/**
 * openchemlib-utils - Various utilities that extends openchemlib-js like HOSE codes or diastereotopic IDs
 * @version v5.11.0
 * @link https://github.com/cheminfo/openchemlib-utils#readme
 * @license MIT
 */
(function (global, factory) {
	typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory() :
	typeof define === 'function' && define.amd ? define(factory) :
	(global = typeof globalThis !== 'undefined' ? globalThis : global || self, global.OCLUtils = factory());
})(this, (function () { 'use strict';

	var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

	function getDefaultExportFromCjs (x) {
		return x && x.__esModule && Object.prototype.hasOwnProperty.call(x, 'default') ? x['default'] : x;
	}

	function getAugmentedNamespace(n) {
		if (n.__esModule) return n;
		var a = Object.defineProperty({}, '__esModule', {value: true});
		Object.keys(n).forEach(function (k) {
			var d = Object.getOwnPropertyDescriptor(n, k);
			Object.defineProperty(a, k, d.get ? d : {
				enumerable: true,
				get: function () {
					return n[k];
				}
			});
		});
		return a;
	}

	var lib = {};

	var TopicMolecule$1 = {};

	var getConnectivityMatrix$1 = {};

	var matrix = {};

	// eslint-disable-next-line @typescript-eslint/unbound-method
	const toString = Object.prototype.toString;
	/**
	 * Checks if an object is an instance of an Array (array or typed array, except those that contain bigint values).
	 *
	 * @param value - Object to check.
	 * @returns True if the object is an array or a typed array.
	 */
	function isAnyArray$1(value) {
	  const tag = toString.call(value);
	  return tag.endsWith('Array]') && !tag.includes('Big');
	}

	var libEsm$2 = /*#__PURE__*/Object.freeze({
		__proto__: null,
		isAnyArray: isAnyArray$1
	});

	var require$$0$3 = /*@__PURE__*/getAugmentedNamespace(libEsm$2);

	function max(input) {
	  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  if (!isAnyArray$1(input)) {
	    throw new TypeError('input must be an array');
	  }
	  if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }
	  var _options$fromIndex = options.fromIndex,
	    fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
	    _options$toIndex = options.toIndex,
	    toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;
	  if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
	    throw new Error('fromIndex must be a positive integer smaller than length');
	  }
	  if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
	    throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
	  }
	  var maxValue = input[fromIndex];
	  for (var i = fromIndex + 1; i < toIndex; i++) {
	    if (input[i] > maxValue) maxValue = input[i];
	  }
	  return maxValue;
	}

	function min(input) {
	  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  if (!isAnyArray$1(input)) {
	    throw new TypeError('input must be an array');
	  }
	  if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }
	  var _options$fromIndex = options.fromIndex,
	    fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
	    _options$toIndex = options.toIndex,
	    toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;
	  if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
	    throw new Error('fromIndex must be a positive integer smaller than length');
	  }
	  if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
	    throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
	  }
	  var minValue = input[fromIndex];
	  for (var i = fromIndex + 1; i < toIndex; i++) {
	    if (input[i] < minValue) minValue = input[i];
	  }
	  return minValue;
	}

	function rescale$1(input) {
	  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  if (!isAnyArray$1(input)) {
	    throw new TypeError('input must be an array');
	  } else if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }
	  var output;
	  if (options.output !== undefined) {
	    if (!isAnyArray$1(options.output)) {
	      throw new TypeError('output option must be an array if specified');
	    }
	    output = options.output;
	  } else {
	    output = new Array(input.length);
	  }
	  var currentMin = min(input);
	  var currentMax = max(input);
	  if (currentMin === currentMax) {
	    throw new RangeError('minimum and maximum input values are equal. Cannot rescale a constant array');
	  }
	  var _options$min = options.min,
	    minValue = _options$min === void 0 ? options.autoMinMax ? currentMin : 0 : _options$min,
	    _options$max = options.max,
	    maxValue = _options$max === void 0 ? options.autoMinMax ? currentMax : 1 : _options$max;
	  if (minValue >= maxValue) {
	    throw new RangeError('min option must be smaller than max option');
	  }
	  var factor = (maxValue - minValue) / (currentMax - currentMin);
	  for (var i = 0; i < input.length; i++) {
	    output[i] = (input[i] - currentMin) * factor + minValue;
	  }
	  return output;
	}

	var libEs6 = /*#__PURE__*/Object.freeze({
		__proto__: null,
		'default': rescale$1
	});

	var require$$1$1 = /*@__PURE__*/getAugmentedNamespace(libEs6);

	Object.defineProperty(matrix, '__esModule', {
	  value: true
	});
	var isAnyArray = require$$0$3;
	var rescale = require$$1$1;
	const indent = ' '.repeat(2);
	const indentData = ' '.repeat(4);
	function inspectMatrix() {
	  return inspectMatrixWithOptions(this);
	}
	function inspectMatrixWithOptions(matrix, options = {}) {
	  const {
	    maxRows = 15,
	    maxColumns = 10,
	    maxNumSize = 8,
	    padMinus = 'auto'
	  } = options;
	  return `${matrix.constructor.name} {
${indent}[
${indentData}${inspectData(matrix, maxRows, maxColumns, maxNumSize, padMinus)}
${indent}]
${indent}rows: ${matrix.rows}
${indent}columns: ${matrix.columns}
}`;
	}
	function inspectData(matrix, maxRows, maxColumns, maxNumSize, padMinus) {
	  const {
	    rows,
	    columns
	  } = matrix;
	  const maxI = Math.min(rows, maxRows);
	  const maxJ = Math.min(columns, maxColumns);
	  const result = [];
	  if (padMinus === 'auto') {
	    padMinus = false;
	    loop: for (let i = 0; i < maxI; i++) {
	      for (let j = 0; j < maxJ; j++) {
	        if (matrix.get(i, j) < 0) {
	          padMinus = true;
	          break loop;
	        }
	      }
	    }
	  }
	  for (let i = 0; i < maxI; i++) {
	    let line = [];
	    for (let j = 0; j < maxJ; j++) {
	      line.push(formatNumber(matrix.get(i, j), maxNumSize, padMinus));
	    }
	    result.push(`${line.join(' ')}`);
	  }
	  if (maxJ !== columns) {
	    result[result.length - 1] += ` ... ${columns - maxColumns} more columns`;
	  }
	  if (maxI !== rows) {
	    result.push(`... ${rows - maxRows} more rows`);
	  }
	  return result.join(`\n${indentData}`);
	}
	function formatNumber(num, maxNumSize, padMinus) {
	  return (num >= 0 && padMinus ? ` ${formatNumber2(num, maxNumSize - 1)}` : formatNumber2(num, maxNumSize)).padEnd(maxNumSize);
	}
	function formatNumber2(num, len) {
	  // small.length numbers should be as is
	  let str = num.toString();
	  if (str.length <= len) return str;

	  // (7)'0.00123' is better then (7)'1.23e-2'
	  // (8)'0.000123' is worse then (7)'1.23e-3',
	  let fix = num.toFixed(len);
	  if (fix.length > len) {
	    fix = num.toFixed(Math.max(0, len - (fix.length - len)));
	  }
	  if (fix.length <= len && !fix.startsWith('0.000') && !fix.startsWith('-0.000')) {
	    return fix;
	  }

	  // well, if it's still too long the user should've used longer numbers
	  let exp = num.toExponential(len);
	  if (exp.length > len) {
	    exp = num.toExponential(Math.max(0, len - (exp.length - len)));
	  }
	  return exp.slice(0);
	}
	function installMathOperations(AbstractMatrix, Matrix) {
	  AbstractMatrix.prototype.add = function add(value) {
	    if (typeof value === 'number') return this.addS(value);
	    return this.addM(value);
	  };
	  AbstractMatrix.prototype.addS = function addS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) + value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.addM = function addM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) + matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.add = function add(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.add(value);
	  };
	  AbstractMatrix.prototype.sub = function sub(value) {
	    if (typeof value === 'number') return this.subS(value);
	    return this.subM(value);
	  };
	  AbstractMatrix.prototype.subS = function subS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) - value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.subM = function subM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) - matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.sub = function sub(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.sub(value);
	  };
	  AbstractMatrix.prototype.subtract = AbstractMatrix.prototype.sub;
	  AbstractMatrix.prototype.subtractS = AbstractMatrix.prototype.subS;
	  AbstractMatrix.prototype.subtractM = AbstractMatrix.prototype.subM;
	  AbstractMatrix.subtract = AbstractMatrix.sub;
	  AbstractMatrix.prototype.mul = function mul(value) {
	    if (typeof value === 'number') return this.mulS(value);
	    return this.mulM(value);
	  };
	  AbstractMatrix.prototype.mulS = function mulS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) * value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.mulM = function mulM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) * matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.mul = function mul(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.mul(value);
	  };
	  AbstractMatrix.prototype.multiply = AbstractMatrix.prototype.mul;
	  AbstractMatrix.prototype.multiplyS = AbstractMatrix.prototype.mulS;
	  AbstractMatrix.prototype.multiplyM = AbstractMatrix.prototype.mulM;
	  AbstractMatrix.multiply = AbstractMatrix.mul;
	  AbstractMatrix.prototype.div = function div(value) {
	    if (typeof value === 'number') return this.divS(value);
	    return this.divM(value);
	  };
	  AbstractMatrix.prototype.divS = function divS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) / value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.divM = function divM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) / matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.div = function div(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.div(value);
	  };
	  AbstractMatrix.prototype.divide = AbstractMatrix.prototype.div;
	  AbstractMatrix.prototype.divideS = AbstractMatrix.prototype.divS;
	  AbstractMatrix.prototype.divideM = AbstractMatrix.prototype.divM;
	  AbstractMatrix.divide = AbstractMatrix.div;
	  AbstractMatrix.prototype.mod = function mod(value) {
	    if (typeof value === 'number') return this.modS(value);
	    return this.modM(value);
	  };
	  AbstractMatrix.prototype.modS = function modS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) % value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.modM = function modM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) % matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.mod = function mod(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.mod(value);
	  };
	  AbstractMatrix.prototype.modulus = AbstractMatrix.prototype.mod;
	  AbstractMatrix.prototype.modulusS = AbstractMatrix.prototype.modS;
	  AbstractMatrix.prototype.modulusM = AbstractMatrix.prototype.modM;
	  AbstractMatrix.modulus = AbstractMatrix.mod;
	  AbstractMatrix.prototype.and = function and(value) {
	    if (typeof value === 'number') return this.andS(value);
	    return this.andM(value);
	  };
	  AbstractMatrix.prototype.andS = function andS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) & value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.andM = function andM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) & matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.and = function and(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.and(value);
	  };
	  AbstractMatrix.prototype.or = function or(value) {
	    if (typeof value === 'number') return this.orS(value);
	    return this.orM(value);
	  };
	  AbstractMatrix.prototype.orS = function orS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) | value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.orM = function orM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) | matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.or = function or(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.or(value);
	  };
	  AbstractMatrix.prototype.xor = function xor(value) {
	    if (typeof value === 'number') return this.xorS(value);
	    return this.xorM(value);
	  };
	  AbstractMatrix.prototype.xorS = function xorS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) ^ value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.xorM = function xorM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) ^ matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.xor = function xor(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.xor(value);
	  };
	  AbstractMatrix.prototype.leftShift = function leftShift(value) {
	    if (typeof value === 'number') return this.leftShiftS(value);
	    return this.leftShiftM(value);
	  };
	  AbstractMatrix.prototype.leftShiftS = function leftShiftS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) << value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.leftShiftM = function leftShiftM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) << matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.leftShift = function leftShift(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.leftShift(value);
	  };
	  AbstractMatrix.prototype.signPropagatingRightShift = function signPropagatingRightShift(value) {
	    if (typeof value === 'number') return this.signPropagatingRightShiftS(value);
	    return this.signPropagatingRightShiftM(value);
	  };
	  AbstractMatrix.prototype.signPropagatingRightShiftS = function signPropagatingRightShiftS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) >> value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.signPropagatingRightShiftM = function signPropagatingRightShiftM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) >> matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.signPropagatingRightShift = function signPropagatingRightShift(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.signPropagatingRightShift(value);
	  };
	  AbstractMatrix.prototype.rightShift = function rightShift(value) {
	    if (typeof value === 'number') return this.rightShiftS(value);
	    return this.rightShiftM(value);
	  };
	  AbstractMatrix.prototype.rightShiftS = function rightShiftS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) >>> value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.rightShiftM = function rightShiftM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) >>> matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.rightShift = function rightShift(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.rightShift(value);
	  };
	  AbstractMatrix.prototype.zeroFillRightShift = AbstractMatrix.prototype.rightShift;
	  AbstractMatrix.prototype.zeroFillRightShiftS = AbstractMatrix.prototype.rightShiftS;
	  AbstractMatrix.prototype.zeroFillRightShiftM = AbstractMatrix.prototype.rightShiftM;
	  AbstractMatrix.zeroFillRightShift = AbstractMatrix.rightShift;
	  AbstractMatrix.prototype.not = function not() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, ~this.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.not = function not(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.not();
	  };
	  AbstractMatrix.prototype.abs = function abs() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.abs(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.abs = function abs(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.abs();
	  };
	  AbstractMatrix.prototype.acos = function acos() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.acos(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.acos = function acos(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.acos();
	  };
	  AbstractMatrix.prototype.acosh = function acosh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.acosh(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.acosh = function acosh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.acosh();
	  };
	  AbstractMatrix.prototype.asin = function asin() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.asin(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.asin = function asin(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.asin();
	  };
	  AbstractMatrix.prototype.asinh = function asinh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.asinh(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.asinh = function asinh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.asinh();
	  };
	  AbstractMatrix.prototype.atan = function atan() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.atan(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.atan = function atan(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.atan();
	  };
	  AbstractMatrix.prototype.atanh = function atanh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.atanh(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.atanh = function atanh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.atanh();
	  };
	  AbstractMatrix.prototype.cbrt = function cbrt() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.cbrt(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.cbrt = function cbrt(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.cbrt();
	  };
	  AbstractMatrix.prototype.ceil = function ceil() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.ceil(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.ceil = function ceil(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.ceil();
	  };
	  AbstractMatrix.prototype.clz32 = function clz32() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.clz32(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.clz32 = function clz32(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.clz32();
	  };
	  AbstractMatrix.prototype.cos = function cos() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.cos(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.cos = function cos(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.cos();
	  };
	  AbstractMatrix.prototype.cosh = function cosh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.cosh(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.cosh = function cosh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.cosh();
	  };
	  AbstractMatrix.prototype.exp = function exp() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.exp(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.exp = function exp(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.exp();
	  };
	  AbstractMatrix.prototype.expm1 = function expm1() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.expm1(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.expm1 = function expm1(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.expm1();
	  };
	  AbstractMatrix.prototype.floor = function floor() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.floor(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.floor = function floor(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.floor();
	  };
	  AbstractMatrix.prototype.fround = function fround() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.fround(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.fround = function fround(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.fround();
	  };
	  AbstractMatrix.prototype.log = function log() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.log(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.log = function log(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.log();
	  };
	  AbstractMatrix.prototype.log1p = function log1p() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.log1p(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.log1p = function log1p(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.log1p();
	  };
	  AbstractMatrix.prototype.log10 = function log10() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.log10(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.log10 = function log10(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.log10();
	  };
	  AbstractMatrix.prototype.log2 = function log2() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.log2(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.log2 = function log2(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.log2();
	  };
	  AbstractMatrix.prototype.round = function round() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.round(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.round = function round(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.round();
	  };
	  AbstractMatrix.prototype.sign = function sign() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.sign(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.sign = function sign(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.sign();
	  };
	  AbstractMatrix.prototype.sin = function sin() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.sin(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.sin = function sin(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.sin();
	  };
	  AbstractMatrix.prototype.sinh = function sinh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.sinh(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.sinh = function sinh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.sinh();
	  };
	  AbstractMatrix.prototype.sqrt = function sqrt() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.sqrt(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.sqrt = function sqrt(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.sqrt();
	  };
	  AbstractMatrix.prototype.tan = function tan() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.tan(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.tan = function tan(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.tan();
	  };
	  AbstractMatrix.prototype.tanh = function tanh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.tanh(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.tanh = function tanh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.tanh();
	  };
	  AbstractMatrix.prototype.trunc = function trunc() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.trunc(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.trunc = function trunc(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.trunc();
	  };
	  AbstractMatrix.pow = function pow(matrix, arg0) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.pow(arg0);
	  };
	  AbstractMatrix.prototype.pow = function pow(value) {
	    if (typeof value === 'number') return this.powS(value);
	    return this.powM(value);
	  };
	  AbstractMatrix.prototype.powS = function powS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.pow(this.get(i, j), value));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.powM = function powM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.pow(this.get(i, j), matrix.get(i, j)));
	      }
	    }
	    return this;
	  };
	}

	/**
	 * @private
	 * Check that a row index is not out of bounds
	 * @param {Matrix} matrix
	 * @param {number} index
	 * @param {boolean} [outer]
	 */
	function checkRowIndex(matrix, index, outer) {
	  let max = outer ? matrix.rows : matrix.rows - 1;
	  if (index < 0 || index > max) {
	    throw new RangeError('Row index out of range');
	  }
	}

	/**
	 * @private
	 * Check that a column index is not out of bounds
	 * @param {Matrix} matrix
	 * @param {number} index
	 * @param {boolean} [outer]
	 */
	function checkColumnIndex(matrix, index, outer) {
	  let max = outer ? matrix.columns : matrix.columns - 1;
	  if (index < 0 || index > max) {
	    throw new RangeError('Column index out of range');
	  }
	}

	/**
	 * @private
	 * Check that the provided vector is an array with the right length
	 * @param {Matrix} matrix
	 * @param {Array|Matrix} vector
	 * @return {Array}
	 * @throws {RangeError}
	 */
	function checkRowVector(matrix, vector) {
	  if (vector.to1DArray) {
	    vector = vector.to1DArray();
	  }
	  if (vector.length !== matrix.columns) {
	    throw new RangeError('vector size must be the same as the number of columns');
	  }
	  return vector;
	}

	/**
	 * @private
	 * Check that the provided vector is an array with the right length
	 * @param {Matrix} matrix
	 * @param {Array|Matrix} vector
	 * @return {Array}
	 * @throws {RangeError}
	 */
	function checkColumnVector(matrix, vector) {
	  if (vector.to1DArray) {
	    vector = vector.to1DArray();
	  }
	  if (vector.length !== matrix.rows) {
	    throw new RangeError('vector size must be the same as the number of rows');
	  }
	  return vector;
	}
	function checkRowIndices(matrix, rowIndices) {
	  if (!isAnyArray.isAnyArray(rowIndices)) {
	    throw new TypeError('row indices must be an array');
	  }
	  for (let i = 0; i < rowIndices.length; i++) {
	    if (rowIndices[i] < 0 || rowIndices[i] >= matrix.rows) {
	      throw new RangeError('row indices are out of range');
	    }
	  }
	}
	function checkColumnIndices(matrix, columnIndices) {
	  if (!isAnyArray.isAnyArray(columnIndices)) {
	    throw new TypeError('column indices must be an array');
	  }
	  for (let i = 0; i < columnIndices.length; i++) {
	    if (columnIndices[i] < 0 || columnIndices[i] >= matrix.columns) {
	      throw new RangeError('column indices are out of range');
	    }
	  }
	}
	function checkRange(matrix, startRow, endRow, startColumn, endColumn) {
	  if (arguments.length !== 5) {
	    throw new RangeError('expected 4 arguments');
	  }
	  checkNumber('startRow', startRow);
	  checkNumber('endRow', endRow);
	  checkNumber('startColumn', startColumn);
	  checkNumber('endColumn', endColumn);
	  if (startRow > endRow || startColumn > endColumn || startRow < 0 || startRow >= matrix.rows || endRow < 0 || endRow >= matrix.rows || startColumn < 0 || startColumn >= matrix.columns || endColumn < 0 || endColumn >= matrix.columns) {
	    throw new RangeError('Submatrix indices are out of range');
	  }
	}
	function newArray(length, value = 0) {
	  let array = [];
	  for (let i = 0; i < length; i++) {
	    array.push(value);
	  }
	  return array;
	}
	function checkNumber(name, value) {
	  if (typeof value !== 'number') {
	    throw new TypeError(`${name} must be a number`);
	  }
	}
	function checkNonEmpty(matrix) {
	  if (matrix.isEmpty()) {
	    throw new Error('Empty matrix has no elements to index');
	  }
	}
	function sumByRow(matrix) {
	  let sum = newArray(matrix.rows);
	  for (let i = 0; i < matrix.rows; ++i) {
	    for (let j = 0; j < matrix.columns; ++j) {
	      sum[i] += matrix.get(i, j);
	    }
	  }
	  return sum;
	}
	function sumByColumn(matrix) {
	  let sum = newArray(matrix.columns);
	  for (let i = 0; i < matrix.rows; ++i) {
	    for (let j = 0; j < matrix.columns; ++j) {
	      sum[j] += matrix.get(i, j);
	    }
	  }
	  return sum;
	}
	function sumAll(matrix) {
	  let v = 0;
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      v += matrix.get(i, j);
	    }
	  }
	  return v;
	}
	function productByRow(matrix) {
	  let sum = newArray(matrix.rows, 1);
	  for (let i = 0; i < matrix.rows; ++i) {
	    for (let j = 0; j < matrix.columns; ++j) {
	      sum[i] *= matrix.get(i, j);
	    }
	  }
	  return sum;
	}
	function productByColumn(matrix) {
	  let sum = newArray(matrix.columns, 1);
	  for (let i = 0; i < matrix.rows; ++i) {
	    for (let j = 0; j < matrix.columns; ++j) {
	      sum[j] *= matrix.get(i, j);
	    }
	  }
	  return sum;
	}
	function productAll(matrix) {
	  let v = 1;
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      v *= matrix.get(i, j);
	    }
	  }
	  return v;
	}
	function varianceByRow(matrix, unbiased, mean) {
	  const rows = matrix.rows;
	  const cols = matrix.columns;
	  const variance = [];
	  for (let i = 0; i < rows; i++) {
	    let sum1 = 0;
	    let sum2 = 0;
	    let x = 0;
	    for (let j = 0; j < cols; j++) {
	      x = matrix.get(i, j) - mean[i];
	      sum1 += x;
	      sum2 += x * x;
	    }
	    if (unbiased) {
	      variance.push((sum2 - sum1 * sum1 / cols) / (cols - 1));
	    } else {
	      variance.push((sum2 - sum1 * sum1 / cols) / cols);
	    }
	  }
	  return variance;
	}
	function varianceByColumn(matrix, unbiased, mean) {
	  const rows = matrix.rows;
	  const cols = matrix.columns;
	  const variance = [];
	  for (let j = 0; j < cols; j++) {
	    let sum1 = 0;
	    let sum2 = 0;
	    let x = 0;
	    for (let i = 0; i < rows; i++) {
	      x = matrix.get(i, j) - mean[j];
	      sum1 += x;
	      sum2 += x * x;
	    }
	    if (unbiased) {
	      variance.push((sum2 - sum1 * sum1 / rows) / (rows - 1));
	    } else {
	      variance.push((sum2 - sum1 * sum1 / rows) / rows);
	    }
	  }
	  return variance;
	}
	function varianceAll(matrix, unbiased, mean) {
	  const rows = matrix.rows;
	  const cols = matrix.columns;
	  const size = rows * cols;
	  let sum1 = 0;
	  let sum2 = 0;
	  let x = 0;
	  for (let i = 0; i < rows; i++) {
	    for (let j = 0; j < cols; j++) {
	      x = matrix.get(i, j) - mean;
	      sum1 += x;
	      sum2 += x * x;
	    }
	  }
	  if (unbiased) {
	    return (sum2 - sum1 * sum1 / size) / (size - 1);
	  } else {
	    return (sum2 - sum1 * sum1 / size) / size;
	  }
	}
	function centerByRow(matrix, mean) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) - mean[i]);
	    }
	  }
	}
	function centerByColumn(matrix, mean) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) - mean[j]);
	    }
	  }
	}
	function centerAll(matrix, mean) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) - mean);
	    }
	  }
	}
	function getScaleByRow(matrix) {
	  const scale = [];
	  for (let i = 0; i < matrix.rows; i++) {
	    let sum = 0;
	    for (let j = 0; j < matrix.columns; j++) {
	      sum += Math.pow(matrix.get(i, j), 2) / (matrix.columns - 1);
	    }
	    scale.push(Math.sqrt(sum));
	  }
	  return scale;
	}
	function scaleByRow(matrix, scale) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) / scale[i]);
	    }
	  }
	}
	function getScaleByColumn(matrix) {
	  const scale = [];
	  for (let j = 0; j < matrix.columns; j++) {
	    let sum = 0;
	    for (let i = 0; i < matrix.rows; i++) {
	      sum += Math.pow(matrix.get(i, j), 2) / (matrix.rows - 1);
	    }
	    scale.push(Math.sqrt(sum));
	  }
	  return scale;
	}
	function scaleByColumn(matrix, scale) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) / scale[j]);
	    }
	  }
	}
	function getScaleAll(matrix) {
	  const divider = matrix.size - 1;
	  let sum = 0;
	  for (let j = 0; j < matrix.columns; j++) {
	    for (let i = 0; i < matrix.rows; i++) {
	      sum += Math.pow(matrix.get(i, j), 2) / divider;
	    }
	  }
	  return Math.sqrt(sum);
	}
	function scaleAll(matrix, scale) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) / scale);
	    }
	  }
	}
	class AbstractMatrix {
	  static from1DArray(newRows, newColumns, newData) {
	    let length = newRows * newColumns;
	    if (length !== newData.length) {
	      throw new RangeError('data length does not match given dimensions');
	    }
	    let newMatrix = new Matrix$1(newRows, newColumns);
	    for (let row = 0; row < newRows; row++) {
	      for (let column = 0; column < newColumns; column++) {
	        newMatrix.set(row, column, newData[row * newColumns + column]);
	      }
	    }
	    return newMatrix;
	  }
	  static rowVector(newData) {
	    let vector = new Matrix$1(1, newData.length);
	    for (let i = 0; i < newData.length; i++) {
	      vector.set(0, i, newData[i]);
	    }
	    return vector;
	  }
	  static columnVector(newData) {
	    let vector = new Matrix$1(newData.length, 1);
	    for (let i = 0; i < newData.length; i++) {
	      vector.set(i, 0, newData[i]);
	    }
	    return vector;
	  }
	  static zeros(rows, columns) {
	    return new Matrix$1(rows, columns);
	  }
	  static ones(rows, columns) {
	    return new Matrix$1(rows, columns).fill(1);
	  }
	  static rand(rows, columns, options = {}) {
	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }
	    const {
	      random = Math.random
	    } = options;
	    let matrix = new Matrix$1(rows, columns);
	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        matrix.set(i, j, random());
	      }
	    }
	    return matrix;
	  }
	  static randInt(rows, columns, options = {}) {
	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }
	    const {
	      min = 0,
	      max = 1000,
	      random = Math.random
	    } = options;
	    if (!Number.isInteger(min)) throw new TypeError('min must be an integer');
	    if (!Number.isInteger(max)) throw new TypeError('max must be an integer');
	    if (min >= max) throw new RangeError('min must be smaller than max');
	    let interval = max - min;
	    let matrix = new Matrix$1(rows, columns);
	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        let value = min + Math.round(random() * interval);
	        matrix.set(i, j, value);
	      }
	    }
	    return matrix;
	  }
	  static eye(rows, columns, value) {
	    if (columns === undefined) columns = rows;
	    if (value === undefined) value = 1;
	    let min = Math.min(rows, columns);
	    let matrix = this.zeros(rows, columns);
	    for (let i = 0; i < min; i++) {
	      matrix.set(i, i, value);
	    }
	    return matrix;
	  }
	  static diag(data, rows, columns) {
	    let l = data.length;
	    if (rows === undefined) rows = l;
	    if (columns === undefined) columns = rows;
	    let min = Math.min(l, rows, columns);
	    let matrix = this.zeros(rows, columns);
	    for (let i = 0; i < min; i++) {
	      matrix.set(i, i, data[i]);
	    }
	    return matrix;
	  }
	  static min(matrix1, matrix2) {
	    matrix1 = this.checkMatrix(matrix1);
	    matrix2 = this.checkMatrix(matrix2);
	    let rows = matrix1.rows;
	    let columns = matrix1.columns;
	    let result = new Matrix$1(rows, columns);
	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        result.set(i, j, Math.min(matrix1.get(i, j), matrix2.get(i, j)));
	      }
	    }
	    return result;
	  }
	  static max(matrix1, matrix2) {
	    matrix1 = this.checkMatrix(matrix1);
	    matrix2 = this.checkMatrix(matrix2);
	    let rows = matrix1.rows;
	    let columns = matrix1.columns;
	    let result = new this(rows, columns);
	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        result.set(i, j, Math.max(matrix1.get(i, j), matrix2.get(i, j)));
	      }
	    }
	    return result;
	  }
	  static checkMatrix(value) {
	    return AbstractMatrix.isMatrix(value) ? value : new Matrix$1(value);
	  }
	  static isMatrix(value) {
	    return value != null && value.klass === 'Matrix';
	  }
	  get size() {
	    return this.rows * this.columns;
	  }
	  apply(callback) {
	    if (typeof callback !== 'function') {
	      throw new TypeError('callback must be a function');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        callback.call(this, i, j);
	      }
	    }
	    return this;
	  }
	  to1DArray() {
	    let array = [];
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        array.push(this.get(i, j));
	      }
	    }
	    return array;
	  }
	  to2DArray() {
	    let copy = [];
	    for (let i = 0; i < this.rows; i++) {
	      copy.push([]);
	      for (let j = 0; j < this.columns; j++) {
	        copy[i].push(this.get(i, j));
	      }
	    }
	    return copy;
	  }
	  toJSON() {
	    return this.to2DArray();
	  }
	  isRowVector() {
	    return this.rows === 1;
	  }
	  isColumnVector() {
	    return this.columns === 1;
	  }
	  isVector() {
	    return this.rows === 1 || this.columns === 1;
	  }
	  isSquare() {
	    return this.rows === this.columns;
	  }
	  isEmpty() {
	    return this.rows === 0 || this.columns === 0;
	  }
	  isSymmetric() {
	    if (this.isSquare()) {
	      for (let i = 0; i < this.rows; i++) {
	        for (let j = 0; j <= i; j++) {
	          if (this.get(i, j) !== this.get(j, i)) {
	            return false;
	          }
	        }
	      }
	      return true;
	    }
	    return false;
	  }
	  isDistance() {
	    if (!this.isSymmetric()) return false;
	    for (let i = 0; i < this.rows; i++) {
	      if (this.get(i, i) !== 0) return false;
	    }
	    return true;
	  }
	  isEchelonForm() {
	    let i = 0;
	    let j = 0;
	    let previousColumn = -1;
	    let isEchelonForm = true;
	    let checked = false;
	    while (i < this.rows && isEchelonForm) {
	      j = 0;
	      checked = false;
	      while (j < this.columns && checked === false) {
	        if (this.get(i, j) === 0) {
	          j++;
	        } else if (this.get(i, j) === 1 && j > previousColumn) {
	          checked = true;
	          previousColumn = j;
	        } else {
	          isEchelonForm = false;
	          checked = true;
	        }
	      }
	      i++;
	    }
	    return isEchelonForm;
	  }
	  isReducedEchelonForm() {
	    let i = 0;
	    let j = 0;
	    let previousColumn = -1;
	    let isReducedEchelonForm = true;
	    let checked = false;
	    while (i < this.rows && isReducedEchelonForm) {
	      j = 0;
	      checked = false;
	      while (j < this.columns && checked === false) {
	        if (this.get(i, j) === 0) {
	          j++;
	        } else if (this.get(i, j) === 1 && j > previousColumn) {
	          checked = true;
	          previousColumn = j;
	        } else {
	          isReducedEchelonForm = false;
	          checked = true;
	        }
	      }
	      for (let k = j + 1; k < this.rows; k++) {
	        if (this.get(i, k) !== 0) {
	          isReducedEchelonForm = false;
	        }
	      }
	      i++;
	    }
	    return isReducedEchelonForm;
	  }
	  echelonForm() {
	    let result = this.clone();
	    let h = 0;
	    let k = 0;
	    while (h < result.rows && k < result.columns) {
	      let iMax = h;
	      for (let i = h; i < result.rows; i++) {
	        if (result.get(i, k) > result.get(iMax, k)) {
	          iMax = i;
	        }
	      }
	      if (result.get(iMax, k) === 0) {
	        k++;
	      } else {
	        result.swapRows(h, iMax);
	        let tmp = result.get(h, k);
	        for (let j = k; j < result.columns; j++) {
	          result.set(h, j, result.get(h, j) / tmp);
	        }
	        for (let i = h + 1; i < result.rows; i++) {
	          let factor = result.get(i, k) / result.get(h, k);
	          result.set(i, k, 0);
	          for (let j = k + 1; j < result.columns; j++) {
	            result.set(i, j, result.get(i, j) - result.get(h, j) * factor);
	          }
	        }
	        h++;
	        k++;
	      }
	    }
	    return result;
	  }
	  reducedEchelonForm() {
	    let result = this.echelonForm();
	    let m = result.columns;
	    let n = result.rows;
	    let h = n - 1;
	    while (h >= 0) {
	      if (result.maxRow(h) === 0) {
	        h--;
	      } else {
	        let p = 0;
	        let pivot = false;
	        while (p < n && pivot === false) {
	          if (result.get(h, p) === 1) {
	            pivot = true;
	          } else {
	            p++;
	          }
	        }
	        for (let i = 0; i < h; i++) {
	          let factor = result.get(i, p);
	          for (let j = p; j < m; j++) {
	            let tmp = result.get(i, j) - factor * result.get(h, j);
	            result.set(i, j, tmp);
	          }
	        }
	        h--;
	      }
	    }
	    return result;
	  }
	  set() {
	    throw new Error('set method is unimplemented');
	  }
	  get() {
	    throw new Error('get method is unimplemented');
	  }
	  repeat(options = {}) {
	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }
	    const {
	      rows = 1,
	      columns = 1
	    } = options;
	    if (!Number.isInteger(rows) || rows <= 0) {
	      throw new TypeError('rows must be a positive integer');
	    }
	    if (!Number.isInteger(columns) || columns <= 0) {
	      throw new TypeError('columns must be a positive integer');
	    }
	    let matrix = new Matrix$1(this.rows * rows, this.columns * columns);
	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        matrix.setSubMatrix(this, this.rows * i, this.columns * j);
	      }
	    }
	    return matrix;
	  }
	  fill(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, value);
	      }
	    }
	    return this;
	  }
	  neg() {
	    return this.mulS(-1);
	  }
	  getRow(index) {
	    checkRowIndex(this, index);
	    let row = [];
	    for (let i = 0; i < this.columns; i++) {
	      row.push(this.get(index, i));
	    }
	    return row;
	  }
	  getRowVector(index) {
	    return Matrix$1.rowVector(this.getRow(index));
	  }
	  setRow(index, array) {
	    checkRowIndex(this, index);
	    array = checkRowVector(this, array);
	    for (let i = 0; i < this.columns; i++) {
	      this.set(index, i, array[i]);
	    }
	    return this;
	  }
	  swapRows(row1, row2) {
	    checkRowIndex(this, row1);
	    checkRowIndex(this, row2);
	    for (let i = 0; i < this.columns; i++) {
	      let temp = this.get(row1, i);
	      this.set(row1, i, this.get(row2, i));
	      this.set(row2, i, temp);
	    }
	    return this;
	  }
	  getColumn(index) {
	    checkColumnIndex(this, index);
	    let column = [];
	    for (let i = 0; i < this.rows; i++) {
	      column.push(this.get(i, index));
	    }
	    return column;
	  }
	  getColumnVector(index) {
	    return Matrix$1.columnVector(this.getColumn(index));
	  }
	  setColumn(index, array) {
	    checkColumnIndex(this, index);
	    array = checkColumnVector(this, array);
	    for (let i = 0; i < this.rows; i++) {
	      this.set(i, index, array[i]);
	    }
	    return this;
	  }
	  swapColumns(column1, column2) {
	    checkColumnIndex(this, column1);
	    checkColumnIndex(this, column2);
	    for (let i = 0; i < this.rows; i++) {
	      let temp = this.get(i, column1);
	      this.set(i, column1, this.get(i, column2));
	      this.set(i, column2, temp);
	    }
	    return this;
	  }
	  addRowVector(vector) {
	    vector = checkRowVector(this, vector);
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) + vector[j]);
	      }
	    }
	    return this;
	  }
	  subRowVector(vector) {
	    vector = checkRowVector(this, vector);
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) - vector[j]);
	      }
	    }
	    return this;
	  }
	  mulRowVector(vector) {
	    vector = checkRowVector(this, vector);
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) * vector[j]);
	      }
	    }
	    return this;
	  }
	  divRowVector(vector) {
	    vector = checkRowVector(this, vector);
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) / vector[j]);
	      }
	    }
	    return this;
	  }
	  addColumnVector(vector) {
	    vector = checkColumnVector(this, vector);
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) + vector[i]);
	      }
	    }
	    return this;
	  }
	  subColumnVector(vector) {
	    vector = checkColumnVector(this, vector);
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) - vector[i]);
	      }
	    }
	    return this;
	  }
	  mulColumnVector(vector) {
	    vector = checkColumnVector(this, vector);
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) * vector[i]);
	      }
	    }
	    return this;
	  }
	  divColumnVector(vector) {
	    vector = checkColumnVector(this, vector);
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) / vector[i]);
	      }
	    }
	    return this;
	  }
	  mulRow(index, value) {
	    checkRowIndex(this, index);
	    for (let i = 0; i < this.columns; i++) {
	      this.set(index, i, this.get(index, i) * value);
	    }
	    return this;
	  }
	  mulColumn(index, value) {
	    checkColumnIndex(this, index);
	    for (let i = 0; i < this.rows; i++) {
	      this.set(i, index, this.get(i, index) * value);
	    }
	    return this;
	  }
	  max(by) {
	    if (this.isEmpty()) {
	      return NaN;
	    }
	    switch (by) {
	      case 'row':
	        {
	          const max = new Array(this.rows).fill(Number.NEGATIVE_INFINITY);
	          for (let row = 0; row < this.rows; row++) {
	            for (let column = 0; column < this.columns; column++) {
	              if (this.get(row, column) > max[row]) {
	                max[row] = this.get(row, column);
	              }
	            }
	          }
	          return max;
	        }
	      case 'column':
	        {
	          const max = new Array(this.columns).fill(Number.NEGATIVE_INFINITY);
	          for (let row = 0; row < this.rows; row++) {
	            for (let column = 0; column < this.columns; column++) {
	              if (this.get(row, column) > max[column]) {
	                max[column] = this.get(row, column);
	              }
	            }
	          }
	          return max;
	        }
	      case undefined:
	        {
	          let max = this.get(0, 0);
	          for (let row = 0; row < this.rows; row++) {
	            for (let column = 0; column < this.columns; column++) {
	              if (this.get(row, column) > max) {
	                max = this.get(row, column);
	              }
	            }
	          }
	          return max;
	        }
	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }
	  maxIndex() {
	    checkNonEmpty(this);
	    let v = this.get(0, 0);
	    let idx = [0, 0];
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        if (this.get(i, j) > v) {
	          v = this.get(i, j);
	          idx[0] = i;
	          idx[1] = j;
	        }
	      }
	    }
	    return idx;
	  }
	  min(by) {
	    if (this.isEmpty()) {
	      return NaN;
	    }
	    switch (by) {
	      case 'row':
	        {
	          const min = new Array(this.rows).fill(Number.POSITIVE_INFINITY);
	          for (let row = 0; row < this.rows; row++) {
	            for (let column = 0; column < this.columns; column++) {
	              if (this.get(row, column) < min[row]) {
	                min[row] = this.get(row, column);
	              }
	            }
	          }
	          return min;
	        }
	      case 'column':
	        {
	          const min = new Array(this.columns).fill(Number.POSITIVE_INFINITY);
	          for (let row = 0; row < this.rows; row++) {
	            for (let column = 0; column < this.columns; column++) {
	              if (this.get(row, column) < min[column]) {
	                min[column] = this.get(row, column);
	              }
	            }
	          }
	          return min;
	        }
	      case undefined:
	        {
	          let min = this.get(0, 0);
	          for (let row = 0; row < this.rows; row++) {
	            for (let column = 0; column < this.columns; column++) {
	              if (this.get(row, column) < min) {
	                min = this.get(row, column);
	              }
	            }
	          }
	          return min;
	        }
	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }
	  minIndex() {
	    checkNonEmpty(this);
	    let v = this.get(0, 0);
	    let idx = [0, 0];
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        if (this.get(i, j) < v) {
	          v = this.get(i, j);
	          idx[0] = i;
	          idx[1] = j;
	        }
	      }
	    }
	    return idx;
	  }
	  maxRow(row) {
	    checkRowIndex(this, row);
	    if (this.isEmpty()) {
	      return NaN;
	    }
	    let v = this.get(row, 0);
	    for (let i = 1; i < this.columns; i++) {
	      if (this.get(row, i) > v) {
	        v = this.get(row, i);
	      }
	    }
	    return v;
	  }
	  maxRowIndex(row) {
	    checkRowIndex(this, row);
	    checkNonEmpty(this);
	    let v = this.get(row, 0);
	    let idx = [row, 0];
	    for (let i = 1; i < this.columns; i++) {
	      if (this.get(row, i) > v) {
	        v = this.get(row, i);
	        idx[1] = i;
	      }
	    }
	    return idx;
	  }
	  minRow(row) {
	    checkRowIndex(this, row);
	    if (this.isEmpty()) {
	      return NaN;
	    }
	    let v = this.get(row, 0);
	    for (let i = 1; i < this.columns; i++) {
	      if (this.get(row, i) < v) {
	        v = this.get(row, i);
	      }
	    }
	    return v;
	  }
	  minRowIndex(row) {
	    checkRowIndex(this, row);
	    checkNonEmpty(this);
	    let v = this.get(row, 0);
	    let idx = [row, 0];
	    for (let i = 1; i < this.columns; i++) {
	      if (this.get(row, i) < v) {
	        v = this.get(row, i);
	        idx[1] = i;
	      }
	    }
	    return idx;
	  }
	  maxColumn(column) {
	    checkColumnIndex(this, column);
	    if (this.isEmpty()) {
	      return NaN;
	    }
	    let v = this.get(0, column);
	    for (let i = 1; i < this.rows; i++) {
	      if (this.get(i, column) > v) {
	        v = this.get(i, column);
	      }
	    }
	    return v;
	  }
	  maxColumnIndex(column) {
	    checkColumnIndex(this, column);
	    checkNonEmpty(this);
	    let v = this.get(0, column);
	    let idx = [0, column];
	    for (let i = 1; i < this.rows; i++) {
	      if (this.get(i, column) > v) {
	        v = this.get(i, column);
	        idx[0] = i;
	      }
	    }
	    return idx;
	  }
	  minColumn(column) {
	    checkColumnIndex(this, column);
	    if (this.isEmpty()) {
	      return NaN;
	    }
	    let v = this.get(0, column);
	    for (let i = 1; i < this.rows; i++) {
	      if (this.get(i, column) < v) {
	        v = this.get(i, column);
	      }
	    }
	    return v;
	  }
	  minColumnIndex(column) {
	    checkColumnIndex(this, column);
	    checkNonEmpty(this);
	    let v = this.get(0, column);
	    let idx = [0, column];
	    for (let i = 1; i < this.rows; i++) {
	      if (this.get(i, column) < v) {
	        v = this.get(i, column);
	        idx[0] = i;
	      }
	    }
	    return idx;
	  }
	  diag() {
	    let min = Math.min(this.rows, this.columns);
	    let diag = [];
	    for (let i = 0; i < min; i++) {
	      diag.push(this.get(i, i));
	    }
	    return diag;
	  }
	  norm(type = 'frobenius') {
	    switch (type) {
	      case 'max':
	        return this.max();
	      case 'frobenius':
	        return Math.sqrt(this.dot(this));
	      default:
	        throw new RangeError(`unknown norm type: ${type}`);
	    }
	  }
	  cumulativeSum() {
	    let sum = 0;
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        sum += this.get(i, j);
	        this.set(i, j, sum);
	      }
	    }
	    return this;
	  }
	  dot(vector2) {
	    if (AbstractMatrix.isMatrix(vector2)) vector2 = vector2.to1DArray();
	    let vector1 = this.to1DArray();
	    if (vector1.length !== vector2.length) {
	      throw new RangeError('vectors do not have the same size');
	    }
	    let dot = 0;
	    for (let i = 0; i < vector1.length; i++) {
	      dot += vector1[i] * vector2[i];
	    }
	    return dot;
	  }
	  mmul(other) {
	    other = Matrix$1.checkMatrix(other);
	    let m = this.rows;
	    let n = this.columns;
	    let p = other.columns;
	    let result = new Matrix$1(m, p);
	    let Bcolj = new Float64Array(n);
	    for (let j = 0; j < p; j++) {
	      for (let k = 0; k < n; k++) {
	        Bcolj[k] = other.get(k, j);
	      }
	      for (let i = 0; i < m; i++) {
	        let s = 0;
	        for (let k = 0; k < n; k++) {
	          s += this.get(i, k) * Bcolj[k];
	        }
	        result.set(i, j, s);
	      }
	    }
	    return result;
	  }
	  strassen2x2(other) {
	    other = Matrix$1.checkMatrix(other);
	    let result = new Matrix$1(2, 2);
	    const a11 = this.get(0, 0);
	    const b11 = other.get(0, 0);
	    const a12 = this.get(0, 1);
	    const b12 = other.get(0, 1);
	    const a21 = this.get(1, 0);
	    const b21 = other.get(1, 0);
	    const a22 = this.get(1, 1);
	    const b22 = other.get(1, 1);

	    // Compute intermediate values.
	    const m1 = (a11 + a22) * (b11 + b22);
	    const m2 = (a21 + a22) * b11;
	    const m3 = a11 * (b12 - b22);
	    const m4 = a22 * (b21 - b11);
	    const m5 = (a11 + a12) * b22;
	    const m6 = (a21 - a11) * (b11 + b12);
	    const m7 = (a12 - a22) * (b21 + b22);

	    // Combine intermediate values into the output.
	    const c00 = m1 + m4 - m5 + m7;
	    const c01 = m3 + m5;
	    const c10 = m2 + m4;
	    const c11 = m1 - m2 + m3 + m6;
	    result.set(0, 0, c00);
	    result.set(0, 1, c01);
	    result.set(1, 0, c10);
	    result.set(1, 1, c11);
	    return result;
	  }
	  strassen3x3(other) {
	    other = Matrix$1.checkMatrix(other);
	    let result = new Matrix$1(3, 3);
	    const a00 = this.get(0, 0);
	    const a01 = this.get(0, 1);
	    const a02 = this.get(0, 2);
	    const a10 = this.get(1, 0);
	    const a11 = this.get(1, 1);
	    const a12 = this.get(1, 2);
	    const a20 = this.get(2, 0);
	    const a21 = this.get(2, 1);
	    const a22 = this.get(2, 2);
	    const b00 = other.get(0, 0);
	    const b01 = other.get(0, 1);
	    const b02 = other.get(0, 2);
	    const b10 = other.get(1, 0);
	    const b11 = other.get(1, 1);
	    const b12 = other.get(1, 2);
	    const b20 = other.get(2, 0);
	    const b21 = other.get(2, 1);
	    const b22 = other.get(2, 2);
	    const m1 = (a00 + a01 + a02 - a10 - a11 - a21 - a22) * b11;
	    const m2 = (a00 - a10) * (-b01 + b11);
	    const m3 = a11 * (-b00 + b01 + b10 - b11 - b12 - b20 + b22);
	    const m4 = (-a00 + a10 + a11) * (b00 - b01 + b11);
	    const m5 = (a10 + a11) * (-b00 + b01);
	    const m6 = a00 * b00;
	    const m7 = (-a00 + a20 + a21) * (b00 - b02 + b12);
	    const m8 = (-a00 + a20) * (b02 - b12);
	    const m9 = (a20 + a21) * (-b00 + b02);
	    const m10 = (a00 + a01 + a02 - a11 - a12 - a20 - a21) * b12;
	    const m11 = a21 * (-b00 + b02 + b10 - b11 - b12 - b20 + b21);
	    const m12 = (-a02 + a21 + a22) * (b11 + b20 - b21);
	    const m13 = (a02 - a22) * (b11 - b21);
	    const m14 = a02 * b20;
	    const m15 = (a21 + a22) * (-b20 + b21);
	    const m16 = (-a02 + a11 + a12) * (b12 + b20 - b22);
	    const m17 = (a02 - a12) * (b12 - b22);
	    const m18 = (a11 + a12) * (-b20 + b22);
	    const m19 = a01 * b10;
	    const m20 = a12 * b21;
	    const m21 = a10 * b02;
	    const m22 = a20 * b01;
	    const m23 = a22 * b22;
	    const c00 = m6 + m14 + m19;
	    const c01 = m1 + m4 + m5 + m6 + m12 + m14 + m15;
	    const c02 = m6 + m7 + m9 + m10 + m14 + m16 + m18;
	    const c10 = m2 + m3 + m4 + m6 + m14 + m16 + m17;
	    const c11 = m2 + m4 + m5 + m6 + m20;
	    const c12 = m14 + m16 + m17 + m18 + m21;
	    const c20 = m6 + m7 + m8 + m11 + m12 + m13 + m14;
	    const c21 = m12 + m13 + m14 + m15 + m22;
	    const c22 = m6 + m7 + m8 + m9 + m23;
	    result.set(0, 0, c00);
	    result.set(0, 1, c01);
	    result.set(0, 2, c02);
	    result.set(1, 0, c10);
	    result.set(1, 1, c11);
	    result.set(1, 2, c12);
	    result.set(2, 0, c20);
	    result.set(2, 1, c21);
	    result.set(2, 2, c22);
	    return result;
	  }
	  mmulStrassen(y) {
	    y = Matrix$1.checkMatrix(y);
	    let x = this.clone();
	    let r1 = x.rows;
	    let c1 = x.columns;
	    let r2 = y.rows;
	    let c2 = y.columns;
	    if (c1 !== r2) {
	      // eslint-disable-next-line no-console
	      console.warn(`Multiplying ${r1} x ${c1} and ${r2} x ${c2} matrix: dimensions do not match.`);
	    }

	    // Put a matrix into the top left of a matrix of zeros.
	    // `rows` and `cols` are the dimensions of the output matrix.
	    function embed(mat, rows, cols) {
	      let r = mat.rows;
	      let c = mat.columns;
	      if (r === rows && c === cols) {
	        return mat;
	      } else {
	        let resultat = AbstractMatrix.zeros(rows, cols);
	        resultat = resultat.setSubMatrix(mat, 0, 0);
	        return resultat;
	      }
	    }

	    // Make sure both matrices are the same size.
	    // This is exclusively for simplicity:
	    // this algorithm can be implemented with matrices of different sizes.

	    let r = Math.max(r1, r2);
	    let c = Math.max(c1, c2);
	    x = embed(x, r, c);
	    y = embed(y, r, c);

	    // Our recursive multiplication function.
	    function blockMult(a, b, rows, cols) {
	      // For small matrices, resort to naive multiplication.
	      if (rows <= 512 || cols <= 512) {
	        return a.mmul(b); // a is equivalent to this
	      }

	      // Apply dynamic padding.
	      if (rows % 2 === 1 && cols % 2 === 1) {
	        a = embed(a, rows + 1, cols + 1);
	        b = embed(b, rows + 1, cols + 1);
	      } else if (rows % 2 === 1) {
	        a = embed(a, rows + 1, cols);
	        b = embed(b, rows + 1, cols);
	      } else if (cols % 2 === 1) {
	        a = embed(a, rows, cols + 1);
	        b = embed(b, rows, cols + 1);
	      }
	      let halfRows = parseInt(a.rows / 2, 10);
	      let halfCols = parseInt(a.columns / 2, 10);
	      // Subdivide input matrices.
	      let a11 = a.subMatrix(0, halfRows - 1, 0, halfCols - 1);
	      let b11 = b.subMatrix(0, halfRows - 1, 0, halfCols - 1);
	      let a12 = a.subMatrix(0, halfRows - 1, halfCols, a.columns - 1);
	      let b12 = b.subMatrix(0, halfRows - 1, halfCols, b.columns - 1);
	      let a21 = a.subMatrix(halfRows, a.rows - 1, 0, halfCols - 1);
	      let b21 = b.subMatrix(halfRows, b.rows - 1, 0, halfCols - 1);
	      let a22 = a.subMatrix(halfRows, a.rows - 1, halfCols, a.columns - 1);
	      let b22 = b.subMatrix(halfRows, b.rows - 1, halfCols, b.columns - 1);

	      // Compute intermediate values.
	      let m1 = blockMult(AbstractMatrix.add(a11, a22), AbstractMatrix.add(b11, b22), halfRows, halfCols);
	      let m2 = blockMult(AbstractMatrix.add(a21, a22), b11, halfRows, halfCols);
	      let m3 = blockMult(a11, AbstractMatrix.sub(b12, b22), halfRows, halfCols);
	      let m4 = blockMult(a22, AbstractMatrix.sub(b21, b11), halfRows, halfCols);
	      let m5 = blockMult(AbstractMatrix.add(a11, a12), b22, halfRows, halfCols);
	      let m6 = blockMult(AbstractMatrix.sub(a21, a11), AbstractMatrix.add(b11, b12), halfRows, halfCols);
	      let m7 = blockMult(AbstractMatrix.sub(a12, a22), AbstractMatrix.add(b21, b22), halfRows, halfCols);

	      // Combine intermediate values into the output.
	      let c11 = AbstractMatrix.add(m1, m4);
	      c11.sub(m5);
	      c11.add(m7);
	      let c12 = AbstractMatrix.add(m3, m5);
	      let c21 = AbstractMatrix.add(m2, m4);
	      let c22 = AbstractMatrix.sub(m1, m2);
	      c22.add(m3);
	      c22.add(m6);

	      // Crop output to the desired size (undo dynamic padding).
	      let result = AbstractMatrix.zeros(2 * c11.rows, 2 * c11.columns);
	      result = result.setSubMatrix(c11, 0, 0);
	      result = result.setSubMatrix(c12, c11.rows, 0);
	      result = result.setSubMatrix(c21, 0, c11.columns);
	      result = result.setSubMatrix(c22, c11.rows, c11.columns);
	      return result.subMatrix(0, rows - 1, 0, cols - 1);
	    }
	    return blockMult(x, y, r, c);
	  }
	  scaleRows(options = {}) {
	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }
	    const {
	      min = 0,
	      max = 1
	    } = options;
	    if (!Number.isFinite(min)) throw new TypeError('min must be a number');
	    if (!Number.isFinite(max)) throw new TypeError('max must be a number');
	    if (min >= max) throw new RangeError('min must be smaller than max');
	    let newMatrix = new Matrix$1(this.rows, this.columns);
	    for (let i = 0; i < this.rows; i++) {
	      const row = this.getRow(i);
	      if (row.length > 0) {
	        rescale(row, {
	          min,
	          max,
	          output: row
	        });
	      }
	      newMatrix.setRow(i, row);
	    }
	    return newMatrix;
	  }
	  scaleColumns(options = {}) {
	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }
	    const {
	      min = 0,
	      max = 1
	    } = options;
	    if (!Number.isFinite(min)) throw new TypeError('min must be a number');
	    if (!Number.isFinite(max)) throw new TypeError('max must be a number');
	    if (min >= max) throw new RangeError('min must be smaller than max');
	    let newMatrix = new Matrix$1(this.rows, this.columns);
	    for (let i = 0; i < this.columns; i++) {
	      const column = this.getColumn(i);
	      if (column.length) {
	        rescale(column, {
	          min,
	          max,
	          output: column
	        });
	      }
	      newMatrix.setColumn(i, column);
	    }
	    return newMatrix;
	  }
	  flipRows() {
	    const middle = Math.ceil(this.columns / 2);
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < middle; j++) {
	        let first = this.get(i, j);
	        let last = this.get(i, this.columns - 1 - j);
	        this.set(i, j, last);
	        this.set(i, this.columns - 1 - j, first);
	      }
	    }
	    return this;
	  }
	  flipColumns() {
	    const middle = Math.ceil(this.rows / 2);
	    for (let j = 0; j < this.columns; j++) {
	      for (let i = 0; i < middle; i++) {
	        let first = this.get(i, j);
	        let last = this.get(this.rows - 1 - i, j);
	        this.set(i, j, last);
	        this.set(this.rows - 1 - i, j, first);
	      }
	    }
	    return this;
	  }
	  kroneckerProduct(other) {
	    other = Matrix$1.checkMatrix(other);
	    let m = this.rows;
	    let n = this.columns;
	    let p = other.rows;
	    let q = other.columns;
	    let result = new Matrix$1(m * p, n * q);
	    for (let i = 0; i < m; i++) {
	      for (let j = 0; j < n; j++) {
	        for (let k = 0; k < p; k++) {
	          for (let l = 0; l < q; l++) {
	            result.set(p * i + k, q * j + l, this.get(i, j) * other.get(k, l));
	          }
	        }
	      }
	    }
	    return result;
	  }
	  kroneckerSum(other) {
	    other = Matrix$1.checkMatrix(other);
	    if (!this.isSquare() || !other.isSquare()) {
	      throw new Error('Kronecker Sum needs two Square Matrices');
	    }
	    let m = this.rows;
	    let n = other.rows;
	    let AxI = this.kroneckerProduct(Matrix$1.eye(n, n));
	    let IxB = Matrix$1.eye(m, m).kroneckerProduct(other);
	    return AxI.add(IxB);
	  }
	  transpose() {
	    let result = new Matrix$1(this.columns, this.rows);
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        result.set(j, i, this.get(i, j));
	      }
	    }
	    return result;
	  }
	  sortRows(compareFunction = compareNumbers) {
	    for (let i = 0; i < this.rows; i++) {
	      this.setRow(i, this.getRow(i).sort(compareFunction));
	    }
	    return this;
	  }
	  sortColumns(compareFunction = compareNumbers) {
	    for (let i = 0; i < this.columns; i++) {
	      this.setColumn(i, this.getColumn(i).sort(compareFunction));
	    }
	    return this;
	  }
	  subMatrix(startRow, endRow, startColumn, endColumn) {
	    checkRange(this, startRow, endRow, startColumn, endColumn);
	    let newMatrix = new Matrix$1(endRow - startRow + 1, endColumn - startColumn + 1);
	    for (let i = startRow; i <= endRow; i++) {
	      for (let j = startColumn; j <= endColumn; j++) {
	        newMatrix.set(i - startRow, j - startColumn, this.get(i, j));
	      }
	    }
	    return newMatrix;
	  }
	  subMatrixRow(indices, startColumn, endColumn) {
	    if (startColumn === undefined) startColumn = 0;
	    if (endColumn === undefined) endColumn = this.columns - 1;
	    if (startColumn > endColumn || startColumn < 0 || startColumn >= this.columns || endColumn < 0 || endColumn >= this.columns) {
	      throw new RangeError('Argument out of range');
	    }
	    let newMatrix = new Matrix$1(indices.length, endColumn - startColumn + 1);
	    for (let i = 0; i < indices.length; i++) {
	      for (let j = startColumn; j <= endColumn; j++) {
	        if (indices[i] < 0 || indices[i] >= this.rows) {
	          throw new RangeError(`Row index out of range: ${indices[i]}`);
	        }
	        newMatrix.set(i, j - startColumn, this.get(indices[i], j));
	      }
	    }
	    return newMatrix;
	  }
	  subMatrixColumn(indices, startRow, endRow) {
	    if (startRow === undefined) startRow = 0;
	    if (endRow === undefined) endRow = this.rows - 1;
	    if (startRow > endRow || startRow < 0 || startRow >= this.rows || endRow < 0 || endRow >= this.rows) {
	      throw new RangeError('Argument out of range');
	    }
	    let newMatrix = new Matrix$1(endRow - startRow + 1, indices.length);
	    for (let i = 0; i < indices.length; i++) {
	      for (let j = startRow; j <= endRow; j++) {
	        if (indices[i] < 0 || indices[i] >= this.columns) {
	          throw new RangeError(`Column index out of range: ${indices[i]}`);
	        }
	        newMatrix.set(j - startRow, i, this.get(j, indices[i]));
	      }
	    }
	    return newMatrix;
	  }
	  setSubMatrix(matrix, startRow, startColumn) {
	    matrix = Matrix$1.checkMatrix(matrix);
	    if (matrix.isEmpty()) {
	      return this;
	    }
	    let endRow = startRow + matrix.rows - 1;
	    let endColumn = startColumn + matrix.columns - 1;
	    checkRange(this, startRow, endRow, startColumn, endColumn);
	    for (let i = 0; i < matrix.rows; i++) {
	      for (let j = 0; j < matrix.columns; j++) {
	        this.set(startRow + i, startColumn + j, matrix.get(i, j));
	      }
	    }
	    return this;
	  }
	  selection(rowIndices, columnIndices) {
	    checkRowIndices(this, rowIndices);
	    checkColumnIndices(this, columnIndices);
	    let newMatrix = new Matrix$1(rowIndices.length, columnIndices.length);
	    for (let i = 0; i < rowIndices.length; i++) {
	      let rowIndex = rowIndices[i];
	      for (let j = 0; j < columnIndices.length; j++) {
	        let columnIndex = columnIndices[j];
	        newMatrix.set(i, j, this.get(rowIndex, columnIndex));
	      }
	    }
	    return newMatrix;
	  }
	  trace() {
	    let min = Math.min(this.rows, this.columns);
	    let trace = 0;
	    for (let i = 0; i < min; i++) {
	      trace += this.get(i, i);
	    }
	    return trace;
	  }
	  clone() {
	    return this.constructor.copy(this, new Matrix$1(this.rows, this.columns));
	  }

	  /**
	   * @template {AbstractMatrix} M
	   * @param {AbstractMatrix} from
	   * @param {M} to
	   * @return {M}
	   */
	  static copy(from, to) {
	    for (const [row, column, value] of from.entries()) {
	      to.set(row, column, value);
	    }
	    return to;
	  }
	  sum(by) {
	    switch (by) {
	      case 'row':
	        return sumByRow(this);
	      case 'column':
	        return sumByColumn(this);
	      case undefined:
	        return sumAll(this);
	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }
	  product(by) {
	    switch (by) {
	      case 'row':
	        return productByRow(this);
	      case 'column':
	        return productByColumn(this);
	      case undefined:
	        return productAll(this);
	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }
	  mean(by) {
	    const sum = this.sum(by);
	    switch (by) {
	      case 'row':
	        {
	          for (let i = 0; i < this.rows; i++) {
	            sum[i] /= this.columns;
	          }
	          return sum;
	        }
	      case 'column':
	        {
	          for (let i = 0; i < this.columns; i++) {
	            sum[i] /= this.rows;
	          }
	          return sum;
	        }
	      case undefined:
	        return sum / this.size;
	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }
	  variance(by, options = {}) {
	    if (typeof by === 'object') {
	      options = by;
	      by = undefined;
	    }
	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }
	    const {
	      unbiased = true,
	      mean = this.mean(by)
	    } = options;
	    if (typeof unbiased !== 'boolean') {
	      throw new TypeError('unbiased must be a boolean');
	    }
	    switch (by) {
	      case 'row':
	        {
	          if (!isAnyArray.isAnyArray(mean)) {
	            throw new TypeError('mean must be an array');
	          }
	          return varianceByRow(this, unbiased, mean);
	        }
	      case 'column':
	        {
	          if (!isAnyArray.isAnyArray(mean)) {
	            throw new TypeError('mean must be an array');
	          }
	          return varianceByColumn(this, unbiased, mean);
	        }
	      case undefined:
	        {
	          if (typeof mean !== 'number') {
	            throw new TypeError('mean must be a number');
	          }
	          return varianceAll(this, unbiased, mean);
	        }
	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }
	  standardDeviation(by, options) {
	    if (typeof by === 'object') {
	      options = by;
	      by = undefined;
	    }
	    const variance = this.variance(by, options);
	    if (by === undefined) {
	      return Math.sqrt(variance);
	    } else {
	      for (let i = 0; i < variance.length; i++) {
	        variance[i] = Math.sqrt(variance[i]);
	      }
	      return variance;
	    }
	  }
	  center(by, options = {}) {
	    if (typeof by === 'object') {
	      options = by;
	      by = undefined;
	    }
	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }
	    const {
	      center = this.mean(by)
	    } = options;
	    switch (by) {
	      case 'row':
	        {
	          if (!isAnyArray.isAnyArray(center)) {
	            throw new TypeError('center must be an array');
	          }
	          centerByRow(this, center);
	          return this;
	        }
	      case 'column':
	        {
	          if (!isAnyArray.isAnyArray(center)) {
	            throw new TypeError('center must be an array');
	          }
	          centerByColumn(this, center);
	          return this;
	        }
	      case undefined:
	        {
	          if (typeof center !== 'number') {
	            throw new TypeError('center must be a number');
	          }
	          centerAll(this, center);
	          return this;
	        }
	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }
	  scale(by, options = {}) {
	    if (typeof by === 'object') {
	      options = by;
	      by = undefined;
	    }
	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }
	    let scale = options.scale;
	    switch (by) {
	      case 'row':
	        {
	          if (scale === undefined) {
	            scale = getScaleByRow(this);
	          } else if (!isAnyArray.isAnyArray(scale)) {
	            throw new TypeError('scale must be an array');
	          }
	          scaleByRow(this, scale);
	          return this;
	        }
	      case 'column':
	        {
	          if (scale === undefined) {
	            scale = getScaleByColumn(this);
	          } else if (!isAnyArray.isAnyArray(scale)) {
	            throw new TypeError('scale must be an array');
	          }
	          scaleByColumn(this, scale);
	          return this;
	        }
	      case undefined:
	        {
	          if (scale === undefined) {
	            scale = getScaleAll(this);
	          } else if (typeof scale !== 'number') {
	            throw new TypeError('scale must be a number');
	          }
	          scaleAll(this, scale);
	          return this;
	        }
	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }
	  toString(options) {
	    return inspectMatrixWithOptions(this, options);
	  }
	  [Symbol.iterator]() {
	    return this.entries();
	  }

	  /**
	   * iterator from left to right, from top to bottom
	   * yield [row, column, value]
	   * @returns {Generator<[number, number, number], void, *>}
	   */
	  *entries() {
	    for (let row = 0; row < this.rows; row++) {
	      for (let col = 0; col < this.columns; col++) {
	        yield [row, col, this.get(row, col)];
	      }
	    }
	  }

	  /**
	   * iterator from left to right, from top to bottom
	   * yield value
	   * @returns {Generator<number, void, *>}
	   */
	  *values() {
	    for (let row = 0; row < this.rows; row++) {
	      for (let col = 0; col < this.columns; col++) {
	        yield this.get(row, col);
	      }
	    }
	  }
	}
	AbstractMatrix.prototype.klass = 'Matrix';
	if (typeof Symbol !== 'undefined') {
	  AbstractMatrix.prototype[Symbol.for('nodejs.util.inspect.custom')] = inspectMatrix;
	}
	function compareNumbers(a, b) {
	  return a - b;
	}
	function isArrayOfNumbers(array) {
	  return array.every(element => {
	    return typeof element === 'number';
	  });
	}

	// Synonyms
	AbstractMatrix.random = AbstractMatrix.rand;
	AbstractMatrix.randomInt = AbstractMatrix.randInt;
	AbstractMatrix.diagonal = AbstractMatrix.diag;
	AbstractMatrix.prototype.diagonal = AbstractMatrix.prototype.diag;
	AbstractMatrix.identity = AbstractMatrix.eye;
	AbstractMatrix.prototype.negate = AbstractMatrix.prototype.neg;
	AbstractMatrix.prototype.tensorProduct = AbstractMatrix.prototype.kroneckerProduct;
	class Matrix$1 extends AbstractMatrix {
	  /**
	   * @type {Float64Array[]}
	   */
	  data;

	  /**
	   * Init an empty matrix
	   * @param {number} nRows
	   * @param {number} nColumns
	   */
	  #initData(nRows, nColumns) {
	    this.data = [];
	    if (Number.isInteger(nColumns) && nColumns >= 0) {
	      for (let i = 0; i < nRows; i++) {
	        this.data.push(new Float64Array(nColumns));
	      }
	    } else {
	      throw new TypeError('nColumns must be a positive integer');
	    }
	    this.rows = nRows;
	    this.columns = nColumns;
	  }
	  constructor(nRows, nColumns) {
	    super();
	    if (Matrix$1.isMatrix(nRows)) {
	      this.#initData(nRows.rows, nRows.columns);
	      Matrix$1.copy(nRows, this);
	    } else if (Number.isInteger(nRows) && nRows >= 0) {
	      this.#initData(nRows, nColumns);
	    } else if (isAnyArray.isAnyArray(nRows)) {
	      // Copy the values from the 2D array
	      const arrayData = nRows;
	      nRows = arrayData.length;
	      nColumns = nRows ? arrayData[0].length : 0;
	      if (typeof nColumns !== 'number') {
	        throw new TypeError('Data must be a 2D array with at least one element');
	      }
	      this.data = [];
	      for (let i = 0; i < nRows; i++) {
	        if (arrayData[i].length !== nColumns) {
	          throw new RangeError('Inconsistent array dimensions');
	        }
	        if (!isArrayOfNumbers(arrayData[i])) {
	          throw new TypeError('Input data contains non-numeric values');
	        }
	        this.data.push(Float64Array.from(arrayData[i]));
	      }
	      this.rows = nRows;
	      this.columns = nColumns;
	    } else {
	      throw new TypeError('First argument must be a positive number or an array');
	    }
	  }
	  set(rowIndex, columnIndex, value) {
	    this.data[rowIndex][columnIndex] = value;
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    return this.data[rowIndex][columnIndex];
	  }
	  removeRow(index) {
	    checkRowIndex(this, index);
	    this.data.splice(index, 1);
	    this.rows -= 1;
	    return this;
	  }
	  addRow(index, array) {
	    if (array === undefined) {
	      array = index;
	      index = this.rows;
	    }
	    checkRowIndex(this, index, true);
	    array = Float64Array.from(checkRowVector(this, array));
	    this.data.splice(index, 0, array);
	    this.rows += 1;
	    return this;
	  }
	  removeColumn(index) {
	    checkColumnIndex(this, index);
	    for (let i = 0; i < this.rows; i++) {
	      const newRow = new Float64Array(this.columns - 1);
	      for (let j = 0; j < index; j++) {
	        newRow[j] = this.data[i][j];
	      }
	      for (let j = index + 1; j < this.columns; j++) {
	        newRow[j - 1] = this.data[i][j];
	      }
	      this.data[i] = newRow;
	    }
	    this.columns -= 1;
	    return this;
	  }
	  addColumn(index, array) {
	    if (typeof array === 'undefined') {
	      array = index;
	      index = this.columns;
	    }
	    checkColumnIndex(this, index, true);
	    array = checkColumnVector(this, array);
	    for (let i = 0; i < this.rows; i++) {
	      const newRow = new Float64Array(this.columns + 1);
	      let j = 0;
	      for (; j < index; j++) {
	        newRow[j] = this.data[i][j];
	      }
	      newRow[j++] = array[i];
	      for (; j < this.columns + 1; j++) {
	        newRow[j] = this.data[i][j - 1];
	      }
	      this.data[i] = newRow;
	    }
	    this.columns += 1;
	    return this;
	  }
	}
	installMathOperations(AbstractMatrix, Matrix$1);

	/**
	 * @typedef {0 | 1 | number | boolean} Mask
	 */

	class SymmetricMatrix extends AbstractMatrix {
	  /** @type {Matrix} */
	  #matrix;
	  get size() {
	    return this.#matrix.size;
	  }
	  get rows() {
	    return this.#matrix.rows;
	  }
	  get columns() {
	    return this.#matrix.columns;
	  }
	  get diagonalSize() {
	    return this.rows;
	  }

	  /**
	   * not the same as matrix.isSymmetric()
	   * Here is to check if it's instanceof SymmetricMatrix without bundling issues
	   *
	   * @param value
	   * @returns {boolean}
	   */
	  static isSymmetricMatrix(value) {
	    return Matrix$1.isMatrix(value) && value.klassType === 'SymmetricMatrix';
	  }

	  /**
	   * @param diagonalSize
	   * @return {SymmetricMatrix}
	   */
	  static zeros(diagonalSize) {
	    return new this(diagonalSize);
	  }

	  /**
	   * @param diagonalSize
	   * @return {SymmetricMatrix}
	   */
	  static ones(diagonalSize) {
	    return new this(diagonalSize).fill(1);
	  }

	  /**
	   * @param {number | AbstractMatrix | ArrayLike<ArrayLike<number>>} diagonalSize
	   * @return {this}
	   */
	  constructor(diagonalSize) {
	    super();
	    if (Matrix$1.isMatrix(diagonalSize)) {
	      if (!diagonalSize.isSymmetric()) {
	        throw new TypeError('not symmetric data');
	      }
	      this.#matrix = Matrix$1.copy(diagonalSize, new Matrix$1(diagonalSize.rows, diagonalSize.rows));
	    } else if (Number.isInteger(diagonalSize) && diagonalSize >= 0) {
	      this.#matrix = new Matrix$1(diagonalSize, diagonalSize);
	    } else {
	      this.#matrix = new Matrix$1(diagonalSize);
	      if (!this.isSymmetric()) {
	        throw new TypeError('not symmetric data');
	      }
	    }
	  }
	  clone() {
	    const matrix = new SymmetricMatrix(this.diagonalSize);
	    for (const [row, col, value] of this.upperRightEntries()) {
	      matrix.set(row, col, value);
	    }
	    return matrix;
	  }
	  toMatrix() {
	    return new Matrix$1(this);
	  }
	  get(rowIndex, columnIndex) {
	    return this.#matrix.get(rowIndex, columnIndex);
	  }
	  set(rowIndex, columnIndex, value) {
	    // symmetric set
	    this.#matrix.set(rowIndex, columnIndex, value);
	    this.#matrix.set(columnIndex, rowIndex, value);
	    return this;
	  }
	  removeCross(index) {
	    // symmetric remove side
	    this.#matrix.removeRow(index);
	    this.#matrix.removeColumn(index);
	    return this;
	  }
	  addCross(index, array) {
	    if (array === undefined) {
	      array = index;
	      index = this.diagonalSize;
	    }
	    const row = array.slice();
	    row.splice(index, 1);
	    this.#matrix.addRow(index, row);
	    this.#matrix.addColumn(index, array);
	    return this;
	  }

	  /**
	   * @param {Mask[]} mask
	   */
	  applyMask(mask) {
	    if (mask.length !== this.diagonalSize) {
	      throw new RangeError('Mask size do not match with matrix size');
	    }

	    // prepare sides to remove from matrix from mask
	    /** @type {number[]} */
	    const sidesToRemove = [];
	    for (const [index, passthroughs] of mask.entries()) {
	      if (passthroughs) continue;
	      sidesToRemove.push(index);
	    }
	    // to remove from highest to lowest for no mutation shifting
	    sidesToRemove.reverse();

	    // remove sides
	    for (const sideIndex of sidesToRemove) {
	      this.removeCross(sideIndex);
	    }
	    return this;
	  }

	  /**
	   * Compact format upper-right corner of matrix
	   * iterate from left to right, from top to bottom.
	   *
	   * ```
	   *   A B C D
	   * A 1 2 3 4
	   * B 2 5 6 7
	   * C 3 6 8 9
	   * D 4 7 9 10
	   * ```
	   *
	   * will return compact 1D array `[1, 2, 3, 4, 5, 6, 7, 8, 9, 10]`
	   *
	   * length is S(i=0, n=sideSize) => 10 for a 4 sideSized matrix
	   *
	   * @returns {number[]}
	   */
	  toCompact() {
	    const {
	      diagonalSize
	    } = this;

	    /** @type {number[]} */
	    const compact = new Array(diagonalSize * (diagonalSize + 1) / 2);
	    for (let col = 0, row = 0, index = 0; index < compact.length; index++) {
	      compact[index] = this.get(row, col);
	      if (++col >= diagonalSize) col = ++row;
	    }
	    return compact;
	  }

	  /**
	   * @param {number[]} compact
	   * @return {SymmetricMatrix}
	   */
	  static fromCompact(compact) {
	    const compactSize = compact.length;
	    // compactSize = (sideSize * (sideSize + 1)) / 2
	    // https://mathsolver.microsoft.com/fr/solve-problem/y%20%3D%20%20x%20%60cdot%20%20%20%60frac%7B%20%20%60left(%20x%2B1%20%20%60right)%20%20%20%20%7D%7B%202%20%20%7D
	    // sideSize = (Sqrt(8 × compactSize + 1) - 1) / 2
	    const diagonalSize = (Math.sqrt(8 * compactSize + 1) - 1) / 2;
	    if (!Number.isInteger(diagonalSize)) {
	      throw new TypeError(`This array is not a compact representation of a Symmetric Matrix, ${JSON.stringify(compact)}`);
	    }
	    const matrix = new SymmetricMatrix(diagonalSize);
	    for (let col = 0, row = 0, index = 0; index < compactSize; index++) {
	      matrix.set(col, row, compact[index]);
	      if (++col >= diagonalSize) col = ++row;
	    }
	    return matrix;
	  }

	  /**
	   * half iterator upper-right-corner from left to right, from top to bottom
	   * yield [row, column, value]
	   *
	   * @returns {Generator<[number, number, number], void, *>}
	   */
	  *upperRightEntries() {
	    for (let row = 0, col = 0; row < this.diagonalSize; void 0) {
	      const value = this.get(row, col);
	      yield [row, col, value];

	      // at the end of row, move cursor to next row at diagonal position
	      if (++col >= this.diagonalSize) col = ++row;
	    }
	  }

	  /**
	   * half iterator upper-right-corner from left to right, from top to bottom
	   * yield value
	   *
	   * @returns {Generator<[number, number, number], void, *>}
	   */
	  *upperRightValues() {
	    for (let row = 0, col = 0; row < this.diagonalSize; void 0) {
	      const value = this.get(row, col);
	      yield value;

	      // at the end of row, move cursor to next row at diagonal position
	      if (++col >= this.diagonalSize) col = ++row;
	    }
	  }
	}
	SymmetricMatrix.prototype.klassType = 'SymmetricMatrix';
	class DistanceMatrix extends SymmetricMatrix {
	  /**
	   * not the same as matrix.isSymmetric()
	   * Here is to check if it's instanceof SymmetricMatrix without bundling issues
	   *
	   * @param value
	   * @returns {boolean}
	   */
	  static isDistanceMatrix(value) {
	    return SymmetricMatrix.isSymmetricMatrix(value) && value.klassSubType === 'DistanceMatrix';
	  }
	  constructor(sideSize) {
	    super(sideSize);
	    if (!this.isDistance()) {
	      throw new TypeError('Provided arguments do no produce a distance matrix');
	    }
	  }
	  set(rowIndex, columnIndex, value) {
	    // distance matrix diagonal is 0
	    if (rowIndex === columnIndex) value = 0;
	    return super.set(rowIndex, columnIndex, value);
	  }
	  addCross(index, array) {
	    if (array === undefined) {
	      array = index;
	      index = this.diagonalSize;
	    }

	    // ensure distance
	    array = array.slice();
	    array[index] = 0;
	    return super.addCross(index, array);
	  }
	  toSymmetricMatrix() {
	    return new SymmetricMatrix(this);
	  }
	  clone() {
	    const matrix = new DistanceMatrix(this.diagonalSize);
	    for (const [row, col, value] of this.upperRightEntries()) {
	      if (row === col) continue;
	      matrix.set(row, col, value);
	    }
	    return matrix;
	  }

	  /**
	   * Compact format upper-right corner of matrix
	   * no diagonal (only zeros)
	   * iterable from left to right, from top to bottom.
	   *
	   * ```
	   *   A B C D
	   * A 0 1 2 3
	   * B 1 0 4 5
	   * C 2 4 0 6
	   * D 3 5 6 0
	   * ```
	   *
	   * will return compact 1D array `[1, 2, 3, 4, 5, 6]`
	   *
	   * length is S(i=0, n=sideSize-1) => 6 for a 4 side sized matrix
	   *
	   * @returns {number[]}
	   */
	  toCompact() {
	    const {
	      diagonalSize
	    } = this;
	    const compactLength = (diagonalSize - 1) * diagonalSize / 2;

	    /** @type {number[]} */
	    const compact = new Array(compactLength);
	    for (let col = 1, row = 0, index = 0; index < compact.length; index++) {
	      compact[index] = this.get(row, col);
	      if (++col >= diagonalSize) col = ++row + 1;
	    }
	    return compact;
	  }

	  /**
	   * @param {number[]} compact
	   */
	  static fromCompact(compact) {
	    const compactSize = compact.length;
	    // compactSize = (sideSize * (sideSize - 1)) / 2
	    // sideSize = (Sqrt(8 × compactSize + 1) + 1) / 2
	    const diagonalSize = (Math.sqrt(8 * compactSize + 1) + 1) / 2;
	    if (!Number.isInteger(diagonalSize)) {
	      throw new TypeError(`This array is not a compact representation of a DistanceMatrix, ${JSON.stringify(compact)}`);
	    }
	    const matrix = new this(diagonalSize);
	    for (let col = 1, row = 0, index = 0; index < compactSize; index++) {
	      matrix.set(col, row, compact[index]);
	      if (++col >= diagonalSize) col = ++row + 1;
	    }
	    return matrix;
	  }
	}
	DistanceMatrix.prototype.klassSubType = 'DistanceMatrix';
	class BaseView extends AbstractMatrix {
	  constructor(matrix, rows, columns) {
	    super();
	    this.matrix = matrix;
	    this.rows = rows;
	    this.columns = columns;
	  }
	}
	class MatrixColumnView extends BaseView {
	  constructor(matrix, column) {
	    checkColumnIndex(matrix, column);
	    super(matrix, matrix.rows, 1);
	    this.column = column;
	  }
	  set(rowIndex, columnIndex, value) {
	    this.matrix.set(rowIndex, this.column, value);
	    return this;
	  }
	  get(rowIndex) {
	    return this.matrix.get(rowIndex, this.column);
	  }
	}
	class MatrixColumnSelectionView extends BaseView {
	  constructor(matrix, columnIndices) {
	    checkColumnIndices(matrix, columnIndices);
	    super(matrix, matrix.rows, columnIndices.length);
	    this.columnIndices = columnIndices;
	  }
	  set(rowIndex, columnIndex, value) {
	    this.matrix.set(rowIndex, this.columnIndices[columnIndex], value);
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    return this.matrix.get(rowIndex, this.columnIndices[columnIndex]);
	  }
	}
	class MatrixFlipColumnView extends BaseView {
	  constructor(matrix) {
	    super(matrix, matrix.rows, matrix.columns);
	  }
	  set(rowIndex, columnIndex, value) {
	    this.matrix.set(rowIndex, this.columns - columnIndex - 1, value);
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    return this.matrix.get(rowIndex, this.columns - columnIndex - 1);
	  }
	}
	class MatrixFlipRowView extends BaseView {
	  constructor(matrix) {
	    super(matrix, matrix.rows, matrix.columns);
	  }
	  set(rowIndex, columnIndex, value) {
	    this.matrix.set(this.rows - rowIndex - 1, columnIndex, value);
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    return this.matrix.get(this.rows - rowIndex - 1, columnIndex);
	  }
	}
	class MatrixRowView extends BaseView {
	  constructor(matrix, row) {
	    checkRowIndex(matrix, row);
	    super(matrix, 1, matrix.columns);
	    this.row = row;
	  }
	  set(rowIndex, columnIndex, value) {
	    this.matrix.set(this.row, columnIndex, value);
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    return this.matrix.get(this.row, columnIndex);
	  }
	}
	class MatrixRowSelectionView extends BaseView {
	  constructor(matrix, rowIndices) {
	    checkRowIndices(matrix, rowIndices);
	    super(matrix, rowIndices.length, matrix.columns);
	    this.rowIndices = rowIndices;
	  }
	  set(rowIndex, columnIndex, value) {
	    this.matrix.set(this.rowIndices[rowIndex], columnIndex, value);
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    return this.matrix.get(this.rowIndices[rowIndex], columnIndex);
	  }
	}
	class MatrixSelectionView extends BaseView {
	  constructor(matrix, rowIndices, columnIndices) {
	    checkRowIndices(matrix, rowIndices);
	    checkColumnIndices(matrix, columnIndices);
	    super(matrix, rowIndices.length, columnIndices.length);
	    this.rowIndices = rowIndices;
	    this.columnIndices = columnIndices;
	  }
	  set(rowIndex, columnIndex, value) {
	    this.matrix.set(this.rowIndices[rowIndex], this.columnIndices[columnIndex], value);
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    return this.matrix.get(this.rowIndices[rowIndex], this.columnIndices[columnIndex]);
	  }
	}
	class MatrixSubView extends BaseView {
	  constructor(matrix, startRow, endRow, startColumn, endColumn) {
	    checkRange(matrix, startRow, endRow, startColumn, endColumn);
	    super(matrix, endRow - startRow + 1, endColumn - startColumn + 1);
	    this.startRow = startRow;
	    this.startColumn = startColumn;
	  }
	  set(rowIndex, columnIndex, value) {
	    this.matrix.set(this.startRow + rowIndex, this.startColumn + columnIndex, value);
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    return this.matrix.get(this.startRow + rowIndex, this.startColumn + columnIndex);
	  }
	}
	class MatrixTransposeView extends BaseView {
	  constructor(matrix) {
	    super(matrix, matrix.columns, matrix.rows);
	  }
	  set(rowIndex, columnIndex, value) {
	    this.matrix.set(columnIndex, rowIndex, value);
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    return this.matrix.get(columnIndex, rowIndex);
	  }
	}
	class WrapperMatrix1D extends AbstractMatrix {
	  constructor(data, options = {}) {
	    const {
	      rows = 1
	    } = options;
	    if (data.length % rows !== 0) {
	      throw new Error('the data length is not divisible by the number of rows');
	    }
	    super();
	    this.rows = rows;
	    this.columns = data.length / rows;
	    this.data = data;
	  }
	  set(rowIndex, columnIndex, value) {
	    let index = this._calculateIndex(rowIndex, columnIndex);
	    this.data[index] = value;
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    let index = this._calculateIndex(rowIndex, columnIndex);
	    return this.data[index];
	  }
	  _calculateIndex(row, column) {
	    return row * this.columns + column;
	  }
	}
	class WrapperMatrix2D extends AbstractMatrix {
	  constructor(data) {
	    super();
	    this.data = data;
	    this.rows = data.length;
	    this.columns = data[0].length;
	  }
	  set(rowIndex, columnIndex, value) {
	    this.data[rowIndex][columnIndex] = value;
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    return this.data[rowIndex][columnIndex];
	  }
	}
	function wrap(array, options) {
	  if (isAnyArray.isAnyArray(array)) {
	    if (array[0] && isAnyArray.isAnyArray(array[0])) {
	      return new WrapperMatrix2D(array);
	    } else {
	      return new WrapperMatrix1D(array, options);
	    }
	  } else {
	    throw new Error('the argument is not an array');
	  }
	}
	class LuDecomposition {
	  constructor(matrix) {
	    matrix = WrapperMatrix2D.checkMatrix(matrix);
	    let lu = matrix.clone();
	    let rows = lu.rows;
	    let columns = lu.columns;
	    let pivotVector = new Float64Array(rows);
	    let pivotSign = 1;
	    let i, j, k, p, s, t, v;
	    let LUcolj, kmax;
	    for (i = 0; i < rows; i++) {
	      pivotVector[i] = i;
	    }
	    LUcolj = new Float64Array(rows);
	    for (j = 0; j < columns; j++) {
	      for (i = 0; i < rows; i++) {
	        LUcolj[i] = lu.get(i, j);
	      }
	      for (i = 0; i < rows; i++) {
	        kmax = Math.min(i, j);
	        s = 0;
	        for (k = 0; k < kmax; k++) {
	          s += lu.get(i, k) * LUcolj[k];
	        }
	        LUcolj[i] -= s;
	        lu.set(i, j, LUcolj[i]);
	      }
	      p = j;
	      for (i = j + 1; i < rows; i++) {
	        if (Math.abs(LUcolj[i]) > Math.abs(LUcolj[p])) {
	          p = i;
	        }
	      }
	      if (p !== j) {
	        for (k = 0; k < columns; k++) {
	          t = lu.get(p, k);
	          lu.set(p, k, lu.get(j, k));
	          lu.set(j, k, t);
	        }
	        v = pivotVector[p];
	        pivotVector[p] = pivotVector[j];
	        pivotVector[j] = v;
	        pivotSign = -pivotSign;
	      }
	      if (j < rows && lu.get(j, j) !== 0) {
	        for (i = j + 1; i < rows; i++) {
	          lu.set(i, j, lu.get(i, j) / lu.get(j, j));
	        }
	      }
	    }
	    this.LU = lu;
	    this.pivotVector = pivotVector;
	    this.pivotSign = pivotSign;
	  }
	  isSingular() {
	    let data = this.LU;
	    let col = data.columns;
	    for (let j = 0; j < col; j++) {
	      if (data.get(j, j) === 0) {
	        return true;
	      }
	    }
	    return false;
	  }
	  solve(value) {
	    value = Matrix$1.checkMatrix(value);
	    let lu = this.LU;
	    let rows = lu.rows;
	    if (rows !== value.rows) {
	      throw new Error('Invalid matrix dimensions');
	    }
	    if (this.isSingular()) {
	      throw new Error('LU matrix is singular');
	    }
	    let count = value.columns;
	    let X = value.subMatrixRow(this.pivotVector, 0, count - 1);
	    let columns = lu.columns;
	    let i, j, k;
	    for (k = 0; k < columns; k++) {
	      for (i = k + 1; i < columns; i++) {
	        for (j = 0; j < count; j++) {
	          X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
	        }
	      }
	    }
	    for (k = columns - 1; k >= 0; k--) {
	      for (j = 0; j < count; j++) {
	        X.set(k, j, X.get(k, j) / lu.get(k, k));
	      }
	      for (i = 0; i < k; i++) {
	        for (j = 0; j < count; j++) {
	          X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
	        }
	      }
	    }
	    return X;
	  }
	  get determinant() {
	    let data = this.LU;
	    if (!data.isSquare()) {
	      throw new Error('Matrix must be square');
	    }
	    let determinant = this.pivotSign;
	    let col = data.columns;
	    for (let j = 0; j < col; j++) {
	      determinant *= data.get(j, j);
	    }
	    return determinant;
	  }
	  get lowerTriangularMatrix() {
	    let data = this.LU;
	    let rows = data.rows;
	    let columns = data.columns;
	    let X = new Matrix$1(rows, columns);
	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        if (i > j) {
	          X.set(i, j, data.get(i, j));
	        } else if (i === j) {
	          X.set(i, j, 1);
	        } else {
	          X.set(i, j, 0);
	        }
	      }
	    }
	    return X;
	  }
	  get upperTriangularMatrix() {
	    let data = this.LU;
	    let rows = data.rows;
	    let columns = data.columns;
	    let X = new Matrix$1(rows, columns);
	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        if (i <= j) {
	          X.set(i, j, data.get(i, j));
	        } else {
	          X.set(i, j, 0);
	        }
	      }
	    }
	    return X;
	  }
	  get pivotPermutationVector() {
	    return Array.from(this.pivotVector);
	  }
	}
	function hypotenuse(a, b) {
	  let r = 0;
	  if (Math.abs(a) > Math.abs(b)) {
	    r = b / a;
	    return Math.abs(a) * Math.sqrt(1 + r * r);
	  }
	  if (b !== 0) {
	    r = a / b;
	    return Math.abs(b) * Math.sqrt(1 + r * r);
	  }
	  return 0;
	}
	class QrDecomposition {
	  constructor(value) {
	    value = WrapperMatrix2D.checkMatrix(value);
	    let qr = value.clone();
	    let m = value.rows;
	    let n = value.columns;
	    let rdiag = new Float64Array(n);
	    let i, j, k, s;
	    for (k = 0; k < n; k++) {
	      let nrm = 0;
	      for (i = k; i < m; i++) {
	        nrm = hypotenuse(nrm, qr.get(i, k));
	      }
	      if (nrm !== 0) {
	        if (qr.get(k, k) < 0) {
	          nrm = -nrm;
	        }
	        for (i = k; i < m; i++) {
	          qr.set(i, k, qr.get(i, k) / nrm);
	        }
	        qr.set(k, k, qr.get(k, k) + 1);
	        for (j = k + 1; j < n; j++) {
	          s = 0;
	          for (i = k; i < m; i++) {
	            s += qr.get(i, k) * qr.get(i, j);
	          }
	          s = -s / qr.get(k, k);
	          for (i = k; i < m; i++) {
	            qr.set(i, j, qr.get(i, j) + s * qr.get(i, k));
	          }
	        }
	      }
	      rdiag[k] = -nrm;
	    }
	    this.QR = qr;
	    this.Rdiag = rdiag;
	  }
	  solve(value) {
	    value = Matrix$1.checkMatrix(value);
	    let qr = this.QR;
	    let m = qr.rows;
	    if (value.rows !== m) {
	      throw new Error('Matrix row dimensions must agree');
	    }
	    if (!this.isFullRank()) {
	      throw new Error('Matrix is rank deficient');
	    }
	    let count = value.columns;
	    let X = value.clone();
	    let n = qr.columns;
	    let i, j, k, s;
	    for (k = 0; k < n; k++) {
	      for (j = 0; j < count; j++) {
	        s = 0;
	        for (i = k; i < m; i++) {
	          s += qr.get(i, k) * X.get(i, j);
	        }
	        s = -s / qr.get(k, k);
	        for (i = k; i < m; i++) {
	          X.set(i, j, X.get(i, j) + s * qr.get(i, k));
	        }
	      }
	    }
	    for (k = n - 1; k >= 0; k--) {
	      for (j = 0; j < count; j++) {
	        X.set(k, j, X.get(k, j) / this.Rdiag[k]);
	      }
	      for (i = 0; i < k; i++) {
	        for (j = 0; j < count; j++) {
	          X.set(i, j, X.get(i, j) - X.get(k, j) * qr.get(i, k));
	        }
	      }
	    }
	    return X.subMatrix(0, n - 1, 0, count - 1);
	  }
	  isFullRank() {
	    let columns = this.QR.columns;
	    for (let i = 0; i < columns; i++) {
	      if (this.Rdiag[i] === 0) {
	        return false;
	      }
	    }
	    return true;
	  }
	  get upperTriangularMatrix() {
	    let qr = this.QR;
	    let n = qr.columns;
	    let X = new Matrix$1(n, n);
	    let i, j;
	    for (i = 0; i < n; i++) {
	      for (j = 0; j < n; j++) {
	        if (i < j) {
	          X.set(i, j, qr.get(i, j));
	        } else if (i === j) {
	          X.set(i, j, this.Rdiag[i]);
	        } else {
	          X.set(i, j, 0);
	        }
	      }
	    }
	    return X;
	  }
	  get orthogonalMatrix() {
	    let qr = this.QR;
	    let rows = qr.rows;
	    let columns = qr.columns;
	    let X = new Matrix$1(rows, columns);
	    let i, j, k, s;
	    for (k = columns - 1; k >= 0; k--) {
	      for (i = 0; i < rows; i++) {
	        X.set(i, k, 0);
	      }
	      X.set(k, k, 1);
	      for (j = k; j < columns; j++) {
	        if (qr.get(k, k) !== 0) {
	          s = 0;
	          for (i = k; i < rows; i++) {
	            s += qr.get(i, k) * X.get(i, j);
	          }
	          s = -s / qr.get(k, k);
	          for (i = k; i < rows; i++) {
	            X.set(i, j, X.get(i, j) + s * qr.get(i, k));
	          }
	        }
	      }
	    }
	    return X;
	  }
	}
	class SingularValueDecomposition {
	  constructor(value, options = {}) {
	    value = WrapperMatrix2D.checkMatrix(value);
	    if (value.isEmpty()) {
	      throw new Error('Matrix must be non-empty');
	    }
	    let m = value.rows;
	    let n = value.columns;
	    const {
	      computeLeftSingularVectors = true,
	      computeRightSingularVectors = true,
	      autoTranspose = false
	    } = options;
	    let wantu = Boolean(computeLeftSingularVectors);
	    let wantv = Boolean(computeRightSingularVectors);
	    let swapped = false;
	    let a;
	    if (m < n) {
	      if (!autoTranspose) {
	        a = value.clone();
	        // eslint-disable-next-line no-console
	        console.warn('Computing SVD on a matrix with more columns than rows. Consider enabling autoTranspose');
	      } else {
	        a = value.transpose();
	        m = a.rows;
	        n = a.columns;
	        swapped = true;
	        let aux = wantu;
	        wantu = wantv;
	        wantv = aux;
	      }
	    } else {
	      a = value.clone();
	    }
	    let nu = Math.min(m, n);
	    let ni = Math.min(m + 1, n);
	    let s = new Float64Array(ni);
	    let U = new Matrix$1(m, nu);
	    let V = new Matrix$1(n, n);
	    let e = new Float64Array(n);
	    let work = new Float64Array(m);
	    let si = new Float64Array(ni);
	    for (let i = 0; i < ni; i++) si[i] = i;
	    let nct = Math.min(m - 1, n);
	    let nrt = Math.max(0, Math.min(n - 2, m));
	    let mrc = Math.max(nct, nrt);
	    for (let k = 0; k < mrc; k++) {
	      if (k < nct) {
	        s[k] = 0;
	        for (let i = k; i < m; i++) {
	          s[k] = hypotenuse(s[k], a.get(i, k));
	        }
	        if (s[k] !== 0) {
	          if (a.get(k, k) < 0) {
	            s[k] = -s[k];
	          }
	          for (let i = k; i < m; i++) {
	            a.set(i, k, a.get(i, k) / s[k]);
	          }
	          a.set(k, k, a.get(k, k) + 1);
	        }
	        s[k] = -s[k];
	      }
	      for (let j = k + 1; j < n; j++) {
	        if (k < nct && s[k] !== 0) {
	          let t = 0;
	          for (let i = k; i < m; i++) {
	            t += a.get(i, k) * a.get(i, j);
	          }
	          t = -t / a.get(k, k);
	          for (let i = k; i < m; i++) {
	            a.set(i, j, a.get(i, j) + t * a.get(i, k));
	          }
	        }
	        e[j] = a.get(k, j);
	      }
	      if (wantu && k < nct) {
	        for (let i = k; i < m; i++) {
	          U.set(i, k, a.get(i, k));
	        }
	      }
	      if (k < nrt) {
	        e[k] = 0;
	        for (let i = k + 1; i < n; i++) {
	          e[k] = hypotenuse(e[k], e[i]);
	        }
	        if (e[k] !== 0) {
	          if (e[k + 1] < 0) {
	            e[k] = 0 - e[k];
	          }
	          for (let i = k + 1; i < n; i++) {
	            e[i] /= e[k];
	          }
	          e[k + 1] += 1;
	        }
	        e[k] = -e[k];
	        if (k + 1 < m && e[k] !== 0) {
	          for (let i = k + 1; i < m; i++) {
	            work[i] = 0;
	          }
	          for (let i = k + 1; i < m; i++) {
	            for (let j = k + 1; j < n; j++) {
	              work[i] += e[j] * a.get(i, j);
	            }
	          }
	          for (let j = k + 1; j < n; j++) {
	            let t = -e[j] / e[k + 1];
	            for (let i = k + 1; i < m; i++) {
	              a.set(i, j, a.get(i, j) + t * work[i]);
	            }
	          }
	        }
	        if (wantv) {
	          for (let i = k + 1; i < n; i++) {
	            V.set(i, k, e[i]);
	          }
	        }
	      }
	    }
	    let p = Math.min(n, m + 1);
	    if (nct < n) {
	      s[nct] = a.get(nct, nct);
	    }
	    if (m < p) {
	      s[p - 1] = 0;
	    }
	    if (nrt + 1 < p) {
	      e[nrt] = a.get(nrt, p - 1);
	    }
	    e[p - 1] = 0;
	    if (wantu) {
	      for (let j = nct; j < nu; j++) {
	        for (let i = 0; i < m; i++) {
	          U.set(i, j, 0);
	        }
	        U.set(j, j, 1);
	      }
	      for (let k = nct - 1; k >= 0; k--) {
	        if (s[k] !== 0) {
	          for (let j = k + 1; j < nu; j++) {
	            let t = 0;
	            for (let i = k; i < m; i++) {
	              t += U.get(i, k) * U.get(i, j);
	            }
	            t = -t / U.get(k, k);
	            for (let i = k; i < m; i++) {
	              U.set(i, j, U.get(i, j) + t * U.get(i, k));
	            }
	          }
	          for (let i = k; i < m; i++) {
	            U.set(i, k, -U.get(i, k));
	          }
	          U.set(k, k, 1 + U.get(k, k));
	          for (let i = 0; i < k - 1; i++) {
	            U.set(i, k, 0);
	          }
	        } else {
	          for (let i = 0; i < m; i++) {
	            U.set(i, k, 0);
	          }
	          U.set(k, k, 1);
	        }
	      }
	    }
	    if (wantv) {
	      for (let k = n - 1; k >= 0; k--) {
	        if (k < nrt && e[k] !== 0) {
	          for (let j = k + 1; j < n; j++) {
	            let t = 0;
	            for (let i = k + 1; i < n; i++) {
	              t += V.get(i, k) * V.get(i, j);
	            }
	            t = -t / V.get(k + 1, k);
	            for (let i = k + 1; i < n; i++) {
	              V.set(i, j, V.get(i, j) + t * V.get(i, k));
	            }
	          }
	        }
	        for (let i = 0; i < n; i++) {
	          V.set(i, k, 0);
	        }
	        V.set(k, k, 1);
	      }
	    }
	    let pp = p - 1;
	    let eps = Number.EPSILON;
	    while (p > 0) {
	      let k, kase;
	      for (k = p - 2; k >= -1; k--) {
	        if (k === -1) {
	          break;
	        }
	        const alpha = Number.MIN_VALUE + eps * Math.abs(s[k] + Math.abs(s[k + 1]));
	        if (Math.abs(e[k]) <= alpha || Number.isNaN(e[k])) {
	          e[k] = 0;
	          break;
	        }
	      }
	      if (k === p - 2) {
	        kase = 4;
	      } else {
	        let ks;
	        for (ks = p - 1; ks >= k; ks--) {
	          if (ks === k) {
	            break;
	          }
	          let t = (ks !== p ? Math.abs(e[ks]) : 0) + (ks !== k + 1 ? Math.abs(e[ks - 1]) : 0);
	          if (Math.abs(s[ks]) <= eps * t) {
	            s[ks] = 0;
	            break;
	          }
	        }
	        if (ks === k) {
	          kase = 3;
	        } else if (ks === p - 1) {
	          kase = 1;
	        } else {
	          kase = 2;
	          k = ks;
	        }
	      }
	      k++;
	      switch (kase) {
	        case 1:
	          {
	            let f = e[p - 2];
	            e[p - 2] = 0;
	            for (let j = p - 2; j >= k; j--) {
	              let t = hypotenuse(s[j], f);
	              let cs = s[j] / t;
	              let sn = f / t;
	              s[j] = t;
	              if (j !== k) {
	                f = -sn * e[j - 1];
	                e[j - 1] = cs * e[j - 1];
	              }
	              if (wantv) {
	                for (let i = 0; i < n; i++) {
	                  t = cs * V.get(i, j) + sn * V.get(i, p - 1);
	                  V.set(i, p - 1, -sn * V.get(i, j) + cs * V.get(i, p - 1));
	                  V.set(i, j, t);
	                }
	              }
	            }
	            break;
	          }
	        case 2:
	          {
	            let f = e[k - 1];
	            e[k - 1] = 0;
	            for (let j = k; j < p; j++) {
	              let t = hypotenuse(s[j], f);
	              let cs = s[j] / t;
	              let sn = f / t;
	              s[j] = t;
	              f = -sn * e[j];
	              e[j] = cs * e[j];
	              if (wantu) {
	                for (let i = 0; i < m; i++) {
	                  t = cs * U.get(i, j) + sn * U.get(i, k - 1);
	                  U.set(i, k - 1, -sn * U.get(i, j) + cs * U.get(i, k - 1));
	                  U.set(i, j, t);
	                }
	              }
	            }
	            break;
	          }
	        case 3:
	          {
	            const scale = Math.max(Math.abs(s[p - 1]), Math.abs(s[p - 2]), Math.abs(e[p - 2]), Math.abs(s[k]), Math.abs(e[k]));
	            const sp = s[p - 1] / scale;
	            const spm1 = s[p - 2] / scale;
	            const epm1 = e[p - 2] / scale;
	            const sk = s[k] / scale;
	            const ek = e[k] / scale;
	            const b = ((spm1 + sp) * (spm1 - sp) + epm1 * epm1) / 2;
	            const c = sp * epm1 * (sp * epm1);
	            let shift = 0;
	            if (b !== 0 || c !== 0) {
	              if (b < 0) {
	                shift = 0 - Math.sqrt(b * b + c);
	              } else {
	                shift = Math.sqrt(b * b + c);
	              }
	              shift = c / (b + shift);
	            }
	            let f = (sk + sp) * (sk - sp) + shift;
	            let g = sk * ek;
	            for (let j = k; j < p - 1; j++) {
	              let t = hypotenuse(f, g);
	              if (t === 0) t = Number.MIN_VALUE;
	              let cs = f / t;
	              let sn = g / t;
	              if (j !== k) {
	                e[j - 1] = t;
	              }
	              f = cs * s[j] + sn * e[j];
	              e[j] = cs * e[j] - sn * s[j];
	              g = sn * s[j + 1];
	              s[j + 1] = cs * s[j + 1];
	              if (wantv) {
	                for (let i = 0; i < n; i++) {
	                  t = cs * V.get(i, j) + sn * V.get(i, j + 1);
	                  V.set(i, j + 1, -sn * V.get(i, j) + cs * V.get(i, j + 1));
	                  V.set(i, j, t);
	                }
	              }
	              t = hypotenuse(f, g);
	              if (t === 0) t = Number.MIN_VALUE;
	              cs = f / t;
	              sn = g / t;
	              s[j] = t;
	              f = cs * e[j] + sn * s[j + 1];
	              s[j + 1] = -sn * e[j] + cs * s[j + 1];
	              g = sn * e[j + 1];
	              e[j + 1] = cs * e[j + 1];
	              if (wantu && j < m - 1) {
	                for (let i = 0; i < m; i++) {
	                  t = cs * U.get(i, j) + sn * U.get(i, j + 1);
	                  U.set(i, j + 1, -sn * U.get(i, j) + cs * U.get(i, j + 1));
	                  U.set(i, j, t);
	                }
	              }
	            }
	            e[p - 2] = f;
	            break;
	          }
	        case 4:
	          {
	            if (s[k] <= 0) {
	              s[k] = s[k] < 0 ? -s[k] : 0;
	              if (wantv) {
	                for (let i = 0; i <= pp; i++) {
	                  V.set(i, k, -V.get(i, k));
	                }
	              }
	            }
	            while (k < pp) {
	              if (s[k] >= s[k + 1]) {
	                break;
	              }
	              let t = s[k];
	              s[k] = s[k + 1];
	              s[k + 1] = t;
	              if (wantv && k < n - 1) {
	                for (let i = 0; i < n; i++) {
	                  t = V.get(i, k + 1);
	                  V.set(i, k + 1, V.get(i, k));
	                  V.set(i, k, t);
	                }
	              }
	              if (wantu && k < m - 1) {
	                for (let i = 0; i < m; i++) {
	                  t = U.get(i, k + 1);
	                  U.set(i, k + 1, U.get(i, k));
	                  U.set(i, k, t);
	                }
	              }
	              k++;
	            }
	            p--;
	            break;
	          }
	        // no default
	      }
	    }
	    if (swapped) {
	      let tmp = V;
	      V = U;
	      U = tmp;
	    }
	    this.m = m;
	    this.n = n;
	    this.s = s;
	    this.U = U;
	    this.V = V;
	  }
	  solve(value) {
	    let Y = value;
	    let e = this.threshold;
	    let scols = this.s.length;
	    let Ls = Matrix$1.zeros(scols, scols);
	    for (let i = 0; i < scols; i++) {
	      if (Math.abs(this.s[i]) <= e) {
	        Ls.set(i, i, 0);
	      } else {
	        Ls.set(i, i, 1 / this.s[i]);
	      }
	    }
	    let U = this.U;
	    let V = this.rightSingularVectors;
	    let VL = V.mmul(Ls);
	    let vrows = V.rows;
	    let urows = U.rows;
	    let VLU = Matrix$1.zeros(vrows, urows);
	    for (let i = 0; i < vrows; i++) {
	      for (let j = 0; j < urows; j++) {
	        let sum = 0;
	        for (let k = 0; k < scols; k++) {
	          sum += VL.get(i, k) * U.get(j, k);
	        }
	        VLU.set(i, j, sum);
	      }
	    }
	    return VLU.mmul(Y);
	  }
	  solveForDiagonal(value) {
	    return this.solve(Matrix$1.diag(value));
	  }
	  inverse() {
	    let V = this.V;
	    let e = this.threshold;
	    let vrows = V.rows;
	    let vcols = V.columns;
	    let X = new Matrix$1(vrows, this.s.length);
	    for (let i = 0; i < vrows; i++) {
	      for (let j = 0; j < vcols; j++) {
	        if (Math.abs(this.s[j]) > e) {
	          X.set(i, j, V.get(i, j) / this.s[j]);
	        }
	      }
	    }
	    let U = this.U;
	    let urows = U.rows;
	    let ucols = U.columns;
	    let Y = new Matrix$1(vrows, urows);
	    for (let i = 0; i < vrows; i++) {
	      for (let j = 0; j < urows; j++) {
	        let sum = 0;
	        for (let k = 0; k < ucols; k++) {
	          sum += X.get(i, k) * U.get(j, k);
	        }
	        Y.set(i, j, sum);
	      }
	    }
	    return Y;
	  }
	  get condition() {
	    return this.s[0] / this.s[Math.min(this.m, this.n) - 1];
	  }
	  get norm2() {
	    return this.s[0];
	  }
	  get rank() {
	    let tol = Math.max(this.m, this.n) * this.s[0] * Number.EPSILON;
	    let r = 0;
	    let s = this.s;
	    for (let i = 0, ii = s.length; i < ii; i++) {
	      if (s[i] > tol) {
	        r++;
	      }
	    }
	    return r;
	  }
	  get diagonal() {
	    return Array.from(this.s);
	  }
	  get threshold() {
	    return Number.EPSILON / 2 * Math.max(this.m, this.n) * this.s[0];
	  }
	  get leftSingularVectors() {
	    return this.U;
	  }
	  get rightSingularVectors() {
	    return this.V;
	  }
	  get diagonalMatrix() {
	    return Matrix$1.diag(this.s);
	  }
	}
	function inverse(matrix, useSVD = false) {
	  matrix = WrapperMatrix2D.checkMatrix(matrix);
	  if (useSVD) {
	    return new SingularValueDecomposition(matrix).inverse();
	  } else {
	    return solve(matrix, Matrix$1.eye(matrix.rows));
	  }
	}
	function solve(leftHandSide, rightHandSide, useSVD = false) {
	  leftHandSide = WrapperMatrix2D.checkMatrix(leftHandSide);
	  rightHandSide = WrapperMatrix2D.checkMatrix(rightHandSide);
	  if (useSVD) {
	    return new SingularValueDecomposition(leftHandSide).solve(rightHandSide);
	  } else {
	    return leftHandSide.isSquare() ? new LuDecomposition(leftHandSide).solve(rightHandSide) : new QrDecomposition(leftHandSide).solve(rightHandSide);
	  }
	}
	function determinant(matrix) {
	  matrix = Matrix$1.checkMatrix(matrix);
	  if (matrix.isSquare()) {
	    if (matrix.columns === 0) {
	      return 1;
	    }
	    let a, b, c, d;
	    if (matrix.columns === 2) {
	      // 2 x 2 matrix
	      a = matrix.get(0, 0);
	      b = matrix.get(0, 1);
	      c = matrix.get(1, 0);
	      d = matrix.get(1, 1);
	      return a * d - b * c;
	    } else if (matrix.columns === 3) {
	      // 3 x 3 matrix
	      let subMatrix0, subMatrix1, subMatrix2;
	      subMatrix0 = new MatrixSelectionView(matrix, [1, 2], [1, 2]);
	      subMatrix1 = new MatrixSelectionView(matrix, [1, 2], [0, 2]);
	      subMatrix2 = new MatrixSelectionView(matrix, [1, 2], [0, 1]);
	      a = matrix.get(0, 0);
	      b = matrix.get(0, 1);
	      c = matrix.get(0, 2);
	      return a * determinant(subMatrix0) - b * determinant(subMatrix1) + c * determinant(subMatrix2);
	    } else {
	      // general purpose determinant using the LU decomposition
	      return new LuDecomposition(matrix).determinant;
	    }
	  } else {
	    throw Error('determinant can only be calculated for a square matrix');
	  }
	}
	function xrange(n, exception) {
	  let range = [];
	  for (let i = 0; i < n; i++) {
	    if (i !== exception) {
	      range.push(i);
	    }
	  }
	  return range;
	}
	function dependenciesOneRow(error, matrix, index, thresholdValue = 10e-10, thresholdError = 10e-10) {
	  if (error > thresholdError) {
	    return new Array(matrix.rows + 1).fill(0);
	  } else {
	    let returnArray = matrix.addRow(index, [0]);
	    for (let i = 0; i < returnArray.rows; i++) {
	      if (Math.abs(returnArray.get(i, 0)) < thresholdValue) {
	        returnArray.set(i, 0, 0);
	      }
	    }
	    return returnArray.to1DArray();
	  }
	}
	function linearDependencies(matrix, options = {}) {
	  const {
	    thresholdValue = 10e-10,
	    thresholdError = 10e-10
	  } = options;
	  matrix = Matrix$1.checkMatrix(matrix);
	  let n = matrix.rows;
	  let results = new Matrix$1(n, n);
	  for (let i = 0; i < n; i++) {
	    let b = Matrix$1.columnVector(matrix.getRow(i));
	    let Abis = matrix.subMatrixRow(xrange(n, i)).transpose();
	    let svd = new SingularValueDecomposition(Abis);
	    let x = svd.solve(b);
	    let error = Matrix$1.sub(b, Abis.mmul(x)).abs().max();
	    results.setRow(i, dependenciesOneRow(error, x, i, thresholdValue, thresholdError));
	  }
	  return results;
	}
	function pseudoInverse(matrix, threshold = Number.EPSILON) {
	  matrix = Matrix$1.checkMatrix(matrix);
	  if (matrix.isEmpty()) {
	    // with a zero dimension, the pseudo-inverse is the transpose, since all 0xn and nx0 matrices are singular
	    // (0xn)*(nx0)*(0xn) = 0xn
	    // (nx0)*(0xn)*(nx0) = nx0
	    return matrix.transpose();
	  }
	  let svdSolution = new SingularValueDecomposition(matrix, {
	    autoTranspose: true
	  });
	  let U = svdSolution.leftSingularVectors;
	  let V = svdSolution.rightSingularVectors;
	  let s = svdSolution.diagonal;
	  for (let i = 0; i < s.length; i++) {
	    if (Math.abs(s[i]) > threshold) {
	      s[i] = 1.0 / s[i];
	    } else {
	      s[i] = 0.0;
	    }
	  }
	  return V.mmul(Matrix$1.diag(s).mmul(U.transpose()));
	}
	function covariance(xMatrix, yMatrix = xMatrix, options = {}) {
	  xMatrix = new Matrix$1(xMatrix);
	  let yIsSame = false;
	  if (typeof yMatrix === 'object' && !Matrix$1.isMatrix(yMatrix) && !isAnyArray.isAnyArray(yMatrix)) {
	    options = yMatrix;
	    yMatrix = xMatrix;
	    yIsSame = true;
	  } else {
	    yMatrix = new Matrix$1(yMatrix);
	  }
	  if (xMatrix.rows !== yMatrix.rows) {
	    throw new TypeError('Both matrices must have the same number of rows');
	  }
	  const {
	    center = true
	  } = options;
	  if (center) {
	    xMatrix = xMatrix.center('column');
	    if (!yIsSame) {
	      yMatrix = yMatrix.center('column');
	    }
	  }
	  const cov = xMatrix.transpose().mmul(yMatrix);
	  for (let i = 0; i < cov.rows; i++) {
	    for (let j = 0; j < cov.columns; j++) {
	      cov.set(i, j, cov.get(i, j) * (1 / (xMatrix.rows - 1)));
	    }
	  }
	  return cov;
	}
	function correlation(xMatrix, yMatrix = xMatrix, options = {}) {
	  xMatrix = new Matrix$1(xMatrix);
	  let yIsSame = false;
	  if (typeof yMatrix === 'object' && !Matrix$1.isMatrix(yMatrix) && !isAnyArray.isAnyArray(yMatrix)) {
	    options = yMatrix;
	    yMatrix = xMatrix;
	    yIsSame = true;
	  } else {
	    yMatrix = new Matrix$1(yMatrix);
	  }
	  if (xMatrix.rows !== yMatrix.rows) {
	    throw new TypeError('Both matrices must have the same number of rows');
	  }
	  const {
	    center = true,
	    scale = true
	  } = options;
	  if (center) {
	    xMatrix.center('column');
	    if (!yIsSame) {
	      yMatrix.center('column');
	    }
	  }
	  if (scale) {
	    xMatrix.scale('column');
	    if (!yIsSame) {
	      yMatrix.scale('column');
	    }
	  }
	  const sdx = xMatrix.standardDeviation('column', {
	    unbiased: true
	  });
	  const sdy = yIsSame ? sdx : yMatrix.standardDeviation('column', {
	    unbiased: true
	  });
	  const corr = xMatrix.transpose().mmul(yMatrix);
	  for (let i = 0; i < corr.rows; i++) {
	    for (let j = 0; j < corr.columns; j++) {
	      corr.set(i, j, corr.get(i, j) * (1 / (sdx[i] * sdy[j])) * (1 / (xMatrix.rows - 1)));
	    }
	  }
	  return corr;
	}
	class EigenvalueDecomposition {
	  constructor(matrix, options = {}) {
	    const {
	      assumeSymmetric = false
	    } = options;
	    matrix = WrapperMatrix2D.checkMatrix(matrix);
	    if (!matrix.isSquare()) {
	      throw new Error('Matrix is not a square matrix');
	    }
	    if (matrix.isEmpty()) {
	      throw new Error('Matrix must be non-empty');
	    }
	    let n = matrix.columns;
	    let V = new Matrix$1(n, n);
	    let d = new Float64Array(n);
	    let e = new Float64Array(n);
	    let value = matrix;
	    let i, j;
	    let isSymmetric = false;
	    if (assumeSymmetric) {
	      isSymmetric = true;
	    } else {
	      isSymmetric = matrix.isSymmetric();
	    }
	    if (isSymmetric) {
	      for (i = 0; i < n; i++) {
	        for (j = 0; j < n; j++) {
	          V.set(i, j, value.get(i, j));
	        }
	      }
	      tred2(n, e, d, V);
	      tql2(n, e, d, V);
	    } else {
	      let H = new Matrix$1(n, n);
	      let ort = new Float64Array(n);
	      for (j = 0; j < n; j++) {
	        for (i = 0; i < n; i++) {
	          H.set(i, j, value.get(i, j));
	        }
	      }
	      orthes(n, H, ort, V);
	      hqr2(n, e, d, V, H);
	    }
	    this.n = n;
	    this.e = e;
	    this.d = d;
	    this.V = V;
	  }
	  get realEigenvalues() {
	    return Array.from(this.d);
	  }
	  get imaginaryEigenvalues() {
	    return Array.from(this.e);
	  }
	  get eigenvectorMatrix() {
	    return this.V;
	  }
	  get diagonalMatrix() {
	    let n = this.n;
	    let e = this.e;
	    let d = this.d;
	    let X = new Matrix$1(n, n);
	    let i, j;
	    for (i = 0; i < n; i++) {
	      for (j = 0; j < n; j++) {
	        X.set(i, j, 0);
	      }
	      X.set(i, i, d[i]);
	      if (e[i] > 0) {
	        X.set(i, i + 1, e[i]);
	      } else if (e[i] < 0) {
	        X.set(i, i - 1, e[i]);
	      }
	    }
	    return X;
	  }
	}
	function tred2(n, e, d, V) {
	  let f, g, h, i, j, k, hh, scale;
	  for (j = 0; j < n; j++) {
	    d[j] = V.get(n - 1, j);
	  }
	  for (i = n - 1; i > 0; i--) {
	    scale = 0;
	    h = 0;
	    for (k = 0; k < i; k++) {
	      scale = scale + Math.abs(d[k]);
	    }
	    if (scale === 0) {
	      e[i] = d[i - 1];
	      for (j = 0; j < i; j++) {
	        d[j] = V.get(i - 1, j);
	        V.set(i, j, 0);
	        V.set(j, i, 0);
	      }
	    } else {
	      for (k = 0; k < i; k++) {
	        d[k] /= scale;
	        h += d[k] * d[k];
	      }
	      f = d[i - 1];
	      g = Math.sqrt(h);
	      if (f > 0) {
	        g = -g;
	      }
	      e[i] = scale * g;
	      h = h - f * g;
	      d[i - 1] = f - g;
	      for (j = 0; j < i; j++) {
	        e[j] = 0;
	      }
	      for (j = 0; j < i; j++) {
	        f = d[j];
	        V.set(j, i, f);
	        g = e[j] + V.get(j, j) * f;
	        for (k = j + 1; k <= i - 1; k++) {
	          g += V.get(k, j) * d[k];
	          e[k] += V.get(k, j) * f;
	        }
	        e[j] = g;
	      }
	      f = 0;
	      for (j = 0; j < i; j++) {
	        e[j] /= h;
	        f += e[j] * d[j];
	      }
	      hh = f / (h + h);
	      for (j = 0; j < i; j++) {
	        e[j] -= hh * d[j];
	      }
	      for (j = 0; j < i; j++) {
	        f = d[j];
	        g = e[j];
	        for (k = j; k <= i - 1; k++) {
	          V.set(k, j, V.get(k, j) - (f * e[k] + g * d[k]));
	        }
	        d[j] = V.get(i - 1, j);
	        V.set(i, j, 0);
	      }
	    }
	    d[i] = h;
	  }
	  for (i = 0; i < n - 1; i++) {
	    V.set(n - 1, i, V.get(i, i));
	    V.set(i, i, 1);
	    h = d[i + 1];
	    if (h !== 0) {
	      for (k = 0; k <= i; k++) {
	        d[k] = V.get(k, i + 1) / h;
	      }
	      for (j = 0; j <= i; j++) {
	        g = 0;
	        for (k = 0; k <= i; k++) {
	          g += V.get(k, i + 1) * V.get(k, j);
	        }
	        for (k = 0; k <= i; k++) {
	          V.set(k, j, V.get(k, j) - g * d[k]);
	        }
	      }
	    }
	    for (k = 0; k <= i; k++) {
	      V.set(k, i + 1, 0);
	    }
	  }
	  for (j = 0; j < n; j++) {
	    d[j] = V.get(n - 1, j);
	    V.set(n - 1, j, 0);
	  }
	  V.set(n - 1, n - 1, 1);
	  e[0] = 0;
	}
	function tql2(n, e, d, V) {
	  let g, h, i, j, k, l, m, p, r, dl1, c, c2, c3, el1, s, s2;
	  for (i = 1; i < n; i++) {
	    e[i - 1] = e[i];
	  }
	  e[n - 1] = 0;
	  let f = 0;
	  let tst1 = 0;
	  let eps = Number.EPSILON;
	  for (l = 0; l < n; l++) {
	    tst1 = Math.max(tst1, Math.abs(d[l]) + Math.abs(e[l]));
	    m = l;
	    while (m < n) {
	      if (Math.abs(e[m]) <= eps * tst1) {
	        break;
	      }
	      m++;
	    }
	    if (m > l) {
	      do {
	        g = d[l];
	        p = (d[l + 1] - g) / (2 * e[l]);
	        r = hypotenuse(p, 1);
	        if (p < 0) {
	          r = -r;
	        }
	        d[l] = e[l] / (p + r);
	        d[l + 1] = e[l] * (p + r);
	        dl1 = d[l + 1];
	        h = g - d[l];
	        for (i = l + 2; i < n; i++) {
	          d[i] -= h;
	        }
	        f = f + h;
	        p = d[m];
	        c = 1;
	        c2 = c;
	        c3 = c;
	        el1 = e[l + 1];
	        s = 0;
	        s2 = 0;
	        for (i = m - 1; i >= l; i--) {
	          c3 = c2;
	          c2 = c;
	          s2 = s;
	          g = c * e[i];
	          h = c * p;
	          r = hypotenuse(p, e[i]);
	          e[i + 1] = s * r;
	          s = e[i] / r;
	          c = p / r;
	          p = c * d[i] - s * g;
	          d[i + 1] = h + s * (c * g + s * d[i]);
	          for (k = 0; k < n; k++) {
	            h = V.get(k, i + 1);
	            V.set(k, i + 1, s * V.get(k, i) + c * h);
	            V.set(k, i, c * V.get(k, i) - s * h);
	          }
	        }
	        p = -s * s2 * c3 * el1 * e[l] / dl1;
	        e[l] = s * p;
	        d[l] = c * p;
	      } while (Math.abs(e[l]) > eps * tst1);
	    }
	    d[l] = d[l] + f;
	    e[l] = 0;
	  }
	  for (i = 0; i < n - 1; i++) {
	    k = i;
	    p = d[i];
	    for (j = i + 1; j < n; j++) {
	      if (d[j] < p) {
	        k = j;
	        p = d[j];
	      }
	    }
	    if (k !== i) {
	      d[k] = d[i];
	      d[i] = p;
	      for (j = 0; j < n; j++) {
	        p = V.get(j, i);
	        V.set(j, i, V.get(j, k));
	        V.set(j, k, p);
	      }
	    }
	  }
	}
	function orthes(n, H, ort, V) {
	  let low = 0;
	  let high = n - 1;
	  let f, g, h, i, j, m;
	  let scale;
	  for (m = low + 1; m <= high - 1; m++) {
	    scale = 0;
	    for (i = m; i <= high; i++) {
	      scale = scale + Math.abs(H.get(i, m - 1));
	    }
	    if (scale !== 0) {
	      h = 0;
	      for (i = high; i >= m; i--) {
	        ort[i] = H.get(i, m - 1) / scale;
	        h += ort[i] * ort[i];
	      }
	      g = Math.sqrt(h);
	      if (ort[m] > 0) {
	        g = -g;
	      }
	      h = h - ort[m] * g;
	      ort[m] = ort[m] - g;
	      for (j = m; j < n; j++) {
	        f = 0;
	        for (i = high; i >= m; i--) {
	          f += ort[i] * H.get(i, j);
	        }
	        f = f / h;
	        for (i = m; i <= high; i++) {
	          H.set(i, j, H.get(i, j) - f * ort[i]);
	        }
	      }
	      for (i = 0; i <= high; i++) {
	        f = 0;
	        for (j = high; j >= m; j--) {
	          f += ort[j] * H.get(i, j);
	        }
	        f = f / h;
	        for (j = m; j <= high; j++) {
	          H.set(i, j, H.get(i, j) - f * ort[j]);
	        }
	      }
	      ort[m] = scale * ort[m];
	      H.set(m, m - 1, scale * g);
	    }
	  }
	  for (i = 0; i < n; i++) {
	    for (j = 0; j < n; j++) {
	      V.set(i, j, i === j ? 1 : 0);
	    }
	  }
	  for (m = high - 1; m >= low + 1; m--) {
	    if (H.get(m, m - 1) !== 0) {
	      for (i = m + 1; i <= high; i++) {
	        ort[i] = H.get(i, m - 1);
	      }
	      for (j = m; j <= high; j++) {
	        g = 0;
	        for (i = m; i <= high; i++) {
	          g += ort[i] * V.get(i, j);
	        }
	        g = g / ort[m] / H.get(m, m - 1);
	        for (i = m; i <= high; i++) {
	          V.set(i, j, V.get(i, j) + g * ort[i]);
	        }
	      }
	    }
	  }
	}
	function hqr2(nn, e, d, V, H) {
	  let n = nn - 1;
	  let low = 0;
	  let high = nn - 1;
	  let eps = Number.EPSILON;
	  let exshift = 0;
	  let norm = 0;
	  let p = 0;
	  let q = 0;
	  let r = 0;
	  let s = 0;
	  let z = 0;
	  let iter = 0;
	  let i, j, k, l, m, t, w, x, y;
	  let ra, sa, vr, vi;
	  let notlast, cdivres;
	  for (i = 0; i < nn; i++) {
	    if (i < low || i > high) {
	      d[i] = H.get(i, i);
	      e[i] = 0;
	    }
	    for (j = Math.max(i - 1, 0); j < nn; j++) {
	      norm = norm + Math.abs(H.get(i, j));
	    }
	  }
	  while (n >= low) {
	    l = n;
	    while (l > low) {
	      s = Math.abs(H.get(l - 1, l - 1)) + Math.abs(H.get(l, l));
	      if (s === 0) {
	        s = norm;
	      }
	      if (Math.abs(H.get(l, l - 1)) < eps * s) {
	        break;
	      }
	      l--;
	    }
	    if (l === n) {
	      H.set(n, n, H.get(n, n) + exshift);
	      d[n] = H.get(n, n);
	      e[n] = 0;
	      n--;
	      iter = 0;
	    } else if (l === n - 1) {
	      w = H.get(n, n - 1) * H.get(n - 1, n);
	      p = (H.get(n - 1, n - 1) - H.get(n, n)) / 2;
	      q = p * p + w;
	      z = Math.sqrt(Math.abs(q));
	      H.set(n, n, H.get(n, n) + exshift);
	      H.set(n - 1, n - 1, H.get(n - 1, n - 1) + exshift);
	      x = H.get(n, n);
	      if (q >= 0) {
	        z = p >= 0 ? p + z : p - z;
	        d[n - 1] = x + z;
	        d[n] = d[n - 1];
	        if (z !== 0) {
	          d[n] = x - w / z;
	        }
	        e[n - 1] = 0;
	        e[n] = 0;
	        x = H.get(n, n - 1);
	        s = Math.abs(x) + Math.abs(z);
	        p = x / s;
	        q = z / s;
	        r = Math.sqrt(p * p + q * q);
	        p = p / r;
	        q = q / r;
	        for (j = n - 1; j < nn; j++) {
	          z = H.get(n - 1, j);
	          H.set(n - 1, j, q * z + p * H.get(n, j));
	          H.set(n, j, q * H.get(n, j) - p * z);
	        }
	        for (i = 0; i <= n; i++) {
	          z = H.get(i, n - 1);
	          H.set(i, n - 1, q * z + p * H.get(i, n));
	          H.set(i, n, q * H.get(i, n) - p * z);
	        }
	        for (i = low; i <= high; i++) {
	          z = V.get(i, n - 1);
	          V.set(i, n - 1, q * z + p * V.get(i, n));
	          V.set(i, n, q * V.get(i, n) - p * z);
	        }
	      } else {
	        d[n - 1] = x + p;
	        d[n] = x + p;
	        e[n - 1] = z;
	        e[n] = -z;
	      }
	      n = n - 2;
	      iter = 0;
	    } else {
	      x = H.get(n, n);
	      y = 0;
	      w = 0;
	      if (l < n) {
	        y = H.get(n - 1, n - 1);
	        w = H.get(n, n - 1) * H.get(n - 1, n);
	      }
	      if (iter === 10) {
	        exshift += x;
	        for (i = low; i <= n; i++) {
	          H.set(i, i, H.get(i, i) - x);
	        }
	        s = Math.abs(H.get(n, n - 1)) + Math.abs(H.get(n - 1, n - 2));
	        // eslint-disable-next-line no-multi-assign
	        x = y = 0.75 * s;
	        w = -0.4375 * s * s;
	      }
	      if (iter === 30) {
	        s = (y - x) / 2;
	        s = s * s + w;
	        if (s > 0) {
	          s = Math.sqrt(s);
	          if (y < x) {
	            s = -s;
	          }
	          s = x - w / ((y - x) / 2 + s);
	          for (i = low; i <= n; i++) {
	            H.set(i, i, H.get(i, i) - s);
	          }
	          exshift += s;
	          // eslint-disable-next-line no-multi-assign
	          x = y = w = 0.964;
	        }
	      }
	      iter = iter + 1;
	      m = n - 2;
	      while (m >= l) {
	        z = H.get(m, m);
	        r = x - z;
	        s = y - z;
	        p = (r * s - w) / H.get(m + 1, m) + H.get(m, m + 1);
	        q = H.get(m + 1, m + 1) - z - r - s;
	        r = H.get(m + 2, m + 1);
	        s = Math.abs(p) + Math.abs(q) + Math.abs(r);
	        p = p / s;
	        q = q / s;
	        r = r / s;
	        if (m === l) {
	          break;
	        }
	        if (Math.abs(H.get(m, m - 1)) * (Math.abs(q) + Math.abs(r)) < eps * (Math.abs(p) * (Math.abs(H.get(m - 1, m - 1)) + Math.abs(z) + Math.abs(H.get(m + 1, m + 1))))) {
	          break;
	        }
	        m--;
	      }
	      for (i = m + 2; i <= n; i++) {
	        H.set(i, i - 2, 0);
	        if (i > m + 2) {
	          H.set(i, i - 3, 0);
	        }
	      }
	      for (k = m; k <= n - 1; k++) {
	        notlast = k !== n - 1;
	        if (k !== m) {
	          p = H.get(k, k - 1);
	          q = H.get(k + 1, k - 1);
	          r = notlast ? H.get(k + 2, k - 1) : 0;
	          x = Math.abs(p) + Math.abs(q) + Math.abs(r);
	          if (x !== 0) {
	            p = p / x;
	            q = q / x;
	            r = r / x;
	          }
	        }
	        if (x === 0) {
	          break;
	        }
	        s = Math.sqrt(p * p + q * q + r * r);
	        if (p < 0) {
	          s = -s;
	        }
	        if (s !== 0) {
	          if (k !== m) {
	            H.set(k, k - 1, -s * x);
	          } else if (l !== m) {
	            H.set(k, k - 1, -H.get(k, k - 1));
	          }
	          p = p + s;
	          x = p / s;
	          y = q / s;
	          z = r / s;
	          q = q / p;
	          r = r / p;
	          for (j = k; j < nn; j++) {
	            p = H.get(k, j) + q * H.get(k + 1, j);
	            if (notlast) {
	              p = p + r * H.get(k + 2, j);
	              H.set(k + 2, j, H.get(k + 2, j) - p * z);
	            }
	            H.set(k, j, H.get(k, j) - p * x);
	            H.set(k + 1, j, H.get(k + 1, j) - p * y);
	          }
	          for (i = 0; i <= Math.min(n, k + 3); i++) {
	            p = x * H.get(i, k) + y * H.get(i, k + 1);
	            if (notlast) {
	              p = p + z * H.get(i, k + 2);
	              H.set(i, k + 2, H.get(i, k + 2) - p * r);
	            }
	            H.set(i, k, H.get(i, k) - p);
	            H.set(i, k + 1, H.get(i, k + 1) - p * q);
	          }
	          for (i = low; i <= high; i++) {
	            p = x * V.get(i, k) + y * V.get(i, k + 1);
	            if (notlast) {
	              p = p + z * V.get(i, k + 2);
	              V.set(i, k + 2, V.get(i, k + 2) - p * r);
	            }
	            V.set(i, k, V.get(i, k) - p);
	            V.set(i, k + 1, V.get(i, k + 1) - p * q);
	          }
	        }
	      }
	    }
	  }
	  if (norm === 0) {
	    return;
	  }
	  for (n = nn - 1; n >= 0; n--) {
	    p = d[n];
	    q = e[n];
	    if (q === 0) {
	      l = n;
	      H.set(n, n, 1);
	      for (i = n - 1; i >= 0; i--) {
	        w = H.get(i, i) - p;
	        r = 0;
	        for (j = l; j <= n; j++) {
	          r = r + H.get(i, j) * H.get(j, n);
	        }
	        if (e[i] < 0) {
	          z = w;
	          s = r;
	        } else {
	          l = i;
	          if (e[i] === 0) {
	            H.set(i, n, w !== 0 ? -r / w : -r / (eps * norm));
	          } else {
	            x = H.get(i, i + 1);
	            y = H.get(i + 1, i);
	            q = (d[i] - p) * (d[i] - p) + e[i] * e[i];
	            t = (x * s - z * r) / q;
	            H.set(i, n, t);
	            H.set(i + 1, n, Math.abs(x) > Math.abs(z) ? (-r - w * t) / x : (-s - y * t) / z);
	          }
	          t = Math.abs(H.get(i, n));
	          if (eps * t * t > 1) {
	            for (j = i; j <= n; j++) {
	              H.set(j, n, H.get(j, n) / t);
	            }
	          }
	        }
	      }
	    } else if (q < 0) {
	      l = n - 1;
	      if (Math.abs(H.get(n, n - 1)) > Math.abs(H.get(n - 1, n))) {
	        H.set(n - 1, n - 1, q / H.get(n, n - 1));
	        H.set(n - 1, n, -(H.get(n, n) - p) / H.get(n, n - 1));
	      } else {
	        cdivres = cdiv(0, -H.get(n - 1, n), H.get(n - 1, n - 1) - p, q);
	        H.set(n - 1, n - 1, cdivres[0]);
	        H.set(n - 1, n, cdivres[1]);
	      }
	      H.set(n, n - 1, 0);
	      H.set(n, n, 1);
	      for (i = n - 2; i >= 0; i--) {
	        ra = 0;
	        sa = 0;
	        for (j = l; j <= n; j++) {
	          ra = ra + H.get(i, j) * H.get(j, n - 1);
	          sa = sa + H.get(i, j) * H.get(j, n);
	        }
	        w = H.get(i, i) - p;
	        if (e[i] < 0) {
	          z = w;
	          r = ra;
	          s = sa;
	        } else {
	          l = i;
	          if (e[i] === 0) {
	            cdivres = cdiv(-ra, -sa, w, q);
	            H.set(i, n - 1, cdivres[0]);
	            H.set(i, n, cdivres[1]);
	          } else {
	            x = H.get(i, i + 1);
	            y = H.get(i + 1, i);
	            vr = (d[i] - p) * (d[i] - p) + e[i] * e[i] - q * q;
	            vi = (d[i] - p) * 2 * q;
	            if (vr === 0 && vi === 0) {
	              vr = eps * norm * (Math.abs(w) + Math.abs(q) + Math.abs(x) + Math.abs(y) + Math.abs(z));
	            }
	            cdivres = cdiv(x * r - z * ra + q * sa, x * s - z * sa - q * ra, vr, vi);
	            H.set(i, n - 1, cdivres[0]);
	            H.set(i, n, cdivres[1]);
	            if (Math.abs(x) > Math.abs(z) + Math.abs(q)) {
	              H.set(i + 1, n - 1, (-ra - w * H.get(i, n - 1) + q * H.get(i, n)) / x);
	              H.set(i + 1, n, (-sa - w * H.get(i, n) - q * H.get(i, n - 1)) / x);
	            } else {
	              cdivres = cdiv(-r - y * H.get(i, n - 1), -s - y * H.get(i, n), z, q);
	              H.set(i + 1, n - 1, cdivres[0]);
	              H.set(i + 1, n, cdivres[1]);
	            }
	          }
	          t = Math.max(Math.abs(H.get(i, n - 1)), Math.abs(H.get(i, n)));
	          if (eps * t * t > 1) {
	            for (j = i; j <= n; j++) {
	              H.set(j, n - 1, H.get(j, n - 1) / t);
	              H.set(j, n, H.get(j, n) / t);
	            }
	          }
	        }
	      }
	    }
	  }
	  for (i = 0; i < nn; i++) {
	    if (i < low || i > high) {
	      for (j = i; j < nn; j++) {
	        V.set(i, j, H.get(i, j));
	      }
	    }
	  }
	  for (j = nn - 1; j >= low; j--) {
	    for (i = low; i <= high; i++) {
	      z = 0;
	      for (k = low; k <= Math.min(j, high); k++) {
	        z = z + V.get(i, k) * H.get(k, j);
	      }
	      V.set(i, j, z);
	    }
	  }
	}
	function cdiv(xr, xi, yr, yi) {
	  let r, d;
	  if (Math.abs(yr) > Math.abs(yi)) {
	    r = yi / yr;
	    d = yr + r * yi;
	    return [(xr + r * xi) / d, (xi - r * xr) / d];
	  } else {
	    r = yr / yi;
	    d = yi + r * yr;
	    return [(r * xr + xi) / d, (r * xi - xr) / d];
	  }
	}
	class CholeskyDecomposition {
	  constructor(value) {
	    value = WrapperMatrix2D.checkMatrix(value);
	    if (!value.isSymmetric()) {
	      throw new Error('Matrix is not symmetric');
	    }
	    let a = value;
	    let dimension = a.rows;
	    let l = new Matrix$1(dimension, dimension);
	    let positiveDefinite = true;
	    let i, j, k;
	    for (j = 0; j < dimension; j++) {
	      let d = 0;
	      for (k = 0; k < j; k++) {
	        let s = 0;
	        for (i = 0; i < k; i++) {
	          s += l.get(k, i) * l.get(j, i);
	        }
	        s = (a.get(j, k) - s) / l.get(k, k);
	        l.set(j, k, s);
	        d = d + s * s;
	      }
	      d = a.get(j, j) - d;
	      positiveDefinite &= d > 0;
	      l.set(j, j, Math.sqrt(Math.max(d, 0)));
	      for (k = j + 1; k < dimension; k++) {
	        l.set(j, k, 0);
	      }
	    }
	    this.L = l;
	    this.positiveDefinite = Boolean(positiveDefinite);
	  }
	  isPositiveDefinite() {
	    return this.positiveDefinite;
	  }
	  solve(value) {
	    value = WrapperMatrix2D.checkMatrix(value);
	    let l = this.L;
	    let dimension = l.rows;
	    if (value.rows !== dimension) {
	      throw new Error('Matrix dimensions do not match');
	    }
	    if (this.isPositiveDefinite() === false) {
	      throw new Error('Matrix is not positive definite');
	    }
	    let count = value.columns;
	    let B = value.clone();
	    let i, j, k;
	    for (k = 0; k < dimension; k++) {
	      for (j = 0; j < count; j++) {
	        for (i = 0; i < k; i++) {
	          B.set(k, j, B.get(k, j) - B.get(i, j) * l.get(k, i));
	        }
	        B.set(k, j, B.get(k, j) / l.get(k, k));
	      }
	    }
	    for (k = dimension - 1; k >= 0; k--) {
	      for (j = 0; j < count; j++) {
	        for (i = k + 1; i < dimension; i++) {
	          B.set(k, j, B.get(k, j) - B.get(i, j) * l.get(i, k));
	        }
	        B.set(k, j, B.get(k, j) / l.get(k, k));
	      }
	    }
	    return B;
	  }
	  get lowerTriangularMatrix() {
	    return this.L;
	  }
	}
	class nipals {
	  constructor(X, options = {}) {
	    X = WrapperMatrix2D.checkMatrix(X);
	    let {
	      Y
	    } = options;
	    const {
	      scaleScores = false,
	      maxIterations = 1000,
	      terminationCriteria = 1e-10
	    } = options;
	    let u;
	    if (Y) {
	      if (isAnyArray.isAnyArray(Y) && typeof Y[0] === 'number') {
	        Y = Matrix$1.columnVector(Y);
	      } else {
	        Y = WrapperMatrix2D.checkMatrix(Y);
	      }
	      if (Y.rows !== X.rows) {
	        throw new Error('Y should have the same number of rows as X');
	      }
	      u = Y.getColumnVector(0);
	    } else {
	      u = X.getColumnVector(0);
	    }
	    let diff = 1;
	    let t, q, w, tOld;
	    for (let counter = 0; counter < maxIterations && diff > terminationCriteria; counter++) {
	      w = X.transpose().mmul(u).div(u.transpose().mmul(u).get(0, 0));
	      w = w.div(w.norm());
	      t = X.mmul(w).div(w.transpose().mmul(w).get(0, 0));
	      if (counter > 0) {
	        diff = t.clone().sub(tOld).pow(2).sum();
	      }
	      tOld = t.clone();
	      if (Y) {
	        q = Y.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
	        q = q.div(q.norm());
	        u = Y.mmul(q).div(q.transpose().mmul(q).get(0, 0));
	      } else {
	        u = t;
	      }
	    }
	    if (Y) {
	      let p = X.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
	      p = p.div(p.norm());
	      let xResidual = X.clone().sub(t.clone().mmul(p.transpose()));
	      let residual = u.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
	      let yResidual = Y.clone().sub(t.clone().mulS(residual.get(0, 0)).mmul(q.transpose()));
	      this.t = t;
	      this.p = p.transpose();
	      this.w = w.transpose();
	      this.q = q;
	      this.u = u;
	      this.s = t.transpose().mmul(t);
	      this.xResidual = xResidual;
	      this.yResidual = yResidual;
	      this.betas = residual;
	    } else {
	      this.w = w.transpose();
	      this.s = t.transpose().mmul(t).sqrt();
	      if (scaleScores) {
	        this.t = t.clone().div(this.s.get(0, 0));
	      } else {
	        this.t = t;
	      }
	      this.xResidual = X.sub(t.mmul(w.transpose()));
	    }
	  }
	}
	matrix.AbstractMatrix = AbstractMatrix;
	matrix.CHO = CholeskyDecomposition;
	matrix.CholeskyDecomposition = CholeskyDecomposition;
	matrix.DistanceMatrix = DistanceMatrix;
	matrix.EVD = EigenvalueDecomposition;
	matrix.EigenvalueDecomposition = EigenvalueDecomposition;
	matrix.LU = LuDecomposition;
	matrix.LuDecomposition = LuDecomposition;
	var Matrix_1 = matrix.Matrix = Matrix$1;
	matrix.MatrixColumnSelectionView = MatrixColumnSelectionView;
	matrix.MatrixColumnView = MatrixColumnView;
	matrix.MatrixFlipColumnView = MatrixFlipColumnView;
	matrix.MatrixFlipRowView = MatrixFlipRowView;
	matrix.MatrixRowSelectionView = MatrixRowSelectionView;
	matrix.MatrixRowView = MatrixRowView;
	matrix.MatrixSelectionView = MatrixSelectionView;
	matrix.MatrixSubView = MatrixSubView;
	matrix.MatrixTransposeView = MatrixTransposeView;
	matrix.NIPALS = nipals;
	matrix.Nipals = nipals;
	matrix.QR = QrDecomposition;
	matrix.QrDecomposition = QrDecomposition;
	matrix.SVD = SingularValueDecomposition;
	matrix.SingularValueDecomposition = SingularValueDecomposition;
	matrix.SymmetricMatrix = SymmetricMatrix;
	matrix.WrapperMatrix1D = WrapperMatrix1D;
	matrix.WrapperMatrix2D = WrapperMatrix2D;
	matrix.correlation = correlation;
	matrix.covariance = covariance;
	var _default = matrix.default = Matrix$1;
	matrix.determinant = determinant;
	matrix.inverse = inverse;
	matrix.linearDependencies = linearDependencies;
	matrix.pseudoInverse = pseudoInverse;
	matrix.solve = solve;
	matrix.wrap = wrap;

	const Matrix = Matrix_1;
	_default.Matrix ? _default.Matrix : Matrix_1;

	/**
	 * Algorithm that finds the shortest distance from one node to the other
	 * @param {Matrix} adjMatrix - A squared adjacency matrix
	 * @return {Matrix} - Distance from a node to the other, -1 if the node is unreachable
	 */
	function floydWarshall(adjMatrix) {
	  if (Matrix.isMatrix(adjMatrix) && adjMatrix.columns !== adjMatrix.rows) {
	    throw new TypeError('The adjacency matrix should be squared');
	  }
	  const numVertices = adjMatrix.columns;
	  let distMatrix = new Matrix(numVertices, numVertices);
	  distMatrix.apply((row, column) => {
	    // principal diagonal is 0
	    if (row === column) {
	      distMatrix.set(row, column, 0);
	    } else {
	      let val = adjMatrix.get(row, column);
	      if (val || Object.is(val, -0)) {
	        // edges values remain the same
	        distMatrix.set(row, column, val);
	      } else {
	        // 0 values become infinity
	        distMatrix.set(row, column, Number.POSITIVE_INFINITY);
	      }
	    }
	  });
	  for (let k = 0; k < numVertices; ++k) {
	    for (let i = 0; i < numVertices; ++i) {
	      for (let j = 0; j < numVertices; ++j) {
	        let dist = distMatrix.get(i, k) + distMatrix.get(k, j);
	        if (distMatrix.get(i, j) > dist) {
	          distMatrix.set(i, j, dist);
	        }
	      }
	    }
	  }
	  // When there's no connection the value is -1
	  distMatrix.apply((row, column) => {
	    if (distMatrix.get(row, column) === Number.POSITIVE_INFINITY) {
	      distMatrix.set(row, column, -1);
	    }
	  });
	  return distMatrix;
	}

	var libEsm$1 = /*#__PURE__*/Object.freeze({
		__proto__: null,
		floydWarshall: floydWarshall
	});

	var require$$0$2 = /*@__PURE__*/getAugmentedNamespace(libEsm$1);

	Object.defineProperty(getConnectivityMatrix$1, "__esModule", {
	  value: true
	});
	getConnectivityMatrix$1.getConnectivityMatrix = void 0;
	const ml_floyd_warshall_1 = require$$0$2;
	const ml_matrix_1 = matrix;
	/**
	 * Returns a connectivity matrix
	 * @param {import('openchemlib').Molecule} molecule
	 * @param {object} [options={}]
	 * @param {boolean} [options.pathLength=false] get the path length between atoms
	 * @param {boolean} [options.mass=false] set the nominal mass of the atoms on diagonal
	 * @param {boolean} [options.atomicNo=false] set the atomic number of the atom on diagonal
	 * @param {boolean} [options.negativeAtomicNo=false] set the atomic number * -1 of the atom on diagonal
	 * @param {boolean} [options.sdt=false] set 1, 2 or 3 depending if single, double or triple bond
	 * @param {boolean} [options.sdta=false] set 1, 2, 3 or 4 depending if single, double, triple or aromatic  bond
	 */
	function getConnectivityMatrix(molecule, options = {}) {
	  const OCL = molecule.getOCL();
	  molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
	  const nbAtoms = molecule.getAllAtoms();
	  let result = new Array(nbAtoms).fill();
	  result = result.map(() => new Array(nbAtoms).fill(0));
	  if (!options.pathLength) {
	    if (options.atomicNo) {
	      for (let i = 0; i < nbAtoms; i++) {
	        result[i][i] = molecule.getAtomicNo(i);
	      }
	    } else if (options.negativeAtomicNo) {
	      for (let i = 0; i < nbAtoms; i++) {
	        result[i][i] = -molecule.getAtomicNo(i);
	      }
	    } else if (options.mass) {
	      for (let i = 0; i < nbAtoms; i++) {
	        result[i][i] = OCL.Molecule.cRoundedMass[molecule.getAtomicNo(i)];
	      }
	    } else {
	      for (let i = 0; i < nbAtoms; i++) {
	        result[i][i] = 1;
	      }
	    }
	  }
	  if (options.sdt) {
	    for (let i = 0; i < nbAtoms; i++) {
	      const l = molecule.getAllConnAtoms(i);
	      for (let j = 0; j < l; j++) {
	        result[i][molecule.getConnAtom(i, j)] = molecule.getConnBondOrder(i, j);
	      }
	    }
	  } else if (options.sdta) {
	    for (let i = 0; i < nbAtoms; i++) {
	      const l = molecule.getAllConnAtoms(i);
	      for (let j = 0; j < l; j++) {
	        const bondNumber = molecule.getConnBond(i, j);
	        if (molecule.isAromaticBond(bondNumber)) {
	          result[i][molecule.getConnAtom(i, j)] = 4;
	        } else {
	          result[i][molecule.getConnAtom(i, j)] = molecule.getConnBondOrder(i, j);
	        }
	      }
	    }
	  } else {
	    for (let i = 0; i < nbAtoms; i++) {
	      const l = molecule.getAllConnAtoms(i);
	      for (let j = 0; j < l; j++) {
	        result[i][molecule.getConnAtom(i, j)] = 1;
	      }
	    }
	  }
	  if (options.pathLength) {
	    result = (0, ml_floyd_warshall_1.floydWarshall)(new ml_matrix_1.Matrix(result)).to2DArray();
	  }
	  return result;
	}
	getConnectivityMatrix$1.getConnectivityMatrix = getConnectivityMatrix;

	var getCanonizedDiaIDs$1 = {};

	var makeRacemic$1 = {};

	Object.defineProperty(makeRacemic$1, "__esModule", {
	  value: true
	});
	makeRacemic$1.makeRacemic = void 0;
	/**
	 *
	 * @param {import('openchemlib').Molecule} molecule An instance of a molecule
	 * @param {object} [options={}]
	 * @param {object} [options.OCL] openchemlib library
	 */
	function makeRacemic(molecule) {
	  const {
	    Molecule
	  } = molecule.getOCL();
	  // if we don't calculate this we have 2 epimers
	  molecule.ensureHelperArrays(Molecule.cHelperCIP);
	  // we need to make one group "AND" for chiral (to force to racemic, this means diastereotopic and not enantiotopic)
	  for (let i = 0; i < molecule.getAllAtoms(); i++) {
	    if (molecule.getAtomParity(i) !== Molecule.cAtomParityNone) {
	      molecule.setAtomESR(i, Molecule.cESRTypeAnd, 0); // changed to group 0; TLS 9.Nov.2015
	    }
	  }
	  // after the change we need to recalculate the CIP
	  molecule.ensureHelperArrays(Molecule.cHelperCIP);
	}
	makeRacemic$1.makeRacemic = makeRacemic;

	var tagAtom$1 = {};

	var getXAtomicNumber$1 = {};

	Object.defineProperty(getXAtomicNumber$1, "__esModule", {
	  value: true
	});
	getXAtomicNumber$1.getXAtomicNumber = void 0;
	let xAtomicNumber = 0;
	/**
	 * Returns the atomic number of the X atom
	 * @param {import('openchemlib').Molecule} molecule An instance of a molecule
	 * @returns
	 */
	function getXAtomicNumber(molecule) {
	  if (!xAtomicNumber) {
	    const OCL = molecule.getOCL();
	    xAtomicNumber = OCL.Molecule.getAtomicNoFromLabel('X', OCL.Molecule.cPseudoAtomX);
	  }
	  return xAtomicNumber;
	}
	getXAtomicNumber$1.getXAtomicNumber = getXAtomicNumber;

	Object.defineProperty(tagAtom$1, "__esModule", {
	  value: true
	});
	tagAtom$1.tagAtom = void 0;
	const getXAtomicNumber_1 = getXAtomicNumber$1;
	/**
	 * Tag an atom to be able to visualize it
	 */
	function tagAtom(molecule, iAtom) {
	  const customLabel = `${molecule.getAtomLabel(iAtom)}*`;
	  molecule.setAtomCustomLabel(iAtom, customLabel);
	  if (molecule.getAtomicNo(iAtom) === 1) {
	    molecule.setAtomicNo(iAtom, (0, getXAtomicNumber_1.getXAtomicNumber)(molecule));
	  } else {
	    // we can not use X because we would have problems with valencies if it is
	    // expanded hydrogens or not
	    // we can not only use a custom label because it does not count for the canonisation
	    molecule.setAtomMass(iAtom, molecule.getAtomMass(iAtom) + 5);
	  }
	  return customLabel;
	}
	tagAtom$1.tagAtom = tagAtom;

	Object.defineProperty(getCanonizedDiaIDs$1, "__esModule", {
	  value: true
	});
	getCanonizedDiaIDs$1.getCanonizedDiaIDs = void 0;
	const makeRacemic_js_1$1 = makeRacemic$1;
	const tagAtom_1$5 = tagAtom$1;
	function getCanonizedDiaIDs(diaMol) {
	  const heterotopicSymmetryRanks = diaMol.heterotopicSymmetryRanks;
	  const moleculeWithH = diaMol.moleculeWithH;
	  const finalRanks = diaMol.finalRanks;
	  const canonizedDiaIDs = new Array(moleculeWithH.getAllAtoms());
	  moleculeWithH.ensureHelperArrays(
	  //@ts-expect-error TODO
	  diaMol.Molecule.cHelperSymmetryStereoHeterotopicity);
	  const cache = {};
	  for (let i = 0; i < diaMol.moleculeWithH.getAllAtoms(); i++) {
	    const rank = heterotopicSymmetryRanks[i];
	    if (rank && cache[rank]) {
	      canonizedDiaIDs[finalRanks[i]] = cache[rank].diaID;
	      continue;
	    }
	    const tempMolecule = diaMol.moleculeWithH.getCompactCopy();
	    (0, tagAtom_1$5.tagAtom)(tempMolecule, i);
	    (0, makeRacemic_js_1$1.makeRacemic)(tempMolecule);
	    const diaID = tempMolecule.getCanonizedIDCode(
	    //@ts-expect-error TODO
	    diaMol.Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS);
	    canonizedDiaIDs[finalRanks[i]] = diaID;
	  }
	  return canonizedDiaIDs;
	}
	getCanonizedDiaIDs$1.getCanonizedDiaIDs = getCanonizedDiaIDs;

	var getCanonizedHoseCodes$1 = {};

	var getHoseCodesForAtomsInternal = {};

	var isCsp3$1 = {};

	/**
	 * Check if a specific atom is a sp3 carbon
	 * @param {import('openchemlib').Molecule} molecule
	 * @param {number} atomID
	 */
	Object.defineProperty(isCsp3$1, "__esModule", {
	  value: true
	});
	isCsp3$1.isCsp3 = void 0;
	function isCsp3(molecule, atomID) {
	  if (molecule.getAtomicNo(atomID) !== 6) return false;
	  if (molecule.getAtomCharge(atomID) !== 0) return false;
	  if (molecule.getImplicitHydrogens(atomID) + molecule.getConnAtoms(atomID) !== 4) {
	    return false;
	  }
	  return true;
	}
	isCsp3$1.isCsp3 = isCsp3;

	(function (exports) {

	  Object.defineProperty(exports, "__esModule", {
	    value: true
	  });
	  exports.getHoseCodesForAtomsInternal = exports.HOSE_CODE_CUT_C_SP3_SP3 = exports.FULL_HOSE_CODE = void 0;
	  const isCsp3_js_1 = isCsp3$1;
	  const makeRacemic_js_1 = makeRacemic$1;
	  exports.FULL_HOSE_CODE = 1;
	  exports.HOSE_CODE_CUT_C_SP3_SP3 = 2;
	  /**
	   * Returns the hose code for specific atom numbers
	   * @param {import('openchemlib').Molecule} molecule - The OCL molecule with expandedImplicitHydrogens and ensureHeterotopicChiralBonds
	   * @param {object} [options={}]
	   * @param {string[]} [options.allowedCustomLabels] Array of the custom labels that are considered as root atoms. By default all atoms having a customLabel
	   * @param {number} [options.minSphereSize=0] Smallest hose code sphere
	   * @param {number} [options.maxSphereSize=4] Largest hose code sphere
	   * @param {number} [options.kind=FULL_HOSE_CODE] Kind of hose code, default usual sphere
	   */
	  function getHoseCodesForAtomsInternal(molecule, options = {}) {
	    const OCL = molecule.getOCL();
	    const {
	      allowedCustomLabels,
	      minSphereSize = 0,
	      maxSphereSize = 4,
	      kind = exports.FULL_HOSE_CODE
	    } = options;
	    // this force reordering of atoms in order to have hydrogens at the end
	    molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
	    const rootAtoms = [];
	    for (let j = 0; j < molecule.getAllAtoms(); j++) {
	      if (allowedCustomLabels?.includes(molecule.getAtomCustomLabel(j)) || molecule.getAtomCustomLabel(j)) {
	        rootAtoms.push(j);
	      }
	    }
	    const fragment = new OCL.Molecule(0, 0);
	    const results = [];
	    let min = 0;
	    let max = 0;
	    const atomMask = new Array(molecule.getAllAtoms());
	    const atomList = new Array(molecule.getAllAtoms());
	    for (let sphere = 0; sphere <= maxSphereSize; sphere++) {
	      if (max === 0) {
	        for (const rootAtom of rootAtoms) {
	          atomList[max] = rootAtom;
	          atomMask[rootAtom] = true;
	          max++;
	        }
	      } else {
	        let newMax = max;
	        for (let i = min; i < max; i++) {
	          const atom = atomList[i];
	          for (let j = 0; j < molecule.getAllConnAtoms(atom); j++) {
	            const connAtom = molecule.getConnAtom(atom, j);
	            if (!atomMask[connAtom]) {
	              switch (kind) {
	                case exports.FULL_HOSE_CODE:
	                  atomMask[connAtom] = true;
	                  atomList[newMax++] = connAtom;
	                  break;
	                case exports.HOSE_CODE_CUT_C_SP3_SP3:
	                  if (!((0, isCsp3_js_1.isCsp3)(molecule, atom) && (0, isCsp3_js_1.isCsp3)(molecule, connAtom))) {
	                    atomMask[connAtom] = true;
	                    atomList[newMax++] = connAtom;
	                  }
	                  break;
	                default:
	                  throw new Error('getHoseCoesForAtom unknown kind');
	              }
	            }
	          }
	        }
	        min = max;
	        max = newMax;
	      }
	      molecule.copyMoleculeByAtoms(fragment, atomMask, true, null);
	      if (sphere >= minSphereSize) {
	        (0, makeRacemic_js_1.makeRacemic)(fragment);
	        results.push(fragment.getCanonizedIDCode(OCL.Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS));
	      }
	    }
	    return results;
	  }
	  exports.getHoseCodesForAtomsInternal = getHoseCodesForAtomsInternal;
	})(getHoseCodesForAtomsInternal);

	Object.defineProperty(getCanonizedHoseCodes$1, "__esModule", {
	  value: true
	});
	getCanonizedHoseCodes$1.getCanonizedHoseCodes = void 0;
	const getHoseCodesForAtomsInternal_js_1$3 = getHoseCodesForAtomsInternal;
	const tagAtom_1$4 = tagAtom$1;
	function getCanonizedHoseCodes(diaMol, options = {}) {
	  const heterotopicSymmetryRanks = diaMol.heterotopicSymmetryRanks;
	  const moleculeWithH = diaMol.moleculeWithH;
	  const finalRanks = diaMol.finalRanks;
	  const canonizedHoseCodes = new Array(moleculeWithH.getAllAtoms());
	  moleculeWithH.ensureHelperArrays(
	  //@ts-expect-error TODO
	  diaMol.Molecule.cHelperSymmetryStereoHeterotopicity);
	  const cache = {};
	  for (let i = 0; i < diaMol.moleculeWithH.getAllAtoms(); i++) {
	    const rank = heterotopicSymmetryRanks[i];
	    if (rank && cache[rank]) {
	      canonizedHoseCodes[finalRanks[i]] = cache[rank].diaID;
	      continue;
	    }
	    const tempMolecule = diaMol.moleculeWithH.getCompactCopy();
	    (0, tagAtom_1$4.tagAtom)(tempMolecule, i);
	    const hoses = (0, getHoseCodesForAtomsInternal_js_1$3.getHoseCodesForAtomsInternal)(tempMolecule, options);
	    canonizedHoseCodes[finalRanks[i]] = hoses;
	  }
	  return canonizedHoseCodes;
	}
	getCanonizedHoseCodes$1.getCanonizedHoseCodes = getCanonizedHoseCodes;

	var getDiaIDsAndInfo$1 = {};

	Object.defineProperty(getDiaIDsAndInfo$1, "__esModule", {
	  value: true
	});
	getDiaIDsAndInfo$1.getDiaIDsAndInfo = void 0;
	function getDiaIDsAndInfo(diaMol, canonizedDiaIDs) {
	  const newDiaIDs = [];
	  const molecule = diaMol.moleculeWithH;
	  const counts = {};
	  for (const diaID of canonizedDiaIDs) {
	    if (!counts[diaID]) {
	      counts[diaID] = 0;
	    }
	    counts[diaID]++;
	  }
	  for (let i = 0; i < canonizedDiaIDs.length; i++) {
	    const diaID = canonizedDiaIDs[diaMol.finalRanks[i]];
	    const newDiaID = {
	      idCode: diaID,
	      attachedHydrogensIDCodes: [],
	      nbAttachedHydrogens: 0,
	      atomLabel: molecule.getAtomLabel(i),
	      nbEquivalentAtoms: counts[diaID],
	      heavyAtom: undefined,
	      atomMapNo: molecule.getAtomMapNo(i)
	    };
	    if (molecule.getAtomicNo(i) === 1) {
	      const atom = molecule.getConnAtom(i, 0);
	      newDiaID.heavyAtom = canonizedDiaIDs[diaMol.finalRanks[atom]];
	    }
	    for (let j = 0; j < molecule.getAllConnAtoms(i); j++) {
	      const atom = molecule.getConnAtom(i, j);
	      if (molecule.getAtomicNo(atom) === 1) {
	        newDiaID.nbAttachedHydrogens++;
	        const hydrogenDiaID = canonizedDiaIDs[diaMol.finalRanks[atom]];
	        if (!newDiaID.attachedHydrogensIDCodes.includes(hydrogenDiaID)) {
	          newDiaID.attachedHydrogensIDCodes.push(hydrogenDiaID);
	        }
	      }
	    }
	    newDiaIDs.push(newDiaID);
	  }
	  return newDiaIDs;
	}
	getDiaIDsAndInfo$1.getDiaIDsAndInfo = getDiaIDsAndInfo;

	var getHeterotopicSymmetryRanks$1 = {};

	Object.defineProperty(getHeterotopicSymmetryRanks$1, "__esModule", {
	  value: true
	});
	getHeterotopicSymmetryRanks$1.getFinalRanks = getHeterotopicSymmetryRanks$1.getHeterotopicSymmetryRanks = void 0;
	/**
	 * Get a unique atomic number for a X
	 * @param xMolecule
	 * @returns
	 */
	function getHeterotopicSymmetryRanks(xMolecule) {
	  xMolecule.ensureHelperArrays(xMolecule.getOCL().Molecule.cHelperSymmetryStereoHeterotopicity);
	  const symmetryRanks = [];
	  for (let i = 0; i < xMolecule.getAllAtoms(); i++) {
	    symmetryRanks.push(xMolecule.getSymmetryRank(i));
	  }
	  return symmetryRanks;
	}
	getHeterotopicSymmetryRanks$1.getHeterotopicSymmetryRanks = getHeterotopicSymmetryRanks;
	function getFinalRanks(xMolecule) {
	  xMolecule.ensureHelperArrays(xMolecule.getOCL().Molecule.cHelperSymmetryStereoHeterotopicity);
	  return xMolecule.getFinalRanks(0).map(rank => rank - 1);
	}
	getHeterotopicSymmetryRanks$1.getFinalRanks = getFinalRanks;

	var getMoleculeWithH$1 = {};

	var ensureHeterotopicChiralBonds$1 = {};

	var getChiralOrHeterotopicCarbons$1 = {};

	Object.defineProperty(getChiralOrHeterotopicCarbons$1, "__esModule", {
	  value: true
	});
	getChiralOrHeterotopicCarbons$1.getChiralOrHeterotopicCarbons = void 0;
	const getXAtomicNumber_js_1$2 = getXAtomicNumber$1;
	/**
	 * Returns the atoms that are chiral or pseudo chiral.
	 * There could be some issues if the original molecule lacks chiral bonds.
	 * The function will add them and this could lead to some issues in the case of pseudochiral atoms.
	 * @param {import('openchemlib').Molecule} molecule
	 * @returns {number[]}
	 */
	function getChiralOrHeterotopicCarbons(molecule) {
	  const {
	    Molecule
	  } = molecule.getOCL();
	  const xAtomicNumber = (0, getXAtomicNumber_js_1$2.getXAtomicNumber)(molecule);
	  const internalMolecule = molecule.getCompactCopy();
	  // hydrogens may be diastereotopic, we need to add them
	  internalMolecule.addImplicitHydrogens();
	  for (let i = 0; i < internalMolecule.getAllAtoms(); i++) {
	    // hydrogens are not taken into account during canonization, we need to change them with an atom with a valence of 1
	    if (internalMolecule.getAtomicNo(i) === 1) {
	      internalMolecule.setAtomicNo(i, xAtomicNumber);
	    }
	  }
	  addPossibleChiralBonds(internalMolecule);
	  internalMolecule.ensureHelperArrays(Molecule.cHelperSymmetryStereoHeterotopicity);
	  const atoms = [];
	  for (let i = 0; i < molecule.getAllAtoms(); i++) {
	    if (internalMolecule.getAtomicNo(i) === xAtomicNumber) {
	      continue;
	    }
	    if (molecule.getAtomicNo(i) !== internalMolecule.getAtomicNo(i)) {
	      throw new Error('getChiralOrHeterotopicCarbons: mismatching atomic numbers');
	    }
	    if (internalMolecule.getAtomicNo(i) !== 6) {
	      continue;
	    }
	    const neighbourSymmetries = getNeighbourSymmetries(internalMolecule, i);
	    if (neighbourSymmetries.length === 4) {
	      atoms.push(i);
	    }
	  }
	  return atoms;
	}
	getChiralOrHeterotopicCarbons$1.getChiralOrHeterotopicCarbons = getChiralOrHeterotopicCarbons;
	function addPossibleChiralBonds(molecule) {
	  const {
	    Molecule
	  } = molecule.getOCL();
	  molecule.ensureHelperArrays(Molecule.cHelperSymmetryStereoHeterotopicity);
	  for (let i = 0; i < molecule.getAtoms(); i++) {
	    if (molecule.getAtomicNo(i) !== 6) continue;
	    if (molecule.getStereoBond(i) >= 0) continue;
	    const neighbourSymmetries = getNeighbourSymmetries(molecule, i);
	    if (neighbourSymmetries.length <= 2) continue;
	    const stereoBond = molecule.getAtomPreferredStereoBond(i);
	    if (stereoBond !== -1) {
	      molecule.setBondType(stereoBond, Molecule.cBondTypeUp);
	      if (molecule.getBondAtom(1, stereoBond) === i) {
	        const connAtom = molecule.getBondAtom(0, stereoBond);
	        molecule.setBondAtom(0, stereoBond, i);
	        molecule.setBondAtom(1, stereoBond, connAtom);
	      }
	      // To me it seems that we have to add all stereo centers into AND group 0. TLS 9.Nov.2015
	      molecule.setAtomESR(i, Molecule.cESRTypeAnd, 0);
	    }
	  }
	}
	function getNeighbourSymmetries(molecule, iAtom) {
	  const neighbourSymmetries = [];
	  for (let j = 0; j < molecule.getAllConnAtoms(iAtom); j++) {
	    const connAtom = molecule.getConnAtom(iAtom, j);
	    const symmetryRank = molecule.getSymmetryRank(connAtom);
	    if (!neighbourSymmetries.includes(symmetryRank)) {
	      neighbourSymmetries.push(molecule.getSymmetryRank(connAtom));
	    }
	  }
	  return neighbourSymmetries;
	}

	Object.defineProperty(ensureHeterotopicChiralBonds$1, "__esModule", {
	  value: true
	});
	ensureHeterotopicChiralBonds$1.ensureHeterotopicChiralBonds = void 0;
	const getChiralOrHeterotopicCarbons_js_1 = getChiralOrHeterotopicCarbons$1;
	/**
	 * This function will add missing chiral bonds on carbons ensure that all enantiotopic
	 * or diastereotopic atoms can be identified uniquely
	 * @param {import('openchemlib').Molecule} molecule
	 * @param {object} [options={}]
	 * @param {number} [options.esrType=Molecule.cESRTypeAnd]
	 * @param {boolean} [options.atLeastThreeAtoms=true] - if true, only carbons with at least three atoms will be considered
	 */
	function ensureHeterotopicChiralBonds(molecule, options = {}) {
	  const {
	    Molecule
	  } = molecule.getOCL();
	  const {
	    esrType = Molecule.cESRTypeAnd,
	    atLeastThreeAtoms = true
	  } = options;
	  molecule.ensureHelperArrays(Molecule.cHelperBitNeighbours);
	  const heterotopicCarbons = (0, getChiralOrHeterotopicCarbons_js_1.getChiralOrHeterotopicCarbons)(molecule);
	  for (const i of heterotopicCarbons) {
	    if (atLeastThreeAtoms && molecule.getAllConnAtoms(i) < 3) continue;
	    if (molecule.getStereoBond(i) === -1) {
	      const stereoBond = molecule.getAtomPreferredStereoBond(i);
	      if (stereoBond !== -1) {
	        molecule.setBondType(stereoBond, Molecule.cBondTypeUp);
	        if (molecule.getBondAtom(1, stereoBond) === i) {
	          const connAtom = molecule.getBondAtom(0, stereoBond);
	          molecule.setBondAtom(0, stereoBond, i);
	          molecule.setBondAtom(1, stereoBond, connAtom);
	        }
	        // To me it seems that we have to add all stereo centers into AND group 0. TLS 9.Nov.2015
	        molecule.setAtomESR(i, esrType, 0);
	      }
	    }
	  }
	}
	ensureHeterotopicChiralBonds$1.ensureHeterotopicChiralBonds = ensureHeterotopicChiralBonds;

	Object.defineProperty(getMoleculeWithH$1, "__esModule", {
	  value: true
	});
	getMoleculeWithH$1.getMoleculeWithH = void 0;
	const ensureHeterotopicChiralBonds_js_1$4 = ensureHeterotopicChiralBonds$1;
	const MAX_NB_ATOMS = 250;
	/**
	 * Expand all the implicit hydrogens and ensure that the heterotopic bonds
	 * @param molecule
	 * @returns
	 */
	function getMoleculeWithH(molecule) {
	  const moleculeWithH = molecule.getCompactCopy();
	  moleculeWithH.addImplicitHydrogens();
	  if (moleculeWithH.getAllAtoms() > MAX_NB_ATOMS) {
	    throw new Error(`too many atoms to add hydrogens: ${moleculeWithH.getAllAtoms()} > ${MAX_NB_ATOMS}`);
	  }
	  (0, ensureHeterotopicChiralBonds_js_1$4.ensureHeterotopicChiralBonds)(moleculeWithH);
	  return moleculeWithH;
	}
	getMoleculeWithH$1.getMoleculeWithH = getMoleculeWithH;

	var getXMolecule$1 = {};

	Object.defineProperty(getXMolecule$1, "__esModule", {
	  value: true
	});
	getXMolecule$1.getXMolecule = void 0;
	const getXAtomicNumber_js_1$1 = getXAtomicNumber$1;
	/**
	 * In order to be able to give a unique ID to all the atoms we are replacing the H by X
	 * @param moleculeWithH
	 * @returns
	 */
	function getXMolecule(moleculeWithH) {
	  const xAtomNumber = (0, getXAtomicNumber_js_1$1.getXAtomicNumber)(moleculeWithH);
	  const xMolecule = moleculeWithH.getCompactCopy();
	  for (let i = 0; i < xMolecule.getAllAtoms(); i++) {
	    // hydrogens are not taken into account during canonization, we need to change them with an atom with a valence of 1
	    if (xMolecule.getAtomicNo(i) === 1) {
	      xMolecule.setAtomicNo(i, xAtomNumber);
	    }
	  }
	  return xMolecule;
	}
	getXMolecule$1.getXMolecule = getXMolecule;

	Object.defineProperty(TopicMolecule$1, "__esModule", {
	  value: true
	});
	TopicMolecule$1.groupDiastereotopicAtomIDs = TopicMolecule$1.TopicMolecule = void 0;
	const getConnectivityMatrix_js_1$1 = getConnectivityMatrix$1;
	const getCanonizedDiaIDs_1 = getCanonizedDiaIDs$1;
	const getCanonizedHoseCodes_1 = getCanonizedHoseCodes$1;
	const getDiaIDsAndInfo_1 = getDiaIDsAndInfo$1;
	const getHeterotopicSymmetryRanks_1 = getHeterotopicSymmetryRanks$1;
	const getMoleculeWithH_1 = getMoleculeWithH$1;
	const getXMolecule_1 = getXMolecule$1;
	/**
	 * This class deals with topicity information and hose codes
	 * It is optimized to avoid recalculation of the same information
	 */
	class TopicMolecule {
	  constructor(molecule) {
	    this.originalMolecule = molecule;
	    this.idCode = molecule.getIDCode();
	    this.molecule = this.originalMolecule.getCompactCopy();
	    this.molecule.ensureHelperArrays(molecule.getOCL().Molecule.cHelperNeighbours);
	    this.Molecule = this.molecule.getOCL().Molecule;
	    //@ts-expect-error TODO
	    this.molecule.ensureHelperArrays(this.Molecule.cHelperNeighbours);
	    this.cache = {};
	  }
	  ensureMapNo() {
	    const existingMapNo = {};
	    for (let i = 0; i < this.molecule.getAllAtoms(); i++) {
	      const mapNo = this.molecule.getAtomMapNo(i);
	      if (mapNo) {
	        if (existingMapNo[mapNo]) {
	          throw new Error('The molecule contains several atoms with the same mapNo');
	        }
	        existingMapNo[mapNo] = true;
	      }
	    }
	    let nextMapNo = 1;
	    for (let i = 0; i < this.molecule.getAllAtoms(); i++) {
	      const mapNo = this.molecule.getAtomMapNo(i);
	      if (!mapNo) {
	        while (existingMapNo[nextMapNo]) {
	          nextMapNo++;
	        }
	        existingMapNo[nextMapNo] = true;
	        this.molecule.setAtomMapNo(i, nextMapNo, false);
	      }
	    }
	  }
	  toMolfile(options = {}) {
	    const {
	      version = 2
	    } = options;
	    if (version === 2) {
	      return this.molecule.toMolfile();
	    }
	    return this.molecule.toMolfileV3();
	  }
	  getMolecule() {
	    return this.molecule;
	  }
	  /**
	   * Returns a new TopicMolecule but will copy precalculated information
	   * if possible (same idCode). This is very practical when expanding hydrogens
	   * for example.
	   * @param molecule
	   * @returns
	   */
	  fromMolecule(molecule) {
	    const idCode = molecule.getIDCode();
	    if (idCode !== this.idCode) {
	      // no way for optimisation
	      return new TopicMolecule(molecule);
	    }
	    const topicMolecule = new TopicMolecule(molecule);
	    topicMolecule.cache = {
	      canonizedDiaIDs: this.cache.canonizedDiaIDs,
	      canonizedHoseCodes: this.cache.canonizedHoseCodes
	    };
	    return topicMolecule;
	  }
	  /**
	   * Returns a molecule with all the hydrogens added. The order is NOT canonized
	   */
	  get moleculeWithH() {
	    if (this.cache.moleculeWithH) return this.cache.moleculeWithH;
	    this.cache.moleculeWithH = (0, getMoleculeWithH_1.getMoleculeWithH)(this.molecule);
	    return this.cache.moleculeWithH;
	  }
	  get xMolecule() {
	    if (this.cache.xMolecule) return this.cache.xMolecule;
	    this.cache.xMolecule = (0, getXMolecule_1.getXMolecule)(this.moleculeWithH);
	    return this.cache.xMolecule;
	  }
	  /**
	   * This is related to the current moleculeWithH. The order is NOT canonized
	   */
	  get diaIDs() {
	    if (this.cache.diaIDs) return this.cache.diaIDs;
	    const diaIDs = [];
	    for (let i = 0; i < this.moleculeWithH.getAllAtoms(); i++) {
	      diaIDs.push(this.canonizedDiaIDs[this.finalRanks[i]]);
	    }
	    this.cache.diaIDs = diaIDs;
	    return diaIDs;
	  }
	  /**
	   * We return the atomIDs corresponding to the specified diaID as well has the attached hydrogens or heavy atoms
	   * @param diaID
	   * @returns
	   */
	  getDiaIDsObject() {
	    return groupDiastereotopicAtomIDsAsObject(this.diaIDs, this.molecule, this.moleculeWithH);
	  }
	  /**
	   * This is related to the current moleculeWithH. The order is NOT canonized
	   */
	  get hoseCodes() {
	    if (this.cache.hoseCodes) return this.cache.hoseCodes;
	    const hoseCodes = [];
	    for (let i = 0; i < this.moleculeWithH.getAllAtoms(); i++) {
	      hoseCodes.push(this.canonizedHoseCodes[this.finalRanks[i]]);
	    }
	    this.cache.hoseCodes = hoseCodes;
	    return hoseCodes;
	  }
	  get canonizedDiaIDs() {
	    if (this.cache.canonizedDiaIDs) return this.cache.canonizedDiaIDs;
	    this.cache.canonizedDiaIDs = (0, getCanonizedDiaIDs_1.getCanonizedDiaIDs)(this);
	    return this.cache.canonizedDiaIDs;
	  }
	  get canonizedHoseCodes() {
	    if (this.cache.canonizedHoseCodes) {
	      return this.cache.canonizedHoseCodes;
	    }
	    this.cache.canonizedHoseCodes = (0, getCanonizedHoseCodes_1.getCanonizedHoseCodes)(this);
	    return this.cache.canonizedHoseCodes;
	  }
	  /**
	   * Returns the distance matrix for the current moleculeWithH
	   */
	  get distanceMatrix() {
	    return (0, getConnectivityMatrix_js_1$1.getConnectivityMatrix)(this.moleculeWithH, {
	      pathLength: true
	    });
	  }
	  get diaIDsAndInfo() {
	    if (this.cache.diaIDsAndInfo) return this.cache.diaIDsAndInfo;
	    this.cache.diaIDsAndInfo = (0, getDiaIDsAndInfo_1.getDiaIDsAndInfo)(this, this.canonizedDiaIDs);
	    return this.cache.diaIDsAndInfo;
	  }
	  /**
	   * Returns symmetryRanks for all the atoms including hydrogens. Those ranks
	   * deals with topicity and is related to the current moleculeWithH.
	   * In order to calculate the ranks we replace all the
	   * hydrogens with a X atom.
	   */
	  get heterotopicSymmetryRanks() {
	    if (this.cache.heterotopicSymmetryRanks) {
	      return this.cache.heterotopicSymmetryRanks;
	    }
	    this.cache.heterotopicSymmetryRanks = (0, getHeterotopicSymmetryRanks_1.getHeterotopicSymmetryRanks)(this.xMolecule);
	    return [...this.cache.heterotopicSymmetryRanks];
	  }
	  /**
	   * Returns finalRanks for all the atoms including hydrogens. Those ranks
	   * deals with topicity and is related to the current moleculeWithH.
	   * All the atoms have a unique identifier.j
	   * In order to calculate the ranks we replace all the
	   * hydrogens with a X atom.
	   */
	  get finalRanks() {
	    if (this.cache.finalRanks) return this.cache.finalRanks;
	    this.cache.finalRanks = (0, getHeterotopicSymmetryRanks_1.getFinalRanks)(this.xMolecule);
	    return this.cache.finalRanks;
	  }
	  toMolfileWithH(options = {}) {
	    const {
	      version = 2
	    } = options;
	    if (version === 2) {
	      return this.moleculeWithH.toMolfile();
	    }
	    return this.moleculeWithH.toMolfileV3();
	  }
	  /**
	   * Returns an array of objects containing the oclID and the corresponding hydrogens and atoms
	   * for the specified atomLabel (if any)
	   * This always applied to the molecule with expanded hydrogens and chirality
	   * @param options
	   * @returns
	   */
	  getGroupedDiastereotopicAtomIDs(options = {}) {
	    return groupDiastereotopicAtomIDs$2(this.diaIDs, this.moleculeWithH, options);
	  }
	  /**
	   * This method returns a mapping between the diaIDs of the current molecule.
	   * It expects that the initial molfile and the final molfile contains atomMapNo
	   * in order to track which atom becomes which one.
	   */
	  getDiaIDsMapping(molecule) {
	    const topicMolecule = new TopicMolecule(molecule);
	    const originalDiaIDs = this.diaIDsAndInfo.filter(diaID => diaID.atomMapNo);
	    const destinationDiaIDs = topicMolecule.diaIDsAndInfo.filter(diaID => diaID.atomMapNo);
	    const mapping = {};
	    // we first check all the atoms present in the molfile
	    for (const destinationDiaID of destinationDiaIDs) {
	      const originalDiaID = originalDiaIDs.find(diaID => diaID.atomMapNo === destinationDiaID.atomMapNo);
	      const newIDCode = destinationDiaID.idCode;
	      const oldIDCode = originalDiaID.idCode;
	      if (oldIDCode in mapping) {
	        if (mapping[oldIDCode] !== newIDCode) {
	          mapping[oldIDCode] = undefined;
	        }
	      } else {
	        mapping[oldIDCode] = newIDCode;
	      }
	    }
	    // we now check all the attached hydrogens that are not defined in the molfile and were not yet mapped
	    for (const destinationDiaID of destinationDiaIDs) {
	      const originalDiaID = originalDiaIDs.find(diaID => diaID.atomMapNo === destinationDiaID.atomMapNo);
	      for (let i = 0; i < originalDiaID.attachedHydrogensIDCodes.length; i++) {
	        const oldHydrogenIDCode = originalDiaID.attachedHydrogensIDCodes[i];
	        if (mapping[oldHydrogenIDCode]) continue;
	        const newHydrogenIDCode = destinationDiaID.attachedHydrogensIDCodes[i];
	        if (oldHydrogenIDCode && newHydrogenIDCode) {
	          if (oldHydrogenIDCode in mapping) {
	            if (mapping[oldHydrogenIDCode] !== newHydrogenIDCode) {
	              mapping[oldHydrogenIDCode] = undefined;
	            }
	          } else {
	            mapping[oldHydrogenIDCode] = newHydrogenIDCode;
	          }
	        }
	      }
	    }
	    return mapping;
	  }
	}
	TopicMolecule$1.TopicMolecule = TopicMolecule;
	function groupDiastereotopicAtomIDs$2(diaIDs, molecule, options = {}) {
	  const diaIDsObject = groupDiastereotopicAtomIDsAsObject(diaIDs, molecule, molecule, options);
	  return Object.keys(diaIDsObject).map(key => diaIDsObject[key]);
	}
	TopicMolecule$1.groupDiastereotopicAtomIDs = groupDiastereotopicAtomIDs$2;
	function groupDiastereotopicAtomIDsAsObject(diaIDs, molecule, moleculeWithH, options = {}) {
	  const {
	    atomLabel
	  } = options;
	  const diaIDsObject = {};
	  for (let i = 0; i < diaIDs.length; i++) {
	    if (!atomLabel || moleculeWithH.getAtomLabel(i) === atomLabel) {
	      const diaID = diaIDs[i];
	      if (!diaIDsObject[diaID]) {
	        diaIDsObject[diaID] = {
	          counter: 0,
	          atoms: [],
	          oclID: diaID,
	          atomLabel: moleculeWithH.getAtomLabel(i),
	          heavyAtoms: [],
	          attachedHydrogens: [],
	          existingAtoms: []
	        };
	      }
	      if (moleculeWithH.getAtomicNo(i) === 1) {
	        const connected = moleculeWithH.getConnAtom(i, 0);
	        if (!diaIDsObject[diaID].heavyAtoms.includes(connected)) {
	          diaIDsObject[diaID].heavyAtoms.push(connected);
	        }
	        if (molecule.getAtomicNo(i)) {
	          diaIDsObject[diaID].existingAtoms.push(i);
	        } else if (!diaIDsObject[diaID].existingAtoms.includes(connected)) {
	          diaIDsObject[diaID].existingAtoms.push(connected);
	        }
	      } else {
	        for (let j = 0; j < moleculeWithH.getAllConnAtoms(i); j++) {
	          const connected = moleculeWithH.getConnAtom(i, j);
	          if (moleculeWithH.getAtomicNo(connected) === 1) {
	            diaIDsObject[diaID].attachedHydrogens.push(connected);
	          }
	        }
	        diaIDsObject[diaID].existingAtoms.push(i);
	      }
	      diaIDsObject[diaID].counter++;
	      diaIDsObject[diaID].atoms.push(i);
	    }
	  }
	  for (const diaID in diaIDsObject) {
	    diaIDsObject[diaID].existingAtoms.sort((a, b) => a - b);
	    diaIDsObject[diaID].attachedHydrogens.sort((a, b) => a - b);
	  }
	  return diaIDsObject;
	}

	var getDiastereotopicAtomIDs$1 = {};

	var getSymmetryRanks$1 = {};

	Object.defineProperty(getSymmetryRanks$1, "__esModule", {
	  value: true
	});
	getSymmetryRanks$1.getSymmetryRanks = void 0;
	const getXAtomicNumber_js_1 = getXAtomicNumber$1;
	/**
	 * Returns an array of symmetry ranks.
	 * @param {import('openchemlib').Molecule} molecule An instance of a molecule
	 * @returns
	 */
	function getSymmetryRanks(molecule) {
	  const {
	    Molecule
	  } = molecule.getOCL();
	  const xAtomNumber = (0, getXAtomicNumber_js_1.getXAtomicNumber)(molecule);
	  // most of the molecules have some symmetry
	  const internalMolecule = molecule.getCompactCopy();
	  for (let i = 0; i < internalMolecule.getAllAtoms(); i++) {
	    // hydrogens are not taken into account during canonization, we need to change them with an atom with a valence of 1
	    if (internalMolecule.getAtomicNo(i) === 1) {
	      internalMolecule.setAtomicNo(i, xAtomNumber);
	    }
	  }
	  internalMolecule.ensureHelperArrays(Molecule.cHelperSymmetryStereoHeterotopicity);
	  const symmetryRanks = [];
	  for (let i = 0; i < molecule.getAllAtoms(); i++) {
	    symmetryRanks.push(internalMolecule.getSymmetryRank(i));
	  }
	  return symmetryRanks;
	}
	getSymmetryRanks$1.getSymmetryRanks = getSymmetryRanks;

	Object.defineProperty(getDiastereotopicAtomIDs$1, "__esModule", {
	  value: true
	});
	getDiastereotopicAtomIDs$1.getDiastereotopicAtomIDs = void 0;
	const getSymmetryRanks_1 = getSymmetryRanks$1;
	const makeRacemic_1$2 = makeRacemic$1;
	const tagAtom_1$3 = tagAtom$1;
	const ensureHeterotopicChiralBonds_1 = ensureHeterotopicChiralBonds$1;
	/**
	 * Returns an array of diastereotopic ID (as idCode)
	 * @param {import('openchemlib').Molecule} molecule
	 */
	function getDiastereotopicAtomIDs(molecule) {
	  const {
	    Molecule
	  } = molecule.getOCL();
	  (0, ensureHeterotopicChiralBonds_1.ensureHeterotopicChiralBonds)(molecule);
	  const symmetryRanks = (0, getSymmetryRanks_1.getSymmetryRanks)(molecule);
	  const numberAtoms = molecule.getAllAtoms();
	  const ids = [];
	  const cache = {};
	  for (let iAtom = 0; iAtom < numberAtoms; iAtom++) {
	    const rank = symmetryRanks[iAtom];
	    if (rank && cache[rank]) {
	      ids[iAtom] = cache[rank];
	      continue;
	    }
	    const tempMolecule = molecule.getCompactCopy();
	    (0, tagAtom_1$3.tagAtom)(tempMolecule, iAtom);
	    (0, makeRacemic_1$2.makeRacemic)(tempMolecule);
	    // We need to ensure the helper array in order to get correctly the result of racemisation
	    ids[iAtom] = tempMolecule.getCanonizedIDCode(Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS);
	    cache[rank] = ids[iAtom];
	  }
	  return ids;
	}
	getDiastereotopicAtomIDs$1.getDiastereotopicAtomIDs = getDiastereotopicAtomIDs;

	var getDiastereotopicAtomIDsAndH$1 = {};

	Object.defineProperty(getDiastereotopicAtomIDsAndH$1, "__esModule", {
	  value: true
	});
	getDiastereotopicAtomIDsAndH$1.getDiastereotopicAtomIDsAndH = void 0;
	const getDiastereotopicAtomIDs_1$2 = getDiastereotopicAtomIDs$1;
	/**
	 *
	 * @param {import('openchemlib').Molecule} molecule
	 */
	function getDiastereotopicAtomIDsAndH(molecule) {
	  const OCL = molecule.getOCL();
	  molecule = molecule.getCompactCopy();
	  molecule.addImplicitHydrogens();
	  molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
	  const diaIDs = (0, getDiastereotopicAtomIDs_1$2.getDiastereotopicAtomIDs)(molecule);
	  const newDiaIDs = [];
	  for (let i = 0; i < diaIDs.length; i++) {
	    const diaID = diaIDs[i];
	    const newDiaID = {
	      oclID: diaID,
	      hydrogenOCLIDs: [],
	      nbHydrogens: 0
	    };
	    if (molecule.getAtomicNo(i) === 1) {
	      const atom = molecule.getConnAtom(i, 0);
	      newDiaID.heavyAtom = diaIDs[atom];
	    }
	    for (let j = 0; j < molecule.getAllConnAtoms(i); j++) {
	      const atom = molecule.getConnAtom(i, j);
	      if (molecule.getAtomicNo(atom) === 1) {
	        newDiaID.nbHydrogens++;
	        if (!newDiaID.hydrogenOCLIDs.includes(diaIDs[atom])) {
	          newDiaID.hydrogenOCLIDs.push(diaIDs[atom]);
	        }
	      }
	    }
	    newDiaIDs.push(newDiaID);
	  }
	  return newDiaIDs;
	}
	getDiastereotopicAtomIDsAndH$1.getDiastereotopicAtomIDsAndH = getDiastereotopicAtomIDsAndH;

	var getGroupedDiastereotopicAtomIDs$1 = {};

	var groupDiastereotopicAtomIDs$1 = {};

	Object.defineProperty(groupDiastereotopicAtomIDs$1, "__esModule", {
	  value: true
	});
	groupDiastereotopicAtomIDs$1.groupDiastereotopicAtomIDs = void 0;
	function groupDiastereotopicAtomIDs(diaIDs, molecule, options = {}) {
	  const {
	    atomLabel
	  } = options;
	  const diaIDsObject = {};
	  for (let i = 0; i < diaIDs.length; i++) {
	    if (!atomLabel || molecule.getAtomLabel(i) === atomLabel) {
	      const diaID = diaIDs[i];
	      if (!diaIDsObject[diaID]) {
	        diaIDsObject[diaID] = {
	          counter: 0,
	          atoms: [],
	          oclID: diaID,
	          atomLabel: molecule.getAtomLabel(i)
	        };
	      }
	      diaIDsObject[diaID].counter++;
	      diaIDsObject[diaID].atoms.push(i);
	    }
	  }
	  return Object.keys(diaIDsObject).map(key => diaIDsObject[key]);
	}
	groupDiastereotopicAtomIDs$1.groupDiastereotopicAtomIDs = groupDiastereotopicAtomIDs;

	Object.defineProperty(getGroupedDiastereotopicAtomIDs$1, "__esModule", {
	  value: true
	});
	getGroupedDiastereotopicAtomIDs$1.getGroupedDiastereotopicAtomIDs = void 0;
	const getDiastereotopicAtomIDs_1$1 = getDiastereotopicAtomIDs$1;
	const groupDiastereotopicAtomIDs_1 = groupDiastereotopicAtomIDs$1;
	/**
	 * This function groups the diasterotopic atomIds of the molecule based on equivalence of atoms. The output object contains
	 * a set of chemically equivalent atoms(element.atoms) and the groups of magnetically equivalent atoms (element.magneticGroups)
	 * @param {import('openchemlib').Molecule} molecule
	 * @param {object} [options={}]
	 * @param {string} [options.atomLabel] Select atoms of the given atomLabel. By default it returns all the explicit atoms in the molecule
	 * @returns {Array}
	 */
	function getGroupedDiastereotopicAtomIDs(molecule, options = {}) {
	  const diaIDs = (0, getDiastereotopicAtomIDs_1$1.getDiastereotopicAtomIDs)(molecule);
	  return (0, groupDiastereotopicAtomIDs_1.groupDiastereotopicAtomIDs)(diaIDs, molecule, options);
	}
	getGroupedDiastereotopicAtomIDs$1.getGroupedDiastereotopicAtomIDs = getGroupedDiastereotopicAtomIDs;

	var getDiastereotopicAtomIDsFromMolfile$1 = {};

	Object.defineProperty(getDiastereotopicAtomIDsFromMolfile$1, "__esModule", {
	  value: true
	});
	getDiastereotopicAtomIDsFromMolfile$1.getDiastereotopicAtomIDsFromMolfile = void 0;
	const getDiastereotopicAtomIDsAndH_js_1 = getDiastereotopicAtomIDsAndH$1;
	/**
	 * Parse a molfile and returns an object containing the molecule, the map and the diaIDs
	 * The map allows to reload properties assigned to the atom molfile
	 * Please take care than numbering of atoms starts at 0 !
	 * @param {import('openchemlib')} OCL - openchemlib library
	 * @param {string} molfile
	 * @returns
	 */
	function getDiastereotopicAtomIDsFromMolfile(OCL, molfile) {
	  const {
	    map,
	    molecule
	  } = OCL.Molecule.fromMolfileWithAtomMap(molfile);
	  const diaIDsArray = (0, getDiastereotopicAtomIDsAndH_js_1.getDiastereotopicAtomIDsAndH)(molecule);
	  const diaIDs = {};
	  for (let i = 0; i < map.length; i++) {
	    diaIDs[map[i]] = {
	      source: map[i],
	      destination: i,
	      ...diaIDsArray[i]
	    };
	  }
	  return {
	    map: diaIDs,
	    molecule,
	    diaIDs: diaIDsArray
	  };
	}
	getDiastereotopicAtomIDsFromMolfile$1.getDiastereotopicAtomIDsFromMolfile = getDiastereotopicAtomIDsFromMolfile;

	var toDiastereotopicSVG$1 = {};

	Object.defineProperty(toDiastereotopicSVG$1, "__esModule", {
	  value: true
	});
	toDiastereotopicSVG$1.toDiastereotopicSVG = void 0;
	const getDiastereotopicAtomIDsAndH_1$1 = getDiastereotopicAtomIDsAndH$1;
	/**
	 * Returns a SVG
	 * @param {*} molecule
	 * @param {*} [options={}]
	 */
	function toDiastereotopicSVG(molecule, options = {}) {
	  const {
	    width = 300,
	    height = 200,
	    prefix = 'ocl',
	    heavyAtomHydrogen = false
	  } = options;
	  let svg = options.svg;
	  let diaIDs = [];
	  const hydrogenInfo = {};
	  (0, getDiastereotopicAtomIDsAndH_1$1.getDiastereotopicAtomIDsAndH)(molecule).forEach(line => {
	    hydrogenInfo[line.oclID] = line;
	  });
	  if (heavyAtomHydrogen) {
	    for (let i = 0; i < molecule.getAtoms(); i++) {
	      diaIDs.push([]);
	    }
	    const groupedDiaIDs = molecule.getGroupedDiastereotopicAtomIDs();
	    groupedDiaIDs.forEach(diaID => {
	      if (hydrogenInfo[diaID.oclID] && hydrogenInfo[diaID.oclID].nbHydrogens > 0) {
	        diaID.atoms.forEach(atom => {
	          hydrogenInfo[diaID.oclID].hydrogenOCLIDs.forEach(id => {
	            if (!diaIDs[atom * 1].includes(id)) diaIDs[atom].push(id);
	          });
	        });
	      }
	    });
	  } else {
	    diaIDs = molecule.getDiastereotopicAtomIDs().map(a => [a]);
	  }
	  if (!svg) svg = molecule.toSVG(width, height, prefix);
	  svg = svg.replace(/Atom:[0-9]+"/g, value => {
	    const atom = value.replace(/[^0-9]/g, '');
	    return `${value} data-diaid="${diaIDs[atom].join(',')}"`;
	  });
	  return svg;
	}
	toDiastereotopicSVG$1.toDiastereotopicSVG = toDiastereotopicSVG;

	var getHoseCodes$1 = {};

	Object.defineProperty(getHoseCodes$1, "__esModule", {
	  value: true
	});
	getHoseCodes$1.getHoseCodes = void 0;
	const ensureHeterotopicChiralBonds_js_1$3 = ensureHeterotopicChiralBonds$1;
	const tagAtom_1$2 = tagAtom$1;
	const getHoseCodesForAtomsInternal_js_1$2 = getHoseCodesForAtomsInternal;
	/**
	 * Returns the hose codes for all atoms in the molecule
	 * @param {*} molecule
	 * @param {object} [options={}]
	 * @param {string[]} [options.atomLabels]
	 * @param {number} [options.minSphereSize=0]
	 * @param {number} [options.maxSphereSize=4]
	 * @returns
	 */
	function getHoseCodes(molecule, options = {}) {
	  const {
	    atomLabels,
	    minSphereSize,
	    maxSphereSize
	  } = options;
	  const {
	    Molecule
	  } = molecule.getOCL();
	  const atomicNumbers = atomLabels?.map(label => Molecule.getAtomicNoFromLabel(label));
	  const internalMolecule = molecule.getCompactCopy();
	  internalMolecule.addImplicitHydrogens();
	  (0, ensureHeterotopicChiralBonds_js_1$3.ensureHeterotopicChiralBonds)(internalMolecule);
	  const hoses = [];
	  for (let i = 0; i < internalMolecule.getAllAtoms(); i++) {
	    if (atomicNumbers && !atomicNumbers.includes(internalMolecule.getAtomicNo(i))) {
	      hoses.push(undefined);
	    } else {
	      const tempMolecule = internalMolecule.getCompactCopy();
	      (0, tagAtom_1$2.tagAtom)(tempMolecule, i);
	      hoses.push((0, getHoseCodesForAtomsInternal_js_1$2.getHoseCodesForAtomsInternal)(tempMolecule, {
	        minSphereSize,
	        maxSphereSize
	      }));
	    }
	  }
	  return hoses;
	}
	getHoseCodes$1.getHoseCodes = getHoseCodes;

	var getHoseCodesAndInfo$1 = {};

	Object.defineProperty(getHoseCodesAndInfo$1, "__esModule", {
	  value: true
	});
	getHoseCodesAndInfo$1.getHoseCodesAndInfo = void 0;
	const ensureHeterotopicChiralBonds_js_1$2 = ensureHeterotopicChiralBonds$1;
	const getConnectivityMatrix_js_1 = getConnectivityMatrix$1;
	const getSymmetryRanks_js_1 = getSymmetryRanks$1;
	const makeRacemic_js_1 = makeRacemic$1;
	const tagAtom_1$1 = tagAtom$1;
	const getHoseCodesForAtomsInternal_js_1$1 = getHoseCodesForAtomsInternal;
	/**
	 * Returns an object containing a molfile, molfile with hydrogens, hoses codes and optionally the diaIDs
	 * and the diaIDs
	 * The map allows to reload properties assigned to the atom molfile
	 * Please take care than numbering of atoms starts at 0 !
	 * @param {import('openchemlib').Molecule} molecule
	 * @param {object} [options={}]
	 * @param {string[]} [options.atomLabels]
	 * @param {boolean} [options.calculateDiastereotopicIDs=true]
	 * @param {number} [options.minSphereSize=0]
	 * @param {number} [options.maxSphereSize=4]
	 * @returns
	 */
	function getHoseCodesAndInfo(molecule, options = {}) {
	  const {
	    minSphereSize,
	    maxSphereSize,
	    calculateDiastereotopicIDs = true
	  } = options;
	  const {
	    Molecule
	  } = molecule.getOCL();
	  molecule = molecule.getCompactCopy();
	  // this will force reordering of the hydrogens to the end, just to have the same order as in the molfile
	  molecule.ensureHelperArrays(Molecule.cHelperNeighbours);
	  const newMolfile = molecule.toMolfile();
	  molecule.addImplicitHydrogens();
	  (0, ensureHeterotopicChiralBonds_js_1$2.ensureHeterotopicChiralBonds)(molecule);
	  molecule.ensureHelperArrays(Molecule.cHelperSymmetryStereoHeterotopicity);
	  const newMolfileWithH = molecule.toMolfile();
	  const symmetryRanks = (0, getSymmetryRanks_js_1.getSymmetryRanks)(molecule);
	  const cache = {};
	  const hoses = [];
	  const diaIDs = [];
	  for (let i = 0; i < molecule.getAllAtoms(); i++) {
	    const rank = symmetryRanks[i];
	    if (rank && cache[rank]) {
	      diaIDs.push(cache[rank].diaID);
	      hoses.push(cache[rank].hose);
	      continue;
	    }
	    const tempMolecule = molecule.getCompactCopy();
	    (0, tagAtom_1$1.tagAtom)(tempMolecule, i);
	    let diaID;
	    if (calculateDiastereotopicIDs) {
	      (0, makeRacemic_js_1.makeRacemic)(tempMolecule);
	      diaID = tempMolecule.getCanonizedIDCode(Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS);
	      diaIDs.push(diaID);
	    }
	    const hose = (0, getHoseCodesForAtomsInternal_js_1$1.getHoseCodesForAtomsInternal)(tempMolecule, {
	      minSphereSize,
	      maxSphereSize
	    });
	    hoses.push(hose);
	    cache[rank] = {
	      diaID,
	      hose
	    };
	  }
	  const distanceMatrix = (0, getConnectivityMatrix_js_1.getConnectivityMatrix)(molecule, {
	    pathLength: true
	  });
	  return {
	    molfile: newMolfile,
	    molfileWithH: newMolfileWithH,
	    hoses,
	    diaIDs: calculateDiastereotopicIDs ? diaIDs : undefined,
	    moleculeWithHydrogens: molecule,
	    distanceMatrix
	  };
	}
	getHoseCodesAndInfo$1.getHoseCodesAndInfo = getHoseCodesAndInfo;

	var getHoseCodesForAtom$1 = {};

	var getHoseCodesForAtoms = {};

	(function (exports) {

	  Object.defineProperty(exports, "__esModule", {
	    value: true
	  });
	  exports.getHoseCodesForAtoms = exports.HOSE_CODE_CUT_C_SP3_SP3 = exports.FULL_HOSE_CODE = void 0;
	  const ensureHeterotopicChiralBonds_js_1 = ensureHeterotopicChiralBonds$1;
	  const tagAtom_1 = tagAtom$1;
	  const getHoseCodesForAtomsInternal_js_1 = getHoseCodesForAtomsInternal;
	  exports.FULL_HOSE_CODE = 1;
	  exports.HOSE_CODE_CUT_C_SP3_SP3 = 2;
	  /**
	   * Returns the hose code for specific atom numbers
	   * @param {import('openchemlib').Molecule} originalMolecule - The OCL molecule to be fragmented
	   * @param {number[]} rootAtoms
	   * @param {object} [options={}]
	   * @param {number} [options.minSphereSize=0] Smallest hose code sphere
	   * @param {number} [options.maxSphereSize=4] Largest hose code sphere
	   * @param {number} [options.kind=FULL_HOSE_CODE] Kind of hose code, default usual sphere
	   */
	  function getHoseCodesForAtoms(originalMolecule, rootAtoms = [], options = {}) {
	    const {
	      minSphereSize = 0,
	      maxSphereSize = 4,
	      kind = exports.FULL_HOSE_CODE
	    } = options;
	    const molecule = originalMolecule.getCompactCopy();
	    // those 2 lines should be done only once
	    molecule.addImplicitHydrogens();
	    (0, ensureHeterotopicChiralBonds_js_1.ensureHeterotopicChiralBonds)(molecule);
	    const allowedCustomLabels = [];
	    for (const rootAtom of rootAtoms) {
	      allowedCustomLabels.push((0, tagAtom_1.tagAtom)(molecule, rootAtom));
	    }
	    return (0, getHoseCodesForAtomsInternal_js_1.getHoseCodesForAtomsInternal)(molecule, {
	      minSphereSize,
	      maxSphereSize,
	      allowedCustomLabels,
	      kind
	    });
	  }
	  exports.getHoseCodesForAtoms = getHoseCodesForAtoms;
	})(getHoseCodesForAtoms);

	Object.defineProperty(getHoseCodesForAtom$1, "__esModule", {
	  value: true
	});
	getHoseCodesForAtom$1.getHoseCodesForAtom = void 0;
	const getHoseCodesForAtoms_js_1 = getHoseCodesForAtoms;
	/**
	 * Returns the hose code for a specific atom number
	 * @param {import('openchemlib').Molecule} originalMolecule
	 * @param {number} rootAtom
	 * @param {object} [options={}]
	 * @param {boolean} [options.isTagged] Specify is the atom is already tagged
	 * @param {number} [options.minSphereSize=0] Smallest hose code sphere
	 * @param {number} [options.maxSphereSize=4] Largest hose code sphere
	 * @param {number} [options.kind=FULL_HOSE_CODE] Kind of hose code, default usual sphere
	 */
	function getHoseCodesForAtom(originalMolecule, rootAtom, options = {}) {
	  return (0, getHoseCodesForAtoms_js_1.getHoseCodesForAtoms)(originalMolecule, [rootAtom], options);
	}
	getHoseCodesForAtom$1.getHoseCodesForAtom = getHoseCodesForAtom;

	var getHoseCodesFromDiastereotopicID$1 = {};

	Object.defineProperty(getHoseCodesFromDiastereotopicID$1, "__esModule", {
	  value: true
	});
	getHoseCodesFromDiastereotopicID$1.getHoseCodesFromDiastereotopicID = void 0;
	const ensureHeterotopicChiralBonds_js_1$1 = ensureHeterotopicChiralBonds$1;
	const getHoseCodesForAtomsInternal_js_1 = getHoseCodesForAtomsInternal;
	/**
	 * Returns the hose code for a specific marked atom
	 * @param {import('openchemlib').Molecule} molecule
	 * @param {object} options
	 */
	function getHoseCodesFromDiastereotopicID(molecule, options = {}) {
	  molecule.addImplicitHydrogens();
	  (0, ensureHeterotopicChiralBonds_js_1$1.ensureHeterotopicChiralBonds)(molecule);
	  return (0, getHoseCodesForAtomsInternal_js_1.getHoseCodesForAtomsInternal)(molecule, options);
	}
	getHoseCodesFromDiastereotopicID$1.getHoseCodesFromDiastereotopicID = getHoseCodesFromDiastereotopicID;

	var getHoseCodesForPath$1 = {};

	Object.defineProperty(getHoseCodesForPath$1, "__esModule", {
	  value: true
	});
	getHoseCodesForPath$1.getHoseCodesForPath = void 0;
	const ensureHeterotopicChiralBonds_js_1 = ensureHeterotopicChiralBonds$1;
	const makeRacemic_1$1 = makeRacemic$1;
	const tagAtom_1 = tagAtom$1;
	let fragment$1;
	/**
	 * Returns the hose code for a specific atom number
	 * @param {import('openchemlib').Molecule} molecule
	 */
	function getHoseCodesForPath(molecule, from, to, maxLength) {
	  const OCL = molecule.getOCL();
	  const originalFrom = from;
	  const originalTo = to;
	  molecule = molecule.getCompactCopy();
	  const originalAtoms = []; // path before renumbering
	  molecule.getPath(originalAtoms, from, to, maxLength + 1);
	  let torsion;
	  if (originalAtoms.length === 4) {
	    torsion = molecule.calculateTorsion(originalAtoms);
	  }
	  const tag1 = (0, tagAtom_1.tagAtom)(molecule, from);
	  const tag2 = (0, tagAtom_1.tagAtom)(molecule, to);
	  molecule.addImplicitHydrogens();
	  (0, ensureHeterotopicChiralBonds_js_1.ensureHeterotopicChiralBonds)(molecule);
	  molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
	  from = -1;
	  to = -1;
	  for (let i = 0; i < molecule.getAllAtoms(); i++) {
	    if (tag1 === tag2) {
	      if (molecule.getAtomCustomLabel(i) === tag1) {
	        if (from === -1) {
	          from = i;
	        } else {
	          to = i;
	        }
	      }
	    } else {
	      if (tag1 === molecule.getAtomCustomLabel(i)) {
	        from = i;
	      }
	      if (tag2 === molecule.getAtomCustomLabel(i)) {
	        to = i;
	      }
	    }
	  }
	  if (!fragment$1) fragment$1 = new OCL.Molecule(0, 0);
	  const atoms = [];
	  molecule.getPath(atoms, from, to, maxLength + 1);
	  let min = 0;
	  let max = 0;
	  const atomMask = new Array(molecule.getAllAtoms()).fill(false);
	  const atomList = new Array(molecule.getAllAtoms()).fill(-1);
	  const hoses = [];
	  for (let sphere = 0; sphere <= 2; sphere++) {
	    if (max === 0) {
	      for (const atom of atoms) {
	        atomMask[atom] = true;
	        atomList[max++] = atom;
	      }
	    } else {
	      let newMax = max;
	      for (let i = min; i < max; i++) {
	        const atom = atomList[i];
	        for (let j = 0; j < molecule.getAllConnAtoms(atom); j++) {
	          const connAtom = molecule.getConnAtom(atom, j);
	          if (!atomMask[connAtom]) {
	            atomMask[connAtom] = true;
	            atomList[newMax++] = connAtom;
	          }
	        }
	      }
	      min = max;
	      max = newMax;
	    }
	    const atomMap = [];
	    molecule.copyMoleculeByAtoms(fragment$1, atomMask, true, atomMap);
	    (0, makeRacemic_1$1.makeRacemic)(fragment$1);
	    const oclID = fragment$1.getCanonizedIDCode(OCL.Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS);
	    hoses.push({
	      sphere,
	      oclID
	    });
	  }
	  return {
	    atoms: originalAtoms,
	    from: originalFrom,
	    to: originalTo,
	    torsion,
	    hoses,
	    length: originalAtoms.length - 1
	  };
	}
	getHoseCodesForPath$1.getHoseCodesForPath = getHoseCodesForPath;

	var createPolymer$1 = {};

	Object.defineProperty(createPolymer$1, "__esModule", {
	  value: true
	});
	createPolymer$1.createPolymer = void 0;
	/**
	 * Create a polymer from a unit, alpha and omega groups
	 * A unit must contain a R1 and a R2
	 * An alpha end group must contain a R1
	 * An omega end group must contain a R2
	 * @param {import('openchemlib').Molecule} molecule an instance of OCL.Molecule
	 * @param {object} options
	 * @param {number} [options.count=10] number of units
	 * @param {boolean} [options.markMonomer=false] mark the different units of the polymer in the atom map
	 * @param {import('openchemlib').Molecule} [options.alpha] alpha end group, default is an hydrogen
	 * @param {import('openchemlib').Molecule} [options.gamma] omega end group, default is an hydrogen
	 */
	function createPolymer(unit, options = {}) {
	  const {
	    count = 10
	  } = options;
	  checkEntity(unit, 'unit');
	  const {
	    Molecule
	  } = unit.getOCL();
	  const {
	    alpha,
	    gamma
	  } = getAlphaGamma(unit, options);
	  checkEntity(alpha, 'alpha');
	  checkEntity(gamma, 'gamma');
	  const {
	    r1AtomicNo,
	    r2AtomicNo
	  } = getR1R2AtomicNo(Molecule);
	  const polymer = alpha.getCompactCopy();
	  polymer.addMolecule(getUnit(unit, 1, options));
	  addBond(polymer, r1AtomicNo, r1AtomicNo);
	  for (let i = 0; i < count - 1; i++) {
	    polymer.addMolecule(getUnit(unit, i + 2, options));
	    addBond(polymer, r2AtomicNo, r1AtomicNo);
	  }
	  polymer.addMolecule(gamma);
	  addBond(polymer, r2AtomicNo, r2AtomicNo);
	  polymer.ensureHelperArrays(Molecule.cHelperNeighbours);
	  // encoding directly in atomNapNo didn't work out because it was removed when deleting atoms
	  for (let i = 0; i < polymer.getAtoms(); i++) {
	    polymer.setAtomMapNo(i, (polymer.getAtomCustomLabel(i) || '').replace(/monomer_/, ''));
	    polymer.setAtomCustomLabel(i, '');
	  }
	  return polymer;
	}
	createPolymer$1.createPolymer = createPolymer;
	function getUnit(unit, index, options) {
	  const {
	    markMonomer = false
	  } = options;
	  if (markMonomer) {
	    unit = unit.getCompactCopy();
	    unit.ensureHelperArrays(unit.getOCL().Molecule.cHelperNeighbours);
	    for (let j = 0; j < unit.getAtoms(); j++) {
	      unit.setAtomCustomLabel(j, `monomer_${index}`);
	    }
	  }
	  return unit;
	}
	function addBond(molecule, firstAtomicNo, secondAtomicNo) {
	  molecule.ensureHelperArrays(molecule.getOCL().Molecule.cHelperNeighbours);
	  let i, j;
	  loop: for (i = 0; i < molecule.getAtoms(); i++) {
	    if (molecule.getAtomicNo(i) === firstAtomicNo) {
	      for (j = i + 1; j < molecule.getAtoms(); j++) {
	        if (molecule.getAtomicNo(j) === secondAtomicNo) {
	          molecule.addBond(molecule.getConnAtom(i, 0), molecule.getConnAtom(j, 0), 1);
	          break loop;
	        }
	      }
	    }
	  }
	  molecule.deleteAtoms([i, j]);
	}
	function checkEntity(unit, kind) {
	  let nbR1 = 1;
	  let nbR2 = 1;
	  switch (kind) {
	    case 'unit':
	      break;
	    case 'alpha':
	      nbR2 = 0;
	      break;
	    case 'gamma':
	      nbR1 = 0;
	      break;
	    default:
	      throw new Error('Unknown kind');
	  }
	  if (!unit) {
	    throw new Error('unit is required');
	  }
	  const {
	    Molecule
	  } = unit.getOCL();
	  // unit must contain ONE R1 and ONE R2
	  const {
	    r1AtomicNo,
	    r2AtomicNo
	  } = getR1R2AtomicNo(Molecule);
	  let r1Count = 0;
	  let r2Count = 0;
	  for (let i = 0; i < unit.getAtoms(); i++) {
	    if (unit.getAtomicNo(i) === r1AtomicNo) {
	      r1Count++;
	    }
	    if (unit.getAtomicNo(i) === r2AtomicNo) {
	      r2Count++;
	    }
	  }
	  if (r1Count !== nbR1) {
	    throw new Error(`${kind} must contain ${nbR1} R1`);
	  }
	  if (r2Count !== nbR2) {
	    throw new Error(`${kind} must contain ${nbR2} R2`);
	  }
	}
	function getAlphaGamma(unit, options) {
	  let {
	    alpha,
	    gamma
	  } = options;
	  const {
	    Molecule
	  } = unit.getOCL();
	  const {
	    r1AtomicNo,
	    r2AtomicNo
	  } = getR1R2AtomicNo(Molecule);
	  if (!alpha) {
	    alpha = Molecule.fromSmiles('CC');
	    alpha.setAtomicNo(0, r1AtomicNo);
	    alpha.setAtomicNo(1, 1);
	  }
	  if (!gamma) {
	    gamma = Molecule.fromSmiles('CC');
	    gamma.setAtomicNo(0, r2AtomicNo);
	    gamma.setAtomicNo(1, 1);
	  }
	  return {
	    alpha,
	    gamma
	  };
	}
	function getR1R2AtomicNo(Molecule) {
	  const r1AtomicNo = Molecule.getAtomicNoFromLabel('R1', Molecule.cPseudoAtomsRGroups);
	  const r2AtomicNo = Molecule.getAtomicNoFromLabel('R2', Molecule.cPseudoAtomsRGroups);
	  return {
	    r1AtomicNo,
	    r2AtomicNo
	  };
	}

	var combineSmiles$1 = {};

	Object.defineProperty(combineSmiles$1, "__esModule", {
	  value: true
	});
	combineSmiles$1.combineSmiles = void 0;
	const MAX_R = 10;
	/**
	 * Generate molecules and calculate predicted properties form a list of smiles and fragments
	 * @param {string} [coreSmiles]
	 * @param {array} [fragments] Array of {smiles,R1,R2,...}
	 * @param {import('openchemlib')} OCL - openchemlib library
	 * @param {object} [options={}]
	 * @param {function} [options.onStep] method to execute each new molecules
	 * @param {boolean} [options.complexity] returns only the number of molecules to evaluate
	 * @return {Promise} promise that resolves to molecules or complexity as a number
	 */
	async function combineSmiles(coreSmiles, fragments, OCL, options = {}) {
	  const {
	    complexity = false
	  } = options;
	  const core = getCore(coreSmiles);
	  const rGroups = getRGroups(core, fragments);
	  if (complexity) {
	    return getComplexity(rGroups);
	  }
	  return generate(core, rGroups, OCL, options);
	}
	combineSmiles$1.combineSmiles = combineSmiles;
	function getComplexity(rGroups) {
	  let complexity = 1;
	  for (const rGroup of rGroups) {
	    complexity *= rGroup.smiles.length;
	  }
	  return complexity;
	}
	async function generate(core, rGroups, OCL, options = {}) {
	  const {
	    onStep
	  } = options;
	  const molecules = {};
	  const sizes = new Array(rGroups.length);
	  const currents = new Array(rGroups.length);
	  for (let i = 0; i < rGroups.length; i++) {
	    sizes[i] = rGroups[i].smiles.length - 1;
	    currents[i] = 0;
	  }
	  let position = 0;
	  let counter = 0;
	  while (true) {
	    counter++;
	    while (position < currents.length) {
	      if (currents[position] < sizes[position]) {
	        if (onStep) {
	          // eslint-disable-next-line no-await-in-loop
	          await onStep(counter);
	        }
	        appendMolecule(molecules, core, rGroups, currents, OCL);
	        currents[position]++;
	        for (let i = 0; i < position; i++) {
	          currents[i] = 0;
	        }
	        position = 0;
	      } else {
	        position++;
	      }
	    }
	    if (position = currents.length) {
	      if (onStep) {
	        // eslint-disable-next-line no-await-in-loop
	        await onStep(counter);
	      }
	      appendMolecule(molecules, core, rGroups, currents, OCL);
	      break;
	    }
	  }
	  return Object.keys(molecules).map(key => molecules[key]).sort((m1, m2) => m1.mw - m2.mw);
	}
	function appendMolecule(molecules, core, rGroups, currents, OCL) {
	  let newSmiles = core.smiles;
	  for (let i = 0; i < currents.length; i++) {
	    newSmiles += `.${rGroups[i].smiles[currents[i]]}`;
	  }
	  const currentMol = OCL.Molecule.fromSmiles(newSmiles);
	  const idCode = currentMol.getIDCode();
	  if (!molecules[idCode]) {
	    const molecule = {};
	    molecules[idCode] = molecule;
	    molecule.smiles = currentMol.toSmiles();
	    molecule.combinedSmiles = newSmiles;
	    molecule.idCode = idCode;
	    molecule.molfile = currentMol.toMolfile();
	    const props = new OCL.MoleculeProperties(currentMol);
	    molecule.nbHAcceptor = props.acceptorCount;
	    molecule.nbHDonor = props.donorCount;
	    molecule.logP = props.logP;
	    molecule.logS = props.logS;
	    molecule.PSA = props.polarSurfaceArea;
	    molecule.nbRottable = props.rotatableBondCount;
	    molecule.nbStereoCenter = props.stereoCenterCount;
	    const mf = currentMol.getMolecularFormula();
	    molecule.mf = mf.formula;
	    molecule.mw = mf.relativeWeight;
	  }
	}
	function getCore(coreSmiles) {
	  const core = {
	    originalSmiles: coreSmiles,
	    smiles: coreSmiles.replace(/\[R(?<group>[1-4])\]/g, '%5$<group>')
	  };
	  for (let i = 0; i < MAX_R; i++) {
	    if (core.originalSmiles.indexOf(`[R${i}]`) > -1) core[`R${i}`] = true;
	  }
	  return core;
	}
	function getRGroups(core, fragments) {
	  const rGroups = {};
	  for (const fragment of fragments) {
	    if (fragment.smiles) {
	      const smiles = updateRPosition(fragment.smiles);
	      for (let i = 0; i < MAX_R; i++) {
	        if (core[`R${i}`]) {
	          // we only consider the R that are in the core
	          if (fragment[`R${i}`]) {
	            if (!rGroups[`R${i}`]) {
	              rGroups[`R${i}`] = {
	                group: `R${i}`,
	                smiles: []
	              };
	            }
	            rGroups[`R${i}`].smiles.push(smiles.replace(/\[R\]/, `(%5${i})`));
	          }
	        }
	      }
	    }
	  }
	  return Object.keys(rGroups).map(key => rGroups[key]);
	}
	function updateRPosition(smiles) {
	  // R group should not be at the beginning
	  if (smiles.indexOf('[R]') !== 0) return smiles;
	  if (smiles.length === 3) return '[H][R]';
	  // we are in trouble ... we need to move the R
	  const newSmiles = smiles.replace('[R]', '');
	  // we need to check where we should put the R group
	  let level = 0;
	  for (let j = 0; j < newSmiles.length; j++) {
	    const currentChar = newSmiles.charAt(j);
	    const currentSubstring = newSmiles.substr(j);
	    if (currentChar === '(') {
	      level++;
	    } else if (currentChar === ')') {
	      level--;
	    } else if (level === 0) {
	      if (currentSubstring.match(/^[a-z]/)) {
	        return `${newSmiles.substr(0, j + 1)}([R])${newSmiles.substr(j + 1)}`;
	      } else if (currentSubstring.match(/^[A-Z][a-z]/)) {
	        return `${newSmiles.substr(0, j + 2)}([R])${newSmiles.substr(j + 2)}`;
	      } else if (currentSubstring.match(/^[A-Z]/)) {
	        return `${newSmiles.substr(0, j + 1)}([R])${newSmiles.substr(j + 1)}`;
	      }
	    }
	  }
	  return smiles;
	}

	var getAtomsInfo$1 = {};

	Object.defineProperty(getAtomsInfo$1, "__esModule", {
	  value: true
	});
	getAtomsInfo$1.getAtomsInfo = void 0;
	const getDiastereotopicAtomIDs_1 = getDiastereotopicAtomIDs$1;
	/**
	 * Returns various information about atoms in the molecule
	 * @param {import('openchemlib').Molecule} [molecule]
	 */
	function getAtomsInfo(molecule) {
	  const OCL = molecule.getOCL();
	  molecule.ensureHelperArrays(OCL.Molecule.cHelperRings);
	  const diaIDs = (0, getDiastereotopicAtomIDs_1.getDiastereotopicAtomIDs)(molecule);
	  const results = [];
	  for (let i = 0; i < diaIDs.length; i++) {
	    const result = {
	      oclID: diaIDs[i],
	      extra: {
	        singleBonds: 0,
	        doubleBonds: 0,
	        tripleBonds: 0,
	        aromaticBonds: 0,
	        cnoHybridation: 0 // should be 1 (sp), 2 (sp2) or 3 (sp3)
	      }
	    };
	    const extra = result.extra;
	    results.push(result);
	    result.abnormalValence = molecule.getAtomAbnormalValence(i); // -1 is normal otherwise specified
	    result.charge = molecule.getAtomCharge(i);
	    result.cipParity = molecule.getAtomCIPParity(i);
	    result.color = molecule.getAtomColor(i);
	    result.customLabel = molecule.getAtomCustomLabel(i);
	    //        result.esrGroup=molecule.getAtomESRGroup(i);
	    //        result.esrType=molecule.getAtomESRType(i);
	    result.atomicNo = molecule.getAtomicNo(i);
	    result.label = molecule.getAtomLabel(i);
	    //        result.list=molecule.getAtomList(i);
	    //        result.listString=molecule.getAtomListString(i);
	    //        result.mapNo=molecule.getAtomMapNo(i);
	    result.mass = molecule.getAtomMass(i);
	    //        result.parity=molecule.getAtomParity(i);
	    //        result.pi=molecule.getAtomPi(i);
	    //        result.preferredStereoBond=molecule.getAtomPreferredStereoBond(i);
	    //        result.queryFeatures=molecule.getAtomQueryFeatures(i);
	    result.radical = molecule.getAtomRadical(i);
	    result.ringBondCount = molecule.getAtomRingBondCount(i);
	    //        result.ringCount=molecule.getAtomRingCount(i);
	    result.ringSize = molecule.getAtomRingSize(i);
	    result.x = molecule.getAtomX(i);
	    result.y = molecule.getAtomY(i);
	    result.z = molecule.getAtomZ(i);
	    result.allHydrogens = molecule.getAllHydrogens(i);
	    result.connAtoms = molecule.getConnAtoms(i);
	    result.allConnAtoms = molecule.getAllConnAtoms(i);
	    result.implicitHydrogens = result.allHydrogens + result.connAtoms - result.allConnAtoms;
	    result.isAromatic = molecule.isAromaticAtom(i);
	    result.isAllylic = molecule.isAllylicAtom(i);
	    result.isStereoCenter = molecule.isAtomStereoCenter(i);
	    result.isRing = molecule.isRingAtom(i);
	    result.isSmallRing = molecule.isSmallRingAtom(i);
	    result.isStabilized = molecule.isStabilizedAtom(i);
	    // todo HACK to circumvent bug in OCL that consider than an hydrogen is connected to itself
	    result.extra.singleBonds = result.atomicNo === 1 ? 0 : result.implicitHydrogens;
	    for (let j = 0; j < molecule.getAllConnAtoms(i); j++) {
	      const bond = molecule.getConnBond(i, j);
	      const bondOrder = molecule.getBondOrder(bond);
	      if (molecule.isAromaticBond(bond)) {
	        extra.aromaticBonds++;
	      } else if (bondOrder === 1) {
	        // not an hydrogen
	        extra.singleBonds++;
	      } else if (bondOrder === 2) {
	        extra.doubleBonds++;
	      } else if (bondOrder === 3) {
	        extra.tripleBonds++;
	      }
	    }
	    result.extra.totalBonds = result.extra.singleBonds + result.extra.doubleBonds + result.extra.tripleBonds + result.extra.aromaticBonds;
	    if (result.atomicNo === 6) {
	      result.extra.cnoHybridation = result.extra.totalBonds - 1;
	    } else if (result.atomicNo === 7) {
	      result.extra.cnoHybridation = result.extra.totalBonds;
	    } else if (result.atomicNo === 8) {
	      result.extra.cnoHybridation = result.extra.totalBonds + 1;
	    } else if (result.atomicNo === 1) {
	      const connectedAtom = molecule.getAllConnAtoms(i) === 0 ? 0 : molecule.getAtomicNo(molecule.getConnAtom(i, 0));
	      result.extra.hydrogenOnAtomicNo = connectedAtom;
	      if (connectedAtom === 7 || connectedAtom === 8) {
	        result.extra.labileHydrogen = true;
	      }
	    }
	  }
	  return results;
	}
	getAtomsInfo$1.getAtomsInfo = getAtomsInfo;

	var getImplicitHydrogensCount$1 = {};

	Object.defineProperty(getImplicitHydrogensCount$1, "__esModule", {
	  value: true
	});
	getImplicitHydrogensCount$1.getImplicitHydrogensCount = void 0;
	function getImplicitHydrogensCount(molecule, atomID) {
	  molecule.ensureHelperArrays(molecule.getOCL().Molecule.cHelperNeighbours);
	  return molecule.getAllHydrogens(atomID) + molecule.getConnAtoms(atomID) - molecule.getAllConnAtoms(atomID);
	}
	getImplicitHydrogensCount$1.getImplicitHydrogensCount = getImplicitHydrogensCount;

	var getMF$1 = {};

	/**
	 * Implementation of the Hill system for sorting atoms
	 * https://en.wikipedia.org/wiki/Chemical_formula#Hill_system
	 * @param {string} a - first atom to compare
	 * @param {string} b - second atom to compare
	 * @returns
	 */

	function atomSorter(a, b) {
	  if (a === b) return 0;
	  if (a === 'C') return -1;
	  if (b === 'C') return 1;
	  if (a === 'H') return -1;
	  if (b === 'H') return 1;
	  if (a < b) return -1;
	  return 1;
	}

	var src$1 = /*#__PURE__*/Object.freeze({
		__proto__: null,
		atomSorter: atomSorter
	});

	var require$$0$1 = /*@__PURE__*/getAugmentedNamespace(src$1);

	Object.defineProperty(getMF$1, "__esModule", {
	  value: true
	});
	getMF$1.getMF = void 0;
	const atom_sorter_1 = require$$0$1;
	/**
	 * Calculate the molecular formula in 'chemcalc' notation taking into account fragments, isotopes and charges
	 * @param {import('openchemlib').Molecule} molecule an instance of OCL.Molecule
	 * @returns {object}
	 */
	function getMF(molecule) {
	  const entries = molecule.getFragments();
	  const result = {};
	  let parts = [];
	  const allAtoms = [];
	  entries.forEach(entry => {
	    const mf = getFragmentMF(entry, allAtoms);
	    parts.push(mf);
	  });
	  const counts = {};
	  for (const part of parts) {
	    if (!counts[part]) counts[part] = 0;
	    counts[part]++;
	  }
	  parts = [];
	  for (const key of Object.keys(counts).sort()) {
	    if (counts[key] > 1) {
	      parts.push(counts[key] + key);
	    } else {
	      parts.push(key);
	    }
	  }
	  result.parts = parts;
	  result.mf = toMFString(allAtoms);
	  return result;
	}
	getMF$1.getMF = getMF;
	function getFragmentMF(molecule, allAtoms) {
	  const atoms = [];
	  for (let i = 0; i < molecule.getAllAtoms(); i++) {
	    const atom = {};
	    atom.charge = molecule.getAtomCharge(i);
	    atom.label = molecule.getAtomLabel(i);
	    atom.mass = molecule.getAtomMass(i);
	    atom.implicitHydrogens = molecule.getImplicitHydrogens(i);
	    atoms.push(atom);
	    allAtoms.push(atom);
	  }
	  return toMFString(atoms);
	}
	function toMFString(atoms) {
	  let charge = 0;
	  const mfs = {};
	  for (const atom of atoms) {
	    let label = atom.label;
	    charge += atom.charge;
	    if (atom.mass) {
	      label = `[${atom.mass}${label}]`;
	    }
	    const mfAtom = mfs[label];
	    if (!mfAtom) {
	      mfs[label] = 0;
	    }
	    mfs[label] += 1;
	    if (atom.implicitHydrogens) {
	      if (!mfs.H) mfs.H = 0;
	      mfs.H += atom.implicitHydrogens;
	    }
	  }
	  let mf = '';
	  const keys = Object.keys(mfs).sort(atom_sorter_1.atomSorter);
	  for (const key of keys) {
	    mf += key;
	    if (mfs[key] > 1) mf += mfs[key];
	  }
	  if (charge > 0) {
	    mf += `(+${charge > 1 ? charge : ''})`;
	  } else if (charge < 0) {
	    mf += `(${charge < -1 ? charge : '-'})`;
	  }
	  return mf;
	}

	var getCharge$1 = {};

	/**
	 * Returns the charge of a molecule
	 * @param {import('openchemlib').Molecule} molecule an instance of OCL.Molecule
	 * @returns {number}
	 */
	Object.defineProperty(getCharge$1, "__esModule", {
	  value: true
	});
	getCharge$1.getCharge = void 0;
	function getCharge(molecule) {
	  let charge = 0;
	  for (let i = 0; i < molecule.getAllAtoms(); i++) {
	    charge += molecule.getAtomCharge(i);
	  }
	  return charge;
	}
	getCharge$1.getCharge = getCharge;

	var getProperties$1 = {};

	Object.defineProperty(getProperties$1, "__esModule", {
	  value: true
	});
	getProperties$1.getProperties = void 0;
	let toxicityPredictor;
	let druglikenessPredictor;
	/**
	 *
	 * @param {import('openchemlib').Molecule} molecule
	 * @param {object} [options={}]
	 * @param {boolean} [options.includeToxicities=false]
	 * @param {boolean} [options.includeDruglikeness=false]
	 */
	function getProperties(molecule, options = {}) {
	  const {
	    includeToxicities = false,
	    includeDruglikeness = false
	  } = options;
	  const OCL = molecule.getOCL();
	  if (!OCL.MoleculeProperties) {
	    throw new Error('OCL.MoleculeProperties is not defined');
	  }
	  const props = new OCL.MoleculeProperties(molecule);
	  const moleculeFormula = molecule.getMolecularFormula();
	  const result = {
	    acceptorCount: props.acceptorCount,
	    donorCount: props.donorCount,
	    logP: props.logP,
	    logS: props.logS,
	    polarSurfaceArea: props.polarSurfaceArea,
	    rotatableBondCount: props.rotatableBondCount,
	    stereoCenterCount: props.stereoCenterCount,
	    mw: moleculeFormula.relativeWeight,
	    mf: moleculeFormula.formula
	  };
	  if (includeToxicities) {
	    const {
	      ToxicityPredictor
	    } = molecule.getOCL();
	    if (!ToxicityPredictor) {
	      throw new Error('OCL.ToxicityPredictor is not defined');
	    }
	    if (!toxicityPredictor) {
	      toxicityPredictor = new ToxicityPredictor();
	    }
	    result.mutagenic = toxicityPredictor.assessRisk(molecule, ToxicityPredictor.TYPE_MUTAGENIC);
	    result.tumorigenic = toxicityPredictor.assessRisk(molecule, ToxicityPredictor.TYPE_TUMORIGENIC);
	    result.irritant = toxicityPredictor.assessRisk(molecule, ToxicityPredictor.TYPE_IRRITANT);
	    result.reproductiveEffective = toxicityPredictor.assessRisk(molecule, ToxicityPredictor.TYPE_REPRODUCTIVE_EFFECTIVE);
	  }
	  if (includeDruglikeness) {
	    const {
	      DruglikenessPredictor
	    } = molecule.getOCL();
	    if (!DruglikenessPredictor) {
	      throw new Error('OCL.DruglikenessPredictor is not defined');
	    }
	    if (!druglikenessPredictor) {
	      druglikenessPredictor = new DruglikenessPredictor();
	    }
	    result.drugLikeness = druglikenessPredictor.assessDruglikeness(molecule);
	  }
	  if (result.drugLikeness !== undefined && result.mutagenic !== undefined) {
	    result.drugScore = OCL.DrugScoreCalculator.calculate(result.logP, result.polarSurfaceArea, result.mw, result.drugLikeness, [result.mutagenic, result.tumurogenic, result.irritant, result.reproductiveEffective]);
	  }
	  return result;
	}
	getProperties$1.getProperties = getProperties;

	var getAtoms$1 = {};

	/**
	 * Calculate the molecular formula in 'chemcalc' notation taking into account fragments, isotopes and charges
	 * @param {OCL.Molecule} [molecule] an instance of OCL.Molecule
	 * @returns {object}
	 */
	Object.defineProperty(getAtoms$1, "__esModule", {
	  value: true
	});
	getAtoms$1.getAtoms = void 0;
	function getAtoms(molecule) {
	  const entries = molecule.getFragments();
	  const atoms = {};
	  const result = {
	    atoms,
	    parts: []
	  };
	  entries.forEach(entry => {
	    const part = {};
	    result.parts.push(part);
	    appendAtomPart(entry, atoms, part);
	  });
	  return result;
	}
	getAtoms$1.getAtoms = getAtoms;
	function appendAtomPart(molecule, atoms, part) {
	  for (let i = 0; i < molecule.getAllAtoms(); i++) {
	    const label = molecule.getAtomLabel(i);
	    if (!atoms[label]) {
	      atoms[label] = 0;
	    }
	    atoms[label] += 1;
	    if (!part[label]) {
	      part[label] = 0;
	    }
	    part[label] += 1;
	    const implicitHydrogens = molecule.getImplicitHydrogens(i);
	    if (implicitHydrogens) {
	      if (!atoms.H) {
	        atoms.H = 0;
	      }
	      atoms.H += implicitHydrogens;
	      if (!part.H) {
	        part.H = 0;
	      }
	      part.H += implicitHydrogens;
	    }
	  }
	}

	var nbOH$1 = {};

	/**
	 * Return the number of Hydroxyl groups in a molecule or fragment
	 * @param {import('openchemlib').Molecule} molecule
	 * @returns {number} 'Number of Hydroxyl groups'
	 */
	Object.defineProperty(nbOH$1, "__esModule", {
	  value: true
	});
	nbOH$1.nbOH = void 0;
	function nbOH(molecule) {
	  let counter = 0;
	  for (let i = 0; i < molecule.getAllAtoms(); i++) {
	    if (molecule.getAtomicNo(i) === 6) {
	      let carbonyl = false;
	      let hydroxyl = false;
	      let carbonOrHydrogen = true;
	      for (let neighbour = 0; neighbour < molecule.getConnAtoms(i); neighbour++) {
	        const neighbourAtom = molecule.getConnAtom(i, neighbour);
	        const neighbourBond = molecule.getConnBond(i, neighbour);
	        if (molecule.getAtomicNo(neighbourAtom) === 8) {
	          if (molecule.getBondOrder(neighbourBond) === 1 && molecule.getAllHydrogens(neighbourAtom) > 0) {
	            // If there is more than a Hydroxyl in the same carbon atome they are not couted as Hydroxyl groups
	            if (hydroxyl) {
	              hydroxyl = false;
	              break;
	            }
	            hydroxyl = true;
	          } else if (molecule.getBondOrder(neighbourBond) === 2) {
	            // If there is Carbonyl group on the same carbon atom it is not couted as Hydroxyl group
	            carbonyl = true;
	          }
	        } else if (
	        // If there is not at least one carbon or hydrogen as neighbour atom it is not counted as Hydroxyl group
	        molecule.getAtomicNo(neighbourAtom) !== 6 && molecule.getAtomicNo(neighbourAtom) !== 1) {
	          carbonOrHydrogen = false;
	        }
	      }
	      if (!carbonyl && hydroxyl && carbonOrHydrogen) counter++;
	    }
	  }
	  return counter;
	}
	nbOH$1.nbOH = nbOH;

	var nbCOOH$1 = {};

	/**
	 * Return the number of Carboxyl groups in a molecule or fragment
	 * @param {import('openchemlib').Molecule} molecule
	 * @returns {number} 'Number of Carboxyl groups'
	 */
	Object.defineProperty(nbCOOH$1, "__esModule", {
	  value: true
	});
	nbCOOH$1.nbCOOH = void 0;
	function nbCOOH(molecule) {
	  let counter = 0;
	  for (let i = 0; i < molecule.getAllAtoms(); i++) {
	    if (molecule.getAtomicNo(i) === 6) {
	      let carbonyl = false;
	      let hydroxyl = false;
	      let carbonOrHydrogen = true;
	      for (let neighbour = 0; neighbour < molecule.getConnAtoms(i); neighbour++) {
	        const neighbourAtom = molecule.getConnAtom(i, neighbour);
	        const neighbourBond = molecule.getConnBond(i, neighbour);
	        if (molecule.getAtomicNo(neighbourAtom) === 8) {
	          if (molecule.getBondOrder(neighbourBond) === 1 && molecule.getAllHydrogens(neighbourAtom) > 0) {
	            // If there is more than a Hydroxyl in the same carbon atom it is not couted as Carboxyl group
	            if (hydroxyl) {
	              hydroxyl = false;
	              break;
	            }
	            hydroxyl = true;
	          } else if (molecule.getBondOrder(neighbourBond) === 2) {
	            // If there is more than one carbonyl in the same carbon atom it is not count as Carboxyl group
	            if (carbonyl) {
	              carbonyl = false;
	              break;
	            }
	            carbonyl = true;
	          }
	        } else if (
	        // If there is not at least one carbon or hydrogen as neighbour atom it is not counted as Carboxyl group
	        molecule.getAtomicNo(neighbourAtom) !== 6 && molecule.getAtomicNo(neighbourAtom) !== 1) {
	          carbonOrHydrogen = false;
	        }
	      }
	      if (carbonyl && hydroxyl && carbonOrHydrogen) counter++;
	    }
	  }
	  return counter;
	}
	nbCOOH$1.nbCOOH = nbCOOH;

	var nbCHO$1 = {};

	/**
	 * Return the number of Carbonyl groups in a molecule or fragment
	 * @param {import('openchemlib').Molecule} molecule
	 * @returns {number} 'Number of Carbonyl groups'
	 */
	Object.defineProperty(nbCHO$1, "__esModule", {
	  value: true
	});
	nbCHO$1.nbCHO = void 0;
	function nbCHO(molecule) {
	  let counter = 0;
	  for (let i = 0; i < molecule.getAllAtoms(); i++) {
	    // if there is not at least one hydrogen in the carbon atom there can not be a carbonyl group
	    if (molecule.getAtomicNo(i) === 6 && molecule.getAllHydrogens(i) > 0) {
	      let carbonyl = false;
	      let carbonOrHydrogen = true;
	      for (let neighbour = 0; neighbour < molecule.getConnAtoms(i); neighbour++) {
	        const neighbourAtom = molecule.getConnAtom(i, neighbour);
	        const neighbourBond = molecule.getConnBond(i, neighbour);
	        if (molecule.getAtomicNo(neighbourAtom) === 8) {
	          if (molecule.getBondOrder(neighbourBond) === 2) {
	            // If there is more than one carbonyl group on the same carbon atom they are not counted as carbonyl groups
	            if (carbonyl) {
	              carbonyl = false;
	              break;
	            }
	            carbonyl = true;
	          }
	        } else if (
	        // If there is not at least one carbon or hydrogen as neighbour atom it is not counted as Carbonyl group
	        molecule.getAtomicNo(neighbourAtom) !== 6 && molecule.getAtomicNo(neighbourAtom) !== 1) {
	          carbonOrHydrogen = false;
	        }
	      }
	      if (carbonyl && carbonOrHydrogen) counter++;
	    }
	  }
	  return counter;
	}
	nbCHO$1.nbCHO = nbCHO;

	var nbNH2$1 = {};

	/**
	 * Return the number of Primary amine groups in a molecule or fragment
	 * @param {import('openchemlib').Molecule} molecule
	 * @returns {number} 'Number of Primary amine groups'
	 */
	Object.defineProperty(nbNH2$1, "__esModule", {
	  value: true
	});
	nbNH2$1.nbNH2 = void 0;
	function nbNH2(molecule) {
	  let counter = 0;
	  for (let i = 0; i < molecule.getAllAtoms(); i++) {
	    if (molecule.getAtomicNo(i) === 6) {
	      let amine = false;
	      let carbonOrHydrogen = true;
	      for (let neighbour = 0; neighbour < molecule.getConnAtoms(i); neighbour++) {
	        const neighbourAtom = molecule.getConnAtom(i, neighbour);
	        const neighbourBond = molecule.getConnBond(i, neighbour);
	        if (molecule.getAtomicNo(neighbourAtom) === 7 && molecule.getBondOrder(neighbourBond) === 1 && molecule.getAllHydrogens(neighbourAtom) > 1) {
	          // If there is more than a Primary amine in the same carbon atom they are not couted as Primary amines groups
	          if (amine) {
	            amine = false;
	            break;
	          }
	          amine = true;
	        } else if (
	        // If there is not at least one carbon or hydrogen as neighbour atom it is not counted as Primary amine group
	        molecule.getAtomicNo(neighbourAtom) !== 6 && molecule.getAtomicNo(neighbourAtom) !== 1) {
	          carbonOrHydrogen = false;
	        }
	      }
	      if (amine && carbonOrHydrogen) counter++;
	    }
	  }
	  return counter;
	}
	nbNH2$1.nbNH2 = nbNH2;

	var nbCN$1 = {};

	/**
	 * Return the number of Nitrile groups in a molecule or fragment
	 * @param {import('openchemlib').Molecule} molecule
	 * @returns {number} 'Number of Nitrile groups'
	 */
	Object.defineProperty(nbCN$1, "__esModule", {
	  value: true
	});
	nbCN$1.nbCN = void 0;
	function nbCN(molecule) {
	  let counter = 0;
	  for (let i = 0; i < molecule.getAllAtoms(); i++) {
	    if (molecule.getAtomicNo(i) === 6) {
	      let cn = false;
	      let carbonOrHydrogen = true;
	      for (let neighbour = 0; neighbour < molecule.getConnAtoms(i); neighbour++) {
	        const neighbourAtom = molecule.getConnAtom(i, neighbour);
	        const neighbourBond = molecule.getConnBond(i, neighbour);
	        if (molecule.getAtomicNo(neighbourAtom) === 7 && molecule.getBondOrder(neighbourBond) === 3) {
	          // If there is more than one Nitrile group in the same carbon atome they are not counted as Nitrile groups
	          if (cn) {
	            cn = false;
	            break;
	          }
	          cn = true;
	        } else if (
	        // If there is not at least one carbon or hydrogen as neighbour atom it is not counted as Nitrile group
	        molecule.getAtomicNo(neighbourAtom) !== 6 && molecule.getAtomicNo(neighbourAtom) !== 1) {
	          carbonOrHydrogen = false;
	        }
	      }
	      if (cn && carbonOrHydrogen) counter++;
	    }
	  }
	  return counter;
	}
	nbCN$1.nbCN = nbCN;

	var nbLabileH$1 = {};

	/**
	 * Return the number of labile protons being either on O, N, Br, Cl, F, I or S
	 * @param {import('openchemlib').Molecule} molecule
	 * @param {object} [options={}]
	 * @param {Array<number>} [options.atomicNumbers=[7, 8, 9, 16, 17, 35, 53]] - atomic numbers of the labile protons
	 * @returns {number} 'Number of labile protons'
	 */
	Object.defineProperty(nbLabileH$1, "__esModule", {
	  value: true
	});
	nbLabileH$1.nbLabileH = void 0;
	function nbLabileH(molecule, options = {}) {
	  const {
	    atomicNumbers = [7, 8, 9, 16, 17, 35, 53]
	  } = options;
	  let counter = 0;
	  for (let i = 0; i < molecule.getAllAtoms(); i++) {
	    if (atomicNumbers.includes(molecule.getAtomicNo(i))) {
	      counter += molecule.getAllHydrogens(i);
	    }
	  }
	  return counter;
	}
	nbLabileH$1.nbLabileH = nbLabileH;

	var toggleHydrogens$1 = {};

	Object.defineProperty(toggleHydrogens$1, "__esModule", {
	  value: true
	});
	toggleHydrogens$1.toggleHydrogens = void 0;
	const getImplicitHydrogensCount_1 = getImplicitHydrogensCount$1;
	/**
	 * Toggle presence of implicity hydrogens on/off
	 * @param molecule
	 * @param atomID
	 */
	function toggleHydrogens(molecule, atomID) {
	  if ((0, getImplicitHydrogensCount_1.getImplicitHydrogensCount)(molecule, atomID) === 0) {
	    const atomsToDelete = [];
	    for (let i = 0; i < molecule.getAllConnAtoms(atomID); i++) {
	      const connectedAtom = molecule.getConnAtom(atomID, i);
	      if (molecule.getAtomicNo(connectedAtom) === 1) {
	        atomsToDelete.push(connectedAtom);
	      }
	    }
	    molecule.deleteAtoms(atomsToDelete);
	  } else {
	    molecule.addImplicitHydrogens(atomID);
	  }
	}
	toggleHydrogens$1.toggleHydrogens = toggleHydrogens;

	var getPathsInfo$1 = {};

	var getPathAndTorsion$1 = {};

	Object.defineProperty(getPathAndTorsion$1, "__esModule", {
	  value: true
	});
	getPathAndTorsion$1.getPathAndTorsion = void 0;
	/**
	 * Calculates the path between 2 atoms
	 * @param {import('openchemlib').Molecule} molecule
	 * @param {number} from - index of the first atom
	 * @param {number} to - index of the end atom
	 * @param {number} maxLength - maximal length of the path
	 */
	function getPathAndTorsion(molecule, from, to, maxLength) {
	  const originalAtoms = []; // path before renumbering
	  molecule.getPath(originalAtoms, from, to, maxLength + 1);
	  let torsion;
	  if (originalAtoms.length === 4) {
	    torsion = molecule.calculateTorsion(originalAtoms);
	  }
	  return {
	    atoms: originalAtoms,
	    from,
	    to,
	    torsion,
	    length: originalAtoms.length - 1
	  };
	}
	getPathAndTorsion$1.getPathAndTorsion = getPathAndTorsion;

	Object.defineProperty(getPathsInfo$1, "__esModule", {
	  value: true
	});
	getPathsInfo$1.getPathsInfo = void 0;
	const getHoseCodesForPath_1 = getHoseCodesForPath$1;
	const getAtomsInfo_1 = getAtomsInfo$1;
	const getConnectivityMatrix_1 = getConnectivityMatrix$1;
	const getPathAndTorsion_js_1 = getPathAndTorsion$1;
	let fragment;
	/**
	 *
	 * @param {import('openchemlib').Molecule} molecule
	 * @param {object} [options={}]
	 * @param {string} [options.fromLabel='H']
	 * @param {string} [options.toLabel='H']
	 * @param {number} [options.minLength=1]
	 * @param {number} [options.maxLength=4]
	 * @param {boolean} [options.withHOSES=false]

	 */
	function getPathsInfo(molecule, options = {}) {
	  const {
	    fromLabel = 'H',
	    toLabel = 'H',
	    minLength = 1,
	    maxLength = 4,
	    withHOSES = false
	  } = options;
	  const OCL = molecule.getOCL();
	  if (!fragment) {
	    fragment = new OCL.Molecule(0, 0);
	  }
	  const fromAtomicNumber = OCL.Molecule.getAtomicNoFromLabel(fromLabel);
	  const toAtomicNumber = OCL.Molecule.getAtomicNoFromLabel(toLabel);
	  // we need to find all the atoms 'fromLabel' and 'toLabel'
	  const atomsInfo = (0, getAtomsInfo_1.getAtomsInfo)(molecule);
	  const pathLengthMatrix = (0, getConnectivityMatrix_1.getConnectivityMatrix)(molecule, {
	    pathLength: true
	  });
	  for (let from = 0; from < molecule.getAllAtoms(); from++) {
	    atomsInfo[from].paths = [];
	    for (let to = 0; to < molecule.getAllAtoms(); to++) {
	      if (from !== to) {
	        if (molecule.getAtomicNo(from) === fromAtomicNumber) {
	          if (molecule.getAtomicNo(to) === toAtomicNumber) {
	            const pathLength = pathLengthMatrix[from][to];
	            if (pathLength >= minLength && pathLength <= maxLength) {
	              if (withHOSES) {
	                atomsInfo[from].paths.push((0, getHoseCodesForPath_1.getHoseCodesForPath)(molecule, from, to, pathLength));
	              } else {
	                atomsInfo[from].paths.push((0, getPathAndTorsion_js_1.getPathAndTorsion)(molecule, from, to, pathLength));
	              }
	            }
	          }
	        }
	      }
	    }
	  }
	  return atomsInfo;
	}
	getPathsInfo$1.getPathsInfo = getPathsInfo;

	var getShortestPaths$1 = {};

	Object.defineProperty(getShortestPaths$1, "__esModule", {
	  value: true
	});
	getShortestPaths$1.getShortestPaths = void 0;
	/**
	 * Get the shortest path between each pair of atoms in the molecule
	 * @param {import('openchemlib').Molecule} molecule
	 * @param {object} [options={}]
	 * @param {string} [opions.fromLabel='H']
	 * @param {string} [opions.toLabel='H']
	 * @param {string} [opions.maxLength=4]
	 * @returns {Array<Array>} A matrix containing on each cell (i,j) the shortest path from atom i to atom j
	 */
	function getShortestPaths(molecule, options = {}) {
	  const OCL = molecule.getOCL();
	  const {
	    fromLabel = '',
	    toLabel = '',
	    maxLength = 3
	  } = options;
	  const fromAtomicNumber = OCL.Molecule.getAtomicNoFromLabel(fromLabel);
	  const toAtomicNumber = OCL.Molecule.getAtomicNoFromLabel(toLabel);
	  const nbAtoms = molecule.getAllAtoms();
	  const allShortestPaths = new Array(nbAtoms);
	  for (let i = 0; i < nbAtoms; i++) {
	    allShortestPaths[i] = new Array(nbAtoms);
	  }
	  for (let from = 0; from < nbAtoms; from++) {
	    allShortestPaths[from][from] = [from];
	    for (let to = from + 1; to < nbAtoms; to++) {
	      if ((fromAtomicNumber === 0 || molecule.getAtomicNo(from) === fromAtomicNumber) && (toAtomicNumber === 0 || molecule.getAtomicNo(to) === toAtomicNumber)) {
	        const path = [];
	        molecule.getPath(path, from, to, maxLength);
	        if (path.length) {
	          allShortestPaths[from][to] = path.slice();
	          allShortestPaths[to][from] = path.reverse();
	        } else {
	          allShortestPaths[from][to] = null;
	          allShortestPaths[to][from] = null;
	        }
	      } else {
	        allShortestPaths[from][to] = null;
	        allShortestPaths[to][from] = null;
	      }
	    }
	  }
	  return allShortestPaths;
	}
	getShortestPaths$1.getShortestPaths = getShortestPaths;

	var MoleculesDB$1 = {};

	var appendCSV$1 = {};

	/*
	    https://tools.ietf.org/html/rfc3629

	    UTF8-char = UTF8-1 / UTF8-2 / UTF8-3 / UTF8-4

	    UTF8-1    = %x00-7F

	    UTF8-2    = %xC2-DF UTF8-tail

	    UTF8-3    = %xE0 %xA0-BF UTF8-tail
	                %xE1-EC 2( UTF8-tail )
	                %xED %x80-9F UTF8-tail
	                %xEE-EF 2( UTF8-tail )

	    UTF8-4    = %xF0 %x90-BF 2( UTF8-tail )
	                %xF1-F3 3( UTF8-tail )
	                %xF4 %x80-8F 2( UTF8-tail )

	    UTF8-tail = %x80-BF
	*/
	/**
	 * Check if a Node.js Buffer or Uint8Array is UTF-8.
	 */
	function isUtf8(buf) {
	  if (!buf) {
	    return false;
	  }
	  var i = 0;
	  var len = buf.length;
	  while (i < len) {
	    // UTF8-1 = %x00-7F
	    if (buf[i] <= 0x7F) {
	      i++;
	      continue;
	    }
	    // UTF8-2 = %xC2-DF UTF8-tail
	    if (buf[i] >= 0xC2 && buf[i] <= 0xDF) {
	      // if(buf[i + 1] >= 0x80 && buf[i + 1] <= 0xBF) {
	      if (buf[i + 1] >> 6 === 2) {
	        i += 2;
	        continue;
	      } else {
	        return false;
	      }
	    }
	    // UTF8-3 = %xE0 %xA0-BF UTF8-tail
	    // UTF8-3 = %xED %x80-9F UTF8-tail
	    if ((buf[i] === 0xE0 && buf[i + 1] >= 0xA0 && buf[i + 1] <= 0xBF || buf[i] === 0xED && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x9F) && buf[i + 2] >> 6 === 2) {
	      i += 3;
	      continue;
	    }
	    // UTF8-3 = %xE1-EC 2( UTF8-tail )
	    // UTF8-3 = %xEE-EF 2( UTF8-tail )
	    if ((buf[i] >= 0xE1 && buf[i] <= 0xEC || buf[i] >= 0xEE && buf[i] <= 0xEF) && buf[i + 1] >> 6 === 2 && buf[i + 2] >> 6 === 2) {
	      i += 3;
	      continue;
	    }
	    // UTF8-4 = %xF0 %x90-BF 2( UTF8-tail )
	    //          %xF1-F3 3( UTF8-tail )
	    //          %xF4 %x80-8F 2( UTF8-tail )
	    if ((buf[i] === 0xF0 && buf[i + 1] >= 0x90 && buf[i + 1] <= 0xBF || buf[i] >= 0xF1 && buf[i] <= 0xF3 && buf[i + 1] >> 6 === 2 || buf[i] === 0xF4 && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x8F) && buf[i + 2] >> 6 === 2 && buf[i + 3] >> 6 === 2) {
	      i += 4;
	      continue;
	    }
	    return false;
	  }
	  return true;
	}

	/**
	 * Ensure that the data is string. If it is an ArrayBuffer it will be converted to string using TextDecoder.
	 * @param blob
	 * @param options
	 * @returns
	 */
	function ensureString(blob, options = {}) {
	  if (typeof blob === 'string') {
	    return blob;
	  }
	  if (ArrayBuffer.isView(blob) || blob instanceof ArrayBuffer) {
	    const {
	      encoding = guessEncoding(blob)
	    } = options;
	    const decoder = new TextDecoder(encoding);
	    return decoder.decode(blob);
	  }
	  throw new TypeError(`blob must be a string, ArrayBuffer or ArrayBufferView`);
	}
	function guessEncoding(blob) {
	  const uint8 = ArrayBuffer.isView(blob) ? new Uint8Array(blob.buffer, blob.byteOffset, blob.byteLength) : new Uint8Array(blob);
	  if (uint8.length >= 2) {
	    if (uint8[0] === 0xfe && uint8[1] === 0xff) {
	      return 'utf-16be';
	    }
	    if (uint8[0] === 0xff && uint8[1] === 0xfe) {
	      return 'utf-16le';
	    }
	  }
	  //@ts-expect-error an ArrayBuffer is also ok
	  if (!isUtf8(blob)) return 'latin1';
	  return 'utf-8';
	}

	var libEsm = /*#__PURE__*/Object.freeze({
		__proto__: null,
		ensureString: ensureString
	});

	var require$$0 = /*@__PURE__*/getAugmentedNamespace(libEsm);

	var papaparse_min = {exports: {}};

	/* @license
	Papa Parse
	v5.4.1
	https://github.com/mholt/PapaParse
	License: MIT
	*/
	(function (module, exports) {
	  !function (e, t) {
	    module.exports = t() ;
	  }(commonjsGlobal, function s() {

	    var f = "undefined" != typeof self ? self : "undefined" != typeof window ? window : void 0 !== f ? f : {};
	    var n = !f.document && !!f.postMessage,
	      o = f.IS_PAPA_WORKER || !1,
	      a = {},
	      u = 0,
	      b = {
	        parse: function (e, t) {
	          var r = (t = t || {}).dynamicTyping || !1;
	          J(r) && (t.dynamicTypingFunction = r, r = {});
	          if (t.dynamicTyping = r, t.transform = !!J(t.transform) && t.transform, t.worker && b.WORKERS_SUPPORTED) {
	            var i = function () {
	              if (!b.WORKERS_SUPPORTED) return !1;
	              var e = (r = f.URL || f.webkitURL || null, i = s.toString(), b.BLOB_URL || (b.BLOB_URL = r.createObjectURL(new Blob(["var global = (function() { if (typeof self !== 'undefined') { return self; } if (typeof window !== 'undefined') { return window; } if (typeof global !== 'undefined') { return global; } return {}; })(); global.IS_PAPA_WORKER=true; ", "(", i, ")();"], {
	                  type: "text/javascript"
	                })))),
	                t = new f.Worker(e);
	              var r, i;
	              return t.onmessage = _, t.id = u++, a[t.id] = t;
	            }();
	            return i.userStep = t.step, i.userChunk = t.chunk, i.userComplete = t.complete, i.userError = t.error, t.step = J(t.step), t.chunk = J(t.chunk), t.complete = J(t.complete), t.error = J(t.error), delete t.worker, void i.postMessage({
	              input: e,
	              config: t,
	              workerId: i.id
	            });
	          }
	          var n = null;
	          b.NODE_STREAM_INPUT, "string" == typeof e ? (e = function (e) {
	            if (65279 === e.charCodeAt(0)) return e.slice(1);
	            return e;
	          }(e), n = t.download ? new l(t) : new p(t)) : !0 === e.readable && J(e.read) && J(e.on) ? n = new g(t) : (f.File && e instanceof File || e instanceof Object) && (n = new c(t));
	          return n.stream(e);
	        },
	        unparse: function (e, t) {
	          var n = !1,
	            _ = !0,
	            m = ",",
	            y = "\r\n",
	            s = '"',
	            a = s + s,
	            r = !1,
	            i = null,
	            o = !1;
	          !function () {
	            if ("object" != typeof t) return;
	            "string" != typeof t.delimiter || b.BAD_DELIMITERS.filter(function (e) {
	              return -1 !== t.delimiter.indexOf(e);
	            }).length || (m = t.delimiter);
	            ("boolean" == typeof t.quotes || "function" == typeof t.quotes || Array.isArray(t.quotes)) && (n = t.quotes);
	            "boolean" != typeof t.skipEmptyLines && "string" != typeof t.skipEmptyLines || (r = t.skipEmptyLines);
	            "string" == typeof t.newline && (y = t.newline);
	            "string" == typeof t.quoteChar && (s = t.quoteChar);
	            "boolean" == typeof t.header && (_ = t.header);
	            if (Array.isArray(t.columns)) {
	              if (0 === t.columns.length) throw new Error("Option columns is empty");
	              i = t.columns;
	            }
	            void 0 !== t.escapeChar && (a = t.escapeChar + s);
	            ("boolean" == typeof t.escapeFormulae || t.escapeFormulae instanceof RegExp) && (o = t.escapeFormulae instanceof RegExp ? t.escapeFormulae : /^[=+\-@\t\r].*$/);
	          }();
	          var u = new RegExp(Q(s), "g");
	          "string" == typeof e && (e = JSON.parse(e));
	          if (Array.isArray(e)) {
	            if (!e.length || Array.isArray(e[0])) return h(null, e, r);
	            if ("object" == typeof e[0]) return h(i || Object.keys(e[0]), e, r);
	          } else if ("object" == typeof e) return "string" == typeof e.data && (e.data = JSON.parse(e.data)), Array.isArray(e.data) && (e.fields || (e.fields = e.meta && e.meta.fields || i), e.fields || (e.fields = Array.isArray(e.data[0]) ? e.fields : "object" == typeof e.data[0] ? Object.keys(e.data[0]) : []), Array.isArray(e.data[0]) || "object" == typeof e.data[0] || (e.data = [e.data])), h(e.fields || [], e.data || [], r);
	          throw new Error("Unable to serialize unrecognized input");
	          function h(e, t, r) {
	            var i = "";
	            "string" == typeof e && (e = JSON.parse(e)), "string" == typeof t && (t = JSON.parse(t));
	            var n = Array.isArray(e) && 0 < e.length,
	              s = !Array.isArray(t[0]);
	            if (n && _) {
	              for (var a = 0; a < e.length; a++) 0 < a && (i += m), i += v(e[a], a);
	              0 < t.length && (i += y);
	            }
	            for (var o = 0; o < t.length; o++) {
	              var u = n ? e.length : t[o].length,
	                h = !1,
	                f = n ? 0 === Object.keys(t[o]).length : 0 === t[o].length;
	              if (r && !n && (h = "greedy" === r ? "" === t[o].join("").trim() : 1 === t[o].length && 0 === t[o][0].length), "greedy" === r && n) {
	                for (var d = [], l = 0; l < u; l++) {
	                  var c = s ? e[l] : l;
	                  d.push(t[o][c]);
	                }
	                h = "" === d.join("").trim();
	              }
	              if (!h) {
	                for (var p = 0; p < u; p++) {
	                  0 < p && !f && (i += m);
	                  var g = n && s ? e[p] : p;
	                  i += v(t[o][g], p);
	                }
	                o < t.length - 1 && (!r || 0 < u && !f) && (i += y);
	              }
	            }
	            return i;
	          }
	          function v(e, t) {
	            if (null == e) return "";
	            if (e.constructor === Date) return JSON.stringify(e).slice(1, 25);
	            var r = !1;
	            o && "string" == typeof e && o.test(e) && (e = "'" + e, r = !0);
	            var i = e.toString().replace(u, a);
	            return (r = r || !0 === n || "function" == typeof n && n(e, t) || Array.isArray(n) && n[t] || function (e, t) {
	              for (var r = 0; r < t.length; r++) if (-1 < e.indexOf(t[r])) return !0;
	              return !1;
	            }(i, b.BAD_DELIMITERS) || -1 < i.indexOf(m) || " " === i.charAt(0) || " " === i.charAt(i.length - 1)) ? s + i + s : i;
	          }
	        }
	      };
	    if (b.RECORD_SEP = String.fromCharCode(30), b.UNIT_SEP = String.fromCharCode(31), b.BYTE_ORDER_MARK = "\ufeff", b.BAD_DELIMITERS = ["\r", "\n", '"', b.BYTE_ORDER_MARK], b.WORKERS_SUPPORTED = !n && !!f.Worker, b.NODE_STREAM_INPUT = 1, b.LocalChunkSize = 10485760, b.RemoteChunkSize = 5242880, b.DefaultDelimiter = ",", b.Parser = E, b.ParserHandle = r, b.NetworkStreamer = l, b.FileStreamer = c, b.StringStreamer = p, b.ReadableStreamStreamer = g, f.jQuery) {
	      var d = f.jQuery;
	      d.fn.parse = function (o) {
	        var r = o.config || {},
	          u = [];
	        return this.each(function (e) {
	          if (!("INPUT" === d(this).prop("tagName").toUpperCase() && "file" === d(this).attr("type").toLowerCase() && f.FileReader) || !this.files || 0 === this.files.length) return !0;
	          for (var t = 0; t < this.files.length; t++) u.push({
	            file: this.files[t],
	            inputElem: this,
	            instanceConfig: d.extend({}, r)
	          });
	        }), e(), this;
	        function e() {
	          if (0 !== u.length) {
	            var e,
	              t,
	              r,
	              i,
	              n = u[0];
	            if (J(o.before)) {
	              var s = o.before(n.file, n.inputElem);
	              if ("object" == typeof s) {
	                if ("abort" === s.action) return e = "AbortError", t = n.file, r = n.inputElem, i = s.reason, void (J(o.error) && o.error({
	                  name: e
	                }, t, r, i));
	                if ("skip" === s.action) return void h();
	                "object" == typeof s.config && (n.instanceConfig = d.extend(n.instanceConfig, s.config));
	              } else if ("skip" === s) return void h();
	            }
	            var a = n.instanceConfig.complete;
	            n.instanceConfig.complete = function (e) {
	              J(a) && a(e, n.file, n.inputElem), h();
	            }, b.parse(n.file, n.instanceConfig);
	          } else J(o.complete) && o.complete();
	        }
	        function h() {
	          u.splice(0, 1), e();
	        }
	      };
	    }
	    function h(e) {
	      this._handle = null, this._finished = !1, this._completed = !1, this._halted = !1, this._input = null, this._baseIndex = 0, this._partialLine = "", this._rowCount = 0, this._start = 0, this._nextChunk = null, this.isFirstChunk = !0, this._completeResults = {
	        data: [],
	        errors: [],
	        meta: {}
	      }, function (e) {
	        var t = w(e);
	        t.chunkSize = parseInt(t.chunkSize), e.step || e.chunk || (t.chunkSize = null);
	        this._handle = new r(t), (this._handle.streamer = this)._config = t;
	      }.call(this, e), this.parseChunk = function (e, t) {
	        if (this.isFirstChunk && J(this._config.beforeFirstChunk)) {
	          var r = this._config.beforeFirstChunk(e);
	          void 0 !== r && (e = r);
	        }
	        this.isFirstChunk = !1, this._halted = !1;
	        var i = this._partialLine + e;
	        this._partialLine = "";
	        var n = this._handle.parse(i, this._baseIndex, !this._finished);
	        if (!this._handle.paused() && !this._handle.aborted()) {
	          var s = n.meta.cursor;
	          this._finished || (this._partialLine = i.substring(s - this._baseIndex), this._baseIndex = s), n && n.data && (this._rowCount += n.data.length);
	          var a = this._finished || this._config.preview && this._rowCount >= this._config.preview;
	          if (o) f.postMessage({
	            results: n,
	            workerId: b.WORKER_ID,
	            finished: a
	          });else if (J(this._config.chunk) && !t) {
	            if (this._config.chunk(n, this._handle), this._handle.paused() || this._handle.aborted()) return void (this._halted = !0);
	            n = void 0, this._completeResults = void 0;
	          }
	          return this._config.step || this._config.chunk || (this._completeResults.data = this._completeResults.data.concat(n.data), this._completeResults.errors = this._completeResults.errors.concat(n.errors), this._completeResults.meta = n.meta), this._completed || !a || !J(this._config.complete) || n && n.meta.aborted || (this._config.complete(this._completeResults, this._input), this._completed = !0), a || n && n.meta.paused || this._nextChunk(), n;
	        }
	        this._halted = !0;
	      }, this._sendError = function (e) {
	        J(this._config.error) ? this._config.error(e) : o && this._config.error && f.postMessage({
	          workerId: b.WORKER_ID,
	          error: e,
	          finished: !1
	        });
	      };
	    }
	    function l(e) {
	      var i;
	      (e = e || {}).chunkSize || (e.chunkSize = b.RemoteChunkSize), h.call(this, e), this._nextChunk = n ? function () {
	        this._readChunk(), this._chunkLoaded();
	      } : function () {
	        this._readChunk();
	      }, this.stream = function (e) {
	        this._input = e, this._nextChunk();
	      }, this._readChunk = function () {
	        if (this._finished) this._chunkLoaded();else {
	          if (i = new XMLHttpRequest(), this._config.withCredentials && (i.withCredentials = this._config.withCredentials), n || (i.onload = v(this._chunkLoaded, this), i.onerror = v(this._chunkError, this)), i.open(this._config.downloadRequestBody ? "POST" : "GET", this._input, !n), this._config.downloadRequestHeaders) {
	            var e = this._config.downloadRequestHeaders;
	            for (var t in e) i.setRequestHeader(t, e[t]);
	          }
	          if (this._config.chunkSize) {
	            var r = this._start + this._config.chunkSize - 1;
	            i.setRequestHeader("Range", "bytes=" + this._start + "-" + r);
	          }
	          try {
	            i.send(this._config.downloadRequestBody);
	          } catch (e) {
	            this._chunkError(e.message);
	          }
	          n && 0 === i.status && this._chunkError();
	        }
	      }, this._chunkLoaded = function () {
	        4 === i.readyState && (i.status < 200 || 400 <= i.status ? this._chunkError() : (this._start += this._config.chunkSize ? this._config.chunkSize : i.responseText.length, this._finished = !this._config.chunkSize || this._start >= function (e) {
	          var t = e.getResponseHeader("Content-Range");
	          if (null === t) return -1;
	          return parseInt(t.substring(t.lastIndexOf("/") + 1));
	        }(i), this.parseChunk(i.responseText)));
	      }, this._chunkError = function (e) {
	        var t = i.statusText || e;
	        this._sendError(new Error(t));
	      };
	    }
	    function c(e) {
	      var i, n;
	      (e = e || {}).chunkSize || (e.chunkSize = b.LocalChunkSize), h.call(this, e);
	      var s = "undefined" != typeof FileReader;
	      this.stream = function (e) {
	        this._input = e, n = e.slice || e.webkitSlice || e.mozSlice, s ? ((i = new FileReader()).onload = v(this._chunkLoaded, this), i.onerror = v(this._chunkError, this)) : i = new FileReaderSync(), this._nextChunk();
	      }, this._nextChunk = function () {
	        this._finished || this._config.preview && !(this._rowCount < this._config.preview) || this._readChunk();
	      }, this._readChunk = function () {
	        var e = this._input;
	        if (this._config.chunkSize) {
	          var t = Math.min(this._start + this._config.chunkSize, this._input.size);
	          e = n.call(e, this._start, t);
	        }
	        var r = i.readAsText(e, this._config.encoding);
	        s || this._chunkLoaded({
	          target: {
	            result: r
	          }
	        });
	      }, this._chunkLoaded = function (e) {
	        this._start += this._config.chunkSize, this._finished = !this._config.chunkSize || this._start >= this._input.size, this.parseChunk(e.target.result);
	      }, this._chunkError = function () {
	        this._sendError(i.error);
	      };
	    }
	    function p(e) {
	      var r;
	      h.call(this, e = e || {}), this.stream = function (e) {
	        return r = e, this._nextChunk();
	      }, this._nextChunk = function () {
	        if (!this._finished) {
	          var e,
	            t = this._config.chunkSize;
	          return t ? (e = r.substring(0, t), r = r.substring(t)) : (e = r, r = ""), this._finished = !r, this.parseChunk(e);
	        }
	      };
	    }
	    function g(e) {
	      h.call(this, e = e || {});
	      var t = [],
	        r = !0,
	        i = !1;
	      this.pause = function () {
	        h.prototype.pause.apply(this, arguments), this._input.pause();
	      }, this.resume = function () {
	        h.prototype.resume.apply(this, arguments), this._input.resume();
	      }, this.stream = function (e) {
	        this._input = e, this._input.on("data", this._streamData), this._input.on("end", this._streamEnd), this._input.on("error", this._streamError);
	      }, this._checkIsFinished = function () {
	        i && 1 === t.length && (this._finished = !0);
	      }, this._nextChunk = function () {
	        this._checkIsFinished(), t.length ? this.parseChunk(t.shift()) : r = !0;
	      }, this._streamData = v(function (e) {
	        try {
	          t.push("string" == typeof e ? e : e.toString(this._config.encoding)), r && (r = !1, this._checkIsFinished(), this.parseChunk(t.shift()));
	        } catch (e) {
	          this._streamError(e);
	        }
	      }, this), this._streamError = v(function (e) {
	        this._streamCleanUp(), this._sendError(e);
	      }, this), this._streamEnd = v(function () {
	        this._streamCleanUp(), i = !0, this._streamData("");
	      }, this), this._streamCleanUp = v(function () {
	        this._input.removeListener("data", this._streamData), this._input.removeListener("end", this._streamEnd), this._input.removeListener("error", this._streamError);
	      }, this);
	    }
	    function r(m) {
	      var a,
	        o,
	        u,
	        i = Math.pow(2, 53),
	        n = -i,
	        s = /^\s*-?(\d+\.?|\.\d+|\d+\.\d+)([eE][-+]?\d+)?\s*$/,
	        h = /^((\d{4}-[01]\d-[0-3]\dT[0-2]\d:[0-5]\d:[0-5]\d\.\d+([+-][0-2]\d:[0-5]\d|Z))|(\d{4}-[01]\d-[0-3]\dT[0-2]\d:[0-5]\d:[0-5]\d([+-][0-2]\d:[0-5]\d|Z))|(\d{4}-[01]\d-[0-3]\dT[0-2]\d:[0-5]\d([+-][0-2]\d:[0-5]\d|Z)))$/,
	        t = this,
	        r = 0,
	        f = 0,
	        d = !1,
	        e = !1,
	        l = [],
	        c = {
	          data: [],
	          errors: [],
	          meta: {}
	        };
	      if (J(m.step)) {
	        var p = m.step;
	        m.step = function (e) {
	          if (c = e, _()) g();else {
	            if (g(), 0 === c.data.length) return;
	            r += e.data.length, m.preview && r > m.preview ? o.abort() : (c.data = c.data[0], p(c, t));
	          }
	        };
	      }
	      function y(e) {
	        return "greedy" === m.skipEmptyLines ? "" === e.join("").trim() : 1 === e.length && 0 === e[0].length;
	      }
	      function g() {
	        return c && u && (k("Delimiter", "UndetectableDelimiter", "Unable to auto-detect delimiting character; defaulted to '" + b.DefaultDelimiter + "'"), u = !1), m.skipEmptyLines && (c.data = c.data.filter(function (e) {
	          return !y(e);
	        })), _() && function () {
	          if (!c) return;
	          function e(e, t) {
	            J(m.transformHeader) && (e = m.transformHeader(e, t)), l.push(e);
	          }
	          if (Array.isArray(c.data[0])) {
	            for (var t = 0; _() && t < c.data.length; t++) c.data[t].forEach(e);
	            c.data.splice(0, 1);
	          } else c.data.forEach(e);
	        }(), function () {
	          if (!c || !m.header && !m.dynamicTyping && !m.transform) return c;
	          function e(e, t) {
	            var r,
	              i = m.header ? {} : [];
	            for (r = 0; r < e.length; r++) {
	              var n = r,
	                s = e[r];
	              m.header && (n = r >= l.length ? "__parsed_extra" : l[r]), m.transform && (s = m.transform(s, n)), s = v(n, s), "__parsed_extra" === n ? (i[n] = i[n] || [], i[n].push(s)) : i[n] = s;
	            }
	            return m.header && (r > l.length ? k("FieldMismatch", "TooManyFields", "Too many fields: expected " + l.length + " fields but parsed " + r, f + t) : r < l.length && k("FieldMismatch", "TooFewFields", "Too few fields: expected " + l.length + " fields but parsed " + r, f + t)), i;
	          }
	          var t = 1;
	          !c.data.length || Array.isArray(c.data[0]) ? (c.data = c.data.map(e), t = c.data.length) : c.data = e(c.data, 0);
	          m.header && c.meta && (c.meta.fields = l);
	          return f += t, c;
	        }();
	      }
	      function _() {
	        return m.header && 0 === l.length;
	      }
	      function v(e, t) {
	        return r = e, m.dynamicTypingFunction && void 0 === m.dynamicTyping[r] && (m.dynamicTyping[r] = m.dynamicTypingFunction(r)), !0 === (m.dynamicTyping[r] || m.dynamicTyping) ? "true" === t || "TRUE" === t || "false" !== t && "FALSE" !== t && (function (e) {
	          if (s.test(e)) {
	            var t = parseFloat(e);
	            if (n < t && t < i) return !0;
	          }
	          return !1;
	        }(t) ? parseFloat(t) : h.test(t) ? new Date(t) : "" === t ? null : t) : t;
	        var r;
	      }
	      function k(e, t, r, i) {
	        var n = {
	          type: e,
	          code: t,
	          message: r
	        };
	        void 0 !== i && (n.row = i), c.errors.push(n);
	      }
	      this.parse = function (e, t, r) {
	        var i = m.quoteChar || '"';
	        if (m.newline || (m.newline = function (e, t) {
	          e = e.substring(0, 1048576);
	          var r = new RegExp(Q(t) + "([^]*?)" + Q(t), "gm"),
	            i = (e = e.replace(r, "")).split("\r"),
	            n = e.split("\n"),
	            s = 1 < n.length && n[0].length < i[0].length;
	          if (1 === i.length || s) return "\n";
	          for (var a = 0, o = 0; o < i.length; o++) "\n" === i[o][0] && a++;
	          return a >= i.length / 2 ? "\r\n" : "\r";
	        }(e, i)), u = !1, m.delimiter) J(m.delimiter) && (m.delimiter = m.delimiter(e), c.meta.delimiter = m.delimiter);else {
	          var n = function (e, t, r, i, n) {
	            var s, a, o, u;
	            n = n || [",", "\t", "|", ";", b.RECORD_SEP, b.UNIT_SEP];
	            for (var h = 0; h < n.length; h++) {
	              var f = n[h],
	                d = 0,
	                l = 0,
	                c = 0;
	              o = void 0;
	              for (var p = new E({
	                  comments: i,
	                  delimiter: f,
	                  newline: t,
	                  preview: 10
	                }).parse(e), g = 0; g < p.data.length; g++) if (r && y(p.data[g])) c++;else {
	                var _ = p.data[g].length;
	                l += _, void 0 !== o ? 0 < _ && (d += Math.abs(_ - o), o = _) : o = _;
	              }
	              0 < p.data.length && (l /= p.data.length - c), (void 0 === a || d <= a) && (void 0 === u || u < l) && 1.99 < l && (a = d, s = f, u = l);
	            }
	            return {
	              successful: !!(m.delimiter = s),
	              bestDelimiter: s
	            };
	          }(e, m.newline, m.skipEmptyLines, m.comments, m.delimitersToGuess);
	          n.successful ? m.delimiter = n.bestDelimiter : (u = !0, m.delimiter = b.DefaultDelimiter), c.meta.delimiter = m.delimiter;
	        }
	        var s = w(m);
	        return m.preview && m.header && s.preview++, a = e, o = new E(s), c = o.parse(a, t, r), g(), d ? {
	          meta: {
	            paused: !0
	          }
	        } : c || {
	          meta: {
	            paused: !1
	          }
	        };
	      }, this.paused = function () {
	        return d;
	      }, this.pause = function () {
	        d = !0, o.abort(), a = J(m.chunk) ? "" : a.substring(o.getCharIndex());
	      }, this.resume = function () {
	        t.streamer._halted ? (d = !1, t.streamer.parseChunk(a, !0)) : setTimeout(t.resume, 3);
	      }, this.aborted = function () {
	        return e;
	      }, this.abort = function () {
	        e = !0, o.abort(), c.meta.aborted = !0, J(m.complete) && m.complete(c), a = "";
	      };
	    }
	    function Q(e) {
	      return e.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
	    }
	    function E(j) {
	      var z,
	        M = (j = j || {}).delimiter,
	        P = j.newline,
	        U = j.comments,
	        q = j.step,
	        N = j.preview,
	        B = j.fastMode,
	        K = z = void 0 === j.quoteChar || null === j.quoteChar ? '"' : j.quoteChar;
	      if (void 0 !== j.escapeChar && (K = j.escapeChar), ("string" != typeof M || -1 < b.BAD_DELIMITERS.indexOf(M)) && (M = ","), U === M) throw new Error("Comment character same as delimiter");
	      !0 === U ? U = "#" : ("string" != typeof U || -1 < b.BAD_DELIMITERS.indexOf(U)) && (U = !1), "\n" !== P && "\r" !== P && "\r\n" !== P && (P = "\n");
	      var W = 0,
	        H = !1;
	      this.parse = function (i, t, r) {
	        if ("string" != typeof i) throw new Error("Input must be a string");
	        var n = i.length,
	          e = M.length,
	          s = P.length,
	          a = U.length,
	          o = J(q),
	          u = [],
	          h = [],
	          f = [],
	          d = W = 0;
	        if (!i) return L();
	        if (j.header && !t) {
	          var l = i.split(P)[0].split(M),
	            c = [],
	            p = {},
	            g = !1;
	          for (var _ in l) {
	            var m = l[_];
	            J(j.transformHeader) && (m = j.transformHeader(m, _));
	            var y = m,
	              v = p[m] || 0;
	            for (0 < v && (g = !0, y = m + "_" + v), p[m] = v + 1; c.includes(y);) y = y + "_" + v;
	            c.push(y);
	          }
	          if (g) {
	            var k = i.split(P);
	            k[0] = c.join(M), i = k.join(P);
	          }
	        }
	        if (B || !1 !== B && -1 === i.indexOf(z)) {
	          for (var b = i.split(P), E = 0; E < b.length; E++) {
	            if (f = b[E], W += f.length, E !== b.length - 1) W += P.length;else if (r) return L();
	            if (!U || f.substring(0, a) !== U) {
	              if (o) {
	                if (u = [], I(f.split(M)), F(), H) return L();
	              } else I(f.split(M));
	              if (N && N <= E) return u = u.slice(0, N), L(!0);
	            }
	          }
	          return L();
	        }
	        for (var w = i.indexOf(M, W), R = i.indexOf(P, W), C = new RegExp(Q(K) + Q(z), "g"), S = i.indexOf(z, W);;) if (i[W] !== z) {
	          if (U && 0 === f.length && i.substring(W, W + a) === U) {
	            if (-1 === R) return L();
	            W = R + s, R = i.indexOf(P, W), w = i.indexOf(M, W);
	          } else if (-1 !== w && (w < R || -1 === R)) f.push(i.substring(W, w)), W = w + e, w = i.indexOf(M, W);else {
	            if (-1 === R) break;
	            if (f.push(i.substring(W, R)), D(R + s), o && (F(), H)) return L();
	            if (N && u.length >= N) return L(!0);
	          }
	        } else for (S = W, W++;;) {
	          if (-1 === (S = i.indexOf(z, S + 1))) return r || h.push({
	            type: "Quotes",
	            code: "MissingQuotes",
	            message: "Quoted field unterminated",
	            row: u.length,
	            index: W
	          }), T();
	          if (S === n - 1) return T(i.substring(W, S).replace(C, z));
	          if (z !== K || i[S + 1] !== K) {
	            if (z === K || 0 === S || i[S - 1] !== K) {
	              -1 !== w && w < S + 1 && (w = i.indexOf(M, S + 1)), -1 !== R && R < S + 1 && (R = i.indexOf(P, S + 1));
	              var O = A(-1 === R ? w : Math.min(w, R));
	              if (i.substr(S + 1 + O, e) === M) {
	                f.push(i.substring(W, S).replace(C, z)), i[W = S + 1 + O + e] !== z && (S = i.indexOf(z, W)), w = i.indexOf(M, W), R = i.indexOf(P, W);
	                break;
	              }
	              var x = A(R);
	              if (i.substring(S + 1 + x, S + 1 + x + s) === P) {
	                if (f.push(i.substring(W, S).replace(C, z)), D(S + 1 + x + s), w = i.indexOf(M, W), S = i.indexOf(z, W), o && (F(), H)) return L();
	                if (N && u.length >= N) return L(!0);
	                break;
	              }
	              h.push({
	                type: "Quotes",
	                code: "InvalidQuotes",
	                message: "Trailing quote on quoted field is malformed",
	                row: u.length,
	                index: W
	              }), S++;
	            }
	          } else S++;
	        }
	        return T();
	        function I(e) {
	          u.push(e), d = W;
	        }
	        function A(e) {
	          var t = 0;
	          if (-1 !== e) {
	            var r = i.substring(S + 1, e);
	            r && "" === r.trim() && (t = r.length);
	          }
	          return t;
	        }
	        function T(e) {
	          return r || (void 0 === e && (e = i.substring(W)), f.push(e), W = n, I(f), o && F()), L();
	        }
	        function D(e) {
	          W = e, I(f), f = [], R = i.indexOf(P, W);
	        }
	        function L(e) {
	          return {
	            data: u,
	            errors: h,
	            meta: {
	              delimiter: M,
	              linebreak: P,
	              aborted: H,
	              truncated: !!e,
	              cursor: d + (t || 0)
	            }
	          };
	        }
	        function F() {
	          q(L()), u = [], h = [];
	        }
	      }, this.abort = function () {
	        H = !0;
	      }, this.getCharIndex = function () {
	        return W;
	      };
	    }
	    function _(e) {
	      var t = e.data,
	        r = a[t.workerId],
	        i = !1;
	      if (t.error) r.userError(t.error, t.file);else if (t.results && t.results.data) {
	        var n = {
	          abort: function () {
	            i = !0, m(t.workerId, {
	              data: [],
	              errors: [],
	              meta: {
	                aborted: !0
	              }
	            });
	          },
	          pause: y,
	          resume: y
	        };
	        if (J(r.userStep)) {
	          for (var s = 0; s < t.results.data.length && (r.userStep({
	            data: t.results.data[s],
	            errors: t.results.errors,
	            meta: t.results.meta
	          }, n), !i); s++);
	          delete t.results;
	        } else J(r.userChunk) && (r.userChunk(t.results, n, t.file), delete t.results);
	      }
	      t.finished && !i && m(t.workerId, t.results);
	    }
	    function m(e, t) {
	      var r = a[e];
	      J(r.userComplete) && r.userComplete(t), r.terminate(), delete a[e];
	    }
	    function y() {
	      throw new Error("Not implemented.");
	    }
	    function w(e) {
	      if ("object" != typeof e || null === e) return e;
	      var t = Array.isArray(e) ? [] : {};
	      for (var r in e) t[r] = w(e[r]);
	      return t;
	    }
	    function v(e, t) {
	      return function () {
	        e.apply(t, arguments);
	      };
	    }
	    function J(e) {
	      return "function" == typeof e;
	    }
	    return o && (f.onmessage = function (e) {
	      var t = e.data;
	      void 0 === b.WORKER_ID && t && (b.WORKER_ID = t.workerId);
	      if ("string" == typeof t.input) f.postMessage({
	        workerId: b.WORKER_ID,
	        results: b.parse(t.input, t.config),
	        finished: !0
	      });else if (f.File && t.input instanceof File || t.input instanceof Object) {
	        var r = b.parse(t.input, t.config);
	        r && f.postMessage({
	          workerId: b.WORKER_ID,
	          results: r,
	          finished: !0
	        });
	      }
	    }), (l.prototype = Object.create(h.prototype)).constructor = l, (c.prototype = Object.create(h.prototype)).constructor = c, (p.prototype = Object.create(p.prototype)).constructor = p, (g.prototype = Object.create(h.prototype)).constructor = g, b;
	  });
	})(papaparse_min);

	var getMoleculeCreators$1 = {};

	Object.defineProperty(getMoleculeCreators$1, "__esModule", {
	  value: true
	});
	function getMoleculeCreators(OCL) {
	  const fields = new Map();
	  fields.set('oclid', OCL.Molecule.fromIDCode);
	  fields.set('idcode', OCL.Molecule.fromIDCode);
	  fields.set('smiles', OCL.Molecule.fromSmiles);
	  fields.set('molfile', OCL.Molecule.fromMolfile);
	  fields.set('smarts', smarts => {
	    const smilesParser = new OCL.SmilesParser({
	      smartsMode: 'smarts'
	    });
	    return smilesParser.parseMolecule(smarts);
	  });
	  return fields;
	}
	getMoleculeCreators$1.default = getMoleculeCreators;

	var __importDefault$3 = commonjsGlobal && commonjsGlobal.__importDefault || function (mod) {
	  return mod && mod.__esModule ? mod : {
	    "default": mod
	  };
	};
	Object.defineProperty(appendCSV$1, "__esModule", {
	  value: true
	});
	const ensure_string_1$3 = require$$0;
	const papaparse_1 = __importDefault$3(papaparse_min.exports);
	const getMoleculeCreators_1$1 = __importDefault$3(getMoleculeCreators$1);
	const defaultCSVOptions = {
	  header: true,
	  dynamicTyping: true,
	  skipEmptyLines: true
	};
	async function appendCSV(moleculesDB, csv, options = {}) {
	  const {
	    onStep
	  } = options;
	  csv = (0, ensure_string_1$3.ensureString)(csv);
	  const moleculeCreators = (0, getMoleculeCreators_1$1.default)(moleculesDB.OCL);
	  if (typeof csv !== 'string') {
	    throw new TypeError('csv must be a string');
	  }
	  options = {
	    ...defaultCSVOptions,
	    ...options
	  };
	  const parsed = papaparse_1.default.parse(csv, options);
	  const fields = parsed.meta.fields;
	  const stats = new Array(fields.length);
	  const firstElement = parsed.data[0];
	  let moleculeCreator, moleculeField;
	  for (let i = 0; i < fields.length; i++) {
	    stats[i] = {
	      label: fields[i],
	      isNumeric: typeof firstElement[fields[i]] === 'number'
	    };
	    const lowerField = fields[i].toLowerCase();
	    if (moleculeCreators.has(lowerField)) {
	      moleculeCreator = moleculeCreators.get(lowerField);
	      moleculeField = fields[i];
	    }
	  }
	  if (!moleculeCreator) {
	    throw new Error('this document does not contain any molecule field');
	  }
	  moleculesDB.statistics = stats;
	  for (let i = 0; i < parsed.data.length; i++) {
	    moleculesDB.pushEntry(moleculeCreator(parsed.data[i][moleculeField]), parsed.data[i]);
	    if (onStep) {
	      // eslint-disable-next-line no-await-in-loop
	      await onStep(i + 1, parsed.data.length);
	    }
	  }
	}
	appendCSV$1.default = appendCSV;

	var appendColor$1 = {};

	Object.defineProperty(appendColor$1, "__esModule", {
	  value: true
	});
	function appendColor(moleculesDB, options = {}) {
	  const {
	    dataLabel,
	    propertyLabel,
	    minValue,
	    maxValue,
	    minHue = 0,
	    maxHue = 360,
	    saturation = 65,
	    lightness = 65,
	    colorLabel = 'color'
	  } = options;
	  const db = moleculesDB.getDB();
	  let values;
	  if (dataLabel) {
	    values = db.map(result => result.data.map(datum => ({
	      value: datum[dataLabel],
	      data: datum
	    }))).flat();
	  } else if (propertyLabel) {
	    values = db.map(result => result.data.map(datum => ({
	      value: result.properties[propertyLabel],
	      data: datum
	    }))).flat();
	  } else {
	    values = db.map(result => result.data.map(datum => ({
	      value: undefined,
	      data: datum
	    }))).flat();
	  }
	  if (minValue !== undefined) {
	    values = values.forEach(value => {
	      if (value.value !== undefined && value.value < minValue) {
	        value.value = minValue;
	      }
	    });
	  }
	  if (maxValue !== undefined) {
	    values = values.forEach(value => {
	      if (value.value !== undefined && value.value > maxValue) {
	        value.value = maxValue;
	      }
	    });
	  }
	  const definedValues = values.filter(value => value.value !== undefined);
	  const min = Math.min(...definedValues.map(value => value.value));
	  const max = Math.max(...definedValues.map(value => value.value));
	  for (const value of values) {
	    if (value.value !== undefined) {
	      value.data[colorLabel] = `hsl(${Math.floor((value.value - min) / (max - min) * (maxHue - minHue) + minHue)},${saturation}%,${lightness}%)`;
	    } else {
	      value.data.color = 'black';
	    }
	  }
	}
	appendColor$1.default = appendColor;

	var appendEntries$1 = {};

	/*!
	 * isobject <https://github.com/jonschlinkert/isobject>
	 *
	 * Copyright (c) 2014-2017, Jon Schlinkert.
	 * Released under the MIT License.
	 */
	var isobject = function isObject(val) {
	  return val != null && typeof val === 'object' && Array.isArray(val) === false;
	};

	/*!
	 * get-value <https://github.com/jonschlinkert/get-value>
	 *
	 * Copyright (c) 2014-2018, Jon Schlinkert.
	 * Released under the MIT License.
	 */
	const isObject = isobject;
	var getValue = function (target, path, options) {
	  if (!isObject(options)) {
	    options = {
	      default: options
	    };
	  }
	  if (!isValidObject(target)) {
	    return typeof options.default !== 'undefined' ? options.default : target;
	  }
	  if (typeof path === 'number') {
	    path = String(path);
	  }
	  const isArray = Array.isArray(path);
	  const isString = typeof path === 'string';
	  const splitChar = options.separator || '.';
	  const joinChar = options.joinChar || (typeof splitChar === 'string' ? splitChar : '.');
	  if (!isString && !isArray) {
	    return target;
	  }
	  if (isString && path in target) {
	    return isValid(path, target, options) ? target[path] : options.default;
	  }
	  let segs = isArray ? path : split(path, splitChar, options);
	  let len = segs.length;
	  let idx = 0;
	  do {
	    let prop = segs[idx];
	    if (typeof prop === 'number') {
	      prop = String(prop);
	    }
	    while (prop && prop.slice(-1) === '\\') {
	      prop = join([prop.slice(0, -1), segs[++idx] || ''], joinChar, options);
	    }
	    if (prop in target) {
	      if (!isValid(prop, target, options)) {
	        return options.default;
	      }
	      target = target[prop];
	    } else {
	      let hasProp = false;
	      let n = idx + 1;
	      while (n < len) {
	        prop = join([prop, segs[n++]], joinChar, options);
	        if (hasProp = prop in target) {
	          if (!isValid(prop, target, options)) {
	            return options.default;
	          }
	          target = target[prop];
	          idx = n - 1;
	          break;
	        }
	      }
	      if (!hasProp) {
	        return options.default;
	      }
	    }
	  } while (++idx < len && isValidObject(target));
	  if (idx === len) {
	    return target;
	  }
	  return options.default;
	};
	function join(segs, joinChar, options) {
	  if (typeof options.join === 'function') {
	    return options.join(segs);
	  }
	  return segs[0] + joinChar + segs[1];
	}
	function split(path, splitChar, options) {
	  if (typeof options.split === 'function') {
	    return options.split(path);
	  }
	  return path.split(splitChar);
	}
	function isValid(key, target, options) {
	  if (typeof options.isValid === 'function') {
	    return options.isValid(key, target);
	  }
	  return true;
	}
	function isValidObject(val) {
	  return isObject(val) || Array.isArray(val) || typeof val === 'function';
	}

	var __importDefault$2 = commonjsGlobal && commonjsGlobal.__importDefault || function (mod) {
	  return mod && mod.__esModule ? mod : {
	    "default": mod
	  };
	};
	Object.defineProperty(appendEntries$1, "__esModule", {
	  value: true
	});
	const get_value_1 = __importDefault$2(getValue);
	/**
	 * Append an array of entries to the current database. An entry is an object that by default should contain a 'ocl' property containing idCode and optionally index and coordinates
	 * @param {*} moleculesDB
	 * @param {object[]} entries
	 * @param {object} [options={}]
	 * @param {string} [options.idCodePath='ocl.idCode']
	 * @param {string} [options.indexPath='ocl.index']
	 * @param {string} [options.coordinatesPath='ocl.coordinates']
	 * @param {string} [options.mwPath='mw']
	 * @param {string} [options.smilesPath]
	 * @param {string} [options.molfilePath]
	 * @param {function} [options.onStep] call back to execute after each molecule
	 * @returns
	 */
	async function appendEntries(moleculesDB, entries, options = {}) {
	  const {
	    onStep,
	    idCodePath = 'ocl.idCode',
	    indexPath = 'ocl.index',
	    coordinatesPath = 'ocl.coordinates',
	    mwPath = 'mw',
	    smilesPath,
	    molfilePath
	  } = options;
	  const Molecule = moleculesDB.OCL.Molecule;
	  for (let i = 0; i < entries.length; i++) {
	    let idCode;
	    const entry = entries[i];
	    let molecule;
	    if (smilesPath) {
	      molecule = Molecule.fromSmiles((0, get_value_1.default)(entry, smilesPath));
	    }
	    if (molfilePath && !molecule) {
	      molecule = Molecule.fromMolfile((0, get_value_1.default)(entry, molfilePath));
	    }
	    if (!molecule) {
	      idCode = (0, get_value_1.default)(entry, idCodePath);
	      if (idCode) {
	        const coordinates = (0, get_value_1.default)(entry, coordinatesPath);
	        molecule = Molecule.fromIDCode(idCode, coordinates || false);
	      }
	    }
	    const index = (0, get_value_1.default)(entry, indexPath);
	    const mw = (0, get_value_1.default)(entry, mwPath);
	    if (molecule) {
	      moleculesDB.pushEntry(molecule, entry, {
	        index,
	        mw
	      });
	    }
	    if (onStep) {
	      // eslint-disable-next-line no-await-in-loop
	      await onStep(i + 1, entries.length);
	    }
	  }
	}
	appendEntries$1.default = appendEntries;

	var appendSDF$1 = {};

	function getEntriesBoundaries(string, substring, eol) {
	  const res = [];
	  let previous = 0;
	  let next = 0;
	  while (next !== -1) {
	    next = string.indexOf(substring, previous);
	    if (next !== -1) {
	      res.push([previous, next]);
	      const nextMatch = string.indexOf(eol, next + substring.length);
	      if (nextMatch === -1) {
	        next = -1;
	      } else {
	        previous = nextMatch + eol.length;
	        next = previous;
	      }
	    } else {
	      res.push([previous, string.length]);
	    }
	  }
	  return res;
	}

	function getMolecule(sdfPart, labels, currentLabels, options) {
	  let parts = sdfPart.split(`${options.eol}>`);
	  if (parts.length === 0 || parts[0].length <= 5) return;
	  let molecule = {};
	  molecule.molfile = parts[0] + options.eol;
	  for (let j = 1; j < parts.length; j++) {
	    let lines = parts[j].split(options.eol);
	    let from = lines[0].indexOf('<');
	    let to = lines[0].indexOf('>');
	    let label = lines[0].substring(from + 1, to);
	    currentLabels.push(label);
	    if (!labels[label]) {
	      labels[label] = {
	        counter: 0,
	        isNumeric: options.dynamicTyping,
	        keep: false
	      };
	      if ((!options.exclude || options.exclude.indexOf(label) === -1) && (!options.include || options.include.indexOf(label) > -1)) {
	        labels[label].keep = true;
	        if (options.modifiers[label]) {
	          labels[label].modifier = options.modifiers[label];
	        }
	        if (options.forEach[label]) {
	          labels[label].forEach = options.forEach[label];
	        }
	      }
	    }
	    if (labels[label].keep) {
	      for (let k = 1; k < lines.length - 1; k++) {
	        if (molecule[label]) {
	          molecule[label] += options.eol + lines[k];
	        } else {
	          molecule[label] = lines[k];
	        }
	      }
	      if (labels[label].modifier) {
	        let modifiedValue = labels[label].modifier(molecule[label]);
	        if (modifiedValue === undefined || modifiedValue === null) {
	          delete molecule[label];
	        } else {
	          molecule[label] = modifiedValue;
	        }
	      }
	      if (labels[label].isNumeric) {
	        if (!isFinite(molecule[label]) || molecule[label].match(/^0[0-9]/)) {
	          labels[label].isNumeric = false;
	        }
	      }
	    }
	  }
	  return molecule;
	}

	/**
	 *  Parse a SDF file
	 * @param {string|ArrayBuffer|Uint8Array} sdf SDF file to parse
	 * @param {object} [options={}]
	 * @param {string[]} [options.include] List of fields to include
	 * @param {string[]} [options.exclude] List of fields to exclude
	 * @param {Function} [options.filter] Callback allowing to filter the molecules
	 * @param {boolean} [options.dynamicTyping] Dynamically type the data
	 * @param {object} [options.modifiers] Object containing callbacks to apply on some specific fields
	 * @param {boolean} [options.mixedEOL=false] Set to true if you know there is a mixture between \r\n and \n
	 * @param {string} [options.eol] Specify the end of line character. Default will be the one found in the file
	 */
	function parse(sdf, options = {}) {
	  options = {
	    ...options
	  };
	  if (options.modifiers === undefined) options.modifiers = {};
	  if (options.forEach === undefined) options.forEach = {};
	  if (options.dynamicTyping === undefined) options.dynamicTyping = true;
	  sdf = ensureString(sdf);
	  if (typeof sdf !== 'string') {
	    throw new TypeError('Parameter "sdf" must be a string');
	  }
	  if (options.eol === undefined) {
	    options.eol = '\n';
	    if (options.mixedEOL) {
	      sdf = sdf.replace(/\r\n/g, '\n');
	      sdf = sdf.replace(/\r/g, '\n');
	    } else {
	      // we will find the delimiter in order to be much faster and not use regular expression
	      let header = sdf.substr(0, 1000);
	      if (header.indexOf('\r\n') > -1) {
	        options.eol = '\r\n';
	      } else if (header.indexOf('\r') > -1) {
	        options.eol = '\r';
	      }
	    }
	  }
	  let entriesBoundaries = getEntriesBoundaries(sdf, `${options.eol}$$$$`, options.eol);
	  let molecules = [];
	  let labels = {};
	  let start = Date.now();
	  for (let i = 0; i < entriesBoundaries.length; i++) {
	    let sdfPart = sdf.substring(...entriesBoundaries[i]);
	    let currentLabels = [];
	    const molecule = getMolecule(sdfPart, labels, currentLabels, options);
	    if (!molecule) continue;
	    if (!options.filter || options.filter(molecule)) {
	      molecules.push(molecule);
	      // only now we can increase the counter
	      for (let j = 0; j < currentLabels.length; j++) {
	        labels[currentLabels[j]].counter++;
	      }
	    }
	  }
	  // all numeric fields should be converted to numbers
	  for (let label in labels) {
	    let currentLabel = labels[label];
	    if (currentLabel.isNumeric) {
	      currentLabel.minValue = Infinity;
	      currentLabel.maxValue = -Infinity;
	      for (let j = 0; j < molecules.length; j++) {
	        if (molecules[j][label]) {
	          let value = parseFloat(molecules[j][label]);
	          molecules[j][label] = value;
	          if (value > currentLabel.maxValue) {
	            currentLabel.maxValue = value;
	          }
	          if (value < currentLabel.minValue) {
	            currentLabel.minValue = value;
	          }
	        }
	      }
	    }
	  }

	  // we check that a label is in all the records
	  for (let key in labels) {
	    if (labels[key].counter === molecules.length) {
	      labels[key].always = true;
	    } else {
	      labels[key].always = false;
	    }
	  }
	  let statistics = [];
	  for (let key in labels) {
	    let statistic = labels[key];
	    statistic.label = key;
	    statistics.push(statistic);
	  }
	  return {
	    time: Date.now() - start,
	    molecules,
	    labels: Object.keys(labels),
	    statistics
	  };
	}

	function iterator() {
	  throw new Error('Iterator not implemented in the browser');
	}

	var src = /*#__PURE__*/Object.freeze({
		__proto__: null,
		parse: parse,
		iterator: iterator
	});

	var require$$1 = /*@__PURE__*/getAugmentedNamespace(src);

	Object.defineProperty(appendSDF$1, "__esModule", {
	  value: true
	});
	const ensure_string_1$2 = require$$0;
	const sdf_parser_1 = require$$1;
	async function appendSDF(moleculesDB, sdf, options = {}) {
	  const {
	    onStep
	  } = options;
	  sdf = (0, ensure_string_1$2.ensureString)(sdf);
	  if (typeof sdf !== 'string') {
	    throw new TypeError('sdf must be a string');
	  }
	  const parsed = (0, sdf_parser_1.parse)(sdf);
	  moleculesDB.statistics = parsed.statistics;
	  for (let i = 0; i < parsed.molecules.length; i++) {
	    const molecule = parsed.molecules[i];
	    moleculesDB.pushEntry(moleculesDB.OCL.Molecule.fromMolfile(molecule.molfile), molecule);
	    if (onStep) {
	      // eslint-disable-next-line no-await-in-loop
	      await onStep(i + 1, parsed.molecules.length);
	    }
	  }
	}
	appendSDF$1.default = appendSDF;

	var appendSmilesList$1 = {};

	Object.defineProperty(appendSmilesList$1, "__esModule", {
	  value: true
	});
	const ensure_string_1$1 = require$$0;
	async function appendSmilesList(moleculesDB, text, options = {}) {
	  const {
	    onStep
	  } = options;
	  text = (0, ensure_string_1$1.ensureString)(text);
	  if (typeof text !== 'string') {
	    throw new TypeError('text must be a string');
	  }
	  const smilesArray = text.split(/\r?\n/).map(line => line.trim()).filter(line => line);
	  for (let i = 0; i < smilesArray.length; i++) {
	    const oneSmiles = smilesArray[i];
	    moleculesDB.pushEntry(moleculesDB.OCL.Molecule.fromSmiles(oneSmiles));
	    if (onStep) {
	      // eslint-disable-next-line no-await-in-loop
	      await onStep(i + 1, smilesArray.length);
	    }
	  }
	}
	appendSmilesList$1.default = appendSmilesList;

	var pushEntry$1 = {};

	/**
	 *
	 * @param {MoleculesDB} moleculesDB
	 * @param {import('openchemlib').Molecule} molecule
	 * @param {object} data
	 * @param {object} [moleculeInfo]
	 * @param {string} [moleculeInfo.idCode]
	 * @param {number[]} [moleculeInfo.index]
	 */
	Object.defineProperty(pushEntry$1, "__esModule", {
	  value: true
	});
	function pushEntry(moleculesDB, molecule, data = {}, moleculeInfo = {}) {
	  // the following line could be the source of problems if the idCode version
	  // changes
	  const moleculeIDCode = getMoleculeIDCode(molecule, moleculeInfo);
	  let entry = moleculesDB.db[moleculeIDCode];
	  if (!entry) {
	    // a new molecule
	    entry = {
	      molecule,
	      properties: {},
	      data: [],
	      idCode: moleculeIDCode
	    };
	    moleculesDB.db[moleculeIDCode] = entry;
	    // ensure helper arrays needed for substructure search
	    molecule.ensureHelperArrays(moleculesDB.OCL.Molecule.cHelperRings);
	    if (!moleculeInfo.index) {
	      entry.index = molecule.getIndex();
	    } else {
	      entry.index = moleculeInfo.index;
	    }
	    let molecularFormula;
	    if (!moleculeInfo.mw) {
	      molecularFormula = molecule.getMolecularFormula();
	      entry.properties.mw = molecularFormula.relativeWeight;
	    } else {
	      entry.properties.mw = moleculeInfo.mw;
	    }
	    if (moleculesDB.computeProperties) {
	      if (!molecularFormula) {
	        molecularFormula = molecule.getMolecularFormula();
	      }
	      const properties = new moleculesDB.OCL.MoleculeProperties(molecule);
	      entry.properties.em = molecularFormula.absoluteWeight;
	      entry.properties.mf = molecularFormula.formula;
	      entry.properties.acceptorCount = properties.acceptorCount;
	      entry.properties.donorCount = properties.donorCount;
	      entry.properties.logP = properties.logP;
	      entry.properties.logS = properties.logS;
	      entry.properties.polarSurfaceArea = properties.polarSurfaceArea;
	      entry.properties.rotatableBondCount = properties.rotatableBondCount;
	      entry.properties.stereoCenterCount = properties.stereoCenterCount;
	    }
	  }
	  entry.data.push(data);
	}
	pushEntry$1.default = pushEntry;
	function getMoleculeIDCode(molecule, moleculeInfo) {
	  if (moleculeInfo.idCode) return moleculeInfo.idCode;
	  return molecule.getIDCode();
	}

	var pushMoleculeInfo$1 = {};

	Object.defineProperty(pushMoleculeInfo$1, "__esModule", {
	  value: true
	});
	function pushMoleculeInfo(moleculesDB, moleculeInfo, data = {}) {
	  if (typeof moleculeInfo !== 'object') {
	    throw new Error('pushMoleculeInfo requires an object as first parameter');
	  }
	  const Molecule = moleculesDB.OCL.Molecule;
	  let molecule;
	  if (moleculeInfo.molfile) {
	    molecule = Molecule.fromMolfile(moleculeInfo.molfile);
	  }
	  if (moleculeInfo.smiles) molecule = Molecule.fromSmiles(moleculeInfo.smiles);
	  if (moleculeInfo.idCode) {
	    if (moleculesDB.db[moleculeInfo.idCode]) {
	      molecule = moleculesDB.db[moleculeInfo.idCode].molecule;
	    } else {
	      molecule = Molecule.fromIDCode(moleculeInfo.idCode, moleculeInfo.coordinates || false);
	    }
	  }
	  if (molecule) {
	    moleculesDB.pushEntry(molecule, data, moleculeInfo);
	  }
	}
	pushMoleculeInfo$1.default = pushMoleculeInfo;

	var search$1 = {};

	var noWait$1 = {};

	Object.defineProperty(noWait$1, "__esModule", {
	  value: true
	});
	noWait$1.noWait = void 0;
	async function noWait() {
	  return new Promise(resolve => {
	    if (typeof setImmediate === 'function') {
	      setImmediate(() => resolve());
	    } else {
	      // didn't find a better way to do it in the browser
	      setTimeout(() => resolve(), 0);
	    }
	  });
	}
	noWait$1.noWait = noWait;

	var __importDefault$1 = commonjsGlobal && commonjsGlobal.__importDefault || function (mod) {
	  return mod && mod.__esModule ? mod : {
	    "default": mod
	  };
	};
	Object.defineProperty(search$1, "__esModule", {
	  value: true
	});
	search$1.searchAsync = search$1.search = void 0;
	const noWait_js_1 = noWait$1;
	const getMoleculeCreators_1 = __importDefault$1(getMoleculeCreators$1);
	class AbortError extends Error {
	  constructor() {
	    super(...arguments);
	    this.name = 'AbortError';
	    this.code = 20;
	  }
	}
	function getQuery(moleculesDB, query, options) {
	  const {
	    format = 'idCode'
	  } = options;
	  if (typeof query === 'string') {
	    const moleculeCreators = (0, getMoleculeCreators_1.default)(moleculesDB.OCL);
	    query = moleculeCreators.get(format.toLowerCase())(query);
	  } else if (!(query instanceof moleculesDB.OCL.Molecule)) {
	    throw new TypeError('toSearch must be a Molecule or string');
	  }
	  return query;
	}
	function search(moleculesDB, query = '', options = {}) {
	  const {
	    mode = 'substructure'
	  } = options;
	  query = getQuery(moleculesDB, query, options);
	  let result;
	  switch (mode.toLowerCase()) {
	    case 'exact':
	      result = exactSearch(moleculesDB, query);
	      break;
	    case 'substructure':
	      result = subStructureSearch(moleculesDB, query);
	      break;
	    case 'similarity':
	      result = similaritySearch(moleculesDB, query);
	      break;
	    default:
	      throw new Error(`unknown search mode: ${options.mode}`);
	  }
	  return processResult(result, options);
	}
	search$1.search = search;
	async function searchAsync(moleculesDB, query = '', options = {}) {
	  const {
	    mode = 'substructure'
	  } = options;
	  query = getQuery(moleculesDB, query, options);
	  let result;
	  switch (mode.toLowerCase()) {
	    case 'exact':
	      result = exactSearch(moleculesDB, query);
	      break;
	    case 'substructure':
	      result = await subStructureSearchAsync(moleculesDB, query, options);
	      break;
	    case 'similarity':
	      result = similaritySearch(moleculesDB, query);
	      break;
	    default:
	      throw new Error(`unknown search mode: ${options.mode}`);
	  }
	  return processResult(result, options);
	}
	search$1.searchAsync = searchAsync;
	function exactSearch(moleculesDB, query) {
	  query = query.getCompactCopy();
	  query.setFragment(false);
	  const queryIDCode = query.getIDCode();
	  const searchResult = moleculesDB.db[queryIDCode] ? [moleculesDB.db[queryIDCode]] : [];
	  return searchResult;
	}
	function substructureSearchBegin(moleculesDB, query) {
	  const searchResult = [];
	  if (query.getAllAtoms() === 0) {
	    for (const idCode in moleculesDB.db) {
	      searchResult.push(moleculesDB.db[idCode]);
	    }
	  }
	  return {
	    searchResult
	  };
	}
	function substructureSearchEnd(searchResult, queryMW) {
	  searchResult.sort((a, b) => {
	    return Math.abs(queryMW - a.properties.mw) - Math.abs(queryMW - b.properties.mw);
	  });
	  return searchResult;
	}
	function subStructureSearch(moleculesDB, query) {
	  const queryMW = getMW(query); // we
	  query = query.getCompactCopy();
	  query.setFragment(true);
	  const {
	    searchResult
	  } = substructureSearchBegin(moleculesDB, query);
	  if (searchResult.length === 0) {
	    const queryIndex = query.getIndex();
	    const searcher = moleculesDB.searcher;
	    searcher.setFragment(query, queryIndex);
	    for (const idCode in moleculesDB.db) {
	      const entry = moleculesDB.db[idCode];
	      searcher.setMolecule(entry.molecule, entry.index);
	      if (searcher.isFragmentInMolecule()) {
	        searchResult.push(entry);
	      }
	    }
	  }
	  return substructureSearchEnd(searchResult, queryMW);
	}
	async function subStructureSearchAsync(moleculesDB, query, options = {}) {
	  const queryMW = getMW(query); // we
	  query = query.getCompactCopy();
	  query.setFragment(true);
	  const {
	    interval = 100,
	    onStep,
	    controller
	  } = options;
	  let shouldAbort = false;
	  if (controller) {
	    const abortEventListener = () => {
	      shouldAbort = true;
	    };
	    controller.signal.addEventListener('abort', abortEventListener);
	  }
	  const {
	    searchResult
	  } = substructureSearchBegin(moleculesDB, query);
	  let begin = performance.now();
	  if (searchResult.length === 0) {
	    const queryIndex = query.getIndex();
	    const searcher = moleculesDB.searcher;
	    searcher.setFragment(query, queryIndex);
	    let index = 0;
	    const length = Object.keys(moleculesDB.db).length;
	    for (const idCode in moleculesDB.db) {
	      if (shouldAbort) {
	        throw new AbortError('Query aborted');
	      }
	      const entry = moleculesDB.db[idCode];
	      searcher.setMolecule(entry.molecule, entry.index);
	      if (searcher.isFragmentInMolecule()) {
	        searchResult.push(entry);
	      }
	      if ((onStep || controller) && performance.now() - begin >= interval) {
	        begin = performance.now();
	        if (onStep) {
	          onStep(index, length);
	        }
	        if (controller && !onStep) {
	          // eslint-disable-next-line no-await-in-loop
	          await (0, noWait_js_1.noWait)();
	        }
	      }
	      index++;
	    }
	  }
	  return substructureSearchEnd(searchResult, queryMW);
	}
	function similaritySearch(moleculesDB, query) {
	  const queryIndex = query.getIndex();
	  const queryMW = getMW(query);
	  const queryIdCode = query.getIDCode();
	  const searchResult = [];
	  let similarity;
	  for (const idCode in moleculesDB.db) {
	    const entry = moleculesDB.db[idCode];
	    if (entry.idCode === queryIdCode) {
	      similarity = Number.MAX_SAFE_INTEGER;
	    } else {
	      similarity = moleculesDB.OCL.SSSearcherWithIndex.getSimilarityTanimoto(queryIndex, entry.index) * 1000000 - Math.abs(queryMW - entry.properties.mw) / 10000;
	    }
	    searchResult.push({
	      similarity,
	      entry
	    });
	  }
	  searchResult.sort((a, b) => {
	    return b.similarity - a.similarity;
	  });
	  return searchResult.map(entry => entry.entry);
	}
	function getMW(query) {
	  const copy = query.getCompactCopy();
	  copy.setFragment(false);
	  return copy.getMolecularFormula().relativeWeight;
	}
	function processResult(entries, options = {}) {
	  const {
	    flattenResult = true,
	    keepMolecule = false,
	    limit = Number.MAX_SAFE_INTEGER
	  } = options;
	  const results = [];
	  if (flattenResult) {
	    for (const entry of entries) {
	      for (const data of entry.data) {
	        const result = {
	          data,
	          idCode: entry.idCode,
	          properties: entry.properties
	        };
	        if (keepMolecule) {
	          result.molecule = entry.molecule;
	        }
	        results.push(result);
	      }
	    }
	  } else {
	    for (const entry of entries) {
	      results.push({
	        data: entry.data,
	        idCode: entry.idCode,
	        properties: entry.properties,
	        molecule: keepMolecule ? entry.molecule : undefined
	      });
	    }
	  }
	  if (limit < results.length) results.length = limit;
	  return results;
	}

	var __importDefault = commonjsGlobal && commonjsGlobal.__importDefault || function (mod) {
	  return mod && mod.__esModule ? mod : {
	    "default": mod
	  };
	};
	Object.defineProperty(MoleculesDB$1, "__esModule", {
	  value: true
	});
	MoleculesDB$1.MoleculesDB = void 0;
	const appendCSV_1 = __importDefault(appendCSV$1);
	const appendColor_1 = __importDefault(appendColor$1);
	const appendEntries_js_1 = __importDefault(appendEntries$1);
	const appendSDF_1 = __importDefault(appendSDF$1);
	const appendSmilesList_1 = __importDefault(appendSmilesList$1);
	const pushEntry_1 = __importDefault(pushEntry$1);
	const pushMoleculeInfo_1 = __importDefault(pushMoleculeInfo$1);
	const search_1 = search$1;
	/*
	    this.db is an object with properties 'oclID' that has as value
	    an object that contains the following properties:
	    * molecule: an OCL molecule instance
	    * index: OCL index used for substructure searching
	    * properties: all the calculates properties
	    * data: array containing free data associated with this molecule
	  */
	class MoleculesDB {
	  /**
	   *
	   * @param {import('openchemlib')} OCL - openchemlib library
	   * @param {object} [options={}]
	   * @param {boolean} [options.computeProperties=false]
	   */
	  constructor(OCL, options = {}) {
	    const {
	      computeProperties = false
	    } = options;
	    this.OCL = OCL;
	    this.db = {};
	    this.statistics = null;
	    this.computeProperties = computeProperties;
	    this.searcher = new OCL.SSSearcherWithIndex();
	  }
	  /**
	   * Append an array of entries to the current database. An entry is an object that by default should contain a 'ocl' property containing idCode and optionally index and coordinates
	   * @param {*} moleculesDB
	   * @param {object[]} entries
	   * @param {object} [options={}]
	   * @param {string} [options.idCodePath='ocl.idCode']
	   * @param {string} [options.indexPath='ocl.index']
	   * @param {string} [options.coordinatesPath='ocl.coordinates']
	   * @param {string} [options.mwPath='mw']
	   * @param {string} [options.smilesPath]
	   * @param {string} [options.molfilePath]
	   * @param {function} [options.onStep] call back to execute after each molecule
	   */
	  appendEntries(entries, options) {
	    return (0, appendEntries_js_1.default)(this, entries, {
	      computeProperties: this.computeProperties,
	      ...options
	    });
	  }
	  /**
	   * append to the current database a CSV file
	   * @param {string|ArrayBuffer} csv - text file containing the comma separated value file
	   * @param {object} [options={}]
	   * @param {boolean} [options.header=true]
	   * @param {boolean} [options.dynamicTyping=true]
	   * @param {boolean} [options.skipEmptyLines=true]
	   * @param {function} [options.onStep] call back to execute after each molecule
	   */
	  appendCSV(csv, options) {
	    return (0, appendCSV_1.default)(this, csv, {
	      computeProperties: this.computeProperties,
	      ...options
	    });
	  }
	  /**
	   * Append a SDF to the current database
	   * @param {string|ArrayBuffer} sdf - text file containing the sdf
	   * @param {object} [options={}]
	   * @param {function} [options.onStep] call back to execute after each molecule
	   * @returns {DB}
	   */
	  appendSDF(sdf, options) {
	    return (0, appendSDF_1.default)(this, sdf, {
	      computeProperties: this.computeProperties,
	      ...options
	    });
	  }
	  /**
	   * Append a SDF to the current database
	   * @param {string|ArrayBuffer} smiles - text file containing a list of smiles
	   * @param {object} [options={}]
	   * @param {function} [options.onStep] call back to execute after each molecule
	   * @returns {DB}
	   */
	  appendSmilesList(text, options) {
	    return (0, appendSmilesList_1.default)(this, text, {
	      computeProperties: this.computeProperties,
	      ...options
	    });
	  }
	  /**
	   * Add a molecule to the current database
	   * @param {import('openchemlib').Molecule} molecule
	   * @param {object} [data={}]
	   * @param {object} [moleculeInfo={}] may contain precalculated index and mw
	   */
	  pushEntry(molecule, data, moleculeInfo) {
	    (0, pushEntry_1.default)(this, molecule, data, moleculeInfo);
	  }
	  /**
	   * Add an entry in the database
	   * @param {object} moleculeInfo - a molecule as a JSON that may contain the following properties: molfile, smiles, idCode, mf, index
	   * @param {object} [data={}]
	   */
	  pushMoleculeInfo(moleculeInfo, data) {
	    return (0, pushMoleculeInfo_1.default)(this, moleculeInfo, data);
	  }
	  /**
	   * Search in a MoleculesDB
	   * Inside the database all the same molecules are group together
	   * @param {string|OCL.Molecule} [query] smiles, molfile, idlCode or instance of Molecule to look for
	   * @param {object} [options={}]
	   * @param {'smiles'|'idCode'|'smarts'|'molfile'} [options.format='idCode'] - query format
	   * @param {string} [options.mode='substructure'] - search by 'substructure', 'exact' or 'similarity'
	   * @param {boolean} [options.flattenResult=true] - The database group the data for the same product. This allows to flatten the result
	   * @param {boolean} [options.keepMolecule=false] - keep the OCL.Molecule object in the result
	   * @param {number} [options.limit=Number.MAX_SAFE_INTEGER] - maximal number of result
	   * @return {Array} array of object of the type {(molecule), idCode, data, properties}
	   */
	  search(query, options) {
	    return (0, search_1.search)(this, query, options);
	  }
	  /**
	   * Search in a MoleculesDB
	   * Inside the database all the same molecules are group together
	   * @param {string|OCL.Molecule} [query] smiles, molfile, idCode or instance of Molecule to look for
	   * @param {object} [options={}]
	   * @param {'smiles'|'idCode'|'smarts'|'molfile'} [options.format='idCode'] - query format
	   * @param {string} [options.mode='substructure'] - search by 'substructure', 'exact' or 'similarity'
	   * @param {boolean} [options.flattenResult=true] - The database group the data for the same product. This allows to flatten the result
	   * @param {boolean} [options.keepMolecule=false] - keep the OCL.Molecule object in the result
	   * @param {number} [options.limit=Number.MAX_SAFE_INTEGER] - maximal number of result
	   * @param {number} [options.interval=100] - interval in ms to call the onStep callback
	   * @param {function} [options.onStep] - callback to execute after each interval
	   * @param {AbortController} [options.controler] - callback to execute to check if the search should be aborted
	   * @return {Promise<Array>} array of object of the type {(molecule), idCode, data, properties}
	   */
	  searchAsync(query, options) {
	    return (0, search_1.searchAsync)(this, query, options);
	  }
	  /**
	   * Returns an array with the current database
	   * @returns
	   */
	  getDB() {
	    return Object.keys(this.db).map(key => this.db[key]);
	  }
	  /**
	   * Append the property `data.color` to each entry based on a data or property label
	   * {object} [options={}]
	   * {string} [options.dataLabel] name of the property from `data` to use
	   * {string} [options.propertyLabel] name of the property from `properties` to use
	   * {number} [options.colorLabel='color'] name of the property to add in data that will contain the color
	   * {number} [options.minValue]
	   * {number} [options.maxValue]
	   * {number} [options.minHue=0]
	   * {number} [options.maxHue=360]
	   * {number} [options.saturation=65] percent of color saturation
	   * {number} [options.lightness=65] percent of color lightness
	   */
	  appendColor(options) {
	    (0, appendColor_1.default)(this, options);
	  }
	}
	MoleculesDB$1.MoleculesDB = MoleculesDB;

	var getAtomFeatures$1 = {};

	Object.defineProperty(getAtomFeatures$1, "__esModule", {
	  value: true
	});
	getAtomFeatures$1.getAtomFeatures = void 0;
	const makeRacemic_1 = makeRacemic$1;
	function getAtomFeatures(originalMolecule, options = {}) {
	  const OCL = originalMolecule.getOCL();
	  const {
	    sphere = 1
	  } = options;
	  const fragment = new OCL.Molecule(0, 0);
	  const results = [];
	  for (let rootAtom = 0; rootAtom < originalMolecule.getAllAtoms(); rootAtom++) {
	    let min = 0;
	    let max = 0;
	    const atomMask = new Array(originalMolecule.getAtoms());
	    const atomList = new Array(originalMolecule.getAtoms());
	    const molecule = originalMolecule.getCompactCopy();
	    for (let currentSphere = 0; currentSphere <= sphere; currentSphere++) {
	      if (max === 0) {
	        atomList[max] = rootAtom;
	        atomMask[rootAtom] = true;
	        max++;
	      } else {
	        let newMax = max;
	        for (let i = min; i < max; i++) {
	          const atom = atomList[i];
	          for (let j = 0; j < molecule.getAllConnAtoms(atom); j++) {
	            const connAtom = molecule.getConnAtom(atom, j);
	            if (!atomMask[connAtom]) {
	              atomMask[connAtom] = true;
	              atomList[newMax++] = connAtom;
	            }
	          }
	        }
	        min = max;
	        max = newMax;
	      }
	      molecule.copyMoleculeByAtoms(fragment, atomMask, true, null);
	      if (currentSphere === sphere) {
	        (0, makeRacemic_1.makeRacemic)(fragment);
	        results.push(fragment.getCanonizedIDCode());
	      }
	    }
	  }
	  const atoms = {};
	  for (const result of results) {
	    if (!atoms[result]) {
	      atoms[result] = 1;
	    } else {
	      atoms[result]++;
	    }
	  }
	  return atoms;
	}
	getAtomFeatures$1.getAtomFeatures = getAtomFeatures;

	var toVisualizerMolfile$1 = {};

	Object.defineProperty(toVisualizerMolfile$1, "__esModule", {
	  value: true
	});
	toVisualizerMolfile$1.toVisualizerMolfile = void 0;
	const getDiastereotopicAtomIDsAndH_1 = getDiastereotopicAtomIDsAndH$1;
	const getGroupedDiastereotopicAtomIDs_1 = getGroupedDiastereotopicAtomIDs$1;
	function toVisualizerMolfile(molecule, options = {}) {
	  const {
	    diastereotopic,
	    heavyAtomHydrogen
	  } = options;
	  let highlight = [];
	  let atoms = {};
	  if (diastereotopic) {
	    const hydrogenInfo = {};
	    const extendedIDs = (0, getDiastereotopicAtomIDsAndH_1.getDiastereotopicAtomIDsAndH)(molecule);
	    for (const line of extendedIDs) {
	      hydrogenInfo[line.oclID] = line;
	    }
	    const diaIDs = (0, getGroupedDiastereotopicAtomIDs_1.getGroupedDiastereotopicAtomIDs)(molecule);
	    for (const diaID of diaIDs) {
	      atoms[diaID.oclID] = diaID.atoms;
	      highlight.push(diaID.oclID);
	      if (heavyAtomHydrogen) {
	        if (hydrogenInfo[diaID.oclID] && hydrogenInfo[diaID.oclID].nbHydrogens > 0) {
	          for (const id of hydrogenInfo[diaID.oclID].hydrogenOCLIDs) {
	            highlight.push(id);
	            atoms[id] = diaID.atoms;
	          }
	        }
	      }
	    }
	  } else {
	    const size = molecule.getAllAtoms();
	    highlight = new Array(size).fill(0).map((a, index) => index);
	    atoms = highlight.map(a => [a]);
	  }
	  const molfile = {
	    type: 'mol2d',
	    value: molecule.toMolfile(),
	    _highlight: highlight,
	    _atoms: atoms
	  };
	  return molfile;
	}
	toVisualizerMolfile$1.toVisualizerMolfile = toVisualizerMolfile;

	var parseDwar$1 = {};

	var getParts$1 = {};

	var getCamelCase$1 = {};

	Object.defineProperty(getCamelCase$1, "__esModule", {
	  value: true
	});
	getCamelCase$1.getCamelCase = void 0;
	/**
	 * Converts a string to camel case.
	 * @param {string} name
	 * @returns {string}
	 */
	function getCamelCase(name) {
	  return name.replace(/[ -][a-z]/g, string => string[1].toUpperCase());
	}
	getCamelCase$1.getCamelCase = getCamelCase;

	Object.defineProperty(getParts$1, "__esModule", {
	  value: true
	});
	getParts$1.getParts = void 0;
	const getCamelCase_1 = getCamelCase$1;
	function getParts(text) {
	  const lines = text.split(/\r?\n/);
	  const parts = {
	    data: []
	  };
	  let currentPart = parts.data;
	  let currentLabel = '';
	  for (const line of lines) {
	    if (line.startsWith('</')) {
	      // close existing part
	      if (!currentLabel === line.slice(2, -1)) {
	        throw new Error('This should not happen');
	      }
	      currentLabel = '';
	      currentPart = parts.data;
	    } else if (line.startsWith('<') && !line.includes('=')) {
	      // open new part
	      if (currentLabel) {
	        throw new Error('This should not happen');
	      }
	      currentLabel = line.slice(1, -1);
	      const target = (0, getCamelCase_1.getCamelCase)(currentLabel);
	      parts[target] = [];
	      currentPart = parts[target];
	    } else if (currentLabel) {
	      // add line to current part
	      currentPart.push(line);
	    } else {
	      //data lines
	      currentPart.push(line);
	    }
	  }
	  return parts;
	}
	getParts$1.getParts = getParts;

	var parseColumnbProperties$1 = {};

	Object.defineProperty(parseColumnbProperties$1, "__esModule", {
	  value: true
	});
	parseColumnbProperties$1.parseColumnbProperties = void 0;
	function parseColumnbProperties(lines) {
	  lines = lines.map(line => {
	    const [key, value] = line.slice(1, -1).split('=');
	    return {
	      key,
	      value: value.slice(1, -1)
	    };
	  });
	  const columnProperties = {};
	  let currentColumnName = '';
	  for (const line of lines) {
	    switch (line.key) {
	      case 'columnName':
	        currentColumnName = line.value;
	        columnProperties[currentColumnName] = {};
	        break;
	      case 'columnProperty':
	        {
	          if (!currentColumnName) {
	            throw new Error('This should not happen');
	          }
	          const [key, value] = line.value.split('\t');
	          columnProperties[currentColumnName][key] = value;
	        }
	        break;
	      default:
	        throw new Error('This should not happen');
	    }
	  }
	  for (const key in columnProperties) {
	    const columnPropery = columnProperties[key];
	    if (columnProperties[key].parent) {
	      const target = columnProperties[columnPropery.parent];
	      if (!target) {
	        throw new Error('Parent column not found');
	      }
	      if (!target.related) {
	        target.related = {};
	      }
	      target.related[columnPropery.specialType] = key;
	    }
	  }
	  return columnProperties;
	}
	parseColumnbProperties$1.parseColumnbProperties = parseColumnbProperties;

	var parseData$1 = {};

	Object.defineProperty(parseData$1, "__esModule", {
	  value: true
	});
	parseData$1.parseData = void 0;
	function parseData(lines, options = {}) {
	  lines = lines.filter(line => !line.match(/^\s*$/));
	  const {
	    columnProperties = {}
	  } = options;
	  const headers = lines.shift().split('\t').map(header => {
	    if (columnProperties[header]) {
	      return {
	        label: header,
	        ...columnProperties[header]
	      };
	    }
	    return {
	      label: header
	    };
	  });
	  const entries = [];
	  const rawEntries = [];
	  for (const line of lines) {
	    const fields = line.split('\t');
	    const rawEntry = {};
	    headers.forEach((header, index) => {
	      rawEntry[header.label] = fields[index];
	    });
	    rawEntries.push(rawEntry);
	    const entry = {};
	    headers.forEach(header => {
	      if (header.parent) return;
	      entry[header.label] = valueEhnhancer(header, rawEntry);
	    });
	    entries.push(entry);
	  }
	  return {
	    entries,
	    rawEntries
	  };
	}
	parseData$1.parseData = parseData;
	function valueEhnhancer(header, rawEntry) {
	  if (header?.specialType === 'rxncode') {
	    return `${rawEntry[header.label]}#${rawEntry[header.related.atomMapping]}#${rawEntry[header.related.idcoordinates2D]}`;
	  }
	  return rawEntry[header.label];
	}

	Object.defineProperty(parseDwar$1, "__esModule", {
	  value: true
	});
	parseDwar$1.parseDwar = void 0;
	const ensure_string_1 = require$$0;
	const getParts_1 = getParts$1;
	const parseColumnbProperties_1 = parseColumnbProperties$1;
	const parseData_1 = parseData$1;
	/**
	 * Convert a DataWarrior database into a JSON object
	 * @param {string} text
	 * @returns
	 */
	function parseDwar(text) {
	  text = (0, ensure_string_1.ensureString)(text);
	  const parts = (0, getParts_1.getParts)(text);
	  improveParts(parts);
	  return parts;
	}
	parseDwar$1.parseDwar = parseDwar;
	function improveParts(parts) {
	  for (const key in parts) {
	    switch (key) {
	      case 'columnProperties':
	        parts[key] = (0, parseColumnbProperties_1.parseColumnbProperties)(parts[key]);
	        break;
	      case 'data':
	        break;
	      default:
	        parts[key] = parseDefault(parts[key]);
	    }
	  }
	  const data = (0, parseData_1.parseData)(parts.data, {
	    columnProperties: parts.columnProperties
	  });
	  parts.data = data.entries;
	  parts.rawData = data.rawEntries;
	}
	function parseDefault(lines) {
	  const result = {};
	  for (const line of lines) {
	    const [key, value] = line.slice(1, -1).split('=');
	    result[key] = value.slice(1, -1);
	  }
	  return result;
	}

	var fragmentAcyclicSingleBonds$1 = {};

	Object.defineProperty(fragmentAcyclicSingleBonds$1, "__esModule", {
	  value: true
	});
	fragmentAcyclicSingleBonds$1.fragmentAcyclicSingleBonds = void 0;
	const getMF_1 = getMF$1;
	function fragmentAcyclicSingleBonds(molecule) {
	  const OCL = molecule.getOCL();
	  const atoms = [];
	  for (let i = 0; i < molecule.getAllAtoms(); i++) {
	    const atom = {};
	    atoms.push(atom);
	    atom.i = i;
	    atom.links = []; // we will store connected atoms of broken bonds
	  }
	  const bonds = [];
	  for (let i = 0; i < molecule.getAllBonds(); i++) {
	    const bond = {};
	    bonds.push(bond);
	    bond.i = i;
	    bond.order = molecule.getBondOrder(i);
	    bond.atom1 = molecule.getBondAtom(0, i);
	    bond.atom2 = molecule.getBondAtom(1, i);
	    bond.type = molecule.getBondType(i);
	    bond.isAromatic = molecule.isAromaticBond(i);
	    bond.isRingBond = molecule.isRingBond(i);
	    if (!bond.isAromatic && (bond.type & 0b11) === 1 && !bond.isRingBond) {
	      bond.selected = true;
	      atoms[bond.atom1].links.push(bond.atom2);
	      atoms[bond.atom2].links.push(bond.atom1);
	    }
	  }
	  const brokenMolecule = molecule.getCompactCopy();
	  for (const bond of bonds) {
	    if (bond.selected) {
	      brokenMolecule.markBondForDeletion(bond.i);
	    }
	  }
	  brokenMolecule.deleteMarkedAtomsAndBonds();
	  const fragmentMap = [];
	  const nbFragments = brokenMolecule.getFragmentNumbers(fragmentMap);
	  const results = [];
	  for (let i = 0; i < nbFragments; i++) {
	    const result = {};
	    result.atomMap = [];
	    const includeAtom = fragmentMap.map(id => {
	      return id === i;
	    });
	    const fragment = new OCL.Molecule(0, 0);
	    const atomMap = [];
	    brokenMolecule.copyMoleculeByAtoms(fragment, includeAtom, false, atomMap);
	    // we will add some R groups at the level of the broken bonds
	    for (let j = 0; j < atomMap.length; j++) {
	      if (atomMap[j] > -1) {
	        result.atomMap.push(j);
	        if (atoms[j].links.length > 0) {
	          atoms[j].links.forEach(() => {
	            fragment.addBond(atomMap[j], fragment.addAtom(154), 1);
	          });
	        }
	      }
	    }
	    fragment.setFragment(false);
	    result.idCode = fragment.getIDCode();
	    result.mf = (0, getMF_1.getMF)(fragment).mf.replace(/R[1-9]?/, '');
	    results.push(result);
	  }
	  return results;
	}
	fragmentAcyclicSingleBonds$1.fragmentAcyclicSingleBonds = fragmentAcyclicSingleBonds;

	var Reactions$1 = {};

	var appendOCLReaction$1 = {};

	Object.defineProperty(appendOCLReaction$1, "__esModule", {
	  value: true
	});
	appendOCLReaction$1.appendOCLReaction = void 0;
	/**
	 * @description Append the OCL reaction to the reaction object
	 * @param {Array} reactions array of reactions objects with rxnCode and label
	 * @param {Object} OCL OCL object
	 * @param {Object} [options={}]
	 * @param {import('cheminfo-types').Logger} [options.logger]
	 *
	 * @returns {Array} array of reactions objects with rxnCode, label and oclReaction (a decoded version of rxnCode reaction)
	 */
	function appendOCLReaction(reactions, OCL, options = {}) {
	  const {
	    logger
	  } = options;
	  const newReactions = [];
	  for (const reaction of reactions) {
	    if (reaction.rxnCode) {
	      newReactions.push({
	        ...reaction,
	        oclReaction: OCL.ReactionEncoder.decode(reaction.rxnCode)
	      });
	    } else if (logger) {
	      logger.warn(reaction, 'Reaction without rxnCode');
	    }
	  }
	  return newReactions;
	}
	appendOCLReaction$1.appendOCLReaction = appendOCLReaction;

	var applyOneReactantReactions$1 = {};

	var checkIfExistsOrAddInfo$1 = {};

	/**
	 *
	 * @param {*} processedMolecules
	 * @param {import('openchemlib').Molecule|string} molecule
	 * @param {*} options
	 * @returns
	 */
	Object.defineProperty(checkIfExistsOrAddInfo$1, "__esModule", {
	  value: true
	});
	checkIfExistsOrAddInfo$1.checkIfExistsOrAddInfo = void 0;
	function checkIfExistsOrAddInfo(processedMolecules, molecule, options) {
	  const {
	    moleculeInfoCallback,
	    asReagent,
	    asProduct
	  } = options;
	  const idCode = typeof molecule === 'string' ? idCode : molecule.getIDCode();
	  if (processedMolecules.has(idCode)) {
	    const entry = processedMolecules.get(idCode);
	    let exists = false;
	    if (asReagent) {
	      if (entry.asReagent) {
	        exists = true;
	      } else {
	        entry.asReagent = true;
	      }
	    }
	    if (asProduct) {
	      if (entry.asProduct) {
	        exists = true;
	      } else {
	        entry.asProduct = true;
	      }
	    }
	    return {
	      exists,
	      info: entry
	    };
	  } else {
	    let info = {
	      idCode,
	      asReagent,
	      asProduct,
	      info: {}
	    };
	    if (moleculeInfoCallback) {
	      info.info = moleculeInfoCallback(molecule);
	    }
	    processedMolecules.set(idCode, info);
	    return {
	      exists: false,
	      info
	    };
	  }
	}
	checkIfExistsOrAddInfo$1.checkIfExistsOrAddInfo = checkIfExistsOrAddInfo;

	Object.defineProperty(applyOneReactantReactions$1, "__esModule", {
	  value: true
	});
	applyOneReactantReactions$1.applyOneReactantReactions = void 0;
	const checkIfExistsOrAddInfo_1 = checkIfExistsOrAddInfo$1;
	/**
	 * @description apply one reaction to one reactant
	 * @param {import('openchemlib').Molecule[]} reactants
	 * @param {Array<Object>} reactions rxnCode of the reaction
	 * @param {Object} options options to apply the reaction
	 * @param {number} options.currentDepth current depth of the recursion
	 * @param {number} options.maxDepth max depth of the recursion
	 * @param {number} options.maxCurrentDepth max depth of the recursion for this set of reactions
	 * @param {number} options.limitReactions limit the number of reactions
	 * @param {Object} options.stats stats of the recursion
	 * @param {number} options.stats.counter number of reactions
	 * @param {Map} options.processedMolecules set of processed molecules
	 * @param {Array} options.trees array of trees of previous recursions
	 * @param {import('openchemlib')} options.OCL OCL object
	 * @returns {Array} array of results
	 */
	function applyOneReactantReactions(tree, reactions, options) {
	  const {
	    currentDepth,
	    maxDepth,
	    maxCurrentDepth,
	    processedMolecules,
	    OCL,
	    logger
	  } = options;
	  if (tree.molecules.length !== 1) {
	    logger?.warn('applyOneReactantReactions:tree.reactants.length!==1', tree.reactants.length);
	    return [];
	  }
	  const reactant = OCL.Molecule.fromIDCode(tree.molecules[0].idCode);
	  const todoNextDepth = [];
	  // if the current depth is greater than the max depth, we stop the recursion and return an empty array
	  if (currentDepth > maxCurrentDepth || tree.depth >= maxDepth) {
	    return [];
	  }
	  const existsAndInfo = (0, checkIfExistsOrAddInfo_1.checkIfExistsOrAddInfo)(processedMolecules, reactant, {
	    ...options,
	    asReagent: true
	  });
	  // check if the reactant has already been processed
	  if (existsAndInfo.exists) {
	    return [];
	  }
	  for (const reaction of reactions) {
	    if (options.stats.counter >= options.limitReactions) {
	      return [];
	    }
	    const reactor = new OCL.Reactor(reaction.oclReaction);
	    // isMatching is true if the reactant is matching the reaction else we continue to the next reaction
	    const isMatching = Boolean(reactor.setReactant(0, reactant));
	    if (isMatching) {
	      options.stats.counter++;
	      // get the products of the reaction
	      const oneReactionProducts = reactor.getProducts();
	      for (const oneReactionProduct of oneReactionProducts) {
	        for (const reactionProduct of oneReactionProduct) {
	          // get the info of the product (idCode, mf)
	          const productExistsAndInfo = (0, checkIfExistsOrAddInfo_1.checkIfExistsOrAddInfo)(processedMolecules, reactionProduct, {
	            ...options,
	            asProduct: true
	          });
	          // if the product has not been processed yet, we add it to the list of products and we add it to the list of todoNextDepth
	          if (!productExistsAndInfo.exists) {
	            // eslint-disable-next-line no-unused-vars
	            const {
	              oclReaction,
	              needToBeCharged,
	              ...reactionWithoutOCL
	            } = reaction;
	            const oneReaction = {
	              reaction: reactionWithoutOCL,
	              depth: tree.depth + 1,
	              isValid: true,
	              currentDepth,
	              molecules: [(0, checkIfExistsOrAddInfo_1.checkIfExistsOrAddInfo)(processedMolecules, reactionProduct, options).info]
	            };
	            if (!tree.children) tree.children = [];
	            tree.children.push(oneReaction);
	            todoNextDepth.push(() => {
	              return applyOneReactantReactions(oneReaction, reactions, {
	                ...options,
	                currentDepth: options.currentDepth + 1
	              });
	            });
	          }
	        }
	      }
	    }
	  }
	  // by returning todoNextDepth, we make sure that the recursion will continue
	  return todoNextDepth;
	}
	applyOneReactantReactions$1.applyOneReactantReactions = applyOneReactantReactions;

	var getFilteredTrees$1 = {};

	Object.defineProperty(getFilteredTrees$1, "__esModule", {
	  value: true
	});
	getFilteredTrees$1.getFilteredTrees = void 0;
	function getFilteredTrees(reactions, options = {}) {
	  const {
	    filter = () => true
	  } = options;
	  const nodesToKeep = reactions.getNodes().filter(filter);
	  const parentMap = reactions.getParentMap();
	  for (let currentNode of nodesToKeep) {
	    const parent = parentMap.get(currentNode);
	    if (parent && nodesToKeep.includes(parent) === false) {
	      nodesToKeep.push(parent);
	    }
	  }
	  return getValidChildren(reactions.trees, {
	    nodesToKeep
	  });
	}
	getFilteredTrees$1.getFilteredTrees = getFilteredTrees;
	function getValidChildren(nodes, options) {
	  const {
	    nodesToKeep
	  } = options;
	  const validNodes = nodes.filter(node => nodesToKeep.includes(node)).map(node => ({
	    ...node
	  }));
	  for (const node of validNodes) {
	    if (node.children) {
	      const validChildren = node.children.filter(child => nodesToKeep.includes(child));
	      if (validChildren.length > 0) {
	        node.children = getValidChildren(validChildren, {
	          nodesToKeep
	        });
	      } else {
	        delete node.children;
	      }
	    }
	  }
	  return validNodes;
	}

	var getLeaves$1 = {};

	Object.defineProperty(getLeaves$1, "__esModule", {
	  value: true
	});
	getLeaves$1.getLeaves = void 0;
	function getLeaves(trees) {
	  const leaves = [];
	  for (const tree of trees) {
	    appendLeavesSS(leaves, tree);
	  }
	  return leaves;
	}
	getLeaves$1.getLeaves = getLeaves;
	function appendLeavesSS(leaves, currentBranch) {
	  if (!currentBranch.children || currentBranch.children.length === 0) {
	    leaves.push(currentBranch);
	    return;
	  }
	  for (const child of currentBranch.children) {
	    appendLeavesSS(leaves, child);
	  }
	}

	var getNodes$1 = {};

	Object.defineProperty(getNodes$1, "__esModule", {
	  value: true
	});
	getNodes$1.getNodes = void 0;
	function getNodes(trees) {
	  const nodes = [];
	  for (const tree of trees) {
	    getNodesSS(nodes, tree);
	  }
	  return nodes;
	}
	getNodes$1.getNodes = getNodes;
	function getNodesSS(nodes, currentBranch) {
	  nodes.push(currentBranch);
	  for (const child of currentBranch?.children || []) {
	    getNodesSS(nodes, child);
	  }
	}

	Object.defineProperty(Reactions$1, "__esModule", {
	  value: true
	});
	Reactions$1.Reactions = void 0;
	const appendOCLReaction_js_1 = appendOCLReaction$1;
	const applyOneReactantReactions_js_1 = applyOneReactantReactions$1;
	const checkIfExistsOrAddInfo_js_1 = checkIfExistsOrAddInfo$1;
	const getFilteredTrees_js_1 = getFilteredTrees$1;
	const getLeaves_js_1 = getLeaves$1;
	const getNodes_js_1 = getNodes$1;
	class Reactions {
	  /**
	   *
	   * @param {object} [options={}]
	   * @param {import('cheminfo-types').Logger} logger
	   * @param {number} [options.maxDepth=5]
	   * @param {function} [options.moleculeInfoCallback]
	   * @param {boolean} [options.skipProcessed=true]
	   */
	  constructor(OCL, options = {}) {
	    this.moleculeInfoCallback = options.moleculeInfoCallback;
	    this.maxDepth = options.maxDepth ?? 5;
	    this.limitReactions = options.limitReactions ?? 200;
	    this.skipProcessed = options.skipProcessed ?? true;
	    this.logger = options.logger;
	    this.processedMolecules = new Map();
	    this.OCL = OCL;
	    this.trees = [];
	    this.moleculeInfo = {}; // a cache containing molecule information like mw, etc.
	  }
	  /**
	   * We need to call this method for all the reactants on which we want to apply the reactions.
	   * If there is only one reactant, we call this method with an array of one reactant.
	   * If there are multiple reactants, we call this method with an array of the reactants.
	   * This method has to be called for all the reactants
	   * @param {import('openchemlib').Molecule[]|string[]} molecules
	   */
	  appendHead(moleculesOrIDCodes) {
	    if (!Array.isArray(moleculesOrIDCodes)) {
	      throw new TypeError('reactants must be an array');
	    }
	    const molecules = moleculesOrIDCodes.map(molecule => (0, checkIfExistsOrAddInfo_js_1.checkIfExistsOrAddInfo)(this.processedMolecules, molecule, {
	      moleculeInfoCallback: this.moleculeInfoCallback
	    }).info);
	    const tree = {
	      molecules,
	      depth: 0,
	      isValid: true // this node could be implied in reactions
	    };
	    this.trees.push(tree);
	  }
	  /**
	   * Returns all the leaves of the trees
	   * @returns
	   */
	  getLeaves() {
	    return (0, getLeaves_js_1.getLeaves)(this.trees);
	  }
	  /**
	   * Returns all the nodes of the trees
	   * @returns
	   */
	  getNodes() {
	    return (0, getNodes_js_1.getNodes)(this.trees);
	  }
	  getParentMap() {
	    const parentMap = new Map();
	    const nodes = this.getNodes();
	    for (const node of nodes) {
	      if (node.children) {
	        for (const child of node.children) {
	          parentMap.set(child, node);
	        }
	      }
	    }
	    return parentMap;
	  }
	  /**
	   * When applying reactions some branches may be dead because it can not be implied in any reaction.
	   * This is the case when we specify a 'min' reaction depth.
	   * This will returno only the valid nodes
	   * @returns
	   */
	  getValidNodes() {
	    return this.getNodes().filter(node => node.isValid);
	  }
	  /**
	   *
	   * @param {object} [options={}]
	   * @param {(object):boolean} [options.filter] - a function that will be called for each node and return true if the node should be kept
	   */
	  getFilteredReactions(options = {}) {
	    const filteredReactions = new Reactions();
	    filteredReactions.moleculeInfoCallback = this.moleculeInfoCallback;
	    filteredReactions.maxDepth = this.maxDepth;
	    filteredReactions.limitReactions = this.limitReactions;
	    filteredReactions.skipProcessed = this.skipProcessed;
	    filteredReactions.logger = this.logger;
	    filteredReactions.processedMolecules = this.processedMolecules;
	    filteredReactions.OCL = this.OCL;
	    filteredReactions.moleculeInfo = this.moleculeInfo; // a cache containing molecule information like mw, etc.
	    filteredReactions.trees = (0, getFilteredTrees_js_1.getFilteredTrees)(this, options);
	    return filteredReactions;
	  }
	  /**
	   *
	   * @param {object[]} reactions - array of reactions that should be applied
	   * @param {object} [options={}]
	   * @param {number} [options.min=0] min depth of the reaction
	   * @param {number} [options.max=3] max depth of the reaction
	   */
	  applyOneReactantReactions(reactions, options = {}) {
	    const {
	      min = 0,
	      max = 3
	    } = options;
	    clearAsFromProcessedMolecules(this.processedMolecules);
	    const nodes = this.getNodes().filter(node => node.isValid);
	    nodes.forEach(node => {
	      node.currentDepth = 0;
	    });
	    reactions = (0, appendOCLReaction_js_1.appendOCLReaction)(reactions, this.OCL);
	    const stats = {
	      counter: 0
	    };
	    // Start the recursion by applying the first level of reactions
	    for (const node of nodes) {
	      let todoCurrentLevel = (0, applyOneReactantReactions_js_1.applyOneReactantReactions)(node, reactions, {
	        OCL: this.OCL,
	        currentDepth: 1,
	        processedMolecules: this.processedMolecules,
	        moleculeInfoCallback: this.moleculeInfoCallback,
	        maxDepth: this.maxDepth,
	        maxCurrentDepth: max,
	        stats,
	        limitReactions: this.limitReactions
	      });
	      do {
	        const nexts = [];
	        for (const todo of todoCurrentLevel) {
	          nexts.push(todo());
	        }
	        todoCurrentLevel = nexts.flat();
	      } while (todoCurrentLevel.length > 0);
	    }
	    const newNodes = this.getNodes().filter(node => node.isValid);
	    for (const node of newNodes) {
	      if (node.currentDepth < min || node.currentDepth > max) {
	        node.isValid = false;
	      }
	      delete node.currentDepth;
	    }
	  }
	}
	Reactions$1.Reactions = Reactions;
	function clearAsFromProcessedMolecules(processedMolecules) {
	  for (const [, value] of processedMolecules) {
	    if (value.asReagent) {
	      value.asReagent = false;
	    }
	    if (value.asProduct) {
	      value.asProduct = false;
	    }
	  }
	}

	var getHints$1 = {};

	// this page allows to debug the hints: https://my.cheminfo.org/?viewURL=https%3A%2F%2Fmyviews.cheminfo.org%2Fdb%2Fvisualizer%2Fentry%2F108024089da99d0cb70a57724486d0c6%2Fview.json
	Object.defineProperty(getHints$1, "__esModule", {
	  value: true
	});
	getHints$1.getHints = void 0;
	const defaultPossibleHints = [{
	  idCode: 'gFp@DiTt@@B',
	  message: 'Did you think about benzene derivatives?'
	}, {
	  idCode: 'gFx@@eJf`@@P',
	  message: 'Did you think about pyridine derivatives?'
	}, {
	  idCode: 'eFHBLFLYpB@QVE_cD',
	  message: 'An aromatic cycle can be an heterocycle.'
	}, {
	  idCode: 'eFHBJD',
	  message: 'Adding a carbonyl could help.'
	}, {
	  idCode: 'gGP`@df]j`MekEZFBpp_zmPl',
	  message: 'What about an ester?'
	}, {
	  idCode: 'gGY@DDf]j`MiXrppUaYl',
	  message: 'There is an amide function in the molecule.'
	}, {
	  idCode: 'gCi@DDfZ@~btl',
	  message: 'There is a primary amide in the molecule.'
	}, {
	  idCode: 'gC``@dfZ@~bl',
	  message: 'You should think about carboxylic acids.'
	}, {
	  idCode: 'eF@Hp\\pcc',
	  message: 'What about a non-aromatic ring?'
	}, {
	  idCode: 'eF@H`_qb@c',
	  message: 'How to get such a high DBE?'
	}, {
	  idCode: 'eF@Hh\\q@',
	  message: 'What about having an olefin?'
	}, {
	  idCode: 'gFp@DiTt@@CxbyZoV',
	  anyMatches: ['daD@@DjWjXHB@CYL{qQZWUmX', 'gFp@DiTt@@CxbyYnv'],
	  message: 'Disubstituted aromatic ring can be o, m or p.',
	  remarks: 'ortho'
	}, {
	  idCode: 'daD@@DjWjXHB@CYL{qQZWUmX',
	  anyMatches: ['gFp@DiTt@@CxbyZoV', 'gFp@DiTt@@CxbyYnv'],
	  message: 'Disubstituted aromatic ring can be o, m or p.',
	  remarks: 'meta'
	}, {
	  idCode: 'gFp@DiTt@@CxbyYnv',
	  anyMatches: ['daD@@DjWjXHB@CYL{qQZWUmX', 'gFp@DiTt@@CxbyZoV'],
	  message: 'Disubstituted aromatic ring can be o, m or p.',
	  remarks: 'para'
	}, {
	  idCode: 'gGP`@df]j`H',
	  anyMatches: ['gGP`@dfUj`H'],
	  message: 'You should check the orientation of the ester.'
	}, {
	  idCode: 'gOq@@drm\\@@Aa@',
	  message: 'Alcohols on aromatic ring may have surprising chemical shifts.'
	}, {
	  idCode: 'gOx@@drm\\@@A}A@',
	  message: 'NH on aromatic ring may have surprising chemical shifts.'
	}, {
	  idCode: 'eFHBJGuP',
	  message: 'Hydrogens on a carbonyl have very high chemical shifts.'
	}];
	function getHints(correct, answer, options = {}) {
	  const hints = [...checkMF(correct, answer), ...checkStereoAndTautomer(correct, answer)];
	  const {
	    possibleHints = defaultPossibleHints
	  } = options;
	  const OCL = correct.getOCL();
	  const searcherCorrect = new OCL.SSSearcher();
	  searcherCorrect.setMolecule(correct);
	  const searcherAnswer = new OCL.SSSearcher();
	  searcherAnswer.setMolecule(answer);
	  for (const possibleTip of possibleHints) {
	    const {
	      anyMatches
	    } = possibleTip;
	    if (anyMatches) {
	      let match = false;
	      for (const anyMatch of anyMatches) {
	        const matchFragment = OCL.Molecule.fromIDCode(anyMatch);
	        searcherAnswer.setFragment(matchFragment);
	        if (searcherAnswer.isFragmentInMolecule()) {
	          match = true;
	          break;
	        }
	      }
	      if (!match) continue;
	    }
	    const fragment = OCL.Molecule.fromIDCode(possibleTip.idCode);
	    searcherCorrect.setFragment(fragment);
	    searcherAnswer.setFragment(fragment);
	    if (searcherCorrect.isFragmentInMolecule() && !searcherAnswer.isFragmentInMolecule()) {
	      hints.push(possibleTip);
	    }
	  }
	  return hints.map(hint => ({
	    ...hint,
	    hash: getHash(JSON.stringify(hint))
	  }));
	}
	getHints$1.getHints = getHints;
	function checkMF(correct, answer) {
	  const mfCorrect = correct.getMolecularFormula().formula;
	  const mfAnswer = answer.getMolecularFormula().formula;
	  if (mfCorrect === mfAnswer) return [];
	  return [{
	    message: 'You should check the molecular formula.'
	  }];
	}
	function checkStereoAndTautomer(correct, answer) {
	  if (correct.getIDCode() === answer.getIDCode()) return [];
	  if (getNoStereoIDCode(correct) === getNoStereoIDCode(answer)) {
	    return [{
	      message: 'There is only a problem with stereochemistry.'
	    }];
	  }
	  if (getTautomerIDCode(correct) === getTautomerIDCode(answer)) {
	    return [{
	      message: "Weird, you didn't draw the expected tautomer."
	    }];
	  }
	  return [];
	}
	function getTautomerIDCode(molecule) {
	  const OCL = molecule.getOCL();
	  return OCL.CanonizerUtil.getIDCode(molecule, OCL.CanonizerUtil.TAUTOMER);
	}
	function getNoStereoIDCode(molecule) {
	  const OCL = molecule.getOCL();
	  return OCL.CanonizerUtil.getIDCode(molecule, OCL.CanonizerUtil.NOSTEREO);
	}
	/*
	    https://github.com/bryc/code/blob/master/jshash/experimental/cyrb53.js
	    cyrb53a (c) 2023 bryc (github.com/bryc)
	    License: Public domain. Attribution appreciated.
	    The original cyrb53 has a slight mixing bias in the low bits of h1.
	    This shouldn't be a huge problem, but I want to try to improve it.
	    This new version should have improved avalanche behavior, but
	    it is not quite final, I may still find improvements.
	    So don't expect it to always produce the same output.
	*/
	function getHash(str, seed = 0) {
	  let h1 = 0xdeadbeef ^ seed;
	  let h2 = 0x41c6ce57 ^ seed;
	  for (let i = 0, ch; i < str.length; i++) {
	    ch = str.charCodeAt(i);
	    h1 = Math.imul(h1 ^ ch, 0x85ebca77);
	    h2 = Math.imul(h2 ^ ch, 0xc2b2ae3d);
	  }
	  h1 ^= Math.imul(h1 ^ h2 >>> 15, 0x735a2d97);
	  h2 ^= Math.imul(h2 ^ h1 >>> 15, 0xcaf649a9);
	  h1 ^= h2 >>> 16;
	  h2 ^= h1 >>> 16;
	  return 2097152 * (h2 >>> 0) + (h1 >>> 11);
	}

	(function (exports) {

	  var __createBinding = commonjsGlobal && commonjsGlobal.__createBinding || (Object.create ? function (o, m, k, k2) {
	    if (k2 === undefined) k2 = k;
	    var desc = Object.getOwnPropertyDescriptor(m, k);
	    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
	      desc = {
	        enumerable: true,
	        get: function () {
	          return m[k];
	        }
	      };
	    }
	    Object.defineProperty(o, k2, desc);
	  } : function (o, m, k, k2) {
	    if (k2 === undefined) k2 = k;
	    o[k2] = m[k];
	  });
	  var __exportStar = commonjsGlobal && commonjsGlobal.__exportStar || function (m, exports) {
	    for (var p in m) if (p !== "default" && !Object.prototype.hasOwnProperty.call(exports, p)) __createBinding(exports, m, p);
	  };
	  Object.defineProperty(exports, "__esModule", {
	    value: true
	  });
	  __exportStar(TopicMolecule$1, exports);
	  __exportStar(ensureHeterotopicChiralBonds$1, exports);
	  __exportStar(getDiastereotopicAtomIDs$1, exports);
	  __exportStar(getDiastereotopicAtomIDsAndH$1, exports);
	  __exportStar(getGroupedDiastereotopicAtomIDs$1, exports);
	  __exportStar(getDiastereotopicAtomIDsFromMolfile$1, exports);
	  __exportStar(toDiastereotopicSVG$1, exports);
	  __exportStar(getHoseCodes$1, exports);
	  __exportStar(getHoseCodesAndInfo$1, exports);
	  __exportStar(getHoseCodesForAtom$1, exports);
	  __exportStar(getHoseCodesForAtoms, exports);
	  __exportStar(getHoseCodesFromDiastereotopicID$1, exports);
	  __exportStar(getHoseCodesForPath$1, exports);
	  __exportStar(createPolymer$1, exports);
	  __exportStar(combineSmiles$1, exports);
	  __exportStar(getAtomsInfo$1, exports);
	  __exportStar(getConnectivityMatrix$1, exports);
	  __exportStar(getImplicitHydrogensCount$1, exports);
	  __exportStar(getMF$1, exports);
	  __exportStar(getCharge$1, exports);
	  __exportStar(getProperties$1, exports);
	  __exportStar(getAtoms$1, exports);
	  __exportStar(isCsp3$1, exports);
	  __exportStar(makeRacemic$1, exports);
	  __exportStar(nbOH$1, exports);
	  __exportStar(nbCOOH$1, exports);
	  __exportStar(nbCHO$1, exports);
	  __exportStar(nbNH2$1, exports);
	  __exportStar(nbCN$1, exports);
	  __exportStar(nbLabileH$1, exports);
	  __exportStar(tagAtom$1, exports);
	  __exportStar(toggleHydrogens$1, exports);
	  __exportStar(getPathsInfo$1, exports);
	  __exportStar(getPathAndTorsion$1, exports);
	  __exportStar(getShortestPaths$1, exports);
	  __exportStar(MoleculesDB$1, exports);
	  __exportStar(getAtomFeatures$1, exports);
	  __exportStar(toVisualizerMolfile$1, exports);
	  __exportStar(parseDwar$1, exports);
	  __exportStar(fragmentAcyclicSingleBonds$1, exports);
	  __exportStar(Reactions$1, exports);
	  __exportStar(getHints$1, exports);
	})(lib);
	var index = /*@__PURE__*/getDefaultExportFromCjs(lib);

	return index;

}));
//# sourceMappingURL=openchemlib-utils.js.map
