/**
 * mass-tools
 * @version v7.20.0
 * @link https://github.com/cheminfo/mass-tools#readme
 * @license MIT
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.MassTools = {}));
})(this, (function (exports) { 'use strict';

    // eslint-disable-next-line @typescript-eslint/unbound-method
    const toString = Object.prototype.toString;
    /**
     * Checks if an object is an instance of an Array (array or typed array, except those that contain bigint values).
     *
     * @param value - Object to check.
     * @returns True if the object is an array or a typed array.
     */
    function isAnyArray$1(value) {
      const tag = toString.call(value);
      return tag.endsWith('Array]') && !tag.includes('Big');
    }

    var libEsm = /*#__PURE__*/Object.freeze({
        __proto__: null,
        isAnyArray: isAnyArray$1
    });

    var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

    function getAugmentedNamespace(n) {
    	if (n.__esModule) return n;
    	var a = Object.defineProperty({}, '__esModule', {value: true});
    	Object.keys(n).forEach(function (k) {
    		var d = Object.getOwnPropertyDescriptor(n, k);
    		Object.defineProperty(a, k, d.get ? d : {
    			enumerable: true,
    			get: function () {
    				return n[k];
    			}
    		});
    	});
    	return a;
    }

    function commonjsRequire (path) {
    	throw new Error('Could not dynamically require "' + path + '". Please configure the dynamicRequireTargets or/and ignoreDynamicRequires option of @rollup/plugin-commonjs appropriately for this require call to work.');
    }

    /**
     * Calculates the median of an array
     *
     * @param input - Array containing values
     * @returns - median
     */
    function xMedian(input) {
      if (!isAnyArray$1(input)) {
        throw new TypeError('input must be an array');
      }
      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      const array = input.slice();
      let low = 0;
      let high = array.length - 1;
      let middle = 0;
      let currentLow = 0;
      let currentHigh = 0;
      const median = calcMiddle(low, high);
      while (true) {
        if (high <= low) {
          return array[median];
        }
        if (high === low + 1) {
          if (array[low] > array[high]) {
            swap(array, low, high);
          }
          return array[median];
        }
        // Find median of low, middle and high items; swap into position low
        middle = calcMiddle(low, high);
        if (array[middle] > array[high]) swap(array, middle, high);
        if (array[low] > array[high]) swap(array, low, high);
        if (array[middle] > array[low]) swap(array, middle, low);
        // Swap low item (now in position middle) into position (low+1)
        swap(array, middle, low + 1);
        // Nibble from each end towards middle, swapping items when stuck
        currentLow = low + 1;
        currentHigh = high;
        while (true) {
          do currentLow++; while (array[low] > array[currentLow]);
          do currentHigh--; while (array[currentHigh] > array[low]);
          if (currentHigh < currentLow) {
            break;
          }
          swap(array, currentLow, currentHigh);
        }
        // Swap middle item (in position low) back into correct position
        swap(array, low, currentHigh);
        // Re-set active partition
        if (currentHigh <= median) {
          low = currentLow;
        }
        if (currentHigh >= median) {
          high = currentHigh - 1;
        }
      }
    }
    function swap(array, i, j) {
      const temp = array[j];
      array[j] = array[i];
      array[i] = temp;
    }
    function calcMiddle(i, j) {
      return Math.floor((i + j) / 2);
    }

    /**
     * This function xAdd the first array by the second array or a constant value to each element of the first array
     *
     * @param array1 - the first array
     * @param array2 - the second array or number
     */
    function xAdd(array1, array2) {
      let isConstant = false;
      let constant = 0;
      if (isAnyArray$1(array2)) {
        if (array1.length !== array2.length) {
          throw new Error('xAdd: size of array1 and array2 must be identical');
        }
      } else {
        isConstant = true;
        constant = array2;
      }
      const array3 = new Float64Array(array1.length);
      if (isConstant) {
        for (let i = 0; i < array1.length; i++) {
          array3[i] = array1[i] + constant;
        }
      } else {
        for (let i = 0; i < array1.length; i++) {
          array3[i] = array1[i] + array2[i];
        }
      }
      return array3;
    }

    /**
     * This function
     * @param output - undefined or a new array
     * @param length - length of the output array
     * @returns
     */
    function getOutputArray(output, length) {
      if (output !== undefined) {
        if (!isAnyArray$1(output)) {
          throw new TypeError('output option must be an array if specified');
        }
        if (output.length !== length) {
          throw new TypeError('the output array does not have the correct length');
        }
        return output;
      } else {
        return new Float64Array(length);
      }
    }

    /**
     * Checks if input is of type array
     *
     * @param input - input
     */
    function xCheck(input, options = {}) {
      const {
        minLength
      } = options;
      if (!isAnyArray$1(input)) {
        throw new TypeError('input must be an array');
      }
      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      if (minLength && input.length < minLength) {
        throw new Error(`input must have a length of at least ${minLength}`);
      }
    }

    var matrix$1 = {};

    var require$$0 = /*@__PURE__*/getAugmentedNamespace(libEsm);

    function max(input) {
      var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      if (!isAnyArray$1(input)) {
        throw new TypeError('input must be an array');
      }
      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;
      if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
        throw new Error('fromIndex must be a positive integer smaller than length');
      }
      if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
        throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
      }
      var maxValue = input[fromIndex];
      for (var i = fromIndex + 1; i < toIndex; i++) {
        if (input[i] > maxValue) maxValue = input[i];
      }
      return maxValue;
    }

    function min(input) {
      var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      if (!isAnyArray$1(input)) {
        throw new TypeError('input must be an array');
      }
      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;
      if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
        throw new Error('fromIndex must be a positive integer smaller than length');
      }
      if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
        throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
      }
      var minValue = input[fromIndex];
      for (var i = fromIndex + 1; i < toIndex; i++) {
        if (input[i] < minValue) minValue = input[i];
      }
      return minValue;
    }

    function rescale$1(input) {
      var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      if (!isAnyArray$1(input)) {
        throw new TypeError('input must be an array');
      } else if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      var output;
      if (options.output !== undefined) {
        if (!isAnyArray$1(options.output)) {
          throw new TypeError('output option must be an array if specified');
        }
        output = options.output;
      } else {
        output = new Array(input.length);
      }
      var currentMin = min(input);
      var currentMax = max(input);
      if (currentMin === currentMax) {
        throw new RangeError('minimum and maximum input values are equal. Cannot rescale a constant array');
      }
      var _options$min = options.min,
        minValue = _options$min === void 0 ? options.autoMinMax ? currentMin : 0 : _options$min,
        _options$max = options.max,
        maxValue = _options$max === void 0 ? options.autoMinMax ? currentMax : 1 : _options$max;
      if (minValue >= maxValue) {
        throw new RangeError('min option must be smaller than max option');
      }
      var factor = (maxValue - minValue) / (currentMax - currentMin);
      for (var i = 0; i < input.length; i++) {
        output[i] = (input[i] - currentMin) * factor + minValue;
      }
      return output;
    }

    var libEs6 = /*#__PURE__*/Object.freeze({
        __proto__: null,
        'default': rescale$1
    });

    var require$$1 = /*@__PURE__*/getAugmentedNamespace(libEs6);

    Object.defineProperty(matrix$1, '__esModule', {
      value: true
    });
    var isAnyArray = require$$0;
    var rescale = require$$1;
    const indent = ' '.repeat(2);
    const indentData = ' '.repeat(4);
    function inspectMatrix() {
      return inspectMatrixWithOptions(this);
    }
    function inspectMatrixWithOptions(matrix, options = {}) {
      const {
        maxRows = 15,
        maxColumns = 10,
        maxNumSize = 8,
        padMinus = 'auto'
      } = options;
      return `${matrix.constructor.name} {
${indent}[
${indentData}${inspectData(matrix, maxRows, maxColumns, maxNumSize, padMinus)}
${indent}]
${indent}rows: ${matrix.rows}
${indent}columns: ${matrix.columns}
}`;
    }
    function inspectData(matrix, maxRows, maxColumns, maxNumSize, padMinus) {
      const {
        rows,
        columns
      } = matrix;
      const maxI = Math.min(rows, maxRows);
      const maxJ = Math.min(columns, maxColumns);
      const result = [];
      if (padMinus === 'auto') {
        padMinus = false;
        loop: for (let i = 0; i < maxI; i++) {
          for (let j = 0; j < maxJ; j++) {
            if (matrix.get(i, j) < 0) {
              padMinus = true;
              break loop;
            }
          }
        }
      }
      for (let i = 0; i < maxI; i++) {
        let line = [];
        for (let j = 0; j < maxJ; j++) {
          line.push(formatNumber(matrix.get(i, j), maxNumSize, padMinus));
        }
        result.push(`${line.join(' ')}`);
      }
      if (maxJ !== columns) {
        result[result.length - 1] += ` ... ${columns - maxColumns} more columns`;
      }
      if (maxI !== rows) {
        result.push(`... ${rows - maxRows} more rows`);
      }
      return result.join(`\n${indentData}`);
    }
    function formatNumber(num, maxNumSize, padMinus) {
      return (num >= 0 && padMinus ? ` ${formatNumber2(num, maxNumSize - 1)}` : formatNumber2(num, maxNumSize)).padEnd(maxNumSize);
    }
    function formatNumber2(num, len) {
      // small.length numbers should be as is
      let str = num.toString();
      if (str.length <= len) return str;

      // (7)'0.00123' is better then (7)'1.23e-2'
      // (8)'0.000123' is worse then (7)'1.23e-3',
      let fix = num.toFixed(len);
      if (fix.length > len) {
        fix = num.toFixed(Math.max(0, len - (fix.length - len)));
      }
      if (fix.length <= len && !fix.startsWith('0.000') && !fix.startsWith('-0.000')) {
        return fix;
      }

      // well, if it's still too long the user should've used longer numbers
      let exp = num.toExponential(len);
      if (exp.length > len) {
        exp = num.toExponential(Math.max(0, len - (exp.length - len)));
      }
      return exp.slice(0);
    }
    function installMathOperations(AbstractMatrix, Matrix) {
      AbstractMatrix.prototype.add = function add(value) {
        if (typeof value === 'number') return this.addS(value);
        return this.addM(value);
      };
      AbstractMatrix.prototype.addS = function addS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) + value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.addM = function addM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) + matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.add = function add(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.add(value);
      };
      AbstractMatrix.prototype.sub = function sub(value) {
        if (typeof value === 'number') return this.subS(value);
        return this.subM(value);
      };
      AbstractMatrix.prototype.subS = function subS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) - value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.subM = function subM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) - matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.sub = function sub(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sub(value);
      };
      AbstractMatrix.prototype.subtract = AbstractMatrix.prototype.sub;
      AbstractMatrix.prototype.subtractS = AbstractMatrix.prototype.subS;
      AbstractMatrix.prototype.subtractM = AbstractMatrix.prototype.subM;
      AbstractMatrix.subtract = AbstractMatrix.sub;
      AbstractMatrix.prototype.mul = function mul(value) {
        if (typeof value === 'number') return this.mulS(value);
        return this.mulM(value);
      };
      AbstractMatrix.prototype.mulS = function mulS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) * value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.mulM = function mulM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) * matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.mul = function mul(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.mul(value);
      };
      AbstractMatrix.prototype.multiply = AbstractMatrix.prototype.mul;
      AbstractMatrix.prototype.multiplyS = AbstractMatrix.prototype.mulS;
      AbstractMatrix.prototype.multiplyM = AbstractMatrix.prototype.mulM;
      AbstractMatrix.multiply = AbstractMatrix.mul;
      AbstractMatrix.prototype.div = function div(value) {
        if (typeof value === 'number') return this.divS(value);
        return this.divM(value);
      };
      AbstractMatrix.prototype.divS = function divS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) / value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.divM = function divM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) / matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.div = function div(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.div(value);
      };
      AbstractMatrix.prototype.divide = AbstractMatrix.prototype.div;
      AbstractMatrix.prototype.divideS = AbstractMatrix.prototype.divS;
      AbstractMatrix.prototype.divideM = AbstractMatrix.prototype.divM;
      AbstractMatrix.divide = AbstractMatrix.div;
      AbstractMatrix.prototype.mod = function mod(value) {
        if (typeof value === 'number') return this.modS(value);
        return this.modM(value);
      };
      AbstractMatrix.prototype.modS = function modS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) % value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.modM = function modM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) % matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.mod = function mod(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.mod(value);
      };
      AbstractMatrix.prototype.modulus = AbstractMatrix.prototype.mod;
      AbstractMatrix.prototype.modulusS = AbstractMatrix.prototype.modS;
      AbstractMatrix.prototype.modulusM = AbstractMatrix.prototype.modM;
      AbstractMatrix.modulus = AbstractMatrix.mod;
      AbstractMatrix.prototype.and = function and(value) {
        if (typeof value === 'number') return this.andS(value);
        return this.andM(value);
      };
      AbstractMatrix.prototype.andS = function andS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) & value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.andM = function andM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) & matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.and = function and(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.and(value);
      };
      AbstractMatrix.prototype.or = function or(value) {
        if (typeof value === 'number') return this.orS(value);
        return this.orM(value);
      };
      AbstractMatrix.prototype.orS = function orS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) | value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.orM = function orM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) | matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.or = function or(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.or(value);
      };
      AbstractMatrix.prototype.xor = function xor(value) {
        if (typeof value === 'number') return this.xorS(value);
        return this.xorM(value);
      };
      AbstractMatrix.prototype.xorS = function xorS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) ^ value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.xorM = function xorM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) ^ matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.xor = function xor(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.xor(value);
      };
      AbstractMatrix.prototype.leftShift = function leftShift(value) {
        if (typeof value === 'number') return this.leftShiftS(value);
        return this.leftShiftM(value);
      };
      AbstractMatrix.prototype.leftShiftS = function leftShiftS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) << value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.leftShiftM = function leftShiftM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) << matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.leftShift = function leftShift(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.leftShift(value);
      };
      AbstractMatrix.prototype.signPropagatingRightShift = function signPropagatingRightShift(value) {
        if (typeof value === 'number') return this.signPropagatingRightShiftS(value);
        return this.signPropagatingRightShiftM(value);
      };
      AbstractMatrix.prototype.signPropagatingRightShiftS = function signPropagatingRightShiftS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) >> value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.signPropagatingRightShiftM = function signPropagatingRightShiftM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) >> matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.signPropagatingRightShift = function signPropagatingRightShift(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.signPropagatingRightShift(value);
      };
      AbstractMatrix.prototype.rightShift = function rightShift(value) {
        if (typeof value === 'number') return this.rightShiftS(value);
        return this.rightShiftM(value);
      };
      AbstractMatrix.prototype.rightShiftS = function rightShiftS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) >>> value);
          }
        }
        return this;
      };
      AbstractMatrix.prototype.rightShiftM = function rightShiftM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) >>> matrix.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.rightShift = function rightShift(matrix, value) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.rightShift(value);
      };
      AbstractMatrix.prototype.zeroFillRightShift = AbstractMatrix.prototype.rightShift;
      AbstractMatrix.prototype.zeroFillRightShiftS = AbstractMatrix.prototype.rightShiftS;
      AbstractMatrix.prototype.zeroFillRightShiftM = AbstractMatrix.prototype.rightShiftM;
      AbstractMatrix.zeroFillRightShift = AbstractMatrix.rightShift;
      AbstractMatrix.prototype.not = function not() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, ~this.get(i, j));
          }
        }
        return this;
      };
      AbstractMatrix.not = function not(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.not();
      };
      AbstractMatrix.prototype.abs = function abs() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.abs(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.abs = function abs(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.abs();
      };
      AbstractMatrix.prototype.acos = function acos() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.acos(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.acos = function acos(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.acos();
      };
      AbstractMatrix.prototype.acosh = function acosh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.acosh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.acosh = function acosh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.acosh();
      };
      AbstractMatrix.prototype.asin = function asin() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.asin(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.asin = function asin(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.asin();
      };
      AbstractMatrix.prototype.asinh = function asinh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.asinh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.asinh = function asinh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.asinh();
      };
      AbstractMatrix.prototype.atan = function atan() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.atan(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.atan = function atan(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.atan();
      };
      AbstractMatrix.prototype.atanh = function atanh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.atanh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.atanh = function atanh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.atanh();
      };
      AbstractMatrix.prototype.cbrt = function cbrt() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.cbrt(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.cbrt = function cbrt(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.cbrt();
      };
      AbstractMatrix.prototype.ceil = function ceil() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.ceil(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.ceil = function ceil(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.ceil();
      };
      AbstractMatrix.prototype.clz32 = function clz32() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.clz32(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.clz32 = function clz32(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.clz32();
      };
      AbstractMatrix.prototype.cos = function cos() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.cos(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.cos = function cos(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.cos();
      };
      AbstractMatrix.prototype.cosh = function cosh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.cosh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.cosh = function cosh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.cosh();
      };
      AbstractMatrix.prototype.exp = function exp() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.exp(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.exp = function exp(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.exp();
      };
      AbstractMatrix.prototype.expm1 = function expm1() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.expm1(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.expm1 = function expm1(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.expm1();
      };
      AbstractMatrix.prototype.floor = function floor() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.floor(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.floor = function floor(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.floor();
      };
      AbstractMatrix.prototype.fround = function fround() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.fround(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.fround = function fround(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.fround();
      };
      AbstractMatrix.prototype.log = function log() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.log(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.log = function log(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.log();
      };
      AbstractMatrix.prototype.log1p = function log1p() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.log1p(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.log1p = function log1p(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.log1p();
      };
      AbstractMatrix.prototype.log10 = function log10() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.log10(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.log10 = function log10(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.log10();
      };
      AbstractMatrix.prototype.log2 = function log2() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.log2(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.log2 = function log2(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.log2();
      };
      AbstractMatrix.prototype.round = function round() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.round(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.round = function round(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.round();
      };
      AbstractMatrix.prototype.sign = function sign() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.sign(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.sign = function sign(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sign();
      };
      AbstractMatrix.prototype.sin = function sin() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.sin(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.sin = function sin(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sin();
      };
      AbstractMatrix.prototype.sinh = function sinh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.sinh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.sinh = function sinh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sinh();
      };
      AbstractMatrix.prototype.sqrt = function sqrt() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.sqrt(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.sqrt = function sqrt(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.sqrt();
      };
      AbstractMatrix.prototype.tan = function tan() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.tan(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.tan = function tan(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.tan();
      };
      AbstractMatrix.prototype.tanh = function tanh() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.tanh(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.tanh = function tanh(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.tanh();
      };
      AbstractMatrix.prototype.trunc = function trunc() {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.trunc(this.get(i, j)));
          }
        }
        return this;
      };
      AbstractMatrix.trunc = function trunc(matrix) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.trunc();
      };
      AbstractMatrix.pow = function pow(matrix, arg0) {
        const newMatrix = new Matrix(matrix);
        return newMatrix.pow(arg0);
      };
      AbstractMatrix.prototype.pow = function pow(value) {
        if (typeof value === 'number') return this.powS(value);
        return this.powM(value);
      };
      AbstractMatrix.prototype.powS = function powS(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.pow(this.get(i, j), value));
          }
        }
        return this;
      };
      AbstractMatrix.prototype.powM = function powM(matrix) {
        matrix = Matrix.checkMatrix(matrix);
        if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
          throw new RangeError('Matrices dimensions must be equal');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, Math.pow(this.get(i, j), matrix.get(i, j)));
          }
        }
        return this;
      };
    }

    /**
     * @private
     * Check that a row index is not out of bounds
     * @param {Matrix} matrix
     * @param {number} index
     * @param {boolean} [outer]
     */
    function checkRowIndex(matrix, index, outer) {
      let max = outer ? matrix.rows : matrix.rows - 1;
      if (index < 0 || index > max) {
        throw new RangeError('Row index out of range');
      }
    }

    /**
     * @private
     * Check that a column index is not out of bounds
     * @param {Matrix} matrix
     * @param {number} index
     * @param {boolean} [outer]
     */
    function checkColumnIndex(matrix, index, outer) {
      let max = outer ? matrix.columns : matrix.columns - 1;
      if (index < 0 || index > max) {
        throw new RangeError('Column index out of range');
      }
    }

    /**
     * @private
     * Check that the provided vector is an array with the right length
     * @param {Matrix} matrix
     * @param {Array|Matrix} vector
     * @return {Array}
     * @throws {RangeError}
     */
    function checkRowVector(matrix, vector) {
      if (vector.to1DArray) {
        vector = vector.to1DArray();
      }
      if (vector.length !== matrix.columns) {
        throw new RangeError('vector size must be the same as the number of columns');
      }
      return vector;
    }

    /**
     * @private
     * Check that the provided vector is an array with the right length
     * @param {Matrix} matrix
     * @param {Array|Matrix} vector
     * @return {Array}
     * @throws {RangeError}
     */
    function checkColumnVector(matrix, vector) {
      if (vector.to1DArray) {
        vector = vector.to1DArray();
      }
      if (vector.length !== matrix.rows) {
        throw new RangeError('vector size must be the same as the number of rows');
      }
      return vector;
    }
    function checkRowIndices(matrix, rowIndices) {
      if (!isAnyArray.isAnyArray(rowIndices)) {
        throw new TypeError('row indices must be an array');
      }
      for (let i = 0; i < rowIndices.length; i++) {
        if (rowIndices[i] < 0 || rowIndices[i] >= matrix.rows) {
          throw new RangeError('row indices are out of range');
        }
      }
    }
    function checkColumnIndices(matrix, columnIndices) {
      if (!isAnyArray.isAnyArray(columnIndices)) {
        throw new TypeError('column indices must be an array');
      }
      for (let i = 0; i < columnIndices.length; i++) {
        if (columnIndices[i] < 0 || columnIndices[i] >= matrix.columns) {
          throw new RangeError('column indices are out of range');
        }
      }
    }
    function checkRange(matrix, startRow, endRow, startColumn, endColumn) {
      if (arguments.length !== 5) {
        throw new RangeError('expected 4 arguments');
      }
      checkNumber('startRow', startRow);
      checkNumber('endRow', endRow);
      checkNumber('startColumn', startColumn);
      checkNumber('endColumn', endColumn);
      if (startRow > endRow || startColumn > endColumn || startRow < 0 || startRow >= matrix.rows || endRow < 0 || endRow >= matrix.rows || startColumn < 0 || startColumn >= matrix.columns || endColumn < 0 || endColumn >= matrix.columns) {
        throw new RangeError('Submatrix indices are out of range');
      }
    }
    function newArray(length, value = 0) {
      let array = [];
      for (let i = 0; i < length; i++) {
        array.push(value);
      }
      return array;
    }
    function checkNumber(name, value) {
      if (typeof value !== 'number') {
        throw new TypeError(`${name} must be a number`);
      }
    }
    function checkNonEmpty(matrix) {
      if (matrix.isEmpty()) {
        throw new Error('Empty matrix has no elements to index');
      }
    }
    function sumByRow(matrix) {
      let sum = newArray(matrix.rows);
      for (let i = 0; i < matrix.rows; ++i) {
        for (let j = 0; j < matrix.columns; ++j) {
          sum[i] += matrix.get(i, j);
        }
      }
      return sum;
    }
    function sumByColumn(matrix) {
      let sum = newArray(matrix.columns);
      for (let i = 0; i < matrix.rows; ++i) {
        for (let j = 0; j < matrix.columns; ++j) {
          sum[j] += matrix.get(i, j);
        }
      }
      return sum;
    }
    function sumAll(matrix) {
      let v = 0;
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          v += matrix.get(i, j);
        }
      }
      return v;
    }
    function productByRow(matrix) {
      let sum = newArray(matrix.rows, 1);
      for (let i = 0; i < matrix.rows; ++i) {
        for (let j = 0; j < matrix.columns; ++j) {
          sum[i] *= matrix.get(i, j);
        }
      }
      return sum;
    }
    function productByColumn(matrix) {
      let sum = newArray(matrix.columns, 1);
      for (let i = 0; i < matrix.rows; ++i) {
        for (let j = 0; j < matrix.columns; ++j) {
          sum[j] *= matrix.get(i, j);
        }
      }
      return sum;
    }
    function productAll(matrix) {
      let v = 1;
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          v *= matrix.get(i, j);
        }
      }
      return v;
    }
    function varianceByRow(matrix, unbiased, mean) {
      const rows = matrix.rows;
      const cols = matrix.columns;
      const variance = [];
      for (let i = 0; i < rows; i++) {
        let sum1 = 0;
        let sum2 = 0;
        let x = 0;
        for (let j = 0; j < cols; j++) {
          x = matrix.get(i, j) - mean[i];
          sum1 += x;
          sum2 += x * x;
        }
        if (unbiased) {
          variance.push((sum2 - sum1 * sum1 / cols) / (cols - 1));
        } else {
          variance.push((sum2 - sum1 * sum1 / cols) / cols);
        }
      }
      return variance;
    }
    function varianceByColumn(matrix, unbiased, mean) {
      const rows = matrix.rows;
      const cols = matrix.columns;
      const variance = [];
      for (let j = 0; j < cols; j++) {
        let sum1 = 0;
        let sum2 = 0;
        let x = 0;
        for (let i = 0; i < rows; i++) {
          x = matrix.get(i, j) - mean[j];
          sum1 += x;
          sum2 += x * x;
        }
        if (unbiased) {
          variance.push((sum2 - sum1 * sum1 / rows) / (rows - 1));
        } else {
          variance.push((sum2 - sum1 * sum1 / rows) / rows);
        }
      }
      return variance;
    }
    function varianceAll(matrix, unbiased, mean) {
      const rows = matrix.rows;
      const cols = matrix.columns;
      const size = rows * cols;
      let sum1 = 0;
      let sum2 = 0;
      let x = 0;
      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < cols; j++) {
          x = matrix.get(i, j) - mean;
          sum1 += x;
          sum2 += x * x;
        }
      }
      if (unbiased) {
        return (sum2 - sum1 * sum1 / size) / (size - 1);
      } else {
        return (sum2 - sum1 * sum1 / size) / size;
      }
    }
    function centerByRow(matrix, mean) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) - mean[i]);
        }
      }
    }
    function centerByColumn(matrix, mean) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) - mean[j]);
        }
      }
    }
    function centerAll(matrix, mean) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) - mean);
        }
      }
    }
    function getScaleByRow(matrix) {
      const scale = [];
      for (let i = 0; i < matrix.rows; i++) {
        let sum = 0;
        for (let j = 0; j < matrix.columns; j++) {
          sum += Math.pow(matrix.get(i, j), 2) / (matrix.columns - 1);
        }
        scale.push(Math.sqrt(sum));
      }
      return scale;
    }
    function scaleByRow(matrix, scale) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) / scale[i]);
        }
      }
    }
    function getScaleByColumn(matrix) {
      const scale = [];
      for (let j = 0; j < matrix.columns; j++) {
        let sum = 0;
        for (let i = 0; i < matrix.rows; i++) {
          sum += Math.pow(matrix.get(i, j), 2) / (matrix.rows - 1);
        }
        scale.push(Math.sqrt(sum));
      }
      return scale;
    }
    function scaleByColumn(matrix, scale) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) / scale[j]);
        }
      }
    }
    function getScaleAll(matrix) {
      const divider = matrix.size - 1;
      let sum = 0;
      for (let j = 0; j < matrix.columns; j++) {
        for (let i = 0; i < matrix.rows; i++) {
          sum += Math.pow(matrix.get(i, j), 2) / divider;
        }
      }
      return Math.sqrt(sum);
    }
    function scaleAll(matrix, scale) {
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          matrix.set(i, j, matrix.get(i, j) / scale);
        }
      }
    }
    class AbstractMatrix {
      static from1DArray(newRows, newColumns, newData) {
        let length = newRows * newColumns;
        if (length !== newData.length) {
          throw new RangeError('data length does not match given dimensions');
        }
        let newMatrix = new Matrix$2(newRows, newColumns);
        for (let row = 0; row < newRows; row++) {
          for (let column = 0; column < newColumns; column++) {
            newMatrix.set(row, column, newData[row * newColumns + column]);
          }
        }
        return newMatrix;
      }
      static rowVector(newData) {
        let vector = new Matrix$2(1, newData.length);
        for (let i = 0; i < newData.length; i++) {
          vector.set(0, i, newData[i]);
        }
        return vector;
      }
      static columnVector(newData) {
        let vector = new Matrix$2(newData.length, 1);
        for (let i = 0; i < newData.length; i++) {
          vector.set(i, 0, newData[i]);
        }
        return vector;
      }
      static zeros(rows, columns) {
        return new Matrix$2(rows, columns);
      }
      static ones(rows, columns) {
        return new Matrix$2(rows, columns).fill(1);
      }
      static rand(rows, columns, options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          random = Math.random
        } = options;
        let matrix = new Matrix$2(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            matrix.set(i, j, random());
          }
        }
        return matrix;
      }
      static randInt(rows, columns, options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          min = 0,
          max = 1000,
          random = Math.random
        } = options;
        if (!Number.isInteger(min)) throw new TypeError('min must be an integer');
        if (!Number.isInteger(max)) throw new TypeError('max must be an integer');
        if (min >= max) throw new RangeError('min must be smaller than max');
        let interval = max - min;
        let matrix = new Matrix$2(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            let value = min + Math.round(random() * interval);
            matrix.set(i, j, value);
          }
        }
        return matrix;
      }
      static eye(rows, columns, value) {
        if (columns === undefined) columns = rows;
        if (value === undefined) value = 1;
        let min = Math.min(rows, columns);
        let matrix = this.zeros(rows, columns);
        for (let i = 0; i < min; i++) {
          matrix.set(i, i, value);
        }
        return matrix;
      }
      static diag(data, rows, columns) {
        let l = data.length;
        if (rows === undefined) rows = l;
        if (columns === undefined) columns = rows;
        let min = Math.min(l, rows, columns);
        let matrix = this.zeros(rows, columns);
        for (let i = 0; i < min; i++) {
          matrix.set(i, i, data[i]);
        }
        return matrix;
      }
      static min(matrix1, matrix2) {
        matrix1 = this.checkMatrix(matrix1);
        matrix2 = this.checkMatrix(matrix2);
        let rows = matrix1.rows;
        let columns = matrix1.columns;
        let result = new Matrix$2(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            result.set(i, j, Math.min(matrix1.get(i, j), matrix2.get(i, j)));
          }
        }
        return result;
      }
      static max(matrix1, matrix2) {
        matrix1 = this.checkMatrix(matrix1);
        matrix2 = this.checkMatrix(matrix2);
        let rows = matrix1.rows;
        let columns = matrix1.columns;
        let result = new this(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            result.set(i, j, Math.max(matrix1.get(i, j), matrix2.get(i, j)));
          }
        }
        return result;
      }
      static checkMatrix(value) {
        return AbstractMatrix.isMatrix(value) ? value : new Matrix$2(value);
      }
      static isMatrix(value) {
        return value != null && value.klass === 'Matrix';
      }
      get size() {
        return this.rows * this.columns;
      }
      apply(callback) {
        if (typeof callback !== 'function') {
          throw new TypeError('callback must be a function');
        }
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            callback.call(this, i, j);
          }
        }
        return this;
      }
      to1DArray() {
        let array = [];
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            array.push(this.get(i, j));
          }
        }
        return array;
      }
      to2DArray() {
        let copy = [];
        for (let i = 0; i < this.rows; i++) {
          copy.push([]);
          for (let j = 0; j < this.columns; j++) {
            copy[i].push(this.get(i, j));
          }
        }
        return copy;
      }
      toJSON() {
        return this.to2DArray();
      }
      isRowVector() {
        return this.rows === 1;
      }
      isColumnVector() {
        return this.columns === 1;
      }
      isVector() {
        return this.rows === 1 || this.columns === 1;
      }
      isSquare() {
        return this.rows === this.columns;
      }
      isEmpty() {
        return this.rows === 0 || this.columns === 0;
      }
      isSymmetric() {
        if (this.isSquare()) {
          for (let i = 0; i < this.rows; i++) {
            for (let j = 0; j <= i; j++) {
              if (this.get(i, j) !== this.get(j, i)) {
                return false;
              }
            }
          }
          return true;
        }
        return false;
      }
      isEchelonForm() {
        let i = 0;
        let j = 0;
        let previousColumn = -1;
        let isEchelonForm = true;
        let checked = false;
        while (i < this.rows && isEchelonForm) {
          j = 0;
          checked = false;
          while (j < this.columns && checked === false) {
            if (this.get(i, j) === 0) {
              j++;
            } else if (this.get(i, j) === 1 && j > previousColumn) {
              checked = true;
              previousColumn = j;
            } else {
              isEchelonForm = false;
              checked = true;
            }
          }
          i++;
        }
        return isEchelonForm;
      }
      isReducedEchelonForm() {
        let i = 0;
        let j = 0;
        let previousColumn = -1;
        let isReducedEchelonForm = true;
        let checked = false;
        while (i < this.rows && isReducedEchelonForm) {
          j = 0;
          checked = false;
          while (j < this.columns && checked === false) {
            if (this.get(i, j) === 0) {
              j++;
            } else if (this.get(i, j) === 1 && j > previousColumn) {
              checked = true;
              previousColumn = j;
            } else {
              isReducedEchelonForm = false;
              checked = true;
            }
          }
          for (let k = j + 1; k < this.rows; k++) {
            if (this.get(i, k) !== 0) {
              isReducedEchelonForm = false;
            }
          }
          i++;
        }
        return isReducedEchelonForm;
      }
      echelonForm() {
        let result = this.clone();
        let h = 0;
        let k = 0;
        while (h < result.rows && k < result.columns) {
          let iMax = h;
          for (let i = h; i < result.rows; i++) {
            if (result.get(i, k) > result.get(iMax, k)) {
              iMax = i;
            }
          }
          if (result.get(iMax, k) === 0) {
            k++;
          } else {
            result.swapRows(h, iMax);
            let tmp = result.get(h, k);
            for (let j = k; j < result.columns; j++) {
              result.set(h, j, result.get(h, j) / tmp);
            }
            for (let i = h + 1; i < result.rows; i++) {
              let factor = result.get(i, k) / result.get(h, k);
              result.set(i, k, 0);
              for (let j = k + 1; j < result.columns; j++) {
                result.set(i, j, result.get(i, j) - result.get(h, j) * factor);
              }
            }
            h++;
            k++;
          }
        }
        return result;
      }
      reducedEchelonForm() {
        let result = this.echelonForm();
        let m = result.columns;
        let n = result.rows;
        let h = n - 1;
        while (h >= 0) {
          if (result.maxRow(h) === 0) {
            h--;
          } else {
            let p = 0;
            let pivot = false;
            while (p < n && pivot === false) {
              if (result.get(h, p) === 1) {
                pivot = true;
              } else {
                p++;
              }
            }
            for (let i = 0; i < h; i++) {
              let factor = result.get(i, p);
              for (let j = p; j < m; j++) {
                let tmp = result.get(i, j) - factor * result.get(h, j);
                result.set(i, j, tmp);
              }
            }
            h--;
          }
        }
        return result;
      }
      set() {
        throw new Error('set method is unimplemented');
      }
      get() {
        throw new Error('get method is unimplemented');
      }
      repeat(options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          rows = 1,
          columns = 1
        } = options;
        if (!Number.isInteger(rows) || rows <= 0) {
          throw new TypeError('rows must be a positive integer');
        }
        if (!Number.isInteger(columns) || columns <= 0) {
          throw new TypeError('columns must be a positive integer');
        }
        let matrix = new Matrix$2(this.rows * rows, this.columns * columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            matrix.setSubMatrix(this, this.rows * i, this.columns * j);
          }
        }
        return matrix;
      }
      fill(value) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, value);
          }
        }
        return this;
      }
      neg() {
        return this.mulS(-1);
      }
      getRow(index) {
        checkRowIndex(this, index);
        let row = [];
        for (let i = 0; i < this.columns; i++) {
          row.push(this.get(index, i));
        }
        return row;
      }
      getRowVector(index) {
        return Matrix$2.rowVector(this.getRow(index));
      }
      setRow(index, array) {
        checkRowIndex(this, index);
        array = checkRowVector(this, array);
        for (let i = 0; i < this.columns; i++) {
          this.set(index, i, array[i]);
        }
        return this;
      }
      swapRows(row1, row2) {
        checkRowIndex(this, row1);
        checkRowIndex(this, row2);
        for (let i = 0; i < this.columns; i++) {
          let temp = this.get(row1, i);
          this.set(row1, i, this.get(row2, i));
          this.set(row2, i, temp);
        }
        return this;
      }
      getColumn(index) {
        checkColumnIndex(this, index);
        let column = [];
        for (let i = 0; i < this.rows; i++) {
          column.push(this.get(i, index));
        }
        return column;
      }
      getColumnVector(index) {
        return Matrix$2.columnVector(this.getColumn(index));
      }
      setColumn(index, array) {
        checkColumnIndex(this, index);
        array = checkColumnVector(this, array);
        for (let i = 0; i < this.rows; i++) {
          this.set(i, index, array[i]);
        }
        return this;
      }
      swapColumns(column1, column2) {
        checkColumnIndex(this, column1);
        checkColumnIndex(this, column2);
        for (let i = 0; i < this.rows; i++) {
          let temp = this.get(i, column1);
          this.set(i, column1, this.get(i, column2));
          this.set(i, column2, temp);
        }
        return this;
      }
      addRowVector(vector) {
        vector = checkRowVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) + vector[j]);
          }
        }
        return this;
      }
      subRowVector(vector) {
        vector = checkRowVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) - vector[j]);
          }
        }
        return this;
      }
      mulRowVector(vector) {
        vector = checkRowVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) * vector[j]);
          }
        }
        return this;
      }
      divRowVector(vector) {
        vector = checkRowVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) / vector[j]);
          }
        }
        return this;
      }
      addColumnVector(vector) {
        vector = checkColumnVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) + vector[i]);
          }
        }
        return this;
      }
      subColumnVector(vector) {
        vector = checkColumnVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) - vector[i]);
          }
        }
        return this;
      }
      mulColumnVector(vector) {
        vector = checkColumnVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) * vector[i]);
          }
        }
        return this;
      }
      divColumnVector(vector) {
        vector = checkColumnVector(this, vector);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            this.set(i, j, this.get(i, j) / vector[i]);
          }
        }
        return this;
      }
      mulRow(index, value) {
        checkRowIndex(this, index);
        for (let i = 0; i < this.columns; i++) {
          this.set(index, i, this.get(index, i) * value);
        }
        return this;
      }
      mulColumn(index, value) {
        checkColumnIndex(this, index);
        for (let i = 0; i < this.rows; i++) {
          this.set(i, index, this.get(i, index) * value);
        }
        return this;
      }
      max(by) {
        if (this.isEmpty()) {
          return NaN;
        }
        switch (by) {
          case 'row':
            {
              const max = new Array(this.rows).fill(Number.NEGATIVE_INFINITY);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) > max[row]) {
                    max[row] = this.get(row, column);
                  }
                }
              }
              return max;
            }
          case 'column':
            {
              const max = new Array(this.columns).fill(Number.NEGATIVE_INFINITY);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) > max[column]) {
                    max[column] = this.get(row, column);
                  }
                }
              }
              return max;
            }
          case undefined:
            {
              let max = this.get(0, 0);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) > max) {
                    max = this.get(row, column);
                  }
                }
              }
              return max;
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      maxIndex() {
        checkNonEmpty(this);
        let v = this.get(0, 0);
        let idx = [0, 0];
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            if (this.get(i, j) > v) {
              v = this.get(i, j);
              idx[0] = i;
              idx[1] = j;
            }
          }
        }
        return idx;
      }
      min(by) {
        if (this.isEmpty()) {
          return NaN;
        }
        switch (by) {
          case 'row':
            {
              const min = new Array(this.rows).fill(Number.POSITIVE_INFINITY);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) < min[row]) {
                    min[row] = this.get(row, column);
                  }
                }
              }
              return min;
            }
          case 'column':
            {
              const min = new Array(this.columns).fill(Number.POSITIVE_INFINITY);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) < min[column]) {
                    min[column] = this.get(row, column);
                  }
                }
              }
              return min;
            }
          case undefined:
            {
              let min = this.get(0, 0);
              for (let row = 0; row < this.rows; row++) {
                for (let column = 0; column < this.columns; column++) {
                  if (this.get(row, column) < min) {
                    min = this.get(row, column);
                  }
                }
              }
              return min;
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      minIndex() {
        checkNonEmpty(this);
        let v = this.get(0, 0);
        let idx = [0, 0];
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            if (this.get(i, j) < v) {
              v = this.get(i, j);
              idx[0] = i;
              idx[1] = j;
            }
          }
        }
        return idx;
      }
      maxRow(row) {
        checkRowIndex(this, row);
        if (this.isEmpty()) {
          return NaN;
        }
        let v = this.get(row, 0);
        for (let i = 1; i < this.columns; i++) {
          if (this.get(row, i) > v) {
            v = this.get(row, i);
          }
        }
        return v;
      }
      maxRowIndex(row) {
        checkRowIndex(this, row);
        checkNonEmpty(this);
        let v = this.get(row, 0);
        let idx = [row, 0];
        for (let i = 1; i < this.columns; i++) {
          if (this.get(row, i) > v) {
            v = this.get(row, i);
            idx[1] = i;
          }
        }
        return idx;
      }
      minRow(row) {
        checkRowIndex(this, row);
        if (this.isEmpty()) {
          return NaN;
        }
        let v = this.get(row, 0);
        for (let i = 1; i < this.columns; i++) {
          if (this.get(row, i) < v) {
            v = this.get(row, i);
          }
        }
        return v;
      }
      minRowIndex(row) {
        checkRowIndex(this, row);
        checkNonEmpty(this);
        let v = this.get(row, 0);
        let idx = [row, 0];
        for (let i = 1; i < this.columns; i++) {
          if (this.get(row, i) < v) {
            v = this.get(row, i);
            idx[1] = i;
          }
        }
        return idx;
      }
      maxColumn(column) {
        checkColumnIndex(this, column);
        if (this.isEmpty()) {
          return NaN;
        }
        let v = this.get(0, column);
        for (let i = 1; i < this.rows; i++) {
          if (this.get(i, column) > v) {
            v = this.get(i, column);
          }
        }
        return v;
      }
      maxColumnIndex(column) {
        checkColumnIndex(this, column);
        checkNonEmpty(this);
        let v = this.get(0, column);
        let idx = [0, column];
        for (let i = 1; i < this.rows; i++) {
          if (this.get(i, column) > v) {
            v = this.get(i, column);
            idx[0] = i;
          }
        }
        return idx;
      }
      minColumn(column) {
        checkColumnIndex(this, column);
        if (this.isEmpty()) {
          return NaN;
        }
        let v = this.get(0, column);
        for (let i = 1; i < this.rows; i++) {
          if (this.get(i, column) < v) {
            v = this.get(i, column);
          }
        }
        return v;
      }
      minColumnIndex(column) {
        checkColumnIndex(this, column);
        checkNonEmpty(this);
        let v = this.get(0, column);
        let idx = [0, column];
        for (let i = 1; i < this.rows; i++) {
          if (this.get(i, column) < v) {
            v = this.get(i, column);
            idx[0] = i;
          }
        }
        return idx;
      }
      diag() {
        let min = Math.min(this.rows, this.columns);
        let diag = [];
        for (let i = 0; i < min; i++) {
          diag.push(this.get(i, i));
        }
        return diag;
      }
      norm(type = 'frobenius') {
        switch (type) {
          case 'max':
            return this.max();
          case 'frobenius':
            return Math.sqrt(this.dot(this));
          default:
            throw new RangeError(`unknown norm type: ${type}`);
        }
      }
      cumulativeSum() {
        let sum = 0;
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            sum += this.get(i, j);
            this.set(i, j, sum);
          }
        }
        return this;
      }
      dot(vector2) {
        if (AbstractMatrix.isMatrix(vector2)) vector2 = vector2.to1DArray();
        let vector1 = this.to1DArray();
        if (vector1.length !== vector2.length) {
          throw new RangeError('vectors do not have the same size');
        }
        let dot = 0;
        for (let i = 0; i < vector1.length; i++) {
          dot += vector1[i] * vector2[i];
        }
        return dot;
      }
      mmul(other) {
        other = Matrix$2.checkMatrix(other);
        let m = this.rows;
        let n = this.columns;
        let p = other.columns;
        let result = new Matrix$2(m, p);
        let Bcolj = new Float64Array(n);
        for (let j = 0; j < p; j++) {
          for (let k = 0; k < n; k++) {
            Bcolj[k] = other.get(k, j);
          }
          for (let i = 0; i < m; i++) {
            let s = 0;
            for (let k = 0; k < n; k++) {
              s += this.get(i, k) * Bcolj[k];
            }
            result.set(i, j, s);
          }
        }
        return result;
      }
      strassen2x2(other) {
        other = Matrix$2.checkMatrix(other);
        let result = new Matrix$2(2, 2);
        const a11 = this.get(0, 0);
        const b11 = other.get(0, 0);
        const a12 = this.get(0, 1);
        const b12 = other.get(0, 1);
        const a21 = this.get(1, 0);
        const b21 = other.get(1, 0);
        const a22 = this.get(1, 1);
        const b22 = other.get(1, 1);

        // Compute intermediate values.
        const m1 = (a11 + a22) * (b11 + b22);
        const m2 = (a21 + a22) * b11;
        const m3 = a11 * (b12 - b22);
        const m4 = a22 * (b21 - b11);
        const m5 = (a11 + a12) * b22;
        const m6 = (a21 - a11) * (b11 + b12);
        const m7 = (a12 - a22) * (b21 + b22);

        // Combine intermediate values into the output.
        const c00 = m1 + m4 - m5 + m7;
        const c01 = m3 + m5;
        const c10 = m2 + m4;
        const c11 = m1 - m2 + m3 + m6;
        result.set(0, 0, c00);
        result.set(0, 1, c01);
        result.set(1, 0, c10);
        result.set(1, 1, c11);
        return result;
      }
      strassen3x3(other) {
        other = Matrix$2.checkMatrix(other);
        let result = new Matrix$2(3, 3);
        const a00 = this.get(0, 0);
        const a01 = this.get(0, 1);
        const a02 = this.get(0, 2);
        const a10 = this.get(1, 0);
        const a11 = this.get(1, 1);
        const a12 = this.get(1, 2);
        const a20 = this.get(2, 0);
        const a21 = this.get(2, 1);
        const a22 = this.get(2, 2);
        const b00 = other.get(0, 0);
        const b01 = other.get(0, 1);
        const b02 = other.get(0, 2);
        const b10 = other.get(1, 0);
        const b11 = other.get(1, 1);
        const b12 = other.get(1, 2);
        const b20 = other.get(2, 0);
        const b21 = other.get(2, 1);
        const b22 = other.get(2, 2);
        const m1 = (a00 + a01 + a02 - a10 - a11 - a21 - a22) * b11;
        const m2 = (a00 - a10) * (-b01 + b11);
        const m3 = a11 * (-b00 + b01 + b10 - b11 - b12 - b20 + b22);
        const m4 = (-a00 + a10 + a11) * (b00 - b01 + b11);
        const m5 = (a10 + a11) * (-b00 + b01);
        const m6 = a00 * b00;
        const m7 = (-a00 + a20 + a21) * (b00 - b02 + b12);
        const m8 = (-a00 + a20) * (b02 - b12);
        const m9 = (a20 + a21) * (-b00 + b02);
        const m10 = (a00 + a01 + a02 - a11 - a12 - a20 - a21) * b12;
        const m11 = a21 * (-b00 + b02 + b10 - b11 - b12 - b20 + b21);
        const m12 = (-a02 + a21 + a22) * (b11 + b20 - b21);
        const m13 = (a02 - a22) * (b11 - b21);
        const m14 = a02 * b20;
        const m15 = (a21 + a22) * (-b20 + b21);
        const m16 = (-a02 + a11 + a12) * (b12 + b20 - b22);
        const m17 = (a02 - a12) * (b12 - b22);
        const m18 = (a11 + a12) * (-b20 + b22);
        const m19 = a01 * b10;
        const m20 = a12 * b21;
        const m21 = a10 * b02;
        const m22 = a20 * b01;
        const m23 = a22 * b22;
        const c00 = m6 + m14 + m19;
        const c01 = m1 + m4 + m5 + m6 + m12 + m14 + m15;
        const c02 = m6 + m7 + m9 + m10 + m14 + m16 + m18;
        const c10 = m2 + m3 + m4 + m6 + m14 + m16 + m17;
        const c11 = m2 + m4 + m5 + m6 + m20;
        const c12 = m14 + m16 + m17 + m18 + m21;
        const c20 = m6 + m7 + m8 + m11 + m12 + m13 + m14;
        const c21 = m12 + m13 + m14 + m15 + m22;
        const c22 = m6 + m7 + m8 + m9 + m23;
        result.set(0, 0, c00);
        result.set(0, 1, c01);
        result.set(0, 2, c02);
        result.set(1, 0, c10);
        result.set(1, 1, c11);
        result.set(1, 2, c12);
        result.set(2, 0, c20);
        result.set(2, 1, c21);
        result.set(2, 2, c22);
        return result;
      }
      mmulStrassen(y) {
        y = Matrix$2.checkMatrix(y);
        let x = this.clone();
        let r1 = x.rows;
        let c1 = x.columns;
        let r2 = y.rows;
        let c2 = y.columns;
        if (c1 !== r2) {
          // eslint-disable-next-line no-console
          console.warn(`Multiplying ${r1} x ${c1} and ${r2} x ${c2} matrix: dimensions do not match.`);
        }

        // Put a matrix into the top left of a matrix of zeros.
        // `rows` and `cols` are the dimensions of the output matrix.
        function embed(mat, rows, cols) {
          let r = mat.rows;
          let c = mat.columns;
          if (r === rows && c === cols) {
            return mat;
          } else {
            let resultat = AbstractMatrix.zeros(rows, cols);
            resultat = resultat.setSubMatrix(mat, 0, 0);
            return resultat;
          }
        }

        // Make sure both matrices are the same size.
        // This is exclusively for simplicity:
        // this algorithm can be implemented with matrices of different sizes.

        let r = Math.max(r1, r2);
        let c = Math.max(c1, c2);
        x = embed(x, r, c);
        y = embed(y, r, c);

        // Our recursive multiplication function.
        function blockMult(a, b, rows, cols) {
          // For small matrices, resort to naive multiplication.
          if (rows <= 512 || cols <= 512) {
            return a.mmul(b); // a is equivalent to this
          }

          // Apply dynamic padding.
          if (rows % 2 === 1 && cols % 2 === 1) {
            a = embed(a, rows + 1, cols + 1);
            b = embed(b, rows + 1, cols + 1);
          } else if (rows % 2 === 1) {
            a = embed(a, rows + 1, cols);
            b = embed(b, rows + 1, cols);
          } else if (cols % 2 === 1) {
            a = embed(a, rows, cols + 1);
            b = embed(b, rows, cols + 1);
          }
          let halfRows = parseInt(a.rows / 2, 10);
          let halfCols = parseInt(a.columns / 2, 10);
          // Subdivide input matrices.
          let a11 = a.subMatrix(0, halfRows - 1, 0, halfCols - 1);
          let b11 = b.subMatrix(0, halfRows - 1, 0, halfCols - 1);
          let a12 = a.subMatrix(0, halfRows - 1, halfCols, a.columns - 1);
          let b12 = b.subMatrix(0, halfRows - 1, halfCols, b.columns - 1);
          let a21 = a.subMatrix(halfRows, a.rows - 1, 0, halfCols - 1);
          let b21 = b.subMatrix(halfRows, b.rows - 1, 0, halfCols - 1);
          let a22 = a.subMatrix(halfRows, a.rows - 1, halfCols, a.columns - 1);
          let b22 = b.subMatrix(halfRows, b.rows - 1, halfCols, b.columns - 1);

          // Compute intermediate values.
          let m1 = blockMult(AbstractMatrix.add(a11, a22), AbstractMatrix.add(b11, b22), halfRows, halfCols);
          let m2 = blockMult(AbstractMatrix.add(a21, a22), b11, halfRows, halfCols);
          let m3 = blockMult(a11, AbstractMatrix.sub(b12, b22), halfRows, halfCols);
          let m4 = blockMult(a22, AbstractMatrix.sub(b21, b11), halfRows, halfCols);
          let m5 = blockMult(AbstractMatrix.add(a11, a12), b22, halfRows, halfCols);
          let m6 = blockMult(AbstractMatrix.sub(a21, a11), AbstractMatrix.add(b11, b12), halfRows, halfCols);
          let m7 = blockMult(AbstractMatrix.sub(a12, a22), AbstractMatrix.add(b21, b22), halfRows, halfCols);

          // Combine intermediate values into the output.
          let c11 = AbstractMatrix.add(m1, m4);
          c11.sub(m5);
          c11.add(m7);
          let c12 = AbstractMatrix.add(m3, m5);
          let c21 = AbstractMatrix.add(m2, m4);
          let c22 = AbstractMatrix.sub(m1, m2);
          c22.add(m3);
          c22.add(m6);

          // Crop output to the desired size (undo dynamic padding).
          let result = AbstractMatrix.zeros(2 * c11.rows, 2 * c11.columns);
          result = result.setSubMatrix(c11, 0, 0);
          result = result.setSubMatrix(c12, c11.rows, 0);
          result = result.setSubMatrix(c21, 0, c11.columns);
          result = result.setSubMatrix(c22, c11.rows, c11.columns);
          return result.subMatrix(0, rows - 1, 0, cols - 1);
        }
        return blockMult(x, y, r, c);
      }
      scaleRows(options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          min = 0,
          max = 1
        } = options;
        if (!Number.isFinite(min)) throw new TypeError('min must be a number');
        if (!Number.isFinite(max)) throw new TypeError('max must be a number');
        if (min >= max) throw new RangeError('min must be smaller than max');
        let newMatrix = new Matrix$2(this.rows, this.columns);
        for (let i = 0; i < this.rows; i++) {
          const row = this.getRow(i);
          if (row.length > 0) {
            rescale(row, {
              min,
              max,
              output: row
            });
          }
          newMatrix.setRow(i, row);
        }
        return newMatrix;
      }
      scaleColumns(options = {}) {
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          min = 0,
          max = 1
        } = options;
        if (!Number.isFinite(min)) throw new TypeError('min must be a number');
        if (!Number.isFinite(max)) throw new TypeError('max must be a number');
        if (min >= max) throw new RangeError('min must be smaller than max');
        let newMatrix = new Matrix$2(this.rows, this.columns);
        for (let i = 0; i < this.columns; i++) {
          const column = this.getColumn(i);
          if (column.length) {
            rescale(column, {
              min,
              max,
              output: column
            });
          }
          newMatrix.setColumn(i, column);
        }
        return newMatrix;
      }
      flipRows() {
        const middle = Math.ceil(this.columns / 2);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < middle; j++) {
            let first = this.get(i, j);
            let last = this.get(i, this.columns - 1 - j);
            this.set(i, j, last);
            this.set(i, this.columns - 1 - j, first);
          }
        }
        return this;
      }
      flipColumns() {
        const middle = Math.ceil(this.rows / 2);
        for (let j = 0; j < this.columns; j++) {
          for (let i = 0; i < middle; i++) {
            let first = this.get(i, j);
            let last = this.get(this.rows - 1 - i, j);
            this.set(i, j, last);
            this.set(this.rows - 1 - i, j, first);
          }
        }
        return this;
      }
      kroneckerProduct(other) {
        other = Matrix$2.checkMatrix(other);
        let m = this.rows;
        let n = this.columns;
        let p = other.rows;
        let q = other.columns;
        let result = new Matrix$2(m * p, n * q);
        for (let i = 0; i < m; i++) {
          for (let j = 0; j < n; j++) {
            for (let k = 0; k < p; k++) {
              for (let l = 0; l < q; l++) {
                result.set(p * i + k, q * j + l, this.get(i, j) * other.get(k, l));
              }
            }
          }
        }
        return result;
      }
      kroneckerSum(other) {
        other = Matrix$2.checkMatrix(other);
        if (!this.isSquare() || !other.isSquare()) {
          throw new Error('Kronecker Sum needs two Square Matrices');
        }
        let m = this.rows;
        let n = other.rows;
        let AxI = this.kroneckerProduct(Matrix$2.eye(n, n));
        let IxB = Matrix$2.eye(m, m).kroneckerProduct(other);
        return AxI.add(IxB);
      }
      transpose() {
        let result = new Matrix$2(this.columns, this.rows);
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            result.set(j, i, this.get(i, j));
          }
        }
        return result;
      }
      sortRows(compareFunction = compareNumbers) {
        for (let i = 0; i < this.rows; i++) {
          this.setRow(i, this.getRow(i).sort(compareFunction));
        }
        return this;
      }
      sortColumns(compareFunction = compareNumbers) {
        for (let i = 0; i < this.columns; i++) {
          this.setColumn(i, this.getColumn(i).sort(compareFunction));
        }
        return this;
      }
      subMatrix(startRow, endRow, startColumn, endColumn) {
        checkRange(this, startRow, endRow, startColumn, endColumn);
        let newMatrix = new Matrix$2(endRow - startRow + 1, endColumn - startColumn + 1);
        for (let i = startRow; i <= endRow; i++) {
          for (let j = startColumn; j <= endColumn; j++) {
            newMatrix.set(i - startRow, j - startColumn, this.get(i, j));
          }
        }
        return newMatrix;
      }
      subMatrixRow(indices, startColumn, endColumn) {
        if (startColumn === undefined) startColumn = 0;
        if (endColumn === undefined) endColumn = this.columns - 1;
        if (startColumn > endColumn || startColumn < 0 || startColumn >= this.columns || endColumn < 0 || endColumn >= this.columns) {
          throw new RangeError('Argument out of range');
        }
        let newMatrix = new Matrix$2(indices.length, endColumn - startColumn + 1);
        for (let i = 0; i < indices.length; i++) {
          for (let j = startColumn; j <= endColumn; j++) {
            if (indices[i] < 0 || indices[i] >= this.rows) {
              throw new RangeError(`Row index out of range: ${indices[i]}`);
            }
            newMatrix.set(i, j - startColumn, this.get(indices[i], j));
          }
        }
        return newMatrix;
      }
      subMatrixColumn(indices, startRow, endRow) {
        if (startRow === undefined) startRow = 0;
        if (endRow === undefined) endRow = this.rows - 1;
        if (startRow > endRow || startRow < 0 || startRow >= this.rows || endRow < 0 || endRow >= this.rows) {
          throw new RangeError('Argument out of range');
        }
        let newMatrix = new Matrix$2(endRow - startRow + 1, indices.length);
        for (let i = 0; i < indices.length; i++) {
          for (let j = startRow; j <= endRow; j++) {
            if (indices[i] < 0 || indices[i] >= this.columns) {
              throw new RangeError(`Column index out of range: ${indices[i]}`);
            }
            newMatrix.set(j - startRow, i, this.get(j, indices[i]));
          }
        }
        return newMatrix;
      }
      setSubMatrix(matrix, startRow, startColumn) {
        matrix = Matrix$2.checkMatrix(matrix);
        if (matrix.isEmpty()) {
          return this;
        }
        let endRow = startRow + matrix.rows - 1;
        let endColumn = startColumn + matrix.columns - 1;
        checkRange(this, startRow, endRow, startColumn, endColumn);
        for (let i = 0; i < matrix.rows; i++) {
          for (let j = 0; j < matrix.columns; j++) {
            this.set(startRow + i, startColumn + j, matrix.get(i, j));
          }
        }
        return this;
      }
      selection(rowIndices, columnIndices) {
        checkRowIndices(this, rowIndices);
        checkColumnIndices(this, columnIndices);
        let newMatrix = new Matrix$2(rowIndices.length, columnIndices.length);
        for (let i = 0; i < rowIndices.length; i++) {
          let rowIndex = rowIndices[i];
          for (let j = 0; j < columnIndices.length; j++) {
            let columnIndex = columnIndices[j];
            newMatrix.set(i, j, this.get(rowIndex, columnIndex));
          }
        }
        return newMatrix;
      }
      trace() {
        let min = Math.min(this.rows, this.columns);
        let trace = 0;
        for (let i = 0; i < min; i++) {
          trace += this.get(i, i);
        }
        return trace;
      }
      clone() {
        let newMatrix = new Matrix$2(this.rows, this.columns);
        for (let row = 0; row < this.rows; row++) {
          for (let column = 0; column < this.columns; column++) {
            newMatrix.set(row, column, this.get(row, column));
          }
        }
        return newMatrix;
      }
      sum(by) {
        switch (by) {
          case 'row':
            return sumByRow(this);
          case 'column':
            return sumByColumn(this);
          case undefined:
            return sumAll(this);
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      product(by) {
        switch (by) {
          case 'row':
            return productByRow(this);
          case 'column':
            return productByColumn(this);
          case undefined:
            return productAll(this);
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      mean(by) {
        const sum = this.sum(by);
        switch (by) {
          case 'row':
            {
              for (let i = 0; i < this.rows; i++) {
                sum[i] /= this.columns;
              }
              return sum;
            }
          case 'column':
            {
              for (let i = 0; i < this.columns; i++) {
                sum[i] /= this.rows;
              }
              return sum;
            }
          case undefined:
            return sum / this.size;
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      variance(by, options = {}) {
        if (typeof by === 'object') {
          options = by;
          by = undefined;
        }
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          unbiased = true,
          mean = this.mean(by)
        } = options;
        if (typeof unbiased !== 'boolean') {
          throw new TypeError('unbiased must be a boolean');
        }
        switch (by) {
          case 'row':
            {
              if (!isAnyArray.isAnyArray(mean)) {
                throw new TypeError('mean must be an array');
              }
              return varianceByRow(this, unbiased, mean);
            }
          case 'column':
            {
              if (!isAnyArray.isAnyArray(mean)) {
                throw new TypeError('mean must be an array');
              }
              return varianceByColumn(this, unbiased, mean);
            }
          case undefined:
            {
              if (typeof mean !== 'number') {
                throw new TypeError('mean must be a number');
              }
              return varianceAll(this, unbiased, mean);
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      standardDeviation(by, options) {
        if (typeof by === 'object') {
          options = by;
          by = undefined;
        }
        const variance = this.variance(by, options);
        if (by === undefined) {
          return Math.sqrt(variance);
        } else {
          for (let i = 0; i < variance.length; i++) {
            variance[i] = Math.sqrt(variance[i]);
          }
          return variance;
        }
      }
      center(by, options = {}) {
        if (typeof by === 'object') {
          options = by;
          by = undefined;
        }
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        const {
          center = this.mean(by)
        } = options;
        switch (by) {
          case 'row':
            {
              if (!isAnyArray.isAnyArray(center)) {
                throw new TypeError('center must be an array');
              }
              centerByRow(this, center);
              return this;
            }
          case 'column':
            {
              if (!isAnyArray.isAnyArray(center)) {
                throw new TypeError('center must be an array');
              }
              centerByColumn(this, center);
              return this;
            }
          case undefined:
            {
              if (typeof center !== 'number') {
                throw new TypeError('center must be a number');
              }
              centerAll(this, center);
              return this;
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      scale(by, options = {}) {
        if (typeof by === 'object') {
          options = by;
          by = undefined;
        }
        if (typeof options !== 'object') {
          throw new TypeError('options must be an object');
        }
        let scale = options.scale;
        switch (by) {
          case 'row':
            {
              if (scale === undefined) {
                scale = getScaleByRow(this);
              } else if (!isAnyArray.isAnyArray(scale)) {
                throw new TypeError('scale must be an array');
              }
              scaleByRow(this, scale);
              return this;
            }
          case 'column':
            {
              if (scale === undefined) {
                scale = getScaleByColumn(this);
              } else if (!isAnyArray.isAnyArray(scale)) {
                throw new TypeError('scale must be an array');
              }
              scaleByColumn(this, scale);
              return this;
            }
          case undefined:
            {
              if (scale === undefined) {
                scale = getScaleAll(this);
              } else if (typeof scale !== 'number') {
                throw new TypeError('scale must be a number');
              }
              scaleAll(this, scale);
              return this;
            }
          default:
            throw new Error(`invalid option: ${by}`);
        }
      }
      toString(options) {
        return inspectMatrixWithOptions(this, options);
      }
    }
    AbstractMatrix.prototype.klass = 'Matrix';
    if (typeof Symbol !== 'undefined') {
      AbstractMatrix.prototype[Symbol.for('nodejs.util.inspect.custom')] = inspectMatrix;
    }
    function compareNumbers(a, b) {
      return a - b;
    }
    function isArrayOfNumbers(array) {
      return array.every(element => {
        return typeof element === 'number';
      });
    }

    // Synonyms
    AbstractMatrix.random = AbstractMatrix.rand;
    AbstractMatrix.randomInt = AbstractMatrix.randInt;
    AbstractMatrix.diagonal = AbstractMatrix.diag;
    AbstractMatrix.prototype.diagonal = AbstractMatrix.prototype.diag;
    AbstractMatrix.identity = AbstractMatrix.eye;
    AbstractMatrix.prototype.negate = AbstractMatrix.prototype.neg;
    AbstractMatrix.prototype.tensorProduct = AbstractMatrix.prototype.kroneckerProduct;
    class Matrix$2 extends AbstractMatrix {
      constructor(nRows, nColumns) {
        super();
        if (Matrix$2.isMatrix(nRows)) {
          // eslint-disable-next-line no-constructor-return
          return nRows.clone();
        } else if (Number.isInteger(nRows) && nRows >= 0) {
          // Create an empty matrix
          this.data = [];
          if (Number.isInteger(nColumns) && nColumns >= 0) {
            for (let i = 0; i < nRows; i++) {
              this.data.push(new Float64Array(nColumns));
            }
          } else {
            throw new TypeError('nColumns must be a positive integer');
          }
        } else if (isAnyArray.isAnyArray(nRows)) {
          // Copy the values from the 2D array
          const arrayData = nRows;
          nRows = arrayData.length;
          nColumns = nRows ? arrayData[0].length : 0;
          if (typeof nColumns !== 'number') {
            throw new TypeError('Data must be a 2D array with at least one element');
          }
          this.data = [];
          for (let i = 0; i < nRows; i++) {
            if (arrayData[i].length !== nColumns) {
              throw new RangeError('Inconsistent array dimensions');
            }
            if (!isArrayOfNumbers(arrayData[i])) {
              throw new TypeError('Input data contains non-numeric values');
            }
            this.data.push(Float64Array.from(arrayData[i]));
          }
        } else {
          throw new TypeError('First argument must be a positive number or an array');
        }
        this.rows = nRows;
        this.columns = nColumns;
      }
      set(rowIndex, columnIndex, value) {
        this.data[rowIndex][columnIndex] = value;
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.data[rowIndex][columnIndex];
      }
      removeRow(index) {
        checkRowIndex(this, index);
        this.data.splice(index, 1);
        this.rows -= 1;
        return this;
      }
      addRow(index, array) {
        if (array === undefined) {
          array = index;
          index = this.rows;
        }
        checkRowIndex(this, index, true);
        array = Float64Array.from(checkRowVector(this, array));
        this.data.splice(index, 0, array);
        this.rows += 1;
        return this;
      }
      removeColumn(index) {
        checkColumnIndex(this, index);
        for (let i = 0; i < this.rows; i++) {
          const newRow = new Float64Array(this.columns - 1);
          for (let j = 0; j < index; j++) {
            newRow[j] = this.data[i][j];
          }
          for (let j = index + 1; j < this.columns; j++) {
            newRow[j - 1] = this.data[i][j];
          }
          this.data[i] = newRow;
        }
        this.columns -= 1;
        return this;
      }
      addColumn(index, array) {
        if (typeof array === 'undefined') {
          array = index;
          index = this.columns;
        }
        checkColumnIndex(this, index, true);
        array = checkColumnVector(this, array);
        for (let i = 0; i < this.rows; i++) {
          const newRow = new Float64Array(this.columns + 1);
          let j = 0;
          for (; j < index; j++) {
            newRow[j] = this.data[i][j];
          }
          newRow[j++] = array[i];
          for (; j < this.columns + 1; j++) {
            newRow[j] = this.data[i][j - 1];
          }
          this.data[i] = newRow;
        }
        this.columns += 1;
        return this;
      }
    }
    installMathOperations(AbstractMatrix, Matrix$2);
    class BaseView extends AbstractMatrix {
      constructor(matrix, rows, columns) {
        super();
        this.matrix = matrix;
        this.rows = rows;
        this.columns = columns;
      }
    }
    class MatrixColumnView extends BaseView {
      constructor(matrix, column) {
        checkColumnIndex(matrix, column);
        super(matrix, matrix.rows, 1);
        this.column = column;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(rowIndex, this.column, value);
        return this;
      }
      get(rowIndex) {
        return this.matrix.get(rowIndex, this.column);
      }
    }
    class MatrixColumnSelectionView extends BaseView {
      constructor(matrix, columnIndices) {
        checkColumnIndices(matrix, columnIndices);
        super(matrix, matrix.rows, columnIndices.length);
        this.columnIndices = columnIndices;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(rowIndex, this.columnIndices[columnIndex], value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(rowIndex, this.columnIndices[columnIndex]);
      }
    }
    class MatrixFlipColumnView extends BaseView {
      constructor(matrix) {
        super(matrix, matrix.rows, matrix.columns);
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(rowIndex, this.columns - columnIndex - 1, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(rowIndex, this.columns - columnIndex - 1);
      }
    }
    class MatrixFlipRowView extends BaseView {
      constructor(matrix) {
        super(matrix, matrix.rows, matrix.columns);
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.rows - rowIndex - 1, columnIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.rows - rowIndex - 1, columnIndex);
      }
    }
    class MatrixRowView extends BaseView {
      constructor(matrix, row) {
        checkRowIndex(matrix, row);
        super(matrix, 1, matrix.columns);
        this.row = row;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.row, columnIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.row, columnIndex);
      }
    }
    class MatrixRowSelectionView extends BaseView {
      constructor(matrix, rowIndices) {
        checkRowIndices(matrix, rowIndices);
        super(matrix, rowIndices.length, matrix.columns);
        this.rowIndices = rowIndices;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.rowIndices[rowIndex], columnIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.rowIndices[rowIndex], columnIndex);
      }
    }
    class MatrixSelectionView extends BaseView {
      constructor(matrix, rowIndices, columnIndices) {
        checkRowIndices(matrix, rowIndices);
        checkColumnIndices(matrix, columnIndices);
        super(matrix, rowIndices.length, columnIndices.length);
        this.rowIndices = rowIndices;
        this.columnIndices = columnIndices;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.rowIndices[rowIndex], this.columnIndices[columnIndex], value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.rowIndices[rowIndex], this.columnIndices[columnIndex]);
      }
    }
    class MatrixSubView extends BaseView {
      constructor(matrix, startRow, endRow, startColumn, endColumn) {
        checkRange(matrix, startRow, endRow, startColumn, endColumn);
        super(matrix, endRow - startRow + 1, endColumn - startColumn + 1);
        this.startRow = startRow;
        this.startColumn = startColumn;
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(this.startRow + rowIndex, this.startColumn + columnIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(this.startRow + rowIndex, this.startColumn + columnIndex);
      }
    }
    class MatrixTransposeView extends BaseView {
      constructor(matrix) {
        super(matrix, matrix.columns, matrix.rows);
      }
      set(rowIndex, columnIndex, value) {
        this.matrix.set(columnIndex, rowIndex, value);
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.matrix.get(columnIndex, rowIndex);
      }
    }
    class WrapperMatrix1D extends AbstractMatrix {
      constructor(data, options = {}) {
        const {
          rows = 1
        } = options;
        if (data.length % rows !== 0) {
          throw new Error('the data length is not divisible by the number of rows');
        }
        super();
        this.rows = rows;
        this.columns = data.length / rows;
        this.data = data;
      }
      set(rowIndex, columnIndex, value) {
        let index = this._calculateIndex(rowIndex, columnIndex);
        this.data[index] = value;
        return this;
      }
      get(rowIndex, columnIndex) {
        let index = this._calculateIndex(rowIndex, columnIndex);
        return this.data[index];
      }
      _calculateIndex(row, column) {
        return row * this.columns + column;
      }
    }
    class WrapperMatrix2D extends AbstractMatrix {
      constructor(data) {
        super();
        this.data = data;
        this.rows = data.length;
        this.columns = data[0].length;
      }
      set(rowIndex, columnIndex, value) {
        this.data[rowIndex][columnIndex] = value;
        return this;
      }
      get(rowIndex, columnIndex) {
        return this.data[rowIndex][columnIndex];
      }
    }
    function wrap(array, options) {
      if (isAnyArray.isAnyArray(array)) {
        if (array[0] && isAnyArray.isAnyArray(array[0])) {
          return new WrapperMatrix2D(array);
        } else {
          return new WrapperMatrix1D(array, options);
        }
      } else {
        throw new Error('the argument is not an array');
      }
    }
    class LuDecomposition$1 {
      constructor(matrix) {
        matrix = WrapperMatrix2D.checkMatrix(matrix);
        let lu = matrix.clone();
        let rows = lu.rows;
        let columns = lu.columns;
        let pivotVector = new Float64Array(rows);
        let pivotSign = 1;
        let i, j, k, p, s, t, v;
        let LUcolj, kmax;
        for (i = 0; i < rows; i++) {
          pivotVector[i] = i;
        }
        LUcolj = new Float64Array(rows);
        for (j = 0; j < columns; j++) {
          for (i = 0; i < rows; i++) {
            LUcolj[i] = lu.get(i, j);
          }
          for (i = 0; i < rows; i++) {
            kmax = Math.min(i, j);
            s = 0;
            for (k = 0; k < kmax; k++) {
              s += lu.get(i, k) * LUcolj[k];
            }
            LUcolj[i] -= s;
            lu.set(i, j, LUcolj[i]);
          }
          p = j;
          for (i = j + 1; i < rows; i++) {
            if (Math.abs(LUcolj[i]) > Math.abs(LUcolj[p])) {
              p = i;
            }
          }
          if (p !== j) {
            for (k = 0; k < columns; k++) {
              t = lu.get(p, k);
              lu.set(p, k, lu.get(j, k));
              lu.set(j, k, t);
            }
            v = pivotVector[p];
            pivotVector[p] = pivotVector[j];
            pivotVector[j] = v;
            pivotSign = -pivotSign;
          }
          if (j < rows && lu.get(j, j) !== 0) {
            for (i = j + 1; i < rows; i++) {
              lu.set(i, j, lu.get(i, j) / lu.get(j, j));
            }
          }
        }
        this.LU = lu;
        this.pivotVector = pivotVector;
        this.pivotSign = pivotSign;
      }
      isSingular() {
        let data = this.LU;
        let col = data.columns;
        for (let j = 0; j < col; j++) {
          if (data.get(j, j) === 0) {
            return true;
          }
        }
        return false;
      }
      solve(value) {
        value = Matrix$2.checkMatrix(value);
        let lu = this.LU;
        let rows = lu.rows;
        if (rows !== value.rows) {
          throw new Error('Invalid matrix dimensions');
        }
        if (this.isSingular()) {
          throw new Error('LU matrix is singular');
        }
        let count = value.columns;
        let X = value.subMatrixRow(this.pivotVector, 0, count - 1);
        let columns = lu.columns;
        let i, j, k;
        for (k = 0; k < columns; k++) {
          for (i = k + 1; i < columns; i++) {
            for (j = 0; j < count; j++) {
              X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
            }
          }
        }
        for (k = columns - 1; k >= 0; k--) {
          for (j = 0; j < count; j++) {
            X.set(k, j, X.get(k, j) / lu.get(k, k));
          }
          for (i = 0; i < k; i++) {
            for (j = 0; j < count; j++) {
              X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
            }
          }
        }
        return X;
      }
      get determinant() {
        let data = this.LU;
        if (!data.isSquare()) {
          throw new Error('Matrix must be square');
        }
        let determinant = this.pivotSign;
        let col = data.columns;
        for (let j = 0; j < col; j++) {
          determinant *= data.get(j, j);
        }
        return determinant;
      }
      get lowerTriangularMatrix() {
        let data = this.LU;
        let rows = data.rows;
        let columns = data.columns;
        let X = new Matrix$2(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            if (i > j) {
              X.set(i, j, data.get(i, j));
            } else if (i === j) {
              X.set(i, j, 1);
            } else {
              X.set(i, j, 0);
            }
          }
        }
        return X;
      }
      get upperTriangularMatrix() {
        let data = this.LU;
        let rows = data.rows;
        let columns = data.columns;
        let X = new Matrix$2(rows, columns);
        for (let i = 0; i < rows; i++) {
          for (let j = 0; j < columns; j++) {
            if (i <= j) {
              X.set(i, j, data.get(i, j));
            } else {
              X.set(i, j, 0);
            }
          }
        }
        return X;
      }
      get pivotPermutationVector() {
        return Array.from(this.pivotVector);
      }
    }
    function hypotenuse(a, b) {
      let r = 0;
      if (Math.abs(a) > Math.abs(b)) {
        r = b / a;
        return Math.abs(a) * Math.sqrt(1 + r * r);
      }
      if (b !== 0) {
        r = a / b;
        return Math.abs(b) * Math.sqrt(1 + r * r);
      }
      return 0;
    }
    class QrDecomposition {
      constructor(value) {
        value = WrapperMatrix2D.checkMatrix(value);
        let qr = value.clone();
        let m = value.rows;
        let n = value.columns;
        let rdiag = new Float64Array(n);
        let i, j, k, s;
        for (k = 0; k < n; k++) {
          let nrm = 0;
          for (i = k; i < m; i++) {
            nrm = hypotenuse(nrm, qr.get(i, k));
          }
          if (nrm !== 0) {
            if (qr.get(k, k) < 0) {
              nrm = -nrm;
            }
            for (i = k; i < m; i++) {
              qr.set(i, k, qr.get(i, k) / nrm);
            }
            qr.set(k, k, qr.get(k, k) + 1);
            for (j = k + 1; j < n; j++) {
              s = 0;
              for (i = k; i < m; i++) {
                s += qr.get(i, k) * qr.get(i, j);
              }
              s = -s / qr.get(k, k);
              for (i = k; i < m; i++) {
                qr.set(i, j, qr.get(i, j) + s * qr.get(i, k));
              }
            }
          }
          rdiag[k] = -nrm;
        }
        this.QR = qr;
        this.Rdiag = rdiag;
      }
      solve(value) {
        value = Matrix$2.checkMatrix(value);
        let qr = this.QR;
        let m = qr.rows;
        if (value.rows !== m) {
          throw new Error('Matrix row dimensions must agree');
        }
        if (!this.isFullRank()) {
          throw new Error('Matrix is rank deficient');
        }
        let count = value.columns;
        let X = value.clone();
        let n = qr.columns;
        let i, j, k, s;
        for (k = 0; k < n; k++) {
          for (j = 0; j < count; j++) {
            s = 0;
            for (i = k; i < m; i++) {
              s += qr.get(i, k) * X.get(i, j);
            }
            s = -s / qr.get(k, k);
            for (i = k; i < m; i++) {
              X.set(i, j, X.get(i, j) + s * qr.get(i, k));
            }
          }
        }
        for (k = n - 1; k >= 0; k--) {
          for (j = 0; j < count; j++) {
            X.set(k, j, X.get(k, j) / this.Rdiag[k]);
          }
          for (i = 0; i < k; i++) {
            for (j = 0; j < count; j++) {
              X.set(i, j, X.get(i, j) - X.get(k, j) * qr.get(i, k));
            }
          }
        }
        return X.subMatrix(0, n - 1, 0, count - 1);
      }
      isFullRank() {
        let columns = this.QR.columns;
        for (let i = 0; i < columns; i++) {
          if (this.Rdiag[i] === 0) {
            return false;
          }
        }
        return true;
      }
      get upperTriangularMatrix() {
        let qr = this.QR;
        let n = qr.columns;
        let X = new Matrix$2(n, n);
        let i, j;
        for (i = 0; i < n; i++) {
          for (j = 0; j < n; j++) {
            if (i < j) {
              X.set(i, j, qr.get(i, j));
            } else if (i === j) {
              X.set(i, j, this.Rdiag[i]);
            } else {
              X.set(i, j, 0);
            }
          }
        }
        return X;
      }
      get orthogonalMatrix() {
        let qr = this.QR;
        let rows = qr.rows;
        let columns = qr.columns;
        let X = new Matrix$2(rows, columns);
        let i, j, k, s;
        for (k = columns - 1; k >= 0; k--) {
          for (i = 0; i < rows; i++) {
            X.set(i, k, 0);
          }
          X.set(k, k, 1);
          for (j = k; j < columns; j++) {
            if (qr.get(k, k) !== 0) {
              s = 0;
              for (i = k; i < rows; i++) {
                s += qr.get(i, k) * X.get(i, j);
              }
              s = -s / qr.get(k, k);
              for (i = k; i < rows; i++) {
                X.set(i, j, X.get(i, j) + s * qr.get(i, k));
              }
            }
          }
        }
        return X;
      }
    }
    class SingularValueDecomposition {
      constructor(value, options = {}) {
        value = WrapperMatrix2D.checkMatrix(value);
        if (value.isEmpty()) {
          throw new Error('Matrix must be non-empty');
        }
        let m = value.rows;
        let n = value.columns;
        const {
          computeLeftSingularVectors = true,
          computeRightSingularVectors = true,
          autoTranspose = false
        } = options;
        let wantu = Boolean(computeLeftSingularVectors);
        let wantv = Boolean(computeRightSingularVectors);
        let swapped = false;
        let a;
        if (m < n) {
          if (!autoTranspose) {
            a = value.clone();
            // eslint-disable-next-line no-console
            console.warn('Computing SVD on a matrix with more columns than rows. Consider enabling autoTranspose');
          } else {
            a = value.transpose();
            m = a.rows;
            n = a.columns;
            swapped = true;
            let aux = wantu;
            wantu = wantv;
            wantv = aux;
          }
        } else {
          a = value.clone();
        }
        let nu = Math.min(m, n);
        let ni = Math.min(m + 1, n);
        let s = new Float64Array(ni);
        let U = new Matrix$2(m, nu);
        let V = new Matrix$2(n, n);
        let e = new Float64Array(n);
        let work = new Float64Array(m);
        let si = new Float64Array(ni);
        for (let i = 0; i < ni; i++) si[i] = i;
        let nct = Math.min(m - 1, n);
        let nrt = Math.max(0, Math.min(n - 2, m));
        let mrc = Math.max(nct, nrt);
        for (let k = 0; k < mrc; k++) {
          if (k < nct) {
            s[k] = 0;
            for (let i = k; i < m; i++) {
              s[k] = hypotenuse(s[k], a.get(i, k));
            }
            if (s[k] !== 0) {
              if (a.get(k, k) < 0) {
                s[k] = -s[k];
              }
              for (let i = k; i < m; i++) {
                a.set(i, k, a.get(i, k) / s[k]);
              }
              a.set(k, k, a.get(k, k) + 1);
            }
            s[k] = -s[k];
          }
          for (let j = k + 1; j < n; j++) {
            if (k < nct && s[k] !== 0) {
              let t = 0;
              for (let i = k; i < m; i++) {
                t += a.get(i, k) * a.get(i, j);
              }
              t = -t / a.get(k, k);
              for (let i = k; i < m; i++) {
                a.set(i, j, a.get(i, j) + t * a.get(i, k));
              }
            }
            e[j] = a.get(k, j);
          }
          if (wantu && k < nct) {
            for (let i = k; i < m; i++) {
              U.set(i, k, a.get(i, k));
            }
          }
          if (k < nrt) {
            e[k] = 0;
            for (let i = k + 1; i < n; i++) {
              e[k] = hypotenuse(e[k], e[i]);
            }
            if (e[k] !== 0) {
              if (e[k + 1] < 0) {
                e[k] = 0 - e[k];
              }
              for (let i = k + 1; i < n; i++) {
                e[i] /= e[k];
              }
              e[k + 1] += 1;
            }
            e[k] = -e[k];
            if (k + 1 < m && e[k] !== 0) {
              for (let i = k + 1; i < m; i++) {
                work[i] = 0;
              }
              for (let i = k + 1; i < m; i++) {
                for (let j = k + 1; j < n; j++) {
                  work[i] += e[j] * a.get(i, j);
                }
              }
              for (let j = k + 1; j < n; j++) {
                let t = -e[j] / e[k + 1];
                for (let i = k + 1; i < m; i++) {
                  a.set(i, j, a.get(i, j) + t * work[i]);
                }
              }
            }
            if (wantv) {
              for (let i = k + 1; i < n; i++) {
                V.set(i, k, e[i]);
              }
            }
          }
        }
        let p = Math.min(n, m + 1);
        if (nct < n) {
          s[nct] = a.get(nct, nct);
        }
        if (m < p) {
          s[p - 1] = 0;
        }
        if (nrt + 1 < p) {
          e[nrt] = a.get(nrt, p - 1);
        }
        e[p - 1] = 0;
        if (wantu) {
          for (let j = nct; j < nu; j++) {
            for (let i = 0; i < m; i++) {
              U.set(i, j, 0);
            }
            U.set(j, j, 1);
          }
          for (let k = nct - 1; k >= 0; k--) {
            if (s[k] !== 0) {
              for (let j = k + 1; j < nu; j++) {
                let t = 0;
                for (let i = k; i < m; i++) {
                  t += U.get(i, k) * U.get(i, j);
                }
                t = -t / U.get(k, k);
                for (let i = k; i < m; i++) {
                  U.set(i, j, U.get(i, j) + t * U.get(i, k));
                }
              }
              for (let i = k; i < m; i++) {
                U.set(i, k, -U.get(i, k));
              }
              U.set(k, k, 1 + U.get(k, k));
              for (let i = 0; i < k - 1; i++) {
                U.set(i, k, 0);
              }
            } else {
              for (let i = 0; i < m; i++) {
                U.set(i, k, 0);
              }
              U.set(k, k, 1);
            }
          }
        }
        if (wantv) {
          for (let k = n - 1; k >= 0; k--) {
            if (k < nrt && e[k] !== 0) {
              for (let j = k + 1; j < n; j++) {
                let t = 0;
                for (let i = k + 1; i < n; i++) {
                  t += V.get(i, k) * V.get(i, j);
                }
                t = -t / V.get(k + 1, k);
                for (let i = k + 1; i < n; i++) {
                  V.set(i, j, V.get(i, j) + t * V.get(i, k));
                }
              }
            }
            for (let i = 0; i < n; i++) {
              V.set(i, k, 0);
            }
            V.set(k, k, 1);
          }
        }
        let pp = p - 1;
        let eps = Number.EPSILON;
        while (p > 0) {
          let k, kase;
          for (k = p - 2; k >= -1; k--) {
            if (k === -1) {
              break;
            }
            const alpha = Number.MIN_VALUE + eps * Math.abs(s[k] + Math.abs(s[k + 1]));
            if (Math.abs(e[k]) <= alpha || Number.isNaN(e[k])) {
              e[k] = 0;
              break;
            }
          }
          if (k === p - 2) {
            kase = 4;
          } else {
            let ks;
            for (ks = p - 1; ks >= k; ks--) {
              if (ks === k) {
                break;
              }
              let t = (ks !== p ? Math.abs(e[ks]) : 0) + (ks !== k + 1 ? Math.abs(e[ks - 1]) : 0);
              if (Math.abs(s[ks]) <= eps * t) {
                s[ks] = 0;
                break;
              }
            }
            if (ks === k) {
              kase = 3;
            } else if (ks === p - 1) {
              kase = 1;
            } else {
              kase = 2;
              k = ks;
            }
          }
          k++;
          switch (kase) {
            case 1:
              {
                let f = e[p - 2];
                e[p - 2] = 0;
                for (let j = p - 2; j >= k; j--) {
                  let t = hypotenuse(s[j], f);
                  let cs = s[j] / t;
                  let sn = f / t;
                  s[j] = t;
                  if (j !== k) {
                    f = -sn * e[j - 1];
                    e[j - 1] = cs * e[j - 1];
                  }
                  if (wantv) {
                    for (let i = 0; i < n; i++) {
                      t = cs * V.get(i, j) + sn * V.get(i, p - 1);
                      V.set(i, p - 1, -sn * V.get(i, j) + cs * V.get(i, p - 1));
                      V.set(i, j, t);
                    }
                  }
                }
                break;
              }
            case 2:
              {
                let f = e[k - 1];
                e[k - 1] = 0;
                for (let j = k; j < p; j++) {
                  let t = hypotenuse(s[j], f);
                  let cs = s[j] / t;
                  let sn = f / t;
                  s[j] = t;
                  f = -sn * e[j];
                  e[j] = cs * e[j];
                  if (wantu) {
                    for (let i = 0; i < m; i++) {
                      t = cs * U.get(i, j) + sn * U.get(i, k - 1);
                      U.set(i, k - 1, -sn * U.get(i, j) + cs * U.get(i, k - 1));
                      U.set(i, j, t);
                    }
                  }
                }
                break;
              }
            case 3:
              {
                const scale = Math.max(Math.abs(s[p - 1]), Math.abs(s[p - 2]), Math.abs(e[p - 2]), Math.abs(s[k]), Math.abs(e[k]));
                const sp = s[p - 1] / scale;
                const spm1 = s[p - 2] / scale;
                const epm1 = e[p - 2] / scale;
                const sk = s[k] / scale;
                const ek = e[k] / scale;
                const b = ((spm1 + sp) * (spm1 - sp) + epm1 * epm1) / 2;
                const c = sp * epm1 * (sp * epm1);
                let shift = 0;
                if (b !== 0 || c !== 0) {
                  if (b < 0) {
                    shift = 0 - Math.sqrt(b * b + c);
                  } else {
                    shift = Math.sqrt(b * b + c);
                  }
                  shift = c / (b + shift);
                }
                let f = (sk + sp) * (sk - sp) + shift;
                let g = sk * ek;
                for (let j = k; j < p - 1; j++) {
                  let t = hypotenuse(f, g);
                  if (t === 0) t = Number.MIN_VALUE;
                  let cs = f / t;
                  let sn = g / t;
                  if (j !== k) {
                    e[j - 1] = t;
                  }
                  f = cs * s[j] + sn * e[j];
                  e[j] = cs * e[j] - sn * s[j];
                  g = sn * s[j + 1];
                  s[j + 1] = cs * s[j + 1];
                  if (wantv) {
                    for (let i = 0; i < n; i++) {
                      t = cs * V.get(i, j) + sn * V.get(i, j + 1);
                      V.set(i, j + 1, -sn * V.get(i, j) + cs * V.get(i, j + 1));
                      V.set(i, j, t);
                    }
                  }
                  t = hypotenuse(f, g);
                  if (t === 0) t = Number.MIN_VALUE;
                  cs = f / t;
                  sn = g / t;
                  s[j] = t;
                  f = cs * e[j] + sn * s[j + 1];
                  s[j + 1] = -sn * e[j] + cs * s[j + 1];
                  g = sn * e[j + 1];
                  e[j + 1] = cs * e[j + 1];
                  if (wantu && j < m - 1) {
                    for (let i = 0; i < m; i++) {
                      t = cs * U.get(i, j) + sn * U.get(i, j + 1);
                      U.set(i, j + 1, -sn * U.get(i, j) + cs * U.get(i, j + 1));
                      U.set(i, j, t);
                    }
                  }
                }
                e[p - 2] = f;
                break;
              }
            case 4:
              {
                if (s[k] <= 0) {
                  s[k] = s[k] < 0 ? -s[k] : 0;
                  if (wantv) {
                    for (let i = 0; i <= pp; i++) {
                      V.set(i, k, -V.get(i, k));
                    }
                  }
                }
                while (k < pp) {
                  if (s[k] >= s[k + 1]) {
                    break;
                  }
                  let t = s[k];
                  s[k] = s[k + 1];
                  s[k + 1] = t;
                  if (wantv && k < n - 1) {
                    for (let i = 0; i < n; i++) {
                      t = V.get(i, k + 1);
                      V.set(i, k + 1, V.get(i, k));
                      V.set(i, k, t);
                    }
                  }
                  if (wantu && k < m - 1) {
                    for (let i = 0; i < m; i++) {
                      t = U.get(i, k + 1);
                      U.set(i, k + 1, U.get(i, k));
                      U.set(i, k, t);
                    }
                  }
                  k++;
                }
                p--;
                break;
              }
            // no default
          }
        }

        if (swapped) {
          let tmp = V;
          V = U;
          U = tmp;
        }
        this.m = m;
        this.n = n;
        this.s = s;
        this.U = U;
        this.V = V;
      }
      solve(value) {
        let Y = value;
        let e = this.threshold;
        let scols = this.s.length;
        let Ls = Matrix$2.zeros(scols, scols);
        for (let i = 0; i < scols; i++) {
          if (Math.abs(this.s[i]) <= e) {
            Ls.set(i, i, 0);
          } else {
            Ls.set(i, i, 1 / this.s[i]);
          }
        }
        let U = this.U;
        let V = this.rightSingularVectors;
        let VL = V.mmul(Ls);
        let vrows = V.rows;
        let urows = U.rows;
        let VLU = Matrix$2.zeros(vrows, urows);
        for (let i = 0; i < vrows; i++) {
          for (let j = 0; j < urows; j++) {
            let sum = 0;
            for (let k = 0; k < scols; k++) {
              sum += VL.get(i, k) * U.get(j, k);
            }
            VLU.set(i, j, sum);
          }
        }
        return VLU.mmul(Y);
      }
      solveForDiagonal(value) {
        return this.solve(Matrix$2.diag(value));
      }
      inverse() {
        let V = this.V;
        let e = this.threshold;
        let vrows = V.rows;
        let vcols = V.columns;
        let X = new Matrix$2(vrows, this.s.length);
        for (let i = 0; i < vrows; i++) {
          for (let j = 0; j < vcols; j++) {
            if (Math.abs(this.s[j]) > e) {
              X.set(i, j, V.get(i, j) / this.s[j]);
            }
          }
        }
        let U = this.U;
        let urows = U.rows;
        let ucols = U.columns;
        let Y = new Matrix$2(vrows, urows);
        for (let i = 0; i < vrows; i++) {
          for (let j = 0; j < urows; j++) {
            let sum = 0;
            for (let k = 0; k < ucols; k++) {
              sum += X.get(i, k) * U.get(j, k);
            }
            Y.set(i, j, sum);
          }
        }
        return Y;
      }
      get condition() {
        return this.s[0] / this.s[Math.min(this.m, this.n) - 1];
      }
      get norm2() {
        return this.s[0];
      }
      get rank() {
        let tol = Math.max(this.m, this.n) * this.s[0] * Number.EPSILON;
        let r = 0;
        let s = this.s;
        for (let i = 0, ii = s.length; i < ii; i++) {
          if (s[i] > tol) {
            r++;
          }
        }
        return r;
      }
      get diagonal() {
        return Array.from(this.s);
      }
      get threshold() {
        return Number.EPSILON / 2 * Math.max(this.m, this.n) * this.s[0];
      }
      get leftSingularVectors() {
        return this.U;
      }
      get rightSingularVectors() {
        return this.V;
      }
      get diagonalMatrix() {
        return Matrix$2.diag(this.s);
      }
    }
    function inverse(matrix, useSVD = false) {
      matrix = WrapperMatrix2D.checkMatrix(matrix);
      if (useSVD) {
        return new SingularValueDecomposition(matrix).inverse();
      } else {
        return solve$1(matrix, Matrix$2.eye(matrix.rows));
      }
    }
    function solve$1(leftHandSide, rightHandSide, useSVD = false) {
      leftHandSide = WrapperMatrix2D.checkMatrix(leftHandSide);
      rightHandSide = WrapperMatrix2D.checkMatrix(rightHandSide);
      if (useSVD) {
        return new SingularValueDecomposition(leftHandSide).solve(rightHandSide);
      } else {
        return leftHandSide.isSquare() ? new LuDecomposition$1(leftHandSide).solve(rightHandSide) : new QrDecomposition(leftHandSide).solve(rightHandSide);
      }
    }
    function determinant(matrix) {
      matrix = Matrix$2.checkMatrix(matrix);
      if (matrix.isSquare()) {
        if (matrix.columns === 0) {
          return 1;
        }
        let a, b, c, d;
        if (matrix.columns === 2) {
          // 2 x 2 matrix
          a = matrix.get(0, 0);
          b = matrix.get(0, 1);
          c = matrix.get(1, 0);
          d = matrix.get(1, 1);
          return a * d - b * c;
        } else if (matrix.columns === 3) {
          // 3 x 3 matrix
          let subMatrix0, subMatrix1, subMatrix2;
          subMatrix0 = new MatrixSelectionView(matrix, [1, 2], [1, 2]);
          subMatrix1 = new MatrixSelectionView(matrix, [1, 2], [0, 2]);
          subMatrix2 = new MatrixSelectionView(matrix, [1, 2], [0, 1]);
          a = matrix.get(0, 0);
          b = matrix.get(0, 1);
          c = matrix.get(0, 2);
          return a * determinant(subMatrix0) - b * determinant(subMatrix1) + c * determinant(subMatrix2);
        } else {
          // general purpose determinant using the LU decomposition
          return new LuDecomposition$1(matrix).determinant;
        }
      } else {
        throw Error('determinant can only be calculated for a square matrix');
      }
    }
    function xrange(n, exception) {
      let range = [];
      for (let i = 0; i < n; i++) {
        if (i !== exception) {
          range.push(i);
        }
      }
      return range;
    }
    function dependenciesOneRow(error, matrix, index, thresholdValue = 10e-10, thresholdError = 10e-10) {
      if (error > thresholdError) {
        return new Array(matrix.rows + 1).fill(0);
      } else {
        let returnArray = matrix.addRow(index, [0]);
        for (let i = 0; i < returnArray.rows; i++) {
          if (Math.abs(returnArray.get(i, 0)) < thresholdValue) {
            returnArray.set(i, 0, 0);
          }
        }
        return returnArray.to1DArray();
      }
    }
    function linearDependencies(matrix, options = {}) {
      const {
        thresholdValue = 10e-10,
        thresholdError = 10e-10
      } = options;
      matrix = Matrix$2.checkMatrix(matrix);
      let n = matrix.rows;
      let results = new Matrix$2(n, n);
      for (let i = 0; i < n; i++) {
        let b = Matrix$2.columnVector(matrix.getRow(i));
        let Abis = matrix.subMatrixRow(xrange(n, i)).transpose();
        let svd = new SingularValueDecomposition(Abis);
        let x = svd.solve(b);
        let error = Matrix$2.sub(b, Abis.mmul(x)).abs().max();
        results.setRow(i, dependenciesOneRow(error, x, i, thresholdValue, thresholdError));
      }
      return results;
    }
    function pseudoInverse(matrix, threshold = Number.EPSILON) {
      matrix = Matrix$2.checkMatrix(matrix);
      if (matrix.isEmpty()) {
        // with a zero dimension, the pseudo-inverse is the transpose, since all 0xn and nx0 matrices are singular
        // (0xn)*(nx0)*(0xn) = 0xn
        // (nx0)*(0xn)*(nx0) = nx0
        return matrix.transpose();
      }
      let svdSolution = new SingularValueDecomposition(matrix, {
        autoTranspose: true
      });
      let U = svdSolution.leftSingularVectors;
      let V = svdSolution.rightSingularVectors;
      let s = svdSolution.diagonal;
      for (let i = 0; i < s.length; i++) {
        if (Math.abs(s[i]) > threshold) {
          s[i] = 1.0 / s[i];
        } else {
          s[i] = 0.0;
        }
      }
      return V.mmul(Matrix$2.diag(s).mmul(U.transpose()));
    }
    function covariance(xMatrix, yMatrix = xMatrix, options = {}) {
      xMatrix = new Matrix$2(xMatrix);
      let yIsSame = false;
      if (typeof yMatrix === 'object' && !Matrix$2.isMatrix(yMatrix) && !isAnyArray.isAnyArray(yMatrix)) {
        options = yMatrix;
        yMatrix = xMatrix;
        yIsSame = true;
      } else {
        yMatrix = new Matrix$2(yMatrix);
      }
      if (xMatrix.rows !== yMatrix.rows) {
        throw new TypeError('Both matrices must have the same number of rows');
      }
      const {
        center = true
      } = options;
      if (center) {
        xMatrix = xMatrix.center('column');
        if (!yIsSame) {
          yMatrix = yMatrix.center('column');
        }
      }
      const cov = xMatrix.transpose().mmul(yMatrix);
      for (let i = 0; i < cov.rows; i++) {
        for (let j = 0; j < cov.columns; j++) {
          cov.set(i, j, cov.get(i, j) * (1 / (xMatrix.rows - 1)));
        }
      }
      return cov;
    }
    function correlation(xMatrix, yMatrix = xMatrix, options = {}) {
      xMatrix = new Matrix$2(xMatrix);
      let yIsSame = false;
      if (typeof yMatrix === 'object' && !Matrix$2.isMatrix(yMatrix) && !isAnyArray.isAnyArray(yMatrix)) {
        options = yMatrix;
        yMatrix = xMatrix;
        yIsSame = true;
      } else {
        yMatrix = new Matrix$2(yMatrix);
      }
      if (xMatrix.rows !== yMatrix.rows) {
        throw new TypeError('Both matrices must have the same number of rows');
      }
      const {
        center = true,
        scale = true
      } = options;
      if (center) {
        xMatrix.center('column');
        if (!yIsSame) {
          yMatrix.center('column');
        }
      }
      if (scale) {
        xMatrix.scale('column');
        if (!yIsSame) {
          yMatrix.scale('column');
        }
      }
      const sdx = xMatrix.standardDeviation('column', {
        unbiased: true
      });
      const sdy = yIsSame ? sdx : yMatrix.standardDeviation('column', {
        unbiased: true
      });
      const corr = xMatrix.transpose().mmul(yMatrix);
      for (let i = 0; i < corr.rows; i++) {
        for (let j = 0; j < corr.columns; j++) {
          corr.set(i, j, corr.get(i, j) * (1 / (sdx[i] * sdy[j])) * (1 / (xMatrix.rows - 1)));
        }
      }
      return corr;
    }
    class EigenvalueDecomposition {
      constructor(matrix, options = {}) {
        const {
          assumeSymmetric = false
        } = options;
        matrix = WrapperMatrix2D.checkMatrix(matrix);
        if (!matrix.isSquare()) {
          throw new Error('Matrix is not a square matrix');
        }
        if (matrix.isEmpty()) {
          throw new Error('Matrix must be non-empty');
        }
        let n = matrix.columns;
        let V = new Matrix$2(n, n);
        let d = new Float64Array(n);
        let e = new Float64Array(n);
        let value = matrix;
        let i, j;
        let isSymmetric = false;
        if (assumeSymmetric) {
          isSymmetric = true;
        } else {
          isSymmetric = matrix.isSymmetric();
        }
        if (isSymmetric) {
          for (i = 0; i < n; i++) {
            for (j = 0; j < n; j++) {
              V.set(i, j, value.get(i, j));
            }
          }
          tred2(n, e, d, V);
          tql2(n, e, d, V);
        } else {
          let H = new Matrix$2(n, n);
          let ort = new Float64Array(n);
          for (j = 0; j < n; j++) {
            for (i = 0; i < n; i++) {
              H.set(i, j, value.get(i, j));
            }
          }
          orthes(n, H, ort, V);
          hqr2(n, e, d, V, H);
        }
        this.n = n;
        this.e = e;
        this.d = d;
        this.V = V;
      }
      get realEigenvalues() {
        return Array.from(this.d);
      }
      get imaginaryEigenvalues() {
        return Array.from(this.e);
      }
      get eigenvectorMatrix() {
        return this.V;
      }
      get diagonalMatrix() {
        let n = this.n;
        let e = this.e;
        let d = this.d;
        let X = new Matrix$2(n, n);
        let i, j;
        for (i = 0; i < n; i++) {
          for (j = 0; j < n; j++) {
            X.set(i, j, 0);
          }
          X.set(i, i, d[i]);
          if (e[i] > 0) {
            X.set(i, i + 1, e[i]);
          } else if (e[i] < 0) {
            X.set(i, i - 1, e[i]);
          }
        }
        return X;
      }
    }
    function tred2(n, e, d, V) {
      let f, g, h, i, j, k, hh, scale;
      for (j = 0; j < n; j++) {
        d[j] = V.get(n - 1, j);
      }
      for (i = n - 1; i > 0; i--) {
        scale = 0;
        h = 0;
        for (k = 0; k < i; k++) {
          scale = scale + Math.abs(d[k]);
        }
        if (scale === 0) {
          e[i] = d[i - 1];
          for (j = 0; j < i; j++) {
            d[j] = V.get(i - 1, j);
            V.set(i, j, 0);
            V.set(j, i, 0);
          }
        } else {
          for (k = 0; k < i; k++) {
            d[k] /= scale;
            h += d[k] * d[k];
          }
          f = d[i - 1];
          g = Math.sqrt(h);
          if (f > 0) {
            g = -g;
          }
          e[i] = scale * g;
          h = h - f * g;
          d[i - 1] = f - g;
          for (j = 0; j < i; j++) {
            e[j] = 0;
          }
          for (j = 0; j < i; j++) {
            f = d[j];
            V.set(j, i, f);
            g = e[j] + V.get(j, j) * f;
            for (k = j + 1; k <= i - 1; k++) {
              g += V.get(k, j) * d[k];
              e[k] += V.get(k, j) * f;
            }
            e[j] = g;
          }
          f = 0;
          for (j = 0; j < i; j++) {
            e[j] /= h;
            f += e[j] * d[j];
          }
          hh = f / (h + h);
          for (j = 0; j < i; j++) {
            e[j] -= hh * d[j];
          }
          for (j = 0; j < i; j++) {
            f = d[j];
            g = e[j];
            for (k = j; k <= i - 1; k++) {
              V.set(k, j, V.get(k, j) - (f * e[k] + g * d[k]));
            }
            d[j] = V.get(i - 1, j);
            V.set(i, j, 0);
          }
        }
        d[i] = h;
      }
      for (i = 0; i < n - 1; i++) {
        V.set(n - 1, i, V.get(i, i));
        V.set(i, i, 1);
        h = d[i + 1];
        if (h !== 0) {
          for (k = 0; k <= i; k++) {
            d[k] = V.get(k, i + 1) / h;
          }
          for (j = 0; j <= i; j++) {
            g = 0;
            for (k = 0; k <= i; k++) {
              g += V.get(k, i + 1) * V.get(k, j);
            }
            for (k = 0; k <= i; k++) {
              V.set(k, j, V.get(k, j) - g * d[k]);
            }
          }
        }
        for (k = 0; k <= i; k++) {
          V.set(k, i + 1, 0);
        }
      }
      for (j = 0; j < n; j++) {
        d[j] = V.get(n - 1, j);
        V.set(n - 1, j, 0);
      }
      V.set(n - 1, n - 1, 1);
      e[0] = 0;
    }
    function tql2(n, e, d, V) {
      let g, h, i, j, k, l, m, p, r, dl1, c, c2, c3, el1, s, s2;
      for (i = 1; i < n; i++) {
        e[i - 1] = e[i];
      }
      e[n - 1] = 0;
      let f = 0;
      let tst1 = 0;
      let eps = Number.EPSILON;
      for (l = 0; l < n; l++) {
        tst1 = Math.max(tst1, Math.abs(d[l]) + Math.abs(e[l]));
        m = l;
        while (m < n) {
          if (Math.abs(e[m]) <= eps * tst1) {
            break;
          }
          m++;
        }
        if (m > l) {
          do {
            g = d[l];
            p = (d[l + 1] - g) / (2 * e[l]);
            r = hypotenuse(p, 1);
            if (p < 0) {
              r = -r;
            }
            d[l] = e[l] / (p + r);
            d[l + 1] = e[l] * (p + r);
            dl1 = d[l + 1];
            h = g - d[l];
            for (i = l + 2; i < n; i++) {
              d[i] -= h;
            }
            f = f + h;
            p = d[m];
            c = 1;
            c2 = c;
            c3 = c;
            el1 = e[l + 1];
            s = 0;
            s2 = 0;
            for (i = m - 1; i >= l; i--) {
              c3 = c2;
              c2 = c;
              s2 = s;
              g = c * e[i];
              h = c * p;
              r = hypotenuse(p, e[i]);
              e[i + 1] = s * r;
              s = e[i] / r;
              c = p / r;
              p = c * d[i] - s * g;
              d[i + 1] = h + s * (c * g + s * d[i]);
              for (k = 0; k < n; k++) {
                h = V.get(k, i + 1);
                V.set(k, i + 1, s * V.get(k, i) + c * h);
                V.set(k, i, c * V.get(k, i) - s * h);
              }
            }
            p = -s * s2 * c3 * el1 * e[l] / dl1;
            e[l] = s * p;
            d[l] = c * p;
          } while (Math.abs(e[l]) > eps * tst1);
        }
        d[l] = d[l] + f;
        e[l] = 0;
      }
      for (i = 0; i < n - 1; i++) {
        k = i;
        p = d[i];
        for (j = i + 1; j < n; j++) {
          if (d[j] < p) {
            k = j;
            p = d[j];
          }
        }
        if (k !== i) {
          d[k] = d[i];
          d[i] = p;
          for (j = 0; j < n; j++) {
            p = V.get(j, i);
            V.set(j, i, V.get(j, k));
            V.set(j, k, p);
          }
        }
      }
    }
    function orthes(n, H, ort, V) {
      let low = 0;
      let high = n - 1;
      let f, g, h, i, j, m;
      let scale;
      for (m = low + 1; m <= high - 1; m++) {
        scale = 0;
        for (i = m; i <= high; i++) {
          scale = scale + Math.abs(H.get(i, m - 1));
        }
        if (scale !== 0) {
          h = 0;
          for (i = high; i >= m; i--) {
            ort[i] = H.get(i, m - 1) / scale;
            h += ort[i] * ort[i];
          }
          g = Math.sqrt(h);
          if (ort[m] > 0) {
            g = -g;
          }
          h = h - ort[m] * g;
          ort[m] = ort[m] - g;
          for (j = m; j < n; j++) {
            f = 0;
            for (i = high; i >= m; i--) {
              f += ort[i] * H.get(i, j);
            }
            f = f / h;
            for (i = m; i <= high; i++) {
              H.set(i, j, H.get(i, j) - f * ort[i]);
            }
          }
          for (i = 0; i <= high; i++) {
            f = 0;
            for (j = high; j >= m; j--) {
              f += ort[j] * H.get(i, j);
            }
            f = f / h;
            for (j = m; j <= high; j++) {
              H.set(i, j, H.get(i, j) - f * ort[j]);
            }
          }
          ort[m] = scale * ort[m];
          H.set(m, m - 1, scale * g);
        }
      }
      for (i = 0; i < n; i++) {
        for (j = 0; j < n; j++) {
          V.set(i, j, i === j ? 1 : 0);
        }
      }
      for (m = high - 1; m >= low + 1; m--) {
        if (H.get(m, m - 1) !== 0) {
          for (i = m + 1; i <= high; i++) {
            ort[i] = H.get(i, m - 1);
          }
          for (j = m; j <= high; j++) {
            g = 0;
            for (i = m; i <= high; i++) {
              g += ort[i] * V.get(i, j);
            }
            g = g / ort[m] / H.get(m, m - 1);
            for (i = m; i <= high; i++) {
              V.set(i, j, V.get(i, j) + g * ort[i]);
            }
          }
        }
      }
    }
    function hqr2(nn, e, d, V, H) {
      let n = nn - 1;
      let low = 0;
      let high = nn - 1;
      let eps = Number.EPSILON;
      let exshift = 0;
      let norm = 0;
      let p = 0;
      let q = 0;
      let r = 0;
      let s = 0;
      let z = 0;
      let iter = 0;
      let i, j, k, l, m, t, w, x, y;
      let ra, sa, vr, vi;
      let notlast, cdivres;
      for (i = 0; i < nn; i++) {
        if (i < low || i > high) {
          d[i] = H.get(i, i);
          e[i] = 0;
        }
        for (j = Math.max(i - 1, 0); j < nn; j++) {
          norm = norm + Math.abs(H.get(i, j));
        }
      }
      while (n >= low) {
        l = n;
        while (l > low) {
          s = Math.abs(H.get(l - 1, l - 1)) + Math.abs(H.get(l, l));
          if (s === 0) {
            s = norm;
          }
          if (Math.abs(H.get(l, l - 1)) < eps * s) {
            break;
          }
          l--;
        }
        if (l === n) {
          H.set(n, n, H.get(n, n) + exshift);
          d[n] = H.get(n, n);
          e[n] = 0;
          n--;
          iter = 0;
        } else if (l === n - 1) {
          w = H.get(n, n - 1) * H.get(n - 1, n);
          p = (H.get(n - 1, n - 1) - H.get(n, n)) / 2;
          q = p * p + w;
          z = Math.sqrt(Math.abs(q));
          H.set(n, n, H.get(n, n) + exshift);
          H.set(n - 1, n - 1, H.get(n - 1, n - 1) + exshift);
          x = H.get(n, n);
          if (q >= 0) {
            z = p >= 0 ? p + z : p - z;
            d[n - 1] = x + z;
            d[n] = d[n - 1];
            if (z !== 0) {
              d[n] = x - w / z;
            }
            e[n - 1] = 0;
            e[n] = 0;
            x = H.get(n, n - 1);
            s = Math.abs(x) + Math.abs(z);
            p = x / s;
            q = z / s;
            r = Math.sqrt(p * p + q * q);
            p = p / r;
            q = q / r;
            for (j = n - 1; j < nn; j++) {
              z = H.get(n - 1, j);
              H.set(n - 1, j, q * z + p * H.get(n, j));
              H.set(n, j, q * H.get(n, j) - p * z);
            }
            for (i = 0; i <= n; i++) {
              z = H.get(i, n - 1);
              H.set(i, n - 1, q * z + p * H.get(i, n));
              H.set(i, n, q * H.get(i, n) - p * z);
            }
            for (i = low; i <= high; i++) {
              z = V.get(i, n - 1);
              V.set(i, n - 1, q * z + p * V.get(i, n));
              V.set(i, n, q * V.get(i, n) - p * z);
            }
          } else {
            d[n - 1] = x + p;
            d[n] = x + p;
            e[n - 1] = z;
            e[n] = -z;
          }
          n = n - 2;
          iter = 0;
        } else {
          x = H.get(n, n);
          y = 0;
          w = 0;
          if (l < n) {
            y = H.get(n - 1, n - 1);
            w = H.get(n, n - 1) * H.get(n - 1, n);
          }
          if (iter === 10) {
            exshift += x;
            for (i = low; i <= n; i++) {
              H.set(i, i, H.get(i, i) - x);
            }
            s = Math.abs(H.get(n, n - 1)) + Math.abs(H.get(n - 1, n - 2));
            // eslint-disable-next-line no-multi-assign
            x = y = 0.75 * s;
            w = -0.4375 * s * s;
          }
          if (iter === 30) {
            s = (y - x) / 2;
            s = s * s + w;
            if (s > 0) {
              s = Math.sqrt(s);
              if (y < x) {
                s = -s;
              }
              s = x - w / ((y - x) / 2 + s);
              for (i = low; i <= n; i++) {
                H.set(i, i, H.get(i, i) - s);
              }
              exshift += s;
              // eslint-disable-next-line no-multi-assign
              x = y = w = 0.964;
            }
          }
          iter = iter + 1;
          m = n - 2;
          while (m >= l) {
            z = H.get(m, m);
            r = x - z;
            s = y - z;
            p = (r * s - w) / H.get(m + 1, m) + H.get(m, m + 1);
            q = H.get(m + 1, m + 1) - z - r - s;
            r = H.get(m + 2, m + 1);
            s = Math.abs(p) + Math.abs(q) + Math.abs(r);
            p = p / s;
            q = q / s;
            r = r / s;
            if (m === l) {
              break;
            }
            if (Math.abs(H.get(m, m - 1)) * (Math.abs(q) + Math.abs(r)) < eps * (Math.abs(p) * (Math.abs(H.get(m - 1, m - 1)) + Math.abs(z) + Math.abs(H.get(m + 1, m + 1))))) {
              break;
            }
            m--;
          }
          for (i = m + 2; i <= n; i++) {
            H.set(i, i - 2, 0);
            if (i > m + 2) {
              H.set(i, i - 3, 0);
            }
          }
          for (k = m; k <= n - 1; k++) {
            notlast = k !== n - 1;
            if (k !== m) {
              p = H.get(k, k - 1);
              q = H.get(k + 1, k - 1);
              r = notlast ? H.get(k + 2, k - 1) : 0;
              x = Math.abs(p) + Math.abs(q) + Math.abs(r);
              if (x !== 0) {
                p = p / x;
                q = q / x;
                r = r / x;
              }
            }
            if (x === 0) {
              break;
            }
            s = Math.sqrt(p * p + q * q + r * r);
            if (p < 0) {
              s = -s;
            }
            if (s !== 0) {
              if (k !== m) {
                H.set(k, k - 1, -s * x);
              } else if (l !== m) {
                H.set(k, k - 1, -H.get(k, k - 1));
              }
              p = p + s;
              x = p / s;
              y = q / s;
              z = r / s;
              q = q / p;
              r = r / p;
              for (j = k; j < nn; j++) {
                p = H.get(k, j) + q * H.get(k + 1, j);
                if (notlast) {
                  p = p + r * H.get(k + 2, j);
                  H.set(k + 2, j, H.get(k + 2, j) - p * z);
                }
                H.set(k, j, H.get(k, j) - p * x);
                H.set(k + 1, j, H.get(k + 1, j) - p * y);
              }
              for (i = 0; i <= Math.min(n, k + 3); i++) {
                p = x * H.get(i, k) + y * H.get(i, k + 1);
                if (notlast) {
                  p = p + z * H.get(i, k + 2);
                  H.set(i, k + 2, H.get(i, k + 2) - p * r);
                }
                H.set(i, k, H.get(i, k) - p);
                H.set(i, k + 1, H.get(i, k + 1) - p * q);
              }
              for (i = low; i <= high; i++) {
                p = x * V.get(i, k) + y * V.get(i, k + 1);
                if (notlast) {
                  p = p + z * V.get(i, k + 2);
                  V.set(i, k + 2, V.get(i, k + 2) - p * r);
                }
                V.set(i, k, V.get(i, k) - p);
                V.set(i, k + 1, V.get(i, k + 1) - p * q);
              }
            }
          }
        }
      }
      if (norm === 0) {
        return;
      }
      for (n = nn - 1; n >= 0; n--) {
        p = d[n];
        q = e[n];
        if (q === 0) {
          l = n;
          H.set(n, n, 1);
          for (i = n - 1; i >= 0; i--) {
            w = H.get(i, i) - p;
            r = 0;
            for (j = l; j <= n; j++) {
              r = r + H.get(i, j) * H.get(j, n);
            }
            if (e[i] < 0) {
              z = w;
              s = r;
            } else {
              l = i;
              if (e[i] === 0) {
                H.set(i, n, w !== 0 ? -r / w : -r / (eps * norm));
              } else {
                x = H.get(i, i + 1);
                y = H.get(i + 1, i);
                q = (d[i] - p) * (d[i] - p) + e[i] * e[i];
                t = (x * s - z * r) / q;
                H.set(i, n, t);
                H.set(i + 1, n, Math.abs(x) > Math.abs(z) ? (-r - w * t) / x : (-s - y * t) / z);
              }
              t = Math.abs(H.get(i, n));
              if (eps * t * t > 1) {
                for (j = i; j <= n; j++) {
                  H.set(j, n, H.get(j, n) / t);
                }
              }
            }
          }
        } else if (q < 0) {
          l = n - 1;
          if (Math.abs(H.get(n, n - 1)) > Math.abs(H.get(n - 1, n))) {
            H.set(n - 1, n - 1, q / H.get(n, n - 1));
            H.set(n - 1, n, -(H.get(n, n) - p) / H.get(n, n - 1));
          } else {
            cdivres = cdiv(0, -H.get(n - 1, n), H.get(n - 1, n - 1) - p, q);
            H.set(n - 1, n - 1, cdivres[0]);
            H.set(n - 1, n, cdivres[1]);
          }
          H.set(n, n - 1, 0);
          H.set(n, n, 1);
          for (i = n - 2; i >= 0; i--) {
            ra = 0;
            sa = 0;
            for (j = l; j <= n; j++) {
              ra = ra + H.get(i, j) * H.get(j, n - 1);
              sa = sa + H.get(i, j) * H.get(j, n);
            }
            w = H.get(i, i) - p;
            if (e[i] < 0) {
              z = w;
              r = ra;
              s = sa;
            } else {
              l = i;
              if (e[i] === 0) {
                cdivres = cdiv(-ra, -sa, w, q);
                H.set(i, n - 1, cdivres[0]);
                H.set(i, n, cdivres[1]);
              } else {
                x = H.get(i, i + 1);
                y = H.get(i + 1, i);
                vr = (d[i] - p) * (d[i] - p) + e[i] * e[i] - q * q;
                vi = (d[i] - p) * 2 * q;
                if (vr === 0 && vi === 0) {
                  vr = eps * norm * (Math.abs(w) + Math.abs(q) + Math.abs(x) + Math.abs(y) + Math.abs(z));
                }
                cdivres = cdiv(x * r - z * ra + q * sa, x * s - z * sa - q * ra, vr, vi);
                H.set(i, n - 1, cdivres[0]);
                H.set(i, n, cdivres[1]);
                if (Math.abs(x) > Math.abs(z) + Math.abs(q)) {
                  H.set(i + 1, n - 1, (-ra - w * H.get(i, n - 1) + q * H.get(i, n)) / x);
                  H.set(i + 1, n, (-sa - w * H.get(i, n) - q * H.get(i, n - 1)) / x);
                } else {
                  cdivres = cdiv(-r - y * H.get(i, n - 1), -s - y * H.get(i, n), z, q);
                  H.set(i + 1, n - 1, cdivres[0]);
                  H.set(i + 1, n, cdivres[1]);
                }
              }
              t = Math.max(Math.abs(H.get(i, n - 1)), Math.abs(H.get(i, n)));
              if (eps * t * t > 1) {
                for (j = i; j <= n; j++) {
                  H.set(j, n - 1, H.get(j, n - 1) / t);
                  H.set(j, n, H.get(j, n) / t);
                }
              }
            }
          }
        }
      }
      for (i = 0; i < nn; i++) {
        if (i < low || i > high) {
          for (j = i; j < nn; j++) {
            V.set(i, j, H.get(i, j));
          }
        }
      }
      for (j = nn - 1; j >= low; j--) {
        for (i = low; i <= high; i++) {
          z = 0;
          for (k = low; k <= Math.min(j, high); k++) {
            z = z + V.get(i, k) * H.get(k, j);
          }
          V.set(i, j, z);
        }
      }
    }
    function cdiv(xr, xi, yr, yi) {
      let r, d;
      if (Math.abs(yr) > Math.abs(yi)) {
        r = yi / yr;
        d = yr + r * yi;
        return [(xr + r * xi) / d, (xi - r * xr) / d];
      } else {
        r = yr / yi;
        d = yi + r * yr;
        return [(r * xr + xi) / d, (r * xi - xr) / d];
      }
    }
    class CholeskyDecomposition$1 {
      constructor(value) {
        value = WrapperMatrix2D.checkMatrix(value);
        if (!value.isSymmetric()) {
          throw new Error('Matrix is not symmetric');
        }
        let a = value;
        let dimension = a.rows;
        let l = new Matrix$2(dimension, dimension);
        let positiveDefinite = true;
        let i, j, k;
        for (j = 0; j < dimension; j++) {
          let d = 0;
          for (k = 0; k < j; k++) {
            let s = 0;
            for (i = 0; i < k; i++) {
              s += l.get(k, i) * l.get(j, i);
            }
            s = (a.get(j, k) - s) / l.get(k, k);
            l.set(j, k, s);
            d = d + s * s;
          }
          d = a.get(j, j) - d;
          positiveDefinite &= d > 0;
          l.set(j, j, Math.sqrt(Math.max(d, 0)));
          for (k = j + 1; k < dimension; k++) {
            l.set(j, k, 0);
          }
        }
        this.L = l;
        this.positiveDefinite = Boolean(positiveDefinite);
      }
      isPositiveDefinite() {
        return this.positiveDefinite;
      }
      solve(value) {
        value = WrapperMatrix2D.checkMatrix(value);
        let l = this.L;
        let dimension = l.rows;
        if (value.rows !== dimension) {
          throw new Error('Matrix dimensions do not match');
        }
        if (this.isPositiveDefinite() === false) {
          throw new Error('Matrix is not positive definite');
        }
        let count = value.columns;
        let B = value.clone();
        let i, j, k;
        for (k = 0; k < dimension; k++) {
          for (j = 0; j < count; j++) {
            for (i = 0; i < k; i++) {
              B.set(k, j, B.get(k, j) - B.get(i, j) * l.get(k, i));
            }
            B.set(k, j, B.get(k, j) / l.get(k, k));
          }
        }
        for (k = dimension - 1; k >= 0; k--) {
          for (j = 0; j < count; j++) {
            for (i = k + 1; i < dimension; i++) {
              B.set(k, j, B.get(k, j) - B.get(i, j) * l.get(i, k));
            }
            B.set(k, j, B.get(k, j) / l.get(k, k));
          }
        }
        return B;
      }
      get lowerTriangularMatrix() {
        return this.L;
      }
    }
    class nipals {
      constructor(X, options = {}) {
        X = WrapperMatrix2D.checkMatrix(X);
        let {
          Y
        } = options;
        const {
          scaleScores = false,
          maxIterations = 1000,
          terminationCriteria = 1e-10
        } = options;
        let u;
        if (Y) {
          if (isAnyArray.isAnyArray(Y) && typeof Y[0] === 'number') {
            Y = Matrix$2.columnVector(Y);
          } else {
            Y = WrapperMatrix2D.checkMatrix(Y);
          }
          if (Y.rows !== X.rows) {
            throw new Error('Y should have the same number of rows as X');
          }
          u = Y.getColumnVector(0);
        } else {
          u = X.getColumnVector(0);
        }
        let diff = 1;
        let t, q, w, tOld;
        for (let counter = 0; counter < maxIterations && diff > terminationCriteria; counter++) {
          w = X.transpose().mmul(u).div(u.transpose().mmul(u).get(0, 0));
          w = w.div(w.norm());
          t = X.mmul(w).div(w.transpose().mmul(w).get(0, 0));
          if (counter > 0) {
            diff = t.clone().sub(tOld).pow(2).sum();
          }
          tOld = t.clone();
          if (Y) {
            q = Y.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
            q = q.div(q.norm());
            u = Y.mmul(q).div(q.transpose().mmul(q).get(0, 0));
          } else {
            u = t;
          }
        }
        if (Y) {
          let p = X.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
          p = p.div(p.norm());
          let xResidual = X.clone().sub(t.clone().mmul(p.transpose()));
          let residual = u.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
          let yResidual = Y.clone().sub(t.clone().mulS(residual.get(0, 0)).mmul(q.transpose()));
          this.t = t;
          this.p = p.transpose();
          this.w = w.transpose();
          this.q = q;
          this.u = u;
          this.s = t.transpose().mmul(t);
          this.xResidual = xResidual;
          this.yResidual = yResidual;
          this.betas = residual;
        } else {
          this.w = w.transpose();
          this.s = t.transpose().mmul(t).sqrt();
          if (scaleScores) {
            this.t = t.clone().div(this.s.get(0, 0));
          } else {
            this.t = t;
          }
          this.xResidual = X.sub(t.mmul(w.transpose()));
        }
      }
    }
    matrix$1.AbstractMatrix = AbstractMatrix;
    matrix$1.CHO = CholeskyDecomposition$1;
    matrix$1.CholeskyDecomposition = CholeskyDecomposition$1;
    matrix$1.EVD = EigenvalueDecomposition;
    matrix$1.EigenvalueDecomposition = EigenvalueDecomposition;
    matrix$1.LU = LuDecomposition$1;
    matrix$1.LuDecomposition = LuDecomposition$1;
    matrix$1.Matrix = Matrix$2;
    matrix$1.MatrixColumnSelectionView = MatrixColumnSelectionView;
    matrix$1.MatrixColumnView = MatrixColumnView;
    matrix$1.MatrixFlipColumnView = MatrixFlipColumnView;
    matrix$1.MatrixFlipRowView = MatrixFlipRowView;
    matrix$1.MatrixRowSelectionView = MatrixRowSelectionView;
    matrix$1.MatrixRowView = MatrixRowView;
    matrix$1.MatrixSelectionView = MatrixSelectionView;
    matrix$1.MatrixSubView = MatrixSubView;
    matrix$1.MatrixTransposeView = MatrixTransposeView;
    matrix$1.NIPALS = nipals;
    matrix$1.Nipals = nipals;
    matrix$1.QR = QrDecomposition;
    matrix$1.QrDecomposition = QrDecomposition;
    matrix$1.SVD = SingularValueDecomposition;
    matrix$1.SingularValueDecomposition = SingularValueDecomposition;
    matrix$1.WrapperMatrix1D = WrapperMatrix1D;
    matrix$1.WrapperMatrix2D = WrapperMatrix2D;
    matrix$1.correlation = correlation;
    matrix$1.covariance = covariance;
    var _default = matrix$1.default = Matrix$2;
    matrix$1.determinant = determinant;
    matrix$1.inverse = inverse;
    matrix$1.linearDependencies = linearDependencies;
    matrix$1.pseudoInverse = pseudoInverse;
    matrix$1.solve = solve$1;
    matrix$1.wrap = wrap;

    _default.AbstractMatrix;
    _default.CHO;
    const CholeskyDecomposition = _default.CholeskyDecomposition;
    _default.EVD;
    _default.EigenvalueDecomposition;
    _default.LU;
    const LuDecomposition = _default.LuDecomposition;
    const Matrix$1 = _default.Matrix;
    _default.MatrixColumnSelectionView;
    _default.MatrixColumnView;
    _default.MatrixFlipColumnView;
    _default.MatrixFlipRowView;
    _default.MatrixRowSelectionView;
    _default.MatrixRowView;
    _default.MatrixSelectionView;
    _default.MatrixSubView;
    _default.MatrixTransposeView;
    _default.NIPALS;
    _default.Nipals;
    _default.QR;
    _default.QrDecomposition;
    _default.SVD;
    _default.SingularValueDecomposition;
    _default.WrapperMatrix1D;
    _default.WrapperMatrix2D;
    _default.correlation;
    _default.covariance;
    _default.default;
    _default.determinant;
    _default.inverse;
    _default.linearDependencies;
    _default.pseudoInverse;
    const solve = _default.solve;
    _default.wrap;

    /**
     * Returns the closest index of a `target`
     *
     * @param array - array of numbers
     * @param target - target
     * @returns - closest index
     */
    function xFindClosestIndex(array, target, options = {}) {
      const {
        sorted = true
      } = options;
      if (sorted) {
        let low = 0;
        let high = array.length - 1;
        let middle = 0;
        while (high - low > 1) {
          middle = low + (high - low >> 1);
          if (array[middle] < target) {
            low = middle;
          } else if (array[middle] > target) {
            high = middle;
          } else {
            return middle;
          }
        }
        if (low < array.length - 1) {
          if (Math.abs(target - array[low]) < Math.abs(array[low + 1] - target)) {
            return low;
          } else {
            return low + 1;
          }
        } else {
          return low;
        }
      } else {
        let index = 0;
        let diff = Number.POSITIVE_INFINITY;
        for (let i = 0; i < array.length; i++) {
          const currentDiff = Math.abs(array[i] - target);
          if (currentDiff < diff) {
            diff = currentDiff;
            index = i;
          }
        }
        return index;
      }
    }

    /**
     * Returns an object with {fromIndex, toIndex} for a specific from / to
     *
     * @param x - array of numbers
     * @param options - Options
     */
    function xGetFromToIndex(x, options = {}) {
      let {
        fromIndex,
        toIndex
      } = options;
      const {
        from,
        to
      } = options;
      if (fromIndex === undefined) {
        if (from !== undefined) {
          fromIndex = xFindClosestIndex(x, from);
        } else {
          fromIndex = 0;
        }
      }
      if (toIndex === undefined) {
        if (to !== undefined) {
          toIndex = xFindClosestIndex(x, to);
        } else {
          toIndex = x.length - 1;
        }
      }
      if (fromIndex < 0) fromIndex = 0;
      if (toIndex < 0) toIndex = 0;
      if (fromIndex >= x.length) fromIndex = x.length - 1;
      if (toIndex >= x.length) toIndex = x.length - 1;
      if (fromIndex > toIndex) [fromIndex, toIndex] = [toIndex, fromIndex];
      return {
        fromIndex,
        toIndex
      };
    }

    /**
     * Computes the maximal value of an array of values
     *
     * @param array - array of numbers
     * @param options - options
     */
    function xMaxValue(array, options = {}) {
      xCheck(array);
      const {
        fromIndex,
        toIndex
      } = xGetFromToIndex(array, options);
      let maxValue = array[fromIndex];
      for (let i = fromIndex + 1; i <= toIndex; i++) {
        if (array[i] > maxValue) {
          maxValue = array[i];
        }
      }
      return maxValue;
    }

    /**
     * Computes the minimal value of an array of values
     *
     * @param array - array of numbers
     * @param options - options
     */
    function xMinValue(array, options = {}) {
      xCheck(array);
      const {
        fromIndex,
        toIndex
      } = xGetFromToIndex(array, options);
      let minValue = array[fromIndex];
      for (let i = fromIndex + 1; i <= toIndex; i++) {
        if (array[i] < minValue) {
          minValue = array[i];
        }
      }
      return minValue;
    }

    /**
     * Check if the values are separated always by the same difference
     *
     * @param array - Monotone growing array of number
     */
    function xIsEquallySpaced(array, options = {}) {
      if (array.length < 3) return true;
      const {
        tolerance = 0.05
      } = options;
      let maxDx = 0;
      let minDx = Number.MAX_SAFE_INTEGER;
      for (let i = 0; i < array.length - 1; ++i) {
        const absoluteDifference = array[i + 1] - array[i];
        if (absoluteDifference < minDx) {
          minDx = absoluteDifference;
        }
        if (absoluteDifference > maxDx) {
          maxDx = absoluteDifference;
        }
      }
      return (maxDx - minDx) / maxDx < tolerance;
    }

    /**
     * Returns true if x is monotone
     *
     * @param array - array of numbers
     */
    function xIsMonotonic(array) {
      if (array.length <= 2) {
        return 1;
      }
      if (array[0] === array[1]) {
        // maybe a constant series
        for (let i = 1; i < array.length - 1; i++) {
          if (array[i] !== array[i + 1]) return 0;
        }
        return 1;
      }
      if (array[0] < array[array.length - 1]) {
        for (let i = 0; i < array.length - 1; i++) {
          if (array[i] >= array[i + 1]) return 0;
        }
        return 1;
      } else {
        for (let i = 0; i < array.length - 1; i++) {
          if (array[i] <= array[i + 1]) return 0;
        }
        return -1;
      }
    }

    /**
     * This function calculates the median absolute deviation (MAD)
     * https://en.wikipedia.org/wiki/Median_absolute_deviation
     * @param array
     */
    function xMedianAbsoluteDeviation(array) {
      const median = xMedian(array);
      const averageDeviations = new Float64Array(array.length);
      for (let i = 0; i < array.length; i++) {
        averageDeviations[i] = Math.abs(array[i] - median);
      }
      return {
        median,
        mad: xMedian(averageDeviations)
      };
    }

    /**
     * Return min and max values of an array
     *
     * @param array - array of number
     * @returns - Object with 2 properties, min and max
     */
    function xMinMaxValues(array) {
      xCheck(array);
      let min = array[0];
      let max = array[0];
      for (const value of array) {
        if (value < min) min = value;
        if (value > max) max = value;
      }
      return {
        min,
        max
      };
    }

    /**
     * Determine noise level using MAD https://en.wikipedia.org/wiki/Median_absolute_deviation
     * Constant to convert mad to sd calculated using https://www.wolframalpha.com/input?i=sqrt%282%29+inverse+erf%280.5%29
     * This assumes a gaussian distribution of the noise
     * @param array
     * @returns noise level corresponding to one standard deviation
     */
    function xNoiseStandardDeviation(array) {
      const {
        mad,
        median
      } = xMedianAbsoluteDeviation(array);
      return {
        sd: mad / 0.6744897501960817,
        mad,
        median
      };
    }

    /**
     * Calculate the sum of the values
     *
     * @param array - Object that contains property x (an ordered increasing array) and y (an array).
     * @param options - Options.
     * @returns XSum value on the specified range.
     */
    function xSum(array, options = {}) {
      xCheck(array);
      const {
        fromIndex,
        toIndex
      } = xGetFromToIndex(array, options);
      let sumValue = array[fromIndex];
      for (let i = fromIndex + 1; i <= toIndex; i++) {
        sumValue += array[i];
      }
      return sumValue;
    }

    /**
     * Divides the data with either the sum, the absolute sum or the maximum of the data
     * @param array - Array containing values
     * @param options - options
     * @returns - normalized data
     */
    function xNormed(input, options = {}) {
      const {
        algorithm = 'absolute',
        value = 1
      } = options;
      xCheck(input);
      const output = getOutputArray(options.output, input.length);
      if (input.length === 0) {
        throw new Error('input must not be empty');
      }
      switch (algorithm.toLowerCase()) {
        case 'absolute':
          {
            const absoluteSumValue = absoluteSum(input) / value;
            if (absoluteSumValue === 0) {
              throw new Error('xNormed: trying to divide by 0');
            }
            for (let i = 0; i < input.length; i++) {
              output[i] = input[i] / absoluteSumValue;
            }
            return output;
          }
        case 'max':
          {
            const currentMaxValue = xMaxValue(input);
            if (currentMaxValue === 0) {
              throw new Error('xNormed: trying to divide by 0');
            }
            const factor = value / currentMaxValue;
            for (let i = 0; i < input.length; i++) {
              output[i] = input[i] * factor;
            }
            return output;
          }
        case 'sum':
          {
            const sumFactor = xSum(input) / value;
            if (sumFactor === 0) {
              throw new Error('xNormed: trying to divide by 0');
            }
            for (let i = 0; i < input.length; i++) {
              output[i] = input[i] / sumFactor;
            }
            return output;
          }
        default:
          throw new Error(`norm: unknown algorithm: ${algorithm}`);
      }
    }
    function absoluteSum(input) {
      let sumValue = 0;
      for (let i = 0; i < input.length; i++) {
        sumValue += Math.abs(input[i]);
      }
      return sumValue;
    }

    /**
     * Throw an error in no an object of x,y arrays
     *
     * @param data - array of points {x,y,z}
     */
    function xyCheck(data, options = {}) {
      const {
        minLength
      } = options;
      if (typeof data !== 'object' || !isAnyArray$1(data.x) || !isAnyArray$1(data.y)) {
        throw new Error('Data must be an object of x and y arrays');
      }
      if (data.x.length !== data.y.length) {
        throw new Error('The x and y arrays must have the same length');
      }
      if (minLength && data.x.length < minLength) {
        throw new Error(`data.x must have a length of at least ${minLength}`);
      }
    }

    /**
     * Filters x,y values to allow strictly growing values in x axis.
     *
     * @param data - Object that contains property x (an ordered increasing array) and y (an array).
     */
    function xyEnsureGrowingX(data) {
      xyCheck(data);
      if (xIsMonotonic(data.x) === 1) return data;
      const x = Array.from(data.x);
      const y = Array.from(data.y);
      let prevX = Number.NEGATIVE_INFINITY;
      let currentIndex = 0;
      for (let index = 0; index < x.length; index++) {
        if (prevX < x[index]) {
          if (currentIndex < index) {
            x[currentIndex] = x[index];
            y[currentIndex] = y[index];
          }
          currentIndex++;
          prevX = x[index];
        }
      }
      x.length = currentIndex;
      y.length = currentIndex;
      return {
        x,
        y
      };
    }

    /** Filter an array x/y based on various criteria x points are expected to be sorted
     *
     * @param data - object containing 2 properties x and y
     * @param minRelativeYValue - the minimum relative value compare to the Y max value
     * @return filtered data
     */
    function xyFilterMinYValue(data, minRelativeYValue) {
      if (minRelativeYValue === undefined) return data;
      const {
        x,
        y
      } = data;
      const {
        min,
        max
      } = xMinMaxValues(y);
      const threshold = max * minRelativeYValue;
      if (min >= threshold) return data;
      const newX = [];
      const newY = [];
      for (let i = 0; i < x.length; i++) {
        if (y[i] >= threshold) {
          newX.push(x[i]);
          newY.push(y[i]);
        }
      }
      return {
        x: newX,
        y: newY
      };
    }

    /** Function that sorts arrays or Float64Arrays in ascending order
     *
     * @param array - array to sort
     * @returns sorted array
     */
    /** Function that sorts arrays or Float64Arrays in descending order
     *
     * @param array - array to sort
     * @returns sorted array
     */
    function xSortDescending(array) {
      if (array instanceof Float64Array) {
        return array.sort().reverse();
      } else if (Array.isArray(array)) {
        return array.sort((a, b) => b - a);
      }
      throw new Error('Trying to sort non aray');
    }

    /** Filter an array x/y based on various criteria x points are expected to be sorted
     *
     * @param data - object containing 2 properties x and y
     * @param options - options
     * @return filtered data
     */
    function xyFilterTopYValues(data, nbPeaks) {
      if (nbPeaks === undefined) return data;
      if (nbPeaks > data.x.length) return data;
      const {
        x,
        y
      } = data;
      const newX = [];
      const newY = [];
      const descending = xSortDescending(y.slice());
      const threshold = descending[nbPeaks - 1];
      let nbThreshold = 0;
      for (let i = 0; i < nbPeaks; i++) {
        if (descending[i] === threshold) {
          nbThreshold++;
        }
      }
      for (let i = 0; i < x.length; i++) {
        if (y[i] > threshold) {
          newX.push(x[i]);
          newY.push(y[i]);
        } else if (y[i] === threshold) {
          nbThreshold--;
          if (nbThreshold >= 0) {
            newX.push(x[i]);
            newY.push(y[i]);
          }
        }
      }
      return {
        x: newX,
        y: newY
      };
    }

    /**
     * Join x / y values when difference in X is closer than delta.
     * When joining, y values are summed and x values are weighted average
     *
     * @param data - Object that contains property x (an ordered increasing array) and y (an array)
     * @param options - Options
     * @returns - An object with the xyIntegration function
     */
    function xyJoinX(data, options = {}) {
      xyCheck(data);
      const {
        delta = 1
      } = options;
      const deltaIsFunction = typeof delta === 'function';
      const x = Array.from(data.x);
      const y = Array.from(data.y);
      if (x.length < 2) {
        return {
          x,
          y
        };
      }
      let position = 0;
      for (let i = 1; i < x.length; i++) {
        const difference = x[i] - x[i - 1];
        const currentDelta = deltaIsFunction ? delta((x[i] + x[i - 1]) / 2) : delta;
        if (difference <= currentDelta) {
          // we join
          if (y[position] !== 0 || y[i] !== 0) {
            x[position] = (x[position] * y[position] + x[i] * y[i]) / (y[position] + y[i]);
            y[position] += y[i];
          }
        } else {
          position++;
          x[position] = x[i];
          y[position] = y[i];
        }
      }
      x.length = position + 1;
      y.length = position + 1;
      return {
        x,
        y
      };
    }

    /**
     * This function performs a quick sort of the x array while transforming the y array to preserve the coordinates.
     *
     * @param data - Object that contains property x (Array) and y (Array)
     */
    function xySortX(data) {
      const {
        x,
        y
      } = data;
      if (xIsMonotonic(x) && x.length > 1) {
        if (x[0] < x[1]) {
          return {
            x: Float64Array.from(x),
            y: Float64Array.from(y)
          };
        } else {
          return {
            x: Float64Array.from(x).reverse(),
            y: Float64Array.from(y).reverse()
          };
        }
      }
      const xyObject = x.map((val, index) => ({
        x: val,
        y: y[index]
      })).sort((a, b) => a.x - b.x);
      const response = {
        x: new Float64Array(x.length),
        y: new Float64Array(y.length)
      };
      for (let i = 0; i < x.length; i++) {
        response.x[i] = xyObject[i].x;
        response.y[i] = xyObject[i].y;
      }
      return response;
    }

    /**
     * GetSlots.
     *
     * @param data - data.
     * @param options - Options.
     */
    function getSlots(data, options = {}) {
      const {
        delta = 1
      } = options;
      const deltaIsFunction = typeof delta === 'function';
      const possibleXs = Float64Array.from([].concat(...data.map(spectrum => spectrum.x))).sort();
      if (possibleXs.length === 0) {
        throw new Error('xyArrayMerge can not process empty arrays');
      }
      let currentSlot = {
        from: possibleXs[0],
        to: possibleXs[0],
        average: possibleXs[0],
        sum: possibleXs[0],
        number: 1
      };
      const slots = [currentSlot];
      for (let i = 1; i < possibleXs.length; i++) {
        const currentDelta = deltaIsFunction ? delta(possibleXs[i]) : delta;
        if (possibleXs[i] - currentSlot.to <= currentDelta) {
          currentSlot.to = possibleXs[i];
          currentSlot.number++;
          currentSlot.sum += possibleXs[i];
          currentSlot.average = currentSlot.sum / currentSlot.number;
        } else {
          currentSlot = {
            from: possibleXs[i],
            to: possibleXs[i],
            average: possibleXs[i],
            sum: possibleXs[i],
            number: 1
          };
          slots.push(currentSlot);
        }
      }
      return slots;
    }

    /**
     * Aligns data, can be used for spectra
     *
     * @param data - data
     * @param options - Options
     */
    function xyArrayAlign(data, options = {}) {
      const {
        delta = 1,
        requiredY = false
      } = options;
      data = data.map(spectrum => xyJoinX(spectrum, {
        delta
      }));
      const slots = getSlots(data, options);
      const x = Float64Array.from(slots.map(slot => slot.average));
      const ys = new Array(data.length).fill(0).map(() => new Float64Array(x.length));
      const positions = new Uint32Array(data.length);
      for (let i = 0; i < slots.length; i++) {
        const slot = slots[i];
        for (let j = 0; j < data.length; j++) {
          const spectrum = data[j];
          while (positions[j] < spectrum.x.length && spectrum.x[positions[j]] <= slot.to) {
            ys[j][i] += spectrum.y[positions[j]];
            positions[j]++;
          }
        }
      }
      if (requiredY) return filterRequiredY(x, ys);
      return {
        x,
        ys
      };
    }
    function filterRequiredY(x, ys) {
      const newX = [];
      const newYs = new Array(ys.length).fill(0).map(() => []);
      for (let i = 0; i < x.length; i++) {
        if (ys.every(y => y[i] !== 0)) {
          newX.push(x[i]);
          for (let j = 0; j < ys.length; j++) {
            newYs[j].push(ys[j][i]);
          }
        }
      }
      return {
        x: newX,
        ys: newYs
      };
    }

    /**
     * Merge DataXY
     * We have an array of DataXY and the goal is to merge all the values for which the deltaX is small or equal to delta.
     * X values are weighted average
     *
     * @param data - data
     * @param options - Options
     */
    function xyArrayWeightedMerge(data, options = {}) {
      let {
        delta = 1
      } = options;
      if (typeof delta === 'number') {
        const deltaNumber = delta;
        delta = () => deltaNumber;
      }
      data = data.filter(spectrum => spectrum.x.length > 0);
      if (data.length === 0) return {
        x: [],
        y: []
      };
      const x = [];
      const y = [];
      const positions = new Array(data.length).fill(0);
      const point = {
        x: 0,
        y: 0
      };
      nextValue(data, positions, point);
      const slot = {
        maxX: point.x + delta(point.x),
        sumY: point.y,
        sumXY: point.y * point.x
      };
      while (data.length > 0) {
        nextValue(data, positions, point);
        const sameSlot = point.x <= slot.maxX;
        if (!sameSlot) {
          if (slot.sumY > 0) {
            x.push(slot.sumXY / slot.sumY);
            y.push(slot.sumY);
          }
          slot.sumY = 0;
          slot.sumXY = 0;
        }
        slot.sumY += point.y;
        slot.sumXY += point.x * point.y;
        slot.maxX = point.x + delta(point.x);
        if (data.length === 0 && slot.sumY > 0) {
          x.push(slot.sumXY / slot.sumY);
          y.push(slot.sumY);
        }
      }
      return {
        x,
        y
      };
    }
    /**
     * NextValue.
     *
     * @param data - data.
     * @param positions - Positions array.
     * @param point - Point.
     */
    function nextValue(data, positions, point) {
      let minIndex = 0;
      let minX = data[0].x[positions[0]];
      for (let i = 1; i < data.length; i++) {
        const currentX = data[i].x[positions[i]];
        if (currentX < minX) {
          minX = currentX;
          minIndex = i;
        }
      }
      point.x = minX;
      point.y = data[minIndex].y[positions[minIndex]];
      positions[minIndex]++;
      if (positions[minIndex] === data[minIndex].x.length) {
        positions.splice(minIndex, 1);
        data.splice(minIndex, 1);
      }
    }

    /**
     * GetSlotsToFirst.
     *
     * @param data - data.
     * @param options - Options.
     */
    function getSlotsToFirst(data, options = {}) {
      const {
        delta = 1
      } = options;
      const deltaIsFunction = typeof delta === 'function';
      const firstXs = data[0].x;
      const slots = [];
      // we first create the slots based on the first spectrum
      for (const element of firstXs) {
        const currentDelta = deltaIsFunction ? delta(element) : delta;
        slots.push({
          from: element - currentDelta,
          to: element + currentDelta,
          value: element
        });
      }
      const otherXs = xyArrayWeightedMerge(data.slice(1), options).x;
      let currentPosition = 0;
      for (const slot of slots) {
        while (otherXs[currentPosition] < slot.to && currentPosition < otherXs.length) {
          if (otherXs[currentPosition] < slot.from) {
            const currentDelta = deltaIsFunction ? delta(otherXs[currentPosition]) : delta;
            slots.push({
              from: otherXs[currentPosition] - currentDelta,
              to: otherXs[currentPosition] + currentDelta,
              value: otherXs[currentPosition]
            });
          }
          currentPosition++;
        }
      }
      for (let i = currentPosition; i < otherXs.length; i++) {
        const currentDelta = deltaIsFunction ? delta(otherXs[i]) : delta;
        slots.push({
          from: otherXs[i] - currentDelta,
          to: otherXs[i] + currentDelta,
          value: otherXs[i]
        });
      }
      slots.sort((a, b) => a.value - b.value);
      // we prevent slots overlap in the first spectrum
      for (let i = 0; i < slots.length - 1; i++) {
        if (slots[i].to > slots[i + 1].from) {
          const middle = (slots[i].value + slots[i + 1].value) / 2;
          slots[i].to = middle;
          slots[i + 1].from = middle;
        }
      }
      return slots;
    }

    /**
     * We align all the data/spectra to the first array of X.
     * The alignment is based on the X values of the first spectrum and the `delta` error allowed.
     * If some x values are missing in the first spectrum we will add them
     *
     * @param data data
     * @param options options
     */
    function xyArrayAlignToFirst(data, options = {}) {
      const slots = getSlotsToFirst(data, options);
      const x = Float64Array.from(slots.map(slot => slot.value));
      const ys = new Array(data.length).fill(0).map(() => new Float64Array(x.length));
      const positions = new Uint32Array(data.length);
      for (let i = 0; i < slots.length; i++) {
        const slot = slots[i];
        for (let j = 0; j < data.length; j++) {
          const spectrum = data[j];
          while (positions[j] < spectrum.x.length && spectrum.x[positions[j]] < slot.to) {
            ys[j][i] += spectrum.y[positions[j]];
            positions[j]++;
          }
        }
      }
      return {
        x,
        ys
      };
    }

    /**
     * Throw an error in no an object of x,y arrays
     *
     * @param points - list of points
     */
    function xyObjectCheck(points = []) {
      if (!Array.isArray(points)) {
        throw new Error('ArrayPoints must be an array of {x,y} object');
      }
      if (points.length > 0 && (points[0].x === undefined || points[0].y === undefined)) {
        throw new Error('ArrayPoints must be an array of {x,y} object');
      }
    }

    /**
     * Finds the max x value and return a {x,y,index} point
     *
     * @param points - Object that contains property x (an ordered increasing array) and y (an array)
     */
    function xyObjectMaxXPoint(points = []) {
      xyObjectCheck(points);
      if (points.length === 0) return {
        x: 0,
        y: 0
      };
      let current = {
        x: points[0].x,
        y: points[0].y,
        index: 0
      };
      for (let i = 1; i < points.length; i++) {
        if (points[i].x > current.x) {
          current = {
            x: points[i].x,
            y: points[i].y,
            index: i
          };
        }
      }
      return current;
    }

    /**
     * Finds the min x value and return a {x,y,index} point
     *
     * @param points - Object that contains property x (an ordered increasing array) and y (an array)
     */
    function xyObjectMinXPoint(points = []) {
      xyObjectCheck(points);
      if (points.length === 0) return {
        x: 0,
        y: 0
      };
      let current = {
        x: points[0].x,
        y: points[0].y,
        index: 0
      };
      for (let i = 1; i < points.length; i++) {
        if (points[i].x < current.x) {
          current = {
            x: points[i].x,
            y: points[i].y,
            index: i
          };
        }
      }
      return current;
    }

    /**
     * Calculate the sum of Y values
     *
     * @param points - Object that contains property x and y (an array)
     */
    function xyObjectSumY(points = []) {
      xyObjectCheck(points);
      let sum = 0;
      for (const point of points) {
        sum += point.y;
      }
      return sum;
    }

    const LOOP = 8;
    const FLOAT_MUL = 1 / 16777216;
    const sh1 = 15;
    const sh2 = 18;
    const sh3 = 11;
    function multiply_uint32(n, m) {
      n >>>= 0;
      m >>>= 0;
      const nlo = n & 0xffff;
      const nhi = n - nlo;
      return (nhi * m >>> 0) + nlo * m >>> 0;
    }
    class XSadd {
      constructor(seed = Date.now()) {
        this.state = new Uint32Array(4);
        this.init(seed);
        this.random = this.getFloat.bind(this);
      }
      /**
       * Returns a 32-bit integer r (0 <= r < 2^32)
       */
      getUint32() {
        this.nextState();
        return this.state[3] + this.state[2] >>> 0;
      }
      /**
       * Returns a floating point number r (0.0 <= r < 1.0)
       */
      getFloat() {
        return (this.getUint32() >>> 8) * FLOAT_MUL;
      }
      init(seed) {
        if (!Number.isInteger(seed)) {
          throw new TypeError('seed must be an integer');
        }
        this.state[0] = seed;
        this.state[1] = 0;
        this.state[2] = 0;
        this.state[3] = 0;
        for (let i = 1; i < LOOP; i++) {
          this.state[i & 3] ^= i + multiply_uint32(1812433253, this.state[i - 1 & 3] ^ this.state[i - 1 & 3] >>> 30 >>> 0) >>> 0;
        }
        this.periodCertification();
        for (let i = 0; i < LOOP; i++) {
          this.nextState();
        }
      }
      periodCertification() {
        if (this.state[0] === 0 && this.state[1] === 0 && this.state[2] === 0 && this.state[3] === 0) {
          this.state[0] = 88; // X
          this.state[1] = 83; // S
          this.state[2] = 65; // A
          this.state[3] = 68; // D
        }
      }

      nextState() {
        let t = this.state[0];
        t ^= t << sh1;
        t ^= t >>> sh2;
        t ^= this.state[3] << sh3;
        this.state[0] = this.state[1];
        this.state[1] = this.state[2];
        this.state[2] = this.state[3];
        this.state[3] = t;
      }
    }

    /**
     * Create a random array of numbers of a specific length
     *
     * @return - array of random floats normally distributed
     */
    let spare;
    let hasSpare = false;
    function createRandomArray(options = {}) {
      const {
        mean = 0,
        standardDeviation = 1,
        length = 1000,
        range = 1,
        seed,
        distribution = 'normal'
      } = options;
      const generator = new XSadd(seed);
      const returnArray = new Float64Array(length);
      switch (distribution) {
        case 'normal':
          for (let i = 0; i < length; i++) {
            returnArray[i] = generateGaussian(mean, standardDeviation, generator);
          }
          break;
        case 'uniform':
          for (let i = 0; i < length; i++) {
            returnArray[i] = (generator.random() - 0.5) * range + mean;
          }
          break;
        default:
          // eslint-disable-next-line @typescript-eslint/restrict-template-expressions
          throw new Error(`unknown distribution: ${distribution}`);
      }
      return returnArray;
    }
    function generateGaussian(mean, standardDeviation, generator) {
      let val, u, v, s;
      if (hasSpare) {
        hasSpare = false;
        val = spare * standardDeviation + mean;
      } else {
        do {
          u = generator.random() * 2 - 1;
          v = generator.random() * 2 - 1;
          s = u * u + v * v;
        } while (s >= 1 || s === 0);
        s = Math.sqrt(-2 * Math.log(s) / s);
        spare = v * s;
        hasSpare = true;
        val = mean + standardDeviation * u * s;
      }
      return val;
    }

    /*
        https://tools.ietf.org/html/rfc3629

        UTF8-char = UTF8-1 / UTF8-2 / UTF8-3 / UTF8-4

        UTF8-1    = %x00-7F

        UTF8-2    = %xC2-DF UTF8-tail

        UTF8-3    = %xE0 %xA0-BF UTF8-tail
                    %xE1-EC 2( UTF8-tail )
                    %xED %x80-9F UTF8-tail
                    %xEE-EF 2( UTF8-tail )

        UTF8-4    = %xF0 %x90-BF 2( UTF8-tail )
                    %xF1-F3 3( UTF8-tail )
                    %xF4 %x80-8F 2( UTF8-tail )

        UTF8-tail = %x80-BF
    */
    /**
     * Check if a Node.js Buffer or Uint8Array is UTF-8.
     */
    function isUtf8(buf) {
      if (!buf) {
        return false;
      }
      var i = 0;
      var len = buf.length;
      while (i < len) {
        // UTF8-1 = %x00-7F
        if (buf[i] <= 0x7F) {
          i++;
          continue;
        }
        // UTF8-2 = %xC2-DF UTF8-tail
        if (buf[i] >= 0xC2 && buf[i] <= 0xDF) {
          // if(buf[i + 1] >= 0x80 && buf[i + 1] <= 0xBF) {
          if (buf[i + 1] >> 6 === 2) {
            i += 2;
            continue;
          } else {
            return false;
          }
        }
        // UTF8-3 = %xE0 %xA0-BF UTF8-tail
        // UTF8-3 = %xED %x80-9F UTF8-tail
        if ((buf[i] === 0xE0 && buf[i + 1] >= 0xA0 && buf[i + 1] <= 0xBF || buf[i] === 0xED && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x9F) && buf[i + 2] >> 6 === 2) {
          i += 3;
          continue;
        }
        // UTF8-3 = %xE1-EC 2( UTF8-tail )
        // UTF8-3 = %xEE-EF 2( UTF8-tail )
        if ((buf[i] >= 0xE1 && buf[i] <= 0xEC || buf[i] >= 0xEE && buf[i] <= 0xEF) && buf[i + 1] >> 6 === 2 && buf[i + 2] >> 6 === 2) {
          i += 3;
          continue;
        }
        // UTF8-4 = %xF0 %x90-BF 2( UTF8-tail )
        //          %xF1-F3 3( UTF8-tail )
        //          %xF4 %x80-8F 2( UTF8-tail )
        if ((buf[i] === 0xF0 && buf[i + 1] >= 0x90 && buf[i + 1] <= 0xBF || buf[i] >= 0xF1 && buf[i] <= 0xF3 && buf[i + 1] >> 6 === 2 || buf[i] === 0xF4 && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x8F) && buf[i + 2] >> 6 === 2 && buf[i + 3] >> 6 === 2) {
          i += 4;
          continue;
        }
        return false;
      }
      return true;
    }

    /**
     * Ensure that the data is string. If it is an ArrayBuffer it will be converted to string using TextDecoder.
     * @param blob
     * @param options
     * @returns
     */
    function ensureString(blob, options = {}) {
      if (typeof blob === 'string') {
        return blob;
      }
      if (ArrayBuffer.isView(blob) || blob instanceof ArrayBuffer) {
        const {
          encoding = guessEncoding(blob)
        } = options;
        const decoder = new TextDecoder(encoding);
        return decoder.decode(blob);
      }
      throw new TypeError(`blob must be a string, ArrayBuffer or ArrayBufferView`);
    }
    function guessEncoding(blob) {
      const uint8 = ArrayBuffer.isView(blob) ? new Uint8Array(blob.buffer, blob.byteOffset, blob.byteLength) : new Uint8Array(blob);
      if (uint8.length >= 2) {
        if (uint8[0] === 0xfe && uint8[1] === 0xff) {
          return 'utf-16be';
        }
        if (uint8[0] === 0xff && uint8[1] === 0xfe) {
          return 'utf-16le';
        }
      }
      //@ts-expect-error an ArrayBuffer is also ok
      if (!isUtf8(blob)) return 'latin1';
      return 'utf-8';
    }

    /**
     * Filter the array by taking the higher peaks and only
     * keep one per slot.
     * There are 2 different slots, the smallest one will have the
     * extra annotation `close` to true
     * @param {array} peaks - array of all the peaks
     * @param {object} [options={}]
     * @param {number} [options.from] - min X value of the window to consider
     * @param {number} [options.to] - max X value of the window to consider
     * @param {number} [options.searchMonoisotopicRatio=0] - search previous peaks with at least ratio height
     * @param {number} [options.limit=20] - max number of peaks
     * @param {number} [options.threshold=0.01] - minimal intensity compare to base peak
     * @param {number} [options.numberSlots=10] - define the number of slots and indirectly the slot width
     * @param {number} [options.numberCloseSlots=50]
     * @returns {array} - copy of peaks with 'close' annotation
     */

    function getBestPeaks$1(peaks, options = {}) {
      const {
        searchMonoisotopicRatio = 0,
        from = xyObjectMinXPoint(peaks).x,
        to = xyObjectMaxXPoint(peaks).x,
        limit = 20,
        threshold = 0.01,
        numberCloseSlots = 50,
        numberSlots = 10
      } = options;
      let slot = (to - from) / numberSlots;
      let closeSlot = (to - from) / numberCloseSlots;
      let selected = peaks.filter(peak => peak.x >= from && peak.x <= to).map(peak => {
        return {
          peak,
          monoisotopic: false
        };
      });
      if (searchMonoisotopicRatio) {
        selected = selected.sort((a, b) => b.peak.x - a.peak.x);
        for (let i = 0; i < selected.length; i++) {
          let item = selected[i];
          for (let j = i + 1; j < selected.length; j++) {
            let nextItem = selected[j];
            if (item.peak.x - nextItem.peak.x < 0.09) continue;
            if (item.peak.x - nextItem.peak.x > 1.1) break;
            if (nextItem.peak.y > item.peak.y * searchMonoisotopicRatio) {
              item.monoisotopic = false;
              nextItem.monoisotopic = true;
              break;
            }
          }
        }
      }
      selected = selected.sort((a, b) => {
        if (a.monoisotopic && !b.monoisotopic) return -1;
        if (b.monoisotopic && !a.monoisotopic) return 1;
        return b.peak.y - a.peak.y;
      });
      let toReturn = [];
      if (selected.length === 0) return [];
      let minY = selected[0].peak.y * threshold;
      peakLoop: for (let item of selected) {
        if (item.peak.y < minY) {
          if (item.monoisotopic) {
            continue;
          } else {
            break;
          }
        }
        let close = false;
        for (let existing of toReturn) {
          if (Math.abs(existing.x - item.peak.x) < closeSlot) {
            continue peakLoop;
          }
          if (Math.abs(existing.x - item.peak.x) < slot) {
            close = true;
          }
        }
        let newPeak = JSON.parse(JSON.stringify(item.peak));
        newPeak.close = close;
        toReturn.push(newPeak);
        if (toReturn.length === limit) break;
      }
      return toReturn.sort((a, b) => a.x - b.x);
    }

    /**
     * Define static variable corresponding to the various Kinds of a molecular formula part.
     */

    const Kind = {
      BEGIN: 'begin',
      ATOM: 'atom',
      MULTIPLIER_RANGE: 'multiplierRange',
      ISOTOPE: 'isotope',
      ISOTOPE_RATIO: 'isotopeRatio',
      CHARGE: 'charge',
      SALT: 'salt',
      OPENING_PARENTHESIS: 'openingParenthesis',
      CLOSING_PARENTHESIS: 'closingParenthesis',
      PRE_MULTIPLIER: 'preMultiplier',
      MULTIPLIER: 'multiplier',
      TEXT: 'text',
      ANCHOR: 'anchor',
      COMMENT: 'comment'
    };

    /**
     * Parse a string to extract the charge
     * The charge may be in the form --, +++, +3, -2, 4+, 2-
     * @param {*} charge
     */

    function parseCharge(charge) {
      charge = charge.replace(/[()]/g, '');
      let chargeNumber = 0;
      if (charge.match(/^[+-]+$/)) {
        for (let i = 0; i < charge.length; i++) {
          if (charge.charAt(i) === '+') chargeNumber++;else chargeNumber--;
        }
      } else if (charge.match(/^[0-9]+[+-]$/)) {
        chargeNumber = Number(charge.charAt(charge.length - 1) + charge.substring(0, charge.length - 1));
      } else {
        chargeNumber = Number(charge);
      }
      return chargeNumber;
    }

    /**
     * Parse a mf to an array of kind / value
     * @param {String} mf
     */

    function parse(mf) {
      return new MFParser().parse(mf);
    }
    class MFParser {
      parse(mf = '') {
        this.mf = mf;
        this.i = 0;
        this.result = [];
        let lastKind = Kind.BEGIN;
        while (this.i < mf.length) {
          if (this.result.length > 0 && this.result[this.result.length - 1].kind !== Kind.TEXT) {
            lastKind = this.result[this.result.length - 1].kind;
          }
          let char = mf.charAt(this.i);
          let ascii = mf.charCodeAt(this.i);
          let nextAscii = 0;
          if (this.i + 1 < mf.length) nextAscii = mf.charCodeAt(this.i + 1);
          if (ascii > 47 && ascii < 58 || char === '-' && nextAscii > 47 && nextAscii < 58) {
            // a number
            let value = this.getNumber(ascii);
            if (lastKind === Kind.SALT || lastKind === Kind.BEGIN || lastKind === Kind.OPENING_PARENTHESIS) {
              if (value.to) {
                throw new MFError(this.mf, this.i, 'Premultiplier may not contain a -');
              }
              this.result.push({
                kind: Kind.PRE_MULTIPLIER,
                value: value.from
              });
            } else if (lastKind === Kind.ANCHOR) {
              if (value.to) {
                throw new MFError(this.mf, this.i, 'Anchor ID may not contain -');
              }
              this.result[this.result.length - 1].value = value.from;
            } else if (value.to) {
              this.result.push({
                kind: Kind.MULTIPLIER_RANGE,
                value: {
                  from: Math.min(value.from, value.to),
                  to: Math.max(value.from, value.to)
                }
              });
            } else {
              this.result.push({
                kind: Kind.MULTIPLIER,
                value: value.from
              });
            }
            continue;
          } else if (char === '.') {
            // a point
            this.result.push({
              kind: Kind.SALT,
              value: char
            });
            // it is not in a number otherwise it would have been taken before
            // it must be in a salt
          } else if (char === '#') {
            // an anchor
            this.result.push({
              kind: Kind.ANCHOR,
              value: 0
            });
            // it is not in a number otherwise it would have been taken before
            // it must be in a salt
          } else if (ascii > 64 && ascii < 91) {
            // an uppercase = new atom
            let value = this.getAtom(ascii);
            this.result.push({
              kind: Kind.ATOM,
              value
            });
            continue;
          } else if (ascii > 96 && ascii < 123) {
            // a lowercase
            throw new MFError(this.mf, this.i, 'found a lowercase not following an uppercase');
          } else if (char === '(') {
            let charge = this.getParenthesisCharge(ascii);
            if (charge) {
              this.result.push({
                kind: Kind.CHARGE,
                value: charge
              });
            } else {
              this.result.push({
                kind: Kind.OPENING_PARENTHESIS,
                value: '('
              });
            }
          } else if (char === ')') {
            this.result.push({
              kind: Kind.CLOSING_PARENTHESIS,
              value: ')'
            });
          } else if (char === '[') {
            // defines an isotope
            let isotope = this.getIsotope(ascii);
            this.result.push({
              kind: Kind.ISOTOPE,
              value: isotope
            });
          } else if (char === ']') {
            throw new MFError(this.mf, this.i, 'should never meet an closing bracket not in isotopes');
          } else if (char === '{') {
            // can define an exotic isotopic ratio or mixtures of groups
            let isotopeRatio = this.getCurlyBracketIsotopeRatio(ascii);
            if (lastKind === Kind.ATOM) {
              let lastResult = this.result[this.result.length - 1];
              lastResult.kind = Kind.ISOTOPE_RATIO;
              lastResult.value = {
                atom: lastResult.value,
                ratio: isotopeRatio
              };
            } else {
              throw new MFError(this.mf, this.i, 'isotopic composition has to follow an atom');
            }
          } else if (char === '}') {
            throw new MFError(this.mf, this.i, 'found a unexpected closing curly bracket');
          } else if (char === '+') {
            // charge not in parenthesis
            let charge = this.getNonParenthesisCharge(ascii);
            this.result.push({
              kind: Kind.CHARGE,
              value: charge
            });
          } else if (char === '-') {
            // charge not in parenthesis
            let charge = this.getNonParenthesisCharge(ascii);
            this.result.push({
              kind: Kind.CHARGE,
              value: charge
            });
          } else if (char === '$') {
            // it is a comment after
            this.result.push({
              kind: Kind.COMMENT,
              value: this.mf.substring(this.i + 1)
            });
            break;
          } else {
            this.result.push({
              kind: Kind.TEXT,
              value: char
            });
          }
          this.i++;
        }
        this.checkParenthesis();
        return this.result;
      }
      checkParenthesis() {
        let counter = 0;
        for (let line of this.result) {
          if (line.kind === Kind.OPENING_PARENTHESIS) counter++;
          if (line.kind === Kind.CLOSING_PARENTHESIS) counter--;
        }
        if (counter !== 0) {
          throw new MFError(this.mf, this.i, 'number of opening and closing parenthesis not equal');
        }
      }
      getNumber(ascii) {
        let number = '';
        let previous;
        do {
          previous = ascii;
          number += String.fromCharCode(ascii);
          this.i++;
          ascii = this.mf.charCodeAt(this.i);
        } while (ascii > 47 && ascii < 58 || ascii === 46 || ascii === 45 || ascii === 47); // number . - /
        // we need to deal with the case there is a from / to
        if (previous === 46) this.i--;
        let indexOfDash = number.indexOf('-', 1);
        if (indexOfDash > -1) {
          return {
            from: parseNumberWithDivision(number.substr(0, indexOfDash)),
            to: parseNumberWithDivision(number.substr(indexOfDash + 1))
          };
        }
        return {
          from: parseNumberWithDivision(number)
        };
      }
      getAtom(ascii) {
        let atom = '';
        do {
          atom += String.fromCharCode(ascii);
          this.i++;
          ascii = this.mf.charCodeAt(this.i);
        } while (ascii > 96 && ascii < 123);
        return atom;
      }
      getIsotope(ascii) {
        // [13C]
        let substring = '';
        do {
          substring += String.fromCharCode(ascii);
          this.i++;
          ascii = this.mf.charCodeAt(this.i);
        } while (ascii !== 93 && this.i <= this.mf.length);
        let atom = substring.replace(/[^a-zA-Z]/g, '');
        let isotope = Number(substring.replace(/[^0-9]/g, ''));
        return {
          atom,
          isotope
        };
      }
      getCurlyBracketIsotopeRatio(ascii) {
        let substring = '';
        let first = true;
        do {
          if (!first) {
            substring += String.fromCharCode(ascii);
          } else {
            first = false;
          }
          this.i++;
          ascii = this.mf.charCodeAt(this.i);
        } while (ascii !== 125 && this.i <= this.mf.length); // closing curly bracket
        if (substring.match(/^[0-9,]+$/)) {
          return substring.split(',').map(a => Number(a));
        }
        throw new MFError(this.mf, this.i, 'Curly brackets should contain only number and comma');
      }
      getParenthesisCharge(ascii) {
        let substring = '';
        let begin = this.i;
        do {
          substring += String.fromCharCode(ascii);
          this.i++;
          ascii = this.mf.charCodeAt(this.i);
        } while (ascii !== 41 && this.i <= this.mf.length); // closing parenthesis
        if (substring.match(/^\([0-9+-]+$/)) {
          return parseCharge(substring.substring(1));
        } else {
          this.i = begin;
          return undefined;
        }
      }
      getNonParenthesisCharge(ascii) {
        let substring = '';
        do {
          substring += String.fromCharCode(ascii);
          this.i++;
          ascii = this.mf.charCodeAt(this.i);
        } while (ascii === 43 || ascii === 45 || ascii > 47 && ascii < 58);
        this.i--;
        return parseCharge(substring);
      }
    }
    class MFError extends SyntaxError {
      constructor(mf, i, message) {
        let text = `${message}\n\n${mf}\n${' '.repeat(i)}^`;
        super(text);
      }
    }
    function parseNumberWithDivision(string) {
      if (string.includes('/')) {
        let parts = string.split('/');
        if (parts.length !== 2) {
          throw new TypeError('Can not parse MF with number like: ', string);
        }
        return Number(parts[0]) / Number(parts[1]);
      } else {
        return Number(string);
      }
    }

    const superscript = {
      0: '⁰',
      1: '¹',
      2: '²',
      3: '³',
      4: '⁴',
      5: '⁵',
      6: '⁶',
      7: '⁷',
      8: '⁸',
      9: '⁹',
      '+': '⁺',
      '-': '⁻',
      '(': '⁽',
      ')': '⁾',
      '{': '⁽',
      '}': '⁾',
      '.': '˙',
      ',': '˙'
    };
    const subscript = {
      0: '₀',
      1: '₁',
      2: '₂',
      3: '₃',
      4: '₄',
      5: '₅',
      6: '₆',
      7: '₇',
      8: '₈',
      9: '₉',
      '(': '₍',
      ')': '₎',
      '{': '₍',
      '}': '₎',
      '.': ' ',
      ',': ' '
    };

    /**
     * Defines static variables corresponding to the various formatting possibilities
     */

    const Format = {
      SUBSCRIPT: 'subscript',
      SUPERSCRIPT: 'superscript',
      SUPERIMPOSE: 'superimpose',
      TEXT: 'text'
    };

    function formatCharge(charge) {
      if (charge === 1) return '+';
      if (charge > 1) return `+${charge}`;
      if (charge < 0) return String(charge);
      return '';
    }

    /**
     * Converts an array of mf elements to an array of formatting information
     * @param {object[]} lines of the parse method
     */

    function toDisplay(lines) {
      let results = [];
      let result = {};
      for (let line of lines) {
        switch (line.kind) {
          case Kind.MULTIPLIER:
            if (line.value !== 1) {
              result = {
                kind: Format.SUBSCRIPT,
                value: String(line.value)
              };
              results.push(result);
            }
            break;
          case Kind.MULTIPLIER_RANGE:
            result = {
              kind: Format.SUBSCRIPT,
              value: `${String(line.value.from)}-${line.value.to}`
            };
            results.push(result);
            break;
          case Kind.CHARGE:
            if (result.kind === Format.SUBSCRIPT) {
              result.kind = Format.SUPERIMPOSE;
              result.over = formatCharge(line.value);
              result.under = result.value;
              result.value = undefined;
            } else {
              result = {
                kind: Format.SUPERSCRIPT,
                value: formatCharge(line.value)
              };
              results.push(result);
            }
            break;
          case Kind.ISOTOPE:
            result = {
              kind: Format.SUPERSCRIPT,
              value: line.value.isotope
            };
            results.push(result);
            result = {
              kind: Format.TEXT,
              value: line.value.atom
            };
            results.push(result);
            break;
          case Kind.ISOTOPE_RATIO:
            if (result.kind === Format.TEXT) {
              result.value += line.value.atom;
            } else {
              result = {
                kind: Format.TEXT,
                value: line.value.atom
              };
              results.push(result);
            }
            result = {
              kind: Format.SUPERSCRIPT,
              value: `{${line.value.ratio.join(',')}}`
            };
            results.push(result);
            break;
          case Kind.SALT:
            if (result.kind === Format.TEXT) {
              result.value += ' • ';
            } else {
              result = {
                kind: Format.TEXT,
                value: ' • '
              };
              results.push(result);
            }
            break;
          default:
            if (result.kind === Format.TEXT) {
              result.value += line.value;
            } else {
              result = {
                kind: Format.TEXT,
                value: line.value
              };
              results.push(result);
            }
        }
      }
      return results;
    }

    const ELECTRON_MASS = 5.4857990907e-4;

    const elementsAndIsotopes = [{
      number: 1,
      isotopes: [{
        nominal: 1,
        mass: 1.00782503223,
        abundance: 0.999885
      }, {
        nominal: 2,
        mass: 2.01410177812,
        abundance: 0.000115
      }, {
        nominal: 3,
        mass: 3.0160492779
      }, {
        nominal: 4,
        mass: 4.02643
      }, {
        nominal: 5,
        mass: 5.035311
      }, {
        nominal: 6,
        mass: 6.04496
      }, {
        nominal: 7,
        mass: 7.0527
      }],
      symbol: 'H',
      mass: 1.0079407540557772,
      name: 'Hydrogen',
      monoisotopicMass: 1.00782503223
    }, {
      number: 2,
      isotopes: [{
        nominal: 3,
        mass: 3.0160293201,
        abundance: 0.00000134
      }, {
        nominal: 4,
        mass: 4.00260325413,
        abundance: 0.99999866
      }, {
        nominal: 5,
        mass: 5.012057
      }, {
        nominal: 6,
        mass: 6.018885891
      }, {
        nominal: 7,
        mass: 7.0279907
      }, {
        nominal: 8,
        mass: 8.03393439
      }, {
        nominal: 9,
        mass: 9.043946
      }, {
        nominal: 10,
        mass: 10.05279
      }],
      symbol: 'He',
      mass: 4.002601932120929,
      name: 'Helium',
      monoisotopicMass: 4.00260325413
    }, {
      number: 3,
      isotopes: [{
        nominal: 3,
        mass: 3.0308
      }, {
        nominal: 4,
        mass: 4.02719
      }, {
        nominal: 5,
        mass: 5.012538
      }, {
        nominal: 6,
        mass: 6.0151228874,
        abundance: 0.0759
      }, {
        nominal: 7,
        mass: 7.0160034366,
        abundance: 0.9241
      }, {
        nominal: 8,
        mass: 8.022486246
      }, {
        nominal: 9,
        mass: 9.02679019
      }, {
        nominal: 10,
        mass: 10.035483
      }, {
        nominal: 11,
        mass: 11.04372358
      }, {
        nominal: 12,
        mass: 12.052517
      }, {
        nominal: 13,
        mass: 13.06263
      }],
      symbol: 'Li',
      mass: 6.94003660291572,
      name: 'Lithium',
      monoisotopicMass: 7.0160034366
    }, {
      number: 4,
      isotopes: [{
        nominal: 5,
        mass: 5.0399
      }, {
        nominal: 6,
        mass: 6.0197264
      }, {
        nominal: 7,
        mass: 7.016928717
      }, {
        nominal: 8,
        mass: 8.005305102
      }, {
        nominal: 9,
        mass: 9.012183065,
        abundance: 1
      }, {
        nominal: 10,
        mass: 10.013534695
      }, {
        nominal: 11,
        mass: 11.02166108
      }, {
        nominal: 12,
        mass: 12.0269221
      }, {
        nominal: 13,
        mass: 13.036135
      }, {
        nominal: 14,
        mass: 14.04289
      }, {
        nominal: 15,
        mass: 15.05342
      }, {
        nominal: 16,
        mass: 16.06167
      }],
      symbol: 'Be',
      mass: 9.012183065,
      name: 'Beryllium',
      monoisotopicMass: 9.012183065
    }, {
      number: 5,
      isotopes: [{
        nominal: 6,
        mass: 6.0508
      }, {
        nominal: 7,
        mass: 7.029712
      }, {
        nominal: 8,
        mass: 8.0246073
      }, {
        nominal: 9,
        mass: 9.01332965
      }, {
        nominal: 10,
        mass: 10.01293695,
        abundance: 0.199
      }, {
        nominal: 11,
        mass: 11.00930536,
        abundance: 0.801
      }, {
        nominal: 12,
        mass: 12.0143527
      }, {
        nominal: 13,
        mass: 13.0177802
      }, {
        nominal: 14,
        mass: 14.025404
      }, {
        nominal: 15,
        mass: 15.031088
      }, {
        nominal: 16,
        mass: 16.039842
      }, {
        nominal: 17,
        mass: 17.04699
      }, {
        nominal: 18,
        mass: 18.05566
      }, {
        nominal: 19,
        mass: 19.0631
      }, {
        nominal: 20,
        mass: 20.07207
      }, {
        nominal: 21,
        mass: 21.08129
      }],
      symbol: 'B',
      mass: 10.811028046410001,
      name: 'Boron',
      monoisotopicMass: 11.00930536
    }, {
      number: 6,
      isotopes: [{
        nominal: 8,
        mass: 8.037643
      }, {
        nominal: 9,
        mass: 9.0310372
      }, {
        nominal: 10,
        mass: 10.01685331
      }, {
        nominal: 11,
        mass: 11.0114336
      }, {
        nominal: 12,
        mass: 12,
        abundance: 0.9893
      }, {
        nominal: 13,
        mass: 13.00335483507,
        abundance: 0.0107
      }, {
        nominal: 14,
        mass: 14.0032419884
      }, {
        nominal: 15,
        mass: 15.01059926
      }, {
        nominal: 16,
        mass: 16.0147013
      }, {
        nominal: 17,
        mass: 17.022577
      }, {
        nominal: 18,
        mass: 18.026751
      }, {
        nominal: 19,
        mass: 19.0348
      }, {
        nominal: 20,
        mass: 20.04032
      }, {
        nominal: 21,
        mass: 21.049
      }, {
        nominal: 22,
        mass: 22.05753
      }, {
        nominal: 23,
        mass: 23.0689
      }],
      symbol: 'C',
      mass: 12.010735896735248,
      name: 'Carbon',
      monoisotopicMass: 12
    }, {
      number: 7,
      isotopes: [{
        nominal: 10,
        mass: 10.04165
      }, {
        nominal: 11,
        mass: 11.026091
      }, {
        nominal: 12,
        mass: 12.0186132
      }, {
        nominal: 13,
        mass: 13.00573861
      }, {
        nominal: 14,
        mass: 14.00307400443,
        abundance: 0.99636
      }, {
        nominal: 15,
        mass: 15.00010889888,
        abundance: 0.00364
      }, {
        nominal: 16,
        mass: 16.0061019
      }, {
        nominal: 17,
        mass: 17.008449
      }, {
        nominal: 18,
        mass: 18.014078
      }, {
        nominal: 19,
        mass: 19.017022
      }, {
        nominal: 20,
        mass: 20.023366
      }, {
        nominal: 21,
        mass: 21.02711
      }, {
        nominal: 22,
        mass: 22.03439
      }, {
        nominal: 23,
        mass: 23.04114
      }, {
        nominal: 24,
        mass: 24.05039
      }, {
        nominal: 25,
        mass: 25.0601
      }],
      symbol: 'N',
      mass: 14.006703211445798,
      name: 'Nitrogen',
      monoisotopicMass: 14.00307400443
    }, {
      number: 8,
      isotopes: [{
        nominal: 12,
        mass: 12.034262
      }, {
        nominal: 13,
        mass: 13.024815
      }, {
        nominal: 14,
        mass: 14.00859636
      }, {
        nominal: 15,
        mass: 15.00306562
      }, {
        nominal: 16,
        mass: 15.99491461957,
        abundance: 0.99757
      }, {
        nominal: 17,
        mass: 16.9991317565,
        abundance: 0.00038
      }, {
        nominal: 18,
        mass: 17.99915961286,
        abundance: 0.00205
      }, {
        nominal: 19,
        mass: 19.003578
      }, {
        nominal: 20,
        mass: 20.00407535
      }, {
        nominal: 21,
        mass: 21.008655
      }, {
        nominal: 22,
        mass: 22.009966
      }, {
        nominal: 23,
        mass: 23.015696
      }, {
        nominal: 24,
        mass: 24.01986
      }, {
        nominal: 25,
        mass: 25.02936
      }, {
        nominal: 26,
        mass: 26.03729
      }, {
        nominal: 27,
        mass: 27.04772
      }, {
        nominal: 28,
        mass: 28.05591
      }],
      symbol: 'O',
      mass: 15.999404924318277,
      name: 'Oxygen',
      monoisotopicMass: 15.99491461957
    }, {
      number: 9,
      isotopes: [{
        nominal: 14,
        mass: 14.034315
      }, {
        nominal: 15,
        mass: 15.018043
      }, {
        nominal: 16,
        mass: 16.0114657
      }, {
        nominal: 17,
        mass: 17.00209524
      }, {
        nominal: 18,
        mass: 18.00093733
      }, {
        nominal: 19,
        mass: 18.99840316273,
        abundance: 1
      }, {
        nominal: 20,
        mass: 19.999981252
      }, {
        nominal: 21,
        mass: 20.9999489
      }, {
        nominal: 22,
        mass: 22.002999
      }, {
        nominal: 23,
        mass: 23.003557
      }, {
        nominal: 24,
        mass: 24.008115
      }, {
        nominal: 25,
        mass: 25.012199
      }, {
        nominal: 26,
        mass: 26.020038
      }, {
        nominal: 27,
        mass: 27.02644
      }, {
        nominal: 28,
        mass: 28.03534
      }, {
        nominal: 29,
        mass: 29.04254
      }, {
        nominal: 30,
        mass: 30.05165
      }, {
        nominal: 31,
        mass: 31.05971
      }],
      symbol: 'F',
      mass: 18.99840316273,
      name: 'Fluorine',
      monoisotopicMass: 18.99840316273
    }, {
      number: 10,
      isotopes: [{
        nominal: 16,
        mass: 16.02575
      }, {
        nominal: 17,
        mass: 17.01771396
      }, {
        nominal: 18,
        mass: 18.0057087
      }, {
        nominal: 19,
        mass: 19.00188091
      }, {
        nominal: 20,
        mass: 19.9924401762,
        abundance: 0.9048
      }, {
        nominal: 21,
        mass: 20.993846685,
        abundance: 0.0027
      }, {
        nominal: 22,
        mass: 21.991385114,
        abundance: 0.0925
      }, {
        nominal: 23,
        mass: 22.99446691
      }, {
        nominal: 24,
        mass: 23.99361065
      }, {
        nominal: 25,
        mass: 24.997789
      }, {
        nominal: 26,
        mass: 26.000515
      }, {
        nominal: 27,
        mass: 27.007553
      }, {
        nominal: 28,
        mass: 28.01212
      }, {
        nominal: 29,
        mass: 29.01975
      }, {
        nominal: 30,
        mass: 30.02473
      }, {
        nominal: 31,
        mass: 31.0331
      }, {
        nominal: 32,
        mass: 32.03972
      }, {
        nominal: 33,
        mass: 33.04938
      }, {
        nominal: 34,
        mass: 34.05673
      }],
      symbol: 'Ne',
      mass: 20.18004638052026,
      name: 'Neon',
      monoisotopicMass: 19.9924401762
    }, {
      number: 11,
      isotopes: [{
        nominal: 18,
        mass: 18.02688
      }, {
        nominal: 19,
        mass: 19.01388
      }, {
        nominal: 20,
        mass: 20.0073544
      }, {
        nominal: 21,
        mass: 20.99765469
      }, {
        nominal: 22,
        mass: 21.99443741
      }, {
        nominal: 23,
        mass: 22.989769282,
        abundance: 1
      }, {
        nominal: 24,
        mass: 23.99096295
      }, {
        nominal: 25,
        mass: 24.989954
      }, {
        nominal: 26,
        mass: 25.9926346
      }, {
        nominal: 27,
        mass: 26.9940765
      }, {
        nominal: 28,
        mass: 27.998939
      }, {
        nominal: 29,
        mass: 29.0028771
      }, {
        nominal: 30,
        mass: 30.0090979
      }, {
        nominal: 31,
        mass: 31.013163
      }, {
        nominal: 32,
        mass: 32.02019
      }, {
        nominal: 33,
        mass: 33.02573
      }, {
        nominal: 34,
        mass: 34.03359
      }, {
        nominal: 35,
        mass: 35.04062
      }, {
        nominal: 36,
        mass: 36.04929
      }, {
        nominal: 37,
        mass: 37.05705
      }],
      symbol: 'Na',
      mass: 22.989769282,
      name: 'Sodium',
      monoisotopicMass: 22.989769282
    }, {
      number: 12,
      isotopes: [{
        nominal: 19,
        mass: 19.034169
      }, {
        nominal: 20,
        mass: 20.01885
      }, {
        nominal: 21,
        mass: 21.011716
      }, {
        nominal: 22,
        mass: 21.99957065
      }, {
        nominal: 23,
        mass: 22.99412421
      }, {
        nominal: 24,
        mass: 23.985041697,
        abundance: 0.7899
      }, {
        nominal: 25,
        mass: 24.985836976,
        abundance: 0.1
      }, {
        nominal: 26,
        mass: 25.982592968,
        abundance: 0.1101
      }, {
        nominal: 27,
        mass: 26.984340624
      }, {
        nominal: 28,
        mass: 27.9838767
      }, {
        nominal: 29,
        mass: 28.988617
      }, {
        nominal: 30,
        mass: 29.9904629
      }, {
        nominal: 31,
        mass: 30.996648
      }, {
        nominal: 32,
        mass: 31.9991102
      }, {
        nominal: 33,
        mass: 33.0053271
      }, {
        nominal: 34,
        mass: 34.008935
      }, {
        nominal: 35,
        mass: 35.01679
      }, {
        nominal: 36,
        mass: 36.02188
      }, {
        nominal: 37,
        mass: 37.03037
      }, {
        nominal: 38,
        mass: 38.03658
      }, {
        nominal: 39,
        mass: 39.04538
      }, {
        nominal: 40,
        mass: 40.05218
      }],
      symbol: 'Mg',
      mass: 24.3050516198371,
      name: 'Magnesium',
      monoisotopicMass: 23.985041697
    }, {
      number: 13,
      isotopes: [{
        nominal: 21,
        mass: 21.02897
      }, {
        nominal: 22,
        mass: 22.01954
      }, {
        nominal: 23,
        mass: 23.00724435
      }, {
        nominal: 24,
        mass: 23.9999489
      }, {
        nominal: 25,
        mass: 24.9904281
      }, {
        nominal: 26,
        mass: 25.986891904
      }, {
        nominal: 27,
        mass: 26.98153853,
        abundance: 1
      }, {
        nominal: 28,
        mass: 27.98191021
      }, {
        nominal: 29,
        mass: 28.9804565
      }, {
        nominal: 30,
        mass: 29.98296
      }, {
        nominal: 31,
        mass: 30.983945
      }, {
        nominal: 32,
        mass: 31.988085
      }, {
        nominal: 33,
        mass: 32.990909
      }, {
        nominal: 34,
        mass: 33.996705
      }, {
        nominal: 35,
        mass: 34.999764
      }, {
        nominal: 36,
        mass: 36.00639
      }, {
        nominal: 37,
        mass: 37.01053
      }, {
        nominal: 38,
        mass: 38.0174
      }, {
        nominal: 39,
        mass: 39.02254
      }, {
        nominal: 40,
        mass: 40.03003
      }, {
        nominal: 41,
        mass: 41.03638
      }, {
        nominal: 42,
        mass: 42.04384
      }, {
        nominal: 43,
        mass: 43.05147
      }],
      symbol: 'Al',
      mass: 26.98153853,
      name: 'Aluminium',
      monoisotopicMass: 26.98153853
    }, {
      number: 14,
      isotopes: [{
        nominal: 22,
        mass: 22.03579
      }, {
        nominal: 23,
        mass: 23.02544
      }, {
        nominal: 24,
        mass: 24.011535
      }, {
        nominal: 25,
        mass: 25.004109
      }, {
        nominal: 26,
        mass: 25.99233384
      }, {
        nominal: 27,
        mass: 26.98670481
      }, {
        nominal: 28,
        mass: 27.97692653465,
        abundance: 0.92223
      }, {
        nominal: 29,
        mass: 28.9764946649,
        abundance: 0.04685
      }, {
        nominal: 30,
        mass: 29.973770136,
        abundance: 0.03092
      }, {
        nominal: 31,
        mass: 30.975363194
      }, {
        nominal: 32,
        mass: 31.97415154
      }, {
        nominal: 33,
        mass: 32.97797696
      }, {
        nominal: 34,
        mass: 33.978576
      }, {
        nominal: 35,
        mass: 34.984583
      }, {
        nominal: 36,
        mass: 35.986695
      }, {
        nominal: 37,
        mass: 36.992921
      }, {
        nominal: 38,
        mass: 37.995523
      }, {
        nominal: 39,
        mass: 39.002491
      }, {
        nominal: 40,
        mass: 40.00583
      }, {
        nominal: 41,
        mass: 41.01301
      }, {
        nominal: 42,
        mass: 42.01778
      }, {
        nominal: 43,
        mass: 43.0248
      }, {
        nominal: 44,
        mass: 44.03061
      }, {
        nominal: 45,
        mass: 45.03995
      }],
      symbol: 'Si',
      mass: 28.085498705705955,
      name: 'Silicon',
      monoisotopicMass: 27.97692653465
    }, {
      number: 15,
      isotopes: [{
        nominal: 24,
        mass: 24.03577
      }, {
        nominal: 25,
        mass: 25.02119
      }, {
        nominal: 26,
        mass: 26.01178
      }, {
        nominal: 27,
        mass: 26.999224
      }, {
        nominal: 28,
        mass: 27.9923266
      }, {
        nominal: 29,
        mass: 28.98180079
      }, {
        nominal: 30,
        mass: 29.97831375
      }, {
        nominal: 31,
        mass: 30.97376199842,
        abundance: 1
      }, {
        nominal: 32,
        mass: 31.973907643
      }, {
        nominal: 33,
        mass: 32.9717257
      }, {
        nominal: 34,
        mass: 33.97364589
      }, {
        nominal: 35,
        mass: 34.9733141
      }, {
        nominal: 36,
        mass: 35.97826
      }, {
        nominal: 37,
        mass: 36.979607
      }, {
        nominal: 38,
        mass: 37.984252
      }, {
        nominal: 39,
        mass: 38.986227
      }, {
        nominal: 40,
        mass: 39.99133
      }, {
        nominal: 41,
        mass: 40.994654
      }, {
        nominal: 42,
        mass: 42.00108
      }, {
        nominal: 43,
        mass: 43.00502
      }, {
        nominal: 44,
        mass: 44.01121
      }, {
        nominal: 45,
        mass: 45.01645
      }, {
        nominal: 46,
        mass: 46.02446
      }, {
        nominal: 47,
        mass: 47.03139
      }],
      symbol: 'P',
      mass: 30.97376199842,
      name: 'Phosphorus',
      monoisotopicMass: 30.97376199842
    }, {
      number: 16,
      isotopes: [{
        nominal: 26,
        mass: 26.02907
      }, {
        nominal: 27,
        mass: 27.01828
      }, {
        nominal: 28,
        mass: 28.00437
      }, {
        nominal: 29,
        mass: 28.996611
      }, {
        nominal: 30,
        mass: 29.98490703
      }, {
        nominal: 31,
        mass: 30.97955701
      }, {
        nominal: 32,
        mass: 31.9720711744,
        abundance: 0.9499
      }, {
        nominal: 33,
        mass: 32.9714589098,
        abundance: 0.0075
      }, {
        nominal: 34,
        mass: 33.967867004,
        abundance: 0.0425
      }, {
        nominal: 35,
        mass: 34.96903231
      }, {
        nominal: 36,
        mass: 35.96708071,
        abundance: 0.0001
      }, {
        nominal: 37,
        mass: 36.97112551
      }, {
        nominal: 38,
        mass: 37.9711633
      }, {
        nominal: 39,
        mass: 38.975134
      }, {
        nominal: 40,
        mass: 39.9754826
      }, {
        nominal: 41,
        mass: 40.9795935
      }, {
        nominal: 42,
        mass: 41.9810651
      }, {
        nominal: 43,
        mass: 42.9869076
      }, {
        nominal: 44,
        mass: 43.9901188
      }, {
        nominal: 45,
        mass: 44.99572
      }, {
        nominal: 46,
        mass: 46.00004
      }, {
        nominal: 47,
        mass: 47.00795
      }, {
        nominal: 48,
        mass: 48.0137
      }, {
        nominal: 49,
        mass: 49.02276
      }],
      symbol: 'S',
      mass: 32.06478740612706,
      name: 'Sulfur',
      monoisotopicMass: 31.9720711744
    }, {
      number: 17,
      isotopes: [{
        nominal: 28,
        mass: 28.02954
      }, {
        nominal: 29,
        mass: 29.01478
      }, {
        nominal: 30,
        mass: 30.00477
      }, {
        nominal: 31,
        mass: 30.992414
      }, {
        nominal: 32,
        mass: 31.98568464
      }, {
        nominal: 33,
        mass: 32.97745199
      }, {
        nominal: 34,
        mass: 33.973762485
      }, {
        nominal: 35,
        mass: 34.968852682,
        abundance: 0.7576
      }, {
        nominal: 36,
        mass: 35.968306809
      }, {
        nominal: 37,
        mass: 36.965902602,
        abundance: 0.2424
      }, {
        nominal: 38,
        mass: 37.96801044
      }, {
        nominal: 39,
        mass: 38.9680082
      }, {
        nominal: 40,
        mass: 39.970415
      }, {
        nominal: 41,
        mass: 40.970685
      }, {
        nominal: 42,
        mass: 41.97325
      }, {
        nominal: 43,
        mass: 42.97389
      }, {
        nominal: 44,
        mass: 43.97787
      }, {
        nominal: 45,
        mass: 44.98029
      }, {
        nominal: 46,
        mass: 45.98517
      }, {
        nominal: 47,
        mass: 46.98916
      }, {
        nominal: 48,
        mass: 47.99564
      }, {
        nominal: 49,
        mass: 49.00123
      }, {
        nominal: 50,
        mass: 50.00905
      }, {
        nominal: 51,
        mass: 51.01554
      }],
      symbol: 'Cl',
      mass: 35.452937582608,
      name: 'Chlorine',
      monoisotopicMass: 34.968852682
    }, {
      number: 18,
      isotopes: [{
        nominal: 30,
        mass: 30.02307
      }, {
        nominal: 31,
        mass: 31.01212
      }, {
        nominal: 32,
        mass: 31.9976378
      }, {
        nominal: 33,
        mass: 32.98992555
      }, {
        nominal: 34,
        mass: 33.98027009
      }, {
        nominal: 35,
        mass: 34.97525759
      }, {
        nominal: 36,
        mass: 35.967545105,
        abundance: 0.003336
      }, {
        nominal: 37,
        mass: 36.96677633
      }, {
        nominal: 38,
        mass: 37.96273211,
        abundance: 0.000629
      }, {
        nominal: 39,
        mass: 38.964313
      }, {
        nominal: 40,
        mass: 39.9623831237,
        abundance: 0.996035
      }, {
        nominal: 41,
        mass: 40.96450057
      }, {
        nominal: 42,
        mass: 41.9630457
      }, {
        nominal: 43,
        mass: 42.9656361
      }, {
        nominal: 44,
        mass: 43.9649238
      }, {
        nominal: 45,
        mass: 44.96803973
      }, {
        nominal: 46,
        mass: 45.968083
      }, {
        nominal: 47,
        mass: 46.972935
      }, {
        nominal: 48,
        mass: 47.97591
      }, {
        nominal: 49,
        mass: 48.9819
      }, {
        nominal: 50,
        mass: 49.98613
      }, {
        nominal: 51,
        mass: 50.9937
      }, {
        nominal: 52,
        mass: 51.99896
      }, {
        nominal: 53,
        mass: 53.00729
      }],
      symbol: 'Ar',
      mass: 39.947798563582005,
      name: 'Argon',
      monoisotopicMass: 39.9623831237
    }, {
      number: 19,
      isotopes: [{
        nominal: 32,
        mass: 32.02265
      }, {
        nominal: 33,
        mass: 33.00756
      }, {
        nominal: 34,
        mass: 33.99869
      }, {
        nominal: 35,
        mass: 34.98800541
      }, {
        nominal: 36,
        mass: 35.98130201
      }, {
        nominal: 37,
        mass: 36.97337589
      }, {
        nominal: 38,
        mass: 37.96908112
      }, {
        nominal: 39,
        mass: 38.9637064864,
        abundance: 0.932581
      }, {
        nominal: 40,
        mass: 39.963998166,
        abundance: 0.000117
      }, {
        nominal: 41,
        mass: 40.9618252579,
        abundance: 0.067302
      }, {
        nominal: 42,
        mass: 41.96240231
      }, {
        nominal: 43,
        mass: 42.9607347
      }, {
        nominal: 44,
        mass: 43.96158699
      }, {
        nominal: 45,
        mass: 44.96069149
      }, {
        nominal: 46,
        mass: 45.96198159
      }, {
        nominal: 47,
        mass: 46.9616616
      }, {
        nominal: 48,
        mass: 47.96534119
      }, {
        nominal: 49,
        mass: 48.96821075
      }, {
        nominal: 50,
        mass: 49.97238
      }, {
        nominal: 51,
        mass: 50.975828
      }, {
        nominal: 52,
        mass: 51.98224
      }, {
        nominal: 53,
        mass: 52.98746
      }, {
        nominal: 54,
        mass: 53.99463
      }, {
        nominal: 55,
        mass: 55.00076
      }, {
        nominal: 56,
        mass: 56.00851
      }],
      symbol: 'K',
      mass: 39.098300910086,
      name: 'Potassium',
      monoisotopicMass: 38.9637064864
    }, {
      number: 20,
      isotopes: [{
        nominal: 34,
        mass: 34.01487
      }, {
        nominal: 35,
        mass: 35.00514
      }, {
        nominal: 36,
        mass: 35.993074
      }, {
        nominal: 37,
        mass: 36.98589785
      }, {
        nominal: 38,
        mass: 37.97631922
      }, {
        nominal: 39,
        mass: 38.97071081
      }, {
        nominal: 40,
        mass: 39.962590863,
        abundance: 0.96941
      }, {
        nominal: 41,
        mass: 40.96227792
      }, {
        nominal: 42,
        mass: 41.95861783,
        abundance: 0.00647
      }, {
        nominal: 43,
        mass: 42.95876644,
        abundance: 0.00135
      }, {
        nominal: 44,
        mass: 43.95548156,
        abundance: 0.02086
      }, {
        nominal: 45,
        mass: 44.95618635
      }, {
        nominal: 46,
        mass: 45.953689,
        abundance: 0.00004
      }, {
        nominal: 47,
        mass: 46.9545424
      }, {
        nominal: 48,
        mass: 47.95252276,
        abundance: 0.00187
      }, {
        nominal: 49,
        mass: 48.95566274
      }, {
        nominal: 50,
        mass: 49.9574992
      }, {
        nominal: 51,
        mass: 50.960989
      }, {
        nominal: 52,
        mass: 51.963217
      }, {
        nominal: 53,
        mass: 52.96945
      }, {
        nominal: 54,
        mass: 53.9734
      }, {
        nominal: 55,
        mass: 54.9803
      }, {
        nominal: 56,
        mass: 55.98508
      }, {
        nominal: 57,
        mass: 56.99262
      }, {
        nominal: 58,
        mass: 57.99794
      }],
      symbol: 'Ca',
      mass: 40.078022511017735,
      name: 'Calcium',
      monoisotopicMass: 39.962590863
    }, {
      number: 21,
      isotopes: [{
        nominal: 36,
        mass: 36.01648
      }, {
        nominal: 37,
        mass: 37.00374
      }, {
        nominal: 38,
        mass: 37.99512
      }, {
        nominal: 39,
        mass: 38.984785
      }, {
        nominal: 40,
        mass: 39.9779673
      }, {
        nominal: 41,
        mass: 40.969251105
      }, {
        nominal: 42,
        mass: 41.96551653
      }, {
        nominal: 43,
        mass: 42.9611505
      }, {
        nominal: 44,
        mass: 43.9594029
      }, {
        nominal: 45,
        mass: 44.95590828,
        abundance: 1
      }, {
        nominal: 46,
        mass: 45.95516826
      }, {
        nominal: 47,
        mass: 46.9524037
      }, {
        nominal: 48,
        mass: 47.9522236
      }, {
        nominal: 49,
        mass: 48.9500146
      }, {
        nominal: 50,
        mass: 49.952176
      }, {
        nominal: 51,
        mass: 50.953592
      }, {
        nominal: 52,
        mass: 51.95688
      }, {
        nominal: 53,
        mass: 52.95909
      }, {
        nominal: 54,
        mass: 53.96393
      }, {
        nominal: 55,
        mass: 54.96782
      }, {
        nominal: 56,
        mass: 55.97345
      }, {
        nominal: 57,
        mass: 56.97777
      }, {
        nominal: 58,
        mass: 57.98403
      }, {
        nominal: 59,
        mass: 58.98894
      }, {
        nominal: 60,
        mass: 59.99565
      }, {
        nominal: 61,
        mass: 61.001
      }],
      symbol: 'Sc',
      mass: 44.95590828,
      name: 'Scandium',
      monoisotopicMass: 44.95590828
    }, {
      number: 22,
      isotopes: [{
        nominal: 38,
        mass: 38.01145
      }, {
        nominal: 39,
        mass: 39.00236
      }, {
        nominal: 40,
        mass: 39.9905
      }, {
        nominal: 41,
        mass: 40.983148
      }, {
        nominal: 42,
        mass: 41.97304903
      }, {
        nominal: 43,
        mass: 42.9685225
      }, {
        nominal: 44,
        mass: 43.95968995
      }, {
        nominal: 45,
        mass: 44.95812198
      }, {
        nominal: 46,
        mass: 45.95262772,
        abundance: 0.0825
      }, {
        nominal: 47,
        mass: 46.95175879,
        abundance: 0.0744
      }, {
        nominal: 48,
        mass: 47.94794198,
        abundance: 0.7372
      }, {
        nominal: 49,
        mass: 48.94786568,
        abundance: 0.0541
      }, {
        nominal: 50,
        mass: 49.94478689,
        abundance: 0.0518
      }, {
        nominal: 51,
        mass: 50.94661065
      }, {
        nominal: 52,
        mass: 51.946893
      }, {
        nominal: 53,
        mass: 52.94973
      }, {
        nominal: 54,
        mass: 53.95105
      }, {
        nominal: 55,
        mass: 54.95527
      }, {
        nominal: 56,
        mass: 55.95791
      }, {
        nominal: 57,
        mass: 56.96364
      }, {
        nominal: 58,
        mass: 57.9666
      }, {
        nominal: 59,
        mass: 58.97247
      }, {
        nominal: 60,
        mass: 59.97603
      }, {
        nominal: 61,
        mass: 60.98245
      }, {
        nominal: 62,
        mass: 61.98651
      }, {
        nominal: 63,
        mass: 62.99375
      }],
      symbol: 'Ti',
      mass: 47.866744962721995,
      name: 'Titanium',
      monoisotopicMass: 47.94794198
    }, {
      number: 23,
      isotopes: [{
        nominal: 40,
        mass: 40.01276
      }, {
        nominal: 41,
        mass: 41.00021
      }, {
        nominal: 42,
        mass: 41.99182
      }, {
        nominal: 43,
        mass: 42.980766
      }, {
        nominal: 44,
        mass: 43.97411
      }, {
        nominal: 45,
        mass: 44.9657748
      }, {
        nominal: 46,
        mass: 45.96019878
      }, {
        nominal: 47,
        mass: 46.95490491
      }, {
        nominal: 48,
        mass: 47.9522522
      }, {
        nominal: 49,
        mass: 48.9485118
      }, {
        nominal: 50,
        mass: 49.94715601,
        abundance: 0.0025
      }, {
        nominal: 51,
        mass: 50.94395704,
        abundance: 0.9975
      }, {
        nominal: 52,
        mass: 51.94477301
      }, {
        nominal: 53,
        mass: 52.9443367
      }, {
        nominal: 54,
        mass: 53.946439
      }, {
        nominal: 55,
        mass: 54.94724
      }, {
        nominal: 56,
        mass: 55.95048
      }, {
        nominal: 57,
        mass: 56.95252
      }, {
        nominal: 58,
        mass: 57.95672
      }, {
        nominal: 59,
        mass: 58.95939
      }, {
        nominal: 60,
        mass: 59.96431
      }, {
        nominal: 61,
        mass: 60.96725
      }, {
        nominal: 62,
        mass: 61.97265
      }, {
        nominal: 63,
        mass: 62.97639
      }, {
        nominal: 64,
        mass: 63.98264
      }, {
        nominal: 65,
        mass: 64.9875
      }, {
        nominal: 66,
        mass: 65.99398
      }],
      symbol: 'V',
      mass: 50.941465037425004,
      name: 'Vanadium',
      monoisotopicMass: 50.94395704
    }, {
      number: 24,
      isotopes: [{
        nominal: 42,
        mass: 42.0067
      }, {
        nominal: 43,
        mass: 42.99753
      }, {
        nominal: 44,
        mass: 43.98536
      }, {
        nominal: 45,
        mass: 44.97905
      }, {
        nominal: 46,
        mass: 45.968359
      }, {
        nominal: 47,
        mass: 46.9628974
      }, {
        nominal: 48,
        mass: 47.9540291
      }, {
        nominal: 49,
        mass: 48.9513333
      }, {
        nominal: 50,
        mass: 49.94604183,
        abundance: 0.04345
      }, {
        nominal: 51,
        mass: 50.94476502
      }, {
        nominal: 52,
        mass: 51.94050623,
        abundance: 0.83789
      }, {
        nominal: 53,
        mass: 52.94064815,
        abundance: 0.09501
      }, {
        nominal: 54,
        mass: 53.93887916,
        abundance: 0.02365
      }, {
        nominal: 55,
        mass: 54.94083843
      }, {
        nominal: 56,
        mass: 55.9406531
      }, {
        nominal: 57,
        mass: 56.943613
      }, {
        nominal: 58,
        mass: 57.94435
      }, {
        nominal: 59,
        mass: 58.94859
      }, {
        nominal: 60,
        mass: 59.95008
      }, {
        nominal: 61,
        mass: 60.95442
      }, {
        nominal: 62,
        mass: 61.9561
      }, {
        nominal: 63,
        mass: 62.96165
      }, {
        nominal: 64,
        mass: 63.96408
      }, {
        nominal: 65,
        mass: 64.96996
      }, {
        nominal: 66,
        mass: 65.97366
      }, {
        nominal: 67,
        mass: 66.98016
      }, {
        nominal: 68,
        mass: 67.98403
      }],
      symbol: 'Cr',
      mass: 51.9961317554337,
      name: 'Chromium',
      monoisotopicMass: 51.94050623
    }, {
      number: 25,
      isotopes: [{
        nominal: 44,
        mass: 44.00715
      }, {
        nominal: 45,
        mass: 44.99449
      }, {
        nominal: 46,
        mass: 45.98609
      }, {
        nominal: 47,
        mass: 46.975775
      }, {
        nominal: 48,
        mass: 47.96852
      }, {
        nominal: 49,
        mass: 48.959595
      }, {
        nominal: 50,
        mass: 49.95423778
      }, {
        nominal: 51,
        mass: 50.94820847
      }, {
        nominal: 52,
        mass: 51.9455639
      }, {
        nominal: 53,
        mass: 52.94128889
      }, {
        nominal: 54,
        mass: 53.9403576
      }, {
        nominal: 55,
        mass: 54.93804391,
        abundance: 1
      }, {
        nominal: 56,
        mass: 55.93890369
      }, {
        nominal: 57,
        mass: 56.9382861
      }, {
        nominal: 58,
        mass: 57.9400666
      }, {
        nominal: 59,
        mass: 58.9403911
      }, {
        nominal: 60,
        mass: 59.9431366
      }, {
        nominal: 61,
        mass: 60.9444525
      }, {
        nominal: 62,
        mass: 61.94795
      }, {
        nominal: 63,
        mass: 62.9496647
      }, {
        nominal: 64,
        mass: 63.9538494
      }, {
        nominal: 65,
        mass: 64.9560198
      }, {
        nominal: 66,
        mass: 65.960547
      }, {
        nominal: 67,
        mass: 66.96424
      }, {
        nominal: 68,
        mass: 67.96962
      }, {
        nominal: 69,
        mass: 68.97366
      }, {
        nominal: 70,
        mass: 69.97937
      }, {
        nominal: 71,
        mass: 70.98368
      }],
      symbol: 'Mn',
      mass: 54.93804391,
      name: 'Manganese',
      monoisotopicMass: 54.93804391
    }, {
      number: 26,
      isotopes: [{
        nominal: 45,
        mass: 45.01442
      }, {
        nominal: 46,
        mass: 46.00063
      }, {
        nominal: 47,
        mass: 46.99185
      }, {
        nominal: 48,
        mass: 47.98023
      }, {
        nominal: 49,
        mass: 48.973429
      }, {
        nominal: 50,
        mass: 49.962975
      }, {
        nominal: 51,
        mass: 50.956841
      }, {
        nominal: 52,
        mass: 51.9481131
      }, {
        nominal: 53,
        mass: 52.9453064
      }, {
        nominal: 54,
        mass: 53.93960899,
        abundance: 0.05845
      }, {
        nominal: 55,
        mass: 54.93829199
      }, {
        nominal: 56,
        mass: 55.93493633,
        abundance: 0.91754
      }, {
        nominal: 57,
        mass: 56.93539284,
        abundance: 0.02119
      }, {
        nominal: 58,
        mass: 57.93327443,
        abundance: 0.00282
      }, {
        nominal: 59,
        mass: 58.93487434
      }, {
        nominal: 60,
        mass: 59.9340711
      }, {
        nominal: 61,
        mass: 60.9367462
      }, {
        nominal: 62,
        mass: 61.9367918
      }, {
        nominal: 63,
        mass: 62.9402727
      }, {
        nominal: 64,
        mass: 63.9409878
      }, {
        nominal: 65,
        mass: 64.9450115
      }, {
        nominal: 66,
        mass: 65.94625
      }, {
        nominal: 67,
        mass: 66.95054
      }, {
        nominal: 68,
        mass: 67.95295
      }, {
        nominal: 69,
        mass: 68.95807
      }, {
        nominal: 70,
        mass: 69.96102
      }, {
        nominal: 71,
        mass: 70.96672
      }, {
        nominal: 72,
        mass: 71.96983
      }, {
        nominal: 73,
        mass: 72.97572
      }, {
        nominal: 74,
        mass: 73.97935
      }],
      symbol: 'Fe',
      mass: 55.845144433865904,
      name: 'Iron',
      monoisotopicMass: 55.93493633
    }, {
      number: 27,
      isotopes: [{
        nominal: 47,
        mass: 47.01057
      }, {
        nominal: 48,
        mass: 48.00093
      }, {
        nominal: 49,
        mass: 48.98891
      }, {
        nominal: 50,
        mass: 49.98091
      }, {
        nominal: 51,
        mass: 50.970647
      }, {
        nominal: 52,
        mass: 51.96351
      }, {
        nominal: 53,
        mass: 52.9542041
      }, {
        nominal: 54,
        mass: 53.94845987
      }, {
        nominal: 55,
        mass: 54.9419972
      }, {
        nominal: 56,
        mass: 55.9398388
      }, {
        nominal: 57,
        mass: 56.93629057
      }, {
        nominal: 58,
        mass: 57.9357521
      }, {
        nominal: 59,
        mass: 58.93319429,
        abundance: 1
      }, {
        nominal: 60,
        mass: 59.9338163
      }, {
        nominal: 61,
        mass: 60.93247662
      }, {
        nominal: 62,
        mass: 61.934059
      }, {
        nominal: 63,
        mass: 62.9336
      }, {
        nominal: 64,
        mass: 63.935811
      }, {
        nominal: 65,
        mass: 64.9364621
      }, {
        nominal: 66,
        mass: 65.939443
      }, {
        nominal: 67,
        mass: 66.9406096
      }, {
        nominal: 68,
        mass: 67.94426
      }, {
        nominal: 69,
        mass: 68.94614
      }, {
        nominal: 70,
        mass: 69.94963
      }, {
        nominal: 71,
        mass: 70.95237
      }, {
        nominal: 72,
        mass: 71.95729
      }, {
        nominal: 73,
        mass: 72.96039
      }, {
        nominal: 74,
        mass: 73.96515
      }, {
        nominal: 75,
        mass: 74.96876
      }, {
        nominal: 76,
        mass: 75.97413
      }],
      symbol: 'Co',
      mass: 58.93319429,
      name: 'Cobalt',
      monoisotopicMass: 58.93319429
    }, {
      number: 28,
      isotopes: [{
        nominal: 48,
        mass: 48.01769
      }, {
        nominal: 49,
        mass: 49.0077
      }, {
        nominal: 50,
        mass: 49.99474
      }, {
        nominal: 51,
        mass: 50.98611
      }, {
        nominal: 52,
        mass: 51.9748
      }, {
        nominal: 53,
        mass: 52.96819
      }, {
        nominal: 54,
        mass: 53.957892
      }, {
        nominal: 55,
        mass: 54.95133063
      }, {
        nominal: 56,
        mass: 55.94212855
      }, {
        nominal: 57,
        mass: 56.93979218
      }, {
        nominal: 58,
        mass: 57.93534241,
        abundance: 0.68077
      }, {
        nominal: 59,
        mass: 58.9343462
      }, {
        nominal: 60,
        mass: 59.93078588,
        abundance: 0.26223
      }, {
        nominal: 61,
        mass: 60.93105557,
        abundance: 0.011399
      }, {
        nominal: 62,
        mass: 61.92834537,
        abundance: 0.036346
      }, {
        nominal: 63,
        mass: 62.92966963
      }, {
        nominal: 64,
        mass: 63.92796682,
        abundance: 0.009255
      }, {
        nominal: 65,
        mass: 64.93008517
      }, {
        nominal: 66,
        mass: 65.9291393
      }, {
        nominal: 67,
        mass: 66.9315694
      }, {
        nominal: 68,
        mass: 67.9318688
      }, {
        nominal: 69,
        mass: 68.9356103
      }, {
        nominal: 70,
        mass: 69.9364313
      }, {
        nominal: 71,
        mass: 70.940519
      }, {
        nominal: 72,
        mass: 71.9417859
      }, {
        nominal: 73,
        mass: 72.9462067
      }, {
        nominal: 74,
        mass: 73.94798
      }, {
        nominal: 75,
        mass: 74.9525
      }, {
        nominal: 76,
        mass: 75.95533
      }, {
        nominal: 77,
        mass: 76.96055
      }, {
        nominal: 78,
        mass: 77.96336
      }, {
        nominal: 79,
        mass: 78.97025
      }],
      symbol: 'Ni',
      mass: 58.69334710994765,
      name: 'Nickel',
      monoisotopicMass: 57.93534241
    }, {
      number: 29,
      isotopes: [{
        nominal: 52,
        mass: 51.99671
      }, {
        nominal: 53,
        mass: 52.98459
      }, {
        nominal: 54,
        mass: 53.97666
      }, {
        nominal: 55,
        mass: 54.96604
      }, {
        nominal: 56,
        mass: 55.95895
      }, {
        nominal: 57,
        mass: 56.9492125
      }, {
        nominal: 58,
        mass: 57.94453305
      }, {
        nominal: 59,
        mass: 58.93949748
      }, {
        nominal: 60,
        mass: 59.9373645
      }, {
        nominal: 61,
        mass: 60.9334576
      }, {
        nominal: 62,
        mass: 61.93259541
      }, {
        nominal: 63,
        mass: 62.92959772,
        abundance: 0.6915
      }, {
        nominal: 64,
        mass: 63.92976434
      }, {
        nominal: 65,
        mass: 64.9277897,
        abundance: 0.3085
      }, {
        nominal: 66,
        mass: 65.92886903
      }, {
        nominal: 67,
        mass: 66.9277303
      }, {
        nominal: 68,
        mass: 67.9296109
      }, {
        nominal: 69,
        mass: 68.9294293
      }, {
        nominal: 70,
        mass: 69.9323921
      }, {
        nominal: 71,
        mass: 70.9326768
      }, {
        nominal: 72,
        mass: 71.9358203
      }, {
        nominal: 73,
        mass: 72.9366744
      }, {
        nominal: 74,
        mass: 73.9398749
      }, {
        nominal: 75,
        mass: 74.9415226
      }, {
        nominal: 76,
        mass: 75.945275
      }, {
        nominal: 77,
        mass: 76.94792
      }, {
        nominal: 78,
        mass: 77.95223
      }, {
        nominal: 79,
        mass: 78.95502
      }, {
        nominal: 80,
        mass: 79.96089
      }, {
        nominal: 81,
        mass: 80.96587
      }, {
        nominal: 82,
        mass: 81.97244
      }],
      symbol: 'Cu',
      mass: 63.54603994583,
      name: 'Copper',
      monoisotopicMass: 62.92959772
    }, {
      number: 30,
      isotopes: [{
        nominal: 54,
        mass: 53.99204
      }, {
        nominal: 55,
        mass: 54.98398
      }, {
        nominal: 56,
        mass: 55.97254
      }, {
        nominal: 57,
        mass: 56.96506
      }, {
        nominal: 58,
        mass: 57.954591
      }, {
        nominal: 59,
        mass: 58.94931266
      }, {
        nominal: 60,
        mass: 59.9418421
      }, {
        nominal: 61,
        mass: 60.939507
      }, {
        nominal: 62,
        mass: 61.93433397
      }, {
        nominal: 63,
        mass: 62.9332115
      }, {
        nominal: 64,
        mass: 63.92914201,
        abundance: 0.4917
      }, {
        nominal: 65,
        mass: 64.92924077
      }, {
        nominal: 66,
        mass: 65.92603381,
        abundance: 0.2773
      }, {
        nominal: 67,
        mass: 66.92712775,
        abundance: 0.0404
      }, {
        nominal: 68,
        mass: 67.92484455,
        abundance: 0.1845
      }, {
        nominal: 69,
        mass: 68.9265507
      }, {
        nominal: 70,
        mass: 69.9253192,
        abundance: 0.0061
      }, {
        nominal: 71,
        mass: 70.9277196
      }, {
        nominal: 72,
        mass: 71.9268428
      }, {
        nominal: 73,
        mass: 72.9295826
      }, {
        nominal: 74,
        mass: 73.9294073
      }, {
        nominal: 75,
        mass: 74.9328402
      }, {
        nominal: 76,
        mass: 75.933115
      }, {
        nominal: 77,
        mass: 76.9368872
      }, {
        nominal: 78,
        mass: 77.9382892
      }, {
        nominal: 79,
        mass: 78.9426381
      }, {
        nominal: 80,
        mass: 79.9445529
      }, {
        nominal: 81,
        mass: 80.9504026
      }, {
        nominal: 82,
        mass: 81.95426
      }, {
        nominal: 83,
        mass: 82.96056
      }, {
        nominal: 84,
        mass: 83.96521
      }, {
        nominal: 85,
        mass: 84.97226
      }],
      symbol: 'Zn',
      mass: 65.37778252952499,
      name: 'Zinc',
      monoisotopicMass: 63.92914201
    }, {
      number: 31,
      isotopes: [{
        nominal: 56,
        mass: 55.99536
      }, {
        nominal: 57,
        mass: 56.9832
      }, {
        nominal: 58,
        mass: 57.97478
      }, {
        nominal: 59,
        mass: 58.96353
      }, {
        nominal: 60,
        mass: 59.95729
      }, {
        nominal: 61,
        mass: 60.949399
      }, {
        nominal: 62,
        mass: 61.94419025
      }, {
        nominal: 63,
        mass: 62.9392942
      }, {
        nominal: 64,
        mass: 63.9368404
      }, {
        nominal: 65,
        mass: 64.93273459
      }, {
        nominal: 66,
        mass: 65.9315894
      }, {
        nominal: 67,
        mass: 66.9282025
      }, {
        nominal: 68,
        mass: 67.9279805
      }, {
        nominal: 69,
        mass: 68.9255735,
        abundance: 0.60108
      }, {
        nominal: 70,
        mass: 69.9260219
      }, {
        nominal: 71,
        mass: 70.92470258,
        abundance: 0.39892
      }, {
        nominal: 72,
        mass: 71.92636747
      }, {
        nominal: 73,
        mass: 72.9251747
      }, {
        nominal: 74,
        mass: 73.9269457
      }, {
        nominal: 75,
        mass: 74.9265002
      }, {
        nominal: 76,
        mass: 75.9288276
      }, {
        nominal: 77,
        mass: 76.9291543
      }, {
        nominal: 78,
        mass: 77.9316088
      }, {
        nominal: 79,
        mass: 78.9328523
      }, {
        nominal: 80,
        mass: 79.9364208
      }, {
        nominal: 81,
        mass: 80.9381338
      }, {
        nominal: 82,
        mass: 81.9431765
      }, {
        nominal: 83,
        mass: 82.9471203
      }, {
        nominal: 84,
        mass: 83.95246
      }, {
        nominal: 85,
        mass: 84.95699
      }, {
        nominal: 86,
        mass: 85.96301
      }, {
        nominal: 87,
        mass: 86.96824
      }],
      symbol: 'Ga',
      mass: 69.7230660725936,
      name: 'Gallium',
      monoisotopicMass: 68.9255735
    }, {
      number: 32,
      isotopes: [{
        nominal: 58,
        mass: 57.99172
      }, {
        nominal: 59,
        mass: 58.98249
      }, {
        nominal: 60,
        mass: 59.97036
      }, {
        nominal: 61,
        mass: 60.96379
      }, {
        nominal: 62,
        mass: 61.95502
      }, {
        nominal: 63,
        mass: 62.949628
      }, {
        nominal: 64,
        mass: 63.9416899
      }, {
        nominal: 65,
        mass: 64.9393681
      }, {
        nominal: 66,
        mass: 65.9338621
      }, {
        nominal: 67,
        mass: 66.9327339
      }, {
        nominal: 68,
        mass: 67.9280953
      }, {
        nominal: 69,
        mass: 68.9279645
      }, {
        nominal: 70,
        mass: 69.92424875,
        abundance: 0.2057
      }, {
        nominal: 71,
        mass: 70.92495233
      }, {
        nominal: 72,
        mass: 71.922075826,
        abundance: 0.2745
      }, {
        nominal: 73,
        mass: 72.923458956,
        abundance: 0.0775
      }, {
        nominal: 74,
        mass: 73.921177761,
        abundance: 0.365
      }, {
        nominal: 75,
        mass: 74.92285837
      }, {
        nominal: 76,
        mass: 75.921402726,
        abundance: 0.0773
      }, {
        nominal: 77,
        mass: 76.923549843
      }, {
        nominal: 78,
        mass: 77.9228529
      }, {
        nominal: 79,
        mass: 78.92536
      }, {
        nominal: 80,
        mass: 79.9253508
      }, {
        nominal: 81,
        mass: 80.9288329
      }, {
        nominal: 82,
        mass: 81.929774
      }, {
        nominal: 83,
        mass: 82.9345391
      }, {
        nominal: 84,
        mass: 83.9375751
      }, {
        nominal: 85,
        mass: 84.9429697
      }, {
        nominal: 86,
        mass: 85.94658
      }, {
        nominal: 87,
        mass: 86.95268
      }, {
        nominal: 88,
        mass: 87.95691
      }, {
        nominal: 89,
        mass: 88.96379
      }, {
        nominal: 90,
        mass: 89.96863
      }],
      symbol: 'Ge',
      mass: 72.6275501646868,
      name: 'Germanium',
      monoisotopicMass: 73.921177761
    }, {
      number: 33,
      isotopes: [{
        nominal: 60,
        mass: 59.99388
      }, {
        nominal: 61,
        mass: 60.98112
      }, {
        nominal: 62,
        mass: 61.97361
      }, {
        nominal: 63,
        mass: 62.9639
      }, {
        nominal: 64,
        mass: 63.95743
      }, {
        nominal: 65,
        mass: 64.949611
      }, {
        nominal: 66,
        mass: 65.9441488
      }, {
        nominal: 67,
        mass: 66.93925111
      }, {
        nominal: 68,
        mass: 67.9367741
      }, {
        nominal: 69,
        mass: 68.932246
      }, {
        nominal: 70,
        mass: 69.930926
      }, {
        nominal: 71,
        mass: 70.9271138
      }, {
        nominal: 72,
        mass: 71.9267523
      }, {
        nominal: 73,
        mass: 72.9238291
      }, {
        nominal: 74,
        mass: 73.9239286
      }, {
        nominal: 75,
        mass: 74.92159457,
        abundance: 1
      }, {
        nominal: 76,
        mass: 75.92239202
      }, {
        nominal: 77,
        mass: 76.9206476
      }, {
        nominal: 78,
        mass: 77.921828
      }, {
        nominal: 79,
        mass: 78.9209484
      }, {
        nominal: 80,
        mass: 79.9224746
      }, {
        nominal: 81,
        mass: 80.9221323
      }, {
        nominal: 82,
        mass: 81.9247412
      }, {
        nominal: 83,
        mass: 82.9252069
      }, {
        nominal: 84,
        mass: 83.9293033
      }, {
        nominal: 85,
        mass: 84.9321637
      }, {
        nominal: 86,
        mass: 85.9367015
      }, {
        nominal: 87,
        mass: 86.9402917
      }, {
        nominal: 88,
        mass: 87.94555
      }, {
        nominal: 89,
        mass: 88.94976
      }, {
        nominal: 90,
        mass: 89.95563
      }, {
        nominal: 91,
        mass: 90.96039
      }, {
        nominal: 92,
        mass: 91.96674
      }],
      symbol: 'As',
      mass: 74.92159457,
      name: 'Arsenic',
      monoisotopicMass: 74.92159457
    }, {
      number: 34,
      isotopes: [{
        nominal: 64,
        mass: 63.97109
      }, {
        nominal: 65,
        mass: 64.9644
      }, {
        nominal: 66,
        mass: 65.95559
      }, {
        nominal: 67,
        mass: 66.949994
      }, {
        nominal: 68,
        mass: 67.94182524
      }, {
        nominal: 69,
        mass: 68.9394148
      }, {
        nominal: 70,
        mass: 69.9335155
      }, {
        nominal: 71,
        mass: 70.9322094
      }, {
        nominal: 72,
        mass: 71.9271405
      }, {
        nominal: 73,
        mass: 72.9267549
      }, {
        nominal: 74,
        mass: 73.922475934,
        abundance: 0.0089
      }, {
        nominal: 75,
        mass: 74.92252287
      }, {
        nominal: 76,
        mass: 75.919213704,
        abundance: 0.0937
      }, {
        nominal: 77,
        mass: 76.919914154,
        abundance: 0.0763
      }, {
        nominal: 78,
        mass: 77.91730928,
        abundance: 0.2377
      }, {
        nominal: 79,
        mass: 78.91849929
      }, {
        nominal: 80,
        mass: 79.9165218,
        abundance: 0.4961
      }, {
        nominal: 81,
        mass: 80.917993
      }, {
        nominal: 82,
        mass: 81.9166995,
        abundance: 0.0873
      }, {
        nominal: 83,
        mass: 82.9191186
      }, {
        nominal: 84,
        mass: 83.9184668
      }, {
        nominal: 85,
        mass: 84.9222608
      }, {
        nominal: 86,
        mass: 85.9243117
      }, {
        nominal: 87,
        mass: 86.9286886
      }, {
        nominal: 88,
        mass: 87.9314175
      }, {
        nominal: 89,
        mass: 88.9366691
      }, {
        nominal: 90,
        mass: 89.9401
      }, {
        nominal: 91,
        mass: 90.94596
      }, {
        nominal: 92,
        mass: 91.94984
      }, {
        nominal: 93,
        mass: 92.95629
      }, {
        nominal: 94,
        mass: 93.96049
      }, {
        nominal: 95,
        mass: 94.9673
      }],
      symbol: 'Se',
      mass: 78.95938855701361,
      name: 'Selenium',
      monoisotopicMass: 79.9165218
    }, {
      number: 35,
      isotopes: [{
        nominal: 67,
        mass: 66.96465
      }, {
        nominal: 68,
        mass: 67.95873
      }, {
        nominal: 69,
        mass: 68.950497
      }, {
        nominal: 70,
        mass: 69.944792
      }, {
        nominal: 71,
        mass: 70.9393422
      }, {
        nominal: 72,
        mass: 71.9365886
      }, {
        nominal: 73,
        mass: 72.9316715
      }, {
        nominal: 74,
        mass: 73.9299102
      }, {
        nominal: 75,
        mass: 74.9258105
      }, {
        nominal: 76,
        mass: 75.924542
      }, {
        nominal: 77,
        mass: 76.9213792
      }, {
        nominal: 78,
        mass: 77.9211459
      }, {
        nominal: 79,
        mass: 78.9183376,
        abundance: 0.5069
      }, {
        nominal: 80,
        mass: 79.9185298
      }, {
        nominal: 81,
        mass: 80.9162897,
        abundance: 0.4931
      }, {
        nominal: 82,
        mass: 81.9168032
      }, {
        nominal: 83,
        mass: 82.9151756
      }, {
        nominal: 84,
        mass: 83.916496
      }, {
        nominal: 85,
        mass: 84.9156458
      }, {
        nominal: 86,
        mass: 85.9188054
      }, {
        nominal: 87,
        mass: 86.920674
      }, {
        nominal: 88,
        mass: 87.9240833
      }, {
        nominal: 89,
        mass: 88.9267046
      }, {
        nominal: 90,
        mass: 89.9312928
      }, {
        nominal: 91,
        mass: 90.9343986
      }, {
        nominal: 92,
        mass: 91.9396316
      }, {
        nominal: 93,
        mass: 92.94313
      }, {
        nominal: 94,
        mass: 93.9489
      }, {
        nominal: 95,
        mass: 94.95301
      }, {
        nominal: 96,
        mass: 95.95903
      }, {
        nominal: 97,
        mass: 96.96344
      }, {
        nominal: 98,
        mass: 97.96946
      }],
      symbol: 'Br',
      mass: 79.90352778050999,
      name: 'Bromine',
      monoisotopicMass: 78.9183376
    }, {
      number: 36,
      isotopes: [{
        nominal: 69,
        mass: 68.96518
      }, {
        nominal: 70,
        mass: 69.95604
      }, {
        nominal: 71,
        mass: 70.95027
      }, {
        nominal: 72,
        mass: 71.9420924
      }, {
        nominal: 73,
        mass: 72.9392892
      }, {
        nominal: 74,
        mass: 73.933084
      }, {
        nominal: 75,
        mass: 74.9309457
      }, {
        nominal: 76,
        mass: 75.9259103
      }, {
        nominal: 77,
        mass: 76.92467
      }, {
        nominal: 78,
        mass: 77.92036494,
        abundance: 0.00355
      }, {
        nominal: 79,
        mass: 78.9200829
      }, {
        nominal: 80,
        mass: 79.91637808,
        abundance: 0.02286
      }, {
        nominal: 81,
        mass: 80.9165912
      }, {
        nominal: 82,
        mass: 81.91348273,
        abundance: 0.11593
      }, {
        nominal: 83,
        mass: 82.91412716,
        abundance: 0.115
      }, {
        nominal: 84,
        mass: 83.9114977282,
        abundance: 0.56987
      }, {
        nominal: 85,
        mass: 84.9125273
      }, {
        nominal: 86,
        mass: 85.9106106269,
        abundance: 0.17279
      }, {
        nominal: 87,
        mass: 86.91335476
      }, {
        nominal: 88,
        mass: 87.9144479
      }, {
        nominal: 89,
        mass: 88.9178355
      }, {
        nominal: 90,
        mass: 89.9195279
      }, {
        nominal: 91,
        mass: 90.9238063
      }, {
        nominal: 92,
        mass: 91.9261731
      }, {
        nominal: 93,
        mass: 92.9311472
      }, {
        nominal: 94,
        mass: 93.93414
      }, {
        nominal: 95,
        mass: 94.939711
      }, {
        nominal: 96,
        mass: 95.943017
      }, {
        nominal: 97,
        mass: 96.94909
      }, {
        nominal: 98,
        mass: 97.95243
      }, {
        nominal: 99,
        mass: 98.95839
      }, {
        nominal: 100,
        mass: 99.96237
      }, {
        nominal: 101,
        mass: 100.96873
      }],
      symbol: 'Kr',
      mass: 83.7979999953261,
      name: 'Krypton',
      monoisotopicMass: 83.9114977282
    }, {
      number: 37,
      isotopes: [{
        nominal: 71,
        mass: 70.96532
      }, {
        nominal: 72,
        mass: 71.95908
      }, {
        nominal: 73,
        mass: 72.95053
      }, {
        nominal: 74,
        mass: 73.9442659
      }, {
        nominal: 75,
        mass: 74.9385732
      }, {
        nominal: 76,
        mass: 75.935073
      }, {
        nominal: 77,
        mass: 76.9304016
      }, {
        nominal: 78,
        mass: 77.9281419
      }, {
        nominal: 79,
        mass: 78.9239899
      }, {
        nominal: 80,
        mass: 79.9225164
      }, {
        nominal: 81,
        mass: 80.9189939
      }, {
        nominal: 82,
        mass: 81.918209
      }, {
        nominal: 83,
        mass: 82.9151142
      }, {
        nominal: 84,
        mass: 83.9143752
      }, {
        nominal: 85,
        mass: 84.9117897379,
        abundance: 0.7217
      }, {
        nominal: 86,
        mass: 85.91116743
      }, {
        nominal: 87,
        mass: 86.909180531,
        abundance: 0.2783
      }, {
        nominal: 88,
        mass: 87.91131559
      }, {
        nominal: 89,
        mass: 88.9122783
      }, {
        nominal: 90,
        mass: 89.9147985
      }, {
        nominal: 91,
        mass: 90.9165372
      }, {
        nominal: 92,
        mass: 91.9197284
      }, {
        nominal: 93,
        mass: 92.9220393
      }, {
        nominal: 94,
        mass: 93.9263948
      }, {
        nominal: 95,
        mass: 94.92926
      }, {
        nominal: 96,
        mass: 95.9341334
      }, {
        nominal: 97,
        mass: 96.9371771
      }, {
        nominal: 98,
        mass: 97.9416869
      }, {
        nominal: 99,
        mass: 98.94503
      }, {
        nominal: 100,
        mass: 99.95003
      }, {
        nominal: 101,
        mass: 100.95404
      }, {
        nominal: 102,
        mass: 101.95952
      }, {
        nominal: 103,
        mass: 102.96392
      }],
      symbol: 'Rb',
      mass: 85.46766359561973,
      name: 'Rubidium',
      monoisotopicMass: 84.9117897379
    }, {
      number: 38,
      isotopes: [{
        nominal: 73,
        mass: 72.9657
      }, {
        nominal: 74,
        mass: 73.95617
      }, {
        nominal: 75,
        mass: 74.94995
      }, {
        nominal: 76,
        mass: 75.941763
      }, {
        nominal: 77,
        mass: 76.9379455
      }, {
        nominal: 78,
        mass: 77.93218
      }, {
        nominal: 79,
        mass: 78.9297077
      }, {
        nominal: 80,
        mass: 79.9245175
      }, {
        nominal: 81,
        mass: 80.9232114
      }, {
        nominal: 82,
        mass: 81.9183999
      }, {
        nominal: 83,
        mass: 82.9175544
      }, {
        nominal: 84,
        mass: 83.9134191,
        abundance: 0.0056
      }, {
        nominal: 85,
        mass: 84.912932
      }, {
        nominal: 86,
        mass: 85.9092606,
        abundance: 0.0986
      }, {
        nominal: 87,
        mass: 86.9088775,
        abundance: 0.07
      }, {
        nominal: 88,
        mass: 87.9056125,
        abundance: 0.8258
      }, {
        nominal: 89,
        mass: 88.9074511
      }, {
        nominal: 90,
        mass: 89.90773
      }, {
        nominal: 91,
        mass: 90.9101954
      }, {
        nominal: 92,
        mass: 91.9110382
      }, {
        nominal: 93,
        mass: 92.9140242
      }, {
        nominal: 94,
        mass: 93.9153556
      }, {
        nominal: 95,
        mass: 94.9193529
      }, {
        nominal: 96,
        mass: 95.9217066
      }, {
        nominal: 97,
        mass: 96.926374
      }, {
        nominal: 98,
        mass: 97.9286888
      }, {
        nominal: 99,
        mass: 98.9328907
      }, {
        nominal: 100,
        mass: 99.93577
      }, {
        nominal: 101,
        mass: 100.940352
      }, {
        nominal: 102,
        mass: 101.943791
      }, {
        nominal: 103,
        mass: 102.94909
      }, {
        nominal: 104,
        mass: 103.95265
      }, {
        nominal: 105,
        mass: 104.95855
      }, {
        nominal: 106,
        mass: 105.96265
      }, {
        nominal: 107,
        mass: 106.96897
      }],
      symbol: 'Sr',
      mass: 87.61664446962,
      name: 'Strontium',
      monoisotopicMass: 87.9056125
    }, {
      number: 39,
      isotopes: [{
        nominal: 76,
        mass: 75.95856
      }, {
        nominal: 77,
        mass: 76.949781
      }, {
        nominal: 78,
        mass: 77.94361
      }, {
        nominal: 79,
        mass: 78.93735
      }, {
        nominal: 80,
        mass: 79.9343561
      }, {
        nominal: 81,
        mass: 80.9294556
      }, {
        nominal: 82,
        mass: 81.9269314
      }, {
        nominal: 83,
        mass: 82.922485
      }, {
        nominal: 84,
        mass: 83.9206721
      }, {
        nominal: 85,
        mass: 84.916433
      }, {
        nominal: 86,
        mass: 85.914886
      }, {
        nominal: 87,
        mass: 86.9108761
      }, {
        nominal: 88,
        mass: 87.9095016
      }, {
        nominal: 89,
        mass: 88.9058403,
        abundance: 1
      }, {
        nominal: 90,
        mass: 89.9071439
      }, {
        nominal: 91,
        mass: 90.9072974
      }, {
        nominal: 92,
        mass: 91.9089451
      }, {
        nominal: 93,
        mass: 92.909578
      }, {
        nominal: 94,
        mass: 93.9115906
      }, {
        nominal: 95,
        mass: 94.9128161
      }, {
        nominal: 96,
        mass: 95.9158968
      }, {
        nominal: 97,
        mass: 96.9182741
      }, {
        nominal: 98,
        mass: 97.9223821
      }, {
        nominal: 99,
        mass: 98.924148
      }, {
        nominal: 100,
        mass: 99.927715
      }, {
        nominal: 101,
        mass: 100.9301477
      }, {
        nominal: 102,
        mass: 101.9343277
      }, {
        nominal: 103,
        mass: 102.937243
      }, {
        nominal: 104,
        mass: 103.94196
      }, {
        nominal: 105,
        mass: 104.94544
      }, {
        nominal: 106,
        mass: 105.95056
      }, {
        nominal: 107,
        mass: 106.95452
      }, {
        nominal: 108,
        mass: 107.95996
      }, {
        nominal: 109,
        mass: 108.96436
      }],
      symbol: 'Y',
      mass: 88.9058403,
      name: 'Yttrium',
      monoisotopicMass: 88.9058403
    }, {
      number: 40,
      isotopes: [{
        nominal: 78,
        mass: 77.95566
      }, {
        nominal: 79,
        mass: 78.94948
      }, {
        nominal: 80,
        mass: 79.9404
      }, {
        nominal: 81,
        mass: 80.93731
      }, {
        nominal: 82,
        mass: 81.93135
      }, {
        nominal: 83,
        mass: 82.9292421
      }, {
        nominal: 84,
        mass: 83.9233269
      }, {
        nominal: 85,
        mass: 84.9214444
      }, {
        nominal: 86,
        mass: 85.9162972
      }, {
        nominal: 87,
        mass: 86.914818
      }, {
        nominal: 88,
        mass: 87.9102213
      }, {
        nominal: 89,
        mass: 88.9088814
      }, {
        nominal: 90,
        mass: 89.9046977,
        abundance: 0.5145
      }, {
        nominal: 91,
        mass: 90.9056396,
        abundance: 0.1122
      }, {
        nominal: 92,
        mass: 91.9050347,
        abundance: 0.1715
      }, {
        nominal: 93,
        mass: 92.9064699
      }, {
        nominal: 94,
        mass: 93.9063108,
        abundance: 0.1738
      }, {
        nominal: 95,
        mass: 94.9080385
      }, {
        nominal: 96,
        mass: 95.9082714,
        abundance: 0.028
      }, {
        nominal: 97,
        mass: 96.9109512
      }, {
        nominal: 98,
        mass: 97.9127289
      }, {
        nominal: 99,
        mass: 98.916667
      }, {
        nominal: 100,
        mass: 99.9180006
      }, {
        nominal: 101,
        mass: 100.921448
      }, {
        nominal: 102,
        mass: 101.9231409
      }, {
        nominal: 103,
        mass: 102.927191
      }, {
        nominal: 104,
        mass: 103.929436
      }, {
        nominal: 105,
        mass: 104.934008
      }, {
        nominal: 106,
        mass: 105.93676
      }, {
        nominal: 107,
        mass: 106.94174
      }, {
        nominal: 108,
        mass: 107.94487
      }, {
        nominal: 109,
        mass: 108.95041
      }, {
        nominal: 110,
        mass: 109.95396
      }, {
        nominal: 111,
        mass: 110.95968
      }, {
        nominal: 112,
        mass: 111.9637
      }],
      symbol: 'Zr',
      mass: 91.22364159706,
      name: 'Zirconium',
      monoisotopicMass: 89.9046977
    }, {
      number: 41,
      isotopes: [{
        nominal: 81,
        mass: 80.9496
      }, {
        nominal: 82,
        mass: 81.94396
      }, {
        nominal: 83,
        mass: 82.93729
      }, {
        nominal: 84,
        mass: 83.93449
      }, {
        nominal: 85,
        mass: 84.9288458
      }, {
        nominal: 86,
        mass: 85.9257828
      }, {
        nominal: 87,
        mass: 86.9206937
      }, {
        nominal: 88,
        mass: 87.918222
      }, {
        nominal: 89,
        mass: 88.913445
      }, {
        nominal: 90,
        mass: 89.9112584
      }, {
        nominal: 91,
        mass: 90.9069897
      }, {
        nominal: 92,
        mass: 91.9071881
      }, {
        nominal: 93,
        mass: 92.906373,
        abundance: 1
      }, {
        nominal: 94,
        mass: 93.9072788
      }, {
        nominal: 95,
        mass: 94.9068324
      }, {
        nominal: 96,
        mass: 95.9080973
      }, {
        nominal: 97,
        mass: 96.9080959
      }, {
        nominal: 98,
        mass: 97.9103265
      }, {
        nominal: 99,
        mass: 98.911613
      }, {
        nominal: 100,
        mass: 99.9143276
      }, {
        nominal: 101,
        mass: 100.9153103
      }, {
        nominal: 102,
        mass: 101.9180772
      }, {
        nominal: 103,
        mass: 102.9194572
      }, {
        nominal: 104,
        mass: 103.9228925
      }, {
        nominal: 105,
        mass: 104.9249465
      }, {
        nominal: 106,
        mass: 105.9289317
      }, {
        nominal: 107,
        mass: 106.9315937
      }, {
        nominal: 108,
        mass: 107.9360748
      }, {
        nominal: 109,
        mass: 108.93922
      }, {
        nominal: 110,
        mass: 109.94403
      }, {
        nominal: 111,
        mass: 110.94753
      }, {
        nominal: 112,
        mass: 111.95247
      }, {
        nominal: 113,
        mass: 112.95651
      }, {
        nominal: 114,
        mass: 113.96201
      }, {
        nominal: 115,
        mass: 114.96634
      }],
      symbol: 'Nb',
      mass: 92.906373,
      name: 'Niobium',
      monoisotopicMass: 92.906373
    }, {
      number: 42,
      isotopes: [{
        nominal: 83,
        mass: 82.94988
      }, {
        nominal: 84,
        mass: 83.94149
      }, {
        nominal: 85,
        mass: 84.938261
      }, {
        nominal: 86,
        mass: 85.9311748
      }, {
        nominal: 87,
        mass: 86.9281962
      }, {
        nominal: 88,
        mass: 87.9219678
      }, {
        nominal: 89,
        mass: 88.9194682
      }, {
        nominal: 90,
        mass: 89.9139309
      }, {
        nominal: 91,
        mass: 90.9117453
      }, {
        nominal: 92,
        mass: 91.90680796,
        abundance: 0.1453
      }, {
        nominal: 93,
        mass: 92.90680958
      }, {
        nominal: 94,
        mass: 93.9050849,
        abundance: 0.0915
      }, {
        nominal: 95,
        mass: 94.90583877,
        abundance: 0.1584
      }, {
        nominal: 96,
        mass: 95.90467612,
        abundance: 0.1667
      }, {
        nominal: 97,
        mass: 96.90601812,
        abundance: 0.096
      }, {
        nominal: 98,
        mass: 97.90540482,
        abundance: 0.2439
      }, {
        nominal: 99,
        mass: 98.90770851
      }, {
        nominal: 100,
        mass: 99.9074718,
        abundance: 0.0982
      }, {
        nominal: 101,
        mass: 100.9103414
      }, {
        nominal: 102,
        mass: 101.9102834
      }, {
        nominal: 103,
        mass: 102.913079
      }, {
        nominal: 104,
        mass: 103.9137344
      }, {
        nominal: 105,
        mass: 104.916969
      }, {
        nominal: 106,
        mass: 105.918259
      }, {
        nominal: 107,
        mass: 106.922106
      }, {
        nominal: 108,
        mass: 107.924033
      }, {
        nominal: 109,
        mass: 108.928424
      }, {
        nominal: 110,
        mass: 109.930704
      }, {
        nominal: 111,
        mass: 110.935654
      }, {
        nominal: 112,
        mass: 111.93831
      }, {
        nominal: 113,
        mass: 112.94335
      }, {
        nominal: 114,
        mass: 113.94653
      }, {
        nominal: 115,
        mass: 114.95196
      }, {
        nominal: 116,
        mass: 115.95545
      }, {
        nominal: 117,
        mass: 116.96117
      }],
      symbol: 'Mo',
      mass: 95.959788541188,
      name: 'Molybdenum',
      monoisotopicMass: 97.90540482
    }, {
      number: 43,
      isotopes: [{
        nominal: 85,
        mass: 84.95058
      }, {
        nominal: 86,
        mass: 85.94493
      }, {
        nominal: 87,
        mass: 86.9380672
      }, {
        nominal: 88,
        mass: 87.93378
      }, {
        nominal: 89,
        mass: 88.9276487
      }, {
        nominal: 90,
        mass: 89.9240739
      }, {
        nominal: 91,
        mass: 90.9184254
      }, {
        nominal: 92,
        mass: 91.9152698
      }, {
        nominal: 93,
        mass: 92.910246
      }, {
        nominal: 94,
        mass: 93.9096536
      }, {
        nominal: 95,
        mass: 94.9076536
      }, {
        nominal: 96,
        mass: 95.907868
      }, {
        nominal: 97,
        mass: 96.9063667
      }, {
        nominal: 98,
        mass: 97.9072124
      }, {
        nominal: 99,
        mass: 98.9062508
      }, {
        nominal: 100,
        mass: 99.9076539
      }, {
        nominal: 101,
        mass: 100.907309
      }, {
        nominal: 102,
        mass: 101.9092097
      }, {
        nominal: 103,
        mass: 102.909176
      }, {
        nominal: 104,
        mass: 103.911425
      }, {
        nominal: 105,
        mass: 104.911655
      }, {
        nominal: 106,
        mass: 105.914358
      }, {
        nominal: 107,
        mass: 106.9154606
      }, {
        nominal: 108,
        mass: 107.9184957
      }, {
        nominal: 109,
        mass: 108.920256
      }, {
        nominal: 110,
        mass: 109.923744
      }, {
        nominal: 111,
        mass: 110.925901
      }, {
        nominal: 112,
        mass: 111.9299458
      }, {
        nominal: 113,
        mass: 112.932569
      }, {
        nominal: 114,
        mass: 113.93691
      }, {
        nominal: 115,
        mass: 114.93998
      }, {
        nominal: 116,
        mass: 115.94476
      }, {
        nominal: 117,
        mass: 116.94806
      }, {
        nominal: 118,
        mass: 117.95299
      }, {
        nominal: 119,
        mass: 118.95666
      }, {
        nominal: 120,
        mass: 119.96187
      }],
      symbol: 'Tc',
      mass: null,
      name: 'Technetium'
    }, {
      number: 44,
      isotopes: [{
        nominal: 87,
        mass: 86.95069
      }, {
        nominal: 88,
        mass: 87.9416
      }, {
        nominal: 89,
        mass: 88.93762
      }, {
        nominal: 90,
        mass: 89.9303444
      }, {
        nominal: 91,
        mass: 90.9267419
      }, {
        nominal: 92,
        mass: 91.9202344
      }, {
        nominal: 93,
        mass: 92.9171044
      }, {
        nominal: 94,
        mass: 93.9113429
      }, {
        nominal: 95,
        mass: 94.910406
      }, {
        nominal: 96,
        mass: 95.90759025,
        abundance: 0.0554
      }, {
        nominal: 97,
        mass: 96.9075471
      }, {
        nominal: 98,
        mass: 97.9052868,
        abundance: 0.0187
      }, {
        nominal: 99,
        mass: 98.9059341,
        abundance: 0.1276
      }, {
        nominal: 100,
        mass: 99.9042143,
        abundance: 0.126
      }, {
        nominal: 101,
        mass: 100.9055769,
        abundance: 0.1706
      }, {
        nominal: 102,
        mass: 101.9043441,
        abundance: 0.3155
      }, {
        nominal: 103,
        mass: 102.9063186
      }, {
        nominal: 104,
        mass: 103.9054275,
        abundance: 0.1862
      }, {
        nominal: 105,
        mass: 104.9077476
      }, {
        nominal: 106,
        mass: 105.9073291
      }, {
        nominal: 107,
        mass: 106.909972
      }, {
        nominal: 108,
        mass: 107.910188
      }, {
        nominal: 109,
        mass: 108.913326
      }, {
        nominal: 110,
        mass: 109.9140407
      }, {
        nominal: 111,
        mass: 110.91757
      }, {
        nominal: 112,
        mass: 111.918809
      }, {
        nominal: 113,
        mass: 112.922844
      }, {
        nominal: 114,
        mass: 113.9246136
      }, {
        nominal: 115,
        mass: 114.92882
      }, {
        nominal: 116,
        mass: 115.9312192
      }, {
        nominal: 117,
        mass: 116.9361
      }, {
        nominal: 118,
        mass: 117.93853
      }, {
        nominal: 119,
        mass: 118.94357
      }, {
        nominal: 120,
        mass: 119.94631
      }, {
        nominal: 121,
        mass: 120.95164
      }, {
        nominal: 122,
        mass: 121.95447
      }, {
        nominal: 123,
        mass: 122.95989
      }, {
        nominal: 124,
        mass: 123.96305
      }],
      symbol: 'Ru',
      mass: 101.06494013916,
      name: 'Ruthenium',
      monoisotopicMass: 101.9043441
    }, {
      number: 45,
      isotopes: [{
        nominal: 89,
        mass: 88.95058
      }, {
        nominal: 90,
        mass: 89.94422
      }, {
        nominal: 91,
        mass: 90.93688
      }, {
        nominal: 92,
        mass: 91.9323677
      }, {
        nominal: 93,
        mass: 92.9259128
      }, {
        nominal: 94,
        mass: 93.9217305
      }, {
        nominal: 95,
        mass: 94.9158979
      }, {
        nominal: 96,
        mass: 95.914453
      }, {
        nominal: 97,
        mass: 96.911329
      }, {
        nominal: 98,
        mass: 97.910708
      }, {
        nominal: 99,
        mass: 98.9081282
      }, {
        nominal: 100,
        mass: 99.908117
      }, {
        nominal: 101,
        mass: 100.9061606
      }, {
        nominal: 102,
        mass: 101.9068374
      }, {
        nominal: 103,
        mass: 102.905498,
        abundance: 1
      }, {
        nominal: 104,
        mass: 103.9066492
      }, {
        nominal: 105,
        mass: 104.9056885
      }, {
        nominal: 106,
        mass: 105.9072868
      }, {
        nominal: 107,
        mass: 106.906748
      }, {
        nominal: 108,
        mass: 107.908714
      }, {
        nominal: 109,
        mass: 108.9087488
      }, {
        nominal: 110,
        mass: 109.911079
      }, {
        nominal: 111,
        mass: 110.9116423
      }, {
        nominal: 112,
        mass: 111.914403
      }, {
        nominal: 113,
        mass: 112.9154393
      }, {
        nominal: 114,
        mass: 113.918718
      }, {
        nominal: 115,
        mass: 114.9203116
      }, {
        nominal: 116,
        mass: 115.924059
      }, {
        nominal: 117,
        mass: 116.9260354
      }, {
        nominal: 118,
        mass: 117.93034
      }, {
        nominal: 119,
        mass: 118.932557
      }, {
        nominal: 120,
        mass: 119.93686
      }, {
        nominal: 121,
        mass: 120.93942
      }, {
        nominal: 122,
        mass: 121.94399
      }, {
        nominal: 123,
        mass: 122.94685
      }, {
        nominal: 124,
        mass: 123.95151
      }, {
        nominal: 125,
        mass: 124.95469
      }, {
        nominal: 126,
        mass: 125.95946
      }],
      symbol: 'Rh',
      mass: 102.905498,
      name: 'Rhodium',
      monoisotopicMass: 102.905498
    }, {
      number: 46,
      isotopes: [{
        nominal: 91,
        mass: 90.95032
      }, {
        nominal: 92,
        mass: 91.94088
      }, {
        nominal: 93,
        mass: 92.93651
      }, {
        nominal: 94,
        mass: 93.9290376
      }, {
        nominal: 95,
        mass: 94.9248898
      }, {
        nominal: 96,
        mass: 95.9182151
      }, {
        nominal: 97,
        mass: 96.916472
      }, {
        nominal: 98,
        mass: 97.9126983
      }, {
        nominal: 99,
        mass: 98.9117748
      }, {
        nominal: 100,
        mass: 99.908505
      }, {
        nominal: 101,
        mass: 100.9082864
      }, {
        nominal: 102,
        mass: 101.9056022,
        abundance: 0.0102
      }, {
        nominal: 103,
        mass: 102.9060809
      }, {
        nominal: 104,
        mass: 103.9040305,
        abundance: 0.1114
      }, {
        nominal: 105,
        mass: 104.9050796,
        abundance: 0.2233
      }, {
        nominal: 106,
        mass: 105.9034804,
        abundance: 0.2733
      }, {
        nominal: 107,
        mass: 106.9051282
      }, {
        nominal: 108,
        mass: 107.9038916,
        abundance: 0.2646
      }, {
        nominal: 109,
        mass: 108.9059504
      }, {
        nominal: 110,
        mass: 109.9051722,
        abundance: 0.1172
      }, {
        nominal: 111,
        mass: 110.90768968
      }, {
        nominal: 112,
        mass: 111.9073297
      }, {
        nominal: 113,
        mass: 112.910261
      }, {
        nominal: 114,
        mass: 113.9103686
      }, {
        nominal: 115,
        mass: 114.913659
      }, {
        nominal: 116,
        mass: 115.914297
      }, {
        nominal: 117,
        mass: 116.9179547
      }, {
        nominal: 118,
        mass: 117.9190667
      }, {
        nominal: 119,
        mass: 118.9233402
      }, {
        nominal: 120,
        mass: 119.9245511
      }, {
        nominal: 121,
        mass: 120.9289503
      }, {
        nominal: 122,
        mass: 121.930632
      }, {
        nominal: 123,
        mass: 122.93514
      }, {
        nominal: 124,
        mass: 123.93714
      }, {
        nominal: 125,
        mass: 124.94179
      }, {
        nominal: 126,
        mass: 125.94416
      }, {
        nominal: 127,
        mass: 126.94907
      }, {
        nominal: 128,
        mass: 127.95183
      }],
      symbol: 'Pd',
      mass: 106.41532750734,
      name: 'Palladium',
      monoisotopicMass: 105.9034804
    }, {
      number: 47,
      isotopes: [{
        nominal: 93,
        mass: 92.95033
      }, {
        nominal: 94,
        mass: 93.94373
      }, {
        nominal: 95,
        mass: 94.93602
      }, {
        nominal: 96,
        mass: 95.930744
      }, {
        nominal: 97,
        mass: 96.92397
      }, {
        nominal: 98,
        mass: 97.92156
      }, {
        nominal: 99,
        mass: 98.9176458
      }, {
        nominal: 100,
        mass: 99.9161154
      }, {
        nominal: 101,
        mass: 100.912684
      }, {
        nominal: 102,
        mass: 101.9117047
      }, {
        nominal: 103,
        mass: 102.9089631
      }, {
        nominal: 104,
        mass: 103.9086239
      }, {
        nominal: 105,
        mass: 104.9065256
      }, {
        nominal: 106,
        mass: 105.9066636
      }, {
        nominal: 107,
        mass: 106.9050916,
        abundance: 0.51839
      }, {
        nominal: 108,
        mass: 107.9059503
      }, {
        nominal: 109,
        mass: 108.9047553,
        abundance: 0.48161
      }, {
        nominal: 110,
        mass: 109.9061102
      }, {
        nominal: 111,
        mass: 110.9052959
      }, {
        nominal: 112,
        mass: 111.9070486
      }, {
        nominal: 113,
        mass: 112.906573
      }, {
        nominal: 114,
        mass: 113.908823
      }, {
        nominal: 115,
        mass: 114.908767
      }, {
        nominal: 116,
        mass: 115.9113868
      }, {
        nominal: 117,
        mass: 116.911774
      }, {
        nominal: 118,
        mass: 117.9145955
      }, {
        nominal: 119,
        mass: 118.91557
      }, {
        nominal: 120,
        mass: 119.9187848
      }, {
        nominal: 121,
        mass: 120.920125
      }, {
        nominal: 122,
        mass: 121.923664
      }, {
        nominal: 123,
        mass: 122.925337
      }, {
        nominal: 124,
        mass: 123.92893
      }, {
        nominal: 125,
        mass: 124.93105
      }, {
        nominal: 126,
        mass: 125.93475
      }, {
        nominal: 127,
        mass: 126.93711
      }, {
        nominal: 128,
        mass: 127.94106
      }, {
        nominal: 129,
        mass: 128.94395
      }, {
        nominal: 130,
        mass: 129.9507
      }],
      symbol: 'Ag',
      mass: 107.868149634557,
      name: 'Silver',
      monoisotopicMass: 106.9050916
    }, {
      number: 48,
      isotopes: [{
        nominal: 95,
        mass: 94.94994
      }, {
        nominal: 96,
        mass: 95.94034
      }, {
        nominal: 97,
        mass: 96.9351
      }, {
        nominal: 98,
        mass: 97.927389
      }, {
        nominal: 99,
        mass: 98.9249258
      }, {
        nominal: 100,
        mass: 99.9203488
      }, {
        nominal: 101,
        mass: 100.9185862
      }, {
        nominal: 102,
        mass: 101.914482
      }, {
        nominal: 103,
        mass: 102.9134165
      }, {
        nominal: 104,
        mass: 103.9098564
      }, {
        nominal: 105,
        mass: 104.9094639
      }, {
        nominal: 106,
        mass: 105.9064599,
        abundance: 0.0125
      }, {
        nominal: 107,
        mass: 106.9066121
      }, {
        nominal: 108,
        mass: 107.9041834,
        abundance: 0.0089
      }, {
        nominal: 109,
        mass: 108.9049867
      }, {
        nominal: 110,
        mass: 109.90300661,
        abundance: 0.1249
      }, {
        nominal: 111,
        mass: 110.90418287,
        abundance: 0.128
      }, {
        nominal: 112,
        mass: 111.90276287,
        abundance: 0.2413
      }, {
        nominal: 113,
        mass: 112.90440813,
        abundance: 0.1222
      }, {
        nominal: 114,
        mass: 113.90336509,
        abundance: 0.2873
      }, {
        nominal: 115,
        mass: 114.90543751
      }, {
        nominal: 116,
        mass: 115.90476315,
        abundance: 0.0749
      }, {
        nominal: 117,
        mass: 116.907226
      }, {
        nominal: 118,
        mass: 117.906922
      }, {
        nominal: 119,
        mass: 118.909847
      }, {
        nominal: 120,
        mass: 119.9098681
      }, {
        nominal: 121,
        mass: 120.9129637
      }, {
        nominal: 122,
        mass: 121.9134591
      }, {
        nominal: 123,
        mass: 122.9168925
      }, {
        nominal: 124,
        mass: 123.9176574
      }, {
        nominal: 125,
        mass: 124.9212576
      }, {
        nominal: 126,
        mass: 125.9224291
      }, {
        nominal: 127,
        mass: 126.926472
      }, {
        nominal: 128,
        mass: 127.9278129
      }, {
        nominal: 129,
        mass: 128.93182
      }, {
        nominal: 130,
        mass: 129.93394
      }, {
        nominal: 131,
        mass: 130.9406
      }, {
        nominal: 132,
        mass: 131.94604
      }, {
        nominal: 133,
        mass: 132.95285
      }],
      symbol: 'Cd',
      mass: 112.411557818268,
      name: 'Cadmium',
      monoisotopicMass: 113.90336509
    }, {
      number: 49,
      isotopes: [{
        nominal: 97,
        mass: 96.94934
      }, {
        nominal: 98,
        mass: 97.94214
      }, {
        nominal: 99,
        mass: 98.93411
      }, {
        nominal: 100,
        mass: 99.93096
      }, {
        nominal: 101,
        mass: 100.92634
      }, {
        nominal: 102,
        mass: 101.9241071
      }, {
        nominal: 103,
        mass: 102.9198819
      }, {
        nominal: 104,
        mass: 103.9182145
      }, {
        nominal: 105,
        mass: 104.914502
      }, {
        nominal: 106,
        mass: 105.913464
      }, {
        nominal: 107,
        mass: 106.91029
      }, {
        nominal: 108,
        mass: 107.9096935
      }, {
        nominal: 109,
        mass: 108.9071514
      }, {
        nominal: 110,
        mass: 109.90717
      }, {
        nominal: 111,
        mass: 110.9051085
      }, {
        nominal: 112,
        mass: 111.9055377
      }, {
        nominal: 113,
        mass: 112.90406184,
        abundance: 0.0429
      }, {
        nominal: 114,
        mass: 113.90491791
      }, {
        nominal: 115,
        mass: 114.903878776,
        abundance: 0.9571
      }, {
        nominal: 116,
        mass: 115.90525999
      }, {
        nominal: 117,
        mass: 116.9045157
      }, {
        nominal: 118,
        mass: 117.9063566
      }, {
        nominal: 119,
        mass: 118.9058507
      }, {
        nominal: 120,
        mass: 119.907967
      }, {
        nominal: 121,
        mass: 120.907851
      }, {
        nominal: 122,
        mass: 121.910281
      }, {
        nominal: 123,
        mass: 122.910434
      }, {
        nominal: 124,
        mass: 123.913182
      }, {
        nominal: 125,
        mass: 124.913605
      }, {
        nominal: 126,
        mass: 125.916507
      }, {
        nominal: 127,
        mass: 126.917446
      }, {
        nominal: 128,
        mass: 127.9204
      }, {
        nominal: 129,
        mass: 128.9218053
      }, {
        nominal: 130,
        mass: 129.924977
      }, {
        nominal: 131,
        mass: 130.9269715
      }, {
        nominal: 132,
        mass: 131.933001
      }, {
        nominal: 133,
        mass: 132.93831
      }, {
        nominal: 134,
        mass: 133.94454
      }, {
        nominal: 135,
        mass: 134.95005
      }],
      symbol: 'In',
      mass: 114.81808662944559,
      name: 'Indium',
      monoisotopicMass: 114.903878776
    }, {
      number: 50,
      isotopes: [{
        nominal: 99,
        mass: 98.94853
      }, {
        nominal: 100,
        mass: 99.9385
      }, {
        nominal: 101,
        mass: 100.93526
      }, {
        nominal: 102,
        mass: 101.93029
      }, {
        nominal: 103,
        mass: 102.928105
      }, {
        nominal: 104,
        mass: 103.9231052
      }, {
        nominal: 105,
        mass: 104.9212684
      }, {
        nominal: 106,
        mass: 105.9169574
      }, {
        nominal: 107,
        mass: 106.9157137
      }, {
        nominal: 108,
        mass: 107.9118943
      }, {
        nominal: 109,
        mass: 108.9112921
      }, {
        nominal: 110,
        mass: 109.907845
      }, {
        nominal: 111,
        mass: 110.9077401
      }, {
        nominal: 112,
        mass: 111.90482387,
        abundance: 0.0097
      }, {
        nominal: 113,
        mass: 112.9051757
      }, {
        nominal: 114,
        mass: 113.9027827,
        abundance: 0.0066
      }, {
        nominal: 115,
        mass: 114.903344699,
        abundance: 0.0034
      }, {
        nominal: 116,
        mass: 115.9017428,
        abundance: 0.1454
      }, {
        nominal: 117,
        mass: 116.90295398,
        abundance: 0.0768
      }, {
        nominal: 118,
        mass: 117.90160657,
        abundance: 0.2422
      }, {
        nominal: 119,
        mass: 118.90331117,
        abundance: 0.0859
      }, {
        nominal: 120,
        mass: 119.90220163,
        abundance: 0.3258
      }, {
        nominal: 121,
        mass: 120.9042426
      }, {
        nominal: 122,
        mass: 121.9034438,
        abundance: 0.0463
      }, {
        nominal: 123,
        mass: 122.9057252
      }, {
        nominal: 124,
        mass: 123.9052766,
        abundance: 0.0579
      }, {
        nominal: 125,
        mass: 124.9077864
      }, {
        nominal: 126,
        mass: 125.907659
      }, {
        nominal: 127,
        mass: 126.91039
      }, {
        nominal: 128,
        mass: 127.910507
      }, {
        nominal: 129,
        mass: 128.913465
      }, {
        nominal: 130,
        mass: 129.9139738
      }, {
        nominal: 131,
        mass: 130.917045
      }, {
        nominal: 132,
        mass: 131.9178267
      }, {
        nominal: 133,
        mass: 132.9239134
      }, {
        nominal: 134,
        mass: 133.9286821
      }, {
        nominal: 135,
        mass: 134.9349086
      }, {
        nominal: 136,
        mass: 135.93999
      }, {
        nominal: 137,
        mass: 136.94655
      }, {
        nominal: 138,
        mass: 137.95184
      }],
      symbol: 'Sn',
      mass: 118.71011259301059,
      name: 'Tin',
      monoisotopicMass: 119.90220163
    }, {
      number: 51,
      isotopes: [{
        nominal: 103,
        mass: 102.93969
      }, {
        nominal: 104,
        mass: 103.93648
      }, {
        nominal: 105,
        mass: 104.931276
      }, {
        nominal: 106,
        mass: 105.928638
      }, {
        nominal: 107,
        mass: 106.9241506
      }, {
        nominal: 108,
        mass: 107.9222267
      }, {
        nominal: 109,
        mass: 108.9181411
      }, {
        nominal: 110,
        mass: 109.9168543
      }, {
        nominal: 111,
        mass: 110.9132182
      }, {
        nominal: 112,
        mass: 111.9124
      }, {
        nominal: 113,
        mass: 112.909375
      }, {
        nominal: 114,
        mass: 113.90929
      }, {
        nominal: 115,
        mass: 114.906598
      }, {
        nominal: 116,
        mass: 115.9067931
      }, {
        nominal: 117,
        mass: 116.9048415
      }, {
        nominal: 118,
        mass: 117.9055321
      }, {
        nominal: 119,
        mass: 118.9039455
      }, {
        nominal: 120,
        mass: 119.9050794
      }, {
        nominal: 121,
        mass: 120.903812,
        abundance: 0.5721
      }, {
        nominal: 122,
        mass: 121.9051699
      }, {
        nominal: 123,
        mass: 122.9042132,
        abundance: 0.4279
      }, {
        nominal: 124,
        mass: 123.905935
      }, {
        nominal: 125,
        mass: 124.905253
      }, {
        nominal: 126,
        mass: 125.907253
      }, {
        nominal: 127,
        mass: 126.9069243
      }, {
        nominal: 128,
        mass: 127.909146
      }, {
        nominal: 129,
        mass: 128.909147
      }, {
        nominal: 130,
        mass: 129.911662
      }, {
        nominal: 131,
        mass: 130.9119888
      }, {
        nominal: 132,
        mass: 131.9145077
      }, {
        nominal: 133,
        mass: 132.9152732
      }, {
        nominal: 134,
        mass: 133.9205357
      }, {
        nominal: 135,
        mass: 134.9251851
      }, {
        nominal: 136,
        mass: 135.9307459
      }, {
        nominal: 137,
        mass: 136.93555
      }, {
        nominal: 138,
        mass: 137.94145
      }, {
        nominal: 139,
        mass: 138.94655
      }, {
        nominal: 140,
        mass: 139.95283
      }],
      symbol: 'Sb',
      mass: 121.75978367348,
      name: 'Antimony',
      monoisotopicMass: 120.903812
    }, {
      number: 52,
      isotopes: [{
        nominal: 105,
        mass: 104.9433
      }, {
        nominal: 106,
        mass: 105.9375
      }, {
        nominal: 107,
        mass: 106.935012
      }, {
        nominal: 108,
        mass: 107.9293805
      }, {
        nominal: 109,
        mass: 108.9273045
      }, {
        nominal: 110,
        mass: 109.9224581
      }, {
        nominal: 111,
        mass: 110.9210006
      }, {
        nominal: 112,
        mass: 111.9167279
      }, {
        nominal: 113,
        mass: 112.915891
      }, {
        nominal: 114,
        mass: 113.912089
      }, {
        nominal: 115,
        mass: 114.911902
      }, {
        nominal: 116,
        mass: 115.90846
      }, {
        nominal: 117,
        mass: 116.908646
      }, {
        nominal: 118,
        mass: 117.905854
      }, {
        nominal: 119,
        mass: 118.9064071
      }, {
        nominal: 120,
        mass: 119.9040593,
        abundance: 0.0009
      }, {
        nominal: 121,
        mass: 120.904944
      }, {
        nominal: 122,
        mass: 121.9030435,
        abundance: 0.0255
      }, {
        nominal: 123,
        mass: 122.9042698,
        abundance: 0.0089
      }, {
        nominal: 124,
        mass: 123.9028171,
        abundance: 0.0474
      }, {
        nominal: 125,
        mass: 124.9044299,
        abundance: 0.0707
      }, {
        nominal: 126,
        mass: 125.9033109,
        abundance: 0.1884
      }, {
        nominal: 127,
        mass: 126.9052257
      }, {
        nominal: 128,
        mass: 127.90446128,
        abundance: 0.3174
      }, {
        nominal: 129,
        mass: 128.90659646
      }, {
        nominal: 130,
        mass: 129.906222748,
        abundance: 0.3408
      }, {
        nominal: 131,
        mass: 130.908522213
      }, {
        nominal: 132,
        mass: 131.9085467
      }, {
        nominal: 133,
        mass: 132.9109688
      }, {
        nominal: 134,
        mass: 133.911394
      }, {
        nominal: 135,
        mass: 134.9165557
      }, {
        nominal: 136,
        mass: 135.9201006
      }, {
        nominal: 137,
        mass: 136.9255989
      }, {
        nominal: 138,
        mass: 137.9294722
      }, {
        nominal: 139,
        mass: 138.9353672
      }, {
        nominal: 140,
        mass: 139.939499
      }, {
        nominal: 141,
        mass: 140.9458
      }, {
        nominal: 142,
        mass: 141.95022
      }, {
        nominal: 143,
        mass: 142.95676
      }],
      symbol: 'Te',
      mass: 127.6031264846604,
      name: 'Tellurium',
      monoisotopicMass: 129.906222748
    }, {
      number: 53,
      isotopes: [{
        nominal: 107,
        mass: 106.94678
      }, {
        nominal: 108,
        mass: 107.94348
      }, {
        nominal: 109,
        mass: 108.9380853
      }, {
        nominal: 110,
        mass: 109.935089
      }, {
        nominal: 111,
        mass: 110.9302692
      }, {
        nominal: 112,
        mass: 111.928005
      }, {
        nominal: 113,
        mass: 112.9236501
      }, {
        nominal: 114,
        mass: 113.92185
      }, {
        nominal: 115,
        mass: 114.918048
      }, {
        nominal: 116,
        mass: 115.91681
      }, {
        nominal: 117,
        mass: 116.913648
      }, {
        nominal: 118,
        mass: 117.913074
      }, {
        nominal: 119,
        mass: 118.910074
      }, {
        nominal: 120,
        mass: 119.910087
      }, {
        nominal: 121,
        mass: 120.9074051
      }, {
        nominal: 122,
        mass: 121.9075888
      }, {
        nominal: 123,
        mass: 122.9055885
      }, {
        nominal: 124,
        mass: 123.906209
      }, {
        nominal: 125,
        mass: 124.9046294
      }, {
        nominal: 126,
        mass: 125.9056233
      }, {
        nominal: 127,
        mass: 126.9044719,
        abundance: 1
      }, {
        nominal: 128,
        mass: 127.9058086
      }, {
        nominal: 129,
        mass: 128.9049837
      }, {
        nominal: 130,
        mass: 129.9066702
      }, {
        nominal: 131,
        mass: 130.9061263
      }, {
        nominal: 132,
        mass: 131.9079935
      }, {
        nominal: 133,
        mass: 132.907797
      }, {
        nominal: 134,
        mass: 133.9097588
      }, {
        nominal: 135,
        mass: 134.9100488
      }, {
        nominal: 136,
        mass: 135.914604
      }, {
        nominal: 137,
        mass: 136.9180282
      }, {
        nominal: 138,
        mass: 137.9227264
      }, {
        nominal: 139,
        mass: 138.926506
      }, {
        nominal: 140,
        mass: 139.93173
      }, {
        nominal: 141,
        mass: 140.93569
      }, {
        nominal: 142,
        mass: 141.9412
      }, {
        nominal: 143,
        mass: 142.94565
      }, {
        nominal: 144,
        mass: 143.95139
      }, {
        nominal: 145,
        mass: 144.95605
      }],
      symbol: 'I',
      mass: 126.9044719,
      name: 'Iodine',
      monoisotopicMass: 126.9044719
    }, {
      number: 54,
      isotopes: [{
        nominal: 109,
        mass: 108.95043
      }, {
        nominal: 110,
        mass: 109.94426
      }, {
        nominal: 111,
        mass: 110.941607
      }, {
        nominal: 112,
        mass: 111.935559
      }, {
        nominal: 113,
        mass: 112.9332217
      }, {
        nominal: 114,
        mass: 113.92798
      }, {
        nominal: 115,
        mass: 114.926294
      }, {
        nominal: 116,
        mass: 115.921581
      }, {
        nominal: 117,
        mass: 116.920359
      }, {
        nominal: 118,
        mass: 117.916179
      }, {
        nominal: 119,
        mass: 118.915411
      }, {
        nominal: 120,
        mass: 119.911784
      }, {
        nominal: 121,
        mass: 120.911453
      }, {
        nominal: 122,
        mass: 121.908368
      }, {
        nominal: 123,
        mass: 122.908482
      }, {
        nominal: 124,
        mass: 123.905892,
        abundance: 0.000952
      }, {
        nominal: 125,
        mass: 124.9063944
      }, {
        nominal: 126,
        mass: 125.9042983,
        abundance: 0.00089
      }, {
        nominal: 127,
        mass: 126.9051829
      }, {
        nominal: 128,
        mass: 127.903531,
        abundance: 0.019102
      }, {
        nominal: 129,
        mass: 128.9047808611,
        abundance: 0.264006
      }, {
        nominal: 130,
        mass: 129.903509349,
        abundance: 0.04071
      }, {
        nominal: 131,
        mass: 130.90508406,
        abundance: 0.212324
      }, {
        nominal: 132,
        mass: 131.9041550856,
        abundance: 0.269086
      }, {
        nominal: 133,
        mass: 132.9059108
      }, {
        nominal: 134,
        mass: 133.90539466,
        abundance: 0.104357
      }, {
        nominal: 135,
        mass: 134.9072278
      }, {
        nominal: 136,
        mass: 135.907214484,
        abundance: 0.088573
      }, {
        nominal: 137,
        mass: 136.91155778
      }, {
        nominal: 138,
        mass: 137.9141463
      }, {
        nominal: 139,
        mass: 138.9187922
      }, {
        nominal: 140,
        mass: 139.9216458
      }, {
        nominal: 141,
        mass: 140.9267872
      }, {
        nominal: 142,
        mass: 141.9299731
      }, {
        nominal: 143,
        mass: 142.9353696
      }, {
        nominal: 144,
        mass: 143.9389451
      }, {
        nominal: 145,
        mass: 144.94472
      }, {
        nominal: 146,
        mass: 145.948518
      }, {
        nominal: 147,
        mass: 146.95426
      }, {
        nominal: 148,
        mass: 147.95813
      }],
      symbol: 'Xe',
      mass: 131.29276144779053,
      name: 'Xenon',
      monoisotopicMass: 131.9041550856
    }, {
      number: 55,
      isotopes: [{
        nominal: 112,
        mass: 111.950309
      }, {
        nominal: 113,
        mass: 112.9444291
      }, {
        nominal: 114,
        mass: 113.941296
      }, {
        nominal: 115,
        mass: 114.93591
      }, {
        nominal: 116,
        mass: 115.93337
      }, {
        nominal: 117,
        mass: 116.928617
      }, {
        nominal: 118,
        mass: 117.92656
      }, {
        nominal: 119,
        mass: 118.922377
      }, {
        nominal: 120,
        mass: 119.920677
      }, {
        nominal: 121,
        mass: 120.917227
      }, {
        nominal: 122,
        mass: 121.916108
      }, {
        nominal: 123,
        mass: 122.912996
      }, {
        nominal: 124,
        mass: 123.9122578
      }, {
        nominal: 125,
        mass: 124.909728
      }, {
        nominal: 126,
        mass: 125.909446
      }, {
        nominal: 127,
        mass: 126.9074174
      }, {
        nominal: 128,
        mass: 127.9077487
      }, {
        nominal: 129,
        mass: 128.9060657
      }, {
        nominal: 130,
        mass: 129.9067093
      }, {
        nominal: 131,
        mass: 130.9054649
      }, {
        nominal: 132,
        mass: 131.9064339
      }, {
        nominal: 133,
        mass: 132.905451961,
        abundance: 1
      }, {
        nominal: 134,
        mass: 133.906718503
      }, {
        nominal: 135,
        mass: 134.905977
      }, {
        nominal: 136,
        mass: 135.9073114
      }, {
        nominal: 137,
        mass: 136.90708923
      }, {
        nominal: 138,
        mass: 137.9110171
      }, {
        nominal: 139,
        mass: 138.9133638
      }, {
        nominal: 140,
        mass: 139.9172831
      }, {
        nominal: 141,
        mass: 140.9200455
      }, {
        nominal: 142,
        mass: 141.924296
      }, {
        nominal: 143,
        mass: 142.927349
      }, {
        nominal: 144,
        mass: 143.932076
      }, {
        nominal: 145,
        mass: 144.935527
      }, {
        nominal: 146,
        mass: 145.940344
      }, {
        nominal: 147,
        mass: 146.944156
      }, {
        nominal: 148,
        mass: 147.94923
      }, {
        nominal: 149,
        mass: 148.95302
      }, {
        nominal: 150,
        mass: 149.95833
      }, {
        nominal: 151,
        mass: 150.96258
      }],
      symbol: 'Cs',
      mass: 132.905451961,
      name: 'Caesium',
      monoisotopicMass: 132.905451961
    }, {
      number: 56,
      isotopes: [{
        nominal: 114,
        mass: 113.95066
      }, {
        nominal: 115,
        mass: 114.94737
      }, {
        nominal: 116,
        mass: 115.94128
      }, {
        nominal: 117,
        mass: 116.93814
      }, {
        nominal: 118,
        mass: 117.93306
      }, {
        nominal: 119,
        mass: 118.93066
      }, {
        nominal: 120,
        mass: 119.92605
      }, {
        nominal: 121,
        mass: 120.92405
      }, {
        nominal: 122,
        mass: 121.919904
      }, {
        nominal: 123,
        mass: 122.918781
      }, {
        nominal: 124,
        mass: 123.915094
      }, {
        nominal: 125,
        mass: 124.914472
      }, {
        nominal: 126,
        mass: 125.91125
      }, {
        nominal: 127,
        mass: 126.911091
      }, {
        nominal: 128,
        mass: 127.908342
      }, {
        nominal: 129,
        mass: 128.908681
      }, {
        nominal: 130,
        mass: 129.9063207,
        abundance: 0.00106
      }, {
        nominal: 131,
        mass: 130.906941
      }, {
        nominal: 132,
        mass: 131.9050611,
        abundance: 0.00101
      }, {
        nominal: 133,
        mass: 132.9060074
      }, {
        nominal: 134,
        mass: 133.90450818,
        abundance: 0.02417
      }, {
        nominal: 135,
        mass: 134.90568838,
        abundance: 0.06592
      }, {
        nominal: 136,
        mass: 135.90457573,
        abundance: 0.07854
      }, {
        nominal: 137,
        mass: 136.90582714,
        abundance: 0.11232
      }, {
        nominal: 138,
        mass: 137.905247,
        abundance: 0.71698
      }, {
        nominal: 139,
        mass: 138.9088411
      }, {
        nominal: 140,
        mass: 139.9106057
      }, {
        nominal: 141,
        mass: 140.9144033
      }, {
        nominal: 142,
        mass: 141.9164324
      }, {
        nominal: 143,
        mass: 142.9206253
      }, {
        nominal: 144,
        mass: 143.9229549
      }, {
        nominal: 145,
        mass: 144.9275184
      }, {
        nominal: 146,
        mass: 145.930284
      }, {
        nominal: 147,
        mass: 146.935304
      }, {
        nominal: 148,
        mass: 147.938171
      }, {
        nominal: 149,
        mass: 148.94308
      }, {
        nominal: 150,
        mass: 149.94605
      }, {
        nominal: 151,
        mass: 150.95127
      }, {
        nominal: 152,
        mass: 151.95481
      }, {
        nominal: 153,
        mass: 152.96036
      }],
      symbol: 'Ba',
      mass: 137.3268916286322,
      name: 'Barium',
      monoisotopicMass: 137.905247
    }, {
      number: 57,
      isotopes: [{
        nominal: 116,
        mass: 115.9563
      }, {
        nominal: 117,
        mass: 116.94999
      }, {
        nominal: 118,
        mass: 117.94673
      }, {
        nominal: 119,
        mass: 118.94099
      }, {
        nominal: 120,
        mass: 119.93807
      }, {
        nominal: 121,
        mass: 120.93315
      }, {
        nominal: 122,
        mass: 121.93071
      }, {
        nominal: 123,
        mass: 122.9263
      }, {
        nominal: 124,
        mass: 123.924574
      }, {
        nominal: 125,
        mass: 124.920816
      }, {
        nominal: 126,
        mass: 125.919513
      }, {
        nominal: 127,
        mass: 126.916375
      }, {
        nominal: 128,
        mass: 127.915592
      }, {
        nominal: 129,
        mass: 128.912694
      }, {
        nominal: 130,
        mass: 129.912369
      }, {
        nominal: 131,
        mass: 130.91007
      }, {
        nominal: 132,
        mass: 131.910119
      }, {
        nominal: 133,
        mass: 132.908218
      }, {
        nominal: 134,
        mass: 133.908514
      }, {
        nominal: 135,
        mass: 134.906984
      }, {
        nominal: 136,
        mass: 135.907635
      }, {
        nominal: 137,
        mass: 136.9064504
      }, {
        nominal: 138,
        mass: 137.9071149,
        abundance: 0.0008881
      }, {
        nominal: 139,
        mass: 138.9063563,
        abundance: 0.9991119
      }, {
        nominal: 140,
        mass: 139.9094806
      }, {
        nominal: 141,
        mass: 140.910966
      }, {
        nominal: 142,
        mass: 141.9140909
      }, {
        nominal: 143,
        mass: 142.9160795
      }, {
        nominal: 144,
        mass: 143.919646
      }, {
        nominal: 145,
        mass: 144.921808
      }, {
        nominal: 146,
        mass: 145.925875
      }, {
        nominal: 147,
        mass: 146.928418
      }, {
        nominal: 148,
        mass: 147.932679
      }, {
        nominal: 149,
        mass: 148.93535
      }, {
        nominal: 150,
        mass: 149.93947
      }, {
        nominal: 151,
        mass: 150.94232
      }, {
        nominal: 152,
        mass: 151.94682
      }, {
        nominal: 153,
        mass: 152.95036
      }, {
        nominal: 154,
        mass: 153.95517
      }, {
        nominal: 155,
        mass: 154.95901
      }],
      symbol: 'La',
      mass: 138.90546887371266,
      name: 'Lanthanum',
      monoisotopicMass: 138.9063563
    }, {
      number: 58,
      isotopes: [{
        nominal: 119,
        mass: 118.95271
      }, {
        nominal: 120,
        mass: 119.94654
      }, {
        nominal: 121,
        mass: 120.94335
      }, {
        nominal: 122,
        mass: 121.93787
      }, {
        nominal: 123,
        mass: 122.93528
      }, {
        nominal: 124,
        mass: 123.93031
      }, {
        nominal: 125,
        mass: 124.92844
      }, {
        nominal: 126,
        mass: 125.923971
      }, {
        nominal: 127,
        mass: 126.922727
      }, {
        nominal: 128,
        mass: 127.918911
      }, {
        nominal: 129,
        mass: 128.918102
      }, {
        nominal: 130,
        mass: 129.914736
      }, {
        nominal: 131,
        mass: 130.914429
      }, {
        nominal: 132,
        mass: 131.911464
      }, {
        nominal: 133,
        mass: 132.91152
      }, {
        nominal: 134,
        mass: 133.908928
      }, {
        nominal: 135,
        mass: 134.909161
      }, {
        nominal: 136,
        mass: 135.90712921,
        abundance: 0.00185
      }, {
        nominal: 137,
        mass: 136.90776236
      }, {
        nominal: 138,
        mass: 137.905991,
        abundance: 0.00251
      }, {
        nominal: 139,
        mass: 138.9066551
      }, {
        nominal: 140,
        mass: 139.9054431,
        abundance: 0.8845
      }, {
        nominal: 141,
        mass: 140.9082807
      }, {
        nominal: 142,
        mass: 141.9092504,
        abundance: 0.11114
      }, {
        nominal: 143,
        mass: 142.9123921
      }, {
        nominal: 144,
        mass: 143.9136529
      }, {
        nominal: 145,
        mass: 144.917265
      }, {
        nominal: 146,
        mass: 145.918802
      }, {
        nominal: 147,
        mass: 146.9226899
      }, {
        nominal: 148,
        mass: 147.924424
      }, {
        nominal: 149,
        mass: 148.928427
      }, {
        nominal: 150,
        mass: 149.930384
      }, {
        nominal: 151,
        mass: 150.934272
      }, {
        nominal: 152,
        mass: 151.9366
      }, {
        nominal: 153,
        mass: 152.94093
      }, {
        nominal: 154,
        mass: 153.9438
      }, {
        nominal: 155,
        mass: 154.94855
      }, {
        nominal: 156,
        mass: 155.95183
      }, {
        nominal: 157,
        mass: 156.95705
      }],
      symbol: 'Ce',
      mass: 140.1157307378545,
      name: 'Cerium',
      monoisotopicMass: 139.9054431
    }, {
      number: 59,
      isotopes: [{
        nominal: 121,
        mass: 120.95532
      }, {
        nominal: 122,
        mass: 121.95175
      }, {
        nominal: 123,
        mass: 122.94596
      }, {
        nominal: 124,
        mass: 123.94294
      }, {
        nominal: 125,
        mass: 124.9377
      }, {
        nominal: 126,
        mass: 125.93524
      }, {
        nominal: 127,
        mass: 126.93071
      }, {
        nominal: 128,
        mass: 127.928791
      }, {
        nominal: 129,
        mass: 128.925095
      }, {
        nominal: 130,
        mass: 129.92359
      }, {
        nominal: 131,
        mass: 130.920235
      }, {
        nominal: 132,
        mass: 131.919255
      }, {
        nominal: 133,
        mass: 132.916331
      }, {
        nominal: 134,
        mass: 133.915697
      }, {
        nominal: 135,
        mass: 134.913112
      }, {
        nominal: 136,
        mass: 135.912677
      }, {
        nominal: 137,
        mass: 136.9106792
      }, {
        nominal: 138,
        mass: 137.910754
      }, {
        nominal: 139,
        mass: 138.9089408
      }, {
        nominal: 140,
        mass: 139.9090803
      }, {
        nominal: 141,
        mass: 140.9076576,
        abundance: 1
      }, {
        nominal: 142,
        mass: 141.9100496
      }, {
        nominal: 143,
        mass: 142.9108228
      }, {
        nominal: 144,
        mass: 143.9133109
      }, {
        nominal: 145,
        mass: 144.9145182
      }, {
        nominal: 146,
        mass: 145.91768
      }, {
        nominal: 147,
        mass: 146.919008
      }, {
        nominal: 148,
        mass: 147.92213
      }, {
        nominal: 149,
        mass: 148.923736
      }, {
        nominal: 150,
        mass: 149.9266765
      }, {
        nominal: 151,
        mass: 150.928309
      }, {
        nominal: 152,
        mass: 151.931553
      }, {
        nominal: 153,
        mass: 152.933904
      }, {
        nominal: 154,
        mass: 153.93753
      }, {
        nominal: 155,
        mass: 154.940509
      }, {
        nominal: 156,
        mass: 155.94464
      }, {
        nominal: 157,
        mass: 156.94789
      }, {
        nominal: 158,
        mass: 157.95241
      }, {
        nominal: 159,
        mass: 158.95589
      }],
      symbol: 'Pr',
      mass: 140.9076576,
      name: 'Praseodymium',
      monoisotopicMass: 140.9076576
    }, {
      number: 60,
      isotopes: [{
        nominal: 124,
        mass: 123.9522
      }, {
        nominal: 125,
        mass: 124.9489
      }, {
        nominal: 126,
        mass: 125.94311
      }, {
        nominal: 127,
        mass: 126.94038
      }, {
        nominal: 128,
        mass: 127.93525
      }, {
        nominal: 129,
        mass: 128.9331
      }, {
        nominal: 130,
        mass: 129.928506
      }, {
        nominal: 131,
        mass: 130.927248
      }, {
        nominal: 132,
        mass: 131.923321
      }, {
        nominal: 133,
        mass: 132.922348
      }, {
        nominal: 134,
        mass: 133.91879
      }, {
        nominal: 135,
        mass: 134.918181
      }, {
        nominal: 136,
        mass: 135.914976
      }, {
        nominal: 137,
        mass: 136.914562
      }, {
        nominal: 138,
        mass: 137.91195
      }, {
        nominal: 139,
        mass: 138.911954
      }, {
        nominal: 140,
        mass: 139.90955
      }, {
        nominal: 141,
        mass: 140.9096147
      }, {
        nominal: 142,
        mass: 141.907729,
        abundance: 0.27152
      }, {
        nominal: 143,
        mass: 142.90982,
        abundance: 0.12174
      }, {
        nominal: 144,
        mass: 143.910093,
        abundance: 0.23798
      }, {
        nominal: 145,
        mass: 144.9125793,
        abundance: 0.08293
      }, {
        nominal: 146,
        mass: 145.9131226,
        abundance: 0.17189
      }, {
        nominal: 147,
        mass: 146.9161061
      }, {
        nominal: 148,
        mass: 147.9168993,
        abundance: 0.05756
      }, {
        nominal: 149,
        mass: 148.9201548
      }, {
        nominal: 150,
        mass: 149.9209022,
        abundance: 0.05638
      }, {
        nominal: 151,
        mass: 150.9238403
      }, {
        nominal: 152,
        mass: 151.924692
      }, {
        nominal: 153,
        mass: 152.927718
      }, {
        nominal: 154,
        mass: 153.92948
      }, {
        nominal: 155,
        mass: 154.9331357
      }, {
        nominal: 156,
        mass: 155.93508
      }, {
        nominal: 157,
        mass: 156.939386
      }, {
        nominal: 158,
        mass: 157.94197
      }, {
        nominal: 159,
        mass: 158.94653
      }, {
        nominal: 160,
        mass: 159.9494
      }, {
        nominal: 161,
        mass: 160.95428
      }],
      symbol: 'Nd',
      mass: 144.241596031827,
      name: 'Neodymium',
      monoisotopicMass: 141.907729
    }, {
      number: 61,
      isotopes: [{
        nominal: 126,
        mass: 125.95792
      }, {
        nominal: 127,
        mass: 126.95192
      }, {
        nominal: 128,
        mass: 127.9487
      }, {
        nominal: 129,
        mass: 128.94323
      }, {
        nominal: 130,
        mass: 129.94053
      }, {
        nominal: 131,
        mass: 130.93567
      }, {
        nominal: 132,
        mass: 131.93384
      }, {
        nominal: 133,
        mass: 132.929782
      }, {
        nominal: 134,
        mass: 133.928353
      }, {
        nominal: 135,
        mass: 134.924823
      }, {
        nominal: 136,
        mass: 135.923585
      }, {
        nominal: 137,
        mass: 136.92048
      }, {
        nominal: 138,
        mass: 137.919548
      }, {
        nominal: 139,
        mass: 138.9168
      }, {
        nominal: 140,
        mass: 139.91604
      }, {
        nominal: 141,
        mass: 140.913555
      }, {
        nominal: 142,
        mass: 141.91289
      }, {
        nominal: 143,
        mass: 142.9109383
      }, {
        nominal: 144,
        mass: 143.9125964
      }, {
        nominal: 145,
        mass: 144.9127559
      }, {
        nominal: 146,
        mass: 145.9147024
      }, {
        nominal: 147,
        mass: 146.915145
      }, {
        nominal: 148,
        mass: 147.9174819
      }, {
        nominal: 149,
        mass: 148.9183423
      }, {
        nominal: 150,
        mass: 149.920991
      }, {
        nominal: 151,
        mass: 150.9212175
      }, {
        nominal: 152,
        mass: 151.923506
      }, {
        nominal: 153,
        mass: 152.9241567
      }, {
        nominal: 154,
        mass: 153.926472
      }, {
        nominal: 155,
        mass: 154.928137
      }, {
        nominal: 156,
        mass: 155.9311175
      }, {
        nominal: 157,
        mass: 156.9331214
      }, {
        nominal: 158,
        mass: 157.936565
      }, {
        nominal: 159,
        mass: 158.939287
      }, {
        nominal: 160,
        mass: 159.9431
      }, {
        nominal: 161,
        mass: 160.94607
      }, {
        nominal: 162,
        mass: 161.95022
      }, {
        nominal: 163,
        mass: 162.95357
      }],
      symbol: 'Pm',
      mass: null,
      name: 'Promethium'
    }, {
      number: 62,
      isotopes: [{
        nominal: 128,
        mass: 127.95842
      }, {
        nominal: 129,
        mass: 128.95476
      }, {
        nominal: 130,
        mass: 129.949
      }, {
        nominal: 131,
        mass: 130.94618
      }, {
        nominal: 132,
        mass: 131.94087
      }, {
        nominal: 133,
        mass: 132.93856
      }, {
        nominal: 134,
        mass: 133.93411
      }, {
        nominal: 135,
        mass: 134.93252
      }, {
        nominal: 136,
        mass: 135.928276
      }, {
        nominal: 137,
        mass: 136.926971
      }, {
        nominal: 138,
        mass: 137.923244
      }, {
        nominal: 139,
        mass: 138.922297
      }, {
        nominal: 140,
        mass: 139.918995
      }, {
        nominal: 141,
        mass: 140.9184816
      }, {
        nominal: 142,
        mass: 141.9152044
      }, {
        nominal: 143,
        mass: 142.9146353
      }, {
        nominal: 144,
        mass: 143.9120065,
        abundance: 0.0307
      }, {
        nominal: 145,
        mass: 144.9134173
      }, {
        nominal: 146,
        mass: 145.913047
      }, {
        nominal: 147,
        mass: 146.9149044,
        abundance: 0.1499
      }, {
        nominal: 148,
        mass: 147.9148292,
        abundance: 0.1124
      }, {
        nominal: 149,
        mass: 148.9171921,
        abundance: 0.1382
      }, {
        nominal: 150,
        mass: 149.9172829,
        abundance: 0.0738
      }, {
        nominal: 151,
        mass: 150.9199398
      }, {
        nominal: 152,
        mass: 151.9197397,
        abundance: 0.2675
      }, {
        nominal: 153,
        mass: 152.9221047
      }, {
        nominal: 154,
        mass: 153.9222169,
        abundance: 0.2275
      }, {
        nominal: 155,
        mass: 154.9246477
      }, {
        nominal: 156,
        mass: 155.925536
      }, {
        nominal: 157,
        mass: 156.9284187
      }, {
        nominal: 158,
        mass: 157.929951
      }, {
        nominal: 159,
        mass: 158.9332172
      }, {
        nominal: 160,
        mass: 159.9353353
      }, {
        nominal: 161,
        mass: 160.9391602
      }, {
        nominal: 162,
        mass: 161.94146
      }, {
        nominal: 163,
        mass: 162.94555
      }, {
        nominal: 164,
        mass: 163.94836
      }, {
        nominal: 165,
        mass: 164.95297
      }],
      symbol: 'Sm',
      mass: 150.36635571193,
      name: 'Samarium',
      monoisotopicMass: 151.9197397
    }, {
      number: 63,
      isotopes: [{
        nominal: 130,
        mass: 129.96369
      }, {
        nominal: 131,
        mass: 130.95784
      }, {
        nominal: 132,
        mass: 131.95467
      }, {
        nominal: 133,
        mass: 132.94929
      }, {
        nominal: 134,
        mass: 133.9464
      }, {
        nominal: 135,
        mass: 134.94187
      }, {
        nominal: 136,
        mass: 135.93962
      }, {
        nominal: 137,
        mass: 136.93546
      }, {
        nominal: 138,
        mass: 137.933709
      }, {
        nominal: 139,
        mass: 138.929792
      }, {
        nominal: 140,
        mass: 139.928088
      }, {
        nominal: 141,
        mass: 140.924932
      }, {
        nominal: 142,
        mass: 141.923442
      }, {
        nominal: 143,
        mass: 142.920299
      }, {
        nominal: 144,
        mass: 143.91882
      }, {
        nominal: 145,
        mass: 144.9162726
      }, {
        nominal: 146,
        mass: 145.917211
      }, {
        nominal: 147,
        mass: 146.9167527
      }, {
        nominal: 148,
        mass: 147.918089
      }, {
        nominal: 149,
        mass: 148.9179378
      }, {
        nominal: 150,
        mass: 149.9197077
      }, {
        nominal: 151,
        mass: 150.9198578,
        abundance: 0.4781
      }, {
        nominal: 152,
        mass: 151.9217522
      }, {
        nominal: 153,
        mass: 152.921238,
        abundance: 0.5219
      }, {
        nominal: 154,
        mass: 153.922987
      }, {
        nominal: 155,
        mass: 154.9229011
      }, {
        nominal: 156,
        mass: 155.9247605
      }, {
        nominal: 157,
        mass: 156.9254334
      }, {
        nominal: 158,
        mass: 157.927799
      }, {
        nominal: 159,
        mass: 158.9291001
      }, {
        nominal: 160,
        mass: 159.931851
      }, {
        nominal: 161,
        mass: 160.933664
      }, {
        nominal: 162,
        mass: 161.936989
      }, {
        nominal: 163,
        mass: 162.939196
      }, {
        nominal: 164,
        mass: 163.94274
      }, {
        nominal: 165,
        mass: 164.94559
      }, {
        nominal: 166,
        mass: 165.94962
      }, {
        nominal: 167,
        mass: 166.95289
      }],
      symbol: 'Eu',
      mass: 151.96437812637998,
      name: 'Europium',
      monoisotopicMass: 152.921238
    }, {
      number: 64,
      isotopes: [{
        nominal: 133,
        mass: 132.96133
      }, {
        nominal: 134,
        mass: 133.95566
      }, {
        nominal: 135,
        mass: 134.95245
      }, {
        nominal: 136,
        mass: 135.9473
      }, {
        nominal: 137,
        mass: 136.94502
      }, {
        nominal: 138,
        mass: 137.94025
      }, {
        nominal: 139,
        mass: 138.93813
      }, {
        nominal: 140,
        mass: 139.933674
      }, {
        nominal: 141,
        mass: 140.932126
      }, {
        nominal: 142,
        mass: 141.928116
      }, {
        nominal: 143,
        mass: 142.92675
      }, {
        nominal: 144,
        mass: 143.922963
      }, {
        nominal: 145,
        mass: 144.921713
      }, {
        nominal: 146,
        mass: 145.9183188
      }, {
        nominal: 147,
        mass: 146.9191014
      }, {
        nominal: 148,
        mass: 147.9181215
      }, {
        nominal: 149,
        mass: 148.9193481
      }, {
        nominal: 150,
        mass: 149.9186644
      }, {
        nominal: 151,
        mass: 150.920356
      }, {
        nominal: 152,
        mass: 151.9197995,
        abundance: 0.002
      }, {
        nominal: 153,
        mass: 152.921758
      }, {
        nominal: 154,
        mass: 153.9208741,
        abundance: 0.0218
      }, {
        nominal: 155,
        mass: 154.9226305,
        abundance: 0.148
      }, {
        nominal: 156,
        mass: 155.9221312,
        abundance: 0.2047
      }, {
        nominal: 157,
        mass: 156.9239686,
        abundance: 0.1565
      }, {
        nominal: 158,
        mass: 157.9241123,
        abundance: 0.2484
      }, {
        nominal: 159,
        mass: 158.926397
      }, {
        nominal: 160,
        mass: 159.9270624,
        abundance: 0.2186
      }, {
        nominal: 161,
        mass: 160.9296775
      }, {
        nominal: 162,
        mass: 161.930993
      }, {
        nominal: 163,
        mass: 162.9341769
      }, {
        nominal: 164,
        mass: 163.93583
      }, {
        nominal: 165,
        mass: 164.93936
      }, {
        nominal: 166,
        mass: 165.94146
      }, {
        nominal: 167,
        mass: 166.94545
      }, {
        nominal: 168,
        mass: 167.94808
      }, {
        nominal: 169,
        mass: 168.9526
      }],
      symbol: 'Gd',
      mass: 157.25213064687998,
      name: 'Gadolinium',
      monoisotopicMass: 157.9241123
    }, {
      number: 65,
      isotopes: [{
        nominal: 135,
        mass: 134.96476
      }, {
        nominal: 136,
        mass: 135.96129
      }, {
        nominal: 137,
        mass: 136.95602
      }, {
        nominal: 138,
        mass: 137.95312
      }, {
        nominal: 139,
        mass: 138.94833
      }, {
        nominal: 140,
        mass: 139.94581
      }, {
        nominal: 141,
        mass: 140.94145
      }, {
        nominal: 142,
        mass: 141.93928
      }, {
        nominal: 143,
        mass: 142.935137
      }, {
        nominal: 144,
        mass: 143.933045
      }, {
        nominal: 145,
        mass: 144.92882
      }, {
        nominal: 146,
        mass: 145.927253
      }, {
        nominal: 147,
        mass: 146.9240548
      }, {
        nominal: 148,
        mass: 147.924282
      }, {
        nominal: 149,
        mass: 148.9232535
      }, {
        nominal: 150,
        mass: 149.9236649
      }, {
        nominal: 151,
        mass: 150.9231096
      }, {
        nominal: 152,
        mass: 151.924083
      }, {
        nominal: 153,
        mass: 152.9234424
      }, {
        nominal: 154,
        mass: 153.924685
      }, {
        nominal: 155,
        mass: 154.923511
      }, {
        nominal: 156,
        mass: 155.9247552
      }, {
        nominal: 157,
        mass: 156.924033
      }, {
        nominal: 158,
        mass: 157.9254209
      }, {
        nominal: 159,
        mass: 158.9253547,
        abundance: 1
      }, {
        nominal: 160,
        mass: 159.9271756
      }, {
        nominal: 161,
        mass: 160.9275778
      }, {
        nominal: 162,
        mass: 161.929495
      }, {
        nominal: 163,
        mass: 162.9306547
      }, {
        nominal: 164,
        mass: 163.93336
      }, {
        nominal: 165,
        mass: 164.93498
      }, {
        nominal: 166,
        mass: 165.93786
      }, {
        nominal: 167,
        mass: 166.93996
      }, {
        nominal: 168,
        mass: 167.9434
      }, {
        nominal: 169,
        mass: 168.94597
      }, {
        nominal: 170,
        mass: 169.94984
      }, {
        nominal: 171,
        mass: 170.95273
      }],
      symbol: 'Tb',
      mass: 158.9253547,
      name: 'Terbium',
      monoisotopicMass: 158.9253547
    }, {
      number: 66,
      isotopes: [{
        nominal: 138,
        mass: 137.9625
      }, {
        nominal: 139,
        mass: 138.95959
      }, {
        nominal: 140,
        mass: 139.95402
      }, {
        nominal: 141,
        mass: 140.95128
      }, {
        nominal: 142,
        mass: 141.94619
      }, {
        nominal: 143,
        mass: 142.943994
      }, {
        nominal: 144,
        mass: 143.9392695
      }, {
        nominal: 145,
        mass: 144.937474
      }, {
        nominal: 146,
        mass: 145.9328445
      }, {
        nominal: 147,
        mass: 146.9310827
      }, {
        nominal: 148,
        mass: 147.927157
      }, {
        nominal: 149,
        mass: 148.927322
      }, {
        nominal: 150,
        mass: 149.9255933
      }, {
        nominal: 151,
        mass: 150.9261916
      }, {
        nominal: 152,
        mass: 151.9247253
      }, {
        nominal: 153,
        mass: 152.9257724
      }, {
        nominal: 154,
        mass: 153.9244293
      }, {
        nominal: 155,
        mass: 154.925759
      }, {
        nominal: 156,
        mass: 155.9242847,
        abundance: 0.00056
      }, {
        nominal: 157,
        mass: 156.9254707
      }, {
        nominal: 158,
        mass: 157.9244159,
        abundance: 0.00095
      }, {
        nominal: 159,
        mass: 158.925747
      }, {
        nominal: 160,
        mass: 159.9252046,
        abundance: 0.02329
      }, {
        nominal: 161,
        mass: 160.9269405,
        abundance: 0.18889
      }, {
        nominal: 162,
        mass: 161.9268056,
        abundance: 0.25475
      }, {
        nominal: 163,
        mass: 162.9287383,
        abundance: 0.24896
      }, {
        nominal: 164,
        mass: 163.9291819,
        abundance: 0.2826
      }, {
        nominal: 165,
        mass: 164.9317105
      }, {
        nominal: 166,
        mass: 165.9328139
      }, {
        nominal: 167,
        mass: 166.935661
      }, {
        nominal: 168,
        mass: 167.93713
      }, {
        nominal: 169,
        mass: 168.94031
      }, {
        nominal: 170,
        mass: 169.94239
      }, {
        nominal: 171,
        mass: 170.94612
      }, {
        nominal: 172,
        mass: 171.94846
      }, {
        nominal: 173,
        mass: 172.95283
      }],
      symbol: 'Dy',
      mass: 162.499472819424,
      name: 'Dysprosium',
      monoisotopicMass: 163.9291819
    }, {
      number: 67,
      isotopes: [{
        nominal: 140,
        mass: 139.96859
      }, {
        nominal: 141,
        mass: 140.96311
      }, {
        nominal: 142,
        mass: 141.96001
      }, {
        nominal: 143,
        mass: 142.95486
      }, {
        nominal: 144,
        mass: 143.9521097
      }, {
        nominal: 145,
        mass: 144.9472674
      }, {
        nominal: 146,
        mass: 145.9449935
      }, {
        nominal: 147,
        mass: 146.9401423
      }, {
        nominal: 148,
        mass: 147.937744
      }, {
        nominal: 149,
        mass: 148.933803
      }, {
        nominal: 150,
        mass: 149.933498
      }, {
        nominal: 151,
        mass: 150.9316983
      }, {
        nominal: 152,
        mass: 151.931724
      }, {
        nominal: 153,
        mass: 152.9302064
      }, {
        nominal: 154,
        mass: 153.9306068
      }, {
        nominal: 155,
        mass: 154.929104
      }, {
        nominal: 156,
        mass: 155.929706
      }, {
        nominal: 157,
        mass: 156.928254
      }, {
        nominal: 158,
        mass: 157.928946
      }, {
        nominal: 159,
        mass: 158.9277197
      }, {
        nominal: 160,
        mass: 159.928737
      }, {
        nominal: 161,
        mass: 160.9278615
      }, {
        nominal: 162,
        mass: 161.9291023
      }, {
        nominal: 163,
        mass: 162.928741
      }, {
        nominal: 164,
        mass: 163.9302403
      }, {
        nominal: 165,
        mass: 164.9303288,
        abundance: 1
      }, {
        nominal: 166,
        mass: 165.9322909
      }, {
        nominal: 167,
        mass: 166.9331385
      }, {
        nominal: 168,
        mass: 167.935522
      }, {
        nominal: 169,
        mass: 168.936878
      }, {
        nominal: 170,
        mass: 169.939625
      }, {
        nominal: 171,
        mass: 170.94147
      }, {
        nominal: 172,
        mass: 171.94473
      }, {
        nominal: 173,
        mass: 172.94702
      }, {
        nominal: 174,
        mass: 173.95095
      }, {
        nominal: 175,
        mass: 174.95362
      }],
      symbol: 'Ho',
      mass: 164.9303288,
      name: 'Holmium',
      monoisotopicMass: 164.9303288
    }, {
      number: 68,
      isotopes: [{
        nominal: 142,
        mass: 141.9701
      }, {
        nominal: 143,
        mass: 142.96662
      }, {
        nominal: 144,
        mass: 143.9607
      }, {
        nominal: 145,
        mass: 144.95805
      }, {
        nominal: 146,
        mass: 145.9524184
      }, {
        nominal: 147,
        mass: 146.949964
      }, {
        nominal: 148,
        mass: 147.944735
      }, {
        nominal: 149,
        mass: 148.942306
      }, {
        nominal: 150,
        mass: 149.937916
      }, {
        nominal: 151,
        mass: 150.937449
      }, {
        nominal: 152,
        mass: 151.935057
      }, {
        nominal: 153,
        mass: 152.93508
      }, {
        nominal: 154,
        mass: 153.9327908
      }, {
        nominal: 155,
        mass: 154.9332159
      }, {
        nominal: 156,
        mass: 155.931067
      }, {
        nominal: 157,
        mass: 156.931949
      }, {
        nominal: 158,
        mass: 157.929893
      }, {
        nominal: 159,
        mass: 158.9306918
      }, {
        nominal: 160,
        mass: 159.929077
      }, {
        nominal: 161,
        mass: 160.9300046
      }, {
        nominal: 162,
        mass: 161.9287884,
        abundance: 0.00139
      }, {
        nominal: 163,
        mass: 162.9300408
      }, {
        nominal: 164,
        mass: 163.9292088,
        abundance: 0.01601
      }, {
        nominal: 165,
        mass: 164.9307345
      }, {
        nominal: 166,
        mass: 165.9302995,
        abundance: 0.33503
      }, {
        nominal: 167,
        mass: 166.9320546,
        abundance: 0.22869
      }, {
        nominal: 168,
        mass: 167.9323767,
        abundance: 0.26978
      }, {
        nominal: 169,
        mass: 168.9345968
      }, {
        nominal: 170,
        mass: 169.9354702,
        abundance: 0.1491
      }, {
        nominal: 171,
        mass: 170.9380357
      }, {
        nominal: 172,
        mass: 171.9393619
      }, {
        nominal: 173,
        mass: 172.9424
      }, {
        nominal: 174,
        mass: 173.94423
      }, {
        nominal: 175,
        mass: 174.94777
      }, {
        nominal: 176,
        mass: 175.94994
      }, {
        nominal: 177,
        mass: 176.95399
      }],
      symbol: 'Er',
      mass: 167.259082649669,
      name: 'Erbium',
      monoisotopicMass: 165.9302995
    }, {
      number: 69,
      isotopes: [{
        nominal: 144,
        mass: 143.97628
      }, {
        nominal: 145,
        mass: 144.97039
      }, {
        nominal: 146,
        mass: 145.96684
      }, {
        nominal: 147,
        mass: 146.9613799
      }, {
        nominal: 148,
        mass: 147.958384
      }, {
        nominal: 149,
        mass: 148.95289
      }, {
        nominal: 150,
        mass: 149.95009
      }, {
        nominal: 151,
        mass: 150.945488
      }, {
        nominal: 152,
        mass: 151.944422
      }, {
        nominal: 153,
        mass: 152.94204
      }, {
        nominal: 154,
        mass: 153.94157
      }, {
        nominal: 155,
        mass: 154.93921
      }, {
        nominal: 156,
        mass: 155.938992
      }, {
        nominal: 157,
        mass: 156.936944
      }, {
        nominal: 158,
        mass: 157.93698
      }, {
        nominal: 159,
        mass: 158.934975
      }, {
        nominal: 160,
        mass: 159.935263
      }, {
        nominal: 161,
        mass: 160.933549
      }, {
        nominal: 162,
        mass: 161.934002
      }, {
        nominal: 163,
        mass: 162.9326592
      }, {
        nominal: 164,
        mass: 163.933544
      }, {
        nominal: 165,
        mass: 164.9324431
      }, {
        nominal: 166,
        mass: 165.933561
      }, {
        nominal: 167,
        mass: 166.9328562
      }, {
        nominal: 168,
        mass: 167.9341774
      }, {
        nominal: 169,
        mass: 168.9342179,
        abundance: 1
      }, {
        nominal: 170,
        mass: 169.935806
      }, {
        nominal: 171,
        mass: 170.9364339
      }, {
        nominal: 172,
        mass: 171.9384055
      }, {
        nominal: 173,
        mass: 172.9396084
      }, {
        nominal: 174,
        mass: 173.942173
      }, {
        nominal: 175,
        mass: 174.943841
      }, {
        nominal: 176,
        mass: 175.947
      }, {
        nominal: 177,
        mass: 176.94904
      }, {
        nominal: 178,
        mass: 177.95264
      }, {
        nominal: 179,
        mass: 178.95534
      }],
      symbol: 'Tm',
      mass: 168.9342179,
      name: 'Thulium',
      monoisotopicMass: 168.9342179
    }, {
      number: 70,
      isotopes: [{
        nominal: 148,
        mass: 147.96758
      }, {
        nominal: 149,
        mass: 148.96436
      }, {
        nominal: 150,
        mass: 149.95852
      }, {
        nominal: 151,
        mass: 150.9554
      }, {
        nominal: 152,
        mass: 151.95027
      }, {
        nominal: 153,
        mass: 152.94932
      }, {
        nominal: 154,
        mass: 153.946396
      }, {
        nominal: 155,
        mass: 154.945783
      }, {
        nominal: 156,
        mass: 155.942825
      }, {
        nominal: 157,
        mass: 156.942645
      }, {
        nominal: 158,
        mass: 157.9398705
      }, {
        nominal: 159,
        mass: 158.940055
      }, {
        nominal: 160,
        mass: 159.937557
      }, {
        nominal: 161,
        mass: 160.937907
      }, {
        nominal: 162,
        mass: 161.935774
      }, {
        nominal: 163,
        mass: 162.93634
      }, {
        nominal: 164,
        mass: 163.934495
      }, {
        nominal: 165,
        mass: 164.93527
      }, {
        nominal: 166,
        mass: 165.9338747
      }, {
        nominal: 167,
        mass: 166.934953
      }, {
        nominal: 168,
        mass: 167.9338896,
        abundance: 0.00123
      }, {
        nominal: 169,
        mass: 168.9351825
      }, {
        nominal: 170,
        mass: 169.9347664,
        abundance: 0.02982
      }, {
        nominal: 171,
        mass: 170.9363302,
        abundance: 0.1409
      }, {
        nominal: 172,
        mass: 171.9363859,
        abundance: 0.2168
      }, {
        nominal: 173,
        mass: 172.9382151,
        abundance: 0.16103
      }, {
        nominal: 174,
        mass: 173.9388664,
        abundance: 0.32026
      }, {
        nominal: 175,
        mass: 174.9412808
      }, {
        nominal: 176,
        mass: 175.9425764,
        abundance: 0.12996
      }, {
        nominal: 177,
        mass: 176.9452656
      }, {
        nominal: 178,
        mass: 177.946651
      }, {
        nominal: 179,
        mass: 178.95004
      }, {
        nominal: 180,
        mass: 179.95212
      }, {
        nominal: 181,
        mass: 180.95589
      }],
      symbol: 'Yb',
      mass: 173.05415016631702,
      name: 'Ytterbium',
      monoisotopicMass: 173.9388664
    }, {
      number: 71,
      isotopes: [{
        nominal: 150,
        mass: 149.97355
      }, {
        nominal: 151,
        mass: 150.96768
      }, {
        nominal: 152,
        mass: 151.96412
      }, {
        nominal: 153,
        mass: 152.95875
      }, {
        nominal: 154,
        mass: 153.95736
      }, {
        nominal: 155,
        mass: 154.954321
      }, {
        nominal: 156,
        mass: 155.953033
      }, {
        nominal: 157,
        mass: 156.950127
      }, {
        nominal: 158,
        mass: 157.949316
      }, {
        nominal: 159,
        mass: 158.946636
      }, {
        nominal: 160,
        mass: 159.946033
      }, {
        nominal: 161,
        mass: 160.943572
      }, {
        nominal: 162,
        mass: 161.943283
      }, {
        nominal: 163,
        mass: 162.941179
      }, {
        nominal: 164,
        mass: 163.941339
      }, {
        nominal: 165,
        mass: 164.939407
      }, {
        nominal: 166,
        mass: 165.939859
      }, {
        nominal: 167,
        mass: 166.93827
      }, {
        nominal: 168,
        mass: 167.938736
      }, {
        nominal: 169,
        mass: 168.9376441
      }, {
        nominal: 170,
        mass: 169.938478
      }, {
        nominal: 171,
        mass: 170.937917
      }, {
        nominal: 172,
        mass: 171.9390891
      }, {
        nominal: 173,
        mass: 172.938934
      }, {
        nominal: 174,
        mass: 173.9403409
      }, {
        nominal: 175,
        mass: 174.9407752,
        abundance: 0.97401
      }, {
        nominal: 176,
        mass: 175.9426897,
        abundance: 0.02599
      }, {
        nominal: 177,
        mass: 176.9437615
      }, {
        nominal: 178,
        mass: 177.945958
      }, {
        nominal: 179,
        mass: 178.9473309
      }, {
        nominal: 180,
        mass: 179.949888
      }, {
        nominal: 181,
        mass: 180.95191
      }, {
        nominal: 182,
        mass: 181.95504
      }, {
        nominal: 183,
        mass: 182.957363
      }, {
        nominal: 184,
        mass: 183.96091
      }, {
        nominal: 185,
        mass: 184.96362
      }],
      symbol: 'Lu',
      mass: 174.96681495785498,
      name: 'Lutetium',
      monoisotopicMass: 174.9407752
    }, {
      number: 72,
      isotopes: [{
        nominal: 153,
        mass: 152.97069
      }, {
        nominal: 154,
        mass: 153.96486
      }, {
        nominal: 155,
        mass: 154.96311
      }, {
        nominal: 156,
        mass: 155.95935
      }, {
        nominal: 157,
        mass: 156.95824
      }, {
        nominal: 158,
        mass: 157.954801
      }, {
        nominal: 159,
        mass: 158.953996
      }, {
        nominal: 160,
        mass: 159.950691
      }, {
        nominal: 161,
        mass: 160.950278
      }, {
        nominal: 162,
        mass: 161.9472148
      }, {
        nominal: 163,
        mass: 162.947113
      }, {
        nominal: 164,
        mass: 163.944371
      }, {
        nominal: 165,
        mass: 164.944567
      }, {
        nominal: 166,
        mass: 165.94218
      }, {
        nominal: 167,
        mass: 166.9426
      }, {
        nominal: 168,
        mass: 167.940568
      }, {
        nominal: 169,
        mass: 168.941259
      }, {
        nominal: 170,
        mass: 169.939609
      }, {
        nominal: 171,
        mass: 170.940492
      }, {
        nominal: 172,
        mass: 171.93945
      }, {
        nominal: 173,
        mass: 172.940513
      }, {
        nominal: 174,
        mass: 173.9400461,
        abundance: 0.0016
      }, {
        nominal: 175,
        mass: 174.9415092
      }, {
        nominal: 176,
        mass: 175.9414076,
        abundance: 0.0526
      }, {
        nominal: 177,
        mass: 176.9432277,
        abundance: 0.186
      }, {
        nominal: 178,
        mass: 177.9437058,
        abundance: 0.2728
      }, {
        nominal: 179,
        mass: 178.9458232,
        abundance: 0.1362
      }, {
        nominal: 180,
        mass: 179.946557,
        abundance: 0.3508
      }, {
        nominal: 181,
        mass: 180.9491083
      }, {
        nominal: 182,
        mass: 181.9505612
      }, {
        nominal: 183,
        mass: 182.95353
      }, {
        nominal: 184,
        mass: 183.955446
      }, {
        nominal: 185,
        mass: 184.958862
      }, {
        nominal: 186,
        mass: 185.960897
      }, {
        nominal: 187,
        mass: 186.96477
      }, {
        nominal: 188,
        mass: 187.96685
      }, {
        nominal: 189,
        mass: 188.97084
      }],
      symbol: 'Hf',
      mass: 178.4849787234,
      name: 'Hafnium',
      monoisotopicMass: 179.946557
    }, {
      number: 73,
      isotopes: [{
        nominal: 155,
        mass: 154.97424
      }, {
        nominal: 156,
        mass: 155.97203
      }, {
        nominal: 157,
        mass: 156.96818
      }, {
        nominal: 158,
        mass: 157.96654
      }, {
        nominal: 159,
        mass: 158.963023
      }, {
        nominal: 160,
        mass: 159.961488
      }, {
        nominal: 161,
        mass: 160.958452
      }, {
        nominal: 162,
        mass: 161.957294
      }, {
        nominal: 163,
        mass: 162.954337
      }, {
        nominal: 164,
        mass: 163.953534
      }, {
        nominal: 165,
        mass: 164.950781
      }, {
        nominal: 166,
        mass: 165.950512
      }, {
        nominal: 167,
        mass: 166.948093
      }, {
        nominal: 168,
        mass: 167.948047
      }, {
        nominal: 169,
        mass: 168.946011
      }, {
        nominal: 170,
        mass: 169.946175
      }, {
        nominal: 171,
        mass: 170.944476
      }, {
        nominal: 172,
        mass: 171.944895
      }, {
        nominal: 173,
        mass: 172.94375
      }, {
        nominal: 174,
        mass: 173.944454
      }, {
        nominal: 175,
        mass: 174.943737
      }, {
        nominal: 176,
        mass: 175.944857
      }, {
        nominal: 177,
        mass: 176.9444795
      }, {
        nominal: 178,
        mass: 177.945678
      }, {
        nominal: 179,
        mass: 178.9459366
      }, {
        nominal: 180,
        mass: 179.9474648,
        abundance: 0.0001201
      }, {
        nominal: 181,
        mass: 180.9479958,
        abundance: 0.9998799
      }, {
        nominal: 182,
        mass: 181.9501519
      }, {
        nominal: 183,
        mass: 182.9513726
      }, {
        nominal: 184,
        mass: 183.954008
      }, {
        nominal: 185,
        mass: 184.955559
      }, {
        nominal: 186,
        mass: 185.958551
      }, {
        nominal: 187,
        mass: 186.960386
      }, {
        nominal: 188,
        mass: 187.963916
      }, {
        nominal: 189,
        mass: 188.96583
      }, {
        nominal: 190,
        mass: 189.96939
      }, {
        nominal: 191,
        mass: 190.97156
      }, {
        nominal: 192,
        mass: 191.97514
      }],
      symbol: 'Ta',
      mass: 180.9478756362269,
      name: 'Tantalum',
      monoisotopicMass: 180.9479958
    }, {
      number: 74,
      isotopes: [{
        nominal: 157,
        mass: 156.97884
      }, {
        nominal: 158,
        mass: 157.97456
      }, {
        nominal: 159,
        mass: 158.97264
      }, {
        nominal: 160,
        mass: 159.96846
      }, {
        nominal: 161,
        mass: 160.9672
      }, {
        nominal: 162,
        mass: 161.963499
      }, {
        nominal: 163,
        mass: 162.962524
      }, {
        nominal: 164,
        mass: 163.958961
      }, {
        nominal: 165,
        mass: 164.958281
      }, {
        nominal: 166,
        mass: 165.955031
      }, {
        nominal: 167,
        mass: 166.954805
      }, {
        nominal: 168,
        mass: 167.951806
      }, {
        nominal: 169,
        mass: 168.951779
      }, {
        nominal: 170,
        mass: 169.949232
      }, {
        nominal: 171,
        mass: 170.949451
      }, {
        nominal: 172,
        mass: 171.947292
      }, {
        nominal: 173,
        mass: 172.947689
      }, {
        nominal: 174,
        mass: 173.946079
      }, {
        nominal: 175,
        mass: 174.946717
      }, {
        nominal: 176,
        mass: 175.945634
      }, {
        nominal: 177,
        mass: 176.946643
      }, {
        nominal: 178,
        mass: 177.945883
      }, {
        nominal: 179,
        mass: 178.947077
      }, {
        nominal: 180,
        mass: 179.9467108,
        abundance: 0.0012
      }, {
        nominal: 181,
        mass: 180.9481978
      }, {
        nominal: 182,
        mass: 181.94820394,
        abundance: 0.265
      }, {
        nominal: 183,
        mass: 182.95022275,
        abundance: 0.1431
      }, {
        nominal: 184,
        mass: 183.95093092,
        abundance: 0.3064
      }, {
        nominal: 185,
        mass: 184.95341897
      }, {
        nominal: 186,
        mass: 185.9543628,
        abundance: 0.2843
      }, {
        nominal: 187,
        mass: 186.9571588
      }, {
        nominal: 188,
        mass: 187.9584862
      }, {
        nominal: 189,
        mass: 188.961763
      }, {
        nominal: 190,
        mass: 189.963091
      }, {
        nominal: 191,
        mass: 190.966531
      }, {
        nominal: 192,
        mass: 191.96817
      }, {
        nominal: 193,
        mass: 192.97178
      }, {
        nominal: 194,
        mass: 193.97367
      }],
      symbol: 'W',
      mass: 183.841777550513,
      name: 'Tungsten',
      monoisotopicMass: 183.95093092
    }, {
      number: 75,
      isotopes: [{
        nominal: 159,
        mass: 158.98418
      }, {
        nominal: 160,
        mass: 159.98182
      }, {
        nominal: 161,
        mass: 160.97757
      }, {
        nominal: 162,
        mass: 161.97584
      }, {
        nominal: 163,
        mass: 162.97208
      }, {
        nominal: 164,
        mass: 163.970453
      }, {
        nominal: 165,
        mass: 164.967103
      }, {
        nominal: 166,
        mass: 165.965761
      }, {
        nominal: 167,
        mass: 166.962595
      }, {
        nominal: 168,
        mass: 167.961573
      }, {
        nominal: 169,
        mass: 168.958766
      }, {
        nominal: 170,
        mass: 169.95822
      }, {
        nominal: 171,
        mass: 170.955716
      }, {
        nominal: 172,
        mass: 171.95542
      }, {
        nominal: 173,
        mass: 172.953243
      }, {
        nominal: 174,
        mass: 173.953115
      }, {
        nominal: 175,
        mass: 174.951381
      }, {
        nominal: 176,
        mass: 175.951623
      }, {
        nominal: 177,
        mass: 176.950328
      }, {
        nominal: 178,
        mass: 177.950989
      }, {
        nominal: 179,
        mass: 178.949989
      }, {
        nominal: 180,
        mass: 179.950792
      }, {
        nominal: 181,
        mass: 180.950058
      }, {
        nominal: 182,
        mass: 181.95121
      }, {
        nominal: 183,
        mass: 182.9508196
      }, {
        nominal: 184,
        mass: 183.9525228
      }, {
        nominal: 185,
        mass: 184.9529545,
        abundance: 0.374
      }, {
        nominal: 186,
        mass: 185.9549856
      }, {
        nominal: 187,
        mass: 186.9557501,
        abundance: 0.626
      }, {
        nominal: 188,
        mass: 187.9581115
      }, {
        nominal: 189,
        mass: 188.959226
      }, {
        nominal: 190,
        mass: 189.961744
      }, {
        nominal: 191,
        mass: 190.963122
      }, {
        nominal: 192,
        mass: 191.966088
      }, {
        nominal: 193,
        mass: 192.967541
      }, {
        nominal: 194,
        mass: 193.97076
      }, {
        nominal: 195,
        mass: 194.97254
      }, {
        nominal: 196,
        mass: 195.9758
      }, {
        nominal: 197,
        mass: 196.97799
      }, {
        nominal: 198,
        mass: 197.9816
      }],
      symbol: 'Re',
      mass: 186.20670454560002,
      name: 'Rhenium',
      monoisotopicMass: 186.9557501
    }, {
      number: 76,
      isotopes: [{
        nominal: 161,
        mass: 160.98903
      }, {
        nominal: 162,
        mass: 161.98443
      }, {
        nominal: 163,
        mass: 162.98241
      }, {
        nominal: 164,
        mass: 163.97802
      }, {
        nominal: 165,
        mass: 164.9766
      }, {
        nominal: 166,
        mass: 165.972692
      }, {
        nominal: 167,
        mass: 166.971549
      }, {
        nominal: 168,
        mass: 167.967808
      }, {
        nominal: 169,
        mass: 168.967018
      }, {
        nominal: 170,
        mass: 169.963578
      }, {
        nominal: 171,
        mass: 170.963174
      }, {
        nominal: 172,
        mass: 171.960017
      }, {
        nominal: 173,
        mass: 172.959808
      }, {
        nominal: 174,
        mass: 173.957064
      }, {
        nominal: 175,
        mass: 174.956945
      }, {
        nominal: 176,
        mass: 175.954806
      }, {
        nominal: 177,
        mass: 176.954966
      }, {
        nominal: 178,
        mass: 177.953254
      }, {
        nominal: 179,
        mass: 178.953817
      }, {
        nominal: 180,
        mass: 179.952375
      }, {
        nominal: 181,
        mass: 180.953247
      }, {
        nominal: 182,
        mass: 181.95211
      }, {
        nominal: 183,
        mass: 182.953125
      }, {
        nominal: 184,
        mass: 183.9524885,
        abundance: 0.0002
      }, {
        nominal: 185,
        mass: 184.9540417
      }, {
        nominal: 186,
        mass: 185.953835,
        abundance: 0.0159
      }, {
        nominal: 187,
        mass: 186.9557474,
        abundance: 0.0196
      }, {
        nominal: 188,
        mass: 187.9558352,
        abundance: 0.1324
      }, {
        nominal: 189,
        mass: 188.9581442,
        abundance: 0.1615
      }, {
        nominal: 190,
        mass: 189.9584437,
        abundance: 0.2626
      }, {
        nominal: 191,
        mass: 190.9609264
      }, {
        nominal: 192,
        mass: 191.961477,
        abundance: 0.4078
      }, {
        nominal: 193,
        mass: 192.9641479
      }, {
        nominal: 194,
        mass: 193.9651772
      }, {
        nominal: 195,
        mass: 194.968318
      }, {
        nominal: 196,
        mass: 195.969641
      }, {
        nominal: 197,
        mass: 196.97283
      }, {
        nominal: 198,
        mass: 197.97441
      }, {
        nominal: 199,
        mass: 198.97801
      }, {
        nominal: 200,
        mass: 199.97984
      }, {
        nominal: 201,
        mass: 200.98364
      }, {
        nominal: 202,
        mass: 201.98595
      }],
      symbol: 'Os',
      mass: 190.22485962823998,
      name: 'Osmium',
      monoisotopicMass: 191.961477
    }, {
      number: 77,
      isotopes: [{
        nominal: 164,
        mass: 163.99191
      }, {
        nominal: 165,
        mass: 164.9875
      }, {
        nominal: 166,
        mass: 165.98566
      }, {
        nominal: 167,
        mass: 166.981666
      }, {
        nominal: 168,
        mass: 167.979907
      }, {
        nominal: 169,
        mass: 168.976298
      }, {
        nominal: 170,
        mass: 169.974922
      }, {
        nominal: 171,
        mass: 170.97164
      }, {
        nominal: 172,
        mass: 171.970607
      }, {
        nominal: 173,
        mass: 172.967506
      }, {
        nominal: 174,
        mass: 173.966861
      }, {
        nominal: 175,
        mass: 174.96415
      }, {
        nominal: 176,
        mass: 175.96365
      }, {
        nominal: 177,
        mass: 176.961301
      }, {
        nominal: 178,
        mass: 177.961082
      }, {
        nominal: 179,
        mass: 178.95912
      }, {
        nominal: 180,
        mass: 179.959229
      }, {
        nominal: 181,
        mass: 180.957625
      }, {
        nominal: 182,
        mass: 181.958076
      }, {
        nominal: 183,
        mass: 182.95684
      }, {
        nominal: 184,
        mass: 183.957476
      }, {
        nominal: 185,
        mass: 184.956698
      }, {
        nominal: 186,
        mass: 185.957944
      }, {
        nominal: 187,
        mass: 186.957542
      }, {
        nominal: 188,
        mass: 187.958828
      }, {
        nominal: 189,
        mass: 188.958715
      }, {
        nominal: 190,
        mass: 189.9605412
      }, {
        nominal: 191,
        mass: 190.9605893,
        abundance: 0.373
      }, {
        nominal: 192,
        mass: 191.9626002
      }, {
        nominal: 193,
        mass: 192.9629216,
        abundance: 0.627
      }, {
        nominal: 194,
        mass: 193.9650735
      }, {
        nominal: 195,
        mass: 194.9659747
      }, {
        nominal: 196,
        mass: 195.968397
      }, {
        nominal: 197,
        mass: 196.969655
      }, {
        nominal: 198,
        mass: 197.97228
      }, {
        nominal: 199,
        mass: 198.973805
      }, {
        nominal: 200,
        mass: 199.9768
      }, {
        nominal: 201,
        mass: 200.97864
      }, {
        nominal: 202,
        mass: 201.98199
      }, {
        nominal: 203,
        mass: 202.98423
      }, {
        nominal: 204,
        mass: 203.9896
      }],
      symbol: 'Ir',
      mass: 192.2160516521,
      name: 'Iridium',
      monoisotopicMass: 192.9629216
    }, {
      number: 78,
      isotopes: [{
        nominal: 166,
        mass: 165.99486
      }, {
        nominal: 167,
        mass: 166.99269
      }, {
        nominal: 168,
        mass: 167.98813
      }, {
        nominal: 169,
        mass: 168.98657
      }, {
        nominal: 170,
        mass: 169.982496
      }, {
        nominal: 171,
        mass: 170.981245
      }, {
        nominal: 172,
        mass: 171.977351
      }, {
        nominal: 173,
        mass: 172.976443
      }, {
        nominal: 174,
        mass: 173.97282
      }, {
        nominal: 175,
        mass: 174.97241
      }, {
        nominal: 176,
        mass: 175.968938
      }, {
        nominal: 177,
        mass: 176.96847
      }, {
        nominal: 178,
        mass: 177.96565
      }, {
        nominal: 179,
        mass: 178.965359
      }, {
        nominal: 180,
        mass: 179.963032
      }, {
        nominal: 181,
        mass: 180.963098
      }, {
        nominal: 182,
        mass: 181.961172
      }, {
        nominal: 183,
        mass: 182.961597
      }, {
        nominal: 184,
        mass: 183.959915
      }, {
        nominal: 185,
        mass: 184.960614
      }, {
        nominal: 186,
        mass: 185.959351
      }, {
        nominal: 187,
        mass: 186.960617
      }, {
        nominal: 188,
        mass: 187.9593889
      }, {
        nominal: 189,
        mass: 188.960831
      }, {
        nominal: 190,
        mass: 189.9599297,
        abundance: 0.00012
      }, {
        nominal: 191,
        mass: 190.9616729
      }, {
        nominal: 192,
        mass: 191.9610387,
        abundance: 0.00782
      }, {
        nominal: 193,
        mass: 192.9629824
      }, {
        nominal: 194,
        mass: 193.9626809,
        abundance: 0.3286
      }, {
        nominal: 195,
        mass: 194.9647917,
        abundance: 0.3378
      }, {
        nominal: 196,
        mass: 195.96495209,
        abundance: 0.2521
      }, {
        nominal: 197,
        mass: 196.96734069
      }, {
        nominal: 198,
        mass: 197.9678949,
        abundance: 0.07356
      }, {
        nominal: 199,
        mass: 198.9705952
      }, {
        nominal: 200,
        mass: 199.971443
      }, {
        nominal: 201,
        mass: 200.974513
      }, {
        nominal: 202,
        mass: 201.975639
      }, {
        nominal: 203,
        mass: 202.97893
      }, {
        nominal: 204,
        mass: 203.98076
      }, {
        nominal: 205,
        mass: 204.98608
      }, {
        nominal: 206,
        mass: 205.98966
      }],
      symbol: 'Pt',
      mass: 195.084456864931,
      name: 'Platinum',
      monoisotopicMass: 194.9647917
    }, {
      number: 79,
      isotopes: [{
        nominal: 169,
        mass: 168.99808
      }, {
        nominal: 170,
        mass: 169.99597
      }, {
        nominal: 171,
        mass: 170.991876
      }, {
        nominal: 172,
        mass: 171.989942
      }, {
        nominal: 173,
        mass: 172.986241
      }, {
        nominal: 174,
        mass: 173.984717
      }, {
        nominal: 175,
        mass: 174.981304
      }, {
        nominal: 176,
        mass: 175.98025
      }, {
        nominal: 177,
        mass: 176.97687
      }, {
        nominal: 178,
        mass: 177.976032
      }, {
        nominal: 179,
        mass: 178.973174
      }, {
        nominal: 180,
        mass: 179.972523
      }, {
        nominal: 181,
        mass: 180.970079
      }, {
        nominal: 182,
        mass: 181.969618
      }, {
        nominal: 183,
        mass: 182.967591
      }, {
        nominal: 184,
        mass: 183.967452
      }, {
        nominal: 185,
        mass: 184.96579
      }, {
        nominal: 186,
        mass: 185.965953
      }, {
        nominal: 187,
        mass: 186.964543
      }, {
        nominal: 188,
        mass: 187.965349
      }, {
        nominal: 189,
        mass: 188.963948
      }, {
        nominal: 190,
        mass: 189.964698
      }, {
        nominal: 191,
        mass: 190.963702
      }, {
        nominal: 192,
        mass: 191.964814
      }, {
        nominal: 193,
        mass: 192.9641373
      }, {
        nominal: 194,
        mass: 193.9654178
      }, {
        nominal: 195,
        mass: 194.9650352
      }, {
        nominal: 196,
        mass: 195.9665699
      }, {
        nominal: 197,
        mass: 196.96656879,
        abundance: 1
      }, {
        nominal: 198,
        mass: 197.96824242
      }, {
        nominal: 199,
        mass: 198.96876528
      }, {
        nominal: 200,
        mass: 199.970756
      }, {
        nominal: 201,
        mass: 200.9716575
      }, {
        nominal: 202,
        mass: 201.973856
      }, {
        nominal: 203,
        mass: 202.9751544
      }, {
        nominal: 204,
        mass: 203.97783
      }, {
        nominal: 205,
        mass: 204.97985
      }, {
        nominal: 206,
        mass: 205.98474
      }, {
        nominal: 207,
        mass: 206.9884
      }, {
        nominal: 208,
        mass: 207.99345
      }, {
        nominal: 209,
        mass: 208.99735
      }, {
        nominal: 210,
        mass: 210.0025
      }],
      symbol: 'Au',
      mass: 196.96656879,
      name: 'Gold',
      monoisotopicMass: 196.96656879
    }, {
      number: 80,
      isotopes: [{
        nominal: 171,
        mass: 171.00353
      }, {
        nominal: 172,
        mass: 171.99881
      }, {
        nominal: 173,
        mass: 172.99709
      }, {
        nominal: 174,
        mass: 173.992865
      }, {
        nominal: 175,
        mass: 174.991441
      }, {
        nominal: 176,
        mass: 175.987361
      }, {
        nominal: 177,
        mass: 176.986277
      }, {
        nominal: 178,
        mass: 177.982484
      }, {
        nominal: 179,
        mass: 178.981831
      }, {
        nominal: 180,
        mass: 179.97826
      }, {
        nominal: 181,
        mass: 180.977819
      }, {
        nominal: 182,
        mass: 181.974689
      }, {
        nominal: 183,
        mass: 182.9744448
      }, {
        nominal: 184,
        mass: 183.971714
      }, {
        nominal: 185,
        mass: 184.971899
      }, {
        nominal: 186,
        mass: 185.969362
      }, {
        nominal: 187,
        mass: 186.969814
      }, {
        nominal: 188,
        mass: 187.967567
      }, {
        nominal: 189,
        mass: 188.968195
      }, {
        nominal: 190,
        mass: 189.966323
      }, {
        nominal: 191,
        mass: 190.967157
      }, {
        nominal: 192,
        mass: 191.965635
      }, {
        nominal: 193,
        mass: 192.966653
      }, {
        nominal: 194,
        mass: 193.9654491
      }, {
        nominal: 195,
        mass: 194.966721
      }, {
        nominal: 196,
        mass: 195.9658326,
        abundance: 0.0015
      }, {
        nominal: 197,
        mass: 196.9672128
      }, {
        nominal: 198,
        mass: 197.9667686,
        abundance: 0.0997
      }, {
        nominal: 199,
        mass: 198.96828064,
        abundance: 0.1687
      }, {
        nominal: 200,
        mass: 199.96832659,
        abundance: 0.231
      }, {
        nominal: 201,
        mass: 200.97030284,
        abundance: 0.1318
      }, {
        nominal: 202,
        mass: 201.9706434,
        abundance: 0.2986
      }, {
        nominal: 203,
        mass: 202.9728728
      }, {
        nominal: 204,
        mass: 203.97349398,
        abundance: 0.0687
      }, {
        nominal: 205,
        mass: 204.9760734
      }, {
        nominal: 206,
        mass: 205.977514
      }, {
        nominal: 207,
        mass: 206.9823
      }, {
        nominal: 208,
        mass: 207.985759
      }, {
        nominal: 209,
        mass: 208.99072
      }, {
        nominal: 210,
        mass: 209.99424
      }, {
        nominal: 211,
        mass: 210.99933
      }, {
        nominal: 212,
        mass: 212.00296
      }, {
        nominal: 213,
        mass: 213.00823
      }, {
        nominal: 214,
        mass: 214.012
      }, {
        nominal: 215,
        mass: 215.0174
      }, {
        nominal: 216,
        mass: 216.02132
      }],
      symbol: 'Hg',
      mass: 200.59916703455602,
      name: 'Mercury',
      monoisotopicMass: 201.9706434
    }, {
      number: 81,
      isotopes: [{
        nominal: 176,
        mass: 176.000624
      }, {
        nominal: 177,
        mass: 176.996431
      }, {
        nominal: 178,
        mass: 177.99485
      }, {
        nominal: 179,
        mass: 178.991111
      }, {
        nominal: 180,
        mass: 179.990057
      }, {
        nominal: 181,
        mass: 180.98626
      }, {
        nominal: 182,
        mass: 181.985713
      }, {
        nominal: 183,
        mass: 182.982193
      }, {
        nominal: 184,
        mass: 183.981886
      }, {
        nominal: 185,
        mass: 184.978789
      }, {
        nominal: 186,
        mass: 185.978651
      }, {
        nominal: 187,
        mass: 186.9759063
      }, {
        nominal: 188,
        mass: 187.976021
      }, {
        nominal: 189,
        mass: 188.973588
      }, {
        nominal: 190,
        mass: 189.973828
      }, {
        nominal: 191,
        mass: 190.9717842
      }, {
        nominal: 192,
        mass: 191.972225
      }, {
        nominal: 193,
        mass: 192.970502
      }, {
        nominal: 194,
        mass: 193.971081
      }, {
        nominal: 195,
        mass: 194.969774
      }, {
        nominal: 196,
        mass: 195.970481
      }, {
        nominal: 197,
        mass: 196.969576
      }, {
        nominal: 198,
        mass: 197.970483
      }, {
        nominal: 199,
        mass: 198.969877
      }, {
        nominal: 200,
        mass: 199.9709633
      }, {
        nominal: 201,
        mass: 200.970822
      }, {
        nominal: 202,
        mass: 201.972102
      }, {
        nominal: 203,
        mass: 202.9723446,
        abundance: 0.2952
      }, {
        nominal: 204,
        mass: 203.9738639
      }, {
        nominal: 205,
        mass: 204.9744278,
        abundance: 0.7048
      }, {
        nominal: 206,
        mass: 205.9761106
      }, {
        nominal: 207,
        mass: 206.9774197
      }, {
        nominal: 208,
        mass: 207.982019
      }, {
        nominal: 209,
        mass: 208.9853594
      }, {
        nominal: 210,
        mass: 209.990074
      }, {
        nominal: 211,
        mass: 210.993475
      }, {
        nominal: 212,
        mass: 211.99834
      }, {
        nominal: 213,
        mass: 213.001915
      }, {
        nominal: 214,
        mass: 214.00694
      }, {
        nominal: 215,
        mass: 215.01064
      }, {
        nominal: 216,
        mass: 216.0158
      }, {
        nominal: 217,
        mass: 217.01966
      }, {
        nominal: 218,
        mass: 218.02479
      }],
      symbol: 'Tl',
      mass: 204.38341283936,
      name: 'Thallium',
      monoisotopicMass: 204.9744278
    }, {
      number: 82,
      isotopes: [{
        nominal: 178,
        mass: 178.003831
      }, {
        nominal: 179,
        mass: 179.002201
      }, {
        nominal: 180,
        mass: 179.997928
      }, {
        nominal: 181,
        mass: 180.996653
      }, {
        nominal: 182,
        mass: 181.992672
      }, {
        nominal: 183,
        mass: 182.991872
      }, {
        nominal: 184,
        mass: 183.988136
      }, {
        nominal: 185,
        mass: 184.98761
      }, {
        nominal: 186,
        mass: 185.984238
      }, {
        nominal: 187,
        mass: 186.9839109
      }, {
        nominal: 188,
        mass: 187.980875
      }, {
        nominal: 189,
        mass: 188.980807
      }, {
        nominal: 190,
        mass: 189.978082
      }, {
        nominal: 191,
        mass: 190.978276
      }, {
        nominal: 192,
        mass: 191.975775
      }, {
        nominal: 193,
        mass: 192.976173
      }, {
        nominal: 194,
        mass: 193.974012
      }, {
        nominal: 195,
        mass: 194.974543
      }, {
        nominal: 196,
        mass: 195.972774
      }, {
        nominal: 197,
        mass: 196.9734312
      }, {
        nominal: 198,
        mass: 197.972034
      }, {
        nominal: 199,
        mass: 198.972913
      }, {
        nominal: 200,
        mass: 199.971819
      }, {
        nominal: 201,
        mass: 200.972883
      }, {
        nominal: 202,
        mass: 201.972152
      }, {
        nominal: 203,
        mass: 202.9733911
      }, {
        nominal: 204,
        mass: 203.973044,
        abundance: 0.014
      }, {
        nominal: 205,
        mass: 204.9744822
      }, {
        nominal: 206,
        mass: 205.9744657,
        abundance: 0.241
      }, {
        nominal: 207,
        mass: 206.9758973,
        abundance: 0.221
      }, {
        nominal: 208,
        mass: 207.9766525,
        abundance: 0.524
      }, {
        nominal: 209,
        mass: 208.9810905
      }, {
        nominal: 210,
        mass: 209.9841889
      }, {
        nominal: 211,
        mass: 210.9887371
      }, {
        nominal: 212,
        mass: 211.9918977
      }, {
        nominal: 213,
        mass: 212.9965629
      }, {
        nominal: 214,
        mass: 213.9998059
      }, {
        nominal: 215,
        mass: 215.00474
      }, {
        nominal: 216,
        mass: 216.00803
      }, {
        nominal: 217,
        mass: 217.01314
      }, {
        nominal: 218,
        mass: 218.01659
      }, {
        nominal: 219,
        mass: 219.02177
      }, {
        nominal: 220,
        mass: 220.02541
      }],
      symbol: 'Pb',
      mass: 207.216908063,
      name: 'Lead',
      monoisotopicMass: 207.9766525
    }, {
      number: 83,
      isotopes: [{
        nominal: 184,
        mass: 184.001275
      }, {
        nominal: 185,
        mass: 184.9976
      }, {
        nominal: 186,
        mass: 185.996644
      }, {
        nominal: 187,
        mass: 186.993147
      }, {
        nominal: 188,
        mass: 187.992287
      }, {
        nominal: 189,
        mass: 188.989195
      }, {
        nominal: 190,
        mass: 189.988622
      }, {
        nominal: 191,
        mass: 190.9857866
      }, {
        nominal: 192,
        mass: 191.985469
      }, {
        nominal: 193,
        mass: 192.98296
      }, {
        nominal: 194,
        mass: 193.982785
      }, {
        nominal: 195,
        mass: 194.9806488
      }, {
        nominal: 196,
        mass: 195.980667
      }, {
        nominal: 197,
        mass: 196.9788651
      }, {
        nominal: 198,
        mass: 197.979206
      }, {
        nominal: 199,
        mass: 198.977673
      }, {
        nominal: 200,
        mass: 199.978131
      }, {
        nominal: 201,
        mass: 200.97701
      }, {
        nominal: 202,
        mass: 201.977734
      }, {
        nominal: 203,
        mass: 202.976893
      }, {
        nominal: 204,
        mass: 203.9778361
      }, {
        nominal: 205,
        mass: 204.9773867
      }, {
        nominal: 206,
        mass: 205.9784993
      }, {
        nominal: 207,
        mass: 206.978471
      }, {
        nominal: 208,
        mass: 207.9797425
      }, {
        nominal: 209,
        mass: 208.9803991,
        abundance: 1
      }, {
        nominal: 210,
        mass: 209.9841207
      }, {
        nominal: 211,
        mass: 210.9872697
      }, {
        nominal: 212,
        mass: 211.991286
      }, {
        nominal: 213,
        mass: 212.9943851
      }, {
        nominal: 214,
        mass: 213.998712
      }, {
        nominal: 215,
        mass: 215.00177
      }, {
        nominal: 216,
        mass: 216.006306
      }, {
        nominal: 217,
        mass: 217.009372
      }, {
        nominal: 218,
        mass: 218.014188
      }, {
        nominal: 219,
        mass: 219.01748
      }, {
        nominal: 220,
        mass: 220.02235
      }, {
        nominal: 221,
        mass: 221.02587
      }, {
        nominal: 222,
        mass: 222.03078
      }, {
        nominal: 223,
        mass: 223.0345
      }, {
        nominal: 224,
        mass: 224.03947
      }],
      symbol: 'Bi',
      mass: 208.9803991,
      name: 'Bismuth',
      monoisotopicMass: 208.9803991
    }, {
      number: 84,
      isotopes: [{
        nominal: 186,
        mass: 186.004393
      }, {
        nominal: 187,
        mass: 187.003041
      }, {
        nominal: 188,
        mass: 187.999416
      }, {
        nominal: 189,
        mass: 188.998473
      }, {
        nominal: 190,
        mass: 189.995101
      }, {
        nominal: 191,
        mass: 190.9945585
      }, {
        nominal: 192,
        mass: 191.991336
      }, {
        nominal: 193,
        mass: 192.991026
      }, {
        nominal: 194,
        mass: 193.988186
      }, {
        nominal: 195,
        mass: 194.988126
      }, {
        nominal: 196,
        mass: 195.985526
      }, {
        nominal: 197,
        mass: 196.98566
      }, {
        nominal: 198,
        mass: 197.983389
      }, {
        nominal: 199,
        mass: 198.983667
      }, {
        nominal: 200,
        mass: 199.981799
      }, {
        nominal: 201,
        mass: 200.9822598
      }, {
        nominal: 202,
        mass: 201.980758
      }, {
        nominal: 203,
        mass: 202.9814161
      }, {
        nominal: 204,
        mass: 203.98031
      }, {
        nominal: 205,
        mass: 204.981203
      }, {
        nominal: 206,
        mass: 205.980474
      }, {
        nominal: 207,
        mass: 206.9815938
      }, {
        nominal: 208,
        mass: 207.9812461
      }, {
        nominal: 209,
        mass: 208.9824308
      }, {
        nominal: 210,
        mass: 209.9828741
      }, {
        nominal: 211,
        mass: 210.9866536
      }, {
        nominal: 212,
        mass: 211.9888684
      }, {
        nominal: 213,
        mass: 212.9928576
      }, {
        nominal: 214,
        mass: 213.9952017
      }, {
        nominal: 215,
        mass: 214.9994201
      }, {
        nominal: 216,
        mass: 216.0019152
      }, {
        nominal: 217,
        mass: 217.0063182
      }, {
        nominal: 218,
        mass: 218.0089735
      }, {
        nominal: 219,
        mass: 219.013614
      }, {
        nominal: 220,
        mass: 220.016386
      }, {
        nominal: 221,
        mass: 221.021228
      }, {
        nominal: 222,
        mass: 222.02414
      }, {
        nominal: 223,
        mass: 223.02907
      }, {
        nominal: 224,
        mass: 224.03211
      }, {
        nominal: 225,
        mass: 225.03707
      }, {
        nominal: 226,
        mass: 226.04031
      }, {
        nominal: 227,
        mass: 227.04539
      }],
      symbol: 'Po',
      mass: null,
      name: 'Polonium'
    }, {
      number: 85,
      isotopes: [{
        nominal: 191,
        mass: 191.004148
      }, {
        nominal: 192,
        mass: 192.003152
      }, {
        nominal: 193,
        mass: 192.999927
      }, {
        nominal: 194,
        mass: 193.999236
      }, {
        nominal: 195,
        mass: 194.9962685
      }, {
        nominal: 196,
        mass: 195.9958
      }, {
        nominal: 197,
        mass: 196.993189
      }, {
        nominal: 198,
        mass: 197.992784
      }, {
        nominal: 199,
        mass: 198.9905277
      }, {
        nominal: 200,
        mass: 199.990351
      }, {
        nominal: 201,
        mass: 200.9884171
      }, {
        nominal: 202,
        mass: 201.98863
      }, {
        nominal: 203,
        mass: 202.986943
      }, {
        nominal: 204,
        mass: 203.987251
      }, {
        nominal: 205,
        mass: 204.986076
      }, {
        nominal: 206,
        mass: 205.986657
      }, {
        nominal: 207,
        mass: 206.9858
      }, {
        nominal: 208,
        mass: 207.9866133
      }, {
        nominal: 209,
        mass: 208.9861702
      }, {
        nominal: 210,
        mass: 209.9871479
      }, {
        nominal: 211,
        mass: 210.9874966
      }, {
        nominal: 212,
        mass: 211.9907377
      }, {
        nominal: 213,
        mass: 212.992937
      }, {
        nominal: 214,
        mass: 213.9963721
      }, {
        nominal: 215,
        mass: 214.9986528
      }, {
        nominal: 216,
        mass: 216.0024236
      }, {
        nominal: 217,
        mass: 217.0047192
      }, {
        nominal: 218,
        mass: 218.008695
      }, {
        nominal: 219,
        mass: 219.0111618
      }, {
        nominal: 220,
        mass: 220.015433
      }, {
        nominal: 221,
        mass: 221.018017
      }, {
        nominal: 222,
        mass: 222.022494
      }, {
        nominal: 223,
        mass: 223.025151
      }, {
        nominal: 224,
        mass: 224.029749
      }, {
        nominal: 225,
        mass: 225.03263
      }, {
        nominal: 226,
        mass: 226.03716
      }, {
        nominal: 227,
        mass: 227.04024
      }, {
        nominal: 228,
        mass: 228.04475
      }, {
        nominal: 229,
        mass: 229.04812
      }],
      symbol: 'At',
      mass: null,
      name: 'Astatine'
    }, {
      number: 86,
      isotopes: [{
        nominal: 193,
        mass: 193.009708
      }, {
        nominal: 194,
        mass: 194.006144
      }, {
        nominal: 195,
        mass: 195.005422
      }, {
        nominal: 196,
        mass: 196.002116
      }, {
        nominal: 197,
        mass: 197.001585
      }, {
        nominal: 198,
        mass: 197.998679
      }, {
        nominal: 199,
        mass: 198.99839
      }, {
        nominal: 200,
        mass: 199.99569
      }, {
        nominal: 201,
        mass: 200.995628
      }, {
        nominal: 202,
        mass: 201.993264
      }, {
        nominal: 203,
        mass: 202.993388
      }, {
        nominal: 204,
        mass: 203.99143
      }, {
        nominal: 205,
        mass: 204.991719
      }, {
        nominal: 206,
        mass: 205.990214
      }, {
        nominal: 207,
        mass: 206.9907303
      }, {
        nominal: 208,
        mass: 207.989635
      }, {
        nominal: 209,
        mass: 208.990415
      }, {
        nominal: 210,
        mass: 209.9896891
      }, {
        nominal: 211,
        mass: 210.9906011
      }, {
        nominal: 212,
        mass: 211.9907039
      }, {
        nominal: 213,
        mass: 212.9938831
      }, {
        nominal: 214,
        mass: 213.995363
      }, {
        nominal: 215,
        mass: 214.9987459
      }, {
        nominal: 216,
        mass: 216.0002719
      }, {
        nominal: 217,
        mass: 217.003928
      }, {
        nominal: 218,
        mass: 218.0056016
      }, {
        nominal: 219,
        mass: 219.0094804
      }, {
        nominal: 220,
        mass: 220.0113941
      }, {
        nominal: 221,
        mass: 221.0155371
      }, {
        nominal: 222,
        mass: 222.0175782
      }, {
        nominal: 223,
        mass: 223.0218893
      }, {
        nominal: 224,
        mass: 224.024096
      }, {
        nominal: 225,
        mass: 225.028486
      }, {
        nominal: 226,
        mass: 226.030861
      }, {
        nominal: 227,
        mass: 227.035304
      }, {
        nominal: 228,
        mass: 228.037835
      }, {
        nominal: 229,
        mass: 229.042257
      }, {
        nominal: 230,
        mass: 230.04514
      }, {
        nominal: 231,
        mass: 231.04987
      }],
      symbol: 'Rn',
      mass: null,
      name: 'Radon'
    }, {
      number: 87,
      isotopes: [{
        nominal: 199,
        mass: 199.007259
      }, {
        nominal: 200,
        mass: 200.006586
      }, {
        nominal: 201,
        mass: 201.003867
      }, {
        nominal: 202,
        mass: 202.00332
      }, {
        nominal: 203,
        mass: 203.0009407
      }, {
        nominal: 204,
        mass: 204.000652
      }, {
        nominal: 205,
        mass: 204.9985939
      }, {
        nominal: 206,
        mass: 205.998666
      }, {
        nominal: 207,
        mass: 206.996946
      }, {
        nominal: 208,
        mass: 207.997138
      }, {
        nominal: 209,
        mass: 208.995955
      }, {
        nominal: 210,
        mass: 209.996422
      }, {
        nominal: 211,
        mass: 210.995556
      }, {
        nominal: 212,
        mass: 211.9962257
      }, {
        nominal: 213,
        mass: 212.996186
      }, {
        nominal: 214,
        mass: 213.9989713
      }, {
        nominal: 215,
        mass: 215.0003418
      }, {
        nominal: 216,
        mass: 216.0031899
      }, {
        nominal: 217,
        mass: 217.0046323
      }, {
        nominal: 218,
        mass: 218.0075787
      }, {
        nominal: 219,
        mass: 219.0092524
      }, {
        nominal: 220,
        mass: 220.0123277
      }, {
        nominal: 221,
        mass: 221.0142552
      }, {
        nominal: 222,
        mass: 222.017552
      }, {
        nominal: 223,
        mass: 223.019736
      }, {
        nominal: 224,
        mass: 224.023398
      }, {
        nominal: 225,
        mass: 225.025573
      }, {
        nominal: 226,
        mass: 226.029566
      }, {
        nominal: 227,
        mass: 227.031869
      }, {
        nominal: 228,
        mass: 228.035823
      }, {
        nominal: 229,
        mass: 229.038298
      }, {
        nominal: 230,
        mass: 230.042416
      }, {
        nominal: 231,
        mass: 231.045158
      }, {
        nominal: 232,
        mass: 232.04937
      }, {
        nominal: 233,
        mass: 233.05264
      }],
      symbol: 'Fr',
      mass: null,
      name: 'Francium'
    }, {
      number: 88,
      isotopes: [{
        nominal: 201,
        mass: 201.01271
      }, {
        nominal: 202,
        mass: 202.00976
      }, {
        nominal: 203,
        mass: 203.009304
      }, {
        nominal: 204,
        mass: 204.006492
      }, {
        nominal: 205,
        mass: 205.006268
      }, {
        nominal: 206,
        mass: 206.003828
      }, {
        nominal: 207,
        mass: 207.003799
      }, {
        nominal: 208,
        mass: 208.001841
      }, {
        nominal: 209,
        mass: 209.00199
      }, {
        nominal: 210,
        mass: 210.000494
      }, {
        nominal: 211,
        mass: 211.0008932
      }, {
        nominal: 212,
        mass: 211.999787
      }, {
        nominal: 213,
        mass: 213.000384
      }, {
        nominal: 214,
        mass: 214.0000997
      }, {
        nominal: 215,
        mass: 215.0027204
      }, {
        nominal: 216,
        mass: 216.0035334
      }, {
        nominal: 217,
        mass: 217.0063207
      }, {
        nominal: 218,
        mass: 218.007141
      }, {
        nominal: 219,
        mass: 219.0100855
      }, {
        nominal: 220,
        mass: 220.0110259
      }, {
        nominal: 221,
        mass: 221.0139177
      }, {
        nominal: 222,
        mass: 222.0153748
      }, {
        nominal: 223,
        mass: 223.0185023
      }, {
        nominal: 224,
        mass: 224.020212
      }, {
        nominal: 225,
        mass: 225.0236119
      }, {
        nominal: 226,
        mass: 226.0254103
      }, {
        nominal: 227,
        mass: 227.0291783
      }, {
        nominal: 228,
        mass: 228.0310707
      }, {
        nominal: 229,
        mass: 229.034942
      }, {
        nominal: 230,
        mass: 230.037055
      }, {
        nominal: 231,
        mass: 231.041027
      }, {
        nominal: 232,
        mass: 232.0434753
      }, {
        nominal: 233,
        mass: 233.047582
      }, {
        nominal: 234,
        mass: 234.050342
      }, {
        nominal: 235,
        mass: 235.05497
      }],
      symbol: 'Ra',
      mass: null,
      name: 'Radium'
    }, {
      number: 89,
      isotopes: [{
        nominal: 206,
        mass: 206.014452
      }, {
        nominal: 207,
        mass: 207.011966
      }, {
        nominal: 208,
        mass: 208.01155
      }, {
        nominal: 209,
        mass: 209.009495
      }, {
        nominal: 210,
        mass: 210.009436
      }, {
        nominal: 211,
        mass: 211.007732
      }, {
        nominal: 212,
        mass: 212.007813
      }, {
        nominal: 213,
        mass: 213.006609
      }, {
        nominal: 214,
        mass: 214.006918
      }, {
        nominal: 215,
        mass: 215.006475
      }, {
        nominal: 216,
        mass: 216.008743
      }, {
        nominal: 217,
        mass: 217.009344
      }, {
        nominal: 218,
        mass: 218.011642
      }, {
        nominal: 219,
        mass: 219.012421
      }, {
        nominal: 220,
        mass: 220.0147549
      }, {
        nominal: 221,
        mass: 221.015592
      }, {
        nominal: 222,
        mass: 222.0178442
      }, {
        nominal: 223,
        mass: 223.0191377
      }, {
        nominal: 224,
        mass: 224.0217232
      }, {
        nominal: 225,
        mass: 225.02323
      }, {
        nominal: 226,
        mass: 226.0260984
      }, {
        nominal: 227,
        mass: 227.0277523
      }, {
        nominal: 228,
        mass: 228.0310215
      }, {
        nominal: 229,
        mass: 229.032956
      }, {
        nominal: 230,
        mass: 230.036327
      }, {
        nominal: 231,
        mass: 231.038393
      }, {
        nominal: 232,
        mass: 232.042034
      }, {
        nominal: 233,
        mass: 233.044346
      }, {
        nominal: 234,
        mass: 234.048139
      }, {
        nominal: 235,
        mass: 235.05084
      }, {
        nominal: 236,
        mass: 236.054988
      }, {
        nominal: 237,
        mass: 237.05827
      }],
      symbol: 'Ac',
      mass: null,
      name: 'Actinium'
    }, {
      number: 90,
      isotopes: [{
        nominal: 208,
        mass: 208.0179
      }, {
        nominal: 209,
        mass: 209.017753
      }, {
        nominal: 210,
        mass: 210.015094
      }, {
        nominal: 211,
        mass: 211.014929
      }, {
        nominal: 212,
        mass: 212.012988
      }, {
        nominal: 213,
        mass: 213.013009
      }, {
        nominal: 214,
        mass: 214.0115
      }, {
        nominal: 215,
        mass: 215.0117248
      }, {
        nominal: 216,
        mass: 216.011056
      }, {
        nominal: 217,
        mass: 217.013117
      }, {
        nominal: 218,
        mass: 218.013276
      }, {
        nominal: 219,
        mass: 219.015537
      }, {
        nominal: 220,
        mass: 220.015748
      }, {
        nominal: 221,
        mass: 221.018184
      }, {
        nominal: 222,
        mass: 222.018469
      }, {
        nominal: 223,
        mass: 223.0208119
      }, {
        nominal: 224,
        mass: 224.021464
      }, {
        nominal: 225,
        mass: 225.0239514
      }, {
        nominal: 226,
        mass: 226.0249034
      }, {
        nominal: 227,
        mass: 227.0277042
      }, {
        nominal: 228,
        mass: 228.0287413
      }, {
        nominal: 229,
        mass: 229.0317627
      }, {
        nominal: 230,
        mass: 230.0331341
      }, {
        nominal: 231,
        mass: 231.0363046
      }, {
        nominal: 232,
        mass: 232.0380558,
        abundance: 1
      }, {
        nominal: 233,
        mass: 233.0415823
      }, {
        nominal: 234,
        mass: 234.0436014
      }, {
        nominal: 235,
        mass: 235.047255
      }, {
        nominal: 236,
        mass: 236.049657
      }, {
        nominal: 237,
        mass: 237.053629
      }, {
        nominal: 238,
        mass: 238.0565
      }, {
        nominal: 239,
        mass: 239.06077
      }],
      symbol: 'Th',
      mass: 232.0380558,
      name: 'Thorium',
      monoisotopicMass: 232.0380558
    }, {
      number: 91,
      isotopes: [{
        nominal: 212,
        mass: 212.023203
      }, {
        nominal: 213,
        mass: 213.021109
      }, {
        nominal: 214,
        mass: 214.020918
      }, {
        nominal: 215,
        mass: 215.019183
      }, {
        nominal: 216,
        mass: 216.019109
      }, {
        nominal: 217,
        mass: 217.018325
      }, {
        nominal: 218,
        mass: 218.020059
      }, {
        nominal: 219,
        mass: 219.019904
      }, {
        nominal: 220,
        mass: 220.021705
      }, {
        nominal: 221,
        mass: 221.021875
      }, {
        nominal: 222,
        mass: 222.023784
      }, {
        nominal: 223,
        mass: 223.023963
      }, {
        nominal: 224,
        mass: 224.0256176
      }, {
        nominal: 225,
        mass: 225.026131
      }, {
        nominal: 226,
        mass: 226.027948
      }, {
        nominal: 227,
        mass: 227.0288054
      }, {
        nominal: 228,
        mass: 228.0310517
      }, {
        nominal: 229,
        mass: 229.0320972
      }, {
        nominal: 230,
        mass: 230.034541
      }, {
        nominal: 231,
        mass: 231.0358842,
        abundance: 1
      }, {
        nominal: 232,
        mass: 232.0385917
      }, {
        nominal: 233,
        mass: 233.0402472
      }, {
        nominal: 234,
        mass: 234.0433072
      }, {
        nominal: 235,
        mass: 235.045399
      }, {
        nominal: 236,
        mass: 236.048668
      }, {
        nominal: 237,
        mass: 237.051023
      }, {
        nominal: 238,
        mass: 238.054637
      }, {
        nominal: 239,
        mass: 239.05726
      }, {
        nominal: 240,
        mass: 240.06098
      }, {
        nominal: 241,
        mass: 241.06408
      }],
      symbol: 'Pa',
      mass: 231.0358842,
      name: 'Protactinium',
      monoisotopicMass: 231.0358842
    }, {
      number: 92,
      isotopes: [{
        nominal: 217,
        mass: 217.02466
      }, {
        nominal: 218,
        mass: 218.023523
      }, {
        nominal: 219,
        mass: 219.024999
      }, {
        nominal: 220,
        mass: 220.02462
      }, {
        nominal: 221,
        mass: 221.02628
      }, {
        nominal: 222,
        mass: 222.026
      }, {
        nominal: 223,
        mass: 223.027739
      }, {
        nominal: 224,
        mass: 224.027605
      }, {
        nominal: 225,
        mass: 225.029391
      }, {
        nominal: 226,
        mass: 226.029339
      }, {
        nominal: 227,
        mass: 227.031157
      }, {
        nominal: 228,
        mass: 228.031371
      }, {
        nominal: 229,
        mass: 229.0335063
      }, {
        nominal: 230,
        mass: 230.0339401
      }, {
        nominal: 231,
        mass: 231.0362939
      }, {
        nominal: 232,
        mass: 232.0371563
      }, {
        nominal: 233,
        mass: 233.0396355
      }, {
        nominal: 234,
        mass: 234.0409523,
        abundance: 0.000054
      }, {
        nominal: 235,
        mass: 235.0439301,
        abundance: 0.007204
      }, {
        nominal: 236,
        mass: 236.0455682
      }, {
        nominal: 237,
        mass: 237.0487304
      }, {
        nominal: 238,
        mass: 238.0507884,
        abundance: 0.992742
      }, {
        nominal: 239,
        mass: 239.0542935
      }, {
        nominal: 240,
        mass: 240.0565934
      }, {
        nominal: 241,
        mass: 241.06033
      }, {
        nominal: 242,
        mass: 242.06293
      }, {
        nominal: 243,
        mass: 243.06699
      }],
      symbol: 'U',
      mass: 238.0289104616574,
      name: 'Uranium',
      monoisotopicMass: 238.0507884
    }, {
      number: 93,
      isotopes: [{
        nominal: 219,
        mass: 219.03143
      }, {
        nominal: 220,
        mass: 220.03254
      }, {
        nominal: 221,
        mass: 221.03204
      }, {
        nominal: 222,
        mass: 222.0333
      }, {
        nominal: 223,
        mass: 223.03285
      }, {
        nominal: 224,
        mass: 224.03422
      }, {
        nominal: 225,
        mass: 225.033911
      }, {
        nominal: 226,
        mass: 226.035188
      }, {
        nominal: 227,
        mass: 227.034957
      }, {
        nominal: 228,
        mass: 228.036067
      }, {
        nominal: 229,
        mass: 229.036264
      }, {
        nominal: 230,
        mass: 230.037828
      }, {
        nominal: 231,
        mass: 231.038245
      }, {
        nominal: 232,
        mass: 232.04011
      }, {
        nominal: 233,
        mass: 233.040741
      }, {
        nominal: 234,
        mass: 234.0428953
      }, {
        nominal: 235,
        mass: 235.0440635
      }, {
        nominal: 236,
        mass: 236.04657
      }, {
        nominal: 237,
        mass: 237.0481736
      }, {
        nominal: 238,
        mass: 238.0509466
      }, {
        nominal: 239,
        mass: 239.0529392
      }, {
        nominal: 240,
        mass: 240.056165
      }, {
        nominal: 241,
        mass: 241.058253
      }, {
        nominal: 242,
        mass: 242.06164
      }, {
        nominal: 243,
        mass: 243.06428
      }, {
        nominal: 244,
        mass: 244.06785
      }, {
        nominal: 245,
        mass: 245.0708
      }],
      symbol: 'Np',
      mass: null,
      name: 'Neptunium'
    }, {
      number: 94,
      isotopes: [{
        nominal: 228,
        mass: 228.038732
      }, {
        nominal: 229,
        mass: 229.040144
      }, {
        nominal: 230,
        mass: 230.03965
      }, {
        nominal: 231,
        mass: 231.041102
      }, {
        nominal: 232,
        mass: 232.041185
      }, {
        nominal: 233,
        mass: 233.042998
      }, {
        nominal: 234,
        mass: 234.0433174
      }, {
        nominal: 235,
        mass: 235.045286
      }, {
        nominal: 236,
        mass: 236.0460581
      }, {
        nominal: 237,
        mass: 237.0484098
      }, {
        nominal: 238,
        mass: 238.0495601
      }, {
        nominal: 239,
        mass: 239.0521636
      }, {
        nominal: 240,
        mass: 240.0538138
      }, {
        nominal: 241,
        mass: 241.0568517
      }, {
        nominal: 242,
        mass: 242.0587428
      }, {
        nominal: 243,
        mass: 243.0620036
      }, {
        nominal: 244,
        mass: 244.0642053
      }, {
        nominal: 245,
        mass: 245.067826
      }, {
        nominal: 246,
        mass: 246.070205
      }, {
        nominal: 247,
        mass: 247.07419
      }],
      symbol: 'Pu',
      mass: null,
      name: 'Plutonium'
    }, {
      number: 95,
      isotopes: [{
        nominal: 230,
        mass: 230.04609
      }, {
        nominal: 231,
        mass: 231.04556
      }, {
        nominal: 232,
        mass: 232.04645
      }, {
        nominal: 233,
        mass: 233.04644
      }, {
        nominal: 234,
        mass: 234.04773
      }, {
        nominal: 235,
        mass: 235.047908
      }, {
        nominal: 236,
        mass: 236.04943
      }, {
        nominal: 237,
        mass: 237.049996
      }, {
        nominal: 238,
        mass: 238.051985
      }, {
        nominal: 239,
        mass: 239.0530247
      }, {
        nominal: 240,
        mass: 240.0553
      }, {
        nominal: 241,
        mass: 241.0568293
      }, {
        nominal: 242,
        mass: 242.0595494
      }, {
        nominal: 243,
        mass: 243.0613813
      }, {
        nominal: 244,
        mass: 244.0642851
      }, {
        nominal: 245,
        mass: 245.0664548
      }, {
        nominal: 246,
        mass: 246.069775
      }, {
        nominal: 247,
        mass: 247.07209
      }, {
        nominal: 248,
        mass: 248.07575
      }, {
        nominal: 249,
        mass: 249.07848
      }],
      symbol: 'Am',
      name: 'Americium',
      mass: null
    }, {
      number: 96,
      isotopes: [{
        nominal: 232,
        mass: 232.04982
      }, {
        nominal: 233,
        mass: 233.05077
      }, {
        nominal: 234,
        mass: 234.05016
      }, {
        nominal: 235,
        mass: 235.05154
      }, {
        nominal: 236,
        mass: 236.051374
      }, {
        nominal: 237,
        mass: 237.052869
      }, {
        nominal: 238,
        mass: 238.053081
      }, {
        nominal: 239,
        mass: 239.05491
      }, {
        nominal: 240,
        mass: 240.0555297
      }, {
        nominal: 241,
        mass: 241.0576532
      }, {
        nominal: 242,
        mass: 242.058836
      }, {
        nominal: 243,
        mass: 243.0613893
      }, {
        nominal: 244,
        mass: 244.0627528
      }, {
        nominal: 245,
        mass: 245.0654915
      }, {
        nominal: 246,
        mass: 246.0672238
      }, {
        nominal: 247,
        mass: 247.0703541
      }, {
        nominal: 248,
        mass: 248.0723499
      }, {
        nominal: 249,
        mass: 249.0759548
      }, {
        nominal: 250,
        mass: 250.078358
      }, {
        nominal: 251,
        mass: 251.082286
      }, {
        nominal: 252,
        mass: 252.08487
      }],
      symbol: 'Cm',
      name: 'Curium',
      mass: null
    }, {
      number: 97,
      isotopes: [{
        nominal: 234,
        mass: 234.05727
      }, {
        nominal: 235,
        mass: 235.05658
      }, {
        nominal: 236,
        mass: 236.05748
      }, {
        nominal: 237,
        mass: 237.0571
      }, {
        nominal: 238,
        mass: 238.0582
      }, {
        nominal: 239,
        mass: 239.05824
      }, {
        nominal: 240,
        mass: 240.05976
      }, {
        nominal: 241,
        mass: 241.06016
      }, {
        nominal: 242,
        mass: 242.06198
      }, {
        nominal: 243,
        mass: 243.0630078
      }, {
        nominal: 244,
        mass: 244.065181
      }, {
        nominal: 245,
        mass: 245.0663618
      }, {
        nominal: 246,
        mass: 246.068673
      }, {
        nominal: 247,
        mass: 247.0703073
      }, {
        nominal: 248,
        mass: 248.073088
      }, {
        nominal: 249,
        mass: 249.0749877
      }, {
        nominal: 250,
        mass: 250.0783167
      }, {
        nominal: 251,
        mass: 251.080762
      }, {
        nominal: 252,
        mass: 252.08431
      }, {
        nominal: 253,
        mass: 253.08688
      }, {
        nominal: 254,
        mass: 254.0906
      }],
      symbol: 'Bk',
      name: 'Berkelium',
      mass: null
    }, {
      number: 98,
      isotopes: [{
        nominal: 237,
        mass: 237.062198
      }, {
        nominal: 238,
        mass: 238.06149
      }, {
        nominal: 239,
        mass: 239.06253
      }, {
        nominal: 240,
        mass: 240.062256
      }, {
        nominal: 241,
        mass: 241.06369
      }, {
        nominal: 242,
        mass: 242.063754
      }, {
        nominal: 243,
        mass: 243.06548
      }, {
        nominal: 244,
        mass: 244.0660008
      }, {
        nominal: 245,
        mass: 245.0680487
      }, {
        nominal: 246,
        mass: 246.0688055
      }, {
        nominal: 247,
        mass: 247.070965
      }, {
        nominal: 248,
        mass: 248.0721851
      }, {
        nominal: 249,
        mass: 249.0748539
      }, {
        nominal: 250,
        mass: 250.0764062
      }, {
        nominal: 251,
        mass: 251.0795886
      }, {
        nominal: 252,
        mass: 252.0816272
      }, {
        nominal: 253,
        mass: 253.0851345
      }, {
        nominal: 254,
        mass: 254.087324
      }, {
        nominal: 255,
        mass: 255.09105
      }, {
        nominal: 256,
        mass: 256.09344
      }],
      symbol: 'Cf',
      name: 'Californium',
      mass: null
    }, {
      number: 99,
      isotopes: [{
        nominal: 239,
        mass: 239.06823
      }, {
        nominal: 240,
        mass: 240.06892
      }, {
        nominal: 241,
        mass: 241.06856
      }, {
        nominal: 242,
        mass: 242.06957
      }, {
        nominal: 243,
        mass: 243.06951
      }, {
        nominal: 244,
        mass: 244.07088
      }, {
        nominal: 245,
        mass: 245.07125
      }, {
        nominal: 246,
        mass: 246.0729
      }, {
        nominal: 247,
        mass: 247.073622
      }, {
        nominal: 248,
        mass: 248.075471
      }, {
        nominal: 249,
        mass: 249.076411
      }, {
        nominal: 250,
        mass: 250.07861
      }, {
        nominal: 251,
        mass: 251.0799936
      }, {
        nominal: 252,
        mass: 252.08298
      }, {
        nominal: 253,
        mass: 253.0848257
      }, {
        nominal: 254,
        mass: 254.0880222
      }, {
        nominal: 255,
        mass: 255.090275
      }, {
        nominal: 256,
        mass: 256.0936
      }, {
        nominal: 257,
        mass: 257.09598
      }, {
        nominal: 258,
        mass: 258.09952
      }],
      symbol: 'Es',
      name: 'Einsteinium',
      mass: null
    }, {
      number: 100,
      isotopes: [{
        nominal: 241,
        mass: 241.07421
      }, {
        nominal: 242,
        mass: 242.07343
      }, {
        nominal: 243,
        mass: 243.07446
      }, {
        nominal: 244,
        mass: 244.07404
      }, {
        nominal: 245,
        mass: 245.07535
      }, {
        nominal: 246,
        mass: 246.07535
      }, {
        nominal: 247,
        mass: 247.07694
      }, {
        nominal: 248,
        mass: 248.0771865
      }, {
        nominal: 249,
        mass: 249.0789275
      }, {
        nominal: 250,
        mass: 250.079521
      }, {
        nominal: 251,
        mass: 251.08154
      }, {
        nominal: 252,
        mass: 252.0824671
      }, {
        nominal: 253,
        mass: 253.0851846
      }, {
        nominal: 254,
        mass: 254.0868544
      }, {
        nominal: 255,
        mass: 255.089964
      }, {
        nominal: 256,
        mass: 256.0917745
      }, {
        nominal: 257,
        mass: 257.0951061
      }, {
        nominal: 258,
        mass: 258.09708
      }, {
        nominal: 259,
        mass: 259.1006
      }, {
        nominal: 260,
        mass: 260.10281
      }],
      symbol: 'Fm',
      name: 'Fermium',
      mass: null
    }, {
      number: 101,
      isotopes: [{
        nominal: 245,
        mass: 245.08081
      }, {
        nominal: 246,
        mass: 246.08171
      }, {
        nominal: 247,
        mass: 247.08152
      }, {
        nominal: 248,
        mass: 248.08282
      }, {
        nominal: 249,
        mass: 249.08291
      }, {
        nominal: 250,
        mass: 250.08441
      }, {
        nominal: 251,
        mass: 251.084774
      }, {
        nominal: 252,
        mass: 252.08643
      }, {
        nominal: 253,
        mass: 253.087144
      }, {
        nominal: 254,
        mass: 254.08959
      }, {
        nominal: 255,
        mass: 255.0910841
      }, {
        nominal: 256,
        mass: 256.09389
      }, {
        nominal: 257,
        mass: 257.0955424
      }, {
        nominal: 258,
        mass: 258.0984315
      }, {
        nominal: 259,
        mass: 259.10051
      }, {
        nominal: 260,
        mass: 260.10365
      }, {
        nominal: 261,
        mass: 261.10583
      }, {
        nominal: 262,
        mass: 262.1091
      }],
      symbol: 'Md',
      name: 'Mendelevium',
      mass: null
    }, {
      number: 102,
      isotopes: [{
        nominal: 248,
        mass: 248.08655
      }, {
        nominal: 249,
        mass: 249.0878
      }, {
        nominal: 250,
        mass: 250.08756
      }, {
        nominal: 251,
        mass: 251.08894
      }, {
        nominal: 252,
        mass: 252.088967
      }, {
        nominal: 253,
        mass: 253.0905641
      }, {
        nominal: 254,
        mass: 254.090956
      }, {
        nominal: 255,
        mass: 255.093191
      }, {
        nominal: 256,
        mass: 256.0942829
      }, {
        nominal: 257,
        mass: 257.0968878
      }, {
        nominal: 258,
        mass: 258.09821
      }, {
        nominal: 259,
        mass: 259.10103
      }, {
        nominal: 260,
        mass: 260.10264
      }, {
        nominal: 261,
        mass: 261.1057
      }, {
        nominal: 262,
        mass: 262.10746
      }, {
        nominal: 263,
        mass: 263.11071
      }, {
        nominal: 264,
        mass: 264.11273
      }],
      symbol: 'No',
      name: 'Nobelium',
      mass: null
    }, {
      number: 103,
      isotopes: [{
        nominal: 251,
        mass: 251.09418
      }, {
        nominal: 252,
        mass: 252.09526
      }, {
        nominal: 253,
        mass: 253.09509
      }, {
        nominal: 254,
        mass: 254.09648
      }, {
        nominal: 255,
        mass: 255.096562
      }, {
        nominal: 256,
        mass: 256.098494
      }, {
        nominal: 257,
        mass: 257.099418
      }, {
        nominal: 258,
        mass: 258.10176
      }, {
        nominal: 259,
        mass: 259.102902
      }, {
        nominal: 260,
        mass: 260.1055
      }, {
        nominal: 261,
        mass: 261.10688
      }, {
        nominal: 262,
        mass: 262.10961
      }, {
        nominal: 263,
        mass: 263.11136
      }, {
        nominal: 264,
        mass: 264.1142
      }, {
        nominal: 265,
        mass: 265.11619
      }, {
        nominal: 266,
        mass: 266.11983
      }],
      symbol: 'Lr',
      name: 'Lawrencium',
      mass: null
    }, {
      number: 104,
      isotopes: [{
        nominal: 253,
        mass: 253.10044
      }, {
        nominal: 254,
        mass: 254.10005
      }, {
        nominal: 255,
        mass: 255.10127
      }, {
        nominal: 256,
        mass: 256.101152
      }, {
        nominal: 257,
        mass: 257.102918
      }, {
        nominal: 258,
        mass: 258.103428
      }, {
        nominal: 259,
        mass: 259.105596
      }, {
        nominal: 260,
        mass: 260.10644
      }, {
        nominal: 261,
        mass: 261.108773
      }, {
        nominal: 262,
        mass: 262.10992
      }, {
        nominal: 263,
        mass: 263.11249
      }, {
        nominal: 264,
        mass: 264.11388
      }, {
        nominal: 265,
        mass: 265.11668
      }, {
        nominal: 266,
        mass: 266.11817
      }, {
        nominal: 267,
        mass: 267.12179
      }, {
        nominal: 268,
        mass: 268.12397
      }],
      symbol: 'Rf',
      name: 'Rutherfordium',
      mass: null
    }, {
      number: 105,
      isotopes: [{
        nominal: 255,
        mass: 255.10707
      }, {
        nominal: 256,
        mass: 256.10789
      }, {
        nominal: 257,
        mass: 257.10758
      }, {
        nominal: 258,
        mass: 258.10928
      }, {
        nominal: 259,
        mass: 259.109492
      }, {
        nominal: 260,
        mass: 260.1113
      }, {
        nominal: 261,
        mass: 261.11192
      }, {
        nominal: 262,
        mass: 262.11407
      }, {
        nominal: 263,
        mass: 263.11499
      }, {
        nominal: 264,
        mass: 264.11741
      }, {
        nominal: 265,
        mass: 265.11861
      }, {
        nominal: 266,
        mass: 266.12103
      }, {
        nominal: 267,
        mass: 267.12247
      }, {
        nominal: 268,
        mass: 268.12567
      }, {
        nominal: 269,
        mass: 269.12791
      }, {
        nominal: 270,
        mass: 270.13136
      }],
      symbol: 'Db',
      name: 'Dubnium',
      mass: null
    }, {
      number: 106,
      isotopes: [{
        nominal: 258,
        mass: 258.11298
      }, {
        nominal: 259,
        mass: 259.1144
      }, {
        nominal: 260,
        mass: 260.114384
      }, {
        nominal: 261,
        mass: 261.115949
      }, {
        nominal: 262,
        mass: 262.116337
      }, {
        nominal: 263,
        mass: 263.11829
      }, {
        nominal: 264,
        mass: 264.11893
      }, {
        nominal: 265,
        mass: 265.12109
      }, {
        nominal: 266,
        mass: 266.12198
      }, {
        nominal: 267,
        mass: 267.12436
      }, {
        nominal: 268,
        mass: 268.12539
      }, {
        nominal: 269,
        mass: 269.12863
      }, {
        nominal: 270,
        mass: 270.13043
      }, {
        nominal: 271,
        mass: 271.13393
      }, {
        nominal: 272,
        mass: 272.13589
      }, {
        nominal: 273,
        mass: 273.13958
      }],
      symbol: 'Sg',
      name: 'Seaborgium',
      mass: null
    }, {
      number: 107,
      isotopes: [{
        nominal: 260,
        mass: 260.12166
      }, {
        nominal: 261,
        mass: 261.12145
      }, {
        nominal: 262,
        mass: 262.12297
      }, {
        nominal: 263,
        mass: 263.12292
      }, {
        nominal: 264,
        mass: 264.12459
      }, {
        nominal: 265,
        mass: 265.12491
      }, {
        nominal: 266,
        mass: 266.12679
      }, {
        nominal: 267,
        mass: 267.1275
      }, {
        nominal: 268,
        mass: 268.12969
      }, {
        nominal: 269,
        mass: 269.13042
      }, {
        nominal: 270,
        mass: 270.13336
      }, {
        nominal: 271,
        mass: 271.13526
      }, {
        nominal: 272,
        mass: 272.13826
      }, {
        nominal: 273,
        mass: 273.14024
      }, {
        nominal: 274,
        mass: 274.14355
      }, {
        nominal: 275,
        mass: 275.14567
      }],
      symbol: 'Bh',
      name: 'Bohrium',
      mass: null
    }, {
      number: 108,
      isotopes: [{
        nominal: 263,
        mass: 263.12852
      }, {
        nominal: 264,
        mass: 264.128357
      }, {
        nominal: 265,
        mass: 265.129793
      }, {
        nominal: 266,
        mass: 266.130046
      }, {
        nominal: 267,
        mass: 267.13167
      }, {
        nominal: 268,
        mass: 268.13186
      }, {
        nominal: 269,
        mass: 269.13375
      }, {
        nominal: 270,
        mass: 270.13429
      }, {
        nominal: 271,
        mass: 271.13717
      }, {
        nominal: 272,
        mass: 272.1385
      }, {
        nominal: 273,
        mass: 273.14168
      }, {
        nominal: 274,
        mass: 274.1433
      }, {
        nominal: 275,
        mass: 275.14667
      }, {
        nominal: 276,
        mass: 276.14846
      }, {
        nominal: 277,
        mass: 277.1519
      }],
      symbol: 'Hs',
      name: 'Hassium',
      mass: null
    }, {
      number: 109,
      isotopes: [{
        nominal: 265,
        mass: 265.136
      }, {
        nominal: 266,
        mass: 266.13737
      }, {
        nominal: 267,
        mass: 267.13719
      }, {
        nominal: 268,
        mass: 268.13865
      }, {
        nominal: 269,
        mass: 269.13882
      }, {
        nominal: 270,
        mass: 270.14033
      }, {
        nominal: 271,
        mass: 271.14074
      }, {
        nominal: 272,
        mass: 272.14341
      }, {
        nominal: 273,
        mass: 273.1444
      }, {
        nominal: 274,
        mass: 274.14724
      }, {
        nominal: 275,
        mass: 275.14882
      }, {
        nominal: 276,
        mass: 276.15159
      }, {
        nominal: 277,
        mass: 277.15327
      }, {
        nominal: 278,
        mass: 278.15631
      }, {
        nominal: 279,
        mass: 279.15808
      }],
      symbol: 'Mt',
      name: 'Meitnerium',
      mass: null
    }, {
      number: 110,
      isotopes: [{
        nominal: 267,
        mass: 267.14377
      }, {
        nominal: 268,
        mass: 268.14348
      }, {
        nominal: 269,
        mass: 269.144752
      }, {
        nominal: 270,
        mass: 270.144584
      }, {
        nominal: 271,
        mass: 271.14595
      }, {
        nominal: 272,
        mass: 272.14602
      }, {
        nominal: 273,
        mass: 273.14856
      }, {
        nominal: 274,
        mass: 274.14941
      }, {
        nominal: 275,
        mass: 275.15203
      }, {
        nominal: 276,
        mass: 276.15303
      }, {
        nominal: 277,
        mass: 277.15591
      }, {
        nominal: 278,
        mass: 278.15704
      }, {
        nominal: 279,
        mass: 279.1601
      }, {
        nominal: 280,
        mass: 280.16131
      }, {
        nominal: 281,
        mass: 281.16451
      }],
      symbol: 'Ds',
      name: 'Darmstadtium',
      mass: null
    }, {
      number: 111,
      isotopes: [{
        nominal: 272,
        mass: 272.15327
      }, {
        nominal: 273,
        mass: 273.15313
      }, {
        nominal: 274,
        mass: 274.15525
      }, {
        nominal: 275,
        mass: 275.15594
      }, {
        nominal: 276,
        mass: 276.15833
      }, {
        nominal: 277,
        mass: 277.15907
      }, {
        nominal: 278,
        mass: 278.16149
      }, {
        nominal: 279,
        mass: 279.16272
      }, {
        nominal: 280,
        mass: 280.16514
      }, {
        nominal: 281,
        mass: 281.16636
      }, {
        nominal: 282,
        mass: 282.16912
      }, {
        nominal: 283,
        mass: 283.17054
      }],
      symbol: 'Rg',
      name: 'Roentgenium',
      mass: null
    }, {
      number: 112,
      isotopes: [{
        nominal: 276,
        mass: 276.16141
      }, {
        nominal: 277,
        mass: 277.16364
      }, {
        nominal: 278,
        mass: 278.16416
      }, {
        nominal: 279,
        mass: 279.16654
      }, {
        nominal: 280,
        mass: 280.16715
      }, {
        nominal: 281,
        mass: 281.16975
      }, {
        nominal: 282,
        mass: 282.1705
      }, {
        nominal: 283,
        mass: 283.17327
      }, {
        nominal: 284,
        mass: 284.17416
      }, {
        nominal: 285,
        mass: 285.17712
      }],
      symbol: 'Cn',
      name: 'Copernicium',
      mass: null
    }, {
      number: 113,
      isotopes: [{
        nominal: 278,
        mass: 278.17058
      }, {
        nominal: 279,
        mass: 279.17095
      }, {
        nominal: 280,
        mass: 280.17293
      }, {
        nominal: 281,
        mass: 281.17348
      }, {
        nominal: 282,
        mass: 282.17567
      }, {
        nominal: 283,
        mass: 283.17657
      }, {
        nominal: 284,
        mass: 284.17873
      }, {
        nominal: 285,
        mass: 285.17973
      }, {
        nominal: 286,
        mass: 286.18221
      }, {
        nominal: 287,
        mass: 287.18339
      }],
      symbol: 'Nh',
      name: 'Nihonium',
      mass: null
    }, {
      number: 114,
      isotopes: [{
        nominal: 285,
        mass: 285.18364
      }, {
        nominal: 286,
        mass: 286.18423
      }, {
        nominal: 287,
        mass: 287.18678
      }, {
        nominal: 288,
        mass: 288.18757
      }, {
        nominal: 289,
        mass: 289.19042
      }],
      symbol: 'Fl',
      name: 'Flerovium',
      mass: null
    }, {
      number: 115,
      isotopes: [{
        nominal: 287,
        mass: 287.1907
      }, {
        nominal: 288,
        mass: 288.19274
      }, {
        nominal: 289,
        mass: 289.19363
      }, {
        nominal: 290,
        mass: 290.19598
      }, {
        nominal: 291,
        mass: 291.19707
      }],
      symbol: 'Mc',
      name: 'Moscovium',
      mass: null
    }, {
      number: 116,
      isotopes: [{
        nominal: 289,
        mass: 289.19816
      }, {
        nominal: 290,
        mass: 290.19864
      }, {
        nominal: 291,
        mass: 291.20108
      }, {
        nominal: 292,
        mass: 292.20174
      }, {
        nominal: 293,
        mass: 293.20449
      }],
      symbol: 'Lv',
      name: 'Livermorium',
      mass: null
    }, {
      number: 117,
      isotopes: [{
        nominal: 291,
        mass: 291.20553
      }, {
        nominal: 292,
        mass: 292.20746
      }, {
        nominal: 293,
        mass: 293.20824
      }, {
        nominal: 294,
        mass: 294.21046
      }],
      symbol: 'Ts',
      name: 'Teennessine',
      mass: null
    }, {
      number: 118,
      isotopes: [{
        nominal: 293,
        mass: 293.21356
      }, {
        nominal: 294,
        mass: 294.21392
      }, {
        nominal: 295,
        mass: 295.21624
      }],
      symbol: 'Og',
      name: 'Oganesson',
      mass: null
    }];

    const elements$2 = elementsAndIsotopes.map(element => ({
      number: element.number,
      symbol: element.symbol,
      mass: element.mass,
      name: element.name,
      monoisotopicMass: element.monoisotopicMass
    }));

    const elementsAndIsotopesObject = {};
    elementsAndIsotopes.forEach(element => {
      elementsAndIsotopesObject[element.symbol] = element;
    });

    const elementsAndStableIsotopes = JSON.parse(JSON.stringify(elementsAndIsotopes));
    elementsAndStableIsotopes.forEach(element => {
      element.isotopes = element.isotopes.filter(i => i.abundance > 0);
    });

    const elementsAndStableIsotopesObject = {};
    elementsAndStableIsotopes.forEach(element => {
      elementsAndStableIsotopesObject[element.symbol] = element;
    });

    const elementsObject = {};
    elements$2.forEach(element => {
      elementsObject[element.symbol] = element;
    });

    const stableIsotopesObject = {};
    for (const element of elementsAndIsotopes) {
      let abundance = 0;
      let mostAbundant = 0;
      for (const isotope of element.isotopes) {
        if (isotope.abundance > abundance) {
          abundance = isotope.abundance;
          mostAbundant = isotope.nominal;
        }
      }
      for (const isotope of element.isotopes) {
        if (isotope.abundance === 0) continue;
        const entry = {
          name: element.name,
          mass: isotope.mass,
          symbol: element.symbol
        };
        if (isotope.nominal === mostAbundant) {
          entry.mostAbundant = true;
        }
        stableIsotopesObject[isotope.nominal + element.symbol] = entry;
      }
    }

    const isotopesObject = {};
    Object.keys(elementsAndIsotopesObject).forEach(key => {
      let e = elementsAndIsotopesObject[key];
      e.isotopes.forEach(i => {
        isotopesObject[i.nominal + key] = {
          abundance: i.abundance,
          mass: i.mass
        };
      });
    });

    const unsaturationsObject = {
      O: 0,
      N: 1,
      H: -1,
      Na: -1,
      K: -1,
      Li: -1,
      Ca: -2,
      C: 2,
      F: -1,
      Si: 2,
      Cl: -1,
      Br: -1,
      I: -1,
      S: 0,
      P: 1
    };

    const groups = [{
      "symbol": "Abu",
      "name": "2-Aminobutyric acid diradical",
      "mf": "C4H7NO",
      "ocl": {
        "value": "dazHPBPOEgEInVZjcH@",
        "coordinates": "!Bb@I~@Ha}_c~H@m]}bGt"
      },
      "mass": 85.10463700109551,
      "monoisotopicMass": 85.05276384961,
      "unsaturation": 2,
      "elements": [{
        "symbol": "C",
        "number": 4
      }, {
        "symbol": "H",
        "number": 7
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Acet",
      "name": "Acetyl",
      "mf": "C2H3O",
      "ocl": {
        "value": "gCaHDEeIi`@",
        "coordinates": "!BbOq~@Ha}"
      },
      "mass": 43.04469897995611,
      "monoisotopicMass": 43.01838971626,
      "unsaturation": 1,
      "elements": [{
        "symbol": "C",
        "number": 2
      }, {
        "symbol": "H",
        "number": 3
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Acm",
      "name": "Acetamidomethyl",
      "mf": "C3H6NO",
      "ocl": {
        "value": "gGYHDPliJuS@@",
        "coordinates": "!BbOrH_Xc|_`BH_P"
      },
      "mass": 72.08596035030448,
      "monoisotopicMass": 72.04493881738,
      "unsaturation": 1,
      "elements": [{
        "symbol": "C",
        "number": 3
      }, {
        "symbol": "H",
        "number": 6
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Adao",
      "name": "Adamantyloxy",
      "mf": "C10H15O",
      "ocl": {
        "value": "dc\\H`HAYRVeV^dUGZjjjj@@",
        "coordinates": "!B]BOXN`EP}CdB\\tbZ@Ijh~hRELdOBBp"
      },
      "mass": 151.2258752025074,
      "monoisotopicMass": 151.11229010302,
      "unsaturation": 5,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 15
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Aib",
      "name": "alpha-Aminoisobutyric acid diradical",
      "mf": "C4H7NO",
      "ocl": {
        "value": "dazHPBPOGgEInfZj@@",
        "coordinates": "!Bb@I~@Ha}b@K|uwwWbGt"
      },
      "mass": 85.10463700109551,
      "monoisotopicMass": 85.05276384961,
      "unsaturation": 2,
      "elements": [{
        "symbol": "C",
        "number": 4
      }, {
        "symbol": "H",
        "number": 7
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Ala",
      "name": "Alanine diradical",
      "mf": "C3H5NO",
      "kind": "aa",
      "oneLetter": "A",
      "alternativeOneLetter": "α",
      "ocl": {
        "value": "gNyDBaxmqR[fZjZ@",
        "coordinates": "!BbOr~@H`}bOr~Wxb}"
      },
      "mass": 71.07801959624871,
      "monoisotopicMass": 71.03711378515,
      "unsaturation": 2,
      "elements": [{
        "symbol": "C",
        "number": 3
      }, {
        "symbol": "H",
        "number": 5
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Arg",
      "name": "Arginine diradical",
      "mf": "C6H12N4O",
      "kind": "aa",
      "oneLetter": "R",
      "alternativeOneLetter": "ρ",
      "ocl": {
        "value": "dkLhPBgSPOEgEInWUijjihr@@",
        "coordinates": "!Bb@I~@Ha}_c~H@m]}bGvHHa}b@I~@Ha}"
      },
      "mass": 156.18592219918227,
      "monoisotopicMass": 156.10111102405,
      "unsaturation": 4,
      "elements": [{
        "symbol": "C",
        "number": 6
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 4
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Argp",
      "name": "Arginine triradical",
      "mf": "C6H11N4O",
      "ocl": {
        "value": "dglhpHpil@gWDEI[UYZfjji`T@",
        "coordinates": "!BbGvHGx@bGvH@ha}bOrH_Wxb@KW_Wx@bGt"
      },
      "mass": 155.1779814451265,
      "monoisotopicMass": 155.09328599182,
      "unsaturation": 5,
      "elements": [{
        "symbol": "C",
        "number": 6
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "N",
        "number": 4
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Asn",
      "name": "Asparagine diradical",
      "mf": "C4H6N2O2",
      "kind": "aa",
      "oneLetter": "N",
      "alternativeOneLetter": "η",
      "ocl": {
        "value": "deeDPBeACqYqR[ezZjZL`@",
        "coordinates": "!BbGu~Ox`B_`BH_X`Bb@I~@Ha}"
      },
      "mass": 114.10280438280381,
      "monoisotopicMass": 114.04292744137999,
      "unsaturation": 4,
      "elements": [{
        "symbol": "C",
        "number": 4
      }, {
        "symbol": "H",
        "number": 6
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Asnp",
      "name": "Asparagine triradical",
      "mf": "C4H5N2O2",
      "ocl": {
        "value": "dmUDpH[E@IEqgqRVvVijjXi@@",
        "coordinates": "!Bb@JH_Wxb@JH_Wxb@KW_Wx@bGt"
      },
      "mass": 113.09486362874803,
      "monoisotopicMass": 113.03510240915,
      "unsaturation": 5,
      "elements": [{
        "symbol": "C",
        "number": 4
      }, {
        "symbol": "H",
        "number": 5
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Asp",
      "name": "Aspartic acid diradical",
      "mf": "C4H5NO3",
      "kind": "aa",
      "oneLetter": "D",
      "alternativeOneLetter": "δ",
      "ocl": {
        "value": "defLPBPYCqYqR[ezZjZL`@",
        "coordinates": "!BbGu~Ox`B_`BH_X`Bb@I~@Ha}"
      },
      "mass": 115.08756534162052,
      "monoisotopicMass": 115.02694302429,
      "unsaturation": 4,
      "elements": [{
        "symbol": "C",
        "number": 4
      }, {
        "symbol": "H",
        "number": 5
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 3
      }]
    }, {
      "symbol": "Aspp",
      "name": "Aspartic acid triradical",
      "mf": "C4H4NO3",
      "ocl": {
        "value": "dmVLpFcE@IEqgqRVvVijjXi@@",
        "coordinates": "!Bb@JH_Wxb@JH_Wxb@KW_Wx@bGt"
      },
      "mass": 114.07962458756472,
      "monoisotopicMass": 114.01911799206,
      "unsaturation": 5,
      "elements": [{
        "symbol": "C",
        "number": 4
      }, {
        "symbol": "H",
        "number": 4
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 3
      }]
    }, {
      "symbol": "Asu",
      "name": "alpha-Aminosuberic acid diradical",
      "mf": "C8H13NO3",
      "ocl": {
        "value": "dgnLPBP{CqYqR[euVfjjihr@@",
        "coordinates": "!BbGu~Ox`B_`BH_Xc|bOrH_X`BbGvHGx@bGt"
      },
      "mass": 171.19403496100773,
      "monoisotopicMass": 171.08954328213002,
      "unsaturation": 4,
      "elements": [{
        "symbol": "C",
        "number": 8
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 3
      }]
    }, {
      "symbol": "Asup",
      "name": "alpha-Aminosuberic acid triradical",
      "mf": "C8H12NO3",
      "ocl": {
        "value": "do^LpEcG@IMqoqRVuUejZjjibT@",
        "coordinates": "!BbOrH_Wxb@JH_Xc|bGvHHa}_c~H@m]}_`BH_P"
      },
      "mass": 170.18609420695194,
      "monoisotopicMass": 170.0817182499,
      "unsaturation": 5,
      "elements": [{
        "symbol": "C",
        "number": 8
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 3
      }]
    }, {
      "symbol": "Boc",
      "name": "t-Butoxycarbonyl",
      "mf": "C5H9O2",
      "ocl": {
        "value": "daxD`DpEeImjZj@@",
        "coordinates": "!B|Ou~_A||Ow}mC}_O@"
      },
      "mass": 101.12395611881479,
      "monoisotopicMass": 101.06025452921,
      "unsaturation": 1,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 9
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Bom",
      "name": "Benzyloxymethyl",
      "mf": "C8H9O",
      "ocl": {
        "value": "deTH`DAYRUYTYj`@@@",
        "coordinates": "!B|Gsp__A||Owp_Gy|Gwp_Wy"
      },
      "mass": 121.15675888470227,
      "monoisotopicMass": 121.06533990964,
      "unsaturation": 7,
      "elements": [{
        "symbol": "C",
        "number": 8
      }, {
        "symbol": "H",
        "number": 9
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Brz",
      "name": "2-Bromobenzyloxycarbonyl",
      "mf": "C8H6BrO2",
      "ocl": {
        "value": "dcLDPDpEd\\QImYgWYjB@@@",
        "coordinates": "!Bb@I~@Hb}b@JH_X`B_c}~@Hb}bGu~Op"
      },
      "mass": 214.03586932736317,
      "monoisotopicMass": 212.95511703252,
      "unsaturation": 9,
      "elements": [{
        "symbol": "C",
        "number": 8
      }, {
        "symbol": "H",
        "number": 6
      }, {
        "symbol": "Br",
        "number": 1
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Bu",
      "name": "Butyl",
      "mf": "C4H9",
      "ocl": {
        "value": "gJPH@liJuP@",
        "coordinates": "!B@Fp@XpAl@FL"
      },
      "mass": 57.114410373442986,
      "monoisotopicMass": 57.07042529007,
      "unsaturation": -1,
      "elements": [{
        "symbol": "C",
        "number": 4
      }, {
        "symbol": "H",
        "number": 9
      }]
    }, {
      "symbol": "Bum",
      "name": "t-Butoxymethyl",
      "mf": "C5H11O",
      "ocl": {
        "value": "gNqHDEeIVjj`@",
        "coordinates": "!B@FL@[@AcXs|@Xvp@"
      },
      "mass": 87.14043270260808,
      "monoisotopicMass": 87.08098997409999,
      "unsaturation": -1,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Bz",
      "name": "Benzoyl",
      "mf": "C7H5O",
      "ocl": {
        "value": "didH`DAYR[e^FX@@@@",
        "coordinates": "!BbOq~@Ha}b@I~Oxa}bGu~Op"
      },
      "mass": 105.1142599717439,
      "monoisotopicMass": 105.03403978072,
      "unsaturation": 9,
      "elements": [{
        "symbol": "C",
        "number": 7
      }, {
        "symbol": "H",
        "number": 5
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Bzl",
      "name": "Benzyl",
      "mf": "C7H7",
      "ocl": {
        "value": "daD@`@VTeeVz`@@@",
        "coordinates": "!B|Gsp_A|_gp_A}_g|"
      },
      "mass": 91.13073655553718,
      "monoisotopicMass": 91.05477522561,
      "unsaturation": 7,
      "elements": [{
        "symbol": "C",
        "number": 7
      }, {
        "symbol": "H",
        "number": 7
      }]
    }, {
      "symbol": "Bn",
      "name": "Benzyl",
      "mf": "C7H7",
      "ocl": {
        "value": "daD@`@VTeeVz`@@@",
        "coordinates": "!B|Gsp_A|_gp_A}_g|"
      },
      "mass": 91.13073655553718,
      "monoisotopicMass": 91.05477522561,
      "unsaturation": 7,
      "elements": [{
        "symbol": "C",
        "number": 7
      }, {
        "symbol": "H",
        "number": 7
      }]
    }, {
      "symbol": "Bzlo",
      "name": "Benzyloxy",
      "mf": "C7H7O",
      "ocl": {
        "value": "didH`HAYRUe^Fh@@@@",
        "coordinates": "!B|Gwp_OC}|Gq~_A}|Gu~_p"
      },
      "mass": 107.13014147985547,
      "monoisotopicMass": 107.04968984518,
      "unsaturation": 7,
      "elements": [{
        "symbol": "C",
        "number": 7
      }, {
        "symbol": "H",
        "number": 7
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Cha",
      "name": "beta-Cyclohexylalanine diradical",
      "mf": "C9H15NO",
      "ocl": {
        "value": "dknHPBPOEgEInWe]NZjjjcH@",
        "coordinates": "!Bb@I~@Ha}_c~H@m]}bGvH@gxbGvH_Wx"
      },
      "mass": 153.22184251721796,
      "monoisotopicMass": 153.11536410745,
      "unsaturation": 4,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 15
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Chxo",
      "name": "Cyclohexyloxy",
      "mf": "C6H11O",
      "ocl": {
        "value": "daDH`HAYRVU[jjj@@",
        "coordinates": "!B|Gsp_A|_gp_A}_g|"
      },
      "mass": 99.15116859934332,
      "monoisotopicMass": 99.08098997409999,
      "unsaturation": 1,
      "elements": [{
        "symbol": "C",
        "number": 6
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Cit",
      "name": "Citrulline diradical",
      "mf": "C6H11N3O2",
      "ocl": {
        "value": "dkODPBdttOEgEInWUijjihr@@",
        "coordinates": "!Bb@I~@Ha}_c~H@m]}bGvHHa}b@I~@Ha}"
      },
      "mass": 157.170683157999,
      "monoisotopicMass": 157.08512660696,
      "unsaturation": 4,
      "elements": [{
        "symbol": "C",
        "number": 6
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Citp",
      "name": "Citrulline triradical",
      "mf": "C6H10N3O2",
      "ocl": {
        "value": "dgoDpHJ\\l@gWDEI[UYZfjji`T@",
        "coordinates": "!BbGvHGx@bGvH@ha}bOrH_Wxb@KW_Wx@bGt"
      },
      "mass": 156.16274240394318,
      "monoisotopicMass": 156.07730157473,
      "unsaturation": 5,
      "elements": [{
        "symbol": "C",
        "number": 6
      }, {
        "symbol": "H",
        "number": 10
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Clz",
      "name": "2-Chlorobenzyloxycarbonyl",
      "mf": "C8H6ClO2",
      "ocl": {
        "value": "dcLDPDpEdXaImYgWYjB@@@",
        "coordinates": "!Bb@I~@Hb}b@JH_X`B_c}~@Hb}bGu~Op"
      },
      "mass": 169.58527912946118,
      "monoisotopicMass": 169.00563211451998,
      "unsaturation": 9,
      "elements": [{
        "symbol": "C",
        "number": 8
      }, {
        "symbol": "H",
        "number": 6
      }, {
        "symbol": "Cl",
        "number": 1
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Cp",
      "name": "Cyclopentadienyl",
      "mf": "C5H5",
      "ocl": {
        "value": "gFpH@liLimRp@",
        "coordinates": "!B\\OtPThyEGl@fP"
      },
      "mass": 65.09338325395512,
      "monoisotopicMass": 65.03912516115,
      "unsaturation": 5,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 5
      }]
    }, {
      "symbol": "Cys",
      "name": "Cysteine diradical",
      "mf": "C3H5NOS",
      "kind": "aa",
      "oneLetter": "C",
      "alternativeOneLetter": "ς",
      "ocl": {
        "value": "dazHpBPOEgG`aInVZjcH@",
        "coordinates": "!Bb@I~@Ha}_c~H@m]}bGt"
      },
      "mass": 103.14280700237578,
      "monoisotopicMass": 103.00918495955,
      "unsaturation": 2,
      "elements": [{
        "symbol": "C",
        "number": 3
      }, {
        "symbol": "H",
        "number": 5
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Cysp",
      "name": "Cysteine triradical",
      "mf": "C3H4NOS",
      "ocl": {
        "value": "diFHHBD@f@agGoEIVVjjfLP@",
        "coordinates": "!BbGvHHa}_c~HM]}_`BH_P"
      },
      "mass": 102.13486624831998,
      "monoisotopicMass": 102.00135992732,
      "unsaturation": 3,
      "elements": [{
        "symbol": "C",
        "number": 3
      }, {
        "symbol": "H",
        "number": 4
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "D",
      "name": "Deuterium",
      "mf": "[2H]",
      "ocl": {
        "value": "eFAAYhBLCEH@",
        "coordinates": "!B@BL"
      },
      "mass": 2.01410177812,
      "monoisotopicMass": 2.01410177812,
      "unsaturation": -1,
      "elements": [{
        "symbol": "H",
        "number": 1,
        "isotope": 2
      }]
    }, {
      "symbol": "Dde",
      "name": "Dde",
      "mf": "C10H13O2",
      "ocl": {
        "value": "dklD`FDEgHhihicIVZfZj@@",
        "coordinates": "!Bb@I~@Ha}upJH@m]}_`BH_Wx@b@I}bOrH"
      },
      "mass": 165.20939861871415,
      "monoisotopicMass": 165.09155465812998,
      "unsaturation": 7,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Dnp",
      "name": "2,4-Dinitrophenyl",
      "mf": "C6H3N2O4",
      "ocl": {
        "value": "dkmB`bWatpVRd^VS{HhheEUFfBAbX@@",
        "coordinates": "!B_c~H_]]}b@I~Owx_`BH_]]}_c~H_]]}"
      },
      "mass": 167.09926376274353,
      "monoisotopicMass": 167.00928158383,
      "unsaturation": 11,
      "elements": [{
        "symbol": "C",
        "number": 6
      }, {
        "symbol": "H",
        "number": 3
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 4
      }]
    }, {
      "symbol": "Et",
      "name": "Ethyl",
      "mf": "C2H5",
      "ocl": {
        "value": "eMBAYRZ@",
        "coordinates": "!B@Fp@Xp"
      },
      "mass": 29.061175563749384,
      "monoisotopicMass": 29.03912516115,
      "unsaturation": -1,
      "elements": [{
        "symbol": "C",
        "number": 2
      }, {
        "symbol": "H",
        "number": 5
      }]
    }, {
      "symbol": "Fmoc",
      "name": "Fluorenylmethoxycarbonyl",
      "mf": "C15H11O2",
      "ocl": {
        "value": "fde@b@DX@liMkLrjxeVCzLuT@@@P@@@",
        "coordinates": "!BbOq~@Ha}bOrH_]ARcm}Tv~i`pAeKv|@fpB[j[~iozfAKvp"
      },
      "mass": 223.24719659427882,
      "monoisotopicMass": 223.07590459367,
      "unsaturation": 19,
      "elements": [{
        "symbol": "C",
        "number": 15
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "For",
      "name": "Formyl",
      "mf": "CHO",
      "ocl": {
        "value": "eMJDVTfP@",
        "coordinates": "!B@Fp@Xp"
      },
      "mass": 29.018081575109303,
      "monoisotopicMass": 29.0027396518,
      "unsaturation": 1,
      "elements": [{
        "symbol": "C",
        "number": 1
      }, {
        "symbol": "H",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Gln",
      "name": "Glutamine diradical",
      "mf": "C5H8N2O2",
      "kind": "aa",
      "oneLetter": "Q",
      "alternativeOneLetter": "ξ",
      "ocl": {
        "value": "dmUDPBUICqYqR[evfjihr@@",
        "coordinates": "!Bb@I~@Ha}_c~H@m]}bGvHGx@bGt"
      },
      "mass": 128.12942178765059,
      "monoisotopicMass": 128.05857750584,
      "unsaturation": 4,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 8
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Glnp",
      "name": "Glutamine triradical",
      "mf": "C5H7N2O2",
      "ocl": {
        "value": "dcuDpH{MAYeqWqRVuejZjiad@",
        "coordinates": "!BbGvHGx@bGvH@ha}_c~HM]}_`BH_P"
      },
      "mass": 127.12148103359483,
      "monoisotopicMass": 127.05075247361,
      "unsaturation": 5,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 7
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Glp",
      "name": "Pyroglutamine",
      "mf": "C5H5NO2",
      "ocl": {
        "value": "deVDPBRP|V\\TfygxYjjZL`@",
        "coordinates": "!Bb@I~@Ha}tEJNwr[@UMo@FXBN"
      },
      "mass": 111.09889631403748,
      "monoisotopicMass": 111.03202840472,
      "unsaturation": 6,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 5
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Glu",
      "name": "Glutamic acid diradical",
      "mf": "C5H7NO3",
      "kind": "aa",
      "oneLetter": "E",
      "alternativeOneLetter": "ε",
      "ocl": {
        "value": "dmVLPBRUCqYqR[evfjihr@@",
        "coordinates": "!Bb@I~@Ha}_c~H@m]}bGvHGx@bGt"
      },
      "mass": 129.11418274646732,
      "monoisotopicMass": 129.04259308875,
      "unsaturation": 4,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 7
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 3
      }]
    }, {
      "symbol": "Glup",
      "name": "Glutamic acid triradical",
      "mf": "C5H6NO3",
      "ocl": {
        "value": "dcvLpNcM@IeqWqRVuejZjiad@",
        "coordinates": "!BbGvHGx@bGvH@ha}_c~HM]}_`BH_P"
      },
      "mass": 128.10624199241153,
      "monoisotopicMass": 128.03476805652002,
      "unsaturation": 5,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 6
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 3
      }]
    }, {
      "symbol": "Gly",
      "name": "Glycine diradical",
      "mf": "C2H3NO",
      "kind": "aa",
      "oneLetter": "G",
      "alternativeOneLetter": "γ",
      "ocl": {
        "value": "gGYDBaxuqR[Yj@@",
        "coordinates": "!BbOq~@Ha}bOrH_P"
      },
      "mass": 57.051402191401905,
      "monoisotopicMass": 57.021463720689994,
      "unsaturation": 2,
      "elements": [{
        "symbol": "C",
        "number": 2
      }, {
        "symbol": "H",
        "number": 3
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Hci",
      "name": "Homocitrulline diradical",
      "mf": "C7H13N3O2",
      "ocl": {
        "value": "dgoDPBVtLOEgEInWUZZjjfcH@",
        "coordinates": "!BbGu~Ox`B_`BH_Xc|bOrH_X`BbGvHGx@bGt"
      },
      "mass": 171.19730056284578,
      "monoisotopicMass": 171.10077667142,
      "unsaturation": 4,
      "elements": [{
        "symbol": "C",
        "number": 7
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Hcip",
      "name": "Homocitrulline triradical",
      "mf": "C7H12N3O2",
      "ocl": {
        "value": "do_DpHI\\\\EdwFEI[UVVijjjfIP@",
        "coordinates": "!BbOrH_Wxb@JH_Xc|bGvHHa}_c~H@m]}_`BH_P"
      },
      "mass": 170.18935980879002,
      "monoisotopicMass": 170.09295163918998,
      "unsaturation": 5,
      "elements": [{
        "symbol": "C",
        "number": 7
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "His",
      "name": "Histidine diradical",
      "mf": "C6H7N3O",
      "kind": "aa",
      "oneLetter": "H",
      "alternativeOneLetter": "ζ",
      "ocl": {
        "value": "dcOHPBGTCqYqR[eyUvZjejL`@",
        "coordinates": "!Bb@I~@Ha}_c~H@m]}bGwPTh{_UMo@FP"
      },
      "mass": 137.13951521745759,
      "monoisotopicMass": 137.05891185847,
      "unsaturation": 8,
      "elements": [{
        "symbol": "C",
        "number": 6
      }, {
        "symbol": "H",
        "number": 7
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Hisp",
      "name": "Histidine triradical",
      "mf": "C6H6N3O",
      "ocl": {
        "value": "dkoHpHHSAYUqwqRY]YXjjVjihy@@",
        "coordinates": "!BTmA}bL@fUHRN`H`BbGu~Ox`Buwu~@Ha}"
      },
      "mass": 136.13157446340182,
      "monoisotopicMass": 136.05108682624,
      "unsaturation": 9,
      "elements": [{
        "symbol": "C",
        "number": 6
      }, {
        "symbol": "H",
        "number": 6
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Hser",
      "name": "Homoserine diradical",
      "mf": "C4H7NO2",
      "ocl": {
        "value": "diFDPBPP|V\\Tfy^Zjhr@@",
        "coordinates": "!BbGu~Ox`B_`BH_X`Bb@JH_P"
      },
      "mass": 101.10404192541378,
      "monoisotopicMass": 101.04767846918,
      "unsaturation": 2,
      "elements": [{
        "symbol": "C",
        "number": 4
      }, {
        "symbol": "H",
        "number": 7
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Hserp",
      "name": "Homoserine triradical",
      "mf": "C4H6NO2",
      "ocl": {
        "value": "defDpJbPV^\\Q|TeVVjji`d@",
        "coordinates": "!Bb@JH_X`BbGu~Oxc|uwu~@Ha}"
      },
      "mass": 100.09610117135801,
      "monoisotopicMass": 100.03985343695001,
      "unsaturation": 3,
      "elements": [{
        "symbol": "C",
        "number": 4
      }, {
        "symbol": "H",
        "number": 6
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Hyp",
      "name": "Hydroxyproline diradical",
      "mf": "C5H7NO2",
      "ocl": {
        "value": "deVDPBRP|V\\\\bfbbOCMUUIdE@@",
        "coordinates": "!Bb@I~@Ha}tEJNwr[@UMo@FUJO"
      },
      "mass": 113.11477782214904,
      "monoisotopicMass": 113.04767846918,
      "unsaturation": 4,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 7
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Hypp",
      "name": "Hydroxyproline triradical",
      "mf": "C5H6NO2",
      "ocl": {
        "value": "dmvDpJaPB^\\Y|TeeWjZjjidRL`@",
        "coordinates": "!BBOpH_UARcc}TNtBY@HyRSpCQDr\\"
      },
      "mass": 112.10683706809326,
      "monoisotopicMass": 112.03985343695001,
      "unsaturation": 5,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 6
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Ile",
      "name": "Isoleucine diradical",
      "mf": "C6H11NO",
      "kind": "aa",
      "oneLetter": "I",
      "alternativeOneLetter": "ι",
      "ocl": {
        "value": "defHPBPOEgEInVyjjdrT`@",
        "coordinates": "!BbGu~Oxc|_`BH_Xc|b@I~Oxa}"
      },
      "mass": 113.15787181078912,
      "monoisotopicMass": 113.08406397853,
      "unsaturation": 2,
      "elements": [{
        "symbol": "C",
        "number": 6
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Ivdde",
      "name": "1-[4,4-dimethyl-2,6-dioxocyclohexylidene)-3-methylbutyl",
      "mf": "C13H19O2",
      "ocl": {
        "value": "f`a@b@NR@lyEEDhhigEVfjYjj`@@",
        "coordinates": "!BbOq~@Ha}urHGxuwu~@Ha}_`CW_Xa}bOq}b@JH"
      },
      "mass": 207.28925083325453,
      "monoisotopicMass": 207.13850485151,
      "unsaturation": 7,
      "elements": [{
        "symbol": "C",
        "number": 13
      }, {
        "symbol": "H",
        "number": 19
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Leu",
      "name": "Leucine diradical",
      "mf": "C6H11NO",
      "kind": "aa",
      "oneLetter": "L",
      "alternativeOneLetter": "λ",
      "ocl": {
        "value": "defHPBPOEgEInWijjhr@@",
        "coordinates": "!BbGu~Ox`B_`BH_X`Bb@I~@Ha}"
      },
      "mass": 113.15787181078912,
      "monoisotopicMass": 113.08406397853,
      "unsaturation": 2,
      "elements": [{
        "symbol": "C",
        "number": 6
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Lys",
      "name": "Lysine diradical",
      "mf": "C6H12N2O",
      "kind": "aa",
      "oneLetter": "K",
      "alternativeOneLetter": "κ",
      "ocl": {
        "value": "dmUHPBU@|V\\Tfy]YjjjL`@",
        "coordinates": "!BbGu~Ox`B_`BHoX`Bb@JH_X`BbKt"
      },
      "mass": 128.17251577629068,
      "monoisotopicMass": 128.09496301519,
      "unsaturation": 2,
      "elements": [{
        "symbol": "C",
        "number": 6
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Lysp",
      "name": "Lysine triradical",
      "mf": "C6H11N2O",
      "ocl": {
        "value": "dcuHpH{PVY\\U|TeUYZjjjXY@@",
        "coordinates": "!Bb@JH_X`BbGvH@ha}_c~H@m]}_`BH_P"
      },
      "mass": 127.16457502223491,
      "monoisotopicMass": 127.08713798295999,
      "unsaturation": 3,
      "elements": [{
        "symbol": "C",
        "number": 6
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Mbh",
      "name": "4,4'-Dimethoxybenzhydryl",
      "mf": "C15H15O2",
      "ocl": {
        "value": "fdy@b@G^@liLsJkzlcZmT@@@UP@@@",
        "coordinates": "!BbGvHGx_`BH_Xa}uwvHHc|_c}~Oxa}uwvHGxbGwW_P"
      },
      "mass": 227.27895961050194,
      "monoisotopicMass": 227.10720472258998,
      "unsaturation": 15,
      "elements": [{
        "symbol": "C",
        "number": 15
      }, {
        "symbol": "H",
        "number": 15
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Me",
      "name": "Methyl",
      "mf": "CH3",
      "ocl": {
        "value": "eFBAYc@@",
        "coordinates": "!B@FL"
      },
      "mass": 15.03455815890258,
      "monoisotopicMass": 15.02347509669,
      "unsaturation": -1,
      "elements": [{
        "symbol": "C",
        "number": 1
      }, {
        "symbol": "H",
        "number": 3
      }]
    }, {
      "symbol": "Mebzl",
      "name": "4-Methylbenzyl",
      "mf": "C8H9",
      "ocl": {
        "value": "did@`@VTee]nh@H@@",
        "coordinates": "!B|Gsp__A|_gp_C}_gp_P"
      },
      "mass": 105.15735396038399,
      "monoisotopicMass": 105.07042529007,
      "unsaturation": 7,
      "elements": [{
        "symbol": "C",
        "number": 8
      }, {
        "symbol": "H",
        "number": 9
      }]
    }, {
      "symbol": "Meobzl",
      "name": "4-Methoxybenzyl",
      "mf": "C8H9O",
      "ocl": {
        "value": "deTH`AAYRVUunh@J@@",
        "coordinates": "!B|Gsp__A|_gp_A}_gp_Wy"
      },
      "mass": 121.15675888470227,
      "monoisotopicMass": 121.06533990964,
      "unsaturation": 7,
      "elements": [{
        "symbol": "C",
        "number": 8
      }, {
        "symbol": "H",
        "number": 9
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Met",
      "name": "Methionine diradical",
      "mf": "C5H9NOS",
      "kind": "aa",
      "oneLetter": "M",
      "alternativeOneLetter": "μ",
      "ocl": {
        "value": "defHpBPOEgDPaInWYjjhr@@",
        "coordinates": "!Bb@I~@Ha}_c~H@m]}bGvHHa}"
      },
      "mass": 131.19604181206938,
      "monoisotopicMass": 131.04048508847,
      "unsaturation": 2,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 9
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Mmt",
      "name": "4-Methoxytrityl",
      "mf": "C20H17O",
      "ocl": {
        "value": "ffcAB@B`V\\bdTTTRRRVvIhnRGMT@@@@AP@@@",
        "coordinates": "!BbKvHM^}_c}~@Hb}dXWHb}j|nHHc|AqOWoWxJV^Ho]\\BuwvHHb}"
      },
      "mass": 273.3491156779715,
      "monoisotopicMass": 273.12794016748,
      "unsaturation": 23,
      "elements": [{
        "symbol": "C",
        "number": 20
      }, {
        "symbol": "H",
        "number": 17
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Mtc",
      "name": "2,2,5,7,8-pentamethylchroman-6-sulphonyl",
      "mf": "C14H19O3S",
      "ocl": {
        "value": "fleAa@DX\\AY`DYEHXhhilmiKW`rpDQUUD@@",
        "coordinates": "!BbGtBbGwWbGvHGxbGu~@Ha}uwu~Ox`B_c~H_Xa}b@H@_osW"
      },
      "mass": 267.36417906043516,
      "monoisotopicMass": 267.10549064548,
      "unsaturation": 9,
      "elements": [{
        "symbol": "C",
        "number": 14
      }, {
        "symbol": "H",
        "number": 19
      }, {
        "symbol": "O",
        "number": 3
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Mtr",
      "name": "4-Methoxy-2,3,6-trimethylbenzenesulphonyl",
      "mf": "C10H13O3S",
      "ocl": {
        "value": "do|LPDrpVXBLbdLTTTngYXBHj@@",
        "coordinates": "!BbOq}b@KWb@I~@Ha}bOsWHc|_c~H_Wx@b@JH_P"
      },
      "mass": 213.27359094915948,
      "monoisotopicMass": 213.05854045209998,
      "unsaturation": 7,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "O",
        "number": 3
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Mts",
      "name": "Mesitylene-2-sulphonyl",
      "mf": "C9H11O2S",
      "ocl": {
        "value": "d@"
      },
      "mass": 183.24756861999438,
      "monoisotopicMass": 183.04797576807,
      "unsaturation": 7,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "O",
        "number": 2
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Mtt",
      "name": "4-Methyltrityl",
      "mf": "C20H17",
      "ocl": {
        "value": "d@"
      },
      "mass": 257.3497107536532,
      "monoisotopicMass": 257.13302554791,
      "unsaturation": 23,
      "elements": [{
        "symbol": "C",
        "number": 20
      }, {
        "symbol": "H",
        "number": 17
      }]
    }, {
      "symbol": "Nle",
      "name": "Norleucine diradical",
      "mf": "C6H11NO",
      "ocl": {
        "value": "defHPBPOEgEInWYjjhr@@",
        "coordinates": "!Bb@I~@Ha}_c~H@m]}bGvHHa}"
      },
      "mass": 113.15787181078912,
      "monoisotopicMass": 113.08406397853,
      "unsaturation": 2,
      "elements": [{
        "symbol": "C",
        "number": 6
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Npys",
      "name": "3-Nitro-2-pyridinesulphenyl",
      "mf": "C5H3N2O2S",
      "ocl": {
        "value": "d@",
        "coordinates": ""
      },
      "mass": 155.1545054234988,
      "monoisotopicMass": 154.99152351908998,
      "unsaturation": 9,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 3
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 2
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Nva",
      "name": "Norvaline diradical",
      "mf": "C5H9NO",
      "ocl": {
        "value": "diFHPBPOEgEInWfjjL`@",
        "coordinates": "!BbGu~Ox`B_`BH_X`Bb@JH_P"
      },
      "mass": 99.13125440594231,
      "monoisotopicMass": 99.06841391407,
      "unsaturation": 2,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 9
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Odmab",
      "name": "Odmab",
      "mf": "C20H26NO3",
      "ocl": {
        "value": "d@"
      },
      "mass": 328.4260955245558,
      "monoisotopicMass": 328.19126870111995,
      "unsaturation": 15,
      "elements": [{
        "symbol": "C",
        "number": 20
      }, {
        "symbol": "H",
        "number": 26
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 3
      }]
    }, {
      "symbol": "Orn",
      "name": "Ornithine diradical",
      "mf": "C5H10N2O",
      "ocl": {
        "value": "deeHPBe@|V\\Tfy]fjjcH@",
        "coordinates": "!Bb@I~@Ha}_c~H@m]}bGvHHa}"
      },
      "mass": 114.14589837144388,
      "monoisotopicMass": 114.07931295072999,
      "unsaturation": 2,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 10
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Ornp",
      "name": "Ornithine triradical",
      "mf": "C5H9N2O",
      "ocl": {
        "value": "dmUHpHYPBQ\\Y|TeUejjjfJP@",
        "coordinates": "!BbGvHHa}b@JH_Wxb@KW_Wx@bGt"
      },
      "mass": 113.13795761738811,
      "monoisotopicMass": 113.0714879185,
      "unsaturation": 3,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 9
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Pbf",
      "name": "2,2,4,6,7-pentamethyldihydrobenzofurane-5-sulfonyl",
      "mf": "C13H17O3S",
      "ocl": {
        "value": "d@",
        "coordinates": ""
      },
      "mass": 253.33756165558833,
      "monoisotopicMass": 253.08984058101998,
      "unsaturation": 9,
      "elements": [{
        "symbol": "C",
        "number": 13
      }, {
        "symbol": "H",
        "number": 17
      }, {
        "symbol": "O",
        "number": 3
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Pen",
      "name": "Penicillamine diradical",
      "mf": "C5H9NOS",
      "ocl": {
        "value": "d@",
        "coordinates": ""
      },
      "mass": 131.19604181206938,
      "monoisotopicMass": 131.04048508847,
      "unsaturation": 2,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 9
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Penp",
      "name": "Penicillamine triradical",
      "mf": "C5H8NOS",
      "ocl": {
        "value": "d@",
        "coordinates": ""
      },
      "mass": 130.1881010580136,
      "monoisotopicMass": 130.03266005624,
      "unsaturation": 3,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 8
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Ph",
      "name": "Phenyl",
      "mf": "C6H5",
      "ocl": {
        "value": "gOpH@liLkW@@@@",
        "coordinates": "!B|Owp_Gy|OwpWy"
      },
      "mass": 77.10411915069038,
      "monoisotopicMass": 77.03912516115,
      "unsaturation": 7,
      "elements": [{
        "symbol": "C",
        "number": 6
      }, {
        "symbol": "H",
        "number": 5
      }]
    }, {
      "symbol": "Phe",
      "name": "Phenylalanine diradical",
      "mf": "C9H9NO",
      "kind": "aa",
      "oneLetter": "F",
      "alternativeOneLetter": "φ",
      "ocl": {
        "value": "dknHPBPOEgEInWe]NZj@@cH@",
        "coordinates": "!Bb@I~@Ha}_c~H@m]}bGvH@gxbGvH_Wx"
      },
      "mass": 147.1741979928833,
      "monoisotopicMass": 147.06841391407002,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 9
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Phepcl",
      "name": "4-Chlorophenylalanine diradical",
      "mf": "C9H8ClNO",
      "ocl": {
        "value": "dg^HpBPOEgFxaInWe_Sfj`@bL`@",
        "coordinates": "!BbOq~@Ha}_c~H@m]}bGvH@gxbGvH_WxbGt"
      },
      "mass": 181.6191948214355,
      "monoisotopicMass": 181.02944156384,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 8
      }, {
        "symbol": "Cl",
        "number": 1
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Pmc",
      "name": "2,2,5,7,8-Pentamethylchroman-6-sulphonyl",
      "mf": "C14H19O3S",
      "ocl": {
        "value": "d@",
        "coordinates": ""
      },
      "mass": 267.36417906043516,
      "monoisotopicMass": 267.10549064548,
      "unsaturation": 9,
      "elements": [{
        "symbol": "C",
        "number": 14
      }, {
        "symbol": "H",
        "number": 19
      }, {
        "symbol": "O",
        "number": 3
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Pro",
      "name": "Proline diradical",
      "mf": "C5H7NO",
      "kind": "aa",
      "oneLetter": "P",
      "alternativeOneLetter": "π",
      "ocl": {
        "value": "difHPBPOEgEInYxYjjhr@@",
        "coordinates": "!Bb@I~@Ha}tEJNwr[@UMo@FP"
      },
      "mass": 97.11537289783075,
      "monoisotopicMass": 97.05276384961,
      "unsaturation": 4,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 7
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Pyr",
      "name": "Pyroglutamine",
      "mf": "C5H5NO2",
      "ocl": {
        "value": "deVDPBRP|V\\TfygxYjjZL`@",
        "coordinates": "!Bb@I~@Ha}tEJNwr[@UMo@FXBN"
      },
      "mass": 111.09889631403748,
      "monoisotopicMass": 111.03202840472,
      "unsaturation": 6,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 5
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Sar",
      "name": "Sarcosine diradical",
      "mf": "C3H5NO",
      "ocl": {
        "value": "d@",
        "coordinates": ""
      },
      "mass": 71.07801959624871,
      "monoisotopicMass": 71.03711378515,
      "unsaturation": 2,
      "elements": [{
        "symbol": "C",
        "number": 3
      }, {
        "symbol": "H",
        "number": 5
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Ser",
      "name": "Serine diradical",
      "mf": "C3H5NO2",
      "kind": "aa",
      "oneLetter": "S",
      "alternativeOneLetter": "σ",
      "ocl": {
        "value": "dazDPBS`|V\\TfyYjjL`@",
        "coordinates": "!Bb@I~@Ha}_c~H@m]}bGt"
      },
      "mass": 87.07742452056698,
      "monoisotopicMass": 87.03202840472,
      "unsaturation": 2,
      "elements": [{
        "symbol": "C",
        "number": 3
      }, {
        "symbol": "H",
        "number": 5
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Serp",
      "name": "Serine triradical",
      "mf": "C3H4NO2",
      "ocl": {
        "value": "diFDpB`PBV\\^|TeYZjjXq@@",
        "coordinates": "!BbGvHHa}_c~HM]}_`BH_P"
      },
      "mass": 86.06948376651121,
      "monoisotopicMass": 86.02420337249,
      "unsaturation": 3,
      "elements": [{
        "symbol": "C",
        "number": 3
      }, {
        "symbol": "H",
        "number": 4
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Sta",
      "name": "Statine diradical",
      "mf": "C8H15NO2",
      "ocl": {
        "value": "d@"
      },
      "mass": 157.210511544801,
      "monoisotopicMass": 157.11027872702002,
      "unsaturation": 2,
      "elements": [{
        "symbol": "C",
        "number": 8
      }, {
        "symbol": "H",
        "number": 15
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Stap",
      "name": "Statine triradical",
      "mf": "C8H14NO2",
      "ocl": {
        "value": "d@"
      },
      "mass": 156.2025707907452,
      "monoisotopicMass": 156.10245369479,
      "unsaturation": 3,
      "elements": [{
        "symbol": "C",
        "number": 8
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Tacm",
      "name": "Trimethylacetamidomethyl",
      "mf": "C6H12NO",
      "ocl": {
        "value": "d@"
      },
      "mass": 114.16581256484488,
      "monoisotopicMass": 114.09188901076,
      "unsaturation": 1,
      "elements": [{
        "symbol": "C",
        "number": 6
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Tbdms",
      "name": "t-Butyldimethylsilyl",
      "mf": "C6H15Si",
      "ocl": {
        "value": "d@"
      },
      "mass": 115.2690253969541,
      "monoisotopicMass": 115.09430201810001,
      "unsaturation": -1,
      "elements": [{
        "symbol": "C",
        "number": 6
      }, {
        "symbol": "H",
        "number": 15
      }, {
        "symbol": "Si",
        "number": 1
      }]
    }, {
      "symbol": "Tbu",
      "name": "t-Butyl",
      "mf": "C4H9",
      "ocl": {
        "value": "d@"
      },
      "mass": 57.114410373442986,
      "monoisotopicMass": 57.07042529007,
      "unsaturation": -1,
      "elements": [{
        "symbol": "C",
        "number": 4
      }, {
        "symbol": "H",
        "number": 9
      }]
    }, {
      "symbol": "Tbuo",
      "name": "t-Butoxy",
      "mf": "C4H9O",
      "ocl": {
        "value": "d@"
      },
      "mass": 73.11381529776126,
      "monoisotopicMass": 73.06533990964,
      "unsaturation": -1,
      "elements": [{
        "symbol": "C",
        "number": 4
      }, {
        "symbol": "H",
        "number": 9
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Tbuthio",
      "name": "t-Butylthio",
      "mf": "C4H9S",
      "ocl": {
        "value": "d@"
      },
      "mass": 89.17919777957005,
      "monoisotopicMass": 89.04249646446999,
      "unsaturation": -1,
      "elements": [{
        "symbol": "C",
        "number": 4
      }, {
        "symbol": "H",
        "number": 9
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Tfa",
      "name": "Trifluoroacetyl",
      "mf": "C2F3O",
      "ocl": {
        "value": "d@"
      },
      "mass": 97.01608620597878,
      "monoisotopicMass": 96.99012410776,
      "unsaturation": 1,
      "elements": [{
        "symbol": "C",
        "number": 2
      }, {
        "symbol": "F",
        "number": 3
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Thr",
      "name": "Threonine diradical",
      "mf": "C4H7NO2",
      "kind": "aa",
      "oneLetter": "T",
      "alternativeOneLetter": "τ",
      "ocl": {
        "value": "d@"
      },
      "mass": 101.10404192541378,
      "monoisotopicMass": 101.04767846918,
      "unsaturation": 2,
      "elements": [{
        "symbol": "C",
        "number": 4
      }, {
        "symbol": "H",
        "number": 7
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Thrp",
      "name": "Threonine triradical",
      "mf": "C4H6NO2",
      "ocl": {
        "value": "d@"
      },
      "mass": 100.09610117135801,
      "monoisotopicMass": 100.03985343695001,
      "unsaturation": 3,
      "elements": [{
        "symbol": "C",
        "number": 4
      }, {
        "symbol": "H",
        "number": 6
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Tfsi",
      "name": "(Bis)(trifluoromethanesulfonyl)imide",
      "mf": "C2F6NO4S2",
      "ocl": {
        "value": "d@"
      },
      "mass": 280.1457884908235,
      "monoisotopicMass": 279.91729380789,
      "unsaturation": -1,
      "elements": [{
        "symbol": "C",
        "number": 2
      }, {
        "symbol": "F",
        "number": 6
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 4
      }, {
        "symbol": "S",
        "number": 2
      }]
    }, {
      "symbol": "Tips",
      "name": "Triisopropylsilyl",
      "mf": "C9H21Si",
      "ocl": {
        "value": "dmT@P@VX\\DffYjjjh@@",
        "coordinates": "!B_a@gHb\\]FBIuWxP^zi~KwxPFAt"
      },
      "mass": 157.34887761149452,
      "monoisotopicMass": 157.14125221148,
      "unsaturation": -1,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 21
      }, {
        "symbol": "Si",
        "number": 1
      }]
    }, {
      "symbol": "Tms",
      "name": "Trimethylsilyl",
      "mf": "C3H9Si",
      "ocl": {
        "value": "gJPD@lqpRZj`@",
        "coordinates": "!BbOq~@GxbGt"
      },
      "mass": 73.1891731824137,
      "monoisotopicMass": 73.04735182472,
      "unsaturation": -1,
      "elements": [{
        "symbol": "C",
        "number": 3
      }, {
        "symbol": "H",
        "number": 9
      }, {
        "symbol": "Si",
        "number": 1
      }]
    }, {
      "symbol": "Tos",
      "name": "Tosyl",
      "mf": "C7H7O2S",
      "ocl": {
        "value": "dmtDPDpEf@cHiCDeafV@B@@",
        "coordinates": "!B|Ou||Ovw|Gwp_Gy|GwpWy|Gt"
      },
      "mass": 155.1943338103008,
      "monoisotopicMass": 155.01667563914998,
      "unsaturation": 7,
      "elements": [{
        "symbol": "C",
        "number": 7
      }, {
        "symbol": "H",
        "number": 7
      }, {
        "symbol": "O",
        "number": 2
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Trp",
      "name": "Tryptophan diradical",
      "mf": "C11H10N2O",
      "kind": "aa",
      "oneLetter": "W",
      "alternativeOneLetter": "ω",
      "ocl": {
        "value": "f`qQA@BFPCqXxiMr|rnhsoSUTa@QCD@@",
        "coordinates": "!BbOq~@Ha}_c~H@m]}bGwPTh{_UMojXL@YpB[@Ini`"
      },
      "mass": 186.21031375185538,
      "monoisotopicMass": 186.07931295073,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 10
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Trpp",
      "name": "Tryptophan triradical",
      "mf": "C11H9N2O",
      "ocl": {
        "value": "fhiQC@HFB@I\\x~|TfYU_ebLDjhDHjibFd@",
        "coordinates": "!BTmA}bL@fUHR_Ihz@iVBeXHc|grZH_WxbOsW_Wx@bGt"
      },
      "mass": 185.20237299779959,
      "monoisotopicMass": 185.07148791850003,
      "unsaturation": 15,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 9
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Trt",
      "name": "Trityl",
      "mf": "C19H15",
      "ocl": {
        "value": "fbm@B@@KJSSLrjkyhnRGMT@@@@@@@@",
        "coordinates": "!BrHI~PGy_rMvW@l`BQCvWw\\bBAg}~PGy@]i}~W|c]cNwH`i_]_e|"
      },
      "mass": 243.32309334880637,
      "monoisotopicMass": 243.11737548345,
      "unsaturation": 23,
      "elements": [{
        "symbol": "C",
        "number": 19
      }, {
        "symbol": "H",
        "number": 15
      }]
    }, {
      "symbol": "Tyr",
      "name": "Tyrosine diradical",
      "mf": "C9H9NO2",
      "kind": "aa",
      "oneLetter": "Y",
      "alternativeOneLetter": "ψ",
      "ocl": {
        "value": "dg^DPBRp|V\\Tfy^U}NZj@BHr@@",
        "coordinates": "!BbOq~@Ha}_c~H@m]}bGvH@gxbGvH_WxbGt"
      },
      "mass": 163.1736029172016,
      "monoisotopicMass": 163.06332853364,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 9
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Tyrp",
      "name": "Tyrosine triradical",
      "mf": "C9H8NO2",
      "ocl": {
        "value": "do~DpEapBS\\[|Tee]YYnh@JjdbT@",
        "coordinates": "!B_`BHGx@bGvH@h`BbKvH@ha}_c~H@m]}_`BHoP"
      },
      "mass": 162.16566216314578,
      "monoisotopicMass": 162.05550350141,
      "unsaturation": 11,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 8
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Val",
      "name": "Valine",
      "mf": "C5H9NO",
      "kind": "aa",
      "oneLetter": "V",
      "alternativeOneLetter": "ν",
      "ocl": {
        "value": "diFHPBPOEgEInVfjjL`@",
        "coordinates": "!Bb@I~@Ha}_c~H@m]}_`BH_P"
      },
      "mass": 99.13125440594231,
      "monoisotopicMass": 99.06841391407,
      "unsaturation": 2,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 9
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Valoh",
      "name": "beta-Hydroxyvaline diradical",
      "mf": "C5H9NO2",
      "ocl": {
        "value": "defDPBS`|V\\TfyZfjjcH@",
        "coordinates": "!Bb@I~@Ha}b@I~Oxa}Owy~OpA~"
      },
      "mass": 115.13065933026058,
      "monoisotopicMass": 115.06332853364,
      "unsaturation": 2,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 9
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Valohp",
      "name": "beta-Hydroxyvaline triradical",
      "mf": "C5H8NO2",
      "ocl": {
        "value": "dmVDpFaPBQ\\Y|\\bTbaTjjjXq@@",
        "coordinates": "!BbGvHHa}_Xc|bGxb@KW_Wx@bGt"
      },
      "mass": 114.1227185762048,
      "monoisotopicMass": 114.05550350141002,
      "unsaturation": 3,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 8
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Xan",
      "name": "Xanthyl",
      "mf": "C13H9O",
      "ocl": {
        "value": "d@",
        "coordinates": ""
      },
      "mass": 181.21043836837848,
      "monoisotopicMass": 181.06533990964002,
      "unsaturation": 17,
      "elements": [{
        "symbol": "C",
        "number": 13
      }, {
        "symbol": "H",
        "number": 9
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Xle",
      "name": "Leucine or Isoleucine diradical",
      "mf": "C6H11NO",
      "kind": "aa",
      "oneLetter": "J",
      "mass": 113.15787181078912,
      "monoisotopicMass": 113.08406397853,
      "unsaturation": 2,
      "elements": [{
        "symbol": "C",
        "number": 6
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Z",
      "name": "Benzyloxycarbonyl",
      "mf": "C8H7O2",
      "ocl": {
        "value": "dmtD`DpEeImYVUfh@@@@",
        "coordinates": "!Bb@I~@Ha}b@JH_Xc|_c~H_Xa}_c|"
      },
      "mass": 135.14028230090898,
      "monoisotopicMass": 135.04460446475,
      "unsaturation": 9,
      "elements": [{
        "symbol": "C",
        "number": 8
      }, {
        "symbol": "H",
        "number": 7
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Damp",
      "name": "Desoxyadenosine monophosphate diradical",
      "mf": "C10H12N5O5P",
      "kind": "DNAp",
      "oneLetter": "A",
      "alternativeOneLetter": "α",
      "ocl": {
        "value": "fnsiS@IASUlJB]xGbkplxyDhhldhiEEUeSdTekUUUULBATXPlKd@@",
        "coordinates": "!Bqc}{JxyO|XoSWC}W]poGQ\\Ou}]rmx\\Ou}]{qpza|qb}MJwlk^sFO|X"
      },
      "mass": 313.2069506932622,
      "monoisotopicMass": 313.05760550518,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 5
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Dcmp",
      "name": "Desoxycytidine monophosphate diradical",
      "mf": "C9H12N3O6P",
      "kind": "DNAp",
      "oneLetter": "C",
      "alternativeOneLetter": "ς",
      "ocl": {
        "value": "fjmps@IQKB`g^BCqUxV\\\\bTTVRTTbb^iqNZjjjifVkBEa\\`@",
        "coordinates": "!Bqc}{JxyO|XoSWA}_W]poGQ\\GuMKuMh\\Gu}]{qpSF]tWQTvatP"
      },
      "mass": 289.18221329795364,
      "monoisotopicMass": 289.04637211589,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 6
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Dgmp",
      "name": "Desoxyguanosine monophosphate diradical",
      "mf": "C10H12N5O6P",
      "kind": "DNAp",
      "oneLetter": "G",
      "alternativeOneLetter": "γ",
      "ocl": {
        "value": "fakhs@IASUlJB]{hOEWaYqrIQQYIQRJJkQTyEIZuUUUSRtsUaBpnP@",
        "coordinates": "!Bqc}{JxyO|XoSWA}W]poGQ\\Gu}]rmx\\Ou}]{qpza|qb}MJwlk^sFza|q`"
      },
      "mass": 329.20635561758047,
      "monoisotopicMass": 329.05252012475,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 6
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Dtmp",
      "name": "Desoxythymidine monophosphate diradical",
      "mf": "C10H13N2O7P",
      "kind": "DNAp",
      "oneLetter": "T",
      "alternativeOneLetter": "τ",
      "ocl": {
        "value": "ff}Qs@IQaPSoAjCqUxV\\\\bTTVRTTbbZUNIsUUUULsSVDKBy@@",
        "coordinates": "!Bqc}{JxyO|XoSWC}_W]poGQ\\GuMKuMh\\Gu}]{qpSF]tWQTvaSZGQ"
      },
      "mass": 304.1935916616171,
      "monoisotopicMass": 304.04603776326,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Dump",
      "name": "Desoxyuridine monophosphate diradical",
      "mf": "C9H11N2O7P",
      "kind": "DNAp",
      "oneLetter": "U",
      "alternativeOneLetter": "υ",
      "ocl": {
        "value": "fjmQs@IQaPSoAJCqUxV\\\\bTTVRTTbb^iqNZjjjifYkBEa\\`@",
        "coordinates": "!Bqc}{JxyO|XoSWA}_W]poGQ\\GuMKuMh\\Gu}]{qpSF]tWQTvatP"
      },
      "mass": 290.1669742567703,
      "monoisotopicMass": 290.0303876988,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Drmp",
      "name": "Desoxyribose monophosphate diradical",
      "mf": "C5H7O5P",
      "kind": "DNAp",
      "ocl": {
        "value": "d@",
        "coordinates": ""
      },
      "mass": 178.08005138207807,
      "monoisotopicMass": 178.00311032188,
      "unsaturation": 4,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 7
      }, {
        "symbol": "O",
        "number": 5
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Dadp",
      "name": "Desoxyadenosine diphosphate diradical",
      "mf": "C10H13N5O8P2",
      "kind": "DNApp",
      "oneLetter": "A",
      "ocl": {
        "value": "fmwhH`IASM\\JBl{wQ`|U^F_AkbdlsjsSOoRtyEMYuUUUM@pSEQaBpnP@",
        "coordinates": "!BIi[Rx{_grZOSXa}_]^H@mQbGu}utnDM^HGwWzf~_Ih}M_`AKvto[_`@_`A~grZ_I`"
      },
      "mass": 393.1868682186928,
      "monoisotopicMass": 393.02393639454,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 2
      }]
    }, {
      "symbol": "Dcdp",
      "name": "Desoxycytidine diphosphate diradical",
      "mf": "C9H13N3O9P2",
      "kind": "DNApp",
      "oneLetter": "C",
      "ocl": {
        "value": "fikqH`IQGB`kN|EoP^JoCOaUqrIQQYIQRJKGRJgDejjjjZYfZkBEa\\`@",
        "coordinates": "!BIi[Rx{_grZOSXa}_]^H@mQbGuMcqLX@m^H@gwWKB__t]Q_`@SFGx@Owx@_mQ"
      },
      "mass": 369.16213082338425,
      "monoisotopicMass": 369.01270300525005,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 9
      }, {
        "symbol": "P",
        "number": 2
      }]
    }, {
      "symbol": "Dgdp",
      "name": "Desoxyguanosine diphosphate diradical",
      "mf": "C10H13N5O9P2",
      "kind": "DNApp",
      "oneLetter": "G",
      "ocl": {
        "value": "fcoiH`IASM\\JBl{wQ{Axj|L~CWEIYgUff_^fZ\\bflzjjjfiZifZlHVEr@@",
        "coordinates": "!BIi[Rx{_grZOSXa}_]^H@mQbGu}utnD@m^H@gwWzf~_Ih}M_`AKvto[_`@_`A~gr[j[y|f"
      },
      "mass": 409.186273143011,
      "monoisotopicMass": 409.01885101411,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 9
      }, {
        "symbol": "P",
        "number": 2
      }]
    }, {
      "symbol": "Dtdp",
      "name": "Desoxythymidine diphosphate diradical",
      "mf": "C10H14N2O10P2",
      "kind": "DNApp",
      "oneLetter": "T",
      "ocl": {
        "value": "fe{Ph`IQaPUg^Ct\\p^JoCO`uqrIQQYIQRJKEJQTxdmUUUSSMTsVDKBy@@",
        "coordinates": "!BIi[Rx{_grZOSXa}_]^HMQbGuMcqLX@m^H@gwWKB__t]Q_`@SFALX_`@_`A~w}D"
      },
      "mass": 384.1735091870477,
      "monoisotopicMass": 384.01236865262,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 2
      }]
    }, {
      "symbol": "Dudp",
      "name": "Desoxyuridine diphosphate diradical",
      "mf": "C9H12N2O10P2",
      "kind": "DNApp",
      "oneLetter": "U",
      "ocl": {
        "value": "fikPh`IQaPUg^Bwhp^JoCOaUqrIQQYIQRJKGRJgDejjjjZYjYkBEa\\`@",
        "coordinates": "!BIi[Rx{_grZOSXa}_]^H@mQbGuMcqLX@m^H@gwWKB__t]Q_`@SFGx@Owx@_mQ"
      },
      "mass": 370.1468917822009,
      "monoisotopicMass": 369.99671858816,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 2
      }]
    }, {
      "symbol": "Datp",
      "name": "Desoxyadenosine triphosphate diradical",
      "mf": "C10H14N5O11P3",
      "kind": "DNAppp",
      "oneLetter": "A",
      "ocl": {
        "value": "eohZMJ@I@diehJAKGOFnakg`OESpr|Mo@yqrIQQYIQRJKYZQKVRcbIJjZjjjihFAhjZcAAXKb@@",
        "coordinates": "!BIi[Rx{_grZOSXa}_]^H@mQbGu}utnDM^H@gwWzf~_Ih}M_`AKvto[@hcW@`A~grZ_Igx@_`@@_c}~"
      },
      "mass": 473.16678574412344,
      "monoisotopicMass": 472.9902672839,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 11
      }, {
        "symbol": "P",
        "number": 3
      }]
    }, {
      "symbol": "Dctp",
      "name": "Desoxycytidine triphosphate diradical",
      "mf": "C9H14N3O12P3",
      "kind": "DNAppp",
      "oneLetter": "C",
      "ocl": {
        "value": "fkopZ`IQGB`kN|Fk^{NCqUxY|I~BwGHeEEdeEHhl]HlYJ\\RVjjjiifVjfkBEa\\`@",
        "coordinates": "!BIi[Rx{_grZOSXa}_]^H@mQbGuMcqLX@m^H@gwWKB__t]Q_`@SFOrHupH@_mQ_`A~@@A~Owx"
      },
      "mass": 449.14204834881485,
      "monoisotopicMass": 448.97903389461004,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 12
      }, {
        "symbol": "P",
        "number": 3
      }]
    }, {
      "symbol": "Dgtp",
      "name": "Desoxyguanosine triphosphate diradical",
      "mf": "C10H14N5O12P3",
      "kind": "DNAppp",
      "oneLetter": "G",
      "ocl": {
        "value": "e`TZCJ@I@diehJAKGOFnamgo`OESpr|CoByqrIQQYIQRJKYZQQYrT\\QIUSUUUUMRuMLtuVBBpWD@@",
        "coordinates": "!BIi[Rx{_grZOSXa}_]^H@mQbGu}utnD@m^H@gwWzf~_Ih}M_`AKvto[@hcW@`A~gr[j[y|f_`A~@@A~Owx"
      },
      "mass": 489.16619066844174,
      "monoisotopicMass": 488.98518190347005,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 12
      }, {
        "symbol": "P",
        "number": 3
      }]
    }, {
      "symbol": "Dttp",
      "name": "Desoxythymidine triphosphate diradical",
      "mf": "C10H15N2O13P3",
      "kind": "DNAppp",
      "oneLetter": "T",
      "ocl": {
        "value": "fgQZ`IQaPUg^BwhygnCqUxY|E~FwGHeEEdeEHhlTiDSISbRuUUUMLuMMMVDKBy@@",
        "coordinates": "!BIi[Rx{_grZOSXa}_]^H@mQbGuMcqLX@m^H@gwWKB__t]Q_`@SFALXHcW@`A~w}E~@Gx@@Gx_`"
      },
      "mass": 464.15342671247834,
      "monoisotopicMass": 463.97869954198,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 15
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 13
      }, {
        "symbol": "P",
        "number": 3
      }]
    }, {
      "symbol": "Dutp",
      "name": "Desoxyuridine triphosphate diradical",
      "mf": "C9H13N2O13P3",
      "kind": "DNAppp",
      "oneLetter": "U",
      "ocl": {
        "value": "fkoQZ`IQaPUg^CUoQ{NCqUxY|I~BwGHeEEdeEHhl]HlYJ\\RVjjjiiffffkBEa\\`@",
        "coordinates": "!BIi[Rx{_grZOSXa}_]^H@mQbGuMcqLX@m^H@gwWKB__t]Q_`@SFOrHupH@_mQ_`A~@@A~Owx"
      },
      "mass": 450.1268093076315,
      "monoisotopicMass": 449.96304947752,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 13
      }, {
        "symbol": "P",
        "number": 3
      }]
    }, {
      "symbol": "Dade",
      "name": "Desoxyadenosine diradical",
      "mf": "C10H11N5O2",
      "kind": "DNA",
      "oneLetter": "A",
      "ocl": {
        "value": "fluha@IF]ELJ@|QNJRsN|rntyYpXuUUTBATXPlKd@@",
        "coordinates": "!B\\KqpQARcg|T^|X@@Id`zeHo@Ie}]vaLcg|T^qAMDDvN_xy"
      },
      "mass": 233.22703316783156,
      "monoisotopicMass": 233.09127461582,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Dcyt",
      "name": "Desoxycytidine diradical",
      "mf": "C9H11N3O3",
      "kind": "DNA",
      "oneLetter": "C",
      "ocl": {
        "value": "fhiqa@IVCBa`^HgEIYg^Y~gG^jjjiejpaXWH@",
        "coordinates": "!BBOpH_UARcc}TN|Y@PIe`zeIO@MDSIrpXTd}RSqLgTd|"
      },
      "mass": 209.202295772523,
      "monoisotopicMass": 209.08004122653,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 3
      }]
    }, {
      "symbol": "Dgua",
      "name": "Desoxyguanosine diradical",
      "mf": "C10H11N5O3",
      "kind": "DNA",
      "oneLetter": "G",
      "ocl": {
        "value": "fbmia@IF]ELJYAxb\\Tef]ye^Z\\lxLZjjjeZfkBEa\\`@",
        "coordinates": "!B\\KqpQARcg|T^|X@@Id`zeHo@Ie}]vaLcg|T^qAMDDvN_vaLcg|"
      },
      "mass": 249.22643809214986,
      "monoisotopicMass": 249.08618923539,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 3
      }]
    }, {
      "symbol": "Dthy",
      "name": "Desoxythymidine diradical",
      "mf": "C10H12N2O4",
      "kind": "DNA",
      "oneLetter": "T",
      "ocl": {
        "value": "fdyPQ@IVaPtP^HgEIYg^YuiqwjjjjYikBEa\\`@",
        "coordinates": "!BBOpH_UARcc}TN|Y@PIe`zeIO@MDSIrpXTd}RSqLgDr]RSp"
      },
      "mass": 224.2136741361865,
      "monoisotopicMass": 224.07970687390002,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 4
      }]
    }, {
      "symbol": "Dura",
      "name": "Desoxyuridine diradical",
      "mf": "C9H10N2O4",
      "kind": "DNA",
      "oneLetter": "U",
      "ocl": {
        "value": "fhiPQ@IVaPpP^HgEIYg^Y~gG^jjjifZpaXWH@",
        "coordinates": "!BBOpH_UARcc}TN|Y@PIe`zeIO@MDSIrpXTd}RSqLgTd|"
      },
      "mass": 210.1870567313397,
      "monoisotopicMass": 210.06405680944,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 10
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 4
      }]
    }, {
      "symbol": "Amp",
      "name": "Adenosine monophosphate diradical",
      "mf": "C10H12N5O6P",
      "kind": "RNAp",
      "oneLetter": "A",
      "alternativeOneLetter": "α",
      "ocl": {
        "value": "fakhs@INBwlJ\\TgHOFwaEqrIQQSYQJIRIMLyxMVuUUUPLpEPQDqBId@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtPCQ@@"
      },
      "mass": 329.20635561758047,
      "monoisotopicMass": 329.05252012475,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 6
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Cmp",
      "name": "Cytidine monophosphate diradical",
      "mf": "C9H12N3O7P",
      "kind": "RNAp",
      "oneLetter": "C",
      "alternativeOneLetter": "ς",
      "ocl": {
        "value": "ff}qs@I^kBgENSdGc[pbxyDhhilheDiLv\\BVjjjjYfZbHfHQL`@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFDuP"
      },
      "mass": 305.1816182222719,
      "monoisotopicMass": 305.04128673546,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Gmp",
      "name": "Guanosine monophosphate diradical",
      "mf": "C10H12N5O7P",
      "kind": "RNAp",
      "oneLetter": "G",
      "alternativeOneLetter": "γ",
      "ocl": {
        "value": "fi{is@INBwlJ\\TgHp^MoBKcdRbbfrbTRdR\\SN^CUmUUUUKMSMQDSDHfP@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtP@gD}D@"
      },
      "mass": 345.20576054189877,
      "monoisotopicMass": 345.04743474432,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Tmp",
      "name": "Thymidine monophosphate diradical",
      "mf": "C10H13N2O8P",
      "kind": "RNAp",
      "oneLetter": "T",
      "alternativeOneLetter": "τ",
      "ocl": {
        "value": "fncPK@I^aSbgIrtGc[pbxyDhhilheDiLjs`RuUUUSLuMDQLPbY@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@"
      },
      "mass": 320.1929965859354,
      "monoisotopicMass": 320.04095238282997,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Ump",
      "name": "Uridine monophosphate diradical",
      "mf": "C9H11N2O8P",
      "kind": "RNAp",
      "oneLetter": "U",
      "alternativeOneLetter": "υ",
      "ocl": {
        "value": "ff}PK@I^aSbgIsTGc[pbxyDhhilheDiLv\\BVjjjjYffbHfHQL`@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFDuP"
      },
      "mass": 306.1663791810886,
      "monoisotopicMass": 306.02530231837,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Rmp",
      "name": "Ribose monophosphate diradical",
      "mf": "C5H7O6P",
      "kind": "RNAp",
      "ocl": {
        "value": "d@",
        "coordinates": ""
      },
      "mass": 194.07945630639637,
      "monoisotopicMass": 193.99802494145,
      "unsaturation": 4,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 7
      }, {
        "symbol": "O",
        "number": 6
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Adp",
      "name": "Adenosine diphosphate diradical",
      "mf": "C10H13N5O9P2",
      "kind": "RNApp",
      "oneLetter": "A",
      "ocl": {
        "value": "fcoiH`INCt\\J\\UENU{Axv|F~DwGHeEEMeDheHd\\eHsg`u{UUUU@mAEMPQDqBId@@",
        "coordinates": "!BvuPfpDnDtEK_t_rHtXBH_TwPbOr_IorHbGtgD}F@RxPBuxc|_]^OTh}R_`CQ`MF@_`@_`A~"
      },
      "mass": 409.186273143011,
      "monoisotopicMass": 409.01885101411,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 9
      }, {
        "symbol": "P",
        "number": 2
      }]
    }, {
      "symbol": "Cdp",
      "name": "Cytidine diphosphate diradical",
      "mf": "C9H13N3O10P2",
      "kind": "RNApp",
      "oneLetter": "C",
      "ocl": {
        "value": "fe{ph`I^[BgENbgHy`|[^E_CkcdRbbfrbTRdqrdYpIZjjjieijZbHfHQL`@",
        "coordinates": "!BvuPfpDnDtEK_t_rHtXBH_TwPb@K_cbpXbKtSItwPS]@Bux`Bo]\\lFGx@S]A~@C}~@Gx"
      },
      "mass": 385.1615357477025,
      "monoisotopicMass": 385.00761762482,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 2
      }]
    }, {
      "symbol": "Gdp",
      "name": "Guanosine diphosphate diradical",
      "mf": "C10H13N5O10P2",
      "kind": "RNApp",
      "oneLetter": "G",
      "ocl": {
        "value": "fkhh`INCt\\J\\UENY{NCqmxM|EnNQJJJ[JIQJQHzIRLyxM^uUUUTkUSLuQDSDHfP@",
        "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@J_I`JHbGtgD}F@RxPBux`B_]^OTh}R_`CQ`B\\StXA~@C}~@Gx"
      },
      "mass": 425.1856780673293,
      "monoisotopicMass": 425.01376563368,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 2
      }]
    }, {
      "symbol": "Tdp",
      "name": "Thymidine diphosphate diradical",
      "mf": "C10H14N2O11P2",
      "kind": "RNApp",
      "oneLetter": "T",
      "ocl": {
        "value": "fmgQh`I^aSbgQSglu`|[^C_@[bdls^rruo}LxDmUUUTruTsTQDqBId@@",
        "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPbOs_cbpXbGtSItwPS]@Bux`B_]\\lFBpX_`AMtGx@Owx@_`"
      },
      "mass": 400.172914111366,
      "monoisotopicMass": 400.00728327219,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 11
      }, {
        "symbol": "P",
        "number": 2
      }]
    }, {
      "symbol": "Udp",
      "name": "Uridine diphosphate diradical",
      "mf": "C9H12N2O11P2",
      "kind": "RNApp",
      "oneLetter": "U",
      "ocl": {
        "value": "fe{Qh`I^aSbgQSehy`|[^E_CkcdRbbfrbTRdqrdYpIZjjjiejfZbHfHQL`@",
        "coordinates": "!BvuPfpDnDtEK_t_rHtXBH_TwPb@K_cbpXbKtSItwPS]@Bux`Bo]\\lFGx@S]A~@C}~@Gx"
      },
      "mass": 386.14629670651925,
      "monoisotopicMass": 385.99163320773005,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 11
      }, {
        "symbol": "P",
        "number": 2
      }]
    }, {
      "symbol": "Atp",
      "name": "Adenosine triphosphate diradical",
      "mf": "C10H14N5O12P3",
      "kind": "RNAppp",
      "oneLetter": "A",
      "ocl": {
        "value": "e`TZCJ@IG@nahJNEHdliemgo`OFspZ|CoByqrIQQSYQJIRIGIRWRL\\^AU]UUUUPKPQMTuABDpaBX`@",
        "coordinates": "!BvuPfpDnDtEK_t_rHtXBH_TwPbOr_IorHbGtgD}F@RxS|uxc|_]^OTh}R_`CQ`MF@@hcW@A~_`A~@@A~Owx"
      },
      "mass": 489.16619066844174,
      "monoisotopicMass": 488.98518190347005,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 12
      }, {
        "symbol": "P",
        "number": 3
      }]
    }, {
      "symbol": "Ctp",
      "name": "Cytidine triphosphate diradical",
      "mf": "C9H14N3O13P3",
      "kind": "RNAppp",
      "oneLetter": "C",
      "ocl": {
        "value": "fgqZ`I^[BgENbgOQsO\\Gc[pkxK|MnNQJJJ[JIQJSGJPzQg@ejjjjfVffjZbHfHQL`@",
        "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@K_cbpXbGtSItwPS]C|ux`B_]\\lFGx@S]@BbM\\B@Gy~@Gx@@Gx_`"
      },
      "mass": 465.1414532731331,
      "monoisotopicMass": 464.97394851418,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 13
      }, {
        "symbol": "P",
        "number": 3
      }]
    }, {
      "symbol": "Gtp",
      "name": "Guanosine triphosphate diradical",
      "mf": "C10H14N5O13P3",
      "kind": "RNAppp",
      "oneLetter": "G",
      "ocl": {
        "value": "eh\\ZKJ@IG@nahJNEHdliemco`POFspZ|KoAyqrIQQSYQJIRIGQJQzQccpJkjjjjjeZjYZijbDIaBDq@@",
        "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@J_I`JHbGtgD}F@RxPBux`B_]^OTh}R_`CQ`B\\StX@BbM_|@Gy~@Gx@@Gx_`"
      },
      "mass": 505.16559559276,
      "monoisotopicMass": 504.98009652304,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 13
      }, {
        "symbol": "P",
        "number": 3
      }]
    }, {
      "symbol": "Ttp",
      "name": "Thymidine triphosphate diradical",
      "mf": "C10H15N2O14P3",
      "kind": "RNAppp",
      "oneLetter": "T",
      "ocl": {
        "value": "eo`TGJ@IOHJNEGHdlnaiekg`OFspZ|Mo@yqrIQQSYQJIRY[ZPzQc`HjjjjjYZjVjZbDIaBDq@@",
        "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@K_cbpXbGtSItwPS]@Bux`B_]\\lFBpX_`AMt@JHupH@_gx@_`@@_c}~"
      },
      "mass": 480.15283163679663,
      "monoisotopicMass": 479.97361416155,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 15
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 14
      }, {
        "symbol": "P",
        "number": 3
      }]
    }, {
      "symbol": "Utp",
      "name": "Uridine triphosphate diradical",
      "mf": "C9H13N2O14P3",
      "kind": "RNAppp",
      "oneLetter": "U",
      "ocl": {
        "value": "fgPz`I^aSbgQSeoQsO\\Gc[pkxK|MnNQJJJ[JIQJSGJPzQg@ejjjjfVjVjZbHfHQL`@",
        "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@K_cbpXbGtSItwPS]C|ux`B_]\\lFGx@S]@BbM\\B@Gy~@Gx@@Gx_`"
      },
      "mass": 466.12621423194986,
      "monoisotopicMass": 465.95796409709004,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 14
      }, {
        "symbol": "P",
        "number": 3
      }]
    }, {
      "symbol": "Ade",
      "name": "Adenosine diradical",
      "mf": "C10H11N5O3",
      "kind": "RNA",
      "oneLetter": "A",
      "ocl": {
        "value": "fbmia@IV|gLJ\\Axj\\Tef[vyWV\\]zJZjjj`PJ`bIbDSH@",
        "coordinates": "!BBOpH_UARccFPEP{PId{RpBN[~i|BEP{iVA@fUARU@QTADBYPId"
      },
      "mass": 249.22643809214986,
      "monoisotopicMass": 249.08618923539,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 3
      }]
    }, {
      "symbol": "Cyt",
      "name": "Cytidine diradical",
      "mf": "C9H11N3O4",
      "kind": "RNA",
      "oneLetter": "C",
      "ocl": {
        "value": "fdypQ@INcBgK@|UNJRsM{\\~sg`uUUULmQDSDHfP@",
        "coordinates": "!BBOpH_UARccFPEP{PId{RpBN[~iRTBpgDq`@c`BNKB\\@c`"
      },
      "mass": 225.20170069684127,
      "monoisotopicMass": 225.0749558461,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 4
      }]
    }, {
      "symbol": "Gua",
      "name": "Guanosine diradical",
      "mf": "C10H11N5O4",
      "kind": "RNA",
      "oneLetter": "G",
      "ocl": {
        "value": "fj}hQ@IV|gLJ\\JCqTxiKLwmroKNN}EMUUUTkTuDQLPbY@@",
        "coordinates": "!BBOpH_UARccFPEP{PId{RpBN[~k|BEP{iVA@fUARU@QTADBYiVA@fP"
      },
      "mass": 265.22584301646816,
      "monoisotopicMass": 265.08110385496,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 4
      }]
    }, {
      "symbol": "Thy",
      "name": "Thymidine diradical",
      "mf": "C10H12N2O5",
      "kind": "RNA",
      "oneLetter": "T",
      "ocl": {
        "value": "fleQQ@INaSed`|UNJRsM{\\zlyxMUUUSMMDQLPbY@@",
        "coordinates": "!BBOpH_UARccFPEP{PId{RpBN[~iRTBpgDq`@c`BNKB\\lIpBN"
      },
      "mass": 240.21307906050478,
      "monoisotopicMass": 240.07462149347,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 5
      }]
    }, {
      "symbol": "Ura",
      "name": "Uridine diradical",
      "mf": "C9H10N2O5",
      "kind": "RNA",
      "oneLetter": "U",
      "ocl": {
        "value": "fdyQQ@INaSeh`|UNJRsM{\\~sg`uUUULsQDSDHfP@",
        "coordinates": "!BBOpH_UARccFPEP{PId{RpBN[~iRTBpgDq`@c`BNKB\\@c`"
      },
      "mass": 226.18646165565798,
      "monoisotopicMass": 226.05897142901,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 10
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 5
      }]
    }, {
      "symbol": "Dam",
      "name": "1,2′-O-dimethyladenosine monophosphate diradical 01A",
      "mf": "C12H16N5O6P",
      "kind": "NucleotideP",
      "oneLetter": "œ",
      "ocl": {
        "value": "feghs@E^ct\\J\\udhOEw`eqrIQQQKZIQJQIiLxFK^uUUUUKLtuQDSDHfP@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuKQ@MD@SuH"
      },
      "mass": 357.2595904272741,
      "monoisotopicMass": 357.08382025367,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 6
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Dgm",
      "name": "1,2′-O-dimethylguanosine monophosphate diradical 01G",
      "mf": "C12H16N5O7P",
      "kind": "NucleotideP",
      "oneLetter": "ε",
      "ocl": {
        "value": "fmwis@E^ct\\J\\udlp^KoAKcdRbbbVtRbTbSbSNAbwmUUUURsMSUDQLPbY@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuKQ@B\\StPAOT`"
      },
      "mass": 373.2589953515923,
      "monoisotopicMass": 373.07873487324,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Dim",
      "name": "1,2′-O-dimethylinosine monophosphate diradical 019A",
      "mf": "C12O7N4H15P",
      "kind": "NucleotideP",
      "oneLetter": "ξ",
      "ocl": {
        "value": "fegIs@E^cvENZrTXOEw`eqrIQQQKZIQJQIiLxFK^uUUUUKLtuQDSDHfP@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuKQ@MD@SuH"
      },
      "mass": 358.2443513860907,
      "monoisotopicMass": 358.06783583658,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 15
      }, {
        "symbol": "N",
        "number": 4
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Tia",
      "name": "2- methylthiomethylenethio-N6-isopentenyl-adenosine monophosphate diradical",
      "mf": "C17H24N5O6PS2",
      "kind": "NucleotideP",
      "oneLetter": "£",
      "ocl": {
        "value": "eh\\ZFJ@IG@nahJNEDl`OFspb\\V`cXHrIQQSYQJIRINIYIKQccpJkjjjjjAfBJjfjBDIaBDq@@",
        "coordinates": "!BpBYTvxBNFY|bEJObGvOS\\@Yt]~DUEJOctu~@Ha}`HzOSTwPTh~H@hc|_`BH_Xa}b@JH@gx@bGvH@h`B_`BH_P"
      },
      "mass": 489.50637075565066,
      "monoisotopicMass": 489.09056286031,
      "unsaturation": 16,
      "elements": [{
        "symbol": "C",
        "number": 17
      }, {
        "symbol": "H",
        "number": 24
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 6
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 2
      }]
    }, {
      "symbol": "Mhc",
      "name": "2′‐O‐Methyl-5-hydroxymethylcytidine monophosphate diradical",
      "mf": "C11H16N3O8P",
      "kind": "NucleotideP",
      "oneLetter": "¡",
      "ocl": {
        "value": "fikpK@EA{BgM^rTXOEw`eqrIQQQKZIQJSJigHujjjjifYjkBHQL`@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@HoTuOSU@HC~NKA`HoQLgSUAMT@a}oS_|BBpXKAaMT@CQ"
      },
      "mass": 349.2342579562838,
      "monoisotopicMass": 349.06750148395,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Odg",
      "name": "N2,2′-O-dimethylguanosine monophosphate diradical 02G",
      "mf": "C12H16N5O7P",
      "kind": "NucleotideP",
      "oneLetter": "γ",
      "ocl": {
        "value": "fmwis@E^ct\\J\\udlp^KoAKcdRbbbVtRbTbSbsNAbwmUUUURsMSUDQLPbY@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuKQ@B\\StPAOT`"
      },
      "mass": 373.2589953515923,
      "monoisotopicMass": 373.07873487324,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Ntg",
      "name": "N2,N2,2′-O-trimethylguanosine monophosphate diradical 022G",
      "mf": "C13H18N5O7P",
      "kind": "NucleotideP",
      "oneLetter": "|",
      "ocl": {
        "value": "fcois@E^ct\\J\\udlp^KoAKcdRbbbVtRbTbSbTYpLVcjjjjjVYjZjHbXaDr@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuHgD}D@tPBNOt}R"
      },
      "mass": 387.2856127564392,
      "monoisotopicMass": 387.0943849377,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 13
      }, {
        "symbol": "H",
        "number": 18
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Otg",
      "name": "N2,7,2′-O-trimethylguanosine monophosphate diradical 027G",
      "mf": "C13H20N5O7P",
      "kind": "NucleotideP",
      "oneLetter": "æ",
      "ocl": {
        "value": "fcoisBE^bN\\J\\udjp^KoAKcFU}dRbbbVtRbTbRlQYpLVcjjjjjVYjjjHbXaDr@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuKFPMD@IqOQ@D}R"
      },
      "mass": 389.30149426455074,
      "monoisotopicMass": 389.11003500216,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 13
      }, {
        "symbol": "H",
        "number": 20
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Rya",
      "name": "2′-O-ribosyladenosine monophosphate diradical 00A",
      "mf": "C15H20N5O9P",
      "kind": "NucleotideP",
      "oneLetter": "^",
      "ocl": {
        "value": "e`\\ZIL@DaegobFAIO@hlm`OGSp\\\\\\bbbfrRbdTT\\rbRQUCDQTrusuUUUUMUU@pET@@@",
        "coordinates": "!BIlAKaMARw}DBbMF@bGuMtHc|KAbH_ZU`@GzH_WwW@h`XKFjKB_jXB\\SiVA`zmG_Irp_hQKctvOSR\\lIrp"
      },
      "mass": 445.3217759066577,
      "monoisotopicMass": 445.09986424130005,
      "unsaturation": 16,
      "elements": [{
        "symbol": "C",
        "number": 15
      }, {
        "symbol": "H",
        "number": 20
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 9
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Ryg",
      "name": "2′-O-ribosylguanosine monophosphate diradical 00G",
      "mf": "C15H20N5O10P",
      "kind": "NucleotideP",
      "oneLetter": "ℑ",
      "ocl": {
        "value": "ehRZEL@DaegobFAIO@hlm`POGSp\\\\\\bbbfrRbdTT\\rbRQbhXbJfVn^jjjjijjjVZfj@@@",
        "coordinates": "!BIlAKaMARw}DBbMF@bGuMtH`BKAbH_ZU`@GzH_WwW@h`XKFjKB_jXB\\SiVA`zmG_Irp_hQKctvOSR\\lt]|gK@"
      },
      "mass": 461.321180830976,
      "monoisotopicMass": 461.09477886087,
      "unsaturation": 16,
      "elements": [{
        "symbol": "C",
        "number": 15
      }, {
        "symbol": "H",
        "number": 20
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Tmu",
      "name": "2-thio-2′-O-methyluridine monophosphate diradical 02U",
      "mf": "C10H13N2O7PS",
      "kind": "NucleotideP",
      "oneLetter": "∏",
      "ocl": {
        "value": "fncQp`EAaSfleZCq]x^BDnNQJJJI[QJIRYlyFmUUUULsSQDSDHfP@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXSU@"
      },
      "mass": 336.25837906774416,
      "monoisotopicMass": 336.01810893766003,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Dmut",
      "name": "3,2′-O-dimethyluridine monophosphate diradical 03U",
      "mf": "C11H15N2O8P",
      "kind": "NucleotideP",
      "oneLetter": "σ",
      "ocl": {
        "value": "fasPK@EAaSfoYKtGb{pRxyDhhhemDheIhv\\cVjjjjfYjZHbXaDr@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXOxyMT@"
      },
      "mass": 334.2196139907822,
      "monoisotopicMass": 334.05660244729,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 15
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Amc",
      "name": "N4-acetyl-2′-O-methylcytidine monophosphate diradical 042C",
      "mf": "C12H16N3O8P",
      "kind": "NucleotideP",
      "oneLetter": "ℵ",
      "ocl": {
        "value": "fe{pK@EA[BgM^rTXOEw`eqrIQQQKZIQJSMJLyFmUUUULsMMQDSDHfP@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXSU@caLgSU@"
      },
      "mass": 361.244993853019,
      "monoisotopicMass": 361.06750148395,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Tmc",
      "name": "N4,N4,2′-O-trimethylcytidine monophosphate diradical 044C",
      "mf": "C12H18N3O7P",
      "kind": "NucleotideP",
      "oneLetter": "β",
      "ocl": {
        "value": "fikqs@EA[BgM^rTGb{pRxyDhhhemDheIfhsdZuUUUTsLuTQDqBId@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXSU@cbpX"
      },
      "mass": 347.2614704368123,
      "monoisotopicMass": 347.08823692884005,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 18
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Dcy",
      "name": "N4,2′-O-dimethylcytidine monophosphate diradical 04C",
      "mf": "C11H16N3O7P",
      "kind": "NucleotideP",
      "oneLetter": "λ",
      "ocl": {
        "value": "fasqs@EA[BgM^rTGb{pRxyDhhhemDheIff\\cVjjjjfYfjHbXaDr@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXSU@lF@"
      },
      "mass": 333.23485303196554,
      "monoisotopicMass": 333.07258686438,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Muo",
      "name": "2′-O-methyluridine 5-oxyacetic acid methyl ester monophosphate diradical 0503U",
      "mf": "C13H17N2O11P",
      "kind": "NucleotideP",
      "oneLetter": "Ͽ",
      "ocl": {
        "value": "fkoQk@EAaSfoYJwj}`|W^BWGHeEEDmheDiLjlif\\cVjjjjfYjZZhbIbDSH@",
        "coordinates": "!BKAb@tURD@m\\YpMAMpBYMcvjbOplIwx@bGuMc}\\Bb@JH@dvOcuKPSXa}bGvHH`BbGu~Oxc|bGt"
      },
      "mass": 408.25518206531905,
      "monoisotopicMass": 408.05699637046,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 13
      }, {
        "symbol": "H",
        "number": 17
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 11
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Xmu",
      "name": "5-carboxymethylaminomethyl-2′-O-methyluridine monophosphate diradical 051U",
      "mf": "C13H18N3O10P",
      "kind": "NucleotideP",
      "oneLetter": ")",
      "ocl": {
        "value": "fkopk@EAGBgM^rWns`|W^BWGHeEEDmheDiLjleF\\cVjjjjfYjZfhbIbDSH@",
        "coordinates": "!BKAb@tURD@m\\YpMAMpBYMcvjb@HlIwx@bGuMc}\\Bb@JH@dvOcuKPSXa}bGvH@h`BbGvH@gx@bKt"
      },
      "mass": 407.2704211065024,
      "monoisotopicMass": 407.07298078755,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 13
      }, {
        "symbol": "H",
        "number": 18
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Mmu",
      "name": "5-methoxycarbonylmethyl-2′-O-methyluridine monophosphate diradical 0521U",
      "mf": "C13H17N2O10P",
      "kind": "NucleotideP",
      "oneLetter": "∩",
      "ocl": {
        "value": "fcwPk@EAaSfoYKvZp^KoAKcdRbbbVtRbTfUVfYrMZjjjjYfifjHbXaDr@@",
        "coordinates": "!BS]@lFJU`@Gyoza`lzf@lI}mK_`B@cm\\Bb@HlI}]}_`A~@BpgIqLXKH`Bb@I~@Ha}_c~HHa}"
      },
      "mass": 392.2557771410008,
      "monoisotopicMass": 392.06208175089,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 13
      }, {
        "symbol": "H",
        "number": 17
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Cue",
      "name": "5-(carboxyhydroxymethyl)-2′-O-methyluridine methyl ester monophosphate diradical 0522U",
      "mf": "C13H17N2O11P",
      "kind": "NucleotideP",
      "oneLetter": "∩",
      "ocl": {
        "value": "fkoQk@EAaSfoYKtZ}`|W^BWGHeEEDmheDiLjhYf\\cVjjjjfYjZZhbIbDSH@",
        "coordinates": "!BS]@lFJU`@Gyoza`lzf@lI}mK_`B@cm\\Bb@HlI}]}_`A~@BpgIqLXKH`Bb@I~@Gx@bGu~Oxc|bGt"
      },
      "mass": 408.25518206531905,
      "monoisotopicMass": 408.05699637046,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 13
      }, {
        "symbol": "H",
        "number": 17
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 11
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Cyu",
      "name": "5-carbamoylmethyl-2′-O-methyluridine monophosphate diradical 053U",
      "mf": "C12H16N3O9P",
      "kind": "NucleotideP",
      "oneLetter": "~",
      "ocl": {
        "value": "fmgqK@EAWBgM^rWlp^KoAKcdRbbbVtRbTfUVcNQkUUUUSLuLuDQLPbY@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXKAaMTDuPOxxlF@"
      },
      "mass": 377.24439877733727,
      "monoisotopicMass": 377.06241610352,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 9
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Ipu",
      "name": "5-(isopentenylaminomethyl)-2′-O-methyluridine monophosphate diradical 0583U",
      "mf": "C16H24N3O8P",
      "kind": "NucleotideP",
      "oneLetter": "¼",
      "ocl": {
        "value": "fgpK@EAGBgM^rWhOEw`eqrIQQQKZIQJSJkIJLyFmUUUULsTuMTQDqBId@@",
        "coordinates": "!BS]@lFJU`@Gyoza`lzf@lI}mK_`B@cm\\Bb@HlI}]}_`A~@BpgIqLXKH`Bb@I~@Ha}b@JH_Xc|_`BH_P"
      },
      "mass": 417.35146347240624,
      "monoisotopicMass": 417.13010174179004,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 16
      }, {
        "symbol": "H",
        "number": 24
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Mcy",
      "name": "5,2′-O-dimethylcytidine monophosphate diradical monophosphate diradical 05C",
      "mf": "C11H16N3O7P",
      "kind": "NucleotideP",
      "oneLetter": "τ",
      "ocl": {
        "value": "fasqs@EA{BgM^rTGb{pRxyDhhhemDheIeV\\cVjjjjfYfjHbXaDr@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXKAaMT@"
      },
      "mass": 333.23485303196554,
      "monoisotopicMass": 333.07258686438,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Dmuf",
      "name": "5,2′-O-dimethyluridine monophosphate diradical 05U",
      "mf": "C11H15N2O8P",
      "kind": "NucleotideP",
      "oneLetter": "\\",
      "ocl": {
        "value": "fasPK@EAaSfoYKtGb{pRxyDhhhemDheIeV\\cVjjjjfYjZHbXaDr@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXKAaMT@"
      },
      "mass": 334.2196139907822,
      "monoisotopicMass": 334.05660244729,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 15
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Tma",
      "name": "N6,N6,2′-O-trimethyladenosine monophosphate diradical 066A",
      "mf": "C13H18N5O6P",
      "kind": "NucleotideP",
      "oneLetter": "η",
      "ocl": {
        "value": "fmwhs@E^ct\\J\\udhOEw`eqrIQQQKZIQJQIkQg@q[vjjjj`Y`JjBHfHQL`@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuKQ@MD@SuHXK@"
      },
      "mass": 371.2862078321209,
      "monoisotopicMass": 371.09947031813005,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 13
      }, {
        "symbol": "H",
        "number": 18
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 6
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Nda",
      "name": "N6,2′-O-dimethyladenosine monophosphate diradical 06A",
      "mf": "C12H16N5O6P",
      "kind": "NucleotideP",
      "oneLetter": "χ",
      "ocl": {
        "value": "feghs@E^ct\\J\\udhOEw`eqrIQQQKZIQJQIkLxFK^uUUUTCLAUADSDHfP@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuKQ@MD@FBp"
      },
      "mass": 357.2595904272741,
      "monoisotopicMass": 357.08382025367,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 6
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Fmc",
      "name": "5-formyl-2′-O-methylcytidine monophosphate diradical 071C",
      "mf": "C10H12N3O8P",
      "kind": "NucleotideP",
      "oneLetter": "°",
      "ocl": {
        "value": "faspK@I^[BgENSghOFwaEqrIQQSYQJIRYULxDmUUUTsLttQDqBId@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@@tP"
      },
      "mass": 333.1917590433254,
      "monoisotopicMass": 333.03620135502996,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Omi",
      "name": "2′-O-methylinosine monophosphate diradical 09A",
      "mf": "C11H13N4O7P",
      "kind": "NucleotideP",
      "oneLetter": "≤",
      "ocl": {
        "value": "fi{Is@E^cvENZrTXOEw`eqrIQQQKZIQJQIig@q[vjjjjiYffhbIbDSH@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuKQ@MD@"
      },
      "mass": 344.21773398124395,
      "monoisotopicMass": 344.05218577211997,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 4
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Opu",
      "name": "2′-O-methylpseudouridine monophosphate diradical 09U",
      "mf": "C10H13N2O8P",
      "kind": "NucleotideP",
      "oneLetter": "Z",
      "ocl": {
        "value": "fncPK@@qaSfoYJtGb{pRxyDhhhemDheIfsdZuUUULuMMDQLPbY@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXSU@"
      },
      "mass": 320.1929965859354,
      "monoisotopicMass": 320.04095238282997,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Oma",
      "name": "2′-O-methyladenosine monophosphate diradical 0A",
      "mf": "C11H14N5O6P",
      "kind": "NucleotideP",
      "oneLetter": ":",
      "ocl": {
        "value": "fi{hs@E^ct\\J\\udhOEw`eqrIQQQKZIQJQIig@q[vjjjj`Y`J`bIbDSH@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuKQ@MD@"
      },
      "mass": 343.2329730224273,
      "monoisotopicMass": 343.06817018921,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 6
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Omc",
      "name": "2′-O-methylcytidine monophosphate diradical 0C",
      "mf": "C10H14N3O7P",
      "kind": "NucleotideP",
      "oneLetter": "B",
      "ocl": {
        "value": "fncqs@EA[BgM^rTGb{pRxyDhhhemDheIfsdZuUUUTsLuDQLPbY@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXSU@"
      },
      "mass": 319.2082356271187,
      "monoisotopicMass": 319.05693679992004,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Omg",
      "name": "2′-O-methylguanosine monophosphate diradical 0G",
      "mf": "C11H14N5O7P",
      "kind": "NucleotideP",
      "oneLetter": "#",
      "ocl": {
        "value": "fegis@E^ct\\J\\udlp^KoAKcdRbbbVtRbTbSbYpLV}jjjjjVYjZbHfHQL`@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@HoTuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuKQ@B\\StP@"
      },
      "mass": 359.23237794674554,
      "monoisotopicMass": 359.06308480878,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Omu",
      "name": "2′-O-methyluridinemonophosphate diradical 0U",
      "mf": "C10H13N2O8P",
      "kind": "NucleotideP",
      "oneLetter": "J",
      "ocl": {
        "value": "fncPK@EAaSfoYJtGb{pRxyDhhhemDheIfsdZuUUUTsMMDQLPbY@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXSU@"
      },
      "mass": 320.1929965859354,
      "monoisotopicMass": 320.04095238282997,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Cdg",
      "name": "7-cyano-7-deazaguanosine monophosphate diradical 100G",
      "mf": "C12H12N5O7P",
      "kind": "NucleotideP",
      "oneLetter": "φ",
      "ocl": {
        "value": "fmwis@INzM\\J\\TgLp^MoBKcdRbbfrbTRdRUbSN^CWmUUUUKLuSuDQLPbY@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RqdCQ@B\\StPCFP@"
      },
      "mass": 369.22723233536925,
      "monoisotopicMass": 369.04743474432,
      "unsaturation": 18,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Azg",
      "name": "7-aminomethyl-7-deazaguanosine monophosphate diradical 101G",
      "mf": "C12H16N5O7P",
      "kind": "NucleotideP",
      "oneLetter": "∉",
      "ocl": {
        "value": "fmwis@INzM\\J\\TgLp^MoBKcdRbbfrbTRdRUbSN^CWmUUUUKLuSUDQLPbY@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RqdCQ@B\\StPA`z`"
      },
      "mass": 373.2589953515923,
      "monoisotopicMass": 373.07873487324,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Eqo",
      "name": "epoxyqueuosine monophosphate diradical 102G",
      "mf": "C17H22N5O10P",
      "kind": "NucleotideP",
      "oneLetter": "ς",
      "ocl": {
        "value": "el^ZEL@IGNaehJNEDlig`TPOFspb\\\\bTTTvTRbTbRlRjbbfXx|Bjz~aAajjjjiYfjZjjjjHPfDHSD@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RqdCQ@B\\StP@{ULY@TEIKA@a}tPA}BOpHAEP"
      },
      "mass": 487.3585341325581,
      "monoisotopicMass": 487.11042892533,
      "unsaturation": 18,
      "elements": [{
        "symbol": "C",
        "number": 17
      }, {
        "symbol": "H",
        "number": 22
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Aes",
      "name": "archaeosine monophosphate diradical 103G",
      "mf": "C12H15N6O7P",
      "kind": "NucleotideP",
      "oneLetter": "(",
      "ocl": {
        "value": "fcoYs@INzM^xTxiNY`|[^DWGHeEEMeDheHdkDhsg`u{UUUURsMTmTQDqBId@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RqdCQ@B\\StP@{UFCj"
      },
      "mass": 386.2577578089824,
      "monoisotopicMass": 386.07398384544,
      "unsaturation": 16,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 15
      }, {
        "symbol": "N",
        "number": 6
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Goq",
      "name": "galactosyl-queuosine monophosphate diradical 104G",
      "mf": "C23H32N5O14P",
      "kind": "NucleotideP",
      "oneLetter": "9",
      "ocl": {
        "value": "ekXzGL@IGNaehJNEDliod\\VU]SPOFspb\\\\bTTTvTRbTbRlRjbTrTrbfRXx|Bjz^AyEjjjjiYfjZijjjjjjbDIaBDq@@",
        "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@J_IorHbGtgD}F@RxRH_WwW@hbOTh}RIlCQ`B\\StXC[UB[@RxSPT`JHbGwQ`H`BaEQ~@Ha}bOq~Ox`BbGu~@Ha}bOrH@`"
      },
      "mass": 633.4999767508004,
      "monoisotopicMass": 633.16833772591,
      "unsaturation": 20,
      "elements": [{
        "symbol": "C",
        "number": 23
      }, {
        "symbol": "H",
        "number": 32
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 14
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Gaq",
      "name": "glutamyl-queuosine monophosphate diradical105G",
      "mf": "C22H29N6O12P",
      "kind": "NucleotideP",
      "oneLetter": "⊄",
      "ocl": {
        "value": "emWVCL@IGNaejXJNEDlioh\\YUPOFspb\\\\bTTTvTRbTbRlRjbTJTtrTXx|Bjz^AjjjjiYfjZijfjfjbDIaBDq@@",
        "coordinates": "!BTmB@c`JHUMmMtL@YtEHYgxQTaDoQ`L@YFY|gKMARH`Ygy|fpAfN`Hz@`H{PTb\\ltEIRtHBNHaTv|@YFYPTha}b@I~@Ha}_c~H@ha}bOq~@Ha}"
      },
      "mass": 600.473311954707,
      "monoisotopicMass": 600.15810739451,
      "unsaturation": 22,
      "elements": [{
        "symbol": "C",
        "number": 22
      }, {
        "symbol": "H",
        "number": 29
      }, {
        "symbol": "N",
        "number": 6
      }, {
        "symbol": "O",
        "number": 12
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Moq",
      "name": "mannosyl-queuosine monophosphate diradical 106G",
      "mf": "C23H32N5O14P",
      "kind": "NucleotideP",
      "oneLetter": "8",
      "ocl": {
        "value": "ekXzGL@IGNaehJNEDliod\\VU]SPOFspb\\\\bTTTvTRbTbRlRjbTrTrbfRXx|Bjz^AyEjjjjiYfjZijjjjjjbDIaBDq@@",
        "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@J_IorHbGtgD}F@RxRH_WwW@hbOTh}RIlCQ`B\\StXC[UB[@RxSPT`JHbGwQ`H`BaEQ~@Ha}bOq~Ox`BbGu~@Ha}bOrH@`"
      },
      "mass": 633.4999767508004,
      "monoisotopicMass": 633.16833772591,
      "unsaturation": 20,
      "elements": [{
        "symbol": "C",
        "number": 23
      }, {
        "symbol": "H",
        "number": 32
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 14
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Qus",
      "name": "queuosine monophosphate diradical 10G",
      "mf": "C17H22N5O9P",
      "kind": "NucleotideP",
      "oneLetter": "Q",
      "ocl": {
        "value": "edZZIL@IGNaehJNEDliohPOFspb\\\\bTTTvTRbTbRlRjbTKGG`UWSpMUUUUKLuSUMUTPaLHPfH@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPHoWtw@aOTd}RqdCQ@B\\StP@{ULY@RpQPTopHBGwQ@@QT"
      },
      "mass": 471.35912920823984,
      "monoisotopicMass": 471.11551430576,
      "unsaturation": 18,
      "elements": [{
        "symbol": "C",
        "number": 17
      }, {
        "symbol": "H",
        "number": 22
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 9
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Cpo",
      "name": "1-methyl-3-(3-amino-3-carboxypropyl)pseudouridine monophosphate diradical 1309U",
      "mf": "C14H20N3O10P",
      "kind": "NucleotideP",
      "oneLetter": "α",
      "ocl": {
        "value": "fgpk@OAWBgENSgi{`|[^DWGHeEEMeDheIhjbihs`RuUUTsTuSUMQDSDHfP@",
        "coordinates": "!BTh|SI~ioOwy`iR\\SiV|SFGxw}FH_]]}DqbH@gx_c|SFA`lIqOW_Xa}uwu~Ox`BbGu~Ox`B_`BH_P"
      },
      "mass": 421.2970385113492,
      "monoisotopicMass": 421.08863085201,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 14
      }, {
        "symbol": "H",
        "number": 20
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Mls",
      "name": "1-methylinosine monophosphate diradical 19A",
      "mf": "C11H13N4O7P",
      "kind": "NucleotideP",
      "oneLetter": "O",
      "ocl": {
        "value": "fi{Is@INBvENJSghOFwaEqrIQQSYQJIRIMIgOAjvjjjjefZZhbIbDSH@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtPCQ@D}R"
      },
      "mass": 344.21773398124395,
      "monoisotopicMass": 344.05218577211997,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 4
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Mpu",
      "name": "1-methylpseudouridine monophosphate diradical 19U",
      "mf": "C10H13N2O8P",
      "kind": "NucleotideP",
      "oneLetter": "]",
      "ocl": {
        "value": "fncPK@OAaSbgIrtGc[pbxyDhhilheDiLjs`RuUUTsTuMDQLPbY@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@"
      },
      "mass": 320.1929965859354,
      "monoisotopicMass": 320.04095238282997,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Mad",
      "name": "1-methyladenosine monophosphate diradical 1A",
      "mf": "C11H14N5O6P",
      "kind": "NucleotideP",
      "oneLetter": "\"",
      "ocl": {
        "value": "fi{hs@INBwlJ\\TgHOFwaEqrIQQSYQJIRIMIgOAjvjjjjefZZhbIbDSH@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtPCQ@D}R"
      },
      "mass": 343.2329730224273,
      "monoisotopicMass": 343.06817018921,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 6
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Mgs",
      "name": "1-methylguanosine monophosphate diradical 1G",
      "mf": "C11H14N5O7P",
      "kind": "NucleotideP",
      "oneLetter": "K",
      "ocl": {
        "value": "fegis@INBwlJ\\TgHp^MoBKcdRbbfrbTRdR\\RYspZmjjjjiYfijbHfHQL`@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtP@gD}D@SuH"
      },
      "mass": 359.23237794674554,
      "monoisotopicMass": 359.06308480878,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Msu",
      "name": "5-aminomethyl-2-selenouridine monophosphate diradical 20510U",
      "mf": "C10H14N3O7PSe",
      "kind": "NucleotideP",
      "oneLetter": "π",
      "ocl": {
        "value": "fasqp`I^{BgEIrtGc[p\\bQ\\\\bTTTvTRbTfUSNAKUUUULsTuDQLPbY@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@@tP"
      },
      "mass": 398.1676241841323,
      "monoisotopicMass": 398.97345859992004,
      "unsaturation": null,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "Se",
        "number": 1
      }]
    }, {
      "symbol": "Mse",
      "name": "5-methylaminomethyl-2-selenouridine monophosphate diradical 20511U",
      "mf": "C11H16N3O7PSe",
      "kind": "NucleotideP",
      "oneLetter": "≅",
      "ocl": {
        "value": "fikqp`I^{BgEIrtGc[p\\bQ\\\\bTTTvTRbTfUVYpIZjjjifZfjHbXaDr@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSUAMTBpX"
      },
      "mass": 412.19424158897914,
      "monoisotopicMass": 412.98910866438,
      "unsaturation": null,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "Se",
        "number": 1
      }]
    }, {
      "symbol": "Cse",
      "name": "5-carboxymethylaminomethyl-2-selenouridine monophosphate diradical 2051U",
      "mf": "C12H16N3O9PSe",
      "kind": "NucleotideP",
      "oneLetter": "⊥",
      "ocl": {
        "value": "fcwqH`I^{BgEIru^p^MoArIEqrIQQSYQJIRYUYJLxDmUUUTsMSTuDQLPbY@@",
        "coordinates": "!BKAb@tURDM\\YpMAMpBYMcx`BKB]~@Ha}SXW@h`Bb@IMcx}RtDvH_Xa}b@JH@ha}b@I~@Ha}"
      },
      "mass": 456.20378733435086,
      "monoisotopicMass": 456.97893790352,
      "unsaturation": null,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 9
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "Se",
        "number": 1
      }]
    }, {
      "symbol": "Agm",
      "name": "agmatidine monophosphate diradical 20C",
      "mf": "C14H26N7O6P",
      "kind": "NucleotideP",
      "oneLetter": "¿",
      "ocl": {
        "value": "fgxs@I^BuY{piqR\\`|[^DWGHeEEMeDeEHmUddhsgbuUUUSTuUUMIDSDHfP@",
        "coordinates": "!BDqc_tTnD_]\\fpH}MgrYRc}_|Dr_W_Wx@ThWM_|bOqRc}ARctu~@Gx@urH@gx@b@I~@H`BbGu~@@"
      },
      "mass": 419.373876184194,
      "monoisotopicMass": 419.16821858483,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 14
      }, {
        "symbol": "H",
        "number": 26
      }, {
        "symbol": "N",
        "number": 7
      }, {
        "symbol": "O",
        "number": 6
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Sou",
      "name": "2-selenouridine monophosphate diradical 20U",
      "mf": "C9H11N2O7PSe",
      "kind": "NucleotideP",
      "oneLetter": "ω",
      "ocl": {
        "value": "ff}Qp`I^aSbdyjCqmxNQHnNQJJJ[JIQJSMg@ejjjjfYihbIbDSH@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFDuP"
      },
      "mass": 369.1263628137839,
      "monoisotopicMass": 369.9469094988,
      "unsaturation": null,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "Se",
        "number": 1
      }]
    }, {
      "symbol": "Agu",
      "name": "5-aminomethyl-2-geranylthiouridine monophosphate diradical 21510U",
      "mf": "C20H30N3O7PS",
      "kind": "NucleotideP",
      "oneLetter": "Δ",
      "ocl": {
        "value": "ed\\\\NB@IOIhJNEDla`OFsp\\BHgGHeEEMeDheHdjdcEdhqpEUUUUURsUSMTuQBDpaBXdDt@",
        "coordinates": "!BDr__cdo[_X`fgx}RgqeRtM]}Dqa~O}\\BTmBH_]]}uwuRtMAMcuI~O}\\BupJH_]]}_`A~Oxa}uwu~Oxa}_cW_Xa}"
      },
      "mass": 487.5074340654907,
      "monoisotopicMass": 487.15420849000003,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 20
      }, {
        "symbol": "H",
        "number": 30
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Mgu",
      "name": "5-methylaminomethyl-2-geranylthiouridine monophosphate diradical 21511U",
      "mf": "C21H32N3O7PS",
      "kind": "NucleotideP",
      "oneLetter": "h",
      "ocl": {
        "value": "elR\\NB@IOIhJNEDla`OFsp\\BHgGHeEEMeDheHdjdlileFN@jjjjjjVZjYjijbDIaBDqHIh",
        "coordinates": "!BTv^cbn{__@fw|}RwqeRdK]}Tva~_{_|TiCp_[]}mwuRdIAMsuI~_{]|mwsp_[]}mwu~_{_||Gvw_Wy|Gu~_{]}|Gt"
      },
      "mass": 501.5340514703375,
      "monoisotopicMass": 501.16985855446006,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 21
      }, {
        "symbol": "H",
        "number": 32
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Cgu",
      "name": "5-carboxymethylaminomethyl-2-geranylthiouridine monophosphate diradical 2151U",
      "mf": "C22H32N3O9PS",
      "kind": "NucleotideP",
      "oneLetter": "f",
      "ocl": {
        "value": "ef^\\IB@IOIhJNEDla`XPOFsp\\BHgGHeEEMeDheHdjdlhehbhqpEUUUUURsUSMUMMTPaLHPfIAM@",
        "coordinates": "!BTv^cbn{_@fw|}RwqeRdK]}Tva~_{]|TiCp[_}muRdIAMsuI~_{]|mwsp_[]}mwu~_{]||Gvw_[_}_g}~_{]||Ou~_{]}|Gt"
      },
      "mass": 545.5435972157093,
      "monoisotopicMass": 545.1596877935999,
      "unsaturation": 16,
      "elements": [{
        "symbol": "C",
        "number": 22
      }, {
        "symbol": "H",
        "number": 32
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 9
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Mha",
      "name": "2-methylthio-N6-(cis-hydroxyisopentenyl) adenosine monophosphate diradical 2160A",
      "mf": "C16H22N5O7PS",
      "kind": "NucleotideP",
      "oneLetter": "≠",
      "ocl": {
        "value": "e`TZNB@IG@nahJNEDlo`OFspb\\V`cHeEEMeDheHdxeleDqqxEUuUUUU@sAETuTDHSBDIbP[P",
        "coordinates": "!BzfC@IeKPaDn}bHCQb@KQwuRDFALYpHCQt]WHc|TmCQw}~N`ME~@Gx@upJH@h`B_`BH_X`BbGvHGxbGt"
      },
      "mass": 459.41437086899504,
      "monoisotopicMass": 459.09775624102,
      "unsaturation": 16,
      "elements": [{
        "symbol": "C",
        "number": 16
      }, {
        "symbol": "H",
        "number": 22
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Mpa",
      "name": "2-methylthio-N6-isopentenyladenosine monophosphate diradical 2161A",
      "mf": "C16H22N5O6PS",
      "kind": "NucleotideP",
      "oneLetter": "*",
      "ocl": {
        "value": "eohZFB@IG@nahJNEDl`OFspb\\V`cHeEEMeDheHdxeleFNO@jnjjjjhFXHjfjBDIaBDq@@",
        "coordinates": "!BpBYTvxBNFY|bEJObGvOS\\@Yt]~DUEJOctu~@Ha}`HzOSTwPTh~H@h`B_`BH_Xa}bOrH@ha}b@I~@Ha}"
      },
      "mass": 443.4149659446768,
      "monoisotopicMass": 443.10284162145,
      "unsaturation": 16,
      "elements": [{
        "symbol": "C",
        "number": 16
      }, {
        "symbol": "H",
        "number": 22
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 6
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Mca",
      "name": "2-methylthio-N6-threonylcarbamoyladenosine monophosphate diradical 2162A",
      "mf": "C16H21N6O10PS",
      "kind": "NucleotideP",
      "oneLetter": "[",
      "ocl": {
        "value": "ebVVEB@IG@nachJNEDlm`XTPOFspb\\V`cHeEEMeDheHdxemLhhhqqxEUuUUUU@sAESUMUABDpaBX`@",
        "coordinates": "!BzfC@IeKPaDn}bHCQb@KQwuRDFALYpHCQt]W@h`BTmCQw}~N`ME~@Gx@upJH@h`B_`BH_Wxb@JH_WxbOrHo]^}_`BH_P"
      },
      "mass": 520.4113480993399,
      "monoisotopicMass": 520.07774907193,
      "unsaturation": 18,
      "elements": [{
        "symbol": "C",
        "number": 16
      }, {
        "symbol": "H",
        "number": 21
      }, {
        "symbol": "N",
        "number": 6
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Mva",
      "name": "2-methylthio-N6-hydroxynorvalylcarbamoyladenosine monophosphate diradical 2163A",
      "mf": "C17H23N6O10PS",
      "kind": "NucleotideP",
      "oneLetter": "≈",
      "ocl": {
        "value": "ej^VEB@IG@nachJNEDlm`XTPOFspb\\V`cHeEEMeDheHdxemLhhiVNO@jnjjjjhFXHjZijjBDIaBDq@@",
        "coordinates": "!BpBYTvxBNFY|BbEJObGvOS\\@Yt]~DUEJOctu~@Ha}`HzOSTwPTh~H@h`B_`BH_Xa}bOrH@gx@bGvHGx@bGwW@h`B_c~H@ha}"
      },
      "mass": 534.4379655041866,
      "monoisotopicMass": 534.09339913639,
      "unsaturation": 18,
      "elements": [{
        "symbol": "C",
        "number": 17
      }, {
        "symbol": "H",
        "number": 23
      }, {
        "symbol": "N",
        "number": 6
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Mya",
      "name": "2-methylthio cyclic N6-threonylcarbamoyladenosine monophosphate diradical 2164A",
      "mf": "C17H20N5O9PS",
      "kind": "NucleotideP",
      "oneLetter": "ÿ",
      "ocl": {
        "value": "elVZIB@IG@nkhJNEDlcghPOFspb\\V`cHeEEMeDheHdxeihiUFNO@jnkojjjjhFXHjfZjbHPfDHSD@@",
        "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@J_IorHbGtgD}F@RxRH_WwW@hbOTh}RIqOQ`MF@cuKW@hQTcttfpL@YS]@BbGvH@Gx"
      },
      "mass": 501.4080351062552,
      "monoisotopicMass": 501.07193541570007,
      "unsaturation": 20,
      "elements": [{
        "symbol": "C",
        "number": 17
      }, {
        "symbol": "H",
        "number": 20
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 9
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Hta",
      "name": "hydroxy-N6-threonylcarbamoyladenosine monophosphate diradical 2165A",
      "mf": "C15H19N6O11P",
      "kind": "NucleotideP",
      "oneLetter": "«",
      "ocl": {
        "value": "elZVML@IG@fnehJNEDligo`TPOFspb\\\\bTTTvTRbTbSVTrbbeXx|BjZjjjj`Y`JZijjBDIaBDq@@",
        "coordinates": "!BpBYTvxBNFY|bEJObGvOS\\@Yt]~DUEJOctu~@Ha}`HzOSTwPTh~HH`BbGvH_Xc|_`BH_Xc|_`BH_]_|bOq~Oxc|bGt"
      },
      "mass": 490.31934821268436,
      "monoisotopicMass": 490.08494245264,
      "unsaturation": 18,
      "elements": [{
        "symbol": "C",
        "number": 15
      }, {
        "symbol": "H",
        "number": 19
      }, {
        "symbol": "N",
        "number": 6
      }, {
        "symbol": "O",
        "number": 11
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Lyd",
      "name": "2-lysidine monophosphate diradical 21C",
      "mf": "C15H24N5O8P",
      "kind": "NucleotideP",
      "oneLetter": "}",
      "ocl": {
        "value": "eo`ZAL@IGOFmhJNEDlkg`OFspb\\\\bTTTvTRbTbSVRTtXxBJjjjjfYjZjfhaBXPaLP@",
        "coordinates": "!BTh|SI~ioOwy`iR\\SiV|SFGxw}FH_]]}Dqa~Oxc|_c|SFA`lIqOW_Xa}_c~HHa}bOrH_WxbOq~@Ha}"
      },
      "mass": 433.3541339985626,
      "monoisotopicMass": 433.13624975064994,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 15
      }, {
        "symbol": "H",
        "number": 24
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Gtu",
      "name": "2-geranylthiouridine monophosphate diradical 21U",
      "mf": "C19H27N2O7PS",
      "kind": "NucleotideP",
      "oneLetter": "Γ",
      "ocl": {
        "value": "e`XTNB@IOHJNEDln`OFsp\\BHgGHeEEMeDheHdtmEdhqpEUUUUURsUKUMTPaLHPfIAu@",
        "coordinates": "!BTv^cbn{__@fw|}RwqeRdK]}Tva~_{_|TiCp_[]}mwuRdIAMsuI~_{]||Gvw_Wy|Gvw_Wy|Gu~_{]}|Gt"
      },
      "mass": 458.46617269514235,
      "monoisotopicMass": 458.12765938888003,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 19
      }, {
        "symbol": "H",
        "number": 27
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Tyg",
      "name": "N2,N2,7-trimethylguanosine cap monophosphate diradical (cap TMG) 2279553N",
      "mf": "C13H20N5O10P2",
      "kind": "NucleotideP",
      "oneLetter": "¶",
      "ocl": {
        "value": "e`TZEBHIG@aihJNEHdleck`OFspz|MgDJTef[vVVe_gifNO@jijjjjjUijifjhaBXPaLP@",
        "coordinates": "!BvuPfpDnDtEK_t_rHtXBH_TwPbOr_I`JHbGtgD}F@RxS|uxc|_]^OTh}RIlA~@B\\StXCQ`Gx@Owx@_h{_cuH"
      },
      "mass": 468.2734710359255,
      "monoisotopicMass": 468.06854085929,
      "unsaturation": 13,
      "elements": [{
        "symbol": "C",
        "number": 13
      }, {
        "symbol": "H",
        "number": 20
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 2
      }]
    }, {
      "symbol": "Tmg",
      "name": "N2,N2,7-trimethylguanosine monophosphate diradical 227G",
      "mf": "C13H20N5O7P",
      "kind": "NucleotideP",
      "oneLetter": "∠",
      "ocl": {
        "value": "fcoisBINCt\\J\\TgLp^MoBKbFY}dRbbfrbTRdRUbtYspZcjjjjiYfjjjHbXaDr@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}Rqd@gD}D@tPBNOt}R"
      },
      "mass": 389.30149426455074,
      "monoisotopicMass": 389.11003500216,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 13
      }, {
        "symbol": "H",
        "number": 20
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Dgu",
      "name": "N2,N2-dimethylguanosine monophosphate diradical 22G",
      "mf": "C12H16N5O7P",
      "kind": "NucleotideP",
      "oneLetter": "R",
      "ocl": {
        "value": "fmwis@INBwlJ\\TgHp^MoBKcdRbbfrbTRdR\\RcN^CWmUUUUKLuMUDQLPbY@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RIqOQ@MD@cc}OT`"
      },
      "mass": 373.2589953515923,
      "monoisotopicMass": 373.07873487324,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Atu",
      "name": "5-aminomethyl-2-thiouridine monophosphate diradical 2510U",
      "mf": "C10H14N3O7PS",
      "kind": "NucleotideP",
      "oneLetter": "∫",
      "ocl": {
        "value": "fasqp`I^{BgEIrtGc[p\\DQ\\\\bTTTvTRbTfUSNAKUUUULsTuDQLPbY@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@@tP"
      },
      "mass": 351.27302303324575,
      "monoisotopicMass": 351.02900797432005,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Mou",
      "name": "5-methylaminomethyl-2-thiouridine monophosphate diradical 2511U",
      "mf": "C11H16N3O7PS",
      "kind": "NucleotideP",
      "oneLetter": "S",
      "ocl": {
        "value": "fikqp`I^{BgEIrtGc[p\\DQ\\\\bTTTvTRbTfUVYpIZjjjifZfjHbXaDr@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSUAMTBpX"
      },
      "mass": 365.2996404380926,
      "monoisotopicMass": 365.04465803878,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Ctu",
      "name": "5-carboxymethylaminomethyl-2-thiouridine monophosphate diradical 251U",
      "mf": "C12H16N3O9PS",
      "kind": "NucleotideP",
      "oneLetter": "$",
      "ocl": {
        "value": "fcwqH`I^{BgEIru^p^MoApQEqrIQQSYQJIRYUYJLxDmUUUTsMSTuDQLPbY@@",
        "coordinates": "!BKAb@tURDM\\YpMAMpBYMcx`BKB]~@Ha}SXW@h`Bb@IMcx}RtDvH_Xa}b@JH@ha}b@I~@Ha}"
      },
      "mass": 409.3091861834643,
      "monoisotopicMass": 409.03448727792,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 9
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Myu",
      "name": "5-methoxycarbonylmethyl-2-thiouridine monophosphate diradical 2521U",
      "mf": "C12H15N2O9PS",
      "kind": "NucleotideP",
      "oneLetter": "3",
      "ocl": {
        "value": "fmgQH`I^aSbdyZNXOFw`xHbxyDhhilheDiLjmLs`RuUUUSLuLuQDSDHfP@",
        "coordinates": "!BS]@lFJU`@Gyoza`lzf@lIwx@`H{WHc|KB_W_Wx@_`@lIr\\SFBrH@h`B_`BH_WxbOrH_P"
      },
      "mass": 394.2945422179627,
      "monoisotopicMass": 394.02358824126003,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 15
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 9
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Cou",
      "name": "5-carbamoylmethyl-2-thiouridine monophosphate diradical 253U",
      "mf": "C11H14N3O8PS",
      "kind": "NucleotideP",
      "oneLetter": "l",
      "ocl": {
        "value": "fe{pH`I^gBgEIrtXOFw`xHbxyDhhilheDiLjmF\\BVjjjjYfifhbIbDSH@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSUAMTC~NKA`"
      },
      "mass": 379.2831638542993,
      "monoisotopicMass": 379.02392259389,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Cau",
      "name": "5-carboxymethyl-2-thiouridine monophosphate diradical 2540U",
      "mf": "C11H13N2O9PS",
      "kind": "NucleotideP",
      "oneLetter": "℘",
      "ocl": {
        "value": "fe{QH`I^aSbdyZNXOFw`xHbxyDhhilheDiLjmF\\BVjjjjYfifhbIbDSH@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSUAMTC~NKA`"
      },
      "mass": 380.26792481311594,
      "monoisotopicMass": 380.00793817680005,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 9
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Tau",
      "name": "5-taurinomethyl-2-thiouridine monophosphate diradical 254U",
      "mf": "C12H18N3O10PS2",
      "kind": "NucleotideP",
      "oneLetter": "∃",
      "ocl": {
        "value": "fgpj`I^{BgEIrwY{`|[^C`bKblHrIQQSYQJIRYUYIRLxDmUUUTsMSUKTQDqBId@@",
        "coordinates": "!BKAb@tURD@m\\YpMAMpBYMcx`BKB]~@Ha}SXW@h`Bb@IMcx}RtDvH_Xa}b@JH@ha}b@JH__rH_]^H_P"
      },
      "mass": 459.3892600220213,
      "monoisotopicMass": 459.01712313635005,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 18
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 2
      }]
    }, {
      "symbol": "Itu",
      "name": "5-(isopentenylaminomethyl)-2-thiouridine monophosphate diradical 2583U",
      "mf": "C15H22N3O7PS",
      "kind": "NucleotideP",
      "oneLetter": "½",
      "ocl": {
        "value": "fkoqp`I^{BgEIrtGc[p\\DQ\\\\bTTTvTRbTfUVRTYpIZjjjifZfijbHfHQL`@",
        "coordinates": "!BS]@lFJU`@Gyoza`lzf@lIwx@`H{W@h`BKB_W_Wx@_`@lIr\\SFBrH@h`B_`BH_Xc|bGvH@gx@bGt"
      },
      "mass": 419.3902285493682,
      "monoisotopicMass": 419.09160823216,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 15
      }, {
        "symbol": "H",
        "number": 22
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Mth",
      "name": "5-methyl-2-thiouridine monophosphate diradical 25U",
      "mf": "C10H13N2O7PS",
      "kind": "NucleotideP",
      "oneLetter": "F",
      "ocl": {
        "value": "fncQp`I^aSbdyZCqmxNBHnNQJJJ[JIQJSJlxDmUUUTsMSQDSDHfP@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@"
      },
      "mass": 336.25837906774416,
      "monoisotopicMass": 336.01810893766003,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Dmg",
      "name": "N2,7-dimethylguanosine monophosphate diradical 27G",
      "mf": "C12H18N5O7P",
      "kind": "NucleotideP",
      "oneLetter": "∨",
      "ocl": {
        "value": "fmwisBINCt\\J\\TgLp^MoBKbFY}dRbbfrbTRdRUbKN^CWmUUUUKLuUUDQLPbY@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RqdCQ@B\\StPAOT`"
      },
      "mass": 375.27487685970397,
      "monoisotopicMass": 375.0943849377,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 18
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Dma",
      "name": "2,8-dimethyladenosine monophosphate diradical 28A",
      "mf": "C12H16N5O6P",
      "kind": "NucleotideP",
      "oneLetter": "±",
      "ocl": {
        "value": "feghs@INCv\\J\\UdhOFw`eqrIQQSYQJJJQKqLyxK^uUUUPMLAUADSDHfP@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpP@c`a}_S_|BD}RSuKQ@B\\StP@"
      },
      "mass": 357.2595904272741,
      "monoisotopicMass": 357.08382025367,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 6
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Mas",
      "name": "2-methyladenosine monophosphate diradical 2A",
      "mf": "C11H14N5O6P",
      "kind": "NucleotideP",
      "oneLetter": "/",
      "ocl": {
        "value": "fi{hs@INBt\\J\\TgHOFwaEqrIQQSYQJIRINIgOAjvjjjjAf@j`bIbDSH@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtP@gD}D@"
      },
      "mass": 343.2329730224273,
      "monoisotopicMass": 343.06817018921,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 6
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Tcy",
      "name": "2-thiocytidine monophosphate diradical 2C",
      "mf": "C9H12N3O6PS",
      "kind": "NucleotideP",
      "oneLetter": "%",
      "ocl": {
        "value": "ff}pp`I^kBgEIrCqmxNBHnNQJJJ[JIQJSMg@ejjjjfYfhbIbDSH@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFDuP"
      },
      "mass": 321.2470007040807,
      "monoisotopicMass": 321.01844329029,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 6
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Nmg",
      "name": "N2-methylguanosine monophosphate diradical 2G",
      "mf": "C11H14N5O7P",
      "kind": "NucleotideP",
      "oneLetter": "L",
      "ocl": {
        "value": "fegis@INBwlJ\\TgHp^MoBKcdRbbfrbTRdR\\VYspZmjjjjiYfijbHfHQL`@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtP@gD}D@SuH"
      },
      "mass": 359.23237794674554,
      "monoisotopicMass": 359.06308480878,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Thu",
      "name": "2-thiouridine monophosphate diradical 2U",
      "mf": "C9H11N2O7PS",
      "kind": "NucleotideP",
      "oneLetter": "2",
      "ocl": {
        "value": "ff}Qp`I^aSbdyjCqmxNBHnNQJJJ[JIQJSMg@ejjjjfYihbIbDSH@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFDuP"
      },
      "mass": 322.2317616628973,
      "monoisotopicMass": 322.0024588732,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Ahu",
      "name": "3-(3-amino-3-carboxypropyl)-5,6-dihydrouridine monophosphate diradical 308U",
      "mf": "C13H20N3O10P",
      "kind": "NucleotideP",
      "oneLetter": "Ð",
      "ocl": {
        "value": "fkopk@I^gBgENSens`|[^DWGHeEEMeDheIhueMF\\BVjjjjZfijfhbIbDSH@",
        "coordinates": "!BTh|SI~ioOwy`iR\\SiV|SFGxw}FH_]]}DqbH@gx_c|SFA`lIqOW_Xa}_c~HHa}_c~H@gx@bGt"
      },
      "mass": 409.28630261461393,
      "monoisotopicMass": 409.08863085201,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 13
      }, {
        "symbol": "H",
        "number": 20
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "App",
      "name": "3-(3-amino-3-carboxypropyl)pseudouridine monophosphate diradical 309U",
      "mf": "C13H18N3O10P",
      "kind": "NucleotideP",
      "oneLetter": "Þ",
      "ocl": {
        "value": "fkopk@OAgBgENSens`|[^DWGHeEEMeDheIhueMF\\BVjjjfZfijfhbIbDSH@",
        "coordinates": "!BTh|SI~ioOwy`iR\\SiV|SFGxw}FH_]]}DqbH@gx_c|SFA`lIqOW_Xa}_c~HHa}_c~H@gx@bGt"
      },
      "mass": 407.2704211065024,
      "monoisotopicMass": 407.07298078755,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 13
      }, {
        "symbol": "H",
        "number": 18
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Apu",
      "name": "3-(3-amino-3-carboxypropyl)uridine monophosphate diradical 30U",
      "mf": "C13H18N3O10P",
      "kind": "NucleotideP",
      "oneLetter": "X",
      "ocl": {
        "value": "fkopk@I^gBgENSens`|[^DWGHeEEMeDheIhueMF\\BVjjjjYfijfhbIbDSH@",
        "coordinates": "!BTh|SI~ioOwy`iR\\SiV|SFGxw}FH_]]}DqbH@gx_c|SFA`lIqOW_Xa}_c~HHa}_c~H@gx@bGt"
      },
      "mass": 407.2704211065024,
      "monoisotopicMass": 407.07298078755,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 13
      }, {
        "symbol": "H",
        "number": 18
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Mws",
      "name": "methylwyosine monophosphate diradical 342G",
      "mf": "C15H18N5O7P",
      "kind": "NucleotideP",
      "oneLetter": "∑",
      "ocl": {
        "value": "eghZNL@IG@nahJNEDli`OFspb\\\\bTTTvTRbTbb\\rVSGG`SPrvuUUUUKMTsUUIBDpaBX`@",
        "coordinates": "!B_`CW@mF@ctvDUI|fRxPYgtwP[zV_IorHFY|gD}F@RxPYg|@YgrZOTh{_cuJOS]F@tXAKaI|fw}EMt@"
      },
      "mass": 411.3070845499097,
      "monoisotopicMass": 411.0943849377,
      "unsaturation": 18,
      "elements": [{
        "symbol": "C",
        "number": 15
      }, {
        "symbol": "H",
        "number": 18
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Uhw",
      "name": "undermodified hydroxywybutosine monophosphate diradical 3470G",
      "mf": "C18H23N6O10P",
      "kind": "NucleotideP",
      "oneLetter": "š",
      "ocl": {
        "value": "ejQVEL@IG@nahXJNEDliolRPOFspb\\\\bTTTvTRbTbb\\rVVTttXx|BZFVvjjjjiYjfZjjfjRDIaBDq@@",
        "coordinates": "!BKB^@ceS[H`Yg}ARpAeMtHa}KAcPTh{_S]CjXES[pAeMtH}MtEK@IdnDpBXBbES[UMo@F]ARaERH_X`B_`BH_WxbOq~@Ha}"
      },
      "mass": 514.3839139947949,
      "monoisotopicMass": 514.12132796199,
      "unsaturation": 20,
      "elements": [{
        "symbol": "C",
        "number": 18
      }, {
        "symbol": "H",
        "number": 23
      }, {
        "symbol": "N",
        "number": 6
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Acw",
      "name": "7-aminocarboxypropylwyosine monophosphate diradical 347G",
      "mf": "C18H23N6O9P",
      "kind": "NucleotideP",
      "oneLetter": "Ω",
      "ocl": {
        "value": "eb^VIL@IG@na`XJNEDlid\\POFspb\\\\bTTTvTRbTbb\\rVRrfcGG`SPrvuUUUUKMTsUUSUIBDpaBX`@",
        "coordinates": "!BDr]RcwwWpAg_tUS[cm~DUAf_XJUTvx}MaEP@_gwWcm~DUDnDUMo|urH@m_@FWwW_]^NwuS[bGtYgx`BbGu~Ox`B_`BH_P"
      },
      "mass": 498.38450907047655,
      "monoisotopicMass": 498.12641334242,
      "unsaturation": 20,
      "elements": [{
        "symbol": "C",
        "number": 18
      }, {
        "symbol": "H",
        "number": 23
      }, {
        "symbol": "N",
        "number": 6
      }, {
        "symbol": "O",
        "number": 9
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Hwy",
      "name": "methylated undermodified hydroxywybutosine monophosphate diradical 3480G",
      "mf": "C19H25N6O10P",
      "kind": "NucleotideP",
      "oneLetter": "y",
      "ocl": {
        "value": "efYVEL@IG@nahXJNEDliolRPOFspb\\\\bTTTvTRbTbb\\rVVTttsGG`SPrvuUUUUKMTsUUTuTdHSBDIb@@",
        "coordinates": "!B`HyRtL@f_XbDRxz@UHS_chc|S]BN`MAMwxyKaL@fUHS_cmG_chCjXI|YzfA}bL@fpBYTaHz@F\\BH@gx@upJH@ha}_`CWHc|_`@"
      },
      "mass": 528.4105313996416,
      "monoisotopicMass": 528.1369780264499,
      "unsaturation": 20,
      "elements": [{
        "symbol": "C",
        "number": 19
      }, {
        "symbol": "H",
        "number": 25
      }, {
        "symbol": "N",
        "number": 6
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Hwb",
      "name": "hydroxywybutosine monophosphate diradical 34830G",
      "mf": "C21H27N6O12P",
      "kind": "NucleotideP",
      "oneLetter": "⊆",
      "ocl": {
        "value": "ee[VCL@IG@nahXJNEDliobZV^POFspb\\\\bTTTvTRbTbb\\rVVTtRbfsGG`SPrvuUUUUKMTsUUULuUIBDpaBX`@",
        "coordinates": "!BKB^@ceS[@h`Yg}ARpAeMtHa}KAcPTh{_S]CjXES[pAeMtH}MtEK@IdnDpBXBbES[UMo@F]ARaERH_X`B_`BH_X`B_c~H_]]}bGu~Ox`B_c~H_P"
      },
      "mass": 586.4466945498602,
      "monoisotopicMass": 586.14245733005,
      "unsaturation": 22,
      "elements": [{
        "symbol": "C",
        "number": 21
      }, {
        "symbol": "H",
        "number": 27
      }, {
        "symbol": "N",
        "number": 6
      }, {
        "symbol": "O",
        "number": 12
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Pwb",
      "name": "peroxywybutosine monophosphate diradical 34832G",
      "mf": "C21H27N6O13P",
      "kind": "NucleotideP",
      "oneLetter": "W",
      "ocl": {
        "value": "emWVKL@IG@nadXJNEDliohZV^QPOFspb\\\\bTTTvTRbTbb\\rVVTRfTTvXx|BZFVvjjjjiYjfZjjjYjjRDIaBDq@@",
        "coordinates": "!BKB^@ceS[@h`Yg}ARpAeMtHa}KAcPTh{_S]CjXES[pAeMtH}MtEK@IdnDpB[|bES[UMo@F]ARaERH_X`B_`BH_X`Bb@I~Oxa}uwvH_Wxb@I~Oxa}"
      },
      "mass": 602.4460994741785,
      "monoisotopicMass": 602.1373719496199,
      "unsaturation": 22,
      "elements": [{
        "symbol": "C",
        "number": 21
      }, {
        "symbol": "H",
        "number": 27
      }, {
        "symbol": "N",
        "number": 6
      }, {
        "symbol": "O",
        "number": 13
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Wyb",
      "name": "wybutosine monophosphate diradical 3483G",
      "mf": "C21H27N6O11P",
      "kind": "NucleotideP",
      "oneLetter": "Y",
      "ocl": {
        "value": "eiSVML@IG@na`XJNEDlilRZVPOFspb\\\\bTTTvTRbTbb\\rVVRbTTvXx|BZFVvjjjjiYjfZjjfZjdaBXPaLP@",
        "coordinates": "!BsJ\\@ciP{@`YWuARPAeMT@a}sNaPThxSUCjhIP{PAeMTD}MTEI@IllDPB[|BIP{eCm@FUARAIPH_Pc|BGtHGzBGtw_Pa}_k|HGzBGt"
      },
      "mass": 570.4472896255419,
      "monoisotopicMass": 570.14754271048,
      "unsaturation": 22,
      "elements": [{
        "symbol": "C",
        "number": 21
      }, {
        "symbol": "H",
        "number": 27
      }, {
        "symbol": "N",
        "number": 6
      }, {
        "symbol": "O",
        "number": 11
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Awo",
      "name": "7-aminocarboxypropylwyosine methyl ester monophosphate diradical 348G",
      "mf": "C19H25N6O9P",
      "kind": "NucleotideP",
      "oneLetter": "⇑",
      "ocl": {
        "value": "ejQVIL@IG@na`XJNEDlid\\POFspb\\\\bTTTvTRbTbb\\rVVRffXx|BZFVvjjjjiYjfZjjZjRDIaBDq@@",
        "coordinates": "!B`HyRtL@f_XbDRxz@UHS_ch`BS]BN`MAMwxyKaL@fUHS_cmG_chCjXI|YzfA}bL@fpBYTaHz@F\\BHHa}bOq~@Ha}_c~H@ha}"
      },
      "mass": 512.4111264753234,
      "monoisotopicMass": 512.14206340688,
      "unsaturation": 20,
      "elements": [{
        "symbol": "C",
        "number": 19
      }, {
        "symbol": "H",
        "number": 25
      }, {
        "symbol": "N",
        "number": 6
      }, {
        "symbol": "O",
        "number": 9
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Wyo",
      "name": "wyosine monophosphate diradical 34G",
      "mf": "C14H16N5O7P",
      "kind": "NucleotideP",
      "oneLetter": "€",
      "ocl": {
        "value": "ek`ZNL@IG@nahJNEDli`OFspb\\\\bTTTvTRbTbb\\rVXx|BZFVvjjjjiYjfZjdaBXPaLP@",
        "coordinates": "!B_`CWMF@ctvDUI|fRxPYgtwP[zV_IorHFY|gD}F@RxPYg|@YgrZOTh{_cuJOS]F@tXAKaI|fw}D"
      },
      "mass": 397.2804671450629,
      "monoisotopicMass": 397.07873487324,
      "unsaturation": 18,
      "elements": [{
        "symbol": "C",
        "number": 14
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Pdu",
      "name": "3-methylpseudouridine monophosphate diradical 39U",
      "mf": "C10H13N2O8P",
      "kind": "NucleotideP",
      "oneLetter": "κ",
      "ocl": {
        "value": "fncPK@OAaSbgIrtGc[pbxyDhhilheDiMFs`RuUUTsTuMDQLPbY@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@HoWtw@`lFC~NSU@"
      },
      "mass": 320.1929965859354,
      "monoisotopicMass": 320.04095238282997,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Mri",
      "name": "3-methyluridine monophosphate diradical 3U",
      "mf": "C10H13N2O8P",
      "kind": "NucleotideP",
      "oneLetter": "δ",
      "ocl": {
        "value": "fncPK@I^aSbgIrtGc[pbxyDhhilheDiMFs`RuUUUSLuMDQLPbY@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@HoWtw@`lFC~NSU@"
      },
      "mass": 320.1929965859354,
      "monoisotopicMass": 320.04095238282997,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Acc",
      "name": "N4-acetylcytidine monophosphate diradical 42C",
      "mf": "C11H14N3O8P",
      "kind": "NucleotideP",
      "oneLetter": "M",
      "ocl": {
        "value": "fikpK@I^kBgENSghOFwaEqrIQQSYQJIRYiQg@ejjjjfYffhbIbDSH@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFDuPOxxSItuP"
      },
      "mass": 347.21837644817225,
      "monoisotopicMass": 347.05185141949,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Iws",
      "name": "isowyosine monophosphate diradical 42G",
      "mf": "C14H16N5O7P",
      "kind": "NucleotideP",
      "oneLetter": "⊇",
      "ocl": {
        "value": "ek`ZNL@IG@fnhJNEDla`OFspb\\\\bTTTvTRbTbSbRrXx|BjzfVjjjjiYjYjjdaBXPaLP@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RStwQ@MD@RpQ_qcQSU@"
      },
      "mass": 397.2804671450629,
      "monoisotopicMass": 397.07873487324,
      "unsaturation": 18,
      "elements": [{
        "symbol": "C",
        "number": 14
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Dmc",
      "name": "N4,N4-dimethylcytidine monophosphate diradical 44C",
      "mf": "C11H16N3O7P",
      "kind": "NucleotideP",
      "oneLetter": "μ",
      "ocl": {
        "value": "fasqs@I^kBgENSdGc[pbxyDhhilheDiLuF\\BVjjjjYfZjHbXaDr@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBGpcbpXBGtSItuPSU@H_Wtw@`lFDuPOxxlF@"
      },
      "mass": 333.23485303196554,
      "monoisotopicMass": 333.07258686438,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Adw",
      "name": "7-aminocarboxypropyl-demethylwyosine monophosphate diradical 47G",
      "mf": "C17H21N6O9P",
      "kind": "NucleotideP",
      "oneLetter": "¥",
      "ocl": {
        "value": "elVVIL@IG@fnohJNEDlahTPOFspb\\\\bTTTvTRbTbSbRrrTtXx|BjzfVjjjjiYjYjjijdaBXPaLP@",
        "coordinates": "!B`MERc|@Y_]^DUH{_UMo_tXa}SXPTh{_w}GjXES[pAg_t]F@cm@Il@f@haTvuS[pAgPThQTbGvH@ha}_c~HGx@bGt"
      },
      "mass": 484.3578916656298,
      "monoisotopicMass": 484.1107632779601,
      "unsaturation": 20,
      "elements": [{
        "symbol": "C",
        "number": 17
      }, {
        "symbol": "H",
        "number": 21
      }, {
        "symbol": "N",
        "number": 6
      }, {
        "symbol": "O",
        "number": 9
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Ncd",
      "name": "N4-methylcytidine monophosphate diradical 4C",
      "mf": "C10H14N3O7P",
      "kind": "NucleotideP",
      "oneLetter": "ν",
      "ocl": {
        "value": "fncqs@I^kBgENSdGc[pbxyDhhilheDiLts`RuUUUSLsUDQLPbY@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFDuPKA`"
      },
      "mass": 319.2082356271187,
      "monoisotopicMass": 319.05693679992004,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Dmw",
      "name": "4-demethylwyosine monophosphate diradical 4G",
      "mf": "C13H14N5O7P",
      "kind": "NucleotideP",
      "oneLetter": "†",
      "ocl": {
        "value": "fcis@INBwlJ\\TgHp^MoBKcdRbbfrbTRdR\\RVYspZ}fnjjjjefifjiHbXaDr@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RStwQ@MD@RpQ_qcQ"
      },
      "mass": 383.253849740216,
      "monoisotopicMass": 383.06308480878,
      "unsaturation": 18,
      "elements": [{
        "symbol": "C",
        "number": 13
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Mtu",
      "name": "5-methoxyuridine monophosphate diradical 501U",
      "mf": "C10H13N2O9P",
      "kind": "NucleotideP",
      "oneLetter": "5",
      "ocl": {
        "value": "fasQK@I^aSbgIsUhOFwaEqrIQQSYQJIRYULxDmUUUTsMSTQDqBId@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@@tP"
      },
      "mass": 336.19240151025366,
      "monoisotopicMass": 336.03586700240004,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 9
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Uoa",
      "name": "uridine 5-oxyacetic acid monophosphate diradical 502U",
      "mf": "C11H13N2O11P",
      "kind": "NucleotideP",
      "oneLetter": "V",
      "ocl": {
        "value": "fmgQk@I^aSbgIsUlu`|[^DWGHeEEMeDheIeUeF\\BVjjjjYfiijHbXaDr@@",
        "coordinates": "!BS]@lFJU`@Gyoza`lzf@lIwx@`H{WHc|KB_W_Wx@_`@lIr\\SFBrHHc|_`BH_Xc|_`BH_P"
      },
      "mass": 380.2019472556255,
      "monoisotopicMass": 380.02569624154,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 11
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Ume",
      "name": "uridine 5-oxyacetic acid methyl ester monophosphate diradical 503U",
      "mf": "C12H15N2O11P",
      "kind": "NucleotideP",
      "oneLetter": "υ",
      "ocl": {
        "value": "fcwQk@I^aSbgIsUlu`|[^DWGHeEEMeDheIeUeLs`RuUUUSLuMMTQDqBId@@",
        "coordinates": "!BKAb@tURDM\\YpMAMpBYMcxc|KB]~@Ha}SXWHc|bOqMcx}RtDvH_Xa}bOrH@ha}_c~HHa}"
      },
      "mass": 394.2285646604723,
      "monoisotopicMass": 394.041346306,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 15
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 11
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Hxc",
      "name": "5-hydroxycytidine monophosphate diradical 50C",
      "mf": "C9H12N3O8P",
      "kind": "NucleotideP",
      "oneLetter": "Ç",
      "ocl": {
        "value": "fncpK@I^[BgENSfhOFwaEqrIQQSYQJIRYUg@ejjjjfYfjHbXaDr@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@"
      },
      "mass": 321.1810231465902,
      "monoisotopicMass": 321.03620135502996,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Hxu",
      "name": "5-hydroxyuridine monophosphate diradical 50U",
      "mf": "C9H11N2O9P",
      "kind": "NucleotideP",
      "oneLetter": "∝",
      "ocl": {
        "value": "fncQK@I^aSbgIsUhOFwaEqrIQQSYQJIRYUg@ejjjjfYjZHbXaDr@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@"
      },
      "mass": 322.1657841054069,
      "monoisotopicMass": 322.02021693794,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 9
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Amu",
      "name": "5-aminomethyluridine monophosphate diradical 510U",
      "mf": "C10H14N3O8P",
      "kind": "NucleotideP",
      "oneLetter": "∪",
      "ocl": {
        "value": "faspK@I^{BgENSehOFwaEqrIQQSYQJIRYULxDmUUUTsMSTQDqBId@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@@tP"
      },
      "mass": 335.207640551437,
      "monoisotopicMass": 335.05185141949,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Mau",
      "name": "5-methylaminomethyluridine monophosphate diradical 511U",
      "mf": "C11H16N3O8P",
      "kind": "NucleotideP",
      "oneLetter": "{",
      "ocl": {
        "value": "fikpK@I^{BgENSehOFwaEqrIQQSYQJIRYUYg@ejjjjfYjZhbIbDSH@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSUAMTBpX"
      },
      "mass": 349.2342579562838,
      "monoisotopicMass": 349.06750148395,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Hmc",
      "name": "5-hydroxymethylcytidine monophosphate diradical 51C",
      "mf": "C10H14N3O8P",
      "kind": "NucleotideP",
      "oneLetter": "∅",
      "ocl": {
        "value": "faspK@I^[BgENSghOFwaEqrIQQSYQJIRYULxDmUUUTsLuTQDqBId@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@@tP"
      },
      "mass": 335.207640551437,
      "monoisotopicMass": 335.05185141949,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Cur",
      "name": "5-carboxymethylaminomethyluridine monophosphate diradical 51U",
      "mf": "C12H16N3O10P",
      "kind": "NucleotideP",
      "oneLetter": "!",
      "ocl": {
        "value": "fcwpk@I^{BgENSej}`|[^DWGHeEEMeDheIeUdhs`RuUUUSLuMSTQDqBId@@",
        "coordinates": "!BKAb@tURDM\\YpMAMpBYMcx`BKB]~@Ha}SXW@h`Bb@IMcx}RtDvH_Xa}b@JH@ha}b@I~@Ha}"
      },
      "mass": 393.24380370165557,
      "monoisotopicMass": 393.05733072309,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Chr",
      "name": "5-carboxyhydroxymethyluridine monophosphate diradical 520U",
      "mf": "C11H13N2O11P",
      "kind": "NucleotideP",
      "oneLetter": "≥",
      "ocl": {
        "value": "fmgQk@I^aSbgIrwlu`|[^DWGHeEEMeDheIeUCF\\BVjjjjYfiijHbXaDr@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@@tTuPOxxlF@"
      },
      "mass": 380.2019472556255,
      "monoisotopicMass": 380.02569624154,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 11
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Mcu",
      "name": "5-methoxycarbonylmethyluridine monophosphate diradical 521U",
      "mf": "C12H15N2O10P",
      "kind": "NucleotideP",
      "oneLetter": "1",
      "ocl": {
        "value": "fmgPk@I^aSbgIrt\\p^MoBKcdRbbfrbTRdrjtsNAKUUUULsTsUDQLPbY@@",
        "coordinates": "!BS]@lFJU`@Gyoza`lzf@lIwx@`H{WHc|KB_W_Wx@_`@lIr\\SFBrH@h`B_`BH_WxbOrH_P"
      },
      "mass": 378.229159736154,
      "monoisotopicMass": 378.04643168643,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 15
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Hme",
      "name": "5-(carboxyhydroxymethyl)uridine methyl ester monophosphate diradical 522U",
      "mf": "C12H15N2O11P",
      "kind": "NucleotideP",
      "oneLetter": ",",
      "ocl": {
        "value": "fcwQk@I^aSbgIrwlu`|[^DWGHeEEMeDheIeUCLs`RuUUUSLuMMTQDqBId@@",
        "coordinates": "!BS]@lFJU`@Gyoza`lzf@lIwx@`H{WHc|KB_W_Wx@_`@lIr\\SFBrHHc|_`A~@Ha}_c~H@ha}"
      },
      "mass": 394.2285646604723,
      "monoisotopicMass": 394.041346306,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 15
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 11
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Cxu",
      "name": "5-carboxymethyluridine monophosphate diradical 52U",
      "mf": "C11H13N2O10P",
      "kind": "NucleotideP",
      "oneLetter": "◊",
      "ocl": {
        "value": "fe{Pk@I^aSbgIrt\\p^MoBKcdRbbfrbTRdrjtYpIZjjjifZfZbHfHQL`@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSUAMTC~NKA`"
      },
      "mass": 364.2025423313072,
      "monoisotopicMass": 364.03078162197,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Hmu",
      "name": "5-carbamoylhydroxymethyluridine monophosphate diradical 531U",
      "mf": "C11H14N3O10P",
      "kind": "NucleotideP",
      "oneLetter": "r",
      "ocl": {
        "value": "fmgpk@I^WBgENSeoY`|[^DWGHeEEMeDheIeUCF\\BVjjjjYfiijHbXaDr@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@@tTuPOxxlF@"
      },
      "mass": 379.21718629680873,
      "monoisotopicMass": 379.04168065863,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Ymu",
      "name": "5-carbamoylmethyluridine monophosphate diradical 53U",
      "mf": "C11H14N3O9P",
      "kind": "NucleotideP",
      "oneLetter": "&",
      "ocl": {
        "value": "fe{qK@I^gBgENSehp^MoBKcdRbbfrbTRdrjtYpIZjjjifZfZbHfHQL`@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSUAMTC~NKA`"
      },
      "mass": 363.2177813724905,
      "monoisotopicMass": 363.04676603906006,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 9
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Ttu",
      "name": "5-taurinomethyluridine monophosphate diradical 54U",
      "mf": "C12H18N3O11PS",
      "kind": "NucleotideP",
      "oneLetter": "Ê",
      "ocl": {
        "value": "fgqh`I^{BgENSenswAxv|HnJpcHeEEMeDheIeUdeHs`RuUUUSLuMTmQDSDHfP@",
        "coordinates": "!BKAb@tURD@m\\YpMAMpBYMcx`BKB]~@Ha}SXW@h`Bb@IMcx}RtDvH_Xa}b@JH@ha}b@JH__rH_]^H_P"
      },
      "mass": 443.32387754021244,
      "monoisotopicMass": 443.03996658152005,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 18
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 11
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Cmu",
      "name": "5-cyanomethyluridine monophosphate diradical 55U",
      "mf": "C11H12N3O8P",
      "kind": "NucleotideP",
      "oneLetter": "Ѷ",
      "ocl": {
        "value": "fikpK@I^GBgENSehOFwaEqrIQQSYQJIRYUYg@ejjjjfYj[hbIbDSH@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@@tPCQ"
      },
      "mass": 345.20249494006066,
      "monoisotopicMass": 345.03620135502996,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Pyu",
      "name": "5-(isopentenylaminomethyl)uridine monophosphate diradical 583U",
      "mf": "C15H22N3O8P",
      "kind": "NucleotideP",
      "oneLetter": "¾",
      "ocl": {
        "value": "fkopK@I^{BgENSehOFwaEqrIQQSYQJIRYUYIQg@ejjjjfYjZfjHbXaDr@@",
        "coordinates": "!BS]@lFJU`@Gyoza`lzf@lIwx@`H{W@h`BKB_W_Wx@_`@lIr\\SFBrH@h`B_`BH_Xc|bGvH@gx@bGt"
      },
      "mass": 403.32484606755946,
      "monoisotopicMass": 403.11445167733,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 15
      }, {
        "symbol": "H",
        "number": 22
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Mdu",
      "name": "5-methyldihydrouridine monophosphate diradical 58U",
      "mf": "C10H15N2O8P",
      "kind": "NucleotideP",
      "oneLetter": "ρ",
      "ocl": {
        "value": "fncPK@I^aSbgIrtGc[pbxyDhhilheDiLjs`RuUUUSTuMDQLPbY@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@"
      },
      "mass": 322.20887809404695,
      "monoisotopicMass": 322.05660244729,
      "unsaturation": 8,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 15
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Mcd",
      "name": "5-methylcytidine monophosphate diradical 5C",
      "mf": "C10H14N3O7P",
      "kind": "NucleotideP",
      "oneLetter": "?",
      "ocl": {
        "value": "fncqs@I^[BgENSdGc[pbxyDhhilheDiLjs`RuUUUSLsUDQLPbY@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@"
      },
      "mass": 319.2082356271187,
      "monoisotopicMass": 319.05693679992004,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Hia",
      "name": "N6-(cis-hydroxyisopentenyl)adenosine monophosphate diradical 60A",
      "mf": "C15H20N5O7P",
      "kind": "NucleotideP",
      "oneLetter": "`",
      "ocl": {
        "value": "eg`ZNL@IG@fnhJNEDlk`OFspb\\\\bTTTvTRbTbSVRTSGG`USUUUUTCLATuTDHSBDIbPSH",
        "coordinates": "!BzfC@IeKPaDn}bHCQb@KQwuRDFALYpHCQt]W@h`BTmCQw}~N`ME~@Gx@b@JH@ha}bOrH_Wxb@JH_P"
      },
      "mass": 413.3229660580212,
      "monoisotopicMass": 413.11003500216003,
      "unsaturation": 16,
      "elements": [{
        "symbol": "C",
        "number": 15
      }, {
        "symbol": "H",
        "number": 20
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Mta",
      "name": "2-methylthio-N6-methyladenosine monophosphate diradical 621A",
      "mf": "C12H16N5O6PS",
      "kind": "NucleotideP",
      "oneLetter": "∞",
      "ocl": {
        "value": "fmwhp`CQstZLDxipEfGa[qZDYEIlheDdhXdmDmKR\\u{MUUUU@aEUAFPTdmH@",
        "coordinates": "!BBGw|B@a}_S\\H@a}TEJNOuP{Ntm@fPBN[~iRSpHUCneXDBYTEITAEPDiVA@fTBYU@Sj[p"
      },
      "mass": 389.3243778334011,
      "monoisotopicMass": 389.05589142807,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 6
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Tca",
      "name": "N6-threonylcarbamoyladenosine monophosphate diradical 62A",
      "mf": "C15H19N6O10P",
      "kind": "NucleotideP",
      "oneLetter": "6",
      "ocl": {
        "value": "edRVEL@IG@fnehJNEDligo`POFspb\\\\bTTTvTRbTbSVTrbbcGG`USUUUUTCLASUMUABDpaBX`@",
        "coordinates": "!BzfC@IeKPaDn}bHCQbOsQwuRDFALYpHCQt]W@h`BTmCQw}~N`ME~@Gx@b@JH@ha}_c~H@ha}_c~H@ha}uwu~@Ha}"
      },
      "mass": 474.31994328836606,
      "monoisotopicMass": 474.09002783307,
      "unsaturation": 18,
      "elements": [{
        "symbol": "C",
        "number": 15
      }, {
        "symbol": "H",
        "number": 19
      }, {
        "symbol": "N",
        "number": 6
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Hva",
      "name": "N6-hydroxynorvalylcarbamoyladenosine monophosphate diradical 63A",
      "mf": "C16H21N6O10P",
      "kind": "NucleotideP",
      "oneLetter": "√",
      "ocl": {
        "value": "elZVIB@IG@fnehJNDligo`POEQql|HgGHeEEMeDheHdueLhhiVNO@jfjjjjhFXBfjZj`aBXPaLP@",
        "coordinates": "!BpBYTvxBNFY|bEJObGvOS\\@Yt]~DUEJOctu~@Ha}`HzOSTwPTh~HH`BbGvH_Xc|_`BH_Xc|_`BH_]_|bOq~Oxc|bGt"
      },
      "mass": 488.34656069321284,
      "monoisotopicMass": 488.10567789753003,
      "unsaturation": 18,
      "elements": [{
        "symbol": "C",
        "number": 16
      }, {
        "symbol": "H",
        "number": 21
      }, {
        "symbol": "N",
        "number": 6
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Aya",
      "name": "N6-acetyladenosine monophosphate diradical 64A",
      "mf": "C12H14N5O7P",
      "kind": "NucleotideP",
      "oneLetter": "⇓",
      "ocl": {
        "value": "fmwis@INBwlJ\\TgLp^MoBKcdRbbfrbTRdRZrcN^CUmUUUTCLASTDQLPbY@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtPCQ@D}RIqOQ@@"
      },
      "mass": 371.2431138434808,
      "monoisotopicMass": 371.06308480878,
      "unsaturation": 16,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Gca",
      "name": "N6-glycinylcarbamoyladenosine monophosphate diradical 65A",
      "mf": "C13H15N6O9P",
      "kind": "NucleotideP",
      "oneLetter": "≡",
      "ocl": {
        "value": "eohVIL@IG@fnehJNEDlikg`OFspb\\\\bTTTvTRbTbSVTrTXx|BjZjjjj`Y`JZfhHPfDHSD@@",
        "coordinates": "!BzfC@IeKPaDn}bHCQb@KQwuRDFALYpHCQt]W@h`BTmCQw}~N`ME~@Gx@bOrHHa}_c~H@ha}bOq~@Ha}"
      },
      "mass": 430.2673035543541,
      "monoisotopicMass": 430.06381308458,
      "unsaturation": 18,
      "elements": [{
        "symbol": "C",
        "number": 13
      }, {
        "symbol": "H",
        "number": 15
      }, {
        "symbol": "N",
        "number": 6
      }, {
        "symbol": "O",
        "number": 9
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Tya",
      "name": "N6-methyl-N6-threonylcarbamoyladenosinemonophosphate diradical 662A",
      "mf": "C16H21N6O10P",
      "kind": "NucleotideP",
      "oneLetter": "E",
      "ocl": {
        "value": "elZVEL@IG@fnmhJNEDleo`XPOFspb\\\\bTTTvTRbTbSVbaTTTXx|BjZjjjj`Y`JfjZjBDIaBDq@@",
        "coordinates": "!BzfC@IeKPaDn}bHCQb@KQwuRDFALYpHCQt]W@h`BTmCQw}~N`ME~@Gx@bOrHHa}_`A~Ox`BbGu~Ox`BbGwW_Wx@bGt"
      },
      "mass": 488.34656069321284,
      "monoisotopicMass": 488.10567789753003,
      "unsaturation": 18,
      "elements": [{
        "symbol": "C",
        "number": 16
      }, {
        "symbol": "H",
        "number": 21
      }, {
        "symbol": "N",
        "number": 6
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Nna",
      "name": "N6,N6-dimethyladenosine monophosphate diradical 66A",
      "mf": "C12H16N5O6P",
      "kind": "NucleotideP",
      "oneLetter": "ζ",
      "ocl": {
        "value": "feghs@INBwlJ\\TgHOFwaEqrIQQSYQJIRIMZLyxMVuUUUPLpEUADSDHfP@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtPCQ@D}RFBp"
      },
      "mass": 357.2595904272741,
      "monoisotopicMass": 357.08382025367,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 6
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Fya",
      "name": "N6-formyladenosine monophosphate diradical 67A",
      "mf": "C11H12N5O7P",
      "kind": "NucleotideP",
      "oneLetter": "Ϩ",
      "ocl": {
        "value": "fegis@INBwlJ\\TgLp^MoBKcdRbbfrbTRdRZrYspZmjjjj`Y`JZBHfHQL`@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtPCQ@D}RtP@"
      },
      "mass": 357.216496438634,
      "monoisotopicMass": 357.04743474432,
      "unsaturation": 16,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Hma",
      "name": "N6-hydroxymethyladenosine monophosphate diradical 68A",
      "mf": "C11H14N5O7P",
      "kind": "NucleotideP",
      "oneLetter": "Ϫ",
      "ocl": {
        "value": "fegis@INBwlJ\\TgLp^MoBKcdRbbfrbTRdRZrYspZmjjjj`Y`JjBHfHQL`@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtPCQ@D}RtP@"
      },
      "mass": 359.23237794674554,
      "monoisotopicMass": 359.06308480878,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Cca",
      "name": "cyclic N6-threonylcarbamoyladenosine monophosphate diradical 69A",
      "mf": "C15H17N6O9P",
      "kind": "NucleotideP",
      "oneLetter": "e",
      "ocl": {
        "value": "ehRVIL@IG@fnehJNEDliko`OFspb\\\\bTTTvTRbTbSVTRRtXx|BjZvNjjjj`Y`IjfjbHPfDHSD`z`",
        "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@J_IorHbGtgD}F@RxRH_WwW@hbOTh}RtXCQ`A`l_`A`iVCjKAcjX@A~@h`Bup"
      },
      "mass": 456.30465685593623,
      "monoisotopicMass": 456.07946314904,
      "unsaturation": 20,
      "elements": [{
        "symbol": "C",
        "number": 15
      }, {
        "symbol": "H",
        "number": 17
      }, {
        "symbol": "N",
        "number": 6
      }, {
        "symbol": "O",
        "number": 9
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Fcy",
      "name": "5-formylcytidine monophosphate diradical71C",
      "mf": "C10H12N3O8P",
      "kind": "NucleotideP",
      "oneLetter": ">",
      "ocl": {
        "value": "faspK@I^[BgENSghOFwaEqrIQQSYQJIRYULxDmUUUTsLttQDqBId@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@@tP"
      },
      "mass": 333.1917590433254,
      "monoisotopicMass": 333.03620135502996,
      "unsaturation": 12,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 12
      }, {
        "symbol": "N",
        "number": 3
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Tur",
      "name": "4-thiouridine monophosphate diradical 74U",
      "mf": "C9H11N2O7PS",
      "kind": "NucleotideP",
      "oneLetter": "4",
      "ocl": {
        "value": "ff}Qp`I^aSbgIrCqmxQ\\ZaFQJJJ[JIQJSMg@ejjjjfYihbIbDSH@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFDuP"
      },
      "mass": 322.2317616628973,
      "monoisotopicMass": 322.0024588732,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Meg",
      "name": "7-methylguanosine monophosphate diradical 7G",
      "mf": "C11H15N5O7P",
      "kind": "NucleotideP",
      "oneLetter": "7",
      "ocl": {
        "value": "fegisDINCt\\J\\TgLp^MoBKbF\\bTTTvTRbTbRlSN^CWmUUUUKLuSTQDqBId@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RqdCQ@B\\StP@"
      },
      "mass": 360.2403187008013,
      "monoisotopicMass": 360.07090984101,
      "unsaturation": 13,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 15
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Mea",
      "name": "8-methyladenosine monophosphate diradical 8A",
      "mf": "C11H14N5O6P",
      "kind": "NucleotideP",
      "oneLetter": "â",
      "ocl": {
        "value": "fi{hs@INCt\\J\\UdhOFw`eqrIQQSYQJJJQKigOA[vjjjjAi`J`bIbDSH@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpP@c`a}_S_|BD}RSuKQ@MD@"
      },
      "mass": 343.2329730224273,
      "monoisotopicMass": 343.06817018921,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 14
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 6
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Dhu",
      "name": "dihydrouridine monophosphate diradical 8U",
      "mf": "C9H13N2O8P",
      "kind": "NucleotideP",
      "oneLetter": "D",
      "ocl": {
        "value": "ff}PK@I^aSbgIsTGc[pbxyDhhilheDiLv\\BVjjjjZffbHfHQL`@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFDuP"
      },
      "mass": 308.1822606892002,
      "monoisotopicMass": 308.04095238282997,
      "unsaturation": 8,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Ins",
      "name": "inosine monophosphate diradical 9A",
      "mf": "C10H11N4O7P",
      "kind": "NucleotideP",
      "oneLetter": "I",
      "ocl": {
        "value": "fakIs@INBvENJSghOFwaEqrIQQSYQJIRIMLyxMVuUUUTlsSTQDqBId@@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtPCQ@@"
      },
      "mass": 330.1911165763972,
      "monoisotopicMass": 330.03653570766,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "N",
        "number": 4
      }, {
        "symbol": "O",
        "number": 7
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Pis",
      "name": "pseudouridine monophosphate diradical 9U",
      "mf": "C9H11N2O8P",
      "kind": "NucleotideP",
      "oneLetter": "P",
      "ocl": {
        "value": "ff}PK@OAaSbgIsTGc[pbxyDhhilheDiLv\\BVjjjfZffbHfHQL`@",
        "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFDuP"
      },
      "mass": 306.1663791810886,
      "monoisotopicMass": 306.02530231837,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 9
      }, {
        "symbol": "H",
        "number": 11
      }, {
        "symbol": "N",
        "number": 2
      }, {
        "symbol": "O",
        "number": 8
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Pqb",
      "name": "preQ0base 100G diradical (base)",
      "mf": "C7H5N5O",
      "kind": "Nucleotide",
      "oneLetter": "ψ",
      "ocl": {
        "value": "dk^h@DxYLLbbTTRekiujYj^`@",
        "coordinates": "!B|Gwp_Gy|Gwp_[lk_gp_Ag_wrYRs}|f"
      },
      "mass": 175.1477760289729,
      "monoisotopicMass": 175.04940980287,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 7
      }, {
        "symbol": "H",
        "number": 5
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Pqg",
      "name": "preQ1base 101G diradical (base)",
      "mf": "C7H9N5O",
      "kind": "Nucleotide",
      "oneLetter": "∇",
      "ocl": {
        "value": "dk^h@DxYLLbbTTRckiUjYij`@",
        "coordinates": "!BWyfe[tlDWye_fXx@RpRe[wtHSuHH@a}"
      },
      "mass": 179.179539045196,
      "monoisotopicMass": 179.08070993179,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 7
      }, {
        "symbol": "H",
        "number": 9
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Qba",
      "name": "Qbase 10G diradical (base)",
      "mf": "C12H15N5O3",
      "kind": "Nucleotide",
      "oneLetter": "∴",
      "ocl": {
        "value": "fbmi`@D\\EHpHyrJIQQJMJIPtyIPTmSMMUMUP@@",
        "coordinates": "!BRpQ_f^i`RpQKAEARzfA_f_pHtP@H_Pc|BGuPThxUCl{RtBYTd|"
      },
      "mass": 277.27967290184347,
      "monoisotopicMass": 277.11748936431,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 15
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 3
      }]
    }, {
      "symbol": "Dgc",
      "name": "N2,7-dimethylguanosine cap (cap DMG) diradical 279553N",
      "mf": "C12H18N5O11P2",
      "kind": "Nucleotide",
      "oneLetter": "®",
      "ocl": {
        "value": "e`TZMBHIG@aihJNEHdlemck`OFspz|OgDJ\\bTTTvTRbTbRvbtfKGG`UPuUUUUJtuTmUTPaLHPfH@@",
        "coordinates": "!BvuPfpDnDtEK_t_rHtXBH_TwPbOr_IorHbGtgD}F@RxS|uxc|_]^OTh}RIlBH_]F@IqOQ`@A~_c|bH}RbGt"
      },
      "mass": 470.24625855539705,
      "monoisotopicMass": 470.04780541440005,
      "unsaturation": 13,
      "elements": [{
        "symbol": "C",
        "number": 12
      }, {
        "symbol": "H",
        "number": 18
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 11
      }, {
        "symbol": "P",
        "number": 2
      }]
    }, {
      "symbol": "Dpa",
      "name": "5′-(3′-dephosphoacetyl-CoA) diradical 4155N",
      "mf": "C23H35N7O16P3S",
      "kind": "Nucleotide",
      "oneLetter": "♣",
      "ocl": {
        "value": "elz~@jDCHlemnSTLBAEKBjfckgbV]XpEfCpB|IoCtHZy{lbdvbbfrbTRdRNRdnTbefRTrRTdTRrFVfjjjj`V`bZjjfjZjZ`bbLSaRP@",
        "coordinates": "!BvtmKaMmKUMlfgto[tDw_cosWt]~H@dvObGv_F_sWbOpgKMG_R}m}bHa}HbOSX}M_cQw}G_OwzH_[wW_c~H_Wx@G{|bM]}bGvHGxbGu~Oxa}bOq~Oxa}_c~H_WxuwvH_P"
      },
      "mass": 790.5483266874629,
      "monoisotopicMass": 790.1073852418399,
      "unsaturation": 21,
      "elements": [{
        "symbol": "C",
        "number": 23
      }, {
        "symbol": "H",
        "number": 35
      }, {
        "symbol": "N",
        "number": 7
      }, {
        "symbol": "O",
        "number": 16
      }, {
        "symbol": "P",
        "number": 3
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Dpm",
      "name": "5′-(3′-dephosphomalonyl-CoA) diradical 4255N",
      "mf": "C24H35N7O18P3S",
      "kind": "Nucleotide",
      "oneLetter": "♥",
      "ocl": {
        "value": "efq~DjDCHlemnSTLBAEKBjfckgbV]XrzpEfCpB|IoCtHZy{lbdvbbfrbTRdRNRdnTbefRTrRTrdbbVPrtuUUUTBtDSUUTuSUSSTDTQb\\JR@@",
        "coordinates": "!BIlB_Ib[@pAe`zni`FALSF@A~FBq~OrpXbGveX@A~_c~OTa`lzf@_ha}_]_Q`MF@bOpXKA`loXbH__rHb@JHoX`B@m]}uwx@bGu~Ox`BbKvH@ha}_c~H@hb}b@JH_Xc|_`BH_X`B_`BHoP"
      },
      "mass": 834.5578724328346,
      "monoisotopicMass": 834.0972144809799,
      "unsaturation": 23,
      "elements": [{
        "symbol": "C",
        "number": 24
      }, {
        "symbol": "H",
        "number": 35
      }, {
        "symbol": "N",
        "number": 7
      }, {
        "symbol": "O",
        "number": 18
      }, {
        "symbol": "P",
        "number": 3
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Dsc",
      "name": "5′-(3′-dephosphosuccinyl-CoA) radical 4355N",
      "mf": "C25H37N7O18P3S",
      "kind": "Nucleotide",
      "oneLetter": "♦",
      "ocl": {
        "value": "eny~DjDCHlemnSTLBAEKBjfckgbV]XzvpOFCpB|IoCtHZy{lbdvbbfrbTRdRNRdnTbefRTrRTrTdTRrFVfjjjj`V`bZjjfjZjZfhHhcDxTd@@",
        "coordinates": "!B[~kjXFjiV[Ry|fcm}MtGwWctvH_]Q_c}KaGwWbGvN`H}MgrX@_gx@h`gKB\\lbGvOSX}M@m^H@gwWbGvH@ha}_Xc|bGxb@I~@Ha}b@JH_X`B_`BH_X`BbGvH@ha}_c~H@ha}b@I~@Ha}"
      },
      "mass": 848.5844898376815,
      "monoisotopicMass": 848.11286454544,
      "unsaturation": 23,
      "elements": [{
        "symbol": "C",
        "number": 25
      }, {
        "symbol": "H",
        "number": 37
      }, {
        "symbol": "N",
        "number": 7
      }, {
        "symbol": "O",
        "number": 18
      }, {
        "symbol": "P",
        "number": 3
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Dpc",
      "name": "5′-(3′-dephospho-CoA) radical 455N",
      "mf": "C21H32N7O13P2S",
      "kind": "Nucleotide",
      "oneLetter": "♠",
      "ocl": {
        "value": "ek_^KBDIG@nabYXJNEHdliemh\\QPEfspZ|CPcKmnrIQQSYQJIRIGIRWJQRsIJYIccpJkjjjjjAZBIjjjZijjBDIaBDq@@",
        "coordinates": "!B[zW[UI|YchAMc{vHcuJH@m~NbGuKvwvHb@JNwx}Rgqe}bHa}@h`gDr\\Sb@JOTh}R@m]~@@A~b@I~@H`B_X`_hb}_`CW@h`B_`BH@gx@upJH@gx@b@I~@@"
      },
      "mass": 684.5310558604504,
      "monoisotopicMass": 684.1254042880199,
      "unsaturation": 19,
      "elements": [{
        "symbol": "C",
        "number": 21
      }, {
        "symbol": "H",
        "number": 32
      }, {
        "symbol": "N",
        "number": 7
      }, {
        "symbol": "O",
        "number": 13
      }, {
        "symbol": "P",
        "number": 2
      }, {
        "symbol": "S",
        "number": 1
      }]
    }, {
      "symbol": "Dpe",
      "name": "5′-diphosphate end 552N",
      "mf": "O3P",
      "kind": "Nucleotide",
      "oneLetter": "ϒ",
      "ocl": {
        "value": "gJQdebGF^Dx|duK@@",
        "coordinates": "!BbOq~@GxbGt"
      },
      "mass": 78.97197677137483,
      "monoisotopicMass": 78.95850585713,
      "unsaturation": 1,
      "elements": [{
        "symbol": "O",
        "number": 3
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Mgc",
      "name": "7-methylguanosine cap (cap 0) diradical 79553N",
      "mf": "C11H16N5O11P2",
      "kind": "Nucleotide",
      "oneLetter": "©",
      "ocl": {
        "value": "eohZMBHIG@aihJNEHdlemck`OFspz|GgDJ\\bTTTvTRbTbRvbtcXx|BjFjjjjiVfjejjHPfDHSD@@",
        "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@J_I`JHbGtgD}F@RxPBux`B_]^OTh}RIlBH_]F@IqOQ`@A~_c|BbHa}"
      },
      "mass": 456.2196411505502,
      "monoisotopicMass": 456.03215534994,
      "unsaturation": 13,
      "elements": [{
        "symbol": "C",
        "number": 11
      }, {
        "symbol": "H",
        "number": 16
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 11
      }, {
        "symbol": "P",
        "number": 2
      }]
    }, {
      "symbol": "Gyy",
      "name": "guanylylated 5′ end (cap G) diradical 9553N",
      "mf": "C10H13N5O10P2",
      "kind": "Nucleotide",
      "oneLetter": "ϑ",
      "ocl": {
        "value": "fkhh`INCt\\J\\UENY{NCqmxM|EnNQJJJ[JIQJQHzIRLyxM^uUUUTkSULuQDSDHfP@",
        "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@J_I`JHbGtgD}F@RxPBux`B_]^OTh}R_`CQ`B\\StXA~@C}~@Gx"
      },
      "mass": 425.1856780673293,
      "monoisotopicMass": 425.01376563368,
      "unsaturation": 14,
      "elements": [{
        "symbol": "C",
        "number": 10
      }, {
        "symbol": "H",
        "number": 13
      }, {
        "symbol": "N",
        "number": 5
      }, {
        "symbol": "O",
        "number": 10
      }, {
        "symbol": "P",
        "number": 2
      }]
    }, {
      "symbol": "Furp",
      "name": "furan phosphate radical",
      "mf": "C5H6O4P",
      "kind": "RNAp",
      "oneLetter": "⬠",
      "ocl": {
        "value": "dmtBPDpnAYcpRZ}eeYjii@@",
        "coordinates": "!BNvw|Vso|kUl{[So|PPAGuU\\z`pP"
      },
      "mass": 161.072705703704,
      "monoisotopicMass": 161.00037067008,
      "unsaturation": 5,
      "elements": [{
        "symbol": "C",
        "number": 5
      }, {
        "symbol": "H",
        "number": 6
      }, {
        "symbol": "O",
        "number": 4
      }, {
        "symbol": "P",
        "number": 1
      }]
    }, {
      "symbol": "Phg",
      "ocl": {
        "value": "dcNHPBPOEgEInVuWYj`@@@"
      },
      "name": "Phenyl glycine diradical",
      "mf": "C8H7NO",
      "kind": "aa",
      "mass": 133.1475805880365,
      "monoisotopicMass": 133.05276384961002,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 8
      }, {
        "symbol": "H",
        "number": 7
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 1
      }]
    }, {
      "symbol": "Hpg",
      "ocl": {
        "value": "dknDPBPp|V\\Tfy[WWYj`@`@@",
        "coordinates": "!BbOq~@Ha}bOq~Oxa}bGwW_Wx?_c?W_Wx?"
      },
      "name": "4-hydroxyphenylglycine diradical",
      "mf": "C8H7NO2",
      "kind": "aa",
      "mass": 149.14698551235477,
      "monoisotopicMass": 149.04767846918,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 8
      }, {
        "symbol": "H",
        "number": 7
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 2
      }]
    }, {
      "symbol": "Dpg",
      "ocl": {
        "value": "dg^LPBS[CqYqR[emYTyj`BH@@",
        "coordinates": "!BbOr~@Hb}bOr~Wxb}bKvWo[y_oe}HoY^}Uwt"
      },
      "name": "3,5-dihydroxyphenylglycine diradical",
      "mf": "C8H7NO3",
      "kind": "aa",
      "mass": 165.14639043667304,
      "monoisotopicMass": 165.04259308875,
      "unsaturation": 10,
      "elements": [{
        "symbol": "C",
        "number": 8
      }, {
        "symbol": "H",
        "number": 7
      }, {
        "symbol": "N",
        "number": 1
      }, {
        "symbol": "O",
        "number": 3
      }]
    }, {
      "symbol": "T",
      "name": "Tritium",
      "mf": "[3H]",
      "ocl": {
        "value": "eFAAYhBLCEX@",
        "coordinates": "!B@@L"
      },
      "mass": 3.0160492779,
      "monoisotopicMass": 3.0160492779,
      "unsaturation": -1,
      "elements": [{
        "symbol": "H",
        "number": 1,
        "isotope": 3
      }]
    }];

    const groupsObject = {};
    groups.forEach(e => {
      if (groupsObject[e.symbol]) {
        console.log('The symbol ' + e.symbol + ' is used more than once');
      }
      groupsObject[e.symbol] = e;
    });

    /**
     * Recreate a one letter sequence
     * @param {object} mf
     */

    function groupsToSequence(mf) {
      mf = mf.replace(/\([^(]*\)/g, '');
      let parts = mf.split(/(?=[A-Z ])/);
      let usefulParts = [];
      for (let part of parts) {
        if (part === ' ') {
          usefulParts.push(' ');
          continue;
        }
        if (!part.match(/^[A-Z][a-z]{2,6}/)) continue;
        if (groupsObject[part] && groupsObject[part].oneLetter) {
          usefulParts.push(groupsObject[part].oneLetter);
        } else {
          usefulParts.push('?');
        }
      }
      return usefulParts.join('').replace(/ +/g, ' ').trim();
    }

    const Style$1 = {
      SUPERIMPOSE: 'flex-direction: column;display: inline-flex;justify-content: center;text-align: left;vertical-align: middle;',
      SUPERIMPOSE_SUP_SUB: 'line-height: 1; font-size: 70%'
    };

    function toHtml(lines) {
      let html = [];
      for (let line of lines) {
        switch (line.kind) {
          case Format.SUBSCRIPT:
            html.push(`<sub>${line.value}</sub>`);
            break;
          case Format.SUPERSCRIPT:
            html.push(`<sup>${line.value}</sup>`);
            break;
          case Format.SUPERIMPOSE:
            html.push(`<span style="${Style$1.SUPERIMPOSE}">`);
            html.push(`<sup style="${Style$1.SUPERIMPOSE_SUP_SUB}">${line.over}</sup>`);
            html.push(`<sub style="${Style$1.SUPERIMPOSE_SUP_SUB}">${line.under}</sub>`);
            html.push('</span>');
            break;
          default:
            html.push(line.value);
        }
      }
      return html.join('');
    }

    const elements$1 = Object.keys(elementsObject).sort((a, b) => b.length - a.length);

    /**
     * Ensure that the mf has been entered with capital letters and not only lowercase
     * If there is only lowercase we try to capitalize the mf
     * @param {string} mf
     */

    function ensureCase(mf) {
      for (let i = 0; i < mf.length; i++) {
        if (mf.charCodeAt(i) > 64 && mf.charCodeAt(i) < 91) {
          return mf;
        }
      }
      let parts = mf.replace(/([a-z]*)([^a-z]*)/g, '$1 $2 ').split(/ +/);
      for (let i = 0; i < parts.length; i++) {
        if (parts[i].match(/^[a-z]$/)) {
          parts[i] = parts[i].toUpperCase();
        } else if (parts[i].match(/^[a-z]+$/)) {
          let newPart = '';
          for (let j = 0; j < parts[i].length; j++) {
            let two = parts[i].substr(j, 2);
            let one = parts[i].charAt(j).toUpperCase();
            if (['c', 'h', 'o', 'n'].includes(two.charAt(0)) && ['h', 'o', 'n'].includes(two.charAt(1))) {
              newPart += two.toUpperCase();
              j++;
            } else {
              two = two.charAt(0).toUpperCase() + two.charAt(1);
              if (elements$1.includes(two)) {
                newPart += two;
                j++;
              } else if (elements$1.includes(one)) {
                newPart += one;
              } else {
                return mf;
              }
            }
          }
          parts[i] = newPart;
        }
      }
      return parts.join('');
    }

    function flatten(parsed, options = {}) {
      const {
        groupIdentical = false,
        limit = 100000
      } = options;
      if (parsed.length === 0) return [''];
      let parts = [];
      let parenthesisLevel = 0;
      let currentPart;
      let comments = [];
      for (const entry of parsed) {
        if ((entry.kind === 'atom' || entry.kind === 'isotope' || entry.kind === 'openingParenthesis' || !currentPart) && parenthesisLevel === 0) {
          currentPart = {
            mf: '',
            min: 1,
            max: 1
          };
          parts.push(currentPart);
        }
        switch (entry.kind) {
          case 'atom':
            currentPart.mf += entry.value;
            break;
          case 'isotope':
            currentPart.mf += `[${entry.value.isotope}${entry.value.atom}]`;
            break;
          case 'multiplier':
            currentPart.mf += entry.value;
            break;
          case 'multiplierRange':
            if (parenthesisLevel !== 0) {
              throw new Error('Range definition inside parenthesis is not allowed.');
            }
            currentPart.min = entry.value.from;
            currentPart.max = entry.value.to;
            break;
          case 'openingParenthesis':
            parenthesisLevel++;
            currentPart.mf += entry.value;
            break;
          case 'charge':
            if (entry.value === 1) {
              currentPart.mf += '+';
            } else if (entry.value > 1) {
              currentPart.mf += `(+${entry.value})`;
            } else if (entry.value < 0) {
              currentPart.mf += `(${entry.value})`;
            }
            break;
          case 'closingParenthesis':
            parenthesisLevel--;
            currentPart.mf += entry.value;
            break;
          case 'comment':
            comments.push(entry.value);
            break;
          case 'text':
            break;
          default:
            throw new Error(`Could not flatten the parsed MF. Unknown kind: ${entry.kind}`);
        }
      }
      if (groupIdentical) {
        parts = optimizeRanges(parts);
      }
      const mfs = createMFs(parts, comments.join(' '), limit);
      return mfs;
    }

    /**
     * If we have many times the same mf we can combine them
     * This should only be applied if there are acutally some ranges
     */
    function optimizeRanges(parts) {
      let newParts = [];
      let mfsObject = {};
      let hasRange = false;
      for (const mf of parts) {
        if (mf.min !== mf.max) {
          hasRange = true;
          break;
        }
      }
      if (!hasRange) return parts;
      for (const mf of parts) {
        if (!mfsObject[mf.mf]) {
          mfsObject[mf.mf] = {
            mf: mf.mf,
            min: mf.min,
            max: mf.max
          };
          newParts.push(mfsObject[mf.mf]);
        } else {
          mfsObject[mf.mf].min = mfsObject[mf.mf].min + mf.min;
          mfsObject[mf.mf].max = mfsObject[mf.mf].max + mf.max;
        }
      }
      return newParts;
    }
    function createMFs(parts, comment, limit) {
      const currents = new Array(parts.length);
      for (let i = 0; i < currents.length; i++) {
        currents[i] = parts[i].min;
      }
      const mfs = [];
      let position = 0;
      while (position < currents.length) {
        if (currents[position] < parts[position].max) {
          mfs.push(getMF$1(parts, currents, comment));
          currents[position]++;
          for (let i = 0; i < position; i++) {
            currents[i] = parts[i].min;
          }
          position = 0;
        } else {
          position++;
        }
        if (mfs.length > limit) {
          throw Error(`MF.flatten generates too many fragments (over ${limit})`);
        }
      }
      mfs.push(getMF$1(parts, currents, comment));
      return mfs;
    }
    function getMF$1(parts, currents, comment) {
      let mf = '';
      for (let i = 0; i < parts.length; i++) {
        if (currents[i] === 0) {
          continue;
        }
        mf += parts[i].mf;
        if (currents[i] !== 1) {
          mf += currents[i];
        }
      }
      if (comment) mf += `$${comment}`;
      return mf;
    }

    function getIsotopeRatioInfo(value) {
      let result = {
        mass: 0,
        monoisotopicMass: 0
      };
      let element = elementsAndStableIsotopesObject[value.atom];
      if (!element) throw new Error(`Element not found: ${value.atom}`);
      let isotopesArray = element.isotopes;
      let ratios = normalize$2(value.ratio);
      let max = Math.max(...ratios);
      if (ratios.length > isotopesArray.length) {
        throw new Error(`the number of specified ratios is bigger that the number of stable isotopes: ${value.atom}`);
      }
      for (let i = 0; i < ratios.length; i++) {
        result.mass += ratios[i] * isotopesArray[i].mass;
        if (max === ratios[i] && result.monoisotopicMass === 0) {
          result.monoisotopicMass = isotopesArray[i].mass;
        }
      }
      return result;
    }
    function normalize$2(array) {
      let sum = array.reduce((prev, current) => prev + current, 0);
      return array.map(a => a / sum);
    }

    /**
     *
     * @param {*} parts
     * @param {*} [options={}]
     */
    function getEA(parts) {
      let results = {};
      for (let part of parts) {
        for (let line of part) {
          switch (line.kind) {
            case Kind.ISOTOPE:
              {
                let isotope = isotopesObject[line.value.isotope + line.value.atom];
                if (!isotope) {
                  throw new Error(`Unknown isotope: ${line.value.isotope}${line.value.atom}`);
                }
                addMass(results, line.value.atom, isotope.mass * line.multiplier);
                break;
              }
            case Kind.ISOTOPE_RATIO:
              {
                let isotopeRatioInfo = getIsotopeRatioInfo(line.value);
                addMass(results, line.value.atom, isotopeRatioInfo.mass * line.multiplier);
                break;
              }
            case Kind.ATOM:
              {
                let element = elementsObject[line.value];
                if (!element) {
                  element = groupsObject[line.value];
                  if (!element) throw Error(`Unknown element: ${line.value}`);
                  // need to explode group ????
                }

                addMass(results, line.value, element.mass * line.multiplier);
                break;
              }
            case Kind.CHARGE:
              break;
            default:
              throw new Error('partToMF unhandled Kind: ', line.kind);
          }
        }
      }
      let eas = [];
      let sum = 0;
      for (let key in results) {
        sum += results[key];
        eas.push({
          element: key,
          mass: results[key]
        });
      }
      eas.forEach(ea => {
        ea.ratio = ea.mass / sum;
      });
      return eas;
    }
    function addMass(results, atom, mass) {
      if (!results[atom]) results[atom] = 0;
      results[atom] += mass;
    }

    /**
     *
     * @param {*} parts
     * @param {*} [options={}]
     */
    function getElements(parts) {
      const elements = [];
      for (const part of parts) {
        for (const line of part) {
          let number = line.multiplier;
          switch (line.kind) {
            case Kind.ATOM:
              {
                let symbol = line.value;
                let element = elementsObject[symbol];
                if (!element) {
                  throw new Error(`element unknown: ${symbol} - ${line}`);
                }
                addElement(elements, {
                  symbol,
                  number
                });
                break;
              }
            case Kind.ISOTOPE:
              {
                let element = elementsAndIsotopesObject[line.value.atom];
                if (!element) {
                  throw new Error(`element unknown: ${part.value.atom} - ${line}`);
                }
                let isotope = element.isotopes.filter(a => a.nominal === line.value.isotope)[0];
                if (!isotope) {
                  throw new Error(`isotope unknown: ${line.value.isotope} - ${line}`);
                }
                addElement(elements, {
                  symbol: line.value.atom,
                  number,
                  isotope: line.value.isotope
                });
                break;
              }
            default:
              throw new Error(`unknown type: ${line.kind}`);
          }
        }
      }
      return elements;
    }
    function addElement(elements, newElement) {
      for (let element of elements) {
        if (element.symbol === newElement.symbol && element.isotope === newElement.isotope) {
          element.number += newElement.number;
          return;
        }
      }
      elements.push(newElement);
    }

    /**
     * Convert a MF part to an array of atoms
     * This procedure will suppress the isotopes !
     * This is mainly used to make queries
     */

    function partToAtoms(part) {
      let atoms = {};
      for (let line of part) {
        switch (line.kind) {
          case Kind.ISOTOPE:
            if (!atoms[line.value.atom]) atoms[line.value.atom] = 0;
            atoms[line.value.atom] += line.multiplier;
            break;
          case Kind.ISOTOPE_RATIO:
            if (!atoms[line.value.atom]) atoms[line.value.atom] = 0;
            atoms[line.value.atom] += line.multiplier;
            break;
          case Kind.ATOM:
            if (!atoms[line.value]) atoms[line.value] = 0;
            atoms[line.value] += line.multiplier;
            break;
          case Kind.CHARGE:
            break;
          case Kind.ANCHOR:
            break;
          default:
            throw new Error('partToMF unhandled Kind: ', line.kind);
        }
      }
      return atoms;
    }

    function partToMF(part, options = {}) {
      let mf = [];
      for (let line of part) {
        switch (line.kind) {
          case Kind.ISOTOPE:
            if (line.multiplier !== 0) {
              mf.push(`[${line.value.isotope}${line.value.atom}]${line.multiplier !== 1 ? line.multiplier : ''}`);
            }
            break;
          case Kind.ISOTOPE_RATIO:
            if (line.multiplier !== 0) {
              mf.push(`${line.value.atom}{${line.value.ratio.join(',')}}${line.multiplier !== 1 ? line.multiplier : ''}`);
            }
            break;
          case Kind.ATOM:
            if (line.multiplier !== 0) {
              mf.push(line.value + (line.multiplier !== 1 ? line.multiplier : ''));
            }
            break;
          case Kind.CHARGE:
            if (line.value === 0 || options.neutral) break;
            mf.push(`(${line.value > 0 ? `+${line.value}` : line.value})`);
            break;
        }
      }
      return mf.join('');
    }

    /**
     *
     * @param {*} parts
     * @param {*} [options={}]
     */
    function getInfo$1(parts, options = {}) {
      let {
        customUnsaturations = {},
        emFieldName = 'monoisotopicMass',
        msemFieldName = 'observedMonoisotopicMass'
      } = options;
      if (parts.length === 0) return {};
      if (parts.length === 1) {
        return getProcessedPart$1(parts[0], {
          customUnsaturations,
          emFieldName,
          msemFieldName
        });
      }
      let result = {
        parts: []
      };
      for (let part of parts) {
        result.parts.push(getProcessedPart$1(part, {
          customUnsaturations,
          emFieldName,
          msemFieldName
        }));
      }
      result[emFieldName] = 0;
      result.mass = 0;
      result.charge = 0;
      result.unsaturation = 0;
      result.atoms = {};
      result.mf = result.parts.map(a => a.mf).join('.');
      for (const part of result.parts) {
        result.mass += part.mass;
        result[emFieldName] += part[emFieldName];
        result.charge += part.charge;
        result.unsaturation += part.unsaturation;
        for (const atom in part.atoms) {
          if (!result.atoms[atom]) {
            result.atoms[atom] = 0;
          }
          result.atoms[atom] += part.atoms[atom];
        }
      }
      return result;
    }
    function getProcessedPart$1(part, options) {
      let {
        customUnsaturations,
        emFieldName,
        msemFieldName
      } = options;
      let currentPart = {
        mass: 0,
        charge: 0,
        mf: '',
        atoms: partToAtoms(part)
      };
      currentPart[emFieldName] = 0;
      let unsaturation = 0;
      let validUnsaturation = true;
      currentPart.mf = partToMF(part);
      for (let line of part) {
        let currentElement = '';
        switch (line.kind) {
          case Kind.ATOM:
            {
              currentElement = line.value;
              let element = elementsAndIsotopesObject[line.value];

              // todo should we have a kind GROUP ?
              if (!element) {
                element = groupsObject[line.value];
                if (!element) throw Error(`Unknown element: ${line.value}`);
                if (!customUnsaturations[line.value]) {
                  customUnsaturations[line.value] = element.unsaturation;
                }
              }
              if (!element) throw new Error(`Unknown element: ${line.value}`);
              currentPart[emFieldName] += element.monoisotopicMass * line.multiplier;
              currentPart.mass += element.mass * line.multiplier;
              break;
            }
          case Kind.ISOTOPE:
            {
              currentElement = line.value.atom;
              let isotope = isotopesObject[line.value.isotope + line.value.atom];
              if (!isotope) {
                throw new Error(`Unknown isotope: ${line.value.isotope}${line.value.atom}`);
              }
              currentPart[emFieldName] += isotope.mass * line.multiplier;
              currentPart.mass += isotope.mass * line.multiplier;
              break;
            }
          case Kind.ISOTOPE_RATIO:
            {
              currentElement = line.value.atom;
              let isotopeRatioInfo = getIsotopeRatioInfo(line.value);
              currentPart[emFieldName] += isotopeRatioInfo[emFieldName] * line.multiplier;
              currentPart.mass += isotopeRatioInfo.mass * line.multiplier;
              break;
            }
          case Kind.CHARGE:
            currentPart.charge = line.value;
            if (validUnsaturation) {
              unsaturation -= line.value;
            }
            break;
          default:
            throw new Error('Unimplemented Kind in getInfo', line.kind);
        }
        if (currentElement) {
          if (customUnsaturations[currentElement] !== undefined) {
            unsaturation += customUnsaturations[currentElement] * line.multiplier;
          } else if (unsaturationsObject[currentElement] !== undefined) {
            unsaturation += unsaturationsObject[currentElement] * line.multiplier;
          } else {
            validUnsaturation = false;
          }
        }
      }

      // need to calculate the observedMonoisotopicMass
      if (currentPart.charge) {
        currentPart[msemFieldName] = (currentPart[emFieldName] - currentPart.charge * ELECTRON_MASS) / Math.abs(currentPart.charge);
      }
      if (validUnsaturation) {
        currentPart.unsaturation = unsaturation / 2 + 1;
      }
      return currentPart;
    }

    /**
     *
     * @param {*} parts
     * @param {*} options
     */
    function getIsotopesInfo(parts) {
      if (parts.length === 0) return [];
      if (parts.length > 1) {
        throw new Error('getIsotopesInfo can not be applied on multipart MF');
      }
      return getProcessedPart(parts[0]);
    }
    function getProcessedPart(part) {
      let result = {
        charge: 0,
        isotopes: []
      };
      for (let line of part) {
        switch (line.kind) {
          case Kind.ISOTOPE:
            {
              let isotope = isotopesObject[line.value.isotope + line.value.atom];
              if (!isotope) {
                throw Error('unknown isotope:', line.value.atom, line.value.isotope);
              }
              result.isotopes.push({
                atom: `[${line.value.isotope}${line.value.atom}]`,
                number: line.multiplier,
                distribution: [{
                  x: isotope.mass,
                  y: 1
                }]
              });
              break;
            }
          case Kind.ISOTOPE_RATIO:
            {
              let element = elementsAndStableIsotopesObject[line.value.atom];
              if (!element) throw new Error('unknown element:', line.value);
              let distribution = getDistribution(element.isotopes, line.value.ratio);
              result.isotopes.push({
                atom: `${line.value.atom}{${line.value.ratio.join(',')}}`,
                number: line.multiplier,
                distribution
              });
            }
            break;
          case Kind.ATOM:
            {
              let element = elementsAndStableIsotopesObject[line.value];
              if (!element) throw new Error('unknown element:', line.value);
              result.isotopes.push({
                atom: line.value,
                number: line.multiplier,
                distribution: element.isotopes.map(e => ({
                  x: e.mass,
                  y: e.abundance
                }))
              });
              break;
            }
          case Kind.CHARGE:
            result.charge += line.value;
            break;
          default:
            throw new Error('partToMF unhandled Kind: ', line.kind);
        }
      }
      return result;
    }
    function getDistribution(isotopesArray, ratio) {
      let ratios = normalize$1(ratio);
      let result = [];
      if (ratios.length > isotopesArray.length) {
        throw new Error(`the number of specified ratios is bigger that the number of stable isotopes: ${isotopesObject}`);
      }
      for (let i = 0; i < ratios.length; i++) {
        result.push({
          x: isotopesArray[i].mass,
          y: ratios[i]
        });
      }
      return result;
    }
    function normalize$1(array) {
      let sum = array.reduce((prev, current) => prev + current, 0);
      return array.map(a => a / sum);
    }

    /**
     * Converts an array of mf elements to an array of formatting information
     * @param {Array<Object>} result of the parse method
     */

    function partsToDisplay(parts) {
      let lines = [];
      for (let part of parts) {
        if (lines.length > 0) lines.push({
          kind: Kind.SALT,
          value: '•'
        });
        for (let partLine of part) {
          lines.push(partLine);
          if (partLine.multiplier) {
            lines.push({
              kind: Kind.MULTIPLIER,
              value: partLine.multiplier
            });
          }
        }
      }
      return toDisplay(lines);
    }

    function partsToMF(parts, options) {
      let mf = [];
      for (let part of parts) {
        mf.push(partToMF(part, options));
      }
      return mf.join(' . ');
    }

    /**
     * Implementation of the Hill system for sorting atoms
     * https://en.wikipedia.org/wiki/Chemical_formula#Hill_system
     * @param {string} a - first atom to compare
     * @param {string} b - second atom to compare
     * @returns
     */

    function atomSorter(a, b) {
      if (a === b) return 0;
      if (a === 'C') return -1;
      if (b === 'C') return 1;
      if (a === 'H') return -1;
      if (b === 'H') return 1;
      if (a < b) return -1;
      return 1;
    }

    /**
     *
     * @param {*} lines
     * @param {object} [options={}]
     * @param {boolean} [options.expand=true] - Should we expand the groupsObject
     */

    function toParts(lines, options = {}) {
      const {
        expand: shouldExpandgroupsObject = true
      } = options;
      let parts = [];
      let currentPart = createNewPart();
      let previousKind = Kind.BEGIN;
      parts.push(currentPart);
      for (let line of lines) {
        switch (line.kind) {
          case Kind.ATOM:
          case Kind.ISOTOPE_RATIO:
          case Kind.ISOTOPE:
          case Kind.CHARGE:
            currentPart.lines.push({
              ...line,
              multiplier: 1
            });
            break;
          case Kind.OPENING_PARENTHESIS:
            openingParenthesis(currentPart);
            break;
          case Kind.CLOSING_PARENTHESIS:
            closingParenthesis(currentPart);
            break;
          case Kind.PRE_MULTIPLIER:
            preMultiplier(currentPart, line);
            break;
          case Kind.MULTIPLIER:
            postMultiplier(currentPart, line.value, previousKind);
            break;
          case Kind.SALT:
            globalPartMultiplier(currentPart);
            currentPart = createNewPart();
            parts.push(currentPart);
            break;
          case Kind.ANCHOR:
            // we ignore anchors to create the parts and canonized MF
            break;
          case Kind.COMMENT:
            // we ignore comments to create the parts and canonized MF
            break;
          case Kind.TEXT:
            break;
          default:
            throw new Error(`Can not process mf having: ${line.kind}`);
        }
        previousKind = line.kind;
      }
      globalPartMultiplier(currentPart);
      if (shouldExpandgroupsObject) expandgroupsObject(parts);
      return combineAtomsIsotopesCharges(parts);
    }
    function createNewPart() {
      let currentMultiplier = {
        value: 1,
        fromIndex: 0
      };
      return {
        lines: [],
        multipliers: [currentMultiplier],
        currentMultiplier
      };
    }
    function openingParenthesis(currentPart) {
      currentPart.currentMultiplier = {
        value: 1,
        fromIndex: currentPart.lines.length
      };
      currentPart.multipliers.push(currentPart.currentMultiplier);
    }
    function closingParenthesis(currentPart) {
      currentPart.currentMultiplier = currentPart.multipliers.pop();
      if (currentPart.currentMultiplier !== 1) {
        for (let i = currentPart.currentMultiplier.fromIndex; i < currentPart.lines.length; i++) {
          currentPart.lines[i].multiplier *= currentPart.currentMultiplier.value;
        }
      }
    }
    function preMultiplier(currentPart, line) {
      currentPart.currentMultiplier.value *= line.value;
    }
    function globalPartMultiplier(currentPart) {
      for (let i = currentPart.multipliers[0].fromIndex; i < currentPart.lines.length; i++) {
        currentPart.lines[i].multiplier *= currentPart.multipliers[0].value;
      }
    }
    function postMultiplier(currentPart, value, previousKind) {
      if (previousKind === Kind.CLOSING_PARENTHESIS) {
        // need to apply to everything till the previous parenthesis
        for (let i = currentPart.currentMultiplier.fromIndex; i < currentPart.lines.length; i++) {
          currentPart.lines[i].multiplier *= value;
        }
      } else {
        // just applies to the previous element
        currentPart.lines[currentPart.lines.length - 1].multiplier *= value;
      }
    }
    function expandgroupsObject(parts) {
      for (let part of parts) {
        let expanded = false;
        for (let i = 0; i < part.lines.length; i++) {
          let line = part.lines[i];
          if (line.kind === Kind.ATOM) {
            let group = groupsObject[line.value];
            if (group) {
              expanded = true;
              for (let element of group.elements) {
                if (element.isotope) {
                  part.lines.push({
                    kind: 'isotope',
                    value: {
                      atom: element.symbol,
                      isotope: element.isotope
                    },
                    multiplier: line.multiplier * element.number
                  });
                } else {
                  part.lines.push({
                    kind: 'atom',
                    value: element.symbol,
                    multiplier: line.multiplier * element.number
                  });
                }
              }
              part.lines[i] = undefined;
            }
          }
        }
        if (expanded) part.lines = part.lines.filter(a => a);
      }
    }
    function combineAtomsIsotopesCharges(parts) {
      let results = [];
      for (let part of parts) {
        let result = [];
        results.push(result);
        calculateAndSortKeys(part);
        let currentKey = '';
        for (let key of part.keys) {
          if (key.key === Kind.CHARGE) {
            if (currentKey !== key.key) {
              result.push({
                kind: Kind.CHARGE,
                value: key.value.value * key.value.multiplier
              });
            } else {
              result[result.length - 1].value += key.value.value * key.value.multiplier;
            }
          } else if (currentKey !== key.key) {
            result.push(key.value);
          } else {
            result[result.length - 1].multiplier += key.value.multiplier;
          }
          currentKey = key.key;
        }
        result.sort((a, b) => {
          if (a.kind === Kind.CHARGE) return 1;
          if (b.kind === Kind.CHARGE) return -1;
          let atomA = a.kind === Kind.ATOM ? a.value : a.value.atom;
          let atomB = b.kind === Kind.ATOM ? b.value : b.value.atom;
          if (atomA !== atomB) return atomSorter(atomA, atomB);
          // same atome but some isotopes ...
          if (a.kind === Kind.ATOM) return -1;
          if (b.kind === Kind.ATOM) return 1;
          if (a.kind === Kind.ISOTOPE) return -1;
          if (b.kind === Kind.ISOTOPE) return 1;
          if (a.kind === Kind.ISOTOPE_RATIO) return -1;
          if (b.kind === Kind.ISOTOPE_RATIO) return 1;
          return 0;
        });
      }
      return results;
    }
    function calculateAndSortKeys(part) {
      part.keys = [];
      for (let line of part.lines) {
        part.keys.push({
          key: getKey(line),
          value: line
        });
      }
      part.keys.sort((a, b) => stringComparator(a.key, b.key));
    }
    function getKey(line) {
      let key = [line.kind];
      switch (line.kind) {
        case Kind.CHARGE:
          break;
        default:
          if (typeof line.value === 'string') {
            key.push(line.value);
          } else {
            for (let prop of Object.keys(line.value).sort()) {
              key.push(line.value[prop]);
            }
          }
      }
      return key.join('-');
    }
    function stringComparator(a, b) {
      if (a < b) return -1;
      if (a > b) return 1;
      return 0;
    }

    function toText(lines) {
      let text = [];
      for (let line of lines) {
        switch (line.kind) {
          case Format.SUBSCRIPT:
            {
              const value = String(line.value);
              for (let i = 0; i < value.length; i++) {
                const char = value[i];
                if (subscript[char]) {
                  text.push(subscript[char]);
                } else {
                  throw new Error(`Subscript problem with: ${char}`);
                }
              }
            }
            break;
          case Format.SUPERSCRIPT:
            {
              const value = String(line.value);
              for (let i = 0; i < value.length; i++) {
                const char = value[i];
                if (superscript[char]) {
                  text.push(superscript[char]);
                } else {
                  throw new Error(`Superscript problem with: ${char}`);
                }
              }
              break;
            }
          case Format.SUPERIMPOSE:
            {
              const under = String(line.under);
              for (let i = 0; i < under.length; i++) {
                const char = under[i];
                if (subscript[char]) {
                  text.push(subscript[char]);
                } else {
                  throw new Error(`Subscript problem with: ${char}`);
                }
              }
              const over = String(line.over);
              for (let i = 0; i < over.length; i++) {
                const char = over[i];
                if (superscript[char]) {
                  text.push(superscript[char]);
                } else {
                  throw new Error(`Superscript problem with: ${char}`);
                }
              }
              break;
            }
          default:
            text.push(line.value);
        }
      }
      return text.join('');
    }

    /**
     * Class allowing to deal with molecular formula and derived information
     */
    class MF {
      constructor(mf, options = {}) {
        if (options.ensureCase) {
          mf = ensureCase(mf);
        }
        this.parsed = parse(mf);
        this.cache = {};
      }
      toDisplay() {
        if (!this.cache.displayed) this.cache.displayed = toDisplay(this.parsed);
        return this.cache.displayed;
      }
      toHtml() {
        if (!this.cache.html) {
          this.toDisplay();
          this.cache.html = toHtml(this.cache.displayed);
        }
        return this.cache.html;
      }
      toText() {
        if (!this.cache.text) {
          this.toDisplay();
          this.cache.text = toText(this.cache.displayed);
        }
        return this.cache.text;
      }
      toCanonicText() {
        if (!this.cache.canonicText) {
          this.cache.canonicText = new MF(this.toMF()).toText(this.cache.displayed);
        }
        return this.cache.canonicText;
      }
      toParts(options) {
        if (!this.cache.parts) {
          this.cache.parts = toParts(this.parsed, options);
        }
        return this.cache.parts;
      }

      /**
       * Returns an object with the global MF, global charge, monoisotopic mass and mass
       * as well as the same information for all the parts
       * @param {object} [options={}] options
       * @param {object} [options.customUnsaturations={}] custom unsaturations
       * @param {string} [options.emFieldName='monoisotopicMass'] name of the monoisotopic mass field
       * @param {string} [options.msemFieldName='observedMonoisotopicMass'] name of the observed monoisotopic mass field
       */
      getInfo(options = {}) {
        if (!this.cache.info) {
          this.toParts();
          this.cache.info = getInfo$1(this.cache.parts, options);
        }
        return this.cache.info;
      }

      /**
       * Returns an object with the elemental analysis
       */
      getEA(options = {}) {
        if (!this.cache.ea) {
          this.toParts();
          this.cache.ea = getEA(this.cache.parts);
        }
        return this.cache.ea;
      }

      /**
       * Get the different elements for each part
       * @returns an array
       */
      getElements() {
        if (!this.cache.elements) {
          this.toParts();
          this.cache.elements = getElements(this.cache.parts);
        }
        return this.cache.elements;
      }

      /**
       * Returns an array with each atom and isotopic composition
       */
      getIsotopesInfo(options = {}) {
        if (!this.cache.isotopesInfo) {
          this.toParts();
          this.cache.isotopesInfo = getIsotopesInfo(this.cache.parts);
        }
        return this.cache.isotopesInfo;
      }

      /**
       * Get a canonized MF
       */
      toMF() {
        if (!this.cache.mf) {
          this.toParts();
          this.cache.mf = partsToMF(this.cache.parts);
        }
        return this.cache.mf;
      }

      /**
       * Get a canonized MF
       */
      toNeutralMF() {
        if (!this.cache.neutralMF) {
          this.toParts();
          this.cache.neutralMF = partsToMF(this.cache.parts, {
            neutral: true
          });
        }
        return this.cache.neutralMF;
      }
      canonize() {
        this.toParts();
        this.cache.displayed = partsToDisplay(this.cache.parts);
        this.cache.html = undefined;
      }
      flatten(options) {
        return flatten(this.parsed, options);
      }
    }

    /**
     * Filter the array of peaks
     * @param {array} peaks - array of all the peaks
     * @param {object} [options={}]
     * @param {number} [options.from] - min X value of the window to consider
     * @param {number} [options.to] - max X value of the window to consider
     * @param {number} [options.threshold=0.01] - minimal intensity compare to base peak
     * @param {number} [options.limit=undefined] - maximal number of peaks (based on intensity)
     * @param {number} [options.sumValue] // if sumValue is defined, maxValue is ignored
     * @returns {array} - copy of peaks with 'close' annotation
     */

    function getPeaks(peaks, options = {}) {
      const {
        from = xyObjectMinXPoint(peaks).x,
        to = xyObjectMaxXPoint(peaks).x,
        threshold = 0.01,
        limit,
        sumValue
      } = options;
      let maxY = Number.MIN_SAFE_INTEGER;
      for (let peak of peaks) {
        if (peak.y > maxY) maxY = peak.y;
      }
      let minY = maxY * threshold;
      peaks = peaks.filter(peak => peak.x >= from && peak.x <= to && peak.y >= minY);
      if (limit && peaks.length > limit) {
        peaks.sort((a, b) => b.y - a.y);
        peaks = peaks.slice(0, limit);
      }
      if (sumValue) {
        peaks = JSON.parse(JSON.stringify(peaks));
        const currentSum = xyObjectSumY(peaks);
        const ratio = sumValue / currentSum;
        peaks.forEach(peak => peak.y *= ratio);
      }
      return peaks.sort((a, b) => a.x - b.x);
    }

    /**
     * Filter the array of peaks
     * @param {array} peaks - array of all the peaks
     * @param {string} mf - Molecular formula of the parent molecule
     * @param {object} [options={}]
     * @param {number} [options.from] - min X value of the window to consider
     * @param {number} [options.to] - max X value of the window to consider
     * @param {number} [options.threshold=0.01] - minimal intensity compare to base peak
     * @param {number} [options.limit=undefined] - maximal number of peaks (based on intensity)
     * @param {string} [options.ionizations]
     * @param {number} [options.precision]
     * @returns {array} - copy of peaks with 'close' annotation
     */

    async function getFragmentPeaks(peaks, mf, options = {}) {
      const emdb = new EMDB$1();
      const {
        ionizations = '',
        precision
      } = options;
      const mfInfo = new MF(mf).getInfo();
      const ranges = Object.keys(mfInfo.atoms).map(key => `${key}0-${mfInfo.atoms[key]}`).join(' ');
      peaks = getPeaks(peaks, options);
      for (let peak of peaks) {
        peak.mfs = (await emdb.fromMonoisotopicMass(peak.x, {
          precision,
          ranges,
          ionizations
        })).mfs;
      }
      peaks = peaks.filter(peak => peak.mfs.length > 0);
      return peaks;
    }

    /**
     * Remove an integer number of time the specifiedd monoisotopic mass
     * Mass remainder analysis (MARA): https://doi.org/10.1021/acs.analchem.7b04730
     * @param {object} spectrum
     * @param {number} mass
     * @param {object} [options={}
     * @param {number} [options.delta=0.001]
     */
    function getMassRemainder(spectrum, mass, options = {}) {
      const {
        delta = 0.001
      } = options;
      const x = spectrum.x.slice();
      const y = spectrum.y;
      for (let i = 0; i < x.length; i++) {
        const factor = Math.floor(x[i] / mass);
        x[i] = x[i] - factor * mass;
      }
      // we sort and join
      return xyJoinX(xySortX({
        x,
        y
      }), {
        delta
      });
    }

    function calculateOverlapFromDiff(diffs) {
      if (diffs[1].length === 0) return 0;
      let sumPos = 0;
      for (let i = 0; i < diffs[1].length; i++) {
        sumPos += Math.abs(diffs[1][i]);
      }
      return 1 - sumPos;
    }

    /**
     * This code requires the use of an array like  [[x1,y1],[x2,y2], ...]
     * If it is not the right format, we will just convert it
     * Otherwise we return the correct format
     * @param {Peaks} peaks
     * @returns [number[], number[]]
     */
    function checkPeaks(peaks) {
      // if it is already a 2D array of points, we just return them
      if (Array.isArray(peaks) && Array.isArray(peaks[0]) && peaks.length === 2) {
        return peaks;
      }
      if (Array.isArray(peaks.x) && Array.isArray(peaks.y)) {
        return [peaks.x, peaks.y];
      }
      const x = new Array(peaks.length);
      const y = new Array(peaks.length);
      for (let i = 0; i < peaks.length; i++) {
        x[i] = peaks[i][0];
        y[i] = peaks[i][1];
      }
      return [x, y];
    }

    function extract(array, from, to) {
      const newArray = [[], []];
      let j = 0;
      const length = array[0] ? array[0].length : 0;
      for (let i = 0; i < length; i++) {
        if ((!from || array[0][i] >= from) && (!to || array[0][i] <= to)) {
          newArray[0][j] = array[0][i];
          newArray[1][j] = array[1][i];
          j++;
        }
      }
      return newArray;
    }

    // returns an new array based on array1 where there is a peak of array2 at a distance under width/2
    function getCommonArray(array1, array2, width) {
      const newArray = [[], []];
      let pos2 = 0;
      width /= 2;
      let j = 0;
      const array1Length = array1[0] ? array1[0].length : 0;
      const array2Length = array2[0] ? array2[0].length : 0;
      for (let i = 0; i < array1Length; i++) {
        while (pos2 < array2Length && array1[0][i] > array2[0][pos2] + width) {
          pos2++;
        }
        if (pos2 < array2Length && array1[0][i] > array2[0][pos2] - width) {
          newArray[0][j] = array1[0][i];
          newArray[1][j] = array1[1][i];
          j++;
        }
      }
      return newArray;
    }

    var mlStat = {};

    var array$2 = {};

    (function (exports) {

      function compareNumbers(a, b) {
        return a - b;
      }

      /**
       * Computes the sum of the given values
       * @param {Array} values
       * @returns {number}
       */
      exports.sum = function sum(values) {
        var sum = 0;
        for (var i = 0; i < values.length; i++) {
          sum += values[i];
        }
        return sum;
      };

      /**
       * Computes the maximum of the given values
       * @param {Array} values
       * @returns {number}
       */
      exports.max = function max(values) {
        var max = values[0];
        var l = values.length;
        for (var i = 1; i < l; i++) {
          if (values[i] > max) max = values[i];
        }
        return max;
      };

      /**
       * Computes the minimum of the given values
       * @param {Array} values
       * @returns {number}
       */
      exports.min = function min(values) {
        var min = values[0];
        var l = values.length;
        for (var i = 1; i < l; i++) {
          if (values[i] < min) min = values[i];
        }
        return min;
      };

      /**
       * Computes the min and max of the given values
       * @param {Array} values
       * @returns {{min: number, max: number}}
       */
      exports.minMax = function minMax(values) {
        var min = values[0];
        var max = values[0];
        var l = values.length;
        for (var i = 1; i < l; i++) {
          if (values[i] < min) min = values[i];
          if (values[i] > max) max = values[i];
        }
        return {
          min: min,
          max: max
        };
      };

      /**
       * Computes the arithmetic mean of the given values
       * @param {Array} values
       * @returns {number}
       */
      exports.arithmeticMean = function arithmeticMean(values) {
        var sum = 0;
        var l = values.length;
        for (var i = 0; i < l; i++) {
          sum += values[i];
        }
        return sum / l;
      };

      /**
       * {@link arithmeticMean}
       */
      exports.mean = exports.arithmeticMean;

      /**
       * Computes the geometric mean of the given values
       * @param {Array} values
       * @returns {number}
       */
      exports.geometricMean = function geometricMean(values) {
        var mul = 1;
        var l = values.length;
        for (var i = 0; i < l; i++) {
          mul *= values[i];
        }
        return Math.pow(mul, 1 / l);
      };

      /**
       * Computes the mean of the log of the given values
       * If the return value is exponentiated, it gives the same result as the
       * geometric mean.
       * @param {Array} values
       * @returns {number}
       */
      exports.logMean = function logMean(values) {
        var lnsum = 0;
        var l = values.length;
        for (var i = 0; i < l; i++) {
          lnsum += Math.log(values[i]);
        }
        return lnsum / l;
      };

      /**
       * Computes the weighted grand mean for a list of means and sample sizes
       * @param {Array} means - Mean values for each set of samples
       * @param {Array} samples - Number of original values for each set of samples
       * @returns {number}
       */
      exports.grandMean = function grandMean(means, samples) {
        var sum = 0;
        var n = 0;
        var l = means.length;
        for (var i = 0; i < l; i++) {
          sum += samples[i] * means[i];
          n += samples[i];
        }
        return sum / n;
      };

      /**
       * Computes the truncated mean of the given values using a given percentage
       * @param {Array} values
       * @param {number} percent - The percentage of values to keep (range: [0,1])
       * @param {boolean} [alreadySorted=false]
       * @returns {number}
       */
      exports.truncatedMean = function truncatedMean(values, percent, alreadySorted) {
        if (alreadySorted === undefined) alreadySorted = false;
        if (!alreadySorted) {
          values = [].concat(values).sort(compareNumbers);
        }
        var l = values.length;
        var k = Math.floor(l * percent);
        var sum = 0;
        for (var i = k; i < l - k; i++) {
          sum += values[i];
        }
        return sum / (l - 2 * k);
      };

      /**
       * Computes the harmonic mean of the given values
       * @param {Array} values
       * @returns {number}
       */
      exports.harmonicMean = function harmonicMean(values) {
        var sum = 0;
        var l = values.length;
        for (var i = 0; i < l; i++) {
          if (values[i] === 0) {
            throw new RangeError('value at index ' + i + 'is zero');
          }
          sum += 1 / values[i];
        }
        return l / sum;
      };

      /**
       * Computes the contraharmonic mean of the given values
       * @param {Array} values
       * @returns {number}
       */
      exports.contraHarmonicMean = function contraHarmonicMean(values) {
        var r1 = 0;
        var r2 = 0;
        var l = values.length;
        for (var i = 0; i < l; i++) {
          r1 += values[i] * values[i];
          r2 += values[i];
        }
        if (r2 < 0) {
          throw new RangeError('sum of values is negative');
        }
        return r1 / r2;
      };

      /**
       * Computes the median of the given values
       * @param {Array} values
       * @param {boolean} [alreadySorted=false]
       * @returns {number}
       */
      exports.median = function median(values, alreadySorted) {
        if (alreadySorted === undefined) alreadySorted = false;
        if (!alreadySorted) {
          values = [].concat(values).sort(compareNumbers);
        }
        var l = values.length;
        var half = Math.floor(l / 2);
        if (l % 2 === 0) {
          return (values[half - 1] + values[half]) * 0.5;
        } else {
          return values[half];
        }
      };

      /**
       * Computes the variance of the given values
       * @param {Array} values
       * @param {boolean} [unbiased=true] - if true, divide by (n-1); if false, divide by n.
       * @returns {number}
       */
      exports.variance = function variance(values, unbiased) {
        if (unbiased === undefined) unbiased = true;
        var theMean = exports.mean(values);
        var theVariance = 0;
        var l = values.length;
        for (var i = 0; i < l; i++) {
          var x = values[i] - theMean;
          theVariance += x * x;
        }
        if (unbiased) {
          return theVariance / (l - 1);
        } else {
          return theVariance / l;
        }
      };

      /**
       * Computes the standard deviation of the given values
       * @param {Array} values
       * @param {boolean} [unbiased=true] - if true, divide by (n-1); if false, divide by n.
       * @returns {number}
       */
      exports.standardDeviation = function standardDeviation(values, unbiased) {
        return Math.sqrt(exports.variance(values, unbiased));
      };
      exports.standardError = function standardError(values) {
        return exports.standardDeviation(values) / Math.sqrt(values.length);
      };

      /**
       * IEEE Transactions on biomedical engineering, vol. 52, no. 1, january 2005, p. 76-
       * Calculate the standard deviation via the Median of the absolute deviation
       *  The formula for the standard deviation only holds for Gaussian random variables.
       * @returns {{mean: number, stdev: number}}
       */
      exports.robustMeanAndStdev = function robustMeanAndStdev(y) {
        var mean = 0,
          stdev = 0;
        var length = y.length,
          i = 0;
        for (i = 0; i < length; i++) {
          mean += y[i];
        }
        mean /= length;
        var averageDeviations = new Array(length);
        for (i = 0; i < length; i++) averageDeviations[i] = Math.abs(y[i] - mean);
        averageDeviations.sort(compareNumbers);
        if (length % 2 === 1) {
          stdev = averageDeviations[(length - 1) / 2] / 0.6745;
        } else {
          stdev = 0.5 * (averageDeviations[length / 2] + averageDeviations[length / 2 - 1]) / 0.6745;
        }
        return {
          mean: mean,
          stdev: stdev
        };
      };
      exports.quartiles = function quartiles(values, alreadySorted) {
        if (typeof alreadySorted === 'undefined') alreadySorted = false;
        if (!alreadySorted) {
          values = [].concat(values).sort(compareNumbers);
        }
        var quart = values.length / 4;
        var q1 = values[Math.ceil(quart) - 1];
        var q2 = exports.median(values, true);
        var q3 = values[Math.ceil(quart * 3) - 1];
        return {
          q1: q1,
          q2: q2,
          q3: q3
        };
      };
      exports.pooledStandardDeviation = function pooledStandardDeviation(samples, unbiased) {
        return Math.sqrt(exports.pooledVariance(samples, unbiased));
      };
      exports.pooledVariance = function pooledVariance(samples, unbiased) {
        if (typeof unbiased === 'undefined') unbiased = true;
        var sum = 0;
        var length = 0,
          l = samples.length;
        for (var i = 0; i < l; i++) {
          var values = samples[i];
          var vari = exports.variance(values);
          sum += (values.length - 1) * vari;
          if (unbiased) length += values.length - 1;else length += values.length;
        }
        return sum / length;
      };
      exports.mode = function mode(values) {
        var l = values.length,
          itemCount = new Array(l),
          i;
        for (i = 0; i < l; i++) {
          itemCount[i] = 0;
        }
        var itemArray = new Array(l);
        var count = 0;
        for (i = 0; i < l; i++) {
          var index = itemArray.indexOf(values[i]);
          if (index >= 0) itemCount[index]++;else {
            itemArray[count] = values[i];
            itemCount[count] = 1;
            count++;
          }
        }
        var maxValue = 0,
          maxIndex = 0;
        for (i = 0; i < count; i++) {
          if (itemCount[i] > maxValue) {
            maxValue = itemCount[i];
            maxIndex = i;
          }
        }
        return itemArray[maxIndex];
      };
      exports.covariance = function covariance(vector1, vector2, unbiased) {
        if (typeof unbiased === 'undefined') unbiased = true;
        var mean1 = exports.mean(vector1);
        var mean2 = exports.mean(vector2);
        if (vector1.length !== vector2.length) throw 'Vectors do not have the same dimensions';
        var cov = 0,
          l = vector1.length;
        for (var i = 0; i < l; i++) {
          var x = vector1[i] - mean1;
          var y = vector2[i] - mean2;
          cov += x * y;
        }
        if (unbiased) return cov / (l - 1);else return cov / l;
      };
      exports.skewness = function skewness(values, unbiased) {
        if (typeof unbiased === 'undefined') unbiased = true;
        var theMean = exports.mean(values);
        var s2 = 0,
          s3 = 0,
          l = values.length;
        for (var i = 0; i < l; i++) {
          var dev = values[i] - theMean;
          s2 += dev * dev;
          s3 += dev * dev * dev;
        }
        var m2 = s2 / l;
        var m3 = s3 / l;
        var g = m3 / Math.pow(m2, 3 / 2.0);
        if (unbiased) {
          var a = Math.sqrt(l * (l - 1));
          var b = l - 2;
          return a / b * g;
        } else {
          return g;
        }
      };
      exports.kurtosis = function kurtosis(values, unbiased) {
        if (typeof unbiased === 'undefined') unbiased = true;
        var theMean = exports.mean(values);
        var n = values.length,
          s2 = 0,
          s4 = 0;
        for (var i = 0; i < n; i++) {
          var dev = values[i] - theMean;
          s2 += dev * dev;
          s4 += dev * dev * dev * dev;
        }
        var m2 = s2 / n;
        var m4 = s4 / n;
        if (unbiased) {
          var v = s2 / (n - 1);
          var a = n * (n + 1) / ((n - 1) * (n - 2) * (n - 3));
          var b = s4 / (v * v);
          var c = (n - 1) * (n - 1) / ((n - 2) * (n - 3));
          return a * b - 3 * c;
        } else {
          return m4 / (m2 * m2) - 3;
        }
      };
      exports.entropy = function entropy(values, eps) {
        if (typeof eps === 'undefined') eps = 0;
        var sum = 0,
          l = values.length;
        for (var i = 0; i < l; i++) sum += values[i] * Math.log(values[i] + eps);
        return -sum;
      };
      exports.weightedMean = function weightedMean(values, weights) {
        var sum = 0,
          l = values.length;
        for (var i = 0; i < l; i++) sum += values[i] * weights[i];
        return sum;
      };
      exports.weightedStandardDeviation = function weightedStandardDeviation(values, weights) {
        return Math.sqrt(exports.weightedVariance(values, weights));
      };
      exports.weightedVariance = function weightedVariance(values, weights) {
        var theMean = exports.weightedMean(values, weights);
        var vari = 0,
          l = values.length;
        var a = 0,
          b = 0;
        for (var i = 0; i < l; i++) {
          var z = values[i] - theMean;
          var w = weights[i];
          vari += w * (z * z);
          b += w;
          a += w * w;
        }
        return vari * (b / (b * b - a));
      };
      exports.center = function center(values, inPlace) {
        if (typeof inPlace === 'undefined') inPlace = false;
        var result = values;
        if (!inPlace) result = [].concat(values);
        var theMean = exports.mean(result),
          l = result.length;
        for (var i = 0; i < l; i++) result[i] -= theMean;
      };
      exports.standardize = function standardize(values, standardDev, inPlace) {
        if (typeof standardDev === 'undefined') standardDev = exports.standardDeviation(values);
        if (typeof inPlace === 'undefined') inPlace = false;
        var l = values.length;
        var result = inPlace ? values : new Array(l);
        for (var i = 0; i < l; i++) result[i] = values[i] / standardDev;
        return result;
      };
      exports.cumulativeSum = function cumulativeSum(array) {
        var l = array.length;
        var result = new Array(l);
        result[0] = array[0];
        for (var i = 1; i < l; i++) result[i] = result[i - 1] + array[i];
        return result;
      };
    })(array$2);

    var matrix = {};

    (function (exports) {

      var arrayStat = array$2;
      function compareNumbers(a, b) {
        return a - b;
      }
      exports.max = function max(matrix) {
        var max = -Infinity;
        for (var i = 0; i < matrix.length; i++) {
          for (var j = 0; j < matrix[i].length; j++) {
            if (matrix[i][j] > max) max = matrix[i][j];
          }
        }
        return max;
      };
      exports.min = function min(matrix) {
        var min = Infinity;
        for (var i = 0; i < matrix.length; i++) {
          for (var j = 0; j < matrix[i].length; j++) {
            if (matrix[i][j] < min) min = matrix[i][j];
          }
        }
        return min;
      };
      exports.minMax = function minMax(matrix) {
        var min = Infinity;
        var max = -Infinity;
        for (var i = 0; i < matrix.length; i++) {
          for (var j = 0; j < matrix[i].length; j++) {
            if (matrix[i][j] < min) min = matrix[i][j];
            if (matrix[i][j] > max) max = matrix[i][j];
          }
        }
        return {
          min: min,
          max: max
        };
      };
      exports.entropy = function entropy(matrix, eps) {
        if (typeof eps === 'undefined') {
          eps = 0;
        }
        var sum = 0,
          l1 = matrix.length,
          l2 = matrix[0].length;
        for (var i = 0; i < l1; i++) {
          for (var j = 0; j < l2; j++) {
            sum += matrix[i][j] * Math.log(matrix[i][j] + eps);
          }
        }
        return -sum;
      };
      exports.mean = function mean(matrix, dimension) {
        if (typeof dimension === 'undefined') {
          dimension = 0;
        }
        var rows = matrix.length,
          cols = matrix[0].length,
          theMean,
          N,
          i,
          j;
        if (dimension === -1) {
          theMean = [0];
          N = rows * cols;
          for (i = 0; i < rows; i++) {
            for (j = 0; j < cols; j++) {
              theMean[0] += matrix[i][j];
            }
          }
          theMean[0] /= N;
        } else if (dimension === 0) {
          theMean = new Array(cols);
          N = rows;
          for (j = 0; j < cols; j++) {
            theMean[j] = 0;
            for (i = 0; i < rows; i++) {
              theMean[j] += matrix[i][j];
            }
            theMean[j] /= N;
          }
        } else if (dimension === 1) {
          theMean = new Array(rows);
          N = cols;
          for (j = 0; j < rows; j++) {
            theMean[j] = 0;
            for (i = 0; i < cols; i++) {
              theMean[j] += matrix[j][i];
            }
            theMean[j] /= N;
          }
        } else {
          throw new Error('Invalid dimension');
        }
        return theMean;
      };
      exports.sum = function sum(matrix, dimension) {
        if (typeof dimension === 'undefined') {
          dimension = 0;
        }
        var rows = matrix.length,
          cols = matrix[0].length,
          theSum,
          i,
          j;
        if (dimension === -1) {
          theSum = [0];
          for (i = 0; i < rows; i++) {
            for (j = 0; j < cols; j++) {
              theSum[0] += matrix[i][j];
            }
          }
        } else if (dimension === 0) {
          theSum = new Array(cols);
          for (j = 0; j < cols; j++) {
            theSum[j] = 0;
            for (i = 0; i < rows; i++) {
              theSum[j] += matrix[i][j];
            }
          }
        } else if (dimension === 1) {
          theSum = new Array(rows);
          for (j = 0; j < rows; j++) {
            theSum[j] = 0;
            for (i = 0; i < cols; i++) {
              theSum[j] += matrix[j][i];
            }
          }
        } else {
          throw new Error('Invalid dimension');
        }
        return theSum;
      };
      exports.product = function product(matrix, dimension) {
        if (typeof dimension === 'undefined') {
          dimension = 0;
        }
        var rows = matrix.length,
          cols = matrix[0].length,
          theProduct,
          i,
          j;
        if (dimension === -1) {
          theProduct = [1];
          for (i = 0; i < rows; i++) {
            for (j = 0; j < cols; j++) {
              theProduct[0] *= matrix[i][j];
            }
          }
        } else if (dimension === 0) {
          theProduct = new Array(cols);
          for (j = 0; j < cols; j++) {
            theProduct[j] = 1;
            for (i = 0; i < rows; i++) {
              theProduct[j] *= matrix[i][j];
            }
          }
        } else if (dimension === 1) {
          theProduct = new Array(rows);
          for (j = 0; j < rows; j++) {
            theProduct[j] = 1;
            for (i = 0; i < cols; i++) {
              theProduct[j] *= matrix[j][i];
            }
          }
        } else {
          throw new Error('Invalid dimension');
        }
        return theProduct;
      };
      exports.standardDeviation = function standardDeviation(matrix, means, unbiased) {
        var vari = exports.variance(matrix, means, unbiased),
          l = vari.length;
        for (var i = 0; i < l; i++) {
          vari[i] = Math.sqrt(vari[i]);
        }
        return vari;
      };
      exports.variance = function variance(matrix, means, unbiased) {
        if (typeof unbiased === 'undefined') {
          unbiased = true;
        }
        means = means || exports.mean(matrix);
        var rows = matrix.length;
        if (rows === 0) return [];
        var cols = matrix[0].length;
        var vari = new Array(cols);
        for (var j = 0; j < cols; j++) {
          var sum1 = 0,
            sum2 = 0,
            x = 0;
          for (var i = 0; i < rows; i++) {
            x = matrix[i][j] - means[j];
            sum1 += x;
            sum2 += x * x;
          }
          if (unbiased) {
            vari[j] = (sum2 - sum1 * sum1 / rows) / (rows - 1);
          } else {
            vari[j] = (sum2 - sum1 * sum1 / rows) / rows;
          }
        }
        return vari;
      };
      exports.median = function median(matrix) {
        var rows = matrix.length,
          cols = matrix[0].length;
        var medians = new Array(cols);
        for (var i = 0; i < cols; i++) {
          var data = new Array(rows);
          for (var j = 0; j < rows; j++) {
            data[j] = matrix[j][i];
          }
          data.sort(compareNumbers);
          var N = data.length;
          if (N % 2 === 0) {
            medians[i] = (data[N / 2] + data[N / 2 - 1]) * 0.5;
          } else {
            medians[i] = data[Math.floor(N / 2)];
          }
        }
        return medians;
      };
      exports.mode = function mode(matrix) {
        var rows = matrix.length,
          cols = matrix[0].length,
          modes = new Array(cols),
          i,
          j;
        for (i = 0; i < cols; i++) {
          var itemCount = new Array(rows);
          for (var k = 0; k < rows; k++) {
            itemCount[k] = 0;
          }
          var itemArray = new Array(rows);
          var count = 0;
          for (j = 0; j < rows; j++) {
            var index = itemArray.indexOf(matrix[j][i]);
            if (index >= 0) {
              itemCount[index]++;
            } else {
              itemArray[count] = matrix[j][i];
              itemCount[count] = 1;
              count++;
            }
          }
          var maxValue = 0,
            maxIndex = 0;
          for (j = 0; j < count; j++) {
            if (itemCount[j] > maxValue) {
              maxValue = itemCount[j];
              maxIndex = j;
            }
          }
          modes[i] = itemArray[maxIndex];
        }
        return modes;
      };
      exports.skewness = function skewness(matrix, unbiased) {
        if (typeof unbiased === 'undefined') unbiased = true;
        var means = exports.mean(matrix);
        var n = matrix.length,
          l = means.length;
        var skew = new Array(l);
        for (var j = 0; j < l; j++) {
          var s2 = 0,
            s3 = 0;
          for (var i = 0; i < n; i++) {
            var dev = matrix[i][j] - means[j];
            s2 += dev * dev;
            s3 += dev * dev * dev;
          }
          var m2 = s2 / n;
          var m3 = s3 / n;
          var g = m3 / Math.pow(m2, 3 / 2);
          if (unbiased) {
            var a = Math.sqrt(n * (n - 1));
            var b = n - 2;
            skew[j] = a / b * g;
          } else {
            skew[j] = g;
          }
        }
        return skew;
      };
      exports.kurtosis = function kurtosis(matrix, unbiased) {
        if (typeof unbiased === 'undefined') unbiased = true;
        var means = exports.mean(matrix);
        var n = matrix.length,
          m = matrix[0].length;
        var kurt = new Array(m);
        for (var j = 0; j < m; j++) {
          var s2 = 0,
            s4 = 0;
          for (var i = 0; i < n; i++) {
            var dev = matrix[i][j] - means[j];
            s2 += dev * dev;
            s4 += dev * dev * dev * dev;
          }
          var m2 = s2 / n;
          var m4 = s4 / n;
          if (unbiased) {
            var v = s2 / (n - 1);
            var a = n * (n + 1) / ((n - 1) * (n - 2) * (n - 3));
            var b = s4 / (v * v);
            var c = (n - 1) * (n - 1) / ((n - 2) * (n - 3));
            kurt[j] = a * b - 3 * c;
          } else {
            kurt[j] = m4 / (m2 * m2) - 3;
          }
        }
        return kurt;
      };
      exports.standardError = function standardError(matrix) {
        var samples = matrix.length;
        var standardDeviations = exports.standardDeviation(matrix);
        var l = standardDeviations.length;
        var standardErrors = new Array(l);
        var sqrtN = Math.sqrt(samples);
        for (var i = 0; i < l; i++) {
          standardErrors[i] = standardDeviations[i] / sqrtN;
        }
        return standardErrors;
      };
      exports.covariance = function covariance(matrix, dimension) {
        return exports.scatter(matrix, undefined, dimension);
      };
      exports.scatter = function scatter(matrix, divisor, dimension) {
        if (typeof dimension === 'undefined') {
          dimension = 0;
        }
        if (typeof divisor === 'undefined') {
          if (dimension === 0) {
            divisor = matrix.length - 1;
          } else if (dimension === 1) {
            divisor = matrix[0].length - 1;
          }
        }
        var means = exports.mean(matrix, dimension);
        var rows = matrix.length;
        if (rows === 0) {
          return [[]];
        }
        var cols = matrix[0].length,
          cov,
          i,
          j,
          s,
          k;
        if (dimension === 0) {
          cov = new Array(cols);
          for (i = 0; i < cols; i++) {
            cov[i] = new Array(cols);
          }
          for (i = 0; i < cols; i++) {
            for (j = i; j < cols; j++) {
              s = 0;
              for (k = 0; k < rows; k++) {
                s += (matrix[k][j] - means[j]) * (matrix[k][i] - means[i]);
              }
              s /= divisor;
              cov[i][j] = s;
              cov[j][i] = s;
            }
          }
        } else if (dimension === 1) {
          cov = new Array(rows);
          for (i = 0; i < rows; i++) {
            cov[i] = new Array(rows);
          }
          for (i = 0; i < rows; i++) {
            for (j = i; j < rows; j++) {
              s = 0;
              for (k = 0; k < cols; k++) {
                s += (matrix[j][k] - means[j]) * (matrix[i][k] - means[i]);
              }
              s /= divisor;
              cov[i][j] = s;
              cov[j][i] = s;
            }
          }
        } else {
          throw new Error('Invalid dimension');
        }
        return cov;
      };
      exports.correlation = function correlation(matrix) {
        var means = exports.mean(matrix),
          standardDeviations = exports.standardDeviation(matrix, true, means),
          scores = exports.zScores(matrix, means, standardDeviations),
          rows = matrix.length,
          cols = matrix[0].length,
          i,
          j;
        var cor = new Array(cols);
        for (i = 0; i < cols; i++) {
          cor[i] = new Array(cols);
        }
        for (i = 0; i < cols; i++) {
          for (j = i; j < cols; j++) {
            var c = 0;
            for (var k = 0, l = scores.length; k < l; k++) {
              c += scores[k][j] * scores[k][i];
            }
            c /= rows - 1;
            cor[i][j] = c;
            cor[j][i] = c;
          }
        }
        return cor;
      };
      exports.zScores = function zScores(matrix, means, standardDeviations) {
        means = means || exports.mean(matrix);
        if (typeof standardDeviations === 'undefined') standardDeviations = exports.standardDeviation(matrix, true, means);
        return exports.standardize(exports.center(matrix, means, false), standardDeviations, true);
      };
      exports.center = function center(matrix, means, inPlace) {
        means = means || exports.mean(matrix);
        var result = matrix,
          l = matrix.length,
          i,
          j,
          jj;
        if (!inPlace) {
          result = new Array(l);
          for (i = 0; i < l; i++) {
            result[i] = new Array(matrix[i].length);
          }
        }
        for (i = 0; i < l; i++) {
          var row = result[i];
          for (j = 0, jj = row.length; j < jj; j++) {
            row[j] = matrix[i][j] - means[j];
          }
        }
        return result;
      };
      exports.standardize = function standardize(matrix, standardDeviations, inPlace) {
        if (typeof standardDeviations === 'undefined') standardDeviations = exports.standardDeviation(matrix);
        var result = matrix,
          l = matrix.length,
          i,
          j,
          jj;
        if (!inPlace) {
          result = new Array(l);
          for (i = 0; i < l; i++) {
            result[i] = new Array(matrix[i].length);
          }
        }
        for (i = 0; i < l; i++) {
          var resultRow = result[i];
          var sourceRow = matrix[i];
          for (j = 0, jj = resultRow.length; j < jj; j++) {
            if (standardDeviations[j] !== 0 && !isNaN(standardDeviations[j])) {
              resultRow[j] = sourceRow[j] / standardDeviations[j];
            }
          }
        }
        return result;
      };
      exports.weightedVariance = function weightedVariance(matrix, weights) {
        var means = exports.mean(matrix);
        var rows = matrix.length;
        if (rows === 0) return [];
        var cols = matrix[0].length;
        var vari = new Array(cols);
        for (var j = 0; j < cols; j++) {
          var sum = 0;
          var a = 0,
            b = 0;
          for (var i = 0; i < rows; i++) {
            var z = matrix[i][j] - means[j];
            var w = weights[i];
            sum += w * (z * z);
            b += w;
            a += w * w;
          }
          vari[j] = sum * (b / (b * b - a));
        }
        return vari;
      };
      exports.weightedMean = function weightedMean(matrix, weights, dimension) {
        if (typeof dimension === 'undefined') {
          dimension = 0;
        }
        var rows = matrix.length;
        if (rows === 0) return [];
        var cols = matrix[0].length,
          means,
          i,
          ii,
          j,
          w,
          row;
        if (dimension === 0) {
          means = new Array(cols);
          for (i = 0; i < cols; i++) {
            means[i] = 0;
          }
          for (i = 0; i < rows; i++) {
            row = matrix[i];
            w = weights[i];
            for (j = 0; j < cols; j++) {
              means[j] += row[j] * w;
            }
          }
        } else if (dimension === 1) {
          means = new Array(rows);
          for (i = 0; i < rows; i++) {
            means[i] = 0;
          }
          for (j = 0; j < rows; j++) {
            row = matrix[j];
            w = weights[j];
            for (i = 0; i < cols; i++) {
              means[j] += row[i] * w;
            }
          }
        } else {
          throw new Error('Invalid dimension');
        }
        var weightSum = arrayStat.sum(weights);
        if (weightSum !== 0) {
          for (i = 0, ii = means.length; i < ii; i++) {
            means[i] /= weightSum;
          }
        }
        return means;
      };
      exports.weightedCovariance = function weightedCovariance(matrix, weights, means, dimension) {
        dimension = dimension || 0;
        means = means || exports.weightedMean(matrix, weights, dimension);
        var s1 = 0,
          s2 = 0;
        for (var i = 0, ii = weights.length; i < ii; i++) {
          s1 += weights[i];
          s2 += weights[i] * weights[i];
        }
        var factor = s1 / (s1 * s1 - s2);
        return exports.weightedScatter(matrix, weights, means, factor, dimension);
      };
      exports.weightedScatter = function weightedScatter(matrix, weights, means, factor, dimension) {
        dimension = dimension || 0;
        means = means || exports.weightedMean(matrix, weights, dimension);
        if (typeof factor === 'undefined') {
          factor = 1;
        }
        var rows = matrix.length;
        if (rows === 0) {
          return [[]];
        }
        var cols = matrix[0].length,
          cov,
          i,
          j,
          k,
          s;
        if (dimension === 0) {
          cov = new Array(cols);
          for (i = 0; i < cols; i++) {
            cov[i] = new Array(cols);
          }
          for (i = 0; i < cols; i++) {
            for (j = i; j < cols; j++) {
              s = 0;
              for (k = 0; k < rows; k++) {
                s += weights[k] * (matrix[k][j] - means[j]) * (matrix[k][i] - means[i]);
              }
              cov[i][j] = s * factor;
              cov[j][i] = s * factor;
            }
          }
        } else if (dimension === 1) {
          cov = new Array(rows);
          for (i = 0; i < rows; i++) {
            cov[i] = new Array(rows);
          }
          for (i = 0; i < rows; i++) {
            for (j = i; j < rows; j++) {
              s = 0;
              for (k = 0; k < cols; k++) {
                s += weights[k] * (matrix[j][k] - means[j]) * (matrix[i][k] - means[i]);
              }
              cov[i][j] = s * factor;
              cov[j][i] = s * factor;
            }
          }
        } else {
          throw new Error('Invalid dimension');
        }
        return cov;
      };
    })(matrix);

    var array$1 = mlStat.array = array$2;
    mlStat.matrix = matrix;

    function normalize(array) {
      const min = array$1.min(array[1]);
      const max = array$1.max(array[1]);
      const sum = array$1.sum(array[1]);
      const length = array[1] ? array[1].length : 0;
      if (sum !== 0) {
        for (let i = 0; i < length; i++) {
          array[1][i] /= sum;
        }
      }
      return {
        sum,
        min,
        max
      };
    }

    // this method will systematically take care of both array
    function commonExtractAndNormalize(array1, array2, width, from, to, common) {
      if (!Array.isArray(array1) || !Array.isArray(array2)) {
        return {
          info: undefined,
          data: undefined
        };
      }
      const extract1 = extract(array1, from, to);
      const extract2 = extract(array2, from, to);
      let common1, common2, info1, info2;
      if (common & COMMON_SECOND) {
        common1 = getCommonArray(extract1, extract2, width);
        info1 = normalize(common1);
      } else {
        common1 = extract1;
        info1 = normalize(common1);
      }
      if (common & COMMON_FIRST) {
        common2 = getCommonArray(extract2, extract1, width);
        info2 = normalize(common2);
      } else {
        common2 = extract2;
        info2 = normalize(common2);
      }
      return {
        info1,
        info2,
        data1: common1,
        data2: common2
      };
    }

    function extractAndNormalize(array, from, to) {
      if (!Array.isArray(array)) {
        return {
          info: undefined,
          data: undefined
        };
      }
      const newArray = extract(array, from, to);
      const info = normalize(newArray);
      return {
        info,
        data: newArray
      };
    }

    // Adapted from: http://stackoverflow.com/questions/563198/how-do-you-detect-where-two-line-segments-intersect/1968345#1968345
    function getIntersection(segment1, segment2) {
      const p0X = segment1[0][0];
      const p0Y = segment1[0][1];
      const p1X = segment1[1][0];
      const p1Y = segment1[1][1];
      const p2X = segment2[0][0];
      const p2Y = segment2[0][1];
      const p3X = segment2[1][0];
      const p3Y = segment2[1][1];
      const s1X = p1X - p0X;
      const s1Y = p1Y - p0Y;
      const s2X = p3X - p2X;
      const s2Y = p3Y - p2Y;
      const s = (-s1Y * (p0X - p2X) + s1X * (p0Y - p2Y)) / (-s2X * s1Y + s1X * s2Y);
      const t = (s2X * (p0Y - p2Y) - s2Y * (p0X - p2X)) / (-s2X * s1Y + s1X * s2Y);
      if (s >= 0 && s <= 1 && t >= 0 && t <= 1) {
        return {
          x: p0X + t * s1X,
          y: p0Y + t * s1Y
        };
      }
      return null; // No collision
    }

    const COMMON_NO = 0;
    const COMMON_FIRST = 1;
    const COMMON_SECOND = 2;
    const COMMON_BOTH = 3;
    /**
     * A number, or a string containing a number.
     * @typedef {([number[],number[]]|[number,number][]|{x:number[],y:number[]})} Peaks
     */
    /**
     * Create a comparator class
     * {object} [options={}]
     * {string} [options.common=''] should we take only common peaks 'first', 'second', 'both', ''
     * {number} [options.widthBottom=2] bottom trapezoid width for similarity evaluation
     * {number} [options.widthTop=1] top trapezoid width for similarity evaluation
     * {number} [options.from] from region used for similarity calculation
     * {number} [options.to] to region used for similarity calculation
     */
    class Comparator {
      constructor(options = {}) {
        this.array1 = [];
        this.array2 = [];
        this.setOptions(options);
      }
      /*
         2 formats are allowed:
         [[x1,x2,...],[y1,y2,...]] or [[x1,y1],[x2,y2], ...]
        */
      setOptions(options = {}) {
        if (typeof options.common === 'string') {
          if (options.common.toLowerCase() === 'first') {
            this.common = COMMON_FIRST;
          } else if (options.common.toLowerCase() === 'second') {
            this.common = COMMON_SECOND;
          } else if (options.common.toLowerCase() === 'both') {
            this.common = COMMON_BOTH;
          } else {
            this.common = COMMON_NO;
          }
        } else if (options.common === true) {
          this.common = COMMON_BOTH;
        } else {
          this.common = COMMON_NO;
        }
        this.trapezoid = options.trapezoid;
        this.commonFactor = options.commonFactor || this.commonFactor || 4;
        const {
          widthBottom = this.widthBottom || 2,
          widthTop = this.widthTop || 1,
          from = this.from,
          to = this.to
        } = options;
        this.setTrapezoid(widthBottom, widthTop);
        this.setFromTo(from, to);
      }
      /**
       *
       * @param {Peaks} peaks
       */
      setPeaks1(peaks) {
        this.array1 = checkPeaks(peaks);
        if (this.common) {
          const extracts = commonExtractAndNormalize(this.array1, this.array2, this.widthBottom, this.from, this.to, this.common);
          this.array1Extract = extracts.data1;
          this.array1ExtractInfo = extracts.info1;
          this.array2Extract = extracts.data2;
          this.array2ExtractInfo = extracts.info2;
        } else {
          const extract = extractAndNormalize(this.array1, this.from, this.to);
          this.array1Extract = extract.data;
          this.array1ExtractInfo = extract.info;
        }
      }
      /**
       *
       * @param {Peaks} peaks
       */
      setPeaks2(peaks) {
        this.array2 = checkPeaks(peaks);
        if (this.common) {
          const extracts = commonExtractAndNormalize(this.array1, this.array2, this.widthBottom, this.from, this.to, this.common);
          this.array1Extract = extracts.data1;
          this.array1ExtractInfo = extracts.info1;
          this.array2Extract = extracts.data2;
          this.array2ExtractInfo = extracts.info2;
        } else {
          const extract = extractAndNormalize(this.array2, this.from, this.to);
          this.array2Extract = extract.data;
          this.array2ExtractInfo = extract.info;
        }
      }
      getExtract1() {
        return this.array1Extract;
      }
      getExtract2() {
        return this.array2Extract;
      }
      getExtractInfo1() {
        return this.array1ExtractInfo;
      }
      getExtractInfo2() {
        return this.array2ExtractInfo;
      }
      /**
       * Set the new bottom and top width of the trapezoid
       * @param {number} newWidthBottom
       * @param {number} newWidthTop
       */
      setTrapezoid(newWidthBottom, newWidthTop) {
        this.widthTop = newWidthTop;
        this.widthBottom = newWidthBottom;
        this.widthSlope = (this.widthBottom - this.widthTop) / 2;
        if (this.widthBottom < this.widthTop) {
          throw new Error('widthBottom has to be larger than widthTop');
        }
      }
      /**
       * Set the from / to for comparison
       * @param {number} newFrom - set the new from value
       * @param {number} newTo - set the new to value
       * @returns
       */
      setFromTo(newFrom, newTo) {
        if (newFrom === this.from && newTo === this.to) return;
        this.from = newFrom;
        this.to = newTo;
        if (this.common) {
          const extracts = commonExtractAndNormalize(this.array1, this.array2, this.widthBottom, this.from, this.to, this.common, this.commonFactor);
          this.array1Extract = extracts.data1;
          this.array1ExtractInfo = extracts.info1;
          this.array2Extract = extracts.data2;
          this.array2ExtractInfo = extracts.info2;
        } else {
          let extract1 = extractAndNormalize(this.array1, this.from, this.to);
          this.array1Extract = extract1.data;
          this.array1ExtractInfo = extract1.info;
          let extract2 = extractAndNormalize(this.array2, this.from, this.to);
          this.array2Extract = extract2.data;
          this.array2ExtractInfo = extract2.info;
        }
      }
      /**
       *
       * @param {number} x1
       * @param {number} y1
       * @param {number} x2
       * @param {number} y2
       * @returns
       */
      getOverlap(x1, y1, x2, y2) {
        if (y1 === 0 || y2 === 0) return 0;
        // TAKE CARE !!! We multiply the diff by 2 !!!
        const diff = Math.abs(x1 - x2) * 2;
        if (diff > this.widthBottom) return 0;
        if (diff <= this.widthTop) {
          return Math.min(y1, y2);
        }
        const maxValue = Math.max(y1, y2) * (this.widthBottom - diff) / (this.widthBottom - this.widthTop);
        return Math.min(y1, y2, maxValue);
      }
      /**
       * This is the old trapezoid similarity
       * @param {number} x1
       * @param {number} y1
       * @param {number} x2
       * @param {number} y2
       * @param {number} widthTop
       * @param {number} widthBottom
       * @returns
       */
      getOverlapTrapezoid(x1, y1, x2, y2, widthTop, widthBottom) {
        // eslint-disable-next-line no-console
        console.error('getOverlapTrapezoid should not be used anymore');
        const factor = 2 / (widthTop + widthBottom); // correction for surface=1
        if (y1 === 0 || y2 === 0) return 0;
        if (x1 === x2) {
          // they have the same position
          return Math.min(y1, y2);
        }
        const diff = Math.abs(x1 - x2);
        if (diff >= widthBottom) return 0;
        if (y1 === y2) {
          // do they have the same height ???
          // we need to find the common length
          if (diff <= widthTop) {
            return ((widthTop + widthBottom) / 2 - diff) * y1 * factor;
          } else if (diff <= widthBottom) {
            return (widthBottom - diff) * y1 / 2 * (diff - widthTop) / (widthBottom - widthTop) * factor;
          }
          return 0;
        } else {
          // the height are different and not the same position ...
          // we need to consider only one segment to find its intersection
          const small = Math.min(y1, y2);
          const big = Math.max(y1, y2);
          const targets = [[[0, 0], [this.widthSlope, small]], [[this.widthSlope, small], [this.widthSlope + widthTop, small]], [[widthTop + this.widthSlope, small], [widthBottom, 0]]];
          let segment;
          if (x1 > x2 && y1 > y2 || x1 < x2 && y1 < y2) {
            segment = [[diff, 0], [diff + this.widthSlope, big]];
          } else {
            segment = [[diff + this.widthSlope, big], [diff, 0]];
          }
          for (let i = 0; i < 3; i++) {
            const intersection = getIntersection(targets[i], segment);
            if (intersection) {
              switch (i) {
                case 0:
                  return small - diff * intersection.y / 2 * factor;
                case 1:
                  // to simplify ...
                  //     console.log("           ",widthSlope,small,big,intersection.x)
                  return (this.widthSlope * small / (2 * big) * small + (widthTop + this.widthSlope - intersection.x) * small + this.widthSlope * small / 2) * factor;
                case 2:
                  return (widthBottom - diff) * intersection.y / 2 * factor;
                default:
                  throw new Error(`unexpected intersection value: ${i}`);
              }
            }
          }
        }
        return NaN;
      }
      /**
       * This method calculates the total diff. The sum of positive value will yield to overlap
       * @returns
       */
      calculateDiff() {
        // we need to take 2 pointers
        // and travel progressively between them ...
        const newFirst = [this.array1Extract[0].slice(), this.array1Extract[1].slice()];
        const newSecond = [this.array2Extract[0].slice(), this.array2Extract[1].slice()];
        const array1Length = this.array1Extract[0] ? this.array1Extract[0].length : 0;
        const array2Length = this.array2Extract[0] ? this.array2Extract[0].length : 0;
        let pos1 = 0;
        let pos2 = 0;
        let previous2 = 0;
        while (pos1 < array1Length) {
          const diff = newFirst[0][pos1] - this.array2Extract[0][pos2];
          if (Math.abs(diff) < this.widthBottom) {
            // there is some overlap
            let overlap;
            if (this.trapezoid) {
              // old trapezoid overlap similarity
              overlap = this.getOverlapTrapezoid(newFirst[0][pos1], newFirst[1][pos1], newSecond[0][pos2], newSecond[1][pos2], this.widthTop, this.widthBottom);
            } else {
              overlap = this.getOverlap(newFirst[0][pos1], newFirst[1][pos1], newSecond[0][pos2], newSecond[1][pos2], this.widthTop, this.widthBottom);
            }
            newFirst[1][pos1] -= overlap;
            newSecond[1][pos2] -= overlap;
            if (pos2 < array2Length - 1) {
              pos2++;
            } else {
              pos1++;
              pos2 = previous2;
            }
          } else if (diff > 0 && pos2 < array2Length - 1) {
            pos2++;
            previous2 = pos2;
          } else {
            pos1++;
            pos2 = previous2;
          }
        }
        return newSecond;
      }
      /**
       * Set the new peaks and return info
       * @param {Peaks} newPeaks1
       * @param {Peaks} newPeaks2
       * @returns
       */
      getSimilarity(newPeaks1, newPeaks2) {
        if (newPeaks1) this.setPeaks1(newPeaks1);
        if (newPeaks2) this.setPeaks2(newPeaks2);
        const diff = this.calculateDiff();
        return {
          diff,
          extract1: this.getExtract1(),
          extract2: this.getExtract2(),
          extractInfo1: this.getExtractInfo1(),
          extractInfo2: this.getExtractInfo2(),
          similarity: calculateOverlapFromDiff(diff),
          widthBottom: this.widthBottom,
          widthTop: this.widthTop
        };
      }
      /**
       * This works mainly when you have a array1 that is fixed
       * newPeaks2 have to be normalized ! (sum to 1)
       * @param {Peaks} newPeaks2
       * @param {number} from
       * @param {number} to
       * @returns
       */
      fastSimilarity(newPeaks2, from, to) {
        this.array1Extract = extract(this.array1, from, to);
        this.array2Extract = newPeaks2;
        if (this.common & COMMON_SECOND) {
          this.array1Extract = getCommonArray(this.array1Extract, this.array2Extract, this.widthBottom);
        }
        normalize(this.array1Extract);
        const diff = this.calculateDiff();
        return calculateOverlapFromDiff(diff);
      }
    }

    /**
     * @param {object}   [options={}]
     * @param {object}   [options.minCharge=1]
     * @param {object}   [options.maxCharge=10]
     * @param {object}   [options.similarity={}]
     * @param {object}   [options.similarity.widthBottom]
     * @param {object}   [options.similarity.widthTop]
     * @param {object}   [options.similarity.widthFunction] - function called with mass that should return an object width containing top and bottom
     * @param {object}   [options.similarity.zone={}]
     * @param {object}   [options.similarity.zone.low=-0.5] - window shift based on observed monoisotopic mass
     * @param {object}   [options.similarity.zone.high=2.5] - to value for the comparison window
     * @param {object}   [options.similarity.common]
     */

    const NEUTRON_MASS$1 = 1;
    function getPeakChargeBySimilarity(spectrum, targetMass, options = {}) {
      let {
        similarity = {},
        minCharge = 1,
        maxCharge = 10
      } = options;
      let {
        zone = {},
        widthFunction
      } = similarity;
      let {
        low = -0.5,
        high = 2.5
      } = zone;
      if (!spectrum || !spectrum.data.x.length > 0) {
        throw Error('You need to add an experimental spectrum first using setMassSpectrum');
      }
      let width = {
        bottom: similarity.widthBottom,
        top: similarity.widthTop
      };
      similarity = JSON.parse(JSON.stringify(similarity));
      similarity.common = 'second';
      let experimentalData = spectrum.data;
      let similarityProcessor = new Comparator(similarity);
      similarityProcessor.setPeaks1([experimentalData.x, experimentalData.y]);
      if (widthFunction && typeof widthFunction === 'string') {
        // eslint-disable-next-line no-new-func
        widthFunction = new Function('mass', widthFunction);
        let checkTopBottom = widthFunction(123);
        if (!checkTopBottom.bottom || !checkTopBottom.top) {
          throw Error('widthFunction should return an object with bottom and top properties');
        }
      }
      let fromCharge = minCharge * maxCharge > 0 ? Math.round(Math.min(Math.abs(minCharge), Math.abs(maxCharge))) : 1;
      let toCharge = Math.round(Math.max(Math.abs(minCharge), Math.abs(maxCharge)));
      let fromIsotope = Math.ceil(low);
      let toIsotope = Math.floor(high);
      let isotopeHeight = 1 / (toIsotope - fromIsotope + 1);
      let results = [];
      for (let charge = fromCharge; charge < toCharge + 1; charge++) {
        let isotopePositions = {
          x: [],
          y: []
        };
        for (let isotopePosition = fromIsotope; isotopePosition < toIsotope + 1; isotopePosition++) {
          isotopePositions.x.push(targetMass + isotopePosition * NEUTRON_MASS$1 / charge);
          isotopePositions.y.push(isotopeHeight);
        }
        let from = targetMass + low / Math.abs(charge);
        let to = targetMass + high / Math.abs(charge);
        similarityProcessor.setFromTo(from, to);
        if (widthFunction) {
          width = widthFunction(targetMass);
          similarityProcessor.setTrapezoid(width.bottom, width.top);
        }
        similarityProcessor.setPeaks2([isotopePositions.x, isotopePositions.y]);
        let result = similarityProcessor.getSimilarity();
        results.push({
          charge,
          similarity: result.similarity
        });
      }
      return results.sort((a, b) => b.similarity - a.similarity)[0].charge;
    }

    /**
     * When a spectrum is continous ?
     * - has more than 100 points
     * - deltaX change can not be more than a factor 2
     * - deltaX may not be larger than 0.1
     * - if y is zero it does not count
     * @param {object} spectrum
     * @param {object} [options={}]
     * @param {number} [options.minLength=100]
     * @param {number} [options.relativeHeightThreshold=0.001] // Under this value the
     * @param {number} [options.maxDeltaRatio=3]
     */

    function isContinuous(spectrum, options = {}) {
      const {
        minLength = 100,
        maxDeltaRatio = 3,
        relativeHeightThreshold = 0.001
      } = options;
      const minHeight = xMaxValue(spectrum.data.y) * relativeHeightThreshold;
      const minRadio = 1 / maxDeltaRatio;
      const maxRatio = 1 * maxDeltaRatio;
      if (spectrum.continuous === undefined) {
        let xs = spectrum.data.x;
        let ys = spectrum.data.y;
        if (xs.length < minLength) {
          spectrum.continuous = false;
        } else {
          let previousDelta = xs[1] - xs[0];
          spectrum.continuous = true;
          let success = 0;
          let failed = 0;
          for (let i = 0; i < xs.length - 1; i++) {
            if (ys[i] < minHeight || ys[i + 1] < minHeight) {
              previousDelta = 0;
              continue;
            }
            let delta = xs[i + 1] - xs[i];
            if (previousDelta) {
              let ratio = delta / previousDelta;
              if ((Math.abs(delta) > 0.1 || ratio < minRadio || ratio > maxRatio) && ys[i] !== 0 && ys[i + 1] !== 0) {
                failed++;
              } else {
                success++;
              }
            }
            previousDelta = delta;
          }
          if (success / failed < 10) {
            spectrum.continuous = false;
          }
        }
      }
      return spectrum.continuous;
    }

    var IDX = 256,
      HEX = [],
      BUFFER;
    while (IDX--) HEX[IDX] = (IDX + 256).toString(16).substring(1);
    function v4() {
      var i = 0,
        num,
        out = '';
      if (!BUFFER || IDX + 16 > 256) {
        BUFFER = Array(i = 256);
        while (i--) BUFFER[i] = 256 * Math.random() | 0;
        i = IDX = 0;
      }
      for (; i < 16; i++) {
        num = BUFFER[IDX + i];
        if (i == 6) out += HEX[num & 15 | 64];else if (i == 8) out += HEX[num & 63 | 128];else out += HEX[num];
        if (i & 1 && i > 1 && i < 11) out += '-';
      }
      IDX++;
      return out;
    }

    /**
     * Apply Savitzky Golay algorithm
     * @param [ys] Array of y values
     * @param [xs] Array of X or deltaX
     * @return  Array containing the new ys (same length)
     */
    function sgg(ys, xs, options = {}) {
      let {
        windowSize = 9,
        derivative = 0,
        polynomial = 3
      } = options;
      if (windowSize % 2 === 0 || windowSize < 5 || !Number.isInteger(windowSize)) {
        throw new RangeError('Invalid window size (should be odd and at least 5 integer number)');
      }
      if (!isAnyArray$1(ys)) {
        throw new TypeError('Y values must be an array');
      }
      if (typeof xs === 'undefined') {
        throw new TypeError('X must be defined');
      }
      if (windowSize > ys.length) {
        throw new RangeError(`Window size is higher than the data length ${windowSize}>${ys.length}`);
      }
      if (derivative < 0 || !Number.isInteger(derivative)) {
        throw new RangeError('Derivative should be a positive integer');
      }
      if (polynomial < 1 || !Number.isInteger(polynomial)) {
        throw new RangeError('Polynomial should be a positive integer');
      }
      if (polynomial >= 6) {
        // eslint-disable-next-line no-console
        console.warn('You should not use polynomial grade higher than 5 if you are' + ' not sure that your data arises from such a model. Possible polynomial oscillation problems');
      }
      let half = Math.floor(windowSize / 2);
      let np = ys.length;
      let ans = new Float64Array(np);
      let weights = fullWeights(windowSize, polynomial, derivative);
      let hs = 0;
      let constantH = true;
      if (isAnyArray$1(xs)) {
        constantH = false;
      } else {
        hs = Math.pow(xs, derivative);
      }
      //For the borders
      for (let i = 0; i < half; i++) {
        let wg1 = weights[half - i - 1];
        let wg2 = weights[half + i + 1];
        let d1 = 0;
        let d2 = 0;
        for (let l = 0; l < windowSize; l++) {
          d1 += wg1[l] * ys[l];
          d2 += wg2[l] * ys[np - windowSize + l];
        }
        if (constantH) {
          ans[half - i - 1] = d1 / hs;
          ans[np - half + i] = d2 / hs;
        } else {
          hs = getHs(xs, half - i - 1, half, derivative);
          ans[half - i - 1] = d1 / hs;
          hs = getHs(xs, np - half + i, half, derivative);
          ans[np - half + i] = d2 / hs;
        }
      }
      //For the internal points
      let wg = weights[half];
      for (let i = windowSize; i <= np; i++) {
        let d = 0;
        for (let l = 0; l < windowSize; l++) d += wg[l] * ys[l + i - windowSize];
        if (!constantH) {
          hs = getHs(xs, i - half - 1, half, derivative);
        }
        ans[i - half - 1] = d / hs;
      }
      return ans;
    }
    function getHs(h, center, half, derivative) {
      let hs = 0;
      let count = 0;
      for (let i = center - half; i < center + half; i++) {
        if (i >= 0 && i < h.length - 1) {
          hs += h[i + 1] - h[i];
          count++;
        }
      }
      return Math.pow(hs / count, derivative);
    }
    function gramPoly(i, m, k, s) {
      let Grampoly = 0;
      if (k > 0) {
        Grampoly = (4 * k - 2) / (k * (2 * m - k + 1)) * (i * gramPoly(i, m, k - 1, s) + s * gramPoly(i, m, k - 1, s - 1)) - (k - 1) * (2 * m + k) / (k * (2 * m - k + 1)) * gramPoly(i, m, k - 2, s);
      } else {
        if (k === 0 && s === 0) {
          Grampoly = 1;
        } else {
          Grampoly = 0;
        }
      }
      return Grampoly;
    }
    function genFact(a, b) {
      let gf = 1;
      if (a >= b) {
        for (let j = a - b + 1; j <= a; j++) {
          gf *= j;
        }
      }
      return gf;
    }
    function weight(i, t, m, n, s) {
      let sum = 0;
      for (let k = 0; k <= n; k++) {
        sum += (2 * k + 1) * (genFact(2 * m, k) / genFact(2 * m + k + 1, k + 1)) * gramPoly(i, m, k, 0) * gramPoly(t, m, k, s);
      }
      return sum;
    }
    /**
     * @private
     * @param m  Number of points
     * @param n  Polynomial grade
     * @param s  Derivative
     */
    function fullWeights(m, n, s) {
      let weights = new Array(m);
      let np = Math.floor(m / 2);
      for (let t = -np; t <= np; t++) {
        weights[t + np] = new Float64Array(m);
        for (let j = -np; j <= np; j++) {
          weights[t + np][j + np] = weight(j, t, np, n, s);
        }
      }
      return weights;
    }

    /**
     * Correction of the x and y coordinates using a quadratic optimizations with the peak and its 3 closest neighbors to determine the true x,y values of the peak.
     * This process is done in place and is very fast.
     * @param data
     * @param peaks
     */
    function optimizeTop(data, peaks) {
      const {
        x,
        y
      } = data;
      for (const peak of peaks) {
        let currentIndex = peak.index;
        // The detected peak could be moved 1 or 2 units to left or right.
        if (y[currentIndex - 1] >= y[currentIndex - 2] && y[currentIndex - 1] >= y[currentIndex]) {
          currentIndex--;
        } else if (y[currentIndex + 1] >= y[currentIndex] && y[currentIndex + 1] >= y[currentIndex + 2]) {
          currentIndex++;
        } else if (y[currentIndex - 2] >= y[currentIndex - 3] && y[currentIndex - 2] >= y[currentIndex - 1]) {
          currentIndex -= 2;
        } else if (y[currentIndex + 2] >= y[currentIndex + 1] && y[currentIndex + 2] >= y[currentIndex + 3]) {
          currentIndex += 2;
        }
        // interpolation to a sin() function
        if (y[currentIndex - 1] > 0 && y[currentIndex + 1] > 0 && y[currentIndex] >= y[currentIndex - 1] && y[currentIndex] >= y[currentIndex + 1] && (y[currentIndex] !== y[currentIndex - 1] || y[currentIndex] !== y[currentIndex + 1])) {
          let alpha = 20 * Math.log10(y[currentIndex - 1]);
          let beta = 20 * Math.log10(y[currentIndex]);
          let gamma = 20 * Math.log10(y[currentIndex + 1]);
          let p = 0.5 * (alpha - gamma) / (alpha - 2 * beta + gamma);
          peak.x = x[currentIndex] + (x[currentIndex] - x[currentIndex - 1]) * p;
          peak.y = y[currentIndex] - 0.25 * (y[currentIndex - 1] - y[currentIndex + 1]) * p;
        }
      }
    }

    /**
     * Global spectra deconvolution
     * @param  data - Object data with x and y arrays. Values in x has to be growing
     * @param {number} [options.broadRatio = 0.00] - If `broadRatio` is higher than 0, then all the peaks which second derivative
     * smaller than `broadRatio * maxAbsSecondDerivative` will be marked with the soft mask equal to true.

     */
    function gsd(data, options = {}) {
      let {
        sgOptions = {
          windowSize: 9,
          polynomial: 3
        },
        noiseLevel,
        smoothY = false,
        maxCriteria = true,
        minMaxRatio = 0.00025,
        realTopDetection = false
      } = options;
      let {
        x,
        y
      } = data;
      if (xIsMonotonic(x) !== 1) {
        throw new Error('GSD only accepts monotone increasing x values');
      }
      //rescale;
      y = y.slice();
      // If the max difference between delta x is less than 5%, then,
      // we can assume it to be equally spaced variable
      let equallySpaced = xIsEquallySpaced(x);
      if (noiseLevel === undefined) {
        if (equallySpaced) {
          const noiseInfo = xNoiseStandardDeviation(y);
          if (maxCriteria) {
            noiseLevel = noiseInfo.median + 1.5 * noiseInfo.sd;
          } else {
            noiseLevel = -noiseInfo.median + 1.5 * noiseInfo.sd;
          }
        } else {
          noiseLevel = 0;
        }
      } else if (!maxCriteria) {
        noiseLevel *= -1;
      }
      if (!maxCriteria) {
        for (let i = 0; i < y.length; i++) {
          y[i] *= -1;
        }
      }
      if (noiseLevel !== undefined) {
        for (let i = 0; i < y.length; i++) {
          if (y[i] < noiseLevel) {
            y[i] = noiseLevel;
          }
        }
      }
      let yData = y;
      let dY, ddY;
      const {
        windowSize,
        polynomial
      } = sgOptions;
      if (equallySpaced) {
        if (smoothY) {
          yData = sgg(y, x[1] - x[0], {
            windowSize,
            polynomial,
            derivative: 0
          });
        }
        dY = sgg(y, x[1] - x[0], {
          windowSize,
          polynomial,
          derivative: 1
        });
        ddY = sgg(y, x[1] - x[0], {
          windowSize,
          polynomial,
          derivative: 2
        });
      } else {
        if (smoothY) {
          yData = sgg(y, x, {
            windowSize,
            polynomial,
            derivative: 0
          });
        }
        dY = sgg(y, x, {
          windowSize,
          polynomial,
          derivative: 1
        });
        ddY = sgg(y, x, {
          windowSize,
          polynomial,
          derivative: 2
        });
      }
      const minY = xMinValue(yData);
      const maxY = xMaxValue(yData);
      if (minY > maxY || minY === maxY) return [];
      const yThreshold = minY + (maxY - minY) * minMaxRatio;
      const dX = x[1] - x[0];
      let lastMax = null;
      let lastMin = null;
      let minddY = [];
      let intervalL = [];
      let intervalR = [];
      // By the intermediate value theorem We cannot find 2 consecutive maximum or minimum
      for (let i = 1; i < yData.length - 1; ++i) {
        if (dY[i] < dY[i - 1] && dY[i] <= dY[i + 1] || dY[i] <= dY[i - 1] && dY[i] < dY[i + 1]) {
          lastMin = {
            x: x[i],
            index: i
          };
          if (dX > 0 && lastMax !== null) {
            intervalL.push(lastMax);
            intervalR.push(lastMin);
          }
        }
        // Maximum in first derivative
        if (dY[i] >= dY[i - 1] && dY[i] > dY[i + 1] || dY[i] > dY[i - 1] && dY[i] >= dY[i + 1]) {
          lastMax = {
            x: x[i],
            index: i
          };
          if (dX < 0 && lastMin !== null) {
            intervalL.push(lastMax);
            intervalR.push(lastMin);
          }
        }
        // Minimum in second derivative
        if (ddY[i] < ddY[i - 1] && ddY[i] < ddY[i + 1]) {
          minddY.push(i);
        }
      }
      let lastK = -1;
      const peaks = [];
      for (const minddYIndex of minddY) {
        let deltaX = x[minddYIndex];
        let possible = -1;
        let k = lastK + 1;
        let minDistance = Number.POSITIVE_INFINITY;
        let currentDistance = 0;
        while (possible === -1 && k < intervalL.length) {
          currentDistance = Math.abs(deltaX - (intervalL[k].x + intervalR[k].x) / 2);
          if (currentDistance < (intervalR[k].x - intervalL[k].x) / 2) {
            possible = k;
            lastK = k;
          }
          ++k;
          // Not getting closer?
          if (currentDistance >= minDistance) {
            break;
          }
          minDistance = currentDistance;
        }
        if (possible !== -1) {
          if (yData[minddYIndex] > yThreshold) {
            let width = Math.abs(intervalR[possible].x - intervalL[possible].x);
            peaks.push({
              id: v4(),
              x: deltaX,
              y: yData[minddYIndex],
              width,
              index: minddYIndex,
              ddY: ddY[minddYIndex],
              inflectionPoints: {
                from: intervalL[possible],
                to: intervalR[possible]
              }
            });
          }
        }
      }
      if (realTopDetection) {
        optimizeTop({
          x,
          y: yData
        }, peaks);
      }
      peaks.forEach(peak => {
        if (!maxCriteria) {
          peak.y *= -1;
          peak.ddY = peak.ddY * -1;
        }
      });
      peaks.sort((a, b) => {
        return a.x - b.x;
      });
      return peaks;
    }

    const GAUSSIAN_EXP_FACTOR = -4 * Math.LN2;
    const ROOT_PI_OVER_LN2 = Math.sqrt(Math.PI / Math.LN2);
    const ROOT_THREE = Math.sqrt(3);
    const ROOT_2LN2 = Math.sqrt(2 * Math.LN2);
    const ROOT_2LN2_MINUS_ONE = Math.sqrt(2 * Math.LN2) - 1;

    // https://en.wikipedia.org/wiki/Error_function#Inverse_functions
    // This code yields to a good approximation
    // If needed a better implementation using polynomial can be found on https://en.wikipedia.org/wiki/Error_function#Inverse_functions
    function erfinv(x) {
      let a = 0.147;
      if (x === 0) return 0;
      let ln1MinusXSqrd = Math.log(1 - x * x);
      let lnEtcBy2Plus2 = ln1MinusXSqrd / 2 + 2 / (Math.PI * a);
      let firstSqrt = Math.sqrt(lnEtcBy2Plus2 ** 2 - ln1MinusXSqrd / a);
      let secondSqrt = Math.sqrt(firstSqrt - lnEtcBy2Plus2);
      return secondSqrt * (x > 0 ? 1 : -1);
    }

    class Gaussian {
      constructor(options = {}) {
        const {
          fwhm = 500,
          sd
        } = options;
        this.fwhm = sd ? gaussianWidthToFWHM(2 * sd) : fwhm;
      }
      fwhmToWidth(fwhm = this.fwhm) {
        return gaussianFwhmToWidth(fwhm);
      }
      widthToFWHM(width) {
        return gaussianWidthToFWHM(width);
      }
      fct(x) {
        return gaussianFct(x, this.fwhm);
      }
      getArea(height = calculateGaussianHeight({
        fwhm: this.fwhm
      })) {
        return getGaussianArea({
          fwhm: this.fwhm,
          height
        });
      }
      getFactor(area) {
        return getGaussianFactor(area);
      }
      getData(options = {}) {
        return getGaussianData(this, options);
      }
      calculateHeight(area = 1) {
        return calculateGaussianHeight({
          fwhm: this.fwhm,
          area
        });
      }
      getParameters() {
        return ['fwhm'];
      }
    }
    function calculateGaussianHeight(options) {
      let {
        fwhm = 500,
        area = 1,
        sd
      } = options;
      if (sd) fwhm = gaussianWidthToFWHM(2 * sd);
      return 2 * area / ROOT_PI_OVER_LN2 / fwhm;
    }
    /**
     * Calculate the height of the gaussian function of a specific width (fwhm) at a speicifc
     * x position (the gaussian is centered on x=0)
     * @param x
     * @param fwhm
     * @returns y
     */
    function gaussianFct(x, fwhm) {
      return Math.exp(GAUSSIAN_EXP_FACTOR * Math.pow(x / fwhm, 2));
    }
    function gaussianWidthToFWHM(width) {
      return width * ROOT_2LN2;
    }
    function gaussianFwhmToWidth(fwhm) {
      return fwhm / ROOT_2LN2;
    }
    function getGaussianArea(options) {
      let {
        fwhm = 500,
        sd,
        height = 1
      } = options;
      if (sd) fwhm = gaussianWidthToFWHM(2 * sd);
      return height * ROOT_PI_OVER_LN2 * fwhm / 2;
    }
    function getGaussianFactor(area = 0.9999) {
      return Math.sqrt(2) * erfinv(area);
    }
    function getGaussianData(shape = {}, options = {}) {
      let {
        fwhm = 500,
        sd
      } = shape;
      if (sd) fwhm = gaussianWidthToFWHM(2 * sd);
      let {
        length,
        factor = getGaussianFactor(),
        height = calculateGaussianHeight({
          fwhm
        })
      } = options;
      if (!length) {
        length = Math.min(Math.ceil(fwhm * factor), Math.pow(2, 25) - 1);
        if (length % 2 === 0) length++;
      }
      const center = (length - 1) / 2;
      const data = new Float64Array(length);
      for (let i = 0; i <= center; i++) {
        data[i] = gaussianFct(i - center, fwhm) * height;
        data[length - 1 - i] = data[i];
      }
      return data;
    }

    class Lorentzian {
      constructor(options = {}) {
        const {
          fwhm = 500
        } = options;
        this.fwhm = fwhm;
      }
      fwhmToWidth(fwhm = this.fwhm) {
        return lorentzianFwhmToWidth(fwhm);
      }
      widthToFWHM(width) {
        return lorentzianWidthToFWHM(width);
      }
      fct(x) {
        return lorentzianFct(x, this.fwhm);
      }
      getArea(height = 1) {
        return getLorentzianArea({
          fwhm: this.fwhm,
          height
        });
      }
      getFactor(area) {
        return getLorentzianFactor(area);
      }
      getData(options = {}) {
        return getLorentzianData(this, options);
      }
      calculateHeight(area = 1) {
        return calculateLorentzianHeight({
          fwhm: this.fwhm,
          area
        });
      }
      getParameters() {
        return ['fwhm'];
      }
    }
    const calculateLorentzianHeight = ({
      fwhm = 1,
      area = 1
    }) => {
      return 2 * area / Math.PI / fwhm;
    };
    const getLorentzianArea = options => {
      const {
        fwhm = 500,
        height = 1
      } = options;
      return height * Math.PI * fwhm / 2;
    };
    const lorentzianFct = (x, fwhm) => {
      return fwhm ** 2 / (4 * x ** 2 + fwhm ** 2);
    };
    const lorentzianWidthToFWHM = width => {
      return width * ROOT_THREE;
    };
    const lorentzianFwhmToWidth = fwhm => {
      return fwhm / ROOT_THREE;
    };
    const getLorentzianFactor = (area = 0.9999) => {
      if (area >= 1) {
        throw new Error('area should be (0 - 1)');
      }
      const halfResidual = (1 - area) * 0.5;
      const quantileFunction = p => Math.tan(Math.PI * (p - 0.5));
      return (quantileFunction(1 - halfResidual) - quantileFunction(halfResidual)) / 2;
    };
    const getLorentzianData = (shape = {}, options = {}) => {
      let {
        fwhm = 500
      } = shape;
      let {
        length,
        factor = getLorentzianFactor(),
        height = calculateLorentzianHeight({
          fwhm,
          area: 1
        })
      } = options;
      if (!length) {
        length = Math.min(Math.ceil(fwhm * factor), Math.pow(2, 25) - 1);
        if (length % 2 === 0) length++;
      }
      const center = (length - 1) / 2;
      const data = new Float64Array(length);
      for (let i = 0; i <= center; i++) {
        data[i] = lorentzianFct(i - center, fwhm) * height;
        data[length - 1 - i] = data[i];
      }
      return data;
    };

    class PseudoVoigt {
      constructor(options = {}) {
        const {
          fwhm = 500,
          mu = 0.5
        } = options;
        this.mu = mu;
        this.fwhm = fwhm;
      }
      fwhmToWidth(fwhm = this.fwhm, mu = this.mu) {
        return pseudoVoigtFwhmToWidth(fwhm, mu);
      }
      widthToFWHM(width, mu = this.mu) {
        return pseudoVoigtWidthToFWHM(width, mu);
      }
      fct(x) {
        return pseudoVoigtFct(x, this.fwhm, this.mu);
      }
      getArea(height = 1) {
        return getPseudoVoigtArea({
          fwhm: this.fwhm,
          height,
          mu: this.mu
        });
      }
      getFactor(area) {
        return getPseudoVoigtFactor(area);
      }
      getData(options = {}) {
        const {
          length,
          factor,
          height = calculatePseudoVoigtHeight({
            fwhm: this.fwhm,
            mu: this.mu,
            area: 1
          })
        } = options;
        return getPseudoVoigtData(this, {
          factor,
          length,
          height
        });
      }
      calculateHeight(area = 1) {
        return calculatePseudoVoigtHeight({
          fwhm: this.fwhm,
          mu: this.mu,
          area
        });
      }
      getParameters() {
        return ['fwhm', 'mu'];
      }
    }
    const calculatePseudoVoigtHeight = (options = {}) => {
      let {
        fwhm = 1,
        mu = 0.5,
        area = 1
      } = options;
      return 2 * area / (fwhm * (mu * ROOT_PI_OVER_LN2 + (1 - mu) * Math.PI));
    };
    const pseudoVoigtFct = (x, fwhm, mu) => {
      return (1 - mu) * lorentzianFct(x, fwhm) + mu * gaussianFct(x, fwhm);
    };
    const pseudoVoigtWidthToFWHM = (width, mu = 0.5) => {
      return width * (mu * ROOT_2LN2_MINUS_ONE + 1);
    };
    const pseudoVoigtFwhmToWidth = (fwhm, mu = 0.5) => {
      return fwhm / (mu * ROOT_2LN2_MINUS_ONE + 1);
    };
    const getPseudoVoigtArea = options => {
      const {
        fwhm = 500,
        height = 1,
        mu = 0.5
      } = options;
      return fwhm * height * (mu * ROOT_PI_OVER_LN2 + (1 - mu) * Math.PI) / 2;
    };
    const getPseudoVoigtFactor = (area = 0.9999, mu = 0.5) => {
      return mu < 1 ? getLorentzianFactor(area) : getGaussianFactor(area);
    };
    const getPseudoVoigtData = (shape = {}, options = {}) => {
      let {
        fwhm = 500,
        mu = 0.5
      } = shape;
      let {
        length,
        factor = getPseudoVoigtFactor(0.999, mu),
        height = calculatePseudoVoigtHeight({
          fwhm,
          mu,
          area: 1
        })
      } = options;
      if (!height) {
        height = 1 / (mu / Math.sqrt(-GAUSSIAN_EXP_FACTOR / Math.PI) * fwhm + (1 - mu) * fwhm * Math.PI / 2);
      }
      if (!length) {
        length = Math.min(Math.ceil(fwhm * factor), Math.pow(2, 25) - 1);
        if (length % 2 === 0) length++;
      }
      const center = (length - 1) / 2;
      const data = new Float64Array(length);
      for (let i = 0; i <= center; i++) {
        data[i] = pseudoVoigtFct(i - center, fwhm, mu) * height;
        data[length - 1 - i] = data[i];
      }
      return data;
    };

    /**
     * Generate a instance of a specific kind of shape.
     */
    function getShape1D(shape) {
      const {
        kind
      } = shape;
      switch (kind) {
        case 'gaussian':
          return new Gaussian(shape);
        case 'lorentzian':
          return new Lorentzian(shape);
        case 'pseudoVoigt':
          return new PseudoVoigt(shape);
        default:
          {
            throw Error(`Unknown distribution ${kind}`);
          }
      }
    }

    /**
     * @param {object}   [options={}]
     * @param {object}   [options.min=1]
     * @param {object}   [options.max=10]
     * @param {object}   [options.low=-1]
     * @param {object}   [options.high=1]
     * @param {object}   [options.precision=100]
     */

    const NEUTRON_MASS = 1;
    function appendPeaksCharge(peaks, options = {}) {
      let {
        precision = 100,
        low = -1,
        high = 1,
        min: minCharge = 1,
        max: maxCharge = 10
      } = options;
      let fromCharge = minCharge * maxCharge > 0 ? Math.round(Math.min(Math.abs(minCharge), Math.abs(maxCharge))) : 1;
      let toCharge = Math.round(Math.max(Math.abs(minCharge), Math.abs(maxCharge)));
      let fromIsotope = Math.ceil(low);
      let toIsotope = Math.floor(high);
      let numberIsotopes = toIsotope - fromIsotope + 1;
      let isotopeIntensity = 1 / numberIsotopes;
      let fromIndex = 0;
      let localFromIndex = 0;
      let localToIndex = 0;
      for (let peakIndex = 0; peakIndex < peaks.length; peakIndex++) {
        let peak = peaks[peakIndex];
        let targetMass = peak.x;
        localFromIndex = fromIndex;
        let bestCharge = fromCharge;
        let bestChargeMatch = 0;
        for (let charge = fromCharge; charge < toCharge + 1; charge++) {
          let theoreticalPositions = {
            x: [],
            y: new Array(numberIsotopes).fill(isotopeIntensity)
          };
          let massRange = precision * 1e-6 * targetMass;
          for (let isotopePosition = fromIsotope; isotopePosition < toIsotope + 1; isotopePosition++) {
            theoreticalPositions.x.push(targetMass + isotopePosition * NEUTRON_MASS / charge);
          }
          let fromMass = targetMass + low / Math.abs(charge) - massRange;
          let toMass = targetMass + high / Math.abs(charge) + massRange;
          if (charge === 1) {
            // we may move the fromIndex
            while (peaks[fromIndex].x < fromMass) {
              fromIndex++;
            }
          }

          /*
           * Find the from / to index for the specific peak and specific charge
           */
          while (peaks[localFromIndex].x < fromMass) {
            localFromIndex++;
          }
          localToIndex = localFromIndex;
          let localHeightSum = 0;
          while (localToIndex < peaks.length && peaks[localToIndex].x < toMass) {
            localHeightSum += peaks[localToIndex].y;
            localToIndex++;
          }
          localToIndex--;

          //  console.log({ localFromIndex, localToIndex });
          /*
            Calculate the overlap for a specific peak and specific charge
          */
          let currentTheoreticalPosition = 0;
          let theoreticalMaxValue = 1 / numberIsotopes;
          let totalMatch = 0;
          for (let index = localFromIndex; index <= localToIndex; index++) {
            let minMass = theoreticalPositions.x[currentTheoreticalPosition] - massRange / charge;
            let maxMass = theoreticalPositions.x[currentTheoreticalPosition] + massRange / charge;
            while (maxMass < peaks[index].x) {
              currentTheoreticalPosition++;
              theoreticalMaxValue = 1 / numberIsotopes;
              minMass = theoreticalPositions.x[currentTheoreticalPosition] - massRange / charge;
              maxMass = theoreticalPositions.x[currentTheoreticalPosition] + massRange / charge;
            }
            while (index < peaks.length && peaks[index].x < minMass) {
              index++;
            }

            //    console.log({ index, minMass, maxMass, massRange, localHeightSum });
            if (index < peaks.length && peaks[index].x <= maxMass) {
              while (index < peaks.length && peaks[index].x <= maxMass) {
                if (peaks[index].x >= minMass && peaks[index].x <= maxMass) {
                  let value = peaks[index].y / localHeightSum;
                  //      console.log({ theoreticalMaxValue, value });
                  value = Math.min(theoreticalMaxValue, value);
                  theoreticalMaxValue -= value;
                  totalMatch += value;
                }
                index++;
              }
              index--;
            }
            if (totalMatch > bestChargeMatch) {
              bestCharge = charge;
              bestChargeMatch = totalMatch;
            }
          }
        }
        peak.charge = bestCharge;
      }
      return peaks;
    }

    /**
     * Filter the array of peaks
     * @param {Spectrum} spectrum - array of all the peaks
     * @param {object} [options={}]
     * @param {object} [options.charge={}]
     * @param {number} [options.charge.min=1]
     * @param {number} [options.charge.max=10]
     * @param {number} [options.charge.low=-1]
     * @param {number} [options.charge.high=1]
     * @param {number} [options.charge.precision=30]
     * @returns {array} - copy of peaks with 'close' annotation
     */

    function peakPicking(spectrum, options = {}) {
      const {
        charge: chargeOptions = {}
      } = options;
      if (!spectrum.peaks || spectrum.peaks.length === 0) {
        spectrum.peaks = [];
        const keys = Object.keys(spectrum.data).filter(key => key !== 'x' && key !== 'y');
        if (spectrum.isContinuous()) {
          // some experimental data are really problematic and we need to add this line
          const data = xyEnsureGrowingX(spectrum.data);
          const gsdPeaks = gsd(data, {
            minMaxRatio: spectrum.options.threshold || 0.00025,
            // Threshold to determine if a given peak should be considered as a noise
            realTopDetection: true,
            smoothY: false,
            sgOptions: {
              windowSize: 7,
              polynomial: 3
            }
          });
          for (let gsdPeak of gsdPeaks) {
            const peak = {
              x: gsdPeak.x,
              y: gsdPeak.y,
              width: gsdPeak.width
            };
            for (let key of keys) {
              peak[key] = spectrum.data[key][gsdPeak.index];
            }
            spectrum.peaks.push(peak);
          }
        } else {
          spectrum.peaks = [];
          let data = spectrum.data;
          for (let i = 0; i < data.x.length; i++) {
            const peak = {
              x: data.x[i],
              y: data.y[i],
              width: 0
            };
            for (let key of keys) {
              peak[key] = spectrum.data[key][i];
            }
            spectrum.peaks.push(peak);
          }
        }
        // required and linked to https://github.com/mljs/global-spectral-deconvolution/issues/17
        spectrum.peaks = spectrum.peaks.filter(peak => !isNaN(peak.x) && !isNaN(peak.y));
        appendPeaksCharge(spectrum.peaks, chargeOptions);
      }
      return spectrum.peaks;
    }

    function maybeToPrecision(value, digits) {
      if (value < 0) {
        value = 0 - value;
        if (typeof digits === 'number') {
          return `- ${value.toPrecision(digits)}`;
        } else {
          return `- ${value.toString()}`;
        }
      } else {
        if (typeof digits === 'number') {
          return value.toPrecision(digits);
        } else {
          return value.toString();
        }
      }
    }

    function checkArraySize(x, y) {
      if (!isAnyArray$1(x) || !isAnyArray$1(y)) {
        throw new TypeError('x and y must be arrays');
      }
      if (x.length !== y.length) {
        throw new RangeError('x and y arrays must have the same length');
      }
    }

    class BaseRegression {
      constructor() {
        if (new.target === BaseRegression) {
          throw new Error('BaseRegression must be subclassed');
        }
      }
      predict(x) {
        if (typeof x === 'number') {
          return this._predict(x);
        } else if (isAnyArray$1(x)) {
          const y = [];
          for (let i = 0; i < x.length; i++) {
            y.push(this._predict(x[i]));
          }
          return y;
        } else {
          throw new TypeError('x must be a number or array');
        }
      }
      _predict() {
        throw new Error('_predict must be implemented');
      }
      train() {
        // Do nothing for this package
      }
      toString() {
        return '';
      }
      toLaTeX() {
        return '';
      }

      /**
       * Return the correlation coefficient of determination (r) and chi-square.
       * @param {Array<number>} x
       * @param {Array<number>} y
       * @return {object}
       */
      score(x, y) {
        if (!isAnyArray$1(x) || !isAnyArray$1(y) || x.length !== y.length) {
          throw new Error('x and y must be arrays of the same length');
        }
        const n = x.length;
        const y2 = new Array(n);
        for (let i = 0; i < n; i++) {
          y2[i] = this._predict(x[i]);
        }
        let xSum = 0;
        let ySum = 0;
        let chi2 = 0;
        let rmsd = 0;
        let xSquared = 0;
        let ySquared = 0;
        let xY = 0;
        for (let i = 0; i < n; i++) {
          xSum += y2[i];
          ySum += y[i];
          xSquared += y2[i] * y2[i];
          ySquared += y[i] * y[i];
          xY += y2[i] * y[i];
          if (y[i] !== 0) {
            chi2 += (y[i] - y2[i]) * (y[i] - y2[i]) / y[i];
          }
          rmsd += (y[i] - y2[i]) * (y[i] - y2[i]);
        }
        const r = (n * xY - xSum * ySum) / Math.sqrt((n * xSquared - xSum * xSum) * (n * ySquared - ySum * ySum));
        return {
          r: r,
          r2: r * r,
          chi2: chi2,
          rmsd: Math.sqrt(rmsd / n)
        };
      }
    }

    class SimpleLinearRegression extends BaseRegression {
      constructor(x, y) {
        super();
        if (x === true) {
          this.slope = y.slope;
          this.intercept = y.intercept;
          this.coefficients = [y.intercept, y.slope];
        } else {
          checkArraySize(x, y);
          regress$1(this, x, y);
        }
      }
      toJSON() {
        return {
          name: 'simpleLinearRegression',
          slope: this.slope,
          intercept: this.intercept
        };
      }
      _predict(x) {
        return this.slope * x + this.intercept;
      }
      computeX(y) {
        return (y - this.intercept) / this.slope;
      }
      toString(precision) {
        let result = 'f(x) = ';
        if (this.slope !== 0) {
          const xFactor = maybeToPrecision(this.slope, precision);
          result += `${xFactor === '1' ? '' : `${xFactor} * `}x`;
          if (this.intercept !== 0) {
            const absIntercept = Math.abs(this.intercept);
            const operator = absIntercept === this.intercept ? '+' : '-';
            result += ` ${operator} ${maybeToPrecision(absIntercept, precision)}`;
          }
        } else {
          result += maybeToPrecision(this.intercept, precision);
        }
        return result;
      }
      toLaTeX(precision) {
        return this.toString(precision);
      }
      static load(json) {
        if (json.name !== 'simpleLinearRegression') {
          throw new TypeError('not a SLR model');
        }
        return new SimpleLinearRegression(true, json);
      }
    }
    function regress$1(slr, x, y) {
      const n = x.length;
      let xSum = 0;
      let ySum = 0;
      let xSquared = 0;
      let xY = 0;
      for (let i = 0; i < n; i++) {
        xSum += x[i];
        ySum += y[i];
        xSquared += x[i] * x[i];
        xY += x[i] * y[i];
      }
      const numerator = n * xY - xSum * ySum;
      slr.slope = numerator / (n * xSquared - xSum * xSum);
      slr.intercept = 1 / n * ySum - slr.slope * (1 / n) * xSum;
      slr.coefficients = [slr.intercept, slr.slope];
    }

    class PowerRegression extends BaseRegression {
      constructor(x, y) {
        super();
        if (x === true) {
          // reloading model
          this.A = y.A;
          this.B = y.B;
        } else {
          checkArraySize(x, y);
          regress(this, x, y);
        }
      }
      _predict(newInputs) {
        return this.A * Math.pow(newInputs, this.B);
      }
      toJSON() {
        return {
          name: 'powerRegression',
          A: this.A,
          B: this.B
        };
      }
      toString(precision) {
        return `f(x) = ${maybeToPrecision(this.A, precision)} * x^${maybeToPrecision(this.B, precision)}`;
      }
      toLaTeX(precision) {
        let latex = '';
        if (this.B >= 0) {
          latex = `f(x) = ${maybeToPrecision(this.A, precision)}x^{${maybeToPrecision(this.B, precision)}}`;
        } else {
          latex = `f(x) = \\frac{${maybeToPrecision(this.A, precision)}}{x^{${maybeToPrecision(-this.B, precision)}}}`;
        }
        latex = latex.replace(/e([+-]?[0-9]+)/g, 'e^{$1}');
        return latex;
      }
      static load(json) {
        if (json.name !== 'powerRegression') {
          throw new TypeError('not a power regression model');
        }
        return new PowerRegression(true, json);
      }
    }
    function regress(pr, x, y) {
      const n = x.length;
      const xl = new Array(n);
      const yl = new Array(n);
      for (let i = 0; i < n; i++) {
        xl[i] = Math.log(x[i]);
        yl[i] = Math.log(y[i]);
      }
      const linear = new SimpleLinearRegression(xl, yl);
      pr.A = Math.exp(linear.intercept);
      pr.B = linear.slope;
    }

    function peaksWidth(peaks) {
      let xs = peaks.map(peak => peak.x);
      let widths = peaks.map(peak => peak.width);
      if (xs.length < 2) {
        throw new Error(`peaksWidth: not enough peaks (less than 2) for automatic width calculation: ${xs.length}`);
      }
      let regression = new PowerRegression(xs, widths, {
        computeQuality: true,
        computeCoefficient: true
      });
      if (isNaN(regression.A) || isNaN(regression.B)) {
        throw new Error('peaksWidth: can not calculate regression');
      }
      let from = xMinValue(xs);
      let to = xMaxValue(xs);
      let regressionChart = {
        x: [],
        y: []
      };
      for (let x = from; x <= to; x += (to - from) / 1000) {
        regressionChart.x.push(x);
        regressionChart.y.push(regression.predict(x));
      }
      return {
        widths: {
          x: xs,
          y: widths
        },
        fit: regressionChart,
        score: regression.score(xs, widths),
        predictFct: regression.predict.bind(regression),
        tex: regression.toLaTeX(3),
        A: regression.A,
        B: regression.B,
        predictFctString: `${regression.A} * mass ** ${regression.B}`
      };
    }

    /**
     * Class dealing with mass spectra and peak picking
     */
    class Spectrum$1 {
      /**
       *
       * @param {import('cheminfo-types').DataXY} data
       * @param {object} [options={}]
       * @param {number} [options.threshold]
       */
      constructor(data = {
        x: [],
        y: []
      }, options = {}) {
        if (typeof data !== 'object' || !isAnyArray$1(data.x) || !isAnyArray$1(data.y)) {
          throw new TypeError('Spectrum data must be an object with x:[], y:[]');
        }
        this.data = {
          ...data
        };
        this.options = options;
        Object.defineProperty(this.data, 'xOriginal', {
          enumerable: false,
          writable: true
        });
        if (this.data && this.data.x.length > 0) {
          this.info = {
            minX: xMinValue(this.data.x),
            maxX: xMaxValue(this.data.x),
            minY: xMinValue(this.data.y),
            maxY: xMaxValue(this.data.y)
          };
        } else {
          this.info = {
            minX: NaN,
            maxX: NaN,
            minY: NaN,
            maxY: NaN
          };
        }
        this.cache = {};
        this.peaks = [];
      }
      minMaxX() {
        return xMinMaxValues(this.data.x);
      }
      maxY() {
        return xMaxValue(this.data.y);
      }
      sumY() {
        if (!this.cache.sumY) {
          this.cache.sumY = xSum(this.data.y);
        }
        return this.cache.sumY;
      }
      scaleY(intensity = 1) {
        this.data.y = Array.from(xNormed(this.data.y, {
          value: intensity,
          algorithm: 'max'
        }));
        return this;
      }
      rescaleX(callback) {
        this.ensureOriginalX();
        for (let i = 0; i < this.data.x.length; i++) {
          this.data.x[i] = callback(this.data.xOriginal[i]);
        }
        return this;
      }
      ensureOriginalX() {
        if (!this.data.xOriginal) {
          this.data.xOriginal = this.data.x.slice(0);
        }
      }
      normedY(total = 1) {
        this.data.y = xNormed(this.data.y, {
          value: total
        });
        return this;
      }
      peakPicking() {
        peakPicking(this);
        return this.peaks;
      }
      peaksWidth() {
        peakPicking(this);
        return peaksWidth(this.peaks);
      }
      getBestPeaks(options) {
        peakPicking(this);
        return getBestPeaks$1(this.peaks, options);
      }
      getPeakChargeBySimilarity(targetMass, options) {
        return getPeakChargeBySimilarity(this, targetMass, options);
      }
      getPeaks(options) {
        peakPicking(this);
        return getPeaks(this.peaks, options);
      }
      getPeaksAsDataXY(options) {
        peakPicking(this);
        const peaks = getPeaks(this.peaks, options);
        return {
          x: peaks.map(peak => peak.x),
          y: peaks.map(peak => peak.y)
        };
      }

      /**
       * Returns also peaks possible for a specific molecular formula
       * @example
       *  const spectrum = new Spectrum({x:[], y:[]})
       *  await spectrum.getFragmentPeaks();
       * @param {string} mf
       * @param {object} options
       * @returns
       */
      getFragmentPeaksFct(mf, options) {
        peakPicking(this);
        return getFragmentPeaks(this.peaks, mf, options);
      }
      isContinuous() {
        return isContinuous(this);
      }

      /**
       * Remove an integer number of time the specifiedd monoisotopic mass
       * Mass remainder analysis (MARA): https://doi.org/10.1021/acs.analchem.7b04730
       */
      getMassRemainderFct(mass, options = {}) {
        return getMassRemainder(this.data, mass, options);
      }
    }

    /**
     * Returns the average of cosine distances between vectors a and b
     * @param a - first vector
     * @param b - second vector
     *
     */
    function cosine(a, b) {
      let p = 0;
      let p2 = 0;
      let q2 = 0;
      for (let i = 0; i < a.length; i++) {
        p += a[i] * b[i];
        p2 += a[i] * a[i];
        q2 += b[i] * b[i];
      }
      return p / (Math.sqrt(p2) * Math.sqrt(q2));
    }

    class MSComparator$1 {
      /**
       * Create a class that will be able to get the similarity between 2 spectra
       * The similarity is based on 'cosine' similarity. The goal is 2 prepare 2 vectors
       * on which the similarity is calculated.
       * The vectors are created by taking the mass and the intensity of the peaks.
       * @param {object} [options={}]
       * @param {number} [options.nbPeaks] - Before comparing spectra how many peaks should be kept
       * @param {number} [options.minNbCommonPeaks] - Minimum number of peaks in common to consider any similarity
       * @param {number} [options.minIntensity] - What is the minimal relative intensity to keep a peak
       * @param {number} [options.massPower=3] - High power will give more weight to the mass. If you would prefer to observe fragments you should use a number less than 1
       * @param {number} [options.intensityPower=0.6] - How important is the intensity. By default we don't give to much importance to it
       * @param {number[]} [options.selectedMasses] - List of allowed masses.
       * @param {number|Function} [options.delta=0.1] - Tolerance in Da (u) to consider 2 peaks as aligned. If a function is provided it will be called with the mass of the peak
       */
      constructor(options = {}) {
        this.options = {
          massPower: 3,
          intensityPower: 0.6,
          delta: 0.1,
          ...options
        };
        this.cache = new WeakMap();
      }

      /**
       *
       * @param {import('cheminfo-types').DataXY} dataXY1
       * @param {import('cheminfo-types').DataXY} dataXY2
       */
      getSimilarity(dataXY1, dataXY2) {
        const data1 = normalizeAndCacheData(this.cache, dataXY1, this.options);
        const data2 = normalizeAndCacheData(this.cache, dataXY2, this.options);
        let aligned;
        if (this.options.selectedMasses?.length > 0) {
          aligned = xyArrayAlign([data1, data2, {
            x: Float64Array.from(this.options.selectedMasses),
            y: new Float64Array(this.options.selectedMasses.length).fill(1)
          }], {
            delta: this.options.delta,
            requiredY: true
          });
        } else {
          aligned = xyArrayAlign([data1, data2], {
            delta: this.options.delta
          });
        }
        if (this.options.minNbCommonPeaks) {
          let commonPeaks = 0;
          for (let i = 0; i < aligned.ys[0].length; i++) {
            if (aligned.ys[0][i] !== 0 && aligned.ys[1][i] !== 0) {
              commonPeaks++;
            }
          }
          if (commonPeaks < this.options.minNbCommonPeaks) return 0;
        }
        const vector1 = new Float64Array(aligned.x.length);
        const vector2 = new Float64Array(aligned.x.length);
        for (let i = 0; i < aligned.x.length; i++) {
          vector1[i] = aligned.x[i] ** this.options.massPower * aligned.ys[0][i] ** this.options.intensityPower;
          vector2[i] = aligned.x[i] ** this.options.massPower * aligned.ys[1][i] ** this.options.intensityPower;
        }
        return cosine(vector1, vector2);
      }
    }

    /**
     *
     * @param {WeakMap} cache
     * @param {import('cheminfo-types').DataXY} dataXY
     * @param {object} [options={}]
     * @param {number} [options.nbPeaks]
     * @param {number} [options.minIntensity]
     */
    function normalizeAndCacheData(cache, dataXY, options = {}) {
      const {
        nbPeaks,
        minIntensity
      } = options;
      if (cache.has(dataXY)) return cache.get(dataXY);
      let data = {
        ...dataXY
      };
      if (xIsMonotonic(data.x) !== 1) {
        data = xySortX(data);
      }
      if (minIntensity !== undefined) {
        data = xyFilterMinYValue(data, minIntensity);
      }
      if (nbPeaks !== undefined) {
        data = xyFilterTopYValues(data, nbPeaks);
      }
      cache.set(dataXY, data);
      return data;
    }

    /**
     *
     * @param {array} bestPeaks
     * @param {object} [options={}]
     * @param {array} [options.mfColors={}]
     * @param {number} [options.numberDigits=5]
     * @param {number} [options.numberMFs=0]
     * @param {boolean} [options.showMF=false]
     * @param {array} [options.mfColors={}]
     * @param {number} [options.charge=1]
     * @param {number} [options.shift=0]
     * @param {object} [options.mfPrefs]
     * @param {number} [options.displayCharge=true]
     * @param {number} [options.displayProperties=[]] Array of properties name to display
     * @returns {Promise}
     */

    async function getPeaksAnnotation(bestPeaks, options = {}) {
      const emdb = new EMDB$1();
      options = {
        limit: 5,
        precision: 100,
        ...options
      };
      let {
        numberDigits = 5,
        shift = 0,
        showMF = false,
        numberMFs = 0,
        charge = 1,
        mfPrefs = {},
        displayCharge = true,
        displayProperties = [],
        mfColors = [{
          limit: 3,
          color: 'green'
        }, {
          limit: 20,
          color: 'lightgreen'
        }, {
          limit: 50,
          color: 'lightorange'
        }]
      } = options;
      if (showMF && !numberMFs) numberMFs = 1;
      let annotations = [];
      bestPeaks.sort((a, b) => a.close ? -1 : b.close ? 1 : 0);
      for (let peak of bestPeaks) {
        let textLine = 0;
        let annotation;
        if (peak.close) {
          annotation = {
            type: 'line',
            _highlight: peak._highlight,
            info: peak,
            position: [{
              y: peak.y,
              dy: '-5px',
              x: peak.x
            }, {
              y: peak.y,
              dy: '-15px',
              x: peak.x
            }]
          };
          annotations.push(annotation);
          annotation = {
            type: 'ellipse',
            _highlight: peak._highlight,
            info: peak,
            position: [{
              y: peak.y,
              dy: '-15px',
              x: peak.x
            }],
            props: {
              rx: '3px',
              ry: '3px',
              fillOpacity: 0.3
            }
          };
        } else {
          annotation = {
            type: 'line',
            _highlight: peak._highlight,
            position: [{
              y: peak.y,
              dy: '-5px',
              x: peak.x
            }, {
              y: peak.y,
              dy: '-25px',
              x: peak.x
            }],
            labels: [{
              text: (peak.x + shift).toFixed(numberDigits),
              color: 'red',
              position: {
                x: peak.x,
                y: peak.y,
                dy: `${textLine++ * -13 - 17}px`,
                dx: '2px'
              }
            }]
          };
          if (displayCharge) {
            annotation.labels.push({
              text: `Z:${peak.charge}`,
              color: 'grey',
              position: {
                x: peak.x,
                y: peak.y,
                dy: '-4px',
                dx: '2px'
              }
            });
          }
          if (numberMFs) {
            // we have 2 cases. Either there is a shift and we deal with differences
            // otherwise it is absolute
            // if there is a shift we consider only a neutral loss and the parameter charge is important
            if (shift) {
              // neutral loss
              let currentMfPrefs = {
                ...mfPrefs,
                allowNeutral: true,
                ionizations: ''
              };
              // we need to deal with the precision and increase it
              currentMfPrefs.precision = currentMfPrefs.precision / Math.max(Math.abs(peak.x + shift), 1) * peak.x;
              await emdb.fromMonoisotopicMass(Math.abs((peak.x + shift) * charge), currentMfPrefs);
            } else {
              await emdb.fromMonoisotopicMass(Math.abs(peak.x * charge), mfPrefs);
            }
            let mfs = emdb.get('monoisotopic');
            let numberOfMFS = Math.min(mfs.length, numberMFs);
            for (let i = 0; i < numberOfMFS; i++) {
              let mf = mfs[i];
              let ppm = shift ? mf.ms.ppm / shift * mfs[0].ms.em : mf.ms.ppm;
              annotation.labels.push({
                text: mf.mf,
                color: getColor$1(mfColors, Math.abs(ppm)),
                position: {
                  x: peak.x,
                  y: peak.y,
                  dy: `${textLine++ * -13 - 17}px`,
                  dx: '2px'
                }
              });
            }
          }
          if (displayProperties.length > 0) {
            for (let property of displayProperties) {
              annotation.labels.push({
                text: peak[property],
                color: 'red',
                position: {
                  x: peak.x,
                  y: peak.y,
                  dy: `${textLine++ * -13 - 17}px`,
                  dx: '2px'
                }
              });
            }
          }
        }
        annotations.push(annotation);
      }
      return annotations;
    }
    function getColor$1(colors, value) {
      for (let color of colors) {
        if (value < color.limit) return color.color;
      }
      return 'lightgrey';
    }

    const JsGraph = {
      getPeaksAnnotation
    };

    /**
     * @param {object}   [entry={}]
     * @param {object}   [options={}]
     * @param {number}   [options.min=-Infinity] - Minimal unsaturation
     * @param {number}   [options.max=+Infinity] - Maximal unsaturation
     * @param {boolean}   [options.onlyInteger=false] - Integer unsaturation
     * @param {boolean}   [options.onlyNonInteger=false] - Non integer unsaturation
     * @return {boolean}
     */

    function unsaturationMatcher(entry, options = {}) {
      const {
        min = Number.MIN_SAFE_INTEGER,
        max = Number.MAX_SAFE_INTEGER,
        onlyInteger,
        onlyNonInteger
      } = options;
      if (entry.unsaturation !== undefined) {
        if (entry.unsaturation < min || entry.unsaturation > max) return false;
        if (onlyInteger && !Number.isInteger(entry.unsaturation)) return false;
        if (onlyNonInteger && Number.isInteger(entry.unsaturation)) return false;
      }
      return true;
    }

    /**
     * Returns true if the entry containing MF information match
     * @param {object}   [entry={}] - object containing mw, ...
     * @param {object}   [options={}]
     * @param {number}   [options.minMW=0] - Minimal molecular weight
     * @param {number}   [options.maxMW=+Infinity] - Maximal molecular weight
     * @param {number}   [options.minEM=0] - Minimal monoisotopic mass
     * @param {number}   [options.maxEM=+Infinity] - Maximal monoisotopic mass
     * @param {number}   [options.minCharge=-Infinity] - Minimal charge
     * @param {number}   [options.maxCharge=+Infinity] - Maximal charge
     * @param {boolean}  [options.absoluteCharge=false] - If true, the charge is absolute (so between 0 and +Infinity by default)
     * @param {object}   [options.unsaturation={}]
     * @param {number}   [options.unsaturation.min=-Infinity] - Minimal unsaturation
     * @param {number}   [options.unsaturation.max=+Infinity] - Maximal unsaturation
     * @param {boolean}   [options.unsaturation.onlyInteger=false] - Integer unsaturation
     * @param {boolean}   [options.unsaturation.onlyNonInteger=false] - Non integer unsaturation
     * @param {object}   [options.atoms] - object of atom:{min, max}
     * @return {boolean}
     */

    function generalMatcher(entry, options = {}) {
      const {
        minMW = 0,
        maxMW = +Infinity,
        minEM = 0,
        maxEM = +Infinity,
        minCharge = Number.MIN_SAFE_INTEGER,
        maxCharge = Number.MAX_SAFE_INTEGER,
        absoluteCharge = false,
        unsaturation = {},
        atoms
      } = options;
      if (entry.mw !== undefined) {
        if (entry.mw < minMW || entry.mw > maxMW) return false;
      }
      if (entry.em !== undefined) {
        if (entry.em < minEM || entry.em > maxEM) return false;
      }
      if (entry.charge !== undefined) {
        let charge = absoluteCharge ? Math.abs(entry.charge) : entry.charge;
        if (charge < minCharge || charge > maxCharge) return false;
      }
      if (unsaturation !== undefined && entry.unsaturation !== undefined) {
        if (!unsaturationMatcher(entry, unsaturation)) return false;
      }
      if (entry.atoms !== undefined && atoms) {
        // all the atoms of the entry must fit in the range
        for (let atom in entry.atoms) {
          if (!atoms[atom]) return false;
          if (entry.atoms[atom] < atoms[atom].min) return false;
          if (entry.atoms[atom] > atoms[atom].max) return false;
        }
      }
      return true;
    }

    function getMsem(em, charge) {
      if (charge > 0) {
        return em / charge - ELECTRON_MASS;
      } else if (charge < 0) {
        return em / (charge * -1) + ELECTRON_MASS;
      } else {
        return 0;
      }
    }

    /**
     * Returns an object containing:
     * {ms: {em, charge, ionization}, ionization: {}}
     * We return the ionization in order to know which one has been selected
     */

    function getMsInfo(entry, options = {}) {
      const {
        allowNeutralMolecules,
        ionization = {
          mf: '',
          em: 0,
          charge: 0
        },
        forceIonization = false,
        targetMass
      } = options;
      let realIonization = ionization;
      if (!forceIonization && entry.ionization && entry.ionization.mf !== '') {
        realIonization = entry.ionization;
      }
      let ms = {
        ionization: realIonization.mf,
        em: 0,
        charge: entry.charge + realIonization.charge
      };
      if (ms.charge !== 0) {
        ms.em = getMsem(entry.em + realIonization.em, ms.charge);
      } else if (allowNeutralMolecules) {
        ms.em = entry.em + realIonization.em;
      }
      if (targetMass) {
        ms.delta = targetMass - ms.em;
        ms.ppm = (targetMass - ms.em) / ms.em * 1e6;
      }
      return {
        ms,
        ionization: realIonization
      };
    }

    function mfDiff(mfString1, mfString2) {
      let mf1 = new MF(mfString1).getInfo().atoms;
      let mf2 = new MF(mfString2).getInfo().atoms;
      let atoms = Object.keys(mf1);
      Object.keys(mf2).forEach(atom => {
        if (!atoms.includes(atom)) atoms.push(atom);
      });
      let mf = '';
      for (let atom of atoms) {
        let diff = (mf1[atom] || 0) - (mf2[atom] || 0);
        if (diff) mf += atom + diff;
      }
      return new MF(mf).toMF();
    }

    function preprocessIonizations(ionizationsString = '') {
      if (Array.isArray(ionizationsString)) return ionizationsString;
      let ionizations = ionizationsString.split(/ *[.,;\t\r\n]+ */);

      // it is allowed to have ranges in Ionizations. We need to explode them.

      let results = [];
      for (let ionization of ionizations) {
        let parts = new MF(ionization).flatten();
        for (let part of parts) {
          let info = new MF(part).getInfo();
          results.push({
            mf: part,
            em: info.monoisotopicMass,
            charge: info.charge,
            atoms: info.atoms
          });
        }
      }
      return results;
    }

    function preprocessRanges(ranges) {
      ranges = JSON.parse(JSON.stringify(ranges));
      if (typeof ranges === 'string') {
        // need to convert to ranges
        let parsed = parse(ranges.replace(/[\r\n\t ]/g, ''));
        let newRanges = [];
        let current = {
          mf: '',
          min: 1,
          max: 1
        };

        // example ClBr2(CH2)0-2NO
        // the idea is that has long as we don't have a range we don't really care
        // there is a limitation is that the range has to be first level of parenthesis
        let parenthesisLevel = 0;
        let currentMF = ''; // start at an atom first level or a parenthesis
        for (let item of parsed) {
          switch (item.kind) {
            case Kind.ATOM:
              if (parenthesisLevel === 0 && currentMF) {
                current.mf += currentMF;
                currentMF = '';
              }
              currentMF += item.value;
              break;
            case Kind.ISOTOPE:
              if (parenthesisLevel === 0 && currentMF) {
                current.mf += currentMF;
                currentMF = '';
              }
              currentMF += `[${item.value.isotope}${item.value.atom}]`;
              break;
            case Kind.MULTIPLIER:
              if (parenthesisLevel === 0 && currentMF) {
                current.mf += currentMF + item.value;
                currentMF = '';
              } else {
                currentMF += item.value;
              }
              break;
            case Kind.MULTIPLIER_RANGE:
              if (parenthesisLevel !== 0) {
                throw new Error('Range multiplier can only be at the first level');
              }
              newRanges.push({
                mf: currentMF,
                min: item.value.from,
                max: item.value.to
              });
              currentMF = '';
              break;
            case Kind.OPENING_PARENTHESIS:
              parenthesisLevel++;
              currentMF += '(';
              break;
            case Kind.CLOSING_PARENTHESIS:
              parenthesisLevel--;
              currentMF += ')';
              break;
            default:
              throw Error(`can not preprocess ${ranges}`);
          }
        }
        if (currentMF) {
          current.mf += currentMF;
        }
        if (current.mf) {
          newRanges.push(current);
        }
        ranges = newRanges;
      }
      let possibilities = [];
      for (let i = 0; i < ranges.length; i++) {
        let range = ranges[i];
        let min = range.min === undefined ? 0 : range.min;
        let max = range.max === undefined ? 1 : range.max;
        let possibility = {
          mf: range.mf,
          originalMinCount: min,
          // value defined by the user
          originalMaxCount: max,
          // value defined by the user
          currentMinCount: min,
          currentMaxCount: max,
          currentCount: min,
          currentMonoisotopicMass: 0,
          currentCharge: 0,
          currentUnsaturation: 0,
          initialOrder: i,
          minInnerMass: 0,
          maxInnerMass: 0,
          minInnerCharge: 0,
          maxInnerCharge: 0,
          minCharge: 0,
          maxCharge: 0,
          minMass: 0,
          maxMass: 0,
          innerCharge: false
        };
        possibilities.push(possibility);
        let info = new MF(range.mf).getInfo();
        possibility.em = range.em || info.monoisotopicMass;
        possibility.charge = range.charge || info.charge;
        possibility.unsaturation = range.unsaturation === undefined ? (info.unsaturation - 1) * 2 : range.unsaturation;
        possibility.atoms = info.atoms;
        if (possibility.mf !== info.mf) {
          possibility.isGroup = true;
        }
      }
      possibilities = possibilities.filter(r => r.originalMinCount !== 0 || r.originalMaxCount !== 0);

      // we will sort the way we analyse the data
      // 1. The one possibility parameter
      // 2. The charged part
      // 3. Decreasing em
      possibilities.sort((a, b) => {
        if (a.originalMinCount === a.originalMaxCount) return -1; // should be in front, they are 'static'
        if (b.originalMinCount === b.originalMaxCount) return 1;
        if (a.charge && b.charge) {
          if (Math.abs(a.charge) > Math.abs(b.charge)) return -1;
          if (Math.abs(a.charge) < Math.abs(b.charge)) return 1;
          return b.em - a.em;
        }
        if (a.charge) return -1;
        if (b.charge) return 1;
        return b.em - a.em;
      });

      // we calculate couple of fixed values

      for (let i = 0; i < possibilities.length; i++) {
        for (let j = i; j < possibilities.length; j++) {
          let possibility = possibilities[j];
          if (possibility.em > 0) {
            possibilities[i].minMass += possibility.em * possibility.originalMinCount;
            possibilities[i].maxMass += possibility.em * possibility.originalMaxCount;
          } else {
            possibilities[i].minMass += possibility.em * possibility.originalMaxCount;
            possibilities[i].maxMass += possibility.em * possibility.originalMinCount;
          }
          if (possibility.charge > 0) {
            possibilities[i].minCharge += possibility.charge * possibility.originalMinCount;
            possibilities[i].maxCharge += possibility.charge * possibility.originalMaxCount;
          } else {
            possibilities[i].minCharge += possibility.charge * possibility.originalMaxCount;
            possibilities[i].maxCharge += possibility.charge * possibility.originalMinCount;
          }
        }
      }
      for (let i = 0; i < possibilities.length; i++) {
        if (i < possibilities.length - 1) {
          let possibility = possibilities[i];
          let innerPossibility = possibilities[i + 1];
          possibility.minInnerMass = innerPossibility.minMass;
          possibility.maxInnerMass = innerPossibility.maxMass;
          possibility.minInnerCharge = innerPossibility.minCharge;
          possibility.maxInnerCharge = innerPossibility.maxCharge;
          if (possibility.minInnerCharge || possibility.maxInnerCharge) {
            possibility.innerCharge = true;
          }
        }
      }
      return possibilities;
    }

    function getRangesForFragment(ranges) {
      ranges = JSON.parse(JSON.stringify(ranges));
      if (typeof ranges === 'string') {
        // need to convert to ranges
        let parsed = parse(ranges.replace(/[\r\n\t ]/g, ''));
        let newRanges = [];

        // example ClBr2(CH2)0-2NO
        // the idea is that has long as we don't have a range we don't really care
        // there is a limitation is that the range has to be first level of parenthesis
        let parenthesisLevel = 0;
        let currentMF = ''; // start at an atom first level or a parenthesis
        for (let item of parsed) {
          switch (item.kind) {
            case Kind.ATOM:
              if (parenthesisLevel === 0 && currentMF) {
                newRanges.push({
                  mf: currentMF
                });
                currentMF = '';
              }
              currentMF += item.value;
              break;
            case Kind.ISOTOPE:
              if (parenthesisLevel === 0 && currentMF) {
                newRanges.push({
                  mf: currentMF
                });
                currentMF = '';
              }
              currentMF += `[${item.value.isotope}${item.value.atom}]`;
              break;
            case Kind.MULTIPLIER:
              if (parenthesisLevel === 0 && currentMF) {
                newRanges.push({
                  mf: currentMF,
                  max: item.value
                });
                currentMF = '';
              } else {
                currentMF += item.value;
              }
              break;
            case Kind.MULTIPLIER_RANGE:
              if (parenthesisLevel !== 0) {
                throw new Error('Range multiplier can only be at the first level');
              }
              newRanges.push({
                mf: currentMF,
                min: item.value.from,
                max: item.value.to
              });
              currentMF = '';
              break;
            case Kind.OPENING_PARENTHESIS:
              if (parenthesisLevel === 0 && currentMF) {
                newRanges.push({
                  mf: currentMF
                });
                currentMF = '';
              }
              parenthesisLevel++;
              currentMF += '(';
              break;
            case Kind.CLOSING_PARENTHESIS:
              parenthesisLevel--;
              currentMF += ')';
              break;
            default:
              throw Error(`can not preprocess ${ranges}`);
          }
        }
        if (currentMF) {
          newRanges.push({
            mf: currentMF
          });
        }
        ranges = newRanges;
      }
      let possibilities = [];
      for (const range of ranges) {
        if (range.max === 0) continue;
        let max = range.max === undefined ? 1 : range.max;
        possibilities.push(`${range.mf}0-${max}`);
      }
      return possibilities.join(' ');
    }

    /**
     * @param {object}         [entry={}]
     * @param {object}         [options={}]
     * @param {object}         [options.ionization={ mf: '', em: 0, charge: 0 }] - ionization method
     * @param {boolean}        [options.forceIonization=false] - If true ignore existing ionizations
     * @param {number}         [options.precision=1000] - The precision on the experimental mass
     * @param {number}         [options.targetMass] - Target mass, allows to calculate error and filter results
     * @param {number[]}       [options.targetMasses] - Target masses: SORTED array of numbers
     * @param {number[]}       [options.targetIntensities] - Target intensities: SORTED array of numbers
     * @param {number}         [options.minEM=0] - Minimal monoisotopic mass
     * @param {number}         [options.maxEM=+Infinity] - Maximal monoisotopic mass
     * @param {number}         [options.minMSEM=0] - Minimal monoisotopic mass observed by mass
     * @param {number}         [options.maxMSEM=+Infinity] - Maximal monoisotopic mass observed by mass
     * @param {number}         [options.minCharge=-Infinity] - Minimal charge
     * @param {number}         [options.maxCharge=+Infinity] - Maximal charge
     * @param {boolean}        [options.absoluteCharge=false] - If true, the charge is absolute (so between 0 and +Infinity by default)
     * @param {boolean}        [options.allowNegativeAtoms=false] - Allow to have negative number of atoms
     * @param {object}         [options.unsaturation={}]
     * @param {number}         [options.unsaturation.min=-Infinity] - Minimal unsaturation
     * @param {number}         [options.unsaturation.max=+Infinity] - Maximal unsaturation
     * @param {boolean}        [options.unsaturation.onlyInteger=false] - Integer unsaturation
     * @param {boolean}        [options.unsaturation.onlyNonInteger=false] - Non integer unsaturation
     * @param {boolean}        [options.atoms] - object of atom:{min, max}
     * @param {Function}       [options.callback] - a function that contains information about the current MF
     * @return {boolean}
     */

    /**
     * We always recalculate msem
     */

    function msemMatcher(entry, options = {}) {
      const {
        ionization = {
          mf: '',
          em: 0,
          charge: 0,
          atoms: {}
        },
        forceIonization = false,
        precision = 1000,
        minCharge = Number.MIN_SAFE_INTEGER,
        maxCharge = Number.MAX_SAFE_INTEGER,
        absoluteCharge = false,
        unsaturation = {},
        targetMass,
        // if present we will calculate the errors
        targetMasses,
        // if present we will calculate the smallest error
        targetIntensities,
        // if present it will be added in the report
        minEM = -Infinity,
        maxEM = +Infinity,
        minMSEM = -Infinity,
        maxMSEM = +Infinity,
        allowNegativeAtoms = false,
        atoms,
        callback
      } = options;
      let msInfo = getMsInfo(entry, {
        ionization,
        forceIonization,
        targetMass
      });
      let ms = msInfo.ms;
      if (entry.em !== undefined) {
        if (entry.em < minEM || entry.em > maxEM) return false;
      }
      if (ms.em !== undefined) {
        if (ms.em < minMSEM || ms.em > maxMSEM) return false;
      }
      if (targetMass && Math.abs(ms.ppm) > precision) return false;
      if (ms.charge !== undefined) {
        let charge = absoluteCharge ? Math.abs(ms.charge) : ms.charge;
        if (charge < minCharge || charge > maxCharge) return false;
      }
      if (unsaturation !== undefined && entry.unsaturation !== undefined) {
        if (!unsaturationMatcher(entry, unsaturation)) {
          return false;
        }
      }
      if (entry.atoms !== undefined && atoms) {
        // all the atoms of the entry must fit in the range
        for (let atom in entry.atoms) {
          if (!atoms[atom]) return false;
          if (entry.atoms[atom] < atoms[atom].min) return false;
          if (entry.atoms[atom] > atoms[atom].max) return false;
        }
      }
      if (entry.atoms !== undefined && !allowNegativeAtoms) {
        const ionizationAtoms = msInfo.ionization && msInfo.ionization.atoms || {};
        const atomKeys = new Set(Object.keys(ionizationAtoms).concat(Object.keys(entry.atoms)));
        for (let atom of atomKeys) {
          if ((entry.atoms[atom] || 0) + (ionizationAtoms[atom] || 0) < 0) {
            return false;
          }
        }
      }
      if (targetMasses && targetMasses.length > 0) {
        let index = xFindClosestIndex(targetMasses, ms.em);
        let closestMass = targetMasses[index];
        msInfo = getMsInfo(entry, {
          ionization,
          forceIonization,
          targetMass: closestMass
        });
        msInfo.ms.target = {
          mass: closestMass
        };
        if (targetIntensities) {
          msInfo.ms.target.intensity = targetIntensities[index];
        }
        // need to find the closest targetMasses
        if (Math.abs(msInfo.ms.ppm) > precision) return false;
      }
      if (callback) {
        if (!callback(entry)) return false;
      }
      return msInfo;
    }

    /**
     * returns all the possible neutral mass for a defined experimental (targetMass) mass
     */
    class TargetMassCache {
      /**
       * @param {number} targetMass
       * @param {Array} possibilities
       * @param {object} [options={}]
       * @param {boolean} [options.allowNeutral=false]
       * @param {number} [options.charge=0]
       * @param {number} [options.precision=100]
       * @param {object} [options.filter={}]
       * @param {number} [options.filter.minCharge=Number.MIN_SAFE_INTEGER]
       * @param {number} [options.filter.maxCharge=Number.MAX_SAFE_INTEGER]
       * @returns
       */
      constructor(targetMass, possibilities, options = {}) {
        const {
          allowNeutral = false,
          // msem because em in this case !
          filter = {},
          charge = 0,
          precision = 100
        } = options;
        const {
          minCharge = Number.MIN_SAFE_INTEGER,
          maxCharge = Number.MAX_SAFE_INTEGER
        } = filter;
        if (!possibilities || possibilities.length === 0) return;
        let firstPossibility = possibilities[0];
        let currentMinCharge = Math.max(minCharge, firstPossibility.minCharge + charge);
        let currentMaxCharge = Math.min(maxCharge, firstPossibility.maxCharge + charge);
        this.minCharge = currentMinCharge;
        this.maxCharge = currentMaxCharge;
        let size = this.maxCharge - this.minCharge + 1;
        this.data = [];
        let minMass = 0;
        let maxMass = 0;
        let range = targetMass * precision / 1e6;
        for (let i = 0; i < size; i++) {
          let currentCharge = i + this.minCharge;
          if (currentCharge === 0) {
            if (allowNeutral) {
              minMass = targetMass - range;
              maxMass = targetMass + range;
            } else {
              minMass = Number.MAX_SAFE_INTEGER;
              maxMass = Number.MIN_SAFE_INTEGER;
            }
          } else {
            minMass = (targetMass - range) * Math.abs(currentCharge) + ELECTRON_MASS * currentCharge;
            maxMass = (targetMass + range) * Math.abs(currentCharge) + ELECTRON_MASS * currentCharge;
          }
          this.data.push({
            charge: currentCharge,
            minMass,
            maxMass
          });
        }
      }
      getMinMass(charge) {
        return this.data[charge - this.minCharge] ? this.data[charge - this.minCharge].minMass : Number.MAX_SAFE_INTEGER;
      }
      getMaxMass(charge) {
        return this.data[charge - this.minCharge] ? this.data[charge - this.minCharge].maxMass : Number.MIN_SAFE_INTEGER;
      }
    }

    /**
     * @param {number}        targetMass - Monoisotopic mass
     * @param {object}        [options={}]
     * @param {number}        [options.maxIterations=10000000] - Maximum number of iterations
     * @param {boolean}       [options.allowNeutral=true]
     * @param {boolean}       [options.uniqueMFs=true]
     * @param {number}        [options.limit=1000] - Maximum number of results
     * @param {string}        [options.ionizations=''] - string containing a comma separated list of modifications
     * @param {string|{mf:string,min:number,max:number}[]}        [options.ranges='C0-100 H0-100 O0-100 N0-100'] - range of mfs to search
     * @param {number}        [options.precision=100] - Allowed mass range based on precision
     * @param {object}        [options.filter={}]
     * @param {number}        [options.filter.minCharge=-Infinity] - Minimal charge
     * @param {number}        [options.filter.maxCharge=+Infinity] - Maximal charge
     * @param {boolean}       [options.filter.absoluteCharge=false] - If true, the charge is absolute (so between 0 and +Infinity by default)
     * @param {object}        [options.filter.unsaturation={}]
     * @param {number}        [options.filter.unsaturation.min=-Infinity] - Minimal unsaturation
     * @param {number}        [options.filter.unsaturation.max=+Infinity] - Maximal unsaturation
     * @param {boolean}       [options.filter.unsaturation.onlyInteger=false] - Integer unsaturation
     * @param {boolean}       [options.filter.unsaturation.onlyNonInteger=false] - Non integer unsaturation
     * @param {object}        [options.filter.atoms] - object of atom:{min, max}
     * @param {function}      [options.filter.callback] - a function to filter the MF
     * @returns {object}
     */

    function findMFsSync(targetMass, options = {}) {
      const {
        filter = {},
        maxIterations = 1e8,
        limit = 1000,
        allowNeutral = true,
        // if there is no msem we use em !
        uniqueMFs = false,
        // if there is no msem we use em !
        ranges = [{
          mf: 'C',
          min: 0,
          max: 100
        }, {
          mf: 'H',
          min: 0,
          max: 100
        }, {
          mf: 'O',
          min: 0,
          max: 100
        }, {
          mf: 'N',
          min: 0,
          max: 100
        }]
      } = options;
      let targetMassCache;
      const {
        minCharge = Number.MIN_SAFE_INTEGER,
        maxCharge = Number.MAX_SAFE_INTEGER,
        unsaturation = {}
      } = filter;
      let filterUnsaturation = !!unsaturation;
      // we calculate not the real unsaturation but the one before dividing by 2 + 1
      let fakeMinUnsaturation = unsaturation.min === undefined ? Number.MIN_SAFE_INTEGER : (unsaturation.min - 1) * 2;
      let fakeMaxUnsaturation = unsaturation.max === undefined ? Number.MAX_SAFE_INTEGER : (unsaturation.max - 1) * 2;
      let filterCharge = minCharge !== Number.MIN_SAFE_INTEGER || maxCharge !== Number.MAX_SAFE_INTEGER;
      let advancedFilter;
      if (filter.atoms || filter.callback) {
        advancedFilter = {
          atoms: filter.atoms,
          callback: filter.callback
        };
      }
      let result = {
        mfs: [],
        info: {
          numberMFEvaluated: 0,
          numberResults: 0
        }
      };
      let orderMapping = []; // used to sort the atoms

      // we need to make the processing for all the ionizations
      let ionizations = preprocessIonizations(options.ionizations);
      for (let ionization of ionizations) {
        let currentIonization = {
          currentMonoisotopicMass: ionization.em || 0,
          currentCharge: ionization.charge,
          currentUnsaturation: 0 // we don't take into account the unsaturation of the ionization agent
        };
        // if (DEBUG) console.log('new ionization', ionization.mf, ionization.em, ionization.charge);
        // ionization em and charge will be used to set the first atom value
        let possibilities = preprocessRanges(ranges);
        orderMapping = getOrderMapping$1(possibilities);
        if (possibilities.length === 0) return {
          mfs: []
        };
        targetMassCache = new TargetMassCache(targetMass, possibilities, {
          ...options,
          charge: ionization.charge
        });
        let theEnd = false;
        let maxPosition = possibilities.length;
        let lastPosition = possibilities.length - 1;
        let currentPosition = 0;
        let currentAtom;
        let previousAtom;
        let lastPossibility = possibilities[lastPosition];
        initializePossibilities(possibilities, currentIonization, targetMassCache);

        //  if (DEBUG) console.log('possibilities', possibilities.map((a) => `${a.mf + a.originalMinCount}-${a.originalMaxCount}`));

        let isValid = false; // designed so that the first time it is not a valid solution
        while (!theEnd) {
          if (result.info.numberMFEvaluated++ > maxIterations) {
            throw Error(`Iteration number is over the current maximum of: ${maxIterations}`);
          }
          if (filterUnsaturation) {
            let unsaturationValue = lastPossibility.currentUnsaturation;
            let isOdd = Math.abs(unsaturationValue % 2);
            if (unsaturation.onlyInteger && isOdd === 1 || unsaturation.onlyNonInteger && isOdd === 0 || fakeMinUnsaturation > unsaturationValue || fakeMaxUnsaturation < unsaturationValue) {
              isValid = false;
            }
          }
          if (filterCharge && (lastPossibility.currentCharge < minCharge || lastPossibility.currentCharge > maxCharge)) {
            isValid = false;
          }
          if (isValid) {
            let minMass = targetMassCache.getMinMass(lastPossibility.currentCharge);
            let maxMass = targetMassCache.getMaxMass(lastPossibility.currentCharge);
            if (lastPossibility.currentMonoisotopicMass < minMass || lastPossibility.currentMonoisotopicMass > maxMass) {
              isValid = false;
            }
          }
          if (isValid) {
            result.info.numberResults++;
            let newResult = getResult$1(possibilities, targetMass, allowNeutral, ionization, orderMapping);
            if (advancedFilter) {
              isValid = msemMatcher(newResult, advancedFilter) !== false;
            }
            if (isValid) {
              result.mfs.push(newResult);
              if (result.mfs.length > 2 * limit) {
                if (uniqueMFs) ensureUniqueMF(result);
                result.mfs.sort((a, b) => Math.abs(a.ms.ppm) - Math.abs(b.ms.ppm));
                result.mfs.length = limit;
              }
            }
          }
          isValid = true;
          // we need to setup all the arrays if possible
          while (currentPosition < maxPosition && currentPosition >= 0) {
            currentAtom = possibilities[currentPosition];
            previousAtom = currentPosition === 0 ? currentIonization : possibilities[currentPosition - 1];
            if (currentAtom.currentCount < currentAtom.currentMaxCount) {
              currentAtom.currentCount++;
              updateCurrentAtom(currentAtom, previousAtom);
              if (currentPosition < lastPosition) {
                currentPosition++;
                setCurrentMinMax(possibilities[currentPosition], possibilities[currentPosition - 1], targetMassCache);
              } else {
                break;
              }
            } else {
              currentPosition--;
            }
          }
          if (currentPosition < 0) {
            theEnd = true;
          }
        }
      }
      if (uniqueMFs) ensureUniqueMF(result);
      result.mfs.sort((a, b) => Math.abs(a.ms.ppm) - Math.abs(b.ms.ppm));
      if (result.mfs.length > limit) {
        result.mfs.length = limit;
      }
      result.mfs.forEach(mf => delete mf.currentCounts);
      return result;
    }

    /**
     * Ensure that we have only once the same MF
     * In order to improve the speed we just consider the em
     * @param {object} result
     */
    function ensureUniqueMF(result) {
      result.mfs.sort((a, b) => a.em - b.em);
      let previousEM = 0;
      let bestCounts = [];
      const mfs = [];
      next: for (let current of result.mfs) {
        if (current.em - previousEM > 1e-8) {
          previousEM = current.em;
          bestCounts = current.currentCounts;
          mfs.push(current);
        } else {
          for (let i = 0; i < current.currentCounts.length; i++) {
            // better priority ???
            if (current.currentCounts[i] > bestCounts[i]) {
              mfs.pop();
              mfs.push(current);
              bestCounts = current.currentCounts;
              continue;
            } else if (current.currentCounts[i] < bestCounts[i]) {
              continue next;
            }
          }
        }
      }
      result.mfs = mfs;
    }
    function updateCurrentAtom(currentAtom, previousAtom) {
      currentAtom.currentMonoisotopicMass = previousAtom.currentMonoisotopicMass + currentAtom.em * currentAtom.currentCount;
      currentAtom.currentCharge = previousAtom.currentCharge + currentAtom.charge * currentAtom.currentCount;
      currentAtom.currentUnsaturation = previousAtom.currentUnsaturation + currentAtom.unsaturation * currentAtom.currentCount;
    }
    function getResult$1(possibilities, targetMass, allowNeutralMolecules, ionization, orderMapping) {
      let lastPossibility = possibilities[possibilities.length - 1];
      let result = {
        em: lastPossibility.currentMonoisotopicMass - ionization.em,
        unsaturation: lastPossibility.currentUnsaturation,
        mf: '',
        charge: lastPossibility.currentCharge - ionization.charge,
        ionization,
        atoms: {},
        groups: {},
        currentCounts: possibilities.map(possibility => possibility.currentCount)
      };

      // we check that the first time we meet the ionization group it does not end
      // in the final result

      for (let i = 0; i < possibilities.length; i++) {
        let possibility = possibilities[orderMapping[i]];
        if (possibility.currentCount !== 0) {
          if (possibility.isGroup) {
            if (possibility.currentCount === 1) {
              result.mf += `${possibility.mf}`;
            } else if (possibility.mf.match(/^\([^()]*\)$/)) {
              result.mf += `${possibility.mf}${possibility.currentCount}`;
            } else {
              result.mf += `(${possibility.mf})${possibility.currentCount}`;
            }
            if (result.groups[possibility.mf]) {
              result.groups[possibility.mf] += possibility.currentCount;
            } else {
              result.groups[possibility.mf] = possibility.currentCount;
            }
          } else {
            result.mf += possibility.mf;
            if (possibility.currentCount !== 1) {
              result.mf += possibility.currentCount;
            }
          }
          for (let atom in possibility.atoms) {
            if (result.atoms[atom]) {
              result.atoms[atom] += possibility.atoms[atom] * possibility.currentCount;
            } else {
              result.atoms[atom] = possibility.atoms[atom] * possibility.currentCount;
            }
          }
        }
      }
      result.unsaturation = (result.unsaturation + Math.abs(result.charge)) / 2 + 1;
      result.ms = getMsInfo(result, {
        targetMass,
        allowNeutralMolecules
      }).ms;
      return result;
    }
    function setCurrentMinMax(currentAtom, previousAtom, targetMassCache) {
      // the current min max can only be optimize if the charge will not change anymore
      // and the currentAtom.em >= 0
      if (currentAtom.innerCharge === true || currentAtom.charge !== 0 || currentAtom.em < 0) {
        currentAtom.currentMinCount = currentAtom.originalMinCount;
        currentAtom.currentMaxCount = currentAtom.originalMaxCount;
        currentAtom.currentCount = currentAtom.currentMinCount - 1;
      } else {
        // no more change of charge, we can optimize
        let currentMass = previousAtom !== undefined ? previousAtom.currentMonoisotopicMass : 0;
        let currentCharge = previousAtom !== undefined ? previousAtom.currentCharge : 0;
        currentAtom.currentMinCount = Math.max(Math.floor((targetMassCache.getMinMass(currentCharge) - currentMass - currentAtom.maxInnerMass) / currentAtom.em), currentAtom.originalMinCount);
        currentAtom.currentMaxCount = Math.min(Math.floor((targetMassCache.getMaxMass(currentCharge) - currentMass - currentAtom.minInnerMass) / currentAtom.em), currentAtom.originalMaxCount);
        currentAtom.currentCount = currentAtom.currentMinCount - 1;
      }
    }
    function initializePossibilities(possibilities, currentIonization, targetMassCache) {
      for (let i = 0; i < possibilities.length; i++) {
        if (i === 0) {
          updateCurrentAtom(possibilities[i], currentIonization);
          setCurrentMinMax(possibilities[i], currentIonization, targetMassCache);
        } else {
          updateCurrentAtom(possibilities[i], possibilities[i - 1]);
        }
      }
    }
    function getOrderMapping$1(possibilities) {
      let mapping = possibilities.map((p, i) => ({
        atom: p.mf,
        index: i
      }));
      mapping.sort((a, b) => {
        return atomSorter(a.atom, b.atom);
      });
      return mapping.map(a => a.index);
    }

    /**
     * @param {number}        targetMass - Monoisotopic mass
     * @param {object}        [options={}]
     * @param {number}        [options.maxIterations=10000000] - Maximum number of iterations
     * @param {boolean}       [options.allowNeutral=true]
     * @param {boolean}       [options.uniqueMFs=true]
     * @param {number}        [options.limit=1000] - Maximum number of results
     * @param {string|{mf:string,min:number,max:number}[]}        [options.ionizations=''] - string containing a comma separated list of modifications
     * @param {string}        [options.ranges='C0-100 H0-100 O0-100 N0-100'] - range of mfs to search
     * @param {number}        [options.precision=100] - Allowed mass range based on precision
     * @param {object}        [options.filter={}]
     * @param {number}        [options.filter.minCharge=-Infinity] - Minimal charge
     * @param {number}        [options.filter.maxCharge=+Infinity] - Maximal charge
     * @param {boolean}       [options.filter.absoluteCharge=false] - If true, the charge is absolute (so between 0 and +Infinity by default)
     * @param {object}        [options.filter.unsaturation={}]
     * @param {number}        [options.filter.unsaturation.min=-Infinity] - Minimal unsaturation
     * @param {number}        [options.filter.unsaturation.max=+Infinity] - Maximal unsaturation
     * @param {boolean}       [options.filter.unsaturation.onlyInteger=false] - Integer unsaturation
     * @param {boolean}       [options.filter.unsaturation.onlyNonInteger=false] - Non integer unsaturation
     * @param {object}        [options.filter.atoms] - object of atom:{min, max}
     * @param {function}      [options.filter.callback] - a function to filter the MF
     * @returns {Promise}
     */

    async function findMFs(targetMass, options = {}) {
      return findMFsSync(targetMass, options);
    }

    /**
     *
     * @param {object}   experimentalSpectrum
     * @param {object}   database
     * @param {object}   [options={}]
     * @param {function} [options.onStep] - Callback to do after each step
     * @param {string}   [options.ionizations=''] - string containing a comma separated list of modifications
     * @param {number}   [options.precision=100] - Allowed mass range based on precision
     */
    async function appendFragmentsInfo(experimentalSpectrum, database, options = {}) {
      const {
        ionizations,
        onStep,
        precision
      } = options;
      if (!experimentalSpectrum) {
        throw new Error('Experimental spectrum is not defined');
      }
      if (!database) {
        throw new Error('Database is not defined');
      }
      const peaks = experimentalSpectrum.getPeaks({
        sumValue: 1
      });
      for (let entry of database) {
        const ranges = Object.keys(entry.atoms).map(atom => `${atom}0-${entry.atoms[atom]}`).join(' ');
        entry.fragments = {
          nbFound: 0,
          intensityFound: 0,
          assignments: []
        };
        for (let i = 0; i < peaks.length; i++) {
          if (onStep) await onStep(i);
          const peak = peaks[i];
          const possibleMFs = await findMFs(peak.x, {
            ionizations,
            precision,
            ranges
          });
          if (possibleMFs.mfs.length > 0) {
            entry.fragments.nbFound++;
            entry.fragments.intensityFound += peak.y;
            entry.fragments.assignments.push({
              peak,
              bestMF: possibleMFs.mfs[0]
            });
          }
        }
      }
    }

    /**
     * Modify object a to join it with b and make the sum of each of the keys
     * @param {*} a
     * @param {*} source1
     *
     * @return {object}
     */

    function sum(target) {
      for (var i = 1; i < arguments.length; i++) {
        let toSum = arguments[i];
        for (var key of Object.keys(toSum)) {
          if (target[key]) {
            target[key] += toSum[key];
          } else {
            target[key] = toSum[key];
          }
        }
      }
      return target;
    }
    var lib = sum;
    var sum$1 = lib;

    /**
     * Generate all the possible combinations of molecular formula and calculate
     * for each of them the monoisotopic mass and observed monoisotopic mass (m/z)
     * In the molecular formula there may be a comment after the '$' symbol
     *
     * @param {Array}         ranges
     * @param {object}        [options={}]
     * @param {number}        [options.limit=10000000] - Maximum number of results
     * @param {boolean}       [options.estimate=false] - estimate the number of MF without filters
     * @param {boolean}       [options.canonizeMF=true] - Canonize molecular formula
     * @param {boolean}       [options.uniqueMFs=true] - Force canonization and make MF unique
     * @param {string}        [options.ionizations=''] - Comma separated list of ionizations (to charge the molecule)
     * @param {function}      [options.onStep] - Callback to do after each step
     * @param {object}        [options.filter={}]
     * @param {number}        [options.filter.minMass=0] - Minimal monoisotopic mass
     * @param {number}        [options.filter.maxMass=+Infinity] - Maximal monoisotopic mass
     * @param {number}        [options.filter.minEM=0] - Minimal neutral monoisotopic mass
     * @param {number}        [options.filter.maxEM=+Infinity] - Maximal neutral monoisotopic mass
     * @param {number}        [options.filter.precision=1000] - The precision on the experimental mass
     * @param {number}        [options.filter.targetMass] - Target mass, allows to calculate error and filter results
     * @param {number[]}      [options.filter.targetMasses] - Target masses: SORTED array of numbers
     * @param {number}        [options.filter.precision=1000] - Precision
     * @param {number}        [options.filter.minCharge=-Infinity] - Minimal charge
     * @param {number}        [options.filter.maxCharge=+Infinity] - Maximal charge
     * @param {boolean}       [options.filter.absoluteCharge=false] - If true, the charge is absolute (so between 0 and +Infinity by default)
     * @param {boolean}       [options.filter.allowNegativeAtoms=false] - Allow to have negative number of atoms
     * @param {object}        [options.filter.unsaturation={}]
     * @param {number}        [options.filter.unsaturation.min=-Infinity] - Minimal unsaturation
     * @param {number}        [options.filter.unsaturation.max=+Infinity] - Maximal unsaturation
     * @param {boolean}       [options.filter.unsaturation.onlyInteger=false] - Integer unsaturation
     * @param {boolean}       [options.filter.unsaturation.onlyNonInteger=false] - Non integer unsaturation
     * @param {object}        [options.filter.atoms] - object of atom:{min, max}
     * @param {function}      [options.filter.callback] - a function to filter the MF
     * @param {string}        [options.filterFct]
     * @param {object}        [options.links]
     * @param {boolean}       [options.links.filter] We filter all the MF that do not match the '*X'
     * @returns {Promise}
     */

    async function generateMFs(ranges, options = {}) {
      if (!Array.isArray(ranges)) {
        throw new Error('Ranges must be an array of string or object');
      }
      options = {
        ...options
      };
      let {
        limit = 100000,
        uniqueMFs = true,
        estimate = false,
        onStep
      } = options;
      options.filterFctVariables = {};
      for (let i = 0; i < ranges.length; i++) {
        const range = ranges[i];
        if (typeof range === 'object' && range.name) {
          options.filterFctVariables[range.name] = i;
          ranges[i] = range.value;
        }
      }
      if (options.filterFct) {
        // we create a real javascript function
        let variables = Object.keys(options.filterFctVariables);
        variables.push('mm', 'mz', 'charge', 'unsaturation', 'atoms');
        // eslint-disable-next-line no-new-func
        options.filterFct = new Function(...variables, `return ${options.filterFct}`);
      }
      if (uniqueMFs === true) options.canonizeMF = true;
      if (options.canonizeMF === undefined) options.canonizeMF = true;
      options.ionizations = preprocessIonizations(options.ionizations);

      // we allow String delimited by ". or ;" instead of an array
      for (let i = 0; i < ranges.length; i++) {
        if (!Array.isArray(ranges[i])) {
          ranges[i] = ranges[i].split(/[.,]/);
        }
      }

      // we allow ranges in a string ...
      // problem with ranges is that we need to know to what the range applies
      for (let i = 0; i < ranges.length; i++) {
        let parts = ranges[i];
        let newParts = [];
        for (let j = 0; j < parts.length; j++) {
          let part = parts[j];
          if (part.match(/[0-9]-[0-9-]/)) {
            // deal with negative numbers
            // there are ranges ... we are in trouble !
            newParts = newParts.concat(new MF(part).flatten({
              groupIdentical: false,
              limit
            }));
          } else {
            newParts.push(parts[j]); // the part with the comments !
          }
        }

        ranges[i] = newParts;
      }
      if (estimate) {
        let total = ranges.reduce((previous, current) => previous * current.length, 1);
        return total * options.ionizations.length;
      }
      let results = [];
      let sizes = [];
      let currents = [];
      for (let i = 0; i < ranges.length; i++) {
        sizes.push(ranges[i].length - 1);
        currents.push(0);
      }
      let position = 0;
      let evolution = 0;
      while (position < currents.length) {
        if (currents[position] < sizes[position]) {
          if (onStep) await onStep(evolution);
          evolution++;
          appendResult(results, currents, ranges, options);
          currents[position]++;
          for (let i = 0; i < position; i++) {
            currents[i] = 0;
          }
          position = 0;
        } else {
          position++;
        }
        if (evolution > limit) {
          throw new Error(`You have reached the limit of ${limit}. You could still change this value using the limit option but it is likely to crash.`);
        }
      }
      appendResult(results, currents, ranges, options);
      if (uniqueMFs) {
        let uniqueMFsObject = {};
        results.forEach(result => {
          uniqueMFsObject[result.mf + result.ionization.mf] = result;
        });
        results = Object.keys(uniqueMFsObject).map(k => uniqueMFsObject[k]);
      }
      results.sort((a, b) => a.em - b.em);
      return results;
    }
    let ems = {};

    // internal method used as a cache
    function getMonoisotopicMass(mfString) {
      if (!ems[mfString]) {
        // we need to calculate based on the mf but not very often ...
        let mf = new MF(mfString);
        let info = mf.getInfo();
        ems[mfString] = {
          em: info.monoisotopicMass,
          charge: info.charge,
          mw: info.mass,
          unsaturation: (info.unsaturation - 1) * 2,
          atoms: info.atoms
        };
      }
      return ems[mfString];
    }
    function getEMFromParts(parts, currents, ionization) {
      let charge = 0;
      let em = 0;
      let mw = 0;
      let unsaturation = 0;
      let validUnsaturation = true;
      let atoms = {};
      for (let i = 0; i < parts.length; i++) {
        let part = parts[i][currents[i]];
        if (part) {
          let info = getMonoisotopicMass(part);
          charge += info.charge;
          em += info.em;
          mw += info.mw;
          sum$1(atoms, info.atoms);
          if (info.unsaturation && validUnsaturation) {
            unsaturation += info.unsaturation;
          }
        }
      }
      return {
        charge,
        em,
        mw,
        ionization,
        unsaturation: unsaturation / 2 + 1 ,
        atoms
      };
    }
    function appendResult(results, currents, keys, options = {}) {
      const {
        canonizeMF,
        filter,
        ionizations,
        links = {}
      } = options;
      // this script is designed to combine molecular formula
      // that may contain comments after a "$" sign
      // therefore we should put all the comments at the ned

      if (links.filter) {
        let sharps = [];
        for (let i = 0; i < keys.length; i++) {
          let anchors = keys[i][currents[i]].match(/#[0-9]+/g);
          if (anchors) sharps.push(...anchors);
        }
        if (sharps.length % 2 === 1) return;
        sharps = sharps.sort();
        for (let i = 0; i < sharps.length; i += 2) {
          if (sharps[i] !== sharps[i + 1]) return;
        }
      }
      for (let ionization of ionizations) {
        let result = getEMFromParts(keys, currents, ionization);
        if (options.filterFct) {
          let variables = [];
          for (let key in options.filterFctVariables) {
            variables.push(currents[options.filterFctVariables[key]]);
          }
          variables.push(result.em, (result.em + ionization.em - ionization.charge * ELECTRON_MASS) / Math.abs(ionization.charge), result.charge + result.ionization.charge, result.unsaturation, result.atoms);
          if (!options.filterFct.apply(null, variables)) continue;
        }
        result.parts = [];
        result.mf = '';
        let comments = [];
        for (let i = 0; i < keys.length; i++) {
          let key = keys[i][currents[i]];
          if (key) {
            if (key.indexOf('$') > -1) {
              comments.push(key.replace(/^[^$]*\$/, ''));
              key = key.replace(/\$.*/, '');
            }
            result.parts[i] = key;
            result.mf += key;
          }
        }
        if (comments.length > 0) {
          result.comment = comments.join(' ');
        }
        let match = msemMatcher(result, filter);
        if (!match) continue;
        result.ms = match.ms;
        result.ionization = match.ionization;
        if (canonizeMF) {
          result.mf = new MF(result.mf).toMF();
        }
        results.push(result);
      }
    }

    /**
     * Generates a database 'generated' from an array of molecular formula
     * @param {array} mfsArray - Array of string or Array of array containing the parts to combine
     * @param {object} [options={}]
     * @param {boolean} [options.estimate=false] - estimate the number of MF without filters
     * @param {string} [options.databaseName='generated']
     * @param {function} [options.onStep] - Callback to do after each step
     * @param {number} [options.limit=10000000] - Maximum number of results
     * @param {boolean} [options.canonizeMF=true] - Canonize molecular formula
     * @param {boolean} [options.uniqueMFs=true] - Force canonization and make MF unique
     * @param {string} [options.ionizations=''] - Comma separated list of ionizations (to charge the molecule)
     * @param {object} [options.filter={}]
     * @param {number} [options.filter.minMass=0] - Minimal monoisotopic mass
     * @param {number} [options.filter.maxMass=+Infinity] - Maximal monoisotopic mass
     * @param {number} [options.filter.minEM=0] - Minimal neutral monoisotopic mass
     * @param {number} [options.filter.maxEM=+Infinity] - Maximal neutral monoisotopic mass
     * @param {number} [options.filter.minMSEM=0] - Minimal observed monoisotopic mass
     * @param {number} [options.filter.maxMSEM=+Infinity] - Maximal observed monoisotopic mass
     * @param {number} [options.filter.minCharge=-Infinity] - Minimal charge
     * @param {number} [options.filter.maxCharge=+Infinity] - Maximal charge
     * @param {boolean}  [options.filter.absoluteCharge=false] - If true, the charge is absolute (so between 0 and +Infinity by default)
     * @param {object} [options.filter.unsaturation={}]
     * @param {number} [options.filter.unsaturation.min=-Infinity] - Minimal unsaturation
     * @param {number} [options.filter.unsaturation.max=+Infinity] - Maximal unsaturation
     * @param {boolean} [options.filter.unsaturation.onlyInteger=false] - Integer unsaturation
     * @param {boolean} [options.filter.unsaturation.onlyNonInteger=false] - Non integer unsaturation
     * @param {object} [options.filter.atoms] - object of atom:{min, max}
     * @param {function} [options.filter.callback] - a function to filter the MF
     * @param {string}  [options.filterFct] - A string representing a function
     *
     * @example
     *
     * const {EMDB} = require('emdb');
     * let emdb = new EMDB();
     * let array = ['C1-10', 'H1-10'];
     * emdb.fromArray(array); // create a database 'generated' combining all possibilies
     * console.log(emdb.get('generated').length); // 100
     *
     * @example
     * const {EMDB} = require('emdb');
     * let emdb = new EMDB();
     * let array = ['C1-10 H1-10'];
     * emdb.fromArray(array); // create a database 'generated' combining all possibilies
     * console.log(emdb.get('generated').length); // 100
     *
     * @example
     * const {EMDB} = require('emdb');
     * let emdb = new EMDB();
     * // in case of an array of array, one of the group is allwed
     * let array = [['C1-10','H1-10'],'Cl0-1 Br0-1'];
     * emdb.fromArray(array); // create a database 'generated' combining all possibilies
     * console.log(emdb.get('generated').length); // 80
     *
     * @example
     * <script src="https://www.lactame.com/lib/mass-tools/HEAD/mass-tools.js" />
     * <script>
     *   let emdb = new MassTools.EMDB();
     *   let array = ['C1-10', 'H1-10'];
     *   emdb.fromArray(array); // create a database 'generated' combining all possibilities
     *   console.log(emdb.get('generated').length); // 100
     * </script>
     *
     * // from the browser
     */

    async function fromArray(mfsArray, options = {}) {
      return generateMFs(mfsArray, options);
    }

    /**
     *
     * @param {import('openchemlib').Molecule} molecule An instance of a molecule
     * @param {object} [options={}]
     * @param {object} [options.OCL] openchemlib library
     */
    function makeRacemic(molecule) {
      const {
        Molecule
      } = molecule.getOCL();
      // if we don't calculate this we have 2 epimers
      molecule.ensureHelperArrays(Molecule.cHelperCIP);
      // we need to make one group "AND" for chiral (to force to racemic, this means diastereotopic and not enantiotopic)
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        if (molecule.getAtomParity(i) !== Molecule.cAtomParityNone) {
          molecule.setAtomESR(i, Molecule.cESRTypeAnd, 0); // changed to group 0; TLS 9.Nov.2015
        }
      }
      // after the change we need to recalculate the CIP
      molecule.ensureHelperArrays(Molecule.cHelperCIP);
    }

    let xAtomicNumber = 0;
    /**
     * Returns the atomic number of the X atom
     * @param {import('openchemlib').Molecule} molecule An instance of a molecule
     * @returns
     */
    function getXAtomicNumber(molecule) {
      if (!xAtomicNumber) {
        const OCL = molecule.getOCL();
        xAtomicNumber = OCL.Molecule.getAtomicNoFromLabel('X', OCL.Molecule.cPseudoAtomX);
      }
      return xAtomicNumber;
    }

    /**
     * Tag an atom to be able to visualize it
     */
    function tagAtom(molecule, iAtom) {
      const customLabel = `${molecule.getAtomLabel(iAtom)}*`;
      molecule.setAtomCustomLabel(iAtom, customLabel);
      if (molecule.getAtomicNo(iAtom) === 1) {
        molecule.setAtomicNo(iAtom, getXAtomicNumber(molecule));
      } else {
        // we can not use X because we would have problems with valencies if it is
        // expanded hydrogens or not
        // we can not only use a custom label because it does not count for the canonisation
        molecule.setAtomMass(iAtom, molecule.getAtomMass(iAtom) + 5);
      }
      return customLabel;
    }

    /**
     * Check if a specific atom is a sp3 carbon
     * @param {import('openchemlib').Molecule} molecule
     * @param {number} atomID
     */
    function isCsp3(molecule, atomID) {
      if (molecule.getAtomicNo(atomID) !== 6) return false;
      if (molecule.getAtomCharge(atomID) !== 0) return false;
      if (molecule.getImplicitHydrogens(atomID) + molecule.getConnAtoms(atomID) !== 4) {
        return false;
      }
      return true;
    }

    const FULL_HOSE_CODE$1 = 1;
    const HOSE_CODE_CUT_C_SP3_SP3 = 2;
    /**
     * Returns the hose code for specific atom numbers
     * @param {import('openchemlib').Molecule} molecule - The OCL molecule with expandedImplicitHydrogens and ensureHeterotopicChiralBonds
     * @param {object} [options={}]
     * @param {string[]} [options.allowedCustomLabels] Array of the custom labels that are considered as root atoms. By default all atoms having a customLabel
     * @param {number} [options.minSphereSize=0] Smallest hose code sphere
     * @param {number} [options.maxSphereSize=4] Largest hose code sphere
     * @param {number} [options.kind=FULL_HOSE_CODE] Kind of hose code, default usual sphere
     */
    function getHoseCodesForAtomsInternal(molecule, options = {}) {
      const OCL = molecule.getOCL();
      const {
        allowedCustomLabels,
        minSphereSize = 0,
        maxSphereSize = 4,
        kind = FULL_HOSE_CODE$1
      } = options;
      // this force reordering of atoms in order to have hydrogens at the end
      molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
      const rootAtoms = [];
      for (let j = 0; j < molecule.getAllAtoms(); j++) {
        if (allowedCustomLabels?.includes(molecule.getAtomCustomLabel(j)) || molecule.getAtomCustomLabel(j)) {
          rootAtoms.push(j);
        }
      }
      const fragment = new OCL.Molecule(0, 0);
      const results = [];
      let min = 0;
      let max = 0;
      const atomMask = new Array(molecule.getAllAtoms());
      const atomList = new Array(molecule.getAllAtoms());
      for (let sphere = 0; sphere <= maxSphereSize; sphere++) {
        if (max === 0) {
          for (const rootAtom of rootAtoms) {
            atomList[max] = rootAtom;
            atomMask[rootAtom] = true;
            max++;
          }
        } else {
          let newMax = max;
          for (let i = min; i < max; i++) {
            const atom = atomList[i];
            for (let j = 0; j < molecule.getAllConnAtoms(atom); j++) {
              const connAtom = molecule.getConnAtom(atom, j);
              if (!atomMask[connAtom]) {
                switch (kind) {
                  case FULL_HOSE_CODE$1:
                    atomMask[connAtom] = true;
                    atomList[newMax++] = connAtom;
                    break;
                  case HOSE_CODE_CUT_C_SP3_SP3:
                    if (!(isCsp3(molecule, atom) && isCsp3(molecule, connAtom))) {
                      atomMask[connAtom] = true;
                      atomList[newMax++] = connAtom;
                    }
                    break;
                  default:
                    throw new Error('getHoseCoesForAtom unknown kind');
                }
              }
            }
          }
          min = max;
          max = newMax;
        }
        molecule.copyMoleculeByAtoms(fragment, atomMask, true, null);
        if (sphere >= minSphereSize) {
          makeRacemic(fragment);
          results.push(fragment.getCanonizedIDCode(OCL.Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS));
        }
      }
      return results;
    }

    /**
     * Returns the atoms that are chiral or pseudo chiral.
     * There could be some issues if the original molecule lacks chiral bonds.
     * The function will add them and this could lead to some issues in the case of pseudochiral atoms.
     * @param {import('openchemlib').Molecule} molecule
     * @returns {number[]}
     */
    function getChiralOrHeterotopicCarbons(molecule) {
      const {
        Molecule
      } = molecule.getOCL();
      const xAtomicNumber = getXAtomicNumber(molecule);
      const internalMolecule = molecule.getCompactCopy();
      // hydrogens may be diastereotopic, we need to add them
      internalMolecule.addImplicitHydrogens();
      for (let i = 0; i < internalMolecule.getAllAtoms(); i++) {
        // hydrogens are not taken into account during canonization, we need to change them with an atom with a valence of 1
        if (internalMolecule.getAtomicNo(i) === 1) {
          internalMolecule.setAtomicNo(i, xAtomicNumber);
        }
      }
      addPossibleChiralBonds(internalMolecule);
      internalMolecule.ensureHelperArrays(Molecule.cHelperSymmetryStereoHeterotopicity);
      const atoms = [];
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        if (internalMolecule.getAtomicNo(i) === xAtomicNumber) {
          continue;
        }
        if (molecule.getAtomicNo(i) !== internalMolecule.getAtomicNo(i)) {
          throw new Error('getChiralOrHeterotopicCarbons: mismatching atomic numbers');
        }
        if (internalMolecule.getAtomicNo(i) !== 6) {
          continue;
        }
        const neighbourSymmetries = getNeighbourSymmetries(internalMolecule, i);
        if (neighbourSymmetries.length === 4) {
          atoms.push(i);
        }
      }
      return atoms;
    }
    function addPossibleChiralBonds(molecule) {
      const {
        Molecule
      } = molecule.getOCL();
      molecule.ensureHelperArrays(Molecule.cHelperSymmetryStereoHeterotopicity);
      for (let i = 0; i < molecule.getAtoms(); i++) {
        if (molecule.getAtomicNo(i) !== 6) continue;
        if (molecule.getStereoBond(i) >= 0) continue;
        const neighbourSymmetries = getNeighbourSymmetries(molecule, i);
        if (neighbourSymmetries.length <= 2) continue;
        const stereoBond = molecule.getAtomPreferredStereoBond(i);
        if (stereoBond !== -1) {
          molecule.setBondType(stereoBond, Molecule.cBondTypeUp);
          if (molecule.getBondAtom(1, stereoBond) === i) {
            const connAtom = molecule.getBondAtom(0, stereoBond);
            molecule.setBondAtom(0, stereoBond, i);
            molecule.setBondAtom(1, stereoBond, connAtom);
          }
          // To me it seems that we have to add all stereo centers into AND group 0. TLS 9.Nov.2015
          molecule.setAtomESR(i, Molecule.cESRTypeAnd, 0);
        }
      }
    }
    function getNeighbourSymmetries(molecule, iAtom) {
      const neighbourSymmetries = [];
      for (let j = 0; j < molecule.getAllConnAtoms(iAtom); j++) {
        const connAtom = molecule.getConnAtom(iAtom, j);
        const symmetryRank = molecule.getSymmetryRank(connAtom);
        if (!neighbourSymmetries.includes(symmetryRank)) {
          neighbourSymmetries.push(molecule.getSymmetryRank(connAtom));
        }
      }
      return neighbourSymmetries;
    }

    /**
     * This function will add missing chiral bonds on carbons ensure that all enantiotopic
     * or diastereotopic atoms can be identified uniquely
     * @param {import('openchemlib').Molecule} molecule
     * @param {object} [options={}]
     * @param {number} [options.esrType=Molecule.cESRTypeAnd]
     * @param {boolean} [options.atLeastThreeAtoms=true] - if true, only carbons with at least three atoms will be considered
     */
    function ensureHeterotopicChiralBonds(molecule, options = {}) {
      const {
        Molecule
      } = molecule.getOCL();
      const {
        esrType = Molecule.cESRTypeAnd,
        atLeastThreeAtoms = true
      } = options;
      molecule.ensureHelperArrays(Molecule.cHelperBitNeighbours);
      const heterotopicCarbons = getChiralOrHeterotopicCarbons(molecule);
      for (const i of heterotopicCarbons) {
        if (atLeastThreeAtoms && molecule.getAllConnAtoms(i) < 3) continue;
        if (molecule.getStereoBond(i) === -1) {
          const stereoBond = molecule.getAtomPreferredStereoBond(i);
          if (stereoBond !== -1) {
            molecule.setBondType(stereoBond, Molecule.cBondTypeUp);
            if (molecule.getBondAtom(1, stereoBond) === i) {
              const connAtom = molecule.getBondAtom(0, stereoBond);
              molecule.setBondAtom(0, stereoBond, i);
              molecule.setBondAtom(1, stereoBond, connAtom);
            }
            // To me it seems that we have to add all stereo centers into AND group 0. TLS 9.Nov.2015
            molecule.setAtomESR(i, esrType, 0);
          }
        }
      }
    }

    const FULL_HOSE_CODE = 1;
    /**
     * Returns the hose code for specific atom numbers
     * @param {import('openchemlib').Molecule} originalMolecule - The OCL molecule to be fragmented
     * @param {number[]} rootAtoms
     * @param {object} [options={}]
     * @param {number} [options.minSphereSize=0] Smallest hose code sphere
     * @param {number} [options.maxSphereSize=4] Largest hose code sphere
     * @param {number} [options.kind=FULL_HOSE_CODE] Kind of hose code, default usual sphere
     */
    function getHoseCodesForAtoms(originalMolecule, rootAtoms = [], options = {}) {
      const {
        minSphereSize = 0,
        maxSphereSize = 4,
        kind = FULL_HOSE_CODE
      } = options;
      const molecule = originalMolecule.getCompactCopy();
      // those 2 lines should be done only once
      molecule.addImplicitHydrogens();
      ensureHeterotopicChiralBonds(molecule);
      const allowedCustomLabels = [];
      for (const rootAtom of rootAtoms) {
        allowedCustomLabels.push(tagAtom(molecule, rootAtom));
      }
      return getHoseCodesForAtomsInternal(molecule, {
        minSphereSize,
        maxSphereSize,
        allowedCustomLabels,
        kind
      });
    }

    /**
     * Calculate the molecular formula in 'chemcalc' notation taking into account fragments, isotopes and charges
     * @param {import('openchemlib').Molecule} molecule an instance of OCL.Molecule
     * @returns {object}
     */
    function getMF(molecule) {
      const entries = molecule.getFragments();
      const result = {};
      let parts = [];
      const allAtoms = [];
      entries.forEach(entry => {
        const mf = getFragmentMF(entry, allAtoms);
        parts.push(mf);
      });
      const counts = {};
      for (const part of parts) {
        if (!counts[part]) counts[part] = 0;
        counts[part]++;
      }
      parts = [];
      for (const key of Object.keys(counts).sort()) {
        if (counts[key] > 1) {
          parts.push(counts[key] + key);
        } else {
          parts.push(key);
        }
      }
      result.parts = parts;
      result.mf = toMFString(allAtoms);
      return result;
    }
    function getFragmentMF(molecule, allAtoms) {
      const atoms = [];
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        const atom = {};
        atom.charge = molecule.getAtomCharge(i);
        atom.label = molecule.getAtomLabel(i);
        atom.mass = molecule.getAtomMass(i);
        atom.implicitHydrogens = molecule.getImplicitHydrogens(i);
        atoms.push(atom);
        allAtoms.push(atom);
      }
      return toMFString(atoms);
    }
    function toMFString(atoms) {
      let charge = 0;
      const mfs = {};
      for (const atom of atoms) {
        let label = atom.label;
        charge += atom.charge;
        if (atom.mass) {
          label = `[${atom.mass}${label}]`;
        }
        const mfAtom = mfs[label];
        if (!mfAtom) {
          mfs[label] = 0;
        }
        mfs[label] += 1;
        if (atom.implicitHydrogens) {
          if (!mfs.H) mfs.H = 0;
          mfs.H += atom.implicitHydrogens;
        }
      }
      let mf = '';
      const keys = Object.keys(mfs).sort(atomSorter);
      for (const key of keys) {
        mf += key;
        if (mfs[key] > 1) mf += mfs[key];
      }
      if (charge > 0) {
        mf += `(+${charge > 1 ? charge : ''})`;
      } else if (charge < 0) {
        mf += `(${charge < -1 ? charge : '-'})`;
      }
      return mf;
    }

    var papaparse_min = {exports: {}};

    /* @license
    Papa Parse
    v5.4.1
    https://github.com/mholt/PapaParse
    License: MIT
    */
    (function (module, exports) {
      !function (e, t) {
        module.exports = t() ;
      }(commonjsGlobal, function s() {

        var f = "undefined" != typeof self ? self : "undefined" != typeof window ? window : void 0 !== f ? f : {};
        var n = !f.document && !!f.postMessage,
          o = f.IS_PAPA_WORKER || !1,
          a = {},
          u = 0,
          b = {
            parse: function (e, t) {
              var r = (t = t || {}).dynamicTyping || !1;
              J(r) && (t.dynamicTypingFunction = r, r = {});
              if (t.dynamicTyping = r, t.transform = !!J(t.transform) && t.transform, t.worker && b.WORKERS_SUPPORTED) {
                var i = function () {
                  if (!b.WORKERS_SUPPORTED) return !1;
                  var e = (r = f.URL || f.webkitURL || null, i = s.toString(), b.BLOB_URL || (b.BLOB_URL = r.createObjectURL(new Blob(["var global = (function() { if (typeof self !== 'undefined') { return self; } if (typeof window !== 'undefined') { return window; } if (typeof global !== 'undefined') { return global; } return {}; })(); global.IS_PAPA_WORKER=true; ", "(", i, ")();"], {
                      type: "text/javascript"
                    })))),
                    t = new f.Worker(e);
                  var r, i;
                  return t.onmessage = _, t.id = u++, a[t.id] = t;
                }();
                return i.userStep = t.step, i.userChunk = t.chunk, i.userComplete = t.complete, i.userError = t.error, t.step = J(t.step), t.chunk = J(t.chunk), t.complete = J(t.complete), t.error = J(t.error), delete t.worker, void i.postMessage({
                  input: e,
                  config: t,
                  workerId: i.id
                });
              }
              var n = null;
              b.NODE_STREAM_INPUT, "string" == typeof e ? (e = function (e) {
                if (65279 === e.charCodeAt(0)) return e.slice(1);
                return e;
              }(e), n = t.download ? new l(t) : new p(t)) : !0 === e.readable && J(e.read) && J(e.on) ? n = new g(t) : (f.File && e instanceof File || e instanceof Object) && (n = new c(t));
              return n.stream(e);
            },
            unparse: function (e, t) {
              var n = !1,
                _ = !0,
                m = ",",
                y = "\r\n",
                s = '"',
                a = s + s,
                r = !1,
                i = null,
                o = !1;
              !function () {
                if ("object" != typeof t) return;
                "string" != typeof t.delimiter || b.BAD_DELIMITERS.filter(function (e) {
                  return -1 !== t.delimiter.indexOf(e);
                }).length || (m = t.delimiter);
                ("boolean" == typeof t.quotes || "function" == typeof t.quotes || Array.isArray(t.quotes)) && (n = t.quotes);
                "boolean" != typeof t.skipEmptyLines && "string" != typeof t.skipEmptyLines || (r = t.skipEmptyLines);
                "string" == typeof t.newline && (y = t.newline);
                "string" == typeof t.quoteChar && (s = t.quoteChar);
                "boolean" == typeof t.header && (_ = t.header);
                if (Array.isArray(t.columns)) {
                  if (0 === t.columns.length) throw new Error("Option columns is empty");
                  i = t.columns;
                }
                void 0 !== t.escapeChar && (a = t.escapeChar + s);
                ("boolean" == typeof t.escapeFormulae || t.escapeFormulae instanceof RegExp) && (o = t.escapeFormulae instanceof RegExp ? t.escapeFormulae : /^[=+\-@\t\r].*$/);
              }();
              var u = new RegExp(Q(s), "g");
              "string" == typeof e && (e = JSON.parse(e));
              if (Array.isArray(e)) {
                if (!e.length || Array.isArray(e[0])) return h(null, e, r);
                if ("object" == typeof e[0]) return h(i || Object.keys(e[0]), e, r);
              } else if ("object" == typeof e) return "string" == typeof e.data && (e.data = JSON.parse(e.data)), Array.isArray(e.data) && (e.fields || (e.fields = e.meta && e.meta.fields || i), e.fields || (e.fields = Array.isArray(e.data[0]) ? e.fields : "object" == typeof e.data[0] ? Object.keys(e.data[0]) : []), Array.isArray(e.data[0]) || "object" == typeof e.data[0] || (e.data = [e.data])), h(e.fields || [], e.data || [], r);
              throw new Error("Unable to serialize unrecognized input");
              function h(e, t, r) {
                var i = "";
                "string" == typeof e && (e = JSON.parse(e)), "string" == typeof t && (t = JSON.parse(t));
                var n = Array.isArray(e) && 0 < e.length,
                  s = !Array.isArray(t[0]);
                if (n && _) {
                  for (var a = 0; a < e.length; a++) 0 < a && (i += m), i += v(e[a], a);
                  0 < t.length && (i += y);
                }
                for (var o = 0; o < t.length; o++) {
                  var u = n ? e.length : t[o].length,
                    h = !1,
                    f = n ? 0 === Object.keys(t[o]).length : 0 === t[o].length;
                  if (r && !n && (h = "greedy" === r ? "" === t[o].join("").trim() : 1 === t[o].length && 0 === t[o][0].length), "greedy" === r && n) {
                    for (var d = [], l = 0; l < u; l++) {
                      var c = s ? e[l] : l;
                      d.push(t[o][c]);
                    }
                    h = "" === d.join("").trim();
                  }
                  if (!h) {
                    for (var p = 0; p < u; p++) {
                      0 < p && !f && (i += m);
                      var g = n && s ? e[p] : p;
                      i += v(t[o][g], p);
                    }
                    o < t.length - 1 && (!r || 0 < u && !f) && (i += y);
                  }
                }
                return i;
              }
              function v(e, t) {
                if (null == e) return "";
                if (e.constructor === Date) return JSON.stringify(e).slice(1, 25);
                var r = !1;
                o && "string" == typeof e && o.test(e) && (e = "'" + e, r = !0);
                var i = e.toString().replace(u, a);
                return (r = r || !0 === n || "function" == typeof n && n(e, t) || Array.isArray(n) && n[t] || function (e, t) {
                  for (var r = 0; r < t.length; r++) if (-1 < e.indexOf(t[r])) return !0;
                  return !1;
                }(i, b.BAD_DELIMITERS) || -1 < i.indexOf(m) || " " === i.charAt(0) || " " === i.charAt(i.length - 1)) ? s + i + s : i;
              }
            }
          };
        if (b.RECORD_SEP = String.fromCharCode(30), b.UNIT_SEP = String.fromCharCode(31), b.BYTE_ORDER_MARK = "\ufeff", b.BAD_DELIMITERS = ["\r", "\n", '"', b.BYTE_ORDER_MARK], b.WORKERS_SUPPORTED = !n && !!f.Worker, b.NODE_STREAM_INPUT = 1, b.LocalChunkSize = 10485760, b.RemoteChunkSize = 5242880, b.DefaultDelimiter = ",", b.Parser = E, b.ParserHandle = r, b.NetworkStreamer = l, b.FileStreamer = c, b.StringStreamer = p, b.ReadableStreamStreamer = g, f.jQuery) {
          var d = f.jQuery;
          d.fn.parse = function (o) {
            var r = o.config || {},
              u = [];
            return this.each(function (e) {
              if (!("INPUT" === d(this).prop("tagName").toUpperCase() && "file" === d(this).attr("type").toLowerCase() && f.FileReader) || !this.files || 0 === this.files.length) return !0;
              for (var t = 0; t < this.files.length; t++) u.push({
                file: this.files[t],
                inputElem: this,
                instanceConfig: d.extend({}, r)
              });
            }), e(), this;
            function e() {
              if (0 !== u.length) {
                var e,
                  t,
                  r,
                  i,
                  n = u[0];
                if (J(o.before)) {
                  var s = o.before(n.file, n.inputElem);
                  if ("object" == typeof s) {
                    if ("abort" === s.action) return e = "AbortError", t = n.file, r = n.inputElem, i = s.reason, void (J(o.error) && o.error({
                      name: e
                    }, t, r, i));
                    if ("skip" === s.action) return void h();
                    "object" == typeof s.config && (n.instanceConfig = d.extend(n.instanceConfig, s.config));
                  } else if ("skip" === s) return void h();
                }
                var a = n.instanceConfig.complete;
                n.instanceConfig.complete = function (e) {
                  J(a) && a(e, n.file, n.inputElem), h();
                }, b.parse(n.file, n.instanceConfig);
              } else J(o.complete) && o.complete();
            }
            function h() {
              u.splice(0, 1), e();
            }
          };
        }
        function h(e) {
          this._handle = null, this._finished = !1, this._completed = !1, this._halted = !1, this._input = null, this._baseIndex = 0, this._partialLine = "", this._rowCount = 0, this._start = 0, this._nextChunk = null, this.isFirstChunk = !0, this._completeResults = {
            data: [],
            errors: [],
            meta: {}
          }, function (e) {
            var t = w(e);
            t.chunkSize = parseInt(t.chunkSize), e.step || e.chunk || (t.chunkSize = null);
            this._handle = new r(t), (this._handle.streamer = this)._config = t;
          }.call(this, e), this.parseChunk = function (e, t) {
            if (this.isFirstChunk && J(this._config.beforeFirstChunk)) {
              var r = this._config.beforeFirstChunk(e);
              void 0 !== r && (e = r);
            }
            this.isFirstChunk = !1, this._halted = !1;
            var i = this._partialLine + e;
            this._partialLine = "";
            var n = this._handle.parse(i, this._baseIndex, !this._finished);
            if (!this._handle.paused() && !this._handle.aborted()) {
              var s = n.meta.cursor;
              this._finished || (this._partialLine = i.substring(s - this._baseIndex), this._baseIndex = s), n && n.data && (this._rowCount += n.data.length);
              var a = this._finished || this._config.preview && this._rowCount >= this._config.preview;
              if (o) f.postMessage({
                results: n,
                workerId: b.WORKER_ID,
                finished: a
              });else if (J(this._config.chunk) && !t) {
                if (this._config.chunk(n, this._handle), this._handle.paused() || this._handle.aborted()) return void (this._halted = !0);
                n = void 0, this._completeResults = void 0;
              }
              return this._config.step || this._config.chunk || (this._completeResults.data = this._completeResults.data.concat(n.data), this._completeResults.errors = this._completeResults.errors.concat(n.errors), this._completeResults.meta = n.meta), this._completed || !a || !J(this._config.complete) || n && n.meta.aborted || (this._config.complete(this._completeResults, this._input), this._completed = !0), a || n && n.meta.paused || this._nextChunk(), n;
            }
            this._halted = !0;
          }, this._sendError = function (e) {
            J(this._config.error) ? this._config.error(e) : o && this._config.error && f.postMessage({
              workerId: b.WORKER_ID,
              error: e,
              finished: !1
            });
          };
        }
        function l(e) {
          var i;
          (e = e || {}).chunkSize || (e.chunkSize = b.RemoteChunkSize), h.call(this, e), this._nextChunk = n ? function () {
            this._readChunk(), this._chunkLoaded();
          } : function () {
            this._readChunk();
          }, this.stream = function (e) {
            this._input = e, this._nextChunk();
          }, this._readChunk = function () {
            if (this._finished) this._chunkLoaded();else {
              if (i = new XMLHttpRequest(), this._config.withCredentials && (i.withCredentials = this._config.withCredentials), n || (i.onload = v(this._chunkLoaded, this), i.onerror = v(this._chunkError, this)), i.open(this._config.downloadRequestBody ? "POST" : "GET", this._input, !n), this._config.downloadRequestHeaders) {
                var e = this._config.downloadRequestHeaders;
                for (var t in e) i.setRequestHeader(t, e[t]);
              }
              if (this._config.chunkSize) {
                var r = this._start + this._config.chunkSize - 1;
                i.setRequestHeader("Range", "bytes=" + this._start + "-" + r);
              }
              try {
                i.send(this._config.downloadRequestBody);
              } catch (e) {
                this._chunkError(e.message);
              }
              n && 0 === i.status && this._chunkError();
            }
          }, this._chunkLoaded = function () {
            4 === i.readyState && (i.status < 200 || 400 <= i.status ? this._chunkError() : (this._start += this._config.chunkSize ? this._config.chunkSize : i.responseText.length, this._finished = !this._config.chunkSize || this._start >= function (e) {
              var t = e.getResponseHeader("Content-Range");
              if (null === t) return -1;
              return parseInt(t.substring(t.lastIndexOf("/") + 1));
            }(i), this.parseChunk(i.responseText)));
          }, this._chunkError = function (e) {
            var t = i.statusText || e;
            this._sendError(new Error(t));
          };
        }
        function c(e) {
          var i, n;
          (e = e || {}).chunkSize || (e.chunkSize = b.LocalChunkSize), h.call(this, e);
          var s = "undefined" != typeof FileReader;
          this.stream = function (e) {
            this._input = e, n = e.slice || e.webkitSlice || e.mozSlice, s ? ((i = new FileReader()).onload = v(this._chunkLoaded, this), i.onerror = v(this._chunkError, this)) : i = new FileReaderSync(), this._nextChunk();
          }, this._nextChunk = function () {
            this._finished || this._config.preview && !(this._rowCount < this._config.preview) || this._readChunk();
          }, this._readChunk = function () {
            var e = this._input;
            if (this._config.chunkSize) {
              var t = Math.min(this._start + this._config.chunkSize, this._input.size);
              e = n.call(e, this._start, t);
            }
            var r = i.readAsText(e, this._config.encoding);
            s || this._chunkLoaded({
              target: {
                result: r
              }
            });
          }, this._chunkLoaded = function (e) {
            this._start += this._config.chunkSize, this._finished = !this._config.chunkSize || this._start >= this._input.size, this.parseChunk(e.target.result);
          }, this._chunkError = function () {
            this._sendError(i.error);
          };
        }
        function p(e) {
          var r;
          h.call(this, e = e || {}), this.stream = function (e) {
            return r = e, this._nextChunk();
          }, this._nextChunk = function () {
            if (!this._finished) {
              var e,
                t = this._config.chunkSize;
              return t ? (e = r.substring(0, t), r = r.substring(t)) : (e = r, r = ""), this._finished = !r, this.parseChunk(e);
            }
          };
        }
        function g(e) {
          h.call(this, e = e || {});
          var t = [],
            r = !0,
            i = !1;
          this.pause = function () {
            h.prototype.pause.apply(this, arguments), this._input.pause();
          }, this.resume = function () {
            h.prototype.resume.apply(this, arguments), this._input.resume();
          }, this.stream = function (e) {
            this._input = e, this._input.on("data", this._streamData), this._input.on("end", this._streamEnd), this._input.on("error", this._streamError);
          }, this._checkIsFinished = function () {
            i && 1 === t.length && (this._finished = !0);
          }, this._nextChunk = function () {
            this._checkIsFinished(), t.length ? this.parseChunk(t.shift()) : r = !0;
          }, this._streamData = v(function (e) {
            try {
              t.push("string" == typeof e ? e : e.toString(this._config.encoding)), r && (r = !1, this._checkIsFinished(), this.parseChunk(t.shift()));
            } catch (e) {
              this._streamError(e);
            }
          }, this), this._streamError = v(function (e) {
            this._streamCleanUp(), this._sendError(e);
          }, this), this._streamEnd = v(function () {
            this._streamCleanUp(), i = !0, this._streamData("");
          }, this), this._streamCleanUp = v(function () {
            this._input.removeListener("data", this._streamData), this._input.removeListener("end", this._streamEnd), this._input.removeListener("error", this._streamError);
          }, this);
        }
        function r(m) {
          var a,
            o,
            u,
            i = Math.pow(2, 53),
            n = -i,
            s = /^\s*-?(\d+\.?|\.\d+|\d+\.\d+)([eE][-+]?\d+)?\s*$/,
            h = /^((\d{4}-[01]\d-[0-3]\dT[0-2]\d:[0-5]\d:[0-5]\d\.\d+([+-][0-2]\d:[0-5]\d|Z))|(\d{4}-[01]\d-[0-3]\dT[0-2]\d:[0-5]\d:[0-5]\d([+-][0-2]\d:[0-5]\d|Z))|(\d{4}-[01]\d-[0-3]\dT[0-2]\d:[0-5]\d([+-][0-2]\d:[0-5]\d|Z)))$/,
            t = this,
            r = 0,
            f = 0,
            d = !1,
            e = !1,
            l = [],
            c = {
              data: [],
              errors: [],
              meta: {}
            };
          if (J(m.step)) {
            var p = m.step;
            m.step = function (e) {
              if (c = e, _()) g();else {
                if (g(), 0 === c.data.length) return;
                r += e.data.length, m.preview && r > m.preview ? o.abort() : (c.data = c.data[0], p(c, t));
              }
            };
          }
          function y(e) {
            return "greedy" === m.skipEmptyLines ? "" === e.join("").trim() : 1 === e.length && 0 === e[0].length;
          }
          function g() {
            return c && u && (k("Delimiter", "UndetectableDelimiter", "Unable to auto-detect delimiting character; defaulted to '" + b.DefaultDelimiter + "'"), u = !1), m.skipEmptyLines && (c.data = c.data.filter(function (e) {
              return !y(e);
            })), _() && function () {
              if (!c) return;
              function e(e, t) {
                J(m.transformHeader) && (e = m.transformHeader(e, t)), l.push(e);
              }
              if (Array.isArray(c.data[0])) {
                for (var t = 0; _() && t < c.data.length; t++) c.data[t].forEach(e);
                c.data.splice(0, 1);
              } else c.data.forEach(e);
            }(), function () {
              if (!c || !m.header && !m.dynamicTyping && !m.transform) return c;
              function e(e, t) {
                var r,
                  i = m.header ? {} : [];
                for (r = 0; r < e.length; r++) {
                  var n = r,
                    s = e[r];
                  m.header && (n = r >= l.length ? "__parsed_extra" : l[r]), m.transform && (s = m.transform(s, n)), s = v(n, s), "__parsed_extra" === n ? (i[n] = i[n] || [], i[n].push(s)) : i[n] = s;
                }
                return m.header && (r > l.length ? k("FieldMismatch", "TooManyFields", "Too many fields: expected " + l.length + " fields but parsed " + r, f + t) : r < l.length && k("FieldMismatch", "TooFewFields", "Too few fields: expected " + l.length + " fields but parsed " + r, f + t)), i;
              }
              var t = 1;
              !c.data.length || Array.isArray(c.data[0]) ? (c.data = c.data.map(e), t = c.data.length) : c.data = e(c.data, 0);
              m.header && c.meta && (c.meta.fields = l);
              return f += t, c;
            }();
          }
          function _() {
            return m.header && 0 === l.length;
          }
          function v(e, t) {
            return r = e, m.dynamicTypingFunction && void 0 === m.dynamicTyping[r] && (m.dynamicTyping[r] = m.dynamicTypingFunction(r)), !0 === (m.dynamicTyping[r] || m.dynamicTyping) ? "true" === t || "TRUE" === t || "false" !== t && "FALSE" !== t && (function (e) {
              if (s.test(e)) {
                var t = parseFloat(e);
                if (n < t && t < i) return !0;
              }
              return !1;
            }(t) ? parseFloat(t) : h.test(t) ? new Date(t) : "" === t ? null : t) : t;
            var r;
          }
          function k(e, t, r, i) {
            var n = {
              type: e,
              code: t,
              message: r
            };
            void 0 !== i && (n.row = i), c.errors.push(n);
          }
          this.parse = function (e, t, r) {
            var i = m.quoteChar || '"';
            if (m.newline || (m.newline = function (e, t) {
              e = e.substring(0, 1048576);
              var r = new RegExp(Q(t) + "([^]*?)" + Q(t), "gm"),
                i = (e = e.replace(r, "")).split("\r"),
                n = e.split("\n"),
                s = 1 < n.length && n[0].length < i[0].length;
              if (1 === i.length || s) return "\n";
              for (var a = 0, o = 0; o < i.length; o++) "\n" === i[o][0] && a++;
              return a >= i.length / 2 ? "\r\n" : "\r";
            }(e, i)), u = !1, m.delimiter) J(m.delimiter) && (m.delimiter = m.delimiter(e), c.meta.delimiter = m.delimiter);else {
              var n = function (e, t, r, i, n) {
                var s, a, o, u;
                n = n || [",", "\t", "|", ";", b.RECORD_SEP, b.UNIT_SEP];
                for (var h = 0; h < n.length; h++) {
                  var f = n[h],
                    d = 0,
                    l = 0,
                    c = 0;
                  o = void 0;
                  for (var p = new E({
                      comments: i,
                      delimiter: f,
                      newline: t,
                      preview: 10
                    }).parse(e), g = 0; g < p.data.length; g++) if (r && y(p.data[g])) c++;else {
                    var _ = p.data[g].length;
                    l += _, void 0 !== o ? 0 < _ && (d += Math.abs(_ - o), o = _) : o = _;
                  }
                  0 < p.data.length && (l /= p.data.length - c), (void 0 === a || d <= a) && (void 0 === u || u < l) && 1.99 < l && (a = d, s = f, u = l);
                }
                return {
                  successful: !!(m.delimiter = s),
                  bestDelimiter: s
                };
              }(e, m.newline, m.skipEmptyLines, m.comments, m.delimitersToGuess);
              n.successful ? m.delimiter = n.bestDelimiter : (u = !0, m.delimiter = b.DefaultDelimiter), c.meta.delimiter = m.delimiter;
            }
            var s = w(m);
            return m.preview && m.header && s.preview++, a = e, o = new E(s), c = o.parse(a, t, r), g(), d ? {
              meta: {
                paused: !0
              }
            } : c || {
              meta: {
                paused: !1
              }
            };
          }, this.paused = function () {
            return d;
          }, this.pause = function () {
            d = !0, o.abort(), a = J(m.chunk) ? "" : a.substring(o.getCharIndex());
          }, this.resume = function () {
            t.streamer._halted ? (d = !1, t.streamer.parseChunk(a, !0)) : setTimeout(t.resume, 3);
          }, this.aborted = function () {
            return e;
          }, this.abort = function () {
            e = !0, o.abort(), c.meta.aborted = !0, J(m.complete) && m.complete(c), a = "";
          };
        }
        function Q(e) {
          return e.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
        }
        function E(j) {
          var z,
            M = (j = j || {}).delimiter,
            P = j.newline,
            U = j.comments,
            q = j.step,
            N = j.preview,
            B = j.fastMode,
            K = z = void 0 === j.quoteChar || null === j.quoteChar ? '"' : j.quoteChar;
          if (void 0 !== j.escapeChar && (K = j.escapeChar), ("string" != typeof M || -1 < b.BAD_DELIMITERS.indexOf(M)) && (M = ","), U === M) throw new Error("Comment character same as delimiter");
          !0 === U ? U = "#" : ("string" != typeof U || -1 < b.BAD_DELIMITERS.indexOf(U)) && (U = !1), "\n" !== P && "\r" !== P && "\r\n" !== P && (P = "\n");
          var W = 0,
            H = !1;
          this.parse = function (i, t, r) {
            if ("string" != typeof i) throw new Error("Input must be a string");
            var n = i.length,
              e = M.length,
              s = P.length,
              a = U.length,
              o = J(q),
              u = [],
              h = [],
              f = [],
              d = W = 0;
            if (!i) return L();
            if (j.header && !t) {
              var l = i.split(P)[0].split(M),
                c = [],
                p = {},
                g = !1;
              for (var _ in l) {
                var m = l[_];
                J(j.transformHeader) && (m = j.transformHeader(m, _));
                var y = m,
                  v = p[m] || 0;
                for (0 < v && (g = !0, y = m + "_" + v), p[m] = v + 1; c.includes(y);) y = y + "_" + v;
                c.push(y);
              }
              if (g) {
                var k = i.split(P);
                k[0] = c.join(M), i = k.join(P);
              }
            }
            if (B || !1 !== B && -1 === i.indexOf(z)) {
              for (var b = i.split(P), E = 0; E < b.length; E++) {
                if (f = b[E], W += f.length, E !== b.length - 1) W += P.length;else if (r) return L();
                if (!U || f.substring(0, a) !== U) {
                  if (o) {
                    if (u = [], I(f.split(M)), F(), H) return L();
                  } else I(f.split(M));
                  if (N && N <= E) return u = u.slice(0, N), L(!0);
                }
              }
              return L();
            }
            for (var w = i.indexOf(M, W), R = i.indexOf(P, W), C = new RegExp(Q(K) + Q(z), "g"), S = i.indexOf(z, W);;) if (i[W] !== z) {
              if (U && 0 === f.length && i.substring(W, W + a) === U) {
                if (-1 === R) return L();
                W = R + s, R = i.indexOf(P, W), w = i.indexOf(M, W);
              } else if (-1 !== w && (w < R || -1 === R)) f.push(i.substring(W, w)), W = w + e, w = i.indexOf(M, W);else {
                if (-1 === R) break;
                if (f.push(i.substring(W, R)), D(R + s), o && (F(), H)) return L();
                if (N && u.length >= N) return L(!0);
              }
            } else for (S = W, W++;;) {
              if (-1 === (S = i.indexOf(z, S + 1))) return r || h.push({
                type: "Quotes",
                code: "MissingQuotes",
                message: "Quoted field unterminated",
                row: u.length,
                index: W
              }), T();
              if (S === n - 1) return T(i.substring(W, S).replace(C, z));
              if (z !== K || i[S + 1] !== K) {
                if (z === K || 0 === S || i[S - 1] !== K) {
                  -1 !== w && w < S + 1 && (w = i.indexOf(M, S + 1)), -1 !== R && R < S + 1 && (R = i.indexOf(P, S + 1));
                  var O = A(-1 === R ? w : Math.min(w, R));
                  if (i.substr(S + 1 + O, e) === M) {
                    f.push(i.substring(W, S).replace(C, z)), i[W = S + 1 + O + e] !== z && (S = i.indexOf(z, W)), w = i.indexOf(M, W), R = i.indexOf(P, W);
                    break;
                  }
                  var x = A(R);
                  if (i.substring(S + 1 + x, S + 1 + x + s) === P) {
                    if (f.push(i.substring(W, S).replace(C, z)), D(S + 1 + x + s), w = i.indexOf(M, W), S = i.indexOf(z, W), o && (F(), H)) return L();
                    if (N && u.length >= N) return L(!0);
                    break;
                  }
                  h.push({
                    type: "Quotes",
                    code: "InvalidQuotes",
                    message: "Trailing quote on quoted field is malformed",
                    row: u.length,
                    index: W
                  }), S++;
                }
              } else S++;
            }
            return T();
            function I(e) {
              u.push(e), d = W;
            }
            function A(e) {
              var t = 0;
              if (-1 !== e) {
                var r = i.substring(S + 1, e);
                r && "" === r.trim() && (t = r.length);
              }
              return t;
            }
            function T(e) {
              return r || (void 0 === e && (e = i.substring(W)), f.push(e), W = n, I(f), o && F()), L();
            }
            function D(e) {
              W = e, I(f), f = [], R = i.indexOf(P, W);
            }
            function L(e) {
              return {
                data: u,
                errors: h,
                meta: {
                  delimiter: M,
                  linebreak: P,
                  aborted: H,
                  truncated: !!e,
                  cursor: d + (t || 0)
                }
              };
            }
            function F() {
              q(L()), u = [], h = [];
            }
          }, this.abort = function () {
            H = !0;
          }, this.getCharIndex = function () {
            return W;
          };
        }
        function _(e) {
          var t = e.data,
            r = a[t.workerId],
            i = !1;
          if (t.error) r.userError(t.error, t.file);else if (t.results && t.results.data) {
            var n = {
              abort: function () {
                i = !0, m(t.workerId, {
                  data: [],
                  errors: [],
                  meta: {
                    aborted: !0
                  }
                });
              },
              pause: y,
              resume: y
            };
            if (J(r.userStep)) {
              for (var s = 0; s < t.results.data.length && (r.userStep({
                data: t.results.data[s],
                errors: t.results.errors,
                meta: t.results.meta
              }, n), !i); s++);
              delete t.results;
            } else J(r.userChunk) && (r.userChunk(t.results, n, t.file), delete t.results);
          }
          t.finished && !i && m(t.workerId, t.results);
        }
        function m(e, t) {
          var r = a[e];
          J(r.userComplete) && r.userComplete(t), r.terminate(), delete a[e];
        }
        function y() {
          throw new Error("Not implemented.");
        }
        function w(e) {
          if ("object" != typeof e || null === e) return e;
          var t = Array.isArray(e) ? [] : {};
          for (var r in e) t[r] = w(e[r]);
          return t;
        }
        function v(e, t) {
          return function () {
            e.apply(t, arguments);
          };
        }
        function J(e) {
          return "function" == typeof e;
        }
        return o && (f.onmessage = function (e) {
          var t = e.data;
          void 0 === b.WORKER_ID && t && (b.WORKER_ID = t.workerId);
          if ("string" == typeof t.input) f.postMessage({
            workerId: b.WORKER_ID,
            results: b.parse(t.input, t.config),
            finished: !0
          });else if (f.File && t.input instanceof File || t.input instanceof Object) {
            var r = b.parse(t.input, t.config);
            r && f.postMessage({
              workerId: b.WORKER_ID,
              results: r,
              finished: !0
            });
          }
        }), (l.prototype = Object.create(h.prototype)).constructor = l, (c.prototype = Object.create(h.prototype)).constructor = c, (p.prototype = Object.create(p.prototype)).constructor = p, (g.prototype = Object.create(h.prototype)).constructor = g, b;
      });
    })(papaparse_min);
    var Papa = papaparse_min.exports;

    /**
     * Converts a string to camel case.
     * @param {string} name
     * @returns {string}
     */
    function getCamelCase(name) {
      return name.replace(/[ -][a-z]/g, string => string[1].toUpperCase());
    }

    function getParts(text) {
      const lines = text.split(/\r?\n/);
      const parts = {
        data: []
      };
      let currentPart = parts.data;
      let currentLabel = '';
      for (const line of lines) {
        if (line.startsWith('</')) {
          // close existing part
          if (!currentLabel === line.slice(2, -1)) {
            throw new Error('This should not happen');
          }
          currentLabel = '';
          currentPart = parts.data;
        } else if (line.startsWith('<') && !line.includes('=')) {
          // open new part
          if (currentLabel) {
            throw new Error('This should not happen');
          }
          currentLabel = line.slice(1, -1);
          const target = getCamelCase(currentLabel);
          parts[target] = [];
          currentPart = parts[target];
        } else if (currentLabel) {
          // add line to current part
          currentPart.push(line);
        } else {
          //data lines
          currentPart.push(line);
        }
      }
      return parts;
    }

    function parseColumnbProperties(lines) {
      lines = lines.map(line => {
        const [key, value] = line.slice(1, -1).split('=');
        return {
          key,
          value: value.slice(1, -1)
        };
      });
      const columnProperties = {};
      let currentColumnName = '';
      for (const line of lines) {
        switch (line.key) {
          case 'columnName':
            currentColumnName = line.value;
            columnProperties[currentColumnName] = {};
            break;
          case 'columnProperty':
            {
              if (!currentColumnName) {
                throw new Error('This should not happen');
              }
              const [key, value] = line.value.split('\t');
              columnProperties[currentColumnName][key] = value;
            }
            break;
          default:
            throw new Error('This should not happen');
        }
      }
      for (const key in columnProperties) {
        const columnPropery = columnProperties[key];
        if (columnProperties[key].parent) {
          const target = columnProperties[columnPropery.parent];
          if (!target) {
            throw new Error('Parent column not found');
          }
          if (!target.related) {
            target.related = {};
          }
          target.related[columnPropery.specialType] = key;
        }
      }
      return columnProperties;
    }

    function parseData(lines, options = {}) {
      lines = lines.filter(line => !line.match(/^\s*$/));
      const {
        columnProperties = {}
      } = options;
      const headers = lines.shift().split('\t').map(header => {
        if (columnProperties[header]) {
          return {
            label: header,
            ...columnProperties[header]
          };
        }
        return {
          label: header
        };
      });
      const entries = [];
      const rawEntries = [];
      for (const line of lines) {
        const fields = line.split('\t');
        const rawEntry = {};
        headers.forEach((header, index) => {
          rawEntry[header.label] = fields[index];
        });
        rawEntries.push(rawEntry);
        const entry = {};
        headers.forEach(header => {
          if (header.parent) return;
          entry[header.label] = valueEhnhancer(header, rawEntry);
        });
        entries.push(entry);
      }
      return {
        entries,
        rawEntries
      };
    }
    function valueEhnhancer(header, rawEntry) {
      if (header?.specialType === 'rxncode') {
        return `${rawEntry[header.label]}#${rawEntry[header.related.atomMapping]}#${rawEntry[header.related.idcoordinates2D]}`;
      }
      return rawEntry[header.label];
    }
    /*
    entry.rxnCode =
    */

    /**
     * Convert a DataWarrior database into a JSON object
     * @param {string} text
     * @returns
     */
    function parseDwar(text) {
      text = ensureString(text);
      const parts = getParts(text);
      improveParts(parts);
      return parts;
    }
    function improveParts(parts) {
      for (const key in parts) {
        switch (key) {
          case 'columnProperties':
            parts[key] = parseColumnbProperties(parts[key]);
            break;
          case 'data':
            break;
          default:
            parts[key] = parseDefault(parts[key]);
        }
      }
      const data = parseData(parts.data, {
        columnProperties: parts.columnProperties
      });
      parts.data = data.entries;
      parts.rawData = data.rawEntries;
    }
    function parseDefault(lines) {
      const result = {};
      for (const line of lines) {
        const [key, value] = line.slice(1, -1).split('=');
        result[key] = value.slice(1, -1);
      }
      return result;
    }

    /**
     * @description Append the OCL reaction to the reaction object
     * @param {Array} reactions array of reactions objects with rxnCode and label
     * @param {Object} OCL OCL object
     * @param {Object} [options={}]
     * @param {import('cheminfo-types').Logger} [options.logger]
     *
     * @returns {Array} array of reactions objects with rxnCode, label and oclReaction (a decoded version of rxnCode reaction)
     */
    function appendOCLReaction(reactions, OCL, options = {}) {
      const {
        logger
      } = options;
      const newReactions = [];
      for (const reaction of reactions) {
        if (reaction.rxnCode) {
          newReactions.push({
            ...reaction,
            oclReaction: OCL.ReactionEncoder.decode(reaction.rxnCode)
          });
        } else if (logger) {
          logger.warn(reaction, 'Reaction without rxnCode');
        }
      }
      return newReactions;
    }

    /**
     *
     * @param {*} processedMolecules
     * @param {import('openchemlib').Molecule|string} molecule
     * @param {*} options
     * @returns
     */
    function checkIfExistsOrAddInfo(processedMolecules, molecule, options) {
      const {
        moleculeInfoCallback,
        asReagent,
        asProduct
      } = options;
      const idCode = typeof molecule === 'string' ? idCode : molecule.getIDCode();
      if (processedMolecules.has(idCode)) {
        const entry = processedMolecules.get(idCode);
        let exists = false;
        if (asReagent) {
          if (entry.asReagent) {
            exists = true;
          } else {
            entry.asReagent = true;
          }
        }
        if (asProduct) {
          if (entry.asProduct) {
            exists = true;
          } else {
            entry.asProduct = true;
          }
        }
        return {
          exists,
          info: entry
        };
      } else {
        let info = {
          idCode,
          molfile: molecule.toMolfile(),
          asReagent,
          asProduct,
          info: {}
        };
        if (moleculeInfoCallback) {
          info.info = moleculeInfoCallback(molecule);
        }
        processedMolecules.set(idCode, info);
        return {
          exists: false,
          info
        };
      }
    }

    /**
     * @description apply one reaction to one reactant
     * @param {import('openchemlib').Molecule[]} reactants
     * @param {Array<Object>} reactions rxnCode of the reaction
     * @param {Object} options options to apply the reaction
     * @param {number} options.currentDepth current depth of the recursion
     * @param {number} options.maxDepth max depth of the recursion
     * @param {number} options.maxCurrentDepth max depth of the recursion for this set of reactions
     * @param {number} options.limitReactions limit the number of reactions
     * @param {Object} options.stats stats of the recursion
     * @param {number} options.stats.counter number of reactions
     * @param {Map} options.processedMolecules set of processed molecules
     * @param {Array} options.trees array of trees of previous recursions
     * @param {import('openchemlib')} options.OCL OCL object
     * @returns {Array} array of results
     */
    function applyOneReactantReactions(tree, reactions, options) {
      const {
        currentDepth,
        maxDepth,
        maxCurrentDepth,
        processedMolecules,
        OCL,
        logger
      } = options;
      if (tree.molecules.length !== 1) {
        logger?.warn('applyOneReactantReactions:tree.reactants.length!==1', tree.reactants.length);
        return [];
      }
      const reactant = OCL.Molecule.fromIDCode(tree.molecules[0].idCode);
      const todoNextDepth = [];
      // if the current depth is greater than the max depth, we stop the recursion and return an empty array
      if (currentDepth > maxCurrentDepth || tree.depth >= maxDepth) {
        return [];
      }
      const existsAndInfo = checkIfExistsOrAddInfo(processedMolecules, reactant, {
        ...options,
        asReagent: true
      });
      // check if the reactant has already been processed
      if (existsAndInfo.exists) {
        return [];
      }
      for (const reaction of reactions) {
        if (options.stats.counter >= options.limitReactions) {
          return [];
        }
        const reactor = new OCL.Reactor(reaction.oclReaction);
        // isMatching is true if the reactant is matching the reaction else we continue to the next reaction
        const isMatching = Boolean(reactor.setReactant(0, reactant));
        if (isMatching) {
          options.stats.counter++;
          // get the products of the reaction
          const oneReactionProducts = reactor.getProducts();
          for (const oneReactionProduct of oneReactionProducts) {
            for (const reactionProduct of oneReactionProduct) {
              // get the info of the product (molfile, idCode, mf)
              const productExistsAndInfo = checkIfExistsOrAddInfo(processedMolecules, reactionProduct, {
                ...options,
                asProduct: true
              });
              // if the product has not been processed yet, we add it to the list of products and we add it to the list of todoNextDepth
              if (!productExistsAndInfo.exists) {
                // eslint-disable-next-line no-unused-vars
                const {
                  oclReaction,
                  needToBeCharged,
                  ...reactionWithoutOCL
                } = reaction;
                const oneReaction = {
                  reaction: reactionWithoutOCL,
                  depth: tree.depth + 1,
                  isValid: true,
                  currentDepth,
                  molecules: [checkIfExistsOrAddInfo(processedMolecules, reactionProduct, options).info]
                };
                if (!tree.children) tree.children = [];
                tree.children.push(oneReaction);
                todoNextDepth.push(() => {
                  return applyOneReactantReactions(oneReaction, reactions, {
                    ...options,
                    currentDepth: options.currentDepth + 1
                  });
                });
              }
            }
          }
        }
      }
      // by returning todoNextDepth, we make sure that the recursion will continue
      return todoNextDepth;
    }

    function getFilteredTrees(reactions, options = {}) {
      const {
        filter = () => true
      } = options;
      const nodesToKeep = reactions.getNodes().filter(filter);
      const parentMap = reactions.getParentMap();
      for (let currentNode of nodesToKeep) {
        const parent = parentMap.get(currentNode);
        if (parent && nodesToKeep.includes(parent) === false) {
          nodesToKeep.push(parent);
        }
      }
      return getValidChildren(reactions.trees, {
        nodesToKeep
      });
    }
    function getValidChildren(nodes, options) {
      const {
        nodesToKeep
      } = options;
      const validNodes = nodes.filter(node => nodesToKeep.includes(node)).map(node => ({
        ...node
      }));
      for (const node of validNodes) {
        if (node.children) {
          const validChildren = node.children.filter(child => nodesToKeep.includes(child));
          if (validChildren.length > 0) {
            node.children = getValidChildren(validChildren, {
              nodesToKeep
            });
          } else {
            delete node.children;
          }
        }
      }
      return validNodes;
    }

    function getLeaves(trees) {
      const leaves = [];
      for (const tree of trees) {
        appendLeavesSS(leaves, tree);
      }
      return leaves;
    }
    function appendLeavesSS(leaves, currentBranch) {
      if (!currentBranch.children || currentBranch.children.length === 0) {
        leaves.push(currentBranch);
        return;
      }
      for (const child of currentBranch.children) {
        appendLeavesSS(leaves, child);
      }
    }

    function getNodes(trees) {
      const nodes = [];
      for (const tree of trees) {
        getNodesSS(nodes, tree);
      }
      return nodes;
    }
    function getNodesSS(nodes, currentBranch) {
      nodes.push(currentBranch);
      for (const child of currentBranch?.children || []) {
        getNodesSS(nodes, child);
      }
    }

    class Reactions {
      /**
       *
       * @param {object} [options={}]
       * @param {import('cheminfo-types').Logger} logger
       * @param {number} [options.maxDepth=5]
       * @param {function} [options.moleculeInfoCallback]
       * @param {boolean} [options.skipProcessed=true]
       */
      constructor(OCL, options = {}) {
        this.moleculeInfoCallback = options.moleculeInfoCallback;
        this.maxDepth = options.maxDepth ?? 5;
        this.limitReactions = options.limitReactions ?? 200;
        this.skipProcessed = options.skipProcessed ?? true;
        this.logger = options.logger;
        this.processedMolecules = new Map();
        this.OCL = OCL;
        this.trees = [];
        this.moleculeInfo = {}; // a cache containing molecule information like mw, etc.
      }
      /**
       * We need to call this method for all the reactants on which we want to apply the reactions.
       * If there is only one reactant, we call this method with an array of one reactant.
       * If there are multiple reactants, we call this method with an array of the reactants.
       * This method has to be called for all the reactants
       * @param {import('openchemlib').Molecule[]|string[]} molecules
       */
      appendHead(moleculesOrIDCodes) {
        if (!Array.isArray(moleculesOrIDCodes)) {
          throw new TypeError('reactants must be an array');
        }
        const molecules = moleculesOrIDCodes.map(molecule => checkIfExistsOrAddInfo(this.processedMolecules, molecule, {
          moleculeInfoCallback: this.moleculeInfoCallback
        }).info);
        const tree = {
          molecules,
          depth: 0,
          isValid: true // this node could be implied in reactions
        };

        this.trees.push(tree);
      }
      /**
       * Returns all the leaves of the trees
       * @returns
       */
      getLeaves() {
        return getLeaves(this.trees);
      }
      /**
       * Returns all the nodes of the trees
       * @returns
       */
      getNodes() {
        return getNodes(this.trees);
      }
      getParentMap() {
        const parentMap = new Map();
        const nodes = this.getNodes();
        for (const node of nodes) {
          if (node.children) {
            for (const child of node.children) {
              parentMap.set(child, node);
            }
          }
        }
        return parentMap;
      }
      /**
       * When applying reactions some branches may be dead because it can not be implied in any reaction.
       * This is the case when we specify a 'min' reaction depth.
       * This will returno only the valid nodes
       * @returns
       */
      getValidNodes() {
        return this.getNodes().filter(node => node.isValid);
      }
      /**
       *
       * @param {object} [options={}]
       * @param {(object):boolean} [options.filter] - a function that will be called for each node and return true if the node should be kept
       */
      getFilteredReactions(options = {}) {
        const filteredReactions = new Reactions();
        filteredReactions.moleculeInfoCallback = this.moleculeInfoCallback;
        filteredReactions.maxDepth = this.maxDepth;
        filteredReactions.limitReactions = this.limitReactions;
        filteredReactions.skipProcessed = this.skipProcessed;
        filteredReactions.logger = this.logger;
        filteredReactions.processedMolecules = this.processedMolecules;
        filteredReactions.OCL = this.OCL;
        filteredReactions.moleculeInfo = this.moleculeInfo; // a cache containing molecule information like mw, etc.
        filteredReactions.trees = getFilteredTrees(this, options);
        return filteredReactions;
      }
      /**
       *
       * @param {object[]} reactions - array of reactions that should be applied
       * @param {object} [options={}]
       * @param {number} [options.min=0] min depth of the reaction
       * @param {number} [options.max=3] max depth of the reaction
       */
      applyOneReactantReactions(reactions, options = {}) {
        const {
          min = 0,
          max = 3
        } = options;
        clearAsFromProcessedMolecules(this.processedMolecules);
        const nodes = this.getNodes().filter(node => node.isValid);
        nodes.forEach(node => {
          node.currentDepth = 0;
        });
        reactions = appendOCLReaction(reactions, this.OCL);
        const stats = {
          counter: 0
        };
        // Start the recursion by applying the first level of reactions
        for (const node of nodes) {
          let todoCurrentLevel = applyOneReactantReactions(node, reactions, {
            OCL: this.OCL,
            currentDepth: 1,
            processedMolecules: this.processedMolecules,
            moleculeInfoCallback: this.moleculeInfoCallback,
            maxDepth: this.maxDepth,
            maxCurrentDepth: max,
            stats,
            limitReactions: this.limitReactions
          });
          do {
            const nexts = [];
            for (const todo of todoCurrentLevel) {
              nexts.push(todo());
            }
            todoCurrentLevel = nexts.flat();
          } while (todoCurrentLevel.length > 0);
        }
        const newNodes = this.getNodes().filter(node => node.isValid);
        for (const node of newNodes) {
          if (node.currentDepth < min || node.currentDepth > max) {
            node.isValid = false;
          }
          delete node.currentDepth;
        }
      }
    }
    function clearAsFromProcessedMolecules(processedMolecules) {
      for (const [, value] of processedMolecules) {
        if (value.asReagent) {
          value.asReagent = false;
        }
        if (value.asProduct) {
          value.asProduct = false;
        }
      }
    }

    /**
     * The function performs the fragmentation of all single linear bonds
     * @param {import('openchemlib').Molecule} molecule - The OCL molecule to be fragmented
     * @param {object} [options={}]
     * @param {boolean} [options.calculateHoseCodes=false] - calculating hose code for bonds is quite time consuming
     * @param {string} [options.parentIDCode=molecule.getIDCode()]
     * @returns Results fragmentation of acyclic bonds
     */

    function fragmentAcyclicBonds(molecule, options = {}) {
      const {
        Molecule
      } = molecule.getOCL();
      const {
        calculateHoseCodes,
        parentIDCode = molecule.getIDCode()
      } = options;
      let atoms = [];
      // Prepare object with lenght equal to number of atoms
      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        let atom = {};
        atoms.push(atom);
        atom.i = i;
        atom.links = [];
      }
      let bonds = [];
      for (let i = 0; i < molecule.getAllBonds(); i++) {
        let bond = {};
        // get informations of bonds
        bond.index = i;
        bond.order = molecule.getBondOrder(i); // dative, single , double, triple
        bond.atom1 = molecule.getBondAtom(0, i); // atom 1 index
        bond.atom2 = molecule.getBondAtom(1, i); // atom 2 index

        bond.type = molecule.getBondType(i); // cBondTypeSingle,cBondTypeDouble,cBondTypeTriple,cBondTypeDelocalized
        bond.isAromatic = molecule.isAromaticBond(i);
        bond.isRingBond = molecule.isRingBond(i);

        // Mapping of bonds to be fragmented, only if they are single bond not aromatic and cyclic the mapping occurs
        if (bond.isAromatic || bond.type > 1 || bond.isRingBond || bond.order !== 1) {
          continue;
        } else {
          bond.selected = true;
          atoms[bond.atom1].links.push(bond.atom2);
          atoms[bond.atom2].links.push(bond.atom1);
        }
        bonds.push(bond);
      }
      let brokenMolecule = {};
      let fragmentMap = [];
      let nbFragments = [];
      let results = [];
      for (let bond of bonds) {
        if (bond.selected) {
          // if bond.selected is true (line 46) the molecule will be fragmented
          brokenMolecule[bond.index] = molecule.getCompactCopy(); // get a copy of the molecule
          brokenMolecule[bond.index].setAtomCustomLabel(bond.atom1, '*');
          brokenMolecule[bond.index].setAtomCustomLabel(bond.atom2, '*');
          brokenMolecule[bond.index].markBondForDeletion(bond.index); //mark bond to be deleted
          // the function returns an array of map

          brokenMolecule[bond.index].deleteMarkedAtomsAndBonds(); // delete marked bonds
        }

        nbFragments = brokenMolecule[bond.index].getFragmentNumbers(fragmentMap);
        // only if there are 2 fragments code can continue
        if (nbFragments === 2) {
          for (let i = 0; i < nbFragments; i++) {
            const result = {};
            if (calculateHoseCodes) {
              result.hoses = getHoseCodesForAtoms(molecule, [bond.atom1, bond.atom2]);
            }
            result.atomMap = [];

            // assign fragment id to index of for loop
            let includeAtom = fragmentMap.map(id => {
              return id === i;
            });
            let fragment = new Molecule(100, 100);
            let atomMap = [];
            brokenMolecule[bond.index].copyMoleculeByAtoms(fragment, includeAtom, false, atomMap);
            for (let j = 0; j < atomMap.length; j++) {
              if (fragment.getAtomCustomLabel(atomMap[j]) === '*') {
                result.atomMap.push(j);
                if (atoms[j].links.length > 0) {
                  fragment.addBond(atomMap[j], fragment.addAtom(154));
                }
              }
            }
            fragment.removeAtomCustomLabels();
            fragment.setFragment(false);
            result.idCode = fragment.getIDCode();
            result.parentIDCode = parentIDCode;
            result.cleavedBonds = [{
              index: bond.index,
              order: bond.order,
              atom1: bond.atom1,
              atom2: bond.atom2
            }];
            result.mfInfo = new MF(getMF(fragment).mf.replace(/R[1-9]?/, '')).getInfo();
            result.fragmentType = 'acyclic';
            results.push(result);
          }
        }
      }
      // sort result in order fragment 1-2; 3-4; ...
      results = results.sort((a, b) => {
        return a.mfInfo.monoisotopicMass - b.mfInfo.monoisotopicMass;
      });
      return results;
    }

    /**
     * This function returns ringBond, and object that contains information about the bonds of each ring
     * @param {import('openchemlib').Molecule} molecule - The OCL molecule to be fragmented
     * @returns Information of ring bonds for each ring in the molecule
     */

    function getRingsInfo(molecule) {
      const ringSet = molecule.getRingSet();
      let ringBonds = [];
      // create a new array with the length of the number of bonds in the molecule and fills it with 0
      let nbRingForBonds = new Array(molecule.getAllBonds()).fill(0);
      for (let i = 0; i < ringSet.getSize(); i++) {
        for (let bond of ringSet.getRingBonds(i)) {
          nbRingForBonds[bond]++;
        }
      }
      for (let i = 0; i < ringSet.getSize(); i++) {
        ringBonds.push({
          bonds: ringSet.getRingBonds(i).map(bondIndex => ({
            index: bondIndex,
            ringIndex: i,
            nbRings: nbRingForBonds[bondIndex],
            // in how many rings this bond is included
            order: molecule.getBondOrder(bondIndex),
            isAromatic: ringSet.isAromatic(i),
            atom1: molecule.getBondAtom(0, bondIndex),
            atom2: molecule.getBondAtom(1, bondIndex)
          }))
        });
      }
      return ringBonds;
    }

    /**
     * This function returns an array of objects with all combination of 2 bonds who can be fragmented in the same ring
     * @param {import('openchemlib').Molecule} molecule - The OCL molecule to be fragmented
     * @returns All combination of 2 bonds who can be fragmented in the same ring
     */
    function getFragmentableRings(molecule) {
      let ringsInfo = getRingsInfo(molecule);
      let fragmentableRingBonds = [];
      for (let ring = 0; ring < ringsInfo.length; ring++) {
        let bonds = ringsInfo[ring].bonds;
        // we prevent to consecutive bonds to be cleaved
        for (let first = 0; first < bonds.length; first++) {
          let end = first === 0 ? bonds.length - 1 : bonds.length;
          for (let second = first + 2; second < end; second++) {
            if (bonds[first].order === 1 && bonds[second].order === 1 && !bonds[first].isAromatic && !bonds[second].isAromatic && bonds[first].nbRings < 2 && bonds[second].nbRings < 2) {
              fragmentableRingBonds.push({
                bonds: [bonds[first], bonds[second]]
              });
            }
          }
        }
      }
      return fragmentableRingBonds;
    }

    /**
     * The function performs the fragmentation of all single ring bonds not belonging to aromatic rings
     * @param {import('openchemlib').Molecule} molecule - The OCL molecule to be fragmented
     * @param {object} [options={}]
     * @param {boolean} [options.calculateHoseCodes=false] - calculating hose code for bonds is quite time consuming
     * @param {string} [options.parentIDCode=molecule.getIDCode()]
     * @returns  Array with results for the fragmentation of ring bonds
     */

    function fragmentRings(molecule, options = {}) {
      const {
        Molecule
      } = molecule.getOCL();
      const {
        calculateHoseCodes,
        parentIDCode = molecule.getIDCode()
      } = options;
      const fragmentableRingBonds = getFragmentableRings(molecule);
      let fragmentationResults = [];
      for (let ringBonds of fragmentableRingBonds) {
        const brokenMolecule = molecule.getCompactCopy();
        let fragmentMap = [];
        let atoms = [];
        let rLinks = {};
        for (let bond of ringBonds.bonds) {
          brokenMolecule.markBondForDeletion(bond.index);
          brokenMolecule.setAtomCustomLabel(bond.atom1, '*');
          brokenMolecule.setAtomCustomLabel(bond.atom2, '*');
          atoms.push(bond.atom1);
          atoms.push(bond.atom2);
          rLinks[bond.atom1] = bond.atom2;
          rLinks[bond.atom2] = bond.atom1;
        }
        brokenMolecule.deleteMarkedAtomsAndBonds();
        const nbFragments = brokenMolecule.getFragmentNumbers(fragmentMap);
        for (let i = 0; i < nbFragments; i++) {
          const result = {};
          if (calculateHoseCodes) {
            result.hoses = getHoseCodesForAtoms(molecule, atoms);
          }
          result.atomMap = [];
          let includeAtom = fragmentMap.map(id => {
            return id === i;
          });
          let fragment = new Molecule(0, 0);
          let atomMap = [];
          brokenMolecule.copyMoleculeByAtoms(fragment, includeAtom, false, atomMap);
          // if includeAtom has more then 3 true all true should become false and all false should become true

          for (let j = 0; j < atomMap.length; j++) {
            if (fragment.getAtomCustomLabel(atomMap[j]) === '*') {
              result.atomMap.push(j);
              if (rLinks[j] !== undefined) {
                fragment.addBond(atomMap[j], fragment.addAtom(154));
              }
            }
          }
          fragment.removeAtomCustomLabels();
          fragment.setFragment(false);
          //      console.log(fragment.getIDCode(), getMF(fragment).mf);
          result.idCode = fragment.getIDCode();
          result.parentIDCode = parentIDCode;
          result.cleavedBonds = ringBonds.bonds;
          result.mfInfo = new MF(getMF(fragment).mf.replace(/R[1-9]?/, '')).getInfo();
          result.fragmentType = 'cyclic';
          fragmentationResults.push(result);
        }
      }
      fragmentationResults = fragmentationResults.sort((a, b) => {
        return a.mfInfo.monoisotopicMass - b.mfInfo.monoisotopicMass;
      });
      return fragmentationResults;
    }

    /**
     * This function fragment both acyclic and cyclic bonds of the molecule
     * @param {import('openchemlib').Molecule} molecule - The OCL molecule to be fragmented
     * @param {object} [options={}]
     * @param {boolean} [options.calculateHoseCodes=false] - calculating hose code for bonds is quite time consuming
     * @param {boolean} [options.cyclic=true] - calculate cyclic fragmentation
     * @param {boolean} [options.acyclic=true] - calculate acyclic fragmentation
     * @param {boolean} [options.full=true] - calculate the molecular formula of the full molecule
     * @returns {object} In-Silico fragmentation results
     */
    function fragment(molecule, options = {}) {
      const {
        cyclic = true,
        acyclic = true,
        full = true,
        calculateHoseCodes = false
      } = options;
      const parentIDCode = molecule.getIDCode();
      let molecularIon = full ? [{
        idCode: parentIDCode,
        parentIDCode,
        mfInfo: new MF(getMF(molecule).mf).getInfo(),
        fragmentType: 'molecule'
      }] : [];
      let acyclicBonds = acyclic ? fragmentAcyclicBonds(molecule, {
        calculateHoseCodes,
        parentIDCode
      }) : [];
      let cyclicBonds = cyclic ? fragmentRings(molecule, {
        calculateHoseCodes,
        parentIDCode
      }) : [];
      let result = [...molecularIon, ...acyclicBonds, ...cyclicBonds];
      return result.sort((a, b) => a.mfInfo.monoisotopicMass - b.mfInfo.monoisotopicMass);
    }

    var core = {};

    var openchemlibCore = {};

    (function(exports){(function(root){function getExports($wnd){var $doc=$wnd.document;var $gwt={};var navigator={userAgent:'webkit'};function noop(){}var __gwtModuleFunction=noop;__gwtModuleFunction.__moduleStartupDone=noop;var Hlb='object',Ilb='anonymous',Jlb='fnStack',Klb='\n',Llb={4:1,10:1,5:1,8:1},Mlb='Unknown',Nlb='boolean',Olb='number',Plb='string',Qlb=2147483647,Rlb='For input string: "',Slb='null',Tlb='__noinit__',Ulb={4:1,5:1,8:1},Vlb={4:1,15:1,22:1},Wlb=' (copy)',Xlb={4:1,10:1,31:1,5:1,17:1,8:1,29:1},Ylb=65536,Zlb=65535,$lb=10000,_lb=', length: ',amb='Index: ',bmb=', Size: ',cmb='fromIndex: ',dmb=', toIndex: ',emb={12:1,4:1,5:1},fmb={4:1},gmb={7:1,4:1,5:1},hmb=16777215,imb=65280,jmb=0.30000001192092896,kmb=545460846592,lmb=4294967296,mmb=8589934592,nmb=17179869184,omb=34359738368,pmb=68719476736,qmb=137438953472,rmb=274877906944,smb='<line x1="',tmb='" ',umb='y1="',vmb='x2="',wmb='y2="',xmb='stroke="',ymb='stroke-width="',zmb='" />',Amb={11:1,4:1,5:1},Bmb=536870912,Cmb=2.617993878,Dmb=3.665191429,Emb=6.283185307179586,Fmb=3.141592653589793,Gmb=1.5707963267948966,Hmb=16384,Imb=4096,Jmb={l:0,m:0,h:1},Kmb={l:0,m:0,h:2},Lmb={l:0,m:0,h:4},Mmb=1920,Nmb=1024,Omb=234881024,Pmb=167772160,Qmb=100663296,Rmb=201326592,Smb=114688,Tmb=4063232,Umb=3538944,Vmb=3014656,Wmb=2097152,Xmb=393216,Ymb=917504,Zmb=1966080,$mb=17042430230528,_mb=16492674416640,anb=15942918602752,bnb=14843406974976,cnb=12644383719424,dnb=8796093022208,enb=549755813888,fnb=1649267441664,gnb=3848290697216,hnb=8246337208320,inb=29360128,jnb=268435456,knb=8192,lnb=2048,mnb=-1.5707963267948966,nnb=130560,onb=6291456,pnb=1.0471975511965976,qnb=0.5235987755982988,rnb=786432,snb=524288,tnb=262144,unb=-16777216,vnb=16711680,wnb=-32768,xnb='<DrawingObject type="',ynb='><\/DrawingObject>',znb={4:1,5:1,13:1},Anb={4:1,5:1},Bnb='unsupported atomicNo:',Cnb=2.8415926535897933,Dnb={26:1,4:1,10:1,5:1,8:1},Enb={4:1,5:1,17:1,8:1},Fnb={9:1,4:1,5:1},Gnb=131072,Hnb={l:6,m:0,h:4},Inb=1048576,Jnb=3221225472,Knb={l:0,m:0,h:3},Lnb=8388608,Mnb=-65536,Nnb={38:1,39:1,45:1},Onb={38:1,39:1,80:1,45:1,98:1},Pnb={4:1,5:1,38:1,39:1,80:1,18:1,45:1,98:1,156:1},Qnb='??',Rnb=2013265920,Snb=32505856,Tnb=31744,Unb=1.7976931348623157E308,Vnb={105:1,4:1,10:1,5:1,17:1,8:1},Wnb=33554432,Xnb=67108864,Ynb=32768,Znb=-33554433,$nb=-33292289,_nb=-3.141592653589793,aob=4194303,bob=3072,cob={36:1,4:1,5:1},dob=0.7853981633974483,eob=3.061592653589793,fob=1073741824,gob=541165879296,hob={14:1,4:1,5:1,8:1},iob=281.17,job=289.19,kob=294.21,lob='0.0000',mob='M  END\n',nob='ATOMS',oob=-2147483648,pob='M  END',qob='$$$$',rob='M  V30 ',sob='M  V30 COUNTS ',tob=')\n',uob='M  V30 MDLV30/STEREL',vob='M  V30 MDLV30/STERAC',wob=3.4028234663852886E38,xob={l:2,m:0,h:4},yob=4194304,zob={l:4179966,m:4063288,h:7},Aob={l:4194302,m:4193343,h:7},Bob={117:1,103:1,104:1,4:1,10:1,5:1,8:1},Cob='class="event" ',Dob='opacity="0" />',Eob='Assignment of aromatic double bonds failed',Fob=1099511627776,Gob=2199023255552,Hob=4398046511104,Iob='Members of ESR groups must only be stereo centers with known configuration.',Job='Over- or under-specified stereo feature or more than one racemic type bond',Kob='Ambiguous configuration at stereo center because of 2 parallel bonds',Lob={91:1,75:1,25:1,4:1},Mob=100000,Nob=3.1415927410125732,Oob=1.5707963705062866,Pob=0.029999999329447746,Qob=6.2831854820251465,Rob=1.9106119871139526,Sob=2.2639999389648438,Tob=2.16759991645813,Uob=2.0662999153137207,Vob=2.3561999797821045,Wob=2.2844998836517334,Xob=1.9106120066311025,Yob=2.094395160675049,Zob=0.6000000238418579,$ob=1.7999999523162842,_ob=0.05000000074505806,apb={4:1,10:1,5:1,8:1,30:1},bpb=0.2617993950843811,cpb=2.0943951023931953,dpb=1.2000000476837158,epb=1.4299999475479126,fpb=1.659999966621399,gpb=1.4600000381469727,hpb=1.8200000524520874,ipb=2.4600000381469727,jpb=2.4200000762939453,kpb=2.450000047683716,lpb=2.440000057220459,mpb=2.4000000953674316,npb=2.319999933242798,opb=2.2899999618530273,ppb=1.8600000143051147,qpb=2.5299999713897705,rpb=2.490000009536743,spb=2.4700000286102295,tpb=2.880000114440918,upb=2.9200000762939453,vpb=2.869999885559082,wpb=2.8299999237060547,xpb=2.7899999618530273,ypb=2.809999942779541,zpb=2.799999952316284,Apb=3.049999952316284,Bpb=1.3300000429153442,Cpb=1.3899999856948853,Dpb=1.159999966621399,Epb=1.1100000143051147,Fpb=1.0700000524520874,Gpb=1.9600000381469727,Hpb=1.7100000381469727,Ipb=1.3600000143051147,Jpb=1.340000033378601,Kpb=1.2200000286102295,Lpb=1.2400000095367432,Mpb=1.2100000381469727,Npb=1.850000023841858,Opb=1.6299999952316284,Ppb=1.4700000286102295,Qpb=1.2799999713897705,Rpb=1.440000057220459,Spb=1.399999976158142,Tpb=1.7599999904632568,Upb=1.7300000190734863,Vpb=1.7200000286102295,Wpb=1.6799999475479126,Xpb=1.690000057220459,Ypb=1.6699999570846558,Zpb=1.649999976158142,$pb=1.7000000476837158,_pb=1.6200000047683716,aqb=1.3700000047683716,bqb=1.2899999618530273,cqb=1.4500000476837158,dqb=1.5700000524520874,eqb=-0.5235987755982988,fqb=-1.0471975511965976,gqb=-0.7853981633974483,hqb=0.017453292519943295,iqb=0.17453292519943295,jqb=1.0E-4,kqb={114:1},lqb=57.29577951308232,mqb=0.021922173386725217,nqb=1.0E-8,oqb=2.5120960256267386,pqb=1.0E-5,qqb='MMFF94s+',rqb='nonbonded cutoff',sqb='dielectric constant',tqb='dielectric model',uqb='angle bend',vqb='bond stretch',wqb='electrostatic',xqb='out of plane',yqb='stretch bend',zqb='torsion angle',Aqb='van der waals',Bqb={4:1,10:1,5:1,17:1,8:1},Cqb=0.07000000000000006,Dqb=0.1200000000000001,Eqb=-7.840000000000001,Fqb='OCL_RXN_V1.0:',Gqb="'$MOL' tag not found",Hqb="'M  END' not found",Iqb='\nActelion Java MolfileCreator 2.0\n\n  0  0  0  0  0  0              0 V3000\n',Jqb='M  V30 END CTAB',Kqb=2475109711874,Lqb=2475109711875,Mqb=2887426572290,Nqb=2887426572291,Oqb=2887627898882,Pqb=2887627898883,Qqb=2887627997186,Rqb=3437182386178,Sqb=3437182386179,Tqb=3437383712770,Uqb=3437383811074,Vqb=3437652148226,Wqb=3437652377602,Xqb=4674132967426,Yqb=4674334294018,Zqb=4674334392322,$qb=4674602729474,_qb=4674602958850,arb=5086449827842,brb=5086449827848,crb=5086651154434,drb=5086651252738,erb=5086651252743,frb=5086919589890,grb=5086919688194,hrb=5636406968322,irb=5636407066626,jrb=5636675502082,krb={l:589826,m:590112,h:4},lrb={l:589827,m:590112,h:4},mrb={l:589826,m:590160,h:4},nrb={l:589827,m:590160,h:4},orb={l:688130,m:590160,h:4},prb={l:589826,m:590224,h:4},qrb={l:589827,m:590224,h:4},rrb={l:819203,m:590224,h:4},srb={l:589826,m:590368,h:4},trb={l:688130,m:590368,h:4},urb={l:819202,m:590368,h:4},vrb={l:819208,m:590368,h:4},wrb={l:589826,m:590416,h:4},xrb={l:589832,m:590416,h:4},yrb={l:688130,m:590416,h:4},zrb={l:819202,m:590416,h:4},Arb={l:589826,m:598304,h:4},Brb={l:589827,m:598304,h:4},Crb={l:688130,m:598304,h:4},Drb={l:1114114,m:598304,h:4},Erb={l:1212418,m:598304,h:4},Frb={l:589826,m:598308,h:4},Grb={l:589826,m:688416,h:4},Hrb={l:589826,m:688464,h:4},Irb={l:589826,m:688720,h:4},Jrb={l:688130,m:688720,h:4},Krb={l:589826,m:590112,h:288},Lrb={l:589826,m:590112,h:336},Mrb={l:589826,m:688416,h:336},Nrb={l:589826,m:688464,h:336},Orb={l:688130,m:688464,h:336},Prb={l:589826,m:590112,h:400},Qrb={l:589826,m:688416,h:400},Rrb={l:589826,m:688464,h:400},Srb={l:688130,m:688464,h:400},Trb={l:589826,m:819488,h:400},Urb={l:589826,m:819536,h:400},Vrb={l:589826,m:819600,h:400},Wrb={l:688130,m:819600,h:400},Xrb={l:819202,m:819600,h:400},Yrb={l:589831,m:688416,h:592},Zrb={l:589831,m:688464,h:592},$rb={l:688135,m:688464,h:592},_rb={l:589831,m:819536,h:592},asb={l:688135,m:819536,h:592},bsb={l:688135,m:819600,h:592},csb={l:589832,m:1212704,h:592},dsb={l:589832,m:1212752,h:592},esb={l:688136,m:1212752,h:592},fsb={l:589831,m:688464,h:656},gsb={l:589826,m:590112,h:8480},hsb={l:589826,m:688416,h:8480},isb={l:589826,m:688464,h:8480},jsb={l:688130,m:688464,h:8480},ksb={l:819202,m:819600,h:8480},lsb={l:688135,m:1212752,h:8480},msb={l:589832,m:1213008,h:8480},nsb={l:688136,m:1213008,h:8480},osb={l:589826,m:590112,h:8484},psb={l:589826,m:590160,h:8484},qsb={l:688130,m:598304,h:8484},rsb=0.6262000203132629,ssb=-1.3825000524520874,tsb=0.33169999718666077,usb=-1.4915000200271606,vsb=0.3540000021457672,wsb=0.38179999589920044,xsb=-0.6019999980926514,ysb=-0.7379999756813049,zsb='Atom-types are 64-bit numbers describing atoms and their near surrounding.',Asb='Recognized atom types and their contributions are:',Bsb='Druglikeness predictor not properly initialized.',Csb=3.009999990463257,Dsb=-0.17000000178813934,Esb=-0.1809999942779541,Fsb=-0.2029999941587448,Gsb=2147483648,Hsb='MMFF94',Isb='MMFF94s',Jsb='117\nint,int,int,int,float\n0,2,0,0,0.020\n1,2,1,2,0.030\n1,2,2,2,0.027\n1,2,2,3,0.026\n1,2,2,5,0.013\n1,2,2,37,0.032\n2,2,2,5,0.013\n2,2,3,5,0.012\n2,2,5,5,0.006\n2,2,5,6,0.027\n2,2,5,37,0.017\n2,2,5,40,0.012\n2,2,5,41,0.008\n0,3,0,0,0.130\n1,3,1,7,0.146\n1,3,2,7,0.138\n1,3,3,7,0.134\n1,3,5,7,0.122\n1,3,6,7,0.141\n1,3,7,10,0.129\n1,3,7,37,0.138\n2,3,5,7,0.113\n2,3,5,9,0.081\n2,3,6,7,0.127\n2,3,7,10,0.116\n3,3,5,7,0.113\n3,3,6,7,0.127\n5,3,5,7,0.103\n5,3,5,9,0.074\n5,3,5,54,0.078\n5,3,6,7,0.119\n5,3,7,10,0.102\n5,3,9,40,0.067\n6,3,7,37,0.127\n7,3,10,10,0.113\n7,3,20,20,0.151\n9,3,40,40,0.057\n0,8,0,0,0.000\n0,10,0,0,0.015\n1,10,1,3,0.015\n1,10,3,6,0.015\n1,10,3,28,0.015\n3,10,3,28,0.015\n3,10,28,28,0.015\n0,17,0,0,0.000\n0,26,0,0,0.000\n0,30,0,0,0.010\n5,30,20,30,0.008\n0,37,0,0,0.035\n1,37,37,37,0.040\n2,37,37,37,0.031\n3,37,37,37,0.027\n5,37,37,37,0.015\n5,37,37,38,0.046\n5,37,37,63,0.008\n5,37,37,64,0.012\n5,37,37,69,0.016\n5,37,38,38,0.084\n6,37,37,37,0.048\n15,37,37,37,0.025\n37,37,37,40,0.046\n0,39,0,0,0.020\n1,39,63,63,0.012\n23,39,63,63,-0.014\n23,39,63,65,0.021\n23,39,65,65,0.062\n0,40,0,0,0.030\n1,40,28,37,0.030\n2,40,28,28,0.030\n3,40,28,28,0.030\n28,40,28,37,0.030\n0,41,0,0,0.180\n1,41,32,32,0.178\n2,41,32,32,0.161\n5,41,32,32,0.158\n0,43,0,0,0.000\n0,45,0,0,0.150\n0,49,0,0,0.000\n50,49,50,50,0.000\n0,54,0,0,0.020\n1,54,3,36,0.016\n3,54,36,36,0.018\n0,55,0,0,0.020\n1,55,36,57,0.020\n36,55,36,57,0.020\n0,56,0,0,0.020\n1,56,36,57,0.020\n36,56,36,57,0.020\n0,57,0,0,0.080\n5,57,55,55,0.038\n56,57,56,56,0.158\n0,58,0,0,0.025\n0,63,0,0,0.050\n5,63,39,64,0.019\n5,63,39,66,0.068\n5,63,44,64,0.014\n5,63,44,66,0.055\n5,63,59,64,0.033\n5,63,59,66,0.085\n37,63,39,64,0.010\n0,64,0,0,0.040\n5,64,63,64,0.006\n5,64,63,66,0.043\n5,64,64,65,0.052\n5,64,65,66,0.094\n37,64,63,64,-0.01\n0,67,0,0,0.070\n0,69,0,0,0.070\n32,69,37,37,0.067\n0,73,0,0,0.000\n0,78,0,0,0.045\n5,78,78,81,0.046\n0,80,0,0,0.080\n5,80,81,81,0.057\n0,81,0,0,0.025\n36,81,78,80,0.016\n0,82,0,0,0.000\n',Ksb={214:1,4:1,5:1,8:1},Lsb='undefined',Msb=0.08726646502812703,Nsb=0.003921568859368563,Osb='Too many percent/per mille characters in pattern "',Psb=1048575,Qsb=17592186044416,Rsb=1000000000,Ssb=-17592186044416,Tsb='CSS1Compat',Usb=5.56,Vsb=11.12,Wsb=13.34,Xsb=14.44,Ysb=0.3010299956639812,Zsb='BigInteger divide by zero',$sb=4294967295,_sb={l:0,m:0,h:524288},atb={27:1,88:1},btb={99:1},ctb={38:1,39:1,45:1,86:1},dtb='delete',etb=15525485,ftb=5.9604644775390625E-8,gtb=16777216,htb={4:1,27:1,52:1,64:1},itb='Invalid UTF8 sequence',jtb=0.20000000298023224,ktb=' distance:',ltb='gGP`@dfyjidNcGI[WQCP`<',mtb={43:1};var Ykb={};var Zkb={};var $kb={};var _kb={};var alb={};var blb={};var clb={};var dlb={};var elb={};var flb={};var glb={};var hlb={};var ilb={};var jlb={};var klb={};var llb={};var mlb={};var nlb={};var olb={};var plb={};var qlb={};var rlb={};var slb={};var tlb={};var ulb={};var vlb={};var wlb={};var xlb={};var ylb={};var zlb={};var Alb={};var Blb={};var Clb={};var Dlb={};var Elb={};var Flb={};var Glb={};var _;var R1;var L1;Ykb._0=function _0(){};function Q1(a,b){typeof window===Hlb&&typeof window['$gwt']===Hlb&&(window['$gwt'][a]=b);}function P1(b,c,d,e){Ykb.O1();var f=L1;function g(){for(var a=0;a<f.length;a++){f[a]();}}if(b){try{Xkb(g)();}catch(a){b(c,a);}}else {Xkb(g)();}}Ykb.O1=function O1(){L1==null&&(L1=[]);};function N1(){Ykb.O1();var a=L1;for(var b=0;b<arguments.length;b++){a.push(arguments[b]);}}Ykb.M1=function M1(){};function a2(){}function _1(a){if(Array.isArray(a)&&a.uc===a2){return Zkb.Mc(a);}return a.toString();}function $1(a,b){var c=$wnd;if(a===''){return c;}var d=a.split('.');!(d[0]in c)&&c.execScript&&c.execScript('var '+d[0]);if(b){var e=b.prototype.sc;e.e=b;}for(var f;d.length&&(f=d.shift());){c=c[f]=c[f]||!d.length&&b||{};}return c;}Ykb.Z1=function Z1(a){function b(){}b.prototype=a||{};return new b();};Ykb.Y1=function Y1(a){return a instanceof Array?a[0]:null;};function X1(a,b,c){var d=function(){return a.apply(d,arguments);};b.apply(d,c);return d;}function W1(){}function V1(a,b){for(var c in b){b[c]['configurable']=true;}Object.defineProperties(a,b);}function U1(a,b,c){var d=R1;var e=d[a];var f=Ykb.Y1(e);if(e&&!f){_=e;}else {_=Ykb.T1(b);_.tc=c;!b&&(_.uc=a2);d[a]=_;}for(var g=3;g<arguments.length;++g){arguments[g].prototype=_;}f&&(_.sc=f);}Ykb.T1=function T1(a){var b=a&&a.prototype;!b&&(b=R1[a]);return Ykb.Z1(b);};Ykb.S1=function S1(){};$wnd.goog=$wnd.goog||{};$wnd.goog.global=$wnd.goog.global||$wnd;R1={};Zkb.Bc=function Bc(a,b){return uX(a)===uX(b);};Zkb.Cc=function Cc(a){return a.sc;};Zkb.Dc=function Dc(a){return alb.wgb(a);};function Ec(){}Zkb.Gc=function Gc(a,b){return rX(a)?Zkb.c5(a,b):oX(a)?Zkb.V3(a,b):nX(a)?Zkb.P2(a,b):lX(a)?a.ib(b):iW(a)?Zkb.Bc(a,b):$kb.pU(a,b);};Zkb.Ic=function Ic(a){return rX(a)?Zkb.e5():oX(a)?Zkb.W3():nX(a)?Zkb.Q2():lX(a)?a.sc:iW(a)?Zkb.Cc(a):$kb.qU(a);};Zkb.Kc=function Kc(a){return rX(a)?Zkb.f5(a):oX(a)?Zkb.X3(a):nX(a)?Zkb.R2(a):lX(a)?a.kb():iW(a)?Zkb.Dc(a):$kb.rU(a);};Zkb.Mc=function Mc(a){var b;return Zkb.q3(Zkb.Ic(a))+'@'+(b=alb.Pgb(alb.Ogb(Zkb.Kc(a))),b.toString(16));};U1(1,null,{},Ec);_.ib=function Fc(a){return Zkb.Bc(this,a);};_.jb=function Hc(){return Zkb.Cc(this);};_.kb=function Jc(){return Zkb.Dc(this);};_.lb=function Lc(){var a;return Zkb.q3(Zkb.Ic(this))+'@'+(a=alb.Pgb(alb.Ogb(Zkb.Kc(this))),a.toString(16));};_.equals=function(a){return this.ib(a);};_.hashCode=function(){return this.kb();};_.toString=function(){return this.lb();};$kb.pU=function pU(a,b){return $kb.vU(a)?$kb.sU(a,b):uX(a)===uX(b);};$kb.qU=function qU(a){return a.sc||Array.isArray(a)&&aW(Ykb.r$,1)||Ykb.r$;};$kb.rU=function rU(a){return $kb.wU(a)?$kb.tU(a):alb.wgb(a);};$kb.sU=function sU(a,b){return a.equals(b);};$kb.tU=function tU(a){return a.hashCode();};$kb.uU=function uU(){return [];};$kb.vU=function vU(a){return !!a&&!!a.equals;};$kb.wU=function wU(a){return !!a&&!!a.hashCode;};$kb.xU=function xU(a){return a.toString?a.toString():'[JavaScriptObject]';};_kb.bV=function bV(){_kb.bV=W1;var a,b;b=!_kb.kV();a=new _kb.tV();_kb.aV=b?new _kb.lV():a;};_kb.cV=function cV(a){_kb.bV();_kb.aV.Tb(a);};_kb.dV=function dV(a){_kb.bV();var b;b=_kb.aV.Ub(a);return _kb.eV(b);};_kb.eV=function eV(a){var b,c,d,e;b='cV';c='mz';e=$wnd.Math.min(a.length,5);for(d=e-1;d>=0;d--){if(Zkb.c5(a[d].d,b)||Zkb.c5(a[d].d,c)){a.length>=d+1&&alb.Pgb(a).splice(0,d+1);break;}}return a;};_kb.fV=function fV(a){var b=/function(?:\s+([\w$]+))?\s*\(/;var c=b.exec(a);return c&&c[1]||Ilb;};_kb.gV=function gV(a){_kb.bV();return a&&a[Jlb]?a[Jlb]:[];};_kb.hV=function hV(a){_kb.bV();return a.name||(a.name=_kb.fV(a.toString()));};_kb.iV=function iV(a){_kb.bV();return parseInt(a)||-1;};_kb.jV=function jV(a){_kb.bV();var b=a.backingJsObject;if(b&&b.stack){var c=b.stack;var d=b+Klb;c.substring(0,d.length)==d&&(c=c.substring(d.length));return c.split(Klb);}return [];};_kb.kV=function kV(){if(Error.stackTraceLimit>0){$wnd.Error.stackTraceLimit=Error.stackTraceLimit=64;return true;}return 'stack'in new Error();};U1(330,1,{});_kb.lV=function lV(){};U1(236,330,{},_kb.lV);_.Tb=function mV(a){var b={};var c=[];a[Jlb]=c;var d=arguments.callee.caller;while(d){var e=_kb.hV(d);c.push(e);var f=':'+e;var g=b[f];if(g){var h,i;for(h=0,i=g.length;h<i;h++){if(g[h]===d){return;}}}(g||(b[f]=[])).push(d);d=d.caller;}};_.Ub=function nV(a){var b,c,d,e;d=_kb.gV(a);c=$kb.DU(d);e=gW(Ykb.h_,Llb,66,c,0,1);for(b=0;b<c;b++){e[b]=new Zkb.T4($kb.CU(d,b),null,-1);}return e;};_kb.oV=function oV(a,b){var c,d,e,f,g,h,i,j,k;j='';if(alb.Pgb(b).length==0){return a.Vb(Mlb,Ilb,-1,-1);}k=Zkb.q5(b);Zkb.c5(alb.Pgb(k).substr(0,3),'at ')&&(k=(alb.Kgb(3,alb.Pgb(k).length+1),alb.Pgb(k).substr(3)));k=_kb.pV(k);g=alb.Pgb(k).indexOf('(');if(g==-1){g=alb.Pgb(k).indexOf('@');if(g==-1){j=k;k='';}else {j=Zkb.q5((alb.Kgb(g+1,alb.Pgb(k).length+1),alb.Pgb(k).substr(g+1)));k=Zkb.q5((alb.Jgb(0,g,alb.Pgb(k).length),alb.Pgb(k).substr(0,g)));}}else {c=alb.Pgb(k).indexOf(')',g);j=(alb.Jgb(g+1,c,alb.Pgb(k).length),alb.Pgb(k).substr(g+1,c-(g+1)));k=Zkb.q5((alb.Jgb(0,g,alb.Pgb(k).length),alb.Pgb(k).substr(0,g)));}g=Zkb.g5(k,s5(46));g!=-1&&(k=(alb.Kgb(g+1,alb.Pgb(k).length+1),alb.Pgb(k).substr(g+1)));(alb.Pgb(k).length==0||Zkb.c5(k,'Anonymous function'))&&(k=Ilb);h=Zkb.j5(j,s5(58));e=Zkb.k5(j,s5(58),h-1);i=-1;d=-1;f=Mlb;if(h!=-1&&e!=-1){f=(alb.Jgb(0,e,alb.Pgb(j).length),alb.Pgb(j).substr(0,e));i=_kb.iV((alb.Jgb(e+1,h,alb.Pgb(j).length),alb.Pgb(j).substr(e+1,h-(e+1))));d=_kb.iV((alb.Kgb(h+1,alb.Pgb(j).length+1),alb.Pgb(j).substr(h+1)));}return a.Vb(f,k,i,d);};_kb.pV=function pV(a){return a.replace(/\[.*?\]/g,'');};U1(331,330,{});_.Tb=function qV(a){};_.Vb=function rV(a,b,c,d){return new Zkb.T4(b,a+'@'+d,c<0?-1:c);};_.Ub=function sV(a){var b,c,d,e,f,g;e=_kb.jV(a);f=gW(Ykb.h_,Llb,66,0,0,1);b=0;d=$kb.DU(e);if(d==0){return f;}g=_kb.oV(this,$kb.CU(e,0));Zkb.c5(g.d,Ilb)||(f[b++]=g);for(c=1;c<d;c++){f[b++]=_kb.oV(this,$kb.CU(e,c));}return f;};_kb.tV=function tV(){};U1(237,331,{},_kb.tV);_.Vb=function uV(a,b,c,d){return new Zkb.T4(b,a,-1);};Ykb._V=function _V(a){return a;};function aW(a,b){return Ykb.bW(a,b);}Ykb.bW=function bW(a,b){return Zkb.A3(a,b);};Ykb.cW=function cW(a){return a.__elementTypeCategory$==null?10:a.__elementTypeCategory$;};Ykb.dW=function dW(a){return a.__elementTypeId$;};function eW(a,b,c,d,e,f){return Ykb.fW(a,b,c,d,e,0,f);}Ykb.fW=function fW(a,b,c,d,e,f,g){var h,i,j,k,l;k=e[f];j=f==g-1;h=j?d:0;l=Ykb.hW(h,k);d!=10&&oW(aW(a,g-f),b[f],c[f],h,l);if(!j){++f;for(i=0;i<k;++i){Ykb.jW(l,i,Ykb.fW(a,b,c,d,e,f,g));}}return l;};function gW(a,b,c,d,e,f){var g;g=Ykb.hW(e,d);e!=10&&oW(aW(a,f),b,c,e,g);return g;}Ykb.hW=function hW(a,b){var c=new Array(b);var d;switch(a){case 14:case 15:d=0;break;case 16:d=false;break;default:return c;}for(var e=0;e<b;++e){c[e]=d;}return c;};function iW(a){return Array.isArray(a)&&a.uc===a2;}Ykb.jW=function jW(a,b,c){return a[b]=c;};function kW(a,b,c){return Ykb.jW(a,b,c);}Ykb.lW=function lW(a,b){a.sc=b;};Ykb.mW=function mW(a,b){a.__elementTypeCategory$=b;};Ykb.nW=function nW(a,b){a.__elementTypeId$=b;};function oW(a,b,c,d,e){Ykb.lW(e,a);e.tc=b;e.uc=a2;Ykb.nW(e,c);Ykb.mW(e,d);return e;}Ykb.pW=function pW(a,b){Ykb.cW(b)!=10&&oW(Zkb.Ic(b),b.tc,Ykb.dW(b),Ykb.cW(b),a);return Ykb._V(a);};function kX(a,b){if(rX(a)){return !!jX[b];}else if(a.tc){return !!a.tc[b];}else if(oX(a)){return !!iX[b];}else if(nX(a)){return !!hX[b];}return false;}function lX(a){return !Array.isArray(a)&&a.uc===a2;}function mX(a,b){return a!=null&&kX(a,b);}function nX(a){return typeof a===Nlb;}function oX(a){return typeof a===Olb;}function pX(a){return a!=null&&Ykb.sX(a)&&!(a.uc===a2);}function qX(a,b){return Ykb.tX(a,b);}function rX(a){return typeof a===Plb;}Ykb.sX=function sX(a){return typeof a===Hlb||typeof a==='function';};Ykb.tX=function tX(a,b){return a&&b&&a instanceof b;};function uX(a){return a==null?null:a;}function vX(a){return Math.max(Math.min(a,Qlb),-2147483648)|0;}var hX;var iX;var jX;Ykb.a1=function a1(a){return a&&a.__java$exception;};function b1(a){var b;if(mX(a,22)){return a;}b=Ykb.a1(a);if(!b){b=new $kb.jU(a);_kb.cV(b);}return b;}function c1(a){return a.backingJsObject;}function s2(a){var b;if(a==null){return false;}b=typeof a;return Zkb.c5(b,Nlb)||Zkb.c5(b,Olb)||Zkb.c5(b,Plb)||a.$implements__java_io_Serializable||Array.isArray(a);}function L3(a){var b;b=typeof a;if(Zkb.c5(b,Nlb)||Zkb.c5(b,Olb)||Zkb.c5(b,Plb)){return true;}return a!=null&&a.$implements__java_lang_Comparable;}Zkb.L2=function L2(){Zkb.L2=W1;K2=alb.Pgb(false);alb.Pgb(true);};Zkb.M2=function M2(a){return alb.Qgb((alb.Egb(a),a));};Zkb.N2=function N2(a,b){return T2(alb.Qgb((alb.Egb(a),a)),alb.Qgb((alb.Egb(b),b)));};Zkb.O2=function O2(a,b){return Zkb.N2(a,b);};Zkb.P2=function P2(a,b){return alb.Egb(a),uX(a)===uX(b);};Zkb.Q2=function Q2(){return Ykb.R$;};Zkb.R2=function R2(a){return alb.Qgb((alb.Egb(a),a))?1231:1237;};function S2(a){Zkb.L2();return Zkb.c5(Nlb,typeof a);}function T2(a,b){Zkb.L2();return a==b?0:a?1:-1;}Zkb.U2=function U2(a,b){Zkb.L2();return rX(a)?Zkb.Y4(a,b):oX(a)?Zkb.T3(a,b):nX(a)?Zkb.O2(a,b):a.qb(b);};hX={4:1,233:1,27:1};var K2;function V2(a){if(Zkb.c5(typeof a,Plb)){return true;}return a!=null&&a.$implements__java_lang_CharSequence;}Zkb.o3=function o3(a){a.i=n3++;};Zkb.p3=function p3(a){if(a.n!=null){return;}Zkb.E3(a);};Zkb.q3=function q3(a){Zkb.p3(a);return a.n;};Zkb.r3=function r3(a){return (a.f&4)!=0;};Zkb.s3=function s3(a){return (a.f&1)!=0;};Zkb.t3=function t3(){Zkb.o3(this);this.n=null;this.j=null;this.g=null;this.d=null;this.b=null;this.k=null;this.a=null;};Zkb.v3=function v3(a){var b;b=new Zkb.t3();b.n='Class$'+(a?'S'+a:''+b.i);b.b=b.n;b.j=b.n;return b;};function w3(a){var b;b=Zkb.v3(a);Zkb.I3(a,b);return b;}function x3(a,b){var c;c=Zkb.v3(a);Zkb.I3(a,c);c.f=b?8:0;return c;}function y3(a){var b;b=Zkb.v3(a);b.k=a;b.f=1;return b;}Zkb.A3=function A3(a,b){var c=a.a=a.a||[];return c[b]||(c[b]=a.Wb(b));};function C3(a){if(a._b()){return null;}var b=a.k;return R1[b];}Zkb.E3=function E3(a){if(a.$b()){var b=a.c;b._b()?a.n='['+b.k:!b.$b()?a.n='[L'+b.Yb()+';':a.n='['+b.Yb();a.b=b.Xb()+'[]';a.j=b.Zb()+'[]';return;}var c=a.g;var d=a.d;d=d.split('/');a.n=Zkb.H3('.',[c,Zkb.H3('$',d)]);a.b=Zkb.H3('.',[c,Zkb.H3('.',d)]);a.j=d[d.length-1];};Zkb.H3=function H3(a,b){var c=0;while(!b[c]||b[c]==''){c++;}var d=b[c++];for(;c<b.length;c++){if(!b[c]||b[c]==''){continue;}d+=a+b[c];}return d;};Zkb.I3=function I3(a,b){if(!a){return;}b.k=a;var d=C3(b);if(!d){R1[a]=[b];return;}d.sc=b;};U1(157,1,{},Zkb.t3);_.Wb=function u3(a){var b;b=new Zkb.t3();b.f=4;a>1?b.c=Zkb.A3(this,a-1):b.c=this;return b;};_.Xb=function z3(){Zkb.p3(this);return this.b;};_.Yb=function B3(){return Zkb.q3(this);};_.Zb=function D3(){Zkb.p3(this);return this.j;};_.$b=function F3(){return Zkb.r3(this);};_._b=function G3(){return Zkb.s3(this);};_.lb=function J3(){return ((this.f&2)!=0?'interface ':(this.f&1)!=0?'':'class ')+(Zkb.p3(this),this.n);};_.f=0;_.i=0;var n3=1;Zkb.N3=function N3(a){return Zkb.c5(Olb,typeof a)||qX(a,$wnd.java.lang.Number$impl);};Zkb.O3=function O3(a){Zkb.M3==null&&(Zkb.M3=new RegExp('^\\s*[+-]?(NaN|Infinity|((\\d+\\.?\\d*)|(\\.\\d+))([eE][+-]?\\d+)?[dDfF]?)\\s*$'));if(!Zkb.M3.test(a)){throw c1(new Zkb.S4(Rlb+a+'"'));}return parseFloat(a);};Zkb.P3=function P3(a,b,c){var d,e,f,g,h;if(a==null){throw c1(new Zkb.S4(Slb));}f=alb.Pgb(a).length;g=f>0&&(alb.Kgb(0,alb.Pgb(a).length),alb.Pgb(a).charCodeAt(0)==45||(alb.Kgb(0,alb.Pgb(a).length),alb.Pgb(a).charCodeAt(0)==43))?1:0;for(d=g;d<f;d++){if(_2((alb.Kgb(d,alb.Pgb(a).length),alb.Pgb(a).charCodeAt(d)))==-1){throw c1(new Zkb.S4(Rlb+a+'"'));}}h=parseInt(a,10);e=h<b;if(isNaN(h)){throw c1(new Zkb.S4(Rlb+a+'"'));}else if(e||h>c){throw c1(new Zkb.S4(Rlb+a+'"'));}return h;};Zkb.Q3=function Q3(a){return oX(a)?Zkb.U3(a):a.ac();};Zkb.R3=function R3(a){return oX(a)?Zkb.Y3(a):a.bc();};U1(95,1,{4:1,95:1});Zkb.S3=function S3(a,b){return $3(alb.Rgb((alb.Egb(a),a)),alb.Rgb((alb.Egb(b),b)));};Zkb.T3=function T3(a,b){return Zkb.S3(a,b);};Zkb.U3=function U3(a){return alb.Rgb((alb.Egb(a),a));};Zkb.V3=function V3(a,b){return alb.Egb(a),uX(a)===uX(b);};Zkb.W3=function W3(){return Ykb.U$;};Zkb.X3=function X3(a){return vX(alb.Rgb((alb.Egb(a),a)));};Zkb.Y3=function Y3(a){return vX(alb.Rgb((alb.Egb(a),a)));};function Z3(a){return Zkb.c5(Olb,typeof a);}function $3(a,b){if(a<b){return -1;}if(a>b){return 1;}if(a==b){return a==0?$3(1/a,1/b):0;}return isNaN(a)?isNaN(b)?0:1:-1;}function _3(a){if(isNaN(a)){return {l:0,m:0,h:524160};}return alb.Lgb(a);}iX={4:1,27:1,234:1,95:1};Zkb.Vy=function Vy(a){a.g=gW(Ykb.h_,Llb,66,0,0,1);};Zkb.Wy=function Wy(a){_kb.cV(a);};Zkb.Xy=function Xy(a){return _kb.dV(a);};Zkb.Yy=function Yy(a){if(a.j){a.backingJsObject!==Tlb&&a.Hb();a.g=null;}return a;};Zkb.Zy=function Zy(a){return a.e;};Zkb.$y=function $y(d,b){if(b instanceof Object){try{b.__java$exception=d;if(navigator.userAgent.toLowerCase().indexOf('msie')!=-1&&$doc.documentMode<9){return;}var c=d;Object.defineProperties(b,{cause:{get:function(){var a=c.Fb();return a&&a.Db();}},suppressed:{get:function(){return c.Eb();}}});}catch(a){}}};Zkb._y=function _y(a,b,c){var d,e,f,g,h;Zkb.az(a);for(e=(a.i==null&&(a.i=gW(Ykb.m_,Llb,22,0,0,1)),a.i),f=0,g=e.length;f<g;++f){d=e[f];Zkb._y(d,b,'\t'+c);}h=a.e;!!h&&Zkb._y(h,b,c);};Zkb.az=function az(a){var b,c,d;for(b=(a.g==null&&(a.g=Zkb.Xy(a)),a.g),c=0,d=b.length;c<d;++c){}};Zkb.bz=function bz(a,b){a.backingJsObject=b;Zkb.$y(a,b);};Zkb.cz=function cz(a){return Zkb.dz(a,a.Gb());};Zkb.dz=function dz(a,b){var c;c=Zkb.q3(a.sc);return b==null?c:c+': '+b;};Zkb.ez=function ez(){Zkb.Vy(this);Zkb.Yy(this);this.Hb();};Zkb.fz=function fz(a){Zkb.Vy(this);this.f=a;Zkb.Yy(this);this.Hb();};Zkb.hz=function hz(b){if(!('stack'in b)){try{throw b;}catch(a){}}return b;};function nz(a){var b;if(a!=null){b=a.__java$exception;if(b){return b;}}return qX(a,TypeError)?new Zkb.P4(a):new Zkb.bU(a);}U1(22,1,{4:1,22:1});_.Cb=function gz(a){return new Error(a);};_.Db=function iz(){return this.backingJsObject;};_.Eb=function jz(){var a,b,c;c=(this.i==null&&(this.i=gW(Ykb.m_,Llb,22,0,0,1)),this.i);b=gW(Ykb.f_,Ulb,1,c.length,5,1);for(a=0;a<c.length;a++){b[a]=c[a].backingJsObject;}return b;};_.Fb=function kz(){return Zkb.Zy(this);};_.Gb=function lz(){return this.f;};_.Hb=function mz(){Zkb.bz(this,Zkb.hz(this.Cb(Zkb.dz(this,this.f))));Zkb.Wy(this);};_.lb=function oz(){return Zkb.cz(this);};_.backingJsObject=Tlb;_.j=true;Zkb.pz=function pz(){Zkb.ez.call(this);};Zkb.qz=function qz(a){Zkb.Vy(this);Zkb.Yy(this);this.backingJsObject=a;Zkb.$y(this,a);this.f=a==null?Slb:_1(a);};Zkb.rz=function rz(a){Zkb.fz.call(this,a);};U1(15,22,Vlb,Zkb.rz);Zkb.sz=function sz(){Zkb.pz.call(this);};Zkb.tz=function tz(a){Zkb.qz.call(this,a);};Zkb.uz=function uz(a){Zkb.rz.call(this,a);};U1(41,15,Vlb);Zkb.G2=function G2(){Zkb.sz.call(this);};Zkb.H2=function H2(a){Zkb.uz.call(this,a);};U1(69,41,Vlb,Zkb.G2,Zkb.H2);Zkb.aU=function aU(){Zkb.sz.call(this);};Zkb.bU=function bU(a){Zkb.tz.call(this,a);};Zkb.cU=function cU(a){Zkb.uz.call(this,a);};U1(100,41,Vlb,Zkb.bU);Zkb.O4=function O4(){Zkb.aU.call(this);};Zkb.P4=function P4(a){Zkb.bU.call(this,a);};Zkb.Q4=function Q4(a){Zkb.cU.call(this,a);};U1(62,100,Vlb,Zkb.O4,Zkb.P4,Zkb.Q4);_.Cb=function R4(a){return new TypeError(a);};Zkb.X4=function X4(a,b){alb.Kgb(b,alb.Pgb(a).length);return alb.Pgb(a).charCodeAt(b);};Zkb.Y4=function Y4(a,b){return Zkb.Z4(a,b);};Zkb.Z4=function Z4(a,b){var c,d;c=Zkb.U3(alb.Pgb(a));d=Zkb.U3(alb.Pgb(b));return c==d?0:c<d?-1:1;};Zkb.$4=function $4(a,b){return Zkb.Z4(alb.Pgb(a).toLowerCase(),alb.Pgb(b).toLowerCase());};Zkb._4=function _4(a,b){return alb.Egb(a),a+(''+(alb.Egb(b),b));};Zkb.a5=function a5(a){var b;return Zkb.u5(alb.ogb(a,0,(b=a.length,alb.lgb(),b)));};Zkb.b5=function b5(a){var b;b=alb.Pgb(Wlb).length;return Zkb.c5(alb.Pgb(a).substr(alb.Pgb(a).length-b,b),Wlb);};Zkb.c5=function c5(a,b){return alb.Egb(a),uX(a)===uX(b);};Zkb.d5=function d5(a,b){alb.Egb(a);if(b==null){return false;}if(Zkb.c5(a,b)){return true;}return alb.Pgb(a).length==alb.Pgb(b).length&&Zkb.c5(alb.Pgb(a).toLowerCase(),alb.Pgb(b).toLowerCase());};Zkb.e5=function e5(){return Ykb.l_;};Zkb.f5=function f5(a){var b,c;b=0;for(c=0;c<alb.Pgb(a).length;c++){b=(b<<5)-b+(alb.Kgb(c,alb.Pgb(a).length),alb.Pgb(a).charCodeAt(c))|0;}return b;};Zkb.g5=function g5(a,b){return alb.Pgb(a).indexOf(b);};Zkb.h5=function h5(a,b,c){return alb.Pgb(a).indexOf(b,c);};function i5(a){return Zkb.c5(Plb,typeof a);}Zkb.j5=function j5(a,b){return alb.Pgb(a).lastIndexOf(b);};Zkb.k5=function k5(a,b,c){return alb.Pgb(a).lastIndexOf(b,c);};Zkb.l5=function l5(a,b){var c;c=Zkb.m5(b,'([/\\\\\\.\\*\\+\\?\\|\\(\\)\\[\\]\\{\\}$^])','\\\\$1');return Zkb.m5(a,c,'');};Zkb.m5=function m5(a,b,c){c=Zkb.t5(c);return alb.Pgb(a).replace(new RegExp(b,'g'),c);};Zkb.n5=function n5(a,b){var c,d,e,f,g,h,i,j;c=new RegExp(b,'g');i=gW(Ykb.l_,Xlb,2,0,6,1);d=0;j=a;f=null;while(true){h=c.exec(j);if(h==null||j==''){i[d]=j;break;}else {g=h.index;i[d]=(alb.Jgb(0,g,alb.Pgb(j).length),alb.Pgb(j).substr(0,g));j=Zkb.p5(j,g+alb.Pgb(alb.Pgb(h)[0]).length,alb.Pgb(j).length);c.lastIndex=0;if(f==j){i[d]=(alb.Jgb(0,1,alb.Pgb(j).length),alb.Pgb(j).substr(0,1));j=(alb.Kgb(1,alb.Pgb(j).length+1),alb.Pgb(j).substr(1));}f=j;++d;}}if(alb.Pgb(a).length>0){e=i.length;while(e>0&&i[e-1]==''){--e;}e<i.length&&(alb.Pgb(i).length=e);}return i;};Zkb.o5=function o5(a,b){alb.Kgb(b,alb.Pgb(a).length+1);return alb.Pgb(a).substr(b);};Zkb.p5=function p5(a,b,c){alb.Jgb(b,c,alb.Pgb(a).length);return alb.Pgb(a).substr(b,c-b);};Zkb.q5=function q5(a){var b,c,d;c=alb.Pgb(a).length;d=0;while(d<c&&(alb.Kgb(d,alb.Pgb(a).length),alb.Pgb(a).charCodeAt(d)<=32)){++d;}b=c;while(b>d&&(alb.Kgb(b-1,alb.Pgb(a).length),alb.Pgb(a).charCodeAt(b-1)<=32)){--b;}return d>0||b<c?(alb.Jgb(d,b,alb.Pgb(a).length),alb.Pgb(a).substr(d,b-d)):a;};Zkb.r5=function r5(a){return String.fromCharCode.apply(null,a);};function s5(a){var b,c;if(a>=Ylb){b=55296+(a-Ylb>>10&1023)&Zlb;c=56320+(a-Ylb&1023)&Zlb;return String.fromCharCode(b)+(''+String.fromCharCode(c));}else {return String.fromCharCode(a&Zlb);}}Zkb.t5=function t5(a){var b;b=0;while(0<=(b=alb.Pgb(a).indexOf('\\',b))){alb.Kgb(b+1,alb.Pgb(a).length);alb.Pgb(a).charCodeAt(b+1)==36?a=(alb.Jgb(0,b,alb.Pgb(a).length),alb.Pgb(a).substr(0,b)+'$'+Zkb.o5(a,++b)):a=(alb.Jgb(0,b,alb.Pgb(a).length),alb.Pgb(a).substr(0,b)+(''+Zkb.o5(a,++b)));}return a;};Zkb.u5=function u5(a){return Zkb.v5(a,0,a.length);};Zkb.v5=function v5(a,b,c){var d,e,f,g;f=b+c;alb.Jgb(b,f,a.length);g='';for(e=b;e<f;){d=$wnd.Math.min(e+$lb,f);g+=Zkb.r5(alb.Pgb(a).slice(e,d));e=d;}return g;};jX={4:1,159:1,27:1,2:1};Zkb.M5=function M5(a){Zkb.H2.call(this,a);};U1(178,69,Vlb,Zkb.M5);alb.agb=function agb(a){var b;b=alb.Pgb(a).slice();return alb.kgb(b,a);};alb.bgb=function bgb(a,b,c,d,e,f){var g,h,i,j,k;if(e==0){return;}if(uX(a)===uX(c)){a=alb.Pgb(a).slice(b,b+e);b=0;}i=alb.Pgb(c);for(h=b,j=b+e;h<j;){g=$wnd.Math.min(h+$lb,j);e=g-h;k=alb.Pgb(a).slice(h,g);alb.Pgb(k).splice(0,0,alb.Pgb(d),alb.Pgb(f?e:0));Array.prototype.splice.apply(i,k);h=g;d+=e;}};alb.cgb=function cgb(a,b){return alb.kgb(new Array(b),a);};alb.dgb=function dgb(a,b,c){alb.Pgb(a).splice(b,0,c);};alb.egb=function egb(a,b,c){alb.bgb(c,0,a,b,c.length,false);};alb.fgb=function fgb(a,b){alb.Pgb(a).push(b);};alb.ggb=function ggb(a,b){alb.Pgb(a).push(b);};alb.hgb=function hgb(a,b,c){alb.Pgb(a).splice(b,c);};alb.igb=function igb(a,b){alb.Pgb(a).length=b;};alb.jgb=function jgb(a,b){alb.Pgb(a).sort(b);};U1(374,1,{});alb.tgb=function tgb(){};alb.ugb=function ugb(a){switch(typeof a){case Plb:return Zkb.f5(alb.Pgb(a));case Olb:return Zkb.X3(alb.Pgb(a));case Nlb:return Zkb.R2(alb.Pgb(a));default:return a==null?0:alb.wgb(a);}};alb.vgb=function vgb(){return ++alb.sgb;};alb.wgb=function wgb(a){return a.$H||(a.$H=alb.vgb());};U1(328,1,{},alb.tgb);alb.sgb=0;alb.xgb=function xgb(a){if(!a){throw c1(new Zkb.b4());}};alb.ygb=function ygb(a,b){if(!a){throw c1(new Zkb.c4(b));}};alb.zgb=function zgb(a,b){if(0>a){throw c1(new Zkb.c4('fromIndex: 0 > toIndex: '+a));}if(a>b){throw c1(new Zkb.J2('fromIndex: 0, toIndex: '+a+_lb+b));}};alb.Agb=function Agb(a){if(a<0){throw c1(new Zkb.N4('Negative array size: '+a));}};alb.Bgb=function Bgb(a,b){if(a!=b){throw c1(new dlb.Icb());}};alb.Cgb=function Cgb(a){if(!a){throw c1(new dlb.Ndb());}};alb.Dgb=function Dgb(a,b){if(a<0||a>=b){throw c1(new Zkb.H2(amb+a+bmb+b));}};alb.Egb=function Egb(a){if(a==null){throw c1(new Zkb.O4());}return a;};alb.Fgb=function Fgb(a,b){if(a==null){throw c1(new Zkb.Q4(b));}};alb.Ggb=function Ggb(a,b){if(a<0||a>b){throw c1(new Zkb.H2(amb+a+bmb+b));}};alb.Hgb=function Hgb(a,b,c){if(a<0||b>c){throw c1(new Zkb.H2(cmb+a+dmb+b+', size: '+c));}if(a>b){throw c1(new Zkb.c4(cmb+a+' > toIndex: '+b));}};alb.Igb=function Igb(a){if(!a){throw c1(new Zkb.d4());}};alb.Jgb=function Jgb(a,b,c){if(a<0||b>c||b<a){throw c1(new Zkb.M5(cmb+a+dmb+b+_lb+c));}};alb.Kgb=function Kgb(a,b){if(a<0||a>=b){throw c1(new Zkb.M5(amb+a+bmb+b));}};alb.Lgb=function Lgb(a){var b,c;b=new ArrayBuffer(8);alb.Pgb(new Float64Array(b))[0]=a;c=alb.Pgb(new Uint32Array(b));return alb.Sgb(c[0]|0,c[1]|0);};alb.Mgb=function Mgb(a,b){return a[b];};alb.Ngb=function Ngb(a){return a===undefined;};alb.Ogb=function Ogb(a){return a>>>0;};alb.Pgb=function Pgb(a){return a;};alb.Qgb=function Qgb(a){return a;};alb.Rgb=function Rgb(a){return a;};U1(329,1,{});Ykb.f_=w3(1);Ykb.r$=w3(0);Ykb.y$=w3(330);Ykb.v$=w3(236);Ykb.x$=w3(331);Ykb.w$=w3(237);Ykb.R$=w3(233);Ykb.T$=w3(157);Ykb.e_=w3(95);Ykb.U$=w3(234);Ykb.m_=w3(22);Ykb.W$=w3(15);Ykb.g_=w3(41);Ykb.Z$=w3(69);Ykb._$=w3(100);Ykb.c_=w3(62);Ykb.l_=w3(2);Ykb.k_=w3(178);Ykb.t0=w3(328);Ykb.u0=w3(329);blb.Nc=function Nc(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N,O,P,Q;this.c=a.length;this.d=a[0].length;G=$wnd.Math.min(this.c,this.d);this.e=gW(Ykb.yX,emb,6,$wnd.Math.min(this.c+1,this.d),15,1);this.a=eW(Ykb.yX,[Llb,emb],[12,6],15,[this.c,G],2);this.b=eW(Ykb.yX,[Llb,emb],[12,6],15,[this.d,this.d],2);e=gW(Ykb.yX,emb,6,this.d,15,1);Q=gW(Ykb.yX,emb,6,this.c,15,1);D=$wnd.Math.min(this.c-1,this.d);F=$wnd.Math.max(0,$wnd.Math.min(this.d-2,this.c));for(v=0;v<$wnd.Math.max(D,F);v++){if(v<D){this.e[v]=0;for(l=v;l<this.c;l++){this.e[v]=blb.Oc(this.e[v],a[l][v]);}if(this.e[v]!=0){a[v][v]<0&&(this.e[v]=-this.e[v]);for(k=v;k<this.c;k++){a[k][v]/=this.e[v];}a[v][v]+=1;}this.e[v]=-this.e[v];}for(q=v+1;q<this.d;q++){if(v<D&this.e[v]!=0){P=0;for(l=v;l<this.c;l++){P+=a[l][v]*a[l][q];}P=-P/a[v][v];for(k=v;k<this.c;k++){a[k][q]+=P*a[k][v];}}e[q]=a[v][q];}if(true&v<D){for(k=v;k<this.c;k++){this.a[k][v]=a[k][v];}}if(v<F){e[v]=0;for(l=v+1;l<this.d;l++){e[v]=blb.Oc(e[v],e[l]);}if(e[v]!=0){e[v+1]<0&&(e[v]=-e[v]);for(m=v+1;m<this.d;m++){e[m]/=e[v];}e[v+1]+=1;}e[v]=-e[v];if(v+1<this.c&e[v]!=0){for(m=v+1;m<this.c;m++){Q[m]=0;}for(r=v+1;r<this.d;r++){for(n=v+1;n<this.c;n++){Q[n]+=e[r]*a[n][r];}}for(s=v+1;s<this.d;s++){P=-e[s]/e[v+1];for(n=v+1;n<this.c;n++){a[n][s]+=P*Q[n];}}}for(k=v+1;k<this.d;k++){this.b[k][v]=e[k];}}}H=$wnd.Math.min(this.d,this.c+1);D<this.d&&(this.e[D]=a[D][D]);this.c<H&&(this.e[H-1]=0);F+1<H&&(e[F]=a[F][H-1]);e[H-1]=0;for(t=D;t<G;t++){for(k=0;k<this.c;k++){this.a[k][t]=0;}this.a[t][t]=1;}for(w=D-1;w>=0;w--){if(this.e[w]!=0){for(p=w+1;p<G;p++){P=0;for(l=w;l<this.c;l++){P+=this.a[l][w]*this.a[l][p];}P=-P/this.a[w][w];for(m=w;m<this.c;m++){this.a[m][p]+=P*this.a[m][w];}}for(n=w;n<this.c;n++){this.a[n][w]=-this.a[n][w];}this.a[w][w]=1+this.a[w][w];for(k=0;k<w-1;k++){this.a[k][w]=0;}}else {for(k=0;k<this.c;k++){this.a[k][w]=0;}this.a[w][w]=1;}}for(A=this.d-1;A>=0;A--){if(A<F&e[A]!=0){for(p=A+1;p<G;p++){P=0;for(l=A+1;l<this.d;l++){P+=this.b[l][A]*this.b[l][p];}P=-P/this.b[A+1][A];for(m=A+1;m<this.d;m++){this.b[m][p]+=P*this.b[m][A];}}}for(k=0;k<this.d;k++){this.b[k][A]=0;}this.b[A][A]=1;}I=H-1;h=$wnd.Math.pow(2,-52);while(H>0){for(u=H-2;u>=-1;u--){if(u==-1){break;}if($wnd.Math.abs(e[u])<=h*($wnd.Math.abs(this.e[u])+$wnd.Math.abs(this.e[u+1]))){e[u]=0;break;}}if(u==H-2){B=4;}else {for(C=H-1;C>=u;C--){if(C==u){break;}P=(C!=H?$wnd.Math.abs(e[C]):0)+(C!=u+1?$wnd.Math.abs(e[C-1]):0);if($wnd.Math.abs(this.e[C])<=h*P){this.e[C]=0;break;}}if(C==u){B=3;}else if(C==H-1){B=1;}else {B=2;u=C;}}++u;switch(B){case 1:{i=e[H-2];e[H-2]=0;for(p=H-2;p>=u;p--){P=blb.Oc(this.e[p],i);d=this.e[p]/P;M=i/P;this.e[p]=P;if(p!=u){i=-M*e[p-1];e[p-1]=d*e[p-1];}for(k=0;k<this.d;k++){P=d*this.b[k][p]+M*this.b[k][H-1];this.b[k][H-1]=-M*this.b[k][p]+d*this.b[k][H-1];this.b[k][p]=P;}}}break;case 2:{i=e[u-1];e[u-1]=0;for(p=u;p<H;p++){P=blb.Oc(this.e[p],i);d=this.e[p]/P;M=i/P;this.e[p]=P;i=-M*e[p];e[p]=d*e[p];for(k=0;k<this.c;k++){P=d*this.a[k][p]+M*this.a[k][u-1];this.a[k][u-1]=-M*this.a[k][p]+d*this.a[k][u-1];this.a[k][p]=P;}}}break;case 3:{J=$wnd.Math.max($wnd.Math.max($wnd.Math.max($wnd.Math.max($wnd.Math.abs(this.e[H-1]),$wnd.Math.abs(this.e[H-2])),$wnd.Math.abs(e[H-2])),$wnd.Math.abs(this.e[u])),$wnd.Math.abs(e[u]));N=this.e[H-1]/J;O=this.e[H-2]/J;g=e[H-2]/J;L=this.e[u]/J;f=e[u]/J;b=((O+N)*(O-N)+g*g)/2;c=N*g*(N*g);K=0;if(b!=0|c!=0){K=$wnd.Math.sqrt(b*b+c);b<0&&(K=-K);K=c/(b+K);}i=(L+N)*(L-N)+K;j=L*f;for(p=u;p<H-1;p++){P=blb.Oc(i,j);d=i/P;M=j/P;p!=u&&(e[p-1]=P);i=d*this.e[p]+M*e[p];e[p]=d*e[p]-M*this.e[p];j=M*this.e[p+1];this.e[p+1]=d*this.e[p+1];for(l=0;l<this.d;l++){P=d*this.b[l][p]+M*this.b[l][p+1];this.b[l][p+1]=-M*this.b[l][p]+d*this.b[l][p+1];this.b[l][p]=P;}P=blb.Oc(i,j);d=i/P;M=j/P;this.e[p]=P;i=d*e[p]+M*this.e[p+1];this.e[p+1]=-M*e[p]+d*this.e[p+1];j=M*e[p+1];e[p+1]=d*e[p+1];if(p<this.c-1){for(k=0;k<this.c;k++){P=d*this.a[k][p]+M*this.a[k][p+1];this.a[k][p+1]=-M*this.a[k][p]+d*this.a[k][p+1];this.a[k][p]=P;}}}e[H-2]=i;}break;case 4:{if(this.e[u]<=0){this.e[u]=this.e[u]<0?-this.e[u]:0;for(k=0;k<=I;k++){this.b[k][u]=-this.b[k][u];}}while(u<I){if(this.e[u]>=this.e[u+1]){break;}P=this.e[u];this.e[u]=this.e[u+1];this.e[u+1]=P;if(u<this.d-1){for(k=0;k<this.d;k++){P=this.b[k][u+1];this.b[k][u+1]=this.b[k][u];this.b[k][u]=P;}}if(u<this.c-1){for(k=0;k<this.c;k++){P=this.a[k][u+1];this.a[k][u+1]=this.a[k][u];this.a[k][u]=P;}}++u;}--H;}}}};blb.Oc=function Oc(a,b){var c;if($wnd.Math.abs(a)>$wnd.Math.abs(b)){c=b/a;c=$wnd.Math.abs(a)*$wnd.Math.sqrt(1+c*c);}else if(b!=0){c=a/b;c=$wnd.Math.abs(b)*$wnd.Math.sqrt(1+c*c);}else {c=0;}return c;};U1(155,1,fmb,blb.Nc);_.c=0;_.d=0;Ykb.CX=w3(155);clb.Qc=function Qc(){clb.Qc=W1;clb.Pc=oW(aW(Ykb.AX,1),gmb,6,15,[0,hmb,14286847,13402367,12779264,16758197,9474192,3166456,16715021,9494608,11789301,11230450,9109248,12560038,15780000,16744448,16777008,2093087,8442339,9388244,4062976,15132390,12567239,10921643,9083335,10255047,14706227,15765664,5296208,13140019,8224944,12750735,6721423,12419299,16752896,10889513,6076625,7351984,imb,9764863,9756896,7586505,5551541,3907230,2396047,687500,27013,12632256,16767375,10909043,6717568,10380213,13924864,9699476,4366000,5707663,51456,7394559,16777159,14286791,13107143,10747847,9437127,6422471,4587463,3211207,2097095,65436,58997,54354,48952,43812,5096191,5089023,2200790,2522539,2516630,1528967,13684960,16765219,12105936,10900557,5724513,10375093,11230208,7688005,4358806,4325478,32000,7384058,47871,41471,36863,33023,27647,5528818,7888099,9064419,10565332,11739092,11739066,11734438,12389767,13041766,13369433,13697103,14221381,14680120,15073326,15400998,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,13158600,1334015,56540,15075850,15132160,56540,15075850,15461355,8553170,1016335,1016335,1334015,15132160,3289770,14456450,16422400,16422400,11819700,3289770,1016335]);};clb.Rc=function Rc(a){a.u=new plb.XF();};clb.Sc=function Sc(a){var b,c;if((a.D&32)!=0)return;c=clb.Jt(a.J);if(c!=null){if(a.v.a==0&&a.v.b==0){b=a.P.c*clb.vk(a.J);a.u=clb.Md(a);clb.bd(a,b);clb.Jd(a,null,b,0);}clb.Os(a,vX(a.w));a.J.O!=1&&clb.Kd(a,448);clb.Is(a,c,a.v.a,a.v.b+jmb*a.w);}};clb.Tc=function Tc(a,b){return a==null?b:b==null?a:a+','+b;};clb.Uc=function Uc(a){var b;b=a.P.c*clb.vk(a.J);a.W=b*0.06;a.R=b*0.15;a.Q=b*0.38;a.U=b*0.47;a.V=vX(b*a.H*0.6+0.5);a.T=b*0.12;a.X=b*0.4;a.w=b*0.5+0.5;};clb.Vc=function Vc(a){var b,c;a=e1(a,kmb);for(c=0;c<(olb.NF(),olb.MF).length;c++)if(o1(a,olb.MF[c]))return olb.LF[c];b=new Zkb.L5('R');z1(e1(a,lmb),0)&&(b.a+='0',b);z1(e1(a,mmb),0)&&(b.a+='3',b);z1(e1(a,nmb),0)&&(b.a+='4',b);z1(e1(a,omb),0)&&(b.a+='5',b);z1(e1(a,pmb),0)&&(b.a+='6',b);z1(e1(a,qmb),0)&&(b.a+='7',b);z1(e1(a,rmb),0)&&(b.a+='8',b);return b.a;};clb.Wc=function Wc(a,b,c,d){var e,f,g;e=new clb.Sd();f=new clb.Sd();e.a=b.a;e.c=b.c;e.b=(b.a+b.b)/2;e.d=(b.c+b.d)/2;f.a=e.b;f.c=e.d;f.b=b.b;f.d=b.d;if(clb.Gd(a,e)){clb.Kd(a,a.p[c]);g=smb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.a),(Blb.f8(),Blb.b8))))+tmb+umb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.c),Blb.b8)))+tmb+vmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.b),Blb.b8)))+tmb+wmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.d),Blb.b8)))+tmb+xmb+a.d+tmb+ymb+a.j+zmb;clb.Rs(a,g);}if(clb.Gd(a,f)){clb.Kd(a,a.p[d]);g=smb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(f.a),(Blb.f8(),Blb.b8))))+tmb+umb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(f.c),Blb.b8)))+tmb+vmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(f.b),Blb.b8)))+tmb+wmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(f.d),Blb.b8)))+tmb+xmb+a.d+tmb+ymb+a.j+zmb;clb.Rs(a,g);}clb.Kd(a,a.O);};clb.Xc=function Xc(a,b,c,d){var e,f,g,h,i,j,k,l,m;l=(b.b-b.a)/10;m=(b.d-b.c)/10;e=new clb.Sd();if(clb.cl(a.J,clb.tn(a.J,c,d))){f=-3;g=-3;}else {f=a.p[c];g=a.p[d];}clb.Kd(a,f);e.a=b.a;e.c=b.c;e.b=b.a+l*2;e.d=b.c+m*2;i=smb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.a),(Blb.f8(),Blb.b8))))+tmb+umb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.c),Blb.b8)))+tmb+vmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.b),Blb.b8)))+tmb+wmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.d),Blb.b8)))+tmb+xmb+a.d+tmb+ymb+a.j+zmb;clb.Rs(a,i);e.a=b.a+l*4;e.c=b.c+m*4;e.b=b.a+l*5;e.d=b.c+m*5;j=smb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.a),Blb.b8)))+tmb+umb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.c),Blb.b8)))+tmb+vmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.b),Blb.b8)))+tmb+wmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.d),Blb.b8)))+tmb+xmb+a.d+tmb+ymb+a.j+zmb;clb.Rs(a,j);clb.Kd(a,g);e.a=b.a+l*5;e.c=b.c+m*5;e.b=b.a+l*6;e.d=b.c+m*6;k=smb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.a),Blb.b8)))+tmb+umb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.c),Blb.b8)))+tmb+vmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.b),Blb.b8)))+tmb+wmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.d),Blb.b8)))+tmb+xmb+a.d+tmb+ymb+a.j+zmb;clb.Rs(a,k);e.a=b.a+l*8;e.c=b.c+m*8;e.b=b.b;e.d=b.d;h=smb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.a),Blb.b8)))+tmb+umb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.c),Blb.b8)))+tmb+vmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.b),Blb.b8)))+tmb+wmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.d),Blb.b8)))+tmb+xmb+a.d+tmb+ymb+a.j+zmb;clb.Rs(a,h);clb.Kd(a,a.O);};clb.Yc=function Yc(a,b,c){clb.Js(a,b-a.T/2,c-a.T/2,a.T);};clb.Zc=function Zc(a,b,c,d){var e;if(clb.cl(a.J,clb.tn(a.J,c,d))){clb.Kd(a,-3);e=smb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(b.a),(Blb.f8(),Blb.b8))))+tmb+umb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(b.c),Blb.b8)))+tmb+vmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(b.b),Blb.b8)))+tmb+wmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(b.d),Blb.b8)))+tmb+xmb+a.d+tmb+ymb+a.j+zmb;clb.Rs(a,e);clb.Kd(a,a.O);}else if(a.p[c]!=a.p[d]){clb.Wc(a,b,c,d);}else if(a.p[c]!=0){clb.Kd(a,a.p[c]);e=smb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(b.a),(Blb.f8(),Blb.b8))))+tmb+umb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(b.c),Blb.b8)))+tmb+vmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(b.b),Blb.b8)))+tmb+wmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(b.d),Blb.b8)))+tmb+xmb+a.d+tmb+ymb+a.j+zmb;clb.Rs(a,e);clb.Kd(a,a.O);}else {e=smb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(b.a),(Blb.f8(),Blb.b8))))+tmb+umb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(b.c),Blb.b8)))+tmb+vmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(b.b),Blb.b8)))+tmb+wmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(b.d),Blb.b8)))+tmb+xmb+a.d+tmb+ymb+a.j+zmb;clb.Rs(a,e);}};clb.$c=function $c(a,b,c,d,e,f,g){g.a=b.a+e;g.c=b.c+f;g.b=b.b+e;g.d=b.d+f;clb.Zc(a,g,c,d);};clb._c=function _c(a,b,c,d){var e,f,g,h,i,j,k,l,m,n,o,p;l=b.b-b.a;o=b.d-b.c;i=$wnd.Math.sqrt(l*l+o*o);j=2*I1(p1($wnd.Math.round(i/(4*a.W))));m=l/(j-1);p=o/(j-1);if(clb.cl(a.J,clb.tn(a.J,c,d))){e=-3;f=-3;}else {e=a.p[c];f=a.p[d];}k=b.a-a.W/2;n=b.c-a.W/2;clb.Kd(a,e);for(h=0;h<(j/2|0);h++){clb.Js(a,k,n,a.W);k+=m;n+=p;}clb.Kd(a,f);for(g=0;g<(j/2|0);g++){clb.Js(a,k,n,a.W);k+=m;n+=p;}clb.Kd(a,a.O);};clb.ad=function ad(a,b,c,d){var e,f,g,h,i,j,k,l,m,n,o,p,q,r;i=(b.c-b.d)/9;n=(b.b-b.a)/9;j=b.b+i;o=b.d+n;k=b.b-i;p=b.d-n;l=(b.a+j)/2;q=(b.c+o)/2;m=(b.a+k)/2;r=(b.c+p)/2;g=new plb.SF(3);h=new plb.SF(4);plb.RF(g,b.a,b.c);plb.RF(g,l,q);plb.RF(g,m,r);plb.RF(h,m,r);plb.RF(h,l,q);plb.RF(h,j,o);plb.RF(h,k,p);if(clb.cl(a.J,clb.tn(a.J,c,d))){e=-3;f=-3;}else {e=a.p[c];f=a.p[d];if(a.J.O!=1){f=clb.ed(a,c);e==clb.ek(a.J,c)&&(e=f);}}clb.Kd(a,e);clb.Hs(a,g);clb.Kd(a,f);clb.Hs(a,h);clb.Kd(a,a.O);};clb.bd=function bd(a,b){var c,d;for(d=0;d<a.Y.a.length;d++)a.u=plb.WF(a.u,dlb.Bi(a.Y,d));clb.cd(a,b);c=0.1*b;a.u.c-=c;a.u.d-=c;a.u.b+=2*c;a.u.a+=2*c;};clb.cd=function cd(a,b){var c,d,e,f,g,h,i;e=gW(Ykb.Z0,Amb,6,a.J.q,16,1);for(d=0;d<a.J.r;d++){if(clb.al(a.J,d)){e[clb.zk(a.J,0,d)]=true;e[clb.zk(a.J,1,d)]=true;}}g=new plb.XF();for(c=0;c<a.J.q;c++){f=z1(e1(clb.pk(a.J,c),Bmb),0)?b*0.47:e[c]?b*0.38:0;if(f!=0){h=clb.Mh(a.P,clb.rk(a.J,c));i=clb.Nh(a.P,clb.sk(a.J,c));plb.VF(g,h-f,i-f,f*2,f*2);a.u=plb.WF(a.u,g);}}};clb.dd=function dd(a,b){var c;c=a.K!=0?a.K:a.B!=0?a.B:-1;return slb.FT(b,c);};clb.ed=function ed(a,b){var c,d;if((a.D&4224)!=0)return a.p[b];d=clb.fd(a,b);if(d==-1){c=clb.an(a.J,b);if(c!=-1){b=c;d=clb.fd(a,b);}}if(d==-1)return a.p[b];switch(d&255){case 1:return 384;case 2:return 64;default:return 448;}};clb.fd=function fd(a,b){var c,d,e;e=-1;d=-1;if((a.D&128)!=0)return e;if(clb.$k(a.J,b)){e=clb.ik(a.J,b);d=clb.hk(a.J,b);}c=clb.cn(a.J,b);if(c!=-1){e=clb.Ek(a.J,c);d=clb.Dk(a.J,c);}e!=-1&&e!=0&&(e|=d<<8);return e;};clb.gd=function gd(a){var b,c,d,e,f;clb.Ms(a,2*a.Q);e=new clb.Sd();for(d=0;d<a.J.r;d++){b=clb.zk(a.J,0,d);c=clb.zk(a.J,1,d);if(clb.al(a.J,d)){e.a=clb.Mh(a.P,clb.rk(a.J,b));e.c=clb.Nh(a.P,clb.sk(a.J,b));e.b=clb.Mh(a.P,clb.rk(a.J,c));e.d=clb.Nh(a.P,clb.sk(a.J,c));clb.Kd(a,-2);f=smb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.a),(Blb.f8(),Blb.b8))))+tmb+umb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.c),Blb.b8)))+tmb+vmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.b),Blb.b8)))+tmb+wmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(e.d),Blb.b8)))+tmb+xmb+a.d+tmb+ymb+a.j+zmb;clb.Rs(a,f);}}};clb.hd=function hd(a){var b,c,d,e,f,g,h;if(a.J.K){g=a.U;clb.Kd(a,-7);for(b=0;b<a.J.f;b++)z1(e1(clb.pk(a.J,b),Bmb),0)&&clb.Js(a,clb.Mh(a.P,clb.rk(a.J,b))-g,clb.Nh(a.P,clb.sk(a.J,b))-g,2*g);clb.Ms(a,2*a.U);f=new clb.Sd();for(e=0;e<a.J.r;e++){c=clb.zk(a.J,0,e);d=clb.zk(a.J,1,e);if(z1(e1(e1(clb.pk(a.J,c),clb.pk(a.J,d)),Bmb),0)){f.a=clb.Mh(a.P,clb.rk(a.J,c));f.c=clb.Nh(a.P,clb.sk(a.J,c));f.b=clb.Mh(a.P,clb.rk(a.J,d));f.d=clb.Nh(a.P,clb.sk(a.J,d));h=smb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(f.a),(Blb.f8(),Blb.b8))))+tmb+umb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(f.c),Blb.b8)))+tmb+vmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(f.b),Blb.b8)))+tmb+wmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(f.d),Blb.b8)))+tmb+xmb+a.d+tmb+ymb+a.j+zmb;clb.Rs(a,h);}}}};clb.jd=function jd(a){var b,c,d,e;if(a.J.K){clb.Kd(a,320);if((a.D&8)!=0)for(b=0;b<a.J.f;b++)z1(e1(clb.pk(a.J,b),-536870913),0)&&clb.Js(a,clb.Mh(a.P,clb.rk(a.J,b))-a.X/2,clb.Nh(a.P,clb.sk(a.J,b))-a.X/2,a.X);for(e=0;e<a.J.g;e++){if(clb.Jk(a.J,e)!=0){c=clb.zk(a.J,0,e);d=clb.zk(a.J,1,e);clb.Js(a,(clb.Mh(a.P,clb.rk(a.J,c))+clb.Mh(a.P,clb.rk(a.J,d))-a.X)/2,(clb.Nh(a.P,clb.sk(a.J,c))+clb.Nh(a.P,clb.sk(a.J,d))-a.X)/2,a.X);}}}};clb.kd=function kd(a){a.H=1;a.P=new clb.Oh();a.Y=new dlb.Qi();a.S=new dlb.Qi();a.r=gW(Ykb.Z0,Amb,6,a.J.q,16,1);a.v=new plb.OF();a.O=0;a.A=-1;clb.Od(a);};clb.ld=function ld(a,b){var c;if(clb.gn(a.J,b)==0)return false;for(c=0;c<clb.gn(a.J,b);c++)if(!clb.cl(a.J,clb.xn(a.J,b,c)))return false;return true;};clb.md=function md(a){var b;a.q=gW(Ykb.Z0,Amb,6,a.J.q,16,1);for(b=0;b<a.J.r;b++){a.q[clb.zk(a.J,0,b)]=true;a.q[clb.zk(a.J,1,b)]=true;}};clb.nd=function nd(a,b){var c;if(clb.wn(a.J,b)!=2)return false;for(c=0;c<2;c++)if(clb.yn(a.J,b,c)!=2)return false;return true;};clb.od=function od(a,b,c,d,e){var f,g,h,i,j,k,l,m,n,o;m=false;e.a=0;e.b=0;d>0?f=Cmb:f=Dmb;o=clb.yk(a.J,b,c);for(k=0;k<clb.wn(a.J,b);k++){g=clb.xn(a.J,b,k);h=o;clb.zk(a.J,0,g)==b?l=clb.zk(a.J,1,g):l=clb.zk(a.J,0,g);if(l==c)continue;n=clb.yk(a.J,b,l);h<n&&(h+=Emb);i=h-n;if(d>0){i<Fmb&&(m=true);i>Cmb&&(i=Cmb);i<0.523598776&&(i=0.523598776);if(i<=f){f=i;j=a.R*$wnd.Math.tan(f-Gmb)/2;e.a=-(j*$wnd.Math.sin(h));e.b=-(j*$wnd.Math.cos(h));}}else {i>=Fmb&&(m=true);i<Dmb&&(i=Dmb);i>5.759586531&&(i=5.759586531);if(i>=f){f=i;j=a.R*$wnd.Math.tan(4.712388981-f)/2;e.a=-(j*$wnd.Math.sin(h));e.b=-(j*$wnd.Math.cos(h));}}}return m;};clb.pd=function pd(a,b,c,d){var e;if(b==0){c<0?d.a=a.R:d.a=-a.R;d.b=0;return;}e=$wnd.Math.atan(c/b);b<0&&(e+=Fmb);d.a=-(a.R*$wnd.Math.sin(e));d.b=a.R*$wnd.Math.cos(e);};clb.qd=function qd(a,b,c,d){var e,f,g,h,i,j,k,l,m,n,o,p;e=new clb.Sd();i=new clb.Sd();k=new plb.OF();j=new plb.OF();g=clb.zk(a.J,0,c);h=clb.zk(a.J,1,c);if(d){m=b.a;b.a=b.b;b.b=m;m=b.c;b.c=b.d;b.d=m;n=g;g=h;h=n;}if(!clb.Gd(a,b))return;if(clb.zo(a.J,c)){e.a=b.a;e.c=b.c;e.b=b.b;e.d=b.d;l=d?-clb.Wn(a.J,c):clb.Wn(a.J,c);l==0&&(l=1);clb.pd(a,b.b-b.a,b.d-b.c,k);if(l>0){i.a=b.a+k.a;i.c=b.c+k.b;i.b=b.b+k.a;i.d=b.d+k.b;if(clb.od(a,g,h,1,j)||clb.wn(a.J,g)>1){i.a+=j.a+k.b;i.c+=j.b-k.a;}}else {i.a=b.a-k.a;i.c=b.c-k.b;i.b=b.b-k.a;i.d=b.d-k.b;if(clb.od(a,g,h,-1,j)||clb.wn(a.J,g)>1){i.a+=j.a+k.b;i.c+=j.b-k.a;}}clb.Kk(a.J,c)==386&&clb.Fd(e,i);clb.Gd(a,e)&&clb.Zc(a,e,g,h);clb.Kk(a.J,c)==64?clb.Gd(a,i)&&clb.Xc(a,i,g,h):clb.Gd(a,i)&&clb.Zc(a,i,g,h);}else {clb.pd(a,b.b-b.a,b.d-b.c,k);o=k.a/2;p=k.b/2;f=false;e.a=b.a+o;e.c=b.c+p;e.b=b.b+o;e.d=b.d+p;if(clb.wn(a.J,g)>1){if(clb.od(a,g,h,1,j)){e.a+=j.a;e.c+=j.b;if(clb.wn(a.J,g)==2){if(j.a!=0||j.b!=0){e.a+=k.b;e.c-=k.a;}}}else {a.o[g]=new plb.PF(e.a,e.c);}}i.a=b.a-o;i.c=b.c-p;i.b=b.b-o;i.d=b.d-p;if(clb.wn(a.J,g)>1){if(clb.od(a,g,h,0,j)){i.a+=j.a;i.c+=j.b;if(clb.wn(a.J,g)==2){if(j.a!=0||j.b!=0){i.a+=k.b;i.c-=k.a;}}}else {a.o[g]=new plb.PF(i.a,i.c);f=true;}}clb.Kk(a.J,c)==386&&clb.Fd(e,i);if(clb.Kk(a.J,c)==64){if(f){clb.Xc(a,e,g,h);clb.Zc(a,i,g,h);}else {clb.Zc(a,e,g,h);clb.Xc(a,i,g,h);}}else {clb.Zc(a,e,g,h);clb.Zc(a,i,g,h);}}};clb.rd=function rd(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o;k=a.O;l=a.C;if((a.D&Hmb)!=0){a.O=-6;a.C=-8355712;clb.Kd(a,1);}a.o=gW(Ykb.QZ,Ulb,50,a.J.q,0,1);for(i=0;i<a.J.r;i++)(clb.Kk(a.J,i)==2||clb.Kk(a.J,i)==386||clb.Kk(a.J,i)==64)&&clb.ud(a,i);for(j=0;j<a.J.r;j++)clb.Kk(a.J,j)!=2&&clb.Kk(a.J,j)!=386&&clb.Kk(a.J,j)!=64&&clb.ud(a,j);if((a.D&64)==0){for(h=0;h<a.J.r;h++){if(clb.Ck(a.J,h)!=0){e=null;clb.Ck(a.J,h)==1||clb.Ck(a.J,h)==2?(clb.Hk(a.J,h)==2||clb.Ek(a.J,h)==0||b[clb.Ek(a.J,h)][clb.Dk(a.J,h)]>1)&&(clb.Ck(a.J,h)==1?e=clb.Hk(a.J,h)==2?'E':clb.el(a.J,h)?'p':'P':e=clb.Hk(a.J,h)==2?'Z':clb.el(a.J,h)?'m':'M'):e='?';if(e!=null){clb.Os(a,(a.V*2+1)/3|0);clb.Kd(a,clb.cl(a.J,h)?-3:a.J.O==1||(a.D&Imb)!=0?a.O:448);c=clb.zk(a.J,0,h);d=clb.zk(a.J,1,h);n=(clb.Mh(a.P,clb.rk(a.J,c))+clb.Mh(a.P,clb.rk(a.J,d)))/2;o=(clb.Nh(a.P,clb.sk(a.J,c))+clb.Nh(a.P,clb.sk(a.J,d)))/2;f=(clb.Mh(a.P,clb.rk(a.J,c))-clb.Mh(a.P,clb.rk(a.J,d)))/3;g=(clb.Nh(a.P,clb.sk(a.J,c))-clb.Nh(a.P,clb.sk(a.J,d)))/3;clb.xd(a,n+g,o-f,e,true);clb.Kd(a,a.O);clb.Os(a,a.V);}}}}if((a.D&4)!=0){clb.Os(a,(a.V*2+1)/3|0);clb.Kd(a,384);for(h=0;h<a.J.r;h++){c=clb.zk(a.J,0,h);d=clb.zk(a.J,1,h);m=clb.qo(a.J,h)?'d':clb.mo(a.J,h)?'a':'';n=(clb.Mh(a.P,clb.rk(a.J,c))+clb.Mh(a.P,clb.rk(a.J,d)))/2;o=(clb.Nh(a.P,clb.sk(a.J,c))+clb.Nh(a.P,clb.sk(a.J,d)))/2;clb.xd(a,n,o,m+(''+h),true);}clb.Kd(a,a.O);clb.Os(a,a.V);}if((a.D&Hmb)!=0){a.O=k;a.C=l;}};clb.sd=function sd(a){var b,c;for(c=new dlb.Obb(a.S);c.a<c.c.a.length;){b=dlb.Nbb(c);clb.Kd(a,b.a);clb.Yc(a,b.b,b.c);}clb.Kd(a,a.O);};clb.td=function td(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N,O,P,Q,R,S,T,U,V,W,X,Y,Z,$;a.I||clb.Ks(a,b,(clb.jk(a.J,b),clb.Mh(a.P,clb.rk(a.J,b))),clb.Nh(a.P,clb.sk(a.J,b)));M=null;if(clb.dk(a.J,b)!=0){U=$wnd.Math.abs(clb.dk(a.J,b))==1?'':''+$wnd.Math.abs(clb.dk(a.J,b));M=clb.dk(a.J,b)<0?U+'-':U+'+';}D=null;N=clb.pk(a.J,b);if(j1(N,0)!=0){z1(e1(N,Jmb),0)&&(D=D==null?'*':D+','+'*');z1(e1(N,Kmb),0)&&(D=D==null?'!*':D+','+'!*');z1(e1(N,Lmb),0)?D=D==null?'ha':D+','+'ha':z1(e1(N,2),0)?D=D==null?'a':D+','+'a':z1(e1(N,4),0)&&(D=D==null?'!a':D+','+'!a');z1(e1(N,Imb),0)&&(D=D==null?'s':D+','+'s');if(z1(e1(N,Mmb),0)){w=e1(N,Mmb);j1(w,1792)==0?D=D==null?'h0':D+','+'h0':j1(w,1664)==0?D=D==null?'h1':D+','+'h1':j1(w,1408)==0?D=D==null?'h2':D+','+'h2':j1(w,128)==0?D=D==null?'h>0':D+','+'h>0':j1(w,384)==0?D=D==null?'h>1':D+','+'h>1':j1(w,896)==0?D=D==null?'h>2':D+','+'h>2':j1(w,Nmb)==0?D=D==null?'h<3':D+','+'h<3':j1(w,1536)==0&&(D=D==null?'h<2':D+','+'h<2');}if(z1(e1(N,Omb),0)){i=e1(N,Omb);j1(i,Pmb)==0?D=D==null?'c0':D+','+'c0':j1(i,Qmb)==0?D=D==null?'c+':D+','+'c+':j1(i,Rmb)==0&&(D=D==null?'c-':D+','+'c-');}if(z1(e1(N,Smb),0)){L=e1(N,Smb);j1(L,98304)==0?D=D==null?'pi0':D+','+'pi0':j1(L,81920)==0?D=D==null?'pi1':D+','+'pi1':j1(L,49152)==0?D=D==null?'pi2':D+','+'pi2':j1(L,Hmb)==0&&(D=D==null?'pi>0':D+','+'pi>0');}if(z1(e1(N,Tmb),0)){K=e1(N,Tmb);j1(K,3801088)==0?D=D==null?'n1':D+','+'n1':j1(K,Umb)==0?D=D==null?'n2':D+','+'n2':j1(K,Vmb)==0?D=D==null?'n3':D+','+'n3':j1(K,3145728)==0?D=D==null?'n<3':D+','+'n<3':j1(K,Wmb)==0?D=D==null?'n<4':D+','+'n<4':j1(K,Xmb)==0?D=D==null?'n>1':D+','+'n>1':j1(K,Ymb)==0?D=D==null?'n>2':D+','+'n>2':j1(K,Zmb)==0&&(D=D==null?'n>3':D+','+'n>3');}if(z1(e1(N,$mb),0)){o=e1(N,$mb);j1(o,_mb)==0?D=D==null?'e0':D+','+'e0':j1(o,anb)==0?D=D==null?'e1':D+','+'e1':j1(o,bnb)==0?D=D==null?'e2':D+','+'e2':j1(o,cnb)==0?D=D==null?'e3':D+','+'e3':j1(o,15393162788864)==0?D=D==null?'e<2':D+','+'e<2':j1(o,13194139533312)==0?D=D==null?'e<3':D+','+'e<3':j1(o,dnb)==0?D=D==null?'e<4':D+','+'e<4':j1(o,enb)==0?D=D==null?'e>0':D+','+'e>0':j1(o,fnb)==0?D=D==null?'e>1':D+','+'e>1':j1(o,gnb)==0?D=D==null?'e>2':D+','+'e>2':j1(o,hnb)==0?D=D==null?'e>3':D+','+'e>3':j1(o,4947802324992)==0?D=D==null?'e1-2':D+','+'e1-2':j1(o,9345848836096)==0?D=D==null?'e1-3':D+','+'e1-3':j1(o,10445360463872)==0&&(D=D==null?'e2-3':D+','+'e2-3');}if(z1(e1(N,120),0)){Q=e1(N,120);j1(Q,112)==0?D=D==null?'!r':D+','+'!r':j1(Q,8)==0?D=D==null?'r':D+','+'r':j1(Q,96)==0?D=D==null?'rb<3':D+','+'rb<3':j1(Q,104)==0?D=D==null?'rb2':D+','+'rb2':j1(Q,88)==0?D=D==null?'rb3':D+','+'rb3':j1(Q,56)==0&&(D=D==null?'rb4':D+','+'rb4');}z1(e1(N,inb),0)&&(D=D==null?'r'+J1(D1(e1(N,inb),22)):D+','+('r'+J1(D1(e1(N,inb),22))));z1(e1(N,kmb),0)&&(D=clb.Tc(D,clb.Vc(N)));z1(e1(N,jnb),0)&&(D=D==null?'f':D+','+'f');}clb.nk(a.J,b)!=0&&(D=clb.Tc(D,''+clb.nk(a.J,b)));T=0;if(clb.qk(a.J,b)!=0){switch(clb.qk(a.J,b)){case 16:M=M==null?'|':M+','+'|';break;case 32:T=1;break;case 48:T=2;}}l=null;if((a.D&64)==0){if(clb.Xk(a.J,b))l='?';else if(clb.ck(a.J,b)!=0){if(clb.ik(a.J,b)==0||c==null||c[clb.ik(a.J,b)][clb.hk(a.J,b)]>1){if(clb.wn(a.J,b)==2){switch(clb.ck(a.J,b)){case 2:l=clb.Zk(a.J,b)?'p':'P';break;case 1:l=clb.Zk(a.J,b)?'m':'M';break;default:l='*';}}else {switch(clb.ck(a.J,b)){case 1:l=clb.Zk(a.J,b)?'r':'R';break;case 2:l=clb.Zk(a.J,b)?'s':'S';break;default:l='*';}}}}}(a.D&768)!=0&&(l=clb.Tc(l,''+clb.Rt(a.J,b)));I=null;(a.D&16)!=0&&clb.mk(a.J,b)!=0&&(I=''+clb.mk(a.J,b));q=null;if(clb._n(a.J,b)!=-1){p=clb.fd(a,b);p!=-1&&(q=p==0?'abs':((p&255)==1?'&':'or')+(1+(p>>8)));}A=0;(a.D&knb)==0&&(a.J.K?z1(e1(clb.pk(a.J,b),lnb),0)&&(A=clb.Kn(a.J,b)):(clb.uk(a.J,b)!=6||clb.nk(a.J,b)!=0||!a.q[b]||clb.qk(a.J,b)!=0)&&(A=clb.Kn(a.J,b)));H=false;f=clb.fk(a.J,b);if(f!=null&&Zkb.c5(alb.Pgb(f).substr(0,1),']')){D=clb.Tc((alb.Kgb(1,alb.Pgb(f).length+1),alb.Pgb(f).substr(1)),D);f=null;H=true;}if(f!=null){A=0;}else if(clb.kk(a.J,b)!=null){e=z1(e1(clb.pk(a.J,b),1),0)?'[!':'[';f=e+clb.lk(a.J,b)+']';alb.Pgb(f).length>5&&(f=e+clb.kk(a.J,b).length+']');z1(e1(clb.pk(a.J,b),lnb),0)&&(A=-1);}else if(z1(e1(clb.pk(a.J,b),1),0)){f='?';z1(e1(clb.pk(a.J,b),lnb),0)&&(A=-1);}else (clb.uk(a.J,b)!=6||M!=null||D!=null||A>0||!a.q[b])&&(f=clb.jk(a.J,b));G=0;!clb.nl(a.J,b)&z1(e1(clb.pk(a.J,b),Bmb),0)&&clb.Kd(a,-8);if(f!=null){G=(O=(V=ylb.h2(a.e,f),new zlb.n2(V)).b,O);clb.xd(a,clb.Mh(a.P,clb.rk(a.J,b)),clb.Nh(a.P,clb.sk(a.J,b)),f,true);a.r[b]=true;}else clb.nd(a,b)&&clb.wd(a,clb.Mh(a.P,clb.rk(a.J,b)),clb.Nh(a.P,clb.sk(a.J,b)),b);if(M!=null){clb.Os(a,(a.V*2+1)/3|0);X=clb.Mh(a.P,clb.rk(a.J,b))+((G+(O=(V=ylb.h2(a.e,M),new zlb.n2(V)).b,O))/2+1);Z=clb.Nh(a.P,clb.sk(a.J,b))-((a.k*4-4)/8|0);clb.xd(a,X,Z,M,true);clb.Os(a,a.V);}(a.D&2)!=0&&(D=''+b);if(D!=null){H?clb.Os(a,(a.V*5+1)/6|0):clb.Os(a,(a.V*2+1)/3|0);X=clb.Mh(a.P,clb.rk(a.J,b))-(G+(O=(V=ylb.h2(a.e,D),new zlb.n2(V)).b,O))/2;Z=clb.Nh(a.P,clb.sk(a.J,b))-((a.k*4-4)/8|0);clb.xd(a,X,Z,D,true);clb.Os(a,a.V);}if(l!=null){clb.Os(a,(a.V*2+1)/3|0);X=clb.Mh(a.P,clb.rk(a.J,b))-(G+(O=(V=ylb.h2(a.e,l),new zlb.n2(V)).b,O))/2;Z=clb.Nh(a.P,clb.sk(a.J,b))+((a.k*4+4)/8|0);S=a.A;a.J.O!=1&&(a.D&Imb)==0&&clb.Kd(a,448);clb.xd(a,X,Z,l,false);clb.Kd(a,S);clb.Os(a,a.V);}if(I!=null){clb.Os(a,(a.V*2+1)/3|0);X=clb.Mh(a.P,clb.rk(a.J,b))+((G+(O=(V=ylb.h2(a.e,I),new zlb.n2(V)).b,O))/2+1);Z=clb.Nh(a.P,clb.sk(a.J,b))+((a.k*4+4)/8|0);S=a.A;clb.Kd(a,clb._k(a.J,b)?384:448);clb.xd(a,X,Z,I,true);clb.Kd(a,S);clb.Os(a,a.V);}if(q!=null){d=clb.Bd(a,b);clb.Os(a,(a.V*2+1)/3|0);X=clb.Mh(a.P,clb.rk(a.J,b))+0.7*a.k*$wnd.Math.sin(d);Z=clb.Nh(a.P,clb.sk(a.J,b))+0.7*a.k*$wnd.Math.cos(d);S=a.A;!a.I&&a.J.O!=1&&clb.Kd(a,clb.ed(a,b));clb.xd(a,X,Z,q,false);clb.Kd(a,S);clb.Os(a,a.V);}if(A==0&&T==0){a.A==-8&&clb.Kd(a,-9);return;}u=gW(Ykb.yX,emb,6,4,15,1);for(C=0;C<clb.hn(a.J,b);C++){h=clb.xn(a.J,b,C);for(F=0;F<2;F++){if(clb.zk(a.J,F,h)==b){R=clb.yk(a.J,clb.zk(a.J,F,h),clb.zk(a.J,1-F,h));if(R<mnb){u[0]-=R+Gmb;u[3]+=R+Fmb;}else if(R<0){u[2]+=R+Gmb;u[3]-=R;}else if(R<Gmb){u[1]+=R;u[2]+=Gmb-R;}else {u[0]+=R-Gmb;u[1]+=Fmb-R;}}}}clb.wn(a.J,b)==0?clb.gl(a.J,b)?u[3]-=0.2:u[1]-=0.2:u[1]-=0.1;(M!=null||I!=null)&&(u[1]+=10);(D!=null||l!=null)&&(u[3]+=10);s='';if(A!=0){v=(P=(W=ylb.h2(a.e,'H'),new zlb.n2(W)).b,P);t=0;r=a.k;if(A==-1){s='n';clb.Os(a,(a.V*2+1)/3|0);t=(O=(V=ylb.h2(a.e,s),new zlb.n2(V)).b,O);}else if(A>1){s=''+A;clb.Os(a,(a.V*2+1)/3|0);t=(O=(V=ylb.h2(a.e,s),new zlb.n2(V)).b,O);}if(u[1]<0.6||u[3]<0.6){k=clb.Nh(a.P,clb.sk(a.J,b));if(u[1]<=u[3]){u[1]+=10;j=clb.Mh(a.P,clb.rk(a.J,b))+(G+v)/2;}else {u[3]+=10;j=clb.Mh(a.P,clb.rk(a.J,b))-(G+v)/2-t;}}else {j=clb.Mh(a.P,clb.rk(a.J,b));if(u[0]<u[2]){u[0]+=10;k=clb.Nh(a.P,clb.sk(a.J,b))-r;}else {u[2]+=10;k=clb.Nh(a.P,clb.sk(a.J,b))+r;}}if(t>0){X=j+(v+t)/2;Z=k+((a.k*4+4)/8|0);clb.xd(a,X,Z,s,true);clb.Os(a,a.V);}clb.xd(a,j,k,'H',true);}g=0;if(T!=0){J=50;m=0;for(B=0;B<4;B++){n=B>1?B-2:B+2;if(u[B]<J){g=B;J=u[B];m=u[n];}else if(u[B]==J){if(u[n]>m){g=B;m=u[n];}}}switch(g){case 0:j=clb.Mh(a.P,clb.rk(a.J,b));k=clb.Nh(a.P,clb.sk(a.J,b))-a.T-G/2;break;case 1:j=clb.Mh(a.P,clb.rk(a.J,b))+a.T+G/2;k=clb.Nh(a.P,clb.sk(a.J,b));break;case 2:j=clb.Mh(a.P,clb.rk(a.J,b));k=clb.Nh(a.P,clb.sk(a.J,b))+a.T+G/2;break;default:j=clb.Mh(a.P,clb.rk(a.J,b))-a.T-G/2;k=clb.Nh(a.P,clb.sk(a.J,b));}if(T==1){dlb.wi(a.Y,new plb.YF(j-a.T,k-a.T,2*a.T,2*a.T));a.I||dlb.wi(a.S,new clb.Rd(j,k,clb.ld(a,b)?-3:a.p[b]));}else {switch(g){case 2:case 0:Y=2*a.T;$=0;j-=a.T;break;case 1:Y=0;$=2*a.T;k-=a.T;break;default:Y=0;$=2*a.T;k-=a.T;}dlb.wi(a.Y,new plb.YF(j-a.T,k-a.T,2*a.T,2*a.T));a.I||dlb.wi(a.S,new clb.Rd(j,k,clb.ld(a,b)?-3:a.p[b]));dlb.wi(a.Y,new plb.YF(j+Y-a.T,k+$-a.T,2*a.T,2*a.T));a.I||dlb.wi(a.S,new clb.Rd(j+Y,k+$,clb.ld(a,b)?-3:a.p[b]));}}a.A==-8&&clb.Kd(a,-9);};clb.ud=function ud(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w;u=new clb.Sd();c=new clb.Sd();g=new clb.Sd();q=new plb.OF();p=new plb.OF();e=clb.zk(a.J,0,b);f=clb.zk(a.J,1,b);clb.Ls(a,b,clb.Mh(a.P,clb.rk(a.J,e)),clb.Nh(a.P,clb.sk(a.J,e)),clb.Mh(a.P,clb.rk(a.J,f)),clb.Nh(a.P,clb.sk(a.J,f)));!clb.nl(a.J,e)&&!clb.nl(a.J,f)&&z1(e1(B1(clb.pk(a.J,e),clb.pk(a.J,f)),Bmb),0)&&clb.Kd(a,-8);if(!a.o[e]){u.a=clb.Mh(a.P,clb.rk(a.J,e));u.c=clb.Nh(a.P,clb.sk(a.J,e));}else {u.a=a.o[e].a;u.c=a.o[e].b;}if(!a.o[f]){u.b=clb.Mh(a.P,clb.rk(a.J,f));u.d=clb.Nh(a.P,clb.sk(a.J,f));}else {u.b=a.o[f].a;u.d=a.o[f].b;}if((clb.Jk(a.J,b)&nnb)!=0){clb.Gd(a,u)&&clb.Gs(a,u);clb.Kd(a,-9);return;}h=clb.Kk(a.J,b)==64?0:clb.Kk(a.J,b)==32?1:clb.Hk(a.J,b);switch(h){case 1:i=clb.Kk(a.J,b);if((a.D&128)!=0&&(i==257||i==129)){t=clb.zk(a.J,0,b);n=clb.ik(a.J,t);if(n!=0){m=clb.hk(a.J,t);l=0;for(d=0;d<a.J.f;d++)clb.ik(a.J,d)==n&&clb.hk(a.J,d)==m&&++l;l==1&&(i=1);}}switch(i){case 1:clb.Gd(a,u)&&clb.Zc(a,u,e,f);break;case 257:clb.Dd(a,u,e,f);break;case 129:v=u.b-u.a;w=u.d-u.c;if(clb.cl(a.J,clb.tn(a.J,e,f))){j=-3;k=-3;}else {j=a.p[e];k=clb.ed(a,e);j==clb.ek(a.J,e)&&(j=k);}for(o=2;o<17;o+=2){c.a=u.a+o*v/17-o*w/128;c.c=u.c+o*w/17+o*v/128;c.b=u.a+o*v/17+o*w/128;c.d=u.c+o*w/17-o*v/128;if(clb.Gd(a,c)){clb.Kd(a,o<9?j:k);r=smb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(c.a),(Blb.f8(),Blb.b8))))+tmb+umb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(c.c),Blb.b8)))+tmb+vmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(c.b),Blb.b8)))+tmb+wmb+Zkb.O3(Blb.g6(Blb.d6(new Blb.h6(c.d),Blb.b8)))+tmb+xmb+a.d+tmb+ymb+a.j+zmb;clb.Rs(a,r);clb.Kd(a,a.O);}}break;case 32:clb.Gd(a,u)&&clb._c(a,u,e,f);}break;case 0:case 2:if((a.r[e]||clb.mn(a.J,e)==2)&&(a.r[f]||clb.mn(a.J,f)==2)&&!clb.zo(a.J,b)&&h==2){if(!clb.Gd(a,u))break;clb.pd(a,u.b-u.a,u.d-u.c,q);v=q.a/2;w=q.b/2;c.a=u.a+v;c.c=u.c+w;c.b=u.b+v;c.d=u.d+w;g.a=u.a-v;g.c=u.c-w;g.b=u.b-v;g.d=u.d-w;clb.Kk(a.J,b)==386&&clb.Fd(c,g);clb.Zc(a,c,e,f);h==2?clb.Zc(a,g,e,f):clb.Xc(a,g,e,f);}else if((a.r[f]||clb.mn(a.J,f)==2)&&h==2){clb.qd(a,u,b,false);}else if((a.r[e]||clb.mn(a.J,e)==2)&&h==2){clb.qd(a,u,b,true);}else {s=clb.Wn(a.J,b);s==0&&(s=1);c.a=u.a;c.c=u.c;c.b=u.b;c.d=u.d;clb.pd(a,u.b-u.a,u.d-u.c,q);if(s>0){g.a=u.a+q.a;g.c=u.c+q.b;g.b=u.b+q.a;g.d=u.d+q.b;if(clb.od(a,e,f,1,p)||clb.wn(a.J,e)>1){g.a+=p.a+q.b;g.c+=p.b-q.a;}if(clb.od(a,f,e,-1,p)||clb.wn(a.J,f)>1){g.b+=p.a-q.b;g.d+=p.b+q.a;}}else {g.a=u.a-q.a;g.c=u.c-q.b;g.b=u.b-q.a;g.d=u.d-q.b;if(clb.od(a,e,f,-1,p)||clb.wn(a.J,e)>1){g.a+=p.a+q.b;g.c+=p.b-q.a;}if(clb.od(a,f,e,1,p)||clb.wn(a.J,f)>1){g.b+=p.a-q.b;g.d+=p.b+q.a;}}clb.Kk(a.J,b)==386&&clb.Fd(c,g);clb.Gd(a,c)&&clb.Zc(a,c,e,f);h==2?clb.Gd(a,g)&&clb.Zc(a,g,e,f):clb.Gd(a,g)&&clb.Xc(a,g,e,f);}break;case 3:if(clb.Gd(a,u)){clb.Zc(a,u,e,f);clb.pd(a,u.b-u.a,u.d-u.c,q);clb.$c(a,u,e,f,q.a,q.b,c);clb.$c(a,u,e,f,-q.a,-q.b,c);}break;case 4:if(clb.Gd(a,u)){clb.pd(a,u.b-u.a,u.d-u.c,q);clb.$c(a,u,e,f,1.5*q.a,1.5*q.b,c);clb.$c(a,u,e,f,0.5*q.a,0.5*q.b,c);clb.$c(a,u,e,f,-0.5*q.a,-0.5*q.b,c);clb.$c(a,u,e,f,-1.5*q.a,-1.5*q.b,c);}break;case 5:if(clb.Gd(a,u)){clb.Zc(a,u,e,f);clb.pd(a,u.b-u.a,u.d-u.c,q);clb.$c(a,u,e,f,2*q.a,2*q.b,c);clb.$c(a,u,e,f,q.a,q.b,c);clb.$c(a,u,e,f,-q.a,-q.b,c);clb.$c(a,u,e,f,-2*q.a,-2*q.b,c);}}a.A==-8&&clb.Kd(a,-9);};clb.vd=function vd(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r;o=false;for(d=0;d<a.J.g;d++){j=null;if(clb.bl(a.J,d)){l=clb.Bk(a.J,d);k=clb.Ak(a.J,d);j=l==k?'['+l+']':'['+l+':'+k+']';}else (clb.Jk(a.J,d)&onb)!=0?j=(clb.Jk(a.J,d)&onb)==Wmb?'a':(clb.Jk(a.J,d)&384)==256?'r!a':'!a':(clb.Jk(a.J,d)&384)!=0&&(j=(clb.Jk(a.J,d)&384)==256?'r':'!r');n=(clb.Jk(a.J,d)&Ymb)>>17;n!=0&&(j=(j==null?'':j)+n);if(j!=null){b=clb.zk(a.J,0,d);c=clb.zk(a.J,1,d);if(!o){clb.Os(a,(a.V*2+1)/3|0);o=true;}q=(clb.Mh(a.P,clb.rk(a.J,b))+clb.Mh(a.P,clb.rk(a.J,c)))/2;r=(clb.Nh(a.P,clb.sk(a.J,b))+clb.Nh(a.P,clb.sk(a.J,c)))/2;f=clb.Mh(a.P,clb.rk(a.J,c))-clb.Mh(a.P,clb.rk(a.J,b));g=clb.Nh(a.P,clb.sk(a.J,c))-clb.Nh(a.P,clb.sk(a.J,b));e=$wnd.Math.sqrt(f*f+g*g);i=(m=(p=ylb.h2(a.e,j),new zlb.n2(p)).b,0.6*m);h=0.55*a.k;e!=0&&(f>0?clb.xd(a,q+i*g/e,r-h*f/e,j,true):clb.xd(a,q-i*g/e,r+h*f/e,j,true));}}o&&clb.Os(a,a.V);};clb.wd=function wd(a,b,c,d){dlb.wi(a.Y,new plb.YF(b-a.T,c-a.T,2*a.T,2*a.T));a.I||dlb.wi(a.S,new clb.Rd(b,c,clb.ld(a,d)?-3:a.p[d]));};clb.xd=function xd(a,b,c,d,e){var f,g,h,i,j;if(e){g=(f=(h=ylb.h2(a.e,d),new zlb.n2(h)).b,f);i=g/2+(a.k/8|0);j=a.k/2|0;(d=='+'||d=='-')&&(j=j*2/3);dlb.wi(a.Y,new plb.YF(b-i,c-j,2*i,2*j));}a.I||clb.Is(a,d,b,c);};clb.yd=function yd(a){var b;b=a.a;a.a=a.b;a.b=b;b=a.c;a.c=a.d;a.d=b;};clb.zd=function zd(a,b,c){var d;d=b==0?Emb+a[0]-a[a.length-1]:a[b]-a[b-1];c>-2.0943951023931953&&c<pnb?d-=2*$wnd.Math.cos(c+qnb):d-=0.5*$wnd.Math.cos(c+qnb);return d;};clb.Ad=function Ad(a){var b;b=new plb.XF();if(a.a<=a.b){b.c=a.a;b.b=a.b-a.a;}else {b.c=a.b;b.b=a.a-a.b;}if(a.c<=a.d){b.d=a.c;b.a=a.d-a.c;}else {b.d=a.d;b.a=a.c-a.d;}return b;};clb.Bd=function Bd(a,b){var c,d,e,f,g,h,i;c=gW(Ykb.yX,emb,6,clb.gn(a.J,b),15,1);for(e=0;e<clb.gn(a.J,b);e++)c[e]=clb.yk(a.J,b,clb.vn(a.J,b,e));alb.jgb(c,alb.Pgb(X1(dlb.ocb.prototype.kc,dlb.ocb,[])));f=clb.Cd(c,0);g=clb.zd(c,0,f);for(d=1;d<c.length;d++){h=clb.Cd(c,d);i=clb.zd(c,d,h);if(g<i){g=i;f=h;}}return f;};clb.Cd=function Cd(a,b){var c;if(b>0)return (a[b]+a[b-1])/2;c=Fmb+(a[0]+a[a.length-1])/2;return c>Fmb?c-Emb:c;};clb.Dd=function Dd(a,b,c,d){var e,f,g,h;h=new clb.Sd();if(b.a==b.b&&b.c==b.d)return;h.a=b.a;h.c=b.c;h.b=b.b;h.d=b.d;g=clb.Ad(h);for(e=0;e<a.Y.a.length;e++){f=dlb.Bi(a.Y,e);if(f.c>g.c+g.b||f.d>g.d+g.a||g.c>f.c+f.b||g.d>f.d+f.a)continue;if(clb.Ed(a,h.a,h.c,e)){if(clb.Ed(a,h.b,h.d,e))return;clb.Hd(a,h,0,e);clb.Dd(a,h,c,d);return;}if(clb.Ed(a,h.b,h.d,e)){clb.Hd(a,h,1,e);clb.Dd(a,h,c,d);return;}}clb.ad(a,h,c,d);};clb.Ed=function Ed(a,b,c,d){var e;if((a.D&1)!=0)return false;e=dlb.Bi(a.Y,d);return b>e.c&&b<e.c+e.b&&c>e.d&&c<e.d+e.a;};clb.Fd=function Fd(a,b){var c;c=a.b;a.b=b.b;b.b=c;c=a.d;a.d=b.d;b.d=c;};clb.Gd=function Gd(a,b){var c,d,e,f,g,h;if(b.a==b.b&&b.c==b.d){for(g=new dlb.Obb(a.Y);g.a<g.c.a.length;){f=dlb.Nbb(g);if(plb.TF(f,b.a,b.c))return false;}return true;}h=clb.Ad(b);c=false;if(b.a>b.b){clb.yd(b);c=true;}for(d=0;d<a.Y.a.length;d++){f=dlb.Bi(a.Y,d);if(f.c>h.c+h.b||f.d>h.d+h.a||h.c>f.c+f.b||h.d>f.d+f.a)continue;if(clb.Ed(a,b.a,b.c,d)){if(clb.Ed(a,b.b,b.d,d)){c&&clb.yd(b);return false;}clb.Hd(a,b,0,d);e=clb.Gd(a,b);c&&clb.yd(b);return e;}if(clb.Ed(a,b.b,b.d,d)){clb.Hd(a,b,1,d);e=clb.Gd(a,b);c&&clb.yd(b);return e;}}c&&clb.yd(b);return true;};clb.Hd=function Hd(a,b,c,d){var e,f,g,h,i,j,k,l,m,n,o;if(c==0){l=b.a;n=b.c;m=b.b;o=b.d;}else {l=b.b;n=b.d;m=b.a;o=b.c;}k=dlb.Bi(a.Y,d);i=m>l?k.c+k.b:k.c;j=o>n?k.d+k.a:k.d;e=m-l;f=o-n;if($wnd.Math.abs(e)>$wnd.Math.abs(f)){if(n==o){g=i;h=n;}else {g=l+e*(j-n)/f;if(m>l==i>g){h=j;}else {g=i;h=n+f*(i-l)/e;}}}else {if(l==m){g=l;h=j;}else {h=n+f*(i-l)/e;if(o>n==j>h){g=i;}else {g=l+e*(j-n)/f;h=j;}}}if(c==0){b.a=g;b.c=h;}else {b.b=g;b.d=h;}};clb.Id=function Id(a){var b,c,d,e;if(a.J.q==0)return;clb.Gt(a.J,(a.D&256)!=0?63:(a.D&512)!=0?95:31);clb.Uc(a);c=clb.Lt(a.J);d=false;a.p=gW(Ykb.AX,gmb,6,a.J.q,15,1);for(b=0;b<a.J.q;b++){a.p[b]=clb.ek(a.J,b);a.p[b]!=0&&(d=true);clb.nl(a.J,b)&&(a.p[b]=128);clb.Sk(a.J,b)&&(a.D&lnb)==0&&(a.p[b]=256);}clb.Kd(a,-10);clb.hd(a);clb.gd(a);clb.jd(a);clb.Sc(a);clb.Os(a,a.V);clb.Ms(a,a.W);clb.Kd(a,a.O);clb.md(a);alb.igb(a.S.a,0);alb.igb(a.Y.a,0);if((a.D&1)!=0){clb.rd(a,c);clb.sd(a);clb.vd(a);}for(e=0;e<a.J.q;e++){if(clb.ld(a,e)){clb.Kd(a,-3);clb.td(a,e,c);clb.Kd(a,a.O);}else if(a.p[e]!=0){clb.Kd(a,a.p[e]);clb.td(a,e,c);clb.Kd(a,a.O);}else if(!d&&a.J.O!=1&&clb.uk(a.J,e)!=1&&clb.uk(a.J,e)!=6&&(a.D&Nmb)==0&&clb.kk(a.J,e)==null&&clb.uk(a.J,e)<clb.Pc.length){clb.Ld(a,clb.dd(a,clb.Pc[clb.uk(a.J,e)]));clb.td(a,e,c);clb.Kd(a,a.O);}else {clb.td(a,e,c);}}if((a.D&1)==0){clb.sd(a);clb.vd(a);clb.rd(a,c);}};clb.Jd=function Jd(a,b,c,d){var e;e=c/2;switch(d&rnb){case rnb:if(b){a.v.a=b.c+b.b/2;a.v.b=b.d+b.a-e;break;}case 0:a.v.a=a.u.c+a.u.b/2;a.v.b=a.u.d+a.u.a+e;!!b&&a.v.b>b.d+b.a-e&&(a.v.b=b.d+b.a-e);break;case snb:if(b){a.v.a=b.c+b.b/2;a.v.b=b.d+e;break;}case tnb:a.v.a=a.u.c+a.u.b/2;a.v.b=a.u.d-e;!!b&&a.v.b<b.d+e&&(a.v.b=b.d+e);}};clb.Kd=function Kd(a,b){if(a.I)return;if(b==-10){a.A=-999;b=a.O;}b!=-2&&b!=-7&&a.L!=0&&(b=-4);if(b==a.A)return;if(a.A==-8&&b!=-9)return;b==-8&&(a.M=a.A);b==-9&&(b=a.M);a.A=b;switch(b){case 0:clb.Ns(a,a.C==0?unb:a.C);break;case-6:clb.Ns(a,a.C);break;case-4:clb.Ns(a,a.L);break;case-2:clb.Ns(a,a.s);break;case-3:clb.Ns(a,a.t);break;case-7:clb.Ns(a,a.F);break;case-8:clb.Ns(a,a.G);break;case-5:clb.Ns(a,a.N);break;case 64:a.d='rgb(32,96,255)';break;case 128:a.d='rgb(255,0,0)';break;case 256:a.d='rgb(192,0,255)';break;case 192:a.d='rgb(0,255,0)';break;case 320:a.d='rgb(255,160,0)';break;case 384:a.d='rgb(0,128,0)';break;case 448:a.d='rgb(160,0,0)';break;case 1:a.d='rgb(128,128,128)';break;default:a.d='rgb(0,0,0)';}};clb.Ld=function Ld(a,b){if(a.L!=0){if(a.A!=-4){a.A=-4;clb.Ns(a,a.L);}return;}a.A=-5;a.N=b;a.d='rgb('+((b&vnb)>>16)+','+((b&imb)>>8)+','+(b&255)+')';};clb.Md=function Md(a){var b,c,d,e,f;e=clb.Mh(a.P,clb.rk(a.J,0));c=clb.Mh(a.P,clb.rk(a.J,0));f=clb.Nh(a.P,clb.sk(a.J,0));d=clb.Nh(a.P,clb.sk(a.J,0));for(b=0;b<a.J.q;b++){e>clb.Mh(a.P,clb.rk(a.J,b))&&(e=clb.Mh(a.P,clb.rk(a.J,b)));c<clb.Mh(a.P,clb.rk(a.J,b))&&(c=clb.Mh(a.P,clb.rk(a.J,b)));f>clb.Nh(a.P,clb.sk(a.J,b))&&(f=clb.Nh(a.P,clb.sk(a.J,b)));d<clb.Nh(a.P,clb.sk(a.J,b))&&(d=clb.Nh(a.P,clb.sk(a.J,b)));}return new plb.YF(e,f,c-e,d-f);};clb.Nd=function Nd(a,b){var c,d;if(a.J.q==0)return null;a.u=clb.Md(a);c=a.P.c*clb.vk(a.J);d=new clb.Ph(a.u,b,c);if(d.c==1&&d.a==0&&d.b==0){d=null;}else {clb.Ih(d,a.P);clb.Kh(d,a.u);}clb.Jd(a,b,c,Ylb);return d;};clb.Od=function Od(a){var b;b=a.K!=0?a.K:a.B!=0?a.B:-1;a.s=slb.GT(b);a.t=slb.FT(wnb,b);a.F=-24321;a.G=-6291392;};clb.Pd=function Pd(a,b){var c,d,e,f;if(a.J.q==0)return null;e=clb.Ps(a,b);clb.Gt(a.J,(a.D&256)!=0?63:(a.D&512)!=0?95:31);clb.md(a);alb.igb(a.S.a,0);alb.igb(a.Y.a,0);clb.Uc(a);clb.Os(a,a.V);a.I=true;for(d=0;d<a.J.q;d++)clb.td(a,d,null);a.I=false;c=a.P.c*clb.vk(a.J);clb.bd(a,c);clb.Jd(a,b,c,Ylb);if(plb.UF(b,a.u))return e;f=new clb.Ph(a.u,b,c);clb.Ih(f,a.P);clb.Kh(f,a.u);clb.Jh(f,a.v);if(!e)return f;clb.Ih(f,e);return e;};clb.Qd=function Qd(a,b){clb.Rc(this);this.J=a;this.D=b;clb.kd(this);};U1(244,1,{});_.s=0;_.t=0;_.w=0;_.A=0;_.B=0;_.C=0;_.D=0;_.F=0;_.G=0;_.H=0;_.I=false;_.K=0;_.L=0;_.M=0;_.N=0;_.O=0;_.Q=0;_.R=0;_.T=0;_.U=0;_.V=0;_.W=0;_.X=0;Ykb.FX=w3(244);clb.Rd=function Rd(a,b,c){this.b=a;this.c=b;this.a=c;};U1(111,1,{111:1},clb.Rd);_.a=0;_.b=0;_.c=0;Ykb.DX=w3(111);clb.Sd=function Sd(){};U1(60,1,{},clb.Sd);_.a=0;_.b=0;_.c=0;_.d=0;Ykb.EX=w3(60);clb.Td=function Td(a){return xnb+a.ob()+'"'+a.nb()+ynb;};U1(153,1,{153:1});_.lb=function Ud(){var a;a=new Zkb.z5();Zkb.y5(a,xnb+this.ob()+'"'+this.nb()+ynb);return a.a;};_.d=false;Ykb.GX=w3(153);clb.Vd=function Vd(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N,O,P,Q,R,S,T,U,V,W,X;J=gW(Ykb.Z0,Amb,6,b.i.a.length,16,1);A=gW(Ykb.AX,gmb,6,a.f.f,15,1);for(T=0;T<b.i.a.length;T++){J[T]=true;for(q=dlb.Bi(b.j,T),r=0,s=q.length;r<s;++r){t=q[r];if(!a.e[t]){J[T]=false;break;}}if(J[T])for(d=dlb.Bi(b.i,T),g=0,j=d.length;g<j;++g){c=d[g];++A[c];}}I=gW(Ykb.Z0,Amb,6,a.f.f,16,1);for(U=0;U<b.i.a.length;U++){V=dlb.Bi(b.j,U).length;if(V==3||V==5||V==6||V==7){if(J[U]){for(e=dlb.Bi(b.i,U),h=0,k=e.length;h<k;++h){c=e[h];I[c]=true;}Q=true;M=-1;N=0;for(f=dlb.Bi(b.i,U),i=0,l=f.length;i<l;++i){c=f[i];if(V==6||A[c]>1){if(!clb.Zd(a,c,false)){Q=false;break;}}else {S=V==5?clb.Wd(a,c,false):clb.Xd(a,c,false);if(clb.Zd(a,c,false)){if(N<S){N=S;M=c;}}else {if(N==10){Q=false;break;}M=c;N=20;}}}if(Q){for(d=dlb.Bi(b.i,U),g=0,j=d.length;g<j;++g){c=d[g];if(c==M){V==5?clb.Wd(a,c,true):clb.Xd(a,c,true);clb.ie(a,c);}else {clb.Zd(a,c,true);}}}}}}w=gW(Ykb.AX,gmb,6,a.f.f,15,1);C=gW(Ykb.Z0,Amb,6,a.f.f,16,1);for(p=0;p<a.f.g;p++){m=clb.zk(a.f,0,p);n=clb.zk(a.f,1,p);if(!I[m]&&!I[n]){if(a.e[p]){++w[m];++w[n];}if(clb.Kk(a.f,p)==32){C[m]=true;C[n]=true;}}}R=gW(Ykb.AX,gmb,6,a.f.f,15,1);B=gW(Ykb.AX,gmb,6,a.f.f,15,1);for(X=0;X<a.f.f;X++){if(w[X]==1){B[0]=X;v=0;D=0;while(v<=D){for(F=0;F<clb.wn(a.f,B[v]);F++){if(a.e[clb.xn(a.f,B[v],F)]){u=clb.vn(a.f,B[v],F);if((v==0||u!=B[v-1])&&w[u]!=0){B[++D]=u;if((w[u]&1)!=0){for(L=1;L<D;L+=2)R[B[L]]=-1;D=0;}break;}}}++v;}}}o=gW(Ykb.Z0,Amb,6,a.f.f,16,1);for(W=0;W<a.f.f;W++){if(!o[W]&&w[W]!=0){B[0]=W;o[W]=true;v=0;D=0;while(v<=D){for(F=0;F<clb.wn(a.f,B[v]);F++){if(a.e[clb.xn(a.f,B[v],F)]){u=clb.vn(a.f,B[v],F);if(!o[u]){B[++D]=u;o[u]=true;}}}++v;}if((D&1)==0){for(G=0;G<=D;G++)R[B[G]]==0&&(R[B[G]]=clb.Yd(a,B[G],false));K=true;for(H=0;H<=D;H++){if(R[B[H]]<=0){if(!clb.Zd(a,B[H],false)){K=false;break;}}}if(K){P=0;O=-1;for(F=0;F<=D;F++){if(P<R[B[F]]){P=R[B[F]];O=B[F];}}if(P>0){clb.Yd(a,O,true);clb.ie(a,O);}}}}}};clb.Wd=function Wd(a,b,c){if(clb.uk(a.f,b)==7){if(clb.gn(a.f,b)==3)return 6;else if(clb.wn(a.f,b)==2)return 4;}else if(clb.uk(a.f,b)==8){return 10;}else if(clb.uk(a.f,b)==15||clb.uk(a.f,b)==33){if(clb.wn(a.f,b)==3)return 8;}else if(clb.uk(a.f,b)==16||clb.uk(a.f,b)==34||clb.uk(a.f,b)==52){if(clb.wn(a.f,b)==2)return 12;}else if(clb.uk(a.f,b)==6){c&&clb.Il(a.f,b,-1);return clb.gn(a.f,b)!=clb.hn(a.f,b)?2:3;}return 0;};clb.Xd=function Xd(a,b,c){if(a.a){if(clb.gn(a.f,b)!=3)return 0;}else {if(clb.gn(a.f,b)>3)return 0;}if(clb.uk(a.f,b)==6){c&&clb.Il(a.f,b,1);return 2;}if(clb.uk(a.f,b)==5){return 4;}return 0;};clb.Yd=function Yd(a,b,c){if(clb.dk(a.f,b)!=0)return 0;if(a.a){if(clb.uk(a.f,b)==5){if(clb.Pn(a.f,b)!=2)return 0;c&&clb.Il(a.f,b,1);return 1;}if(clb.uk(a.f,b)==7){if(clb.Pn(a.f,b)!=2)return 0;c&&clb.Il(a.f,b,-1);return clb._d(a,b)?6:3;}if(clb.uk(a.f,b)==8){if(clb.Pn(a.f,b)!=1)return 0;c&&clb.Il(a.f,b,-1);return clb._d(a,b)?7:4;}if(clb.uk(a.f,b)==16){if(clb.Pn(a.f,b)!=1)return 0;c&&clb.Il(a.f,b,-1);return clb._d(a,b)?5:2;}if(clb.uk(a.f,b)==34){if(clb.Pn(a.f,b)!=1)return 0;c&&clb.Il(a.f,b,-1);return clb._d(a,b)?4:1;}}else {if(clb.uk(a.f,b)==5){if(clb.Pn(a.f,b)>2)return 0;c&&clb.Il(a.f,b,1);return 1;}if(clb.uk(a.f,b)==7){if(clb.Pn(a.f,b)>2)return 0;c&&clb.Il(a.f,b,-1);return clb._d(a,b)?5:3;}if(clb.uk(a.f,b)==8){if(clb.Pn(a.f,b)>1)return 0;c&&clb.Il(a.f,b,-1);return clb._d(a,b)?7:4;}if(clb.uk(a.f,b)==16){if(clb.Pn(a.f,b)>1)return 0;c&&clb.Il(a.f,b,-1);return clb._d(a,b)?5:2;}}return 0;};clb.Zd=function Zd(a,b,c){var d,e,f;d=clb.uk(a.f,b);if(d>=5&&d<=8||d==15||d==16||d==33||d==34||d==52){f=clb.Ln(a.f,b);if(f!=0)return true;e=clb.dk(a.f,b);if(d==5&&e>=0){c&&clb.Il(a.f,b,e-1);return true;}if(d!=5&&e<=0){c&&clb.Il(a.f,b,e+1);return true;}}return false;};clb.$d=function $d(a,b,c,d){var e;for(e=0;e<clb.wn(a.f,b);e++)if(d[clb.xn(a.f,b,e)]==1&&clb.vn(a.f,b,e)!=c)return e;return -1;};clb._d=function _d(a,b){var c;for(c=0;c<clb.wn(a.f,b);c++)if(clb.jl(a.f,clb.vn(a.f,b,c)))return true;return false;};clb.ae=function ae(a,b,c){var d;for(d=0;d<clb.wn(a.f,b);d++)if(c[clb.xn(a.f,b,d)]>1)return true;return false;};clb.be=function be(a,b){var c;for(c=0;c<clb.wn(a.f,b);c++)if(a.e[clb.xn(a.f,b,c)])return true;return false;};clb.ce=function ce(a,b,c){var d,e,f,g,h,i,j,k,l,m;clb.Gt(a.f,1);if(b!=null){a.e=b;}else {a.e=gW(Ykb.Z0,Amb,6,a.f.g,16,1);for(e=0;e<a.f.g;e++){if(clb.Kk(a.f,e)==64){a.e[e]=true;clb.jm(a.f,e,1);}}}a.g=0;a.d=gW(Ykb.Z0,Amb,6,a.f.f,16,1);for(f=0;f<a.f.g;f++){if(a.e[f]){++a.c;for(h=0;h<2;h++){if(!a.d[clb.zk(a.f,h,f)]){a.d[clb.zk(a.f,h,f)]=true;++a.b;}}}}if(a.c==0)return true;a.a=false;clb.ke(a,c);a.f.K&&clb.ee(a);m=new clb.Fr(a.f,1);c&&clb.Vd(a,m);clb.le(a,m);clb.he(a,m);clb.je(a);clb.fe(a);while(clb.ge(a,m))clb.fe(a);while(a.c!=0){g=false;if(!g){for(k=0;k<m.i.a.length;k++){if(dlb.Bi(m.j,k).length==6){j=true;l=dlb.Bi(m.j,k);for(i=0;i<6;i++){if(!a.e[l[i]]){j=false;break;}}if(j){for(h=0;h<6;h+=2)clb.de(a,l[h]);g=true;break;}}}}if(!g){for(d=0;d<a.f.g;d++){if(a.e[d]){clb.de(a,d);clb.fe(a);g=true;break;}}}}return a.b==a.g;};clb.de=function de(a,b){var c,d,e,f;if(clb.Kk(a.f,b)==1){clb.jm(a.f,b,2);a.g+=2;}for(e=0;e<2;e++){c=clb.zk(a.f,e,b);a.d[c]=false;for(f=0;f<clb.wn(a.f,c);f++){d=clb.xn(a.f,c,f);if(a.e[d]){a.e[d]=false;--a.c;}}}};clb.ee=function ee(a){var b,c,d,e,f,g,h,i;for(c=0;c<a.f.g;c++){if(a.e[c]){for(e=0;e<2;e++){h=clb.zk(a.f,e,c);b=false;for(g=0;g<clb.wn(a.f,h);g++){if(c!=clb.xn(a.f,h,g)&&a.e[clb.xn(a.f,h,g)]){b=true;break;}}if(!b){i=c;d=clb.zk(a.f,1-e,c);while(i!=-1){a.e[i]=false;--a.c;clb.jm(a.f,i,64);i=-1;h=d;for(f=0;f<clb.wn(a.f,h);f++){if(a.e[clb.xn(a.f,h,f)]){if(i==-1){i=clb.xn(a.f,h,f);d=clb.vn(a.f,h,f);}else {h=-1;i=-1;break;}}}}break;}}}}};clb.fe=function fe(a){var b,c,d,e,f,g,h;do{h=false;for(c=0;c<a.f.g;c++){if(a.e[c]){f=false;for(e=0;e<2;e++){d=clb.zk(a.f,e,c);b=false;for(g=0;g<clb.wn(a.f,d);g++){if(c!=clb.xn(a.f,d,g)&&a.e[clb.xn(a.f,d,g)]){b=true;break;}}if(!b){f=true;break;}}if(f){h=true;clb.de(a,c);}}}}while(h);};clb.ge=function ge(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o;o=gW(Ykb.AX,gmb,6,a.f.g,15,1);for(m=0;m<b.i.a.length;m++){n=dlb.Bi(b.j,m);l=true;for(k=0;k<n.length;k++){if(!a.e[n[k]]){l=false;break;}}if(l)for(j=0;j<n.length;j++)++o[n[j]];}i=a.c;for(f=0;f<a.f.g;f++){if(o[f]==1){for(j=0;j<2&&a.e[f];j++){c=clb.zk(a.f,j,f);d=clb.zk(a.f,1-j,f);if(clb.ae(a,c,o)&&!clb.ae(a,d,o)){while(-1!=(h=clb.$d(a,d,c,o))){e=clb.vn(a.f,d,h);g=clb.xn(a.f,d,h);if(!a.e[g])break;clb.de(a,g);h=clb.$d(a,e,d,o);if(h==-1)break;c=e;d=clb.vn(a.f,e,h);}}}}}return i!=a.c;};clb.he=function he(a,b){var c;for(c=0;c<a.f.g;c++){if(a.e[c]&&clb.yr(b,c)){clb.ie(a,clb.zk(a.f,0,c));clb.ie(a,clb.zk(a.f,1,c));}}};clb.ie=function ie(a,b){var c,d;if(a.d[b]){a.d[b]=false;--a.b;}for(d=0;d<clb.wn(a.f,b);d++){c=clb.xn(a.f,b,d);if(a.e[c]){a.e[c]=false;--a.c;}}};clb.je=function je(a){var b,c,d,e,f;for(c=0;c<a.f.g;c++){if(clb.Hk(a.f,c)==2){for(e=0;e<2;e++){b=clb.zk(a.f,e,c);if(clb.uk(a.f,b)<=8){for(f=0;f<clb.wn(a.f,b);f++){d=clb.xn(a.f,b,f);if(a.e[d]){clb.ie(a,b);break;}}}}}}};clb.ke=function ke(a,b){var c;for(c=0;c<a.f.f;c++)a.d[c]&&clb.Ln(a.f,c)==0&&(!b||clb.uk(a.f,c)==5&&clb.dk(a.f,c)<0||clb.uk(a.f,c)==6||clb.uk(a.f,c)==14||clb.gl(a.f,c)&&clb.dk(a.f,c)>0)&&clb.ie(a,c);};clb.le=function le(a,b){var c,d,e,f,g,h,i,j,k,l,m,n;for(k=0;k<b.i.a.length;k++){n=dlb.Bi(b.j,k).length;if(n==3||n==5||n==7){l=dlb.Bi(b.i,k);for(d=0;d<n;d++){c=l[d];clb.be(a,c)&&(n==5?(clb.uk(a.f,c)==6&&clb.dk(a.f,c)==-1&&clb.gn(a.f,c)==3||clb.uk(a.f,c)==7&&clb.dk(a.f,c)==0&&clb.gn(a.f,c)==3||clb.uk(a.f,c)==8&&clb.dk(a.f,c)==0&&clb.wn(a.f,c)==2||clb.uk(a.f,c)==16&&clb.dk(a.f,c)==0&&clb.wn(a.f,c)==2||clb.uk(a.f,c)==34&&clb.dk(a.f,c)==0&&clb.wn(a.f,c)==2)&&clb.ie(a,c):(clb.uk(a.f,c)==5&&clb.dk(a.f,c)==0&&clb.gn(a.f,c)==3||clb.uk(a.f,c)==6&&clb.dk(a.f,c)==1)&&clb.ie(a,c));}}}for(j=0;j<b.i.a.length;j++){if(dlb.Bi(b.j,j).length==5){m=dlb.Bi(b.j,j);f=true;for(e=0;e<m.length;e++){if(!a.e[m[e]]){f=false;break;}}if(f){l=dlb.Bi(b.i,j);h=0;g=-1;for(d=0;d<m.length;d++){if(clb.dk(a.f,l[d])==-1&&clb.uk(a.f,l[d])==6){i=clb.gn(a.f,l[d])==3?3:clb.hn(a.f,l[d])==3?2:1;if(h<i){h=i;g=l[d];}}}g!=-1&&clb.ie(a,g);}}}};clb.me=function me(a){this.f=a;};U1(76,1,{},clb.me);_.a=false;_.b=0;_.c=0;_.g=0;Ykb.HX=w3(76);clb.ne=function ne(a,b){var c,d;c=0;for(d=0;d<a.j[b];d++)a.n[b][d]==2&&(clb.uk(a,a.i[b][d])==7||clb.uk(a,a.i[b][d])==8||clb.uk(a,a.i[b][d])==16)&&++c;return c;};clb.oe=function oe(a,b){var c,d,e,f,g,h;if(a.s[b]==0){return false;}h=true;c=a.s[b];f=a.j[b];g=0;for(d=0;d<f;d++){e=a.i[b][d];g+=a.s[e];}$wnd.Math.abs(c)<=$wnd.Math.abs(g)&&Zkb.M4(c)!=Zkb.M4(g)&&(h=false);return h;};clb.pe=function pe(a,b){var c,d,e,f,g,h,i;if(a.C[b]!=8||a.s[b]!=0||a.j[b]!=1||a.n[b][0]!=1)return false;c=a.i[b][0];if(a.C[c]==6){g=a.j[c];for(d=0;d<g;d++){e=a.i[c][d];if(e==b){continue;}if(a.C[e]!=8){continue;}f=clb.tn(a,c,e);if(a.H[f]==2)return true;}}else if(a.C[c]==7){if(a.s[c]==1)return true;}else if(a.C[c]==16){h=a.j[c];i=0;for(d=0;d<h;d++){e=a.i[c][d];if(e==b)continue;if(a.C[e]!=8)continue;f=clb.tn(a,c,e);a.H[f]==2&&++i;}if(i==2)return true;}else if(clb.qe(a,b))return true;return false;};clb.qe=function qe(a,b){var c,d,e,f,g;if(a.C[b]!=8)return false;if(a.j[b]!=1)return false;c=a.i[b][0];if(a.C[c]==15){g=a.j[c];for(d=0;d<g;d++){e=a.i[c][d];if(e==b)continue;if(a.C[e]!=8)continue;f=clb.tn(a,c,e);if(a.H[f]==2)return true;}}return false;};clb.re=function re(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A;if(a.C[b]!=7||a.s[b]!=0||a.j[b]+a.o[b]>3)return false;if(b<a.f&&clb.rr(a.p,b)){if(a.o[b]!=1)return false;if(clb.pn(a,b,7)!=1)return false;u=(clb.Gt(a,7),a.p);for(s=0;s<u.i.a.length;s++){if(clb.tr(u,s,b)){if(dlb.Bi(u.j,s).length==5||dlb.Bi(u.j,s).length==6){v=dlb.Bi(u.i,s);q=-1;for(i=0;i<v.length;i++){if(v[i]==b){q=i;break;}}e=0;r=null;p=null;if(v.length==5){r=gW(Ykb.AX,gmb,6,2,15,1);r[0]=v[q-1<0?q+4:q-1];r[1]=v[q-4<0?q+1:q-4];p=gW(Ykb.AX,gmb,6,2,15,1);p[0]=v[q-2<0?q+3:q-2];p[1]=v[q-3<0?q+2:q-3];}if(v.length==6){r=gW(Ykb.AX,gmb,6,3,15,1);r[0]=v[q-1<0?q+5:q-1];r[1]=v[q-3<0?q+3:q-3];r[2]=v[q-5<0?q+1:q-5];p=gW(Ykb.AX,gmb,6,2,15,1);p[0]=v[q-2<0?q+4:q-2];p[1]=v[q-4<0?q+2:q-4];}for(j=0;j<v.length;j++)b!=v[j]&&clb.uk(a,v[j])==7&&clb.mn(a,v[j])==1&&--e;for(k=0;k<r.length;k++){f=-1;g=-1;for(o=0;o<clb.wn(a,r[k]);o++){if(!clb.mo(a,clb.xn(a,r[k],o))){f=clb.vn(a,r[k],o);g=clb.xn(a,r[k],o);break;}}if(f!=-1){if(a.C[f]==7&&a.o[f]==0&&a.j[f]+a.o[f]<=3&&!clb.se(a,f,false)){++e;continue;}if(a.C[f]==8&&a.j[f]==1){e+=2;continue;}if(g<a.g&&clb.sr(a.p,g)){for(w=0;w<u.i.a.length;w++){if(u.d[w]&&clb.tr(u,w,f)){t=dlb.Bi(u.i,w);for(n=0;n<t.length;n++){if(clb.uk(a,t[n])==7&&clb.mn(a,t[n])==1){--e;break;}}break;}}}}}for(l=0;l<p.length;l++){f=-1;for(n=0;n<clb.wn(a,p[l]);n++)clb.mo(a,clb.xn(a,p[l],n))||(f=clb.vn(a,p[l],n));clb.uk(a,p[l])==6?f!=-1&&clb.ne(a,f)!=0&&--e:clb.uk(a,p[l])==7&&clb.mn(a,p[l])==0&&(f==-1||!(f<a.f&&clb.rr(a.p,f))&&clb.ne(a,f)==0)&&++e;}return e>0;}break;}}return false;}if(a.o[b]>1)return false;if(a.o[b]==1){m=-1;A=0;for(i=0;i<a.j[b];i++){d=a.i[b][i];if(a.n[b][i]==2){if(a.C[d]!=6)return false;m=d;continue;}if(a.C[d]==8)return false;if(a.C[d]==7){--A;clb.se(a,d,false)&&--A;continue;}d<a.f&&clb.rr(a.p,d)&&--A;}if(m==-1)return false;c=0;for(j=0;j<a.j[m];j++){if(a.n[m][j]==1){d=a.i[m][j];if(clb.ne(a,d)!=0)return false;d<a.f&&clb.rr(a.p,d)&&++c;a.C[d]==7&&!clb.se(a,d,true)&&++A;(a.C[d]==8||a.C[d]==16)&&--A;}}c==2&&--A;return A>=0;}for(h=0;h<a.j[b];h++){d=a.i[b][h];if(d<a.f&&clb.rr(a.p,d))return false;if(a.C[d]!=6)return false;if(clb.ne(a,d)!=0)return false;if(a.o[d]!=0&&clb.te(a,d))return false;}return true;};clb.se=function se(a,b,c){var d,e,f;d=false;for(f=0;f<a.j[b];f++){if(!clb.mo(a,a.k[b][f])&&a.n[b][f]==1){e=a.i[b][f];if(!(e<a.f&&clb.rr(a.p,e))&&(a.C[e]==6&&clb.ne(a,e)==1||a.C[e]==16&&clb.ne(a,e)==2)){if(d||!c)return true;d=true;}}}return false;};clb.te=function te(a,b){var c,d,e;for(d=0;d<a.j[b];d++){if(a.n[b][d]!=1){c=a.i[b][d];for(e=0;e<a.j[c];e++)if(a.n[c][e]==1&&clb.ne(a,a.i[c][e])!=0)return true;}}return false;};clb.we=function we(){clb.we=W1;clb.ue=oW(aW(Ykb.Y0,1),znb,6,15,[-1,-1,-1,0,0,1,2,3,4,5,-1,0,0,0,6,7,8,9,-1,0,0,10,10,10,10,10,10,10,10,10,10,1,11,11,12,13,-1,0,0,10,10,10,10,10,10,10,10,10,10,0,0,0,11,14,-1,0,0,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,10,10,10,10,10,10,10,10,1,1,1,1,-1,-1,-1,-1,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1]);clb.ve=oW(aW(Ykb.Y0,1),znb,6,15,[-1,-1,-1,0,0,0,2,5,5,5,-1,0,0,0,0,9,9,9,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,9,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,-1,-1,-1,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1]);};clb.xe=function xe(a,b,c){clb.we();var d,e,f,g,h,i,j,k,l,m,n,o,p,q;clb.Gt(a,7);n=gW(Ykb.BX,Anb,6,a.j[b],14,1);m=0;for(i=0;i<a.j[b];i++){e=a.i[b][i];if(a.C[e]==1)continue;g=0;if((c&32)!=0){f=q1(a.n[b][i]);c==32190?j1(f,3)<0&&clb.qo(a,a.k[b][i])&&a.o[b]==1&&(f=0):j1(f,3)<0&&clb.mo(a,a.k[b][i])&&(f=0);g=B1(g,C1(f,4));}if((c&128)!=0){if(clb.ue[a.C[e]]==-1)throw c1(new Zkb.rz(Bnb+a.C[e]));g=d1(g,q1(clb.ue[a.C[e]]));}else if((c&64)!=0){if(clb.ve[a.C[e]]==-1)throw c1(new Zkb.rz(Bnb+a.C[e]));g=d1(g,q1(clb.ve[a.C[e]]));}if((c&256)!=0){o=a.j[e]-1;o>3&&(o=3);(c&512)==0&&o>1&&(o=1);g=B1(g,Ykb.m1(o<<6));}(c&Nmb)!=0&&(a.u[e]&8)!=0&&(g=B1(g,256));(c&lnb)!=0&&e<a.f&&clb.rr(a.p,e)&&(g=B1(g,512));(c&Hmb)!=0&&(a.u[e]&knb)!=0&&(g=B1(g,Nmb));k=0;while(j1(g,n[k])<0)++k;for(l=i;l>k;l--)n[l]=n[l-1];n[k]=g;++m;}m>4&&(m=4);d=0;for(j=0;j<m;j++){d=C1(d,11);d=B1(d,n[j]);}d=C1(d,15);if(clb.ue[a.C[b]]==-1)throw c1(new Zkb.rz(Bnb+a.C[b]));d=B1(d,q1(clb.ue[a.C[b]]));if((c&2)!=0){q=!!a.p&&b<a.f?clb.gr(a.p,b):0;q>9&&(q=9);q>2&&(q-=2);d=B1(d,Ykb.m1(q<<4));}else (c&1)!=0&&(a.u[b]&8)!=0&&(d=B1(d,64));(c&4)!=0&&b<a.f&&clb.rr(a.p,b)&&(d=B1(d,Nmb));(c&8)!=0&&(a.u[b]&Imb)!=0&&(d=B1(d,lnb));(c&16)!=0&&(a.u[b]&knb)!=0&&(d=B1(d,Imb));if((c&Imb)!=0){clb.oe(a,b)&&(d=B1(d,knb));if(clb.re(a,b)){for(h=0;h<a.f;h++){if(clb.pe(a,h)){d=B1(d,Hmb);break;}}}}if((c&knb)!=0){p=q1(clb.pn(a,b,10));d=B1(d,C1(p,7));}return d;};clb.ye=function ye(a,b,c,d,e,f){var g,h,i,j;i=0;for(h=0;h<a.P.f;h++)z1(e1(clb.pk(a.P,a.w[h]),d),0)&&++i;if(i==0)return;clb.wf(a,b);clb.rf(a,Ykb.m1(i),c);for(g=0;g<a.P.f;g++){j=e1(clb.pk(a.P,a.w[g]),d);if(j1(j,0)!=0){clb.rf(a,Ykb.m1(g),c);e!=1&&clb.rf(a,D1(j,f),e);}}};clb.ze=function ze(a,b,c,d,e,f){var g,h,i,j;i=0;for(h=0;h<a.P.g;h++)(clb.Jk(a.P,a.A[h])&d)!=0&&++i;if(i==0)return;clb.wf(a,b);clb.rf(a,Ykb.m1(i),c);for(g=0;g<a.P.g;g++){j=clb.Jk(a.P,a.A[g])&d;if(j!=0){clb.rf(a,Ykb.m1(g),c);e!=1&&clb.rf(a,Ykb.m1(j>>f),e);}}};clb.Ae=function Ae(a){var b,c,d,e,f,g;while(true){f=gW(Ykb.Z0,Amb,6,a.S+1,16,1);d=-1;for(c=0;c<a.P.f;c++){if(clb.Ln(a.P,c)!=0){f[a.d[c]]&&d<a.d[c]&&(d=a.d[c]);f[a.d[c]]=true;}}if(d==-1)break;e=0;for(b=0;b<a.P.f;b++){g=0;a.d[b]==d&&(g=++e);clb.sg(a.c[b],b);clb.pg(a.c[b],a.b,q1(a.d[b]));clb.pg(a.c[b],8,Ykb.m1(g));}a.S=clb._e(a);}};clb.Be=function Be(a){var b,c,d,e;for(c=0;c<a.P.f;c++){clb.sg(a.c[c],c);clb.pg(a.c[c],2*a.b+4,C1(q1(a.d[c]),a.b+4));}e=false;for(b=0;b<a.P.f;b++)e=e|clb.Oe(a,b,3);for(d=0;d<a.P.g;d++)e=e|clb.Je(a,d,3);if(!e)return false;while(a.S<a.P.f){e=clb.Ye(a);if(!e)break;clb.$e(a);!!a.N&&clb.Ig(a.N,a.d);}return true;};clb.Ce=function Ce(a){var b,c,d,e,f;for(c=0;c<a.P.f;c++){clb.sg(a.c[c],c);clb.pg(a.c[c],a.b+1,q1(2*a.d[c]));}f=gW(Ykb.AX,gmb,6,a.S+1,15,1);f