/**
 * mrz - Parse MRZ (Machine Readable Zone) from identity documents
 * @version v3.1.3
 * @link https://github.com/cheminfo/mrz#readme
 * @license MIT
 */
(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory() :
  typeof define === 'function' && define.amd ? define(factory) :
  (global = typeof globalThis !== 'undefined' ? globalThis : global || self, global.mrz = factory());
}(this, (function () { 'use strict';

  const formats = {
    TD1: 'TD1',
    TD2: 'TD2',
    TD3: 'TD3',
    SWISS_DRIVING_LICENSE: 'SWISS_DRIVING_LICENSE',
    FRENCH_NATIONAL_ID: 'FRENCH_NATIONAL_ID'
  };
  Object.freeze(formats);
  var formats_1 = formats;

  const states = {
    "AFG": "Afghanistan",
    "ALA": "Åland Islands",
    "ALB": "Albania",
    "DZA": "Algeria",
    "ASM": "American Samoa",
    "AND": "Andorra",
    "AGO": "Angola",
    "AIA": "Anguilla",
    "ATA": "Antarctica",
    "ATG": "Antigua and Barbuda",
    "ARG": "Argentina",
    "ARM": "Armenia",
    "ABW": "Aruba",
    "AUS": "Australia",
    "AUT": "Austria",
    "AZE": "Azerbaijan",
    "BHS": "Bahamas",
    "BHR": "Bahrain",
    "BGD": "Bangladesh",
    "BRB": "Barbados",
    "BLR": "Belarus",
    "BEL": "Belgium",
    "BLZ": "Belize",
    "BEN": "Benin",
    "BMU": "Bermuda",
    "BTN": "Bhutan",
    "BOL": "Bolivia, Plurinational State of",
    "BES": "Bonaire, Sint Eustatius and Saba",
    "BIH": "Bosnia and Herzegovina",
    "BWA": "Botswana",
    "BVT": "Bouvet Island",
    "BRA": "Brazil",
    "IOT": "British Indian Ocean Territory",
    "BRN": "Brunei Darussalam",
    "BGR": "Bulgaria",
    "BFA": "Burkina Faso",
    "BDI": "Burundi",
    "CPV": "Cabo Verde",
    "KHM": "Cambodia",
    "CMR": "Cameroon",
    "CAN": "Canada",
    "CYM": "Cayman Islands",
    "CAF": "Central African Republic",
    "TCD": "Chad",
    "CHL": "Chile",
    "CHN": "China",
    "CXR": "Christmas Island",
    "CCK": "Cocos (Keeling) Islands",
    "COL": "Colombia",
    "COM": "Comoros",
    "COG": "Congo",
    "COD": "Congo, Democratic Republic of the",
    "COK": "Cook Islands",
    "CRI": "Costa Rica",
    "CIV": "Côte d’Ivoire",
    "HRV": "Croatia",
    "CUB": "Cuba",
    "CUW": "Curaçao",
    "CYP": "Cyprus",
    "CZE": "Czech Republic",
    "DNK": "Denmark",
    "DJI": "Djibouti",
    "DMA": "Dominica",
    "DOM": "Dominican Republic",
    "ECU": "Ecuador",
    "EGY": "Egypt",
    "SLV": "El Salvador",
    "GNQ": "Equatorial Guinea",
    "ERI": "Eritrea",
    "EST": "Estonia",
    "ETH": "Ethiopia",
    "FLK": "Falkland Islands (Malvinas)",
    "FRO": "Faroe Islands",
    "FJI": "Fiji",
    "FIN": "Finland",
    "FRA": "France",
    "GUF": "French Guiana",
    "PYF": "French Polynesia",
    "ATF": "French Southern Territories",
    "GAB": "Gabon",
    "GMB": "Gambia",
    "GEO": "Georgia",
    "D": "Germany",
    "GHA": "Ghana",
    "GIB": "Gibraltar",
    "GRC": "Greece",
    "GRL": "Greenland",
    "GRD": "Grenada",
    "GLP": "Guadeloupe",
    "GUM": "Guam",
    "GTM": "Guatemala",
    "GGY": "Guernsey",
    "GIN": "Guinea",
    "GNB": "Guinea-Bissau",
    "GUY": "Guyana",
    "HTI": "Haiti",
    "HMD": "Heard Island and McDonald Islands",
    "VAT": "Holy See (Vatican City State)",
    "HND": "Honduras",
    "HKG": "Hong Kong Special Administrative Region of China",
    "HUN": "Hungary",
    "ISL": "Iceland",
    "IND": "India",
    "IDN": "Indonesia",
    "IRN": "Iran (Islamic Republic of)",
    "IRQ": "Iraq",
    "IRL": "Ireland",
    "IMN": "Isle of Man",
    "ISR": "Israel",
    "ITA": "Italy",
    "JAM": "Jamaica",
    "JPN": "Japan",
    "JEY": "Jersey",
    "JOR": "Jordan",
    "KAZ": "Kazakhstan",
    "KEN": "Kenya",
    "KIR": "Kiribati",
    "PRK": "Korea, Democratic People’s Republic of",
    "KOR": "Korea, Republic of",
    "KWT": "Kuwait",
    "KGZ": "Kyrgyzstan",
    "LAO": "Lao People’s Democratic Republic",
    "LVA": "Latvia",
    "LBN": "Lebanon",
    "LSO": "Lesotho",
    "LBR": "Liberia",
    "LBY": "Libya",
    "LIE": "Liechtenstein",
    "LTU": "Lithuania",
    "LUX": "Luxembourg",
    "MAC": "Macao Special Administrative Region of China",
    "MKD": "Macedonia, the former Yugoslav Republic of",
    "MDG": "Madagascar",
    "MWI": "Malawi",
    "MYS": "Malaysia",
    "MDV": "Maldives",
    "MLI": "Mali",
    "MLT": "Malta",
    "MHL": "Marshall Islands",
    "MTQ": "Martinique",
    "MRT": "Mauritania",
    "MUS": "Mauritius",
    "MYT": "Mayotte",
    "MEX": "Mexico",
    "FSM": "Micronesia (Federated States of)",
    "MDA": "Moldova, Republic of",
    "MCO": "Monaco",
    "MNG": "Mongolia",
    "MNE": "Montenegro",
    "MSR": "Montserrat",
    "MAR": "Morocco",
    "MOZ": "Mozambique",
    "MMR": "Myanmar",
    "NAM": "Namibia",
    "NRU": "Nauru",
    "NPL": "Nepal",
    "NLD": "Netherlands",
    "ANT": "Netherlands Antilles",
    "NTZ": "Neutral Zone",
    "NCL": "New Caledonia",
    "NZL": "New Zealand",
    "NIC": "Nicaragua",
    "NER": "Niger",
    "NGA": "Nigeria",
    "NIU": "Niue",
    "NFK": "Norfolk Island",
    "MNP": "Northern Mariana Islands",
    "NOR": "Norway",
    "OMN": "Oman",
    "PAK": "Pakistan",
    "PLW": "Palau",
    "PSE": "Palestine, State of",
    "PAN": "Panama",
    "PNG": "Papua New Guinea",
    "PRY": "Paraguay",
    "PER": "Peru",
    "PHL": "Philippines",
    "PCN": "Pitcairn",
    "POL": "Poland",
    "PRT": "Portugal",
    "PRI": "Puerto Rico",
    "QAT": "Qatar",
    "REU": "Réunion",
    "ROU": "Romania",
    "RUS": "Russian Federation",
    "RWA": "Rwanda",
    "BLM": "Saint Barthélemy",
    "SHN": "Saint Helena, Ascension and Tristan da Cunha",
    "KNA": "Saint Kitts and Nevis",
    "LCA": "Saint Lucia",
    "MAF": "Saint Martin (French part)",
    "SPM": "Saint Pierre and Miquelon",
    "VCT": "Saint Vincent and the Grenadines",
    "WSM": "Samoa",
    "SMR": "San Marino",
    "STP": "Sao Tome and Principe",
    "SAU": "Saudi Arabia",
    "SEN": "Senegal",
    "SRB": "Serbia",
    "SYC": "Seychelles",
    "SLE": "Sierra Leone",
    "SGP": "Singapore",
    "SXM": "Sint Maarten (Dutch part)",
    "SVK": "Slovakia",
    "SVN": "Slovenia",
    "SLB": "Solomon Islands",
    "SOM": "Somalia",
    "ZAF": "South Africa",
    "SGS": "South Georgia and the South Sandwich Islands",
    "SSD": "South Sudan",
    "ESP": "Spain",
    "LKA": "Sri Lanka",
    "SDN": "Sudan",
    "SUR": "Suriname",
    "SJM": "Svalbard and Jan Mayen",
    "SWZ": "Swaziland",
    "SWE": "Sweden",
    "CHE": "Switzerland",
    "SYR": "Syrian Arab Republic",
    "TWN": "Taiwan Province of China",
    "TJK": "Tajikistan",
    "TZA": "Tanzania, United Republic of",
    "THA": "Thailand",
    "TLS": "Timor-Leste",
    "TGO": "Togo",
    "TKL": "Tokelau",
    "TON": "Tonga",
    "TTO": "Trinidad and Tobago",
    "TUN": "Tunisia",
    "TUR": "Turkey",
    "TKM": "Turkmenistan",
    "TCA": "Turks and Caicos Islands",
    "TUV": "Tuvalu",
    "UGA": "Uganda",
    "UKR": "Ukraine",
    "ARE": "United Arab Emirates",
    "GBR": "United Kingdom - British Citizen",
    "GBD": "United Kingdom - British Overseas Territories Citizen",
    "GBN": "United Kingdom - British National (Overseas)",
    "GBO": "United Kingdom - British Overseas Citizen",
    "GBP": "United Kingdom - British Protected person",
    "USA": "United States",
    "UMI": "United States Minor Outlying Islands",
    "URY": "Uruguay",
    "UZB": "Uzbekistan",
    "VUT": "Vanuatu",
    "VEN": "Venezuela, Bolivarian Republic of",
    "VNM": "Viet Nam",
    "VGB": "Virgin Islands (British)",
    "VIR": "Virgin Islands (U.S.)",
    "WLF": "Wallis and Futuna",
    "ESH": "Western Sahara",
    "YEM": "Yemen",
    "ZMB": "Zambia",
    "ZWE": "Zimbabwe",
    "EUE": "European Union (EU)",
    "UNO": "United Nations Organization or one of its officials",
    "UNA": "United Nations specialized agency or one of its officials",
    "UNK": "Resident of Kosovo to whom a travel document has been issued by the United Nations Interim Administration Mission in Kosovo (UNMIK)",
    "XBA": "African Development Bank (ADB)",
    "XIM": "African Export-Import Bank (AFREXIM bank)",
    "XCC": "Caribbean Community or one of its emissaries (CARICOM)",
    "XCO": "Common Market for Eastern and Southern Africa (COMESA)",
    "XEC": "Economic Community of West African States (ECOWAS)",
    "XPO": "International Criminal Police Organization (INTERPOL)",
    "XOM": "Sovereign Military Order of Malta or one of its emissaries",
    "XXA": "Stateless person, as defined in Article 1 of the 1954 Convention Relating to the Status of Stateless Persons",
    "XXB": "Refugee, as defined in Article 1 of the 1951 Convention Relating to the Status of Refugees as amended by the 1967 Protocol",
    "XXC": "Refugee, other than as defined under the code XXB above",
    "XXX": "Person of unspecified nationality, for whom issuing State does not consider it necessary to specify any of the codes XXA, XXB or XXC above, whatever that person’s status may be. This category may include a person who is neither stateless nor a refugee but who is of unknown nationality and legally residing in the State of issue."
  };
  Object.freeze(states);
  var states_1 = states;

  function checkLines(lines) {
    if (typeof lines === 'string') {
      lines = lines.split(/[\r\n]+/);
    }

    if (!Array.isArray(lines)) {
      throw new TypeError('input must be an array or string');
    }

    for (const line of lines) {
      if (!line.match(/[A-Z0-9<]+/)) {
        throw new TypeError('lines must be composed of only alphanumerical characters and "<"');
      }
    }

    return lines;
  }

  var checkLines_1 = checkLines;

  var cleanText = function cleanText(string) {
    return string.replace(/<+$/g, '').replace(/</g, ' ');
  };

  var parseAlpha = function parseAlpha(source) {
    if (!source.match(/^[A-Z<]+$/)) {
      throw new Error(`invalid text: ${source}. Must be only alphabetical with <`);
    }

    return cleanText(source);
  };

  var parseDocumentCodeId = function parseDocumentCodeId(source) {
    const first = source.charAt(0);

    if (first !== 'A' && first !== 'C' && first !== 'I') {
      throw new Error(`invalid document code: ${source}. First character must be A, C or I`);
    }

    const second = source.charAt(1);

    if (second === 'V') {
      throw new Error(`invalid document code: ${source}. Second character may not be V`);
    }

    if (second === '<') {
      return {
        value: first,
        start: 0,
        end: 1
      };
    } else {
      return source;
    }
  };

  var parseText = function parseText(source, regexp = /^[0-9A-Z<]+$/) {
    if (!source.match(regexp)) {
      throw new Error(`invalid text: ${source}. Must match the following regular expression: ${regexp}`);
    }

    return cleanText(source);
  };

  var parseOptional = function parseOptional(source) {
    const value = parseText(source);
    return {
      value,
      start: 0,
      end: 0 + value.length
    };
  };

  var createFieldParser = function (fieldOptions) {
    checkType(fieldOptions, 'label', 'string');

    if (fieldOptions.field !== null) {
      checkType(fieldOptions, 'field', 'string');
    }

    checkType(fieldOptions, 'line', 'number');
    checkType(fieldOptions, 'start', 'number');
    checkType(fieldOptions, 'end', 'number');
    checkType(fieldOptions, 'parser', 'function');
    const ranges = [{
      line: fieldOptions.line,
      start: fieldOptions.start,
      end: fieldOptions.end
    }];

    if (Array.isArray(fieldOptions.related)) {
      for (const related of fieldOptions.related) {
        checkType(related, 'line', 'number');
        checkType(related, 'start', 'number');
        checkType(related, 'end', 'number');
        ranges.push(related);
      }
    }

    return function parseField(lines) {
      const source = getText(lines, fieldOptions);
      let related = fieldOptions.related || [];
      related = related.map(r => getText(lines, r));
      const result = {
        label: fieldOptions.label,
        field: fieldOptions.field,
        value: null,
        valid: false,
        ranges: ranges.map(range => ({
          line: range.line,
          start: range.start,
          end: range.end,
          raw: getText(lines, range)
        }))
      };
      const range = result.ranges[0];
      result.line = range.line;
      result.start = range.start;
      result.end = range.end;

      try {
        let parsed = fieldOptions.parser(source, ...related);
        result.value = typeof parsed === 'object' ? parsed.value : parsed;
        result.valid = true;

        if (typeof parsed === 'object') {
          result.start = range.start + parsed.start;
          result.end = range.start + parsed.end;
        }
      } catch (e) {
        result.error = e.message;
      }

      return result;
    };
  };

  function getText(lines, options) {
    const line = lines[options.line];
    return line.substring(options.start, options.end);
  }

  function checkType(options, name, type) {
    if (typeof options[name] !== type) {
      throw new TypeError(`${name} must be a ${type}`);
    }
  }

  var parseDocumentNumber$1 = function parseDocumentNumber(source, checkDigit, optional) {
    let end, value;

    if (checkDigit === '<' && optional) {
      const firstFiller = optional.indexOf('<');
      const tail = optional.substring(0, firstFiller - 1);
      value = source + tail;
      end = value.length + 1;
    } else {
      value = cleanText(source);
      end = value.length;
    }

    return {
      value,
      start: 0,
      end
    };
  };

  var check = function check(string, value) {
    let code = 0;
    let factors = [7, 3, 1];

    for (let i = 0; i < string.length; i++) {
      let charCode = string.charCodeAt(i);
      if (charCode === 60) charCode = 0;
      if (charCode >= 65) charCode -= 55;
      if (charCode >= 48) charCode -= 48;
      charCode *= factors[i % 3];
      code += charCode;
    }

    code %= 10;

    if (code !== Number(value)) {
      throw new Error(`invalid check digit: ${value}. Must be ${code}`);
    }
  };

  var parseDocumentNumberCheckDigit = function parseDocumentNumberCheckDigit(checkDigit, source, optional) {
    if (checkDigit === '<' && optional) {
      const firstFiller = optional.indexOf('<');
      const tail = optional.substring(0, firstFiller - 1);
      source = `${source}<${tail}`;
      checkDigit = optional.charAt(firstFiller - 1);
      check(source, checkDigit);
      return {
        value: checkDigit,
        start: firstFiller,
        end: firstFiller + 1
      };
    } else {
      check(source, checkDigit);
      return checkDigit;
    }
  };

  var parseState = function parseState(source) {
    source = cleanText(source);
    let state = states_1[source];

    if (!state) {
      throw new Error(`invalid state code: ${source}`);
    }

    return {
      value: source,
      start: 0,
      end: source.length
    };
  };

  var parseSex = function parseSex(source) {
    switch (source) {
      case 'M':
        return 'male';

      case 'F':
        return 'female';

      case '<':
        return 'nonspecified';

      default:
        throw new Error(`invalid sex: ${source}. Must be M, F or <.`);
    }
  };

  var parseDate = function parseDate(value) {
    if (!value.match(/^[0-9<]{4,6}$/)) {
      throw new Error(`invalid date: ${value}`);
    }

    const month = value.substring(2, 4);

    if (month !== '<<' && (month < 1 || month > 12)) {
      throw new Error(`invalid date month: ${month}`);
    }

    if (value.length === 6) {
      const day = value.substring(4, 6);

      if (day !== '<<' && (day < 1 || day > 31)) {
        throw new Error(`invalid date day: ${day}`);
      }
    }

    return value;
  };

  var parseDateCheckDigit = function parseCheckDigit(checkDigit, value) {
    if (checkDigit !== false) {
      check(value, checkDigit);
    }

    return checkDigit;
  };

  var parseCompositeCheckDigit = function parseCompositeCheckDigit(checkDigit, ...sources) {
    const source = sources.join('');
    check(source, checkDigit);
    return checkDigit;
  };

  var parseFirstName = function parseFirstName(source) {
    const withoutStart = source.replace(/.*?<{2}/, '');
    const value = parseText(withoutStart, /^[A-Z<]+<*$/);
    const start = source.length - withoutStart.length;
    return {
      value,
      start,
      end: start + value.length
    };
  };

  var parseLastName = function parseLastName(source) {
    const parsed = parseText(source.replace(/<{2}.*/, ''), /^[A-Z<]*<*$/);
    return {
      value: parsed,
      start: 0,
      end: parsed.length
    };
  };

  const documentNumberTemplate$5 = {
    label: 'Document number',
    field: 'documentNumber',
    parser: parseDocumentNumber$1
  };
  const documentNumberCheckDigitTemplate$4 = {
    label: 'Document number check digit',
    field: 'documentNumberCheckDigit',
    parser: parseDocumentNumberCheckDigit
  };
  const documentCodeTemplate$5 = {
    label: 'Document code',
    field: 'documentCode'
  };
  const nationalityTemplate$3 = {
    label: 'Nationality',
    field: 'nationality',
    parser: parseState
  };
  const sexTemplate$4 = {
    label: 'Sex',
    field: 'sex',
    parser: parseSex
  };
  const expirationDateTemplate$3 = {
    label: 'Expiration date',
    field: 'expirationDate',
    parser: parseDate
  };
  const expirationDateCheckDigitTemplate$3 = {
    label: 'Expiration date check digit',
    field: 'expirationDateCheckDigit',
    parser: parseDateCheckDigit
  };
  const compositeCheckDigitTemplate$4 = {
    label: 'Composite check digit',
    field: 'compositeCheckDigit',
    parser: parseCompositeCheckDigit
  };
  const birthDateTemplate$5 = {
    label: 'Birth date',
    field: 'birthDate',
    parser: parseDate
  };
  const birthDateCheckDigitTemplate$4 = {
    label: 'Birth date check digit',
    field: 'birthDateCheckDigit',
    parser: parseDateCheckDigit
  };
  const issueDateTemplate$1 = {
    label: 'Issue date',
    field: 'issueDate',
    parser: parseDate
  };
  const firstNameTemplate$5 = {
    label: 'First name',
    field: 'firstName',
    parser: parseFirstName
  };
  const lastNameTemplate$5 = {
    label: 'Last name',
    field: 'lastName',
    parser: parseLastName
  };
  const issuingStateTemplate$5 = {
    label: 'Issuing state',
    field: 'issuingState',
    parser: parseState
  };
  var fieldTemplates = {
    documentNumberTemplate: documentNumberTemplate$5,
    documentNumberCheckDigitTemplate: documentNumberCheckDigitTemplate$4,
    documentCodeTemplate: documentCodeTemplate$5,
    nationalityTemplate: nationalityTemplate$3,
    sexTemplate: sexTemplate$4,
    expirationDateTemplate: expirationDateTemplate$3,
    expirationDateCheckDigitTemplate: expirationDateCheckDigitTemplate$3,
    birthDateTemplate: birthDateTemplate$5,
    birthDateCheckDigitTemplate: birthDateCheckDigitTemplate$4,
    issueDateTemplate: issueDateTemplate$1,
    compositeCheckDigitTemplate: compositeCheckDigitTemplate$4,
    firstNameTemplate: firstNameTemplate$5,
    lastNameTemplate: lastNameTemplate$5,
    issuingStateTemplate: issuingStateTemplate$5
  };

  const {
    documentCodeTemplate: documentCodeTemplate$4,
    issuingStateTemplate: issuingStateTemplate$4,
    lastNameTemplate: lastNameTemplate$4,
    issueDateTemplate,
    firstNameTemplate: firstNameTemplate$4,
    documentNumberTemplate: documentNumberTemplate$4,
    documentNumberCheckDigitTemplate: documentNumberCheckDigitTemplate$3,
    birthDateTemplate: birthDateTemplate$4,
    birthDateCheckDigitTemplate: birthDateCheckDigitTemplate$3,
    sexTemplate: sexTemplate$3,
    compositeCheckDigitTemplate: compositeCheckDigitTemplate$3
  } = fieldTemplates;
  var frenchNationalIdFields = [Object.assign({}, documentCodeTemplate$4, {
    line: 0,
    start: 0,
    end: 2,
    parser: parseDocumentCodeId
  }), Object.assign({}, issuingStateTemplate$4, {
    line: 0,
    start: 2,
    end: 5
  }), Object.assign({}, lastNameTemplate$4, {
    line: 0,
    start: 5,
    end: 30,
    parser: parseAlpha
  }), {
    label: 'Administrative code',
    field: 'administrativeCode',
    line: 0,
    start: 30,
    end: 36,
    parser: parseOptional
  }, Object.assign({}, issueDateTemplate, {
    line: 1,
    start: 0,
    end: 4
  }), {
    label: 'Administrative code 2',
    field: 'administrativeCode2',
    line: 1,
    start: 4,
    end: 7,
    parser: parseOptional
  }, Object.assign({}, documentNumberTemplate$4, {
    line: 1,
    start: 7,
    end: 12
  }), Object.assign({}, documentNumberCheckDigitTemplate$3, {
    line: 1,
    start: 12,
    end: 13,
    related: [{
      line: 1,
      start: 0,
      end: 12
    }]
  }), Object.assign({}, firstNameTemplate$4, {
    line: 1,
    start: 13,
    end: 27,
    parser: parseAlpha
  }), Object.assign({}, birthDateTemplate$4, {
    line: 1,
    start: 27,
    end: 33
  }), Object.assign({}, birthDateCheckDigitTemplate$3, {
    line: 1,
    start: 33,
    end: 34,
    related: [{
      line: 1,
      start: 27,
      end: 33
    }]
  }), Object.assign({}, sexTemplate$3, {
    line: 1,
    start: 34,
    end: 35
  }), Object.assign({}, compositeCheckDigitTemplate$3, {
    line: 1,
    start: 35,
    end: 36,
    related: [{
      line: 0,
      start: 0,
      end: 36
    }, {
      line: 1,
      start: 0,
      end: 35
    }]
  })].map(createFieldParser);

  function getDetails(lines, fieldParsers) {
    const details = [];

    for (const parser of fieldParsers) {
      details.push(parser(lines));
    }

    return details;
  }

  function getFields(details) {
    const fields = {};
    let valid = true;

    for (const detail of details) {
      if (!detail.valid) valid = false;

      if (detail.field) {
        fields[detail.field] = detail.value;
      }
    }

    return {
      fields,
      valid
    };
  }

  function getResult(format, lines, fieldParsers) {
    const details = getDetails(lines, fieldParsers);
    const fields = getFields(details);
    const result = {
      format,
      details,
      fields: fields.fields,
      valid: fields.valid
    };
    return result;
  }

  var getResult_1 = getResult;

  const {
    FRENCH_NATIONAL_ID
  } = formats_1;

  var frenchNationalId = function parseFrenchNationalId(lines) {
    lines = checkLines_1(lines);

    if (lines.length !== 2) {
      throw new Error(`invalid number of lines: ${lines.length}: Must be 2 for ${FRENCH_NATIONAL_ID}`);
    }

    lines.forEach((line, index) => {
      if (line.length !== 36) {
        throw new Error(`invalid number of characters for line ${index + 1}: ${line.length}. Must be 36 for ${FRENCH_NATIONAL_ID}`);
      }
    });
    return getResult_1(FRENCH_NATIONAL_ID, lines, frenchNationalIdFields);
  };

  var parseNumber = function parseNumber(source) {
    if (!source.match(/^[0-9]+$/)) {
      throw new Error(`invalid number: ${source}`);
    }

    return source;
  };

  var checkSeparator = function checkSeparator(source) {
    if (!source.match(/^<*$/)) {
      throw new Error(`invalid separator: ${source}. Must be composed only of "<"`);
    }

    return source;
  };

  var parseDocumentCode = function parseDocumentCode(source) {
    if (source !== 'FA') {
      throw new Error(`invalid document code: ${source}. Must be FA`);
    }

    return source;
  };

  var parseLanguageCode = function parseLanguageCode(languageCode) {
    switch (languageCode) {
      case 'D':
      case 'F':
      case 'I':
      case 'R':
        return languageCode;

      default:
        throw new Error(`invalid languageCode code: ${languageCode}. Must be D, F, I or R`);
    }
  };

  var parseDocumentNumber = function parseDocumentNumber(source) {
    // swiss driving license number
    let first = source.substring(0, 3);
    let second = source.substring(3, 6);
    let languageCode = source.charAt(6);
    let end = source.substring(7);

    if (!first.match(/^[A-Z0-9]{3}$/)) {
      throw new Error(`invalid document number: ${source}. Must start with three alphanumeric digits`);
    }

    if (!second.match(/^[0-9]{3}$/)) {
      throw new Error(`invalid document number: ${source}. Must have numeric digits in positions 4, 5 and 6`);
    }

    if (end !== '<<') {
      throw new Error(`invalid document number: ${source}. Must end with <<`);
    } // calling this method to throw if languageCode invalid


    parseLanguageCode(languageCode);
    return {
      value: source.substring(0, 7),
      start: 0,
      end: 7
    };
  };

  var parseIssuingState = function parseIssuingState(source) {
    if (source !== 'CHE' && source !== 'LIE') {
      throw new Error(`invalid state code: ${source}. Must be CHE or LIE`);
    }

    return source;
  };

  const {
    documentNumberTemplate: documentNumberTemplate$3,
    documentCodeTemplate: documentCodeTemplate$3,
    issuingStateTemplate: issuingStateTemplate$3,
    birthDateTemplate: birthDateTemplate$3,
    lastNameTemplate: lastNameTemplate$3,
    firstNameTemplate: firstNameTemplate$3
  } = fieldTemplates;
  var swissDrivingLicenseFields = [Object.assign({}, documentNumberTemplate$3, {
    line: 0,
    start: 0,
    end: 9,
    parser: parseDocumentNumber
  }), {
    label: 'Language code',
    field: 'languageCode',
    line: 0,
    start: 6,
    end: 7,
    parser: parseLanguageCode
  }, Object.assign({}, documentCodeTemplate$3, {
    line: 1,
    start: 0,
    end: 2,
    parser: parseDocumentCode
  }), Object.assign({}, issuingStateTemplate$3, {
    line: 1,
    start: 2,
    end: 5,
    parser: parseIssuingState
  }), {
    label: 'PIN code',
    field: 'pinCode',
    line: 1,
    start: 5,
    end: 14,
    parser: parseNumber
  }, {
    label: 'Version number',
    field: 'versionNumber',
    line: 1,
    start: 14,
    end: 17,
    parser: parseNumber
  }, {
    label: 'Separator 1',
    field: null,
    line: 1,
    start: 17,
    end: 19,
    parser: checkSeparator
  }, Object.assign({}, birthDateTemplate$3, {
    line: 1,
    start: 19,
    end: 25
  }), {
    label: 'Separator 2',
    field: null,
    line: 1,
    start: 25,
    end: 30,
    parser: checkSeparator
  }, Object.assign({}, lastNameTemplate$3, {
    line: 2,
    start: 0,
    end: 30
  }), Object.assign({}, firstNameTemplate$3, {
    line: 2,
    start: 0,
    end: 30
  })].map(createFieldParser);

  const {
    SWISS_DRIVING_LICENSE
  } = formats_1;

  var swissDrivingLicense = function parseSwissDrivingLicense(lines) {
    lines = checkLines_1(lines);

    if (lines.length !== 3) {
      throw new Error(`invalid number of lines: ${lines.length}: Must be 3 for ${SWISS_DRIVING_LICENSE}`);
    }

    if (lines[0].length !== 9) {
      throw new Error(`invalid number of characters for line 1: ${lines[0].length}. Must be 9 for ${SWISS_DRIVING_LICENSE}`);
    }

    if (lines[1].length !== 30) {
      throw new Error(`invalid number of characters for line 2: ${lines[1].length}. Must be 30 for ${SWISS_DRIVING_LICENSE}`);
    }

    if (lines[2].length !== 30) {
      throw new Error(`invalid number of characters for line 3: ${lines[2].length}. Must be 30 for ${SWISS_DRIVING_LICENSE}`);
    }

    return getResult_1(SWISS_DRIVING_LICENSE, lines, swissDrivingLicenseFields);
  };

  var parseDocumentNumberOptional = function parseDocumentNumberOptional(optional, documentNumber, checkDigit) {
    if (checkDigit === '<') {
      const firstFiller = optional.indexOf('<');
      const value = parseText(optional.substring(firstFiller + 1));
      return {
        value,
        start: firstFiller + 1,
        end: firstFiller + 1 + value.length
      };
    } else {
      const value = parseText(optional);
      return {
        value,
        start: 0,
        end: 0 + value.length
      };
    }
  };

  const {
    documentCodeTemplate: documentCodeTemplate$2,
    issuingStateTemplate: issuingStateTemplate$2,
    documentNumberTemplate: documentNumberTemplate$2,
    documentNumberCheckDigitTemplate: documentNumberCheckDigitTemplate$2,
    birthDateTemplate: birthDateTemplate$2,
    birthDateCheckDigitTemplate: birthDateCheckDigitTemplate$2,
    sexTemplate: sexTemplate$2,
    expirationDateTemplate: expirationDateTemplate$2,
    expirationDateCheckDigitTemplate: expirationDateCheckDigitTemplate$2,
    nationalityTemplate: nationalityTemplate$2,
    compositeCheckDigitTemplate: compositeCheckDigitTemplate$2,
    lastNameTemplate: lastNameTemplate$2,
    firstNameTemplate: firstNameTemplate$2
  } = fieldTemplates;
  var td1Fields = [Object.assign({}, documentCodeTemplate$2, {
    line: 0,
    start: 0,
    end: 2,
    parser: parseDocumentCodeId
  }), Object.assign({}, issuingStateTemplate$2, {
    line: 0,
    start: 2,
    end: 5
  }), Object.assign({}, documentNumberTemplate$2, {
    line: 0,
    start: 5,
    end: 14,
    related: [{
      line: 0,
      start: 14,
      end: 15
    }, {
      line: 0,
      start: 15,
      end: 30
    }]
  }), Object.assign(documentNumberCheckDigitTemplate$2, {
    line: 0,
    start: 14,
    end: 15,
    related: [{
      line: 0,
      start: 5,
      end: 14
    }, {
      line: 0,
      start: 15,
      end: 30
    }]
  }), {
    label: 'Optional field 1',
    field: 'optional1',
    line: 0,
    start: 15,
    end: 30,
    related: [{
      line: 0,
      start: 5,
      end: 14
    }, {
      line: 0,
      start: 14,
      end: 15
    }],
    parser: parseDocumentNumberOptional
  }, Object.assign({}, birthDateTemplate$2, {
    start: 0,
    end: 6,
    line: 1
  }), Object.assign({}, birthDateCheckDigitTemplate$2, {
    line: 1,
    start: 6,
    end: 7,
    related: [{
      line: 1,
      start: 0,
      end: 6
    }]
  }), Object.assign({}, sexTemplate$2, {
    line: 1,
    start: 7,
    end: 8
  }), Object.assign({}, expirationDateTemplate$2, {
    line: 1,
    start: 8,
    end: 14
  }), Object.assign({}, expirationDateCheckDigitTemplate$2, {
    line: 1,
    start: 14,
    end: 15,
    related: [{
      line: 1,
      start: 8,
      end: 14
    }]
  }), Object.assign({}, nationalityTemplate$2, {
    line: 1,
    start: 15,
    end: 18
  }), {
    label: 'Optional field 2',
    field: 'optional2',
    line: 1,
    start: 18,
    end: 29,
    parser: parseOptional
  }, Object.assign({}, compositeCheckDigitTemplate$2, {
    line: 1,
    start: 29,
    end: 30,
    related: [{
      line: 0,
      start: 5,
      end: 30
    }, {
      line: 1,
      start: 0,
      end: 7
    }, {
      line: 1,
      start: 8,
      end: 15
    }, {
      line: 1,
      start: 18,
      end: 29
    }]
  }), Object.assign({}, lastNameTemplate$2, {
    line: 2,
    start: 0,
    end: 30
  }), Object.assign({}, firstNameTemplate$2, {
    line: 2,
    start: 0,
    end: 30
  })].map(createFieldParser);

  const {
    TD1
  } = formats_1;

  var td1 = function parseTD1(lines) {
    lines = checkLines_1(lines);

    if (lines.length !== 3) {
      throw new Error(`invalid number of lines: ${lines.length}: Must be 3 for ${TD1}`);
    }

    lines.forEach((line, index) => {
      if (line.length !== 30) {
        throw new Error(`invalid number of characters for line ${index + 1}: ${line.length}. Must be 30 for ${TD1}`);
      }
    });
    return getResult_1(TD1, lines, td1Fields);
  };

  const {
    documentCodeTemplate: documentCodeTemplate$1,
    issuingStateTemplate: issuingStateTemplate$1,
    firstNameTemplate: firstNameTemplate$1,
    lastNameTemplate: lastNameTemplate$1,
    documentNumberTemplate: documentNumberTemplate$1,
    documentNumberCheckDigitTemplate: documentNumberCheckDigitTemplate$1,
    nationalityTemplate: nationalityTemplate$1,
    birthDateTemplate: birthDateTemplate$1,
    birthDateCheckDigitTemplate: birthDateCheckDigitTemplate$1,
    sexTemplate: sexTemplate$1,
    expirationDateTemplate: expirationDateTemplate$1,
    expirationDateCheckDigitTemplate: expirationDateCheckDigitTemplate$1,
    compositeCheckDigitTemplate: compositeCheckDigitTemplate$1
  } = fieldTemplates;
  var td2Fields = [Object.assign({}, documentCodeTemplate$1, {
    line: 0,
    start: 0,
    end: 2,
    parser: parseDocumentCodeId
  }), Object.assign({}, issuingStateTemplate$1, {
    line: 0,
    start: 2,
    end: 5
  }), Object.assign({}, lastNameTemplate$1, {
    line: 0,
    start: 5,
    end: 36
  }), Object.assign({}, firstNameTemplate$1, {
    line: 0,
    start: 5,
    end: 36
  }), Object.assign({}, documentNumberTemplate$1, {
    line: 1,
    start: 0,
    end: 9,
    related: [{
      line: 1,
      start: 9,
      end: 10
    }, {
      line: 1,
      start: 28,
      end: 35
    }]
  }), Object.assign({}, documentNumberCheckDigitTemplate$1, {
    line: 1,
    start: 9,
    end: 10,
    related: [{
      line: 1,
      start: 0,
      end: 9
    }, {
      line: 1,
      start: 28,
      end: 35
    }]
  }), Object.assign({}, nationalityTemplate$1, {
    line: 1,
    start: 10,
    end: 13
  }), Object.assign({}, birthDateTemplate$1, {
    line: 1,
    start: 13,
    end: 19
  }), Object.assign({}, birthDateCheckDigitTemplate$1, {
    line: 1,
    start: 19,
    end: 20,
    related: [{
      line: 1,
      start: 13,
      end: 19
    }]
  }), Object.assign({}, sexTemplate$1, {
    line: 1,
    start: 20,
    end: 21
  }), Object.assign({}, expirationDateTemplate$1, {
    line: 1,
    start: 21,
    end: 27
  }), Object.assign({}, expirationDateCheckDigitTemplate$1, {
    line: 1,
    start: 27,
    end: 28,
    related: [{
      line: 1,
      start: 21,
      end: 27
    }]
  }), {
    label: 'Optional field',
    field: 'optional',
    line: 1,
    start: 28,
    end: 35,
    parser: parseOptional
  }, Object.assign({}, compositeCheckDigitTemplate$1, {
    line: 1,
    start: 35,
    end: 36,
    related: [{
      line: 1,
      start: 0,
      end: 10
    }, {
      line: 1,
      start: 13,
      end: 20
    }, {
      line: 1,
      start: 21,
      end: 35
    }]
  })].map(createFieldParser);

  const {
    TD2
  } = formats_1;

  var td2 = function parseTD2(lines) {
    lines = checkLines_1(lines);

    if (lines.length !== 2) {
      throw new Error(`invalid number of lines: ${lines.length}: Must be 2 for ${TD2}`);
    }

    lines.forEach((line, index) => {
      if (line.length !== 36) {
        throw new Error(`invalid number of characters for line ${index + 1}: ${line.length}. Must be 36 for TD2`);
      }
    });
    return getResult_1(TD2, lines, td2Fields);
  };

  var parseDocumentCodePassport = function parseDocumentCodePassport(source) {
    const first = source.charAt(0);

    if (first !== 'P') {
      throw new Error(`invalid document code: ${source}. First character must be P`);
    }

    const second = source.charAt(1);

    if (!second.match(/[A-Z<]/)) {
      throw new Error(`invalid document code: ${source}. Second character must be a letter or <`);
    }

    if (second === '<') {
      return {
        value: first,
        start: 0,
        end: 1
      };
    } else {
      return source;
    }
  };

  var parsePersonalNumber = function parsePersonalNumber(source) {
    const value = parseText(source, /^[A-Z0-9<]+<*$/);
    return {
      value,
      start: 0,
      end: value.length
    };
  };

  var parsePersonalNumberCheckDigit = function parsePersonalNumberCheckDigit(checkDigit, personalNumber) {
    const cleanNumber = cleanText(personalNumber);

    if (cleanNumber === '') {
      if (checkDigit !== '<' && checkDigit !== '0') {
        throw new Error(`invalid check digit ${checkDigit}: must be 0 or <`);
      } else {
        return checkDigit;
      }
    }

    check(personalNumber, checkDigit);
    return checkDigit;
  };

  const {
    documentCodeTemplate,
    issuingStateTemplate,
    lastNameTemplate,
    firstNameTemplate,
    documentNumberTemplate,
    documentNumberCheckDigitTemplate,
    nationalityTemplate,
    birthDateTemplate,
    birthDateCheckDigitTemplate,
    sexTemplate,
    expirationDateTemplate,
    expirationDateCheckDigitTemplate,
    compositeCheckDigitTemplate
  } = fieldTemplates;
  var td3Fields = [Object.assign({}, documentCodeTemplate, {
    line: 0,
    start: 0,
    end: 2,
    parser: parseDocumentCodePassport
  }), Object.assign({}, issuingStateTemplate, {
    line: 0,
    start: 2,
    end: 5
  }), Object.assign({}, lastNameTemplate, {
    line: 0,
    start: 5,
    end: 44
  }), Object.assign({}, firstNameTemplate, {
    line: 0,
    start: 5,
    end: 44
  }), Object.assign({}, documentNumberTemplate, {
    line: 1,
    start: 0,
    end: 9
  }), Object.assign({}, documentNumberCheckDigitTemplate, {
    line: 1,
    start: 9,
    end: 10,
    related: [{
      line: 1,
      start: 0,
      end: 9
    }]
  }), Object.assign({}, nationalityTemplate, {
    line: 1,
    start: 10,
    end: 13
  }), Object.assign({}, birthDateTemplate, {
    line: 1,
    start: 13,
    end: 19
  }), Object.assign({}, birthDateCheckDigitTemplate, {
    line: 1,
    start: 19,
    end: 20,
    related: [{
      line: 1,
      start: 13,
      end: 19
    }]
  }), Object.assign({}, sexTemplate, {
    line: 1,
    start: 20,
    end: 21
  }), Object.assign({}, expirationDateTemplate, {
    line: 1,
    start: 21,
    end: 27
  }), Object.assign({}, expirationDateCheckDigitTemplate, {
    line: 1,
    start: 27,
    end: 28,
    related: [{
      line: 1,
      start: 21,
      end: 27
    }]
  }), {
    label: 'Personal number',
    field: 'personalNumber',
    line: 1,
    start: 28,
    end: 42,
    parser: parsePersonalNumber
  }, {
    label: 'Personal number check digit',
    field: 'personalNumberCheckDigit',
    line: 1,
    start: 42,
    end: 43,
    related: [{
      line: 1,
      start: 28,
      end: 42
    }],
    parser: parsePersonalNumberCheckDigit
  }, Object.assign({}, compositeCheckDigitTemplate, {
    line: 1,
    start: 43,
    end: 44,
    related: [{
      line: 1,
      start: 0,
      end: 10
    }, {
      line: 1,
      start: 13,
      end: 20
    }, {
      line: 1,
      start: 21,
      end: 43
    }]
  })].map(createFieldParser);

  const {
    TD3
  } = formats_1;

  var td3 = function parseTD3(lines) {
    lines = checkLines_1(lines);

    if (lines.length !== 2) {
      throw new Error(`invalid number of lines: ${lines.length}: Must be 2 for ${TD3}`);
    }

    lines.forEach((line, index) => {
      if (line.length !== 44) {
        throw new Error(`invalid number of characters for line ${index + 1}: ${line.length}. Must be 44 for TD3`);
      }
    });
    return getResult_1(TD3, lines, td3Fields);
  };

  var parsers = {
    TD1: td1,
    TD2: td2,
    TD3: td3,
    SWISS_DRIVING_LICENSE: swissDrivingLicense,
    FRENCH_NATIONAL_ID: frenchNationalId
  };

  function parseMRZ(lines) {
    lines = checkLines_1(lines);

    switch (lines.length) {
      case 2:
      case 3:
        {
          switch (lines[0].length) {
            case 30:
              return parsers.TD1(lines);

            case 36:
              {
                const endLine1 = lines[0].substr(30, 36);

                if (endLine1.match(/[0-9]/)) {
                  return parsers.FRENCH_NATIONAL_ID(lines);
                } else {
                  return parsers.TD2(lines);
                }
              }

            case 44:
              return parsers.TD3(lines);

            case 9:
              return parsers.SWISS_DRIVING_LICENSE(lines);

            default:
              throw new Error('unrecognized document format. First line of input must have 30 (TD1), 36 (TD2 or French National Id), 44 (TD3) or 9 (Swiss Driving License) characters');
          }
        }

      default:
        throw new Error(`unrecognized document format. Input must have two or three lines, found${lines.length}`);
    }
  }

  for (const format in formats_1) {
    parseMRZ[format] = parsers[format];
  }

  var parse = parseMRZ;

  var src = {
    states: states_1,
    formats: formats_1,
    parse
  };

  return src;

})));
//# sourceMappingURL=mrz.js.map
