/**
 * openchemlib-utils - Various utilities that extends openchemlib-js like HOSE codes or diastereotopic IDs
 * @version v1.7.0
 * @link https://github.com/cheminfo/openchemlib-utils#readme
 * @license MIT
 */
(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
  typeof define === 'function' && define.amd ? define(['exports'], factory) :
  (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.OCLUtils = {}));
})(this, (function (exports) { 'use strict';

  let xAtomicNumber = 0;
  /**
   * Tag an atom to be able to visualize it
   * @param {OCL.Molecule} molecule
   * @param {number} iAtom
   */

  function tagAtom(molecule, iAtom) {
    let customLabel = `${molecule.getAtomLabel(iAtom)}*`;
    molecule.setAtomCustomLabel(iAtom, customLabel);

    if (molecule.getAtomicNo(iAtom) === 1) {
      molecule.setAtomicNo(iAtom, getXAtomicNumber(molecule));
    } else {
      // we can not use X because we would have problems with valencies if it is
      // expanded hydrogens or not
      // we can not only use a custom label because it does not count for the canonisation
      molecule.setAtomMass(iAtom, molecule.getAtomMass(iAtom) + 5);
    }

    return customLabel;
  }

  function getXAtomicNumber(molecule) {
    if (!xAtomicNumber) {
      const OCL = molecule.getOCL();
      xAtomicNumber = OCL.Molecule.getAtomicNoFromLabel('X');
    }

    return xAtomicNumber;
  }

  /**
   * Add either missing chirality of diastereotopic missing chirality
   * The problem is that sometimes we need to add chiral bond that was not planned because it is the same group
   * This is the case for example for the valine where the 2 C of the methyl groups are diastereotopic
   * @param {OCL.Molecule} molecule
   * @param {object} [options={}]
   * @param {number} [options.esrType=cESRTypeAnd]
   */

  function addDiastereotopicMissingChirality(molecule) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    const {
      Molecule
    } = molecule.getOCL();
    const {
      esrType = Molecule.cESRTypeAnd
    } = options;

    for (let iAtom = 0; iAtom < molecule.getAllAtoms(); iAtom++) {
      let tempMolecule = molecule.getCompactCopy();
      tagAtom(tempMolecule, iAtom); // After copy, helpers must be recalculated

      tempMolecule.ensureHelperArrays(Molecule.cHelperBitsStereo); // We need to have >0 and not >1 because there could be unspecified chirality in racemate

      for (let i = 0; i < tempMolecule.getAtoms(); i++) {
        // changed from from handling below; TLS 9.Nov.2015
        if (tempMolecule.isAtomStereoCenter(i) && tempMolecule.getStereoBond(i) === -1) {
          let stereoBond = tempMolecule.getAtomPreferredStereoBond(i);

          if (stereoBond !== -1) {
            molecule.setBondType(stereoBond, Molecule.cBondTypeUp);

            if (molecule.getBondAtom(1, stereoBond) === i) {
              let connAtom = molecule.getBondAtom(0, stereoBond);
              molecule.setBondAtom(0, stereoBond, i);
              molecule.setBondAtom(1, stereoBond, connAtom);
            } // To me it seems that we have to add all stereo centers into AND group 0. TLS 9.Nov.2015


            molecule.setAtomESR(i, esrType, 0);
          }
        }
      }
    }
  }

  /**
   *
   * @param {OCL.Molecule} [molecule] An instance of a molecule
   * @param {object} [options={}]
   * @param {object} [options.OCL] openchemlib library
   */
  function makeRacemic(molecule) {
    const {
      Molecule
    } = molecule.getOCL(); // if we don't calculate this we have 2 epimers

    molecule.ensureHelperArrays(Molecule.cHelperCIP); // we need to make one group "AND" for chiral (to force to racemic, this means diastereotopic and not enantiotopic)

    for (let i = 0; i < molecule.getAllAtoms(); i++) {
      if (molecule.getAtomParity(i) !== Molecule.cAtomParityNone) {
        molecule.setAtomESR(i, Molecule.cESRTypeAnd, 0); // changed to group 0; TLS 9.Nov.2015
      }
    }
  }

  /**
   * Returns an array of diastereotopic ID (as oclCode)
   * @param {OCL.Molecule} molecule
   */

  function getDiastereotopicAtomIDs(molecule) {
    const OCL = molecule.getOCL();
    addDiastereotopicMissingChirality(molecule);
    let numberAtoms = molecule.getAllAtoms();
    let ids = [];

    for (let iAtom = 0; iAtom < numberAtoms; iAtom++) {
      let tempMolecule = molecule.getCompactCopy();
      tagAtom(tempMolecule, iAtom);
      makeRacemic(tempMolecule); // We need to ensure the helper array in order to get correctly the result of racemisation

      ids[iAtom] = tempMolecule.getCanonizedIDCode(OCL.Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS);
    }

    return ids;
  }

  /**
   *
   * @param {OCL.Molecule} molecule
   */

  function getDiastereotopicAtomIDsAndH(molecule) {
    const OCL = molecule.getOCL();
    molecule = molecule.getCompactCopy();
    molecule.addImplicitHydrogens(); // TODO Temporary code ???

    molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
    const diaIDs = getDiastereotopicAtomIDs(molecule);
    const newDiaIDs = [];

    for (let i = 0; i < diaIDs.length; i++) {
      const diaID = diaIDs[i];
      const newDiaID = {
        oclID: diaID,
        hydrogenOCLIDs: [],
        nbHydrogens: 0
      };

      if (molecule.getAtomicNo(i) === 1) {
        const atom = molecule.getConnAtom(i, 0);
        newDiaID.heavyAtom = diaIDs[atom];
      }

      for (let j = 0; j < molecule.getAllConnAtoms(i); j++) {
        const atom = molecule.getConnAtom(i, j);

        if (molecule.getAtomicNo(atom) === 1) {
          newDiaID.nbHydrogens++;

          if (newDiaID.hydrogenOCLIDs.indexOf(diaIDs[atom]) === -1) {
            newDiaID.hydrogenOCLIDs.push(diaIDs[atom]);
          }
        }
      }

      newDiaIDs.push(newDiaID);
    }

    return newDiaIDs;
  }

  /**
   * Returns a SVG
   * @param {*} molecule
   * @param {*} [options={}]
   */

  function toDiastereotopicSVG(molecule) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    let {
      width = 300,
      height = 200,
      prefix = 'ocl',
      heavyAtomHydrogen = false
    } = options;
    let svg = options.svg;
    let diaIDs = [];
    let hydrogenInfo = {};
    getDiastereotopicAtomIDsAndH(molecule).forEach(line => {
      hydrogenInfo[line.oclID] = line;
    });

    if (heavyAtomHydrogen) {
      for (let i = 0; i < molecule.getAtoms(); i++) {
        diaIDs.push([]);
      }

      let groupedDiaIDs = molecule.getGroupedDiastereotopicAtomIDs();
      groupedDiaIDs.forEach(diaID => {
        if (hydrogenInfo[diaID.oclID] && hydrogenInfo[diaID.oclID].nbHydrogens > 0) {
          diaID.atoms.forEach(atom => {
            hydrogenInfo[diaID.oclID].hydrogenOCLIDs.forEach(id => {
              if (!diaIDs[atom * 1].includes(id)) diaIDs[atom].push(id);
            });
          });
        }
      });
    } else {
      diaIDs = molecule.getDiastereotopicAtomIDs().map(a => [a]);
    }

    if (!svg) svg = molecule.toSVG(width, height, prefix);
    svg = svg.replace(/Atom:[0-9]+"/g, value => {
      let atom = value.replace(/[^0-9]/g, '');
      return `${value} data-diaid="${diaIDs[atom].join(',')}"`;
    });
    return svg;
  }

  /**
   * This function groups the diasterotopic atomIds of the molecule based on equivalence of atoms. The output object contains
   * a set of chemically equivalent atoms(element.atoms) and the groups of magnetically equivalent atoms (element.magneticGroups)
   * @param {OCL.Molecule} molecule
   * @param {object} [options={}]
   * @param {string} [options.atomLabel] Select atoms of the given atomLabel. By default it returns all the explicit atoms in the molecule
   * @returns {Array}
   */

  function getGroupedDiastereotopicAtomIDs(molecule) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    const {
      atomLabel
    } = options;
    let diaIDs = getDiastereotopicAtomIDs(molecule);
    let diaIDsObject = {};

    for (let i = 0; i < diaIDs.length; i++) {
      if (!atomLabel || molecule.getAtomLabel(i) === atomLabel) {
        let diaID = diaIDs[i];

        if (!diaIDsObject[diaID]) {
          diaIDsObject[diaID] = {
            counter: 0,
            atoms: [],
            oclID: diaID,
            atomLabel: molecule.getAtomLabel(i)
          };
        }

        diaIDsObject[diaID].counter++;
        diaIDsObject[diaID].atoms.push(i);
      }
    }

    return Object.keys(diaIDsObject).map(key => diaIDsObject[key]);
  }

  /**
   * Check if a specific atom is a sp3 carbon
   * @param {OCL.Molecule} molecule
   * @param {number} atomID
   */
  function isCsp3(molecule, atomID) {
    if (molecule.getAtomicNo(atomID) !== 6) return false;
    if (molecule.getAtomCharge(atomID) !== 0) return false;

    if (molecule.getImplicitHydrogens(atomID) + molecule.getConnAtoms(atomID) !== 4) {
      return false;
    }

    return true;
  }

  const FULL_HOSE_CODE = 1;
  const HOSE_CODE_CUT_C_SP3_SP3 = 2;
  /**
   * Returns the hose code for a specific atom number
   * @param {OCL.Molecule} originalMolecule
   * @param {number} rootAtom
   * @param {object} [options={}]
   * @param {boolean} [options.isTagged] Specify is the atom is already tagged
   */

  function getHoseCodesForAtom(originalMolecule, rootAtom) {
    let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
    const OCL = originalMolecule.getOCL();
    const {
      minSphereSize = 0,
      maxSphereSize = 4,
      kind = FULL_HOSE_CODE,
      isTagged = false
    } = options;
    const molecule = originalMolecule.getCompactCopy();

    if (!isTagged) {
      let tag = tagAtom(molecule, rootAtom);
      molecule.addImplicitHydrogens();
      molecule.addMissingChirality();
      molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours); // because ensuring helper reorder atoms we need to look again for it

      for (let i = 0; i < molecule.getAllAtoms(); i++) {
        if (tag === molecule.getAtomCustomLabel(i)) {
          rootAtom = i;
          break;
        }
      }
    }

    let fragment = new OCL.Molecule(0, 0);
    let results = [];
    let min = 0;
    let max = 0;
    let atomMask = new Array(molecule.getAllAtoms());
    let atomList = new Array(molecule.getAllAtoms());

    for (let sphere = 0; sphere <= maxSphereSize; sphere++) {
      if (max === 0) {
        atomList[0] = rootAtom;
        atomMask[rootAtom] = true;
        max = 1;
      } else {
        let newMax = max;

        for (let i = min; i < max; i++) {
          let atom = atomList[i];

          for (let j = 0; j < molecule.getAllConnAtoms(atom); j++) {
            let connAtom = molecule.getConnAtom(atom, j);

            if (!atomMask[connAtom]) {
              switch (kind) {
                case FULL_HOSE_CODE:
                  atomMask[connAtom] = true;
                  atomList[newMax++] = connAtom;
                  break;

                case HOSE_CODE_CUT_C_SP3_SP3:
                  if (!(isCsp3(molecule, atom) && isCsp3(molecule, connAtom))) {
                    atomMask[connAtom] = true;
                    atomList[newMax++] = connAtom;
                  }

                  break;

                default:
                  throw new Error('getHoseCoesForAtom unknown kind');
              }
            }
          }
        }

        min = max;
        max = newMax;
      }

      molecule.copyMoleculeByAtoms(fragment, atomMask, true, null);

      if (sphere >= minSphereSize) {
        makeRacemic(fragment);
        results.push(fragment.getCanonizedIDCode(OCL.Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS));
      }
    }

    return results;
  }

  /**
   * Returns the hose code for a specific marked atom
   * @param {OCL.Molecule} diastereotopicID
   * @param {object} options
   */

  function getHoseCodesFromDiastereotopicID(molecule) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    molecule.addImplicitHydrogens();
    molecule.addMissingChirality(); // One of the atom has to be marked !

    let atomID = -1;

    for (let i = 0; i < molecule.getAllAtoms(); i++) {
      // we need to find the marked atom
      const atomCustomLabel = molecule.getAtomCustomLabel(i);

      if (atomCustomLabel != null && atomCustomLabel.endsWith('*')) {
        atomID = i;
        break;
      }
    }

    if (atomID >= 0) {
      options.isTagged = true;
      return getHoseCodesForAtom(molecule, atomID, options);
    }

    return undefined;
  }

  /**
   * Returns an array containing one entry per atom containing
   * diaID and hose code
   * @param {OCL.Molecule} molecule
   * @param {object} options
   */

  function getHoseCodesAndDiastereotopicIDs(molecule, options) {
    const diaIDs = getDiastereotopicAtomIDs(molecule).map(oclID => ({
      oclID
    }));
    const OCL = molecule.getOCL(); // TODO: seems like a very slow approach

    diaIDs.forEach(diaID => {
      const hoses = getHoseCodesFromDiastereotopicID(OCL.Molecule.fromIDCode(diaID.oclID), options);
      diaID.hoses = [];
      let sphere = 0;

      for (const hose of hoses) {
        diaID.hoses.push({
          sphere: sphere++,
          oclID: hose
        });
      }
    });
    return diaIDs;
  }

  let fragment$1;
  /**
   * Returns the hose code for a specific atom number
   * @param {OCL.Molecule} molecule
   */

  function getHoseCodesForPath(molecule, from, to, maxLength) {
    const OCL = molecule.getOCL();
    const originalFrom = from;
    const originalTo = to;
    molecule = molecule.getCompactCopy();
    let originalAtoms = []; // path before renumbering

    molecule.getPath(originalAtoms, from, to, maxLength + 1);
    let torsion;

    if (originalAtoms.length === 4) {
      torsion = molecule.calculateTorsion(originalAtoms);
    }

    const tag1 = tagAtom(molecule, from);
    const tag2 = tagAtom(molecule, to);
    molecule.addImplicitHydrogens();
    molecule.addMissingChirality();
    molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
    from = -1;
    to = -1;

    for (let i = 0; i < molecule.getAllAtoms(); i++) {
      if (tag1 === tag2) {
        if (molecule.getAtomCustomLabel(i) === tag1) {
          if (from === -1) {
            from = i;
          } else {
            to = i;
          }
        }
      } else {
        if (tag1 === molecule.getAtomCustomLabel(i)) {
          from = i;
        }

        if (tag2 === molecule.getAtomCustomLabel(i)) {
          to = i;
        }
      }
    }

    if (!fragment$1) fragment$1 = new OCL.Molecule(0, 0);
    let atoms = [];
    molecule.getPath(atoms, from, to, maxLength + 1);
    let min = 0;
    let max = 0;
    let atomMask = new Array(molecule.getAllAtoms()).fill(false);
    let atomList = new Array(molecule.getAllAtoms()).fill(-1);
    let hoses = [];

    for (let sphere = 0; sphere <= 2; sphere++) {
      if (max === 0) {
        for (let atom of atoms) {
          atomMask[atom] = true;
          atomList[max++] = atom;
        }
      } else {
        let newMax = max;

        for (let i = min; i < max; i++) {
          let atom = atomList[i];

          for (let j = 0; j < molecule.getAllConnAtoms(atom); j++) {
            let connAtom = molecule.getConnAtom(atom, j);

            if (!atomMask[connAtom]) {
              atomMask[connAtom] = true;
              atomList[newMax++] = connAtom;
            }
          }
        }

        min = max;
        max = newMax;
      }

      let atomMap = [];
      molecule.copyMoleculeByAtoms(fragment$1, atomMask, true, atomMap);
      makeRacemic(fragment$1);
      let oclID = fragment$1.getCanonizedIDCode(OCL.Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS);
      hoses.push({
        sphere,
        oclID
      });
    }

    return {
      atoms: originalAtoms,
      from: originalFrom,
      to: originalTo,
      torsion,
      hoses,
      length: originalAtoms.length - 1
    };
  }

  const MAX_R = 10;
  /**
   * Generate molecules and calcule predicted properties form a list of smiles and fragments
   * @param {string} [coreSmiles]
   * @param {array} [fragments] Array of {smiles,R1,R2,...}
   * @param {OCL} [OCL] The openchemlib library
   * @param {object} [options={}]
   * @param {function} [options.onStep] method to execute each new molecules
   * @param {boolean} [options.complexity] returns only the number of molecules to evaluate
   * @return {Promise} promise that resolves to molecules or complexity as a number
   */

  async function combineSmiles(coreSmiles, fragments, OCL) {
    let options = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : {};
    const {
      complexity = false
    } = options;
    const core = getCore(coreSmiles);
    const rGroups = getRGroups(core, fragments);

    if (complexity) {
      return getComplexity(rGroups);
    }

    return generate(core, rGroups, OCL, options);
  }

  function getComplexity(rGroups) {
    let complexity = 1;

    for (let rGroup of rGroups) {
      complexity *= rGroup.smiles.length;
    }

    return complexity;
  }

  async function generate(core, rGroups, OCL) {
    let options = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : {};
    const {
      onStep
    } = options;
    const molecules = {};
    const sizes = new Array(rGroups.length);
    const currents = new Array(rGroups.length);

    for (let i = 0; i < rGroups.length; i++) {
      sizes[i] = rGroups[i].smiles.length - 1;
      currents[i] = 0;
    }

    let position = 0;
    let counter = 0;

    while (true) {
      counter++;

      while (position < currents.length) {
        if (currents[position] < sizes[position]) {
          if (onStep) {
            await onStep(counter);
          }

          appendMolecule(molecules, core, rGroups, currents, OCL);
          currents[position]++;

          for (let i = 0; i < position; i++) {
            currents[i] = 0;
          }

          position = 0;
        } else {
          position++;
        }
      }

      if (position = currents.length) {
        if (onStep) {
          await onStep(counter);
        }

        appendMolecule(molecules, core, rGroups, currents, OCL);
        break;
      }
    }

    return Object.keys(molecules).map(key => molecules[key]).sort((m1, m2) => m1.mw - m2.mw);
  }

  function appendMolecule(molecules, core, rGroups, currents, OCL) {
    let newSmiles = core.smiles;

    for (let i = 0; i < currents.length; i++) {
      newSmiles += `.${rGroups[i].smiles[currents[i]]}`;
    }

    const currentMol = OCL.Molecule.fromSmiles(newSmiles);
    const idCode = currentMol.getIDCode();

    if (!molecules[idCode]) {
      let molecule = {};
      molecules[idCode] = molecule;
      molecule.smiles = currentMol.toSmiles();
      molecule.combinedSmiles = newSmiles;
      molecule.idCode = idCode;
      molecule.molfile = currentMol.toMolfile();
      const props = new OCL.MoleculeProperties(currentMol);
      molecule.nbHAcceptor = props.acceptorCount;
      molecule.nbHDonor = props.donorCount;
      molecule.logP = props.logP;
      molecule.logS = props.logS;
      molecule.PSA = props.polarSurfaceArea;
      molecule.nbRottable = props.rotatableBondCount;
      molecule.nbStereoCenter = props.stereoCenterCount;
      let mf = currentMol.getMolecularFormula();
      molecule.mf = mf.formula;
      molecule.mw = mf.relativeWeight;
    }
  }

  function getCore(coreSmiles) {
    let core = {
      originalSmiles: coreSmiles,
      smiles: coreSmiles.replace(/\[R(?<group>[1-4])\]/g, '%5$<group>')
    };

    for (let i = 0; i < MAX_R; i++) {
      if (core.originalSmiles.indexOf(`[R${i}]`) > -1) core[`R${i}`] = true;
    }

    return core;
  }

  function getRGroups(core, fragments) {
    let rGroups = {};

    for (const fragment of fragments) {
      if (fragment.smiles) {
        const smiles = updateRPosition(fragment.smiles);

        for (let i = 0; i < MAX_R; i++) {
          if (core[`R${i}`]) {
            // we only consider the R that are in the core
            if (fragment[`R${i}`]) {
              if (!rGroups[`R${i}`]) {
                rGroups[`R${i}`] = {
                  group: `R${i}`,
                  smiles: []
                };
              }

              rGroups[`R${i}`].smiles.push(smiles.replace(/\[R\]/, `(%5${i})`));
            }
          }
        }
      }
    }

    return Object.keys(rGroups).map(key => rGroups[key]);
  }

  function updateRPosition(smiles) {
    // R group should not be at the beginning
    if (smiles.indexOf('[R]') !== 0) return smiles;
    if (smiles.length === 3) return '[H][R]'; // we are in trouble ... we need to move the R

    let newSmiles = smiles.replace('[R]', ''); // we need to check where we should put the R group

    let level = 0;

    for (let j = 0; j < newSmiles.length; j++) {
      let currentChar = newSmiles.charAt(j);
      let currentSubstring = newSmiles.substr(j);

      if (currentChar === '(') {
        level++;
      } else if (currentChar === ')') {
        level--;
      } else if (level === 0) {
        if (currentSubstring.match(/^[a-z]/)) {
          return `${newSmiles.substr(0, j + 1)}([R])${newSmiles.substr(j + 1)}`;
        } else if (currentSubstring.match(/^[A-Z][a-z]/)) {
          return `${newSmiles.substr(0, j + 2)}([R])${newSmiles.substr(j + 2)}`;
        } else if (currentSubstring.match(/^[A-Z]/)) {
          return `${newSmiles.substr(0, j + 1)}([R])${newSmiles.substr(j + 1)}`;
        }
      }
    }

    return smiles;
  }

  /**
   * Returns various information about atoms in the molecule
   * @param {OCL.Molecule} [molecule]
   */

  function getAtomsInfo(molecule) {
    const OCL = molecule.getOCL();
    molecule.ensureHelperArrays(OCL.Molecule.cHelperRings);
    let diaIDs = getDiastereotopicAtomIDs(molecule);
    let results = [];

    for (let i = 0; i < diaIDs.length; i++) {
      let result = {
        oclID: diaIDs[i],
        extra: {
          singleBonds: 0,
          doubleBonds: 0,
          tripleBonds: 0,
          aromaticBonds: 0,
          cnoHybridation: 0 // should be 1 (sp), 2 (sp2) or 3 (sp3)

        }
      };
      let extra = result.extra;
      results.push(result);
      result.abnormalValence = molecule.getAtomAbnormalValence(i); // -1 is normal otherwise specified

      result.charge = molecule.getAtomCharge(i);
      result.cipParity = molecule.getAtomCIPParity(i);
      result.color = molecule.getAtomColor(i);
      result.customLabel = molecule.getAtomCustomLabel(i); //        result.esrGroup=molecule.getAtomESRGroup(i);
      //        result.esrType=molecule.getAtomESRType(i);

      result.atomicNo = molecule.getAtomicNo(i);
      result.label = molecule.getAtomLabel(i); //        result.list=molecule.getAtomList(i);
      //        result.listString=molecule.getAtomListString(i);
      //        result.mapNo=molecule.getAtomMapNo(i);

      result.mass = molecule.getAtomMass(i); //        result.parity=molecule.getAtomParity(i);
      //        result.pi=molecule.getAtomPi(i);
      //        result.preferredStereoBond=molecule.getAtomPreferredStereoBond(i);
      //        result.queryFeatures=molecule.getAtomQueryFeatures(i);

      result.radical = molecule.getAtomRadical(i);
      result.ringBondCount = molecule.getAtomRingBondCount(i); //        result.ringCount=molecule.getAtomRingCount(i);

      result.ringSize = molecule.getAtomRingSize(i);
      result.x = molecule.getAtomX(i);
      result.y = molecule.getAtomY(i);
      result.z = molecule.getAtomZ(i);
      result.allHydrogens = molecule.getAllHydrogens(i);
      result.connAtoms = molecule.getConnAtoms(i);
      result.allConnAtoms = molecule.getAllConnAtoms(i);
      result.implicitHydrogens = result.allHydrogens + result.connAtoms - result.allConnAtoms;
      result.isAromatic = molecule.isAromaticAtom(i);
      result.isAllylic = molecule.isAllylicAtom(i);
      result.isStereoCenter = molecule.isAtomStereoCenter(i);
      result.isRing = molecule.isRingAtom(i);
      result.isSmallRing = molecule.isSmallRingAtom(i);
      result.isStabilized = molecule.isStabilizedAtom(i); // todo HACK to circumvent bug in OCL that consider than an hydrogen is connected to itself

      result.extra.singleBonds = result.atomicNo === 1 ? 0 : result.implicitHydrogens;

      for (let j = 0; j < molecule.getAllConnAtoms(i); j++) {
        let bond = molecule.getConnBond(i, j);
        let bondOrder = molecule.getBondOrder(bond);

        if (molecule.isAromaticBond(bond)) {
          extra.aromaticBonds++;
        } else if (bondOrder === 1) {
          // not an hydrogen
          extra.singleBonds++;
        } else if (bondOrder === 2) {
          extra.doubleBonds++;
        } else if (bondOrder === 3) {
          extra.tripleBonds++;
        }
      }

      result.extra.totalBonds = result.extra.singleBonds + result.extra.doubleBonds + result.extra.tripleBonds + result.extra.aromaticBonds;

      if (result.atomicNo === 6) {
        result.extra.cnoHybridation = result.extra.totalBonds - 1;
      } else if (result.atomicNo === 7) {
        result.extra.cnoHybridation = result.extra.totalBonds;
      } else if (result.atomicNo === 8) {
        result.extra.cnoHybridation = result.extra.totalBonds + 1;
      } else if (result.atomicNo === 1) {
        let connectedAtom = molecule.getAllConnAtoms(i) === 0 ? 0 : molecule.getAtomicNo(molecule.getConnAtom(i, 0));
        result.extra.hydrogenOnAtomicNo = connectedAtom;

        if (connectedAtom === 7 || connectedAtom === 8) {
          result.extra.labileHydrogen = true;
        }
      }
    }

    return results;
  }

  const toString = Object.prototype.toString;
  function isAnyArray(object) {
    return toString.call(object).endsWith('Array]');
  }

  function max(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;

    if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
      throw new Error('fromIndex must be a positive integer smaller than length');
    }

    if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
      throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
    }

    var maxValue = input[fromIndex];

    for (var i = fromIndex + 1; i < toIndex; i++) {
      if (input[i] > maxValue) maxValue = input[i];
    }

    return maxValue;
  }

  function min(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;

    if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
      throw new Error('fromIndex must be a positive integer smaller than length');
    }

    if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
      throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
    }

    var minValue = input[fromIndex];

    for (var i = fromIndex + 1; i < toIndex; i++) {
      if (input[i] < minValue) minValue = input[i];
    }

    return minValue;
  }

  function rescale(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    } else if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    var output;

    if (options.output !== undefined) {
      if (!isAnyArray(options.output)) {
        throw new TypeError('output option must be an array if specified');
      }

      output = options.output;
    } else {
      output = new Array(input.length);
    }

    var currentMin = min(input);
    var currentMax = max(input);

    if (currentMin === currentMax) {
      throw new RangeError('minimum and maximum input values are equal. Cannot rescale a constant array');
    }

    var _options$min = options.min,
        minValue = _options$min === void 0 ? options.autoMinMax ? currentMin : 0 : _options$min,
        _options$max = options.max,
        maxValue = _options$max === void 0 ? options.autoMinMax ? currentMax : 1 : _options$max;

    if (minValue >= maxValue) {
      throw new RangeError('min option must be smaller than max option');
    }

    var factor = (maxValue - minValue) / (currentMax - currentMin);

    for (var i = 0; i < input.length; i++) {
      output[i] = (input[i] - currentMin) * factor + minValue;
    }

    return output;
  }

  const indent = ' '.repeat(2);
  const indentData = ' '.repeat(4);
  function inspectMatrix() {
    return inspectMatrixWithOptions(this);
  }
  function inspectMatrixWithOptions(matrix) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    const {
      maxRows = 15,
      maxColumns = 10,
      maxNumSize = 8
    } = options;
    return `${matrix.constructor.name} {
${indent}[
${indentData}${inspectData(matrix, maxRows, maxColumns, maxNumSize)}
${indent}]
${indent}rows: ${matrix.rows}
${indent}columns: ${matrix.columns}
}`;
  }

  function inspectData(matrix, maxRows, maxColumns, maxNumSize) {
    const {
      rows,
      columns
    } = matrix;
    const maxI = Math.min(rows, maxRows);
    const maxJ = Math.min(columns, maxColumns);
    const result = [];

    for (let i = 0; i < maxI; i++) {
      let line = [];

      for (let j = 0; j < maxJ; j++) {
        line.push(formatNumber(matrix.get(i, j), maxNumSize));
      }

      result.push(`${line.join(' ')}`);
    }

    if (maxJ !== columns) {
      result[result.length - 1] += ` ... ${columns - maxColumns} more columns`;
    }

    if (maxI !== rows) {
      result.push(`... ${rows - maxRows} more rows`);
    }

    return result.join(`\n${indentData}`);
  }

  function formatNumber(num, maxNumSize) {
    const numStr = String(num);

    if (numStr.length <= maxNumSize) {
      return numStr.padEnd(maxNumSize, ' ');
    }

    const precise = num.toPrecision(maxNumSize - 2);

    if (precise.length <= maxNumSize) {
      return precise;
    }

    const exponential = num.toExponential(maxNumSize - 2);
    const eIndex = exponential.indexOf('e');
    const e = exponential.slice(eIndex);
    return exponential.slice(0, maxNumSize - e.length) + e;
  }

  function installMathOperations(AbstractMatrix, Matrix) {
    AbstractMatrix.prototype.add = function add(value) {
      if (typeof value === 'number') return this.addS(value);
      return this.addM(value);
    };

    AbstractMatrix.prototype.addS = function addS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.addM = function addM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.add = function add(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.add(value);
    };

    AbstractMatrix.prototype.sub = function sub(value) {
      if (typeof value === 'number') return this.subS(value);
      return this.subM(value);
    };

    AbstractMatrix.prototype.subS = function subS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.subM = function subM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.sub = function sub(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sub(value);
    };

    AbstractMatrix.prototype.subtract = AbstractMatrix.prototype.sub;
    AbstractMatrix.prototype.subtractS = AbstractMatrix.prototype.subS;
    AbstractMatrix.prototype.subtractM = AbstractMatrix.prototype.subM;
    AbstractMatrix.subtract = AbstractMatrix.sub;

    AbstractMatrix.prototype.mul = function mul(value) {
      if (typeof value === 'number') return this.mulS(value);
      return this.mulM(value);
    };

    AbstractMatrix.prototype.mulS = function mulS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.mulM = function mulM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.mul = function mul(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.mul(value);
    };

    AbstractMatrix.prototype.multiply = AbstractMatrix.prototype.mul;
    AbstractMatrix.prototype.multiplyS = AbstractMatrix.prototype.mulS;
    AbstractMatrix.prototype.multiplyM = AbstractMatrix.prototype.mulM;
    AbstractMatrix.multiply = AbstractMatrix.mul;

    AbstractMatrix.prototype.div = function div(value) {
      if (typeof value === 'number') return this.divS(value);
      return this.divM(value);
    };

    AbstractMatrix.prototype.divS = function divS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.divM = function divM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.div = function div(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.div(value);
    };

    AbstractMatrix.prototype.divide = AbstractMatrix.prototype.div;
    AbstractMatrix.prototype.divideS = AbstractMatrix.prototype.divS;
    AbstractMatrix.prototype.divideM = AbstractMatrix.prototype.divM;
    AbstractMatrix.divide = AbstractMatrix.div;

    AbstractMatrix.prototype.mod = function mod(value) {
      if (typeof value === 'number') return this.modS(value);
      return this.modM(value);
    };

    AbstractMatrix.prototype.modS = function modS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) % value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.modM = function modM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) % matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.mod = function mod(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.mod(value);
    };

    AbstractMatrix.prototype.modulus = AbstractMatrix.prototype.mod;
    AbstractMatrix.prototype.modulusS = AbstractMatrix.prototype.modS;
    AbstractMatrix.prototype.modulusM = AbstractMatrix.prototype.modM;
    AbstractMatrix.modulus = AbstractMatrix.mod;

    AbstractMatrix.prototype.and = function and(value) {
      if (typeof value === 'number') return this.andS(value);
      return this.andM(value);
    };

    AbstractMatrix.prototype.andS = function andS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) & value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.andM = function andM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) & matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.and = function and(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.and(value);
    };

    AbstractMatrix.prototype.or = function or(value) {
      if (typeof value === 'number') return this.orS(value);
      return this.orM(value);
    };

    AbstractMatrix.prototype.orS = function orS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) | value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.orM = function orM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) | matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.or = function or(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.or(value);
    };

    AbstractMatrix.prototype.xor = function xor(value) {
      if (typeof value === 'number') return this.xorS(value);
      return this.xorM(value);
    };

    AbstractMatrix.prototype.xorS = function xorS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) ^ value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.xorM = function xorM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) ^ matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.xor = function xor(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.xor(value);
    };

    AbstractMatrix.prototype.leftShift = function leftShift(value) {
      if (typeof value === 'number') return this.leftShiftS(value);
      return this.leftShiftM(value);
    };

    AbstractMatrix.prototype.leftShiftS = function leftShiftS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) << value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.leftShiftM = function leftShiftM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) << matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.leftShift = function leftShift(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.leftShift(value);
    };

    AbstractMatrix.prototype.signPropagatingRightShift = function signPropagatingRightShift(value) {
      if (typeof value === 'number') return this.signPropagatingRightShiftS(value);
      return this.signPropagatingRightShiftM(value);
    };

    AbstractMatrix.prototype.signPropagatingRightShiftS = function signPropagatingRightShiftS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >> value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.signPropagatingRightShiftM = function signPropagatingRightShiftM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >> matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.signPropagatingRightShift = function signPropagatingRightShift(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.signPropagatingRightShift(value);
    };

    AbstractMatrix.prototype.rightShift = function rightShift(value) {
      if (typeof value === 'number') return this.rightShiftS(value);
      return this.rightShiftM(value);
    };

    AbstractMatrix.prototype.rightShiftS = function rightShiftS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >>> value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.rightShiftM = function rightShiftM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >>> matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.rightShift = function rightShift(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.rightShift(value);
    };

    AbstractMatrix.prototype.zeroFillRightShift = AbstractMatrix.prototype.rightShift;
    AbstractMatrix.prototype.zeroFillRightShiftS = AbstractMatrix.prototype.rightShiftS;
    AbstractMatrix.prototype.zeroFillRightShiftM = AbstractMatrix.prototype.rightShiftM;
    AbstractMatrix.zeroFillRightShift = AbstractMatrix.rightShift;

    AbstractMatrix.prototype.not = function not() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, ~this.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.not = function not(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.not();
    };

    AbstractMatrix.prototype.abs = function abs() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.abs(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.abs = function abs(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.abs();
    };

    AbstractMatrix.prototype.acos = function acos() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.acos(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.acos = function acos(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.acos();
    };

    AbstractMatrix.prototype.acosh = function acosh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.acosh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.acosh = function acosh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.acosh();
    };

    AbstractMatrix.prototype.asin = function asin() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.asin(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.asin = function asin(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.asin();
    };

    AbstractMatrix.prototype.asinh = function asinh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.asinh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.asinh = function asinh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.asinh();
    };

    AbstractMatrix.prototype.atan = function atan() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.atan(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.atan = function atan(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.atan();
    };

    AbstractMatrix.prototype.atanh = function atanh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.atanh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.atanh = function atanh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.atanh();
    };

    AbstractMatrix.prototype.cbrt = function cbrt() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.cbrt(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.cbrt = function cbrt(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.cbrt();
    };

    AbstractMatrix.prototype.ceil = function ceil() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.ceil(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.ceil = function ceil(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.ceil();
    };

    AbstractMatrix.prototype.clz32 = function clz32() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.clz32(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.clz32 = function clz32(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.clz32();
    };

    AbstractMatrix.prototype.cos = function cos() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.cos(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.cos = function cos(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.cos();
    };

    AbstractMatrix.prototype.cosh = function cosh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.cosh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.cosh = function cosh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.cosh();
    };

    AbstractMatrix.prototype.exp = function exp() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.exp(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.exp = function exp(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.exp();
    };

    AbstractMatrix.prototype.expm1 = function expm1() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.expm1(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.expm1 = function expm1(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.expm1();
    };

    AbstractMatrix.prototype.floor = function floor() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.floor(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.floor = function floor(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.floor();
    };

    AbstractMatrix.prototype.fround = function fround() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.fround(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.fround = function fround(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.fround();
    };

    AbstractMatrix.prototype.log = function log() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.log = function log(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log();
    };

    AbstractMatrix.prototype.log1p = function log1p() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log1p(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.log1p = function log1p(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log1p();
    };

    AbstractMatrix.prototype.log10 = function log10() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log10(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.log10 = function log10(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log10();
    };

    AbstractMatrix.prototype.log2 = function log2() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log2(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.log2 = function log2(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log2();
    };

    AbstractMatrix.prototype.round = function round() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.round(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.round = function round(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.round();
    };

    AbstractMatrix.prototype.sign = function sign() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sign(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.sign = function sign(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sign();
    };

    AbstractMatrix.prototype.sin = function sin() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sin(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.sin = function sin(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sin();
    };

    AbstractMatrix.prototype.sinh = function sinh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sinh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.sinh = function sinh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sinh();
    };

    AbstractMatrix.prototype.sqrt = function sqrt() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sqrt(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.sqrt = function sqrt(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sqrt();
    };

    AbstractMatrix.prototype.tan = function tan() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.tan(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.tan = function tan(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.tan();
    };

    AbstractMatrix.prototype.tanh = function tanh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.tanh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.tanh = function tanh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.tanh();
    };

    AbstractMatrix.prototype.trunc = function trunc() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.trunc(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.trunc = function trunc(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.trunc();
    };

    AbstractMatrix.pow = function pow(matrix, arg0) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.pow(arg0);
    };

    AbstractMatrix.prototype.pow = function pow(value) {
      if (typeof value === 'number') return this.powS(value);
      return this.powM(value);
    };

    AbstractMatrix.prototype.powS = function powS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.pow(this.get(i, j), value));
        }
      }

      return this;
    };

    AbstractMatrix.prototype.powM = function powM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.pow(this.get(i, j), matrix.get(i, j)));
        }
      }

      return this;
    };
  }

  /**
   * @private
   * Check that a row index is not out of bounds
   * @param {Matrix} matrix
   * @param {number} index
   * @param {boolean} [outer]
   */
  function checkRowIndex(matrix, index, outer) {
    let max = outer ? matrix.rows : matrix.rows - 1;

    if (index < 0 || index > max) {
      throw new RangeError('Row index out of range');
    }
  }
  /**
   * @private
   * Check that a column index is not out of bounds
   * @param {Matrix} matrix
   * @param {number} index
   * @param {boolean} [outer]
   */

  function checkColumnIndex(matrix, index, outer) {
    let max = outer ? matrix.columns : matrix.columns - 1;

    if (index < 0 || index > max) {
      throw new RangeError('Column index out of range');
    }
  }
  /**
   * @private
   * Check that the provided vector is an array with the right length
   * @param {Matrix} matrix
   * @param {Array|Matrix} vector
   * @return {Array}
   * @throws {RangeError}
   */

  function checkRowVector(matrix, vector) {
    if (vector.to1DArray) {
      vector = vector.to1DArray();
    }

    if (vector.length !== matrix.columns) {
      throw new RangeError('vector size must be the same as the number of columns');
    }

    return vector;
  }
  /**
   * @private
   * Check that the provided vector is an array with the right length
   * @param {Matrix} matrix
   * @param {Array|Matrix} vector
   * @return {Array}
   * @throws {RangeError}
   */

  function checkColumnVector(matrix, vector) {
    if (vector.to1DArray) {
      vector = vector.to1DArray();
    }

    if (vector.length !== matrix.rows) {
      throw new RangeError('vector size must be the same as the number of rows');
    }

    return vector;
  }
  function checkIndices(matrix, rowIndices, columnIndices) {
    return {
      row: checkRowIndices(matrix, rowIndices),
      column: checkColumnIndices(matrix, columnIndices)
    };
  }
  function checkRowIndices(matrix, rowIndices) {
    if (typeof rowIndices !== 'object') {
      throw new TypeError('unexpected type for row indices');
    }

    let rowOut = rowIndices.some(r => {
      return r < 0 || r >= matrix.rows;
    });

    if (rowOut) {
      throw new RangeError('row indices are out of range');
    }

    if (!Array.isArray(rowIndices)) rowIndices = Array.from(rowIndices);
    return rowIndices;
  }
  function checkColumnIndices(matrix, columnIndices) {
    if (typeof columnIndices !== 'object') {
      throw new TypeError('unexpected type for column indices');
    }

    let columnOut = columnIndices.some(c => {
      return c < 0 || c >= matrix.columns;
    });

    if (columnOut) {
      throw new RangeError('column indices are out of range');
    }

    if (!Array.isArray(columnIndices)) columnIndices = Array.from(columnIndices);
    return columnIndices;
  }
  function checkRange(matrix, startRow, endRow, startColumn, endColumn) {
    if (arguments.length !== 5) {
      throw new RangeError('expected 4 arguments');
    }

    checkNumber('startRow', startRow);
    checkNumber('endRow', endRow);
    checkNumber('startColumn', startColumn);
    checkNumber('endColumn', endColumn);

    if (startRow > endRow || startColumn > endColumn || startRow < 0 || startRow >= matrix.rows || endRow < 0 || endRow >= matrix.rows || startColumn < 0 || startColumn >= matrix.columns || endColumn < 0 || endColumn >= matrix.columns) {
      throw new RangeError('Submatrix indices are out of range');
    }
  }
  function newArray(length) {
    let value = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0;
    let array = [];

    for (let i = 0; i < length; i++) {
      array.push(value);
    }

    return array;
  }

  function checkNumber(name, value) {
    if (typeof value !== 'number') {
      throw new TypeError(`${name} must be a number`);
    }
  }

  function checkNonEmpty(matrix) {
    if (matrix.isEmpty()) {
      throw new Error('Empty matrix has no elements to index');
    }
  }

  function sumByRow(matrix) {
    let sum = newArray(matrix.rows);

    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[i] += matrix.get(i, j);
      }
    }

    return sum;
  }
  function sumByColumn(matrix) {
    let sum = newArray(matrix.columns);

    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[j] += matrix.get(i, j);
      }
    }

    return sum;
  }
  function sumAll(matrix) {
    let v = 0;

    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        v += matrix.get(i, j);
      }
    }

    return v;
  }
  function productByRow(matrix) {
    let sum = newArray(matrix.rows, 1);

    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[i] *= matrix.get(i, j);
      }
    }

    return sum;
  }
  function productByColumn(matrix) {
    let sum = newArray(matrix.columns, 1);

    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[j] *= matrix.get(i, j);
      }
    }

    return sum;
  }
  function productAll(matrix) {
    let v = 1;

    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        v *= matrix.get(i, j);
      }
    }

    return v;
  }
  function varianceByRow(matrix, unbiased, mean) {
    const rows = matrix.rows;
    const cols = matrix.columns;
    const variance = [];

    for (let i = 0; i < rows; i++) {
      let sum1 = 0;
      let sum2 = 0;
      let x = 0;

      for (let j = 0; j < cols; j++) {
        x = matrix.get(i, j) - mean[i];
        sum1 += x;
        sum2 += x * x;
      }

      if (unbiased) {
        variance.push((sum2 - sum1 * sum1 / cols) / (cols - 1));
      } else {
        variance.push((sum2 - sum1 * sum1 / cols) / cols);
      }
    }

    return variance;
  }
  function varianceByColumn(matrix, unbiased, mean) {
    const rows = matrix.rows;
    const cols = matrix.columns;
    const variance = [];

    for (let j = 0; j < cols; j++) {
      let sum1 = 0;
      let sum2 = 0;
      let x = 0;

      for (let i = 0; i < rows; i++) {
        x = matrix.get(i, j) - mean[j];
        sum1 += x;
        sum2 += x * x;
      }

      if (unbiased) {
        variance.push((sum2 - sum1 * sum1 / rows) / (rows - 1));
      } else {
        variance.push((sum2 - sum1 * sum1 / rows) / rows);
      }
    }

    return variance;
  }
  function varianceAll(matrix, unbiased, mean) {
    const rows = matrix.rows;
    const cols = matrix.columns;
    const size = rows * cols;
    let sum1 = 0;
    let sum2 = 0;
    let x = 0;

    for (let i = 0; i < rows; i++) {
      for (let j = 0; j < cols; j++) {
        x = matrix.get(i, j) - mean;
        sum1 += x;
        sum2 += x * x;
      }
    }

    if (unbiased) {
      return (sum2 - sum1 * sum1 / size) / (size - 1);
    } else {
      return (sum2 - sum1 * sum1 / size) / size;
    }
  }
  function centerByRow(matrix, mean) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) - mean[i]);
      }
    }
  }
  function centerByColumn(matrix, mean) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) - mean[j]);
      }
    }
  }
  function centerAll(matrix, mean) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) - mean);
      }
    }
  }
  function getScaleByRow(matrix) {
    const scale = [];

    for (let i = 0; i < matrix.rows; i++) {
      let sum = 0;

      for (let j = 0; j < matrix.columns; j++) {
        sum += Math.pow(matrix.get(i, j), 2) / (matrix.columns - 1);
      }

      scale.push(Math.sqrt(sum));
    }

    return scale;
  }
  function scaleByRow(matrix, scale) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) / scale[i]);
      }
    }
  }
  function getScaleByColumn(matrix) {
    const scale = [];

    for (let j = 0; j < matrix.columns; j++) {
      let sum = 0;

      for (let i = 0; i < matrix.rows; i++) {
        sum += Math.pow(matrix.get(i, j), 2) / (matrix.rows - 1);
      }

      scale.push(Math.sqrt(sum));
    }

    return scale;
  }
  function scaleByColumn(matrix, scale) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) / scale[j]);
      }
    }
  }
  function getScaleAll(matrix) {
    const divider = matrix.size - 1;
    let sum = 0;

    for (let j = 0; j < matrix.columns; j++) {
      for (let i = 0; i < matrix.rows; i++) {
        sum += Math.pow(matrix.get(i, j), 2) / divider;
      }
    }

    return Math.sqrt(sum);
  }
  function scaleAll(matrix, scale) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) / scale);
      }
    }
  }

  class AbstractMatrix {
    static from1DArray(newRows, newColumns, newData) {
      let length = newRows * newColumns;

      if (length !== newData.length) {
        throw new RangeError('data length does not match given dimensions');
      }

      let newMatrix = new Matrix(newRows, newColumns);

      for (let row = 0; row < newRows; row++) {
        for (let column = 0; column < newColumns; column++) {
          newMatrix.set(row, column, newData[row * newColumns + column]);
        }
      }

      return newMatrix;
    }

    static rowVector(newData) {
      let vector = new Matrix(1, newData.length);

      for (let i = 0; i < newData.length; i++) {
        vector.set(0, i, newData[i]);
      }

      return vector;
    }

    static columnVector(newData) {
      let vector = new Matrix(newData.length, 1);

      for (let i = 0; i < newData.length; i++) {
        vector.set(i, 0, newData[i]);
      }

      return vector;
    }

    static zeros(rows, columns) {
      return new Matrix(rows, columns);
    }

    static ones(rows, columns) {
      return new Matrix(rows, columns).fill(1);
    }

    static rand(rows, columns) {
      let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        random = Math.random
      } = options;
      let matrix = new Matrix(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          matrix.set(i, j, random());
        }
      }

      return matrix;
    }

    static randInt(rows, columns) {
      let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        min = 0,
        max = 1000,
        random = Math.random
      } = options;
      if (!Number.isInteger(min)) throw new TypeError('min must be an integer');
      if (!Number.isInteger(max)) throw new TypeError('max must be an integer');
      if (min >= max) throw new RangeError('min must be smaller than max');
      let interval = max - min;
      let matrix = new Matrix(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          let value = min + Math.round(random() * interval);
          matrix.set(i, j, value);
        }
      }

      return matrix;
    }

    static eye(rows, columns, value) {
      if (columns === undefined) columns = rows;
      if (value === undefined) value = 1;
      let min = Math.min(rows, columns);
      let matrix = this.zeros(rows, columns);

      for (let i = 0; i < min; i++) {
        matrix.set(i, i, value);
      }

      return matrix;
    }

    static diag(data, rows, columns) {
      let l = data.length;
      if (rows === undefined) rows = l;
      if (columns === undefined) columns = rows;
      let min = Math.min(l, rows, columns);
      let matrix = this.zeros(rows, columns);

      for (let i = 0; i < min; i++) {
        matrix.set(i, i, data[i]);
      }

      return matrix;
    }

    static min(matrix1, matrix2) {
      matrix1 = this.checkMatrix(matrix1);
      matrix2 = this.checkMatrix(matrix2);
      let rows = matrix1.rows;
      let columns = matrix1.columns;
      let result = new Matrix(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          result.set(i, j, Math.min(matrix1.get(i, j), matrix2.get(i, j)));
        }
      }

      return result;
    }

    static max(matrix1, matrix2) {
      matrix1 = this.checkMatrix(matrix1);
      matrix2 = this.checkMatrix(matrix2);
      let rows = matrix1.rows;
      let columns = matrix1.columns;
      let result = new this(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          result.set(i, j, Math.max(matrix1.get(i, j), matrix2.get(i, j)));
        }
      }

      return result;
    }

    static checkMatrix(value) {
      return AbstractMatrix.isMatrix(value) ? value : new Matrix(value);
    }

    static isMatrix(value) {
      return value != null && value.klass === 'Matrix';
    }

    get size() {
      return this.rows * this.columns;
    }

    apply(callback) {
      if (typeof callback !== 'function') {
        throw new TypeError('callback must be a function');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          callback.call(this, i, j);
        }
      }

      return this;
    }

    to1DArray() {
      let array = [];

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          array.push(this.get(i, j));
        }
      }

      return array;
    }

    to2DArray() {
      let copy = [];

      for (let i = 0; i < this.rows; i++) {
        copy.push([]);

        for (let j = 0; j < this.columns; j++) {
          copy[i].push(this.get(i, j));
        }
      }

      return copy;
    }

    toJSON() {
      return this.to2DArray();
    }

    isRowVector() {
      return this.rows === 1;
    }

    isColumnVector() {
      return this.columns === 1;
    }

    isVector() {
      return this.rows === 1 || this.columns === 1;
    }

    isSquare() {
      return this.rows === this.columns;
    }

    isEmpty() {
      return this.rows === 0 || this.columns === 0;
    }

    isSymmetric() {
      if (this.isSquare()) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j <= i; j++) {
            if (this.get(i, j) !== this.get(j, i)) {
              return false;
            }
          }
        }

        return true;
      }

      return false;
    }

    isEchelonForm() {
      let i = 0;
      let j = 0;
      let previousColumn = -1;
      let isEchelonForm = true;
      let checked = false;

      while (i < this.rows && isEchelonForm) {
        j = 0;
        checked = false;

        while (j < this.columns && checked === false) {
          if (this.get(i, j) === 0) {
            j++;
          } else if (this.get(i, j) === 1 && j > previousColumn) {
            checked = true;
            previousColumn = j;
          } else {
            isEchelonForm = false;
            checked = true;
          }
        }

        i++;
      }

      return isEchelonForm;
    }

    isReducedEchelonForm() {
      let i = 0;
      let j = 0;
      let previousColumn = -1;
      let isReducedEchelonForm = true;
      let checked = false;

      while (i < this.rows && isReducedEchelonForm) {
        j = 0;
        checked = false;

        while (j < this.columns && checked === false) {
          if (this.get(i, j) === 0) {
            j++;
          } else if (this.get(i, j) === 1 && j > previousColumn) {
            checked = true;
            previousColumn = j;
          } else {
            isReducedEchelonForm = false;
            checked = true;
          }
        }

        for (let k = j + 1; k < this.rows; k++) {
          if (this.get(i, k) !== 0) {
            isReducedEchelonForm = false;
          }
        }

        i++;
      }

      return isReducedEchelonForm;
    }

    echelonForm() {
      let result = this.clone();
      let h = 0;
      let k = 0;

      while (h < result.rows && k < result.columns) {
        let iMax = h;

        for (let i = h; i < result.rows; i++) {
          if (result.get(i, k) > result.get(iMax, k)) {
            iMax = i;
          }
        }

        if (result.get(iMax, k) === 0) {
          k++;
        } else {
          result.swapRows(h, iMax);
          let tmp = result.get(h, k);

          for (let j = k; j < result.columns; j++) {
            result.set(h, j, result.get(h, j) / tmp);
          }

          for (let i = h + 1; i < result.rows; i++) {
            let factor = result.get(i, k) / result.get(h, k);
            result.set(i, k, 0);

            for (let j = k + 1; j < result.columns; j++) {
              result.set(i, j, result.get(i, j) - result.get(h, j) * factor);
            }
          }

          h++;
          k++;
        }
      }

      return result;
    }

    reducedEchelonForm() {
      let result = this.echelonForm();
      let m = result.columns;
      let n = result.rows;
      let h = n - 1;

      while (h >= 0) {
        if (result.maxRow(h) === 0) {
          h--;
        } else {
          let p = 0;
          let pivot = false;

          while (p < n && pivot === false) {
            if (result.get(h, p) === 1) {
              pivot = true;
            } else {
              p++;
            }
          }

          for (let i = 0; i < h; i++) {
            let factor = result.get(i, p);

            for (let j = p; j < m; j++) {
              let tmp = result.get(i, j) - factor * result.get(h, j);
              result.set(i, j, tmp);
            }
          }

          h--;
        }
      }

      return result;
    }

    set() {
      throw new Error('set method is unimplemented');
    }

    get() {
      throw new Error('get method is unimplemented');
    }

    repeat() {
      let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        rows = 1,
        columns = 1
      } = options;

      if (!Number.isInteger(rows) || rows <= 0) {
        throw new TypeError('rows must be a positive integer');
      }

      if (!Number.isInteger(columns) || columns <= 0) {
        throw new TypeError('columns must be a positive integer');
      }

      let matrix = new Matrix(this.rows * rows, this.columns * columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          matrix.setSubMatrix(this, this.rows * i, this.columns * j);
        }
      }

      return matrix;
    }

    fill(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, value);
        }
      }

      return this;
    }

    neg() {
      return this.mulS(-1);
    }

    getRow(index) {
      checkRowIndex(this, index);
      let row = [];

      for (let i = 0; i < this.columns; i++) {
        row.push(this.get(index, i));
      }

      return row;
    }

    getRowVector(index) {
      return Matrix.rowVector(this.getRow(index));
    }

    setRow(index, array) {
      checkRowIndex(this, index);
      array = checkRowVector(this, array);

      for (let i = 0; i < this.columns; i++) {
        this.set(index, i, array[i]);
      }

      return this;
    }

    swapRows(row1, row2) {
      checkRowIndex(this, row1);
      checkRowIndex(this, row2);

      for (let i = 0; i < this.columns; i++) {
        let temp = this.get(row1, i);
        this.set(row1, i, this.get(row2, i));
        this.set(row2, i, temp);
      }

      return this;
    }

    getColumn(index) {
      checkColumnIndex(this, index);
      let column = [];

      for (let i = 0; i < this.rows; i++) {
        column.push(this.get(i, index));
      }

      return column;
    }

    getColumnVector(index) {
      return Matrix.columnVector(this.getColumn(index));
    }

    setColumn(index, array) {
      checkColumnIndex(this, index);
      array = checkColumnVector(this, array);

      for (let i = 0; i < this.rows; i++) {
        this.set(i, index, array[i]);
      }

      return this;
    }

    swapColumns(column1, column2) {
      checkColumnIndex(this, column1);
      checkColumnIndex(this, column2);

      for (let i = 0; i < this.rows; i++) {
        let temp = this.get(i, column1);
        this.set(i, column1, this.get(i, column2));
        this.set(i, column2, temp);
      }

      return this;
    }

    addRowVector(vector) {
      vector = checkRowVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + vector[j]);
        }
      }

      return this;
    }

    subRowVector(vector) {
      vector = checkRowVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - vector[j]);
        }
      }

      return this;
    }

    mulRowVector(vector) {
      vector = checkRowVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * vector[j]);
        }
      }

      return this;
    }

    divRowVector(vector) {
      vector = checkRowVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / vector[j]);
        }
      }

      return this;
    }

    addColumnVector(vector) {
      vector = checkColumnVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + vector[i]);
        }
      }

      return this;
    }

    subColumnVector(vector) {
      vector = checkColumnVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - vector[i]);
        }
      }

      return this;
    }

    mulColumnVector(vector) {
      vector = checkColumnVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * vector[i]);
        }
      }

      return this;
    }

    divColumnVector(vector) {
      vector = checkColumnVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / vector[i]);
        }
      }

      return this;
    }

    mulRow(index, value) {
      checkRowIndex(this, index);

      for (let i = 0; i < this.columns; i++) {
        this.set(index, i, this.get(index, i) * value);
      }

      return this;
    }

    mulColumn(index, value) {
      checkColumnIndex(this, index);

      for (let i = 0; i < this.rows; i++) {
        this.set(i, index, this.get(i, index) * value);
      }

      return this;
    }

    max() {
      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(0, 0);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) > v) {
            v = this.get(i, j);
          }
        }
      }

      return v;
    }

    maxIndex() {
      checkNonEmpty(this);
      let v = this.get(0, 0);
      let idx = [0, 0];

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) > v) {
            v = this.get(i, j);
            idx[0] = i;
            idx[1] = j;
          }
        }
      }

      return idx;
    }

    min() {
      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(0, 0);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) < v) {
            v = this.get(i, j);
          }
        }
      }

      return v;
    }

    minIndex() {
      checkNonEmpty(this);
      let v = this.get(0, 0);
      let idx = [0, 0];

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) < v) {
            v = this.get(i, j);
            idx[0] = i;
            idx[1] = j;
          }
        }
      }

      return idx;
    }

    maxRow(row) {
      checkRowIndex(this, row);

      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(row, 0);

      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) > v) {
          v = this.get(row, i);
        }
      }

      return v;
    }

    maxRowIndex(row) {
      checkRowIndex(this, row);
      checkNonEmpty(this);
      let v = this.get(row, 0);
      let idx = [row, 0];

      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) > v) {
          v = this.get(row, i);
          idx[1] = i;
        }
      }

      return idx;
    }

    minRow(row) {
      checkRowIndex(this, row);

      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(row, 0);

      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) < v) {
          v = this.get(row, i);
        }
      }

      return v;
    }

    minRowIndex(row) {
      checkRowIndex(this, row);
      checkNonEmpty(this);
      let v = this.get(row, 0);
      let idx = [row, 0];

      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) < v) {
          v = this.get(row, i);
          idx[1] = i;
        }
      }

      return idx;
    }

    maxColumn(column) {
      checkColumnIndex(this, column);

      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(0, column);

      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) > v) {
          v = this.get(i, column);
        }
      }

      return v;
    }

    maxColumnIndex(column) {
      checkColumnIndex(this, column);
      checkNonEmpty(this);
      let v = this.get(0, column);
      let idx = [0, column];

      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) > v) {
          v = this.get(i, column);
          idx[0] = i;
        }
      }

      return idx;
    }

    minColumn(column) {
      checkColumnIndex(this, column);

      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(0, column);

      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) < v) {
          v = this.get(i, column);
        }
      }

      return v;
    }

    minColumnIndex(column) {
      checkColumnIndex(this, column);
      checkNonEmpty(this);
      let v = this.get(0, column);
      let idx = [0, column];

      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) < v) {
          v = this.get(i, column);
          idx[0] = i;
        }
      }

      return idx;
    }

    diag() {
      let min = Math.min(this.rows, this.columns);
      let diag = [];

      for (let i = 0; i < min; i++) {
        diag.push(this.get(i, i));
      }

      return diag;
    }

    norm() {
      let type = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 'frobenius';
      let result = 0;

      if (type === 'max') {
        return this.max();
      } else if (type === 'frobenius') {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            result = result + this.get(i, j) * this.get(i, j);
          }
        }

        return Math.sqrt(result);
      } else {
        throw new RangeError(`unknown norm type: ${type}`);
      }
    }

    cumulativeSum() {
      let sum = 0;

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          sum += this.get(i, j);
          this.set(i, j, sum);
        }
      }

      return this;
    }

    dot(vector2) {
      if (AbstractMatrix.isMatrix(vector2)) vector2 = vector2.to1DArray();
      let vector1 = this.to1DArray();

      if (vector1.length !== vector2.length) {
        throw new RangeError('vectors do not have the same size');
      }

      let dot = 0;

      for (let i = 0; i < vector1.length; i++) {
        dot += vector1[i] * vector2[i];
      }

      return dot;
    }

    mmul(other) {
      other = Matrix.checkMatrix(other);
      let m = this.rows;
      let n = this.columns;
      let p = other.columns;
      let result = new Matrix(m, p);
      let Bcolj = new Float64Array(n);

      for (let j = 0; j < p; j++) {
        for (let k = 0; k < n; k++) {
          Bcolj[k] = other.get(k, j);
        }

        for (let i = 0; i < m; i++) {
          let s = 0;

          for (let k = 0; k < n; k++) {
            s += this.get(i, k) * Bcolj[k];
          }

          result.set(i, j, s);
        }
      }

      return result;
    }

    strassen2x2(other) {
      other = Matrix.checkMatrix(other);
      let result = new Matrix(2, 2);
      const a11 = this.get(0, 0);
      const b11 = other.get(0, 0);
      const a12 = this.get(0, 1);
      const b12 = other.get(0, 1);
      const a21 = this.get(1, 0);
      const b21 = other.get(1, 0);
      const a22 = this.get(1, 1);
      const b22 = other.get(1, 1); // Compute intermediate values.

      const m1 = (a11 + a22) * (b11 + b22);
      const m2 = (a21 + a22) * b11;
      const m3 = a11 * (b12 - b22);
      const m4 = a22 * (b21 - b11);
      const m5 = (a11 + a12) * b22;
      const m6 = (a21 - a11) * (b11 + b12);
      const m7 = (a12 - a22) * (b21 + b22); // Combine intermediate values into the output.

      const c00 = m1 + m4 - m5 + m7;
      const c01 = m3 + m5;
      const c10 = m2 + m4;
      const c11 = m1 - m2 + m3 + m6;
      result.set(0, 0, c00);
      result.set(0, 1, c01);
      result.set(1, 0, c10);
      result.set(1, 1, c11);
      return result;
    }

    strassen3x3(other) {
      other = Matrix.checkMatrix(other);
      let result = new Matrix(3, 3);
      const a00 = this.get(0, 0);
      const a01 = this.get(0, 1);
      const a02 = this.get(0, 2);
      const a10 = this.get(1, 0);
      const a11 = this.get(1, 1);
      const a12 = this.get(1, 2);
      const a20 = this.get(2, 0);
      const a21 = this.get(2, 1);
      const a22 = this.get(2, 2);
      const b00 = other.get(0, 0);
      const b01 = other.get(0, 1);
      const b02 = other.get(0, 2);
      const b10 = other.get(1, 0);
      const b11 = other.get(1, 1);
      const b12 = other.get(1, 2);
      const b20 = other.get(2, 0);
      const b21 = other.get(2, 1);
      const b22 = other.get(2, 2);
      const m1 = (a00 + a01 + a02 - a10 - a11 - a21 - a22) * b11;
      const m2 = (a00 - a10) * (-b01 + b11);
      const m3 = a11 * (-b00 + b01 + b10 - b11 - b12 - b20 + b22);
      const m4 = (-a00 + a10 + a11) * (b00 - b01 + b11);
      const m5 = (a10 + a11) * (-b00 + b01);
      const m6 = a00 * b00;
      const m7 = (-a00 + a20 + a21) * (b00 - b02 + b12);
      const m8 = (-a00 + a20) * (b02 - b12);
      const m9 = (a20 + a21) * (-b00 + b02);
      const m10 = (a00 + a01 + a02 - a11 - a12 - a20 - a21) * b12;
      const m11 = a21 * (-b00 + b02 + b10 - b11 - b12 - b20 + b21);
      const m12 = (-a02 + a21 + a22) * (b11 + b20 - b21);
      const m13 = (a02 - a22) * (b11 - b21);
      const m14 = a02 * b20;
      const m15 = (a21 + a22) * (-b20 + b21);
      const m16 = (-a02 + a11 + a12) * (b12 + b20 - b22);
      const m17 = (a02 - a12) * (b12 - b22);
      const m18 = (a11 + a12) * (-b20 + b22);
      const m19 = a01 * b10;
      const m20 = a12 * b21;
      const m21 = a10 * b02;
      const m22 = a20 * b01;
      const m23 = a22 * b22;
      const c00 = m6 + m14 + m19;
      const c01 = m1 + m4 + m5 + m6 + m12 + m14 + m15;
      const c02 = m6 + m7 + m9 + m10 + m14 + m16 + m18;
      const c10 = m2 + m3 + m4 + m6 + m14 + m16 + m17;
      const c11 = m2 + m4 + m5 + m6 + m20;
      const c12 = m14 + m16 + m17 + m18 + m21;
      const c20 = m6 + m7 + m8 + m11 + m12 + m13 + m14;
      const c21 = m12 + m13 + m14 + m15 + m22;
      const c22 = m6 + m7 + m8 + m9 + m23;
      result.set(0, 0, c00);
      result.set(0, 1, c01);
      result.set(0, 2, c02);
      result.set(1, 0, c10);
      result.set(1, 1, c11);
      result.set(1, 2, c12);
      result.set(2, 0, c20);
      result.set(2, 1, c21);
      result.set(2, 2, c22);
      return result;
    }

    mmulStrassen(y) {
      y = Matrix.checkMatrix(y);
      let x = this.clone();
      let r1 = x.rows;
      let c1 = x.columns;
      let r2 = y.rows;
      let c2 = y.columns;

      if (c1 !== r2) {
        // eslint-disable-next-line no-console
        console.warn(`Multiplying ${r1} x ${c1} and ${r2} x ${c2} matrix: dimensions do not match.`);
      } // Put a matrix into the top left of a matrix of zeros.
      // `rows` and `cols` are the dimensions of the output matrix.


      function embed(mat, rows, cols) {
        let r = mat.rows;
        let c = mat.columns;

        if (r === rows && c === cols) {
          return mat;
        } else {
          let resultat = AbstractMatrix.zeros(rows, cols);
          resultat = resultat.setSubMatrix(mat, 0, 0);
          return resultat;
        }
      } // Make sure both matrices are the same size.
      // This is exclusively for simplicity:
      // this algorithm can be implemented with matrices of different sizes.


      let r = Math.max(r1, r2);
      let c = Math.max(c1, c2);
      x = embed(x, r, c);
      y = embed(y, r, c); // Our recursive multiplication function.

      function blockMult(a, b, rows, cols) {
        // For small matrices, resort to naive multiplication.
        if (rows <= 512 || cols <= 512) {
          return a.mmul(b); // a is equivalent to this
        } // Apply dynamic padding.


        if (rows % 2 === 1 && cols % 2 === 1) {
          a = embed(a, rows + 1, cols + 1);
          b = embed(b, rows + 1, cols + 1);
        } else if (rows % 2 === 1) {
          a = embed(a, rows + 1, cols);
          b = embed(b, rows + 1, cols);
        } else if (cols % 2 === 1) {
          a = embed(a, rows, cols + 1);
          b = embed(b, rows, cols + 1);
        }

        let halfRows = parseInt(a.rows / 2, 10);
        let halfCols = parseInt(a.columns / 2, 10); // Subdivide input matrices.

        let a11 = a.subMatrix(0, halfRows - 1, 0, halfCols - 1);
        let b11 = b.subMatrix(0, halfRows - 1, 0, halfCols - 1);
        let a12 = a.subMatrix(0, halfRows - 1, halfCols, a.columns - 1);
        let b12 = b.subMatrix(0, halfRows - 1, halfCols, b.columns - 1);
        let a21 = a.subMatrix(halfRows, a.rows - 1, 0, halfCols - 1);
        let b21 = b.subMatrix(halfRows, b.rows - 1, 0, halfCols - 1);
        let a22 = a.subMatrix(halfRows, a.rows - 1, halfCols, a.columns - 1);
        let b22 = b.subMatrix(halfRows, b.rows - 1, halfCols, b.columns - 1); // Compute intermediate values.

        let m1 = blockMult(AbstractMatrix.add(a11, a22), AbstractMatrix.add(b11, b22), halfRows, halfCols);
        let m2 = blockMult(AbstractMatrix.add(a21, a22), b11, halfRows, halfCols);
        let m3 = blockMult(a11, AbstractMatrix.sub(b12, b22), halfRows, halfCols);
        let m4 = blockMult(a22, AbstractMatrix.sub(b21, b11), halfRows, halfCols);
        let m5 = blockMult(AbstractMatrix.add(a11, a12), b22, halfRows, halfCols);
        let m6 = blockMult(AbstractMatrix.sub(a21, a11), AbstractMatrix.add(b11, b12), halfRows, halfCols);
        let m7 = blockMult(AbstractMatrix.sub(a12, a22), AbstractMatrix.add(b21, b22), halfRows, halfCols); // Combine intermediate values into the output.

        let c11 = AbstractMatrix.add(m1, m4);
        c11.sub(m5);
        c11.add(m7);
        let c12 = AbstractMatrix.add(m3, m5);
        let c21 = AbstractMatrix.add(m2, m4);
        let c22 = AbstractMatrix.sub(m1, m2);
        c22.add(m3);
        c22.add(m6); // Crop output to the desired size (undo dynamic padding).

        let resultat = AbstractMatrix.zeros(2 * c11.rows, 2 * c11.columns);
        resultat = resultat.setSubMatrix(c11, 0, 0);
        resultat = resultat.setSubMatrix(c12, c11.rows, 0);
        resultat = resultat.setSubMatrix(c21, 0, c11.columns);
        resultat = resultat.setSubMatrix(c22, c11.rows, c11.columns);
        return resultat.subMatrix(0, rows - 1, 0, cols - 1);
      }

      return blockMult(x, y, r, c);
    }

    scaleRows() {
      let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        min = 0,
        max = 1
      } = options;
      if (!Number.isFinite(min)) throw new TypeError('min must be a number');
      if (!Number.isFinite(max)) throw new TypeError('max must be a number');
      if (min >= max) throw new RangeError('min must be smaller than max');
      let newMatrix = new Matrix(this.rows, this.columns);

      for (let i = 0; i < this.rows; i++) {
        const row = this.getRow(i);

        if (row.length > 0) {
          rescale(row, {
            min,
            max,
            output: row
          });
        }

        newMatrix.setRow(i, row);
      }

      return newMatrix;
    }

    scaleColumns() {
      let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        min = 0,
        max = 1
      } = options;
      if (!Number.isFinite(min)) throw new TypeError('min must be a number');
      if (!Number.isFinite(max)) throw new TypeError('max must be a number');
      if (min >= max) throw new RangeError('min must be smaller than max');
      let newMatrix = new Matrix(this.rows, this.columns);

      for (let i = 0; i < this.columns; i++) {
        const column = this.getColumn(i);

        if (column.length) {
          rescale(column, {
            min: min,
            max: max,
            output: column
          });
        }

        newMatrix.setColumn(i, column);
      }

      return newMatrix;
    }

    flipRows() {
      const middle = Math.ceil(this.columns / 2);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < middle; j++) {
          let first = this.get(i, j);
          let last = this.get(i, this.columns - 1 - j);
          this.set(i, j, last);
          this.set(i, this.columns - 1 - j, first);
        }
      }

      return this;
    }

    flipColumns() {
      const middle = Math.ceil(this.rows / 2);

      for (let j = 0; j < this.columns; j++) {
        for (let i = 0; i < middle; i++) {
          let first = this.get(i, j);
          let last = this.get(this.rows - 1 - i, j);
          this.set(i, j, last);
          this.set(this.rows - 1 - i, j, first);
        }
      }

      return this;
    }

    kroneckerProduct(other) {
      other = Matrix.checkMatrix(other);
      let m = this.rows;
      let n = this.columns;
      let p = other.rows;
      let q = other.columns;
      let result = new Matrix(m * p, n * q);

      for (let i = 0; i < m; i++) {
        for (let j = 0; j < n; j++) {
          for (let k = 0; k < p; k++) {
            for (let l = 0; l < q; l++) {
              result.set(p * i + k, q * j + l, this.get(i, j) * other.get(k, l));
            }
          }
        }
      }

      return result;
    }

    kroneckerSum(other) {
      other = Matrix.checkMatrix(other);

      if (!this.isSquare() || !other.isSquare()) {
        throw new Error('Kronecker Sum needs two Square Matrices');
      }

      let m = this.rows;
      let n = other.rows;
      let AxI = this.kroneckerProduct(Matrix.eye(n, n));
      let IxB = Matrix.eye(m, m).kroneckerProduct(other);
      return AxI.add(IxB);
    }

    transpose() {
      let result = new Matrix(this.columns, this.rows);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          result.set(j, i, this.get(i, j));
        }
      }

      return result;
    }

    sortRows() {
      let compareFunction = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : compareNumbers;

      for (let i = 0; i < this.rows; i++) {
        this.setRow(i, this.getRow(i).sort(compareFunction));
      }

      return this;
    }

    sortColumns() {
      let compareFunction = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : compareNumbers;

      for (let i = 0; i < this.columns; i++) {
        this.setColumn(i, this.getColumn(i).sort(compareFunction));
      }

      return this;
    }

    subMatrix(startRow, endRow, startColumn, endColumn) {
      checkRange(this, startRow, endRow, startColumn, endColumn);
      let newMatrix = new Matrix(endRow - startRow + 1, endColumn - startColumn + 1);

      for (let i = startRow; i <= endRow; i++) {
        for (let j = startColumn; j <= endColumn; j++) {
          newMatrix.set(i - startRow, j - startColumn, this.get(i, j));
        }
      }

      return newMatrix;
    }

    subMatrixRow(indices, startColumn, endColumn) {
      if (startColumn === undefined) startColumn = 0;
      if (endColumn === undefined) endColumn = this.columns - 1;

      if (startColumn > endColumn || startColumn < 0 || startColumn >= this.columns || endColumn < 0 || endColumn >= this.columns) {
        throw new RangeError('Argument out of range');
      }

      let newMatrix = new Matrix(indices.length, endColumn - startColumn + 1);

      for (let i = 0; i < indices.length; i++) {
        for (let j = startColumn; j <= endColumn; j++) {
          if (indices[i] < 0 || indices[i] >= this.rows) {
            throw new RangeError(`Row index out of range: ${indices[i]}`);
          }

          newMatrix.set(i, j - startColumn, this.get(indices[i], j));
        }
      }

      return newMatrix;
    }

    subMatrixColumn(indices, startRow, endRow) {
      if (startRow === undefined) startRow = 0;
      if (endRow === undefined) endRow = this.rows - 1;

      if (startRow > endRow || startRow < 0 || startRow >= this.rows || endRow < 0 || endRow >= this.rows) {
        throw new RangeError('Argument out of range');
      }

      let newMatrix = new Matrix(endRow - startRow + 1, indices.length);

      for (let i = 0; i < indices.length; i++) {
        for (let j = startRow; j <= endRow; j++) {
          if (indices[i] < 0 || indices[i] >= this.columns) {
            throw new RangeError(`Column index out of range: ${indices[i]}`);
          }

          newMatrix.set(j - startRow, i, this.get(j, indices[i]));
        }
      }

      return newMatrix;
    }

    setSubMatrix(matrix, startRow, startColumn) {
      matrix = Matrix.checkMatrix(matrix);

      if (matrix.isEmpty()) {
        return this;
      }

      let endRow = startRow + matrix.rows - 1;
      let endColumn = startColumn + matrix.columns - 1;
      checkRange(this, startRow, endRow, startColumn, endColumn);

      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          this.set(startRow + i, startColumn + j, matrix.get(i, j));
        }
      }

      return this;
    }

    selection(rowIndices, columnIndices) {
      let indices = checkIndices(this, rowIndices, columnIndices);
      let newMatrix = new Matrix(rowIndices.length, columnIndices.length);

      for (let i = 0; i < indices.row.length; i++) {
        let rowIndex = indices.row[i];

        for (let j = 0; j < indices.column.length; j++) {
          let columnIndex = indices.column[j];
          newMatrix.set(i, j, this.get(rowIndex, columnIndex));
        }
      }

      return newMatrix;
    }

    trace() {
      let min = Math.min(this.rows, this.columns);
      let trace = 0;

      for (let i = 0; i < min; i++) {
        trace += this.get(i, i);
      }

      return trace;
    }

    clone() {
      let newMatrix = new Matrix(this.rows, this.columns);

      for (let row = 0; row < this.rows; row++) {
        for (let column = 0; column < this.columns; column++) {
          newMatrix.set(row, column, this.get(row, column));
        }
      }

      return newMatrix;
    }

    sum(by) {
      switch (by) {
        case 'row':
          return sumByRow(this);

        case 'column':
          return sumByColumn(this);

        case undefined:
          return sumAll(this);

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    product(by) {
      switch (by) {
        case 'row':
          return productByRow(this);

        case 'column':
          return productByColumn(this);

        case undefined:
          return productAll(this);

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    mean(by) {
      const sum = this.sum(by);

      switch (by) {
        case 'row':
          {
            for (let i = 0; i < this.rows; i++) {
              sum[i] /= this.columns;
            }

            return sum;
          }

        case 'column':
          {
            for (let i = 0; i < this.columns; i++) {
              sum[i] /= this.rows;
            }

            return sum;
          }

        case undefined:
          return sum / this.size;

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    variance(by) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }

      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        unbiased = true,
        mean = this.mean(by)
      } = options;

      if (typeof unbiased !== 'boolean') {
        throw new TypeError('unbiased must be a boolean');
      }

      switch (by) {
        case 'row':
          {
            if (!Array.isArray(mean)) {
              throw new TypeError('mean must be an array');
            }

            return varianceByRow(this, unbiased, mean);
          }

        case 'column':
          {
            if (!Array.isArray(mean)) {
              throw new TypeError('mean must be an array');
            }

            return varianceByColumn(this, unbiased, mean);
          }

        case undefined:
          {
            if (typeof mean !== 'number') {
              throw new TypeError('mean must be a number');
            }

            return varianceAll(this, unbiased, mean);
          }

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    standardDeviation(by, options) {
      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }

      const variance = this.variance(by, options);

      if (by === undefined) {
        return Math.sqrt(variance);
      } else {
        for (let i = 0; i < variance.length; i++) {
          variance[i] = Math.sqrt(variance[i]);
        }

        return variance;
      }
    }

    center(by) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }

      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        center = this.mean(by)
      } = options;

      switch (by) {
        case 'row':
          {
            if (!Array.isArray(center)) {
              throw new TypeError('center must be an array');
            }

            centerByRow(this, center);
            return this;
          }

        case 'column':
          {
            if (!Array.isArray(center)) {
              throw new TypeError('center must be an array');
            }

            centerByColumn(this, center);
            return this;
          }

        case undefined:
          {
            if (typeof center !== 'number') {
              throw new TypeError('center must be a number');
            }

            centerAll(this, center);
            return this;
          }

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    scale(by) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }

      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      let scale = options.scale;

      switch (by) {
        case 'row':
          {
            if (scale === undefined) {
              scale = getScaleByRow(this);
            } else if (!Array.isArray(scale)) {
              throw new TypeError('scale must be an array');
            }

            scaleByRow(this, scale);
            return this;
          }

        case 'column':
          {
            if (scale === undefined) {
              scale = getScaleByColumn(this);
            } else if (!Array.isArray(scale)) {
              throw new TypeError('scale must be an array');
            }

            scaleByColumn(this, scale);
            return this;
          }

        case undefined:
          {
            if (scale === undefined) {
              scale = getScaleAll(this);
            } else if (typeof scale !== 'number') {
              throw new TypeError('scale must be a number');
            }

            scaleAll(this, scale);
            return this;
          }

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    toString(options) {
      return inspectMatrixWithOptions(this, options);
    }

  }
  AbstractMatrix.prototype.klass = 'Matrix';

  if (typeof Symbol !== 'undefined') {
    AbstractMatrix.prototype[Symbol.for('nodejs.util.inspect.custom')] = inspectMatrix;
  }

  function compareNumbers(a, b) {
    return a - b;
  } // Synonyms


  AbstractMatrix.random = AbstractMatrix.rand;
  AbstractMatrix.randomInt = AbstractMatrix.randInt;
  AbstractMatrix.diagonal = AbstractMatrix.diag;
  AbstractMatrix.prototype.diagonal = AbstractMatrix.prototype.diag;
  AbstractMatrix.identity = AbstractMatrix.eye;
  AbstractMatrix.prototype.negate = AbstractMatrix.prototype.neg;
  AbstractMatrix.prototype.tensorProduct = AbstractMatrix.prototype.kroneckerProduct;
  class Matrix extends AbstractMatrix {
    constructor(nRows, nColumns) {
      super();

      if (Matrix.isMatrix(nRows)) {
        // eslint-disable-next-line no-constructor-return
        return nRows.clone();
      } else if (Number.isInteger(nRows) && nRows >= 0) {
        // Create an empty matrix
        this.data = [];

        if (Number.isInteger(nColumns) && nColumns >= 0) {
          for (let i = 0; i < nRows; i++) {
            this.data.push(new Float64Array(nColumns));
          }
        } else {
          throw new TypeError('nColumns must be a positive integer');
        }
      } else if (Array.isArray(nRows)) {
        // Copy the values from the 2D array
        const arrayData = nRows;
        nRows = arrayData.length;
        nColumns = nRows ? arrayData[0].length : 0;

        if (typeof nColumns !== 'number') {
          throw new TypeError('Data must be a 2D array with at least one element');
        }

        this.data = [];

        for (let i = 0; i < nRows; i++) {
          if (arrayData[i].length !== nColumns) {
            throw new RangeError('Inconsistent array dimensions');
          }

          this.data.push(Float64Array.from(arrayData[i]));
        }
      } else {
        throw new TypeError('First argument must be a positive number or an array');
      }

      this.rows = nRows;
      this.columns = nColumns;
    }

    set(rowIndex, columnIndex, value) {
      this.data[rowIndex][columnIndex] = value;
      return this;
    }

    get(rowIndex, columnIndex) {
      return this.data[rowIndex][columnIndex];
    }

    removeRow(index) {
      checkRowIndex(this, index);
      this.data.splice(index, 1);
      this.rows -= 1;
      return this;
    }

    addRow(index, array) {
      if (array === undefined) {
        array = index;
        index = this.rows;
      }

      checkRowIndex(this, index, true);
      array = Float64Array.from(checkRowVector(this, array));
      this.data.splice(index, 0, array);
      this.rows += 1;
      return this;
    }

    removeColumn(index) {
      checkColumnIndex(this, index);

      for (let i = 0; i < this.rows; i++) {
        const newRow = new Float64Array(this.columns - 1);

        for (let j = 0; j < index; j++) {
          newRow[j] = this.data[i][j];
        }

        for (let j = index + 1; j < this.columns; j++) {
          newRow[j - 1] = this.data[i][j];
        }

        this.data[i] = newRow;
      }

      this.columns -= 1;
      return this;
    }

    addColumn(index, array) {
      if (typeof array === 'undefined') {
        array = index;
        index = this.columns;
      }

      checkColumnIndex(this, index, true);
      array = checkColumnVector(this, array);

      for (let i = 0; i < this.rows; i++) {
        const newRow = new Float64Array(this.columns + 1);
        let j = 0;

        for (; j < index; j++) {
          newRow[j] = this.data[i][j];
        }

        newRow[j++] = array[i];

        for (; j < this.columns + 1; j++) {
          newRow[j] = this.data[i][j - 1];
        }

        this.data[i] = newRow;
      }

      this.columns += 1;
      return this;
    }

  }
  installMathOperations(AbstractMatrix, Matrix);

  /**
   * Algorithm that finds the shortest distance from one node to the other
   * @param {Matrix} adjMatrix - A squared adjacency matrix
   * @return {Matrix} - Distance from a node to the other, -1 if the node is unreachable
   */

  function floydWarshall(adjMatrix) {
    if (Matrix.isMatrix(adjMatrix) && adjMatrix.columns !== adjMatrix.rows) {
      throw new TypeError('The adjacency matrix should be squared');
    }

    const numVertices = adjMatrix.columns;
    let distMatrix = new Matrix(numVertices, numVertices);
    distMatrix.apply((row, column) => {
      // principal diagonal is 0
      if (row === column) {
        distMatrix.set(row, column, 0);
      } else {
        let val = adjMatrix.get(row, column);

        if (val) {
          // edges values remain the same
          distMatrix.set(row, column, val);
        } else {
          // 0 values become infinity
          distMatrix.set(row, column, Number.POSITIVE_INFINITY);
        }
      }
    });

    for (let k = 0; k < numVertices; ++k) {
      for (let i = 0; i < numVertices; ++i) {
        for (let j = 0; j < numVertices; ++j) {
          let dist = distMatrix.get(i, k) + distMatrix.get(k, j);

          if (distMatrix.get(i, j) > dist) {
            distMatrix.set(i, j, dist);
          }
        }
      }
    } // When there's no connection the value is -1


    distMatrix.apply((row, column) => {
      if (distMatrix.get(row, column) === Number.POSITIVE_INFINITY) {
        distMatrix.set(row, column, -1);
      }
    });
    return distMatrix;
  }

  /**
   * Returns a connectivity matrix
   * @param {OCL.Molecule} molecule
   * @param {object} [options={}]
   * @param {boolean} [options.pathLength=false] get the path length between atoms
   * @param {boolean} [options.mass=false] set the nominal mass of the atoms on diagonal
   * @param {boolean} [options.atomicNo=false] set the atomic number of the atom on diagonal
   * @param {boolean} [options.negativeAtomicNo=false] set the atomic number * -1 of the atom on diagonal
   * @param {boolean} [options.sdt=false] set 1, 2 or 3 depending if single, double or triple bond
   * @param {boolean} [options.sdta=false] set 1, 2, 3 or 4 depending if single, double, triple or aromatic  bond
   */

  function getConnectivityMatrix(molecule) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    const OCL = molecule.getOCL();
    molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
    let nbAtoms = molecule.getAllAtoms();
    let result = new Array(nbAtoms).fill();
    result = result.map(() => new Array(nbAtoms).fill(0));

    if (!options.pathLength) {
      if (options.atomicNo) {
        for (let i = 0; i < nbAtoms; i++) {
          result[i][i] = molecule.getAtomicNo(i);
        }
      } else if (options.negativeAtomicNo) {
        for (let i = 0; i < nbAtoms; i++) {
          result[i][i] = -molecule.getAtomicNo(i);
        }
      } else if (options.mass) {
        for (let i = 0; i < nbAtoms; i++) {
          result[i][i] = OCL.Molecule.cRoundedMass[molecule.getAtomicNo(i)];
        }
      } else {
        for (let i = 0; i < nbAtoms; i++) {
          result[i][i] = 1;
        }
      }
    }

    if (options.sdt) {
      for (let i = 0; i < nbAtoms; i++) {
        let l = molecule.getAllConnAtoms(i);

        for (let j = 0; j < l; j++) {
          result[i][molecule.getConnAtom(i, j)] = molecule.getConnBondOrder(i, j);
        }
      }
    } else if (options.sdta) {
      for (let i = 0; i < nbAtoms; i++) {
        let l = molecule.getAllConnAtoms(i);

        for (let j = 0; j < l; j++) {
          let bondNumber = molecule.getConnBond(i, j);

          if (molecule.isAromaticBond(bondNumber)) {
            result[i][molecule.getConnAtom(i, j)] = 4;
          } else {
            result[i][molecule.getConnAtom(i, j)] = molecule.getConnBondOrder(i, j);
          }
        }
      }
    } else {
      for (let i = 0; i < nbAtoms; i++) {
        let l = molecule.getAllConnAtoms(i);

        for (let j = 0; j < l; j++) {
          result[i][molecule.getConnAtom(i, j)] = 1;
        }
      }
    }

    if (options.pathLength) {
      result = floydWarshall(new Matrix(result)).to2DArray();
    }

    return result;
  }

  var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

  function atomSorter(a, b) {
    if (a === b) return 0;
    if (a === 'C') return -1;
    if (b === 'C') return 1;
    if (a === 'H') return -1;
    if (b === 'H') return 1;
    if (a < b) return -1;
    return 1;
  }

  var src = atomSorter;
  var atomSorter$1 = src;

  /**
   * Calculate the molecular formula in 'chemcalc' notation taking into account fragments, isotopes and charges
   * {OCL.Molecule} [molecule] an instance of OCL.Molecule
   * @returns {object}
   */

  function getMF(molecule) {
    let entries = molecule.getFragments();
    let result = {};
    let parts = [];
    let allAtoms = [];
    entries.forEach(entry => {
      let mf = getFragmentMF(entry, allAtoms);
      parts.push(mf);
    });
    let counts = {};

    for (let part of parts) {
      if (!counts[part]) counts[part] = 0;
      counts[part]++;
    }

    parts = [];

    for (let key of Object.keys(counts).sort()) {
      if (counts[key] > 1) {
        parts.push(counts[key] + key);
      } else {
        parts.push(key);
      }
    }

    result.parts = parts;
    result.mf = toMFString(allAtoms);
    return result;
  }

  function getFragmentMF(molecule, allAtoms) {
    let atoms = [];

    for (let i = 0; i < molecule.getAllAtoms(); i++) {
      let atom = {};
      atom.charge = molecule.getAtomCharge(i);
      atom.label = molecule.getAtomLabel(i);
      atom.mass = molecule.getAtomMass(i);
      atom.implicitHydrogens = molecule.getImplicitHydrogens(i);
      atoms.push(atom);
      allAtoms.push(atom);
    }

    return toMFString(atoms);
  }

  function toMFString(atoms) {
    let charge = 0;
    let mfs = {};

    for (let atom of atoms) {
      let label = atom.label;
      charge += atom.charge;

      if (atom.mass) {
        label = `[${atom.mass}${label}]`;
      }

      let mfAtom = mfs[label];

      if (!mfAtom) {
        mfs[label] = 0;
      }

      mfs[label] += 1;

      if (atom.implicitHydrogens) {
        if (!mfs.H) mfs.H = 0;
        mfs.H += atom.implicitHydrogens;
      }
    }

    let mf = '';
    let keys = Object.keys(mfs).sort(atomSorter$1);

    for (let key of keys) {
      mf += key;
      if (mfs[key] > 1) mf += mfs[key];
    }

    if (charge > 0) {
      mf += `(+${charge > 1 ? charge : ''})`;
    } else if (charge < 0) {
      mf += `(${charge < -1 ? charge : '-'})`;
    }

    return mf;
  }

  let fragment;
  /**
   *
   * @param {OCL.Molecule} molecule
   * @param {object} [options={}]
   * @param {string} [opions.fromLabel='H']
   * @param {string} [opions.toLabel='H']
   * @param {string} [opions.minLength=1]
   * @param {string} [opions.maxLength=4]

   */

  function getPathsInfo(molecule) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    const {
      fromLabel = 'H',
      toLabel = 'H',
      minLength = 1,
      maxLength = 4
    } = options;
    const OCL = molecule.getOCL();

    if (!fragment) {
      fragment = new OCL.Molecule(0, 0);
    }

    let fromAtomicNumber = OCL.Molecule.getAtomicNoFromLabel(fromLabel);
    let toAtomicNumber = OCL.Molecule.getAtomicNoFromLabel(toLabel); // we need to find all the atoms 'fromLabel' and 'toLabel'

    let atomsInfo = getAtomsInfo(molecule);
    let pathLengthMatrix = getConnectivityMatrix(molecule, {
      pathLength: true
    });

    for (let from = 0; from < molecule.getAllAtoms(); from++) {
      atomsInfo[from].paths = [];

      for (let to = 0; to < molecule.getAllAtoms(); to++) {
        if (from !== to) {
          if (molecule.getAtomicNo(from) === fromAtomicNumber) {
            if (molecule.getAtomicNo(to) === toAtomicNumber) {
              let pathLength = pathLengthMatrix[from][to];

              if (pathLength >= minLength && pathLength <= maxLength) {
                atomsInfo[from].paths.push(getHoseCodesForPath(molecule, from, to, pathLength, atomsInfo[to].oclID));
              }
            }
          }
        }
      }
    }

    return atomsInfo;
  }

  /**
   * Get the shortest path between each pair of atoms in the molecule
   * @param {OCL.Molecule} molecule
   * @param {object} [options={}]
   * @param {string} [opions.fromLabel='H']
   * @param {string} [opions.toLabel='H']
   * @param {string} [opions.maxLength=4]
   * @returns {Array<Array>} A matrix containing on each cell (i,j) the shortest path from atom i to atom j
   */
  function getShortestPaths(molecule) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    const OCL = molecule.getOCL();
    const {
      fromLabel = '',
      toLabel = '',
      maxLength = 3
    } = options;
    let fromAtomicNumber = OCL.Molecule.getAtomicNoFromLabel(fromLabel);
    let toAtomicNumber = OCL.Molecule.getAtomicNoFromLabel(toLabel);
    const nbAtoms = molecule.getAllAtoms();
    let allShortestPaths = new Array(nbAtoms);

    for (let i = 0; i < nbAtoms; i++) {
      allShortestPaths[i] = new Array(nbAtoms);
    }

    for (let from = 0; from < nbAtoms; from++) {
      allShortestPaths[from][from] = [from];

      for (let to = from + 1; to < nbAtoms; to++) {
        if ((fromAtomicNumber === 0 || molecule.getAtomicNo(from) === fromAtomicNumber) && (toAtomicNumber === 0 || molecule.getAtomicNo(to) === toAtomicNumber)) {
          let path = [];
          molecule.getPath(path, from, to, maxLength);

          if (path.length) {
            allShortestPaths[from][to] = path.slice();
            allShortestPaths[to][from] = path.reverse();
          } else {
            allShortestPaths[from][to] = null;
            allShortestPaths[to][from] = null;
          }
        } else {
          allShortestPaths[from][to] = null;
          allShortestPaths[to][from] = null;
        }
      }
    }

    return allShortestPaths;
  }

  /**
   * Ensure that the data is string. If it is an ArrayBuffer it will be converted to string using TextDecoder.
   * @param blob
   * @param options
   * @returns
   */
  function ensureString(blob) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (typeof blob === 'string') {
      return blob;
    }

    if (ArrayBuffer.isView(blob) || blob instanceof ArrayBuffer) {
      const {
        encoding = guessEncoding(blob)
      } = options;
      const decoder = new TextDecoder(encoding);
      return decoder.decode(blob);
    }

    throw new TypeError(`blob must be a string, ArrayBuffer or ArrayBufferView`);
  }

  function guessEncoding(blob) {
    const uint8 = ArrayBuffer.isView(blob) ? new Uint8Array(blob.buffer, blob.byteOffset, blob.byteLength) : new Uint8Array(blob);

    if (uint8.length >= 2) {
      if (uint8[0] === 0xfe && uint8[1] === 0xff) {
        return 'utf-16be';
      }

      if (uint8[0] === 0xff && uint8[1] === 0xfe) {
        return 'utf-16le';
      }
    }

    return 'utf-8';
  }

  var papaparse_min = {exports: {}};

  /* @license
  Papa Parse
  v5.3.1
  https://github.com/mholt/PapaParse
  License: MIT
  */

  (function (module, exports) {
    !function (e, t) {
      module.exports = t() ;
    }(commonjsGlobal, function s() {

      var f = "undefined" != typeof self ? self : "undefined" != typeof window ? window : void 0 !== f ? f : {};
      var n = !f.document && !!f.postMessage,
          o = n && /blob:/i.test((f.location || {}).protocol),
          a = {},
          h = 0,
          b = {
        parse: function (e, t) {
          var i = (t = t || {}).dynamicTyping || !1;
          M(i) && (t.dynamicTypingFunction = i, i = {});

          if (t.dynamicTyping = i, t.transform = !!M(t.transform) && t.transform, t.worker && b.WORKERS_SUPPORTED) {
            var r = function () {
              if (!b.WORKERS_SUPPORTED) return !1;
              var e = (i = f.URL || f.webkitURL || null, r = s.toString(), b.BLOB_URL || (b.BLOB_URL = i.createObjectURL(new Blob(["(", r, ")();"], {
                type: "text/javascript"
              })))),
                  t = new f.Worker(e);
              var i, r;
              return t.onmessage = _, t.id = h++, a[t.id] = t;
            }();

            return r.userStep = t.step, r.userChunk = t.chunk, r.userComplete = t.complete, r.userError = t.error, t.step = M(t.step), t.chunk = M(t.chunk), t.complete = M(t.complete), t.error = M(t.error), delete t.worker, void r.postMessage({
              input: e,
              config: t,
              workerId: r.id
            });
          }

          var n = null;
          b.NODE_STREAM_INPUT, "string" == typeof e ? n = t.download ? new l(t) : new p(t) : !0 === e.readable && M(e.read) && M(e.on) ? n = new g(t) : (f.File && e instanceof File || e instanceof Object) && (n = new c(t));
          return n.stream(e);
        },
        unparse: function (e, t) {
          var n = !1,
              _ = !0,
              m = ",",
              y = "\r\n",
              s = '"',
              a = s + s,
              i = !1,
              r = null,
              o = !1;

          !function () {
            if ("object" != typeof t) return;
            "string" != typeof t.delimiter || b.BAD_DELIMITERS.filter(function (e) {
              return -1 !== t.delimiter.indexOf(e);
            }).length || (m = t.delimiter);
            ("boolean" == typeof t.quotes || "function" == typeof t.quotes || Array.isArray(t.quotes)) && (n = t.quotes);
            "boolean" != typeof t.skipEmptyLines && "string" != typeof t.skipEmptyLines || (i = t.skipEmptyLines);
            "string" == typeof t.newline && (y = t.newline);
            "string" == typeof t.quoteChar && (s = t.quoteChar);
            "boolean" == typeof t.header && (_ = t.header);

            if (Array.isArray(t.columns)) {
              if (0 === t.columns.length) throw new Error("Option columns is empty");
              r = t.columns;
            }

            void 0 !== t.escapeChar && (a = t.escapeChar + s);
            "boolean" == typeof t.escapeFormulae && (o = t.escapeFormulae);
          }();
          var h = new RegExp(j(s), "g");
          "string" == typeof e && (e = JSON.parse(e));

          if (Array.isArray(e)) {
            if (!e.length || Array.isArray(e[0])) return u(null, e, i);
            if ("object" == typeof e[0]) return u(r || Object.keys(e[0]), e, i);
          } else if ("object" == typeof e) return "string" == typeof e.data && (e.data = JSON.parse(e.data)), Array.isArray(e.data) && (e.fields || (e.fields = e.meta && e.meta.fields), e.fields || (e.fields = Array.isArray(e.data[0]) ? e.fields : "object" == typeof e.data[0] ? Object.keys(e.data[0]) : []), Array.isArray(e.data[0]) || "object" == typeof e.data[0] || (e.data = [e.data])), u(e.fields || [], e.data || [], i);

          throw new Error("Unable to serialize unrecognized input");

          function u(e, t, i) {
            var r = "";
            "string" == typeof e && (e = JSON.parse(e)), "string" == typeof t && (t = JSON.parse(t));
            var n = Array.isArray(e) && 0 < e.length,
                s = !Array.isArray(t[0]);

            if (n && _) {
              for (var a = 0; a < e.length; a++) 0 < a && (r += m), r += v(e[a], a);

              0 < t.length && (r += y);
            }

            for (var o = 0; o < t.length; o++) {
              var h = n ? e.length : t[o].length,
                  u = !1,
                  f = n ? 0 === Object.keys(t[o]).length : 0 === t[o].length;

              if (i && !n && (u = "greedy" === i ? "" === t[o].join("").trim() : 1 === t[o].length && 0 === t[o][0].length), "greedy" === i && n) {
                for (var d = [], l = 0; l < h; l++) {
                  var c = s ? e[l] : l;
                  d.push(t[o][c]);
                }

                u = "" === d.join("").trim();
              }

              if (!u) {
                for (var p = 0; p < h; p++) {
                  0 < p && !f && (r += m);
                  var g = n && s ? e[p] : p;
                  r += v(t[o][g], p);
                }

                o < t.length - 1 && (!i || 0 < h && !f) && (r += y);
              }
            }

            return r;
          }

          function v(e, t) {
            if (null == e) return "";
            if (e.constructor === Date) return JSON.stringify(e).slice(1, 25);
            !0 === o && "string" == typeof e && null !== e.match(/^[=+\-@].*$/) && (e = "'" + e);

            var i = e.toString().replace(h, a),
                r = "boolean" == typeof n && n || "function" == typeof n && n(e, t) || Array.isArray(n) && n[t] || function (e, t) {
              for (var i = 0; i < t.length; i++) if (-1 < e.indexOf(t[i])) return !0;

              return !1;
            }(i, b.BAD_DELIMITERS) || -1 < i.indexOf(m) || " " === i.charAt(0) || " " === i.charAt(i.length - 1);

            return r ? s + i + s : i;
          }
        }
      };

      if (b.RECORD_SEP = String.fromCharCode(30), b.UNIT_SEP = String.fromCharCode(31), b.BYTE_ORDER_MARK = "\ufeff", b.BAD_DELIMITERS = ["\r", "\n", '"', b.BYTE_ORDER_MARK], b.WORKERS_SUPPORTED = !n && !!f.Worker, b.NODE_STREAM_INPUT = 1, b.LocalChunkSize = 10485760, b.RemoteChunkSize = 5242880, b.DefaultDelimiter = ",", b.Parser = E, b.ParserHandle = i, b.NetworkStreamer = l, b.FileStreamer = c, b.StringStreamer = p, b.ReadableStreamStreamer = g, f.jQuery) {
        var d = f.jQuery;

        d.fn.parse = function (o) {
          var i = o.config || {},
              h = [];
          return this.each(function (e) {
            if (!("INPUT" === d(this).prop("tagName").toUpperCase() && "file" === d(this).attr("type").toLowerCase() && f.FileReader) || !this.files || 0 === this.files.length) return !0;

            for (var t = 0; t < this.files.length; t++) h.push({
              file: this.files[t],
              inputElem: this,
              instanceConfig: d.extend({}, i)
            });
          }), e(), this;

          function e() {
            if (0 !== h.length) {
              var e,
                  t,
                  i,
                  r,
                  n = h[0];

              if (M(o.before)) {
                var s = o.before(n.file, n.inputElem);

                if ("object" == typeof s) {
                  if ("abort" === s.action) return e = "AbortError", t = n.file, i = n.inputElem, r = s.reason, void (M(o.error) && o.error({
                    name: e
                  }, t, i, r));
                  if ("skip" === s.action) return void u();
                  "object" == typeof s.config && (n.instanceConfig = d.extend(n.instanceConfig, s.config));
                } else if ("skip" === s) return void u();
              }

              var a = n.instanceConfig.complete;
              n.instanceConfig.complete = function (e) {
                M(a) && a(e, n.file, n.inputElem), u();
              }, b.parse(n.file, n.instanceConfig);
            } else M(o.complete) && o.complete();
          }

          function u() {
            h.splice(0, 1), e();
          }
        };
      }

      function u(e) {
        this._handle = null, this._finished = !1, this._completed = !1, this._halted = !1, this._input = null, this._baseIndex = 0, this._partialLine = "", this._rowCount = 0, this._start = 0, this._nextChunk = null, this.isFirstChunk = !0, this._completeResults = {
          data: [],
          errors: [],
          meta: {}
        }, function (e) {
          var t = w(e);
          t.chunkSize = parseInt(t.chunkSize), e.step || e.chunk || (t.chunkSize = null);
          this._handle = new i(t), (this._handle.streamer = this)._config = t;
        }.call(this, e), this.parseChunk = function (e, t) {
          if (this.isFirstChunk && M(this._config.beforeFirstChunk)) {
            var i = this._config.beforeFirstChunk(e);

            void 0 !== i && (e = i);
          }

          this.isFirstChunk = !1, this._halted = !1;
          var r = this._partialLine + e;
          this._partialLine = "";

          var n = this._handle.parse(r, this._baseIndex, !this._finished);

          if (!this._handle.paused() && !this._handle.aborted()) {
            var s = n.meta.cursor;
            this._finished || (this._partialLine = r.substring(s - this._baseIndex), this._baseIndex = s), n && n.data && (this._rowCount += n.data.length);
            var a = this._finished || this._config.preview && this._rowCount >= this._config.preview;
            if (o) f.postMessage({
              results: n,
              workerId: b.WORKER_ID,
              finished: a
            });else if (M(this._config.chunk) && !t) {
              if (this._config.chunk(n, this._handle), this._handle.paused() || this._handle.aborted()) return void (this._halted = !0);
              n = void 0, this._completeResults = void 0;
            }
            return this._config.step || this._config.chunk || (this._completeResults.data = this._completeResults.data.concat(n.data), this._completeResults.errors = this._completeResults.errors.concat(n.errors), this._completeResults.meta = n.meta), this._completed || !a || !M(this._config.complete) || n && n.meta.aborted || (this._config.complete(this._completeResults, this._input), this._completed = !0), a || n && n.meta.paused || this._nextChunk(), n;
          }

          this._halted = !0;
        }, this._sendError = function (e) {
          M(this._config.error) ? this._config.error(e) : o && this._config.error && f.postMessage({
            workerId: b.WORKER_ID,
            error: e,
            finished: !1
          });
        };
      }

      function l(e) {
        var r;
        (e = e || {}).chunkSize || (e.chunkSize = b.RemoteChunkSize), u.call(this, e), this._nextChunk = n ? function () {
          this._readChunk(), this._chunkLoaded();
        } : function () {
          this._readChunk();
        }, this.stream = function (e) {
          this._input = e, this._nextChunk();
        }, this._readChunk = function () {
          if (this._finished) this._chunkLoaded();else {
            if (r = new XMLHttpRequest(), this._config.withCredentials && (r.withCredentials = this._config.withCredentials), n || (r.onload = v(this._chunkLoaded, this), r.onerror = v(this._chunkError, this)), r.open(this._config.downloadRequestBody ? "POST" : "GET", this._input, !n), this._config.downloadRequestHeaders) {
              var e = this._config.downloadRequestHeaders;

              for (var t in e) r.setRequestHeader(t, e[t]);
            }

            if (this._config.chunkSize) {
              var i = this._start + this._config.chunkSize - 1;
              r.setRequestHeader("Range", "bytes=" + this._start + "-" + i);
            }

            try {
              r.send(this._config.downloadRequestBody);
            } catch (e) {
              this._chunkError(e.message);
            }

            n && 0 === r.status && this._chunkError();
          }
        }, this._chunkLoaded = function () {
          4 === r.readyState && (r.status < 200 || 400 <= r.status ? this._chunkError() : (this._start += this._config.chunkSize ? this._config.chunkSize : r.responseText.length, this._finished = !this._config.chunkSize || this._start >= function (e) {
            var t = e.getResponseHeader("Content-Range");
            if (null === t) return -1;
            return parseInt(t.substring(t.lastIndexOf("/") + 1));
          }(r), this.parseChunk(r.responseText)));
        }, this._chunkError = function (e) {
          var t = r.statusText || e;

          this._sendError(new Error(t));
        };
      }

      function c(e) {
        var r, n;
        (e = e || {}).chunkSize || (e.chunkSize = b.LocalChunkSize), u.call(this, e);
        var s = "undefined" != typeof FileReader;
        this.stream = function (e) {
          this._input = e, n = e.slice || e.webkitSlice || e.mozSlice, s ? ((r = new FileReader()).onload = v(this._chunkLoaded, this), r.onerror = v(this._chunkError, this)) : r = new FileReaderSync(), this._nextChunk();
        }, this._nextChunk = function () {
          this._finished || this._config.preview && !(this._rowCount < this._config.preview) || this._readChunk();
        }, this._readChunk = function () {
          var e = this._input;

          if (this._config.chunkSize) {
            var t = Math.min(this._start + this._config.chunkSize, this._input.size);
            e = n.call(e, this._start, t);
          }

          var i = r.readAsText(e, this._config.encoding);
          s || this._chunkLoaded({
            target: {
              result: i
            }
          });
        }, this._chunkLoaded = function (e) {
          this._start += this._config.chunkSize, this._finished = !this._config.chunkSize || this._start >= this._input.size, this.parseChunk(e.target.result);
        }, this._chunkError = function () {
          this._sendError(r.error);
        };
      }

      function p(e) {
        var i;
        u.call(this, e = e || {}), this.stream = function (e) {
          return i = e, this._nextChunk();
        }, this._nextChunk = function () {
          if (!this._finished) {
            var e,
                t = this._config.chunkSize;
            return t ? (e = i.substring(0, t), i = i.substring(t)) : (e = i, i = ""), this._finished = !i, this.parseChunk(e);
          }
        };
      }

      function g(e) {
        u.call(this, e = e || {});
        var t = [],
            i = !0,
            r = !1;
        this.pause = function () {
          u.prototype.pause.apply(this, arguments), this._input.pause();
        }, this.resume = function () {
          u.prototype.resume.apply(this, arguments), this._input.resume();
        }, this.stream = function (e) {
          this._input = e, this._input.on("data", this._streamData), this._input.on("end", this._streamEnd), this._input.on("error", this._streamError);
        }, this._checkIsFinished = function () {
          r && 1 === t.length && (this._finished = !0);
        }, this._nextChunk = function () {
          this._checkIsFinished(), t.length ? this.parseChunk(t.shift()) : i = !0;
        }, this._streamData = v(function (e) {
          try {
            t.push("string" == typeof e ? e : e.toString(this._config.encoding)), i && (i = !1, this._checkIsFinished(), this.parseChunk(t.shift()));
          } catch (e) {
            this._streamError(e);
          }
        }, this), this._streamError = v(function (e) {
          this._streamCleanUp(), this._sendError(e);
        }, this), this._streamEnd = v(function () {
          this._streamCleanUp(), r = !0, this._streamData("");
        }, this), this._streamCleanUp = v(function () {
          this._input.removeListener("data", this._streamData), this._input.removeListener("end", this._streamEnd), this._input.removeListener("error", this._streamError);
        }, this);
      }

      function i(m) {
        var a,
            o,
            h,
            r = Math.pow(2, 53),
            n = -r,
            s = /^\s*-?(\d+\.?|\.\d+|\d+\.\d+)([eE][-+]?\d+)?\s*$/,
            u = /^(\d{4}-[01]\d-[0-3]\dT[0-2]\d:[0-5]\d:[0-5]\d\.\d+([+-][0-2]\d:[0-5]\d|Z))|(\d{4}-[01]\d-[0-3]\dT[0-2]\d:[0-5]\d:[0-5]\d([+-][0-2]\d:[0-5]\d|Z))|(\d{4}-[01]\d-[0-3]\dT[0-2]\d:[0-5]\d([+-][0-2]\d:[0-5]\d|Z))$/,
            t = this,
            i = 0,
            f = 0,
            d = !1,
            e = !1,
            l = [],
            c = {
          data: [],
          errors: [],
          meta: {}
        };

        if (M(m.step)) {
          var p = m.step;

          m.step = function (e) {
            if (c = e, _()) g();else {
              if (g(), 0 === c.data.length) return;
              i += e.data.length, m.preview && i > m.preview ? o.abort() : (c.data = c.data[0], p(c, t));
            }
          };
        }

        function y(e) {
          return "greedy" === m.skipEmptyLines ? "" === e.join("").trim() : 1 === e.length && 0 === e[0].length;
        }

        function g() {
          if (c && h && (k("Delimiter", "UndetectableDelimiter", "Unable to auto-detect delimiting character; defaulted to '" + b.DefaultDelimiter + "'"), h = !1), m.skipEmptyLines) for (var e = 0; e < c.data.length; e++) y(c.data[e]) && c.data.splice(e--, 1);
          return _() && function () {
            if (!c) return;

            function e(e, t) {
              M(m.transformHeader) && (e = m.transformHeader(e, t)), l.push(e);
            }

            if (Array.isArray(c.data[0])) {
              for (var t = 0; _() && t < c.data.length; t++) c.data[t].forEach(e);

              c.data.splice(0, 1);
            } else c.data.forEach(e);
          }(), function () {
            if (!c || !m.header && !m.dynamicTyping && !m.transform) return c;

            function e(e, t) {
              var i,
                  r = m.header ? {} : [];

              for (i = 0; i < e.length; i++) {
                var n = i,
                    s = e[i];
                m.header && (n = i >= l.length ? "__parsed_extra" : l[i]), m.transform && (s = m.transform(s, n)), s = v(n, s), "__parsed_extra" === n ? (r[n] = r[n] || [], r[n].push(s)) : r[n] = s;
              }

              return m.header && (i > l.length ? k("FieldMismatch", "TooManyFields", "Too many fields: expected " + l.length + " fields but parsed " + i, f + t) : i < l.length && k("FieldMismatch", "TooFewFields", "Too few fields: expected " + l.length + " fields but parsed " + i, f + t)), r;
            }

            var t = 1;
            !c.data.length || Array.isArray(c.data[0]) ? (c.data = c.data.map(e), t = c.data.length) : c.data = e(c.data, 0);
            m.header && c.meta && (c.meta.fields = l);
            return f += t, c;
          }();
        }

        function _() {
          return m.header && 0 === l.length;
        }

        function v(e, t) {
          return i = e, m.dynamicTypingFunction && void 0 === m.dynamicTyping[i] && (m.dynamicTyping[i] = m.dynamicTypingFunction(i)), !0 === (m.dynamicTyping[i] || m.dynamicTyping) ? "true" === t || "TRUE" === t || "false" !== t && "FALSE" !== t && (function (e) {
            if (s.test(e)) {
              var t = parseFloat(e);
              if (n < t && t < r) return !0;
            }

            return !1;
          }(t) ? parseFloat(t) : u.test(t) ? new Date(t) : "" === t ? null : t) : t;
          var i;
        }

        function k(e, t, i, r) {
          var n = {
            type: e,
            code: t,
            message: i
          };
          void 0 !== r && (n.row = r), c.errors.push(n);
        }

        this.parse = function (e, t, i) {
          var r = m.quoteChar || '"';
          if (m.newline || (m.newline = function (e, t) {
            e = e.substring(0, 1048576);
            var i = new RegExp(j(t) + "([^]*?)" + j(t), "gm"),
                r = (e = e.replace(i, "")).split("\r"),
                n = e.split("\n"),
                s = 1 < n.length && n[0].length < r[0].length;
            if (1 === r.length || s) return "\n";

            for (var a = 0, o = 0; o < r.length; o++) "\n" === r[o][0] && a++;

            return a >= r.length / 2 ? "\r\n" : "\r";
          }(e, r)), h = !1, m.delimiter) M(m.delimiter) && (m.delimiter = m.delimiter(e), c.meta.delimiter = m.delimiter);else {
            var n = function (e, t, i, r, n) {
              var s, a, o, h;
              n = n || [",", "\t", "|", ";", b.RECORD_SEP, b.UNIT_SEP];

              for (var u = 0; u < n.length; u++) {
                var f = n[u],
                    d = 0,
                    l = 0,
                    c = 0;
                o = void 0;

                for (var p = new E({
                  comments: r,
                  delimiter: f,
                  newline: t,
                  preview: 10
                }).parse(e), g = 0; g < p.data.length; g++) if (i && y(p.data[g])) c++;else {
                  var _ = p.data[g].length;
                  l += _, void 0 !== o ? 0 < _ && (d += Math.abs(_ - o), o = _) : o = _;
                }

                0 < p.data.length && (l /= p.data.length - c), (void 0 === a || d <= a) && (void 0 === h || h < l) && 1.99 < l && (a = d, s = f, h = l);
              }

              return {
                successful: !!(m.delimiter = s),
                bestDelimiter: s
              };
            }(e, m.newline, m.skipEmptyLines, m.comments, m.delimitersToGuess);

            n.successful ? m.delimiter = n.bestDelimiter : (h = !0, m.delimiter = b.DefaultDelimiter), c.meta.delimiter = m.delimiter;
          }
          var s = w(m);
          return m.preview && m.header && s.preview++, a = e, o = new E(s), c = o.parse(a, t, i), g(), d ? {
            meta: {
              paused: !0
            }
          } : c || {
            meta: {
              paused: !1
            }
          };
        }, this.paused = function () {
          return d;
        }, this.pause = function () {
          d = !0, o.abort(), a = M(m.chunk) ? "" : a.substring(o.getCharIndex());
        }, this.resume = function () {
          t.streamer._halted ? (d = !1, t.streamer.parseChunk(a, !0)) : setTimeout(t.resume, 3);
        }, this.aborted = function () {
          return e;
        }, this.abort = function () {
          e = !0, o.abort(), c.meta.aborted = !0, M(m.complete) && m.complete(c), a = "";
        };
      }

      function j(e) {
        return e.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
      }

      function E(e) {
        var S,
            O = (e = e || {}).delimiter,
            x = e.newline,
            I = e.comments,
            T = e.step,
            D = e.preview,
            A = e.fastMode,
            L = S = void 0 === e.quoteChar ? '"' : e.quoteChar;
        if (void 0 !== e.escapeChar && (L = e.escapeChar), ("string" != typeof O || -1 < b.BAD_DELIMITERS.indexOf(O)) && (O = ","), I === O) throw new Error("Comment character same as delimiter");
        !0 === I ? I = "#" : ("string" != typeof I || -1 < b.BAD_DELIMITERS.indexOf(I)) && (I = !1), "\n" !== x && "\r" !== x && "\r\n" !== x && (x = "\n");
        var F = 0,
            z = !1;
        this.parse = function (r, t, i) {
          if ("string" != typeof r) throw new Error("Input must be a string");
          var n = r.length,
              e = O.length,
              s = x.length,
              a = I.length,
              o = M(T),
              h = [],
              u = [],
              f = [],
              d = F = 0;
          if (!r) return C();

          if (A || !1 !== A && -1 === r.indexOf(S)) {
            for (var l = r.split(x), c = 0; c < l.length; c++) {
              if (f = l[c], F += f.length, c !== l.length - 1) F += x.length;else if (i) return C();

              if (!I || f.substring(0, a) !== I) {
                if (o) {
                  if (h = [], k(f.split(O)), R(), z) return C();
                } else k(f.split(O));

                if (D && D <= c) return h = h.slice(0, D), C(!0);
              }
            }

            return C();
          }

          for (var p = r.indexOf(O, F), g = r.indexOf(x, F), _ = new RegExp(j(L) + j(S), "g"), m = r.indexOf(S, F);;) if (r[F] !== S) {
            if (I && 0 === f.length && r.substring(F, F + a) === I) {
              if (-1 === g) return C();
              F = g + s, g = r.indexOf(x, F), p = r.indexOf(O, F);
            } else if (-1 !== p && (p < g || -1 === g)) f.push(r.substring(F, p)), F = p + e, p = r.indexOf(O, F);else {
              if (-1 === g) break;
              if (f.push(r.substring(F, g)), w(g + s), o && (R(), z)) return C();
              if (D && h.length >= D) return C(!0);
            }
          } else for (m = F, F++;;) {
            if (-1 === (m = r.indexOf(S, m + 1))) return i || u.push({
              type: "Quotes",
              code: "MissingQuotes",
              message: "Quoted field unterminated",
              row: h.length,
              index: F
            }), E();
            if (m === n - 1) return E(r.substring(F, m).replace(_, S));

            if (S !== L || r[m + 1] !== L) {
              if (S === L || 0 === m || r[m - 1] !== L) {
                -1 !== p && p < m + 1 && (p = r.indexOf(O, m + 1)), -1 !== g && g < m + 1 && (g = r.indexOf(x, m + 1));
                var y = b(-1 === g ? p : Math.min(p, g));

                if (r[m + 1 + y] === O) {
                  f.push(r.substring(F, m).replace(_, S)), r[F = m + 1 + y + e] !== S && (m = r.indexOf(S, F)), p = r.indexOf(O, F), g = r.indexOf(x, F);
                  break;
                }

                var v = b(g);

                if (r.substring(m + 1 + v, m + 1 + v + s) === x) {
                  if (f.push(r.substring(F, m).replace(_, S)), w(m + 1 + v + s), p = r.indexOf(O, F), m = r.indexOf(S, F), o && (R(), z)) return C();
                  if (D && h.length >= D) return C(!0);
                  break;
                }

                u.push({
                  type: "Quotes",
                  code: "InvalidQuotes",
                  message: "Trailing quote on quoted field is malformed",
                  row: h.length,
                  index: F
                }), m++;
              }
            } else m++;
          }

          return E();

          function k(e) {
            h.push(e), d = F;
          }

          function b(e) {
            var t = 0;

            if (-1 !== e) {
              var i = r.substring(m + 1, e);
              i && "" === i.trim() && (t = i.length);
            }

            return t;
          }

          function E(e) {
            return i || (void 0 === e && (e = r.substring(F)), f.push(e), F = n, k(f), o && R()), C();
          }

          function w(e) {
            F = e, k(f), f = [], g = r.indexOf(x, F);
          }

          function C(e) {
            return {
              data: h,
              errors: u,
              meta: {
                delimiter: O,
                linebreak: x,
                aborted: z,
                truncated: !!e,
                cursor: d + (t || 0)
              }
            };
          }

          function R() {
            T(C()), h = [], u = [];
          }
        }, this.abort = function () {
          z = !0;
        }, this.getCharIndex = function () {
          return F;
        };
      }

      function _(e) {
        var t = e.data,
            i = a[t.workerId],
            r = !1;
        if (t.error) i.userError(t.error, t.file);else if (t.results && t.results.data) {
          var n = {
            abort: function () {
              r = !0, m(t.workerId, {
                data: [],
                errors: [],
                meta: {
                  aborted: !0
                }
              });
            },
            pause: y,
            resume: y
          };

          if (M(i.userStep)) {
            for (var s = 0; s < t.results.data.length && (i.userStep({
              data: t.results.data[s],
              errors: t.results.errors,
              meta: t.results.meta
            }, n), !r); s++);

            delete t.results;
          } else M(i.userChunk) && (i.userChunk(t.results, n, t.file), delete t.results);
        }
        t.finished && !r && m(t.workerId, t.results);
      }

      function m(e, t) {
        var i = a[e];
        M(i.userComplete) && i.userComplete(t), i.terminate(), delete a[e];
      }

      function y() {
        throw new Error("Not implemented.");
      }

      function w(e) {
        if ("object" != typeof e || null === e) return e;
        var t = Array.isArray(e) ? [] : {};

        for (var i in e) t[i] = w(e[i]);

        return t;
      }

      function v(e, t) {
        return function () {
          e.apply(t, arguments);
        };
      }

      function M(e) {
        return "function" == typeof e;
      }

      return o && (f.onmessage = function (e) {
        var t = e.data;
        void 0 === b.WORKER_ID && t && (b.WORKER_ID = t.workerId);
        if ("string" == typeof t.input) f.postMessage({
          workerId: b.WORKER_ID,
          results: b.parse(t.input, t.config),
          finished: !0
        });else if (f.File && t.input instanceof File || t.input instanceof Object) {
          var i = b.parse(t.input, t.config);
          i && f.postMessage({
            workerId: b.WORKER_ID,
            results: i,
            finished: !0
          });
        }
      }), (l.prototype = Object.create(u.prototype)).constructor = l, (c.prototype = Object.create(u.prototype)).constructor = c, (p.prototype = Object.create(p.prototype)).constructor = p, (g.prototype = Object.create(u.prototype)).constructor = g, b;
    });
  })(papaparse_min);

  var Papa = papaparse_min.exports;

  function getMoleculeCreators(Molecule) {
    const fields = new Map();
    fields.set('oclid', Molecule.fromIDCode);
    fields.set('idcode', Molecule.fromIDCode);
    fields.set('smiles', Molecule.fromSmiles);
    fields.set('molfile', Molecule.fromMolfile);
    return fields;
  }

  const defaultCSVOptions = {
    header: true,
    dynamicTyping: true,
    skipEmptyLines: true
  };
  async function appendCSV(moleculesDB, csv) {
    let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
    const {
      onStep
    } = options;
    csv = ensureString(csv);
    const moleculeCreators = getMoleculeCreators(moleculesDB.OCL.Molecule);

    if (typeof csv !== 'string') {
      throw new TypeError('csv must be a string');
    }

    options = { ...defaultCSVOptions,
      ...options
    };
    const parsed = Papa.parse(csv, options);
    const fields = parsed.meta.fields;
    const stats = new Array(fields.length);
    const firstElement = parsed.data[0];
    let moleculeCreator, moleculeField;

    for (let i = 0; i < fields.length; i++) {
      stats[i] = {
        label: fields[i],
        isNumeric: typeof firstElement[fields[i]] === 'number'
      };
      const lowerField = fields[i].toLowerCase();

      if (moleculeCreators.has(lowerField)) {
        moleculeCreator = moleculeCreators.get(lowerField);
        moleculeField = fields[i];
      }
    }

    if (!moleculeCreator) {
      throw new Error('this document does not contain any molecule field');
    }

    moleculesDB.statistics = stats;

    for (let i = 0; i < parsed.data.length; i++) {
      moleculesDB.pushEntry(moleculeCreator(parsed.data[i][moleculeField]), parsed.data[i]);

      if (onStep) {
        await onStep(i + 1, parsed.data.length);
      }
    }
  }

  function appendColor(moleculesDB) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    const {
      dataLabel,
      propertyLabel,
      minValue,
      maxValue,
      minHue = 0,
      maxHue = 360,
      saturation = 65,
      lightness = 65,
      colorLabel = 'color'
    } = options;
    const db = moleculesDB.getDB();
    let values;

    if (dataLabel) {
      values = db.map(result => result.data.map(datum => ({
        value: datum[dataLabel],
        data: datum
      }))).flat();
    } else if (propertyLabel) {
      values = db.map(result => result.data.map(datum => ({
        value: result.properties[propertyLabel],
        data: datum
      }))).flat();
    } else {
      values = db.map(result => result.data.map(datum => ({
        value: undefined,
        data: datum
      }))).flat();
    }

    if (minValue !== undefined) {
      values = values.forEach(value => {
        if (value.value !== undefined && value.value < minValue) {
          value.value = minValue;
        }
      });
    }

    if (maxValue !== undefined) {
      values = values.forEach(value => {
        if (value.value !== undefined && value.value > maxValue) {
          value.value = maxValue;
        }
      });
    }

    const definedValues = values.filter(value => value.value !== undefined);
    const min = Math.min(...definedValues.map(value => value.value));
    const max = Math.max(...definedValues.map(value => value.value));

    for (let value of values) {
      if (value.value !== undefined) {
        value.data[colorLabel] = `hsl(${Math.floor((value.value - min) / (max - min) * (maxHue - minHue) + minHue)},${saturation}%,${lightness}%)`;
      } else {
        value.data.color = 'black';
      }
    }
  }

  function getEntriesBoundaries(string, substring, eol) {
    const res = [];
    let previous = 0;
    let next = 0;

    while (next !== -1) {
      next = string.indexOf(substring, previous);

      if (next !== -1) {
        res.push([previous, next]);
        previous = next = string.indexOf(eol, next + substring.length) + eol.length;
      } else {
        res.push([previous, string.length]);
      }
    }

    return res;
  }

  function parse(sdf) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    const {
      include,
      exclude,
      filter,
      modifiers = {},
      forEach = {},
      dynamicTyping = true
    } = options;

    if (typeof sdf !== 'string') {
      throw new TypeError('Parameter "sdf" must be a string');
    }

    let eol = '\n';

    if (options.mixedEOL) {
      sdf = sdf.replace(/\r\n/g, '\n');
      sdf = sdf.replace(/\r/g, '\n');
    } else {
      // we will find the delimiter in order to be much faster and not use regular expression
      let header = sdf.substr(0, 1000);

      if (header.indexOf('\r\n') > -1) {
        eol = '\r\n';
      } else if (header.indexOf('\r') > -1) {
        eol = '\r';
      }
    }

    let entriesBoundaries = getEntriesBoundaries(sdf, `${eol}$$$$`, eol);
    let molecules = [];
    let labels = {};
    let start = Date.now();

    for (let i = 0; i < entriesBoundaries.length; i++) {
      let sdfPart = sdf.substring(...entriesBoundaries[i]);
      let parts = sdfPart.split(`${eol}>`);

      if (parts.length > 0 && parts[0].length > 5) {
        let molecule = {};
        let currentLabels = [];
        molecule.molfile = parts[0] + eol;

        for (let j = 1; j < parts.length; j++) {
          let lines = parts[j].split(eol);
          let from = lines[0].indexOf('<');
          let to = lines[0].indexOf('>');
          let label = lines[0].substring(from + 1, to);
          currentLabels.push(label);

          if (!labels[label]) {
            labels[label] = {
              counter: 0,
              isNumeric: dynamicTyping,
              keep: false
            };

            if ((!exclude || exclude.indexOf(label) === -1) && (!include || include.indexOf(label) > -1)) {
              labels[label].keep = true;

              if (modifiers[label]) {
                labels[label].modifier = modifiers[label];
              }

              if (forEach[label]) {
                labels[label].forEach = forEach[label];
              }
            }
          }

          if (labels[label].keep) {
            for (let k = 1; k < lines.length - 1; k++) {
              if (molecule[label]) {
                molecule[label] += eol + lines[k];
              } else {
                molecule[label] = lines[k];
              }
            }

            if (labels[label].modifier) {
              let modifiedValue = labels[label].modifier(molecule[label]);

              if (modifiedValue === undefined || modifiedValue === null) {
                delete molecule[label];
              } else {
                molecule[label] = modifiedValue;
              }
            }

            if (labels[label].isNumeric) {
              if (!isFinite(molecule[label]) || molecule[label].match(/^0[0-9]/)) {
                labels[label].isNumeric = false;
              }
            }
          }
        }

        if (!filter || filter(molecule)) {
          molecules.push(molecule); // only now we can increase the counter

          for (let j = 0; j < currentLabels.length; j++) {
            labels[currentLabels[j]].counter++;
          }
        }
      }
    } // all numeric fields should be converted to numbers


    for (let label in labels) {
      let currentLabel = labels[label];

      if (currentLabel.isNumeric) {
        currentLabel.minValue = Infinity;
        currentLabel.maxValue = -Infinity;

        for (let j = 0; j < molecules.length; j++) {
          if (molecules[j][label]) {
            let value = parseFloat(molecules[j][label]);
            molecules[j][label] = value;

            if (value > currentLabel.maxValue) {
              currentLabel.maxValue = value;
            }

            if (value < currentLabel.minValue) {
              currentLabel.minValue = value;
            }
          }
        }
      }
    } // we check that a label is in all the records


    for (let key in labels) {
      if (labels[key].counter === molecules.length) {
        labels[key].always = true;
      } else {
        labels[key].always = false;
      }
    }

    let statistics = [];

    for (let key in labels) {
      let statistic = labels[key];
      statistic.label = key;
      statistics.push(statistic);
    }

    return {
      time: Date.now() - start,
      molecules: molecules,
      labels: Object.keys(labels),
      statistics: statistics
    };
  }

  async function appendSDF(moleculesDB, sdf) {
    let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
    const {
      onStep
    } = options;
    sdf = ensureString(sdf);

    if (typeof sdf !== 'string') {
      throw new TypeError('sdf must be a string');
    }

    const parsed = parse(sdf);
    moleculesDB.statistics = parsed.statistics;

    for (let i = 0; i < parsed.molecules.length; i++) {
      const molecule = parsed.molecules[i];
      moleculesDB.pushEntry(moleculesDB.OCL.Molecule.fromMolfile(molecule.molfile), molecule);

      if (onStep) {
        await onStep(i + 1, parsed.molecules.length);
      }
    }
  }

  async function appendSmilesList(moleculesDB, text) {
    let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
    const {
      onStep
    } = options;
    text = ensureString(text);

    if (typeof text !== 'string') {
      throw new TypeError('text must be a string');
    }

    const smilesArray = text.split(/\r?\n/).map(line => line.trim()).filter(line => line);

    for (let i = 0; i < smilesArray.length; i++) {
      const oneSmiles = smilesArray[i];
      moleculesDB.pushEntry(moleculesDB.OCL.Molecule.fromSmiles(oneSmiles));

      if (onStep) {
        await onStep(i + 1, smilesArray.length);
      }
    }
  }

  function pushEntry(moleculesDB, molecule) {
    let data = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
    let moleculeInfo = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : {};
    // the following line could be the source of problems if the idCode version
    // changes
    let moleculeIDCode = moleculeInfo.idCode ? moleculeInfo.idCode : molecule.getIDCode();
    let entry = moleculesDB.db[moleculeIDCode];

    if (!entry) {
      // a new molecule
      entry = {
        molecule,
        properties: {},
        data: [],
        idCode: moleculeIDCode
      };
      moleculesDB.db[moleculeIDCode] = entry; // ensure helper arrays needed for substructure search

      molecule.ensureHelperArrays(moleculesDB.OCL.Molecule.cHelperRings);

      if (!moleculeInfo.index) {
        entry.index = molecule.getIndex();
      } else {
        entry.index = moleculeInfo.index;
      }

      let molecularFormula;

      if (!moleculeInfo.mw) {
        molecularFormula = molecule.getMolecularFormula();
        entry.properties.mw = molecularFormula.relativeWeight;
      } else {
        entry.properties.mw = moleculeInfo.mw;
      }

      if (moleculesDB.computeProperties) {
        if (!molecularFormula) {
          molecularFormula = molecule.getMolecularFormula();
        }

        const properties = new moleculesDB.OCL.MoleculeProperties(molecule);
        entry.properties.em = molecularFormula.absoluteWeight;
        entry.properties.mf = molecularFormula.formula;
        entry.properties.acceptorCount = properties.acceptorCount;
        entry.properties.donorCount = properties.donorCount;
        entry.properties.logP = properties.logP;
        entry.properties.logS = properties.logS;
        entry.properties.polarSurfaceArea = properties.polarSurfaceArea;
        entry.properties.rotatableBondCount = properties.rotatableBondCount;
        entry.properties.stereoCenterCount = properties.stereoCenterCount;
      }
    }

    entry.data.push(data);
  }

  function pushMoleculeInfo(moleculesDB, moleculeInfo) {
    let data = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

    if (typeof moleculeInfo !== 'object') {
      throw new Error('pushMoleculeInfo requires an object as first parameter');
    }

    const Molecule = moleculesDB.OCL.Molecule;
    let molecule;

    if (moleculeInfo.molfile) {
      molecule = Molecule.fromMolfile(moleculeInfo.molfile);
    }

    if (moleculeInfo.smiles) molecule = Molecule.fromSmiles(moleculeInfo.smiles);

    if (moleculeInfo.idCode) {
      if (moleculesDB.db[moleculeInfo.idCode]) {
        molecule = moleculesDB.db[moleculeInfo.idCode].molecule;
      } else {
        molecule = Molecule.fromIDCode(moleculeInfo.idCode, moleculeInfo.coordinates || false);
      }
    }

    if (molecule) {
      moleculesDB.pushEntry(molecule, data, moleculeInfo);
    }
  }

  function search(moleculesDB, query) {
    let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
    const {
      format = 'idCode',
      mode = 'substructure',
      flattenResult = true,
      keepMolecule = false,
      limit = Number.MAX_SAFE_INTEGER
    } = options;

    if (typeof query === 'string') {
      const moleculeCreators = getMoleculeCreators(moleculesDB.OCL.Molecule);
      query = moleculeCreators.get(format.toLowerCase())(query);
    } else if (!(query instanceof moleculesDB.OCL.Molecule)) {
      throw new TypeError('toSearch must be a Molecule or string');
    }

    let result;

    switch (mode.toLowerCase()) {
      case 'exact':
        result = exactSearch(moleculesDB, query);
        break;

      case 'substructure':
        result = subStructureSearch(moleculesDB, query);
        break;

      case 'similarity':
        result = similaritySearch(moleculesDB, query);
        break;

      default:
        throw new Error(`unknown search mode: ${options.mode}`);
    }

    return processResult(result, {
      flattenResult,
      keepMolecule,
      limit
    });
  }

  function exactSearch(moleculesDB, query) {
    const queryIDCode = query.getIDCode();
    let searchResult = moleculesDB.db[queryIDCode] ? [moleculesDB.db[queryIDCode]] : [];
    return searchResult;
  }

  function subStructureSearch(moleculesDB, query) {
    let resetFragment = false;

    if (!query.isFragment()) {
      resetFragment = true;
      query.setFragment(true);
    }

    const queryMW = getMW(query);
    const searchResult = [];

    if (query.getAllAtoms() === 0) {
      for (let idCode in moleculesDB.db) {
        searchResult.push(moleculesDB.db[idCode]);
      }
    } else {
      const queryIndex = query.getIndex();
      const searcher = moleculesDB.searcher;
      searcher.setFragment(query, queryIndex);

      for (let idCode in moleculesDB.db) {
        let entry = moleculesDB.db[idCode];
        searcher.setMolecule(entry.molecule, entry.index);

        if (searcher.isFragmentInMolecule()) {
          searchResult.push(entry);
        }
      }
    }

    searchResult.sort((a, b) => {
      return Math.abs(queryMW - a.properties.mw) - Math.abs(queryMW - b.properties.mw);
    });

    if (resetFragment) {
      query.setFragment(false);
    }

    return searchResult;
  }

  function similaritySearch(moleculesDB, query) {
    const queryIndex = query.getIndex();
    const queryMW = getMW(query);
    const queryIdCode = query.getIDCode();
    const searchResult = [];
    let similarity;

    for (let idCode in moleculesDB.db) {
      let entry = moleculesDB.db[idCode];

      if (entry.idCode === queryIdCode) {
        similarity = Number.MAX_SAFE_INTEGER;
      } else {
        similarity = moleculesDB.OCL.SSSearcherWithIndex.getSimilarityTanimoto(queryIndex, entry.index) * 1000000 - Math.abs(queryMW - entry.properties.mw) / 10000;
      }

      searchResult.push({
        similarity,
        entry
      });
    }

    searchResult.sort((a, b) => {
      return b.similarity - a.similarity;
    });
    return searchResult.map(entry => entry.entry);
  }

  function getMW(query) {
    let copy = query.getCompactCopy();
    copy.setFragment(false);
    return copy.getMolecularFormula().relativeWeight;
  }

  function processResult(entries) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    const {
      flattenResult = true,
      keepMolecule = false,
      limit = Number.MAX_SAFE_INTEGER
    } = options;
    let results = [];

    if (flattenResult) {
      for (let entry of entries) {
        for (let data of entry.data) {
          results.push({
            data,
            idCode: entry.idCode,
            properties: entry.properties,
            molecule: keepMolecule ? entry.molecule : undefined
          });
        }
      }
    } else {
      for (let entry of entries) {
        results.push({
          data: entry.data,
          idCode: entry.idCode,
          properties: entry.properties,
          molecule: keepMolecule ? entry.molecule : undefined
        });
      }
    }

    if (limit < results.length) results.length = limit;
    return results;
  }

  /*
      this.db is an object with properties 'oclID' that has as value
      an object that contains the following properties:
      * molecule: an OCL molecule instance
      * index: OCL index used for substructure searching
      * properties: all the calculates properties
      * data: array containing free data associated with this molecule
    */

  class MoleculesDB {
    /**
     *
     * @param {OCL} [OCL] The openchemlib library
     * @param {object} [options={}]
     * @param {boolean} [options.computeProperties=false]
     */
    constructor(OCL) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        computeProperties = false
      } = options;
      this.OCL = OCL;
      this.db = {};
      this.statistics = null;
      this.computeProperties = computeProperties;
      this.searcher = new OCL.SSSearcherWithIndex();
    }
    /**
     * append to the current database a CSV file
     * @param {text|ArrayBuffer} csv - text file containing the comma separated value file
     * @param {object} [options={}]
     * @param {boolean} [options.header=true]
     * @param {boolean} [options.dynamicTyping=true]
     * @param {boolean} [options.skipEmptyLines=true]
     * @param {function} [options.onStep] call back to execute after each molecule
     */


    appendCSV(csv, options) {
      return appendCSV(this, csv, {
        computeProperties: this.computeProperties,
        ...options
      });
    }
    /**
     * Append a SDF to the current database
     * @param {text|ArrayBuffer} sdf - text file containing the sdf
     * @param {object} [options={}]
     * @param {function} [options.onStep] call back to execute after each molecule
     * @returns {DB}
     */


    appendSDF(sdf, options) {
      return appendSDF(this, sdf, {
        computeProperties: this.computeProperties,
        ...options
      });
    }
    /**
     * Append a SDF to the current database
     * @param {text|ArrayBuffer} smiles - text file containing a list of smiles
     * @param {object} [options={}]
     * @param {function} [options.onStep] call back to execute after each molecule
     * @returns {DB}
     */


    appendSmilesList(text, options) {
      return appendSmilesList(this, text, {
        computeProperties: this.computeProperties,
        ...options
      });
    }
    /**
     * Add a molecule to the current database
     * @param {OCL.Molecule} molecule
     * @param {object} [data={}]
     * @param {object} [moleculeInfo={}] may contain precalculated index and mw
     */


    pushEntry(molecule, data, moleculeInfo) {
      pushEntry(this, molecule, data, moleculeInfo);
    }
    /**
     * Add an netry in the database
     * @param {object} moleculeInfo - a molecule as a JSON that may contain the following properties: molfile, smiles, idCode, mf, index
     * @param {object} [data={}]
     */


    pushMoleculeInfo(moleculeInfo, data) {
      return pushMoleculeInfo(this, moleculeInfo, data);
    }
    /**
     * Search in a MoleculesDB
     * Inside the database all the same molecules are group together
     * @param {string|OCL.Molecule} [query] smiles, molfile, oclCode or instance of Molecule to look for
     * @param {object} [options={}]
     * @param {string} [options.format='idCode'] - query is in the format 'smiles', 'oclid' or 'molfile'
     * @param {string} [options.mode='substructure'] - search by 'substructure', 'exact' or 'similarity'
     * @param {boolean} [options.flattenResult=true] - The database group the data for the same product. This allows to flatten the result
     * @param {boolean} [options.keepMolecule=false] - keep the OCL.Molecule object in the result
     * @param {number} [options.limit=Number.MAX_SAFE_INTEGER] - maximal number of result
     * @return {Array} array of object of the type {(molecule), idCode, data, properties}
     */


    search(query, options) {
      return search(this, query, options);
    }
    /**
     * Returns an array with the current database
     * @returns
     */


    getDB() {
      return Object.keys(this.db).map(key => this.db[key]);
    }
    /**
     * Append the property `data.color` to each entry based on a data or property label
     * {object} [options={}]
     * {string} [options.dataLabel] name of the property from `data` to use
     * {string} [options.propertyLabel] name of the property from `properties` to use
     * {number} [options.colorLabel='color'] name of the property to add in data that will contain the color
     * {number} [options.minValue]
     * {number} [options.maxValue]
     * {number} [options.minHue=0]
     * {number} [options.maxHue=360]
     * {number} [options.saturation=65] percent of color saturation
     * {number} [options.lightness=65] percent of color lightness
     */


    appendColor(options) {
      appendColor(this, options);
    }

  }

  function toVisualizerMolfile(molecule) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    const {
      diastereotopic,
      heavyAtomHydrogen
    } = options;
    let highlight = [];
    let atoms = {};

    if (diastereotopic) {
      let hydrogenInfo = {};
      let extendedIDs = getDiastereotopicAtomIDsAndH(molecule);

      for (let line of extendedIDs) {
        hydrogenInfo[line.oclID] = line;
      }

      let diaIDs = getGroupedDiastereotopicAtomIDs(molecule);

      for (const diaID of diaIDs) {
        atoms[diaID.oclID] = diaID.atoms;
        highlight.push(diaID.oclID);

        if (heavyAtomHydrogen) {
          if (hydrogenInfo[diaID.oclID] && hydrogenInfo[diaID.oclID].nbHydrogens > 0) {
            for (let id of hydrogenInfo[diaID.oclID].hydrogenOCLIDs) {
              highlight.push(id);
              atoms[id] = diaID.atoms;
            }
          }
        }
      }
    } else {
      let size = molecule.getAllAtoms();
      highlight = new Array(size).fill(0).map((a, index) => index);
      atoms = highlight.map(a => [a]);
    }

    let molfile = {
      type: 'mol2d',
      value: molecule.toMolfile(),
      _highlight: highlight,
      _atoms: atoms
    };
    return molfile;
  }

  function fragmentAcyclicSingleBonds(molecule) {
    const OCL = molecule.getOCL();
    let atoms = [];

    for (let i = 0; i < molecule.getAllAtoms(); i++) {
      let atom = {};
      atoms.push(atom);
      atom.i = i;
      atom.links = []; // we will store connected atoms of broken bonds
    }

    let bonds = [];

    for (let i = 0; i < molecule.getAllBonds(); i++) {
      let bond = {};
      bonds.push(bond);
      bond.i = i;
      bond.order = molecule.getBondOrder(i);
      bond.atom1 = molecule.getBondAtom(0, i);
      bond.atom2 = molecule.getBondAtom(1, i);
      bond.type = molecule.getBondType(i);
      bond.isAromatic = molecule.isAromaticBond(i);
      bond.isRingBond = molecule.isRingBond(i);

      if (!bond.isAromatic && (bond.type & 0b11) === 1 && !bond.isRingBond) {
        bond.selected = true;
        atoms[bond.atom1].links.push(bond.atom2);
        atoms[bond.atom2].links.push(bond.atom1);
      }
    } //  console.log(bonds);


    let brokenMolecule = molecule.getCompactCopy();

    for (let bond of bonds) {
      if (bond.selected) {
        brokenMolecule.markBondForDeletion(bond.i);
      }
    }

    brokenMolecule.deleteMarkedAtomsAndBonds();
    let fragmentMap = [];
    let nbFragments = brokenMolecule.getFragmentNumbers(fragmentMap);
    let results = [];

    for (let i = 0; i < nbFragments; i++) {
      let result = {};
      result.atomMap = [];
      let includeAtom = fragmentMap.map(id => {
        return id === i;
      });
      let fragment = new OCL.Molecule(0, 0);
      let atomMap = [];
      brokenMolecule.copyMoleculeByAtoms(fragment, includeAtom, false, atomMap); // we will add some R groups at the level of the broken bonds

      for (let j = 0; j < atomMap.length; j++) {
        if (atomMap[j] > -1) {
          result.atomMap.push(j);

          if (atoms[j].links.length > 0) {
            for (let k = 0; k < atoms[j].links.length; k++) {
              fragment.addBond(atomMap[j], fragment.addAtom(154), 1);
            }
          }
        }
      }

      fragment.setFragment(false);
      result.idCode = fragment.getIDCode();
      result.mf = getMF(fragment).mf.replace(/R[1-9]?/, '');
      results.push(result);
    }

    return results;
  }

  exports.FULL_HOSE_CODE = FULL_HOSE_CODE;
  exports.HOSE_CODE_CUT_C_SP3_SP3 = HOSE_CODE_CUT_C_SP3_SP3;
  exports.MoleculesDB = MoleculesDB;
  exports.addDiastereotopicMissingChirality = addDiastereotopicMissingChirality;
  exports.combineSmiles = combineSmiles;
  exports.fragmentAcyclicSingleBonds = fragmentAcyclicSingleBonds;
  exports.getAtomsInfo = getAtomsInfo;
  exports.getConnectivityMatrix = getConnectivityMatrix;
  exports.getDiastereotopicAtomIDs = getDiastereotopicAtomIDs;
  exports.getDiastereotopicAtomIDsAndH = getDiastereotopicAtomIDsAndH;
  exports.getGroupedDiastereotopicAtomIDs = getGroupedDiastereotopicAtomIDs;
  exports.getHoseCodesAndDiastereotopicIDs = getHoseCodesAndDiastereotopicIDs;
  exports.getHoseCodesForAtom = getHoseCodesForAtom;
  exports.getHoseCodesForPath = getHoseCodesForPath;
  exports.getHoseCodesFromDiastereotopicID = getHoseCodesFromDiastereotopicID;
  exports.getMF = getMF;
  exports.getPathsInfo = getPathsInfo;
  exports.getShortestPaths = getShortestPaths;
  exports.isCsp3 = isCsp3;
  exports.makeRacemic = makeRacemic;
  exports.tagAtom = tagAtom;
  exports.toDiastereotopicSVG = toDiastereotopicSVG;
  exports.toVisualizerMolfile = toVisualizerMolfile;

  Object.defineProperty(exports, '__esModule', { value: true });

}));
//# sourceMappingURL=openchemlib-utils.js.map
