/**
 * epfl-safety - Calculate hazard for a reaction
 * @version v1.1.0
 * @link https://github.com/cheminfo/epfl-safety#readme
 * @license MIT
 */
(function (global, factory) {
	typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
	typeof define === 'function' && define.amd ? define(['exports'], factory) :
	(global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.EPFLSafety = {}));
})(this, (function (exports) { 'use strict';

	var IDX = 256,
	  HEX = [],
	  BUFFER;
	while (IDX--) HEX[IDX] = (IDX + 256).toString(16).substring(1);
	function v4() {
	  var i = 0,
	    num,
	    out = '';
	  if (!BUFFER || IDX + 16 > 256) {
	    BUFFER = Array(i = 256);
	    while (i--) BUFFER[i] = 256 * Math.random() | 0;
	    i = IDX = 0;
	  }
	  for (; i < 16; i++) {
	    num = BUFFER[IDX + i];
	    if (i == 6) out += HEX[num & 15 | 64];else if (i == 8) out += HEX[num & 63 | 128];else out += HEX[num];
	    if (i & 1 && i > 1 && i < 11) out += '-';
	  }
	  IDX++;
	  return out;
	}

	const levels = {
	  values: {
	    fatal: 60,
	    error: 50,
	    warn: 40,
	    info: 30,
	    debug: 20,
	    trace: 10,
	    silent: 0
	  },
	  labels: {
	    0: 'silent',
	    10: 'trace',
	    20: 'debug',
	    30: 'info',
	    40: 'warn',
	    50: 'error',
	    60: 'fatal'
	  }
	};

	/**
	 * A FIFO logger that stores the last events in an array.
	 */
	class FifoLogger {
	  constructor(options = {}) {
	    this.lastID = {
	      id: 0
	    };
	    this.initialOptions = options;
	    this.uuids = [v4()];
	    this.events = [];
	    this.level = options.level || 'info';
	    this.levelAsNumber = levels.values[this.level];
	    this.limit = options.limit ?? 1000;
	    this.bindings = options.bindings ?? {};
	    this.onChange = options.onChange;
	  }
	  setLevel(level) {
	    this.level = level;
	    this.levelAsNumber = levels.values[level];
	  }
	  setLimit(limit) {
	    this.limit = limit;
	    this.checkSize();
	  }
	  checkSize() {
	    if (this.events.length > this.limit) {
	      this.events.splice(0, this.events.length - this.limit);
	    }
	  }
	  /**
	   * Remove events from the current logger and its children.
	   * @param options
	   */
	  clear() {
	    for (let i = this.events.length - 1; i >= 0; i--) {
	      if (this.events[i].uuids.includes(this.uuids[0])) {
	        this.events.splice(i, 1);
	      }
	    }
	    this.onChange?.(undefined, this.events, {
	      depth: this.uuids.length
	    });
	  }
	  getLogs(options = {}) {
	    const {
	      level,
	      minLevel,
	      includeChildren
	    } = options;
	    let logs = this.events.slice();
	    if (includeChildren) {
	      logs = logs.filter(log => log.uuids.includes(this.uuids[0]));
	    } else {
	      logs = logs.filter(log => log.uuids[0] === this.uuids[0]);
	    }
	    if (level) {
	      const levelNumber = Number(levels.values[level]);
	      if (Number.isNaN(levelNumber)) {
	        throw new Error('Invalid level');
	      }
	      logs = logs.filter(log => log.level === levelNumber);
	    }
	    if (minLevel) {
	      const levelNumber = Number(levels.values[minLevel]);
	      if (Number.isNaN(levelNumber)) {
	        throw new Error('Invalid level');
	      }
	      logs = logs.filter(log => log.level >= levelNumber);
	    }
	    return logs;
	  }
	  /**
	   * @param bindings: an object of key-value pairs to include in log lines as properties.
	   * @param options: an options object that will override child logger inherited options.
	   */
	  child(bindings) {
	    const newFifoLogger = new FifoLogger(this.initialOptions);
	    newFifoLogger.events = this.events;
	    newFifoLogger.uuids = [v4(), ...this.uuids];
	    newFifoLogger.lastID = this.lastID;
	    newFifoLogger.bindings = {
	      ...this.bindings,
	      ...bindings
	    };
	    return newFifoLogger;
	  }
	  trace(value, message) {
	    addEvent(this, levels.values.trace, value, message);
	  }
	  debug(value, message) {
	    addEvent(this, levels.values.debug, value, message);
	  }
	  info(value, message) {
	    addEvent(this, levels.values.info, value, message);
	  }
	  warn(value, message) {
	    addEvent(this, levels.values.warn, value, message);
	  }
	  error(value, message) {
	    addEvent(this, levels.values.error, value, message);
	  }
	  fatal(value, message) {
	    addEvent(this, levels.values.fatal, value, message);
	  }
	}
	function addEvent(logger, level, value, message) {
	  if (level < logger.levelAsNumber) return;
	  const event = {
	    id: ++logger.lastID.id,
	    level,
	    levelLabel: levels.labels[level],
	    time: Date.now(),
	    uuids: logger.uuids
	  };
	  if (value instanceof Error) {
	    event.message = value.toString();
	    event.error = value;
	    event.meta = {
	      ...logger.bindings
	    };
	  } else if (message !== undefined && typeof value === 'object') {
	    event.message = message;
	    event.meta = {
	      ...logger.bindings,
	      ...value
	    };
	  } else if (message === undefined && typeof value === 'string') {
	    event.message = value;
	    event.meta = {
	      ...logger.bindings
	    };
	  } else {
	    throw new Error('Invalid arguments');
	  }
	  logger.events.push(event);
	  logger.checkSize();
	  if (logger.onChange) {
	    logger.onChange(event, logger.events, {
	      depth: logger.uuids.length
	    });
	  }
	}

	const redAcuteToxicitiesHazards = ['H300', 'H310', 'H330'];
	const rules$3 = [{
	  description: 'Extreme acute toxicity',
	  info: {
	    highOrExtreme: true
	  },
	  matcher: {
	    hazards: ['H300', 'H310', 'H330']
	  },
	  masses: [{
	    lessThan: 0.001,
	    score: 3
	  }, {
	    lessThan: 0.003,
	    score: 6
	  }, {
	    lessThan: +Infinity,
	    score: 12
	  }]
	}, {
	  description: 'High acute toxicity',
	  info: {
	    highOrExtreme: true
	  },
	  matcher: {
	    hazards: ['H311', 'H318', 'H331']
	  },
	  masses: [{
	    lessThan: 0.001,
	    score: 2
	  }, {
	    lessThan: 0.003,
	    score: 3
	  }, {
	    lessThan: +Infinity,
	    score: 4
	  }]
	}, {
	  description: 'Medium acute toxicity',
	  info: {
	    highOrExtreme: false
	  },
	  matcher: {
	    hazards: ['H302', 'H303', 'H305', 'H312', 'H313', 'H330', 'H332', 'H333', 'H334']
	  },
	  masses: [{
	    lessThan: 0.001,
	    score: 1
	  }, {
	    lessThan: 0.003,
	    score: 2
	  }, {
	    lessThan: +Infinity,
	    score: 3
	  }]
	}, {
	  description: 'Low acute toxicity',
	  info: {
	    highOrExtreme: false
	  },
	  matcher: {
	    hazards: ['H301', 'H304', 'H315', 'H316', 'H317', 'H319,H320', 'H334', 'H335', 'H336']
	  },
	  masses: [{
	    lessThan: 0.001,
	    score: 0.5
	  }, {
	    lessThan: 0.003,
	    score: 1
	  }, {
	    lessThan: +Infinity,
	    score: 2
	  }]
	}, {
	  description: 'No known acute toxicity',
	  info: {
	    highOrExtreme: false
	  },
	  matcher: {},
	  masses: [{
	    lessThan: +Infinity,
	    score: 0
	  }]
	}];
	function appendAcuteToxicity(reagents, report, options) {
	  const {
	    logger
	  } = options;
	  let redAcuteToxicities = new Set();
	  // what is the worse hazard ?
	  for (const rule of rules$3) {
	    for (const reagent of reagents) {
	      if (reagent.safety.acuteToxicity) continue;
	      appendRed$1(reagent, redAcuteToxicities);
	      const matching = isMatching$3(reagent, rule);
	      if (!matching) continue;
	      reagent.safety.acuteToxicity = {
	        description: rule.description,
	        matchingHazards: matching.matchingHazards || [],
	        info: rule.info,
	        score: rule.masses.find(mass => reagent.mass.value <= mass.lessThan).score
	      };
	    }
	  }

	  /*
	  If high or extreme are present only take sum of high and extreme values.
	  If no high or extreme, simply take the score for the one reagent that leads to the highest score.
	  */
	  const highOrExtremes = reagents.filter(reagent => reagent.safety.acuteToxicity.info.highOrExtreme);
	  if (highOrExtremes.length === 0) {
	    logger?.info('AcuteToxicity: No high or extreme acute toxicity. Taking the highest score of the other.');
	    report.scores.acuteToxicity = Math.max(...reagents.map(reagent => reagent.safety.acuteToxicity.score));
	  } else {
	    logger?.info('AcuteToxicity: High or extreme acute toxicity. Taking the sum of them.');
	    report.scores.acuteToxicity = highOrExtremes.reduce((sum, reagent) => sum + reagent.safety.acuteToxicity.score, 0);
	  }
	  report.redAcuteToxicity = [...redAcuteToxicities];
	}
	function isMatching$3(reagent, rule) {
	  if (!rule.matcher.hazards) return true;
	  const matchingHazards = rule.matcher.hazards.filter(hazard => reagent.hazards.includes(hazard));
	  if (matchingHazards.length === 0) return false;
	  return {
	    matchingHazards
	  };
	}
	function appendRed$1(reagent, redAcuteToxicities) {
	  reagent.hazards.forEach(hazard => {
	    if (redAcuteToxicitiesHazards.includes(hazard)) {
	      redAcuteToxicities.add(hazard);
	    }
	  });
	}

	const rules$2 = [{
	  description: 'Extreme long-term toxicity',
	  matcher: {
	    hazards: ['H340', 'H350', 'H360', 'H370']
	  },
	  volumes: [{
	    lessThan: 0.05,
	    score: 2
	  }, {
	    lessThan: 0.5,
	    score: 3
	  }, {
	    lessThan: 1,
	    score: 6
	  }, {
	    lessThan: +Infinity,
	    score: 10
	  }]
	}, {
	  description: 'High long-term toxicity',
	  matcher: {
	    hazards: ['H341', 'H351', 'H361', 'H371', 'H372', 'H373']
	  },
	  volumes: [{
	    lessThan: 0.05,
	    score: 1
	  }, {
	    lessThan: 0.5,
	    score: 2
	  }, {
	    lessThan: 1,
	    score: 4
	  }, {
	    lessThan: +Infinity,
	    score: 8
	  }]
	}, {
	  description: 'No known long-term toxicity',
	  matcher: {},
	  volumes: [{
	    lessThan: +Infinity,
	    score: 0
	  }]
	}];
	function appendLongTermToxicity(reagents, report, options = {}) {
	  const {
	    logger
	  } = options;
	  // what is the worse hazard ?
	  for (const rule of rules$2) {
	    for (const reagent of reagents) {
	      if (reagent.safety.longTermToxicity) continue;
	      const matching = isMatching$2(reagent, rule);
	      if (!matching) continue;
	      reagent.safety.longTermToxicity = {
	        description: rule.description,
	        matchingHazards: matching.matchingHazards || [],
	        score: rule.volumes.find(volume => reagent.volume.value <= volume.lessThan).score
	      };
	    }
	  }
	  logger?.info('LongTermToxicity: Taking the highest score.');
	  report.scores.longTermToxicity = Math.max(...reagents.map(reagent => reagent.safety.longTermToxicity.score));
	}
	function isMatching$2(reagent, rule) {
	  if (!rule.matcher.hazards) return true;
	  const matchingHazards = rule.matcher.hazards.filter(hazard => reagent.hazards.includes(hazard));
	  if (matchingHazards.length === 0) return false;
	  return {
	    matchingHazards
	  };
	}

	function appendMiscellaneous(reagents, report, options = {}) {
	  const {
	    miscellaneous = {}
	  } = options;
	  const {
	    highVoltage,
	    dustRisk,
	    reactionTime,
	    temperature,
	    gasLines,
	    pressure,
	    overheadStirring
	  } = miscellaneous;
	  const results = [];
	  results.push(getDustRisk(dustRisk, options));
	  results.push(getReactionTime(reactionTime, options));
	  results.push(getTemperature(temperature, options));
	  results.push(getGasLines(gasLines, options));
	  results.push(getPressure(pressure, options));
	  results.push(getOverheadStirring(overheadStirring, options));
	  results.push(getHighVoltage(highVoltage, options));
	  report.miscellaneous = results;
	}
	function getDustRisk(dustRisk, options) {
	  const {
	    logger
	  } = options;
	  const entry = {
	    label: 'Dust risk',
	    info: '(for more than 5g of solids)',
	    scores: []
	  };
	  switch (dustRisk) {
	    case 'yes':
	      entry.scores.push({
	        remarks: 'Mask still required for all quantities of fine particles!',
	        label: 'Fine powders, fibers air suspended',
	        score: 4
	      });
	      break;
	    case 'none':
	      entry.scores.push({
	        label: 'None',
	        score: 0
	      });
	      break;
	    default:
	      logger.error(`Unknown dustRisk: ${dustRisk}`);
	  }
	  entry.score = entry.scores.reduce((acc, cur) => acc + cur.score, 0);
	  return entry;
	}
	function getReactionTime(reactionTime, options) {
	  const {
	    logger
	  } = options;
	  const entry = {
	    label: 'Reaction time',
	    info: '',
	    scores: []
	  };
	  switch (reactionTime) {
	    case 'daytime':
	      entry.scores.push({
	        label: 'Daytime',
	        score: 0
	      });
	      break;
	    case 'overnight':
	      entry.scores.push({
	        label: 'Overnight',
	        score: 2
	      });
	      break;
	    default:
	      logger.error(`Unknown reactionTime: ${reactionTime}`);
	  }
	  entry.score = entry.scores.reduce((acc, cur) => acc + cur.score, 0);
	  return entry;
	}
	function getTemperature(temperature, options) {
	  const {
	    logger
	  } = options;
	  const entry = {
	    label: 'Temperature',
	    info: '',
	    scores: []
	  };
	  switch (temperature) {
	    case '<100':
	      entry.scores.push({
	        label: '< 100°C',
	        score: 0
	      });
	      break;
	    case '100-200':
	      entry.scores.push({
	        label: '100-200°C',
	        score: 1
	      });
	      break;
	    case 'extreme':
	      entry.scores.push({
	        label: '> 200°C or < -30°C',
	        score: 2
	      });
	      break;
	    default:
	      logger.error(`Unknown temperature: ${temperature}`);
	  }
	  entry.score = entry.scores.reduce((acc, cur) => acc + cur.score, 0);
	  return entry;
	}
	function getGasLines(gasLines, options) {
	  const {
	    logger
	  } = options;
	  const entry = {
	    label: 'Handling High Pressure Gas Lines',
	    info: '(Gas line pressure)',
	    scores: []
	  };
	  const inertDescription = 'Inert (e.g., N2, He, Ar, CO2)';
	  switch (gasLines.inert) {
	    case 'none':
	      entry.scores.push({
	        description: inertDescription,
	        label: 'None',
	        score: 0
	      });
	      break;
	    case '<20':
	      entry.scores.push({
	        description: inertDescription,
	        label: '<20 barg',
	        score: 0
	      });
	      break;
	    case '20-40':
	      entry.scores.push({
	        description: inertDescription,
	        label: '20-40 barg',
	        score: 1
	      });
	      break;
	    case '>40':
	      entry.scores.push({
	        description: inertDescription,
	        label: '>40 barg',
	        score: 2
	      });
	      break;
	    default:
	      logger.error(`Unknown gasLines.inert: ${gasLines.inert}`);
	  }
	  const reactiveDescription = 'Reactive/Hazardous (e.g., CO, NH3, H2, hydrocarbons)';
	  switch (gasLines.reactive) {
	    case 'none':
	      entry.scores.push({
	        description: reactiveDescription,
	        label: 'None',
	        score: 0
	      });
	      break;
	    case '<20':
	      entry.scores.push({
	        description: reactiveDescription,
	        label: '<20 barg',
	        score: 2
	      });
	      break;
	    case '20-40':
	      entry.scores.push({
	        description: reactiveDescription,
	        label: '20-40 barg',
	        score: 3
	      });
	      break;
	    case '>40':
	      entry.scores.push({
	        description: reactiveDescription,
	        label: '>40 barg',
	        score: 4
	      });
	      break;
	    default:
	      logger.error(`Unknown gasLines.inert: ${gasLines.reactive}`);
	  }
	  entry.score = entry.scores.reduce((acc, cur) => acc + cur.score, 0);
	  return entry;
	}
	function getPressure(pressure, options) {
	  const {
	    logger
	  } = options;
	  const entry = {
	    label: 'For pressurized reactions in Parr/flow',
	    info: 'Did you pass the PARR / Flow reactor exam and sign the pressurized SOP and PHA?',
	    scores: []
	  };
	  const parrLowDescription = 'Parr < 50 mL or flow reactor(1 / 4';
	  switch (pressure.parrLow) {
	    case 'none':
	      entry.scores.push({
	        description: parrLowDescription,
	        label: 'None',
	        score: 0
	      });
	      break;
	    case '1-20':
	      entry.scores.push({
	        description: parrLowDescription,
	        label: '1-20 barg',
	        score: 1
	      });
	      break;
	    case '20-100':
	      entry.scores.push({
	        description: parrLowDescription,
	        label: '20-100 barg',
	        score: 2
	      });
	      break;
	    case '>100':
	      entry.scores.push({
	        description: parrLowDescription,
	        label: '>100 barg',
	        score: 3
	      });
	      break;
	    default:
	      logger.error(`Unknown pressure.parrLow: ${pressure.parrLow}`);
	  }
	  const parrHighDescription = 'Parr > 50 mL or flow reactor(1 / 2';
	  switch (pressure.parrHigh) {
	    case 'none':
	      entry.scores.push({
	        description: parrHighDescription,
	        label: 'None',
	        score: 0
	      });
	      break;
	    case '1-2':
	      entry.scores.push({
	        description: parrHighDescription,
	        label: '1-2 barg',
	        score: 1
	      });
	      break;
	    case '2-5':
	      entry.scores.push({
	        description: parrHighDescription,
	        label: '2-5 barg',
	        score: 2
	      });
	      break;
	    case '>10':
	      entry.scores.push({
	        description: parrHighDescription,
	        label: '>10 barg',
	        score: 3
	      });
	      break;
	    default:
	      logger.error(`Unknown pressure.parrHigh: ${pressure.parrHigh}`);
	  }
	  const pRatedDescription = 'Glass (P-Rated)';
	  switch (pressure.pRated) {
	    case 'none':
	      entry.scores.push({
	        description: pRatedDescription,
	        label: 'None',
	        score: 0
	      });
	      break;
	    case '1-2':
	      entry.scores.push({
	        description: pRatedDescription,
	        label: '1-2 barg',
	        score: 1
	      });
	      break;
	    case '2-5':
	      entry.scores.push({
	        description: pRatedDescription,
	        label: '2-5 barg',
	        score: 2
	      });
	      break;
	    case '>5':
	      entry.scores.push({
	        description: pRatedDescription,
	        label: '>5 barg',
	        score: 3
	      });
	      break;
	    default:
	      logger.error(`Unknown pressure.pRated: ${pressure.pRated}`);
	  }
	  const nonRatedDescription = 'Glass (Not rated)';
	  switch (pressure.nonRated) {
	    case 'none':
	      entry.scores.push({
	        description: nonRatedDescription,
	        label: 'None',
	        score: 0
	      });
	      break;
	    case 'yes':
	      entry.scores.push({
	        description: nonRatedDescription,
	        label: 'Yes',
	        score: 100
	      });
	      break;
	    default:
	      logger.error(`Unknown pressure.nonRated: ${pressure.nonRated}`);
	  }
	  entry.score = entry.scores.reduce((acc, cur) => acc + cur.score, 0);
	  return entry;
	}
	function getOverheadStirring(overheadStirring, options) {
	  const {
	    logger
	  } = options;
	  const entry = {
	    label: 'Overheads stirring ?',
	    info: '',
	    scores: []
	  };
	  switch (overheadStirring) {
	    case 'none':
	      entry.scores.push({
	        label: 'None',
	        score: 0
	      });
	      break;
	    case 'yes':
	      entry.scores.push({
	        label: 'Yes',
	        score: 2
	      });
	      break;
	    default:
	      logger.error(`Unknown overheadStirring: ${overheadStirring}`);
	  }
	  entry.score = entry.scores.reduce((acc, cur) => acc + cur.score, 0);
	  return entry;
	}
	function getHighVoltage(highVoltage, options) {
	  const {
	    logger
	  } = options;
	  const entry = {
	    label: 'High voltage ?',
	    info: '',
	    scores: []
	  };
	  switch (highVoltage) {
	    case 'none':
	      entry.scores.push({
	        label: 'None',
	        score: 0
	      });
	      break;
	    case 'yes':
	      entry.scores.push({
	        label: 'Yes',
	        score: 100
	      });
	      break;
	    default:
	      logger.error(`Unknown highVoltage: ${highVoltage}`);
	  }
	  entry.score = entry.scores.reduce((acc, cur) => acc + cur.score, 0);
	  return entry;
	}

	const redReactivityAndInstabilityHazards = ['H260', 'H261', 'H250', 'H271', 'H272', 'H200', 'H201', 'H202', 'H203', 'H240', 'H241', 'H242', 'H251', 'H252'];
	const rules$1 = [{
	  description: 'Flammable gas generation',
	  matcher: {
	    hazards: ['H260', 'H261']
	  },
	  masses: [{
	    lessThan: 0.001,
	    score: 3
	  }, {
	    lessThan: 0.003,
	    score: 5
	  }, {
	    lessThan: +Infinity,
	    score: 10
	  }]
	}, {
	  description: 'Pyrophoric',
	  matcher: {
	    hazards: ['H250']
	  },
	  masses: [{
	    lessThan: 0.001,
	    score: 3
	  }, {
	    lessThan: 0.003,
	    score: 6
	  }, {
	    lessThan: +Infinity,
	    score: 10
	  }]
	}, {
	  description: 'Oxidizer',
	  matcher: {
	    hazards: ['H270', 'H271', 'H272']
	  },
	  masses: [{
	    lessThan: 0.001,
	    score: 1
	  }, {
	    lessThan: 0.003,
	    score: 3
	  }, {
	    lessThan: +Infinity,
	    score: 6
	  }]
	}, {
	  description: 'Explosive / Self-heating',
	  matcher: {
	    hazards: ['H200', 'H201', 'H202', 'H203', 'H240', 'H241', 'H242', 'H251', 'H252']
	  },
	  masses: [{
	    lessThan: 0.001,
	    score: 3
	  }, {
	    lessThan: 0.003,
	    score: 5
	  }, {
	    lessThan: +Infinity,
	    score: 10
	  }]
	}, {
	  description: 'Corrosive',
	  matcher: {
	    hazards: ['H314']
	  },
	  masses: [{
	    lessThan: 0.001,
	    score: 0.5
	  }, {
	    lessThan: 0.003,
	    score: 1
	  }, {
	    lessThan: +Infinity,
	    score: 2
	  }]
	}, {
	  description: 'No reactivity or instability hazards known',
	  matcher: {},
	  masses: [{
	    lessThan: +Infinity,
	    score: 0
	  }]
	}];
	function appendReactivityAndInstability(reagents, report) {
	  // what is the worse hazard ?
	  let sum = 0;
	  let redReactivityAndInstabilities = new Set();
	  for (const rule of rules$1) {
	    for (const reagent of reagents) {
	      if (!reagent.safety.reactivityAndInstabilities) {
	        reagent.safety.reactivityAndInstabilities = [];
	      }
	      const matching = isMatching$1(reagent, rule);
	      appendRed(reagent, redReactivityAndInstabilities);
	      if (!matching) continue;
	      const score = rule.masses.find(mass => reagent.mass.value <= mass.lessThan).score;
	      if (score > 0 || reagent.safety.reactivityAndInstabilities.length === 0) {
	        reagent.safety.reactivityAndInstabilities.push({
	          description: rule.description,
	          matchingHazards: matching.matchingHazards || [],
	          score
	        });
	        sum += score;
	      }
	    }
	  }
	  report.redReactivityAndInstabilities = redReactivityAndInstabilities;
	  report.scores.reactivityAndInstability = sum;
	}
	function appendRed(reagent, reactivityAndInstabilities) {
	  reagent.hazards.forEach(hazard => {
	    if (redReactivityAndInstabilityHazards.includes(hazard)) {
	      reactivityAndInstabilities.add(hazard);
	    }
	  });
	}
	function isMatching$1(reagent, rule) {
	  if (!rule.matcher.hazards) return true;
	  const matchingHazards = rule.matcher.hazards.filter(hazard => reagent.hazards.includes(hazard));
	  if (matchingHazards.length === 0) return false;
	  return {
	    matchingHazards
	  };
	}

	const rules = [{
	  description: 'Very flammable, low bp',
	  matcher: {
	    hazards: ['H222', 'H224', 'H225'],
	    bpLessThan: 35
	  },
	  volumes: [{
	    lessThan: 0.01,
	    score: 1.5
	  }, {
	    lessThan: 0.05,
	    score: 3
	  }, {
	    lessThan: 0.5,
	    score: 4
	  }, {
	    lessThan: 1,
	    score: 5
	  }, {
	    lessThan: +Infinity,
	    score: 10
	  }]
	}, {
	  description: 'Very flammable, high bp',
	  matcher: {
	    hazards: ['H222', 'H224', 'H225']
	  },
	  volumes: [{
	    lessThan: 0.01,
	    score: 1
	  }, {
	    lessThan: 0.05,
	    score: 2
	  }, {
	    lessThan: 0.5,
	    score: 3
	  }, {
	    lessThan: 1,
	    score: 4
	  }, {
	    lessThan: +Infinity,
	    score: 8
	  }]
	}, {
	  description: 'Flammable, low bp',
	  matcher: {
	    hazards: ['H223', 'H226', 'H227', 'H228'],
	    bpLessThan: 35
	  },
	  volumes: [{
	    lessThan: 0.01,
	    score: 0.5
	  }, {
	    lessThan: 0.05,
	    score: 1
	  }, {
	    lessThan: 0.5,
	    score: 2
	  }, {
	    lessThan: 1,
	    score: 3
	  }, {
	    lessThan: +Infinity,
	    score: 6
	  }]
	}, {
	  description: 'Flammable, high bp',
	  matcher: {
	    hazards: ['H223', 'H226', 'H227', 'H228']
	  },
	  volumes: [{
	    lessThan: 0.01,
	    score: 0
	  }, {
	    lessThan: 0.05,
	    score: 0.5
	  }, {
	    lessThan: 0.5,
	    score: 1
	  }, {
	    lessThan: 1,
	    score: 2
	  }, {
	    lessThan: +Infinity,
	    score: 4
	  }]
	}, {
	  description: 'Not flammable',
	  matcher: {},
	  volumes: [{
	    lessThan: 0.5,
	    score: 0
	  }, {
	    lessThan: 1,
	    score: 1
	  }, {
	    lessThan: +Infinity,
	    score: 3
	  }]
	}];
	function appendVolumeAndFlammability(reagents, report, options) {
	  const totalVolume = getTotalVolume(reagents, options);
	  // what is the worse hazard ?
	  let bestScore = 0;
	  for (const rule of rules) {
	    for (const reagent of reagents) {
	      if (reagent.safety.volumeAndFlammability) continue;
	      const matching = isMatching(reagent, rule);
	      if (!matching) continue;
	      const score = rule.volumes.find(volume => totalVolume <= volume.lessThan).score;
	      if (score > bestScore) {
	        bestScore = score;
	      }
	      reagent.safety.volumeAndFlammability = {
	        description: rule.description,
	        matchingHazards: matching.matchingHazards || [],
	        score
	      };
	    }
	  }
	  report.scores.volumeAndFlammability = bestScore;
	}
	function isMatching(reagent, rule) {
	  if (!rule.matcher.hazards) return true;
	  if (!rule.matcher.bpLessThan && reagent.bp > rule.matcher.bpLessThan) {
	    return false;
	  }
	  const matchingHazards = rule.matcher.hazards.filter(hazard => reagent.hazards.includes(hazard));
	  if (matchingHazards.length === 0) return false;
	  return {
	    matchingHazards
	  };
	}
	function getTotalVolume(reagents, options) {
	  const {
	    logger
	  } = options;
	  let totalVolume = 0;
	  for (const reagent of reagents) {
	    if (reagent.volume) {
	      totalVolume += reagent.volume.value;
	    } else {
	      logger.warn(`reagent ${reagent.smiles} has no volume`);
	      totalVolume += reagent.mass.value;
	    }
	  }
	  return totalVolume;
	}

	function filterReagents(reagents, options) {
	  const reagentsKind = options.miscellaneous?.reagentsKind;
	  if (reagentsKind === 'nonWorkup') {
	    reagents = reagents.filter(reagent => reagent.kind?.replace(/[^a-z]/gi, '') !== 'workup');
	  } else if (reagentsKind === 'onlyWorkup') {
	    reagents = reagents.filter(reagent => reagent.kind?.replace(/[^a-z]/gi, '') === 'workup');
	  }
	  return reagents;
	}

	var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

	function commonjsRequire (path) {
		throw new Error('Could not dynamically require "' + path + '". Please configure the dynamicRequireTargets or/and ignoreDynamicRequires option of @rollup/plugin-commonjs appropriately for this require call to work.');
	}

	var delay$2 = {exports: {}};

	// From https://github.com/sindresorhus/random-int/blob/c37741b56f76b9160b0b63dae4e9c64875128146/index.js#L13-L15
	const randomInteger = (minimum, maximum) => Math.floor(Math.random() * (maximum - minimum + 1) + minimum);
	const createAbortError = () => {
	  const error = new Error('Delay aborted');
	  error.name = 'AbortError';
	  return error;
	};
	const createDelay = ({
	  clearTimeout: defaultClear,
	  setTimeout: set,
	  willResolve
	}) => (ms, {
	  value,
	  signal
	} = {}) => {
	  if (signal && signal.aborted) {
	    return Promise.reject(createAbortError());
	  }
	  let timeoutId;
	  let settle;
	  let rejectFn;
	  const clear = defaultClear || clearTimeout;
	  const signalListener = () => {
	    clear(timeoutId);
	    rejectFn(createAbortError());
	  };
	  const cleanup = () => {
	    if (signal) {
	      signal.removeEventListener('abort', signalListener);
	    }
	  };
	  const delayPromise = new Promise((resolve, reject) => {
	    settle = () => {
	      cleanup();
	      if (willResolve) {
	        resolve(value);
	      } else {
	        reject(value);
	      }
	    };
	    rejectFn = reject;
	    timeoutId = (set || setTimeout)(settle, ms);
	  });
	  if (signal) {
	    signal.addEventListener('abort', signalListener, {
	      once: true
	    });
	  }
	  delayPromise.clear = () => {
	    clear(timeoutId);
	    timeoutId = null;
	    settle();
	  };
	  return delayPromise;
	};
	const createWithTimers = clearAndSet => {
	  const delay = createDelay({
	    ...clearAndSet,
	    willResolve: true
	  });
	  delay.reject = createDelay({
	    ...clearAndSet,
	    willResolve: false
	  });
	  delay.range = (minimum, maximum, options) => delay(randomInteger(minimum, maximum), options);
	  return delay;
	};
	const delay = createWithTimers();
	delay.createWithTimers = createWithTimers;
	delay$2.exports = delay;
	// TODO: Remove this for the next major release
	delay$2.exports.default = delay;
	var delay$1 = delay$2.exports;

	var browserPonyfill = {exports: {}};

	(function (module, exports) {
	  // Save global object in a variable
	  var __global__ = typeof globalThis !== 'undefined' && globalThis || typeof self !== 'undefined' && self || typeof commonjsGlobal !== 'undefined' && commonjsGlobal;
	  // Create an object that extends from __global__ without the fetch function
	  var __globalThis__ = function () {
	    function F() {
	      this.fetch = false;
	      this.DOMException = __global__.DOMException;
	    }
	    F.prototype = __global__; // Needed for feature detection on whatwg-fetch's code
	    return new F();
	  }();
	  // Wraps whatwg-fetch with a function scope to hijack the global object
	  // "globalThis" that's going to be patched
	  (function (globalThis) {
	    (function (exports) {
	      var global = typeof globalThis !== 'undefined' && globalThis || typeof self !== 'undefined' && self || typeof global !== 'undefined' && global;
	      var support = {
	        searchParams: 'URLSearchParams' in global,
	        iterable: 'Symbol' in global && 'iterator' in Symbol,
	        blob: 'FileReader' in global && 'Blob' in global && function () {
	          try {
	            new Blob();
	            return true;
	          } catch (e) {
	            return false;
	          }
	        }(),
	        formData: 'FormData' in global,
	        arrayBuffer: 'ArrayBuffer' in global
	      };
	      function isDataView(obj) {
	        return obj && DataView.prototype.isPrototypeOf(obj);
	      }
	      if (support.arrayBuffer) {
	        var viewClasses = ['[object Int8Array]', '[object Uint8Array]', '[object Uint8ClampedArray]', '[object Int16Array]', '[object Uint16Array]', '[object Int32Array]', '[object Uint32Array]', '[object Float32Array]', '[object Float64Array]'];
	        var isArrayBufferView = ArrayBuffer.isView || function (obj) {
	          return obj && viewClasses.indexOf(Object.prototype.toString.call(obj)) > -1;
	        };
	      }
	      function normalizeName(name) {
	        if (typeof name !== 'string') {
	          name = String(name);
	        }
	        if (/[^a-z0-9\-#$%&'*+.^_`|~!]/i.test(name) || name === '') {
	          throw new TypeError('Invalid character in header field name: "' + name + '"');
	        }
	        return name.toLowerCase();
	      }
	      function normalizeValue(value) {
	        if (typeof value !== 'string') {
	          value = String(value);
	        }
	        return value;
	      }

	      // Build a destructive iterator for the value list
	      function iteratorFor(items) {
	        var iterator = {
	          next: function () {
	            var value = items.shift();
	            return {
	              done: value === undefined,
	              value: value
	            };
	          }
	        };
	        if (support.iterable) {
	          iterator[Symbol.iterator] = function () {
	            return iterator;
	          };
	        }
	        return iterator;
	      }
	      function Headers(headers) {
	        this.map = {};
	        if (headers instanceof Headers) {
	          headers.forEach(function (value, name) {
	            this.append(name, value);
	          }, this);
	        } else if (Array.isArray(headers)) {
	          headers.forEach(function (header) {
	            this.append(header[0], header[1]);
	          }, this);
	        } else if (headers) {
	          Object.getOwnPropertyNames(headers).forEach(function (name) {
	            this.append(name, headers[name]);
	          }, this);
	        }
	      }
	      Headers.prototype.append = function (name, value) {
	        name = normalizeName(name);
	        value = normalizeValue(value);
	        var oldValue = this.map[name];
	        this.map[name] = oldValue ? oldValue + ', ' + value : value;
	      };
	      Headers.prototype['delete'] = function (name) {
	        delete this.map[normalizeName(name)];
	      };
	      Headers.prototype.get = function (name) {
	        name = normalizeName(name);
	        return this.has(name) ? this.map[name] : null;
	      };
	      Headers.prototype.has = function (name) {
	        return this.map.hasOwnProperty(normalizeName(name));
	      };
	      Headers.prototype.set = function (name, value) {
	        this.map[normalizeName(name)] = normalizeValue(value);
	      };
	      Headers.prototype.forEach = function (callback, thisArg) {
	        for (var name in this.map) {
	          if (this.map.hasOwnProperty(name)) {
	            callback.call(thisArg, this.map[name], name, this);
	          }
	        }
	      };
	      Headers.prototype.keys = function () {
	        var items = [];
	        this.forEach(function (value, name) {
	          items.push(name);
	        });
	        return iteratorFor(items);
	      };
	      Headers.prototype.values = function () {
	        var items = [];
	        this.forEach(function (value) {
	          items.push(value);
	        });
	        return iteratorFor(items);
	      };
	      Headers.prototype.entries = function () {
	        var items = [];
	        this.forEach(function (value, name) {
	          items.push([name, value]);
	        });
	        return iteratorFor(items);
	      };
	      if (support.iterable) {
	        Headers.prototype[Symbol.iterator] = Headers.prototype.entries;
	      }
	      function consumed(body) {
	        if (body.bodyUsed) {
	          return Promise.reject(new TypeError('Already read'));
	        }
	        body.bodyUsed = true;
	      }
	      function fileReaderReady(reader) {
	        return new Promise(function (resolve, reject) {
	          reader.onload = function () {
	            resolve(reader.result);
	          };
	          reader.onerror = function () {
	            reject(reader.error);
	          };
	        });
	      }
	      function readBlobAsArrayBuffer(blob) {
	        var reader = new FileReader();
	        var promise = fileReaderReady(reader);
	        reader.readAsArrayBuffer(blob);
	        return promise;
	      }
	      function readBlobAsText(blob) {
	        var reader = new FileReader();
	        var promise = fileReaderReady(reader);
	        reader.readAsText(blob);
	        return promise;
	      }
	      function readArrayBufferAsText(buf) {
	        var view = new Uint8Array(buf);
	        var chars = new Array(view.length);
	        for (var i = 0; i < view.length; i++) {
	          chars[i] = String.fromCharCode(view[i]);
	        }
	        return chars.join('');
	      }
	      function bufferClone(buf) {
	        if (buf.slice) {
	          return buf.slice(0);
	        } else {
	          var view = new Uint8Array(buf.byteLength);
	          view.set(new Uint8Array(buf));
	          return view.buffer;
	        }
	      }
	      function Body() {
	        this.bodyUsed = false;
	        this._initBody = function (body) {
	          /*
	            fetch-mock wraps the Response object in an ES6 Proxy to
	            provide useful test harness features such as flush. However, on
	            ES5 browsers without fetch or Proxy support pollyfills must be used;
	            the proxy-pollyfill is unable to proxy an attribute unless it exists
	            on the object before the Proxy is created. This change ensures
	            Response.bodyUsed exists on the instance, while maintaining the
	            semantic of setting Request.bodyUsed in the constructor before
	            _initBody is called.
	          */
	          this.bodyUsed = this.bodyUsed;
	          this._bodyInit = body;
	          if (!body) {
	            this._bodyText = '';
	          } else if (typeof body === 'string') {
	            this._bodyText = body;
	          } else if (support.blob && Blob.prototype.isPrototypeOf(body)) {
	            this._bodyBlob = body;
	          } else if (support.formData && FormData.prototype.isPrototypeOf(body)) {
	            this._bodyFormData = body;
	          } else if (support.searchParams && URLSearchParams.prototype.isPrototypeOf(body)) {
	            this._bodyText = body.toString();
	          } else if (support.arrayBuffer && support.blob && isDataView(body)) {
	            this._bodyArrayBuffer = bufferClone(body.buffer);
	            // IE 10-11 can't handle a DataView body.
	            this._bodyInit = new Blob([this._bodyArrayBuffer]);
	          } else if (support.arrayBuffer && (ArrayBuffer.prototype.isPrototypeOf(body) || isArrayBufferView(body))) {
	            this._bodyArrayBuffer = bufferClone(body);
	          } else {
	            this._bodyText = body = Object.prototype.toString.call(body);
	          }
	          if (!this.headers.get('content-type')) {
	            if (typeof body === 'string') {
	              this.headers.set('content-type', 'text/plain;charset=UTF-8');
	            } else if (this._bodyBlob && this._bodyBlob.type) {
	              this.headers.set('content-type', this._bodyBlob.type);
	            } else if (support.searchParams && URLSearchParams.prototype.isPrototypeOf(body)) {
	              this.headers.set('content-type', 'application/x-www-form-urlencoded;charset=UTF-8');
	            }
	          }
	        };
	        if (support.blob) {
	          this.blob = function () {
	            var rejected = consumed(this);
	            if (rejected) {
	              return rejected;
	            }
	            if (this._bodyBlob) {
	              return Promise.resolve(this._bodyBlob);
	            } else if (this._bodyArrayBuffer) {
	              return Promise.resolve(new Blob([this._bodyArrayBuffer]));
	            } else if (this._bodyFormData) {
	              throw new Error('could not read FormData body as blob');
	            } else {
	              return Promise.resolve(new Blob([this._bodyText]));
	            }
	          };
	          this.arrayBuffer = function () {
	            if (this._bodyArrayBuffer) {
	              var isConsumed = consumed(this);
	              if (isConsumed) {
	                return isConsumed;
	              }
	              if (ArrayBuffer.isView(this._bodyArrayBuffer)) {
	                return Promise.resolve(this._bodyArrayBuffer.buffer.slice(this._bodyArrayBuffer.byteOffset, this._bodyArrayBuffer.byteOffset + this._bodyArrayBuffer.byteLength));
	              } else {
	                return Promise.resolve(this._bodyArrayBuffer);
	              }
	            } else {
	              return this.blob().then(readBlobAsArrayBuffer);
	            }
	          };
	        }
	        this.text = function () {
	          var rejected = consumed(this);
	          if (rejected) {
	            return rejected;
	          }
	          if (this._bodyBlob) {
	            return readBlobAsText(this._bodyBlob);
	          } else if (this._bodyArrayBuffer) {
	            return Promise.resolve(readArrayBufferAsText(this._bodyArrayBuffer));
	          } else if (this._bodyFormData) {
	            throw new Error('could not read FormData body as text');
	          } else {
	            return Promise.resolve(this._bodyText);
	          }
	        };
	        if (support.formData) {
	          this.formData = function () {
	            return this.text().then(decode);
	          };
	        }
	        this.json = function () {
	          return this.text().then(JSON.parse);
	        };
	        return this;
	      }

	      // HTTP methods whose capitalization should be normalized
	      var methods = ['DELETE', 'GET', 'HEAD', 'OPTIONS', 'POST', 'PUT'];
	      function normalizeMethod(method) {
	        var upcased = method.toUpperCase();
	        return methods.indexOf(upcased) > -1 ? upcased : method;
	      }
	      function Request(input, options) {
	        if (!(this instanceof Request)) {
	          throw new TypeError('Please use the "new" operator, this DOM object constructor cannot be called as a function.');
	        }
	        options = options || {};
	        var body = options.body;
	        if (input instanceof Request) {
	          if (input.bodyUsed) {
	            throw new TypeError('Already read');
	          }
	          this.url = input.url;
	          this.credentials = input.credentials;
	          if (!options.headers) {
	            this.headers = new Headers(input.headers);
	          }
	          this.method = input.method;
	          this.mode = input.mode;
	          this.signal = input.signal;
	          if (!body && input._bodyInit != null) {
	            body = input._bodyInit;
	            input.bodyUsed = true;
	          }
	        } else {
	          this.url = String(input);
	        }
	        this.credentials = options.credentials || this.credentials || 'same-origin';
	        if (options.headers || !this.headers) {
	          this.headers = new Headers(options.headers);
	        }
	        this.method = normalizeMethod(options.method || this.method || 'GET');
	        this.mode = options.mode || this.mode || null;
	        this.signal = options.signal || this.signal;
	        this.referrer = null;
	        if ((this.method === 'GET' || this.method === 'HEAD') && body) {
	          throw new TypeError('Body not allowed for GET or HEAD requests');
	        }
	        this._initBody(body);
	        if (this.method === 'GET' || this.method === 'HEAD') {
	          if (options.cache === 'no-store' || options.cache === 'no-cache') {
	            // Search for a '_' parameter in the query string
	            var reParamSearch = /([?&])_=[^&]*/;
	            if (reParamSearch.test(this.url)) {
	              // If it already exists then set the value with the current time
	              this.url = this.url.replace(reParamSearch, '$1_=' + new Date().getTime());
	            } else {
	              // Otherwise add a new '_' parameter to the end with the current time
	              var reQueryString = /\?/;
	              this.url += (reQueryString.test(this.url) ? '&' : '?') + '_=' + new Date().getTime();
	            }
	          }
	        }
	      }
	      Request.prototype.clone = function () {
	        return new Request(this, {
	          body: this._bodyInit
	        });
	      };
	      function decode(body) {
	        var form = new FormData();
	        body.trim().split('&').forEach(function (bytes) {
	          if (bytes) {
	            var split = bytes.split('=');
	            var name = split.shift().replace(/\+/g, ' ');
	            var value = split.join('=').replace(/\+/g, ' ');
	            form.append(decodeURIComponent(name), decodeURIComponent(value));
	          }
	        });
	        return form;
	      }
	      function parseHeaders(rawHeaders) {
	        var headers = new Headers();
	        // Replace instances of \r\n and \n followed by at least one space or horizontal tab with a space
	        // https://tools.ietf.org/html/rfc7230#section-3.2
	        var preProcessedHeaders = rawHeaders.replace(/\r?\n[\t ]+/g, ' ');
	        // Avoiding split via regex to work around a common IE11 bug with the core-js 3.6.0 regex polyfill
	        // https://github.com/github/fetch/issues/748
	        // https://github.com/zloirock/core-js/issues/751
	        preProcessedHeaders.split('\r').map(function (header) {
	          return header.indexOf('\n') === 0 ? header.substr(1, header.length) : header;
	        }).forEach(function (line) {
	          var parts = line.split(':');
	          var key = parts.shift().trim();
	          if (key) {
	            var value = parts.join(':').trim();
	            headers.append(key, value);
	          }
	        });
	        return headers;
	      }
	      Body.call(Request.prototype);
	      function Response(bodyInit, options) {
	        if (!(this instanceof Response)) {
	          throw new TypeError('Please use the "new" operator, this DOM object constructor cannot be called as a function.');
	        }
	        if (!options) {
	          options = {};
	        }
	        this.type = 'default';
	        this.status = options.status === undefined ? 200 : options.status;
	        this.ok = this.status >= 200 && this.status < 300;
	        this.statusText = options.statusText === undefined ? '' : '' + options.statusText;
	        this.headers = new Headers(options.headers);
	        this.url = options.url || '';
	        this._initBody(bodyInit);
	      }
	      Body.call(Response.prototype);
	      Response.prototype.clone = function () {
	        return new Response(this._bodyInit, {
	          status: this.status,
	          statusText: this.statusText,
	          headers: new Headers(this.headers),
	          url: this.url
	        });
	      };
	      Response.error = function () {
	        var response = new Response(null, {
	          status: 0,
	          statusText: ''
	        });
	        response.type = 'error';
	        return response;
	      };
	      var redirectStatuses = [301, 302, 303, 307, 308];
	      Response.redirect = function (url, status) {
	        if (redirectStatuses.indexOf(status) === -1) {
	          throw new RangeError('Invalid status code');
	        }
	        return new Response(null, {
	          status: status,
	          headers: {
	            location: url
	          }
	        });
	      };
	      exports.DOMException = global.DOMException;
	      try {
	        new exports.DOMException();
	      } catch (err) {
	        exports.DOMException = function (message, name) {
	          this.message = message;
	          this.name = name;
	          var error = Error(message);
	          this.stack = error.stack;
	        };
	        exports.DOMException.prototype = Object.create(Error.prototype);
	        exports.DOMException.prototype.constructor = exports.DOMException;
	      }
	      function fetch(input, init) {
	        return new Promise(function (resolve, reject) {
	          var request = new Request(input, init);
	          if (request.signal && request.signal.aborted) {
	            return reject(new exports.DOMException('Aborted', 'AbortError'));
	          }
	          var xhr = new XMLHttpRequest();
	          function abortXhr() {
	            xhr.abort();
	          }
	          xhr.onload = function () {
	            var options = {
	              status: xhr.status,
	              statusText: xhr.statusText,
	              headers: parseHeaders(xhr.getAllResponseHeaders() || '')
	            };
	            options.url = 'responseURL' in xhr ? xhr.responseURL : options.headers.get('X-Request-URL');
	            var body = 'response' in xhr ? xhr.response : xhr.responseText;
	            setTimeout(function () {
	              resolve(new Response(body, options));
	            }, 0);
	          };
	          xhr.onerror = function () {
	            setTimeout(function () {
	              reject(new TypeError('Network request failed'));
	            }, 0);
	          };
	          xhr.ontimeout = function () {
	            setTimeout(function () {
	              reject(new TypeError('Network request failed'));
	            }, 0);
	          };
	          xhr.onabort = function () {
	            setTimeout(function () {
	              reject(new exports.DOMException('Aborted', 'AbortError'));
	            }, 0);
	          };
	          function fixUrl(url) {
	            try {
	              return url === '' && global.location.href ? global.location.href : url;
	            } catch (e) {
	              return url;
	            }
	          }
	          xhr.open(request.method, fixUrl(request.url), true);
	          if (request.credentials === 'include') {
	            xhr.withCredentials = true;
	          } else if (request.credentials === 'omit') {
	            xhr.withCredentials = false;
	          }
	          if ('responseType' in xhr) {
	            if (support.blob) {
	              xhr.responseType = 'blob';
	            } else if (support.arrayBuffer && request.headers.get('Content-Type') && request.headers.get('Content-Type').indexOf('application/octet-stream') !== -1) {
	              xhr.responseType = 'arraybuffer';
	            }
	          }
	          if (init && typeof init.headers === 'object' && !(init.headers instanceof Headers)) {
	            Object.getOwnPropertyNames(init.headers).forEach(function (name) {
	              xhr.setRequestHeader(name, normalizeValue(init.headers[name]));
	            });
	          } else {
	            request.headers.forEach(function (value, name) {
	              xhr.setRequestHeader(name, value);
	            });
	          }
	          if (request.signal) {
	            request.signal.addEventListener('abort', abortXhr);
	            xhr.onreadystatechange = function () {
	              // DONE (success or failure)
	              if (xhr.readyState === 4) {
	                request.signal.removeEventListener('abort', abortXhr);
	              }
	            };
	          }
	          xhr.send(typeof request._bodyInit === 'undefined' ? null : request._bodyInit);
	        });
	      }
	      fetch.polyfill = true;
	      if (!global.fetch) {
	        global.fetch = fetch;
	        global.Headers = Headers;
	        global.Request = Request;
	        global.Response = Response;
	      }
	      exports.Headers = Headers;
	      exports.Request = Request;
	      exports.Response = Response;
	      exports.fetch = fetch;
	      return exports;
	    })({});
	  })(__globalThis__);
	  // This is a ponyfill, so...
	  __globalThis__.fetch.ponyfill = true;
	  delete __globalThis__.fetch.polyfill;
	  // Choose between native implementation (__global__) or custom implementation (__globalThis__)
	  var ctx = __global__.fetch ? __global__ : __globalThis__;
	  exports = ctx.fetch; // To enable: import fetch from 'cross-fetch'
	  exports.default = ctx.fetch; // For TypeScript consumers without esModuleInterop.
	  exports.fetch = ctx.fetch; // To enable: import {fetch} from 'cross-fetch'
	  exports.Headers = ctx.Headers;
	  exports.Request = ctx.Request;
	  exports.Response = ctx.Response;
	  module.exports = exports;
	})(browserPonyfill, browserPonyfill.exports);
	var fetch = browserPonyfill.exports;

	var jsonpath = {exports: {}};

	/*! jsonpath 1.1.1 */
	(function (module, exports) {
	  (function (f) {
	    {
	      module.exports = f();
	    }
	  })(function () {
	    return function e(t, n, r) {
	      function s(o, u) {
	        if (!n[o]) {
	          if (!t[o]) {
	            var a = typeof commonjsRequire == "function" && commonjsRequire;
	            if (!u && a) return a(o, !0);
	            if (i) return i(o, !0);
	            var f = new Error("Cannot find module '" + o + "'");
	            throw f.code = "MODULE_NOT_FOUND", f;
	          }
	          var l = n[o] = {
	            exports: {}
	          };
	          t[o][0].call(l.exports, function (e) {
	            var n = t[o][1][e];
	            return s(n ? n : e);
	          }, l, l.exports, e, t, n, r);
	        }
	        return n[o].exports;
	      }
	      var i = typeof commonjsRequire == "function" && commonjsRequire;
	      for (var o = 0; o < r.length; o++) s(r[o]);
	      return s;
	    }({
	      "./aesprim": [function (require, module, exports) {
	        /*
	          Copyright (C) 2013 Ariya Hidayat <ariya.hidayat@gmail.com>
	          Copyright (C) 2013 Thaddee Tyl <thaddee.tyl@gmail.com>
	          Copyright (C) 2013 Mathias Bynens <mathias@qiwi.be>
	          Copyright (C) 2012 Ariya Hidayat <ariya.hidayat@gmail.com>
	          Copyright (C) 2012 Mathias Bynens <mathias@qiwi.be>
	          Copyright (C) 2012 Joost-Wim Boekesteijn <joost-wim@boekesteijn.nl>
	          Copyright (C) 2012 Kris Kowal <kris.kowal@cixar.com>
	          Copyright (C) 2012 Yusuke Suzuki <utatane.tea@gmail.com>
	          Copyright (C) 2012 Arpad Borsos <arpad.borsos@googlemail.com>
	          Copyright (C) 2011 Ariya Hidayat <ariya.hidayat@gmail.com>
	        
	          Redistribution and use in source and binary forms, with or without
	          modification, are permitted provided that the following conditions are met:
	        
	            * Redistributions of source code must retain the above copyright
	              notice, this list of conditions and the following disclaimer.
	            * Redistributions in binary form must reproduce the above copyright
	              notice, this list of conditions and the following disclaimer in the
	              documentation and/or other materials provided with the distribution.
	        
	          THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
	          AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
	          IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
	          ARE DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
	          DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
	          (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
	          LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
	          ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
	          (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
	          THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
	        */

	        /*jslint bitwise:true plusplus:true */
	        /*global esprima:true, define:true, exports:true, window: true,
	        throwErrorTolerant: true,
	        throwError: true, generateStatement: true, peek: true,
	        parseAssignmentExpression: true, parseBlock: true, parseExpression: true,
	        parseFunctionDeclaration: true, parseFunctionExpression: true,
	        parseFunctionSourceElements: true, parseVariableIdentifier: true,
	        parseLeftHandSideExpression: true,
	        parseUnaryExpression: true,
	        parseStatement: true, parseSourceElement: true */

	        (function (root, factory) {

	          // Universal Module Definition (UMD) to support AMD, CommonJS/Node.js,
	          // Rhino, and plain browser loading.

	          /* istanbul ignore next */
	          if (typeof exports !== 'undefined') {
	            factory(exports);
	          } else {
	            factory(root.esprima = {});
	          }
	        })(this, function (exports) {

	          var Token, TokenName, FnExprTokens, Syntax, PropertyKind, Messages, Regex, SyntaxTreeDelegate, source, strict, index, lineNumber, lineStart, length, delegate, lookahead, state, extra;
	          Token = {
	            BooleanLiteral: 1,
	            EOF: 2,
	            Identifier: 3,
	            Keyword: 4,
	            NullLiteral: 5,
	            NumericLiteral: 6,
	            Punctuator: 7,
	            StringLiteral: 8,
	            RegularExpression: 9
	          };
	          TokenName = {};
	          TokenName[Token.BooleanLiteral] = 'Boolean';
	          TokenName[Token.EOF] = '<end>';
	          TokenName[Token.Identifier] = 'Identifier';
	          TokenName[Token.Keyword] = 'Keyword';
	          TokenName[Token.NullLiteral] = 'Null';
	          TokenName[Token.NumericLiteral] = 'Numeric';
	          TokenName[Token.Punctuator] = 'Punctuator';
	          TokenName[Token.StringLiteral] = 'String';
	          TokenName[Token.RegularExpression] = 'RegularExpression';

	          // A function following one of those tokens is an expression.
	          FnExprTokens = ['(', '{', '[', 'in', 'typeof', 'instanceof', 'new', 'return', 'case', 'delete', 'throw', 'void',
	          // assignment operators
	          '=', '+=', '-=', '*=', '/=', '%=', '<<=', '>>=', '>>>=', '&=', '|=', '^=', ',',
	          // binary/unary operators
	          '+', '-', '*', '/', '%', '++', '--', '<<', '>>', '>>>', '&', '|', '^', '!', '~', '&&', '||', '?', ':', '===', '==', '>=', '<=', '<', '>', '!=', '!=='];
	          Syntax = {
	            AssignmentExpression: 'AssignmentExpression',
	            ArrayExpression: 'ArrayExpression',
	            BlockStatement: 'BlockStatement',
	            BinaryExpression: 'BinaryExpression',
	            BreakStatement: 'BreakStatement',
	            CallExpression: 'CallExpression',
	            CatchClause: 'CatchClause',
	            ConditionalExpression: 'ConditionalExpression',
	            ContinueStatement: 'ContinueStatement',
	            DoWhileStatement: 'DoWhileStatement',
	            DebuggerStatement: 'DebuggerStatement',
	            EmptyStatement: 'EmptyStatement',
	            ExpressionStatement: 'ExpressionStatement',
	            ForStatement: 'ForStatement',
	            ForInStatement: 'ForInStatement',
	            FunctionDeclaration: 'FunctionDeclaration',
	            FunctionExpression: 'FunctionExpression',
	            Identifier: 'Identifier',
	            IfStatement: 'IfStatement',
	            Literal: 'Literal',
	            LabeledStatement: 'LabeledStatement',
	            LogicalExpression: 'LogicalExpression',
	            MemberExpression: 'MemberExpression',
	            NewExpression: 'NewExpression',
	            ObjectExpression: 'ObjectExpression',
	            Program: 'Program',
	            Property: 'Property',
	            ReturnStatement: 'ReturnStatement',
	            SequenceExpression: 'SequenceExpression',
	            SwitchStatement: 'SwitchStatement',
	            SwitchCase: 'SwitchCase',
	            ThisExpression: 'ThisExpression',
	            ThrowStatement: 'ThrowStatement',
	            TryStatement: 'TryStatement',
	            UnaryExpression: 'UnaryExpression',
	            UpdateExpression: 'UpdateExpression',
	            VariableDeclaration: 'VariableDeclaration',
	            VariableDeclarator: 'VariableDeclarator',
	            WhileStatement: 'WhileStatement',
	            WithStatement: 'WithStatement'
	          };
	          PropertyKind = {
	            Data: 1,
	            Get: 2,
	            Set: 4
	          };

	          // Error messages should be identical to V8.
	          Messages = {
	            UnexpectedToken: 'Unexpected token %0',
	            UnexpectedNumber: 'Unexpected number',
	            UnexpectedString: 'Unexpected string',
	            UnexpectedIdentifier: 'Unexpected identifier',
	            UnexpectedReserved: 'Unexpected reserved word',
	            UnexpectedEOS: 'Unexpected end of input',
	            NewlineAfterThrow: 'Illegal newline after throw',
	            InvalidRegExp: 'Invalid regular expression',
	            UnterminatedRegExp: 'Invalid regular expression: missing /',
	            InvalidLHSInAssignment: 'Invalid left-hand side in assignment',
	            InvalidLHSInForIn: 'Invalid left-hand side in for-in',
	            MultipleDefaultsInSwitch: 'More than one default clause in switch statement',
	            NoCatchOrFinally: 'Missing catch or finally after try',
	            UnknownLabel: 'Undefined label \'%0\'',
	            Redeclaration: '%0 \'%1\' has already been declared',
	            IllegalContinue: 'Illegal continue statement',
	            IllegalBreak: 'Illegal break statement',
	            IllegalReturn: 'Illegal return statement',
	            StrictModeWith: 'Strict mode code may not include a with statement',
	            StrictCatchVariable: 'Catch variable may not be eval or arguments in strict mode',
	            StrictVarName: 'Variable name may not be eval or arguments in strict mode',
	            StrictParamName: 'Parameter name eval or arguments is not allowed in strict mode',
	            StrictParamDupe: 'Strict mode function may not have duplicate parameter names',
	            StrictFunctionName: 'Function name may not be eval or arguments in strict mode',
	            StrictOctalLiteral: 'Octal literals are not allowed in strict mode.',
	            StrictDelete: 'Delete of an unqualified identifier in strict mode.',
	            StrictDuplicateProperty: 'Duplicate data property in object literal not allowed in strict mode',
	            AccessorDataProperty: 'Object literal may not have data and accessor property with the same name',
	            AccessorGetSet: 'Object literal may not have multiple get/set accessors with the same name',
	            StrictLHSAssignment: 'Assignment to eval or arguments is not allowed in strict mode',
	            StrictLHSPostfix: 'Postfix increment/decrement may not have eval or arguments operand in strict mode',
	            StrictLHSPrefix: 'Prefix increment/decrement may not have eval or arguments operand in strict mode',
	            StrictReservedWord: 'Use of future reserved word in strict mode'
	          };

	          // See also tools/generate-unicode-regex.py.
	          Regex = {
	            NonAsciiIdentifierStart: new RegExp('[\xAA\xB5\xBA\xC0-\xD6\xD8-\xF6\xF8-\u02C1\u02C6-\u02D1\u02E0-\u02E4\u02EC\u02EE\u0370-\u0374\u0376\u0377\u037A-\u037D\u0386\u0388-\u038A\u038C\u038E-\u03A1\u03A3-\u03F5\u03F7-\u0481\u048A-\u0527\u0531-\u0556\u0559\u0561-\u0587\u05D0-\u05EA\u05F0-\u05F2\u0620-\u064A\u066E\u066F\u0671-\u06D3\u06D5\u06E5\u06E6\u06EE\u06EF\u06FA-\u06FC\u06FF\u0710\u0712-\u072F\u074D-\u07A5\u07B1\u07CA-\u07EA\u07F4\u07F5\u07FA\u0800-\u0815\u081A\u0824\u0828\u0840-\u0858\u08A0\u08A2-\u08AC\u0904-\u0939\u093D\u0950\u0958-\u0961\u0971-\u0977\u0979-\u097F\u0985-\u098C\u098F\u0990\u0993-\u09A8\u09AA-\u09B0\u09B2\u09B6-\u09B9\u09BD\u09CE\u09DC\u09DD\u09DF-\u09E1\u09F0\u09F1\u0A05-\u0A0A\u0A0F\u0A10\u0A13-\u0A28\u0A2A-\u0A30\u0A32\u0A33\u0A35\u0A36\u0A38\u0A39\u0A59-\u0A5C\u0A5E\u0A72-\u0A74\u0A85-\u0A8D\u0A8F-\u0A91\u0A93-\u0AA8\u0AAA-\u0AB0\u0AB2\u0AB3\u0AB5-\u0AB9\u0ABD\u0AD0\u0AE0\u0AE1\u0B05-\u0B0C\u0B0F\u0B10\u0B13-\u0B28\u0B2A-\u0B30\u0B32\u0B33\u0B35-\u0B39\u0B3D\u0B5C\u0B5D\u0B5F-\u0B61\u0B71\u0B83\u0B85-\u0B8A\u0B8E-\u0B90\u0B92-\u0B95\u0B99\u0B9A\u0B9C\u0B9E\u0B9F\u0BA3\u0BA4\u0BA8-\u0BAA\u0BAE-\u0BB9\u0BD0\u0C05-\u0C0C\u0C0E-\u0C10\u0C12-\u0C28\u0C2A-\u0C33\u0C35-\u0C39\u0C3D\u0C58\u0C59\u0C60\u0C61\u0C85-\u0C8C\u0C8E-\u0C90\u0C92-\u0CA8\u0CAA-\u0CB3\u0CB5-\u0CB9\u0CBD\u0CDE\u0CE0\u0CE1\u0CF1\u0CF2\u0D05-\u0D0C\u0D0E-\u0D10\u0D12-\u0D3A\u0D3D\u0D4E\u0D60\u0D61\u0D7A-\u0D7F\u0D85-\u0D96\u0D9A-\u0DB1\u0DB3-\u0DBB\u0DBD\u0DC0-\u0DC6\u0E01-\u0E30\u0E32\u0E33\u0E40-\u0E46\u0E81\u0E82\u0E84\u0E87\u0E88\u0E8A\u0E8D\u0E94-\u0E97\u0E99-\u0E9F\u0EA1-\u0EA3\u0EA5\u0EA7\u0EAA\u0EAB\u0EAD-\u0EB0\u0EB2\u0EB3\u0EBD\u0EC0-\u0EC4\u0EC6\u0EDC-\u0EDF\u0F00\u0F40-\u0F47\u0F49-\u0F6C\u0F88-\u0F8C\u1000-\u102A\u103F\u1050-\u1055\u105A-\u105D\u1061\u1065\u1066\u106E-\u1070\u1075-\u1081\u108E\u10A0-\u10C5\u10C7\u10CD\u10D0-\u10FA\u10FC-\u1248\u124A-\u124D\u1250-\u1256\u1258\u125A-\u125D\u1260-\u1288\u128A-\u128D\u1290-\u12B0\u12B2-\u12B5\u12B8-\u12BE\u12C0\u12C2-\u12C5\u12C8-\u12D6\u12D8-\u1310\u1312-\u1315\u1318-\u135A\u1380-\u138F\u13A0-\u13F4\u1401-\u166C\u166F-\u167F\u1681-\u169A\u16A0-\u16EA\u16EE-\u16F0\u1700-\u170C\u170E-\u1711\u1720-\u1731\u1740-\u1751\u1760-\u176C\u176E-\u1770\u1780-\u17B3\u17D7\u17DC\u1820-\u1877\u1880-\u18A8\u18AA\u18B0-\u18F5\u1900-\u191C\u1950-\u196D\u1970-\u1974\u1980-\u19AB\u19C1-\u19C7\u1A00-\u1A16\u1A20-\u1A54\u1AA7\u1B05-\u1B33\u1B45-\u1B4B\u1B83-\u1BA0\u1BAE\u1BAF\u1BBA-\u1BE5\u1C00-\u1C23\u1C4D-\u1C4F\u1C5A-\u1C7D\u1CE9-\u1CEC\u1CEE-\u1CF1\u1CF5\u1CF6\u1D00-\u1DBF\u1E00-\u1F15\u1F18-\u1F1D\u1F20-\u1F45\u1F48-\u1F4D\u1F50-\u1F57\u1F59\u1F5B\u1F5D\u1F5F-\u1F7D\u1F80-\u1FB4\u1FB6-\u1FBC\u1FBE\u1FC2-\u1FC4\u1FC6-\u1FCC\u1FD0-\u1FD3\u1FD6-\u1FDB\u1FE0-\u1FEC\u1FF2-\u1FF4\u1FF6-\u1FFC\u2071\u207F\u2090-\u209C\u2102\u2107\u210A-\u2113\u2115\u2119-\u211D\u2124\u2126\u2128\u212A-\u212D\u212F-\u2139\u213C-\u213F\u2145-\u2149\u214E\u2160-\u2188\u2C00-\u2C2E\u2C30-\u2C5E\u2C60-\u2CE4\u2CEB-\u2CEE\u2CF2\u2CF3\u2D00-\u2D25\u2D27\u2D2D\u2D30-\u2D67\u2D6F\u2D80-\u2D96\u2DA0-\u2DA6\u2DA8-\u2DAE\u2DB0-\u2DB6\u2DB8-\u2DBE\u2DC0-\u2DC6\u2DC8-\u2DCE\u2DD0-\u2DD6\u2DD8-\u2DDE\u2E2F\u3005-\u3007\u3021-\u3029\u3031-\u3035\u3038-\u303C\u3041-\u3096\u309D-\u309F\u30A1-\u30FA\u30FC-\u30FF\u3105-\u312D\u3131-\u318E\u31A0-\u31BA\u31F0-\u31FF\u3400-\u4DB5\u4E00-\u9FCC\uA000-\uA48C\uA4D0-\uA4FD\uA500-\uA60C\uA610-\uA61F\uA62A\uA62B\uA640-\uA66E\uA67F-\uA697\uA6A0-\uA6EF\uA717-\uA71F\uA722-\uA788\uA78B-\uA78E\uA790-\uA793\uA7A0-\uA7AA\uA7F8-\uA801\uA803-\uA805\uA807-\uA80A\uA80C-\uA822\uA840-\uA873\uA882-\uA8B3\uA8F2-\uA8F7\uA8FB\uA90A-\uA925\uA930-\uA946\uA960-\uA97C\uA984-\uA9B2\uA9CF\uAA00-\uAA28\uAA40-\uAA42\uAA44-\uAA4B\uAA60-\uAA76\uAA7A\uAA80-\uAAAF\uAAB1\uAAB5\uAAB6\uAAB9-\uAABD\uAAC0\uAAC2\uAADB-\uAADD\uAAE0-\uAAEA\uAAF2-\uAAF4\uAB01-\uAB06\uAB09-\uAB0E\uAB11-\uAB16\uAB20-\uAB26\uAB28-\uAB2E\uABC0-\uABE2\uAC00-\uD7A3\uD7B0-\uD7C6\uD7CB-\uD7FB\uF900-\uFA6D\uFA70-\uFAD9\uFB00-\uFB06\uFB13-\uFB17\uFB1D\uFB1F-\uFB28\uFB2A-\uFB36\uFB38-\uFB3C\uFB3E\uFB40\uFB41\uFB43\uFB44\uFB46-\uFBB1\uFBD3-\uFD3D\uFD50-\uFD8F\uFD92-\uFDC7\uFDF0-\uFDFB\uFE70-\uFE74\uFE76-\uFEFC\uFF21-\uFF3A\uFF41-\uFF5A\uFF66-\uFFBE\uFFC2-\uFFC7\uFFCA-\uFFCF\uFFD2-\uFFD7\uFFDA-\uFFDC]'),
	            NonAsciiIdentifierPart: new RegExp('[\xAA\xB5\xBA\xC0-\xD6\xD8-\xF6\xF8-\u02C1\u02C6-\u02D1\u02E0-\u02E4\u02EC\u02EE\u0300-\u0374\u0376\u0377\u037A-\u037D\u0386\u0388-\u038A\u038C\u038E-\u03A1\u03A3-\u03F5\u03F7-\u0481\u0483-\u0487\u048A-\u0527\u0531-\u0556\u0559\u0561-\u0587\u0591-\u05BD\u05BF\u05C1\u05C2\u05C4\u05C5\u05C7\u05D0-\u05EA\u05F0-\u05F2\u0610-\u061A\u0620-\u0669\u066E-\u06D3\u06D5-\u06DC\u06DF-\u06E8\u06EA-\u06FC\u06FF\u0710-\u074A\u074D-\u07B1\u07C0-\u07F5\u07FA\u0800-\u082D\u0840-\u085B\u08A0\u08A2-\u08AC\u08E4-\u08FE\u0900-\u0963\u0966-\u096F\u0971-\u0977\u0979-\u097F\u0981-\u0983\u0985-\u098C\u098F\u0990\u0993-\u09A8\u09AA-\u09B0\u09B2\u09B6-\u09B9\u09BC-\u09C4\u09C7\u09C8\u09CB-\u09CE\u09D7\u09DC\u09DD\u09DF-\u09E3\u09E6-\u09F1\u0A01-\u0A03\u0A05-\u0A0A\u0A0F\u0A10\u0A13-\u0A28\u0A2A-\u0A30\u0A32\u0A33\u0A35\u0A36\u0A38\u0A39\u0A3C\u0A3E-\u0A42\u0A47\u0A48\u0A4B-\u0A4D\u0A51\u0A59-\u0A5C\u0A5E\u0A66-\u0A75\u0A81-\u0A83\u0A85-\u0A8D\u0A8F-\u0A91\u0A93-\u0AA8\u0AAA-\u0AB0\u0AB2\u0AB3\u0AB5-\u0AB9\u0ABC-\u0AC5\u0AC7-\u0AC9\u0ACB-\u0ACD\u0AD0\u0AE0-\u0AE3\u0AE6-\u0AEF\u0B01-\u0B03\u0B05-\u0B0C\u0B0F\u0B10\u0B13-\u0B28\u0B2A-\u0B30\u0B32\u0B33\u0B35-\u0B39\u0B3C-\u0B44\u0B47\u0B48\u0B4B-\u0B4D\u0B56\u0B57\u0B5C\u0B5D\u0B5F-\u0B63\u0B66-\u0B6F\u0B71\u0B82\u0B83\u0B85-\u0B8A\u0B8E-\u0B90\u0B92-\u0B95\u0B99\u0B9A\u0B9C\u0B9E\u0B9F\u0BA3\u0BA4\u0BA8-\u0BAA\u0BAE-\u0BB9\u0BBE-\u0BC2\u0BC6-\u0BC8\u0BCA-\u0BCD\u0BD0\u0BD7\u0BE6-\u0BEF\u0C01-\u0C03\u0C05-\u0C0C\u0C0E-\u0C10\u0C12-\u0C28\u0C2A-\u0C33\u0C35-\u0C39\u0C3D-\u0C44\u0C46-\u0C48\u0C4A-\u0C4D\u0C55\u0C56\u0C58\u0C59\u0C60-\u0C63\u0C66-\u0C6F\u0C82\u0C83\u0C85-\u0C8C\u0C8E-\u0C90\u0C92-\u0CA8\u0CAA-\u0CB3\u0CB5-\u0CB9\u0CBC-\u0CC4\u0CC6-\u0CC8\u0CCA-\u0CCD\u0CD5\u0CD6\u0CDE\u0CE0-\u0CE3\u0CE6-\u0CEF\u0CF1\u0CF2\u0D02\u0D03\u0D05-\u0D0C\u0D0E-\u0D10\u0D12-\u0D3A\u0D3D-\u0D44\u0D46-\u0D48\u0D4A-\u0D4E\u0D57\u0D60-\u0D63\u0D66-\u0D6F\u0D7A-\u0D7F\u0D82\u0D83\u0D85-\u0D96\u0D9A-\u0DB1\u0DB3-\u0DBB\u0DBD\u0DC0-\u0DC6\u0DCA\u0DCF-\u0DD4\u0DD6\u0DD8-\u0DDF\u0DF2\u0DF3\u0E01-\u0E3A\u0E40-\u0E4E\u0E50-\u0E59\u0E81\u0E82\u0E84\u0E87\u0E88\u0E8A\u0E8D\u0E94-\u0E97\u0E99-\u0E9F\u0EA1-\u0EA3\u0EA5\u0EA7\u0EAA\u0EAB\u0EAD-\u0EB9\u0EBB-\u0EBD\u0EC0-\u0EC4\u0EC6\u0EC8-\u0ECD\u0ED0-\u0ED9\u0EDC-\u0EDF\u0F00\u0F18\u0F19\u0F20-\u0F29\u0F35\u0F37\u0F39\u0F3E-\u0F47\u0F49-\u0F6C\u0F71-\u0F84\u0F86-\u0F97\u0F99-\u0FBC\u0FC6\u1000-\u1049\u1050-\u109D\u10A0-\u10C5\u10C7\u10CD\u10D0-\u10FA\u10FC-\u1248\u124A-\u124D\u1250-\u1256\u1258\u125A-\u125D\u1260-\u1288\u128A-\u128D\u1290-\u12B0\u12B2-\u12B5\u12B8-\u12BE\u12C0\u12C2-\u12C5\u12C8-\u12D6\u12D8-\u1310\u1312-\u1315\u1318-\u135A\u135D-\u135F\u1380-\u138F\u13A0-\u13F4\u1401-\u166C\u166F-\u167F\u1681-\u169A\u16A0-\u16EA\u16EE-\u16F0\u1700-\u170C\u170E-\u1714\u1720-\u1734\u1740-\u1753\u1760-\u176C\u176E-\u1770\u1772\u1773\u1780-\u17D3\u17D7\u17DC\u17DD\u17E0-\u17E9\u180B-\u180D\u1810-\u1819\u1820-\u1877\u1880-\u18AA\u18B0-\u18F5\u1900-\u191C\u1920-\u192B\u1930-\u193B\u1946-\u196D\u1970-\u1974\u1980-\u19AB\u19B0-\u19C9\u19D0-\u19D9\u1A00-\u1A1B\u1A20-\u1A5E\u1A60-\u1A7C\u1A7F-\u1A89\u1A90-\u1A99\u1AA7\u1B00-\u1B4B\u1B50-\u1B59\u1B6B-\u1B73\u1B80-\u1BF3\u1C00-\u1C37\u1C40-\u1C49\u1C4D-\u1C7D\u1CD0-\u1CD2\u1CD4-\u1CF6\u1D00-\u1DE6\u1DFC-\u1F15\u1F18-\u1F1D\u1F20-\u1F45\u1F48-\u1F4D\u1F50-\u1F57\u1F59\u1F5B\u1F5D\u1F5F-\u1F7D\u1F80-\u1FB4\u1FB6-\u1FBC\u1FBE\u1FC2-\u1FC4\u1FC6-\u1FCC\u1FD0-\u1FD3\u1FD6-\u1FDB\u1FE0-\u1FEC\u1FF2-\u1FF4\u1FF6-\u1FFC\u200C\u200D\u203F\u2040\u2054\u2071\u207F\u2090-\u209C\u20D0-\u20DC\u20E1\u20E5-\u20F0\u2102\u2107\u210A-\u2113\u2115\u2119-\u211D\u2124\u2126\u2128\u212A-\u212D\u212F-\u2139\u213C-\u213F\u2145-\u2149\u214E\u2160-\u2188\u2C00-\u2C2E\u2C30-\u2C5E\u2C60-\u2CE4\u2CEB-\u2CF3\u2D00-\u2D25\u2D27\u2D2D\u2D30-\u2D67\u2D6F\u2D7F-\u2D96\u2DA0-\u2DA6\u2DA8-\u2DAE\u2DB0-\u2DB6\u2DB8-\u2DBE\u2DC0-\u2DC6\u2DC8-\u2DCE\u2DD0-\u2DD6\u2DD8-\u2DDE\u2DE0-\u2DFF\u2E2F\u3005-\u3007\u3021-\u302F\u3031-\u3035\u3038-\u303C\u3041-\u3096\u3099\u309A\u309D-\u309F\u30A1-\u30FA\u30FC-\u30FF\u3105-\u312D\u3131-\u318E\u31A0-\u31BA\u31F0-\u31FF\u3400-\u4DB5\u4E00-\u9FCC\uA000-\uA48C\uA4D0-\uA4FD\uA500-\uA60C\uA610-\uA62B\uA640-\uA66F\uA674-\uA67D\uA67F-\uA697\uA69F-\uA6F1\uA717-\uA71F\uA722-\uA788\uA78B-\uA78E\uA790-\uA793\uA7A0-\uA7AA\uA7F8-\uA827\uA840-\uA873\uA880-\uA8C4\uA8D0-\uA8D9\uA8E0-\uA8F7\uA8FB\uA900-\uA92D\uA930-\uA953\uA960-\uA97C\uA980-\uA9C0\uA9CF-\uA9D9\uAA00-\uAA36\uAA40-\uAA4D\uAA50-\uAA59\uAA60-\uAA76\uAA7A\uAA7B\uAA80-\uAAC2\uAADB-\uAADD\uAAE0-\uAAEF\uAAF2-\uAAF6\uAB01-\uAB06\uAB09-\uAB0E\uAB11-\uAB16\uAB20-\uAB26\uAB28-\uAB2E\uABC0-\uABEA\uABEC\uABED\uABF0-\uABF9\uAC00-\uD7A3\uD7B0-\uD7C6\uD7CB-\uD7FB\uF900-\uFA6D\uFA70-\uFAD9\uFB00-\uFB06\uFB13-\uFB17\uFB1D-\uFB28\uFB2A-\uFB36\uFB38-\uFB3C\uFB3E\uFB40\uFB41\uFB43\uFB44\uFB46-\uFBB1\uFBD3-\uFD3D\uFD50-\uFD8F\uFD92-\uFDC7\uFDF0-\uFDFB\uFE00-\uFE0F\uFE20-\uFE26\uFE33\uFE34\uFE4D-\uFE4F\uFE70-\uFE74\uFE76-\uFEFC\uFF10-\uFF19\uFF21-\uFF3A\uFF3F\uFF41-\uFF5A\uFF66-\uFFBE\uFFC2-\uFFC7\uFFCA-\uFFCF\uFFD2-\uFFD7\uFFDA-\uFFDC]')
	          };

	          // Ensure the condition is true, otherwise throw an error.
	          // This is only to have a better contract semantic, i.e. another safety net
	          // to catch a logic error. The condition shall be fulfilled in normal case.
	          // Do NOT use this to enforce a certain condition on any user input.

	          function assert(condition, message) {
	            /* istanbul ignore if */
	            if (!condition) {
	              throw new Error('ASSERT: ' + message);
	            }
	          }
	          function isDecimalDigit(ch) {
	            return ch >= 48 && ch <= 57; // 0..9
	          }
	          function isHexDigit(ch) {
	            return '0123456789abcdefABCDEF'.indexOf(ch) >= 0;
	          }
	          function isOctalDigit(ch) {
	            return '01234567'.indexOf(ch) >= 0;
	          }

	          // 7.2 White Space

	          function isWhiteSpace(ch) {
	            return ch === 0x20 || ch === 0x09 || ch === 0x0B || ch === 0x0C || ch === 0xA0 || ch >= 0x1680 && [0x1680, 0x180E, 0x2000, 0x2001, 0x2002, 0x2003, 0x2004, 0x2005, 0x2006, 0x2007, 0x2008, 0x2009, 0x200A, 0x202F, 0x205F, 0x3000, 0xFEFF].indexOf(ch) >= 0;
	          }

	          // 7.3 Line Terminators

	          function isLineTerminator(ch) {
	            return ch === 0x0A || ch === 0x0D || ch === 0x2028 || ch === 0x2029;
	          }

	          // 7.6 Identifier Names and Identifiers

	          function isIdentifierStart(ch) {
	            return ch == 0x40 || ch === 0x24 || ch === 0x5F ||
	            // $ (dollar) and _ (underscore)
	            ch >= 0x41 && ch <= 0x5A ||
	            // A..Z
	            ch >= 0x61 && ch <= 0x7A ||
	            // a..z
	            ch === 0x5C ||
	            // \ (backslash)
	            ch >= 0x80 && Regex.NonAsciiIdentifierStart.test(String.fromCharCode(ch));
	          }
	          function isIdentifierPart(ch) {
	            return ch === 0x24 || ch === 0x5F ||
	            // $ (dollar) and _ (underscore)
	            ch >= 0x41 && ch <= 0x5A ||
	            // A..Z
	            ch >= 0x61 && ch <= 0x7A ||
	            // a..z
	            ch >= 0x30 && ch <= 0x39 ||
	            // 0..9
	            ch === 0x5C ||
	            // \ (backslash)
	            ch >= 0x80 && Regex.NonAsciiIdentifierPart.test(String.fromCharCode(ch));
	          }

	          // 7.6.1.2 Future Reserved Words

	          function isFutureReservedWord(id) {
	            switch (id) {
	              case 'class':
	              case 'enum':
	              case 'export':
	              case 'extends':
	              case 'import':
	              case 'super':
	                return true;
	              default:
	                return false;
	            }
	          }
	          function isStrictModeReservedWord(id) {
	            switch (id) {
	              case 'implements':
	              case 'interface':
	              case 'package':
	              case 'private':
	              case 'protected':
	              case 'public':
	              case 'static':
	              case 'yield':
	              case 'let':
	                return true;
	              default:
	                return false;
	            }
	          }
	          function isRestrictedWord(id) {
	            return id === 'eval' || id === 'arguments';
	          }

	          // 7.6.1.1 Keywords

	          function isKeyword(id) {
	            if (strict && isStrictModeReservedWord(id)) {
	              return true;
	            }

	            // 'const' is specialized as Keyword in V8.
	            // 'yield' and 'let' are for compatiblity with SpiderMonkey and ES.next.
	            // Some others are from future reserved words.

	            switch (id.length) {
	              case 2:
	                return id === 'if' || id === 'in' || id === 'do';
	              case 3:
	                return id === 'var' || id === 'for' || id === 'new' || id === 'try' || id === 'let';
	              case 4:
	                return id === 'this' || id === 'else' || id === 'case' || id === 'void' || id === 'with' || id === 'enum';
	              case 5:
	                return id === 'while' || id === 'break' || id === 'catch' || id === 'throw' || id === 'const' || id === 'yield' || id === 'class' || id === 'super';
	              case 6:
	                return id === 'return' || id === 'typeof' || id === 'delete' || id === 'switch' || id === 'export' || id === 'import';
	              case 7:
	                return id === 'default' || id === 'finally' || id === 'extends';
	              case 8:
	                return id === 'function' || id === 'continue' || id === 'debugger';
	              case 10:
	                return id === 'instanceof';
	              default:
	                return false;
	            }
	          }

	          // 7.4 Comments

	          function addComment(type, value, start, end, loc) {
	            var comment;
	            assert(typeof start === 'number', 'Comment must have valid position');

	            // Because the way the actual token is scanned, often the comments
	            // (if any) are skipped twice during the lexical analysis.
	            // Thus, we need to skip adding a comment if the comment array already
	            // handled it.
	            if (state.lastCommentStart >= start) {
	              return;
	            }
	            state.lastCommentStart = start;
	            comment = {
	              type: type,
	              value: value
	            };
	            if (extra.range) {
	              comment.range = [start, end];
	            }
	            if (extra.loc) {
	              comment.loc = loc;
	            }
	            extra.comments.push(comment);
	            if (extra.attachComment) {
	              extra.leadingComments.push(comment);
	              extra.trailingComments.push(comment);
	            }
	          }
	          function skipSingleLineComment(offset) {
	            var start, loc, ch, comment;
	            start = index - offset;
	            loc = {
	              start: {
	                line: lineNumber,
	                column: index - lineStart - offset
	              }
	            };
	            while (index < length) {
	              ch = source.charCodeAt(index);
	              ++index;
	              if (isLineTerminator(ch)) {
	                if (extra.comments) {
	                  comment = source.slice(start + offset, index - 1);
	                  loc.end = {
	                    line: lineNumber,
	                    column: index - lineStart - 1
	                  };
	                  addComment('Line', comment, start, index - 1, loc);
	                }
	                if (ch === 13 && source.charCodeAt(index) === 10) {
	                  ++index;
	                }
	                ++lineNumber;
	                lineStart = index;
	                return;
	              }
	            }
	            if (extra.comments) {
	              comment = source.slice(start + offset, index);
	              loc.end = {
	                line: lineNumber,
	                column: index - lineStart
	              };
	              addComment('Line', comment, start, index, loc);
	            }
	          }
	          function skipMultiLineComment() {
	            var start, loc, ch, comment;
	            if (extra.comments) {
	              start = index - 2;
	              loc = {
	                start: {
	                  line: lineNumber,
	                  column: index - lineStart - 2
	                }
	              };
	            }
	            while (index < length) {
	              ch = source.charCodeAt(index);
	              if (isLineTerminator(ch)) {
	                if (ch === 0x0D && source.charCodeAt(index + 1) === 0x0A) {
	                  ++index;
	                }
	                ++lineNumber;
	                ++index;
	                lineStart = index;
	                if (index >= length) {
	                  throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
	                }
	              } else if (ch === 0x2A) {
	                // Block comment ends with '*/'.
	                if (source.charCodeAt(index + 1) === 0x2F) {
	                  ++index;
	                  ++index;
	                  if (extra.comments) {
	                    comment = source.slice(start + 2, index - 2);
	                    loc.end = {
	                      line: lineNumber,
	                      column: index - lineStart
	                    };
	                    addComment('Block', comment, start, index, loc);
	                  }
	                  return;
	                }
	                ++index;
	              } else {
	                ++index;
	              }
	            }
	            throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
	          }
	          function skipComment() {
	            var ch, start;
	            start = index === 0;
	            while (index < length) {
	              ch = source.charCodeAt(index);
	              if (isWhiteSpace(ch)) {
	                ++index;
	              } else if (isLineTerminator(ch)) {
	                ++index;
	                if (ch === 0x0D && source.charCodeAt(index) === 0x0A) {
	                  ++index;
	                }
	                ++lineNumber;
	                lineStart = index;
	                start = true;
	              } else if (ch === 0x2F) {
	                // U+002F is '/'
	                ch = source.charCodeAt(index + 1);
	                if (ch === 0x2F) {
	                  ++index;
	                  ++index;
	                  skipSingleLineComment(2);
	                  start = true;
	                } else if (ch === 0x2A) {
	                  // U+002A is '*'
	                  ++index;
	                  ++index;
	                  skipMultiLineComment();
	                } else {
	                  break;
	                }
	              } else if (start && ch === 0x2D) {
	                // U+002D is '-'
	                // U+003E is '>'
	                if (source.charCodeAt(index + 1) === 0x2D && source.charCodeAt(index + 2) === 0x3E) {
	                  // '-->' is a single-line comment
	                  index += 3;
	                  skipSingleLineComment(3);
	                } else {
	                  break;
	                }
	              } else if (ch === 0x3C) {
	                // U+003C is '<'
	                if (source.slice(index + 1, index + 4) === '!--') {
	                  ++index; // `<`
	                  ++index; // `!`
	                  ++index; // `-`
	                  ++index; // `-`
	                  skipSingleLineComment(4);
	                } else {
	                  break;
	                }
	              } else {
	                break;
	              }
	            }
	          }
	          function scanHexEscape(prefix) {
	            var i,
	              len,
	              ch,
	              code = 0;
	            len = prefix === 'u' ? 4 : 2;
	            for (i = 0; i < len; ++i) {
	              if (index < length && isHexDigit(source[index])) {
	                ch = source[index++];
	                code = code * 16 + '0123456789abcdef'.indexOf(ch.toLowerCase());
	              } else {
	                return '';
	              }
	            }
	            return String.fromCharCode(code);
	          }
	          function getEscapedIdentifier() {
	            var ch, id;
	            ch = source.charCodeAt(index++);
	            id = String.fromCharCode(ch);

	            // '\u' (U+005C, U+0075) denotes an escaped character.
	            if (ch === 0x5C) {
	              if (source.charCodeAt(index) !== 0x75) {
	                throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
	              }
	              ++index;
	              ch = scanHexEscape('u');
	              if (!ch || ch === '\\' || !isIdentifierStart(ch.charCodeAt(0))) {
	                throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
	              }
	              id = ch;
	            }
	            while (index < length) {
	              ch = source.charCodeAt(index);
	              if (!isIdentifierPart(ch)) {
	                break;
	              }
	              ++index;
	              id += String.fromCharCode(ch);

	              // '\u' (U+005C, U+0075) denotes an escaped character.
	              if (ch === 0x5C) {
	                id = id.substr(0, id.length - 1);
	                if (source.charCodeAt(index) !== 0x75) {
	                  throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
	                }
	                ++index;
	                ch = scanHexEscape('u');
	                if (!ch || ch === '\\' || !isIdentifierPart(ch.charCodeAt(0))) {
	                  throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
	                }
	                id += ch;
	              }
	            }
	            return id;
	          }
	          function getIdentifier() {
	            var start, ch;
	            start = index++;
	            while (index < length) {
	              ch = source.charCodeAt(index);
	              if (ch === 0x5C) {
	                // Blackslash (U+005C) marks Unicode escape sequence.
	                index = start;
	                return getEscapedIdentifier();
	              }
	              if (isIdentifierPart(ch)) {
	                ++index;
	              } else {
	                break;
	              }
	            }
	            return source.slice(start, index);
	          }
	          function scanIdentifier() {
	            var start, id, type;
	            start = index;

	            // Backslash (U+005C) starts an escaped character.
	            id = source.charCodeAt(index) === 0x5C ? getEscapedIdentifier() : getIdentifier();

	            // There is no keyword or literal with only one character.
	            // Thus, it must be an identifier.
	            if (id.length === 1) {
	              type = Token.Identifier;
	            } else if (isKeyword(id)) {
	              type = Token.Keyword;
	            } else if (id === 'null') {
	              type = Token.NullLiteral;
	            } else if (id === 'true' || id === 'false') {
	              type = Token.BooleanLiteral;
	            } else {
	              type = Token.Identifier;
	            }
	            return {
	              type: type,
	              value: id,
	              lineNumber: lineNumber,
	              lineStart: lineStart,
	              start: start,
	              end: index
	            };
	          }

	          // 7.7 Punctuators

	          function scanPunctuator() {
	            var start = index,
	              code = source.charCodeAt(index),
	              code2,
	              ch1 = source[index],
	              ch2,
	              ch3,
	              ch4;
	            switch (code) {
	              // Check for most common single-character punctuators.
	              case 0x2E: // . dot
	              case 0x28: // ( open bracket
	              case 0x29: // ) close bracket
	              case 0x3B: // ; semicolon
	              case 0x2C: // , comma
	              case 0x7B: // { open curly brace
	              case 0x7D: // } close curly brace
	              case 0x5B: // [
	              case 0x5D: // ]
	              case 0x3A: // :
	              case 0x3F: // ?
	              case 0x7E:
	                // ~
	                ++index;
	                if (extra.tokenize) {
	                  if (code === 0x28) {
	                    extra.openParenToken = extra.tokens.length;
	                  } else if (code === 0x7B) {
	                    extra.openCurlyToken = extra.tokens.length;
	                  }
	                }
	                return {
	                  type: Token.Punctuator,
	                  value: String.fromCharCode(code),
	                  lineNumber: lineNumber,
	                  lineStart: lineStart,
	                  start: start,
	                  end: index
	                };
	              default:
	                code2 = source.charCodeAt(index + 1);

	                // '=' (U+003D) marks an assignment or comparison operator.
	                if (code2 === 0x3D) {
	                  switch (code) {
	                    case 0x2B: // +
	                    case 0x2D: // -
	                    case 0x2F: // /
	                    case 0x3C: // <
	                    case 0x3E: // >
	                    case 0x5E: // ^
	                    case 0x7C: // |
	                    case 0x25: // %
	                    case 0x26: // &
	                    case 0x2A:
	                      // *
	                      index += 2;
	                      return {
	                        type: Token.Punctuator,
	                        value: String.fromCharCode(code) + String.fromCharCode(code2),
	                        lineNumber: lineNumber,
	                        lineStart: lineStart,
	                        start: start,
	                        end: index
	                      };
	                    case 0x21: // !
	                    case 0x3D:
	                      // =
	                      index += 2;

	                      // !== and ===
	                      if (source.charCodeAt(index) === 0x3D) {
	                        ++index;
	                      }
	                      return {
	                        type: Token.Punctuator,
	                        value: source.slice(start, index),
	                        lineNumber: lineNumber,
	                        lineStart: lineStart,
	                        start: start,
	                        end: index
	                      };
	                  }
	                }
	            }

	            // 4-character punctuator: >>>=

	            ch4 = source.substr(index, 4);
	            if (ch4 === '>>>=') {
	              index += 4;
	              return {
	                type: Token.Punctuator,
	                value: ch4,
	                lineNumber: lineNumber,
	                lineStart: lineStart,
	                start: start,
	                end: index
	              };
	            }

	            // 3-character punctuators: === !== >>> <<= >>=

	            ch3 = ch4.substr(0, 3);
	            if (ch3 === '>>>' || ch3 === '<<=' || ch3 === '>>=') {
	              index += 3;
	              return {
	                type: Token.Punctuator,
	                value: ch3,
	                lineNumber: lineNumber,
	                lineStart: lineStart,
	                start: start,
	                end: index
	              };
	            }

	            // Other 2-character punctuators: ++ -- << >> && ||
	            ch2 = ch3.substr(0, 2);
	            if (ch1 === ch2[1] && '+-<>&|'.indexOf(ch1) >= 0 || ch2 === '=>') {
	              index += 2;
	              return {
	                type: Token.Punctuator,
	                value: ch2,
	                lineNumber: lineNumber,
	                lineStart: lineStart,
	                start: start,
	                end: index
	              };
	            }

	            // 1-character punctuators: < > = ! + - * % & | ^ /
	            if ('<>=!+-*%&|^/'.indexOf(ch1) >= 0) {
	              ++index;
	              return {
	                type: Token.Punctuator,
	                value: ch1,
	                lineNumber: lineNumber,
	                lineStart: lineStart,
	                start: start,
	                end: index
	              };
	            }
	            throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
	          }

	          // 7.8.3 Numeric Literals

	          function scanHexLiteral(start) {
	            var number = '';
	            while (index < length) {
	              if (!isHexDigit(source[index])) {
	                break;
	              }
	              number += source[index++];
	            }
	            if (number.length === 0) {
	              throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
	            }
	            if (isIdentifierStart(source.charCodeAt(index))) {
	              throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
	            }
	            return {
	              type: Token.NumericLiteral,
	              value: parseInt('0x' + number, 16),
	              lineNumber: lineNumber,
	              lineStart: lineStart,
	              start: start,
	              end: index
	            };
	          }
	          function scanOctalLiteral(start) {
	            var number = '0' + source[index++];
	            while (index < length) {
	              if (!isOctalDigit(source[index])) {
	                break;
	              }
	              number += source[index++];
	            }
	            if (isIdentifierStart(source.charCodeAt(index)) || isDecimalDigit(source.charCodeAt(index))) {
	              throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
	            }
	            return {
	              type: Token.NumericLiteral,
	              value: parseInt(number, 8),
	              octal: true,
	              lineNumber: lineNumber,
	              lineStart: lineStart,
	              start: start,
	              end: index
	            };
	          }
	          function scanNumericLiteral() {
	            var number, start, ch;
	            ch = source[index];
	            assert(isDecimalDigit(ch.charCodeAt(0)) || ch === '.', 'Numeric literal must start with a decimal digit or a decimal point');
	            start = index;
	            number = '';
	            if (ch !== '.') {
	              number = source[index++];
	              ch = source[index];

	              // Hex number starts with '0x'.
	              // Octal number starts with '0'.
	              if (number === '0') {
	                if (ch === 'x' || ch === 'X') {
	                  ++index;
	                  return scanHexLiteral(start);
	                }
	                if (isOctalDigit(ch)) {
	                  return scanOctalLiteral(start);
	                }

	                // decimal number starts with '0' such as '09' is illegal.
	                if (ch && isDecimalDigit(ch.charCodeAt(0))) {
	                  throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
	                }
	              }
	              while (isDecimalDigit(source.charCodeAt(index))) {
	                number += source[index++];
	              }
	              ch = source[index];
	            }
	            if (ch === '.') {
	              number += source[index++];
	              while (isDecimalDigit(source.charCodeAt(index))) {
	                number += source[index++];
	              }
	              ch = source[index];
	            }
	            if (ch === 'e' || ch === 'E') {
	              number += source[index++];
	              ch = source[index];
	              if (ch === '+' || ch === '-') {
	                number += source[index++];
	              }
	              if (isDecimalDigit(source.charCodeAt(index))) {
	                while (isDecimalDigit(source.charCodeAt(index))) {
	                  number += source[index++];
	                }
	              } else {
	                throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
	              }
	            }
	            if (isIdentifierStart(source.charCodeAt(index))) {
	              throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
	            }
	            return {
	              type: Token.NumericLiteral,
	              value: parseFloat(number),
	              lineNumber: lineNumber,
	              lineStart: lineStart,
	              start: start,
	              end: index
	            };
	          }

	          // 7.8.4 String Literals

	          function scanStringLiteral() {
	            var str = '',
	              quote,
	              start,
	              ch,
	              code,
	              unescaped,
	              restore,
	              octal = false,
	              startLineNumber,
	              startLineStart;
	            startLineNumber = lineNumber;
	            startLineStart = lineStart;
	            quote = source[index];
	            assert(quote === '\'' || quote === '"', 'String literal must starts with a quote');
	            start = index;
	            ++index;
	            while (index < length) {
	              ch = source[index++];
	              if (ch === quote) {
	                quote = '';
	                break;
	              } else if (ch === '\\') {
	                ch = source[index++];
	                if (!ch || !isLineTerminator(ch.charCodeAt(0))) {
	                  switch (ch) {
	                    case 'u':
	                    case 'x':
	                      restore = index;
	                      unescaped = scanHexEscape(ch);
	                      if (unescaped) {
	                        str += unescaped;
	                      } else {
	                        index = restore;
	                        str += ch;
	                      }
	                      break;
	                    case 'n':
	                      str += '\n';
	                      break;
	                    case 'r':
	                      str += '\r';
	                      break;
	                    case 't':
	                      str += '\t';
	                      break;
	                    case 'b':
	                      str += '\b';
	                      break;
	                    case 'f':
	                      str += '\f';
	                      break;
	                    case 'v':
	                      str += '\x0B';
	                      break;
	                    default:
	                      if (isOctalDigit(ch)) {
	                        code = '01234567'.indexOf(ch);

	                        // \0 is not octal escape sequence
	                        if (code !== 0) {
	                          octal = true;
	                        }
	                        if (index < length && isOctalDigit(source[index])) {
	                          octal = true;
	                          code = code * 8 + '01234567'.indexOf(source[index++]);

	                          // 3 digits are only allowed when string starts
	                          // with 0, 1, 2, 3
	                          if ('0123'.indexOf(ch) >= 0 && index < length && isOctalDigit(source[index])) {
	                            code = code * 8 + '01234567'.indexOf(source[index++]);
	                          }
	                        }
	                        str += String.fromCharCode(code);
	                      } else {
	                        str += ch;
	                      }
	                      break;
	                  }
	                } else {
	                  ++lineNumber;
	                  if (ch === '\r' && source[index] === '\n') {
	                    ++index;
	                  }
	                  lineStart = index;
	                }
	              } else if (isLineTerminator(ch.charCodeAt(0))) {
	                break;
	              } else {
	                str += ch;
	              }
	            }
	            if (quote !== '') {
	              throwError({}, Messages.UnexpectedToken, 'ILLEGAL');
	            }
	            return {
	              type: Token.StringLiteral,
	              value: str,
	              octal: octal,
	              startLineNumber: startLineNumber,
	              startLineStart: startLineStart,
	              lineNumber: lineNumber,
	              lineStart: lineStart,
	              start: start,
	              end: index
	            };
	          }
	          function testRegExp(pattern, flags) {
	            var value;
	            try {
	              value = new RegExp(pattern, flags);
	            } catch (e) {
	              throwError({}, Messages.InvalidRegExp);
	            }
	            return value;
	          }
	          function scanRegExpBody() {
	            var ch, str, classMarker, terminated, body;
	            ch = source[index];
	            assert(ch === '/', 'Regular expression literal must start with a slash');
	            str = source[index++];
	            classMarker = false;
	            terminated = false;
	            while (index < length) {
	              ch = source[index++];
	              str += ch;
	              if (ch === '\\') {
	                ch = source[index++];
	                // ECMA-262 7.8.5
	                if (isLineTerminator(ch.charCodeAt(0))) {
	                  throwError({}, Messages.UnterminatedRegExp);
	                }
	                str += ch;
	              } else if (isLineTerminator(ch.charCodeAt(0))) {
	                throwError({}, Messages.UnterminatedRegExp);
	              } else if (classMarker) {
	                if (ch === ']') {
	                  classMarker = false;
	                }
	              } else {
	                if (ch === '/') {
	                  terminated = true;
	                  break;
	                } else if (ch === '[') {
	                  classMarker = true;
	                }
	              }
	            }
	            if (!terminated) {
	              throwError({}, Messages.UnterminatedRegExp);
	            }

	            // Exclude leading and trailing slash.
	            body = str.substr(1, str.length - 2);
	            return {
	              value: body,
	              literal: str
	            };
	          }
	          function scanRegExpFlags() {
	            var ch, str, flags, restore;
	            str = '';
	            flags = '';
	            while (index < length) {
	              ch = source[index];
	              if (!isIdentifierPart(ch.charCodeAt(0))) {
	                break;
	              }
	              ++index;
	              if (ch === '\\' && index < length) {
	                ch = source[index];
	                if (ch === 'u') {
	                  ++index;
	                  restore = index;
	                  ch = scanHexEscape('u');
	                  if (ch) {
	                    flags += ch;
	                    for (str += '\\u'; restore < index; ++restore) {
	                      str += source[restore];
	                    }
	                  } else {
	                    index = restore;
	                    flags += 'u';
	                    str += '\\u';
	                  }
	                  throwErrorTolerant({}, Messages.UnexpectedToken, 'ILLEGAL');
	                } else {
	                  str += '\\';
	                  throwErrorTolerant({}, Messages.UnexpectedToken, 'ILLEGAL');
	                }
	              } else {
	                flags += ch;
	                str += ch;
	              }
	            }
	            return {
	              value: flags,
	              literal: str
	            };
	          }
	          function scanRegExp() {
	            var start, body, flags, value;
	            lookahead = null;
	            skipComment();
	            start = index;
	            body = scanRegExpBody();
	            flags = scanRegExpFlags();
	            value = testRegExp(body.value, flags.value);
	            if (extra.tokenize) {
	              return {
	                type: Token.RegularExpression,
	                value: value,
	                lineNumber: lineNumber,
	                lineStart: lineStart,
	                start: start,
	                end: index
	              };
	            }
	            return {
	              literal: body.literal + flags.literal,
	              value: value,
	              start: start,
	              end: index
	            };
	          }
	          function collectRegex() {
	            var pos, loc, regex, token;
	            skipComment();
	            pos = index;
	            loc = {
	              start: {
	                line: lineNumber,
	                column: index - lineStart
	              }
	            };
	            regex = scanRegExp();
	            loc.end = {
	              line: lineNumber,
	              column: index - lineStart
	            };

	            /* istanbul ignore next */
	            if (!extra.tokenize) {
	              // Pop the previous token, which is likely '/' or '/='
	              if (extra.tokens.length > 0) {
	                token = extra.tokens[extra.tokens.length - 1];
	                if (token.range[0] === pos && token.type === 'Punctuator') {
	                  if (token.value === '/' || token.value === '/=') {
	                    extra.tokens.pop();
	                  }
	                }
	              }
	              extra.tokens.push({
	                type: 'RegularExpression',
	                value: regex.literal,
	                range: [pos, index],
	                loc: loc
	              });
	            }
	            return regex;
	          }
	          function isIdentifierName(token) {
	            return token.type === Token.Identifier || token.type === Token.Keyword || token.type === Token.BooleanLiteral || token.type === Token.NullLiteral;
	          }
	          function advanceSlash() {
	            var prevToken, checkToken;
	            // Using the following algorithm:
	            // https://github.com/mozilla/sweet.js/wiki/design
	            prevToken = extra.tokens[extra.tokens.length - 1];
	            if (!prevToken) {
	              // Nothing before that: it cannot be a division.
	              return collectRegex();
	            }
	            if (prevToken.type === 'Punctuator') {
	              if (prevToken.value === ']') {
	                return scanPunctuator();
	              }
	              if (prevToken.value === ')') {
	                checkToken = extra.tokens[extra.openParenToken - 1];
	                if (checkToken && checkToken.type === 'Keyword' && (checkToken.value === 'if' || checkToken.value === 'while' || checkToken.value === 'for' || checkToken.value === 'with')) {
	                  return collectRegex();
	                }
	                return scanPunctuator();
	              }
	              if (prevToken.value === '}') {
	                // Dividing a function by anything makes little sense,
	                // but we have to check for that.
	                if (extra.tokens[extra.openCurlyToken - 3] && extra.tokens[extra.openCurlyToken - 3].type === 'Keyword') {
	                  // Anonymous function.
	                  checkToken = extra.tokens[extra.openCurlyToken - 4];
	                  if (!checkToken) {
	                    return scanPunctuator();
	                  }
	                } else if (extra.tokens[extra.openCurlyToken - 4] && extra.tokens[extra.openCurlyToken - 4].type === 'Keyword') {
	                  // Named function.
	                  checkToken = extra.tokens[extra.openCurlyToken - 5];
	                  if (!checkToken) {
	                    return collectRegex();
	                  }
	                } else {
	                  return scanPunctuator();
	                }
	                // checkToken determines whether the function is
	                // a declaration or an expression.
	                if (FnExprTokens.indexOf(checkToken.value) >= 0) {
	                  // It is an expression.
	                  return scanPunctuator();
	                }
	                // It is a declaration.
	                return collectRegex();
	              }
	              return collectRegex();
	            }
	            if (prevToken.type === 'Keyword') {
	              return collectRegex();
	            }
	            return scanPunctuator();
	          }
	          function advance() {
	            var ch;
	            skipComment();
	            if (index >= length) {
	              return {
	                type: Token.EOF,
	                lineNumber: lineNumber,
	                lineStart: lineStart,
	                start: index,
	                end: index
	              };
	            }
	            ch = source.charCodeAt(index);
	            if (isIdentifierStart(ch)) {
	              return scanIdentifier();
	            }

	            // Very common: ( and ) and ;
	            if (ch === 0x28 || ch === 0x29 || ch === 0x3B) {
	              return scanPunctuator();
	            }

	            // String literal starts with single quote (U+0027) or double quote (U+0022).
	            if (ch === 0x27 || ch === 0x22) {
	              return scanStringLiteral();
	            }

	            // Dot (.) U+002E can also start a floating-point number, hence the need
	            // to check the next character.
	            if (ch === 0x2E) {
	              if (isDecimalDigit(source.charCodeAt(index + 1))) {
	                return scanNumericLiteral();
	              }
	              return scanPunctuator();
	            }
	            if (isDecimalDigit(ch)) {
	              return scanNumericLiteral();
	            }

	            // Slash (/) U+002F can also start a regex.
	            if (extra.tokenize && ch === 0x2F) {
	              return advanceSlash();
	            }
	            return scanPunctuator();
	          }
	          function collectToken() {
	            var loc, token, value;
	            skipComment();
	            loc = {
	              start: {
	                line: lineNumber,
	                column: index - lineStart
	              }
	            };
	            token = advance();
	            loc.end = {
	              line: lineNumber,
	              column: index - lineStart
	            };
	            if (token.type !== Token.EOF) {
	              value = source.slice(token.start, token.end);
	              extra.tokens.push({
	                type: TokenName[token.type],
	                value: value,
	                range: [token.start, token.end],
	                loc: loc
	              });
	            }
	            return token;
	          }
	          function lex() {
	            var token;
	            token = lookahead;
	            index = token.end;
	            lineNumber = token.lineNumber;
	            lineStart = token.lineStart;
	            lookahead = typeof extra.tokens !== 'undefined' ? collectToken() : advance();
	            index = token.end;
	            lineNumber = token.lineNumber;
	            lineStart = token.lineStart;
	            return token;
	          }
	          function peek() {
	            var pos, line, start;
	            pos = index;
	            line = lineNumber;
	            start = lineStart;
	            lookahead = typeof extra.tokens !== 'undefined' ? collectToken() : advance();
	            index = pos;
	            lineNumber = line;
	            lineStart = start;
	          }
	          function Position(line, column) {
	            this.line = line;
	            this.column = column;
	          }
	          function SourceLocation(startLine, startColumn, line, column) {
	            this.start = new Position(startLine, startColumn);
	            this.end = new Position(line, column);
	          }
	          SyntaxTreeDelegate = {
	            name: 'SyntaxTree',
	            processComment: function (node) {
	              var lastChild, trailingComments;
	              if (node.type === Syntax.Program) {
	                if (node.body.length > 0) {
	                  return;
	                }
	              }
	              if (extra.trailingComments.length > 0) {
	                if (extra.trailingComments[0].range[0] >= node.range[1]) {
	                  trailingComments = extra.trailingComments;
	                  extra.trailingComments = [];
	                } else {
	                  extra.trailingComments.length = 0;
	                }
	              } else {
	                if (extra.bottomRightStack.length > 0 && extra.bottomRightStack[extra.bottomRightStack.length - 1].trailingComments && extra.bottomRightStack[extra.bottomRightStack.length - 1].trailingComments[0].range[0] >= node.range[1]) {
	                  trailingComments = extra.bottomRightStack[extra.bottomRightStack.length - 1].trailingComments;
	                  delete extra.bottomRightStack[extra.bottomRightStack.length - 1].trailingComments;
	                }
	              }

	              // Eating the stack.
	              while (extra.bottomRightStack.length > 0 && extra.bottomRightStack[extra.bottomRightStack.length - 1].range[0] >= node.range[0]) {
	                lastChild = extra.bottomRightStack.pop();
	              }
	              if (lastChild) {
	                if (lastChild.leadingComments && lastChild.leadingComments[lastChild.leadingComments.length - 1].range[1] <= node.range[0]) {
	                  node.leadingComments = lastChild.leadingComments;
	                  delete lastChild.leadingComments;
	                }
	              } else if (extra.leadingComments.length > 0 && extra.leadingComments[extra.leadingComments.length - 1].range[1] <= node.range[0]) {
	                node.leadingComments = extra.leadingComments;
	                extra.leadingComments = [];
	              }
	              if (trailingComments) {
	                node.trailingComments = trailingComments;
	              }
	              extra.bottomRightStack.push(node);
	            },
	            markEnd: function (node, startToken) {
	              if (extra.range) {
	                node.range = [startToken.start, index];
	              }
	              if (extra.loc) {
	                node.loc = new SourceLocation(startToken.startLineNumber === undefined ? startToken.lineNumber : startToken.startLineNumber, startToken.start - (startToken.startLineStart === undefined ? startToken.lineStart : startToken.startLineStart), lineNumber, index - lineStart);
	                this.postProcess(node);
	              }
	              if (extra.attachComment) {
	                this.processComment(node);
	              }
	              return node;
	            },
	            postProcess: function (node) {
	              if (extra.source) {
	                node.loc.source = extra.source;
	              }
	              return node;
	            },
	            createArrayExpression: function (elements) {
	              return {
	                type: Syntax.ArrayExpression,
	                elements: elements
	              };
	            },
	            createAssignmentExpression: function (operator, left, right) {
	              return {
	                type: Syntax.AssignmentExpression,
	                operator: operator,
	                left: left,
	                right: right
	              };
	            },
	            createBinaryExpression: function (operator, left, right) {
	              var type = operator === '||' || operator === '&&' ? Syntax.LogicalExpression : Syntax.BinaryExpression;
	              return {
	                type: type,
	                operator: operator,
	                left: left,
	                right: right
	              };
	            },
	            createBlockStatement: function (body) {
	              return {
	                type: Syntax.BlockStatement,
	                body: body
	              };
	            },
	            createBreakStatement: function (label) {
	              return {
	                type: Syntax.BreakStatement,
	                label: label
	              };
	            },
	            createCallExpression: function (callee, args) {
	              return {
	                type: Syntax.CallExpression,
	                callee: callee,
	                'arguments': args
	              };
	            },
	            createCatchClause: function (param, body) {
	              return {
	                type: Syntax.CatchClause,
	                param: param,
	                body: body
	              };
	            },
	            createConditionalExpression: function (test, consequent, alternate) {
	              return {
	                type: Syntax.ConditionalExpression,
	                test: test,
	                consequent: consequent,
	                alternate: alternate
	              };
	            },
	            createContinueStatement: function (label) {
	              return {
	                type: Syntax.ContinueStatement,
	                label: label
	              };
	            },
	            createDebuggerStatement: function () {
	              return {
	                type: Syntax.DebuggerStatement
	              };
	            },
	            createDoWhileStatement: function (body, test) {
	              return {
	                type: Syntax.DoWhileStatement,
	                body: body,
	                test: test
	              };
	            },
	            createEmptyStatement: function () {
	              return {
	                type: Syntax.EmptyStatement
	              };
	            },
	            createExpressionStatement: function (expression) {
	              return {
	                type: Syntax.ExpressionStatement,
	                expression: expression
	              };
	            },
	            createForStatement: function (init, test, update, body) {
	              return {
	                type: Syntax.ForStatement,
	                init: init,
	                test: test,
	                update: update,
	                body: body
	              };
	            },
	            createForInStatement: function (left, right, body) {
	              return {
	                type: Syntax.ForInStatement,
	                left: left,
	                right: right,
	                body: body,
	                each: false
	              };
	            },
	            createFunctionDeclaration: function (id, params, defaults, body) {
	              return {
	                type: Syntax.FunctionDeclaration,
	                id: id,
	                params: params,
	                defaults: defaults,
	                body: body,
	                rest: null,
	                generator: false,
	                expression: false
	              };
	            },
	            createFunctionExpression: function (id, params, defaults, body) {
	              return {
	                type: Syntax.FunctionExpression,
	                id: id,
	                params: params,
	                defaults: defaults,
	                body: body,
	                rest: null,
	                generator: false,
	                expression: false
	              };
	            },
	            createIdentifier: function (name) {
	              return {
	                type: Syntax.Identifier,
	                name: name
	              };
	            },
	            createIfStatement: function (test, consequent, alternate) {
	              return {
	                type: Syntax.IfStatement,
	                test: test,
	                consequent: consequent,
	                alternate: alternate
	              };
	            },
	            createLabeledStatement: function (label, body) {
	              return {
	                type: Syntax.LabeledStatement,
	                label: label,
	                body: body
	              };
	            },
	            createLiteral: function (token) {
	              return {
	                type: Syntax.Literal,
	                value: token.value,
	                raw: source.slice(token.start, token.end)
	              };
	            },
	            createMemberExpression: function (accessor, object, property) {
	              return {
	                type: Syntax.MemberExpression,
	                computed: accessor === '[',
	                object: object,
	                property: property
	              };
	            },
	            createNewExpression: function (callee, args) {
	              return {
	                type: Syntax.NewExpression,
	                callee: callee,
	                'arguments': args
	              };
	            },
	            createObjectExpression: function (properties) {
	              return {
	                type: Syntax.ObjectExpression,
	                properties: properties
	              };
	            },
	            createPostfixExpression: function (operator, argument) {
	              return {
	                type: Syntax.UpdateExpression,
	                operator: operator,
	                argument: argument,
	                prefix: false
	              };
	            },
	            createProgram: function (body) {
	              return {
	                type: Syntax.Program,
	                body: body
	              };
	            },
	            createProperty: function (kind, key, value) {
	              return {
	                type: Syntax.Property,
	                key: key,
	                value: value,
	                kind: kind
	              };
	            },
	            createReturnStatement: function (argument) {
	              return {
	                type: Syntax.ReturnStatement,
	                argument: argument
	              };
	            },
	            createSequenceExpression: function (expressions) {
	              return {
	                type: Syntax.SequenceExpression,
	                expressions: expressions
	              };
	            },
	            createSwitchCase: function (test, consequent) {
	              return {
	                type: Syntax.SwitchCase,
	                test: test,
	                consequent: consequent
	              };
	            },
	            createSwitchStatement: function (discriminant, cases) {
	              return {
	                type: Syntax.SwitchStatement,
	                discriminant: discriminant,
	                cases: cases
	              };
	            },
	            createThisExpression: function () {
	              return {
	                type: Syntax.ThisExpression
	              };
	            },
	            createThrowStatement: function (argument) {
	              return {
	                type: Syntax.ThrowStatement,
	                argument: argument
	              };
	            },
	            createTryStatement: function (block, guardedHandlers, handlers, finalizer) {
	              return {
	                type: Syntax.TryStatement,
	                block: block,
	                guardedHandlers: guardedHandlers,
	                handlers: handlers,
	                finalizer: finalizer
	              };
	            },
	            createUnaryExpression: function (operator, argument) {
	              if (operator === '++' || operator === '--') {
	                return {
	                  type: Syntax.UpdateExpression,
	                  operator: operator,
	                  argument: argument,
	                  prefix: true
	                };
	              }
	              return {
	                type: Syntax.UnaryExpression,
	                operator: operator,
	                argument: argument,
	                prefix: true
	              };
	            },
	            createVariableDeclaration: function (declarations, kind) {
	              return {
	                type: Syntax.VariableDeclaration,
	                declarations: declarations,
	                kind: kind
	              };
	            },
	            createVariableDeclarator: function (id, init) {
	              return {
	                type: Syntax.VariableDeclarator,
	                id: id,
	                init: init
	              };
	            },
	            createWhileStatement: function (test, body) {
	              return {
	                type: Syntax.WhileStatement,
	                test: test,
	                body: body
	              };
	            },
	            createWithStatement: function (object, body) {
	              return {
	                type: Syntax.WithStatement,
	                object: object,
	                body: body
	              };
	            }
	          };

	          // Return true if there is a line terminator before the next token.

	          function peekLineTerminator() {
	            var pos, line, start, found;
	            pos = index;
	            line = lineNumber;
	            start = lineStart;
	            skipComment();
	            found = lineNumber !== line;
	            index = pos;
	            lineNumber = line;
	            lineStart = start;
	            return found;
	          }

	          // Throw an exception

	          function throwError(token, messageFormat) {
	            var error,
	              args = Array.prototype.slice.call(arguments, 2),
	              msg = messageFormat.replace(/%(\d)/g, function (whole, index) {
	                assert(index < args.length, 'Message reference must be in range');
	                return args[index];
	              });
	            if (typeof token.lineNumber === 'number') {
	              error = new Error('Line ' + token.lineNumber + ': ' + msg);
	              error.index = token.start;
	              error.lineNumber = token.lineNumber;
	              error.column = token.start - lineStart + 1;
	            } else {
	              error = new Error('Line ' + lineNumber + ': ' + msg);
	              error.index = index;
	              error.lineNumber = lineNumber;
	              error.column = index - lineStart + 1;
	            }
	            error.description = msg;
	            throw error;
	          }
	          function throwErrorTolerant() {
	            try {
	              throwError.apply(null, arguments);
	            } catch (e) {
	              if (extra.errors) {
	                extra.errors.push(e);
	              } else {
	                throw e;
	              }
	            }
	          }

	          // Throw an exception because of the token.

	          function throwUnexpected(token) {
	            if (token.type === Token.EOF) {
	              throwError(token, Messages.UnexpectedEOS);
	            }
	            if (token.type === Token.NumericLiteral) {
	              throwError(token, Messages.UnexpectedNumber);
	            }
	            if (token.type === Token.StringLiteral) {
	              throwError(token, Messages.UnexpectedString);
	            }
	            if (token.type === Token.Identifier) {
	              throwError(token, Messages.UnexpectedIdentifier);
	            }
	            if (token.type === Token.Keyword) {
	              if (isFutureReservedWord(token.value)) {
	                throwError(token, Messages.UnexpectedReserved);
	              } else if (strict && isStrictModeReservedWord(token.value)) {
	                throwErrorTolerant(token, Messages.StrictReservedWord);
	                return;
	              }
	              throwError(token, Messages.UnexpectedToken, token.value);
	            }

	            // BooleanLiteral, NullLiteral, or Punctuator.
	            throwError(token, Messages.UnexpectedToken, token.value);
	          }

	          // Expect the next token to match the specified punctuator.
	          // If not, an exception will be thrown.

	          function expect(value) {
	            var token = lex();
	            if (token.type !== Token.Punctuator || token.value !== value) {
	              throwUnexpected(token);
	            }
	          }

	          // Expect the next token to match the specified keyword.
	          // If not, an exception will be thrown.

	          function expectKeyword(keyword) {
	            var token = lex();
	            if (token.type !== Token.Keyword || token.value !== keyword) {
	              throwUnexpected(token);
	            }
	          }

	          // Return true if the next token matches the specified punctuator.

	          function match(value) {
	            return lookahead.type === Token.Punctuator && lookahead.value === value;
	          }

	          // Return true if the next token matches the specified keyword

	          function matchKeyword(keyword) {
	            return lookahead.type === Token.Keyword && lookahead.value === keyword;
	          }

	          // Return true if the next token is an assignment operator

	          function matchAssign() {
	            var op;
	            if (lookahead.type !== Token.Punctuator) {
	              return false;
	            }
	            op = lookahead.value;
	            return op === '=' || op === '*=' || op === '/=' || op === '%=' || op === '+=' || op === '-=' || op === '<<=' || op === '>>=' || op === '>>>=' || op === '&=' || op === '^=' || op === '|=';
	          }
	          function consumeSemicolon() {
	            var line;

	            // Catch the very common case first: immediately a semicolon (U+003B).
	            if (source.charCodeAt(index) === 0x3B || match(';')) {
	              lex();
	              return;
	            }
	            line = lineNumber;
	            skipComment();
	            if (lineNumber !== line) {
	              return;
	            }
	            if (lookahead.type !== Token.EOF && !match('}')) {
	              throwUnexpected(lookahead);
	            }
	          }

	          // Return true if provided expression is LeftHandSideExpression

	          function isLeftHandSide(expr) {
	            return expr.type === Syntax.Identifier || expr.type === Syntax.MemberExpression;
	          }

	          // 11.1.4 Array Initialiser

	          function parseArrayInitialiser() {
	            var elements = [],
	              startToken;
	            startToken = lookahead;
	            expect('[');
	            while (!match(']')) {
	              if (match(',')) {
	                lex();
	                elements.push(null);
	              } else {
	                elements.push(parseAssignmentExpression());
	                if (!match(']')) {
	                  expect(',');
	                }
	              }
	            }
	            lex();
	            return delegate.markEnd(delegate.createArrayExpression(elements), startToken);
	          }

	          // 11.1.5 Object Initialiser

	          function parsePropertyFunction(param, first) {
	            var previousStrict, body, startToken;
	            previousStrict = strict;
	            startToken = lookahead;
	            body = parseFunctionSourceElements();
	            if (first && strict && isRestrictedWord(param[0].name)) {
	              throwErrorTolerant(first, Messages.StrictParamName);
	            }
	            strict = previousStrict;
	            return delegate.markEnd(delegate.createFunctionExpression(null, param, [], body), startToken);
	          }
	          function parseObjectPropertyKey() {
	            var token, startToken;
	            startToken = lookahead;
	            token = lex();

	            // Note: This function is called only from parseObjectProperty(), where
	            // EOF and Punctuator tokens are already filtered out.

	            if (token.type === Token.StringLiteral || token.type === Token.NumericLiteral) {
	              if (strict && token.octal) {
	                throwErrorTolerant(token, Messages.StrictOctalLiteral);
	              }
	              return delegate.markEnd(delegate.createLiteral(token), startToken);
	            }
	            return delegate.markEnd(delegate.createIdentifier(token.value), startToken);
	          }
	          function parseObjectProperty() {
	            var token, key, id, value, param, startToken;
	            token = lookahead;
	            startToken = lookahead;
	            if (token.type === Token.Identifier) {
	              id = parseObjectPropertyKey();

	              // Property Assignment: Getter and Setter.

	              if (token.value === 'get' && !match(':')) {
	                key = parseObjectPropertyKey();
	                expect('(');
	                expect(')');
	                value = parsePropertyFunction([]);
	                return delegate.markEnd(delegate.createProperty('get', key, value), startToken);
	              }
	              if (token.value === 'set' && !match(':')) {
	                key = parseObjectPropertyKey();
	                expect('(');
	                token = lookahead;
	                if (token.type !== Token.Identifier) {
	                  expect(')');
	                  throwErrorTolerant(token, Messages.UnexpectedToken, token.value);
	                  value = parsePropertyFunction([]);
	                } else {
	                  param = [parseVariableIdentifier()];
	                  expect(')');
	                  value = parsePropertyFunction(param, token);
	                }
	                return delegate.markEnd(delegate.createProperty('set', key, value), startToken);
	              }
	              expect(':');
	              value = parseAssignmentExpression();
	              return delegate.markEnd(delegate.createProperty('init', id, value), startToken);
	            }
	            if (token.type === Token.EOF || token.type === Token.Punctuator) {
	              throwUnexpected(token);
	            } else {
	              key = parseObjectPropertyKey();
	              expect(':');
	              value = parseAssignmentExpression();
	              return delegate.markEnd(delegate.createProperty('init', key, value), startToken);
	            }
	          }
	          function parseObjectInitialiser() {
	            var properties = [],
	              property,
	              name,
	              key,
	              kind,
	              map = {},
	              toString = String,
	              startToken;
	            startToken = lookahead;
	            expect('{');
	            while (!match('}')) {
	              property = parseObjectProperty();
	              if (property.key.type === Syntax.Identifier) {
	                name = property.key.name;
	              } else {
	                name = toString(property.key.value);
	              }
	              kind = property.kind === 'init' ? PropertyKind.Data : property.kind === 'get' ? PropertyKind.Get : PropertyKind.Set;
	              key = '$' + name;
	              if (Object.prototype.hasOwnProperty.call(map, key)) {
	                if (map[key] === PropertyKind.Data) {
	                  if (strict && kind === PropertyKind.Data) {
	                    throwErrorTolerant({}, Messages.StrictDuplicateProperty);
	                  } else if (kind !== PropertyKind.Data) {
	                    throwErrorTolerant({}, Messages.AccessorDataProperty);
	                  }
	                } else {
	                  if (kind === PropertyKind.Data) {
	                    throwErrorTolerant({}, Messages.AccessorDataProperty);
	                  } else if (map[key] & kind) {
	                    throwErrorTolerant({}, Messages.AccessorGetSet);
	                  }
	                }
	                map[key] |= kind;
	              } else {
	                map[key] = kind;
	              }
	              properties.push(property);
	              if (!match('}')) {
	                expect(',');
	              }
	            }
	            expect('}');
	            return delegate.markEnd(delegate.createObjectExpression(properties), startToken);
	          }

	          // 11.1.6 The Grouping Operator

	          function parseGroupExpression() {
	            var expr;
	            expect('(');
	            expr = parseExpression();
	            expect(')');
	            return expr;
	          }

	          // 11.1 Primary Expressions

	          function parsePrimaryExpression() {
	            var type, token, expr, startToken;
	            if (match('(')) {
	              return parseGroupExpression();
	            }
	            if (match('[')) {
	              return parseArrayInitialiser();
	            }
	            if (match('{')) {
	              return parseObjectInitialiser();
	            }
	            type = lookahead.type;
	            startToken = lookahead;
	            if (type === Token.Identifier) {
	              expr = delegate.createIdentifier(lex().value);
	            } else if (type === Token.StringLiteral || type === Token.NumericLiteral) {
	              if (strict && lookahead.octal) {
	                throwErrorTolerant(lookahead, Messages.StrictOctalLiteral);
	              }
	              expr = delegate.createLiteral(lex());
	            } else if (type === Token.Keyword) {
	              if (matchKeyword('function')) {
	                return parseFunctionExpression();
	              }
	              if (matchKeyword('this')) {
	                lex();
	                expr = delegate.createThisExpression();
	              } else {
	                throwUnexpected(lex());
	              }
	            } else if (type === Token.BooleanLiteral) {
	              token = lex();
	              token.value = token.value === 'true';
	              expr = delegate.createLiteral(token);
	            } else if (type === Token.NullLiteral) {
	              token = lex();
	              token.value = null;
	              expr = delegate.createLiteral(token);
	            } else if (match('/') || match('/=')) {
	              if (typeof extra.tokens !== 'undefined') {
	                expr = delegate.createLiteral(collectRegex());
	              } else {
	                expr = delegate.createLiteral(scanRegExp());
	              }
	              peek();
	            } else {
	              throwUnexpected(lex());
	            }
	            return delegate.markEnd(expr, startToken);
	          }

	          // 11.2 Left-Hand-Side Expressions

	          function parseArguments() {
	            var args = [];
	            expect('(');
	            if (!match(')')) {
	              while (index < length) {
	                args.push(parseAssignmentExpression());
	                if (match(')')) {
	                  break;
	                }
	                expect(',');
	              }
	            }
	            expect(')');
	            return args;
	          }
	          function parseNonComputedProperty() {
	            var token, startToken;
	            startToken = lookahead;
	            token = lex();
	            if (!isIdentifierName(token)) {
	              throwUnexpected(token);
	            }
	            return delegate.markEnd(delegate.createIdentifier(token.value), startToken);
	          }
	          function parseNonComputedMember() {
	            expect('.');
	            return parseNonComputedProperty();
	          }
	          function parseComputedMember() {
	            var expr;
	            expect('[');
	            expr = parseExpression();
	            expect(']');
	            return expr;
	          }
	          function parseNewExpression() {
	            var callee, args, startToken;
	            startToken = lookahead;
	            expectKeyword('new');
	            callee = parseLeftHandSideExpression();
	            args = match('(') ? parseArguments() : [];
	            return delegate.markEnd(delegate.createNewExpression(callee, args), startToken);
	          }
	          function parseLeftHandSideExpressionAllowCall() {
	            var previousAllowIn, expr, args, property, startToken;
	            startToken = lookahead;
	            previousAllowIn = state.allowIn;
	            state.allowIn = true;
	            expr = matchKeyword('new') ? parseNewExpression() : parsePrimaryExpression();
	            state.allowIn = previousAllowIn;
	            for (;;) {
	              if (match('.')) {
	                property = parseNonComputedMember();
	                expr = delegate.createMemberExpression('.', expr, property);
	              } else if (match('(')) {
	                args = parseArguments();
	                expr = delegate.createCallExpression(expr, args);
	              } else if (match('[')) {
	                property = parseComputedMember();
	                expr = delegate.createMemberExpression('[', expr, property);
	              } else {
	                break;
	              }
	              delegate.markEnd(expr, startToken);
	            }
	            return expr;
	          }
	          function parseLeftHandSideExpression() {
	            var previousAllowIn, expr, property, startToken;
	            startToken = lookahead;
	            previousAllowIn = state.allowIn;
	            expr = matchKeyword('new') ? parseNewExpression() : parsePrimaryExpression();
	            state.allowIn = previousAllowIn;
	            while (match('.') || match('[')) {
	              if (match('[')) {
	                property = parseComputedMember();
	                expr = delegate.createMemberExpression('[', expr, property);
	              } else {
	                property = parseNonComputedMember();
	                expr = delegate.createMemberExpression('.', expr, property);
	              }
	              delegate.markEnd(expr, startToken);
	            }
	            return expr;
	          }

	          // 11.3 Postfix Expressions

	          function parsePostfixExpression() {
	            var expr,
	              token,
	              startToken = lookahead;
	            expr = parseLeftHandSideExpressionAllowCall();
	            if (lookahead.type === Token.Punctuator) {
	              if ((match('++') || match('--')) && !peekLineTerminator()) {
	                // 11.3.1, 11.3.2
	                if (strict && expr.type === Syntax.Identifier && isRestrictedWord(expr.name)) {
	                  throwErrorTolerant({}, Messages.StrictLHSPostfix);
	                }
	                if (!isLeftHandSide(expr)) {
	                  throwErrorTolerant({}, Messages.InvalidLHSInAssignment);
	                }
	                token = lex();
	                expr = delegate.markEnd(delegate.createPostfixExpression(token.value, expr), startToken);
	              }
	            }
	            return expr;
	          }

	          // 11.4 Unary Operators

	          function parseUnaryExpression() {
	            var token, expr, startToken;
	            if (lookahead.type !== Token.Punctuator && lookahead.type !== Token.Keyword) {
	              expr = parsePostfixExpression();
	            } else if (match('++') || match('--')) {
	              startToken = lookahead;
	              token = lex();
	              expr = parseUnaryExpression();
	              // 11.4.4, 11.4.5
	              if (strict && expr.type === Syntax.Identifier && isRestrictedWord(expr.name)) {
	                throwErrorTolerant({}, Messages.StrictLHSPrefix);
	              }
	              if (!isLeftHandSide(expr)) {
	                throwErrorTolerant({}, Messages.InvalidLHSInAssignment);
	              }
	              expr = delegate.createUnaryExpression(token.value, expr);
	              expr = delegate.markEnd(expr, startToken);
	            } else if (match('+') || match('-') || match('~') || match('!')) {
	              startToken = lookahead;
	              token = lex();
	              expr = parseUnaryExpression();
	              expr = delegate.createUnaryExpression(token.value, expr);
	              expr = delegate.markEnd(expr, startToken);
	            } else if (matchKeyword('delete') || matchKeyword('void') || matchKeyword('typeof')) {
	              startToken = lookahead;
	              token = lex();
	              expr = parseUnaryExpression();
	              expr = delegate.createUnaryExpression(token.value, expr);
	              expr = delegate.markEnd(expr, startToken);
	              if (strict && expr.operator === 'delete' && expr.argument.type === Syntax.Identifier) {
	                throwErrorTolerant({}, Messages.StrictDelete);
	              }
	            } else {
	              expr = parsePostfixExpression();
	            }
	            return expr;
	          }
	          function binaryPrecedence(token, allowIn) {
	            var prec = 0;
	            if (token.type !== Token.Punctuator && token.type !== Token.Keyword) {
	              return 0;
	            }
	            switch (token.value) {
	              case '||':
	                prec = 1;
	                break;
	              case '&&':
	                prec = 2;
	                break;
	              case '|':
	                prec = 3;
	                break;
	              case '^':
	                prec = 4;
	                break;
	              case '&':
	                prec = 5;
	                break;
	              case '==':
	              case '!=':
	              case '===':
	              case '!==':
	                prec = 6;
	                break;
	              case '<':
	              case '>':
	              case '<=':
	              case '>=':
	              case 'instanceof':
	                prec = 7;
	                break;
	              case 'in':
	                prec = allowIn ? 7 : 0;
	                break;
	              case '<<':
	              case '>>':
	              case '>>>':
	                prec = 8;
	                break;
	              case '+':
	              case '-':
	                prec = 9;
	                break;
	              case '*':
	              case '/':
	              case '%':
	                prec = 11;
	                break;
	            }
	            return prec;
	          }

	          // 11.5 Multiplicative Operators
	          // 11.6 Additive Operators
	          // 11.7 Bitwise Shift Operators
	          // 11.8 Relational Operators
	          // 11.9 Equality Operators
	          // 11.10 Binary Bitwise Operators
	          // 11.11 Binary Logical Operators

	          function parseBinaryExpression() {
	            var marker, markers, expr, token, prec, stack, right, operator, left, i;
	            marker = lookahead;
	            left = parseUnaryExpression();
	            token = lookahead;
	            prec = binaryPrecedence(token, state.allowIn);
	            if (prec === 0) {
	              return left;
	            }
	            token.prec = prec;
	            lex();
	            markers = [marker, lookahead];
	            right = parseUnaryExpression();
	            stack = [left, token, right];
	            while ((prec = binaryPrecedence(lookahead, state.allowIn)) > 0) {
	              // Reduce: make a binary expression from the three topmost entries.
	              while (stack.length > 2 && prec <= stack[stack.length - 2].prec) {
	                right = stack.pop();
	                operator = stack.pop().value;
	                left = stack.pop();
	                expr = delegate.createBinaryExpression(operator, left, right);
	                markers.pop();
	                marker = markers[markers.length - 1];
	                delegate.markEnd(expr, marker);
	                stack.push(expr);
	              }

	              // Shift.
	              token = lex();
	              token.prec = prec;
	              stack.push(token);
	              markers.push(lookahead);
	              expr = parseUnaryExpression();
	              stack.push(expr);
	            }

	            // Final reduce to clean-up the stack.
	            i = stack.length - 1;
	            expr = stack[i];
	            markers.pop();
	            while (i > 1) {
	              expr = delegate.createBinaryExpression(stack[i - 1].value, stack[i - 2], expr);
	              i -= 2;
	              marker = markers.pop();
	              delegate.markEnd(expr, marker);
	            }
	            return expr;
	          }

	          // 11.12 Conditional Operator

	          function parseConditionalExpression() {
	            var expr, previousAllowIn, consequent, alternate, startToken;
	            startToken = lookahead;
	            expr = parseBinaryExpression();
	            if (match('?')) {
	              lex();
	              previousAllowIn = state.allowIn;
	              state.allowIn = true;
	              consequent = parseAssignmentExpression();
	              state.allowIn = previousAllowIn;
	              expect(':');
	              alternate = parseAssignmentExpression();
	              expr = delegate.createConditionalExpression(expr, consequent, alternate);
	              delegate.markEnd(expr, startToken);
	            }
	            return expr;
	          }

	          // 11.13 Assignment Operators

	          function parseAssignmentExpression() {
	            var token, left, right, node, startToken;
	            token = lookahead;
	            startToken = lookahead;
	            node = left = parseConditionalExpression();
	            if (matchAssign()) {
	              // LeftHandSideExpression
	              if (!isLeftHandSide(left)) {
	                throwErrorTolerant({}, Messages.InvalidLHSInAssignment);
	              }

	              // 11.13.1
	              if (strict && left.type === Syntax.Identifier && isRestrictedWord(left.name)) {
	                throwErrorTolerant(token, Messages.StrictLHSAssignment);
	              }
	              token = lex();
	              right = parseAssignmentExpression();
	              node = delegate.markEnd(delegate.createAssignmentExpression(token.value, left, right), startToken);
	            }
	            return node;
	          }

	          // 11.14 Comma Operator

	          function parseExpression() {
	            var expr,
	              startToken = lookahead;
	            expr = parseAssignmentExpression();
	            if (match(',')) {
	              expr = delegate.createSequenceExpression([expr]);
	              while (index < length) {
	                if (!match(',')) {
	                  break;
	                }
	                lex();
	                expr.expressions.push(parseAssignmentExpression());
	              }
	              delegate.markEnd(expr, startToken);
	            }
	            return expr;
	          }

	          // 12.1 Block

	          function parseStatementList() {
	            var list = [],
	              statement;
	            while (index < length) {
	              if (match('}')) {
	                break;
	              }
	              statement = parseSourceElement();
	              if (typeof statement === 'undefined') {
	                break;
	              }
	              list.push(statement);
	            }
	            return list;
	          }
	          function parseBlock() {
	            var block, startToken;
	            startToken = lookahead;
	            expect('{');
	            block = parseStatementList();
	            expect('}');
	            return delegate.markEnd(delegate.createBlockStatement(block), startToken);
	          }

	          // 12.2 Variable Statement

	          function parseVariableIdentifier() {
	            var token, startToken;
	            startToken = lookahead;
	            token = lex();
	            if (token.type !== Token.Identifier) {
	              throwUnexpected(token);
	            }
	            return delegate.markEnd(delegate.createIdentifier(token.value), startToken);
	          }
	          function parseVariableDeclaration(kind) {
	            var init = null,
	              id,
	              startToken;
	            startToken = lookahead;
	            id = parseVariableIdentifier();

	            // 12.2.1
	            if (strict && isRestrictedWord(id.name)) {
	              throwErrorTolerant({}, Messages.StrictVarName);
	            }
	            if (kind === 'const') {
	              expect('=');
	              init = parseAssignmentExpression();
	            } else if (match('=')) {
	              lex();
	              init = parseAssignmentExpression();
	            }
	            return delegate.markEnd(delegate.createVariableDeclarator(id, init), startToken);
	          }
	          function parseVariableDeclarationList(kind) {
	            var list = [];
	            do {
	              list.push(parseVariableDeclaration(kind));
	              if (!match(',')) {
	                break;
	              }
	              lex();
	            } while (index < length);
	            return list;
	          }
	          function parseVariableStatement() {
	            var declarations;
	            expectKeyword('var');
	            declarations = parseVariableDeclarationList();
	            consumeSemicolon();
	            return delegate.createVariableDeclaration(declarations, 'var');
	          }

	          // kind may be `const` or `let`
	          // Both are experimental and not in the specification yet.
	          // see http://wiki.ecmascript.org/doku.php?id=harmony:const
	          // and http://wiki.ecmascript.org/doku.php?id=harmony:let
	          function parseConstLetDeclaration(kind) {
	            var declarations, startToken;
	            startToken = lookahead;
	            expectKeyword(kind);
	            declarations = parseVariableDeclarationList(kind);
	            consumeSemicolon();
	            return delegate.markEnd(delegate.createVariableDeclaration(declarations, kind), startToken);
	          }

	          // 12.3 Empty Statement

	          function parseEmptyStatement() {
	            expect(';');
	            return delegate.createEmptyStatement();
	          }

	          // 12.4 Expression Statement

	          function parseExpressionStatement() {
	            var expr = parseExpression();
	            consumeSemicolon();
	            return delegate.createExpressionStatement(expr);
	          }

	          // 12.5 If statement

	          function parseIfStatement() {
	            var test, consequent, alternate;
	            expectKeyword('if');
	            expect('(');
	            test = parseExpression();
	            expect(')');
	            consequent = parseStatement();
	            if (matchKeyword('else')) {
	              lex();
	              alternate = parseStatement();
	            } else {
	              alternate = null;
	            }
	            return delegate.createIfStatement(test, consequent, alternate);
	          }

	          // 12.6 Iteration Statements

	          function parseDoWhileStatement() {
	            var body, test, oldInIteration;
	            expectKeyword('do');
	            oldInIteration = state.inIteration;
	            state.inIteration = true;
	            body = parseStatement();
	            state.inIteration = oldInIteration;
	            expectKeyword('while');
	            expect('(');
	            test = parseExpression();
	            expect(')');
	            if (match(';')) {
	              lex();
	            }
	            return delegate.createDoWhileStatement(body, test);
	          }
	          function parseWhileStatement() {
	            var test, body, oldInIteration;
	            expectKeyword('while');
	            expect('(');
	            test = parseExpression();
	            expect(')');
	            oldInIteration = state.inIteration;
	            state.inIteration = true;
	            body = parseStatement();
	            state.inIteration = oldInIteration;
	            return delegate.createWhileStatement(test, body);
	          }
	          function parseForVariableDeclaration() {
	            var token, declarations, startToken;
	            startToken = lookahead;
	            token = lex();
	            declarations = parseVariableDeclarationList();
	            return delegate.markEnd(delegate.createVariableDeclaration(declarations, token.value), startToken);
	          }
	          function parseForStatement() {
	            var init, test, update, left, right, body, oldInIteration;
	            init = test = update = null;
	            expectKeyword('for');
	            expect('(');
	            if (match(';')) {
	              lex();
	            } else {
	              if (matchKeyword('var') || matchKeyword('let')) {
	                state.allowIn = false;
	                init = parseForVariableDeclaration();
	                state.allowIn = true;
	                if (init.declarations.length === 1 && matchKeyword('in')) {
	                  lex();
	                  left = init;
	                  right = parseExpression();
	                  init = null;
	                }
	              } else {
	                state.allowIn = false;
	                init = parseExpression();
	                state.allowIn = true;
	                if (matchKeyword('in')) {
	                  // LeftHandSideExpression
	                  if (!isLeftHandSide(init)) {
	                    throwErrorTolerant({}, Messages.InvalidLHSInForIn);
	                  }
	                  lex();
	                  left = init;
	                  right = parseExpression();
	                  init = null;
	                }
	              }
	              if (typeof left === 'undefined') {
	                expect(';');
	              }
	            }
	            if (typeof left === 'undefined') {
	              if (!match(';')) {
	                test = parseExpression();
	              }
	              expect(';');
	              if (!match(')')) {
	                update = parseExpression();
	              }
	            }
	            expect(')');
	            oldInIteration = state.inIteration;
	            state.inIteration = true;
	            body = parseStatement();
	            state.inIteration = oldInIteration;
	            return typeof left === 'undefined' ? delegate.createForStatement(init, test, update, body) : delegate.createForInStatement(left, right, body);
	          }

	          // 12.7 The continue statement

	          function parseContinueStatement() {
	            var label = null,
	              key;
	            expectKeyword('continue');

	            // Optimize the most common form: 'continue;'.
	            if (source.charCodeAt(index) === 0x3B) {
	              lex();
	              if (!state.inIteration) {
	                throwError({}, Messages.IllegalContinue);
	              }
	              return delegate.createContinueStatement(null);
	            }
	            if (peekLineTerminator()) {
	              if (!state.inIteration) {
	                throwError({}, Messages.IllegalContinue);
	              }
	              return delegate.createContinueStatement(null);
	            }
	            if (lookahead.type === Token.Identifier) {
	              label = parseVariableIdentifier();
	              key = '$' + label.name;
	              if (!Object.prototype.hasOwnProperty.call(state.labelSet, key)) {
	                throwError({}, Messages.UnknownLabel, label.name);
	              }
	            }
	            consumeSemicolon();
	            if (label === null && !state.inIteration) {
	              throwError({}, Messages.IllegalContinue);
	            }
	            return delegate.createContinueStatement(label);
	          }

	          // 12.8 The break statement

	          function parseBreakStatement() {
	            var label = null,
	              key;
	            expectKeyword('break');

	            // Catch the very common case first: immediately a semicolon (U+003B).
	            if (source.charCodeAt(index) === 0x3B) {
	              lex();
	              if (!(state.inIteration || state.inSwitch)) {
	                throwError({}, Messages.IllegalBreak);
	              }
	              return delegate.createBreakStatement(null);
	            }
	            if (peekLineTerminator()) {
	              if (!(state.inIteration || state.inSwitch)) {
	                throwError({}, Messages.IllegalBreak);
	              }
	              return delegate.createBreakStatement(null);
	            }
	            if (lookahead.type === Token.Identifier) {
	              label = parseVariableIdentifier();
	              key = '$' + label.name;
	              if (!Object.prototype.hasOwnProperty.call(state.labelSet, key)) {
	                throwError({}, Messages.UnknownLabel, label.name);
	              }
	            }
	            consumeSemicolon();
	            if (label === null && !(state.inIteration || state.inSwitch)) {
	              throwError({}, Messages.IllegalBreak);
	            }
	            return delegate.createBreakStatement(label);
	          }

	          // 12.9 The return statement

	          function parseReturnStatement() {
	            var argument = null;
	            expectKeyword('return');
	            if (!state.inFunctionBody) {
	              throwErrorTolerant({}, Messages.IllegalReturn);
	            }

	            // 'return' followed by a space and an identifier is very common.
	            if (source.charCodeAt(index) === 0x20) {
	              if (isIdentifierStart(source.charCodeAt(index + 1))) {
	                argument = parseExpression();
	                consumeSemicolon();
	                return delegate.createReturnStatement(argument);
	              }
	            }
	            if (peekLineTerminator()) {
	              return delegate.createReturnStatement(null);
	            }
	            if (!match(';')) {
	              if (!match('}') && lookahead.type !== Token.EOF) {
	                argument = parseExpression();
	              }
	            }
	            consumeSemicolon();
	            return delegate.createReturnStatement(argument);
	          }

	          // 12.10 The with statement

	          function parseWithStatement() {
	            var object, body;
	            if (strict) {
	              // TODO(ikarienator): Should we update the test cases instead?
	              skipComment();
	              throwErrorTolerant({}, Messages.StrictModeWith);
	            }
	            expectKeyword('with');
	            expect('(');
	            object = parseExpression();
	            expect(')');
	            body = parseStatement();
	            return delegate.createWithStatement(object, body);
	          }

	          // 12.10 The swith statement

	          function parseSwitchCase() {
	            var test,
	              consequent = [],
	              statement,
	              startToken;
	            startToken = lookahead;
	            if (matchKeyword('default')) {
	              lex();
	              test = null;
	            } else {
	              expectKeyword('case');
	              test = parseExpression();
	            }
	            expect(':');
	            while (index < length) {
	              if (match('}') || matchKeyword('default') || matchKeyword('case')) {
	                break;
	              }
	              statement = parseStatement();
	              consequent.push(statement);
	            }
	            return delegate.markEnd(delegate.createSwitchCase(test, consequent), startToken);
	          }
	          function parseSwitchStatement() {
	            var discriminant, cases, clause, oldInSwitch, defaultFound;
	            expectKeyword('switch');
	            expect('(');
	            discriminant = parseExpression();
	            expect(')');
	            expect('{');
	            cases = [];
	            if (match('}')) {
	              lex();
	              return delegate.createSwitchStatement(discriminant, cases);
	            }
	            oldInSwitch = state.inSwitch;
	            state.inSwitch = true;
	            defaultFound = false;
	            while (index < length) {
	              if (match('}')) {
	                break;
	              }
	              clause = parseSwitchCase();
	              if (clause.test === null) {
	                if (defaultFound) {
	                  throwError({}, Messages.MultipleDefaultsInSwitch);
	                }
	                defaultFound = true;
	              }
	              cases.push(clause);
	            }
	            state.inSwitch = oldInSwitch;
	            expect('}');
	            return delegate.createSwitchStatement(discriminant, cases);
	          }

	          // 12.13 The throw statement

	          function parseThrowStatement() {
	            var argument;
	            expectKeyword('throw');
	            if (peekLineTerminator()) {
	              throwError({}, Messages.NewlineAfterThrow);
	            }
	            argument = parseExpression();
	            consumeSemicolon();
	            return delegate.createThrowStatement(argument);
	          }

	          // 12.14 The try statement

	          function parseCatchClause() {
	            var param, body, startToken;
	            startToken = lookahead;
	            expectKeyword('catch');
	            expect('(');
	            if (match(')')) {
	              throwUnexpected(lookahead);
	            }
	            param = parseVariableIdentifier();
	            // 12.14.1
	            if (strict && isRestrictedWord(param.name)) {
	              throwErrorTolerant({}, Messages.StrictCatchVariable);
	            }
	            expect(')');
	            body = parseBlock();
	            return delegate.markEnd(delegate.createCatchClause(param, body), startToken);
	          }
	          function parseTryStatement() {
	            var block,
	              handlers = [],
	              finalizer = null;
	            expectKeyword('try');
	            block = parseBlock();
	            if (matchKeyword('catch')) {
	              handlers.push(parseCatchClause());
	            }
	            if (matchKeyword('finally')) {
	              lex();
	              finalizer = parseBlock();
	            }
	            if (handlers.length === 0 && !finalizer) {
	              throwError({}, Messages.NoCatchOrFinally);
	            }
	            return delegate.createTryStatement(block, [], handlers, finalizer);
	          }

	          // 12.15 The debugger statement

	          function parseDebuggerStatement() {
	            expectKeyword('debugger');
	            consumeSemicolon();
	            return delegate.createDebuggerStatement();
	          }

	          // 12 Statements

	          function parseStatement() {
	            var type = lookahead.type,
	              expr,
	              labeledBody,
	              key,
	              startToken;
	            if (type === Token.EOF) {
	              throwUnexpected(lookahead);
	            }
	            if (type === Token.Punctuator && lookahead.value === '{') {
	              return parseBlock();
	            }
	            startToken = lookahead;
	            if (type === Token.Punctuator) {
	              switch (lookahead.value) {
	                case ';':
	                  return delegate.markEnd(parseEmptyStatement(), startToken);
	                case '(':
	                  return delegate.markEnd(parseExpressionStatement(), startToken);
	              }
	            }
	            if (type === Token.Keyword) {
	              switch (lookahead.value) {
	                case 'break':
	                  return delegate.markEnd(parseBreakStatement(), startToken);
	                case 'continue':
	                  return delegate.markEnd(parseContinueStatement(), startToken);
	                case 'debugger':
	                  return delegate.markEnd(parseDebuggerStatement(), startToken);
	                case 'do':
	                  return delegate.markEnd(parseDoWhileStatement(), startToken);
	                case 'for':
	                  return delegate.markEnd(parseForStatement(), startToken);
	                case 'function':
	                  return delegate.markEnd(parseFunctionDeclaration(), startToken);
	                case 'if':
	                  return delegate.markEnd(parseIfStatement(), startToken);
	                case 'return':
	                  return delegate.markEnd(parseReturnStatement(), startToken);
	                case 'switch':
	                  return delegate.markEnd(parseSwitchStatement(), startToken);
	                case 'throw':
	                  return delegate.markEnd(parseThrowStatement(), startToken);
	                case 'try':
	                  return delegate.markEnd(parseTryStatement(), startToken);
	                case 'var':
	                  return delegate.markEnd(parseVariableStatement(), startToken);
	                case 'while':
	                  return delegate.markEnd(parseWhileStatement(), startToken);
	                case 'with':
	                  return delegate.markEnd(parseWithStatement(), startToken);
	              }
	            }
	            expr = parseExpression();

	            // 12.12 Labelled Statements
	            if (expr.type === Syntax.Identifier && match(':')) {
	              lex();
	              key = '$' + expr.name;
	              if (Object.prototype.hasOwnProperty.call(state.labelSet, key)) {
	                throwError({}, Messages.Redeclaration, 'Label', expr.name);
	              }
	              state.labelSet[key] = true;
	              labeledBody = parseStatement();
	              delete state.labelSet[key];
	              return delegate.markEnd(delegate.createLabeledStatement(expr, labeledBody), startToken);
	            }
	            consumeSemicolon();
	            return delegate.markEnd(delegate.createExpressionStatement(expr), startToken);
	          }

	          // 13 Function Definition

	          function parseFunctionSourceElements() {
	            var sourceElement,
	              sourceElements = [],
	              token,
	              directive,
	              firstRestricted,
	              oldLabelSet,
	              oldInIteration,
	              oldInSwitch,
	              oldInFunctionBody,
	              startToken;
	            startToken = lookahead;
	            expect('{');
	            while (index < length) {
	              if (lookahead.type !== Token.StringLiteral) {
	                break;
	              }
	              token = lookahead;
	              sourceElement = parseSourceElement();
	              sourceElements.push(sourceElement);
	              if (sourceElement.expression.type !== Syntax.Literal) {
	                // this is not directive
	                break;
	              }
	              directive = source.slice(token.start + 1, token.end - 1);
	              if (directive === 'use strict') {
	                strict = true;
	                if (firstRestricted) {
	                  throwErrorTolerant(firstRestricted, Messages.StrictOctalLiteral);
	                }
	              } else {
	                if (!firstRestricted && token.octal) {
	                  firstRestricted = token;
	                }
	              }
	            }
	            oldLabelSet = state.labelSet;
	            oldInIteration = state.inIteration;
	            oldInSwitch = state.inSwitch;
	            oldInFunctionBody = state.inFunctionBody;
	            state.labelSet = {};
	            state.inIteration = false;
	            state.inSwitch = false;
	            state.inFunctionBody = true;
	            while (index < length) {
	              if (match('}')) {
	                break;
	              }
	              sourceElement = parseSourceElement();
	              if (typeof sourceElement === 'undefined') {
	                break;
	              }
	              sourceElements.push(sourceElement);
	            }
	            expect('}');
	            state.labelSet = oldLabelSet;
	            state.inIteration = oldInIteration;
	            state.inSwitch = oldInSwitch;
	            state.inFunctionBody = oldInFunctionBody;
	            return delegate.markEnd(delegate.createBlockStatement(sourceElements), startToken);
	          }
	          function parseParams(firstRestricted) {
	            var param,
	              params = [],
	              token,
	              stricted,
	              paramSet,
	              key,
	              message;
	            expect('(');
	            if (!match(')')) {
	              paramSet = {};
	              while (index < length) {
	                token = lookahead;
	                param = parseVariableIdentifier();
	                key = '$' + token.value;
	                if (strict) {
	                  if (isRestrictedWord(token.value)) {
	                    stricted = token;
	                    message = Messages.StrictParamName;
	                  }
	                  if (Object.prototype.hasOwnProperty.call(paramSet, key)) {
	                    stricted = token;
	                    message = Messages.StrictParamDupe;
	                  }
	                } else if (!firstRestricted) {
	                  if (isRestrictedWord(token.value)) {
	                    firstRestricted = token;
	                    message = Messages.StrictParamName;
	                  } else if (isStrictModeReservedWord(token.value)) {
	                    firstRestricted = token;
	                    message = Messages.StrictReservedWord;
	                  } else if (Object.prototype.hasOwnProperty.call(paramSet, key)) {
	                    firstRestricted = token;
	                    message = Messages.StrictParamDupe;
	                  }
	                }
	                params.push(param);
	                paramSet[key] = true;
	                if (match(')')) {
	                  break;
	                }
	                expect(',');
	              }
	            }
	            expect(')');
	            return {
	              params: params,
	              stricted: stricted,
	              firstRestricted: firstRestricted,
	              message: message
	            };
	          }
	          function parseFunctionDeclaration() {
	            var id,
	              params = [],
	              body,
	              token,
	              stricted,
	              tmp,
	              firstRestricted,
	              message,
	              previousStrict,
	              startToken;
	            startToken = lookahead;
	            expectKeyword('function');
	            token = lookahead;
	            id = parseVariableIdentifier();
	            if (strict) {
	              if (isRestrictedWord(token.value)) {
	                throwErrorTolerant(token, Messages.StrictFunctionName);
	              }
	            } else {
	              if (isRestrictedWord(token.value)) {
	                firstRestricted = token;
	                message = Messages.StrictFunctionName;
	              } else if (isStrictModeReservedWord(token.value)) {
	                firstRestricted = token;
	                message = Messages.StrictReservedWord;
	              }
	            }
	            tmp = parseParams(firstRestricted);
	            params = tmp.params;
	            stricted = tmp.stricted;
	            firstRestricted = tmp.firstRestricted;
	            if (tmp.message) {
	              message = tmp.message;
	            }
	            previousStrict = strict;
	            body = parseFunctionSourceElements();
	            if (strict && firstRestricted) {
	              throwError(firstRestricted, message);
	            }
	            if (strict && stricted) {
	              throwErrorTolerant(stricted, message);
	            }
	            strict = previousStrict;
	            return delegate.markEnd(delegate.createFunctionDeclaration(id, params, [], body), startToken);
	          }
	          function parseFunctionExpression() {
	            var token,
	              id = null,
	              stricted,
	              firstRestricted,
	              message,
	              tmp,
	              params = [],
	              body,
	              previousStrict,
	              startToken;
	            startToken = lookahead;
	            expectKeyword('function');
	            if (!match('(')) {
	              token = lookahead;
	              id = parseVariableIdentifier();
	              if (strict) {
	                if (isRestrictedWord(token.value)) {
	                  throwErrorTolerant(token, Messages.StrictFunctionName);
	                }
	              } else {
	                if (isRestrictedWord(token.value)) {
	                  firstRestricted = token;
	                  message = Messages.StrictFunctionName;
	                } else if (isStrictModeReservedWord(token.value)) {
	                  firstRestricted = token;
	                  message = Messages.StrictReservedWord;
	                }
	              }
	            }
	            tmp = parseParams(firstRestricted);
	            params = tmp.params;
	            stricted = tmp.stricted;
	            firstRestricted = tmp.firstRestricted;
	            if (tmp.message) {
	              message = tmp.message;
	            }
	            previousStrict = strict;
	            body = parseFunctionSourceElements();
	            if (strict && firstRestricted) {
	              throwError(firstRestricted, message);
	            }
	            if (strict && stricted) {
	              throwErrorTolerant(stricted, message);
	            }
	            strict = previousStrict;
	            return delegate.markEnd(delegate.createFunctionExpression(id, params, [], body), startToken);
	          }

	          // 14 Program

	          function parseSourceElement() {
	            if (lookahead.type === Token.Keyword) {
	              switch (lookahead.value) {
	                case 'const':
	                case 'let':
	                  return parseConstLetDeclaration(lookahead.value);
	                case 'function':
	                  return parseFunctionDeclaration();
	                default:
	                  return parseStatement();
	              }
	            }
	            if (lookahead.type !== Token.EOF) {
	              return parseStatement();
	            }
	          }
	          function parseSourceElements() {
	            var sourceElement,
	              sourceElements = [],
	              token,
	              directive,
	              firstRestricted;
	            while (index < length) {
	              token = lookahead;
	              if (token.type !== Token.StringLiteral) {
	                break;
	              }
	              sourceElement = parseSourceElement();
	              sourceElements.push(sourceElement);
	              if (sourceElement.expression.type !== Syntax.Literal) {
	                // this is not directive
	                break;
	              }
	              directive = source.slice(token.start + 1, token.end - 1);
	              if (directive === 'use strict') {
	                strict = true;
	                if (firstRestricted) {
	                  throwErrorTolerant(firstRestricted, Messages.StrictOctalLiteral);
	                }
	              } else {
	                if (!firstRestricted && token.octal) {
	                  firstRestricted = token;
	                }
	              }
	            }
	            while (index < length) {
	              sourceElement = parseSourceElement();
	              /* istanbul ignore if */
	              if (typeof sourceElement === 'undefined') {
	                break;
	              }
	              sourceElements.push(sourceElement);
	            }
	            return sourceElements;
	          }
	          function parseProgram() {
	            var body, startToken;
	            skipComment();
	            peek();
	            startToken = lookahead;
	            strict = false;
	            body = parseSourceElements();
	            return delegate.markEnd(delegate.createProgram(body), startToken);
	          }
	          function filterTokenLocation() {
	            var i,
	              entry,
	              token,
	              tokens = [];
	            for (i = 0; i < extra.tokens.length; ++i) {
	              entry = extra.tokens[i];
	              token = {
	                type: entry.type,
	                value: entry.value
	              };
	              if (extra.range) {
	                token.range = entry.range;
	              }
	              if (extra.loc) {
	                token.loc = entry.loc;
	              }
	              tokens.push(token);
	            }
	            extra.tokens = tokens;
	          }
	          function tokenize(code, options) {
	            var toString, token, tokens;
	            toString = String;
	            if (typeof code !== 'string' && !(code instanceof String)) {
	              code = toString(code);
	            }
	            delegate = SyntaxTreeDelegate;
	            source = code;
	            index = 0;
	            lineNumber = source.length > 0 ? 1 : 0;
	            lineStart = 0;
	            length = source.length;
	            lookahead = null;
	            state = {
	              allowIn: true,
	              labelSet: {},
	              inFunctionBody: false,
	              inIteration: false,
	              inSwitch: false,
	              lastCommentStart: -1
	            };
	            extra = {};

	            // Options matching.
	            options = options || {};

	            // Of course we collect tokens here.
	            options.tokens = true;
	            extra.tokens = [];
	            extra.tokenize = true;
	            // The following two fields are necessary to compute the Regex tokens.
	            extra.openParenToken = -1;
	            extra.openCurlyToken = -1;
	            extra.range = typeof options.range === 'boolean' && options.range;
	            extra.loc = typeof options.loc === 'boolean' && options.loc;
	            if (typeof options.comment === 'boolean' && options.comment) {
	              extra.comments = [];
	            }
	            if (typeof options.tolerant === 'boolean' && options.tolerant) {
	              extra.errors = [];
	            }
	            try {
	              peek();
	              if (lookahead.type === Token.EOF) {
	                return extra.tokens;
	              }
	              token = lex();
	              while (lookahead.type !== Token.EOF) {
	                try {
	                  token = lex();
	                } catch (lexError) {
	                  token = lookahead;
	                  if (extra.errors) {
	                    extra.errors.push(lexError);
	                    // We have to break on the first error
	                    // to avoid infinite loops.
	                    break;
	                  } else {
	                    throw lexError;
	                  }
	                }
	              }
	              filterTokenLocation();
	              tokens = extra.tokens;
	              if (typeof extra.comments !== 'undefined') {
	                tokens.comments = extra.comments;
	              }
	              if (typeof extra.errors !== 'undefined') {
	                tokens.errors = extra.errors;
	              }
	            } catch (e) {
	              throw e;
	            } finally {
	              extra = {};
	            }
	            return tokens;
	          }
	          function parse(code, options) {
	            var program, toString;
	            toString = String;
	            if (typeof code !== 'string' && !(code instanceof String)) {
	              code = toString(code);
	            }
	            delegate = SyntaxTreeDelegate;
	            source = code;
	            index = 0;
	            lineNumber = source.length > 0 ? 1 : 0;
	            lineStart = 0;
	            length = source.length;
	            lookahead = null;
	            state = {
	              allowIn: true,
	              labelSet: {},
	              inFunctionBody: false,
	              inIteration: false,
	              inSwitch: false,
	              lastCommentStart: -1
	            };
	            extra = {};
	            if (typeof options !== 'undefined') {
	              extra.range = typeof options.range === 'boolean' && options.range;
	              extra.loc = typeof options.loc === 'boolean' && options.loc;
	              extra.attachComment = typeof options.attachComment === 'boolean' && options.attachComment;
	              if (extra.loc && options.source !== null && options.source !== undefined) {
	                extra.source = toString(options.source);
	              }
	              if (typeof options.tokens === 'boolean' && options.tokens) {
	                extra.tokens = [];
	              }
	              if (typeof options.comment === 'boolean' && options.comment) {
	                extra.comments = [];
	              }
	              if (typeof options.tolerant === 'boolean' && options.tolerant) {
	                extra.errors = [];
	              }
	              if (extra.attachComment) {
	                extra.range = true;
	                extra.comments = [];
	                extra.bottomRightStack = [];
	                extra.trailingComments = [];
	                extra.leadingComments = [];
	              }
	            }
	            try {
	              program = parseProgram();
	              if (typeof extra.comments !== 'undefined') {
	                program.comments = extra.comments;
	              }
	              if (typeof extra.tokens !== 'undefined') {
	                filterTokenLocation();
	                program.tokens = extra.tokens;
	              }
	              if (typeof extra.errors !== 'undefined') {
	                program.errors = extra.errors;
	              }
	            } catch (e) {
	              throw e;
	            } finally {
	              extra = {};
	            }
	            return program;
	          }

	          // Sync with *.json manifests.
	          exports.version = '1.2.2';
	          exports.tokenize = tokenize;
	          exports.parse = parse;

	          // Deep copy.
	          /* istanbul ignore next */
	          exports.Syntax = function () {
	            var name,
	              types = {};
	            if (typeof Object.create === 'function') {
	              types = Object.create(null);
	            }
	            for (name in Syntax) {
	              if (Syntax.hasOwnProperty(name)) {
	                types[name] = Syntax[name];
	              }
	            }
	            if (typeof Object.freeze === 'function') {
	              Object.freeze(types);
	            }
	            return types;
	          }();
	        });
	        /* vim: set sw=4 ts=4 et tw=80 : */
	      }, {}],
	      1: [function (require, module, exports) {
	        (function (process) {
	          /* parser generated by jison 0.4.13 */
	          /*
	            Returns a Parser object of the following structure:
	          
	            Parser: {
	              yy: {}
	            }
	          
	            Parser.prototype: {
	              yy: {},
	              trace: function(),
	              symbols_: {associative list: name ==> number},
	              terminals_: {associative list: number ==> name},
	              productions_: [...],
	              performAction: function anonymous(yytext, yyleng, yylineno, yy, yystate, $$, _$),
	              table: [...],
	              defaultActions: {...},
	              parseError: function(str, hash),
	              parse: function(input),
	          
	              lexer: {
	                  EOF: 1,
	                  parseError: function(str, hash),
	                  setInput: function(input),
	                  input: function(),
	                  unput: function(str),
	                  more: function(),
	                  less: function(n),
	                  pastInput: function(),
	                  upcomingInput: function(),
	                  showPosition: function(),
	                  test_match: function(regex_match_array, rule_index),
	                  next: function(),
	                  lex: function(),
	                  begin: function(condition),
	                  popState: function(),
	                  _currentRules: function(),
	                  topState: function(),
	                  pushState: function(condition),
	          
	                  options: {
	                      ranges: boolean           (optional: true ==> token location info will include a .range[] member)
	                      flex: boolean             (optional: true ==> flex-like lexing behaviour where the rules are tested exhaustively to find the longest match)
	                      backtrack_lexer: boolean  (optional: true ==> lexer regexes are tested in order and for each matching regex the action code is invoked; the lexer terminates the scan when a token is returned by the action code)
	                  },
	          
	                  performAction: function(yy, yy_, $avoiding_name_collisions, YY_START),
	                  rules: [...],
	                  conditions: {associative list: name ==> set},
	              }
	            }
	          
	          
	            token location info (@$, _$, etc.): {
	              first_line: n,
	              last_line: n,
	              first_column: n,
	              last_column: n,
	              range: [start_number, end_number]       (where the numbers are indexes into the input string, regular zero-based)
	            }
	          
	          
	            the parseError function receives a 'hash' object with these members for lexer and parser errors: {
	              text:        (matched text)
	              token:       (the produced terminal token, if any)
	              line:        (yylineno)
	            }
	            while parser (grammar) errors will also provide these members, i.e. parser errors deliver a superset of attributes: {
	              loc:         (yylloc)
	              expected:    (string describing the set of expected tokens)
	              recoverable: (boolean: TRUE when the parser has a error recovery rule available for this particular error)
	            }
	          */
	          var parser = function () {
	            var parser = {
	              trace: function trace() {},
	              yy: {},
	              symbols_: {
	                "error": 2,
	                "JSON_PATH": 3,
	                "DOLLAR": 4,
	                "PATH_COMPONENTS": 5,
	                "LEADING_CHILD_MEMBER_EXPRESSION": 6,
	                "PATH_COMPONENT": 7,
	                "MEMBER_COMPONENT": 8,
	                "SUBSCRIPT_COMPONENT": 9,
	                "CHILD_MEMBER_COMPONENT": 10,
	                "DESCENDANT_MEMBER_COMPONENT": 11,
	                "DOT": 12,
	                "MEMBER_EXPRESSION": 13,
	                "DOT_DOT": 14,
	                "STAR": 15,
	                "IDENTIFIER": 16,
	                "SCRIPT_EXPRESSION": 17,
	                "INTEGER": 18,
	                "END": 19,
	                "CHILD_SUBSCRIPT_COMPONENT": 20,
	                "DESCENDANT_SUBSCRIPT_COMPONENT": 21,
	                "[": 22,
	                "SUBSCRIPT": 23,
	                "]": 24,
	                "SUBSCRIPT_EXPRESSION": 25,
	                "SUBSCRIPT_EXPRESSION_LIST": 26,
	                "SUBSCRIPT_EXPRESSION_LISTABLE": 27,
	                ",": 28,
	                "STRING_LITERAL": 29,
	                "ARRAY_SLICE": 30,
	                "FILTER_EXPRESSION": 31,
	                "QQ_STRING": 32,
	                "Q_STRING": 33,
	                "$accept": 0,
	                "$end": 1
	              },
	              terminals_: {
	                2: "error",
	                4: "DOLLAR",
	                12: "DOT",
	                14: "DOT_DOT",
	                15: "STAR",
	                16: "IDENTIFIER",
	                17: "SCRIPT_EXPRESSION",
	                18: "INTEGER",
	                19: "END",
	                22: "[",
	                24: "]",
	                28: ",",
	                30: "ARRAY_SLICE",
	                31: "FILTER_EXPRESSION",
	                32: "QQ_STRING",
	                33: "Q_STRING"
	              },
	              productions_: [0, [3, 1], [3, 2], [3, 1], [3, 2], [5, 1], [5, 2], [7, 1], [7, 1], [8, 1], [8, 1], [10, 2], [6, 1], [11, 2], [13, 1], [13, 1], [13, 1], [13, 1], [13, 1], [9, 1], [9, 1], [20, 3], [21, 4], [23, 1], [23, 1], [26, 1], [26, 3], [27, 1], [27, 1], [27, 1], [25, 1], [25, 1], [25, 1], [29, 1], [29, 1]],
	              performAction: function anonymous(yytext, yyleng, yylineno, yy, yystate /* action[1] */, $$ /* vstack */, _$ /* lstack */
	              /**/) {
	                /* this == yyval */
	                if (!yy.ast) {
	                  yy.ast = _ast;
	                  _ast.initialize();
	                }
	                var $0 = $$.length - 1;
	                switch (yystate) {
	                  case 1:
	                    yy.ast.set({
	                      expression: {
	                        type: "root",
	                        value: $$[$0]
	                      }
	                    });
	                    yy.ast.unshift();
	                    return yy.ast.yield();
	                  case 2:
	                    yy.ast.set({
	                      expression: {
	                        type: "root",
	                        value: $$[$0 - 1]
	                      }
	                    });
	                    yy.ast.unshift();
	                    return yy.ast.yield();
	                  case 3:
	                    yy.ast.unshift();
	                    return yy.ast.yield();
	                  case 4:
	                    yy.ast.set({
	                      operation: "member",
	                      scope: "child",
	                      expression: {
	                        type: "identifier",
	                        value: $$[$0 - 1]
	                      }
	                    });
	                    yy.ast.unshift();
	                    return yy.ast.yield();
	                  case 5:
	                    break;
	                  case 6:
	                    break;
	                  case 7:
	                    yy.ast.set({
	                      operation: "member"
	                    });
	                    yy.ast.push();
	                    break;
	                  case 8:
	                    yy.ast.set({
	                      operation: "subscript"
	                    });
	                    yy.ast.push();
	                    break;
	                  case 9:
	                    yy.ast.set({
	                      scope: "child"
	                    });
	                    break;
	                  case 10:
	                    yy.ast.set({
	                      scope: "descendant"
	                    });
	                    break;
	                  case 11:
	                    break;
	                  case 12:
	                    yy.ast.set({
	                      scope: "child",
	                      operation: "member"
	                    });
	                    break;
	                  case 13:
	                    break;
	                  case 14:
	                    yy.ast.set({
	                      expression: {
	                        type: "wildcard",
	                        value: $$[$0]
	                      }
	                    });
	                    break;
	                  case 15:
	                    yy.ast.set({
	                      expression: {
	                        type: "identifier",
	                        value: $$[$0]
	                      }
	                    });
	                    break;
	                  case 16:
	                    yy.ast.set({
	                      expression: {
	                        type: "script_expression",
	                        value: $$[$0]
	                      }
	                    });
	                    break;
	                  case 17:
	                    yy.ast.set({
	                      expression: {
	                        type: "numeric_literal",
	                        value: parseInt($$[$0])
	                      }
	                    });
	                    break;
	                  case 18:
	                    break;
	                  case 19:
	                    yy.ast.set({
	                      scope: "child"
	                    });
	                    break;
	                  case 20:
	                    yy.ast.set({
	                      scope: "descendant"
	                    });
	                    break;
	                  case 21:
	                    break;
	                  case 22:
	                    break;
	                  case 23:
	                    break;
	                  case 24:
	                    $$[$0].length > 1 ? yy.ast.set({
	                      expression: {
	                        type: "union",
	                        value: $$[$0]
	                      }
	                    }) : this.$ = $$[$0];
	                    break;
	                  case 25:
	                    this.$ = [$$[$0]];
	                    break;
	                  case 26:
	                    this.$ = $$[$0 - 2].concat($$[$0]);
	                    break;
	                  case 27:
	                    this.$ = {
	                      expression: {
	                        type: "numeric_literal",
	                        value: parseInt($$[$0])
	                      }
	                    };
	                    yy.ast.set(this.$);
	                    break;
	                  case 28:
	                    this.$ = {
	                      expression: {
	                        type: "string_literal",
	                        value: $$[$0]
	                      }
	                    };
	                    yy.ast.set(this.$);
	                    break;
	                  case 29:
	                    this.$ = {
	                      expression: {
	                        type: "slice",
	                        value: $$[$0]
	                      }
	                    };
	                    yy.ast.set(this.$);
	                    break;
	                  case 30:
	                    this.$ = {
	                      expression: {
	                        type: "wildcard",
	                        value: $$[$0]
	                      }
	                    };
	                    yy.ast.set(this.$);
	                    break;
	                  case 31:
	                    this.$ = {
	                      expression: {
	                        type: "script_expression",
	                        value: $$[$0]
	                      }
	                    };
	                    yy.ast.set(this.$);
	                    break;
	                  case 32:
	                    this.$ = {
	                      expression: {
	                        type: "filter_expression",
	                        value: $$[$0]
	                      }
	                    };
	                    yy.ast.set(this.$);
	                    break;
	                  case 33:
	                    this.$ = $$[$0];
	                    break;
	                  case 34:
	                    this.$ = $$[$0];
	                    break;
	                }
	              },
	              table: [{
	                3: 1,
	                4: [1, 2],
	                6: 3,
	                13: 4,
	                15: [1, 5],
	                16: [1, 6],
	                17: [1, 7],
	                18: [1, 8],
	                19: [1, 9]
	              }, {
	                1: [3]
	              }, {
	                1: [2, 1],
	                5: 10,
	                7: 11,
	                8: 12,
	                9: 13,
	                10: 14,
	                11: 15,
	                12: [1, 18],
	                14: [1, 19],
	                20: 16,
	                21: 17,
	                22: [1, 20]
	              }, {
	                1: [2, 3],
	                5: 21,
	                7: 11,
	                8: 12,
	                9: 13,
	                10: 14,
	                11: 15,
	                12: [1, 18],
	                14: [1, 19],
	                20: 16,
	                21: 17,
	                22: [1, 20]
	              }, {
	                1: [2, 12],
	                12: [2, 12],
	                14: [2, 12],
	                22: [2, 12]
	              }, {
	                1: [2, 14],
	                12: [2, 14],
	                14: [2, 14],
	                22: [2, 14]
	              }, {
	                1: [2, 15],
	                12: [2, 15],
	                14: [2, 15],
	                22: [2, 15]
	              }, {
	                1: [2, 16],
	                12: [2, 16],
	                14: [2, 16],
	                22: [2, 16]
	              }, {
	                1: [2, 17],
	                12: [2, 17],
	                14: [2, 17],
	                22: [2, 17]
	              }, {
	                1: [2, 18],
	                12: [2, 18],
	                14: [2, 18],
	                22: [2, 18]
	              }, {
	                1: [2, 2],
	                7: 22,
	                8: 12,
	                9: 13,
	                10: 14,
	                11: 15,
	                12: [1, 18],
	                14: [1, 19],
	                20: 16,
	                21: 17,
	                22: [1, 20]
	              }, {
	                1: [2, 5],
	                12: [2, 5],
	                14: [2, 5],
	                22: [2, 5]
	              }, {
	                1: [2, 7],
	                12: [2, 7],
	                14: [2, 7],
	                22: [2, 7]
	              }, {
	                1: [2, 8],
	                12: [2, 8],
	                14: [2, 8],
	                22: [2, 8]
	              }, {
	                1: [2, 9],
	                12: [2, 9],
	                14: [2, 9],
	                22: [2, 9]
	              }, {
	                1: [2, 10],
	                12: [2, 10],
	                14: [2, 10],
	                22: [2, 10]
	              }, {
	                1: [2, 19],
	                12: [2, 19],
	                14: [2, 19],
	                22: [2, 19]
	              }, {
	                1: [2, 20],
	                12: [2, 20],
	                14: [2, 20],
	                22: [2, 20]
	              }, {
	                13: 23,
	                15: [1, 5],
	                16: [1, 6],
	                17: [1, 7],
	                18: [1, 8],
	                19: [1, 9]
	              }, {
	                13: 24,
	                15: [1, 5],
	                16: [1, 6],
	                17: [1, 7],
	                18: [1, 8],
	                19: [1, 9],
	                22: [1, 25]
	              }, {
	                15: [1, 29],
	                17: [1, 30],
	                18: [1, 33],
	                23: 26,
	                25: 27,
	                26: 28,
	                27: 32,
	                29: 34,
	                30: [1, 35],
	                31: [1, 31],
	                32: [1, 36],
	                33: [1, 37]
	              }, {
	                1: [2, 4],
	                7: 22,
	                8: 12,
	                9: 13,
	                10: 14,
	                11: 15,
	                12: [1, 18],
	                14: [1, 19],
	                20: 16,
	                21: 17,
	                22: [1, 20]
	              }, {
	                1: [2, 6],
	                12: [2, 6],
	                14: [2, 6],
	                22: [2, 6]
	              }, {
	                1: [2, 11],
	                12: [2, 11],
	                14: [2, 11],
	                22: [2, 11]
	              }, {
	                1: [2, 13],
	                12: [2, 13],
	                14: [2, 13],
	                22: [2, 13]
	              }, {
	                15: [1, 29],
	                17: [1, 30],
	                18: [1, 33],
	                23: 38,
	                25: 27,
	                26: 28,
	                27: 32,
	                29: 34,
	                30: [1, 35],
	                31: [1, 31],
	                32: [1, 36],
	                33: [1, 37]
	              }, {
	                24: [1, 39]
	              }, {
	                24: [2, 23]
	              }, {
	                24: [2, 24],
	                28: [1, 40]
	              }, {
	                24: [2, 30]
	              }, {
	                24: [2, 31]
	              }, {
	                24: [2, 32]
	              }, {
	                24: [2, 25],
	                28: [2, 25]
	              }, {
	                24: [2, 27],
	                28: [2, 27]
	              }, {
	                24: [2, 28],
	                28: [2, 28]
	              }, {
	                24: [2, 29],
	                28: [2, 29]
	              }, {
	                24: [2, 33],
	                28: [2, 33]
	              }, {
	                24: [2, 34],
	                28: [2, 34]
	              }, {
	                24: [1, 41]
	              }, {
	                1: [2, 21],
	                12: [2, 21],
	                14: [2, 21],
	                22: [2, 21]
	              }, {
	                18: [1, 33],
	                27: 42,
	                29: 34,
	                30: [1, 35],
	                32: [1, 36],
	                33: [1, 37]
	              }, {
	                1: [2, 22],
	                12: [2, 22],
	                14: [2, 22],
	                22: [2, 22]
	              }, {
	                24: [2, 26],
	                28: [2, 26]
	              }],
	              defaultActions: {
	                27: [2, 23],
	                29: [2, 30],
	                30: [2, 31],
	                31: [2, 32]
	              },
	              parseError: function parseError(str, hash) {
	                if (hash.recoverable) {
	                  this.trace(str);
	                } else {
	                  throw new Error(str);
	                }
	              },
	              parse: function parse(input) {
	                var self = this,
	                  stack = [0],
	                  vstack = [null],
	                  lstack = [],
	                  table = this.table,
	                  yytext = '',
	                  yylineno = 0,
	                  yyleng = 0,
	                  TERROR = 2,
	                  EOF = 1;
	                var args = lstack.slice.call(arguments, 1);
	                this.lexer.setInput(input);
	                this.lexer.yy = this.yy;
	                this.yy.lexer = this.lexer;
	                this.yy.parser = this;
	                if (typeof this.lexer.yylloc == 'undefined') {
	                  this.lexer.yylloc = {};
	                }
	                var yyloc = this.lexer.yylloc;
	                lstack.push(yyloc);
	                var ranges = this.lexer.options && this.lexer.options.ranges;
	                if (typeof this.yy.parseError === 'function') {
	                  this.parseError = this.yy.parseError;
	                } else {
	                  this.parseError = Object.getPrototypeOf(this).parseError;
	                }
	                function lex() {
	                  var token;
	                  token = self.lexer.lex() || EOF;
	                  if (typeof token !== 'number') {
	                    token = self.symbols_[token] || token;
	                  }
	                  return token;
	                }
	                var symbol,
	                  state,
	                  action,
	                  r,
	                  yyval = {},
	                  p,
	                  len,
	                  newState,
	                  expected;
	                while (true) {
	                  state = stack[stack.length - 1];
	                  if (this.defaultActions[state]) {
	                    action = this.defaultActions[state];
	                  } else {
	                    if (symbol === null || typeof symbol == 'undefined') {
	                      symbol = lex();
	                    }
	                    action = table[state] && table[state][symbol];
	                  }
	                  if (typeof action === 'undefined' || !action.length || !action[0]) {
	                    var errStr = '';
	                    expected = [];
	                    for (p in table[state]) {
	                      if (this.terminals_[p] && p > TERROR) {
	                        expected.push('\'' + this.terminals_[p] + '\'');
	                      }
	                    }
	                    if (this.lexer.showPosition) {
	                      errStr = 'Parse error on line ' + (yylineno + 1) + ':\n' + this.lexer.showPosition() + '\nExpecting ' + expected.join(', ') + ', got \'' + (this.terminals_[symbol] || symbol) + '\'';
	                    } else {
	                      errStr = 'Parse error on line ' + (yylineno + 1) + ': Unexpected ' + (symbol == EOF ? 'end of input' : '\'' + (this.terminals_[symbol] || symbol) + '\'');
	                    }
	                    this.parseError(errStr, {
	                      text: this.lexer.match,
	                      token: this.terminals_[symbol] || symbol,
	                      line: this.lexer.yylineno,
	                      loc: yyloc,
	                      expected: expected
	                    });
	                  }
	                  if (action[0] instanceof Array && action.length > 1) {
	                    throw new Error('Parse Error: multiple actions possible at state: ' + state + ', token: ' + symbol);
	                  }
	                  switch (action[0]) {
	                    case 1:
	                      stack.push(symbol);
	                      vstack.push(this.lexer.yytext);
	                      lstack.push(this.lexer.yylloc);
	                      stack.push(action[1]);
	                      symbol = null;
	                      {
	                        yyleng = this.lexer.yyleng;
	                        yytext = this.lexer.yytext;
	                        yylineno = this.lexer.yylineno;
	                        yyloc = this.lexer.yylloc;
	                      }
	                      break;
	                    case 2:
	                      len = this.productions_[action[1]][1];
	                      yyval.$ = vstack[vstack.length - len];
	                      yyval._$ = {
	                        first_line: lstack[lstack.length - (len || 1)].first_line,
	                        last_line: lstack[lstack.length - 1].last_line,
	                        first_column: lstack[lstack.length - (len || 1)].first_column,
	                        last_column: lstack[lstack.length - 1].last_column
	                      };
	                      if (ranges) {
	                        yyval._$.range = [lstack[lstack.length - (len || 1)].range[0], lstack[lstack.length - 1].range[1]];
	                      }
	                      r = this.performAction.apply(yyval, [yytext, yyleng, yylineno, this.yy, action[1], vstack, lstack].concat(args));
	                      if (typeof r !== 'undefined') {
	                        return r;
	                      }
	                      if (len) {
	                        stack = stack.slice(0, -1 * len * 2);
	                        vstack = vstack.slice(0, -1 * len);
	                        lstack = lstack.slice(0, -1 * len);
	                      }
	                      stack.push(this.productions_[action[1]][0]);
	                      vstack.push(yyval.$);
	                      lstack.push(yyval._$);
	                      newState = table[stack[stack.length - 2]][stack[stack.length - 1]];
	                      stack.push(newState);
	                      break;
	                    case 3:
	                      return true;
	                  }
	                }
	                return true;
	              }
	            };
	            var _ast = {
	              initialize: function () {
	                this._nodes = [];
	                this._node = {};
	                this._stash = [];
	              },
	              set: function (props) {
	                for (var k in props) this._node[k] = props[k];
	                return this._node;
	              },
	              node: function (obj) {
	                if (arguments.length) this._node = obj;
	                return this._node;
	              },
	              push: function () {
	                this._nodes.push(this._node);
	                this._node = {};
	              },
	              unshift: function () {
	                this._nodes.unshift(this._node);
	                this._node = {};
	              },
	              yield: function () {
	                var _nodes = this._nodes;
	                this.initialize();
	                return _nodes;
	              }
	            };
	            /* generated by jison-lex 0.2.1 */
	            var lexer = function () {
	              var lexer = {
	                EOF: 1,
	                parseError: function parseError(str, hash) {
	                  if (this.yy.parser) {
	                    this.yy.parser.parseError(str, hash);
	                  } else {
	                    throw new Error(str);
	                  }
	                },
	                // resets the lexer, sets new input
	                setInput: function (input) {
	                  this._input = input;
	                  this._more = this._backtrack = this.done = false;
	                  this.yylineno = this.yyleng = 0;
	                  this.yytext = this.matched = this.match = '';
	                  this.conditionStack = ['INITIAL'];
	                  this.yylloc = {
	                    first_line: 1,
	                    first_column: 0,
	                    last_line: 1,
	                    last_column: 0
	                  };
	                  if (this.options.ranges) {
	                    this.yylloc.range = [0, 0];
	                  }
	                  this.offset = 0;
	                  return this;
	                },
	                // consumes and returns one char from the input
	                input: function () {
	                  var ch = this._input[0];
	                  this.yytext += ch;
	                  this.yyleng++;
	                  this.offset++;
	                  this.match += ch;
	                  this.matched += ch;
	                  var lines = ch.match(/(?:\r\n?|\n).*/g);
	                  if (lines) {
	                    this.yylineno++;
	                    this.yylloc.last_line++;
	                  } else {
	                    this.yylloc.last_column++;
	                  }
	                  if (this.options.ranges) {
	                    this.yylloc.range[1]++;
	                  }
	                  this._input = this._input.slice(1);
	                  return ch;
	                },
	                // unshifts one char (or a string) into the input
	                unput: function (ch) {
	                  var len = ch.length;
	                  var lines = ch.split(/(?:\r\n?|\n)/g);
	                  this._input = ch + this._input;
	                  this.yytext = this.yytext.substr(0, this.yytext.length - len - 1);
	                  //this.yyleng -= len;
	                  this.offset -= len;
	                  var oldLines = this.match.split(/(?:\r\n?|\n)/g);
	                  this.match = this.match.substr(0, this.match.length - 1);
	                  this.matched = this.matched.substr(0, this.matched.length - 1);
	                  if (lines.length - 1) {
	                    this.yylineno -= lines.length - 1;
	                  }
	                  var r = this.yylloc.range;
	                  this.yylloc = {
	                    first_line: this.yylloc.first_line,
	                    last_line: this.yylineno + 1,
	                    first_column: this.yylloc.first_column,
	                    last_column: lines ? (lines.length === oldLines.length ? this.yylloc.first_column : 0) + oldLines[oldLines.length - lines.length].length - lines[0].length : this.yylloc.first_column - len
	                  };
	                  if (this.options.ranges) {
	                    this.yylloc.range = [r[0], r[0] + this.yyleng - len];
	                  }
	                  this.yyleng = this.yytext.length;
	                  return this;
	                },
	                // When called from action, caches matched text and appends it on next action
	                more: function () {
	                  this._more = true;
	                  return this;
	                },
	                // When called from action, signals the lexer that this rule fails to match the input, so the next matching rule (regex) should be tested instead.
	                reject: function () {
	                  if (this.options.backtrack_lexer) {
	                    this._backtrack = true;
	                  } else {
	                    return this.parseError('Lexical error on line ' + (this.yylineno + 1) + '. You can only invoke reject() in the lexer when the lexer is of the backtracking persuasion (options.backtrack_lexer = true).\n' + this.showPosition(), {
	                      text: "",
	                      token: null,
	                      line: this.yylineno
	                    });
	                  }
	                  return this;
	                },
	                // retain first n characters of the match
	                less: function (n) {
	                  this.unput(this.match.slice(n));
	                },
	                // displays already matched input, i.e. for error messages
	                pastInput: function () {
	                  var past = this.matched.substr(0, this.matched.length - this.match.length);
	                  return (past.length > 20 ? '...' : '') + past.substr(-20).replace(/\n/g, "");
	                },
	                // displays upcoming input, i.e. for error messages
	                upcomingInput: function () {
	                  var next = this.match;
	                  if (next.length < 20) {
	                    next += this._input.substr(0, 20 - next.length);
	                  }
	                  return (next.substr(0, 20) + (next.length > 20 ? '...' : '')).replace(/\n/g, "");
	                },
	                // displays the character position where the lexing error occurred, i.e. for error messages
	                showPosition: function () {
	                  var pre = this.pastInput();
	                  var c = new Array(pre.length + 1).join("-");
	                  return pre + this.upcomingInput() + "\n" + c + "^";
	                },
	                // test the lexed token: return FALSE when not a match, otherwise return token
	                test_match: function (match, indexed_rule) {
	                  var token, lines, backup;
	                  if (this.options.backtrack_lexer) {
	                    // save context
	                    backup = {
	                      yylineno: this.yylineno,
	                      yylloc: {
	                        first_line: this.yylloc.first_line,
	                        last_line: this.last_line,
	                        first_column: this.yylloc.first_column,
	                        last_column: this.yylloc.last_column
	                      },
	                      yytext: this.yytext,
	                      match: this.match,
	                      matches: this.matches,
	                      matched: this.matched,
	                      yyleng: this.yyleng,
	                      offset: this.offset,
	                      _more: this._more,
	                      _input: this._input,
	                      yy: this.yy,
	                      conditionStack: this.conditionStack.slice(0),
	                      done: this.done
	                    };
	                    if (this.options.ranges) {
	                      backup.yylloc.range = this.yylloc.range.slice(0);
	                    }
	                  }
	                  lines = match[0].match(/(?:\r\n?|\n).*/g);
	                  if (lines) {
	                    this.yylineno += lines.length;
	                  }
	                  this.yylloc = {
	                    first_line: this.yylloc.last_line,
	                    last_line: this.yylineno + 1,
	                    first_column: this.yylloc.last_column,
	                    last_column: lines ? lines[lines.length - 1].length - lines[lines.length - 1].match(/\r?\n?/)[0].length : this.yylloc.last_column + match[0].length
	                  };
	                  this.yytext += match[0];
	                  this.match += match[0];
	                  this.matches = match;
	                  this.yyleng = this.yytext.length;
	                  if (this.options.ranges) {
	                    this.yylloc.range = [this.offset, this.offset += this.yyleng];
	                  }
	                  this._more = false;
	                  this._backtrack = false;
	                  this._input = this._input.slice(match[0].length);
	                  this.matched += match[0];
	                  token = this.performAction.call(this, this.yy, this, indexed_rule, this.conditionStack[this.conditionStack.length - 1]);
	                  if (this.done && this._input) {
	                    this.done = false;
	                  }
	                  if (token) {
	                    return token;
	                  } else if (this._backtrack) {
	                    // recover context
	                    for (var k in backup) {
	                      this[k] = backup[k];
	                    }
	                    return false; // rule action called reject() implying the next rule should be tested instead.
	                  }
	                  return false;
	                },
	                // return next match in input
	                next: function () {
	                  if (this.done) {
	                    return this.EOF;
	                  }
	                  if (!this._input) {
	                    this.done = true;
	                  }
	                  var token, match, tempMatch, index;
	                  if (!this._more) {
	                    this.yytext = '';
	                    this.match = '';
	                  }
	                  var rules = this._currentRules();
	                  for (var i = 0; i < rules.length; i++) {
	                    tempMatch = this._input.match(this.rules[rules[i]]);
	                    if (tempMatch && (!match || tempMatch[0].length > match[0].length)) {
	                      match = tempMatch;
	                      index = i;
	                      if (this.options.backtrack_lexer) {
	                        token = this.test_match(tempMatch, rules[i]);
	                        if (token !== false) {
	                          return token;
	                        } else if (this._backtrack) {
	                          match = false;
	                          continue; // rule action called reject() implying a rule MISmatch.
	                        } else {
	                          // else: this is a lexer rule which consumes input without producing a token (e.g. whitespace)
	                          return false;
	                        }
	                      } else if (!this.options.flex) {
	                        break;
	                      }
	                    }
	                  }
	                  if (match) {
	                    token = this.test_match(match, rules[index]);
	                    if (token !== false) {
	                      return token;
	                    }
	                    // else: this is a lexer rule which consumes input without producing a token (e.g. whitespace)
	                    return false;
	                  }
	                  if (this._input === "") {
	                    return this.EOF;
	                  } else {
	                    return this.parseError('Lexical error on line ' + (this.yylineno + 1) + '. Unrecognized text.\n' + this.showPosition(), {
	                      text: "",
	                      token: null,
	                      line: this.yylineno
	                    });
	                  }
	                },
	                // return next match that has a token
	                lex: function lex() {
	                  var r = this.next();
	                  if (r) {
	                    return r;
	                  } else {
	                    return this.lex();
	                  }
	                },
	                // activates a new lexer condition state (pushes the new lexer condition state onto the condition stack)
	                begin: function begin(condition) {
	                  this.conditionStack.push(condition);
	                },
	                // pop the previously active lexer condition state off the condition stack
	                popState: function popState() {
	                  var n = this.conditionStack.length - 1;
	                  if (n > 0) {
	                    return this.conditionStack.pop();
	                  } else {
	                    return this.conditionStack[0];
	                  }
	                },
	                // produce the lexer rule set which is active for the currently active lexer condition state
	                _currentRules: function _currentRules() {
	                  if (this.conditionStack.length && this.conditionStack[this.conditionStack.length - 1]) {
	                    return this.conditions[this.conditionStack[this.conditionStack.length - 1]].rules;
	                  } else {
	                    return this.conditions["INITIAL"].rules;
	                  }
	                },
	                // return the currently active lexer condition state; when an index argument is provided it produces the N-th previous condition state, if available
	                topState: function topState(n) {
	                  n = this.conditionStack.length - 1 - Math.abs(n || 0);
	                  if (n >= 0) {
	                    return this.conditionStack[n];
	                  } else {
	                    return "INITIAL";
	                  }
	                },
	                // alias for begin(condition)
	                pushState: function pushState(condition) {
	                  this.begin(condition);
	                },
	                // return the number of states currently on the stack
	                stateStackSize: function stateStackSize() {
	                  return this.conditionStack.length;
	                },
	                options: {},
	                performAction: function anonymous(yy, yy_, $avoiding_name_collisions, YY_START
	                /**/) {
	                  switch ($avoiding_name_collisions) {
	                    case 0:
	                      return 4;
	                    case 1:
	                      return 14;
	                    case 2:
	                      return 12;
	                    case 3:
	                      return 15;
	                    case 4:
	                      return 16;
	                    case 5:
	                      return 22;
	                    case 6:
	                      return 24;
	                    case 7:
	                      return 28;
	                    case 8:
	                      return 30;
	                    case 9:
	                      return 18;
	                    case 10:
	                      yy_.yytext = yy_.yytext.substr(1, yy_.yyleng - 2);
	                      return 32;
	                    case 11:
	                      yy_.yytext = yy_.yytext.substr(1, yy_.yyleng - 2);
	                      return 33;
	                    case 12:
	                      return 17;
	                    case 13:
	                      return 31;
	                  }
	                },
	                rules: [/^(?:\$)/, /^(?:\.\.)/, /^(?:\.)/, /^(?:\*)/, /^(?:[a-zA-Z_]+[a-zA-Z0-9_]*)/, /^(?:\[)/, /^(?:\])/, /^(?:,)/, /^(?:((-?(?:0|[1-9][0-9]*)))?\:((-?(?:0|[1-9][0-9]*)))?(\:((-?(?:0|[1-9][0-9]*)))?)?)/, /^(?:(-?(?:0|[1-9][0-9]*)))/, /^(?:"(?:\\["bfnrt/\\]|\\u[a-fA-F0-9]{4}|[^"\\])*")/, /^(?:'(?:\\['bfnrt/\\]|\\u[a-fA-F0-9]{4}|[^'\\])*')/, /^(?:\(.+?\)(?=\]))/, /^(?:\?\(.+?\)(?=\]))/],
	                conditions: {
	                  "INITIAL": {
	                    "rules": [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13],
	                    "inclusive": true
	                  }
	                }
	              };
	              return lexer;
	            }();
	            parser.lexer = lexer;
	            function Parser() {
	              this.yy = {};
	            }
	            Parser.prototype = parser;
	            parser.Parser = Parser;
	            return new Parser();
	          }();
	          if (typeof require !== 'undefined' && typeof exports !== 'undefined') {
	            exports.parser = parser;
	            exports.Parser = parser.Parser;
	            exports.parse = function () {
	              return parser.parse.apply(parser, arguments);
	            };
	            exports.main = function commonjsMain(args) {
	              if (!args[1]) {
	                console.log('Usage: ' + args[0] + ' FILE');
	                process.exit(1);
	              }
	              var source = require('fs').readFileSync(require('path').normalize(args[1]), "utf8");
	              return exports.parser.parse(source);
	            };
	            if (typeof module !== 'undefined' && require.main === module) {
	              exports.main(process.argv.slice(1));
	            }
	          }
	        }).call(this, require('_process'));
	      }, {
	        "_process": 14,
	        "fs": 12,
	        "path": 13
	      }],
	      2: [function (require, module, exports) {
	        module.exports = {
	          identifier: "[a-zA-Z_]+[a-zA-Z0-9_]*",
	          integer: "-?(?:0|[1-9][0-9]*)",
	          qq_string: "\"(?:\\\\[\"bfnrt/\\\\]|\\\\u[a-fA-F0-9]{4}|[^\"\\\\])*\"",
	          q_string: "'(?:\\\\[\'bfnrt/\\\\]|\\\\u[a-fA-F0-9]{4}|[^\'\\\\])*'"
	        };
	      }, {}],
	      3: [function (require, module, exports) {
	        var dict = require('./dict');
	        var fs = require('fs');
	        var grammar = {
	          lex: {
	            macros: {
	              esc: "\\\\",
	              int: dict.integer
	            },
	            rules: [["\\$", "return 'DOLLAR'"], ["\\.\\.", "return 'DOT_DOT'"], ["\\.", "return 'DOT'"], ["\\*", "return 'STAR'"], [dict.identifier, "return 'IDENTIFIER'"], ["\\[", "return '['"], ["\\]", "return ']'"], [",", "return ','"], ["({int})?\\:({int})?(\\:({int})?)?", "return 'ARRAY_SLICE'"], ["{int}", "return 'INTEGER'"], [dict.qq_string, "yytext = yytext.substr(1,yyleng-2); return 'QQ_STRING';"], [dict.q_string, "yytext = yytext.substr(1,yyleng-2); return 'Q_STRING';"], ["\\(.+?\\)(?=\\])", "return 'SCRIPT_EXPRESSION'"], ["\\?\\(.+?\\)(?=\\])", "return 'FILTER_EXPRESSION'"]]
	          },
	          start: "JSON_PATH",
	          bnf: {
	            JSON_PATH: [['DOLLAR', 'yy.ast.set({ expression: { type: "root", value: $1 } }); yy.ast.unshift(); return yy.ast.yield()'], ['DOLLAR PATH_COMPONENTS', 'yy.ast.set({ expression: { type: "root", value: $1 } }); yy.ast.unshift(); return yy.ast.yield()'], ['LEADING_CHILD_MEMBER_EXPRESSION', 'yy.ast.unshift(); return yy.ast.yield()'], ['LEADING_CHILD_MEMBER_EXPRESSION PATH_COMPONENTS', 'yy.ast.set({ operation: "member", scope: "child", expression: { type: "identifier", value: $1 }}); yy.ast.unshift(); return yy.ast.yield()']],
	            PATH_COMPONENTS: [['PATH_COMPONENT', ''], ['PATH_COMPONENTS PATH_COMPONENT', '']],
	            PATH_COMPONENT: [['MEMBER_COMPONENT', 'yy.ast.set({ operation: "member" }); yy.ast.push()'], ['SUBSCRIPT_COMPONENT', 'yy.ast.set({ operation: "subscript" }); yy.ast.push() ']],
	            MEMBER_COMPONENT: [['CHILD_MEMBER_COMPONENT', 'yy.ast.set({ scope: "child" })'], ['DESCENDANT_MEMBER_COMPONENT', 'yy.ast.set({ scope: "descendant" })']],
	            CHILD_MEMBER_COMPONENT: [['DOT MEMBER_EXPRESSION', '']],
	            LEADING_CHILD_MEMBER_EXPRESSION: [['MEMBER_EXPRESSION', 'yy.ast.set({ scope: "child", operation: "member" })']],
	            DESCENDANT_MEMBER_COMPONENT: [['DOT_DOT MEMBER_EXPRESSION', '']],
	            MEMBER_EXPRESSION: [['STAR', 'yy.ast.set({ expression: { type: "wildcard", value: $1 } })'], ['IDENTIFIER', 'yy.ast.set({ expression: { type: "identifier", value: $1 } })'], ['SCRIPT_EXPRESSION', 'yy.ast.set({ expression: { type: "script_expression", value: $1 } })'], ['INTEGER', 'yy.ast.set({ expression: { type: "numeric_literal", value: parseInt($1) } })'], ['END', '']],
	            SUBSCRIPT_COMPONENT: [['CHILD_SUBSCRIPT_COMPONENT', 'yy.ast.set({ scope: "child" })'], ['DESCENDANT_SUBSCRIPT_COMPONENT', 'yy.ast.set({ scope: "descendant" })']],
	            CHILD_SUBSCRIPT_COMPONENT: [['[ SUBSCRIPT ]', '']],
	            DESCENDANT_SUBSCRIPT_COMPONENT: [['DOT_DOT [ SUBSCRIPT ]', '']],
	            SUBSCRIPT: [['SUBSCRIPT_EXPRESSION', ''], ['SUBSCRIPT_EXPRESSION_LIST', '$1.length > 1? yy.ast.set({ expression: { type: "union", value: $1 } }) : $$ = $1']],
	            SUBSCRIPT_EXPRESSION_LIST: [['SUBSCRIPT_EXPRESSION_LISTABLE', '$$ = [$1]'], ['SUBSCRIPT_EXPRESSION_LIST , SUBSCRIPT_EXPRESSION_LISTABLE', '$$ = $1.concat($3)']],
	            SUBSCRIPT_EXPRESSION_LISTABLE: [['INTEGER', '$$ = { expression: { type: "numeric_literal", value: parseInt($1) } }; yy.ast.set($$)'], ['STRING_LITERAL', '$$ = { expression: { type: "string_literal", value: $1 } }; yy.ast.set($$)'], ['ARRAY_SLICE', '$$ = { expression: { type: "slice", value: $1 } }; yy.ast.set($$)']],
	            SUBSCRIPT_EXPRESSION: [['STAR', '$$ = { expression: { type: "wildcard", value: $1 } }; yy.ast.set($$)'], ['SCRIPT_EXPRESSION', '$$ = { expression: { type: "script_expression", value: $1 } }; yy.ast.set($$)'], ['FILTER_EXPRESSION', '$$ = { expression: { type: "filter_expression", value: $1 } }; yy.ast.set($$)']],
	            STRING_LITERAL: [['QQ_STRING', "$$ = $1"], ['Q_STRING', "$$ = $1"]]
	          }
	        };
	        if (fs.readFileSync) {
	          grammar.moduleInclude = fs.readFileSync(require.resolve("../include/module.js"));
	          grammar.actionInclude = fs.readFileSync(require.resolve("../include/action.js"));
	        }
	        module.exports = grammar;
	      }, {
	        "./dict": 2,
	        "fs": 12
	      }],
	      4: [function (require, module, exports) {
	        var aesprim = require('./aesprim');
	        var slice = require('./slice');
	        var _evaluate = require('static-eval');
	        var _uniq = require('underscore').uniq;
	        var Handlers = function () {
	          return this.initialize.apply(this, arguments);
	        };
	        Handlers.prototype.initialize = function () {
	          this.traverse = traverser(true);
	          this.descend = traverser();
	        };
	        Handlers.prototype.keys = Object.keys;
	        Handlers.prototype.resolve = function (component) {
	          var key = [component.operation, component.scope, component.expression.type].join('-');
	          var method = this._fns[key];
	          if (!method) throw new Error("couldn't resolve key: " + key);
	          return method.bind(this);
	        };
	        Handlers.prototype.register = function (key, handler) {
	          if (!handler instanceof Function) {
	            throw new Error("handler must be a function");
	          }
	          this._fns[key] = handler;
	        };
	        Handlers.prototype._fns = {
	          'member-child-identifier': function (component, partial) {
	            var key = component.expression.value;
	            var value = partial.value;
	            if (value instanceof Object && key in value) {
	              return [{
	                value: value[key],
	                path: partial.path.concat(key)
	              }];
	            }
	          },
	          'member-descendant-identifier': _traverse(function (key, value, ref) {
	            return key == ref;
	          }),
	          'subscript-child-numeric_literal': _descend(function (key, value, ref) {
	            return key === ref;
	          }),
	          'member-child-numeric_literal': _descend(function (key, value, ref) {
	            return String(key) === String(ref);
	          }),
	          'subscript-descendant-numeric_literal': _traverse(function (key, value, ref) {
	            return key === ref;
	          }),
	          'member-child-wildcard': _descend(function () {
	            return true;
	          }),
	          'member-descendant-wildcard': _traverse(function () {
	            return true;
	          }),
	          'subscript-descendant-wildcard': _traverse(function () {
	            return true;
	          }),
	          'subscript-child-wildcard': _descend(function () {
	            return true;
	          }),
	          'subscript-child-slice': function (component, partial) {
	            if (is_array(partial.value)) {
	              var args = component.expression.value.split(':').map(_parse_nullable_int);
	              var values = partial.value.map(function (v, i) {
	                return {
	                  value: v,
	                  path: partial.path.concat(i)
	                };
	              });
	              return slice.apply(null, [values].concat(args));
	            }
	          },
	          'subscript-child-union': function (component, partial) {
	            var results = [];
	            component.expression.value.forEach(function (component) {
	              var _component = {
	                operation: 'subscript',
	                scope: 'child',
	                expression: component.expression
	              };
	              var handler = this.resolve(_component);
	              var _results = handler(_component, partial);
	              if (_results) {
	                results = results.concat(_results);
	              }
	            }, this);
	            return unique(results);
	          },
	          'subscript-descendant-union': function (component, partial, count) {
	            var jp = require('..');
	            var self = this;
	            var results = [];
	            var nodes = jp.nodes(partial, '$..*').slice(1);
	            nodes.forEach(function (node) {
	              if (results.length >= count) return;
	              component.expression.value.forEach(function (component) {
	                var _component = {
	                  operation: 'subscript',
	                  scope: 'child',
	                  expression: component.expression
	                };
	                var handler = self.resolve(_component);
	                var _results = handler(_component, node);
	                results = results.concat(_results);
	              });
	            });
	            return unique(results);
	          },
	          'subscript-child-filter_expression': function (component, partial, count) {
	            // slice out the expression from ?(expression)
	            var src = component.expression.value.slice(2, -1);
	            var ast = aesprim.parse(src).body[0].expression;
	            var passable = function (key, value) {
	              return evaluate(ast, {
	                '@': value
	              });
	            };
	            return this.descend(partial, null, passable, count);
	          },
	          'subscript-descendant-filter_expression': function (component, partial, count) {
	            // slice out the expression from ?(expression)
	            var src = component.expression.value.slice(2, -1);
	            var ast = aesprim.parse(src).body[0].expression;
	            var passable = function (key, value) {
	              return evaluate(ast, {
	                '@': value
	              });
	            };
	            return this.traverse(partial, null, passable, count);
	          },
	          'subscript-child-script_expression': function (component, partial) {
	            var exp = component.expression.value.slice(1, -1);
	            return eval_recurse(partial, exp, '$[{{value}}]');
	          },
	          'member-child-script_expression': function (component, partial) {
	            var exp = component.expression.value.slice(1, -1);
	            return eval_recurse(partial, exp, '$.{{value}}');
	          },
	          'member-descendant-script_expression': function (component, partial) {
	            var exp = component.expression.value.slice(1, -1);
	            return eval_recurse(partial, exp, '$..value');
	          }
	        };
	        Handlers.prototype._fns['subscript-child-string_literal'] = Handlers.prototype._fns['member-child-identifier'];
	        Handlers.prototype._fns['member-descendant-numeric_literal'] = Handlers.prototype._fns['subscript-descendant-string_literal'] = Handlers.prototype._fns['member-descendant-identifier'];
	        function eval_recurse(partial, src, template) {
	          var jp = require('./index');
	          var ast = aesprim.parse(src).body[0].expression;
	          var value = evaluate(ast, {
	            '@': partial.value
	          });
	          var path = template.replace(/\{\{\s*value\s*\}\}/g, value);
	          var results = jp.nodes(partial.value, path);
	          results.forEach(function (r) {
	            r.path = partial.path.concat(r.path.slice(1));
	          });
	          return results;
	        }
	        function is_array(val) {
	          return Array.isArray(val);
	        }
	        function is_object(val) {
	          // is this a non-array, non-null object?
	          return val && !(val instanceof Array) && val instanceof Object;
	        }
	        function traverser(recurse) {
	          return function (partial, ref, passable, count) {
	            var value = partial.value;
	            var path = partial.path;
	            var results = [];
	            var descend = function (value, path) {
	              if (is_array(value)) {
	                value.forEach(function (element, index) {
	                  if (results.length >= count) {
	                    return;
	                  }
	                  if (passable(index, element, ref)) {
	                    results.push({
	                      path: path.concat(index),
	                      value: element
	                    });
	                  }
	                });
	                value.forEach(function (element, index) {
	                  if (results.length >= count) {
	                    return;
	                  }
	                  if (recurse) {
	                    descend(element, path.concat(index));
	                  }
	                });
	              } else if (is_object(value)) {
	                this.keys(value).forEach(function (k) {
	                  if (results.length >= count) {
	                    return;
	                  }
	                  if (passable(k, value[k], ref)) {
	                    results.push({
	                      path: path.concat(k),
	                      value: value[k]
	                    });
	                  }
	                });
	                this.keys(value).forEach(function (k) {
	                  if (results.length >= count) {
	                    return;
	                  }
	                  if (recurse) {
	                    descend(value[k], path.concat(k));
	                  }
	                });
	              }
	            }.bind(this);
	            descend(value, path);
	            return results;
	          };
	        }
	        function _descend(passable) {
	          return function (component, partial, count) {
	            return this.descend(partial, component.expression.value, passable, count);
	          };
	        }
	        function _traverse(passable) {
	          return function (component, partial, count) {
	            return this.traverse(partial, component.expression.value, passable, count);
	          };
	        }
	        function evaluate() {
	          try {
	            return _evaluate.apply(this, arguments);
	          } catch (e) {}
	        }
	        function unique(results) {
	          results = results.filter(function (d) {
	            return d;
	          });
	          return _uniq(results, function (r) {
	            return r.path.map(function (c) {
	              return String(c).replace('-', '--');
	            }).join('-');
	          });
	        }
	        function _parse_nullable_int(val) {
	          var sval = String(val);
	          return sval.match(/^-?[0-9]+$/) ? parseInt(sval) : null;
	        }
	        module.exports = Handlers;
	      }, {
	        "..": "jsonpath",
	        "./aesprim": "./aesprim",
	        "./index": 5,
	        "./slice": 7,
	        "static-eval": 15,
	        "underscore": 12
	      }],
	      5: [function (require, module, exports) {
	        var assert = require('assert');
	        var dict = require('./dict');
	        var Parser = require('./parser');
	        var Handlers = require('./handlers');
	        var JSONPath = function () {
	          this.initialize.apply(this, arguments);
	        };
	        JSONPath.prototype.initialize = function () {
	          this.parser = new Parser();
	          this.handlers = new Handlers();
	        };
	        JSONPath.prototype.parse = function (string) {
	          assert.ok(_is_string(string), "we need a path");
	          return this.parser.parse(string);
	        };
	        JSONPath.prototype.parent = function (obj, string) {
	          assert.ok(obj instanceof Object, "obj needs to be an object");
	          assert.ok(string, "we need a path");
	          var node = this.nodes(obj, string)[0];
	          node.path.pop(); /* jshint unused:false */
	          return this.value(obj, node.path);
	        };
	        JSONPath.prototype.apply = function (obj, string, fn) {
	          assert.ok(obj instanceof Object, "obj needs to be an object");
	          assert.ok(string, "we need a path");
	          assert.equal(typeof fn, "function", "fn needs to be function");
	          var nodes = this.nodes(obj, string).sort(function (a, b) {
	            // sort nodes so we apply from the bottom up
	            return b.path.length - a.path.length;
	          });
	          nodes.forEach(function (node) {
	            var key = node.path.pop();
	            var parent = this.value(obj, this.stringify(node.path));
	            var val = node.value = fn.call(obj, parent[key]);
	            parent[key] = val;
	          }, this);
	          return nodes;
	        };
	        JSONPath.prototype.value = function (obj, path, value) {
	          assert.ok(obj instanceof Object, "obj needs to be an object");
	          assert.ok(path, "we need a path");
	          if (arguments.length >= 3) {
	            var node = this.nodes(obj, path).shift();
	            if (!node) return this._vivify(obj, path, value);
	            var key = node.path.slice(-1).shift();
	            var parent = this.parent(obj, this.stringify(node.path));
	            parent[key] = value;
	          }
	          return this.query(obj, this.stringify(path), 1).shift();
	        };
	        JSONPath.prototype._vivify = function (obj, string, value) {
	          var self = this;
	          assert.ok(obj instanceof Object, "obj needs to be an object");
	          assert.ok(string, "we need a path");
	          var path = this.parser.parse(string).map(function (component) {
	            return component.expression.value;
	          });
	          var setValue = function (path, value) {
	            var key = path.pop();
	            var node = self.value(obj, path);
	            if (!node) {
	              setValue(path.concat(), typeof key === 'string' ? {} : []);
	              node = self.value(obj, path);
	            }
	            node[key] = value;
	          };
	          setValue(path, value);
	          return this.query(obj, string)[0];
	        };
	        JSONPath.prototype.query = function (obj, string, count) {
	          assert.ok(obj instanceof Object, "obj needs to be an object");
	          assert.ok(_is_string(string), "we need a path");
	          var results = this.nodes(obj, string, count).map(function (r) {
	            return r.value;
	          });
	          return results;
	        };
	        JSONPath.prototype.paths = function (obj, string, count) {
	          assert.ok(obj instanceof Object, "obj needs to be an object");
	          assert.ok(string, "we need a path");
	          var results = this.nodes(obj, string, count).map(function (r) {
	            return r.path;
	          });
	          return results;
	        };
	        JSONPath.prototype.nodes = function (obj, string, count) {
	          assert.ok(obj instanceof Object, "obj needs to be an object");
	          assert.ok(string, "we need a path");
	          if (count === 0) return [];
	          var path = this.parser.parse(string);
	          var handlers = this.handlers;
	          var partials = [{
	            path: ['$'],
	            value: obj
	          }];
	          var matches = [];
	          if (path.length && path[0].expression.type == 'root') path.shift();
	          if (!path.length) return partials;
	          path.forEach(function (component, index) {
	            if (matches.length >= count) return;
	            var handler = handlers.resolve(component);
	            var _partials = [];
	            partials.forEach(function (p) {
	              if (matches.length >= count) return;
	              var results = handler(component, p, count);
	              if (index == path.length - 1) {
	                // if we're through the components we're done
	                matches = matches.concat(results || []);
	              } else {
	                // otherwise accumulate and carry on through
	                _partials = _partials.concat(results || []);
	              }
	            });
	            partials = _partials;
	          });
	          return count ? matches.slice(0, count) : matches;
	        };
	        JSONPath.prototype.stringify = function (path) {
	          assert.ok(path, "we need a path");
	          var string = '$';
	          var templates = {
	            'descendant-member': '..{{value}}',
	            'child-member': '.{{value}}',
	            'descendant-subscript': '..[{{value}}]',
	            'child-subscript': '[{{value}}]'
	          };
	          path = this._normalize(path);
	          path.forEach(function (component) {
	            if (component.expression.type == 'root') return;
	            var key = [component.scope, component.operation].join('-');
	            var template = templates[key];
	            var value;
	            if (component.expression.type == 'string_literal') {
	              value = JSON.stringify(component.expression.value);
	            } else {
	              value = component.expression.value;
	            }
	            if (!template) throw new Error("couldn't find template " + key);
	            string += template.replace(/{{value}}/, value);
	          });
	          return string;
	        };
	        JSONPath.prototype._normalize = function (path) {
	          assert.ok(path, "we need a path");
	          if (typeof path == "string") {
	            return this.parser.parse(path);
	          } else if (Array.isArray(path) && typeof path[0] == "string") {
	            var _path = [{
	              expression: {
	                type: "root",
	                value: "$"
	              }
	            }];
	            path.forEach(function (component, index) {
	              if (component == '$' && index === 0) return;
	              if (typeof component == "string" && component.match("^" + dict.identifier + "$")) {
	                _path.push({
	                  operation: 'member',
	                  scope: 'child',
	                  expression: {
	                    value: component,
	                    type: 'identifier'
	                  }
	                });
	              } else {
	                var type = typeof component == "number" ? 'numeric_literal' : 'string_literal';
	                _path.push({
	                  operation: 'subscript',
	                  scope: 'child',
	                  expression: {
	                    value: component,
	                    type: type
	                  }
	                });
	              }
	            });
	            return _path;
	          } else if (Array.isArray(path) && typeof path[0] == "object") {
	            return path;
	          }
	          throw new Error("couldn't understand path " + path);
	        };
	        function _is_string(obj) {
	          return Object.prototype.toString.call(obj) == '[object String]';
	        }
	        JSONPath.Handlers = Handlers;
	        JSONPath.Parser = Parser;
	        var instance = new JSONPath();
	        instance.JSONPath = JSONPath;
	        module.exports = instance;
	      }, {
	        "./dict": 2,
	        "./handlers": 4,
	        "./parser": 6,
	        "assert": 8
	      }],
	      6: [function (require, module, exports) {
	        var grammar = require('./grammar');
	        var gparser = require('../generated/parser');
	        var Parser = function () {
	          var parser = new gparser.Parser();
	          var _parseError = parser.parseError;
	          parser.yy.parseError = function () {
	            if (parser.yy.ast) {
	              parser.yy.ast.initialize();
	            }
	            _parseError.apply(parser, arguments);
	          };
	          return parser;
	        };
	        Parser.grammar = grammar;
	        module.exports = Parser;
	      }, {
	        "../generated/parser": 1,
	        "./grammar": 3
	      }],
	      7: [function (require, module, exports) {
	        module.exports = function (arr, start, end, step) {
	          if (typeof start == 'string') throw new Error("start cannot be a string");
	          if (typeof end == 'string') throw new Error("end cannot be a string");
	          if (typeof step == 'string') throw new Error("step cannot be a string");
	          var len = arr.length;
	          if (step === 0) throw new Error("step cannot be zero");
	          step = step ? integer(step) : 1;

	          // normalize negative values
	          start = start < 0 ? len + start : start;
	          end = end < 0 ? len + end : end;

	          // default extents to extents
	          start = integer(start === 0 ? 0 : !start ? step > 0 ? 0 : len - 1 : start);
	          end = integer(end === 0 ? 0 : !end ? step > 0 ? len : -1 : end);

	          // clamp extents
	          start = step > 0 ? Math.max(0, start) : Math.min(len, start);
	          end = step > 0 ? Math.min(end, len) : Math.max(-1, end);

	          // return empty if extents are backwards
	          if (step > 0 && end <= start) return [];
	          if (step < 0 && start <= end) return [];
	          var result = [];
	          for (var i = start; i != end; i += step) {
	            if (step < 0 && i <= end || step > 0 && i >= end) break;
	            result.push(arr[i]);
	          }
	          return result;
	        };
	        function integer(val) {
	          return String(val).match(/^[0-9]+$/) ? parseInt(val) : Number.isFinite(val) ? parseInt(val, 10) : 0;
	        }
	      }, {}],
	      8: [function (require, module, exports) {
	        // http://wiki.commonjs.org/wiki/Unit_Testing/1.0
	        //
	        // THIS IS NOT TESTED NOR LIKELY TO WORK OUTSIDE V8!
	        //
	        // Originally from narwhal.js (http://narwhaljs.org)
	        // Copyright (c) 2009 Thomas Robinson <280north.com>
	        //
	        // Permission is hereby granted, free of charge, to any person obtaining a copy
	        // of this software and associated documentation files (the 'Software'), to
	        // deal in the Software without restriction, including without limitation the
	        // rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
	        // sell copies of the Software, and to permit persons to whom the Software is
	        // furnished to do so, subject to the following conditions:
	        //
	        // The above copyright notice and this permission notice shall be included in
	        // all copies or substantial portions of the Software.
	        //
	        // THE SOFTWARE IS PROVIDED 'AS IS', WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	        // IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
	        // FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
	        // AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
	        // ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
	        // WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

	        // when used in node, this will actually load the util module we depend on
	        // versus loading the builtin util module as happens otherwise
	        // this is a bug in node module loading as far as I am concerned
	        var util = require('util/');
	        var pSlice = Array.prototype.slice;
	        var hasOwn = Object.prototype.hasOwnProperty;

	        // 1. The assert module provides functions that throw
	        // AssertionError's when particular conditions are not met. The
	        // assert module must conform to the following interface.

	        var assert = module.exports = ok;

	        // 2. The AssertionError is defined in assert.
	        // new assert.AssertionError({ message: message,
	        //                             actual: actual,
	        //                             expected: expected })

	        assert.AssertionError = function AssertionError(options) {
	          this.name = 'AssertionError';
	          this.actual = options.actual;
	          this.expected = options.expected;
	          this.operator = options.operator;
	          if (options.message) {
	            this.message = options.message;
	            this.generatedMessage = false;
	          } else {
	            this.message = getMessage(this);
	            this.generatedMessage = true;
	          }
	          var stackStartFunction = options.stackStartFunction || fail;
	          if (Error.captureStackTrace) {
	            Error.captureStackTrace(this, stackStartFunction);
	          } else {
	            // non v8 browsers so we can have a stacktrace
	            var err = new Error();
	            if (err.stack) {
	              var out = err.stack;

	              // try to strip useless frames
	              var fn_name = stackStartFunction.name;
	              var idx = out.indexOf('\n' + fn_name);
	              if (idx >= 0) {
	                // once we have located the function frame
	                // we need to strip out everything before it (and its line)
	                var next_line = out.indexOf('\n', idx + 1);
	                out = out.substring(next_line + 1);
	              }
	              this.stack = out;
	            }
	          }
	        };

	        // assert.AssertionError instanceof Error
	        util.inherits(assert.AssertionError, Error);
	        function replacer(key, value) {
	          if (util.isUndefined(value)) {
	            return '' + value;
	          }
	          if (util.isNumber(value) && !isFinite(value)) {
	            return value.toString();
	          }
	          if (util.isFunction(value) || util.isRegExp(value)) {
	            return value.toString();
	          }
	          return value;
	        }
	        function truncate(s, n) {
	          if (util.isString(s)) {
	            return s.length < n ? s : s.slice(0, n);
	          } else {
	            return s;
	          }
	        }
	        function getMessage(self) {
	          return truncate(JSON.stringify(self.actual, replacer), 128) + ' ' + self.operator + ' ' + truncate(JSON.stringify(self.expected, replacer), 128);
	        }

	        // At present only the three keys mentioned above are used and
	        // understood by the spec. Implementations or sub modules can pass
	        // other keys to the AssertionError's constructor - they will be
	        // ignored.

	        // 3. All of the following functions must throw an AssertionError
	        // when a corresponding condition is not met, with a message that
	        // may be undefined if not provided.  All assertion methods provide
	        // both the actual and expected values to the assertion error for
	        // display purposes.

	        function fail(actual, expected, message, operator, stackStartFunction) {
	          throw new assert.AssertionError({
	            message: message,
	            actual: actual,
	            expected: expected,
	            operator: operator,
	            stackStartFunction: stackStartFunction
	          });
	        }

	        // EXTENSION! allows for well behaved errors defined elsewhere.
	        assert.fail = fail;

	        // 4. Pure assertion tests whether a value is truthy, as determined
	        // by !!guard.
	        // assert.ok(guard, message_opt);
	        // This statement is equivalent to assert.equal(true, !!guard,
	        // message_opt);. To test strictly for the value true, use
	        // assert.strictEqual(true, guard, message_opt);.

	        function ok(value, message) {
	          if (!value) fail(value, true, message, '==', assert.ok);
	        }
	        assert.ok = ok;

	        // 5. The equality assertion tests shallow, coercive equality with
	        // ==.
	        // assert.equal(actual, expected, message_opt);

	        assert.equal = function equal(actual, expected, message) {
	          if (actual != expected) fail(actual, expected, message, '==', assert.equal);
	        };

	        // 6. The non-equality assertion tests for whether two objects are not equal
	        // with != assert.notEqual(actual, expected, message_opt);

	        assert.notEqual = function notEqual(actual, expected, message) {
	          if (actual == expected) {
	            fail(actual, expected, message, '!=', assert.notEqual);
	          }
	        };

	        // 7. The equivalence assertion tests a deep equality relation.
	        // assert.deepEqual(actual, expected, message_opt);

	        assert.deepEqual = function deepEqual(actual, expected, message) {
	          if (!_deepEqual(actual, expected)) {
	            fail(actual, expected, message, 'deepEqual', assert.deepEqual);
	          }
	        };
	        function _deepEqual(actual, expected) {
	          // 7.1. All identical values are equivalent, as determined by ===.
	          if (actual === expected) {
	            return true;
	          } else if (util.isBuffer(actual) && util.isBuffer(expected)) {
	            if (actual.length != expected.length) return false;
	            for (var i = 0; i < actual.length; i++) {
	              if (actual[i] !== expected[i]) return false;
	            }
	            return true;

	            // 7.2. If the expected value is a Date object, the actual value is
	            // equivalent if it is also a Date object that refers to the same time.
	          } else if (util.isDate(actual) && util.isDate(expected)) {
	            return actual.getTime() === expected.getTime();

	            // 7.3 If the expected value is a RegExp object, the actual value is
	            // equivalent if it is also a RegExp object with the same source and
	            // properties (`global`, `multiline`, `lastIndex`, `ignoreCase`).
	          } else if (util.isRegExp(actual) && util.isRegExp(expected)) {
	            return actual.source === expected.source && actual.global === expected.global && actual.multiline === expected.multiline && actual.lastIndex === expected.lastIndex && actual.ignoreCase === expected.ignoreCase;

	            // 7.4. Other pairs that do not both pass typeof value == 'object',
	            // equivalence is determined by ==.
	          } else if (!util.isObject(actual) && !util.isObject(expected)) {
	            return actual == expected;

	            // 7.5 For all other Object pairs, including Array objects, equivalence is
	            // determined by having the same number of owned properties (as verified
	            // with Object.prototype.hasOwnProperty.call), the same set of keys
	            // (although not necessarily the same order), equivalent values for every
	            // corresponding key, and an identical 'prototype' property. Note: this
	            // accounts for both named and indexed properties on Arrays.
	          } else {
	            return objEquiv(actual, expected);
	          }
	        }
	        function isArguments(object) {
	          return Object.prototype.toString.call(object) == '[object Arguments]';
	        }
	        function objEquiv(a, b) {
	          if (util.isNullOrUndefined(a) || util.isNullOrUndefined(b)) return false;
	          // an identical 'prototype' property.
	          if (a.prototype !== b.prototype) return false;
	          // if one is a primitive, the other must be same
	          if (util.isPrimitive(a) || util.isPrimitive(b)) {
	            return a === b;
	          }
	          var aIsArgs = isArguments(a),
	            bIsArgs = isArguments(b);
	          if (aIsArgs && !bIsArgs || !aIsArgs && bIsArgs) return false;
	          if (aIsArgs) {
	            a = pSlice.call(a);
	            b = pSlice.call(b);
	            return _deepEqual(a, b);
	          }
	          var ka = objectKeys(a),
	            kb = objectKeys(b),
	            key,
	            i;
	          // having the same number of owned properties (keys incorporates
	          // hasOwnProperty)
	          if (ka.length != kb.length) return false;
	          //the same set of keys (although not necessarily the same order),
	          ka.sort();
	          kb.sort();
	          //~~~cheap key test
	          for (i = ka.length - 1; i >= 0; i--) {
	            if (ka[i] != kb[i]) return false;
	          }
	          //equivalent values for every corresponding key, and
	          //~~~possibly expensive deep test
	          for (i = ka.length - 1; i >= 0; i--) {
	            key = ka[i];
	            if (!_deepEqual(a[key], b[key])) return false;
	          }
	          return true;
	        }

	        // 8. The non-equivalence assertion tests for any deep inequality.
	        // assert.notDeepEqual(actual, expected, message_opt);

	        assert.notDeepEqual = function notDeepEqual(actual, expected, message) {
	          if (_deepEqual(actual, expected)) {
	            fail(actual, expected, message, 'notDeepEqual', assert.notDeepEqual);
	          }
	        };

	        // 9. The strict equality assertion tests strict equality, as determined by ===.
	        // assert.strictEqual(actual, expected, message_opt);

	        assert.strictEqual = function strictEqual(actual, expected, message) {
	          if (actual !== expected) {
	            fail(actual, expected, message, '===', assert.strictEqual);
	          }
	        };

	        // 10. The strict non-equality assertion tests for strict inequality, as
	        // determined by !==.  assert.notStrictEqual(actual, expected, message_opt);

	        assert.notStrictEqual = function notStrictEqual(actual, expected, message) {
	          if (actual === expected) {
	            fail(actual, expected, message, '!==', assert.notStrictEqual);
	          }
	        };
	        function expectedException(actual, expected) {
	          if (!actual || !expected) {
	            return false;
	          }
	          if (Object.prototype.toString.call(expected) == '[object RegExp]') {
	            return expected.test(actual);
	          } else if (actual instanceof expected) {
	            return true;
	          } else if (expected.call({}, actual) === true) {
	            return true;
	          }
	          return false;
	        }
	        function _throws(shouldThrow, block, expected, message) {
	          var actual;
	          if (util.isString(expected)) {
	            message = expected;
	            expected = null;
	          }
	          try {
	            block();
	          } catch (e) {
	            actual = e;
	          }
	          message = (expected && expected.name ? ' (' + expected.name + ').' : '.') + (message ? ' ' + message : '.');
	          if (shouldThrow && !actual) {
	            fail(actual, expected, 'Missing expected exception' + message);
	          }
	          if (!shouldThrow && expectedException(actual, expected)) {
	            fail(actual, expected, 'Got unwanted exception' + message);
	          }
	          if (shouldThrow && actual && expected && !expectedException(actual, expected) || !shouldThrow && actual) {
	            throw actual;
	          }
	        }

	        // 11. Expected to throw an error:
	        // assert.throws(block, Error_opt, message_opt);

	        assert.throws = function (block, /*optional*/error, /*optional*/message) {
	          _throws.apply(this, [true].concat(pSlice.call(arguments)));
	        };

	        // EXTENSION! This is annoying to write outside this module.
	        assert.doesNotThrow = function (block, /*optional*/message) {
	          _throws.apply(this, [false].concat(pSlice.call(arguments)));
	        };
	        assert.ifError = function (err) {
	          if (err) {
	            throw err;
	          }
	        };
	        var objectKeys = Object.keys || function (obj) {
	          var keys = [];
	          for (var key in obj) {
	            if (hasOwn.call(obj, key)) keys.push(key);
	          }
	          return keys;
	        };
	      }, {
	        "util/": 11
	      }],
	      9: [function (require, module, exports) {
	        if (typeof Object.create === 'function') {
	          // implementation from standard node.js 'util' module
	          module.exports = function inherits(ctor, superCtor) {
	            ctor.super_ = superCtor;
	            ctor.prototype = Object.create(superCtor.prototype, {
	              constructor: {
	                value: ctor,
	                enumerable: false,
	                writable: true,
	                configurable: true
	              }
	            });
	          };
	        } else {
	          // old school shim for old browsers
	          module.exports = function inherits(ctor, superCtor) {
	            ctor.super_ = superCtor;
	            var TempCtor = function () {};
	            TempCtor.prototype = superCtor.prototype;
	            ctor.prototype = new TempCtor();
	            ctor.prototype.constructor = ctor;
	          };
	        }
	      }, {}],
	      10: [function (require, module, exports) {
	        module.exports = function isBuffer(arg) {
	          return arg && typeof arg === 'object' && typeof arg.copy === 'function' && typeof arg.fill === 'function' && typeof arg.readUInt8 === 'function';
	        };
	      }, {}],
	      11: [function (require, module, exports) {
	        (function (process, global) {
	          // Copyright Joyent, Inc. and other Node contributors.
	          //
	          // Permission is hereby granted, free of charge, to any person obtaining a
	          // copy of this software and associated documentation files (the
	          // "Software"), to deal in the Software without restriction, including
	          // without limitation the rights to use, copy, modify, merge, publish,
	          // distribute, sublicense, and/or sell copies of the Software, and to permit
	          // persons to whom the Software is furnished to do so, subject to the
	          // following conditions:
	          //
	          // The above copyright notice and this permission notice shall be included
	          // in all copies or substantial portions of the Software.
	          //
	          // THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
	          // OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
	          // MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
	          // NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
	          // DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
	          // OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
	          // USE OR OTHER DEALINGS IN THE SOFTWARE.

	          var formatRegExp = /%[sdj%]/g;
	          exports.format = function (f) {
	            if (!isString(f)) {
	              var objects = [];
	              for (var i = 0; i < arguments.length; i++) {
	                objects.push(inspect(arguments[i]));
	              }
	              return objects.join(' ');
	            }
	            var i = 1;
	            var args = arguments;
	            var len = args.length;
	            var str = String(f).replace(formatRegExp, function (x) {
	              if (x === '%%') return '%';
	              if (i >= len) return x;
	              switch (x) {
	                case '%s':
	                  return String(args[i++]);
	                case '%d':
	                  return Number(args[i++]);
	                case '%j':
	                  try {
	                    return JSON.stringify(args[i++]);
	                  } catch (_) {
	                    return '[Circular]';
	                  }
	                default:
	                  return x;
	              }
	            });
	            for (var x = args[i]; i < len; x = args[++i]) {
	              if (isNull(x) || !isObject(x)) {
	                str += ' ' + x;
	              } else {
	                str += ' ' + inspect(x);
	              }
	            }
	            return str;
	          };

	          // Mark that a method should not be used.
	          // Returns a modified function which warns once by default.
	          // If --no-deprecation is set, then it is a no-op.
	          exports.deprecate = function (fn, msg) {
	            // Allow for deprecating things in the process of starting up.
	            if (isUndefined(global.process)) {
	              return function () {
	                return exports.deprecate(fn, msg).apply(this, arguments);
	              };
	            }
	            if (process.noDeprecation === true) {
	              return fn;
	            }
	            var warned = false;
	            function deprecated() {
	              if (!warned) {
	                if (process.throwDeprecation) {
	                  throw new Error(msg);
	                } else if (process.traceDeprecation) {
	                  console.trace(msg);
	                } else {
	                  console.error(msg);
	                }
	                warned = true;
	              }
	              return fn.apply(this, arguments);
	            }
	            return deprecated;
	          };
	          var debugs = {};
	          var debugEnviron;
	          exports.debuglog = function (set) {
	            if (isUndefined(debugEnviron)) debugEnviron = process.env.NODE_DEBUG || '';
	            set = set.toUpperCase();
	            if (!debugs[set]) {
	              if (new RegExp('\\b' + set + '\\b', 'i').test(debugEnviron)) {
	                var pid = process.pid;
	                debugs[set] = function () {
	                  var msg = exports.format.apply(exports, arguments);
	                  console.error('%s %d: %s', set, pid, msg);
	                };
	              } else {
	                debugs[set] = function () {};
	              }
	            }
	            return debugs[set];
	          };

	          /**
	           * Echos the value of a value. Trys to print the value out
	           * in the best way possible given the different types.
	           *
	           * @param {Object} obj The object to print out.
	           * @param {Object} opts Optional options object that alters the output.
	           */
	          /* legacy: obj, showHidden, depth, colors*/
	          function inspect(obj, opts) {
	            // default options
	            var ctx = {
	              seen: [],
	              stylize: stylizeNoColor
	            };
	            // legacy...
	            if (arguments.length >= 3) ctx.depth = arguments[2];
	            if (arguments.length >= 4) ctx.colors = arguments[3];
	            if (isBoolean(opts)) {
	              // legacy...
	              ctx.showHidden = opts;
	            } else if (opts) {
	              // got an "options" object
	              exports._extend(ctx, opts);
	            }
	            // set default options
	            if (isUndefined(ctx.showHidden)) ctx.showHidden = false;
	            if (isUndefined(ctx.depth)) ctx.depth = 2;
	            if (isUndefined(ctx.colors)) ctx.colors = false;
	            if (isUndefined(ctx.customInspect)) ctx.customInspect = true;
	            if (ctx.colors) ctx.stylize = stylizeWithColor;
	            return formatValue(ctx, obj, ctx.depth);
	          }
	          exports.inspect = inspect;

	          // http://en.wikipedia.org/wiki/ANSI_escape_code#graphics
	          inspect.colors = {
	            'bold': [1, 22],
	            'italic': [3, 23],
	            'underline': [4, 24],
	            'inverse': [7, 27],
	            'white': [37, 39],
	            'grey': [90, 39],
	            'black': [30, 39],
	            'blue': [34, 39],
	            'cyan': [36, 39],
	            'green': [32, 39],
	            'magenta': [35, 39],
	            'red': [31, 39],
	            'yellow': [33, 39]
	          };

	          // Don't use 'blue' not visible on cmd.exe
	          inspect.styles = {
	            'special': 'cyan',
	            'number': 'yellow',
	            'boolean': 'yellow',
	            'undefined': 'grey',
	            'null': 'bold',
	            'string': 'green',
	            'date': 'magenta',
	            // "name": intentionally not styling
	            'regexp': 'red'
	          };
	          function stylizeWithColor(str, styleType) {
	            var style = inspect.styles[styleType];
	            if (style) {
	              return '\u001b[' + inspect.colors[style][0] + 'm' + str + '\u001b[' + inspect.colors[style][1] + 'm';
	            } else {
	              return str;
	            }
	          }
	          function stylizeNoColor(str, styleType) {
	            return str;
	          }
	          function arrayToHash(array) {
	            var hash = {};
	            array.forEach(function (val, idx) {
	              hash[val] = true;
	            });
	            return hash;
	          }
	          function formatValue(ctx, value, recurseTimes) {
	            // Provide a hook for user-specified inspect functions.
	            // Check that value is an object with an inspect function on it
	            if (ctx.customInspect && value && isFunction(value.inspect) &&
	            // Filter out the util module, it's inspect function is special
	            value.inspect !== exports.inspect &&
	            // Also filter out any prototype objects using the circular check.
	            !(value.constructor && value.constructor.prototype === value)) {
	              var ret = value.inspect(recurseTimes, ctx);
	              if (!isString(ret)) {
	                ret = formatValue(ctx, ret, recurseTimes);
	              }
	              return ret;
	            }

	            // Primitive types cannot have properties
	            var primitive = formatPrimitive(ctx, value);
	            if (primitive) {
	              return primitive;
	            }

	            // Look up the keys of the object.
	            var keys = Object.keys(value);
	            var visibleKeys = arrayToHash(keys);
	            if (ctx.showHidden) {
	              keys = Object.getOwnPropertyNames(value);
	            }

	            // IE doesn't make error fields non-enumerable
	            // http://msdn.microsoft.com/en-us/library/ie/dww52sbt(v=vs.94).aspx
	            if (isError(value) && (keys.indexOf('message') >= 0 || keys.indexOf('description') >= 0)) {
	              return formatError(value);
	            }

	            // Some type of object without properties can be shortcutted.
	            if (keys.length === 0) {
	              if (isFunction(value)) {
	                var name = value.name ? ': ' + value.name : '';
	                return ctx.stylize('[Function' + name + ']', 'special');
	              }
	              if (isRegExp(value)) {
	                return ctx.stylize(RegExp.prototype.toString.call(value), 'regexp');
	              }
	              if (isDate(value)) {
	                return ctx.stylize(Date.prototype.toString.call(value), 'date');
	              }
	              if (isError(value)) {
	                return formatError(value);
	              }
	            }
	            var base = '',
	              array = false,
	              braces = ['{', '}'];

	            // Make Array say that they are Array
	            if (isArray(value)) {
	              array = true;
	              braces = ['[', ']'];
	            }

	            // Make functions say that they are functions
	            if (isFunction(value)) {
	              var n = value.name ? ': ' + value.name : '';
	              base = ' [Function' + n + ']';
	            }

	            // Make RegExps say that they are RegExps
	            if (isRegExp(value)) {
	              base = ' ' + RegExp.prototype.toString.call(value);
	            }

	            // Make dates with properties first say the date
	            if (isDate(value)) {
	              base = ' ' + Date.prototype.toUTCString.call(value);
	            }

	            // Make error with message first say the error
	            if (isError(value)) {
	              base = ' ' + formatError(value);
	            }
	            if (keys.length === 0 && (!array || value.length == 0)) {
	              return braces[0] + base + braces[1];
	            }
	            if (recurseTimes < 0) {
	              if (isRegExp(value)) {
	                return ctx.stylize(RegExp.prototype.toString.call(value), 'regexp');
	              } else {
	                return ctx.stylize('[Object]', 'special');
	              }
	            }
	            ctx.seen.push(value);
	            var output;
	            if (array) {
	              output = formatArray(ctx, value, recurseTimes, visibleKeys, keys);
	            } else {
	              output = keys.map(function (key) {
	                return formatProperty(ctx, value, recurseTimes, visibleKeys, key, array);
	              });
	            }
	            ctx.seen.pop();
	            return reduceToSingleString(output, base, braces);
	          }
	          function formatPrimitive(ctx, value) {
	            if (isUndefined(value)) return ctx.stylize('undefined', 'undefined');
	            if (isString(value)) {
	              var simple = '\'' + JSON.stringify(value).replace(/^"|"$/g, '').replace(/'/g, "\\'").replace(/\\"/g, '"') + '\'';
	              return ctx.stylize(simple, 'string');
	            }
	            if (isNumber(value)) return ctx.stylize('' + value, 'number');
	            if (isBoolean(value)) return ctx.stylize('' + value, 'boolean');
	            // For some reason typeof null is "object", so special case here.
	            if (isNull(value)) return ctx.stylize('null', 'null');
	          }
	          function formatError(value) {
	            return '[' + Error.prototype.toString.call(value) + ']';
	          }
	          function formatArray(ctx, value, recurseTimes, visibleKeys, keys) {
	            var output = [];
	            for (var i = 0, l = value.length; i < l; ++i) {
	              if (hasOwnProperty(value, String(i))) {
	                output.push(formatProperty(ctx, value, recurseTimes, visibleKeys, String(i), true));
	              } else {
	                output.push('');
	              }
	            }
	            keys.forEach(function (key) {
	              if (!key.match(/^\d+$/)) {
	                output.push(formatProperty(ctx, value, recurseTimes, visibleKeys, key, true));
	              }
	            });
	            return output;
	          }
	          function formatProperty(ctx, value, recurseTimes, visibleKeys, key, array) {
	            var name, str, desc;
	            desc = Object.getOwnPropertyDescriptor(value, key) || {
	              value: value[key]
	            };
	            if (desc.get) {
	              if (desc.set) {
	                str = ctx.stylize('[Getter/Setter]', 'special');
	              } else {
	                str = ctx.stylize('[Getter]', 'special');
	              }
	            } else {
	              if (desc.set) {
	                str = ctx.stylize('[Setter]', 'special');
	              }
	            }
	            if (!hasOwnProperty(visibleKeys, key)) {
	              name = '[' + key + ']';
	            }
	            if (!str) {
	              if (ctx.seen.indexOf(desc.value) < 0) {
	                if (isNull(recurseTimes)) {
	                  str = formatValue(ctx, desc.value, null);
	                } else {
	                  str = formatValue(ctx, desc.value, recurseTimes - 1);
	                }
	                if (str.indexOf('\n') > -1) {
	                  if (array) {
	                    str = str.split('\n').map(function (line) {
	                      return '  ' + line;
	                    }).join('\n').substr(2);
	                  } else {
	                    str = '\n' + str.split('\n').map(function (line) {
	                      return '   ' + line;
	                    }).join('\n');
	                  }
	                }
	              } else {
	                str = ctx.stylize('[Circular]', 'special');
	              }
	            }
	            if (isUndefined(name)) {
	              if (array && key.match(/^\d+$/)) {
	                return str;
	              }
	              name = JSON.stringify('' + key);
	              if (name.match(/^"([a-zA-Z_][a-zA-Z_0-9]*)"$/)) {
	                name = name.substr(1, name.length - 2);
	                name = ctx.stylize(name, 'name');
	              } else {
	                name = name.replace(/'/g, "\\'").replace(/\\"/g, '"').replace(/(^"|"$)/g, "'");
	                name = ctx.stylize(name, 'string');
	              }
	            }
	            return name + ': ' + str;
	          }
	          function reduceToSingleString(output, base, braces) {
	            var length = output.reduce(function (prev, cur) {
	              if (cur.indexOf('\n') >= 0) ;
	              return prev + cur.replace(/\u001b\[\d\d?m/g, '').length + 1;
	            }, 0);
	            if (length > 60) {
	              return braces[0] + (base === '' ? '' : base + '\n ') + ' ' + output.join(',\n  ') + ' ' + braces[1];
	            }
	            return braces[0] + base + ' ' + output.join(', ') + ' ' + braces[1];
	          }

	          // NOTE: These type checking functions intentionally don't use `instanceof`
	          // because it is fragile and can be easily faked with `Object.create()`.
	          function isArray(ar) {
	            return Array.isArray(ar);
	          }
	          exports.isArray = isArray;
	          function isBoolean(arg) {
	            return typeof arg === 'boolean';
	          }
	          exports.isBoolean = isBoolean;
	          function isNull(arg) {
	            return arg === null;
	          }
	          exports.isNull = isNull;
	          function isNullOrUndefined(arg) {
	            return arg == null;
	          }
	          exports.isNullOrUndefined = isNullOrUndefined;
	          function isNumber(arg) {
	            return typeof arg === 'number';
	          }
	          exports.isNumber = isNumber;
	          function isString(arg) {
	            return typeof arg === 'string';
	          }
	          exports.isString = isString;
	          function isSymbol(arg) {
	            return typeof arg === 'symbol';
	          }
	          exports.isSymbol = isSymbol;
	          function isUndefined(arg) {
	            return arg === void 0;
	          }
	          exports.isUndefined = isUndefined;
	          function isRegExp(re) {
	            return isObject(re) && objectToString(re) === '[object RegExp]';
	          }
	          exports.isRegExp = isRegExp;
	          function isObject(arg) {
	            return typeof arg === 'object' && arg !== null;
	          }
	          exports.isObject = isObject;
	          function isDate(d) {
	            return isObject(d) && objectToString(d) === '[object Date]';
	          }
	          exports.isDate = isDate;
	          function isError(e) {
	            return isObject(e) && (objectToString(e) === '[object Error]' || e instanceof Error);
	          }
	          exports.isError = isError;
	          function isFunction(arg) {
	            return typeof arg === 'function';
	          }
	          exports.isFunction = isFunction;
	          function isPrimitive(arg) {
	            return arg === null || typeof arg === 'boolean' || typeof arg === 'number' || typeof arg === 'string' || typeof arg === 'symbol' ||
	            // ES6 symbol
	            typeof arg === 'undefined';
	          }
	          exports.isPrimitive = isPrimitive;
	          exports.isBuffer = require('./support/isBuffer');
	          function objectToString(o) {
	            return Object.prototype.toString.call(o);
	          }
	          function pad(n) {
	            return n < 10 ? '0' + n.toString(10) : n.toString(10);
	          }
	          var months = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];

	          // 26 Feb 16:19:34
	          function timestamp() {
	            var d = new Date();
	            var time = [pad(d.getHours()), pad(d.getMinutes()), pad(d.getSeconds())].join(':');
	            return [d.getDate(), months[d.getMonth()], time].join(' ');
	          }

	          // log is just a thin wrapper to console.log that prepends a timestamp
	          exports.log = function () {
	            console.log('%s - %s', timestamp(), exports.format.apply(exports, arguments));
	          };

	          /**
	           * Inherit the prototype methods from one constructor into another.
	           *
	           * The Function.prototype.inherits from lang.js rewritten as a standalone
	           * function (not on Function.prototype). NOTE: If this file is to be loaded
	           * during bootstrapping this function needs to be rewritten using some native
	           * functions as prototype setup using normal JavaScript does not work as
	           * expected during bootstrapping (see mirror.js in r114903).
	           *
	           * @param {function} ctor Constructor function which needs to inherit the
	           *     prototype.
	           * @param {function} superCtor Constructor function to inherit prototype from.
	           */
	          exports.inherits = require('inherits');
	          exports._extend = function (origin, add) {
	            // Don't do anything if add isn't an object
	            if (!add || !isObject(add)) return origin;
	            var keys = Object.keys(add);
	            var i = keys.length;
	            while (i--) {
	              origin[keys[i]] = add[keys[i]];
	            }
	            return origin;
	          };
	          function hasOwnProperty(obj, prop) {
	            return Object.prototype.hasOwnProperty.call(obj, prop);
	          }
	        }).call(this, require('_process'), typeof commonjsGlobal !== "undefined" ? commonjsGlobal : typeof self !== "undefined" ? self : typeof window !== "undefined" ? window : {});
	      }, {
	        "./support/isBuffer": 10,
	        "_process": 14,
	        "inherits": 9
	      }],
	      12: [function (require, module, exports) {}, {}],
	      13: [function (require, module, exports) {
	        (function (process) {
	          // .dirname, .basename, and .extname methods are extracted from Node.js v8.11.1,
	          // backported and transplited with Babel, with backwards-compat fixes

	          // Copyright Joyent, Inc. and other Node contributors.
	          //
	          // Permission is hereby granted, free of charge, to any person obtaining a
	          // copy of this software and associated documentation files (the
	          // "Software"), to deal in the Software without restriction, including
	          // without limitation the rights to use, copy, modify, merge, publish,
	          // distribute, sublicense, and/or sell copies of the Software, and to permit
	          // persons to whom the Software is furnished to do so, subject to the
	          // following conditions:
	          //
	          // The above copyright notice and this permission notice shall be included
	          // in all copies or substantial portions of the Software.
	          //
	          // THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
	          // OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
	          // MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
	          // NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
	          // DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
	          // OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
	          // USE OR OTHER DEALINGS IN THE SOFTWARE.

	          // resolves . and .. elements in a path array with directory names there
	          // must be no slashes, empty elements, or device names (c:\) in the array
	          // (so also no leading and trailing slashes - it does not distinguish
	          // relative and absolute paths)
	          function normalizeArray(parts, allowAboveRoot) {
	            // if the path tries to go above the root, `up` ends up > 0
	            var up = 0;
	            for (var i = parts.length - 1; i >= 0; i--) {
	              var last = parts[i];
	              if (last === '.') {
	                parts.splice(i, 1);
	              } else if (last === '..') {
	                parts.splice(i, 1);
	                up++;
	              } else if (up) {
	                parts.splice(i, 1);
	                up--;
	              }
	            }

	            // if the path is allowed to go above the root, restore leading ..s
	            if (allowAboveRoot) {
	              for (; up--; up) {
	                parts.unshift('..');
	              }
	            }
	            return parts;
	          }

	          // path.resolve([from ...], to)
	          // posix version
	          exports.resolve = function () {
	            var resolvedPath = '',
	              resolvedAbsolute = false;
	            for (var i = arguments.length - 1; i >= -1 && !resolvedAbsolute; i--) {
	              var path = i >= 0 ? arguments[i] : process.cwd();

	              // Skip empty and invalid entries
	              if (typeof path !== 'string') {
	                throw new TypeError('Arguments to path.resolve must be strings');
	              } else if (!path) {
	                continue;
	              }
	              resolvedPath = path + '/' + resolvedPath;
	              resolvedAbsolute = path.charAt(0) === '/';
	            }

	            // At this point the path should be resolved to a full absolute path, but
	            // handle relative paths to be safe (might happen when process.cwd() fails)

	            // Normalize the path
	            resolvedPath = normalizeArray(filter(resolvedPath.split('/'), function (p) {
	              return !!p;
	            }), !resolvedAbsolute).join('/');
	            return (resolvedAbsolute ? '/' : '') + resolvedPath || '.';
	          };

	          // path.normalize(path)
	          // posix version
	          exports.normalize = function (path) {
	            var isAbsolute = exports.isAbsolute(path),
	              trailingSlash = substr(path, -1) === '/';

	            // Normalize the path
	            path = normalizeArray(filter(path.split('/'), function (p) {
	              return !!p;
	            }), !isAbsolute).join('/');
	            if (!path && !isAbsolute) {
	              path = '.';
	            }
	            if (path && trailingSlash) {
	              path += '/';
	            }
	            return (isAbsolute ? '/' : '') + path;
	          };

	          // posix version
	          exports.isAbsolute = function (path) {
	            return path.charAt(0) === '/';
	          };

	          // posix version
	          exports.join = function () {
	            var paths = Array.prototype.slice.call(arguments, 0);
	            return exports.normalize(filter(paths, function (p, index) {
	              if (typeof p !== 'string') {
	                throw new TypeError('Arguments to path.join must be strings');
	              }
	              return p;
	            }).join('/'));
	          };

	          // path.relative(from, to)
	          // posix version
	          exports.relative = function (from, to) {
	            from = exports.resolve(from).substr(1);
	            to = exports.resolve(to).substr(1);
	            function trim(arr) {
	              var start = 0;
	              for (; start < arr.length; start++) {
	                if (arr[start] !== '') break;
	              }
	              var end = arr.length - 1;
	              for (; end >= 0; end--) {
	                if (arr[end] !== '') break;
	              }
	              if (start > end) return [];
	              return arr.slice(start, end - start + 1);
	            }
	            var fromParts = trim(from.split('/'));
	            var toParts = trim(to.split('/'));
	            var length = Math.min(fromParts.length, toParts.length);
	            var samePartsLength = length;
	            for (var i = 0; i < length; i++) {
	              if (fromParts[i] !== toParts[i]) {
	                samePartsLength = i;
	                break;
	              }
	            }
	            var outputParts = [];
	            for (var i = samePartsLength; i < fromParts.length; i++) {
	              outputParts.push('..');
	            }
	            outputParts = outputParts.concat(toParts.slice(samePartsLength));
	            return outputParts.join('/');
	          };
	          exports.sep = '/';
	          exports.delimiter = ':';
	          exports.dirname = function (path) {
	            if (typeof path !== 'string') path = path + '';
	            if (path.length === 0) return '.';
	            var code = path.charCodeAt(0);
	            var hasRoot = code === 47 /*/*/;
	            var end = -1;
	            var matchedSlash = true;
	            for (var i = path.length - 1; i >= 1; --i) {
	              code = path.charCodeAt(i);
	              if (code === 47 /*/*/) {
	                if (!matchedSlash) {
	                  end = i;
	                  break;
	                }
	              } else {
	                // We saw the first non-path separator
	                matchedSlash = false;
	              }
	            }
	            if (end === -1) return hasRoot ? '/' : '.';
	            if (hasRoot && end === 1) {
	              // return '//';
	              // Backwards-compat fix:
	              return '/';
	            }
	            return path.slice(0, end);
	          };
	          function basename(path) {
	            if (typeof path !== 'string') path = path + '';
	            var start = 0;
	            var end = -1;
	            var matchedSlash = true;
	            var i;
	            for (i = path.length - 1; i >= 0; --i) {
	              if (path.charCodeAt(i) === 47 /*/*/) {
	                // If we reached a path separator that was not part of a set of path
	                // separators at the end of the string, stop now
	                if (!matchedSlash) {
	                  start = i + 1;
	                  break;
	                }
	              } else if (end === -1) {
	                // We saw the first non-path separator, mark this as the end of our
	                // path component
	                matchedSlash = false;
	                end = i + 1;
	              }
	            }
	            if (end === -1) return '';
	            return path.slice(start, end);
	          }

	          // Uses a mixed approach for backwards-compatibility, as ext behavior changed
	          // in new Node.js versions, so only basename() above is backported here
	          exports.basename = function (path, ext) {
	            var f = basename(path);
	            if (ext && f.substr(-1 * ext.length) === ext) {
	              f = f.substr(0, f.length - ext.length);
	            }
	            return f;
	          };
	          exports.extname = function (path) {
	            if (typeof path !== 'string') path = path + '';
	            var startDot = -1;
	            var startPart = 0;
	            var end = -1;
	            var matchedSlash = true;
	            // Track the state of characters (if any) we see before our first dot and
	            // after any path separator we find
	            var preDotState = 0;
	            for (var i = path.length - 1; i >= 0; --i) {
	              var code = path.charCodeAt(i);
	              if (code === 47 /*/*/) {
	                // If we reached a path separator that was not part of a set of path
	                // separators at the end of the string, stop now
	                if (!matchedSlash) {
	                  startPart = i + 1;
	                  break;
	                }
	                continue;
	              }
	              if (end === -1) {
	                // We saw the first non-path separator, mark this as the end of our
	                // extension
	                matchedSlash = false;
	                end = i + 1;
	              }
	              if (code === 46 /*.*/) {
	                // If this is our first dot, mark it as the start of our extension
	                if (startDot === -1) startDot = i;else if (preDotState !== 1) preDotState = 1;
	              } else if (startDot !== -1) {
	                // We saw a non-dot and non-path separator before our dot, so we should
	                // have a good chance at having a non-empty extension
	                preDotState = -1;
	              }
	            }
	            if (startDot === -1 || end === -1 ||
	            // We saw a non-dot character immediately before the dot
	            preDotState === 0 ||
	            // The (right-most) trimmed path component is exactly '..'
	            preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
	              return '';
	            }
	            return path.slice(startDot, end);
	          };
	          function filter(xs, f) {
	            if (xs.filter) return xs.filter(f);
	            var res = [];
	            for (var i = 0; i < xs.length; i++) {
	              if (f(xs[i], i, xs)) res.push(xs[i]);
	            }
	            return res;
	          }

	          // String.prototype.substr - negative index don't work in IE8
	          var substr = 'ab'.substr(-1) === 'b' ? function (str, start, len) {
	            return str.substr(start, len);
	          } : function (str, start, len) {
	            if (start < 0) start = str.length + start;
	            return str.substr(start, len);
	          };
	        }).call(this, require('_process'));
	      }, {
	        "_process": 14
	      }],
	      14: [function (require, module, exports) {
	        // shim for using process in browser
	        var process = module.exports = {};

	        // cached from whatever global is present so that test runners that stub it
	        // don't break things.  But we need to wrap it in a try catch in case it is
	        // wrapped in strict mode code which doesn't define any globals.  It's inside a
	        // function because try/catches deoptimize in certain engines.

	        var cachedSetTimeout;
	        var cachedClearTimeout;
	        function defaultSetTimout() {
	          throw new Error('setTimeout has not been defined');
	        }
	        function defaultClearTimeout() {
	          throw new Error('clearTimeout has not been defined');
	        }
	        (function () {
	          try {
	            if (typeof setTimeout === 'function') {
	              cachedSetTimeout = setTimeout;
	            } else {
	              cachedSetTimeout = defaultSetTimout;
	            }
	          } catch (e) {
	            cachedSetTimeout = defaultSetTimout;
	          }
	          try {
	            if (typeof clearTimeout === 'function') {
	              cachedClearTimeout = clearTimeout;
	            } else {
	              cachedClearTimeout = defaultClearTimeout;
	            }
	          } catch (e) {
	            cachedClearTimeout = defaultClearTimeout;
	          }
	        })();
	        function runTimeout(fun) {
	          if (cachedSetTimeout === setTimeout) {
	            //normal enviroments in sane situations
	            return setTimeout(fun, 0);
	          }
	          // if setTimeout wasn't available but was latter defined
	          if ((cachedSetTimeout === defaultSetTimout || !cachedSetTimeout) && setTimeout) {
	            cachedSetTimeout = setTimeout;
	            return setTimeout(fun, 0);
	          }
	          try {
	            // when when somebody has screwed with setTimeout but no I.E. maddness
	            return cachedSetTimeout(fun, 0);
	          } catch (e) {
	            try {
	              // When we are in I.E. but the script has been evaled so I.E. doesn't trust the global object when called normally
	              return cachedSetTimeout.call(null, fun, 0);
	            } catch (e) {
	              // same as above but when it's a version of I.E. that must have the global object for 'this', hopfully our context correct otherwise it will throw a global error
	              return cachedSetTimeout.call(this, fun, 0);
	            }
	          }
	        }
	        function runClearTimeout(marker) {
	          if (cachedClearTimeout === clearTimeout) {
	            //normal enviroments in sane situations
	            return clearTimeout(marker);
	          }
	          // if clearTimeout wasn't available but was latter defined
	          if ((cachedClearTimeout === defaultClearTimeout || !cachedClearTimeout) && clearTimeout) {
	            cachedClearTimeout = clearTimeout;
	            return clearTimeout(marker);
	          }
	          try {
	            // when when somebody has screwed with setTimeout but no I.E. maddness
	            return cachedClearTimeout(marker);
	          } catch (e) {
	            try {
	              // When we are in I.E. but the script has been evaled so I.E. doesn't  trust the global object when called normally
	              return cachedClearTimeout.call(null, marker);
	            } catch (e) {
	              // same as above but when it's a version of I.E. that must have the global object for 'this', hopfully our context correct otherwise it will throw a global error.
	              // Some versions of I.E. have different rules for clearTimeout vs setTimeout
	              return cachedClearTimeout.call(this, marker);
	            }
	          }
	        }
	        var queue = [];
	        var draining = false;
	        var currentQueue;
	        var queueIndex = -1;
	        function cleanUpNextTick() {
	          if (!draining || !currentQueue) {
	            return;
	          }
	          draining = false;
	          if (currentQueue.length) {
	            queue = currentQueue.concat(queue);
	          } else {
	            queueIndex = -1;
	          }
	          if (queue.length) {
	            drainQueue();
	          }
	        }
	        function drainQueue() {
	          if (draining) {
	            return;
	          }
	          var timeout = runTimeout(cleanUpNextTick);
	          draining = true;
	          var len = queue.length;
	          while (len) {
	            currentQueue = queue;
	            queue = [];
	            while (++queueIndex < len) {
	              if (currentQueue) {
	                currentQueue[queueIndex].run();
	              }
	            }
	            queueIndex = -1;
	            len = queue.length;
	          }
	          currentQueue = null;
	          draining = false;
	          runClearTimeout(timeout);
	        }
	        process.nextTick = function (fun) {
	          var args = new Array(arguments.length - 1);
	          if (arguments.length > 1) {
	            for (var i = 1; i < arguments.length; i++) {
	              args[i - 1] = arguments[i];
	            }
	          }
	          queue.push(new Item(fun, args));
	          if (queue.length === 1 && !draining) {
	            runTimeout(drainQueue);
	          }
	        };

	        // v8 likes predictible objects
	        function Item(fun, array) {
	          this.fun = fun;
	          this.array = array;
	        }
	        Item.prototype.run = function () {
	          this.fun.apply(null, this.array);
	        };
	        process.title = 'browser';
	        process.browser = true;
	        process.env = {};
	        process.argv = [];
	        process.version = ''; // empty string to avoid regexp issues
	        process.versions = {};
	        function noop() {}
	        process.on = noop;
	        process.addListener = noop;
	        process.once = noop;
	        process.off = noop;
	        process.removeListener = noop;
	        process.removeAllListeners = noop;
	        process.emit = noop;
	        process.prependListener = noop;
	        process.prependOnceListener = noop;
	        process.listeners = function (name) {
	          return [];
	        };
	        process.binding = function (name) {
	          throw new Error('process.binding is not supported');
	        };
	        process.cwd = function () {
	          return '/';
	        };
	        process.chdir = function (dir) {
	          throw new Error('process.chdir is not supported');
	        };
	        process.umask = function () {
	          return 0;
	        };
	      }, {}],
	      15: [function (require, module, exports) {
	        var unparse = require('escodegen').generate;
	        module.exports = function (ast, vars) {
	          if (!vars) vars = {};
	          var FAIL = {};
	          var result = function walk(node, scopeVars) {
	            if (node.type === 'Literal') {
	              return node.value;
	            } else if (node.type === 'UnaryExpression') {
	              var val = walk(node.argument);
	              if (node.operator === '+') return +val;
	              if (node.operator === '-') return -val;
	              if (node.operator === '~') return ~val;
	              if (node.operator === '!') return !val;
	              return FAIL;
	            } else if (node.type === 'ArrayExpression') {
	              var xs = [];
	              for (var i = 0, l = node.elements.length; i < l; i++) {
	                var x = walk(node.elements[i]);
	                if (x === FAIL) return FAIL;
	                xs.push(x);
	              }
	              return xs;
	            } else if (node.type === 'ObjectExpression') {
	              var obj = {};
	              for (var i = 0; i < node.properties.length; i++) {
	                var prop = node.properties[i];
	                var value = prop.value === null ? prop.value : walk(prop.value);
	                if (value === FAIL) return FAIL;
	                obj[prop.key.value || prop.key.name] = value;
	              }
	              return obj;
	            } else if (node.type === 'BinaryExpression' || node.type === 'LogicalExpression') {
	              var l = walk(node.left);
	              if (l === FAIL) return FAIL;
	              var r = walk(node.right);
	              if (r === FAIL) return FAIL;
	              var op = node.operator;
	              if (op === '==') return l == r;
	              if (op === '===') return l === r;
	              if (op === '!=') return l != r;
	              if (op === '!==') return l !== r;
	              if (op === '+') return l + r;
	              if (op === '-') return l - r;
	              if (op === '*') return l * r;
	              if (op === '/') return l / r;
	              if (op === '%') return l % r;
	              if (op === '<') return l < r;
	              if (op === '<=') return l <= r;
	              if (op === '>') return l > r;
	              if (op === '>=') return l >= r;
	              if (op === '|') return l | r;
	              if (op === '&') return l & r;
	              if (op === '^') return l ^ r;
	              if (op === '&&') return l && r;
	              if (op === '||') return l || r;
	              return FAIL;
	            } else if (node.type === 'Identifier') {
	              if ({}.hasOwnProperty.call(vars, node.name)) {
	                return vars[node.name];
	              } else return FAIL;
	            } else if (node.type === 'ThisExpression') {
	              if ({}.hasOwnProperty.call(vars, 'this')) {
	                return vars['this'];
	              } else return FAIL;
	            } else if (node.type === 'CallExpression') {
	              var callee = walk(node.callee);
	              if (callee === FAIL) return FAIL;
	              if (typeof callee !== 'function') return FAIL;
	              var ctx = node.callee.object ? walk(node.callee.object) : FAIL;
	              if (ctx === FAIL) ctx = null;
	              var args = [];
	              for (var i = 0, l = node.arguments.length; i < l; i++) {
	                var x = walk(node.arguments[i]);
	                if (x === FAIL) return FAIL;
	                args.push(x);
	              }
	              return callee.apply(ctx, args);
	            } else if (node.type === 'MemberExpression') {
	              var obj = walk(node.object);
	              // do not allow access to methods on Function 
	              if (obj === FAIL || typeof obj == 'function') {
	                return FAIL;
	              }
	              if (node.property.type === 'Identifier') {
	                return obj[node.property.name];
	              }
	              var prop = walk(node.property);
	              if (prop === FAIL) return FAIL;
	              return obj[prop];
	            } else if (node.type === 'ConditionalExpression') {
	              var val = walk(node.test);
	              if (val === FAIL) return FAIL;
	              return val ? walk(node.consequent) : walk(node.alternate);
	            } else if (node.type === 'ExpressionStatement') {
	              var val = walk(node.expression);
	              if (val === FAIL) return FAIL;
	              return val;
	            } else if (node.type === 'ReturnStatement') {
	              return walk(node.argument);
	            } else if (node.type === 'FunctionExpression') {
	              var bodies = node.body.body;

	              // Create a "scope" for our arguments
	              var oldVars = {};
	              Object.keys(vars).forEach(function (element) {
	                oldVars[element] = vars[element];
	              });
	              for (var i = 0; i < node.params.length; i++) {
	                var key = node.params[i];
	                if (key.type == 'Identifier') {
	                  vars[key.name] = null;
	                } else return FAIL;
	              }
	              for (var i in bodies) {
	                if (walk(bodies[i]) === FAIL) {
	                  return FAIL;
	                }
	              }
	              // restore the vars and scope after we walk
	              vars = oldVars;
	              var keys = Object.keys(vars);
	              var vals = keys.map(function (key) {
	                return vars[key];
	              });
	              return Function(keys.join(', '), 'return ' + unparse(node)).apply(null, vals);
	            } else if (node.type === 'TemplateLiteral') {
	              var str = '';
	              for (var i = 0; i < node.expressions.length; i++) {
	                str += walk(node.quasis[i]);
	                str += walk(node.expressions[i]);
	              }
	              str += walk(node.quasis[i]);
	              return str;
	            } else if (node.type === 'TaggedTemplateExpression') {
	              var tag = walk(node.tag);
	              var quasi = node.quasi;
	              var strings = quasi.quasis.map(walk);
	              var values = quasi.expressions.map(walk);
	              return tag.apply(null, [strings].concat(values));
	            } else if (node.type === 'TemplateElement') {
	              return node.value.cooked;
	            } else return FAIL;
	          }(ast);
	          return result === FAIL ? undefined : result;
	        };
	      }, {
	        "escodegen": 12
	      }],
	      "jsonpath": [function (require, module, exports) {
	        module.exports = require('./lib/index');
	      }, {
	        "./lib/index": 5
	      }]
	    }, {}, ["jsonpath"])("jsonpath");
	  });
	})(jsonpath);
	var jp = jsonpath.exports;

	/**
	 * Returns the computed data section of a compound
	 *
	 * @param data Data of a compound data request to the PubChem API
	 * @returns Computed Data section
	 */
	function getComputedDataSection(data) {
	  const computedData = jp.query(data, '$.Section[?(@.TOCHeading==="Chemical and Physical Properties")]' + '.Section[?(@.TOCHeading==="Computed Properties")]');
	  return computedData;
	}

	/**
	 * Returns a computed data section of a compound
	 *
	 * @param data Data of a compound data request to the PubChem API
	 * @param sectionName Name of the section to be returned
	 * @returns Computed Data section
	 */
	function getComputedPropertySection(data, sectionName) {
	  const returnData = jp.query(data, `$.Section[?(@.TOCHeading==="${sectionName}")].Information[*]`)[0];
	  returnData.Description = jp.query(data, `$.Section[?(@.TOCHeading==="${sectionName}")].Description`)[0];
	  return returnData;
	}

	/**
	 * Returns the number properties of a compound
	 *
	 * @param data Data of a compound data request to the PubChem API
	 * @param sectionName Name of the section to get the data from
	 * @returns Computed Data
	 */
	function getNumberProperties(data, sectionName) {
	  const output = {
	    value: null,
	    label: null,
	    reference: {
	      description: null
	    },
	    description: null
	  };
	  try {
	    const computationalData = getComputedDataSection(data);
	    const section = getComputedPropertySection(computationalData[0], sectionName);
	    output.value = section.Value.Number[0];
	    output.label = sectionName;
	    output.description = section.Description;
	    output.reference.description = section.Reference[0];
	    if (section.Value.Unit) {
	      output.units = section.Value.Unit;
	    }
	    // eslint-disable-next-line no-empty
	  } catch (error) {}
	  return output;
	}

	/**
	 * Returns the complexity of a compound
	 *
	 * @param data Data of a compound data request to the PubChem API
	 * @returns ComputedData
	 */
	function getComplexity(data) {
	  const complexity = getNumberProperties(data, 'Complexity');
	  return complexity;
	}

	/**
	 * Returns the formal charge of a compound
	 *
	 * @param data Data of a compound data request to the PubChem API
	 * @returns Computed Data
	 */
	function getFormalCharge(data) {
	  const formalCharge = getNumberProperties(data, 'Formal Charge');
	  return formalCharge;
	}

	/**
	 * Returns the heavy atom count of a compound
	 *
	 * @param data Data of a compound data request to the PubChem API
	 * @returns Computed Data
	 */
	function getHeavyAtomCount(data) {
	  const heavyAtomCount = getNumberProperties(data, 'Heavy Atom Count');
	  return heavyAtomCount;
	}

	/**
	 * Returns the hydrogen bond acceptor count of a compound
	 *
	 * @param data Data of a compound data request to the PubChem API
	 * @returns Computed Data
	 */
	function getHydrogenBondAcceptorCount(data) {
	  const hydrogenBondAcceptorCount = getNumberProperties(data, 'Hydrogen Bond Acceptor Count');
	  return hydrogenBondAcceptorCount;
	}

	/**
	 * Returns the hydrogen bond donor count of a compound
	 *
	 * @param data Data of a compound data request to the PubChem API
	 * @returns ComputedData
	 */
	function getHydrogenBondDonorCount(data) {
	  const hydrogenBondDonorCount = getNumberProperties(data, 'Hydrogen Bond Donor Count');
	  return hydrogenBondDonorCount;
	}

	/**
	 * Returns the rotable bond count of a compound
	 *
	 * @param data Data of a compound data request to the PubChem API
	 * @returns ComputedData
	 */
	function getRotableBondCount(data) {
	  const rotableBondCount = getNumberProperties(data, 'Rotatable Bond Count');
	  return rotableBondCount;
	}

	/**
	 * Returns the topological polar surface area in angstrom^2
	 *
	 * @param data Data of a compound data request to the PubChem API
	 * @returns Computed Data
	 */
	function getTPSA(data) {
	  const tpsa = getNumberProperties(data, 'Topological Polar Surface Area');
	  return tpsa;
	}

	/**
	 * Returns the xLogP3 of a compound
	 *
	 * @param data Data of a compound data request to the PubChem API
	 * @returns Computed Data
	 */
	function getxLogP3(data) {
	  const xLogP3 = getNumberProperties(data, 'XLogP3');
	  return xLogP3;
	}

	/**
	 * Returns an object of computed data
	 *
	 * @param data Data of a compound data request to the PubChem API
	 * @returns Object of computed data
	 */
	function getComputedData(data) {
	  return {
	    hydrogenBondDonorCount: getHydrogenBondDonorCount(data),
	    hydrogenBondAcceptorCount: getHydrogenBondAcceptorCount(data),
	    rotableBondCount: getRotableBondCount(data),
	    heavyAtomCount: getHeavyAtomCount(data),
	    formalCharge: getFormalCharge(data),
	    complexity: getComplexity(data),
	    tpsa: getTPSA(data),
	    xLogP3: getxLogP3(data)
	  };
	}

	var isMergeableObject = function isMergeableObject(value) {
	  return isNonNullObject(value) && !isSpecial(value);
	};
	function isNonNullObject(value) {
	  return !!value && typeof value === 'object';
	}
	function isSpecial(value) {
	  var stringValue = Object.prototype.toString.call(value);
	  return stringValue === '[object RegExp]' || stringValue === '[object Date]' || isReactElement(value);
	}

	// see https://github.com/facebook/react/blob/b5ac963fb791d1298e7f396236383bc955f916c1/src/isomorphic/classic/element/ReactElement.js#L21-L25
	var canUseSymbol = typeof Symbol === 'function' && Symbol.for;
	var REACT_ELEMENT_TYPE = canUseSymbol ? Symbol.for('react.element') : 0xeac7;
	function isReactElement(value) {
	  return value.$$typeof === REACT_ELEMENT_TYPE;
	}
	function emptyTarget(val) {
	  return Array.isArray(val) ? [] : {};
	}
	function cloneUnlessOtherwiseSpecified(value, options) {
	  return options.clone !== false && options.isMergeableObject(value) ? deepmerge(emptyTarget(value), value, options) : value;
	}
	function defaultArrayMerge(target, source, options) {
	  return target.concat(source).map(function (element) {
	    return cloneUnlessOtherwiseSpecified(element, options);
	  });
	}
	function getMergeFunction(key, options) {
	  if (!options.customMerge) {
	    return deepmerge;
	  }
	  var customMerge = options.customMerge(key);
	  return typeof customMerge === 'function' ? customMerge : deepmerge;
	}
	function getEnumerableOwnPropertySymbols(target) {
	  return Object.getOwnPropertySymbols ? Object.getOwnPropertySymbols(target).filter(function (symbol) {
	    return Object.propertyIsEnumerable.call(target, symbol);
	  }) : [];
	}
	function getKeys(target) {
	  return Object.keys(target).concat(getEnumerableOwnPropertySymbols(target));
	}
	function propertyIsOnObject(object, property) {
	  try {
	    return property in object;
	  } catch (_) {
	    return false;
	  }
	}

	// Protects from prototype poisoning and unexpected merging up the prototype chain.
	function propertyIsUnsafe(target, key) {
	  return propertyIsOnObject(target, key) // Properties are safe to merge if they don't exist in the target yet,
	  && !(Object.hasOwnProperty.call(target, key) // unsafe if they exist up the prototype chain,
	  && Object.propertyIsEnumerable.call(target, key)); // and also unsafe if they're nonenumerable.
	}
	function mergeObject(target, source, options) {
	  var destination = {};
	  if (options.isMergeableObject(target)) {
	    getKeys(target).forEach(function (key) {
	      destination[key] = cloneUnlessOtherwiseSpecified(target[key], options);
	    });
	  }
	  getKeys(source).forEach(function (key) {
	    if (propertyIsUnsafe(target, key)) {
	      return;
	    }
	    if (propertyIsOnObject(target, key) && options.isMergeableObject(source[key])) {
	      destination[key] = getMergeFunction(key, options)(target[key], source[key], options);
	    } else {
	      destination[key] = cloneUnlessOtherwiseSpecified(source[key], options);
	    }
	  });
	  return destination;
	}
	function deepmerge(target, source, options) {
	  options = options || {};
	  options.arrayMerge = options.arrayMerge || defaultArrayMerge;
	  options.isMergeableObject = options.isMergeableObject || isMergeableObject;
	  // cloneUnlessOtherwiseSpecified is added to `options` so that custom arrayMerge()
	  // implementations can use it. The caller may not replace it.
	  options.cloneUnlessOtherwiseSpecified = cloneUnlessOtherwiseSpecified;
	  var sourceIsArray = Array.isArray(source);
	  var targetIsArray = Array.isArray(target);
	  var sourceAndTargetTypesMatch = sourceIsArray === targetIsArray;
	  if (!sourceAndTargetTypesMatch) {
	    return cloneUnlessOtherwiseSpecified(source, options);
	  } else if (sourceIsArray) {
	    return options.arrayMerge(target, source, options);
	  } else {
	    return mergeObject(target, source, options);
	  }
	}
	deepmerge.all = function deepmergeAll(array, options) {
	  if (!Array.isArray(array)) {
	    throw new Error('first argument should be an array');
	  }
	  return array.reduce(function (prev, next) {
	    return deepmerge(prev, next, options);
	  }, {});
	};
	var deepmerge_1 = deepmerge;
	var cjs = deepmerge_1;
	var merge = cjs;

	var quantities = {exports: {}};

	/*
	The MIT License (MIT)
	Copyright © 2006-2007 Kevin C. Olbrich
	Copyright © 2010-2016 LIM SAS (http://lim.eu) - Julien Sanchez

	Permission is hereby granted, free of charge, to any person obtaining a copy of
	this software and associated documentation files (the "Software"), to deal in
	the Software without restriction, including without limitation the rights to
	use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
	of the Software, and to permit persons to whom the Software is furnished to do
	so, subject to the following conditions:

	The above copyright notice and this permission notice shall be included in all
	copies or substantial portions of the Software.

	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
	FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
	AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
	LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
	OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
	SOFTWARE.
	*/
	(function (module, exports) {
	  (function (global, factory) {
	    module.exports = factory() ;
	  })(commonjsGlobal, function () {

	    /**
	     * Tests if a value is a string
	     *
	     * @param {*} value - Value to test
	     *
	     * @returns {boolean} true if value is a string, false otherwise
	     */
	    function isString(value) {
	      return typeof value === "string" || value instanceof String;
	    }

	    /*
	     * Prefer stricter Number.isFinite if currently supported.
	     * To be dropped when ES6 is finalized. Obsolete browsers will
	     * have to use ES6 polyfills.
	     */
	    var isFiniteImpl = Number.isFinite || window.isFinite;
	    /**
	     * Tests if a value is a number
	     *
	     * @param {*} value - Value to test
	     *
	     * @returns {boolean} true if value is a number, false otherwise
	     */
	    function isNumber(value) {
	      // Number.isFinite allows not to consider NaN or '1' as numbers
	      return isFiniteImpl(value);
	    }

	    /*
	     * Identity function
	     */
	    function identity(value) {
	      return value;
	    }

	    /**
	     * Returns unique strings from list
	     *
	     * @param {string[]} strings - array of strings
	     *
	     *
	     * @returns {string[]} a new array of strings without duplicates
	     */
	    function uniq(strings) {
	      var seen = {};
	      return strings.filter(function (item) {
	        return seen.hasOwnProperty(item) ? false : seen[item] = true;
	      });
	    }
	    function compareArray(array1, array2) {
	      if (array2.length !== array1.length) {
	        return false;
	      }
	      for (var i = 0; i < array1.length; i++) {
	        if (array2[i].compareArray) {
	          if (!array2[i].compareArray(array1[i])) {
	            return false;
	          }
	        }
	        if (array2[i] !== array1[i]) {
	          return false;
	        }
	      }
	      return true;
	    }
	    function assign(target, properties) {
	      Object.keys(properties).forEach(function (key) {
	        target[key] = properties[key];
	      });
	    }

	    /**
	     * Safely multiplies numbers while avoiding floating errors
	     * like 0.1 * 0.1 => 0.010000000000000002
	     *
	     * @param {...number} numbers - numbers to multiply
	     *
	     * @returns {number} result
	     */
	    function mulSafe() {
	      var result = 1,
	        decimals = 0;
	      for (var i = 0; i < arguments.length; i++) {
	        var arg = arguments[i];
	        decimals = decimals + getFractional(arg);
	        result *= arg;
	      }
	      return decimals !== 0 ? round(result, decimals) : result;
	    }

	    /**
	     * Safely divides two numbers while avoiding floating errors
	     * like 0.3 / 0.05 => 5.999999999999999
	     *
	     * @returns {number} result
	     * @param {number} num Numerator
	     * @param {number} den Denominator
	     */
	    function divSafe(num, den) {
	      if (den === 0) {
	        throw new Error("Divide by zero");
	      }
	      var factor = Math.pow(10, getFractional(den));
	      var invDen = factor / (factor * den);
	      return mulSafe(num, invDen);
	    }

	    /**
	     * Rounds value at the specified number of decimals
	     *
	     * @param {number} val - value to round
	     * @param {number} decimals - number of decimals
	     *
	     * @returns {number} rounded number
	     */
	    function round(val, decimals) {
	      return Math.round(val * Math.pow(10, decimals)) / Math.pow(10, decimals);
	    }
	    function getFractional(num) {
	      // Check for NaNs or Infinities
	      if (!isFinite(num)) {
	        return 0;
	      }

	      // Faster than parsing strings
	      // http://jsperf.com/count-decimals/2
	      var count = 0;
	      while (num % 1 !== 0) {
	        num *= 10;
	        count++;
	      }
	      return count;
	    }

	    /**
	     * Custom error type definition
	     * @constructor
	     */
	    function QtyError() {
	      var err;
	      if (!this) {
	        // Allows to instantiate QtyError without new()
	        err = Object.create(QtyError.prototype);
	        QtyError.apply(err, arguments);
	        return err;
	      }
	      err = Error.apply(this, arguments);
	      this.name = "QtyError";
	      this.message = err.message;
	      this.stack = err.stack;
	    }
	    QtyError.prototype = Object.create(Error.prototype, {
	      constructor: {
	        value: QtyError
	      }
	    });

	    /*
	     * Throws incompatible units error
	     * @param {string} left - units
	     * @param {string} right - units incompatible with first argument
	     * @throws "Incompatible units" error
	     */
	    function throwIncompatibleUnits(left, right) {
	      throw new QtyError("Incompatible units: " + left + " and " + right);
	    }
	    var UNITS = {
	      /* prefixes */
	      "<googol>": [["googol"], 1e100, "prefix"],
	      "<kibi>": [["Ki", "Kibi", "kibi"], Math.pow(2, 10), "prefix"],
	      "<mebi>": [["Mi", "Mebi", "mebi"], Math.pow(2, 20), "prefix"],
	      "<gibi>": [["Gi", "Gibi", "gibi"], Math.pow(2, 30), "prefix"],
	      "<tebi>": [["Ti", "Tebi", "tebi"], Math.pow(2, 40), "prefix"],
	      "<pebi>": [["Pi", "Pebi", "pebi"], Math.pow(2, 50), "prefix"],
	      "<exi>": [["Ei", "Exi", "exi"], Math.pow(2, 60), "prefix"],
	      "<zebi>": [["Zi", "Zebi", "zebi"], Math.pow(2, 70), "prefix"],
	      "<yebi>": [["Yi", "Yebi", "yebi"], Math.pow(2, 80), "prefix"],
	      "<yotta>": [["Y", "Yotta", "yotta"], 1e24, "prefix"],
	      "<zetta>": [["Z", "Zetta", "zetta"], 1e21, "prefix"],
	      "<exa>": [["E", "Exa", "exa"], 1e18, "prefix"],
	      "<peta>": [["P", "Peta", "peta"], 1e15, "prefix"],
	      "<tera>": [["T", "Tera", "tera"], 1e12, "prefix"],
	      "<giga>": [["G", "Giga", "giga"], 1e9, "prefix"],
	      "<mega>": [["M", "Mega", "mega"], 1e6, "prefix"],
	      "<kilo>": [["k", "kilo"], 1e3, "prefix"],
	      "<hecto>": [["h", "Hecto", "hecto"], 1e2, "prefix"],
	      "<deca>": [["da", "Deca", "deca", "deka"], 1e1, "prefix"],
	      "<deci>": [["d", "Deci", "deci"], 1e-1, "prefix"],
	      "<centi>": [["c", "Centi", "centi"], 1e-2, "prefix"],
	      "<milli>": [["m", "Milli", "milli"], 1e-3, "prefix"],
	      "<micro>": [["u", "\u03BC" /*µ as greek letter*/, "\u00B5" /*µ as micro sign*/, "Micro", "mc", "micro"], 1e-6, "prefix"],
	      "<nano>": [["n", "Nano", "nano"], 1e-9, "prefix"],
	      "<pico>": [["p", "Pico", "pico"], 1e-12, "prefix"],
	      "<femto>": [["f", "Femto", "femto"], 1e-15, "prefix"],
	      "<atto>": [["a", "Atto", "atto"], 1e-18, "prefix"],
	      "<zepto>": [["z", "Zepto", "zepto"], 1e-21, "prefix"],
	      "<yocto>": [["y", "Yocto", "yocto"], 1e-24, "prefix"],
	      "<1>": [["1", "<1>"], 1, ""],
	      /* length units */
	      "<meter>": [["m", "meter", "meters", "metre", "metres"], 1.0, "length", ["<meter>"]],
	      "<inch>": [["in", "inch", "inches", "\""], 0.0254, "length", ["<meter>"]],
	      "<foot>": [["ft", "foot", "feet", "'"], 0.3048, "length", ["<meter>"]],
	      "<yard>": [["yd", "yard", "yards"], 0.9144, "length", ["<meter>"]],
	      "<mile>": [["mi", "mile", "miles"], 1609.344, "length", ["<meter>"]],
	      "<naut-mile>": [["nmi", "naut-mile"], 1852, "length", ["<meter>"]],
	      "<league>": [["league", "leagues"], 4828, "length", ["<meter>"]],
	      "<furlong>": [["furlong", "furlongs"], 201.2, "length", ["<meter>"]],
	      "<rod>": [["rd", "rod", "rods"], 5.029, "length", ["<meter>"]],
	      "<mil>": [["mil", "mils"], 0.0000254, "length", ["<meter>"]],
	      "<angstrom>": [["ang", "angstrom", "angstroms"], 1e-10, "length", ["<meter>"]],
	      "<fathom>": [["fathom", "fathoms"], 1.829, "length", ["<meter>"]],
	      "<pica>": [["pica", "picas"], 0.00423333333, "length", ["<meter>"]],
	      "<point>": [["pt", "point", "points"], 0.000352777778, "length", ["<meter>"]],
	      "<redshift>": [["z", "red-shift", "redshift"], 1.302773e26, "length", ["<meter>"]],
	      "<AU>": [["AU", "astronomical-unit"], 149597900000, "length", ["<meter>"]],
	      "<light-second>": [["ls", "light-second"], 299792500, "length", ["<meter>"]],
	      "<light-minute>": [["lmin", "light-minute"], 17987550000, "length", ["<meter>"]],
	      "<light-year>": [["ly", "light-year"], 9460528000000000, "length", ["<meter>"]],
	      "<parsec>": [["pc", "parsec", "parsecs"], 30856780000000000, "length", ["<meter>"]],
	      "<datamile>": [["DM", "datamile"], 1828.8, "length", ["<meter>"]],
	      /* mass */
	      "<kilogram>": [["kg", "kilogram", "kilograms"], 1.0, "mass", ["<kilogram>"]],
	      "<AMU>": [["u", "AMU", "amu"], 1.660538921e-27, "mass", ["<kilogram>"]],
	      "<dalton>": [["Da", "Dalton", "Daltons", "dalton", "daltons"], 1.660538921e-27, "mass", ["<kilogram>"]],
	      "<slug>": [["slug", "slugs"], 14.5939029, "mass", ["<kilogram>"]],
	      "<short-ton>": [["tn", "ton", "short-ton"], 907.18474, "mass", ["<kilogram>"]],
	      "<metric-ton>": [["t", "tonne", "metric-ton"], 1000, "mass", ["<kilogram>"]],
	      "<carat>": [["ct", "carat", "carats"], 0.0002, "mass", ["<kilogram>"]],
	      "<pound>": [["lbs", "lb", "pound", "pounds", "#"], 0.45359237, "mass", ["<kilogram>"]],
	      "<ounce>": [["oz", "ounce", "ounces"], 0.0283495231, "mass", ["<kilogram>"]],
	      "<gram>": [["g", "gram", "grams", "gramme", "grammes"], 1e-3, "mass", ["<kilogram>"]],
	      "<grain>": [["grain", "grains", "gr"], 6.479891e-5, "mass", ["<kilogram>"]],
	      "<dram>": [["dram", "drams", "dr"], 0.0017718452, "mass", ["<kilogram>"]],
	      "<stone>": [["stone", "stones", "st"], 6.35029318, "mass", ["<kilogram>"]],
	      /* area */
	      "<hectare>": [["hectare"], 10000, "area", ["<meter>", "<meter>"]],
	      "<acre>": [["acre", "acres"], 4046.85642, "area", ["<meter>", "<meter>"]],
	      "<sqft>": [["sqft"], 1, "area", ["<foot>", "<foot>"]],
	      /* volume */
	      "<liter>": [["l", "L", "liter", "liters", "litre", "litres"], 0.001, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<gallon>": [["gal", "gallon", "gallons"], 0.0037854118, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<gallon-imp>": [["galimp", "gallon-imp", "gallons-imp"], 0.0045460900, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<quart>": [["qt", "quart", "quarts"], 0.00094635295, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<pint>": [["pt", "pint", "pints"], 0.000473176475, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<pint-imp>": [["ptimp", "pint-imp", "pints-imp"], 5.6826125e-4, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<cup>": [["cu", "cup", "cups"], 0.000236588238, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<fluid-ounce>": [["floz", "fluid-ounce", "fluid-ounces"], 2.95735297e-5, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<fluid-ounce-imp>": [["flozimp", "floz-imp", "fluid-ounce-imp", "fluid-ounces-imp"], 2.84130625e-5, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<tablespoon>": [["tb", "tbsp", "tbs", "tablespoon", "tablespoons"], 1.47867648e-5, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<teaspoon>": [["tsp", "teaspoon", "teaspoons"], 4.92892161e-6, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<bushel>": [["bu", "bsh", "bushel", "bushels"], 0.035239072, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<oilbarrel>": [["bbl", "oilbarrel", "oilbarrels", "oil-barrel", "oil-barrels"], 0.158987294928, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<beerbarrel>": [["bl", "bl-us", "beerbarrel", "beerbarrels", "beer-barrel", "beer-barrels"], 0.1173477658, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<beerbarrel-imp>": [["blimp", "bl-imp", "beerbarrel-imp", "beerbarrels-imp", "beer-barrel-imp", "beer-barrels-imp"], 0.16365924, "volume", ["<meter>", "<meter>", "<meter>"]],
	      /* speed */
	      "<kph>": [["kph"], 0.277777778, "speed", ["<meter>"], ["<second>"]],
	      "<mph>": [["mph"], 0.44704, "speed", ["<meter>"], ["<second>"]],
	      "<knot>": [["kt", "kn", "kts", "knot", "knots"], 0.514444444, "speed", ["<meter>"], ["<second>"]],
	      "<fps>": [["fps"], 0.3048, "speed", ["<meter>"], ["<second>"]],
	      /* acceleration */
	      "<gee>": [["gee"], 9.80665, "acceleration", ["<meter>"], ["<second>", "<second>"]],
	      "<Gal>": [["Gal"], 1e-2, "acceleration", ["<meter>"], ["<second>", "<second>"]],
	      /* temperature_difference */
	      "<kelvin>": [["degK", "kelvin"], 1.0, "temperature", ["<kelvin>"]],
	      "<celsius>": [["degC", "celsius", "celsius", "centigrade"], 1.0, "temperature", ["<kelvin>"]],
	      "<fahrenheit>": [["degF", "fahrenheit"], 5 / 9, "temperature", ["<kelvin>"]],
	      "<rankine>": [["degR", "rankine"], 5 / 9, "temperature", ["<kelvin>"]],
	      "<temp-K>": [["tempK", "temp-K"], 1.0, "temperature", ["<temp-K>"]],
	      "<temp-C>": [["tempC", "temp-C"], 1.0, "temperature", ["<temp-K>"]],
	      "<temp-F>": [["tempF", "temp-F"], 5 / 9, "temperature", ["<temp-K>"]],
	      "<temp-R>": [["tempR", "temp-R"], 5 / 9, "temperature", ["<temp-K>"]],
	      /* time */
	      "<second>": [["s", "sec", "secs", "second", "seconds"], 1.0, "time", ["<second>"]],
	      "<minute>": [["min", "mins", "minute", "minutes"], 60.0, "time", ["<second>"]],
	      "<hour>": [["h", "hr", "hrs", "hour", "hours"], 3600.0, "time", ["<second>"]],
	      "<day>": [["d", "day", "days"], 3600 * 24, "time", ["<second>"]],
	      "<week>": [["wk", "week", "weeks"], 7 * 3600 * 24, "time", ["<second>"]],
	      "<fortnight>": [["fortnight", "fortnights"], 1209600, "time", ["<second>"]],
	      "<year>": [["y", "yr", "year", "years", "annum"], 31556926, "time", ["<second>"]],
	      "<decade>": [["decade", "decades"], 315569260, "time", ["<second>"]],
	      "<century>": [["century", "centuries"], 3155692600, "time", ["<second>"]],
	      /* pressure */
	      "<pascal>": [["Pa", "pascal", "Pascal"], 1.0, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<bar>": [["bar", "bars"], 100000, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<mmHg>": [["mmHg"], 133.322368, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<inHg>": [["inHg"], 3386.3881472, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<torr>": [["torr"], 133.322368, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<atm>": [["atm", "ATM", "atmosphere", "atmospheres"], 101325, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<psi>": [["psi"], 6894.76, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<cmh2o>": [["cmH2O", "cmh2o"], 98.0638, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<inh2o>": [["inH2O", "inh2o"], 249.082052, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      /* viscosity */
	      "<poise>": [["P", "poise"], 0.1, "viscosity", ["<kilogram>"], ["<meter>", "<second>"]],
	      "<stokes>": [["St", "stokes"], 1e-4, "viscosity", ["<meter>", "<meter>"], ["<second>"]],
	      /* substance */
	      "<mole>": [["mol", "mole"], 1.0, "substance", ["<mole>"]],
	      /* molar_concentration */
	      "<molar>": [["M", "molar"], 1000, "molar_concentration", ["<mole>"], ["<meter>", "<meter>", "<meter>"]],
	      "<wtpercent>": [["wt%", "wtpercent"], 10, "molar_concentration", ["<kilogram>"], ["<meter>", "<meter>", "<meter>"]],
	      /* activity */
	      "<katal>": [["kat", "katal", "Katal"], 1.0, "activity", ["<mole>"], ["<second>"]],
	      "<unit>": [["U", "enzUnit", "unit"], 16.667e-16, "activity", ["<mole>"], ["<second>"]],
	      /* capacitance */
	      "<farad>": [["F", "farad", "Farad"], 1.0, "capacitance", ["<second>", "<second>", "<second>", "<second>", "<ampere>", "<ampere>"], ["<meter>", "<meter>", "<kilogram>"]],
	      /* charge */
	      "<coulomb>": [["C", "coulomb", "Coulomb"], 1.0, "charge", ["<ampere>", "<second>"]],
	      "<Ah>": [["Ah"], 3600, "charge", ["<ampere>", "<second>"]],
	      /* current */
	      "<ampere>": [["A", "Ampere", "ampere", "amp", "amps"], 1.0, "current", ["<ampere>"]],
	      /* conductance */
	      "<siemens>": [["S", "Siemens", "siemens"], 1.0, "conductance", ["<second>", "<second>", "<second>", "<ampere>", "<ampere>"], ["<kilogram>", "<meter>", "<meter>"]],
	      /* inductance */
	      "<henry>": [["H", "Henry", "henry"], 1.0, "inductance", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<ampere>", "<ampere>"]],
	      /* potential */
	      "<volt>": [["V", "Volt", "volt", "volts"], 1.0, "potential", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<second>", "<ampere>"]],
	      /* resistance */
	      "<ohm>": [["Ohm", "ohm", "\u03A9" /*Ω as greek letter*/, "\u2126" /*Ω as ohm sign*/], 1.0, "resistance", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<second>", "<ampere>", "<ampere>"]],
	      /* magnetism */
	      "<weber>": [["Wb", "weber", "webers"], 1.0, "magnetism", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<ampere>"]],
	      "<tesla>": [["T", "tesla", "teslas"], 1.0, "magnetism", ["<kilogram>"], ["<second>", "<second>", "<ampere>"]],
	      "<gauss>": [["G", "gauss"], 1e-4, "magnetism", ["<kilogram>"], ["<second>", "<second>", "<ampere>"]],
	      "<maxwell>": [["Mx", "maxwell", "maxwells"], 1e-8, "magnetism", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<ampere>"]],
	      "<oersted>": [["Oe", "oersted", "oersteds"], 250.0 / Math.PI, "magnetism", ["<ampere>"], ["<meter>"]],
	      /* energy */
	      "<joule>": [["J", "joule", "Joule", "joules", "Joules"], 1.0, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      "<erg>": [["erg", "ergs"], 1e-7, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      "<btu>": [["BTU", "btu", "BTUs"], 1055.056, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      "<calorie>": [["cal", "calorie", "calories"], 4.18400, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      "<Calorie>": [["Cal", "Calorie", "Calories"], 4184.00, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      "<therm-US>": [["th", "therm", "therms", "Therm", "therm-US"], 105480400, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      "<Wh>": [["Wh"], 3600, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      "<electronvolt>": [["eV", "electronvolt", "electronvolts"], 1.602176634E-19, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      /* force */
	      "<newton>": [["N", "Newton", "newton"], 1.0, "force", ["<kilogram>", "<meter>"], ["<second>", "<second>"]],
	      "<dyne>": [["dyn", "dyne"], 1e-5, "force", ["<kilogram>", "<meter>"], ["<second>", "<second>"]],
	      "<pound-force>": [["lbf", "pound-force"], 4.448222, "force", ["<kilogram>", "<meter>"], ["<second>", "<second>"]],
	      /* frequency */
	      "<hertz>": [["Hz", "hertz", "Hertz"], 1.0, "frequency", ["<1>"], ["<second>"]],
	      /* angle */
	      "<radian>": [["rad", "radian", "radians"], 1.0, "angle", ["<radian>"]],
	      "<degree>": [["deg", "degree", "degrees"], Math.PI / 180.0, "angle", ["<radian>"]],
	      "<arcminute>": [["arcmin", "arcminute", "arcminutes"], Math.PI / 10800.0, "angle", ["<radian>"]],
	      "<arcsecond>": [["arcsec", "arcsecond", "arcseconds"], Math.PI / 648000.0, "angle", ["<radian>"]],
	      "<gradian>": [["gon", "grad", "gradian", "grads"], Math.PI / 200.0, "angle", ["<radian>"]],
	      "<steradian>": [["sr", "steradian", "steradians"], 1.0, "solid_angle", ["<steradian>"]],
	      /* rotation */
	      "<rotation>": [["rotation"], 2.0 * Math.PI, "angle", ["<radian>"]],
	      "<rpm>": [["rpm"], 2.0 * Math.PI / 60.0, "angular_velocity", ["<radian>"], ["<second>"]],
	      /* information */
	      "<byte>": [["B", "byte", "bytes"], 1.0, "information", ["<byte>"]],
	      "<bit>": [["b", "bit", "bits"], 0.125, "information", ["<byte>"]],
	      /* information rate */
	      "<Bps>": [["Bps"], 1.0, "information_rate", ["<byte>"], ["<second>"]],
	      "<bps>": [["bps"], 0.125, "information_rate", ["<byte>"], ["<second>"]],
	      /* currency */
	      "<dollar>": [["USD", "dollar"], 1.0, "currency", ["<dollar>"]],
	      "<cents>": [["cents"], 0.01, "currency", ["<dollar>"]],
	      /* luminosity */
	      "<candela>": [["cd", "candela"], 1.0, "luminosity", ["<candela>"]],
	      "<lumen>": [["lm", "lumen"], 1.0, "luminous_power", ["<candela>", "<steradian>"]],
	      "<lux>": [["lux"], 1.0, "illuminance", ["<candela>", "<steradian>"], ["<meter>", "<meter>"]],
	      /* power */
	      "<watt>": [["W", "watt", "watts"], 1.0, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
	      "<volt-ampere>": [["VA", "volt-ampere"], 1.0, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
	      "<volt-ampere-reactive>": [["var", "Var", "VAr", "VAR", "volt-ampere-reactive"], 1.0, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
	      "<horsepower>": [["hp", "horsepower"], 745.699872, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
	      /* radiation */
	      "<gray>": [["Gy", "gray", "grays"], 1.0, "radiation", ["<meter>", "<meter>"], ["<second>", "<second>"]],
	      "<roentgen>": [["R", "roentgen"], 0.009330, "radiation", ["<meter>", "<meter>"], ["<second>", "<second>"]],
	      "<sievert>": [["Sv", "sievert", "sieverts"], 1.0, "radiation", ["<meter>", "<meter>"], ["<second>", "<second>"]],
	      "<becquerel>": [["Bq", "becquerel", "becquerels"], 1.0, "radiation", ["<1>"], ["<second>"]],
	      "<curie>": [["Ci", "curie", "curies"], 3.7e10, "radiation", ["<1>"], ["<second>"]],
	      /* rate */
	      "<cpm>": [["cpm"], 1.0 / 60.0, "rate", ["<count>"], ["<second>"]],
	      "<dpm>": [["dpm"], 1.0 / 60.0, "rate", ["<count>"], ["<second>"]],
	      "<bpm>": [["bpm"], 1.0 / 60.0, "rate", ["<count>"], ["<second>"]],
	      /* resolution / typography */
	      "<dot>": [["dot", "dots"], 1, "resolution", ["<each>"]],
	      "<pixel>": [["pixel", "px"], 1, "resolution", ["<each>"]],
	      "<ppi>": [["ppi"], 1, "resolution", ["<pixel>"], ["<inch>"]],
	      "<dpi>": [["dpi"], 1, "typography", ["<dot>"], ["<inch>"]],
	      /* other */
	      "<cell>": [["cells", "cell"], 1, "counting", ["<each>"]],
	      "<each>": [["each"], 1.0, "counting", ["<each>"]],
	      "<count>": [["count"], 1.0, "counting", ["<each>"]],
	      "<base-pair>": [["bp", "base-pair"], 1.0, "counting", ["<each>"]],
	      "<nucleotide>": [["nt", "nucleotide"], 1.0, "counting", ["<each>"]],
	      "<molecule>": [["molecule", "molecules"], 1.0, "counting", ["<1>"]],
	      "<dozen>": [["doz", "dz", "dozen"], 12.0, "prefix_only", ["<each>"]],
	      "<percent>": [["%", "percent"], 0.01, "prefix_only", ["<1>"]],
	      "<ppm>": [["ppm"], 1e-6, "prefix_only", ["<1>"]],
	      "<ppb>": [["ppb"], 1e-9, "prefix_only", ["<1>"]],
	      "<ppt>": [["ppt"], 1e-12, "prefix_only", ["<1>"]],
	      "<ppq>": [["ppq"], 1e-15, "prefix_only", ["<1>"]],
	      "<gross>": [["gr", "gross"], 144.0, "prefix_only", ["<dozen>", "<dozen>"]],
	      "<decibel>": [["dB", "decibel", "decibels"], 1.0, "logarithmic", ["<decibel>"]]
	    };
	    var BASE_UNITS = ["<meter>", "<kilogram>", "<second>", "<mole>", "<ampere>", "<radian>", "<kelvin>", "<temp-K>", "<byte>", "<dollar>", "<candela>", "<each>", "<steradian>", "<decibel>"];
	    var UNITY = "<1>";
	    var UNITY_ARRAY = [UNITY];

	    // Setup

	    /**
	     * Asserts unit definition is valid
	     *
	     * @param {string} unitDef - Name of unit to test
	     * @param {Object} definition - Definition of unit to test
	     *
	     * @returns {void}
	     * @throws {QtyError} if unit definition is not valid
	     */
	    function validateUnitDefinition(unitDef, definition) {
	      var scalar = definition[1];
	      var numerator = definition[3] || [];
	      var denominator = definition[4] || [];
	      if (!isNumber(scalar)) {
	        throw new QtyError(unitDef + ": Invalid unit definition. " + "'scalar' must be a number");
	      }
	      numerator.forEach(function (unit) {
	        if (UNITS[unit] === undefined) {
	          throw new QtyError(unitDef + ": Invalid unit definition. " + "Unit " + unit + " in 'numerator' is not recognized");
	        }
	      });
	      denominator.forEach(function (unit) {
	        if (UNITS[unit] === undefined) {
	          throw new QtyError(unitDef + ": Invalid unit definition. " + "Unit " + unit + " in 'denominator' is not recognized");
	        }
	      });
	    }
	    var PREFIX_VALUES = {};
	    var PREFIX_MAP = {};
	    var UNIT_VALUES = {};
	    var UNIT_MAP = {};
	    var OUTPUT_MAP = {};
	    for (var unitDef in UNITS) {
	      if (UNITS.hasOwnProperty(unitDef)) {
	        var definition = UNITS[unitDef];
	        if (definition[2] === "prefix") {
	          PREFIX_VALUES[unitDef] = definition[1];
	          for (var i = 0; i < definition[0].length; i++) {
	            PREFIX_MAP[definition[0][i]] = unitDef;
	          }
	        } else {
	          validateUnitDefinition(unitDef, definition);
	          UNIT_VALUES[unitDef] = {
	            scalar: definition[1],
	            numerator: definition[3],
	            denominator: definition[4]
	          };
	          for (var j = 0; j < definition[0].length; j++) {
	            UNIT_MAP[definition[0][j]] = unitDef;
	          }
	        }
	        OUTPUT_MAP[unitDef] = definition[0][0];
	      }
	    }

	    /**
	     * Returns a list of available units of kind
	     *
	     * @param {string} [kind] - kind of units
	     * @returns {array} names of units
	     * @throws {QtyError} if kind is unknown
	     */
	    function getUnits(kind) {
	      var i;
	      var units = [];
	      var unitKeys = Object.keys(UNITS);
	      if (typeof kind === "undefined") {
	        for (i = 0; i < unitKeys.length; i++) {
	          if (["", "prefix"].indexOf(UNITS[unitKeys[i]][2]) === -1) {
	            units.push(unitKeys[i].substr(1, unitKeys[i].length - 2));
	          }
	        }
	      } else if (this.getKinds().indexOf(kind) === -1) {
	        throw new QtyError("Kind not recognized");
	      } else {
	        for (i = 0; i < unitKeys.length; i++) {
	          if (UNITS[unitKeys[i]][2] === kind) {
	            units.push(unitKeys[i].substr(1, unitKeys[i].length - 2));
	          }
	        }
	      }
	      return units.sort(function (a, b) {
	        if (a.toLowerCase() < b.toLowerCase()) {
	          return -1;
	        }
	        if (a.toLowerCase() > b.toLowerCase()) {
	          return 1;
	        }
	        return 0;
	      });
	    }

	    /**
	     * Returns a list of alternative names for a unit
	     *
	     * @param {string} unitName - name of unit
	     * @returns {string[]} aliases for unit
	     * @throws {QtyError} if unit is unknown
	     */
	    function getAliases(unitName) {
	      if (!UNIT_MAP[unitName]) {
	        throw new QtyError("Unit not recognized");
	      }
	      return UNITS[UNIT_MAP[unitName]][0];
	    }
	    var SIGNATURE_VECTOR = ["length", "time", "temperature", "mass", "current", "substance", "luminosity", "currency", "information", "angle"];

	    /*
	    calculates the unit signature id for use in comparing compatible units and simplification
	    the signature is based on a simple classification of units and is based on the following publication
	     Novak, G.S., Jr. "Conversion of units of measurement", IEEE Transactions on Software Engineering,
	    21(8), Aug 1995, pp.651-661
	    doi://10.1109/32.403789
	    http://ieeexplore.ieee.org/Xplore/login.jsp?url=/iel1/32/9079/00403789.pdf?isnumber=9079&prod=JNL&arnumber=403789&arSt=651&ared=661&arAuthor=Novak%2C+G.S.%2C+Jr.
	    */
	    function unitSignature() {
	      if (this.signature) {
	        return this.signature;
	      }
	      var vector = unitSignatureVector.call(this);
	      for (var i = 0; i < vector.length; i++) {
	        vector[i] *= Math.pow(20, i);
	      }
	      return vector.reduce(function (previous, current) {
	        return previous + current;
	      }, 0);
	    }

	    // calculates the unit signature vector used by unit_signature
	    function unitSignatureVector() {
	      if (!this.isBase()) {
	        return unitSignatureVector.call(this.toBase());
	      }
	      var vector = new Array(SIGNATURE_VECTOR.length);
	      for (var i = 0; i < vector.length; i++) {
	        vector[i] = 0;
	      }
	      var r, n;
	      for (var j = 0; j < this.numerator.length; j++) {
	        if (r = UNITS[this.numerator[j]]) {
	          n = SIGNATURE_VECTOR.indexOf(r[2]);
	          if (n >= 0) {
	            vector[n] = vector[n] + 1;
	          }
	        }
	      }
	      for (var k = 0; k < this.denominator.length; k++) {
	        if (r = UNITS[this.denominator[k]]) {
	          n = SIGNATURE_VECTOR.indexOf(r[2]);
	          if (n >= 0) {
	            vector[n] = vector[n] - 1;
	          }
	        }
	      }
	      return vector;
	    }
	    var SIGN = "[+-]";
	    var INTEGER = "\\d+";
	    var SIGNED_INTEGER = SIGN + "?" + INTEGER;
	    var FRACTION = "\\." + INTEGER;
	    var FLOAT = "(?:" + INTEGER + "(?:" + FRACTION + ")?" + ")" + "|" + "(?:" + FRACTION + ")";
	    var EXPONENT = "[Ee]" + SIGNED_INTEGER;
	    var SCI_NUMBER = "(?:" + FLOAT + ")(?:" + EXPONENT + ")?";
	    var SIGNED_NUMBER = SIGN + "?\\s*" + SCI_NUMBER;
	    var QTY_STRING = "(" + SIGNED_NUMBER + ")?" + "\\s*([^/]*)(?:\/(.+))?";
	    var QTY_STRING_REGEX = new RegExp("^" + QTY_STRING + "$");
	    var POWER_OP = "\\^|\\*{2}";
	    // Allow unit powers representing scalar, length, area, volume; 4 is for some
	    // special case representations in SI base units.
	    var SAFE_POWER = "[01234]";
	    var TOP_REGEX = new RegExp("([^ \\*\\d]+?)(?:" + POWER_OP + ")?(-?" + SAFE_POWER + "(?![a-zA-Z]))");
	    var BOTTOM_REGEX = new RegExp("([^ \\*\\d]+?)(?:" + POWER_OP + ")?(" + SAFE_POWER + "(?![a-zA-Z]))");

	    /* parse a string into a unit object.
	     * Typical formats like :
	     * "5.6 kg*m/s^2"
	     * "5.6 kg*m*s^-2"
	     * "5.6 kilogram*meter*second^-2"
	     * "2.2 kPa"
	     * "37 degC"
	     * "1"  -- creates a unitless constant with value 1
	     * "GPa"  -- creates a unit with scalar 1 with units 'GPa'
	     * 6'4"  -- recognized as 6 feet + 4 inches
	     * 8 lbs 8 oz -- recognized as 8 lbs + 8 ounces
	     */
	    function parse(val) {
	      if (!isString(val)) {
	        val = val.toString();
	      }
	      val = val.trim();
	      var result = QTY_STRING_REGEX.exec(val);
	      if (!result) {
	        throw new QtyError(val + ": Quantity not recognized");
	      }
	      var scalarMatch = result[1];
	      if (scalarMatch) {
	        // Allow whitespaces between sign and scalar for loose parsing
	        scalarMatch = scalarMatch.replace(/\s/g, "");
	        this.scalar = parseFloat(scalarMatch);
	      } else {
	        this.scalar = 1;
	      }
	      var top = result[2];
	      var bottom = result[3];
	      var n, x, nx;
	      // TODO DRY me
	      while (result = TOP_REGEX.exec(top)) {
	        n = parseFloat(result[2]);
	        if (isNaN(n)) {
	          // Prevents infinite loops
	          throw new QtyError("Unit exponent is not a number");
	        }
	        // Disallow unrecognized unit even if exponent is 0
	        if (n === 0 && !UNIT_TEST_REGEX.test(result[1])) {
	          throw new QtyError("Unit not recognized");
	        }
	        x = result[1] + " ";
	        nx = "";
	        for (var i = 0; i < Math.abs(n); i++) {
	          nx += x;
	        }
	        if (n >= 0) {
	          top = top.replace(result[0], nx);
	        } else {
	          bottom = bottom ? bottom + nx : nx;
	          top = top.replace(result[0], "");
	        }
	      }
	      while (result = BOTTOM_REGEX.exec(bottom)) {
	        n = parseFloat(result[2]);
	        if (isNaN(n)) {
	          // Prevents infinite loops
	          throw new QtyError("Unit exponent is not a number");
	        }
	        // Disallow unrecognized unit even if exponent is 0
	        if (n === 0 && !UNIT_TEST_REGEX.test(result[1])) {
	          throw new QtyError("Unit not recognized");
	        }
	        x = result[1] + " ";
	        nx = "";
	        for (var j = 0; j < n; j++) {
	          nx += x;
	        }
	        bottom = bottom.replace(result[0], nx);
	      }
	      if (top) {
	        this.numerator = parseUnits(top.trim());
	      }
	      if (bottom) {
	        this.denominator = parseUnits(bottom.trim());
	      }
	    }
	    var PREFIX_REGEX = Object.keys(PREFIX_MAP).sort(function (a, b) {
	      return b.length - a.length;
	    }).join("|");
	    var UNIT_REGEX = Object.keys(UNIT_MAP).sort(function (a, b) {
	      return b.length - a.length;
	    }).join("|");
	    /*
	     * Minimal boundary regex to support units with Unicode characters
	     * \b only works for ASCII
	     */
	    var BOUNDARY_REGEX = "\\b|$";
	    var UNIT_MATCH = "(" + PREFIX_REGEX + ")??(" + UNIT_REGEX + ")(?:" + BOUNDARY_REGEX + ")";
	    var UNIT_TEST_REGEX = new RegExp("^\\s*(" + UNIT_MATCH + "[\\s\\*]*)+$");
	    var UNIT_MATCH_REGEX = new RegExp(UNIT_MATCH, "g"); // g flag for multiple occurences
	    var parsedUnitsCache = {};
	    /**
	     * Parses and converts units string to normalized unit array.
	     * Result is cached to speed up next calls.
	     *
	     * @param {string} units Units string
	     * @returns {string[]} Array of normalized units
	     *
	     * @example
	     * // Returns ["<second>", "<meter>", "<second>"]
	     * parseUnits("s m s");
	     *
	     */
	    function parseUnits(units) {
	      var cached = parsedUnitsCache[units];
	      if (cached) {
	        return cached;
	      }
	      var unitMatch,
	        normalizedUnits = [];

	      // Scan
	      if (!UNIT_TEST_REGEX.test(units)) {
	        throw new QtyError("Unit not recognized");
	      }
	      while (unitMatch = UNIT_MATCH_REGEX.exec(units)) {
	        normalizedUnits.push(unitMatch.slice(1));
	      }
	      normalizedUnits = normalizedUnits.map(function (item) {
	        return PREFIX_MAP[item[0]] ? [PREFIX_MAP[item[0]], UNIT_MAP[item[1]]] : [UNIT_MAP[item[1]]];
	      });

	      // Flatten and remove null elements
	      normalizedUnits = normalizedUnits.reduce(function (a, b) {
	        return a.concat(b);
	      }, []);
	      normalizedUnits = normalizedUnits.filter(function (item) {
	        return item;
	      });
	      parsedUnitsCache[units] = normalizedUnits;
	      return normalizedUnits;
	    }

	    /**
	     * Parses a string as a quantity
	     * @param {string} value - quantity as text
	     * @throws if value is not a string
	     * @returns {Qty|null} Parsed quantity or null if unrecognized
	     */
	    function globalParse(value) {
	      if (!isString(value)) {
	        throw new QtyError("Argument should be a string");
	      }
	      try {
	        return this(value);
	      } catch (e) {
	        return null;
	      }
	    }

	    /**
	     * Tests if a value is a Qty instance
	     *
	     * @param {*} value - Value to test
	     *
	     * @returns {boolean} true if value is a Qty instance, false otherwise
	     */
	    function isQty(value) {
	      return value instanceof Qty;
	    }
	    function Qty(initValue, initUnits) {
	      assertValidConstructorArgs.apply(null, arguments);
	      if (!isQty(this)) {
	        return new Qty(initValue, initUnits);
	      }
	      this.scalar = null;
	      this.baseScalar = null;
	      this.signature = null;
	      this._conversionCache = {};
	      this.numerator = UNITY_ARRAY;
	      this.denominator = UNITY_ARRAY;
	      if (isDefinitionObject(initValue)) {
	        this.scalar = initValue.scalar;
	        this.numerator = initValue.numerator && initValue.numerator.length !== 0 ? initValue.numerator : UNITY_ARRAY;
	        this.denominator = initValue.denominator && initValue.denominator.length !== 0 ? initValue.denominator : UNITY_ARRAY;
	      } else if (initUnits) {
	        parse.call(this, initUnits);
	        this.scalar = initValue;
	      } else {
	        parse.call(this, initValue);
	      }

	      // math with temperatures is very limited
	      if (this.denominator.join("*").indexOf("temp") >= 0) {
	        throw new QtyError("Cannot divide with temperatures");
	      }
	      if (this.numerator.join("*").indexOf("temp") >= 0) {
	        if (this.numerator.length > 1) {
	          throw new QtyError("Cannot multiply by temperatures");
	        }
	        if (!compareArray(this.denominator, UNITY_ARRAY)) {
	          throw new QtyError("Cannot divide with temperatures");
	        }
	      }
	      this.initValue = initValue;
	      updateBaseScalar.call(this);
	      if (this.isTemperature() && this.baseScalar < 0) {
	        throw new QtyError("Temperatures must not be less than absolute zero");
	      }
	    }
	    Qty.prototype = {
	      // Properly set up constructor
	      constructor: Qty
	    };

	    /**
	     * Asserts constructor arguments are valid
	     *
	     * @param {*} value - Value to test
	     * @param {string} [units] - Optional units when value is passed as a number
	     *
	     * @returns {void}
	     * @throws {QtyError} if constructor arguments are invalid
	     */
	    function assertValidConstructorArgs(value, units) {
	      if (units) {
	        if (!(isNumber(value) && isString(units))) {
	          throw new QtyError("Only number accepted as initialization value " + "when units are explicitly provided");
	        }
	      } else {
	        if (!(isString(value) || isNumber(value) || isQty(value) || isDefinitionObject(value))) {
	          throw new QtyError("Only string, number or quantity accepted as " + "single initialization value");
	        }
	      }
	    }

	    /**
	     * Tests if a value is a Qty definition object
	     *
	     * @param {*} value - Value to test
	     *
	     * @returns {boolean} true if value is a definition object, false otherwise
	     */
	    function isDefinitionObject(value) {
	      return value && typeof value === "object" && value.hasOwnProperty("scalar");
	    }
	    function updateBaseScalar() {
	      if (this.baseScalar) {
	        return this.baseScalar;
	      }
	      if (this.isBase()) {
	        this.baseScalar = this.scalar;
	        this.signature = unitSignature.call(this);
	      } else {
	        var base = this.toBase();
	        this.baseScalar = base.scalar;
	        this.signature = base.signature;
	      }
	    }
	    var KINDS = {
	      "-312078": "elastance",
	      "-312058": "resistance",
	      "-312038": "inductance",
	      "-152058": "potential",
	      "-152040": "magnetism",
	      "-152038": "magnetism",
	      "-7997": "specific_volume",
	      "-79": "snap",
	      "-59": "jolt",
	      "-39": "acceleration",
	      "-38": "radiation",
	      "-20": "frequency",
	      "-19": "speed",
	      "-18": "viscosity",
	      "-17": "volumetric_flow",
	      "-1": "wavenumber",
	      "0": "unitless",
	      "1": "length",
	      "2": "area",
	      "3": "volume",
	      "20": "time",
	      "400": "temperature",
	      "7941": "yank",
	      "7942": "power",
	      "7959": "pressure",
	      "7961": "force",
	      "7962": "energy",
	      "7979": "viscosity",
	      "7981": "momentum",
	      "7982": "angular_momentum",
	      "7997": "density",
	      "7998": "area_density",
	      "8000": "mass",
	      "152020": "radiation_exposure",
	      "159999": "magnetism",
	      "160000": "current",
	      "160020": "charge",
	      "312058": "conductance",
	      "312078": "capacitance",
	      "3199980": "activity",
	      "3199997": "molar_concentration",
	      "3200000": "substance",
	      "63999998": "illuminance",
	      "64000000": "luminous_power",
	      "1280000000": "currency",
	      "25599999980": "information_rate",
	      "25600000000": "information",
	      "511999999980": "angular_velocity",
	      "512000000000": "angle"
	    };

	    /**
	     * Returns the list of available well-known kinds of units, e.g.
	     * "radiation" or "length".
	     *
	     * @returns {string[]} names of kinds of units
	     */
	    function getKinds() {
	      return uniq(Object.keys(KINDS).map(function (knownSignature) {
	        return KINDS[knownSignature];
	      }));
	    }
	    Qty.prototype.kind = function () {
	      return KINDS[this.signature.toString()];
	    };
	    assign(Qty.prototype, {
	      isDegrees: function () {
	        // signature may not have been calculated yet
	        return (this.signature === null || this.signature === 400) && this.numerator.length === 1 && compareArray(this.denominator, UNITY_ARRAY) && (this.numerator[0].match(/<temp-[CFRK]>/) || this.numerator[0].match(/<(kelvin|celsius|rankine|fahrenheit)>/));
	      },
	      isTemperature: function () {
	        return this.isDegrees() && this.numerator[0].match(/<temp-[CFRK]>/);
	      }
	    });
	    function subtractTemperatures(lhs, rhs) {
	      var lhsUnits = lhs.units();
	      var rhsConverted = rhs.to(lhsUnits);
	      var dstDegrees = Qty(getDegreeUnits(lhsUnits));
	      return Qty({
	        "scalar": lhs.scalar - rhsConverted.scalar,
	        "numerator": dstDegrees.numerator,
	        "denominator": dstDegrees.denominator
	      });
	    }
	    function subtractTempDegrees(temp, deg) {
	      var tempDegrees = deg.to(getDegreeUnits(temp.units()));
	      return Qty({
	        "scalar": temp.scalar - tempDegrees.scalar,
	        "numerator": temp.numerator,
	        "denominator": temp.denominator
	      });
	    }
	    function addTempDegrees(temp, deg) {
	      var tempDegrees = deg.to(getDegreeUnits(temp.units()));
	      return Qty({
	        "scalar": temp.scalar + tempDegrees.scalar,
	        "numerator": temp.numerator,
	        "denominator": temp.denominator
	      });
	    }
	    function getDegreeUnits(units) {
	      if (units === "tempK") {
	        return "degK";
	      } else if (units === "tempC") {
	        return "degC";
	      } else if (units === "tempF") {
	        return "degF";
	      } else if (units === "tempR") {
	        return "degR";
	      } else {
	        throw new QtyError("Unknown type for temp conversion from: " + units);
	      }
	    }
	    function toDegrees(src, dst) {
	      var srcDegK = toDegK(src);
	      var dstUnits = dst.units();
	      var dstScalar;
	      if (dstUnits === "degK") {
	        dstScalar = srcDegK.scalar;
	      } else if (dstUnits === "degC") {
	        dstScalar = srcDegK.scalar;
	      } else if (dstUnits === "degF") {
	        dstScalar = srcDegK.scalar * 9 / 5;
	      } else if (dstUnits === "degR") {
	        dstScalar = srcDegK.scalar * 9 / 5;
	      } else {
	        throw new QtyError("Unknown type for degree conversion to: " + dstUnits);
	      }
	      return Qty({
	        "scalar": dstScalar,
	        "numerator": dst.numerator,
	        "denominator": dst.denominator
	      });
	    }
	    function toDegK(qty) {
	      var units = qty.units();
	      var q;
	      if (units.match(/(deg)[CFRK]/)) {
	        q = qty.baseScalar;
	      } else if (units === "tempK") {
	        q = qty.scalar;
	      } else if (units === "tempC") {
	        q = qty.scalar;
	      } else if (units === "tempF") {
	        q = qty.scalar * 5 / 9;
	      } else if (units === "tempR") {
	        q = qty.scalar * 5 / 9;
	      } else {
	        throw new QtyError("Unknown type for temp conversion from: " + units);
	      }
	      return Qty({
	        "scalar": q,
	        "numerator": ["<kelvin>"],
	        "denominator": UNITY_ARRAY
	      });
	    }
	    function toTemp(src, dst) {
	      var dstUnits = dst.units();
	      var dstScalar;
	      if (dstUnits === "tempK") {
	        dstScalar = src.baseScalar;
	      } else if (dstUnits === "tempC") {
	        dstScalar = src.baseScalar - 273.15;
	      } else if (dstUnits === "tempF") {
	        dstScalar = src.baseScalar * 9 / 5 - 459.67;
	      } else if (dstUnits === "tempR") {
	        dstScalar = src.baseScalar * 9 / 5;
	      } else {
	        throw new QtyError("Unknown type for temp conversion to: " + dstUnits);
	      }
	      return Qty({
	        "scalar": dstScalar,
	        "numerator": dst.numerator,
	        "denominator": dst.denominator
	      });
	    }
	    function toTempK(qty) {
	      var units = qty.units();
	      var q;
	      if (units.match(/(deg)[CFRK]/)) {
	        q = qty.baseScalar;
	      } else if (units === "tempK") {
	        q = qty.scalar;
	      } else if (units === "tempC") {
	        q = qty.scalar + 273.15;
	      } else if (units === "tempF") {
	        q = (qty.scalar + 459.67) * 5 / 9;
	      } else if (units === "tempR") {
	        q = qty.scalar * 5 / 9;
	      } else {
	        throw new QtyError("Unknown type for temp conversion from: " + units);
	      }
	      return Qty({
	        "scalar": q,
	        "numerator": ["<temp-K>"],
	        "denominator": UNITY_ARRAY
	      });
	    }
	    assign(Qty.prototype, {
	      /**
	       * Converts to other compatible units.
	       * Instance's converted quantities are cached for faster subsequent calls.
	       *
	       * @param {(string|Qty)} other - Target units as string or retrieved from
	       *                               other Qty instance (scalar is ignored)
	       *
	       * @returns {Qty} New converted Qty instance with target units
	       *
	       * @throws {QtyError} if target units are incompatible
	       *
	       * @example
	       * var weight = Qty("25 kg");
	       * weight.to("lb"); // => Qty("55.11556554621939 lbs");
	       * weight.to(Qty("3 g")); // => Qty("25000 g"); // scalar of passed Qty is ignored
	       */
	      to: function (other) {
	        var cached, target;
	        if (other === undefined || other === null) {
	          return this;
	        }
	        if (!isString(other)) {
	          return this.to(other.units());
	        }
	        cached = this._conversionCache[other];
	        if (cached) {
	          return cached;
	        }

	        // Instantiating target to normalize units
	        target = Qty(other);
	        if (target.units() === this.units()) {
	          return this;
	        }
	        if (!this.isCompatible(target)) {
	          if (this.isInverse(target)) {
	            target = this.inverse().to(other);
	          } else {
	            throwIncompatibleUnits(this.units(), target.units());
	          }
	        } else {
	          if (target.isTemperature()) {
	            target = toTemp(this, target);
	          } else if (target.isDegrees()) {
	            target = toDegrees(this, target);
	          } else {
	            var q = divSafe(this.baseScalar, target.baseScalar);
	            target = Qty({
	              "scalar": q,
	              "numerator": target.numerator,
	              "denominator": target.denominator
	            });
	          }
	        }
	        this._conversionCache[other] = target;
	        return target;
	      },
	      // convert to base SI units
	      // results of the conversion are cached so subsequent calls to this will be fast
	      toBase: function () {
	        if (this.isBase()) {
	          return this;
	        }
	        if (this.isTemperature()) {
	          return toTempK(this);
	        }
	        var cached = baseUnitCache[this.units()];
	        if (!cached) {
	          cached = toBaseUnits(this.numerator, this.denominator);
	          baseUnitCache[this.units()] = cached;
	        }
	        return cached.mul(this.scalar);
	      },
	      // Converts the unit back to a float if it is unitless.  Otherwise raises an exception
	      toFloat: function () {
	        if (this.isUnitless()) {
	          return this.scalar;
	        }
	        throw new QtyError("Can't convert to Float unless unitless.  Use Unit#scalar");
	      },
	      /**
	       * Returns the nearest multiple of quantity passed as
	       * precision
	       *
	       * @param {(Qty|string|number)} precQuantity - Quantity, string formated
	       *   quantity or number as expected precision
	       *
	       * @returns {Qty} Nearest multiple of precQuantity
	       *
	       * @example
	       * Qty('5.5 ft').toPrec('2 ft'); // returns 6 ft
	       * Qty('0.8 cu').toPrec('0.25 cu'); // returns 0.75 cu
	       * Qty('6.3782 m').toPrec('cm'); // returns 6.38 m
	       * Qty('1.146 MPa').toPrec('0.1 bar'); // returns 1.15 MPa
	       *
	       */
	      toPrec: function (precQuantity) {
	        if (isString(precQuantity)) {
	          precQuantity = Qty(precQuantity);
	        }
	        if (isNumber(precQuantity)) {
	          precQuantity = Qty(precQuantity + " " + this.units());
	        }
	        if (!this.isUnitless()) {
	          precQuantity = precQuantity.to(this.units());
	        } else if (!precQuantity.isUnitless()) {
	          throwIncompatibleUnits(this.units(), precQuantity.units());
	        }
	        if (precQuantity.scalar === 0) {
	          throw new QtyError("Divide by zero");
	        }
	        var precRoundedResult = mulSafe(Math.round(this.scalar / precQuantity.scalar), precQuantity.scalar);
	        return Qty(precRoundedResult + this.units());
	      }
	    });

	    /**
	     * Configures and returns a fast function to convert
	     * Number values from units to others.
	     * Useful to efficiently convert large array of values
	     * with same units into others with iterative methods.
	     * Does not take care of rounding issues.
	     *
	     * @param {string} srcUnits Units of values to convert
	     * @param {string} dstUnits Units to convert to
	     *
	     * @returns {Function} Converting function accepting Number value
	     *   and returning converted value
	     *
	     * @throws "Incompatible units" if units are incompatible
	     *
	     * @example
	     * // Converting large array of numbers with the same units
	     * // into other units
	     * var converter = Qty.swiftConverter("m/h", "ft/s");
	     * var convertedSerie = largeSerie.map(converter);
	     *
	     */
	    function swiftConverter(srcUnits, dstUnits) {
	      var srcQty = Qty(srcUnits);
	      var dstQty = Qty(dstUnits);
	      if (srcQty.eq(dstQty)) {
	        return identity;
	      }
	      var convert;
	      if (!srcQty.isTemperature()) {
	        convert = function (value) {
	          return value * srcQty.baseScalar / dstQty.baseScalar;
	        };
	      } else {
	        convert = function (value) {
	          // TODO Not optimized
	          return srcQty.mul(value).to(dstQty).scalar;
	        };
	      }
	      return function converter(value) {
	        var i, length, result;
	        if (!Array.isArray(value)) {
	          return convert(value);
	        } else {
	          length = value.length;
	          result = [];
	          for (i = 0; i < length; i++) {
	            result.push(convert(value[i]));
	          }
	          return result;
	        }
	      };
	    }
	    var baseUnitCache = {};
	    function toBaseUnits(numerator, denominator) {
	      var num = [];
	      var den = [];
	      var q = 1;
	      var unit;
	      for (var i = 0; i < numerator.length; i++) {
	        unit = numerator[i];
	        if (PREFIX_VALUES[unit]) {
	          // workaround to fix
	          // 0.1 * 0.1 => 0.010000000000000002
	          q = mulSafe(q, PREFIX_VALUES[unit]);
	        } else {
	          if (UNIT_VALUES[unit]) {
	            q *= UNIT_VALUES[unit].scalar;
	            if (UNIT_VALUES[unit].numerator) {
	              num.push(UNIT_VALUES[unit].numerator);
	            }
	            if (UNIT_VALUES[unit].denominator) {
	              den.push(UNIT_VALUES[unit].denominator);
	            }
	          }
	        }
	      }
	      for (var j = 0; j < denominator.length; j++) {
	        unit = denominator[j];
	        if (PREFIX_VALUES[unit]) {
	          q /= PREFIX_VALUES[unit];
	        } else {
	          if (UNIT_VALUES[unit]) {
	            q /= UNIT_VALUES[unit].scalar;
	            if (UNIT_VALUES[unit].numerator) {
	              den.push(UNIT_VALUES[unit].numerator);
	            }
	            if (UNIT_VALUES[unit].denominator) {
	              num.push(UNIT_VALUES[unit].denominator);
	            }
	          }
	        }
	      }

	      // Flatten
	      num = num.reduce(function (a, b) {
	        return a.concat(b);
	      }, []);
	      den = den.reduce(function (a, b) {
	        return a.concat(b);
	      }, []);
	      return Qty({
	        "scalar": q,
	        "numerator": num,
	        "denominator": den
	      });
	    }
	    Qty.parse = globalParse;
	    Qty.getUnits = getUnits;
	    Qty.getAliases = getAliases;
	    Qty.mulSafe = mulSafe;
	    Qty.divSafe = divSafe;
	    Qty.getKinds = getKinds;
	    Qty.swiftConverter = swiftConverter;
	    Qty.Error = QtyError;
	    assign(Qty.prototype, {
	      // Returns new instance with units of this
	      add: function (other) {
	        if (isString(other)) {
	          other = Qty(other);
	        }
	        if (!this.isCompatible(other)) {
	          throwIncompatibleUnits(this.units(), other.units());
	        }
	        if (this.isTemperature() && other.isTemperature()) {
	          throw new QtyError("Cannot add two temperatures");
	        } else if (this.isTemperature()) {
	          return addTempDegrees(this, other);
	        } else if (other.isTemperature()) {
	          return addTempDegrees(other, this);
	        }
	        return Qty({
	          "scalar": this.scalar + other.to(this).scalar,
	          "numerator": this.numerator,
	          "denominator": this.denominator
	        });
	      },
	      sub: function (other) {
	        if (isString(other)) {
	          other = Qty(other);
	        }
	        if (!this.isCompatible(other)) {
	          throwIncompatibleUnits(this.units(), other.units());
	        }
	        if (this.isTemperature() && other.isTemperature()) {
	          return subtractTemperatures(this, other);
	        } else if (this.isTemperature()) {
	          return subtractTempDegrees(this, other);
	        } else if (other.isTemperature()) {
	          throw new QtyError("Cannot subtract a temperature from a differential degree unit");
	        }
	        return Qty({
	          "scalar": this.scalar - other.to(this).scalar,
	          "numerator": this.numerator,
	          "denominator": this.denominator
	        });
	      },
	      mul: function (other) {
	        if (isNumber(other)) {
	          return Qty({
	            "scalar": mulSafe(this.scalar, other),
	            "numerator": this.numerator,
	            "denominator": this.denominator
	          });
	        } else if (isString(other)) {
	          other = Qty(other);
	        }
	        if ((this.isTemperature() || other.isTemperature()) && !(this.isUnitless() || other.isUnitless())) {
	          throw new QtyError("Cannot multiply by temperatures");
	        }

	        // Quantities should be multiplied with same units if compatible, with base units else
	        var op1 = this;
	        var op2 = other;

	        // so as not to confuse results, multiplication and division between temperature degrees will maintain original unit info in num/den
	        // multiplication and division between deg[CFRK] can never factor each other out, only themselves: "degK*degC/degC^2" == "degK/degC"
	        if (op1.isCompatible(op2) && op1.signature !== 400) {
	          op2 = op2.to(op1);
	        }
	        var numdenscale = cleanTerms(op1.numerator, op1.denominator, op2.numerator, op2.denominator);
	        return Qty({
	          "scalar": mulSafe(op1.scalar, op2.scalar, numdenscale[2]),
	          "numerator": numdenscale[0],
	          "denominator": numdenscale[1]
	        });
	      },
	      div: function (other) {
	        if (isNumber(other)) {
	          if (other === 0) {
	            throw new QtyError("Divide by zero");
	          }
	          return Qty({
	            "scalar": this.scalar / other,
	            "numerator": this.numerator,
	            "denominator": this.denominator
	          });
	        } else if (isString(other)) {
	          other = Qty(other);
	        }
	        if (other.scalar === 0) {
	          throw new QtyError("Divide by zero");
	        }
	        if (other.isTemperature()) {
	          throw new QtyError("Cannot divide with temperatures");
	        } else if (this.isTemperature() && !other.isUnitless()) {
	          throw new QtyError("Cannot divide with temperatures");
	        }

	        // Quantities should be multiplied with same units if compatible, with base units else
	        var op1 = this;
	        var op2 = other;

	        // so as not to confuse results, multiplication and division between temperature degrees will maintain original unit info in num/den
	        // multiplication and division between deg[CFRK] can never factor each other out, only themselves: "degK*degC/degC^2" == "degK/degC"
	        if (op1.isCompatible(op2) && op1.signature !== 400) {
	          op2 = op2.to(op1);
	        }
	        var numdenscale = cleanTerms(op1.numerator, op1.denominator, op2.denominator, op2.numerator);
	        return Qty({
	          "scalar": mulSafe(op1.scalar, numdenscale[2]) / op2.scalar,
	          "numerator": numdenscale[0],
	          "denominator": numdenscale[1]
	        });
	      },
	      // Returns a Qty that is the inverse of this Qty,
	      inverse: function () {
	        if (this.isTemperature()) {
	          throw new QtyError("Cannot divide with temperatures");
	        }
	        if (this.scalar === 0) {
	          throw new QtyError("Divide by zero");
	        }
	        return Qty({
	          "scalar": 1 / this.scalar,
	          "numerator": this.denominator,
	          "denominator": this.numerator
	        });
	      }
	    });
	    function cleanTerms(num1, den1, num2, den2) {
	      function notUnity(val) {
	        return val !== UNITY;
	      }
	      num1 = num1.filter(notUnity);
	      num2 = num2.filter(notUnity);
	      den1 = den1.filter(notUnity);
	      den2 = den2.filter(notUnity);
	      var combined = {};
	      function combineTerms(terms, direction) {
	        var k;
	        var prefix;
	        var prefixValue;
	        for (var i = 0; i < terms.length; i++) {
	          if (PREFIX_VALUES[terms[i]]) {
	            k = terms[i + 1];
	            prefix = terms[i];
	            prefixValue = PREFIX_VALUES[prefix];
	            i++;
	          } else {
	            k = terms[i];
	            prefix = null;
	            prefixValue = 1;
	          }
	          if (k && k !== UNITY) {
	            if (combined[k]) {
	              combined[k][0] += direction;
	              var combinedPrefixValue = combined[k][2] ? PREFIX_VALUES[combined[k][2]] : 1;
	              combined[k][direction === 1 ? 3 : 4] *= divSafe(prefixValue, combinedPrefixValue);
	            } else {
	              combined[k] = [direction, k, prefix, 1, 1];
	            }
	          }
	        }
	      }
	      combineTerms(num1, 1);
	      combineTerms(den1, -1);
	      combineTerms(num2, 1);
	      combineTerms(den2, -1);
	      var num = [];
	      var den = [];
	      var scale = 1;
	      for (var prop in combined) {
	        if (combined.hasOwnProperty(prop)) {
	          var item = combined[prop];
	          var n;
	          if (item[0] > 0) {
	            for (n = 0; n < item[0]; n++) {
	              num.push(item[2] === null ? item[1] : [item[2], item[1]]);
	            }
	          } else if (item[0] < 0) {
	            for (n = 0; n < -item[0]; n++) {
	              den.push(item[2] === null ? item[1] : [item[2], item[1]]);
	            }
	          }
	          scale *= divSafe(item[3], item[4]);
	        }
	      }
	      if (num.length === 0) {
	        num = UNITY_ARRAY;
	      }
	      if (den.length === 0) {
	        den = UNITY_ARRAY;
	      }

	      // Flatten
	      num = num.reduce(function (a, b) {
	        return a.concat(b);
	      }, []);
	      den = den.reduce(function (a, b) {
	        return a.concat(b);
	      }, []);
	      return [num, den, scale];
	    }
	    assign(Qty.prototype, {
	      eq: function (other) {
	        return this.compareTo(other) === 0;
	      },
	      lt: function (other) {
	        return this.compareTo(other) === -1;
	      },
	      lte: function (other) {
	        return this.eq(other) || this.lt(other);
	      },
	      gt: function (other) {
	        return this.compareTo(other) === 1;
	      },
	      gte: function (other) {
	        return this.eq(other) || this.gt(other);
	      },
	      // Compare two Qty objects. Throws an exception if they are not of compatible types.
	      // Comparisons are done based on the value of the quantity in base SI units.
	      //
	      // NOTE: We cannot compare inverses as that breaks the general compareTo contract:
	      //   if a.compareTo(b) < 0 then b.compareTo(a) > 0
	      //   if a.compareTo(b) == 0 then b.compareTo(a) == 0
	      //
	      //   Since "10S" == ".1ohm" (10 > .1) and "10ohm" == ".1S" (10 > .1)
	      //     Qty("10S").inverse().compareTo("10ohm") == -1
	      //     Qty("10ohm").inverse().compareTo("10S") == -1
	      //
	      //   If including inverses in the sort is needed, I suggest writing: Qty.sort(qtyArray,units)
	      compareTo: function (other) {
	        if (isString(other)) {
	          return this.compareTo(Qty(other));
	        }
	        if (!this.isCompatible(other)) {
	          throwIncompatibleUnits(this.units(), other.units());
	        }
	        if (this.baseScalar < other.baseScalar) {
	          return -1;
	        } else if (this.baseScalar === other.baseScalar) {
	          return 0;
	        } else if (this.baseScalar > other.baseScalar) {
	          return 1;
	        }
	      },
	      // Return true if quantities and units match
	      // Unit("100 cm").same(Unit("100 cm"))  # => true
	      // Unit("100 cm").same(Unit("1 m"))     # => false
	      same: function (other) {
	        return this.scalar === other.scalar && this.units() === other.units();
	      }
	    });
	    assign(Qty.prototype, {
	      // returns true if no associated units
	      // false, even if the units are "unitless" like 'radians, each, etc'
	      isUnitless: function () {
	        return [this.numerator, this.denominator].every(function (item) {
	          return compareArray(item, UNITY_ARRAY);
	        });
	      },
	      /*
	      check to see if units are compatible, but not the scalar part
	      this check is done by comparing signatures for performance reasons
	      if passed a string, it will create a unit object with the string and then do the comparison
	      this permits a syntax like:
	      unit =~ "mm"
	      if you want to do a regexp on the unit string do this ...
	      unit.units =~ /regexp/
	      */
	      isCompatible: function (other) {
	        if (isString(other)) {
	          return this.isCompatible(Qty(other));
	        }
	        if (!isQty(other)) {
	          return false;
	        }
	        if (other.signature !== undefined) {
	          return this.signature === other.signature;
	        } else {
	          return false;
	        }
	      },
	      /*
	      check to see if units are inverse of each other, but not the scalar part
	      this check is done by comparing signatures for performance reasons
	      if passed a string, it will create a unit object with the string and then do the comparison
	      this permits a syntax like:
	      unit =~ "mm"
	      if you want to do a regexp on the unit string do this ...
	      unit.units =~ /regexp/
	      */
	      isInverse: function (other) {
	        return this.inverse().isCompatible(other);
	      },
	      // Returns 'true' if the Unit is represented in base units
	      isBase: function () {
	        if (this._isBase !== undefined) {
	          return this._isBase;
	        }
	        if (this.isDegrees() && this.numerator[0].match(/<(kelvin|temp-K)>/)) {
	          this._isBase = true;
	          return this._isBase;
	        }
	        this.numerator.concat(this.denominator).forEach(function (item) {
	          if (item !== UNITY && BASE_UNITS.indexOf(item) === -1) {
	            this._isBase = false;
	          }
	        }, this);
	        if (this._isBase === false) {
	          return this._isBase;
	        }
	        this._isBase = true;
	        return this._isBase;
	      }
	    });
	    function NestedMap() {}
	    NestedMap.prototype.get = function (keys) {
	      // Allows to pass key1, key2, ... instead of [key1, key2, ...]
	      if (arguments.length > 1) {
	        // Slower with Firefox but faster with Chrome than
	        // Array.prototype.slice.call(arguments)
	        // See http://jsperf.com/array-apply-versus-array-prototype-slice-call
	        keys = Array.apply(null, arguments);
	      }
	      return keys.reduce(function (map, key, index) {
	        if (map) {
	          var childMap = map[key];
	          if (index === keys.length - 1) {
	            return childMap ? childMap.data : undefined;
	          } else {
	            return childMap;
	          }
	        }
	      }, this);
	    };
	    NestedMap.prototype.set = function (keys, value) {
	      if (arguments.length > 2) {
	        keys = Array.prototype.slice.call(arguments, 0, -1);
	        value = arguments[arguments.length - 1];
	      }
	      return keys.reduce(function (map, key, index) {
	        var childMap = map[key];
	        if (childMap === undefined) {
	          childMap = map[key] = {};
	        }
	        if (index === keys.length - 1) {
	          childMap.data = value;
	          return value;
	        } else {
	          return childMap;
	        }
	      }, this);
	    };

	    /**
	     * Default formatter
	     *
	     * @param {number} scalar - scalar value
	     * @param {string} units - units as string
	     *
	     * @returns {string} formatted result
	     */
	    function defaultFormatter(scalar, units) {
	      return (scalar + " " + units).trim();
	    }

	    /**
	     *
	     * Configurable Qty default formatter
	     *
	     * @type {function}
	     *
	     * @param {number} scalar
	     * @param {string} units
	     *
	     * @returns {string} formatted result
	     */
	    Qty.formatter = defaultFormatter;
	    assign(Qty.prototype, {
	      // returns the 'unit' part of the Unit object without the scalar
	      units: function () {
	        if (this._units !== undefined) {
	          return this._units;
	        }
	        var numIsUnity = compareArray(this.numerator, UNITY_ARRAY);
	        var denIsUnity = compareArray(this.denominator, UNITY_ARRAY);
	        if (numIsUnity && denIsUnity) {
	          this._units = "";
	          return this._units;
	        }
	        var numUnits = stringifyUnits(this.numerator);
	        var denUnits = stringifyUnits(this.denominator);
	        this._units = numUnits + (denIsUnity ? "" : "/" + denUnits);
	        return this._units;
	      },
	      /**
	       * Stringifies the quantity
	       * Deprecation notice: only units parameter is supported.
	       *
	       * @param {(number|string|Qty)} targetUnitsOrMaxDecimalsOrPrec -
	       *                              target units if string,
	       *                              max number of decimals if number,
	       *                              passed to #toPrec before converting if Qty
	       *
	       * @param {number=} maxDecimals - Maximum number of decimals of
	       *                                formatted output
	       *
	       * @returns {string} reparseable quantity as string
	       */
	      toString: function (targetUnitsOrMaxDecimalsOrPrec, maxDecimals) {
	        var targetUnits;
	        if (isNumber(targetUnitsOrMaxDecimalsOrPrec)) {
	          targetUnits = this.units();
	          maxDecimals = targetUnitsOrMaxDecimalsOrPrec;
	        } else if (isString(targetUnitsOrMaxDecimalsOrPrec)) {
	          targetUnits = targetUnitsOrMaxDecimalsOrPrec;
	        } else if (isQty(targetUnitsOrMaxDecimalsOrPrec)) {
	          return this.toPrec(targetUnitsOrMaxDecimalsOrPrec).toString(maxDecimals);
	        }
	        var out = this.to(targetUnits);
	        var outScalar = maxDecimals !== undefined ? round(out.scalar, maxDecimals) : out.scalar;
	        out = (outScalar + " " + out.units()).trim();
	        return out;
	      },
	      /**
	       * Format the quantity according to optional passed target units
	       * and formatter
	       *
	       * @param {string} [targetUnits=current units] -
	       *                 optional units to convert to before formatting
	       *
	       * @param {function} [formatter=Qty.formatter] -
	       *                   delegates formatting to formatter callback.
	       *                   formatter is called back with two parameters (scalar, units)
	       *                   and should return formatted result.
	       *                   If unspecified, formatting is delegated to default formatter
	       *                   set to Qty.formatter
	       *
	       * @example
	       * var roundingAndLocalizingFormatter = function(scalar, units) {
	       *   // localize or limit scalar to n max decimals for instance
	       *   // return formatted result
	       * };
	       * var qty = Qty('1.1234 m');
	       * qty.format(); // same units, default formatter => "1.234 m"
	       * qty.format("cm"); // converted to "cm", default formatter => "123.45 cm"
	       * qty.format(roundingAndLocalizingFormatter); // same units, custom formatter => "1,2 m"
	       * qty.format("cm", roundingAndLocalizingFormatter); // convert to "cm", custom formatter => "123,4 cm"
	       *
	       * @returns {string} quantity as string
	       */
	      format: function (targetUnits, formatter) {
	        if (arguments.length === 1) {
	          if (typeof targetUnits === "function") {
	            formatter = targetUnits;
	            targetUnits = undefined;
	          }
	        }
	        formatter = formatter || Qty.formatter;
	        var targetQty = this.to(targetUnits);
	        return formatter.call(this, targetQty.scalar, targetQty.units());
	      }
	    });
	    var stringifiedUnitsCache = new NestedMap();
	    /**
	     * Returns a string representing a normalized unit array
	     *
	     * @param {string[]} units Normalized unit array
	     * @returns {string} String representing passed normalized unit array and
	     *   suitable for output
	     *
	     */
	    function stringifyUnits(units) {
	      var stringified = stringifiedUnitsCache.get(units);
	      if (stringified) {
	        return stringified;
	      }
	      var isUnity = compareArray(units, UNITY_ARRAY);
	      if (isUnity) {
	        stringified = "1";
	      } else {
	        stringified = simplify(getOutputNames(units)).join("*");
	      }

	      // Cache result
	      stringifiedUnitsCache.set(units, stringified);
	      return stringified;
	    }
	    function getOutputNames(units) {
	      var unitNames = [],
	        token,
	        tokenNext;
	      for (var i = 0; i < units.length; i++) {
	        token = units[i];
	        tokenNext = units[i + 1];
	        if (PREFIX_VALUES[token]) {
	          unitNames.push(OUTPUT_MAP[token] + OUTPUT_MAP[tokenNext]);
	          i++;
	        } else {
	          unitNames.push(OUTPUT_MAP[token]);
	        }
	      }
	      return unitNames;
	    }
	    function simplify(units) {
	      // this turns ['s','m','s'] into ['s2','m']

	      var unitCounts = units.reduce(function (acc, unit) {
	        var unitCounter = acc[unit];
	        if (!unitCounter) {
	          acc.push(unitCounter = acc[unit] = [unit, 0]);
	        }
	        unitCounter[1]++;
	        return acc;
	      }, []);
	      return unitCounts.map(function (unitCount) {
	        return unitCount[0] + (unitCount[1] > 1 ? unitCount[1] : "");
	      });
	    }
	    Qty.version = "1.8.0";
	    return Qty;
	  });
	})(quantities);
	var Qty = quantities.exports;

	/**
	 * Parse a string that may contain min / max and units values
	 * @param {string} string
	 * @param {object} [options={}]
	 * @param {number} [options.defaultValue]
	 * @param {string} [options.defjaultUnits]
	 * @param {string} [options.targetUnits]
	 * @param {string} [options.optional=false] Is this value optional. Would through an error if empty and not optional
	 */
	function parseNumbersUnits(string = '', options = {}) {
	  const {
	    defaultUnits = undefined,
	    targetUnits = undefined,
	    defaultValue = undefined,
	    optional = false
	  } = options;
	  let normalized = String(string).replace(/ /g, '').replace(/,/g, '.');
	  normalized = normalized.replace(/Â°/g, '°');
	  if (normalized.match(/^\(.*\)$/)) {
	    normalized = normalized.substring(1, normalized.length - 2);
	  }

	  // still parenthesis ??? We remove what is after
	  normalized = normalized.replace(/\(.*/, '');
	  if (optional && !normalized) return {};
	  let splitter = /^([0-9.Ee-]*)([a-zA-Z^/ °0-9]*)$/;
	  if (!normalized.match(splitter)) {
	    throw Error(`Can not parseNumbersUnits of: ${string}`);
	  }
	  let numbers = normalized.replace(splitter, '$1');
	  let units = normalized.replace(splitter, '$2').trim();
	  let low;
	  let high;
	  if (numbers === '') {
	    if (defaultValue) {
	      numbers = String(defaultValue).replace(/ /g, '');
	    } else {
	      throw Error(`Can not parseNumbersUnits of: ${string}`);
	    }
	  }
	  if (numbers.includes('--')) {
	    // -5--4
	    low = Number(numbers.split('--')[0]);
	    high = Number(`-${numbers.split('--')[1]}`);
	  } else if (numbers.startsWith('-')) {
	    // -5 or -1-1 (minus 1 to plus 1)
	    let subparts = numbers.split('-');
	    switch (subparts.length) {
	      case 2:
	        low = Number(numbers);
	        break;
	      case 3:
	        low = Number(`-${subparts[1]}`);
	        high = Number(+subparts[2]);
	        break;
	      default:
	        throw new Error(`Can not parse: ${string}`);
	    }
	  } else {
	    let subparts = numbers.split('-');
	    low = subparts[0] !== '' ? Number(subparts[0]) : undefined;
	    high = subparts[1] !== undefined ? Number(subparts[1]) : undefined;
	  }
	  if (!units) units = defaultUnits;

	  // Should we convert the units
	  if (targetUnits) {
	    let convert = Qty.swiftConverter(units.replace(/°/g, 'temp'), targetUnits.replace(/°/g, 'temp'));
	    if (low !== undefined) {
	      low = convert(low);
	    }
	    if (high !== undefined) {
	      high = convert(high);
	    }
	    units = targetUnits;
	  }
	  return {
	    low,
	    high,
	    units
	  };
	}

	/**
	 *
	 * @param {String} string
	 * @param {object} [options={}]
	 * @param {object} [options.temperature={}]
	 * @param {string} [options.temperature.defaultUnits]
	 * @param {string} [options.temperature.targetUnits]
	 * @param {object} [options.pressure={}]
	 * @param {number} [options.pressure.defaultValue]
	 * @param {string} [options.pressure.defaultUnits]
	 * @param {string} [options.pressure.targetUnits]
	 * @param {string} [options.pressure.optional]
	 */
	function parseBoilingPoint(string, options = {}) {
	  let parts = String(string).split(/[@/]|, | at /);
	  return {
	    temperature: parseNumbersUnits(parts[0], options.temperature),
	    pressure: parseNumbersUnits(parts[1], options.pressure)
	  };
	}

	/**
	 *
	 * @param {String} string
	 * @param {object} [options={}]
	 * @param {object} [options.value={}]
	 * @param {string} [options.value.defaultUnits]
	 * @param {string} [options.value.targetUnits]
	 * @param {object} [options.temperature={}]
	 * @param {number} [options.temperature.defaultValue]
	 * @param {string} [options.temperature.defaultUnits]
	 * @param {string} [options.temperature.targetUnits]
	 */
	function parseDensity(string, options = {}) {
	  string = String(string);
	  string = string.replace(/\(.*/, '');
	  string = string.replace('g/cu cm', 'g/cm^3');
	  let parts = String(string).split(/[@]|, | at /);
	  let value = parts[0];
	  let temperature = parts[1];
	  if (temperature) {
	    temperature = temperature.replace(/ *t= */, '');
	  }
	  return {
	    value: parseNumbersUnits(value, options.value),
	    temperature: parseNumbersUnits(temperature, options.temperature)
	  };
	}

	/**
	 *
	 * @param {String} string
	 * @param {object} [options={}]
	 * @param {object} [options.value={}]
	 * @param {string} [options.value.defaultUnits]
	 * @param {string} [options.value.targetUnits]
	 * @param {object} [options.temperature={}]
	 * @param {number} [options.temperature.defaultValue]
	 * @param {string} [options.temperature.defaultUnits]
	 * @param {string} [options.temperature.targetUnits]
	 */
	function parseRefractiveIndex(string, options = {}) {
	  string = String(string).replace(/^[^0-9.]*/, '');
	  let parts = string.split(/at|[@/]|, /);
	  let value = parts[0];
	  let temperature = parts[1];
	  if (temperature) {
	    temperature = temperature.replace(/ *t= */, '');
	  }
	  return {
	    value: parseNumbersUnits(value, options.value),
	    temperature: parseNumbersUnits(temperature, options.temperature)
	  };
	}

	/**
	 *
	 * @param {String} string
	 * @param {object} [options={}]
	 * @param {object} [options.temperature={}]
	 * @param {string} [options.temperature.defaultUnits]
	 * @param {string} [options.temperature.targetUnits]
	 * @param {string} [options.pressure.optional]
	 * @param {object} [options.pressure={}]
	 * @param {string} [options.pressure.defaultUnits]
	 * @param {string} [options.pressure.targetUnits]

	 */
	function parseVaporPressure(string, options = {}) {
	  let parts = String(string).split(/[@/]|, | at /);
	  return {
	    pressure: parseNumbersUnits(parts[0], options.pressure),
	    temperature: parseNumbersUnits(parts[1], options.temperature)
	  };
	}

	/**
	 * Returns all references cited in the PubChem entry
	 *
	 * @param data Data of a compound data request to the PubChem API
	 * @returns References
	 */
	function getReferences(data) {
	  const references = jp.query(data, '$.Reference[*]').reduce((ref, entry) => {
	    ref[entry.ReferenceNumber] = {
	      url: entry.URL,
	      sourceName: entry.SourceName,
	      name: entry.Name,
	      description: entry.Description
	    };
	    return ref;
	  }, {});
	  return references;
	}

	function extractExperimentalData(data, sectionName, options = {}) {
	  const {
	    parser
	  } = options;
	  const references = getReferences(data);
	  const experimentalSection = getExperimentalDataSection(data);
	  const results = [];
	  if (!experimentalSection || experimentalSection.length === 0) {
	    return results;
	  }
	  jp.query(experimentalSection[0], `$.Section[?(@.TOCHeading==="${sectionName}")].Information[*]`).forEach(entry => {
	    jp.query(entry, '$.Value.StringWithMarkup[*].String').forEach(original => {
	      const result = {
	        reference: references[entry.ReferenceNumber],
	        data: {
	          original
	        }
	      };
	      if (parser) {
	        try {
	          result.data.parsed = parser(original, options);
	        } catch (e) {
	          result.data.parsed = {
	            error: e.toString()
	          };
	        }
	      }
	      results.push(result);
	    });
	  });
	  return results;
	}
	function getExperimentalDataSection(data) {
	  const experimentalData = jp.query(data, '$.Section[?(@.TOCHeading==="Chemical and Physical Properties")]' + '.Section[?(@.TOCHeading==="Experimental Properties")]');
	  return experimentalData;
	}

	/**
	 * Returns the boiling point of a compound
	 *
	 * @param data Data of a compound data request to the PubChem API
	 * @param options Options for the compound
	 * @returns ExperimentalData
	 */
	function getBoilingPoint(data, options = {}) {
	  const boilingPoint = extractExperimentalData(data, 'Boiling Point', {
	    parser: value => parseBoilingPoint(value, merge({
	      pressure: {
	        defaultValue: 760,
	        defaultUnits: 'torr'
	      }
	    }, options))
	  });
	  return boilingPoint;
	}

	/**
	 * Returns the refractive index of a compound
	 * @param data Data of a compound data request to the PubChem API
	 * @returns ExperimentalData
	 */
	function getDensity(data = {}, options = {}) {
	  const density = extractExperimentalData(data, 'Density', {
	    parser: value => {
	      return parseDensity(value, options);
	    }
	  });
	  return density;
	}

	/**
	 *
	 *
	 * @export
	 * @param {Object} data response of a compound data request to the PubChem API
	 * @param {object} [options={}]
	 * @param {object} [options.targetUnits]
	 * @returns {ExperimentalData}
	 */
	function getFlashPoint(data, options = {}) {
	  const flashPoint = extractExperimentalData(data, 'Flash Point', {
	    parser: value => parseNumbersUnits(value, options.temperature)
	  });
	  return flashPoint;
	}

	/**
	 * Returns the melting point of a compound
	 * @param data Data of a compound data request to the PubChem API
	 * @param options Options for the compound
	 * @returns ExperimentalData
	 */
	function getMeltingPoint(data, options = {}) {
	  const meltingPoint = extractExperimentalData(data, 'Melting Point', {
	    parser: value => parseNumbersUnits(value, options.temperature)
	  });
	  return meltingPoint;
	}

	/**
	 * Returns the refractive index of a compound
	 * @param data Data of a compound data request to the PubChem API
	 * @returns ExperimentalData
	 */
	function getRefractiveIndex(data = {}) {
	  const refractiveIndex = extractExperimentalData(data, 'Refractive Index', {
	    parser: value => parseRefractiveIndex(value)
	  });
	  return refractiveIndex;
	}

	/**
	 * Returns the solubility of a compound
	 * @param data Data of a compound data request to the PubChem API
	 * @returns ExperimentalData
	 */
	function getSolubility(data) {
	  const solubility = extractExperimentalData(data, 'Solubility');
	  return solubility;
	}

	/**
	 * Returns the vapor pressure of a compound
	 * @param data Data of a compound data request to the PubChem API
	 * @param options Options for the compound
	 */
	function getVaporPressure(data, options = {}) {
	  const vaporPressure = extractExperimentalData(data, 'Vapor Pressure', {
	    parser: value => parseVaporPressure(value, merge({
	      temperature: {
	        optional: true
	      }
	    }, options))
	  });
	  return vaporPressure;
	}

	/**
	 * Returns the experimental data of a compound
	 * @param data Data of a compound data request to the PubChem API
	 * @param options Options for the compound
	 * @returns ExperimentalData
	 */
	function getExperimentalData(data, options) {
	  return {
	    boilingPoint: getBoilingPoint(data, options),
	    density: getDensity(data, options),
	    flashPoint: getFlashPoint(data, options),
	    meltingPoint: getMeltingPoint(data, options),
	    solubility: getSolubility(data),
	    vaporPressure: getVaporPressure(data, options),
	    refractiveIndex: getRefractiveIndex(data)
	  };
	}

	function getFormula(data) {
	  return {
	    value: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Molecular Formula")]' + '.Information[0]')[0].Value.StringWithMarkup[0].String,
	    label: 'Molecular Formula',
	    description: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Molecular Formula")]')[0].Description
	  };
	}

	// Names and Identifiers
	function getInChI(data) {
	  return {
	    value: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Computed Descriptors")]' + '.Section[?(@.TOCHeading==="InChI")]' + '.Information[0]')[0].Value.StringWithMarkup[0].String,
	    label: 'InChI',
	    description: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Computed Descriptors")]' + '.Section[?(@.TOCHeading==="InChI")]')[0].Description
	  };
	}

	// Names and Identifiers
	function getInChIKey(data) {
	  return {
	    value: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Computed Descriptors")]' + '.Section[?(@.TOCHeading==="InChIKey")]' + '.Information[0]')[0].Value.StringWithMarkup[0].String,
	    label: 'InChIKey',
	    description: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Computed Descriptors")]' + '.Section[?(@.TOCHeading==="InChIKey")]')[0].Description
	  };
	}

	// Names and Identifiers
	function getSMILES(data) {
	  return {
	    value: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Computed Descriptors")]' + '.Section[?(@.TOCHeading==="Canonical SMILES")]' + '.Information[0]')[0].Value.StringWithMarkup[0].String,
	    label: 'Canonical SMILES',
	    description: jp.query(data, '$.Section[?(@.TOCHeading==="Names and Identifiers")]' + '.Section[?(@.TOCHeading==="Computed Descriptors")]' + '.Section[?(@.TOCHeading==="Canonical SMILES")]')[0].Description
	  };
	}

	function getIdentifiers(data) {
	  return {
	    formula: getFormula(data),
	    inchi: getInChI(data),
	    inchiKey: getInChIKey(data),
	    smiles: getSMILES(data)
	  };
	}

	const ghsPictogramText = {
	  GHS01: 'Explosive',
	  GHS02: 'Flammable',
	  GHS03: 'Oxidizing',
	  GHS04: 'Compressed Gas',
	  GHS05: 'Corrosive',
	  GHS06: 'Toxic',
	  GHS07: 'Harmful Irritant',
	  GHS08: 'Health Hazard',
	  GHS09: 'Environmental Hazard'
	};

	const hazardStatements = {
	  H200: '(Deleted) Unstable Explosive',
	  H201: '(Deleted) Explosive; mass explosion hazard',
	  H202: '(Deleted) Explosive; severe projection hazard',
	  H203: '(Deleted) Explosive; fire, blast or projection hazard',
	  H204: 'Fire or projection hazard',
	  H205: '(Deleted) May mass explode in fire',
	  H206: 'Fire, blast or projection hazard; increased risk of explosion if desensitizing agent is reduced',
	  H207: 'Fire or projection hazard; increased risk of explosion if desensitizing agent is reduced',
	  H208: 'Fire hazard; increased risk of explosion if desensitizing agent is reduced',
	  H209: 'Explosive',
	  H210: 'Very sensitive',
	  H211: 'Maybe sensitive',
	  H220: 'Extremely flammable gas',
	  H221: 'Flammable gas',
	  H222: 'Extremely flammable aerosol',
	  H223: 'Flammable aerosol',
	  H224: 'Extremely flammable liquid and vapor',
	  H225: 'Highly Flammable liquid and vapor',
	  H226: 'Flammable liquid and vapor',
	  H227: 'Combustible liquid',
	  H228: 'Flammable solid',
	  H229: 'Pressurized container: may burst if heated',
	  H230: 'May react explosively even in the absence of air',
	  H231: 'May react explosively even in the absence of air at elevated pressure and/or temperature',
	  H232: 'May ignite spontaneously if exposed to air',
	  H240: 'Heating may cause an explosion',
	  H241: 'Heating may cause a fire or explosion',
	  H242: 'Heating may cause a fire',
	  H250: 'Catches fire spontaneously if exposed to air',
	  H251: 'Self-heating; may catch fire',
	  H252: 'Self-heating in large quantities; may catch fire',
	  H260: 'In contact with water releases flammable gases which may ignite spontaneously',
	  H261: 'In contact with water releases flammable gas',
	  H270: 'May cause or intensify fire; oxidizer',
	  H271: 'May cause fire or explosion; strong Oxidizer',
	  H272: 'May intensify fire; oxidizer',
	  H280: 'Contains gas under pressure; may explode if heated',
	  H281: 'Contains refrigerated gas; may cause cryogenic burns or injury',
	  H282: 'Extremely flammable chemical under pressure: may explode if heated',
	  H283: 'Flammable chemical under pressure: may explode if heated',
	  H284: 'Chemical under pressure: may explode if heated',
	  H290: 'May be corrosive to metals',
	  H300: 'Fatal if swallowed',
	  H301: 'Toxic if swallowed',
	  H302: 'Harmful if swallowed',
	  H303: 'May be harmful if swallowed',
	  H304: 'May be fatal if swallowed and enters airways',
	  H305: 'May be harmful if swallowed and enters airways',
	  H310: 'Fatal in contact with skin',
	  H311: 'Toxic in contact with skin',
	  H312: 'Harmful in contact with skin',
	  H313: 'May be harmful in contact with skin',
	  H314: 'Causes severe skin burns and eye damage',
	  H315: 'Causes skin irritation',
	  H316: 'Causes mild skin irritation',
	  H317: 'May cause an allergic skin reaction',
	  H318: 'Causes serious eye damage',
	  H319: 'Causes serious eye irritation',
	  H320: 'Causes eye irritation',
	  H330: 'Fatal if inhaled',
	  H331: 'Toxic if inhaled',
	  H332: 'Harmful if inhaled',
	  H333: 'May be harmful if inhaled',
	  H334: 'May cause allergy or asthma symptoms or breathing difficulties if inhaled',
	  H335: 'May cause respiratory irritation',
	  H336: 'May cause drowsiness or dizziness',
	  H340: 'May cause genetic defects',
	  H341: 'Suspected of causing genetic defects',
	  H350: 'May cause cancer',
	  H350i: 'May cause cancer by inhalation',
	  H351: 'Suspected of causing cancer',
	  H360: 'May damage fertility or the unborn child',
	  H360F: 'May damage fertility',
	  H360D: 'May damage the unborn child',
	  H360FD: 'May damage fertility; May damage the unborn child',
	  H360Fd: 'May damage fertility; Suspected of damaging the unborn child',
	  H360Df: 'May damage the unborn child; Suspected of damaging fertility',
	  H361: 'Suspected of damaging fertility or the unborn child',
	  H361f: 'Suspected of damaging fertility',
	  H361d: 'Suspected of damaging the unborn child',
	  H361fd: 'Suspected of damaging fertility; Suspected of damaging the unborn child',
	  H362: 'May cause harm to breast-fed children',
	  H370: 'Causes damage to organs',
	  H371: 'May cause damage to organs',
	  H372: 'Causes damage to organs through prolonged or repeated exposure',
	  H373: 'May causes damage to organs through prolonged or repeated exposure',
	  H400: 'Very toxic to aquatic life',
	  H401: 'Toxic to aquatic life',
	  H402: 'Harmful to aquatic life',
	  H410: 'Very toxic to aquatic life with long lasting effects',
	  H411: 'Toxic to aquatic life with long lasting effects',
	  H412: 'Harmful to aquatic life with long lasting effects',
	  H413: 'May cause long lasting harmful effects to aquatic life',
	  H420: 'Harms public health and the environment by destroying ozone in the upper atmosphere',
	  'H300+H310': 'Fatal if swallowed or in contact with skin',
	  'H300+H330': 'Fatal if swallowed or if inhaled',
	  'H310+H330': 'Fatal in contact with skin or if inhaled',
	  'H300+H310+H330': 'Fatal if swallowed, in contact with skin or if inhaled',
	  'H301+H311': 'Toxic if swallowed or in contact with skin',
	  'H301+H331': 'Toxic if swallowed or if inhaled',
	  'H311+H331': 'Toxic in contact with skin or if inhaled.',
	  'H301+H311+H331': 'Toxic if swallowed, in contact with skin or if inhaled',
	  'H302+H312': 'Harmful if swallowed or in contact with skin',
	  'H302+H332': 'Harmful if swallowed or if inhaled',
	  'H312+H332': 'Harmful in contact with skin or if inhaled',
	  'H302+H312+H332': 'Harmful if swallowed, in contact with skin or if inhaled',
	  'H303+H313': 'May be harmful if swallowed or in contact with skin',
	  'H303+H333': 'May be harmful if swallowed or if inhaled',
	  'H313+H333': 'May be harmful in contact with skin or if inhaled',
	  'H303+H313+H333': 'May be harmful if swallowed, in contact with skin or if inhaled',
	  'H315+H320': 'Cause skin and eye irritation',
	  EUH001: 'Explosive when dry',
	  EUH006: 'Explosive with or without contact with air',
	  EUH014: 'Reacts violently with water',
	  EUH018: 'In use may form flammable/explosive vapour-air mixture',
	  EUH019: 'May form explosive peroxides',
	  EUH044: 'Risk of explosion if heated under confinement',
	  EUH029: 'Contact with water liberates toxic gas',
	  EUH031: 'Contact with acids liberates toxic gas',
	  EUH032: 'Contact with acids liberates very toxic gas',
	  EUH066: 'Repeated exposure may cause skin dryness or cracking',
	  EUH070: 'Toxic by eye contact',
	  EUH071: 'Corrosive to the respiratory tract',
	  EUH059: 'Hazardous to the ozone layer, superseded by GHS Class 5.1 in the second adaptation to technical progress of CLP.',
	  EUH201: 'Contains lead. Should not be used on surfaces liable to be chewed or sucked by children.',
	  EUH201A: 'Warning! Contains lead.',
	  EUH202: 'Cyanoacrylate. Danger. Bonds skin and eyes in seconds. Keep out of the reach of children.',
	  EUH203: 'Contains chromium(VI). May produce an allergic reaction.',
	  EUH204: 'Contains isocyanates. May produce an allergic reaction.',
	  EUH205: 'Contains epoxy constituents. May produce an allergic reaction.',
	  EUH206: 'Warning! Do not use together with other products. May release dangerous gases (chlorine).',
	  EUH207: 'Warning! Contains cadmium. Dangerous fumes are formed during use. See information supplied by the manufacturer. Comply with the safety instructions.',
	  EUH208: 'Contains <name of sensitising substance>. May produce an allergic reaction.',
	  EUH209: 'Can become highly flammable in use.',
	  EUH209A: 'Can become flammable in use.',
	  EUH210: 'Safety data sheet available on request.',
	  EUH401: 'To avoid risks to human health and the environment, comply with the instructions for use.'
	};

	const precautionaryStatements = {
	  P101: 'If medical advice is needed, have product container or label at hand.',
	  P102: 'Keep out of reach of children.',
	  P103: 'Read carefully and follow all instructions.',
	  P201: '(Deleted) Obtain special instructions before use.',
	  P202: '(Deleted) Do not handle until all safety precautions have been read and understood.',
	  P203: 'Obtain, read and follow all safety instructions before use.',
	  P210: 'Keep away from heat, hot surface, sparks, open flames and other ignition sources. No smoking.',
	  P211: 'Do not spray on an open flame or other ignition source.',
	  P212: 'Avoid heating under confinement or reduction of the desensitized agent.',
	  P220: 'Keep away from clothing and other combustible materials.',
	  P221: '(Deleted) Take any precaution to avoid mixing with combustibles/...',
	  P222: 'Do not allow contact with air.',
	  P223: 'Do not allow contact with water.',
	  P230: 'Keep wetted with ...',
	  P231: 'Handle and store contents under inert gas/...',
	  P232: 'Protect from moisture.',
	  P233: 'Keep container tightly closed.',
	  P234: 'Keep only in original container.',
	  P235: 'Keep cool.',
	  P236: 'Keep only in original packaging; Division .. in the transport configuration.',
	  P240: 'Ground/bond container and receiving equipment.',
	  P241: 'Use explosion-proof [electrical/ventilating/lighting/.../] equipment.',
	  P242: 'Use only non-sparking tools.',
	  P243: 'Take precautionary measures against static discharge.',
	  P244: 'Keep valves and fittings free from oil and grease.',
	  P250: 'Do not subject to grinding/shock/friction/...',
	  P251: 'Do not pierce or burn, even after use.',
	  P260: 'Do not breathe dust/fume/gas/mist/vapors/spray.',
	  P261: 'Avoid breathing dust/fume/gas/mist/vapors/spray.',
	  P262: 'Do not get in eyes, on skin, or on clothing.',
	  P263: 'Avoid contact during pregnancy/while nursing.',
	  P264: 'Wash hands [and ...] thoroughly after handling.',
	  P265: 'Do not touch eyes.',
	  P270: 'Do not eat, drink or smoke when using this product.',
	  P271: 'Use only outdoors or in a well-ventilated area.',
	  P272: 'Contaminated work clothing should not be allowed out of the workplace.',
	  P273: 'Avoid release to the environment.',
	  P280: 'Wear protective gloves/protective clothing/eye protection/face protection/hearing protection/...',
	  P281: '(Deleted)Use personal protective equipment as required.',
	  P282: 'Wear cold insulating gloves and either face shield or eye protection.',
	  P283: 'Wear fire resistant or flame retardant clothing.',
	  P284: '[In case of inadequate ventilation] Wear respiratory protection.',
	  P285: '(Deleted) In case of inadequate ventilation wear respiratory protection.',
	  'P231+P232': 'Handle and store contents under inert gas/... Protect from moisture.',
	  'P264+P265': 'Wash hands [and ...] thoroughly after handling. Do not touch eyes.',
	  'P235+P410': '(Deleted) Keep cool. Protect from sunlight.',
	  P301: 'IF SWALLOWED:',
	  P302: 'IF ON SKIN:',
	  P303: 'IF ON SKIN (or hair):',
	  P304: 'IF INHALED:',
	  P305: 'IF IN EYES:',
	  P306: 'IF ON CLOTHING:',
	  P307: '(Deleted) IF exposed:',
	  P308: 'IF exposed or concerned:',
	  P309: '(Deleted) IF exposed or if you feel unwell',
	  P310: '(Deleted) Immediately call a POISON CENTER or doctor/physician.',
	  P311: '(Deleted) Call a POISON CENTER or doctor/...',
	  P312: '(Deleted) Call a POISON CENTER or doctor/... if you feel unwell.',
	  P313: '(Deleted) Get medical advice/attention.',
	  P314: '(Deleted) Get medical advice/attention if you feel unwell.',
	  P315: '(Deleted) Get immediate medical advice/attention.',
	  P316: 'Get emergency medical help immediately.',
	  P317: 'Get emergency medical help.',
	  P318: 'if exposed or concerned, get medical advice.',
	  P319: 'Get medical help if you feel unwell.',
	  P320: 'Specific treatment is urgent (see ... on this label).',
	  P321: 'Specific treatment (see ... on this label).',
	  P322: '(Deleted) Specific measures (see ...on this label).',
	  P330: 'Rinse mouth.',
	  P331: 'Do NOT induce vomiting.',
	  P332: 'IF SKIN irritation occurs:',
	  P333: 'If skin irritation or rash occurs:',
	  P334: 'Immerse in cool water [or wrap in wet bandages].',
	  P335: 'Brush off loose particles from skin.',
	  P336: 'Thaw frosted parts with lukewarm water. Do not rub affected area.',
	  P337: 'If eye irritation persists:',
	  P338: 'Remove contact lenses, if present and easy to do. Continue rinsing.',
	  P340: 'Remove person to fresh air and keep comfortable for breathing.',
	  P341: '(Deleted) If breathing is difficult, remove victim to fresh air and keep at rest in a position comfortable for breathing.',
	  P342: 'If experiencing respiratory symptoms:',
	  P350: '(Deleted) Gently wash with plenty of soap and water.',
	  P351: 'Rinse cautiously with water for several minutes.',
	  P352: 'Wash with plenty of water/...',
	  P353: 'Rinse skin with water [or shower].',
	  P354: 'Immediately rinse with water for several minutes.',
	  P360: 'Rinse immediately contaminated clothing and skin with plenty of water before removing clothes.',
	  P361: 'Take off immediately all contaminated clothing.',
	  P362: 'Take off contaminated clothing.',
	  P363: 'Wash contaminated clothing before reuse.',
	  P364: 'And wash it before reuse.',
	  P370: 'In case of fire:',
	  P371: 'In case of major fire and large quantities:',
	  P372: 'Explosion risk.',
	  P373: 'DO NOT fight fire when fire reaches explosives.',
	  P374: '(Deleted) Fight fire with normal precautions from a reasonable distance.',
	  P375: 'Fight fire remotely due to the risk of explosion.',
	  P376: 'Stop leak if safe to do so.',
	  P377: 'Leaking gas fire: Do not extinguish, unless leak can be stopped safely.',
	  P378: 'Use ... to extinguish.',
	  P380: 'Evacuate area.',
	  P381: 'In case of leakage, eliminate all ignition sources.',
	  P390: 'Absorb spillage to prevent material damage.',
	  P391: 'Collect spillage.',
	  'P301+P310': '(Deleted) IF SWALLOWED: Immediately call a POISON CENTER/doctor/...',
	  'P301+P312': '(Deleted) IF SWALLOWED: call a POISON CENTER/doctor/... IF you feel unwell.',
	  'P301+P316': 'IF SWALLOWED: Get emergency medical help immediately.',
	  'P301+P317': 'IF SWALLOWED: Get medical help.',
	  'P301+P330+P331': 'IF SWALLOWED: Rinse mouth. Do NOT induce vomiting.',
	  'P302+P317': 'IF ON SKIN: Get medical help.',
	  'P302+P334': 'IF ON SKIN: Immerse in cool water or wrap in wet bandages.',
	  'P302+P335+P334': 'Brush off loose particles from skin. Immerse in cool water [or wrap in wet bandages].',
	  'P302+P350': '(Deleted) IF ON SKIN: Gently wash with plenty of soap and water.',
	  'P302+P352': 'IF ON SKIN: wash with plenty of water/...',
	  'P302+P361+P354': 'IF ON SKIN: Take off Immediately all contaminated clothing. Immediately rinse with water for several minutes.',
	  'P303+P361+P353': 'IF ON SKIN (or hair): Take off Immediately all contaminated clothing. Rinse SKIN with water [or shower].',
	  'P304+P312': '(Deleted) IF INHALED: Call a POISON CENTER/doctor/... if you feel unwell.',
	  'P304+P317': 'IF INHALED: Get medical help.',
	  'P304+P340': 'IF INHALED: Remove person to fresh air and keep comfortable for breathing.',
	  'P304+P341': '(Deleted) IF INHALED: If breathing is difficult, remove victim to fresh air and keep at rest in a position comfortable for breathing.',
	  'P305+P351+P338': 'IF IN EYES: Rinse cautiously with water for several minutes. Remove contact lenses if present and easy to do - continue rinsing.',
	  'P305+P354+P338': 'IF IN EYES: Immediately rinse with water for several minutes. Remove contact lenses if present and easy to do. Continue rinsing.',
	  'P306+P360': 'IF ON CLOTHING: Rinse Immediately contaminated CLOTHING and SKIN with plenty of water before removing clothes.',
	  'P308+P316': 'IF exposed or concerned: Get emergency medical help immediately.',
	  'P307+P311': '(Deleted) IF exposed: call a POISON CENTER or doctor/physician.',
	  'P308+P311': '(Deleted) IF exposed or concerned: Call a POISON CENTER/doctor/...',
	  'P308+P313': '(Deleted) IF exposed or concerned: Get medical advice/attention.',
	  'P309+P311': '(Deleted) IF exposed or if you feel unwell: call a POISON CENTER or doctor/physician.',
	  'P332+P313': '(Deleted) IF SKIN irritation occurs: Get medical advice/attention.',
	  'P332+P317': 'If skin irritation occurs: Get medical help.',
	  'P333+P317': 'If skin irritation or rash occurs: Get medical help.',
	  'P336+P317': 'Immediately thaw frosted parts with lukewarm water. Do not rub affected area. Get medical help.',
	  'P337+P317': 'If eye irritation persists: Get medical help.',
	  'P342+P316': 'If experiencing respiratory symptoms: Get emergence medical help immediately.',
	  'P333+P313': '(Deleted) IF SKIN irritation or rash occurs: Get medical advice/attention.',
	  'P335+P334': '(Deleted) Brush off loose particles from skin. Immerse in cool water/wrap in wet bandages.',
	  'P337+P313': '(Deleted) IF eye irritation persists: Get medical advice/attention.',
	  'P342+P311': '(Deleted) IF experiencing respiratory symptoms: Call a POISON CENTER/doctor/...',
	  'P361+P364': 'Take off immediately all contaminated clothing and wash it before reuse.',
	  'P362+P364': 'Take off contaminated clothing and wash it before reuse.',
	  'P370+P376': 'in case of fire: Stop leak if safe to do so.',
	  'P370+P378': 'In case of fire: Use ... to extinguish.',
	  'P370+P380': '(Deleted) In case of fire: Evacuate area.',
	  'P370+P380+P375': 'In case of fire: Evacuate area. Fight fire remotely due to the risk of explosion.',
	  'P371+P380+P375': 'In case of major fire and large quantities: Evacuate area. Fight fire remotely due to the risk of explosion.',
	  'P370+P372+P380+P373': 'In case of fire: Explosion risk. Evacuate area. DO NOT fight fire when fire reaches explosives.',
	  'P370+P380+P375[+P378]': 'In case of fire: Evacuate area. Fight fire remotely due to the risk of explosion. [Use...to extinguish].]',
	  P401: 'Store in accordance with ...',
	  P402: 'Store in a dry place.',
	  P403: 'Store in a well-ventilated place.',
	  P404: 'Store in a closed container.',
	  P405: 'Store locked up.',
	  P406: 'Store in corrosive resistant/... container with a resistant inner liner.',
	  P407: 'Maintain air gap between stacks or pallets.',
	  P410: 'Protect from sunlight.',
	  P411: 'Store at temperatures not exceeding ... °C/...°F.',
	  P412: 'Do not expose to temperatures exceeding 50 °C/ 122 °F.',
	  P413: 'Store bulk masses greater than ... kg/...lbs at temperatures not exceeding ... °C/...°F.',
	  P420: 'Store separately.',
	  P422: '(Deleted) Store contents under ...',
	  'P402+P404': 'Store in a dry place. Store in a closed container.',
	  'P403+P233': 'Store in a well-ventilated place. Keep container tightly closed.',
	  'P403+P235': 'Store in a well-ventilated place. Keep cool.',
	  'P410+P403': 'Protect from sunlight. Store in a well-ventilated place.',
	  'P410+P412': 'Protect from sunlight. Do not expose to temperatures exceeding 50 °C/122°F.',
	  'P411+P235': '(Deleted) Store at temperatures not exceeding ... °C/...°F. Keep cool.',
	  P501: 'Dispose of contents/container to ...',
	  P502: 'Refer to manufacturer or supplier for information on recovery or recycling',
	  P503: 'Refer to manufacturer/supplier... for information on disposal/recovery/recycling.'
	};

	/**
	 * Extracts GHS information (H/P-Statements and pictograms)
	 * @param data The data to extract from
	 * @return The extracted data
	 */
	function getGHS(data) {
	  const references = getReferences(data);
	  let pictograms;
	  try {
	    pictograms = jp.query(data, '$.Section[?(@.TOCHeading==="Safety and Hazards")]' + '.Section[?(@.TOCHeading==="Hazards Identification")]' + '.Section[?(@.TOCHeading==="GHS Classification")]' + '.Information[?(@.Name==="Pictogram(s)")]').map(entry => ({
	      reference: references[entry.ReferenceNumber],
	      data: jp.query(entry, '$.Value.StringWithMarkup[*].Markup[*]').map(entry => {
	        const code = entry.URL.match(/GHS\d+/)[0];
	        return {
	          code,
	          description: ghsPictogramText[code]
	        };
	      })
	    }));
	  } catch (error) {
	    pictograms = [];
	  }
	  let hStatements;
	  try {
	    hStatements = jp.query(data, '$.Section[?(@.TOCHeading==="Safety and Hazards")]' + '.Section[?(@.TOCHeading==="Hazards Identification")]' + '.Section[?(@.TOCHeading==="GHS Classification")]' + '.Information[?(@.Name==="GHS Hazard Statements")]').map(entry => ({
	      reference: references[entry.ReferenceNumber],
	      data: jp.query(entry, '$.Value.StringWithMarkup[*]').map(entry => entry.String.match(/H\d+/)[0]).map(hCode => ({
	        code: hCode,
	        description: hazardStatements[hCode]
	      }))
	    }));
	  } catch (error) {
	    hStatements = [];
	  }
	  //ToDo(kjappelbaum): investigate in more detail why they do not have the full P statements
	  //For P statements the full sentence (with conditions) is more important than just the number
	  let pStatements;
	  try {
	    pStatements = jp.query(data, '$.Section[?(@.TOCHeading==="Safety and Hazards")]' + '.Section[?(@.TOCHeading==="Hazards Identification")]' + '.Section[?(@.TOCHeading==="GHS Classification")]' + '.Information[?(@.Name==="Precautionary Statement Codes")]').map(entry => ({
	      reference: references[entry.ReferenceNumber],
	      data: jp.query(entry, '$.Value.StringWithMarkup[*]').map(entry => entry.String.match(
	      // eslint-disable-next-line no-lookahead-lookbehind-regexp/no-lookahead-lookbehind-regexp
	      /(?<oneP>(?<!\+)P\d\d\d(?!\+))|(?<twoP>P\d\d\d\+P\d\d\d(?!\+))|(?<threeP>P\d\d\d\+P\d\d\d\+P\d\d\d(?!\+))/gm))[0].map(pCode => ({
	        code: pCode,
	        description: precautionaryStatements[pCode]
	      }))
	    }));
	  } catch (error) {
	    pStatements = [];
	  }
	  return {
	    pictograms,
	    hStatements,
	    pStatements
	  };
	}
	/**
	 * We combine all the results and keep
	 */
	function getGHSSummary(data, options = {}) {
	  const {
	    sourceName
	  } = options;
	  const result = getGHS(data);
	  if (sourceName) {
	    result.pictograms = result.pictograms.filter(entry => entry.reference.sourceName.match(sourceName));
	    result.hStatements = result.hStatements.filter(entry => entry.reference.sourceName.match(sourceName));
	    result.pStatements = result.pStatements.filter(entry => entry.reference.sourceName.match(sourceName));
	  }
	  return {
	    pictograms: getUnique(result.pictograms),
	    hStatements: getUnique(result.hStatements),
	    pStatements: getUnique(result.pStatements)
	  };
	}
	function getUnique(entries) {
	  const uniqueMap = {};
	  entries.map(entry => entry.data.forEach(line => {
	    uniqueMap[line.code] = line;
	  }));
	  return Object.keys(uniqueMap).map(code => uniqueMap[code]).sort((a, b) => a.code < b.code ? -1 : 1);
	}

	class CompoundData {
	  constructor(data) {
	    this.data = data;
	  }
	  getReferences() {
	    return getReferences(this.data);
	  }
	  getExperimentalData(options) {
	    return getExperimentalData(this.data, options);
	  }
	  getIdentifiers() {
	    return getIdentifiers(this.data);
	  }
	  get computed() {
	    return getComputedData(this.data);
	  }
	  /**
	   * Property containing a summary of GHS information
	   */
	  get ghs() {
	    return getGHSSummary(this.data);
	  }
	  getGHSSummary(options) {
	    return getGHSSummary(this.data, options);
	  }
	  /**
	   * Canonical SMILES
	   */
	  getSMILES() {
	    return getSMILES(this.data);
	  }
	  /**
	   * Return molecular formula in Hill notation
	   */
	  getFormula() {
	    return getFormula(this.data);
	  }
	  /**
	   * Return the International Chemical Identifier (InChI) computed from chemical structure using the International Union of Pure and Applied Chemistry (IUPAC) standard
	   */
	  getInChI() {
	    return getInChI(this.data);
	  }
	  /**
	   * International Chemical Identifier hash (InChIKey) computed from chemical structure using the International Union of Pure and Applied Chemistry (IUPAC) standard.
	   */
	  getInChIKey() {
	    return getInChIKey(this.data);
	  }
	  /**
	   * Detailed information about GHS information
	   */
	  getGHS() {
	    return getGHS(this.data);
	  }
	  getMeltingPoint(options) {
	    return getMeltingPoint(this.data, options);
	  }
	  getBoilingPoint(options) {
	    return getBoilingPoint(this.data, options);
	  }
	  getDensity(options) {
	    return getDensity(this.data, options);
	  }
	  getVaporPressure(options) {
	    return getVaporPressure(this.data, options);
	  }
	  getSolubility() {
	    return getSolubility(this.data);
	  }
	  getFlashPoint(options) {
	    return getFlashPoint(this.data, options);
	  }
	  toJSON() {
	    const methods = Object.entries(Object.getOwnPropertyDescriptors(CompoundData.prototype)).filter(([, descriptor]) => typeof descriptor.get === 'function').map(([key]) => key);
	    const result = {};
	    for (const method of methods) {
	      result[method] = this[method];
	    }
	    return result;
	  }
	}

	async function compoundDataFromCID(cid, options = {}) {
	  const {
	    cache
	  } = options;
	  let compoundData;
	  if (cache?.(cid)) {
	    compoundData = cache(cid);
	  } else {
	    const dataResponse = await fetch(`https://pubchem.ncbi.nlm.nih.gov/rest/pug_view/data/compound/${cid}/JSON`);
	    compoundData = (await dataResponse.json()).Record;
	    if (cache) {
	      cache(cid, compoundData);
	    }
	  }
	  return new CompoundData(compoundData);
	}

	function checkCompoundsResult(compounds) {
	  if (!Array.isArray(compounds) || compounds.length === 0) {
	    throw new Error('No compound found');
	  }
	  if (compounds.length !== 1) {
	    throw new Error('More than one compound found');
	  }
	  if (!('id' in compounds[0].id)) {
	    throw new Error('No compound found');
	  }
	}

	async function compoundFromInchiKey(inchiKey, options = {}) {
	  const {
	    cache
	  } = options;
	  let compounds;
	  if (cache?.(inchiKey)) {
	    compounds = cache(inchiKey);
	  } else {
	    const encodedInchiKey = encodeURIComponent(inchiKey);
	    const moleculeResponse = await fetch(`https://pubchem.ncbi.nlm.nih.gov/rest/pug/compound/inchikey/${encodedInchiKey}/json`);
	    compounds = (await moleculeResponse.json()).PC_Compounds;
	    if (cache) {
	      cache(inchiKey, compounds);
	    }
	  }
	  checkCompoundsResult(compounds);
	  return new Compound(compounds[0], {
	    cache
	  });
	}

	async function compoundFromName(smiles, options = {}) {
	  const {
	    cache
	  } = options;
	  let compounds;
	  if (cache?.(smiles)) {
	    compounds = cache(smiles);
	  } else {
	    const encodedName = encodeURIComponent(smiles);
	    const moleculeResponse = await fetch(`https://pubchem.ncbi.nlm.nih.gov/rest/pug/compound/name/${encodedName}/json`);
	    compounds = (await moleculeResponse.json()).PC_Compounds;
	    if (cache) {
	      cache(smiles, compounds);
	    }
	  }
	  checkCompoundsResult(compounds);
	  return new Compound(compounds[0], {
	    cache
	  });
	}

	async function compoundFromSmiles(smiles, options = {}) {
	  const {
	    cache
	  } = options;
	  let compounds;
	  if (cache?.(smiles)) {
	    compounds = cache(smiles);
	  } else {
	    const encodedSmiles = encodeURIComponent(smiles);
	    const moleculeResponse = await fetch(`https://pubchem.ncbi.nlm.nih.gov/rest/pug/compound/smiles/${encodedSmiles}/json`);
	    compounds = (await moleculeResponse.json()).PC_Compounds;
	    if (cache) {
	      cache(smiles, compounds);
	    }
	  }
	  checkCompoundsResult(compounds);
	  return new Compound(compounds[0], {
	    cache
	  });
	}

	class Compound {
	  constructor(data, options = {}) {
	    this.data = data;
	    this.cache = options.cache;
	  }
	  getCID() {
	    return this.data?.id?.id?.cid;
	  }
	  getData() {
	    const cid = this.getCID();
	    const cd = compoundDataFromCID(cid, {
	      cache: this.cache
	    });
	    return cd;
	  }
	  toJSON() {
	    const methods = Object.entries(Object.getOwnPropertyDescriptors(Compound.prototype)).filter(([, descriptor]) => typeof descriptor.get === 'function').map(([key]) => key);
	    const result = {};
	    for (const method of methods) {
	      result[method] = this[method];
	    }
	    return result;
	  }
	}
	Compound.fromSmiles = compoundFromSmiles;
	Compound.fromName = compoundFromName;
	Compound.fromInchiKey = compoundFromInchiKey;

	/**
	 * We check that the hazards are present in the reagents
	 * @param {object} reagent
	 */
	async function ensureHazardAndPhysical(reagent, options = {}) {
	  const {
	    logger
	  } = options;
	  if (reagent.hazards) return;
	  try {
	    const compound = await Compound.fromSmiles(reagent.smiles);
	    const data = await compound.getData();
	    const summary = data.getGHSSummary({
	      sourceName: 'European'
	    });
	    reagent.hStatements = summary.hStatements;
	    reagent.hazards = reagent.hStatements.map(h => h.code);
	  } catch (e) {
	    logger.error(`Could not get hazards for ${reagent.smiles}: ${e.toString()}`);
	    reagent.hStatements = [];
	    reagent.hazards = [];
	  }
	}

	var core = {};

	var openchemlibCore = {};

	(function(exports){(function(root){function getExports($wnd){var $doc=$wnd.document;var $gwt={};var navigator={userAgent:'webkit'};function noop(){}var __gwtModuleFunction=noop;__gwtModuleFunction.__moduleStartupDone=noop;var Plb='object',Qlb='anonymous',Rlb='fnStack',Slb='\n',Tlb={4:1,10:1,5:1,8:1},Ulb='Unknown',Vlb='boolean',Wlb='number',Xlb='string',Ylb=2147483647,Zlb='For input string: "',$lb='null',_lb='__noinit__',amb={4:1,5:1,8:1},bmb={4:1,15:1,22:1},cmb=' (copy)',dmb={4:1,10:1,31:1,5:1,17:1,8:1,29:1},emb=65536,fmb=65535,gmb=10000,hmb=', length: ',imb='Index: ',jmb=', Size: ',kmb='fromIndex: ',lmb=', toIndex: ',mmb={12:1,4:1,5:1},nmb={4:1},omb={7:1,4:1,5:1},pmb=16777215,qmb=65280,rmb=0.30000001192092896,smb=545460846592,tmb=4294967296,umb=8589934592,vmb=17179869184,wmb=34359738368,xmb=68719476736,ymb=137438953472,zmb=274877906944,Amb='<line x1="',Bmb='" ',Cmb='y1="',Dmb='x2="',Emb='y2="',Fmb='stroke="',Gmb='stroke-width="',Hmb='" />',Imb={11:1,4:1,5:1},Jmb=536870912,Kmb=2.617993878,Lmb=3.665191429,Mmb=6.283185307179586,Nmb=3.141592653589793,Omb=1.5707963267948966,Pmb=16384,Qmb=4096,Rmb={l:0,m:0,h:1},Smb={l:0,m:0,h:2},Tmb={l:0,m:0,h:4},Umb=1920,Vmb=1024,Wmb=234881024,Xmb=167772160,Ymb=100663296,Zmb=201326592,$mb=114688,_mb=4063232,anb=3538944,bnb=3014656,cnb=2097152,dnb=393216,enb=917504,fnb=1966080,gnb=17042430230528,hnb=16492674416640,inb=15942918602752,jnb=14843406974976,knb=12644383719424,lnb=8796093022208,mnb=549755813888,nnb=1649267441664,onb=3848290697216,pnb=8246337208320,qnb=29360128,rnb=268435456,snb=8192,tnb=2048,unb=-1.5707963267948966,vnb=130560,wnb=6291456,xnb=1.0471975511965976,ynb=0.5235987755982988,znb=786432,Anb=524288,Bnb=262144,Cnb=-16777216,Dnb=16711680,Enb=-32768,Fnb='<DrawingObject type="',Gnb='><\/DrawingObject>',Hnb={4:1,5:1,13:1},Inb={4:1,5:1},Jnb='unsupported atomicNo:',Knb=2.8415926535897933,Lnb={26:1,4:1,10:1,5:1,8:1},Mnb={4:1,5:1,17:1,8:1},Nnb={9:1,4:1,5:1},Onb=131072,Pnb={l:6,m:0,h:4},Qnb=1048576,Rnb=3221225472,Snb={l:0,m:0,h:3},Tnb=8388608,Unb=-65536,Vnb={38:1,39:1,45:1},Wnb={38:1,39:1,80:1,45:1,98:1},Xnb={4:1,5:1,38:1,39:1,80:1,18:1,45:1,98:1,157:1},Ynb='??',Znb=2013265920,$nb=32505856,_nb=31744,aob=1.7976931348623157E308,bob={105:1,4:1,10:1,5:1,17:1,8:1},cob=33554432,dob=67108864,eob=32768,fob=-33554433,gob=-33292289,hob=-3.141592653589793,iob=4194303,job=3072,kob={36:1,4:1,5:1},lob=0.7853981633974483,mob=3.061592653589793,nob=1073741824,oob=541165879296,pob={14:1,4:1,5:1,8:1},qob=281.17,rob=289.19,sob=294.21,tob='0.0000',uob='M  END\n',vob='ATOMS',wob=-2147483648,xob='M  END',yob='$$$$',zob='M  V30 ',Aob='M  V30 COUNTS ',Bob=')\n',Cob='M  V30 MDLV30/STEREL',Dob='M  V30 MDLV30/STERAC',Eob=3.4028234663852886E38,Fob={l:2,m:0,h:4},Gob=4194304,Hob={l:4179966,m:4063288,h:7},Iob={l:4194302,m:4193343,h:7},Job={117:1,103:1,104:1,4:1,10:1,5:1,8:1},Kob='class="event" ',Lob='opacity="0" />',Mob='Assignment of aromatic double bonds failed',Nob=1099511627776,Oob=2199023255552,Pob=4398046511104,Qob='Members of ESR groups must only be stereo centers with known configuration.',Rob='Over- or under-specified stereo feature or more than one racemic type bond',Sob='Ambiguous configuration at stereo center because of 2 parallel bonds',Tob={91:1,75:1,25:1,4:1},Uob=100000,Vob=3.1415927410125732,Wob=1.5707963705062866,Xob=0.029999999329447746,Yob=6.2831854820251465,Zob=1.9106119871139526,$ob=2.2639999389648438,_ob=2.16759991645813,apb=2.0662999153137207,bpb=2.3561999797821045,cpb=2.2844998836517334,dpb=1.9106120066311025,epb=2.094395160675049,fpb=0.6000000238418579,gpb=1.7999999523162842,hpb=0.05000000074505806,ipb={4:1,10:1,5:1,8:1,30:1},jpb=0.2617993950843811,kpb=2.0943951023931953,lpb=1.2000000476837158,mpb=1.4299999475479126,npb=1.659999966621399,opb=1.4600000381469727,ppb=1.8200000524520874,qpb=2.4600000381469727,rpb=2.4200000762939453,spb=2.450000047683716,tpb=2.440000057220459,upb=2.4000000953674316,vpb=2.319999933242798,wpb=2.2899999618530273,xpb=1.8600000143051147,ypb=2.5299999713897705,zpb=2.490000009536743,Apb=2.4700000286102295,Bpb=2.880000114440918,Cpb=2.9200000762939453,Dpb=2.869999885559082,Epb=2.8299999237060547,Fpb=2.7899999618530273,Gpb=2.809999942779541,Hpb=2.799999952316284,Ipb=3.049999952316284,Jpb=1.3300000429153442,Kpb=1.3899999856948853,Lpb=1.159999966621399,Mpb=1.1100000143051147,Npb=1.0700000524520874,Opb=1.9600000381469727,Ppb=1.7100000381469727,Qpb=1.3600000143051147,Rpb=1.340000033378601,Spb=1.2200000286102295,Tpb=1.2400000095367432,Upb=1.2100000381469727,Vpb=1.850000023841858,Wpb=1.6299999952316284,Xpb=1.4700000286102295,Ypb=1.2799999713897705,Zpb=1.440000057220459,$pb=1.399999976158142,_pb=1.7599999904632568,aqb=1.7300000190734863,bqb=1.7200000286102295,cqb=1.6799999475479126,dqb=1.690000057220459,eqb=1.6699999570846558,fqb=1.649999976158142,gqb=1.7000000476837158,hqb=1.6200000047683716,iqb=1.3700000047683716,jqb=1.2899999618530273,kqb=1.4500000476837158,lqb=1.5700000524520874,mqb=-0.5235987755982988,nqb=-1.0471975511965976,oqb=-0.7853981633974483,pqb=0.017453292519943295,qqb=0.17453292519943295,rqb=1.0E-4,sqb={114:1},tqb=57.29577951308232,uqb=0.021922173386725217,vqb=1.0E-8,wqb=2.5120960256267386,xqb=1.0E-5,yqb='MMFF94s+',zqb='nonbonded cutoff',Aqb='dielectric constant',Bqb='dielectric model',Cqb='angle bend',Dqb='bond stretch',Eqb='electrostatic',Fqb='out of plane',Gqb='stretch bend',Hqb='torsion angle',Iqb='van der waals',Jqb={4:1,10:1,5:1,17:1,8:1},Kqb=0.07000000000000006,Lqb=0.1200000000000001,Mqb=-7.840000000000001,Nqb='OCL_RXN_V1.0:',Oqb="'$MOL' tag not found",Pqb="'M  END' not found",Qqb='\nActelion Java MolfileCreator 2.0\n\n  0  0  0  0  0  0              0 V3000\n',Rqb='M  V30 END CTAB',Sqb=2475109711874,Tqb=2475109711875,Uqb=2887426572290,Vqb=2887426572291,Wqb=2887627898882,Xqb=2887627898883,Yqb=2887627997186,Zqb=3437182386178,$qb=3437182386179,_qb=3437383712770,arb=3437383811074,brb=3437652148226,crb=3437652377602,drb=4674132967426,erb=4674334294018,frb=4674334392322,grb=4674602729474,hrb=4674602958850,irb=5086449827842,jrb=5086449827848,krb=5086651154434,lrb=5086651252738,mrb=5086651252743,nrb=5086919589890,orb=5086919688194,prb=5636406968322,qrb=5636407066626,rrb=5636675502082,srb={l:589826,m:590112,h:4},trb={l:589827,m:590112,h:4},urb={l:589826,m:590160,h:4},vrb={l:589827,m:590160,h:4},wrb={l:688130,m:590160,h:4},xrb={l:589826,m:590224,h:4},yrb={l:589827,m:590224,h:4},zrb={l:819203,m:590224,h:4},Arb={l:589826,m:590368,h:4},Brb={l:688130,m:590368,h:4},Crb={l:819202,m:590368,h:4},Drb={l:819208,m:590368,h:4},Erb={l:589826,m:590416,h:4},Frb={l:589832,m:590416,h:4},Grb={l:688130,m:590416,h:4},Hrb={l:819202,m:590416,h:4},Irb={l:589826,m:598304,h:4},Jrb={l:589827,m:598304,h:4},Krb={l:688130,m:598304,h:4},Lrb={l:1114114,m:598304,h:4},Mrb={l:1212418,m:598304,h:4},Nrb={l:589826,m:598308,h:4},Orb={l:589826,m:688416,h:4},Prb={l:589826,m:688464,h:4},Qrb={l:589826,m:688720,h:4},Rrb={l:688130,m:688720,h:4},Srb={l:589826,m:590112,h:288},Trb={l:589826,m:590112,h:336},Urb={l:589826,m:688416,h:336},Vrb={l:589826,m:688464,h:336},Wrb={l:688130,m:688464,h:336},Xrb={l:589826,m:590112,h:400},Yrb={l:589826,m:688416,h:400},Zrb={l:589826,m:688464,h:400},$rb={l:688130,m:688464,h:400},_rb={l:589826,m:819488,h:400},asb={l:589826,m:819536,h:400},bsb={l:589826,m:819600,h:400},csb={l:688130,m:819600,h:400},dsb={l:819202,m:819600,h:400},esb={l:589831,m:688416,h:592},fsb={l:589831,m:688464,h:592},gsb={l:688135,m:688464,h:592},hsb={l:589831,m:819536,h:592},isb={l:688135,m:819536,h:592},jsb={l:688135,m:819600,h:592},ksb={l:589832,m:1212704,h:592},lsb={l:589832,m:1212752,h:592},msb={l:688136,m:1212752,h:592},nsb={l:589831,m:688464,h:656},osb={l:589826,m:590112,h:8480},psb={l:589826,m:688416,h:8480},qsb={l:589826,m:688464,h:8480},rsb={l:688130,m:688464,h:8480},ssb={l:819202,m:819600,h:8480},tsb={l:688135,m:1212752,h:8480},usb={l:589832,m:1213008,h:8480},vsb={l:688136,m:1213008,h:8480},wsb={l:589826,m:590112,h:8484},xsb={l:589826,m:590160,h:8484},ysb={l:688130,m:598304,h:8484},zsb=0.6262000203132629,Asb=-1.3825000524520874,Bsb=0.33169999718666077,Csb=-1.4915000200271606,Dsb=0.3540000021457672,Esb=0.38179999589920044,Fsb=-0.6019999980926514,Gsb=-0.7379999756813049,Hsb='Atom-types are 64-bit numbers describing atoms and their near surrounding.',Isb='Recognized atom types and their contributions are:',Jsb='Druglikeness predictor not properly initialized.',Ksb=3.009999990463257,Lsb=-0.17000000178813934,Msb=-0.1809999942779541,Nsb=-0.2029999941587448,Osb=2147483648,Psb='MMFF94',Qsb='MMFF94s',Rsb='117\nint,int,int,int,float\n0,2,0,0,0.020\n1,2,1,2,0.030\n1,2,2,2,0.027\n1,2,2,3,0.026\n1,2,2,5,0.013\n1,2,2,37,0.032\n2,2,2,5,0.013\n2,2,3,5,0.012\n2,2,5,5,0.006\n2,2,5,6,0.027\n2,2,5,37,0.017\n2,2,5,40,0.012\n2,2,5,41,0.008\n0,3,0,0,0.130\n1,3,1,7,0.146\n1,3,2,7,0.138\n1,3,3,7,0.134\n1,3,5,7,0.122\n1,3,6,7,0.141\n1,3,7,10,0.129\n1,3,7,37,0.138\n2,3,5,7,0.113\n2,3,5,9,0.081\n2,3,6,7,0.127\n2,3,7,10,0.116\n3,3,5,7,0.113\n3,3,6,7,0.127\n5,3,5,7,0.103\n5,3,5,9,0.074\n5,3,5,54,0.078\n5,3,6,7,0.119\n5,3,7,10,0.102\n5,3,9,40,0.067\n6,3,7,37,0.127\n7,3,10,10,0.113\n7,3,20,20,0.151\n9,3,40,40,0.057\n0,8,0,0,0.000\n0,10,0,0,0.015\n1,10,1,3,0.015\n1,10,3,6,0.015\n1,10,3,28,0.015\n3,10,3,28,0.015\n3,10,28,28,0.015\n0,17,0,0,0.000\n0,26,0,0,0.000\n0,30,0,0,0.010\n5,30,20,30,0.008\n0,37,0,0,0.035\n1,37,37,37,0.040\n2,37,37,37,0.031\n3,37,37,37,0.027\n5,37,37,37,0.015\n5,37,37,38,0.046\n5,37,37,63,0.008\n5,37,37,64,0.012\n5,37,37,69,0.016\n5,37,38,38,0.084\n6,37,37,37,0.048\n15,37,37,37,0.025\n37,37,37,40,0.046\n0,39,0,0,0.020\n1,39,63,63,0.012\n23,39,63,63,-0.014\n23,39,63,65,0.021\n23,39,65,65,0.062\n0,40,0,0,0.030\n1,40,28,37,0.030\n2,40,28,28,0.030\n3,40,28,28,0.030\n28,40,28,37,0.030\n0,41,0,0,0.180\n1,41,32,32,0.178\n2,41,32,32,0.161\n5,41,32,32,0.158\n0,43,0,0,0.000\n0,45,0,0,0.150\n0,49,0,0,0.000\n50,49,50,50,0.000\n0,54,0,0,0.020\n1,54,3,36,0.016\n3,54,36,36,0.018\n0,55,0,0,0.020\n1,55,36,57,0.020\n36,55,36,57,0.020\n0,56,0,0,0.020\n1,56,36,57,0.020\n36,56,36,57,0.020\n0,57,0,0,0.080\n5,57,55,55,0.038\n56,57,56,56,0.158\n0,58,0,0,0.025\n0,63,0,0,0.050\n5,63,39,64,0.019\n5,63,39,66,0.068\n5,63,44,64,0.014\n5,63,44,66,0.055\n5,63,59,64,0.033\n5,63,59,66,0.085\n37,63,39,64,0.010\n0,64,0,0,0.040\n5,64,63,64,0.006\n5,64,63,66,0.043\n5,64,64,65,0.052\n5,64,65,66,0.094\n37,64,63,64,-0.01\n0,67,0,0,0.070\n0,69,0,0,0.070\n32,69,37,37,0.067\n0,73,0,0,0.000\n0,78,0,0,0.045\n5,78,78,81,0.046\n0,80,0,0,0.080\n5,80,81,81,0.057\n0,81,0,0,0.025\n36,81,78,80,0.016\n0,82,0,0,0.000\n',Ssb={131:1,4:1,5:1,8:1},Tsb='undefined',Usb=0.08726646502812703,Vsb=0.003921568859368563,Wsb='Too many percent/per mille characters in pattern "',Xsb=1048575,Ysb=17592186044416,Zsb=1000000000,$sb=-17592186044416,_sb='CSS1Compat',atb=5.56,btb=11.12,ctb=13.34,dtb=14.44,etb=0.3010299956639812,ftb='BigInteger divide by zero',gtb=4294967295,htb={l:0,m:0,h:524288},itb={27:1,88:1},jtb={99:1},ktb={38:1,39:1,45:1,86:1},ltb='delete',mtb=15525485,ntb=5.960464477539063E-8,otb=16777216,ptb={4:1,27:1,52:1,64:1},qtb='Invalid UTF8 sequence',rtb=0.20000000298023224,stb=' distance:',ttb='gGP`@dfyjidNcGI[WQCP`<',utb={43:1};var elb={};var flb={};var glb={};var hlb={};var ilb={};var jlb={};var klb={};var llb={};var mlb={};var nlb={};var olb={};var plb={};var qlb={};var rlb={};var slb={};var tlb={};var ulb={};var vlb={};var wlb={};var xlb={};var ylb={};var zlb={};var Alb={};var Blb={};var Clb={};var Dlb={};var Elb={};var Flb={};var Glb={};var Hlb={};var Ilb={};var Jlb={};var Klb={};var Llb={};var Mlb={};var Nlb={};var Olb={};var _;var $1;var U1;elb.i1=function i1(){};function Z1(a,b){typeof window===Plb&&typeof window['$gwt']===Plb&&(window['$gwt'][a]=b);}function Y1(b,c,d,e){elb.X1();var f=U1;function g(){for(var a=0;a<f.length;a++){f[a]();}}if(b){try{dlb(g)();}catch(a){b(c,a);}}else {dlb(g)();}}elb.X1=function X1(){U1==null&&(U1=[]);};function W1(){elb.X1();var a=U1;for(var b=0;b<arguments.length;b++){a.push(arguments[b]);}}elb.V1=function V1(){};function j2(){}function i2(a){if(Array.isArray(a)&&a.uc===j2){return flb.Mc(a);}return a.toString();}function h2(a,b){var c=$wnd;if(a===''){return c;}var d=a.split('.');!(d[0]in c)&&c.execScript&&c.execScript('var '+d[0]);if(b){var e=b.prototype.sc;e.e=b;}for(var f;d.length&&(f=d.shift());){c=c[f]=c[f]||!d.length&&b||{};}return c;}elb.g2=function g2(a){function b(){}b.prototype=a||{};return new b();};elb.f2=function f2(a){return a instanceof Array?a[0]:null;};function e2(a,b,c){var d=function(){return a.apply(d,arguments);};b.apply(d,c);return d;}function d2(){}function c2(a,b){for(var c in b){b[c]['configurable']=true;}Object.defineProperties(a,b);}function b2(a,b,c){var d=$1;var e=d[a];var f=elb.f2(e);if(e&&!f){_=e;}else {_=elb.a2(b);_.tc=c;!b&&(_.uc=j2);d[a]=_;}for(var g=3;g<arguments.length;++g){arguments[g].prototype=_;}f&&(_.sc=f);}elb.a2=function a2(a){var b=a&&a.prototype;!b&&(b=$1[a]);return elb.g2(b);};elb._1=function _1(){};$wnd.goog=$wnd.goog||{};$wnd.goog.global=$wnd.goog.global||$wnd;$1={};flb.Bc=function Bc(a,b){return CX(a)===CX(b);};flb.Cc=function Cc(a){return a.sc;};flb.Dc=function Dc(a){return ilb.Egb(a);};function Ec(){}flb.Gc=function Gc(a,b){return zX(a)?flb.j5(a,b):wX(a)?flb.a4(a,b):vX(a)?flb.Y2(a,b):tX(a)?a.ib(b):qW(a)?flb.Bc(a,b):glb.xU(a,b);};flb.Ic=function Ic(a){return zX(a)?flb.l5():wX(a)?flb.b4():vX(a)?flb.Z2():tX(a)?a.sc:qW(a)?flb.Cc(a):glb.yU(a);};flb.Kc=function Kc(a){return zX(a)?flb.m5(a):wX(a)?flb.c4(a):vX(a)?flb.$2(a):tX(a)?a.kb():qW(a)?flb.Dc(a):glb.zU(a);};flb.Mc=function Mc(a){var b;return flb.x3(flb.Ic(a))+'@'+(b=ilb.Xgb(ilb.Wgb(flb.Kc(a))),b.toString(16));};b2(1,null,{},Ec);_.ib=function Fc(a){return flb.Bc(this,a);};_.jb=function Hc(){return flb.Cc(this);};_.kb=function Jc(){return flb.Dc(this);};_.lb=function Lc(){var a;return flb.x3(flb.Ic(this))+'@'+(a=ilb.Xgb(ilb.Wgb(flb.Kc(this))),a.toString(16));};_.equals=function(a){return this.ib(a);};_.hashCode=function(){return this.kb();};_.toString=function(){return this.lb();};glb.xU=function xU(a,b){return glb.DU(a)?glb.AU(a,b):CX(a)===CX(b);};glb.yU=function yU(a){return a.sc||Array.isArray(a)&&iW(elb.A$,1)||elb.A$;};glb.zU=function zU(a){return glb.EU(a)?glb.BU(a):ilb.Egb(a);};glb.AU=function AU(a,b){return a.equals(b);};glb.BU=function BU(a){return a.hashCode();};glb.CU=function CU(){return [];};glb.DU=function DU(a){return !!a&&!!a.equals;};glb.EU=function EU(a){return !!a&&!!a.hashCode;};glb.FU=function FU(a){return a.toString?a.toString():'[JavaScriptObject]';};hlb.jV=function jV(){hlb.jV=d2;var a,b;b=!hlb.sV();a=new hlb.BV();hlb.iV=b?new hlb.tV():a;};hlb.kV=function kV(a){hlb.jV();hlb.iV.Tb(a);};hlb.lV=function lV(a){hlb.jV();var b;b=hlb.iV.Ub(a);return hlb.mV(b);};hlb.mV=function mV(a){var b,c,d,e;b='kV';c='pz';e=$wnd.Math.min(a.length,5);for(d=e-1;d>=0;d--){if(flb.j5(a[d].d,b)||flb.j5(a[d].d,c)){a.length>=d+1&&ilb.Xgb(a).splice(0,d+1);break;}}return a;};hlb.nV=function nV(a){var b=/function(?:\s+([\w$]+))?\s*\(/;var c=b.exec(a);return c&&c[1]||Qlb;};hlb.oV=function oV(a){hlb.jV();return a&&a[Rlb]?a[Rlb]:[];};hlb.pV=function pV(a){hlb.jV();return a.name||(a.name=hlb.nV(a.toString()));};hlb.qV=function qV(a){hlb.jV();return parseInt(a)||-1;};hlb.rV=function rV(a){hlb.jV();var b=a.backingJsObject;if(b&&b.stack){var c=b.stack;var d=b+Slb;c.substring(0,d.length)==d&&(c=c.substring(d.length));return c.split(Slb);}return [];};hlb.sV=function sV(){if(Error.stackTraceLimit>0){$wnd.Error.stackTraceLimit=Error.stackTraceLimit=64;return true;}return 'stack'in new Error();};b2(331,1,{});hlb.tV=function tV(){};b2(237,331,{},hlb.tV);_.Tb=function uV(a){var b={};var c=[];a[Rlb]=c;var d=arguments.callee.caller;while(d){var e=hlb.pV(d);c.push(e);var f=':'+e;var g=b[f];if(g){var h,i;for(h=0,i=g.length;h<i;h++){if(g[h]===d){return;}}}(g||(b[f]=[])).push(d);d=d.caller;}};_.Ub=function vV(a){var b,c,d,e;d=hlb.oV(a);c=glb.LU(d);e=oW(elb.q_,Tlb,66,c,0,1);for(b=0;b<c;b++){e[b]=new flb.$4(glb.KU(d,b),null,-1);}return e;};hlb.wV=function wV(a,b){var c,d,e,f,g,h,i,j,k;j='';if(ilb.Xgb(b).length==0){return a.Vb(Ulb,Qlb,-1,-1);}k=flb.x5(b);flb.j5(ilb.Xgb(k).substr(0,3),'at ')&&(k=(ilb.Sgb(3,ilb.Xgb(k).length+1),ilb.Xgb(k).substr(3)));k=hlb.xV(k);g=ilb.Xgb(k).indexOf('(');if(g==-1){g=ilb.Xgb(k).indexOf('@');if(g==-1){j=k;k='';}else {j=flb.x5((ilb.Sgb(g+1,ilb.Xgb(k).length+1),ilb.Xgb(k).substr(g+1)));k=flb.x5((ilb.Rgb(0,g,ilb.Xgb(k).length),ilb.Xgb(k).substr(0,g)));}}else {c=ilb.Xgb(k).indexOf(')',g);j=(ilb.Rgb(g+1,c,ilb.Xgb(k).length),ilb.Xgb(k).substr(g+1,c-(g+1)));k=flb.x5((ilb.Rgb(0,g,ilb.Xgb(k).length),ilb.Xgb(k).substr(0,g)));}g=flb.n5(k,z5(46));g!=-1&&(k=(ilb.Sgb(g+1,ilb.Xgb(k).length+1),ilb.Xgb(k).substr(g+1)));(ilb.Xgb(k).length==0||flb.j5(k,'Anonymous function'))&&(k=Qlb);h=flb.q5(j,z5(58));e=flb.r5(j,z5(58),h-1);i=-1;d=-1;f=Ulb;if(h!=-1&&e!=-1){f=(ilb.Rgb(0,e,ilb.Xgb(j).length),ilb.Xgb(j).substr(0,e));i=hlb.qV((ilb.Rgb(e+1,h,ilb.Xgb(j).length),ilb.Xgb(j).substr(e+1,h-(e+1))));d=hlb.qV((ilb.Sgb(h+1,ilb.Xgb(j).length+1),ilb.Xgb(j).substr(h+1)));}return a.Vb(f,k,i,d);};hlb.xV=function xV(a){return a.replace(/\[.*?\]/g,'');};b2(332,331,{});_.Tb=function yV(a){};_.Vb=function zV(a,b,c,d){return new flb.$4(b,a+'@'+d,c<0?-1:c);};_.Ub=function AV(a){var b,c,d,e,f,g;e=hlb.rV(a);f=oW(elb.q_,Tlb,66,0,0,1);b=0;d=glb.LU(e);if(d==0){return f;}g=hlb.wV(this,glb.KU(e,0));flb.j5(g.d,Qlb)||(f[b++]=g);for(c=1;c<d;c++){f[b++]=hlb.wV(this,glb.KU(e,c));}return f;};hlb.BV=function BV(){};b2(238,332,{},hlb.BV);_.Vb=function CV(a,b,c,d){return new flb.$4(b,a,-1);};elb.hW=function hW(a){return a;};function iW(a,b){return elb.jW(a,b);}elb.jW=function jW(a,b){return flb.H3(a,b);};elb.kW=function kW(a){return a.__elementTypeCategory$==null?10:a.__elementTypeCategory$;};elb.lW=function lW(a){return a.__elementTypeId$;};function mW(a,b,c,d,e,f){return elb.nW(a,b,c,d,e,0,f);}elb.nW=function nW(a,b,c,d,e,f,g){var h,i,j,k,l;k=e[f];j=f==g-1;h=j?d:0;l=elb.pW(h,k);d!=10&&wW(iW(a,g-f),b[f],c[f],h,l);if(!j){++f;for(i=0;i<k;++i){elb.rW(l,i,elb.nW(a,b,c,d,e,f,g));}}return l;};function oW(a,b,c,d,e,f){var g;g=elb.pW(e,d);e!=10&&wW(iW(a,f),b,c,e,g);return g;}elb.pW=function pW(a,b){var c=new Array(b);var d;switch(a){case 14:case 15:d=0;break;case 16:d=false;break;default:return c;}for(var e=0;e<b;++e){c[e]=d;}return c;};function qW(a){return Array.isArray(a)&&a.uc===j2;}elb.rW=function rW(a,b,c){return a[b]=c;};function sW(a,b,c){return elb.rW(a,b,c);}elb.tW=function tW(a,b){a.sc=b;};elb.uW=function uW(a,b){a.__elementTypeCategory$=b;};elb.vW=function vW(a,b){a.__elementTypeId$=b;};function wW(a,b,c,d,e){elb.tW(e,a);e.tc=b;e.uc=j2;elb.vW(e,c);elb.uW(e,d);return e;}elb.xW=function xW(a,b){elb.kW(b)!=10&&wW(flb.Ic(b),b.tc,elb.lW(b),elb.kW(b),a);return elb.hW(a);};function sX(a,b){if(zX(a)){return !!rX[b];}else if(a.tc){return !!a.tc[b];}else if(wX(a)){return !!qX[b];}else if(vX(a)){return !!pX[b];}return false;}function tX(a){return !Array.isArray(a)&&a.uc===j2;}function uX(a,b){return a!=null&&sX(a,b);}function vX(a){return typeof a===Vlb;}function wX(a){return typeof a===Wlb;}function xX(a){return a!=null&&elb.AX(a)&&!(a.uc===j2);}function yX(a,b){return elb.BX(a,b);}function zX(a){return typeof a===Xlb;}elb.AX=function AX(a){return typeof a===Plb||typeof a==='function';};elb.BX=function BX(a,b){return a&&b&&a instanceof b;};function CX(a){return a==null?null:a;}function DX(a){return Math.max(Math.min(a,Ylb),-2147483648)|0;}var pX;var qX;var rX;elb.j1=function j1(a){return a&&a.__java$exception;};function k1(a){var b;if(uX(a,22)){return a;}b=elb.j1(a);if(!b){b=new glb.rU(a);hlb.kV(b);}return b;}function l1(a){return a.backingJsObject;}function B2(a){var b;if(a==null){return false;}b=typeof a;return flb.j5(b,Vlb)||flb.j5(b,Wlb)||flb.j5(b,Xlb)||a.$implements__java_io_Serializable||Array.isArray(a);}function S3(a){var b;b=typeof a;if(flb.j5(b,Vlb)||flb.j5(b,Wlb)||flb.j5(b,Xlb)){return true;}return a!=null&&a.$implements__java_lang_Comparable;}flb.U2=function U2(){flb.U2=d2;T2=ilb.Xgb(false);ilb.Xgb(true);};flb.V2=function V2(a){return ilb.Ygb((ilb.Mgb(a),a));};flb.W2=function W2(a,b){return a3(ilb.Ygb((ilb.Mgb(a),a)),ilb.Ygb((ilb.Mgb(b),b)));};flb.X2=function X2(a,b){return flb.W2(a,b);};flb.Y2=function Y2(a,b){return ilb.Mgb(a),CX(a)===CX(b);};flb.Z2=function Z2(){return elb.$$;};flb.$2=function $2(a){return ilb.Ygb((ilb.Mgb(a),a))?1231:1237;};function _2(a){flb.U2();return flb.j5(Vlb,typeof a);}function a3(a,b){flb.U2();return a==b?0:a?1:-1;}flb.b3=function b3(a,b){flb.U2();return zX(a)?flb.d5(a,b):wX(a)?flb.$3(a,b):vX(a)?flb.X2(a,b):a.qb(b);};pX={4:1,234:1,27:1};var T2;function c3(a){if(flb.j5(typeof a,Xlb)){return true;}return a!=null&&a.$implements__java_lang_CharSequence;}flb.v3=function v3(a){a.i=u3++;};flb.w3=function w3(a){if(a.n!=null){return;}flb.L3(a);};flb.x3=function x3(a){flb.w3(a);return a.n;};flb.y3=function y3(a){return (a.f&4)!=0;};flb.z3=function z3(a){return (a.f&1)!=0;};flb.A3=function A3(){flb.v3(this);this.n=null;this.j=null;this.g=null;this.d=null;this.b=null;this.k=null;this.a=null;};flb.C3=function C3(a){var b;b=new flb.A3();b.n='Class$'+(a?'S'+a:''+b.i);b.b=b.n;b.j=b.n;return b;};function D3(a){var b;b=flb.C3(a);flb.P3(a,b);return b;}function E3(a,b){var c;c=flb.C3(a);flb.P3(a,c);c.f=b?8:0;return c;}function F3(a){var b;b=flb.C3(a);b.k=a;b.f=1;return b;}flb.H3=function H3(a,b){var c=a.a=a.a||[];return c[b]||(c[b]=a.Wb(b));};function J3(a){if(a._b()){return null;}var b=a.k;return $1[b];}flb.L3=function L3(a){if(a.$b()){var b=a.c;b._b()?a.n='['+b.k:!b.$b()?a.n='[L'+b.Yb()+';':a.n='['+b.Yb();a.b=b.Xb()+'[]';a.j=b.Zb()+'[]';return;}var c=a.g;var d=a.d;d=d.split('/');a.n=flb.O3('.',[c,flb.O3('$',d)]);a.b=flb.O3('.',[c,flb.O3('.',d)]);a.j=d[d.length-1];};flb.O3=function O3(a,b){var c=0;while(!b[c]||b[c]==''){c++;}var d=b[c++];for(;c<b.length;c++){if(!b[c]||b[c]==''){continue;}d+=a+b[c];}return d;};flb.P3=function P3(a,b){if(!a){return;}b.k=a;var d=J3(b);if(!d){$1[a]=[b];return;}d.sc=b;};b2(158,1,{},flb.A3);_.Wb=function B3(a){var b;b=new flb.A3();b.f=4;a>1?b.c=flb.H3(this,a-1):b.c=this;return b;};_.Xb=function G3(){flb.w3(this);return this.b;};_.Yb=function I3(){return flb.x3(this);};_.Zb=function K3(){flb.w3(this);return this.j;};_.$b=function M3(){return flb.y3(this);};_._b=function N3(){return flb.z3(this);};_.lb=function Q3(){return ((this.f&2)!=0?'interface ':(this.f&1)!=0?'':'class ')+(flb.w3(this),this.n);};_.f=0;_.i=0;var u3=1;flb.U3=function U3(a){return flb.j5(Wlb,typeof a)||yX(a,$wnd.java.lang.Number$impl);};flb.V3=function V3(a){flb.T3==null&&(flb.T3=new RegExp('^\\s*[+-]?(NaN|Infinity|((\\d+\\.?\\d*)|(\\.\\d+))([eE][+-]?\\d+)?[dDfF]?)\\s*$'));if(!flb.T3.test(a)){throw l1(new flb.Z4(Zlb+a+'"'));}return parseFloat(a);};flb.W3=function W3(a,b,c){var d,e,f,g,h;if(a==null){throw l1(new flb.Z4($lb));}f=ilb.Xgb(a).length;g=f>0&&(ilb.Sgb(0,ilb.Xgb(a).length),ilb.Xgb(a).charCodeAt(0)==45||(ilb.Sgb(0,ilb.Xgb(a).length),ilb.Xgb(a).charCodeAt(0)==43))?1:0;for(d=g;d<f;d++){if(i3((ilb.Sgb(d,ilb.Xgb(a).length),ilb.Xgb(a).charCodeAt(d)))==-1){throw l1(new flb.Z4(Zlb+a+'"'));}}h=parseInt(a,10);e=h<b;if(isNaN(h)){throw l1(new flb.Z4(Zlb+a+'"'));}else if(e||h>c){throw l1(new flb.Z4(Zlb+a+'"'));}return h;};flb.X3=function X3(a){return wX(a)?flb._3(a):a.ac();};flb.Y3=function Y3(a){return wX(a)?flb.d4(a):a.bc();};b2(95,1,{4:1,95:1});flb.Z3=function Z3(a,b){return f4(ilb.Zgb((ilb.Mgb(a),a)),ilb.Zgb((ilb.Mgb(b),b)));};flb.$3=function $3(a,b){return flb.Z3(a,b);};flb._3=function _3(a){return ilb.Zgb((ilb.Mgb(a),a));};flb.a4=function a4(a,b){return ilb.Mgb(a),CX(a)===CX(b);};flb.b4=function b4(){return elb.b_;};flb.c4=function c4(a){return DX(ilb.Zgb((ilb.Mgb(a),a)));};flb.d4=function d4(a){return DX(ilb.Zgb((ilb.Mgb(a),a)));};function e4(a){return flb.j5(Wlb,typeof a);}function f4(a,b){if(a<b){return -1;}if(a>b){return 1;}if(a==b){return a==0?f4(1/a,1/b):0;}return isNaN(a)?isNaN(b)?0:1:-1;}function g4(a){if(isNaN(a)){return {l:0,m:0,h:524160};}return ilb.Tgb(a);}qX={4:1,27:1,235:1,95:1};flb.Yy=function Yy(a){a.g=oW(elb.q_,Tlb,66,0,0,1);};flb.Zy=function Zy(a){hlb.kV(a);};flb.$y=function $y(a){return hlb.lV(a);};flb._y=function _y(a){if(a.j){a.backingJsObject!==_lb&&a.Hb();a.g=null;}return a;};flb.az=function az(a){return a.e;};flb.bz=function bz(d,b){if(b instanceof Object){try{b.__java$exception=d;if(navigator.userAgent.toLowerCase().indexOf('msie')!=-1&&$doc.documentMode<9){return;}var c=d;Object.defineProperties(b,{cause:{get:function(){var a=c.Fb();return a&&a.Db();}},suppressed:{get:function(){return c.Eb();}}});}catch(a){}}};flb.cz=function cz(a,b,c){var d,e,f,g,h;flb.dz(a);for(e=(a.i==null&&(a.i=oW(elb.v_,Tlb,22,0,0,1)),a.i),f=0,g=e.length;f<g;++f){d=e[f];flb.cz(d,b,'\t'+c);}h=a.e;!!h&&flb.cz(h,b,c);};flb.dz=function dz(a){var b,c,d;for(b=(a.g==null&&(a.g=flb.$y(a)),a.g),c=0,d=b.length;c<d;++c){}};flb.ez=function ez(a,b){a.backingJsObject=b;flb.bz(a,b);};flb.fz=function fz(a){return flb.gz(a,a.Gb());};flb.gz=function gz(a,b){var c;c=flb.x3(a.sc);return b==null?c:c+': '+b;};flb.hz=function hz(){flb.Yy(this);flb._y(this);this.Hb();};flb.iz=function iz(a){flb.Yy(this);this.f=a;flb._y(this);this.Hb();};flb.kz=function kz(b){if(!('stack'in b)){try{throw b;}catch(a){}}return b;};function qz(a){var b;if(a!=null){b=a.__java$exception;if(b){return b;}}return yX(a,TypeError)?new flb.W4(a):new flb.jU(a);}b2(22,1,{4:1,22:1});_.Cb=function jz(a){return new Error(a);};_.Db=function lz(){return this.backingJsObject;};_.Eb=function mz(){var a,b,c;c=(this.i==null&&(this.i=oW(elb.v_,Tlb,22,0,0,1)),this.i);b=oW(elb.o_,amb,1,c.length,5,1);for(a=0;a<c.length;a++){b[a]=c[a].backingJsObject;}return b;};_.Fb=function nz(){return flb.az(this);};_.Gb=function oz(){return this.f;};_.Hb=function pz(){flb.ez(this,flb.kz(this.Cb(flb.gz(this,this.f))));flb.Zy(this);};_.lb=function rz(){return flb.fz(this);};_.backingJsObject=_lb;_.j=true;flb.sz=function sz(){flb.hz.call(this);};flb.tz=function tz(a){flb.Yy(this);flb._y(this);this.backingJsObject=a;flb.bz(this,a);this.f=a==null?$lb:i2(a);};flb.uz=function uz(a){flb.iz.call(this,a);};b2(15,22,bmb,flb.uz);flb.vz=function vz(){flb.sz.call(this);};flb.wz=function wz(a){flb.tz.call(this,a);};flb.xz=function xz(a){flb.uz.call(this,a);};b2(41,15,bmb);flb.P2=function P2(){flb.vz.call(this);};flb.Q2=function Q2(a){flb.xz.call(this,a);};b2(69,41,bmb,flb.P2,flb.Q2);flb.iU=function iU(){flb.vz.call(this);};flb.jU=function jU(a){flb.wz.call(this,a);};flb.kU=function kU(a){flb.xz.call(this,a);};b2(100,41,bmb,flb.jU);flb.V4=function V4(){flb.iU.call(this);};flb.W4=function W4(a){flb.jU.call(this,a);};flb.X4=function X4(a){flb.kU.call(this,a);};b2(62,100,bmb,flb.V4,flb.W4,flb.X4);_.Cb=function Y4(a){return new TypeError(a);};flb.c5=function c5(a,b){ilb.Sgb(b,ilb.Xgb(a).length);return ilb.Xgb(a).charCodeAt(b);};flb.d5=function d5(a,b){return flb.e5(a,b);};flb.e5=function e5(a,b){var c,d;c=flb._3(ilb.Xgb(a));d=flb._3(ilb.Xgb(b));return c==d?0:c<d?-1:1;};flb.f5=function f5(a,b){return flb.e5(ilb.Xgb(a).toLowerCase(),ilb.Xgb(b).toLowerCase());};flb.g5=function g5(a,b){return ilb.Mgb(a),a+(''+(ilb.Mgb(b),b));};flb.h5=function h5(a){return flb.B5(ilb.wgb(a,0,a.length));};flb.i5=function i5(a){var b;b=ilb.Xgb(cmb).length;return flb.j5(ilb.Xgb(a).substr(ilb.Xgb(a).length-b,b),cmb);};flb.j5=function j5(a,b){return ilb.Mgb(a),CX(a)===CX(b);};flb.k5=function k5(a,b){ilb.Mgb(a);if(b==null){return false;}if(flb.j5(a,b)){return true;}return ilb.Xgb(a).length==ilb.Xgb(b).length&&flb.j5(ilb.Xgb(a).toLowerCase(),ilb.Xgb(b).toLowerCase());};flb.l5=function l5(){return elb.u_;};flb.m5=function m5(a){var b,c;b=0;for(c=0;c<ilb.Xgb(a).length;c++){b=(b<<5)-b+(ilb.Sgb(c,ilb.Xgb(a).length),ilb.Xgb(a).charCodeAt(c))|0;}return b;};flb.n5=function n5(a,b){return ilb.Xgb(a).indexOf(b);};flb.o5=function o5(a,b,c){return ilb.Xgb(a).indexOf(b,c);};function p5(a){return flb.j5(Xlb,typeof a);}flb.q5=function q5(a,b){return ilb.Xgb(a).lastIndexOf(b);};flb.r5=function r5(a,b,c){return ilb.Xgb(a).lastIndexOf(b,c);};flb.s5=function s5(a,b){var c;c=flb.t5(b,'([/\\\\\\.\\*\\+\\?\\|\\(\\)\\[\\]\\{\\}$^])','\\\\$1');return flb.t5(a,c,'');};flb.t5=function t5(a,b,c){c=flb.A5(c);return ilb.Xgb(a).replace(new RegExp(b,'g'),c);};flb.u5=function u5(a,b){var c,d,e,f,g,h,i,j;c=new RegExp(b,'g');i=oW(elb.u_,dmb,2,0,6,1);d=0;j=a;f=null;while(true){h=c.exec(j);if(h==null||j==''){i[d]=j;break;}else {g=h.index;i[d]=(ilb.Rgb(0,g,ilb.Xgb(j).length),ilb.Xgb(j).substr(0,g));j=flb.w5(j,g+ilb.Xgb(ilb.Xgb(h)[0]).length,ilb.Xgb(j).length);c.lastIndex=0;if(f==j){i[d]=(ilb.Rgb(0,1,ilb.Xgb(j).length),ilb.Xgb(j).substr(0,1));j=(ilb.Sgb(1,ilb.Xgb(j).length+1),ilb.Xgb(j).substr(1));}f=j;++d;}}if(ilb.Xgb(a).length>0){e=i.length;while(e>0&&i[e-1]==''){--e;}e<i.length&&(ilb.Xgb(i).length=e);}return i;};flb.v5=function v5(a,b){ilb.Sgb(b,ilb.Xgb(a).length+1);return ilb.Xgb(a).substr(b);};flb.w5=function w5(a,b,c){ilb.Rgb(b,c,ilb.Xgb(a).length);return ilb.Xgb(a).substr(b,c-b);};flb.x5=function x5(a){var b,c,d;c=ilb.Xgb(a).length;d=0;while(d<c&&(ilb.Sgb(d,ilb.Xgb(a).length),ilb.Xgb(a).charCodeAt(d)<=32)){++d;}b=c;while(b>d&&(ilb.Sgb(b-1,ilb.Xgb(a).length),ilb.Xgb(a).charCodeAt(b-1)<=32)){--b;}return d>0||b<c?(ilb.Rgb(d,b,ilb.Xgb(a).length),ilb.Xgb(a).substr(d,b-d)):a;};flb.y5=function y5(a){return String.fromCharCode.apply(null,a);};function z5(a){var b,c;if(a>=emb){b=55296+(a-emb>>10&1023)&fmb;c=56320+(a-emb&1023)&fmb;return String.fromCharCode(b)+(''+String.fromCharCode(c));}else {return String.fromCharCode(a&fmb);}}flb.A5=function A5(a){var b;b=0;while(0<=(b=ilb.Xgb(a).indexOf('\\',b))){ilb.Sgb(b+1,ilb.Xgb(a).length);ilb.Xgb(a).charCodeAt(b+1)==36?a=(ilb.Rgb(0,b,ilb.Xgb(a).length),ilb.Xgb(a).substr(0,b)+'$'+flb.v5(a,++b)):a=(ilb.Rgb(0,b,ilb.Xgb(a).length),ilb.Xgb(a).substr(0,b)+(''+flb.v5(a,++b)));}return a;};flb.B5=function B5(a){return flb.C5(a,0,a.length);};flb.C5=function C5(a,b,c){var d,e,f,g;f=b+c;ilb.Rgb(b,f,a.length);g='';for(e=b;e<f;){d=$wnd.Math.min(e+gmb,f);g+=flb.y5(ilb.Xgb(a).slice(e,d));e=d;}return g;};rX={4:1,160:1,27:1,2:1};flb.T5=function T5(a){flb.Q2.call(this,a);};b2(179,69,bmb,flb.T5);ilb.igb=function igb(a){var b;b=ilb.Xgb(a).slice();return ilb.sgb(b,a);};ilb.jgb=function jgb(a,b,c,d,e,f){var g,h,i,j,k;if(e==0){return;}if(CX(a)===CX(c)){a=ilb.Xgb(a).slice(b,b+e);b=0;}i=ilb.Xgb(c);for(h=b,j=b+e;h<j;){g=$wnd.Math.min(h+gmb,j);e=g-h;k=ilb.Xgb(a).slice(h,g);ilb.Xgb(k).splice(0,0,ilb.Xgb(d),ilb.Xgb(f?e:0));Array.prototype.splice.apply(i,k);h=g;d+=e;}};ilb.kgb=function kgb(a,b){return ilb.sgb(new Array(b),a);};ilb.lgb=function lgb(a,b,c){ilb.Xgb(a).splice(b,0,c);};ilb.mgb=function mgb(a,b,c){ilb.jgb(c,0,a,b,c.length,false);};ilb.ngb=function ngb(a,b){ilb.Xgb(a).push(b);};ilb.ogb=function ogb(a,b){ilb.Xgb(a).push(b);};ilb.pgb=function pgb(a,b,c){ilb.Xgb(a).splice(b,c);};ilb.qgb=function qgb(a,b){ilb.Xgb(a).length=b;};ilb.rgb=function rgb(a,b){ilb.Xgb(a).sort(b);};b2(375,1,{});ilb.Bgb=function Bgb(){};ilb.Cgb=function Cgb(a){switch(typeof a){case Xlb:return flb.m5(ilb.Xgb(a));case Wlb:return flb.c4(ilb.Xgb(a));case Vlb:return flb.$2(ilb.Xgb(a));default:return a==null?0:ilb.Egb(a);}};ilb.Dgb=function Dgb(){return ++ilb.Agb;};ilb.Egb=function Egb(a){return a.$H||(a.$H=ilb.Dgb());};b2(329,1,{},ilb.Bgb);ilb.Agb=0;ilb.Fgb=function Fgb(a){if(!a){throw l1(new flb.i4());}};ilb.Ggb=function Ggb(a,b){if(!a){throw l1(new flb.j4(b));}};ilb.Hgb=function Hgb(a,b){if(0>a){throw l1(new flb.j4('fromIndex: 0 > toIndex: '+a));}if(a>b){throw l1(new flb.S2('fromIndex: 0, toIndex: '+a+hmb+b));}};ilb.Igb=function Igb(a){if(a<0){throw l1(new flb.U4('Negative array size: '+a));}};ilb.Jgb=function Jgb(a,b){if(a!=b){throw l1(new llb.Qcb());}};ilb.Kgb=function Kgb(a){if(!a){throw l1(new llb.Vdb());}};ilb.Lgb=function Lgb(a,b){if(a<0||a>=b){throw l1(new flb.Q2(imb+a+jmb+b));}};ilb.Mgb=function Mgb(a){if(a==null){throw l1(new flb.V4());}return a;};ilb.Ngb=function Ngb(a,b){if(a==null){throw l1(new flb.X4(b));}};ilb.Ogb=function Ogb(a,b){if(a<0||a>b){throw l1(new flb.Q2(imb+a+jmb+b));}};ilb.Pgb=function Pgb(a,b,c){if(a<0||b>c){throw l1(new flb.Q2(kmb+a+lmb+b+', size: '+c));}if(a>b){throw l1(new flb.j4(kmb+a+' > toIndex: '+b));}};ilb.Qgb=function Qgb(a){if(!a){throw l1(new flb.k4());}};ilb.Rgb=function Rgb(a,b,c){if(a<0||b>c||b<a){throw l1(new flb.T5(kmb+a+lmb+b+hmb+c));}};ilb.Sgb=function Sgb(a,b){if(a<0||a>=b){throw l1(new flb.T5(imb+a+jmb+b));}};ilb.Tgb=function Tgb(a){var b,c;b=new ArrayBuffer(8);ilb.Xgb(new Float64Array(b))[0]=a;c=ilb.Xgb(new Uint32Array(b));return ilb.$gb(c[0]|0,c[1]|0);};ilb.Ugb=function Ugb(a,b){return a[b];};ilb.Vgb=function Vgb(a){return a===undefined;};ilb.Wgb=function Wgb(a){return a>>>0;};ilb.Xgb=function Xgb(a){return a;};ilb.Ygb=function Ygb(a){return a;};ilb.Zgb=function Zgb(a){return a;};b2(330,1,{});elb.o_=D3(1);elb.A$=D3(0);elb.H$=D3(331);elb.E$=D3(237);elb.G$=D3(332);elb.F$=D3(238);elb.$$=D3(234);elb.a_=D3(158);elb.n_=D3(95);elb.b_=D3(235);elb.v_=D3(22);elb.d_=D3(15);elb.p_=D3(41);elb.g_=D3(69);elb.i_=D3(100);elb.l_=D3(62);elb.u_=D3(2);elb.t_=D3(179);elb.C0=D3(329);elb.D0=D3(330);jlb.Nc=function Nc(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N,O,P,Q;this.c=a.length;this.d=a[0].length;G=$wnd.Math.min(this.c,this.d);this.e=oW(elb.GX,mmb,6,$wnd.Math.min(this.c+1,this.d),15,1);this.a=mW(elb.GX,[Tlb,mmb],[12,6],15,[this.c,G],2);this.b=mW(elb.GX,[Tlb,mmb],[12,6],15,[this.d,this.d],2);e=oW(elb.GX,mmb,6,this.d,15,1);Q=oW(elb.GX,mmb,6,this.c,15,1);D=$wnd.Math.min(this.c-1,this.d);F=$wnd.Math.max(0,$wnd.Math.min(this.d-2,this.c));for(v=0;v<$wnd.Math.max(D,F);v++){if(v<D){this.e[v]=0;for(l=v;l<this.c;l++){this.e[v]=jlb.Oc(this.e[v],a[l][v]);}if(this.e[v]!=0){a[v][v]<0&&(this.e[v]=-this.e[v]);for(k=v;k<this.c;k++){a[k][v]/=this.e[v];}a[v][v]+=1;}this.e[v]=-this.e[v];}for(q=v+1;q<this.d;q++){if(v<D&this.e[v]!=0){P=0;for(l=v;l<this.c;l++){P+=a[l][v]*a[l][q];}P=-P/a[v][v];for(k=v;k<this.c;k++){a[k][q]+=P*a[k][v];}}e[q]=a[v][q];}if(true&v<D){for(k=v;k<this.c;k++){this.a[k][v]=a[k][v];}}if(v<F){e[v]=0;for(l=v+1;l<this.d;l++){e[v]=jlb.Oc(e[v],e[l]);}if(e[v]!=0){e[v+1]<0&&(e[v]=-e[v]);for(m=v+1;m<this.d;m++){e[m]/=e[v];}e[v+1]+=1;}e[v]=-e[v];if(v+1<this.c&e[v]!=0){for(m=v+1;m<this.c;m++){Q[m]=0;}for(r=v+1;r<this.d;r++){for(n=v+1;n<this.c;n++){Q[n]+=e[r]*a[n][r];}}for(s=v+1;s<this.d;s++){P=-e[s]/e[v+1];for(n=v+1;n<this.c;n++){a[n][s]+=P*Q[n];}}}for(k=v+1;k<this.d;k++){this.b[k][v]=e[k];}}}H=$wnd.Math.min(this.d,this.c+1);D<this.d&&(this.e[D]=a[D][D]);this.c<H&&(this.e[H-1]=0);F+1<H&&(e[F]=a[F][H-1]);e[H-1]=0;for(t=D;t<G;t++){for(k=0;k<this.c;k++){this.a[k][t]=0;}this.a[t][t]=1;}for(w=D-1;w>=0;w--){if(this.e[w]!=0){for(p=w+1;p<G;p++){P=0;for(l=w;l<this.c;l++){P+=this.a[l][w]*this.a[l][p];}P=-P/this.a[w][w];for(m=w;m<this.c;m++){this.a[m][p]+=P*this.a[m][w];}}for(n=w;n<this.c;n++){this.a[n][w]=-this.a[n][w];}this.a[w][w]=1+this.a[w][w];for(k=0;k<w-1;k++){this.a[k][w]=0;}}else {for(k=0;k<this.c;k++){this.a[k][w]=0;}this.a[w][w]=1;}}for(A=this.d-1;A>=0;A--){if(A<F&e[A]!=0){for(p=A+1;p<G;p++){P=0;for(l=A+1;l<this.d;l++){P+=this.b[l][A]*this.b[l][p];}P=-P/this.b[A+1][A];for(m=A+1;m<this.d;m++){this.b[m][p]+=P*this.b[m][A];}}}for(k=0;k<this.d;k++){this.b[k][A]=0;}this.b[A][A]=1;}I=H-1;h=$wnd.Math.pow(2,-52);while(H>0){for(u=H-2;u>=-1;u--){if(u==-1){break;}if($wnd.Math.abs(e[u])<=h*($wnd.Math.abs(this.e[u])+$wnd.Math.abs(this.e[u+1]))){e[u]=0;break;}}if(u==H-2){B=4;}else {for(C=H-1;C>=u;C--){if(C==u){break;}P=(C!=H?$wnd.Math.abs(e[C]):0)+(C!=u+1?$wnd.Math.abs(e[C-1]):0);if($wnd.Math.abs(this.e[C])<=h*P){this.e[C]=0;break;}}if(C==u){B=3;}else if(C==H-1){B=1;}else {B=2;u=C;}}++u;switch(B){case 1:{i=e[H-2];e[H-2]=0;for(p=H-2;p>=u;p--){P=jlb.Oc(this.e[p],i);d=this.e[p]/P;M=i/P;this.e[p]=P;if(p!=u){i=-M*e[p-1];e[p-1]=d*e[p-1];}for(k=0;k<this.d;k++){P=d*this.b[k][p]+M*this.b[k][H-1];this.b[k][H-1]=-M*this.b[k][p]+d*this.b[k][H-1];this.b[k][p]=P;}}}break;case 2:{i=e[u-1];e[u-1]=0;for(p=u;p<H;p++){P=jlb.Oc(this.e[p],i);d=this.e[p]/P;M=i/P;this.e[p]=P;i=-M*e[p];e[p]=d*e[p];for(k=0;k<this.c;k++){P=d*this.a[k][p]+M*this.a[k][u-1];this.a[k][u-1]=-M*this.a[k][p]+d*this.a[k][u-1];this.a[k][p]=P;}}}break;case 3:{J=$wnd.Math.max($wnd.Math.max($wnd.Math.max($wnd.Math.max($wnd.Math.abs(this.e[H-1]),$wnd.Math.abs(this.e[H-2])),$wnd.Math.abs(e[H-2])),$wnd.Math.abs(this.e[u])),$wnd.Math.abs(e[u]));N=this.e[H-1]/J;O=this.e[H-2]/J;g=e[H-2]/J;L=this.e[u]/J;f=e[u]/J;b=((O+N)*(O-N)+g*g)/2;c=N*g*(N*g);K=0;if(b!=0|c!=0){K=$wnd.Math.sqrt(b*b+c);b<0&&(K=-K);K=c/(b+K);}i=(L+N)*(L-N)+K;j=L*f;for(p=u;p<H-1;p++){P=jlb.Oc(i,j);d=i/P;M=j/P;p!=u&&(e[p-1]=P);i=d*this.e[p]+M*e[p];e[p]=d*e[p]-M*this.e[p];j=M*this.e[p+1];this.e[p+1]=d*this.e[p+1];for(l=0;l<this.d;l++){P=d*this.b[l][p]+M*this.b[l][p+1];this.b[l][p+1]=-M*this.b[l][p]+d*this.b[l][p+1];this.b[l][p]=P;}P=jlb.Oc(i,j);d=i/P;M=j/P;this.e[p]=P;i=d*e[p]+M*this.e[p+1];this.e[p+1]=-M*e[p]+d*this.e[p+1];j=M*e[p+1];e[p+1]=d*e[p+1];if(p<this.c-1){for(k=0;k<this.c;k++){P=d*this.a[k][p]+M*this.a[k][p+1];this.a[k][p+1]=-M*this.a[k][p]+d*this.a[k][p+1];this.a[k][p]=P;}}}e[H-2]=i;}break;case 4:{if(this.e[u]<=0){this.e[u]=this.e[u]<0?-this.e[u]:0;for(k=0;k<=I;k++){this.b[k][u]=-this.b[k][u];}}while(u<I){if(this.e[u]>=this.e[u+1]){break;}P=this.e[u];this.e[u]=this.e[u+1];this.e[u+1]=P;if(u<this.d-1){for(k=0;k<this.d;k++){P=this.b[k][u+1];this.b[k][u+1]=this.b[k][u];this.b[k][u]=P;}}if(u<this.c-1){for(k=0;k<this.c;k++){P=this.a[k][u+1];this.a[k][u+1]=this.a[k][u];this.a[k][u]=P;}}++u;}--H;}}}};jlb.Oc=function Oc(a,b){var c;if($wnd.Math.abs(a)>$wnd.Math.abs(b)){c=b/a;c=$wnd.Math.abs(a)*$wnd.Math.sqrt(1+c*c);}else if(b!=0){c=a/b;c=$wnd.Math.abs(b)*$wnd.Math.sqrt(1+c*c);}else {c=0;}return c;};b2(156,1,nmb,jlb.Nc);_.c=0;_.d=0;elb.KX=D3(156);klb.Qc=function Qc(){klb.Qc=d2;klb.Pc=wW(iW(elb.IX,1),omb,6,15,[0,pmb,14286847,13402367,12779264,16758197,9474192,3166456,16715021,9494608,11789301,11230450,9109248,12560038,15780000,16744448,16777008,2093087,8442339,9388244,4062976,15132390,12567239,10921643,9083335,10255047,14706227,15765664,5296208,13140019,8224944,12750735,6721423,12419299,16752896,10889513,6076625,7351984,qmb,9764863,9756896,7586505,5551541,3907230,2396047,687500,27013,12632256,16767375,10909043,6717568,10380213,13924864,9699476,4366000,5707663,51456,7394559,16777159,14286791,13107143,10747847,9437127,6422471,4587463,3211207,2097095,65436,58997,54354,48952,43812,5096191,5089023,2200790,2522539,2516630,1528967,13684960,16765219,12105936,10900557,5724513,10375093,11230208,7688005,4358806,4325478,32000,7384058,47871,41471,36863,33023,27647,5528818,7888099,9064419,10565332,11739092,11739066,11734438,12389767,13041766,13369433,13697103,14221381,14680120,15073326,15400998,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,13158600,1334015,56540,15075850,15132160,56540,15075850,15461355,8553170,1016335,1016335,1334015,15132160,3289770,14456450,16422400,16422400,11819700,3289770,1016335]);};klb.Rc=function Rc(a){a.v=new xlb.cG();};klb.Sc=function Sc(a){var b,c;if((a.F&32)!=0)return;c=klb.Mt(a.K);if(c!=null){if(a.w.a==0&&a.w.b==0){b=a.Q.c*klb.vk(a.K);a.v=klb.Md(a);klb.bd(a,b);klb.Jd(a,null,b,0);}klb.Ps(a,DX(a.A));a.K.O!=1&&klb.Kd(a,448);klb.Js(a,c,a.w.a,a.w.b+rmb*a.A);}};klb.Tc=function Tc(a,b){return a==null?b:b==null?a:a+','+b;};klb.Uc=function Uc(a){var b;b=a.Q.c*(a.r!=0?a.r:klb.vk(a.K));a.X=b*0.06;a.S=b*0.15;a.R=b*0.38;a.V=b*0.47;a.W=DX(b*a.I*0.6+0.5);a.U=b*0.12;a.Y=b*0.4;a.A=b*0.5+0.5;};klb.Vc=function Vc(a){var b,c;a=n1(a,smb);for(c=0;c<(wlb.UF(),wlb.TF).length;c++)if(x1(a,wlb.TF[c]))return wlb.SF[c];b=new flb.S5('R');I1(n1(a,tmb),0)&&(b.a+='0',b);I1(n1(a,umb),0)&&(b.a+='3',b);I1(n1(a,vmb),0)&&(b.a+='4',b);I1(n1(a,wmb),0)&&(b.a+='5',b);I1(n1(a,xmb),0)&&(b.a+='6',b);I1(n1(a,ymb),0)&&(b.a+='7',b);I1(n1(a,zmb),0)&&(b.a+='8',b);return b.a;};klb.Wc=function Wc(a,b,c,d){var e,f,g;e=new klb.Sd();f=new klb.Sd();e.a=b.a;e.c=b.c;e.b=(b.a+b.b)/2;e.d=(b.c+b.d)/2;f.a=e.b;f.c=e.d;f.b=b.b;f.d=b.d;if(klb.Gd(a,e)){klb.Kd(a,a.p[c]);g=Amb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.a),(Jlb.m8(),Jlb.i8))))+Bmb+Cmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.c),Jlb.i8)))+Bmb+Dmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.b),Jlb.i8)))+Bmb+Emb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.d),Jlb.i8)))+Bmb+Fmb+a.d+Bmb+Gmb+a.j+Hmb;klb.Ss(a,g);}if(klb.Gd(a,f)){klb.Kd(a,a.p[d]);g=Amb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(f.a),(Jlb.m8(),Jlb.i8))))+Bmb+Cmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(f.c),Jlb.i8)))+Bmb+Dmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(f.b),Jlb.i8)))+Bmb+Emb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(f.d),Jlb.i8)))+Bmb+Fmb+a.d+Bmb+Gmb+a.j+Hmb;klb.Ss(a,g);}klb.Kd(a,a.P);};klb.Xc=function Xc(a,b,c,d){var e,f,g,h,i,j,k,l,m;l=(b.b-b.a)/10;m=(b.d-b.c)/10;e=new klb.Sd();if(klb.dl(a.K,klb.un(a.K,c,d))){f=-3;g=-3;}else {f=a.p[c];g=a.p[d];}klb.Kd(a,f);e.a=b.a;e.c=b.c;e.b=b.a+l*2;e.d=b.c+m*2;i=Amb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.a),(Jlb.m8(),Jlb.i8))))+Bmb+Cmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.c),Jlb.i8)))+Bmb+Dmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.b),Jlb.i8)))+Bmb+Emb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.d),Jlb.i8)))+Bmb+Fmb+a.d+Bmb+Gmb+a.j+Hmb;klb.Ss(a,i);e.a=b.a+l*4;e.c=b.c+m*4;e.b=b.a+l*5;e.d=b.c+m*5;j=Amb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.a),Jlb.i8)))+Bmb+Cmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.c),Jlb.i8)))+Bmb+Dmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.b),Jlb.i8)))+Bmb+Emb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.d),Jlb.i8)))+Bmb+Fmb+a.d+Bmb+Gmb+a.j+Hmb;klb.Ss(a,j);klb.Kd(a,g);e.a=b.a+l*5;e.c=b.c+m*5;e.b=b.a+l*6;e.d=b.c+m*6;k=Amb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.a),Jlb.i8)))+Bmb+Cmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.c),Jlb.i8)))+Bmb+Dmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.b),Jlb.i8)))+Bmb+Emb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.d),Jlb.i8)))+Bmb+Fmb+a.d+Bmb+Gmb+a.j+Hmb;klb.Ss(a,k);e.a=b.a+l*8;e.c=b.c+m*8;e.b=b.b;e.d=b.d;h=Amb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.a),Jlb.i8)))+Bmb+Cmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.c),Jlb.i8)))+Bmb+Dmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.b),Jlb.i8)))+Bmb+Emb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.d),Jlb.i8)))+Bmb+Fmb+a.d+Bmb+Gmb+a.j+Hmb;klb.Ss(a,h);klb.Kd(a,a.P);};klb.Yc=function Yc(a,b,c){klb.Ks(a,b-a.U/2,c-a.U/2,a.U);};klb.Zc=function Zc(a,b,c,d){var e;if(klb.dl(a.K,klb.un(a.K,c,d))){klb.Kd(a,-3);e=Amb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(b.a),(Jlb.m8(),Jlb.i8))))+Bmb+Cmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(b.c),Jlb.i8)))+Bmb+Dmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(b.b),Jlb.i8)))+Bmb+Emb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(b.d),Jlb.i8)))+Bmb+Fmb+a.d+Bmb+Gmb+a.j+Hmb;klb.Ss(a,e);klb.Kd(a,a.P);}else if(a.p[c]!=a.p[d]){klb.Wc(a,b,c,d);}else if(a.p[c]!=0){klb.Kd(a,a.p[c]);e=Amb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(b.a),(Jlb.m8(),Jlb.i8))))+Bmb+Cmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(b.c),Jlb.i8)))+Bmb+Dmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(b.b),Jlb.i8)))+Bmb+Emb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(b.d),Jlb.i8)))+Bmb+Fmb+a.d+Bmb+Gmb+a.j+Hmb;klb.Ss(a,e);klb.Kd(a,a.P);}else {e=Amb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(b.a),(Jlb.m8(),Jlb.i8))))+Bmb+Cmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(b.c),Jlb.i8)))+Bmb+Dmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(b.b),Jlb.i8)))+Bmb+Emb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(b.d),Jlb.i8)))+Bmb+Fmb+a.d+Bmb+Gmb+a.j+Hmb;klb.Ss(a,e);}};klb.$c=function $c(a,b,c,d,e,f,g){g.a=b.a+e;g.c=b.c+f;g.b=b.b+e;g.d=b.d+f;klb.Zc(a,g,c,d);};klb._c=function _c(a,b,c,d){var e,f,g,h,i,j,k,l,m,n,o,p;l=b.b-b.a;o=b.d-b.c;i=$wnd.Math.sqrt(l*l+o*o);j=2*R1(y1($wnd.Math.round(i/(4*a.X))));m=l/(j-1);p=o/(j-1);if(klb.dl(a.K,klb.un(a.K,c,d))){e=-3;f=-3;}else {e=a.p[c];f=a.p[d];}k=b.a-a.X/2;n=b.c-a.X/2;klb.Kd(a,e);for(h=0;h<(j/2|0);h++){klb.Ks(a,k,n,a.X);k+=m;n+=p;}klb.Kd(a,f);for(g=0;g<(j/2|0);g++){klb.Ks(a,k,n,a.X);k+=m;n+=p;}klb.Kd(a,a.P);};klb.ad=function ad(a,b,c,d){var e,f,g,h,i,j,k,l,m,n,o,p,q,r;i=(b.c-b.d)/9;n=(b.b-b.a)/9;j=b.b+i;o=b.d+n;k=b.b-i;p=b.d-n;l=(b.a+j)/2;q=(b.c+o)/2;m=(b.a+k)/2;r=(b.c+p)/2;g=new xlb.ZF(3);h=new xlb.ZF(4);xlb.YF(g,b.a,b.c);xlb.YF(g,l,q);xlb.YF(g,m,r);xlb.YF(h,m,r);xlb.YF(h,l,q);xlb.YF(h,j,o);xlb.YF(h,k,p);if(klb.dl(a.K,klb.un(a.K,c,d))){e=-3;f=-3;}else {e=a.p[c];f=a.p[d];if(a.K.O!=1){f=klb.ed(a,c);e==klb.ek(a.K,c)&&(e=f);}}klb.Kd(a,e);klb.Is(a,g);klb.Kd(a,f);klb.Is(a,h);klb.Kd(a,a.P);};klb.bd=function bd(a,b){var c,d;for(d=0;d<a.Z.a.length;d++)a.v=xlb.bG(a.v,llb.Bi(a.Z,d));klb.cd(a,b);c=0.1*b;a.v.c-=c;a.v.d-=c;a.v.b+=2*c;a.v.a+=2*c;};klb.cd=function cd(a,b){var c,d,e,f,g,h,i;e=oW(elb.g1,Imb,6,a.K.q,16,1);for(d=0;d<a.K.r;d++){if(klb.bl(a.K,d)){e[klb.zk(a.K,0,d)]=true;e[klb.zk(a.K,1,d)]=true;}}g=new xlb.cG();for(c=0;c<a.K.q;c++){f=I1(n1(klb.pk(a.K,c),Jmb),0)?b*0.47:e[c]?b*0.38:0;if(f!=0){h=klb.Mh(a.Q,klb.rk(a.K,c));i=klb.Nh(a.Q,klb.sk(a.K,c));xlb.aG(g,h-f,i-f,f*2,f*2);a.v=xlb.bG(a.v,g);}}};klb.dd=function dd(a,b){var c;c=a.L!=0?a.L:a.C!=0?a.C:-1;return Alb.NT(b,c);};klb.ed=function ed(a,b){var c,d;if((a.F&4224)!=0)return a.p[b];d=klb.fd(a,b);if(d==-1){c=klb.bn(a.K,b);if(c!=-1){b=c;d=klb.fd(a,b);}}if(d==-1)return a.p[b];switch(d&255){case 1:return 384;case 2:return 64;default:return 448;}};klb.fd=function fd(a,b){var c,d,e;e=-1;d=-1;if((a.F&128)!=0)return e;if(klb._k(a.K,b)){e=klb.ik(a.K,b);d=klb.hk(a.K,b);}c=klb.dn(a.K,b);if(c!=-1){e=klb.Ek(a.K,c);d=klb.Dk(a.K,c);}e!=-1&&e!=0&&(e|=d<<8);return e;};klb.gd=function gd(a){var b,c,d,e,f;klb.Ns(a,2*a.R);e=new klb.Sd();for(d=0;d<a.K.r;d++){b=klb.zk(a.K,0,d);c=klb.zk(a.K,1,d);if(klb.bl(a.K,d)){e.a=klb.Mh(a.Q,klb.rk(a.K,b));e.c=klb.Nh(a.Q,klb.sk(a.K,b));e.b=klb.Mh(a.Q,klb.rk(a.K,c));e.d=klb.Nh(a.Q,klb.sk(a.K,c));klb.Kd(a,-2);f=Amb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.a),(Jlb.m8(),Jlb.i8))))+Bmb+Cmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.c),Jlb.i8)))+Bmb+Dmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.b),Jlb.i8)))+Bmb+Emb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e.d),Jlb.i8)))+Bmb+Fmb+a.d+Bmb+Gmb+a.j+Hmb;klb.Ss(a,f);}}};klb.hd=function hd(a){var b,c,d,e,f,g,h;if(a.K.K){g=a.V;klb.Kd(a,-7);for(b=0;b<a.K.f;b++)I1(n1(klb.pk(a.K,b),Jmb),0)&&klb.Ks(a,klb.Mh(a.Q,klb.rk(a.K,b))-g,klb.Nh(a.Q,klb.sk(a.K,b))-g,2*g);klb.Ns(a,2*a.V);f=new klb.Sd();for(e=0;e<a.K.r;e++){c=klb.zk(a.K,0,e);d=klb.zk(a.K,1,e);if(I1(n1(n1(klb.pk(a.K,c),klb.pk(a.K,d)),Jmb),0)){f.a=klb.Mh(a.Q,klb.rk(a.K,c));f.c=klb.Nh(a.Q,klb.sk(a.K,c));f.b=klb.Mh(a.Q,klb.rk(a.K,d));f.d=klb.Nh(a.Q,klb.sk(a.K,d));h=Amb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(f.a),(Jlb.m8(),Jlb.i8))))+Bmb+Cmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(f.c),Jlb.i8)))+Bmb+Dmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(f.b),Jlb.i8)))+Bmb+Emb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(f.d),Jlb.i8)))+Bmb+Fmb+a.d+Bmb+Gmb+a.j+Hmb;klb.Ss(a,h);}}}};klb.jd=function jd(a){var b,c,d,e;if(a.K.K){klb.Kd(a,320);if((a.F&8)!=0)for(b=0;b<a.K.f;b++)I1(n1(klb.pk(a.K,b),-536870913),0)&&klb.Ks(a,klb.Mh(a.Q,klb.rk(a.K,b))-a.Y/2,klb.Nh(a.Q,klb.sk(a.K,b))-a.Y/2,a.Y);for(e=0;e<a.K.g;e++){if(klb.Jk(a.K,e)!=0){c=klb.zk(a.K,0,e);d=klb.zk(a.K,1,e);klb.Ks(a,(klb.Mh(a.Q,klb.rk(a.K,c))+klb.Mh(a.Q,klb.rk(a.K,d))-a.Y)/2,(klb.Nh(a.Q,klb.sk(a.K,c))+klb.Nh(a.Q,klb.sk(a.K,d))-a.Y)/2,a.Y);}}}};klb.kd=function kd(a){a.I=1;a.Q=new klb.Oh();a.Z=new llb.Qi();a.T=new llb.Qi();a.s=oW(elb.g1,Imb,6,a.K.q,16,1);a.w=new xlb.VF();a.P=0;a.B=-1;klb.Od(a);};klb.ld=function ld(a,b){var c;if(klb.hn(a.K,b)==0)return false;for(c=0;c<klb.hn(a.K,b);c++)if(!klb.dl(a.K,klb.yn(a.K,b,c)))return false;return true;};klb.md=function md(a){var b;a.q=oW(elb.g1,Imb,6,a.K.q,16,1);for(b=0;b<a.K.r;b++){a.q[klb.zk(a.K,0,b)]=true;a.q[klb.zk(a.K,1,b)]=true;}};klb.nd=function nd(a,b){var c;if(klb.xn(a.K,b)!=2)return false;for(c=0;c<2;c++)if(klb.zn(a.K,b,c)!=2)return false;return true;};klb.od=function od(a,b,c,d,e){var f,g,h,i,j,k,l,m,n,o;m=false;e.a=0;e.b=0;d>0?f=Kmb:f=Lmb;o=klb.yk(a.K,b,c);for(k=0;k<klb.xn(a.K,b);k++){g=klb.yn(a.K,b,k);h=o;klb.zk(a.K,0,g)==b?l=klb.zk(a.K,1,g):l=klb.zk(a.K,0,g);if(l==c)continue;n=klb.yk(a.K,b,l);h<n&&(h+=Mmb);i=h-n;if(d>0){i<Nmb&&(m=true);i>Kmb&&(i=Kmb);i<0.523598776&&(i=0.523598776);if(i<=f){f=i;j=a.S*$wnd.Math.tan(f-Omb)/2;e.a=-(j*$wnd.Math.sin(h));e.b=-(j*$wnd.Math.cos(h));}}else {i>=Nmb&&(m=true);i<Lmb&&(i=Lmb);i>5.759586531&&(i=5.759586531);if(i>=f){f=i;j=a.S*$wnd.Math.tan(4.712388981-f)/2;e.a=-(j*$wnd.Math.sin(h));e.b=-(j*$wnd.Math.cos(h));}}}return m;};klb.pd=function pd(a,b,c,d){var e;if(b==0){c<0?d.a=a.S:d.a=-a.S;d.b=0;return;}e=$wnd.Math.atan(c/b);b<0&&(e+=Nmb);d.a=-(a.S*$wnd.Math.sin(e));d.b=a.S*$wnd.Math.cos(e);};klb.qd=function qd(a,b,c,d){var e,f,g,h,i,j,k,l,m,n,o,p;e=new klb.Sd();i=new klb.Sd();k=new xlb.VF();j=new xlb.VF();g=klb.zk(a.K,0,c);h=klb.zk(a.K,1,c);if(d){m=b.a;b.a=b.b;b.b=m;m=b.c;b.c=b.d;b.d=m;n=g;g=h;h=n;}if(!klb.Gd(a,b))return;if(klb.Co(a.K,c)){e.a=b.a;e.c=b.c;e.b=b.b;e.d=b.d;l=d?-klb.Zn(a.K,c):klb.Zn(a.K,c);l==0&&(l=1);klb.pd(a,b.b-b.a,b.d-b.c,k);if(l>0){i.a=b.a+k.a;i.c=b.c+k.b;i.b=b.b+k.a;i.d=b.d+k.b;if(klb.od(a,g,h,1,j)||klb.xn(a.K,g)>1){i.a+=j.a+k.b;i.c+=j.b-k.a;}}else {i.a=b.a-k.a;i.c=b.c-k.b;i.b=b.b-k.a;i.d=b.d-k.b;if(klb.od(a,g,h,-1,j)||klb.xn(a.K,g)>1){i.a+=j.a+k.b;i.c+=j.b-k.a;}}klb.Kk(a.K,c)==386&&klb.Fd(e,i);klb.Gd(a,e)&&klb.Zc(a,e,g,h);klb.Kk(a.K,c)==64?klb.Gd(a,i)&&klb.Xc(a,i,g,h):klb.Gd(a,i)&&klb.Zc(a,i,g,h);}else {klb.pd(a,b.b-b.a,b.d-b.c,k);o=k.a/2;p=k.b/2;f=false;e.a=b.a+o;e.c=b.c+p;e.b=b.b+o;e.d=b.d+p;if(klb.xn(a.K,g)>1){if(klb.od(a,g,h,1,j)){e.a+=j.a;e.c+=j.b;if(klb.xn(a.K,g)==2){if(j.a!=0||j.b!=0){e.a+=k.b;e.c-=k.a;}}}else {a.o[g]=new xlb.WF(e.a,e.c);}}i.a=b.a-o;i.c=b.c-p;i.b=b.b-o;i.d=b.d-p;if(klb.xn(a.K,g)>1){if(klb.od(a,g,h,0,j)){i.a+=j.a;i.c+=j.b;if(klb.xn(a.K,g)==2){if(j.a!=0||j.b!=0){i.a+=k.b;i.c-=k.a;}}}else {a.o[g]=new xlb.WF(i.a,i.c);f=true;}}klb.Kk(a.K,c)==386&&klb.Fd(e,i);if(klb.Kk(a.K,c)==64){if(f){klb.Xc(a,e,g,h);klb.Zc(a,i,g,h);}else {klb.Zc(a,e,g,h);klb.Xc(a,i,g,h);}}else {klb.Zc(a,e,g,h);klb.Zc(a,i,g,h);}}};klb.rd=function rd(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o;k=a.P;l=a.D;if((a.F&Pmb)!=0){a.P=-6;a.D=-8355712;klb.Kd(a,1);}a.o=oW(elb.ZZ,amb,50,a.K.q,0,1);for(i=0;i<a.K.r;i++)(klb.Kk(a.K,i)==2||klb.Kk(a.K,i)==386||klb.Kk(a.K,i)==64)&&klb.ud(a,i);for(j=0;j<a.K.r;j++)klb.Kk(a.K,j)!=2&&klb.Kk(a.K,j)!=386&&klb.Kk(a.K,j)!=64&&klb.ud(a,j);if((a.F&64)==0){for(h=0;h<a.K.r;h++){if(klb.Ck(a.K,h)!=0){e=null;klb.Ck(a.K,h)==1||klb.Ck(a.K,h)==2?(klb.Hk(a.K,h)==2||klb.Ek(a.K,h)==0||b[klb.Ek(a.K,h)][klb.Dk(a.K,h)]>1)&&(klb.Ck(a.K,h)==1?e=klb.Hk(a.K,h)==2?'E':klb.fl(a.K,h)?'p':'P':e=klb.Hk(a.K,h)==2?'Z':klb.fl(a.K,h)?'m':'M'):e='?';if(e!=null){klb.Ps(a,(a.W*2+1)/3|0);klb.Kd(a,klb.dl(a.K,h)?-3:a.K.O==1||(a.F&Qmb)!=0?a.P:448);c=klb.zk(a.K,0,h);d=klb.zk(a.K,1,h);n=(klb.Mh(a.Q,klb.rk(a.K,c))+klb.Mh(a.Q,klb.rk(a.K,d)))/2;o=(klb.Nh(a.Q,klb.sk(a.K,c))+klb.Nh(a.Q,klb.sk(a.K,d)))/2;f=(klb.Mh(a.Q,klb.rk(a.K,c))-klb.Mh(a.Q,klb.rk(a.K,d)))/3;g=(klb.Nh(a.Q,klb.sk(a.K,c))-klb.Nh(a.Q,klb.sk(a.K,d)))/3;klb.xd(a,n+g,o-f,e,true);klb.Kd(a,a.P);klb.Ps(a,a.W);}}}}if((a.F&4)!=0){klb.Ps(a,(a.W*2+1)/3|0);klb.Kd(a,384);for(h=0;h<a.K.r;h++){c=klb.zk(a.K,0,h);d=klb.zk(a.K,1,h);m=klb.to(a.K,h)?'d':klb.po(a.K,h)?'a':'';n=(klb.Mh(a.Q,klb.rk(a.K,c))+klb.Mh(a.Q,klb.rk(a.K,d)))/2;o=(klb.Nh(a.Q,klb.sk(a.K,c))+klb.Nh(a.Q,klb.sk(a.K,d)))/2;klb.xd(a,n,o,m+(''+h),true);}klb.Kd(a,a.P);klb.Ps(a,a.W);}if((a.F&Pmb)!=0){a.P=k;a.D=l;}};klb.sd=function sd(a){var b,c;for(c=new llb.Wbb(a.T);c.a<c.c.a.length;){b=llb.Vbb(c);klb.Kd(a,b.a);klb.Yc(a,b.b,b.c);}klb.Kd(a,a.P);};klb.td=function td(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N,O,P,Q,R,S,T,U,V,W,X,Y,Z,$;a.J||klb.Ls(a,b,(klb.jk(a.K,b),klb.Mh(a.Q,klb.rk(a.K,b))),klb.Nh(a.Q,klb.sk(a.K,b)));M=null;if(klb.dk(a.K,b)!=0){U=$wnd.Math.abs(klb.dk(a.K,b))==1?'':''+$wnd.Math.abs(klb.dk(a.K,b));M=klb.dk(a.K,b)<0?U+'-':U+'+';}D=null;N=klb.pk(a.K,b);if(s1(N,0)!=0){I1(n1(N,Rmb),0)&&(D=D==null?'*':D+','+'*');I1(n1(N,Smb),0)&&(D=D==null?'!*':D+','+'!*');I1(n1(N,Tmb),0)?D=D==null?'ha':D+','+'ha':I1(n1(N,2),0)?D=D==null?'a':D+','+'a':I1(n1(N,4),0)&&(D=D==null?'!a':D+','+'!a');I1(n1(N,Qmb),0)&&(D=D==null?'s':D+','+'s');if(I1(n1(N,Umb),0)){w=n1(N,Umb);s1(w,1792)==0?D=D==null?'h0':D+','+'h0':s1(w,1664)==0?D=D==null?'h1':D+','+'h1':s1(w,1408)==0?D=D==null?'h2':D+','+'h2':s1(w,128)==0?D=D==null?'h>0':D+','+'h>0':s1(w,384)==0?D=D==null?'h>1':D+','+'h>1':s1(w,896)==0?D=D==null?'h>2':D+','+'h>2':s1(w,Vmb)==0?D=D==null?'h<3':D+','+'h<3':s1(w,1536)==0&&(D=D==null?'h<2':D+','+'h<2');}if(I1(n1(N,Wmb),0)){i=n1(N,Wmb);s1(i,Xmb)==0?D=D==null?'c0':D+','+'c0':s1(i,Ymb)==0?D=D==null?'c+':D+','+'c+':s1(i,Zmb)==0&&(D=D==null?'c-':D+','+'c-');}if(I1(n1(N,$mb),0)){L=n1(N,$mb);s1(L,98304)==0?D=D==null?'pi0':D+','+'pi0':s1(L,81920)==0?D=D==null?'pi1':D+','+'pi1':s1(L,49152)==0?D=D==null?'pi2':D+','+'pi2':s1(L,Pmb)==0&&(D=D==null?'pi>0':D+','+'pi>0');}if(I1(n1(N,_mb),0)){K=n1(N,_mb);s1(K,3801088)==0?D=D==null?'n1':D+','+'n1':s1(K,anb)==0?D=D==null?'n2':D+','+'n2':s1(K,bnb)==0?D=D==null?'n3':D+','+'n3':s1(K,3145728)==0?D=D==null?'n<3':D+','+'n<3':s1(K,cnb)==0?D=D==null?'n<4':D+','+'n<4':s1(K,dnb)==0?D=D==null?'n>1':D+','+'n>1':s1(K,enb)==0?D=D==null?'n>2':D+','+'n>2':s1(K,fnb)==0&&(D=D==null?'n>3':D+','+'n>3');}if(I1(n1(N,gnb),0)){o=n1(N,gnb);s1(o,hnb)==0?D=D==null?'e0':D+','+'e0':s1(o,inb)==0?D=D==null?'e1':D+','+'e1':s1(o,jnb)==0?D=D==null?'e2':D+','+'e2':s1(o,knb)==0?D=D==null?'e3':D+','+'e3':s1(o,15393162788864)==0?D=D==null?'e<2':D+','+'e<2':s1(o,13194139533312)==0?D=D==null?'e<3':D+','+'e<3':s1(o,lnb)==0?D=D==null?'e<4':D+','+'e<4':s1(o,mnb)==0?D=D==null?'e>0':D+','+'e>0':s1(o,nnb)==0?D=D==null?'e>1':D+','+'e>1':s1(o,onb)==0?D=D==null?'e>2':D+','+'e>2':s1(o,pnb)==0?D=D==null?'e>3':D+','+'e>3':s1(o,4947802324992)==0?D=D==null?'e1-2':D+','+'e1-2':s1(o,9345848836096)==0?D=D==null?'e1-3':D+','+'e1-3':s1(o,10445360463872)==0&&(D=D==null?'e2-3':D+','+'e2-3');}if(I1(n1(N,120),0)){Q=n1(N,120);s1(Q,112)==0?D=D==null?'!r':D+','+'!r':s1(Q,8)==0?D=D==null?'r':D+','+'r':s1(Q,96)==0?D=D==null?'rb<3':D+','+'rb<3':s1(Q,104)==0?D=D==null?'rb2':D+','+'rb2':s1(Q,88)==0?D=D==null?'rb3':D+','+'rb3':s1(Q,56)==0&&(D=D==null?'rb4':D+','+'rb4');}I1(n1(N,qnb),0)&&(D=D==null?'r'+S1(M1(n1(N,qnb),22)):D+','+('r'+S1(M1(n1(N,qnb),22))));I1(n1(N,smb),0)&&(D=klb.Tc(D,klb.Vc(N)));I1(n1(N,rnb),0)&&(D=D==null?'f':D+','+'f');}klb.nk(a.K,b)!=0&&(D=klb.Tc(D,''+klb.nk(a.K,b)));T=0;if(klb.qk(a.K,b)!=0){switch(klb.qk(a.K,b)){case 16:M=M==null?'|':M+','+'|';break;case 32:T=1;break;case 48:T=2;}}l=null;if((a.F&64)==0){if(klb.Yk(a.K,b))l='?';else if(klb.ck(a.K,b)!=0){if(klb.ik(a.K,b)==0||c==null||c[klb.ik(a.K,b)][klb.hk(a.K,b)]>1){if(klb.xn(a.K,b)==2){switch(klb.ck(a.K,b)){case 2:l=klb.$k(a.K,b)?'p':'P';break;case 1:l=klb.$k(a.K,b)?'m':'M';break;default:l='*';}}else {switch(klb.ck(a.K,b)){case 1:l=klb.$k(a.K,b)?'r':'R';break;case 2:l=klb.$k(a.K,b)?'s':'S';break;default:l='*';}}}}}(a.F&768)!=0&&(l=klb.Tc(l,''+klb.Ut(a.K,b)));I=null;(a.F&16)!=0&&klb.mk(a.K,b)!=0&&(I=''+klb.mk(a.K,b));q=null;if(klb.co(a.K,b)!=-1){p=klb.fd(a,b);p!=-1&&(q=p==0?'abs':((p&255)==1?'&':'or')+(1+(p>>8)));}A=0;(a.F&snb)==0&&(a.K.K?I1(n1(klb.pk(a.K,b),tnb),0)&&(A=klb.Mn(a.K,b)):(klb.uk(a.K,b)!=6||klb.nk(a.K,b)!=0||!a.q[b]||klb.qk(a.K,b)!=0)&&(A=klb.Mn(a.K,b)));H=false;f=klb.fk(a.K,b);if(f!=null&&flb.j5(ilb.Xgb(f).substr(0,1),']')){D=klb.Tc((ilb.Sgb(1,ilb.Xgb(f).length+1),ilb.Xgb(f).substr(1)),D);f=null;H=true;}if(f!=null){A=0;}else if(klb.kk(a.K,b)!=null){e=I1(n1(klb.pk(a.K,b),1),0)?'[!':'[';f=e+klb.lk(a.K,b)+']';ilb.Xgb(f).length>5&&(f=e+klb.kk(a.K,b).length+']');I1(n1(klb.pk(a.K,b),tnb),0)&&(A=-1);}else if(I1(n1(klb.pk(a.K,b),1),0)){f='?';I1(n1(klb.pk(a.K,b),tnb),0)&&(A=-1);}else (klb.uk(a.K,b)!=6||M!=null||D!=null||A>0||!a.q[b])&&(f=klb.jk(a.K,b));G=0;!klb.ol(a.K,b)&I1(n1(klb.pk(a.K,b),Jmb),0)&&klb.Kd(a,-8);if(f!=null){G=(O=(V=Glb.q2(a.e,f),new Hlb.w2(V)).b,O);klb.xd(a,klb.Mh(a.Q,klb.rk(a.K,b)),klb.Nh(a.Q,klb.sk(a.K,b)),f,true);a.s[b]=true;}else klb.nd(a,b)&&klb.wd(a,klb.Mh(a.Q,klb.rk(a.K,b)),klb.Nh(a.Q,klb.sk(a.K,b)),b);if(M!=null){klb.Ps(a,(a.W*2+1)/3|0);X=klb.Mh(a.Q,klb.rk(a.K,b))+((G+(O=(V=Glb.q2(a.e,M),new Hlb.w2(V)).b,O))/2+1);Z=klb.Nh(a.Q,klb.sk(a.K,b))-((a.k*4-4)/8|0);klb.xd(a,X,Z,M,true);klb.Ps(a,a.W);}(a.F&2)!=0&&(D=''+b);if(D!=null){H?klb.Ps(a,(a.W*5+1)/6|0):klb.Ps(a,(a.W*2+1)/3|0);X=klb.Mh(a.Q,klb.rk(a.K,b))-(G+(O=(V=Glb.q2(a.e,D),new Hlb.w2(V)).b,O))/2;Z=klb.Nh(a.Q,klb.sk(a.K,b))-((a.k*4-4)/8|0);klb.xd(a,X,Z,D,true);klb.Ps(a,a.W);}if(l!=null){klb.Ps(a,(a.W*2+1)/3|0);X=klb.Mh(a.Q,klb.rk(a.K,b))-(G+(O=(V=Glb.q2(a.e,l),new Hlb.w2(V)).b,O))/2;Z=klb.Nh(a.Q,klb.sk(a.K,b))+((a.k*4+4)/8|0);S=a.B;a.K.O!=1&&(a.F&Qmb)==0&&klb.Kd(a,448);klb.xd(a,X,Z,l,false);klb.Kd(a,S);klb.Ps(a,a.W);}if(I!=null){klb.Ps(a,(a.W*2+1)/3|0);X=klb.Mh(a.Q,klb.rk(a.K,b))+((G+(O=(V=Glb.q2(a.e,I),new Hlb.w2(V)).b,O))/2+1);Z=klb.Nh(a.Q,klb.sk(a.K,b))+((a.k*4+4)/8|0);S=a.B;klb.Kd(a,klb.al(a.K,b)?384:448);klb.xd(a,X,Z,I,true);klb.Kd(a,S);klb.Ps(a,a.W);}if(q!=null){d=klb.Bd(a,b);klb.Ps(a,(a.W*2+1)/3|0);X=klb.Mh(a.Q,klb.rk(a.K,b))+0.7*a.k*$wnd.Math.sin(d);Z=klb.Nh(a.Q,klb.sk(a.K,b))+0.7*a.k*$wnd.Math.cos(d);S=a.B;!a.J&&a.K.O!=1&&klb.Kd(a,klb.ed(a,b));klb.xd(a,X,Z,q,false);klb.Kd(a,S);klb.Ps(a,a.W);}if(A==0&&T==0){a.B==-8&&klb.Kd(a,-9);return;}u=oW(elb.GX,mmb,6,4,15,1);for(C=0;C<klb.jn(a.K,b);C++){h=klb.yn(a.K,b,C);for(F=0;F<2;F++){if(klb.zk(a.K,F,h)==b){R=klb.yk(a.K,klb.zk(a.K,F,h),klb.zk(a.K,1-F,h));if(R<unb){u[0]-=R+Omb;u[3]+=R+Nmb;}else if(R<0){u[2]+=R+Omb;u[3]-=R;}else if(R<Omb){u[1]+=R;u[2]+=Omb-R;}else {u[0]+=R-Omb;u[1]+=Nmb-R;}}}}klb.xn(a.K,b)==0?klb.hl(a.K,b)?u[3]-=0.2:u[1]-=0.2:u[1]-=0.1;(M!=null||I!=null)&&(u[1]+=10);(D!=null||l!=null)&&(u[3]+=10);s='';if(A!=0){v=(P=(W=Glb.q2(a.e,'H'),new Hlb.w2(W)).b,P);t=0;r=a.k;if(A==-1){s='n';klb.Ps(a,(a.W*2+1)/3|0);t=(O=(V=Glb.q2(a.e,s),new Hlb.w2(V)).b,O);}else if(A>1){s=''+A;klb.Ps(a,(a.W*2+1)/3|0);t=(O=(V=Glb.q2(a.e,s),new Hlb.w2(V)).b,O);}if(u[1]<0.6||u[3]<0.6){k=klb.Nh(a.Q,klb.sk(a.K,b));if(u[1]<=u[3]){u[1]+=10;j=klb.Mh(a.Q,klb.rk(a.K,b))+(G+v)/2;}else {u[3]+=10;j=klb.Mh(a.Q,klb.rk(a.K,b))-(G+v)/2-t;}}else {j=klb.Mh(a.Q,klb.rk(a.K,b));if(u[0]<u[2]){u[0]+=10;k=klb.Nh(a.Q,klb.sk(a.K,b))-r;}else {u[2]+=10;k=klb.Nh(a.Q,klb.sk(a.K,b))+r;}}if(t>0){X=j+(v+t)/2;Z=k+((a.k*4+4)/8|0);klb.xd(a,X,Z,s,true);klb.Ps(a,a.W);}klb.xd(a,j,k,'H',true);}g=0;if(T!=0){J=50;m=0;for(B=0;B<4;B++){n=B>1?B-2:B+2;if(u[B]<J){g=B;J=u[B];m=u[n];}else if(u[B]==J){if(u[n]>m){g=B;m=u[n];}}}switch(g){case 0:j=klb.Mh(a.Q,klb.rk(a.K,b));k=klb.Nh(a.Q,klb.sk(a.K,b))-a.U-G/2;break;case 1:j=klb.Mh(a.Q,klb.rk(a.K,b))+a.U+G/2;k=klb.Nh(a.Q,klb.sk(a.K,b));break;case 2:j=klb.Mh(a.Q,klb.rk(a.K,b));k=klb.Nh(a.Q,klb.sk(a.K,b))+a.U+G/2;break;default:j=klb.Mh(a.Q,klb.rk(a.K,b))-a.U-G/2;k=klb.Nh(a.Q,klb.sk(a.K,b));}if(T==1){llb.wi(a.Z,new xlb.dG(j-a.U,k-a.U,2*a.U,2*a.U));a.J||llb.wi(a.T,new klb.Rd(j,k,klb.ld(a,b)?-3:a.p[b]));}else {switch(g){case 2:case 0:Y=2*a.U;$=0;j-=a.U;break;case 1:Y=0;$=2*a.U;k-=a.U;break;default:Y=0;$=2*a.U;k-=a.U;}llb.wi(a.Z,new xlb.dG(j-a.U,k-a.U,2*a.U,2*a.U));a.J||llb.wi(a.T,new klb.Rd(j,k,klb.ld(a,b)?-3:a.p[b]));llb.wi(a.Z,new xlb.dG(j+Y-a.U,k+$-a.U,2*a.U,2*a.U));a.J||llb.wi(a.T,new klb.Rd(j+Y,k+$,klb.ld(a,b)?-3:a.p[b]));}}a.B==-8&&klb.Kd(a,-9);};klb.ud=function ud(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w;u=new klb.Sd();c=new klb.Sd();g=new klb.Sd();q=new xlb.VF();p=new xlb.VF();e=klb.zk(a.K,0,b);f=klb.zk(a.K,1,b);klb.Ms(a,b,klb.Mh(a.Q,klb.rk(a.K,e)),klb.Nh(a.Q,klb.sk(a.K,e)),klb.Mh(a.Q,klb.rk(a.K,f)),klb.Nh(a.Q,klb.sk(a.K,f)));!klb.ol(a.K,e)&&!klb.ol(a.K,f)&&I1(n1(K1(klb.pk(a.K,e),klb.pk(a.K,f)),Jmb),0)&&klb.Kd(a,-8);if(!a.o[e]){u.a=klb.Mh(a.Q,klb.rk(a.K,e));u.c=klb.Nh(a.Q,klb.sk(a.K,e));}else {u.a=a.o[e].a;u.c=a.o[e].b;}if(!a.o[f]){u.b=klb.Mh(a.Q,klb.rk(a.K,f));u.d=klb.Nh(a.Q,klb.sk(a.K,f));}else {u.b=a.o[f].a;u.d=a.o[f].b;}if((klb.Jk(a.K,b)&vnb)!=0){klb.Gd(a,u)&&klb.Hs(a,u);klb.Kd(a,-9);return;}h=klb.Kk(a.K,b)==64?0:klb.Kk(a.K,b)==32?1:klb.Hk(a.K,b);switch(h){case 1:i=klb.Kk(a.K,b);if((a.F&128)!=0&&(i==257||i==129)){t=klb.zk(a.K,0,b);n=klb.ik(a.K,t);if(n!=0){m=klb.hk(a.K,t);l=0;for(d=0;d<a.K.f;d++)klb.ik(a.K,d)==n&&klb.hk(a.K,d)==m&&++l;l==1&&(i=1);}}switch(i){case 1:klb.Gd(a,u)&&klb.Zc(a,u,e,f);break;case 257:klb.Dd(a,u,e,f);break;case 129:v=u.b-u.a;w=u.d-u.c;if(klb.dl(a.K,klb.un(a.K,e,f))){j=-3;k=-3;}else {j=a.p[e];k=klb.ed(a,e);j==klb.ek(a.K,e)&&(j=k);}for(o=2;o<17;o+=2){c.a=u.a+o*v/17-o*w/128;c.c=u.c+o*w/17+o*v/128;c.b=u.a+o*v/17+o*w/128;c.d=u.c+o*w/17-o*v/128;if(klb.Gd(a,c)){klb.Kd(a,o<9?j:k);r=Amb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(c.a),(Jlb.m8(),Jlb.i8))))+Bmb+Cmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(c.c),Jlb.i8)))+Bmb+Dmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(c.b),Jlb.i8)))+Bmb+Emb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(c.d),Jlb.i8)))+Bmb+Fmb+a.d+Bmb+Gmb+a.j+Hmb;klb.Ss(a,r);klb.Kd(a,a.P);}}break;case 32:klb.Gd(a,u)&&klb._c(a,u,e,f);}break;case 0:case 2:if((a.s[e]||klb.nn(a.K,e)==2)&&(a.s[f]||klb.nn(a.K,f)==2)&&!klb.Co(a.K,b)&&h==2){if(!klb.Gd(a,u))break;klb.pd(a,u.b-u.a,u.d-u.c,q);v=q.a/2;w=q.b/2;c.a=u.a+v;c.c=u.c+w;c.b=u.b+v;c.d=u.d+w;g.a=u.a-v;g.c=u.c-w;g.b=u.b-v;g.d=u.d-w;klb.Kk(a.K,b)==386&&klb.Fd(c,g);klb.Zc(a,c,e,f);h==2?klb.Zc(a,g,e,f):klb.Xc(a,g,e,f);}else if((a.s[f]||klb.nn(a.K,f)==2)&&h==2){klb.qd(a,u,b,false);}else if((a.s[e]||klb.nn(a.K,e)==2)&&h==2){klb.qd(a,u,b,true);}else {s=klb.Zn(a.K,b);s==0&&(s=1);c.a=u.a;c.c=u.c;c.b=u.b;c.d=u.d;klb.pd(a,u.b-u.a,u.d-u.c,q);if(s>0){g.a=u.a+q.a;g.c=u.c+q.b;g.b=u.b+q.a;g.d=u.d+q.b;if(klb.od(a,e,f,1,p)||klb.xn(a.K,e)>1){g.a+=p.a+q.b;g.c+=p.b-q.a;}if(klb.od(a,f,e,-1,p)||klb.xn(a.K,f)>1){g.b+=p.a-q.b;g.d+=p.b+q.a;}}else {g.a=u.a-q.a;g.c=u.c-q.b;g.b=u.b-q.a;g.d=u.d-q.b;if(klb.od(a,e,f,-1,p)||klb.xn(a.K,e)>1){g.a+=p.a+q.b;g.c+=p.b-q.a;}if(klb.od(a,f,e,1,p)||klb.xn(a.K,f)>1){g.b+=p.a-q.b;g.d+=p.b+q.a;}}klb.Kk(a.K,b)==386&&klb.Fd(c,g);klb.Gd(a,c)&&klb.Zc(a,c,e,f);h==2?klb.Gd(a,g)&&klb.Zc(a,g,e,f):klb.Gd(a,g)&&klb.Xc(a,g,e,f);}break;case 3:if(klb.Gd(a,u)){klb.Zc(a,u,e,f);klb.pd(a,u.b-u.a,u.d-u.c,q);klb.$c(a,u,e,f,q.a,q.b,c);klb.$c(a,u,e,f,-q.a,-q.b,c);}break;case 4:if(klb.Gd(a,u)){klb.pd(a,u.b-u.a,u.d-u.c,q);klb.$c(a,u,e,f,1.5*q.a,1.5*q.b,c);klb.$c(a,u,e,f,0.5*q.a,0.5*q.b,c);klb.$c(a,u,e,f,-0.5*q.a,-0.5*q.b,c);klb.$c(a,u,e,f,-1.5*q.a,-1.5*q.b,c);}break;case 5:if(klb.Gd(a,u)){klb.Zc(a,u,e,f);klb.pd(a,u.b-u.a,u.d-u.c,q);klb.$c(a,u,e,f,2*q.a,2*q.b,c);klb.$c(a,u,e,f,q.a,q.b,c);klb.$c(a,u,e,f,-q.a,-q.b,c);klb.$c(a,u,e,f,-2*q.a,-2*q.b,c);}}a.B==-8&&klb.Kd(a,-9);};klb.vd=function vd(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r;o=false;for(d=0;d<a.K.g;d++){j=null;if(klb.cl(a.K,d)){l=klb.Bk(a.K,d);k=klb.Ak(a.K,d);j=l==k?'['+l+']':'['+l+':'+k+']';}else (klb.Jk(a.K,d)&wnb)!=0?j=(klb.Jk(a.K,d)&wnb)==cnb?'a':(klb.Jk(a.K,d)&384)==256?'r!a':'!a':(klb.Jk(a.K,d)&384)!=0&&(j=(klb.Jk(a.K,d)&384)==256?'r':'!r');n=(klb.Jk(a.K,d)&enb)>>17;n!=0&&(j=(j==null?'':j)+n);if(j!=null){b=klb.zk(a.K,0,d);c=klb.zk(a.K,1,d);if(!o){klb.Ps(a,(a.W*2+1)/3|0);o=true;}q=(klb.Mh(a.Q,klb.rk(a.K,b))+klb.Mh(a.Q,klb.rk(a.K,c)))/2;r=(klb.Nh(a.Q,klb.sk(a.K,b))+klb.Nh(a.Q,klb.sk(a.K,c)))/2;f=klb.Mh(a.Q,klb.rk(a.K,c))-klb.Mh(a.Q,klb.rk(a.K,b));g=klb.Nh(a.Q,klb.sk(a.K,c))-klb.Nh(a.Q,klb.sk(a.K,b));e=$wnd.Math.sqrt(f*f+g*g);i=(m=(p=Glb.q2(a.e,j),new Hlb.w2(p)).b,0.6*m);h=0.55*a.k;e!=0&&(f>0?klb.xd(a,q+i*g/e,r-h*f/e,j,true):klb.xd(a,q-i*g/e,r+h*f/e,j,true));}}o&&klb.Ps(a,a.W);};klb.wd=function wd(a,b,c,d){llb.wi(a.Z,new xlb.dG(b-a.U,c-a.U,2*a.U,2*a.U));a.J||llb.wi(a.T,new klb.Rd(b,c,klb.ld(a,d)?-3:a.p[d]));};klb.xd=function xd(a,b,c,d,e){var f,g,h,i,j;if(e){g=(f=(h=Glb.q2(a.e,d),new Hlb.w2(h)).b,f);i=g/2+(a.k/8|0);j=a.k/2|0;(d=='+'||d=='-')&&(j=j*2/3);llb.wi(a.Z,new xlb.dG(b-i,c-j,2*i,2*j));}a.J||klb.Js(a,d,b,c);};klb.yd=function yd(a){var b;b=a.a;a.a=a.b;a.b=b;b=a.c;a.c=a.d;a.d=b;};klb.zd=function zd(a,b,c){var d;d=b==0?Mmb+a[0]-a[a.length-1]:a[b]-a[b-1];c>-2.0943951023931953&&c<xnb?d-=2*$wnd.Math.cos(c+ynb):d-=0.5*$wnd.Math.cos(c+ynb);return d;};klb.Ad=function Ad(a){var b;b=new xlb.cG();if(a.a<=a.b){b.c=a.a;b.b=a.b-a.a;}else {b.c=a.b;b.b=a.a-a.b;}if(a.c<=a.d){b.d=a.c;b.a=a.d-a.c;}else {b.d=a.d;b.a=a.c-a.d;}return b;};klb.Bd=function Bd(a,b){var c,d,e,f,g,h,i;c=oW(elb.GX,mmb,6,klb.hn(a.K,b),15,1);for(e=0;e<klb.hn(a.K,b);e++)c[e]=klb.yk(a.K,b,klb.wn(a.K,b,e));ilb.rgb(c,ilb.Xgb(e2(llb.wcb.prototype.kc,llb.wcb,[])));f=klb.Cd(c,0);g=klb.zd(c,0,f);for(d=1;d<c.length;d++){h=klb.Cd(c,d);i=klb.zd(c,d,h);if(g<i){g=i;f=h;}}return f;};klb.Cd=function Cd(a,b){var c;if(b>0)return (a[b]+a[b-1])/2;c=Nmb+(a[0]+a[a.length-1])/2;return c>Nmb?c-Mmb:c;};klb.Dd=function Dd(a,b,c,d){var e,f,g,h;h=new klb.Sd();if(b.a==b.b&&b.c==b.d)return;h.a=b.a;h.c=b.c;h.b=b.b;h.d=b.d;g=klb.Ad(h);for(e=0;e<a.Z.a.length;e++){f=llb.Bi(a.Z,e);if(f.c>g.c+g.b||f.d>g.d+g.a||g.c>f.c+f.b||g.d>f.d+f.a)continue;if(klb.Ed(a,h.a,h.c,e)){if(klb.Ed(a,h.b,h.d,e))return;klb.Hd(a,h,0,e);klb.Dd(a,h,c,d);return;}if(klb.Ed(a,h.b,h.d,e)){klb.Hd(a,h,1,e);klb.Dd(a,h,c,d);return;}}klb.ad(a,h,c,d);};klb.Ed=function Ed(a,b,c,d){var e;if((a.F&1)!=0)return false;e=llb.Bi(a.Z,d);return b>e.c&&b<e.c+e.b&&c>e.d&&c<e.d+e.a;};klb.Fd=function Fd(a,b){var c;c=a.b;a.b=b.b;b.b=c;c=a.d;a.d=b.d;b.d=c;};klb.Gd=function Gd(a,b){var c,d,e,f,g,h;if(b.a==b.b&&b.c==b.d){for(g=new llb.Wbb(a.Z);g.a<g.c.a.length;){f=llb.Vbb(g);if(xlb.$F(f,b.a,b.c))return false;}return true;}h=klb.Ad(b);c=false;if(b.a>b.b){klb.yd(b);c=true;}for(d=0;d<a.Z.a.length;d++){f=llb.Bi(a.Z,d);if(f.c>h.c+h.b||f.d>h.d+h.a||h.c>f.c+f.b||h.d>f.d+f.a)continue;if(klb.Ed(a,b.a,b.c,d)){if(klb.Ed(a,b.b,b.d,d)){c&&klb.yd(b);return false;}klb.Hd(a,b,0,d);e=klb.Gd(a,b);c&&klb.yd(b);return e;}if(klb.Ed(a,b.b,b.d,d)){klb.Hd(a,b,1,d);e=klb.Gd(a,b);c&&klb.yd(b);return e;}}c&&klb.yd(b);return true;};klb.Hd=function Hd(a,b,c,d){var e,f,g,h,i,j,k,l,m,n,o;if(c==0){l=b.a;n=b.c;m=b.b;o=b.d;}else {l=b.b;n=b.d;m=b.a;o=b.c;}k=llb.Bi(a.Z,d);i=m>l?k.c+k.b:k.c;j=o>n?k.d+k.a:k.d;e=m-l;f=o-n;if($wnd.Math.abs(e)>$wnd.Math.abs(f)){if(n==o){g=i;h=n;}else {g=l+e*(j-n)/f;if(m>l==i>g){h=j;}else {g=i;h=n+f*(i-l)/e;}}}else {if(l==m){g=l;h=j;}else {h=n+f*(i-l)/e;if(o>n==j>h){g=i;}else {g=l+e*(j-n)/f;h=j;}}}if(c==0){b.a=g;b.c=h;}else {b.b=g;b.d=h;}};klb.Id=function Id(a){var b,c,d,e;if(a.K.q==0)return;klb.Jt(a.K,(a.F&256)!=0?63:(a.F&512)!=0?95:31);klb.Uc(a);c=klb.Ot(a.K);d=false;a.p=oW(elb.IX,omb,6,a.K.q,15,1);for(b=0;b<a.K.q;b++){a.p[b]=klb.ek(a.K,b);a.p[b]!=0&&(d=true);klb.ol(a.K,b)&&(a.p[b]=128);klb.Tk(a.K,b)&&(a.F&tnb)==0&&(a.p[b]=256);}klb.Kd(a,-10);klb.hd(a);klb.gd(a);klb.jd(a);klb.Sc(a);klb.Ps(a,a.W);klb.Ns(a,a.X);klb.Kd(a,a.P);klb.md(a);ilb.qgb(a.T.a,0);ilb.qgb(a.Z.a,0);if((a.F&1)!=0){klb.rd(a,c);klb.sd(a);klb.vd(a);}for(e=0;e<a.K.q;e++){if(klb.ld(a,e)){klb.Kd(a,-3);klb.td(a,e,c);klb.Kd(a,a.P);}else if(a.p[e]!=0){klb.Kd(a,a.p[e]);klb.td(a,e,c);klb.Kd(a,a.P);}else if(!d&&a.K.O!=1&&klb.uk(a.K,e)!=1&&klb.uk(a.K,e)!=6&&(a.F&Vmb)==0&&klb.kk(a.K,e)==null&&klb.uk(a.K,e)<klb.Pc.length){klb.Ld(a,klb.dd(a,klb.Pc[klb.uk(a.K,e)]));klb.td(a,e,c);klb.Kd(a,a.P);}else {klb.td(a,e,c);}}if((a.F&1)==0){klb.sd(a);klb.vd(a);klb.rd(a,c);}};klb.Jd=function Jd(a,b,c,d){var e;e=c/2;switch(d&znb){case znb:if(b){a.w.a=b.c+b.b/2;a.w.b=b.d+b.a-e;break;}case 0:a.w.a=a.v.c+a.v.b/2;a.w.b=a.v.d+a.v.a+e;!!b&&a.w.b>b.d+b.a-e&&(a.w.b=b.d+b.a-e);break;case Anb:if(b){a.w.a=b.c+b.b/2;a.w.b=b.d+e;break;}case Bnb:a.w.a=a.v.c+a.v.b/2;a.w.b=a.v.d-e;!!b&&a.w.b<b.d+e&&(a.w.b=b.d+e);}};klb.Kd=function Kd(a,b){if(a.J)return;if(b==-10){a.B=-999;b=a.P;}b!=-2&&b!=-7&&a.M!=0&&(b=-4);if(b==a.B)return;if(a.B==-8&&b!=-9)return;b==-8&&(a.N=a.B);b==-9&&(b=a.N);a.B=b;switch(b){case 0:klb.Os(a,a.D==0?Cnb:a.D);break;case-6:klb.Os(a,a.D);break;case-4:klb.Os(a,a.M);break;case-2:klb.Os(a,a.t);break;case-3:klb.Os(a,a.u);break;case-7:klb.Os(a,a.G);break;case-8:klb.Os(a,a.H);break;case-5:klb.Os(a,a.O);break;case 64:a.d='rgb(32,96,255)';break;case 128:a.d='rgb(255,0,0)';break;case 256:a.d='rgb(192,0,255)';break;case 192:a.d='rgb(0,255,0)';break;case 320:a.d='rgb(255,160,0)';break;case 384:a.d='rgb(0,128,0)';break;case 448:a.d='rgb(160,0,0)';break;case 1:a.d='rgb(128,128,128)';break;default:a.d='rgb(0,0,0)';}};klb.Ld=function Ld(a,b){if(a.M!=0){if(a.B!=-4){a.B=-4;klb.Os(a,a.M);}return;}a.B=-5;a.O=b;a.d='rgb('+((b&Dnb)>>16)+','+((b&qmb)>>8)+','+(b&255)+')';};klb.Md=function Md(a){var b,c,d,e,f;e=klb.Mh(a.Q,klb.rk(a.K,0));c=klb.Mh(a.Q,klb.rk(a.K,0));f=klb.Nh(a.Q,klb.sk(a.K,0));d=klb.Nh(a.Q,klb.sk(a.K,0));for(b=0;b<a.K.q;b++){e>klb.Mh(a.Q,klb.rk(a.K,b))&&(e=klb.Mh(a.Q,klb.rk(a.K,b)));c<klb.Mh(a.Q,klb.rk(a.K,b))&&(c=klb.Mh(a.Q,klb.rk(a.K,b)));f>klb.Nh(a.Q,klb.sk(a.K,b))&&(f=klb.Nh(a.Q,klb.sk(a.K,b)));d<klb.Nh(a.Q,klb.sk(a.K,b))&&(d=klb.Nh(a.Q,klb.sk(a.K,b)));}return new xlb.dG(e,f,c-e,d-f);};klb.Nd=function Nd(a,b){var c,d;if(a.K.q==0)return null;a.v=klb.Md(a);c=a.Q.c*klb.vk(a.K);d=new klb.Ph(a.v,b,c);if(d.c==1&&d.a==0&&d.b==0){d=null;}else {klb.Ih(d,a.Q);klb.Kh(d,a.v);}klb.Jd(a,b,c,emb);return d;};klb.Od=function Od(a){var b;b=a.L!=0?a.L:a.C!=0?a.C:-1;a.t=Alb.OT(b);a.u=Alb.NT(Enb,b);a.G=-24321;a.H=-6291392;};klb.Pd=function Pd(a,b){var c,d,e,f;if(a.K.q==0)return null;e=klb.Qs(a,b);klb.Jt(a.K,(a.F&256)!=0?63:(a.F&512)!=0?95:31);klb.md(a);ilb.qgb(a.T.a,0);ilb.qgb(a.Z.a,0);klb.Uc(a);klb.Ps(a,a.W);a.J=true;for(d=0;d<a.K.q;d++)klb.td(a,d,null);a.J=false;c=a.Q.c*klb.vk(a.K);klb.bd(a,c);klb.Jd(a,b,c,emb);if(xlb._F(b,a.v))return e;f=new klb.Ph(a.v,b,c);klb.Ih(f,a.Q);klb.Kh(f,a.v);klb.Jh(f,a.w);if(!e)return f;klb.Ih(f,e);return e;};klb.Qd=function Qd(a,b){klb.Rc(this);this.K=a;this.F=b;klb.kd(this);};b2(245,1,{});_.r=0;_.t=0;_.u=0;_.A=0;_.B=0;_.C=0;_.D=0;_.F=0;_.G=0;_.H=0;_.I=0;_.J=false;_.L=0;_.M=0;_.N=0;_.O=0;_.P=0;_.R=0;_.S=0;_.U=0;_.V=0;_.W=0;_.X=0;_.Y=0;elb.NX=D3(245);klb.Rd=function Rd(a,b,c){this.b=a;this.c=b;this.a=c;};b2(111,1,{111:1},klb.Rd);_.a=0;_.b=0;_.c=0;elb.LX=D3(111);klb.Sd=function Sd(){};b2(60,1,{},klb.Sd);_.a=0;_.b=0;_.c=0;_.d=0;elb.MX=D3(60);klb.Td=function Td(a){return Fnb+a.ob()+'"'+a.nb()+Gnb;};b2(154,1,{154:1});_.lb=function Ud(){var a;a=new flb.G5();flb.F5(a,Fnb+this.ob()+'"'+this.nb()+Gnb);return a.a;};_.d=false;elb.OX=D3(154);klb.Vd=function Vd(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N,O,P,Q,R,S,T,U,V,W,X;J=oW(elb.g1,Imb,6,b.i.a.length,16,1);A=oW(elb.IX,omb,6,a.f.f,15,1);for(T=0;T<b.i.a.length;T++){J[T]=true;for(q=llb.Bi(b.j,T),r=0,s=q.length;r<s;++r){t=q[r];if(!a.e[t]){J[T]=false;break;}}if(J[T])for(d=llb.Bi(b.i,T),g=0,j=d.length;g<j;++g){c=d[g];++A[c];}}I=oW(elb.g1,Imb,6,a.f.f,16,1);for(U=0;U<b.i.a.length;U++){V=llb.Bi(b.j,U).length;if(V==3||V==5||V==6||V==7){if(J[U]){for(e=llb.Bi(b.i,U),h=0,k=e.length;h<k;++h){c=e[h];I[c]=true;}Q=true;M=-1;N=0;for(f=llb.Bi(b.i,U),i=0,l=f.length;i<l;++i){c=f[i];if(V==6||A[c]>1){if(!klb.Zd(a,c,false)){Q=false;break;}}else {S=V==5?klb.Wd(a,c,false):klb.Xd(a,c,false);if(klb.Zd(a,c,false)){if(N<S){N=S;M=c;}}else {if(N==10){Q=false;break;}M=c;N=20;}}}if(Q){for(d=llb.Bi(b.i,U),g=0,j=d.length;g<j;++g){c=d[g];if(c==M){V==5?klb.Wd(a,c,true):klb.Xd(a,c,true);klb.ie(a,c);}else {klb.Zd(a,c,true);}}}}}}w=oW(elb.IX,omb,6,a.f.f,15,1);C=oW(elb.g1,Imb,6,a.f.f,16,1);for(p=0;p<a.f.g;p++){m=klb.zk(a.f,0,p);n=klb.zk(a.f,1,p);if(!I[m]&&!I[n]){if(a.e[p]){++w[m];++w[n];}if(klb.Kk(a.f,p)==32){C[m]=true;C[n]=true;}}}R=oW(elb.IX,omb,6,a.f.f,15,1);B=oW(elb.IX,omb,6,a.f.f,15,1);for(X=0;X<a.f.f;X++){if(w[X]==1){B[0]=X;v=0;D=0;while(v<=D){for(F=0;F<klb.xn(a.f,B[v]);F++){if(a.e[klb.yn(a.f,B[v],F)]){u=klb.wn(a.f,B[v],F);if((v==0||u!=B[v-1])&&w[u]!=0){B[++D]=u;if((w[u]&1)!=0){for(L=1;L<D;L+=2)R[B[L]]=-1;D=0;}break;}}}++v;}}}o=oW(elb.g1,Imb,6,a.f.f,16,1);for(W=0;W<a.f.f;W++){if(!o[W]&&w[W]!=0){B[0]=W;o[W]=true;v=0;D=0;while(v<=D){for(F=0;F<klb.xn(a.f,B[v]);F++){if(a.e[klb.yn(a.f,B[v],F)]){u=klb.wn(a.f,B[v],F);if(!o[u]){B[++D]=u;o[u]=true;}}}++v;}if((D&1)==0){for(G=0;G<=D;G++)R[B[G]]==0&&(R[B[G]]=klb.Yd(a,B[G],false));K=true;for(H=0;H<=D;H++){if(R[B[H]]<=0){if(!klb.Zd(a,B[H],false)){K=false;break;}}}if(K){P=0;O=-1;for(F=0;F<=D;F++){if(P<R[B[F]]){P=R[B[F]];O=B[F];}}if(P>0){klb.Yd(a,O,true);klb.ie(a,O);}}}}}};klb.Wd=function Wd(a,b,c){if(klb.uk(a.f,b)==7){if(klb.hn(a.f,b)==3)return 6;else if(klb.xn(a.f,b)==2)return 4;}else if(klb.uk(a.f,b)==8){return 10;}else if(klb.uk(a.f,b)==15||klb.uk(a.f,b)==33){if(klb.xn(a.f,b)==3)return 8;}else if(klb.uk(a.f,b)==16||klb.uk(a.f,b)==34||klb.uk(a.f,b)==52){if(klb.xn(a.f,b)==2)return 12;}else if(klb.uk(a.f,b)==6){c&&klb.Jl(a.f,b,-1);return klb.hn(a.f,b)!=klb.jn(a.f,b)?2:3;}return 0;};klb.Xd=function Xd(a,b,c){if(a.a){if(klb.hn(a.f,b)!=3)return 0;}else {if(klb.hn(a.f,b)>3)return 0;}if(klb.uk(a.f,b)==6){c&&klb.Jl(a.f,b,1);return 2;}if(klb.uk(a.f,b)==5){return 4;}return 0;};klb.Yd=function Yd(a,b,c){if(klb.dk(a.f,b)!=0)return 0;if(a.a){if(klb.uk(a.f,b)==5){if(klb.Sn(a.f,b)!=2)return 0;c&&klb.Jl(a.f,b,1);return 1;}if(klb.uk(a.f,b)==7){if(klb.Sn(a.f,b)!=2)return 0;c&&klb.Jl(a.f,b,-1);return klb._d(a,b)?6:3;}if(klb.uk(a.f,b)==8){if(klb.Sn(a.f,b)!=1)return 0;c&&klb.Jl(a.f,b,-1);return klb._d(a,b)?7:4;}if(klb.uk(a.f,b)==16){if(klb.Sn(a.f,b)!=1)return 0;c&&klb.Jl(a.f,b,-1);return klb._d(a,b)?5:2;}if(klb.uk(a.f,b)==34){if(klb.Sn(a.f,b)!=1)return 0;c&&klb.Jl(a.f,b,-1);return klb._d(a,b)?4:1;}}else {if(klb.uk(a.f,b)==5){if(klb.Sn(a.f,b)>2)return 0;c&&klb.Jl(a.f,b,1);return 1;}if(klb.uk(a.f,b)==7){if(klb.Sn(a.f,b)>2)return 0;c&&klb.Jl(a.f,b,-1);return klb._d(a,b)?5:3;}if(klb.uk(a.f,b)==8){if(klb.Sn(a.f,b)>1)return 0;c&&klb.Jl(a.f,b,-1);return klb._d(a,b)?7:4;}if(klb.uk(a.f,b)==16){if(klb.Sn(a.f,b)>1)return 0;c&&klb.Jl(a.f,b,-1);return klb._d(a,b)?5:2;}}return 0;};klb.Zd=function Zd(a,b,c){var d,e,f;d=klb.uk(a.f,b);if(d>=5&&d<=8||d==15||d==16||d==33||d==34||d==52){f=klb.Nn(a.f,b);if(f!=0)return true;e=klb.dk(a.f,b);if(d==5&&e>=0){c&&klb.Jl(a.f,b,e-1);return true;}if(d!=5&&e<=0){c&&klb.Jl(a.f,b,e+1);return true;}}return false;};klb.$d=function $d(a,b,c,d){var e;for(e=0;e<klb.xn(a.f,b);e++)if(d[klb.yn(a.f,b,e)]==1&&klb.wn(a.f,b,e)!=c)return e;return -1;};klb._d=function _d(a,b){var c;for(c=0;c<klb.xn(a.f,b);c++)if(klb.kl(a.f,klb.wn(a.f,b,c)))return true;return false;};klb.ae=function ae(a,b,c){var d;for(d=0;d<klb.xn(a.f,b);d++)if(c[klb.yn(a.f,b,d)]>1)return true;return false;};klb.be=function be(a,b){var c;for(c=0;c<klb.xn(a.f,b);c++)if(a.e[klb.yn(a.f,b,c)])return true;return false;};klb.ce=function ce(a,b,c){var d,e,f,g,h,i,j,k,l,m;klb.Jt(a.f,1);if(b!=null){a.e=b;}else {a.e=oW(elb.g1,Imb,6,a.f.g,16,1);for(e=0;e<a.f.g;e++){if(klb.Kk(a.f,e)==64){a.e[e]=true;klb.km(a.f,e,1);}}}a.g=0;a.d=oW(elb.g1,Imb,6,a.f.f,16,1);for(f=0;f<a.f.g;f++){if(a.e[f]){++a.c;for(h=0;h<2;h++){if(!a.d[klb.zk(a.f,h,f)]){a.d[klb.zk(a.f,h,f)]=true;++a.b;}}}}if(a.c==0)return true;a.a=false;klb.ke(a,c);a.f.K&&klb.ee(a);m=new klb.Gr(a.f,1);c&&klb.Vd(a,m);klb.le(a,m);klb.he(a,m);klb.je(a);klb.fe(a);while(klb.ge(a,m))klb.fe(a);while(a.c!=0){g=false;if(!g){for(k=0;k<m.i.a.length;k++){if(llb.Bi(m.j,k).length==6){j=true;l=llb.Bi(m.j,k);for(i=0;i<6;i++){if(!a.e[l[i]]){j=false;break;}}if(j){for(h=0;h<6;h+=2)klb.de(a,l[h]);g=true;break;}}}}if(!g){for(d=0;d<a.f.g;d++){if(a.e[d]){klb.de(a,d);klb.fe(a);g=true;break;}}}}return a.b==a.g;};klb.de=function de(a,b){var c,d,e,f;if(klb.Kk(a.f,b)==1){klb.km(a.f,b,2);a.g+=2;}for(e=0;e<2;e++){c=klb.zk(a.f,e,b);a.d[c]=false;for(f=0;f<klb.xn(a.f,c);f++){d=klb.yn(a.f,c,f);if(a.e[d]){a.e[d]=false;--a.c;}}}};klb.ee=function ee(a){var b,c,d,e,f,g,h,i;for(c=0;c<a.f.g;c++){if(a.e[c]){for(e=0;e<2;e++){h=klb.zk(a.f,e,c);b=false;for(g=0;g<klb.xn(a.f,h);g++){if(c!=klb.yn(a.f,h,g)&&a.e[klb.yn(a.f,h,g)]){b=true;break;}}if(!b){i=c;d=klb.zk(a.f,1-e,c);while(i!=-1){a.e[i]=false;--a.c;klb.km(a.f,i,64);i=-1;h=d;for(f=0;f<klb.xn(a.f,h);f++){if(a.e[klb.yn(a.f,h,f)]){if(i==-1){i=klb.yn(a.f,h,f);d=klb.wn(a.f,h,f);}else {h=-1;i=-1;break;}}}}break;}}}}};klb.fe=function fe(a){var b,c,d,e,f,g,h;do{h=false;for(c=0;c<a.f.g;c++){if(a.e[c]){f=false;for(e=0;e<2;e++){d=klb.zk(a.f,e,c);b=false;for(g=0;g<klb.xn(a.f,d);g++){if(c!=klb.yn(a.f,d,g)&&a.e[klb.yn(a.f,d,g)]){b=true;break;}}if(!b){f=true;break;}}if(f){h=true;klb.de(a,c);}}}}while(h);};klb.ge=function ge(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o;o=oW(elb.IX,omb,6,a.f.g,15,1);for(m=0;m<b.i.a.length;m++){n=llb.Bi(b.j,m);l=true;for(k=0;k<n.length;k++){if(!a.e[n[k]]){l=false;break;}}if(l)for(j=0;j<n.length;j++)++o[n[j]];}i=a.c;for(f=0;f<a.f.g;f++){if(o[f]==1){for(j=0;j<2&&a.e[f];j++){c=klb.zk(a.f,j,f);d=klb.zk(a.f,1-j,f);if(klb.ae(a,c,o)&&!klb.ae(a,d,o)){while(-1!=(h=klb.$d(a,d,c,o))){e=klb.wn(a.f,d,h);g=klb.yn(a.f,d,h);if(!a.e[g])break;klb.de(a,g);h=klb.$d(a,e,d,o);if(h==-1)break;c=e;d=klb.wn(a.f,e,h);}}}}}return i!=a.c;};klb.he=function he(a,b){var c;for(c=0;c<a.f.g;c++){if(a.e[c]&&klb.zr(b,c)){klb.ie(a,klb.zk(a.f,0,c));klb.ie(a,klb.zk(a.f,1,c));}}};klb.ie=function ie(a,b){var c,d;if(a.d[b]){a.d[b]=false;--a.b;}for(d=0;d<klb.xn(a.f,b);d++){c=klb.yn(a.f,b,d);if(a.e[c]){a.e[c]=false;--a.c;}}};klb.je=function je(a){var b,c,d,e,f;for(c=0;c<a.f.g;c++){if(klb.Hk(a.f,c)==2){for(e=0;e<2;e++){b=klb.zk(a.f,e,c);if(klb.uk(a.f,b)<=8){for(f=0;f<klb.xn(a.f,b);f++){d=klb.yn(a.f,b,f);if(a.e[d]){klb.ie(a,b);break;}}}}}}};klb.ke=function ke(a,b){var c;for(c=0;c<a.f.f;c++)a.d[c]&&klb.Nn(a.f,c)==0&&(!b||klb.uk(a.f,c)==5&&klb.dk(a.f,c)<0||klb.uk(a.f,c)==6||klb.uk(a.f,c)==14||klb.hl(a.f,c)&&klb.dk(a.f,c)>0)&&klb.ie(a,c);};klb.le=function le(a,b){var c,d,e,f,g,h,i,j,k,l,m,n;for(k=0;k<b.i.a.length;k++){n=llb.Bi(b.j,k).length;if(n==3||n==5||n==7){l=llb.Bi(b.i,k);for(d=0;d<n;d++){c=l[d];klb.be(a,c)&&(n==5?(klb.uk(a.f,c)==6&&klb.dk(a.f,c)==-1&&klb.hn(a.f,c)==3||klb.uk(a.f,c)==7&&klb.dk(a.f,c)==0&&klb.hn(a.f,c)==3||klb.uk(a.f,c)==8&&klb.dk(a.f,c)==0&&klb.xn(a.f,c)==2||klb.uk(a.f,c)==16&&klb.dk(a.f,c)==0&&klb.xn(a.f,c)==2||klb.uk(a.f,c)==34&&klb.dk(a.f,c)==0&&klb.xn(a.f,c)==2)&&klb.ie(a,c):(klb.uk(a.f,c)==5&&klb.dk(a.f,c)==0&&klb.hn(a.f,c)==3||klb.uk(a.f,c)==6&&klb.dk(a.f,c)==1)&&klb.ie(a,c));}}}for(j=0;j<b.i.a.length;j++){if(llb.Bi(b.j,j).length==5){m=llb.Bi(b.j,j);f=true;for(e=0;e<m.length;e++){if(!a.e[m[e]]){f=false;break;}}if(f){l=llb.Bi(b.i,j);h=0;g=-1;for(d=0;d<m.length;d++){if(klb.dk(a.f,l[d])==-1&&klb.uk(a.f,l[d])==6){i=klb.hn(a.f,l[d])==3?3:klb.jn(a.f,l[d])==3?2:1;if(h<i){h=i;g=l[d];}}}g!=-1&&klb.ie(a,g);}}}};klb.me=function me(a){this.f=a;};b2(76,1,{},klb.me);_.a=false;_.b=0;_.c=0;_.g=0;elb.PX=D3(76);klb.ne=function ne(a,b){var c,d;c=0;for(d=0;d<a.j[b];d++)a.n[b][d]==2&&(klb.uk(a,a.i[b][d])==7||klb.uk(a,a.i[b][d])==8||klb.uk(a,a.i[b][d])==16)&&++c;return c;};klb.oe=function oe(a,b){var c,d,e,f,g,h;if(a.s[b]==0){return false;}h=true;c=a.s[b];f=a.j[b];g=0;for(d=0;d<f;d++){e=a.i[b][d];g+=a.s[e];}$wnd.Math.abs(c)<=$wnd.Math.abs(g)&&flb.T4(c)!=flb.T4(g)&&(h=false);return h;};klb.pe=function pe(a,b){var c,d,e,f,g,h,i;if(a.C[b]!=8||a.s[b]!=0||a.j[b]!=1||a.n[b][0]!=1)return false;c=a.i[b][0];if(a.C[c]==6){g=a.j[c];for(d=0;d<g;d++){e=a.i[c][d];if(e==b){continue;}if(a.C[e]!=8){continue;}f=klb.un(a,c,e);if(a.H[f]==2)return true;}}else if(a.C[c]==7){if(a.s[c]==1)return true;}else if(a.C[c]==16){h=a.j[c];i=0;for(d=0;d<h;d++){e=a.i[c][d];if(e==b)continue;if(a.C[e]!=8)continue;f=klb.un(a,c,e);a.H[f]==2&&++i;}if(i==2)return true;}else if(klb.qe(a,b))return true;return false;};klb.qe=function qe(a,b){var c,d,e,f,g;if(a.C[b]!=8)return false;if(a.j[b]!=1)return false;c=a.i[b][0];if(a.C[c]==15){g=a.j[c];for(d=0;d<g;d++){e=a.i[c][d];if(e==b)continue;if(a.C[e]!=8)continue;f=klb.un(a,c,e);if(a.H[f]==2)return true;}}return false;};klb.re=function re(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A;if(a.C[b]!=7||a.s[b]!=0||a.j[b]+a.o[b]>3)return false;if(b<a.f&&klb.sr(a.p,b)){if(a.o[b]!=1)return false;if(klb.qn(a,b,7)!=1)return false;u=(klb.Jt(a,7),a.p);for(s=0;s<u.i.a.length;s++){if(klb.ur(u,s,b)){if(llb.Bi(u.j,s).length==5||llb.Bi(u.j,s).length==6){v=llb.Bi(u.i,s);q=-1;for(i=0;i<v.length;i++){if(v[i]==b){q=i;break;}}e=0;r=null;p=null;if(v.length==5){r=oW(elb.IX,omb,6,2,15,1);r[0]=v[q-1<0?q+4:q-1];r[1]=v[q-4<0?q+1:q-4];p=oW(elb.IX,omb,6,2,15,1);p[0]=v[q-2<0?q+3:q-2];p[1]=v[q-3<0?q+2:q-3];}if(v.length==6){r=oW(elb.IX,omb,6,3,15,1);r[0]=v[q-1<0?q+5:q-1];r[1]=v[q-3<0?q+3:q-3];r[2]=v[q-5<0?q+1:q-5];p=oW(elb.IX,omb,6,2,15,1);p[0]=v[q-2<0?q+4:q-2];p[1]=v[q-4<0?q+2:q-4];}for(j=0;j<v.length;j++)b!=v[j]&&klb.uk(a,v[j])==7&&klb.nn(a,v[j])==1&&--e;for(k=0;k<r.length;k++){f=-1;g=-1;for(o=0;o<klb.xn(a,r[k]);o++){if(!klb.po(a,klb.yn(a,r[k],o))){f=klb.wn(a,r[k],o);g=klb.yn(a,r[k],o);break;}}if(f!=-1){if(a.C[f]==7&&a.o[f]==0&&a.j[f]+a.o[f]<=3&&!klb.se(a,f,false)){++e;continue;}if(a.C[f]==8&&a.j[f]==1){e+=2;continue;}if(g<a.g&&klb.tr(a.p,g)){for(w=0;w<u.i.a.length;w++){if(u.d[w]&&klb.ur(u,w,f)){t=llb.Bi(u.i,w);for(n=0;n<t.length;n++){if(klb.uk(a,t[n])==7&&klb.nn(a,t[n])==1){--e;break;}}break;}}}}}for(l=0;l<p.length;l++){f=-1;for(n=0;n<klb.xn(a,p[l]);n++)klb.po(a,klb.yn(a,p[l],n))||(f=klb.wn(a,p[l],n));klb.uk(a,p[l])==6?f!=-1&&klb.ne(a,f)!=0&&--e:klb.uk(a,p[l])==7&&klb.nn(a,p[l])==0&&(f==-1||!(f<a.f&&klb.sr(a.p,f))&&klb.ne(a,f)==0)&&++e;}return e>0;}break;}}return false;}if(a.o[b]>1)return false;if(a.o[b]==1){m=-1;A=0;for(i=0;i<a.j[b];i++){d=a.i[b][i];if(a.n[b][i]==2){if(a.C[d]!=6)return false;m=d;continue;}if(a.C[d]==8)return false;if(a.C[d]==7){--A;klb.se(a,d,false)&&--A;continue;}d<a.f&&klb.sr(a.p,d)&&--A;}if(m==-1)return false;c=0;for(j=0;j<a.j[m];j++){if(a.n[m][j]==1){d=a.i[m][j];if(klb.ne(a,d)!=0)return false;d<a.f&&klb.sr(a.p,d)&&++c;a.C[d]==7&&!klb.se(a,d,true)&&++A;(a.C[d]==8||a.C[d]==16)&&--A;}}c==2&&--A;return A>=0;}for(h=0;h<a.j[b];h++){d=a.i[b][h];if(d<a.f&&klb.sr(a.p,d))return false;if(a.C[d]!=6)return false;if(klb.ne(a,d)!=0)return false;if(a.o[d]!=0&&klb.te(a,d))return false;}return true;};klb.se=function se(a,b,c){var d,e,f;d=false;for(f=0;f<a.j[b];f++){if(!klb.po(a,a.k[b][f])&&a.n[b][f]==1){e=a.i[b][f];if(!(e<a.f&&klb.sr(a.p,e))&&(a.C[e]==6&&klb.ne(a,e)==1||a.C[e]==16&&klb.ne(a,e)==2)){if(d||!c)return true;d=true;}}}return false;};klb.te=function te(a,b){var c,d,e;for(d=0;d<a.j[b];d++){if(a.n[b][d]!=1){c=a.i[b][d];for(e=0;e<a.j[c];e++)if(a.n[c][e]==1&&klb.ne(a,a.i[c][e])!=0)return true;}}return false;};klb.we=function we(){klb.we=d2;klb.ue=wW(iW(elb.f1,1),Hnb,6,15,[-1,-1,-1,0,0,1,2,3,4,5,-1,0,0,0,6,7,8,9,-1,0,0,10,10,10,10,10,10,10,10,10,10,1,11,11,12,13,-1,0,0,10,10,10,10,10,10,10,10,10,10,0,0,0,11,14,-1,0,0,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,10,10,10,10,10,10,10,10,1,1,1,1,-1,-1,-1,-1,15,15,15,15,15,15,15,15,15,15,15,15,15,15,15,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1]);klb.ve=wW(iW(elb.f1,1),Hnb,6,15,[-1,-1,-1,0,0,0,2,5,5,5,-1,0,0,0,0,9,9,9,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,9,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,9,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,-1,-1,-1,-1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1]);};klb.xe=function xe(a,b,c){klb.we();var d,e,f,g,h,i,j,k,l,m,n,o,p,q;klb.Jt(a,7);n=oW(elb.JX,Inb,6,a.j[b],14,1);m=0;for(i=0;i<a.j[b];i++){e=a.i[b][i];if(a.C[e]==1)continue;g=0;if((c&32)!=0){f=z1(a.n[b][i]);c==32190?s1(f,3)<0&&klb.to(a,a.k[b][i])&&a.o[b]==1&&(f=0):s1(f,3)<0&&klb.po(a,a.k[b][i])&&(f=0);g=K1(g,L1(f,4));}if((c&128)!=0){if(klb.ue[a.C[e]]==-1)throw l1(new flb.uz(Jnb+a.C[e]));g=m1(g,z1(klb.ue[a.C[e]]));}else if((c&64)!=0){if(klb.ve[a.C[e]]==-1)throw l1(new flb.uz(Jnb+a.C[e]));g=m1(g,z1(klb.ve[a.C[e]]));}if((c&256)!=0){o=a.j[e]-1;o>3&&(o=3);(c&512)==0&&o>1&&(o=1);g=K1(g,elb.v1(o<<6));}(c&Vmb)!=0&&(a.u[e]&8)!=0&&(g=K1(g,256));(c&tnb)!=0&&e<a.f&&klb.sr(a.p,e)&&(g=K1(g,512));(c&Pmb)!=0&&(a.u[e]&snb)!=0&&(g=K1(g,Vmb));k=0;while(s1(g,n[k])<0)++k;for(l=i;l>k;l--)n[l]=n[l-1];n[k]=g;++m;}m>4&&(m=4);d=0;for(j=0;j<m;j++){d=L1(d,11);d=K1(d,n[j]);}d=L1(d,15);if(klb.ue[a.C[b]]==-1)throw l1(new flb.uz(Jnb+a.C[b]));d=K1(d,z1(klb.ue[a.C[b]]));if((c&2)!=0){q=!!a.p&&b<a.f?klb.hr(a.p,b):0;q>9&&(q=9);q>2&&(q-=2);d=K1(d,elb.v1(q<<4));}else (c&1)!=0&&(a.u[b]&8)!=0&&(d=K1(d,64));(c&4)!=0&&b<a.f&&klb.sr(a.p,b)&&(d=K1(d,Vmb));(c&8)!=0&&(a.u[b]&Qmb)!=0&&(d=K1(d,tnb));(c&16)!=0&&(a.u[b]&snb)!=0&&(d=K1(d,Qmb));if((c&Qmb)!=0){klb.oe(a,b)&&(d=K1(d,snb));if(klb.re(a,b)){for(h=0;h<a.f;h++){if(klb.pe(a,h)){d=K1(d,Pmb);break;}}}}if((c&snb)!=0){p=z1(klb.qn(a,b,10));d=K1(d,L1(p,7));}return d;};klb.ye=function ye(a,b,c,d,e,f){var g,h,i,j;i=0;for(h=0;h<a.P.f;h++)I1(n1(klb.pk(a.P,a.w[h]),d),0)&&++i;if(i==0)return;klb.wf(a,b);klb.rf(a,elb.v1(i),c);for(g=0;g<a.P.f;g++){j=n1(klb.pk(a.P,a.w[g]),d);if(s1(j,0)!=0){klb.rf(a,elb.v1(g),c);e!=1&&klb.rf(a,M1(j,f),e);}}};klb.ze=function ze(a,b,c,d,e,f){var g,h,i,j;i=0;for(h=0;h<a.P.g;h++)(klb.Jk(a.P,a.A[h])&d)!=0&&++i;if(i==0)return;klb.wf(a,b);klb.rf(a,elb.v1(i),c);for(g=0;g<a.P.g;g++){j=klb.Jk(a.P,a.A[g])&d;if(j!=0){klb.rf(a,elb.v1(g),c);e!=1&&klb.rf(a,elb.v1(j>>f),e);}}};klb.Ae=function Ae(a){var b,c,d,e,f,g;while(true){f=oW(elb.g1,Imb,6,a.S+1,16,1);d=-1;for(c=0;c<a.P.f;c++){if(klb.Nn(a.P,c)!=0){f[a.d[c]]&&d<a.d[c]&&(d=a.d[c]);f[a.d[c]]=true;}}if(d==-1)break;e=0;for(b=0;b<a.P.f;b++){g=0;a.d[b]==d&&(g=++e);klb.sg(a.c[b],b);klb.pg(a.c[b],a.b,z1(a.d[b]));klb.pg(a.c[b],8,elb.v1(g));}a.S=klb._e(a);}};klb.Be=function Be(a){var b,c,d,e;for(c=0;c<a.P.f;c++){klb.sg(a.c[c],c);klb.pg(a.c[c],2*a.b+4,L1(z1(a.d[c]),a.b+4));}e=false;for(b=0;b<a.P.f;b++)e=e|klb.Oe(a,b,3);for(d=0;d<a.P.g;d++)e=e|klb.Je(a,d,3);if(!e)return false;while(a.S<a.P.f){e=klb.Ye(a);if(!e)break;klb.$e(a);!!a.N&&klb.Ig(a.N,a.d);}return true;};klb.Ce=function Ce(a){var b,c,d,e,f;for(c=0;c<a.P.f;c++){klb.sg(a.c[c],c);klb.pg(a.c[c],a.b+1,z1(2*a.d[c]));}f=oW(elb.IX,omb,6,a.S+1,15,1);for(d=0;d<a.P.f;d++)++f[a.d[d]];e=1;while(f[e]==1)++e;for(b=0;b<a.P.f;b++){if(a.d[b]==e){klb.qg(a.c[b],1);break;}}a.S=klb._e(a);};klb.De=function De(a,b,c){var d,e,f,g,h;if(klb.uk(a.P,b)!=6&&klb.uk(a.P,b)!=7)return false;e=klb.wn(a.P,b,0);f=klb.wn(a.P,b,1);if(klb.nn(a.P,e)!=1||klb.nn(a.P,f)!=1)return false;if(klb.xn(a.P,e)==1||klb.xn(a.P,f)==1)return false;if(klb.hn(a.P,e)>3||klb.hn(a.P,f)>3)return false;g=new klb.oj(a.P,a.d,b,e);if(g.f&&c==1)return false;h=new klb.oj(a.P,a.d,b,f);if(h.f&&c==1)return false;if(g.f&&h.f)return false;if(c==3){g.f&&g.c&&(a.U[b]=true);h.f&&h.c&&(a.U[b]=true);}d=a.hb?klb.Fe(a,g,h):klb.Ee(g,h);if(c==1){a.bb[b]=d;}else if(c==2){g.f&&(d==1?klb.qg(a.c[g.b],z1(a.d[e])):klb.qg(a.c[g.d],z1(a.d[e])));h.f&&(d==2?klb.qg(a.c[h.b],z1(a.d[f])):klb.qg(a.c[h.d],z1(a.d[f])));}return true;};klb.Ee=function Ee(a,b){var c,d,e;d=klb.nj(a);e=klb.nj(b);if(d==-1||e==-1||(d+e&1)==0)return 3;c=0;switch(d+e){case 3:case 7:c=2;break;case 5:c=1;}return c;};klb.Fe=function Fe(a,b,c){var d,e;d=oW(elb.IX,omb,6,4,15,1);d[0]=b.b;d[1]=b.a;d[2]=c.a;d[3]=c.b;e=klb.Jj(a.P,d);if($wnd.Math.abs(e)<0.3||$wnd.Math.abs(e)>Knb)return 3;return e<0?2:1;};klb.Ge=function Ge(a,b,c){var d,e,f,g,h;if(!klb.qo(a.P,b))return false;d=klb.zk(a.P,0,b);e=klb.zk(a.P,1,b);g=new klb.oj(a.P,a.d,d,e);if(g.f&&c==1)return false;h=new klb.oj(a.P,a.d,e,d);if(h.f&&c==1)return false;if(g.f&&h.f)return false;if(c==3){g.f&&(a.T[b]=klb.Mf(a,e));h.f&&(a.T[b]=klb.Mf(a,d));}f=a.hb?klb.Ie(a,g,h):klb.He(g,h);if(c==1){a.k[b]=f;}else if(c==2){g.f&&(f==2?klb.qg(a.c[g.b],z1(a.d[e])):klb.qg(a.c[g.d],z1(a.d[e])));h.f&&(f==2?klb.qg(a.c[h.b],z1(a.d[d])):klb.qg(a.c[h.d],z1(a.d[d])));}return true;};klb.He=function He(a,b){var c,d,e;d=klb.nj(a);e=klb.nj(b);if(d==-1||e==-1||(d+e&1)==0)return 3;c=0;switch(d+e){case 3:case 7:c=1;break;case 5:c=2;}return c;};klb.Ie=function Ie(a,b,c){var d,e;d=oW(elb.IX,omb,6,4,15,1);d[0]=b.b;d[1]=b.a;d[2]=c.a;d[3]=c.b;e=klb.Jj(a.P,d);if($wnd.Math.abs(e)<0.3||$wnd.Math.abs(e)>Knb)return 3;return e<0?1:2;};klb.Je=function Je(a,b,c){var d,e,f,g,h;if(a.k[b]!=0)return false;if(klb.Hk(a.P,b)==1)return klb.Ge(a,b,c);if(klb.Hk(a.P,b)!=2)return false;if(klb.po(a.P,b))return false;e=klb.zk(a.P,0,b);f=klb.zk(a.P,1,b);if(klb.xn(a.P,e)==1||klb.xn(a.P,f)==1)return false;if(klb.xn(a.P,e)>3||klb.xn(a.P,f)>3)return false;if(klb.nn(a.P,e)==2||klb.nn(a.P,f)==2)return false;g=new klb.oj(a.P,a.d,f,e);if(g.f&&c==1)return false;h=new klb.oj(a.P,a.d,e,f);if(h.f&&c==1)return false;if(g.f&&h.f)return false;if(c==3){g.f&&g.c&&(a.T[b]=true);h.f&&h.c&&(a.T[b]=true);}d=klb.gl(a.P,b)?3:a.hb?klb.Le(a,g,h):klb.Ke(g,h);if(c==1){a.k[b]=d;}else if(c==2){g.f&&(d==1?klb.qg(a.c[g.b],z1(a.d[e])):d==2&&klb.qg(a.c[g.d],z1(a.d[e])));h.f&&(d==1?klb.qg(a.c[h.b],z1(a.d[f])):d==2&&klb.qg(a.c[h.d],z1(a.d[f])));}return true;};klb.Ke=function Ke(a,b){if(klb.nj(a)==-1||klb.nj(b)==-1)return 3;if(((klb.nj(a)|klb.nj(b))&1)!=0)return 3;return klb.nj(a)==klb.nj(b)?1:2;};klb.Le=function Le(a,b,c){var d,e,f,g,h,i,j;f=oW(elb.GX,mmb,6,3,15,1);f[0]=klb.rk(a.P,c.a)-klb.rk(a.P,b.a);f[1]=klb.sk(a.P,c.a)-klb.sk(a.P,b.a);f[2]=klb.tk(a.P,c.a)-klb.tk(a.P,b.a);i=oW(elb.GX,mmb,6,3,15,1);i[0]=klb.rk(a.P,b.b)-klb.rk(a.P,b.a);i[1]=klb.sk(a.P,b.b)-klb.sk(a.P,b.a);i[2]=klb.tk(a.P,b.b)-klb.tk(a.P,b.a);j=oW(elb.GX,mmb,6,3,15,1);j[0]=klb.rk(a.P,c.b)-klb.rk(a.P,c.a);j[1]=klb.sk(a.P,c.b)-klb.sk(a.P,c.a);j[2]=klb.tk(a.P,c.b)-klb.tk(a.P,c.a);g=oW(elb.GX,mmb,6,3,15,1);g[0]=f[1]*i[2]-f[2]*i[1];g[1]=f[2]*i[0]-f[0]*i[2];g[2]=f[0]*i[1]-f[1]*i[0];h=oW(elb.GX,mmb,6,3,15,1);h[0]=f[1]*g[2]-f[2]*g[1];h[1]=f[2]*g[0]-f[0]*g[2];h[2]=f[0]*g[1]-f[1]*g[0];d=(i[0]*h[0]+i[1]*h[1]+i[2]*h[2])/($wnd.Math.sqrt(i[0]*i[0]+i[1]*i[1]+i[2]*i[2])*$wnd.Math.sqrt(h[0]*h[0]+h[1]*h[1]+h[2]*h[2]));e=(j[0]*h[0]+j[1]*h[1]+j[2]*h[2])/($wnd.Math.sqrt(j[0]*j[0]+j[1]*j[1]+j[2]*j[2])*$wnd.Math.sqrt(h[0]*h[0]+h[1]*h[1]+h[2]*h[2]));return d<0^e<0?1:2;};klb.Me=function Me(a,b){var c,d,e,f;c=klb.bk(a.P,b);d=klb.Ln(a.P,b,false);e=klb.Ln(a.P,b,true);f=-1;if(d!=e){c!=-1&&c>d?f=c<<24>>24:f=d<<24>>24;}else if(c!=-1){(c>e||c<e&&c>=klb.Sn(a.P,b))&&(f=c<<24>>24);}else if(!klb.Wo(a.P,b)&&klb.Bn(a.P,b)!=0){f=klb.Sn(a.P,b);f-=klb.Ok(a.P,b,f);}klb.gf(a,b,f);return f;};klb.Ne=function Ne(a){var b,c,d,e,f,g,h,i,j,k,l;d=oW(elb.IX,omb,6,a.M,15,1);for(b=0;b<a.P.f;b++){k=klb.xn(a.P,b)+klb.Pn(a.P,b);j=0;for(f=0;f<klb.jn(a.P,b);f++){if(f<klb.xn(a.P,b)||f>=klb.hn(a.P,b)){l=2*a.d[klb.wn(a.P,b,f)];c=klb.yn(a.P,b,f);klb.Hk(a.P,c)==2&&(klb.po(a.P,c)||++l);for(h=0;h<j;h++)if(l<d[h])break;for(i=j;i>h;i--)d[i]=d[i-1];d[h]=l;++j;}}klb.sg(a.c[b],b);klb.pg(a.c[b],a.b,z1(a.d[b]));for(g=k;g<a.M;g++)klb.pg(a.c[b],a.b+1,0);for(e=0;e<k;e++)klb.pg(a.c[b],a.b+1,elb.v1(d[e]));}};klb.Oe=function Oe(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o;if(a.bb[b]!=0)return false;if(klb.uk(a.P,b)!=5&&klb.uk(a.P,b)!=6&&klb.uk(a.P,b)!=7&&klb.uk(a.P,b)!=14&&klb.uk(a.P,b)!=15&&klb.uk(a.P,b)!=16)return false;if(klb.nn(a.P,b)!=0){if(klb.ro(a.P,b))return klb.De(a,b,c);if(klb.uk(a.P,b)!=15&&klb.uk(a.P,b)!=16)return false;}if(klb.xn(a.P,b)<3||klb.hn(a.P,b)>4)return false;if(klb.dk(a.P,b)>0&&klb.uk(a.P,b)==6)return false;if(klb.uk(a.P,b)==5&&klb.hn(a.P,b)!=4)return false;if(klb.uk(a.P,b)==7&&!a.Q[b])return false;n=oW(elb.IX,omb,6,4,15,1);o=oW(elb.IX,omb,6,4,15,1);j=oW(elb.g1,Imb,6,4,16,1);for(h=0;h<klb.hn(a.P,b);h++){f=-1;e=0;for(i=0;i<klb.hn(a.P,b);i++){if(!j[i]){if(f<a.d[klb.wn(a.P,b,i)]){f=a.d[klb.wn(a.P,b,i)];e=i;}}}n[h]=e;o[h]=f;j[e]=true;}if(klb.hn(a.P,b)==4&&o[0]==o[1]&&o[2]==o[3])return false;if(klb.hn(a.P,b)==4&&(o[0]==o[2]||o[1]==o[3]))return false;if(klb.hn(a.P,b)==3&&o[0]==o[2])return false;k=0;l=0;m=false;for(g=1;g<klb.hn(a.P,b);g++){if(o[g-1]==o[g]){if(c==1||o[g]==0)return false;k=klb.wn(a.P,b,n[g-1]);l=klb.wn(a.P,b,n[g]);c==3&&klb.Co(a.P,klb.yn(a.P,b,n[g]))&&(a.U[b]=true);m=true;}}if(c!=1&&!m)return false;d=a.hb?klb.Qe(a,b,n):klb.Pe(a,b,n);c==1?a.bb[b]=d:c==2&&(d==1?klb.qg(a.c[k],z1(a.d[b])):d==2&&klb.qg(a.c[l],z1(a.d[b])));return true;};klb.Pe=function Pe(a,b,c){var d,e,f,g,h,i,j,k,l,m;m=wW(iW(elb.IX,2),Lnb,7,0,[wW(iW(elb.IX,1),omb,6,15,[2,1,2,1]),wW(iW(elb.IX,1),omb,6,15,[1,2,2,1]),wW(iW(elb.IX,1),omb,6,15,[1,1,2,2]),wW(iW(elb.IX,1),omb,6,15,[2,1,1,2]),wW(iW(elb.IX,1),omb,6,15,[2,2,1,1]),wW(iW(elb.IX,1),omb,6,15,[1,2,1,2])]);d=oW(elb.GX,mmb,6,klb.hn(a.P,b),15,1);for(g=0;g<klb.hn(a.P,b);g++)d[g]=klb.yk(a.P,klb.wn(a.P,b,c[g]),b);j=klb.Dn(a.P,b,c,d,null)<<24>>24;if(j!=3)return j;k=0;l=0;for(h=0;h<klb.hn(a.P,b);h++){e=klb.yn(a.P,b,c[h]);if(klb.zk(a.P,0,e)==b){if(klb.Kk(a.P,e)==129){l!=0&&klb.tm(a.P,b);k=h;l=1;}if(klb.Kk(a.P,e)==257){l!=0&&klb.tm(a.P,b);k=h;l=2;}}}if(l==0)return 3;for(f=1;f<klb.hn(a.P,b);f++)d[f]<d[0]&&(d[f]+=Mmb);if(klb.hn(a.P,b)==3){switch(k){case 0:(d[1]<d[2]&&d[2]-d[1]<Nmb||d[1]>d[2]&&d[1]-d[2]>Nmb)&&(l=3-l);break;case 1:d[2]-d[0]>Nmb&&(l=3-l);break;case 2:d[1]-d[0]<Nmb&&(l=3-l);}return l==1?2:1;}i=0;d[1]<=d[2]&&d[2]<=d[3]?i=0:d[1]<=d[3]&&d[3]<=d[2]?i=1:d[2]<=d[1]&&d[1]<=d[3]?i=2:d[2]<=d[3]&&d[3]<=d[1]?i=3:d[3]<=d[1]&&d[1]<=d[2]?i=4:d[3]<=d[2]&&d[2]<=d[1]&&(i=5);return m[i][k]==l?2:1;};klb.Qe=function Qe(a,b,c){var d,e,f,g,h,i;d=oW(elb.IX,omb,6,4,15,1);for(h=0;h<klb.hn(a.P,b);h++)d[h]=klb.wn(a.P,b,c[h]);klb.hn(a.P,b)==3&&(d[3]=b);e=mW(elb.GX,[Tlb,mmb],[12,6],15,[3,3],2);for(g=0;g<3;g++){e[g][0]=klb.rk(a.P,d[g+1])-klb.rk(a.P,d[0]);e[g][1]=klb.sk(a.P,d[g+1])-klb.sk(a.P,d[0]);e[g][2]=klb.tk(a.P,d[g+1])-klb.tk(a.P,d[0]);}i=oW(elb.GX,mmb,6,3,15,1);i[0]=e[0][1]*e[1][2]-e[0][2]*e[1][1];i[1]=e[0][2]*e[1][0]-e[0][0]*e[1][2];i[2]=e[0][0]*e[1][1]-e[0][1]*e[1][0];f=(e[2][0]*i[0]+e[2][1]*i[1]+e[2][2]*i[2])/($wnd.Math.sqrt(e[2][0]*e[2][0]+e[2][1]*e[2][1]+e[2][2]*e[2][2])*$wnd.Math.sqrt(i[0]*i[0]+i[1]*i[1]+i[2]*i[2]));return f>0?1:2;};klb.Re=function Re(a){var b,c;b=0;llb.scb(a.c);for(c=0;c<a.c.length;c++){(c==0||klb.rg(a.c[c],a.c[c-1])!=0)&&++b;a.d[a.c[c].a]=b;}return b;};klb.Se=function Se(a){var b,c,d,e,f,g,h,i,j,k,l,m;if(a.v)return;a.v=new llb.Qi();k=0;l=oW(elb.IX,omb,6,a.P.f,15,1);g=oW(elb.IX,omb,6,a.P.f,15,1);i=oW(elb.IX,omb,6,a.P.g,15,1);for(b=0;b<a.P.f;b++){if(l[b]==0&&(klb.Bo(a.P,b)||klb.nn(a.P,b)==1)){g[0]=b;h=1;j=0;l[b]=++k;c=oW(elb.g1,Imb,6,a.P.g,16,1);for(f=0;f<h;f++){for(m=0;m<klb.xn(a.P,g[f]);m++){e=klb.yn(a.P,g[f],m);if(klb.Co(a.P,e)||klb.Hk(a.P,e)==2||klb.qo(a.P,e)){d=klb.wn(a.P,g[f],m);if(!c[e]){i[j++]=e;c[e]=true;}if(l[d]==0){g[h++]=d;l[d]=k;}}}}llb.wi(a.v,new klb.yg(g,h,i,j));}}};klb.Te=function Te(a){var b;a.Q=oW(elb.g1,Imb,6,a.P.f,16,1);for(b=0;b<a.P.f;b++){if(klb.uk(a.P,b)==7){if(klb.xn(a.P,b)==4){a.Q[b]=true;continue;}if(klb.xn(a.P,b)==3){if(klb.rn(a.P,b)==3){a.Q[b]=true;continue;}if(klb.dk(a.P,b)==1){a.Q[b]=true;continue;}if(klb.vo(a.P,b,true))continue;if((a.O&32)!=0){a.Q[b]=true;continue;}if(klb.Ao(a.P,b)){a.Q[b]=true;continue;}}}}};klb.Ue=function Ue(a,b){var c,d,e,f;e=false;for(d=0;d<a.P.g;d++)if(klb.Je(a,d,1)){a.o[d]=a.J;b&&klb.hf(a,d);e=true;}f=false;for(c=0;c<a.P.f;c++)if(klb.Oe(a,c,1)){a.gb[c]=a.J;b&&klb.jf(a,c);f=true;}f&&(a.J=!a.J);return e||f;};klb.Ve=function Ve(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B;s=oW(elb.g1,Imb,6,a.P.f,16,1);t=oW(elb.g1,Imb,6,a.P.g,16,1);b=0;v=false;if((a.O&128)!=0){a.W=oW(elb.IX,omb,6,a.P.f,15,1);a.V=oW(elb.IX,omb,6,a.P.g,15,1);}for(d=0;d<a.P.f;d++){if(a.U[d]){if(!a.db[d]){if(klb.Oe(a,d,1)){a.db[d]=true;s[d]=true;++b;}}}}for(f=0;f<a.P.g;f++){if(a.T[f]){if(!a.n[f]){if(klb.Je(a,f,1)){a.n[f]=true;t[f]=true;++b;}}}}if(b==1){for(c=0;c<a.P.f;c++){if(s[c]){a.bb[c]=0;break;}}for(e=0;e<a.P.g;e++){if(t[e]){a.k[e]=0;break;}}}else if(b>1){klb.Se(a);a.R=0;for(h=new llb.Wbb(a.v);h.a<h.c.a.length;){g=llb.Vbb(h);u=0;w=0;k=0;j=0;l=-1;i=-1;for(o=0;o<g.a.length;o++){if(s[g.a[o]]){++u;if(a.bb[g.a[o]]==1||a.bb[g.a[o]]==2){++w;v=true;if(l<a.d[g.a[o]]){l=a.d[g.a[o]];k=g.a[o];}}}}for(p=0;p<g.b.length;p++){if(t[g.b[p]]){++u;A=a.d[klb.zk(a.P,0,g.b[p])];B=a.d[klb.zk(a.P,1,g.b[p])];m=A>B?(A<<16)+B:(B<<16)+A;if(a.k[g.b[p]]==1||a.k[g.b[p]]==2){++w;v=true;if(i<m){i=m;j=g.b[p];}}}}if(u==0)continue;if(u==1){for(q=0;q<g.a.length;q++)s[g.a[q]]&&(a.bb[g.a[q]]=0);for(n=0;n<g.b.length;n++)t[g.b[n]]&&(a.k[g.b[n]]=0);}else {if(w==1){for(q=0;q<g.a.length;q++)s[g.a[q]]&&(a.bb[g.a[q]]=3);for(n=0;n<g.b.length;n++)t[g.b[n]]&&(a.k[g.b[n]]=3);}else {if((a.O&128)!=0){++a.R;for(q=0;q<g.a.length;q++)s[g.a[q]]&&(a.W[g.a[q]]=a.R);for(n=0;n<g.b.length;n++)t[g.b[n]]&&(a.V[g.b[n]]=a.R);}r=false;l!=-1?a.bb[k]==2&&(r=true):a.k[j]==2&&(r=true);if(r){for(q=0;q<g.a.length;q++){if(s[g.a[q]]){switch(a.bb[g.a[q]]){case 1:a.bb[g.a[q]]=2;break;case 2:a.bb[g.a[q]]=1;}}}for(n=0;n<g.b.length;n++){if(t[g.b[n]]){switch(a.k[g.b[n]]){case 1:a.k[g.b[n]]=2;break;case 2:a.k[g.b[n]]=1;}}}}}}}}return v;};klb.We=function We(a,b){var c,d,e,f,g,h,i,j,k,l,m;f=mW(elb.IX,[Lnb,omb],[7,6],15,[2,32],2);for(g=0;g<2;g++){c=oW(elb.IX,Lnb,7,32,0,2);m=0;for(e=0;e<32;e++){if(b[g][e]!=null){k=b[g][e].length;c[e]=oW(elb.IX,omb,6,k,15,1);for(h=0;h<k;h++)c[e][h]=a.d[b[g][e][h]];ilb.rgb(c[e],ilb.Xgb(e2(llb.Ccb.prototype.kc,llb.Ccb,[])));++m;}}for(l=m;l>0;l--){j=0;i=null;for(d=0;d<32;d++){if(c[d]!=null){if(i==null||i.length<c[d].length){i=c[d];j=d;}else if(i.length==c[d].length){for(h=i.length-1;h>=0;h--){if(i[h]<c[d][h]){i=c[d];j=d;break;}}}}}f[g][j]=l;c[j]=null;}}return f;};klb.Xe=function Xe(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t;l=false;if(a.P.K){for(j=0;j<a.P.g;j++){if(klb.Jk(a.P,j)!=0){l=true;break;}}}a.M=2;for(c=0;c<a.P.f;c++)a.M=$wnd.Math.max(a.M,klb.xn(a.P,c)+klb.Pn(a.P,c));i=$wnd.Math.max(2,l?(62+a.b+a.M*(a.b+23))/63|0:(62+a.b+a.M*(a.b+5))/63|0);a.d=oW(elb.IX,omb,6,a.P.q,15,1);a.c=oW(elb.UX,Mnb,110,a.P.f,0,1);for(d=0;d<a.P.f;d++)a.c[d]=new klb.tg(i);h=false;for(e=0;e<a.P.f;e++){klb.sg(a.c[e],e);I1(n1(klb.pk(a.P,e),1),0)||klb.kk(a.P,e)!=null?klb.pg(a.c[e],8,6):klb.pg(a.c[e],8,z1(klb.uk(a.P,e)));klb.pg(a.c[e],8,z1(klb.nk(a.P,e)));klb.pg(a.c[e],2,z1(klb.nn(a.P,e)));klb.pg(a.c[e],4,z1(klb.xn(a.P,e)+klb.Pn(a.P,e)));I1(n1(klb.pk(a.P,e),1),0)?klb.pg(a.c[e],4,8):klb.pg(a.c[e],4,z1(8+klb.dk(a.P,e)));klb.pg(a.c[e],5,z1($wnd.Math.min(31,klb.rn(a.P,e))));klb.pg(a.c[e],4,z1(klb.Me(a,e)+1));klb.pg(a.c[e],2,z1(klb.qk(a.P,e)>>4));if(a.P.K){klb.pg(a.c[e],46,klb.pk(a.P,e));klb.kk(a.P,e)!=null&&(h=true);}}a.S=klb._e(a);if(a.S<a.P.f){for(b=0;b<a.P.f;b++){klb.sg(a.c[b],b);klb.pg(a.c[b],a.b,z1(a.d[b]));m=oW(elb.IX,omb,6,klb.xn(a.P,b),15,1);for(o=0;o<klb.xn(a.P,b);o++){m[o]=a.d[klb.wn(a.P,b,o)]<<5;m[o]|=$wnd.Math.min(31,klb.vn(a.P,klb.yn(a.P,b,o)));}ilb.rgb(m,ilb.Xgb(e2(llb.Ccb.prototype.kc,llb.Ccb,[])));for(p=a.M;p>m.length;p--)klb.pg(a.c[b],a.b+5,0);for(n=m.length-1;n>=0;n--)klb.pg(a.c[b],a.b+5,elb.v1(m[n]));}a.S=klb._e(a);}if(h&&a.S<a.P.f){for(b=0;b<a.P.f;b++){klb.sg(a.c[b],b);klb.pg(a.c[b],a.b,z1(a.d[b]));g=klb.kk(a.P,b);s=g==null?0:$wnd.Math.min(12,g.length);for(o=12;o>s;o--)klb.pg(a.c[b],8,0);for(n=s-1;n>=0;n--)klb.pg(a.c[b],8,elb.v1(g[n]));}a.S=klb._e(a);}if(l&&a.S<a.P.f){for(b=0;b<a.P.f;b++){klb.sg(a.c[b],b);klb.pg(a.c[b],a.b,z1(a.d[b]));k=oW(elb.JX,Inb,6,klb.xn(a.P,b)+klb.Pn(a.P,b),14,1);q=0;for(o=0;o<klb.jn(a.P,b);o++){if(o<klb.xn(a.P,b)||o>=klb.hn(a.P,b)){k[q]=z1(a.d[klb.wn(a.P,b,o)]);k[q]=L1(k[q],23);k[q]=K1(k[q],z1(klb.Jk(a.P,klb.yn(a.P,b,o))));++q;}}ilb.rgb(k,ilb.Xgb(e2(llb.zcb.prototype.lc,llb.zcb,[])));for(p=a.M;p>k.length;p--)klb.pg(a.c[b],a.b+23,0);for(n=k.length-1;n>=0;n--)klb.pg(a.c[b],a.b+23,k[n]);}a.S=klb._e(a);}if((a.O&8)!=0&&a.S<a.P.f){r=new klb.Gt();for(f=0;f<a.P.f;f++)klb.fk(a.P,f)!=null&&klb.Et(r,klb.fk(a.P,f));for(b=0;b<a.P.f;b++){t=klb.fk(a.P,b)==null?0:1+klb.Ft(r,klb.fk(a.P,b));klb.sg(a.c[b],b);klb.pg(a.c[b],a.b,z1(a.d[b]));klb.pg(a.c[b],a.b,elb.v1(t));}a.S=klb._e(a);}if((a.O&16)!=0&&a.S<a.P.f){for(b=0;b<a.P.f;b++){klb.sg(a.c[b],b);klb.pg(a.c[b],a.b,z1(a.d[b]));klb.pg(a.c[b],1,z1(klb.ol(a.P,b)?1:0));}a.S=klb._e(a);}(a.O&512)!=0&&a.P.K&&klb.Ae(a);};klb.Ye=function Ye(a){var b,c,d,e,f,g,h,i,j;for(c=0;c<a.P.f;c++){klb.sg(a.c[c],c);klb.pg(a.c[c],2*a.b+4,L1(z1(a.d[c]),a.b+4));}for(i=1;i<=a.S;i++){e=false;for(d=0;d<a.P.f;d++)a.d[d]==i&&(e=e|klb.Oe(a,d,2));if(e){h=a.S;a.S=klb._e(a);if(a.S!=h)return true;for(b=0;b<a.P.f;b++){klb.sg(a.c[b],b);klb.pg(a.c[b],2*a.b+4,L1(z1(a.d[b]),a.b+4));}}}j=oW(elb.VX,Mnb,141,a.P.g,0,1);for(g=0;g<j.length;g++)j[g]=new klb.wg(a.d[klb.zk(a.P,0,g)],a.d[klb.zk(a.P,1,g)],g);llb.qcb(j,0,j.length,null);for(f=0;f<j.length;f++){if(klb.Je(a,j[f].a,2)){while(f+1<j.length&&klb.vg(j[f],j[f+1])==0)klb.Je(a,j[++f].a,2);h=a.S;a.S=klb._e(a);if(a.S!=h)return true;for(b=0;b<a.P.f;b++){klb.sg(a.c[b],b);klb.pg(a.c[b],2*a.b+4,L1(z1(a.d[b]),a.b+4));}}}return false;};klb.Ze=function Ze(a){var b,c,d,e,f,g,h,i,j,k;f=0;for(c=0;c<a.P.f;c++)a._[c]!=0&&(a._[c]!=2||(a.O&256)==0)&&++f;if(f==0)return;k=oW(elb.IX,omb,6,f,15,1);f=0;for(d=0;d<a.P.f;d++){if(a._[d]!=0&&(a._[d]!=2||(a.O&256)==0)){k[f]=a._[d]<<29|a.$[d]<<24|a.d[d]<<12|d;++f;}}ilb.rgb(k,ilb.Xgb(e2(llb.Ccb.prototype.kc,llb.Ccb,[])));g=0;j=0;h=k[0]&Cnb;while(true){++j;if(j==k.length||h!=(k[j]&Cnb)){e=oW(elb.IX,omb,6,j-g,15,1);for(i=g;i<j;i++){b=k[i]&4095;e[i-g]=b;a.eb[b]=true;}llb.wi(a.fb,e);if(j==k.length)break;h=k[j]&Cnb;g=j;}}};klb.$e=function $e(a){var b,c,d,e,f,g,h,i,j,k,l,m,n;h=false;for(i=0;i<a.fb.a.length;i++){g=llb.Bi(a.fb,i);b=true;n=-1;j=false;for(l=0;l<g.length;l++){c=g[l];if(a.bb[c]==0){b=false;break;}if(a.bb[c]!=3){k=true;for(m=0;m<g.length;m++){if(m!=l&&a.d[c]==a.d[g[m]]){k=false;break;}}if(k&&n<a.d[c]){n=a.d[c];j=a.bb[c]==1;}}}if(b&&n!=-1){for(d=g,e=0,f=d.length;e<f;++e){c=d[e];(a.bb[c]==1||a.bb[c]==2)&&(a.cb[c]=j);a.eb[c]=false;}llb.Ji(a.fb,g);h=true;--i;}}return h;};klb._e=function _e(a){var b,c;b=klb.Re(a);do{c=b;klb.Ne(a);b=klb.Re(a);}while(c!=b);return b;};klb.af=function af(a){(a.O&1)!=0&&(a.O&2)==0&&(a.e=llb.acb(a.d,a.P.f));if((a.O&tnb)==0){a.U=oW(elb.g1,Imb,6,a.P.f,16,1);a.T=oW(elb.g1,Imb,6,a.P.g,16,1);if(a.S<a.P.f){klb.Be(a);if((a.O&tnb)==0){klb.$e(a);!!a.N&&klb.Ig(a.N,a.d);}}}a.e==null&&(a.O&1)!=0&&(a.O&2)!=0&&(a.e=llb.acb(a.d,a.P.f));while(a.S<a.P.f){klb.Ce(a);if((a.O&tnb)==0){klb.$e(a);!!a.N&&klb.Ig(a.N,a.d);}}if((a.O&tnb)==0){klb.$e(a);klb.Ve(a);klb.zf(a);}};klb.bf=function bf(a){var b,c,d,e,f;f=a.S;e=llb.acb(a.d,a.P.f);if(!a.P.K){klb.df(a);klb.Rf(a,f,e);}a._=oW(elb.EX,Nnb,6,a.P.f,15,1);a.$=oW(elb.EX,Nnb,6,a.P.f,15,1);for(c=0;c<a.P.f;c++){a._[c]=klb.ik(a.P,c)<<24>>24;a.$[c]=klb.hk(a.P,c)<<24>>24;}a.j=oW(elb.EX,Nnb,6,a.P.g,15,1);a.i=oW(elb.EX,Nnb,6,a.P.g,15,1);for(d=0;d<a.P.g;d++){a.j[d]=klb.Ek(a.P,d)<<24>>24;a.i[d]=klb.Dk(a.P,d)<<24>>24;}klb.cf(a);a.X=false;a.K=oW(elb.g1,Imb,6,a.P.f,16,1);for(b=0;b<a.P.f;b++){if(a.bb[b]!=0){a.K[b]=true;a.X=true;}}klb.ff(a);a.N=null;a.ab=oW(elb.g1,Imb,6,a.P.f,16,1);if(a.X){a.N=new klb.Og(a.P,e,a.K,a.bb,a.k,a._,a.$,a.gb,a.o,a.ab);klb.Jg(a.N);}a.cb=oW(elb.g1,Imb,6,a.P.f,16,1);a.eb=oW(elb.g1,Imb,6,a.P.f,16,1);a.fb=new llb.Qi();!!a.N&&klb.Ig(a.N,a.d);klb.Ze(a);klb.Rf(a,f,e);klb.ef(a);!!a.N&&(a.I=klb.Fg(a.N));klb.qf(a);};klb.cf=function cf(a){var b,c,d,e,f,g;a.J=true;f=klb.Ue(a,false);while(a.S<a.P.f&&f){for(b=0;b<a.P.f;b++){klb.sg(a.c[b],b);klb.pg(a.c[b],a.b,z1(a.d[b]));g=a.bb[b]<<7;if((a.bb[b]==1||a.bb[b]==2)&&a._[b]!=0){g|=a._[b]<<5;g|=a.$[b];}klb.pg(a.c[b],18,elb.v1(g<<9));}for(c=0;c<a.P.g;c++){d=a.k[c]<<7;if((a.k[c]==1||a.k[c]==2)&&klb.Kk(a.P,c)==1&&a.j[c]!=0){d|=a.j[c]<<5;d|=a.i[c];}klb.qg(a.c[klb.zk(a.P,0,c)],elb.v1(d));klb.qg(a.c[klb.zk(a.P,1,c)],elb.v1(d));}e=klb._e(a);if(a.S==e)break;a.S=e;f=klb.Ue(a,false);}};klb.df=function df(a){var b,c,d,e;a.J=true;a.Y=oW(elb.EX,Nnb,6,a.P.f,15,1);a.f=oW(elb.EX,Nnb,6,a.P.g,15,1);e=klb.Ue(a,true);while(a.S<a.P.f&&e){for(b=0;b<a.P.f;b++){klb.sg(a.c[b],b);klb.pg(a.c[b],a.b+4,z1(a.d[b]<<4|a.bb[b]<<2));}for(c=0;c<a.P.g;c++){klb.qg(a.c[klb.zk(a.P,0,c)],z1(a.k[c]));klb.qg(a.c[klb.zk(a.P,1,c)],z1(a.k[c]));}d=klb._e(a);if(a.S==d)break;a.S=d;e=klb.Ue(a,true);}};klb.ef=function ef(a){var b,c,d,e,f,g,h;a.J=true;d=klb.pf(a);!!a.N&&klb.Ig(a.N,a.d)&&(d=klb.pf(a));klb.Ue(a,false)&&klb.$e(a);g=true;while(a.S<a.P.f&&g){e=klb.We(a,d);for(b=0;b<a.P.f;b++){klb.sg(a.c[b],b);klb.pg(a.c[b],a.b,z1(a.d[b]));klb.pg(a.c[b],20,0);!a.ab[b]&&a._[b]!=0&&klb.qg(a.c[b],z1((a._[b]<<18)+(e[a._[b]==1?0:1][a.$[b]]<<8)));h=a.bb[b];a.cb[b]&&(h==1?h=2:h==2&&(h=1));klb.qg(a.c[b],elb.v1(h<<4));}for(c=0;c<a.P.g;c++){klb.qg(a.c[klb.zk(a.P,0,c)],z1(a.k[c]));klb.qg(a.c[klb.zk(a.P,1,c)],z1(a.k[c]));}f=klb._e(a);if(a.S==f)break;a.S=f;g=false;if(!!a.N&&klb.Ig(a.N,a.d)){g=true;d=klb.pf(a);}if(klb.Ue(a,false)){g=true;klb.$e(a);}}};klb.ff=function ff(a){var b,c;for(b=0;b<a.P.f;b++)(!a.K[b]||a.bb[b]==3)&&(a._[b]=0);for(c=0;c<a.P.g;c++)(klb.Kk(a.P,c)!=1||a.k[c]==0||a.k[c]==3)&&(a.j[c]=0);};klb.gf=function gf(a,b,c){if(a.a==null){a.a=oW(elb.EX,Nnb,6,a.P.f,15,1);llb.dcb(a.a);}a.a[b]=c<<24>>24;};klb.hf=function hf(b,c){var d,e,f,g,h,i;if((b.k[c]==1||b.k[c]==2)&&!klb.Fo(b.P,c)){h=false;try{for(g=0;g<2;g++){d=klb.zk(b.P,g,c);if(klb.xn(b.P,d)==3){e=oW(elb.IX,omb,6,2,15,1);f=0;for(i=0;i<klb.xn(b.P,d);i++)klb.yn(b.P,d,i)!=c&&(e[f++]=klb.wn(b.P,d,i));b.d[e[0]]>b.d[e[1]]^klb.kf(b,d,e[0],e[1])&&(h=!h);}}}catch(a){a=k1(a);if(uX(a,15)){b.f[c]=3;return;}else throw l1(a);}b.k[c]==1^h?b.f[c]=1:b.f[c]=2;}};klb.jf=function jf(b,c){var d,e,f,g,h,i,j;if(b.bb[c]==1||b.bb[c]==2){i=false;if(klb.nn(b.P,c)==2&&klb.xn(b.P,c)==2){try{for(h=0;h<2;h++){d=klb.wn(b.P,c,h);if(klb.xn(b.P,d)==3){f=oW(elb.IX,omb,6,2,15,1);g=0;for(j=0;j<klb.xn(b.P,d);j++)klb.zn(b.P,d,j)==1&&(f[g++]=klb.wn(b.P,d,j));b.d[f[0]]>b.d[f[1]]^klb.kf(b,d,f[0],f[1])&&(i=!i);}}}catch(a){a=k1(a);if(uX(a,15)){b.Y[c]=3;return;}else throw l1(a);}}else {try{e=klb.mf(b,c);}catch(a){a=k1(a);if(uX(a,15)){b.Y[c]=3;return;}else throw l1(a);}for(h=1;h<e.length;h++)for(j=0;j<h;j++)b.d[e[h]]<b.d[e[j]]&&(i=!i);}b.bb[c]==1^i?b.Y[c]=1:b.Y[c]=2;}};klb.kf=function kf(a,b,c,d){var e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L;if(klb.uk(a.P,c)!=klb.uk(a.P,d))return klb.uk(a.P,c)>klb.uk(a.P,d);if(klb.nk(a.P,c)!=klb.nk(a.P,d)){H=klb.ll(a.P,c)?(klb.wj(),klb.uj)[klb.uk(a.P,c)]:klb.nk(a.P,c);I=klb.ll(a.P,d)?(klb.wj(),klb.uj)[klb.uk(a.P,d)]:klb.nk(a.P,d);return H>I;}w=a.P.f;s=oW(elb.IX,omb,6,w,15,1);u=oW(elb.IX,omb,6,w,15,1);v=oW(elb.IX,omb,6,w,15,1);t=oW(elb.g1,Imb,6,w,16,1);i=oW(elb.g1,Imb,6,a.P.q,16,1);s[0]=b;s[1]=c;s[2]=d;u[0]=-1;u[1]=0;u[2]=0;i[b]=true;i[c]=true;i[d]=true;m=1;A=2;G=oW(elb.IX,omb,6,64,15,1);G[1]=1;G[2]=3;o=2;while(m<=A){while(m<G[o]){n=s[m];if(!t[m]){p=0;q=0;for(C=0;C<klb.xn(a.P,n);C++){k=klb.wn(a.P,n,C);if(A+klb.zn(a.P,n,C)+1>=w){w+=a.P.f;s=klb.Vf(s,w);u=klb.Vf(u,w);v=klb.Vf(v,w);t=klb.Wf(t,w);}if(klb.to(a.P,klb.yn(a.P,n,C))){++p;q+=klb.uk(a.P,k);}else if(k!=b){for(F=1;F<klb.zn(a.P,n,C);F++){++A;s[A]=k;u[A]=m;t[A]=true;}}K=u[m];if(k==s[K])continue;h=false;if(i[k]){J=u[K];while(J!=-1){if(k==s[J]){h=true;break;}J=u[J];}}if(h){++A;s[A]=k;u[A]=m;t[A]=true;}else {++A;s[A]=k;u[A]=m;i[k]=true;}}if(p!=0){++A;v[A]=(q<<2)/p|0;u[A]=m;t[A]=true;}}++m;if(m==gmb){throw l1(new flb.uz('Emergency break in while loop.'));}}G.length==o+1&&(G=klb.Vf(G,G.length+64));G[o+1]=A+1;for(B=G[o];B<G[o+1];B++){v[B]==0&&(v[B]=(klb.uk(a.P,s[B])==151?1:klb.uk(a.P,s[B])==152?1:klb.uk(a.P,s[B]))<<2);v[B]+=v[u[B]]<<16;}klb.of(a,t,v,u,s,G,o);if(v[1]!=v[2])return v[1]>v[2];o>1&&klb.lf(v,u,G,o);++o;}l=oW(elb.IX,omb,6,a.P.f,15,1);D=false;for(f=0;f<a.P.f;f++){if(i[f]&&!klb.ll(a.P,f)){D=true;break;}}if(D){for(g=0;g<a.P.f;g++)l[g]=klb.ll(a.P,g)?(klb.wj(),klb.uj)[klb.uk(a.P,g)]:klb.nk(a.P,g);if(klb.nf(a,t,v,u,s,l,G,o))return v[1]>v[2];}llb.jcb(l,l.length,0);r=false;for(j=0;j<a.P.g;j++){if(i[klb.zk(a.P,0,j)]||i[klb.zk(a.P,1,j)]){if(a.f[j]==1){l[klb.zk(a.P,0,j)]=1;l[klb.zk(a.P,1,j)]=1;r=true;}else if(a.f[j]==2){l[klb.zk(a.P,0,j)]=2;l[klb.zk(a.P,1,j)]=2;r=true;}}}if(r&&klb.nf(a,t,v,u,s,l,G,o))return v[1]>v[2];llb.jcb(l,l.length,0);L=false;for(e=0;e<a.P.f;e++){if(i[e]){if(a.Y[e]==2){l[e]=1;L=true;}else if(a.Y[e]==1){l[e]=2;L=true;}}}if(L&&klb.nf(a,t,v,u,s,l,G,o))return v[1]>v[2];throw l1(new flb.uz('no distinction applying CIP rules'));};klb.lf=function lf(a,b,c,d){var e,f,g,h,i,j,k,l,m;l=c[d];g=c[d+1]-l;m=oW(elb.SX,amb,143,g,0,1);for(i=0;i<g;i++){m[i]=new klb.og();m[i].c=a[i+l];m[i].b=b[i+l];m[i].a=i+l;}e=new klb.lg();for(k=d;k>1;k--){for(j=0;j<g;j++){m[j].c+=a[m[j].b]<<16;m[j].b=b[m[j].b];}llb.qcb(m,0,m.length,e);f=1;for(h=0;h<g;h++){a[m[h].a]=f;h!=g-1&&klb.kg(m[h],m[h+1])!=0&&++f;}}};klb.mf=function mf(a,b){var c,d,e,f,g,h,i;g=klb.hn(a.P,b);h=oW(elb.IX,omb,6,g,15,1);for(e=0;e<g;e++)h[e]=klb.wn(a.P,b,e);for(d=g;d>1;d--){c=false;for(f=1;f<d;f++){if(klb.kf(a,b,h[f-1],h[f])){c=true;i=h[f-1];h[f-1]=h[f];h[f]=i;}}if(!c)break;}return h;};klb.nf=function nf(a,b,c,d,e,f,g,h){var i,j;for(j=1;j<h;j++){for(i=g[j];i<g[j+1];i++)c[i]=f[e[i]]+(c[d[i]]<<8);klb.of(a,b,c,d,e,g,j);if(c[1]!=c[2])return true;j>1&&klb.lf(c,d,g,j);}return false;};klb.of=function of(a,b,c,d,e,f,g){var h,i,j,k,l,m,n,o,p,q,r;for(l=g;l>1;l--){p=f[l]-f[l-1];r=oW(elb.QX,amb,142,p,0,1);h=f[l];for(o=0;o<p;o++){q=f[l-1]+o;m=h;while(m<f[l+1]&&d[m]==q)++m;r[o]=new klb.ig();r[o].c=q;r[o].d=c[q];r[o].b=b[q]?0:klb.Yn(a.P,e[q]);r[o].a=oW(elb.IX,omb,6,m-h,15,1);for(k=h;k<m;k++)r[o].a[k-h]=c[k];ilb.rgb(r[o].a,ilb.Xgb(e2(llb.Ccb.prototype.kc,llb.Ccb,[])));h=m;}i=new klb.fg();llb.qcb(r,0,r.length,i);j=1;for(n=0;n<p;n++){c[r[n].c]=j;n!=p-1&&klb.eg(r[n],r[n+1])!=0&&++j;}}};klb.pf=function pf(a){var b,c;c=mW(elb.IX,[Tlb,Lnb],[26,7],0,[2,32],2);for(b=0;b<a.P.f;b++){a.K[b]&&(a._[b]==1?c[0][a.$[b]]=klb.Pg(c[0][a.$[b]],b):a._[b]==2&&(c[1][a.$[b]]=klb.Pg(c[0][a.$[b]],b)));}return c;};klb.qf=function qf(a){var b,c,d,e,f,g,h,i,j,k,l,m;f=0;k=0;g=0;h=0;i=0;j=0;l=0;m=false;b=oW(elb.g1,Imb,6,32,16,1);for(c=0;c<a.P.f;c++){if(a.bb[c]!=0){++f;if(a.bb[c]==3){++k;}else {if(a._[c]==0){++g;!!a.N&&klb.Eg(a.N,c)&&++h;}else if(a._[c]==2){a.$[c]==0&&++j;}else if(a._[c]==1){e=a.$[c];if(!b[e]){++l;b[e]=true;}a.$[c]==0&&++i;!!a.N&&klb.Eg(a.N,c)&&(m=true);}}}}for(d=0;d<a.P.g;d++){if(a.k[d]!=0&&klb.Kk(a.P,d)==1){++f;if(a.k[d]==3){++k;}else {if(a.j[d]==0){++g;!!a.N&&klb.Eg(a.N,klb.zk(a.P,0,d))&&klb.Eg(a.N,klb.zk(a.P,1,d))&&++h;}else if(a.j[d]==2){a.i[d]==0&&++j;}else if(a.j[d]==1){e=a.i[d];if(!b[e]){++l;b[e]=true;}a.i[d]==0&&++i;!!a.N&&klb.Eg(a.N,klb.zk(a.P,0,d))&&klb.Eg(a.N,klb.zk(a.P,1,d))&&(m=true);}}}}if(f==0){klb.lm(a.P,emb);return;}if(k!=0){klb.lm(a.P,0);return;}if(a.I){klb.lm(a.P,Onb+(1<<l));return;}i+h==f&&!m?klb.lm(a.P,196608):g==f?klb.lm(a.P,Bnb):j==f?klb.lm(a.P,327680):g==f-1&&i==1?klb.lm(a.P,dnb):klb.lm(a.P,458752+(1<<l));};klb.rf=function rf(a,b,c){while(c!=0){if(a.r==0){(!a.p||a.t!=63)&&(a.t+=64);flb.J5(a.s,a.t&fmb);a.r=6;a.t=0;}a.t<<=1;a.t=R1(K1(z1(a.t),n1(b,1)));b=M1(b,1);--c;--a.r;}};klb.sf=function sf(a){a.t<<=a.r;(!a.p||a.t!=63)&&(a.t+=64);flb.J5(a.s,a.t&fmb);return a.s.a;};klb.tf=function tf(a,b){a.s=new flb.Q5();a.r=6;a.t=0;a.p=b;};klb.uf=function uf(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q;if(a.P.f==0){a.q='';return;}l=false;if(a.hb&&a.P.q>a.P.f&&!a.P.K){l=true;for(i=0;i<a.P.f;i++){if(klb.Mn(a.P,i)!=0){l=false;break;}}}q=a.hb?16:8;klb.tf(a,true);flb.J5(a.s,l?35:33);klb.rf(a,z1(a.hb?1:0),1);klb.rf(a,elb.v1(b?1:0),1);klb.rf(a,elb.v1(q/2|0),4);o=0;for(j=1;j<a.P.f;j++)o=klb.Hf(a,a.w[j],a.C[j]==-1?-1:a.w[a.C[j]],o,c);if(l){for(i=0;i<a.P.f;i++){d=a.w[i];for(n=klb.xn(a.P,d);n<klb.hn(a.P,d);n++)o=klb.Hf(a,klb.wn(a.P,d,n),d,o,c);}}if(a.P.f>1&&o==0){a.q='';return;}g=1<<q;m=o/(g/2-1);p=o+m/2;for(k=1;k<a.P.f;k++)klb.vf(a,a.w[k],a.C[k]==-1?-1:a.w[a.C[k]],p,m,q,c);if(l){for(h=0;h<a.P.f;h++){d=a.w[h];for(n=klb.xn(a.P,d);n<klb.hn(a.P,d);n++)klb.vf(a,klb.wn(a.P,d,n),d,p,m,q,c);}}if(b){f=a.hb?1.5:(klb.wj(),klb.wj(),klb.vj);e=klb.xk(a.P,l?a.P.q:a.P.f,l?a.P.r:a.P.g,f,c);klb.rf(a,z1($wnd.Math.min(g-1,$wnd.Math.max(0,DX(0.5+$wnd.Math.log(e/0.1)*$wnd.Math.LOG10E/($wnd.Math.log(2000)*$wnd.Math.LOG10E)*(g-1))))),q);klb.rf(a,z1(klb.yf(c[a.w[0]].a/e,g)),q);klb.rf(a,z1(klb.yf(c[a.w[0]].b/e,g)),q);a.hb&&klb.rf(a,z1(klb.yf(c[a.w[0]].c/e,g)),q);}a.q=klb.sf(a);};klb.vf=function vf(a,b,c,d,e,f,g){var h,i,j;h=c==-1?(g[b].a-g[a.w[0]].a)/8:g[b].a-g[c].a;i=c==-1?(g[b].b-g[a.w[0]].b)/8:g[b].b-g[c].b;klb.rf(a,z1(DX((d+h)/e)),f);klb.rf(a,z1(DX((d+i)/e)),f);if(a.hb){j=c==-1?(g[b].c-g[a.w[0]].c)/8:g[b].c-g[c].c;klb.rf(a,z1(DX((d+j)/e)),f);}};klb.wf=function wf(a,b){var c;for(c=0;c<a.u;c++)b-=16;b<0&&(flb.V5(),String.fromCharCode(10));while(b>15){klb.rf(a,1,1);klb.rf(a,15,4);b-=16;++a.u;}klb.rf(a,1,1);klb.rf(a,elb.v1(b),4);};klb.xf=function xf(a){var b,c,d,e,f,g;if(a.P.f==0){a.L='';return;}f=0;d=false;e=false;for(c=0;c<a.P.f;c++){f<klb.mk(a.P,c)&&(f=klb.mk(a.P,c));klb.al(a.P,c)?d=true:e=true;}if(f==0){a.L='';return;}g=klb.cg(f);klb.tf(a,true);klb.rf(a,elb.v1(g),4);klb.rf(a,elb.v1(d?1:0),1);klb.rf(a,elb.v1(e?1:0),1);for(b=0;b<a.P.f;b++){klb.rf(a,z1(klb.mk(a.P,a.w[b])),g);d&&e&&klb.rf(a,z1(klb.al(a.P,a.w[b])?1:0),1);}a.L=klb.sf(a);};klb.yf=function yf(a,b){var c,d,e,f;c=b/2|0;e=a<0;a=$wnd.Math.abs(a);f=b/32|0;d=$wnd.Math.min(c-1,R1(y1($wnd.Math.round(a*c/(a+f)))));return e?c+d:d;};klb.zf=function zf(a){var b,c,d;for(b=0;b<a.P.f;b++){a.bb[b]==3&&!klb.Yk(a.P,b)&&klb.tm(a.P,b);(klb.ik(a.P,b)==1||klb.ik(a.P,b)==2)&&a.bb[b]==3&&klb.tm(a.P,b);klb.Yk(a.P,b)&&a.bb[b]!=3&&!klb.Tf(a,b)&&klb.tm(a.P,b);}for(d=0;d<a.P.r;d++)klb.ql(a.P,d)&&!klb.Sf(a,d)&&klb.tm(a.P,klb.zk(a.P,0,d));for(c=0;c<a.P.g;c++){if(klb.Hk(a.P,c)==2){if(klb.gl(a.P,c)&&(a.k[c]==1||a.k[c]==2)){a.k[c]=3;klb.km(a.P,c,386);}if(a.k[c]==3&&!a.n[c]){if(klb.Kk(a.P,c)!=386){klb.tm(a.P,klb.zk(a.P,0,c));klb.tm(a.P,klb.zk(a.P,1,c));}}}if(klb.Kk(a.P,c)==1&&a.k[c]==3&&!klb.Yk(a.P,klb.zk(a.P,0,c))&&!klb.Yk(a.P,klb.zk(a.P,1,c))){klb.tm(a.P,klb.zk(a.P,0,c));klb.tm(a.P,klb.zk(a.P,1,c));}if((klb.Ek(a.P,c)==1||klb.Ek(a.P,c)==2)&&(klb.Kk(a.P,c)!=1||a.k[c]!=1&&a.k[c]!=2)){klb.tm(a.P,klb.zk(a.P,0,c));klb.tm(a.P,klb.zk(a.P,1,c));}}};klb.Af=function Af(a){var b,c,d,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v;if(a.P.f==0)return;if(a.D)return;a.G=0;v=0;for(c=1;c<a.P.f;c++)a.d[c]>a.d[v]&&(v=c);d=oW(elb.g1,Imb,6,a.P.f,16,1);g=oW(elb.g1,Imb,6,a.P.g,16,1);a.F=oW(elb.IX,omb,6,a.P.f,15,1);a.w=oW(elb.IX,omb,6,a.P.f,15,1);a.C=oW(elb.IX,omb,6,a.P.f,15,1);a.A=oW(elb.IX,omb,6,a.P.g,15,1);a.w[0]=v;a.F[v]=0;d[v]=true;i=0;j=1;k=0;while(i<a.P.f){if(i<j){while(true){o=0;p=0;m=-1;b=a.w[i];for(q=0;q<klb.jn(a.P,b);q++){if(q<klb.xn(a.P,b)||q>=klb.hn(a.P,b)){h=klb.wn(a.P,b,q);if(!d[h]&&a.d[h]>m){o=h;p=klb.yn(a.P,b,q);m=a.d[h];}}}if(m==-1)break;a.F[o]=j;a.C[j]=i;a.w[j++]=o;a.A[k++]=p;d[o]=true;g[p]=true;}++i;}else {n=0;m=-1;for(b=0;b<a.P.f;b++){if(!d[b]&&a.d[b]>m){n=b;m=a.d[b];}}a.F[n]=j;a.C[j]=-1;a.w[j++]=n;d[n]=true;}}a.B=oW(elb.IX,omb,6,2*(a.P.g-k),15,1);while(true){s=a.P.M;t=a.P.M;u=-1;for(f=0;f<a.P.g;f++){if(!g[f]){if(a.F[klb.zk(a.P,0,f)]<a.F[klb.zk(a.P,1,f)]){r=a.F[klb.zk(a.P,0,f)];l=a.F[klb.zk(a.P,1,f)];}else {r=a.F[klb.zk(a.P,1,f)];l=a.F[klb.zk(a.P,0,f)];}if(r<s||r==s&&l<t){s=r;t=l;u=f;}}}if(u==-1)break;g[u]=true;a.A[k++]=u;a.B[2*a.G]=s;a.B[2*a.G+1]=t;++a.G;}a.D=true;};klb.Bf=function Bf(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,o;i=null;o=klb.$n(a.P);for(l=0;l<o.i.a.length;l++){if(o.e[l]){f=0;m=llb.Bi(o.i,l);for(c=m,d=0,e=c.length;d<e;++d){b=c[d];klb.Nf(a,b)&&++f;}if(f!=0){n=llb.Bi(o.j,l);i==null&&(i=oW(elb.g1,Imb,6,a.P.g,16,1));if(f==m.length){j=-1;k=Ylb;for(g=0;g<m.length;g++){if(k>a.w[n[g]]){k=a.w[n[g]];j=g;}}while(f>0){i[n[j]]=true;j=klb._f(j+2,m.length);f-=2;}}else {h=0;while(klb.Nf(a,m[h]))++h;while(!klb.Nf(a,m[h]))h=klb._f(h+1,m.length);while(f>0){i[n[h]]=true;h=klb._f(h+2,m.length);f-=2;while(!klb.Nf(a,m[h]))h=klb._f(h+1,m.length);}}}}}return i;};klb.Cf=function Cf(a,b){return a.k[b];};klb.Df=function Df(a){return klb.Ef(a,a.hb);};klb.Ef=function Ef(a,b){if(a.q==null){klb.Af(a);klb.uf(a,b,a.P.J);}return a.q;};klb.Ff=function Ff(a){if(a.L==null){klb.Af(a);klb.xf(a);}return a.L;};klb.Gf=function Gf(a){if(a.H==null){klb.Af(a);if((a.O&tnb)==0){klb.Pf(a);klb.Qf(a,1);klb.Qf(a,2);}klb.Of(a);}return a.H;};klb.Hf=function Hf(a,b,c,d,e){var f,g,h;f=c==-1?$wnd.Math.abs(e[b].a-e[a.w[0]].a)/8:$wnd.Math.abs(e[b].a-e[c].a);d<f&&(d=f);g=c==-1?$wnd.Math.abs(e[b].b-e[a.w[0]].b)/8:$wnd.Math.abs(e[b].b-e[c].b);d<g&&(d=g);if(a.hb){h=c==-1?$wnd.Math.abs(e[b].c-e[a.w[0]].c)/8:$wnd.Math.abs(e[b].c-e[c].c);d<h&&(d=h);}return d;};klb.If=function If(a,b){return a.V[b];};klb.Jf=function Jf(a,b){return a.W[b];};klb.Kf=function Kf(a,b){return a.e==null?-1:a.e[b];};klb.Lf=function Lf(a,b){return a.bb[b];};klb.Mf=function Mf(a,b){var c,d,e,f,g,h,i;i=klb.$n(a.P);for(c=0;c<i.i.a.length;c++){if(i.d[c]&&klb.ur(i,c,b)){for(e=llb.Bi(i.i,c),f=0,g=e.length;f<g;++f){d=e[f];if(d!=b)for(h=0;h<klb.xn(a.P,d);h++)if(klb.qo(a.P,klb.yn(a.P,d,h)))return true;}return false;}}return false;};klb.Nf=function Nf(a,b){var c,d,e;if(klb.nn(a.P,b)<2)return false;if(klb.xn(a.P,b)==2)return true;c=0;for(e=0;e<klb.xn(a.P,b);e++){d=klb.yn(a.P,b,e);klb.po(a.P,d)&&(c+=klb.Hk(a.P,d)-1);}return c>1;};klb.Of=function Of(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N,O,P,Q,R,S,T,U,V,W,X;klb.tf(a,false);klb.rf(a,9,4);T=$wnd.Math.max(klb.cg(a.P.f),klb.cg(a.P.g));klb.rf(a,elb.v1(T),4);if(T==0){klb.rf(a,z1(a.P.K?1:0),1);klb.rf(a,0,1);a.H=klb.sf(a);return;}U=W=V=H=0;for(i=0;i<a.P.f;i++){if(x1(n1(klb.pk(a.P,i),1),0)){switch(klb.uk(a.P,i)){case 6:break;case 7:++U;break;case 8:++W;break;default:++V;}klb.dk(a.P,i)!=0&&++H;}}klb.rf(a,z1(a.P.f),T);klb.rf(a,z1(a.P.g),T);klb.rf(a,elb.v1(U),T);klb.rf(a,elb.v1(W),T);klb.rf(a,elb.v1(V),T);klb.rf(a,elb.v1(H),T);for(j=0;j<a.P.f;j++)klb.uk(a.P,a.w[j])==7&&x1(n1(klb.pk(a.P,a.w[j]),1),0)&&klb.rf(a,elb.v1(j),T);for(m=0;m<a.P.f;m++)klb.uk(a.P,a.w[m])==8&&x1(n1(klb.pk(a.P,a.w[m]),1),0)&&klb.rf(a,elb.v1(m),T);for(n=0;n<a.P.f;n++)if(klb.uk(a.P,a.w[n])!=6&&klb.uk(a.P,a.w[n])!=7&&klb.uk(a.P,a.w[n])!=8&&x1(n1(klb.pk(a.P,a.w[n]),1),0)){klb.rf(a,elb.v1(n),T);klb.rf(a,z1(klb.uk(a.P,a.w[n])),8);}for(o=0;o<a.P.f;o++)if(klb.dk(a.P,a.w[o])!=0&&x1(n1(klb.pk(a.P,a.w[o]),1),0)){klb.rf(a,elb.v1(o),T);klb.rf(a,z1(8+klb.dk(a.P,a.w[o])),4);}S=0;v=0;for(p=1;p<a.P.f;p++){if(a.C[p]==-1){L=0;}else {L=1+a.C[p]-v;v=a.C[p];}S<L&&(S=L);}K=klb.cg(S);klb.rf(a,elb.v1(K),4);v=0;for(q=1;q<a.P.f;q++){if(a.C[q]==-1){L=0;}else {L=1+a.C[q]-v;v=a.C[q];}klb.rf(a,elb.v1(L),K);}for(N=0;N<2*a.G;N++)klb.rf(a,z1(a.B[N]),T);for(A=0;A<a.P.g;A++){G=(klb.Jk(a.P,a.A[A])&vnb)!=0||klb.Kk(a.P,a.A[A])==32?1:klb.to(a.P,a.A[A])?0:$wnd.Math.min(3,klb.Hk(a.P,a.A[A]));klb.rf(a,elb.v1(G),2);}c=0;if((a.O&tnb)==0){for(r=0;r<a.P.f;r++)a.Z[a.w[r]]!=0&&a.Z[a.w[r]]!=3&&++c;}klb.rf(a,elb.v1(c),T);if((a.O&tnb)==0){for(r=0;r<a.P.f;r++){if(a.Z[a.w[r]]!=0&&a.Z[a.w[r]]!=3){klb.rf(a,elb.v1(r),T);if(a._[a.w[r]]==0){klb.rf(a,z1(a.Z[a.w[r]]),3);}else {X=a.Z[a.w[r]]==1?a._[a.w[r]]==1?4:6:a._[a.w[r]]==1?5:7;klb.rf(a,elb.v1(X),3);klb.rf(a,z1(a.$[a.w[r]]),3);}}}}b=0;if((a.O&tnb)==0){for(B=0;B<a.P.g;B++)a.g[a.A[B]]!=0&&a.g[a.A[B]]!=3&&(!klb.Fo(a.P,a.A[B])||klb.Kk(a.P,a.A[B])==1)&&++b;}klb.rf(a,elb.v1(b),T);if((a.O&tnb)==0){for(B=0;B<a.P.g;B++){if(a.g[a.A[B]]!=0&&a.g[a.A[B]]!=3&&(!klb.Fo(a.P,a.A[B])||klb.Kk(a.P,a.A[B])==1)){klb.rf(a,elb.v1(B),T);if(klb.Kk(a.P,a.A[B])==1){if(a.j[a.A[B]]==0){klb.rf(a,z1(a.g[a.A[B]]),3);}else {X=a.g[a.A[B]]==1?a.j[a.A[B]]==1?4:6:a.j[a.A[B]]==1?5:7;klb.rf(a,elb.v1(X),3);klb.rf(a,z1(a.i[a.A[B]]),3);}}else {klb.rf(a,z1(a.g[a.A[B]]),2);}}}}klb.rf(a,z1(a.P.K?1:0),1);I=0;for(s=0;s<a.P.f;s++)klb.nk(a.P,a.w[s])!=0&&++I;if(I!=0){klb.wf(a,1);klb.rf(a,elb.v1(I),T);for(r=0;r<a.P.f;r++){if(klb.nk(a.P,a.w[r])!=0){klb.rf(a,elb.v1(r),T);klb.rf(a,z1(klb.nk(a.P,a.w[r])),8);}}}a.u=0;if(a.P.K){klb.ye(a,0,T,tnb,1,-1);klb.ye(a,3,T,Qmb,1,-1);klb.ye(a,4,T,120,4,3);klb.ye(a,5,T,Pnb,2,1);klb.ye(a,6,T,1,1,-1);klb.ye(a,7,T,Umb,4,7);I=0;for(r=0;r<a.P.f;r++)klb.kk(a.P,a.w[r])!=null&&++I;if(I>0){klb.wf(a,8);klb.rf(a,elb.v1(I),T);for(t=0;t<a.P.f;t++){u=klb.kk(a.P,a.w[t]);if(u!=null){klb.rf(a,elb.v1(t),T);klb.rf(a,z1(u.length),4);for(e=u,f=0,g=e.length;f<g;++f){d=e[f];klb.rf(a,elb.v1(d),8);}}}}klb.ze(a,9,T,384,2,7);klb.ze(a,10,T,31,5,0);klb.ye(a,11,T,snb,1,-1);klb.ze(a,12,T,vnb,8,9);klb.ye(a,13,T,$mb,3,14);klb.ye(a,14,T,_mb,5,17);klb.ye(a,16,T,qnb,3,22);}I=0;for(k=0;k<a.P.f;k++)a.a!=null&&a.a[a.w[k]]!=-1&&++I;if(I!=0){klb.wf(a,17);klb.rf(a,elb.v1(I),T);for(r=0;r<a.P.f;r++){if(a.a!=null&&a.a[a.w[r]]!=-1){klb.rf(a,elb.v1(r),T);klb.rf(a,z1(a.a[a.w[r]]),4);}}}if((a.O&8)!=0||(a.O&Vmb)!=0){I=0;R=0;for(r=0;r<a.P.f;r++){P=klb.fk(a.P,a.w[r]);if(P!=null){++I;R=$wnd.Math.max(R,ilb.Xgb(P).length);}}if(I!=0){Q=klb.cg(R);klb.wf(a,18);klb.rf(a,elb.v1(I),T);klb.rf(a,elb.v1(Q),4);for(t=0;t<a.P.f;t++){J=klb.fk(a.P,a.w[t]);if(J!=null){klb.rf(a,elb.v1(t),T);klb.rf(a,z1(ilb.Xgb(J).length),Q);for(M=0;M<ilb.Xgb(J).length;M++)klb.rf(a,(ilb.Sgb(M,ilb.Xgb(J).length),z1(ilb.Xgb(J).charCodeAt(M))),7);}}}}if(a.P.K){klb.ye(a,19,T,Wmb,3,25);klb.ze(a,20,T,enb,3,17);}I=0;for(l=0;l<a.P.f;l++)klb.qk(a.P,a.w[l])!=0&&++I;if(I!=0){klb.wf(a,21);klb.rf(a,elb.v1(I),T);for(h=0;h<a.P.f;h++){if(klb.qk(a.P,a.w[h])!=0){klb.rf(a,elb.v1(h),T);klb.rf(a,z1(klb.qk(a.P,a.w[h])>>4),2);}}}if(a.P.K){klb.ye(a,22,T,rnb,1,-1);klb.ze(a,23,T,Qnb,1,-1);klb.ze(a,24,T,wnb,2,21);}if((a.O&16)!=0){for(h=0;h<a.P.f;h++){if(klb.ol(a.P,a.w[h])){klb.wf(a,25);for(d=0;d<a.P.f;d++)klb.rf(a,z1(klb.ol(a.P,a.w[d])?1:0),1);break;}}}O=klb.Bf(a);if(O!=null){I=0;for(B=0;B<a.P.g;B++)O[a.A[B]]&&++I;klb.wf(a,26);klb.rf(a,elb.v1(I),T);for(C=0;C<a.P.g;C++)O[a.A[C]]&&klb.rf(a,elb.v1(C),T);}a.P.K&&klb.ye(a,27,T,Jmb,1,-1);I=0;for(D=0;D<a.P.g;D++)klb.Kk(a.P,a.A[D])==32&&++I;if(I!=0){klb.wf(a,28);klb.rf(a,elb.v1(I),T);for(B=0;B<a.P.g;B++)klb.Kk(a.P,a.A[B])==32&&klb.rf(a,elb.v1(B),T);}if(a.P.K){klb.ye(a,29,T,Rnb,2,30);klb.ye(a,30,T,smb,7,32);klb.ye(a,32,T,Snb,2,44);klb.ye(a,33,T,gnb,5,39);klb.ye(a,34,T,Tmb,1,-1);klb.ze(a,35,T,Tnb,1,-1);klb.ze(a,36,T,96,2,5);}I=0;for(F=0;F<a.P.g;F++)(klb.Kk(a.P,a.A[F])==8||klb.Kk(a.P,a.A[F])==16)&&++I;if(I!=0){klb.wf(a,37);klb.rf(a,elb.v1(I),T);for(w=0;w<a.P.g;w++){if(klb.Kk(a.P,a.A[w])==8||klb.Kk(a.P,a.A[w])==16){klb.rf(a,elb.v1(w),T);klb.rf(a,z1(klb.Kk(a.P,a.A[w])==8?0:1),1);}}}klb.rf(a,0,1);a.H=klb.sf(a);};klb.Pf=function Pf(a){var b,c,d,e,f,g,h,i,j,k,l;a.Z=oW(elb.EX,Nnb,6,a.P.f,15,1);for(b=0;b<a.P.f;b++){if(a.bb[b]==1||a.bb[b]==2){i=a.cb[b];if(klb.ro(a.P,b)){for(h=0;h<klb.xn(a.P,b);h++){e=klb.wn(a.P,b,h);l=0;k=oW(elb.IX,omb,6,3,15,1);for(j=0;j<klb.xn(a.P,e);j++){k[l]=klb.wn(a.P,e,j);k[l]!=b&&++l;}l==2&&a.d[k[0]]>a.d[k[1]]^a.F[k[0]]<a.F[k[1]]&&(i=!i);}}else {for(h=1;h<klb.xn(a.P,b);h++){for(j=0;j<h;j++){f=klb.wn(a.P,b,h);g=klb.wn(a.P,b,j);a.d[f]>a.d[g]&&(i=!i);a.F[f]<a.F[g]&&(i=!i);}}}a.Z[b]=a.bb[b]==1^i?1:2;}else {a.Z[b]=a.bb[b];}}a.g=oW(elb.EX,Nnb,6,a.P.g,15,1);for(c=0;c<a.P.g;c++){if(a.k[c]==1||a.k[c]==2){i=false;for(h=0;h<2;h++){d=klb.zk(a.P,h,c);if(klb.xn(a.P,d)==3){k=oW(elb.IX,omb,6,2,15,1);l=0;for(j=0;j<3;j++)klb.wn(a.P,d,j)!=klb.zk(a.P,1-h,c)&&(k[l++]=klb.wn(a.P,d,j));a.d[k[0]]>a.d[k[1]]&&(i=!i);a.F[k[0]]<a.F[k[1]]&&(i=!i);}}a.g[c]=a.k[c]==1^i?1:2;}else {a.g[c]=a.k[c];}}};klb.Qf=function Qf(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o;i=oW(elb.IX,omb,6,32,15,1);j=0;for(d=0;d<a.P.f;d++){if((a.Z[d]==1||a.Z[d]==2)&&a._[d]==b){h=a.$[d];if(i[h]<a.d[d]){i[h]==0&&++j;i[h]=a.d[d];}}}for(f=0;f<a.P.g;f++){if((a.g[f]==1||a.g[f]==2)&&a.j[f]==b&&klb.Kk(a.P,f)==1){h=a.i[f];o=$wnd.Math.max(a.d[klb.zk(a.P,0,f)],a.d[klb.zk(a.P,1,f)]);if(i[h]<o){i[h]==0&&++j;i[h]=o;}}}g=oW(elb.EX,Nnb,6,32,15,1);for(k=0;k<j;k++){m=-1;n=0;for(l=0;l<32;l++){if(n<i[l]){n=i[l];m=l;}}i[m]=0;g[m]=k<<24>>24;}for(c=0;c<a.P.f;c++)(a.Z[c]==1||a.Z[c]==2)&&a._[c]==b&&(a.$[c]=g[a.$[c]]);for(e=0;e<a.P.g;e++)(a.g[e]==1||a.g[e]==2)&&a.j[e]==b&&klb.Kk(a.P,e)==1&&(a.i[e]=g[a.i[e]]);};klb.Rf=function Rf(a,b,c){var d,e;a.S=b;for(d=0;d<a.P.f;d++){a.d[d]=c[d];a.bb[d]=0;a.gb[d]=false;}for(e=0;e<a.P.g;e++){a.k[e]=0;a.o[e]=false;}};klb.Sf=function Sf(a,b){var c,d,e;c=klb.zk(a.P,0,b);if(c>=a.P.f)return false;if(a.bb[c]==1||a.bb[c]==2)return true;if(a.bb[c]==3)return false;d=klb.dn(a.P,c);if(d!=-1)return a.k[d]==1||a.k[d]==2;for(e=0;e<klb.xn(a.P,c);e++){if(klb.zn(a.P,c,e)==2){if(a.bb[klb.wn(a.P,c,e)]==1||a.bb[klb.wn(a.P,c,e)]==2)return true;}}return false;};klb.Tf=function Tf(a,b){var c;for(c=0;c<klb.xn(a.P,b);c++)if(a.k[klb.yn(a.P,b,c)]==3&&klb.zn(a.P,b,c)==1)return true;return false;};klb.Uf=function Uf(a){var b,c,d,e,f,g,h;g=oW(elb.IX,omb,6,a.S+1,15,1);for(c=0;c<a.P.f;c++){klb.ik(a.P,c)==0&&(a.bb[c]==1?++g[a.d[c]]:a.bb[c]==2&&--g[a.d[c]]);}for(e=0;e<a.P.g;e++){if(klb.Hk(a.P,e)==1&&klb.Ek(a.P,e)==0){if(a.k[e]==1){++g[a.d[klb.zk(a.P,0,e)]];++g[a.d[klb.zk(a.P,1,e)]];}else if(a.k[e]==2){--g[a.d[klb.zk(a.P,0,e)]];--g[a.d[klb.zk(a.P,1,e)]];}}}for(h=1;h<=a.S;h++){if(g[h]!=0){f=g[h]<0;if(f){for(b=0;b<a.P.f;b++){klb.ik(a.P,b)==0&&(a.bb[b]==1?a.bb[b]=2:a.bb[b]==2&&(a.bb[b]=1));}for(d=0;d<a.P.g;d++){klb.Hk(a.P,d)==1&&klb.Ek(a.P,d)==0&&(a.k[d]==1?a.k[d]=2:a.k[d]==2&&(a.k[d]=1));}}return f;}}return false;};klb.Vf=function Vf(a,b){var c;c=oW(elb.IX,omb,6,b,15,1);flb.W5(a,0,c,0,a.length);return c;};klb.Wf=function Wf(a,b){var c;c=oW(elb.g1,Imb,6,b,16,1);flb.W5(a,0,c,0,a.length);return c;};klb.Xf=function Xf(a){var b,c;if(a.Y!=null)for(b=0;b<a.P.f;b++)klb.Il(a.P,b,a.Y[b]);if(a.f!=null)for(c=0;c<a.P.g;c++)klb.dm(a.P,c,a.f[c]);};klb.Yf=function Yf(a){var b,c,d,e,f,g,h,i,j,k,l;for(b=0;b<a.P.f;b++){if(a.bb[b]==1||a.bb[b]==2){i=false;if(klb.ro(a.P,b)){for(h=0;h<klb.xn(a.P,b);h++){e=klb.wn(a.P,b,h);l=0;k=oW(elb.IX,omb,6,3,15,1);for(j=0;j<klb.xn(a.P,e);j++){k[l]=klb.wn(a.P,e,j);k[l]!=b&&++l;}l==2&&a.d[k[0]]>a.d[k[1]]^k[0]<k[1]&&(i=!i);}}else {for(h=1;h<klb.xn(a.P,b);h++){for(j=0;j<h;j++){f=klb.wn(a.P,b,h);g=klb.wn(a.P,b,j);a.d[f]>a.d[g]&&(i=!i);f<g&&(i=!i);}}}klb.Ul(a.P,b,a.bb[b]==1^i?1:2,a.db[b]);}else {klb.Ul(a.P,b,a.bb[b],a.db[b]);}}for(c=0;c<a.P.g;c++){if(a.k[c]==1||a.k[c]==2){i=false;for(h=0;h<2;h++){d=klb.zk(a.P,h,c);if(klb.xn(a.P,d)==3){k=oW(elb.IX,omb,6,2,15,1);l=0;for(j=0;j<3;j++)klb.wn(a.P,d,j)!=klb.zk(a.P,1-h,c)&&(k[l++]=klb.wn(a.P,d,j));a.d[k[0]]>a.d[k[1]]&&(i=!i);k[0]<k[1]&&(i=!i);}}klb.hm(a.P,c,a.k[c]==1^i?1:2,a.n[c]);}else {klb.hm(a.P,c,a.k[c],a.n[c]);}}};klb.Zf=function Zf(a){var b;for(b=0;b<a.P.f;b++){klb.Yl(a.P,b,a.K[b]);}};klb.$f=function $f(a){var b,c,d;for(b=0;b<a.P.f;b++)!klb.Yk(a.P,b)&&a.bb[b]==3&&klb.Ll(a.P,b,true);for(c=0;c<a.P.g;c++){if(a.k[c]==3){d=klb.Hk(a.P,c);d==1?klb.Ll(a.P,klb.zk(a.P,0,c),true):d==2&&klb.km(a.P,c,386);}}};klb._f=function _f(a,b){return a<b?a:a-b;};klb.ag=function ag(a){klb.bg.call(this,a,0);};klb.bg=function bg(a,b){this.P=a;this.O=b;klb.Jt(this.P,7);this.b=klb.cg(this.P.f);(this.O&tnb)==0&&klb.Te(this);this.hb=(b&64)!=0||klb.Xk(this.P);if((this.O&tnb)==0){this.bb=oW(elb.EX,Nnb,6,this.P.f,15,1);this.db=oW(elb.g1,Imb,6,this.P.f,16,1);this.gb=oW(elb.g1,Imb,6,this.P.f,16,1);this.k=oW(elb.EX,Nnb,6,this.P.g,15,1);this.o=oW(elb.g1,Imb,6,this.P.g,16,1);this.n=oW(elb.g1,Imb,6,this.P.g,16,1);}klb.Xe(this);(this.O&tnb)==0&&klb.bf(this);klb.af(this);};klb.cg=function cg(a){var b;b=0;while(a>0){a>>=1;++b;}return b;};b2(28,1,{},klb.ag,klb.bg);_.b=0;_.p=false;_.r=0;_.t=0;_.u=0;_.D=false;_.G=0;_.I=false;_.J=false;_.M=0;_.O=0;_.R=0;_.S=0;_.X=false;_.hb=false;elb.$X=D3(28);klb.dg=function dg(a,b){var c,d,e,f;if(a.d!=b.d)return a.d>b.d?1:-1;e=a.a.length;f=b.a.length;c=$wnd.Math.min(e,f);for(d=0;d<c;d++){--e;--f;if(a.a[e]!=b.a[f])return a.a[e]>b.a[f]?1:-1;}if(e!=f)return e>f?1:-1;if(a.b!=b.b)return a.b>b.b?1:-1;return 0;};klb.eg=function eg(a,b){return klb.dg(a,b);};klb.fg=function fg(){};b2(239,1,{},klb.fg);_.pb=function gg(a,b){return klb.eg(a,b);};_.ib=function hg(a){return this===a;};elb.RX=D3(239);klb.ig=function ig(){};b2(142,1,{142:1},klb.ig);_.b=0;_.c=0;_.d=0;elb.QX=D3(142);klb.jg=function jg(a,b){if(a.c!=b.c)return a.c>b.c?1:-1;return 0;};klb.kg=function kg(a,b){return klb.jg(a,b);};klb.lg=function lg(){};b2(240,1,{},klb.lg);_.pb=function mg(a,b){return klb.kg(a,b);};_.ib=function ng(a){return this===a;};elb.TX=D3(240);klb.og=function og(){};b2(143,1,{143:1},klb.og);_.a=0;_.b=0;_.c=0;elb.SX=D3(143);klb.pg=function pg(a,b,c){if(a.b==0){++a.c;a.b=63;}if(a.b==63){a.d[a.c]=K1(a.d[a.c],c);a.b-=b;}else {if(a.b>=b){a.d[a.c]=L1(a.d[a.c],b);a.d[a.c]=K1(a.d[a.c],c);a.b-=b;}else {a.d[a.c]=L1(a.d[a.c],a.b);a.d[a.c]=K1(a.d[a.c],M1(c,b-a.b));b-=a.b;++a.c;a.b=63-b;a.d[a.c]=K1(a.d[a.c],n1(c,elb.v1((1<<b)-1)));}}};klb.qg=function qg(a,b){a.d[a.c]=m1(a.d[a.c],b);};klb.rg=function rg(a,b){var c;for(c=0;c<a.c;c++)if(I1(a.d[c],b.d[c]))return D1(a.d[c],b.d[c])?-1:1;return x1(a.d[a.c],b.d[a.c])?0:D1(a.d[a.c],b.d[a.c])?-1:1;};klb.sg=function sg(a,b){a.a=b;a.c=0;a.b=63;llb.fcb(a.d);};klb.tg=function tg(a){this.d=oW(elb.JX,Inb,6,a,14,1);};b2(110,1,{110:1,27:1},klb.tg);_.qb=function ug(a){return klb.rg(this,a);};_.a=0;_.b=0;_.c=0;elb.UX=D3(110);klb.vg=function vg(a,b){if(a.b!=b.b)return a.b>b.b?-1:1;if(a.c!=b.c)return a.c>b.c?-1:1;return 0;};klb.wg=function wg(a,b,c){this.b=$wnd.Math.max(a,b);this.c=$wnd.Math.min(a,b);this.a=c;};b2(141,1,{141:1,27:1},klb.wg);_.qb=function xg(a){return klb.vg(this,a);};_.a=0;_.b=0;_.c=0;elb.VX=D3(141);klb.yg=function yg(a,b,c,d){this.a=(ilb.Igb(b),llb.ccb(a,oW(elb.IX,omb,6,b,15,1),b));this.b=(ilb.Igb(d),llb.ccb(c,oW(elb.IX,omb,6,d,15,1),d));};b2(173,1,{173:1},klb.yg);elb.WX=D3(173);klb.zg=function zg(a,b){var c,d;for(d=0;d<a.g[b].length;d++){c=a.g[b][d];if(a.f[c]&&(a.o[c]==1||a.o[c]==2)&&a.k[c]==0)return true;}return false;};klb.Ag=function Ag(a,b,c){var d,e,f,g,h;e=0;g=0;for(h=0;h<a.g[b].length;h++){d=a.g[b][h];if(a.k[d]==c){f=1<<a.j[d];if((g&f)==0){g|=f;++e;}}}return e;};klb.Bg=function Bg(a){var b,c,d,e,f,g,h,i,j,k,l,m;k=new llb.pfb();for(l=0;l<a.i.f;l++){if(klb.nn(a.i,l)<2||klb.xn(a.i,l)>2){for(g=1;g<klb.xn(a.i,l);g++){b=klb.wn(a.i,l,g);for(h=0;h<g;h++){c=klb.wn(a.i,l,h);klb.Gg(a,b,c)&&(i=klb.Ng(a,b,c),!!i&&klb.Zp(i,a.f)&&llb.seb(k.a,i,(flb.U2(),T2))==null);}}}}for(m=0;m<a.i.g;m++){if(a.c[m]!=0){if(klb.Hk(a.i,m)!=2||a.c[m]!=2)continue;}b=klb.zk(a.i,0,m);c=klb.zk(a.i,1,m);klb.Gg(a,b,c)&&(i=klb.Ng(a,b,c),!!i&&klb.Zp(i,a.f)&&llb.seb(k.a,i,(flb.U2(),T2))==null);}a.g=oW(elb.IX,Lnb,7,k.a.c,0,2);a.e=oW(elb.g1,Imb,6,a.i.f,16,1);f=0;for(j=(e=new llb.Keb(new llb.Peb(new llb.cbb(k.a).a).b),new llb.kbb(e));llb.P9(j.a.a);){i=(d=llb.Ieb(j.a),d.hc());a.g[f++]=i.b;for(g=0;g<i.b.length;g++)a.e[i.b[g]]=true;}};klb.Cg=function Cg(a,b,c,d){var e,f,g,h,i,j,k;e=oW(elb.IX,omb,6,klb.xn(a.i,c),15,1);h=0;for(g=0;g<klb.xn(a.i,c);g++){e[h]=klb.wn(a.i,c,g);!d[e[h]]&&klb.Gg(a,b,e[h])&&++h;}if(h==0)return -1;if(h==1)return e[0];i=-1;j=Ylb;for(f=0;f<h;f++){k=klb.Xn(a.i,b,e[f],Ylb,d);if(k<j){j=k;i=e[f];}}return i;};klb.Dg=function Dg(a,b){return a.f[b]&&(a.o[b]==1||a.o[b]==2);};klb.Eg=function Eg(a,b){return a.e[b];};klb.Fg=function Fg(a){var b,c;c=true;for(b=0;b<a.i.f;b++){if(a.o[b]!=0&&!a.e[b]){c=false;break;}}return c;};klb.Gg=function Gg(a,b,c){var d,e,f,g,h;if(b==c)return false;if(a.a[b]!=a.a[c])return false;if(a.o[b]!=0){if(a.o[b]==3||a.o[c]==3)return false;if(a.p[b]^a.o[b]!==a.o[c])return false;if(a.k[b]!==a.k[c]||a.j[b]!==a.j[c])return false;}d=klb.un(a.i,b,c);if(d!=-1){if(klb.Hk(a.i,d)==1&&a.c[d]!=0)return false;if(klb.Hk(a.i,d)==2&&a.c[d]==1)return false;}if(klb.nn(a.i,b)==1&&!klb.oo(a.i,b)){e=-1;for(h=0;h<klb.xn(a.i,b);h++){if(klb.wn(a.i,b,h)!=c&&klb.zn(a.i,b,h)==2){e=klb.yn(a.i,b,h);break;}}f=-1;for(g=0;g<klb.xn(a.i,c);g++){if(klb.wn(a.i,c,g)!=b&&klb.zn(a.i,c,g)==2){f=klb.yn(a.i,c,g);break;}}if(e!=-1&&a.c[e]!=0&&a.d[e]^a.c[e]===a.c[f])return false;}return true;};klb.Hg=function Hg(a,b,c,d,e){var f,g,h,i,j,k;i=null;f=null;for(k=0;k<a.g[b].length;k++){g=a.g[b][k];a.f[g]&&(a.o[g]==1||a.o[g]==2)&&(a.k[g]==0?f=klb.Pg(f,(e[g]<<16)+g):a.k[g]==d&&a.j[g]==c&&(i=klb.Pg(i,(e[g]<<16)+g)));}h=klb.bh(i,f);if(h==0)return false;if(h<0){for(j=0;j<a.g[b].length;j++){g=a.g[b][j];if(a.f[g]&&(a.o[g]==1||a.o[g]==2)){if(a.k[g]==0){a.k[g]=d<<24>>24;a.j[g]=c<<24>>24;}else if(a.k[g]==d&&a.j[g]==c){a.k[g]=0;a.j[g]=-1;}}}}return true;};klb.Ig=function Ig(a,b){var c,d,e,f,g,h;if(!a.b)return false;e=false;for(f=a.b.a.length-1;f>=0;f--){d=false;g=llb.Bi(a.b,f);g.a==2?d=klb.Hg(a,g.b,g.c,g.d,b):g.a==1&&(d=klb.Mg(a,g.b,b));if(d){llb.Ji(a.b,g);for(h=0;h<a.g[g.b].length;h++){c=a.g[g.b][h];a.n[c]=false;}e=true;}}return e;};klb.Jg=function Jg(a){var b,c,d,e,f,g,h,i;if(a.g!=null){g=new klb.ah(a);a.b=new llb.Qi();for(e=0;e<a.g.length;e++){d=klb.Ug(g,e);if(d==0){klb.Qg(g,e);h=klb.Ag(a,e,2);b=klb.Ag(a,e,1);c=klb.zg(a,e);if(h==1&&b==1&&!c){klb.Lg(a,e,g.a+g.f++);llb.wi(a.b,new klb.mj(e,1,-1,-1));}if(h>0){if(c){klb.Kg(a,e,g.i+g.g++,2);++h;}llb.wi(a.b,new klb.mj(e,1,-1,-1));}else if(b>0){c&&klb.Kg(a,e,g.a+g.f++,1);llb.wi(a.b,new klb.mj(e,1,-1,-1));}else if(c){klb.Kg(a,e,g.a+g.f++,1);llb.wi(a.b,new klb.mj(e,1,-1,-1));}}else if(d==1){if(klb.zg(a,e)){f=klb.Tg(g,e);i=klb.Vg(g,e);llb.wi(a.b,new klb.mj(e,2,f,i));}else {klb.Qg(g,e);llb.wi(a.b,new klb.mj(e,1,-1,-1));}}}}};klb.Kg=function Kg(a,b,c,d){var e,f;for(f=0;f<a.g[b].length;f++){e=a.g[b][f];if(a.f[e]&&(a.o[e]==1||a.o[e]==2)&&a.k[e]==0){a.k[e]=d<<24>>24;a.j[e]=c<<24>>24;}}};klb.Lg=function Lg(a,b,c){var d,e;for(e=0;e<a.g[b].length;e++){d=a.g[b][e];if(a.k[d]==2){a.k[d]=1;a.j[d]=c<<24>>24;}}};klb.Mg=function Mg(a,b,c){var d,e,f,g,h,i,j,k;f=a.g[b];e=1;for(i=0;i<f.length;i++){d=f[i];if(a.f[d]&&a.k[d]==2){e=2;break;}}g=oW(elb.IX,Lnb,7,32,0,2);for(j=0;j<f.length;j++){d=f[j];a.f[d]&&a.k[d]==e&&(g[a.j[d]]=klb.Pg(g[a.j[d]],(c[d]<<16)+d));}for(k=0;k<32;k++)g[k]!=null&&ilb.rgb(g[k],ilb.Xgb(e2(llb.Ccb.prototype.kc,llb.Ccb,[])));llb.ucb(g,new klb.dh());if(klb.bh(g[0],g[1])==0)return false;for(h=0;h<g[0].length;h++){d=g[0][h]&fmb;a.k[d]=0;a.j[d]=-1;}return true;};klb.Ng=function Ng(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v;i=oW(elb.IX,omb,6,a.i.f,15,1);r=oW(elb.IX,omb,6,a.i.f,15,1);o=oW(elb.g1,Imb,6,a.i.f,16,1);j=oW(elb.g1,Imb,6,a.i.f,16,1);s=new klb.$p(a.i.f);i[0]=b;r[b]=c;r[c]=-2;s.a[b]=true;s.a[c]=true;f=0;k=0;while(f<=k){g=i[f];if(r[g]==g){for(l=0;l<klb.xn(a.i,g);l++){d=klb.wn(a.i,g,l);if(!s.a[d]){if(klb.zn(a.i,g,l)==2&&klb.uk(a.i,d)<10){i[++k]=d;r[d]=d;j[d]=j[g]||klb.nn(a.i,d)==2;o[d]=j[g]&&!o[g];s.a[d]=true;}else if(j[g]&&o[g]){t=klb.Cg(a,d,r[g],s.a);if(t==-1)return null;i[++k]=d;r[d]=t;r[t]=-2;j[d]=false;s.a[d]=true;s.a[t]=true;}else if(klb.Co(a.i,klb.yn(a.i,g,l))){i[++k]=d;r[d]=d;j[d]=false;s.a[d]=true;if((klb.uk(a.i,d)==6&&klb.nn(a.i,d)==0||klb.uk(a.i,d)==7&&klb.dk(a.i,d)==1||klb.uk(a.i,d)==14||klb.uk(a.i,d)==15&&klb.xn(a.i,d)>2||klb.uk(a.i,d)==16&&klb.xn(a.i,d)>2)&&klb.xn(a.i,d)>2){h=false;for(p=1;p<klb.xn(a.i,d);p++){u=klb.wn(a.i,d,p);if(!s.a[u]){for(q=0;q<p;q++){v=klb.wn(a.i,d,q);if(!s.a[v]){if(klb.Gg(a,u,v)){i[++k]=u;r[u]=v;r[v]=-2;j[u]=false;s.a[u]=true;s.a[v]=true;h=true;}}}}}if(!h)return null;}}}}}else {e=oW(elb.g1,Imb,6,klb.xn(a.i,g),16,1);for(m=0;m<klb.xn(a.i,g);m++){d=klb.wn(a.i,g,m);if(s.a[d]){e[m]=r[d]==d;}else {for(p=0;p<klb.xn(a.i,d);p++){if(klb.wn(a.i,d,p)==r[g]){e[m]=true;break;}}}}for(n=0;n<klb.xn(a.i,g);n++){if(e[n]){d=klb.wn(a.i,g,n);if(s.a[d]){if(klb.un(a.i,d,r[g])==-1)return null;}else {i[++k]=d;r[d]=d;o[d]=false;j[d]=true;s.a[d]=true;}}}for(l=0;l<klb.xn(a.i,g);l++){if(!e[l]){d=klb.wn(a.i,g,l);if(!s.a[d]){t=klb.Cg(a,d,r[g],s.a);if(t==-1)return null;i[++k]=d;r[d]=t;r[t]=-2;j[d]=false;s.a[d]=true;s.a[t]=true;}}}}++f;}return s;};klb.Og=function Og(a,b,c,d,e,f,g,h,i,j){this.i=a;this.a=b;this.f=c;this.o=d;this.c=e;this.k=f;this.j=g;this.p=h;this.d=i;this.n=j;klb.Bg(this);};klb.Pg=function Pg(a,b){var c,d;d=oW(elb.IX,omb,6,a==null?1:a.length+1,15,1);for(c=0;c<d.length-1;c++)d[c]=a[c];d[d.length-1]=b;return d;};b2(264,1,{},klb.Og);elb.YX=D3(264);klb.Qg=function Qg(a,b){var c,d,e,f,g,h;for(e=0;e<a.b;e++){if(a.e[e][b]&&a.c[e]!=-3){for(d=0;d<=a.j.g.length;d++){if(d!=b&&a.e[e][d]){a.e[e][b]=false;h=e<a.a?e:e<a.b?e-a.a:-1;g=klb.$g(a,e<a.a?1:e<a.b?2:0);for(f=0;f<a.j.g[b].length;f++){c=a.j.g[b][f];klb.Dg(a.j,c)&&a.j.j[c]==h&&(a.j.j[c]=g<<24>>24);}}}}}};klb.Rg=function Rg(a,b,c){var d,e,f,g,h;h=false;g=1;b[c]=g;d=true;while(d){d=false;for(e=0;e<a.b;e++){if(b[e]==g){for(f=0;f<a.b;f++){if(b[f]==0&&klb.Zg(a,e,f)){if(a.c[f]==-2){b[f]=g+1;d=true;}else if(a.c[f]!=a.c[c]){b[f]=g+1;h=true;}}}}}++g;}return h;};klb.Sg=function Sg(a,b,c,d){var e,f,g,h,i,j,k,l,m,n,o;for(i=d+1;i<a.j.g.length;i++){if(i!=d&&a.e[b][i]&&a.e[c][i]){g=oW(elb.IX,omb,6,2,15,1);g[0]=c;g[1]=b;return g;}}o=oW(elb.IX,omb,6,a.b,15,1);k=oW(elb.IX,omb,6,a.b,15,1);j=oW(elb.IX,omb,6,a.b,15,1);f=0;l=0;j[0]=b;k[b]=1;while(f<=l){for(m=0;m<a.d[j[f]].length;m++){e=a.d[j[f]][m];if(e==c){if(f==0)continue;h=k[j[f]]+1;g=oW(elb.IX,omb,6,h,15,1);g[0]=e;g[1]=j[f];for(n=2;n<h;n++)g[n]=o[g[n-1]];return g;}if(k[e]==0&&a.c[e]!=-3){k[e]=k[j[f]]+1;j[++l]=e;o[e]=j[f];}}++f;}return null;};klb.Tg=function Tg(a,b){var c;for(c=0;c<a.b;c++)if(a.e[c][b]&&a.c[c]==-3)return c<a.a?c:c<a.b?c-a.a:-1;return -1;};klb.Ug=function Ug(a,b){var c,d;c=0;for(d=0;d<a.b;d++)a.e[d][b]&&a.c[d]==-3&&++c;return c;};klb.Vg=function Vg(a,b){var c;for(c=0;c<a.b;c++)if(a.e[c][b]&&a.c[c]==-3)return c<a.a?1:c<a.b?2:0;return -1;};klb.Wg=function Wg(a,b){return b<a.a?b:b<a.b?b-a.a:-1;};klb.Xg=function Xg(a,b){return b<a.a?1:b<a.b?2:0;};klb.Yg=function Yg(a,b){var c,d;d=a.j.k[b];c=a.j.j[b];return d==0?a.b:d==1?c:a.a+c;};klb.Zg=function Zg(a,b,c){var d;for(d=0;d<a.j.g.length;d++)if(a.e[b][d]&&a.e[c][d])return true;return false;};klb.$g=function $g(a,b){return b==1?a.a+a.f++:a.i+a.g++;};klb._g=function _g(a,b){var c,d,e,f,g,h,i,j,k,l;k=Ylb;i=-1;l=-1;j=-1;for(d=0;d<a.j.i.f;d++){if(klb.Dg(a.j,d)&&a.j.k[d]!=0){for(h=0;h<b.length;h++){e=klb.Wg(a,b[h]);f=klb.Xg(a,b[h]);if(a.j.k[d]==f&&a.j.j[d]==e){if(k>a.j.a[d]+(f==1?emb:0)){k=a.j.a[d]+(f==1?emb:0);i=e;l=f;j=b[h];}}}}}for(c=0;c<a.j.i.f;c++){if(klb.Dg(a.j,c)&&a.j.k[c]==l&&a.j.j[c]==i){a.j.k[c]=0;a.j.j[c]=-1;}}for(g=0;g<a.j.g.length;g++)a.e[j][g]=false;};klb.ah=function ah(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q;this.j=a;for(d=0;d<a.i.f;d++){a.f[d]&&(a.o[d]==1||a.o[d]==2)&&(a.k[d]==1?this.a<=a.j[d]&&(this.a=1+a.j[d]):a.k[d]==2&&this.i<=a.j[d]&&(this.i=1+a.j[d]));}this.b=this.a+this.i;this.e=mW(elb.g1,[Tlb,Imb],[11,6],16,[this.b+1,a.g.length+1],2);for(e=0;e<a.i.f;e++)a.f[e]&&(a.o[e]==1||a.o[e]==2)&&!a.e[e]&&(this.e[klb.Yg(this,e)][a.g.length]=true);for(i=0;i<a.g.length;i++){for(q=0;q<a.g[i].length;q++){c=a.g[i][q];a.f[c]&&(a.o[c]==1||a.o[c]==2)&&(this.e[klb.Yg(this,c)][i]=true);}}this.d=oW(elb.IX,Lnb,7,this.b,0,2);for(j=0;j<a.g.length;j++){for(n=1;n<this.b;n++){if(this.e[n][j]){for(o=0;o<n;o++){if(this.e[o][j]){this.d[n]=klb.Pg(this.d[n],o);this.d[o]=klb.Pg(this.d[o],n);}}}}}this.c=oW(elb.IX,omb,6,this.b+1,15,1);for(m=0;m<this.b;m++){this.e[m][a.g.length]?this.c[m]=-1:this.c[m]=-2;}for(k=0;k<a.g.length;k++){if(this.e[this.b][k]){for(l=0;l<this.b;l++){this.e[l][k]&&this.c[l]!=k&&(this.c[l]==-2?this.c[l]=k:this.c[l]=-3);}}}for(b=0;b<this.b;b++){if(this.c[b]>=-1){f=oW(elb.IX,omb,6,this.b,15,1);if(klb.Rg(this,f,b)){for(l=0;l<this.b;l++){f[l]!=0&&(this.c[l]=-3);}}}}for(h=0;h<a.g.length-1;h++){for(n=1;n<this.b;n++){if(this.e[n][h]&&this.c[n]!=-3){for(o=0;o<n;o++){if(this.e[o][h]&&this.c[o]!=-3){g=klb.Sg(this,n,o,h);if(g!=null){for(p=0;p<g.length;p++)this.c[g[p]]=-3;klb._g(this,g);break;}}}}}}};b2(265,1,{},klb.ah);_.a=0;_.b=0;_.f=0;_.g=0;_.i=0;elb.XX=D3(265);klb.bh=function bh(a,b){var c,d;if(a==null)return b==null?0:1;if(b==null)return -1;c=$wnd.Math.min(a.length,b.length);for(d=0;d<c;d++)if((a[d]&Unb)!=(b[d]&Unb))return (a[d]&Unb)<(b[d]&Unb)?-1:1;return a.length==b.length?0:a.length<b.length?-1:1;};klb.dh=function dh(){};b2(266,1,{},klb.dh);_.pb=function eh(a,b){return klb.bh(a,b);};_.ib=function fh(a){return this===a;};elb.ZX=D3(266);klb.gh=function gh(a,b){switch(b){case 0:return klb.hh(a,false);case 1:return klb.jh(a);case 2:return klb.ih(a);case 3:return klb.lh(a);case 4:return klb.kh(a);}return null;};klb.hh=function hh(b,c){try{if(!c)return klb.Gf(new klb.ag(b));b=klb.Nt(b);klb.Vo(b,true);klb.kq(b);return klb.Gf(new klb.ag(b));}catch(a){a=k1(a);if(uX(a,22)){flb.V5();return null;}else throw l1(a);}};klb.ih=function ih(b){var c,d;try{b=klb.Nt(b);klb.Xt(b);c=b.r;for(d=0;d<c;d++){b.H[d]=1;b.T=0;}return klb.Gf(new klb.ag(b));}catch(a){a=k1(a);if(uX(a,22)){flb.V5();return null;}else throw l1(a);}};klb.jh=function jh(b){try{b=klb.Nt(b);klb.Xt(b);return klb.Gf(new klb.ag(b));}catch(a){a=k1(a);if(uX(a,22)){flb.V5();return null;}else throw l1(a);}};klb.kh=function kh(b){var c;try{b=klb.Nt(b);klb.Xt(b);c=klb.nu(new klb.qu(b));return klb.Gf(new klb.bg(c,8));}catch(a){a=k1(a);if(uX(a,22)){flb.V5();return null;}else throw l1(a);}};klb.lh=function lh(b){var c;try{c=klb.nu(new klb.qu(b));return klb.Gf(new klb.bg(c,8));}catch(a){a=k1(a);if(uX(a,22)){flb.V5();return null;}else throw l1(a);}};klb.mh=function mh(a,b,c,d){a.a+=b;a.b+=c;a.c+=d;};klb.nh=function nh(a,b){a.a+=b.a;a.b+=b.b;a.c+=b.c;return a;};klb.oh=function oh(a,b){return new klb.Dh(a.a+b.a,a.b+b.b,a.c+b.c);};klb.ph=function ph(a,b){a.a=(a.a+b.a)/2;a.b=(a.b+b.b)/2;a.c=(a.c+b.c)/2;return a;};klb.qh=function qh(a,b){if(a.a!=b.a)return a.a<b.a?-1:1;if(a.b!=b.b)return a.b<b.b?-1:1;if(a.c!=b.c)return a.c<b.c?-1:1;return 0;};klb.rh=function rh(a,b){return new klb.Dh(a.b*b.c-a.c*b.b,-(a.a*b.c-a.c*b.a),a.a*b.b-a.b*b.a);};klb.sh=function sh(a,b){return $wnd.Math.sqrt((b.a-a.a)*(b.a-a.a)+(b.b-a.b)*(b.b-a.b)+(b.c-a.c)*(b.c-a.c));};klb.th=function th(a,b){var c;if(b==null||!uX(b,24))return false;c=b;return $wnd.Math.abs(c.a-a.a)+$wnd.Math.abs(c.b-a.b)+$wnd.Math.abs(c.c-a.c)<1.0E-6;};klb.uh=function uh(a,b){var c,d,e;d=a.a*a.a+a.b*a.b+a.c*a.c;e=b.a*b.a+b.b*b.b+b.c*b.c;if(d==0||e==0)return 0;c=(a.a*b.a+a.b*b.b+a.c*b.c)/$wnd.Math.sqrt(d*e);if(c>=1)return 0;if(c<=-1)return Nmb;return $wnd.Math.acos(c);};klb.vh=function vh(a,b){var c,d,e;c=a.a;d=a.b;e=a.c;a.a=c*b[0][0]+d*b[1][0]+e*b[2][0];a.b=c*b[0][1]+d*b[1][1]+e*b[2][1];a.c=c*b[0][2]+d*b[1][2]+e*b[2][2];return a;};klb.wh=function wh(a,b){a.a*=b;a.b*=b;a.c*=b;return a;};klb.xh=function xh(a,b,c,d){a.a=b;a.b=c;a.c=d;};klb.yh=function yh(a,b){klb.xh(a,b.a,b.b,b.c);return a;};klb.zh=function zh(a,b){a.a-=b.a;a.b-=b.b;a.c-=b.c;return a;};klb.Ah=function Ah(a,b){return new klb.Dh(a.a-b.a,a.b-b.b,a.c-b.c);};klb.Bh=function Bh(a){var b;b=$wnd.Math.sqrt(a.a*a.a+a.b*a.b+a.c*a.c);if(b==0){flb.V5();a.a=1;a.b=0;a.c=0;return a;}a.a/=b;a.b/=b;a.c/=b;return a;};klb.Ch=function Ch(){};klb.Dh=function Dh(a,b,c){this.a=a;this.b=b;this.c=c;};klb.Eh=function Eh(a){klb.Dh.call(this,a.a,a.b,a.c);};b2(24,1,{24:1,4:1,27:1},klb.Ch,klb.Dh,klb.Eh);_.qb=function Fh(a){return klb.qh(this,a);};_.ib=function Gh(a){return klb.th(this,a);};_.lb=function Hh(){var a;a=new Llb.y8('0.00');return '['+Llb.w8(a,this.a)+', '+Llb.w8(a,this.b)+', '+Llb.w8(a,this.c)+']';};_.a=0;_.b=0;_.c=0;elb._X=D3(24);klb.Ih=function Ih(a,b){b.c*=a.c;b.a=b.a*a.c+a.a;b.b=b.b*a.c+a.b;};klb.Jh=function Jh(a,b){b.a=b.a*a.c+a.a;b.b=b.b*a.c+a.b;};klb.Kh=function Kh(a,b){b.c=b.c*a.c+a.a;b.d=b.d*a.c+a.b;b.b*=a.c;b.a*=a.c;};klb.Lh=function Lh(a){a.a=0;a.b=0;a.c=1;};klb.Mh=function Mh(a,b){return b*a.c+a.a;};klb.Nh=function Nh(a,b){return b*a.c+a.b;};klb.Oh=function Oh(){klb.Lh(this);};klb.Ph=function Ph(a,b,c){var d,e,f,g;klb.Lh(this);e=b.b/a.b;g=b.a/a.a;f=0;f==0&&(f=24);d=f/c;this.c=$wnd.Math.min(d,$wnd.Math.min(e,g));this.a=b.c+b.b/2-this.c*(a.c+a.b/2);this.b=b.d+b.a/2-this.c*(a.d+a.a/2);};b2(144,1,{},klb.Oh,klb.Ph);_.lb=function Qh(){return 'DepictorTransformation Offset: '+this.a+','+this.b+' Scaling: '+this.c;};_.a=0;_.b=0;_.c=0;elb.aY=D3(144);klb.Rh=function Rh(a){var b,c,d,e;if(!flb.j5(ilb.Xgb(a).substr(0,21),Fnb)||(d=ilb.Xgb(Gnb).length,!flb.j5(ilb.Xgb(a).substr(ilb.Xgb(a).length-d,d),Gnb)))return null;c=flb.o5(a,z5(34),21);if(c==-1)return null;e=(ilb.Rgb(21,c,ilb.Xgb(a).length),ilb.Xgb(a).substr(21,c-21));b=flb.w5(a,21+ilb.Xgb(e).length+1,ilb.Xgb(a).length-17);if(flb.j5(e,'arrow'))return new vlb.cF(b);if(flb.j5(e,'text'))return new klb.Fu(b);return null;};llb.Sh=function Sh(a,b,c){var d,e;for(e=a.rb();e.ec();){d=e.fc();if(CX(b)===CX(d)||b!=null&&flb.Gc(b,d)){c&&e.gc();return true;}}return false;};llb.Th=function Th(a,b){var c,d;ilb.Mgb(b);for(d=b.rb();d.ec();){c=d.fc();if(!a.contains(c)){return false;}}return true;};llb.Uh=function Uh(a,b){var c,d,e;ilb.Mgb(b);c=false;for(d=a.rb();d.ec();){e=d.fc();if(b.contains(e)){d.gc();c=true;}}return c;};llb.Vh=function Vh(a){var b,c,d;d=new llb.ieb('[',']');for(c=a.rb();c.ec();){b=c.fc();llb.geb(d,CX(b)===CX(a)?'(this Collection)':b==null?$lb:i2(b));}return !d.a?d.c:ilb.Xgb(d.e).length==0?d.a.a:d.a.a+(''+d.e);};b2(39,1,Vnb);_.add=function Wh(a){throw l1(new flb.Z5('Add not supported on this collection'));};_.addAll=function Xh(a){var b,c,d;ilb.Mgb(a);b=false;for(d=a.rb();d.ec();){c=d.fc();b=b|this.add(c);}return b;};_.clear=function Yh(){var a;for(a=this.rb();a.ec();){a.fc();a.gc();}};_.contains=function Zh(a){return llb.Sh(this,a,false);};_.containsAll=function $h(a){return llb.Th(this,a);};_.isEmpty=function _h(){return this.size()==0;};_.remove=function ai(a){return llb.Sh(this,a,true);};_.removeAll=function bi(a){return llb.Uh(this,a);};_.retainAll=function ci(a){var b,c,d;ilb.Mgb(a);b=false;for(c=this.rb();c.ec();){d=c.fc();if(!a.contains(d)){c.gc();b=true;}}return b;};_.toArray=function di(){return this.sb(oW(elb.o_,amb,1,this.size(),5,1));};_.sb=function ei(a){var b,c,d,e;e=this.size();a.length<e&&(a=ilb.sgb(new Array(e),a));d=a;c=this.rb();for(b=0;b<e;++b){sW(d,b,c.fc());}a.length>e&&elb.rW(a,e,null);return a;};_.lb=function fi(){return llb.Vh(this);};elb.D_=D3(39);llb.gi=function gi(a,b,c){return new llb.Z9(a,b,c);};b2(80,39,Wnb);_.addAtIndex=function hi(a,b){throw l1(new flb.Z5('Add not supported on this list'));};_.add=function ii(a){this.addAtIndex(this.size(),a);return true;};_.addAllAtIndex=function ji(a,b){var c,d,e;ilb.Mgb(b);c=false;for(e=b.rb();e.ec();){d=e.fc();this.addAtIndex(a++,d);c=true;}return c;};_.clear=function ki(){this.tb(0,this.size());};_.ib=function li(a){var b,c,d,e,f;if(a===this){return true;}if(!uX(a,98)){return false;}f=a;if(this.size()!=f.size()){return false;}e=f.rb();for(c=this.rb();c.ec();){b=c.fc();d=e.fc();if(!(CX(b)===CX(d)||b!=null&&flb.Gc(b,d))){return false;}}return true;};_.kb=function mi(){return llb.Fcb(this);};_.indexOf=function ni(a){var b,c;for(b=0,c=this.size();b<c;++b){if(llb.Wdb(a,this.getAtIndex(b))){return b;}}return -1;};_.rb=function oi(){return new llb.S9(this);};_.lastIndexOf=function pi(a){var b;for(b=this.size()-1;b>-1;--b){if(llb.Wdb(a,this.getAtIndex(b))){return b;}}return -1;};_.removeAtIndex=function qi(a){throw l1(new flb.Z5('Remove not supported on this list'));};_.tb=function ri(a,b){var c,d;d=new llb.X9(this,a);for(c=a;c<b;++c){ilb.Kgb(d.a<d.c.size());d.c.getAtIndex(d.b=d.a++);llb.R9(d);}};_.setAtIndex=function si(a,b){throw l1(new flb.Z5('Set not supported on this list'));};_.subList=function ti(a,b){return llb.gi(this,a,b);};elb.K_=D3(80);function R3(a){if(a==null){return false;}return a.$implements__java_lang_Cloneable||Array.isArray(a);}llb.ui=function ui(a){a.a=oW(elb.o_,amb,1,0,5,1);};llb.vi=function vi(a,b,c){ilb.Ogb(b,a.a.length);ilb.lgb(a.a,b,c);};llb.wi=function wi(a,b){ilb.ogb(a.a,b);return true;};llb.xi=function xi(a,b,c){var d,e;ilb.Ogb(b,a.a.length);d=c.toArray();e=d.length;if(e==0){return false;}ilb.mgb(a.a,b,d);return true;};llb.yi=function yi(a,b){var c,d;c=b.toArray();d=c.length;if(d==0){return false;}ilb.mgb(a.a,a.a.length,c);return true;};llb.zi=function zi(a){ilb.qgb(a.a,0);};llb.Ai=function Ai(a,b){return llb.Di(a,b,0)!=-1;};llb.Bi=function Bi(a,b){ilb.Lgb(b,a.a.length);return a.a[b];};llb.Ci=function Ci(a,b){return llb.Di(a,b,0);};llb.Di=function Di(a,b,c){for(;c<a.a.length;++c){if(llb.Wdb(b,a.a[c])){return c;}}return -1;};llb.Ei=function Ei(a){return a.a.length==0;};llb.Fi=function Fi(a){return new llb.Wbb(a);};llb.Gi=function Gi(a,b){return llb.Hi(a,b,a.a.length-1);};llb.Hi=function Hi(a,b,c){for(;c>=0;--c){if(llb.Wdb(b,a.a[c])){return c;}}return -1;};llb.Ii=function Ii(a,b){var c;c=(ilb.Lgb(b,a.a.length),a.a[b]);ilb.pgb(a.a,b,1);return c;};llb.Ji=function Ji(a,b){var c;c=llb.Di(a,b,0);if(c==-1){return false;}llb.Ii(a,c);return true;};llb.Ki=function Ki(a,b,c){var d;ilb.Pgb(b,c,a.a.length);d=c-b;ilb.pgb(a.a,b,d);};llb.Li=function Li(a,b,c){var d;d=(ilb.Lgb(b,a.a.length),a.a[b]);a.a[b]=c;return d;};llb.Mi=function Mi(a){return a.a.length;};llb.Ni=function Ni(a,b){llb.tcb(a.a,a.a.length,b);};llb.Oi=function Oi(a){return ilb.igb(a.a);};llb.Pi=function Pi(a,b){var c,d;d=a.a.length;b.length<d&&(b=ilb.sgb(new Array(d),b));for(c=0;c<d;++c){sW(b,c,a.a[c]);}b.length>d&&elb.rW(b,d,null);return b;};llb.Qi=function Qi(){llb.ui(this);};llb.Ri=function Ri(a){llb.ui(this);ilb.Ggb(a>=0,'Initial capacity must not be negative');};b2(18,80,Xnb,llb.Qi,llb.Ri);_.addAtIndex=function Si(a,b){llb.vi(this,a,b);};_.add=function Ti(a){return llb.wi(this,a);};_.addAllAtIndex=function Ui(a,b){return llb.xi(this,a,b);};_.addAll=function Vi(a){return llb.yi(this,a);};_.clear=function Wi(){llb.zi(this);};_.contains=function Xi(a){return llb.Ai(this,a);};_.getAtIndex=function Yi(a){return llb.Bi(this,a);};_.indexOf=function Zi(a){return llb.Ci(this,a);};_.isEmpty=function $i(){return llb.Ei(this);};_.rb=function _i(){return llb.Fi(this);};_.lastIndexOf=function aj(a){return llb.Gi(this,a);};_.removeAtIndex=function bj(a){return llb.Ii(this,a);};_.remove=function cj(a){return llb.Ji(this,a);};_.tb=function dj(a,b){llb.Ki(this,a,b);};_.setAtIndex=function ej(a,b){return llb.Li(this,a,b);};_.size=function fj(){return llb.Mi(this);};_.toArray=function gj(){return llb.Oi(this);};_.sb=function hj(a){return llb.Pi(this,a);};elb.__=D3(18);klb.ij=function ij(a){var b,c;c=new flb.G5();for(b=0;b<a.a.length;b++)flb.F5(c,klb.Td((ilb.Lgb(b,a.a.length),a.a[b]))+Slb);return c.a;};klb.jj=function jj(b){var c;llb.Qi.call(this);try{if(b){for(c=0;c<b.a.length;c++)llb.wi(this,(ilb.Lgb(c,b.a.length),b.a[c]).mb());}}catch(a){a=k1(a);if(!uX(a,15))throw l1(a);}};klb.kj=function kj(a){var b,c,d;llb.Qi.call(this);if(a==null||ilb.Xgb(a).length==0)return;b=0;c=flb.n5(a,z5(10));while(c!=-1){d=klb.Rh((ilb.Rgb(b,c,ilb.Xgb(a).length),ilb.Xgb(a).substr(b,c-b)));!!d&&(ilb.ogb(this.a,d),true);b=c+1;c=flb.o5(a,z5(10),b);}};b2(166,18,Xnb,klb.jj,klb.kj);_.lb=function lj(){return klb.ij(this);};elb.bY=D3(166);klb.mj=function mj(a,b,c,d){this.b=a;this.a=b;this.c=c;this.d=d;};b2(83,1,{83:1},klb.mj);_.a=0;_.b=0;_.c=0;_.d=0;elb.cY=D3(83);klb.nj=function nj(a){var b,c,d,e,f,g;if(a.j!=0)return a.j;if(a.i&&klb.uk(a.e,a.a)!=15&&klb.uk(a.e,a.a)!=16){for(g=0;g<klb.hn(a.e,a.a);g++){f=klb.yn(a.e,a.a,g);if(klb.rl(a.e,f,a.a)){klb.wn(a.e,a.a,g)==a.b?a.j=klb.Kk(a.e,f)==257?3:1:a.j=klb.Kk(a.e,f)==257?1:3;return a.j;}}}b=klb.yk(a.e,a.a,a.g);d=klb.yk(a.e,a.a,a.b);d<b&&(d+=Mmb);if(klb.hn(a.e,a.a)==2){c=d-b;if(c>3.0915926535897933&&c<3.191592653589793){a.j=-1;return a.j;}a.j=c<Nmb?4:2;return a.j;}else {e=klb.yk(a.e,a.a,a.d);e<b&&(e+=Mmb);a.j=e<d?2:4;return a.j;}};klb.oj=function oj(a,b,c,d){var e,f,g,h;this.e=a;this.g=c;this.a=d;g=-1;for(h=0;h<klb.hn(this.e,this.a);h++){e=klb.wn(this.e,this.a,h);f=klb.yn(this.e,this.a,h);if(e==this.g){klb.Kk(this.e,f)==386&&(this.j=-1);continue;}if(klb.rl(this.e,f,this.a)){this.i&&(a.u[d]|=emb);this.i=true;}if(g==b[e]){this.d=e;this.f=true;this.c=klb.Co(this.e,f);}else if(g<b[e]){g=b[e];this.d=this.b;this.b=e;}else {this.d=e;}}};b2(96,1,{},klb.oj);_.a=0;_.b=0;_.c=false;_.d=0;_.f=false;_.g=0;_.i=false;_.j=0;elb.dY=D3(96);klb.wj=function wj(){klb.wj=d2;klb.qj=wW(iW(elb.u_,1),dmb,2,6,['?','H','He','Li','Be','B','C','N','O','F','Ne','Na','Mg','Al','Si','P','S','Cl','Ar','K','Ca','Sc','Ti','V','Cr','Mn','Fe','Co','Ni','Cu','Zn','Ga','Ge','As','Se','Br','Kr','Rb','Sr','Y','Zr','Nb','Mo','Tc','Ru','Rh','Pd','Ag','Cd','In','Sn','Sb','Te','I','Xe','Cs','Ba','La','Ce','Pr','Nd','Pm','Sm','Eu','Gd','Tb','Dy','Ho','Er','Tm','Yb','Lu','Hf','Ta','W','Re','Os','Ir','Pt','Au','Hg','Tl','Pb','Bi','Po','At','Rn','Fr','Ra','Ac','Th','Pa','U','Np','Pu','Am','Cm','Bk','Cf','Es','Fm','Md','No','Lr','Rf','Db','Sg','Bh','Hs','Mt','Ds','Rg','Cn','Nh','Fl','Mc','Lv','Ts','Og',Ynb,Ynb,Ynb,Ynb,Ynb,Ynb,Ynb,Ynb,Ynb,Ynb,'R4','R5','R6','R7','R8','R9','R10','R11','R12','R13','R14','R15','R16','R1','R2','R3','A','A1','A2','A3',Ynb,Ynb,'D','T','X','R','H2','H+','Nnn','HYD','Pol',Ynb,Ynb,Ynb,Ynb,Ynb,Ynb,Ynb,Ynb,Ynb,Ynb,Ynb,'Ala','Arg','Asn','Asp','Cys','Gln','Glu','Gly','His','Ile','Leu','Lys','Met','Phe','Pro','Ser','Thr','Trp','Tyr','Val']);klb.uj=wW(iW(elb.f1,1),Hnb,6,15,[0,1,4,7,9,11,12,14,16,19,20,23,24,27,28,31,32,35,40,39,40,45,48,51,52,55,56,59,58,63,64,69,74,75,80,79,84,85,88,89,90,93,98,0,102,103,106,107,114,115,120,121,130,127,132,133,138,139,140,141,142,0,152,153,158,159,164,165,166,169,174,175,180,181,184,187,192,193,195,197,202,205,208,209,209,210,222,223,226,227,232,231,238,237,244,243,247,247,251,252,257,258,259,262,267,268,271,270,277,276,281,281,283,285,289,289,293,294,294,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,71,156,114,115,103,128,129,57,137,113,113,128,131,147,97,87,101,186,163,99]);klb.tj=wW(iW(elb.EX,1),Nnb,6,15,[6]);klb.pj=wW(iW(elb.EX,1),Nnb,6,15,[2]);klb.rj=wW(iW(elb.EX,2),Tlb,9,0,[null,wW(iW(elb.EX,1),Nnb,6,15,[1]),wW(iW(elb.EX,1),Nnb,6,15,[0]),wW(iW(elb.EX,1),Nnb,6,15,[1]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[4]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[1]),wW(iW(elb.EX,1),Nnb,6,15,[0]),wW(iW(elb.EX,1),Nnb,6,15,[1]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[4]),wW(iW(elb.EX,1),Nnb,6,15,[3,5]),wW(iW(elb.EX,1),Nnb,6,15,[2,4,6]),wW(iW(elb.EX,1),Nnb,6,15,[1,3,5,7]),wW(iW(elb.EX,1),Nnb,6,15,[0]),wW(iW(elb.EX,1),Nnb,6,15,[1]),wW(iW(elb.EX,1),Nnb,6,15,[2]),null,null,null,null,null,null,null,null,null,null,wW(iW(elb.EX,1),Nnb,6,15,[2,3]),wW(iW(elb.EX,1),Nnb,6,15,[2,4]),wW(iW(elb.EX,1),Nnb,6,15,[3,5]),wW(iW(elb.EX,1),Nnb,6,15,[2,4,6]),wW(iW(elb.EX,1),Nnb,6,15,[1,3,5,7]),wW(iW(elb.EX,1),Nnb,6,15,[0,2]),wW(iW(elb.EX,1),Nnb,6,15,[1]),wW(iW(elb.EX,1),Nnb,6,15,[2]),null,null,null,null,null,null,null,null,null,null,wW(iW(elb.EX,1),Nnb,6,15,[1,2,3]),wW(iW(elb.EX,1),Nnb,6,15,[2,4]),wW(iW(elb.EX,1),Nnb,6,15,[3,5]),wW(iW(elb.EX,1),Nnb,6,15,[2,4,6]),wW(iW(elb.EX,1),Nnb,6,15,[1,3,5,7]),wW(iW(elb.EX,1),Nnb,6,15,[0,2,4,6]),wW(iW(elb.EX,1),Nnb,6,15,[1]),wW(iW(elb.EX,1),Nnb,6,15,[2]),null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,null,wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[2])]);klb.sj=wW(iW(elb.EX,2),Tlb,9,0,[null,wW(iW(elb.EX,1),Nnb,6,15,[1]),null,wW(iW(elb.EX,1),Nnb,6,15,[1]),wW(iW(elb.EX,1),Nnb,6,15,[2]),null,null,wW(iW(elb.EX,1),Nnb,6,15,[-3]),wW(iW(elb.EX,1),Nnb,6,15,[-2]),wW(iW(elb.EX,1),Nnb,6,15,[-1]),null,wW(iW(elb.EX,1),Nnb,6,15,[1]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[3]),null,wW(iW(elb.EX,1),Nnb,6,15,[-3]),wW(iW(elb.EX,1),Nnb,6,15,[-2]),wW(iW(elb.EX,1),Nnb,6,15,[-1]),null,wW(iW(elb.EX,1),Nnb,6,15,[1]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[2,3,4]),wW(iW(elb.EX,1),Nnb,6,15,[2,3,4,5]),wW(iW(elb.EX,1),Nnb,6,15,[2,3,6]),wW(iW(elb.EX,1),Nnb,6,15,[2,3,4,7]),wW(iW(elb.EX,1),Nnb,6,15,[2,3]),wW(iW(elb.EX,1),Nnb,6,15,[2,3]),wW(iW(elb.EX,1),Nnb,6,15,[2,3]),wW(iW(elb.EX,1),Nnb,6,15,[1,2]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[2,4]),wW(iW(elb.EX,1),Nnb,6,15,[-3,3,5]),wW(iW(elb.EX,1),Nnb,6,15,[-2]),wW(iW(elb.EX,1),Nnb,6,15,[-1]),null,wW(iW(elb.EX,1),Nnb,6,15,[1]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[4]),wW(iW(elb.EX,1),Nnb,6,15,[3,5]),wW(iW(elb.EX,1),Nnb,6,15,[6]),wW(iW(elb.EX,1),Nnb,6,15,[4,6,7]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[2,4]),wW(iW(elb.EX,1),Nnb,6,15,[1]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[2,4]),wW(iW(elb.EX,1),Nnb,6,15,[-3,3,5]),wW(iW(elb.EX,1),Nnb,6,15,[-2,4,6]),wW(iW(elb.EX,1),Nnb,6,15,[-1]),null,wW(iW(elb.EX,1),Nnb,6,15,[1]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[3,4]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[2,3]),wW(iW(elb.EX,1),Nnb,6,15,[2,3]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[2,3]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[4]),wW(iW(elb.EX,1),Nnb,6,15,[5]),wW(iW(elb.EX,1),Nnb,6,15,[6]),wW(iW(elb.EX,1),Nnb,6,15,[4,6,7]),wW(iW(elb.EX,1),Nnb,6,15,[3,4]),wW(iW(elb.EX,1),Nnb,6,15,[3,4]),wW(iW(elb.EX,1),Nnb,6,15,[2,4]),wW(iW(elb.EX,1),Nnb,6,15,[1,3]),wW(iW(elb.EX,1),Nnb,6,15,[1,2]),wW(iW(elb.EX,1),Nnb,6,15,[1,3]),wW(iW(elb.EX,1),Nnb,6,15,[2,4]),wW(iW(elb.EX,1),Nnb,6,15,[3,5]),wW(iW(elb.EX,1),Nnb,6,15,[-2,2,4]),wW(iW(elb.EX,1),Nnb,6,15,[-1,1]),null,wW(iW(elb.EX,1),Nnb,6,15,[1]),wW(iW(elb.EX,1),Nnb,6,15,[2]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[4]),wW(iW(elb.EX,1),Nnb,6,15,[4,5]),wW(iW(elb.EX,1),Nnb,6,15,[3,4,5,6]),wW(iW(elb.EX,1),Nnb,6,15,[3,4,5,6]),wW(iW(elb.EX,1),Nnb,6,15,[3,4,5,6]),wW(iW(elb.EX,1),Nnb,6,15,[3,4,5,6]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[3,4]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[3]),wW(iW(elb.EX,1),Nnb,6,15,[2,3]),wW(iW(elb.EX,1),Nnb,6,15,[2,3]),wW(iW(elb.EX,1),Nnb,6,15,[3])]);};klb.xj=function xj(a,b,c,d){var e;e=klb.yj(a,6);klb.xh(a.J[e],b,c,d);return e;};klb.yj=function yj(a,b){a.q>=a.M&&klb.pm(a,a.M*2);a.C[a.q]=0;klb.am(a,a.q,b);a.s[a.q]=0;a.u[a.q]=0;a.B[a.q]=0;a.w[a.q]=0;klb.xh(a.J[a.q],0,0,0);a.v!=null&&(a.v[a.q]=null);a.t!=null&&(a.t[a.q]=null);a.T=0;return a.q++;};klb.zj=function zj(a,b,c){return klb.Aj(a,b,c,klb.kl(a,b)||klb.kl(a,c)?32:1);};klb.Aj=function Aj(a,b,c,d){var e;if(b==c)return -1;for(e=0;e<a.r;e++){if(a.D[0][e]==b&&a.D[1][e]==c||a.D[0][e]==c&&a.D[1][e]==b){a.H[e]<d&&(a.H[e]=d);return e;}}a.r>=a.N&&klb.qm(a,a.N*2);a.D[0][a.r]=b;a.D[1][a.r]=c;a.H[a.r]=d;a.F[a.r]=0;a.G[a.r]=0;a.T=0;return a.r++;};klb.Bj=function Bj(a,b){return klb.Cj(a,b,b.q,b.r);};klb.Cj=function Cj(a,b,c,d){var e,f,g,h,i;a.K=a.K|b.K;f=oW(elb.IX,omb,6,b.q,15,1);h=klb.Dl(a,1);i=klb.Dl(a,2);for(e=0;e<c;e++){f[e]=klb.Pj(b,a,e,h,i);}for(g=0;g<d;g++){klb.Qj(b,a,g,h,i,f[b.D[0][g]],f[b.D[1][g]],false);}a.L=a.L&&b.L;a.I=0;a.T=0;return f;};klb.Dj=function Dj(a,b,c,d,e,f,g,h){var i,j;i=klb._j(a,b,c);if(i==-1){a.q>=a.M&&klb.pm(a,a.M*2);i=klb.yj(a,d);klb.xh(a.J[i],b,c,0);a.A[i]=e;klb.Hl(a,i,f);klb.Wl(a,i,g);klb.Ml(a,i,h);return true;}j=klb.Kj(a,i,d,e,f,g);klb.Ml(a,i,h);return j;};klb.Ej=function Ej(a,b,c,d){var e;for(e=0;e<a.r;e++){if(a.D[0][e]==b&&a.D[1][e]==c||a.D[0][e]==c&&a.D[1][e]==b){klb.Mj(a,e,d);a.T=0;return e;}}a.r>=a.N&&klb.qm(a,a.N*2);a.D[0][a.r]=b;a.D[1][a.r]=c;a.H[a.r]=d;a.F[a.r]=0;a.G[a.r]=0;a.T=0;return a.r++;};klb.Fj=function Fj(a,b,c,d,e,f){var g,h,i;while(a.q+d>a.M)klb.pm(a,a.M*2);while(a.r+d>a.N)klb.qm(a,a.N*2);g=klb._j(a,b,c);if(g!=-1)return klb.Gj(a,g,d,e,f);h=klb.ak(a,b,c);if(h!=-1)return klb.Hj(a,h,d,e,f);g=klb.xj(a,b,c,0);i=Nmb*(d-2)/d;klb.vl(a,g,d,g,e,0,Nmb-i,f);a.T=0;return true;};klb.Gj=function Gj(a,b,c,d,e){var f,g,h,i,j,k;if(d&&klb.Sn(a,b)>1||!d&&klb.Sn(a,b)>2)return false;g=0;f=oW(elb.GX,mmb,6,4,15,1);for(i=0;i<a.r;i++){for(j=0;j<2;j++){if(a.D[j][i]==b){if(g==2){g=3;break;}f[g++]=klb.yk(a,b,a.D[1-j][i]);}}if(g==3)break;}if(g==3)return false;k=g==1?f[0]+Nmb:$wnd.Math.abs(f[0]-f[1])>Nmb?(f[0]+f[1])/2:(f[0]+f[1])/2+Nmb;h=Nmb*(c-2)/c;klb.vl(a,b,c,b,d,k-h/2,Nmb-h,e);a.T=0;return true;};klb.Hj=function Hj(a,b,c,d,e){var f,g,h,i,j,k,l,m,n,o,p;j=oW(elb.IX,omb,6,2,15,1);i=oW(elb.GX,mmb,6,2,15,1);j[0]=a.D[0][b];j[1]=a.D[1][b];if(klb.Sn(a,j[0])>3)return false;if(klb.Sn(a,j[1])>3)return false;g=0;f=oW(elb.GX,mmb,6,4,15,1);for(m=0;m<a.r;m++){if(m==b)continue;for(n=0;n<2;n++){for(o=0;o<2;o++){if(a.D[n][m]==j[o]){if(g==4){g=5;break;}f[g++]=klb.yk(a,j[o],a.D[1-n][m]);}}if(g==5)break;}if(g==5)break;}if(g==5)return false;i[0]=klb.yk(a,j[0],j[1]);if(i[0]<0){i[1]=i[0]+Nmb;h=0;}else {i[1]=i[0];i[0]=i[1]-Nmb;h=1;}p=0;for(l=0;l<g;l++){f[l]>i[0]&&f[l]<i[1]?--p:++p;}h=p>0?1-h:h;k=Nmb*(c-2)/c;klb.vl(a,j[h],c-1,j[1-h],d,i[p>0?0:1]+Nmb-k,Nmb-k,e);a.T=0;return true;};klb.Ij=function Ij(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p;e=oW(elb.IX,omb,6,b.q,15,1);i=klb.Dl(a,1);j=klb.Dl(a,2);for(d=0;d<b.q;d++){if(b.C[d]!=0||(b.t==null?null:b.t[d]==null?null:flb.h5((n=b.t[d],Klb.v8(),n)))!=null){e[d]=klb.Pj(b,a,d,i,j);}else {e[d]=c;if((b.T&8)!=0){for(h=0;h<b.r;h++){for(k=0;k<2;k++){if(b.D[k][h]==d){o=b.D[1-k][h];p=b.u[o]&3;l=false;if(p==1||p==2){for(f=0;f<b.r;f++){if(f!=h){for(m=0;m<2;m++){b.D[m][f]==o&&b.D[1-m][f]<d&&(l=!l);}}}}l&&klb.Ul(b,o,p==1?2:1,(b.u[o]&4)!=0);}}}}}}for(g=0;g<b.r;g++){klb.Qj(b,a,g,i,j,e[b.D[0][g]],e[b.D[1][g]],false);}a.L=a.L&&b.L;a.I=0;a.T=0;return e;};klb.Jj=function Jj(a,b){var c,d,e,f,g,h,i,j,k;c=a.J[b[0]];d=a.J[b[1]];e=a.J[b[2]];f=a.J[b[3]];i=new klb.Dh(d.a-c.a,d.b-c.b,d.c-c.c);j=new klb.Dh(e.a-d.a,e.b-d.b,e.c-d.c);k=new klb.Dh(f.a-e.a,f.b-e.b,f.c-e.c);g=new klb.Dh(i.b*j.c-i.c*j.b,-(i.a*j.c-i.c*j.a),i.a*j.b-i.b*j.a);h=new klb.Dh(j.b*k.c-j.c*k.b,-(j.a*k.c-j.c*k.a),j.a*k.b-j.b*k.a);return -$wnd.Math.atan2($wnd.Math.sqrt(j.a*j.a+j.b*j.b+j.c*j.c)*(i.a*h.a+i.b*h.b+i.c*h.c),g.a*h.a+g.b*h.b+g.c*h.c);};klb.Kj=function Kj(a,b,c,d,e,f){if((c==1||c==151||c==152)&&klb.Sn(a,b)>1)return false;a.B[b]=n1(a.B[b],-2);a.v!=null&&(a.v[b]=null);a.t!=null&&(a.t[b]=null);if(c==a.C[b]&&d==a.A[b]&&e==((a.u[b]&Znb)>>>27)-1&&f==(a.u[b]&48))return false;if(c==151||c==152){d=c-149;c=1;}a.u[b]&=960;a.C[b]=c;a.A[b]=d;a.s[b]=0;a.B[b]=0;klb.Hl(a,b,e);klb.Wl(a,b,f);klb.Bl(a,a.w[b]);a.T=0;return true;};klb.Lj=function Lj(a,b,c){if(c){if(a.s[b]>8)return false;++a.s[b];}else {if(a.s[b]<-8)return false;--a.s[b];}a.T=0;return true;};klb.Mj=function Mj(a,b,c){var d,e,f,g,h;f=false;g=a.H[b];if(c==511){f=klb.Uk(a,b);}else if(klb.Yo(a,b,c)){if(c==257||c==129){d=klb.wl(a,b,a.D[0][b]);e=klb.wl(a,b,a.D[1][b]);if(c==g){if(d==e||e){h=a.D[0][b];a.D[0][b]=a.D[1][b];a.D[1][b]=h;f=true;}}else {if(!d&&e){h=a.D[0][b];a.D[0][b]=a.D[1][b];a.D[1][b]=h;}a.H[b]=c;f=true;}}else {a.H[b]=c;f=true;}}if(f){a.T=(g&127)==(c&127)?a.T&7:0;a.G[b]=0;}return f;};klb.Nj=function Nj(a){a.q=0;a.r=0;a.K=false;a.L=false;a.I=0;a.v=null;a.t=null;a.P=null;a.T=0;};klb.Oj=function Oj(a){var b,c,d,e,f,g,h,i;for(g=0;g<a.r;g++){if(a.H[g]==512){c=a.D[0][g];d=a.D[1][g];if(a.C[c]==-1^a.C[d]==-1){if(a.s[c]!=0&&a.s[d]!=0){if(a.s[c]<0^a.s[d]<0){if(a.s[c]<0){++a.s[c];--a.s[d];}else {--a.s[c];++a.s[d];}}}}}}i=oW(elb.IX,omb,6,a.q,15,1);e=0;for(b=0;b<a.q;b++){if(a.C[b]==-1){i[b]=-1;continue;}if(e<b){a.C[e]=a.C[b];a.s[e]=a.s[b];a.A[e]=a.A[b];a.u[e]=a.u[b];a.B[e]=a.B[b];a.w[e]=a.w[b];klb.yh(a.J[e],a.J[b]);a.v!=null&&(a.v[e]=a.v[b]);a.t!=null&&(a.t[e]=a.t[b]);}i[b]=e;++e;}a.q=e;h=0;for(f=0;f<a.r;f++){if(a.H[f]==512)continue;a.H[h]=a.H[f];a.F[h]=a.F[f];a.G[h]=a.G[f];a.D[0][h]=i[a.D[0][f]];a.D[1][h]=i[a.D[1][f]];++h;}a.r=h;return i;};klb.Pj=function Pj(a,b,c,d,e){var f,g,h;f=b.q;f>=b.M&&klb.pm(b,b.M*2);h=(a.u[c]&znb)>>18;g=-1;h==1?d==-1?g=klb.Dl(b,h):g=$wnd.Math.min(31,d+((a.u[c]&znb)>>18!=1&&(a.u[c]&znb)>>18!=2?-1:(a.u[c]&$nb)>>20)):h==2&&(e==-1?g=klb.Dl(b,h):g=$wnd.Math.min(31,e+((a.u[c]&znb)>>18!=1&&(a.u[c]&znb)>>18!=2?-1:(a.u[c]&$nb)>>20)));b.C[f]=a.C[c];b.s[f]=a.s[c];b.A[f]=a.A[c];b.u[f]=a.u[c];b.B[f]=b.K?a.B[c]:0;klb.yh(b.J[f],a.J[c]);b.w[f]=a.w[c];b.v!=null&&(b.v[f]=null);if(a.v!=null&&a.v[c]!=null&&b.K){b.v==null&&(b.v=oW(elb.IX,Lnb,7,b.C.length,0,2));b.v[f]=llb.acb(a.v[c],a.v[c].length);}b.t!=null&&(b.t[f]=null);if(a.t!=null&&a.t[c]!=null){b.t==null&&(b.t=oW(elb.EX,Tlb,9,b.C.length,0,2));b.t[f]=llb.$bb(a.t[c],a.t[c].length);}if(g!=-1){b.u[f]&=-32505857;b.u[f]|=g<<20;}++b.q;b.T=0;return f;};klb.Qj=function Qj(a,b,c,d,e,f,g,h){var i,j,k,l;j=b.r;j>=b.N&&klb.qm(b,b.N*2);l=(a.F[c]&768)>>8;k=-1;l==1&&(d==-1?k=klb.Dl(b,l):k=$wnd.Math.min(32,d+((a.F[c]&768)>>8!=1&&(a.F[c]&768)>>8!=2?-1:(a.F[c]&_nb)>>10)));l==2&&(e==-1?k=klb.Dl(b,l):k=$wnd.Math.min(32,e+((a.F[c]&768)>>8!=1&&(a.F[c]&768)>>8!=2?-1:(a.F[c]&_nb)>>10)));b.D[0][j]=f;b.D[1][j]=g;i=h&&a.vb(c)?64:a.H[c];b.H[j]=i;b.F[j]=a.F[c];b.G[j]=b.K?a.G[c]:0;if(k!=-1){b.F[j]&=-31745;b.F[j]|=k<<10;}++b.r;b.T=0;return j;};klb.Rj=function Rj(a,b,c,d,e,f,g){return klb.Qj(a,b,c,d,e,f==null?a.D[0][c]:f[a.D[0][c]],f==null?a.D[1][c]:f[a.D[1][c]],g);};klb.Sj=function Sj(a,b){var c,d;b.v=null;b.t=null;b.K=a.K;b.q=0;for(c=0;c<a.q;c++)klb.Pj(a,b,c,0,0);b.r=0;for(d=0;d<a.r;d++)klb.Qj(a,b,d,0,0,a.D[0][d],a.D[1][d],false);a.ub(b);};klb.Tj=function Tj(a,b){b.K=a.K;b.L=a.L;b.S=a.S;b.I=a.I;b.P=a.P;b.T=a.T&24;};klb.Uj=function Uj(a,b){var c,d,e,f;for(c=0;c<a.r;c++){for(e=0;e<2;e++){if(a.D[e][c]==b){a.H[c]=512;d=0;for(f=0;f<a.r;f++){if(f==c)continue;(a.D[0][f]===a.D[1-e][c]||a.D[1][f]===a.D[1-e][c])&&++d;}if(d==0){klb.Bl(a,a.w[a.D[1-e][c]]);a.C[a.D[1-e][c]]=-1;}}}}klb.Bl(a,a.w[b]);a.C[b]=-1;a.v!=null&&(a.v[b]=null);a.t!=null&&(a.t[b]=null);klb.Oj(a);a.T=0;};klb.Vj=function Vj(a,b,c){var d,e;d=klb._j(a,b,c);if(d!=-1){(a.u[d]&512)!=0?klb.$j(a):klb.Uj(a,d);a.T=0;return true;}e=klb.ak(a,b,c);if(e!=-1){(a.u[a.D[0][e]]&a.u[a.D[1][e]]&512)!=0?klb.$j(a):klb.Yj(a,e);a.T=0;return true;}return false;};klb.Wj=function Wj(a,b){var c,d,e,f;if(b.length==0)return null;for(d=b,e=0,f=d.length;e<f;++e){c=d[e];a.C[c]=-1;}return klb.Zj(a);};klb.Xj=function Xj(a,b){a.H[b]=512;klb.Oj(a);a.T=0;};klb.Yj=function Yj(a,b){var c,d,e;for(d=0;d<2;d++){c=0;for(e=0;e<a.r;e++){if(e==b)continue;(a.D[0][e]===a.D[d][b]||a.D[1][e]===a.D[d][b])&&++c;}if(c==0){klb.Bl(a,a.w[a.D[d][b]]);a.C[a.D[d][b]]=-1;}}a.H[b]=512;klb.Oj(a);a.T=0;};klb.Zj=function Zj(a){var b,c,d;d=false;for(b=0;b<a.q;b++){if(a.C[b]==-1){d=true;klb.Bl(a,a.w[b]);}}for(c=0;c<a.r;c++){if(a.H[c]==512){d=true;}else if(a.C[a.D[0][c]]==-1||a.C[a.D[1][c]]==-1){a.H[c]=512;d=true;}}if(d){a.T=0;return klb.Oj(a);}return null;};klb.$j=function $j(a){var b,c;c=false;for(b=0;b<a.q;b++){if((a.u[b]&512)!=0){a.C[b]=-1;c=true;}}return c&&klb.Zj(a)!=null;};klb._j=function _j(a,b,c){var d,e,f,g,h,i,j,k;g=-1;e=klb.wk(a,a.q,a.r,klb.vj);h=aob;i=e*e/12;for(d=0;d<a.q;d++){j=a.J[d].a;k=a.J[d].b;f=(b-j)*(b-j)+(c-k)*(c-k);if(f<i&&f<h){h=f;g=d;}}return g;};klb.ak=function ak(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s;m=-1;o=klb.wk(a,a.q,a.r,klb.vj);n=aob;for(d=0;d<a.r;d++){p=a.J[a.D[0][d]].a;r=a.J[a.D[0][d]].b;q=a.J[a.D[1][d]].a;s=a.J[a.D[1][d]].b;k=q-p;l=s-r;e=$wnd.Math.sqrt(k*k+l*l);f=(p+q)/2;g=(r+s)/2;k=b-f;l=c-g;if($wnd.Math.sqrt(k*k+l*l)>e/2)continue;if(q==p)j=$wnd.Math.abs(p-b);else {h=(s-r)/(p-q);i=-h*p-r;j=$wnd.Math.abs((h*b+c+i)/$wnd.Math.sqrt(h*h+1));}if(j<o&&j<n){n=j;m=d;}}return m;};klb.bk=function bk(a,b){return ((a.u[b]&Znb)>>>27)-1;};klb.ck=function ck(a,b){return (a.u[b]&49152)>>14;};klb.dk=function dk(a,b){return a.s[b];};klb.ek=function ek(a,b){return a.u[b]&448;};klb.fk=function fk(a,b){var c;return a.t==null?null:a.t[b]==null?null:flb.h5((c=a.t[b],Klb.v8(),c));};klb.gk=function gk(a,b){return a.t==null?null:a.t[b];};klb.hk=function hk(a,b){return (a.u[b]&znb)>>18!=1&&(a.u[b]&znb)>>18!=2?-1:(a.u[b]&$nb)>>20;};klb.ik=function ik(a,b){return (a.u[b]&znb)>>18;};klb.jk=function jk(a,b){return klb.qj[a.C[b]];};klb.kk=function kk(a,b){return a.v==null?null:a.v[b];};klb.lk=function lk(a,b){var c,d,e;if(a.v==null||a.v[b]==null)return I1(n1(a.B[b],1),0)?'':klb.qj[a.C[b]];e='';for(d=0;d<a.v[b].length;d++){d>0&&(e=(ilb.Mgb(e),e+(ilb.Mgb(','),',')));c=a.v[b][d];e=flb.g5(e,klb.qj[c]);}return e;};klb.mk=function mk(a,b){return $wnd.Math.abs(a.w[b]);};klb.nk=function nk(a,b){return a.A[b];};klb.ok=function ok(a,b){return a.u[b]&3;};klb.pk=function pk(a,b){return a.B[b];};klb.qk=function qk(a,b){return a.u[b]&48;};klb.rk=function rk(a,b){return a.J[b].a;};klb.sk=function sk(a,b){return a.J[b].b;};klb.tk=function tk(a,b){return a.J[b].c;};klb.uk=function uk(a,b){return a.C[b];};klb.vk=function vk(a){return klb.wk(a,a.q,a.r,klb.vj);};klb.wk=function wk(a,b,c,d){return klb.xk(a,b,c,d,a.J);};klb.xk=function xk(a,b,c,d,e){var f,g,h,i,j,k,l,m,n,o;l=false;m=0;for(j=0;j<c;j++)a.H[j]!=32&&(a.G[j]&vnb)==0&&++m;if(m==0){for(k=0;k<c;k++)(a.G[k]&vnb)==0&&++m;l=true;}if(m==0){if(b<2)return d;o=aob;for(f=1;f<b;f++){for(g=0;g<f;g++){n=klb.sh(e[f],e[g]);n>0&&n<o&&(o=n);}}return o!=aob?0.6*o:d;}h=0;for(i=0;i<c;i++){(l||a.H[i]!=32)&&(a.G[i]&vnb)==0&&(h+=klb.sh(e[a.D[1][i]],e[a.D[0][i]]));}return h/m;};klb.yk=function yk(a,b,c){return klb.Mm(a.J[b].a,a.J[b].b,a.J[c].a,a.J[c].b);};klb.zk=function zk(a,b,c){return a.D[b][c];};klb.Ak=function Ak(a,b){return ((a.G[b]&7680)>>9)+((a.G[b]&122880)>>13);};klb.Bk=function Bk(a,b){return (a.G[b]&7680)>>9;};klb.Ck=function Ck(a,b){return (a.F[b]&48)>>4;};klb.Dk=function Dk(a,b){return (a.F[b]&768)>>8!=1&&(a.F[b]&768)>>8!=2?-1:(a.F[b]&_nb)>>10;};klb.Ek=function Ek(a,b){return (a.F[b]&768)>>8;};klb.Fk=function Fk(a,b){var c,d,e,f;c=a.D[0][b];d=a.D[1][b];e=a.J[d].a-a.J[c].a;f=a.J[d].b-a.J[c].b;return $wnd.Math.sqrt(e*e+f*f);};klb.Gk=function Gk(a,b,c){var d;for(d=0;d<a.r;d++)if(a.D[0][d]==b&&a.D[1][d]==c||a.D[0][d]==c&&a.D[1][d]==b)if(a.H[d]!=512)return d;return -1;};klb.Hk=function Hk(a,b){switch(a.H[b]&127){case 1:case 64:return 1;case 2:return 2;case 4:return 3;case 8:return 4;case 16:return 5;default:return 0;}};klb.Ik=function Ik(a,b){return a.F[b]&3;};klb.Jk=function Jk(a,b){return a.G[b];};klb.Kk=function Kk(a,b){return a.H[b];};klb.Lk=function Lk(a,b){return a.H[b]&127;};klb.Mk=function Mk(a,b){var c,d,e,f,g;if(a.q==0)return null;d=a.J[0].a;f=a.J[0].b;e=a.J[0].a;g=a.J[0].b;for(c=1;c<a.q;c++){d>a.J[c].a?d=a.J[c].a:e<a.J[c].a&&(e=a.J[c].a);f>a.J[c].b?f=a.J[c].b:g<a.J[c].b&&(g=a.J[c].b);}if(!b){b=new xlb.dG(d,f,e-d,g-f);}else {b.c=d;b.d=f;b.b=e-d;b.a=g-f;}return b;};klb.Nk=function Nk(a,b){var c;c=a.C[b]<klb.rj.length?klb.rj[a.C[b]]:null;return c==null?6:c[c.length-1];};klb.Ok=function Ok(a,b,c){return klb.Pk(a,b,c,a.C[b]);};klb.Pk=function Pk(a,b,c,d){var e,f;if(d>=171&&d<=190)return 0;f=0;(a.u[b]&48)==32&&(f-=1);((a.u[b]&48)==16||(a.u[b]&48)==48)&&(f-=2);e=a.s[b];if(e==0&&a.K){x1(n1(a.B[b],Wmb),Zmb)&&(e=-1);x1(n1(a.B[b],Wmb),Ymb)&&(e=1);}d==7||d==8||d==9?f+=e:d==6||d==14||d==32?f-=$wnd.Math.abs(e):d==15||d==33?c-f-e<=3?f+=e:f-=e:d==16||d==34||d==52?c-f-e<=4?f+=e:f-=$wnd.Math.abs(e):d==17||d==35||d==53?c-f-e<=5?f+=e:f-=$wnd.Math.abs(e):f-=e;return f;};klb.Qk=function Qk(a,b){var c;c=klb.Rk(a,b);return c+klb.Pk(a,b,c,a.C[b]);};klb.Rk=function Rk(a,b){var c,d;c=((a.u[b]&Znb)>>>27)-1;c==-1&&(c=(d=a.C[b]<klb.rj.length?klb.rj[a.C[b]]:null,d==null?6:d[d.length-1]));return c;};klb.Sk=function Sk(a,b){var c,d,e,f,g;f=klb.sl(a,a.D[0][b])||klb.sl(a,a.D[1][b])?5:3;for(d=0;d<2;d++){c=a.D[d][b];e=klb.Hk(a,b)+(g=klb.Rk(a,c),g+klb.Pk(a,c,g,a.C[c]))-klb.Sn(a,c);f>e&&(f=e);}return f;};klb.Tk=function Tk(a,b){return (a.u[b]&emb)!=0;};klb.Uk=function Uk(a,b){var c,d,e;d=klb.Sk(a,b);c=klb.kl(a,a.D[0][b])||klb.kl(a,a.D[1][b]);e=c?32:1;if(a.H[b]==16){a.H[b]=e;a.T=0;return true;}if(a.H[b]==8){a.H[b]=d>4?16:e;a.T=0;return true;}if(a.H[b]==4){a.H[b]=d>3?8:e;a.T=0;return true;}if(a.H[b]==2){a.H[b]=386;a.T&=7;if((a.F[b]&128)==0)return true;}if(a.H[b]==386){d>2?a.H[b]=4:a.H[b]=e;a.T=0;return true;}if((384&a.H[b])!=0){a.H[b]=1;a.T&=7;return true;}if(!c&&d<2)return false;if(a.H[b]==1){a.H[b]=2;a.T=0;return true;}if(d<1)return false;if(a.H[b]==32){a.H[b]=1;a.T=0;return true;}return false;};klb.Vk=function Vk(a){var b;a.T=0;a.C=oW(elb.IX,omb,6,a.M,15,1);a.s=oW(elb.IX,omb,6,a.M,15,1);a.w=oW(elb.IX,omb,6,a.M,15,1);a.J=oW(elb._X,bob,24,a.M,0,1);for(b=0;b<a.M;b++)a.J[b]=new klb.Ch();a.A=oW(elb.IX,omb,6,a.M,15,1);a.u=oW(elb.IX,omb,6,a.M,15,1);a.B=oW(elb.JX,Inb,6,a.M,14,1);a.v=null;a.t=null;a.D=mW(elb.IX,[Lnb,omb],[7,6],15,[2,a.N],2);a.H=oW(elb.IX,omb,6,a.N,15,1);a.F=oW(elb.IX,omb,6,a.N,15,1);a.G=oW(elb.IX,omb,6,a.N,15,1);};klb.Wk=function Wk(a,b){a.T&=~b;};klb.Xk=function Xk(a){var b;for(b=0;b<a.q;b++)if(a.J[b].c!=0)return true;return false;};klb.Yk=function Yk(a,b){return (a.u[b]&cob)!=0;};klb.Zk=function Zk(a,b){return a.C[b]==-1;};klb.$k=function $k(a,b){return (a.u[b]&4)!=0;};klb._k=function _k(a,b){return (a.u[b]&dob)!=0;};klb.al=function al(a,b){return a.w[b]<0;};klb.bl=function bl(a,b){return (a.F[b]&eob)!=0;};klb.cl=function cl(a,b){return (a.G[b]&vnb)!=0;};klb.dl=function dl(a,b){return (a.F[b]&emb)!=0;};klb.el=function el(a,b){return a.H[b]==512;};klb.fl=function fl(a,b){return (a.F[b]&4)!=0;};klb.gl=function gl(a,b){return (a.F[b]&Onb)!=0;};klb.hl=function hl(a,b){var c,d,e,f;if(a.K){if(I1(n1(a.B[b],1),0))return false;if(a.v!=null&&a.v[b]!=null)for(d=a.v[b],e=0,f=d.length;e<f;++e){c=d[e];if(!klb.Pm(c))return false;}}return klb.Pm(a.C[b]);};klb.il=function il(a,b){var c,d,e,f;if(a.K){if(I1(n1(a.B[b],1),0))return false;if(a.v!=null&&a.v[b]!=null)for(d=a.v[b],e=0,f=d.length;e<f;++e){c=d[e];if(!klb.Qm(c))return false;}}return klb.Qm(a.C[b]);};klb.jl=function jl(a,b){return (a.u[b]&Onb)!=0;};klb.kl=function kl(a,b){var c,d,e,f;if(a.K){if(I1(n1(a.B[b],1),0))return false;if(a.v!=null&&a.v[b]!=null)for(d=a.v[b],e=0,f=d.length;e<f;++e){c=d[e];if(!(c>=3&&c<=4||c>=11&&c<=13||c>=19&&c<=31||c>=37&&c<=51||c>=55&&c<=84||c>=87&&c<=112))return false;}}return klb.Rm(a.C[b]);};klb.ll=function ll(a,b){return a.A[b]==0;};klb.ml=function ml(a,b){var c,d,e,f;if(a.K){if(I1(n1(a.B[b],1),0))return false;if(a.v!=null&&a.v[b]!=null)for(d=a.v[b],e=0,f=d.length;e<f;++e){c=d[e];if(!(c==1||c>=5&&c<=9||c>=14&&c<=17||c>=32&&c<=35||c>=52&&c<=53))return false;}}return klb.Sm(a.C[b]);};klb.nl=function nl(a){var b;for(b=0;b<a.q;b++){switch(a.C[b]){case 1:case 5:case 6:case 7:case 8:case 9:case 14:case 15:case 16:case 17:case 33:case 34:case 35:case 52:case 53:continue;default:return false;}}return true;};klb.ol=function ol(a,b){return (a.u[b]&512)!=0;};klb.pl=function pl(a,b){return (a.u[a.D[0][b]]&a.u[a.D[1][b]]&512)!=0;};klb.ql=function ql(a,b){return a.H[b]==257||a.H[b]==129;};klb.rl=function rl(a,b,c){return (a.H[b]==257||a.H[b]==129)&&a.D[0][b]==c;};klb.sl=function sl(a,b){var c,d,e,f;if(a.K){if(I1(n1(a.B[b],1),0))return false;if(a.v!=null&&a.v[b]!=null)for(d=a.v[b],e=0,f=d.length;e<f;++e){c=d[e];if(!(c>=21&&c<=30||c>=39&&c<=48||c==57||c>=72&&c<=80||c==89||c>=104&&c<=112))return false;}}return klb.Tm(a.C[b]);};klb.tl=function tl(a,b){a.C[b]=-1;};klb.ul=function ul(a,b){a.H[b]=512;};klb.vl=function vl(a,b,c,d,e,f,g,h){var i,j,k,l,m,n,o,p,q,r;if(b!=d){q=a.J[b].a-a.J[d].a;r=a.J[b].b-a.J[d].b;h=$wnd.Math.sqrt(q*q+r*r);}i=b;k=klb.um(a,b)!=3;for(p=1;p<c;p++){m=a.J[i].a+h*$wnd.Math.sin(f);n=a.J[i].b+h*$wnd.Math.cos(f);o=-1;for(l=0;l<a.q;l++){if($wnd.Math.abs(m-a.J[l].a)<4&&$wnd.Math.abs(n-a.J[l].b)<4){o=l;break;}}if(o==-1){o=klb.xj(a,m,n,0);a.J[o].a=m;a.J[o].b=n;a.J[o].c=0;}j=klb.Gk(a,i,o);if(j==-1){j=klb.Aj(a,i,o,klb.kl(a,i)||klb.kl(a,o)?32:1);if(e){k&&klb.um(a,a.D[0][j])<4&&klb.um(a,a.D[1][j])<3&&(a.H[j]=2);k=!k;}}i=o;f+=g;}j=klb.Gk(a,i,d);j==-1&&(j=klb.Aj(a,i,d,klb.kl(a,i)||klb.kl(a,d)?32:1));e&&k&&klb.um(a,a.D[0][j])<4&&klb.um(a,a.D[1][j])<4&&(a.H[j]=2);};klb.wl=function wl(a,b,c){var d,e;if(klb.Hk(a,b)!=1)return false;if((a.u[c]&3)!=0)return true;for(e=0;e<a.r;e++)if(e!=b&&a.H[e]==2&&(a.D[0][e]==c&&(a.u[a.D[1][e]]&3)!=0||a.D[1][e]==c&&(a.u[a.D[0][e]]&3)!=0))return true;for(d=0;d<a.r;d++)if(d!=b&&a.H[d]==1&&(a.D[0][d]==c||a.D[1][d]==c)&&(a.F[d]&3)!=0)return true;return false;};klb.xl=function xl(a){var b;for(b=0;b<a.q;b++)a.u[b]&=-449;};klb.yl=function yl(a){var b;for(b=0;b<a.q;b++)a.u[b]&=-131073;};klb.zl=function zl(a){var b;for(b=0;b<a.q;b++)a.u[b]&=-513;};klb.Al=function Al(a){var b;for(b=0;b<a.r;b++)a.F[b]&=-98305;};klb.Bl=function Bl(a,b){var c;for(c=0;c<a.q;c++)$wnd.Math.abs(a.w[c])==$wnd.Math.abs(b)&&(a.w[c]=0);};klb.Cl=function Cl(a){var b,c,d,e;e=false;for(c=0;c<a.q;c++){if(I1(n1(a.B[c],Jmb),0)){a.C[c]=-1;e=true;}}e&&klb.Zj(a);if(a.v!=null){a.v=null;e=true;}for(b=0;b<a.q;b++){if(I1(a.B[b],0)){a.B[b]=0;e=true;}}for(d=0;d<a.r;d++){if(a.G[d]!=0){a.G[d]=0;e=true;}if(a.H[d]==64){a.H[d]=1;e=true;}}e&&(a.T=0);return e;};klb.Dl=function Dl(a,b){var c,d,e,f,g,h,i,j,k;if(b==0)return 0;h=null;for(d=0;d<a.q;d++){if((a.u[d]&znb)>>18==b){h==null&&(h=oW(elb.g1,Imb,6,32,16,1));h[(a.u[d]&znb)>>18!=1&&(a.u[d]&znb)>>18!=2?-1:(a.u[d]&$nb)>>20]=true;}}for(f=0;f<a.r;f++){if((a.F[f]&768)>>8==b){h==null&&(h=oW(elb.g1,Imb,6,32,16,1));h[(a.F[f]&768)>>8!=1&&(a.F[f]&768)>>8!=2?-1:(a.F[f]&_nb)>>10]=true;}}k=0;if(h!=null){j=oW(elb.IX,omb,6,32,15,1);for(i=0;i<32;i++)h[i]&&(j[i]=k++);for(c=0;c<a.q;c++){if((a.u[c]&znb)>>18==b){g=j[(a.u[c]&znb)>>18!=1&&(a.u[c]&znb)>>18!=2?-1:(a.u[c]&$nb)>>20];a.u[c]&=-32505857;a.u[c]|=g<<20;}}for(e=0;e<a.r;e++){if((a.F[e]&768)>>8==b){g=j[(a.F[e]&768)>>8!=1&&(a.F[e]&768)>>8!=2?-1:(a.F[e]&_nb)>>10];a.F[e]&=-31745;a.F[e]|=g<<10;}}}return k;};klb.El=function El(a,b){var c;for(c=0;c<a.q;c++){a.J[c].a*=b;a.J[c].b*=b;}};klb.Fl=function Fl(a,b){a.q=b;a.T=0;};klb.Gl=function Gl(a,b){a.r=b;a.T=0;};klb.Hl=function Hl(a,b,c){if(c>=-1&&c<=14){a.u[b]&=-2013265921;a.u[b]|=1+c<<27;if(a.C[b]==6){if(c==-1||c==0||c==2||c==4){a.u[b]&=-49;c==2&&(a.u[b]|=16);}}}};klb.Il=function Il(a,b,c){a.u[b]&=-49153;a.u[b]|=c<<14;};klb.Jl=function Jl(a,b,c){a.s[b]=c;a.T=0;};klb.Kl=function Kl(a,b,c){a.u[b]&=-449;a.u[b]|=c;};klb.Ll=function Ll(a,b,c){c?a.u[b]|=cob:a.u[b]&=fob;a.T&=7;};klb.Ml=function Ml(a,b,c){var d,e;if(c!=null){if(ilb.Xgb(c).length==0)c=null;else {d=klb.Om(c,321);if(d!=0&&flb.j5(c,klb.qj[d])||flb.j5(c,'?')){klb.am(a,b,d);c=null;}}}if(c==null){a.t!=null&&(a.t[b]=null);}else {a.t==null&&(a.t=oW(elb.EX,Tlb,9,a.M,0,2));a.t[b]=ilb.ygb((e=c,Klb.v8(),e));}};klb.Nl=function Nl(a,b,c){c!=null&&c.length==0&&(c=null);if(c==null){a.t!=null&&(a.t[b]=null);}else {a.t==null&&(a.t=oW(elb.EX,Tlb,9,a.M,0,2));a.t[b]=c;}};klb.Ol=function Ol(a,b,c,d){var e,f,g;if(c==0){a.u[b]&=gob;a.u[b]|=c<<18;}else {if(d>=32)return;if(d==-1){g=-1;for(f=0;f<a.q;f++)f!=b&&c==(a.u[f]&znb)>>18&&g<((a.u[f]&znb)>>18!=1&&(a.u[f]&znb)>>18!=2?-1:(a.u[f]&$nb)>>20)&&(g=(a.u[f]&znb)>>18!=1&&(a.u[f]&znb)>>18!=2?-1:(a.u[f]&$nb)>>20);for(e=0;e<a.r;e++)c==(a.F[e]&768)>>8&&g<((a.F[e]&768)>>8!=1&&(a.F[e]&768)>>8!=2?-1:(a.F[e]&_nb)>>10)&&(g=(a.F[e]&768)>>8!=1&&(a.F[e]&768)>>8!=2?-1:(a.F[e]&_nb)>>10);d=g+1;if(d>=32)return;}a.u[b]&=gob;a.u[b]|=c<<18|d<<20;}a.T&=7;};klb.Pl=function Pl(a,b,c){a.v==null&&(a.v=oW(elb.IX,Lnb,7,a.M,0,2));ilb.rgb(c,ilb.Xgb(e2(llb.Ccb.prototype.kc,llb.Ccb,[])));a.v[b]=c;a.T=0;a.K=true;};klb.Ql=function Ql(a,b,c,d){var e;if(c==null){a.v!=null&&(a.v[b]=null);return;}if(c.length==1&&!d){e=c[0];a.C[b]!=e&&klb.Kj(a,b,e,0,-1,0);a.v!=null&&(a.v[b]=null);return;}a.v==null&&(a.v=oW(elb.IX,Lnb,7,a.M,0,2));a.v[b]=c;d&&(a.B[b]=K1(a.B[b],1));a.T=0;a.K=true;};klb.Rl=function Rl(a,b,c,d){a.w[b]=d?-c:c;};klb.Sl=function Sl(a,b,c){c?a.u[b]|=Onb:a.u[b]&=-131073;};klb.Tl=function Tl(a,b,c){a.A[b]=c;a.T&=a.C[b]==1?0:7;};klb.Ul=function Ul(a,b,c,d){a.u[b]&=-8;c!=3&&(a.u[b]&=fob);a.u[b]|=c;d&&(a.u[b]|=4);};klb.Vl=function Vl(a,b,c,d){d?a.B[b]=K1(a.B[b],c):a.B[b]=n1(a.B[b],J1(c));a.T=0;a.K=true;};klb.Wl=function Wl(a,b,c){a.u[b]&=-49;a.u[b]|=c;a.T&=7;};klb.Xl=function Xl(a,b,c){c?a.u[b]|=512:a.u[b]&=-513;};klb.Yl=function Yl(a,b,c){a.u[b]&=-67108865;c&&(a.u[b]|=dob);};klb.Zl=function Zl(a,b,c){a.J[b].a=c;a.T&=7;};klb.$l=function $l(a,b,c){a.J[b].b=c;a.T&=7;};klb._l=function _l(a,b,c){a.J[b].c=c;a.T&=7;};klb.am=function am(a,b,c){if(c>=0&&c<=190){if(c==151||c==152){a.C[b]=1;a.A[b]=c-149;}else {a.C[b]=c;a.A[b]=0;}a.u[b]&=-2013265921;a.T=0;}};klb.bm=function bm(a,b,c,d){a.D[b][c]=d;a.T=0;};klb.cm=function cm(a,b,c){c?a.F[b]|=eob:a.F[b]&=-32769;};klb.dm=function dm(a,b,c){a.F[b]&=-49;a.F[b]|=c<<4;};klb.em=function em(a,b,c,d){var e,f,g;if(c==0){a.F[b]&=-32513;a.F[b]|=c<<8;}else {if(d>=32)return;if(d==-1){g=-1;for(f=0;f<a.q;f++)c==(a.u[f]&znb)>>18&&g<((a.u[f]&znb)>>18!=1&&(a.u[f]&znb)>>18!=2?-1:(a.u[f]&$nb)>>20)&&(g=(a.u[f]&znb)>>18!=1&&(a.u[f]&znb)>>18!=2?-1:(a.u[f]&$nb)>>20);for(e=0;e<a.r;e++)e!=b&&c==(a.F[e]&768)>>8&&g<((a.F[e]&768)>>8!=1&&(a.F[e]&768)>>8!=2?-1:(a.F[e]&_nb)>>10)&&(g=(a.F[e]&768)>>8!=1&&(a.F[e]&768)>>8!=2?-1:(a.F[e]&_nb)>>10);d=g+1;if(d>=32)return;}a.F[b]&=-32513;a.F[b]|=c<<8|d<<10;}a.T&=7;};klb.fm=function fm(a,b,c){c?a.F[b]|=emb:a.F[b]&=-65537;};klb.gm=function gm(a,b,c){a.H[b]=c==1?1:c==2?2:c==3?4:32;a.T=0;};klb.hm=function hm(a,b,c,d){a.F[b]&=-131080;a.F[b]|=c;d&&(a.F[b]|=4);};klb.im=function im(a,b){a.F[b]|=Onb;};klb.jm=function jm(a,b,c,d){d?a.G[b]|=c:a.G[b]&=~c;a.T=0;a.K=true;};klb.km=function km(a,b,c){a.H[b]=c;a.T=0;};klb.lm=function lm(a,b){a.I=b;};klb.mm=function mm(a,b){if(a.K!=b){a.K=b;b||klb.Cl(a);a.T=0;}};klb.nm=function nm(a,b){a.T=b;};klb.om=function om(a,b){a.S=b;};klb.pm=function pm(a,b){var c,d;a.C=klb.Gm(a.C,b);a.s=klb.Gm(a.s,b);a.w=klb.Gm(a.w,b);d=a.J.length;a.J=klb.Im(a.J,b);for(c=d;c<b;c++)a.J[c]=new klb.Ch();a.A=klb.Gm(a.A,b);a.u=klb.Gm(a.u,b);a.B=klb.Hm(a.B,b);a.v!=null&&(a.v=klb.Km(a.v,b));a.t!=null&&(a.t=klb.Jm(a.t,b));a.M=b;};klb.qm=function qm(a,b){a.D[0]=klb.Gm(a.D[0],b);a.D[1]=klb.Gm(a.D[1],b);a.H=klb.Gm(a.H,b);a.F=klb.Gm(a.F,b);a.G=klb.Gm(a.G,b);a.N=b;};klb.rm=function rm(a,b){a.O=b;};klb.sm=function sm(a,b){a.P=b;};klb.tm=function tm(a,b){a.u[b]|=emb;};klb.um=function um(a,b){var c,d;d=0;for(c=0;c<a.r;c++)(a.D[0][c]==b||a.D[1][c]==b)&&(d+=klb.Hk(a,c));return d;};klb.vm=function vm(a){var b,c,d;c=false;d=false;for(b=0;b<a.q;b++){if(a.A[b]!=0){a.A[b]=0;c=true;a.C[b]==1&&(d=true);}}d&&(a.T=0);return c;};klb.wm=function wm(a,b,c){return klb.kl(a,b)||klb.kl(a,c)?32:1;};klb.xm=function xm(a,b,c){var d,e,f,g,h,i;g=a.C[b];a.C[b]=a.C[c];a.C[c]=g;g=a.s[b];a.s[b]=a.s[c];a.s[c]=g;g=a.A[b];a.A[b]=a.A[c];a.A[c]=g;g=a.u[b];a.u[b]=a.u[c];a.u[c]=g;i=a.B[b];a.B[b]=a.B[c];a.B[c]=i;g=a.w[b];a.w[b]=a.w[c];a.w[c]=g;f=a.J[b];a.J[b]=a.J[c];a.J[c]=f;if(a.v!=null){h=a.v[b];a.v[b]=a.v[c];a.v[c]=h;}if(a.t!=null){h=a.t[b];a.t[b]=a.t[c];a.t[c]=h;}for(d=0;d<a.r;d++){for(e=0;e<2;e++){a.D[e][d]==b?a.D[e][d]=c:a.D[e][d]==c&&(a.D[e][d]=b);}}a.T=0;};klb.ym=function ym(a,b,c){var d;d=a.D[0][b];a.D[0][b]=a.D[0][c];a.D[0][c]=d;d=a.D[1][b];a.D[1][b]=a.D[1][c];a.D[1][c]=d;d=a.H[b];a.H[b]=a.H[c];a.H[c]=d;d=a.F[b];a.F[b]=a.F[c];a.F[c]=d;d=a.G[b];a.G[b]=a.G[c];a.G[c]=d;a.T=0;};klb.zm=function zm(a,b,c){var d;for(d=0;d<a.q;d++){a.J[d].a+=b;a.J[d].b+=c;}a.U+=b;a.V+=c;};klb.Am=function Am(a,b,c){var d,e;e=c&127;d=klb.Sk(a,b);switch(e){case 1:case 64:return d>=1;case 2:return d>=2;case 4:return d>=3;case 8:return d>=4;case 16:return d>=5;case 32:return true;default:return false;}};klb.Bm=function Bm(a,b,c,d){var e,f,g;for(e=0;e<a.q;e++){if(!d||(a.u[e]&512)!=0){g=a.R[e]*b;f=a.Q[e]-c;a.J[e].a=a.U+g*$wnd.Math.sin(f);a.J[e].b=a.V+g*$wnd.Math.cos(f);}}d&&(a.T&=7);};klb.Cm=function Cm(a,b,c){var d,e,f;a.U=b;a.V=c;a.Q=oW(elb.GX,mmb,6,a.q,15,1);a.R=oW(elb.GX,mmb,6,a.q,15,1);for(d=0;d<a.q;d++){e=b-a.J[d].a;f=c-a.J[d].b;a.R[d]=$wnd.Math.sqrt(e*e+f*f);a.Q[d]=klb.Mm(b,c,a.J[d].a,a.J[d].b);}};klb.Dm=function Dm(){this.M=this.N=256;klb.Vk(this);};klb.Em=function Em(a,b){this.M=$wnd.Math.max(1,a);this.N=$wnd.Math.max(1,b);klb.Vk(this);};klb.Gm=function Gm(a,b){var c;c=oW(elb.IX,omb,6,b,15,1);flb.W5(a,0,c,0,$wnd.Math.min(a.length,b));return c;};klb.Hm=function Hm(a,b){var c;c=oW(elb.JX,Inb,6,b,14,1);flb.W5(a,0,c,0,$wnd.Math.min(a.length,b));return c;};klb.Im=function Im(a,b){var c,d;c=oW(elb._X,bob,24,b,0,1);for(d=0;d<a.length;d++)!!a[d]&&(c[d]=new klb.Eh(a[d]));return c;};klb.Jm=function Jm(a,b){var c,d;c=oW(elb.EX,Tlb,9,b,0,2);for(d=0;d<a.length;d++){if(a[d]!=null){c[d]=oW(elb.EX,Nnb,6,a[d].length,15,1);flb.W5(a[d],0,c[d],0,a[d].length);}}return c;};klb.Km=function Km(a,b){var c,d;c=oW(elb.IX,Lnb,7,b,0,2);for(d=0;d<a.length;d++){if(a[d]!=null){c[d]=oW(elb.IX,omb,6,a[d].length,15,1);flb.W5(a[d],0,c[d],0,a[d].length);}}return c;};klb.Lm=function Lm(a){klb.wj();return a>=0&&a<klb.rj.length&&klb.rj[a]!=null?klb.rj[a]:a>=171&&a<=190?klb.pj:klb.tj;};klb.Mm=function Mm(a,b,c,d){klb.wj();var e,f,g;f=c-a;g=d-b;if(g!=0){e=$wnd.Math.atan(f/g);g<0&&(f<0?e-=Nmb:e+=Nmb);}else e=f>0?Omb:unb;return e;};klb.Nm=function Nm(a,b){klb.wj();var c;c=a-b;while(c<hob)c+=Mmb;while(c>Nmb)c-=Mmb;return c;};klb.Om=function Om(a,b){klb.wj();var c,d,e,f,g;if((b&256)!=0&&flb.j5(a,'?'))return 0;for(d=1;d<=128;d++)if(!flb.j5(a,Ynb)&&flb.k5(a,klb.qj[d]))return d;if((b&2)!=0)for(e=129;e<=144;e++)if(flb.k5(a,klb.qj[e]))return e;if((b&4)!=0)for(f=146;f<=148;f++)if(flb.k5(a,klb.qj[f]))return f;if((b&1)!=0)for(g=151;g<=152;g++)if(flb.k5(a,klb.qj[g]))return g;if((b&32)!=0)if(flb.k5(a,klb.qj[153]))return 153;if((b&8)!=0)if(flb.k5(a,klb.qj[154]))return 154;if((b&16)!=0)if(flb.k5(a,klb.qj[145]))return 145;if((b&128)!=0)if(flb.k5(a,klb.qj[159]))return 159;if((b&64)!=0)for(c=171;c<=190;c++)if(flb.k5(a,klb.qj[c]))return c;return 0;};klb.Pm=function Pm(a){klb.wj();switch(a){case 7:case 8:case 9:case 15:case 16:case 17:case 33:case 34:case 35:case 52:case 53:return true;}return false;};klb.Qm=function Qm(a){klb.wj();if(a==1||a==6)return false;if(klb.Pm(a))return false;if(a==2||a==10||a==18||a==36||a==54)return false;if(a>103)return false;return true;};klb.Rm=function Rm(a){return a>=3&&a<=4||a>=11&&a<=13||a>=19&&a<=31||a>=37&&a<=51||a>=55&&a<=84||a>=87&&a<=112;};klb.Sm=function Sm(a){return a==1||a>=5&&a<=9||a>=14&&a<=17||a>=32&&a<=35||a>=52&&a<=53;};klb.Tm=function Tm(a){return a>=21&&a<=30||a>=39&&a<=48||a==57||a>=72&&a<=80||a==89||a>=104&&a<=112;};b2(75,1,{75:1,4:1});_.ub=function Fm(a){klb.Tj(this,a);};_.vb=function Um(a){return this.H[a]==64;};_.q=0;_.r=0;_.I=0;_.K=false;_.L=false;_.M=0;_.N=0;_.O=0;_.S=false;_.T=0;_.U=0;_.V=0;klb.vj=24;elb.lY=D3(75);klb.Vm=function Vm(a,b,c,d){var e,f,g,h,i,j,k,l,m;klb.Jt(b,1);d==null&&(d=oW(elb.IX,omb,6,b.q,15,1));h=klb.Dl(a,1);i=klb.Dl(a,2);m=oW(elb.g1,Imb,6,b.q,16,1);j=oW(elb.IX,omb,6,b.q,15,1);j[0]=c;m[c]=true;d[c]=klb.Pj(b,a,c,h,i);g=0;k=0;while(g<=k){for(l=0;l<klb.hn(b,j[g]);l++){f=b.i[j[g]][l];if(!m[f]){j[++k]=f;m[f]=true;d[f]=klb.Pj(b,a,f,h,i);}}++g;}for(e=0;e<b.r;e++)m[b.D[0][e]]&&klb.Qj(b,a,e,h,i,d==null?b.D[0][e]:d[b.D[0][e]],d==null?b.D[1][e]:d[b.D[1][e]],false);klb.Dl(a,1);klb.Dl(a,2);a.T=0;};klb.Wm=function Wm(a){var b,c,d,e,f,g,h,i,j,k,l,m;a.j=oW(elb.IX,omb,6,a.q,15,1);a.e=oW(elb.IX,omb,6,a.q,15,1);a.i=oW(elb.IX,Lnb,7,a.q,0,2);a.k=oW(elb.IX,Lnb,7,a.q,0,2);a.n=oW(elb.IX,Lnb,7,a.q,0,2);a.o=oW(elb.IX,omb,6,a.f,15,1);j=oW(elb.IX,omb,6,a.q,15,1);for(g=0;g<a.r;g++){++j[a.D[0][g]];++j[a.D[1][g]];}for(d=0;d<a.q;d++){a.i[d]=oW(elb.IX,omb,6,j[d],15,1);a.k[d]=oW(elb.IX,omb,6,j[d],15,1);a.n[d]=oW(elb.IX,omb,6,j[d],15,1);}l=false;for(h=0;h<a.g;h++){m=klb.Hk(a,h);if(m==0){l=true;continue;}for(k=0;k<2;k++){c=a.D[k][h];b=a.e[c];a.n[c][b]=m;a.i[c][b]=a.D[1-k][h];a.k[c][b]=h;++a.e[c];++a.j[c];c<a.f&&(m>1?a.o[c]+=m-1:a.H[h]==64&&(a.o[c]=1));}}for(i=a.g;i<a.r;i++){m=klb.Hk(a,i);if(m==0){l=true;continue;}for(k=0;k<2;k++){c=a.D[k][i];b=a.e[c];a.n[c][b]=m;a.i[c][b]=a.D[1-k][i];a.k[c][b]=i;++a.e[c];a.D[1-k][i]<a.f&&++a.j[c];}}if(l){b=oW(elb.IX,omb,6,a.q,15,1);for(e=0;e<a.q;e++)b[e]=a.e[e];for(f=0;f<a.r;f++){m=klb.Hk(a,f);if(m==0){for(k=0;k<2;k++){c=a.D[k][f];a.n[c][b[c]]=m;a.i[c][b[c]]=a.D[1-k][f];a.k[c][b[c]]=f;++b[c];}}}}};klb.Xm=function Xm(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w;klb.Jt(a,1);c&&(b=true);for(i=0;i<a.r;i++){j=klb.Hk(a,i);if(j==1||j==2){if(a.s[a.D[0][i]]>0&&a.s[a.D[1][i]]<0){f=a.D[0][i];g=a.D[1][i];}else if(a.s[a.D[0][i]]<0&&a.s[a.D[1][i]]>0){f=a.D[1][i];g=a.D[0][i];}else continue;if(klb.kl(a,f)||klb.kl(a,g))continue;if(a.C[f]<9&&klb.Sn(a,f)>3||a.C[g]<9&&klb.Sn(a,g)>3)continue;l=klb.Mn(a,f)!=0;a.s[f]-=1;a.s[g]+=1;if(!l){s=a.H[i];j==1?a.H[i]=2:a.H[i]=4;if(s==129||s==257){w=a.D[0][i];r=klb.Lo(a,w,false);if(a.D[0][r]!=w){a.D[1][r]=a.D[0][r];a.D[1][r]=w;}}}a.T=0;}}t=0;p=0;n=0;for(e=0;e<a.q;e++){t+=a.s[e];if(a.s[e]<0&&!klb.ko(a,e)){++p;klb.hl(a,e)&&(n-=a.s[e]);}}if(!b&&t!=0)throw l1(new flb.uz("molecule's overall charges are not balanced"));klb.Jt(a,1);u=0;v=c?t+n:n;for(h=0;h<a.q;h++){if(a.s[h]>0){if(!klb.jo(a,h)&&klb.hl(a,h)){k=$wnd.Math.min(klb.Mn(a,h),a.s[h]);if(k!=0&&v>=k){t-=k;u-=k;v-=k;a.s[h]-=k;a.T&=1;}}}}q=c?t:u;if(q<0){o=oW(elb.IX,omb,6,p,15,1);p=0;for(f=0;f<a.q;f++){a.s[f]<0&&!klb.ko(a,f)&&(o[p++]=(a.C[f]<<22)+f);}ilb.rgb(o,ilb.Xgb(e2(llb.Ccb.prototype.kc,llb.Ccb,[])));for(m=o.length-1;q<0&&m>=o.length-p;m--){d=o[m]&iob;if(klb.hl(a,d)){k=$wnd.Math.min(-q,-a.s[d]);t+=k;q+=k;a.s[d]+=k;a.T&=1;}}}return t;};klb.Ym=function Ym(a){var b,c,d,e,f,g,h,i,j,k,l,m,n;if(!a.K)return false;for(c=0;c<a.q;c++){n=klb.Rk(a,c);n+klb.Pk(a,c,n,a.C[c])-klb.Sn(a,c)<=0&&!(a.s[c]==0&&(a.C[c]==5||(e=a.C[c],e==7||e==15||e==33)||(d=a.C[c],d==8||d==16||d==34||d==52)))&&(a.B[c]=n1(a.B[c],-6145));}g=false;for(b=0;b<a.f;b++){h=a.e[b]-a.j[b];if(!a.S&&h>0){if(x1(n1(a.B[b],tnb),0)){k=x1(n1(a.B[b],Umb),896)?3:x1(n1(a.B[b],Umb),384)?2:x1(n1(a.B[b],128),128)?1:0;i=(n=klb.Rk(a,b),n+klb.Pk(a,b,n,a.C[b])-klb.Sn(a,b));a.s[b]==0&&x1(n1(a.B[b],Wmb),0)&&a.C[b]!=6&&++i;l=h;l>3-k&&(l=3-k);l>i+h-k&&(l=i+h-k);if(l>0){m=k==0?0:L1(n1(a.B[b],Umb),l);m=K1(m,elb.v1((l==3?7:h==2?3:1)<<7));a.B[b]=n1(a.B[b],-1921);a.B[b]=K1(a.B[b],n1(Umb,m));}}for(j=a.j[b];j<a.e[b];j++){f=a.k[b][j];if(a.H[f]==1){a.C[a.i[b][j]]=-1;a.H[f]=512;g=true;}}}}g&&klb.Oj(a);return g;};klb.Zm=function Zm(a,b){var c,d,e,f;if(a.o[b]==2&&a.j[b]==2&&a.n[b][0]==2){for(e=0;e<2;e++){c=klb.cn(a,b,a.i[b][e]);if(c!=-1){for(f=0;f<a.j[c];f++){d=a.k[c][f];(a.H[d]==257||a.H[d]==129)&&a.D[0][d]==c&&(a.H[a.k[c][f]]=1);}}}return;}if(a.o[b]==0||a.C[b]>=15){for(e=0;e<a.e[b];e++){d=a.k[b][e];(a.H[d]==257||a.H[d]==129)&&a.D[0][d]==b&&a.D[0][d]==b&&(a.H[d]=1);}}};klb.$m=function $m(a,b,c,d,e){var f,g,h,i,j,k,l,m;klb.Jt(a,d?7:1);b.v=null;a.K&&klb.mm(b,true);i=c.length;e==null&&(e=oW(elb.IX,omb,6,i,15,1));b.q=0;for(f=0;f<i;f++){if(c[f]){e[f]=klb.Pj(a,b,f,0,0);if(a.C[f]==1){m=klb.Sn(a,f);if(m>1){for(l=0;l<a.e[f];l++){if(!c[a.i[f][l]]){klb.Hl(b,e[f],m);break;}}}}}else {e[f]=-1;}}k=oW(elb.IX,omb,6,a.r,15,1);llb.jcb(k,k.length,-1);b.r=0;for(j=0;j<a.r;j++){g=a.D[0][j];h=a.D[1][j];if(g<i&&h<i){if(c[g]&&c[h])k[j]=klb.Qj(a,b,j,0,0,e==null?a.D[0][j]:e[a.D[0][j]],e==null?a.D[1][j]:e[a.D[1][j]],d);else if(a.s[g]!=0&&a.s[h]!=0&&a.s[g]<0^a.s[h]<0){c[g]&&(b.s[e[g]]+=a.s[g]<0?1:-1);c[h]&&(b.s[e[h]]+=a.s[h]<0?1:-1);}}}klb.Tj(a,b);!!a.d&&(b.T=0);b.T=0;klb.Dl(b,1);klb.Dl(b,2);b.q!=i&&klb.mm(b,true);klb.Oo(a,b,i,e,k);klb.No(a,b,i,e);d&&klb.ce(new klb.me(b),null,false);};klb._m=function _m(a,b,c,d,e){var f,g,h,i,j,k,l,m;klb.Jt(a,d?7:1);b.v=null;a.K&&klb.mm(b,true);e==null&&(e=oW(elb.IX,omb,6,a.q,15,1));b.q=0;for(f=0;f<a.q;f++){e[f]=-1;for(k=0;k<a.j[f];k++){if(c[a.k[f][k]]){e[f]=klb.Pj(a,b,f,0,0);if(a.C[f]==1){m=klb.Sn(a,f);if(m>1){for(l=0;l<a.e[f];l++){if(!c[a.k[f][l]]){klb.Hl(b,e[f],m);break;}}}}break;}}}j=oW(elb.IX,omb,6,a.r,15,1);b.r=0;for(i=0;i<a.r;i++)if(c[i]){j[i]=klb.Qj(a,b,i,0,0,e==null?a.D[0][i]:e[a.D[0][i]],e==null?a.D[1][i]:e[a.D[1][i]],d);}else {j[i]=-1;g=a.D[0][i];h=a.D[1][i];if(e[g]==-1^e[h]==-1){if(a.s[g]!=0&&a.s[h]!=0&&a.s[g]<0^a.s[h]<0){e[g]!=-1&&(b.s[e[g]]+=a.s[g]<0?1:-1);e[h]!=-1&&(b.s[e[h]]+=a.s[h]<0?1:-1);}}}klb.Tj(a,b);!!a.d&&(b.T=0);b.T=0;klb.Dl(b,1);klb.Dl(b,2);b.q!=a.q&&klb.mm(b,true);klb.Oo(a,b,a.q,e,j);klb.No(a,b,a.q,e);d&&klb.ce(new klb.me(b),null,false);return e;};klb.an=function an(a,b){var c,d,e,f,g,h,i,j,k,l;if((b&~a.T)==0)return;if((a.T&1)==0){klb.io(a);klb.Wm(a);a.T|=1;if(klb.Ym(a)){klb.io(a);klb.Wm(a);}}if((b&~a.T)==0)return;if((a.T&-7)!=0){for(d=0;d<a.f;d++)a.u[d]&=-15369;for(f=0;f<a.g;f++)a.F[f]&=-705;if((b&4)==0){klb.fn(a,1);a.T|=2;return;}klb.fn(a,7);for(e=0;e<a.f;e++){for(k=0;k<a.j[e];k++){i=a.k[e][k];if(i<a.g&&klb.tr(a.p,i))continue;h=a.i[e][k];for(l=0;l<a.j[h];l++){if(a.k[h][l]==i)continue;a.n[h][l]>1&&(a.C[a.i[h][l]]==6?a.u[e]|=Qmb:!klb.po(a,a.k[h][l])&&klb.hl(a,a.i[h][l])&&(a.u[e]|=snb));}}}while(true){j=false;for(c=0;c<a.f;c++){if(a.o[c]>0&&(a.u[c]&snb)!=0&&!klb.sr(a.p,c)){for(k=0;k<a.j[c];k++){if(a.n[c][k]>1){h=a.i[c][k];i=a.k[c][k];for(l=0;l<a.j[h];l++){if(a.k[h][l]!=i){g=a.i[h][l];if((a.u[g]&snb)==0){a.u[g]|=snb;j=true;}}}}}}}if(!j)break;}}a.T|=6;};klb.bn=function bn(a,b){var c,d,e,f,g;c=-1;if(a.o[b]==1){for(f=0;f<a.j[b];f++){if(a.n[b][f]==2){d=a.i[b][f];if(a.j[d]==2&&a.o[d]==2){for(g=0;g<2;g++){e=a.i[d][g];if(e!=b&&a.o[e]==1){c=d;break;}}}break;}}}return c;};klb.cn=function cn(a,b,c){var d,e;d=b;while(a.j[c]==2&&a.o[c]==2&&c!=d){e=c;c=a.i[c][0]==b?a.i[c][1]:a.i[c][0];b=e;}return c==d?-1:c;};klb.dn=function dn(a,b){var c;if(a.j[b]==3&&b<a.f&&klb.sr(a.p,b)&&(!!a.p&&b<a.f?klb.hr(a.p,b):0)>=5)for(c=0;c<a.j[b];c++)if(klb.qo(a,a.k[b][c]))return a.k[b][c];return -1;};klb.en=function en(a,b,c,d,e){var f,g,h,i,j,k;klb.Jt(a,7);if((a.u[b]&job)==0||c&&!(b<a.f&&klb.sr(a.p,b)))return;i=oW(elb.IX,omb,6,a.f,15,1);i[0]=b;d[b]=true;h=0;j=0;while(h<=j){for(k=0;k<a.j[i[h]];k++){g=a.k[i[h]][k];if(!e[g]&&(a.F[g]&64)!=0&&(!c||g<a.g&&klb.tr(a.p,g))){e[g]=true;f=a.i[i[h]][k];if(!d[f]){d[f]=true;i[++j]=f;}}}++h;}};klb.fn=function fn(a,b){var c,d,e,f,g,h,i,j;a.p=new klb.Gr(a,b);d=oW(elb.IX,omb,6,a.f,15,1);for(e=0;e<a.g;e++){if(klb.jr(a.p,e)!=0){a.F[e]|=64;++d[a.D[0][e]];++d[a.D[1][e]];}}for(c=0;c<a.f;c++){d[c]==2?a.u[c]|=Vmb:d[c]==3?a.u[c]|=tnb:d[c]>3&&(a.u[c]|=job);}for(j=0;j<a.p.i.a.length;j++){g=klb.lr(a.p,j);i=klb.mr(a.p,j);h=g.length;for(f=0;f<h;f++){a.u[g[f]]|=8;a.F[i[f]]|=128;a.H[i[f]]==386&&(a.H[i[f]]=2);}}};klb.gn=function gn(a){var b,c,d,e,f,g,h,i;h=oW(elb.g1,Imb,6,a.q,16,1);for(c=0;c<a.q;c++)h[c]=a.C[c]==1&&a.A[c]==0&&a.s[c]==0&&(a.u[c]&Znb)==0&&(a.t==null||a.t[c]==null);i=oW(elb.g1,Imb,6,a.q,16,1);for(g=0;g<a.r;g++){d=a.D[0][g];e=a.D[1][g];if(klb.Hk(a,g)!=1){h[d]=false;h[e]=false;continue;}i[d]&&(h[d]=false);i[e]&&(h[e]=false);h[d]&&klb.kl(a,e)&&a.C[e]!=13&&(h[d]=false);h[e]&&klb.kl(a,d)&&a.C[d]!=13&&(h[e]=false);i[d]=true;i[e]=true;}for(f=0;f<a.r;f++){if(h[a.D[0][f]]&&h[a.D[1][f]]){h[a.D[0][f]]=false;h[a.D[1][f]]=false;}}for(b=0;b<a.q;b++)i[b]||(h[b]=false);return h;};klb.hn=function hn(a,b){return a.e[b];};klb.jn=function jn(a,b){return a.i[b].length;};klb.kn=function kn(a,b){return a.e[b]-klb.Rn(a,b)+klb.Mn(a,b);};klb.ln=function ln(a){var b,c;klb.Jt(a,7);b=0;for(c=0;c<a.p.i.a.length;c++)klb.rr(a.p,c)&&++b;return b;};klb.mn=function mn(a,b){var c,d;c=0;for(d=0;d<a.j[b];d++)klb.hl(a,a.i[b][d])&&!klb.cl(a,a.k[b][d])&&++c;return c;};klb.nn=function nn(a,b){return a.o[b];};klb.on=function on(a,b){klb.Jt(a,7);return a.o[b]==2&&a.j[b]==2?klb.Jo(a,b,false):klb.Lo(a,b,false);};klb.pn=function pn(a,b){var c;c=a.u[b]&job;return c==0?0:c==Vmb?2:c==tnb?3:4;};klb.qn=function qn(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o;klb.Jt(a,7);f=oW(elb.g1,Imb,6,a.g,16,1);l=oW(elb.g1,Imb,6,a.g,16,1);o=oW(elb.IX,omb,6,a.f,15,1);g=0;for(h=1;h<a.j[b];h++){d=a.k[b][h];if((a.F[d]&64)!=0){for(j=0;j<h;j++){e=a.k[b][j];if((a.F[e]&64)!=0){l[d]=true;l[e]=true;n=klb.Un(a,o,a.i[b][h],a.i[b][j],c-2,null,l);l[d]=false;l[e]=false;if(n!=-1){i=false;m=oW(elb.IX,omb,6,n,15,1);klb.Vn(a,o,m,n);for(k=0;k<n;k++){if(!f[m[k]]){f[m[k]]=true;i=true;}}i&&++g;}}}}}return g;};klb.rn=function rn(a,b){return !!a.p&&b<a.f?klb.hr(a.p,b):0;};klb.sn=function sn(a,b){if(b){klb.Jt(a,1);return klb.wk(a,a.f,a.g,klb.vj);}else {return klb.wk(a,a.q,a.r,klb.vj);}};klb.tn=function tn(a){var b,c,d,e,f,g,h,i;klb.Jt(a,1);h=oW(elb.HX,kob,6,a.f,15,1);d=oW(elb.IX,omb,6,a.f,15,1);for(i=0;i<a.f;i++){d[0]=i;e=oW(elb.IX,omb,6,a.f,15,1);e[i]=1;c=0;f=0;while(c<=f){for(g=0;g<a.j[d[c]];g++){b=a.i[d[c]][g];if(e[b]==0){e[b]=e[d[c]]+1;d[++f]=b;h[i]+=e[b]-1;}}++c;}h[i]/=f;}return h;};klb.un=function un(a,b,c){var d;for(d=0;d<a.i[b].length;d++)if(a.i[b][d]==c)return a.k[b][d];return -1;};klb.vn=function vn(a,b){return !!a.p&&b<a.g?klb.jr(a.p,b):0;};klb.wn=function wn(a,b,c){return a.i[b][c];};klb.xn=function xn(a,b){return a.j[b];};klb.yn=function yn(a,b,c){return a.k[b][c];};klb.zn=function zn(a,b,c){return a.n[b][c];};klb.An=function An(a,b){var c,d;c=0;for(d=0;d<a.j[b];d++)I1(n1(a.B[d],Jmb),0)&&++c;return c;};klb.Bn=function Bn(a,b){return a.e[b]-a.j[b];};klb.Cn=function Cn(a,b,c,d,e){var f,g,h,i;g=a.e[b];if(a.o[b]!=0||b<a.f&&klb.sr(a.p,b)||a.j[b]<3||g>4)return false;i=oW(elb.g1,Imb,6,4,16,1);for(h=0;h<g;h++){f=3.9269908169872414-d[h];if($wnd.Math.abs(lob-f%Omb)>0.0872664675116539)return false;e[h]=3&DX(f/Omb);if(i[e[h]])return false;i[e[h]]=true;if((e[h]&1)==0){if(a.H[a.k[b][c[h]]]!=1)return false;}else {if(!klb.rl(a,a.k[b][c[h]],b))return false;}}return i[0]&&i[2];};klb.Dn=function Dn(a,b,c,d,e){var f,g,h,i,j,k,l,m;if((!!a.p&&b<a.f?klb.hr(a.p,b):0)>24)return 3;f=a.e[b];e==null&&(e=oW(elb.IX,omb,6,f,15,1));if(!klb.Cn(a,b,c,d,e))return 3;i=-1;for(j=0;j<f;j++){if((e[j]&1)==1){g=a.H[a.k[b][c[j]]];if(i!=-1&&i!=g)return 3;i=g;}}k=$wnd.Math.abs(e[0]-e[1])==2?1:0;h=e[k]-e[k+1];m=$wnd.Math.abs(h)==3^e[k]<e[k+1];l=f==3||(e[3]&1)==1;return m^l^i==129?1:2;};klb.En=function En(a,b,c){var d,e,f,g;g=g=oW(elb.g1,Imb,6,a.q,16,1);f=klb.Fn(a,b,c,g);e=oW(elb.IX,omb,6,f,15,1);f=0;for(d=0;d<a.q;d++)g[d]&&(e[f++]=d);return e;};klb.Fn=function Fn(a,b,c,d){var e,f,g,h,i,j,k;klb.Jt(a,1);d==null&&(d=oW(elb.g1,Imb,6,a.q,16,1));i=oW(elb.IX,omb,6,a.q,15,1);i[0]=b;d[b]=true;g=0;j=0;h=1;while(g<=j){f=c?klb.jn(a,i[g]):a.e[i[g]];for(k=0;k<f;k++){e=a.i[i[g]][k];if(e<d.length&&!d[e]){i[++j]=e;d[e]=true;++h;}}++g;}return h;};klb.Gn=function Gn(a,b,c,d){var e,f,g,h,i,j,k,l,m;klb.Jt(a,1);for(f=0;f<a.q;f++)b[f]=-1;j=0;for(e=0;e<a.q;e++){if(b[e]==-1&&(!c||(a.u[e]&Onb)!=0)){b[e]=j;k=oW(elb.IX,omb,6,a.q,15,1);k[0]=e;i=0;l=0;while(i<=l){h=d?klb.jn(a,k[i]):a.e[k[i]];for(m=0;m<h;m++){g=a.i[k[i]][m];if(b[g]==-1&&(!c||(a.u[g]&Onb)!=0)){k[++l]=g;b[g]=j;}}++i;}++j;}}return j;};klb.Hn=function Hn(a,b,c,d){var e,f,g,h,i,j,k,l,m;klb.Jt(a,1);for(f=0;f<a.q;f++)b[f]=-1;j=0;for(e=0;e<a.q;e++){if(b[e]==-1){b[e]=j;k=oW(elb.IX,omb,6,a.q,15,1);k[0]=e;i=0;l=0;while(i<=l){h=d?klb.jn(a,k[i]):a.e[k[i]];for(m=0;m<h;m++){g=a.i[k[i]][m];if(b[g]==-1&&!c[a.k[k[i]][m]]){k[++l]=g;b[g]=j;}}++i;}++j;}}return j;};klb.In=function In(a,b){var c;return c=klb.Rk(a,b),c+klb.Pk(a,b,c,a.C[b])-klb.Sn(a,b);};klb.Jn=function Jn(a,b){var c,d,e,f,g,h;f=oW(elb.IX,omb,6,a.q,15,1);for(d=0;d<a.q;d++)f[d]=d;e=a.q;do--e;while(e>=0&&b[e]);for(c=0;c<e;c++){if(b[f[c]]){h=f[c];f[c]=f[e];f[e]=h;g=b[c];b[c]=b[e];b[e]=g;do--e;while(b[e]);}}return f;};klb.Kn=function Kn(a){return klb.Jn(a,klb.gn(a));};klb.Ln=function Ln(a,b,c){var d,e,f;e=klb.Sn(a,b);e-=klb.Pk(a,b,e,a.C[b]);c&&(e-=a.e[b]-a.j[b]);f=klb.Lm(a.C[b]);if(e<=f[0])return -1;for(d=1;d<f.length;d++)if(f[d]>=e)return f[d];return e;};klb.Mn=function Mn(a,b){var c,d,e,f,g,h;if(a.K&&x1(n1(a.B[b],tnb),0))return 0;if(!klb.Wo(a,b))return 0;klb.Jt(a,1);g=0;for(e=0;e<a.e[b];e++)g+=a.n[b][e];if(a.K){c=1;for(d=0;d<a.j[b];d++)a.H[a.k[b][d]]==64&&++c;g+=c>>1;}g-=klb.Pk(a,b,g,a.C[b]);f=((a.u[b]&Znb)>>>27)-1;if(f==-1){h=klb.Lm(a.C[b]);f=h[0];for(d=1;f<g&&d<h.length;d++)f=h[d];}return $wnd.Math.max(0,f-g);};klb.Nn=function Nn(a,b){var c,d,e,f,g;if(!a.K||a.v==null||a.v[b]==null)return klb.On(a,b,a.C[b]);g=0;for(d=a.v[b],e=0,f=d.length;e<f;++e){c=d[e];g=$wnd.Math.max(g,klb.On(a,b,c));}return g;};klb.On=function On(a,b,c){var d,e,f,g,h;f=klb.Sn(a,b);d=klb.Pk(a,b,f,c);g=((a.u[b]&Znb)>>>27)-1;if(g==-1){h=klb.Lm(a.C[b]);e=0;while(f>h[e]+d&&e<h.length-1)++e;g=h[e];}return g+d-f;};klb.Pn=function Pn(a,b){return a.i[b].length-a.e[b];};klb.Qn=function Qn(a){var b,c,d,e;klb.Jt(a,1);e=0;for(b=0;b<a.q;b++){d=a.A[b]!=0?a.A[b]:klb.uj[a.C[b]];e+=d+klb.Mn(a,b)*klb.uj[1];if(a.C[b]>=171&&a.C[b]<=190){c=a.e[b];c>2&&(e-=(c-2)*klb.uj[1]);}}return e;};klb.Rn=function Rn(a,b){var c,d;c=a.j[b];for(d=0;d<a.j[b];d++)a.C[a.i[b][d]]==1&&--c;return c;};klb.Sn=function Sn(a,b){var c,d,e,f,g,h;a.wb(1);g=false;d=false;h=0;for(e=0;e<a.e[b];e++){if(!a.K||x1(n1(a.B[a.i[b][e]],Jmb),0)){f=a.n[b][e];h+=f;f>1&&(g=true);c=a.k[b][e];a.H[c]==64&&(d=true);}}d&&!g&&++h;return h;};klb.Tn=function Tn(a,b,c){var d,e,f;e=0;for(f=0;f<a.j[b];f++){d=a.i[b][f];d!=c&&a.j[d]>2&&++e;}return e;};klb.Un=function Un(a,b,c,d,e,f,g){var h,i,j,k,l,m,n,o,p;if(c==d){b[0]=c;return 0;}a.wb(1);k=oW(elb.IX,omb,6,a.q,15,1);j=oW(elb.IX,omb,6,a.q,15,1);p=oW(elb.IX,omb,6,a.q,15,1);j[0]=c;k[c]=1;i=0;l=0;while(i<=l&&k[j[i]]<=e){o=j[i];for(m=0;m<a.e[o];m++){if(g==null||g.length<=a.k[o][m]||!g[a.k[o][m]]){h=a.i[o][m];if(h==d){n=k[o];b[n]=h;b[--n]=o;while(n>0){b[n-1]=p[b[n]];--n;}return k[o];}if(k[h]==0&&(f==null||f.length<=h||!f[h])){j[++l]=h;k[h]=k[o]+1;p[h]=o;}}}++i;}return -1;};klb.Vn=function Vn(a,b,c,d){var e,f;klb.Jt(a,1);for(e=0;e<d;e++){for(f=0;f<a.e[b[e]];f++){if(a.i[b[e]][f]==b[e+1]){c[e]=a.k[b[e]][f];break;}}}};klb.Wn=function Wn(a,b,c){var d,e,f,g,h,i;if(b==c)return 0;klb.Jt(a,1);g=oW(elb.IX,omb,6,a.q,15,1);f=oW(elb.IX,omb,6,a.q,15,1);f[0]=b;g[b]=1;e=0;h=0;while(e<=h){for(i=0;i<a.e[f[e]];i++){d=a.i[f[e]][i];if(d==c)return g[f[e]];if(g[d]==0){f[++h]=d;g[d]=g[f[e]]+1;}}++e;}return -1;};klb.Xn=function Xn(a,b,c,d,e){var f,g,h,i,j,k;if(b==c)return 0;a.wb(1);i=oW(elb.IX,omb,6,a.q,15,1);h=oW(elb.IX,omb,6,a.q,15,1);h[0]=b;i[b]=1;g=0;j=0;while(g<=j&&i[h[g]]<=d){for(k=0;k<a.e[h[g]];k++){f=a.i[h[g]][k];if(f==c)return i[h[g]];if(i[f]==0&&(e==null||e.length<=f||!e[f])){h[++j]=f;i[f]=i[h[g]]+1;}}++g;}return -1;};klb.Yn=function Yn(a,b){return a.e[b]-a.j[b]+klb.Mn(a,b);};klb.Zn=function Zn(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o;o=oW(elb.IX,omb,6,16,15,1);c=oW(elb.GX,mmb,6,16,15,1);f=oW(elb.GX,mmb,6,2,15,1);d=0;for(k=0;k<2;k++){e=a.D[k][b];for(l=0;l<a.j[e];l++){i=a.k[e][l];if(i==b)continue;if(d==4)return 0;h=a.i[e][l];o[d]=16;a.o[h]!=0&&(o[d]+=(a.u[h]&job)!=0?1:4);if((a.F[b]&64)!=0&&(a.F[i]&64)!=0){m=klb.pr(a.p,b,i);m!=-1&&(o[d]+=klb.rr(a.p,m)?64:6);}c[d++]=klb.Mm(a.J[e].a,a.J[e].b,a.J[h].a,a.J[h].b);}}f[0]=klb.yk(a,a.D[0][b],a.D[1][b]);if(f[0]<0){f[1]=f[0]+Nmb;g=false;}else {f[1]=f[0];f[0]=f[1]-Nmb;g=true;}n=0;for(j=0;j<d;j++){c[j]>f[0]&&c[j]<f[1]?n-=o[j]:n+=o[j];}return g?-n:n;};klb.$n=function $n(a){klb.Jt(a,7);return a.p;};klb._n=function _n(a){var b,c,d,e,f,g,h,i,j;j=0;klb.Jt(a,7);for(d=0;d<a.g;d++){if(klb.Hk(a,d)==1&&(a.F[d]&64)==0){h=true;for(g=0;g<2;g++){b=a.D[g][d];if(a.j[b]==1){h=false;break;}if(a.C[b]==7&&!(b<a.f&&klb.sr(a.p,b))){c=a.D[1-g][d];for(i=0;i<a.j[c];i++){e=a.i[c][i];f=a.k[c][i];if(f!=d&&klb.Hk(a,f)>1&&!(e<a.f&&klb.sr(a.p,e))&&klb.hl(a,e)){h=false;break;}}}}h&&!klb.yo(a,d)&&++j;}}return j;};klb.ao=function ao(a,b,c){var d,e,f,g,h,i;e=0;for(f=0;f<a.j[b];f++){d=a.i[b][f];if(d!=c){g=0;for(h=0;h<a.j[d];h++){i=a.i[d][h];i!=b&&klb.po(a,a.k[d][h])&&a.j[i]>2&&++g;}g==2&&++e;}}return e;};klb.bo=function bo(a,b){var c,d,e,f;c=a.e[b];f=oW(elb.IX,omb,6,c,15,1);for(e=0;e<c;e++)f[e]=(a.i[b][e]<<16)+e;ilb.rgb(f,ilb.Xgb(e2(llb.Ccb.prototype.kc,llb.Ccb,[])));for(d=0;d<c;d++)f[d]&=fmb;return f;};klb.co=function co(a,b){var c,d;klb.Jt(a,1);if(a.j[b]==2&&a.n[b][0]==2&&a.n[b][1]==2){for(c=0;c<2;c++)for(d=0;d<a.e[a.i[b][c]];d++)if(klb.rl(a,a.k[a.i[b][c]][d],a.i[b][c]))return a.k[a.i[b][c]][d];}else {for(c=0;c<a.e[b];c++)if(klb.rl(a,a.k[b][c],b))return a.k[b][c];}return -1;};klb.eo=function eo(a,b,c){if(klb.Hk(a,b)!=1)return 0;return 16-a.e[c]+16*$wnd.Math.max(0,$wnd.Math.min(9,!!a.p&&b<a.g?klb.jr(a.p,b):0)-2)+((a.H[b]&384)==0||a.D[0][b]!=c?eob:0)+((a.u[c]&3)==0?Qmb:0)+(a.C[c]==1?tnb:0)+(a.e[c]==1?Vmb:0)+((a.F[b]&64)!=0?0:512)+(a.C[c]!=6?256:0);};klb.fo=function fo(a,b,c,d,e,f){var g,h,i,j,k;klb.Jt(a,1);if(e){klb.Nj(e);e.K=false;}i=oW(elb.IX,omb,6,a.q,15,1);d==null?d=oW(elb.g1,Imb,6,a.q,16,1):llb.mcb(d,d.length);i[0]=b;i[1]=c;d[b]=true;d[c]=true;h=1;j=1;while(h<=j){for(k=0;k<a.e[i[h]];k++){g=a.i[i[h]][k];if(g==b){if(h!=1)return -1;}if(!d[g]){d[g]=true;i[++j]=g;}}++h;}if(e){f==null&&(f=oW(elb.IX,omb,6,d.length,15,1));klb.$m(a,e,d,false,f);klb.Kj(e,f[b],0,0,-1,0);}d[b]=false;return j;};klb.go=function go(a,b,c){var d,e,f,g,h,i;klb.Jt(a,1);f=oW(elb.IX,omb,6,a.f,15,1);i=oW(elb.g1,Imb,6,a.f,16,1);f[0]=b;f[1]=c;i[b]=true;i[c]=true;e=1;g=1;while(e<=g){for(h=0;h<a.j[f[e]];h++){d=a.i[f[e]][h];if(d==b){if(e!=1)return -1;}if(!i[d]){i[d]=true;f[++g]=d;}}++e;}return g;};klb.ho=function ho(a,b,c){var d,e,f,g,h,i,j,k,l,m,n;if(klb.Hk(a,c)!=2&&!(c<a.g&&klb.tr(a.p,c)))return -1;n=a.F[c]&3;if(n!=1&&n!=2)return -1;for(i=0;i<2;i++){d=a.D[i][c];e=a.D[1-i][c];m=-1;g=false;for(k=0;k<a.j[d];k++){f=a.i[d][k];f!=e&&(f==b?g=true:m=f);}if(g){l=-1;h=-1;for(j=0;j<a.j[e];j++){f=a.i[e][j];if(f!=d){if(l==-1)l=f;else if(f>l)h=f;else {h=l;l=f;}}}if(a.j[d]==2){if(a.j[e]==2)return n==2?l:-1;return n==2?l:h;}else {if(a.j[e]==2)return n==2^b<m?-1:l;return n==2^b<m?h:l;}}}return -1;};klb.io=function io(a){var b,c,d,e,f,g,h,i,j,k;g=klb.gn(a);i=a.q;do--i;while(i>=0&&g[i]);for(b=0;b<i;b++){if(g[b]){klb.xm(a,b,i);k=g[b];g[b]=g[i];g[i]=k;do--i;while(g[i]);}}a.f=i+1;if(a.q==a.f){a.g=a.r;return;}h=oW(elb.g1,Imb,6,a.r,16,1);for(f=0;f<a.r;f++){c=a.D[0][f];d=a.D[1][f];(g[c]||g[d])&&(h[f]=true);}j=a.r;do--j;while(j>=0&&h[j]);for(e=0;e<j;e++){if(h[e]){klb.ym(a,e,j);h[e]=false;do--j;while(h[j]);}}a.g=j+1;};klb.jo=function jo(a,b){var c;for(c=0;c<a.j[b];c++)if(a.s[a.i[b][c]]<0)return true;return false;};klb.ko=function ko(a,b){var c;for(c=0;c<a.j[b];c++)if(a.s[a.i[b][c]]>0)return true;return false;};klb.lo=function lo(a,b){var c;c=a.C[b];return c==3||c==11||c==19||c==37||c==55;};klb.mo=function mo(a,b){return (a.u[b]&Qmb)!=0;};klb.no=function no(a,b){var c,d,e,f,g,h;klb.Jt(a,1);for(g=0;g<2;g++){c=a.D[g][b];if(a.C[c]==7){d=a.D[1-g][b];for(h=0;h<a.j[d];h++){e=a.i[d][h];f=a.k[d][h];if((a.C[e]==7||a.C[e]==8||a.C[e]==16)&&klb.Hk(a,f)>=2)return true;}}}return false;};klb.oo=function oo(a,b){return b<a.f&&klb.sr(a.p,b);};klb.po=function po(a,b){return b<a.g&&klb.tr(a.p,b);};klb.qo=function qo(a,b){var c,d,e,f,g,h;if(a.H[b]!=1||b<a.g&&klb.tr(a.p,b)||(a.F[b]&64)!=0&&(!!a.p&&b<a.g?klb.jr(a.p,b):0)<7)return false;c=a.D[0][b];if(!(c<a.f&&klb.sr(a.p,c))||(!!a.p&&c<a.f?klb.hr(a.p,c):0)<5)return false;d=a.D[1][b];if(!(d<a.f&&klb.sr(a.p,d))||(!!a.p&&d<a.f?klb.hr(a.p,d):0)<5)return false;e=klb.Tn(a,c,d);f=klb.Tn(a,d,c);if((!!a.p&&c<a.f?klb.hr(a.p,c):0)>5&&(!!a.p&&d<a.f?klb.hr(a.p,d):0)>5)return e+f>2;g=klb.ao(a,c,d);h=klb.ao(a,d,c);if(e==2&&h>=1)return true;if(f==2&&g>=1)return true;if(g==2&&(f>=1||h>=1))return true;if(h==2&&(e>=1||g>=1))return true;return false;};klb.ro=function ro(a,b){return a.o[b]==2&&a.j[b]==2&&a.n[b][0]==2&&a.n[b][1]==2&&a.C[b]<=7;};klb.so=function so(a,b){var c;c=a.C[b];return c==8||c==16||c==34||c==52;};klb.to=function to(a,b){return b<a.g&&(klb.xr(a.p,b)||a.H[b]==64);};klb.uo=function uo(a,b){var c;c=a.C[b];return c==12||c==20||c==38||c==56;};klb.vo=function vo(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p;if(a.C[b]!=7||a.j[b]==4)return false;if(b<a.f&&klb.sr(a.p,b)||a.o[b]!=0||I1(n1(a.B[b],rnb),0))return true;if(a.s[b]==1)return false;for(i=0;i<a.j[b];i++)if(klb.vn(a,a.k[b][i])==3)return false;g=0;for(j=0;j<a.j[b];j++){if(a.n[b][j]==1){d=a.C[a.i[b][j]];(d==8||d==9||d==17)&&++g;}}if(g==0){for(h=0;h<a.j[b];h++){e=a.i[b][h];if(a.o[e]!=0){if(e<a.f&&klb.sr(a.p,e)){if((!!a.p&&e<a.f?klb.hr(a.p,e):0)>=5){p=0;for(m=0;m<a.j[e];m++){o=a.i[e][m];o!=b&&klb.Rn(a,o)>=3&&++p;}n=klb.Rn(a,b);if(p==2&&n>=2||p==1&&n==3)continue;}return !c||!klb.Ao(a,b);}for(l=0;l<a.j[e];l++){if(a.n[e][l]==2||klb.po(a,a.k[e][l]))return !c||!klb.Ao(a,b);}}}}if(g<2){for(h=0;h<a.j[b];h++){e=a.i[b][h];k=false;f=false;for(l=0;l<a.j[e];l++){if(a.i[e][l]!=b){a.n[e][l]!=1&&(a.C[a.i[e][l]]==7||a.C[a.i[e][l]]==8||a.C[a.i[e][l]]==16)&&(k=true);a.n[e][l]==1&&a.C[a.i[e][l]]==7&&(f=true);}}if(k&&(!f||g==0))return !c||!klb.Ao(a,b);}}return false;};klb.wo=function wo(a,b){var c;c=a.C[b];return c==9||c==17||c==35||c==53;};klb.xo=function xo(a,b){var c;c=a.C[b];return c==7||c==15||c==33;};klb.yo=function yo(a,b){var c,d,e,f,g,h;if(klb.Hk(a,b)!=1)return false;for(f=0;f<2;f++){c=a.D[f][b];h=a.D[1-f][b];while(a.o[c]==2&&a.j[c]==2&&a.C[c]<10){for(g=0;g<2;g++){d=a.i[c][g];if(d!=h){if(a.j[d]==1)return true;e=a.k[c][g];if(klb.Hk(a,e)==1&&e<b)return true;h=c;c=d;break;}}}if(a.j[c]==1)return true;}return false;};klb.zo=function zo(a,b,c,d,e){var f,g,h,i,j,k,l,m,n,o,p,q,r,s;m=oW(elb.g1,Imb,6,a.g,16,1);m[e]=true;n=oW(elb.IX,omb,6,11,15,1);o=klb.Un(a,n,d,b,10,null,m);if(o==-1)return false;g=1;while(!klb.ur(a.p,c,n[g]))++g;f=o-g;h=n[g];s=klb.or(a.p,c);if(s==6&&f==2&&g==3){if(klb.pn(a,n[1])>=3){r=klb.lr(a.p,c);for(l=0;l<6;l++){if(b==r[l]){q=klb.Fr(a.p,c,h==r[klb.Fr(a.p,c,l+2)]?l-2:l+2);p=r[q];k=a.u[p]&job;if((k==0?0:k==Vmb?2:k==tnb?3:4)>=3&&klb.Xn(a,n[1],p,2,null)==2)return true;break;}}}}i=a.o[h]==1||h<a.f&&klb.sr(a.p,h)||klb.vo(a,h,false);j=!i&&a.C[h]==7&&a.s[h]!=1;if(f==1)return !i&&!j&&s<=4&&g<=3;switch(s){case 4:if(!i&&!j&&g<=4)return true;break;case 5:if(j){if(g<=3)return true;}else if(!i){if(g<=4)return true;}break;case 6:if(f==2){if(i){if(g<=4)return true;}else if(!j){if(g<=3)return true;}}else if(f==3){if(i){if(g<=6)return true;}else {if(g<=4)return true;}}break;case 7:if(f==3){if(g<=3)return true;}}return false;};klb.Ao=function Ao(a,b){var c,d,e,f,g;if(b<a.f&&klb.sr(a.p,b)||a.o[b]!=0||I1(n1(a.B[b],rnb),0)||(d=a.u[b]&job,(d==0?0:d==Vmb?2:d==tnb?3:4)<3))return false;g=!!a.p&&b<a.f?klb.hr(a.p,b):0;if(g>7)return false;f=0;while(f<a.p.i.a.length){if(klb.or(a.p,f)==g&&klb.ur(a.p,f,b))break;++f;}if(f>=Vmb&&f==a.p.i.a.length)return false;for(e=0;e<a.j[b];e++){c=a.k[b][e];if(!klb.vr(a.p,f,c)&&klb.zo(a,b,f,a.i[b][e],c))return true;}return false;};klb.Bo=function Bo(a,b){return (a.u[b]&job)!=0;};klb.Co=function Co(a,b){return (a.F[b]&64)!=0;};klb.Do=function Do(a,b){return a.C[b]==1&&a.A[b]==0&&a.s[b]==0&&(a.u[b]&Znb)==0&&(a.t==null||a.t[b]==null);};klb.Eo=function Eo(a,b){return (a.u[b]&8)!=0;};klb.Fo=function Fo(a,b){return (a.F[b]&128)!=0;};klb.Go=function Go(a,b){return (a.u[b]&snb)!=0;};klb.Ho=function Ho(a){var b,c,d,e,f,g,h,i,j,k,l;klb.Jt(a,1);klb.Io(a);j=false;for(c=0;c<a.f;c++){if(a.C[c]==7&&a.s[c]==0){l=klb.Sn(a,c);if(l==4){for(k=0;k<a.j[c];k++){h=a.i[c][k];if(a.n[c][k]==1&&a.C[h]==8&&a.j[h]==1&&a.s[h]==0){j=true;++a.s[c];--a.s[h];break;}}}else if(l==5){for(k=0;k<a.j[c];k++){h=a.i[c][k];i=a.k[c][k];if(a.n[c][k]==2&&a.C[h]==8){j=true;++a.s[c];--a.s[h];a.H[i]=1;break;}if(a.n[c][k]==3&&a.C[h]==7){j=true;++a.s[c];--a.s[h];a.H[i]=2;break;}}}}}g=false;for(f=0;f<a.g;f++){for(k=0;k<2;k++){if(klb.hl(a,a.D[k][f])){b=a.D[1-k][f];e=a.C[b];if(e==3||e==11||e==19||e==37||e==55||(d=a.C[b],d==12||d==20||d==38||d==56)){if(klb.Hk(a,f)==1){++a.s[b];--a.s[a.D[k][f]];a.H[f]=512;g=true;}else if(a.H[f]==32){a.H[f]=512;g=true;}}break;}}}if(g){klb.Oj(a);j=true;}j&&(a.T=0);return j;};klb.Io=function Io(a){var b;for(b=0;b<a.g;b++)if(a.H[b]==64)return klb.ce(new klb.me(a),null,false);return false;};klb.Jo=function Jo(a,b,c){var d,e,f,g,h,i,j,k;j=-1;e=0;for(h=0;h<2;h++){d=a.i[b][h];for(i=0;i<a.e[d];i++){f=a.i[d][i];if(f!=b){g=a.k[d][i];k=klb.eo(a,g,f);if(e<k&&(!c||!(a.H[g]==257||a.H[g]==129))){e=k;j=g;}}}}return j;};klb.Ko=function Ko(a,b){var c,d,e,f,g,h,i,j;i=-1;d=0;for(g=0;g<2;g++){c=a.D[g][b];for(h=0;h<a.e[c];h++){e=a.i[c][h];if(e!=a.D[1-g][b]){f=a.k[c][h];j=klb.eo(a,f,e);if(d<j){d=j;i=f;}}}}return i;};klb.Lo=function Lo(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s;d=a.e[b];e=oW(elb.GX,mmb,6,d,15,1);for(n=0;n<d;n++)e[n]=klb.yk(a,b,a.i[b][n]);for(o=1;o<d;o++){for(p=0;p<o;p++){f=$wnd.Math.abs(klb.Nm(e[o],e[p]));if(f<0.08||f>mob){g=0;h=0;for(q=0;q<d;q++){if(q!=o&&q!=p){g+=$wnd.Math.abs(Alb.HT(e[o],e[q]));h+=$wnd.Math.abs(Alb.HT(e[p],e[q]));}}j=g<h?a.k[b][o]:a.k[b][p];if(klb.Hk(a,j)==1&&(!c||!(a.H[j]==257||a.H[j]==129)))return j;}}}r=-1;i=0;for(m=0;m<d;m++){k=a.i[b][m];l=a.k[b][m];s=klb.eo(a,l,k);if(i<s&&(!c||!(a.H[l]==257||a.H[l]==129))){i=s;r=l;}}return r;};klb.Mo=function Mo(a,b){var c,d,e,f;klb.Jt(a,b?15:1);a.q=a.f;a.r=a.g;for(d=0;d<a.f;d++){if(a.e[d]!=a.j[d]){c=klb.Ln(a,d,false);a.e[d]=a.j[d];if(c!=-1){f=klb.Ln(a,d,true);if(c!=f){e=((a.u[d]&Znb)>>>27)-1;(e==-1||e<c)&&klb.Hl(a,d,c);}}}}b&&klb.Uo(a);a.T=0;};klb.No=function No(a,b,c,d){var e,f,g,h;klb.Jt(b,1);for(f=0;f<c;f++){if(d[f]!=-1&&a.C[f]!=1&&a.e[f]!=klb.hn(b,d[f])){e=klb.Ln(a,f,false);if(e!=-1){h=klb.Ln(b,d[f],false);if(e!=h){g=klb.bk(b,d[f]);(g==-1||g<e)&&klb.Hl(b,d[f],e);}}}}};klb.Oo=function Oo(a,b,c,d,e){var f,g,h,i,j,k,l,m,n,o,p;for(h=0;h<c;h++){if(d[h]!=-1&&a.e[h]>3&&(a.u[h]&dob)!=0){p=0;m=-1;l=-1;for(k=0;k<a.e[h];k++){if(d.length>a.i[h][k]&&d[a.i[h][k]]!=-1)++p;else if(a.n[h][k]==1&&klb.ql(a,a.k[h][k])&&a.D[0][a.k[h][k]]==h){m=a.k[h][k];l=a.i[h][k];}}if(m!=-1&&p>=3){f=klb.Mm(a.J[h].a,a.J[h].b,a.J[l].a,a.J[l].b);n=10;o=-1;for(j=0;j<a.e[h];j++){if(a.n[h][j]==1&&(!klb.ql(a,a.k[h][j])||a.D[0][a.k[h][j]]==h)&&d.length>a.i[h][j]&&d[a.i[h][j]]!=-1){g=$wnd.Math.abs(klb.Nm(f,klb.yk(a,h,a.i[h][j])));if(n>g){n=g;o=a.k[h][j];}}}if(o!=-1){i=e[o];klb.km(b,i,a.H[o]==257?129:257);if(a.D[0][o]!=h){klb.bm(b,1,i,d[a.D[0][o]]);klb.bm(b,0,i,d[h]);}}}}}};klb.Po=function Po(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A;if(a.j[b]!=2||a.n[b][0]!=2||a.n[b][1]!=2||a.j[a.i[b][0]]<2||a.j[a.i[b][1]]<2||a.o[a.i[b][0]]!=1||a.o[a.i[b][1]]!=1){klb.Ul(a,b,0,false);return;}v=-1;u=-1;t=-1;q=-1;f=0;for(l=0;l<2;l++){d=a.i[b][l];for(o=0;o<a.e[d];o++){g=a.i[d][o];if(g!=b){h=a.k[d][o];w=klb.eo(a,h,g);if(f<w){f=w;u=g;v=h;t=d;q=a.i[b][1-l];}}}}if(u==-1)return;for(m=0;m<2;m++){d=a.i[b][m];for(o=0;o<a.e[d];o++){g=a.i[d][o];h=a.k[d][o];g!=b&&a.D[0][h]==d&&(a.H[h]=1);}}if(a.D[1][v]!=u){a.D[0][v]=a.D[1][v];a.D[1][v]=u;}i=Ylb;for(n=0;n<a.j[t];n++){g=a.i[t][n];g!=b&&i>g&&(i=g);}r=oW(elb.IX,omb,6,2,15,1);s=0;for(k=0;k<a.j[q];k++){g=a.i[q][k];g!=b&&(r[s++]=g);}c=klb.Mm(a.J[b].a,a.J[b].b,a.J[q].a,a.J[q].b);e=0;if(s==2){if(r[0]>r[1]){A=r[0];r[0]=r[1];r[1]=A;}j=klb.Nm(c,klb.yk(a,q,r[0]));p=klb.Nm(c,klb.yk(a,q,r[1]));e=j-p;}else {e=klb.Nm(c,klb.yk(a,q,r[0]));}e<0^(a.u[b]&3)==1^i==u?a.H[v]=257:a.H[v]=129;};klb.Qo=function Qo(a,b,c,d){var e,f,g,h,i,j;e=a.e[b];h=oW(elb.IX,omb,6,e,15,1);j=klb.Dn(a,b,c,d,h);if(j==3)return false;g=(a.u[b]&3)==j?257:129;for(i=0;i<e;i++){if((h[i]&1)==1){f=a.k[b][c[i]];a.H[f]=g;if(a.D[0][f]!=b){a.D[1][f]=a.D[0][f];a.D[0][f]=b;}}}return true;};klb.Ro=function Ro(a,b){a.T|=248&(8|b);};klb.So=function So(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q;klb.Zm(a,b);if((a.u[b]&3)==0||(a.u[b]&3)==3)return;if(a.o[b]==2&&a.j[b]==2){klb.Po(a,b);return;}if(a.j[b]<3||a.j[b]>4){klb.Ul(a,b,0,false);return;}c=a.e[b];o=false;for(g=0;g<c;g++){if(klb.Hk(a,a.k[b][g])==1){o=true;break;}}if(!o)return;p=klb.bo(a,b);d=oW(elb.GX,mmb,6,c,15,1);for(h=0;h<c;h++)d[h]=klb.yk(a,a.i[b][p[h]],b);for(i=0;i<c;i++)a.D[0][a.k[b][i]]==b&&klb.Hk(a,a.k[b][i])==1&&(a.H[a.k[b][i]]=1);if((!!a.p&&b<a.f?klb.hr(a.p,b):0)<=24&&klb.Qo(a,b,p,d))return;m=klb.Lo(a,b,true);if(m==-1)return;if(a.D[0][m]!=b){a.D[1][m]=a.D[0][m];a.D[0][m]=b;}n=-1;for(j=0;j<c;j++){if(m==a.k[b][p[j]]){n=j;break;}}q=wW(iW(elb.IX,2),Lnb,7,0,[wW(iW(elb.IX,1),omb,6,15,[2,1,2,1]),wW(iW(elb.IX,1),omb,6,15,[1,2,2,1]),wW(iW(elb.IX,1),omb,6,15,[1,1,2,2]),wW(iW(elb.IX,1),omb,6,15,[2,1,1,2]),wW(iW(elb.IX,1),omb,6,15,[2,2,1,1]),wW(iW(elb.IX,1),omb,6,15,[1,2,1,2])]);for(f=1;f<c;f++)d[f]<d[0]&&(d[f]+=Mmb);if(c==3){k=false;switch(n){case 0:k=d[1]<d[2]&&d[2]-d[1]<Nmb||d[1]>d[2]&&d[1]-d[2]>Nmb;break;case 1:k=d[2]-d[0]>Nmb;break;case 2:k=d[1]-d[0]<Nmb;}e=(a.u[b]&3)==1^k?257:129;}else {l=0;d[1]<=d[2]&&d[2]<=d[3]?l=0:d[1]<=d[3]&&d[3]<=d[2]?l=1:d[2]<=d[1]&&d[1]<=d[3]?l=2:d[2]<=d[3]&&d[3]<=d[1]?l=3:d[3]<=d[1]&&d[1]<=d[2]?l=4:d[3]<=d[2]&&d[2]<=d[1]&&(l=5);e=(a.u[b]&3)==1^q[l][n]==1?129:257;}a.H[m]=e;};klb.To=function To(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A;if((a.F[b]&3)==0||(a.F[b]&3)==3||!klb.qo(a,b))return;v=-1;t=-1;u=-1;s=-1;e=0;for(l=0;l<2;l++){d=a.D[l][b];for(o=0;o<a.e[d];o++){h=a.k[d][o];if(h!=b&&klb.Hk(a,h)==1){g=a.i[d][o];w=klb.eo(a,h,g);if(e<w){e=w;t=g;v=h;u=d;s=a.D[1-l][b];}}}}if(t==-1)return;for(m=0;m<2;m++){for(o=0;o<a.e[a.D[m][b]];o++){h=a.k[a.D[m][b]][o];h!=b&&klb.Hk(a,h)==1&&(a.H[h]=1);}}if(a.D[1][v]!=t){a.D[0][v]=a.D[1][v];a.D[1][v]=t;}i=Ylb;for(n=0;n<a.j[u];n++){g=a.i[u][n];a.k[u][n]!=b&&i>g&&(i=g);}q=oW(elb.IX,omb,6,2,15,1);r=0;for(k=0;k<a.j[s];k++)a.k[s][k]!=b&&(q[r++]=a.i[s][k]);f=klb.Mm(a.J[u].a,a.J[u].b,a.J[s].a,a.J[s].b);c=0;if(r==2){if(q[0]>q[1]){A=q[0];q[0]=q[1];q[1]=A;}j=klb.Nm(f,klb.yk(a,s,q[0]));p=klb.Nm(f,klb.yk(a,s,q[1]));c=j-p;}else {c=klb.Nm(f,klb.yk(a,s,q[0]));}c<0^(a.F[b]&3)==2^i==t?a.H[v]=257:a.H[v]=129;};klb.Uo=function Uo(a){var b,c,d,e;klb.Jt(a,7);for(d=0;d<a.g;d++)(a.H[d]==257||a.H[d]==129)&&(a.H[d]=1);for(b=0;b<a.f;b++)klb.So(a,b);for(e=0;e<a.g;e++)klb.To(a,e);for(c=0;c<a.g;c++)a.H[c]==2&&(a.F[c]&3)==3&&(a.H[c]=386);};klb.Vo=function Vo(b,c){var d,e,f,g,h,i,j,k,l,m;i=oW(elb.IX,omb,6,b.q,15,1);h=klb.Gn(b,i,false,c);if(h<=1)return null;j=oW(elb.IX,omb,6,h,15,1);for(e=0;e<b.f;e++)++j[i[e]];l=0;m=j[0];for(k=1;k<h;k++){if(m<j[k]){m=j[k];l=k;}}for(d=0;d<b.q;d++)i[d]!=l&&(b.C[d]=-1);for(g=0;g<b.r;g++)(!c&&b.H[g]==32||i[b.D[0][g]]!=l)&&(b.H[g]=512);f=klb.Oj(b);b.T=0;try{klb.Xm(b,true,true);}catch(a){a=k1(a);if(!uX(a,15))throw l1(a);}return f;};klb.Wo=function Wo(a,b){if((a.u[b]&Znb)!=0)return true;if(a.C[b]==1)return false;return klb.ml(a,b)||a.C[b]==13||a.C[b]>=171;};klb.Xo=function Xo(a){var b,c,d,e,f,g,h,i,j,k;f=klb.wk(a,a.q,a.r,klb.vj);g=f*f/16;for(d=1;d<a.q;d++){for(e=0;e<d;e++){i=a.J[e].a-a.J[d].a;j=a.J[e].b-a.J[d].b;k=a.J[e].c-a.J[d].c;if(i*i+j*j+k*k<g)throw l1(new flb.uz('The distance between two atoms is too close.'));}}klb.Jt(a,1);b=0;for(c=0;c<a.f;c++){if(klb.Sn(a,c)>(h=klb.Rk(a,c),h+klb.Pk(a,c,h,a.C[c])))throw l1(new flb.uz('atom valence exceeded'));b+=a.s[c];}if(b!=0)throw l1(new flb.uz('unbalanced atom charge'));};klb.Yo=function Yo(a,b,c){var d;d=klb.Am(a,b,c);if(d&&c==386){klb.Jt(a,7);d=d&(a.F[b]&128)==0;}return d;};klb.Zo=function Zo(){klb.Dm.call(this);};klb.$o=function $o(a,b){klb.Em.call(this,a,b);};klb._o=function _o(a){klb.Em.call(this,!a?256:a.M,!a?256:a.N);!!a&&klb.Sj(a,this);};b2(91,75,{91:1,75:1,4:1});_.wb=function ap(a){klb.an(this,a);};_.vb=function bp(a){return klb.to(this,a);};_.f=0;_.g=0;elb.eY=D3(91);klb.cp=function cp(a,b){return $wnd.Math.pow(10,$wnd.Math.log(2000)*$wnd.Math.LOG10E*a/(b-1)-1);};klb.dp=function dp(a,b){var c,d;c=b;d=0;while(b!=0){if(a.d==0){a.f=(a.c[++a.e]&63)<<11;a.d=6;}d|=(emb&a.f)>>16-c+b;a.f<<=1;--b;--a.d;}return d;};klb.ep=function ep(a,b,c){a.d=6;a.e=c;a.c=b;a.f=(b[a.e]&63)<<11;};klb.fp=function fp(a,b){var c,d,e,f;d=b/2|0;e=a>=d;e&&(a-=d);f=b/32|0;c=f*a/(d-a);return e?-c:c;};klb.gp=function gp(a,b){var c;return b==null||ilb.Xgb(b).length==0?null:klb.ip(a,ilb.ygb((c=b,Klb.v8(),c)),null);};klb.hp=function hp(a,b,c){var d,e;return b==null?null:klb.ip(a,ilb.ygb((e=b,Klb.v8(),e)),c==null?null:ilb.ygb((d=c,d)));};klb.ip=function ip(a,b,c){var d,e,f,g,h;klb.ep(a,b,0);d=klb.dp(a,4);g=klb.dp(a,4);d>8&&(d=g);e=klb.dp(a,d);f=klb.dp(a,g);h=new klb.au(e,f);klb.lp(a,h,b,c,0);return h;};klb.jp=function jp(a,b,c){var d,e,f,g;if(c==null||ilb.Xgb(c).length==0){klb.kp(a,b,null,null);return;}d=flb.n5(c,z5(32));d>0&&d<ilb.Xgb(c).length-1?klb.kp(a,b,ilb.ygb((f=(ilb.Rgb(0,d,ilb.Xgb(c).length),ilb.Xgb(c).substr(0,d)),Klb.v8(),f)),ilb.ygb((g=(ilb.Sgb(d+1,ilb.Xgb(c).length+1),ilb.Xgb(c).substr(d+1)),g))):klb.kp(a,b,ilb.ygb((e=c,Klb.v8(),e)),null);};klb.kp=function kp(a,b,c,d){if(c==null||c.length==0){klb.Nj(b);return;}klb.lp(a,b,c,d,0);};klb.lp=function lp(b,c,d,e,f){var g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N,O,P,Q,R,S,T,U,V,W,X,Y,Z,$,ab,bb,cb,db,eb,fb,gb,hb,ib,jb,kb,lb,mb,nb,ob,pb,qb,rb,sb,tb,ub,vb,wb,xb,yb,zb,Ab,Bb,Cb,Db,Eb,Fb,Gb,Hb,Ib,Jb,Kb,Lb,Mb,Nb,Ob,Pb,Qb,Rb,Sb,Tb,Ub,Vb,Wb,Xb,Yb,Zb,$b,_b,ac,bc,cc,dc,ec,fc,gc,hc,ic,jc,kc,lc,mc,nc,oc,pc,qc,rc,sc,tc,uc,vc,wc,xc,yc,zc,Ac;klb.Nj(c);if(d==null||0>=d.length)return;b.g=c;uc=8;e!=null&&(f<0||f>=e.length)&&(e=null);klb.ep(b,d,0);i=klb.dp(b,4);B=klb.dp(b,4);if(i>8){uc=i;i=B;}if(i==0){klb.mm(b.g,klb.dp(b,1)==1);return;}j=klb.dp(b,i);k=klb.dp(b,B);ic=klb.dp(b,i);mc=klb.dp(b,i);lc=klb.dp(b,i);O=klb.dp(b,i);for(o=0;o<j;o++)klb.yj(b.g,6);for(kb=0;kb<ic;kb++)klb.am(b.g,klb.dp(b,i),7);for(lb=0;lb<mc;lb++)klb.am(b.g,klb.dp(b,i),8);for(wb=0;wb<lc;wb++)klb.am(b.g,klb.dp(b,i),klb.dp(b,8));for(Hb=0;Hb<O;Hb++)klb.Jl(b.g,klb.dp(b,i),klb.dp(b,4)-8);P=1+k-j;V=klb.dp(b,4);A=0;klb.Zl(b.g,0,0);klb.$l(b.g,0,0);klb._l(b.g,0,0);W=e!=null&&e[f]>=39;tc=0;wc=0;yc=0;Ac=0;R=false;S=false;if(W){if(e.length>2*j-2&&e[2*j-2]==39||e.length>3*j-3&&e[3*j-3]==39){S=true;R=e.length==3*j-3+9;_b=R?3*j-3:2*j-2;w=86*(e[_b+1]-40)+e[_b+2]-40;tc=$wnd.Math.pow(10,w/2000-1);_b+=2;vc=86*(e[_b+1]-40)+e[_b+2]-40;wc=$wnd.Math.pow(10,vc/1500-1);_b+=2;xc=86*(e[_b+1]-40)+e[_b+2]-40;yc=$wnd.Math.pow(10,xc/1500-1);if(R){_b+=2;zc=86*(e[_b+1]-40)+e[_b+2]-40;Ac=$wnd.Math.pow(10,zc/1500-1);}}else {R=e.length==3*j-3;}}if(b.xb()&&R){e=null;W=false;}for(Sb=1;Sb<j;Sb++){X=klb.dp(b,V);if(X==0){if(W){klb.Zl(b.g,Sb,klb.rk(b.g,0)+8*(e[Sb*2-2]-83));klb.$l(b.g,Sb,klb.sk(b.g,0)+8*(e[Sb*2-1]-83));R&&klb._l(b.g,Sb,klb.tk(b.g,0)+8*(e[2*j-3+Sb]-83));}++P;continue;}A+=X-1;if(W){klb.Zl(b.g,Sb,klb.rk(b.g,A)+e[Sb*2-2]-83);klb.$l(b.g,Sb,klb.sk(b.g,A)+e[Sb*2-1]-83);R&&klb._l(b.g,Sb,klb.tk(b.g,A)+(e[2*j-3+Sb]-83));}klb.Aj(b.g,A,Sb,1);}for(Wb=0;Wb<P;Wb++)klb.Aj(b.g,klb.dp(b,i),klb.dp(b,i),1);ac=oW(elb.g1,Imb,6,k,16,1);for(I=0;I<k;I++){J=klb.dp(b,2);switch(J){case 0:ac[I]=true;break;case 2:klb.km(b.g,I,2);break;case 3:klb.km(b.g,I,4);}}h=klb.dp(b,i);for(Xb=0;Xb<h;Xb++){n=klb.dp(b,i);if(uc==8){nc=klb.dp(b,2);if(nc==3){klb.Ol(b.g,n,1,0);klb.Ul(b.g,n,1,false);}else {klb.Ul(b.g,n,nc,false);}}else {nc=klb.dp(b,3);switch(nc){case 4:klb.Ul(b.g,n,1,false);klb.Ol(b.g,n,1,klb.dp(b,3));break;case 5:klb.Ul(b.g,n,2,false);klb.Ol(b.g,n,1,klb.dp(b,3));break;case 6:klb.Ul(b.g,n,1,false);klb.Ol(b.g,n,2,klb.dp(b,3));break;case 7:klb.Ul(b.g,n,2,false);klb.Ol(b.g,n,2,klb.dp(b,3));break;default:klb.Ul(b.g,n,nc,false);}}}uc==8&&klb.dp(b,1)==0&&(b.g.L=true);g=klb.dp(b,B);for(Yb=0;Yb<g;Yb++){D=klb.dp(b,B);if(klb.Kk(b.g,D)==1){nc=klb.dp(b,3);switch(nc){case 4:klb.hm(b.g,D,1,false);klb.em(b.g,D,1,klb.dp(b,3));break;case 5:klb.hm(b.g,D,2,false);klb.em(b.g,D,1,klb.dp(b,3));break;case 6:klb.hm(b.g,D,1,false);klb.em(b.g,D,2,klb.dp(b,3));break;case 7:klb.hm(b.g,D,2,false);klb.em(b.g,D,2,klb.dp(b,3));break;default:klb.hm(b.g,D,nc,false);}}else {klb.hm(b.g,D,klb.dp(b,2),false);}}klb.mm(b.g,klb.dp(b,1)==1);m=null;kc=0;while(klb.dp(b,1)==1){U=kc+klb.dp(b,4);switch(U){case 0:jc=klb.dp(b,i);for(Zb=0;Zb<jc;Zb++){n=klb.dp(b,i);klb.Vl(b.g,n,tnb,true);}break;case 1:jc=klb.dp(b,i);for($b=0;$b<jc;$b++){n=klb.dp(b,i);gc=klb.dp(b,8);klb.Tl(b.g,n,gc);}break;case 2:jc=klb.dp(b,B);for(mb=0;mb<jc;mb++){klb.dp(b,B);}break;case 3:jc=klb.dp(b,i);for(nb=0;nb<jc;nb++){n=klb.dp(b,i);klb.Vl(b.g,n,Qmb,true);}break;case 4:jc=klb.dp(b,i);for(ob=0;ob<jc;ob++){n=klb.dp(b,i);rc=L1(z1(klb.dp(b,4)),3);klb.Vl(b.g,n,rc,true);}break;case 5:jc=klb.dp(b,i);for(pb=0;pb<jc;pb++){n=klb.dp(b,i);l=L1(z1(klb.dp(b,2)),1);klb.Vl(b.g,n,l,true);}break;case 6:jc=klb.dp(b,i);for(qb=0;qb<jc;qb++){n=klb.dp(b,i);klb.Vl(b.g,n,1,true);}break;case 7:jc=klb.dp(b,i);for(rb=0;rb<jc;rb++){n=klb.dp(b,i);hb=L1(z1(klb.dp(b,4)),7);klb.Vl(b.g,n,hb,true);}break;case 8:jc=klb.dp(b,i);for(sb=0;sb<jc;sb++){n=klb.dp(b,i);t=klb.dp(b,4);r=oW(elb.IX,omb,6,t,15,1);for(bc=0;bc<t;bc++){s=klb.dp(b,8);r[bc]=s;}klb.Pl(b.g,n,r);}break;case 9:jc=klb.dp(b,B);for(tb=0;tb<jc;tb++){D=klb.dp(b,B);rc=klb.dp(b,2)<<7;klb.jm(b.g,D,rc,true);}break;case 10:jc=klb.dp(b,B);for(ub=0;ub<jc;ub++){D=klb.dp(b,B);L=klb.dp(b,5);klb.jm(b.g,D,L,true);}break;case 11:jc=klb.dp(b,i);for(vb=0;vb<jc;vb++){n=klb.dp(b,i);klb.Vl(b.g,n,snb,true);}break;case 12:jc=klb.dp(b,B);for(xb=0;xb<jc;xb++){D=klb.dp(b,B);M=klb.dp(b,8)<<9;klb.jm(b.g,D,M,true);}break;case 13:jc=klb.dp(b,i);for(yb=0;yb<jc;yb++){n=klb.dp(b,i);oc=L1(z1(klb.dp(b,3)),14);klb.Vl(b.g,n,oc,true);}break;case 14:jc=klb.dp(b,i);for(zb=0;zb<jc;zb++){n=klb.dp(b,i);hc=L1(z1(klb.dp(b,5)),17);klb.Vl(b.g,n,hc,true);}break;case 15:case 31:kc+=16;break;case 16:jc=klb.dp(b,i);for(Ab=0;Ab<jc;Ab++){n=klb.dp(b,i);qc=L1(z1(klb.dp(b,3)),22);klb.Vl(b.g,n,qc,true);}break;case 17:jc=klb.dp(b,i);for(Bb=0;Bb<jc;Bb++){n=klb.dp(b,i);klb.Hl(b.g,n,klb.dp(b,4));}break;case 18:jc=klb.dp(b,i);fc=klb.dp(b,4);for(Cb=0;Cb<jc;Cb++){n=klb.dp(b,i);T=klb.dp(b,fc);cc=oW(elb.EX,Nnb,6,T,15,1);for(bc=0;bc<T;bc++)cc[bc]=klb.dp(b,7)<<24>>24;klb.Ml(b.g,n,flb.h5((dc=cc,Klb.v8(),dc)));}break;case 19:jc=klb.dp(b,i);for(Db=0;Db<jc;Db++){n=klb.dp(b,i);N=L1(z1(klb.dp(b,3)),25);klb.Vl(b.g,n,N,true);}break;case 20:jc=klb.dp(b,B);for(Eb=0;Eb<jc;Eb++){D=klb.dp(b,B);qc=klb.dp(b,3)<<17;klb.jm(b.g,D,qc,true);}break;case 21:jc=klb.dp(b,i);for(Fb=0;Fb<jc;Fb++){n=klb.dp(b,i);klb.Wl(b.g,n,klb.dp(b,2)<<4);}break;case 22:jc=klb.dp(b,i);for(Gb=0;Gb<jc;Gb++){n=klb.dp(b,i);klb.Vl(b.g,n,rnb,true);}break;case 23:jc=klb.dp(b,B);for(Ib=0;Ib<jc;Ib++){D=klb.dp(b,B);klb.jm(b.g,D,Qnb,true);}break;case 24:jc=klb.dp(b,B);for(Jb=0;Jb<jc;Jb++){D=klb.dp(b,B);l=klb.dp(b,2)<<21;klb.jm(b.g,D,l,true);}break;case 25:for(Kb=0;Kb<j;Kb++)klb.dp(b,1)==1&&klb.Xl(b.g,Kb,true);break;case 26:jc=klb.dp(b,B);m=oW(elb.IX,omb,6,jc,15,1);for(Lb=0;Lb<jc;Lb++)m[Lb]=klb.dp(b,B);break;case 27:jc=klb.dp(b,i);for(Mb=0;Mb<jc;Mb++){n=klb.dp(b,i);klb.Vl(b.g,n,Jmb,true);}break;case 28:jc=klb.dp(b,B);for(Nb=0;Nb<jc;Nb++)klb.km(b.g,klb.dp(b,B),32);break;case 29:jc=klb.dp(b,i);for(Ob=0;Ob<jc;Ob++){n=klb.dp(b,i);gb=L1(z1(klb.dp(b,2)),30);klb.Vl(b.g,n,gb,true);}break;case 30:jc=klb.dp(b,i);for(Pb=0;Pb<jc;Pb++){n=klb.dp(b,i);qc=L1(z1(klb.dp(b,7)),32);klb.Vl(b.g,n,qc,true);}break;case 32:jc=klb.dp(b,i);for(Qb=0;Qb<jc;Qb++){n=klb.dp(b,i);sc=L1(z1(klb.dp(b,2)),44);klb.Vl(b.g,n,sc,true);}break;case 33:jc=klb.dp(b,i);for(Rb=0;Rb<jc;Rb++){n=klb.dp(b,i);bb=L1(z1(klb.dp(b,5)),39);klb.Vl(b.g,n,bb,true);}break;case 34:jc=klb.dp(b,i);for(Tb=0;Tb<jc;Tb++){n=klb.dp(b,i);klb.Vl(b.g,n,Tmb,true);}break;case 35:jc=klb.dp(b,B);for(Ub=0;Ub<jc;Ub++){D=klb.dp(b,B);klb.jm(b.g,D,Tnb,true);}break;case 36:jc=klb.dp(b,B);for(Vb=0;Vb<jc;Vb++){D=klb.dp(b,B);K=klb.dp(b,2)<<5;klb.jm(b.g,D,K,true);}break;case 37:jc=klb.dp(b,B);for(jb=0;jb<jc;jb++){D=klb.dp(b,B);K=klb.dp(b,1)==0?8:16;klb.km(b.g,D,K);}}}klb.ce(new klb.me(b.g),ac,false);if(m!=null)for(F=m,G=0,H=F.length;G<H;++G){D=F[G];klb.km(b.g,D,klb.Kk(b.g,D)==2?4:2);}if(e==null&&!b.i&&d.length>b.e+1&&(d[b.e+1]==32||d[b.e+1]==9)){e=d;f=b.e+2;}if(e!=null){try{if(e[f]==33||e[f]==35){klb.ep(b,e,f+1);R=klb.dp(b,1)==1;S=klb.dp(b,1)==1;pc=2*klb.dp(b,4);C=1<<pc;eb=0;D=0;for(p=1;p<j;p++){if(D<k&&klb.zk(b.g,1,D)==p){eb=klb.zk(b.g,0,D++);db=1;}else {eb=0;db=8;}klb.Zl(b.g,p,klb.rk(b.g,eb)+db*(klb.dp(b,pc)-(C/2|0)));klb.$l(b.g,p,klb.sk(b.g,eb)+db*(klb.dp(b,pc)-(C/2|0)));R&&klb._l(b.g,p,klb.tk(b.g,eb)+db*(klb.dp(b,pc)-(C/2|0)));}if(e[f]==35){ib=0;fb=oW(elb.IX,omb,6,j,15,1);for(q=0;q<j;q++)ib+=fb[q]=klb.Mn(b.g,q);for(n=0;n<j;n++){for(jb=0;jb<fb[n];jb++){hb=klb.yj(b.g,1);klb.Aj(b.g,n,hb,1);klb.Zl(b.g,hb,klb.rk(b.g,n)+(klb.dp(b,pc)-(C/2|0)));klb.$l(b.g,hb,klb.sk(b.g,n)+(klb.dp(b,pc)-(C/2|0)));R&&klb._l(b.g,hb,klb.tk(b.g,n)+(klb.dp(b,pc)-(C/2|0)));}}j+=ib;k+=ib;}v=R?1.5:(klb.wj(),klb.wj(),klb.vj);u=klb.wk(b.g,j,k,v);if(S){tc=klb.cp(klb.dp(b,pc),C);wc=tc*klb.fp(klb.dp(b,pc),C);yc=tc*klb.fp(klb.dp(b,pc),C);R&&(Ac=tc*klb.fp(klb.dp(b,pc),C));db=tc/u;for(n=0;n<j;n++){klb.Zl(b.g,n,wc+db*klb.rk(b.g,n));klb.$l(b.g,n,yc+db*klb.sk(b.g,n));R&&klb._l(b.g,n,Ac+db*klb.tk(b.g,n));}}else {tc=1.5;db=tc/u;for(n=0;n<j;n++){klb.Zl(b.g,n,db*klb.rk(b.g,n));klb.$l(b.g,n,db*klb.sk(b.g,n));R&&klb._l(b.g,n,db*klb.tk(b.g,n));}}}else {R&&!S&&tc==0&&(tc=1.5);if(tc!=0&&b.g.r!=0){u=0;for(D=0;D<b.g.r;D++){Y=klb.rk(b.g,klb.zk(b.g,0,D))-klb.rk(b.g,klb.zk(b.g,1,D));Z=klb.sk(b.g,klb.zk(b.g,0,D))-klb.sk(b.g,klb.zk(b.g,1,D));$=R?klb.tk(b.g,klb.zk(b.g,0,D))-klb.tk(b.g,klb.zk(b.g,1,D)):0;u+=$wnd.Math.sqrt(Y*Y+Z*Z+$*$);}u/=b.g.r;cb=tc/u;for(n=0;n<b.g.q;n++){klb.Zl(b.g,n,klb.rk(b.g,n)*cb+wc);klb.$l(b.g,n,klb.sk(b.g,n)*cb+yc);R&&klb._l(b.g,n,klb.tk(b.g,n)*cb+Ac);}}}}catch(a){a=k1(a);if(uX(a,15)){ab=a;flb.cz(ab,(flb.V5(),flb.U5),'');'Faulty id-coordinates:'+ab+' '+flb.h5((ec=d,Klb.v8(),ec))+' '+flb.h5((dc=e,dc));e=null;R=false;}else throw l1(a);}}Q=e!=null&&!R;if(Q||b.xb()){klb.Jt(b.g,7);for(D=0;D<b.g.g;D++)klb.Hk(b.g,D)==2&&!klb.Fo(b.g,D)&&klb.Ik(b.g,D)==0&&klb.im(b.g,D);}klb.Ro(b.g,0);if(!Q&&b.xb()){try{b.yb(b.g);Q=true;}catch(a){a=k1(a);if(uX(a,15)){ab=a;flb.cz(ab,(flb.V5(),flb.U5),'');'2D-coordinate creation failed:'+ab+' '+flb.h5((dc=d,Klb.v8(),dc));}else throw l1(a);}}if(Q){klb.Uo(b.g);klb.Wt(b.g);}else R||klb.Ro(b.g,0);};klb.mp=function mp(a,b){var c,d,e,f,g,h;if(b.length<=0||b[0]<64)return;klb.ep(a,b,0);h=klb.dp(a,4);e=klb.dp(a,1)==1;f=klb.dp(a,1)==1;for(c=0;c<a.g.f;c++){g=klb.dp(a,h);d=e;e&&f&&(d=klb.dp(a,1)==1);klb.Rl(a.g,c,g,d);}};klb.np=function np(){};b2(246,1,{},klb.np);_.xb=function op(){return false;};_.yb=function pp(a){throw l1(new flb.uz('Unexpected request to invent coordinates. Check source code logic!'));};_.d=0;_.e=0;_.f=0;_.i=false;elb.fY=D3(246);klb.qp=function qp(a){};klb.rp=function rp(a){klb.qp(this);this.b=a;};b2(48,246,{},klb.rp);_.xb=function sp(){return this.b;};_.yb=function tp(a){var b;b=new olb.Ux(this.a);b.g=new llb.feb(78187493520);olb.Bx(b,a);};_.a=2;_.b=false;elb.gY=D3(48);klb.up=function up(a,b,c,d){var e,f,g,h,i,j,k,l,m,n,o,p,q,r;e=b.a;n=b.g;j=I1(n1(klb.pk(a.k,e),1),0);f=klb.kk(a.k,e);l=f!=null?klb.Bp(a,e,f,d):j?'*':klb.jk(a.k,e);!j&&f==null&&klb.oo(a.k,e)&&(a.j&4)==0&&(klb.nn(a.k,e)!=0||klb.bk(a.k,e)==-1&&klb.qk(a.k,e)==0)&&(l=ilb.Xgb(l).toLowerCase());b.f&&(c.a+='(',c);n!=-1&&klb.yp(a,klb.un(a.k,b.a,b.g),b.g,c);g=klb.dk(a.k,e);if(g==0&&(a.j&1)!=0){h=n1(klb.pk(a.k,e),Wmb);s1(h,Zmb)==0?g=-1:s1(h,Ymb)==0&&(g=1);}k=klb.nk(a.k,e);m=(a.j&2)!=0?klb.mk(a.k,e):0;o=(a.j&1)!=0?klb.Hp(a,e,d):null;r=!j&&!klb.Op(klb.uk(a.k,e))||f!=null||(klb.ok(a.k,e)==1||klb.ok(a.k,e)==2)&&(q=klb.ik(a.k,e)-1,!(q!=-1&&a.i[q][klb.hk(a.k,e)]<=1))&&(klb.uk(a.k,e)!=7||klb.dk(a.k,e)>0)||klb.oo(a.k,e)&&klb.nn(a.k,e)==0&&(a.j&4)==0||g!=0||k!=0||m!=0||klb.bk(a.k,e)!=-1||klb.qk(a.k,e)!=0||o!=null;r&&(c.a+='[',c);k!=0&&(c.a+=k,c);c.a+=''+l;(klb.ok(a.k,e)==1||klb.ok(a.k,e)==2)&&(p=klb.ik(a.k,e)-1,!(p!=-1&&a.i[p][klb.hk(a.k,e)]<=1))&&(klb.uk(a.k,e)!=7||klb.dk(a.k,e)>0)&&flb.N5(c,klb.Gp(a,e,n));if((a.j&1)==0&&r){i=klb.Yn(a.k,e);i==1?(c.a+='H',c):i>1&&(c.a+='H'+i,c);}if(g!=0){c.a+=String.fromCharCode(g>0?43:45);$wnd.Math.abs(g)>1&&flb.K5(c,$wnd.Math.abs(g));}o!=null&&(c.a+=''+o,c);if(m!=0){c.a+=':';c.a+=m;}r&&(c.a+=']',c);klb.zp(a,b,c);b.e&&(c.a+=')',c);};klb.vp=function vp(a,b,c){var d,e,f,g,h;f=true;h=llb.Bi(a.g,b).a;g=klb.Kp(a,h);while(g!=-1){d=klb.wn(a.k,h,g);e=klb.yn(a.k,h,g);g=klb.Kp(a,d);klb.xp(a,new klb.Ws(d,e,h,c&&f,c&&g==-1),++b);h=d;f=false;}};klb.wp=function wp(a,b,c){var d,e,f,g;for(e=b,f=0,g=e.length;f<g;++f){d=e[f];a.f[d&1073741823]=(d&nob)!=0^c?2:1;}};klb.xp=function xp(a,b,c){llb.vi(a.g,c,b);a.b[b.a]=true;b.g!=-1&&(a.c[klb.un(a.k,b.a,b.g)]=true);};klb.yp=function yp(a,b,c,d){var e,f,g,h,i;i=ilb.Xgb(d.a).length;a.f[b]!=0&&flb.J5(d,a.f[b]==1?47:92);if(a.j==1){f=klb.Jk(a.k,127);if(f!=0){(f&1)!=0&&a.f[b]==0&&(d.a+='-',d);if((f&2)!=0){ilb.Xgb(d.a).length!=i&&(d.a+=',',d);d.a+='=';}if((f&4)!=0){ilb.Xgb(d.a).length!=i&&(d.a+=',',d);d.a+='#';}if((f&8)!=0){ilb.Xgb(d.a).length!=i&&(d.a+=',',d);d.a+='$';}if((f&16)!=0){ilb.Xgb(d.a).length!=i&&(d.a+=',',d);d.a+='$';}if((f&64)!=0){ilb.Xgb(d.a).length!=i&&(d.a+=',',d);d.a+=':';}if((f&32)!=0){ilb.Xgb(d.a).length!=i&&(d.a+=',',d);flb.N5(d,klb.kl(a.k,c)?'<-':'->');}}}if(i==ilb.Xgb(d.a).length&&(!klb.po(a.k,b)||(a.j&4)!=0)){e=klb.Kk(a.k,b)&127;e==1?klb.oo(a.k,klb.zk(a.k,0,b))&&klb.oo(a.k,klb.zk(a.k,1,b))&&(a.j&4)==0&&a.f[b]==0&&(d.a+='-',d):e==2?(d.a+='=',d):e==4?(d.a+='#',d):e==8?(d.a+='$',d):e==16?(d.a+='$',d):e==64?(d.a+=':',d):e==32&&flb.N5(d,klb.kl(a.k,c)?'<-':'->');}if(a.j==1){g=i==ilb.Xgb(d.a).length?'':';';h=klb.Jk(a.k,b)&384;h==256?(d.a+=g+'@',d):h==128&&(d.a+=g+'!@',d);}};klb.zp=function zp(a,b,c){var d,e,f;if(b.c!=null){for(e=0;e<b.c.length;e++){for(f=0;f<klb.xn(a.k,b.a);f++){if(b.c[e]==klb.wn(a.k,b.a,f)){d=klb.yn(a.k,b.a,f);b.d[e]||klb.yp(a,d,b.a,c);a.e[d]>9&&(c.a+='%',c);flb.K5(c,a.e[d]);}}}}};klb.Ap=function Ap(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N,O;M=new llb.Qi();for(r=new llb.Wbb(a.g);r.a<r.c.a.length;){q=llb.Vbb(r);if(q.g!=-1){t=klb.un(a.k,q.a,q.g);if(!klb.qo(a.k,t)&&!klb.Fo(a.k,t)&&(klb.Ik(a.k,t)==1||klb.Ik(a.k,t)==2)){K=llb.Bi(a.g,a.q[q.g]);c=oW(elb.IX,omb,6,klb.xn(a.k,q.a)+klb.xn(a.k,K.a)-2,15,1);B=0;L=false;if(K.g!=-1){c[B++]=K.b;}else {u=-1;N=-1;v=Ylb;for(D=0;D<klb.xn(a.k,K.a);D++){l=klb.wn(a.k,K.a,D);if(l!=q.a){if(u==-1){u=D;v=a.q[l];}else {if(v<a.q[l]){N=D;}else {N=u;u=D;}}}}if(N==-1){H=klb.wn(a.k,K.a,u);I=klb.yn(a.k,K.a,u);c[B++]=I|(klb.Mp(a,K.a,H)?nob:0);}else {m=klb.wn(a.k,K.a,u);o=klb.yn(a.k,K.a,u);n=klb.wn(a.k,K.a,N);p=klb.yn(a.k,K.a,N);c[B++]=o|(klb.Mp(a,K.a,m)?nob:0);c[B++]=p|(klb.Mp(a,K.a,n)?0:nob);}}if(klb.xn(a.k,K.a)==3&&K.g!=-1){for(D=0;D<klb.xn(a.k,K.a);D++){l=klb.wn(a.k,K.a,D);if(l!=K.g&&l!=q.a){d=klb.yn(a.k,K.a,D);c[B++]=d|(klb.Mp(a,K.a,l)?nob:0);l<K.g&&(L=!L);break;}}}klb.Ik(a.k,t)==2&&(L=!L);for(C=0;C<klb.xn(a.k,q.a);C++){i=klb.wn(a.k,q.a,C);if(i!=q.g){A=L;if(klb.xn(a.k,q.a)==3){for(G=0;G<klb.xn(a.k,q.a);G++){l=klb.wn(a.k,q.a,G);if(l!=q.g&&l!=i){l<i&&(A=!A);break;}}}if(klb.fl(a.k,t)){w=klb.If(a.d,t);if(!a.n[w]){a.n[w]=true;a.o[w]=A;}a.o[w]&&(A=!A);}j=klb.un(a.k,q.a,i);c[B++]=j|(A^klb.Mp(a,q.a,i)?0:nob);}}ilb.ogb(M.a,c);}}}a.f=oW(elb.IX,omb,6,a.k.g,15,1);M.a.length!=0&&klb.wp(a,llb.Ii(M,0),false);while(M.a.length!=0){O=M.a.length;for(C=M.a.length-1;C>=0;C--){c=(ilb.Lgb(C,M.a.length),M.a[C]);J=0;F=false;k=false;for(f=c,g=0,h=f.length;g<h;++g){e=f[g];b=e&1073741823;if(a.f[b]!=0){s=(e&nob)!=0^a.f[b]==2;J==0?F=s:F!=s&&(k=true);++J;}}if(J!=0){c=llb.Ii(M,C);k||klb.wp(a,c,F);}}O==M.a.length&&klb.wp(a,llb.Ii(M,0),false);}};klb.Bp=function Bp(a,b,c,d){var e,f,g,h,i,j,k;k=ilb.Xgb(d.a).length;0<k?d.a=flb.w5(d.a,0,0):0>k&&(d.a+=flb.B5(oW(elb.FX,Inb,6,-k,15,1)));i=klb.oo(a.k,b)&&(a.j&4)==0;for(f=c,g=0,h=f.length;g<h;++g){e=f[g];ilb.Xgb(d.a).length!=0&&(d.a+=',',d);j=(klb.wj(),klb.qj)[e];flb.N5(d,i?ilb.Xgb(j).toLowerCase():j);}return d.a;};klb.Cp=function Cp(a){var b,c,d,e,f,g,h,i;if(!a.k||a.k.q==0)return '';klb.Jt(a.k,15);a.d=new klb.bg(a.k,129);e=a.d.R;a.o=oW(elb.g1,Imb,6,e+1,16,1);a.n=oW(elb.g1,Imb,6,e+1,16,1);a.i=mW(elb.IX,[Lnb,omb],[7,6],15,[2,32],2);for(b=0;b<a.k.f;b++){i=klb.ik(a.k,b)-1;i!=-1&&++a.i[i][klb.hk(a.k,b)];}klb.Fp(a);klb.Dp(a);klb.Ap(a);d=new flb.Q5();c=new flb.Q5();f=true;for(h=new llb.Wbb(a.g);h.a<h.c.a.length;){g=llb.Vbb(h);g.g==-1&&(f?f=false:(d.a+='.',d));klb.up(a,g,d,c);}return d.a;};klb.Dp=function Dp(a){var b,c,d,e,f,g,h,i,j,k,l;for(l=new llb.Wbb(a.g);l.a<l.c.a.length;){j=llb.Vbb(l);c=0;for(f=0;f<klb.xn(a.k,j.a);f++)a.c[klb.yn(a.k,j.a,f)]||++c;if(c!=0){j.c=oW(elb.IX,omb,6,c,15,1);c=0;for(g=0;g<klb.xn(a.k,j.a);g++){if(!a.c[klb.yn(a.k,j.a,g)]){i=klb.wn(a.k,j.a,g);j.c[c++]=a.q[i]<<16|i;}}ilb.rgb(j.c,ilb.Xgb(e2(llb.Ccb.prototype.kc,llb.Ccb,[])));for(e=0;e<j.c.length;e++)j.c[e]=fmb&j.c[e];}}d=oW(elb.g1,Imb,6,a.k.g,16,1);a.e=oW(elb.IX,omb,6,a.k.g,15,1);for(k=new llb.Wbb(a.g);k.a<k.c.a.length;){j=llb.Vbb(k);if(j.c!=null){j.d=oW(elb.g1,Imb,6,j.c.length,16,1);for(e=0;e<j.c.length;e++){for(h=0;h<klb.xn(a.k,j.a);h++){if(j.c[e]==klb.wn(a.k,j.a,h)){b=klb.yn(a.k,j.a,h);if(a.c[b]){d[a.e[b]]=false;}else {a.c[b]=true;j.d[e]=true;a.e[b]=1;while(d[a.e[b]])++a.e[b];d[a.e[b]]=true;}}}}}}};klb.Ep=function Ep(a){var b,c,d,e;d=-1;e=Ylb;for(b=0;b<a.k.f;b++){if(!a.b[b]){c=a.a[b];I1(n1(klb.pk(a.k,b),Jmb),0)&&(c+=nob);klb.xn(a.k,b)==0?c+=1056964608:c+=klb.xn(a.k,b)<<24;if(e>c){e=c;d=b;}}}return d;};klb.Fp=function Fp(a){var b,c,d,e,f;a.a=a.d.d;a.b=oW(elb.g1,Imb,6,a.k.f,16,1);a.c=oW(elb.g1,Imb,6,a.k.g,16,1);a.g=new llb.Qi();b=klb.Ep(a);while(b!=-1){c=a.g.a.length;klb.xp(a,new klb.Ws(b,-1,-1,false,false),c);if(klb.xn(a.k,b)!=0){klb.vp(a,c,false);while(c<a.g.a.length-1){while(klb.Lp(a,llb.Bi(a.g,c).a))klb.vp(a,c,true);++c;}}b=klb.Ep(a);}a.q=oW(elb.IX,omb,6,a.k.f,15,1);d=0;for(f=new llb.Wbb(a.g);f.a<f.c.a.length;){e=llb.Vbb(f);a.q[e.a]=d++;}};klb.Gp=function Gp(a,b,c){var d,e,f,g,h,i,j,k,l,m,n;h=false;if(klb.nn(a.k,b)!=0&&klb.xn(a.k,b)==2&&klb.zn(a.k,b,0)==2&&klb.zn(a.k,b,1)==2){for(f=0;f<klb.xn(a.k,b);f++){d=klb.wn(a.k,b,f);n=0;m=oW(elb.IX,omb,6,3,15,1);for(j=0;j<klb.xn(a.k,d);j++){m[n]=klb.wn(a.k,d,j);m[n]!=b&&++n;}n==2&&a.q[m[0]]<a.q[m[1]]^m[0]<m[1]&&(h=!h);}}else {k=oW(elb.IX,omb,6,4,15,1);l=oW(elb.IX,omb,6,4,15,1);g=0;if(c!=-1){k[g]=c;l[g++]=8*a.q[c];}if(klb.Mn(a.k,b)!=0){k[g]=Ylb;l[g++]=8*a.q[b];}else if(klb.xn(a.k,b)==3){k[g]=Ylb;l[g++]=8*a.q[b];}for(f=0;f<klb.xn(a.k,b);f++){d=klb.wn(a.k,b,f);if(d!=c){k[g]=d;l[g++]=klb.Jp(a,b,f);}}h=klb.Np(k,l);}i=klb.ok(a.k,b)==1^h;if(klb.$k(a.k,b)){e=klb.Jf(a.d,b);if(!a.n[e]){a.n[e]=true;a.o[e]=i;}a.o[e]&&(i=!i);}return i?'@@':'@';};klb.Hp=function Hp(a,b,c){var d,e,f,g,h,i,j,k;h=ilb.Xgb(c.a).length;0<h?c.a=flb.w5(c.a,0,0):0>h&&(c.a+=flb.B5(oW(elb.FX,Inb,6,-h,15,1)));i=klb.pk(a.k,b);e=R1(M1(n1(i,Wmb),3));switch(e){case 20971520:c.a+='+0';break;case 25165824:klb.dk(a.k,b)==0&&(c.a+='-',c);break;case 12582912:klb.dk(a.k,b)==0&&(c.a+='+',c);}d=n1(i,Pnb);s1(d,2)==0?(c.a+=';a',c):s1(d,4)==0&&(c.a+=';A',c);f=n1(i,Umb);s1(f,0)!=0&&(s1(f,1792)==0?(c.a+=';H0',c):s1(f,1664)==0?(c.a+=';H1',c):s1(f,1408)==0?(c.a+=';H2',c):s1(f,896)==0?(c.a+=';H3',c):s1(f,128)==0?(c.a+=';!H0',c):s1(f,384)==0?(c.a+=';!H0;!H1',c):s1(f,1536)==0?(c.a+=';!H2;!H3',c):s1(f,Vmb)==0&&(c.a+=';!H3',c));k=n1(i,120);s1(k,8)==0?(c.a+=';!R0',c):s1(k,16)==0?(c.a+=';!R1',c):s1(k,32)==0?(c.a+=';!R2',c):s1(k,64)==0?(c.a+=';!R3',c):s1(k,112)==0?(c.a+=';R0',c):s1(k,104)==0?(c.a+=';R1',c):s1(k,88)==0?(c.a+=';R2',c):s1(k,56)==0&&(c.a+=';R3',c);j=n1(i,smb);if(s1(j,tmb)==0)c.a+=';!r'+S1(j);else if(s1(j,oob)==0)c.a+=';r'+S1(j);else if(s1(j,0)!=0){if(I1(n1(j,zmb),0)){x1(n1(j,tmb),0)&&(c.a+=';!r0'+S1(j),c);x1(n1(j,umb),0)&&(c.a+=';!r3'+S1(j),c);x1(n1(j,vmb),0)&&(c.a+=';!r4'+S1(j),c);x1(n1(j,wmb),0)&&(c.a+=';!r5'+S1(j),c);x1(n1(j,xmb),0)&&(c.a+=';!r6'+S1(j),c);x1(n1(j,ymb),0)&&(c.a+=';!r7'+S1(j),c);}else {c.a+=';';I1(n1(j,tmb),0)&&(c.a+='r0,'+S1(j),c);I1(n1(j,umb),0)&&(c.a+='r3,'+S1(j),c);I1(n1(j,vmb),0)&&(c.a+='r4,'+S1(j),c);I1(n1(j,wmb),0)&&(c.a+='r5,'+S1(j),c);I1(n1(j,xmb),0)&&(c.a+='r6,'+S1(j),c);I1(n1(j,ymb),0)&&(c.a+='r7,'+S1(j),c);flb.K2(c,ilb.Xgb(c.a).length-1);}}if(s1(j,0)==0){j=M1(n1(i,qnb),22);s1(j,0)!=0&&(c.a+=';r'+S1(j),c);}g=n1(i,_mb);s1(g,3801088)==0&&(c.a+=';D1',c);s1(g,anb)==0&&(c.a+=';D2',c);s1(g,bnb)==0&&(c.a+=';D3',c);s1(g,3145728)==0&&(c.a+=';!D3;!D4',c);s1(g,cnb)==0&&(c.a+=';!D4',c);s1(g,dnb)==0&&(c.a+=';!D0;!D1',c);s1(g,enb)==0&&(c.a+=';!D0;!D1;!D2',c);s1(g,fnb)==0&&(c.a+=';!D0;!D1;!D2;!D3',c);I1(n1(i,tnb),0)&&flb.N5(c,';D'+klb.xn(a.k,b));I1(n1(i,Qmb),0)&&flb.N5(c,';!D'+klb.xn(a.k,b));return ilb.Xgb(c.a).length==0?null:c.a;};klb.Ip=function Ip(a){a.p==null&&(a.p=klb.Cp(a));return a.p;};klb.Jp=function Jp(a,b,c){var d,e,f,g,h;d=klb.yn(a.k,b,c);g=klb.wn(a.k,b,c);if(a.e[d]!=0){h=8*a.q[b]+1;e=llb.Bi(a.g,a.q[b]).c;for(f=0;f<e.length&&g!=e[f];f++)++h;return h;}return 8*a.q[g];};klb.Kp=function Kp(a,b){var c,d,e,f,g;c=-1;d=-1;for(f=0;f<klb.xn(a.k,b);f++){e=klb.wn(a.k,b,f);g=(klb.zn(a.k,b,f)<<24)+a.a[e];if(!a.b[e]&&(c==-1||d<g)){c=f;d=g;}}return c;};klb.Lp=function Lp(a,b){var c;for(c=0;c<klb.xn(a.k,b);c++)if(!a.b[klb.wn(a.k,b,c)])return true;return false;};klb.Mp=function Mp(a,b,c){var d,e;d=llb.Bi(a.g,a.q[b]);if(d.g==c)return false;e=llb.Bi(a.g,a.q[c]);if(e.g==b)return true;return klb.Vs(e,b);};klb.Np=function Np(a,b){var c,d,e;d=false;for(c=1;c<4;c++){for(e=0;e<c;e++){a[e]>a[c]&&(d=!d);b[e]>b[c]&&(d=!d);}}return d;};klb.Op=function Op(a){return a>=5&&a<=9||a>=15&&a<=17||a==35||a==53;};klb.Pp=function Pp(a){klb.Rp.call(this,a,false);};klb.Qp=function Qp(a,b){this.k=a;this.j=b;};klb.Rp=function Rp(a,b){klb.Qp.call(this,a,b?2:0);};klb.Sp=function Sp(a){var b,c,d,e;e=new flb.Q5();for(c=0;c<a.g.a.length;c++){c!=0&&(e.a+='.',e);flb.N5(e,klb.Ip(new klb.Qp(llb.Bi(a.g,c),2)));}e.a+='>';for(d=0;d<a.a.a.length;d++){d!=0&&(e.a+='.',e);flb.N5(e,klb.Ip(new klb.Pp(llb.Bi(a.a,d))));}e.a+='>';for(b=0;b<a.f.a.length;b++){b!=0&&(e.a+='.',e);flb.N5(e,klb.Ip(new klb.Qp(llb.Bi(a.f,b),2)));}return e.a;};b2(90,1,{},klb.Pp,klb.Qp,klb.Rp);_.j=0;elb.hY=D3(90);klb.Tp=function Tp(a,b){this.b=a;this.a=b;};b2(3,1,{3:1},klb.Tp);_.a=0;_.b=0;elb.iY=D3(3);klb.Vp=function Vp(){klb.Vp=d2;klb.Up=wW(iW(elb.iY,2),Tlb,14,0,[null,wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(0,1.007825032),new klb.Tp(1,2.014101778),new klb.Tp(2,3.016049268),new klb.Tp(3,4.027834627),new klb.Tp(4,5.039542911),new klb.Tp(5,6.044942608)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(1,3.01602931),new klb.Tp(2,4.00260325),new klb.Tp(3,5.012223628),new klb.Tp(4,6.018888072),new klb.Tp(5,7.028030527),new klb.Tp(6,8.033921838),new klb.Tp(7,9.043820323),new klb.Tp(8,10.052399713)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(1,4.027182329),new klb.Tp(2,5.012537796),new klb.Tp(3,6.015122281),new klb.Tp(4,7.016004049),new klb.Tp(5,8.02248667),new klb.Tp(6,9.026789122),new klb.Tp(7,10.035480884),new klb.Tp(8,11.043796166),new klb.Tp(9,12.05378)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(1,5.04079),new klb.Tp(2,6.019725804),new klb.Tp(3,7.016929246),new klb.Tp(4,8.005305094),new klb.Tp(5,9.012182135),new klb.Tp(6,10.01353372),new klb.Tp(7,11.021657653),new klb.Tp(8,12.026920631),new klb.Tp(9,13.036133834),new klb.Tp(10,14.042815522)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(2,7.029917389),new klb.Tp(3,8.024606713),new klb.Tp(4,9.013328806),new klb.Tp(5,10.012937027),new klb.Tp(6,11.009305466),new klb.Tp(7,12.014352109),new klb.Tp(8,13.017780267),new klb.Tp(9,14.025404064),new klb.Tp(10,15.031097291),new klb.Tp(11,16.039808836),new klb.Tp(12,17.046931399),new klb.Tp(13,18.05617),new klb.Tp(14,19.06373)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(2,8.037675026),new klb.Tp(3,9.031040087),new klb.Tp(4,10.01685311),new klb.Tp(5,11.011433818),new klb.Tp(6,12),new klb.Tp(7,13.003354838),new klb.Tp(8,14.003241988),new klb.Tp(9,15.010599258),new klb.Tp(10,16.014701243),new klb.Tp(11,17.022583712),new klb.Tp(12,18.026757058),new klb.Tp(13,19.035248094),new klb.Tp(14,20.040322395),new klb.Tp(15,21.04934),new klb.Tp(16,22.05645)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(3,10.042618),new klb.Tp(4,11.026796226),new klb.Tp(5,12.018613202),new klb.Tp(6,13.005738584),new klb.Tp(7,14.003074005),new klb.Tp(8,15.000108898),new klb.Tp(9,16.006101417),new klb.Tp(10,17.008449673),new klb.Tp(11,18.014081827),new klb.Tp(12,19.017026896),new klb.Tp(13,20.023367295),new klb.Tp(14,21.027087574),new klb.Tp(15,22.034440259),new klb.Tp(16,23.04051),new klb.Tp(17,24.0505)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(4,12.034404776),new klb.Tp(5,13.0248104),new klb.Tp(6,14.008595285),new klb.Tp(7,15.003065386),new klb.Tp(8,15.994914622),new klb.Tp(9,16.999131501),new klb.Tp(10,17.999160419),new klb.Tp(11,19.00357873),new klb.Tp(12,20.00407615),new klb.Tp(13,21.008654631),new klb.Tp(14,22.009967157),new klb.Tp(15,23.015691325),new klb.Tp(16,24.020369922),new klb.Tp(17,25.02914),new klb.Tp(18,26.03775)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(5,14.03608),new klb.Tp(6,15.018010856),new klb.Tp(7,16.01146573),new klb.Tp(8,17.002095238),new klb.Tp(9,18.000937667),new klb.Tp(10,18.998403205),new klb.Tp(11,19.999981324),new klb.Tp(12,20.999948921),new klb.Tp(13,22.00299925),new klb.Tp(14,23.003574385),new klb.Tp(15,24.008099371),new klb.Tp(16,25.012094963),new klb.Tp(17,26.019633157),new klb.Tp(18,27.026892316),new klb.Tp(19,28.03567),new klb.Tp(20,29.04326)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(6,16.025756907),new klb.Tp(7,17.017697565),new klb.Tp(8,18.005697066),new klb.Tp(9,19.001879839),new klb.Tp(10,19.992440176),new klb.Tp(11,20.993846744),new klb.Tp(12,21.99138551),new klb.Tp(13,22.994467337),new klb.Tp(14,23.993615074),new klb.Tp(15,24.997789899),new klb.Tp(16,26.000461498),new klb.Tp(17,27.0076152),new klb.Tp(18,28.012108072),new klb.Tp(19,29.019345902),new klb.Tp(20,30.023872),new klb.Tp(21,31.03311),new klb.Tp(22,32.03991)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(7,18.02718),new klb.Tp(8,19.01387945),new klb.Tp(9,20.00734826),new klb.Tp(10,20.997655099),new klb.Tp(11,21.994436782),new klb.Tp(12,22.989769675),new klb.Tp(13,23.990963332),new klb.Tp(14,24.989954352),new klb.Tp(15,25.992589898),new klb.Tp(16,26.994008702),new klb.Tp(17,27.99889041),new klb.Tp(18,29.002811301),new klb.Tp(19,30.009226487),new klb.Tp(20,31.013595108),new klb.Tp(21,32.019649792),new klb.Tp(22,33.027386),new klb.Tp(23,34.0349),new klb.Tp(24,35.04418)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(8,20.018862744),new klb.Tp(9,21.011714174),new klb.Tp(10,21.999574055),new klb.Tp(11,22.99412485),new klb.Tp(12,23.985041898),new klb.Tp(13,24.985837023),new klb.Tp(14,25.98259304),new klb.Tp(15,26.984340742),new klb.Tp(16,27.983876703),new klb.Tp(17,28.988554743),new klb.Tp(18,29.990464529),new klb.Tp(19,30.996548459),new klb.Tp(20,31.999145889),new klb.Tp(21,33.005586975),new klb.Tp(22,34.00907244),new klb.Tp(23,35.018669),new klb.Tp(24,36.02245),new klb.Tp(25,37.03124)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(8,21.02804),new klb.Tp(9,22.01952),new klb.Tp(10,23.0072649),new klb.Tp(11,23.999940911),new klb.Tp(12,24.990428555),new klb.Tp(13,25.986891659),new klb.Tp(14,26.981538441),new klb.Tp(15,27.981910184),new klb.Tp(16,28.980444848),new klb.Tp(17,29.982960304),new klb.Tp(18,30.983946023),new klb.Tp(19,31.988124379),new klb.Tp(20,32.990869587),new klb.Tp(21,33.996927255),new klb.Tp(22,34.99993765),new klb.Tp(23,36.006351501),new klb.Tp(24,37.01031),new klb.Tp(25,38.0169),new klb.Tp(26,39.0219)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(8,22.03453),new klb.Tp(9,23.02552),new klb.Tp(10,24.011545711),new klb.Tp(11,25.00410664),new klb.Tp(12,25.992329935),new klb.Tp(13,26.986704764),new klb.Tp(14,27.976926533),new klb.Tp(15,28.976494719),new klb.Tp(16,29.973770218),new klb.Tp(17,30.975363275),new klb.Tp(18,31.974148129),new klb.Tp(19,32.97800052),new klb.Tp(20,33.978575745),new klb.Tp(21,34.984584158),new klb.Tp(22,35.986687363),new klb.Tp(23,36.99299599),new klb.Tp(24,37.99598),new klb.Tp(25,39.0023),new klb.Tp(26,40.0058),new klb.Tp(27,41.0127),new klb.Tp(28,42.0161)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(9,24.03435),new klb.Tp(10,25.02026),new klb.Tp(11,26.01178),new klb.Tp(12,26.999191645),new klb.Tp(13,27.99231233),new klb.Tp(14,28.981801376),new klb.Tp(15,29.978313807),new klb.Tp(16,30.973761512),new klb.Tp(17,31.973907163),new klb.Tp(18,32.971725281),new klb.Tp(19,33.973636381),new klb.Tp(20,34.973314249),new klb.Tp(21,35.978259824),new klb.Tp(22,36.979608338),new klb.Tp(23,37.98447),new klb.Tp(24,38.98642),new klb.Tp(25,39.99105),new klb.Tp(26,40.9948),new klb.Tp(27,42.00009),new klb.Tp(28,43.00331),new klb.Tp(29,44.00988),new klb.Tp(30,45.01514),new klb.Tp(31,46.02383)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(10,26.02788),new klb.Tp(11,27.018795),new klb.Tp(12,28.004372661),new klb.Tp(13,28.996608805),new klb.Tp(14,29.984902954),new klb.Tp(15,30.979554421),new klb.Tp(16,31.97207069),new klb.Tp(17,32.971458497),new klb.Tp(18,33.967866831),new klb.Tp(19,34.96903214),new klb.Tp(20,35.96708088),new klb.Tp(21,36.971125716),new klb.Tp(22,37.971163443),new klb.Tp(23,38.975135275),new klb.Tp(24,39.97547),new klb.Tp(25,40.98003),new klb.Tp(26,41.98149),new klb.Tp(27,42.9866),new klb.Tp(28,43.98832),new klb.Tp(29,44.99482),new klb.Tp(30,45.99957),new klb.Tp(31,47.00762),new klb.Tp(32,48.01299),new klb.Tp(33,49.02201)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(11,28.02851),new klb.Tp(12,29.01411),new klb.Tp(13,30.00477),new klb.Tp(14,30.992416014),new klb.Tp(15,31.985688908),new klb.Tp(16,32.977451798),new klb.Tp(17,33.973761967),new klb.Tp(18,34.968852707),new klb.Tp(19,35.968306945),new klb.Tp(20,36.9659026),new klb.Tp(21,37.96801055),new klb.Tp(22,38.968007677),new klb.Tp(23,39.970415555),new klb.Tp(24,40.970650212),new klb.Tp(25,41.973174994),new klb.Tp(26,42.974203385),new klb.Tp(27,43.978538712),new klb.Tp(28,44.9797),new klb.Tp(29,45.98412),new klb.Tp(30,46.98795),new klb.Tp(31,47.99485),new klb.Tp(32,48.99989),new klb.Tp(33,50.00773),new klb.Tp(34,51.01353)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(12,30.02156),new klb.Tp(13,31.012126),new klb.Tp(14,31.99766066),new klb.Tp(15,32.989928719),new klb.Tp(16,33.980270118),new klb.Tp(17,34.975256726),new klb.Tp(18,35.967546282),new klb.Tp(19,36.966775912),new klb.Tp(20,37.962732161),new klb.Tp(21,38.964313413),new klb.Tp(22,39.962383123),new klb.Tp(23,40.964500828),new klb.Tp(24,41.963046386),new klb.Tp(25,42.965670701),new klb.Tp(26,43.965365269),new klb.Tp(27,44.968094979),new klb.Tp(28,45.968093467),new klb.Tp(29,46.972186238),new klb.Tp(30,47.97507),new klb.Tp(31,48.98218),new klb.Tp(32,49.98594),new klb.Tp(33,50.99324),new klb.Tp(34,51.99817),new klb.Tp(35,53.006227)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(13,32.02192),new klb.Tp(14,33.00726),new klb.Tp(15,33.99841),new klb.Tp(16,34.988011615),new klb.Tp(17,35.981293405),new klb.Tp(18,36.973376915),new klb.Tp(19,37.969080107),new klb.Tp(20,38.963706861),new klb.Tp(21,39.963998672),new klb.Tp(22,40.961825972),new klb.Tp(23,41.962403059),new klb.Tp(24,42.960715746),new klb.Tp(25,43.961556146),new klb.Tp(26,44.960699658),new klb.Tp(27,45.961976203),new klb.Tp(28,46.961677807),new klb.Tp(29,47.965512946),new klb.Tp(30,48.967450084),new klb.Tp(31,49.972782832),new klb.Tp(32,50.97638),new klb.Tp(33,51.98261),new klb.Tp(34,52.98712),new klb.Tp(35,53.99399),new klb.Tp(36,54.999388)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(14,34.01412),new klb.Tp(15,35.004765),new klb.Tp(16,35.993087234),new klb.Tp(17,36.985871505),new klb.Tp(18,37.976318637),new klb.Tp(19,38.970717729),new klb.Tp(20,39.962591155),new klb.Tp(21,40.962278349),new klb.Tp(22,41.958618337),new klb.Tp(23,42.958766833),new klb.Tp(24,43.955481094),new klb.Tp(25,44.956185938),new klb.Tp(26,45.953692759),new klb.Tp(27,46.954546459),new klb.Tp(28,47.952533512),new klb.Tp(29,48.955673302),new klb.Tp(30,49.957518286),new klb.Tp(31,50.961474238),new klb.Tp(32,51.9651),new klb.Tp(33,52.97005),new klb.Tp(34,53.97468),new klb.Tp(35,54.98055),new klb.Tp(36,55.98579),new klb.Tp(37,56.992356)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(15,36.01492),new klb.Tp(16,37.00305),new klb.Tp(17,37.9947),new klb.Tp(18,38.984790009),new klb.Tp(19,39.977964014),new klb.Tp(20,40.969251316),new klb.Tp(21,41.965516761),new klb.Tp(22,42.96115098),new klb.Tp(23,43.959403048),new klb.Tp(24,44.955910243),new klb.Tp(25,45.95517025),new klb.Tp(26,46.952408027),new klb.Tp(27,47.952234991),new klb.Tp(28,48.950024065),new klb.Tp(29,49.952187008),new klb.Tp(30,50.9536027),new klb.Tp(31,51.95665),new klb.Tp(32,52.95817),new klb.Tp(33,53.963),new klb.Tp(34,54.9694),new klb.Tp(35,55.97266),new klb.Tp(36,56.97704),new klb.Tp(37,57.98307),new klb.Tp(38,58.988041)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(16,38.00977),new klb.Tp(17,39.001323),new klb.Tp(18,39.990498907),new klb.Tp(19,40.983131),new klb.Tp(20,41.973031622),new klb.Tp(21,42.968523342),new klb.Tp(22,43.959690235),new klb.Tp(23,44.958124349),new klb.Tp(24,45.952629491),new klb.Tp(25,46.951763792),new klb.Tp(26,47.947947053),new klb.Tp(27,48.947870789),new klb.Tp(28,49.944792069),new klb.Tp(29,50.946616017),new klb.Tp(30,51.946898175),new klb.Tp(31,52.949731709),new klb.Tp(32,53.95087),new klb.Tp(33,54.95512),new klb.Tp(34,55.95799),new klb.Tp(35,56.9643),new klb.Tp(36,57.96611),new klb.Tp(37,58.97196),new klb.Tp(38,59.97564),new klb.Tp(39,60.982018)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(17,40.01109),new klb.Tp(18,40.99974),new klb.Tp(19,41.99123),new klb.Tp(20,42.98065),new klb.Tp(21,43.9744),new klb.Tp(22,44.965782286),new klb.Tp(23,45.960199491),new klb.Tp(24,46.954906918),new klb.Tp(25,47.95225448),new klb.Tp(26,48.948516914),new klb.Tp(27,49.947162792),new klb.Tp(28,50.943963675),new klb.Tp(29,51.944779658),new klb.Tp(30,52.944342517),new klb.Tp(31,53.946444381),new klb.Tp(32,54.947238194),new klb.Tp(33,55.95036),new klb.Tp(34,56.95236),new klb.Tp(35,57.95665),new klb.Tp(36,58.9593),new klb.Tp(37,59.9645),new klb.Tp(38,60.96741),new klb.Tp(39,61.97314),new klb.Tp(40,62.97675)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(18,42.00643),new klb.Tp(19,42.997707),new klb.Tp(20,43.98547),new klb.Tp(21,44.97916),new klb.Tp(22,45.968361649),new klb.Tp(23,46.962906512),new klb.Tp(24,47.954035861),new klb.Tp(25,48.951341135),new klb.Tp(26,49.946049607),new klb.Tp(27,50.944771767),new klb.Tp(28,51.940511904),new klb.Tp(29,52.940653781),new klb.Tp(30,53.938884921),new klb.Tp(31,54.940844164),new klb.Tp(32,55.940645238),new klb.Tp(33,56.9437538),new klb.Tp(34,57.94425),new klb.Tp(35,58.94863),new klb.Tp(36,59.94973),new klb.Tp(37,60.95409),new klb.Tp(38,61.9558),new klb.Tp(39,62.96186),new klb.Tp(40,63.9642),new klb.Tp(41,64.97037)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(19,44.00687),new klb.Tp(20,44.99451),new klb.Tp(21,45.98672),new klb.Tp(22,46.9761),new klb.Tp(23,47.96887),new klb.Tp(24,48.959623415),new klb.Tp(25,49.95424396),new klb.Tp(26,50.948215487),new klb.Tp(27,51.945570079),new klb.Tp(28,52.941294702),new klb.Tp(29,53.940363247),new klb.Tp(30,54.938049636),new klb.Tp(31,55.938909366),new klb.Tp(32,56.938287458),new klb.Tp(33,57.939986451),new klb.Tp(34,58.940447166),new klb.Tp(35,59.943193998),new klb.Tp(36,60.94446),new klb.Tp(37,61.94797),new klb.Tp(38,62.94981),new klb.Tp(39,63.95373),new klb.Tp(40,64.9561),new klb.Tp(41,65.96082),new klb.Tp(42,66.96382)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(19,45.01456),new klb.Tp(20,46.00081),new klb.Tp(21,46.99289),new klb.Tp(22,47.98056),new klb.Tp(23,48.97361),new klb.Tp(24,49.962993316),new klb.Tp(25,50.956824936),new klb.Tp(26,51.948116526),new klb.Tp(27,52.945312282),new klb.Tp(28,53.939614836),new klb.Tp(29,54.938298029),new klb.Tp(30,55.934942133),new klb.Tp(31,56.935398707),new klb.Tp(32,57.933280458),new klb.Tp(33,58.934880493),new klb.Tp(34,59.934076943),new klb.Tp(35,60.936749461),new klb.Tp(36,61.936770495),new klb.Tp(37,62.940118442),new klb.Tp(38,63.94087),new klb.Tp(39,64.94494),new klb.Tp(40,65.94598),new klb.Tp(41,66.95),new klb.Tp(42,67.95251),new klb.Tp(43,68.9577)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(21,48.00176),new klb.Tp(22,48.98972),new klb.Tp(23,49.98154),new klb.Tp(24,50.97072),new klb.Tp(25,51.96359),new klb.Tp(26,52.954224985),new klb.Tp(27,53.948464147),new klb.Tp(28,54.942003149),new klb.Tp(29,55.939843937),new klb.Tp(30,56.936296235),new klb.Tp(31,57.935757571),new klb.Tp(32,58.933200194),new klb.Tp(33,59.933822196),new klb.Tp(34,60.932479381),new klb.Tp(35,61.934054212),new klb.Tp(36,62.933615218),new klb.Tp(37,63.935813523),new klb.Tp(38,64.936484581),new klb.Tp(39,65.939825412),new klb.Tp(40,66.94061),new klb.Tp(41,67.94436),new klb.Tp(42,68.9452),new klb.Tp(43,69.94981),new klb.Tp(44,70.95173),new klb.Tp(45,71.95641)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(22,49.99593),new klb.Tp(23,50.98772),new klb.Tp(24,51.97568),new klb.Tp(25,52.96846),new klb.Tp(26,53.957910508),new klb.Tp(27,54.951336329),new klb.Tp(28,55.942136339),new klb.Tp(29,56.939800489),new klb.Tp(30,57.935347922),new klb.Tp(31,58.934351553),new klb.Tp(32,59.930790633),new klb.Tp(33,60.931060442),new klb.Tp(34,61.928348763),new klb.Tp(35,62.929672948),new klb.Tp(36,63.927969574),new klb.Tp(37,64.930088013),new klb.Tp(38,65.929115232),new klb.Tp(39,66.931569638),new klb.Tp(40,67.931844932),new klb.Tp(41,68.935181837),new klb.Tp(42,69.93614),new klb.Tp(43,70.94),new klb.Tp(44,71.9413),new klb.Tp(45,72.94608),new klb.Tp(46,73.94791),new klb.Tp(47,74.95297),new klb.Tp(48,75.95533),new klb.Tp(49,76.96083),new klb.Tp(50,77.9638)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(23,51.99718),new klb.Tp(24,52.98555),new klb.Tp(25,53.97671),new klb.Tp(26,54.96605),new klb.Tp(27,55.95856),new klb.Tp(28,56.949215695),new klb.Tp(29,57.944540734),new klb.Tp(30,58.939504114),new klb.Tp(31,59.937368123),new klb.Tp(32,60.933462181),new klb.Tp(33,61.932587299),new klb.Tp(34,62.929601079),new klb.Tp(35,63.929767865),new klb.Tp(36,64.927793707),new klb.Tp(37,65.928873041),new klb.Tp(38,66.927750294),new klb.Tp(39,67.929637875),new klb.Tp(40,68.929425281),new klb.Tp(41,69.932409287),new klb.Tp(42,70.932619818),new klb.Tp(43,71.93552),new klb.Tp(44,72.93649),new klb.Tp(45,73.9402),new klb.Tp(46,74.9417),new klb.Tp(47,75.94599),new klb.Tp(48,76.94795),new klb.Tp(49,77.95281),new klb.Tp(50,78.95528),new klb.Tp(51,79.96189)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(24,53.99295),new klb.Tp(25,54.98398),new klb.Tp(26,55.97238),new klb.Tp(27,56.96491),new klb.Tp(28,57.954596465),new klb.Tp(29,58.949267074),new klb.Tp(30,59.941832031),new klb.Tp(31,60.939513907),new klb.Tp(32,61.934334132),new klb.Tp(33,62.933215563),new klb.Tp(34,63.929146578),new klb.Tp(35,64.929245079),new klb.Tp(36,65.926036763),new klb.Tp(37,66.927130859),new klb.Tp(38,67.924847566),new klb.Tp(39,68.926553538),new klb.Tp(40,69.92532487),new klb.Tp(41,70.927727195),new klb.Tp(42,71.926861122),new klb.Tp(43,72.929779469),new klb.Tp(44,73.929458261),new klb.Tp(45,74.932937379),new klb.Tp(46,75.933394207),new klb.Tp(47,76.937085857),new klb.Tp(48,77.938569576),new klb.Tp(49,78.942095175),new klb.Tp(50,79.944414722),new klb.Tp(51,80.95048),new klb.Tp(52,81.95484)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(25,55.99491),new klb.Tp(26,56.98293),new klb.Tp(27,57.97425),new klb.Tp(28,58.96337),new klb.Tp(29,59.95706),new klb.Tp(30,60.94917),new klb.Tp(31,61.944179608),new klb.Tp(32,62.939141527),new klb.Tp(33,63.936838307),new klb.Tp(34,64.932739322),new klb.Tp(35,65.931592355),new klb.Tp(36,66.928204915),new klb.Tp(37,67.927983497),new klb.Tp(38,68.925580912),new klb.Tp(39,69.926027741),new klb.Tp(40,70.92470501),new klb.Tp(41,71.92636935),new klb.Tp(42,72.925169832),new klb.Tp(43,73.926940999),new klb.Tp(44,74.926500645),new klb.Tp(45,75.928928262),new klb.Tp(46,76.929281189),new klb.Tp(47,77.93165595),new klb.Tp(48,78.932916371),new klb.Tp(49,79.936588154),new klb.Tp(50,80.937752955),new klb.Tp(51,81.94316),new klb.Tp(52,82.94687),new klb.Tp(53,83.95234)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(26,57.99101),new klb.Tp(27,58.98175),new klb.Tp(28,59.97019),new klb.Tp(29,60.96379),new klb.Tp(30,61.95465),new klb.Tp(31,62.94964),new klb.Tp(32,63.941572638),new klb.Tp(33,64.939440762),new klb.Tp(34,65.933846798),new klb.Tp(35,66.932738415),new klb.Tp(36,67.928097266),new klb.Tp(37,68.927972002),new klb.Tp(38,69.924250365),new klb.Tp(39,70.924953991),new klb.Tp(40,71.922076184),new klb.Tp(41,72.923459361),new klb.Tp(42,73.921178213),new klb.Tp(43,74.922859494),new klb.Tp(44,75.921402716),new klb.Tp(45,76.923548462),new klb.Tp(46,77.922852886),new klb.Tp(47,78.92540156),new klb.Tp(48,79.925444764),new klb.Tp(49,80.928821065),new klb.Tp(50,81.929550326),new klb.Tp(51,82.93451),new klb.Tp(52,83.93731),new klb.Tp(53,84.94269),new klb.Tp(54,85.94627)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(27,59.99313),new klb.Tp(28,60.98062),new klb.Tp(29,61.9732),new klb.Tp(30,62.96369),new klb.Tp(31,63.957572),new klb.Tp(32,64.949484),new klb.Tp(33,65.944099147),new klb.Tp(34,66.939190417),new klb.Tp(35,67.936792976),new klb.Tp(36,68.932280154),new klb.Tp(37,69.930927811),new klb.Tp(38,70.927114724),new klb.Tp(39,71.926752647),new klb.Tp(40,72.923825288),new klb.Tp(41,73.923929076),new klb.Tp(42,74.921596417),new klb.Tp(43,75.922393933),new klb.Tp(44,76.920647703),new klb.Tp(45,77.921828577),new klb.Tp(46,78.920948498),new klb.Tp(47,79.922578162),new klb.Tp(48,80.922132884),new klb.Tp(49,81.924504668),new klb.Tp(50,82.924980625),new klb.Tp(51,83.92906),new klb.Tp(52,84.93181),new klb.Tp(53,85.93623),new klb.Tp(54,86.93958),new klb.Tp(55,87.94456),new klb.Tp(56,88.94923)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(31,64.96466),new klb.Tp(32,65.95521),new klb.Tp(33,66.95009),new klb.Tp(34,67.94187),new klb.Tp(35,68.939562155),new klb.Tp(36,69.933504),new klb.Tp(37,70.931868378),new klb.Tp(38,71.927112313),new klb.Tp(39,72.9267668),new klb.Tp(40,73.922476561),new klb.Tp(41,74.922523571),new klb.Tp(42,75.919214107),new klb.Tp(43,76.91991461),new klb.Tp(44,77.917309522),new klb.Tp(45,78.918499802),new klb.Tp(46,79.916521828),new klb.Tp(47,80.917992931),new klb.Tp(48,81.9167),new klb.Tp(49,82.919119072),new klb.Tp(50,83.918464523),new klb.Tp(51,84.922244678),new klb.Tp(52,85.924271165),new klb.Tp(53,86.928520749),new klb.Tp(54,87.931423982),new klb.Tp(55,88.93602),new klb.Tp(56,89.93942),new klb.Tp(57,90.94537),new klb.Tp(58,91.94933)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(32,66.96479),new klb.Tp(33,67.958248),new klb.Tp(34,68.950178),new klb.Tp(35,69.944208),new klb.Tp(36,70.939246),new klb.Tp(37,71.936496876),new klb.Tp(38,72.931794889),new klb.Tp(39,73.929891152),new klb.Tp(40,74.92577641),new klb.Tp(41,75.924541974),new klb.Tp(42,76.921380123),new klb.Tp(43,77.92114613),new klb.Tp(44,78.918337647),new klb.Tp(45,79.918529952),new klb.Tp(46,80.91629106),new klb.Tp(47,81.916804666),new klb.Tp(48,82.915180219),new klb.Tp(49,83.916503685),new klb.Tp(50,84.915608027),new klb.Tp(51,85.918797162),new klb.Tp(52,86.920710713),new klb.Tp(53,87.924065908),new klb.Tp(54,88.92638726),new klb.Tp(55,89.930634988),new klb.Tp(56,90.9339653),new klb.Tp(57,91.939255258),new klb.Tp(58,92.9431),new klb.Tp(59,93.94868)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(33,68.96532),new klb.Tp(34,69.95601),new klb.Tp(35,70.95051),new klb.Tp(36,71.94190754),new klb.Tp(37,72.938931115),new klb.Tp(38,73.933258225),new klb.Tp(39,74.931033794),new klb.Tp(40,75.925948304),new klb.Tp(41,76.92466788),new klb.Tp(42,77.920386271),new klb.Tp(43,78.920082992),new klb.Tp(44,79.91637804),new klb.Tp(45,80.916592419),new klb.Tp(46,81.913484601),new klb.Tp(47,82.914135952),new klb.Tp(48,83.911506627),new klb.Tp(49,84.912526954),new klb.Tp(50,85.910610313),new klb.Tp(51,86.913354251),new klb.Tp(52,87.914446951),new klb.Tp(53,88.917632505),new klb.Tp(54,89.919523803),new klb.Tp(55,90.923442418),new klb.Tp(56,91.926152752),new klb.Tp(57,92.931265246),new klb.Tp(58,93.934362),new klb.Tp(59,94.93984),new klb.Tp(60,95.94307),new klb.Tp(61,96.94856)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(34,70.96532),new klb.Tp(35,71.95908),new klb.Tp(36,72.950366),new klb.Tp(37,73.944470376),new klb.Tp(38,74.938569199),new klb.Tp(39,75.935071448),new klb.Tp(40,76.930406599),new klb.Tp(41,77.928141485),new klb.Tp(42,78.923996719),new klb.Tp(43,79.922519322),new klb.Tp(44,80.918994165),new klb.Tp(45,81.918207691),new klb.Tp(46,82.915111951),new klb.Tp(47,83.914384676),new klb.Tp(48,84.911789341),new klb.Tp(49,85.91116708),new klb.Tp(50,86.909183465),new klb.Tp(51,87.911318556),new klb.Tp(52,88.912279939),new klb.Tp(53,89.914808941),new klb.Tp(54,90.91653416),new klb.Tp(55,91.919725442),new klb.Tp(56,92.922032765),new klb.Tp(57,93.926407326),new klb.Tp(58,94.92931926),new klb.Tp(59,95.934283962),new klb.Tp(60,96.937342863),new klb.Tp(61,97.941703557),new klb.Tp(62,98.945420616),new klb.Tp(63,99.94987),new klb.Tp(64,100.953195994),new klb.Tp(65,101.95921)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(35,72.96597),new klb.Tp(36,73.95631),new klb.Tp(37,74.94992),new klb.Tp(38,75.94161),new klb.Tp(39,76.937761511),new klb.Tp(40,77.932179362),new klb.Tp(41,78.929707076),new klb.Tp(42,79.924524588),new klb.Tp(43,80.923213095),new klb.Tp(44,81.918401258),new klb.Tp(45,82.917555029),new klb.Tp(46,83.913424778),new klb.Tp(47,84.912932689),new klb.Tp(48,85.909262351),new klb.Tp(49,86.908879316),new klb.Tp(50,87.905614339),new klb.Tp(51,88.907452906),new klb.Tp(52,89.907737596),new klb.Tp(53,90.910209845),new klb.Tp(54,91.911029895),new klb.Tp(55,92.91402241),new klb.Tp(56,93.915359856),new klb.Tp(57,94.919358213),new klb.Tp(58,95.921680473),new klb.Tp(59,96.926148757),new klb.Tp(60,97.928471177),new klb.Tp(61,98.933315038),new klb.Tp(62,99.935351729),new klb.Tp(63,100.940517434),new klb.Tp(64,101.943018795),new klb.Tp(65,102.94895),new klb.Tp(66,103.95233)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(38,76.94962),new klb.Tp(39,77.9435),new klb.Tp(40,78.937350712),new klb.Tp(41,79.931982402),new klb.Tp(42,80.929128719),new klb.Tp(43,81.926792071),new klb.Tp(44,82.922352572),new klb.Tp(45,83.920387768),new klb.Tp(46,84.916427076),new klb.Tp(47,85.914887724),new klb.Tp(48,86.910877833),new klb.Tp(49,87.909503361),new klb.Tp(50,88.905847902),new klb.Tp(51,89.907151443),new klb.Tp(52,90.907303415),new klb.Tp(53,91.908946832),new klb.Tp(54,92.909581582),new klb.Tp(55,93.911594008),new klb.Tp(56,94.912823709),new klb.Tp(57,95.915897787),new klb.Tp(58,96.918131017),new klb.Tp(59,97.922219525),new klb.Tp(60,98.924634736),new klb.Tp(61,99.927756402),new klb.Tp(62,100.930313395),new klb.Tp(63,101.933555501),new klb.Tp(64,102.93694),new klb.Tp(65,103.94145),new klb.Tp(66,104.94509),new klb.Tp(67,105.95022)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(39,78.94916),new klb.Tp(40,79.94055),new klb.Tp(41,80.936815296),new klb.Tp(42,81.931086249),new klb.Tp(43,82.92865213),new klb.Tp(44,83.92325),new klb.Tp(45,84.92146522),new klb.Tp(46,85.916472851),new klb.Tp(47,86.914816578),new klb.Tp(48,87.910226179),new klb.Tp(49,88.908888916),new klb.Tp(50,89.904703679),new klb.Tp(51,90.905644968),new klb.Tp(52,91.905040106),new klb.Tp(53,92.906475627),new klb.Tp(54,93.906315765),new klb.Tp(55,94.908042739),new klb.Tp(56,95.908275675),new klb.Tp(57,96.910950716),new klb.Tp(58,97.912746366),new klb.Tp(59,98.916511084),new klb.Tp(60,99.917761704),new klb.Tp(61,100.921139958),new klb.Tp(62,101.922981089),new klb.Tp(63,102.926597062),new klb.Tp(64,103.92878),new klb.Tp(65,104.93305),new klb.Tp(66,105.93591),new klb.Tp(67,106.94086),new klb.Tp(68,107.94428)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(40,80.94905),new klb.Tp(41,81.94313),new klb.Tp(42,82.936703713),new klb.Tp(43,83.93357),new klb.Tp(44,84.927906486),new klb.Tp(45,85.925037588),new klb.Tp(46,86.920361435),new klb.Tp(47,87.91833144),new klb.Tp(48,88.913495503),new klb.Tp(49,89.911264109),new klb.Tp(50,90.906990538),new klb.Tp(51,91.907193214),new klb.Tp(52,92.906377543),new klb.Tp(53,93.907283457),new klb.Tp(54,94.906835178),new klb.Tp(55,95.908100076),new klb.Tp(56,96.908097144),new klb.Tp(57,97.91033069),new klb.Tp(58,98.911617864),new klb.Tp(59,99.914181434),new klb.Tp(60,100.915251567),new klb.Tp(61,101.918037417),new klb.Tp(62,102.919141297),new klb.Tp(63,103.922459464),new klb.Tp(64,104.923934023),new klb.Tp(65,105.92819),new klb.Tp(66,106.93031),new klb.Tp(67,107.93501),new klb.Tp(68,108.93763),new klb.Tp(69,109.94268)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(41,82.94874),new klb.Tp(42,83.94009),new klb.Tp(43,84.93659),new klb.Tp(44,85.930695167),new klb.Tp(45,86.92732683),new klb.Tp(46,87.921952728),new klb.Tp(47,88.919480562),new klb.Tp(48,89.913936161),new klb.Tp(49,90.911750754),new klb.Tp(50,91.90681048),new klb.Tp(51,92.906812213),new klb.Tp(52,93.905087578),new klb.Tp(53,94.905841487),new klb.Tp(54,95.904678904),new klb.Tp(55,96.906021033),new klb.Tp(56,97.905407846),new klb.Tp(57,98.907711598),new klb.Tp(58,99.907477149),new klb.Tp(59,100.910346543),new klb.Tp(60,101.910297162),new klb.Tp(61,102.913204596),new klb.Tp(62,103.913758387),new klb.Tp(63,104.916972087),new klb.Tp(64,105.918134284),new klb.Tp(65,106.921694724),new klb.Tp(66,107.923973837),new klb.Tp(67,108.92781),new klb.Tp(68,109.92973),new klb.Tp(69,110.93451),new klb.Tp(70,111.93684),new klb.Tp(71,112.94203)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(42,84.94894),new klb.Tp(43,85.94288),new klb.Tp(44,86.93653),new klb.Tp(45,87.93283),new klb.Tp(46,88.92754288),new klb.Tp(47,89.92355583),new klb.Tp(48,90.9184282),new klb.Tp(49,91.915259655),new klb.Tp(50,92.910248473),new klb.Tp(51,93.909656309),new klb.Tp(52,94.907656454),new klb.Tp(53,95.907870803),new klb.Tp(54,96.906364843),new klb.Tp(55,97.907215692),new klb.Tp(56,98.906254554),new klb.Tp(57,99.907657594),new klb.Tp(58,100.90731438),new klb.Tp(59,101.909212938),new klb.Tp(60,102.909178805),new klb.Tp(61,103.911444898),new klb.Tp(62,104.911658043),new klb.Tp(63,105.914355408),new klb.Tp(64,106.915081691),new klb.Tp(65,107.918479973),new klb.Tp(66,108.919980998),new klb.Tp(67,109.92339),new klb.Tp(68,110.92505),new klb.Tp(69,111.92924),new klb.Tp(70,112.93133),new klb.Tp(71,113.93588),new klb.Tp(72,114.93828)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(43,86.94918),new klb.Tp(44,87.94042),new klb.Tp(45,88.93611),new klb.Tp(46,89.92978),new klb.Tp(47,90.926377434),new klb.Tp(48,91.92012),new klb.Tp(49,92.917051523),new klb.Tp(50,93.911359569),new klb.Tp(51,94.910412729),new klb.Tp(52,95.907597681),new klb.Tp(53,96.907554546),new klb.Tp(54,97.905287111),new klb.Tp(55,98.905939307),new klb.Tp(56,99.904219664),new klb.Tp(57,100.905582219),new klb.Tp(58,101.904349503),new klb.Tp(59,102.906323677),new klb.Tp(60,103.905430145),new klb.Tp(61,104.907750341),new klb.Tp(62,105.907326913),new klb.Tp(63,106.909907207),new klb.Tp(64,107.910192211),new klb.Tp(65,108.913201565),new klb.Tp(66,109.913966185),new klb.Tp(67,110.91756),new klb.Tp(68,111.918821673),new klb.Tp(69,112.92254),new klb.Tp(70,113.923891981),new klb.Tp(71,114.92831),new klb.Tp(72,115.93016),new klb.Tp(73,116.93479),new klb.Tp(74,117.93703)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(44,88.94938),new klb.Tp(45,89.94287),new klb.Tp(46,90.93655),new klb.Tp(47,91.93198),new klb.Tp(48,92.92574),new klb.Tp(49,93.921698),new klb.Tp(50,94.915898541),new klb.Tp(51,95.914518212),new klb.Tp(52,96.911336643),new klb.Tp(53,97.910716431),new klb.Tp(54,98.908132101),new klb.Tp(55,99.90811663),new klb.Tp(56,100.906163526),new klb.Tp(57,101.906842845),new klb.Tp(58,102.905504182),new klb.Tp(59,103.906655315),new klb.Tp(60,104.905692444),new klb.Tp(61,105.907284615),new klb.Tp(62,106.90675054),new klb.Tp(63,107.908730768),new klb.Tp(64,108.908735621),new klb.Tp(65,109.910949525),new klb.Tp(66,110.91166),new klb.Tp(67,111.913969253),new klb.Tp(68,112.91542),new klb.Tp(69,113.91734336),new klb.Tp(70,114.920124676),new klb.Tp(71,115.922746643),new klb.Tp(72,116.92535),new klb.Tp(73,117.92943),new klb.Tp(74,118.93136),new klb.Tp(75,119.93578),new klb.Tp(76,120.93808)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(45,90.94948),new klb.Tp(46,91.94042),new klb.Tp(47,92.93591),new klb.Tp(48,93.92877),new klb.Tp(49,94.92469),new klb.Tp(50,95.91822194),new klb.Tp(51,96.916478921),new klb.Tp(52,97.912720751),new klb.Tp(53,98.911767757),new klb.Tp(54,99.908504596),new klb.Tp(55,100.908289144),new klb.Tp(56,101.905607716),new klb.Tp(57,102.906087204),new klb.Tp(58,103.904034912),new klb.Tp(59,104.905084046),new klb.Tp(60,105.903483087),new klb.Tp(61,106.905128453),new klb.Tp(62,107.903894451),new klb.Tp(63,108.905953535),new klb.Tp(64,109.905152385),new klb.Tp(65,110.907643952),new klb.Tp(66,111.907313277),new klb.Tp(67,112.910151346),new klb.Tp(68,113.910365322),new klb.Tp(69,114.91368341),new klb.Tp(70,115.914158288),new klb.Tp(71,116.91784),new klb.Tp(72,117.918983915),new klb.Tp(73,118.92268),new klb.Tp(74,119.92403),new klb.Tp(75,120.92818),new klb.Tp(76,121.9298),new klb.Tp(77,122.93426)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(47,93.94278),new klb.Tp(48,94.93548),new klb.Tp(49,95.93068),new klb.Tp(50,96.924),new klb.Tp(51,97.921759995),new klb.Tp(52,98.917597103),new klb.Tp(53,99.916069387),new klb.Tp(54,100.912802135),new klb.Tp(55,101.911999996),new klb.Tp(56,102.908972453),new klb.Tp(57,103.908628228),new klb.Tp(58,104.906528234),new klb.Tp(59,105.906666431),new klb.Tp(60,106.90509302),new klb.Tp(61,107.905953705),new klb.Tp(62,108.904755514),new klb.Tp(63,109.90611046),new klb.Tp(64,110.905294679),new klb.Tp(65,111.907004132),new klb.Tp(66,112.906565708),new klb.Tp(67,113.908807907),new klb.Tp(68,114.908762282),new klb.Tp(69,115.911359558),new klb.Tp(70,116.911684187),new klb.Tp(71,117.914582383),new klb.Tp(72,118.915666045),new klb.Tp(73,119.918788609),new klb.Tp(74,120.919851074),new klb.Tp(75,121.92332),new klb.Tp(76,122.9249),new klb.Tp(77,123.92853),new klb.Tp(78,124.93054),new klb.Tp(79,125.9345),new klb.Tp(80,126.93688)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(48,95.93977),new klb.Tp(49,96.93494),new klb.Tp(50,97.927579),new klb.Tp(51,98.92501),new klb.Tp(52,99.920230232),new klb.Tp(53,100.918681442),new klb.Tp(54,101.914777255),new klb.Tp(55,102.913418952),new klb.Tp(56,103.909848091),new klb.Tp(57,104.909467818),new klb.Tp(58,105.906458007),new klb.Tp(59,106.906614232),new klb.Tp(60,107.904183403),new klb.Tp(61,108.904985569),new klb.Tp(62,109.903005578),new klb.Tp(63,110.904181628),new klb.Tp(64,111.902757226),new klb.Tp(65,112.904400947),new klb.Tp(66,113.903358121),new klb.Tp(67,114.905430553),new klb.Tp(68,115.904755434),new klb.Tp(69,116.907218242),new klb.Tp(70,117.906914144),new klb.Tp(71,118.909922582),new klb.Tp(72,119.909851352),new klb.Tp(73,120.91298039),new klb.Tp(74,121.9135),new klb.Tp(75,122.917003675),new klb.Tp(76,123.917648302),new klb.Tp(77,124.92124717),new klb.Tp(78,125.922353996),new klb.Tp(79,126.926434822),new klb.Tp(80,127.927760617),new klb.Tp(81,128.93226),new klb.Tp(82,129.93398)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(49,97.94224),new klb.Tp(50,98.93461),new klb.Tp(51,99.931149033),new klb.Tp(52,100.92656),new klb.Tp(53,101.924707541),new klb.Tp(54,102.919913896),new klb.Tp(55,103.918338416),new klb.Tp(56,104.914673434),new klb.Tp(57,105.913461134),new klb.Tp(58,106.910292195),new klb.Tp(59,107.909719683),new klb.Tp(60,108.907154078),new klb.Tp(61,109.907168783),new klb.Tp(62,110.905110677),new klb.Tp(63,111.905533338),new klb.Tp(64,112.904061223),new klb.Tp(65,113.904916758),new klb.Tp(66,114.903878328),new klb.Tp(67,115.905259995),new klb.Tp(68,116.904515731),new klb.Tp(69,117.906354623),new klb.Tp(70,118.905846334),new klb.Tp(71,119.907961505),new klb.Tp(72,120.907848847),new klb.Tp(73,121.910277103),new klb.Tp(74,122.910438951),new klb.Tp(75,123.913175916),new klb.Tp(76,124.913601387),new klb.Tp(77,125.916464532),new klb.Tp(78,126.917344048),new klb.Tp(79,127.920170658),new klb.Tp(80,128.921657958),new klb.Tp(81,129.924854941),new klb.Tp(82,130.926767408),new klb.Tp(83,131.932919005),new klb.Tp(84,132.93834),new klb.Tp(85,133.94466)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(50,99.938954),new klb.Tp(51,100.93606),new klb.Tp(52,101.93049),new klb.Tp(53,102.92813),new klb.Tp(54,103.923185469),new klb.Tp(55,104.921390409),new klb.Tp(56,105.916880472),new klb.Tp(57,106.915666702),new klb.Tp(58,107.911965339),new klb.Tp(59,108.911286879),new klb.Tp(60,109.907852688),new klb.Tp(61,110.907735404),new klb.Tp(62,111.90482081),new klb.Tp(63,112.905173373),new klb.Tp(64,113.902781816),new klb.Tp(65,114.903345973),new klb.Tp(66,115.901744149),new klb.Tp(67,116.902953765),new klb.Tp(68,117.901606328),new klb.Tp(69,118.90330888),new klb.Tp(70,119.902196571),new klb.Tp(71,120.904236867),new klb.Tp(72,121.903440138),new klb.Tp(73,122.905721901),new klb.Tp(74,123.90527463),new klb.Tp(75,124.907784924),new klb.Tp(76,125.907653953),new klb.Tp(77,126.91035098),new klb.Tp(78,127.910534953),new klb.Tp(79,128.913439976),new klb.Tp(80,129.913852185),new klb.Tp(81,130.916919144),new klb.Tp(82,131.917744455),new klb.Tp(83,132.923814085),new klb.Tp(84,133.928463576),new klb.Tp(85,134.93473),new klb.Tp(86,135.93934),new klb.Tp(87,136.94579)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(52,102.94012),new klb.Tp(53,103.936287),new klb.Tp(54,104.931528593),new klb.Tp(55,105.928183134),new klb.Tp(56,106.92415),new klb.Tp(57,107.92216),new klb.Tp(58,108.918136092),new klb.Tp(59,109.917533911),new klb.Tp(60,110.912534147),new klb.Tp(61,111.91239464),new klb.Tp(62,112.909377941),new klb.Tp(63,113.909095876),new klb.Tp(64,114.906598812),new klb.Tp(65,115.906797235),new klb.Tp(66,116.90483959),new klb.Tp(67,117.905531885),new klb.Tp(68,118.90394646),new klb.Tp(69,119.905074315),new klb.Tp(70,120.903818044),new klb.Tp(71,121.905175415),new klb.Tp(72,122.904215696),new klb.Tp(73,123.905937525),new klb.Tp(74,124.905247804),new klb.Tp(75,125.907248153),new klb.Tp(76,126.906914564),new klb.Tp(77,127.90916733),new klb.Tp(78,128.909150092),new klb.Tp(79,129.911546459),new klb.Tp(80,130.911946487),new klb.Tp(81,131.914413247),new klb.Tp(82,132.915236466),new klb.Tp(83,133.920551554),new klb.Tp(84,134.925167962),new klb.Tp(85,135.93066),new klb.Tp(86,136.93531),new klb.Tp(87,137.94096),new klb.Tp(88,138.94571)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(54,105.937702),new klb.Tp(55,106.935036),new klb.Tp(56,107.929486838),new klb.Tp(57,108.927456483),new klb.Tp(58,109.922407164),new klb.Tp(59,110.921120589),new klb.Tp(60,111.917061617),new klb.Tp(61,112.915452551),new klb.Tp(62,113.912498025),new klb.Tp(63,114.911578627),new klb.Tp(64,115.908420253),new klb.Tp(65,116.90863418),new klb.Tp(66,117.905825187),new klb.Tp(67,118.90640811),new klb.Tp(68,119.904019891),new klb.Tp(69,120.904929815),new klb.Tp(70,121.903047064),new klb.Tp(71,122.904272951),new klb.Tp(72,123.902819466),new klb.Tp(73,124.904424718),new klb.Tp(74,125.903305543),new klb.Tp(75,126.90521729),new klb.Tp(76,127.904461383),new klb.Tp(77,128.906595593),new klb.Tp(78,129.906222753),new klb.Tp(79,130.90852188),new klb.Tp(80,131.908523782),new klb.Tp(81,132.910939068),new klb.Tp(82,133.911540546),new klb.Tp(83,134.916450782),new klb.Tp(84,135.920103155),new klb.Tp(85,136.925324769),new klb.Tp(86,137.92922),new klb.Tp(87,138.93473),new klb.Tp(88,139.9387),new klb.Tp(89,140.94439),new klb.Tp(90,141.9485)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(55,107.943291),new klb.Tp(56,108.938191658),new klb.Tp(57,109.934634181),new klb.Tp(58,110.930276),new klb.Tp(59,111.92797),new klb.Tp(60,112.923644245),new klb.Tp(61,113.92185),new klb.Tp(62,114.918272),new klb.Tp(63,115.916735014),new klb.Tp(64,116.913647692),new klb.Tp(65,117.91337523),new klb.Tp(66,118.910180837),new klb.Tp(67,119.910047843),new klb.Tp(68,120.907366063),new klb.Tp(69,121.907592451),new klb.Tp(70,122.905597944),new klb.Tp(71,123.906211423),new klb.Tp(72,124.90462415),new klb.Tp(73,125.905619387),new klb.Tp(74,126.90446842),new klb.Tp(75,127.905805254),new klb.Tp(76,128.904987487),new klb.Tp(77,129.906674018),new klb.Tp(78,130.906124168),new klb.Tp(79,131.907994525),new klb.Tp(80,132.907806465),new klb.Tp(81,133.909876552),new klb.Tp(82,134.91005031),new klb.Tp(83,135.914655105),new klb.Tp(84,136.917872653),new klb.Tp(85,137.922383666),new klb.Tp(86,138.926093402),new klb.Tp(87,139.93121),new klb.Tp(88,140.93483),new klb.Tp(89,141.94018),new klb.Tp(90,142.94407),new klb.Tp(91,143.94961)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(56,109.944476),new klb.Tp(57,110.941632),new klb.Tp(58,111.93566535),new klb.Tp(59,112.933382836),new klb.Tp(60,113.928145),new klb.Tp(61,114.926979032),new klb.Tp(62,115.921394197),new klb.Tp(63,116.920564355),new klb.Tp(64,117.91657092),new klb.Tp(65,118.915554295),new klb.Tp(66,119.91215199),new klb.Tp(67,120.911386497),new klb.Tp(68,121.908548396),new klb.Tp(69,122.908470748),new klb.Tp(70,123.905895774),new klb.Tp(71,124.906398236),new klb.Tp(72,125.904268868),new klb.Tp(73,126.905179581),new klb.Tp(74,127.903530436),new klb.Tp(75,128.904779458),new klb.Tp(76,129.903507903),new klb.Tp(77,130.90508192),new klb.Tp(78,131.904154457),new klb.Tp(79,132.90590566),new klb.Tp(80,133.905394504),new klb.Tp(81,134.907207499),new klb.Tp(82,135.907219526),new klb.Tp(83,136.911562939),new klb.Tp(84,137.913988549),new klb.Tp(85,138.918786859),new klb.Tp(86,139.921635665),new klb.Tp(87,140.926646282),new klb.Tp(88,141.929702981),new klb.Tp(89,142.93489),new klb.Tp(90,143.93823),new klb.Tp(91,144.94367),new klb.Tp(92,145.9473),new klb.Tp(93,146.95301)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(57,111.950331),new klb.Tp(58,112.944535512),new klb.Tp(59,113.940841319),new klb.Tp(60,114.935939),new klb.Tp(61,115.932914152),new klb.Tp(62,116.928639484),new klb.Tp(63,117.926554883),new klb.Tp(64,118.922370879),new klb.Tp(65,119.920678219),new klb.Tp(66,120.917183637),new klb.Tp(67,121.916121946),new klb.Tp(68,122.912990168),new klb.Tp(69,123.912245731),new klb.Tp(70,124.909724871),new klb.Tp(71,125.909447953),new klb.Tp(72,126.9074176),new klb.Tp(73,127.907747919),new klb.Tp(74,128.906063369),new klb.Tp(75,129.906706163),new klb.Tp(76,130.905460232),new klb.Tp(77,131.906429799),new klb.Tp(78,132.90544687),new klb.Tp(79,133.906713419),new klb.Tp(80,134.905971903),new klb.Tp(81,135.907305741),new klb.Tp(82,136.907083505),new klb.Tp(83,137.911010537),new klb.Tp(84,138.913357921),new klb.Tp(85,139.917277075),new klb.Tp(86,140.920043984),new klb.Tp(87,141.924292317),new klb.Tp(88,142.927330292),new klb.Tp(89,143.932027373),new klb.Tp(90,144.935388226),new klb.Tp(91,145.940162028),new klb.Tp(92,146.943864435),new klb.Tp(93,147.948899539),new klb.Tp(94,148.95272),new klb.Tp(95,149.95797),new klb.Tp(96,150.962)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(58,113.950941),new klb.Tp(59,114.94771),new klb.Tp(60,115.94168),new klb.Tp(61,116.937700229),new klb.Tp(62,117.93344),new klb.Tp(63,118.931051927),new klb.Tp(64,119.926045941),new klb.Tp(65,120.924485908),new klb.Tp(66,121.92026),new klb.Tp(67,122.91885),new klb.Tp(68,123.915088437),new klb.Tp(69,124.914620234),new klb.Tp(70,125.911244146),new klb.Tp(71,126.911121328),new klb.Tp(72,127.90830887),new klb.Tp(73,128.908673749),new klb.Tp(74,129.906310478),new klb.Tp(75,130.906930798),new klb.Tp(76,131.905056152),new klb.Tp(77,132.906002368),new klb.Tp(78,133.904503347),new klb.Tp(79,134.905682749),new klb.Tp(80,135.904570109),new klb.Tp(81,136.905821414),new klb.Tp(82,137.905241273),new klb.Tp(83,138.908835384),new klb.Tp(84,139.910599485),new klb.Tp(85,140.914406439),new klb.Tp(86,141.916448175),new klb.Tp(87,142.920617184),new klb.Tp(88,143.922940468),new klb.Tp(89,144.926923807),new klb.Tp(90,145.930106645),new klb.Tp(91,146.933992519),new klb.Tp(92,147.937682377),new klb.Tp(93,148.94246),new klb.Tp(94,149.94562),new klb.Tp(95,150.9507),new klb.Tp(96,151.95416),new klb.Tp(97,152.95961)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(60,116.95001),new klb.Tp(61,117.94657),new klb.Tp(62,118.94099),new klb.Tp(63,119.93807),new klb.Tp(64,120.93301),new klb.Tp(65,121.93071),new klb.Tp(66,122.92624),new klb.Tp(67,123.92453),new klb.Tp(68,124.92067),new klb.Tp(69,125.91937),new klb.Tp(70,126.91616),new klb.Tp(71,127.91544794),new klb.Tp(72,128.912667334),new klb.Tp(73,129.91232),new klb.Tp(74,130.910108489),new klb.Tp(75,131.910110399),new klb.Tp(76,132.908396372),new klb.Tp(77,133.908489607),new klb.Tp(78,134.906971003),new klb.Tp(79,135.907651181),new klb.Tp(80,136.906465656),new klb.Tp(81,137.907106826),new klb.Tp(82,138.90634816),new klb.Tp(83,139.909472552),new klb.Tp(84,140.910957016),new klb.Tp(85,141.914074489),new klb.Tp(86,142.916058646),new klb.Tp(87,143.919591666),new klb.Tp(88,144.92163837),new klb.Tp(89,145.925700146),new klb.Tp(90,146.927819639),new klb.Tp(91,147.932191197),new klb.Tp(92,148.93437),new klb.Tp(93,149.93857),new klb.Tp(94,150.94156),new klb.Tp(95,151.94611),new klb.Tp(96,152.94945),new klb.Tp(97,153.9544),new klb.Tp(98,154.95813)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(61,118.95276),new klb.Tp(62,119.94664),new klb.Tp(63,120.94367),new klb.Tp(64,121.93801),new klb.Tp(65,122.93551),new klb.Tp(66,123.93052),new klb.Tp(67,124.92854),new klb.Tp(68,125.9241),new klb.Tp(69,126.92275),new klb.Tp(70,127.91887),new klb.Tp(71,128.918679183),new klb.Tp(72,129.914339361),new klb.Tp(73,130.914424137),new klb.Tp(74,131.91149),new klb.Tp(75,132.91155),new klb.Tp(76,133.909026379),new klb.Tp(77,134.909145555),new klb.Tp(78,135.907143574),new klb.Tp(79,136.907777634),new klb.Tp(80,137.905985574),new klb.Tp(81,138.906646605),new klb.Tp(82,139.905434035),new klb.Tp(83,140.908271103),new klb.Tp(84,141.909239733),new klb.Tp(85,142.912381158),new klb.Tp(86,143.913642686),new klb.Tp(87,144.917227871),new klb.Tp(88,145.918689722),new klb.Tp(89,146.922510962),new klb.Tp(90,147.924394738),new klb.Tp(91,148.928289207),new klb.Tp(92,149.930226399),new klb.Tp(93,150.93404),new klb.Tp(94,151.93638),new klb.Tp(95,152.94058),new klb.Tp(96,153.94332),new klb.Tp(97,154.94804),new klb.Tp(98,155.95126),new klb.Tp(99,156.95634)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(62,120.955364),new klb.Tp(63,121.95165),new klb.Tp(64,122.94596),new klb.Tp(65,123.94296),new klb.Tp(66,124.93783),new klb.Tp(67,125.93531),new klb.Tp(68,126.93083),new klb.Tp(69,127.9288),new klb.Tp(70,128.92486),new klb.Tp(71,129.92338),new klb.Tp(72,130.920060245),new klb.Tp(73,131.91912),new klb.Tp(74,132.9162),new klb.Tp(75,133.915672),new klb.Tp(76,134.91313914),new klb.Tp(77,135.912646935),new klb.Tp(78,136.910678351),new klb.Tp(79,137.910748891),new klb.Tp(80,138.908932181),new klb.Tp(81,139.909071204),new klb.Tp(82,140.907647726),new klb.Tp(83,141.910039865),new klb.Tp(84,142.910812233),new klb.Tp(85,143.913300595),new klb.Tp(86,144.914506897),new klb.Tp(87,145.917588016),new klb.Tp(88,146.918979001),new klb.Tp(89,147.922183237),new klb.Tp(90,148.923791056),new klb.Tp(91,149.926995031),new klb.Tp(92,150.928227869),new klb.Tp(93,151.9316),new klb.Tp(94,152.93365),new klb.Tp(95,153.93739),new klb.Tp(96,154.93999),new klb.Tp(97,155.94412),new klb.Tp(98,156.94717),new klb.Tp(99,157.95178),new klb.Tp(100,158.95523)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(66,125.94307),new klb.Tp(67,126.9405),new klb.Tp(68,127.93539),new klb.Tp(69,128.932385),new klb.Tp(70,129.92878),new klb.Tp(71,130.927102697),new klb.Tp(72,131.92312),new klb.Tp(73,132.92221),new klb.Tp(74,133.918645),new klb.Tp(75,134.91824),new klb.Tp(76,135.915020542),new klb.Tp(77,136.91463973),new klb.Tp(78,137.91291745),new klb.Tp(79,138.91192415),new klb.Tp(80,139.909309824),new klb.Tp(81,140.9096048),new klb.Tp(82,141.907718643),new klb.Tp(83,142.909809626),new klb.Tp(84,143.910082629),new klb.Tp(85,144.912568847),new klb.Tp(86,145.913112139),new klb.Tp(87,146.916095794),new klb.Tp(88,147.916888516),new klb.Tp(89,148.92014419),new klb.Tp(90,149.920886563),new klb.Tp(91,150.923824739),new klb.Tp(92,151.924682428),new klb.Tp(93,152.927694534),new klb.Tp(94,153.929483295),new klb.Tp(95,154.932629551),new klb.Tp(96,155.9352),new klb.Tp(97,156.93927),new klb.Tp(98,157.94187),new klb.Tp(99,158.94639),new klb.Tp(100,159.94939),new klb.Tp(101,160.95433)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(67,127.94826),new klb.Tp(68,128.94316),new klb.Tp(69,129.94045),new klb.Tp(70,130.9358),new klb.Tp(71,131.93375),new klb.Tp(72,132.92972),new klb.Tp(73,133.92849),new klb.Tp(74,134.924617),new klb.Tp(75,135.923447865),new klb.Tp(76,136.920713),new klb.Tp(77,137.920432261),new klb.Tp(78,138.916759814),new klb.Tp(79,139.915801649),new klb.Tp(80,140.913606636),new klb.Tp(81,141.912950738),new klb.Tp(82,142.910927571),new klb.Tp(83,143.912585768),new klb.Tp(84,144.912743879),new klb.Tp(85,145.914692165),new klb.Tp(86,146.915133898),new klb.Tp(87,147.917467786),new klb.Tp(88,148.918329195),new klb.Tp(89,149.920979477),new klb.Tp(90,150.921202693),new klb.Tp(91,151.923490557),new klb.Tp(92,152.924113189),new klb.Tp(93,153.926547019),new klb.Tp(94,154.928097047),new klb.Tp(95,155.931060357),new klb.Tp(96,156.9332),new klb.Tp(97,157.93669),new klb.Tp(98,158.93913),new klb.Tp(99,159.94299),new klb.Tp(100,160.94586),new klb.Tp(101,161.95029),new klb.Tp(102,162.95352)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(68,129.94863),new klb.Tp(69,130.94589),new klb.Tp(70,131.94082),new klb.Tp(71,132.93873),new klb.Tp(72,133.93402),new klb.Tp(73,134.93235),new klb.Tp(74,135.9283),new klb.Tp(75,136.927046709),new klb.Tp(76,137.92354),new klb.Tp(77,138.922302),new klb.Tp(78,139.918991),new klb.Tp(79,140.918468512),new klb.Tp(80,141.915193274),new klb.Tp(81,142.914623555),new klb.Tp(82,143.91199473),new klb.Tp(83,144.913405611),new klb.Tp(84,145.91303676),new klb.Tp(85,146.914893275),new klb.Tp(86,147.914817914),new klb.Tp(87,148.917179521),new klb.Tp(88,149.917271454),new klb.Tp(89,150.919928351),new klb.Tp(90,151.919728244),new klb.Tp(91,152.922093907),new klb.Tp(92,153.922205303),new klb.Tp(93,154.92463594),new klb.Tp(94,155.925526236),new klb.Tp(95,156.928354506),new klb.Tp(96,157.929987938),new klb.Tp(97,158.9332),new klb.Tp(98,159.93514),new klb.Tp(99,160.93883),new klb.Tp(100,161.94122),new klb.Tp(101,162.94536),new klb.Tp(102,163.94828),new klb.Tp(103,164.95298)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(69,131.95416),new klb.Tp(70,132.9489),new klb.Tp(71,133.94632),new klb.Tp(72,134.94172),new klb.Tp(73,135.9395),new klb.Tp(74,136.93521),new klb.Tp(75,137.93345),new klb.Tp(76,138.92882915),new klb.Tp(77,139.928083921),new klb.Tp(78,140.924885867),new klb.Tp(79,141.923400033),new klb.Tp(80,142.920286634),new klb.Tp(81,143.918774116),new klb.Tp(82,144.916261285),new klb.Tp(83,145.917199714),new klb.Tp(84,146.916741206),new klb.Tp(85,147.918153775),new klb.Tp(86,148.917925922),new klb.Tp(87,149.919698294),new klb.Tp(88,150.919846022),new klb.Tp(89,151.921740399),new klb.Tp(90,152.921226219),new klb.Tp(91,153.922975386),new klb.Tp(92,154.922889429),new klb.Tp(93,155.924750855),new klb.Tp(94,156.925419435),new klb.Tp(95,157.927841923),new klb.Tp(96,158.9290845),new klb.Tp(97,159.931460406),new klb.Tp(98,160.93368),new klb.Tp(99,161.93704),new klb.Tp(100,162.93921),new klb.Tp(101,163.94299),new klb.Tp(102,164.94572),new klb.Tp(103,165.94997),new klb.Tp(104,166.95305)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(72,135.94707),new klb.Tp(73,136.94465),new klb.Tp(74,137.93997),new klb.Tp(75,138.93808),new klb.Tp(76,139.933236934),new klb.Tp(77,140.93221),new klb.Tp(78,141.927908919),new klb.Tp(79,142.926738636),new klb.Tp(80,143.923390357),new klb.Tp(81,144.921687498),new klb.Tp(82,145.918305344),new klb.Tp(83,146.919089446),new klb.Tp(84,147.918109771),new klb.Tp(85,148.919336427),new klb.Tp(86,149.918655455),new klb.Tp(87,150.920344273),new klb.Tp(88,151.919787882),new klb.Tp(89,152.921746283),new klb.Tp(90,153.920862271),new klb.Tp(91,154.922618801),new klb.Tp(92,155.922119552),new klb.Tp(93,156.923956686),new klb.Tp(94,157.924100533),new klb.Tp(95,158.926385075),new klb.Tp(96,159.927050616),new klb.Tp(97,160.929665688),new klb.Tp(98,161.930981211),new klb.Tp(99,162.93399),new klb.Tp(100,163.93586),new klb.Tp(101,164.93938),new klb.Tp(102,165.9416),new klb.Tp(103,166.94557),new klb.Tp(104,167.94836),new klb.Tp(105,168.95287)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(73,137.95287),new klb.Tp(74,138.94803),new klb.Tp(75,139.945367985),new klb.Tp(76,140.94116),new klb.Tp(77,141.939073781),new klb.Tp(78,142.93475),new klb.Tp(79,143.93253),new klb.Tp(80,144.92888),new klb.Tp(81,145.927180629),new klb.Tp(82,146.924037176),new klb.Tp(83,147.924298636),new klb.Tp(84,148.92324163),new klb.Tp(85,149.923654158),new klb.Tp(86,150.923098169),new klb.Tp(87,151.924071324),new klb.Tp(88,152.923430858),new klb.Tp(89,153.924686236),new klb.Tp(90,154.923500411),new klb.Tp(91,155.924743749),new klb.Tp(92,156.924021155),new klb.Tp(93,157.92541026),new klb.Tp(94,158.925343135),new klb.Tp(95,159.927164021),new klb.Tp(96,160.927566289),new klb.Tp(97,161.929484803),new klb.Tp(98,162.930643942),new klb.Tp(99,163.933347253),new klb.Tp(100,164.93488),new klb.Tp(101,165.93805),new klb.Tp(102,166.94005),new klb.Tp(103,167.94364),new klb.Tp(104,168.94622),new klb.Tp(105,169.95025),new klb.Tp(106,170.9533)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(74,139.95379),new klb.Tp(75,140.95119),new klb.Tp(76,141.946695946),new klb.Tp(77,142.94383),new klb.Tp(78,143.93907),new klb.Tp(79,144.936717),new klb.Tp(80,145.932720118),new klb.Tp(81,146.930878496),new klb.Tp(82,147.927177882),new klb.Tp(83,148.927333981),new klb.Tp(84,149.925579728),new klb.Tp(85,150.92617963),new klb.Tp(86,151.924713874),new klb.Tp(87,152.925760865),new klb.Tp(88,153.924422046),new klb.Tp(89,154.92574895),new klb.Tp(90,155.924278273),new klb.Tp(91,156.925461256),new klb.Tp(92,157.924404637),new klb.Tp(93,158.92573566),new klb.Tp(94,159.925193718),new klb.Tp(95,160.926929595),new klb.Tp(96,161.926794731),new klb.Tp(97,162.928727532),new klb.Tp(98,163.929171165),new klb.Tp(99,164.931699828),new klb.Tp(100,165.932803241),new klb.Tp(101,166.935649025),new klb.Tp(102,167.93723),new klb.Tp(103,168.940303648),new klb.Tp(104,169.94267),new klb.Tp(105,170.94648),new klb.Tp(106,171.94911),new klb.Tp(107,172.95344)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(75,141.95986),new klb.Tp(76,142.95469),new klb.Tp(77,143.95164),new klb.Tp(78,144.94688),new klb.Tp(79,145.9441),new klb.Tp(80,146.93984),new klb.Tp(81,147.937269),new klb.Tp(82,148.933789944),new klb.Tp(83,149.932760914),new klb.Tp(84,150.931680791),new klb.Tp(85,151.931740598),new klb.Tp(86,152.930194506),new klb.Tp(87,153.930596268),new klb.Tp(88,154.929079084),new klb.Tp(89,155.929001869),new klb.Tp(90,156.928188059),new klb.Tp(91,157.92894573),new klb.Tp(92,158.927708537),new klb.Tp(93,159.928725679),new klb.Tp(94,160.927851662),new klb.Tp(95,161.92909242),new klb.Tp(96,162.928730286),new klb.Tp(97,163.930230577),new klb.Tp(98,164.930319169),new klb.Tp(99,165.932281267),new klb.Tp(100,166.933126195),new klb.Tp(101,167.935496424),new klb.Tp(102,168.936868306),new klb.Tp(103,169.939614951),new klb.Tp(104,170.941461227),new klb.Tp(105,171.94482),new klb.Tp(106,172.94729),new klb.Tp(107,173.95115),new klb.Tp(108,174.95405)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(76,143.96059),new klb.Tp(77,144.95746),new klb.Tp(78,145.95212),new klb.Tp(79,146.94931),new klb.Tp(80,147.94444),new klb.Tp(81,148.942780527),new klb.Tp(82,149.937171034),new klb.Tp(83,150.93746),new klb.Tp(84,151.935078452),new klb.Tp(85,152.935093125),new klb.Tp(86,153.932777294),new klb.Tp(87,154.933204273),new klb.Tp(88,155.931015001),new klb.Tp(89,156.931945517),new klb.Tp(90,157.929912),new klb.Tp(91,158.930680718),new klb.Tp(92,159.929078924),new klb.Tp(93,160.930001348),new klb.Tp(94,161.928774923),new klb.Tp(95,162.930029273),new klb.Tp(96,163.929196996),new klb.Tp(97,164.9307228),new klb.Tp(98,165.93028997),new klb.Tp(99,166.932045448),new klb.Tp(100,167.932367781),new klb.Tp(101,168.934588082),new klb.Tp(102,169.935460334),new klb.Tp(103,170.938025885),new klb.Tp(104,171.939352149),new klb.Tp(105,172.9424),new klb.Tp(106,173.94434),new klb.Tp(107,174.94793),new klb.Tp(108,175.95029),new klb.Tp(109,176.95437)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(77,145.966495),new klb.Tp(78,146.961081),new klb.Tp(79,147.95755),new klb.Tp(80,148.95265),new klb.Tp(81,149.94967),new klb.Tp(82,150.944842),new klb.Tp(83,151.9443),new klb.Tp(84,152.942027631),new klb.Tp(85,153.940832325),new klb.Tp(86,154.939191562),new klb.Tp(87,155.939006895),new klb.Tp(88,156.936756069),new klb.Tp(89,157.936996),new klb.Tp(90,158.934808966),new klb.Tp(91,159.935090772),new klb.Tp(92,160.933398042),new klb.Tp(93,161.933970147),new klb.Tp(94,162.932647648),new klb.Tp(95,163.933450972),new klb.Tp(96,164.932432463),new klb.Tp(97,165.933553133),new klb.Tp(98,166.932848844),new klb.Tp(99,167.934170375),new klb.Tp(100,168.934211117),new klb.Tp(101,169.935797877),new klb.Tp(102,170.936425817),new klb.Tp(103,171.938396118),new klb.Tp(104,172.939600336),new klb.Tp(105,173.942164618),new klb.Tp(106,174.943832897),new klb.Tp(107,175.946991412),new klb.Tp(108,176.94904),new klb.Tp(109,177.95264),new klb.Tp(110,178.95534)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(78,147.96676),new klb.Tp(79,148.96348),new klb.Tp(80,149.95799),new klb.Tp(81,150.954657965),new klb.Tp(82,151.950167),new klb.Tp(83,152.94921),new klb.Tp(84,153.945651145),new klb.Tp(85,154.945792),new klb.Tp(86,155.942847109),new klb.Tp(87,156.94265865),new klb.Tp(88,157.939857897),new klb.Tp(89,158.940153735),new klb.Tp(90,159.93756),new klb.Tp(91,160.937357719),new klb.Tp(92,161.93575),new klb.Tp(93,162.936265492),new klb.Tp(94,163.93452),new klb.Tp(95,164.935397592),new klb.Tp(96,165.933879623),new klb.Tp(97,166.934946862),new klb.Tp(98,167.933894465),new klb.Tp(99,168.93518712),new klb.Tp(100,169.934758652),new klb.Tp(101,170.936322297),new klb.Tp(102,171.936377696),new klb.Tp(103,172.938206756),new klb.Tp(104,173.938858101),new klb.Tp(105,174.941272494),new klb.Tp(106,175.942568409),new klb.Tp(107,176.945257126),new klb.Tp(108,177.946643396),new klb.Tp(109,178.95017),new klb.Tp(110,179.95233),new klb.Tp(111,180.95615)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(79,149.972668),new klb.Tp(80,150.967147),new klb.Tp(81,151.96361),new klb.Tp(82,152.95869),new klb.Tp(83,153.9571),new klb.Tp(84,154.953641324),new klb.Tp(85,155.952907),new klb.Tp(86,156.950101536),new klb.Tp(87,157.948577981),new klb.Tp(88,158.946615113),new klb.Tp(89,159.945383),new klb.Tp(90,160.943047504),new klb.Tp(91,161.943222),new klb.Tp(92,162.941203796),new klb.Tp(93,163.941215),new klb.Tp(94,164.939605886),new klb.Tp(95,165.939762646),new klb.Tp(96,166.938307056),new klb.Tp(97,167.938698576),new klb.Tp(98,168.937648757),new klb.Tp(99,169.93847219),new klb.Tp(100,170.937909903),new klb.Tp(101,171.939082239),new klb.Tp(102,172.938926901),new klb.Tp(103,173.940333522),new klb.Tp(104,174.940767904),new klb.Tp(105,175.942682399),new klb.Tp(106,176.943754987),new klb.Tp(107,177.945951366),new klb.Tp(108,178.947324216),new klb.Tp(109,179.949879968),new klb.Tp(110,180.95197),new klb.Tp(111,181.95521),new klb.Tp(112,182.95757),new klb.Tp(113,183.96117)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(82,153.96425),new klb.Tp(83,154.96276),new klb.Tp(84,155.959247),new klb.Tp(85,156.958127),new klb.Tp(86,157.95405528),new klb.Tp(87,158.954003),new klb.Tp(88,159.950713588),new klb.Tp(89,160.950330852),new klb.Tp(90,161.947202977),new klb.Tp(91,162.947057),new klb.Tp(92,163.944422),new klb.Tp(93,164.94454),new klb.Tp(94,165.94225),new klb.Tp(95,166.9426),new klb.Tp(96,167.94063),new klb.Tp(97,168.941158567),new klb.Tp(98,169.93965),new klb.Tp(99,170.94049),new klb.Tp(100,171.93945798),new klb.Tp(101,172.94065),new klb.Tp(102,173.940040159),new klb.Tp(103,174.941502991),new klb.Tp(104,175.941401828),new klb.Tp(105,176.943220013),new klb.Tp(106,177.943697732),new klb.Tp(107,178.945815073),new klb.Tp(108,179.94654876),new klb.Tp(109,180.949099124),new klb.Tp(110,181.950552893),new klb.Tp(111,182.953531012),new klb.Tp(112,183.95544788),new klb.Tp(113,184.95878),new klb.Tp(114,185.96092)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(83,155.971689),new klb.Tp(84,156.968145),new klb.Tp(85,157.966368),new klb.Tp(86,158.96232309),new klb.Tp(87,159.961358),new klb.Tp(88,160.958372992),new klb.Tp(89,161.956556553),new klb.Tp(90,162.95431665),new klb.Tp(91,163.95357),new klb.Tp(92,164.950817),new klb.Tp(93,165.95047),new klb.Tp(94,166.948639),new klb.Tp(95,167.947787),new klb.Tp(96,168.94592),new klb.Tp(97,169.94609),new klb.Tp(98,170.94446),new klb.Tp(99,171.944739818),new klb.Tp(100,172.94459),new klb.Tp(101,173.944167937),new klb.Tp(102,174.94365),new klb.Tp(103,175.944740551),new klb.Tp(104,176.944471766),new klb.Tp(105,177.945750349),new klb.Tp(106,178.945934113),new klb.Tp(107,179.947465655),new klb.Tp(108,180.947996346),new klb.Tp(109,181.950152414),new klb.Tp(110,182.951373188),new klb.Tp(111,183.954009331),new klb.Tp(112,184.955559086),new klb.Tp(113,185.9585501),new klb.Tp(114,186.96041),new klb.Tp(115,187.96371)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(84,157.973939),new klb.Tp(85,158.97228),new klb.Tp(86,159.968369),new klb.Tp(87,160.967089),new klb.Tp(88,161.962750303),new klb.Tp(89,162.962532),new klb.Tp(90,163.95898381),new klb.Tp(91,164.958335962),new klb.Tp(92,165.955019896),new klb.Tp(93,166.954672),new klb.Tp(94,167.951863),new klb.Tp(95,168.951759),new klb.Tp(96,169.948473988),new klb.Tp(97,170.94946),new klb.Tp(98,171.948228837),new klb.Tp(99,172.948884),new klb.Tp(100,173.94616),new klb.Tp(101,174.94677),new klb.Tp(102,175.94559),new klb.Tp(103,176.94662),new klb.Tp(104,177.945848364),new klb.Tp(105,178.947071733),new klb.Tp(106,179.946705734),new klb.Tp(107,180.948198054),new klb.Tp(108,181.948205519),new klb.Tp(109,182.950224458),new klb.Tp(110,183.950932553),new klb.Tp(111,184.953420586),new klb.Tp(112,185.954362204),new klb.Tp(113,186.957158365),new klb.Tp(114,187.958486954),new klb.Tp(115,188.96191222),new klb.Tp(116,189.963179541)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(85,159.981485),new klb.Tp(86,160.977661),new klb.Tp(87,161.975707),new klb.Tp(88,162.971375872),new klb.Tp(89,163.970319),new klb.Tp(90,164.967050268),new klb.Tp(91,165.965211372),new klb.Tp(92,166.962564),new klb.Tp(93,167.961609),new klb.Tp(94,168.95883),new klb.Tp(95,169.958163),new klb.Tp(96,170.955547),new klb.Tp(97,171.955285),new klb.Tp(98,172.953062),new klb.Tp(99,173.952114),new klb.Tp(100,174.951393),new klb.Tp(101,175.95157),new klb.Tp(102,176.95027),new klb.Tp(103,177.950851081),new klb.Tp(104,178.949981038),new klb.Tp(105,179.95078768),new klb.Tp(106,180.950064596),new klb.Tp(107,181.951211444),new klb.Tp(108,182.950821349),new klb.Tp(109,183.952524289),new klb.Tp(110,184.952955747),new klb.Tp(111,185.954986529),new klb.Tp(112,186.955750787),new klb.Tp(113,187.958112287),new klb.Tp(114,188.959228359),new klb.Tp(115,189.961816139),new klb.Tp(116,190.963123592),new klb.Tp(117,191.96596)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(86,161.983819),new klb.Tp(87,162.982048),new klb.Tp(88,163.977927),new klb.Tp(89,164.976475),new klb.Tp(90,165.971934911),new klb.Tp(91,166.971554),new klb.Tp(92,167.967832911),new klb.Tp(93,168.967076205),new klb.Tp(94,169.963569716),new klb.Tp(95,170.96304),new klb.Tp(96,171.960078),new klb.Tp(97,172.959791),new klb.Tp(98,173.956307704),new klb.Tp(99,174.95708),new klb.Tp(100,175.953757941),new klb.Tp(101,176.955045),new klb.Tp(102,177.953348225),new klb.Tp(103,178.953951),new klb.Tp(104,179.952308241),new klb.Tp(105,180.953274494),new klb.Tp(106,181.952186222),new klb.Tp(107,182.95311),new klb.Tp(108,183.952490808),new klb.Tp(109,184.954043023),new klb.Tp(110,185.953838355),new klb.Tp(111,186.955747928),new klb.Tp(112,187.955835993),new klb.Tp(113,188.958144866),new klb.Tp(114,189.95844521),new klb.Tp(115,190.960927951),new klb.Tp(116,191.961479047),new klb.Tp(117,192.964148083),new klb.Tp(118,193.965179314),new klb.Tp(119,194.968123889),new klb.Tp(120,195.96962255)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(88,164.98758),new klb.Tp(89,165.985506),new klb.Tp(90,166.980951577),new klb.Tp(91,167.979966),new klb.Tp(92,168.976390868),new klb.Tp(93,169.974441697),new klb.Tp(94,170.971779),new klb.Tp(95,171.970643),new klb.Tp(96,172.967707),new klb.Tp(97,173.966804),new klb.Tp(98,174.964279),new klb.Tp(99,175.963511),new klb.Tp(100,176.96117),new klb.Tp(101,177.960084944),new klb.Tp(102,178.95915),new klb.Tp(103,179.958555615),new klb.Tp(104,180.957642156),new klb.Tp(105,181.958127689),new klb.Tp(106,182.956814),new klb.Tp(107,183.957388318),new klb.Tp(108,184.95659),new klb.Tp(109,185.957951104),new klb.Tp(110,186.95736083),new klb.Tp(111,187.958851962),new klb.Tp(112,188.958716473),new klb.Tp(113,189.960592299),new klb.Tp(114,190.960591191),new klb.Tp(115,191.962602198),new klb.Tp(116,192.9629237),new klb.Tp(117,193.96507561),new klb.Tp(118,194.9659768),new klb.Tp(119,195.968379906),new klb.Tp(120,196.969636496),new klb.Tp(121,197.97228),new klb.Tp(122,198.973787159)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(90,167.988035),new klb.Tp(91,168.986421),new klb.Tp(92,169.981734918),new klb.Tp(93,170.981251),new klb.Tp(94,171.977376138),new klb.Tp(95,172.976499642),new klb.Tp(96,173.972811276),new klb.Tp(97,174.972276),new klb.Tp(98,175.969),new klb.Tp(99,176.968453),new klb.Tp(100,177.964894223),new klb.Tp(101,178.965475),new klb.Tp(102,179.962023729),new klb.Tp(103,180.963177),new klb.Tp(104,181.961267637),new klb.Tp(105,182.961729),new klb.Tp(106,183.959851685),new klb.Tp(107,184.960753782),new klb.Tp(108,185.959432346),new klb.Tp(109,186.960697),new klb.Tp(110,187.959395697),new klb.Tp(111,188.9608319),new klb.Tp(112,189.959930073),new klb.Tp(113,190.961684653),new klb.Tp(114,191.961035158),new klb.Tp(115,192.962984504),new klb.Tp(116,193.962663581),new klb.Tp(117,194.964774449),new klb.Tp(118,195.964934884),new klb.Tp(119,196.967323401),new klb.Tp(120,197.967876009),new klb.Tp(121,198.970576213),new klb.Tp(122,199.971423885),new klb.Tp(123,200.974496467),new klb.Tp(124,201.97574)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(92,170.991183),new klb.Tp(93,171.990109),new klb.Tp(94,172.986398138),new klb.Tp(95,173.984325861),new klb.Tp(96,174.981552),new klb.Tp(97,175.980269),new klb.Tp(98,176.977215),new klb.Tp(99,177.975975),new klb.Tp(100,178.973412),new klb.Tp(101,179.972396),new klb.Tp(102,180.969948),new klb.Tp(103,181.968621416),new klb.Tp(104,182.96762),new klb.Tp(105,183.966776046),new klb.Tp(106,184.965806956),new klb.Tp(107,185.965997671),new klb.Tp(108,186.964562),new klb.Tp(109,187.965321662),new klb.Tp(110,188.9642243),new klb.Tp(111,189.964698757),new klb.Tp(112,190.963649239),new klb.Tp(113,191.964810107),new klb.Tp(114,192.964131745),new klb.Tp(115,193.96533889),new klb.Tp(116,194.965017928),new klb.Tp(117,195.966551315),new klb.Tp(118,196.966551609),new klb.Tp(119,197.968225244),new klb.Tp(120,198.968748016),new klb.Tp(121,199.970717886),new klb.Tp(122,200.971640839),new klb.Tp(123,201.973788431),new klb.Tp(124,202.975137256),new klb.Tp(125,203.977705),new klb.Tp(126,204.97961)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(95,174.991411),new klb.Tp(96,175.987413248),new klb.Tp(97,176.986336874),new klb.Tp(98,177.982476325),new klb.Tp(99,178.981783),new klb.Tp(100,179.978322),new klb.Tp(101,180.977806),new klb.Tp(102,181.97393546),new klb.Tp(103,182.974561),new klb.Tp(104,183.970705219),new klb.Tp(105,184.971983),new klb.Tp(106,185.969460021),new klb.Tp(107,186.969785),new klb.Tp(108,187.967511693),new klb.Tp(109,188.968733187),new klb.Tp(110,189.966958568),new klb.Tp(111,190.96706311),new klb.Tp(112,191.965921572),new klb.Tp(113,192.966644169),new klb.Tp(114,193.965381832),new klb.Tp(115,194.966638981),new klb.Tp(116,195.965814846),new klb.Tp(117,196.967195333),new klb.Tp(118,197.96675183),new klb.Tp(119,198.968262489),new klb.Tp(120,199.968308726),new klb.Tp(121,200.970285275),new klb.Tp(122,201.970625604),new klb.Tp(123,202.972857096),new klb.Tp(124,203.97347564),new klb.Tp(125,204.976056104),new klb.Tp(126,205.977498672),new klb.Tp(127,206.982577025),new klb.Tp(128,207.98594)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(96,176.996881),new klb.Tp(97,177.994637),new klb.Tp(98,178.991466),new klb.Tp(99,179.990194),new klb.Tp(100,180.986904),new klb.Tp(101,181.98561),new klb.Tp(102,182.982697),new klb.Tp(103,183.98176),new klb.Tp(104,184.9791),new klb.Tp(105,185.977549881),new klb.Tp(106,186.97617),new klb.Tp(107,187.97592),new klb.Tp(108,188.974290451),new klb.Tp(109,189.974473379),new klb.Tp(110,190.972261952),new klb.Tp(111,191.972770785),new klb.Tp(112,192.970548),new klb.Tp(113,193.971053),new klb.Tp(114,194.96965),new klb.Tp(115,195.970515),new klb.Tp(116,196.9695362),new klb.Tp(117,197.970466294),new klb.Tp(118,198.969813837),new klb.Tp(119,199.970945394),new klb.Tp(120,200.97080377),new klb.Tp(121,201.972090569),new klb.Tp(122,202.972329088),new klb.Tp(123,203.973848646),new klb.Tp(124,204.97441227),new klb.Tp(125,205.976095321),new klb.Tp(126,206.977407908),new klb.Tp(127,207.982004653),new klb.Tp(128,208.985349125),new klb.Tp(129,209.990065574)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(99,180.996714),new klb.Tp(100,181.992676101),new klb.Tp(101,182.99193),new klb.Tp(102,183.988198),new klb.Tp(103,184.98758),new klb.Tp(104,185.983485388),new klb.Tp(105,186.98403),new klb.Tp(106,187.979869108),new klb.Tp(107,188.98088),new klb.Tp(108,189.978180008),new klb.Tp(109,190.9782),new klb.Tp(110,191.975719811),new klb.Tp(111,192.97608),new klb.Tp(112,193.974648056),new klb.Tp(113,194.975920279),new klb.Tp(114,195.97271),new klb.Tp(115,196.97338),new klb.Tp(116,197.97198),new klb.Tp(117,198.972909384),new klb.Tp(118,199.97181556),new klb.Tp(119,200.972846589),new klb.Tp(120,201.972143786),new klb.Tp(121,202.973375491),new klb.Tp(122,203.973028761),new klb.Tp(123,204.974467112),new klb.Tp(124,205.974449002),new klb.Tp(125,206.975880605),new klb.Tp(126,207.97663585),new klb.Tp(127,208.981074801),new klb.Tp(128,209.984173129),new klb.Tp(129,210.988731474),new klb.Tp(130,211.991887495),new klb.Tp(131,212.9965),new klb.Tp(132,213.999798147)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(102,184.997708),new klb.Tp(103,185.99648),new klb.Tp(104,186.993458),new klb.Tp(105,187.992173),new klb.Tp(106,188.989505),new klb.Tp(107,189.987520007),new klb.Tp(108,190.986053),new klb.Tp(109,191.985368),new klb.Tp(110,192.983662229),new klb.Tp(111,193.983430186),new klb.Tp(112,194.98112697),new klb.Tp(113,195.981236107),new klb.Tp(114,196.978934287),new klb.Tp(115,197.979024396),new klb.Tp(116,198.977576953),new klb.Tp(117,199.978141983),new klb.Tp(118,200.976970721),new klb.Tp(119,201.977674504),new klb.Tp(120,202.976868118),new klb.Tp(121,203.977805161),new klb.Tp(122,204.977374688),new klb.Tp(123,205.978482854),new klb.Tp(124,206.978455217),new klb.Tp(125,207.979726699),new klb.Tp(126,208.980383241),new klb.Tp(127,209.984104944),new klb.Tp(128,210.987258139),new klb.Tp(129,211.991271542),new klb.Tp(130,212.994374836),new klb.Tp(131,213.998698664),new klb.Tp(132,215.001832349),new klb.Tp(133,216.006199)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(106,189.994293888),new klb.Tp(107,190.994653),new klb.Tp(108,191.99033039),new klb.Tp(109,192.991102),new klb.Tp(110,193.988284107),new klb.Tp(111,194.988045),new klb.Tp(112,195.985469432),new klb.Tp(113,196.985567),new klb.Tp(114,197.984024384),new klb.Tp(115,198.985044507),new klb.Tp(116,199.981735),new klb.Tp(117,200.982209),new klb.Tp(118,201.980704),new klb.Tp(119,202.981412863),new klb.Tp(120,203.980307113),new klb.Tp(121,204.981165396),new klb.Tp(122,205.980465241),new klb.Tp(123,206.981578228),new klb.Tp(124,207.981231059),new klb.Tp(125,208.982415788),new klb.Tp(126,209.982857396),new klb.Tp(127,210.986636869),new klb.Tp(128,211.988851755),new klb.Tp(129,212.992842522),new klb.Tp(130,213.995185949),new klb.Tp(131,214.999414609),new klb.Tp(132,216.001905198),new klb.Tp(133,217.006253),new klb.Tp(134,218.008965773)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(108,193.000188),new klb.Tp(109,193.997973),new klb.Tp(110,194.996554),new klb.Tp(111,195.995702),new klb.Tp(112,196.993891293),new klb.Tp(113,197.99343368),new klb.Tp(114,198.991008569),new klb.Tp(115,199.990920883),new klb.Tp(116,200.988486908),new klb.Tp(117,201.988448629),new klb.Tp(118,202.986847216),new klb.Tp(119,203.987261559),new klb.Tp(120,204.986036352),new klb.Tp(121,205.986599242),new klb.Tp(122,206.985775861),new klb.Tp(123,207.986582508),new klb.Tp(124,208.986158678),new klb.Tp(125,209.987131308),new klb.Tp(126,210.987480806),new klb.Tp(127,211.990734657),new klb.Tp(128,212.99292115),new klb.Tp(129,213.996356412),new klb.Tp(130,214.998641245),new klb.Tp(131,216.002408839),new klb.Tp(132,217.004709619),new klb.Tp(133,218.008681458),new klb.Tp(134,219.011296478),new klb.Tp(135,220.015301),new klb.Tp(136,221.01814),new klb.Tp(137,222.02233),new klb.Tp(138,223.02534)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(110,196.001117268),new klb.Tp(111,197.001661),new klb.Tp(112,197.998779978),new klb.Tp(113,198.998309),new klb.Tp(114,199.995634148),new klb.Tp(115,200.995535),new klb.Tp(116,201.993899382),new klb.Tp(117,202.994765192),new klb.Tp(118,203.991365),new klb.Tp(119,204.991668),new klb.Tp(120,205.99016),new klb.Tp(121,206.990726826),new klb.Tp(122,207.989631237),new klb.Tp(123,208.990376634),new klb.Tp(124,209.989679862),new klb.Tp(125,210.99058541),new klb.Tp(126,211.990688899),new klb.Tp(127,212.993868354),new klb.Tp(128,213.995346275),new klb.Tp(129,214.998729195),new klb.Tp(130,216.000258153),new klb.Tp(131,217.003914555),new klb.Tp(132,218.005586315),new klb.Tp(133,219.009474831),new klb.Tp(134,220.011384149),new klb.Tp(135,221.015455),new klb.Tp(136,222.017570472),new klb.Tp(137,223.02179),new klb.Tp(138,224.02409),new klb.Tp(139,225.02844),new klb.Tp(140,226.03089),new klb.Tp(141,227.035407),new klb.Tp(142,228.038084)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(113,200.006499),new klb.Tp(114,201.00458692),new klb.Tp(115,202.00396885),new klb.Tp(116,203.001423829),new klb.Tp(117,204.001221209),new klb.Tp(118,204.998663961),new klb.Tp(119,205.998486886),new klb.Tp(120,206.996859385),new klb.Tp(121,207.997133849),new klb.Tp(122,208.995915421),new klb.Tp(123,209.996398327),new klb.Tp(124,210.995529332),new klb.Tp(125,211.996194988),new klb.Tp(126,212.996174845),new klb.Tp(127,213.99895474),new klb.Tp(128,215.000326029),new klb.Tp(129,216.003187873),new klb.Tp(130,217.004616452),new klb.Tp(131,218.007563326),new klb.Tp(132,219.009240843),new klb.Tp(133,220.012312978),new klb.Tp(134,221.014245654),new klb.Tp(135,222.017543957),new klb.Tp(136,223.019730712),new klb.Tp(137,224.023235513),new klb.Tp(138,225.025606914),new klb.Tp(139,226.029343423),new klb.Tp(140,227.031833167),new klb.Tp(141,228.034776087),new klb.Tp(142,229.038426),new klb.Tp(143,230.04251),new klb.Tp(144,231.045407),new klb.Tp(145,232.049654)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(115,203.00921),new klb.Tp(116,204.006434513),new klb.Tp(117,205.006187),new klb.Tp(118,206.004463814),new klb.Tp(119,207.005176607),new klb.Tp(120,208.001776),new klb.Tp(121,209.001944),new klb.Tp(122,210.000446),new klb.Tp(123,211.000893996),new klb.Tp(124,211.999783492),new klb.Tp(125,213.000345847),new klb.Tp(126,214.000091141),new klb.Tp(127,215.002704195),new klb.Tp(128,216.003518402),new klb.Tp(129,217.00630601),new klb.Tp(130,218.007123948),new klb.Tp(131,219.010068787),new klb.Tp(132,220.011014669),new klb.Tp(133,221.013907762),new klb.Tp(134,222.01536182),new klb.Tp(135,223.01849714),new klb.Tp(136,224.020202004),new klb.Tp(137,225.023604463),new klb.Tp(138,226.025402555),new klb.Tp(139,227.029170677),new klb.Tp(140,228.031064101),new klb.Tp(141,229.034820309),new klb.Tp(142,230.037084774),new klb.Tp(143,231.04122),new klb.Tp(144,232.043693),new klb.Tp(145,233.047995),new klb.Tp(146,234.050547)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(118,207.012469754),new klb.Tp(119,208.012112949),new klb.Tp(120,209.009568736),new klb.Tp(121,210.009256802),new klb.Tp(122,211.007648196),new klb.Tp(123,212.007811441),new klb.Tp(124,213.006573689),new klb.Tp(125,214.006893072),new klb.Tp(126,215.006450832),new klb.Tp(127,216.008721268),new klb.Tp(128,217.009332676),new klb.Tp(129,218.011625045),new klb.Tp(130,219.012404918),new klb.Tp(131,220.014752105),new klb.Tp(132,221.015575746),new klb.Tp(133,222.017828852),new klb.Tp(134,223.01912603),new klb.Tp(135,224.021708435),new klb.Tp(136,225.023220576),new klb.Tp(137,226.026089848),new klb.Tp(138,227.027746979),new klb.Tp(139,228.031014825),new klb.Tp(140,229.032930871),new klb.Tp(141,230.036025144),new klb.Tp(142,231.038551503),new klb.Tp(143,232.042022474),new klb.Tp(144,233.04455),new klb.Tp(145,234.04842),new klb.Tp(146,235.051102),new klb.Tp(147,236.055178)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(120,210.015711883),new klb.Tp(121,211.016306912),new klb.Tp(122,212.012916),new klb.Tp(123,213.012962),new klb.Tp(124,214.011451),new klb.Tp(125,215.011726597),new klb.Tp(126,216.011050963),new klb.Tp(127,217.013066169),new klb.Tp(128,218.013267744),new klb.Tp(129,219.015521253),new klb.Tp(130,220.015733126),new klb.Tp(131,221.018171499),new klb.Tp(132,222.018454131),new klb.Tp(133,223.020795153),new klb.Tp(134,224.02145925),new klb.Tp(135,225.023941441),new klb.Tp(136,226.024890681),new klb.Tp(137,227.027698859),new klb.Tp(138,228.028731348),new klb.Tp(139,229.03175534),new klb.Tp(140,230.033126574),new klb.Tp(141,231.03629706),new klb.Tp(142,232.03805036),new klb.Tp(143,233.041576923),new klb.Tp(144,234.043595497),new klb.Tp(145,235.04750442),new klb.Tp(146,236.04971),new klb.Tp(147,237.053894),new klb.Tp(148,238.056243)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(122,213.021183209),new klb.Tp(123,214.02073923),new klb.Tp(124,215.019097612),new klb.Tp(125,216.019109649),new klb.Tp(126,217.018288571),new klb.Tp(127,218.020007906),new klb.Tp(128,219.019880348),new klb.Tp(129,220.021876493),new klb.Tp(130,221.021863742),new klb.Tp(131,222.023726),new klb.Tp(132,223.023963748),new klb.Tp(133,224.025614854),new klb.Tp(134,225.026115172),new klb.Tp(135,226.02793275),new klb.Tp(136,227.028793151),new klb.Tp(137,228.031036942),new klb.Tp(138,229.032088601),new klb.Tp(139,230.034532562),new klb.Tp(140,231.035878898),new klb.Tp(141,232.03858172),new klb.Tp(142,233.040240235),new klb.Tp(143,234.043302325),new klb.Tp(144,235.045436759),new klb.Tp(145,236.048675176),new klb.Tp(146,237.05113943),new klb.Tp(147,238.054497046),new klb.Tp(148,239.05713),new klb.Tp(149,240.06098)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(126,218.023487),new klb.Tp(127,219.024915423),new klb.Tp(128,220.024712),new klb.Tp(129,221.026351),new klb.Tp(130,222.02607),new klb.Tp(131,223.027722956),new klb.Tp(132,224.027590139),new klb.Tp(133,225.029384369),new klb.Tp(134,226.02933975),new klb.Tp(135,227.031140069),new klb.Tp(136,228.031366357),new klb.Tp(137,229.033496137),new klb.Tp(138,230.033927392),new klb.Tp(139,231.036289158),new klb.Tp(140,232.03714628),new klb.Tp(141,233.039628196),new klb.Tp(142,234.040945606),new klb.Tp(143,235.043923062),new klb.Tp(144,236.045561897),new klb.Tp(145,237.048723955),new klb.Tp(146,238.050782583),new klb.Tp(147,239.054287777),new klb.Tp(148,240.056585734),new klb.Tp(149,241.06033),new klb.Tp(150,242.062925)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(132,225.033899689),new klb.Tp(133,226.035129),new klb.Tp(134,227.034958261),new klb.Tp(135,228.03618),new klb.Tp(136,229.036246866),new klb.Tp(137,230.037812591),new klb.Tp(138,231.038233161),new klb.Tp(139,232.040099),new klb.Tp(140,233.04073235),new klb.Tp(141,234.042888556),new klb.Tp(142,235.044055876),new klb.Tp(143,236.046559724),new klb.Tp(144,237.048167253),new klb.Tp(145,238.050940464),new klb.Tp(146,239.052931399),new klb.Tp(147,240.056168828),new klb.Tp(148,241.058246266),new klb.Tp(149,242.061635),new klb.Tp(150,243.064273),new klb.Tp(151,244.06785)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(134,228.038727686),new klb.Tp(135,229.040138934),new klb.Tp(136,230.039645603),new klb.Tp(137,231.041258),new klb.Tp(138,232.041179445),new klb.Tp(139,233.04298757),new klb.Tp(140,234.043304681),new klb.Tp(141,235.0452815),new klb.Tp(142,236.046048088),new klb.Tp(143,237.048403774),new klb.Tp(144,238.0495534),new klb.Tp(145,239.052156519),new klb.Tp(146,240.05380746),new klb.Tp(147,241.056845291),new klb.Tp(148,242.058736847),new klb.Tp(149,243.061997013),new klb.Tp(150,244.06419765),new klb.Tp(151,245.067738657),new klb.Tp(152,246.070198429),new klb.Tp(153,247.07407)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(136,231.04556),new klb.Tp(137,232.04659),new klb.Tp(138,233.046472),new klb.Tp(139,234.047794),new klb.Tp(140,235.048029),new klb.Tp(141,236.049569),new klb.Tp(142,237.049970748),new klb.Tp(143,238.051977839),new klb.Tp(144,239.053018481),new klb.Tp(145,240.055287826),new klb.Tp(146,241.056822944),new klb.Tp(147,242.059543039),new klb.Tp(148,243.061372686),new klb.Tp(149,244.064279429),new klb.Tp(150,245.066445398),new klb.Tp(151,246.069768438),new klb.Tp(152,247.072086),new klb.Tp(153,248.075745),new klb.Tp(154,249.07848)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(137,233.0508),new klb.Tp(138,234.05024),new klb.Tp(139,235.051591),new klb.Tp(140,236.051405),new klb.Tp(141,237.052891),new klb.Tp(142,238.053016298),new klb.Tp(143,239.054951),new klb.Tp(144,240.055519046),new klb.Tp(145,241.057646736),new klb.Tp(146,242.058829326),new klb.Tp(147,243.061382249),new klb.Tp(148,244.062746349),new klb.Tp(149,245.065485586),new klb.Tp(150,246.067217551),new klb.Tp(151,247.070346811),new klb.Tp(152,248.072342247),new klb.Tp(153,249.075947062),new klb.Tp(154,250.078350687),new klb.Tp(155,251.082277873),new klb.Tp(156,252.08487)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(138,235.05658),new klb.Tp(139,236.05733),new klb.Tp(140,237.057127),new klb.Tp(141,238.058266),new klb.Tp(142,239.058362),new klb.Tp(143,240.059749),new klb.Tp(144,241.060223),new klb.Tp(145,242.06205),new klb.Tp(146,243.06300157),new klb.Tp(147,244.065167882),new klb.Tp(148,245.066355386),new klb.Tp(149,246.068666836),new klb.Tp(150,247.070298533),new klb.Tp(151,248.07308),new klb.Tp(152,249.074979937),new klb.Tp(153,250.078310529),new klb.Tp(154,251.08075344),new klb.Tp(155,252.084303),new klb.Tp(156,253.08688),new klb.Tp(157,254.0906)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(139,237.06207),new klb.Tp(140,238.06141),new klb.Tp(141,239.062579),new klb.Tp(142,240.062295),new klb.Tp(143,241.063716),new klb.Tp(144,242.063688713),new klb.Tp(145,243.065421),new klb.Tp(146,244.06599039),new klb.Tp(147,245.068039),new klb.Tp(148,246.068798807),new klb.Tp(149,247.070992043),new klb.Tp(150,248.07217808),new klb.Tp(151,249.074846818),new klb.Tp(152,250.076399951),new klb.Tp(153,251.079580056),new klb.Tp(154,252.081619582),new klb.Tp(155,253.085126791),new klb.Tp(156,254.087316198),new klb.Tp(157,255.091039),new klb.Tp(158,256.09344)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(141,240.06892),new klb.Tp(142,241.068662),new klb.Tp(143,242.069699),new klb.Tp(144,243.069631),new klb.Tp(145,244.070969),new klb.Tp(146,245.071317),new klb.Tp(147,246.072965),new klb.Tp(148,247.07365),new klb.Tp(149,248.075458),new klb.Tp(150,249.076405),new klb.Tp(151,250.078654),new klb.Tp(152,251.079983592),new klb.Tp(153,252.082972247),new klb.Tp(154,253.084817974),new klb.Tp(155,254.088016026),new klb.Tp(156,255.090266386),new klb.Tp(157,256.093592),new klb.Tp(158,257.095979)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(142,242.07343),new klb.Tp(143,243.07451),new klb.Tp(144,244.074077),new klb.Tp(145,245.075375),new klb.Tp(146,246.075281634),new klb.Tp(147,247.076819),new klb.Tp(148,248.077184411),new klb.Tp(149,249.079024),new klb.Tp(150,250.079514759),new klb.Tp(151,251.081566467),new klb.Tp(152,252.082460071),new klb.Tp(153,253.085176259),new klb.Tp(154,254.086847795),new klb.Tp(155,255.089955466),new klb.Tp(156,256.091766522),new klb.Tp(157,257.095098635),new klb.Tp(158,258.097069),new klb.Tp(159,259.100588)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(144,245.081017),new klb.Tp(145,246.081933),new klb.Tp(146,247.081804),new klb.Tp(147,248.082909),new klb.Tp(148,249.083002),new klb.Tp(149,250.084488),new klb.Tp(150,251.084919),new klb.Tp(151,252.08663),new klb.Tp(152,253.08728),new klb.Tp(153,254.089725),new klb.Tp(154,255.091075196),new klb.Tp(155,256.094052757),new klb.Tp(156,257.095534643),new klb.Tp(157,258.098425321),new klb.Tp(158,259.100503),new klb.Tp(159,260.103645)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(147,249.087823),new klb.Tp(148,250.087493),new klb.Tp(149,251.08896),new klb.Tp(150,252.088965909),new klb.Tp(151,253.090649),new klb.Tp(152,254.090948746),new klb.Tp(153,255.093232449),new klb.Tp(154,256.094275879),new klb.Tp(155,257.096852778),new klb.Tp(156,258.0982),new klb.Tp(157,259.101024),new klb.Tp(158,260.102636),new klb.Tp(159,261.105743),new klb.Tp(160,262.10752)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(148,251.09436),new klb.Tp(149,252.09533),new klb.Tp(150,253.095258),new klb.Tp(151,254.096587),new klb.Tp(152,255.096769),new klb.Tp(153,256.098763),new klb.Tp(154,257.099606),new klb.Tp(155,258.101883),new klb.Tp(156,259.10299),new klb.Tp(157,260.105572),new klb.Tp(158,261.106941),new klb.Tp(159,262.109692),new klb.Tp(160,263.111394)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(149,253.100679),new klb.Tp(150,254.100166),new klb.Tp(151,255.101492),new klb.Tp(152,256.101179573),new klb.Tp(153,257.103072),new klb.Tp(154,258.103568),new klb.Tp(155,259.105628),new klb.Tp(156,260.106434),new klb.Tp(157,261.108752),new klb.Tp(158,262.109918),new klb.Tp(159,263.11254),new klb.Tp(160,264.113978)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(150,255.107398),new klb.Tp(151,256.10811),new klb.Tp(152,257.107858),new klb.Tp(153,258.109438),new klb.Tp(154,259.109721),new klb.Tp(155,260.111427),new klb.Tp(156,261.112106),new klb.Tp(157,262.114153),new klb.Tp(158,263.115078),new klb.Tp(159,264.117473),new klb.Tp(160,265.118659)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(152,258.113151),new klb.Tp(153,259.114652),new klb.Tp(154,260.114435447),new klb.Tp(155,261.116199),new klb.Tp(156,262.116477),new klb.Tp(157,263.118313),new klb.Tp(158,264.118924),new klb.Tp(159,265.121066),new klb.Tp(160,266.121928)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(153,260.121803),new klb.Tp(154,261.1218),new klb.Tp(155,262.123009),new klb.Tp(156,263.123146),new klb.Tp(157,264.12473),new klb.Tp(158,265.125198),new klb.Tp(159,266.127009),new klb.Tp(160,267.12774)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(155,263.12871),new klb.Tp(156,264.128408258),new klb.Tp(157,265.130001),new klb.Tp(158,266.130042),new klb.Tp(159,267.131774),new klb.Tp(160,268.132156),new klb.Tp(161,269.134114)]),wW(iW(elb.iY,1),pob,3,0,[new klb.Tp(156,265.136567),new klb.Tp(157,266.13794),new klb.Tp(158,267.137526),new klb.Tp(159,268.138816),new klb.Tp(160,269.139106),new klb.Tp(161,270.140723),new klb.Tp(162,271.141229)])]);};klb.Wp=function Wp(a,b){klb.Vp();var c,d;d=b-a;for(c=0;c<klb.Up[a].length;c++)if(klb.Up[a][c].b==d)return klb.Up[a][c].a;return NaN;};klb.Xp=function Xp(a,b){var c;if(a.b.length!=b.b.length)return a.b.length<b.b.length?-1:1;for(c=0;c<a.b.length;c++)if(a.b[c]!=b.b[c])return a.b[c]<b.b[c]?-1:1;return 0;};klb.Yp=function Yp(a){var b,c,d,e,f,g;c=0;for(e=a.a,f=0,g=e.length;f<g;++f){d=e[f];d&&++c;}a.b=oW(elb.IX,omb,6,c,15,1);c=0;for(b=0;b<a.a.length;b++)a.a[b]&&(a.b[c++]=b);};klb.Zp=function Zp(a,b){var c;klb.Yp(a);for(c=0;c<a.b.length;c++)if(b[a.b[c]])return true;return false;};klb.$p=function $p(a){this.a=oW(elb.g1,Imb,6,a,16,1);};b2(186,1,{186:1,27:1},klb.$p);_.qb=function _p(a){return klb.Xp(this,a);};elb.jY=D3(186);klb.dq=function dq(){klb.dq=d2;klb.cq=wW(iW(elb.GX,1),mmb,6,15,[0,1.00794,4.0026,6.941,9.0122,10.811,12.011,14.007,15.999,18.998,20.18,22.99,24.305,26.982,28.086,30.974,32.066,35.453,39.948,39.098,40.078,44.956,47.867,50.942,51.996,54.938,55.845,58.933,58.693,63.546,65.39,69.723,72.61,74.922,78.96,79.904,83.8,85.468,87.62,88.906,91.224,92.906,95.94,98.906,101.07,102.91,106.42,107.87,112.41,114.82,118.71,121.76,127.6,126.9,131.29,132.91,137.33,138.91,140.12,140.91,144.24,146.92,150.36,151.96,157.25,158.93,162.5,164.93,167.26,168.93,173.04,174.97,178.49,180.95,183.84,186.21,190.23,192.22,195.08,196.97,200.59,204.38,207.2,208.98,209.98,209.99,222.02,223.02,226.03,227.03,232.04,231.04,238.03,237.05,239.05,241.06,244.06,249.08,252.08,252.08,257.1,258.1,259.1,262.11,267.12,268.13,271.13,270.13,277.15,276.15,qob,qob,283.17,285.18,rob,rob,293.2,sob,sob,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2.0141,3.016,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,71.0787,156.18828,114.10364,115.0877,103.1447,128.13052,129.11458,57.05182,137.14158,113.15934,113.15934,128.17428,131.19846,147.17646,97.11658,87.0777,101.10458,186.2134,163.17546,99.13246]);klb.aq=wW(iW(elb.GX,1),mmb,6,15,[0,1.007825,4.0026,7.016003,9.012182,11.009305,12,14.003074,15.994915,18.998403,19.992435,22.989767,23.985042,26.98153,27.976927,30.973762,31.97207,34.968852,39.962384,38.963707,39.962591,44.95591,47.947947,50.943962,51.940509,54.938047,55.934939,58.933198,57.935346,62.939598,63.929145,68.92558,73.921177,74.921594,79.91652,78.918336,83.911507,84.911794,87.905619,88.905849,89.904703,92.906377,97.905406,89.92381,101.904348,102.9055,105.903478,106.905092,113.903357,114.90388,119.9022,120.903821,129.906229,126.904473,131.904144,132.905429,137.905232,138.906346,139.905433,140.907647,141.907719,135.92398,151.919729,152.921225,157.924099,158.925342,163.929171,164.930319,165.93029,168.934212,173.938859,174.94077,179.946545,180.947992,183.950928,186.955744,191.961467,192.962917,194.964766,196.966543,201.970617,204.974401,207.976627,208.980374,193.98818,195.99573,199.9957,201.00411,206.0038,210.00923,232.038054,216.01896,238.050784,229.03623,232.041169,237.05005,238.05302,242.06194,240.06228,243.06947,243.07446,248.08275,251.08887,253.09515,257.10295,257.10777,271.13,270.13,277.15,276.15,qob,qob,283.17,285.18,rob,rob,291.2,sob,sob,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2.014,3.01605,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]);klb.bq=wW(iW(elb.IX,1),omb,6,15,[6,1,7,8]);};klb.eq=function eq(a){var b,c;c=a.a;for(b=0;b<a.b.length;b++)c+=a.b[b]*klb.aq[a.c[b]];return c;};klb.fq=function fq(a){var b,c;b=new flb.G5();for(c=0;c<a.b.length;c++){flb.F5(b,(klb.wj(),klb.qj)[a.c[c]]);a.b[c]>1&&flb.E5(b,a.b[c]);}return b.a;};klb.gq=function gq(a){var b,c;c=a.d;for(b=0;b<a.b.length;b++)c+=a.b[b]*klb.cq[a.c[b]];return c;};klb.hq=function hq(a){var b,c,d,e,f,g,h,i,j,k,l;klb.Jt(a,1);e=oW(elb.IX,omb,6,191,15,1);for(c=0;c<a.q;c++){switch(a.C[c]){case 171:e[1]+=5;e[6]+=3;e[7]+=1;e[8]+=1;break;case 172:e[1]+=12;e[6]+=6;e[7]+=4;e[8]+=1;break;case 173:e[1]+=6;e[6]+=4;e[7]+=2;e[8]+=2;break;case 174:e[1]+=5;e[6]+=4;e[7]+=1;e[8]+=3;break;case 175:e[1]+=5;e[6]+=3;e[7]+=1;e[8]+=1;e[16]+=1;break;case 176:e[1]+=8;e[6]+=5;e[7]+=2;e[8]+=2;break;case 177:e[1]+=7;e[6]+=5;e[7]+=1;e[8]+=3;break;case 178:e[1]+=3;e[6]+=2;e[7]+=1;e[8]+=1;break;case 179:e[1]+=7;e[6]+=6;e[7]+=3;e[8]+=1;break;case 181:case 180:e[1]+=11;e[6]+=6;e[7]+=1;e[8]+=1;break;case 182:e[1]+=12;e[6]+=6;e[7]+=2;e[8]+=1;break;case 183:e[1]+=9;e[6]+=5;e[7]+=1;e[8]+=1;e[16]+=1;break;case 184:e[1]+=9;e[6]+=9;e[7]+=1;e[8]+=1;break;case 185:e[1]+=7;e[6]+=5;e[7]+=1;e[8]+=1;break;case 186:e[1]+=5;e[6]+=3;e[7]+=1;e[8]+=2;break;case 187:e[1]+=7;e[6]+=4;e[7]+=1;e[8]+=2;break;case 188:e[1]+=10;e[6]+=11;e[7]+=2;e[8]+=1;break;case 189:e[1]+=9;e[6]+=9;e[7]+=1;e[8]+=2;break;case 190:e[1]+=9;e[6]+=5;e[7]+=1;e[8]+=1;break;case 1:switch(a.A[c]){case 0:case 1:++e[1];break;case 2:++e[151];break;case 3:++e[152];}break;default:++e[a.C[c]];}}for(d=0;d<a.q;d++)a.C[d]>=171&&a.C[d]<=190?e[1]+=2-klb.Sn(a,d):e[1]+=klb.Mn(a,d);h=0;for(j=1;j<=190;j++)e[j]!=0&&++h;this.b=oW(elb.IX,omb,6,h,15,1);this.c=oW(elb.IX,omb,6,h,15,1);h=0;for(i=0;i<klb.bq.length;i++){if(e[klb.bq[i]]!=0){this.b[h]=e[klb.bq[i]];this.c[h]=klb.bq[i];++h;e[klb.bq[i]]=0;}}while(true){l='zzz';k=-1;for(g=1;g<=190;g++)if(e[g]>0&&flb.e5(l,(klb.wj(),klb.qj)[g])>0){l=(klb.wj(),klb.qj)[g];k=g;}if(k==-1)break;this.b[h]=e[k];this.c[h]=k;++h;e[k]=0;}this.a=0;this.d=0;for(b=0;b<a.f;b++){if(a.C[b]!=1&&a.A[b]!=0){g=a.C[b];f=a.A[b];this.a+=klb.Wp(g,f)-klb.aq[g];this.d+=klb.Wp(g,f)-klb.cq[g];}}};b2(119,1,{119:1});_.ib=function iq(a){var b;if(a===this)return true;if(!uX(a,119))return false;for(b=0;b<this.b.length;b++)if(this.b[b]!=a.b[b])return false;return true;};_.a=0;_.d=0;elb.kY=D3(119);klb.jq=function jq(a){switch(a){case 6:return 1;case 53:return 2;case 33:return 3;case 34:return 4;case 35:return 5;case 15:return 6;case 16:return 7;case 17:return 8;case 7:return 9;case 8:return 10;case 9:return 11;}return 0;};klb.kq=function kq(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C;klb.Jt(a,7);B=0;n=oW(elb.g1,Imb,6,a.f,16,1);C=oW(elb.IX,omb,6,a.f,15,1);for(f=0;f<a.g;f++){c=a.D[0][f];d=a.D[1][f];g=a.s[c];h=a.s[d];if(g!=0&&h!=0&&g<0^h<0){n[c]=true;n[d]=true;}C[c]+=klb.Hk(a,f);C[d]+=klb.Hk(a,f);}for(i=0;i<a.f;i++){B+=a.s[i];if(a.s[i]==1){if(a.C[i]==7){if(!n[i]){if(C[i]<=3){B-=1;a.s[i]=0;a.T=0;if(a.j[i]!=a.e[i]){klb.Uj(a,klb.wn(a,i,a.e[i]-1));klb.Jt(a,7);}}else if(i<a.f&&klb.sr(a.p,i)){r=oW(elb.g1,Imb,6,a.f,16,1);s=oW(elb.g1,Imb,6,a.g,16,1);klb.en(a,i,true,r,s);for(k=0;k<a.f;k++){if(r[k]&&a.C[k]==7&&a.s[k]==0&&C[k]==2){if(klb.nq(a,s,i,k)){B-=1;break;}}}}}}}else if(a.s[i]<0){if(a.C[i]==6||a.C[i]==7||a.C[i]==8||a.C[i]==16){if(n[i]){v=oW(elb.IX,omb,6,16,15,1);v[0]=i;A=1;w=0;while(w<A){for(e=0;e<a.r;e++){b=-1;a.D[0][e]==v[w]?b=a.D[1][e]:a.D[1][e]==v[w]&&(b=a.D[0][e]);if(b==-1)continue;if(a.s[b]!=0){l=false;for(p=0;p<A;p++){if(b==v[p]){l=true;break;}}if(!l){if(A==v.length){j=oW(elb.IX,omb,6,2*v.length,15,1);flb.W5(v,0,j,0,v.length);v=j;}v[A++]=b;}}}++w;}m=0;for(q=0;q<A;q++){m+=klb.dk(a,v[q]);}if(m<0){t=-1;u=99;for(o=0;o<A;o++){if(klb.dk(a,v[o])<0){if(u>klb.jq(klb.uk(a,v[o]))){u=klb.jq(klb.uk(a,v[o]));t=v[o];}}}if(t!=-1){B-=a.s[t];a.s[t]=0;a.T=0;}}}else {B-=a.s[i];a.s[i]=0;a.T=0;}}}}B>0&&(B=klb.lq(a,B));return B;};klb.lq=function lq(a,b){var c,d,e,f;b>0&&(b=klb.oq(a,b,9));b>0&&(b=klb.oq(a,b,17));b>0&&(b=klb.oq(a,b,35));b>0&&(b=klb.oq(a,b,53));if(b>0){for(c=0;c<a.f;c++){if(a.s[c]>0){e=false;for(f=0;f<a.j[c];f++){d=a.i[c][f];if(a.s[d]==0&&klb.hl(a,d)&&klb.Mn(a,d)>0){a.s[d]=-1;a.T=0;--b;if(b==0)return 0;e=true;break;}}if(e)continue;}}}b>0&&(b=klb.mq(a,b,8,16));b>0&&(b=klb.mq(a,b,8,15));b>0&&(b=klb.mq(a,b,8,6));b>0&&(b=klb.mq(a,b,7,16));return b;};klb.mq=function mq(a,b,c,d){var e,f,g,h,i,j,k,l;for(e=0;e<a.f;e++){if(a.C[e]==c&&a.s[e]==0&&klb.Mn(a,e)>0){h=false;for(j=0;j<a.j[e];j++){if(a.n[e][j]==1){f=a.i[e][j];if(a.C[f]==d){l=false;i=false;for(k=0;k<a.j[f];k++){g=a.i[f][k];if(a.s[g]<0){i=true;break;}g!=e&&a.C[g]==8&&a.n[f][k]==2&&(l=true);}if(!i&&l){a.s[e]=-1;a.T=0;--b;if(b==0)return 0;h=true;}}}if(h)break;}}}return b;};klb.nq=function nq(a,b,c,d){var e,f,g,h,i,j,k,l;l=oW(elb.IX,omb,6,a.g,15,1);i=oW(elb.g1,Imb,6,a.g,16,1);for(g=0;g<a.g;g++){l[g]=a.H[g];if(b[g]){i[g]=true;a.H[g]=1;a.T=0;}}for(e=0;e<a.f;e++){if(e!=d&&a.o[e]==0&&i[a.k[d][0]]){for(k=0;k<a.j[e];k++){h=a.k[e][k];a.H[h]=1;a.T=0;i[h]=false;}}}for(j=0;j<a.j[c];j++){h=a.k[c][j];a.H[h]=1;a.T=0;i[h]=false;}if(klb.ce(new klb.me(a),i,false)){a.s[c]=0;a.T=0;a.j[d]!=a.e[d]&&klb.Uj(a,klb.wn(a,d,a.e[d]-1));klb.Jt(a,7);return true;}for(f=0;f<a.g;f++){klb.km(a,f,l[f]);}return false;};klb.oq=function oq(a,b,c){var d;for(d=0;d<a.f;d++){if(a.C[d]==c&&a.s[d]==0&&a.j[d]==0){a.s[d]=-1;a.T=0;--b;if(b==0)return 0;}}return b;};klb.pq=function pq(a,b){var c,d;d=Llb.w8(a.b,b);for(c=ilb.Xgb(d).length;c<10;c++)flb.J5(a.a,32);flb.N5(a.a,d);};klb.qq=function qq(a,b){var c,d,e;if(b<0||b>999){flb.N5(a.a,'  ?');return;}c=false;for(d=0;d<3;d++){e=b/100|0;if(e==0){d==2||c?flb.J5(a.a,48):flb.J5(a.a,32);}else {flb.J5(a.a,48+e&fmb);c=true;}b=10*(b%100);}};klb.rq=function rq(a,b){Ilb.E2(b,a.a.a);};klb.sq=function sq(a){klb.tq.call(this,a);};klb.tq=function tq(a){klb.vq.call(this,a,new flb.R5());};klb.uq=function uq(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N,O,P,Q,R,S,T,U,V,W,X,Y,Z;this.b=new Llb.z8((N=tob,N));klb.Jt(a,15);L=true;for(e=0;e<a.f;e++){if((a.u[e]&3)!=0&&(a.u[e]&3)!=3&&(a.u[e]&znb)>>18!=1){L=false;break;}}O=-1;if(L){F=oW(elb.IX,omb,6,32,15,1);P=0;for(f=0;f<a.f;f++){if((a.u[f]&3)!=0&&(a.u[f]&3)!=3&&(a.u[f]&znb)>>18==1){H=(a.u[f]&znb)>>18!=1&&(a.u[f]&znb)>>18!=2?-1:(a.u[f]&$nb)>>20;++F[H];if(P<F[H]){P=F[H];O=H;}break;}}}this.a=!c?new flb.Q5():c;R=a.P!=null?a.P:'';flb.N5(this.a,R+Slb);flb.N5(this.a,'Actelion Java MolfileCreator 1.0\n\n');klb.qq(this,a.q);klb.qq(this,a.r);flb.N5(this.a,'  0  0');klb.qq(this,L?0:1);flb.N5(this.a,'  0  0  0  0  0999 V2000\n');I=a.q==1;for(i=1;i<a.q;i++){if(a.J[i].a!=a.J[0].a||a.J[i].b!=a.J[0].b||a.J[i].c!=a.J[0].c){I=true;break;}}G=1;if(I){if(b!=0){G=b;}else {s=klb.wk(a,a.q,a.r,(klb.wj(),klb.vj));if(s!=0){(s<1||s>3)&&(G=1.5/s);}else {Q=aob;for(f=1;f<a.q;f++){for(h=0;h<f;h++){B=a.J[h].a-a.J[f].a;C=a.J[h].b-a.J[f].b;D=a.J[h].c-a.J[f].c;A=B*B+C*C+D*D;Q>A&&(Q=A);}}G=3/Q;}}}for(j=0;j<a.q;j++){if(I){klb.pq(this,G*a.J[j].a);klb.pq(this,G*-a.J[j].b);klb.pq(this,G*-a.J[j].c);}else {flb.N5(this.a,'    0.0000    0.0000    0.0000');}if((a.v==null?null:a.v[j])!=null)flb.N5(this.a,' L  ');else if(I1(n1(a.B[j],1),0))flb.N5(this.a,' A  ');else if(a.C[j]>=129&&a.C[j]<=144||a.C[j]==154)flb.N5(this.a,' R# ');else {p=(klb.wj(),klb.qj)[a.C[j]];flb.N5(this.a,' '+p);ilb.Xgb(p).length==1?flb.N5(this.a,'  '):ilb.Xgb(p).length==2&&flb.N5(this.a,' ');}flb.N5(this.a,' 0  0  0');J=n1(Umb,a.B[j]);s1(J,0)==0?flb.N5(this.a,'  0'):s1(J,384)==0?flb.N5(this.a,'  3'):s1(J,128)==0?flb.N5(this.a,'  2'):s1(J,1792)==0?flb.N5(this.a,'  1'):s1(J,1664)==0&&flb.N5(this.a,'  2');flb.N5(this.a,I1(n1(a.B[j],snb),0)?'  1':'  0');Z=((a.u[j]&Znb)>>>27)-1;Z==-1?flb.N5(this.a,'  0'):Z==0?flb.N5(this.a,' 15'):klb.qq(this,Z);flb.N5(this.a,'  0  0  0');klb.qq(this,$wnd.Math.abs(a.w[j]));flb.N5(this.a,'  0  0\n');}for(t=0;t<a.r;t++){switch(a.H[t]){case 1:T=1;W=0;break;case 2:T=2;W=0;break;case 4:T=3;W=0;break;case 129:T=1;W=6;break;case 257:T=1;W=1;break;case 386:T=2;W=3;break;case 64:T=4;W=0;break;case 32:T=8;W=0;break;default:T=1;W=0;}if(L&&(W==1||W==6)){f=a.D[0][t];(a.u[f]&znb)>>18==2?W=0:(a.u[f]&znb)>>18==1&&((a.u[f]&znb)>>18!=1&&(a.u[f]&znb)>>18!=2?-1:(a.u[f]&$nb)>>20)!=O&&(W=4);}u=a.G[t]&31;u!=0&&(u==8?T=4:u==3?T=5:u==9?T=6:u==10?T=7:T=8);V=a.G[t]&384;Y=V==0?0:V==256?1:2;klb.qq(this,1+a.D[0][t]);klb.qq(this,1+a.D[1][t]);klb.qq(this,T);klb.qq(this,W);flb.N5(this.a,'  0');klb.qq(this,Y);flb.N5(this.a,'  0\n');}S=0;for(k=0;k<a.q;k++)a.s[k]!=0&&++S;if(S!=0){w=0;for(f=0;f<a.q;f++){if(a.s[f]!=0){if(w==0){flb.N5(this.a,'M  CHG');klb.qq(this,$wnd.Math.min(8,S));}flb.N5(this.a,' ');klb.qq(this,f+1);v=a.s[f];if(v<0){flb.N5(this.a,'  -');v=-v;}else flb.N5(this.a,'   ');flb.J5(this.a,48+v&fmb);--S;if(++w==8||S==0){w=0;flb.N5(this.a,Slb);}}}}S=0;for(l=0;l<a.q;l++)a.A[l]==0||++S;if(S!=0){w=0;for(f=0;f<a.q;f++){if(a.A[f]!=0){if(w==0){flb.N5(this.a,'M  ISO');klb.qq(this,$wnd.Math.min(8,S));}flb.N5(this.a,' ');klb.qq(this,f+1);flb.N5(this.a,' ');klb.qq(this,a.A[f]);--S;if(++w==8||S==0){w=0;flb.N5(this.a,Slb);}}}}S=0;for(m=0;m<a.q;m++)(a.u[m]&48)!=0&&++S;if(S!=0){w=0;for(f=0;f<a.q;f++){if((a.u[f]&48)!=0){if(w==0){flb.N5(this.a,'M  RAD');klb.qq(this,$wnd.Math.min(8,S));}flb.N5(this.a,' ');klb.qq(this,f+1);switch(a.u[f]&48){case 16:flb.N5(this.a,'   1');break;case 32:flb.N5(this.a,'   2');break;case 48:flb.N5(this.a,'   3');}--S;if(++w==8||S==0){w=0;flb.N5(this.a,Slb);}}}}S=0;for(n=0;n<a.q;n++)(a.C[n]>=129&&a.C[n]<=144||a.C[n]==154)&&++S;if(S!=0){w=0;for(d=0;d<a.q;d++){r=a.C[d];if(r>=129&&r<=144||r==154){if(w==0){flb.N5(this.a,'M  RGP');klb.qq(this,$wnd.Math.min(8,S));}flb.N5(this.a,' ');klb.qq(this,d+1);flb.N5(this.a,' ');klb.qq(this,r==154?0:r>=142?r-141:r-125);--S;if(++w==8||S==0){w=0;flb.N5(this.a,Slb);}}}}if(a.K){S=0;for(f=0;f<a.q;f++)I1(n1(a.B[f],120),0)&&++S;if(S!=0){w=0;for(h=0;h<a.q;h++){U=n1(a.B[h],120);if(s1(U,0)!=0){if(w==0){flb.N5(this.a,'M  RBC');klb.qq(this,$wnd.Math.min(8,S));}flb.N5(this.a,' ');klb.qq(this,h+1);s1(U,112)==0?flb.N5(this.a,'  -1'):s1(U,8)==0?flb.N5(this.a,'   1'):s1(U,104)==0?flb.N5(this.a,'   2'):s1(U,88)==0?flb.N5(this.a,'   3'):s1(U,56)==0&&flb.N5(this.a,'   4');--S;if(++w==8||S==0){w=0;flb.N5(this.a,Slb);}}}}for(o=0;o<a.q;o++){q=a.v==null?null:a.v[o];if(q!=null){flb.N5(this.a,'M  ALS ');klb.qq(this,o+1);klb.qq(this,q.length);flb.N5(this.a,I1(n1(a.B[o],1),0)?' T ':' F ');for(K=0;K<q.length;K++){M=(klb.wj(),klb.qj)[q[K]];switch(ilb.Xgb(M).length){case 1:flb.N5(this.a,M+'   ');break;case 2:flb.N5(this.a,M+'  ');break;case 3:flb.N5(this.a,M+' ');break;default:flb.N5(this.a,'   ?');}}flb.N5(this.a,Slb);}}S=0;for(g=0;g<a.q;g++)I1(n1(a.B[g],6144),0)&&++S;if(S!=0){w=0;for(d=0;d<a.q;d++){X=n1(a.B[d],6144);if(s1(X,0)!=0){if(w==0){flb.N5(this.a,'M  SUB');klb.qq(this,$wnd.Math.min(8,S));}flb.N5(this.a,' ');klb.qq(this,d+1);I1(n1(X,Qmb),0)?flb.N5(this.a,'   '+(a.e[d]+1)):flb.N5(this.a,'  -2');--S;if(++w==8||S==0){w=0;flb.N5(this.a,Slb);}}}}}flb.N5(this.a,uob);};klb.vq=function vq(a,b){klb.uq.call(this,a,0,b);};b2(102,1,{},klb.sq,klb.uq);elb.mY=D3(102);klb.wq=function wq(a,b,c,d,e,f){var g,h,i,j;j=1;h=false;switch(e){case 1:j=257;break;case 3:j=386;break;case 4:j=257;h=true;break;case 6:j=129;break;default:switch(d){case 1:j=1;break;case 2:j=2;break;case 3:j=4;break;case 4:j=64;break;case 9:j=32;}}g=klb.Aj(a.j,b,c,j);i=0;h&&klb.Ol(a.j,b,1,-1);if(d>4){switch(d){case 5:i|=3;break;case 6:i|=9;break;case 7:i|=10;break;case 8:j!=32&&(i|=31);}}f==1&&(i|=256);f==2&&(i|=128);i!=0&&klb.jm(a.j,g,i,true);return g;};klb.xq=function xq(a){var b,c,d,e,f,g,h,i,j,k,l;g=oW(elb.IX,omb,6,a.j.q,15,1);for(c=0;c<a.j.q;c++)g[c]=-klb.dk(a.j,c);klb.ce(new klb.me(a.j),null,true);for(d=0;d<a.j.q;d++)g[d]+=klb.dk(a.j,d);for(b=0;b<a.j.q;b++){if(g[b]!=0){h=-g[b];for(e=0;e<a.j.r;e++){for(j=0;j<2;j++){if(h>0&&klb.Kk(a.j,e)==32&&klb.zk(a.j,1-j,e)==b){l=klb.zk(a.j,j,e);if(klb.kl(a.j,l)){k=klb.zq(a,l);f=klb.dk(a.j,l);if(f<k){i=$wnd.Math.min(h,k-f);klb.Jl(a.j,l,f+i);h-=i;}}}}}}}};klb.yq=function yq(a,b){a.j=null;return klb.Qq(a,new Ilb.z2(new Ilb.D2(b)))?a.j:null;};klb.zq=function zq(a,b){var c,d;c=klb.uk(a.j,b);d=c<(klb.wj(),klb.sj).length?klb.sj[c]:null;return d==null?0:d[d.length-1];};klb.Aq=function Aq(a,b){var c;c=!a.b?null:llb.Qab(a.b,new flb.o4(b));return !c?b-1:c.a;};klb.Bq=function Bq(a,b){var c;c=!a.c?null:llb.Qab(a.c,new flb.o4(b));return !c?b-1:c.a;};klb.Cq=function Cq(a,b){var c,d;if(b!=null){klb.Jt(a.j,1);for(c=0;c<a.j.f;c++){if(b[c]!=0){d=klb.Ok(a.j,c,klb.Sn(a.j,c));b[c]==15?d>=0&&klb.Hl(a.j,c,0):b[c]!=klb.Qk(a.j,c)&&klb.Hl(a.j,c,b[c]-d);}}}};klb.Dq=function Dq(a,b){var c;if(b==-1){return -1;}for(c=b+1;c<ilb.Xgb(a).length;c++){ilb.Sgb(c,ilb.Xgb(a).length);if(ilb.Xgb(a).charCodeAt(c)!=32&&(ilb.Sgb(c,ilb.Xgb(a).length),ilb.Xgb(a).charCodeAt(c)!=9)){return c;}}return -1;};klb.Eq=function Eq(a,b){var c;for(c=b;c<ilb.Xgb(a).length;c++){ilb.Sgb(c,ilb.Xgb(a).length);if(ilb.Xgb(a).charCodeAt(c)==32||(ilb.Sgb(c,ilb.Xgb(a).length),ilb.Xgb(a).charCodeAt(c)==9)){return c;}}return -1;};klb.Fq=function Fq(a){if(ilb.Xgb(a).indexOf('ATOMS=(')!=-1)return vob;if(ilb.Xgb(a).indexOf('BONDS=(')!=-1)return 'BONDS';return null;};klb.Gq=function Gq(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F;p=0;q=(h=klb.Eq(b,p+1),h==-1?ilb.Xgb(b).length:h);d=flb.W3((ilb.Rgb(p,q,ilb.Xgb(b).length),ilb.Xgb(b).substr(p,q-p)),wob,Ylb);p=klb.Dq(b,q);q=(i=klb.Eq(b,p+1),i==-1?ilb.Xgb(b).length:i);s=(ilb.Rgb(p,q,ilb.Xgb(b).length),ilb.Xgb(b).substr(p,q-p));A=null;e=false;r=klb.Nq(b);if(r!=0){A=klb.Hq(b);r<0&&(e=true);q=$wnd.Math.abs(r);}p=klb.Dq(b,q);q=(j=klb.Eq(b,p+1),j==-1?ilb.Xgb(b).length:j);C=h4((ilb.Rgb(p,q,ilb.Xgb(b).length),ilb.Xgb(b).substr(p,q-p)));p=klb.Dq(b,q);q=(k=klb.Eq(b,p+1),k==-1?ilb.Xgb(b).length:k);D=h4((ilb.Rgb(p,q,ilb.Xgb(b).length),ilb.Xgb(b).substr(p,q-p)));p=klb.Dq(b,q);q=(l=klb.Eq(b,p+1),l==-1?ilb.Xgb(b).length:l);F=h4((ilb.Rgb(p,q,ilb.Xgb(b).length),ilb.Xgb(b).substr(p,q-p)));p=klb.Dq(b,q);q=(m=klb.Eq(b,p+1),m==-1?ilb.Xgb(b).length:m);u=flb.W3((ilb.Rgb(p,q,ilb.Xgb(b).length),ilb.Xgb(b).substr(p,q-p)),wob,Ylb);c=klb.xj(a.j,C,-D,-F);c+1!=d&&(!a.b&&(a.b=new llb.Aeb()),llb.seb(a.b,new flb.o4(d),new flb.o4(c)));if(A!=null){klb.am(a.j,c,6);klb.Ql(a.j,c,A,e);}u!=0&&klb.Rl(a.j,c,u,false);if(flb.j5(s,'A')||flb.j5(s,'*')){klb.Vl(a.j,c,1,true);}else if(flb.j5(s,'Q')){t=oW(elb.IX,omb,6,1,15,1);t[0]=6;klb.Ql(a.j,c,t,true);}else {klb.am(a.j,c,klb.Om(s,67));}while((p=klb.Dq(b,q))!=-1){q=(g=klb.Eq(b,p+1),g==-1?ilb.Xgb(b).length:g);v=(ilb.Rgb(p,q,ilb.Xgb(b).length),ilb.Xgb(b).substr(p,q-p));o=flb.n5(v,z5(61));n=(ilb.Rgb(0,o,ilb.Xgb(v).length),ilb.Xgb(v).substr(0,o));B=flb.W3((ilb.Sgb(o+1,ilb.Xgb(v).length+1),ilb.Xgb(v).substr(o+1)),wob,Ylb);if(flb.j5(n,'CHG')){klb.Jl(a.j,c,B);}else if(flb.j5(n,'RAD')){switch(B){case 1:klb.Wl(a.j,c,16);break;case 2:klb.Wl(a.j,c,32);break;case 3:klb.Wl(a.j,c,48);}}else if(flb.j5(n,'CFG'));else if(flb.j5(n,'MASS')){klb.Tl(a.j,c,B);}else if(flb.j5(n,'VAL')){klb.Hl(a.j,c,B==-1?0:B==0?-1:B);}else if(flb.j5(n,'HCOUNT')){switch(B){case 0:break;case-1:klb.Vl(a.j,c,1792,true);break;case 1:klb.Vl(a.j,c,128,true);break;case 2:klb.Vl(a.j,c,384,true);break;default:klb.Vl(a.j,c,896,true);}}else if(flb.j5(n,'SUBST')){if(B==-1){klb.Vl(a.j,c,tnb,true);}else if(B>0){w=0;for(f=0;f<a.j.r;f++){(klb.zk(a.j,0,f)==c||klb.zk(a.j,1,f)==c)&&++w;}B>w&&klb.Vl(a.j,c,Qmb,true);}}else if(flb.j5(n,'RBCNT')){switch(B){case 3:case-1:klb.Vl(a.j,c,112,true);break;case 1:klb.Vl(a.j,c,8,true);break;case 2:klb.Vl(a.j,c,104,true);break;case 4:klb.Vl(a.j,c,56,true);}}}};klb.Hq=function Hq(a){var b,c,d,e,f,g,h,i;h=null;c=ilb.Xgb(a).indexOf('[');d=ilb.Xgb(a).indexOf(']',c);if(c>=0&&d>0){b=oW(elb.IX,omb,6,16,15,1);i=(ilb.Rgb(c+1,d,ilb.Xgb(a).length),ilb.Xgb(a).substr(c+1,d-(c+1)));e=0;g=true;while(g&&e<16){c=ilb.Xgb(i).indexOf(',');f=null;if(c==-1){f=i;g=false;}else {f=(ilb.Rgb(0,c,ilb.Xgb(i).length),ilb.Xgb(i).substr(0,c));i=(ilb.Sgb(c+1,ilb.Xgb(i).length+1),ilb.Xgb(i).substr(c+1));}b[e++]=klb.Om(f,1);}h=oW(elb.IX,omb,6,e,15,1);flb.W5(b,0,h,0,e);}return h;};klb.Iq=function Iq(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t;o=0;p=(i=klb.Eq(b,o+1),i==-1?ilb.Xgb(b).length:i);f=flb.W3((ilb.Rgb(o,p,ilb.Xgb(b).length),ilb.Xgb(b).substr(o,p-o)),wob,Ylb);o=klb.Dq(b,p);p=(j=klb.Eq(b,o+1),j==-1?ilb.Xgb(b).length:j);g=flb.W3((ilb.Rgb(o,p,ilb.Xgb(b).length),ilb.Xgb(b).substr(o,p-o)),wob,Ylb);o=klb.Dq(b,p);p=(k=klb.Eq(b,o+1),k==-1?ilb.Xgb(b).length:k);c=klb.Aq(a,flb.W3((ilb.Rgb(o,p,ilb.Xgb(b).length),ilb.Xgb(b).substr(o,p-o)),wob,Ylb));o=klb.Dq(b,p);p=(l=klb.Eq(b,o+1),l==-1?ilb.Xgb(b).length:l);d=klb.Aq(a,flb.W3((ilb.Rgb(o,p,ilb.Xgb(b).length),ilb.Xgb(b).substr(o,p-o)),wob,Ylb));r=0;s=0;while((o=klb.Dq(b,p))!=-1){p=(h=klb.Eq(b,o+1),h==-1?ilb.Xgb(b).length:h);q=(ilb.Rgb(o,p,ilb.Xgb(b).length),ilb.Xgb(b).substr(o,p-o));n=flb.n5(q,z5(61));m=(ilb.Rgb(0,n,ilb.Xgb(q).length),ilb.Xgb(q).substr(0,n));t=flb.W3((ilb.Sgb(n+1,ilb.Xgb(q).length+1),ilb.Xgb(q).substr(n+1)),wob,Ylb);if(flb.j5(m,'CFG')){switch(t){case 1:r=1;break;case 2:r=g==2?3:4;break;case 3:r=6;}}else flb.j5(m,'TOPO')?s=t:undefined;}e=klb.wq(a,c,d,g,r,s);e+1!=f&&(!a.c&&(a.c=new llb.Aeb()),llb.seb(a.c,new flb.o4(f),new flb.o4(e)));};klb.Jq=function Jq(a,b){var c,d,e,f,g,h;h=klb.Fq(b);if(h!=null){g=klb.Lq(b,h);if(flb.j5(ilb.Xgb(b).substr(0,13),'MDLV30/STEABS')){if(flb.j5(h,vob))for(f=0;f<g.length;f++)klb.Ol(a.j,klb.Aq(a,g[f]),0,-1);else for(e=0;e<g.length;e++)klb.em(a.j,klb.Bq(a,g[e]),0,-1);}else if(flb.j5(ilb.Xgb(b).substr(0,13),'MDLV30/STERAC')){d=flb.W3(flb.w5(b,13,klb.Eq(b,13)),wob,Ylb);if(flb.j5(h,vob))for(f=0;f<g.length;f++)klb.Ol(a.j,klb.Aq(a,g[f]),1,d-1);else for(e=0;e<g.length;e++)klb.em(a.j,klb.Bq(a,g[e]),1,d-1);}else if(flb.j5(ilb.Xgb(b).substr(0,13),'MDLV30/STEREL')){d=flb.W3(flb.w5(b,13,klb.Eq(b,13)),wob,Ylb);if(flb.j5(h,vob))for(f=0;f<g.length;f++)klb.Ol(a.j,klb.Aq(a,g[f]),2,d-1);else for(e=0;e<g.length;e++)klb.em(a.j,klb.Bq(a,g[e]),2,d-1);}else if(flb.j5(ilb.Xgb(b).substr(0,13),'MDLV30/HILITE')){if(flb.j5(h,vob)){for(e=0;e<g.length;e++)klb.Kl(a.j,klb.Aq(a,g[e]),448);}else {for(e=0;e<g.length;e++){c=klb.Bq(a,g[e]);klb.Kl(a.j,klb.zk(a.j,0,c),448);klb.Kl(a.j,klb.zk(a.j,1,c),448);}}}}};klb.Kq=function Kq(a,b){var c,d,e;if(!a.j){if(flb.j5(ilb.Xgb(b).substr(0,6),'COUNTS')){c=klb.Dq(b,klb.Eq(b,7));d=flb.W3(flb.w5(b,7,klb.Eq(b,7)),wob,Ylb);e=flb.W3(flb.w5(b,c,klb.Eq(b,c)),wob,Ylb);a.j=new klb.au(d,e);}}};klb.Lq=function Lq(a,b){var c,d,e,f,g,h;f=ilb.Xgb(a).indexOf(b+'=(')+ilb.Xgb(b).length+2;g=flb.o5(a,z5(41),f);e=klb.Eq(a,f);c=flb.W3((ilb.Rgb(f,e,ilb.Xgb(a).length),ilb.Xgb(a).substr(f,e-f)),wob,Ylb);h=oW(elb.IX,omb,6,c,15,1);for(d=0;d<c;d++){f=klb.Dq(a,e);e=klb.Eq(a,f);(e==-1||e>g)&&(e=g);h[d]=flb.W3((ilb.Rgb(f,e,ilb.Xgb(a).length),ilb.Xgb(a).substr(f,e-f)),wob,Ylb);}return h;};klb.Mq=function Mq(a){var b,c,d,e,f,g,h,i,j;i=oW(elb.IX,omb,6,a.j.q,15,1);for(d=0;d<a.j.r;d++)if(klb.Kk(a.j,d)==64)for(g=0;g<2;g++)i[klb.zk(a.j,g,d)]=1;for(e=0;e<a.j.r;e++){j=klb.Hk(a.j,e);for(f=0;f<2;f++)i[klb.zk(a.j,f,e)]+=j;}for(c=0;c<a.j.r;c++){if(klb.Hk(a.j,c)==1){for(f=0;f<2;f++){h=klb.zk(a.j,1-f,c);if(klb.kl(a.j,h)){b=klb.zk(a.j,f,c);if(klb.hl(a.j,b)&&i[b]>klb.Qk(a.j,b)){klb.km(a.j,c,32);continue;}}}}}};klb.Nq=function Nq(a){var b,c;if(ilb.Xgb(a).indexOf('[')>=0){b=ilb.Xgb(a).indexOf(' NOT[');c=ilb.Xgb(a).indexOf(']',b);if(b>=0&&c>0){return -(c+1);}else {b=ilb.Xgb(a).indexOf(' [');c=ilb.Xgb(a).indexOf(']',b);if(b>=0&&c>0){return c+1;}}b=ilb.Xgb(a).indexOf(" 'NOT[");c=ilb.Xgb(a).indexOf("]'",b);if(b>=0&&c>0){return -(c+2);}else {b=ilb.Xgb(a).indexOf(" '[");c=ilb.Xgb(a).indexOf("]'",b);if(b>=0&&c>0){return c+2;}}flb.V5();}return 0;};klb.Oq=function Oq(a,b,c){a.j=b;return klb.Qq(a,c);};klb.Pq=function Pq(a){return ilb.Xgb(a).length==0?0:flb.W3(a,wob,Ylb);};klb.Qq=function Qq(b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N,O,P,Q,R,S,T,U,V,W,X;T=null;try{b.f=null;if(b.j){klb.Nj(b.j);klb.mm(b.j,false);}D=v=Ilb.y2(c);if(null==D){return false;}if(null==(v=Ilb.y2(c))){return false;}if(null==(v=Ilb.y2(c))){return false;}b.k=ilb.Xgb(v).indexOf("From CSD data. Using bond type 'Any'")!=-1;b.e=ilb.Xgb(v).indexOf('From CSD data.')!=-1;if(null==(v=Ilb.y2(c))){return false;}b.g=false;b.d=b.a;try{F=flb.W3(flb.x5((ilb.Rgb(0,3,ilb.Xgb(v).length),ilb.Xgb(v).substr(0,3))),wob,Ylb);G=flb.W3(flb.x5((ilb.Rgb(3,6,ilb.Xgb(v).length),ilb.Xgb(v).substr(3,3))),wob,Ylb);H=klb.Pq(flb.x5((ilb.Rgb(6,9,ilb.Xgb(v).length),ilb.Xgb(v).substr(6,3))));b.d=b.d|1==klb.Pq(flb.x5((ilb.Rgb(12,15,ilb.Xgb(v).length),ilb.Xgb(v).substr(12,3))));b.g=ilb.Xgb(v).length>=39&&flb.j5(ilb.Xgb(v).substr(34,ilb.Xgb('V3000').length),'V3000');}catch(a){a=k1(a);if(uX(a,15)){return false;}else throw l1(a);}if(b.g){K=klb.Rq(b,c);klb.sm(b.j,D);return K;}!b.j&&(b.j=new klb.au(F,G));klb.sm(b.j,D);b.d||(b.j.L=true);if(0==F){while(v!=null&&!(flb.j5(v,xob)||flb.j5(v,yob)||flb.j5((ilb.Sgb(1,ilb.Xgb(v).length+1),ilb.Xgb(v).substr(1)),'$'))){v=Ilb.y2(c);}return true;}for(q=0;q<F;q++){if(null==(v=Ilb.y2(c))){return false;}V=h4(flb.x5((ilb.Rgb(0,10,ilb.Xgb(v).length),ilb.Xgb(v).substr(0,10))));W=h4(flb.x5((ilb.Rgb(10,20,ilb.Xgb(v).length),ilb.Xgb(v).substr(10,10))));X=h4(flb.x5((ilb.Rgb(20,30,ilb.Xgb(v).length),ilb.Xgb(v).substr(20,10))));e=klb.xj(b.j,V,-W,-X);u=flb.x5((ilb.Rgb(31,34,ilb.Xgb(v).length),ilb.Xgb(v).substr(31,3)));if(flb.j5(u,'A')||flb.j5(u,'*')){klb.Vl(b.j,e,1,true);}else if(flb.j5(u,'Q')){w=oW(elb.IX,omb,6,1,15,1);w[0]=6;klb.Ql(b.j,e,w,true);}else {h=klb.Om(u,67);klb.am(b.j,e,h);}C=klb.Pq(flb.x5((ilb.Rgb(34,36,ilb.Xgb(v).length),ilb.Xgb(v).substr(34,2))));C!=0&&klb.Tl(b.j,e,(klb.wj(),klb.uj)[klb.uk(b.j,e)]+C);m=klb.Pq(flb.x5((ilb.Rgb(36,39,ilb.Xgb(v).length),ilb.Xgb(v).substr(36,3))));m!=0&&(m==4?klb.Wl(b.j,e,32):klb.Jl(b.j,e,4-m));A=ilb.Xgb(v).length<63?0:klb.Pq(flb.x5((ilb.Rgb(60,63,ilb.Xgb(v).length),ilb.Xgb(v).substr(60,3))));klb.Rl(b.j,e,A,false);o=ilb.Xgb(v).length<45?0:klb.Pq(flb.x5((ilb.Rgb(42,45,ilb.Xgb(v).length),ilb.Xgb(v).substr(42,3))));switch(o){case 0:break;case 1:klb.Vl(b.j,e,768,true);break;case 2:klb.Vl(b.j,e,128,true);break;case 3:klb.Vl(b.j,e,384,true);break;default:klb.Vl(b.j,e,896,true);}ilb.Xgb(v).length>=48&&(ilb.Sgb(47,ilb.Xgb(v).length),ilb.Xgb(v).charCodeAt(47)==49)&&klb.Vl(b.j,e,snb,true);S=ilb.Xgb(v).length<51?0:klb.Pq(flb.x5((ilb.Rgb(48,51,ilb.Xgb(v).length),ilb.Xgb(v).substr(48,3))));if(S!=0){T==null&&(T=oW(elb.IX,omb,6,F,15,1));T[e]=S;}}for(r=0;r<G;r++){if(null==(v=Ilb.y2(c))){return false;}f=flb.W3(flb.x5((ilb.Rgb(0,3,ilb.Xgb(v).length),ilb.Xgb(v).substr(0,3))),wob,Ylb)-1;g=flb.W3(flb.x5((ilb.Rgb(3,6,ilb.Xgb(v).length),ilb.Xgb(v).substr(3,3))),wob,Ylb)-1;k=flb.W3(flb.x5((ilb.Rgb(6,9,ilb.Xgb(v).length),ilb.Xgb(v).substr(6,3))),wob,Ylb);N=ilb.Xgb(v).length<12?0:klb.Pq(flb.x5((ilb.Rgb(9,12,ilb.Xgb(v).length),ilb.Xgb(v).substr(9,3))));R=ilb.Xgb(v).length<18?0:klb.Pq(flb.x5((ilb.Rgb(15,18,ilb.Xgb(v).length),ilb.Xgb(v).substr(15,3))));k==8&&(b.k||klb.kl(b.j,f)||klb.kl(b.j,g))&&(k=9);klb.wq(b,f,g,k,N,R);}for(p=0;p<H;p++){if(null==(v=Ilb.y2(c))){return false;}}if(null==(v=Ilb.y2(c))){(b.i&1)!=0&&(b.f=klb.Kn(b.j));klb.Cq(b,T);b.d||klb.Jt(b.j,15);return true;}while(v!=null&&!(flb.j5(v,xob)||flb.j5(v,yob))){if(flb.j5(ilb.Xgb(v).substr(0,6),'M  CHG')){s=flb.W3(flb.x5((ilb.Rgb(6,9,ilb.Xgb(v).length),ilb.Xgb(v).substr(6,3))),wob,Ylb);if(s>0){d=10;U=14;for(t=1;t<=s;++t,d+=8,U+=8){e=flb.W3(flb.x5((ilb.Rgb(d,d+3,ilb.Xgb(v).length),ilb.Xgb(v).substr(d,d+3-d))),wob,Ylb)-1;l=flb.W3(flb.x5((ilb.Rgb(U,U+3,ilb.Xgb(v).length),ilb.Xgb(v).substr(U,U+3-U))),wob,Ylb);klb.Jl(b.j,e,l);}}}if(flb.j5(ilb.Xgb(v).substr(0,6),'M  ISO')){s=flb.W3(flb.x5((ilb.Rgb(6,9,ilb.Xgb(v).length),ilb.Xgb(v).substr(6,3))),wob,Ylb);if(s>0){d=10;U=14;for(t=1;t<=s;++t,d+=8,U+=8){e=flb.W3(flb.x5((ilb.Rgb(d,d+3,ilb.Xgb(v).length),ilb.Xgb(v).substr(d,d+3-d))),wob,Ylb)-1;B=flb.W3(flb.x5((ilb.Rgb(U,U+3,ilb.Xgb(v).length),ilb.Xgb(v).substr(U,U+3-U))),wob,Ylb);klb.Tl(b.j,e,B);}}}if(flb.j5(ilb.Xgb(v).substr(0,6),'M  RAD')){s=flb.W3(flb.x5((ilb.Rgb(6,9,ilb.Xgb(v).length),ilb.Xgb(v).substr(6,3))),wob,Ylb);if(s>0){d=10;U=14;for(t=1;t<=s;++t,d+=8,U+=8){e=flb.W3(flb.x5((ilb.Rgb(d,d+3,ilb.Xgb(v).length),ilb.Xgb(v).substr(d,d+3-d))),wob,Ylb)-1;J=flb.W3(flb.x5((ilb.Rgb(U,U+3,ilb.Xgb(v).length),ilb.Xgb(v).substr(U,U+3-U))),wob,Ylb);switch(J){case 1:klb.Wl(b.j,e,16);break;case 2:klb.Wl(b.j,e,32);break;case 3:klb.Wl(b.j,e,48);}}}}if(flb.j5(ilb.Xgb(v).substr(0,6),'M  RBC')||flb.j5(ilb.Xgb(v).substr(0,6),'M  RBD')){s=flb.W3(flb.x5((ilb.Rgb(6,9,ilb.Xgb(v).length),ilb.Xgb(v).substr(6,3))),wob,Ylb);if(s>0){d=10;U=14;for(t=1;t<=s;++t,d+=8,U+=8){e=flb.W3(flb.x5((ilb.Rgb(d,d+3,ilb.Xgb(v).length),ilb.Xgb(v).substr(d,d+3-d))),wob,Ylb)-1;L=flb.W3(flb.x5((ilb.Rgb(U,U+3,ilb.Xgb(v).length),ilb.Xgb(v).substr(U,U+3-U))),wob,Ylb);switch(L){case 3:case-1:klb.Vl(b.j,e,112,true);break;case 1:klb.Vl(b.j,e,8,true);break;case 2:klb.Vl(b.j,e,104,true);break;case 4:klb.Vl(b.j,e,56,true);}}}}if(flb.j5(ilb.Xgb(v).substr(0,6),'M  ALS')){e=flb.W3(flb.x5((ilb.Rgb(7,10,ilb.Xgb(v).length),ilb.Xgb(v).substr(7,3))),wob,Ylb)-1;if(e>=0){I=flb.W3(flb.x5((ilb.Rgb(10,13,ilb.Xgb(v).length),ilb.Xgb(v).substr(10,3))),wob,Ylb);i=(ilb.Sgb(14,ilb.Xgb(v).length),ilb.Xgb(v).charCodeAt(14)==84);S=oW(elb.IX,omb,6,I,15,1);d=16;for(t=0;t<I;++t,d+=4){Q=flb.x5((ilb.Rgb(d,d+4,ilb.Xgb(v).length),ilb.Xgb(v).substr(d,d+4-d)));S[t]=klb.Om(Q,1);}klb.am(b.j,e,6);klb.Ql(b.j,e,S,i);}}if(flb.j5(ilb.Xgb(v).substr(0,6),'M  SUB')){s=flb.W3(flb.x5((ilb.Rgb(6,9,ilb.Xgb(v).length),ilb.Xgb(v).substr(6,3))),wob,Ylb);if(s>0){d=10;U=14;for(t=1;t<=s;++t,d+=8,U+=8){e=flb.W3(flb.x5((ilb.Rgb(d,d+3,ilb.Xgb(v).length),ilb.Xgb(v).substr(d,d+3-d))),wob,Ylb)-1;O=flb.W3(flb.x5((ilb.Rgb(U,U+3,ilb.Xgb(v).length),ilb.Xgb(v).substr(U,U+3-U))),wob,Ylb);if(O==-2){klb.Vl(b.j,e,tnb,true);}else if(O>0){P=0;for(j=0;j<b.j.r;j++){(klb.zk(b.j,0,j)==e||klb.zk(b.j,1,j)==e)&&++P;}O>P&&klb.Vl(b.j,e,Qmb,true);}}}}if(flb.j5(ilb.Xgb(v).substr(0,6),'M  RGP')){s=flb.W3(flb.x5((ilb.Rgb(6,9,ilb.Xgb(v).length),ilb.Xgb(v).substr(6,3))),wob,Ylb);if(s>0){d=10;U=14;for(t=1;t<=s;++t,d+=8,U+=8){e=flb.W3(flb.x5((ilb.Rgb(d,d+3,ilb.Xgb(v).length),ilb.Xgb(v).substr(d,d+3-d))),wob,Ylb)-1;M=flb.W3(flb.x5((ilb.Rgb(U,U+3,ilb.Xgb(v).length),ilb.Xgb(v).substr(U,U+3-U))),wob,Ylb);M>=1&&M<=20&&klb.am(b.j,e,klb.Om('R'+M,2));}}}v=Ilb.y2(c);}}catch(a){a=k1(a);if(uX(a,15)){n=a;flb.cz(n,(flb.V5(),flb.U5),'');return false;}else throw l1(a);}if(b.e){klb.Mq(b);klb.xq(b);}(b.i&1)!=0&&(b.f=klb.Kn(b.j));klb.Cq(b,T);klb.Jt(b.j,15);return true;};klb.Rq=function Rq(a,b){var c,d,e,f,g;!!a.b&&llb.leb(a.b);!!a.c&&llb.leb(a.c);e=0;d=Ilb.y2(b);while(d!=null&&flb.j5(ilb.Xgb(d).substr(0,7),zob)){d=flb.x5((ilb.Sgb(7,ilb.Xgb(d).length+1),ilb.Xgb(d).substr(7)));while(g=ilb.Xgb('-').length,flb.j5(ilb.Xgb(d).substr(ilb.Xgb(d).length-g,g),'-')){c=Ilb.y2(b);if(!flb.j5(ilb.Xgb(c).substr(0,7),zob)){return false;}d=flb.x5(flb.g5(flb.w5(d,0,ilb.Xgb(d).length-1),(ilb.Sgb(7,ilb.Xgb(c).length+1),ilb.Xgb(c).substr(7))));}if(flb.j5(ilb.Xgb(d).substr(0,5),'BEGIN')){f=flb.x5((ilb.Sgb(6,ilb.Xgb(d).length+1),ilb.Xgb(d).substr(6)));if(flb.j5(ilb.Xgb(f).substr(0,4),'CTAB')){e=1;}else if(flb.j5(ilb.Xgb(f).substr(0,4),'ATOM')){e=2;}else if(flb.j5(ilb.Xgb(f).substr(0,4),'BOND')){e=3;}else if(flb.j5(ilb.Xgb(f).substr(0,10),'COLLECTION')){e=4;}else {return false;}}else if(flb.j5(ilb.Xgb(d).substr(0,3),'END')){e=0;}else if(e==1){klb.Kq(a,d);}else if(e==2){klb.Gq(a,d);}else if(e==3){klb.Iq(a,d);}else if(e==4){klb.Jq(a,d);}else {return false;}d=Ilb.y2(b);}while(d!=null&&!(flb.j5(ilb.Xgb(d).substr(0,6),xob)||flb.j5(d,yob))){d=Ilb.y2(b);}return true;};klb.Sq=function Sq(){this.i=0;};klb.Tq=function Tq(){this.i=1;};b2(101,1,{},klb.Sq,klb.Tq);_.a=false;_.d=false;_.e=false;_.g=false;_.i=0;_.k=false;elb.nY=D3(101);klb.Uq=function Uq(a){};klb.Vq=function Vq(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J;flb.N5(a.a,'M  V30 BEGIN CTAB\n');flb.N5(a.a,Aob+b.q+' '+b.r+' 0 0 0'+Slb);flb.N5(a.a,'M  V30 BEGIN ATOM\n');for(i=0;i<b.q;i++){flb.N5(a.a,zob+(i+1));if((b.v==null?null:b.v[i])!=null){k=b.v==null?null:b.v[i];u=I1(n1(b.B[i],1),0);flb.N5(a.a,u?' NOT[':' [');for(s=0;s<k.length;s++){s>0&&flb.N5(a.a,',');t=(klb.wj(),klb.qj)[k[s]];switch(ilb.Xgb(t).length){case 2:case 3:case 1:flb.N5(a.a,t);break;default:flb.N5(a.a,'?');}}flb.N5(a.a,']');}else I1(n1(b.B[i],1),0)?flb.N5(a.a,' A'):b.C[i]>=129&&b.C[i]<=144||b.C[i]==154?flb.N5(a.a,' R#'):flb.N5(a.a,' '+(klb.wj(),klb.qj)[b.C[i]]);if(c){flb.N5(a.a,' '+DX(gmb*a.b*b.J[i].a)/gmb);flb.N5(a.a,' '+DX(gmb*a.b*-b.J[i].b)/gmb);flb.N5(a.a,' '+DX(gmb*a.b*-b.J[i].c)/gmb);}else {flb.N5(a.a,' 0 0 0');}flb.N5(a.a,' '+$wnd.Math.abs(b.w[i]));b.s[i]!=0&&flb.N5(a.a,' CHG='+b.s[i]);if((b.u[i]&48)!=0){flb.N5(a.a,' RAD=');switch(b.u[i]&48){case 16:flb.N5(a.a,'1');break;case 32:flb.N5(a.a,'2');break;case 48:flb.N5(a.a,'3');}}if((b.u[i]&3)==1||(b.u[i]&3)==2){flb.N5(a.a,' CFG=');(b.u[i]&3)==1?flb.N5(a.a,'1'):flb.N5(a.a,'2');}b.A[i]!=0&&flb.N5(a.a,' MASS='+b.A[i]);J=((b.u[i]&Znb)>>>27)-1;J!=-1&&flb.N5(a.a,' VAL='+(J==0?'-1':flb.A4(J)));l=b.C[i];(l>=129&&l<=144||l==154)&&flb.N5(a.a,' RGROUPS=(1 '+(l==154?0:l>=142?l-141:l-125)+')');r=n1(Umb,b.B[i]);s1(r,384)==0?flb.N5(a.a,' HCOUNT=2'):s1(r,128)==0?flb.N5(a.a,' HCOUNT=1'):s1(r,1792)==0?flb.N5(a.a,' HCOUNT=-1'):s1(r,1664)==0&&flb.N5(a.a,' HCOUNT=1');G=n1(b.B[i],6144);s1(G,0)!=0&&(I1(n1(G,Qmb),0)?flb.N5(a.a,' SUBST='+(b.e[i]+1)):flb.N5(a.a,' SUBST=-1'));C=n1(b.B[i],120);s1(C,0)!=0&&(s1(C,112)==0?flb.N5(a.a,' RBCNT=-1'):s1(C,8)==0?flb.N5(a.a,' RBCNT=2'):s1(C,104)==0?flb.N5(a.a,' RBCNT=2'):s1(C,88)==0?flb.N5(a.a,' RBCNT=3'):s1(C,56)==0&&flb.N5(a.a,' RBCNT=4'));flb.N5(a.a,Slb);}flb.N5(a.a,'M  V30 END ATOM\n');flb.N5(a.a,'M  V30 BEGIN BOND\n');for(n=0;n<b.r;n++){flb.N5(a.a,zob+(n+1));switch(b.H[n]){case 1:A=1;F=0;break;case 2:A=2;F=0;break;case 4:A=3;F=0;break;case 129:A=1;F=3;break;case 257:A=1;F=1;break;case 386:A=2;F=2;break;case 64:A=4;F=0;break;case 32:A=9;F=0;break;default:A=1;F=0;}p=b.G[n]&31;p!=0&&(p==8?A=4:p==3?A=5:p==9?A=6:p==10?A=7:A=8);flb.N5(a.a,' '+A+' '+(b.D[0][n]+1)+' '+(b.D[1][n]+1));F!=0&&flb.N5(a.a,' CFG='+F);D=b.G[n]&384;H=D==0?0:D==256?1:2;H!=0&&flb.N5(a.a,' TOPO='+H);flb.N5(a.a,Slb);}flb.N5(a.a,'M  V30 END BOND\n');B=false;d=0;v=oW(elb.IX,omb,6,32,15,1);f=oW(elb.IX,omb,6,32,15,1);for(j=0;j<b.f;j++){if((b.u[j]&3)==1||(b.u[j]&3)==2){B=true;I=(b.u[j]&znb)>>18;I==1?++f[(b.u[j]&znb)>>18!=1&&(b.u[j]&znb)>>18!=2?-1:(b.u[j]&$nb)>>20]:I==2?++v[(b.u[j]&znb)>>18!=1&&(b.u[j]&znb)>>18!=2?-1:(b.u[j]&$nb)>>20]:++d;}}e=0;w=oW(elb.IX,omb,6,32,15,1);g=oW(elb.IX,omb,6,32,15,1);for(o=0;o<b.g;o++){if(klb.Hk(b,o)!=2&&((b.F[o]&3)==1||(b.F[o]&3)==2)){B=true;I=(b.F[o]&768)>>8;I==1?++g[(b.F[o]&768)>>8!=1&&(b.F[o]&768)>>8!=2?-1:(b.F[o]&_nb)>>10]:I==2?++w[(b.F[o]&768)>>8!=1&&(b.F[o]&768)>>8!=2?-1:(b.F[o]&_nb)>>10]:++e;}}if(B){flb.N5(a.a,'M  V30 BEGIN COLLECTION\n');if(d!=0){flb.N5(a.a,'M  V30 MDLV30/STEABS ATOMS=('+d);for(h=0;h<b.f;h++){((b.u[h]&3)==1||(b.u[h]&3)==2)&&(b.u[h]&znb)>>18==0&&flb.N5(a.a,' '+(h+1));}flb.N5(a.a,Bob);}if(e!=0){flb.N5(a.a,'M  V30 MDLV30/STEABS BONDS=('+e);for(m=0;m<b.g;m++){klb.Hk(b,m)!=2&&((b.F[m]&3)==1||(b.F[m]&3)==2)&&(b.F[m]&768)>>8==0&&flb.N5(a.a,' '+(m+1));}flb.N5(a.a,Bob);}for(q=0;q<32;q++){if(v[q]!=0){flb.N5(a.a,Cob+(q+1)+' ATOMS=('+v[q]);for(h=0;h<b.f;h++){((b.u[h]&3)==1||(b.u[h]&3)==2)&&(b.u[h]&znb)>>18==2&&((b.u[h]&znb)>>18!=1&&(b.u[h]&znb)>>18!=2?-1:(b.u[h]&$nb)>>20)==q&&flb.N5(a.a,' '+(h+1));}flb.N5(a.a,Bob);}if(f[q]!=0){flb.N5(a.a,Dob+(q+1)+' ATOMS=('+f[q]);for(h=0;h<b.f;h++){((b.u[h]&3)==1||(b.u[h]&3)==2)&&(b.u[h]&znb)>>18==1&&((b.u[h]&znb)>>18!=1&&(b.u[h]&znb)>>18!=2?-1:(b.u[h]&$nb)>>20)==q&&flb.N5(a.a,' '+(h+1));}flb.N5(a.a,Bob);}if(w[q]!=0){flb.N5(a.a,Cob+(q+1)+' BONDS=('+w[q]);for(m=0;m<b.g;m++){klb.Hk(b,m)!=2&&((b.F[m]&3)==1||(b.F[m]&3)==2)&&(b.F[m]&768)>>8==2&&((b.F[m]&768)>>8!=1&&(b.F[m]&768)>>8!=2?-1:(b.F[m]&_nb)>>10)==q&&flb.N5(a.a,' '+(m+1));}flb.N5(a.a,Bob);}if(g[q]!=0){flb.N5(a.a,Dob+(q+1)+' BONDS=('+g[q]);for(m=0;m<b.g;m++){klb.Hk(b,m)!=2&&((b.F[m]&3)==1||(b.F[m]&3)==2)&&(b.F[m]&768)>>8==1&&((b.F[m]&768)>>8!=1&&(b.F[m]&768)>>8!=2?-1:(b.F[m]&_nb)>>10)==q&&flb.N5(a.a,' '+(m+1));}flb.N5(a.a,Bob);}}flb.N5(a.a,'M  V30 END COLLECTION\n');}flb.N5(a.a,'M  V30 END CTAB\n');};klb.Wq=function Wq(){klb.Uq(this);this.a=new flb.R5();};klb.Xq=function Xq(a){klb.Yq.call(this,a);};klb.Yq=function Yq(a){klb.Zq.call(this,a,new flb.R5());};klb.Zq=function Zq(a,b){var c,d;klb.Uq(this);klb.Jt(a,15);this.a=b;d=a.P!=null?a.P:'';flb.N5(this.a,d+Slb);flb.N5(this.a,'Actelion Java MolfileCreator 2.0\n\n');flb.N5(this.a,'  0  0  0  0  0  0              0 V3000\n');this.b=1;c=klb._q(a);c&&(this.b=klb.$q(a));klb.Vq(this,a,c);flb.N5(this.a,uob);};klb.$q=function $q(a){var b,c,d,e,f,g,h,i,j;j=1;d=klb.wk(a,a.q,a.r,(klb.wj(),klb.vj));if(d!=0){(d<1||d>3)&&(j=1.5/d);}else {i=Eob;for(b=1;b<a.q;b++){for(c=0;c<b;c++){f=a.J[c].a-a.J[b].a;g=a.J[c].b-a.J[b].b;h=a.J[c].c-a.J[b].c;e=f*f+g*g+h*h;i>e&&(i=e);}}j=3/$wnd.Math.max(0.75,i);}return j;};klb._q=function _q(a){var b;if(a.q==1)return true;for(b=1;b<a.q;b++){if(a.J[b].a!=a.J[0].a||a.J[b].b!=a.J[0].b||a.J[b].c!=a.J[0].c){return true;}}return false;};klb.ar=function ar(a,b){var c;c=new klb.Wq();c.b=b;klb.Jt(a,15);klb.Vq(c,a,true);return c.a.a;};b2(115,1,{},klb.Wq,klb.Xq);_.b=1;elb.oY=D3(115);klb.br=function br(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p;k=a.g.M;l=0;for(f=0;f<c;f++){if(k>b[f]){k=b[f];l=f;}}p=oW(elb.IX,omb,6,c,15,1);j=l>0?l-1:c-1;m=l<c-1?l+1:0;h=b[j]<b[m];for(g=0;g<c;g++){p[g]=b[l];h?--l<0&&(l=c-1):++l==c&&(l=0);}for(e=0;e<a.i.a.length;e++){o=llb.Bi(a.i,e);if(o.length!=c)continue;d=true;for(i=0;i<c;i++){if(o[i]!=p[i]){d=false;break;}}if(d)return;}llb.wi(a.i,p);n=klb.nr(a,p);llb.wi(a.j,n);klb.Er(a,p,n);};klb.cr=function cr(a,b,c){var d,e,f,g,h,i,j;i=oW(elb.IX,omb,6,a.f,15,1);g=oW(elb.IX,omb,6,a.f,15,1);j=oW(elb.g1,Imb,6,a.g.f,16,1);d=klb.zk(a.g,0,b);e=klb.zk(a.g,1,b);i[0]=d;i[1]=e;g[1]=-1;j[e]=true;h=1;while(h>=1){++g[h];if(g[h]==klb.xn(a.g,i[h])){j[i[h]]=false;--h;continue;}f=klb.wn(a.g,i[h],g[h]);if(j[f]||c[f])continue;if(f==d&&h>1){klb.br(a,i,h+1);if(a.i.a.length>=Vmb)return;continue;}if(h+1<a.f){++h;i[h]=f;j[f]=true;g[h]=-1;}}};klb.dr=function dr(a,b,c,d,e,f,g,h){var i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D;A=llb.Bi(a.i,b);for(l=A,m=0,n=l.length;m<n;++m){k=l[m];if(!klb.Ar(a,k))return true;}B=llb.Bi(a.j,b);C=B.length;o=0;j=0;D=false;for(v=0;v<C;v++){o<<=1;j<<=1;if(klb.Cr(a,B[v])){o|=1;}else if(h&&klb.zr(a,B[v])){o|=1;j|=1;}else {i=c[b][v];if(i!=-1){if(d[i]){if(e[i]){o|=1;f[i]||(j|=1);}}else {D=true;}}}}u=false;switch(C){case 3:p=wW(iW(elb.IX,1),omb,6,15,[2,1,4]);u=true;for(t=0;t<3;t++){if((o&p[t])==p[t]){if(klb.uk(a.g,A[t])==6&&klb.dk(a.g,A[t])==1||klb.uk(a.g,A[t])==5&&klb.dk(a.g,A[t])==0){e[b]=true;g[b]=t;(j&p[t])==0&&(u=false);}}}break;case 5:q=wW(iW(elb.IX,1),omb,6,15,[10,5,18,9,20]);u=true;for(w=0;w<5;w++){if((o&q[w])==q[w]){switch(klb.uk(a.g,A[w])){case 6:if(klb.dk(a.g,A[w])==-1){e[b]=true;g[b]=w;(j&q[w])==0&&(u=false);}break;case 7:if(klb.dk(a.g,A[w])<=0){e[b]=true;g[b]=w;}break;case 8:e[b]=true;g[b]=w;break;case 16:case 34:case 52:if(klb.xn(a.g,A[w])==2){e[b]=true;g[b]=w;}}}}break;case 6:u=true;if((o&21)==21){e[b]=true;(j&21)==0&&(u=false);}if((o&42)==42){e[b]=true;(j&42)==0&&(u=false);}break;case 7:r=wW(iW(elb.IX,1),omb,6,15,[42,21,74,37,82,41,84]);u=true;for(s=0;s<7;s++){if((o&r[s])==r[s]){if(klb.uk(a.g,A[s])==6&&(klb.dk(a.g,A[s])==1||h&&klb.qr(a,A[s]))||klb.uk(a.g,A[s])==5&&klb.dk(a.g,A[s])==0){e[b]=true;g[b]=s;(j&r[s])==0&&(u=false);}}}}e[b]&&!u&&(f[b]=true);if(e[b])return true;return !D;};klb.er=function er(a,b,c,d,e){var f,g,h,i,j,k,l,m,n,o,p,q;f=oW(elb.IX,Lnb,7,a.i.a.length,0,2);for(j=0;j<a.i.a.length;j++){f[j]=oW(elb.IX,omb,6,llb.Bi(a.i,j).length,15,1);for(k=0;k<llb.Bi(a.i,j).length;k++)f[j][k]=-1;}p=oW(elb.IX,omb,6,a.g.g,15,1);for(n=0;n<a.j.a.length;n++){o=llb.Bi(a.j,n);if(o.length==3||o.length>=5&&o.length<=7){for(i=0;i<o.length;i++){h=o[i];if(klb.xn(a.g,klb.zk(a.g,0,h))==3&&klb.xn(a.g,klb.zk(a.g,1,h))==3){if(p[h]>0){f[p[h]>>>16][p[h]&32767]=n;f[n][i]=p[h]>>>16;}else {p[h]=(n<<16)+eob+i;}}}}}g=oW(elb.g1,Imb,6,a.i.a.length,16,1);q=0;l=-1;while(q>l){l=q;for(m=0;m<a.i.a.length;m++){if(!g[m]){if(klb.dr(a,m,f,g,b,c,d,e)){g[m]=true;++q;}}}}};klb.fr=function fr(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o;e=klb.zk(a.g,0,b);f=klb.zk(a.g,1,b);i=oW(elb.IX,omb,6,a.g.f,15,1);j=oW(elb.IX,omb,6,a.g.f,15,1);k=oW(elb.IX,omb,6,a.g.f,15,1);i[0]=e;i[1]=f;j[e]=1;j[f]=2;k[e]=-1;k[f]=e;h=1;l=1;while(h<=l){for(m=0;m<klb.xn(a.g,i[h]);m++){g=klb.wn(a.g,i[h],m);if(h>1&&g==e){o=oW(elb.IX,omb,6,j[i[h]],15,1);d=i[h];for(n=0;n<o.length;n++){o[n]=d;d=k[d];}return o;}if(j[g]==0&&!c[g]){i[++l]=g;j[g]=j[i[h]]+1;k[g]=i[h];}}++h;}return null;};klb.gr=function gr(a,b,c){var d,e;e=llb.Bi(a.i,b);for(d=0;d<e.length;d++)if(c==e[d])return d;return -1;};klb.hr=function hr(a,b){return a.a[b]&fmb;};klb.ir=function ir(a,b,c){var d,e;e=llb.Bi(a.j,b);for(d=0;d<e.length;d++)if(c==e[d])return d;return -1;};klb.jr=function jr(a,b){return a.b[b]&fmb;};klb.kr=function kr(a,b){return a.c[b];};klb.lr=function lr(a,b){return llb.Bi(a.i,b);};klb.mr=function mr(a,b){return llb.Bi(a.j,b);};klb.nr=function nr(a,b){var c,d,e,f,g;f=b.length;g=oW(elb.IX,omb,6,f,15,1);for(d=0;d<f;d++){c=d==f-1?b[0]:b[d+1];for(e=0;e<klb.xn(a.g,b[d]);e++){if(klb.wn(a.g,b[d],e)==c){g[d]=klb.yn(a.g,b[d],e);break;}}}return g;};klb.or=function or(a,b){return llb.Bi(a.j,b).length;};klb.pr=function pr(a,b,c){var d;for(d=0;d<a.j.a.length;d++)if(klb.vr(a,d,b)&&klb.vr(a,d,c))return d;return -1;};klb.qr=function qr(a,b){var c;for(c=0;c<klb.xn(a.g,b);c++)if(klb.zn(a.g,b,c)==2&&klb.uk(a.g,klb.wn(a.g,b,c))==8)return true;return false;};klb.rr=function rr(a,b){return a.d[b];};klb.sr=function sr(a,b){return (a.a[b]&emb)!=0;};klb.tr=function tr(a,b){return (a.b[b]&emb)!=0;};klb.ur=function ur(a,b,c){var d,e;e=llb.Bi(a.i,b);for(d=0;d<e.length;d++)if(c==e[d])return true;return false;};klb.vr=function vr(a,b,c){var d,e;e=llb.Bi(a.j,b);for(d=0;d<e.length;d++)if(c==e[d])return true;return false;};klb.wr=function wr(a,b){return a.e[b];};klb.xr=function xr(a,b){return (a.b[b]&Onb)!=0;};klb.yr=function yr(a,b){return (a.a[b]&Bnb)!=0;};klb.zr=function zr(a,b){var c,d,e,f,g,h;for(g=0;g<2;g++){c=klb.zk(a.g,g,b);if(klb.uk(a.g,c)==7&&klb.xn(a.g,c)==2){d=klb.zk(a.g,1-g,b);if(klb.uk(a.g,d)==6){for(h=0;h<klb.xn(a.g,d);h++){e=klb.wn(a.g,d,h);f=klb.yn(a.g,d,h);if((klb.uk(a.g,e)==8||klb.uk(a.g,e)==16)&&klb.Hk(a.g,f)==2&&klb.xn(a.g,e)==1)return true;}}}}return false;};klb.Ar=function Ar(a,b){var c,d,e,f,g;if(a.g.K){if(I1(n1(klb.pk(a.g,b),1),0)){return true;}else {g=klb.kk(a.g,b);if(g!=null){for(d=g,e=0,f=d.length;e<f;++e){c=d[e];if(c==5||c==6||c==7||c==8||c==15||c==16||c==33||c==34)return true;}return false;}}}return klb.Ir(klb.uk(a.g,b));};klb.Br=function Br(a,b){var c,d,e,f,g;if(a.g.K){if(I1(n1(klb.pk(a.g,b),1),0))return false;c=klb.kk(a.g,b);if(c!=null){for(e=c,f=0,g=e.length;f<g;++f){d=e[f];if(!klb.Pm(d))return false;}return true;}}return klb.Pm(klb.uk(a.g,b));};klb.Cr=function Cr(a,b){return klb.Hk(a.g,b)>1||klb.Kk(a.g,b)==64;};klb.Dr=function Dr(a){var b,c,d,e,f,g,h,i,j,k,l,m,n,o,p,q;for(q=0;q<a.d.length;q++){if(a.d[q]){p=false;for(d=llb.Bi(a.i,q),f=0,h=d.length;f<h;++f){b=d[f];a.a[b]|=emb;klb.Br(a,b)&&(p=true);}for(k=llb.Bi(a.j,q),m=0,o=k.length;m<o;++m){i=k[m];a.b[i]|=emb;}if(a.e[q]){for(c=llb.Bi(a.i,q),e=0,g=c.length;e<g;++e){b=c[e];a.a[b]|=Onb;}for(j=llb.Bi(a.j,q),l=0,n=j.length;l<n;++l){i=j[l];a.b[i]|=Onb;}}if(p){for(c=llb.Bi(a.i,q),e=0,g=c.length;e<g;++e){b=c[e];a.a[b]|=Bnb;}for(j=llb.Bi(a.j,q),l=0,n=j.length;l<n;++l){i=j[l];a.b[i]|=Bnb;}}}}};klb.Er=function Er(a,b,c){var d,e,f,g;g=b.length;for(f=0;f<g;f++){d=a.a[b[f]]&fmb;if(d==0||d>g){a.a[b[f]]&=Unb;a.a[b[f]]|=g;}}for(e=0;e<g;e++){d=a.b[c[e]]&fmb;if(d==0||d>g){a.b[c[e]]&=Unb;a.b[c[e]]|=g;}}};klb.Fr=function Fr(a,b,c){var d;d=llb.Bi(a.j,b).length;while(c>=d)c-=d;while(c<0)c+=d;return c;};klb.Gr=function Gr(a,b){klb.Hr.call(this,a,b);};klb.Hr=function Hr(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s;this.g=a;this.f=7;this.i=new llb.Qi();this.j=new llb.Qi();this.a=oW(elb.IX,omb,6,this.g.f,15,1);this.b=oW(elb.IX,omb,6,this.g.g,15,1);this.g.wb(1);m=oW(elb.g1,Imb,6,this.g.f,16,1);n=oW(elb.g1,Imb,6,this.g.g,16,1);do{g=false;for(c=0;c<this.g.f;c++){if(!m[c]){q=0;for(l=0;l<klb.xn(this.g,c);l++)m[klb.wn(this.g,c,l)]||++q;if(q<2){m[c]=true;for(k=0;k<klb.xn(this.g,c);k++)n[klb.yn(this.g,c,k)]=true;g=true;}}}}while(g);s=0;while(s<this.g.f&&m[s])++s;if(s==this.g.f)return;i=oW(elb.IX,omb,6,this.g.f,15,1);i[0]=s;p=oW(elb.IX,omb,6,this.g.f,15,1);p[0]=-1;h=oW(elb.IX,omb,6,this.g.f,15,1);h[s]=1;f=0;j=0;o=1;while(f<=j){for(k=0;k<klb.xn(this.g,i[f]);k++){e=klb.wn(this.g,i[f],k);if(e==p[i[f]])continue;if(h[e]!=0){klb.cr(this,klb.yn(this.g,i[f],k),m);continue;}if(!m[e]){h[e]=o;p[e]=i[f];i[++j]=e;}}++f;if(f>j){for(c=0;c<this.g.f;c++){if(h[c]==0&&!m[c]){h[c]=++o;i[++j]=c;p[c]=-1;break;}}}}if((b&4)!=0){this.d=oW(elb.g1,Imb,6,this.i.a.length,16,1);this.e=oW(elb.g1,Imb,6,this.i.a.length,16,1);this.c=oW(elb.IX,omb,6,this.i.a.length,15,1);klb.er(this,this.d,this.e,this.c,(b&8)!=0);klb.Dr(this);}if((b&2)!=0){for(d=0;d<this.g.g;d++){if(!n[d]&&klb.Hk(this.g,d)!=0){r=klb.fr(this,d,m);r!=null&&klb.Er(this,r,klb.nr(this,r));}}}};klb.Ir=function Ir(a){return a==5||a==6||a==7||a==8||a==15||a==16||a==33||a==34;};b2(118,1,{},klb.Gr);_.f=0;elb.pY=D3(118);klb.Jr=function Jr(a,b){var c,d,e,f,g,h,i,j,k,l,m;if(b==2||b==5){llb.wi(a.D,klb.ms(a.F,a.F.length));!!a.b&&llb.wi(a.a,klb.ns(a.B,a.B.length));}else if(b==4){m=klb.Vr(klb.ms(a.F,a.F.length));if(!llb.mfb(a.N,m)){llb.kfb(a.N,m);llb.wi(a.D,klb.ms(a.F,a.F.length));!!a.b&&llb.wi(a.a,klb.ns(a.B,a.B.length));}}else if(b==3){m=klb.Vr(klb.ms(a.F,a.F.length));if(!llb.mfb(a.N,m)){l=false;for(j=(h=new llb.Keb(new llb.Peb(new llb.cbb(a.N.a).a).b),new llb.kbb(h));llb.P9(j.a.a);){i=(g=llb.Ieb(j.a),g.hc());k=0;for(d=m,e=0,f=d.length;e<f;++e){c=d[e];while(k<i.length&&i[k]<c)++k;if(k<i.length){if(c==i[k]){l=true;break;}}}if(l)break;}if(!l){llb.kfb(a.N,m);llb.wi(a.D,klb.ms(a.F,a.F.length));!!a.b&&llb.wi(a.a,klb.ns(a.B,a.B.length));}}}else if(b==6){m=klb.Wr(a,klb.ms(a.F,a.F.length));if(!llb.mfb(a.N,m)){llb.kfb(a.N,m);llb.wi(a.D,klb.ms(a.F,a.F.length));!!a.b&&llb.wi(a.a,klb.ns(a.B,a.B.length));}}};klb.Kr=function Kr(a,b,c){var d,e,f,g,h,i,j,k,l,m;h=klb.xn(a.G,b);d=a.n[c];if(d>h)return false;j=klb.pk(a.G,b);f=klb.pk(a.g,c);e=klb.kk(a.g,c);i=klb.kk(a.G,b);if(I1(n1(f,1),0)){if(e!=null){if(I1(n1(j,1),0)){if(i==null)return false;if(!klb._r(e,i))return false;}else {if(i!=null){if(klb.bs(i,e))return false;}else {if(klb.$r(klb.uk(a.G,b),e))return false;}}}}else {if(I1(n1(j,1),0))return false;if(e!=null){if(i!=null){if(!klb._r(i,e))return false;}else {if(!klb.$r(klb.uk(a.G,b),e))return false;}}else {if(i!=null)return false;if(a.I[b]!=a.j[c])return false;}}if(I1(K1(j,f),0)){if(I1(n1(f,tnb),0)){if(a.G.K&&x1(n1(j,tnb),0))return false;else if(d!=h)return false;}if(I1(n1(f,Qmb),0)){if(d>=h&&x1(n1(j,Qmb),0))return false;}}if(I1(n1(a.H[b],J1(a.i[c])),0))return false;if(I1(n1(a.A[c],J1(a.L[b])),0))return false;g=n1(f,smb);if(a.G.K){k=n1(f,smb);if(s1(k,0)!=0&&(s1(g,0)==0||I1(n1(g,J1(k)),0)))return false;}else {if(s1(g,0)!=0&&x1(n1(g,a.L[b]),0))return false;}if(klb.dk(a.g,c)!=0&&klb.dk(a.g,c)!=klb.dk(a.G,b))return false;if(klb.nk(a.g,c)!=0&&klb.nk(a.g,c)!=klb.nk(a.G,b))return false;if(klb.qk(a.g,c)!=0&&klb.qk(a.g,c)!=klb.qk(a.G,b))return false;l=R1(M1(n1(klb.pk(a.g,c),qnb),22));if(l!=0){if(a.G.K){m=R1(M1(n1(klb.pk(a.G,b),qnb),22));if(l!=m)return false;}else {if(klb.rn(a.G,b)!=l)return false;}}return true;};klb.Lr=function Lr(a,b,c){var d,e,f,g,h,i,j,k,l;j=a.J[b];g=a.k[c];if((klb.Jk(a.g,c)&Tnb)!=0){i=klb.Lk(a.G,b);e=klb.Lk(a.g,c);f=klb.Jk(a.g,c)&31;if(i!=e&&!(i==1&&(f&1)!=0)&&!(i==2&&(f&2)!=0)&&!(i==4&&(f&4)!=0)&&!(i==8&&(f&32)!=0)&&!(i==16&&(f&64)!=0)&&!(i==32&&(f&16)!=0)&&!(i==64&&(f&8)!=0))return false;j&=-32;g&=-32;}if((j&~g)!=0)return false;l=(klb.Jk(a.g,c)&enb)>>17;if(l!=0){if(a.G.K&&l==(klb.Jk(a.G,c)&enb)>>17)return true;d=false;k=klb.$n(a.G);for(h=0;h<k.i.a.length;h++){if(llb.Bi(k.j,h).length==l){if(klb.vr(k,h,b)){d=true;break;}}}if(!d)return false;}return true;};klb.Mr=function Mr(a){var b,c;a.b=null;for(b=0;b<a.g.g;b++){if(klb.cl(a.g,b)){!a.b&&(a.b=new llb.Qi());c=new klb.os();c.a=klb.zk(a.g,0,b);c.b=klb.zk(a.g,1,b);c.d=klb.Bk(a.g,b);c.c=klb.Ak(a.g,b);llb.wi(a.b,c);}}};klb.Nr=function Nr(a){var b,c,d,e,f,g,h,i,j,k,l,m;klb.Jt(a.g,a.M);i=$wnd.Math.max(a.g.f,a.g.g)+16;a.r=oW(elb.IX,omb,6,i,15,1);a.t=oW(elb.IX,omb,6,i,15,1);a.u=oW(elb.IX,omb,6,i,15,1);a.s=oW(elb.g1,Imb,6,i+1,16,1);g=oW(elb.g1,Imb,6,a.g.f,16,1);h=oW(elb.g1,Imb,6,a.g.g,16,1);d=0;for(c=0;c<a.g.f;c++){if(!a.C[c]&&!g[c]){a.r[d]=c;a.u[d]=-1;a.t[d]=-1;j=d;while(d<=j){for(k=0;k<klb.jn(a.g,a.r[d]);k++)j=klb.js(a,d,j,k,g,h,-1);while(a.s[++d]);}}}a.v=d;if(a.o!=0){j=a.v-1;for(f=0;f<a.d;f++){d=0;while(d<=j){for(l=0;l<klb.jn(a.g,a.r[d]);l++)j=klb.js(a,d,j,l,g,h,f);while(a.s[++d]);}}for(b=0;b<a.g.f;b++){if(a.C[b]&&!g[b]){a.r[d]=b;a.u[d]=-1;a.t[d]=-1;j=d;while(d<=j){for(l=0;l<klb.jn(a.g,a.r[d]);l++)klb.wn(a.g,a.r[d],l)<a.g.f&&(j=klb.js(a,d,j,l,g,h,a.f[b]));while(a.s[++d]);}}}a.e=oW(elb.IX,omb,6,a.d,15,1);for(m=0;m<a.d;m++)a.e[m]=-1;for(k=a.v;k<d;k++){e=a.f[a.r[k]];a.e[e]==-1&&(a.e[e]=k);}}a.w=d;};klb.Or=function Or(a,b){var c,d,e,f,g,h,i,j;j=klb.$n(a.G);for(h=0;h<j.i.a.length;h++){if(klb.vr(j,h,b)){i=oW(elb.IX,omb,6,2,15,1);for(e=0;e<2;e++){i[e]=Ylb;d=klb.zk(a.G,e,b);for(f=0;f<klb.xn(a.G,d);f++){c=klb.wn(a.G,d,f);c!=klb.zk(a.G,1-e,b)&&i[e]>c&&(i[e]=c);}}g=0;klb.ur(j,h,i[0])&&++g;klb.ur(j,h,i[1])&&++g;if(g==2)return 2;if(g==1)return 1;return 2;}}return 0;};klb.Pr=function Pr(a,b,c){var d,e,f,g,h;if(a.b){a.B=oW(elb.g1,Imb,6,b.length,16,1);for(e=new llb.Wbb(a.b);e.a<e.c.a.length;){d=llb.Vbb(e);if(a.f==null||c==-1&&a.f[d.a]==-1&&a.f[d.b]==-1||c!=-1&&(a.f[d.a]==c||a.f[d.b]==c)){h=oW(elb.IX,omb,6,d.c+2,15,1);f=klb.Un(a.G,h,a.F[d.a],a.F[d.b],d.c+1,b,null)-1;if(f<d.d||f>d.c)return false;for(g=1;g<=f;g++)a.B[h[g]]=true;}}}return true;};klb.Qr=function Qr(a,b){var c,d,e,f,g,h,i,j;for(e=0;e<a.g.g;e++){if((klb.Jk(a.g,e)&Qnb)!=0){f=klb.Ik(a.g,e);if(f==0)continue;c=klb.zk(a.g,0,e);d=klb.zk(a.g,1,e);if(a.f==null||b==-1&&a.f[c]==-1&&a.f[d]==-1||b!=-1&&(a.f[c]==b||a.f[d]==b)){g=a.F[c];h=a.F[d];i=klb.un(a.G,g,h);j=klb.Ik(a.G,i);if(j==0){klb.Fo(a.G,i)&&(j=klb.Or(a,i));if(j==0)continue;}if(f==3)continue;if(j==3)continue;if(klb.Xr(a,e)==(f==j))return false;}}}return true;};klb.Rr=function Rr(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p;g=0;for(i=0;i<a.g.f;i++){if((a.f==null||a.f[i]==b)&&I1(n1(klb.pk(a.g,i),snb),0)){m=a.F[i];l=klb.ok(a.g,i);o=klb.ok(a.G,m);if(l==0)continue;if(l==3)continue;if(o==0||o==3)return false;if(klb.ik(a.g,i)==1){++g;continue;}if(klb.ik(a.G,m)==1)return false;if(klb.ik(a.g,i)==2){++g;continue;}if(klb.ik(a.G,m)==2)return false;if(klb.as(a,i)==(l==o))return false;}}if(g!=0){e=oW(elb.IX,omb,6,g,15,1);f=0;for(j=0;j<a.g.f;j++){if((a.f==null||a.f[j]==b)&&I1(n1(klb.pk(a.g,j),snb),0)){l=klb.ok(a.g,j);l!=0&&l!=3&&(e[f++]=klb.hk(a.g,j)<<24|klb.ik(a.g,j)<<22|j);}}ilb.rgb(e,ilb.Xgb(e2(llb.Ccb.prototype.kc,llb.Ccb,[])));f=0;while(f<e.length){k=e[f]&iob;n=a.F[k];c=e[f]&-4194304;d=klb.as(a,k)^klb.ok(a.g,k)==klb.ok(a.G,n);for(++f;f<e.length&&(e[f]&-4194304)==c;f++){h=e[f]&iob;m=a.F[h];if(klb.ik(a.G,m)!=klb.ik(a.G,n)||klb.hk(a.G,m)!=klb.hk(a.G,n))return false;p=klb.as(a,h)^klb.ok(a.g,h)==klb.ok(a.G,m);if(p!=d)return false;}}}return true;};klb.Sr=function Sr(a,b,c){var d,e,f,g,h,i,j;a.O=false;a.D=new llb.Qi();a.a=new llb.Qi();llb.leb(a.N.a);if(!a.G||!a.g)return 0;if(a.g.f-a.o>a.G.f||a.g.g-a.p>a.G.g)return 0;if(a.g.f==0)return 0;b==6&&(a.M=63);klb.fs(a,c);d=oW(elb.g1,Imb,6,a.G.f,16,1);a.F=oW(elb.IX,omb,6,a.g.f,15,1);llb.ecb(a.F);h=oW(elb.IX,omb,6,a.w,15,1);llb.jcb(h,h.length,-1);f=0;while(!a.O){if(a.v!=0){j=a.t[f]==-1?a.G.f:klb.jn(a.G,a.F[a.t[f]]);++h[f];if(h[f]==j){h[f]=-1;if(f==0)break;--f;a.s[f]||(d[a.F[a.r[f]]]=false);continue;}if(a.t[f]==-1){if(!d[h[f]]){if(klb.Kr(a,h[f],a.r[f])){a.F[a.r[f]]=h[f];d[h[f]]=true;++f;}}}else {if(klb.wn(a.G,a.F[a.t[f]],h[f])>=a.G.f)continue;e=klb.wn(a.G,a.F[a.t[f]],h[f]);if(a.s[f]){e==a.F[a.r[f]]&&klb.Lr(a,klb.yn(a.G,a.F[a.t[f]],h[f]),a.u[f])&&++f;}else {if(!d[e]){if(klb.Kr(a,e,a.r[f])&&klb.Lr(a,klb.yn(a.G,a.F[a.t[f]],h[f]),a.u[f])){d[e]=true;a.F[a.r[f]]=e;++f;}}}}}if(f==a.v){if(klb.Rr(a,-1)&&klb.Qr(a,-1)&&klb.Pr(a,d,-1)){i=false;for(g=0;g<a.d;g++){if(klb.Yr(a,d,h,g)){i=true;break;}}if(b==1&&!i)return 1;if(!i){klb.Jr(a,b);if(b==2)return 1;}}if(f==0)break;--f;a.s[f]||(d[a.F[a.r[f]]]=false);}}return a.D.a.length;};klb.Tr=function Tr(a,b){var c,d,e,f,g,h,i,j;i=0;if(a.K){b<a.f&&klb.yr(a.p,b)?i=K1(i,Fob):b<a.f&&klb.sr(a.p,b)&&(i=K1(i,2));j=(e=a.u[b]&job,e==0?0:e==Vmb?2:e==tnb?3:4);if(j!=0){i=K1(i,8);j>2&&(i=K1(i,16));j>3&&(i=K1(i,32));}c=a.s[b];c<0?i=K1(i,Zmb):c>0&&(i=K1(i,Ymb));g=a.j[b];switch(g){case 0:break;case 1:i=K1(i,Onb);break;case 2:i=K1(i,dnb);break;case 3:i=K1(i,enb);break;default:i=K1(i,fnb);}d=klb.mn(a,b);switch(d){case 0:break;case 1:i=K1(i,mnb);break;case 2:i=K1(i,nnb);break;case 3:i=K1(i,onb);break;default:i=K1(i,pnb);}h=a.o[b];h>0&&(i=K1(i,Pmb));h>1&&(i=K1(i,eob));}else {b<a.f&&klb.yr(a.p,b)?i=K1(i,Fob):b<a.f&&klb.sr(a.p,b)?i=K1(i,2):i=K1(i,4);(a.u[b]&dob)!=0?i=K1(i,Rmb):i=K1(i,Smb);j=(e=a.u[b]&job,e==0?0:e==Vmb?2:e==tnb?3:4);j==0?i=K1(i,112):j==2?i=K1(i,104):j==3?i=K1(i,88):i=K1(i,56);c=a.s[b];c==0?i=K1(i,Xmb):c<0?i=K1(i,Zmb):c>0&&(i=K1(i,Ymb));f=a.e[b]-klb.Rn(a,b)+klb.Mn(a,b);switch(f){case 0:i=K1(i,1792);break;case 1:i=K1(i,1664);break;case 2:i=K1(i,1408);break;default:i=K1(i,896);}g=a.j[b];switch(g){case 0:i=K1(i,3932160);break;case 1:i=K1(i,3801088);break;case 2:i=K1(i,anb);break;case 3:i=K1(i,bnb);break;default:i=K1(i,fnb);}d=klb.mn(a,b);switch(d){case 0:i=K1(i,hnb);break;case 1:i=K1(i,inb);break;case 2:i=K1(i,jnb);break;case 3:i=K1(i,knb);break;default:i=K1(i,pnb);}h=a.o[b];switch(h){case 0:i=K1(i,98304);break;case 1:i=K1(i,81920);break;default:i=K1(i,49152);}}return i;};klb.Ur=function Ur(a,b){var c;c=0;if(b<a.g&&(klb.xr(a.p,b)||a.H[b]==64)||a.H[b]==64)c|=8;else switch(klb.Hk(a,b)){case 4:case 0:c|=32;break;case 1:c|=1;break;case 2:c|=2;break;case 3:c|=4;break;case 5:c|=64;}(a.F[b]&64)!=0?c|=256:a.K||(c|=128);b<a.g&&klb.tr(a.p,b)?c|=cnb:a.K||(c|=Gob);return c;};klb.Vr=function Vr(a){var b,c,d,e,f,g,h,i,j,k;i=0;for(d=a,f=0,h=d.length;f<h;++f){b=d[f];b==-1&&++i;}if(i!=0){k=a;a=oW(elb.IX,omb,6,k.length-i,15,1);j=0;for(c=k,e=0,g=c.length;e<g;++e){b=c[e];b!=-1&&(a[j++]=b);}}ilb.rgb(a,ilb.Xgb(e2(llb.Ccb.prototype.kc,llb.Ccb,[])));return a;};klb.Wr=function Wr(a,b){var c,d,e,f,g,h,i,j;g=0;for(d=b,e=0,f=d.length;e<f;++e){c=d[e];c==-1&&++g;}j=oW(elb.IX,omb,6,b.length-g,15,1);i=0;for(h=0;h<b.length;h++){if(b[h]!=-1){j[i]=klb.Ut(a.g,h)<<16|klb.Ut(a.G,b[h]);++i;}}ilb.rgb(j,ilb.Xgb(e2(llb.Ccb.prototype.kc,llb.Ccb,[])));return j;};klb.Xr=function Xr(a,b){var c,d,e,f,g,h,i,j,k,l,m,n;f=false;for(e=0;e<2;e++){c=klb.zk(a.g,e,b);k=a.F[c];if(klb.xn(a.G,k)>2){m=klb.zk(a.g,1-e,b);i=Ylb;for(h=0;h<klb.xn(a.g,c);h++){d=klb.wn(a.g,c,h);d!=m&&i>d&&(i=d);}n=a.F[m];j=Ylb;for(g=0;g<klb.xn(a.G,k);g++){l=klb.wn(a.G,k,g);l!=n&&j>l&&(j=l);}a.F[i]!=j&&(f=!f);}}return f;};klb.Yr=function Yr(a,b,c,d){var e,f,g,h,i,j,k,l;h=a.e[d];i=h+1;while(i<a.w&&a.f[a.r[i]]==d)++i;for(k=h;k<i;k++)c[k]=-1;g=h;while(true){l=a.t[g]==-1?a.G.f:klb.jn(a.G,a.F[a.t[g]]);++c[g];if(c[g]==l){c[g]=-1;if(g==h)break;--g;if(!a.s[g]){b[a.F[a.r[g]]]=false;a.F[a.r[g]]=-1;}continue;}if(a.t[g]==-1){if(!b[c[g]]){if(klb.Kr(a,c[g],a.r[g])){a.F[a.r[g]]=c[g];b[c[g]]=true;++g;}}}else {if(klb.wn(a.G,a.F[a.t[g]],c[g])>=a.G.f){++c[g];continue;}f=klb.wn(a.G,a.F[a.t[g]],c[g]);if(a.s[g]){f==a.F[a.r[g]]&&klb.Lr(a,klb.yn(a.G,a.F[a.t[g]],c[g]),a.u[g])&&++g;}else {if(!b[f]){if(klb.Kr(a,f,a.r[g])&&klb.Lr(a,klb.yn(a.G,a.F[a.t[g]],c[g]),a.u[g])){b[f]=true;a.F[a.r[g]]=f;++g;}}}}if(g==i){if(klb.Rr(a,d)&&klb.Qr(a,d)&&klb.Pr(a,b,d)){for(j=h;j<i;j++){if(!a.s[j]){e=a.r[j];b[a.F[e]]=false;a.F[e]=-1;}}return true;}--g;if(!a.s[g]){b[a.F[a.r[g]]]=false;a.F[a.r[g]]=-1;}}}return false;};klb.Zr=function Zr(a){return klb.Sr(a,1,a.c)>0;};klb.$r=function $r(a,b){var c;for(c=0;c<b.length;c++)if(b[c]==a)return true;return false;};klb._r=function _r(a,b){var c,d,e;e=0;for(d=0;d<a.length;d++){c=a[d];while(b[e]<c){++e;if(e==b.length)return false;}if(b[e]>c)return false;}return true;};klb.as=function as(a,b){var c,d,e,f,g,h,i,j;g=false;if(klb.nn(a.g,b)==0){for(f=1;f<klb.xn(a.g,b);f++){for(h=0;h<f;h++){d=klb.wn(a.g,b,f);e=klb.wn(a.g,b,h);a.F[d]>a.F[e]^d>e&&(g=!g);}}}else {for(f=0;f<klb.xn(a.g,b);f++){c=klb.wn(a.g,b,f);j=0;i=oW(elb.IX,omb,6,3,15,1);for(h=0;h<klb.xn(a.g,c);h++){i[j]=klb.wn(a.g,c,h);i[j]!=b&&++j;}j==2&&a.F[i[0]]>a.F[i[1]]^i[0]>i[1]&&(g=!g);}}return g;};klb.bs=function bs(a,b){var c,d,e,f;e=0;f=0;while(e<a.length&&f<b.length){c=a[e];d=b[f];if(c==d)return true;c<d?++e:++f;}return false;};klb.cs=function cs(a,b){var c,d,e,f,g;if(!b||b.q==0||!b.K){a.g=null;return;}a.g=b;a.q=false;klb.Jt(a.g,1);a.M=7;for(d=0;d<a.g.f;d++)I1(n1(klb.pk(a.g,d),{l:snb,m:0,h:3}),0)&&(a.M=15);for(g=0;g<a.g.g;g++)(klb.Jk(a.g,g)&Qnb)!=0&&(a.M=15);a.K&&a.M!=7&&klb.Jt(a.G,a.M);a.o=0;a.p=0;a.C=oW(elb.g1,Imb,6,a.g.f,16,1);for(e=0;e<a.g.f;e++){a.C[e]=I1(n1(klb.pk(a.g,e),Jmb),0);a.C[e]&&++a.o;}a.d=0;a.f=null;if(a.o!=0){if(a.o!=0)for(f=0;f<a.g.g;f++)(a.C[klb.zk(a.g,0,f)]||a.C[klb.zk(a.g,1,f)])&&++a.p;for(c=0;c<a.g.q;c++)klb.Sl(a.g,c,a.C[c]);a.f=oW(elb.IX,omb,6,a.g.q,15,1);a.d=klb.Gn(a.g,a.f,true,false);}};klb.ds=function ds(a,b,c){klb.es(a,c);klb.cs(a,b);};klb.es=function es(a,b){if(!b||b.q==0){a.G=null;return;}a.G=b;a.K=false;klb.Jt(a.G,1);};klb.fs=function fs(a,b){if(!a.K){klb.is(a,b);a.K=true;}if(!a.q){klb.gs(a,b);klb.Nr(a);klb.Mr(a);a.q=true;}};klb.gs=function gs(a,b){var c,d,e,f,g,h,i,j,k,l;f=null;i=null;g=null;klb.Jt(a.g,a.M);a.n=oW(elb.IX,omb,6,a.g.f,15,1);for(d=0;d<a.g.f;d++)a.n[d]=klb.xn(a.g,d);if(a.o!=0){j=new klb.au(a.g.q,a.g.r);l=oW(elb.g1,Imb,6,a.g.q,16,1);for(e=0;e<a.g.q;e++)l[e]=!a.C[e];klb.$m(a.g,j,l,true,null);klb.Jt(j,a.M);klb.hs(a,j,b);f=a.i;i=a.k;g=a.j;k=0;for(c=0;c<a.g.f;c++)a.C[c]||(a.n[c]=klb.xn(j,k++));}klb.hs(a,a.g,b);if(a.o!=0){k=0;for(c=0;c<a.g.q;c++){if(!a.C[c]){a.i[c]=f[k];a.j[c]=g[k++];}}k=0;for(h=0;h<a.g.r;h++){!a.C[klb.zk(a.g,0,h)]&&!a.C[klb.zk(a.g,1,h)]&&(a.k[h]=i[k++]);}}};klb.hs=function hs(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s;p=b.f;a.i=oW(elb.JX,Inb,6,b.f,14,1);a.j=oW(elb.IX,omb,6,b.f,15,1);for(h=0;h<p;h++){a.i[h]=T1(n1(K1(klb.Tr(b,h),b.B[h]),Hob),Iob);a.j[h]=b.C[h];(c&1)!=0&&(a.j[h]+=b.s[h]+16<<8);(c&2)!=0&&(a.j[h]+=b.A[h]<<16);}a.A=oW(elb.JX,Inb,6,b.f,14,1);r=(klb.Jt(b,7),b.p);for(o=0;o<r.i.a.length;o++){n=false;for(j=llb.Bi(r.j,o),k=0,l=j.length;k<l;++k){m=j[k];if((b.G[m]&vnb)!=0){n=true;break;}}if(!n){s=llb.Bi(r.j,o).length;for(e=llb.Bi(r.i,o),f=0,g=e.length;f<g;++f){d=e[f];s==3?a.A[d]=K1(a.A[d],umb):s==4?a.A[d]=K1(a.A[d],vmb):s==5?a.A[d]=K1(a.A[d],wmb):s==6?a.A[d]=K1(a.A[d],xmb):s==7&&(a.A[d]=K1(a.A[d],ymb));}}}q=b.g;a.k=oW(elb.IX,omb,6,b.g,15,1);for(i=0;i<q;i++){a.k[i]=(klb.Ur(b,i)|b.G[i])&6291967^6291840;(c&4)!=0?(a.k[i]&2)!=0&&(a.k[i]|=8):(c&8)!=0&&(a.k[i]&2)!=0&&i<b.g&&klb.tr(b.p,i)&&(a.k[i]|=8);}};klb.is=function is(a,b){var c,d,e,f,g,h,i,j,k,l,m,n;klb.Jt(a.G,a.M);k=a.G.f;a.I=oW(elb.IX,omb,6,k,15,1);a.H=oW(elb.JX,Inb,6,k,14,1);for(g=0;g<k;g++){a.H[g]=T1(n1(K1(klb.Tr(a.G,g),klb.pk(a.G,g)),Hob),Iob);a.I[g]=klb.uk(a.G,g);(b&1)!=0&&(a.I[g]+=klb.dk(a.G,g)+16<<8);(b&2)!=0&&(a.I[g]+=klb.nk(a.G,g)<<16);}a.L=oW(elb.JX,Inb,6,k,14,1);m=klb.$n(a.G);for(j=0;j<m.i.a.length;j++){n=llb.Bi(m.j,j).length;for(d=llb.Bi(m.i,j),e=0,f=d.length;e<f;++e){h=d[e];n==3?a.L[h]=K1(a.L[h],umb):n==4?a.L[h]=K1(a.L[h],vmb):n==5?a.L[h]=K1(a.L[h],wmb):n==6?a.L[h]=K1(a.L[h],xmb):n==7&&(a.L[h]=K1(a.L[h],ymb));}}for(c=0;c<k;c++){n=klb.rn(a.G,c);n==0?a.L[c]=K1(a.L[c],tmb):n>7&&(a.L[c]=K1(a.L[c],zmb));}l=a.G.g;a.J=oW(elb.IX,omb,6,l,15,1);for(i=0;i<l;i++)a.J[i]=(klb.Ur(a.G,i)|klb.Jk(a.G,i))&6422527^6291840;};klb.js=function js(a,b,c,d,e,f,g){var h,i;h=klb.wn(a.g,a.r[b],d);if((!a.C[h]||a.f[h]==g)&&h!=a.t[b]){i=klb.yn(a.g,a.r[b],d);if(!f[i]&&!klb.cl(a.g,i)){a.r[++c]=h;a.t[c]=a.r[b];a.u[c]=i;f[i]=true;e[h]?a.s[c]=true:e[h]=true;}}return c;};klb.ks=function ks(){this.c=8;this.N=new llb.qfb(new Alb.VT());};klb.ls=function ls(){this.c=1;this.N=new llb.qfb(new Alb.VT());};klb.ms=function ms(a,b){var c;c=oW(elb.IX,omb,6,b,15,1);flb.W5(a,0,c,0,$wnd.Math.min(a.length,b));return c;};klb.ns=function ns(a,b){var c;c=oW(elb.g1,Imb,6,b,16,1);flb.W5(a,0,c,0,$wnd.Math.min(a.length,b));return c;};b2(63,1,{},klb.ks,klb.ls);_.c=0;_.d=0;_.o=0;_.p=0;_.q=false;_.v=0;_.w=0;_.K=false;_.M=0;_.O=false;elb.sY=D3(63);klb.os=function os(){};b2(168,1,{168:1},klb.os);_.a=0;_.b=0;_.c=0;_.d=0;elb.qY=D3(168);klb.rs=function rs(){klb.rs=d2;klb.ps=wW(iW(elb.u_,1),dmb,2,6,['QM@HzAmdqjF@','RF@Q``','qC`@ISTAlQE`','`J@H','QM@HzAmdqbF@','qC`@ISTAlQEhqPp@','sJP@DiZhAmQEb','RF@QPvR@','QM@HzA@','qC`@ISTAlQEhpPp@','qC`@Qz`MbHl','sJP@DiZhAmQEcFZF@','RFPDXH','qC`@IVtAlQE`','QM@HvAmdqfF@','sGP@DiVj`FsDVM@','`L@H','sJP@DizhAmQEcFBF@','sJP@DjvhAmQEb','sFp@DiTt@@AlqEcP','sGP@LdbMU@MfHlZ','QMHAIhD','QM@HzAy@','sJP@DkVhAmQEb','sNp@DiUjj@[\\QXu`','sJP@DiZhAmQEcFBF@','sGP@DjVj`FsDVM@','RFPDTH','RG@DXOH@','sGP@Divj`FsDVMcAC@','sGP@Dj}j`FsDVM@','qC`@Qz`MbHmFRF@','sNp@LdbJjj@[\\QXu`','QMHAIhGe@','QM@HzAyd`','QM`AIhD','qC`@ISTA@','sGP@DkUj`FsDVM@','qC`@IVtAlQEhqPp@','sNp@DiUjj@[\\QXuqea`@','KAx@@IRjuUPAlHPfES\\','QM`BN`P','sJP@DjZhAmQEcFJF@','Hid@@DjU^nBBH@FtaBXUMp`','sNp@Diujj@[\\QXuq`a`@','sJP@DjvhAmQEcFZF@','sJP@DjZhAmQEcFFF@','sOp@DjWkB@@FwDVM\\YhX@','sNp@Dj}Zj@[\\QXu`','sNp@DiWjj@[\\QXuq`a`@','sOp@DjWkB@@D','KAx@@ITouUPAlHPfES\\','KAx@@YIDTjjh@vDHSBin@','sNp@DkUZj@[\\QXu`','RFPDXOH@','QM`BN`^L`','qC`@ISTAy@','sGP@LdbMU@MfHl[FVF@','qCb@AIZ`H','KAx@@IRjuUPAlHPfES]FFa`@','KAx@@ITnuUPAlHPfES\\','HiD@@DiUVjj`AmHPfES\\H','sNp@DjUjj@[\\QXu`','sJP@DkVhAmQEcFJF@','sGP@DjVj`FsDVMcCC@','qC`@Qz`MbHmFBF@','sJP@DkfhAmQEb','qC`@IVtAlQEhsPp@','sGP@Djuj`FsDVM@','sGP@Dj}j`FsDVMcMC@','sJP@DiZhA@','KAx@@ISjuUPAlHPfES]F@a`@','sJP@DjZhAmQEcFRF@','KAx@@IRnuUPAlHPfES]F@a`@','HiD@@DjWvjj`AmHPfES\\H','QMHAIhGd@','sNp@DiUjj@[\\QXuq`a`@','KAx@@IVjmUPAlHPfES\\','sGP@DjVj`FsDVMcMC@','QM`AIhGe@','HiD@@LdbJRjjh@[RDIaTwB','qCp@AIZ`H','sGP@LdbMU@MfHl[FFF@','QMDARVA@','sNp@LdbJjj@[\\QXuqba`@','sNp@LdbJjj@[\\QXuqca`@','sGP@Dkej`FsDVM@','qCb@AIZ`OI@','HaD@@DjUZxHH@AlHPfES]FLa`@','sGP@DkYj`FsDVM@','qCb@AIV`H','sNp@LdbJjj@[\\QXuqea`@','sGP@DkUj`FsDVMcEC@','sFp@DiTt@@Axa@','Hmt@@DjU_ZxHHj@AmhPfES\\Lj','QM`BN`^P','qCb@AIZ`OH`','sFp@DiTt@@AxaP','sGP@Djuj`FsDVMcEC@','sGP@Djuj`FsDVMcIC@','sGP@DkUj`FsDVMcKC@','sJP@DkfhAmQEcFRF@','sGP@DjVj`FsDVMcIC@','HaD@@DjUZxHH@AlHPfES]FFa`@','qC`@IRtDVqDV@','sNp@Dj}Zj@[\\QXuqfa`@','KAx@@ITnuUPAlHPfES]FFa`@','HiD@@DkUUjj`AmHPfES\\H','sJQ@@dkU@H','qC`@Qz`H','KAx@@IUkmUPAlHPfES\\','KAx@@ITouUPAlHPfES]FJa`@','sJP@H~j@[TQX`','sGP@DjZj`FsDVM@','sJP@DkVhAmQEcFFF@','sJX@@eKU@H','sJP@DizhAy@','QMHAIhGbP','KAx@@ITouUPAlHPfES]FNa`@','HaD@@DjUZxHD@AlHPfES\\','HaD@@DjUZxHH@A@','sNp@LdbJjj@[\\QXuqaa`@','Hed@@LdbRQUUUP@vTHSBinFP','KAx@@ITouUPAlHPfES]FLa`@','sNp@DkUZj@[\\QXuqba`@','KAx@@ITjuUPAlHPfES]FNa`@','KAx@@YIDTjjh@vDHSBincGPp@','HaD@@DjYvxH`@AlHPfES]FLa`@','RF@QP`','qCb@AIj`H','sNp@DjUjj@[\\QXuqaa`@','sNp@DkVZj@[\\QXu`','KAx@@YIDUJjh@vDHSBin@','sGP@DkYj`FsDVMcIC@','sGP@DjVj`FsDVMcAC@','sGP@DiVj`D','sJP@DkVhAmQEcFZF@','sNp@LdbLjj@[\\QXu`','QM@HvAmdqbF@','HaD@@DjWjXHB@AlHPfES\\','sNp@DjwZj@[\\QXuqba`@','sNp@LdbJjj@[\\QXuqda`@','sFp@DiTt@@Axa`','HiD@@Djuujj`AmHPfES\\H','sNp@DkUZj@[\\QXuqca`@','sJP@DiZhAy@','KAx@@YIDTjjh@vDHSBincCPp@','KAx@@IWNmUPAlHPfES\\','KAx@@IVkMUPAlHPfES\\','sJQ@@dju@H','qCb@AIZ`OH@','qC`@ISTAxa@','sNp@DjyZj@[\\QXu`','Hid@@DjUfaBB`@FtaBXUMp`','HiD@@DiUVjj`AmHPfES\\LXBF@','KAx@@IUjmUPAlHPfES\\','HiD@@DjWvjj`AmHPfES\\LXjF@','sJP@DjVhAmQEb','qCb@AIV`OH`','HiD@@LdbJRjjh@[RDIaTwCFDa`@','KAx@@YIDTjjh@vDHSBinc@Pp@','sNp@DjUjj@[\\QXuqda`@','qC`@Qz`OED','sJP@DkfhAmQEcFZF@','KAx@@YIDbjjh@vDHSBincDPp@','sGP@Djyj`FsDVMcMC@','KAx@@IVrmUPAlHPfES\\','qCp@AIZ`OI@','sJX@@dkU@H','sJQ@@dkU@OH`','sNp@Di]ZjBBvxbqk@','Hkl@@DjU_Uk``bj`@[VDIaTwCJzX','sGP@DjZj`FsDVMcEC@','Hid@@DjU^nBBH@FtaBXUMpqcHX@','sNp@DkeZj@[\\QXu`','sNp@DjYjj@[\\QXuqca`@','sGQ@@djuT@`','HiD@@LdbJTjjh@[RDIaTwB','sOp@DjWkB@@Gd`','HeT@@LdbbRKBDQD@CYPaLJfxY@','qCr@XIKTA@','HiD@@DjW^jj`AmHPfES\\LXJF@','HeT@@DjU]k``b`@[JDIaTwCH','sGP@Djuj`FsDVMcCC@','`IH`B','sOp@DjWkB@@GdX','sJQ@@eKU@H','KAx@@YIDUJjh@vDHSBincBPp@','sJX@@eKU@OH@','KAx@@YIDTjjh@vDHSBincAPp@','sOq@@drm\\@@@`','KAx@@IUkMUPAlHPfES\\','qCp@AIj`H','Hed@@DjUUjjj@FraBXUMpr','sGX@@eJuT@`','sGP@DkUj`FsDVMcCC@','HiD@@Dj}Ujj`AmHPfES\\LXrF@','KAx@@ITouUPAlHPfES]FHa`@','Hed@@DjWujjj@FraBXUMpsFIa`@','sGP@DiUj``mfHlZ','sFp@DiTvjhAlqEcP','Hid@@DjU^nBBH@FtaBXUMpq`XX@','sJP@DkVdAmQEb','qCp@AIZ`OH`','QMhDRVA@','qC`@ISJAlQE`','qCp@BOTAyhl','sJX@@eOU@ODB','sFp@DiTt@@AyaB','sGP@DkUj`FsDVMcMC@','Hid@@DjYUaBH`@FtaBXUMpqcHX@','qC`@Qz`OH@','HiD@@DjUVjj`AmHPfES\\LXZF@','sJP@H~j@[TQXqda`@','sJX@@eKU@OI@','sNp@Djejj@[\\QXu`','sJQ@@dsU@H','sJQ@@dkU@OI`','KAx@@YIMDVjh@vDHSBin@','Hid@@DjU^nBBD@FtaBXUMp`','sNp@DkgZj@[\\QXuqca`@','qC`@IRtDVqDVcEC@','Hed@@LdbRQeUUP@vTHSBinFP','sNp@DiUjj@P','qC`@IRtDT','sNp@DkYZj@[\\QXuqca`@','KAx@@IUkmUPAlHPfES]FDa`@','KAx@@IVjmUPAlHPfES]FNa`@','sOx@@drm\\@@@`','KAx@@ITjuUPAlHPfES]FBa`@','QMDARVAyH','sJP`@dfvhA@','HeT@@DjU_k``b`@[JDIaTwCLXfF@','KAx@@IToUUPAlHPfES]FJa`@','sGP@DkYj`FsDVMcEC@','qCb@AIZ`ODH','`I@`B','KAx@@IUzmUPAlHPfES]FFa`@','sNp@DkfZj@[\\QXu`','KAx@@ITnuUPAlHPfES]F@a`@','HiD@@LddURjjh@[RDIaTwB','sNp@Dj~Zj@[\\QXuqfa`@','Hed@@Dj{uZjj@FraBXUMpr','KAx@@ITsUUPAlHPfES\\','Hid@@LdbRQk``b@AmHPfES\\LXrF@','sOp@DjWkB@@GdH','sJQ@@dkU@OH@','Hid@@DjU^nBBH@FtaBXUMpqahX@','sGP@DiYj``mfHlZ','KAx@@IToUUPAlHPfES]FLa`@','qCp@AJZ`ODH','Hmt@@DjU]ZxHHj@AmhPfES\\Lj','sGP@DkUjPFsDVM@','qC`@IVtA@','Hed@@LdbJReUUP@vTHSBinFP','sNp@DjuZj@[\\QXuqea`@','KAx@@IUkmUPAlHPfES]FNa`@','HiD@@DkVUjj`AmHPfES\\H','Hed@@DkUeZjj@FraBXUMpr','sNp@DkVZj@[\\QXuqea`@','sJP@DiVhHKZbKFLLL@','HiD@@Djuyjj`AmHPfES\\H','sNp@DjUjj@[\\QXuq`a`@','HeT@@DjYUXPbH`@[JDIaTwCH','HiD@@DjwUjj`AmHPfES\\LXRF@','sNq@@djmUPB','KAx@@YIEEZjh@vDHSBincCPp@','sGP@Di^V`dmfHlZ','Hid@@DjYUaBHP@FtaBXUMp`','sNp@DjYjj@[\\QXuqba`@','sGP@Dkej`FsDVMcKC@','HeT@@DjU^k``b`@[JDIaTwCH','qC`@Qv`MbHmFBF@','sGQ@@djmT@`','qCr@XIKTAyH','qC`@IVtAlQEhpPp@','Hid@@LdbbQxXF@@AmHPfES\\LXjF@','sGP@DkYj`FsDVMcCC@','KAx@@IVsMUPAlHPfES\\','qCp@AIj`ODl','HiD@@DkeUjj`AmHPfES\\H','deT@@DjU[k``b`@vTHSBinFP','sJP@DkVdAmQEcFRF@','HiD@@LdbJTjjh@[RDIaTwCFDa`@','HiD@@DkYyjj`AmHPfES\\H','sJP@DjZhAyH','KAx@@IVkMUPAlHPfES]FDa`@','sJX@@dkU@OI@','Hed@@LdbRQUUUP@vTHSBinFXpLL@','Hed@@DjuUZjj@FraBXUMpr','sGP@Djfj`FsDVMcKC@','sNp@DkVZj@[\\QXuqba`@','sNp@DjyZj@[\\QXuqfa`@','qCb@AIj`OH@','sNp@DjUZj@[\\QXu`','KAx@@IWOMUPAlHPfES\\','Hid@@DjU^nBBH@D','Hed@@DjuvZjj@FraBXUMpr','sJP@DiVhHKZbKFLtL@','dmt@@DjU_ZxHHj@C[PaLJfxYVLKC@','sNp@DjuZj@[\\QXuqca`@','sJP@DkfhAmQEcFJF@','sNp@LdbJZj@[\\QXu`','HeT@@DjU_k``b`@[JDIaTwCLXFF@','KAx@@IVlmUPAlHPfES]FNa`@','HeT@@LdbbRKBDQD@CYPaLJfxYcEPp@','Hid@@DjUZnBBH@FtaBXUMpqcHX@','qCa@CIKTA@','HiD@@Dj~]jj`AmHPfES\\LXFF@','sKP@Di\\Zj@[TQX`','sGP@Djfj`FsDVMcEC@','HiD@@DkgYjj`AmHPfES\\H','sNp@DjuZj@[\\QXuqaa`@','KAx@@YIMDVjh@vDHSBincDPp@','sJP@DjVhHKZbKFLTL@','Hid@@LdbRQk``b@AmHPfES\\LXZF@','HiD@@Dj}Ujj`AmHPfES\\LXzF@','HeT@@DjU_k``bP@[JDIaTwCH','sNp@DkUZi@[\\QXu`','HiD@@DjYfjj`AmHPfES\\H','sGP@DjZj`FsDVMcAC@','Hmt@@DjU_jxHHj@AmhPfES\\Lj','Hid@@LdbRQk``R@AmHPfES\\H','KAx@@YIDUJjh@vDHSBincDPp@','qCr@XIKTAyD','sOq@@drm\\@@@|`@','Hed@@DjW^jjj@FraBXUMpsFBa`@','HeT@@DjY]zXFB@@[JDIaTwCH','Hkl@@DjU_Vk``bj`@[VDIaTwCJzX','Hid@@DjY}nBHH@FtaBXUMpqcHX@','sGX@@eKuT@|d@','sGP@Dj^Y`FsDVM@','HcL@@DjU_ZnBBJh@FqaBXUMprn`','sJP@DkVdAmQEcFJF@','sOq@@drm\\@@@|b@','sNp@DjyZj@[\\QXuqaa`@','HaD@@DjUZxHH@AyD@','qC`@Qv`H','dmt@@DjU_ZxHHj@C[PaLJfxYVLYC@','sGP@Dkej`FsDVMcMC@','Hed@@DjUUjjj@FraBXUMpsFHa`@','HeT@@LdbbRkBDQD@CYPaLJfxY@','KAx@@IU{MUPAlHPfES]FLa`@','RG@DTH','sJY@DDeVhA@','KAx@@YIDUJjh@vDHSBinc@Pp@','sJX@@dkU@OI`','sJQ@@dju@OI`','HeT@@LdbbRKBDQD@CYPaLJfxYcFPp@','sFp@DiTvjhAlqEcXpPp@','HaD@@DjUZxHH@AyG@','sNx@@eJ}UPB','sNp@LddUjj@[\\QXuqca`@','HaDH@@RVU[j@@@D','sNp@DkgZi@[\\QXu`','sGY@LDeVj`D','sNp@LdbJfZBZvxbqk@','sJP`@dfvhAyL','sGX@AddQjhAxe`','Hmt@@DjU_ZxHHj@AmhPfES\\LkFIa`@','qCh@CIKTA@','sNp@LdbLjj@[\\QXuq`a`@','sOq@@drm\\@@@|a@','KAx@@IUzmUPAlHPfES]FJa`@','sNx@AddQUUPB','sGP@Di]jP`mfHlZ','sJP`@TeZhA@','KAx@@IRjmUPHKXPaLJfx','HeT@@LdbRTM\\DDT@CYPaLJfxY@','HaF@@@Rfu[j@@@D','Hid@@DjYUaBH`@FtaBXUMpqchX@','KAx@@IUjmTpAlHPfES\\','Hid@@DjU^nBBD@FtaBXUMpqcHX@','sGP@DiUj``mfHl[FFF@','KAx@@IUvmUPAlHPfES]FLa`@','Hed@@LdbQTUUUP@vTHSBinFXqDL@','sJP@DkVhA@','sOx@@drm\\@@@|b@','KAx@@IUkMUPAlHPfES]FDa`@','HeT@@LdbRQU\\DDT@CYPaLJfxY@','HiD@@Dj}Yjj`AmHPfES\\LXrF@','HiD@@Dj{ujj`AmHPfES\\LXFF@','KAx@@IWNmUPAlHPfES]FFa`@','KAx@@IRkMUPHKXPaLJfx','sJP@DjYdAmQEcFZF@','sJY@LDeZhAyL','HaDH@@RVU[f@@@D','sJP`@deVhAyB','HaD@@DjWjZjj`AlHPfES\\','sGP@DkYj`FsDVMcMC@','sNp@DkgZj@[\\QXuqea`@','sJQ@@dlu@H','HeT@@DjU]k``b`@[JDIaTwCLXrF@','sJX@@dkU@OH`','RFDDQFCr`','sJP@DiYXIKZbKFLLL@','KAx@@YIHjjjh@vDHSBincGPp@','Hk\\@@DjU^ukmLHH@@@AmXPfES\\Lki`','sGQ@@djmT@|b@','Hid@@DjUfaBB`@FtaBXUMpqahX@','sNx@@eRmUPB','Hmt@@LdbRVak``ah@FvaBXUMprh','qCr@XIJtA@','KAx@@IWMmUPAlHPfES]FNa`@','HeT@@DjYYZPbJ@@[JDIaTwCH','sNp@DkfZj@[\\QXuqea`@','Hid@@DjU^nBAHAEVtaBXUMp`','dmt@@DjUgZDHJJ@C[PaLJfxYT','sGP@DkejPFsDVM@','sNx@@eJmUPB','qCb@AIf`H','HcL@@DjU_VnBBJh@FqaBXUMprnqcXX@','Hid@@DjUZnBBH@FtaBXUMpqahX@','sNp@LdbQZjBBvxbqkcGC@','sOx@@drm\\@@@|c@','sJP@H~j@^R@','KAx@@YIDcFjhDElHPfES\\','Hid@@DjUZnBAH@FtaBXUMp`','sNp@LddUji@[\\QXu`','sGP@DjfjPFsDVM@','HeT@@DjYUXPbD`@[JDIaTwCH','KAx@@IUoMUPAlHPfES]FDa`@','sFp@DiTt@@AyaD','Hed@@DjuuZjj@FraBXUMpsFIa`@','HeT@@DjUghP`h`@[JDIaTwCLXfF@','sOp@DjWkjj`FwDVM\\YhX@','sGP@Djfj`FsDVMcIC@','KAx@@IRkmUPHKXPaLJfzL]C@','sNx@@djmUPB','QM`AIdD','sOp@DjWkB@@Gbe@','sNp@DjyZj@[\\QXuqca`@','QM@HuAmd`','sNp@LddUjj@[\\QXuqea`@','daD@@DjUZxHD@CXPaLJfzLYC@','qCb@AIZPH','HiD@@LdbJTjjh@[RDIaTwCF@a`@','Hmt@@DjU_ZxHHi@AmhPfES\\Lj','HaDH@@RYWih@H@D','HiD@@LdbJTjjh@[RDIaTwCFHa`@','sGX@@djuT@|a@','sNp@DkfZj@[\\QXuqaa`@','Hid@@DjU^nBBH@GdL','KAx@@IVkMUPAlHPfES]FJa`@','qCr@XIKTAy@','HmT@@Dj{uVjjh@[ZDIaTwCJqaXX@','dmt@@DjYWVDHbJ@C[PaLJfxYT','Hif@@@RUe^Fh@@@P','HaDH@@Rfu[j@@@GdH','KAx@@IVsMUPAlHPfES]FDa`@','sKP@Di\\Zj@[TQXq`a`@','sJX@@eMU@OH@','HeT@@DjU^k``b`@[JDIaTwCLXFF@','Hmt@@LdbbRJXPbHh@FvaBXUMprh','sJP@DjvhAmQEcFBF@','dmt@@LdbbRNXPbHh@MmBDpj[aeXplL@','sJP`@dfvhAyD','sGP@Di^V`dmfHl[FVF@','KAx@@IVsmUPAlHPfES]FBa`@','sOq@@drm\\@@@|PP','sJY@BDeZhA@','HeT@@LdbRbmBDED@CYPaLJfxY@','Hed@@Djy[Zjj@FraBXUMpr','HeT@@DjU]k``b`@[JDIaTwCLXFF@','Hid@@DjUfaBB`@D','qCa@CIJtA@','QMPARVA@','Hid@@DjUfaBB`@FtaBXUMpqcHX@','sJY@BDfZhA@','HeT@@DjUghP`hP@[JDIaTwCH','Hed@@Dj{uZjj@FraBXUMpsFIa`@','dmt@@DjYUvDHbJ@C[PaLJfxYVLUC@','sNp`@dfuZj@P','sJQ@@dmU@OH@','sJX@@dmU@H','HeT@@DjU]k``b`@[JDIaTwCLXZF@','HiD@@LdfbJZjh@[RDIaTwCFAa`@','sOx@@drm\\@@@|a@','deT@@DjUfhP`h`@vTHSBinFP','dmt@@DjU]ZxHHj@C[PaLJfxYVLKC@','sOp@DjWkjj`FwDVM\\XHX@','HcL@@LdbbRNSBDQEP@McBDpj[ae]cFpp@','HiD@@Dj}Yji`AmHPfES\\H','HaDH@@RYe[hB@@D','Hid@@DjU^njjj@FtaBXUMpq`XX@','deT@@LdbbRKBDQB@FraBXUMpr','QMPARZA@','sOq@@drm\\@@@|QX','HaD@@DjYvxH`@A@','HcL@@LdbbRNcBDQEP@McBDpj[ae]@','QMhDRZA@','RG@DXLHmP','QM`BN`XQYd','RG@DTLHmP','QMHAIXFEVd','QMDARVAaH','RFPDXLHmP','RF@Q`vRbdLEC@','RF@QpvR@','QO@HyjAmd`','`II@B','`II@CFspqJp','`II@CF[@hM@prB`','`H@[T[|B`XN@PdM@p|@bHrBcDk@','RG@DXMj}F@','QM`BN`[L~b@','RG@DTMj}D@','QMHAIXFt~j@','QMDARVA}L@','RFPDXMj}D@','sKP@Di\\YZ@[TQXqaa`@','eF`BHD']);};klb.ss=function ss(a,b){var c,d;if(!b)return null;d=oW(elb.IX,omb,6,(klb.ps.length+31)/32|0,15,1);b=klb.xs(b);klb.es(a.e,b);for(c=0;c<klb.ps.length;c++){klb.cs(a.e,klb.qs[c]);klb.Sr(a.e,1,4)>0&&(d[c/32|0]|=1<<31-c%32);}return d;};klb.ts=function ts(a,b){var c,d;if(!b)return null;d=oW(elb.JX,Inb,6,(klb.ps.length+63)/64|0,14,1);b=klb.xs(b);klb.es(a.e,b);for(c=0;c<klb.ps.length;c++){klb.cs(a.e,klb.qs[c]);klb.Sr(a.e,1,4)>0&&(d[c/64|0]=K1(d[c/64|0],L1(1,63-c%64)));}return d;};klb.us=function us(){var a,b;if(klb.qs==null){b=new klb.rp(false);klb.qs=oW(elb.CY,Job,25,klb.ps.length,0,1);for(a=0;a<klb.ps.length;a++){klb.qs[a]=klb.gp(b,klb.ps[a]);klb.Jt(klb.qs[a],1);}}};klb.vs=function vs(a){var b;if(a.d!=null){for(b=0;b<a.d.length;b++)if((a.b[b]&~a.d[b])!=0)return false;}else {return false;}return klb.ws(a);};klb.ws=function ws(a){!a.c&&(a.c=(new klb.rp(false),null));!a.a&&(a.a=(new klb.rp(false),null));klb.es(a.e,a.c);klb.cs(a.e,a.a);return klb.Zr(a.e);};klb.xs=function xs(a){var b,c;if(a.K){for(b=0;b<a.q;b++){if(I1(n1(a.B[b],Jmb),0)){a=new klb.bu(a);for(c=b;c<a.q;c++)I1(n1(a.B[c],Jmb),0)&&(a.C[c]=-1);klb.Zj(a);}}}return a;};klb.ys=function ys(a,b,c){a.a=b;c==null?a.b=klb.ss(a,b):a.b=c;};klb.zs=function zs(a,b,c){a.c=b;c==null?a.d=klb.ss(a,b):a.d=c;};klb.As=function As(){klb.rs();this.e=new klb.ks();klb.us();};klb.Bs=function Bs(a){klb.rs();var b,c,d,e,f,g;if(a==null)return null;b=oW(elb.EX,Nnb,6,a.length*8,15,1);for(d=0;d<a.length;d++){g=a[d];for(e=7;e>=0;e--){c=g&15;c>9&&(c+=7);b[d*8+e]=48+c<<24>>24;g>>=4;}}return flb.h5((f=b,Klb.v8(),f));};klb.Cs=function Cs(a){klb.rs();var b,c,d,e;if(ilb.Xgb(a).length==0||(ilb.Xgb(a).length&7)!=0)return null;d=oW(elb.IX,omb,6,ilb.Xgb(a).length/8|0,15,1);for(c=0;c<ilb.Xgb(a).length;c++){e=c/8|0;b=(ilb.Sgb(c,ilb.Xgb(a).length),ilb.Xgb(a).charCodeAt(c)-48);b>16&&(b-=7);d[e]<<=4;d[e]+=b;}return d;};klb.Ds=function Ds(a,b){klb.rs();var c,d,e,f;f=0;d=0;e=0;for(c=0;c<a.length;c++){f+=p4(a[c]&b[c]);d+=p4(a[c]);e+=p4(b[c]);}return f/$wnd.Math.sqrt(d*e);};klb.Es=function Es(a,b){klb.rs();var c,d,e;e=0;c=0;for(d=0;d<a.length;d++){e+=p4(a[d]&b[d]);c+=p4(a[d]|b[d]);}return e/c;};b2(132,1,{},klb.As);elb.rY=D3(132);klb.Gs=function Gs(a){a.j=1;a.k=10;a.n=400;a.f=400;a.i=true;a.d='black';a.b=new llb.Qi();a.a=new llb.Qi();a.c=new flb.Q5();a.e=new Glb.s2(12);};klb.Hs=function Hs(a,b){var c,d;c=flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(3*a.j),(Jlb.m8(),Jlb.i8))));d='<line stroke-dasharray="'+c+','+c+Bmb+'x1="'+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(b.a),Jlb.i8)))+Bmb+Cmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(b.c),Jlb.i8)))+Bmb+Dmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(b.b),Jlb.i8)))+Bmb+Emb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(b.d),Jlb.i8)))+Bmb+Fmb+a.d+Bmb+'stroke-width:'+a.j+Hmb;klb.Ss(a,d);};klb.Is=function Is(a,b){var c,d;d=new flb.S5('<polygon points="');for(c=0;c<b.a;c++){flb.L5(d,y1($wnd.Math.round(b.b[c])));d.a+=',';flb.L5(d,y1($wnd.Math.round(b.c[c])));d.a+=' ';}flb.N5(d,'" fill="'+a.d+Bmb+Fmb+a.d+Bmb+Gmb+a.j+Hmb);klb.Ss(a,d.a);};klb.Js=function Js(a,b,c,d){var e,f,g,h;g=(e=(h=Glb.q2(a.e,b),new Hlb.w2(h)).b,e);f='<text x="'+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(c-g/2),(Jlb.m8(),Jlb.i8))))+Bmb+'y="'+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(d+a.k/3),Jlb.i8)))+Bmb+'stroke="none" '+'font-size="'+a.e.b+Bmb+'fill="'+a.d+'">'+b+'<\/text>';klb.Ss(a,f);};klb.Ks=function Ks(a,b,c,d){var e;e='<circle cx="'+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(b+d/2),(Jlb.m8(),Jlb.i8))))+Bmb+'cy="'+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(c+d/2),Jlb.i8)))+Bmb+'r="'+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(d/2),Jlb.i8)))+Bmb+'fill="'+a.d+Hmb;klb.Ss(a,e);};klb.Ls=function Ls(a,b,c,d){var e,f;e=klb.mk(a.K,b);f='<circle id="'+(a.g!=null?a.g:'mol'+klb.Fs)+':Atom:'+b+Bmb+Kob+(e==0?'':'data-atom-map-no="'+e+Bmb)+'cx="'+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(c),(Jlb.m8(),Jlb.i8))))+Bmb+'cy="'+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(d),Jlb.i8)))+Bmb+'r="'+8+Bmb+Lob;llb.wi(a.a,f);};klb.Ms=function Ms(a,b,c,d,e,f){var g;g='<line id="'+(a.g!=null?a.g:'mol'+klb.Fs)+':Bond:'+b+Bmb+Kob+'x1="'+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(c),(Jlb.m8(),Jlb.i8))))+Bmb+Cmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(d),Jlb.i8)))+Bmb+Dmb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(e),Jlb.i8)))+Bmb+Emb+flb.V3(Jlb.n6(Jlb.k6(new Jlb.o6(f),Jlb.i8)))+Bmb+Gmb+8+Bmb+Lob;llb.wi(a.b,g);};klb.Ns=function Ns(a,b){a.j=Q1(y1($wnd.Math.round(100*$wnd.Math.max(b,1))))/100;};klb.Os=function Os(a,b){a.d='rgb('+((b&Dnb)>>16)+','+((b&qmb)>>8)+','+(b&255)+')';};klb.Ps=function Ps(a,b){if(a.k!=b){a.k=b;a.e=new Glb.s2(b);}};klb.Qs=function Qs(a,b){a.n=R1(y1($wnd.Math.round(b.b)));a.f=R1(y1($wnd.Math.round(b.a)));return klb.Nd(a,b);};klb.Rs=function Rs(a){var b,c,d,e,f,g;f='<svg id="'+(a.g!=null?a.g:'mol'+klb.Fs)+Bmb+'xmlns="http://www.w3.org/2000/svg" version="1.1" '+'width="'+a.n+'px" '+'height="'+a.f+'px" '+'viewBox="0 0 '+a.n+' '+a.f+'">\n';g=a.i?'<style> #'+(a.g!=null?a.g:'mol'+klb.Fs)+' { pointer-events:none; }'+' #'+(a.g!=null?a.g:'mol'+klb.Fs)+' .event '+' { pointer-events:all; }'+' line { stroke-linecap:round; }'+' polygon { stroke-linejoin:round; }'+' <\/style>\n':'<g style="font-size:'+a.k+'px; fill-opacity:1; stroke-opacity:1; fill:black; stroke:black;'+' font-weight:normal; text-rendering:optimizeLegibility; font-family:sans-serif;'+' stroke-linejoin:round; stroke-linecap:round; stroke-dashoffset:0;">';f+='  ';f+=g;if(a.i){for(e=new llb.Wbb(a.b);e.a<e.c.a.length;){d=llb.Vbb(e);klb.Ss(a,d);}for(c=new llb.Wbb(a.a);c.a<c.c.a.length;){b=llb.Vbb(c);klb.Ss(a,b);}}a.i||klb.Ss(a,'<\/g>');return f+a.c.a+'<\/svg>';};klb.Ss=function Ss(a,b){flb.N5(a.c,'  ');flb.N5(a.c,b);flb.N5(a.c,Slb);};klb.Ts=function Ts(a,b,c){klb.Qc();klb.Qd.call(this,a,b);klb.Gs(this);this.g=c;++klb.Fs;};b2(212,245,{},klb.Ts);_.lb=function Us(){return klb.Rs(this);};_.f=0;_.i=false;_.j=0;_.k=0;_.n=0;klb.Fs=0;elb.tY=D3(212);klb.Vs=function Vs(a,b){var c;if(a.c!=null)for(c=0;c<a.c.length;c++)if(b==a.c[c]&&a.d[c])return true;return false;};klb.Ws=function Ws(a,b,c,d,e){this.a=a;this.b=b;this.g=c;this.f=d;this.e=e;};b2(140,1,{140:1},klb.Ws);_.a=0;_.b=0;_.e=false;_.f=false;_.g=0;elb.uY=D3(140);klb.Xs=function Xs(a,b){var c,d,e,f;f=false;a.b=b;klb.Jt(a.b,15);c=a.b.f;d=a.b.g;a.j=oW(elb.g1,Imb,6,d,16,1);for(e=0;e<d;++e)a.j[e]=false;a.g=oW(elb.g1,Imb,6,c,16,1);a.c=oW(elb.IX,omb,6,c,15,1);for(e=0;e<c;++e){a.g[e]=false;a.c[e]=-1;}a.e=oW(elb.u_,dmb,2,3*c,6,1);a.i=0;a.d=0;a.a=0;while(!f){for(e=0;e<c;++e){if(!a.g[e]){a.a>0&&(a.e[a.i++]='.');klb.Zs(a,e,-1);++a.a;break;}}e==c&&(f=true);}a.f='';for(e=0;e<a.i;++e)a.f+=''+a.e[e];return a.f;};klb.Ys=function Ys(a){switch(a){case 5:case 6:case 7:case 8:case 9:case 15:case 16:case 17:case 35:case 53:return true;default:return false;}};klb.Zs=function Zs(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p;d=true;i=0;p=0;m=a.i;a.c[b]=m;h=klb.uk(a.b,b);g=klb.jk(a.b,b);e=klb.dk(a.b,b);f=klb.nk(a.b,b);k=klb.xn(a.b,b);e==0&&f==0&&klb.Ys(h)&&(d=false);a.e[m]='';if(c!=-1){switch(klb.Hk(a.b,c)){case 0:a.e[m]+='~';break;case 2:a.e[m]+='=';break;case 3:a.e[m]+='#';}}d&&(a.e[m]+='[');f!=0&&(a.e[m]+=''+f);a.e[m]+=''+g;if(d){if(0<(o=klb.Mn(a.b,b))){a.e[m]+='H';1<o&&(a.e[m]+=o);}}if(e!=0){e>0?a.e[m]+='+':a.e[m]+='-';$wnd.Math.abs(e)>1&&(a.e[m]+=''+$wnd.Math.abs(e));}d&&(a.e[m]+=']');c!=-1&&(a.j[c]=true);a.g[b]=true;++a.i;for(n=0;n<k;++n)a.j[klb.yn(a.b,b,n)]||++i;for(n=0;n<k;++n){j=klb.wn(a.b,b,n);l=klb.yn(a.b,b,n);if(a.j[l]){++p;continue;}if(a.g[j]){++a.d;a.j[l]=true;switch(klb.Hk(a.b,l)){case 0:a.e[a.c[j]]+='~';a.e[m]+='~';break;case 2:a.e[a.c[j]]+='=';a.e[m]+='=';break;case 3:a.e[a.c[j]]+='#';a.e[m]+='3';}if(a.d>9){a.e[a.c[j]]+='%';a.e[m]+='%';}a.e[a.c[j]]+=''+a.d;a.e[m]+=''+a.d;continue;}n-p<i-1&&(a.e[a.i++]='(');klb.Zs(a,j,l);n-p<i-1&&(a.e[a.i++]=')');}};klb.$s=function $s(){};b2(211,1,{},klb.$s);_.a=0;_.d=0;_.i=0;elb.vY=D3(211);klb._s=function _s(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o;j=oW(elb.IX,omb,6,a.i.f,15,1);h=oW(elb.IX,omb,6,a.i.f,15,1);i=oW(elb.IX,omb,6,a.i.f,15,1);k=oW(elb.IX,omb,6,a.i.f,15,1);c=klb.zk(a.i,0,b);d=klb.zk(a.i,1,b);h[0]=c;h[1]=d;i[0]=-1;i[1]=b;j[c]=1;j[d]=2;k[c]=-1;k[d]=c;g=1;l=1;while(g<=l&&j[h[g]]<15){o=h[g];for(m=0;m<klb.xn(a.i,o);m++){e=klb.wn(a.i,o,m);if(e!=k[o]){f=klb.yn(a.i,o,m);if(e==c){i[0]=f;for(n=0;n<=l;n++){if(!a.f[i[m]]){a.f[i[m]]=true;++a.c;}}return;}if(klb.jl(a.i,e)&&j[e]==0){++l;h[l]=e;i[l]=f;j[e]=j[o]+1;k[e]=o;}}}++g;}return;};klb.at=function at(a){var b,c,d,e,f,g,h,i,j,k,l,m,n;klb.Jt(a.i,7);l=false;m=oW(elb.IX,omb,6,2,15,1);n=oW(elb.IX,omb,6,2,15,1);k=oW(elb.IX,omb,6,2,15,1);for(d=0;d<a.i.g;d++){if(!klb.Fo(a.i,d)&&klb.Kk(a.i,d)==2){for(g=0;g<2;g++){m[g]=-1;k[g]=-1;b=klb.zk(a.i,g,d);for(j=0;j<klb.xn(a.i,b);j++){e=klb.yn(a.i,b,j);if(e!=d){if(m[g]==-1&&(klb.Kk(a.i,e)==257||klb.Kk(a.i,e)==129)){m[g]=klb.wn(a.i,b,j);n[g]=e;}else {k[g]=klb.wn(a.i,b,j);}}}if(m[g]==-1)break;}if(m[0]!=-1&&m[1]!=-1){i=klb.Kk(a.i,n[0])==klb.Kk(a.i,n[1]);for(h=0;h<2;h++)m[h]==klb.zk(a.i,0,n[h])&&(i=!i);for(f=0;f<2;f++)k[f]!=-1&&k[f]<m[f]&&(i=!i);klb.hm(a.i,d,i?2:1,false);l=true;}}}for(c=0;c<a.i.g;c++)(klb.Kk(a.i,c)==257||klb.Kk(a.i,c)==129)&&klb.km(a.i,c,1);return l;};klb.bt=function bt(a){return a==61?2:a==35?4:a==36?32:a==58?8:a==62?16:a==126?31:1;};klb.ct=function ct(a,b){var c,d,e,f,g,h,i,j,k,l,m;for(c=0;c<a.i.f;c++){if(klb.jl(a.i,c)){i=oW(elb.IX,omb,6,a.i.f,15,1);h=oW(elb.IX,omb,6,a.i.f,15,1);j=oW(elb.IX,omb,6,a.i.f,15,1);h[0]=c;i[c]=1;j[c]=-1;g=0;k=0;while(g<=k){e=(i[h[g]]&1)==1?1:2;for(l=0;l<klb.xn(a.i,h[g]);l++){d=klb.yn(a.i,h[g],l);if(klb.Hk(a.i,d)==e&&b[d]){f=klb.wn(a.i,h[g],l);if(i[f]==0){if(e==1&&klb.jl(a.i,f)){m=h[g];while(m!=-1){klb.km(a.i,klb.un(a.i,f,m),e==1?2:1);e=3-e;f=m;m=j[m];}klb.Sl(a.i,c,false);klb.Sl(a.i,f,false);a.b-=2;return true;}h[++k]=f;j[f]=h[g];i[f]=i[h[g]]+1;}}}++g;}}}return false;};klb.dt=function dt(a){var b,c,d,e;for(b=0;b<a.i.f;b++){if(klb.uk(a.i,b)==7&&klb.dk(a.i,b)==0&&klb.Sn(a.i,b)>3&&klb.nn(a.i,b)>0){for(e=0;e<klb.xn(a.i,b);e++){c=klb.wn(a.i,b,e);d=klb.yn(a.i,b,e);if(klb.Hk(a.i,d)>1&&klb.hl(a.i,c)){klb.Kk(a.i,d)==4?klb.km(a.i,d,2):klb.km(a.i,d,1);klb.Jl(a.i,b,klb.dk(a.i,b)+1);klb.Jl(a.i,c,klb.dk(a.i,c)-1);klb.Hl(a.i,b,-1);break;}}}}};klb.et=function et(a,b,c,d){var e;d.d=true;d.c=true;if(a[b]==35){++b;d.a=0;d.b=1;while(b<c&&m3(a[b]&fmb)){d.a=10*d.a+a[b]-48;++d.b;++b;}if(d.a==0||d.a>=(klb.wj(),klb.qj).length)throw l1(new flb.uz('SmilesParser: Atomic number out of range.'));}else if(a[b]>=65&&a[b]<=90){d.b=a[b+1]>=97&&a[b+1]<=122?2:1;d.a=(klb.wj(),klb.Om(flb.B5(ilb.wgb(a,b,(e=d.b,Klb.v8(),e))),321));d.d=false;}else if(a[b]>=97&&a[b]<=122){d.b=a[b+1]>=97&&a[b+1]<=122?2:1;d.a=(klb.wj(),klb.Om(flb.B5(ilb.wgb(a,b,(e=d.b,Klb.v8(),e))),321));d.c=false;}else throw l1(new flb.uz("SmilesParser: Unexpected character within brackets:'"+String.fromCharCode(a[b]&fmb)+"'"));};klb.ft=function ft(a){return !a.n?'':'Unresolved SMARTS features:'+a.n;};klb.gt=function gt(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K;klb.Jt(a.i,1);a.f=oW(elb.g1,Imb,6,a.i.g,16,1);a.c=0;for(k=0;k<a.i.g;k++){if(klb.Kk(a.i,k)==64){klb.km(a.i,k,1);a.f[k]=true;++a.c;}}w=oW(elb.g1,Imb,6,a.i.f,16,1);K=new klb.Gr(a.i,3);v=oW(elb.g1,Imb,6,K.i.a.length,16,1);for(G=0;G<K.i.a.length;G++){I=llb.Bi(K.i,G);v[G]=true;for(q=0;q<I.length;q++){if(!klb.jl(a.i,I[q])){v[G]=false;break;}}if(v[G]){for(r=0;r<I.length;r++)w[I[r]]=true;J=llb.Bi(K.j,G);for(s=0;s<J.length;s++){if(!a.f[J[s]]){a.f[J[s]]=true;++a.c;}}}}for(l=0;l<a.i.g;l++){!a.f[l]&&(K.b[l]&fmb)!=0&&klb.jl(a.i,klb.zk(a.i,0,l))&&klb.jl(a.i,klb.zk(a.i,1,l))&&klb._s(a,l);}for(m=0;m<a.i.g;m++){if(!a.f[m]){e=klb.zk(a.i,0,m);f=klb.zk(a.i,1,m);if(!w[e]&&!w[f]&&klb.jl(a.i,e)&&klb.jl(a.i,f)){a.f[m]=true;++a.c;}}}klb.Jt(a.i,7);u=oW(elb.g1,Imb,6,a.i.g,16,1);for(t=0;t<a.i.g;t++)u[t]=a.f[t];for(H=0;H<K.i.a.length;H++){if(v[H]){I=llb.Bi(K.i,H);for(p=0;p<I.length;p++){if(!klb.nt(a,I[p])){if(klb.jl(a.i,I[p])){klb.Sl(a.i,I[p],false);--a.b;}for(B=0;B<klb.xn(a.i,I[p]);B++){o=klb.yn(a.i,I[p],B);if(a.f[o]){a.f[o]=false;--a.c;}}}}}}klb.mt(a);for(F=0;F<K.i.a.length;F++){if(v[F]&&llb.Bi(K.j,F).length==6){J=llb.Bi(K.j,F);A=true;for(h=J,i=0,j=h.length;i<j;++i){g=h[i];if(!a.f[g]){A=false;break;}}if(A){klb.lt(a,J[0]);klb.lt(a,J[2]);klb.lt(a,J[4]);klb.mt(a);}}}for(D=5;D>=4;D--){do{C=false;for(g=0;g<a.i.g;g++){if(a.f[g]){c=0;for(p=0;p<2;p++){n=klb.zk(a.i,p,g);for(B=0;B<klb.xn(a.i,n);B++)a.f[klb.yn(a.i,n,B)]&&++c;}if(c==D){klb.lt(a,g);klb.mt(a);C=true;break;}}}}while(C);}while(a.b>=2)if(!klb.ct(a,u))break;if(b){if(a.b!=0){for(d=0;d<a.i.f;d++){if(klb.jl(a.i,d)){klb.Sl(a.i,d,false);klb.Vl(a.i,d,2,true);--a.b;}}}if(a.c!=0){for(g=0;g<a.i.g;g++){if(a.f[g]){a.f[g]=false;klb.km(a.i,g,64);--a.c;}}}}else {for(d=0;d<a.i.f;d++){if(klb.jl(a.i,d)&&klb.Mn(a.i,d)!=0){klb.Sl(a.i,d,false);klb.Wl(a.i,d,32);--a.b;}}}if(a.b!=0)throw l1(new flb.uz(Mob));if(a.c!=0)throw l1(new flb.uz(Mob));};klb.ht=function ht(a,b,c,d,e,f,g){var h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N,O,P,Q,R,S,T,U,V,W,X,Y,Z,$,ab,bb,cb,db,eb,fb,gb,hb,ib,jb,kb,lb,mb,nb,ob,pb,qb,rb,sb,tb,ub,vb,wb;a.i=b;klb.Nj(a.i);!!a.n&&flb.K2(a.n,0);a.b=0;i=a.k!=0;cb=null;q=oW(elb.IX,omb,6,32,15,1);q[0]=-1;ib=oW(elb.IX,omb,6,16,15,1);lb=oW(elb.IX,omb,6,16,15,1);kb=oW(elb.IX,omb,6,16,15,1);jb=oW(elb.IX,omb,6,16,15,1);for(N=0;N<16;N++)ib[N]=-1;n=0;I=-1;ob=false;Q=false;nb=false;v=0;t=1;s=0;m=new Alb.cU();db=new klb.Bt(c);l=new klb.st();while(c[d]<=32)++d;while(d<e){qb=c[d++]&fmb;if(n3(qb)||qb==42||qb==63||qb==33&&i&&ob||qb==35&&i&&ob){p=-1;A=0;T=0;h=-1;G=-1;bb=false;P=false;o=0;if(ob){if(qb==42){p=6;o=R1(K1(elb.v1(o),1));}else if(qb==63){p=0;}else {R=qb==33;if(R){nb=true;o=R1(K1(elb.v1(o),1));++d;}if(c[d-1]==82&&i&&(m3(c[d]&fmb)||a.a&&c[d]==123)){p=6;o=R1(K1(elb.v1(o),1));--d;R&&--d;}else {klb.et(c,d-1,e,l);p=l.a;d+=l.b-1;G=9;if(i&&(c[d]==44||R)){V=l.d;U=l.c;pb=d-l.b;while(pb<e){klb.et(c,pb,e,l);Alb.ZT(m,flb.A4(l.a));V=V|l.d;U=U|l.c;pb+=l.b;if(c[pb]!=44)break;++pb;if(c[pb]==33){if(!R)throw l1(new flb.uz("SmilesParser: inconsistent '!' in atom list."));++pb;}}if(m.d.a.length>1){G=-1;U?V||(o=R1(K1(elb.v1(o),4))):o=R1(K1(elb.v1(o),2));}d=pb;}}}while(ob){if(c[d]==64){++d;if(c[d]==64){P=true;++d;}bb=true;continue;}if(c[d]==58){++d;while(m3(c[d]&fmb)){T=10*T+c[d]-48;++d;}continue;}if(c[d]==91)throw l1(new flb.uz('SmilesParser: nested square brackets found'));if(c[d]==93){++d;ob=false;continue;}if(c[d]==43){A=1;++d;while(c[d]==43){++A;++d;}if(A==1&&m3(c[d]&fmb)){A=c[d]-48;++d;}A==0&&(o=R1(K1(elb.v1(o),Xmb)));continue;}if(c[d]==45){A=-1;++d;while(c[d]==45){--A;++d;}if(A==-1&&m3(c[d]&fmb)){A=48-c[d];++d;}A==0&&(o=R1(K1(elb.v1(o),Xmb)));continue;}R=c[d]==33;R&&++d;if(c[d]==72){++d;d+=klb.yt(db,c,d,1);G=db.c;H=0;db.c<=0&&db.b>=0&&(H=R1(K1(elb.v1(H),128)));db.c<=1&&db.b>=1&&(H=R1(K1(elb.v1(H),256)));db.c<=2&&db.b>=2&&(H=R1(K1(elb.v1(H),512)));db.c<=3&&db.b>=3&&(H=R1(K1(elb.v1(H),Vmb)));if(R){o|=H;G=-1;}else {if(db.b==db.c){G=db.c;}else {o=R1(K1(elb.v1(o),n1(Umb,elb.v1(~H))));G=-1;}}continue;}if(c[d]==68){++d;d+=klb.yt(db,c,d,1);H=0;db.c<=0&&db.b>=0&&(H=K1(H,Onb));db.c<=1&&db.b>=1&&(H=K1(H,Bnb));db.c<=2&&db.b>=2&&(H=K1(H,Anb));db.c<=3&&db.b>=3&&(H=K1(H,Qnb));db.c<=4&&db.b>=4&&(H=K1(H,cnb));if(s1(H,0)!=0){R||(H=T1(H,_mb));o=R1(K1(elb.v1(o),H));}continue;}if(c[d]==122&&a.a){++d;d+=klb.yt(db,c,d,4);H=0;db.c<=0&&db.b>=0&&(H=K1(H,mnb));db.c<=1&&db.b>=1&&(H=K1(H,Nob));db.c<=2&&db.b>=2&&(H=K1(H,Oob));db.c<=3&&db.b>=3&&(H=K1(H,Pob));db.c<=4&&db.b>=4&&(H=K1(H,lnb));if(s1(H,0)!=0){R||(H=T1(H,gnb));o=R1(K1(elb.v1(o),H));}continue;}if(c[d]==88){++d;d+=klb.yt(db,c,d,1);wb=(klb.wj(),klb.rj)[p];if(wb==null)continue;sb=wb[0];H=0;sb-db.c<=0&&sb-db.b>=0&&(H=K1(H,Pmb));sb-db.c<=1&&sb-db.b>=1&&(H=K1(H,eob));sb-db.c<=2&&sb-db.b>=2&&(H=K1(H,emb));if(s1(H,0)!=0){R||(H=T1(H,$mb));o=R1(K1(elb.v1(o),H));}continue;}if(c[d]==65||c[d]==97){++d;o=R1(K1(elb.v1(o),R^c[d]==65?4:2));continue;}if(c[d]==82){++d;d+=klb.yt(db,c,d,3);H=0;db.c<=0&&db.b>=0&&(H=K1(H,8));db.c<=1&&db.b>=1&&(H=K1(H,16));db.c<=2&&db.b>=2&&(H=K1(H,32));db.c<=3&&db.b>=3&&(H=K1(H,64));db.b>3&&klb.pt(a,(R?'!R':'R')+db.b);if(s1(H,0)!=0){R||(H=T1(H,120));o=R1(K1(elb.v1(o),H));}continue;}if(c[d]==114){++d;d+=klb.yt(db,c,d,1);if(db.a){R?o=R1(K1(elb.v1(o),384)):o=R1(K1(elb.v1(o),8));continue;}mb=db.c;db.b>db.c&&klb.pt(a,(R?'!r':'r')+('{'+db.c+'-'+db.b+'}'));!R&&mb>=3&&mb<=7?o|=mb<<22:db.b>db.c||klb.pt(a,(R?'!r':'r')+mb);continue;}if(c[d]==118){++d;d+=klb.yt(db,c,d,1);sb=db.c;db.b>db.c&&klb.pt(a,(R?'!v':'v')+('{'+db.c+'-'+db.b+'}'));!R&&sb<=14?h=sb:db.b>db.c||klb.pt(a,(R?'!v':'v')+sb);continue;}if(i&&(c[d]==59||c[d]==38)){nb=true;++d;continue;}throw l1(new flb.uz("SmilesParser: unexpected character inside brackets: '"+String.fromCharCode(c[d]&fmb)+"'"));}}else if(qb==42){p=6;o=R1(K1(elb.v1(o),1));}else if(qb==63){p=0;}else if((qb==65||qb==97)&&i){p=6;o=R1(K1(elb.v1(o),1));o=R1(K1(elb.v1(o),qb==65?4:2));nb=true;}else {switch(String.fromCharCode(qb).toUpperCase().charCodeAt(0)){case 66:if(d<e&&c[d]==114){p=35;++d;}else p=5;break;case 67:if(d<e&&c[d]==108){p=17;++d;}else p=6;break;case 70:p=9;break;case 73:p=53;break;case 78:p=7;break;case 79:p=8;break;case 80:p=15;break;case 83:p=16;}}if(p==-1&&qb!=63)throw l1(new flb.uz('SmilesParser: unknown element label found'));k=klb.yj(a.i,p);klb.Jl(a.i,k,A);klb.Rl(a.i,k,T,false);klb.Hl(a.i,k,h);if(o!=0){nb=true;if(I1(n1(elb.v1(o),2),0)){o=R1(n1(elb.v1(o),-3));klb.Sl(a.i,k,true);++a.b;}else {klb.Sl(a.i,k,false);}klb.Vl(a.i,k,elb.v1(o),true);}if(m.d.a.length!=0){nb=true;S=oW(elb.IX,omb,6,m.d.a.length,15,1);for(M=0;M<m.d.a.length;M++)S[M]=(M<0?null:llb.Bi(m.d,M)).a;klb.Pl(a.i,k,S);ilb.qgb(m.d.a,0);}else {if(String.fromCharCode(qb).toLowerCase().charCodeAt(0)==qb&&n3(qb)){if(p!=5&&p!=6&&p!=7&&p!=8&&p!=15&&p!=16&&p!=33&&p!=34)throw l1(new flb.uz('SmilesParser: atomicNo '+p+' must not be aromatic'));klb.Sl(a.i,k,true);++a.b;}else {klb.Sl(a.i,k,false);}}if(G!=-1&&p!=1){w=oW(elb.EX,Nnb,6,1,15,1);w[0]=(G==9?0:G)<<24>>24;klb.Nl(a.i,k,w);}I=q[v];if(q[v]!=-1&&t!=512){r=klb.Aj(a.i,q[v],k,t);if(s!=0){nb=true;klb.jm(a.i,r,s,true);}}t=1;s=0;q[v]=k;if(n!=0){klb.Tl(a.i,k,n);n=0;}if(g){$=!cb?null:llb.Qab(cb,flb.A4(I));!!$&&klb.ut($,k,d);if(bb){!cb&&(cb=new llb.Aeb());L=G==9?0:G;llb.seb(cb,flb.A4(k),new klb.xt(k,d-2,I,L,P));}}continue;}if(qb==46){q[v]=-1;t=512;continue;}if(qb==45||qb==61||qb==35||qb==36||qb==58||qb==47||qb==92||qb==60||qb==126||qb==33||qb==64){if(ob)throw l1(new flb.uz("SmilesParser: unexpected bond symbol inside square brackets: '"+String.fromCharCode(qb)+"'"));D=0;while(qb==45||qb==61||qb==35||qb==36||qb==58||qb==47||qb==92||qb==60||qb==126||qb==33||qb==64){if(qb==33){qb=c[d++]&fmb;qb==64&&(s|=128);if(qb==45&&c[d]==62||qb==60&&c[d]==45){D|=32;++d;}else if(qb==45)D|=1;else if(qb==61)D|=2;else if(qb==35)D|=4;else if(qb==36)D|=32;else if(qb==58)D|=8;else throw l1(new flb.uz("SmilesParser: bond symbol '"+String.fromCharCode(qb)+"' not allowed after '!'."));}else {if(qb==64)s|=256;else if(qb==61)t=2;else if(qb==35)t=4;else if(qb==36)t=8;else if(qb==58)t=64;else if(qb==126)s|=31;else if(qb==47){g&&(t=257);}else if(qb==92){g&&(t=129);}else if(qb==45&&c[d]==62||qb==60&&c[d]==45){t=32;++d;}if(c[d]==44){s|=(t==32?62:qb)==61?2:(t==32?62:qb)==35?4:(t==32?62:qb)==36?32:(t==32?62:qb)==58?8:(t==32?62:qb)==62?16:(t==32?62:qb)==126?31:1;while(c[d]==44){if(c[d+1]==60&&c[d+2]==45||c[d+1]==45&&c[d+2]==62){s|=16;d+=3;}else {s|=klb.bt(c[d+1]&fmb);d+=2;}}}}if(c[d]==59){++d;qb=c[d++]&fmb;continue;}D!=0&&(s|=31&~D);break;}continue;}if(qb<=32){d=e;continue;}if(m3(qb)){X=qb-48;if(ob){while(d<e&&m3(c[d]&fmb)){X=10*X+c[d]-48;++d;}n=X;}else {u=Q?d-3:d-2;K=c[u]==45||c[u]==47||c[u]==92||c[u]==61||c[u]==35||c[u]==36||c[u]==58||c[u]==62||c[u]==126;if(Q&&d<e&&m3(c[d]&fmb)){X=10*X+c[d]-48;Q=false;++d;}if(X>=ib.length){if(X>=100)throw l1(new flb.uz('SmilesParser: ringClosureAtom number out of range'));Y=ib.length;W=ib.length;while(W<=X)W=$wnd.Math.min(100,W+16);ib=(ilb.Igb(W),llb.ccb(ib,oW(elb.IX,omb,6,W,15,1),W));lb=(ilb.Igb(W),llb.ccb(lb,oW(elb.IX,omb,6,W,15,1),W));kb=(ilb.Igb(W),llb.ccb(kb,oW(elb.IX,omb,6,W,15,1),W));jb=(ilb.Igb(W),llb.ccb(jb,oW(elb.IX,omb,6,W,15,1),W));for(M=Y;M<W;M++)ib[M]=-1;}if(ib[X]==-1){ib[X]=q[v];lb[X]=d-1;kb[X]=K?t:-1;jb[X]=K?s:0;}else {if(ib[X]==q[v])throw l1(new flb.uz('SmilesParser: ring closure to same atom'));if(g&&!!cb){$=llb.Qab(cb,flb.A4(ib[X]));!!$&&klb.ut($,q[v],lb[X]);$=llb.Qab(cb,flb.A4(q[v]));!!$&&klb.ut($,ib[X],d-1);}kb[X]!=-1?t=kb[X]:t==257?t=129:t==129&&(t=257);r=klb.Aj(a.i,ib[X],q[v],t);jb[X]!=0&&(s=jb[X]);if(s!=0){nb=true;klb.jm(a.i,r,jb[X],true);}ib[X]=-1;}t=1;s=0;}continue;}if(qb==43){throw l1(new flb.uz("SmilesParser: '+' found outside brackets"));}if(qb==40){if(q[v]==-1)throw l1(new flb.uz('Smiles with leading parenthesis are not supported'));++v;q.length==v&&(q=llb.acb(q,q.length+32));q[v]=q[v-1];continue;}if(qb==41){--v;continue;}if(qb==91){ob=true;continue;}if(qb==93){throw l1(new flb.uz('SmilesParser: closing bracket at unexpected position'));}if(qb==37){Q=true;continue;}throw l1(new flb.uz("SmilesParser: unexpected character outside brackets: '"+String.fromCharCode(qb)+"'"));}if(t!=1)throw l1(new flb.uz('SmilesParser: dangling open bond'));for(fb=ib,gb=0,hb=fb.length;gb<hb;++gb){eb=fb[gb];if(eb!=-1)throw l1(new flb.uz('SmilesParser: dangling ring closure'));}J=klb.Kn(a.i);klb.om(a.i,true);klb.Jt(a.i,1);for(j=0;j<a.i.q;j++){if(klb.fk(a.i,j)!=null){F=klb.gk(a.i,j)[0];if(nb||a.k==2){if(a.g){for(M=0;M<F;M++)klb.Aj(a.i,j,klb.yj(a.i,1),1);}else {F==0&&klb.Vl(a.i,j,1792,true);F==1&&klb.Vl(a.i,j,1664,true);F==2&&klb.Vl(a.i,j,1408,true);F==3&&klb.Vl(a.i,j,896,true);}}else {if(!klb.kl(a.i,j)&&(!klb.jl(a.i,j)||klb.uk(a.i,j)==6&&klb.dk(a.i,j)==0)){wb=klb.Lm(klb.uk(a.i,j));B=false;rb=klb.Sn(a.i,j);rb-=klb.Ok(a.i,j,rb);rb+=F;klb.jl(a.i,j)&&++rb;for(tb=wb,ub=0,vb=tb.length;ub<vb;++ub){sb=tb[ub];if(rb<=sb){B=true;sb==rb+2?klb.Wl(a.i,j,48):sb==rb+1?klb.Wl(a.i,j,32):(sb!=rb||sb!=wb[0])&&klb.Hl(a.i,j,rb);break;}}B||klb.Hl(a.i,j,rb);}if(a.g||!klb.Wo(a.i,j))for(M=0;M<F;M++)klb.Aj(a.i,j,klb.yj(a.i,1),1);}}else if(!a.g&&(nb||a.k==2)){F=klb.Bn(a.i,j);F>=1&&klb.Vl(a.i,j,128,true);F>=2&&klb.Vl(a.i,j,256,true);F>=3&&klb.Vl(a.i,j,512,true);F>=4&&klb.Vl(a.i,j,Vmb,true);}}!a.g&&(nb||a.k==2)&&klb.Mo(a.i,true);klb.Jt(a.i,1);klb.dt(a);klb.gt(a,i);a.i.t=null;klb.om(a.i,false);if(g){klb.at(a);if(cb){for(ab=(Z=new llb.pab(cb).a.dc().rb(),new llb.wab(Z));ab.a.ec();){$=(C=ab.a.fc(),C.ic());klb.Ul(a.i,$.a,klb.vt($,J),false);}klb.Ro(a.i,0);}}klb.Ro(a.i,0);if(f){O=new olb.Ux(a.d);I1(a.j,0)&&olb.Qx(O,a.j);olb.Bx(O,a.i);g&&klb.Wt(a.i);}(nb||a.k==2)&&klb.mm(a.i,true);};klb.it=function it(a,b,c,d,e){klb.ht(a,b,c,0,c.length,d,e);};klb.jt=function jt(a,b){var c;return b==null?null:klb.kt(a,ilb.ygb((c=b,Klb.v8(),c)));};klb.kt=function kt(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o;h=Alb.IT(b,0);while(h>0&&b[h-1]==45)h=Alb.IT(b,h+1);i=h==-1?-1:Alb.IT(b,h+1);while(i>0&&b[i-1]==45)i=Alb.IT(b,i+1);if(i==-1)throw l1(new flb.uz("Missing one or both separators ('>')."));if(Alb.IT(b,i+1)!=-1)throw l1(new flb.uz("Found more than 2 separators ('>')."));n=new vlb.$E();o=0;for(f=o;f<h-1;f++){if(b[f]==46&&b[f+1]==46){if(f>o){l=new klb._t();klb.ht(a,l,b,o,f,true,true);llb.wi(n.g,l);n.d=-1;}o=f+2;}}m=new klb._t();klb.ht(a,m,b,o,h,true,true);llb.wi(n.g,m);n.d=-1;if(i-h>1){o=h+1;for(g=o;g<i-1;g++){if(b[g]==46&&b[g+1]==46){if(g>o){c=new klb._t();klb.ht(a,c,b,o,g,true,true);llb.wi(n.a,c);}o=g+2;}}d=new klb._t();klb.ht(a,d,b,o,i,true,true);llb.wi(n.a,d);}o=i+1;for(e=o;e<b.length-1;e++){if(b[e]==46&&b[e+1]==46){if(e>o){j=new klb._t();klb.ht(a,j,b,o,e,true,true);llb.wi(n.f,j);n.d=-1;}o=e+2;}}k=new klb._t();klb.ht(a,k,b,o,b.length,true,true);llb.wi(n.f,k);n.d=-1;return n;};klb.lt=function lt(a,b){var c,d,e,f;klb.Kk(a.i,b)==1&&klb.km(a.i,b,2);for(e=0;e<2;e++){c=klb.zk(a.i,e,b);if(klb.jl(a.i,c)){klb.Sl(a.i,c,false);--a.b;}for(f=0;f<klb.xn(a.i,c);f++){d=klb.yn(a.i,c,f);if(a.f[d]){a.f[d]=false;--a.c;}}}};klb.mt=function mt(a){var b,c,d,e,f,g,h;do{h=false;for(c=0;c<a.i.g;c++){if(a.f[c]){f=false;for(e=0;e<2;e++){b=false;d=klb.zk(a.i,e,c);for(g=0;g<klb.xn(a.i,d);g++){if(c!=klb.yn(a.i,d,g)&&a.f[klb.yn(a.i,d,g)]){b=true;break;}}if(!b){f=true;break;}}if(f){h=true;klb.lt(a,c);}}}}while(h);};klb.nt=function nt(a,b){var c,d;if(!klb.Ir(klb.uk(a.i,b)))return false;if(klb.uk(a.i,b)==6){if(!klb.jl(a.i,b))return false;if(klb.dk(a.i,b)>0)return false;}c=klb.fk(a.i,b)==null?0:klb.gk(a.i,b)[0];d=klb.In(a.i,b)-c;if(d<1)return false;if(klb.uk(a.i,b)==16||klb.uk(a.i,b)==34||klb.uk(a.i,b)==52){if(klb.xn(a.i,b)==2&&klb.dk(a.i,b)<=0)return false;if(d==2)return false;}return true;};klb.ot=function ot(a,b){a.j=b;};klb.pt=function pt(a,b){if(a.e){!a.n&&(a.n=new flb.Q5());flb.N5(a.n,' ');flb.N5(a.n,b);}};klb.qt=function qt(){klb.rt.call(this,0,false);};klb.rt=function rt(a,b){this.k=a&3;this.a=(a&16)==0;this.e=b;this.g=(a&8)!=0;this.d=2;(a&4)!=0&&(this.d|=1);this.g&&(this.d&=-3);};b2(116,1,{},klb.qt,klb.rt);_.a=false;_.b=0;_.c=0;_.d=0;_.e=false;_.g=false;_.j=0;_.k=0;elb.zY=D3(116);klb.st=function st(){};b2(233,1,{},klb.st);_.a=0;_.b=0;_.c=false;_.d=false;elb.wY=D3(233);klb.tt=function tt(a,b){this.a=a;this.b=b;};b2(139,1,{139:1},klb.tt);_.a=0;_.b=0;elb.xY=D3(139);klb.ut=function ut(a,b,c){if(!a.c){if(a.e.a.length==4){a.c=true;return;}llb.wi(a.e,new klb.tt(b,c));}};klb.vt=function vt(a,b){var c,d,e;if(a.c)return 3;for(d=new llb.Wbb(a.e);d.a<d.c.a.length;){c=llb.Vbb(d);c.a!=2147483646&&c.a!=Ylb&&(c.a=b[c.a]);}if(a.e.a.length==3)llb.wi(a.e,new klb.tt(Ylb,a.b));else if(a.e.a.length!=4)return 3;e=a.d^klb.wt(a)?1:2;return e;};klb.wt=function wt(a){var b,c,d;c=false;for(b=1;b<a.e.a.length;b++){for(d=0;d<b;d++){llb.Bi(a.e,d).a>llb.Bi(a.e,b).a&&(c=!c);llb.Bi(a.e,d).b>llb.Bi(a.e,b).b&&(c=!c);}}return c;};klb.xt=function xt(a,b,c,d,e){if(d!=0&&d!=1){this.c=true;}else {this.a=a;this.b=b;this.d=e;this.e=new llb.Qi();c!=-1&&klb.ut(this,c,b-1);c!=-1&&d==1&&klb.ut(this,2147483646,b+1);}};b2(169,1,{169:1},klb.xt);_.a=0;_.b=0;_.c=false;_.d=false;elb.yY=D3(169);klb.yt=function yt(a,b,c,d){a.a=false;a.d=c;if(m3(b[c]&fmb)){a.c=a.b=klb.zt(a);return a.d-c;}if(b[c]==123&&m3(b[c+1]&fmb)){++a.d;a.c=klb.zt(a);if(b[a.d++]!=45)return 0;if(!m3(b[a.d]&fmb))return 0;a.b=klb.zt(a);if(b[a.d++]!=125)return 0;return a.d-c;}a.c=1;a.b=d;a.a=true;return 0;};klb.zt=function zt(a){var b;b=a.e[a.d++]-48;m3(a.e[a.d]&fmb)&&(b=10*b+(a.e[a.d++]-48));return b;};klb.At=function At(a){return '{'+a.c+'-'+a.b+'}';};klb.Bt=function Bt(a){this.e=a;};b2(232,1,{},klb.Bt);_.lb=function Ct(){return klb.At(this);};_.a=false;_.b=0;_.c=0;_.d=0;elb.AY=D3(232);klb.Dt=function Dt(a){a.c=new llb.Qi();};klb.Et=function Et(a,b){var c,d,e,f;f=a.c.a.length;if(f==0){llb.vi(a.c,0,b);return 0;}e=1;while(2*e<=f)e<<=1;d=e;--e;while(d!=0){d>>=1;if(e>=f){e-=d;continue;}c=flb.e5(b,llb.Bi(a.c,e));if(c==0)return -1;if(d==0)break;c<0?e-=d:e+=d;}e<f&&flb.e5(b,llb.Bi(a.c,e))>0&&++e;llb.vi(a.c,e,b);return e;};klb.Ft=function Ft(a,b){var c,d,e,f;f=a.c.a.length;if(f==0)return -1;e=1;while(2*e<=f)e<<=1;d=e;--e;while(d!=0){d>>=1;if(e>=f){e-=d;continue;}c=flb.e5(b,llb.Bi(a.c,e));if(c==0)return e;if(d==0)break;c<0?e-=d:e+=d;}return -1;};klb.Gt=function Gt(){klb.Dt(this);};b2(77,1,{77:1,4:1},klb.Gt);_.zb=function Ht(a){return klb.Ft(this,a);};elb.BY=D3(77);klb.It=function It(a,b){klb.Tj(a,b);!!a.d&&(b.T=0);};klb.Jt=function Jt(a,b){var c,d,e,f;klb.an(a,b);if((b&~a.T)==0)return;(a.T&15)!=0&&a.q>1&&klb.th(a.J[0],a.J[1])&&olb.Bx(new olb.Ux(0),a);a.c&&(b|=128);for(c=0;c<a.q;c++)a.u[c]&=-67223560;for(d=0;d<a.g;d++)a.F[d]&=-64;e=0;f=0;if((b&32)!=0){e=32;f=1;}else if((b&64)!=0){e=64;f=3;}if((b&128)!=0){e|=128;f|=32;}a.d=new klb.bg(a,f);klb.Yf(a.d);klb.Zf(a.d);klb.Xf(a.d);klb.$t(a)&&(a.d=new klb.bg(a,f));a.T|=24|e;};klb.Kt=function Kt(a,b){return klb.Lf(a.d,b);};klb.Lt=function Lt(a,b){return klb.Cf(a.d,b);};klb.Mt=function Mt(a){var b;klb.Jt(a,31);b=a.I&fmb;switch(a.I&Unb){case emb:return null;case Onb:return b==1?'meso':''+b+' meso diastereomers';case 0:return 'unknown chirality';case 196608:return 'both enantiomers';case Bnb:return 'this enantiomer';case 327680:return 'this or other enantiomer';case dnb:return 'two epimers';default:return b==1?'one stereo isomer':''+b+' stereo isomers';}};klb.Nt=function Nt(a){var b;b=new klb.au(a.q,a.r);klb.Sj(a,b);return b;};klb.Ot=function Ot(a){var b,c,d,e,f,g,h;klb.Jt(a,15);g=oW(elb.IX,omb,6,3,15,1);for(c=0;c<a.f;c++){if((a.u[c]&dob)!=0){h=(a.u[c]&znb)>>18;h!=0&&(g[h]=$wnd.Math.max(g[h],(a.u[c]&znb)>>18!=1&&(a.u[c]&znb)>>18!=2?-1:(a.u[c]&$nb)>>20));}}for(e=0;e<a.g;e++){if(((a.F[e]&3)==1||(a.F[e]&3)==2)&&a.H[e]==1){h=(a.F[e]&768)>>8;h!=0&&(g[h]=$wnd.Math.max(g[h],(a.F[e]&768)>>8!=1&&(a.F[e]&768)>>8!=2?-1:(a.F[e]&_nb)>>10));}}f=oW(elb.IX,Lnb,7,3,0,2);f[1]=oW(elb.IX,omb,6,1+g[1],15,1);f[2]=oW(elb.IX,omb,6,1+g[2],15,1);for(b=0;b<a.f;b++){if((a.u[b]&dob)!=0){h=(a.u[b]&znb)>>18;h!=0&&++f[h][(a.u[b]&znb)>>18!=1&&(a.u[b]&znb)>>18!=2?-1:(a.u[b]&$nb)>>20];}}for(d=0;d<a.g;d++){if(((a.F[d]&3)==1||(a.F[d]&3)==2)&&a.H[d]==1){h=(a.F[d]&768)>>8;h!=0&&++f[h][(a.F[d]&768)>>8!=1&&(a.F[d]&768)>>8!=2?-1:(a.F[d]&_nb)>>10];}}return f;};klb.Pt=function Pt(a){var b,c;b=oW(elb.IX,omb,6,a.q,15,1);c=klb.Gn(a,b,false,false);return klb.Qt(a,b,c);};klb.Qt=function Qt(a,b,c){var d,e,f,g,h,i,j,k,l,m,n,o,p,q,r;q=oW(elb.CY,Job,25,c,0,1);g=oW(elb.IX,omb,6,c,15,1);j=oW(elb.IX,omb,6,c,15,1);f=oW(elb.IX,omb,6,a.q,15,1);for(e=0;e<a.q;e++)b[e]!=-1&&(f[e]=g[b[e]]++);for(i=0;i<a.r;i++){o=b[a.D[0][i]];p=b[a.D[1][i]];o==p&&o!=-1&&++j[o];}for(r=0;r<c;r++){q[r]=new klb.au(g[r],j[r]);klb.It(a,q[r]);}for(d=0;d<a.q;d++)b[d]!=-1&&klb.Pj(a,q[b[d]],d,0,0);for(h=0;h<a.r;h++){o=b[a.D[0][h]];p=b[a.D[1][h]];o==p&&o!=-1&&klb.Rj(a,q[o],h,0,0,f,false);}for(l=q,m=0,n=l.length;m<n;++m){k=l[m];klb.Dl(k,1);klb.Dl(k,2);}return q;};klb.Rt=function Rt(a){klb.Jt(a,15);!a.d&&(a.f<2||!klb.th(a.J[0],a.J[1]))&&(a.d=new klb.ag(a));return !a.d?null:klb.Gf(a.d);};klb.St=function St(a){klb.Jt(a,15);!a.d&&(a.f<2||!klb.th(a.J[0],a.J[1]))&&(a.d=new klb.ag(a));return !a.d?null:klb.Df(a.d);};klb.Tt=function Tt(a){var b,c;klb.Jt(a,31);c=0;for(b=0;b<a.f;b++)(a.u[b]&3)!=0&&(a.u[b]&4)==0&&++c;return c;};klb.Ut=function Ut(a,b){return klb.Kf(a.d,b);};klb.Vt=function Vt(a,b){a.c=b;a.T&=-144;};klb.Wt=function Wt(a){klb.Jt(a,31);!!a.d&&klb.$f(a.d);};klb.Xt=function Xt(a){var b,c;klb.Jt(a,15);a.L=false;for(b=0;b<a.q;b++){a.u[b]&=gob;(a.u[b]&3)!=0?a.u[b]|=cob:a.u[b]&=fob;}for(c=0;c<a.r;c++)(a.F[c]&3)!=0&&klb.Hk(a,c)==2?a.H[c]=386:a.H[c]&=-385;a.T&=-249;};klb.Yt=function Yt(a,b,c){var d,e,f,g,h,i,j,k,l;l=a.u[b]&3;if(l==1||l==2){h=false;if(a.o[b]==2&&a.j[b]==2&&a.n[b][0]==2&&a.n[b][1]==2&&a.C[b]<=7){for(g=0;g<a.j[b];g++){d=a.i[b][g];k=0;j=oW(elb.IX,omb,6,3,15,1);for(i=0;i<a.j[d];i++){j[k]=a.i[d][i];j[k]!=b&&++k;}k==2&&j[0]<j[1]^c[j[0]]<c[j[1]]&&(h=!h);}}else {for(g=1;g<a.j[b];g++){for(i=0;i<g;i++){e=a.i[b][g];f=a.i[b][i];e<f^c[e]<c[f]&&(h=!h);}}}h&&(l=l==1?2:1);}return l;};klb.Zt=function Zt(a){var b,c,d,e,f,g;klb.Xo(a);klb.Jt(a,31);for(d=0;d<a.f;d++){if(((a.u[d]&znb)>>18==1||(a.u[d]&znb)>>18==2)&&((a.u[d]&dob)==0||(a.u[d]&3)==3))throw l1(new flb.uz(Qob));if((a.u[d]&emb)!=0)throw l1(new flb.uz(Rob));if(((a.u[d]&3)==1||(a.u[d]&3)==2)&&a.o[d]==0){b=oW(elb.GX,mmb,6,a.j[d],15,1);for(f=0;f<a.j[d];f++)b[f]=klb.yk(a,d,a.i[d][f]);for(e=1;e<a.j[d];e++)if(!klb.rl(a,a.k[d][e],d))for(g=0;g<e;g++)if(!klb.rl(a,a.k[d][g],d)){c=$wnd.Math.abs(klb.Nm(b[e],b[g]));if(c<0.08||c>mob)throw l1(new flb.uz(Sob));}}}};klb.$t=function $t(a){var b,c,d,e,f,g,h;h=false;for(c=0;c<a.f;c++)((a.u[c]&dob)==0||(a.u[c]&3)==3)&&(a.u[c]&=gob);for(f=0;f<a.g;f++)(klb.Hk(a,f)!=1||(a.F[f]&3)==0||(a.F[f]&3)==3)&&(a.F[f]&=-32513);if(a.L){if((a.I&Unb)!=Onb){g=oW(elb.g1,Imb,6,a.f,16,1);for(d=0;d<a.f;d++)(a.u[d]&dob)!=0&&(a.u[d]&3)!=3&&(a.u[d]&znb)>>18==1&&(g[d]=true);for(e=0;e<a.f;e++){if((a.u[e]&dob)!=0&&(a.u[e]&3)!=3){klb.Ol(a,e,1,0);h=true;}}for(b=0;b<a.f;b++){if(g[b]){klb.Ul(a,b,1,false);klb.Ol(a,b,1,-1);h=true;}}}a.L=false;}klb.Dl(a,1);klb.Dl(a,2);return h;};klb._t=function _t(){klb.wj();klb.Zo.call(this);};klb.au=function au(a,b){klb.wj();klb.$o.call(this,a,b);};klb.bu=function bu(a){klb.wj();klb._o.call(this,a);};b2(25,91,Tob,klb._t,klb.au,klb.bu);_.ub=function cu(a){klb.It(this,a);};_.wb=function du(a){klb.Jt(this,a);};_.c=false;elb.CY=D3(25);klb.fu=function fu(a,b){var c,d,e,f,g,h,i,j,k,l,m;h=new llb.Qi();klb.Jt(b,1);j=oW(elb.g1,Imb,6,b.f,16,1);for(c=0;c<b.f;c++){if(a.f[c]&&(klb.uk(a.i,c)==7||klb.uk(a.i,c)==8||klb.uk(a.i,c)==16||klb.uk(a.i,c)==34||klb.uk(a.i,c)==52)){j[c]=true;for(i=0;i<b.j[c];i++){d=b.i[c][i];f=b.k[c][i];l=b.n[c][i];if(b.o[d]!=0&&b.o[c]<l){j[d]=true;llb.wi(h,flb.A4(f));for(k=0;k<b.j[d];k++){e=b.i[d][k];if(!j[e]){j[e]=true;g=b.k[d][k];m=b.n[d][k];if(b.o[d]+2==l+m){llb.wi(h,flb.A4(g));l>=m&&(b.o[e]<m?klb.ou(b,e)&&klb.iu(a,b,e,true,false,j,h):klb.iu(a,b,e,true,true,j,h));m>=l&&klb.ou(b,c)&&klb.iu(a,b,e,false,false,j,h);if(a.f[e]&&(klb.uk(a.i,e)==5||klb.uk(a.i,e)==6||klb.uk(a.i,e)==7||klb.uk(a.i,e)==8||klb.uk(a.i,e)==16||klb.uk(a.i,e)==34||klb.uk(a.i,e)==52)&&b.o[e]<m){l<=2&&m>=2&&klb.ou(b,c)&&klb.gu(a,b,f,g);l>=2&&m<=2&&klb.ou(b,e)&&klb.gu(a,b,g,f);}llb.Ii(h,h.a.length-1);}j[e]=false;}}llb.Ii(h,h.a.length-1);j[d]=false;}}j[c]=false;}}};klb.gu=function gu(a,b,c,d){var e;e=new klb.uu(a,b);klb.tu(e,c,klb.Hk(b,c)==1?2:3);klb.tu(e,d,klb.Hk(b,d)==2?1:2);a.g[c]=true;a.g[d]=true;llb.kfb(a.e,e)&&(llb.pbb(a.d,e),true);};klb.hu=function hu(a,b){llb.kfb(a.e,b)&&(llb.pbb(a.d,b),true);};klb.iu=function iu(a,b,c,d,e,f,g){var h,i,j,k,l,m,n,o,p,q,r,s;for(n=0;n<b.j[c];n++){h=b.i[c][n];if(!f[h]){k=b.k[c][n];r=klb.Hk(b,k);if(d&&r>=2||!d&&r<=2){f[h]=true;g.add(flb.A4(k));for(o=0;o<b.j[h];o++){i=b.i[h][o];if(!f[i]){l=b.k[h][o];s=klb.Hk(b,l);if(b.o[h]+2==r+s&&(d&&s<=2||!d&&s>=2)){f[i]=true;g.add(flb.A4(l));if(a.f[i]&&(klb.uk(a.i,i)==5||klb.uk(a.i,i)==6||klb.uk(a.i,i)==7||klb.uk(a.i,i)==8||klb.uk(a.i,i)==16||klb.uk(a.i,i)==34||klb.uk(a.i,i)==52)&&(!d||klb.ou(b,i))){m=new klb.uu(a,b);for(p=0;p<g.a.length;p++){j=(ilb.Lgb(p,g.a.length),g.a[p]).a;q=p<2?d^(p&1)==0:e^(p&1)==0;q?klb.tu(m,j,klb.Hk(b,j)==1?2:3):klb.tu(m,j,klb.Hk(b,j)==2?1:2);a.g[j]=true;}llb.kfb(a.e,m)&&(llb.pbb(a.d,m),true);}else {klb.iu(a,b,i,d,e,f,g);}llb.Ii(g,g.a.length-1);f[i]=false;}}}llb.Ii(g,g.a.length-1);f[h]=false;}}}return false;};klb.ju=function ju(a){var b,c,d,e,f,g,h,i,j,k,l;a.b=oW(elb.IX,omb,6,a.i.f,15,1);h=oW(elb.IX,omb,6,a.i.f,15,1);d=oW(elb.g1,Imb,6,a.i.g,16,1);l=0;for(c=0;c<a.i.g;c++){if(!d[c]&&a.g[c]){++l;a.b[klb.zk(a.i,0,c)]=l;a.b[klb.zk(a.i,1,c)]=l;d[c]=true;for(j=0;j<2;j++){b=klb.zk(a.i,j,c);a.b[b]=l;g=0;i=0;h[0]=b;while(g<=i){for(k=0;k<klb.xn(a.i,h[g]);k++){f=klb.yn(a.i,h[g],k);if(!d[f]&&a.g[f]){d[f]=true;e=klb.wn(a.i,h[g],k);if(a.b[e]==0){a.b[e]=l;h[++i]=e;}}}++g;}}}}a.j=l;};klb.ku=function ku(a){var b,c,d;a.n=oW(elb.IX,omb,6,a.j,15,1);a.k=oW(elb.IX,omb,6,a.j,15,1);a.o=oW(elb.IX,omb,6,a.j,15,1);for(b=0;b<a.i.f;b++){if(a.b[b]!=0){d=a.b[b]-1;a.a!=null&&(a.k[d]+=a.a[b]);a.c!=null&&(a.o[d]+=a.c[b]);}}for(c=0;c<a.i.g;c++){a.g[c]&&klb.Hk(a.i,c)==2&&(a.n[a.b[klb.zk(a.i,0,c)]-1]+=2);}};klb.lu=function lu(a){var b,c,d,e;for(d=0;d<a.i.r;d++){for(e=0;e<2;e++){b=klb.zk(a.i,e,d);c=klb.zk(a.i,1-e,d);if(klb.uk(a.i,b)==1&&klb.nk(a.i,b)>1&&klb.uk(a.i,c)>1&&a.b[c]!=0){if(klb.nk(a.i,b)==2){a.a==null&&(a.a=oW(elb.IX,omb,6,a.i.q,15,1));++a.a[c];}else {a.c==null&&(a.c=oW(elb.IX,omb,6,a.i.q,15,1));++a.c[c];}klb.tl(a.i,b);}}}(a.a!=null||a.c!=null)&&klb.Zj(a.i);};klb.mu=function mu(a){var b;a.e=new llb.pfb();a.d=new llb.Bbb();klb.hu(a,new klb.uu(a,a.i));b=klb.Nt(a.i);while(!llb.tbb(a.d)){klb.su(llb.vbb(a.d),b);klb.fu(a,b);if(a.e.a.c>=klb.eu){flb.V5();'Tautomer count exceeds maximum: '+klb.Gf(new klb.ag(a.i));break;}}};klb.nu=function nu(a){var b,c,d,e,f,g,h,i,j,k;if(a.e.a.c==1)return a.i;j=klb.Nt(a.i);klb.mm(j,true);klb.Jt(j,7);for(e=0;e<j.g;e++){if(a.g[e]){j.H[e]=1;j.T=0;klb.jm(j,e,3,true);}}for(c=0;c<j.f;c++){if(a.b[c]!=0&&klb.Rn(a.i,c)<4){klb.Zm(j,c);j.u[c]&=fob;j.T&=7;klb.Ol(j,c,0,-1);}}h=oW(elb.IX,omb,6,a.j,15,1);i=oW(elb.IX,omb,6,a.j,15,1);d=new klb.ag(j).d;for(b=0;b<j.f;b++){if(a.b[b]!=0){k=a.b[b]-1;if(i[k]<d[b]){i[k]=d[b];h[k]=b;}}}for(f=0;f<a.j;f++){g=''+a.n[f]+'|'+a.k[f]+'|'+a.o[f];klb.Ml(j,h[f],g);}return j;};klb.ou=function ou(a,b){if(a.e[b]-klb.Rn(a,b)+klb.Mn(a,b)<=0)return false;if(klb.hl(a,b))return true;if(a.o[b]!=0)return false;return true;};klb.pu=function pu(a){var b,c,d,e,f,g,h,i;klb.Jt(a.i,1);h=a.i.f;do--h;while(h>=0&&klb.uk(a.i,h)==1);for(b=0;b<h;b++){if(klb.uk(a.i,b)==1){klb.xm(a.i,b,h);do--h;while(klb.uk(a.i,h)==1);}}if(h==a.i.f-1)return;g=oW(elb.g1,Imb,6,a.i.g,16,1);for(f=0;f<a.i.g;f++){c=klb.zk(a.i,0,f);d=klb.zk(a.i,1,f);(klb.uk(a.i,c)==1||klb.uk(a.i,d)==1)&&(g[f]=true);}i=a.i.g;do--i;while(i>=0&&g[i]);for(e=0;e<i;e++){if(g[e]){klb.ym(a.i,e,i);g[e]=false;do--i;while(g[i]);}}};klb.qu=function qu(a){var b,c;this.i=klb.Nt(a);klb.pu(this);klb.Jt(this.i,7);this.g=oW(elb.g1,Imb,6,this.i.g,16,1);this.f=oW(elb.g1,Imb,6,this.i.f,16,1);for(b=0;b<this.i.f;b++){c=klb.Lm(klb.uk(this.i,b))[0];this.f[b]=klb.Rn(this.i,b)<c;}klb.mu(this);klb.ju(this);klb.lu(this);klb.ku(this);};b2(188,1,{},klb.qu);_.j=0;klb.eu=Uob;elb.EY=D3(188);klb.ru=function ru(a,b){return Alb.UT(a.a,b.a);};klb.su=function su(a,b){var c,d;for(d=0;d<a.b.i.g;d++){if(a.b.g[d]){c=3&a.a[d>>4]>>2*(d&15);klb.km(b,d,c==1?1:c==2?a.b.g[d]&&!klb.Fo(a.b.i,d)?386:2:c==3?4:32);}}};klb.tu=function tu(a,b,c){var d,e;d=b>>4;e=2*(b&15);a.a[d]&=~(3<<e);a.a[d]|=c<<e;};klb.uu=function uu(a,b){var c;this.b=a;this.a=oW(elb.IX,omb,6,(a.i.g+15)/16|0,15,1);for(c=0;c<a.i.g;c++)this.a[c>>4]|=$wnd.Math.min(3,klb.Hk(b,c))<<2*(c&15);};b2(127,1,{127:1,27:1},klb.uu);_.qb=function vu(a){return klb.ru(this,a);};elb.DY=D3(127);klb.wu=function wu(a){var b,c,d,e,f;c=flb.n5(a,z5(38));if(c==-1)return a;e=0;f=new flb.G5();while(c!=-1){flb.F5(f,(ilb.Rgb(e,c,ilb.Xgb(a).length),ilb.Xgb(a).substr(e,c-e)));ilb.Sgb(c+1,ilb.Xgb(a).length);if(ilb.Xgb(a).charCodeAt(c+1)==38){f.a+='&';e=c+2;}else {b=(ilb.Sgb(c+1,ilb.Xgb(a).length),ilb.Xgb(a).charCodeAt(c+1));b=b-(b<65?48:b<97?65:97);d=(ilb.Sgb(c+2,ilb.Xgb(a).length),ilb.Xgb(a).charCodeAt(c+2));d=d-(d<65?48:d<97?65:97);flb.D5(f,16*b+d&fmb);e=c+3;}c=flb.o5(a,z5(38),e);}flb.F5(f,(ilb.Sgb(e,ilb.Xgb(a).length+1),ilb.Xgb(a).substr(e)));return f.a;};klb.xu=function xu(a){var b,c;c=new flb.G5();for(b=0;b<ilb.Xgb(a).length;b++){switch(ilb.Sgb(b,ilb.Xgb(a).length),ilb.Xgb(a).charCodeAt(b)){case 38:c.a+='&&';break;case 9:c.a+='&09';break;case 10:c.a+='&0A';break;case 32:c.a+='&20';break;default:flb.D5(c,(ilb.Sgb(b,ilb.Xgb(a).length),ilb.Xgb(a).charCodeAt(b)));}}return c.a;};klb.yu=function yu(a,b,c){a.e[0].a=b;a.e[0].b=c;};klb.zu=function zu(b,c){try{b.a=h4(c);}catch(a){a=k1(a);if(!uX(a,35))throw l1(a);}};klb.Au=function Au(b,c){try{b.b=flb.W3(c,wob,Ylb);}catch(a){a=k1(a);if(!uX(a,35))throw l1(a);}};klb.Bu=function Bu(a,b,c,d){a.c=b;a.a=c;a.b=d;};klb.Cu=function Cu(b,c){try{b.e[0].a=h4(c);}catch(a){a=k1(a);if(!uX(a,35))throw l1(a);}};klb.Du=function Du(b,c){try{b.e[0].b=h4(c);}catch(a){a=k1(a);if(!uX(a,35))throw l1(a);}};klb.Eu=function Eu(){klb.Gu.call(this,new xlb.VF());};klb.Fu=function Fu(a){var b,c,d,e;klb.Eu.call(this);b=0;while(b!=-1){c=ilb.Xgb(a).indexOf('="',b);if(c==-1)break;d=(ilb.Rgb(b+1,c,ilb.Xgb(a).length),ilb.Xgb(a).substr(b+1,c-(b+1)));b=ilb.Xgb(a).indexOf('"',c+2);e=b==-1?(ilb.Sgb(c+1,ilb.Xgb(a).length+1),ilb.Xgb(a).substr(c+1)):(ilb.Rgb(c+1,b,ilb.Xgb(a).length),ilb.Xgb(a).substr(c+1,b-(c+1)));flb.j5(d,'text')?this.c=klb.wu(e):flb.j5(d,'x')?klb.Cu(this,e):flb.j5(d,'y')?klb.Du(this,e):flb.j5(d,'size')?klb.zu(this,e):flb.j5(d,'style')&&klb.Au(this,e);}};klb.Gu=function Gu(a){this.c='';this.a=9;this.b=0;this.e=oW(elb.ZZ,amb,50,1,0,1);this.e[0]=a;};b2(129,154,{154:1},klb.Eu,klb.Fu);_.mb=function Hu(){var a;a=new klb.Eu();klb.Bu(a,this.c,this.a,this.b);klb.yu(a,this.e[0].a,this.e[0].b);a.d=this.d;return a;};_.nb=function Iu(){var a;a=new flb.Q5();flb.N5(a,' text="'+klb.xu(this.c)+'"');flb.N5(a,' x="'+this.e[0].a+'"');flb.N5(a,' y="'+this.e[0].b+'"');this.a!=9&&flb.N5(a,' size="'+this.a+'"');this.b!=0&&flb.N5(a,' style="'+this.b+'"');return a.a;};_.ob=function Ju(){return 'text';};_.a=0;_.b=0;elb.FY=D3(129);klb.Ku=function Ku(a){a.b=new llb.Qi();a.a=new llb.Qi();};klb.Lu=function Lu(a,b){var c,d;c=klb.Et(a,b);if(c==-1)return -1;d=a.b.a.length;llb.wi(a.b,b);llb.vi(a.a,c,new flb.o4(d));return d;};klb.Mu=function Mu(){klb.Gt.call(this);klb.Ku(this);};b2(231,77,{77:1,4:1},klb.Mu);_.zb=function Nu(a){var b;b=klb.Ft(this,a);if(b==-1)return -1;return llb.Bi(this.a,b).a;};elb.GY=D3(231);mlb.Ou=function Ou(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C,D,F,G,H,I,J,K,L,M,N;K=klb.$n(a.e);L=llb.Bi(K.j,b).length;I=llb.Bi(K.i,b);J=llb.Bi(K.j,b);C=true;for(r=1;r<L;r++){if(mlb.bv(a.b,J[r])!=mlb.bv(a.b,J[0])){C=false;break;}}if(C){f=(L-2)*Vob/L;for(s=0;s<L;s++)mlb.Tu(a,I[s],J[s],J[s==0?L-1:s-1],f);return;}G=oW(elb.HX,kob,6,L,15,1);e=0;for(t=0;t<L;t++){G[t]=klb.nn(a.e,I[t])==0?1.9106119386631804:klb.nn(a.e,I[t])==1?2.094499897810209:Vob;e+=G[t];}d=((L-2)*Vob-e)/L;for(u=0;u<L;u++)G[u]+=d;k=oW(elb.HX,kob,6,L,15,1);for(v=1;v<L;v++)k[v]=k[v-1]+Vob-G[v];h=oW(elb.HX,kob,6,L,15,1);for(g=0;g<100;g++){M=0;N=0;for(s=0;s<L;s++){M+=mlb.bv(a.b,J[s])*$wnd.Math.sin(k[s]);N+=mlb.bv(a.b,J[s])*$wnd.Math.cos(k[s]);}n=klb.Mm(0,0,M,N);p=$wnd.Math.sqrt(M*M+N*N);F=-1;D=0;for(w=0;w<L;w++){A=w==0?L-1:w-1;B=w+1==L?0:w+1;i=klb.Nm(k[w],k[A]);j=klb.Nm(k[B],k[w]);H=klb.Nm(G[w],G[B]);h[w]=2*i-2*j+2*H;o=$wnd.Math.cos(k[w]-Wob-n);m=p*o-Xob*h[w];if($wnd.Math.abs(m)>$wnd.Math.abs(D)){D=m;F=w;}}l=$wnd.Math.exp(-5*g/100);k[F]+=l*D;}for(q=0;q<L;q++){A=q==0?L-1:q-1;c=k[A]+Vob-k[q];c>Mmb&&(c-=Mmb);mlb.Tu(a,I[q],J[A],J[q],c);}};mlb.Pu=function Pu(a,b){var c,d,e,f,g,h,i,j,k;i=klb.$n(a.e);j=llb.Bi(i.j,b).length;h=llb.Bi(i.j,b);k=false;for(e=0;e<j;e++){if(a.f[h[e]]==0){k=true;break;}}if(k){c=(180*j-360)/j*0.01745329238474369;g=llb.Bi(i.i,b);for(f=0;f<j;f++){mlb.Tu(a,g[f],h[f==0?j-1:f-1],h[f],c);}if(j<=4){for(d=0;d<j;d++)a.f[h[d]]+=j;}}};mlb.Qu=function Qu(a,b,c,d){var e,f,g,h;f=-1;g=-1;for(h=0;h<klb.hn(a.e,b);h++){e=klb.wn(a.e,b,h);if(e==c||e==d){if(g==-1){g=h;}else {f=h;break;}}}return a.a[b][f][g];};mlb.Ru=function Ru(a,b,c,d){return c<d?a.a[b][d][c]:a.a[b][c][d];};mlb.Su=function Su(a,b){var c,d,e,f,g,h,i;d=oW(elb.g1,Imb,6,klb.xn(a.e,b),16,1);i=0;for(e=0;e<klb.xn(a.e,b);e++){h=0;g=-1;for(f=0;f<klb.xn(a.e,b);f++){if(!d[f]){c=klb.yn(a.e,b,f);if(h<a.f[c]){h=a.f[c];g=f;}}}if(h==0)return i;i<<=8;i+=h;d[g]=true;}return i;};mlb.Tu=function Tu(a,b,c,d,e){var f,g,h,i;f=-1;g=-1;for(i=0;i<klb.hn(a.e,b);i++){h=klb.yn(a.e,b,i);if(h==c||h==d){if(f==-1){f=i;}else {g=i;break;}}}if(a.a[b][g][f]==0){a.a[b][g][f]=e;a.d[b]+=e;++a.c[b];}};mlb.Uu=function Uu(a,b,c,d,e){var f,g,h,i,j,k;f=klb.hn(a.e,b);i=klb.xn(a.e,b);for(j=1;j<f;j++){g=j<i?a.f[klb.yn(a.e,b,j)]:0;for(k=0;k<j;k++){if(a.a[b][j][k]==0){h=k<i?a.f[klb.yn(a.e,b,k)]:0;(g==c&&h==d||g==d&&h==c)&&(a.a[b][j][k]=e);}}}};mlb.Vu=function Vu(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A;this.b=b;this.e=a;this.a=oW(elb.HX,Tlb,344,this.e.f,0,3);for(e=0;e<this.e.f;e++){this.a[e]=oW(elb.HX,{344:1,4:1,10:1,5:1,8:1},36,klb.hn(this.e,e),0,2);for(i=1;i<klb.hn(this.e,e);i++)this.a[e][i]=oW(elb.HX,kob,6,i,15,1);}this.c=oW(elb.IX,omb,6,this.e.f,15,1);this.d=oW(elb.HX,kob,6,this.e.f,15,1);this.f=oW(elb.IX,omb,6,this.e.g,15,1);w=klb.$n(this.e);m=oW(elb.g1,Imb,6,w.i.a.length,16,1);klb.er(w,m,oW(elb.g1,Imb,6,w.i.a.length,16,1),oW(elb.IX,omb,6,w.i.a.length,15,1),true);for(h=3;h<=7;h++){for(r=0;r<w.i.a.length;r++){A=llb.Bi(w.j,r).length;A==h&&(m[r]?mlb.Ou(this,r):A<=4&&mlb.Pu(this,r));}}l=oW(elb.g1,Imb,6,this.e.f,16,1);for(q=0;q<w.i.a.length;q++)if(m[q])for(t=llb.Bi(w.i,q),u=0,v=t.length;u<v;++u){s=t[u];l[s]=true;}f=wW(iW(elb.IX,1),omb,6,15,[0,0,1,3,6,10,15,21]);for(d=0;d<this.e.f;d++){g=klb.hn(this.e,d);if(g>4){for(i=1;i<g;i++)for(o=0;o<i;o++)this.a[d][i][o]=Wob;this.c[d]=f[g];continue;}if(this.c[d]==f[g])continue;if(klb.Eo(this.e,d)&&(l[d]||klb.rn(this.e,d)<=4&&klb.nn(this.e,d)>0)){if(g>2){if(this.c[d]==1){klb.uk(this.e,d)<=14?c=(Yob-this.d[d])/2:c=Zob+(Zob-this.d[d])*0.18000000715255737;}else {c=Yob-this.d[d];if(g>3){if(this.c[d]==2){n=oW(elb.g1,Imb,6,klb.hn(this.e,d),16,1);for(j=1;j<g;j++){for(p=0;p<j;p++){if(this.a[d][j][p]!=0){n[j]=!n[j];n[p]=!n[p];}}}for(k=0;k<g;k++){if(n[k]){for(p=k+1;p<g;p++){if(n[p]){this.a[d][p][k]=c;break;}}break;}}}c=Wob;}}for(i=1;i<g;i++)for(o=0;o<i;o++)this.a[d][i][o]==0&&(this.a[d][i][o]=c);}}else if(klb.Eo(this.e,d)&&klb.rn(this.e,d)<=4){switch(mlb.Su(this,d)){case 771:mlb.Uu(this,d,0,3,2.0653998851776123);mlb.Uu(this,d,0,0,1.9814722631346626);break;case 1028:mlb.Uu(this,d,0,4,1.9797999858856201);mlb.Uu(this,d,0,0,1.94691481878138);break;case 393987:mlb.Uu(this,d,0,3,$ob);mlb.Uu(this,d,0,6,$ob);mlb.Uu(this,d,3,3,1.7229016938441077);break;case 459779:mlb.Uu(this,d,0,3,_ob);mlb.Uu(this,d,0,4,_ob);mlb.Uu(this,d,0,7,_ob);mlb.Uu(this,d,3,4,1.9322539839360076);break;case 525316:mlb.Uu(this,d,0,4,apb);mlb.Uu(this,d,0,8,apb);mlb.Uu(this,d,4,4,1.99944913298566);case 394758:mlb.Uu(this,d,0,6,2.526099920272827);break;case 460550:mlb.Uu(this,d,0,6,bpb);mlb.Uu(this,d,0,7,bpb);break;case 526087:mlb.Uu(this,d,0,7,cpb);mlb.Uu(this,d,0,8,cpb);break;case 526344:mlb.Uu(this,d,0,8,2.186300039291382);break;case 50529027:mlb.Uu(this,d,3,3,2.4189000129699707);break;case 67371779:mlb.Uu(this,d,3,4,2.2298998832702637);break;case 67372036:mlb.Uu(this,d,4,4,2.094399929046631);break;case 101057283:mlb.Uu(this,d,3,6,1.839926051241747);mlb.Uu(this,d,3,3,2.9061476191098734);break;case 117834755:mlb.Uu(this,d,3,4,2.812249087174905);mlb.Uu(this,d,3,7,1.7910569124592968);mlb.Uu(this,d,4,6,2.1224948975613245);break;case 134677507:mlb.Uu(this,d,3,4,2.642428360523752);mlb.Uu(this,d,3,8,2.027723514585844);mlb.Uu(this,d,4,7,2.251474717631936);break;case 117900035:mlb.Uu(this,d,3,7,2.109753935530918);mlb.Uu(this,d,3,3,3.1052897491356646);break;case 117900292:mlb.Uu(this,d,4,7,2.090729910747413);mlb.Uu(this,d,4,4,2.551671293386306);break;case 134743044:mlb.Uu(this,d,4,8,2.139250042271712);mlb.Uu(this,d,4,4,2.3520055858942612);}}else {c=klb.uk(this.e,d)>10?dpb:klb.nn(this.e,d)==2?Vob:klb.vo(this.e,d,true)?epb:klb.nn(this.e,d)==0?dpb:epb;for(i=1;i<g;i++)for(o=0;o<i;o++)this.a[d][i][o]=c;}}};b2(278,1,{},mlb.Vu);elb.HY=D3(278);mlb.av=function av(){mlb.av=d2;mlb.$u=wW(iW(elb.g1,1),Imb,6,16,[false,false,false,false,false,true,true,true,true,false,false,false,false,false,false,true,true]);};mlb.bv=function bv(a,b){return a.a[b];};mlb.cv=function cv(a){mlb.av();var b,c;klb.Jt(a,7);this.a=oW(elb.HX,kob,6,a.r,15,1);this.b=oW(elb.HX,kob,6,a.r,15,1);for(b=0;b<a.r;b++){c=mlb.fv(a,b);if(c==-1){this.a[b]=mlb.gv(a,b);this.b[b]=mlb.hv(a,b);}else {this.a[b]=c==-1?2.000499963760376:mlb.Yu[c];this.b[b]=c==-1?1:mlb.Zu[c];}}};mlb.dv=function dv(a,b){if(b>=a.f)return 0;if(b<a.f&&klb.sr(a.p,b)&&a.C[b]==6&&a.s[b]!=0)return 1;return a.o[b];};mlb.ev=function ev(a,b,c,d,e,f,g){var h,i,j,k,l;k=d<mlb.$u.length&&mlb.$u[d]?f<<8:0;l=e<mlb.$u.length&&mlb.$u[e]?g<<8:0;h=k+d;i=l+e;j=c?0:b?4+a:a;return (j<<24)+(h<i?(h<<12)+i:(i<<12)+h);};mlb.fv=function fv(a,b){var c,d,e,f;c=a.D[0][b];d=a.D[1][b];e=a.C[c];f=a.C[d];return mlb.jv(mlb.ev(klb.Hk(a,b),b<a.g&&klb.tr(a.p,b),b<a.g&&(klb.xr(a.p,b)||a.H[b]==64),e,f,mlb.dv(a,c),mlb.dv(a,d)));};mlb.gv=function gv(a,b){var c,d;c=klb.uk(a,a.D[0][b]);d=klb.uk(a,a.D[1][b]);return (c<(mlb.Iw(),mlb.Gw).length?mlb.Gw[c]:c<mlb.Hw.length?fpb*mlb.Hw[c]:gpb)+(d<mlb.Gw.length?mlb.Gw[d]:d<mlb.Hw.length?fpb*mlb.Hw[d]:gpb);};mlb.hv=function hv(a,b){var c,d;c=klb.uk(a,a.D[0][b]);d=klb.uk(a,a.D[1][b]);return (c<(mlb.Iw(),mlb.Gw).length?hpb:0.125)+(d<mlb.Gw.length?hpb:0.125);};mlb.iv=function iv(){var b,c,d,e,f,g,h;if(!mlb._u){b=(h=flb.u5('2312\n4099\t2.403\t0.282\t466\n4100\t1.407\t0.2\t28\n4103\t1.561\t0.023\t5\n4104\t1.42\t0.092\t491\n4105\t1.313\t0.149\t32\n4107\t2.677\t0.401\t190\n4108\t2.118\t0.244\t325\n4109\t1.978\t0.26\t968\n4113\t1.819\t0.033\t4\n4115\t3.194\t0.482\t381\n4116\t2.771\t0.32\t198\n4117\t2.398\t0.235\t145\n4118\t2.317\t0.22\t208\n4119\t2.034\t0.212\t34\n4120\t2.033\t0.286\t103\n4121\t1.959\t0.264\t122\n4122\t1.918\t0.267\t664\n4123\t1.893\t0.257\t185\n4124\t1.877\t0.218\t259\n4125\t2.05\t0.257\t556\n4126\t2.012\t0.239\t129\n4127\t1.979\t0.292\t196\n4133\t3.305\t0.429\t34\n4134\t2.593\t0.367\t83\n4135\t2.74\t0.174\t706\n4136\t2.462\t0.196\t611\n4137\t2.207\t0.264\t78\n4138\t2.16\t0.225\t359\n4140\t2.004\t0.188\t2204\n4141\t1.916\t0.256\t610\n4142\t1.95\t0.234\t126\n4143\t2.3\t0.213\t81\n4144\t2.187\t0.116\t6\n4145\t2.103\t0.153\t21\n4146\t2.031\t0.308\t83\n4147\t1.752\t0.443\t21\n4151\t3.49\t0.4\t83\n4152\t3.017\t0.331\t40\n4153\t2.932\t0.276\t176\n4154\t2.39\t0.34\t24\n4155\t3.124\t0.377\t16\n4156\t2.907\t0.232\t198\n4158\t2.857\t0.237\t142\n4159\t2.687\t0.244\t49\n4160\t2.787\t0.359\t29\n4161\t2.785\t0.306\t37\n4162\t2.623\t0.298\t175\n4163\t2.812\t0.154\t70\n4164\t2.745\t0.281\t100\n4165\t2.719\t0.184\t37\n4166\t2.683\t0.361\t231\n4167\t2.655\t0.253\t198\n4168\t2.546\t0.208\t37\n4169\t2.259\t0.231\t190\n4170\t2.043\t0.289\t251\n4171\t2.001\t0.221\t391\n4172\t1.984\t0.164\t885\n4173\t1.904\t0.228\t1074\n4174\t2.036\t0.224\t250\n4175\t1.955\t0.184\t6\n4176\t2.35\t0.15\t12\n4177\t2.367\t0.216\t7\n4178\t2.47\t0.186\t11\n4179\t0.95\t-1.0\t1\n4186\t2.88\t0.309\t39\n4188\t2.804\t0.289\t136\n4189\t2.172\t0.082\t12\n4190\t2.321\t0.0\t2\n4359\t1.503\t0.106\t24\n4360\t1.425\t0.052\t40\n12291\t2.861\t0.27\t1069\n12292\t2.773\t0.086\t4\n12293\t2.507\t0.251\t27\n12294\t2.769\t0.373\t370\n12299\t3.1\t0.221\t116\n12300\t2.844\t0.146\t9\n12301\t3.636\t0.002\t3\n12302\t2.779\t0.318\t189\n12307\t3.639\t0.518\t25\n12308\t3.205\t0.245\t16\n12317\t3.036\t0.064\t4\n12318\t2.88\t-1.0\t1\n12319\t2.615\t0.864\t3\n12320\t2.857\t0.129\t247\n12325\t3.557\t0.41\t18\n12326\t3.273\t0.208\t23\n12330\t3.496\t0.025\t2\n12333\t3.012\t0.276\t2\n12334\t2.679\t0.0\t2\n12338\t2.886\t0.196\t80\n12339\t2.871\t0.221\t18\n12343\t3.628\t0.266\t9\n12344\t3.37\t0.145\t35\n12346\t2.859\t0.0\t4\n12352\t2.722\t0.017\t4\n12354\t3.317\t0.016\t4\n12355\t2.714\t0.006\t2\n12357\t2.703\t0.005\t2\n12365\t2.615\t-1.0\t1\n12367\t2.729\t0.0\t2\n12368\t2.841\t0.0\t2\n12369\t2.889\t0.082\t5\n12370\t2.913\t0.126\t64\n12549\t3.003\t0.175\t18\n12550\t2.797\t0.21\t1156\n12805\t2.63\t0.0\t2\n12806\t2.939\t0.243\t85\n16388\t2.29\t0.104\t57\n16389\t2.022\t0.097\t6\n16390\t2.097\t0.127\t21\n16391\t1.771\t0.395\t281\n16392\t1.648\t0.242\t634\n16393\t1.567\t0.555\t264\n16395\t2.926\t0.104\t14\n16397\t2.368\t-1.0\t1\n16398\t3.269\t-1.0\t1\n16399\t2.211\t0.15\t6\n16400\t2.1\t-1.0\t1\n16401\t2.065\t0.622\t37\n16403\t3.355\t0.087\t11\n16404\t2.9\t0.136\t2\n16416\t2.399\t0.154\t3\n16417\t2.06\t0.0\t2\n16418\t2.195\t-1.0\t1\n16419\t2.167\t0.95\t16\n16421\t3.378\t-1.0\t1\n16422\t2.886\t0.031\t2\n16429\t2.375\t0.124\t6\n16435\t2.635\t-1.0\t1\n16436\t2.399\t-1.0\t1\n16437\t2.417\t0.96\t23\n16439\t3.716\t0.078\t2\n16442\t3.052\t0.0\t4\n16463\t2.64\t-1.0\t1\n16466\t2.23\t-1.0\t1\n16474\t3.061\t0.0\t4\n16476\t3.017\t0.004\t8\n16645\t1.954\t-1.0\t1\n16646\t2.276\t0.281\t56\n16647\t1.966\t0.14\t51\n16648\t1.843\t0.124\t41\n16902\t1.739\t0.003\t2\n16903\t2.003\t-1.0\t1\n20491\t2.681\t-1.0\t1\n20492\t2.402\t0.287\t12\n20493\t2.373\t0.17\t6\n20499\t3.054\t0.395\t11\n20500\t3.076\t0.081\t2\n20501\t2.3\t1.168\t20\n20502\t2.221\t0.057\t3\n20503\t1.964\t-1.0\t1\n20504\t2.515\t0.236\t21\n20505\t1.97\t0.231\t70\n20506\t2.056\t0.352\t138\n20507\t2.553\t0.302\t81\n20508\t2.489\t0.321\t24\n20509\t2.215\t0.363\t36\n20510\t2.786\t0.026\t6\n20511\t2.418\t0.106\t10\n20517\t3.193\t0.063\t3\n20519\t2.598\t0.187\t9\n20520\t2.376\t0.309\t9\n20522\t2.708\t0.265\t154\n20524\t2.526\t0.319\t123\n20525\t2.262\t0.289\t122\n20526\t2.502\t0.287\t18\n20527\t2.365\t0.284\t11\n20528\t2.966\t0.038\t6\n20529\t2.584\t0.061\t7\n20530\t2.771\t0.387\t5\n20531\t1.29\t0.001\t12\n20535\t3.368\t0.154\t5\n20536\t3.023\t0.074\t6\n20537\t2.913\t0.0\t2\n20538\t3.021\t0.331\t2\n20540\t2.651\t0.201\t5\n20543\t2.735\t0.344\t5\n20546\t2.531\t0.277\t7\n20549\t3.267\t-1.0\t1\n20550\t2.329\t0.34\t7\n20552\t2.499\t0.02\t6\n20553\t2.703\t0.218\t20\n20554\t2.693\t0.292\t46\n20555\t2.639\t0.27\t40\n20556\t2.066\t0.059\t39\n20557\t2.599\t0.324\t183\n20558\t2.048\t0.149\t134\n20559\t2.327\t0.386\t34\n20560\t2.179\t-1.0\t1\n20561\t2.619\t-1.0\t1\n20562\t2.967\t-1.0\t1\n20570\t2.665\t0.007\t2\n20572\t2.63\t0.19\t9\n20742\t1.87\t0.155\t40\n24587\t3.525\t0.393\t28\n24588\t2.763\t0.313\t380\n24589\t2.462\t0.308\t5373\n24595\t4.077\t0.356\t19\n24596\t3.17\t0.362\t45\n24597\t2.829\t0.525\t432\n24598\t2.656\t0.164\t627\n24599\t2.654\t0.158\t57\n24600\t2.57\t0.26\t309\n24601\t2.625\t0.288\t208\n24602\t2.45\t0.223\t1004\n24603\t2.182\t0.257\t1071\n24604\t2.355\t0.185\t1159\n24605\t2.384\t0.162\t130\n24606\t2.479\t0.188\t1375\n24607\t2.547\t0.23\t805\n24613\t3.789\t0.188\t2\n24614\t3.484\t0.291\t8\n24615\t3.089\t0.34\t499\n24616\t2.809\t0.177\t912\n24617\t2.665\t0.147\t106\n24618\t2.632\t0.3\t598\n24620\t2.419\t0.207\t1127\n24621\t2.466\t0.217\t582\n24622\t2.456\t0.168\t1672\n24623\t2.573\t0.135\t12\n24624\t2.753\t0.503\t34\n24625\t2.695\t0.487\t478\n24626\t2.644\t0.444\t4321\n24627\t2.655\t0.961\t518\n24631\t3.712\t0.709\t3\n24632\t3.745\t0.575\t6\n24633\t3.382\t0.306\t128\n24634\t3.417\t0.336\t41\n24635\t3.236\t0.278\t27\n24636\t3.365\t0.272\t72\n24638\t3.236\t0.316\t52\n24639\t3.097\t0.0\t2\n24640\t2.718\t0.258\t39\n24641\t3.353\t0.103\t6\n24642\t3.087\t0.178\t51\n24643\t2.988\t0.348\t31\n24644\t2.995\t0.182\t58\n24645\t3.221\t0.798\t58\n24646\t3.104\t0.353\t86\n24647\t3.019\t0.515\t320\n24648\t2.735\t0.176\t341\n24649\t2.703\t0.204\t299\n24650\t2.595\t0.331\t829\n24651\t2.536\t0.58\t227\n24652\t2.472\t0.25\t206\n24653\t2.534\t0.168\t933\n24654\t2.491\t0.326\t1889\n24655\t2.445\t0.168\t254\n24656\t2.51\t0.162\t95\n24657\t3.004\t0.801\t27\n24658\t2.961\t0.516\t71\n24659\t3.105\t0.444\t106\n24666\t3.114\t0.326\t123\n24668\t3.098\t0.303\t230\n28685\t2.353\t0.307\t3006\n28693\t2.684\t0.241\t703\n28694\t2.452\t0.242\t3992\n28695\t2.409\t0.319\t1209\n28696\t2.464\t0.534\t2219\n28697\t2.577\t0.289\t2590\n28698\t2.456\t0.2\t6911\n28699\t2.41\t0.36\t7053\n28700\t2.487\t0.231\t7446\n28701\t2.406\t0.194\t9772\n28702\t2.479\t0.227\t5502\n28703\t2.471\t0.309\t929\n28711\t2.921\t0.239\t1714\n28712\t2.679\t0.232\t2798\n28713\t2.442\t0.272\t911\n28714\t2.445\t0.293\t2412\n28716\t2.475\t0.298\t2904\n28717\t2.428\t0.335\t1207\n28718\t2.458\t0.222\t3289\n28719\t2.487\t0.247\t1033\n28720\t2.81\t0.231\t768\n28721\t2.785\t0.35\t616\n28722\t2.606\t0.37\t1703\n28723\t2.599\t0.29\t531\n28729\t3.125\t0.329\t626\n28730\t3.082\t0.337\t728\n28731\t3.179\t0.249\t153\n28732\t3.022\t0.208\t458\n28734\t3.032\t0.245\t890\n28735\t3.194\t0.258\t616\n28736\t3.148\t0.29\t455\n28737\t3.124\t0.266\t301\n28738\t3.028\t0.263\t602\n28739\t2.987\t0.306\t159\n28740\t2.919\t0.28\t496\n28741\t3.182\t0.365\t119\n28742\t2.892\t0.229\t1179\n28743\t2.821\t0.212\t549\n28744\t2.607\t0.225\t683\n28745\t2.503\t0.227\t1194\n28746\t2.395\t0.326\t1012\n28747\t2.447\t0.267\t941\n28748\t2.383\t0.233\t326\n28749\t2.511\t0.279\t970\n28750\t2.457\t0.24\t1884\n28751\t2.393\t0.172\t533\n28752\t2.849\t0.285\t194\n28753\t2.973\t0.272\t62\n28754\t2.876\t0.236\t326\n28755\t2.753\t0.232\t424\n28762\t3.088\t0.249\t588\n28764\t2.972\t0.34\t2780\n28765\t3.016\t0.335\t10\n28766\t3.297\t0.204\t7\n32781\t2.145\t0.448\t7124\n32789\t2.625\t0.845\t799\n32790\t2.338\t0.282\t13095\n32791\t2.352\t0.77\t18035\n32792\t2.299\t0.427\t5228\n32793\t2.387\t0.292\t15567\n32794\t2.358\t0.225\t12153\n32795\t2.38\t0.29\t13623\n32796\t2.421\t0.301\t14996\n32797\t2.272\t0.223\t19019\n32798\t2.378\t0.357\t12982\n32799\t2.317\t0.208\t831\n32807\t2.796\t0.315\t3153\n32808\t2.497\t0.249\t4631\n32809\t2.375\t0.365\t1523\n32810\t2.347\t0.467\t31965\n32812\t2.419\t0.158\t3311\n32813\t2.436\t0.193\t2109\n32814\t2.4\t0.154\t2649\n32815\t2.661\t0.256\t865\n32816\t2.694\t0.218\t2847\n32817\t2.616\t0.276\t949\n32818\t2.579\t0.438\t4702\n32819\t2.309\t0.553\t1993\n32825\t2.994\t0.343\t2113\n32826\t2.856\t0.3\t2621\n32827\t2.93\t0.228\t1218\n32828\t2.905\t0.207\t2518\n32830\t2.869\t0.237\t2353\n32831\t2.827\t0.209\t3697\n32832\t2.834\t0.197\t5222\n32833\t2.78\t0.177\t4383\n32834\t2.738\t0.205\t8482\n32835\t2.747\t0.193\t1707\n32836\t2.742\t0.193\t2196\n32837\t2.798\t0.21\t502\n32838\t2.734\t0.235\t2716\n32839\t2.767\t0.212\t885\n32840\t2.498\t0.29\t774\n32841\t2.373\t0.261\t1221\n32842\t2.285\t0.417\t9618\n32843\t2.218\t0.348\t3304\n32844\t2.411\t0.2\t303\n32845\t2.462\t0.304\t1297\n32846\t2.393\t0.203\t1953\n32847\t2.305\t0.24\t286\n32848\t2.736\t0.309\t162\n32849\t2.822\t0.336\t202\n32850\t2.624\t0.233\t1404\n32851\t2.579\t0.268\t1906\n32858\t2.759\t0.291\t759\n32860\t2.611\t0.412\t7083\n32861\t2.503\t0.425\t287\n32862\t2.802\t0.399\t255\n32863\t2.837\t0.483\t20\n32864\t3.097\t0.448\t5\n32865\t2.94\t0.02\t3\n32866\t2.916\t0.021\t3\n36877\t1.811\t0.461\t1163\n36885\t2.022\t0.295\t129\n36886\t2.262\t0.492\t1106\n36887\t2.576\t0.889\t536\n36888\t1.937\t0.267\t474\n36889\t2.093\t0.73\t296\n36890\t1.929\t0.743\t373\n36891\t2.283\t0.257\t113\n36892\t2.221\t0.196\t116\n36893\t2.345\t0.817\t110\n36894\t2.065\t0.336\t75\n36895\t2.412\t0.358\t76\n36903\t2.125\t0.259\t44\n36904\t2.154\t1.055\t667\n36905\t2.12\t0.459\t502\n36906\t2.031\t0.453\t512\n36908\t2.395\t0.388\t43\n36909\t2.339\t0.338\t54\n36910\t2.361\t0.307\t101\n36911\t2.57\t0.191\t9\n36912\t2.427\t0.207\t26\n36913\t2.059\t1.008\t121\n36914\t2.126\t0.626\t419\n36915\t2.193\t0.262\t8674\n36921\t2.731\t0.286\t10\n36922\t2.995\t1.207\t51\n36923\t2.406\t0.044\t8\n36924\t2.742\t0.246\t33\n36926\t2.795\t0.182\t52\n36927\t2.799\t0.303\t24\n36928\t2.454\t0.018\t4\n36929\t1.96\t0.0\t6\n36930\t2.434\t0.192\t19\n36931\t2.857\t-1.0\t1\n36932\t2.747\t0.211\t25\n36934\t2.687\t0.139\t19\n36935\t2.806\t0.003\t4\n36936\t2.107\t0.444\t90\n36937\t2.149\t0.225\t250\n36938\t1.882\t0.247\t232\n36939\t2.0\t0.93\t274\n36940\t2.276\t0.144\t56\n36941\t2.293\t1.188\t69\n36942\t2.356\t0.364\t53\n36943\t1.893\t0.555\t149\n36944\t2.373\t0.0\t2\n36945\t2.015\t0.261\t45\n36946\t2.333\t0.203\t37\n36947\t2.152\t0.196\t94\n36954\t2.666\t0.268\t30\n36955\t2.205\t0.007\t16\n36956\t2.285\t0.379\t153\n36957\t2.26\t0.017\t10\n45067\t3.369\t0.477\t389\n45068\t3.242\t-1.0\t1\n45069\t3.524\t0.425\t2\n45070\t3.558\t0.0\t2\n45075\t3.592\t0.346\t36\n45076\t3.383\t0.133\t22\n45081\t3.221\t0.182\t3\n45082\t2.923\t0.336\t4\n45083\t3.382\t-1.0\t1\n45084\t2.961\t0.097\t3\n45087\t3.136\t-1.0\t1\n45088\t3.161\t0.174\t9\n45093\t3.645\t0.249\t17\n45094\t3.529\t0.167\t10\n45101\t3.618\t-1.0\t1\n45106\t3.243\t0.311\t72\n45107\t3.33\t0.219\t17\n45111\t3.704\t0.248\t22\n45112\t3.585\t0.155\t9\n45134\t3.136\t-1.0\t1\n45136\t3.234\t0.09\t7\n45137\t3.405\t0.105\t12\n45138\t3.441\t0.118\t15\n45139\t3.281\t0.068\t7\n45317\t3.245\t0.017\t2\n45318\t3.261\t0.178\t294\n45573\t2.685\t-1.0\t1\n45574\t3.187\t0.261\t35\n49164\t3.052\t0.123\t490\n49165\t3.401\t-1.0\t1\n49166\t2.887\t0.569\t39\n49171\t4.275\t0.055\t2\n49172\t3.412\t0.019\t8\n49177\t3.408\t-1.0\t1\n49178\t3.072\t0.063\t8\n49181\t3.334\t0.086\t3\n49182\t3.217\t0.061\t2\n49183\t2.836\t0.772\t13\n49184\t2.807\t0.438\t17\n49190\t3.648\t0.307\t28\n49191\t2.187\t0.0\t6\n49194\t2.466\t0.0\t2\n49197\t2.818\t0.099\t91\n49202\t2.927\t0.036\t9\n49203\t2.862\t0.432\t19\n49207\t3.807\t0.0\t2\n49208\t3.659\t0.045\t20\n49209\t3.53\t0.075\t16\n49210\t3.447\t0.003\t12\n49211\t3.506\t0.0\t3\n49212\t3.54\t0.0\t3\n49214\t3.463\t0.0\t3\n49215\t3.48\t0.014\t8\n49216\t3.364\t0.0\t3\n49221\t3.144\t0.0\t12\n49229\t2.7\t0.086\t48\n49230\t3.087\t0.019\t2\n49233\t3.145\t0.004\t2\n49413\t2.826\t0.154\t5\n49414\t2.811\t0.216\t411\n49670\t2.681\t0.238\t49\n53261\t3.147\t0.471\t69\n53262\t3.263\t0.176\t31\n53263\t3.143\t0.346\t218\n53264\t2.983\t0.385\t264\n53265\t2.601\t0.322\t2017\n53268\t3.669\t-1.0\t1\n53272\t2.557\t-1.0\t1\n53273\t2.309\t-1.0\t1\n53274\t2.908\t0.179\t18\n53275\t2.828\t0.234\t10\n53276\t2.655\t0.245\t6\n53277\t2.352\t0.06\t13\n53278\t3.3\t0.187\t3\n53279\t3.414\t0.37\t3\n53280\t2.499\t0.185\t12\n53281\t2.448\t0.051\t41\n53282\t2.875\t0.469\t72\n53283\t2.741\t0.288\t381\n53286\t3.495\t0.045\t15\n53292\t2.999\t-1.0\t1\n53293\t3.023\t0.354\t19\n53294\t2.951\t0.089\t21\n53295\t3.08\t-1.0\t1\n53297\t3.715\t-1.0\t1\n53298\t3.418\t0.0\t2\n53299\t2.655\t0.502\t19\n53300\t2.915\t0.495\t30\n53301\t3.049\t0.382\t137\n53303\t2.804\t1.394\t2\n53304\t3.582\t0.06\t4\n53305\t3.584\t0.347\t3\n53308\t3.56\t0.086\t2\n53322\t3.218\t0.048\t2\n53325\t3.468\t0.243\t2\n53326\t3.202\t0.139\t4\n53331\t3.74\t0.255\t4\n53340\t4.01\t0.006\t2\n53509\t2.605\t0.31\t8\n53510\t2.407\t0.249\t1700\n53511\t2.374\t0.163\t1982\n53512\t2.216\t0.23\t533\n53519\t2.935\t0.196\t5\n53520\t2.843\t0.186\t40\n53766\t2.468\t0.199\t96\n53767\t2.448\t0.135\t44\n53768\t1.978\t0.199\t4\n57363\t4.548\t0.464\t19\n57364\t3.083\t0.417\t13\n57366\t3.671\t0.019\t4\n57367\t3.246\t0.035\t4\n57368\t2.795\t0.193\t8\n57369\t2.811\t0.292\t27\n57370\t2.847\t0.192\t233\n57371\t2.715\t0.262\t77\n57372\t2.749\t0.228\t147\n57373\t2.884\t0.306\t43\n57374\t3.288\t0.273\t42\n57375\t3.561\t0.36\t4\n57382\t3.347\t0.097\t2\n57384\t3.775\t0.176\t18\n57385\t3.436\t0.115\t26\n57386\t3.224\t0.255\t49\n57388\t2.956\t0.338\t151\n57389\t2.78\t0.227\t106\n57390\t2.921\t0.275\t97\n57391\t3.357\t0.302\t20\n57393\t3.19\t0.085\t14\n57394\t3.662\t0.527\t210\n57395\t3.917\t0.191\t58\n57399\t3.91\t0.0\t4\n57400\t3.327\t0.062\t5\n57402\t3.01\t0.585\t8\n57406\t4.511\t0.25\t17\n57407\t4.503\t0.08\t6\n57409\t3.926\t0.003\t2\n57411\t3.905\t0.012\t2\n57414\t4.362\t0.205\t19\n57416\t3.87\t0.205\t13\n57417\t3.437\t0.24\t5\n57418\t3.085\t0.21\t66\n57419\t2.96\t0.12\t14\n57420\t3.028\t0.125\t7\n57421\t2.797\t0.286\t150\n57422\t2.978\t0.215\t175\n57423\t2.761\t0.465\t12\n57424\t3.53\t0.211\t22\n57426\t3.874\t0.316\t17\n57427\t4.031\t0.321\t9\n61461\t3.38\t0.299\t100\n61462\t3.21\t0.369\t276\n61463\t3.131\t0.207\t110\n61464\t2.873\t0.244\t730\n61465\t2.696\t0.32\t774\n61466\t2.697\t0.243\t4802\n61467\t2.679\t0.235\t3056\n61468\t2.676\t0.196\t4702\n61469\t2.64\t0.179\t5253\n61470\t2.912\t0.244\t119\n61471\t3.023\t0.242\t152\n61479\t3.617\t0.27\t57\n61480\t3.41\t0.235\t240\n61481\t3.284\t0.297\t129\n61482\t3.002\t0.261\t2058\n61484\t2.758\t0.203\t6616\n61485\t2.714\t0.205\t4052\n61486\t2.713\t0.193\t6710\n61487\t2.863\t0.16\t1964\n61488\t3.199\t0.271\t42\n61489\t3.285\t0.255\t93\n61490\t3.243\t0.31\t221\n61491\t3.183\t0.277\t77\n61497\t3.201\t0.305\t27\n61498\t2.934\t0.476\t9\n61500\t3.451\t0.1\t14\n61502\t3.794\t0.117\t26\n61506\t3.579\t0.083\t13\n61507\t3.584\t0.109\t8\n61508\t3.774\t0.0\t2\n61510\t3.577\t0.217\t22\n61511\t3.739\t0.336\t7\n61512\t3.408\t0.222\t72\n61513\t3.244\t0.257\t147\n61514\t3.003\t0.256\t1665\n61515\t2.906\t0.239\t1623\n61516\t2.798\t0.241\t1057\n61517\t2.754\t0.215\t2863\n61518\t2.714\t0.201\t6668\n61519\t2.668\t0.172\t4216\n61520\t2.879\t0.203\t76\n61521\t3.471\t0.204\t33\n61522\t3.487\t0.164\t39\n61523\t3.141\t0.234\t30\n61530\t3.726\t0.298\t60\n61532\t3.891\t0.328\t80\n65557\t3.412\t0.368\t35\n65558\t2.901\t0.267\t387\n65559\t2.852\t0.348\t489\n65560\t2.881\t0.289\t268\n65561\t2.726\t0.281\t883\n65562\t2.718\t0.226\t7921\n65563\t2.724\t0.272\t1489\n65564\t2.606\t0.234\t5975\n65565\t2.755\t0.225\t5038\n65566\t2.788\t0.225\t1360\n65567\t2.767\t0.214\t227\n65575\t3.418\t0.184\t159\n65576\t3.376\t0.277\t123\n65577\t2.954\t0.528\t373\n65578\t2.911\t0.288\t3355\n65580\t2.816\t0.192\t1907\n65581\t2.839\t0.173\t765\n65582\t2.775\t0.192\t2485\n65583\t3.093\t0.232\t4438\n65584\t3.087\t0.238\t609\n65585\t3.111\t0.393\t484\n65586\t2.949\t0.477\t1247\n65587\t2.995\t0.415\t696\n65593\t3.611\t0.276\t65\n65594\t3.5\t0.164\t59\n65595\t3.551\t0.145\t18\n65596\t3.459\t0.169\t45\n65598\t3.453\t0.18\t175\n65599\t3.683\t0.232\t44\n65600\t3.412\t0.05\t10\n65601\t3.114\t0.282\t6\n65602\t3.516\t0.333\t73\n65603\t3.018\t0.572\t5\n65604\t3.488\t0.196\t81\n65605\t3.266\t0.272\t10\n65606\t3.254\t0.188\t179\n65607\t3.336\t0.194\t34\n65608\t3.554\t0.326\t21\n65609\t2.461\t0.31\t329\n65610\t2.919\t0.288\t2082\n65611\t2.8\t0.267\t1996\n65612\t2.475\t0.252\t240\n65613\t2.833\t0.153\t655\n65614\t2.73\t0.196\t2208\n65615\t2.776\t0.277\t3697\n65616\t2.944\t0.23\t538\n65617\t3.031\t0.277\t17\n65618\t3.147\t0.26\t222\n65619\t3.099\t0.284\t350\n65626\t3.319\t0.35\t170\n65628\t3.382\t0.265\t290\n65629\t3.399\t0.077\t8\n65630\t2.931\t0.387\t17\n65631\t3.438\t0.046\t3\n65632\t3.441\t0.047\t3\n69653\t3.008\t0.308\t233\n69654\t2.799\t0.246\t2597\n69655\t2.789\t0.215\t490\n69656\t2.8\t0.199\t1394\n69657\t2.875\t0.267\t1182\n69658\t2.712\t0.238\t3474\n69659\t2.694\t0.227\t2713\n69660\t2.758\t0.28\t2082\n69661\t2.701\t0.25\t5780\n69662\t2.642\t0.425\t3822\n69663\t2.661\t0.254\t1485\n69671\t3.218\t0.274\t409\n69672\t2.991\t0.248\t1900\n69673\t2.903\t0.356\t1449\n69674\t2.907\t0.475\t2267\n69676\t2.885\t0.203\t5377\n69677\t2.863\t0.235\t2604\n69678\t2.776\t0.213\t6900\n69679\t3.032\t0.245\t617\n69680\t2.974\t0.221\t1026\n69681\t2.957\t0.377\t932\n69682\t2.928\t0.391\t2452\n69683\t2.872\t0.46\t1745\n69689\t3.393\t0.345\t120\n69690\t3.542\t0.471\t133\n69691\t3.384\t0.328\t54\n69692\t3.319\t0.289\t122\n69694\t3.354\t0.248\t151\n69695\t3.179\t0.349\t113\n69696\t3.251\t0.372\t116\n69697\t3.069\t0.3\t85\n69698\t3.114\t0.336\t295\n69699\t3.273\t0.344\t35\n69700\t3.214\t0.336\t160\n69701\t2.947\t0.43\t25\n69702\t3.224\t0.207\t183\n69703\t3.182\t0.198\t73\n69704\t2.972\t0.302\t357\n69705\t2.911\t0.327\t1000\n69706\t2.885\t0.369\t1787\n69707\t2.801\t0.257\t2221\n69708\t2.851\t0.255\t681\n69709\t2.879\t0.222\t2186\n69710\t2.757\t0.265\t4336\n69711\t2.675\t0.201\t2687\n69712\t2.804\t0.27\t1229\n69713\t2.554\t0.473\t192\n69714\t3.224\t0.457\t625\n69715\t3.107\t0.371\t1402\n69722\t3.417\t0.306\t157\n69723\t2.64\t0.0\t4\n69724\t3.21\t0.343\t893\n69725\t3.181\t0.284\t122\n69726\t3.088\t0.296\t154\n69727\t2.915\t-1.0\t1\n69728\t2.915\t-1.0\t1\n69729\t2.465\t0.0\t6\n77843\t3.904\t0.444\t476\n77844\t3.931\t0.219\t13\n77847\t3.213\t0.001\t2\n77848\t3.318\t0.114\t5\n77849\t3.574\t0.313\t3\n77850\t3.205\t-1.0\t1\n77851\t3.463\t0.411\t4\n77852\t3.491\t0.534\t2\n77853\t3.822\t0.039\t3\n77854\t3.511\t0.765\t6\n77856\t3.502\t0.647\t78\n77861\t3.9\t0.078\t12\n77862\t3.806\t0.091\t8\n77864\t3.807\t-1.0\t1\n77865\t3.535\t0.137\t7\n77870\t3.386\t0.218\t17\n77871\t3.245\t0.006\t2\n77874\t3.754\t0.595\t67\n77875\t3.655\t0.42\t25\n77879\t3.899\t0.365\t31\n77880\t4.729\t0.676\t2\n77898\t4.496\t0.606\t3\n77902\t3.48\t0.0\t4\n77905\t3.736\t0.08\t11\n77906\t3.724\t0.099\t26\n77907\t3.562\t0.045\t4\n77917\t4.592\t-1.0\t1\n78085\t3.76\t0.113\t8\n78086\t3.713\t0.29\t1287\n78342\t3.557\t0.28\t73\n81940\t3.661\t0.289\t176\n81943\t3.019\t-1.0\t1\n81944\t2.85\t-1.0\t1\n81945\t3.324\t0.0\t2\n81951\t3.151\t-1.0\t1\n81952\t3.185\t0.116\t25\n81957\t3.75\t0.247\t3\n81958\t3.579\t0.106\t3\n81967\t3.517\t0.128\t5\n81969\t3.26\t0.128\t3\n81970\t3.346\t0.13\t71\n81971\t3.246\t0.132\t128\n81975\t3.895\t0.219\t5\n81976\t3.719\t0.202\t10\n81997\t3.133\t0.007\t2\n81999\t3.088\t0.086\t20\n82000\t3.26\t0.098\t4\n82001\t3.332\t-1.0\t1\n82002\t3.427\t0.083\t10\n82003\t3.25\t0.171\t45\n82012\t3.805\t-1.0\t1\n82182\t3.212\t0.212\t411\n82438\t2.973\t0.227\t13\n86044\t2.357\t-1.0\t1\n86050\t3.301\t0.106\t5\n86051\t3.223\t0.408\t15\n86053\t3.906\t-1.0\t1\n86062\t2.711\t0.034\t2\n86068\t3.534\t0.111\t4\n86069\t3.014\t0.266\t86\n86277\t3.132\t0.206\t3\n86278\t3.08\t0.205\t1002\n86279\t2.704\t0.168\t460\n86280\t2.561\t0.163\t44\n86287\t3.292\t0.078\t26\n86288\t3.128\t0.068\t7\n86534\t2.902\t0.101\t13\n86535\t2.446\t0.18\t5\n90134\t3.158\t0.093\t9\n90138\t2.576\t0.185\t3\n90139\t2.835\t0.161\t13\n90140\t2.925\t0.174\t8\n90141\t3.761\t-1.0\t1\n90143\t3.058\t0.0\t2\n90144\t2.522\t0.784\t3\n90145\t2.482\t0.209\t46\n90146\t3.164\t0.558\t44\n90147\t2.774\t0.281\t113\n90150\t3.233\t0.034\t7\n90156\t3.193\t-1.0\t1\n90157\t2.921\t-1.0\t1\n90161\t3.271\t0.0\t2\n90162\t3.869\t0.023\t3\n90163\t3.573\t0.143\t28\n90164\t3.347\t0.022\t18\n90165\t2.62\t0.414\t54\n90168\t3.427\t0.036\t8\n90194\t3.897\t-1.0\t1\n90373\t3.171\t0.268\t3\n90374\t2.926\t0.185\t13105\n90375\t2.587\t0.178\t1875\n90376\t2.456\t0.185\t917\n90383\t3.113\t0.322\t22\n90384\t3.044\t0.167\t14\n90630\t2.624\t0.183\t73\n90631\t2.431\t0.217\t59\n90632\t2.541\t0.214\t13\n94231\t2.269\t0.138\t9\n94234\t2.503\t0.095\t6\n94235\t2.637\t0.592\t3\n94236\t3.236\t0.203\t2\n94242\t2.529\t0.487\t29\n94243\t2.969\t0.254\t17\n94264\t3.486\t-1.0\t1\n94286\t2.784\t0.083\t10\n94470\t2.664\t0.196\t1656\n94471\t2.46\t0.141\t1730\n94472\t2.373\t0.21\t563\n94479\t3.002\t0.084\t9\n94480\t2.809\t0.211\t72\n94726\t2.339\t0.204\t93\n94727\t2.461\t0.173\t28\n94728\t2.22\t0.033\t36\n98328\t2.585\t0.33\t139\n98329\t3.026\t0.483\t6\n98330\t3.053\t0.116\t7\n98331\t2.861\t0.157\t7\n98332\t3.016\t0.06\t2\n98333\t2.906\t-1.0\t1\n98334\t3.171\t-1.0\t1\n98335\t2.778\t0.078\t6\n98336\t2.83\t0.006\t4\n98337\t2.984\t0.301\t63\n98338\t3.557\t0.536\t97\n98339\t2.852\t0.31\t66\n98341\t3.47\t0.104\t3\n98349\t3.572\t-1.0\t1\n98350\t3.259\t0.092\t4\n98354\t3.11\t0.137\t19\n98355\t3.017\t0.193\t20\n98356\t3.203\t0.219\t43\n98357\t3.129\t0.376\t67\n98378\t3.04\t-1.0\t1\n98381\t3.568\t-1.0\t1\n98382\t3.15\t0.188\t5\n98383\t3.133\t0.037\t3\n98384\t3.045\t0.0\t2\n98386\t3.121\t0.097\t12\n98387\t3.325\t0.016\t2\n98565\t2.688\t0.12\t18\n98566\t2.619\t0.153\t6878\n98567\t2.404\t0.141\t2515\n98568\t2.41\t0.174\t638\n98575\t3.098\t0.146\t259\n98576\t2.876\t0.25\t37\n98822\t2.158\t0.152\t4143\n98823\t2.42\t0.131\t56\n98824\t2.468\t0.236\t5\n102425\t3.087\t0.258\t19\n102426\t2.867\t0.103\t4\n102427\t3.03\t0.131\t2\n102428\t2.886\t-1.0\t1\n102431\t3.326\t0.375\t7\n102432\t2.775\t0.168\t35\n102433\t2.737\t0.092\t14\n102434\t3.052\t0.275\t320\n102435\t2.947\t0.279\t501\n102438\t3.136\t0.442\t2\n102442\t3.198\t-1.0\t1\n102444\t2.998\t0.081\t3\n102445\t2.848\t0.318\t2\n102447\t3.116\t0.312\t4\n102449\t2.668\t0.007\t12\n102450\t2.65\t0.156\t37\n102451\t2.804\t0.128\t97\n102452\t3.71\t0.389\t143\n102453\t3.196\t0.352\t77\n102476\t2.993\t0.038\t4\n102478\t3.256\t-1.0\t1\n102479\t3.089\t0.272\t5\n102480\t2.61\t0.003\t4\n102481\t3.397\t-1.0\t1\n102482\t2.754\t0.035\t4\n102483\t2.878\t0.096\t25\n102661\t2.718\t0.111\t3\n102662\t2.482\t0.226\t2788\n102663\t2.511\t0.177\t8370\n102664\t2.493\t0.179\t2602\n102671\t2.643\t0.184\t45\n102672\t2.962\t0.176\t53\n102918\t2.052\t0.149\t4796\n102919\t2.51\t0.179\t433\n102920\t2.374\t0.161\t19\n106522\t2.941\t0.371\t219\n106523\t2.892\t0.261\t5\n106524\t2.877\t0.111\t7\n106525\t2.863\t0.149\t85\n106526\t2.98\t0.096\t34\n106527\t2.834\t0.242\t89\n106528\t2.743\t0.335\t128\n106529\t2.701\t0.254\t143\n106530\t2.739\t0.223\t715\n106531\t2.904\t0.242\t866\n106536\t3.002\t0.349\t8\n106537\t2.799\t0.229\t2\n106538\t3.086\t0.144\t4\n106540\t2.735\t0.302\t14\n106541\t2.768\t0.387\t2\n106542\t3.159\t0.157\t17\n106543\t2.781\t0.205\t11\n106545\t3.031\t0.266\t33\n106546\t3.013\t0.266\t96\n106547\t2.844\t0.229\t38\n106548\t2.92\t0.185\t202\n106549\t3.146\t0.267\t200\n106551\t3.701\t0.004\t2\n106552\t3.508\t0.08\t2\n106554\t3.692\t-1.0\t1\n106556\t3.882\t-1.0\t1\n106566\t3.665\t0.215\t7\n106570\t3.257\t0.114\t4\n106571\t2.863\t0.015\t4\n106572\t2.976\t0.164\t5\n106574\t3.224\t0.085\t11\n106575\t2.779\t0.247\t4\n106576\t2.981\t0.124\t18\n106577\t3.014\t0.072\t10\n106579\t3.124\t0.117\t28\n106757\t2.221\t0.408\t15\n106758\t2.401\t0.141\t81624\n106759\t2.351\t0.184\t21226\n106760\t2.431\t0.13\t2185\n106767\t2.782\t0.218\t230\n106768\t2.742\t0.183\t221\n107014\t2.077\t0.157\t13871\n107015\t2.286\t0.144\t1126\n107016\t2.261\t0.186\t6\n107023\t2.555\t-1.0\t1\n110619\t2.657\t0.367\t240\n110620\t2.816\t0.234\t12\n110621\t2.849\t0.093\t2\n110622\t2.914\t0.068\t12\n110623\t2.683\t0.337\t23\n110624\t2.349\t0.547\t34\n110625\t2.914\t0.287\t44\n110626\t2.37\t0.334\t254\n110627\t2.823\t0.245\t593\n110629\t3.216\t-1.0\t1\n110632\t3.168\t0.262\t18\n110633\t3.517\t-1.0\t1\n110634\t3.188\t-1.0\t1\n110636\t2.935\t0.025\t5\n110637\t2.659\t0.073\t4\n110638\t2.964\t0.14\t43\n110641\t2.808\t0.245\t6\n110642\t2.669\t0.36\t18\n110643\t2.899\t0.004\t2\n110644\t2.907\t0.266\t28\n110645\t3.111\t0.233\t239\n110654\t3.746\t0.042\t4\n110664\t3.213\t0.095\t2\n110665\t3.535\t-1.0\t1\n110666\t3.211\t-1.0\t1\n110669\t2.652\t-1.0\t1\n110670\t3.108\t0.335\t20\n110671\t3.078\t0.114\t48\n110672\t2.505\t0.007\t4\n110673\t2.66\t0.009\t3\n110674\t2.75\t0.014\t4\n110675\t2.821\t0.241\t50\n110682\t3.786\t-1.0\t1\n110684\t3.602\t0.211\t2\n110853\t2.51\t0.032\t24\n110854\t2.424\t0.18\t7855\n110855\t2.35\t0.165\t15280\n110856\t2.397\t0.151\t2600\n110863\t2.753\t0.315\t114\n110864\t2.718\t0.212\t256\n111110\t2.057\t0.162\t5916\n111111\t2.347\t0.183\t580\n111120\t3.119\t-1.0\t1\n114716\t2.861\t0.204\t423\n114717\t2.956\t0.144\t62\n114718\t3.07\t0.105\t55\n114719\t2.884\t0.235\t61\n114720\t2.67\t0.388\t26\n114721\t2.959\t0.219\t18\n114722\t2.832\t0.233\t173\n114723\t2.869\t0.229\t1269\n114727\t2.635\t0.109\t4\n114728\t2.753\t-1.0\t1\n114730\t2.796\t0.0\t2\n114733\t2.529\t0.017\t3\n114737\t2.923\t0.325\t25\n114738\t3.191\t0.145\t154\n114739\t3.085\t0.122\t9\n114740\t3.439\t0.421\t52\n114741\t3.061\t0.179\t190\n114743\t3.288\t-1.0\t1\n114745\t2.617\t0.062\t2\n114759\t3.289\t0.325\t3\n114760\t2.798\t-1.0\t1\n114762\t2.961\t-1.0\t1\n114765\t2.702\t-1.0\t1\n114766\t3.089\t0.135\t10\n114767\t3.047\t0.066\t62\n114768\t3.431\t0.011\t4\n114769\t3.198\t-1.0\t1\n114770\t3.331\t0.009\t16\n114771\t3.287\t0.091\t62\n114780\t3.048\t-1.0\t1\n114949\t2.438\t0.289\t13\n114950\t2.447\t0.207\t5143\n114951\t2.359\t0.153\t15318\n114952\t2.422\t0.153\t2635\n114959\t2.635\t0.231\t43\n114960\t2.669\t0.201\t795\n115206\t2.167\t0.159\t1651\n115207\t2.384\t0.147\t685\n115208\t2.377\t-1.0\t1\n118813\t2.913\t0.463\t3054\n118814\t2.839\t0.079\t11\n118815\t2.711\t0.001\t4\n118816\t2.916\t0.195\t36\n118817\t2.906\t0.257\t48\n118818\t3.003\t0.19\t1185\n118819\t2.89\t0.233\t2162\n118821\t4.015\t-1.0\t1\n118826\t2.96\t-1.0\t1\n118828\t2.952\t0.11\t39\n118829\t2.507\t0.024\t3\n118830\t2.938\t0.19\t22\n118831\t2.849\t0.2\t37\n118834\t2.765\t0.134\t305\n118835\t3.108\t0.155\t31\n118836\t3.39\t0.219\t189\n118837\t3.139\t0.235\t3644\n118839\t3.839\t-1.0\t1\n118858\t2.994\t-1.0\t1\n118861\t2.937\t0.024\t4\n118862\t2.749\t0.231\t11\n118863\t3.089\t0.14\t128\n118864\t2.73\t0.419\t10\n118867\t4.236\t-1.0\t1\n119046\t2.301\t0.205\t1784\n119047\t2.304\t0.132\t28601\n119048\t2.278\t0.16\t3943\n119055\t2.752\t0.18\t205\n119056\t2.764\t0.209\t1501\n119302\t2.425\t0.207\t1326\n119303\t2.302\t0.163\t1333\n119304\t2.514\t0.012\t2\n122910\t2.933\t0.511\t46\n122911\t2.702\t0.368\t7\n122912\t2.663\t0.513\t47\n122913\t2.561\t0.197\t22\n122914\t2.985\t0.238\t269\n122915\t2.799\t0.235\t1020\n122917\t3.448\t-1.0\t1\n122920\t3.428\t0.0\t2\n122922\t3.262\t0.063\t13\n122924\t3.141\t0.257\t36\n122925\t3.024\t0.038\t68\n122926\t3.08\t0.273\t30\n122929\t3.163\t0.018\t2\n122930\t2.751\t0.402\t11\n122932\t3.359\t0.441\t18\n122933\t3.021\t0.24\t382\n122935\t3.714\t-1.0\t1\n122955\t3.014\t0.026\t6\n122957\t3.238\t0.02\t4\n122958\t3.092\t0.105\t29\n122960\t3.074\t0.0\t2\n122961\t2.928\t0.007\t10\n122962\t3.518\t0.023\t8\n122963\t2.553\t0.0\t2\n123142\t2.737\t0.301\t850\n123143\t2.41\t0.137\t13630\n123144\t2.395\t0.157\t2260\n123151\t2.989\t0.176\t11\n123152\t2.86\t0.169\t504\n123398\t2.392\t0.233\t92\n123399\t2.348\t0.155\t241\n127007\t3.02\t0.246\t86\n127008\t2.974\t0.088\t8\n127009\t3.494\t0.451\t21\n127010\t2.829\t0.38\t37\n127011\t2.708\t0.245\t196\n127014\t3.469\t0.079\t7\n127016\t3.23\t0.0\t2\n127018\t3.049\t0.131\t23\n127020\t2.953\t0.101\t22\n127021\t3.009\t0.238\t45\n127022\t3.086\t0.158\t56\n127023\t3.058\t0.009\t4\n127025\t3.335\t0.196\t5\n127026\t3.283\t0.062\t7\n127027\t2.651\t0.769\t24\n127028\t3.434\t0.203\t18\n127029\t3.123\t0.373\t261\n127031\t3.789\t0.084\t6\n127032\t4.457\t-1.0\t1\n127033\t3.667\t-1.0\t1\n127036\t4.074\t0.0\t2\n127039\t4.048\t-1.0\t1\n127046\t4.093\t-1.0\t1\n127050\t2.953\t0.101\t7\n127052\t3.076\t0.218\t4\n127053\t3.176\t0.156\t3\n127054\t2.924\t0.124\t38\n127055\t2.909\t0.082\t10\n127059\t4.205\t0.525\t4\n127068\t3.948\t0.013\t2\n127237\t2.513\t0.281\t7\n127238\t2.422\t0.173\t417\n127239\t2.418\t0.176\t782\n127240\t2.39\t0.178\t153\n127247\t2.978\t0.075\t57\n127248\t2.775\t0.184\t61\n127494\t2.47\t0.44\t42\n127495\t2.406\t0.044\t2\n131109\t3.618\t0.424\t54\n131110\t3.321\t0.053\t36\n131111\t2.973\t0.025\t4\n131112\t3.771\t0.94\t5\n131113\t3.619\t0.341\t9\n131114\t3.143\t0.38\t30\n131116\t2.916\t0.159\t157\n131117\t2.917\t0.159\t121\n131118\t3.04\t0.268\t107\n131119\t2.922\t0.222\t15\n131120\t3.307\t0.038\t4\n131122\t3.411\t0.477\t35\n131123\t3.351\t-1.0\t1\n131127\t3.814\t0.114\t37\n131128\t3.49\t0.107\t29\n131130\t2.988\t0.0\t6\n131137\t2.864\t-1.0\t1\n131138\t3.615\t0.013\t2\n131144\t3.909\t0.169\t5\n131146\t3.079\t0.227\t59\n131147\t2.946\t0.309\t17\n131148\t2.938\t0.152\t48\n131149\t2.889\t0.075\t105\n131150\t2.932\t0.142\t86\n131151\t2.965\t0.315\t19\n131154\t4.158\t0.0\t3\n131155\t3.469\t0.0\t2\n135208\t3.392\t0.193\t9\n135209\t2.553\t0.506\t113\n135210\t2.846\t0.341\t42\n135212\t2.866\t0.212\t135\n135213\t2.927\t0.123\t16\n135214\t2.871\t0.163\t78\n135215\t2.95\t0.206\t41\n135216\t2.799\t0.488\t10\n135217\t2.858\t0.374\t47\n135218\t3.171\t0.414\t88\n135219\t3.525\t0.292\t6\n135226\t2.999\t0.0\t6\n135230\t3.926\t0.069\t11\n135234\t3.555\t0.069\t7\n135237\t4.014\t0.007\t2\n135240\t2.589\t0.032\t8\n135241\t2.464\t0.036\t10\n135242\t2.986\t0.141\t77\n135243\t2.924\t-1.0\t1\n135244\t2.773\t0.037\t5\n135245\t3.046\t0.0\t2\n135246\t2.825\t0.185\t175\n135247\t2.784\t0.162\t65\n135248\t3.475\t0.422\t13\n135249\t3.243\t0.492\t4\n135250\t3.627\t0.385\t15\n135251\t3.822\t0.36\t16\n135258\t3.001\t0.0\t2\n135260\t3.608\t0.228\t9\n135430\t2.312\t0.051\t4\n139303\t3.535\t0.292\t31\n139304\t3.38\t0.183\t24\n139305\t2.623\t0.529\t104\n139306\t3.083\t0.33\t321\n139308\t2.96\t0.162\t229\n139309\t2.923\t0.274\t63\n139310\t2.882\t0.149\t259\n139311\t3.27\t0.202\t438\n139312\t2.668\t0.417\t260\n139313\t3.241\t0.429\t63\n139314\t3.163\t0.399\t546\n139315\t3.191\t0.461\t290\n139321\t3.855\t0.226\t50\n139322\t3.709\t0.138\t100\n139323\t3.684\t0.123\t46\n139324\t3.633\t0.072\t70\n139326\t3.545\t0.123\t112\n139327\t3.991\t0.038\t4\n139328\t3.786\t0.526\t8\n139329\t3.979\t0.001\t4\n139330\t3.537\t0.014\t6\n139331\t3.355\t0.108\t12\n139332\t3.159\t0.173\t21\n139333\t3.624\t0.136\t27\n139334\t3.334\t0.167\t126\n139336\t3.239\t0.487\t10\n139337\t2.614\t0.35\t62\n139338\t3.123\t0.251\t218\n139339\t3.229\t0.309\t1334\n139340\t3.562\t0.717\t14\n139341\t2.945\t0.19\t49\n139342\t2.914\t0.165\t139\n139343\t2.866\t0.183\t165\n139344\t3.14\t0.274\t105\n139345\t3.078\t0.295\t50\n139346\t3.558\t0.371\t43\n139347\t3.512\t0.445\t86\n139354\t3.623\t0.348\t57\n139356\t3.581\t0.414\t207\n139357\t3.553\t0.26\t12\n139358\t3.738\t0.14\t26\n139526\t2.278\t0.056\t6\n143399\t2.881\t0.269\t29\n143400\t3.289\t0.207\t165\n143401\t2.903\t0.415\t136\n143402\t2.928\t0.263\t822\n143404\t2.981\t0.274\t225\n143405\t3.002\t0.373\t123\n143406\t2.929\t0.211\t1350\n143407\t3.15\t0.245\t392\n143408\t3.115\t0.214\t390\n143409\t2.892\t0.454\t277\n143410\t3.152\t0.327\t288\n143411\t3.118\t0.344\t421\n143417\t3.489\t0.306\t26\n143418\t3.66\t0.441\t102\n143419\t2.889\t0.02\t3\n143420\t3.311\t0.239\t14\n143422\t3.519\t0.117\t10\n143423\t3.405\t0.182\t14\n143424\t3.622\t0.201\t10\n143425\t2.902\t0.279\t14\n143426\t3.441\t0.397\t31\n143427\t2.869\t0.248\t16\n143428\t2.769\t0.111\t12\n143430\t3.342\t0.101\t8\n143433\t3.029\t0.413\t218\n143434\t2.63\t0.358\t237\n143435\t2.974\t0.223\t787\n143436\t2.871\t0.256\t91\n143437\t3.02\t0.255\t120\n143438\t2.97\t0.262\t730\n143439\t2.839\t0.219\t570\n143440\t2.939\t0.303\t600\n143441\t2.709\t0.226\t130\n143442\t3.393\t0.358\t470\n143443\t3.237\t0.326\t869\n143450\t3.569\t0.067\t21\n143452\t3.3\t0.463\t144\n143453\t2.771\t0.0\t6\n143454\t3.655\t0.021\t3\n143455\t2.882\t0.023\t3\n151589\t4.027\t0.433\t171\n151590\t3.818\t0.355\t2\n151594\t3.871\t0.0\t2\n151598\t3.586\t0.162\t20\n151599\t3.353\t0.049\t2\n151601\t3.937\t0.0\t2\n151602\t3.839\t0.296\t15\n151603\t3.859\t0.378\t36\n151607\t4.188\t0.029\t2\n151631\t3.626\t0.064\t26\n151633\t3.814\t0.025\t4\n151634\t3.849\t0.071\t10\n151635\t4.057\t0.0\t2\n151814\t3.815\t0.247\t64\n152069\t3.392\t0.0\t2\n152070\t3.728\t0.185\t7\n155686\t3.715\t0.263\t197\n155692\t3.291\t0.0\t2\n155693\t3.275\t0.0\t2\n155695\t3.371\t-1.0\t1\n155696\t3.544\t0.083\t4\n155697\t3.415\t0.077\t36\n155698\t3.446\t0.074\t26\n155699\t3.346\t0.137\t70\n155703\t4.039\t0.114\t2\n155704\t3.785\t-1.0\t1\n155725\t3.3\t0.0\t4\n155726\t3.174\t0.064\t8\n155727\t3.236\t0.051\t4\n155728\t3.411\t0.02\t2\n155729\t3.492\t0.006\t3\n155730\t3.53\t0.066\t10\n155731\t3.424\t0.132\t88\n155740\t3.534\t-1.0\t1\n155910\t3.584\t0.272\t112\n156166\t2.793\t0.019\t4\n159783\t4.713\t0.009\t2\n159790\t3.627\t0.388\t4\n159795\t3.832\t0.039\t12\n159797\t3.719\t0.347\t56\n159799\t4.168\t0.0\t2\n159822\t3.558\t0.003\t2\n160005\t3.476\t0.074\t35\n160006\t3.258\t0.191\t4054\n160007\t2.913\t0.142\t1412\n160008\t2.796\t0.249\t708\n160015\t3.636\t0.026\t3\n160016\t3.367\t0.408\t27\n160262\t2.99\t0.127\t72\n160263\t3.031\t0.264\t24\n163884\t3.712\t0.084\t3\n163885\t3.713\t0.26\t5\n163889\t3.399\t0.0\t2\n163890\t3.989\t0.377\t5\n163892\t3.476\t0.0\t2\n163893\t3.302\t0.424\t87\n163917\t3.364\t0.045\t5\n163918\t3.314\t0.355\t6\n163922\t4.034\t-1.0\t1\n164101\t3.255\t0.125\t9\n164102\t3.029\t0.219\t13290\n164103\t2.812\t0.186\t1007\n164104\t2.599\t0.231\t718\n164111\t3.355\t0.275\t19\n164112\t3.184\t0.119\t22\n164358\t2.805\t0.151\t103\n164359\t2.642\t0.21\t22\n164360\t2.844\t0.216\t9\n167977\t2.746\t0.433\t12\n167986\t3.166\t-1.0\t1\n167988\t3.668\t0.819\t15\n167989\t2.936\t0.216\t20\n168198\t2.975\t0.237\t953\n168199\t2.749\t0.24\t450\n168200\t2.532\t0.296\t44\n168207\t3.165\t0.13\t49\n168208\t3.318\t0.08\t14\n168454\t2.601\t0.169\t111\n168455\t2.727\t0.161\t25\n172074\t2.912\t0.373\t1298\n172076\t3.157\t0.34\t9\n172078\t3.301\t0.025\t5\n172079\t3.222\t0.067\t8\n172082\t3.28\t0.259\t30\n172083\t3.177\t0.165\t15\n172084\t3.327\t0.193\t23\n172085\t3.272\t0.292\t557\n172087\t3.799\t-1.0\t1\n172088\t3.528\t-1.0\t1\n172106\t2.727\t0.462\t6\n172107\t3.735\t-1.0\t1\n172109\t3.252\t0.082\t43\n172110\t3.239\t0.183\t9\n172111\t3.197\t0.132\t22\n172112\t3.584\t0.076\t4\n172114\t3.693\t0.232\t12\n172115\t3.359\t0.28\t40\n172124\t3.744\t0.013\t2\n172293\t3.03\t0.256\t14\n172294\t2.789\t0.167\t9278\n172295\t2.577\t0.18\t3098\n172296\t2.467\t0.259\t1296\n172303\t2.908\t0.239\t150\n172304\t2.957\t0.217\t165\n172550\t2.341\t0.159\t4663\n172551\t2.511\t0.149\t218\n172559\t2.202\t0.279\t10\n180268\t3.063\t0.299\t1102\n180269\t2.945\t0.515\t10\n180270\t3.082\t0.36\t14\n180271\t3.086\t-1.0\t1\n180272\t3.242\t0.131\t2\n180273\t3.439\t0.268\t6\n180274\t3.072\t0.196\t129\n180275\t3.213\t0.211\t24\n180276\t3.091\t0.153\t98\n180277\t3.22\t0.208\t173\n180295\t3.709\t0.109\t4\n180296\t3.674\t0.024\t2\n180298\t3.338\t0.368\t8\n180299\t3.08\t0.02\t2\n180300\t3.051\t0.136\t6\n180301\t3.2\t0.237\t31\n180302\t3.182\t0.186\t59\n180303\t2.975\t0.164\t24\n180304\t3.255\t-1.0\t1\n180306\t2.931\t0.041\t12\n180307\t2.734\t0.077\t20\n180485\t2.755\t0.204\t14\n180486\t2.611\t0.146\t28221\n180487\t2.363\t0.139\t11820\n180488\t2.408\t0.153\t1326\n180495\t2.75\t0.202\t77\n180496\t2.828\t0.175\t621\n180742\t2.163\t0.16\t10524\n180743\t2.394\t0.13\t880\n180744\t2.447\t0.224\t6\n180751\t2.688\t0.012\t5\n180752\t2.464\t0.284\t5\n184365\t2.871\t0.285\t392\n184366\t3.302\t0.52\t4\n184367\t2.634\t0.029\t2\n184368\t2.701\t-1.0\t1\n184369\t3.216\t0.427\t13\n184370\t3.333\t0.264\t124\n184371\t3.063\t0.086\t29\n184372\t3.343\t0.188\t5\n184373\t3.193\t0.194\t327\n184396\t2.794\t0.148\t26\n184397\t2.962\t0.816\t2\n184398\t2.985\t0.39\t19\n184399\t3.255\t0.176\t61\n184400\t3.215\t0.059\t6\n184402\t3.402\t0.003\t2\n184403\t3.348\t0.247\t21\n184412\t3.283\t0.005\t3\n184581\t2.634\t0.295\t6\n184582\t2.568\t0.171\t12855\n184583\t2.441\t0.153\t3081\n184584\t2.416\t0.166\t1225\n184591\t2.739\t0.094\t61\n184592\t2.805\t0.202\t148\n184837\t2.886\t-1.0\t1\n184838\t2.196\t0.199\t2670\n184839\t2.45\t0.148\t279\n184848\t2.287\t0.009\t6\n188462\t3.197\t0.284\t549\n188463\t3.567\t0.174\t71\n188465\t3.129\t0.275\t2\n188466\t3.378\t0.343\t81\n188467\t3.278\t0.49\t122\n188468\t3.121\t0.318\t140\n188469\t3.118\t0.179\t936\n188472\t3.155\t0.189\t11\n188486\t2.788\t-1.0\t1\n188487\t3.697\t0.043\t2\n188490\t3.361\t-1.0\t1\n188491\t3.319\t0.01\t2\n188492\t3.071\t0.101\t16\n188494\t3.253\t0.286\t17\n188495\t3.308\t0.347\t126\n188496\t3.082\t0.313\t5\n188497\t3.342\t0.379\t39\n188498\t3.685\t0.035\t11\n188499\t2.981\t0.028\t91\n188508\t3.265\t0.257\t3\n188677\t2.545\t0.218\t11\n188678\t2.381\t0.169\t6266\n188679\t2.365\t0.135\t7580\n188680\t2.446\t0.147\t1073\n188687\t2.778\t0.159\t82\n188688\t2.732\t0.177\t615\n188934\t2.405\t0.21\t1341\n188935\t2.38\t0.131\t281\n188936\t2.152\t0.004\t2\n188944\t2.657\t0.082\t9\n192559\t3.554\t0.226\t4083\n192561\t2.716\t-1.0\t1\n192562\t3.443\t0.292\t14\n192563\t3.163\t0.241\t12\n192564\t3.698\t0.229\t62\n192565\t3.522\t0.275\t796\n192572\t3.827\t0.068\t3\n192587\t3.3\t0.096\t6\n192588\t3.119\t0.06\t30\n192589\t3.417\t0.237\t27\n192590\t3.436\t0.273\t279\n192591\t3.326\t0.24\t399\n192592\t2.823\t0.436\t7\n192774\t2.472\t0.2\t1611\n192775\t2.535\t0.173\t5301\n192776\t2.731\t0.17\t305\n192783\t2.898\t0.163\t18\n192784\t3.194\t0.23\t847\n193030\t2.719\t0.257\t1011\n193031\t2.571\t0.187\t351\n196656\t2.962\t0.143\t4\n196659\t2.984\t0.064\t32\n196660\t3.488\t0.2\t53\n196661\t3.212\t0.24\t422\n196664\t3.568\t0.009\t4\n196686\t3.302\t0.12\t162\n196687\t3.375\t0.136\t15\n196690\t3.639\t0.028\t8\n196691\t3.016\t0.052\t32\n196870\t2.546\t0.218\t32\n196871\t2.669\t0.125\t3379\n196872\t2.711\t0.152\t719\n196880\t3.123\t0.225\t349\n197126\t2.828\t0.381\t77\n197127\t2.606\t0.119\t56\n197128\t2.751\t-1.0\t1\n200753\t2.968\t0.291\t109\n200754\t3.781\t0.18\t17\n200755\t2.926\t0.006\t4\n200756\t3.55\t0.394\t50\n200757\t3.215\t0.314\t179\n200759\t3.813\t0.152\t8\n200760\t3.68\t0.079\t15\n200778\t3.2\t0.021\t2\n200779\t2.798\t0.024\t24\n200781\t2.816\t0.38\t2\n200782\t3.146\t0.138\t51\n200783\t3.783\t-1.0\t1\n200784\t3.608\t0.06\t2\n200787\t3.701\t0.063\t13\n200966\t2.737\t0.248\t164\n200967\t2.638\t0.193\t666\n200968\t2.735\t0.196\t128\n200975\t3.61\t0.145\t5\n200976\t2.787\t0.336\t60\n201222\t2.754\t0.18\t12\n201223\t2.897\t0.149\t12\n204850\t3.629\t0.476\t976\n204851\t2.793\t0.054\t34\n204852\t3.367\t0.645\t98\n204853\t3.325\t0.463\t310\n204855\t3.996\t0.344\t33\n204856\t3.682\t0.08\t11\n204857\t3.947\t0.023\t6\n204858\t3.186\t0.0\t6\n204863\t3.44\t0.0\t11\n204866\t3.929\t-1.0\t1\n204870\t3.525\t0.118\t8\n204872\t4.135\t0.14\t3\n204874\t3.301\t0.205\t57\n204875\t3.121\t0.234\t55\n204876\t3.203\t0.168\t70\n204877\t3.061\t0.08\t42\n204878\t3.284\t0.173\t281\n204879\t4.309\t0.589\t23\n204881\t3.669\t0.346\t5\n204882\t3.46\t0.055\t10\n204883\t3.636\t0.704\t18\n204892\t4.537\t-1.0\t1\n205061\t2.753\t0.308\t17\n205062\t2.528\t0.15\t3869\n205063\t2.708\t0.199\t1070\n205064\t2.716\t0.2\t545\n205071\t3.421\t0.246\t38\n205072\t3.123\t0.212\t212\n205318\t2.574\t0.174\t88\n205319\t2.626\t0.158\t25\n208947\t2.969\t0.506\t259\n208948\t3.77\t0.671\t12\n208949\t3.392\t0.45\t229\n208951\t3.952\t0.118\t44\n208952\t3.545\t0.115\t138\n208953\t3.197\t0.0\t6\n208954\t3.176\t0.0\t6\n208962\t3.825\t0.039\t12\n208968\t3.44\t-1.0\t1\n208970\t3.273\t0.166\t30\n208971\t3.245\t0.214\t83\n208972\t2.968\t0.115\t113\n208973\t3.43\t0.067\t4\n208974\t3.1\t0.205\t80\n208975\t3.122\t0.204\t60\n208976\t3.551\t0.081\t6\n208977\t4.668\t0.032\t6\n208978\t4.088\t0.033\t12\n208979\t2.917\t0.271\t11\n209158\t2.495\t0.15\t1844\n209159\t2.721\t0.28\t225\n209160\t2.668\t0.436\t78\n209167\t3.017\t0.046\t5\n209168\t3.149\t0.165\t38\n209414\t2.509\t0.24\t10\n209415\t2.427\t0.146\t6\n213050\t4.433\t0.045\t6\n213054\t4.024\t0.099\t10\n213065\t5.027\t-1.0\t1\n213066\t2.704\t0.455\t23\n213067\t3.276\t0.286\t165\n213068\t3.025\t0.034\t20\n213070\t3.155\t0.284\t48\n213071\t3.03\t0.209\t38\n213072\t3.343\t0.249\t92\n213073\t4.446\t0.01\t2\n213074\t4.01\t0.314\t16\n213075\t3.489\t0.653\t32\n213084\t3.85\t0.333\t22\n213086\t4.355\t0.043\t6\n217145\t3.893\t0.312\t115\n217146\t4.057\t0.272\t72\n217147\t3.309\t0.409\t38\n217148\t3.875\t0.23\t104\n217150\t3.913\t0.338\t122\n217151\t4.078\t0.166\t14\n217152\t3.772\t0.324\t18\n217153\t3.606\t0.456\t25\n217154\t3.192\t0.524\t27\n217156\t3.103\t0.413\t22\n217157\t3.863\t0.489\t46\n217158\t3.761\t0.157\t82\n217159\t3.66\t0.413\t6\n217160\t3.35\t0.322\t22\n217161\t2.816\t0.4\t52\n217162\t2.801\t0.247\t341\n217163\t2.814\t0.353\t94\n217164\t2.999\t0.191\t26\n217165\t3.222\t0.232\t233\n217166\t3.234\t0.277\t811\n217167\t3.066\t0.296\t281\n217168\t3.131\t0.252\t665\n217169\t2.89\t0.413\t19\n217170\t3.801\t0.427\t565\n217171\t3.582\t0.409\t827\n217178\t3.927\t0.242\t20\n217180\t3.819\t0.264\t406\n217182\t3.899\t0.189\t7\n217183\t3.278\t-1.0\t1\n225335\t4.276\t0.49\t265\n225336\t4.479\t0.154\t3\n225338\t4.3\t0.0\t2\n225345\t4.243\t-1.0\t1\n225346\t4.182\t-1.0\t1\n225354\t3.789\t-1.0\t1\n225359\t3.687\t0.153\t5\n225360\t4.965\t0.0\t3\n225361\t4.086\t0.077\t5\n225362\t3.938\t0.124\t17\n225363\t4.209\t0.153\t5\n225542\t4.026\t0.331\t148\n225797\t3.377\t0.162\t3\n225798\t3.864\t0.248\t33\n229432\t4.077\t0.334\t194\n229450\t3.552\t0.0\t2\n229455\t3.61\t0.0\t2\n229458\t3.779\t0.076\t5\n229459\t3.62\t0.105\t15\n229638\t3.726\t0.142\t169\n229893\t2.88\t0.148\t17\n233547\t3.663\t0.041\t4\n233733\t3.661\t0.127\t9\n233734\t3.511\t0.169\t1320\n233735\t3.103\t0.143\t699\n233736\t2.952\t0.191\t544\n233743\t4.137\t0.779\t6\n233744\t3.541\t0.191\t21\n233990\t3.356\t0.17\t18\n233991\t3.128\t0.157\t55\n237830\t3.517\t0.262\t866\n237831\t3.053\t0.238\t780\n237832\t2.942\t0.272\t364\n237839\t3.058\t0.078\t3\n237840\t3.557\t0.208\t14\n238086\t3.309\t0.197\t15\n238087\t3.084\t0.184\t40\n241926\t3.477\t0.124\t197\n241927\t3.016\t0.163\t300\n241928\t2.871\t0.151\t320\n241936\t3.545\t0.074\t6\n242182\t3.213\t0.142\t3\n242183\t3.034\t0.049\t17\n246021\t3.749\t-1.0\t1\n246022\t3.442\t0.199\t803\n246023\t3.027\t0.153\t733\n246024\t2.83\t0.16\t737\n246031\t3.715\t0.509\t10\n246032\t3.489\t0.117\t10\n246278\t3.07\t0.175\t7\n246279\t3.018\t0.113\t65\n254014\t3.621\t0.145\t11\n254027\t3.568\t0.017\t3\n254214\t3.39\t0.158\t2671\n254215\t2.996\t0.161\t998\n254216\t2.805\t0.192\t556\n254223\t3.737\t0.252\t40\n254224\t3.439\t0.148\t13\n254470\t3.227\t0.142\t6\n254471\t3.139\t0.173\t24\n254472\t2.911\t0.206\t20\n258111\t2.215\t0.23\t2\n258310\t3.466\t0.262\t271\n258311\t2.927\t0.187\t1239\n258312\t2.76\t0.166\t1083\n258320\t3.383\t0.05\t12\n258566\t3.457\t0.042\t4\n258567\t2.819\t0.112\t11\n262406\t3.329\t0.213\t453\n262407\t2.926\t0.184\t889\n262408\t2.796\t0.186\t1231\n262416\t3.355\t0.381\t10\n262662\t2.919\t0.044\t2\n262663\t2.999\t0.226\t11\n266502\t3.401\t0.179\t388\n266503\t2.89\t0.17\t992\n266504\t2.734\t0.193\t1159\n266511\t3.732\t-1.0\t1\n266512\t3.445\t0.0\t2\n266758\t2.926\t-1.0\t1\n266759\t2.857\t0.231\t31\n270597\t3.404\t0.076\t3\n270598\t3.222\t0.253\t1595\n270599\t2.866\t0.193\t2042\n270600\t2.707\t0.189\t2073\n270607\t3.718\t-1.0\t1\n270608\t3.41\t0.38\t13\n270854\t2.9\t-1.0\t1\n270855\t2.794\t0.192\t30\n274510\t4.001\t0.0\t2\n274694\t3.388\t0.171\t354\n274695\t2.87\t0.172\t416\n274696\t2.724\t0.185\t439\n274950\t2.925\t0.039\t8\n274951\t2.479\t0.253\t12\n278789\t3.409\t0.069\t2\n278790\t3.158\t0.169\t1145\n278791\t2.849\t0.175\t801\n278792\t2.672\t0.179\t577\n278799\t3.625\t-1.0\t1\n278800\t3.474\t0.092\t7\n279046\t2.934\t0.097\t12\n279047\t2.89\t0.09\t12\n282693\t3.553\t0.0\t4\n282886\t3.443\t0.178\t536\n282887\t2.879\t0.209\t168\n282888\t2.725\t0.147\t129\n282895\t3.631\t0.142\t21\n282896\t3.222\t0.01\t8\n283143\t2.527\t0.136\t9\n286981\t3.507\t0.084\t5\n286982\t3.184\t0.155\t3390\n286983\t2.859\t0.153\t1245\n286984\t2.682\t0.159\t499\n286991\t3.54\t0.155\t13\n286992\t3.365\t0.159\t9\n287238\t3.019\t0.138\t24\n287239\t2.855\t0.076\t16\n287240\t2.979\t0.21\t26\n290891\t3.455\t0.011\t2\n290894\t3.514\t-1.0\t1\n291077\t3.476\t0.16\t11\n291078\t3.284\t0.204\t878\n291079\t2.84\t0.187\t460\n291080\t2.635\t0.193\t132\n291088\t3.528\t0.062\t2\n291334\t2.883\t0.06\t12\n291335\t2.948\t0.137\t14\n294994\t3.999\t-1.0\t1\n295173\t3.165\t0.193\t2\n295174\t3.107\t0.177\t2432\n295175\t2.774\t0.182\t341\n295176\t2.652\t0.204\t104\n295184\t3.109\t0.166\t9\n295430\t2.767\t0.112\t12\n295431\t2.657\t0.278\t12\n299081\t2.935\t0.369\t17\n299085\t3.119\t0.214\t10\n299270\t2.918\t0.218\t1400\n299271\t2.737\t0.199\t305\n299272\t2.492\t0.248\t40\n299279\t2.559\t0.037\t4\n299526\t2.488\t0.241\t28\n299527\t2.804\t0.155\t6\n303178\t2.675\t0.417\t434\n303181\t3.222\t0.078\t41\n303182\t3.654\t-1.0\t1\n303183\t3.21\t0.023\t4\n303186\t3.733\t0.044\t8\n303187\t3.128\t0.192\t8\n303365\t2.968\t0.028\t2\n303366\t2.8\t0.213\t3774\n303367\t2.584\t0.227\t2258\n303368\t2.49\t0.297\t312\n303375\t2.904\t0.202\t124\n303376\t3.053\t0.272\t32\n303622\t2.342\t0.159\t6180\n303623\t2.471\t0.154\t65\n307275\t3.127\t0.383\t1161\n307276\t3.076\t0.401\t28\n307277\t3.191\t0.02\t2\n307278\t3.567\t0.117\t38\n307279\t3.269\t0.094\t25\n307280\t3.228\t0.038\t20\n307281\t3.587\t0.03\t2\n307283\t3.205\t0.081\t30\n307292\t3.713\t0.164\t4\n307462\t2.629\t0.197\t1542\n307463\t2.457\t0.127\t3999\n307464\t2.435\t0.206\t207\n307471\t2.732\t0.251\t11\n307472\t2.825\t0.122\t121\n307718\t2.178\t0.135\t8471\n307719\t2.383\t0.156\t157\n307720\t2.299\t0.121\t3\n307728\t2.409\t0.014\t6\n311372\t3.106\t0.207\t804\n311373\t3.181\t-1.0\t1\n311374\t3.548\t0.146\t32\n311375\t3.12\t0.036\t62\n311379\t2.887\t0.06\t22\n311558\t2.597\t0.187\t2089\n311559\t2.357\t0.157\t1187\n311560\t2.486\t0.163\t95\n311567\t3.048\t0.26\t4\n311568\t3.082\t0.291\t44\n311814\t2.108\t0.152\t4572\n311815\t2.428\t0.139\t86\n311824\t2.543\t0.001\t4\n315469\t3.151\t0.278\t365\n315470\t3.549\t0.122\t63\n315471\t3.157\t0.118\t31\n315475\t3.22\t0.061\t39\n315653\t2.65\t0.215\t20\n315654\t2.552\t0.195\t16080\n315655\t2.382\t0.126\t6084\n315656\t2.483\t0.127\t418\n315663\t2.762\t0.14\t37\n315664\t2.816\t0.196\t162\n315910\t2.215\t0.168\t2203\n315911\t2.436\t0.105\t204\n315920\t2.977\t-1.0\t1\n319566\t3.225\t0.248\t552\n319567\t3.367\t0.293\t115\n319568\t3.169\t0.205\t20\n319569\t3.36\t0.238\t50\n319570\t3.392\t0.286\t42\n319571\t2.929\t0.286\t8\n319580\t2.758\t-1.0\t1\n319749\t2.528\t0.434\t19\n319750\t2.374\t0.187\t4743\n319751\t2.364\t0.145\t6282\n319752\t2.404\t0.169\t399\n319759\t2.743\t0.15\t63\n319760\t2.674\t0.189\t638\n320005\t2.585\t0.095\t3\n320006\t2.331\t0.173\t2612\n320007\t2.357\t0.121\t135\n320015\t2.781\t0.006\t2\n320016\t2.898\t0.124\t6\n323663\t3.245\t0.336\t4562\n323664\t3.526\t0.142\t2\n323665\t3.286\t0.102\t22\n323666\t3.306\t0.022\t2\n323667\t2.881\t0.196\t25\n323846\t2.35\t0.15\t3014\n323847\t2.354\t0.137\t1158\n323848\t2.403\t0.253\t6\n323855\t2.769\t0.169\t45\n323856\t2.739\t0.213\t222\n324102\t2.337\t0.19\t1691\n324103\t2.388\t0.156\t45\n327760\t2.694\t0.474\t53\n327941\t2.577\t-1.0\t1\n327942\t2.266\t0.136\t922\n327943\t2.741\t0.157\t953\n327944\t2.741\t0.235\t37\n327951\t2.904\t0.093\t8\n327952\t2.866\t0.242\t158\n328198\t2.38\t0.168\t94\n328199\t2.875\t0.15\t3\n331857\t3.111\t0.21\t60\n332038\t2.542\t0.227\t19\n332039\t2.984\t0.13\t175\n332040\t2.942\t0.295\t20\n332048\t3.399\t0.21\t18\n332294\t2.174\t0.0\t4\n335954\t3.231\t0.429\t436\n336134\t2.776\t0.248\t252\n336135\t2.833\t0.135\t693\n336136\t2.76\t0.196\t207\n336143\t3.678\t0.123\t15\n336144\t3.263\t0.218\t66\n336390\t2.874\t0.122\t4\n336391\t2.699\t0.147\t7\n340051\t3.126\t0.27\t404\n340229\t2.797\t-1.0\t1\n340230\t2.623\t0.24\t807\n340231\t2.807\t0.169\t491\n340232\t2.771\t0.212\t269\n340239\t3.206\t0.012\t2\n340240\t3.325\t0.293\t98\n340486\t2.701\t0.167\t16\n340487\t3.129\t0.34\t2\n368902\t3.528\t0.229\t2516\n368903\t3.077\t0.237\t337\n368904\t2.966\t0.304\t84\n368911\t3.616\t0.121\t7\n368912\t3.535\t0.376\t64\n369158\t3.14\t0.113\t14\n369159\t3.146\t0.323\t8\n369160\t3.179\t0.006\t2\n376924\t3.747\t0.0\t2\n377094\t3.44\t0.227\t3394\n377095\t2.982\t0.201\t1971\n377096\t2.826\t0.242\t770\n377103\t2.998\t0.426\t10\n377104\t3.462\t0.179\t110\n377350\t3.258\t0.302\t74\n377351\t3.02\t0.219\t96\n377352\t3.011\t0.211\t9\n381190\t3.595\t0.186\t49\n381191\t3.016\t0.171\t33\n381192\t3.071\t0.23\t47\n381200\t3.429\t0.053\t8\n385286\t3.676\t0.046\t15\n385287\t3.005\t0.218\t31\n385288\t3.065\t0.263\t47\n385296\t3.84\t0.251\t6\n389383\t3.107\t0.032\t5\n389384\t2.858\t0.234\t18\n389392\t3.517\t0.038\t3\n393479\t3.128\t0.008\t2\n393488\t3.419\t0.064\t3\n397575\t3.05\t0.029\t3\n397576\t2.942\t0.008\t3\n401671\t3.038\t0.025\t3\n401672\t2.931\t0.016\t3\n1073414\t1.594\t0.077\t2000853\n1073415\t1.527\t0.071\t91806\n1073423\t2.001\t0.161\t74\n1073424\t1.981\t0.067\t42\n1073679\t2.05\t0.093\t4\n1073680\t1.992\t0.039\t7\n1077511\t1.504\t0.087\t791\n1077519\t1.839\t0.106\t1862\n1077520\t1.811\t0.097\t38\n1077776\t1.753\t0.11\t23\n1110287\t2.325\t0.221\t8\n16781317\t1.299\t0.138\t2655\n16781318\t1.157\t0.073\t2154155\n16781319\t1.027\t0.087\t120477\n16781320\t1.0\t0.102\t99234\n16781321\t1.229\t0.768\t39\n16781326\t1.698\t0.206\t492\n16781327\t1.491\t0.16\t461\n16781328\t1.414\t0.148\t260\n16781329\t1.46\t0.267\t16\n16781344\t1.768\t0.207\t170\n16781345\t1.64\t0.233\t13\n16781346\t1.184\t0.24\t11\n16781347\t1.193\t0.14\t10\n16781365\t1.612\t0.001\t2\n16781573\t1.266\t-1.0\t1\n16781574\t1.083\t0.054\t1399983\n16781575\t1.026\t0.08\t10086\n16781576\t1.035\t0.083\t62\n16781583\t1.555\t0.169\t170\n16781830\t1.082\t0.064\t1503\n16781831\t1.012\t0.075\t9\n16797701\t2.055\t0.175\t200\n16797702\t1.917\t0.136\t1968\n16797703\t1.74\t0.145\t4307\n16797704\t1.627\t0.111\t7838\n16797705\t1.579\t0.095\t5410\n16797710\t2.643\t0.184\t32\n16797711\t2.367\t0.191\t736\n16797712\t2.279\t0.266\t207\n16797713\t2.22\t0.188\t513\n16797728\t2.564\t0.128\t21\n16797729\t2.481\t0.101\t5\n16797730\t2.577\t0.148\t48\n16797731\t2.401\t0.231\t244\n16797748\t2.725\t0.14\t4\n16797749\t2.87\t0.374\t77\n16797958\t1.838\t0.106\t11969\n16797959\t1.825\t0.108\t1915\n16797967\t2.268\t0.255\t20\n16798214\t1.868\t0.127\t268\n16798215\t1.772\t0.141\t42\n16798224\t2.259\t-1.0\t1\n16801798\t1.819\t0.111\t650683\n16801799\t1.721\t0.104\t153277\n16801800\t1.674\t0.089\t194202\n16801801\t1.483\t0.083\t37441\n16801806\t2.363\t0.237\t31159\n16801807\t2.243\t0.189\t5173\n16801808\t2.128\t0.132\t11654\n16801809\t2.024\t0.124\t15529\n16801824\t2.527\t0.248\t999\n16801825\t2.422\t0.325\t206\n16801826\t2.378\t0.18\t872\n16801827\t2.279\t0.155\t2704\n16801844\t2.666\t0.308\t172\n16801845\t2.522\t0.166\t717\n16802054\t1.762\t0.092\t352183\n16802055\t1.72\t0.097\t14388\n16802056\t1.688\t0.108\t6\n16802063\t2.207\t0.172\t3307\n16802064\t2.105\t0.131\t3350\n16802310\t1.696\t0.091\t7665\n16802311\t1.656\t0.128\t75\n16802320\t2.082\t0.116\t4919\n16805895\t1.623\t0.108\t3208\n16805896\t1.666\t0.126\t2100\n16805897\t1.415\t0.162\t17\n16805902\t2.205\t0.191\t1967\n16805903\t2.079\t0.152\t1682\n16805904\t1.93\t0.103\t249\n16805905\t1.958\t0.111\t24\n16805920\t2.362\t0.195\t1315\n16805921\t2.241\t0.173\t128\n16805922\t2.3\t0.193\t96\n16805923\t2.162\t0.118\t32\n16805940\t2.674\t0.201\t118\n16805941\t2.473\t0.194\t38\n16806150\t1.59\t0.082\t164495\n16806151\t1.574\t0.081\t7627\n16806159\t1.986\t0.139\t3197\n16806160\t2.009\t0.09\t401\n16806406\t1.512\t0.069\t153\n16806407\t1.394\t0.086\t533\n16806416\t1.9\t0.093\t9234\n16809992\t1.741\t0.264\t647\n16809998\t1.995\t0.155\t7813\n16809999\t1.925\t0.144\t1208\n16810000\t1.916\t0.201\t51\n16810001\t1.673\t0.087\t998\n16810016\t2.152\t0.242\t1021\n16810017\t1.796\t0.244\t670\n16810018\t2.006\t0.2\t174\n16810019\t1.965\t0.381\t4\n16810036\t2.404\t0.302\t460\n16810037\t2.386\t0.22\t205\n16810246\t1.562\t0.08\t163577\n16810247\t1.393\t0.102\t21110\n16810255\t1.861\t0.129\t9068\n16810256\t1.926\t0.104\t123\n16810502\t1.408\t0.07\t41\n16810503\t1.354\t0.073\t7\n16810512\t1.734\t0.114\t5958\n16814089\t1.446\t0.064\t5\n16814094\t1.979\t0.162\t679\n16814095\t1.82\t0.095\t9307\n16814096\t1.773\t0.108\t625\n16814097\t1.615\t0.213\t20\n16814112\t2.054\t0.303\t209\n16814113\t1.845\t0.169\t415\n16814114\t2.218\t0.193\t16\n16814115\t1.836\t0.098\t23\n16814132\t1.837\t0.254\t287\n16814133\t2.14\t0.294\t84\n16814342\t1.478\t0.072\t44091\n16814343\t1.246\t0.169\t3\n16814351\t1.618\t0.194\t150\n16814352\t1.613\t0.103\t16\n16814608\t1.556\t0.127\t44\n16834574\t3.378\t0.364\t2682\n16834575\t2.882\t0.272\t162\n16834576\t2.626\t0.23\t157\n16834577\t2.549\t0.274\t1011\n16834592\t3.473\t0.286\t507\n16834593\t3.314\t0.371\t24\n16834594\t2.897\t0.327\t42\n16834595\t2.87\t0.389\t169\n16834612\t3.665\t0.536\t15\n16834613\t3.09\t0.417\t78\n16834822\t2.235\t0.149\t6897\n16834823\t2.299\t0.166\t545\n16835078\t2.237\t0.123\t1405\n16835079\t2.102\t0.162\t46\n16838671\t2.751\t0.246\t664\n16838672\t2.368\t0.224\t165\n16838673\t2.447\t0.27\t425\n16838688\t3.18\t0.313\t89\n16838689\t2.773\t0.119\t26\n16838690\t2.582\t0.333\t127\n16838691\t2.563\t0.288\t58\n16838708\t3.027\t0.178\t12\n16838709\t2.915\t0.394\t101\n16838918\t2.115\t0.115\t8657\n16838919\t1.855\t0.183\t439\n16838927\t2.715\t0.169\t135\n16838928\t2.658\t0.069\t6\n16839174\t1.936\t0.185\t23\n16839175\t2.101\t0.064\t2\n16839184\t2.948\t-1.0\t1\n16842768\t2.281\t0.201\t1234\n16842769\t2.015\t0.03\t13\n16842784\t2.754\t0.331\t616\n16842785\t2.241\t0.25\t917\n16842786\t2.475\t0.266\t49\n16842787\t2.694\t0.214\t14\n16842804\t3.261\t0.399\t44\n16842805\t3.241\t0.251\t46\n16843014\t2.01\t0.115\t20813\n16843015\t1.834\t0.133\t156\n16843023\t2.464\t0.209\t339\n16843024\t2.494\t0.161\t19\n16843270\t1.935\t0.13\t225\n16843280\t2.489\t0.19\t50\n16846865\t1.992\t0.399\t44\n16846880\t2.745\t0.286\t511\n16846881\t2.728\t0.235\t235\n16846882\t2.384\t0.337\t109\n16846900\t3.03\t0.369\t277\n16846901\t2.906\t0.468\t87\n16847110\t1.969\t0.095\t20992\n16847111\t1.77\t0.177\t16\n16847119\t2.289\t0.187\t1133\n16847120\t2.181\t0.113\t8\n16847366\t1.923\t0.203\t4\n16847376\t2.231\t0.165\t28\n16908320\t3.119\t0.561\t1100\n16908321\t2.481\t0.4\t70\n16908322\t2.624\t0.585\t344\n16908323\t2.911\t0.375\t127\n16908340\t3.181\t0.424\t124\n16908341\t3.271\t0.469\t73\n16908550\t2.342\t0.169\t2233\n16908551\t2.432\t0.175\t414\n16908552\t2.284\t0.062\t14\n16908559\t3.148\t0.384\t31\n16908806\t2.273\t0.193\t233\n16912417\t2.568\t0.286\t226\n16912418\t2.384\t0.277\t52\n16912419\t2.383\t0.526\t37\n16912436\t2.868\t0.102\t3\n16912437\t3.175\t0.376\t43\n16912646\t2.247\t0.155\t563\n16912647\t2.196\t0.137\t13\n16912655\t2.8\t0.063\t13\n16912902\t1.902\t0.429\t4\n16912903\t2.164\t0.088\t7\n16916514\t2.705\t0.288\t257\n16916515\t2.826\t0.329\t129\n16916532\t3.054\t0.385\t17\n16916533\t3.596\t0.24\t25\n16916742\t2.215\t0.127\t1635\n16916743\t1.989\t0.141\t55\n16916751\t2.686\t0.173\t226\n16916752\t2.64\t-1.0\t1\n16916998\t2.197\t0.154\t196\n16920611\t3.015\t0.264\t187\n16920628\t3.279\t0.255\t239\n16920629\t3.195\t0.403\t63\n16920838\t2.164\t0.108\t13670\n16920839\t2.524\t0.134\t43\n16920847\t2.176\t0.207\t50\n16920848\t2.785\t0.214\t9\n16921094\t2.117\t0.097\t16\n16990260\t3.228\t0.382\t40\n16990261\t3.606\t0.35\t107\n16990470\t2.49\t0.154\t812\n16990471\t2.273\t0.267\t50\n16990472\t3.042\t-1.0\t1\n16990479\t3.31\t0.124\t16\n16990726\t2.535\t0.253\t17\n16994357\t3.54\t0.331\t863\n16994566\t2.373\t0.134\t5123\n16994567\t2.597\t0.154\t114\n16994575\t3.258\t0.143\t18\n16994822\t2.308\t0.108\t265\n16994823\t2.683\t0.0\t2\n17846534\t1.807\t0.004\t6\n17850630\t1.686\t0.082\t194786\n17850631\t1.634\t0.082\t26774\n17850632\t1.542\t0.034\t5\n17850639\t2.101\t0.105\t7379\n17850640\t2.077\t0.073\t1123\n17850886\t1.624\t0.087\t18128\n17850887\t1.587\t0.111\t90\n17850896\t2.041\t0.093\t13784\n17854727\t1.605\t0.077\t739\n17854735\t1.988\t0.193\t481\n17854736\t2.01\t0.157\t103\n17854982\t1.518\t0.05\t118\n17854983\t1.299\t0.035\t2\n17854991\t2.079\t0.0\t2\n17854992\t1.872\t0.081\t564\n17858824\t1.212\t0.0\t2\n17887503\t2.816\t0.232\t70\n17887504\t2.775\t-1.0\t1\n17887750\t2.038\t0.159\t42\n17887760\t2.72\t0.033\t2\n17891600\t2.75\t0.187\t12\n17891856\t2.984\t-1.0\t1\n18899462\t1.586\t0.08\t708\n18899472\t2.099\t0.117\t10\n18940432\t2.57\t0.156\t5\n33611790\t2.862\t0.219\t107\n33611809\t3.023\t-1.0\t1\n33611810\t1.697\t0.952\t3\n33611828\t3.382\t-1.0\t1\n33612038\t2.267\t0.06\t19\n33612039\t2.076\t0.084\t9\n33612040\t2.213\t0.116\t7\n33612047\t3.299\t0.356\t32\n33612048\t2.829\t-1.0\t1\n33612294\t2.498\t0.086\t4\n33624117\t2.68\t0.0\t2\n33624326\t2.051\t0.166\t7\n33624328\t1.644\t0.085\t2970\n33685536\t3.587\t0.192\t22\n33685537\t3.362\t0.618\t7\n33685539\t2.768\t-1.0\t1\n33685766\t2.418\t0.096\t23\n33685767\t2.3\t-1.0\t1\n33685768\t1.911\t0.307\t75\n33685775\t3.061\t0.334\t74\n33685776\t3.159\t0.258\t11\n33686022\t2.244\t-1.0\t1\n33689633\t2.268\t0.368\t3\n33689634\t3.2\t0.0\t2\n33689653\t3.142\t0.113\t4\n33689862\t1.874\t0.13\t10\n33689863\t1.999\t-1.0\t1\n33689864\t2.138\t0.163\t58\n33689871\t2.766\t0.099\t11\n33689872\t2.702\t0.904\t6\n33693731\t3.211\t0.049\t4\n33693748\t3.96\t0.381\t8\n33693749\t3.32\t0.198\t7\n33693958\t2.081\t0.047\t6\n33693959\t2.112\t0.21\t9\n33693960\t1.88\t0.205\t162\n33693967\t2.503\t0.176\t332\n33697844\t3.872\t0.0\t2\n33698055\t1.967\t-1.0\t1\n33698056\t1.612\t0.028\t8\n33767476\t2.657\t-1.0\t1\n33767477\t3.446\t-1.0\t1\n33767686\t2.131\t0.005\t3\n33767687\t2.1\t-1.0\t1\n33767688\t2.267\t0.518\t6\n33767695\t2.857\t0.194\t7\n33767696\t3.719\t0.957\t3\n33771573\t3.614\t0.457\t10\n33771783\t2.249\t0.088\t5\n33771784\t2.127\t0.16\t106\n33771791\t3.098\t-1.0\t1\n33771792\t3.166\t0.428\t3\n34623750\t1.78\t0.012\t7\n34627846\t1.55\t0.083\t65623\n34627847\t1.486\t0.077\t27424\n34627848\t1.404\t0.071\t156637\n34627855\t2.044\t0.135\t373\n34627856\t1.945\t0.103\t4751\n34628102\t1.526\t0.109\t716\n34628112\t1.965\t0.141\t8\n34631943\t1.436\t0.08\t1641\n34631944\t1.371\t0.063\t17326\n34631951\t1.879\t0.175\t1347\n34631952\t1.901\t0.158\t89\n34632198\t1.396\t0.123\t200\n34632199\t1.374\t0.136\t116\n34632207\t2.095\t0.0\t2\n34632208\t1.812\t0.114\t109\n34636040\t1.131\t0.253\t79\n34636047\t1.751\t0.123\t5003\n34636048\t1.76\t0.092\t2467\n34636294\t1.228\t0.157\t96\n34636295\t1.111\t0.035\t20\n34636304\t1.661\t0.085\t34493\n34664719\t2.605\t0.159\t86\n34664720\t2.299\t0.162\t912\n34664966\t1.951\t0.174\t35\n34664967\t1.804\t0.217\t28\n34664975\t2.749\t0.0\t2\n34664976\t2.399\t-1.0\t1\n34668816\t2.417\t0.111\t4\n34669062\t1.893\t0.17\t143\n34669071\t2.291\t0.138\t2\n35676678\t1.475\t0.093\t37\n52453894\t1.376\t0.081\t8518\n52453895\t1.293\t0.068\t14577\n52453896\t1.593\t0.165\t45\n52453903\t1.768\t0.204\t5\n52457991\t1.263\t0.085\t581\n52458000\t1.392\t0.156\t3\n83914759\t1.547\t0.085\t5\n83915013\t1.739\t0.204\t7\n83915014\t1.572\t0.079\t50584\n83915015\t1.539\t0.074\t8699\n83915023\t1.977\t0.137\t53\n83919110\t1.558\t0.068\t8984\n83919111\t1.575\t0.101\t961\n83919119\t2.014\t0.019\t9\n83951650\t2.389\t0.04\t4\n83951878\t1.973\t0.109\t22593\n83951879\t1.799\t0.11\t991\n83951887\t2.464\t-1.0\t1\n83951888\t2.672\t0.158\t11\n84025378\t2.575\t0.005\t2\n84025606\t2.151\t0.135\t549\n84025607\t2.077\t0.132\t184\n84959494\t1.626\t0.085\t32317\n84959495\t1.575\t0.081\t14689\n84959503\t2.111\t0.074\t20\n84959504\t2.015\t0.127\t36\n84963591\t1.557\t0.073\t2345\n84963599\t1.935\t0.256\t2\n84963600\t2.2\t0.296\t11\n100802822\t2.071\t0.051\t5\n100802823\t1.97\t0.021\t4\n101732615\t1.731\t0.197\t7\n101736710\t1.567\t0.082\t49304\n101736711\t1.502\t0.079\t23379\n101736719\t2.028\t0.119\t39\n101736720\t1.978\t0.107\t31\n101740807\t1.481\t0.072\t2974\n101740815\t1.989\t0.139\t38\n101740816\t1.737\t0.078\t11\n101773584\t2.915\t0.094\t4',Slb),h);d=b[0];c=flb.W3(d,wob,Ylb);mlb.Xu=oW(elb.IX,omb,6,c,15,1);mlb.Yu=oW(elb.HX,kob,6,c,15,1);mlb.Zu=oW(elb.HX,kob,6,c,15,1);mlb.Wu=oW(elb.IX,omb,6,c,15,1);for(e=0;e<c;e++){g=b[e+1];f=flb.u5(g,'\t');if(f.length==4){try{mlb.Xu[e]=flb.W3(f[0],wob,Ylb);mlb.Yu[e]=h4(f[1]);mlb.Zu[e]=h4(f[2]);mlb.Wu[e]=flb.W3(f[3],wob,Ylb);}catch(a){a=k1(a);if(uX(a,35)){break;}else throw l1(a);}}}mlb._u=true;}};mlb.jv=function jv(a){var b,c,d,e;mlb._u||mlb.iv();e=tnb;d=Vmb;for(c=0;c<12;c++){b=e>=mlb.Xu.length||a<mlb.Xu[e]?-1:a==mlb.Xu[e]?0:1;if(b==0)return e;e=b<0?e-d:e+d;d=d/2|0;}return -1;};b2(277,1,{},mlb.cv);mlb._u=false;elb.IY=D3(277);mlb.kv=function kv(a,b){a.q=mlb.hw(new mlb.kw(a.o,b),a);};mlb.lv=function lv(a,b){var c,d,e,f,g,h,i,j,k;c=mlb.rv(a,b[0]);d=mlb.rv(a,b[1]);e=mlb.rv(a,b[2]);f=mlb.rv(a,b[3]);i=new klb.Dh(d.a-c.a,d.b-c.b,d.c-c.c);j=new klb.Dh(e.a-d.a,e.b-d.b,e.c-d.c);k=new klb.Dh(f.a-e.a,f.b-e.b,f.c-e.c);g=new klb.Dh(i.b*j.c-i.c*j.b,-(i.a*j.c-i.c*j.a),i.a*j.b-i.b*j.a);h=new klb.Dh(j.b*k.c-j.c*k.b,-(j.a*k.c-j.c*k.a),j.a*k.b-j.b*k.a);return -$wnd.Math.atan2($wnd.Math.sqrt(j.a*j.a+j.b*j.b+j.c*j.c)*(i.a*h.a+i.b*h.b+i.c*h.c),g.a*h.a+g.b*h.b+g.c*h.c);};mlb.mv=function mv(a,b){mlb.pv(a,b);return mlb.Xv(a.q,b.q);};mlb.nv=function nv(a,b){var c;for(c=0;c<b.q;c++)klb.yh(a.k[c],b.J[c]);};mlb.ov=function ov(a,b){var c;for(c=0;c<b.k.length;c++)klb.yh(a.k[c],b.k[c]);b.j==null?a.j=null:a.j=llb.bcb(b.j,b.j.length);};mlb.pv=function pv(a,b){var c;if(!a.q||!b.q){c=new mlb.jw(a.o);!a.q&&(a.q=mlb.hw(c,a));!b.q&&(b.q=mlb.hw(c,b));}};mlb.qv=function qv(a,b){return a.j==null?-1:a.j[b];};mlb.rv=function rv(a,b){return a.k[b];};mlb.sv=function sv(a){return a.p==null?a.o.P:a.p;};mlb.tv=function tv(a,b){return a.k[b].a;};mlb.uv=function uv(a,b){return a.k[b].b;};mlb.vv=function vv(a,b){return a.k[b].c;};mlb.wv=function wv(a,b,c){if(a.j==null){a.j=oW(elb.f1,Hnb,6,a.o.r,15,1);llb.gcb(a.j);}while(c<0)c=c+360<<16>>16;while(c>=360)c=c-360<<16>>16;a.j[b]=c;};mlb.xv=function xv(a,b,c){klb.yh(a.k[b],c);};mlb.yv=function yv(a,b,c){a.k[b]=c;};mlb.zv=function zv(a,b){a.p=b;};mlb.Av=function Av(a,b,c){a.k[b].a=c;};mlb.Bv=function Bv(a,b,c){a.k[b].b=c;};mlb.Cv=function Cv(a,b,c){a.k[b].c=c;};mlb.Dv=function Dv(a,b){var c;!b&&(b=klb.Nt(a.o));for(c=0;c<b.q;c++)klb.yh(b.J[c],a.k[c]);a.p!=null&&klb.sm(b,a.p);return b;};mlb.Ev=function Ev(a){var b;this.o=a;this.k=oW(elb._X,bob,24,a.q,0,1);for(b=0;b<a.q;b++)this.k[b]=new klb.Eh(a.J[b]);this.n=NaN;};mlb.Fv=function Fv(a){mlb.Gv.call(this,a,a.o);};mlb.Gv=function Gv(a,b){var c;this.o=b;this.k=oW(elb._X,bob,24,a.k.length,0,1);for(c=0;c<this.k.length;c++)this.k[c]=new klb.Eh(a.k[c]);a.j!=null&&(this.j=llb.bcb(a.j,a.j.length));this.p=a.p==null||flb.i5(a.p)?a.p:flb.g5(a.p,cmb);this.n=NaN;};b2(58,1,{58:1,27:1},mlb.Fv);_.qb=function Hv(a){return mlb.mv(this,a);};_.n=0;elb.JY=D3(58);mlb.Jv=function Jv(a,b){var c,d,e,f,g;if(b==null)return null;g=llb.Qab(a.c,b);if(g)return g;d=ilb.Xgb('<').length;if(flb.j5(ilb.Xgb(b).substr(ilb.Xgb(b).length-d,d),'<')||(e=ilb.Xgb('-').length,flb.j5(ilb.Xgb(b).substr(ilb.Xgb(b).length-e,e),'-'))){g=llb.Qab(a.c,(f=ilb.Xgb('<').length,flb.j5(ilb.Xgb(b).substr(ilb.Xgb(b).length-f,f),'<')?flb.w5(b,0,ilb.Xgb(b).length-1)+'>':(c=ilb.Xgb('-').length,flb.j5(ilb.Xgb(b).substr(ilb.Xgb(b).length-c,c),'-')?flb.w5(b,0,ilb.Xgb(b).length-1)+'+':b)));if(g){g=new mlb.Ew(g);llb.seb(a.c,b,g);return g;}}return null;};mlb.Kv=function Kv(a,b){var c,d,e,f,g,h,i,j,k,l,m,n,o,p,q,r,s,t,u,v,w,A,B,C;a.b|=b;A=(m=flb.u5('daD@h@\\H\\D\\L\\B\\Djfdrjj`Cj`XZV^[lgl|WjBp^C~RQdp\ndaDD`HP@cIICISKUT@]LCbURss\\bCPVCp_rPmV@\ndaDH@@RfzaJjjbHGWPhxdtl|yXEGDfgf}HDoTE`|G|dC]`<\ndaDH@@RfzaJjjbHGWPhxdtl|yqGDf}HDoTE`|G|dC]`<\ndaDH@@RfzaJjjbHGWPhxdtl|yqGDf}HDoTE`|G|dC]`>\ndaT@`@\\DjfjZn[jjjkJcKhGP`phxdtl|wY@xD`uyo]{|lKUfp\ndax@@DiU[zh@zHfegfzLG``\ndax@@DiYn`@@wHmumm}l=\ndax@@DiYn`@@wHmumm}t=\ndax@@DiYn`@@wHmumn}t=\ndax@@DiYn`@@wHmuum}t\ndax@@DiYn`@@wHmuun}t=\ndax@@DiYn`@@wHnuun}t=\ndax@@DiYn`h@zsEgfyEnmlG``=\ndax@@DiYnf@@zaJyEnnmom`=\ndax@@DiYnf@@zaJyEvnmom`=\ndax@@DiYnf@@zqDffyEv`mom`=\ndax@@DiYnf@@zqDffyEv`mon`\ndax@@DiYnf@@zqDff~TEm}l=\ndax@@DiYnf@@zqDff~Uyn}t=\ndax@@DiYnf@@zqDfjyEv`mom`=\ndax@@DiYnf@@zqDfj~TEm}l=\ndax@@Djfjjh@zDFEGDfegf{HG@dF`lG``\ndax@@Djfjjh@zDFEGDfeggI@phdwY@xD`tE`|D\ndax@@Djfjjh@zDFEGDfeggM@phdtwY@xD`tE`|D\ndax@@Djfjjh@zDFEGDfeggN@pdwY@xD`tE`|D\ndax@@Djfjjh@zDFEGDfeggN@plwY@xD`tE`}x\ndax@@Djfjjh@zDFEGDjegj{HG@dF`lG``\ndax@@DjfjjjHP]BCBcbSRssd`XRV[l`\\BPZBp^|>\ndax@@DjfjjjHP]BCBcbSRsug@XR[l`\\BPZBp^B<\ndax@@DjfjjjHP]BCBcbSRsug@XR[l`\\BPZBp^B>\ndax@D@bH\\DbL\\B\\JbFbN\\Dfnjjh@zRGGNHxlwPo]x\ndax@H@\\H\\D\\J\\Djfjjh@zXFGDegf{I{@dFoLG``\ndax@H@\\H\\D\\J\\Djfjjh@zXFGDeggN@pdwYOXD`uy`|D\ndax@H@\\H\\D\\L\\Djfjjh@zXFDfegf{I{ODF`lG``\ndax@H@\\H\\J\\F\\Djfjjh@zXFEGDgj{HG@dFoM{``\ndax@H@\\J\\F\\N\\Djfjjh@zhFEGDf{HG@dFoM{o@\ndax@H@\\J\\F\\N\\Djfjjh@zhFEGDf}HG@dFoMx\ndax@H@dDdJdFdDfnjjh@zJGDgfzDGo@\ndax@H@dDdJdFdDfnjjh@zJGDgfzE{o@\ndax@H@dDdJdFdDfnjjh@zJGDggNHxlwP`|D\ndax@H@dDdJdFdDfnjjh@zJGDggNHxlwPo]x\ndax@H@dDdJdFdDfnjjh@zJGDkfzDGo@\ndax@H@dDdJdFdDfnjjh@zJGDkgNHxlwP`}x\ndax@H@dDdJdFdDfnjjh@zJGDkjzDG``\ndax@H@dDdLdJdLbdLeUT@]EBRss]F}w`\ndax@H@dDdLdJdLbdLeUUAPNbaIZyncA{p<\ndax@H@dDdLdJdLbdLeUUAPNbaIZyncA{p>\ndax@P@BFBDiuZ{h@zHEEgfzHG``\ndax@P@QFQDiuZ{h@zHEEgfzHG``\ndax@P@\\H\\Djfjjh@zDFEGDfegf{HG@dF`lG``\ndax@P@\\H\\Djfjjh@zDFEGDjegj{HG@dF`lG``\ndax@P@\\H\\Djnjjh@zxEGDfegf{Q{@dF`lG``\ndax@P@\\H\\Djnjjh@zxEGDfegf{Q{OEz`lG``\ndax@P@\\H\\Djnjjh@zxEGDfegf}YxoTE`|D\ndax@P@\\H\\Djnjjj@P]\\BcbSRss]h}`R}PVCpP<\ndax@P@\\H\\Djnjjj@P]\\BcbSRss]h}`R}PVCpP>\ndax@P@\\H\\Djnjjj@P]\\BcbSRss^lBWjBp^B<\ndax@P@\\H\\Djnjjj@P]\\BcbSRss^lBWjBp^B>\ndax@Q@|Bh@dnRjZjj`Ck`XT\\ZV^[t`\\CPVCpP\ndax@R@BH|HeIijjj@NnaQqIiYyoV^KuAXOA@\ndax@R@QH|HeIijjj@NnaQqIiYyoV^KuAXOA@\ndax@R@\\H\\HyJkjjj@NnAQqIiYyoV^KuAXOA@\ndax@X@\\H\\D\\L\\B\\J\\Djfjjh@zHFEgf{I{OEzoLG``\ndax@X@\\H\\D\\L\\B\\J\\Djfjjh@zHFEggI@phdwYOYxoUy`|D\ndax@X@\\H\\D\\L\\B\\J\\Djfjjh@zHFEggN@pdwYOYxoUy`|D\ndax@X@kDdLdJkFdNdLbdLeUT@]IBR\ndax@`@BDf]j^@@zbJyAoMmom`=\ndax@`@BDf]j^@@zbJyAoNmom`=\ndax@`@BDf]j^@@zbJyAoNmwn`=\ndax@`@\\Df]j^@@zRGFyA{Nmom`=\ndax@`@\\Djfjjh@zDFEGDfegf{HG@dF`lG``\ndax@`@\\Djfjjh@zDFEGDfegf{HG@ez`lG``\ndax@`@\\Djfjjh@zDFEGDjegf{HG@dF`lG``\ndax@`@\\Djfjjh@zDFEGHjegf{HG@dF`lG``\ndax@`@\\Djfjjh@zDFEKHjegf{HG@dF`lG``\ndax@`@\\LddUUUr@]LBcbRss\\dC`RCwP\ndax@`@dDffjjjHP]]BcbSRssgHTR[u`RCPVCpP<\ndax@b@|@eJijjj@NaAaQqIiYynrApI^hKAxH\ndax@b@|@eJijjj@NaAaQqIiYynrAsq^hKAxH\ndax@b@|@eJijjj`DGP`phyDtl|wY@xDoTE`|D<\ndax@b@|@eJijjj`DGP`phyDtl|wY@xDoTE`|D>\ndax@h@BDdLdJdFdLbdLeUT@]YBSs]@}w`\ndax@h@\\H\\D\\L\\B\\Djfjjh@zhFFegf{I{OEz`lG``\ndax@h@\\H\\D\\L\\B\\Djfjjh@zhFFegf{I{OEz`lG`\x7FddEB\ndax@h@\\H\\D\\L\\B\\Djfjjh@zhFFegf}I{OTE`|D\ndax@h@\\H\\D\\L\\J\\Djfjjh@zhFDegf{I{ODFoLG``\ndax@h@\\H\\D\\L\\J\\Djfjjh@zhFDeggI@phdwYOYx`uy`|D\ndax@h@\\H\\D\\L\\J\\Djfjjj@P]TCBRssg@XR[lgl|PZ|p^B<\ndax@h@\\H\\D\\L\\J\\Djfjjj@P]TCBRssg@XR[lgl|PZ|p^B>\ndax@h@dDdLdJdFdLbdLeUT@]YBSs^NB\ndax@h@dDdLdJdFdLbdLeUT@]YBSs^N|\ndax@h@dDdLdJdFdLbdLeUT@]YBSu^NB\ndax@h@kDdLdJdFdLbdLeUT@]YBSs^N|\ndax@p@\\H\\D\\Djfjjh@zxFGDfegf{I{@dF`lG``\ndax@p@\\H\\D\\Djfjjh@zxFGDfegj{I{@dF`lG``\ndax@p@\\J\\F\\Djfjjh@zXFEGDgf{HG@dFoM{``\ndax@p@\\J\\F\\Djfjjh@zXFEGDgj{HG@dFoM{``\ndax@p@\\J\\F\\DjfjjjHP]LCBcbSs^dC`RCWnB<\ndax@p@\\J\\F\\DjfjjjHP]LCBcbSs^dC`RCWnB>\ndax@p@dDdLdLbdLeUT@]UBSRss_JBp^B\ndax@p@dDdLdLbdLeUT@]UBSRss_JBp^|\ndax@p@dDdLdLbdLeUT@]UBSRsu_JBp^B\ndax@p@dDdLdLbdLeUT@]UBSRuu_JBp^B\ndax@p@dDdLdLbdLeUT@]UBSTus_JBp^|\ndax@p@|D\\L\\DffjjjHP]MBbSRss^l|PZBp^B<\ndax@p@|D\\L\\DffjjjHP]MBbSRss^l|PZBp^B>\ndaxB@@RfRZzjjdaX`zJGDggN@hdwP`|D<\ndaxB@@RfRZzjjdaX`zJGDggN@hdwP`|D>\ndaxBRHSFCpkpslddlcUT`HdGV@d|wRo\\D>\ndaxB`HSJCprRRrMUPAtPIKOMtXOA@\ndaxD@@QIkjjj@NfaqIiYysbNIMrHMAXOA@\ndaxD@@QIkjjjbDGSPxdtl|yHEGDfyDF`lG``<\ndaxD@@QImYk`@Nhanc[{X=\ndaxD@@iIijjj@NfaQqIYynVAHKAxH\ndaxD@@iIijjj@NfaQqIYysTJNMKMrpIAXOA@\ndaxD@@iIijjjRDbCihT\\RV^\\yBcS\\lBPVCpP\ndaxD@@iIijjjRDdCihT\\RV^\\yBbs\\lBPVCw`>\ndaxD@@iIijjjRDdCihT\\RV^\\yBcS\\lBPVCpP<\ndaxD@@yIUVnz@NdaY@\ndaxDHHP@biAiCiCIICHmU@GV@xdw```\ndaxDPHP@b`cIICHmU@GU@xeL|wH`uy`|D\ndaxD`HP@cIICHmU@GS@xdtl|wH`tE`|D\ndaxD`HP@cIICHmU@GS@xeTl|wH`tE`|D\ndaxD`HPGCIICIUU@GS@xdtl|wHoTE`|D\ndaxDpJXIAICICHiCIUU@GVPd|wc``\ndaxDpJXIAICICHiCIUU@GVPd|wco@\ndaxH@@RYvjxHaAtdNM|pK[{X+\ndaxH@@RZZjj`CkhT\\RZV^[u`RCPVCpP\ndaxH@@RZZjj`CkhT\\RZV^\\lCBcbSS^lBPZBp^B\ndaxH@@RZZjj`CkhT\\RZV^\\uBcbSS^lBPZBp^B\ndaxH@@RZZjj`CkhT\\RZV^lmBcbSRs^lBPZBp^B\ndaxH@@RZZjjhQAutJNIMKONZaQqIYoVAHMAXO^@<\ndaxH@@RZZjjhaAutJNIMKONVAaQqIioVAKuAXOA@<\ndaxH@@RZZjjhaAutJNIMKONZAaQIioVAHMAXOA@<\ndaxH@@RZZjjhaAutJNIMKONZaQqIioVAKuAXOA@>\ndaxH@@RZZjjhaAutJNIMKON\\aQIoVAHMAXOA@<\ndaxH@@RZZjjhaAutJNIMKON\\aQIoVAHMAXOA@>\ndaxH@@RZZjjhaAutJNIMKON\\aQYoVAHMAXO^@>\ndaxH@@RZZjjhaAutJNIMKON\\aqIoVAKuAXOA@<\ndaxH@@RZZjjhaAutJNIMKON\\aqIoVAKuAXOA@>\ndaxH@@RZZjjhaAutJNIMKON\\aqYoVAKuAXO^@<\ndaxHHBApapQpqpRjZjj`Cj`XZV^[tgl}PVCpP\ndaxHPJBPRPRZzjjhrAuTNIKWM|KsA{p<\ndaxHPJBPRPRZzjjhrAuTNIKWM|KsA{p>\ndaxH`HApRjzjj`Ck`T\\RZV^[u`RCPVCpP\ndaxH`HApRjzjj`Ck`T\\RZV^[ugb}PVCpP\ndaxH`HApRjzjj`Ck`T\\bjV^[u`RCPVCpP\ndaxH`HApRjzjjhAAupJNIMKOMzpI^hKAxH<\ndaxH`HApRjzjjhAAupJNIMKOMzpI^hKAxH>\ndaxH`JBHRZZjjhaAttJNIKONRaQiYnVAHKAxH>\ndaxHpJBHRHrHrJPrUUPAtTIKOMtXOA@\ndaxHpJBPRPrPrJPrUUPAtTIKOMtXOA@\ndaxHpJBPRPrPrJPrUUPAtTIKOMt[w^@\ndaxHpJBPRPrPrJPrUUTE@zJDegfzLGo@<\ndaxHpJBPRPrPrJPrUUTE@zJDegfzLGo@>\ndaxHpJBPRPrPrJPrUUTE@zJDegjzLG``<\ndaxHpJBPRPrPrJPrUUTE@zJDegjzLG``>\ndaxHpJBPRPrPrJPrUUTE@zJDekfzLGo@<\ndaxHpJBPRPrPrJPrUUTE@zJDekfzLGo@>\ndaxL@@RdfnjjjHP]UCbRsse`T\\RZV[xPVCpP<\ndaxLHHS@BJdFdNdLddlbuT@]HBR\ndaxLJHS@BJdFdNdL{IIKHmU@GR@d`\ndaxL`HS@\\LddleUT@]TBSRss_JBp^B\ndaxLhJ[`\\H\\D\\L\\B\\Djfjjh@zPFF~I{OEx\nday@@@QIkjjj@NfaqIiYysPLJIMMrHMAXOA@\ndayDaLJD@HrR{IICHmU@GU@xtl|wr`lG``\ndaz@@@RYvHx@C\\`wfvwvp=\ndaz@@@RYvHx@C\\`wfvwwP\ndaz@@@RYvHx@C\\`wgVwvp=\ndaz@@@RYvix@CjHkdF|zv~v=\ndaz@@@RYvix@CjHkdG\\zv~v=\ndaz@B@BTf]bN@@wHMymm}t\ndaz@HBApapQpqpRjZjj`Cj`XZV^[tgl}PVCpP\ndaz@bHCpBTjnjjh@zxEGDfegf}YxoTE`|D\ndaz@pJBPRPrPrJPrUUPAtTIKOMtXOA@\ndaz@pJBPRPrPrJPrUUTE@zJDegjzLG``>\ndazDHLa@BJdFdNdLddlbuT@]HBR\ndazDJLa@BJdFdNdL{IIKHmU@GR@d`\ndazD`La@BLddlbuT@]TBSRss_JBp^B\ndazH`D`OCIICHuUPBCi`\\RZV^[dWjBp^B<\ndazH`D`OCIICHuUPBCi`\\RZV^[dWjBp^B>\ndazHpFhIAICICHiCIUUPRCkHR^[qpP<\ndazHpFhIAICICHiCIUUPRCkHR^[qpP>\ndazLQFaLCpkpsfgIIKHuUHBIAu`IOMtkwA@<\ndazLQFaLCpkpsfgIIKHuUHBIAu`IOMtkwA@>\ndctD@@gHiDTeIU]uPAttNIHjnmrpIATNa@\ndctD@@gHiDTeIU]uPAttNQPjvurpIATNa@\nded@@DiUfoh@@Mr[_[G\\{@=\nded@@DiUfoh@@Mr[_]G\\}@\nded@P@\\F\\LddUTeUuPAtHJNIKOHlmvPNAHOADLa@\ndedD@@{HiDUIU]T@]MCdTrMM\\lBPQCHP\ndiD@@DiYkhJ`ChlV^Q[tVzvp^BHP=\ndiD@@DiYkhJ`ChlV^Q[tVzzp^BHP\ndiD@@DiYkhJ`ChlV^aktVzvp^BHP=\ndiD@@DiYkhJ`ChlVnaktVzvp^BHP=\ndiD@@DiYkhJ`ChlVnaktVzzp^BHP\ndiD@H@\\H\\D\\L\\LddjTjn@@z`F}QyOY{nct=\ndiD@`@\\LddUTjn@@zHEGDf}HG@dGmcl=\ndiD@`@\\LddUTjn@@zHEGDf}HG@dGmct\ndiD@`@\\LddUTjn@@zHEGDf}HG@dGnct=\ndiD@`@\\LddUTjnX@zhEGDej}HG@dGnct\ndiD@`@\\LddUTjnX@zxEGDegdV}HG@dGnbD\ndiD@`@\\LddUTjnX@zxEGDegdZ}HG@dGnbD\ndiDH@@rJQERjx@CkH\\R[e`RCvqv=\ndiDH@@rJQERjx@CkH\\R[e`RCvqz\ndiDH@@rJQERjx@CkH\\bke`RCvqv=\ndiDH@@rJQERjx@CkH\\bke`RCvqz\ndiDH@@rJQERjxHaAudNIMrpI^{X{@+\ndiDH@@rJQERjxHaAudNIMrpI^{X{@-\ndiDH@@rJQERjxHaAudNIMrpI^{X}@<\ndiDH@@rJQERjy`Cih\\RV^Q[y`RBO`\ndiDH@@rJQERjy`Cih\\bf^Q[e`RCwQB\ndiF@@@rJQERjx@CkH\\R[e`RCvqv=\ndiF@`NAprRQURjx@Ch`T\\R[t`\\BP^vNp\ndiF@a@b`BX{HiDUJk`@NlarJnVAHO[Gh\ndiFH@NCHiDUJk`@NlaqInVAHO[GX\ngC`@Die@zHXir\ngC`@Die@zHXis]Knz\ngC`@Die@zHXis^Kh\ngC`@Die@zHXisgAbd\ngC`@Die@zHXisgAbfzW]t\ngC`@Die@zHXisgAbf|WP\ngC`@Dij@zHXis]HNB\ngC`@Dij@zHXis]HN|\ngC`@Dij@zHXis]Kv|\ngC`@Dij@zHXisgAbfzP\\D\ngC`@Dij@zHXisgAbfzP]x\ngC`@Dij@zHXisgAbfzWmx\ngC`@Dij@zHXiu]HNB\ngC`@Dij@zHXiugAbfzP\\D\ngC`@Dij@zHXjs]HN|\ngC`@Dij@zHXjsgAbfzP]x\ngC`@Dij@zHXju]HNB\ngC`@Dij@zHXjugAbfzP\\D\ngC`@DkZ@zHXis]SnB\ngC`@DkZ@zHXis]Sn|\ngC`@DkZ@zHXis^XH\ngC`@DkZ@zHXis^[p\ngC`@DkZ@zHXisgAbfzg\\D\ngC`@DkZ@zHXisgAbfzg]x\ngC`@DkZ@zHXisgAbf|pP\ngC`@DkZ@zHXisgAbf|w`\ngC`@DkZ@zHXiu]SnB\ngC`@DkZ@zHXiu^XH\ngC`@DkZ@zHXiugAbfzg\\D\ngC`@DkZ@zHXiugAbf|pP\ngC`B@DPHPQ`aJZ`NdFH\ngC`B@DPHPQ`aJZ`NdFN\\FJP\ngC`B@DPHWaoAJZ`NdFMto[p\ngC`D@DPHRfhChabgMt`xH\ngC`D@DPHRfhChabgN\\FJ[iApP\ngC`D@DPHRfhChabgN\\FJ[i^w`\ngC`D@DPHRfhChabkUt`xH\ngC`D@DPHRnhCkASf|pP\ngC`D@DPHRnhCkASj|pP\ngC`D@DXHRVhCiQR\ngC`D@DXHRVhCiQSgAbd\ngC`D@DXHRfhCkAbf|PP\ngC`D@DXHRfhCkAbgNCEMx``\ngC`D@DXHRfhCkAbj|PP\ngC`D@DYpRVhCiQS]Cv|\ngC`D@VPlRfhChabkUt`xH\ngC`D@VYpRfhCkAbfzP]x\ngC`D@bQDRfhChabgMt`xH\ngC`D@bQDRfhChabkUt`xH\ngC`D@bSpRnhCkASfzglD\ngC`D@xYpRVhCiQS]Cv|\ngC`D@xYpRVhCiQSgAbfzGmx\ngC`DABPHRVhCkQSj|pP\ngC`DADZHRVhCiQR\ngC`DAb[DRVhCiQR\ngC`DAx[pRVhCiQR\ngC`DAx[pRVhCiQS]Cv|\ngC`Dax[pDzTej@zTTwP}o@\ngC`H@DIKRAuhir\ngC`H@DIKTAuhis]CvB\ngC`H@DIKTAuhis^XH\ngC`H@DIKTAuhis^[p\ngC`H@DIKTAuhisgAbfzGlD\ngC`H@DIKTAuhisgAbf|pP\ngC`H@DIKTAuhisgAbf|w`\ngC`H@DIKTAuhiu]CvB\ngC`H@DIKTAuhiu^XH\ngC`H@DISRAtPqSfzP]t\ngC`H@DISRAtPqSf|PP\ngC`H@DISRAtPqSgNCEMt`{h\ngC`H@DISRAtPqSgNCEMx``\ngC`H@DISRAtPqUfzP]t\ngC`H@DISRAtPqUgNCEMt`{h\ngC`H@DISTAtPqSfzP\\D\ngC`H@DISTAtPqSfzP]x\ngC`H@DISTAtPqSfzWlD\ngC`H@DISTAtPqSgNCEMt`xH\ngC`H@DISTAtPqSgNCEMt`{p\ngC`H@DISTAtPqSgNCEMtoXH\ngC`H@DISTAtPqSjzP\\D\ngC`H@DISTAtPqSjzWlD\ngC`H@DISTAtPqSkNCEMt`xH\ngC`H@DISTAtPqUfzP\\D\ngC`H@DISTAtPqUfzP]x\ngC`H@DISTAtPqUgNCEMt`xH\ngC`H@DISTAtPqUjzP\\D\ngC`H@VIKTAuhis]CvB\ngC`H@bIKTAuhis^XH\ngC`H@bISRAtPqSf|PP\ngC`H@bISTAtPqSfzP\\D\ngC`H@bISTAtPqSgNCEMt`xH\ngC`H@bISTAtPqSjzP\\D\ngC`H@bISTAtPqUfzP\\D\ngC`H@bISTAtPqUjzP\\D\ngC`H@xIKRAuhis^Cp\ngC`H@xIKTAuhis]CvB\ngC`H@xIKTAuhis]Cv|\ngC`H@xIKTAuhis^XH\ngC`H@xIKTAuhisgAbfzGlD\ngC`H@xIKTAuhiu]CvB\ngC`HADIKTAuhis^XH\ngC`HAVIKTAuhis^XH\ngC`HAbIKTAuhis^XH\ngC`HAxIKTAuhis]CvB\ngC`HAxIKTAuhis^XH\ngC`HAxIKTAuhisgAbfzGlD\ngC`HAxIKTAuhiu]CvB\ngC`HMPISTAu`qS]HN|\ngC`HMPISTAu`qS^HH\ngC`IAVBTii@zHXjs]HNz\ngC`IAxBTej@ztTyna{A@\ngC`IAxBTej@ztTys`qS]CvB\ngC`IMP[dij@zpXindG^@\ngC`L@DPHPPeMPGVCGMy``\ngC`L@DPHPPeMPGVCGMyo@\ngC`L@DPHPPeMPGVCGUy``\ngC`L@DPHPPeMPGVCGV\\FJ[sA@\ngC`L@VPHPPeMPGVCGMy``\ngC`L@bPHPPeMPGVCGMy``\ngC`PAEdij@zPXypXi@\ngC``@deZ@ztTyoLD\ngC``ADij@zHXis]HN|\ngC``ADij@zHXis]Kv|\ngC``ADij@zHXisgAbfzP\\D\ngC``ADij@zHXisgAbfzP]x\ngC``ADij@zHXisgAbfzWmx\ngC``Adej@zTT`\ngC``Adej@zTTypXi@\ngC``Adij@zpXioDD\ngC``Adij@zpXioEx\ngC``Adij@zpXis`qS^HH\ngC``Adij@zpXjoDD\ngC``Adij@zpXjs`qS^HH\ngC`hH`xIKTAuhis]Cv|\ngCa@@dkHGVbgH\ngCa@@dkHGVbgMyn`\ngCa@@dkPGVbgMy``\ngCa@@dmHGVbgH\ngCa@@dmHGVbgMyn`\ngCa@@dmHGVbgN\\FJP\ngCa@@dmHGVbgN\\FJ[s]@\ngCa@@dmPGVbgMy``\ngCa@@dmPGVbgMyo@\ngCa@@dmPGVbgN\\FJ[sA@\ngCa@@dmPGVbgN\\FJ[s^@\ngCa@@dmPGVbgUy``\ngCa@@dmPGVbgV\\FJ[sA@\ngCa@@eMHGQCEN[iAwP\ngCa@@eMHGQCEN[qA@\ngCa@@eMHGQCEN[q^@\ngCa@@eMHGQCEN\\xLTwRCn`\ngCa@@eMHGQCEN\\xLTwbB\ngCa@@eMHGQCEV[iAwP\ngCa@@eMHGQCEV\\xLTwRCn`\ngCa@@eMHGQCEV\\xLTwbB\ngCa@@eMPGQCEN[iApP\ngCa@@eMPGQCEN[iAw`\ngCa@@eMPGQCEN[i^pP\ngCa@@eMPGQCEN[i^w`\ngCa@@eMPGQCEN\\xLTwRC``\ngCa@@eMPGQCEN\\xLTwRCo@\ngCa@@eMPGQCEN\\xLTwR}``\ngCa@@eMPGQCEN\\xLTwR}o@\ngCa@@eMPGQCENkiApP\ngCa@@eMPGQCENki^pP\ngCa@@eMPGQCENlxLTwRC``\ngCa@@eMPGQCENlxLTwR}``\ngCa@@eMPGQCEV[iApP\ngCa@@eMPGQCEV[iAw`\ngCa@@eMPGQCEV\\xLTwRC``\ngCa@@eMPGQCEV\\xLTwRCo@\ngCa@@eMPGQCEVkiApP\ngCa@@eMPGQCEVlxLTwRC``\ngCaH@jAJZ`NlFN[i^pP\ngCaH@jAJZ`NlFN\\xLTwR}``\ngCaH@jAJZ`NlFN\\xLTwfB\ngCaH@jAJZ`NlFN\\xLTwf|\ngCaHDGAIZ`NmEN[h^pP\ngCaHH@aIZ`NmEN[h^pP\ngCaHH@aIZ`NmEN[h^w`\ngCaHH@aIZ`NmEN\\xLTwP}``\ngCaHHGAIZ`NmEN[h^pP\ngCaHHGAIZ`NmEN[h^w`\ngCaHHGAIZ`NmEN\\xLTwP}``\ngCaHHGAIZ`NmENkh^pP\ngCaHHOAIZ`NmEN[h^pP\ngCaHHOAIZ`NmEN[h^w`\ngCaHHOAIZ`NmEN[sA@\ngCaHHOAIZ`NmEN\\xLTwP}``\ngCaHHOAIZ`NmEN\\xLTwP}o@\ngCaHHOAIZ`NmEN\\xLTwf|\ngCaHHOAIZ`NmENkh^pP\ngCaHHOAIZ`NmENlxLTwP}``\ngCaHL@aIZ`NeEH\ngCaHL@aIZ`NeEN\\FJP\ngCaHLDQIZ`NeEH\ngCaHLGAIZ`NeEMxO@\ngCaHLHaIZ`NeEH\ngCaHLLQIZ`NeEH\ngCaHLOAIZ`NeEMxO@\ngCaI@jA\\eMPGVCGMtoXH\ngCaIHO@RdmPGVbgMtOXH\ngCaIHO@RdmPGVbgN\\FJ[h^pP\ngCaIL@`RdmPGRbd\ngCd@@deZ@ztTyoLG}h`\ngCd@@deZ@ztTys`qS^XO{Q@\ngCd@@deZ@ztTzoLG}h`\ngCd@@deZ@ztTzs`qS^XO{Q@\ngCd@ADie@zHXis]Kn{~rD@\ngCd@ADie@zHXis^Ko{HP\ngCd@ADie@zHXisgAbfzW]w}dH\ngCd@ADie@zHXis~rD@\ngCd@ADif@zHXis]KnC~rD@\ngCd@ADif@zHXis^XO{HP\ngCd@ADif@zHXisgAbfzW\\G}dH\ngCd@ADif@zHXisgAbf|p_vP`\ngCd@ADif@zHXiu]KnC~rD@\ngCd@ADif@zHXiu^XO{HP\ngCd@ADij@zHXis]HNC~rD@\ngCd@ADij@zHXis]HNC~t@\ngCd@ADij@zHXiu]HNC~rD@\ngCd@ADij@zHXju]HNC~rD@\ngCd@ADkj@zpTzoLD\ngCd@AdeY@zTT`\ngCd@Adej@zTT`\ngCd@Adej@zTTypXi@\ngCd@Adij@zpXioDG}h@\ngCd@Adij@zpXis`qS^HO{P@\ngCd@Adij@zpXjoDG}h@\ngCd@Adij@zpXjs`qS^HO{P@\ngCd@aAteJz`NlEN[sA\x7FZ@\ngCd@aAteJz`NlEN[s^\x7FZ@\ngCd@aaNeIZ`NeEH\ngCd@aaNeIZ`NeEN\\FJP\ngCd@aaNeIZ`NeEN\\FJ[h^w`\ngCd@aaNeIZ`NeEN\\FJ[p^@\ngCdAAIrTs@]DLTznewA\x7FYB@\ngCdAAaJRu@]JJP\ngCdAAaJRu@]JJ[p^@\ngCdAAaJRu@]JJ\\xLT`\ngCdAAaJRu@]JJ\\xLTw`|\ngCdDE`DPHRfhCiAb\ngCdHH`DIKRAuhis]Cv{~rb@\ngCdHH`DIKRAuhis^Cw{JH\ngCdHI`DIKTAthis`qS^Cw{R@\ngCe@E`dkPGRbg}h`\ngCh@@dkHGVbgH\ngCh@@dkPGVbgMy``\ngCh@@dmHGVbgH\ngCh@@dmHGVbgMyn`\ngCh@@dmHGVbgN\\FJP\ngCh@@dmHGVbgN\\FJ[s]@\ngCh@@dmPGVbgMtOXH\ngCh@@dmPGVbgMy``\ngCh@@dmPGVbgMyo@\ngCh@@dmPGVbgN\\FJ[h^pP\ngCh@@dmPGVbgN\\FJ[sA@\ngCh@@dmPGVbgN\\FJ[s^@\ngCh@@dmPGVbgUy``\ngCh@@dmPGVbgV\\FJ[sA@\ngCh@@eLhGQCEN\\xLTwR{n\x7Fm@\ngCh@@eLhGQCEN\\xLTwb{~t@\ngCh@@eMHGQCEN[iAw_v`@\ngCh@@eMHGQCEN[i^w_v`@\ngCh@@eMHGQCEN[qA\x7FZ@\ngCh@@eMHGQCEN\\xLTwRCn\x7Fm@\ngCh@@eMHGQCEN\\xLTwbC~t@\ngCh@@eMHGQCEV[iAw_v`@\ngCh@@eMHGQCEV[qA\x7FZ@\ngCh@@eMPGQCEN[iApP\ngCh@@eMPGQCEN[iAp_v`@\ngCh@@eMPGQCEN[iAw`\ngCh@@eMPGQCEN[iAwov`@\ngCh@@eMPGQCEN[i^pP\ngCh@@eMPGQCEN\\xLTwRC``\ngCh@@eMPGQCEN\\xLTwRC`\x7Fm@\ngCh@@eMPGQCEN\\xLTwRCo@\ngCh@@eMPGQCEN\\xLTwRCo_m@\ngCh@@eMPGQCEN\\xLTwR}``\ngCh@@eMPGQCENkiApP\ngCh@@eMPGQCENkiAp_v`@\ngCh@@eMPGQCENki^pP\ngCh@@eMPGQCENlxLTwRC``\ngCh@@eMPGQCEV[iApP\ngCh@@eMPGQCEV[iAp_v`@\ngCh@@eMPGQCEV[iAwov`@\ngCh@@eMPGQCEV\\xLTwRC``\ngCh@@eMPGQCEV\\xLTwRC`\x7Fm@\ngCh@@eMPGQCEVkiApP\ngCh@@eMPGQCEVkiAp_v`@\ngCh@@e[PGQCEN[j]p_v`@\ngCh@@e[PGQCEN[sA\x7FZ@\ngCh@@e[PGQCEN[s^\x7FZ@\ngCh@@e[PGQCEN\\xLTwT{`\x7Fm@\ngCh@@e[PGQCENkj]p_v`@\ngCh@@e[PGQCENksA\x7FZ@\ngCh@@e[PGQCENlxLTwT{`\x7Fm@\ngChA@IRUdCkQSf|GP\ngChA@IRVdCkQSd\ngChA@IRVhCkQSfzGlD\ngChA@IRVhCkQSf|pP\ngChA@IRVhCkQSf|w`\ngChA@IRVhCkQSgNCEMtOXH\ngChA@IRVhCkQSgNCEMy``\ngChA@IRVhCkQSgNCEMyo@\ngChA@IRVhCkQSj|pP\ngChA@IRVhCkQSkNCEMy``\ngChA@IRfhChabgMt`xH\ngChA@IRfhChabgMt`{p\ngChA@IRfhChabgMtoXH\ngChA@IRfhChabgN\\FJ[iApP\ngChA@IRfhChabgUt`xH\ngChA@IRfhChabgUtoXH\ngChA@IRfhChabgV\\FJ[iApP\ngChA@IRfhChabkUt`xH\ngChA@IRmhChabgMuNxO{P@\ngChA@IRmhChabgMy`\x7Fm@\ngChA@IRmhChabgN\\FJ[j]p_v`@\ngChA@IRmhChabgUuNxO{P@\ngChDD@bABUt`]XJ\\wf{~tP\ngChDDDRHbUt`]XJ\\wf{~tP\ngChHD@aIZ`NmEN[h^p_vb@\ngChHDOAIZ`NmENkh^p_vb@\ngChHH@aIZPNmEN[h^w_vd@\ngChHH@aIZ`NmEN[h^p_vd@\ngChHH@aIZ`NmEN[h^wovd@\ngChHH@aIZ`NmEN\\xLTwP}`\x7FmH\ngChHH@aIZ`NmENkh^p_vd@\ngChHH@aJZPNbFJlypXindG]\x7FZP\ngChHHOAIZ`NmEN[h^p_vd@\ngChHHOAIZ`NmEN\\xLTwP}`\x7FmH\ngChHHOAIZ`NmENkh^p_vd@\ngChHL@aIZ`NeEH\ngChHL@aIZ`NeEMxO@\ngChHL@aIZ`NeEN\\FJP\ngChHL@aJZ`NlFJ[qA@\ngChHL@aJZ`NlFJ\\xLTwbB\ngChHLHaIZ`NeEH\ngChHLLQIZ`NeEH\ngChHLOAIZ`NeEH\ngChIDOARdmPGVbgMtOXH\ngChIHOBRdkPGVbgMy`\x7FmH\ngChIL@cRdmPGRbd\ngChIL@cRdmPGRbf|G`\ngChILLSRdmPGRbd\ngCh`LHe]PGRBd\ngCi@DDeZ@ztTyoLG}h`\ngCi@DDeZ@ztTyoM{}h`\ngCi@DDeZ@ztTzoLG}h`\ngCi@HDii@zHXis]HN{~t`\ngCi@HDii@zHXis^HO{R@\ngCi@HDii@zHXisgAbfzP]w}i@\ngCi@HDii@zHXjs]HN{~t`\ngCi@HDij@zHXis]HNB\ngCi@HDij@zHXis]HNC~t`\ngCi@HDkj@zpTyoLD\ngCi@HDkj@zpTyoMx\ngCi@HDkj@zpTys`qS^XH\ngCi@HDkj@zpTzoLD\ngCi@LDej@zTT`\ngCi@LDej@zTTypXi@\ngCi@LDej@zTTypXioMx\ngCi@LDij@zpXioDD\ngCi@LDij@zpXis`qS^HH\ngCi@LDij@zpXjs`qS^HH\ngCiALMJRu@]JJP\ngCiALMJRu@]JJ[s^@\ngCiALMJRu@]JJ\\xLT`\ngCiALMJRu@]JJ\\xLTwf|\ngCl@ADe[PGVBgUy`\x7Fl``\ngClAADnRmhCkASj|p_vPP\ngFp@DjYZXHCkASfyW]x{n\x7FrTrp=\ngFp@DjYZjjQCiabgIZyHXis\\kv|_WwyJYX<\ngFp@DjxujhCiabgIZwL{`dMC~RvN@\ngFp@LdaahUPGVrVmsMkYCP\x7Fdmc`=\ngFp@LdaahUPGVrVmsMkY}__dmc`=\ngFp@LdaahUPGVrVmsMkiCP\x7Fdmc`\ngFp@LdaahUTrGVrVmsMkYC__dmc`+\ngFp@LdaahUTrGVrVmsMkYC__dmc`-\ngFp@LdaahUTrGVrVmsMkiC__dmc`<\ngFp@LdaahUTrGVrVmsMkiC__dmc`>\ngFp@LddLUUPGSCENRusdyK\\pJBPtOyHZX\ngFp@LddLUUPGSCENRusdyK\\sr|PtOyHZX\ngFp@LddLUUTBGSCENRurPqRVy`Uxah_rPtp>\ngFp@LddLUUTBGSCENRus`iK\\sr|PtOyHZX>\ngFp@LddLUUTBGSCENRusdyK\\pJ|PtOyHZX<\ngFp@LddLUUTBGSCENRusdyK\\pJ|PtOyHZX>\ngFp@LddLUUTBGSCENRushiK\\pJ|PtOyHZX<\ngFp`@TfXujjJChqSfnSENRwVCP\x7Fdmc`<\ngFp`@TizJjiRYChqSfn\\dm[iAh_rPup<\ngFp`@TizJjiRZChqSfmtoTOyHZx<\ngFp`@TizJjiRZChqSfmtoTOyHZx>\ngFp`@TizJjiRZChqSfn\\dm[iAh_rPup<\ngFp`@TizJjiRZChqSfn\\dm[iAh_rPup>\ngFp`@dfxujhChqrVmtPtOyKXx\ngFp`@dfxujhChqrVmt_WwyKXx\ngFp`@dfxujjYChqrVmtPwwyKXx<\ngFp`@dfxujjYChqrVmtPwwyKXx>\ngFq@@dmXuU@]VJ\\ek_XICP\x7Fdem`\ngFq@@dsFmU@]VJ\\ek_XICP\x7Fdmc`\ngFq@@dsFmU@]VJ]Ek_XICP\x7Fdmc`\ngFq@@dsFmUQH]VJ]Ek_XIC__dmc`<\ngFq@@dsFmUQH]VJ]Ek_XIC__dmc`>\ngFq@@dsFuURH]VJ\\ek_XICP\x7Fdmc`<\ngFq@@dsFuURH]VJ\\ek_XICP\x7Fdmc`>\ngFq@@eMXuURH]LLTyKVyP\\D~h_rRvp<\ngFq@@eMqUURH]VJ\\ek_HICP\x7Fdak`<\ngFq@@eMqUURH]VJ\\ek_HICP\x7Fdak`>\ngFq@@eMqUURH]VJ\\ek_KqCP\x7Fdak`<\ngFq@@eMqUURH]VJ\\ekdaRVm|oDMC~RFn@>\ngFq@@eMqUURH]VJ\\ekgIKV~WbFa\x7FICW@<\ngFq@@eMqUURH]VJ\\ekgIKV~WbFa\x7FICW@>\ngFq@@eMqUURH]VJ\\ekgIrV~PRFa\x7FICW@<\ngFq@@eMqUURH]VJ\\ekgIrV~PRFa\x7FICW@>\ngFq@@eMqUURH]VJ\\ekgIrV~PSza\x7FICW@<\ngFq@@eOQUARP]JJ[yAN{o|dM\\+\ngFq@@eOQUARP]JJ[yAN{o|dM\\-\ngFq@@eOQUARP]JJ[y^N{o|dM\\+\ngFq@@eOQUARP]JJ[y^N{o|dM\\-\ngFq`@bdwFmUSP]ZNZweC~RvN@>\ngFq`ALe\\qUU@]ENRunbFa\x7FISW@\ngFx@@eMXuU@]LLTyKVyP\\Dah_rRvuh@\ngFx@@eOQMA@]JJ[y^N{o|dM\\=\ngFx@@eOQUARP]JJ[yAN{o|dM\\+\ngFx@@eOQUARP]JJ[yAN{o|dM\\-\ngFx@@eOQUARP]JJ[y^N{o|dM\\+\ngFx@@eOQUARP]JJ[y^N{o|dM\\-\ngFx@@eSFuU@]LLTyKVyP\\Dah_rVquh@=\ngFx@@eSFuU@]LLTzKVyP\\Dah_rVquh@\ngFx@@eSFuU@]LLUZKVyP\\Dah_rVqp\ngFxAAIRicZ`@NbFR\\wJC`gm{~RvN@=\ngFxAAYRicZ`@NbFR\\wJC`gm{~RvNm@=\ngFx`DBdwFmUSP]ZNZweC~RvN@<\ngFy@DDfxujhChqrVmtPtOyKXx\ngFy@DDfxujjYChqrVmtPwwyKXx<\ngFy@DDfxujjYChqrVmtPwwyKXx>\ngGP@DiWj`NbgIZwQCP`\ngGP@DjZ@@Mrm{Yw]`=\ngGP@DjZ@@Mrm{Yw^`=\ngGP@DjZ@@Mrm{Y{^`=\ngGP@DjZ@@Mrm{iw^`\ngGP@DjZ@@Mrm{i{^`=\ngGP@DjZ@@Mrn{i{^`=\ngGP@DjZ@@NZFJ\\enUw]O[t=\ngGP@DjZ@@N\\EI[e[wSn}@\ngGP@DjZ@@N\\EN[e[wSv}@\ngGP@DjZ@@N\\FN[e]wSv}@\ngGP@DjZX@NhJyV}l{np=\ngGP@DjZX@NhJyV}l{oP\ngGP@DjZX@NhJyV}l}oP=\ngGP@DjZX@NhJyV}t{np=\ngGP@DjZX@NhJyV}t{oP\ngGP@DjZX@NhJyW]l{np=\ngGP@DjZX@NhJyW]l{oP\ngGP@DjZX@NhJyW]l}oP=\ngGP@DjZX@NhJyW]t{np=\ngGP@DjZX@NhJyW]t{oP\ngGP@DjZX@NhJyW]t}oP=\ngGP@DjZX@NhKNBdmrn{iw^`\ngGP@DjZX@NhKNCDmrn{Yw^`\ngGP@DjZX@NhKNCDmrn{iw^`\ngGP@DjZX@NlEN[e]pSn{@=\ngGP@DjZX@NlEN[e]pSn{\x7FIPh`=\ngGP@DjZX@NlEN[e]pSn}@\ngGP@DjZX@NlEN[e]pSv}@=\ngGP@DjZX@NlEN[e]wcn{@=\ngGP@DjZX@NlEN[e]wcn{\x7FIPh`=\ngGP@DjZX@NlEN[e]wcn}@\ngGP@DjZX@NlEN[e]wcv}@=\ngGP@DjZX@NlEN[{AN{l=\ngGP@DjZX@NlEN[{AN{t\ngGP@DjZX@NlEN[{AO[t=\ngGP@DjZX@NlEN[{^N{l=\ngGP@DjZX@NlEN[{^N{t\ngGP@DjZX@NlEN\\xJRwJ{`g]z\ngGP@DjZX@NlEN\\xJRwJ{oG]z\ngGP@DjZX@NlEN\\xJRwvB]wh\ngGP@DjZX@NlEN\\xJRwv|]wh\ngGP@DjZX@NlEN\\xJ\\wJ{`gmz\ngGP@DjZX@NlEN\\xJ\\wvB^wh\ngGP@DjZX@NlEN\\xLRwJ{`g]z\ngGP@DjZX@NlEN\\xLRwvB]wh\ngGP@DjZX@NlEN\\xLRwv|]wh\ngGP@DjZX@NlEN\\xL\\wJ{`gmz\ngGP@DjZX@NlEN\\xL\\wvB^wh\ngGP@DjZX@NlENke]pSn{@=\ngGP@DjZX@NlENke]pSn}@\ngGP@DjZX@NlENke]pSv}@=\ngGP@DjZX@NlENk{AN{l=\ngGP@DjZX@NlENk{AN{t\ngGP@DjZX@NlENk{AO[t=\ngGP@DjZX@NlENlxJRwJ{`g]z\ngGP@DjZX@NlENlxJRwvB]wh\ngGP@DjZX@NlENlxLRwJ{`g]z\ngGP@DjZX@NlENlxLRwvB]wh\ngGP@DjZX@NlENlxL\\wJ{`gmz\ngGP@DjZX@NlENlxL\\wvB^wh\ngGP@DjZY`NbFR\\wJ{`g]z\ngGP@DjZY`NbFR\\wJ{`gmv\ngGP@DjZY`NbFR\\wJ{`gmz\ngGP@DjZY`NbFR\\wJ{oGmv\ngGP@DjZY`NbFR\\wJ{oGmz\ngGP@DjZY`NbFR\\wvB]wX\ngGP@DjZY`NbFR\\wvB^wh\ngGP@DjZY`NbFR]WJ{`gmv\ngGP@DjZY`NbFR]WJ{`gmz\ngGP@DjZY`NdJUrm{Yw]`\ngGP@DjZY`NdJUrm{iw^`\ngGP@DjZY`NdJUrn{Y{]`\ngGP@DjZY`NdJUrn{i{^`\ngGP@DjZY`NfFJ\\ek\\knB^tH\ngGP@DjZY`NfFJ\\ek]XMB\ngGP@DjZY`NfFJ\\ek_KnCP`\ngGP@DjZY`NfFJ\\ek_XI{_@\ngGP@DjZY`NfFJ\\ekfabgI[e]pSva@\ngGP@DjZY`NfFJ\\ekfabgI[kAhP\ngGP@DjZY`NfFJ\\ekfabgI[y]pZD\ngGP@DjZY`NfFJ\\ekgARVyW\\D}hP\ngGP@DjZY`NfFJ\\ekgARV~W\\Fa@\ngGP@DjZY`NfFJ\\ekgASf~W\\Fa@\ngGP@DjZY`NfFJ\\ekgAbVzpZD\ngGP@DjZY`NfFJ\\ekgAbV~W\\Fa@\ngGP@DjZY`NfFJ\\ekgAcfyW\\D}hP\ngGP@DjZY`NfFJ\\ekgAcf~W\\Fa@\ngGP@DjZY`NfFJ\\em\\knB^tH\ngGP@DjZY`NfFJ\\em]XMB\ngGP@DjZY`NfFJ\\em_KnCP`\ngGP@DjZY`NfFJ\\em_XI{P`\ngGP@DjZY`NfFJ\\emgAcfyW\\D}hP\ngGP@DjZY`NfFJ\\emgAcf~W\\Fa@\ngGP@DjZY`NfFJ]Em\\knB^tH\ngGP@DjZY`NfFJ]Em]XMB\ngGP@DjZY`NfFJ]Em_KnCP`\ngGP@DjZY`NfFJ]EmgAcfyW\\D}hP\ngGP@DjZj`NfFJ\\ek\\hNBPtH\ngGP@DjZj`NfFJ\\ekfabgI[eApRFa@\ngGP@DjZj`NfFJ\\fm\\hNBPtH\ngGP@DjZj`NfFJmFm\\hNBPtH\ngGP@DjZjdEDNfFJ\\ekdacdmr`xIC_@<\ngGP@DjZjdEDNfFJ\\ekdacdmr`xIC_@>\ngGP@DjZjdEDNfFJ\\ekfabgI[eAwbF~@<\ngGP@DjZjdEDNfFJ\\ekfabgI[eAwbF~@>\ngGP@DjZjdEDNfFJ\\ekgARVyP]xao`<\ngGP@DjZjdEDNfFJ\\ekgARVyP]xao`>\ngGP@DjZjdEDNfFJ\\ekgAbVyP\\Dao`<\ngGP@DjZjdEDNfFJ\\ekgAbVyP\\Dao`>\ngGP@DjZjdEDNfFJ\\ekgAcfyP\\DahP<\ngGP@DjZjdEDNfFJ\\ekgAcfyP\\DahP>\ngGP@DjZjdEDNfFJ\\ekgAcfyP\\Dao`<\ngGP@DjZjdEDNfFJ\\ekgAcfyP\\Dao`>\ngGP@DjZjdEDNfFJ\\emgAbVyP\\DahP<\ngGP@DjZjdEDNfFJ\\emgAbVyP\\DahP>\ngGP@DjZjdEDNfFJ\\emgAcfyP\\DahP<\ngGP@DjZjdEDNfFJ\\emgAcfyP\\DahP>\ngGP@DjZjdEDNfFJ\\fmgAcfyP\\DahP<\ngGP@DjZjdEDNfFJ\\fmgAcfyP\\DahP>\ngGP@DjZjdEDNfFJ]EkfabgI[eApRF~@<\ngGP@DjZjdEDNfFJ]EkfabgI[eApRF~@>\ngGP@DjZjdEDNfFJ]Em\\hNBPtH<\ngGP@DjZjdEDNfFJ]Em\\hNBPtH>\ngGP@DjZjdEDNfFJ]EmgAcfyP\\DahP<\ngGP@DjZjdEDNfFJ]EmgAcfyP\\DahP>\ngGP@DjZjdEDNfFJ]FmgAcfyP\\DahP<\ngGP@DjZjdEHNfFJ\\ekfabgI[eAwbF~@\ngGP@DjZjdEHNfFJ\\ekgARVyP]xao`\ngGP@DjZjdEHNfFJ\\ekgAbVyP\\Dao`<\ngGP@DjZjdEHNfFJ\\ekgAbVyP\\Dao`>\ngGP@DjZjdEHNfFJ\\ekgAcfyP\\DahP\ngGP@DjZjdEHNfFJ\\emgAbVyP\\DahP<\ngGP@DjZjdEHNfFJ\\emgAbVyP\\DahP>\ngGP@DjZjdEHNfFJ\\emgAcfyP\\DahP<\ngGP@DjZjdEHNfFJ\\emgAcfyP\\DahP>\ngGP@DjZjdEHNfFJ\\fkgAbVyP\\Dao`<\ngGP@DjZjdEHNfFJ\\fkgAbVyP\\Dao`>\ngGP@DjZjdEHNfFJ\\fmgAcfyP\\DahP<\ngGP@DjZjdEHNfFJ\\fmgAcfyP\\DahP>\ngGP@DjZjdEHNfFJ]EkfabgI[eApRF~@>\ngGP@DjZjdEHNfFJ]EkgAbVyP\\Dao`<\ngGP@DjZjdEHNfFJ]EkgAbVyP\\Dao`>\ngGP@DjZjdEHNfFJ]Em\\hNBPtH\ngGP@DjZjdEHNfFJ]EmgAbVyP\\DahP<\ngGP@DjZjdEHNfFJ]EmgAbVyP\\DahP>\ngGP@DjZjhDNfFJ\\fmdacdmr`xICP`<\ngGP@DjZjhDNfFJ\\fmdacdmr`xICP`>\ngGP@DjZjhDNfFJ]Fm\\hNBPtH>\ngGP@DjZjiDNfFJ\\ek\\hNBPwp<\ngGP@DjZjiDNfFJ\\ek\\hNBPwp>\ngGP@DjZjiDNfFJ\\ekdabdmr`xIC_@<\ngGP@DjZjiDNfFJ\\ekdabdmr`xIC_@>\ngGP@DjZjiDNfFJ\\ekdabgMr`xICP`<\ngGP@DjZjiDNfFJ\\ekdabgMr`xICP`>\ngGP@DjZjiDNfFJ\\ekfabgI[eApRF~@<\ngGP@DjZjiDNfFJ\\ekfabgI[eApRF~@>\ngGP@DjZjiDNfFJ\\em\\hNBPtH<\ngGP@DjZjiDNfFJ\\em\\hNBPtH>\ngGP@DjZjiDNfFJ\\emdabgMr`xICP`<\ngGP@DjZjiDNfFJ\\emdabgMr`xICP`>\ngGP@DjZjiDNfFJ\\emfabgI[eApRFa@<\ngGP@DjZjiDNfFJ\\emfabgI[eApRFa@>\ngGP@Djfj`NfFJ\\ek\\hNBPtH\ngGP@Djfj`NfFJ\\ek\\hNBPtOyJED\ngGP@Djfj`NfFJ\\ek\\hNBPwp\ngGP@Djfj`NfFJ\\ek\\hNB_TH\ngGP@Djfj`NfFJ\\ek\\hNB_Wp\ngGP@Djfj`NfFJ\\ek\\hN|_TH\ngGP@Djfj`NfFJ\\ekdaSdmr`{q}P`\ngGP@Djfj`NfFJ\\ekdaSdmr`{q}_@\ngGP@Djfj`NfFJ\\ekdabdmr`xICP`\ngGP@Djfj`NfFJ\\ekdabdmr`xIC_@\ngGP@Djfj`NfFJ\\ekdabdmr`xI}P`\ngGP@Djfj`NfFJ\\ekgAbVyP\\DahP\ngGP@Djfj`NfFJ\\ekgAbVyP\\Dao`\ngGP@Djfj`NfFJ\\ekgArVyP\\Dah_rTJH\ngGP@Djfj`NfFJ\\ekgArVyP\\D~hP\ngGP@Djfj`NfFJ\\ekgArVyP\\D~o`\ngGP@Djfj`NfFJ\\em\\hNBPtH\ngGP@Djfj`NfFJ\\em\\hNB_TH\ngGP@Djfj`NfFJ\\emgAbVyP\\DahP\ngGP@Djfj`NfFJ\\emgArVyP\\D~hP\ngGP@Djfj`NfFJ\\fk\\hNBPtH\ngGP@Djfj`NfFJ\\fk\\hNBPwp\ngGP@Djfj`NfFJ\\fkdabdmr`xICP`\ngGP@Djfj`NfFJ\\fkdabdmr`xIC_@\ngGP@Djfj`NfFJ\\fkgArVyP\\DahP\ngGP@Djfj`NfFJ\\fkgArVyP\\Dao`\ngGP@Djfj`NfFJ\\fm\\hNBPtH\ngGP@Djfj`NfFJ]Fk\\hNBPtH\ngGP@Djfj`NfFJ]Fk\\hNBPwp\ngGP@Djfj`NfFJ]FkgAbVyP\\DahP\ngGP@Djfj`NfFJ]Fm\\hNBPtH\ngGP@Djfj`NfFJmEk\\hNB_TH\ngGP@Djfj`NfFJmFk\\hNBPtH\ngGP@DjfjhDNfFJ\\ekdabdmr`{q}P`<\ngGP@DjfjhDNfFJ\\ekdabdmr`{q}P`>\ngGP@DjfjhDNfFJ\\ekdacdmr`xI}P`<\ngGP@DjfjhDNfFJ\\ekdacdmr`xI}P`>\ngGP@DjfjhDNfFJ\\ekdacdmr`xI}_@<\ngGP@DjfjhDNfFJ\\ekdacdmr`xI}_@>\ngGP@DjfjhDNfFJ\\ekgARVyP]x~hP<\ngGP@DjfjhDNfFJ\\ekgARVyP]x~hP>\ngGP@DjfjhDNfFJ\\ekgAbVyP\\D~hP<\ngGP@DjfjhDNfFJ\\ekgAbVyP\\D~hP>\ngGP@DjfjhDNfFJ\\fkgAbVyP\\DahP<\ngGP@DjfjhDNfFJ\\fkgAbVyP\\DahP>\ngGP@DjfjhDNfFJ]Ek\\hNB_TH<\ngGP@DjfjhDNfFJ]Ek\\hNB_TH>\ngGP@DjfjhDNfFJ]EkgAbVyP\\D~hP<\ngGP@DjfjhDNfFJ]EkgArVyP\\D~hP<\ngGP@DjfjhDNfFJ]EkgArVyP\\D~hP>\ngGP@DjfjhDNfFJ]FkgARVyP\\DahP<\ngGP@DjfjhDNfFJ]FkgARVyP\\DahP>\ngGP@DjfjhDNfFJ]FkgARVyP\\Dao`<\ngGP@DjfjhDNfFJ]FkgARVyP\\Dao`>\ngGP@Djyj`NbIIZwLwmdMB\ngGP@Djyj`NbIIZwLwndMB\ngGP@Djyj`NbIIZwL{mdMB\ngGP@Djyj`NbIIZwL{ndMB\ngGP@Djyj`NbIIkWLwmdMB\ngGP@Djyj`NbIIkWLwndMB\ngGP@Djyj`NbIIkWL{ndMB\ngGP@Djyj`NbIIkYLTyK\\s^zPtH\ngGP@Djyj`NfFJ\\ek\\snBPtH\ngGP@Djyj`NfFJ\\ek_XICP`\ngGP@Djyj`NfFJ\\ek_XICP\x7FdlRP\ngGP@Djyj`NfFJ\\ekdacdmsNxICP`\ngGP@Djyj`NfFJ\\ekdacdm}`dMB\ngGP@Djyj`NfFJ\\ekdqSdmsNxICP`\ngGP@Djyj`NfFJ\\ekdqSdmsN{qCP`\ngGP@Djyj`NfFJ\\ekdqSdm}`dMB\ngGP@Djyj`NfFJ\\ekfabgI[f]pRFa@\ngGP@Djyj`NfFJ\\ekfabgI[{AHZD\ngGP@Djyj`NfFJ\\ekfabgI[{^HZD\ngGP@Djyj`NfFJ\\fm\\snBPtH\ngGP@Djyj`NfFJ\\fm_XICP`\ngGP@Djyj`NfFJ\\fmdacdmsNxICP`\ngGP@Djyj`NfFJ]Ek\\snBPtH\ngGP@Djyj`NfFJ]Ek_XICP`\ngGP@Djyj`NfFJ]Ekdacdm}`dMB\ngGP@Djyj`NfFJ]EkfabgI[f]pRFa@\ngGP@Djyj`NfFJ]EkfabgI[{AHZD\ngGP@Djyj`NfFJ]FmdacdmsNxICP`\ngGP@DjyjhdNbIIZwLwmdM|>\ngGP@DjyjhdNbIIZwLwndM|<\ngGP@DjyjhdNbIIZypXynYw[HZD<\ngGP@DjyjhdNbIIZypXynYw[HZD>\ngGP@DjyjhdNbIIZypXynYw]HZD<\ngGP@DjyjhdNbIIZypXynYw]HZD>\ngGP@DjyjhdNbII[WLwndMB<\ngGP@DjyjhdNbII[WLwndMB>\ngGP@DjyjhdNbII[WL{ndMB<\ngGP@DjyjhdNbII[WL{ndMB>\ngGP@DjyjhdNbII[YpXynYw]HZD<\ngGP@DjyjhdNbII[YpXynYw]HZD>\ngGP@DjyjhdNbII[YtTynYw]HZD<\ngGP@DjyjhdNbIIjypXenYw]H[x<\ngGP@DjyjhdNfFJ\\ek_XIC_@<\ngGP@DjyjhdNfFJ\\ek_XIC_@>\ngGP@DjyjhdNfFJ\\ekdacdm}`dM|<\ngGP@DjyjhdNfFJ\\ekdacdm}`dM|>\ngGP@DjyjhdNfFJ\\ekdacdm}`dM}~RqI@>\ngGP@DjyjhdNfFJ\\ekfabgI[f]pRF~@>\ngGP@DjyjhdNfFJ\\ekfabgI[{AH[x<\ngGP@DjyjhdNfFJ\\ekfabgI[{AH[x>\ngGP@DjyjhdNfFJ\\ekfabgI[{^H[x<\ngGP@DjyjhdNfFJ\\ekfabgI[{^H[x>\ngGP@DjyjhdNfFJ\\ekgAbVyg\\Dao`<\ngGP@DjyjhdNfFJ\\ekgAbVyg\\Dao`>\ngGP@DjyjhdNfFJ\\ekgAbV~pRF~@<\ngGP@DjyjhdNfFJ\\ekgAbV~pRF~@>\ngGP@DjyjhdNfFJ\\ekgAcfyg\\DahP<\ngGP@DjyjhdNfFJ\\ekgAcfyg\\DahP>\ngGP@DjyjhdNfFJ\\ekgAcf~pRFa@<\ngGP@DjyjhdNfFJ\\ekgAcf~pRFa@>\ngGP@DjyjhdNfFJ\\ekgQRV~pRF~@<\ngGP@DjyjhdNfFJ\\ekgQRV~pRF~@>\ngGP@DjyjhdNfFJ\\ekgQRV~wbF~@<\ngGP@DjyjhdNfFJ\\ekgQRV~wbF~@>\ngGP@DjyjhdNfFJ\\ekgQSfyg\\DahP<\ngGP@DjyjhdNfFJ\\ekgQSfyg\\DahP>\ngGP@DjyjhdNfFJ\\ekgQSf~pRFa@<\ngGP@DjyjhdNfFJ\\ekgQSf~pRFa@>\ngGP@DjyjhdNfFJ\\ekgQSf~wbFa@<\ngGP@DjyjhdNfFJ\\ekgQSf~wbFa@>\ngGP@DjyjhdNfFJ\\em\\snBPtH<\ngGP@DjyjhdNfFJ\\em\\snBPtH>\ngGP@DjyjhdNfFJ\\em_XICP`<\ngGP@DjyjhdNfFJ\\em_XICP`>\ngGP@DjyjhdNfFJ\\emdacdmsNxICP`<\ngGP@DjyjhdNfFJ\\emdacdmsNxICP`>\ngGP@DjyjhdNfFJ\\emgAbVyg\\DahP<\ngGP@DjyjhdNfFJ\\emgAbVyg\\DahP>\ngGP@DjyjhdNfFJ\\emgAbV~pRFa@<\ngGP@DjyjhdNfFJ\\emgAcfyg\\DahP<\ngGP@DjyjhdNfFJ\\emgAcfyg\\DahP>\ngGP@DjyjhdNfFJ\\emgAcf~pRFa@<\ngGP@DjyjhdNfFJ\\emgAcf~pRFa@>\ngGP@DjyjhdNfFJ\\emgQSfyg\\DahP<\ngGP@DjyjhdNfFJ\\emgQSf~pRFa@>\ngGP@DjyjhdNfFJ\\fkgAbVyg\\Dao`<\ngGP@DjyjhdNfFJ\\fkgAbVyg\\Dao`>\ngGP@DjyjhdNfFJ\\fkgAbV~pRF~@<\ngGP@DjyjhdNfFJ\\fmgAcfyg\\DahP<\ngGP@DjyjhdNfFJ\\fmgAcfyg\\DahP>\ngGP@DjyjhdNfFJ]EkgAcfyg\\DahP<\ngGP@DjyjhdNfFJ]EkgAcfyg\\DahP>\ngGP@DjyjhdNfFJ]EkgAcf~pRFa@<\ngGP@DjyjhdNfFJ]EkgQSfyg\\DahP<\ngGP@DjyjhdNfFJ]EkgQSfyg\\DahP>\ngGP@DjyjhdNfFJ]EmgAcf~pRFa@>\ngGP@Dkij`NfFJ\\ek\\snBPtH\ngGP@Dkij`NfFJ\\ek_XICP`\ngGP@Dkij`NfFJ\\ek_XIC_@\ngGP@Dkij`NfFJ\\ekdabgMsNxICP`\ngGP@Dkij`NfFJ\\ekdabgM}`dMB\ngGP@Dkij`NfFJ\\ekgAbVyg\\Dao`\ngGP@Dkij`NfFJ\\ekgAbV~pRF~@\ngGP@Dkij`NfFJ\\ekgAbfyg\\DahP\ngGP@Dkij`NfFJ\\ekgAbf~pRFa@\ngGP@Dkij`NfFJ\\em_XICP`\ngGP@Dkij`NfFJ\\fmgAbfyg\\DahP\ngGP@Dkij`NfFJ\\fmgAbf~pRFa@\ngGP@DkijhdNfFJ\\ekdabdm}`dM|<\ngGP@DkijhdNfFJ\\ekgAbfyg\\Dao`>\ngGP@DkijhdNfFJ\\ekgAbf~pRF~@<\ngGP@DkijhdNfFJ\\ekgAbf~pRF~@>\ngGP@DkijhdNfFJ\\emgAbfyg\\DahP<\ngGP@DkijhdNfFJ\\emgAbfyg\\DahP>\ngGP@DkijhdNfFJ\\emgAbf~pRFa@<\ngGP@DkijhdNfFJ\\emgAbf~pRFa@>\ngGP@Lda`U@][IZwLvmdMB=\ngGP@Lda`U@][IZwLvmdMC~RqI@=\ngGP@Lda`U@][IZwLvmgu|=\ngGP@Lda`U@][IZwLvmgu}~RqI@=\ngGP@Lda`U@][IZwLvndMB\ngGP@Lda`U@][IZwLvndMC~RqI@\ngGP@Lda`U@][IZwLzndMB=\ngGP@Lda`U@][IZwLzndMC~RqI@=\ngGP@Lda`U@][IZyHTyK\\sjzPtH\ngGP@Lda`U@][IZyLTyK\\sZzPtH\ngGP@Lda`U@][IZyhXirVygUtahP=\ngGP@Lda`U@][IkWLvmdMB=\ngGP@Lda`U@][IkWLvndMB\ngGP@Lda`U@][IkWLzndMB=\ngGP@Lda`U@][IkYHTyK\\sjzPtH\ngGP@Lda`U@][IkYLTyK\\sZzPtH\ngGP@Lda`USH][IZwLvmdM|+\ngGP@Lda`USH][IZwLvmdM|-\ngGP@Lda`USH][IZwLvmdM}~RqI@+\ngGP@Lda`USH][IZwLvmdM}~RqI@-\ngGP@Lda`USH][IZwLvndM|<\ngGP@Lda`USH][IZwLvndM|>\ngGP@Lda`USH][IZyHTyK\\sjzPwp>\ngGP@Lda`USH][IZyLTyK\\sZzPwp>\ngGP@Lda`USH][IZyLTyK\\sZzPwwyKDd>\ngGP@Lda`USH][IZypTynYu]HZD<\ngGP@Lda`USH][IZypTynYu]HZD>\ngGP@Lda`USH][IZytTenYm]H[x<\ngGP@Lda`USH][IZytTenYm]H[x>\ngGP@Lda`USH][IZytTenYm]H[{|ebR<\ngGP@Lda`USH][IZytTenYm]H[{|ebR>\ngGP@Lda`USH][IZytTynYm]HZD<\ngGP@Lda`USH][IZytTynYm]HZD>\ngGP@Lda`USH][IZytTynYm]HZG|ebR<\ngGP@Lda`USH][IZytTynYm]HZG|ebR>\ngGP@Lda`USH][IZytTynYm]H[x<\ngGP@Lda`USH][IZytTynYm]H[x>\ngGP@Lda`USH][IZytTynYm]Okx<\ngGP@Lda`USH][IZytTynYm]Okx>\ngGP@Lda`USH][I[WLvmdMB+\ngGP@Lda`USH][I[WLvmdMB-\ngGP@Lda`USH][I[WLvndMB<\ngGP@Lda`USH][I[WLvndMB>\ngGP@Lda`USH][I[WLzndMB+\ngGP@Lda`USH][I[WLzndMB-\ngGP@Lda`USH][I[YHTyK\\sjzPtH<\ngGP@Lda`USH][I[YHTyK\\sjzPtH>\ngGP@Lda`USH][I[YLTyK\\sZzPtH<\ngGP@Lda`USH][I[YLTyK\\sZzPtH>\ngGP@Lda`USH][I[YpTenYu]HZD<\ngGP@Lda`USH][I[YpTenYu]HZD>\ngGP@Lda`USH][I[YpTynYu]HZD<\ngGP@Lda`USH][I[YpTynYu]HZD>\ngGP@Lda`USH][I[YtTenYm]HZD<\ngGP@Lda`USH][I[YtTenYm]HZD>\ngGP@Lda`USH][I[YtTynYm]HZD<\ngGP@Lda`USH][I[YtTynYm]HZD>\ngGP@Lda`USH][IjwLvmdM|+\ngGP@Lda`USH][IjwLvmdM|-\ngGP@Lda`USH][IjwLvndM|<\ngGP@Lda`USH][IjwLvndM|>\ngGP@Lda`USH][IjyHTyK\\sjzPwp<\ngGP@Lda`USH][IjyHTyK\\sjzPwp>\ngGP@Lda`USH][IjyLTyK\\sZzPwp<\ngGP@Lda`USH][IjyLTyK\\sZzPwp>\ngGP@Lda`USH][IjytTenYm]H[x<\ngGP@Lda`USH][IjytTenYm]H[x>\ngGP@Lda`USH][IjytTynYm]H[x<\ngGP@Lda`USH][IjytTynYm]H[x>\ngGP@Lda`USH][IkYpTynYu]HZD<\ngGP@Lda`USH][IkYpTynYu]HZD>\ngGP@Lda`USH][IkYtTynYm]HZD<\ngGP@Lda`USH][IkYtTynYm]HZD>\ngGPAEJU\\uPGSCENRusPqSdmsNxICP`\ngGPAEJU\\uPGSCENRusPqSdm}`dMB\ngGPAKrUM@@FyV}l}oP=\ngGPAMJUML@GVBgN\\EI[{AN{t\ngGPAMNRPpJ`Nmdm[f]WRFa@=\ngGPAMNRPpJ`Nmdukf[VrFa@=\ngGPAMNRPpJ`Nmdukf[WRFa@\ngGPAMNRPpJ`Nmdukf]WRFa@=\ngGPAMNRPpJ`NmdultLTyK\\sjzPtH=\ngGPAMNRPpJ`NmdulxJ\\wLzndMB>\ngGPAMNRPpJ`NmdulzJ\\wLvndMB>\ngGPAMvRPpJ`Nijyful}oP=\ngGPB@DZPLaYAJZjhCkAbf|PP\ngGPB@DZPLaYAJZjhCkAbj|PP\ngGPBABRDKaWAJnjhCkASf~pSz~@>\ngGPBADJHLQXaInjhCiQr\ngGPD@DDHRYjY@zTeWHwmg]v\ngGPD@DDHRYjY@zTeWHwmg]z\ngGPD@DDHRYjY@zTeWHwng]z\ngGPD@DDHRYjY@zTeYtTenQo]N{t\ngGPD@DDHRYjY@ztdunao[@\ngGPD@DDHRYjjTR`zLTykgAcfzpZD>\ngGPD@DFPRYfi@zLTykgQRVzp[t\ngGPD@DGpRYjY@ztduoao[oP\ngGPD@DHHRZzj@zL\\ek]DMB\ngGPD@DHHRZzj@zL\\ekgIrVyGexahP\ngGPD@DHHRZzj@zL\\ekgIrVzHZD\ngGPD@DHHR[f`@zd[h{np=\ngGPD@DHHR[ff@zL\\ek]GmB\ngGPD@DHHR[j`@zd[h{np=\ngGPD@DHHR[j`@zd[h{oP\ngGPD@DKpRYjZ@zLdfm\\c^vPtH\ngGPD@DSpRZVj@zlTyKV~pRFa@\ngGPD@DXHRfjj@zhXiKV~PRFa@\ngGPD@DXHRfjj@zhXiKWNCEM|`dMB\ngGPD@DXHRfjj@zhXjKV~PRFa@\ngGPD@DXHRfjjdPzhXiK[NCEM|`dMB<\ngGPD@DXHRfjjdPzhXiK[NCEM|`dMB>\ngGPD@DZ`R[ij@zLlfm\\cZvPtH\ngGPD@VIpRVjj@zlTyKV~pRFa@\ngGPD@bSpRZVj@zlTzMV~pRFa@\ngGPD@xIpR[j`@zd[d^Wcn{@=\ngGPD@xIpR[j`@zd[d^Wcn}@\ngGPD@xQpRjzj@zhTyKVyglDahP\ngGPD@xQpRkf`@zP\\wL{`gmz+\ngGPD@xQpRkf`@zP]WL{`g]v+\ngGPD@xQpRkf`@zP]WL{`gmz+\ngGPD@xUpRjZj@zhXirVyP\\Dao`\ngGPDADFHRYjY@zLTykgQRVzw[t\ngGPDADFHRYjjTRPzLTyk]XMB\ngGPDAHFPRYjjTRPzLTykgQRVzpZD\ngGPDAHFPRYjjTRPzLTykgQRVzp[x<\ngGPDAHFPRYjjTRPzLTykgQRVzp[x>\ngGPDAHJPRZzj@zL\\ek]DMB\ngGPDAHJPRZzj@zL\\ek]DM|\ngGPDAHJPRZzj@zL\\ek]GuB\ngGPDAbGDRYjjTRPzLTyk]XMB\ngGPDAbGDRYjjTRPzLTym]XMB>\ngGPDAbR`RYj`@zT]YtTeolD{oP>\ngGPDAxKpRYfj@zlTyKV~pRFa@\ngGPDAxSpRijj@zXXirVmr`xICP`\ngGPDAxSpRkj``PzP]WL}`g]v+\ngGPDAxSpRkj``PzP]WL}`g]v-\ngGPDAxSpRkj``PzP]WvB]wX+\ngGPDAxSpRkj``PzP]WvB]wX-\ngGPDAxSpRkjj`PzhTyKV~pRFa@<\ngGPDAxSpRkjj`PzhTyKV~pRFa@>\ngGPDaxF`Dgdjfj`NjFJ\\unTGAOjD\ngGPDaxF`Dgdjfj`NjFJ\\uodGAhP\ngGPDaxF`DgdjfjhDNjFJ]UodGAhP<\ngGPDaxF`DgdjfjhDNjFJ]UodGAhP>\ngGPEAxSpdiUuPPH]HNkf^pSn{@+\ngGPEAxSpdiUuPPH]HNkf^pSn{@-\ngGPFAVRDTIhPueaDfZj`NeEN]EIP>\ngGPH@DIJm]@]ZIZweB\ngGPH@DIKUU@]VJ\\ek\\cvBPtH\ngGPH@DIKUU@]VJ\\ek\\cvBPwp\ngGPH@DIKUU@]VJ\\ek_XICP`\ngGPH@DIKUU@]VJ\\ekgAbf~pRFa@\ngGPH@DILsP@]JNlzJRwvB]wh\ngGPH@DILuH@]JN[d^wSn{@=\ngGPH@DILuH@]JN[h{np=\ngGPH@DILuH@]JN[x^N{l=\ngGPH@DILuH@]JN[{]N{l=\ngGPH@DILuH@]RUrM{Yw]`=\ngGPH@DILuH@]RUrM{Yw^`\ngGPH@DILuH@]RUrM{Y{^`=\ngGPH@DILuH@]RUrM{iw]`=\ngGPH@DILuH@]RUrM{iw^`\ngGPH@DILuH@]RV]EI[d[wSn}@\ngGPH@DILuK@]FRRunQo[OZD\ngGPH@DILuK@]FRRunQo]OZD\ngGPH@DILuK@]FRRunQo]O[x\ngGPH@DILuK@]FRRuoao[hP\ngGPH@DILuK@]FRRvnQo[OZD\ngGPH@DILuK@]JRkd[vsn}@\ngGPH@DILuK@]JRkd[vsv}@\ngGPH@DILuPRH]JN[d^pSn{@+\ngGPH@DILuPRH]JN[d^pSn{@-\ngGPH@DILuPRH]JN[{AN{l+\ngGPH@DILuPRH]JN[{AN{l-\ngGPH@DILuPRH]JN[{AN{t>\ngGPH@DILuPRH]JNk{AN{l-\ngGPH@DIMMU@]VJ\\ekdqSdm}`dMB\ngGPH@DIMMU@]VJ\\ekgAbV~pRFa@\ngGPH@DIMMURH]VJ\\ekgQRV~pRFa@<\ngGPH@DIMMURH]VJ\\ekgQRV~pRFa@>\ngGPH@DIMtu@]FVRunQm[HZD\ngGPH@DIMtu@]FVRunQm]HZD\ngGPH@DISUU@]LLTyKVyP\\DahP\ngGPH@DISUU@]LLTyKWNCDmr`xIC_@\ngGPH@DISUU@]LLTyKWNCEMr`xICP`\ngGPH@DISUU@]LLTzMZyP\\DahP\ngGPH@DISUU@]LLUYKVyP\\DahP\ngGPH@DISUU@]LLUYKWNCEMr`xICP`\ngGPH@DISUU@]LLUYKZyP\\DahP\ngGPH@DISm]@]DJRundFa@\ngGPH@VILuH@]JN[h{np=\ngGPH@VILuH@]JN[x^N{l=\ngGPH@VILuH@]JN[{]N{l=\ngGPH@bILuH@]JN[h{np=\ngGPH@bILuH@]JN[{]N{l=\ngGPH@bILuH@]RUrM{Yw]`=\ngGPH@bILuH@]RUrM{iw]`=\ngGPH@bISUU@]LLUYKVyP\\DahP\ngGPH@xIKUU@]VJ\\ek\\cvBPtH\ngGPH@xILuH@]JN[d^wSn{@=\ngGPH@xILuH@]JN[x^N{l=\ngGPH@xIMMU@]VJ\\ek\\cvBPtH\ngGPH@xITuP@]XJ\\wJC`g]z\ngGPH@xITuP@]XJmWJC`g]v=\ngGPH@xITuP@]XJmWJC`g]z\ngGPH@xITuPPH]XJ]WJC`g]v+\ngGPH@xITuPPH]XJ]WJC`g]v-\ngGPH@xITuU@]LLTyKVyP\\DahP\ngGPH@xIUMT`]LLTyKVyP\\DaoP\ngGPH@xIUMT`]LLTyKV~P\\Da@\ngGPH@xIUMT`]LLTyMV~P\\Da@\ngGPH@xIUMT`]LLTzMV~P\\Da@\ngGPH@xIUMU@]LLTyKVyP\\DahP\ngGPH@xIUMU@]LLTyKVyP\\Dao`\ngGPH@xIUMU@]LLTyKVyP\\D~hP\ngGPH@xIUMU@]LLTyKWNCDmr`xICP`\ngGPH@xIUMU@]LLTyKZyP\\DahP\ngGPH@xIUMU@]LLTyMVyP\\DahP\ngGPH@xIUMU@]LLTzMVyP\\DahP\ngGPH@xIUMU@]LLTzMWNCDmr`xICP`\ngGPH@xIUMU@]LLTzMZyP\\DahP\ngGPHABILuH@]JN[h{np=\ngGPHABITuP@]XJmWJC`g]v=\ngGPHABITuP@]XJmWJC`g]z\ngGPHABITuP@]XJmWJC`gmz=\ngGPHABITuS@]LLTyKWNCGMr`xI{P`>\ngGPHADILuH@]JN[h{np=\ngGPHADILuH@]JN[{]N{l=\ngGPHADILuH@]JN\\zJRwvz]wh\ngGPHADILuPRH]JN[{AN{l+\ngGPHADILuPRH]JN[{AN{l-\ngGPHADILuUJIP]VJ\\ekgQRV~pRF~@<\ngGPHADIMMU@]VJ\\ek_XICP`\ngGPHADIMMU@]VJ\\ekdqSdm}`dMB\ngGPHADIMMURH]VJ\\ekgQRV~pRFa@<\ngGPHADIMMURH]VJ\\ekgQRV~pRFa@>\ngGPHADIMMURH]VJ\\ekgQrV~pSza@>\ngGPHAHILuH@]JN[{]N{l=\ngGPHAHILuK@]VJ\\ekgQRVzOZD\ngGPHAHILuK@]VJ\\ekgQRV~wSva@\ngGPHAHILuPRH]JN[{AN{l-\ngGPHAHIMMU@]VJ\\ekdqSdm}`dMB\ngGPHAHIMMURH]VJ\\ekgQRV~pRFa@<\ngGPHAHIMMURH]VJ\\ekgQRV~pRFa@>\ngGPHAVILuH@]JN[{]N{l=\ngGPHAVILuH@]JN\\zJRwvz]wh\ngGPHAVIMMU@]VJ\\ekdqSdm}`dMB\ngGPHAbIKUU@]VJ\\ek_XICP`\ngGPHAbILuH@]JN[{]N{l=\ngGPHAbILuPRH]JN[{AN{l+\ngGPHAbILuPRH]JN[{AN{l-\ngGPHAbILuPRH]JN\\zJ\\wvB^wh<\ngGPHAbILuUJIH]VJ]EmgQSf~pRFa@>\ngGPHAbIMMU@]VJ\\ekdqSdm}`dM|\ngGPHAxILsP@]JNk{AN{l=\ngGPHAxILuH@]JN[d^wSn{@=\ngGPHAxILuH@]JN[h{np=\ngGPHAxILuH@]JN[x^N{l=\ngGPHAxILuH@]JN[x^N{t\ngGPHAxILuH@]JN[{]N{l=\ngGPHAxILuPRH]JN[d^pSn{@+\ngGPHAxILuPRH]JN[d^pSn{@-\ngGPHAxILuUJIH]VJ\\ekgQSfyGlDahP<\ngGPHAxIMMU@]VJ\\ekdqSdmrOXICP`\ngGPHAxIMMU@]VJ\\ekdqSdm}`dMB\ngGPHAxITuP@]XJ\\wJC`g]v=\ngGPHAxITuP@]XJ\\wJC`g]z\ngGPHAxITuP@]XJ\\wJC`gmz=\ngGPHAxITuP@]XJmWJC`g]v=\ngGPHAxITuP@]XJmWJC`g]z\ngGPHAxITuP@]XJmWJC`gmz=\ngGPHAxITuP@]XJmYHXiK\\hNB]wh\ngGPHAxITuP@]XJmYhXirVyP\\D}oP=\ngGPHAxITuPPH]XJ]WJC`g]v+\ngGPHAxITuPPH]XJ]WJC`g]v-\ngGPHAxITuPPH]XJ]WJC`g]z<\ngGPHAxITuPPH]XJ]WJC`g]z>\ngGPHAxITuPPH]XJ]WJC`gmz+\ngGPHAxITuPPH]XJ]WJC`gmz-\ngGPHAxITuPPH]XJlwJCoG]v+\ngGPHAxITuPPH]XJlwJCoG]v-\ngGPHAxITuS@]LLUZKVyP\\D}hP\ngGPHAxITuU@]LLTyKVyP\\DahP\ngGPHAxITuU@]LLUZKVyP\\DahP\ngGPHAxITuUPH]LLTzKVyP\\DahP<\ngGPHAxITuUPH]LLTzKVyP\\DahP>\ngGPHKPITuURH]TLTyM\\hNBPwp<\ngGPHKPYIC@jfPzVRwtvmdH+\ngGPHKPYIC@jfPzVRwtvmdH-\ngGPHKPYIC@jfPzVRytTenYm]H[x>\ngGPHMPITtp@]HJ[e]wcn{@=\ngGPHMPITtp@]HJ[e]wcv}@=\ngGPHMPITtp@]HJ[h{np=\ngGPHMPITtp@]HJ[y]N{l=\ngGPHMPITtp@]HJ[y]O[t=\ngGPHMPYIC@j@zvRunYu]HZD=\ngGPHMPYIC@j@zvRvnYu]HZD+\ngGPHMPYIC@j@zvRvshis\\sZzPtH>\ngGPHMPYIC@j@zvSVnYm[HZD=\ngGPHMPYIC@j@zvSVnYm]HZD\ngGPHMPYIC@j@zvSVnYu]HZD=\ngGPHMPYIC@j@zvSVsPqSdmsNkiCP`=\ngGPHMPYIC@j@zvSVshis\\sZzPtH>\ngGPHMPYICFj@zXXjrVv\\EN[f]PRFa@>\ngGPHMPYICFj@zXXjrZusNhICP`\ngGPHMPYICJj@zXXirVnRFJ\\wLB`dMB>\ngGPHaxCeJSLuPGUbgI[YtTynQwAHZD>\ngGPI@DBTjZh@NlEVkeApSn{@=\ngGPI@DBTjZhHDNlENkeApSn}@<\ngGPI@DBTjZhHDNlENlxJRwJC`g]z<\ngGPI@DBTjZhHDNlENlxJRwJC`g]z>\ngGPI@DZTfzZ`NcKIkWHvmdMB\ngGPI@VBTjZh@NlEVkeApSn{@=\ngGPIAxBTfZd@NeGMrO[iw]`=\ngGPIAxBTfZhIDNeGMrOXIw]`+\ngGPIAxBTfZhIDNeGMrOXIw]`-\ngGPIAxBTffj`NkENRurXirVyGlDahP\ngGPIAxBTffj`NkENRurXirV~pRFa@\ngGPIAxBTijj`NfFJ\\ek\\kvBPtH\ngGPIAxBTjZh@NlEN[eApSn}@\ngGPIAxBTjZh@NlEVkeApSn{@=\ngGPIAxBTjZh@NlEVkeApSn}@\ngGPIAxBTjZh@NlEVkeApSv}@=\ngGPIAxBTjZhHDNlENkeApSv}@+\ngGPIAxBTjZhHDNlENkeApSv}@-\ngGPIAxBTjZj`NfFJ\\ek\\hNBPtH\ngGPIAxBTjZjhDNfFJ]Ek\\hNBPtH<\ngGPIAxBTjZjhDNfFJ]Ek\\hNBPtH>\ngGPIAxBTjfjPNfFJmFk\\hNBPwh\ngGPIAxBTjfj`NfFJ\\ek\\hNBPtH\ngGPIAxBTjfj`NfFJ\\em\\hNBPtH\ngGPIAxBTjfj`NfFJ\\fk\\hNBPtH\ngGPIAxBTjfj`NfFJ]Fk\\hNBPtH\ngGPIAxBTjfj`NfFJ]FkgAbVyP\\DahP\ngGPIAxBTjfj`NfFJmFk\\hNBPtH\ngGPIAxBTjfj`NfFJmFm\\hNBPtH\ngGPIEPKdkij`NfFJ]Fm\\snBPtH\ngGPIEPKdkjj`NfFJ\\em\\pNBPtH\ngGPIEPKdkjj`NfFJ\\fm\\pNBPtH\ngGPIEPKdkjj`NfFJ\\fmgAbfy`\\DahP\ngGPIEPKdkjj`NfFJ]Fm\\pNBPtH\ngGPIMP[djZX@NdEMrn{qw]`=\ngGPIMP[djZX@NdEMrn{q{^`=\ngGPIMP[lda`U@][I[WLzndMB+\ngGPIMP[lda`U@][IkWLvndMB\ngGPIMP[lda`U@][IkWLzndMB=\ngGPIMP[ldaeU@]LLTyK[ICEN[fAPRFa@>\ngGPIMP[ldaeU@]LLTyMZy`TDahP\ngGPJ@DQpSa``aBTuUPGVCFvRFJ\\wR}P`\ngGPJ@xQpSagBnFRRVj`NdDmsOKv}_@\ngGPL@DXHHPeMUTAtPqSWICEN[i^hP\ngGPL@xIpsadTajhCkQKWNSdm|OKuB\ngGPL@xQpSaddLZ@CjAnYu^O[t+\ngGPL@xQpSaddajhCharVmsOKqCP`\ngGPL@xQpSaddajhCharVmsOKqC__d`XP\ngGPLADHHhPdtuTAtXirVzwc{|dcb\ngGPLAHJPtadTajdCkQKT\ngGPLAHJPtadTajdCkQKV|oP\ngGPLAHJPtadTajhCkQKV|hP\ngGPLAHJPtadTajhCkQKV|o`\ngGPLAHJPtadTajhCkQKZ|hP\ngGPP@cTfyi`NeGH\ngGPPAcTijj`NlFJ\\xLTwbB\ngGPP`cUuyInZXCiQr\ngGPQ@c[rS\\tpGRcd\ngGPT@cYPZ`R[jj@zT\\`\ngGPTHlPxUpRjZj@zPXwe|\ngGPTHlPxUpRjZj@zPXypXeoKx\ngGPTHlQxUpRjvj@zPTwe|\ngGPTHlQxWpRjvj@zPTwe|\ngGPTIcQxIpRVjj@zTTw`|\ngGPUHlQxWpOIU[UPH]HJ[r~@<\ngGPUHlQxWpOIU[UPH]HJ[r~@>\ngGPXHlPDYIHUj@zp\\uoJD\ngGPXHlPDYIHUj@zp\\usdyK^TH\ngGPXHlPxIUMU@]XLZweB\ngGPXHlPxIUMU@]XL[WeB\ngGPXHlQxIU[U@]XJZweB\ngGPXHlQxIU[U@]XJZyLTyK^TH\ngGPXHlQxIU[U@]XJ[WeB\ngGPXHlQxIU[UPH]XJZytTeoJD<\ngGPXHlQxIU[UPH]XJZytTeoJD>\ngGPXHlQxIU[UPH]XJ[YtTeoJD<\ngGPXHlQxIU[UPH]XJ[YtTeoJD>\ngGPXIcPVIKUU@]JJP\ngGPXhlQx[crUVuPGVBfmyP`\ngGPYAcUPKdkjj`NlFJ[rA@\ngGPYAcUPKdkjj`NlFJ\\xLTwdB\ngGPYAcUPKdkjj`NlFR\\xLTwdB\ngGPYHlPDGlddJu@]XNZweB\ngGPYHlQxGdjmjhDNlEM[ra@<\ngGPYHlQxGdjmjhDNlEM[ra@>\ngGPYHlQxGdjmjhDNlEM\\zJRweB<\ngGPYHlQxGdjmjhDNlEM\\zJRweB>\ngGPYHlQxGdjmjhDNlEMkra@<\ngGPYHlQxGdjmjhDNlEMkra@>\ngGPYHlQxGdjmjhDNlEMlzJRweB<\ngGPYHlQxGdjmjhDNlEMlzJRweB>\ngGP`@TfYi`NcENZytTenlFa@\ngGP`@TfYi`NcEN[YtTenlFa@\ngGP`@TfYi`NcENkWVCP`\ngGP`@TfYi`NcENkYtTenlFa@\ngGP`@TfYjPNcENZytTenlF}@\ngGP`@TfYjPNcENZytTenmz}@\ngGP`@TfYjPNcENZytTeoLD\ngGP`@TfYjPNcENjytTenlF}@\ngGP`@TfYjPNcENjytTeoLD\ngGP`@TfYjPNmEV[{AN{l\ngGP`@TfYjPNmEV[{AN{t\ngGP`@TfYjPNmEVk{AN{l\ngGP`@TfYjPNmEVk{AN{t\ngGP`@TfYjPNmEVlzJRwvB]wh\ngGP`@TfYjhhNcENZwVCP`<\ngGP`@TfYjhhNcENZwVCP`>\ngGP`@TfYjhhNcENZwV}P`<\ngGP`@TfYjhhNcENZyLTyK]XMB<\ngGP`@TfYjhhNcENZyLTyK]XMB>\ngGP`@TfYjhhNcENZytTenlFa@<\ngGP`@TfYjhhNcENZytTenlFa@>\ngGP`@TfYjhhNcENZytTenlF~@<\ngGP`@TfYjhhNcENZytTenlF~@>\ngGP`@TfYjhhNcENZytTenmza@<\ngGP`@TfYjhhNcENjwVCP`<\ngGP`@TfYjhhNcENjwVCP`>\ngGP`@TfZfPNeIUrM{Yw]`\ngGP`@TfZfPNeIV]EI[d[wSn}@\ngGP`@TfZjeDdNcENZwVCP`\ngGP`@TfZjeDdNcENZwVC_@<\ngGP`@TfZjeDdNcENZyHXiK]XMB<\ngGP`@TfZjeDdNcENZyHXiK]XMB>\ngGP`@TfZjeDdNcENZyHXiK]XM|<\ngGP`@TfZjeDdNcENZyhXirVzpZD\ngGP`@TfZjeDdNcENZypXenlFa@<\ngGP`@TfZjeDdNcENZypXenlFa@>\ngGP`@TfZjeDdNcENZypXenlF~@>\ngGP`@TfZjeDdNcENZytTenlFa@\ngGP`@TfZjeDdNcENZytTenlF~@<\ngGP`@TfZjeDdNcENZytTenlF~@>\ngGP`@TfZjeDdNcEN[WVCP`<\ngGP`@TfZjeDdNcEN[WVCP`>\ngGP`@TfZjeDdNcEN[YtTynlFa@<\ngGP`@TfZjeDdNcEN[YtTynlFa@>\ngGP`@TfZjeDdNcENkWVCP`\ngGP`@TfZjeDhNcENZwVCP`<\ngGP`@TfZjeDhNcENZwVCP`>\ngGP`@TfZjeDhNcENZwVC_@<\ngGP`@TfZjeDhNcENZwVC_@>\ngGP`@TfZjeDhNcENZyHXiK]XMB<\ngGP`@TfZjeDhNcENZyHXiK]XMB>\ngGP`@TfZjeDhNcENZyhXirVzpZD<\ngGP`@TfZjeDhNcENZyhXirVzpZD>\ngGP`@TfZjeDhNcENZypXenlFa@<\ngGP`@TfZjeDhNcENZypXenlFa@>\ngGP`@TfZjeDhNcENZypXynlFa@>\ngGP`@TfZjeDhNcENZytTenlFa@<\ngGP`@TfZjeDhNcENZytTenlFa@>\ngGP`@TfZjeDhNcENZytTenlF~@<\ngGP`@TfZjeDhNcENZytTenlF~@>\ngGP`@TfZjeDhNcENZytTenmz~@<\ngGP`@TfZjeDhNcEN[WVCP`<\ngGP`@TfZjeDhNcEN[WVCP`>\ngGP`@TfZjeDhNcEN[YpXenlFa@<\ngGP`@TfZjeDhNcENkWVCP`<\ngGP`@TfZjeDhNcENkWVCP`>\ngGP`@TfzZihNcENZweB>\ngGP`@TfzZihNcENZytTeneva@>\ngGP`@TfzZihNmKM[x[VzD<\ngGP`@TfzZihNmKM[x[VzD>\ngGP`@dfnjPNcGIZyr\\enbF}@\ngGP`@dfnjPNcGIZyr\\eoBD\ngGP`@dfnj`NcGIZwQCP`\ngGP`@dfnj`NcGIZyHXiK]DMB\ngGP`@dfnj`NcGIZyHXiK]DM|\ngGP`@dfnj`NcGIZyr\\enbFa@\ngGP`@dfnj`NcGIZyr\\enbF~@\ngGP`@dfnj`NcGIZyr\\encza@\ngGP`@dfnj`NcGI[WQCP`\ngGP`@dfnj`NcGIjyHXiK]DMB\ngGP`@dfnjiDNcGIZyHTyK]DMB<\ngGP`@dfnjiDNcGIZyHTyK]DMB>\ngGP`@dfnjiDNcGIZypTenbFa@<\ngGP`@dfnjiDNcGIZypTenbFa@>\ngGP`@dfnjiDNcGIZypTenbF~@<\ngGP`@dfnjiDNcGIZypTencza@<\ngGP`@dfnjiDNcGIZypTencza@>\ngGP`@dfyh@NiFzN{l=\ngGP`@dfyh@NiFzN{t\ngGP`@dfyh@NiFzO[t=\ngGP`@dfyh@NiGNbdmt]wh\ngGP`@dfyh@NiGNbgMt^wh\ngGP`@dfyi`NcGIZwQ{P`\ngGP`@dfyi`NcGIZwQ{P\x7FdlRP\ngGP`@dfyi`NcGIZwQ{_@\ngGP`@dfyi`NcGIZweB\ngGP`@dfyi`NcGIZwe|\ngGP`@dfyi`NcGIZytTencva@\ngGP`@dfyi`NcGIZytTeoJD\ngGP`@dfyi`NcGIZytTyncva@\ngGP`@dfyi`NcGIZytTyncv~@\ngGP`@dfyi`NcGIZytTyoJD\ngGP`@dfyi`NcGI[WQ{P`\ngGP`@dfyi`NcGI[WeB\ngGP`@dfyi`NcGI[YtTyncva@\ngGP`@dfyi`NcGI[YtTyoJD\ngGP`@dfyi`NeGUt]wX\ngGP`@dfyi`NeGUt]wh\ngGP`@dfyi`NeGUt^wX\ngGP`@dfyi`NeGUt^wh\ngGP`@dfyi`NeGV]EN[h}oP\ngGP`@dfyj`NcGIZwQCP`\ngGP`@dfyj`NcGIZwQCP\x7FdlRP\ngGP`@dfyj`NcGIZwQ}_@\ngGP`@dfyj`NcGIkWQCP`\ngGP`@dfyjidNcGIZwQC_@<\ngGP`@dfyjidNcGIZwQC_@>\ngGP`@dfyjidNcGIZyLTyK]DM|>\ngGP`@dfyjidNcGIZytTenbF~@<\ngGP`@dfyjidNcGIZytTenbF~@>\ngGP`@dfyjidNcGIZytTynbFa@<\ngGP`@dfyjidNcGIZytTynbFa@>\ngGP`@dfyjidNcGIZytTynbF~@<\ngGP`@dfyjidNcGIZytTynbF~@>\ngGP`@dfyjidNcGI[WQCP`<\ngGP`@dfyjidNcGI[WQCP`>\ngGP`@dfyjidNcGI[YtTynbFa@>\ngGP`@dfyjidNcGIjwQC_@<\ngGP`@dfyjidNcGIjwQC_@>\ngGP`@dfzh@NiFzN{l=\ngGP`@dfzh@NiFzN{t\ngGP`@dfzh@NiGICEI[h{oP\ngGP`@dfzhIDNiGNBdmt]wh<\ngGP`@dfzhIDNiGNBdmt]wh>\ngGP`@dfzj`NcGIZwQCP`\ngGP`@dfzj`NcGIZyhXirVzHZD\ngGP`@dfzjeEdNcGIZypTynbFa@<\ngGP`@dfzjeEdNcGIZypTynbFa@>\ngGP`@dfzjeEhNcGIZypTynbFa@>\ngGP`ADkjj`NjENRuolDahP\ngGP`ADkjj`NjENRurPqSf~pRFa@\ngGP`ADkjj`NjENRus`qK_XIC_@\ngGP`ADkjj`NjENRus`qS_XICP`\ngGP`ADkjj`NjENRvolDahP\ngGP`ADkjj`NjENcVolDahP\ngGP`ADkjj`NjENcVrPqSf~pRFa@\ngGP`ATeVn`NeDh\ngGP`ATffj`NcENRwVBP`\ngGP`ATffjiDNcENRwVB_@<\ngGP`ATffjiDNcENRwVB_@>\ngGP`ATffjiDNcENRytTenlDa@<\ngGP`ATffjiDNcENRytTenlDa@>\ngGP`ATffjiDNcENSWVBP`>\ngGP`Adeij`NcEIZwQCP`\ngGP`Adijj`NjFJRuodDahP\ngGP`Adijj`NjFJRuodD~o`\ngGP`Adijj`NjFJRurPqrV~WbFa@\ngGP`Adijj`NjFJRus`qS_HICP`\ngGP`Adijj`NjFJRus`qS_KqCP`\ngGP`Adijj`NjFJSVs`qS_HICP`\ngGP`Adijj`NjFJbus`qS_HICP`\ngGP`AdijjiDNjFJRuodDao`<\ngGP`AdijjiDNjFJRuodDao`>\ngGP`AdijjiDNjFJRurPqSf~PRFa@>\ngGP`AdijjiDNjFJRurPqSf~WbFa@<\ngGP`AdijjiDNjFJRurPqSf~WbFa@>\ngGP`AdijjiDNjFJRurPqrV~PRF~@>\ngGP`AdijjiDNjFJRus`qK_HIC_@<\ngGP`AdijjiDNjFJRus`qK_HIC_@>\ngGP`AdijjiDNjFJRus`qK_KqC_@>\ngGP`AdijjiDNjFJRus`qS_HIC_@<\ngGP`AdijjiDNjFJRus`qs_HICP`<\ngGP`AdijjiDNjFJRus`qs_HICP`>\ngGP`AdijjiDNjFJRus`qs_HIC_@<\ngGP`AdijjiDNjFJRus`qs_HIC_@>\ngGP`AdijjiDNjFJRus`qs_KqCP`<\ngGP`AdijjiDNjFJRus`qs_KqCP`>\ngGP`AdijjiDNjFJRvodDahP<\ngGP`AdijjiDNjFJRvodDahP>\ngGP`AdijjiDNjFJRvs`qS_HICP`<\ngGP`AdijjiDNjFJRvs`qS_HICP`>\ngGP`AdijjiDNjFJRvs`qs_HICP`<\ngGP`AdijjiDNjFJRvs`qs_HICP`>\ngGP`AdijjiDNjFJSUs`qK_HIC_@<\ngGP`AdijjiDNjFJSUs`qK_HIC_@>\ngGP`AdijjiDNjFJbus`qK_HIC_@<\ngGP`AdijjiDNjFJbus`qs_HICP`<\ngGP`AdijjiDNjFJbus`qs_HICP`>\ngGP`AdijjiDNjFJbvs`qs_HICP`<\ngGP`AdijjiDNjFJbvs`qs_HICP`>\ngGPa@SrSLuHGQbgM\\zJRwVC^`\ngGPa@SrSMSHGQbgM\\zJRwV{^`\ngGPa@erS\\t@GTc]G]v=\ngGPa@erS\\t@GTc]G]z\ngGPa@erS\\t@GTc]Gmz=\ngGPa@erS\\tpGQcdm[ra@\ngGPa@erS\\uPGQcdm[hahP\ngGPa@erS\\uPGQcdm[hah_rVIH\ngGPa@erS\\uTrGQcdmkhahP<\ngGPa@iJS]T@GTc]G]v=\ngGPa@krS\\t@GTc]Gmz\ngGPa@mrS\\uPGRcjzO[t\ngGPdCPVZ`R[ij@ztTyoEt\ngGPdE`DRPrRRrtAu`ek^TH\ngGPdH`DTHrRPkTAu`yK]DM|\ngGPdH`DWprRPkTAu`yK]DM|\ngGPdH`VTlRjZj@zHXzM_XIC_@\ngGPdH`VUpRjZj@zHXzM_XIC_@\ngGPdH`xUpRjZj@zHXyK_XIC_@\ngGPdH`xUpRjZj@zHXyKgAbV~pRF~@\ngGPdH`xUpRjZj@zHXzM_XIC_@\ngGPdH`xUpRjZj@zHXzMgAbV~pRF~@\ngGPdH`xUpRjZj`PzHXyM_XIC_@<\ngGPdH`xUpRjZj`PzHXyM_XIC_@>\ngGPdH`xV`RjZj@zHXzM]XIB\ngGPdI`VIpRVjj@zLTfm_CqCP`\ngGPeE`DSpGIM\\uRH]FNSUnbFa@>\ngGPeH`VUDliUMU@]DL]FnlDa@\ngGPh@iPIMuP@]RMt]wX=\ngGPh@iPIMuP@]RMt]wh\ngGPh@iPIMuP@]RMt^wh=\ngGPh@iPIMuP@]RN\\EI[h{oP>\ngGPh@iPIMuS@]FNRuncva@\ngGPh@iPIMuS@]FNRuoJD\ngGPh@iPIMuS@]FNRvncva@\ngGPh@iPIMuS@]FNRvoJD\ngGPh@iPIMuS@]JNkh{oP\ngGPh@iPIMuU@]FNRunbFa@\ngGPh@iPIMuUSH]FNRvnbFa@>\ngGPhCPDILuK@]JR[h[vp\ngGPhCPDILuK@]JR[h[wP\ngGPhCPDILuK@]ZJ\\`\ngGPhCPDILuK@]ZJ\\wfz\ngGPhCPDILuSRH]ZJ\\wP}``>\ngGPhCPDILuSRH]ZJ\\wfB<\ngGPhCPDILuSRH]ZJ\\wfB>\ngGPhCQBILuK@]ZJ\\wfz\ngGPhCQDILuSRH]ZJ\\wfB>\ngGPhCQbILuSRH]ZJ\\wfB>\ngGPhCQxILsS@]ZJ\\wfB\ngGPhCQxILuK@]ZJ\\wP}n`\ngGPhCQxILuK@]ZJ\\wfz\ngGPhDPDIMuUJKH]ZNZytTeoJD<\ngGPhDPDIMuUJKH]ZNZytTeoJD>\ngGPhH`DYIHUi@zH\\ek^DH\ngGPhH`DYIHUi@zH\\ekgIrVzH[t\ngGPhH`DYIHUi@zH\\fk]DMz\ngGPhH`DYIHUi@zH\\fk^DH\ngGPhH`DYIHUj@zH\\ek]DMB\ngGPhH`DYIHUj@zH\\ekgIrVzHZD\ngGPhH`DYIHUj@zH\\em]DMB\ngGPhH`DYIHUj@zH\\fk]DMB\ngGPhH`DYIHUj@zH\\fk]DM|\ngGPhH`DYIHUj@zH\\fm]DMB\ngGPhH`xIUMU@]TL\\ek_XICP`\ngGPhH`xIUMU@]TL\\ek_[q}P`\ngGPhH`xIUMU@]TL\\ekgAbV~pRF~@\ngGPhH`xIUMU@]TL\\ekgAbV~wcza@\ngGPhH`xIUMU@]TL\\em_XICP`\ngGPhH`xIUMU@]TL]Fm_XICP`\ngGPhH`xYIHZj@zH\\ek]DMB\ngGPhHaVYICE`@z`[h{oP\ngGPhHaVYICJ`@z`[h{np=\ngGPhHaxIU[U@]TJ\\ek_XICP`\ngGPhHaxIU[U@]TJ]Fm_XICP`\ngGPhHaxIU[UPH]TJ\\fk_XICP`<\ngGPhHaxYICJ`@z`[h{oP\ngGPhI`xIKUU@]FJRuoaxahP\ngGPhaePRUrUuUPGUCEIkWtBPtH\ngGPhaePRUrUuUPGUCEIkYpXiohDahP\ngGPiAePKdkjj`NjFJSVs`qS_PICP`\ngGPiCPDWdfZe`NeIMtM{X\ngGPllQHR`TaNliM]U@]ZNZyr\\eoJD\ngGPmLQHR`TaNRZzj@zt\\usdyK^TH\ngGQ@@dmMTAuXirVm}`dMB\ngGQ@@dmMTAuXirVm}`dM|\ngGQ@@dmMTAuXirVu}`dMB\ngGQ@@dmMTAuXirZu}`dMB\ngGQ@@dmUTAuXirVm}`dMB\ngGQ@@dmUTAuXirVm}`dM|\ngGQ@@dmUTAuXirVnRFJ\\wvBPtH\ngGQ@@dmUTAuXirVn\\FJ[{AHZD\ngGQ@@dmUTAuXirVu}`dMB\ngGQ@@dsM@AthyolD{np=\ngGQ@@dsM@AthyolD{oP\ngGQ@@dsM@AthyolD}oP=\ngGQ@@dsM@Athyomx{np=\ngGQ@@dsM@Athyomx{oP\ngGQ@@dsM@AthyshiK_XIw^`\ngGQ@@dsM@AthyshiK_[qw^`\ngGQ@@dsM@Athyshis_XI{^`\ngGQ@@dsM@Athyshis_[q{^`\ngGQ@@dsM@AthzolD{np=\ngGQ@@dsM@AthzolD{oP\ngGQ@@dsM@AthzolD}oP=\ngGQ@@dsM@AthzshiK_XIw^`\ngGQ@@dsM@Athzshis_XI{^`\ngGQ@@dsMLAuXirVmu`tH\ngGQ@@dsMLAuXirVmu`wp\ngGQ@@dsMLAuXirVmuoTH\ngGQ@@dsMLAuXirVm}`gmB\ngGQ@@dsMLAuXirVm}`gm|\ngGQ@@dsMLAuXirVn]EI[kAhP\ngGQ@@dsMLAuXirVn]EI[kAo`\ngGQ@@dsMLAuXirVn]EI[k^hP\ngGQ@@dsMLAuXirVn]EI[{AOZD\ngGQ@@dsMLAuXirVn]EI[{AO[x\ngGQ@@dsMLAuXirVn]EI[{^OZD\ngGQ@@dsMLAuXirVn]EN[kAhP\ngGQ@@dsMLAuXirVn]EN[kAo`\ngGQ@@dsMLAuXirVn]EN[k^hP\ngGQ@@dsMLAuXirVn]EN[{AOZD\ngGQ@@dsMLAuXirVn]EN[{AO[x\ngGQ@@dsMLAuXirVn]EN[{^OZD\ngGQ@@dsMLAuXirVu}`gmB\ngGQ@@dsMLAuXirVv]EI[{AOZD\ngGQ@@dsMLAuXirVv]EN[kAhP\ngGQ@@dsMLAuXirVv]EN[{AOZD\ngGQ@@dsMLAuXitVmu`tH\ngGQ@@dsMLAuXitVmu`wp\ngGQ@@dsMLAuXitVm}`gmB\ngGQ@@dsMLAuXitVm}`gm|\ngGQ@@dsMLAuXitVn]EI[kAhP\ngGQ@@dsMLAuXitVn]EI[kAo`\ngGQ@@dsMLAuXitVn]EI[{AOZD\ngGQ@@dsMLAuXitVn]EI[{AO[x\ngGQ@@dsMLAuXitVn]EN[kAhP\ngGQ@@dsMLAuXitVn]EN[kAo`\ngGQ@@dsMLAuXitVn]EN[{AOZD\ngGQ@@dsMLAuXitVn]EN[{AO[x\ngGQ@@dsMLAuXitVuu`tH\ngGQ@@dsMLAuXitVu}`gmB\ngGQ@@dsMLAuXitVv]EN[kAhP\ngGQ@@dsMLAuXitVv]EN[{AOZD\ngGQ@@dsMLAuhjs_XIw]`\ngGQ@@dsMLAuhjs_XIw^`\ngGQ@@dsMLAuhjs_XI{]`\ngGQ@@dsMLAuhjs_XI{^`\ngGQ@@dsMLAuhjs_[qw^`\ngGQ@@dsMLAuhjsgQSf~pSv{@\ngGQ@@dsMLAuhjsgQSf~pSv}@\ngGQ@@dsMLAuhju_XIw^`\ngGQ@@dsMLAuhju_XI{]`\ngGQ@@dsMLAuhju_XI{^`\ngGQ@@dsMLAuhjugQSf~pSv{@\ngGQ@@dsMLAuhjugQSf~pSv}@\ngGQ@@dsMTAuXirVm}`dMB\ngGQ@@dsMTAuXirVm}`dMC~RqI@\ngGQ@@dsMTAuXirVm}oDMB\ngGQ@@dsMTAuXirVnSENRwvBPtH\ngGQ@@dsMTAuXirVnSENRwv|PtH\ngGQ@@dsMTAuXirZvSENRwvBPtH\ngGQ@@dsMTAuXitVm}`dMB\ngGQ@@dsMTAuXitVm}`dMC~RqI@\ngGQ@@dsMTAuXitVm}`gu|\ngGQ@@dsMTAuXitZu}`dMB\ngGQ@@dsMUDauXirVm}`dM|<\ngGQ@@dsMUDauXirVm}`dM|>\ngGQ@@dsMUDauXirVnSENRwvBPwp<\ngGQ@@dsMUDauXirVnSENRwvBPwp>\ngGQ@@dsMUDauXirVnSENRwv|Pwp>\ngGQ@@dsMUDauXirVn]EI[{AH[x<\ngGQ@@dsMUDauXirVn]EI[{AH[x>\ngGQ@@dsMUDauXirVn]EI[{AH[{|ebR>\ngGQ@@dsMUDauXirVn]EN[{AHZD<\ngGQ@@dsMUDauXirVn]EN[{AHZD>\ngGQ@@dsMUDauXirVn]EN[{AHZG|ebR<\ngGQ@@dsMUDauXirVn]EN[{AHZG|ebR>\ngGQ@@dsMUDauXirVn]EN[{AH[x<\ngGQ@@dsMUDauXirVn]EN[{AH[x>\ngGQ@@dsMUDauXirVn]EN[{AOkx<\ngGQ@@dsMUDauXirVn]EN[{AOkx>\ngGQ@@dsMUDauXirVn]EN[{^HZD<\ngGQ@@dsMUDauXirVn]EN[{^HZD>\ngGQ@@dsMUDauXirVv]EN[{AHZD<\ngGQ@@dsMUDauXirVv]EN[{AHZD>\ngGQ@@dsMUDauXitVm}`dM|<\ngGQ@@dsMUDauXitVm}`dM|>\ngGQ@@dsMUDauXitVn]EI[{AH[x<\ngGQ@@dsMUDauXitVn]EN[{AHZD<\ngGQ@@dsMUDauXitVn]EN[{AHZD>\ngGQ@@dsMUDauXitVu}`dMB<\ngGQ@@dsMUDauXitVu}`dMB>\ngGQ@@dsT`Athyncn{@=\ngGQ@@dsT`Athyncn}@\ngGQ@@dsT`Athyncv}@=\ngGQ@@dsT`Athyomt{np=\ngGQ@@dsT`Athyomt{oP\ngGQ@@dsT`Athys`qK]G]z\ngGQ@@dsT`Athys`qK_[iw^`\ngGQ@@dsT`AthyshiK]G]z\ngGQ@@dsT`AthyshiK_[iw^`\ngGQ@@dsT`Athyshis_[i{^`\ngGQ@@dsT`AuIWHwmg]v=\ngGQ@@dsT`AuIWHwmg]z\ngGQ@@dsT`AuIWHwmgmz=\ngGQ@@dsT`AuIWHwng]v=\ngGQ@@dsT`AuIWHwng]z\ngGQ@@dsT`AuIYtTenQo]N{t\ngGQ@@dsTlAtYIKV~F}na@\ngGQ@@dsTlAtYIKZyF}l}hP\ngGQ@@dsTlAuXirVnZFJ\\encva@\ngGQ@@dsTlAuXirVn]EN[{]OZD\ngGQ@@dsUAHathyolD{np+\ngGQ@@dsUAHathyolD{np-\ngGQ@@dsUAHathyolD{oP<\ngGQ@@dsUAHathyolD{oP>\ngGQ@@dsUAHathyolD}oP+\ngGQ@@dsUAHathyolD}oP-\ngGQ@@dsUAHathyomx{np+\ngGQ@@dsUAHathyomx{np-\ngGQ@@dsUAHathyomx{oP<\ngGQ@@dsUAHathyomx{oP>\ngGQ@@dsUAHathyrPqRV~pSn}@<\ngGQ@@dsUAHathyrPqRV~pSn}@>\ngGQ@@dsUAHathys`qK_XIw^`<\ngGQ@@dsUAHathys`qK_XIw^`>\ngGQ@@dsUAHathyshiK_XIw^`<\ngGQ@@dsUAHathyshiK_XIw^`>\ngGQ@@dsUAHathyshiK_[qw^`<\ngGQ@@dsUAHathyshiK_[qw^`>\ngGQ@@dsUAHathyshis_XI{^`<\ngGQ@@dsUAHathyshis_XI{^`>\ngGQ@@dsUAHathyshis_[q{^`<\ngGQ@@dsUAHathyshis_[q{^`>\ngGQ@@dsUAHathzolD{np+\ngGQ@@dsUAHathzolD{np-\ngGQ@@dsUAHathzolD{oP<\ngGQ@@dsUAHathzolD{oP>\ngGQ@@dsUAHathzolD}oP-\ngGQ@@dsUAHathzs`qK_XIw^`<\ngGQ@@dsUAHathzs`qK_XIw^`>\ngGQ@@dsUMHauXirVmu`tH<\ngGQ@@dsUMHauXirVmu`tH>\ngGQ@@dsUMHauXirVm}`gmB<\ngGQ@@dsUMHauXirVm}`gmB>\ngGQ@@dsUMHauXirVm}`gm|>\ngGQ@@dsUMHauXirVnRFJ\\wVCP`<\ngGQ@@dsUMHauXirVnZFJ\\enlFa@>\ngGQ@@dsUMHauXirVn\\FN[{AOZD<\ngGQ@@dsUMHauXirVn\\FN[{AOZD>\ngGQ@@dsUMHauXirVn]EI[kAhP<\ngGQ@@dsUMHauXirVn]EI[kAhP>\ngGQ@@dsUMHauXirVn]EI[kAo`<\ngGQ@@dsUMHauXirVn]EI[kAo`>\ngGQ@@dsUMHauXirVn]EI[k^hP>\ngGQ@@dsUMHauXirVn]EI[{AO[x>\ngGQ@@dsUMHauXirVn]EI[{^OZD<\ngGQ@@dsUMHauXirVn]EN[kAhP<\ngGQ@@dsUMHauXirVn]EN[kAhP>\ngGQ@@dsUMHauXirVn]EN[{AOZD<\ngGQ@@dsUMHauXirVn]EN[{AOZD>\ngGQ@@dsUMHauXirVuu`tH<\ngGQ@@dsUMHauXitVmu`tH<\ngGQ@@dsUMHauXitVmu`tH>\ngGQ@@dsUMHauhjs_XIw]`<\ngGQ@@dsUMHauhjs_XIw]`>\ngGQ@@dsUMHauhjs_[qw]`<\ngGQ@@dsUMHauhjs_[qw]`>\ngGQ@@dsUThdauXirVm}`dM|<\ngGQ@@dsUThdauXirVm}`dM|>\ngGQ@@dsUThdauXirVnRFJRwvBPwp<\ngGQ@@dsUThdauXirVnRFJ\\wvBPtH>\ngGQ@@dsUThdauXirVnSENRwvBPwp>\ngGQ@@dsUThdauXirVn\\FN[{AHZD<\ngGQ@@dsUThdauXirVn\\FN[{AHZD>\ngGQ@@dsUThdauXirVn]EI[{AH[x<\ngGQ@@dsUThdauXirVn]EI[{AH[x>\ngGQ@@dsUThdauXirVn]EN[{AHZD<\ngGQ@@dsUThdauXirVn]EN[{AHZD>\ngGQ@@dsUThdauXirVn]EN[{AH[x<\ngGQ@@dsUThdauXirVn]EN[{AH[x>\ngGQ@@dsUThdauXirVn]EN[{AOkx<\ngGQ@@dsUThdauXirVn]EN[{AOkx>\ngGQ@@dsUThdauXirVn]EN[{^HZD<\ngGQ@@dsUThdauXirVn]EN[{^HZD>\ngGQ@@dsUThdauXirVn]EN[{^Okx<\ngGQ@@dsUThdauXirVu}`dMB<\ngGQ@@dsUThdauXirVv]EN[{AHZD<\ngGQ@@dsUThdauXirVv]EN[{AHZD>\ngGQ@@dsUThdauXitVn]EN[{AHZD<\ngGQ@@dsUThdauXitVn]EN[{AHZD>\ngGQ@@dsUTheAuXirVm}`dM|<\ngGQ@@dsUTheAuXirVm}`dM|>\ngGQ@@dsUTheAuXirVm}oDM|<\ngGQ@@dsUTheAuXirVnRFJRwvBPwp>\ngGQ@@dsUTheAuXirVnRFJ\\wvBPtH<\ngGQ@@dsUTheAuXirVnRFJ\\wvBPtH>\ngGQ@@dsUTheAuXirVn\\FN[{AHZD<\ngGQ@@dsUTheAuXirVn\\FN[{AHZD>\ngGQ@@dsUTheAuXirVn]EI[{AH[x<\ngGQ@@dsUTheAuXirVn]EI[{AH[x>\ngGQ@@dsUTheAuXirVn]EI[{^H[x>\ngGQ@@dsUTheAuXirVn]EN[{AHZD<\ngGQ@@dsUTheAuXirVn]EN[{AHZD>\ngGQ@@dsUTheAuXirVn]EN[{AH[x<\ngGQ@@dsUTheAuXirVn]EN[{AH[x>\ngGQ@@dsUTheAuXirVn]EN[{AOkx<\ngGQ@@dsUTheAuXirVn]EN[{AOkx>\ngGQ@@dsUTheAuXirVu}`dMB<\ngGQ@@dsUTheAuXirVu}`dMB>\ngGQ@@dsUTheAuXirVv]EN[{AHZD<\ngGQ@@dsUTheAuXitVm}`dM|>\ngGQ@@dsUTiDauXirVn]EN[{^HZD<\ngGQ@@dsUTiDauXirVn]EN[{^HZD>\ngGQ@@dsUTiDauXitVn\\FN[{AHZD<\ngGQ@@dsUTiDauXitVn\\FN[{AHZD>\ngGQ@@dsUTiDauXitVn]EN[{AHZD>\ngGQ@@dsUUHauXirVm}`dMB<\ngGQ@@dsUUHauXirVm}`dMB>\ngGQ@@dsUUHauXirVm}`dMC~RqI@<\ngGQ@@dsUUHauXirVm}`gu|>\ngGQ@@dsUUHauXirVm}oDMB<\ngGQ@@dsUUHauXirVm}oDMB>\ngGQ@@dsUUHauXirVm}oGu|<\ngGQ@@dsUUHauXirVnSENRwvBPtH<\ngGQ@@dsUUHauXirVnSENRwvBPtH>\ngGQ@@dsUUHauXirVnSENRwv|PtH<\ngGQ@@dsUUHauXirVnZFJ\\eolDahP<\ngGQ@@dsUUHauXirVnZFJ\\eolDahP>\ngGQ@@dsUUHauXirVnZFJ\\eomxahP<\ngGQ@@dsUUHauXitVm}`dMB<\ngGQ@@dsUUHauXitVm}`dMB>\ngGQ@@dtuRAuXirVmu`dH\ngGQ@@dtuRAuXirVnSENRwVBP`\ngGQ@@dtuRAuXirVn\\FI[kAHP\ngGQ@@dtuRAuXirVn\\FI[{AH[t\ngGQ@@dtuRAuXitZn\\FI[kAHP\ngGQ@@dtuSHauXirVmu`gp>\ngGQ@@dtuSHauXirVnRFJRwVBP`<\ngGQ@@dtuSHauXirVnRFNRwVB_@<\ngGQ@@dtuSHauXirVnSENRwVB_@>\ngGQ@@dtuSHauXirVn]EI[kAHP<\ngGQ@@dtuSHauXirVn]EI[kAHP>\ngGQ@@dtuSHauXirVn]EI[{AH[t<\ngGQ@@dtuSHauXirVn]EI[{AH[t>\ngGQ@@dtuSHauXirVn]GI[kAO`<\ngGQ@@dtuSHauXirZn]EI[kAHP<\ngGQ@@dtuSHauXirZn]EI[kAHP>\ngGQ@@dtuTAuXirVm}`dMB\ngGQ@@dtuTAuXirVm}oGuB\ngGQ@@dtuTAuXirVnSENRwvBPtH\ngGQ@@dtuTAuXirVnSENRwvBPwp\ngGQ@@dtuTAuXirVn\\FI[{AHZD\ngGQ@@dtuTAuXirVn\\FI[{AH[x\ngGQ@@dtuTAuXirVu}`dMB\ngGQ@@dtuTAuXitZm}`dMB\ngGQ@@dtuUHauXirVm}`guB<\ngGQ@@dtuUHauXirVm}`guB>\ngGQ@@dtuUHauXirVnRFJRwvBPtH<\ngGQ@@dtuUHauXirVnRFJRwvBPtH>\ngGQ@@dtuUHauXirVnRFNRwvB_TH<\ngGQ@@dtuUHauXirVnRFNRwvB_TH>\ngGQ@@dtuUHauXirVnSENRwvB_TH<\ngGQ@@dtuUHauXirVn]EI[{AHZD<\ngGQ@@dtuUHauXirVn]EI[{AHZD>\ngGQ@@dtuUHauXirVn]EI[{AHZG|dcb>\ngGQ@@dtuUHauXirVn]EI[{AH[x<\ngGQ@@dtuUHauXirVn]EI[{AH[x>\ngGQ@@dtuUHauXirVn]EI[{AOjD<\ngGQ@@dtuUHauXirVn]EI[{AOjD>\ngGQ@@dtuUHauXirVn]EI[{^OjD<\ngGQ@@dtuUHauXirVn]EI[{^OjD>\ngGQ@@dtuUHauXirVn]GI[{AOjD<\ngGQ@@dtuUHauXirVn]GI[{AOjD>\ngGQ@@dtuUHauXirVn]GI[{AOkx<\ngGQ@@dtuUHauXirVn]GI[{AOkx>\ngGQ@@dtuUHauXirVv]EI[{AHZD<\ngGQ@@dtuUHauXirZm}`dMB<\ngGQ@@dtuUHauXirZm}`dMB>\ngGQ@@dtuUHauXirZm}`dM|<\ngGQ@@dtuUHauXirZn\\FI[{AHZD<\ngGQ@@dtuUHauXirZn\\FI[{AHZD>\ngGQ@@dtuUHauXirZn]EI[{AHZD<\ngGQ@@dtuUHauXirZn]EI[{AHZD>\ngGQ@@dtuUHauXirZu}`dMB<\ngGQ@@dtuUHauXirZu}`dMB>\ngGQ@@dtuUHauXitVm}`guB<\ngGQ@@dtuUHauXitVm}`guB>\ngGQ@@dtuUHauXitVn]GI[{AOjD>\ngGQ@@dwSTAtYYKVyFulahP\ngGQ@@dwSTAuXirVmtPtH\ngGQ@@dwSTAuXirVm|ndMB\ngGQ@@dwSULauXirVn]EN[hahP>\ngGQ@@dwSULauXirVn]EN[y]HZD<\ngGQ@@eMUTAtpqSdm[eApRFa@\ngGQ@@eMUTAtpqSdm[eApRFa\x7FIHx`\ngGQ@@eMUTAtpqSdm[eApRF~@\ngGQ@@eMUTAtpqSdm[eApSz~@\ngGQ@@eMUTAtpqSdm\\dLTynTGAHZD\ngGQ@@eMUTAtpqSdm\\dLTynU{AHZD\ngGQ@@eMUTAtpqSdm\\xLRwJC`dM|\ngGQ@@eMUTAtpqSdm\\xLRwJ}`dM|\ngGQ@@eMUTAtpqSdm\\xLTwJC`dMB\ngGQ@@eMUTAtpqSdm\\xLTwJ}`dMB\ngGQ@@eMUTAtpqSdmkeApRFa@\ngGQ@@eMUTAtpqSdmkeApRFa\x7FIHx`\ngGQ@@eMUTAtpqSdukeApRFa@\ngGQ@@eMUTAtpqShukeApRFa@\ngGQ@@eMUTAtpqShulxLTwJ}`dMB\ngGQ@@eMUTAtpqUdm[eApRFa@\ngGQ@@eMUTAtpqUdm\\xLRwJC`dM|\ngGQ@@eMUTAtpqUdm\\xLTwJC`dMB\ngGQ@@eMUTAtpqUdulxLTwJC`dMB\ngGQ@@eMUUHatpqSdm\\dLTenU{AH[x>\ngGQ@@eMUUHatpqSdm\\xLTwJC`dM|<\ngGQ@@eMUUHatpqSdm\\xLTwJC`dM|>\ngGQ@@eMUUHatpqSdmlxLTwJC`dMB<\ngGQ@@eMUUHatpqSdmlxLTwJC`dMB>\ngGQ@@eMUUHatpqSdu[eApRF~@<\ngGQ@@eMUUHatpqSdu[eApRF~@>\ngGQ@@eMUUHatpqUdmlxLTwJC`dMB>\ngGQA@NRYjP@zT\\wQw]`=\ngGQA@NRYjP@zT\\wvz]wX=\ngGQA@NRYjP@zT\\ytTeomt{oP\ngGQAAIRYj`@zT\\wvB]wX+\ngGQAAIRYjf@zlTyK[Nbdm}`gmB>\ngGQBBGBNB\\LxYIIZj@zpRunYy^wjD\ngGQDHGBnBUSUPGUCENRwJC`dM|\ngGQDHGBnBUSUPGUCENRwJC`gu|\ngGQDHGBnBUSUPGUCENRypXenTGAH[x\ngGQDHGBnBUSUPGUCENSWJC`dM|\ngGQDHGBnBUSUPGUCEVcWJC`dM|\ngGQDJ@aABSWUPGVcdmxP`\ngGQDJIARBSWUPGVcdmx_@\ngGQDJIARBSWUPGVcdn\\gI[p~@\ngGQDLOA^FQRFj`NmDm[x^WjD\ngGQHAJAIfj@CiQs_XIw]`+\ngGQHAJAIfj@CiQs_XI{^`+\ngGQHAJAIfj@CiQu_XIw^`>\ngGQHAJAIfj@CiQu_XI{^`+\ngGQHAJAIfj@CiQugQRV~pSn}@>\ngGQHAJAIfj@CiQugQSf~pSv}@>\ngGQHAJAIfjXCjqShmk{AOZD>\ngGQHAJAIfjhCjqSdm[{AHZD>\ngGQHAjAInfhChrrZurMkYCP`\ngGQHB@aIfiXChqSfn]EI[k]hP\ngGQHB@aIfiXChqSfuyP`\ngGQHB@aIfiXChqSfv]EI[k]hP\ngGQHB@aIfiXCkRSV~F}na@\ngGQHB@aIfiXCkRSV~F}va@\ngGQHB@aIfiXCkRSWNbdm|M{mB\ngGQHB@aIfiXCkRSZ~F}na@\ngGQHB@aIfiXCkRSZ~F}va@\ngGQHB@aIfiXCkRS[Nbdm|M{mB\ngGQHB@aIfidCiRU\\c^v]wX\ngGQHB@aIfjiQJChqSjn\\FI[kAhP>\ngGQHB@aInfjZCkRsV~Fuva@<\ngGQHBGAIfiXChqSfn]EI[h^hP\ngGQHBHaIfiXChqSfn]EI[k]hP\ngGQHBHaIfiXChqSfn]EI[k]o`\ngGQHBHaIfiXChqSfn]EI[ra@\ngGQHBHaIfiXChqSfv]EI[k]hP\ngGQHBHaIfjiQIChqSfmu`tH<\ngGQHBHaIfjiQJChqSfmu`tH>\ngGQHBHaIfjiQJChqSfn]EI[kAhP<\ngGQHBHaIfjiQJChqSfn]EI[kAhP>\ngGQHBIAIfiXChqSfn]EI[k]hP\ngGQHBIAIfjiQIChqSfnSENRwVCP`>\ngGQHBIAIfjiQIChqSfn]EI[kAhP>\ngGQHBIAIfjiQJChqSfn]EI[kAhP<\ngGQHBIAIfjiQJChqSfn]EI[kAhP>\ngGQHBJqIfiXChqSfn]EI[k]hP\ngGQHBJqIfiXChqSfn]EI[ra@\ngGQHBJqIfjiQJChqSfn]EI[kAhP<\ngGQHBLQIfiXChqSfn]EI[k]hP\ngGQHBLQIfiXChqSfn]EI[ra@\ngGQHBLQIfiXChqSfv]EI[k]hP\ngGQHBLQIfjZQChqSfuu`tH<\ngGQHBLQIfjZQChqSfuu`tH>\ngGQHBLQIfjiQIChqSfmu`tH>\ngGQHBLQIfjiQJChqSfmu`tH>\ngGQHBLQInfjZChqSfn]EI[ra@>\ngGQHD@aInZ@CjQshiK]G]z\ngGQHD@aInf@CjQncn{@=\ngGQHD@aInfhChqrVmtPtH\ngGQHDHaInf@CjQncn{@=\ngGQHDOAIijjQCjqSdm\\dLTenQ{AH[x<\ngGQHDOAIijjQCjqSdm\\dLTeolDao`<\ngGQHDOAIkjjQChqrVnRFJRwQC_@<\ngGQHDOAIkjjQChqrVnRFJRwp|Pwp<\ngGQHDOAInf@CjQncn{@=\ngGQHDOAInjBQCjQoax{np+\ngGQHDOAInjBQCjQoax{np-\ngGQHH@aJnZBACiAs_XIw]`+\ngGQHH@aJnZBACiAs_XIw]`-\ngGQHH@aJnZBACiAs_XIw^`>\ngGQHH@aJnZBACiAs_[qw]`-\ngGQHH@aJnZBACiAu_XIw]`+\ngGQHH@aJnZBACiAu_XIw]`-\ngGQHH@aJnZBACiAu_XIw^`<\ngGQHH@aJnZBACiAu_XIw^`>\ngGQHH@aJnZBACiAugQRV~pSn}@<\ngGQHH@aJnZBACiAugQRV~pSn}@>\ngGQHH@aJnZZACkAUf~pSv}@<\ngGQHH@aJnZZACkAUf~pSv}@>\ngGQHH@aJnZZACkAUj~pSv}@<\ngGQHH@aJnZZACkAUj~pSv}@>\ngGQHH@aJnZjACjaShm[{AHZD>\ngGQHHDQJnZ@CiAu_XIw^`>\ngGQHHGAIZjhCjqSdm[d^pRFa@\ngGQHHGAIZjhCjqSdm[d^pRF~@\ngGQHHGAIZjhCjqSdm\\xLTwH}`dMB\ngGQHHGAIZjhCjqSdmkd^pRFa@\ngGQHHGA