/*
 * array-of-objects-utilities v1.0.0
 * Utilities to reduce depths of objects and convert to text
 * https://github.com/cheminfo/array-of-objects-utilities#readme
 *
 * Licensed under the MIT license.
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.ArrayOfObjectsUtilities = {}));
})(this, (function (exports) { 'use strict';

    // eslint-disable-next-line @typescript-eslint/unbound-method
    const toString = Object.prototype.toString;
    /**
     * Checks if an object is an instance of an Array (array or typed array, except those that contain bigint values).
     *
     * @param value - Object to check.
     * @returns True if the object is an array or a typed array.
     */
    function isAnyArray(value) {
      const tag = toString.call(value);
      return tag.endsWith('Array]') && !tag.includes('Big');
    }

    /**
     * Recursively extracts all JSON paths with their corresponding primitive values from an object.
     * @param obj - The input object or value to traverse.
     * @param options - Optional parameters to control traversal and filtering.
     * @returns an array of objects each containing a jpath and its corresponding primitive value.
     */
    function getJPaths(obj, options = {}) {
      const {
        maxArrayElements = 5,
        maxDepth = 3,
        includeJPaths = [],
        excludeJPaths = [],
        includeJPathRegexps = [],
        excludeJPathRegexps = [],
        modifiers = {}
      } = options;
      // convert modifiers to a Map for faster access
      const modifiersMap = new Map(Object.entries(modifiers));
      function matchesInclude(path) {
        if (includeJPaths.length === 0 && includeJPathRegexps.length === 0) {
          return true;
        }
        if (includeJPaths.length > 0 && includeJPaths.includes(path)) return true;
        return includeJPathRegexps.some(re => re.test(path));
      }
      function isExcluded(path) {
        if (excludeJPaths.length === 0 && excludeJPathRegexps.length === 0) {
          return false;
        }
        if (excludeJPaths.length > 0 && excludeJPaths.includes(path)) return true;
        return excludeJPathRegexps.some(re => re.test(path));
      }
      function traverse(value, prefix = '', depth = 0) {
        // Stop if depth exceeds maximum
        if (depth > maxDepth) {
          return [];
        }
        const modifier = modifiersMap.get(prefix);
        if (modifier) {
          value = modifier(value);
        }
        let result = [];
        if (isAnyArray(value)) {
          const limitedItems = value.slice(0, maxArrayElements);
          for (const [idx, item] of limitedItems.entries()) {
            const newPrefix = prefix ? `${prefix}.${idx}` : `${idx}`;
            result = result.concat(traverse(item, newPrefix, depth + 1));
          }
        } else if (typeof value === 'object' && value !== null) {
          for (const key of Object.keys(value)) {
            const child = value[key];
            const newPrefix = prefix ? `${prefix}.${key}` : key;
            result = result.concat(traverse(child, newPrefix, depth + 1));
          }
        } else if ((typeof value === 'string' || typeof value === 'number' || typeof value === 'boolean') && matchesInclude(prefix) && !isExcluded(prefix)) {
          result.push({
            jpath: prefix,
            value
          });
        }
        return result;
      }
      return traverse(obj);
    }

    /**
     * Recursively extracts all JSON paths with their corresponding primitive values from an object.
     * @param obj - The input object or value to traverse.
     * @param options - Optional parameters to control traversal and filtering.
     * @returns an object mapping jpaths to their corresponding primitive values.
     */
    function getJPathsAsObject(obj, options = {}) {
      const jpaths = getJPaths(obj, options);
      const result = {};
      for (const entry of jpaths) {
        result[entry.jpath] = entry.value;
      }
      return result;
    }

    /**
     * When given an array of objects, it flattens each object to its JSON paths representation.
     * For example if you have an array of objects like:
     * [{ a: { b: 'ab', c: 'cd' }, d: 4 }, { e: 5 }]
     * It will return:
     * [{ 'a.b': 'ab', 'a.c': 'cd', 'd': 4 }, { 'e': 5 }]
     * Numerous options allows to select jPaths and modify values.
     * @param data - Array of objects to flatten
     * @param options - Options to customize the flattening process based on the get-jpaths library
     * @returns Array of flattened objects
     */
    function flattenToJPaths(data, options = {}) {
      const results = [];
      for (const item of data) {
        results.push(getJPathsAsObject(item, options));
      }
      return results;
    }

    /**
     * When given an array of objects, it returns the distinct JSON paths found in the objects.
     * For example if you have an array of objects like:
     * [{ a: { b: 'ab', c: 'cd' }, d: 4 }, { e: 5 }]
     * It will return:
     * ['a.b', 'a.c', 'd', 'e']
     * @param data - array of objects
     * @param options - options to get jpaths
     * @returns Array of distinct JSON paths
     */
    function getDistinctJPaths(data, options = {}) {
      const jPathSet = new Set();
      for (const item of data) {
        const jpaths = getJPaths(item, options);
        for (const jpath of jpaths) {
          jPathSet.add(jpath.jpath);
        }
      }
      return Array.from(jPathSet);
    }

    /**
     * Converts an array of objects into a text representation (e.g., CSV or TSV).
     * @param data - Array of objects
     * @param options - Options for text conversion
     * @returns Text representation of the object array
     */
    function objectArrayToText(data, options = {}) {
      const {
        delimiter = '\t',
        eol = '\n',
        header = true,
        headerMapping = {}
      } = options;
      const allKeys = new Set();
      for (const item of data) {
        for (const key in item) {
          allKeys.add(key);
        }
      }
      const keys = Array.from(allKeys);
      let text = '';
      if (header) {
        const headerRow = keys.map(key => headerMapping[key] || key).join(delimiter);
        text += headerRow + eol;
      }
      for (const item of data) {
        const row = keys.map(key => {
          const value = item[key];
          return value !== undefined ? String(value) : '';
        }).join(delimiter);
        text += row + eol;
      }
      return text;
    }

    exports.flattenToJPaths = flattenToJPaths;
    exports.getDistinctJPaths = getDistinctJPaths;
    exports.objectArrayToText = objectArrayToText;

}));
//# sourceMappingURL=array-of-objects-utilities.umd.js.map
