/**
 * ml-integral-transforms - Line broadening through integral transforms
 * @version v0.0.1
 * @link https://github.com/mljs/ml-integral-transforms#readme
 * @license MIT
 */
(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
  typeof define === 'function' && define.amd ? define(['exports'], factory) :
  (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.IntegralTransform = {}));
})(this, (function (exports) { 'use strict';

  function checkSize(size) {
    if (!Number.isInteger(size) || size < 1) {
      throw new TypeError(`size must be a positive integer. Got ${size}`);
    }
  }
  function checkKernel(kernel) {
    if (kernel.length === 0 || kernel.length % 2 !== 1) {
      throw new RangeError(`kernel must have an odd positive length. Got ${kernel.length}`);
    }
  }
  function checkBorderType(borderType) {
    if (borderType !== 'CONSTANT' && borderType !== 'CUT') {
      throw new RangeError(`unexpected border type: ${borderType}`);
    }
  }
  function checkInputLength(actual, expected) {
    if (actual !== expected) {
      throw new RangeError(`input length (${actual}) does not match setup size (${expected})`);
    }
  }
  function createArray(len) {
    const array = [];
    for (var i = 0; i < len; i++) {
      array.push(0);
    }
    return array;
  }

  function FFT(size) {
    this.size = size | 0;
    if (this.size <= 1 || (this.size & this.size - 1) !== 0) throw new Error('FFT size must be a power of two and bigger than 1');
    this._csize = size << 1;

    // NOTE: Use of `var` is intentional for old V8 versions
    var table = new Array(this.size * 2);
    for (var i = 0; i < table.length; i += 2) {
      const angle = Math.PI * i / this.size;
      table[i] = Math.cos(angle);
      table[i + 1] = -Math.sin(angle);
    }
    this.table = table;

    // Find size's power of two
    var power = 0;
    for (var t = 1; this.size > t; t <<= 1) power++;

    // Calculate initial step's width:
    //   * If we are full radix-4 - it is 2x smaller to give inital len=8
    //   * Otherwise it is the same as `power` to give len=4
    this._width = power % 2 === 0 ? power - 1 : power;

    // Pre-compute bit-reversal patterns
    this._bitrev = new Array(1 << this._width);
    for (var j = 0; j < this._bitrev.length; j++) {
      this._bitrev[j] = 0;
      for (var shift = 0; shift < this._width; shift += 2) {
        var revShift = this._width - shift - 2;
        this._bitrev[j] |= (j >>> shift & 3) << revShift;
      }
    }
    this._out = null;
    this._data = null;
    this._inv = 0;
  }
  var fft = FFT;
  FFT.prototype.fromComplexArray = function fromComplexArray(complex, storage) {
    var res = storage || new Array(complex.length >>> 1);
    for (var i = 0; i < complex.length; i += 2) res[i >>> 1] = complex[i];
    return res;
  };
  FFT.prototype.createComplexArray = function createComplexArray() {
    const res = new Array(this._csize);
    for (var i = 0; i < res.length; i++) res[i] = 0;
    return res;
  };
  FFT.prototype.toComplexArray = function toComplexArray(input, storage) {
    var res = storage || this.createComplexArray();
    for (var i = 0; i < res.length; i += 2) {
      res[i] = input[i >>> 1];
      res[i + 1] = 0;
    }
    return res;
  };
  FFT.prototype.completeSpectrum = function completeSpectrum(spectrum) {
    var size = this._csize;
    var half = size >>> 1;
    for (var i = 2; i < half; i += 2) {
      spectrum[size - i] = spectrum[i];
      spectrum[size - i + 1] = -spectrum[i + 1];
    }
  };
  FFT.prototype.transform = function transform(out, data) {
    if (out === data) throw new Error('Input and output buffers must be different');
    this._out = out;
    this._data = data;
    this._inv = 0;
    this._transform4();
    this._out = null;
    this._data = null;
  };
  FFT.prototype.realTransform = function realTransform(out, data) {
    if (out === data) throw new Error('Input and output buffers must be different');
    this._out = out;
    this._data = data;
    this._inv = 0;
    this._realTransform4();
    this._out = null;
    this._data = null;
  };
  FFT.prototype.inverseTransform = function inverseTransform(out, data) {
    if (out === data) throw new Error('Input and output buffers must be different');
    this._out = out;
    this._data = data;
    this._inv = 1;
    this._transform4();
    for (var i = 0; i < out.length; i++) out[i] /= this.size;
    this._out = null;
    this._data = null;
  };

  // radix-4 implementation
  //
  // NOTE: Uses of `var` are intentional for older V8 version that do not
  // support both `let compound assignments` and `const phi`
  FFT.prototype._transform4 = function _transform4() {
    var out = this._out;
    var size = this._csize;

    // Initial step (permute and transform)
    var width = this._width;
    var step = 1 << width;
    var len = size / step << 1;
    var outOff;
    var t;
    var bitrev = this._bitrev;
    if (len === 4) {
      for (outOff = 0, t = 0; outOff < size; outOff += len, t++) {
        const off = bitrev[t];
        this._singleTransform2(outOff, off, step);
      }
    } else {
      // len === 8
      for (outOff = 0, t = 0; outOff < size; outOff += len, t++) {
        const off = bitrev[t];
        this._singleTransform4(outOff, off, step);
      }
    }

    // Loop through steps in decreasing order
    var inv = this._inv ? -1 : 1;
    var table = this.table;
    for (step >>= 2; step >= 2; step >>= 2) {
      len = size / step << 1;
      var quarterLen = len >>> 2;

      // Loop through offsets in the data
      for (outOff = 0; outOff < size; outOff += len) {
        // Full case
        var limit = outOff + quarterLen;
        for (var i = outOff, k = 0; i < limit; i += 2, k += step) {
          const A = i;
          const B = A + quarterLen;
          const C = B + quarterLen;
          const D = C + quarterLen;

          // Original values
          const Ar = out[A];
          const Ai = out[A + 1];
          const Br = out[B];
          const Bi = out[B + 1];
          const Cr = out[C];
          const Ci = out[C + 1];
          const Dr = out[D];
          const Di = out[D + 1];

          // Middle values
          const MAr = Ar;
          const MAi = Ai;
          const tableBr = table[k];
          const tableBi = inv * table[k + 1];
          const MBr = Br * tableBr - Bi * tableBi;
          const MBi = Br * tableBi + Bi * tableBr;
          const tableCr = table[2 * k];
          const tableCi = inv * table[2 * k + 1];
          const MCr = Cr * tableCr - Ci * tableCi;
          const MCi = Cr * tableCi + Ci * tableCr;
          const tableDr = table[3 * k];
          const tableDi = inv * table[3 * k + 1];
          const MDr = Dr * tableDr - Di * tableDi;
          const MDi = Dr * tableDi + Di * tableDr;

          // Pre-Final values
          const T0r = MAr + MCr;
          const T0i = MAi + MCi;
          const T1r = MAr - MCr;
          const T1i = MAi - MCi;
          const T2r = MBr + MDr;
          const T2i = MBi + MDi;
          const T3r = inv * (MBr - MDr);
          const T3i = inv * (MBi - MDi);

          // Final values
          const FAr = T0r + T2r;
          const FAi = T0i + T2i;
          const FCr = T0r - T2r;
          const FCi = T0i - T2i;
          const FBr = T1r + T3i;
          const FBi = T1i - T3r;
          const FDr = T1r - T3i;
          const FDi = T1i + T3r;
          out[A] = FAr;
          out[A + 1] = FAi;
          out[B] = FBr;
          out[B + 1] = FBi;
          out[C] = FCr;
          out[C + 1] = FCi;
          out[D] = FDr;
          out[D + 1] = FDi;
        }
      }
    }
  };

  // radix-2 implementation
  //
  // NOTE: Only called for len=4
  FFT.prototype._singleTransform2 = function _singleTransform2(outOff, off, step) {
    const out = this._out;
    const data = this._data;
    const evenR = data[off];
    const evenI = data[off + 1];
    const oddR = data[off + step];
    const oddI = data[off + step + 1];
    const leftR = evenR + oddR;
    const leftI = evenI + oddI;
    const rightR = evenR - oddR;
    const rightI = evenI - oddI;
    out[outOff] = leftR;
    out[outOff + 1] = leftI;
    out[outOff + 2] = rightR;
    out[outOff + 3] = rightI;
  };

  // radix-4
  //
  // NOTE: Only called for len=8
  FFT.prototype._singleTransform4 = function _singleTransform4(outOff, off, step) {
    const out = this._out;
    const data = this._data;
    const inv = this._inv ? -1 : 1;
    const step2 = step * 2;
    const step3 = step * 3;

    // Original values
    const Ar = data[off];
    const Ai = data[off + 1];
    const Br = data[off + step];
    const Bi = data[off + step + 1];
    const Cr = data[off + step2];
    const Ci = data[off + step2 + 1];
    const Dr = data[off + step3];
    const Di = data[off + step3 + 1];

    // Pre-Final values
    const T0r = Ar + Cr;
    const T0i = Ai + Ci;
    const T1r = Ar - Cr;
    const T1i = Ai - Ci;
    const T2r = Br + Dr;
    const T2i = Bi + Di;
    const T3r = inv * (Br - Dr);
    const T3i = inv * (Bi - Di);

    // Final values
    const FAr = T0r + T2r;
    const FAi = T0i + T2i;
    const FBr = T1r + T3i;
    const FBi = T1i - T3r;
    const FCr = T0r - T2r;
    const FCi = T0i - T2i;
    const FDr = T1r - T3i;
    const FDi = T1i + T3r;
    out[outOff] = FAr;
    out[outOff + 1] = FAi;
    out[outOff + 2] = FBr;
    out[outOff + 3] = FBi;
    out[outOff + 4] = FCr;
    out[outOff + 5] = FCi;
    out[outOff + 6] = FDr;
    out[outOff + 7] = FDi;
  };

  // Real input radix-4 implementation
  FFT.prototype._realTransform4 = function _realTransform4() {
    var out = this._out;
    var size = this._csize;

    // Initial step (permute and transform)
    var width = this._width;
    var step = 1 << width;
    var len = size / step << 1;
    var outOff;
    var t;
    var bitrev = this._bitrev;
    if (len === 4) {
      for (outOff = 0, t = 0; outOff < size; outOff += len, t++) {
        const off = bitrev[t];
        this._singleRealTransform2(outOff, off >>> 1, step >>> 1);
      }
    } else {
      // len === 8
      for (outOff = 0, t = 0; outOff < size; outOff += len, t++) {
        const off = bitrev[t];
        this._singleRealTransform4(outOff, off >>> 1, step >>> 1);
      }
    }

    // Loop through steps in decreasing order
    var inv = this._inv ? -1 : 1;
    var table = this.table;
    for (step >>= 2; step >= 2; step >>= 2) {
      len = size / step << 1;
      var halfLen = len >>> 1;
      var quarterLen = halfLen >>> 1;
      var hquarterLen = quarterLen >>> 1;

      // Loop through offsets in the data
      for (outOff = 0; outOff < size; outOff += len) {
        for (var i = 0, k = 0; i <= hquarterLen; i += 2, k += step) {
          var A = outOff + i;
          var B = A + quarterLen;
          var C = B + quarterLen;
          var D = C + quarterLen;

          // Original values
          var Ar = out[A];
          var Ai = out[A + 1];
          var Br = out[B];
          var Bi = out[B + 1];
          var Cr = out[C];
          var Ci = out[C + 1];
          var Dr = out[D];
          var Di = out[D + 1];

          // Middle values
          var MAr = Ar;
          var MAi = Ai;
          var tableBr = table[k];
          var tableBi = inv * table[k + 1];
          var MBr = Br * tableBr - Bi * tableBi;
          var MBi = Br * tableBi + Bi * tableBr;
          var tableCr = table[2 * k];
          var tableCi = inv * table[2 * k + 1];
          var MCr = Cr * tableCr - Ci * tableCi;
          var MCi = Cr * tableCi + Ci * tableCr;
          var tableDr = table[3 * k];
          var tableDi = inv * table[3 * k + 1];
          var MDr = Dr * tableDr - Di * tableDi;
          var MDi = Dr * tableDi + Di * tableDr;

          // Pre-Final values
          var T0r = MAr + MCr;
          var T0i = MAi + MCi;
          var T1r = MAr - MCr;
          var T1i = MAi - MCi;
          var T2r = MBr + MDr;
          var T2i = MBi + MDi;
          var T3r = inv * (MBr - MDr);
          var T3i = inv * (MBi - MDi);

          // Final values
          var FAr = T0r + T2r;
          var FAi = T0i + T2i;
          var FBr = T1r + T3i;
          var FBi = T1i - T3r;
          out[A] = FAr;
          out[A + 1] = FAi;
          out[B] = FBr;
          out[B + 1] = FBi;

          // Output final middle point
          if (i === 0) {
            var FCr = T0r - T2r;
            var FCi = T0i - T2i;
            out[C] = FCr;
            out[C + 1] = FCi;
            continue;
          }

          // Do not overwrite ourselves
          if (i === hquarterLen) continue;

          // In the flipped case:
          // MAi = -MAi
          // MBr=-MBi, MBi=-MBr
          // MCr=-MCr
          // MDr=MDi, MDi=MDr
          var ST0r = T1r;
          var ST0i = -T1i;
          var ST1r = T0r;
          var ST1i = -T0i;
          var ST2r = -inv * T3i;
          var ST2i = -inv * T3r;
          var ST3r = -inv * T2i;
          var ST3i = -inv * T2r;
          var SFAr = ST0r + ST2r;
          var SFAi = ST0i + ST2i;
          var SFBr = ST1r + ST3i;
          var SFBi = ST1i - ST3r;
          var SA = outOff + quarterLen - i;
          var SB = outOff + halfLen - i;
          out[SA] = SFAr;
          out[SA + 1] = SFAi;
          out[SB] = SFBr;
          out[SB + 1] = SFBi;
        }
      }
    }
  };

  // radix-2 implementation
  //
  // NOTE: Only called for len=4
  FFT.prototype._singleRealTransform2 = function _singleRealTransform2(outOff, off, step) {
    const out = this._out;
    const data = this._data;
    const evenR = data[off];
    const oddR = data[off + step];
    const leftR = evenR + oddR;
    const rightR = evenR - oddR;
    out[outOff] = leftR;
    out[outOff + 1] = 0;
    out[outOff + 2] = rightR;
    out[outOff + 3] = 0;
  };

  // radix-4
  //
  // NOTE: Only called for len=8
  FFT.prototype._singleRealTransform4 = function _singleRealTransform4(outOff, off, step) {
    const out = this._out;
    const data = this._data;
    const inv = this._inv ? -1 : 1;
    const step2 = step * 2;
    const step3 = step * 3;

    // Original values
    const Ar = data[off];
    const Br = data[off + step];
    const Cr = data[off + step2];
    const Dr = data[off + step3];

    // Pre-Final values
    const T0r = Ar + Cr;
    const T1r = Ar - Cr;
    const T2r = Br + Dr;
    const T3r = inv * (Br - Dr);

    // Final values
    const FAr = T0r + T2r;
    const FBr = T1r;
    const FBi = -T3r;
    const FCr = T0r - T2r;
    const FDr = T1r;
    const FDi = T3r;
    out[outOff] = FAr;
    out[outOff + 1] = 0;
    out[outOff + 2] = FBr;
    out[outOff + 3] = FBi;
    out[outOff + 4] = FCr;
    out[outOff + 5] = 0;
    out[outOff + 6] = FDr;
    out[outOff + 7] = FDi;
  };
  var FFT$1 = fft;

  var nextPowerOfTwo_1 = nextPowerOfTwo;
  function nextPowerOfTwo(n) {
    if (n === 0) return 1;
    n--;
    n |= n >> 1;
    n |= n >> 2;
    n |= n >> 4;
    n |= n >> 8;
    n |= n >> 16;
    return n + 1;
  }
  var nextPOT = nextPowerOfTwo_1;

  class FFTConvolution {
    constructor(size, kernel) {
      let borderType = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : 'CONSTANT';
      checkSize(size);
      checkKernel(kernel);
      checkBorderType(borderType);
      this.size = size;
      this.kernelOffset = (kernel.length - 1) / 2;
      this.doubleOffset = 2 * this.kernelOffset;
      this.borderType = borderType;
      const resultLength = size + this.doubleOffset;
      this.fftLength = nextPOT(Math.max(resultLength, 2));
      this.fftComplexLength = this.fftLength * 2;
      this.fft = new FFT$1(this.fftLength);
      kernel = kernel.slice().reverse();
      const paddedKernel = createArray(this.fftComplexLength);
      this.fftKernel = createArray(this.fftComplexLength);
      pad(kernel, paddedKernel, this.fftComplexLength);
      this.fft.transform(this.fftKernel, paddedKernel);
      this.paddedInput = createArray(this.fftComplexLength);
      this.fftInput = createArray(this.fftComplexLength);
      this.ifftOutput = createArray(this.fftComplexLength);
      this.result = paddedKernel;
    }
    convolve(input) {
      checkInputLength(input.length, this.size);
      pad(input, this.paddedInput, this.fftComplexLength);
      this.fft.transform(this.fftInput, this.paddedInput);
      for (var i = 0; i < this.fftInput.length; i += 2) {
        const tmp = this.fftInput[i] * this.fftKernel[i] - this.fftInput[i + 1] * this.fftKernel[i + 1];
        this.fftInput[i + 1] = this.fftInput[i] * this.fftKernel[i + 1] + this.fftInput[i + 1] * this.fftKernel[i];
        this.fftInput[i] = tmp;
      }
      this.fft.inverseTransform(this.ifftOutput, this.fftInput);
      const r = this.fft.fromComplexArray(this.ifftOutput, this.result);
      if (this.borderType === 'CONSTANT') {
        return r.slice(this.kernelOffset, this.kernelOffset + input.length);
      } else {
        return r.slice(this.doubleOffset, input.length);
      }
    }
  }
  function fftConvolution(input, kernel, borderType) {
    return new FFTConvolution(input.length, kernel, borderType).convolve(input);
  }
  function pad(data, out, len) {
    let i = 0;
    for (; i < data.length; i++) {
      out[i * 2] = data[i];
      out[i * 2 + 1] = 0;
    }
    i *= 2;
    for (; i < len; i += 2) {
      out[i] = 0;
      out[i + 1] = 0;
    }
  }

  const GAUSSIAN_EXP_FACTOR = -4 * Math.LN2;
  const ROOT_PI_OVER_LN2 = Math.sqrt(Math.PI / Math.LN2);
  const ROOT_THREE = Math.sqrt(3);
  const ROOT_2LN2 = Math.sqrt(2 * Math.LN2);
  const ROOT_2LN2_MINUS_ONE = Math.sqrt(2 * Math.LN2) - 1;

  // https://en.wikipedia.org/wiki/Error_function#Inverse_functions
  // This code yields to a good approximation
  // If needed a better implementation using polynomial can be found on https://en.wikipedia.org/wiki/Error_function#Inverse_functions
  function erfinv(x) {
    let a = 0.147;
    if (x === 0) return 0;
    let ln1MinusXSqrd = Math.log(1 - x * x);
    let lnEtcBy2Plus2 = ln1MinusXSqrd / 2 + 2 / (Math.PI * a);
    let firstSqrt = Math.sqrt(lnEtcBy2Plus2 ** 2 - ln1MinusXSqrd / a);
    let secondSqrt = Math.sqrt(firstSqrt - lnEtcBy2Plus2);
    return secondSqrt * (x > 0 ? 1 : -1);
  }

  class Gaussian {
    constructor() {
      let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      const {
        fwhm = 500,
        sd
      } = options;
      this.fwhm = sd ? gaussianWidthToFWHM(2 * sd) : fwhm;
    }
    fwhmToWidth() {
      let fwhm = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : this.fwhm;
      return gaussianFwhmToWidth(fwhm);
    }
    widthToFWHM(width) {
      return gaussianWidthToFWHM(width);
    }
    fct(x) {
      return gaussianFct(x, this.fwhm);
    }
    getArea() {
      let height = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : calculateGaussianHeight({
        fwhm: this.fwhm
      });
      return getGaussianArea({
        fwhm: this.fwhm,
        height
      });
    }
    getFactor(area) {
      return getGaussianFactor(area);
    }
    getData() {
      let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      return getGaussianData(this, options);
    }
    calculateHeight() {
      let area = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
      return calculateGaussianHeight({
        fwhm: this.fwhm,
        area
      });
    }
    getParameters() {
      return ['fwhm'];
    }
  }
  function calculateGaussianHeight(options) {
    let {
      fwhm = 500,
      area = 1,
      sd
    } = options;
    if (sd) fwhm = gaussianWidthToFWHM(2 * sd);
    return 2 * area / ROOT_PI_OVER_LN2 / fwhm;
  }
  /**
   * Calculate the height of the gaussian function of a specific width (fwhm) at a speicifc
   * x position (the gaussian is centered on x=0)
   * @param x
   * @param fwhm
   * @returns y
   */
  function gaussianFct(x, fwhm) {
    return Math.exp(GAUSSIAN_EXP_FACTOR * Math.pow(x / fwhm, 2));
  }
  function gaussianWidthToFWHM(width) {
    return width * ROOT_2LN2;
  }
  function gaussianFwhmToWidth(fwhm) {
    return fwhm / ROOT_2LN2;
  }
  function getGaussianArea(options) {
    let {
      fwhm = 500,
      sd,
      height = 1
    } = options;
    if (sd) fwhm = gaussianWidthToFWHM(2 * sd);
    return height * ROOT_PI_OVER_LN2 * fwhm / 2;
  }
  function getGaussianFactor() {
    let area = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 0.9999;
    return Math.sqrt(2) * erfinv(area);
  }
  function getGaussianData() {
    let shape = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    let {
      fwhm = 500,
      sd
    } = shape;
    if (sd) fwhm = gaussianWidthToFWHM(2 * sd);
    let {
      length,
      factor = getGaussianFactor(),
      height = calculateGaussianHeight({
        fwhm
      })
    } = options;
    if (!length) {
      length = Math.min(Math.ceil(fwhm * factor), Math.pow(2, 25) - 1);
      if (length % 2 === 0) length++;
    }
    const center = (length - 1) / 2;
    const data = new Float64Array(length);
    for (let i = 0; i <= center; i++) {
      data[i] = gaussianFct(i - center, fwhm) * height;
      data[length - 1 - i] = data[i];
    }
    return data;
  }

  class Lorentzian {
    constructor() {
      let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      const {
        fwhm = 500
      } = options;
      this.fwhm = fwhm;
    }
    fwhmToWidth() {
      let fwhm = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : this.fwhm;
      return lorentzianFwhmToWidth(fwhm);
    }
    widthToFWHM(width) {
      return lorentzianWidthToFWHM(width);
    }
    fct(x) {
      return lorentzianFct(x, this.fwhm);
    }
    getArea() {
      let height = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
      return getLorentzianArea({
        fwhm: this.fwhm,
        height
      });
    }
    getFactor(area) {
      return getLorentzianFactor(area);
    }
    getData() {
      let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      return getLorentzianData(this, options);
    }
    calculateHeight() {
      let area = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
      return calculateLorentzianHeight({
        fwhm: this.fwhm,
        area
      });
    }
    getParameters() {
      return ['fwhm'];
    }
  }
  const calculateLorentzianHeight = _ref => {
    let {
      fwhm = 1,
      area = 1
    } = _ref;
    return 2 * area / Math.PI / fwhm;
  };
  const getLorentzianArea = options => {
    const {
      fwhm = 500,
      height = 1
    } = options;
    return height * Math.PI * fwhm / 2;
  };
  const lorentzianFct = (x, fwhm) => {
    return fwhm ** 2 / (4 * x ** 2 + fwhm ** 2);
  };
  const lorentzianWidthToFWHM = width => {
    return width * ROOT_THREE;
  };
  const lorentzianFwhmToWidth = fwhm => {
    return fwhm / ROOT_THREE;
  };
  const getLorentzianFactor = function () {
    let area = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 0.9999;
    if (area >= 1) {
      throw new Error('area should be (0 - 1)');
    }
    const halfResidual = (1 - area) * 0.5;
    const quantileFunction = p => Math.tan(Math.PI * (p - 0.5));
    return (quantileFunction(1 - halfResidual) - quantileFunction(halfResidual)) / 2;
  };
  const getLorentzianData = function () {
    let shape = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    let {
      fwhm = 500
    } = shape;
    let {
      length,
      factor = getLorentzianFactor(),
      height = calculateLorentzianHeight({
        fwhm,
        area: 1
      })
    } = options;
    if (!length) {
      length = Math.min(Math.ceil(fwhm * factor), Math.pow(2, 25) - 1);
      if (length % 2 === 0) length++;
    }
    const center = (length - 1) / 2;
    const data = new Float64Array(length);
    for (let i = 0; i <= center; i++) {
      data[i] = lorentzianFct(i - center, fwhm) * height;
      data[length - 1 - i] = data[i];
    }
    return data;
  };

  class PseudoVoigt {
    constructor() {
      let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      const {
        fwhm = 500,
        mu = 0.5
      } = options;
      this.mu = mu;
      this.fwhm = fwhm;
    }
    fwhmToWidth() {
      let fwhm = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : this.fwhm;
      let mu = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : this.mu;
      return pseudoVoigtFwhmToWidth(fwhm, mu);
    }
    widthToFWHM(width) {
      let mu = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : this.mu;
      return pseudoVoigtWidthToFWHM(width, mu);
    }
    fct(x) {
      return pseudoVoigtFct(x, this.fwhm, this.mu);
    }
    getArea() {
      let height = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
      return getPseudoVoigtArea({
        fwhm: this.fwhm,
        height,
        mu: this.mu
      });
    }
    getFactor(area) {
      return getPseudoVoigtFactor(area);
    }
    getData() {
      let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      const {
        length,
        factor,
        height = calculatePseudoVoigtHeight({
          fwhm: this.fwhm,
          mu: this.mu,
          area: 1
        })
      } = options;
      return getPseudoVoigtData(this, {
        factor,
        length,
        height
      });
    }
    calculateHeight() {
      let area = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
      return calculatePseudoVoigtHeight({
        fwhm: this.fwhm,
        mu: this.mu,
        area
      });
    }
    getParameters() {
      return ['fwhm', 'mu'];
    }
  }
  const calculatePseudoVoigtHeight = function () {
    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    let {
      fwhm = 1,
      mu = 0.5,
      area = 1
    } = options;
    return 2 * area / (fwhm * (mu * ROOT_PI_OVER_LN2 + (1 - mu) * Math.PI));
  };
  const pseudoVoigtFct = (x, fwhm, mu) => {
    return (1 - mu) * lorentzianFct(x, fwhm) + mu * gaussianFct(x, fwhm);
  };
  const pseudoVoigtWidthToFWHM = function (width) {
    let mu = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0.5;
    return width * (mu * ROOT_2LN2_MINUS_ONE + 1);
  };
  const pseudoVoigtFwhmToWidth = function (fwhm) {
    let mu = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0.5;
    return fwhm / (mu * ROOT_2LN2_MINUS_ONE + 1);
  };
  const getPseudoVoigtArea = options => {
    const {
      fwhm = 500,
      height = 1,
      mu = 0.5
    } = options;
    return fwhm * height * (mu * ROOT_PI_OVER_LN2 + (1 - mu) * Math.PI) / 2;
  };
  const getPseudoVoigtFactor = function () {
    let area = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 0.9999;
    let mu = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0.5;
    return mu < 1 ? getLorentzianFactor(area) : getGaussianFactor(area);
  };
  const getPseudoVoigtData = function () {
    let shape = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    let {
      fwhm = 500,
      mu = 0.5
    } = shape;
    let {
      length,
      factor = getPseudoVoigtFactor(0.999, mu),
      height = calculatePseudoVoigtHeight({
        fwhm,
        mu,
        area: 1
      })
    } = options;
    if (!height) {
      height = 1 / (mu / Math.sqrt(-GAUSSIAN_EXP_FACTOR / Math.PI) * fwhm + (1 - mu) * fwhm * Math.PI / 2);
    }
    if (!length) {
      length = Math.min(Math.ceil(fwhm * factor), Math.pow(2, 25) - 1);
      if (length % 2 === 0) length++;
    }
    const center = (length - 1) / 2;
    const data = new Float64Array(length);
    for (let i = 0; i <= center; i++) {
      data[i] = pseudoVoigtFct(i - center, fwhm, mu) * height;
      data[length - 1 - i] = data[i];
    }
    return data;
  };

  /**
   * Generate a instance of a specific kind of shape.
   */
  function getShape1D(shape) {
    const {
      kind
    } = shape;
    switch (kind) {
      case 'gaussian':
        return new Gaussian(shape);
      case 'lorentzian':
        return new Lorentzian(shape);
      case 'pseudoVoigt':
        return new PseudoVoigt(shape);
      default:
        {
          throw Error(`Unknown distribution ${kind}`);
        }
    }
  }

  const toString = Object.prototype.toString;
  /**
   * Checks if an object is an instance of an Array (array or typed array).
   *
   * @param {any} value - Object to check.
   * @returns {boolean} True if the object is an array.
   */
  function isAnyArray(value) {
    return toString.call(value).endsWith('Array]');
  }

  /**
   * This function
   * @param output - undefined or a new array
   * @param length - length of the output array
   * @returns
   */
  function getOutputArray(output, length) {
    if (output !== undefined) {
      if (!isAnyArray(output)) {
        throw new TypeError('output option must be an array if specified');
      }
      if (output.length !== length) {
        throw new TypeError('the output array does not have the correct length');
      }
      return output;
    } else {
      return new Float64Array(length);
    }
  }

  /**
   * Checks if input is of type array
   *
   * @param input - input
   */
  function xCheck(input) {
    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }
    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }
  }

  /**
   * Returns the closest index of a `target`
   *
   * @param array - array of numbers
   * @param target - target
   * @returns - closest index
   */
  function xFindClosestIndex(array, target) {
    let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
    const {
      sorted = true
    } = options;
    if (sorted) {
      let low = 0;
      let high = array.length - 1;
      let middle = 0;
      while (high - low > 1) {
        middle = low + (high - low >> 1);
        if (array[middle] < target) {
          low = middle;
        } else if (array[middle] > target) {
          high = middle;
        } else {
          return middle;
        }
      }
      if (low < array.length - 1) {
        if (Math.abs(target - array[low]) < Math.abs(array[low + 1] - target)) {
          return low;
        } else {
          return low + 1;
        }
      } else {
        return low;
      }
    } else {
      let index = 0;
      let diff = Number.POSITIVE_INFINITY;
      for (let i = 0; i < array.length; i++) {
        const currentDiff = Math.abs(array[i] - target);
        if (currentDiff < diff) {
          diff = currentDiff;
          index = i;
        }
      }
      return index;
    }
  }

  /**
   * Returns an object with {fromIndex, toIndex} for a specific from / to
   *
   * @param x - array of numbers
   * @param options - Options
   */
  function xGetFromToIndex(x) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    let {
      fromIndex,
      toIndex,
      from,
      to
    } = options;
    if (fromIndex === undefined) {
      if (from !== undefined) {
        fromIndex = xFindClosestIndex(x, from);
      } else {
        fromIndex = 0;
      }
    }
    if (toIndex === undefined) {
      if (to !== undefined) {
        toIndex = xFindClosestIndex(x, to);
      } else {
        toIndex = x.length - 1;
      }
    }
    if (fromIndex < 0) fromIndex = 0;
    if (toIndex < 0) toIndex = 0;
    if (fromIndex >= x.length) fromIndex = x.length - 1;
    if (toIndex >= x.length) toIndex = x.length - 1;
    if (fromIndex > toIndex) [fromIndex, toIndex] = [toIndex, fromIndex];
    return {
      fromIndex,
      toIndex
    };
  }

  /**
   * Computes the maximal value of an array of values
   *
   * @param array - array of numbers
   * @param options - options
   */
  function xMaxValue(array) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    xCheck(array);
    const {
      fromIndex,
      toIndex
    } = xGetFromToIndex(array, options);
    let maxValue = array[fromIndex];
    for (let i = fromIndex + 1; i <= toIndex; i++) {
      if (array[i] > maxValue) {
        maxValue = array[i];
      }
    }
    return maxValue;
  }

  /**
   * Calculate the sum of the values
   *
   * @param array - Object that contains property x (an ordered increasing array) and y (an array).
   * @param options - Options.
   * @returns XSum value on the specified range.
   */
  function xSum(array) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    xCheck(array);
    const {
      fromIndex,
      toIndex
    } = xGetFromToIndex(array, options);
    let sumValue = array[fromIndex];
    for (let i = fromIndex + 1; i <= toIndex; i++) {
      sumValue += array[i];
    }
    return sumValue;
  }

  /**
   * Divides the data with either the sum, the absolute sum or the maximum of the data
   * @param array - Array containing values
   * @param options - options
   * @returns - normalized data
   */
  function xNormed(input) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    const {
      algorithm = 'absolute',
      value = 1
    } = options;
    xCheck(input);
    const output = getOutputArray(options.output, input.length);
    if (input.length === 0) {
      throw new Error('input must not be empty');
    }
    switch (algorithm.toLowerCase()) {
      case 'absolute':
        {
          let absoluteSumValue = absoluteSum(input) / value;
          if (absoluteSumValue === 0) {
            throw new Error('xNormed: trying to divide by 0');
          }
          for (let i = 0; i < input.length; i++) {
            output[i] = input[i] / absoluteSumValue;
          }
          return output;
        }
      case 'max':
        {
          let currentMaxValue = xMaxValue(input);
          if (currentMaxValue === 0) {
            throw new Error('xNormed: trying to divide by 0');
          }
          const factor = value / currentMaxValue;
          for (let i = 0; i < input.length; i++) {
            output[i] = input[i] * factor;
          }
          return output;
        }
      case 'sum':
        {
          let sumFactor = xSum(input) / value;
          if (sumFactor === 0) {
            throw new Error('xNormed: trying to divide by 0');
          }
          for (let i = 0; i < input.length; i++) {
            output[i] = input[i] / sumFactor;
          }
          return output;
        }
      default:
        throw new Error(`norm: unknown algorithm: ${algorithm}`);
    }
  }
  function absoluteSum(input) {
    let sumValue = 0;
    for (let i = 0; i < input.length; i++) {
      sumValue += Math.abs(input[i]);
    }
    return sumValue;
  }

  /**
   * This function returns a broadened spectrum
   * @param array - The original spectrum
   * @param options - options
   * @returns - broadened spectrum
   */
  function integralTransform(array) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    const {
      shape = {
        kind: 'gaussian',
        sd: 1.2
      },
      kernelWidth = 7,
      normalized = false,
      kernelHeight = 1,
      maxHeight = 1
    } = options;
    const kernelBasis = getShape1D({
      ...shape,
      fwhm: kernelWidth
    });
    const kernel = kernelBasis.getData({
      length: kernelWidth,
      height: kernelHeight
    });
    const result = fftConvolution(array, kernel, 'CONSTANT');
    return normalized ? xNormed(result, {
      algorithm: 'max',
      value: maxHeight
    }) : result;
  }

  exports.integralTransform = integralTransform;

  Object.defineProperty(exports, '__esModule', { value: true });

}));
//# sourceMappingURL=ml-integral-transforms.js.map
