/**
 * convert-to-jcamp - Convert strings into JCAMP
 * @version v5.0.0
 * @link https://github.com/cheminfo/convert-to-jcamp#readme
 * @license MIT
 */
(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
  typeof define === 'function' && define.amd ? define(['exports'], factory) :
  (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.ConvertToJcamp = {}));
})(this, (function (exports) { 'use strict';

  const toString = Object.prototype.toString;
  function isAnyArray(object) {
    return toString.call(object).endsWith('Array]');
  }

  function max(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;

    if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
      throw new Error('fromIndex must be a positive integer smaller than length');
    }

    if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
      throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
    }

    var maxValue = input[fromIndex];

    for (var i = fromIndex + 1; i < toIndex; i++) {
      if (input[i] > maxValue) maxValue = input[i];
    }

    return maxValue;
  }

  function min(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;

    if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
      throw new Error('fromIndex must be a positive integer smaller than length');
    }

    if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
      throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
    }

    var minValue = input[fromIndex];

    for (var i = fromIndex + 1; i < toIndex; i++) {
      if (input[i] < minValue) minValue = input[i];
    }

    return minValue;
  }

  const addInfoData = function (data) {
    let keys = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : Object.keys(data);
    let prefix = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : '##$';
    let header = '';

    for (const key of keys) {
      header += typeof data[key] === 'object' ? `${prefix}${key}=${JSON.stringify(data[key])}\n` : `${prefix}${key}=${data[key]}\n`;
    }

    return header;
  };

  /**
   * Parse from a xyxy data array
   * @param variables - Variables to convert to jcamp
   * @param [options={}] - options that allows to add meta data in the jcamp
   * @return JCAMP-DX text file corresponding to the variables
   */

  function creatorNtuples(variables, options) {
    const {
      meta = {},
      info = {}
    } = options;
    const {
      title = '',
      owner = '',
      origin = '',
      dataType = ''
    } = info;
    const symbol = [];
    const varName = [];
    const varType = [];
    const varDim = [];
    const units = [];
    const first = [];
    const last = [];
    const min$1 = [];
    const max$1 = [];
    const keys = Object.keys(variables);

    for (let i = 0; i < keys.length; i++) {
      const key = keys[i];
      let variable = variables[key];
      if (!variable) continue;
      let name = variable === null || variable === void 0 ? void 0 : variable.label.replace(/ *\[.*/, '');
      let unit = variable === null || variable === void 0 ? void 0 : variable.label.replace(/.*\[(?<units>.*)\].*/, '$<units>');
      symbol.push(variable.symbol || key);
      varName.push(name || key);
      varDim.push(variable.data.length);

      if (variable.isDependent !== undefined) {
        varType.push(variable.isDependent ? 'DEPENDENT' : 'INDEPENDENT');
      } else {
        varType.push(variable.isDependent !== undefined ? !variable.isDependent : i === 0 ? 'INDEPENDENT' : 'DEPENDENT');
      }

      units.push(variable.units || unit || '');
      first.push(variable.data[0]);
      last.push(variable.data[variable.data.length - 1]);
      min$1.push(min(variable.data));
      max$1.push(max(variable.data));
    }

    let header = `##TITLE=${title}
##JCAMP-DX=6.00
##DATA TYPE=${dataType}
##ORIGIN=${origin}
##OWNER=${owner}\n`;
    const infoKeys = Object.keys(info).filter(e => !['title', 'owner', 'origin', 'dataType'].includes(e));
    header += addInfoData(info, infoKeys, '##');
    header += addInfoData(meta);
    header += `##NTUPLES= ${dataType}
##VAR_NAME=  ${varName.join()}
##SYMBOL=    ${symbol.join()}
##VAR_TYPE=  ${varType.join()}
##VAR_DIM=   ${varDim.join()}
##UNITS=     ${units.join()}
##PAGE= N=1\n`;
    header += `##DATA TABLE= (${symbol.join('')}..${symbol.join('')}), PEAKS\n`;

    for (let i = 0; i < variables.x.data.length; i++) {
      let point = [];

      for (let key of keys) {
        let variable = variables[key];
        if (!variable) continue;
        point.push(variable.data[i]);
      }

      header += `${point.join('\t')}\n`;
    }

    header += '##END';
    return header;
  }

  /**
   * Reconvert number to original value
   * @param number Number used for computation
   * @param factor Multiplying factor
   * @returns Original value
   */
  function getNumber(number, factor) {
    if (factor !== 1) number /= factor;
    const rounded = Math.round(number);

    if (rounded !== number && Math.abs(rounded - number) <= Number.EPSILON) {
      return rounded;
    }

    return number;
  }

  function peakTableCreator(data) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    const {
      xFactor = 1,
      yFactor = 1
    } = options.info || {};
    let firstX = Number.POSITIVE_INFINITY;
    let lastX = Number.NEGATIVE_INFINITY;
    let firstY = Number.POSITIVE_INFINITY;
    let lastY = Number.NEGATIVE_INFINITY;
    let lines = [];

    for (let i = 0; i < data.x.length; i++) {
      let x = data.x[i];
      let y = data.y[i];

      if (firstX > x) {
        firstX = x;
      }

      if (lastX < x) {
        lastX = x;
      }

      if (firstY > y) {
        firstY = y;
      }

      if (lastY < y) {
        lastY = y;
      }
    }

    lines.push(`##FIRSTX=${firstX}`);
    lines.push(`##LASTX=${lastX}`);
    lines.push(`##FIRSTY=${firstY}`);
    lines.push(`##LASTY=${lastY}`);
    lines.push(`##XFACTOR=${xFactor}`);
    lines.push(`##YFACTOR=${yFactor}`);
    lines.push('##PEAK TABLE=(XY..XY)');

    for (let i = 0; i < data.x.length; i++) {
      lines.push(`${getNumber(data.x[i], xFactor)} ${getNumber(data.y[i], yFactor)}`);
    }

    return lines;
  }

  /**
   * class encodes a integer vector as a String in order to store it in a text file.
   * The algorithms used to encode the data are describe in:
   *            http://www.iupac.org/publications/pac/pdf/2001/pdf/7311x1765.pdf
   */
  const newLine = '\r\n';
  const pseudoDigits = [['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'], ['@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I'], ['@', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i'], ['%', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R'], ['%', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r'], [' ', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 's']];
  const SQZ_P = 1;
  const SQZ_N = 2;
  const DIF_P = 3;
  const DIF_N = 4;
  const DUP = 5;
  const MaxLinelength = 100;
  /**
   * This function encodes the given vector. The xyEncoding format is specified by the
   * xyEncoding option
   * @param xyEncoding: ('FIX','SQZ','DIF','DIFDUP','CVS','PAC') Default 'DIFDUP'
   * @return {string}
   */

  function encode(data, firstX, intervalX, xyEncoding) {
    switch (xyEncoding) {
      case 'FIX':
        return fixEncoding(data, firstX, intervalX);

      case 'SQZ':
        return squeezedEncoding(data, firstX, intervalX);

      case 'DIF':
        return differenceEncoding(data, firstX, intervalX);

      case 'DIFDUP':
        return differenceDuplicateEncoding(data, firstX, intervalX);

      case 'CSV':
        return commaSeparatedValuesEncoding(data, firstX, intervalX);

      case 'PAC':
        return packedEncoding(data, firstX, intervalX);

      default:
        return differenceEncoding(data, firstX, intervalX);
    }
  }
  /**
   * @private
   * No data compression used. The data is separated by a comma(',').
   */

  function commaSeparatedValuesEncoding(data, firstX, intervalX) {
    return fixEncoding(data, firstX, intervalX, ',');
  }
  /**
   * @private
   * No data compression used. The data is separated by the specified separator.
   */

  function fixEncoding(data, firstX, intervalX) {
    let separator = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : ' ';
    let outputData = '';
    let j = 0;
    let TD = data.length;
    let i;

    while (j < TD - 7) {
      outputData += Math.ceil(firstX + j * intervalX);

      for (i = 0; i < 8; i++) {
        outputData += separator + data[j++];
      }

      outputData += newLine;
    }

    if (j < TD) {
      // We add last numbers
      outputData += Math.ceil(firstX + j * intervalX);

      for (i = j; i < TD; i++) {
        outputData += separator + data[i];
      }
    }

    return outputData;
  }
  /**
   * @private
   * No data compression used. The data is separated by the sign of the number.
   */

  function packedEncoding(data, firstX, intervalX) {
    let outputData = '';
    let j = 0;
    let TD = data.length;

    while (j < TD - 7) {
      outputData += Math.ceil(firstX + j * intervalX);

      for (let i = 0; i < 8; i++) {
        outputData += data[j] < 0 ? data[j++] : `+${data[j++]}`;
      }

      outputData += newLine;
    }

    if (j < TD) {
      // We add last numbers
      outputData += Math.ceil(firstX + j * intervalX);

      for (let i = j; i < TD; i++) {
        outputData += data[i] < 0 ? data[i] : `+${data[i]}`;
      }
    }

    return outputData;
  }
  /**
   * @private
   * Data compression is possible using the squeezed form (SQZ) in which the delimiter, the leading digit,
   * and sign are replaced by a pseudo-digit from Table 1. For example, the Y-values 30, 32 would be
   * represented as C0C2.
   */

  function squeezedEncoding(data, firstX, intervalX) {
    let outputData = ''; // String outputData = new String();

    let j = 0;
    let TD = data.length;

    while (j < TD - 10) {
      outputData += Math.ceil(firstX + j * intervalX);

      for (let i = 0; i < 10; i++) {
        outputData += squeezedDigit(data[j++].toString());
      }

      outputData += newLine;
    }

    if (j < TD) {
      // We add last numbers
      outputData += Math.ceil(firstX + j * intervalX);

      for (let i = j; i < TD; i++) {
        outputData += squeezedDigit(data[i].toString());
      }
    }

    return outputData;
  }
  /**
   * @private
   * Duplicate suppression xyEncoding
   */

  function differenceDuplicateEncoding(data, firstX, intervalX) {
    let mult = 0;
    let index = 0;
    let charCount = 0; // We built a string where we store the encoded data.

    let encodedData = '';
    let encodedNumber = '';
    let temp = ''; // We calculate the differences vector

    let diffData = new Array(data.length - 1);

    for (let i = 0; i < diffData.length; i++) {
      diffData[i] = data[i + 1] - data[i];
    } // We simulate a line carry


    let numDiff = diffData.length;

    while (index < numDiff) {
      if (charCount === 0) {
        // Start line
        encodedNumber = Math.ceil(firstX + index * intervalX) + squeezedDigit(data[index].toString()) + differenceDigit(diffData[index].toString());
        encodedData += encodedNumber;
        charCount += encodedNumber.length;
      } else {
        // Try to insert next difference
        if (diffData[index - 1] === diffData[index]) {
          mult++;
        } else {
          if (mult > 0) {
            // Now we know that it can be in line
            mult++;
            encodedNumber = duplicateDigit(mult.toString());
            encodedData += encodedNumber;
            charCount += encodedNumber.length;
            mult = 0;
            index--;
          } else {
            // Mirar si cabe, en caso contrario iniciar una nueva linea
            encodedNumber = differenceDigit(diffData[index].toString());

            if (encodedNumber.length + charCount < MaxLinelength) {
              encodedData += encodedNumber;
              charCount += encodedNumber.length;
            } else {
              // Iniciar nueva linea
              encodedData += newLine;
              temp = Math.ceil(firstX + index * intervalX) + squeezedDigit(data[index].toString()) + encodedNumber;
              encodedData += temp; // Each line start with first index number.

              charCount = temp.length;
            }
          }
        }
      }

      index++;
    }

    if (mult > 0) {
      encodedData += duplicateDigit((mult + 1).toString());
    } // We insert the last data from fid. It is done to control of data
    // The last line start with the number of datas in the fid.


    encodedData += newLine + Math.ceil(firstX + index * intervalX) + squeezedDigit(data[index].toString());
    return encodedData;
  }
  /**
   * @private
   * Differential xyEncoding
   */

  function differenceEncoding(data, firstX, intervalX) {
    let index = 0;
    let charCount = 0;
    let i;
    let encodedData = '';
    let encodedNumber = '';
    let temp = ''; // We calculate the differences vector

    let diffData = new Array(data.length - 1);

    for (i = 0; i < diffData.length; i++) {
      diffData[i] = data[i + 1] - data[i];
    }

    let numDiff = diffData.length;

    while (index < numDiff) {
      if (charCount === 0) {
        // We convert the first number.
        encodedNumber = Math.ceil(firstX + index * intervalX) + squeezedDigit(data[index].toString()) + differenceDigit(diffData[index].toString());
        encodedData += encodedNumber;
        charCount += encodedNumber.length;
      } else {
        encodedNumber = differenceDigit(diffData[index].toString());

        if (encodedNumber.length + charCount < MaxLinelength) {
          encodedData += encodedNumber;
          charCount += encodedNumber.length;
        } else {
          encodedData += newLine;
          temp = Math.ceil(firstX + index * intervalX) + squeezedDigit(data[index].toString()) + encodedNumber;
          encodedData += temp; // Each line start with first index number.

          charCount = temp.length;
        }
      }

      index++;
    } // We insert the last number from data. It is done to control of data


    encodedData += newLine + Math.ceil(firstX + index * intervalX) + squeezedDigit(data[index].toString());
    return encodedData;
  }
  /**
   * @private
   * Convert number to the ZQZ format, using pseudo digits.
   */

  function squeezedDigit(num) {
    let sqzDigits = '';

    if (num.startsWith('-')) {
      sqzDigits += pseudoDigits[SQZ_N][num.charCodeAt(1) - 48];

      if (num.length > 2) {
        sqzDigits += num.substring(2);
      }
    } else {
      sqzDigits += pseudoDigits[SQZ_P][num.charCodeAt(0) - 48];

      if (num.length > 1) {
        sqzDigits += num.substring(1);
      }
    }

    return sqzDigits;
  }
  /**
   * Convert number to the DIF format, using pseudo digits.
   */


  function differenceDigit(num) {
    let diffDigits = '';

    if (num.startsWith('-')) {
      diffDigits += pseudoDigits[DIF_N][num.charCodeAt(1) - 48];

      if (num.length > 2) {
        diffDigits += num.substring(2);
      }
    } else {
      diffDigits += pseudoDigits[DIF_P][num.charCodeAt(0) - 48];

      if (num.length > 1) {
        diffDigits += num.substring(1);
      }
    }

    return diffDigits;
  }
  /**
   * Convert number to the DUP format, using pseudo digits.
   */


  function duplicateDigit(num) {
    let dupDigits = '';
    dupDigits += pseudoDigits[DUP][num.charCodeAt(0) - 48];

    if (num.length > 1) {
      dupDigits += num.substring(1);
    }

    return dupDigits;
  }

  // import { getNumber } from './getNumber';
  function xyDataCreator(data) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    const {
      xyEncoding = 'DIF'
    } = options;
    const {
      xFactor = 1,
      yFactor = 1
    } = options.info || {};
    let firstX = data.x[0];
    let lastX = data.x[data.x.length - 1];
    let firstY = data.y[0];
    let lastY = data.y[data.y.length - 1];
    let nbPoints = data.x.length;
    let deltaX = (lastX - firstX) / (nbPoints - 1);
    let lines = [];
    lines.push(`##FIRSTX=${firstX}`);
    lines.push(`##LASTX=${lastX}`);
    lines.push(`##FIRSTY=${firstY}`);
    lines.push(`##LASTY=${lastY}`);
    lines.push(`##DELTAX=${deltaX}`);
    lines.push(`##XFACTOR=${xFactor}`);
    lines.push(`##YFACTOR=${yFactor}`);
    lines.push('##XYDATA=(X++(Y..Y))');
    let line = encode(data.y, data.x[0], deltaX, xyEncoding);
    if (line) lines.push(line);
    return lines;
  }

  const infoDefaultKeys = ['title', 'owner', 'origin', 'dataType', 'xUnits', 'yUnits', 'xFactor', 'yFactor'];
  /**
   * Create a jcamp
   * @param data object of array
   * @param [options={meta:{},info:{}} - metadata object
   * @returns JCAMP of the input
   */

  function fromJSON(data) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    const {
      meta = {},
      info = {},
      xyData = false
    } = options;
    const {
      title = '',
      owner = '',
      origin = '',
      dataType = '',
      xUnits = '',
      yUnits = '',
      xFactor = 1,
      yFactor = 1
    } = info;
    data = {
      x: data.x,
      y: data.y
    };
    let header = `##TITLE=${title}
##JCAMP-DX=4.24
##DATA TYPE=${dataType}
##ORIGIN=${origin}
##OWNER=${owner}
##XUNITS=${xUnits}
##YUNITS=${yUnits}\n`;
    const infoKeys = Object.keys(info).filter(e => !infoDefaultKeys.includes(e));
    header += addInfoData(info, infoKeys, '##');
    header += addInfoData(meta); // we leave the header and utf8 fonts ${header.replace(/[^\t\r\n\x20-\x7F]/g, '')

    return `${header}##NPOINTS=${data.x.length}
${(xyData ? xyDataCreator(data, {
    info: {
      xFactor,
      yFactor
    }
  }) : peakTableCreator(data, {
    info: {
      xFactor,
      yFactor
    }
  })).join('\n')}
##END=`;
  }

  /**
   * Create a jcamp from variables
   */

  function fromVariables(
  /** object of variables */
  variables) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    const {
      info = {},
      meta = {},
      forceNtuples = false
    } = options;
    let jcampOptions = {
      info,
      meta
    };
    let keys = Object.keys(variables).map(key => key.toLowerCase());

    if (keys.length === 2 && keys.includes('x') && keys.includes('y') && !forceNtuples) {
      let x = variables.x;
      let xLabel = x.label || 'x';

      if (variables.x.units) {
        if (xLabel.includes(variables.x.units)) {
          jcampOptions.info.xUnits = xLabel;
        } else {
          jcampOptions.info.xUnits = `${xLabel} (${variables.x.units})`;
        }
      } else {
        jcampOptions.info.xUnits = xLabel;
      }

      let y = variables.y;
      let yLabel = y.label || 'y';

      if (variables.y.units) {
        if (yLabel.includes(variables.y.units)) {
          jcampOptions.info.xUnits = yLabel;
        } else {
          jcampOptions.info.yUnits = `${yLabel} (${variables.y.units})`;
        }
      } else {
        jcampOptions.info.yUnits = yLabel;
      }

      return fromJSON({
        x: variables.x.data,
        y: variables.y.data
      }, jcampOptions);
    } else {
      return creatorNtuples(variables, options);
    }
  }

  exports.fromJSON = fromJSON;
  exports.fromVariables = fromVariables;

  Object.defineProperty(exports, '__esModule', { value: true });

}));
//# sourceMappingURL=convert-to-jcamp.js.map
