/**
 * mass-tools
 * @version v0.63.0
 * @link https://github.com/cheminfo/mass-tools#readme
 * @license MIT
 */
(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory() :
  typeof define === 'function' && define.amd ? define(factory) :
  (global = typeof globalThis !== 'undefined' ? globalThis : global || self, global.MassTools = factory());
}(this, (function () { 'use strict';

  const toString$a = Object.prototype.toString;
  function isAnyArray$7(object) {
    return toString$a.call(object).endsWith('Array]');
  }

  var src$q = /*#__PURE__*/Object.freeze({
    __proto__: null,
    'default': isAnyArray$7
  });

  var commonjsGlobal$1 = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

  function getAugmentedNamespace(n) {
  	if (n.__esModule) return n;
  	var a = Object.defineProperty({}, '__esModule', {value: true});
  	Object.keys(n).forEach(function (k) {
  		var d = Object.getOwnPropertyDescriptor(n, k);
  		Object.defineProperty(a, k, d.get ? d : {
  			enumerable: true,
  			get: function () {
  				return n[k];
  			}
  		});
  	});
  	return a;
  }

  function createCommonjsModule$1(fn) {
    var module = { exports: {} };
  	return fn(module, module.exports), module.exports;
  }

  function commonjsRequire (target) {
  	throw new Error('Could not dynamically require "' + target + '". Please configure the dynamicRequireTargets option of @rollup/plugin-commonjs appropriately for this require call to behave properly.');
  }

  var isArray$2 = /*@__PURE__*/getAugmentedNamespace(src$q);

  function _interopDefaultLegacy$5(e) {
    return e && typeof e === 'object' && 'default' in e ? e : {
      'default': e
    };
  }

  var isArray__default$4 = /*#__PURE__*/_interopDefaultLegacy$5(isArray$2);

  function max$8(input, options = {}) {
    if (!isArray__default$4['default'](input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    const {
      fromIndex = 0,
      toIndex = input.length
    } = options;

    if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
      throw new Error('fromIndex must be a positive integer smaller than length');
    }

    if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
      throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
    }

    let maxValue = input[fromIndex];

    for (let i = fromIndex + 1; i < toIndex; i++) {
      if (input[i] > maxValue) maxValue = input[i];
    }

    return maxValue;
  }

  var lib$7 = max$8;

  function max$7(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (!isAnyArray$7(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;

    if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
      throw new Error('fromIndex must be a positive integer smaller than length');
    }

    if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
      throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
    }

    var maxValue = input[fromIndex];

    for (var i = fromIndex + 1; i < toIndex; i++) {
      if (input[i] > maxValue) maxValue = input[i];
    }

    return maxValue;
  }

  var libEs6$2 = /*#__PURE__*/Object.freeze({
    __proto__: null,
    'default': max$7
  });

  function sum$4(input) {
    if (!isAnyArray$7(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    var sumValue = 0;

    for (var i = 0; i < input.length; i++) {
      sumValue += input[i];
    }

    return sumValue;
  }

  var libEs6$1 = /*#__PURE__*/Object.freeze({
    __proto__: null,
    'default': sum$4
  });

  var max$6 = /*@__PURE__*/getAugmentedNamespace(libEs6$2);

  var sum$3 = /*@__PURE__*/getAugmentedNamespace(libEs6$1);

  function _interopDefaultLegacy$4(e) {
    return e && typeof e === 'object' && 'default' in e ? e : {
      'default': e
    };
  }

  var isArray__default$3 = /*#__PURE__*/_interopDefaultLegacy$4(isArray$2);

  var max__default = /*#__PURE__*/_interopDefaultLegacy$4(max$6);

  var sum__default = /*#__PURE__*/_interopDefaultLegacy$4(sum$3);

  function norm(input, options = {}) {
    const {
      algorithm = 'absolute',
      sumValue = 1,
      maxValue = 1
    } = options;

    if (!isArray__default$3['default'](input)) {
      throw new Error('input must be an array');
    }

    let output;

    if (options.output !== undefined) {
      if (!isArray__default$3['default'](options.output)) {
        throw new TypeError('output option must be an array if specified');
      }

      output = options.output;
    } else {
      output = new Array(input.length);
    }

    if (input.length === 0) {
      throw new Error('input must not be empty');
    }

    switch (algorithm.toLowerCase()) {
      case 'absolute':
        {
          let absoluteSumValue = absoluteSum(input) / sumValue;
          if (absoluteSumValue === 0) return input.slice(0);

          for (let i = 0; i < input.length; i++) {
            output[i] = input[i] / absoluteSumValue;
          }

          return output;
        }

      case 'max':
        {
          let currentMaxValue = max__default['default'](input);
          if (currentMaxValue === 0) return input.slice(0);
          const factor = maxValue / currentMaxValue;

          for (let i = 0; i < input.length; i++) {
            output[i] = input[i] * factor;
          }

          return output;
        }

      case 'sum':
        {
          let sumFactor = sum__default['default'](input) / sumValue;
          if (sumFactor === 0) return input.slice(0);

          for (let i = 0; i < input.length; i++) {
            output[i] = input[i] / sumFactor;
          }

          return output;
        }

      default:
        throw new Error(`norm: unknown algorithm: ${algorithm}`);
    }
  }

  function absoluteSum(input) {
    let sumValue = 0;

    for (let i = 0; i < input.length; i++) {
      sumValue += Math.abs(input[i]);
    }

    return sumValue;
  }

  var lib$6 = norm;

  /**
   * In place modification of the 2 arrays to make X unique and sum the Y if X has the same value
   * @param {object} [points={}] : Object of points contains property x (an array) and y (an array)
   * @return points
   */
  function uniqueX(points = {}) {
    const {
      x,
      y
    } = points;
    if (x.length < 2) return;

    if (x.length !== y.length) {
      throw new Error('The X and Y arrays mush have the same length');
    }

    let current = x[0];
    let counter = 0;

    for (let i = 1; i < x.length; i++) {
      if (current !== x[i]) {
        counter++;
        current = x[i];
        x[counter] = x[i];

        if (i !== counter) {
          y[counter] = 0;
        }
      }

      if (i !== counter) {
        y[counter] += y[i];
      }
    }

    x.length = counter + 1;
    y.length = counter + 1;
  }

  /**
   * Returns true if x is monotone
   * @param {Array} array
   * @return {boolean}
   */
  function xIsMonotone$1(array) {
    if (array.length <= 2) {
      return true;
    }

    if (array[0] === array[1]) {
      // maybe a constant series
      for (let i = 1; i < array.length - 1; i++) {
        if (array[i] !== array[i + 1]) return false;
      }

      return true;
    }

    if (array[0] < array[array.length - 1]) {
      for (let i = 0; i < array.length - 1; i++) {
        if (array[i] >= array[i + 1]) return false;
      }
    } else {
      for (let i = 0; i < array.length - 1; i++) {
        if (array[i] <= array[i + 1]) return false;
      }
    }

    return true;
  }

  function min$8(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (!isAnyArray$7(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;

    if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
      throw new Error('fromIndex must be a positive integer smaller than length');
    }

    if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
      throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
    }

    var minValue = input[fromIndex];

    for (var i = fromIndex + 1; i < toIndex; i++) {
      if (input[i] < minValue) minValue = input[i];
    }

    return minValue;
  }

  const GAUSSIAN_EXP_FACTOR$2 = -4 * Math.LN2;
  const ROOT_PI_OVER_LN2$2 = Math.sqrt(Math.PI / Math.LN2);
  const ROOT_THREE$2 = Math.sqrt(3);
  const ROOT_2LN2$2 = Math.sqrt(2 * Math.LN2);
  const ROOT_2LN2_MINUS_ONE$2 = Math.sqrt(2 * Math.LN2) - 1;

  // https://en.wikipedia.org/wiki/Error_function#Inverse_functions
  // This code yields to a good approximation
  // If needed a better implementation using polynomial can be found on https://en.wikipedia.org/wiki/Error_function#Inverse_functions
  function erfinv$2(x) {
    let a = 0.147;
    if (x === 0) return 0;
    let ln1MinusXSqrd = Math.log(1 - x * x);
    let lnEtcBy2Plus2 = ln1MinusXSqrd / 2 + 2 / (Math.PI * a);
    let firstSqrt = Math.sqrt(lnEtcBy2Plus2 ** 2 - ln1MinusXSqrd / a);
    let secondSqrt = Math.sqrt(firstSqrt - lnEtcBy2Plus2);
    return secondSqrt * (x > 0 ? 1 : -1);
  }

  class Gaussian$2 {
    /**
     * @param {object} [options = {}]
     * @param {number} [options.height=4*LN2/(PI*FWHM)] Define the height of the peak, by default area=1 (normalized)
     * @param {number} [options.fwhm = 500] - Full Width at Half Maximum in the number of points in FWHM.
     * @param {number} [options.sd] - Standard deviation, if it's defined options.fwhm will be ignored and the value will be computed sd * Math.sqrt(8 * Math.LN2);
     */
    constructor(options = {}) {
      this.fwhm = options.sd ? Gaussian$2.widthToFWHM(2 * options.sd) : options.fwhm ? options.fwhm : 500;
      this.height = options.height === undefined ? Math.sqrt(-GAUSSIAN_EXP_FACTOR$2 / Math.PI) / this.fwhm : options.height;
    }
    /**
     * Calculate a gaussian shape
     * @param {object} [options = {}]
     * @param {number} [options.factor = 6] - Number of time to take fwhm to calculate length. Default covers 99.99 % of area.
     * @param {number} [options.length = fwhm * factor + 1] - total number of points to calculate
     * @return {Float64Array} y values
     */


    getData(options = {}) {
      let {
        length,
        factor = this.getFactor()
      } = options;

      if (!length) {
        length = Math.min(Math.ceil(this.fwhm * factor), Math.pow(2, 25) - 1);
        if (length % 2 === 0) length++;
      }

      const center = (length - 1) / 2;
      const data = new Float64Array(length);

      for (let i = 0; i <= center; i++) {
        data[i] = this.fct(i - center) * this.height;
        data[length - 1 - i] = data[i];
      }

      return data;
    }
    /**
     * Return a parameterized function of a gaussian shape (see README for equation).
     * @param {number} x - x value to calculate.
     * @returns {number} - the y value of gaussian with the current parameters.
     */


    fct(x) {
      return Gaussian$2.fct(x, this.fwhm);
    }
    /**
     * Calculate the number of times FWHM allows to reach a specific area coverage
     * @param {number} [area=0.9999]
     * @returns {number}
     */


    getFactor(area = 0.9999) {
      return Gaussian$2.getFactor(area);
    }
    /**
     * Calculate the area of the shape.
     * @returns {number} - returns the area.
     */


    getArea() {
      return Gaussian$2.getArea(this.fwhm, {
        height: this.height
      });
    }
    /**
     * Compute the value of Full Width at Half Maximum (FWHM) from the width between the inflection points.
     * //https://mathworld.wolfram.com/GaussianFunction.html
     * @param {number} width - Width between the inflection points
     * @returns {number} fwhm
     */


    widthToFWHM(width) {
      //https://mathworld.wolfram.com/GaussianFunction.html
      return Gaussian$2.widthToFWHM(width);
    }
    /**
     * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
     * //https://mathworld.wolfram.com/GaussianFunction.html
     * @param {number} fwhm - Full Width at Half Maximum.
     * @returns {number} width
     */


    fwhmToWidth(fwhm = this.fwhm) {
      return Gaussian$2.fwhmToWidth(fwhm);
    }
    /**
     * set a new full width at half maximum
     * @param {number} fwhm - full width at half maximum
     */


    setFWHM(fwhm) {
      this.fwhm = fwhm;
    }
    /**
     * set a new height
     * @param {number} height - The maximal intensity of the shape.
     */


    setHeight(height) {
      this.height = height;
    }

  }
  /**
   * Return a parameterized function of a gaussian shape (see README for equation).
   * @param {number} x - x value to calculate.
   * @param {number} fwhm - full width half maximum
   * @returns {number} - the y value of gaussian with the current parameters.
   */

  Gaussian$2.fct = function fct(x, fwhm = 500) {
    return Math.exp(GAUSSIAN_EXP_FACTOR$2 * Math.pow(x / fwhm, 2));
  };
  /**
   * Compute the value of Full Width at Half Maximum (FWHM) from the width between the inflection points.
   * //https://mathworld.wolfram.com/GaussianFunction.html
   * @param {number} width - Width between the inflection points
   * @returns {number} fwhm
   */


  Gaussian$2.widthToFWHM = function widthToFWHM(width) {
    return width * ROOT_2LN2$2;
  };
  /**
   * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
   * //https://mathworld.wolfram.com/GaussianFunction.html
   * @param {number} fwhm - Full Width at Half Maximum.
   * @returns {number} width
   */


  Gaussian$2.fwhmToWidth = function fwhmToWidth(fwhm) {
    return fwhm / ROOT_2LN2$2;
  };
  /**
   * Calculate the area of a specific shape.
   * @param {number} fwhm - Full width at half maximum.
   * @param {object} [options = {}] - options.
   * @param {number} [options.height = 1] - Maximum y value of the shape.
   * @returns {number} - returns the area of the specific shape and parameters.
   */


  Gaussian$2.getArea = function getArea(fwhm, options = {}) {
    let {
      height = 1
    } = options;
    return height * ROOT_PI_OVER_LN2$2 * fwhm / 2;
  };
  /**
   * Calculate the number of times FWHM allows to reach a specific area coverage.
   * @param {number} [area=0.9999]
   * @returns {number}
   */


  Gaussian$2.getFactor = function getFactor(area = 0.9999) {
    return Math.sqrt(2) * erfinv$2(area);
  };

  class Lorentzian$2 {
    /**
     * @param {object} [options = {}]
     * @param {number} [options.height=2/(PI*FWHM)] Define the height of the peak, by default area=1 (normalized)
     * @param {number} [options.fwhm = 500] - Full Width at Half Maximum in the number of points in FWHM.
     * @param {number} [options.sd] - Standard deviation, if it's defined options.fwhm will be ignored and the value will be computed sd * Math.sqrt(8 * Math.LN2);
     */
    constructor(options = {}) {
      this.fwhm = options.fwhm === undefined ? 500 : options.fwhm;
      this.height = options.height === undefined ? 2 / Math.PI / this.fwhm : options.height;
    }
    /**
     * Calculate a lorentzian shape
     * @param {object} [options = {}]
     * @param {number} [options.factor = Math.tan(Math.PI * (0.9999 - 0.5))] - Number of time to take fwhm to calculate length. Default covers 99.99 % of area.
     * @param {number} [options.length = fwhm * factor + 1] - total number of points to calculate
     * @return {Float64Array} y values
     */


    getData(options = {}) {
      let {
        length,
        factor = this.getFactor()
      } = options;

      if (!length) {
        length = Math.min(Math.ceil(this.fwhm * factor), Math.pow(2, 25) - 1);
        if (length % 2 === 0) length++;
      }

      const center = (length - 1) / 2;
      const data = new Float64Array(length);

      for (let i = 0; i <= center; i++) {
        data[i] = this.fct(i - center) * this.height;
        data[length - 1 - i] = data[i];
      }

      return data;
    }
    /**
     * Return a parameterized function of a lorentzian shape (see README for equation).
     * @param {number} x - x value to calculate.
     * @returns {number} - the y value of lorentzian with the current parameters.
     */


    fct(x) {
      return Lorentzian$2.fct(x, this.fwhm);
    }
    /**
     * Calculate the number of times FWHM allows to reach a specific area coverage
     * @param {number} [area=0.9999]
     * @returns {number}
     */


    getFactor(area = 0.9999) {
      return Lorentzian$2.getFactor(area);
    }
    /**
     * Calculate the area of the shape.
     * @returns {number} - returns the area.
     */


    getArea() {
      return Lorentzian$2.getArea(this.fwhm, {
        height: this.height
      });
    }
    /**
     * Compute the value of width between the inflection points of a specific shape from Full Width at Half Maximum (FWHM).
     * //https://mathworld.wolfram.com/LorentzianFunction.html
     * @param {number} [fwhm] - Full Width at Half Maximum.
     * @returns {number} width between the inflection points
     */


    fwhmToWidth(fwhm = this.fwhm) {
      return Lorentzian$2.fwhmToWidth(fwhm);
    }
    /**
     * Compute the value of Full Width at Half Maximum (FWHM) of a specific shape from the width between the inflection points.
     * //https://mathworld.wolfram.com/LorentzianFunction.html
     * @param {number} [width] Width between the inflection points
     * @returns {number} fwhm
     */


    widthToFWHM(width) {
      return Lorentzian$2.widthToFWHM(width);
    }
    /**
     * set a new full width at half maximum
     * @param {number} fwhm - full width at half maximum
     */


    setFWHM(fwhm) {
      this.fwhm = fwhm;
    }
    /**
     * set a new height
     * @param {number} height - The maximal intensity of the shape.
     */


    setHeight(height) {
      this.height = height;
    }

  }
  /**
   * Return a parameterized function of a gaussian shape (see README for equation).
   * @param {number} x - x value to calculate.
   * @param {number} fwhm - full width half maximum
   * @returns {number} - the y value of gaussian with the current parameters.
   */

  Lorentzian$2.fct = function fct(x, fwhm) {
    const squareFWHM = fwhm * fwhm;
    return squareFWHM / (4 * Math.pow(x, 2) + squareFWHM);
  };
  /**
   * Compute the value of width between the inflection points of a specific shape from Full Width at Half Maximum (FWHM).
   * //https://mathworld.wolfram.com/LorentzianFunction.html
   * @param {number} [fwhm] - Full Width at Half Maximum.
   * @returns {number} width between the inflection points
   */


  Lorentzian$2.fwhmToWidth = function fwhmToWidth(fwhm) {
    return fwhm / ROOT_THREE$2;
  };
  /**
   * Compute the value of Full Width at Half Maximum (FWHM) of a specific shape from the width between the inflection points.
   * //https://mathworld.wolfram.com/LorentzianFunction.html
   * @param {number} [width] Width between the inflection points
   * @returns {number} fwhm
   */


  Lorentzian$2.widthToFWHM = function widthToFWHM(width) {
    return width * ROOT_THREE$2;
  };
  /**
   * Calculate the area of a specific shape.
   * @param {number} fwhm - Full width at half maximum.
   * @param {*} [options = {}] - options.
   * @param {number} [options.height = 1] - Maximum y value of the shape.
   * @returns {number} - returns the area of the specific shape and parameters.
   */


  Lorentzian$2.getArea = function getArea(fwhm, options = {}) {
    let {
      height = 1
    } = options;
    return height * Math.PI * fwhm / 2;
  };
  /**
   * Calculate the number of times FWHM allows to reach a specific area coverage
   * @param {number} [area=0.9999]
   * @returns {number}
   */


  Lorentzian$2.getFactor = function getFactor(area = 0.9999) {
    return 2 * Math.tan(Math.PI * (area - 0.5));
  };

  class PseudoVoigt$2 {
    /**
     * @param {object} [options={}]
     * @param {number} [options.height=1/(mu*FWHM/sqrt(4*LN2/PI)+(1-mu)*fwhm*PI*0.5)] Define the height of the peak, by default area=1 (normalized)
     * @param {number} [options.fwhm=500] - Full Width at Half Maximum in the number of points in FWHM.
     * @param {number} [options.mu=0.5] - ratio of gaussian contribution.
     */
    constructor(options = {}) {
      this.mu = options.mu === undefined ? 0.5 : options.mu;
      this.fwhm = options.fwhm === undefined ? 500 : options.fwhm;
      this.height = options.height === undefined ? 1 / (this.mu / Math.sqrt(-GAUSSIAN_EXP_FACTOR$2 / Math.PI) * this.fwhm + (1 - this.mu) * this.fwhm * Math.PI / 2) : options.height;
    }
    /**
     * Calculate a linear combination of gaussian and lorentzian function width an same full width at half maximum
     * @param { object } [options = {}]
     * @param { number } [options.factor = 2 * Math.tan(Math.PI * (0.9999 - 0.5))] - Number of time to take fwhm in the calculation of the length.Default covers 99.99 % of area.
     * @param { number } [options.length = fwhm * factor + 1] - total number of points to calculate
     * @return { object } - { fwhm, data<Float64Array>} - An with the number of points at half maximum and the array of y values covering the 99.99 % of the area.
     */


    getData(options = {}) {
      let {
        length,
        factor = this.getFactor()
      } = options;

      if (!length) {
        length = Math.ceil(this.fwhm * factor);
        if (length % 2 === 0) length++;
      }

      const center = (length - 1) / 2;
      let data = new Float64Array(length);

      for (let i = 0; i <= center; i++) {
        data[i] = this.fct(i - center) * this.height;
        data[length - 1 - i] = data[i];
      }

      return data;
    }
    /**
     * Return a parameterized function of a linear combination of Gaussian and Lorentzian shapes where the full width at half maximum are the same for both kind of shapes (see README for equation).
     * @param {number} [x] x value to calculate.
     * @returns {number} - the y value of a pseudo voigt with the current parameters.
     */


    fct(x) {
      return PseudoVoigt$2.fct(x, this.fwhm, this.mu);
    }
    /**
     * Calculate the number of times FWHM allows to reach a specific area coverage
     * @param {number} [area=0.9999] - required area to be coverage
     * @param {number} [mu=this.mu] - ratio of gaussian contribution.
     * @returns {number}
     */


    getFactor(area = 0.9999, mu = this.mu) {
      return PseudoVoigt$2.getFactor(area, mu);
    }
    /**
     * Calculate the area of the shape.
     * @returns {number} - returns the area.
     */


    getArea() {
      return PseudoVoigt$2.getArea(this.fwhm, {
        height: this.height,
        mu: this.mu
      });
    }
    /**
     * Compute the value of Full Width at Half Maximum (FMHM) from width between the inflection points.
     * @param {number} width - width between the inflection points
     * @param {number} [mu = 0.5] - ratio of gaussian contribution.
     * @returns {number} Full Width at Half Maximum (FMHM).
     */


    widthToFWHM(width, mu) {
      return PseudoVoigt$2.widthToFWHM(width, mu);
    }
    /**
     * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
     * @param {number} fwhm - Full Width at Half Maximum.
     * @param {number} [mu] - ratio of gaussian contribution.
     * @returns {number} width between the inflection points.
     */


    fwhmToWidth(fwhm = this.fwhm, mu = this.mu) {
      return PseudoVoigt$2.fwhmToWidth(fwhm, mu);
    }
    /**
     * set a new full width at half maximum
     * @param {number} fwhm - full width at half maximum
     */


    setFWHM(fwhm) {
      this.fwhm = fwhm;
    }
    /**
     * set a new height
     * @param {number} height - The maximal intensity of the shape.
     */


    setHeight(height) {
      this.height = height;
    }
    /**
     * set a new mu
     * @param {number} mu - ratio of gaussian contribution.
     */


    setMu(mu) {
      this.mu = mu;
    }

  }
  /**
   * Return a parameterized function of a gaussian shape (see README for equation).
   * @param {number} x - x value to calculate.
   * @param {number} fwhm - full width half maximum
   * @param {number} [mu=0.5] - ratio of gaussian contribution.
   * @returns {number} - the y value of gaussian with the current parameters.
   */

  PseudoVoigt$2.fct = function fct(x, fwhm, mu = 0.5) {
    return (1 - mu) * Lorentzian$2.fct(x, fwhm) + mu * Gaussian$2.fct(x, fwhm);
  };
  /**
   * Compute the value of Full Width at Half Maximum (FMHM) from width between the inflection points.
   * @param {number} width - width between the inflection points
   * @param {number} [mu = 0.5] - ratio of gaussian contribution.
   * @returns {number} Full Width at Half Maximum (FMHM).
   */


  PseudoVoigt$2.widthToFWHM = function widthToFWHM(width, mu = 0.5) {
    return width * (mu * ROOT_2LN2_MINUS_ONE$2 + 1);
  };
  /**
   * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
   * @param {number} fwhm - Full Width at Half Maximum.
   * @param {number} [mu = 0.5] - ratio of gaussian contribution.
   * @returns {number} width between the inflection points.
   */


  PseudoVoigt$2.fwhmToWidth = function fwhmToWidth(fwhm, mu = 0.5) {
    return fwhm / (mu * ROOT_2LN2_MINUS_ONE$2 + 1);
  };
  /**
   * Calculate the area of a specific shape.
   * @param {number} fwhm - Full width at half maximum.
   * @param {*} [options = {}] - options.
   * @param {number} [options.height = 1] - Maximum y value of the shape.
   * @param {number} [options.mu = 0.5] - ratio of gaussian contribution.
   * @returns {number} - returns the area of the specific shape and parameters.
   */


  PseudoVoigt$2.getArea = function getArea(fwhm, options = {}) {
    let {
      height = 1,
      mu = 0.5
    } = options;
    return fwhm * height * (mu * ROOT_PI_OVER_LN2$2 + (1 - mu) * Math.PI) / 2;
  };
  /**
   * Calculate the number of times FWHM allows to reach a specific area coverage
   * @param {number} [area=0.9999] - required area to be coverage
   * @param {number} [mu=this.mu] - ratio of gaussian contribution.
   * @returns {number}
   */


  PseudoVoigt$2.getFactor = function getFactor(area = 0.9999, mu = 0.5) {
    return mu < 1 ? Lorentzian$2.getFactor(area) : Gaussian$2.getFactor(area);
  };

  let axis$2 = ['x', 'y'];
  class Gaussian2D$2 {
    /**
     * @param {object} [options = {}]
     * @param {number} [options.height=4*LN2/(PI*xFWHM*yFWHM)] Define the height of the peak, by default area=1 (normalized).
     * @param {number} [options.fwhm = 500] - Full Width at Half Maximum in the number of points in FWHM used if x or y has not the fwhm property.
     * @param {object} [options.x] - Options for x axis.
     * @param {number} [options.x.fwhm = fwhm] - Full Width at Half Maximum in the number of points in FWHM for x axis.
     * @param {number} [options.x.sd] - Standard deviation for x axis, if it's defined options.x.fwhm will be ignored and the value will be computed sd * Math.sqrt(8 * Math.LN2);
     * @param {object} [options.y] - Options for y axis.
     * @param {number} [options.y.fwhm = fwhm] - Full Width at Half Maximum in the number of points in FWHM for y axis.
     * @param {number} [options.y.sd] - Standard deviation for y axis, if it's defined options.y.fwhm will be ignored and the value will be computed sd * Math.sqrt(8 * Math.LN2);
     */
    constructor(options = {}) {
      let {
        fwhm: globalFWHM = 500
      } = options;

      for (let i of axis$2) {
        let fwhm;

        if (!options[i]) {
          fwhm = globalFWHM;
        } else {
          fwhm = options[i].sd ? Gaussian2D$2.widthToFWHM(2 * options[i].sd) : options[i].fwhm || globalFWHM;
        }

        this[i] = {
          fwhm
        };
      }

      this.height = options.height === undefined ? -GAUSSIAN_EXP_FACTOR$2 / Math.PI / this.x.fwhm / this.y.fwhm : options.height;
    }
    /**
     * Calculate a Gaussian2D shape
     * @param {object} [options = {}]
     * @param {number} [options.factor] - Number of time to take fwhm to calculate length. Default covers 99.99 % of area.
     * @param {object} [options.x] - parameter for x axis.
     * @param {number} [options.x.length=fwhm*factor+1] - length on x axis.
     * @param {number} [options.x.factor=factor] - Number of time to take fwhm to calculate length. Default covers 99.99 % of area.
     * @param {object} [options.y] - parameter for y axis.
     * @param {number} [options.y.length=fwhm*factor+1] - length on y axis.
     * @param {number} [options.y.factor=factor] - Number of time to take fwhm to calculate length. Default covers 99.99 % of area.
     * @return {Array<Float64Array>} - z values.
     */


    getData(options = {}) {
      let {
        x = {},
        y = {},
        factor = this.getFactor(),
        length
      } = options;
      let xLength = x.length || length;

      if (!xLength) {
        let {
          factor: xFactor = factor
        } = x;
        xLength = Math.min(Math.ceil(this.x.fwhm * xFactor), Math.pow(2, 25) - 1);
        if (xLength % 2 === 0) xLength++;
      }

      let yLength = y.length || length;

      if (!yLength) {
        let {
          factor: yFactor = factor
        } = y;
        yLength = Math.min(Math.ceil(this.y.fwhm * yFactor), Math.pow(2, 25) - 1);
        if (yLength % 2 === 0) yLength++;
      }

      const xCenter = (xLength - 1) / 2;
      const yCenter = (yLength - 1) / 2;
      const data = new Array(xLength);

      for (let i = 0; i < xLength; i++) {
        data[i] = new Array(yLength);
      }

      for (let i = 0; i < xLength; i++) {
        for (let j = 0; j < yLength; j++) {
          data[i][j] = this.fct(i - xCenter, j - yCenter) * this.height;
        }
      }

      return data;
    }
    /**
     * Return the intensity value of a 2D gaussian shape (see README for equation).
     * @param {number} x - x value to calculate.
     * @param {number} y - y value to calculate.
     * @returns {number} - the z value of bi-dimensional gaussian with the current parameters.
     */


    fct(x, y) {
      return Gaussian2D$2.fct(x, y, this.x.fwhm, this.y.fwhm);
    }
    /**
     * Calculate the number of times FWHM allows to reach a specific volume coverage.
     * @param {number} [volume=0.9999]
     * @returns {number}
     */


    getFactor(volume = 0.9999) {
      return Gaussian2D$2.getFactor(volume);
    }
    /**
     * Calculate the volume of the shape.
     * @returns {number} - returns the volume.
     */


    getVolume() {
      return Gaussian2D$2.getVolume(this.x.fwhm, this.y.fwhm, {
        height: this.height
      });
    }
    /**
     * Compute the value of Full Width at Half Maximum (FWHM) from the width between the inflection points.
     * //https://mathworld.wolfram.com/Gaussian2DFunction.html
     * @param {number} width - Width between the inflection points
     * @returns {number} fwhm
     */


    widthToFWHM(width) {
      //https://mathworld.wolfram.com/Gaussian2DFunction.html
      return Gaussian2D$2.widthToFWHM(width);
    }
    /**
     * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
     * //https://mathworld.wolfram.com/Gaussian2DFunction.html
     * @param {number} fwhm - Full Width at Half Maximum.
     * @returns {number} width
     */


    fwhmToWidth(fwhm = this.x.fwhm) {
      return Gaussian2D$2.fwhmToWidth(fwhm);
    }
    /**
     * set a new full width at half maximum
     * @param {number} fwhm - full width at half maximum
     * @param {string|Array<string>} axisLabel - label of axis, if it is undefined fwhm is set to both axis.
     */


    setFWHM(fwhm, axisLabel) {
      if (!axisLabel) axisLabel = axis$2;
      if (!Array.isArray(axisLabel)) axisLabel = [axisLabel];

      for (let i of axisLabel) {
        let axisName = i.toLowerCase();

        if (axisName !== 'y' && axisName !== 'x') {
          throw new Error('axis label should be x or y');
        }

        this[axisName].fwhm = fwhm;
      }
    }
    /**
     * set a new height
     * @param {number} height - The maximal intensity of the shape.
     */


    setHeight(height) {
      this.height = height;
    }

  }
  /**
   * Return a parameterized function of a Gaussian2D shape (see README for equation).
   * @param {number} x - x value to calculate.
   * @param {number} y - y value to calculate.
   * @param {number} fwhmX - full width half maximum in the x axis.
   * @param {number} fwhmY - full width half maximum in the y axis.
   * @returns {number} - the z value of bi-dimensional gaussian with the current parameters.
   */

  Gaussian2D$2.fct = function fct(x, y, xFWHM = 500, yFWHM = 500) {
    return Math.exp(GAUSSIAN_EXP_FACTOR$2 * (Math.pow(x / xFWHM, 2) + Math.pow(y / yFWHM, 2)));
  };
  /**
   * Compute the value of Full Width at Half Maximum (FWHM) from the width between the inflection points.
   * //https://mathworld.wolfram.com/Gaussian2DFunction.html
   * @param {number} width - Width between the inflection points
   * @returns {number} fwhm
   */


  Gaussian2D$2.widthToFWHM = function widthToFWHM(width) {
    return width * ROOT_2LN2$2;
  };
  /**
   * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
   * //https://mathworld.wolfram.com/Gaussian2DFunction.html
   * @param {number} fwhm - Full Width at Half Maximum.
   * @returns {number} width
   */


  Gaussian2D$2.fwhmToWidth = function fwhmToWidth(fwhm) {
    return fwhm / ROOT_2LN2$2;
  };
  /**
   * Calculate the volume of a specific shape.
   * @param {number} xFWHM - Full width at half maximum for x axis.
   * @param {number} yFWHM - Full width at half maximum for y axis.
   * @param {object} [options = {}] - options.
   * @param {number} [options.height = 1] - Maximum z value of the shape.
   * @returns {number} - returns the area of the specific shape and parameters.
   */


  Gaussian2D$2.getVolume = function getVolume(xFWHM, yFWHM, options = {}) {
    let {
      height = 1
    } = options;
    return height * Math.PI * xFWHM * yFWHM / Math.LN2 / 4;
  };
  /**@TODO look for a better factor
   * Calculate the number of times FWHM allows to reach a specific volume coverage.
   * @param {number} [volume=0.9999]
   * @returns {number}
   */


  Gaussian2D$2.getFactor = function getFactor(volume = 0.9999) {
    return Math.sqrt(2) * erfinv$2(volume);
  };

  function getShapeGenerator$2(options) {
    let {
      kind = 'Gaussian',
      options: shapeOptions
    } = options;

    switch (kind.toLowerCase().replace(/[^a-z^0-9]/g, '')) {
      case 'gaussian':
        return new Gaussian$2(shapeOptions);

      case 'lorentzian':
        return new Lorentzian$2(shapeOptions);

      case 'pseudovoigt':
        return new PseudoVoigt$2(shapeOptions);

      case 'gaussian2d':
        return new Gaussian2D$2(shapeOptions);

      default:
        throw new Error(`Unknown kind: ${kind}`);
    }
  }

  /**
   * Apply Savitzky Golay algorithm
   * @param {array} [ys] Array of y values
   * @param {array|number} [xs] Array of X or deltaX
   * @param {object} [options={}]
   * @param {number} [options.windowSize=9]
   * @param {number} [options.derivative=0]
   * @param {number} [options.polynomial=3]
   * @return {array} Array containing the new ys (same length)
   */
  function SavitzkyGolay$1(ys, xs, options = {}) {
    let {
      windowSize = 9,
      derivative = 0,
      polynomial = 3
    } = options;

    if (windowSize % 2 === 0 || windowSize < 5 || !Number.isInteger(windowSize)) {
      throw new RangeError('Invalid window size (should be odd and at least 5 integer number)');
    }

    if (windowSize > ys.length) {
      throw new RangeError(`Window size is higher than the data length ${windowSize}>${ys.length}`);
    }

    if (derivative < 0 || !Number.isInteger(derivative)) {
      throw new RangeError('Derivative should be a positive integer');
    }

    if (polynomial < 1 || !Number.isInteger(polynomial)) {
      throw new RangeError('Polynomial should be a positive integer');
    }

    if (polynomial >= 6) {
      // eslint-disable-next-line no-console
      console.warn('You should not use polynomial grade higher than 5 if you are' + ' not sure that your data arises from such a model. Possible polynomial oscillation problems');
    }

    let half = Math.floor(windowSize / 2);
    let np = ys.length;
    let ans = new Array(np);
    let weights = fullWeights$1(windowSize, polynomial, derivative);
    let hs = 0;
    let constantH = true;

    if (Array.isArray(xs)) {
      constantH = false;
    } else {
      hs = Math.pow(xs, derivative);
    } //For the borders


    for (let i = 0; i < half; i++) {
      let wg1 = weights[half - i - 1];
      let wg2 = weights[half + i + 1];
      let d1 = 0;
      let d2 = 0;

      for (let l = 0; l < windowSize; l++) {
        d1 += wg1[l] * ys[l];
        d2 += wg2[l] * ys[np - windowSize + l];
      }

      if (constantH) {
        ans[half - i - 1] = d1 / hs;
        ans[np - half + i] = d2 / hs;
      } else {
        hs = getHs$1(xs, half - i - 1, half, derivative);
        ans[half - i - 1] = d1 / hs;
        hs = getHs$1(xs, np - half + i, half, derivative);
        ans[np - half + i] = d2 / hs;
      }
    } //For the internal points


    let wg = weights[half];

    for (let i = windowSize; i <= np; i++) {
      let d = 0;

      for (let l = 0; l < windowSize; l++) d += wg[l] * ys[l + i - windowSize];

      if (!constantH) hs = getHs$1(xs, i - half - 1, half, derivative);
      ans[i - half - 1] = d / hs;
    }

    return ans;
  }

  function getHs$1(h, center, half, derivative) {
    let hs = 0;
    let count = 0;

    for (let i = center - half; i < center + half; i++) {
      if (i >= 0 && i < h.length - 1) {
        hs += h[i + 1] - h[i];
        count++;
      }
    }

    return Math.pow(hs / count, derivative);
  }

  function GramPoly$1(i, m, k, s) {
    let Grampoly = 0;

    if (k > 0) {
      Grampoly = (4 * k - 2) / (k * (2 * m - k + 1)) * (i * GramPoly$1(i, m, k - 1, s) + s * GramPoly$1(i, m, k - 1, s - 1)) - (k - 1) * (2 * m + k) / (k * (2 * m - k + 1)) * GramPoly$1(i, m, k - 2, s);
    } else {
      if (k === 0 && s === 0) {
        Grampoly = 1;
      } else {
        Grampoly = 0;
      }
    }

    return Grampoly;
  }

  function GenFact$1(a, b) {
    let gf = 1;

    if (a >= b) {
      for (let j = a - b + 1; j <= a; j++) {
        gf *= j;
      }
    }

    return gf;
  }

  function Weight$1(i, t, m, n, s) {
    let sum = 0;

    for (let k = 0; k <= n; k++) {
      //console.log(k);
      sum += (2 * k + 1) * (GenFact$1(2 * m, k) / GenFact$1(2 * m + k + 1, k + 1)) * GramPoly$1(i, m, k, 0) * GramPoly$1(t, m, k, s);
    }

    return sum;
  }
  /**
   *
   * @param m  Number of points
   * @param n  Polynomial grade
   * @param s  Derivative
   */


  function fullWeights$1(m, n, s) {
    let weights = new Array(m);
    let np = Math.floor(m / 2);

    for (let t = -np; t <= np; t++) {
      weights[t + np] = new Array(m);

      for (let j = -np; j <= np; j++) {
        weights[t + np][j + np] = Weight$1(j, t, np, n, s);
      }
    }

    return weights;
  }
  /*function entropy(data,h,options){
      var trend = SavitzkyGolay(data,h,trendOptions);
      var copy = new Array(data.length);
      var sum = 0;
      var max = 0;
      for(var i=0;i<data.length;i++){
          copy[i] = data[i]-trend[i];
      }

      sum/=data.length;
      console.log(sum+" "+max);
      console.log(stat.array.standardDeviation(copy));
      console.log(Math.abs(stat.array.mean(copy))/stat.array.standardDeviation(copy));
      return sum;

  }



  function guessWindowSize(data, h){
      console.log("entropy "+entropy(data,h,trendOptions));
      return 5;
  }
  */

  /**
   * Global spectra deconvolution
   * @param {object} data - Object data with x and y arrays
   * @param {Array<number>} [data.x] - Independent variable
   * @param {Array<number>} [data.y] - Dependent variable
   * @param {object} [options={}] - Options object
   * @param {object} [options.shape={}] - Object that specified the kind of shape to calculate the FWHM instead of width between inflection points. see https://mljs.github.io/peak-shape-generator/#inflectionpointswidthtofwhm
   * @param {object} [options.shape.kind='gaussian']
   * @param {object} [options.shape.options={}]
   * @param {object} [options.sgOptions] - Options object for Savitzky-Golay filter. See https://github.com/mljs/savitzky-golay-generalized#options
   * @param {number} [options.sgOptions.windowSize = 9] - points to use in the approximations
   * @param {number} [options.sgOptions.polynomial = 3] - degree of the polynomial to use in the approximations
   * @param {number} [options.minMaxRatio = 0.00025] - Threshold to determine if a given peak should be considered as a noise
   * @param {number} [options.broadRatio = 0.00] - If `broadRatio` is higher than 0, then all the peaks which second derivative
   * smaller than `broadRatio * maxAbsSecondDerivative` will be marked with the soft mask equal to true.
   * @param {number} [options.noiseLevel = 0] - Noise threshold in spectrum units
   * @param {boolean} [options.maxCriteria = true] - Peaks are local maximum(true) or minimum(false)
   * @param {boolean} [options.smoothY = true] - Select the peak intensities from a smoothed version of the independent variables
   * @param {boolean} [options.realTopDetection = false] - Use a quadratic optimizations with the peak and its 3 closest neighbors
   * to determine the true x,y values of the peak?
   * @param {number} [options.heightFactor = 0] - Factor to multiply the calculated height (usually 2)
   * @param {number} [options.derivativeThreshold = -1] - Filters based on the amplitude of the first derivative
   * @return {Array<object>}
   */

  function gsd$2(data, options = {}) {
    let {
      noiseLevel,
      sgOptions = {
        windowSize: 9,
        polynomial: 3
      },
      shape = {},
      smoothY = true,
      heightFactor = 0,
      broadRatio = 0.0,
      maxCriteria = true,
      minMaxRatio = 0.00025,
      derivativeThreshold = -1,
      realTopDetection = false
    } = options;
    let {
      y: yIn,
      x
    } = data;
    const y = yIn.slice();
    let equalSpaced = isEqualSpaced$1(x);

    if (noiseLevel === undefined) {
      noiseLevel = equalSpaced ? getNoiseLevel$1(y) : 0;
    }

    const yCorrection = {
      m: 1,
      b: noiseLevel
    };

    if (!maxCriteria) {
      yCorrection.m = -1;
      yCorrection.b *= -1;
    }

    for (let i = 0; i < y.length; i++) {
      y[i] = yCorrection.m * y[i] - yCorrection.b;
    }

    for (let i = 0; i < y.length; i++) {
      if (y[i] < 0) {
        y[i] = 0;
      }
    } // If the max difference between delta x is less than 5%, then,
    // we can assume it to be equally spaced variable


    let yData = y;
    let dY, ddY;
    const {
      windowSize,
      polynomial
    } = sgOptions;

    if (equalSpaced) {
      if (smoothY) {
        yData = SavitzkyGolay$1(y, x[1] - x[0], {
          windowSize,
          polynomial,
          derivative: 0
        });
      }

      dY = SavitzkyGolay$1(y, x[1] - x[0], {
        windowSize,
        polynomial,
        derivative: 1
      });
      ddY = SavitzkyGolay$1(y, x[1] - x[0], {
        windowSize,
        polynomial,
        derivative: 2
      });
    } else {
      if (smoothY) {
        yData = SavitzkyGolay$1(y, x, {
          windowSize,
          polynomial,
          derivative: 0
        });
      }

      dY = SavitzkyGolay$1(y, x, {
        windowSize,
        polynomial,
        derivative: 1
      });
      ddY = SavitzkyGolay$1(y, x, {
        windowSize,
        polynomial,
        derivative: 2
      });
    }

    const xData = x;
    const dX = x[1] - x[0];
    let maxDdy = 0;
    let maxY = 0;

    for (let i = 0; i < yData.length; i++) {
      if (Math.abs(ddY[i]) > maxDdy) {
        maxDdy = Math.abs(ddY[i]);
      }

      if (Math.abs(yData[i]) > maxY) {
        maxY = Math.abs(yData[i]);
      }
    }

    let lastMax = null;
    let lastMin = null;
    let minddY = [];
    let intervalL = [];
    let intervalR = [];
    let broadMask = []; // By the intermediate value theorem We cannot find 2 consecutive maximum or minimum

    for (let i = 1; i < yData.length - 1; ++i) {
      // filter based on derivativeThreshold
      // console.log('pasa', y[i], dY[i], ddY[i]);
      if (Math.abs(dY[i]) > derivativeThreshold) {
        // Minimum in first derivative
        if (dY[i] < dY[i - 1] && dY[i] <= dY[i + 1] || dY[i] <= dY[i - 1] && dY[i] < dY[i + 1]) {
          lastMin = {
            x: xData[i],
            index: i
          };

          if (dX > 0 && lastMax !== null) {
            intervalL.push(lastMax);
            intervalR.push(lastMin);
          }
        } // Maximum in first derivative


        if (dY[i] >= dY[i - 1] && dY[i] > dY[i + 1] || dY[i] > dY[i - 1] && dY[i] >= dY[i + 1]) {
          lastMax = {
            x: xData[i],
            index: i
          };

          if (dX < 0 && lastMin !== null) {
            intervalL.push(lastMax);
            intervalR.push(lastMin);
          }
        }
      } // Minimum in second derivative


      if (ddY[i] < ddY[i - 1] && ddY[i] < ddY[i + 1]) {
        minddY.push(i);
        broadMask.push(Math.abs(ddY[i]) <= broadRatio * maxDdy);
      }
    }

    let widthProcessor = shape.kind ? getShapeGenerator$2(shape.kind, shape.options).widthToFWHM : x => x;
    let signals = [];
    let lastK = -1;
    let possible, frequency, distanceJ, minDistance, gettingCloser;

    for (let j = 0; j < minddY.length; ++j) {
      frequency = xData[minddY[j]];
      possible = -1;
      let k = lastK + 1;
      minDistance = Number.MAX_VALUE;
      distanceJ = 0;
      gettingCloser = true;

      while (possible === -1 && k < intervalL.length && gettingCloser) {
        distanceJ = Math.abs(frequency - (intervalL[k].x + intervalR[k].x) / 2); // Still getting closer?

        if (distanceJ < minDistance) {
          minDistance = distanceJ;
        } else {
          gettingCloser = false;
        }

        if (distanceJ < Math.abs(intervalL[k].x - intervalR[k].x) / 2) {
          possible = k;
          lastK = k;
        }

        ++k;
      }

      if (possible !== -1) {
        if (Math.abs(yData[minddY[j]]) > minMaxRatio * maxY) {
          let width = Math.abs(intervalR[possible].x - intervalL[possible].x);
          signals.push({
            index: minddY[j],
            x: frequency,
            y: (yData[minddY[j]] + yCorrection.b) / yCorrection.m,
            width: widthProcessor(width),
            soft: broadMask[j]
          });
          signals[signals.length - 1].left = intervalL[possible];
          signals[signals.length - 1].right = intervalR[possible];

          if (heightFactor) {
            let yLeft = yData[intervalL[possible].index];
            let yRight = yData[intervalR[possible].index];
            signals[signals.length - 1].height = heightFactor * (signals[signals.length - 1].y - (yLeft + yRight) / 2);
          }
        }
      }
    }

    if (realTopDetection) {
      determineRealTop$1(signals, xData, yData);
    } // Correct the values to fit the original spectra data


    for (let j = 0; j < signals.length; j++) {
      signals[j].base = noiseLevel;
    }

    signals.sort(function (a, b) {
      return a.x - b.x;
    });
    return signals;
  }

  const isEqualSpaced$1 = x => {
    let tmp;
    let maxDx = 0;
    let minDx = Number.MAX_SAFE_INTEGER;

    for (let i = 0; i < x.length - 1; ++i) {
      tmp = Math.abs(x[i + 1] - x[i]);

      if (tmp < minDx) {
        minDx = tmp;
      }

      if (tmp > maxDx) {
        maxDx = tmp;
      }
    }

    return (maxDx - minDx) / maxDx < 0.05;
  };

  const getNoiseLevel$1 = y => {
    let mean = 0;
    let stddev = 0;
    let length = y.length;

    for (let i = 0; i < length; ++i) {
      mean += y[i];
    }

    mean /= length;
    let averageDeviations = new Array(length);

    for (let i = 0; i < length; ++i) {
      averageDeviations[i] = Math.abs(y[i] - mean);
    }

    averageDeviations.sort((a, b) => a - b);

    if (length % 2 === 1) {
      stddev = averageDeviations[(length - 1) / 2] / 0.6745;
    } else {
      stddev = 0.5 * (averageDeviations[length / 2] + averageDeviations[length / 2 - 1]) / 0.6745;
    }

    return stddev;
  };

  const determineRealTop$1 = (peakList, x, y) => {
    let alpha, beta, gamma, p, currentPoint;

    for (let j = 0; j < peakList.length; j++) {
      currentPoint = peakList[j].index; // peakList[j][2];
      // The detected peak could be moved 1 or 2 units to left or right.

      if (y[currentPoint - 1] >= y[currentPoint - 2] && y[currentPoint - 1] >= y[currentPoint]) {
        currentPoint--;
      } else {
        if (y[currentPoint + 1] >= y[currentPoint] && y[currentPoint + 1] >= y[currentPoint + 2]) {
          currentPoint++;
        } else {
          if (y[currentPoint - 2] >= y[currentPoint - 3] && y[currentPoint - 2] >= y[currentPoint - 1]) {
            currentPoint -= 2;
          } else {
            if (y[currentPoint + 2] >= y[currentPoint + 1] && y[currentPoint + 2] >= y[currentPoint + 3]) {
              currentPoint += 2;
            }
          }
        }
      } // interpolation to a sin() function


      if (y[currentPoint - 1] > 0 && y[currentPoint + 1] > 0 && y[currentPoint] >= y[currentPoint - 1] && y[currentPoint] >= y[currentPoint + 1] && (y[currentPoint] !== y[currentPoint - 1] || y[currentPoint] !== y[currentPoint + 1])) {
        alpha = 20 * Math.log10(y[currentPoint - 1]);
        beta = 20 * Math.log10(y[currentPoint]);
        gamma = 20 * Math.log10(y[currentPoint + 1]);
        p = 0.5 * (alpha - gamma) / (alpha - 2 * beta + gamma); // console.log(alpha, beta, gamma, `p: ${p}`);
        // console.log(x[currentPoint]+" "+tmp+" "+currentPoint);

        peakList[j].x = x[currentPoint] + (x[currentPoint] - x[currentPoint - 1]) * p;
        peakList[j].y = y[currentPoint] - 0.25 * (y[currentPoint - 1] - y[currentPoint + 1]) * p;
      }
    }
  };

  /*!
   * assign-symbols <https://github.com/jonschlinkert/assign-symbols>
   *
   * Copyright (c) 2015-present, Jon Schlinkert.
   * Licensed under the MIT License.
   */

  const toString$9 = Object.prototype.toString;
  const isEnumerable = Object.prototype.propertyIsEnumerable;
  const getSymbols = Object.getOwnPropertySymbols;

  var assignSymbols = (target, ...args) => {
    if (!isObject$1(target)) {
      throw new TypeError('expected the first argument to be an object');
    }

    if (args.length === 0 || typeof Symbol !== 'function' || typeof getSymbols !== 'function') {
      return target;
    }

    for (let arg of args) {
      let names = getSymbols(arg);

      for (let key of names) {
        if (isEnumerable.call(arg, key)) {
          target[key] = arg[key];
        }
      }
    }

    return target;
  };

  function isObject$1(val) {
    return typeof val === 'function' || toString$9.call(val) === '[object Object]' || Array.isArray(val);
  }

  /*!
   * assign-deep <https://github.com/jonschlinkert/assign-deep>
   *
   * Copyright (c) 2017-present, Jon Schlinkert.
   * Released under the MIT License.
   */
  var assignDeep = createCommonjsModule$1(function (module) {

    const toString = Object.prototype.toString;

    const isValidKey = key => {
      return key !== '__proto__' && key !== 'constructor' && key !== 'prototype';
    };

    const assign = module.exports = (target, ...args) => {
      let i = 0;
      if (isPrimitive(target)) target = args[i++];
      if (!target) target = {};

      for (; i < args.length; i++) {
        if (isObject(args[i])) {
          for (const key of Object.keys(args[i])) {
            if (isValidKey(key)) {
              if (isObject(target[key]) && isObject(args[i][key])) {
                assign(target[key], args[i][key]);
              } else {
                target[key] = args[i][key];
              }
            }
          }

          assignSymbols(target, args[i]);
        }
      }

      return target;
    };

    function isObject(val) {
      return typeof val === 'function' || toString.call(val) === '[object Object]';
    }

    function isPrimitive(val) {
      return typeof val === 'object' ? val === null : typeof val !== 'function';
    }
  });

  /**
   * This function calculates the spectrum as a sum of linear combination of gaussian and lorentzian functions. The pseudo voigt
   * parameters are divided in 4 batches. 1st: centers; 2nd: heights; 3th: widths; 4th: mu's ;
   * @param t Ordinate value
   * @param p Lorentzian parameters
   * @returns {*}
   */

  function sumOfGaussianLorentzians(p) {
    return function (t) {
      let nL = p.length / 4;
      let result = 0;

      for (let i = 0; i < nL; i++) {
        result += p[i + nL] * PseudoVoigt$2.fct(t - p[i], p[i + nL * 2], p[i + nL * 3]);
      }

      return result;
    };
  }

  /**
   * This function calculates the spectrum as a sum of gaussian functions. The Gaussian
   * parameters are divided in 3 batches. 1st: centers; 2nd: height; 3th: widths;
   * @param t Ordinate values
   * @param p Gaussian parameters
   * @returns {*}
   */

  function sumOfGaussians(p) {
    return function (t) {
      let nL = p.length / 3;
      let result = 0;

      for (let i = 0; i < nL; i++) {
        result += p[i + nL] * Gaussian$2.fct(t - p[i], p[i + nL * 2]);
      }

      return result;
    };
  }

  /**
   * This function calculates the spectrum as a sum of lorentzian functions. The Lorentzian
   * parameters are divided in 3 batches. 1st: centers; 2nd: heights; 3th: widths;
   * @param t Ordinate values
   * @param p Lorentzian parameters
   * @returns {*}
   */

  function sumOfLorentzians(p) {
    return function (t) {
      let nL = p.length / 3;
      let result = 0;

      for (let i = 0; i < nL; i++) {
        result += p[i + nL] * Lorentzian$2.fct(t - p[i], p[i + nL * 2]);
      }

      return result;
    };
  }

  function checkInput(data, peaks, options) {
    let {
      shape = {
        kind: 'gaussian'
      },
      optimization = {
        kind: 'lm'
      }
    } = options;

    if (typeof shape.kind !== 'string') {
      throw new Error('kind should be a string');
    }

    let kind = shape.kind.toLowerCase().replace(/[^a-z]/g, '');
    let paramsFunc;
    let defaultParameters;

    switch (kind) {
      case 'gaussian':
        paramsFunc = sumOfGaussians;
        defaultParameters = {
          x: {
            init: peak => peak.x,
            max: peak => peak.x + peak.width * 2,
            min: peak => peak.x - peak.width * 2,
            gradientDifference: peak => peak.width * 2e-3
          },
          y: {
            init: peak => peak.y,
            max: () => 1.5,
            min: () => 0,
            gradientDifference: () => 1e-3
          },
          width: {
            init: peak => peak.width,
            max: peak => peak.width * 4,
            min: peak => peak.width * 0.25,
            gradientDifference: peak => peak.width * 2e-3
          }
        };
        break;

      case 'lorentzian':
        paramsFunc = sumOfLorentzians;
        defaultParameters = {
          x: {
            init: peak => peak.x,
            max: peak => peak.x + peak.width * 2,
            min: peak => peak.x - peak.width * 2,
            gradientDifference: peak => peak.width * 2e-3
          },
          y: {
            init: peak => peak.y,
            max: () => 1.5,
            min: () => 0,
            gradientDifference: () => 1e-3
          },
          width: {
            init: peak => peak.width,
            max: peak => peak.width * 4,
            min: peak => peak.width * 0.25,
            gradientDifference: peak => peak.width * 2e-3
          }
        };
        break;

      case 'pseudovoigt':
        paramsFunc = sumOfGaussianLorentzians;
        defaultParameters = {
          x: {
            init: peak => peak.x,
            max: peak => peak.x + peak.width * 2,
            min: peak => peak.x - peak.width * 2,
            gradientDifference: peak => peak.width * 2e-3
          },
          y: {
            init: peak => peak.y,
            max: () => 1.5,
            min: () => 0,
            gradientDifference: () => 1e-3
          },
          width: {
            init: peak => peak.width,
            max: peak => peak.width * 4,
            min: peak => peak.width * 0.25,
            gradientDifference: peak => peak.width * 2e-3
          },
          mu: {
            init: peak => peak.mu !== undefined ? peak.mu : 0.5,
            min: () => 0,
            max: () => 1,
            gradientDifference: () => 0.01
          }
        };
        break;

      default:
        throw new Error('kind of shape is not supported');
    }

    let x = data.x;
    let maxY = max$7(data.y);
    let y = new Array(x.length);

    for (let i = 0; i < x.length; i++) {
      y[i] = data.y[i] / maxY;
    }

    for (let i = 0; i < peaks.length; i++) {
      peaks[i].y /= maxY;
    }

    let parameters = assignDeep({}, optimization.parameters, defaultParameters);

    for (let key in parameters) {
      for (let par in parameters[key]) {
        if (!Array.isArray(parameters[key][par])) {
          parameters[key][par] = [parameters[key][par]];
        }

        if (parameters[key][par].length !== 1 && parameters[key][par].length !== peaks.length) {
          throw new Error(`The length of ${key}-${par} is not correct`);
        }

        for (let index = 0; index < parameters[key][par].length; index++) {
          if (typeof parameters[key][par][index] === 'number') {
            let value = parameters[key][par][index];

            parameters[key][par][index] = () => value;
          }
        }
      }
    }

    optimization.parameters = parameters;
    return {
      y,
      x,
      maxY,
      peaks,
      paramsFunc,
      optimization
    };
  }

  function checkOptions$1(data, parameterizedFunction, options) {
    let {
      timeout,
      minValues,
      maxValues,
      initialValues,
      weights = 1,
      damping = 1e-2,
      dampingStepUp = 11,
      dampingStepDown = 9,
      maxIterations = 100,
      errorTolerance = 1e-7,
      centralDifference = false,
      gradientDifference = 10e-2,
      improvementThreshold = 1e-3
    } = options;

    if (damping <= 0) {
      throw new Error('The damping option must be a positive number');
    } else if (!data.x || !data.y) {
      throw new Error('The data parameter must have x and y elements');
    } else if (!isAnyArray$7(data.x) || data.x.length < 2 || !isAnyArray$7(data.y) || data.y.length < 2) {
      throw new Error('The data parameter elements must be an array with more than 2 points');
    } else if (data.x.length !== data.y.length) {
      throw new Error('The data parameter elements must have the same size');
    }

    let parameters = initialValues || new Array(parameterizedFunction.length).fill(1);
    let nbPoints = data.y.length;
    let parLen = parameters.length;
    maxValues = maxValues || new Array(parLen).fill(Number.MAX_SAFE_INTEGER);
    minValues = minValues || new Array(parLen).fill(Number.MIN_SAFE_INTEGER);

    if (maxValues.length !== minValues.length) {
      throw new Error('minValues and maxValues must be the same size');
    }

    if (!isAnyArray$7(parameters)) {
      throw new Error('initialValues must be an array');
    }

    if (typeof gradientDifference === 'number') {
      gradientDifference = new Array(parameters.length).fill(gradientDifference);
    } else if (isAnyArray$7(gradientDifference)) {
      if (gradientDifference.length !== parLen) {
        gradientDifference = new Array(parLen).fill(gradientDifference[0]);
      }
    } else {
      throw new Error('gradientDifference should be a number or array with length equal to the number of parameters');
    }

    let filler;

    if (typeof weights === 'number') {
      let value = 1 / weights ** 2;

      filler = () => value;
    } else if (isAnyArray$7(weights)) {
      if (weights.length < data.x.length) {
        let value = 1 / weights[0] ** 2;

        filler = () => value;
      } else {
        filler = i => 1 / weights[i] ** 2;
      }
    } else {
      throw new Error('weights should be a number or array with length equal to the number of data points');
    }

    let checkTimeout;

    if (timeout !== undefined) {
      if (typeof timeout !== 'number') {
        throw new Error('timeout should be a number');
      }

      let endTime = Date.now() + timeout * 1000;

      checkTimeout = () => Date.now() > endTime;
    } else {
      checkTimeout = () => false;
    }

    let weightSquare = new Array(data.x.length);

    for (let i = 0; i < nbPoints; i++) {
      weightSquare[i] = filler(i);
    }

    return {
      checkTimeout,
      minValues,
      maxValues,
      parameters,
      weightSquare,
      damping,
      dampingStepUp,
      dampingStepDown,
      maxIterations,
      errorTolerance,
      centralDifference,
      gradientDifference,
      improvementThreshold
    };
  }

  /**
   * the sum of the weighted squares of the errors (or weighted residuals) between the data.y
   * and the curve-fit function.
   * @ignore
   * @param {{x:Array<number>, y:Array<number>}} data - Array of points to fit in the format [x1, x2, ... ], [y1, y2, ... ]
   * @param {Array<number>} parameters - Array of current parameter values
   * @param {function} parameterizedFunction - The parameters and returns a function with the independent variable as a parameter
   * @param {Array} weightSquare - Square of weights
   * @return {number}
   */
  function errorCalculation(data, parameters, parameterizedFunction, weightSquare) {
    let error = 0;
    const func = parameterizedFunction(parameters);

    for (let i = 0; i < data.x.length; i++) {
      error += Math.pow(data.y[i] - func(data.x[i]), 2) / weightSquare[i];
    }

    return error;
  }

  function rescale$1(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (!isAnyArray$7(input)) {
      throw new TypeError('input must be an array');
    } else if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    var output;

    if (options.output !== undefined) {
      if (!isAnyArray$7(options.output)) {
        throw new TypeError('output option must be an array if specified');
      }

      output = options.output;
    } else {
      output = new Array(input.length);
    }

    var currentMin = min$8(input);
    var currentMax = max$7(input);

    if (currentMin === currentMax) {
      throw new RangeError('minimum and maximum input values are equal. Cannot rescale a constant array');
    }

    var _options$min = options.min,
        minValue = _options$min === void 0 ? options.autoMinMax ? currentMin : 0 : _options$min,
        _options$max = options.max,
        maxValue = _options$max === void 0 ? options.autoMinMax ? currentMax : 1 : _options$max;

    if (minValue >= maxValue) {
      throw new RangeError('min option must be smaller than max option');
    }

    var factor = (maxValue - minValue) / (currentMax - currentMin);

    for (var i = 0; i < input.length; i++) {
      output[i] = (input[i] - currentMin) * factor + minValue;
    }

    return output;
  }

  const indent$1 = ' '.repeat(2);
  const indentData$1 = ' '.repeat(4);
  function inspectMatrix$1() {
    return inspectMatrixWithOptions$1(this);
  }
  function inspectMatrixWithOptions$1(matrix, options = {}) {
    const {
      maxRows = 15,
      maxColumns = 10,
      maxNumSize = 8
    } = options;
    return `${matrix.constructor.name} {
${indent$1}[
${indentData$1}${inspectData$1(matrix, maxRows, maxColumns, maxNumSize)}
${indent$1}]
${indent$1}rows: ${matrix.rows}
${indent$1}columns: ${matrix.columns}
}`;
  }

  function inspectData$1(matrix, maxRows, maxColumns, maxNumSize) {
    const {
      rows,
      columns
    } = matrix;
    const maxI = Math.min(rows, maxRows);
    const maxJ = Math.min(columns, maxColumns);
    const result = [];

    for (let i = 0; i < maxI; i++) {
      let line = [];

      for (let j = 0; j < maxJ; j++) {
        line.push(formatNumber$1(matrix.get(i, j), maxNumSize));
      }

      result.push(`${line.join(' ')}`);
    }

    if (maxJ !== columns) {
      result[result.length - 1] += ` ... ${columns - maxColumns} more columns`;
    }

    if (maxI !== rows) {
      result.push(`... ${rows - maxRows} more rows`);
    }

    return result.join(`\n${indentData$1}`);
  }

  function formatNumber$1(num, maxNumSize) {
    const numStr = String(num);

    if (numStr.length <= maxNumSize) {
      return numStr.padEnd(maxNumSize, ' ');
    }

    const precise = num.toPrecision(maxNumSize - 2);

    if (precise.length <= maxNumSize) {
      return precise;
    }

    const exponential = num.toExponential(maxNumSize - 2);
    const eIndex = exponential.indexOf('e');
    const e = exponential.slice(eIndex);
    return exponential.slice(0, maxNumSize - e.length) + e;
  }

  function installMathOperations$1(AbstractMatrix, Matrix) {
    AbstractMatrix.prototype.add = function add(value) {
      if (typeof value === 'number') return this.addS(value);
      return this.addM(value);
    };

    AbstractMatrix.prototype.addS = function addS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.addM = function addM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.add = function add(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.add(value);
    };

    AbstractMatrix.prototype.sub = function sub(value) {
      if (typeof value === 'number') return this.subS(value);
      return this.subM(value);
    };

    AbstractMatrix.prototype.subS = function subS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.subM = function subM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.sub = function sub(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sub(value);
    };

    AbstractMatrix.prototype.subtract = AbstractMatrix.prototype.sub;
    AbstractMatrix.prototype.subtractS = AbstractMatrix.prototype.subS;
    AbstractMatrix.prototype.subtractM = AbstractMatrix.prototype.subM;
    AbstractMatrix.subtract = AbstractMatrix.sub;

    AbstractMatrix.prototype.mul = function mul(value) {
      if (typeof value === 'number') return this.mulS(value);
      return this.mulM(value);
    };

    AbstractMatrix.prototype.mulS = function mulS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.mulM = function mulM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.mul = function mul(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.mul(value);
    };

    AbstractMatrix.prototype.multiply = AbstractMatrix.prototype.mul;
    AbstractMatrix.prototype.multiplyS = AbstractMatrix.prototype.mulS;
    AbstractMatrix.prototype.multiplyM = AbstractMatrix.prototype.mulM;
    AbstractMatrix.multiply = AbstractMatrix.mul;

    AbstractMatrix.prototype.div = function div(value) {
      if (typeof value === 'number') return this.divS(value);
      return this.divM(value);
    };

    AbstractMatrix.prototype.divS = function divS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.divM = function divM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.div = function div(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.div(value);
    };

    AbstractMatrix.prototype.divide = AbstractMatrix.prototype.div;
    AbstractMatrix.prototype.divideS = AbstractMatrix.prototype.divS;
    AbstractMatrix.prototype.divideM = AbstractMatrix.prototype.divM;
    AbstractMatrix.divide = AbstractMatrix.div;

    AbstractMatrix.prototype.mod = function mod(value) {
      if (typeof value === 'number') return this.modS(value);
      return this.modM(value);
    };

    AbstractMatrix.prototype.modS = function modS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) % value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.modM = function modM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) % matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.mod = function mod(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.mod(value);
    };

    AbstractMatrix.prototype.modulus = AbstractMatrix.prototype.mod;
    AbstractMatrix.prototype.modulusS = AbstractMatrix.prototype.modS;
    AbstractMatrix.prototype.modulusM = AbstractMatrix.prototype.modM;
    AbstractMatrix.modulus = AbstractMatrix.mod;

    AbstractMatrix.prototype.and = function and(value) {
      if (typeof value === 'number') return this.andS(value);
      return this.andM(value);
    };

    AbstractMatrix.prototype.andS = function andS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) & value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.andM = function andM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) & matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.and = function and(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.and(value);
    };

    AbstractMatrix.prototype.or = function or(value) {
      if (typeof value === 'number') return this.orS(value);
      return this.orM(value);
    };

    AbstractMatrix.prototype.orS = function orS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) | value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.orM = function orM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) | matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.or = function or(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.or(value);
    };

    AbstractMatrix.prototype.xor = function xor(value) {
      if (typeof value === 'number') return this.xorS(value);
      return this.xorM(value);
    };

    AbstractMatrix.prototype.xorS = function xorS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) ^ value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.xorM = function xorM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) ^ matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.xor = function xor(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.xor(value);
    };

    AbstractMatrix.prototype.leftShift = function leftShift(value) {
      if (typeof value === 'number') return this.leftShiftS(value);
      return this.leftShiftM(value);
    };

    AbstractMatrix.prototype.leftShiftS = function leftShiftS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) << value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.leftShiftM = function leftShiftM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) << matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.leftShift = function leftShift(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.leftShift(value);
    };

    AbstractMatrix.prototype.signPropagatingRightShift = function signPropagatingRightShift(value) {
      if (typeof value === 'number') return this.signPropagatingRightShiftS(value);
      return this.signPropagatingRightShiftM(value);
    };

    AbstractMatrix.prototype.signPropagatingRightShiftS = function signPropagatingRightShiftS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >> value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.signPropagatingRightShiftM = function signPropagatingRightShiftM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >> matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.signPropagatingRightShift = function signPropagatingRightShift(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.signPropagatingRightShift(value);
    };

    AbstractMatrix.prototype.rightShift = function rightShift(value) {
      if (typeof value === 'number') return this.rightShiftS(value);
      return this.rightShiftM(value);
    };

    AbstractMatrix.prototype.rightShiftS = function rightShiftS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >>> value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.rightShiftM = function rightShiftM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >>> matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.rightShift = function rightShift(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.rightShift(value);
    };

    AbstractMatrix.prototype.zeroFillRightShift = AbstractMatrix.prototype.rightShift;
    AbstractMatrix.prototype.zeroFillRightShiftS = AbstractMatrix.prototype.rightShiftS;
    AbstractMatrix.prototype.zeroFillRightShiftM = AbstractMatrix.prototype.rightShiftM;
    AbstractMatrix.zeroFillRightShift = AbstractMatrix.rightShift;

    AbstractMatrix.prototype.not = function not() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, ~this.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.not = function not(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.not();
    };

    AbstractMatrix.prototype.abs = function abs() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.abs(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.abs = function abs(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.abs();
    };

    AbstractMatrix.prototype.acos = function acos() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.acos(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.acos = function acos(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.acos();
    };

    AbstractMatrix.prototype.acosh = function acosh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.acosh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.acosh = function acosh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.acosh();
    };

    AbstractMatrix.prototype.asin = function asin() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.asin(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.asin = function asin(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.asin();
    };

    AbstractMatrix.prototype.asinh = function asinh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.asinh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.asinh = function asinh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.asinh();
    };

    AbstractMatrix.prototype.atan = function atan() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.atan(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.atan = function atan(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.atan();
    };

    AbstractMatrix.prototype.atanh = function atanh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.atanh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.atanh = function atanh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.atanh();
    };

    AbstractMatrix.prototype.cbrt = function cbrt() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.cbrt(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.cbrt = function cbrt(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.cbrt();
    };

    AbstractMatrix.prototype.ceil = function ceil() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.ceil(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.ceil = function ceil(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.ceil();
    };

    AbstractMatrix.prototype.clz32 = function clz32() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.clz32(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.clz32 = function clz32(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.clz32();
    };

    AbstractMatrix.prototype.cos = function cos() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.cos(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.cos = function cos(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.cos();
    };

    AbstractMatrix.prototype.cosh = function cosh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.cosh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.cosh = function cosh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.cosh();
    };

    AbstractMatrix.prototype.exp = function exp() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.exp(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.exp = function exp(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.exp();
    };

    AbstractMatrix.prototype.expm1 = function expm1() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.expm1(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.expm1 = function expm1(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.expm1();
    };

    AbstractMatrix.prototype.floor = function floor() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.floor(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.floor = function floor(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.floor();
    };

    AbstractMatrix.prototype.fround = function fround() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.fround(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.fround = function fround(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.fround();
    };

    AbstractMatrix.prototype.log = function log() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.log = function log(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log();
    };

    AbstractMatrix.prototype.log1p = function log1p() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log1p(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.log1p = function log1p(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log1p();
    };

    AbstractMatrix.prototype.log10 = function log10() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log10(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.log10 = function log10(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log10();
    };

    AbstractMatrix.prototype.log2 = function log2() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log2(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.log2 = function log2(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log2();
    };

    AbstractMatrix.prototype.round = function round() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.round(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.round = function round(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.round();
    };

    AbstractMatrix.prototype.sign = function sign() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sign(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.sign = function sign(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sign();
    };

    AbstractMatrix.prototype.sin = function sin() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sin(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.sin = function sin(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sin();
    };

    AbstractMatrix.prototype.sinh = function sinh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sinh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.sinh = function sinh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sinh();
    };

    AbstractMatrix.prototype.sqrt = function sqrt() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sqrt(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.sqrt = function sqrt(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sqrt();
    };

    AbstractMatrix.prototype.tan = function tan() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.tan(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.tan = function tan(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.tan();
    };

    AbstractMatrix.prototype.tanh = function tanh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.tanh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.tanh = function tanh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.tanh();
    };

    AbstractMatrix.prototype.trunc = function trunc() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.trunc(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.trunc = function trunc(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.trunc();
    };

    AbstractMatrix.pow = function pow(matrix, arg0) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.pow(arg0);
    };

    AbstractMatrix.prototype.pow = function pow(value) {
      if (typeof value === 'number') return this.powS(value);
      return this.powM(value);
    };

    AbstractMatrix.prototype.powS = function powS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.pow(this.get(i, j), value));
        }
      }

      return this;
    };

    AbstractMatrix.prototype.powM = function powM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.pow(this.get(i, j), matrix.get(i, j)));
        }
      }

      return this;
    };
  }

  /**
   * @private
   * Check that a row index is not out of bounds
   * @param {Matrix} matrix
   * @param {number} index
   * @param {boolean} [outer]
   */
  function checkRowIndex$1(matrix, index, outer) {
    let max = outer ? matrix.rows : matrix.rows - 1;

    if (index < 0 || index > max) {
      throw new RangeError('Row index out of range');
    }
  }
  /**
   * @private
   * Check that a column index is not out of bounds
   * @param {Matrix} matrix
   * @param {number} index
   * @param {boolean} [outer]
   */

  function checkColumnIndex$1(matrix, index, outer) {
    let max = outer ? matrix.columns : matrix.columns - 1;

    if (index < 0 || index > max) {
      throw new RangeError('Column index out of range');
    }
  }
  /**
   * @private
   * Check that the provided vector is an array with the right length
   * @param {Matrix} matrix
   * @param {Array|Matrix} vector
   * @return {Array}
   * @throws {RangeError}
   */

  function checkRowVector$1(matrix, vector) {
    if (vector.to1DArray) {
      vector = vector.to1DArray();
    }

    if (vector.length !== matrix.columns) {
      throw new RangeError('vector size must be the same as the number of columns');
    }

    return vector;
  }
  /**
   * @private
   * Check that the provided vector is an array with the right length
   * @param {Matrix} matrix
   * @param {Array|Matrix} vector
   * @return {Array}
   * @throws {RangeError}
   */

  function checkColumnVector$1(matrix, vector) {
    if (vector.to1DArray) {
      vector = vector.to1DArray();
    }

    if (vector.length !== matrix.rows) {
      throw new RangeError('vector size must be the same as the number of rows');
    }

    return vector;
  }
  function checkIndices$1(matrix, rowIndices, columnIndices) {
    return {
      row: checkRowIndices$1(matrix, rowIndices),
      column: checkColumnIndices$1(matrix, columnIndices)
    };
  }
  function checkRowIndices$1(matrix, rowIndices) {
    if (typeof rowIndices !== 'object') {
      throw new TypeError('unexpected type for row indices');
    }

    let rowOut = rowIndices.some(r => {
      return r < 0 || r >= matrix.rows;
    });

    if (rowOut) {
      throw new RangeError('row indices are out of range');
    }

    if (!Array.isArray(rowIndices)) rowIndices = Array.from(rowIndices);
    return rowIndices;
  }
  function checkColumnIndices$1(matrix, columnIndices) {
    if (typeof columnIndices !== 'object') {
      throw new TypeError('unexpected type for column indices');
    }

    let columnOut = columnIndices.some(c => {
      return c < 0 || c >= matrix.columns;
    });

    if (columnOut) {
      throw new RangeError('column indices are out of range');
    }

    if (!Array.isArray(columnIndices)) columnIndices = Array.from(columnIndices);
    return columnIndices;
  }
  function checkRange$1(matrix, startRow, endRow, startColumn, endColumn) {
    if (arguments.length !== 5) {
      throw new RangeError('expected 4 arguments');
    }

    checkNumber$1('startRow', startRow);
    checkNumber$1('endRow', endRow);
    checkNumber$1('startColumn', startColumn);
    checkNumber$1('endColumn', endColumn);

    if (startRow > endRow || startColumn > endColumn || startRow < 0 || startRow >= matrix.rows || endRow < 0 || endRow >= matrix.rows || startColumn < 0 || startColumn >= matrix.columns || endColumn < 0 || endColumn >= matrix.columns) {
      throw new RangeError('Submatrix indices are out of range');
    }
  }
  function newArray$1(length, value = 0) {
    let array = [];

    for (let i = 0; i < length; i++) {
      array.push(value);
    }

    return array;
  }

  function checkNumber$1(name, value) {
    if (typeof value !== 'number') {
      throw new TypeError(`${name} must be a number`);
    }
  }

  function checkNonEmpty$1(matrix) {
    if (matrix.isEmpty()) {
      throw new Error('Empty matrix has no elements to index');
    }
  }

  function sumByRow$1(matrix) {
    let sum = newArray$1(matrix.rows);

    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[i] += matrix.get(i, j);
      }
    }

    return sum;
  }
  function sumByColumn$1(matrix) {
    let sum = newArray$1(matrix.columns);

    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[j] += matrix.get(i, j);
      }
    }

    return sum;
  }
  function sumAll$1(matrix) {
    let v = 0;

    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        v += matrix.get(i, j);
      }
    }

    return v;
  }
  function productByRow$1(matrix) {
    let sum = newArray$1(matrix.rows, 1);

    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[i] *= matrix.get(i, j);
      }
    }

    return sum;
  }
  function productByColumn$1(matrix) {
    let sum = newArray$1(matrix.columns, 1);

    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[j] *= matrix.get(i, j);
      }
    }

    return sum;
  }
  function productAll$1(matrix) {
    let v = 1;

    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        v *= matrix.get(i, j);
      }
    }

    return v;
  }
  function varianceByRow$1(matrix, unbiased, mean) {
    const rows = matrix.rows;
    const cols = matrix.columns;
    const variance = [];

    for (let i = 0; i < rows; i++) {
      let sum1 = 0;
      let sum2 = 0;
      let x = 0;

      for (let j = 0; j < cols; j++) {
        x = matrix.get(i, j) - mean[i];
        sum1 += x;
        sum2 += x * x;
      }

      if (unbiased) {
        variance.push((sum2 - sum1 * sum1 / cols) / (cols - 1));
      } else {
        variance.push((sum2 - sum1 * sum1 / cols) / cols);
      }
    }

    return variance;
  }
  function varianceByColumn$1(matrix, unbiased, mean) {
    const rows = matrix.rows;
    const cols = matrix.columns;
    const variance = [];

    for (let j = 0; j < cols; j++) {
      let sum1 = 0;
      let sum2 = 0;
      let x = 0;

      for (let i = 0; i < rows; i++) {
        x = matrix.get(i, j) - mean[j];
        sum1 += x;
        sum2 += x * x;
      }

      if (unbiased) {
        variance.push((sum2 - sum1 * sum1 / rows) / (rows - 1));
      } else {
        variance.push((sum2 - sum1 * sum1 / rows) / rows);
      }
    }

    return variance;
  }
  function varianceAll$1(matrix, unbiased, mean) {
    const rows = matrix.rows;
    const cols = matrix.columns;
    const size = rows * cols;
    let sum1 = 0;
    let sum2 = 0;
    let x = 0;

    for (let i = 0; i < rows; i++) {
      for (let j = 0; j < cols; j++) {
        x = matrix.get(i, j) - mean;
        sum1 += x;
        sum2 += x * x;
      }
    }

    if (unbiased) {
      return (sum2 - sum1 * sum1 / size) / (size - 1);
    } else {
      return (sum2 - sum1 * sum1 / size) / size;
    }
  }
  function centerByRow$1(matrix, mean) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) - mean[i]);
      }
    }
  }
  function centerByColumn$1(matrix, mean) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) - mean[j]);
      }
    }
  }
  function centerAll$1(matrix, mean) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) - mean);
      }
    }
  }
  function getScaleByRow$1(matrix) {
    const scale = [];

    for (let i = 0; i < matrix.rows; i++) {
      let sum = 0;

      for (let j = 0; j < matrix.columns; j++) {
        sum += Math.pow(matrix.get(i, j), 2) / (matrix.columns - 1);
      }

      scale.push(Math.sqrt(sum));
    }

    return scale;
  }
  function scaleByRow$1(matrix, scale) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) / scale[i]);
      }
    }
  }
  function getScaleByColumn$1(matrix) {
    const scale = [];

    for (let j = 0; j < matrix.columns; j++) {
      let sum = 0;

      for (let i = 0; i < matrix.rows; i++) {
        sum += Math.pow(matrix.get(i, j), 2) / (matrix.rows - 1);
      }

      scale.push(Math.sqrt(sum));
    }

    return scale;
  }
  function scaleByColumn$1(matrix, scale) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) / scale[j]);
      }
    }
  }
  function getScaleAll$1(matrix) {
    const divider = matrix.size - 1;
    let sum = 0;

    for (let j = 0; j < matrix.columns; j++) {
      for (let i = 0; i < matrix.rows; i++) {
        sum += Math.pow(matrix.get(i, j), 2) / divider;
      }
    }

    return Math.sqrt(sum);
  }
  function scaleAll$1(matrix, scale) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) / scale);
      }
    }
  }

  class AbstractMatrix$1 {
    static from1DArray(newRows, newColumns, newData) {
      let length = newRows * newColumns;

      if (length !== newData.length) {
        throw new RangeError('data length does not match given dimensions');
      }

      let newMatrix = new Matrix$2(newRows, newColumns);

      for (let row = 0; row < newRows; row++) {
        for (let column = 0; column < newColumns; column++) {
          newMatrix.set(row, column, newData[row * newColumns + column]);
        }
      }

      return newMatrix;
    }

    static rowVector(newData) {
      let vector = new Matrix$2(1, newData.length);

      for (let i = 0; i < newData.length; i++) {
        vector.set(0, i, newData[i]);
      }

      return vector;
    }

    static columnVector(newData) {
      let vector = new Matrix$2(newData.length, 1);

      for (let i = 0; i < newData.length; i++) {
        vector.set(i, 0, newData[i]);
      }

      return vector;
    }

    static zeros(rows, columns) {
      return new Matrix$2(rows, columns);
    }

    static ones(rows, columns) {
      return new Matrix$2(rows, columns).fill(1);
    }

    static rand(rows, columns, options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        random = Math.random
      } = options;
      let matrix = new Matrix$2(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          matrix.set(i, j, random());
        }
      }

      return matrix;
    }

    static randInt(rows, columns, options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        min = 0,
        max = 1000,
        random = Math.random
      } = options;
      if (!Number.isInteger(min)) throw new TypeError('min must be an integer');
      if (!Number.isInteger(max)) throw new TypeError('max must be an integer');
      if (min >= max) throw new RangeError('min must be smaller than max');
      let interval = max - min;
      let matrix = new Matrix$2(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          let value = min + Math.round(random() * interval);
          matrix.set(i, j, value);
        }
      }

      return matrix;
    }

    static eye(rows, columns, value) {
      if (columns === undefined) columns = rows;
      if (value === undefined) value = 1;
      let min = Math.min(rows, columns);
      let matrix = this.zeros(rows, columns);

      for (let i = 0; i < min; i++) {
        matrix.set(i, i, value);
      }

      return matrix;
    }

    static diag(data, rows, columns) {
      let l = data.length;
      if (rows === undefined) rows = l;
      if (columns === undefined) columns = rows;
      let min = Math.min(l, rows, columns);
      let matrix = this.zeros(rows, columns);

      for (let i = 0; i < min; i++) {
        matrix.set(i, i, data[i]);
      }

      return matrix;
    }

    static min(matrix1, matrix2) {
      matrix1 = this.checkMatrix(matrix1);
      matrix2 = this.checkMatrix(matrix2);
      let rows = matrix1.rows;
      let columns = matrix1.columns;
      let result = new Matrix$2(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          result.set(i, j, Math.min(matrix1.get(i, j), matrix2.get(i, j)));
        }
      }

      return result;
    }

    static max(matrix1, matrix2) {
      matrix1 = this.checkMatrix(matrix1);
      matrix2 = this.checkMatrix(matrix2);
      let rows = matrix1.rows;
      let columns = matrix1.columns;
      let result = new this(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          result.set(i, j, Math.max(matrix1.get(i, j), matrix2.get(i, j)));
        }
      }

      return result;
    }

    static checkMatrix(value) {
      return AbstractMatrix$1.isMatrix(value) ? value : new Matrix$2(value);
    }

    static isMatrix(value) {
      return value != null && value.klass === 'Matrix';
    }

    get size() {
      return this.rows * this.columns;
    }

    apply(callback) {
      if (typeof callback !== 'function') {
        throw new TypeError('callback must be a function');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          callback.call(this, i, j);
        }
      }

      return this;
    }

    to1DArray() {
      let array = [];

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          array.push(this.get(i, j));
        }
      }

      return array;
    }

    to2DArray() {
      let copy = [];

      for (let i = 0; i < this.rows; i++) {
        copy.push([]);

        for (let j = 0; j < this.columns; j++) {
          copy[i].push(this.get(i, j));
        }
      }

      return copy;
    }

    toJSON() {
      return this.to2DArray();
    }

    isRowVector() {
      return this.rows === 1;
    }

    isColumnVector() {
      return this.columns === 1;
    }

    isVector() {
      return this.rows === 1 || this.columns === 1;
    }

    isSquare() {
      return this.rows === this.columns;
    }

    isEmpty() {
      return this.rows === 0 || this.columns === 0;
    }

    isSymmetric() {
      if (this.isSquare()) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j <= i; j++) {
            if (this.get(i, j) !== this.get(j, i)) {
              return false;
            }
          }
        }

        return true;
      }

      return false;
    }

    isEchelonForm() {
      let i = 0;
      let j = 0;
      let previousColumn = -1;
      let isEchelonForm = true;
      let checked = false;

      while (i < this.rows && isEchelonForm) {
        j = 0;
        checked = false;

        while (j < this.columns && checked === false) {
          if (this.get(i, j) === 0) {
            j++;
          } else if (this.get(i, j) === 1 && j > previousColumn) {
            checked = true;
            previousColumn = j;
          } else {
            isEchelonForm = false;
            checked = true;
          }
        }

        i++;
      }

      return isEchelonForm;
    }

    isReducedEchelonForm() {
      let i = 0;
      let j = 0;
      let previousColumn = -1;
      let isReducedEchelonForm = true;
      let checked = false;

      while (i < this.rows && isReducedEchelonForm) {
        j = 0;
        checked = false;

        while (j < this.columns && checked === false) {
          if (this.get(i, j) === 0) {
            j++;
          } else if (this.get(i, j) === 1 && j > previousColumn) {
            checked = true;
            previousColumn = j;
          } else {
            isReducedEchelonForm = false;
            checked = true;
          }
        }

        for (let k = j + 1; k < this.rows; k++) {
          if (this.get(i, k) !== 0) {
            isReducedEchelonForm = false;
          }
        }

        i++;
      }

      return isReducedEchelonForm;
    }

    echelonForm() {
      let result = this.clone();
      let h = 0;
      let k = 0;

      while (h < result.rows && k < result.columns) {
        let iMax = h;

        for (let i = h; i < result.rows; i++) {
          if (result.get(i, k) > result.get(iMax, k)) {
            iMax = i;
          }
        }

        if (result.get(iMax, k) === 0) {
          k++;
        } else {
          result.swapRows(h, iMax);
          let tmp = result.get(h, k);

          for (let j = k; j < result.columns; j++) {
            result.set(h, j, result.get(h, j) / tmp);
          }

          for (let i = h + 1; i < result.rows; i++) {
            let factor = result.get(i, k) / result.get(h, k);
            result.set(i, k, 0);

            for (let j = k + 1; j < result.columns; j++) {
              result.set(i, j, result.get(i, j) - result.get(h, j) * factor);
            }
          }

          h++;
          k++;
        }
      }

      return result;
    }

    reducedEchelonForm() {
      let result = this.echelonForm();
      let m = result.columns;
      let n = result.rows;
      let h = n - 1;

      while (h >= 0) {
        if (result.maxRow(h) === 0) {
          h--;
        } else {
          let p = 0;
          let pivot = false;

          while (p < n && pivot === false) {
            if (result.get(h, p) === 1) {
              pivot = true;
            } else {
              p++;
            }
          }

          for (let i = 0; i < h; i++) {
            let factor = result.get(i, p);

            for (let j = p; j < m; j++) {
              let tmp = result.get(i, j) - factor * result.get(h, j);
              result.set(i, j, tmp);
            }
          }

          h--;
        }
      }

      return result;
    }

    set() {
      throw new Error('set method is unimplemented');
    }

    get() {
      throw new Error('get method is unimplemented');
    }

    repeat(options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        rows = 1,
        columns = 1
      } = options;

      if (!Number.isInteger(rows) || rows <= 0) {
        throw new TypeError('rows must be a positive integer');
      }

      if (!Number.isInteger(columns) || columns <= 0) {
        throw new TypeError('columns must be a positive integer');
      }

      let matrix = new Matrix$2(this.rows * rows, this.columns * columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          matrix.setSubMatrix(this, this.rows * i, this.columns * j);
        }
      }

      return matrix;
    }

    fill(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, value);
        }
      }

      return this;
    }

    neg() {
      return this.mulS(-1);
    }

    getRow(index) {
      checkRowIndex$1(this, index);
      let row = [];

      for (let i = 0; i < this.columns; i++) {
        row.push(this.get(index, i));
      }

      return row;
    }

    getRowVector(index) {
      return Matrix$2.rowVector(this.getRow(index));
    }

    setRow(index, array) {
      checkRowIndex$1(this, index);
      array = checkRowVector$1(this, array);

      for (let i = 0; i < this.columns; i++) {
        this.set(index, i, array[i]);
      }

      return this;
    }

    swapRows(row1, row2) {
      checkRowIndex$1(this, row1);
      checkRowIndex$1(this, row2);

      for (let i = 0; i < this.columns; i++) {
        let temp = this.get(row1, i);
        this.set(row1, i, this.get(row2, i));
        this.set(row2, i, temp);
      }

      return this;
    }

    getColumn(index) {
      checkColumnIndex$1(this, index);
      let column = [];

      for (let i = 0; i < this.rows; i++) {
        column.push(this.get(i, index));
      }

      return column;
    }

    getColumnVector(index) {
      return Matrix$2.columnVector(this.getColumn(index));
    }

    setColumn(index, array) {
      checkColumnIndex$1(this, index);
      array = checkColumnVector$1(this, array);

      for (let i = 0; i < this.rows; i++) {
        this.set(i, index, array[i]);
      }

      return this;
    }

    swapColumns(column1, column2) {
      checkColumnIndex$1(this, column1);
      checkColumnIndex$1(this, column2);

      for (let i = 0; i < this.rows; i++) {
        let temp = this.get(i, column1);
        this.set(i, column1, this.get(i, column2));
        this.set(i, column2, temp);
      }

      return this;
    }

    addRowVector(vector) {
      vector = checkRowVector$1(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + vector[j]);
        }
      }

      return this;
    }

    subRowVector(vector) {
      vector = checkRowVector$1(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - vector[j]);
        }
      }

      return this;
    }

    mulRowVector(vector) {
      vector = checkRowVector$1(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * vector[j]);
        }
      }

      return this;
    }

    divRowVector(vector) {
      vector = checkRowVector$1(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / vector[j]);
        }
      }

      return this;
    }

    addColumnVector(vector) {
      vector = checkColumnVector$1(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + vector[i]);
        }
      }

      return this;
    }

    subColumnVector(vector) {
      vector = checkColumnVector$1(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - vector[i]);
        }
      }

      return this;
    }

    mulColumnVector(vector) {
      vector = checkColumnVector$1(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * vector[i]);
        }
      }

      return this;
    }

    divColumnVector(vector) {
      vector = checkColumnVector$1(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / vector[i]);
        }
      }

      return this;
    }

    mulRow(index, value) {
      checkRowIndex$1(this, index);

      for (let i = 0; i < this.columns; i++) {
        this.set(index, i, this.get(index, i) * value);
      }

      return this;
    }

    mulColumn(index, value) {
      checkColumnIndex$1(this, index);

      for (let i = 0; i < this.rows; i++) {
        this.set(i, index, this.get(i, index) * value);
      }

      return this;
    }

    max() {
      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(0, 0);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) > v) {
            v = this.get(i, j);
          }
        }
      }

      return v;
    }

    maxIndex() {
      checkNonEmpty$1(this);
      let v = this.get(0, 0);
      let idx = [0, 0];

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) > v) {
            v = this.get(i, j);
            idx[0] = i;
            idx[1] = j;
          }
        }
      }

      return idx;
    }

    min() {
      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(0, 0);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) < v) {
            v = this.get(i, j);
          }
        }
      }

      return v;
    }

    minIndex() {
      checkNonEmpty$1(this);
      let v = this.get(0, 0);
      let idx = [0, 0];

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) < v) {
            v = this.get(i, j);
            idx[0] = i;
            idx[1] = j;
          }
        }
      }

      return idx;
    }

    maxRow(row) {
      checkRowIndex$1(this, row);

      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(row, 0);

      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) > v) {
          v = this.get(row, i);
        }
      }

      return v;
    }

    maxRowIndex(row) {
      checkRowIndex$1(this, row);
      checkNonEmpty$1(this);
      let v = this.get(row, 0);
      let idx = [row, 0];

      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) > v) {
          v = this.get(row, i);
          idx[1] = i;
        }
      }

      return idx;
    }

    minRow(row) {
      checkRowIndex$1(this, row);

      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(row, 0);

      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) < v) {
          v = this.get(row, i);
        }
      }

      return v;
    }

    minRowIndex(row) {
      checkRowIndex$1(this, row);
      checkNonEmpty$1(this);
      let v = this.get(row, 0);
      let idx = [row, 0];

      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) < v) {
          v = this.get(row, i);
          idx[1] = i;
        }
      }

      return idx;
    }

    maxColumn(column) {
      checkColumnIndex$1(this, column);

      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(0, column);

      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) > v) {
          v = this.get(i, column);
        }
      }

      return v;
    }

    maxColumnIndex(column) {
      checkColumnIndex$1(this, column);
      checkNonEmpty$1(this);
      let v = this.get(0, column);
      let idx = [0, column];

      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) > v) {
          v = this.get(i, column);
          idx[0] = i;
        }
      }

      return idx;
    }

    minColumn(column) {
      checkColumnIndex$1(this, column);

      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(0, column);

      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) < v) {
          v = this.get(i, column);
        }
      }

      return v;
    }

    minColumnIndex(column) {
      checkColumnIndex$1(this, column);
      checkNonEmpty$1(this);
      let v = this.get(0, column);
      let idx = [0, column];

      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) < v) {
          v = this.get(i, column);
          idx[0] = i;
        }
      }

      return idx;
    }

    diag() {
      let min = Math.min(this.rows, this.columns);
      let diag = [];

      for (let i = 0; i < min; i++) {
        diag.push(this.get(i, i));
      }

      return diag;
    }

    norm(type = 'frobenius') {
      let result = 0;

      if (type === 'max') {
        return this.max();
      } else if (type === 'frobenius') {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            result = result + this.get(i, j) * this.get(i, j);
          }
        }

        return Math.sqrt(result);
      } else {
        throw new RangeError(`unknown norm type: ${type}`);
      }
    }

    cumulativeSum() {
      let sum = 0;

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          sum += this.get(i, j);
          this.set(i, j, sum);
        }
      }

      return this;
    }

    dot(vector2) {
      if (AbstractMatrix$1.isMatrix(vector2)) vector2 = vector2.to1DArray();
      let vector1 = this.to1DArray();

      if (vector1.length !== vector2.length) {
        throw new RangeError('vectors do not have the same size');
      }

      let dot = 0;

      for (let i = 0; i < vector1.length; i++) {
        dot += vector1[i] * vector2[i];
      }

      return dot;
    }

    mmul(other) {
      other = Matrix$2.checkMatrix(other);
      let m = this.rows;
      let n = this.columns;
      let p = other.columns;
      let result = new Matrix$2(m, p);
      let Bcolj = new Float64Array(n);

      for (let j = 0; j < p; j++) {
        for (let k = 0; k < n; k++) {
          Bcolj[k] = other.get(k, j);
        }

        for (let i = 0; i < m; i++) {
          let s = 0;

          for (let k = 0; k < n; k++) {
            s += this.get(i, k) * Bcolj[k];
          }

          result.set(i, j, s);
        }
      }

      return result;
    }

    strassen2x2(other) {
      other = Matrix$2.checkMatrix(other);
      let result = new Matrix$2(2, 2);
      const a11 = this.get(0, 0);
      const b11 = other.get(0, 0);
      const a12 = this.get(0, 1);
      const b12 = other.get(0, 1);
      const a21 = this.get(1, 0);
      const b21 = other.get(1, 0);
      const a22 = this.get(1, 1);
      const b22 = other.get(1, 1); // Compute intermediate values.

      const m1 = (a11 + a22) * (b11 + b22);
      const m2 = (a21 + a22) * b11;
      const m3 = a11 * (b12 - b22);
      const m4 = a22 * (b21 - b11);
      const m5 = (a11 + a12) * b22;
      const m6 = (a21 - a11) * (b11 + b12);
      const m7 = (a12 - a22) * (b21 + b22); // Combine intermediate values into the output.

      const c00 = m1 + m4 - m5 + m7;
      const c01 = m3 + m5;
      const c10 = m2 + m4;
      const c11 = m1 - m2 + m3 + m6;
      result.set(0, 0, c00);
      result.set(0, 1, c01);
      result.set(1, 0, c10);
      result.set(1, 1, c11);
      return result;
    }

    strassen3x3(other) {
      other = Matrix$2.checkMatrix(other);
      let result = new Matrix$2(3, 3);
      const a00 = this.get(0, 0);
      const a01 = this.get(0, 1);
      const a02 = this.get(0, 2);
      const a10 = this.get(1, 0);
      const a11 = this.get(1, 1);
      const a12 = this.get(1, 2);
      const a20 = this.get(2, 0);
      const a21 = this.get(2, 1);
      const a22 = this.get(2, 2);
      const b00 = other.get(0, 0);
      const b01 = other.get(0, 1);
      const b02 = other.get(0, 2);
      const b10 = other.get(1, 0);
      const b11 = other.get(1, 1);
      const b12 = other.get(1, 2);
      const b20 = other.get(2, 0);
      const b21 = other.get(2, 1);
      const b22 = other.get(2, 2);
      const m1 = (a00 + a01 + a02 - a10 - a11 - a21 - a22) * b11;
      const m2 = (a00 - a10) * (-b01 + b11);
      const m3 = a11 * (-b00 + b01 + b10 - b11 - b12 - b20 + b22);
      const m4 = (-a00 + a10 + a11) * (b00 - b01 + b11);
      const m5 = (a10 + a11) * (-b00 + b01);
      const m6 = a00 * b00;
      const m7 = (-a00 + a20 + a21) * (b00 - b02 + b12);
      const m8 = (-a00 + a20) * (b02 - b12);
      const m9 = (a20 + a21) * (-b00 + b02);
      const m10 = (a00 + a01 + a02 - a11 - a12 - a20 - a21) * b12;
      const m11 = a21 * (-b00 + b02 + b10 - b11 - b12 - b20 + b21);
      const m12 = (-a02 + a21 + a22) * (b11 + b20 - b21);
      const m13 = (a02 - a22) * (b11 - b21);
      const m14 = a02 * b20;
      const m15 = (a21 + a22) * (-b20 + b21);
      const m16 = (-a02 + a11 + a12) * (b12 + b20 - b22);
      const m17 = (a02 - a12) * (b12 - b22);
      const m18 = (a11 + a12) * (-b20 + b22);
      const m19 = a01 * b10;
      const m20 = a12 * b21;
      const m21 = a10 * b02;
      const m22 = a20 * b01;
      const m23 = a22 * b22;
      const c00 = m6 + m14 + m19;
      const c01 = m1 + m4 + m5 + m6 + m12 + m14 + m15;
      const c02 = m6 + m7 + m9 + m10 + m14 + m16 + m18;
      const c10 = m2 + m3 + m4 + m6 + m14 + m16 + m17;
      const c11 = m2 + m4 + m5 + m6 + m20;
      const c12 = m14 + m16 + m17 + m18 + m21;
      const c20 = m6 + m7 + m8 + m11 + m12 + m13 + m14;
      const c21 = m12 + m13 + m14 + m15 + m22;
      const c22 = m6 + m7 + m8 + m9 + m23;
      result.set(0, 0, c00);
      result.set(0, 1, c01);
      result.set(0, 2, c02);
      result.set(1, 0, c10);
      result.set(1, 1, c11);
      result.set(1, 2, c12);
      result.set(2, 0, c20);
      result.set(2, 1, c21);
      result.set(2, 2, c22);
      return result;
    }

    mmulStrassen(y) {
      y = Matrix$2.checkMatrix(y);
      let x = this.clone();
      let r1 = x.rows;
      let c1 = x.columns;
      let r2 = y.rows;
      let c2 = y.columns;

      if (c1 !== r2) {
        // eslint-disable-next-line no-console
        console.warn(`Multiplying ${r1} x ${c1} and ${r2} x ${c2} matrix: dimensions do not match.`);
      } // Put a matrix into the top left of a matrix of zeros.
      // `rows` and `cols` are the dimensions of the output matrix.


      function embed(mat, rows, cols) {
        let r = mat.rows;
        let c = mat.columns;

        if (r === rows && c === cols) {
          return mat;
        } else {
          let resultat = AbstractMatrix$1.zeros(rows, cols);
          resultat = resultat.setSubMatrix(mat, 0, 0);
          return resultat;
        }
      } // Make sure both matrices are the same size.
      // This is exclusively for simplicity:
      // this algorithm can be implemented with matrices of different sizes.


      let r = Math.max(r1, r2);
      let c = Math.max(c1, c2);
      x = embed(x, r, c);
      y = embed(y, r, c); // Our recursive multiplication function.

      function blockMult(a, b, rows, cols) {
        // For small matrices, resort to naive multiplication.
        if (rows <= 512 || cols <= 512) {
          return a.mmul(b); // a is equivalent to this
        } // Apply dynamic padding.


        if (rows % 2 === 1 && cols % 2 === 1) {
          a = embed(a, rows + 1, cols + 1);
          b = embed(b, rows + 1, cols + 1);
        } else if (rows % 2 === 1) {
          a = embed(a, rows + 1, cols);
          b = embed(b, rows + 1, cols);
        } else if (cols % 2 === 1) {
          a = embed(a, rows, cols + 1);
          b = embed(b, rows, cols + 1);
        }

        let halfRows = parseInt(a.rows / 2, 10);
        let halfCols = parseInt(a.columns / 2, 10); // Subdivide input matrices.

        let a11 = a.subMatrix(0, halfRows - 1, 0, halfCols - 1);
        let b11 = b.subMatrix(0, halfRows - 1, 0, halfCols - 1);
        let a12 = a.subMatrix(0, halfRows - 1, halfCols, a.columns - 1);
        let b12 = b.subMatrix(0, halfRows - 1, halfCols, b.columns - 1);
        let a21 = a.subMatrix(halfRows, a.rows - 1, 0, halfCols - 1);
        let b21 = b.subMatrix(halfRows, b.rows - 1, 0, halfCols - 1);
        let a22 = a.subMatrix(halfRows, a.rows - 1, halfCols, a.columns - 1);
        let b22 = b.subMatrix(halfRows, b.rows - 1, halfCols, b.columns - 1); // Compute intermediate values.

        let m1 = blockMult(AbstractMatrix$1.add(a11, a22), AbstractMatrix$1.add(b11, b22), halfRows, halfCols);
        let m2 = blockMult(AbstractMatrix$1.add(a21, a22), b11, halfRows, halfCols);
        let m3 = blockMult(a11, AbstractMatrix$1.sub(b12, b22), halfRows, halfCols);
        let m4 = blockMult(a22, AbstractMatrix$1.sub(b21, b11), halfRows, halfCols);
        let m5 = blockMult(AbstractMatrix$1.add(a11, a12), b22, halfRows, halfCols);
        let m6 = blockMult(AbstractMatrix$1.sub(a21, a11), AbstractMatrix$1.add(b11, b12), halfRows, halfCols);
        let m7 = blockMult(AbstractMatrix$1.sub(a12, a22), AbstractMatrix$1.add(b21, b22), halfRows, halfCols); // Combine intermediate values into the output.

        let c11 = AbstractMatrix$1.add(m1, m4);
        c11.sub(m5);
        c11.add(m7);
        let c12 = AbstractMatrix$1.add(m3, m5);
        let c21 = AbstractMatrix$1.add(m2, m4);
        let c22 = AbstractMatrix$1.sub(m1, m2);
        c22.add(m3);
        c22.add(m6); // Crop output to the desired size (undo dynamic padding).

        let resultat = AbstractMatrix$1.zeros(2 * c11.rows, 2 * c11.columns);
        resultat = resultat.setSubMatrix(c11, 0, 0);
        resultat = resultat.setSubMatrix(c12, c11.rows, 0);
        resultat = resultat.setSubMatrix(c21, 0, c11.columns);
        resultat = resultat.setSubMatrix(c22, c11.rows, c11.columns);
        return resultat.subMatrix(0, rows - 1, 0, cols - 1);
      }

      return blockMult(x, y, r, c);
    }

    scaleRows(options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        min = 0,
        max = 1
      } = options;
      if (!Number.isFinite(min)) throw new TypeError('min must be a number');
      if (!Number.isFinite(max)) throw new TypeError('max must be a number');
      if (min >= max) throw new RangeError('min must be smaller than max');
      let newMatrix = new Matrix$2(this.rows, this.columns);

      for (let i = 0; i < this.rows; i++) {
        const row = this.getRow(i);

        if (row.length > 0) {
          rescale$1(row, {
            min,
            max,
            output: row
          });
        }

        newMatrix.setRow(i, row);
      }

      return newMatrix;
    }

    scaleColumns(options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        min = 0,
        max = 1
      } = options;
      if (!Number.isFinite(min)) throw new TypeError('min must be a number');
      if (!Number.isFinite(max)) throw new TypeError('max must be a number');
      if (min >= max) throw new RangeError('min must be smaller than max');
      let newMatrix = new Matrix$2(this.rows, this.columns);

      for (let i = 0; i < this.columns; i++) {
        const column = this.getColumn(i);

        if (column.length) {
          rescale$1(column, {
            min: min,
            max: max,
            output: column
          });
        }

        newMatrix.setColumn(i, column);
      }

      return newMatrix;
    }

    flipRows() {
      const middle = Math.ceil(this.columns / 2);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < middle; j++) {
          let first = this.get(i, j);
          let last = this.get(i, this.columns - 1 - j);
          this.set(i, j, last);
          this.set(i, this.columns - 1 - j, first);
        }
      }

      return this;
    }

    flipColumns() {
      const middle = Math.ceil(this.rows / 2);

      for (let j = 0; j < this.columns; j++) {
        for (let i = 0; i < middle; i++) {
          let first = this.get(i, j);
          let last = this.get(this.rows - 1 - i, j);
          this.set(i, j, last);
          this.set(this.rows - 1 - i, j, first);
        }
      }

      return this;
    }

    kroneckerProduct(other) {
      other = Matrix$2.checkMatrix(other);
      let m = this.rows;
      let n = this.columns;
      let p = other.rows;
      let q = other.columns;
      let result = new Matrix$2(m * p, n * q);

      for (let i = 0; i < m; i++) {
        for (let j = 0; j < n; j++) {
          for (let k = 0; k < p; k++) {
            for (let l = 0; l < q; l++) {
              result.set(p * i + k, q * j + l, this.get(i, j) * other.get(k, l));
            }
          }
        }
      }

      return result;
    }

    kroneckerSum(other) {
      other = Matrix$2.checkMatrix(other);

      if (!this.isSquare() || !other.isSquare()) {
        throw new Error('Kronecker Sum needs two Square Matrices');
      }

      let m = this.rows;
      let n = other.rows;
      let AxI = this.kroneckerProduct(Matrix$2.eye(n, n));
      let IxB = Matrix$2.eye(m, m).kroneckerProduct(other);
      return AxI.add(IxB);
    }

    transpose() {
      let result = new Matrix$2(this.columns, this.rows);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          result.set(j, i, this.get(i, j));
        }
      }

      return result;
    }

    sortRows(compareFunction = compareNumbers$1) {
      for (let i = 0; i < this.rows; i++) {
        this.setRow(i, this.getRow(i).sort(compareFunction));
      }

      return this;
    }

    sortColumns(compareFunction = compareNumbers$1) {
      for (let i = 0; i < this.columns; i++) {
        this.setColumn(i, this.getColumn(i).sort(compareFunction));
      }

      return this;
    }

    subMatrix(startRow, endRow, startColumn, endColumn) {
      checkRange$1(this, startRow, endRow, startColumn, endColumn);
      let newMatrix = new Matrix$2(endRow - startRow + 1, endColumn - startColumn + 1);

      for (let i = startRow; i <= endRow; i++) {
        for (let j = startColumn; j <= endColumn; j++) {
          newMatrix.set(i - startRow, j - startColumn, this.get(i, j));
        }
      }

      return newMatrix;
    }

    subMatrixRow(indices, startColumn, endColumn) {
      if (startColumn === undefined) startColumn = 0;
      if (endColumn === undefined) endColumn = this.columns - 1;

      if (startColumn > endColumn || startColumn < 0 || startColumn >= this.columns || endColumn < 0 || endColumn >= this.columns) {
        throw new RangeError('Argument out of range');
      }

      let newMatrix = new Matrix$2(indices.length, endColumn - startColumn + 1);

      for (let i = 0; i < indices.length; i++) {
        for (let j = startColumn; j <= endColumn; j++) {
          if (indices[i] < 0 || indices[i] >= this.rows) {
            throw new RangeError(`Row index out of range: ${indices[i]}`);
          }

          newMatrix.set(i, j - startColumn, this.get(indices[i], j));
        }
      }

      return newMatrix;
    }

    subMatrixColumn(indices, startRow, endRow) {
      if (startRow === undefined) startRow = 0;
      if (endRow === undefined) endRow = this.rows - 1;

      if (startRow > endRow || startRow < 0 || startRow >= this.rows || endRow < 0 || endRow >= this.rows) {
        throw new RangeError('Argument out of range');
      }

      let newMatrix = new Matrix$2(endRow - startRow + 1, indices.length);

      for (let i = 0; i < indices.length; i++) {
        for (let j = startRow; j <= endRow; j++) {
          if (indices[i] < 0 || indices[i] >= this.columns) {
            throw new RangeError(`Column index out of range: ${indices[i]}`);
          }

          newMatrix.set(j - startRow, i, this.get(j, indices[i]));
        }
      }

      return newMatrix;
    }

    setSubMatrix(matrix, startRow, startColumn) {
      matrix = Matrix$2.checkMatrix(matrix);

      if (matrix.isEmpty()) {
        return this;
      }

      let endRow = startRow + matrix.rows - 1;
      let endColumn = startColumn + matrix.columns - 1;
      checkRange$1(this, startRow, endRow, startColumn, endColumn);

      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          this.set(startRow + i, startColumn + j, matrix.get(i, j));
        }
      }

      return this;
    }

    selection(rowIndices, columnIndices) {
      let indices = checkIndices$1(this, rowIndices, columnIndices);
      let newMatrix = new Matrix$2(rowIndices.length, columnIndices.length);

      for (let i = 0; i < indices.row.length; i++) {
        let rowIndex = indices.row[i];

        for (let j = 0; j < indices.column.length; j++) {
          let columnIndex = indices.column[j];
          newMatrix.set(i, j, this.get(rowIndex, columnIndex));
        }
      }

      return newMatrix;
    }

    trace() {
      let min = Math.min(this.rows, this.columns);
      let trace = 0;

      for (let i = 0; i < min; i++) {
        trace += this.get(i, i);
      }

      return trace;
    }

    clone() {
      let newMatrix = new Matrix$2(this.rows, this.columns);

      for (let row = 0; row < this.rows; row++) {
        for (let column = 0; column < this.columns; column++) {
          newMatrix.set(row, column, this.get(row, column));
        }
      }

      return newMatrix;
    }

    sum(by) {
      switch (by) {
        case 'row':
          return sumByRow$1(this);

        case 'column':
          return sumByColumn$1(this);

        case undefined:
          return sumAll$1(this);

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    product(by) {
      switch (by) {
        case 'row':
          return productByRow$1(this);

        case 'column':
          return productByColumn$1(this);

        case undefined:
          return productAll$1(this);

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    mean(by) {
      const sum = this.sum(by);

      switch (by) {
        case 'row':
          {
            for (let i = 0; i < this.rows; i++) {
              sum[i] /= this.columns;
            }

            return sum;
          }

        case 'column':
          {
            for (let i = 0; i < this.columns; i++) {
              sum[i] /= this.rows;
            }

            return sum;
          }

        case undefined:
          return sum / this.size;

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    variance(by, options = {}) {
      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }

      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        unbiased = true,
        mean = this.mean(by)
      } = options;

      if (typeof unbiased !== 'boolean') {
        throw new TypeError('unbiased must be a boolean');
      }

      switch (by) {
        case 'row':
          {
            if (!Array.isArray(mean)) {
              throw new TypeError('mean must be an array');
            }

            return varianceByRow$1(this, unbiased, mean);
          }

        case 'column':
          {
            if (!Array.isArray(mean)) {
              throw new TypeError('mean must be an array');
            }

            return varianceByColumn$1(this, unbiased, mean);
          }

        case undefined:
          {
            if (typeof mean !== 'number') {
              throw new TypeError('mean must be a number');
            }

            return varianceAll$1(this, unbiased, mean);
          }

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    standardDeviation(by, options) {
      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }

      const variance = this.variance(by, options);

      if (by === undefined) {
        return Math.sqrt(variance);
      } else {
        for (let i = 0; i < variance.length; i++) {
          variance[i] = Math.sqrt(variance[i]);
        }

        return variance;
      }
    }

    center(by, options = {}) {
      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }

      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        center = this.mean(by)
      } = options;

      switch (by) {
        case 'row':
          {
            if (!Array.isArray(center)) {
              throw new TypeError('center must be an array');
            }

            centerByRow$1(this, center);
            return this;
          }

        case 'column':
          {
            if (!Array.isArray(center)) {
              throw new TypeError('center must be an array');
            }

            centerByColumn$1(this, center);
            return this;
          }

        case undefined:
          {
            if (typeof center !== 'number') {
              throw new TypeError('center must be a number');
            }

            centerAll$1(this, center);
            return this;
          }

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    scale(by, options = {}) {
      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }

      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      let scale = options.scale;

      switch (by) {
        case 'row':
          {
            if (scale === undefined) {
              scale = getScaleByRow$1(this);
            } else if (!Array.isArray(scale)) {
              throw new TypeError('scale must be an array');
            }

            scaleByRow$1(this, scale);
            return this;
          }

        case 'column':
          {
            if (scale === undefined) {
              scale = getScaleByColumn$1(this);
            } else if (!Array.isArray(scale)) {
              throw new TypeError('scale must be an array');
            }

            scaleByColumn$1(this, scale);
            return this;
          }

        case undefined:
          {
            if (scale === undefined) {
              scale = getScaleAll$1(this);
            } else if (typeof scale !== 'number') {
              throw new TypeError('scale must be a number');
            }

            scaleAll$1(this, scale);
            return this;
          }

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    toString(options) {
      return inspectMatrixWithOptions$1(this, options);
    }

  }
  AbstractMatrix$1.prototype.klass = 'Matrix';

  if (typeof Symbol !== 'undefined') {
    AbstractMatrix$1.prototype[Symbol.for('nodejs.util.inspect.custom')] = inspectMatrix$1;
  }

  function compareNumbers$1(a, b) {
    return a - b;
  } // Synonyms


  AbstractMatrix$1.random = AbstractMatrix$1.rand;
  AbstractMatrix$1.randomInt = AbstractMatrix$1.randInt;
  AbstractMatrix$1.diagonal = AbstractMatrix$1.diag;
  AbstractMatrix$1.prototype.diagonal = AbstractMatrix$1.prototype.diag;
  AbstractMatrix$1.identity = AbstractMatrix$1.eye;
  AbstractMatrix$1.prototype.negate = AbstractMatrix$1.prototype.neg;
  AbstractMatrix$1.prototype.tensorProduct = AbstractMatrix$1.prototype.kroneckerProduct;
  class Matrix$2 extends AbstractMatrix$1 {
    constructor(nRows, nColumns) {
      super();

      if (Matrix$2.isMatrix(nRows)) {
        // eslint-disable-next-line no-constructor-return
        return nRows.clone();
      } else if (Number.isInteger(nRows) && nRows >= 0) {
        // Create an empty matrix
        this.data = [];

        if (Number.isInteger(nColumns) && nColumns >= 0) {
          for (let i = 0; i < nRows; i++) {
            this.data.push(new Float64Array(nColumns));
          }
        } else {
          throw new TypeError('nColumns must be a positive integer');
        }
      } else if (Array.isArray(nRows)) {
        // Copy the values from the 2D array
        const arrayData = nRows;
        nRows = arrayData.length;
        nColumns = nRows ? arrayData[0].length : 0;

        if (typeof nColumns !== 'number') {
          throw new TypeError('Data must be a 2D array with at least one element');
        }

        this.data = [];

        for (let i = 0; i < nRows; i++) {
          if (arrayData[i].length !== nColumns) {
            throw new RangeError('Inconsistent array dimensions');
          }

          this.data.push(Float64Array.from(arrayData[i]));
        }
      } else {
        throw new TypeError('First argument must be a positive number or an array');
      }

      this.rows = nRows;
      this.columns = nColumns;
    }

    set(rowIndex, columnIndex, value) {
      this.data[rowIndex][columnIndex] = value;
      return this;
    }

    get(rowIndex, columnIndex) {
      return this.data[rowIndex][columnIndex];
    }

    removeRow(index) {
      checkRowIndex$1(this, index);
      this.data.splice(index, 1);
      this.rows -= 1;
      return this;
    }

    addRow(index, array) {
      if (array === undefined) {
        array = index;
        index = this.rows;
      }

      checkRowIndex$1(this, index, true);
      array = Float64Array.from(checkRowVector$1(this, array));
      this.data.splice(index, 0, array);
      this.rows += 1;
      return this;
    }

    removeColumn(index) {
      checkColumnIndex$1(this, index);

      for (let i = 0; i < this.rows; i++) {
        const newRow = new Float64Array(this.columns - 1);

        for (let j = 0; j < index; j++) {
          newRow[j] = this.data[i][j];
        }

        for (let j = index + 1; j < this.columns; j++) {
          newRow[j - 1] = this.data[i][j];
        }

        this.data[i] = newRow;
      }

      this.columns -= 1;
      return this;
    }

    addColumn(index, array) {
      if (typeof array === 'undefined') {
        array = index;
        index = this.columns;
      }

      checkColumnIndex$1(this, index, true);
      array = checkColumnVector$1(this, array);

      for (let i = 0; i < this.rows; i++) {
        const newRow = new Float64Array(this.columns + 1);
        let j = 0;

        for (; j < index; j++) {
          newRow[j] = this.data[i][j];
        }

        newRow[j++] = array[i];

        for (; j < this.columns + 1; j++) {
          newRow[j] = this.data[i][j - 1];
        }

        this.data[i] = newRow;
      }

      this.columns += 1;
      return this;
    }

  }
  installMathOperations$1(AbstractMatrix$1, Matrix$2);

  class WrapperMatrix2D extends AbstractMatrix$1 {
    constructor(data) {
      super();
      this.data = data;
      this.rows = data.length;
      this.columns = data[0].length;
    }

    set(rowIndex, columnIndex, value) {
      this.data[rowIndex][columnIndex] = value;
      return this;
    }

    get(rowIndex, columnIndex) {
      return this.data[rowIndex][columnIndex];
    }

  }

  class LuDecomposition {
    constructor(matrix) {
      matrix = WrapperMatrix2D.checkMatrix(matrix);
      let lu = matrix.clone();
      let rows = lu.rows;
      let columns = lu.columns;
      let pivotVector = new Float64Array(rows);
      let pivotSign = 1;
      let i, j, k, p, s, t, v;
      let LUcolj, kmax;

      for (i = 0; i < rows; i++) {
        pivotVector[i] = i;
      }

      LUcolj = new Float64Array(rows);

      for (j = 0; j < columns; j++) {
        for (i = 0; i < rows; i++) {
          LUcolj[i] = lu.get(i, j);
        }

        for (i = 0; i < rows; i++) {
          kmax = Math.min(i, j);
          s = 0;

          for (k = 0; k < kmax; k++) {
            s += lu.get(i, k) * LUcolj[k];
          }

          LUcolj[i] -= s;
          lu.set(i, j, LUcolj[i]);
        }

        p = j;

        for (i = j + 1; i < rows; i++) {
          if (Math.abs(LUcolj[i]) > Math.abs(LUcolj[p])) {
            p = i;
          }
        }

        if (p !== j) {
          for (k = 0; k < columns; k++) {
            t = lu.get(p, k);
            lu.set(p, k, lu.get(j, k));
            lu.set(j, k, t);
          }

          v = pivotVector[p];
          pivotVector[p] = pivotVector[j];
          pivotVector[j] = v;
          pivotSign = -pivotSign;
        }

        if (j < rows && lu.get(j, j) !== 0) {
          for (i = j + 1; i < rows; i++) {
            lu.set(i, j, lu.get(i, j) / lu.get(j, j));
          }
        }
      }

      this.LU = lu;
      this.pivotVector = pivotVector;
      this.pivotSign = pivotSign;
    }

    isSingular() {
      let data = this.LU;
      let col = data.columns;

      for (let j = 0; j < col; j++) {
        if (data.get(j, j) === 0) {
          return true;
        }
      }

      return false;
    }

    solve(value) {
      value = Matrix$2.checkMatrix(value);
      let lu = this.LU;
      let rows = lu.rows;

      if (rows !== value.rows) {
        throw new Error('Invalid matrix dimensions');
      }

      if (this.isSingular()) {
        throw new Error('LU matrix is singular');
      }

      let count = value.columns;
      let X = value.subMatrixRow(this.pivotVector, 0, count - 1);
      let columns = lu.columns;
      let i, j, k;

      for (k = 0; k < columns; k++) {
        for (i = k + 1; i < columns; i++) {
          for (j = 0; j < count; j++) {
            X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
          }
        }
      }

      for (k = columns - 1; k >= 0; k--) {
        for (j = 0; j < count; j++) {
          X.set(k, j, X.get(k, j) / lu.get(k, k));
        }

        for (i = 0; i < k; i++) {
          for (j = 0; j < count; j++) {
            X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
          }
        }
      }

      return X;
    }

    get determinant() {
      let data = this.LU;

      if (!data.isSquare()) {
        throw new Error('Matrix must be square');
      }

      let determinant = this.pivotSign;
      let col = data.columns;

      for (let j = 0; j < col; j++) {
        determinant *= data.get(j, j);
      }

      return determinant;
    }

    get lowerTriangularMatrix() {
      let data = this.LU;
      let rows = data.rows;
      let columns = data.columns;
      let X = new Matrix$2(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          if (i > j) {
            X.set(i, j, data.get(i, j));
          } else if (i === j) {
            X.set(i, j, 1);
          } else {
            X.set(i, j, 0);
          }
        }
      }

      return X;
    }

    get upperTriangularMatrix() {
      let data = this.LU;
      let rows = data.rows;
      let columns = data.columns;
      let X = new Matrix$2(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          if (i <= j) {
            X.set(i, j, data.get(i, j));
          } else {
            X.set(i, j, 0);
          }
        }
      }

      return X;
    }

    get pivotPermutationVector() {
      return Array.from(this.pivotVector);
    }

  }

  function hypotenuse(a, b) {
    let r = 0;

    if (Math.abs(a) > Math.abs(b)) {
      r = b / a;
      return Math.abs(a) * Math.sqrt(1 + r * r);
    }

    if (b !== 0) {
      r = a / b;
      return Math.abs(b) * Math.sqrt(1 + r * r);
    }

    return 0;
  }

  class QrDecomposition {
    constructor(value) {
      value = WrapperMatrix2D.checkMatrix(value);
      let qr = value.clone();
      let m = value.rows;
      let n = value.columns;
      let rdiag = new Float64Array(n);
      let i, j, k, s;

      for (k = 0; k < n; k++) {
        let nrm = 0;

        for (i = k; i < m; i++) {
          nrm = hypotenuse(nrm, qr.get(i, k));
        }

        if (nrm !== 0) {
          if (qr.get(k, k) < 0) {
            nrm = -nrm;
          }

          for (i = k; i < m; i++) {
            qr.set(i, k, qr.get(i, k) / nrm);
          }

          qr.set(k, k, qr.get(k, k) + 1);

          for (j = k + 1; j < n; j++) {
            s = 0;

            for (i = k; i < m; i++) {
              s += qr.get(i, k) * qr.get(i, j);
            }

            s = -s / qr.get(k, k);

            for (i = k; i < m; i++) {
              qr.set(i, j, qr.get(i, j) + s * qr.get(i, k));
            }
          }
        }

        rdiag[k] = -nrm;
      }

      this.QR = qr;
      this.Rdiag = rdiag;
    }

    solve(value) {
      value = Matrix$2.checkMatrix(value);
      let qr = this.QR;
      let m = qr.rows;

      if (value.rows !== m) {
        throw new Error('Matrix row dimensions must agree');
      }

      if (!this.isFullRank()) {
        throw new Error('Matrix is rank deficient');
      }

      let count = value.columns;
      let X = value.clone();
      let n = qr.columns;
      let i, j, k, s;

      for (k = 0; k < n; k++) {
        for (j = 0; j < count; j++) {
          s = 0;

          for (i = k; i < m; i++) {
            s += qr.get(i, k) * X.get(i, j);
          }

          s = -s / qr.get(k, k);

          for (i = k; i < m; i++) {
            X.set(i, j, X.get(i, j) + s * qr.get(i, k));
          }
        }
      }

      for (k = n - 1; k >= 0; k--) {
        for (j = 0; j < count; j++) {
          X.set(k, j, X.get(k, j) / this.Rdiag[k]);
        }

        for (i = 0; i < k; i++) {
          for (j = 0; j < count; j++) {
            X.set(i, j, X.get(i, j) - X.get(k, j) * qr.get(i, k));
          }
        }
      }

      return X.subMatrix(0, n - 1, 0, count - 1);
    }

    isFullRank() {
      let columns = this.QR.columns;

      for (let i = 0; i < columns; i++) {
        if (this.Rdiag[i] === 0) {
          return false;
        }
      }

      return true;
    }

    get upperTriangularMatrix() {
      let qr = this.QR;
      let n = qr.columns;
      let X = new Matrix$2(n, n);
      let i, j;

      for (i = 0; i < n; i++) {
        for (j = 0; j < n; j++) {
          if (i < j) {
            X.set(i, j, qr.get(i, j));
          } else if (i === j) {
            X.set(i, j, this.Rdiag[i]);
          } else {
            X.set(i, j, 0);
          }
        }
      }

      return X;
    }

    get orthogonalMatrix() {
      let qr = this.QR;
      let rows = qr.rows;
      let columns = qr.columns;
      let X = new Matrix$2(rows, columns);
      let i, j, k, s;

      for (k = columns - 1; k >= 0; k--) {
        for (i = 0; i < rows; i++) {
          X.set(i, k, 0);
        }

        X.set(k, k, 1);

        for (j = k; j < columns; j++) {
          if (qr.get(k, k) !== 0) {
            s = 0;

            for (i = k; i < rows; i++) {
              s += qr.get(i, k) * X.get(i, j);
            }

            s = -s / qr.get(k, k);

            for (i = k; i < rows; i++) {
              X.set(i, j, X.get(i, j) + s * qr.get(i, k));
            }
          }
        }
      }

      return X;
    }

  }

  class SingularValueDecomposition {
    constructor(value, options = {}) {
      value = WrapperMatrix2D.checkMatrix(value);

      if (value.isEmpty()) {
        throw new Error('Matrix must be non-empty');
      }

      let m = value.rows;
      let n = value.columns;
      const {
        computeLeftSingularVectors = true,
        computeRightSingularVectors = true,
        autoTranspose = false
      } = options;
      let wantu = Boolean(computeLeftSingularVectors);
      let wantv = Boolean(computeRightSingularVectors);
      let swapped = false;
      let a;

      if (m < n) {
        if (!autoTranspose) {
          a = value.clone(); // eslint-disable-next-line no-console

          console.warn('Computing SVD on a matrix with more columns than rows. Consider enabling autoTranspose');
        } else {
          a = value.transpose();
          m = a.rows;
          n = a.columns;
          swapped = true;
          let aux = wantu;
          wantu = wantv;
          wantv = aux;
        }
      } else {
        a = value.clone();
      }

      let nu = Math.min(m, n);
      let ni = Math.min(m + 1, n);
      let s = new Float64Array(ni);
      let U = new Matrix$2(m, nu);
      let V = new Matrix$2(n, n);
      let e = new Float64Array(n);
      let work = new Float64Array(m);
      let si = new Float64Array(ni);

      for (let i = 0; i < ni; i++) si[i] = i;

      let nct = Math.min(m - 1, n);
      let nrt = Math.max(0, Math.min(n - 2, m));
      let mrc = Math.max(nct, nrt);

      for (let k = 0; k < mrc; k++) {
        if (k < nct) {
          s[k] = 0;

          for (let i = k; i < m; i++) {
            s[k] = hypotenuse(s[k], a.get(i, k));
          }

          if (s[k] !== 0) {
            if (a.get(k, k) < 0) {
              s[k] = -s[k];
            }

            for (let i = k; i < m; i++) {
              a.set(i, k, a.get(i, k) / s[k]);
            }

            a.set(k, k, a.get(k, k) + 1);
          }

          s[k] = -s[k];
        }

        for (let j = k + 1; j < n; j++) {
          if (k < nct && s[k] !== 0) {
            let t = 0;

            for (let i = k; i < m; i++) {
              t += a.get(i, k) * a.get(i, j);
            }

            t = -t / a.get(k, k);

            for (let i = k; i < m; i++) {
              a.set(i, j, a.get(i, j) + t * a.get(i, k));
            }
          }

          e[j] = a.get(k, j);
        }

        if (wantu && k < nct) {
          for (let i = k; i < m; i++) {
            U.set(i, k, a.get(i, k));
          }
        }

        if (k < nrt) {
          e[k] = 0;

          for (let i = k + 1; i < n; i++) {
            e[k] = hypotenuse(e[k], e[i]);
          }

          if (e[k] !== 0) {
            if (e[k + 1] < 0) {
              e[k] = 0 - e[k];
            }

            for (let i = k + 1; i < n; i++) {
              e[i] /= e[k];
            }

            e[k + 1] += 1;
          }

          e[k] = -e[k];

          if (k + 1 < m && e[k] !== 0) {
            for (let i = k + 1; i < m; i++) {
              work[i] = 0;
            }

            for (let i = k + 1; i < m; i++) {
              for (let j = k + 1; j < n; j++) {
                work[i] += e[j] * a.get(i, j);
              }
            }

            for (let j = k + 1; j < n; j++) {
              let t = -e[j] / e[k + 1];

              for (let i = k + 1; i < m; i++) {
                a.set(i, j, a.get(i, j) + t * work[i]);
              }
            }
          }

          if (wantv) {
            for (let i = k + 1; i < n; i++) {
              V.set(i, k, e[i]);
            }
          }
        }
      }

      let p = Math.min(n, m + 1);

      if (nct < n) {
        s[nct] = a.get(nct, nct);
      }

      if (m < p) {
        s[p - 1] = 0;
      }

      if (nrt + 1 < p) {
        e[nrt] = a.get(nrt, p - 1);
      }

      e[p - 1] = 0;

      if (wantu) {
        for (let j = nct; j < nu; j++) {
          for (let i = 0; i < m; i++) {
            U.set(i, j, 0);
          }

          U.set(j, j, 1);
        }

        for (let k = nct - 1; k >= 0; k--) {
          if (s[k] !== 0) {
            for (let j = k + 1; j < nu; j++) {
              let t = 0;

              for (let i = k; i < m; i++) {
                t += U.get(i, k) * U.get(i, j);
              }

              t = -t / U.get(k, k);

              for (let i = k; i < m; i++) {
                U.set(i, j, U.get(i, j) + t * U.get(i, k));
              }
            }

            for (let i = k; i < m; i++) {
              U.set(i, k, -U.get(i, k));
            }

            U.set(k, k, 1 + U.get(k, k));

            for (let i = 0; i < k - 1; i++) {
              U.set(i, k, 0);
            }
          } else {
            for (let i = 0; i < m; i++) {
              U.set(i, k, 0);
            }

            U.set(k, k, 1);
          }
        }
      }

      if (wantv) {
        for (let k = n - 1; k >= 0; k--) {
          if (k < nrt && e[k] !== 0) {
            for (let j = k + 1; j < n; j++) {
              let t = 0;

              for (let i = k + 1; i < n; i++) {
                t += V.get(i, k) * V.get(i, j);
              }

              t = -t / V.get(k + 1, k);

              for (let i = k + 1; i < n; i++) {
                V.set(i, j, V.get(i, j) + t * V.get(i, k));
              }
            }
          }

          for (let i = 0; i < n; i++) {
            V.set(i, k, 0);
          }

          V.set(k, k, 1);
        }
      }

      let pp = p - 1;
      let eps = Number.EPSILON;

      while (p > 0) {
        let k, kase;

        for (k = p - 2; k >= -1; k--) {
          if (k === -1) {
            break;
          }

          const alpha = Number.MIN_VALUE + eps * Math.abs(s[k] + Math.abs(s[k + 1]));

          if (Math.abs(e[k]) <= alpha || Number.isNaN(e[k])) {
            e[k] = 0;
            break;
          }
        }

        if (k === p - 2) {
          kase = 4;
        } else {
          let ks;

          for (ks = p - 1; ks >= k; ks--) {
            if (ks === k) {
              break;
            }

            let t = (ks !== p ? Math.abs(e[ks]) : 0) + (ks !== k + 1 ? Math.abs(e[ks - 1]) : 0);

            if (Math.abs(s[ks]) <= eps * t) {
              s[ks] = 0;
              break;
            }
          }

          if (ks === k) {
            kase = 3;
          } else if (ks === p - 1) {
            kase = 1;
          } else {
            kase = 2;
            k = ks;
          }
        }

        k++;

        switch (kase) {
          case 1:
            {
              let f = e[p - 2];
              e[p - 2] = 0;

              for (let j = p - 2; j >= k; j--) {
                let t = hypotenuse(s[j], f);
                let cs = s[j] / t;
                let sn = f / t;
                s[j] = t;

                if (j !== k) {
                  f = -sn * e[j - 1];
                  e[j - 1] = cs * e[j - 1];
                }

                if (wantv) {
                  for (let i = 0; i < n; i++) {
                    t = cs * V.get(i, j) + sn * V.get(i, p - 1);
                    V.set(i, p - 1, -sn * V.get(i, j) + cs * V.get(i, p - 1));
                    V.set(i, j, t);
                  }
                }
              }

              break;
            }

          case 2:
            {
              let f = e[k - 1];
              e[k - 1] = 0;

              for (let j = k; j < p; j++) {
                let t = hypotenuse(s[j], f);
                let cs = s[j] / t;
                let sn = f / t;
                s[j] = t;
                f = -sn * e[j];
                e[j] = cs * e[j];

                if (wantu) {
                  for (let i = 0; i < m; i++) {
                    t = cs * U.get(i, j) + sn * U.get(i, k - 1);
                    U.set(i, k - 1, -sn * U.get(i, j) + cs * U.get(i, k - 1));
                    U.set(i, j, t);
                  }
                }
              }

              break;
            }

          case 3:
            {
              const scale = Math.max(Math.abs(s[p - 1]), Math.abs(s[p - 2]), Math.abs(e[p - 2]), Math.abs(s[k]), Math.abs(e[k]));
              const sp = s[p - 1] / scale;
              const spm1 = s[p - 2] / scale;
              const epm1 = e[p - 2] / scale;
              const sk = s[k] / scale;
              const ek = e[k] / scale;
              const b = ((spm1 + sp) * (spm1 - sp) + epm1 * epm1) / 2;
              const c = sp * epm1 * (sp * epm1);
              let shift = 0;

              if (b !== 0 || c !== 0) {
                if (b < 0) {
                  shift = 0 - Math.sqrt(b * b + c);
                } else {
                  shift = Math.sqrt(b * b + c);
                }

                shift = c / (b + shift);
              }

              let f = (sk + sp) * (sk - sp) + shift;
              let g = sk * ek;

              for (let j = k; j < p - 1; j++) {
                let t = hypotenuse(f, g);
                if (t === 0) t = Number.MIN_VALUE;
                let cs = f / t;
                let sn = g / t;

                if (j !== k) {
                  e[j - 1] = t;
                }

                f = cs * s[j] + sn * e[j];
                e[j] = cs * e[j] - sn * s[j];
                g = sn * s[j + 1];
                s[j + 1] = cs * s[j + 1];

                if (wantv) {
                  for (let i = 0; i < n; i++) {
                    t = cs * V.get(i, j) + sn * V.get(i, j + 1);
                    V.set(i, j + 1, -sn * V.get(i, j) + cs * V.get(i, j + 1));
                    V.set(i, j, t);
                  }
                }

                t = hypotenuse(f, g);
                if (t === 0) t = Number.MIN_VALUE;
                cs = f / t;
                sn = g / t;
                s[j] = t;
                f = cs * e[j] + sn * s[j + 1];
                s[j + 1] = -sn * e[j] + cs * s[j + 1];
                g = sn * e[j + 1];
                e[j + 1] = cs * e[j + 1];

                if (wantu && j < m - 1) {
                  for (let i = 0; i < m; i++) {
                    t = cs * U.get(i, j) + sn * U.get(i, j + 1);
                    U.set(i, j + 1, -sn * U.get(i, j) + cs * U.get(i, j + 1));
                    U.set(i, j, t);
                  }
                }
              }

              e[p - 2] = f;
              break;
            }

          case 4:
            {
              if (s[k] <= 0) {
                s[k] = s[k] < 0 ? -s[k] : 0;

                if (wantv) {
                  for (let i = 0; i <= pp; i++) {
                    V.set(i, k, -V.get(i, k));
                  }
                }
              }

              while (k < pp) {
                if (s[k] >= s[k + 1]) {
                  break;
                }

                let t = s[k];
                s[k] = s[k + 1];
                s[k + 1] = t;

                if (wantv && k < n - 1) {
                  for (let i = 0; i < n; i++) {
                    t = V.get(i, k + 1);
                    V.set(i, k + 1, V.get(i, k));
                    V.set(i, k, t);
                  }
                }

                if (wantu && k < m - 1) {
                  for (let i = 0; i < m; i++) {
                    t = U.get(i, k + 1);
                    U.set(i, k + 1, U.get(i, k));
                    U.set(i, k, t);
                  }
                }

                k++;
              }
              p--;
              break;
            }
          // no default
        }
      }

      if (swapped) {
        let tmp = V;
        V = U;
        U = tmp;
      }

      this.m = m;
      this.n = n;
      this.s = s;
      this.U = U;
      this.V = V;
    }

    solve(value) {
      let Y = value;
      let e = this.threshold;
      let scols = this.s.length;
      let Ls = Matrix$2.zeros(scols, scols);

      for (let i = 0; i < scols; i++) {
        if (Math.abs(this.s[i]) <= e) {
          Ls.set(i, i, 0);
        } else {
          Ls.set(i, i, 1 / this.s[i]);
        }
      }

      let U = this.U;
      let V = this.rightSingularVectors;
      let VL = V.mmul(Ls);
      let vrows = V.rows;
      let urows = U.rows;
      let VLU = Matrix$2.zeros(vrows, urows);

      for (let i = 0; i < vrows; i++) {
        for (let j = 0; j < urows; j++) {
          let sum = 0;

          for (let k = 0; k < scols; k++) {
            sum += VL.get(i, k) * U.get(j, k);
          }

          VLU.set(i, j, sum);
        }
      }

      return VLU.mmul(Y);
    }

    solveForDiagonal(value) {
      return this.solve(Matrix$2.diag(value));
    }

    inverse() {
      let V = this.V;
      let e = this.threshold;
      let vrows = V.rows;
      let vcols = V.columns;
      let X = new Matrix$2(vrows, this.s.length);

      for (let i = 0; i < vrows; i++) {
        for (let j = 0; j < vcols; j++) {
          if (Math.abs(this.s[j]) > e) {
            X.set(i, j, V.get(i, j) / this.s[j]);
          }
        }
      }

      let U = this.U;
      let urows = U.rows;
      let ucols = U.columns;
      let Y = new Matrix$2(vrows, urows);

      for (let i = 0; i < vrows; i++) {
        for (let j = 0; j < urows; j++) {
          let sum = 0;

          for (let k = 0; k < ucols; k++) {
            sum += X.get(i, k) * U.get(j, k);
          }

          Y.set(i, j, sum);
        }
      }

      return Y;
    }

    get condition() {
      return this.s[0] / this.s[Math.min(this.m, this.n) - 1];
    }

    get norm2() {
      return this.s[0];
    }

    get rank() {
      let tol = Math.max(this.m, this.n) * this.s[0] * Number.EPSILON;
      let r = 0;
      let s = this.s;

      for (let i = 0, ii = s.length; i < ii; i++) {
        if (s[i] > tol) {
          r++;
        }
      }

      return r;
    }

    get diagonal() {
      return Array.from(this.s);
    }

    get threshold() {
      return Number.EPSILON / 2 * Math.max(this.m, this.n) * this.s[0];
    }

    get leftSingularVectors() {
      return this.U;
    }

    get rightSingularVectors() {
      return this.V;
    }

    get diagonalMatrix() {
      return Matrix$2.diag(this.s);
    }

  }

  function inverse(matrix, useSVD = false) {
    matrix = WrapperMatrix2D.checkMatrix(matrix);

    if (useSVD) {
      return new SingularValueDecomposition(matrix).inverse();
    } else {
      return solve(matrix, Matrix$2.eye(matrix.rows));
    }
  }
  function solve(leftHandSide, rightHandSide, useSVD = false) {
    leftHandSide = WrapperMatrix2D.checkMatrix(leftHandSide);
    rightHandSide = WrapperMatrix2D.checkMatrix(rightHandSide);

    if (useSVD) {
      return new SingularValueDecomposition(leftHandSide).solve(rightHandSide);
    } else {
      return leftHandSide.isSquare() ? new LuDecomposition(leftHandSide).solve(rightHandSide) : new QrDecomposition(leftHandSide).solve(rightHandSide);
    }
  }

  /**
   * Difference of the matrix function over the parameters
   * @ignore
   * @param {{x:Array<number>, y:Array<number>}} data - Array of points to fit in the format [x1, x2, ... ], [y1, y2, ... ]
   * @param {Array<number>} evaluatedData - Array of previous evaluated function values
   * @param {Array<number>} params - Array of previous parameter values
   * @param {number|array} gradientDifference - The step size to approximate the jacobian matrix
   * @param {boolean} centralDifference - If true the jacobian matrix is approximated by central differences otherwise by forward differences
   * @param {function} paramFunction - The parameters and returns a function with the independent variable as a parameter
   * @return {Matrix}
   */

  function gradientFunction(data, evaluatedData, params, gradientDifference, paramFunction, centralDifference) {
    const nbParams = params.length;
    const nbPoints = data.x.length;
    let ans = Matrix$2.zeros(nbParams, nbPoints);
    let rowIndex = 0;

    for (let param = 0; param < nbParams; param++) {
      if (gradientDifference[param] === 0) continue;
      let delta = gradientDifference[param];
      let auxParams = params.slice();
      auxParams[param] += delta;
      let funcParam = paramFunction(auxParams);

      if (!centralDifference) {
        for (let point = 0; point < nbPoints; point++) {
          ans.set(rowIndex, point, (evaluatedData[point] - funcParam(data.x[point])) / delta);
        }
      } else {
        auxParams = params.slice();
        auxParams[param] -= delta;
        delta *= 2;
        let funcParam2 = paramFunction(auxParams);

        for (let point = 0; point < nbPoints; point++) {
          ans.set(rowIndex, point, (funcParam2(data.x[point]) - funcParam(data.x[point])) / delta);
        }
      }

      rowIndex++;
    }

    return ans;
  }

  /**
   * Matrix function over the samples
   * @ignore
   * @param {{x:Array<number>, y:Array<number>}} data - Array of points to fit in the format [x1, x2, ... ], [y1, y2, ... ]
   * @param {Array<number>} evaluatedData - Array of previous evaluated function values
   * @return {Matrix}
   */

  function matrixFunction(data, evaluatedData) {
    const m = data.x.length;
    let ans = new Matrix$2(m, 1);

    for (let point = 0; point < m; point++) {
      ans.set(point, 0, data.y[point] - evaluatedData[point]);
    }

    return ans;
  }
  /**
   * Iteration for Levenberg-Marquardt
   * @ignore
   * @param {{x:Array<number>, y:Array<number>}} data - Array of points to fit in the format [x1, x2, ... ], [y1, y2, ... ]
   * @param {Array<number>} params - Array of previous parameter values
   * @param {number} damping - Levenberg-Marquardt parameter
   * @param {number|array} gradientDifference - The step size to approximate the jacobian matrix
   * @param {boolean} centralDifference - If true the jacobian matrix is approximated by central differences otherwise by forward differences
   * @param {function} parameterizedFunction - The parameters and returns a function with the independent variable as a parameter
   * @return {Array<number>}
   */


  function step(data, params, damping, gradientDifference, parameterizedFunction, centralDifference, weights) {
    let value = damping;
    let identity = Matrix$2.eye(params.length, params.length, value);
    const func = parameterizedFunction(params);
    let evaluatedData = new Float64Array(data.x.length);

    for (let i = 0; i < data.x.length; i++) {
      evaluatedData[i] = func(data.x[i]);
    }

    let gradientFunc = gradientFunction(data, evaluatedData, params, gradientDifference, parameterizedFunction, centralDifference);
    let residualError = matrixFunction(data, evaluatedData);
    let inverseMatrix = inverse(identity.add(gradientFunc.mmul(gradientFunc.transpose().scale('row', {
      scale: weights
    }))));
    let jacobianWeigthResidualError = gradientFunc.mmul(residualError.scale('row', {
      scale: weights
    }));
    let perturbations = inverseMatrix.mmul(jacobianWeigthResidualError);
    return {
      perturbations,
      jacobianWeigthResidualError
    };
  }

  /**
   * Curve fitting algorithm
   * @param {{x:Array<number>, y:Array<number>}} data - Array of points to fit in the format [x1, x2, ... ], [y1, y2, ... ]
   * @param {function} parameterizedFunction - The parameters and returns a function with the independent variable as a parameter
   * @param {object} [options] - Options object
   * @param {number|array} [options.weights = 1] - weighting vector, if the length does not match with the number of data points, the vector is reconstructed with first value.
   * @param {number} [options.damping = 1e-2] - Levenberg-Marquardt parameter, small values of the damping parameter λ result in a Gauss-Newton update and large
  values of λ result in a gradient descent update
   * @param {number} [options.dampingStepDown = 9] - factor to reduce the damping (Levenberg-Marquardt parameter) when there is not an improvement when updating parameters.
   * @param {number} [options.dampingStepUp = 11] - factor to increase the damping (Levenberg-Marquardt parameter) when there is an improvement when updating parameters.
   * @param {number} [options.improvementThreshold = 1e-3] - the threshold to define an improvement through an update of parameters
   * @param {number|array} [options.gradientDifference = 10e-2] - The step size to approximate the jacobian matrix
   * @param {boolean} [options.centralDifference = false] - If true the jacobian matrix is approximated by central differences otherwise by forward differences
   * @param {Array<number>} [options.minValues] - Minimum allowed values for parameters
   * @param {Array<number>} [options.maxValues] - Maximum allowed values for parameters
   * @param {Array<number>} [options.initialValues] - Array of initial parameter values
   * @param {number} [options.maxIterations = 100] - Maximum of allowed iterations
   * @param {number} [options.errorTolerance = 10e-3] - Minimum uncertainty allowed for each point.
   * @param {number} [options.timeout] - maximum time running before throw in seconds.
   * @return {{parameterValues: Array<number>, parameterError: number, iterations: number}}
   */

  function levenbergMarquardt(data, parameterizedFunction, options = {}) {
    let {
      checkTimeout,
      minValues,
      maxValues,
      parameters,
      weightSquare,
      damping,
      dampingStepUp,
      dampingStepDown,
      maxIterations,
      errorTolerance,
      centralDifference,
      gradientDifference,
      improvementThreshold
    } = checkOptions$1(data, parameterizedFunction, options);
    let error = errorCalculation(data, parameters, parameterizedFunction, weightSquare);
    let converged = error <= errorTolerance;
    let iteration = 0;

    for (; iteration < maxIterations && !converged; iteration++) {
      let previousError = error;
      let {
        perturbations,
        jacobianWeigthResidualError
      } = step(data, parameters, damping, gradientDifference, parameterizedFunction, centralDifference, weightSquare);

      for (let k = 0; k < parameters.length; k++) {
        parameters[k] = Math.min(Math.max(minValues[k], parameters[k] - perturbations.get(k, 0)), maxValues[k]);
      }

      error = errorCalculation(data, parameters, parameterizedFunction, weightSquare);
      if (isNaN(error)) break;
      let improvementMetric = (previousError - error) / perturbations.transpose().mmul(perturbations.mulS(damping).add(jacobianWeigthResidualError)).get(0, 0);

      if (improvementMetric > improvementThreshold) {
        damping = Math.max(damping / dampingStepDown, 1e-7);
      } else {
        error = previousError;
        damping = Math.min(damping * dampingStepUp, 1e7);
      }

      if (checkTimeout()) {
        throw new Error(`The execution time is over to ${options.timeout} seconds`);
      }

      converged = error <= errorTolerance;
    }

    return {
      parameterValues: parameters,
      parameterError: error,
      iterations: iteration
    };
  }

  const LEVENBERG_MARQUARDT = 1;
  function selectMethod(optimizationOptions = {}) {
    let {
      kind,
      options
    } = optimizationOptions;
    kind = getKind(kind);

    switch (kind) {
      case LEVENBERG_MARQUARDT:
        return {
          algorithm: levenbergMarquardt,
          optimizationOptions: checkOptions(kind, options)
        };

      default:
        throw new Error(`Unknown kind algorithm`);
    }
  }

  function checkOptions(kind, options = {}) {
    // eslint-disable-next-line default-case
    switch (kind) {
      case LEVENBERG_MARQUARDT:
        return Object.assign({}, lmOptions, options);
    }
  }

  function getKind(kind) {
    if (typeof kind !== 'string') return kind;

    switch (kind.toLowerCase().replace(/[^a-z]/g, '')) {
      case 'lm':
      case 'levenbergmarquardt':
        return LEVENBERG_MARQUARDT;

      default:
        throw new Error(`Unknown kind algorithm`);
    }
  }

  const lmOptions = {
    damping: 1.5,
    maxIterations: 100,
    errorTolerance: 1e-8
  };

  // const STATE_MIN = 1;
  // const STATE_MAX = 2;
  // const STATE_GRADIENT_DIFFERENCE = 3;
  // const X = 0;
  // const Y = 1;
  // const WIDTH = 2;
  // const MU = 3;
  // const keys = ['x', 'y', 'width', 'mu'];

  /**
   * Fits a set of points to the sum of a set of bell functions.
   * @param {object} data - An object containing the x and y data to be fitted.
   * @param {array} peaks - A list of initial parameters to be optimized. e.g. coming from a peak picking [{x, y, width}].
   * @param {object} [options = {}]
   * @param {object} [options.shape={}] - it's specify the kind of shape used to fitting.
   * @param {string} [options.shape.kind = 'gaussian'] - kind of shape; lorentzian, gaussian and pseudovoigt are supported.
   * @param {object} [options.optimization = {}] - it's specify the kind and options of the algorithm use to optimize parameters.
   * @param {object} [options.optimization.kind = 'lm'] - kind of algorithm. By default it's levenberg-marquardt.
   * @param {object} [options.optimization.parameters] - options of each parameter to be optimized e.g. For a gaussian shape
   *  it could have x, y and with properties, each of which could contain init, min, max and gradientDifference, those options will define the guess,
   *  the min and max value of the parameter (search space) and the step size to approximate the jacobian matrix respectively. Those options could be a number,
   *  array of numbers, callback, or array of callbacks. Each kind of shape has default parameters so it could be undefined.
   * @param {object} [options.optimization.parameters.x] - options for x parameter.
   * @param {number|callback|array<number|callback>} [options.optimization.parameters.x.init] - definition of the starting point of the parameter (the guess),
   *  if it is a callback the method pass the peak as the unique input, if it is an array the first element define the guess of the first peak and so on.
   * @param {number|callback|array<number|callback>} [options.optimization.parameters.x.min] - definition of the lower limit of the parameter,
   *  if it is a callback the method pass the peak as the unique input, if it is an array the first element define the min of the first peak and so on.
   * @param {number|callback|array<number|callback>} [options.optimization.parameters.x.max] - definition of the upper limit of the parameter,
   *  if it is a callback the method pass the peak as the unique input, if it is an array the first element define the max of the first peak and so on.
   * @param {number|callback|array<number|callback>} [options.optimization.parameters.x.gradientDifference] - definition of  the step size to approximate the jacobian matrix of the parameter,
   *  if it is a callback the method pass the peak as the unique input, if it is an array the first element define the gradientDifference of the first peak and so on.
   * @param {object} [options.optimization.options = {}] - options for the specific kind of algorithm.
   * @param {number} [options.optimization.options.timeout] - maximum time running before break in seconds.
   * @param {number} [options.optimization.options.damping=1.5]
   * @param {number} [options.optimization.options.maxIterations=100]
   * @param {number} [options.optimization.options.errorTolerance=1e-8]
   * @returns {object} - A object with fitting error and the list of optimized parameters { parameters: [ {x, y, width} ], error } if the kind of shape is pseudoVoigt mu parameter is optimized.
   */

  function optimize(data, peakList, options = {}) {
    const {
      y,
      x,
      maxY,
      peaks,
      paramsFunc,
      optimization
    } = checkInput(data, peakList, options);
    let parameters = optimization.parameters;
    let nbShapes = peaks.length;
    let parameterKey = Object.keys(parameters);
    let nbParams = nbShapes * parameterKey.length;
    let pMin = new Float64Array(nbParams);
    let pMax = new Float64Array(nbParams);
    let pInit = new Float64Array(nbParams);
    let gradientDifference = new Float64Array(nbParams);

    for (let i = 0; i < nbShapes; i++) {
      let peak = peaks[i];

      for (let k = 0; k < parameterKey.length; k++) {
        let key = parameterKey[k];
        let init = parameters[key].init;
        let min = parameters[key].min;
        let max = parameters[key].max;
        let gradientDifferenceValue = parameters[key].gradientDifference;
        pInit[i + k * nbShapes] = init[i % init.length](peak);
        pMin[i + k * nbShapes] = min[i % min.length](peak);
        pMax[i + k * nbShapes] = max[i % max.length](peak);
        gradientDifference[i + k * nbShapes] = gradientDifferenceValue[i % gradientDifferenceValue.length](peak);
      }
    }

    let {
      algorithm,
      optimizationOptions
    } = selectMethod(optimization);
    optimizationOptions.minValues = pMin;
    optimizationOptions.maxValues = pMax;
    optimizationOptions.initialValues = pInit;
    optimizationOptions.gradientDifference = gradientDifference;
    let pFit = algorithm({
      x,
      y
    }, paramsFunc, optimizationOptions);
    let {
      parameterError: error,
      iterations
    } = pFit;
    let result = {
      error,
      iterations,
      peaks
    };

    for (let i = 0; i < nbShapes; i++) {
      pFit.parameterValues[i + nbShapes] *= maxY;

      for (let k = 0; k < parameterKey.length; k++) {
        // we modify the optimized parameters
        peaks[i][parameterKey[k]] = pFit.parameterValues[i + k * nbShapes];
      }
    }

    return result;
  }

  /**
   * Returns the closest index of a `target` in an ordered array
   * @param {array<Number>} array
   * @param {number} target
   */
  function xFindClosestIndex$2(array, target) {
    let low = 0;
    let high = array.length - 1;
    let middle = 0;

    while (high - low > 1) {
      middle = low + (high - low >> 1);

      if (array[middle] < target) {
        low = middle;
      } else if (array[middle] > target) {
        high = middle;
      } else {
        return middle;
      }
    }

    if (low < array.length - 1) {
      if (Math.abs(target - array[low]) < Math.abs(array[low + 1] - target)) {
        return low;
      } else {
        return low + 1;
      }
    } else {
      return low;
    }
  }

  /**
   * Returns an object with {fromIndex, toIndex} for a specific from / to
   * @param {array} x
   * @param {object} [options={}]
   * @param {number} [options.from] - First value for xyIntegration in the X scale
   * @param {number} [options.fromIndex=0] - First point for xyIntegration
   * @param {number} [options.to] - Last value for xyIntegration in the X scale
   * @param {number} [options.toIndex=x.length-1] - Last point for xyIntegration
   */

  function xGetFromToIndex$1(x, options = {}) {
    let {
      fromIndex,
      toIndex,
      from,
      to
    } = options;

    if (fromIndex === undefined) {
      if (from !== undefined) {
        fromIndex = xFindClosestIndex$2(x, from);
      } else {
        fromIndex = 0;
      }
    }

    if (toIndex === undefined) {
      if (to !== undefined) {
        toIndex = xFindClosestIndex$2(x, to);
      } else {
        toIndex = x.length - 1;
      }
    }

    if (fromIndex > toIndex) [fromIndex, toIndex] = [toIndex, fromIndex];
    return {
      fromIndex,
      toIndex
    };
  }

  /**
   * Group peaks based on factor and add group property in peaks
   * @param {array} peakList
   * @param {number} factor
   */
  function groupPeaks(peakList, factor = 1) {
    if (peakList.length === 0) return [];
    let peaks = peakList.sort((a, b) => a.x - b.x);
    let previousPeak = {
      x: Number.NEGATIVE_INFINITY,
      width: 1
    };
    let currentGroup = [previousPeak];
    let groups = [];

    for (let peak of peaks) {
      if ((peak.x - previousPeak.x) / (peak.width + previousPeak.width) <= factor / 2) {
        currentGroup.push(peak);
      } else {
        currentGroup = [peak];
        groups.push(currentGroup);
      }

      peak.group = groups.length - 1;
      previousPeak = peak;
    }

    return groups;
  }

  /**
   * Optimize the position (x), max intensity (y), full width at half maximum (width)
   * and the ratio of gaussian contribution (mu) if it's required. It supports three kind of shapes: gaussian, lorentzian and pseudovoigt
   * @param {object} data - An object containing the x and y data to be fitted.
   * @param {Array} peakList - A list of initial parameters to be optimized. e.g. coming from a peak picking [{x, y, width}].
   * @param {object} [options = {}] -
   * @param {number} [options.factorWidth = 1] - times of width to group peaks.
   * @param {number} [options.factorLimits = 2] - times of width to use to optimize peaks
   * @param {object} [options.shape={}] - it's specify the kind of shape used to fitting.
   * @param {string} [options.shape.kind='gaussian'] - kind of shape; lorentzian, gaussian and pseudovoigt are supported.
   * @param {string} [options.shape.options={}] - options depending the kind of shape
   * @param {object} [options.optimization={}] - it's specify the kind and options of the algorithm use to optimize parameters.
   * @param {string} [options.optimization.kind='lm'] - kind of algorithm. By default it's levenberg-marquardt.
   * @param {object} [options.optimization.options={}] - options for the specific kind of algorithm.
   * @param {number} [options.optimization.options.timeout=10] - maximum time running before break in seconds.
   */

  function optimizePeaks(data, peakList, options = {}) {
    const {
      factorWidth = 1,
      factorLimits = 2,
      shape = {
        kind: 'gaussian'
      },
      optimization = {
        kind: 'lm',
        options: {
          timeout: 10
        }
      }
    } = options;

    if (data.x[0] > data.x[1]) {
      data.x.reverse();
      data.y.reverse();
    }

    let groups = groupPeaks(peakList, factorWidth);
    let results = [];

    for (const peaks of groups) {
      const firstPeak = peaks[0];
      const lastPeak = peaks[peaks.length - 1];
      const from = firstPeak.x - firstPeak.width * factorLimits;
      const to = lastPeak.x + lastPeak.width * factorLimits;
      const {
        fromIndex,
        toIndex
      } = xGetFromToIndex$1(data.x, {
        from,
        to
      }); // Multiple peaks

      const currentRange = {
        x: data.x.slice(fromIndex, toIndex),
        y: data.y.slice(fromIndex, toIndex)
      };

      if (currentRange.x.length > 5) {
        let {
          peaks: optimizedPeaks
        } = optimize(currentRange, peaks, {
          shape,
          optimization
        });
        results = results.concat(optimizedPeaks);
      } else {
        results = results.concat(peaks);
      }
    }

    return results;
  }

  /**
   * This function try to join the peaks that seems to belong to a broad signal in a single broad peak.
   * @param {Array} peakList - A list of initial parameters to be optimized. e.g. coming from a peak picking [{x, y, width}].
   * @param {object} [options = {}] - options
   * @param {number} [options.width=0.25] - width limit to join peaks.
   * @param {object} [options.shape={}] - it's specify the kind of shape used to fitting.
   * @param {string} [options.shape.kind = 'gaussian'] - kind of shape; lorentzian, gaussian and pseudovoigt are supported.
   * @param {object} [options.optimization = {}] - it's specify the kind and options of the algorithm use to optimize parameters.
   * @param {string} [options.optimization.kind = 'lm'] - kind of algorithm. By default it's levenberg-marquardt.
   * @param {number} [options.optimization.options.timeout = 10] - maximum time running before break in seconds.
   * @param {object} [options.optimization.options = {}] - options for the specific kind of algorithm.
   */

  function joinBroadPeaks(peakList, options = {}) {
    let {
      width = 0.25,
      shape = {
        kind: 'gaussian'
      },
      optimization = {
        kind: 'lm',
        timeout: 10
      }
    } = options;
    let broadLines = []; // Optimize the possible broad lines

    let max = 0;
    let maxI = 0;
    let count = 1;

    for (let i = peakList.length - 1; i >= 0; i--) {
      if (peakList[i].soft) {
        broadLines.push(peakList.splice(i, 1)[0]);
      }
    } // Push a feke peak


    broadLines.push({
      x: Number.MAX_VALUE
    });
    let candidates = {
      x: [broadLines[0].x],
      y: [broadLines[0].y]
    };
    let indexes = [0];

    for (let i = 1; i < broadLines.length; i++) {
      if (Math.abs(broadLines[i - 1].x - broadLines[i].x) < width) {
        candidates.x.push(broadLines[i].x);
        candidates.y.push(broadLines[i].y);

        if (broadLines[i].y > max) {
          max = broadLines[i].y;
          maxI = i;
        }

        indexes.push(i);
        count++;
      } else {
        if (count > 2) {
          let fitted = optimize(candidates, [{
            x: broadLines[maxI].x,
            y: max,
            width: Math.abs(candidates.x[0] - candidates.x[candidates.x.length - 1])
          }], {
            shape,
            optimization
          });
          let {
            peaks: peak
          } = fitted;
          peak[0].index = Math.floor(indexes.reduce((a, b) => a + b, 0) / indexes.length);
          peak[0].soft = false;
          peakList.push(peak[0]);
        } else {
          // Put back the candidates to the signals list
          indexes.forEach(index => {
            peakList.push(broadLines[index]);
          });
        }

        candidates = {
          x: [broadLines[i].x],
          y: [broadLines[i].y]
        };
        indexes = [i];
        max = broadLines[i].y;
        maxI = i;
        count = 1;
      }
    }

    peakList.sort(function (a, b) {
      return a.x - b.x;
    });
    return peakList;
  }

  /**
   * This method will allow to enlarge peaks and prevent overlap between peaks
   * Because peaks may not be symmetric after we add 2 properties, from and to.
   * @param {Array} peakList
   * @param {object} [options={}]
   * @param {number} [options.factor=2]
   * @param {boolean} [options.overlap=false] by default we don't allow overlap
   * @return {Array} peakList
   */
  function broadenPeaks(peakList, options = {}) {
    const {
      factor = 2,
      overlap = false
    } = options;

    for (let peak of peakList) {
      if (!peak.right || !peak.left) {
        peak.from = peak.x - peak.width / 2 * factor;
        peak.to = peak.x + peak.width / 2 * factor;
      } else {
        peak.from = peak.x - (peak.x - peak.left.x) * factor;
        peak.to = peak.x + (peak.right.x - peak.x) * factor;
      }
    }

    if (!overlap) {
      for (let i = 0; i < peakList.length - 1; i++) {
        let peak = peakList[i];
        let nextPeak = peakList[i + 1];

        if (peak.to > nextPeak.from) {
          peak.to = nextPeak.from = (peak.to + nextPeak.from) / 2;
        }
      }
    }

    for (let peak of peakList) {
      peak.width = peak.to - peak.from;
    }

    return peakList;
  }

  var src$p = /*#__PURE__*/Object.freeze({
    __proto__: null,
    gsd: gsd$2,
    optimizePeaks: optimizePeaks,
    joinBroadPeaks: joinBroadPeaks,
    broadenPeaks: broadenPeaks
  });

  /**
   * Parse a text-file and convert it to an array of XY points
   * @param {string} text - csv or tsv strings
   * @param {object} [options={}]
   * @param {boolean} [options.rescale = false] - will set the maximum value to 1
   * @param {boolean} [options.uniqueX = false] - Make the X values unique (works only with 'xxyy' format). If the X value is repeated the sum of Y is done.
   * @param {number} [options.xColumn = 0] - A number that specifies the x column
   * @param {number} [options.yColumn = 1] - A number that specifies the y column
   * @param {boolean} [options.bestGuess=false] Will try to guess which columns are the best
   * @param {number} [options.numberColumns=Number.MAX_SAFE_INTEGER] If the file has 10 columns and you specify here 2 it will reflow the file
   * @param {number} [options.maxNumberColumns = (Math.max(xColumn, yColumn)+1)] - A number that specifies the maximum number of y columns
   * @param {number} [options.minNumberColumns = (Math.min(xColumn, yColumn)+1)] - A number that specifies the minimum number of y columns
   * @param {boolean} [options.keepInfo = false] - shoud we keep the non numeric lines. In this case the system will return an object {data, info}
   * @return {object{x:<Array<number>>,y:<Array<number>>}
   */

  function parseXY$1(text, options = {}) {
    let {
      rescale = false,
      uniqueX: uniqueX$1 = false,
      xColumn = 0,
      yColumn = 1,
      keepInfo = false,
      bestGuess = false,
      numberColumns = Number.MAX_SAFE_INTEGER,
      maxNumberColumns = Number.MAX_SAFE_INTEGER,
      minNumberColumns = 2
    } = options;
    maxNumberColumns = Math.max(maxNumberColumns, xColumn + 1, yColumn + 1);
    minNumberColumns = Math.max(xColumn + 1, yColumn + 1, minNumberColumns);
    let lines = text.split(/[\r\n]+/);
    let matrix = [];
    let info = [];
    let position = 0;

    for (let l = 0; l < lines.length; l++) {
      let line = lines[l].trim(); // we will consider only lines that contains only numbers

      if (line.match(/[0-9]+/) && line.match(/^[0-9eE,;. \t+-]+$/)) {
        let fields = line.split(/,[; \t]+|[; \t]+/);

        if (fields.length === 1) {
          fields = line.split(/[,; \t]+/);
        }

        if (fields && fields.length >= minNumberColumns && // we filter lines that have not enough or too many columns
        fields.length <= maxNumberColumns) {
          matrix.push(fields.map(value => parseFloat(value.replace(',', '.'))));
          position++;
        }
      } else if (line) {
        info.push({
          position,
          value: line
        });
      }
    }

    if (bestGuess) {
      if (matrix[0] && matrix[0].length === 3 && options.xColumn === undefined && options.yColumn === undefined) {
        // is the first column a seuqnetial number ?
        let skipFirstColumn = true;

        for (let i = 0; i < matrix.length - 1; i++) {
          if (Math.abs(matrix[i][0] - matrix[i + 1][0]) !== 1) {
            skipFirstColumn = false;
          }
        }

        if (skipFirstColumn) {
          xColumn = 1;
          yColumn = 2;
        }
      }

      if (matrix[0] && matrix[0].length > 3) {
        let xs = [];

        for (let row of matrix) {
          for (let i = xColumn; i < row.length; i += 2) {
            xs.push(row[i]);
          }
        }

        if (xIsMonotone$1(xs)) {
          numberColumns = 2;
        }
      }
    }

    if (numberColumns) {
      const newMatrix = [];

      for (const row of matrix) {
        for (let i = 0; i < row.length; i += numberColumns) {
          newMatrix.push(row.slice(i, i + numberColumns));
        }
      }

      matrix = newMatrix;
    }

    const result = {
      x: matrix.map(row => row[xColumn]),
      y: matrix.map(row => row[yColumn])
    };

    if (uniqueX$1) {
      uniqueX(result);
    }

    if (rescale) {
      let maxY = max$7(result.y);

      for (let i = 0; i < result.y.length; i++) {
        result.y[i] /= maxY;
      }
    }

    if (!keepInfo) return result;
    return {
      info,
      data: result
    };
  }

  var src$o = /*#__PURE__*/Object.freeze({
    __proto__: null,
    parseXY: parseXY$1
  });

  /**
   * Filter the array by taking the higher peaks and only
   * keep one per slot.
   * There are 2 different slots, the smallest one will have the
   * extra annotation `close` to true
   * @param {array} peaks - array of all the peaks
   * @param {object} [options={}]
   * @param {number} [options.from] - min X value of the window to consider
   * @param {number} [options.to] - max X value of the window to consider
   * @param {number} [options.searchMonoisotopicRatio=0] - search previous peaks with at least ratio height
   * @param {number} [options.limit=20] - max number of peaks
   * @param {number} [options.threshold=0.01] - minimal intensity compare to base peak
   * @param {number} [options.numberSlots=10] - define the number of slots and indirectly the slot width
   * @param {number} [options.numberCloseSlots=50]
   * @returns {array} - copy of peaks with 'close' annotation
   */

  function getBestPeaks$1(peaks, options = {}) {
    const {
      searchMonoisotopicRatio = 0,
      from = peaks.reduce((previous, peak) => Math.min(peak.x, previous), Number.MAX_SAFE_INTEGER),
      to = peaks.reduce((previous, peak) => Math.max(peak.x, previous), Number.MIN_SAFE_INTEGER),
      limit = 20,
      threshold = 0.01,
      numberCloseSlots = 50,
      numberSlots = 10
    } = options;
    let slot = (to - from) / numberSlots;
    let closeSlot = (to - from) / numberCloseSlots;
    let selected = peaks.filter(peak => peak.x >= from && peak.x <= to).map(peak => {
      return {
        peak,
        monoisotopic: false
      };
    });

    if (searchMonoisotopicRatio) {
      selected = selected.sort((a, b) => b.peak.x - a.peak.x);

      for (let i = 0; i < selected.length; i++) {
        let item = selected[i];

        for (let j = i + 1; j < selected.length; j++) {
          let nextItem = selected[j];
          if (item.peak.x - nextItem.peak.x < 0.09) continue;
          if (item.peak.x - nextItem.peak.x > 1.1) break;

          if (nextItem.peak.y > item.peak.y * searchMonoisotopicRatio) {
            item.monoisotopic = false;
            nextItem.monoisotopic = true;
            break;
          }
        }
      }
    }

    selected = selected.sort((a, b) => {
      if (a.monoisotopic && !b.monoisotopic) return -1;
      if (b.monoisotopic && !a.monoisotopic) return 1;
      return b.peak.y - a.peak.y;
    });
    let toReturn = [];
    if (selected.length === 0) return [];
    let minY = selected[0].peak.y * threshold;

    peakLoop: for (let item of selected) {
      if (item.peak.y < minY) {
        if (item.monoisotopic) {
          continue;
        } else {
          break;
        }
      }

      let close = false;

      for (let existing of toReturn) {
        if (Math.abs(existing.x - item.peak.x) < closeSlot) {
          continue peakLoop;
        }

        if (Math.abs(existing.x - item.peak.x) < slot) {
          close = true;
        }
      }

      let newPeak = JSON.parse(JSON.stringify(item.peak));
      newPeak.close = close;
      toReturn.push(newPeak);
      if (toReturn.length === limit) break;
    }

    return toReturn.sort((a, b) => a.x - b.x);
  }

  var getBestPeaks_1 = getBestPeaks$1;

  var array$4 = createCommonjsModule$1(function (module, exports) {

    function compareNumbers(a, b) {
      return a - b;
    }
    /**
     * Computes the sum of the given values
     * @param {Array} values
     * @returns {number}
     */


    exports.sum = function sum(values) {
      var sum = 0;

      for (var i = 0; i < values.length; i++) {
        sum += values[i];
      }

      return sum;
    };
    /**
     * Computes the maximum of the given values
     * @param {Array} values
     * @returns {number}
     */


    exports.max = function max(values) {
      var max = values[0];
      var l = values.length;

      for (var i = 1; i < l; i++) {
        if (values[i] > max) max = values[i];
      }

      return max;
    };
    /**
     * Computes the minimum of the given values
     * @param {Array} values
     * @returns {number}
     */


    exports.min = function min(values) {
      var min = values[0];
      var l = values.length;

      for (var i = 1; i < l; i++) {
        if (values[i] < min) min = values[i];
      }

      return min;
    };
    /**
     * Computes the min and max of the given values
     * @param {Array} values
     * @returns {{min: number, max: number}}
     */


    exports.minMax = function minMax(values) {
      var min = values[0];
      var max = values[0];
      var l = values.length;

      for (var i = 1; i < l; i++) {
        if (values[i] < min) min = values[i];
        if (values[i] > max) max = values[i];
      }

      return {
        min: min,
        max: max
      };
    };
    /**
     * Computes the arithmetic mean of the given values
     * @param {Array} values
     * @returns {number}
     */


    exports.arithmeticMean = function arithmeticMean(values) {
      var sum = 0;
      var l = values.length;

      for (var i = 0; i < l; i++) {
        sum += values[i];
      }

      return sum / l;
    };
    /**
     * {@link arithmeticMean}
     */


    exports.mean = exports.arithmeticMean;
    /**
     * Computes the geometric mean of the given values
     * @param {Array} values
     * @returns {number}
     */

    exports.geometricMean = function geometricMean(values) {
      var mul = 1;
      var l = values.length;

      for (var i = 0; i < l; i++) {
        mul *= values[i];
      }

      return Math.pow(mul, 1 / l);
    };
    /**
     * Computes the mean of the log of the given values
     * If the return value is exponentiated, it gives the same result as the
     * geometric mean.
     * @param {Array} values
     * @returns {number}
     */


    exports.logMean = function logMean(values) {
      var lnsum = 0;
      var l = values.length;

      for (var i = 0; i < l; i++) {
        lnsum += Math.log(values[i]);
      }

      return lnsum / l;
    };
    /**
     * Computes the weighted grand mean for a list of means and sample sizes
     * @param {Array} means - Mean values for each set of samples
     * @param {Array} samples - Number of original values for each set of samples
     * @returns {number}
     */


    exports.grandMean = function grandMean(means, samples) {
      var sum = 0;
      var n = 0;
      var l = means.length;

      for (var i = 0; i < l; i++) {
        sum += samples[i] * means[i];
        n += samples[i];
      }

      return sum / n;
    };
    /**
     * Computes the truncated mean of the given values using a given percentage
     * @param {Array} values
     * @param {number} percent - The percentage of values to keep (range: [0,1])
     * @param {boolean} [alreadySorted=false]
     * @returns {number}
     */


    exports.truncatedMean = function truncatedMean(values, percent, alreadySorted) {
      if (alreadySorted === undefined) alreadySorted = false;

      if (!alreadySorted) {
        values = [].concat(values).sort(compareNumbers);
      }

      var l = values.length;
      var k = Math.floor(l * percent);
      var sum = 0;

      for (var i = k; i < l - k; i++) {
        sum += values[i];
      }

      return sum / (l - 2 * k);
    };
    /**
     * Computes the harmonic mean of the given values
     * @param {Array} values
     * @returns {number}
     */


    exports.harmonicMean = function harmonicMean(values) {
      var sum = 0;
      var l = values.length;

      for (var i = 0; i < l; i++) {
        if (values[i] === 0) {
          throw new RangeError('value at index ' + i + 'is zero');
        }

        sum += 1 / values[i];
      }

      return l / sum;
    };
    /**
     * Computes the contraharmonic mean of the given values
     * @param {Array} values
     * @returns {number}
     */


    exports.contraHarmonicMean = function contraHarmonicMean(values) {
      var r1 = 0;
      var r2 = 0;
      var l = values.length;

      for (var i = 0; i < l; i++) {
        r1 += values[i] * values[i];
        r2 += values[i];
      }

      if (r2 < 0) {
        throw new RangeError('sum of values is negative');
      }

      return r1 / r2;
    };
    /**
     * Computes the median of the given values
     * @param {Array} values
     * @param {boolean} [alreadySorted=false]
     * @returns {number}
     */


    exports.median = function median(values, alreadySorted) {
      if (alreadySorted === undefined) alreadySorted = false;

      if (!alreadySorted) {
        values = [].concat(values).sort(compareNumbers);
      }

      var l = values.length;
      var half = Math.floor(l / 2);

      if (l % 2 === 0) {
        return (values[half - 1] + values[half]) * 0.5;
      } else {
        return values[half];
      }
    };
    /**
     * Computes the variance of the given values
     * @param {Array} values
     * @param {boolean} [unbiased=true] - if true, divide by (n-1); if false, divide by n.
     * @returns {number}
     */


    exports.variance = function variance(values, unbiased) {
      if (unbiased === undefined) unbiased = true;
      var theMean = exports.mean(values);
      var theVariance = 0;
      var l = values.length;

      for (var i = 0; i < l; i++) {
        var x = values[i] - theMean;
        theVariance += x * x;
      }

      if (unbiased) {
        return theVariance / (l - 1);
      } else {
        return theVariance / l;
      }
    };
    /**
     * Computes the standard deviation of the given values
     * @param {Array} values
     * @param {boolean} [unbiased=true] - if true, divide by (n-1); if false, divide by n.
     * @returns {number}
     */


    exports.standardDeviation = function standardDeviation(values, unbiased) {
      return Math.sqrt(exports.variance(values, unbiased));
    };

    exports.standardError = function standardError(values) {
      return exports.standardDeviation(values) / Math.sqrt(values.length);
    };
    /**
     * IEEE Transactions on biomedical engineering, vol. 52, no. 1, january 2005, p. 76-
     * Calculate the standard deviation via the Median of the absolute deviation
     *  The formula for the standard deviation only holds for Gaussian random variables.
     * @returns {{mean: number, stdev: number}}
     */


    exports.robustMeanAndStdev = function robustMeanAndStdev(y) {
      var mean = 0,
          stdev = 0;
      var length = y.length,
          i = 0;

      for (i = 0; i < length; i++) {
        mean += y[i];
      }

      mean /= length;
      var averageDeviations = new Array(length);

      for (i = 0; i < length; i++) averageDeviations[i] = Math.abs(y[i] - mean);

      averageDeviations.sort(compareNumbers);

      if (length % 2 === 1) {
        stdev = averageDeviations[(length - 1) / 2] / 0.6745;
      } else {
        stdev = 0.5 * (averageDeviations[length / 2] + averageDeviations[length / 2 - 1]) / 0.6745;
      }

      return {
        mean: mean,
        stdev: stdev
      };
    };

    exports.quartiles = function quartiles(values, alreadySorted) {
      if (typeof alreadySorted === 'undefined') alreadySorted = false;

      if (!alreadySorted) {
        values = [].concat(values).sort(compareNumbers);
      }

      var quart = values.length / 4;
      var q1 = values[Math.ceil(quart) - 1];
      var q2 = exports.median(values, true);
      var q3 = values[Math.ceil(quart * 3) - 1];
      return {
        q1: q1,
        q2: q2,
        q3: q3
      };
    };

    exports.pooledStandardDeviation = function pooledStandardDeviation(samples, unbiased) {
      return Math.sqrt(exports.pooledVariance(samples, unbiased));
    };

    exports.pooledVariance = function pooledVariance(samples, unbiased) {
      if (typeof unbiased === 'undefined') unbiased = true;
      var sum = 0;
      var length = 0,
          l = samples.length;

      for (var i = 0; i < l; i++) {
        var values = samples[i];
        var vari = exports.variance(values);
        sum += (values.length - 1) * vari;
        if (unbiased) length += values.length - 1;else length += values.length;
      }

      return sum / length;
    };

    exports.mode = function mode(values) {
      var l = values.length,
          itemCount = new Array(l),
          i;

      for (i = 0; i < l; i++) {
        itemCount[i] = 0;
      }

      var itemArray = new Array(l);
      var count = 0;

      for (i = 0; i < l; i++) {
        var index = itemArray.indexOf(values[i]);
        if (index >= 0) itemCount[index]++;else {
          itemArray[count] = values[i];
          itemCount[count] = 1;
          count++;
        }
      }

      var maxValue = 0,
          maxIndex = 0;

      for (i = 0; i < count; i++) {
        if (itemCount[i] > maxValue) {
          maxValue = itemCount[i];
          maxIndex = i;
        }
      }

      return itemArray[maxIndex];
    };

    exports.covariance = function covariance(vector1, vector2, unbiased) {
      if (typeof unbiased === 'undefined') unbiased = true;
      var mean1 = exports.mean(vector1);
      var mean2 = exports.mean(vector2);
      if (vector1.length !== vector2.length) throw 'Vectors do not have the same dimensions';
      var cov = 0,
          l = vector1.length;

      for (var i = 0; i < l; i++) {
        var x = vector1[i] - mean1;
        var y = vector2[i] - mean2;
        cov += x * y;
      }

      if (unbiased) return cov / (l - 1);else return cov / l;
    };

    exports.skewness = function skewness(values, unbiased) {
      if (typeof unbiased === 'undefined') unbiased = true;
      var theMean = exports.mean(values);
      var s2 = 0,
          s3 = 0,
          l = values.length;

      for (var i = 0; i < l; i++) {
        var dev = values[i] - theMean;
        s2 += dev * dev;
        s3 += dev * dev * dev;
      }

      var m2 = s2 / l;
      var m3 = s3 / l;
      var g = m3 / Math.pow(m2, 3 / 2.0);

      if (unbiased) {
        var a = Math.sqrt(l * (l - 1));
        var b = l - 2;
        return a / b * g;
      } else {
        return g;
      }
    };

    exports.kurtosis = function kurtosis(values, unbiased) {
      if (typeof unbiased === 'undefined') unbiased = true;
      var theMean = exports.mean(values);
      var n = values.length,
          s2 = 0,
          s4 = 0;

      for (var i = 0; i < n; i++) {
        var dev = values[i] - theMean;
        s2 += dev * dev;
        s4 += dev * dev * dev * dev;
      }

      var m2 = s2 / n;
      var m4 = s4 / n;

      if (unbiased) {
        var v = s2 / (n - 1);
        var a = n * (n + 1) / ((n - 1) * (n - 2) * (n - 3));
        var b = s4 / (v * v);
        var c = (n - 1) * (n - 1) / ((n - 2) * (n - 3));
        return a * b - 3 * c;
      } else {
        return m4 / (m2 * m2) - 3;
      }
    };

    exports.entropy = function entropy(values, eps) {
      if (typeof eps === 'undefined') eps = 0;
      var sum = 0,
          l = values.length;

      for (var i = 0; i < l; i++) sum += values[i] * Math.log(values[i] + eps);

      return -sum;
    };

    exports.weightedMean = function weightedMean(values, weights) {
      var sum = 0,
          l = values.length;

      for (var i = 0; i < l; i++) sum += values[i] * weights[i];

      return sum;
    };

    exports.weightedStandardDeviation = function weightedStandardDeviation(values, weights) {
      return Math.sqrt(exports.weightedVariance(values, weights));
    };

    exports.weightedVariance = function weightedVariance(values, weights) {
      var theMean = exports.weightedMean(values, weights);
      var vari = 0,
          l = values.length;
      var a = 0,
          b = 0;

      for (var i = 0; i < l; i++) {
        var z = values[i] - theMean;
        var w = weights[i];
        vari += w * (z * z);
        b += w;
        a += w * w;
      }

      return vari * (b / (b * b - a));
    };

    exports.center = function center(values, inPlace) {
      if (typeof inPlace === 'undefined') inPlace = false;
      var result = values;
      if (!inPlace) result = [].concat(values);
      var theMean = exports.mean(result),
          l = result.length;

      for (var i = 0; i < l; i++) result[i] -= theMean;
    };

    exports.standardize = function standardize(values, standardDev, inPlace) {
      if (typeof standardDev === 'undefined') standardDev = exports.standardDeviation(values);
      if (typeof inPlace === 'undefined') inPlace = false;
      var l = values.length;
      var result = inPlace ? values : new Array(l);

      for (var i = 0; i < l; i++) result[i] = values[i] / standardDev;

      return result;
    };

    exports.cumulativeSum = function cumulativeSum(array) {
      var l = array.length;
      var result = new Array(l);
      result[0] = array[0];

      for (var i = 1; i < l; i++) result[i] = result[i - 1] + array[i];

      return result;
    };
  });

  var matrix$3 = createCommonjsModule$1(function (module, exports) {

    function compareNumbers(a, b) {
      return a - b;
    }

    exports.max = function max(matrix) {
      var max = -Infinity;

      for (var i = 0; i < matrix.length; i++) {
        for (var j = 0; j < matrix[i].length; j++) {
          if (matrix[i][j] > max) max = matrix[i][j];
        }
      }

      return max;
    };

    exports.min = function min(matrix) {
      var min = Infinity;

      for (var i = 0; i < matrix.length; i++) {
        for (var j = 0; j < matrix[i].length; j++) {
          if (matrix[i][j] < min) min = matrix[i][j];
        }
      }

      return min;
    };

    exports.minMax = function minMax(matrix) {
      var min = Infinity;
      var max = -Infinity;

      for (var i = 0; i < matrix.length; i++) {
        for (var j = 0; j < matrix[i].length; j++) {
          if (matrix[i][j] < min) min = matrix[i][j];
          if (matrix[i][j] > max) max = matrix[i][j];
        }
      }

      return {
        min: min,
        max: max
      };
    };

    exports.entropy = function entropy(matrix, eps) {
      if (typeof eps === 'undefined') {
        eps = 0;
      }

      var sum = 0,
          l1 = matrix.length,
          l2 = matrix[0].length;

      for (var i = 0; i < l1; i++) {
        for (var j = 0; j < l2; j++) {
          sum += matrix[i][j] * Math.log(matrix[i][j] + eps);
        }
      }

      return -sum;
    };

    exports.mean = function mean(matrix, dimension) {
      if (typeof dimension === 'undefined') {
        dimension = 0;
      }

      var rows = matrix.length,
          cols = matrix[0].length,
          theMean,
          N,
          i,
          j;

      if (dimension === -1) {
        theMean = [0];
        N = rows * cols;

        for (i = 0; i < rows; i++) {
          for (j = 0; j < cols; j++) {
            theMean[0] += matrix[i][j];
          }
        }

        theMean[0] /= N;
      } else if (dimension === 0) {
        theMean = new Array(cols);
        N = rows;

        for (j = 0; j < cols; j++) {
          theMean[j] = 0;

          for (i = 0; i < rows; i++) {
            theMean[j] += matrix[i][j];
          }

          theMean[j] /= N;
        }
      } else if (dimension === 1) {
        theMean = new Array(rows);
        N = cols;

        for (j = 0; j < rows; j++) {
          theMean[j] = 0;

          for (i = 0; i < cols; i++) {
            theMean[j] += matrix[j][i];
          }

          theMean[j] /= N;
        }
      } else {
        throw new Error('Invalid dimension');
      }

      return theMean;
    };

    exports.sum = function sum(matrix, dimension) {
      if (typeof dimension === 'undefined') {
        dimension = 0;
      }

      var rows = matrix.length,
          cols = matrix[0].length,
          theSum,
          i,
          j;

      if (dimension === -1) {
        theSum = [0];

        for (i = 0; i < rows; i++) {
          for (j = 0; j < cols; j++) {
            theSum[0] += matrix[i][j];
          }
        }
      } else if (dimension === 0) {
        theSum = new Array(cols);

        for (j = 0; j < cols; j++) {
          theSum[j] = 0;

          for (i = 0; i < rows; i++) {
            theSum[j] += matrix[i][j];
          }
        }
      } else if (dimension === 1) {
        theSum = new Array(rows);

        for (j = 0; j < rows; j++) {
          theSum[j] = 0;

          for (i = 0; i < cols; i++) {
            theSum[j] += matrix[j][i];
          }
        }
      } else {
        throw new Error('Invalid dimension');
      }

      return theSum;
    };

    exports.product = function product(matrix, dimension) {
      if (typeof dimension === 'undefined') {
        dimension = 0;
      }

      var rows = matrix.length,
          cols = matrix[0].length,
          theProduct,
          i,
          j;

      if (dimension === -1) {
        theProduct = [1];

        for (i = 0; i < rows; i++) {
          for (j = 0; j < cols; j++) {
            theProduct[0] *= matrix[i][j];
          }
        }
      } else if (dimension === 0) {
        theProduct = new Array(cols);

        for (j = 0; j < cols; j++) {
          theProduct[j] = 1;

          for (i = 0; i < rows; i++) {
            theProduct[j] *= matrix[i][j];
          }
        }
      } else if (dimension === 1) {
        theProduct = new Array(rows);

        for (j = 0; j < rows; j++) {
          theProduct[j] = 1;

          for (i = 0; i < cols; i++) {
            theProduct[j] *= matrix[j][i];
          }
        }
      } else {
        throw new Error('Invalid dimension');
      }

      return theProduct;
    };

    exports.standardDeviation = function standardDeviation(matrix, means, unbiased) {
      var vari = exports.variance(matrix, means, unbiased),
          l = vari.length;

      for (var i = 0; i < l; i++) {
        vari[i] = Math.sqrt(vari[i]);
      }

      return vari;
    };

    exports.variance = function variance(matrix, means, unbiased) {
      if (typeof unbiased === 'undefined') {
        unbiased = true;
      }

      means = means || exports.mean(matrix);
      var rows = matrix.length;
      if (rows === 0) return [];
      var cols = matrix[0].length;
      var vari = new Array(cols);

      for (var j = 0; j < cols; j++) {
        var sum1 = 0,
            sum2 = 0,
            x = 0;

        for (var i = 0; i < rows; i++) {
          x = matrix[i][j] - means[j];
          sum1 += x;
          sum2 += x * x;
        }

        if (unbiased) {
          vari[j] = (sum2 - sum1 * sum1 / rows) / (rows - 1);
        } else {
          vari[j] = (sum2 - sum1 * sum1 / rows) / rows;
        }
      }

      return vari;
    };

    exports.median = function median(matrix) {
      var rows = matrix.length,
          cols = matrix[0].length;
      var medians = new Array(cols);

      for (var i = 0; i < cols; i++) {
        var data = new Array(rows);

        for (var j = 0; j < rows; j++) {
          data[j] = matrix[j][i];
        }

        data.sort(compareNumbers);
        var N = data.length;

        if (N % 2 === 0) {
          medians[i] = (data[N / 2] + data[N / 2 - 1]) * 0.5;
        } else {
          medians[i] = data[Math.floor(N / 2)];
        }
      }

      return medians;
    };

    exports.mode = function mode(matrix) {
      var rows = matrix.length,
          cols = matrix[0].length,
          modes = new Array(cols),
          i,
          j;

      for (i = 0; i < cols; i++) {
        var itemCount = new Array(rows);

        for (var k = 0; k < rows; k++) {
          itemCount[k] = 0;
        }

        var itemArray = new Array(rows);
        var count = 0;

        for (j = 0; j < rows; j++) {
          var index = itemArray.indexOf(matrix[j][i]);

          if (index >= 0) {
            itemCount[index]++;
          } else {
            itemArray[count] = matrix[j][i];
            itemCount[count] = 1;
            count++;
          }
        }

        var maxValue = 0,
            maxIndex = 0;

        for (j = 0; j < count; j++) {
          if (itemCount[j] > maxValue) {
            maxValue = itemCount[j];
            maxIndex = j;
          }
        }

        modes[i] = itemArray[maxIndex];
      }

      return modes;
    };

    exports.skewness = function skewness(matrix, unbiased) {
      if (typeof unbiased === 'undefined') unbiased = true;
      var means = exports.mean(matrix);
      var n = matrix.length,
          l = means.length;
      var skew = new Array(l);

      for (var j = 0; j < l; j++) {
        var s2 = 0,
            s3 = 0;

        for (var i = 0; i < n; i++) {
          var dev = matrix[i][j] - means[j];
          s2 += dev * dev;
          s3 += dev * dev * dev;
        }

        var m2 = s2 / n;
        var m3 = s3 / n;
        var g = m3 / Math.pow(m2, 3 / 2);

        if (unbiased) {
          var a = Math.sqrt(n * (n - 1));
          var b = n - 2;
          skew[j] = a / b * g;
        } else {
          skew[j] = g;
        }
      }

      return skew;
    };

    exports.kurtosis = function kurtosis(matrix, unbiased) {
      if (typeof unbiased === 'undefined') unbiased = true;
      var means = exports.mean(matrix);
      var n = matrix.length,
          m = matrix[0].length;
      var kurt = new Array(m);

      for (var j = 0; j < m; j++) {
        var s2 = 0,
            s4 = 0;

        for (var i = 0; i < n; i++) {
          var dev = matrix[i][j] - means[j];
          s2 += dev * dev;
          s4 += dev * dev * dev * dev;
        }

        var m2 = s2 / n;
        var m4 = s4 / n;

        if (unbiased) {
          var v = s2 / (n - 1);
          var a = n * (n + 1) / ((n - 1) * (n - 2) * (n - 3));
          var b = s4 / (v * v);
          var c = (n - 1) * (n - 1) / ((n - 2) * (n - 3));
          kurt[j] = a * b - 3 * c;
        } else {
          kurt[j] = m4 / (m2 * m2) - 3;
        }
      }

      return kurt;
    };

    exports.standardError = function standardError(matrix) {
      var samples = matrix.length;
      var standardDeviations = exports.standardDeviation(matrix);
      var l = standardDeviations.length;
      var standardErrors = new Array(l);
      var sqrtN = Math.sqrt(samples);

      for (var i = 0; i < l; i++) {
        standardErrors[i] = standardDeviations[i] / sqrtN;
      }

      return standardErrors;
    };

    exports.covariance = function covariance(matrix, dimension) {
      return exports.scatter(matrix, undefined, dimension);
    };

    exports.scatter = function scatter(matrix, divisor, dimension) {
      if (typeof dimension === 'undefined') {
        dimension = 0;
      }

      if (typeof divisor === 'undefined') {
        if (dimension === 0) {
          divisor = matrix.length - 1;
        } else if (dimension === 1) {
          divisor = matrix[0].length - 1;
        }
      }

      var means = exports.mean(matrix, dimension);
      var rows = matrix.length;

      if (rows === 0) {
        return [[]];
      }

      var cols = matrix[0].length,
          cov,
          i,
          j,
          s,
          k;

      if (dimension === 0) {
        cov = new Array(cols);

        for (i = 0; i < cols; i++) {
          cov[i] = new Array(cols);
        }

        for (i = 0; i < cols; i++) {
          for (j = i; j < cols; j++) {
            s = 0;

            for (k = 0; k < rows; k++) {
              s += (matrix[k][j] - means[j]) * (matrix[k][i] - means[i]);
            }

            s /= divisor;
            cov[i][j] = s;
            cov[j][i] = s;
          }
        }
      } else if (dimension === 1) {
        cov = new Array(rows);

        for (i = 0; i < rows; i++) {
          cov[i] = new Array(rows);
        }

        for (i = 0; i < rows; i++) {
          for (j = i; j < rows; j++) {
            s = 0;

            for (k = 0; k < cols; k++) {
              s += (matrix[j][k] - means[j]) * (matrix[i][k] - means[i]);
            }

            s /= divisor;
            cov[i][j] = s;
            cov[j][i] = s;
          }
        }
      } else {
        throw new Error('Invalid dimension');
      }

      return cov;
    };

    exports.correlation = function correlation(matrix) {
      var means = exports.mean(matrix),
          standardDeviations = exports.standardDeviation(matrix, true, means),
          scores = exports.zScores(matrix, means, standardDeviations),
          rows = matrix.length,
          cols = matrix[0].length,
          i,
          j;
      var cor = new Array(cols);

      for (i = 0; i < cols; i++) {
        cor[i] = new Array(cols);
      }

      for (i = 0; i < cols; i++) {
        for (j = i; j < cols; j++) {
          var c = 0;

          for (var k = 0, l = scores.length; k < l; k++) {
            c += scores[k][j] * scores[k][i];
          }

          c /= rows - 1;
          cor[i][j] = c;
          cor[j][i] = c;
        }
      }

      return cor;
    };

    exports.zScores = function zScores(matrix, means, standardDeviations) {
      means = means || exports.mean(matrix);
      if (typeof standardDeviations === 'undefined') standardDeviations = exports.standardDeviation(matrix, true, means);
      return exports.standardize(exports.center(matrix, means, false), standardDeviations, true);
    };

    exports.center = function center(matrix, means, inPlace) {
      means = means || exports.mean(matrix);
      var result = matrix,
          l = matrix.length,
          i,
          j,
          jj;

      if (!inPlace) {
        result = new Array(l);

        for (i = 0; i < l; i++) {
          result[i] = new Array(matrix[i].length);
        }
      }

      for (i = 0; i < l; i++) {
        var row = result[i];

        for (j = 0, jj = row.length; j < jj; j++) {
          row[j] = matrix[i][j] - means[j];
        }
      }

      return result;
    };

    exports.standardize = function standardize(matrix, standardDeviations, inPlace) {
      if (typeof standardDeviations === 'undefined') standardDeviations = exports.standardDeviation(matrix);
      var result = matrix,
          l = matrix.length,
          i,
          j,
          jj;

      if (!inPlace) {
        result = new Array(l);

        for (i = 0; i < l; i++) {
          result[i] = new Array(matrix[i].length);
        }
      }

      for (i = 0; i < l; i++) {
        var resultRow = result[i];
        var sourceRow = matrix[i];

        for (j = 0, jj = resultRow.length; j < jj; j++) {
          if (standardDeviations[j] !== 0 && !isNaN(standardDeviations[j])) {
            resultRow[j] = sourceRow[j] / standardDeviations[j];
          }
        }
      }

      return result;
    };

    exports.weightedVariance = function weightedVariance(matrix, weights) {
      var means = exports.mean(matrix);
      var rows = matrix.length;
      if (rows === 0) return [];
      var cols = matrix[0].length;
      var vari = new Array(cols);

      for (var j = 0; j < cols; j++) {
        var sum = 0;
        var a = 0,
            b = 0;

        for (var i = 0; i < rows; i++) {
          var z = matrix[i][j] - means[j];
          var w = weights[i];
          sum += w * (z * z);
          b += w;
          a += w * w;
        }

        vari[j] = sum * (b / (b * b - a));
      }

      return vari;
    };

    exports.weightedMean = function weightedMean(matrix, weights, dimension) {
      if (typeof dimension === 'undefined') {
        dimension = 0;
      }

      var rows = matrix.length;
      if (rows === 0) return [];
      var cols = matrix[0].length,
          means,
          i,
          ii,
          j,
          w,
          row;

      if (dimension === 0) {
        means = new Array(cols);

        for (i = 0; i < cols; i++) {
          means[i] = 0;
        }

        for (i = 0; i < rows; i++) {
          row = matrix[i];
          w = weights[i];

          for (j = 0; j < cols; j++) {
            means[j] += row[j] * w;
          }
        }
      } else if (dimension === 1) {
        means = new Array(rows);

        for (i = 0; i < rows; i++) {
          means[i] = 0;
        }

        for (j = 0; j < rows; j++) {
          row = matrix[j];
          w = weights[j];

          for (i = 0; i < cols; i++) {
            means[j] += row[i] * w;
          }
        }
      } else {
        throw new Error('Invalid dimension');
      }

      var weightSum = array$4.sum(weights);

      if (weightSum !== 0) {
        for (i = 0, ii = means.length; i < ii; i++) {
          means[i] /= weightSum;
        }
      }

      return means;
    };

    exports.weightedCovariance = function weightedCovariance(matrix, weights, means, dimension) {
      dimension = dimension || 0;
      means = means || exports.weightedMean(matrix, weights, dimension);
      var s1 = 0,
          s2 = 0;

      for (var i = 0, ii = weights.length; i < ii; i++) {
        s1 += weights[i];
        s2 += weights[i] * weights[i];
      }

      var factor = s1 / (s1 * s1 - s2);
      return exports.weightedScatter(matrix, weights, means, factor, dimension);
    };

    exports.weightedScatter = function weightedScatter(matrix, weights, means, factor, dimension) {
      dimension = dimension || 0;
      means = means || exports.weightedMean(matrix, weights, dimension);

      if (typeof factor === 'undefined') {
        factor = 1;
      }

      var rows = matrix.length;

      if (rows === 0) {
        return [[]];
      }

      var cols = matrix[0].length,
          cov,
          i,
          j,
          k,
          s;

      if (dimension === 0) {
        cov = new Array(cols);

        for (i = 0; i < cols; i++) {
          cov[i] = new Array(cols);
        }

        for (i = 0; i < cols; i++) {
          for (j = i; j < cols; j++) {
            s = 0;

            for (k = 0; k < rows; k++) {
              s += weights[k] * (matrix[k][j] - means[j]) * (matrix[k][i] - means[i]);
            }

            cov[i][j] = s * factor;
            cov[j][i] = s * factor;
          }
        }
      } else if (dimension === 1) {
        cov = new Array(rows);

        for (i = 0; i < rows; i++) {
          cov[i] = new Array(rows);
        }

        for (i = 0; i < rows; i++) {
          for (j = i; j < rows; j++) {
            s = 0;

            for (k = 0; k < cols; k++) {
              s += weights[k] * (matrix[j][k] - means[j]) * (matrix[i][k] - means[i]);
            }

            cov[i][j] = s * factor;
            cov[j][i] = s * factor;
          }
        }
      } else {
        throw new Error('Invalid dimension');
      }

      return cov;
    };
  });

  var array$3 = array$4;
  var matrix$2 = matrix$3;
  var mlStat$1 = {
    array: array$3,
    matrix: matrix$2
  };

  var COMMON_NO$1 = 0;
  var COMMON_FIRST$1 = 1;
  var COMMON_SECOND$1 = 2;
  var COMMON_BOTH$1 = 3; // should be a binary operation !

  var Stat$1 = mlStat$1.array;
  /**
   * Create a comparator class
   * {object} [options={}]
   * {string} [options.common=''] should we take only common peaks 'first', 'second', 'both', ''
   * {number} [options.widthBottom=2] bottom trapezoid width for similarity evaluation
   * {number} [options.widthTop=1] top trapezoid width for similarity evaluation
   * {number} [options.from] from region used for similarity calculation
   * {number} [options.to] to region used for similarity calculation
   */

  var src$n = function Comparator(options = {}) {
    var widthTop, widthBottom, from, to;
    var array1Extract, array2Extract, widthSlope, array1ExtractInfo, array2ExtractInfo;
    var common, commonFactor;
    setOptions(options);
    var array1 = [];
    var array2 = [];
    /*
       2 formats are allowed:
       [[x1,x2,...],[y1,y2,...]] or [[x1,y1],[x2,y2], ...]
      */

    function setOptions(newOptions) {
      options = newOptions || {};

      if (typeof options.common === 'string') {
        if (options.common.toLowerCase() === 'first') {
          common = COMMON_FIRST$1;
        } else if (options.common.toLowerCase() === 'second') {
          common = COMMON_SECOND$1;
        } else if (options.common.toLowerCase() === 'both') {
          common = COMMON_BOTH$1;
        } else {
          common = COMMON_NO$1;
        }
      } else {
        if (options.common === true) {
          common = COMMON_BOTH$1;
        } else {
          common = COMMON_NO$1;
        }
      }

      commonFactor = options.commonFactor || commonFactor || 4;

      if (options.widthBottom === undefined) {
        options.widthBottom = widthBottom || 2;
      }

      if (options.widthTop === undefined) {
        options.widthTop = widthTop || 1;
      }

      setTrapezoid(options.widthBottom, options.widthTop);
      setFromTo(options.from || from, options.to || to);
    }

    function setPeaks1(anArray) {
      array1 = checkArray(anArray);

      if (common) {
        var extracts = commonExtractAndNormalize$1(array1, array2, widthBottom, from, to, common);
        array1Extract = extracts.data1;
        array1ExtractInfo = extracts.info1;
        array2Extract = extracts.data2;
        array2ExtractInfo = extracts.info2;
      } else {
        var extract = extractAndNormalize$1(array1, from, to);
        array1Extract = extract.data;
        array1ExtractInfo = extract.info;
      }
    }

    function setPeaks2(anArray) {
      array2 = checkArray(anArray);

      if (common) {
        var extracts = commonExtractAndNormalize$1(array1, array2, widthBottom, from, to, common);
        array1Extract = extracts.data1;
        array1ExtractInfo = extracts.info1;
        array2Extract = extracts.data2;
        array2ExtractInfo = extracts.info2;
      } else {
        var extract = extractAndNormalize$1(array2, from, to);
        array2Extract = extract.data;
        array2ExtractInfo = extract.info;
      }
    }

    function getExtract1() {
      return array1Extract;
    }

    function getExtract2() {
      return array2Extract;
    }

    function getExtractInfo1() {
      return array1ExtractInfo;
    }

    function getExtractInfo2() {
      return array2ExtractInfo;
    }

    function setTrapezoid(newWidthBottom, newWidthTop) {
      widthTop = newWidthTop;
      widthBottom = newWidthBottom;
      widthSlope = (widthBottom - widthTop) / 2;
      if (widthBottom < widthTop) throw new Error('widthBottom has to be larger than widthTop');
    }

    function setFromTo(newFrom, newTo) {
      if (newFrom === from && newTo === to) return;
      from = newFrom;
      to = newTo;

      if (common) {
        var extracts = commonExtractAndNormalize$1(array1, array2, widthBottom, from, to, common);
        array1Extract = extracts.data1;
        array1ExtractInfo = extracts.info1;
        array2Extract = extracts.data2;
        array2ExtractInfo = extracts.info2;
      } else {
        var extract = extractAndNormalize$1(array1, from, to);
        array1Extract = extract.data;
        array1ExtractInfo = extract.info;
        extract = extractAndNormalize$1(array2, from, to);
        array2Extract = extract.data;
        array2ExtractInfo = extract.info;
      }
    }

    function getOverlap(x1, y1, x2, y2) {
      if (y1 === 0 || y2 === 0) return 0; // TAKE CARE !!! We multiply the diff by 2 !!!

      var diff = Math.abs(x1 - x2) * 2;
      if (diff > widthBottom) return 0;

      if (diff <= widthTop) {
        return Math.min(y1, y2);
      }

      var maxValue = Math.max(y1, y2) * (widthBottom - diff) / (widthBottom - widthTop);
      return Math.min(y1, y2, maxValue);
    } // This is the old trapezoid similarity


    function getOverlapTrapezoid(x1, y1, x2, y2) {
      var factor = 2 / (widthTop + widthBottom); // correction for surface=1

      if (y1 === 0 || y2 === 0) return 0;

      if (x1 === x2) {
        // they have the same position
        return Math.min(y1, y2);
      }

      var diff = Math.abs(x1 - x2);
      if (diff >= widthBottom) return 0;

      if (y1 === y2) {
        // do they have the same height ???
        // we need to find the common length
        if (diff <= widthTop) {
          return ((widthTop + widthBottom) / 2 - diff) * y1 * factor;
        } else if (diff <= widthBottom) {
          return (widthBottom - diff) * y1 / 2 * (diff - widthTop) / (widthBottom - widthTop) * factor;
        }

        return 0;
      } else {
        // the height are different and not the same position ...
        // we need to consider only one segment to find its intersection
        var small = Math.min(y1, y2);
        var big = Math.max(y1, y2);
        var targets = [[[0, 0], [widthSlope, small]], [[widthSlope, small], [widthSlope + widthTop, small]], [[widthTop + widthSlope, small], [widthBottom, 0]]];
        var segment;

        if (x1 > x2 && y1 > y2 || x1 < x2 && y1 < y2) {
          segment = [[diff, 0], [diff + widthSlope, big]];
        } else {
          segment = [[diff + widthSlope, big], [diff, 0]];
        }

        for (var i = 0; i < 3; i++) {
          var intersection = getIntersection$1(targets[i], segment);

          if (intersection) {
            switch (i) {
              case 0:
                return small - diff * intersection.y / 2 * factor;

              case 1:
                // to simplify ...
                //     console.log("           ",widthSlope,small,big,intersection.x)
                return (widthSlope * small / (2 * big) * small + (widthTop + widthSlope - intersection.x) * small + widthSlope * small / 2) * factor;

              case 2:
                return (widthBottom - diff) * intersection.y / 2 * factor;

              default:
                throw new Error(`unexpected intersection value: ${i}`);
            }
          }
        }
      }

      return NaN;
    } // this method calculates the total diff. The sum of positive value will yield to overlap


    function calculateDiff() {
      // we need to take 2 pointers
      // and travel progressively between them ...
      var newFirst = [[].concat(array1Extract[0]), [].concat(array1Extract[1])];
      var newSecond = [[].concat(array2Extract[0]), [].concat(array2Extract[1])];
      var array1Length = array1Extract[0] ? array1Extract[0].length : 0;
      var array2Length = array2Extract[0] ? array2Extract[0].length : 0;
      var pos1 = 0;
      var pos2 = 0;
      var previous2 = 0;

      while (pos1 < array1Length) {
        var diff = newFirst[0][pos1] - array2Extract[0][pos2];

        if (Math.abs(diff) < widthBottom) {
          // there is some overlap
          var overlap;

          if (options.trapezoid) {
            overlap = getOverlapTrapezoid(newFirst[0][pos1], newFirst[1][pos1], newSecond[0][pos2], newSecond[1][pos2]);
          } else {
            overlap = getOverlap(newFirst[0][pos1], newFirst[1][pos1], newSecond[0][pos2], newSecond[1][pos2]);
          }

          newFirst[1][pos1] -= overlap;
          newSecond[1][pos2] -= overlap;

          if (pos2 < array2Length - 1) {
            pos2++;
          } else {
            pos1++;
            pos2 = previous2;
          }
        } else {
          if (diff > 0 && pos2 < array2Length - 1) {
            pos2++;
            previous2 = pos2;
          } else {
            pos1++;
            pos2 = previous2;
          }
        }
      }

      return newSecond;
    }
    /*
          This code requires the use of an array like  [[x1,y1],[x2,y2], ...]
          If it is not the right format, we will just convert it
          Otherwise we return the correct format
       */


    function checkArray(points) {
      // if it is already a 2D array of points, we just return them
      if (Array.isArray(points) && Array.isArray(points[0]) && points.length === 2) return points;
      var x = new Array(points.length);
      var y = new Array(points.length);

      for (var i = 0; i < points.length; i++) {
        x[i] = points[i][0];
        y[i] = points[i][1];
      }

      return [x, y];
    }

    function getSimilarity(newPeaks1, newPeaks2) {
      if (newPeaks1) setPeaks1(newPeaks1);
      if (newPeaks2) setPeaks2(newPeaks2);
      var result = {};
      result.diff = calculateDiff();
      result.extract1 = getExtract1();
      result.extract2 = getExtract2();
      result.extractInfo1 = getExtractInfo1();
      result.extractInfo2 = getExtractInfo2();
      result.similarity = calculateOverlapFromDiff$1(result.diff);
      result.widthBottom = widthBottom;
      result.widthTop = widthTop;
      return result;
    }
    /*
      This works mainly when you have a array1 that is fixed
      newPeaks2 have to be normalized ! (sum to 1)
    */


    function fastSimilarity(newPeaks2, from, to) {
      array1Extract = extract$1(array1, from, to);
      array2Extract = newPeaks2;
      if (common & COMMON_SECOND$1) array1Extract = getCommonArray$1(array1Extract, array2Extract, widthBottom);
      normalize$5(array1Extract);
      var diff = calculateDiff();
      return calculateOverlapFromDiff$1(diff);
    }

    this.setPeaks1 = setPeaks1;
    this.setPeaks2 = setPeaks2;
    this.getExtract1 = getExtract1;
    this.getExtract2 = getExtract2;
    this.getExtractInfo1 = getExtractInfo1;
    this.getExtractInfo2 = getExtractInfo2;
    this.setFromTo = setFromTo;
    this.setOptions = setOptions;
    this.setTrapezoid = setTrapezoid;
    this.getSimilarity = getSimilarity;
    this.getCommonArray = getCommonArray$1;
    this.fastSimilarity = fastSimilarity;
  }; // returns an new array based on array1 where there is a peak of array2 at a distance under width/2


  function getCommonArray$1(array1, array2, width) {
    var newArray = [[], []];
    var pos2 = 0;
    width /= 2;
    var j = 0;
    var array1Length = array1[0] ? array1[0].length : 0;
    var array2Length = array2[0] ? array2[0].length : 0;

    for (var i = 0; i < array1Length; i++) {
      while (pos2 < array2Length && array1[0][i] > array2[0][pos2] + width) {
        pos2++;
      }

      if (pos2 < array2Length && array1[0][i] > array2[0][pos2] - width) {
        newArray[0][j] = array1[0][i];
        newArray[1][j] = array1[1][i];
        j++;
      }
    }

    return newArray;
  } // Adapted from: http://stackoverflow.com/questions/563198/how-do-you-detect-where-two-line-segments-intersect/1968345#1968345


  function getIntersection$1(segment1, segment2) {
    var p0X = segment1[0][0];
    var p0Y = segment1[0][1];
    var p1X = segment1[1][0];
    var p1Y = segment1[1][1];
    var p2X = segment2[0][0];
    var p2Y = segment2[0][1];
    var p3X = segment2[1][0];
    var p3Y = segment2[1][1];
    var s1X, s1Y, s2X, s2Y;
    s1X = p1X - p0X;
    s1Y = p1Y - p0Y;
    s2X = p3X - p2X;
    s2Y = p3Y - p2Y;
    var s, t;
    s = (-s1Y * (p0X - p2X) + s1X * (p0Y - p2Y)) / (-s2X * s1Y + s1X * s2Y);
    t = (s2X * (p0Y - p2Y) - s2Y * (p0X - p2X)) / (-s2X * s1Y + s1X * s2Y);

    if (s >= 0 && s <= 1 && t >= 0 && t <= 1) {
      return {
        x: p0X + t * s1X,
        y: p0Y + t * s1Y
      };
    }

    return null; // No collision
  }

  function normalize$5(array) {
    var min = Stat$1.min(array[1]);
    var max = Stat$1.max(array[1]);
    var sum = Stat$1.sum(array[1]);
    var length = array[1] ? array[1].length : 0;

    if (sum !== 0) {
      for (var i = 0; i < length; i++) {
        array[1][i] /= sum;
      }
    }

    return {
      sum: sum,
      min: min,
      max: max
    };
  } // this method will systematically take care of both array


  function commonExtractAndNormalize$1(array1, array2, width, from, to, common) {
    if (!Array.isArray(array1) || !Array.isArray(array2)) {
      return {
        info: undefined,
        data: undefined
      };
    }

    var extract1 = extract$1(array1, from, to);
    var extract2 = extract$1(array2, from, to);
    var common1, common2, info1, info2;

    if (common & COMMON_SECOND$1) {
      common1 = getCommonArray$1(extract1, extract2, width);
      info1 = normalize$5(common1);
    } else {
      common1 = extract1;
      info1 = normalize$5(common1);
    }

    if (common & COMMON_FIRST$1) {
      common2 = getCommonArray$1(extract2, extract1, width);
      info2 = normalize$5(common2);
    } else {
      common2 = extract2;
      info2 = normalize$5(common2);
    }

    return {
      info1: info1,
      info2: info2,
      data1: common1,
      data2: common2
    };
  }

  function extract$1(array, from, to) {
    var newArray = [[], []];
    var j = 0;
    var length = array[0] ? array[0].length : 0;

    for (var i = 0; i < length; i++) {
      if ((!from || array[0][i] >= from) && (!to || array[0][i] <= to)) {
        newArray[0][j] = array[0][i];
        newArray[1][j] = array[1][i];
        j++;
      }
    }

    return newArray;
  }

  function extractAndNormalize$1(array, from, to) {
    if (!Array.isArray(array)) {
      return {
        info: undefined,
        data: undefined
      };
    }

    var newArray = extract$1(array, from, to);
    var info = normalize$5(newArray);
    return {
      info: info,
      data: newArray
    };
  }

  function calculateOverlapFromDiff$1(diffs) {
    if (diffs[1].length === 0) return 0;
    var sumPos = 0;

    for (var i = 0; i < diffs[1].length; i++) {
      sumPos += Math.abs(diffs[1][i]);
    }

    return 1 - sumPos;
  }

  /**
   * @param {object}   [options={}]
   * @param {object}   [options.minCharge=1]
   * @param {object}   [options.maxCharge=10]
   * @param {object}   [options.similarity={}]
   * @param {object}   [options.similarity.widthBottom]
   * @param {object}   [options.similarity.widthTop]
   * @param {object}   [options.similarity.widthFunction] - function called with mass that should return an object width containing top and bottom
   * @param {object}   [options.similarity.zone={}]
   * @param {object}   [options.similarity.zone.low=-0.5] - window shift based on observed monoisotopic mass
   * @param {object}   [options.similarity.zone.high=2.5] - to value for the comparison window
   * @param {object}   [options.similarity.common]
   */


  const NEUTRON_MASS$1 = 1;

  function getPeakChargeBySimilarity(spectrum, targetMass, options = {}) {
    let {
      similarity = {},
      minCharge = 1,
      maxCharge = 10
    } = options;
    let {
      zone = {},
      widthFunction
    } = similarity;
    let {
      low = -0.5,
      high = 2.5
    } = zone;

    if (!spectrum || !spectrum.data.x.length > 0) {
      throw Error('You need to add an experimental spectrum first using setMassSpectrum');
    }

    let width = {
      bottom: similarity.widthBottom,
      top: similarity.widthTop
    };
    similarity = JSON.parse(JSON.stringify(similarity));
    similarity.common = 'second';
    let experimentalData = spectrum.data;
    let similarityProcessor = new src$n(similarity);
    similarityProcessor.setPeaks1([experimentalData.x, experimentalData.y]);

    if (widthFunction && typeof widthFunction === 'string') {
      // eslint-disable-next-line no-new-func
      widthFunction = new Function('mass', widthFunction);
      let checkTopBottom = widthFunction(123);

      if (!checkTopBottom.bottom || !checkTopBottom.top) {
        throw Error('widthFunction should return an object with bottom and top properties');
      }
    }

    let fromCharge = minCharge * maxCharge > 0 ? Math.round(Math.min(Math.abs(minCharge), Math.abs(maxCharge))) : 1;
    let toCharge = Math.round(Math.max(Math.abs(minCharge), Math.abs(maxCharge)));
    let fromIsotope = Math.ceil(low);
    let toIsotope = Math.floor(high);
    let isotopeHeight = 1 / (toIsotope - fromIsotope + 1);
    let results = [];

    for (let charge = fromCharge; charge < toCharge + 1; charge++) {
      let isotopePositions = {
        x: [],
        y: []
      };

      for (let isotopePosition = fromIsotope; isotopePosition < toIsotope + 1; isotopePosition++) {
        isotopePositions.x.push(targetMass + isotopePosition * NEUTRON_MASS$1 / charge);
        isotopePositions.y.push(isotopeHeight);
      }

      let from = targetMass + low / Math.abs(charge);
      let to = targetMass + high / Math.abs(charge);
      similarityProcessor.setFromTo(from, to);

      if (widthFunction) {
        width = widthFunction(targetMass);
        similarityProcessor.setTrapezoid(width.bottom, width.top);
      }

      similarityProcessor.setPeaks2([isotopePositions.x, isotopePositions.y]);
      let result = similarityProcessor.getSimilarity();
      results.push({
        charge,
        similarity: result.similarity
      });
    }

    return results.sort((a, b) => b.similarity - a.similarity)[0].charge;
  }

  var getPeakChargeBySimilarity_1 = getPeakChargeBySimilarity;

  /**
   * Filter the array of peaks
   * @param {array} peaks - array of all the peaks
   * @param {object} [options={}]
   * @param {number} [options.from] - min X value of the window to consider
   * @param {number} [options.to] - max X value of the window to consider
   * @param {number} [options.threshold=0.01] - minimal intensity compare to base peak
   * @param {number} [options.limit=undefined] - maximal number of peaks (based on intensity)
   * @returns {array} - copy of peaks with 'close' annotation
   */

  function getPeaks$1(peaks, options = {}) {
    const {
      from = peaks.reduce((previous, peak) => Math.min(peak.x, previous), Number.MAX_SAFE_INTEGER),
      to = peaks.reduce((previous, peak) => Math.max(peak.x, previous), Number.MIN_SAFE_INTEGER),
      threshold = 0.01,
      limit
    } = options;
    let maxY = Number.MIN_SAFE_INTEGER;

    for (let peak of peaks) {
      if (peak.y > maxY) maxY = peak.y;
    }

    let minY = maxY * threshold;
    peaks = peaks.filter(peak => peak.x >= from && peak.x <= to && peak.y >= minY);

    if (limit && peaks.length > limit) {
      peaks.sort((a, b) => b.y - a.y);
      peaks = peaks.slice(0, limit);
    }

    return peaks.sort((a, b) => a.x - b.x);
  }

  var getPeaks_1 = getPeaks$1;

  /**
   * When a spectrum is continous ?
   * - has more than 100 points
   * - deltaX change can not be more than a factor 2
   * - deltaX may not be larger than 0.1
   * - if y is zero it does not count
   * @param {object} spectrum
   * @param {object} [options={}]
   * @param {number} [options.minLength=100]
   * @param {number} [options.maxDeltaRatio=3]
   */

  function isContinuous(spectrum, options = {}) {
    const {
      minLength = 100,
      maxDeltaRatio = 3
    } = options;
    const minRadio = 1 / maxDeltaRatio;
    const maxRatio = 1 * maxDeltaRatio;

    if (spectrum.continuous === undefined) {
      let xs = spectrum.data.x;
      let ys = spectrum.data.y;

      if (xs.length < minLength) {
        spectrum.continuous = false;
      } else {
        let previousDelta = xs[1] - xs[0];
        spectrum.continuous = true;
        let success = 0;
        let failed = 0;

        for (let i = 0; i < xs.length - 1; i++) {
          if (ys[i] === 0 || ys[i + 1] === 0) {
            previousDelta = 0;
            continue;
          }

          let delta = xs[i + 1] - xs[i];

          if (previousDelta) {
            let ratio = delta / previousDelta;

            if ((Math.abs(delta) > 0.1 || ratio < minRadio || ratio > maxRatio) && ys[i] !== 0 && ys[i + 1] !== 0) {
              failed++;
              break;
            } else {
              success++;
            }
          }

          previousDelta = delta;
        }

        if (success / failed < 10) {
          spectrum.continuous = false;
        }
      }
    }

    return spectrum.continuous;
  }

  var isContinuous_1 = isContinuous;

  /**
   * @param {object}   [options={}]
   * @param {object}   [options.min=1]
   * @param {object}   [options.max=10]
   * @param {object}   [options.low=-1]
   * @param {object}   [options.high=1]
   * @param {object}   [options.precision=100]
   */

  const NEUTRON_MASS = 1;

  function appendPeaksCharge(peaks, options = {}) {
    let {
      precision = 100,
      low = -1,
      high = 1,
      min: minCharge = 1,
      max: maxCharge = 10
    } = options;
    let fromCharge = minCharge * maxCharge > 0 ? Math.round(Math.min(Math.abs(minCharge), Math.abs(maxCharge))) : 1;
    let toCharge = Math.round(Math.max(Math.abs(minCharge), Math.abs(maxCharge)));
    let fromIsotope = Math.ceil(low);
    let toIsotope = Math.floor(high);
    let numberIsotopes = toIsotope - fromIsotope + 1;
    let isotopeIntensity = 1 / numberIsotopes;
    let fromIndex = 0;
    let localFromIndex = 0;
    let localToIndex = 0;

    for (let peakIndex = 0; peakIndex < peaks.length; peakIndex++) {
      let peak = peaks[peakIndex];
      let targetMass = peak.x;
      localFromIndex = fromIndex;
      let bestCharge = fromCharge;
      let bestChargeMatch = 0;

      for (let charge = fromCharge; charge < toCharge + 1; charge++) {
        let theoreticalPositions = {
          x: [],
          y: new Array(numberIsotopes).fill(isotopeIntensity)
        };
        let massRange = precision * 1e-6 * targetMass;

        for (let isotopePosition = fromIsotope; isotopePosition < toIsotope + 1; isotopePosition++) {
          theoreticalPositions.x.push(targetMass + isotopePosition * NEUTRON_MASS / charge);
        }

        let fromMass = targetMass + low / Math.abs(charge) - massRange;
        let toMass = targetMass + high / Math.abs(charge) + massRange;

        if (charge === 1) {
          // we may move the fromIndex
          while (peaks[fromIndex].x < fromMass) {
            fromIndex++;
          }
        }
        /*
         * Find the from / to index for the specific peak and specific charge
         */


        while (peaks[localFromIndex].x < fromMass) {
          localFromIndex++;
        }

        localToIndex = localFromIndex;
        let localHeightSum = 0;

        while (localToIndex < peaks.length && peaks[localToIndex].x < toMass) {
          localHeightSum += peaks[localToIndex].y;
          localToIndex++;
        }

        localToIndex--; //  console.log({ localFromIndex, localToIndex });

        /*
          Calculate the overlap for a specific peak and specific charge
        */

        let currentTheoreticalPosition = 0;
        let theoreticalMaxValue = 1 / numberIsotopes;
        let totalMatch = 0;

        for (let index = localFromIndex; index <= localToIndex; index++) {
          let minMass = theoreticalPositions.x[currentTheoreticalPosition] - massRange / charge;
          let maxMass = theoreticalPositions.x[currentTheoreticalPosition] + massRange / charge;

          while (maxMass < peaks[index].x) {
            currentTheoreticalPosition++;
            theoreticalMaxValue = 1 / numberIsotopes;
            minMass = theoreticalPositions.x[currentTheoreticalPosition] - massRange / charge;
            maxMass = theoreticalPositions.x[currentTheoreticalPosition] + massRange / charge;
          }

          while (index < peaks.length && peaks[index].x < minMass) {
            index++;
          } //    console.log({ index, minMass, maxMass, massRange, localHeightSum });


          if (index < peaks.length && peaks[index].x <= maxMass) {
            while (index < peaks.length && peaks[index].x <= maxMass) {
              if (peaks[index].x >= minMass && peaks[index].x <= maxMass) {
                let value = peaks[index].y / localHeightSum; //      console.log({ theoreticalMaxValue, value });

                value = Math.min(theoreticalMaxValue, value);
                theoreticalMaxValue -= value;
                totalMatch += value;
              }

              index++;
            }

            index--;
          }

          if (totalMatch > bestChargeMatch) {
            bestCharge = charge;
            bestChargeMatch = totalMatch;
          }
        }
      }

      peak.charge = bestCharge;
    }

    return peaks;
  }

  var appendPeaksCharge_1 = appendPeaksCharge;

  var require$$0$5 = /*@__PURE__*/getAugmentedNamespace(src$p);

  const gsd$1 = require$$0$5.gsd;
  /**
   * Filter the array of peaks
   * @param {Spectrum} spectrum - array of all the peaks
   * @param {object} [options={}]
   * @param {object} [options.charge={}]
   * @param {number} [options.charge.min=1]
   * @param {number} [options.charge.max=10]
   * @param {number} [options.charge.low=-1]
   * @param {number} [options.charge.high=1]
   * @param {number} [options.charge.precision=30]
   * @returns {array} - copy of peaks with 'close' annotation
   */

  function peakPicking(spectrum, options = {}) {
    const {
      charge: chargeOptions = {}
    } = options;

    if (!spectrum.peaks) {
      if (spectrum.isContinuous()) {
        spectrum.peaks = gsd$1(spectrum.data, {
          noiseLevel: 0,
          minMaxRatio: 0.00025,
          // Threshold to determine if a given peak should be considered as a noise
          realTopDetection: true,
          maxCriteria: true,
          // inverted:false
          smoothY: false,
          sgOptions: {
            windowSize: 7,
            polynomial: 3
          }
        });
      } else {
        spectrum.peaks = [];
        let data = spectrum.data;

        for (let i = 0; i < data.x.length; i++) {
          spectrum.peaks.push({
            x: data.x[i],
            y: data.y[i],
            width: 0
          });
        }
      } // required and linked to https://github.com/mljs/global-spectral-deconvolution/issues/17


      spectrum.peaks = spectrum.peaks.filter(peak => !isNaN(peak.x) && !isNaN(peak.y));
      appendPeaksCharge_1(spectrum.peaks, chargeOptions);
    }

    return spectrum.peaks;
  }

  var peakPicking_1 = peakPicking;

  function _interopDefaultLegacy$3(e) {
    return e && typeof e === 'object' && 'default' in e ? e : {
      'default': e
    };
  }

  var isArray__default$2 = /*#__PURE__*/_interopDefaultLegacy$3(isArray$2);

  function min$7(input, options = {}) {
    if (!isArray__default$2['default'](input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    const {
      fromIndex = 0,
      toIndex = input.length
    } = options;

    if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
      throw new Error('fromIndex must be a positive integer smaller than length');
    }

    if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
      throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
    }

    let minValue = input[fromIndex];

    for (let i = fromIndex + 1; i < toIndex; i++) {
      if (input[i] < minValue) minValue = input[i];
    }

    return minValue;
  }

  var lib$5 = min$7;

  function maybeToPrecision$1(value, digits) {
    if (value < 0) {
      value = 0 - value;

      if (typeof digits === 'number') {
        return `- ${value.toPrecision(digits)}`;
      } else {
        return `- ${value.toString()}`;
      }
    } else {
      if (typeof digits === 'number') {
        return value.toPrecision(digits);
      } else {
        return value.toString();
      }
    }
  }

  function checkArraySize$1(x, y) {
    if (!isAnyArray$7(x) || !isAnyArray$7(y)) {
      throw new TypeError('x and y must be arrays');
    }

    if (x.length !== y.length) {
      throw new RangeError('x and y arrays must have the same length');
    }
  }

  class BaseRegression$3 {
    constructor() {
      if (new.target === BaseRegression$3) {
        throw new Error('BaseRegression must be subclassed');
      }
    }

    predict(x) {
      if (typeof x === 'number') {
        return this._predict(x);
      } else if (isAnyArray$7(x)) {
        const y = [];

        for (let i = 0; i < x.length; i++) {
          y.push(this._predict(x[i]));
        }

        return y;
      } else {
        throw new TypeError('x must be a number or array');
      }
    }

    _predict() {
      throw new Error('_predict must be implemented');
    }

    train() {// Do nothing for this package
    }

    toString() {
      return '';
    }

    toLaTeX() {
      return '';
    }
    /**
     * Return the correlation coefficient of determination (r) and chi-square.
     * @param {Array<number>} x
     * @param {Array<number>} y
     * @return {object}
     */


    score(x, y) {
      if (!isAnyArray$7(x) || !isAnyArray$7(y) || x.length !== y.length) {
        throw new Error('x and y must be arrays of the same length');
      }

      const n = x.length;
      const y2 = new Array(n);

      for (let i = 0; i < n; i++) {
        y2[i] = this._predict(x[i]);
      }

      let xSum = 0;
      let ySum = 0;
      let chi2 = 0;
      let rmsd = 0;
      let xSquared = 0;
      let ySquared = 0;
      let xY = 0;

      for (let i = 0; i < n; i++) {
        xSum += y2[i];
        ySum += y[i];
        xSquared += y2[i] * y2[i];
        ySquared += y[i] * y[i];
        xY += y2[i] * y[i];

        if (y[i] !== 0) {
          chi2 += (y[i] - y2[i]) * (y[i] - y2[i]) / y[i];
        }

        rmsd += (y[i] - y2[i]) * (y[i] - y2[i]);
      }

      const r = (n * xY - xSum * ySum) / Math.sqrt((n * xSquared - xSum * xSum) * (n * ySquared - ySum * ySum));
      return {
        r: r,
        r2: r * r,
        chi2: chi2,
        rmsd: Math.sqrt(rmsd / n)
      };
    }

  }

  var src$m = /*#__PURE__*/Object.freeze({
    __proto__: null,
    'default': BaseRegression$3,
    maybeToPrecision: maybeToPrecision$1,
    checkArrayLength: checkArraySize$1
  });

  class SimpleLinearRegression$1 extends BaseRegression$3 {
    constructor(x, y) {
      super();

      if (x === true) {
        this.slope = y.slope;
        this.intercept = y.intercept;
        this.coefficients = [y.intercept, y.slope];
      } else {
        checkArraySize$1(x, y);
        regress$1(this, x, y);
      }
    }

    toJSON() {
      return {
        name: 'simpleLinearRegression',
        slope: this.slope,
        intercept: this.intercept
      };
    }

    _predict(x) {
      return this.slope * x + this.intercept;
    }

    computeX(y) {
      return (y - this.intercept) / this.slope;
    }

    toString(precision) {
      let result = 'f(x) = ';

      if (this.slope !== 0) {
        const xFactor = maybeToPrecision$1(this.slope, precision);
        result += `${xFactor === '1' ? '' : `${xFactor} * `}x`;

        if (this.intercept !== 0) {
          const absIntercept = Math.abs(this.intercept);
          const operator = absIntercept === this.intercept ? '+' : '-';
          result += ` ${operator} ${maybeToPrecision$1(absIntercept, precision)}`;
        }
      } else {
        result += maybeToPrecision$1(this.intercept, precision);
      }

      return result;
    }

    toLaTeX(precision) {
      return this.toString(precision);
    }

    static load(json) {
      if (json.name !== 'simpleLinearRegression') {
        throw new TypeError('not a SLR model');
      }

      return new SimpleLinearRegression$1(true, json);
    }

  }

  function regress$1(slr, x, y) {
    const n = x.length;
    let xSum = 0;
    let ySum = 0;
    let xSquared = 0;
    let xY = 0;

    for (let i = 0; i < n; i++) {
      xSum += x[i];
      ySum += y[i];
      xSquared += x[i] * x[i];
      xY += x[i] * y[i];
    }

    const numerator = n * xY - xSum * ySum;
    slr.slope = numerator / (n * xSquared - xSum * xSum);
    slr.intercept = 1 / n * ySum - slr.slope * (1 / n) * xSum;
    slr.coefficients = [slr.intercept, slr.slope];
  }

  var src$l = /*#__PURE__*/Object.freeze({
    __proto__: null,
    'default': SimpleLinearRegression$1
  });

  var BaseRegression$2 = /*@__PURE__*/getAugmentedNamespace(src$m);

  var require$$0$4 = /*@__PURE__*/getAugmentedNamespace(src$l);

  function _interopDefault$2(ex) {
    return ex && typeof ex === 'object' && 'default' in ex ? ex['default'] : ex;
  }

  var BaseRegression__default$1 = _interopDefault$2(BaseRegression$2);

  var SimpleLinearRegression = _interopDefault$2(require$$0$4);

  class PowerRegression extends BaseRegression__default$1 {
    constructor(x, y) {
      super();

      if (x === true) {
        // reloading model
        this.A = y.A;
        this.B = y.B;
      } else {
        BaseRegression$2.checkArrayLength(x, y);
        regress(this, x, y);
      }
    }

    _predict(newInputs) {
      return this.A * Math.pow(newInputs, this.B);
    }

    toJSON() {
      return {
        name: 'powerRegression',
        A: this.A,
        B: this.B
      };
    }

    toString(precision) {
      return `f(x) = ${BaseRegression$2.maybeToPrecision(this.A, precision)} * x^${BaseRegression$2.maybeToPrecision(this.B, precision)}`;
    }

    toLaTeX(precision) {
      let latex = '';

      if (this.B >= 0) {
        latex = `f(x) = ${BaseRegression$2.maybeToPrecision(this.A, precision)}x^{${BaseRegression$2.maybeToPrecision(this.B, precision)}}`;
      } else {
        latex = `f(x) = \\frac{${BaseRegression$2.maybeToPrecision(this.A, precision)}}{x^{${BaseRegression$2.maybeToPrecision(-this.B, precision)}}}`;
      }

      latex = latex.replace(/e([+-]?[0-9]+)/g, 'e^{$1}');
      return latex;
    }

    static load(json) {
      if (json.name !== 'powerRegression') {
        throw new TypeError('not a power regression model');
      }

      return new PowerRegression(true, json);
    }

  }

  function regress(pr, x, y) {
    const n = x.length;
    const xl = new Array(n);
    const yl = new Array(n);

    for (let i = 0; i < n; i++) {
      xl[i] = Math.log(x[i]);
      yl[i] = Math.log(y[i]);
    }

    const linear = new SimpleLinearRegression(xl, yl);
    pr.A = Math.exp(linear.intercept);
    pr.B = linear.slope;
  }

  var lib$4 = PowerRegression;

  function peaksWidth(peaks) {
    let xs = peaks.map(peak => peak.x);
    let widths = peaks.map(peak => peak.width);

    if (xs.length < 2) {
      throw new Error(`peaksWidth: not enough peaks (less than 2) for automatic width calculation: ${xs.length}`);
    }

    let regression = new lib$4(xs, widths, {
      computeQuality: true,
      computeCoefficient: true
    });

    if (isNaN(regression.A) || isNaN(regression.B)) {
      throw new Error('peaksWidth: can not calculate regression');
    }

    let from = lib$5(xs);
    let to = lib$7(xs);
    let regressionChart = {
      x: [],
      y: []
    };

    for (let x = from; x <= to; x += (to - from) / 1000) {
      regressionChart.x.push(x);
      regressionChart.y.push(regression.predict(x));
    }

    return {
      widths: {
        x: xs,
        y: widths
      },
      fit: regressionChart,
      score: regression.score(xs, widths),
      // eslint-disable-next-line no-new-func
      predictFct: regression.predict.bind(regression),
      tex: regression.toLaTeX(3),
      A: regression.A,
      B: regression.B,
      predictFctString: `${regression.A} * mass ** ${regression.B}`
    };
  }

  var peaksWidth_1 = peaksWidth;

  /**
   *
   * @param {array} bestPeaks
   * @param {object} [options={}]
   * @param {array} [options.mfColors={}]
   * @param {number} [options.numberDigits=5]
   * @param {number} [options.numberMFs=0]
   * @param {boolean} [options.showMF=false]
   * @param {array} [options.mfColors={}]
   * @param {number} [options.charge=1]
   * @param {number} [options.shift=0]
   * @param {object} [options.mfPrefs]
   * @param {number} [options.displayCharge=true]
   *
   */


  function getPeaksAnnotation(bestPeaks, options = {}) {
    const emdb = new src$3();
    options = Object.assign({
      limit: 5,
      precision: 100
    }, options);
    let {
      numberDigits = 5,
      shift = 0,
      showMF = false,
      numberMFs = 0,
      charge = 1,
      mfPrefs = {},
      displayCharge = true,
      mfColors = [{
        limit: 3,
        color: 'green'
      }, {
        limit: 20,
        color: 'lightgreen'
      }, {
        limit: 50,
        color: 'lightorange'
      }]
    } = options;
    if (showMF && !numberMFs) numberMFs = 1;
    let annotations = [];
    bestPeaks.sort((a, b) => a.close ? -1 : b.close ? 1 : 0);

    for (let peak of bestPeaks) {
      let annotation;

      if (peak.close) {
        annotation = {
          type: 'line',
          _highlight: peak._highlight,
          info: peak,
          position: [{
            y: peak.y,
            dy: '-5px',
            x: peak.x
          }, {
            y: peak.y,
            dy: '-15px',
            x: peak.x
          }]
        };
        annotations.push(annotation);
        annotation = {
          type: 'ellipse',
          _highlight: peak._highlight,
          info: peak,
          position: [{
            y: peak.y,
            dy: '-15px',
            x: peak.x
          }],
          props: {
            rx: '3px',
            ry: '3px',
            fillOpacity: 0.3
          }
        };
      } else {
        annotation = {
          type: 'line',
          _highlight: peak._highlight,
          position: [{
            y: peak.y,
            dy: '-5px',
            x: peak.x
          }, {
            y: peak.y,
            dy: '-25px',
            x: peak.x
          }],
          labels: [{
            text: (peak.x + shift).toFixed(numberDigits),
            color: 'red',
            position: {
              x: peak.x,
              y: peak.y,
              dy: '-17px',
              dx: '2px'
            }
          }]
        };

        if (displayCharge) {
          annotation.labels.push({
            text: `Z:${peak.charge}`,
            color: 'grey',
            position: {
              x: peak.x,
              y: peak.y,
              dy: '-4px',
              dx: '2px'
            }
          });
        }

        if (numberMFs) {
          // we have 2 cases. Either there is a shift and we deal with differences
          // otherwise it is absolute
          // if there is a shift we consider only a neutral loss and the parameter charge is important
          if (shift) {
            // neutral loss
            let currentMfPrefs = Object.assign({}, mfPrefs, {
              allowNeutral: true,
              ionizations: ''
            }); // we need to deal with the precision and increase it

            currentMfPrefs.precision = currentMfPrefs.precision / Math.max(Math.abs(peak.x + shift), 1) * peak.x;
            emdb.fromMonoisotopicMass(Math.abs((peak.x + shift) * charge), currentMfPrefs);
          } else {
            emdb.fromMonoisotopicMass(Math.abs(peak.x * charge), mfPrefs);
          }

          let mfs = emdb.get('monoisotopic');
          let numberOfMFS = Math.min(mfs.length, numberMFs);

          for (let i = 0; i < numberOfMFS; i++) {
            let mf = mfs[i];
            let ppm = shift ? mf.ms.ppm / shift * mfs[0].ms.em : mf.ms.ppm;
            annotation.labels.push({
              text: mf.mf,
              color: getColor$1(mfColors, Math.abs(ppm)),
              position: {
                x: peak.x,
                y: peak.y,
                dy: `${-13 * (numberOfMFS - i) - 17}px`,
                dx: '2px'
              }
            });
          }
        }
      }

      annotations.push(annotation);
    }

    return annotations;
  }

  function getColor$1(colors, value) {
    for (let color of colors) {
      if (value < color.limit) return color.color;
    }

    return 'lightgrey';
  }

  var getPeaksAnnotation_1 = getPeaksAnnotation;

  var jsgraph = {
    getPeaksAnnotation: getPeaksAnnotation_1
  };

  var require$$0$3 = /*@__PURE__*/getAugmentedNamespace(src$o);

  const {
    parseXY
  } = require$$0$3;

  function Spectrum$2(data = {
    x: [],
    y: []
  }) {
    if (typeof data !== 'object' || !Array.isArray(data.x) || !Array.isArray(data.y)) {
      throw new TypeError('Spectrum data must be an object with x:[], y:[]');
    }

    this.data = {
      x: data.x,
      y: data.y
    };
    Object.defineProperty(this.data, 'xOriginal', {
      enumerable: false,
      writable: true
    });
    this.cache = {};
  }

  Spectrum$2.fromText = function fromText(text) {
    const data = parseXY(text);
    return new Spectrum$2(data);
  };

  Spectrum$2.prototype.maxY = function maxY() {
    return lib$7(this.data.y);
  };

  Spectrum$2.prototype.sumY = function sumY() {
    if (!this.cache.sumY) {
      this.cache.sumY = this.data.y.reduce((previous, current) => previous + current, 0);
    }

    return this.cache.sumY;
  };

  Spectrum$2.prototype.scaleY = function scaleY(intensity = 1) {
    let basePeak = this.maxY() / intensity;
    this.data.y = this.data.y.map(y => y / basePeak);
    return this;
  };

  Spectrum$2.prototype.rescaleX = function rescaleX(callback) {
    this.ensureOriginalX();

    for (let i = 0; i < this.data.x.length; i++) {
      this.data.x[i] = callback(this.data.xOriginal[i]);
    }

    return this;
  };

  Spectrum$2.prototype.ensureOriginalX = function ensureOriginalX() {
    if (!this.data.xOriginal) {
      this.data.xOriginal = this.data.x.slice(0);
    }
  };

  Spectrum$2.prototype.normedY = function normedY(total = 1) {
    this.data.y = lib$6(this.data.y);

    if (total !== 1) {
      this.data.y = this.data.y.map(y => y * total);
    }

    return this;
  };

  Spectrum$2.prototype.peakPicking = function () {
    peakPicking_1(this);
    return this.peaks;
  };

  Spectrum$2.prototype.peaksWidth = function () {
    peakPicking_1(this);
    return peaksWidth_1(this.peaks);
  };

  Spectrum$2.prototype.getBestPeaks = function (options) {
    peakPicking_1(this);
    return getBestPeaks_1(this.peaks, options);
  };

  Spectrum$2.prototype.getPeakChargeBySimilarity = function (targetMass, options) {
    return getPeakChargeBySimilarity_1(this, targetMass, options);
  };

  Spectrum$2.prototype.getPeaks = function (options) {
    peakPicking_1(this);
    return getPeaks_1(this.peaks, options);
  };

  Spectrum$2.prototype.isContinuous = function () {
    return isContinuous_1(this);
  };

  Spectrum$2.JsGraph = Spectrum$2.prototype.JsGraph = jsgraph;
  var Spectrum_1 = Spectrum$2;

  var Spectrum$1 = Spectrum_1;
  var getPeaks = getPeaks_1;
  var getBestPeaks = getBestPeaks_1;
  var src$k = {
    Spectrum: Spectrum$1,
    getPeaks: getPeaks,
    getBestPeaks: getBestPeaks
  };

  /*!

  JSZip v3.6.0 - A JavaScript class for generating and reading zip files
  <http://stuartk.com/jszip>

  (c) 2009-2016 Stuart Knightley <stuart [at] stuartk.com>
  Dual licenced under the MIT license or GPLv3. See https://raw.github.com/Stuk/jszip/master/LICENSE.markdown.

  JSZip uses the library pako released under the MIT license :
  https://github.com/nodeca/pako/blob/master/LICENSE
  */
  var jszip = createCommonjsModule$1(function (module, exports) {
    (function (f) {
      {
        module.exports = f();
      }
    })(function () {
      return function e(t, n, r) {
        function s(o, u) {
          if (!n[o]) {
            if (!t[o]) {
              var a = typeof commonjsRequire == "function" && commonjsRequire;
              if (!u && a) return a(o, !0);
              if (i) return i(o, !0);
              var f = new Error("Cannot find module '" + o + "'");
              throw f.code = "MODULE_NOT_FOUND", f;
            }

            var l = n[o] = {
              exports: {}
            };
            t[o][0].call(l.exports, function (e) {
              var n = t[o][1][e];
              return s(n ? n : e);
            }, l, l.exports, e, t, n, r);
          }

          return n[o].exports;
        }

        var i = typeof commonjsRequire == "function" && commonjsRequire;

        for (var o = 0; o < r.length; o++) s(r[o]);

        return s;
      }({
        1: [function (require, module, exports) {
          (function (global) {
            /*!
            
            JSZip v3.5.0 - A JavaScript class for generating and reading zip files
            <http://stuartk.com/jszip>
            
            (c) 2009-2016 Stuart Knightley <stuart [at] stuartk.com>
            Dual licenced under the MIT license or GPLv3. See https://raw.github.com/Stuk/jszip/master/LICENSE.markdown.
            
            JSZip uses the library pako released under the MIT license :
            https://github.com/nodeca/pako/blob/master/LICENSE
            */
            !function (e) {
              if ("object" == typeof exports && "undefined" != typeof module) module.exports = e();else {
                ("undefined" != typeof window ? window : "undefined" != typeof global ? global : "undefined" != typeof self ? self : this).JSZip = e();
              }
            }(function () {
              return function s(a, o, u) {
                function h(r, e) {
                  if (!o[r]) {
                    if (!a[r]) {
                      var t = "function" == typeof require && require;
                      if (!e && t) return t(r, !0);
                      if (f) return f(r, !0);
                      var n = new Error("Cannot find module '" + r + "'");
                      throw n.code = "MODULE_NOT_FOUND", n;
                    }

                    var i = o[r] = {
                      exports: {}
                    };
                    a[r][0].call(i.exports, function (e) {
                      var t = a[r][1][e];
                      return h(t || e);
                    }, i, i.exports, s, a, o, u);
                  }

                  return o[r].exports;
                }

                for (var f = "function" == typeof require && require, e = 0; e < u.length; e++) h(u[e]);

                return h;
              }({
                1: [function (l, t, n) {
                  (function (r) {
                    !function (e) {
                      "object" == typeof n && void 0 !== t ? t.exports = e() : ("undefined" != typeof window ? window : void 0 !== r ? r : "undefined" != typeof self ? self : this).JSZip = e();
                    }(function () {
                      return function s(a, o, u) {
                        function h(t, e) {
                          if (!o[t]) {
                            if (!a[t]) {
                              var r = "function" == typeof l && l;
                              if (!e && r) return r(t, !0);
                              if (f) return f(t, !0);
                              var n = new Error("Cannot find module '" + t + "'");
                              throw n.code = "MODULE_NOT_FOUND", n;
                            }

                            var i = o[t] = {
                              exports: {}
                            };
                            a[t][0].call(i.exports, function (e) {
                              return h(a[t][1][e] || e);
                            }, i, i.exports, s, a, o, u);
                          }

                          return o[t].exports;
                        }

                        for (var f = "function" == typeof l && l, e = 0; e < u.length; e++) h(u[e]);

                        return h;
                      }({
                        1: [function (l, t, n) {
                          (function (r) {
                            !function (e) {
                              "object" == typeof n && void 0 !== t ? t.exports = e() : ("undefined" != typeof window ? window : void 0 !== r ? r : "undefined" != typeof self ? self : this).JSZip = e();
                            }(function () {
                              return function s(a, o, u) {
                                function h(t, e) {
                                  if (!o[t]) {
                                    if (!a[t]) {
                                      var r = "function" == typeof l && l;
                                      if (!e && r) return r(t, !0);
                                      if (f) return f(t, !0);
                                      var n = new Error("Cannot find module '" + t + "'");
                                      throw n.code = "MODULE_NOT_FOUND", n;
                                    }

                                    var i = o[t] = {
                                      exports: {}
                                    };
                                    a[t][0].call(i.exports, function (e) {
                                      return h(a[t][1][e] || e);
                                    }, i, i.exports, s, a, o, u);
                                  }

                                  return o[t].exports;
                                }

                                for (var f = "function" == typeof l && l, e = 0; e < u.length; e++) h(u[e]);

                                return h;
                              }({
                                1: [function (l, t, n) {
                                  (function (r) {
                                    !function (e) {
                                      "object" == typeof n && void 0 !== t ? t.exports = e() : ("undefined" != typeof window ? window : void 0 !== r ? r : "undefined" != typeof self ? self : this).JSZip = e();
                                    }(function () {
                                      return function s(a, o, u) {
                                        function h(t, e) {
                                          if (!o[t]) {
                                            if (!a[t]) {
                                              var r = "function" == typeof l && l;
                                              if (!e && r) return r(t, !0);
                                              if (f) return f(t, !0);
                                              var n = new Error("Cannot find module '" + t + "'");
                                              throw n.code = "MODULE_NOT_FOUND", n;
                                            }

                                            var i = o[t] = {
                                              exports: {}
                                            };
                                            a[t][0].call(i.exports, function (e) {
                                              return h(a[t][1][e] || e);
                                            }, i, i.exports, s, a, o, u);
                                          }

                                          return o[t].exports;
                                        }

                                        for (var f = "function" == typeof l && l, e = 0; e < u.length; e++) h(u[e]);

                                        return h;
                                      }({
                                        1: [function (l, t, n) {
                                          (function (r) {
                                            !function (e) {
                                              "object" == typeof n && void 0 !== t ? t.exports = e() : ("undefined" != typeof window ? window : void 0 !== r ? r : "undefined" != typeof self ? self : this).JSZip = e();
                                            }(function () {
                                              return function s(a, o, u) {
                                                function h(t, e) {
                                                  if (!o[t]) {
                                                    if (!a[t]) {
                                                      var r = "function" == typeof l && l;
                                                      if (!e && r) return r(t, !0);
                                                      if (f) return f(t, !0);
                                                      var n = new Error("Cannot find module '" + t + "'");
                                                      throw n.code = "MODULE_NOT_FOUND", n;
                                                    }

                                                    var i = o[t] = {
                                                      exports: {}
                                                    };
                                                    a[t][0].call(i.exports, function (e) {
                                                      return h(a[t][1][e] || e);
                                                    }, i, i.exports, s, a, o, u);
                                                  }

                                                  return o[t].exports;
                                                }

                                                for (var f = "function" == typeof l && l, e = 0; e < u.length; e++) h(u[e]);

                                                return h;
                                              }({
                                                1: [function (e, t, r) {

                                                  var c = e("./utils"),
                                                      l = e("./support"),
                                                      p = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";
                                                  r.encode = function (e) {
                                                    for (var t, r, n, i, s, a, o, u = [], h = 0, f = e.length, l = f, d = "string" !== c.getTypeOf(e); h < e.length;) l = f - h, n = d ? (t = e[h++], r = h < f ? e[h++] : 0, h < f ? e[h++] : 0) : (t = e.charCodeAt(h++), r = h < f ? e.charCodeAt(h++) : 0, h < f ? e.charCodeAt(h++) : 0), i = t >> 2, s = (3 & t) << 4 | r >> 4, a = 1 < l ? (15 & r) << 2 | n >> 6 : 64, o = 2 < l ? 63 & n : 64, u.push(p.charAt(i) + p.charAt(s) + p.charAt(a) + p.charAt(o));

                                                    return u.join("");
                                                  }, r.decode = function (e) {
                                                    var t,
                                                        r,
                                                        n,
                                                        i,
                                                        s,
                                                        a,
                                                        o = 0,
                                                        u = 0;
                                                    if ("data:" === e.substr(0, "data:".length)) throw new Error("Invalid base64 input, it looks like a data url.");
                                                    var h,
                                                        f = 3 * (e = e.replace(/[^A-Za-z0-9\+\/\=]/g, "")).length / 4;
                                                    if (e.charAt(e.length - 1) === p.charAt(64) && f--, e.charAt(e.length - 2) === p.charAt(64) && f--, f % 1 != 0) throw new Error("Invalid base64 input, bad content length.");

                                                    for (h = l.uint8array ? new Uint8Array(0 | f) : new Array(0 | f); o < e.length;) t = p.indexOf(e.charAt(o++)) << 2 | (i = p.indexOf(e.charAt(o++))) >> 4, r = (15 & i) << 4 | (s = p.indexOf(e.charAt(o++))) >> 2, n = (3 & s) << 6 | (a = p.indexOf(e.charAt(o++))), h[u++] = t, 64 !== s && (h[u++] = r), 64 !== a && (h[u++] = n);

                                                    return h;
                                                  };
                                                }, {
                                                  "./support": 30,
                                                  "./utils": 32
                                                }],
                                                2: [function (e, t, r) {

                                                  var n = e("./external"),
                                                      i = e("./stream/DataWorker"),
                                                      s = e("./stream/Crc32Probe"),
                                                      a = e("./stream/DataLengthProbe");

                                                  function o(e, t, r, n, i) {
                                                    this.compressedSize = e, this.uncompressedSize = t, this.crc32 = r, this.compression = n, this.compressedContent = i;
                                                  }

                                                  o.prototype = {
                                                    getContentWorker: function () {
                                                      var e = new i(n.Promise.resolve(this.compressedContent)).pipe(this.compression.uncompressWorker()).pipe(new a("data_length")),
                                                          t = this;
                                                      return e.on("end", function () {
                                                        if (this.streamInfo.data_length !== t.uncompressedSize) throw new Error("Bug : uncompressed data size mismatch");
                                                      }), e;
                                                    },
                                                    getCompressedWorker: function () {
                                                      return new i(n.Promise.resolve(this.compressedContent)).withStreamInfo("compressedSize", this.compressedSize).withStreamInfo("uncompressedSize", this.uncompressedSize).withStreamInfo("crc32", this.crc32).withStreamInfo("compression", this.compression);
                                                    }
                                                  }, o.createWorkerFrom = function (e, t, r) {
                                                    return e.pipe(new s()).pipe(new a("uncompressedSize")).pipe(t.compressWorker(r)).pipe(new a("compressedSize")).withStreamInfo("compression", t);
                                                  }, t.exports = o;
                                                }, {
                                                  "./external": 6,
                                                  "./stream/Crc32Probe": 25,
                                                  "./stream/DataLengthProbe": 26,
                                                  "./stream/DataWorker": 27
                                                }],
                                                3: [function (e, t, r) {

                                                  var n = e("./stream/GenericWorker");
                                                  r.STORE = {
                                                    magic: "\0\0",
                                                    compressWorker: function (e) {
                                                      return new n("STORE compression");
                                                    },
                                                    uncompressWorker: function () {
                                                      return new n("STORE decompression");
                                                    }
                                                  }, r.DEFLATE = e("./flate");
                                                }, {
                                                  "./flate": 7,
                                                  "./stream/GenericWorker": 28
                                                }],
                                                4: [function (e, t, r) {

                                                  var n = e("./utils"),
                                                      a = function () {
                                                    for (var e, t = [], r = 0; r < 256; r++) {
                                                      e = r;

                                                      for (var n = 0; n < 8; n++) e = 1 & e ? 3988292384 ^ e >>> 1 : e >>> 1;

                                                      t[r] = e;
                                                    }

                                                    return t;
                                                  }();

                                                  t.exports = function (e, t) {
                                                    return void 0 !== e && e.length ? "string" !== n.getTypeOf(e) ? function (e, t, r) {
                                                      var n = a,
                                                          i = 0 + r;
                                                      e ^= -1;

                                                      for (var s = 0; s < i; s++) e = e >>> 8 ^ n[255 & (e ^ t[s])];

                                                      return -1 ^ e;
                                                    }(0 | t, e, e.length) : function (e, t, r) {
                                                      var n = a,
                                                          i = 0 + r;
                                                      e ^= -1;

                                                      for (var s = 0; s < i; s++) e = e >>> 8 ^ n[255 & (e ^ t.charCodeAt(s))];

                                                      return -1 ^ e;
                                                    }(0 | t, e, e.length) : 0;
                                                  };
                                                }, {
                                                  "./utils": 32
                                                }],
                                                5: [function (e, t, r) {

                                                  r.base64 = !1, r.binary = !1, r.dir = !1, r.createFolders = !0, r.date = null, r.compression = null, r.compressionOptions = null, r.comment = null, r.unixPermissions = null, r.dosPermissions = null;
                                                }, {}],
                                                6: [function (e, t, r) {

                                                  var n;
                                                  n = "undefined" != typeof Promise ? Promise : e("lie"), t.exports = {
                                                    Promise: n
                                                  };
                                                }, {
                                                  lie: 37
                                                }],
                                                7: [function (e, t, r) {

                                                  var n = "undefined" != typeof Uint8Array && "undefined" != typeof Uint16Array && "undefined" != typeof Uint32Array,
                                                      i = e("pako"),
                                                      s = e("./utils"),
                                                      a = e("./stream/GenericWorker"),
                                                      o = n ? "uint8array" : "array";

                                                  function u(e, t) {
                                                    a.call(this, "FlateWorker/" + e), this._pako = null, this._pakoAction = e, this._pakoOptions = t, this.meta = {};
                                                  }

                                                  r.magic = "\b\0", s.inherits(u, a), u.prototype.processChunk = function (e) {
                                                    this.meta = e.meta, null === this._pako && this._createPako(), this._pako.push(s.transformTo(o, e.data), !1);
                                                  }, u.prototype.flush = function () {
                                                    a.prototype.flush.call(this), null === this._pako && this._createPako(), this._pako.push([], !0);
                                                  }, u.prototype.cleanUp = function () {
                                                    a.prototype.cleanUp.call(this), this._pako = null;
                                                  }, u.prototype._createPako = function () {
                                                    this._pako = new i[this._pakoAction]({
                                                      raw: !0,
                                                      level: this._pakoOptions.level || -1
                                                    });
                                                    var t = this;

                                                    this._pako.onData = function (e) {
                                                      t.push({
                                                        data: e,
                                                        meta: t.meta
                                                      });
                                                    };
                                                  }, r.compressWorker = function (e) {
                                                    return new u("Deflate", e);
                                                  }, r.uncompressWorker = function () {
                                                    return new u("Inflate", {});
                                                  };
                                                }, {
                                                  "./stream/GenericWorker": 28,
                                                  "./utils": 32,
                                                  pako: 38
                                                }],
                                                8: [function (e, t, r) {

                                                  function I(e, t) {
                                                    var r,
                                                        n = "";

                                                    for (r = 0; r < t; r++) n += String.fromCharCode(255 & e), e >>>= 8;

                                                    return n;
                                                  }

                                                  function i(e, t, r, n, i, s) {
                                                    var a,
                                                        o,
                                                        u = e.file,
                                                        h = e.compression,
                                                        f = s !== B.utf8encode,
                                                        l = O.transformTo("string", s(u.name)),
                                                        d = O.transformTo("string", B.utf8encode(u.name)),
                                                        c = u.comment,
                                                        p = O.transformTo("string", s(c)),
                                                        m = O.transformTo("string", B.utf8encode(c)),
                                                        _ = d.length !== u.name.length,
                                                        g = m.length !== c.length,
                                                        v = "",
                                                        b = "",
                                                        w = "",
                                                        y = u.dir,
                                                        k = u.date,
                                                        x = {
                                                      crc32: 0,
                                                      compressedSize: 0,
                                                      uncompressedSize: 0
                                                    };

                                                    t && !r || (x.crc32 = e.crc32, x.compressedSize = e.compressedSize, x.uncompressedSize = e.uncompressedSize);
                                                    var S = 0;
                                                    t && (S |= 8), f || !_ && !g || (S |= 2048);
                                                    var z,
                                                        C = 0,
                                                        E = 0;
                                                    y && (C |= 16), "UNIX" === i ? (E = 798, C |= ((z = u.unixPermissions) || (z = y ? 16893 : 33204), (65535 & z) << 16)) : (E = 20, C |= 63 & (u.dosPermissions || 0)), a = k.getUTCHours(), a <<= 6, a |= k.getUTCMinutes(), a <<= 5, a |= k.getUTCSeconds() / 2, o = k.getUTCFullYear() - 1980, o <<= 4, o |= k.getUTCMonth() + 1, o <<= 5, o |= k.getUTCDate(), _ && (v += "up" + I((b = I(1, 1) + I(R(l), 4) + d).length, 2) + b), g && (v += "uc" + I((w = I(1, 1) + I(R(p), 4) + m).length, 2) + w);
                                                    var A = "";
                                                    return A += "\n\0", A += I(S, 2), A += h.magic, A += I(a, 2), A += I(o, 2), A += I(x.crc32, 4), A += I(x.compressedSize, 4), A += I(x.uncompressedSize, 4), A += I(l.length, 2), A += I(v.length, 2), {
                                                      fileRecord: T.LOCAL_FILE_HEADER + A + l + v,
                                                      dirRecord: T.CENTRAL_FILE_HEADER + I(E, 2) + A + I(p.length, 2) + "\0\0\0\0" + I(C, 4) + I(n, 4) + l + v + p
                                                    };
                                                  }

                                                  var O = e("../utils"),
                                                      s = e("../stream/GenericWorker"),
                                                      B = e("../utf8"),
                                                      R = e("../crc32"),
                                                      T = e("../signature");

                                                  function n(e, t, r, n) {
                                                    s.call(this, "ZipFileWorker"), this.bytesWritten = 0, this.zipComment = t, this.zipPlatform = r, this.encodeFileName = n, this.streamFiles = e, this.accumulate = !1, this.contentBuffer = [], this.dirRecords = [], this.currentSourceOffset = 0, this.entriesCount = 0, this.currentFile = null, this._sources = [];
                                                  }

                                                  O.inherits(n, s), n.prototype.push = function (e) {
                                                    var t = e.meta.percent || 0,
                                                        r = this.entriesCount,
                                                        n = this._sources.length;
                                                    this.accumulate ? this.contentBuffer.push(e) : (this.bytesWritten += e.data.length, s.prototype.push.call(this, {
                                                      data: e.data,
                                                      meta: {
                                                        currentFile: this.currentFile,
                                                        percent: r ? (t + 100 * (r - n - 1)) / r : 100
                                                      }
                                                    }));
                                                  }, n.prototype.openedSource = function (e) {
                                                    this.currentSourceOffset = this.bytesWritten, this.currentFile = e.file.name;
                                                    var t = this.streamFiles && !e.file.dir;

                                                    if (t) {
                                                      var r = i(e, t, !1, this.currentSourceOffset, this.zipPlatform, this.encodeFileName);
                                                      this.push({
                                                        data: r.fileRecord,
                                                        meta: {
                                                          percent: 0
                                                        }
                                                      });
                                                    } else this.accumulate = !0;
                                                  }, n.prototype.closedSource = function (e) {
                                                    this.accumulate = !1;
                                                    var t,
                                                        r = this.streamFiles && !e.file.dir,
                                                        n = i(e, r, !0, this.currentSourceOffset, this.zipPlatform, this.encodeFileName);
                                                    if (this.dirRecords.push(n.dirRecord), r) this.push({
                                                      data: (t = e, T.DATA_DESCRIPTOR + I(t.crc32, 4) + I(t.compressedSize, 4) + I(t.uncompressedSize, 4)),
                                                      meta: {
                                                        percent: 100
                                                      }
                                                    });else for (this.push({
                                                      data: n.fileRecord,
                                                      meta: {
                                                        percent: 0
                                                      }
                                                    }); this.contentBuffer.length;) this.push(this.contentBuffer.shift());
                                                    this.currentFile = null;
                                                  }, n.prototype.flush = function () {
                                                    for (var e = this.bytesWritten, t = 0; t < this.dirRecords.length; t++) this.push({
                                                      data: this.dirRecords[t],
                                                      meta: {
                                                        percent: 100
                                                      }
                                                    });

                                                    var r,
                                                        n,
                                                        i,
                                                        s,
                                                        a,
                                                        o,
                                                        u = this.bytesWritten - e,
                                                        h = (r = this.dirRecords.length, n = u, i = e, s = this.zipComment, a = this.encodeFileName, o = O.transformTo("string", a(s)), T.CENTRAL_DIRECTORY_END + "\0\0\0\0" + I(r, 2) + I(r, 2) + I(n, 4) + I(i, 4) + I(o.length, 2) + o);
                                                    this.push({
                                                      data: h,
                                                      meta: {
                                                        percent: 100
                                                      }
                                                    });
                                                  }, n.prototype.prepareNextSource = function () {
                                                    this.previous = this._sources.shift(), this.openedSource(this.previous.streamInfo), this.isPaused ? this.previous.pause() : this.previous.resume();
                                                  }, n.prototype.registerPrevious = function (e) {
                                                    this._sources.push(e);

                                                    var t = this;
                                                    return e.on("data", function (e) {
                                                      t.processChunk(e);
                                                    }), e.on("end", function () {
                                                      t.closedSource(t.previous.streamInfo), t._sources.length ? t.prepareNextSource() : t.end();
                                                    }), e.on("error", function (e) {
                                                      t.error(e);
                                                    }), this;
                                                  }, n.prototype.resume = function () {
                                                    return !!s.prototype.resume.call(this) && (!this.previous && this._sources.length ? (this.prepareNextSource(), !0) : this.previous || this._sources.length || this.generatedError ? void 0 : (this.end(), !0));
                                                  }, n.prototype.error = function (e) {
                                                    var t = this._sources;
                                                    if (!s.prototype.error.call(this, e)) return !1;

                                                    for (var r = 0; r < t.length; r++) try {
                                                      t[r].error(e);
                                                    } catch (e) {}

                                                    return !0;
                                                  }, n.prototype.lock = function () {
                                                    s.prototype.lock.call(this);

                                                    for (var e = this._sources, t = 0; t < e.length; t++) e[t].lock();
                                                  }, t.exports = n;
                                                }, {
                                                  "../crc32": 4,
                                                  "../signature": 23,
                                                  "../stream/GenericWorker": 28,
                                                  "../utf8": 31,
                                                  "../utils": 32
                                                }],
                                                9: [function (e, t, r) {

                                                  var h = e("../compressions"),
                                                      n = e("./ZipFileWorker");

                                                  r.generateWorker = function (e, a, t) {
                                                    var o = new n(a.streamFiles, t, a.platform, a.encodeFileName),
                                                        u = 0;

                                                    try {
                                                      e.forEach(function (e, t) {
                                                        u++;

                                                        var r = function (e, t) {
                                                          var r = e || t,
                                                              n = h[r];
                                                          if (!n) throw new Error(r + " is not a valid compression method !");
                                                          return n;
                                                        }(t.options.compression, a.compression),
                                                            n = t.options.compressionOptions || a.compressionOptions || {},
                                                            i = t.dir,
                                                            s = t.date;

                                                        t._compressWorker(r, n).withStreamInfo("file", {
                                                          name: e,
                                                          dir: i,
                                                          date: s,
                                                          comment: t.comment || "",
                                                          unixPermissions: t.unixPermissions,
                                                          dosPermissions: t.dosPermissions
                                                        }).pipe(o);
                                                      }), o.entriesCount = u;
                                                    } catch (e) {
                                                      o.error(e);
                                                    }

                                                    return o;
                                                  };
                                                }, {
                                                  "../compressions": 3,
                                                  "./ZipFileWorker": 8
                                                }],
                                                10: [function (e, t, r) {

                                                  function n() {
                                                    if (!(this instanceof n)) return new n();
                                                    if (arguments.length) throw new Error("The constructor with parameters has been removed in JSZip 3.0, please check the upgrade guide.");
                                                    this.files = {}, this.comment = null, this.root = "", this.clone = function () {
                                                      var e = new n();

                                                      for (var t in this) "function" != typeof this[t] && (e[t] = this[t]);

                                                      return e;
                                                    };
                                                  }

                                                  (n.prototype = e("./object")).loadAsync = e("./load"), n.support = e("./support"), n.defaults = e("./defaults"), n.version = "3.5.0", n.loadAsync = function (e, t) {
                                                    return new n().loadAsync(e, t);
                                                  }, n.external = e("./external"), t.exports = n;
                                                }, {
                                                  "./defaults": 5,
                                                  "./external": 6,
                                                  "./load": 11,
                                                  "./object": 15,
                                                  "./support": 30
                                                }],
                                                11: [function (e, t, r) {

                                                  var n = e("./utils"),
                                                      i = e("./external"),
                                                      o = e("./utf8"),
                                                      u = e("./zipEntries"),
                                                      s = e("./stream/Crc32Probe"),
                                                      h = e("./nodejsUtils");

                                                  function f(n) {
                                                    return new i.Promise(function (e, t) {
                                                      var r = n.decompressed.getContentWorker().pipe(new s());
                                                      r.on("error", function (e) {
                                                        t(e);
                                                      }).on("end", function () {
                                                        r.streamInfo.crc32 !== n.decompressed.crc32 ? t(new Error("Corrupted zip : CRC32 mismatch")) : e();
                                                      }).resume();
                                                    });
                                                  }

                                                  t.exports = function (e, s) {
                                                    var a = this;
                                                    return s = n.extend(s || {}, {
                                                      base64: !1,
                                                      checkCRC32: !1,
                                                      optimizedBinaryString: !1,
                                                      createFolders: !1,
                                                      decodeFileName: o.utf8decode
                                                    }), h.isNode && h.isStream(e) ? i.Promise.reject(new Error("JSZip can't accept a stream when loading a zip file.")) : n.prepareContent("the loaded zip file", e, !0, s.optimizedBinaryString, s.base64).then(function (e) {
                                                      var t = new u(s);
                                                      return t.load(e), t;
                                                    }).then(function (e) {
                                                      var t = [i.Promise.resolve(e)],
                                                          r = e.files;
                                                      if (s.checkCRC32) for (var n = 0; n < r.length; n++) t.push(f(r[n]));
                                                      return i.Promise.all(t);
                                                    }).then(function (e) {
                                                      for (var t = e.shift(), r = t.files, n = 0; n < r.length; n++) {
                                                        var i = r[n];
                                                        a.file(i.fileNameStr, i.decompressed, {
                                                          binary: !0,
                                                          optimizedBinaryString: !0,
                                                          date: i.date,
                                                          dir: i.dir,
                                                          comment: i.fileCommentStr.length ? i.fileCommentStr : null,
                                                          unixPermissions: i.unixPermissions,
                                                          dosPermissions: i.dosPermissions,
                                                          createFolders: s.createFolders
                                                        });
                                                      }

                                                      return t.zipComment.length && (a.comment = t.zipComment), a;
                                                    });
                                                  };
                                                }, {
                                                  "./external": 6,
                                                  "./nodejsUtils": 14,
                                                  "./stream/Crc32Probe": 25,
                                                  "./utf8": 31,
                                                  "./utils": 32,
                                                  "./zipEntries": 33
                                                }],
                                                12: [function (e, t, r) {

                                                  var n = e("../utils"),
                                                      i = e("../stream/GenericWorker");

                                                  function s(e, t) {
                                                    i.call(this, "Nodejs stream input adapter for " + e), this._upstreamEnded = !1, this._bindStream(t);
                                                  }

                                                  n.inherits(s, i), s.prototype._bindStream = function (e) {
                                                    var t = this;
                                                    (this._stream = e).pause(), e.on("data", function (e) {
                                                      t.push({
                                                        data: e,
                                                        meta: {
                                                          percent: 0
                                                        }
                                                      });
                                                    }).on("error", function (e) {
                                                      t.isPaused ? this.generatedError = e : t.error(e);
                                                    }).on("end", function () {
                                                      t.isPaused ? t._upstreamEnded = !0 : t.end();
                                                    });
                                                  }, s.prototype.pause = function () {
                                                    return !!i.prototype.pause.call(this) && (this._stream.pause(), !0);
                                                  }, s.prototype.resume = function () {
                                                    return !!i.prototype.resume.call(this) && (this._upstreamEnded ? this.end() : this._stream.resume(), !0);
                                                  }, t.exports = s;
                                                }, {
                                                  "../stream/GenericWorker": 28,
                                                  "../utils": 32
                                                }],
                                                13: [function (e, t, r) {

                                                  var i = e("readable-stream").Readable;

                                                  function n(e, t, r) {
                                                    i.call(this, t), this._helper = e;
                                                    var n = this;
                                                    e.on("data", function (e, t) {
                                                      n.push(e) || n._helper.pause(), r && r(t);
                                                    }).on("error", function (e) {
                                                      n.emit("error", e);
                                                    }).on("end", function () {
                                                      n.push(null);
                                                    });
                                                  }

                                                  e("../utils").inherits(n, i), n.prototype._read = function () {
                                                    this._helper.resume();
                                                  }, t.exports = n;
                                                }, {
                                                  "../utils": 32,
                                                  "readable-stream": 16
                                                }],
                                                14: [function (e, t, r) {

                                                  t.exports = {
                                                    isNode: "undefined" != typeof Buffer,
                                                    newBufferFrom: function (e, t) {
                                                      if (Buffer.from && Buffer.from !== Uint8Array.from) return Buffer.from(e, t);
                                                      if ("number" == typeof e) throw new Error('The "data" argument must not be a number');
                                                      return new Buffer(e, t);
                                                    },
                                                    allocBuffer: function (e) {
                                                      if (Buffer.alloc) return Buffer.alloc(e);
                                                      var t = new Buffer(e);
                                                      return t.fill(0), t;
                                                    },
                                                    isBuffer: function (e) {
                                                      return Buffer.isBuffer(e);
                                                    },
                                                    isStream: function (e) {
                                                      return e && "function" == typeof e.on && "function" == typeof e.pause && "function" == typeof e.resume;
                                                    }
                                                  };
                                                }, {}],
                                                15: [function (e, t, r) {

                                                  function s(e, t, r) {
                                                    var n,
                                                        i = f.getTypeOf(t),
                                                        s = f.extend(r || {}, d);
                                                    s.date = s.date || new Date(), null !== s.compression && (s.compression = s.compression.toUpperCase()), "string" == typeof s.unixPermissions && (s.unixPermissions = parseInt(s.unixPermissions, 8)), s.unixPermissions && 16384 & s.unixPermissions && (s.dir = !0), s.dosPermissions && 16 & s.dosPermissions && (s.dir = !0), s.dir && (e = h(e)), s.createFolders && (n = function (e) {
                                                      "/" === e.slice(-1) && (e = e.substring(0, e.length - 1));
                                                      var t = e.lastIndexOf("/");
                                                      return 0 < t ? e.substring(0, t) : "";
                                                    }(e)) && g.call(this, n, !0);
                                                    var a,
                                                        o = "string" === i && !1 === s.binary && !1 === s.base64;
                                                    r && void 0 !== r.binary || (s.binary = !o), (t instanceof c && 0 === t.uncompressedSize || s.dir || !t || 0 === t.length) && (s.base64 = !1, s.binary = !0, t = "", s.compression = "STORE", i = "string"), a = t instanceof c || t instanceof l ? t : m.isNode && m.isStream(t) ? new _(e, t) : f.prepareContent(e, t, s.binary, s.optimizedBinaryString, s.base64);
                                                    var u = new p(e, a, s);
                                                    this.files[e] = u;
                                                  }

                                                  function h(e) {
                                                    return "/" !== e.slice(-1) && (e += "/"), e;
                                                  }

                                                  var i = e("./utf8"),
                                                      f = e("./utils"),
                                                      l = e("./stream/GenericWorker"),
                                                      a = e("./stream/StreamHelper"),
                                                      d = e("./defaults"),
                                                      c = e("./compressedObject"),
                                                      p = e("./zipObject"),
                                                      o = e("./generate"),
                                                      m = e("./nodejsUtils"),
                                                      _ = e("./nodejs/NodejsStreamInputAdapter"),
                                                      g = function (e, t) {
                                                    return t = void 0 !== t ? t : d.createFolders, e = h(e), this.files[e] || s.call(this, e, null, {
                                                      dir: !0,
                                                      createFolders: t
                                                    }), this.files[e];
                                                  };

                                                  function u(e) {
                                                    return "[object RegExp]" === Object.prototype.toString.call(e);
                                                  }

                                                  var n = {
                                                    load: function () {
                                                      throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.");
                                                    },
                                                    forEach: function (e) {
                                                      var t, r, n;

                                                      for (t in this.files) this.files.hasOwnProperty(t) && (n = this.files[t], (r = t.slice(this.root.length, t.length)) && t.slice(0, this.root.length) === this.root && e(r, n));
                                                    },
                                                    filter: function (r) {
                                                      var n = [];
                                                      return this.forEach(function (e, t) {
                                                        r(e, t) && n.push(t);
                                                      }), n;
                                                    },
                                                    file: function (e, t, r) {
                                                      if (1 !== arguments.length) return e = this.root + e, s.call(this, e, t, r), this;

                                                      if (u(e)) {
                                                        var n = e;
                                                        return this.filter(function (e, t) {
                                                          return !t.dir && n.test(e);
                                                        });
                                                      }

                                                      var i = this.files[this.root + e];
                                                      return i && !i.dir ? i : null;
                                                    },
                                                    folder: function (r) {
                                                      if (!r) return this;
                                                      if (u(r)) return this.filter(function (e, t) {
                                                        return t.dir && r.test(e);
                                                      });
                                                      var e = this.root + r,
                                                          t = g.call(this, e),
                                                          n = this.clone();
                                                      return n.root = t.name, n;
                                                    },
                                                    remove: function (r) {
                                                      r = this.root + r;
                                                      var e = this.files[r];
                                                      if (e || ("/" !== r.slice(-1) && (r += "/"), e = this.files[r]), e && !e.dir) delete this.files[r];else for (var t = this.filter(function (e, t) {
                                                        return t.name.slice(0, r.length) === r;
                                                      }), n = 0; n < t.length; n++) delete this.files[t[n].name];
                                                      return this;
                                                    },
                                                    generate: function (e) {
                                                      throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.");
                                                    },
                                                    generateInternalStream: function (e) {
                                                      var t,
                                                          r = {};

                                                      try {
                                                        if ((r = f.extend(e || {}, {
                                                          streamFiles: !1,
                                                          compression: "STORE",
                                                          compressionOptions: null,
                                                          type: "",
                                                          platform: "DOS",
                                                          comment: null,
                                                          mimeType: "application/zip",
                                                          encodeFileName: i.utf8encode
                                                        })).type = r.type.toLowerCase(), r.compression = r.compression.toUpperCase(), "binarystring" === r.type && (r.type = "string"), !r.type) throw new Error("No output type specified.");
                                                        f.checkSupport(r.type), "darwin" !== r.platform && "freebsd" !== r.platform && "linux" !== r.platform && "sunos" !== r.platform || (r.platform = "UNIX"), "win32" === r.platform && (r.platform = "DOS");
                                                        var n = r.comment || this.comment || "";
                                                        t = o.generateWorker(this, r, n);
                                                      } catch (e) {
                                                        (t = new l("error")).error(e);
                                                      }

                                                      return new a(t, r.type || "string", r.mimeType);
                                                    },
                                                    generateAsync: function (e, t) {
                                                      return this.generateInternalStream(e).accumulate(t);
                                                    },
                                                    generateNodeStream: function (e, t) {
                                                      return (e = e || {}).type || (e.type = "nodebuffer"), this.generateInternalStream(e).toNodejsStream(t);
                                                    }
                                                  };
                                                  t.exports = n;
                                                }, {
                                                  "./compressedObject": 2,
                                                  "./defaults": 5,
                                                  "./generate": 9,
                                                  "./nodejs/NodejsStreamInputAdapter": 12,
                                                  "./nodejsUtils": 14,
                                                  "./stream/GenericWorker": 28,
                                                  "./stream/StreamHelper": 29,
                                                  "./utf8": 31,
                                                  "./utils": 32,
                                                  "./zipObject": 35
                                                }],
                                                16: [function (e, t, r) {
                                                  t.exports = e("stream");
                                                }, {
                                                  stream: void 0
                                                }],
                                                17: [function (e, t, r) {

                                                  var n = e("./DataReader");

                                                  function i(e) {
                                                    n.call(this, e);

                                                    for (var t = 0; t < this.data.length; t++) e[t] = 255 & e[t];
                                                  }

                                                  e("../utils").inherits(i, n), i.prototype.byteAt = function (e) {
                                                    return this.data[this.zero + e];
                                                  }, i.prototype.lastIndexOfSignature = function (e) {
                                                    for (var t = e.charCodeAt(0), r = e.charCodeAt(1), n = e.charCodeAt(2), i = e.charCodeAt(3), s = this.length - 4; 0 <= s; --s) if (this.data[s] === t && this.data[s + 1] === r && this.data[s + 2] === n && this.data[s + 3] === i) return s - this.zero;

                                                    return -1;
                                                  }, i.prototype.readAndCheckSignature = function (e) {
                                                    var t = e.charCodeAt(0),
                                                        r = e.charCodeAt(1),
                                                        n = e.charCodeAt(2),
                                                        i = e.charCodeAt(3),
                                                        s = this.readData(4);
                                                    return t === s[0] && r === s[1] && n === s[2] && i === s[3];
                                                  }, i.prototype.readData = function (e) {
                                                    if (this.checkOffset(e), 0 === e) return [];
                                                    var t = this.data.slice(this.zero + this.index, this.zero + this.index + e);
                                                    return this.index += e, t;
                                                  }, t.exports = i;
                                                }, {
                                                  "../utils": 32,
                                                  "./DataReader": 18
                                                }],
                                                18: [function (e, t, r) {

                                                  var n = e("../utils");

                                                  function i(e) {
                                                    this.data = e, this.length = e.length, this.index = 0, this.zero = 0;
                                                  }

                                                  i.prototype = {
                                                    checkOffset: function (e) {
                                                      this.checkIndex(this.index + e);
                                                    },
                                                    checkIndex: function (e) {
                                                      if (this.length < this.zero + e || e < 0) throw new Error("End of data reached (data length = " + this.length + ", asked index = " + e + "). Corrupted zip ?");
                                                    },
                                                    setIndex: function (e) {
                                                      this.checkIndex(e), this.index = e;
                                                    },
                                                    skip: function (e) {
                                                      this.setIndex(this.index + e);
                                                    },
                                                    byteAt: function (e) {},
                                                    readInt: function (e) {
                                                      var t,
                                                          r = 0;

                                                      for (this.checkOffset(e), t = this.index + e - 1; t >= this.index; t--) r = (r << 8) + this.byteAt(t);

                                                      return this.index += e, r;
                                                    },
                                                    readString: function (e) {
                                                      return n.transformTo("string", this.readData(e));
                                                    },
                                                    readData: function (e) {},
                                                    lastIndexOfSignature: function (e) {},
                                                    readAndCheckSignature: function (e) {},
                                                    readDate: function () {
                                                      var e = this.readInt(4);
                                                      return new Date(Date.UTC(1980 + (e >> 25 & 127), (e >> 21 & 15) - 1, e >> 16 & 31, e >> 11 & 31, e >> 5 & 63, (31 & e) << 1));
                                                    }
                                                  }, t.exports = i;
                                                }, {
                                                  "../utils": 32
                                                }],
                                                19: [function (e, t, r) {

                                                  var n = e("./Uint8ArrayReader");

                                                  function i(e) {
                                                    n.call(this, e);
                                                  }

                                                  e("../utils").inherits(i, n), i.prototype.readData = function (e) {
                                                    this.checkOffset(e);
                                                    var t = this.data.slice(this.zero + this.index, this.zero + this.index + e);
                                                    return this.index += e, t;
                                                  }, t.exports = i;
                                                }, {
                                                  "../utils": 32,
                                                  "./Uint8ArrayReader": 21
                                                }],
                                                20: [function (e, t, r) {

                                                  var n = e("./DataReader");

                                                  function i(e) {
                                                    n.call(this, e);
                                                  }

                                                  e("../utils").inherits(i, n), i.prototype.byteAt = function (e) {
                                                    return this.data.charCodeAt(this.zero + e);
                                                  }, i.prototype.lastIndexOfSignature = function (e) {
                                                    return this.data.lastIndexOf(e) - this.zero;
                                                  }, i.prototype.readAndCheckSignature = function (e) {
                                                    return e === this.readData(4);
                                                  }, i.prototype.readData = function (e) {
                                                    this.checkOffset(e);
                                                    var t = this.data.slice(this.zero + this.index, this.zero + this.index + e);
                                                    return this.index += e, t;
                                                  }, t.exports = i;
                                                }, {
                                                  "../utils": 32,
                                                  "./DataReader": 18
                                                }],
                                                21: [function (e, t, r) {

                                                  var n = e("./ArrayReader");

                                                  function i(e) {
                                                    n.call(this, e);
                                                  }

                                                  e("../utils").inherits(i, n), i.prototype.readData = function (e) {
                                                    if (this.checkOffset(e), 0 === e) return new Uint8Array(0);
                                                    var t = this.data.subarray(this.zero + this.index, this.zero + this.index + e);
                                                    return this.index += e, t;
                                                  }, t.exports = i;
                                                }, {
                                                  "../utils": 32,
                                                  "./ArrayReader": 17
                                                }],
                                                22: [function (e, t, r) {

                                                  var n = e("../utils"),
                                                      i = e("../support"),
                                                      s = e("./ArrayReader"),
                                                      a = e("./StringReader"),
                                                      o = e("./NodeBufferReader"),
                                                      u = e("./Uint8ArrayReader");

                                                  t.exports = function (e) {
                                                    var t = n.getTypeOf(e);
                                                    return n.checkSupport(t), "string" !== t || i.uint8array ? "nodebuffer" === t ? new o(e) : i.uint8array ? new u(n.transformTo("uint8array", e)) : new s(n.transformTo("array", e)) : new a(e);
                                                  };
                                                }, {
                                                  "../support": 30,
                                                  "../utils": 32,
                                                  "./ArrayReader": 17,
                                                  "./NodeBufferReader": 19,
                                                  "./StringReader": 20,
                                                  "./Uint8ArrayReader": 21
                                                }],
                                                23: [function (e, t, r) {

                                                  r.LOCAL_FILE_HEADER = "PK", r.CENTRAL_FILE_HEADER = "PK", r.CENTRAL_DIRECTORY_END = "PK", r.ZIP64_CENTRAL_DIRECTORY_LOCATOR = "PK", r.ZIP64_CENTRAL_DIRECTORY_END = "PK", r.DATA_DESCRIPTOR = "PK\b";
                                                }, {}],
                                                24: [function (e, t, r) {

                                                  var n = e("./GenericWorker"),
                                                      i = e("../utils");

                                                  function s(e) {
                                                    n.call(this, "ConvertWorker to " + e), this.destType = e;
                                                  }

                                                  i.inherits(s, n), s.prototype.processChunk = function (e) {
                                                    this.push({
                                                      data: i.transformTo(this.destType, e.data),
                                                      meta: e.meta
                                                    });
                                                  }, t.exports = s;
                                                }, {
                                                  "../utils": 32,
                                                  "./GenericWorker": 28
                                                }],
                                                25: [function (e, t, r) {

                                                  var n = e("./GenericWorker"),
                                                      i = e("../crc32");

                                                  function s() {
                                                    n.call(this, "Crc32Probe"), this.withStreamInfo("crc32", 0);
                                                  }

                                                  e("../utils").inherits(s, n), s.prototype.processChunk = function (e) {
                                                    this.streamInfo.crc32 = i(e.data, this.streamInfo.crc32 || 0), this.push(e);
                                                  }, t.exports = s;
                                                }, {
                                                  "../crc32": 4,
                                                  "../utils": 32,
                                                  "./GenericWorker": 28
                                                }],
                                                26: [function (e, t, r) {

                                                  var n = e("../utils"),
                                                      i = e("./GenericWorker");

                                                  function s(e) {
                                                    i.call(this, "DataLengthProbe for " + e), this.propName = e, this.withStreamInfo(e, 0);
                                                  }

                                                  n.inherits(s, i), s.prototype.processChunk = function (e) {
                                                    if (e) {
                                                      var t = this.streamInfo[this.propName] || 0;
                                                      this.streamInfo[this.propName] = t + e.data.length;
                                                    }

                                                    i.prototype.processChunk.call(this, e);
                                                  }, t.exports = s;
                                                }, {
                                                  "../utils": 32,
                                                  "./GenericWorker": 28
                                                }],
                                                27: [function (e, t, r) {

                                                  var n = e("../utils"),
                                                      i = e("./GenericWorker");

                                                  function s(e) {
                                                    i.call(this, "DataWorker");
                                                    var t = this;
                                                    this.dataIsReady = !1, this.index = 0, this.max = 0, this.data = null, this.type = "", this._tickScheduled = !1, e.then(function (e) {
                                                      t.dataIsReady = !0, t.data = e, t.max = e && e.length || 0, t.type = n.getTypeOf(e), t.isPaused || t._tickAndRepeat();
                                                    }, function (e) {
                                                      t.error(e);
                                                    });
                                                  }

                                                  n.inherits(s, i), s.prototype.cleanUp = function () {
                                                    i.prototype.cleanUp.call(this), this.data = null;
                                                  }, s.prototype.resume = function () {
                                                    return !!i.prototype.resume.call(this) && (!this._tickScheduled && this.dataIsReady && (this._tickScheduled = !0, n.delay(this._tickAndRepeat, [], this)), !0);
                                                  }, s.prototype._tickAndRepeat = function () {
                                                    this._tickScheduled = !1, this.isPaused || this.isFinished || (this._tick(), this.isFinished || (n.delay(this._tickAndRepeat, [], this), this._tickScheduled = !0));
                                                  }, s.prototype._tick = function () {
                                                    if (this.isPaused || this.isFinished) return !1;
                                                    var e = null,
                                                        t = Math.min(this.max, this.index + 16384);
                                                    if (this.index >= this.max) return this.end();

                                                    switch (this.type) {
                                                      case "string":
                                                        e = this.data.substring(this.index, t);
                                                        break;

                                                      case "uint8array":
                                                        e = this.data.subarray(this.index, t);
                                                        break;

                                                      case "array":
                                                      case "nodebuffer":
                                                        e = this.data.slice(this.index, t);
                                                    }

                                                    return this.index = t, this.push({
                                                      data: e,
                                                      meta: {
                                                        percent: this.max ? this.index / this.max * 100 : 0
                                                      }
                                                    });
                                                  }, t.exports = s;
                                                }, {
                                                  "../utils": 32,
                                                  "./GenericWorker": 28
                                                }],
                                                28: [function (e, t, r) {

                                                  function n(e) {
                                                    this.name = e || "default", this.streamInfo = {}, this.generatedError = null, this.extraStreamInfo = {}, this.isPaused = !0, this.isFinished = !1, this.isLocked = !1, this._listeners = {
                                                      data: [],
                                                      end: [],
                                                      error: []
                                                    }, this.previous = null;
                                                  }

                                                  n.prototype = {
                                                    push: function (e) {
                                                      this.emit("data", e);
                                                    },
                                                    end: function () {
                                                      if (this.isFinished) return !1;
                                                      this.flush();

                                                      try {
                                                        this.emit("end"), this.cleanUp(), this.isFinished = !0;
                                                      } catch (e) {
                                                        this.emit("error", e);
                                                      }

                                                      return !0;
                                                    },
                                                    error: function (e) {
                                                      return !this.isFinished && (this.isPaused ? this.generatedError = e : (this.isFinished = !0, this.emit("error", e), this.previous && this.previous.error(e), this.cleanUp()), !0);
                                                    },
                                                    on: function (e, t) {
                                                      return this._listeners[e].push(t), this;
                                                    },
                                                    cleanUp: function () {
                                                      this.streamInfo = this.generatedError = this.extraStreamInfo = null, this._listeners = [];
                                                    },
                                                    emit: function (e, t) {
                                                      if (this._listeners[e]) for (var r = 0; r < this._listeners[e].length; r++) this._listeners[e][r].call(this, t);
                                                    },
                                                    pipe: function (e) {
                                                      return e.registerPrevious(this);
                                                    },
                                                    registerPrevious: function (e) {
                                                      if (this.isLocked) throw new Error("The stream '" + this + "' has already been used.");
                                                      this.streamInfo = e.streamInfo, this.mergeStreamInfo(), this.previous = e;
                                                      var t = this;
                                                      return e.on("data", function (e) {
                                                        t.processChunk(e);
                                                      }), e.on("end", function () {
                                                        t.end();
                                                      }), e.on("error", function (e) {
                                                        t.error(e);
                                                      }), this;
                                                    },
                                                    pause: function () {
                                                      return !this.isPaused && !this.isFinished && (this.isPaused = !0, this.previous && this.previous.pause(), !0);
                                                    },
                                                    resume: function () {
                                                      if (!this.isPaused || this.isFinished) return !1;
                                                      var e = this.isPaused = !1;
                                                      return this.generatedError && (this.error(this.generatedError), e = !0), this.previous && this.previous.resume(), !e;
                                                    },
                                                    flush: function () {},
                                                    processChunk: function (e) {
                                                      this.push(e);
                                                    },
                                                    withStreamInfo: function (e, t) {
                                                      return this.extraStreamInfo[e] = t, this.mergeStreamInfo(), this;
                                                    },
                                                    mergeStreamInfo: function () {
                                                      for (var e in this.extraStreamInfo) this.extraStreamInfo.hasOwnProperty(e) && (this.streamInfo[e] = this.extraStreamInfo[e]);
                                                    },
                                                    lock: function () {
                                                      if (this.isLocked) throw new Error("The stream '" + this + "' has already been used.");
                                                      this.isLocked = !0, this.previous && this.previous.lock();
                                                    },
                                                    toString: function () {
                                                      var e = "Worker " + this.name;
                                                      return this.previous ? this.previous + " -> " + e : e;
                                                    }
                                                  }, t.exports = n;
                                                }, {}],
                                                29: [function (e, t, r) {

                                                  var h = e("../utils"),
                                                      i = e("./ConvertWorker"),
                                                      s = e("./GenericWorker"),
                                                      f = e("../base64"),
                                                      n = e("../support"),
                                                      a = e("../external"),
                                                      o = null;
                                                  if (n.nodestream) try {
                                                    o = e("../nodejs/NodejsStreamOutputAdapter");
                                                  } catch (e) {}

                                                  function u(e, t, r) {
                                                    var n = t;

                                                    switch (t) {
                                                      case "blob":
                                                      case "arraybuffer":
                                                        n = "uint8array";
                                                        break;

                                                      case "base64":
                                                        n = "string";
                                                    }

                                                    try {
                                                      this._internalType = n, this._outputType = t, this._mimeType = r, h.checkSupport(n), this._worker = e.pipe(new i(n)), e.lock();
                                                    } catch (e) {
                                                      this._worker = new s("error"), this._worker.error(e);
                                                    }
                                                  }

                                                  u.prototype = {
                                                    accumulate: function (e) {
                                                      return o = this, u = e, new a.Promise(function (t, r) {
                                                        var n = [],
                                                            i = o._internalType,
                                                            s = o._outputType,
                                                            a = o._mimeType;
                                                        o.on("data", function (e, t) {
                                                          n.push(e), u && u(t);
                                                        }).on("error", function (e) {
                                                          n = [], r(e);
                                                        }).on("end", function () {
                                                          try {
                                                            var e = function (e, t, r) {
                                                              switch (e) {
                                                                case "blob":
                                                                  return h.newBlob(h.transformTo("arraybuffer", t), r);

                                                                case "base64":
                                                                  return f.encode(t);

                                                                default:
                                                                  return h.transformTo(e, t);
                                                              }
                                                            }(s, function (e, t) {
                                                              var r,
                                                                  n = 0,
                                                                  i = null,
                                                                  s = 0;

                                                              for (r = 0; r < t.length; r++) s += t[r].length;

                                                              switch (e) {
                                                                case "string":
                                                                  return t.join("");

                                                                case "array":
                                                                  return Array.prototype.concat.apply([], t);

                                                                case "uint8array":
                                                                  for (i = new Uint8Array(s), r = 0; r < t.length; r++) i.set(t[r], n), n += t[r].length;

                                                                  return i;

                                                                case "nodebuffer":
                                                                  return Buffer.concat(t);

                                                                default:
                                                                  throw new Error("concat : unsupported type '" + e + "'");
                                                              }
                                                            }(i, n), a);

                                                            t(e);
                                                          } catch (e) {
                                                            r(e);
                                                          }

                                                          n = [];
                                                        }).resume();
                                                      });
                                                      var o, u;
                                                    },
                                                    on: function (e, t) {
                                                      var r = this;
                                                      return "data" === e ? this._worker.on(e, function (e) {
                                                        t.call(r, e.data, e.meta);
                                                      }) : this._worker.on(e, function () {
                                                        h.delay(t, arguments, r);
                                                      }), this;
                                                    },
                                                    resume: function () {
                                                      return h.delay(this._worker.resume, [], this._worker), this;
                                                    },
                                                    pause: function () {
                                                      return this._worker.pause(), this;
                                                    },
                                                    toNodejsStream: function (e) {
                                                      if (h.checkSupport("nodestream"), "nodebuffer" !== this._outputType) throw new Error(this._outputType + " is not supported by this method");
                                                      return new o(this, {
                                                        objectMode: "nodebuffer" !== this._outputType
                                                      }, e);
                                                    }
                                                  }, t.exports = u;
                                                }, {
                                                  "../base64": 1,
                                                  "../external": 6,
                                                  "../nodejs/NodejsStreamOutputAdapter": 13,
                                                  "../support": 30,
                                                  "../utils": 32,
                                                  "./ConvertWorker": 24,
                                                  "./GenericWorker": 28
                                                }],
                                                30: [function (e, t, r) {

                                                  if (r.base64 = !0, r.array = !0, r.string = !0, r.arraybuffer = "undefined" != typeof ArrayBuffer && "undefined" != typeof Uint8Array, r.nodebuffer = "undefined" != typeof Buffer, r.uint8array = "undefined" != typeof Uint8Array, "undefined" == typeof ArrayBuffer) r.blob = !1;else {
                                                    var n = new ArrayBuffer(0);

                                                    try {
                                                      r.blob = 0 === new Blob([n], {
                                                        type: "application/zip"
                                                      }).size;
                                                    } catch (e) {
                                                      try {
                                                        var i = new (self.BlobBuilder || self.WebKitBlobBuilder || self.MozBlobBuilder || self.MSBlobBuilder)();
                                                        i.append(n), r.blob = 0 === i.getBlob("application/zip").size;
                                                      } catch (e) {
                                                        r.blob = !1;
                                                      }
                                                    }
                                                  }

                                                  try {
                                                    r.nodestream = !!e("readable-stream").Readable;
                                                  } catch (e) {
                                                    r.nodestream = !1;
                                                  }
                                                }, {
                                                  "readable-stream": 16
                                                }],
                                                31: [function (e, t, s) {

                                                  for (var o = e("./utils"), u = e("./support"), r = e("./nodejsUtils"), n = e("./stream/GenericWorker"), h = new Array(256), i = 0; i < 256; i++) h[i] = 252 <= i ? 6 : 248 <= i ? 5 : 240 <= i ? 4 : 224 <= i ? 3 : 192 <= i ? 2 : 1;

                                                  function a() {
                                                    n.call(this, "utf-8 decode"), this.leftOver = null;
                                                  }

                                                  function f() {
                                                    n.call(this, "utf-8 encode");
                                                  }

                                                  h[254] = h[254] = 1, s.utf8encode = function (e) {
                                                    return u.nodebuffer ? r.newBufferFrom(e, "utf-8") : function (e) {
                                                      var t,
                                                          r,
                                                          n,
                                                          i,
                                                          s,
                                                          a = e.length,
                                                          o = 0;

                                                      for (i = 0; i < a; i++) 55296 == (64512 & (r = e.charCodeAt(i))) && i + 1 < a && 56320 == (64512 & (n = e.charCodeAt(i + 1))) && (r = 65536 + (r - 55296 << 10) + (n - 56320), i++), o += r < 128 ? 1 : r < 2048 ? 2 : r < 65536 ? 3 : 4;

                                                      for (t = u.uint8array ? new Uint8Array(o) : new Array(o), i = s = 0; s < o; i++) 55296 == (64512 & (r = e.charCodeAt(i))) && i + 1 < a && 56320 == (64512 & (n = e.charCodeAt(i + 1))) && (r = 65536 + (r - 55296 << 10) + (n - 56320), i++), r < 128 ? t[s++] = r : (r < 2048 ? t[s++] = 192 | r >>> 6 : (r < 65536 ? t[s++] = 224 | r >>> 12 : (t[s++] = 240 | r >>> 18, t[s++] = 128 | r >>> 12 & 63), t[s++] = 128 | r >>> 6 & 63), t[s++] = 128 | 63 & r);

                                                      return t;
                                                    }(e);
                                                  }, s.utf8decode = function (e) {
                                                    return u.nodebuffer ? o.transformTo("nodebuffer", e).toString("utf-8") : function (e) {
                                                      var t,
                                                          r,
                                                          n,
                                                          i,
                                                          s = e.length,
                                                          a = new Array(2 * s);

                                                      for (t = r = 0; t < s;) if ((n = e[t++]) < 128) a[r++] = n;else if (4 < (i = h[n])) a[r++] = 65533, t += i - 1;else {
                                                        for (n &= 2 === i ? 31 : 3 === i ? 15 : 7; 1 < i && t < s;) n = n << 6 | 63 & e[t++], i--;

                                                        1 < i ? a[r++] = 65533 : n < 65536 ? a[r++] = n : (n -= 65536, a[r++] = 55296 | n >> 10 & 1023, a[r++] = 56320 | 1023 & n);
                                                      }

                                                      return a.length !== r && (a.subarray ? a = a.subarray(0, r) : a.length = r), o.applyFromCharCode(a);
                                                    }(e = o.transformTo(u.uint8array ? "uint8array" : "array", e));
                                                  }, o.inherits(a, n), a.prototype.processChunk = function (e) {
                                                    var t = o.transformTo(u.uint8array ? "uint8array" : "array", e.data);

                                                    if (this.leftOver && this.leftOver.length) {
                                                      if (u.uint8array) {
                                                        var r = t;
                                                        (t = new Uint8Array(r.length + this.leftOver.length)).set(this.leftOver, 0), t.set(r, this.leftOver.length);
                                                      } else t = this.leftOver.concat(t);

                                                      this.leftOver = null;
                                                    }

                                                    var n = function (e, t) {
                                                      var r;

                                                      for ((t = t || e.length) > e.length && (t = e.length), r = t - 1; 0 <= r && 128 == (192 & e[r]);) r--;

                                                      return r < 0 ? t : 0 === r ? t : r + h[e[r]] > t ? r : t;
                                                    }(t),
                                                        i = t;

                                                    n !== t.length && (u.uint8array ? (i = t.subarray(0, n), this.leftOver = t.subarray(n, t.length)) : (i = t.slice(0, n), this.leftOver = t.slice(n, t.length))), this.push({
                                                      data: s.utf8decode(i),
                                                      meta: e.meta
                                                    });
                                                  }, a.prototype.flush = function () {
                                                    this.leftOver && this.leftOver.length && (this.push({
                                                      data: s.utf8decode(this.leftOver),
                                                      meta: {}
                                                    }), this.leftOver = null);
                                                  }, s.Utf8DecodeWorker = a, o.inherits(f, n), f.prototype.processChunk = function (e) {
                                                    this.push({
                                                      data: s.utf8encode(e.data),
                                                      meta: e.meta
                                                    });
                                                  }, s.Utf8EncodeWorker = f;
                                                }, {
                                                  "./nodejsUtils": 14,
                                                  "./stream/GenericWorker": 28,
                                                  "./support": 30,
                                                  "./utils": 32
                                                }],
                                                32: [function (e, t, o) {

                                                  var u = e("./support"),
                                                      h = e("./base64"),
                                                      r = e("./nodejsUtils"),
                                                      n = e("set-immediate-shim"),
                                                      f = e("./external");

                                                  function i(e) {
                                                    return e;
                                                  }

                                                  function l(e, t) {
                                                    for (var r = 0; r < e.length; ++r) t[r] = 255 & e.charCodeAt(r);

                                                    return t;
                                                  }

                                                  o.newBlob = function (t, r) {
                                                    o.checkSupport("blob");

                                                    try {
                                                      return new Blob([t], {
                                                        type: r
                                                      });
                                                    } catch (e) {
                                                      try {
                                                        var n = new (self.BlobBuilder || self.WebKitBlobBuilder || self.MozBlobBuilder || self.MSBlobBuilder)();
                                                        return n.append(t), n.getBlob(r);
                                                      } catch (e) {
                                                        throw new Error("Bug : can't construct the Blob.");
                                                      }
                                                    }
                                                  };

                                                  var s = {
                                                    stringifyByChunk: function (e, t, r) {
                                                      var n = [],
                                                          i = 0,
                                                          s = e.length;
                                                      if (s <= r) return String.fromCharCode.apply(null, e);

                                                      for (; i < s;) "array" === t || "nodebuffer" === t ? n.push(String.fromCharCode.apply(null, e.slice(i, Math.min(i + r, s)))) : n.push(String.fromCharCode.apply(null, e.subarray(i, Math.min(i + r, s)))), i += r;

                                                      return n.join("");
                                                    },
                                                    stringifyByChar: function (e) {
                                                      for (var t = "", r = 0; r < e.length; r++) t += String.fromCharCode(e[r]);

                                                      return t;
                                                    },
                                                    applyCanBeUsed: {
                                                      uint8array: function () {
                                                        try {
                                                          return u.uint8array && 1 === String.fromCharCode.apply(null, new Uint8Array(1)).length;
                                                        } catch (e) {
                                                          return !1;
                                                        }
                                                      }(),
                                                      nodebuffer: function () {
                                                        try {
                                                          return u.nodebuffer && 1 === String.fromCharCode.apply(null, r.allocBuffer(1)).length;
                                                        } catch (e) {
                                                          return !1;
                                                        }
                                                      }()
                                                    }
                                                  };

                                                  function a(e) {
                                                    var t = 65536,
                                                        r = o.getTypeOf(e),
                                                        n = !0;
                                                    if ("uint8array" === r ? n = s.applyCanBeUsed.uint8array : "nodebuffer" === r && (n = s.applyCanBeUsed.nodebuffer), n) for (; 1 < t;) try {
                                                      return s.stringifyByChunk(e, r, t);
                                                    } catch (e) {
                                                      t = Math.floor(t / 2);
                                                    }
                                                    return s.stringifyByChar(e);
                                                  }

                                                  function d(e, t) {
                                                    for (var r = 0; r < e.length; r++) t[r] = e[r];

                                                    return t;
                                                  }

                                                  o.applyFromCharCode = a;
                                                  var c = {};
                                                  c.string = {
                                                    string: i,
                                                    array: function (e) {
                                                      return l(e, new Array(e.length));
                                                    },
                                                    arraybuffer: function (e) {
                                                      return c.string.uint8array(e).buffer;
                                                    },
                                                    uint8array: function (e) {
                                                      return l(e, new Uint8Array(e.length));
                                                    },
                                                    nodebuffer: function (e) {
                                                      return l(e, r.allocBuffer(e.length));
                                                    }
                                                  }, c.array = {
                                                    string: a,
                                                    array: i,
                                                    arraybuffer: function (e) {
                                                      return new Uint8Array(e).buffer;
                                                    },
                                                    uint8array: function (e) {
                                                      return new Uint8Array(e);
                                                    },
                                                    nodebuffer: function (e) {
                                                      return r.newBufferFrom(e);
                                                    }
                                                  }, c.arraybuffer = {
                                                    string: function (e) {
                                                      return a(new Uint8Array(e));
                                                    },
                                                    array: function (e) {
                                                      return d(new Uint8Array(e), new Array(e.byteLength));
                                                    },
                                                    arraybuffer: i,
                                                    uint8array: function (e) {
                                                      return new Uint8Array(e);
                                                    },
                                                    nodebuffer: function (e) {
                                                      return r.newBufferFrom(new Uint8Array(e));
                                                    }
                                                  }, c.uint8array = {
                                                    string: a,
                                                    array: function (e) {
                                                      return d(e, new Array(e.length));
                                                    },
                                                    arraybuffer: function (e) {
                                                      return e.buffer;
                                                    },
                                                    uint8array: i,
                                                    nodebuffer: function (e) {
                                                      return r.newBufferFrom(e);
                                                    }
                                                  }, c.nodebuffer = {
                                                    string: a,
                                                    array: function (e) {
                                                      return d(e, new Array(e.length));
                                                    },
                                                    arraybuffer: function (e) {
                                                      return c.nodebuffer.uint8array(e).buffer;
                                                    },
                                                    uint8array: function (e) {
                                                      return d(e, new Uint8Array(e.length));
                                                    },
                                                    nodebuffer: i
                                                  }, o.transformTo = function (e, t) {
                                                    if (t = t || "", !e) return t;
                                                    o.checkSupport(e);
                                                    var r = o.getTypeOf(t);
                                                    return c[r][e](t);
                                                  }, o.getTypeOf = function (e) {
                                                    return "string" == typeof e ? "string" : "[object Array]" === Object.prototype.toString.call(e) ? "array" : u.nodebuffer && r.isBuffer(e) ? "nodebuffer" : u.uint8array && e instanceof Uint8Array ? "uint8array" : u.arraybuffer && e instanceof ArrayBuffer ? "arraybuffer" : void 0;
                                                  }, o.checkSupport = function (e) {
                                                    if (!u[e.toLowerCase()]) throw new Error(e + " is not supported by this platform");
                                                  }, o.MAX_VALUE_16BITS = 65535, o.MAX_VALUE_32BITS = -1, o.pretty = function (e) {
                                                    var t,
                                                        r,
                                                        n = "";

                                                    for (r = 0; r < (e || "").length; r++) n += "\\x" + ((t = e.charCodeAt(r)) < 16 ? "0" : "") + t.toString(16).toUpperCase();

                                                    return n;
                                                  }, o.delay = function (e, t, r) {
                                                    n(function () {
                                                      e.apply(r || null, t || []);
                                                    });
                                                  }, o.inherits = function (e, t) {
                                                    function r() {}

                                                    r.prototype = t.prototype, e.prototype = new r();
                                                  }, o.extend = function () {
                                                    var e,
                                                        t,
                                                        r = {};

                                                    for (e = 0; e < arguments.length; e++) for (t in arguments[e]) arguments[e].hasOwnProperty(t) && void 0 === r[t] && (r[t] = arguments[e][t]);

                                                    return r;
                                                  }, o.prepareContent = function (n, e, i, s, a) {
                                                    return f.Promise.resolve(e).then(function (n) {
                                                      return u.blob && (n instanceof Blob || -1 !== ["[object File]", "[object Blob]"].indexOf(Object.prototype.toString.call(n))) && "undefined" != typeof FileReader ? new f.Promise(function (t, r) {
                                                        var e = new FileReader();
                                                        e.onload = function (e) {
                                                          t(e.target.result);
                                                        }, e.onerror = function (e) {
                                                          r(e.target.error);
                                                        }, e.readAsArrayBuffer(n);
                                                      }) : n;
                                                    }).then(function (e) {
                                                      var t,
                                                          r = o.getTypeOf(e);
                                                      return r ? ("arraybuffer" === r ? e = o.transformTo("uint8array", e) : "string" === r && (a ? e = h.decode(e) : i && !0 !== s && (e = l(t = e, u.uint8array ? new Uint8Array(t.length) : new Array(t.length)))), e) : f.Promise.reject(new Error("Can't read the data of '" + n + "'. Is it in a supported JavaScript type (String, Blob, ArrayBuffer, etc) ?"));
                                                    });
                                                  };
                                                }, {
                                                  "./base64": 1,
                                                  "./external": 6,
                                                  "./nodejsUtils": 14,
                                                  "./support": 30,
                                                  "set-immediate-shim": 54
                                                }],
                                                33: [function (e, t, r) {

                                                  var n = e("./reader/readerFor"),
                                                      i = e("./utils"),
                                                      s = e("./signature"),
                                                      a = e("./zipEntry"),
                                                      o = (e("./utf8"), e("./support"));

                                                  function u(e) {
                                                    this.files = [], this.loadOptions = e;
                                                  }

                                                  u.prototype = {
                                                    checkSignature: function (e) {
                                                      if (!this.reader.readAndCheckSignature(e)) {
                                                        this.reader.index -= 4;
                                                        var t = this.reader.readString(4);
                                                        throw new Error("Corrupted zip or bug: unexpected signature (" + i.pretty(t) + ", expected " + i.pretty(e) + ")");
                                                      }
                                                    },
                                                    isSignature: function (e, t) {
                                                      var r = this.reader.index;
                                                      this.reader.setIndex(e);
                                                      var n = this.reader.readString(4) === t;
                                                      return this.reader.setIndex(r), n;
                                                    },
                                                    readBlockEndOfCentral: function () {
                                                      this.diskNumber = this.reader.readInt(2), this.diskWithCentralDirStart = this.reader.readInt(2), this.centralDirRecordsOnThisDisk = this.reader.readInt(2), this.centralDirRecords = this.reader.readInt(2), this.centralDirSize = this.reader.readInt(4), this.centralDirOffset = this.reader.readInt(4), this.zipCommentLength = this.reader.readInt(2);
                                                      var e = this.reader.readData(this.zipCommentLength),
                                                          t = o.uint8array ? "uint8array" : "array",
                                                          r = i.transformTo(t, e);
                                                      this.zipComment = this.loadOptions.decodeFileName(r);
                                                    },
                                                    readBlockZip64EndOfCentral: function () {
                                                      this.zip64EndOfCentralSize = this.reader.readInt(8), this.reader.skip(4), this.diskNumber = this.reader.readInt(4), this.diskWithCentralDirStart = this.reader.readInt(4), this.centralDirRecordsOnThisDisk = this.reader.readInt(8), this.centralDirRecords = this.reader.readInt(8), this.centralDirSize = this.reader.readInt(8), this.centralDirOffset = this.reader.readInt(8), this.zip64ExtensibleData = {};

                                                      for (var e, t, r, n = this.zip64EndOfCentralSize - 44; 0 < n;) e = this.reader.readInt(2), t = this.reader.readInt(4), r = this.reader.readData(t), this.zip64ExtensibleData[e] = {
                                                        id: e,
                                                        length: t,
                                                        value: r
                                                      };
                                                    },
                                                    readBlockZip64EndOfCentralLocator: function () {
                                                      if (this.diskWithZip64CentralDirStart = this.reader.readInt(4), this.relativeOffsetEndOfZip64CentralDir = this.reader.readInt(8), this.disksCount = this.reader.readInt(4), 1 < this.disksCount) throw new Error("Multi-volumes zip are not supported");
                                                    },
                                                    readLocalFiles: function () {
                                                      var e, t;

                                                      for (e = 0; e < this.files.length; e++) t = this.files[e], this.reader.setIndex(t.localHeaderOffset), this.checkSignature(s.LOCAL_FILE_HEADER), t.readLocalPart(this.reader), t.handleUTF8(), t.processAttributes();
                                                    },
                                                    readCentralDir: function () {
                                                      var e;

                                                      for (this.reader.setIndex(this.centralDirOffset); this.reader.readAndCheckSignature(s.CENTRAL_FILE_HEADER);) (e = new a({
                                                        zip64: this.zip64
                                                      }, this.loadOptions)).readCentralPart(this.reader), this.files.push(e);

                                                      if (this.centralDirRecords !== this.files.length && 0 !== this.centralDirRecords && 0 === this.files.length) throw new Error("Corrupted zip or bug: expected " + this.centralDirRecords + " records in central dir, got " + this.files.length);
                                                    },
                                                    readEndOfCentral: function () {
                                                      var e = this.reader.lastIndexOfSignature(s.CENTRAL_DIRECTORY_END);
                                                      if (e < 0) throw this.isSignature(0, s.LOCAL_FILE_HEADER) ? new Error("Corrupted zip: can't find end of central directory") : new Error("Can't find end of central directory : is this a zip file ? If it is, see https://stuk.github.io/jszip/documentation/howto/read_zip.html");
                                                      this.reader.setIndex(e);
                                                      var t = e;

                                                      if (this.checkSignature(s.CENTRAL_DIRECTORY_END), this.readBlockEndOfCentral(), this.diskNumber === i.MAX_VALUE_16BITS || this.diskWithCentralDirStart === i.MAX_VALUE_16BITS || this.centralDirRecordsOnThisDisk === i.MAX_VALUE_16BITS || this.centralDirRecords === i.MAX_VALUE_16BITS || this.centralDirSize === i.MAX_VALUE_32BITS || this.centralDirOffset === i.MAX_VALUE_32BITS) {
                                                        if (this.zip64 = !0, (e = this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR)) < 0) throw new Error("Corrupted zip: can't find the ZIP64 end of central directory locator");
                                                        if (this.reader.setIndex(e), this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR), this.readBlockZip64EndOfCentralLocator(), !this.isSignature(this.relativeOffsetEndOfZip64CentralDir, s.ZIP64_CENTRAL_DIRECTORY_END) && (this.relativeOffsetEndOfZip64CentralDir = this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_END), this.relativeOffsetEndOfZip64CentralDir < 0)) throw new Error("Corrupted zip: can't find the ZIP64 end of central directory");
                                                        this.reader.setIndex(this.relativeOffsetEndOfZip64CentralDir), this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_END), this.readBlockZip64EndOfCentral();
                                                      }

                                                      var r = this.centralDirOffset + this.centralDirSize;
                                                      this.zip64 && (r += 20, r += 12 + this.zip64EndOfCentralSize);
                                                      var n = t - r;
                                                      if (0 < n) this.isSignature(t, s.CENTRAL_FILE_HEADER) || (this.reader.zero = n);else if (n < 0) throw new Error("Corrupted zip: missing " + Math.abs(n) + " bytes.");
                                                    },
                                                    prepareReader: function (e) {
                                                      this.reader = n(e);
                                                    },
                                                    load: function (e) {
                                                      this.prepareReader(e), this.readEndOfCentral(), this.readCentralDir(), this.readLocalFiles();
                                                    }
                                                  }, t.exports = u;
                                                }, {
                                                  "./reader/readerFor": 22,
                                                  "./signature": 23,
                                                  "./support": 30,
                                                  "./utf8": 31,
                                                  "./utils": 32,
                                                  "./zipEntry": 34
                                                }],
                                                34: [function (e, t, r) {

                                                  var n = e("./reader/readerFor"),
                                                      s = e("./utils"),
                                                      i = e("./compressedObject"),
                                                      a = e("./crc32"),
                                                      o = e("./utf8"),
                                                      u = e("./compressions"),
                                                      h = e("./support");

                                                  function f(e, t) {
                                                    this.options = e, this.loadOptions = t;
                                                  }

                                                  f.prototype = {
                                                    isEncrypted: function () {
                                                      return 1 == (1 & this.bitFlag);
                                                    },
                                                    useUTF8: function () {
                                                      return 2048 == (2048 & this.bitFlag);
                                                    },
                                                    readLocalPart: function (e) {
                                                      var t, r;
                                                      if (e.skip(22), this.fileNameLength = e.readInt(2), r = e.readInt(2), this.fileName = e.readData(this.fileNameLength), e.skip(r), -1 === this.compressedSize || -1 === this.uncompressedSize) throw new Error("Bug or corrupted zip : didn't get enough information from the central directory (compressedSize === -1 || uncompressedSize === -1)");
                                                      if (null === (t = function (e) {
                                                        for (var t in u) if (u.hasOwnProperty(t) && u[t].magic === e) return u[t];

                                                        return null;
                                                      }(this.compressionMethod))) throw new Error("Corrupted zip : compression " + s.pretty(this.compressionMethod) + " unknown (inner file : " + s.transformTo("string", this.fileName) + ")");
                                                      this.decompressed = new i(this.compressedSize, this.uncompressedSize, this.crc32, t, e.readData(this.compressedSize));
                                                    },
                                                    readCentralPart: function (e) {
                                                      this.versionMadeBy = e.readInt(2), e.skip(2), this.bitFlag = e.readInt(2), this.compressionMethod = e.readString(2), this.date = e.readDate(), this.crc32 = e.readInt(4), this.compressedSize = e.readInt(4), this.uncompressedSize = e.readInt(4);
                                                      var t = e.readInt(2);
                                                      if (this.extraFieldsLength = e.readInt(2), this.fileCommentLength = e.readInt(2), this.diskNumberStart = e.readInt(2), this.internalFileAttributes = e.readInt(2), this.externalFileAttributes = e.readInt(4), this.localHeaderOffset = e.readInt(4), this.isEncrypted()) throw new Error("Encrypted zip are not supported");
                                                      e.skip(t), this.readExtraFields(e), this.parseZIP64ExtraField(e), this.fileComment = e.readData(this.fileCommentLength);
                                                    },
                                                    processAttributes: function () {
                                                      this.unixPermissions = null, this.dosPermissions = null;
                                                      var e = this.versionMadeBy >> 8;
                                                      this.dir = !!(16 & this.externalFileAttributes), 0 == e && (this.dosPermissions = 63 & this.externalFileAttributes), 3 == e && (this.unixPermissions = this.externalFileAttributes >> 16 & 65535), this.dir || "/" !== this.fileNameStr.slice(-1) || (this.dir = !0);
                                                    },
                                                    parseZIP64ExtraField: function (e) {
                                                      if (this.extraFields[1]) {
                                                        var t = n(this.extraFields[1].value);
                                                        this.uncompressedSize === s.MAX_VALUE_32BITS && (this.uncompressedSize = t.readInt(8)), this.compressedSize === s.MAX_VALUE_32BITS && (this.compressedSize = t.readInt(8)), this.localHeaderOffset === s.MAX_VALUE_32BITS && (this.localHeaderOffset = t.readInt(8)), this.diskNumberStart === s.MAX_VALUE_32BITS && (this.diskNumberStart = t.readInt(4));
                                                      }
                                                    },
                                                    readExtraFields: function (e) {
                                                      var t,
                                                          r,
                                                          n,
                                                          i = e.index + this.extraFieldsLength;

                                                      for (this.extraFields || (this.extraFields = {}); e.index + 4 < i;) t = e.readInt(2), r = e.readInt(2), n = e.readData(r), this.extraFields[t] = {
                                                        id: t,
                                                        length: r,
                                                        value: n
                                                      };

                                                      e.setIndex(i);
                                                    },
                                                    handleUTF8: function () {
                                                      var e = h.uint8array ? "uint8array" : "array";
                                                      if (this.useUTF8()) this.fileNameStr = o.utf8decode(this.fileName), this.fileCommentStr = o.utf8decode(this.fileComment);else {
                                                        var t = this.findExtraFieldUnicodePath();
                                                        if (null !== t) this.fileNameStr = t;else {
                                                          var r = s.transformTo(e, this.fileName);
                                                          this.fileNameStr = this.loadOptions.decodeFileName(r);
                                                        }
                                                        var n = this.findExtraFieldUnicodeComment();
                                                        if (null !== n) this.fileCommentStr = n;else {
                                                          var i = s.transformTo(e, this.fileComment);
                                                          this.fileCommentStr = this.loadOptions.decodeFileName(i);
                                                        }
                                                      }
                                                    },
                                                    findExtraFieldUnicodePath: function () {
                                                      var e = this.extraFields[28789];

                                                      if (e) {
                                                        var t = n(e.value);
                                                        return 1 !== t.readInt(1) ? null : a(this.fileName) !== t.readInt(4) ? null : o.utf8decode(t.readData(e.length - 5));
                                                      }

                                                      return null;
                                                    },
                                                    findExtraFieldUnicodeComment: function () {
                                                      var e = this.extraFields[25461];

                                                      if (e) {
                                                        var t = n(e.value);
                                                        return 1 !== t.readInt(1) ? null : a(this.fileComment) !== t.readInt(4) ? null : o.utf8decode(t.readData(e.length - 5));
                                                      }

                                                      return null;
                                                    }
                                                  }, t.exports = f;
                                                }, {
                                                  "./compressedObject": 2,
                                                  "./compressions": 3,
                                                  "./crc32": 4,
                                                  "./reader/readerFor": 22,
                                                  "./support": 30,
                                                  "./utf8": 31,
                                                  "./utils": 32
                                                }],
                                                35: [function (e, t, r) {

                                                  function n(e, t, r) {
                                                    this.name = e, this.dir = r.dir, this.date = r.date, this.comment = r.comment, this.unixPermissions = r.unixPermissions, this.dosPermissions = r.dosPermissions, this._data = t, this._dataBinary = r.binary, this.options = {
                                                      compression: r.compression,
                                                      compressionOptions: r.compressionOptions
                                                    };
                                                  }

                                                  var s = e("./stream/StreamHelper"),
                                                      i = e("./stream/DataWorker"),
                                                      a = e("./utf8"),
                                                      o = e("./compressedObject"),
                                                      u = e("./stream/GenericWorker");
                                                  n.prototype = {
                                                    internalStream: function (e) {
                                                      var t = null,
                                                          r = "string";

                                                      try {
                                                        if (!e) throw new Error("No output type specified.");
                                                        var n = "string" === (r = e.toLowerCase()) || "text" === r;
                                                        "binarystring" !== r && "text" !== r || (r = "string"), t = this._decompressWorker();
                                                        var i = !this._dataBinary;
                                                        i && !n && (t = t.pipe(new a.Utf8EncodeWorker())), !i && n && (t = t.pipe(new a.Utf8DecodeWorker()));
                                                      } catch (e) {
                                                        (t = new u("error")).error(e);
                                                      }

                                                      return new s(t, r, "");
                                                    },
                                                    async: function (e, t) {
                                                      return this.internalStream(e).accumulate(t);
                                                    },
                                                    nodeStream: function (e, t) {
                                                      return this.internalStream(e || "nodebuffer").toNodejsStream(t);
                                                    },
                                                    _compressWorker: function (e, t) {
                                                      if (this._data instanceof o && this._data.compression.magic === e.magic) return this._data.getCompressedWorker();

                                                      var r = this._decompressWorker();

                                                      return this._dataBinary || (r = r.pipe(new a.Utf8EncodeWorker())), o.createWorkerFrom(r, e, t);
                                                    },
                                                    _decompressWorker: function () {
                                                      return this._data instanceof o ? this._data.getContentWorker() : this._data instanceof u ? this._data : new i(this._data);
                                                    }
                                                  };

                                                  for (var h = ["asText", "asBinary", "asNodeBuffer", "asUint8Array", "asArrayBuffer"], f = function () {
                                                    throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.");
                                                  }, l = 0; l < h.length; l++) n.prototype[h[l]] = f;

                                                  t.exports = n;
                                                }, {
                                                  "./compressedObject": 2,
                                                  "./stream/DataWorker": 27,
                                                  "./stream/GenericWorker": 28,
                                                  "./stream/StreamHelper": 29,
                                                  "./utf8": 31
                                                }],
                                                36: [function (e, f, t) {
                                                  (function (t) {

                                                    var r,
                                                        n,
                                                        e = t.MutationObserver || t.WebKitMutationObserver;

                                                    if (e) {
                                                      var i = 0,
                                                          s = new e(h),
                                                          a = t.document.createTextNode("");
                                                      s.observe(a, {
                                                        characterData: !0
                                                      }), r = function () {
                                                        a.data = i = ++i % 2;
                                                      };
                                                    } else if (t.setImmediate || void 0 === t.MessageChannel) r = "document" in t && "onreadystatechange" in t.document.createElement("script") ? function () {
                                                      var e = t.document.createElement("script");
                                                      e.onreadystatechange = function () {
                                                        h(), e.onreadystatechange = null, e.parentNode.removeChild(e), e = null;
                                                      }, t.document.documentElement.appendChild(e);
                                                    } : function () {
                                                      setTimeout(h, 0);
                                                    };else {
                                                      var o = new t.MessageChannel();
                                                      o.port1.onmessage = h, r = function () {
                                                        o.port2.postMessage(0);
                                                      };
                                                    }

                                                    var u = [];

                                                    function h() {
                                                      var e, t;
                                                      n = !0;

                                                      for (var r = u.length; r;) {
                                                        for (t = u, u = [], e = -1; ++e < r;) t[e]();

                                                        r = u.length;
                                                      }

                                                      n = !1;
                                                    }

                                                    f.exports = function (e) {
                                                      1 !== u.push(e) || n || r();
                                                    };
                                                  }).call(this, void 0 !== r ? r : "undefined" != typeof self ? self : "undefined" != typeof window ? window : {});
                                                }, {}],
                                                37: [function (e, t, r) {

                                                  var i = e("immediate");

                                                  function h() {}

                                                  var f = {},
                                                      s = ["REJECTED"],
                                                      a = ["FULFILLED"],
                                                      n = ["PENDING"];

                                                  function o(e) {
                                                    if ("function" != typeof e) throw new TypeError("resolver must be a function");
                                                    this.state = n, this.queue = [], this.outcome = void 0, e !== h && c(this, e);
                                                  }

                                                  function u(e, t, r) {
                                                    this.promise = e, "function" == typeof t && (this.onFulfilled = t, this.callFulfilled = this.otherCallFulfilled), "function" == typeof r && (this.onRejected = r, this.callRejected = this.otherCallRejected);
                                                  }

                                                  function l(t, r, n) {
                                                    i(function () {
                                                      var e;

                                                      try {
                                                        e = r(n);
                                                      } catch (e) {
                                                        return f.reject(t, e);
                                                      }

                                                      e === t ? f.reject(t, new TypeError("Cannot resolve promise with itself")) : f.resolve(t, e);
                                                    });
                                                  }

                                                  function d(e) {
                                                    var t = e && e.then;
                                                    if (e && ("object" == typeof e || "function" == typeof e) && "function" == typeof t) return function () {
                                                      t.apply(e, arguments);
                                                    };
                                                  }

                                                  function c(t, e) {
                                                    var r = !1;

                                                    function n(e) {
                                                      r || (r = !0, f.reject(t, e));
                                                    }

                                                    function i(e) {
                                                      r || (r = !0, f.resolve(t, e));
                                                    }

                                                    var s = p(function () {
                                                      e(i, n);
                                                    });
                                                    "error" === s.status && n(s.value);
                                                  }

                                                  function p(e, t) {
                                                    var r = {};

                                                    try {
                                                      r.value = e(t), r.status = "success";
                                                    } catch (e) {
                                                      r.status = "error", r.value = e;
                                                    }

                                                    return r;
                                                  }

                                                  (t.exports = o).prototype.finally = function (t) {
                                                    if ("function" != typeof t) return this;
                                                    var r = this.constructor;
                                                    return this.then(function (e) {
                                                      return r.resolve(t()).then(function () {
                                                        return e;
                                                      });
                                                    }, function (e) {
                                                      return r.resolve(t()).then(function () {
                                                        throw e;
                                                      });
                                                    });
                                                  }, o.prototype.catch = function (e) {
                                                    return this.then(null, e);
                                                  }, o.prototype.then = function (e, t) {
                                                    if ("function" != typeof e && this.state === a || "function" != typeof t && this.state === s) return this;
                                                    var r = new this.constructor(h);
                                                    return this.state !== n ? l(r, this.state === a ? e : t, this.outcome) : this.queue.push(new u(r, e, t)), r;
                                                  }, u.prototype.callFulfilled = function (e) {
                                                    f.resolve(this.promise, e);
                                                  }, u.prototype.otherCallFulfilled = function (e) {
                                                    l(this.promise, this.onFulfilled, e);
                                                  }, u.prototype.callRejected = function (e) {
                                                    f.reject(this.promise, e);
                                                  }, u.prototype.otherCallRejected = function (e) {
                                                    l(this.promise, this.onRejected, e);
                                                  }, f.resolve = function (e, t) {
                                                    var r = p(d, t);
                                                    if ("error" === r.status) return f.reject(e, r.value);
                                                    var n = r.value;
                                                    if (n) c(e, n);else {
                                                      e.state = a, e.outcome = t;

                                                      for (var i = -1, s = e.queue.length; ++i < s;) e.queue[i].callFulfilled(t);
                                                    }
                                                    return e;
                                                  }, f.reject = function (e, t) {
                                                    e.state = s, e.outcome = t;

                                                    for (var r = -1, n = e.queue.length; ++r < n;) e.queue[r].callRejected(t);

                                                    return e;
                                                  }, o.resolve = function (e) {
                                                    return e instanceof this ? e : f.resolve(new this(h), e);
                                                  }, o.reject = function (e) {
                                                    var t = new this(h);
                                                    return f.reject(t, e);
                                                  }, o.all = function (e) {
                                                    var r = this;
                                                    if ("[object Array]" !== Object.prototype.toString.call(e)) return this.reject(new TypeError("must be an array"));
                                                    var n = e.length,
                                                        i = !1;
                                                    if (!n) return this.resolve([]);

                                                    for (var s = new Array(n), a = 0, t = -1, o = new this(h); ++t < n;) u(e[t], t);

                                                    return o;

                                                    function u(e, t) {
                                                      r.resolve(e).then(function (e) {
                                                        s[t] = e, ++a !== n || i || (i = !0, f.resolve(o, s));
                                                      }, function (e) {
                                                        i || (i = !0, f.reject(o, e));
                                                      });
                                                    }
                                                  }, o.race = function (e) {
                                                    if ("[object Array]" !== Object.prototype.toString.call(e)) return this.reject(new TypeError("must be an array"));
                                                    var t = e.length,
                                                        r = !1;
                                                    if (!t) return this.resolve([]);

                                                    for (var n, i = -1, s = new this(h); ++i < t;) n = e[i], this.resolve(n).then(function (e) {
                                                      r || (r = !0, f.resolve(s, e));
                                                    }, function (e) {
                                                      r || (r = !0, f.reject(s, e));
                                                    });

                                                    return s;
                                                  };
                                                }, {
                                                  immediate: 36
                                                }],
                                                38: [function (e, t, r) {

                                                  var n = {};
                                                  (0, e("./lib/utils/common").assign)(n, e("./lib/deflate"), e("./lib/inflate"), e("./lib/zlib/constants")), t.exports = n;
                                                }, {
                                                  "./lib/deflate": 39,
                                                  "./lib/inflate": 40,
                                                  "./lib/utils/common": 41,
                                                  "./lib/zlib/constants": 44
                                                }],
                                                39: [function (e, t, r) {

                                                  var a = e("./zlib/deflate"),
                                                      o = e("./utils/common"),
                                                      u = e("./utils/strings"),
                                                      i = e("./zlib/messages"),
                                                      s = e("./zlib/zstream"),
                                                      h = Object.prototype.toString,
                                                      f = 0,
                                                      l = -1,
                                                      d = 0,
                                                      c = 8;

                                                  function p(e) {
                                                    if (!(this instanceof p)) return new p(e);
                                                    this.options = o.assign({
                                                      level: l,
                                                      method: c,
                                                      chunkSize: 16384,
                                                      windowBits: 15,
                                                      memLevel: 8,
                                                      strategy: d,
                                                      to: ""
                                                    }, e || {});
                                                    var t = this.options;
                                                    t.raw && 0 < t.windowBits ? t.windowBits = -t.windowBits : t.gzip && 0 < t.windowBits && t.windowBits < 16 && (t.windowBits += 16), this.err = 0, this.msg = "", this.ended = !1, this.chunks = [], this.strm = new s(), this.strm.avail_out = 0;
                                                    var r = a.deflateInit2(this.strm, t.level, t.method, t.windowBits, t.memLevel, t.strategy);
                                                    if (r !== f) throw new Error(i[r]);

                                                    if (t.header && a.deflateSetHeader(this.strm, t.header), t.dictionary) {
                                                      var n;
                                                      if (n = "string" == typeof t.dictionary ? u.string2buf(t.dictionary) : "[object ArrayBuffer]" === h.call(t.dictionary) ? new Uint8Array(t.dictionary) : t.dictionary, (r = a.deflateSetDictionary(this.strm, n)) !== f) throw new Error(i[r]);
                                                      this._dict_set = !0;
                                                    }
                                                  }

                                                  function n(e, t) {
                                                    var r = new p(t);
                                                    if (r.push(e, !0), r.err) throw r.msg || i[r.err];
                                                    return r.result;
                                                  }

                                                  p.prototype.push = function (e, t) {
                                                    var r,
                                                        n,
                                                        i = this.strm,
                                                        s = this.options.chunkSize;
                                                    if (this.ended) return !1;
                                                    n = t === ~~t ? t : !0 === t ? 4 : 0, "string" == typeof e ? i.input = u.string2buf(e) : "[object ArrayBuffer]" === h.call(e) ? i.input = new Uint8Array(e) : i.input = e, i.next_in = 0, i.avail_in = i.input.length;

                                                    do {
                                                      if (0 === i.avail_out && (i.output = new o.Buf8(s), i.next_out = 0, i.avail_out = s), 1 !== (r = a.deflate(i, n)) && r !== f) return this.onEnd(r), !(this.ended = !0);
                                                      0 !== i.avail_out && (0 !== i.avail_in || 4 !== n && 2 !== n) || ("string" === this.options.to ? this.onData(u.buf2binstring(o.shrinkBuf(i.output, i.next_out))) : this.onData(o.shrinkBuf(i.output, i.next_out)));
                                                    } while ((0 < i.avail_in || 0 === i.avail_out) && 1 !== r);

                                                    return 4 === n ? (r = a.deflateEnd(this.strm), this.onEnd(r), this.ended = !0, r === f) : 2 !== n || (this.onEnd(f), !(i.avail_out = 0));
                                                  }, p.prototype.onData = function (e) {
                                                    this.chunks.push(e);
                                                  }, p.prototype.onEnd = function (e) {
                                                    e === f && ("string" === this.options.to ? this.result = this.chunks.join("") : this.result = o.flattenChunks(this.chunks)), this.chunks = [], this.err = e, this.msg = this.strm.msg;
                                                  }, r.Deflate = p, r.deflate = n, r.deflateRaw = function (e, t) {
                                                    return (t = t || {}).raw = !0, n(e, t);
                                                  }, r.gzip = function (e, t) {
                                                    return (t = t || {}).gzip = !0, n(e, t);
                                                  };
                                                }, {
                                                  "./utils/common": 41,
                                                  "./utils/strings": 42,
                                                  "./zlib/deflate": 46,
                                                  "./zlib/messages": 51,
                                                  "./zlib/zstream": 53
                                                }],
                                                40: [function (e, t, r) {

                                                  var d = e("./zlib/inflate"),
                                                      c = e("./utils/common"),
                                                      p = e("./utils/strings"),
                                                      m = e("./zlib/constants"),
                                                      n = e("./zlib/messages"),
                                                      i = e("./zlib/zstream"),
                                                      s = e("./zlib/gzheader"),
                                                      _ = Object.prototype.toString;

                                                  function a(e) {
                                                    if (!(this instanceof a)) return new a(e);
                                                    this.options = c.assign({
                                                      chunkSize: 16384,
                                                      windowBits: 0,
                                                      to: ""
                                                    }, e || {});
                                                    var t = this.options;
                                                    t.raw && 0 <= t.windowBits && t.windowBits < 16 && (t.windowBits = -t.windowBits, 0 === t.windowBits && (t.windowBits = -15)), !(0 <= t.windowBits && t.windowBits < 16) || e && e.windowBits || (t.windowBits += 32), 15 < t.windowBits && t.windowBits < 48 && 0 == (15 & t.windowBits) && (t.windowBits |= 15), this.err = 0, this.msg = "", this.ended = !1, this.chunks = [], this.strm = new i(), this.strm.avail_out = 0;
                                                    var r = d.inflateInit2(this.strm, t.windowBits);
                                                    if (r !== m.Z_OK) throw new Error(n[r]);
                                                    this.header = new s(), d.inflateGetHeader(this.strm, this.header);
                                                  }

                                                  function o(e, t) {
                                                    var r = new a(t);
                                                    if (r.push(e, !0), r.err) throw r.msg || n[r.err];
                                                    return r.result;
                                                  }

                                                  a.prototype.push = function (e, t) {
                                                    var r,
                                                        n,
                                                        i,
                                                        s,
                                                        a,
                                                        o,
                                                        u = this.strm,
                                                        h = this.options.chunkSize,
                                                        f = this.options.dictionary,
                                                        l = !1;
                                                    if (this.ended) return !1;
                                                    n = t === ~~t ? t : !0 === t ? m.Z_FINISH : m.Z_NO_FLUSH, "string" == typeof e ? u.input = p.binstring2buf(e) : "[object ArrayBuffer]" === _.call(e) ? u.input = new Uint8Array(e) : u.input = e, u.next_in = 0, u.avail_in = u.input.length;

                                                    do {
                                                      if (0 === u.avail_out && (u.output = new c.Buf8(h), u.next_out = 0, u.avail_out = h), (r = d.inflate(u, m.Z_NO_FLUSH)) === m.Z_NEED_DICT && f && (o = "string" == typeof f ? p.string2buf(f) : "[object ArrayBuffer]" === _.call(f) ? new Uint8Array(f) : f, r = d.inflateSetDictionary(this.strm, o)), r === m.Z_BUF_ERROR && !0 === l && (r = m.Z_OK, l = !1), r !== m.Z_STREAM_END && r !== m.Z_OK) return this.onEnd(r), !(this.ended = !0);
                                                      u.next_out && (0 !== u.avail_out && r !== m.Z_STREAM_END && (0 !== u.avail_in || n !== m.Z_FINISH && n !== m.Z_SYNC_FLUSH) || ("string" === this.options.to ? (i = p.utf8border(u.output, u.next_out), s = u.next_out - i, a = p.buf2string(u.output, i), u.next_out = s, u.avail_out = h - s, s && c.arraySet(u.output, u.output, i, s, 0), this.onData(a)) : this.onData(c.shrinkBuf(u.output, u.next_out)))), 0 === u.avail_in && 0 === u.avail_out && (l = !0);
                                                    } while ((0 < u.avail_in || 0 === u.avail_out) && r !== m.Z_STREAM_END);

                                                    return r === m.Z_STREAM_END && (n = m.Z_FINISH), n === m.Z_FINISH ? (r = d.inflateEnd(this.strm), this.onEnd(r), this.ended = !0, r === m.Z_OK) : n !== m.Z_SYNC_FLUSH || (this.onEnd(m.Z_OK), !(u.avail_out = 0));
                                                  }, a.prototype.onData = function (e) {
                                                    this.chunks.push(e);
                                                  }, a.prototype.onEnd = function (e) {
                                                    e === m.Z_OK && ("string" === this.options.to ? this.result = this.chunks.join("") : this.result = c.flattenChunks(this.chunks)), this.chunks = [], this.err = e, this.msg = this.strm.msg;
                                                  }, r.Inflate = a, r.inflate = o, r.inflateRaw = function (e, t) {
                                                    return (t = t || {}).raw = !0, o(e, t);
                                                  }, r.ungzip = o;
                                                }, {
                                                  "./utils/common": 41,
                                                  "./utils/strings": 42,
                                                  "./zlib/constants": 44,
                                                  "./zlib/gzheader": 47,
                                                  "./zlib/inflate": 49,
                                                  "./zlib/messages": 51,
                                                  "./zlib/zstream": 53
                                                }],
                                                41: [function (e, t, r) {

                                                  var n = "undefined" != typeof Uint8Array && "undefined" != typeof Uint16Array && "undefined" != typeof Int32Array;
                                                  r.assign = function (e) {
                                                    for (var t = Array.prototype.slice.call(arguments, 1); t.length;) {
                                                      var r = t.shift();

                                                      if (r) {
                                                        if ("object" != typeof r) throw new TypeError(r + "must be non-object");

                                                        for (var n in r) r.hasOwnProperty(n) && (e[n] = r[n]);
                                                      }
                                                    }

                                                    return e;
                                                  }, r.shrinkBuf = function (e, t) {
                                                    return e.length === t ? e : e.subarray ? e.subarray(0, t) : (e.length = t, e);
                                                  };
                                                  var i = {
                                                    arraySet: function (e, t, r, n, i) {
                                                      if (t.subarray && e.subarray) e.set(t.subarray(r, r + n), i);else for (var s = 0; s < n; s++) e[i + s] = t[r + s];
                                                    },
                                                    flattenChunks: function (e) {
                                                      var t, r, n, i, s, a;

                                                      for (t = n = 0, r = e.length; t < r; t++) n += e[t].length;

                                                      for (a = new Uint8Array(n), t = i = 0, r = e.length; t < r; t++) s = e[t], a.set(s, i), i += s.length;

                                                      return a;
                                                    }
                                                  },
                                                      s = {
                                                    arraySet: function (e, t, r, n, i) {
                                                      for (var s = 0; s < n; s++) e[i + s] = t[r + s];
                                                    },
                                                    flattenChunks: function (e) {
                                                      return [].concat.apply([], e);
                                                    }
                                                  };
                                                  r.setTyped = function (e) {
                                                    e ? (r.Buf8 = Uint8Array, r.Buf16 = Uint16Array, r.Buf32 = Int32Array, r.assign(r, i)) : (r.Buf8 = Array, r.Buf16 = Array, r.Buf32 = Array, r.assign(r, s));
                                                  }, r.setTyped(n);
                                                }, {}],
                                                42: [function (e, t, r) {

                                                  var u = e("./common"),
                                                      i = !0,
                                                      s = !0;

                                                  try {
                                                    String.fromCharCode.apply(null, [0]);
                                                  } catch (e) {
                                                    i = !1;
                                                  }

                                                  try {
                                                    String.fromCharCode.apply(null, new Uint8Array(1));
                                                  } catch (e) {
                                                    s = !1;
                                                  }

                                                  for (var h = new u.Buf8(256), n = 0; n < 256; n++) h[n] = 252 <= n ? 6 : 248 <= n ? 5 : 240 <= n ? 4 : 224 <= n ? 3 : 192 <= n ? 2 : 1;

                                                  function f(e, t) {
                                                    if (t < 65537 && (e.subarray && s || !e.subarray && i)) return String.fromCharCode.apply(null, u.shrinkBuf(e, t));

                                                    for (var r = "", n = 0; n < t; n++) r += String.fromCharCode(e[n]);

                                                    return r;
                                                  }

                                                  h[254] = h[254] = 1, r.string2buf = function (e) {
                                                    var t,
                                                        r,
                                                        n,
                                                        i,
                                                        s,
                                                        a = e.length,
                                                        o = 0;

                                                    for (i = 0; i < a; i++) 55296 == (64512 & (r = e.charCodeAt(i))) && i + 1 < a && 56320 == (64512 & (n = e.charCodeAt(i + 1))) && (r = 65536 + (r - 55296 << 10) + (n - 56320), i++), o += r < 128 ? 1 : r < 2048 ? 2 : r < 65536 ? 3 : 4;

                                                    for (t = new u.Buf8(o), i = s = 0; s < o; i++) 55296 == (64512 & (r = e.charCodeAt(i))) && i + 1 < a && 56320 == (64512 & (n = e.charCodeAt(i + 1))) && (r = 65536 + (r - 55296 << 10) + (n - 56320), i++), r < 128 ? t[s++] = r : (r < 2048 ? t[s++] = 192 | r >>> 6 : (r < 65536 ? t[s++] = 224 | r >>> 12 : (t[s++] = 240 | r >>> 18, t[s++] = 128 | r >>> 12 & 63), t[s++] = 128 | r >>> 6 & 63), t[s++] = 128 | 63 & r);

                                                    return t;
                                                  }, r.buf2binstring = function (e) {
                                                    return f(e, e.length);
                                                  }, r.binstring2buf = function (e) {
                                                    for (var t = new u.Buf8(e.length), r = 0, n = t.length; r < n; r++) t[r] = e.charCodeAt(r);

                                                    return t;
                                                  }, r.buf2string = function (e, t) {
                                                    var r,
                                                        n,
                                                        i,
                                                        s,
                                                        a = t || e.length,
                                                        o = new Array(2 * a);

                                                    for (r = n = 0; r < a;) if ((i = e[r++]) < 128) o[n++] = i;else if (4 < (s = h[i])) o[n++] = 65533, r += s - 1;else {
                                                      for (i &= 2 === s ? 31 : 3 === s ? 15 : 7; 1 < s && r < a;) i = i << 6 | 63 & e[r++], s--;

                                                      1 < s ? o[n++] = 65533 : i < 65536 ? o[n++] = i : (i -= 65536, o[n++] = 55296 | i >> 10 & 1023, o[n++] = 56320 | 1023 & i);
                                                    }

                                                    return f(o, n);
                                                  }, r.utf8border = function (e, t) {
                                                    var r;

                                                    for ((t = t || e.length) > e.length && (t = e.length), r = t - 1; 0 <= r && 128 == (192 & e[r]);) r--;

                                                    return r < 0 ? t : 0 === r ? t : r + h[e[r]] > t ? r : t;
                                                  };
                                                }, {
                                                  "./common": 41
                                                }],
                                                43: [function (e, t, r) {

                                                  t.exports = function (e, t, r, n) {
                                                    for (var i = 65535 & e | 0, s = e >>> 16 & 65535 | 0, a = 0; 0 !== r;) {
                                                      for (r -= a = 2e3 < r ? 2e3 : r; s = s + (i = i + t[n++] | 0) | 0, --a;);

                                                      i %= 65521, s %= 65521;
                                                    }

                                                    return i | s << 16 | 0;
                                                  };
                                                }, {}],
                                                44: [function (e, t, r) {

                                                  t.exports = {
                                                    Z_NO_FLUSH: 0,
                                                    Z_PARTIAL_FLUSH: 1,
                                                    Z_SYNC_FLUSH: 2,
                                                    Z_FULL_FLUSH: 3,
                                                    Z_FINISH: 4,
                                                    Z_BLOCK: 5,
                                                    Z_TREES: 6,
                                                    Z_OK: 0,
                                                    Z_STREAM_END: 1,
                                                    Z_NEED_DICT: 2,
                                                    Z_ERRNO: -1,
                                                    Z_STREAM_ERROR: -2,
                                                    Z_DATA_ERROR: -3,
                                                    Z_BUF_ERROR: -5,
                                                    Z_NO_COMPRESSION: 0,
                                                    Z_BEST_SPEED: 1,
                                                    Z_BEST_COMPRESSION: 9,
                                                    Z_DEFAULT_COMPRESSION: -1,
                                                    Z_FILTERED: 1,
                                                    Z_HUFFMAN_ONLY: 2,
                                                    Z_RLE: 3,
                                                    Z_FIXED: 4,
                                                    Z_DEFAULT_STRATEGY: 0,
                                                    Z_BINARY: 0,
                                                    Z_TEXT: 1,
                                                    Z_UNKNOWN: 2,
                                                    Z_DEFLATED: 8
                                                  };
                                                }, {}],
                                                45: [function (e, t, r) {

                                                  var o = function () {
                                                    for (var e, t = [], r = 0; r < 256; r++) {
                                                      e = r;

                                                      for (var n = 0; n < 8; n++) e = 1 & e ? 3988292384 ^ e >>> 1 : e >>> 1;

                                                      t[r] = e;
                                                    }

                                                    return t;
                                                  }();

                                                  t.exports = function (e, t, r, n) {
                                                    var i = o,
                                                        s = n + r;
                                                    e ^= -1;

                                                    for (var a = n; a < s; a++) e = e >>> 8 ^ i[255 & (e ^ t[a])];

                                                    return -1 ^ e;
                                                  };
                                                }, {}],
                                                46: [function (e, t, r) {

                                                  var u,
                                                      d = e("../utils/common"),
                                                      h = e("./trees"),
                                                      c = e("./adler32"),
                                                      p = e("./crc32"),
                                                      n = e("./messages"),
                                                      f = 0,
                                                      l = 0,
                                                      m = -2,
                                                      i = 2,
                                                      _ = 8,
                                                      s = 286,
                                                      a = 30,
                                                      o = 19,
                                                      g = 2 * s + 1,
                                                      v = 15,
                                                      b = 3,
                                                      w = 258,
                                                      y = w + b + 1,
                                                      k = 42,
                                                      x = 113;

                                                  function S(e, t) {
                                                    return e.msg = n[t], t;
                                                  }

                                                  function z(e) {
                                                    return (e << 1) - (4 < e ? 9 : 0);
                                                  }

                                                  function C(e) {
                                                    for (var t = e.length; 0 <= --t;) e[t] = 0;
                                                  }

                                                  function E(e) {
                                                    var t = e.state,
                                                        r = t.pending;
                                                    r > e.avail_out && (r = e.avail_out), 0 !== r && (d.arraySet(e.output, t.pending_buf, t.pending_out, r, e.next_out), e.next_out += r, t.pending_out += r, e.total_out += r, e.avail_out -= r, t.pending -= r, 0 === t.pending && (t.pending_out = 0));
                                                  }

                                                  function A(e, t) {
                                                    h._tr_flush_block(e, 0 <= e.block_start ? e.block_start : -1, e.strstart - e.block_start, t), e.block_start = e.strstart, E(e.strm);
                                                  }

                                                  function I(e, t) {
                                                    e.pending_buf[e.pending++] = t;
                                                  }

                                                  function O(e, t) {
                                                    e.pending_buf[e.pending++] = t >>> 8 & 255, e.pending_buf[e.pending++] = 255 & t;
                                                  }

                                                  function B(e, t) {
                                                    var r,
                                                        n,
                                                        i = e.max_chain_length,
                                                        s = e.strstart,
                                                        a = e.prev_length,
                                                        o = e.nice_match,
                                                        u = e.strstart > e.w_size - y ? e.strstart - (e.w_size - y) : 0,
                                                        h = e.window,
                                                        f = e.w_mask,
                                                        l = e.prev,
                                                        d = e.strstart + w,
                                                        c = h[s + a - 1],
                                                        p = h[s + a];
                                                    e.prev_length >= e.good_match && (i >>= 2), o > e.lookahead && (o = e.lookahead);

                                                    do {
                                                      if (h[(r = t) + a] === p && h[r + a - 1] === c && h[r] === h[s] && h[++r] === h[s + 1]) {
                                                        s += 2, r++;

                                                        do {} while (h[++s] === h[++r] && h[++s] === h[++r] && h[++s] === h[++r] && h[++s] === h[++r] && h[++s] === h[++r] && h[++s] === h[++r] && h[++s] === h[++r] && h[++s] === h[++r] && s < d);

                                                        if (n = w - (d - s), s = d - w, a < n) {
                                                          if (e.match_start = t, o <= (a = n)) break;
                                                          c = h[s + a - 1], p = h[s + a];
                                                        }
                                                      }
                                                    } while ((t = l[t & f]) > u && 0 != --i);

                                                    return a <= e.lookahead ? a : e.lookahead;
                                                  }

                                                  function R(e) {
                                                    var t,
                                                        r,
                                                        n,
                                                        i,
                                                        s,
                                                        a,
                                                        o,
                                                        u,
                                                        h,
                                                        f,
                                                        l = e.w_size;

                                                    do {
                                                      if (i = e.window_size - e.lookahead - e.strstart, e.strstart >= l + (l - y)) {
                                                        for (d.arraySet(e.window, e.window, l, l, 0), e.match_start -= l, e.strstart -= l, e.block_start -= l, t = r = e.hash_size; n = e.head[--t], e.head[t] = l <= n ? n - l : 0, --r;);

                                                        for (t = r = l; n = e.prev[--t], e.prev[t] = l <= n ? n - l : 0, --r;);

                                                        i += l;
                                                      }

                                                      if (0 === e.strm.avail_in) break;
                                                      if (a = e.strm, o = e.window, u = e.strstart + e.lookahead, f = void 0, (h = i) < (f = a.avail_in) && (f = h), r = 0 === f ? 0 : (a.avail_in -= f, d.arraySet(o, a.input, a.next_in, f, u), 1 === a.state.wrap ? a.adler = c(a.adler, o, f, u) : 2 === a.state.wrap && (a.adler = p(a.adler, o, f, u)), a.next_in += f, a.total_in += f, f), e.lookahead += r, e.lookahead + e.insert >= b) for (s = e.strstart - e.insert, e.ins_h = e.window[s], e.ins_h = (e.ins_h << e.hash_shift ^ e.window[s + 1]) & e.hash_mask; e.insert && (e.ins_h = (e.ins_h << e.hash_shift ^ e.window[s + b - 1]) & e.hash_mask, e.prev[s & e.w_mask] = e.head[e.ins_h], e.head[e.ins_h] = s, s++, e.insert--, !(e.lookahead + e.insert < b)););
                                                    } while (e.lookahead < y && 0 !== e.strm.avail_in);
                                                  }

                                                  function T(e, t) {
                                                    for (var r, n;;) {
                                                      if (e.lookahead < y) {
                                                        if (R(e), e.lookahead < y && t === f) return 1;
                                                        if (0 === e.lookahead) break;
                                                      }

                                                      if (r = 0, e.lookahead >= b && (e.ins_h = (e.ins_h << e.hash_shift ^ e.window[e.strstart + b - 1]) & e.hash_mask, r = e.prev[e.strstart & e.w_mask] = e.head[e.ins_h], e.head[e.ins_h] = e.strstart), 0 !== r && e.strstart - r <= e.w_size - y && (e.match_length = B(e, r)), e.match_length >= b) {
                                                        if (n = h._tr_tally(e, e.strstart - e.match_start, e.match_length - b), e.lookahead -= e.match_length, e.match_length <= e.max_lazy_match && e.lookahead >= b) {
                                                          for (e.match_length--; e.strstart++, e.ins_h = (e.ins_h << e.hash_shift ^ e.window[e.strstart + b - 1]) & e.hash_mask, r = e.prev[e.strstart & e.w_mask] = e.head[e.ins_h], e.head[e.ins_h] = e.strstart, 0 != --e.match_length;);

                                                          e.strstart++;
                                                        } else e.strstart += e.match_length, e.match_length = 0, e.ins_h = e.window[e.strstart], e.ins_h = (e.ins_h << e.hash_shift ^ e.window[e.strstart + 1]) & e.hash_mask;
                                                      } else n = h._tr_tally(e, 0, e.window[e.strstart]), e.lookahead--, e.strstart++;
                                                      if (n && (A(e, !1), 0 === e.strm.avail_out)) return 1;
                                                    }

                                                    return e.insert = e.strstart < b - 1 ? e.strstart : b - 1, 4 === t ? (A(e, !0), 0 === e.strm.avail_out ? 3 : 4) : e.last_lit && (A(e, !1), 0 === e.strm.avail_out) ? 1 : 2;
                                                  }

                                                  function D(e, t) {
                                                    for (var r, n, i;;) {
                                                      if (e.lookahead < y) {
                                                        if (R(e), e.lookahead < y && t === f) return 1;
                                                        if (0 === e.lookahead) break;
                                                      }

                                                      if (r = 0, e.lookahead >= b && (e.ins_h = (e.ins_h << e.hash_shift ^ e.window[e.strstart + b - 1]) & e.hash_mask, r = e.prev[e.strstart & e.w_mask] = e.head[e.ins_h], e.head[e.ins_h] = e.strstart), e.prev_length = e.match_length, e.prev_match = e.match_start, e.match_length = b - 1, 0 !== r && e.prev_length < e.max_lazy_match && e.strstart - r <= e.w_size - y && (e.match_length = B(e, r), e.match_length <= 5 && (1 === e.strategy || e.match_length === b && 4096 < e.strstart - e.match_start) && (e.match_length = b - 1)), e.prev_length >= b && e.match_length <= e.prev_length) {
                                                        for (i = e.strstart + e.lookahead - b, n = h._tr_tally(e, e.strstart - 1 - e.prev_match, e.prev_length - b), e.lookahead -= e.prev_length - 1, e.prev_length -= 2; ++e.strstart <= i && (e.ins_h = (e.ins_h << e.hash_shift ^ e.window[e.strstart + b - 1]) & e.hash_mask, r = e.prev[e.strstart & e.w_mask] = e.head[e.ins_h], e.head[e.ins_h] = e.strstart), 0 != --e.prev_length;);

                                                        if (e.match_available = 0, e.match_length = b - 1, e.strstart++, n && (A(e, !1), 0 === e.strm.avail_out)) return 1;
                                                      } else if (e.match_available) {
                                                        if ((n = h._tr_tally(e, 0, e.window[e.strstart - 1])) && A(e, !1), e.strstart++, e.lookahead--, 0 === e.strm.avail_out) return 1;
                                                      } else e.match_available = 1, e.strstart++, e.lookahead--;
                                                    }

                                                    return e.match_available && (n = h._tr_tally(e, 0, e.window[e.strstart - 1]), e.match_available = 0), e.insert = e.strstart < b - 1 ? e.strstart : b - 1, 4 === t ? (A(e, !0), 0 === e.strm.avail_out ? 3 : 4) : e.last_lit && (A(e, !1), 0 === e.strm.avail_out) ? 1 : 2;
                                                  }

                                                  function F(e, t, r, n, i) {
                                                    this.good_length = e, this.max_lazy = t, this.nice_length = r, this.max_chain = n, this.func = i;
                                                  }

                                                  function N() {
                                                    this.strm = null, this.status = 0, this.pending_buf = null, this.pending_buf_size = 0, this.pending_out = 0, this.pending = 0, this.wrap = 0, this.gzhead = null, this.gzindex = 0, this.method = _, this.last_flush = -1, this.w_size = 0, this.w_bits = 0, this.w_mask = 0, this.window = null, this.window_size = 0, this.prev = null, this.head = null, this.ins_h = 0, this.hash_size = 0, this.hash_bits = 0, this.hash_mask = 0, this.hash_shift = 0, this.block_start = 0, this.match_length = 0, this.prev_match = 0, this.match_available = 0, this.strstart = 0, this.match_start = 0, this.lookahead = 0, this.prev_length = 0, this.max_chain_length = 0, this.max_lazy_match = 0, this.level = 0, this.strategy = 0, this.good_match = 0, this.nice_match = 0, this.dyn_ltree = new d.Buf16(2 * g), this.dyn_dtree = new d.Buf16(2 * (2 * a + 1)), this.bl_tree = new d.Buf16(2 * (2 * o + 1)), C(this.dyn_ltree), C(this.dyn_dtree), C(this.bl_tree), this.l_desc = null, this.d_desc = null, this.bl_desc = null, this.bl_count = new d.Buf16(v + 1), this.heap = new d.Buf16(2 * s + 1), C(this.heap), this.heap_len = 0, this.heap_max = 0, this.depth = new d.Buf16(2 * s + 1), C(this.depth), this.l_buf = 0, this.lit_bufsize = 0, this.last_lit = 0, this.d_buf = 0, this.opt_len = 0, this.static_len = 0, this.matches = 0, this.insert = 0, this.bi_buf = 0, this.bi_valid = 0;
                                                  }

                                                  function U(e) {
                                                    var t;
                                                    return e && e.state ? (e.total_in = e.total_out = 0, e.data_type = i, (t = e.state).pending = 0, t.pending_out = 0, t.wrap < 0 && (t.wrap = -t.wrap), t.status = t.wrap ? k : x, e.adler = 2 === t.wrap ? 0 : 1, t.last_flush = f, h._tr_init(t), l) : S(e, m);
                                                  }

                                                  function P(e) {
                                                    var t,
                                                        r = U(e);
                                                    return r === l && ((t = e.state).window_size = 2 * t.w_size, C(t.head), t.max_lazy_match = u[t.level].max_lazy, t.good_match = u[t.level].good_length, t.nice_match = u[t.level].nice_length, t.max_chain_length = u[t.level].max_chain, t.strstart = 0, t.block_start = 0, t.lookahead = 0, t.insert = 0, t.match_length = t.prev_length = b - 1, t.match_available = 0, t.ins_h = 0), r;
                                                  }

                                                  function L(e, t, r, n, i, s) {
                                                    if (!e) return m;
                                                    var a = 1;
                                                    if (-1 === t && (t = 6), n < 0 ? (a = 0, n = -n) : 15 < n && (a = 2, n -= 16), i < 1 || 9 < i || r !== _ || n < 8 || 15 < n || t < 0 || 9 < t || s < 0 || 4 < s) return S(e, m);
                                                    8 === n && (n = 9);
                                                    var o = new N();
                                                    return (e.state = o).strm = e, o.wrap = a, o.gzhead = null, o.w_bits = n, o.w_size = 1 << o.w_bits, o.w_mask = o.w_size - 1, o.hash_bits = i + 7, o.hash_size = 1 << o.hash_bits, o.hash_mask = o.hash_size - 1, o.hash_shift = ~~((o.hash_bits + b - 1) / b), o.window = new d.Buf8(2 * o.w_size), o.head = new d.Buf16(o.hash_size), o.prev = new d.Buf16(o.w_size), o.lit_bufsize = 1 << i + 6, o.pending_buf_size = 4 * o.lit_bufsize, o.pending_buf = new d.Buf8(o.pending_buf_size), o.d_buf = 1 * o.lit_bufsize, o.l_buf = 3 * o.lit_bufsize, o.level = t, o.strategy = s, o.method = r, P(e);
                                                  }

                                                  u = [new F(0, 0, 0, 0, function (e, t) {
                                                    var r = 65535;

                                                    for (r > e.pending_buf_size - 5 && (r = e.pending_buf_size - 5);;) {
                                                      if (e.lookahead <= 1) {
                                                        if (R(e), 0 === e.lookahead && t === f) return 1;
                                                        if (0 === e.lookahead) break;
                                                      }

                                                      e.strstart += e.lookahead, e.lookahead = 0;
                                                      var n = e.block_start + r;
                                                      if ((0 === e.strstart || e.strstart >= n) && (e.lookahead = e.strstart - n, e.strstart = n, A(e, !1), 0 === e.strm.avail_out)) return 1;
                                                      if (e.strstart - e.block_start >= e.w_size - y && (A(e, !1), 0 === e.strm.avail_out)) return 1;
                                                    }

                                                    return e.insert = 0, 4 === t ? (A(e, !0), 0 === e.strm.avail_out ? 3 : 4) : (e.strstart > e.block_start && (A(e, !1), e.strm.avail_out), 1);
                                                  }), new F(4, 4, 8, 4, T), new F(4, 5, 16, 8, T), new F(4, 6, 32, 32, T), new F(4, 4, 16, 16, D), new F(8, 16, 32, 32, D), new F(8, 16, 128, 128, D), new F(8, 32, 128, 256, D), new F(32, 128, 258, 1024, D), new F(32, 258, 258, 4096, D)], r.deflateInit = function (e, t) {
                                                    return L(e, t, _, 15, 8, 0);
                                                  }, r.deflateInit2 = L, r.deflateReset = P, r.deflateResetKeep = U, r.deflateSetHeader = function (e, t) {
                                                    return e && e.state ? 2 !== e.state.wrap ? m : (e.state.gzhead = t, l) : m;
                                                  }, r.deflate = function (e, t) {
                                                    var r, n, i, s;
                                                    if (!e || !e.state || 5 < t || t < 0) return e ? S(e, m) : m;
                                                    if (n = e.state, !e.output || !e.input && 0 !== e.avail_in || 666 === n.status && 4 !== t) return S(e, 0 === e.avail_out ? -5 : m);
                                                    if (n.strm = e, r = n.last_flush, n.last_flush = t, n.status === k) if (2 === n.wrap) e.adler = 0, I(n, 31), I(n, 139), I(n, 8), n.gzhead ? (I(n, (n.gzhead.text ? 1 : 0) + (n.gzhead.hcrc ? 2 : 0) + (n.gzhead.extra ? 4 : 0) + (n.gzhead.name ? 8 : 0) + (n.gzhead.comment ? 16 : 0)), I(n, 255 & n.gzhead.time), I(n, n.gzhead.time >> 8 & 255), I(n, n.gzhead.time >> 16 & 255), I(n, n.gzhead.time >> 24 & 255), I(n, 9 === n.level ? 2 : 2 <= n.strategy || n.level < 2 ? 4 : 0), I(n, 255 & n.gzhead.os), n.gzhead.extra && n.gzhead.extra.length && (I(n, 255 & n.gzhead.extra.length), I(n, n.gzhead.extra.length >> 8 & 255)), n.gzhead.hcrc && (e.adler = p(e.adler, n.pending_buf, n.pending, 0)), n.gzindex = 0, n.status = 69) : (I(n, 0), I(n, 0), I(n, 0), I(n, 0), I(n, 0), I(n, 9 === n.level ? 2 : 2 <= n.strategy || n.level < 2 ? 4 : 0), I(n, 3), n.status = x);else {
                                                      var a = _ + (n.w_bits - 8 << 4) << 8;
                                                      a |= (2 <= n.strategy || n.level < 2 ? 0 : n.level < 6 ? 1 : 6 === n.level ? 2 : 3) << 6, 0 !== n.strstart && (a |= 32), a += 31 - a % 31, n.status = x, O(n, a), 0 !== n.strstart && (O(n, e.adler >>> 16), O(n, 65535 & e.adler)), e.adler = 1;
                                                    }
                                                    if (69 === n.status) if (n.gzhead.extra) {
                                                      for (i = n.pending; n.gzindex < (65535 & n.gzhead.extra.length) && (n.pending !== n.pending_buf_size || (n.gzhead.hcrc && n.pending > i && (e.adler = p(e.adler, n.pending_buf, n.pending - i, i)), E(e), i = n.pending, n.pending !== n.pending_buf_size));) I(n, 255 & n.gzhead.extra[n.gzindex]), n.gzindex++;

                                                      n.gzhead.hcrc && n.pending > i && (e.adler = p(e.adler, n.pending_buf, n.pending - i, i)), n.gzindex === n.gzhead.extra.length && (n.gzindex = 0, n.status = 73);
                                                    } else n.status = 73;
                                                    if (73 === n.status) if (n.gzhead.name) {
                                                      i = n.pending;

                                                      do {
                                                        if (n.pending === n.pending_buf_size && (n.gzhead.hcrc && n.pending > i && (e.adler = p(e.adler, n.pending_buf, n.pending - i, i)), E(e), i = n.pending, n.pending === n.pending_buf_size)) {
                                                          s = 1;
                                                          break;
                                                        }

                                                        s = n.gzindex < n.gzhead.name.length ? 255 & n.gzhead.name.charCodeAt(n.gzindex++) : 0, I(n, s);
                                                      } while (0 !== s);

                                                      n.gzhead.hcrc && n.pending > i && (e.adler = p(e.adler, n.pending_buf, n.pending - i, i)), 0 === s && (n.gzindex = 0, n.status = 91);
                                                    } else n.status = 91;
                                                    if (91 === n.status) if (n.gzhead.comment) {
                                                      i = n.pending;

                                                      do {
                                                        if (n.pending === n.pending_buf_size && (n.gzhead.hcrc && n.pending > i && (e.adler = p(e.adler, n.pending_buf, n.pending - i, i)), E(e), i = n.pending, n.pending === n.pending_buf_size)) {
                                                          s = 1;
                                                          break;
                                                        }

                                                        s = n.gzindex < n.gzhead.comment.length ? 255 & n.gzhead.comment.charCodeAt(n.gzindex++) : 0, I(n, s);
                                                      } while (0 !== s);

                                                      n.gzhead.hcrc && n.pending > i && (e.adler = p(e.adler, n.pending_buf, n.pending - i, i)), 0 === s && (n.status = 103);
                                                    } else n.status = 103;

                                                    if (103 === n.status && (n.gzhead.hcrc ? (n.pending + 2 > n.pending_buf_size && E(e), n.pending + 2 <= n.pending_buf_size && (I(n, 255 & e.adler), I(n, e.adler >> 8 & 255), e.adler = 0, n.status = x)) : n.status = x), 0 !== n.pending) {
                                                      if (E(e), 0 === e.avail_out) return n.last_flush = -1, l;
                                                    } else if (0 === e.avail_in && z(t) <= z(r) && 4 !== t) return S(e, -5);

                                                    if (666 === n.status && 0 !== e.avail_in) return S(e, -5);

                                                    if (0 !== e.avail_in || 0 !== n.lookahead || t !== f && 666 !== n.status) {
                                                      var o = 2 === n.strategy ? function (e, t) {
                                                        for (var r;;) {
                                                          if (0 === e.lookahead && (R(e), 0 === e.lookahead)) {
                                                            if (t === f) return 1;
                                                            break;
                                                          }

                                                          if (e.match_length = 0, r = h._tr_tally(e, 0, e.window[e.strstart]), e.lookahead--, e.strstart++, r && (A(e, !1), 0 === e.strm.avail_out)) return 1;
                                                        }

                                                        return e.insert = 0, 4 === t ? (A(e, !0), 0 === e.strm.avail_out ? 3 : 4) : e.last_lit && (A(e, !1), 0 === e.strm.avail_out) ? 1 : 2;
                                                      }(n, t) : 3 === n.strategy ? function (e, t) {
                                                        for (var r, n, i, s, a = e.window;;) {
                                                          if (e.lookahead <= w) {
                                                            if (R(e), e.lookahead <= w && t === f) return 1;
                                                            if (0 === e.lookahead) break;
                                                          }

                                                          if (e.match_length = 0, e.lookahead >= b && 0 < e.strstart && (n = a[i = e.strstart - 1]) === a[++i] && n === a[++i] && n === a[++i]) {
                                                            s = e.strstart + w;

                                                            do {} while (n === a[++i] && n === a[++i] && n === a[++i] && n === a[++i] && n === a[++i] && n === a[++i] && n === a[++i] && n === a[++i] && i < s);

                                                            e.match_length = w - (s - i), e.match_length > e.lookahead && (e.match_length = e.lookahead);
                                                          }

                                                          if (e.match_length >= b ? (r = h._tr_tally(e, 1, e.match_length - b), e.lookahead -= e.match_length, e.strstart += e.match_length, e.match_length = 0) : (r = h._tr_tally(e, 0, e.window[e.strstart]), e.lookahead--, e.strstart++), r && (A(e, !1), 0 === e.strm.avail_out)) return 1;
                                                        }

                                                        return e.insert = 0, 4 === t ? (A(e, !0), 0 === e.strm.avail_out ? 3 : 4) : e.last_lit && (A(e, !1), 0 === e.strm.avail_out) ? 1 : 2;
                                                      }(n, t) : u[n.level].func(n, t);
                                                      if (3 !== o && 4 !== o || (n.status = 666), 1 === o || 3 === o) return 0 === e.avail_out && (n.last_flush = -1), l;
                                                      if (2 === o && (1 === t ? h._tr_align(n) : 5 !== t && (h._tr_stored_block(n, 0, 0, !1), 3 === t && (C(n.head), 0 === n.lookahead && (n.strstart = 0, n.block_start = 0, n.insert = 0))), E(e), 0 === e.avail_out)) return n.last_flush = -1, l;
                                                    }

                                                    return 4 !== t ? l : n.wrap <= 0 ? 1 : (2 === n.wrap ? (I(n, 255 & e.adler), I(n, e.adler >> 8 & 255), I(n, e.adler >> 16 & 255), I(n, e.adler >> 24 & 255), I(n, 255 & e.total_in), I(n, e.total_in >> 8 & 255), I(n, e.total_in >> 16 & 255), I(n, e.total_in >> 24 & 255)) : (O(n, e.adler >>> 16), O(n, 65535 & e.adler)), E(e), 0 < n.wrap && (n.wrap = -n.wrap), 0 !== n.pending ? l : 1);
                                                  }, r.deflateEnd = function (e) {
                                                    var t;
                                                    return e && e.state ? (t = e.state.status) !== k && 69 !== t && 73 !== t && 91 !== t && 103 !== t && t !== x && 666 !== t ? S(e, m) : (e.state = null, t === x ? S(e, -3) : l) : m;
                                                  }, r.deflateSetDictionary = function (e, t) {
                                                    var r,
                                                        n,
                                                        i,
                                                        s,
                                                        a,
                                                        o,
                                                        u,
                                                        h,
                                                        f = t.length;
                                                    if (!e || !e.state) return m;
                                                    if (2 === (s = (r = e.state).wrap) || 1 === s && r.status !== k || r.lookahead) return m;

                                                    for (1 === s && (e.adler = c(e.adler, t, f, 0)), r.wrap = 0, f >= r.w_size && (0 === s && (C(r.head), r.strstart = 0, r.block_start = 0, r.insert = 0), h = new d.Buf8(r.w_size), d.arraySet(h, t, f - r.w_size, r.w_size, 0), t = h, f = r.w_size), a = e.avail_in, o = e.next_in, u = e.input, e.avail_in = f, e.next_in = 0, e.input = t, R(r); r.lookahead >= b;) {
                                                      for (n = r.strstart, i = r.lookahead - (b - 1); r.ins_h = (r.ins_h << r.hash_shift ^ r.window[n + b - 1]) & r.hash_mask, r.prev[n & r.w_mask] = r.head[r.ins_h], r.head[r.ins_h] = n, n++, --i;);

                                                      r.strstart = n, r.lookahead = b - 1, R(r);
                                                    }

                                                    return r.strstart += r.lookahead, r.block_start = r.strstart, r.insert = r.lookahead, r.lookahead = 0, r.match_length = r.prev_length = b - 1, r.match_available = 0, e.next_in = o, e.input = u, e.avail_in = a, r.wrap = s, l;
                                                  }, r.deflateInfo = "pako deflate (from Nodeca project)";
                                                }, {
                                                  "../utils/common": 41,
                                                  "./adler32": 43,
                                                  "./crc32": 45,
                                                  "./messages": 51,
                                                  "./trees": 52
                                                }],
                                                47: [function (e, t, r) {

                                                  t.exports = function () {
                                                    this.text = 0, this.time = 0, this.xflags = 0, this.os = 0, this.extra = null, this.extra_len = 0, this.name = "", this.comment = "", this.hcrc = 0, this.done = !1;
                                                  };
                                                }, {}],
                                                48: [function (e, t, r) {

                                                  t.exports = function (e, t) {
                                                    var r, n, i, s, a, o, u, h, f, l, d, c, p, m, _, g, v, b, w, y, k, x, S, z, C;

                                                    r = e.state, n = e.next_in, z = e.input, i = n + (e.avail_in - 5), s = e.next_out, C = e.output, a = s - (t - e.avail_out), o = s + (e.avail_out - 257), u = r.dmax, h = r.wsize, f = r.whave, l = r.wnext, d = r.window, c = r.hold, p = r.bits, m = r.lencode, _ = r.distcode, g = (1 << r.lenbits) - 1, v = (1 << r.distbits) - 1;

                                                    e: do {
                                                      p < 15 && (c += z[n++] << p, p += 8, c += z[n++] << p, p += 8), b = m[c & g];

                                                      t: for (;;) {
                                                        if (c >>>= w = b >>> 24, p -= w, 0 == (w = b >>> 16 & 255)) C[s++] = 65535 & b;else {
                                                          if (!(16 & w)) {
                                                            if (0 == (64 & w)) {
                                                              b = m[(65535 & b) + (c & (1 << w) - 1)];
                                                              continue t;
                                                            }

                                                            if (32 & w) {
                                                              r.mode = 12;
                                                              break e;
                                                            }

                                                            e.msg = "invalid literal/length code", r.mode = 30;
                                                            break e;
                                                          }

                                                          y = 65535 & b, (w &= 15) && (p < w && (c += z[n++] << p, p += 8), y += c & (1 << w) - 1, c >>>= w, p -= w), p < 15 && (c += z[n++] << p, p += 8, c += z[n++] << p, p += 8), b = _[c & v];

                                                          r: for (;;) {
                                                            if (c >>>= w = b >>> 24, p -= w, !(16 & (w = b >>> 16 & 255))) {
                                                              if (0 == (64 & w)) {
                                                                b = _[(65535 & b) + (c & (1 << w) - 1)];
                                                                continue r;
                                                              }

                                                              e.msg = "invalid distance code", r.mode = 30;
                                                              break e;
                                                            }

                                                            if (k = 65535 & b, p < (w &= 15) && (c += z[n++] << p, (p += 8) < w && (c += z[n++] << p, p += 8)), u < (k += c & (1 << w) - 1)) {
                                                              e.msg = "invalid distance too far back", r.mode = 30;
                                                              break e;
                                                            }

                                                            if (c >>>= w, p -= w, (w = s - a) < k) {
                                                              if (f < (w = k - w) && r.sane) {
                                                                e.msg = "invalid distance too far back", r.mode = 30;
                                                                break e;
                                                              }

                                                              if (S = d, (x = 0) === l) {
                                                                if (x += h - w, w < y) {
                                                                  for (y -= w; C[s++] = d[x++], --w;);

                                                                  x = s - k, S = C;
                                                                }
                                                              } else if (l < w) {
                                                                if (x += h + l - w, (w -= l) < y) {
                                                                  for (y -= w; C[s++] = d[x++], --w;);

                                                                  if (x = 0, l < y) {
                                                                    for (y -= w = l; C[s++] = d[x++], --w;);

                                                                    x = s - k, S = C;
                                                                  }
                                                                }
                                                              } else if (x += l - w, w < y) {
                                                                for (y -= w; C[s++] = d[x++], --w;);

                                                                x = s - k, S = C;
                                                              }

                                                              for (; 2 < y;) C[s++] = S[x++], C[s++] = S[x++], C[s++] = S[x++], y -= 3;

                                                              y && (C[s++] = S[x++], 1 < y && (C[s++] = S[x++]));
                                                            } else {
                                                              for (x = s - k; C[s++] = C[x++], C[s++] = C[x++], C[s++] = C[x++], 2 < (y -= 3););

                                                              y && (C[s++] = C[x++], 1 < y && (C[s++] = C[x++]));
                                                            }

                                                            break;
                                                          }
                                                        }
                                                        break;
                                                      }
                                                    } while (n < i && s < o);

                                                    n -= y = p >> 3, c &= (1 << (p -= y << 3)) - 1, e.next_in = n, e.next_out = s, e.avail_in = n < i ? i - n + 5 : 5 - (n - i), e.avail_out = s < o ? o - s + 257 : 257 - (s - o), r.hold = c, r.bits = p;
                                                  };
                                                }, {}],
                                                49: [function (e, t, r) {

                                                  var I = e("../utils/common"),
                                                      O = e("./adler32"),
                                                      B = e("./crc32"),
                                                      R = e("./inffast"),
                                                      T = e("./inftrees"),
                                                      D = 1,
                                                      F = 2,
                                                      N = 0,
                                                      U = -2,
                                                      P = 1,
                                                      n = 852,
                                                      i = 592;

                                                  function L(e) {
                                                    return (e >>> 24 & 255) + (e >>> 8 & 65280) + ((65280 & e) << 8) + ((255 & e) << 24);
                                                  }

                                                  function s() {
                                                    this.mode = 0, this.last = !1, this.wrap = 0, this.havedict = !1, this.flags = 0, this.dmax = 0, this.check = 0, this.total = 0, this.head = null, this.wbits = 0, this.wsize = 0, this.whave = 0, this.wnext = 0, this.window = null, this.hold = 0, this.bits = 0, this.length = 0, this.offset = 0, this.extra = 0, this.lencode = null, this.distcode = null, this.lenbits = 0, this.distbits = 0, this.ncode = 0, this.nlen = 0, this.ndist = 0, this.have = 0, this.next = null, this.lens = new I.Buf16(320), this.work = new I.Buf16(288), this.lendyn = null, this.distdyn = null, this.sane = 0, this.back = 0, this.was = 0;
                                                  }

                                                  function a(e) {
                                                    var t;
                                                    return e && e.state ? (t = e.state, e.total_in = e.total_out = t.total = 0, e.msg = "", t.wrap && (e.adler = 1 & t.wrap), t.mode = P, t.last = 0, t.havedict = 0, t.dmax = 32768, t.head = null, t.hold = 0, t.bits = 0, t.lencode = t.lendyn = new I.Buf32(n), t.distcode = t.distdyn = new I.Buf32(i), t.sane = 1, t.back = -1, N) : U;
                                                  }

                                                  function o(e) {
                                                    var t;
                                                    return e && e.state ? ((t = e.state).wsize = 0, t.whave = 0, t.wnext = 0, a(e)) : U;
                                                  }

                                                  function u(e, t) {
                                                    var r, n;
                                                    return e && e.state ? (n = e.state, t < 0 ? (r = 0, t = -t) : (r = 1 + (t >> 4), t < 48 && (t &= 15)), t && (t < 8 || 15 < t) ? U : (null !== n.window && n.wbits !== t && (n.window = null), n.wrap = r, n.wbits = t, o(e))) : U;
                                                  }

                                                  function h(e, t) {
                                                    var r, n;
                                                    return e ? (n = new s(), (e.state = n).window = null, (r = u(e, t)) !== N && (e.state = null), r) : U;
                                                  }

                                                  var f,
                                                      l,
                                                      d = !0;

                                                  function j(e) {
                                                    if (d) {
                                                      var t;

                                                      for (f = new I.Buf32(512), l = new I.Buf32(32), t = 0; t < 144;) e.lens[t++] = 8;

                                                      for (; t < 256;) e.lens[t++] = 9;

                                                      for (; t < 280;) e.lens[t++] = 7;

                                                      for (; t < 288;) e.lens[t++] = 8;

                                                      for (T(D, e.lens, 0, 288, f, 0, e.work, {
                                                        bits: 9
                                                      }), t = 0; t < 32;) e.lens[t++] = 5;

                                                      T(F, e.lens, 0, 32, l, 0, e.work, {
                                                        bits: 5
                                                      }), d = !1;
                                                    }

                                                    e.lencode = f, e.lenbits = 9, e.distcode = l, e.distbits = 5;
                                                  }

                                                  function Z(e, t, r, n) {
                                                    var i,
                                                        s = e.state;
                                                    return null === s.window && (s.wsize = 1 << s.wbits, s.wnext = 0, s.whave = 0, s.window = new I.Buf8(s.wsize)), n >= s.wsize ? (I.arraySet(s.window, t, r - s.wsize, s.wsize, 0), s.wnext = 0, s.whave = s.wsize) : (n < (i = s.wsize - s.wnext) && (i = n), I.arraySet(s.window, t, r - n, i, s.wnext), (n -= i) ? (I.arraySet(s.window, t, r - n, n, 0), s.wnext = n, s.whave = s.wsize) : (s.wnext += i, s.wnext === s.wsize && (s.wnext = 0), s.whave < s.wsize && (s.whave += i))), 0;
                                                  }

                                                  r.inflateReset = o, r.inflateReset2 = u, r.inflateResetKeep = a, r.inflateInit = function (e) {
                                                    return h(e, 15);
                                                  }, r.inflateInit2 = h, r.inflate = function (e, t) {
                                                    var r,
                                                        n,
                                                        i,
                                                        s,
                                                        a,
                                                        o,
                                                        u,
                                                        h,
                                                        f,
                                                        l,
                                                        d,
                                                        c,
                                                        p,
                                                        m,
                                                        _,
                                                        g,
                                                        v,
                                                        b,
                                                        w,
                                                        y,
                                                        k,
                                                        x,
                                                        S,
                                                        z,
                                                        C = 0,
                                                        E = new I.Buf8(4),
                                                        A = [16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15];

                                                    if (!e || !e.state || !e.output || !e.input && 0 !== e.avail_in) return U;
                                                    12 === (r = e.state).mode && (r.mode = 13), a = e.next_out, i = e.output, u = e.avail_out, s = e.next_in, n = e.input, o = e.avail_in, h = r.hold, f = r.bits, l = o, d = u, x = N;

                                                    e: for (;;) switch (r.mode) {
                                                      case P:
                                                        if (0 === r.wrap) {
                                                          r.mode = 13;
                                                          break;
                                                        }

                                                        for (; f < 16;) {
                                                          if (0 === o) break e;
                                                          o--, h += n[s++] << f, f += 8;
                                                        }

                                                        if (2 & r.wrap && 35615 === h) {
                                                          E[r.check = 0] = 255 & h, E[1] = h >>> 8 & 255, r.check = B(r.check, E, 2, 0), f = h = 0, r.mode = 2;
                                                          break;
                                                        }

                                                        if (r.flags = 0, r.head && (r.head.done = !1), !(1 & r.wrap) || (((255 & h) << 8) + (h >> 8)) % 31) {
                                                          e.msg = "incorrect header check", r.mode = 30;
                                                          break;
                                                        }

                                                        if (8 != (15 & h)) {
                                                          e.msg = "unknown compression method", r.mode = 30;
                                                          break;
                                                        }

                                                        if (f -= 4, k = 8 + (15 & (h >>>= 4)), 0 === r.wbits) r.wbits = k;else if (k > r.wbits) {
                                                          e.msg = "invalid window size", r.mode = 30;
                                                          break;
                                                        }
                                                        r.dmax = 1 << k, e.adler = r.check = 1, r.mode = 512 & h ? 10 : 12, f = h = 0;
                                                        break;

                                                      case 2:
                                                        for (; f < 16;) {
                                                          if (0 === o) break e;
                                                          o--, h += n[s++] << f, f += 8;
                                                        }

                                                        if (r.flags = h, 8 != (255 & r.flags)) {
                                                          e.msg = "unknown compression method", r.mode = 30;
                                                          break;
                                                        }

                                                        if (57344 & r.flags) {
                                                          e.msg = "unknown header flags set", r.mode = 30;
                                                          break;
                                                        }

                                                        r.head && (r.head.text = h >> 8 & 1), 512 & r.flags && (E[0] = 255 & h, E[1] = h >>> 8 & 255, r.check = B(r.check, E, 2, 0)), f = h = 0, r.mode = 3;

                                                      case 3:
                                                        for (; f < 32;) {
                                                          if (0 === o) break e;
                                                          o--, h += n[s++] << f, f += 8;
                                                        }

                                                        r.head && (r.head.time = h), 512 & r.flags && (E[0] = 255 & h, E[1] = h >>> 8 & 255, E[2] = h >>> 16 & 255, E[3] = h >>> 24 & 255, r.check = B(r.check, E, 4, 0)), f = h = 0, r.mode = 4;

                                                      case 4:
                                                        for (; f < 16;) {
                                                          if (0 === o) break e;
                                                          o--, h += n[s++] << f, f += 8;
                                                        }

                                                        r.head && (r.head.xflags = 255 & h, r.head.os = h >> 8), 512 & r.flags && (E[0] = 255 & h, E[1] = h >>> 8 & 255, r.check = B(r.check, E, 2, 0)), f = h = 0, r.mode = 5;

                                                      case 5:
                                                        if (1024 & r.flags) {
                                                          for (; f < 16;) {
                                                            if (0 === o) break e;
                                                            o--, h += n[s++] << f, f += 8;
                                                          }

                                                          r.length = h, r.head && (r.head.extra_len = h), 512 & r.flags && (E[0] = 255 & h, E[1] = h >>> 8 & 255, r.check = B(r.check, E, 2, 0)), f = h = 0;
                                                        } else r.head && (r.head.extra = null);

                                                        r.mode = 6;

                                                      case 6:
                                                        if (1024 & r.flags && (o < (c = r.length) && (c = o), c && (r.head && (k = r.head.extra_len - r.length, r.head.extra || (r.head.extra = new Array(r.head.extra_len)), I.arraySet(r.head.extra, n, s, c, k)), 512 & r.flags && (r.check = B(r.check, n, c, s)), o -= c, s += c, r.length -= c), r.length)) break e;
                                                        r.length = 0, r.mode = 7;

                                                      case 7:
                                                        if (2048 & r.flags) {
                                                          if (0 === o) break e;

                                                          for (c = 0; k = n[s + c++], r.head && k && r.length < 65536 && (r.head.name += String.fromCharCode(k)), k && c < o;);

                                                          if (512 & r.flags && (r.check = B(r.check, n, c, s)), o -= c, s += c, k) break e;
                                                        } else r.head && (r.head.name = null);

                                                        r.length = 0, r.mode = 8;

                                                      case 8:
                                                        if (4096 & r.flags) {
                                                          if (0 === o) break e;

                                                          for (c = 0; k = n[s + c++], r.head && k && r.length < 65536 && (r.head.comment += String.fromCharCode(k)), k && c < o;);

                                                          if (512 & r.flags && (r.check = B(r.check, n, c, s)), o -= c, s += c, k) break e;
                                                        } else r.head && (r.head.comment = null);

                                                        r.mode = 9;

                                                      case 9:
                                                        if (512 & r.flags) {
                                                          for (; f < 16;) {
                                                            if (0 === o) break e;
                                                            o--, h += n[s++] << f, f += 8;
                                                          }

                                                          if (h !== (65535 & r.check)) {
                                                            e.msg = "header crc mismatch", r.mode = 30;
                                                            break;
                                                          }

                                                          f = h = 0;
                                                        }

                                                        r.head && (r.head.hcrc = r.flags >> 9 & 1, r.head.done = !0), e.adler = r.check = 0, r.mode = 12;
                                                        break;

                                                      case 10:
                                                        for (; f < 32;) {
                                                          if (0 === o) break e;
                                                          o--, h += n[s++] << f, f += 8;
                                                        }

                                                        e.adler = r.check = L(h), f = h = 0, r.mode = 11;

                                                      case 11:
                                                        if (0 === r.havedict) return e.next_out = a, e.avail_out = u, e.next_in = s, e.avail_in = o, r.hold = h, r.bits = f, 2;
                                                        e.adler = r.check = 1, r.mode = 12;

                                                      case 12:
                                                        if (5 === t || 6 === t) break e;

                                                      case 13:
                                                        if (r.last) {
                                                          h >>>= 7 & f, f -= 7 & f, r.mode = 27;
                                                          break;
                                                        }

                                                        for (; f < 3;) {
                                                          if (0 === o) break e;
                                                          o--, h += n[s++] << f, f += 8;
                                                        }

                                                        switch (r.last = 1 & h, f -= 1, 3 & (h >>>= 1)) {
                                                          case 0:
                                                            r.mode = 14;
                                                            break;

                                                          case 1:
                                                            if (j(r), r.mode = 20, 6 !== t) break;
                                                            h >>>= 2, f -= 2;
                                                            break e;

                                                          case 2:
                                                            r.mode = 17;
                                                            break;

                                                          case 3:
                                                            e.msg = "invalid block type", r.mode = 30;
                                                        }

                                                        h >>>= 2, f -= 2;
                                                        break;

                                                      case 14:
                                                        for (h >>>= 7 & f, f -= 7 & f; f < 32;) {
                                                          if (0 === o) break e;
                                                          o--, h += n[s++] << f, f += 8;
                                                        }

                                                        if ((65535 & h) != (h >>> 16 ^ 65535)) {
                                                          e.msg = "invalid stored block lengths", r.mode = 30;
                                                          break;
                                                        }

                                                        if (r.length = 65535 & h, f = h = 0, r.mode = 15, 6 === t) break e;

                                                      case 15:
                                                        r.mode = 16;

                                                      case 16:
                                                        if (c = r.length) {
                                                          if (o < c && (c = o), u < c && (c = u), 0 === c) break e;
                                                          I.arraySet(i, n, s, c, a), o -= c, s += c, u -= c, a += c, r.length -= c;
                                                          break;
                                                        }

                                                        r.mode = 12;
                                                        break;

                                                      case 17:
                                                        for (; f < 14;) {
                                                          if (0 === o) break e;
                                                          o--, h += n[s++] << f, f += 8;
                                                        }

                                                        if (r.nlen = 257 + (31 & h), h >>>= 5, f -= 5, r.ndist = 1 + (31 & h), h >>>= 5, f -= 5, r.ncode = 4 + (15 & h), h >>>= 4, f -= 4, 286 < r.nlen || 30 < r.ndist) {
                                                          e.msg = "too many length or distance symbols", r.mode = 30;
                                                          break;
                                                        }

                                                        r.have = 0, r.mode = 18;

                                                      case 18:
                                                        for (; r.have < r.ncode;) {
                                                          for (; f < 3;) {
                                                            if (0 === o) break e;
                                                            o--, h += n[s++] << f, f += 8;
                                                          }

                                                          r.lens[A[r.have++]] = 7 & h, h >>>= 3, f -= 3;
                                                        }

                                                        for (; r.have < 19;) r.lens[A[r.have++]] = 0;

                                                        if (r.lencode = r.lendyn, r.lenbits = 7, S = {
                                                          bits: r.lenbits
                                                        }, x = T(0, r.lens, 0, 19, r.lencode, 0, r.work, S), r.lenbits = S.bits, x) {
                                                          e.msg = "invalid code lengths set", r.mode = 30;
                                                          break;
                                                        }

                                                        r.have = 0, r.mode = 19;

                                                      case 19:
                                                        for (; r.have < r.nlen + r.ndist;) {
                                                          for (; g = (C = r.lencode[h & (1 << r.lenbits) - 1]) >>> 16 & 255, v = 65535 & C, !((_ = C >>> 24) <= f);) {
                                                            if (0 === o) break e;
                                                            o--, h += n[s++] << f, f += 8;
                                                          }

                                                          if (v < 16) h >>>= _, f -= _, r.lens[r.have++] = v;else {
                                                            if (16 === v) {
                                                              for (z = _ + 2; f < z;) {
                                                                if (0 === o) break e;
                                                                o--, h += n[s++] << f, f += 8;
                                                              }

                                                              if (h >>>= _, f -= _, 0 === r.have) {
                                                                e.msg = "invalid bit length repeat", r.mode = 30;
                                                                break;
                                                              }

                                                              k = r.lens[r.have - 1], c = 3 + (3 & h), h >>>= 2, f -= 2;
                                                            } else if (17 === v) {
                                                              for (z = _ + 3; f < z;) {
                                                                if (0 === o) break e;
                                                                o--, h += n[s++] << f, f += 8;
                                                              }

                                                              f -= _, k = 0, c = 3 + (7 & (h >>>= _)), h >>>= 3, f -= 3;
                                                            } else {
                                                              for (z = _ + 7; f < z;) {
                                                                if (0 === o) break e;
                                                                o--, h += n[s++] << f, f += 8;
                                                              }

                                                              f -= _, k = 0, c = 11 + (127 & (h >>>= _)), h >>>= 7, f -= 7;
                                                            }

                                                            if (r.have + c > r.nlen + r.ndist) {
                                                              e.msg = "invalid bit length repeat", r.mode = 30;
                                                              break;
                                                            }

                                                            for (; c--;) r.lens[r.have++] = k;
                                                          }
                                                        }

                                                        if (30 === r.mode) break;

                                                        if (0 === r.lens[256]) {
                                                          e.msg = "invalid code -- missing end-of-block", r.mode = 30;
                                                          break;
                                                        }

                                                        if (r.lenbits = 9, S = {
                                                          bits: r.lenbits
                                                        }, x = T(D, r.lens, 0, r.nlen, r.lencode, 0, r.work, S), r.lenbits = S.bits, x) {
                                                          e.msg = "invalid literal/lengths set", r.mode = 30;
                                                          break;
                                                        }

                                                        if (r.distbits = 6, r.distcode = r.distdyn, S = {
                                                          bits: r.distbits
                                                        }, x = T(F, r.lens, r.nlen, r.ndist, r.distcode, 0, r.work, S), r.distbits = S.bits, x) {
                                                          e.msg = "invalid distances set", r.mode = 30;
                                                          break;
                                                        }

                                                        if (r.mode = 20, 6 === t) break e;

                                                      case 20:
                                                        r.mode = 21;

                                                      case 21:
                                                        if (6 <= o && 258 <= u) {
                                                          e.next_out = a, e.avail_out = u, e.next_in = s, e.avail_in = o, r.hold = h, r.bits = f, R(e, d), a = e.next_out, i = e.output, u = e.avail_out, s = e.next_in, n = e.input, o = e.avail_in, h = r.hold, f = r.bits, 12 === r.mode && (r.back = -1);
                                                          break;
                                                        }

                                                        for (r.back = 0; g = (C = r.lencode[h & (1 << r.lenbits) - 1]) >>> 16 & 255, v = 65535 & C, !((_ = C >>> 24) <= f);) {
                                                          if (0 === o) break e;
                                                          o--, h += n[s++] << f, f += 8;
                                                        }

                                                        if (g && 0 == (240 & g)) {
                                                          for (b = _, w = g, y = v; g = (C = r.lencode[y + ((h & (1 << b + w) - 1) >> b)]) >>> 16 & 255, v = 65535 & C, !(b + (_ = C >>> 24) <= f);) {
                                                            if (0 === o) break e;
                                                            o--, h += n[s++] << f, f += 8;
                                                          }

                                                          h >>>= b, f -= b, r.back += b;
                                                        }

                                                        if (h >>>= _, f -= _, r.back += _, r.length = v, 0 === g) {
                                                          r.mode = 26;
                                                          break;
                                                        }

                                                        if (32 & g) {
                                                          r.back = -1, r.mode = 12;
                                                          break;
                                                        }

                                                        if (64 & g) {
                                                          e.msg = "invalid literal/length code", r.mode = 30;
                                                          break;
                                                        }

                                                        r.extra = 15 & g, r.mode = 22;

                                                      case 22:
                                                        if (r.extra) {
                                                          for (z = r.extra; f < z;) {
                                                            if (0 === o) break e;
                                                            o--, h += n[s++] << f, f += 8;
                                                          }

                                                          r.length += h & (1 << r.extra) - 1, h >>>= r.extra, f -= r.extra, r.back += r.extra;
                                                        }

                                                        r.was = r.length, r.mode = 23;

                                                      case 23:
                                                        for (; g = (C = r.distcode[h & (1 << r.distbits) - 1]) >>> 16 & 255, v = 65535 & C, !((_ = C >>> 24) <= f);) {
                                                          if (0 === o) break e;
                                                          o--, h += n[s++] << f, f += 8;
                                                        }

                                                        if (0 == (240 & g)) {
                                                          for (b = _, w = g, y = v; g = (C = r.distcode[y + ((h & (1 << b + w) - 1) >> b)]) >>> 16 & 255, v = 65535 & C, !(b + (_ = C >>> 24) <= f);) {
                                                            if (0 === o) break e;
                                                            o--, h += n[s++] << f, f += 8;
                                                          }

                                                          h >>>= b, f -= b, r.back += b;
                                                        }

                                                        if (h >>>= _, f -= _, r.back += _, 64 & g) {
                                                          e.msg = "invalid distance code", r.mode = 30;
                                                          break;
                                                        }

                                                        r.offset = v, r.extra = 15 & g, r.mode = 24;

                                                      case 24:
                                                        if (r.extra) {
                                                          for (z = r.extra; f < z;) {
                                                            if (0 === o) break e;
                                                            o--, h += n[s++] << f, f += 8;
                                                          }

                                                          r.offset += h & (1 << r.extra) - 1, h >>>= r.extra, f -= r.extra, r.back += r.extra;
                                                        }

                                                        if (r.offset > r.dmax) {
                                                          e.msg = "invalid distance too far back", r.mode = 30;
                                                          break;
                                                        }

                                                        r.mode = 25;

                                                      case 25:
                                                        if (0 === u) break e;

                                                        if (c = d - u, r.offset > c) {
                                                          if ((c = r.offset - c) > r.whave && r.sane) {
                                                            e.msg = "invalid distance too far back", r.mode = 30;
                                                            break;
                                                          }

                                                          p = c > r.wnext ? (c -= r.wnext, r.wsize - c) : r.wnext - c, c > r.length && (c = r.length), m = r.window;
                                                        } else m = i, p = a - r.offset, c = r.length;

                                                        for (u < c && (c = u), u -= c, r.length -= c; i[a++] = m[p++], --c;);

                                                        0 === r.length && (r.mode = 21);
                                                        break;

                                                      case 26:
                                                        if (0 === u) break e;
                                                        i[a++] = r.length, u--, r.mode = 21;
                                                        break;

                                                      case 27:
                                                        if (r.wrap) {
                                                          for (; f < 32;) {
                                                            if (0 === o) break e;
                                                            o--, h |= n[s++] << f, f += 8;
                                                          }

                                                          if (d -= u, e.total_out += d, r.total += d, d && (e.adler = r.check = r.flags ? B(r.check, i, d, a - d) : O(r.check, i, d, a - d)), d = u, (r.flags ? h : L(h)) !== r.check) {
                                                            e.msg = "incorrect data check", r.mode = 30;
                                                            break;
                                                          }

                                                          f = h = 0;
                                                        }

                                                        r.mode = 28;

                                                      case 28:
                                                        if (r.wrap && r.flags) {
                                                          for (; f < 32;) {
                                                            if (0 === o) break e;
                                                            o--, h += n[s++] << f, f += 8;
                                                          }

                                                          if (h !== (4294967295 & r.total)) {
                                                            e.msg = "incorrect length check", r.mode = 30;
                                                            break;
                                                          }

                                                          f = h = 0;
                                                        }

                                                        r.mode = 29;

                                                      case 29:
                                                        x = 1;
                                                        break e;

                                                      case 30:
                                                        x = -3;
                                                        break e;

                                                      case 31:
                                                        return -4;

                                                      case 32:
                                                      default:
                                                        return U;
                                                    }

                                                    return e.next_out = a, e.avail_out = u, e.next_in = s, e.avail_in = o, r.hold = h, r.bits = f, (r.wsize || d !== e.avail_out && r.mode < 30 && (r.mode < 27 || 4 !== t)) && Z(e, e.output, e.next_out, d - e.avail_out) ? (r.mode = 31, -4) : (l -= e.avail_in, d -= e.avail_out, e.total_in += l, e.total_out += d, r.total += d, r.wrap && d && (e.adler = r.check = r.flags ? B(r.check, i, d, e.next_out - d) : O(r.check, i, d, e.next_out - d)), e.data_type = r.bits + (r.last ? 64 : 0) + (12 === r.mode ? 128 : 0) + (20 === r.mode || 15 === r.mode ? 256 : 0), (0 == l && 0 === d || 4 === t) && x === N && (x = -5), x);
                                                  }, r.inflateEnd = function (e) {
                                                    if (!e || !e.state) return U;
                                                    var t = e.state;
                                                    return t.window && (t.window = null), e.state = null, N;
                                                  }, r.inflateGetHeader = function (e, t) {
                                                    var r;
                                                    return e && e.state ? 0 == (2 & (r = e.state).wrap) ? U : ((r.head = t).done = !1, N) : U;
                                                  }, r.inflateSetDictionary = function (e, t) {
                                                    var r,
                                                        n = t.length;
                                                    return e && e.state ? 0 !== (r = e.state).wrap && 11 !== r.mode ? U : 11 === r.mode && O(1, t, n, 0) !== r.check ? -3 : Z(e, t, n, n) ? (r.mode = 31, -4) : (r.havedict = 1, N) : U;
                                                  }, r.inflateInfo = "pako inflate (from Nodeca project)";
                                                }, {
                                                  "../utils/common": 41,
                                                  "./adler32": 43,
                                                  "./crc32": 45,
                                                  "./inffast": 48,
                                                  "./inftrees": 50
                                                }],
                                                50: [function (e, t, r) {

                                                  var D = e("../utils/common"),
                                                      F = [3, 4, 5, 6, 7, 8, 9, 10, 11, 13, 15, 17, 19, 23, 27, 31, 35, 43, 51, 59, 67, 83, 99, 115, 131, 163, 195, 227, 258, 0, 0],
                                                      N = [16, 16, 16, 16, 16, 16, 16, 16, 17, 17, 17, 17, 18, 18, 18, 18, 19, 19, 19, 19, 20, 20, 20, 20, 21, 21, 21, 21, 16, 72, 78],
                                                      U = [1, 2, 3, 4, 5, 7, 9, 13, 17, 25, 33, 49, 65, 97, 129, 193, 257, 385, 513, 769, 1025, 1537, 2049, 3073, 4097, 6145, 8193, 12289, 16385, 24577, 0, 0],
                                                      P = [16, 16, 16, 16, 17, 17, 18, 18, 19, 19, 20, 20, 21, 21, 22, 22, 23, 23, 24, 24, 25, 25, 26, 26, 27, 27, 28, 28, 29, 29, 64, 64];

                                                  t.exports = function (e, t, r, n, i, s, a, o) {
                                                    var u,
                                                        h,
                                                        f,
                                                        l,
                                                        d,
                                                        c,
                                                        p,
                                                        m,
                                                        _,
                                                        g = o.bits,
                                                        v = 0,
                                                        b = 0,
                                                        w = 0,
                                                        y = 0,
                                                        k = 0,
                                                        x = 0,
                                                        S = 0,
                                                        z = 0,
                                                        C = 0,
                                                        E = 0,
                                                        A = null,
                                                        I = 0,
                                                        O = new D.Buf16(16),
                                                        B = new D.Buf16(16),
                                                        R = null,
                                                        T = 0;

                                                    for (v = 0; v <= 15; v++) O[v] = 0;

                                                    for (b = 0; b < n; b++) O[t[r + b]]++;

                                                    for (k = g, y = 15; 1 <= y && 0 === O[y]; y--);

                                                    if (y < k && (k = y), 0 === y) return i[s++] = 20971520, i[s++] = 20971520, o.bits = 1, 0;

                                                    for (w = 1; w < y && 0 === O[w]; w++);

                                                    for (k < w && (k = w), v = z = 1; v <= 15; v++) if (z <<= 1, (z -= O[v]) < 0) return -1;

                                                    if (0 < z && (0 === e || 1 !== y)) return -1;

                                                    for (B[1] = 0, v = 1; v < 15; v++) B[v + 1] = B[v] + O[v];

                                                    for (b = 0; b < n; b++) 0 !== t[r + b] && (a[B[t[r + b]]++] = b);

                                                    if (c = 0 === e ? (A = R = a, 19) : 1 === e ? (A = F, I -= 257, R = N, T -= 257, 256) : (A = U, R = P, -1), v = w, d = s, S = b = E = 0, f = -1, l = (C = 1 << (x = k)) - 1, 1 === e && 852 < C || 2 === e && 592 < C) return 1;

                                                    for (;;) {
                                                      for (p = v - S, _ = a[b] < c ? (m = 0, a[b]) : a[b] > c ? (m = R[T + a[b]], A[I + a[b]]) : (m = 96, 0), u = 1 << v - S, w = h = 1 << x; i[d + (E >> S) + (h -= u)] = p << 24 | m << 16 | _ | 0, 0 !== h;);

                                                      for (u = 1 << v - 1; E & u;) u >>= 1;

                                                      if (0 !== u ? (E &= u - 1, E += u) : E = 0, b++, 0 == --O[v]) {
                                                        if (v === y) break;
                                                        v = t[r + a[b]];
                                                      }

                                                      if (k < v && (E & l) !== f) {
                                                        for (0 === S && (S = k), d += w, z = 1 << (x = v - S); x + S < y && !((z -= O[x + S]) <= 0);) x++, z <<= 1;

                                                        if (C += 1 << x, 1 === e && 852 < C || 2 === e && 592 < C) return 1;
                                                        i[f = E & l] = k << 24 | x << 16 | d - s | 0;
                                                      }
                                                    }

                                                    return 0 !== E && (i[d + E] = v - S << 24 | 64 << 16 | 0), o.bits = k, 0;
                                                  };
                                                }, {
                                                  "../utils/common": 41
                                                }],
                                                51: [function (e, t, r) {

                                                  t.exports = {
                                                    2: "need dictionary",
                                                    1: "stream end",
                                                    0: "",
                                                    "-1": "file error",
                                                    "-2": "stream error",
                                                    "-3": "data error",
                                                    "-4": "insufficient memory",
                                                    "-5": "buffer error",
                                                    "-6": "incompatible version"
                                                  };
                                                }, {}],
                                                52: [function (e, t, r) {

                                                  var o = e("../utils/common");

                                                  function n(e) {
                                                    for (var t = e.length; 0 <= --t;) e[t] = 0;
                                                  }

                                                  var _ = 15,
                                                      i = 16,
                                                      u = [0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 5, 5, 5, 5, 0],
                                                      h = [0, 0, 0, 0, 1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9, 10, 10, 11, 11, 12, 12, 13, 13],
                                                      a = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 3, 7],
                                                      f = [16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15],
                                                      l = new Array(576);
                                                  n(l);
                                                  var d = new Array(60);
                                                  n(d);
                                                  var c = new Array(512);
                                                  n(c);
                                                  var p = new Array(256);
                                                  n(p);
                                                  var m = new Array(29);
                                                  n(m);
                                                  var g,
                                                      v,
                                                      b,
                                                      w = new Array(30);

                                                  function y(e, t, r, n, i) {
                                                    this.static_tree = e, this.extra_bits = t, this.extra_base = r, this.elems = n, this.max_length = i, this.has_stree = e && e.length;
                                                  }

                                                  function s(e, t) {
                                                    this.dyn_tree = e, this.max_code = 0, this.stat_desc = t;
                                                  }

                                                  function k(e) {
                                                    return e < 256 ? c[e] : c[256 + (e >>> 7)];
                                                  }

                                                  function x(e, t) {
                                                    e.pending_buf[e.pending++] = 255 & t, e.pending_buf[e.pending++] = t >>> 8 & 255;
                                                  }

                                                  function S(e, t, r) {
                                                    e.bi_valid > i - r ? (e.bi_buf |= t << e.bi_valid & 65535, x(e, e.bi_buf), e.bi_buf = t >> i - e.bi_valid, e.bi_valid += r - i) : (e.bi_buf |= t << e.bi_valid & 65535, e.bi_valid += r);
                                                  }

                                                  function z(e, t, r) {
                                                    S(e, r[2 * t], r[2 * t + 1]);
                                                  }

                                                  function C(e, t) {
                                                    for (var r = 0; r |= 1 & e, e >>>= 1, r <<= 1, 0 < --t;);

                                                    return r >>> 1;
                                                  }

                                                  function E(e, t, r) {
                                                    var n,
                                                        i,
                                                        s = new Array(_ + 1),
                                                        a = 0;

                                                    for (n = 1; n <= _; n++) s[n] = a = a + r[n - 1] << 1;

                                                    for (i = 0; i <= t; i++) {
                                                      var o = e[2 * i + 1];
                                                      0 !== o && (e[2 * i] = C(s[o]++, o));
                                                    }
                                                  }

                                                  function A(e) {
                                                    var t;

                                                    for (t = 0; t < 286; t++) e.dyn_ltree[2 * t] = 0;

                                                    for (t = 0; t < 30; t++) e.dyn_dtree[2 * t] = 0;

                                                    for (t = 0; t < 19; t++) e.bl_tree[2 * t] = 0;

                                                    e.dyn_ltree[512] = 1, e.opt_len = e.static_len = 0, e.last_lit = e.matches = 0;
                                                  }

                                                  function I(e) {
                                                    8 < e.bi_valid ? x(e, e.bi_buf) : 0 < e.bi_valid && (e.pending_buf[e.pending++] = e.bi_buf), e.bi_buf = 0, e.bi_valid = 0;
                                                  }

                                                  function O(e, t, r, n) {
                                                    var i = 2 * t,
                                                        s = 2 * r;
                                                    return e[i] < e[s] || e[i] === e[s] && n[t] <= n[r];
                                                  }

                                                  function B(e, t, r) {
                                                    for (var n = e.heap[r], i = r << 1; i <= e.heap_len && (i < e.heap_len && O(t, e.heap[i + 1], e.heap[i], e.depth) && i++, !O(t, n, e.heap[i], e.depth));) e.heap[r] = e.heap[i], r = i, i <<= 1;

                                                    e.heap[r] = n;
                                                  }

                                                  function R(e, t, r) {
                                                    var n,
                                                        i,
                                                        s,
                                                        a,
                                                        o = 0;
                                                    if (0 !== e.last_lit) for (; n = e.pending_buf[e.d_buf + 2 * o] << 8 | e.pending_buf[e.d_buf + 2 * o + 1], i = e.pending_buf[e.l_buf + o], o++, 0 === n ? z(e, i, t) : (z(e, (s = p[i]) + 256 + 1, t), 0 !== (a = u[s]) && S(e, i -= m[s], a), z(e, s = k(--n), r), 0 !== (a = h[s]) && S(e, n -= w[s], a)), o < e.last_lit;);
                                                    z(e, 256, t);
                                                  }

                                                  function T(e, t) {
                                                    var r,
                                                        n,
                                                        i,
                                                        s = t.dyn_tree,
                                                        a = t.stat_desc.static_tree,
                                                        o = t.stat_desc.has_stree,
                                                        u = t.stat_desc.elems,
                                                        h = -1;

                                                    for (e.heap_len = 0, e.heap_max = 573, r = 0; r < u; r++) 0 !== s[2 * r] ? (e.heap[++e.heap_len] = h = r, e.depth[r] = 0) : s[2 * r + 1] = 0;

                                                    for (; e.heap_len < 2;) s[2 * (i = e.heap[++e.heap_len] = h < 2 ? ++h : 0)] = 1, e.depth[i] = 0, e.opt_len--, o && (e.static_len -= a[2 * i + 1]);

                                                    for (t.max_code = h, r = e.heap_len >> 1; 1 <= r; r--) B(e, s, r);

                                                    for (i = u; r = e.heap[1], e.heap[1] = e.heap[e.heap_len--], B(e, s, 1), n = e.heap[1], e.heap[--e.heap_max] = r, e.heap[--e.heap_max] = n, s[2 * i] = s[2 * r] + s[2 * n], e.depth[i] = (e.depth[r] >= e.depth[n] ? e.depth[r] : e.depth[n]) + 1, s[2 * r + 1] = s[2 * n + 1] = i, e.heap[1] = i++, B(e, s, 1), 2 <= e.heap_len;);

                                                    e.heap[--e.heap_max] = e.heap[1], function (e, t) {
                                                      var r,
                                                          n,
                                                          i,
                                                          s,
                                                          a,
                                                          o,
                                                          u = t.dyn_tree,
                                                          h = t.max_code,
                                                          f = t.stat_desc.static_tree,
                                                          l = t.stat_desc.has_stree,
                                                          d = t.stat_desc.extra_bits,
                                                          c = t.stat_desc.extra_base,
                                                          p = t.stat_desc.max_length,
                                                          m = 0;

                                                      for (s = 0; s <= _; s++) e.bl_count[s] = 0;

                                                      for (u[2 * e.heap[e.heap_max] + 1] = 0, r = e.heap_max + 1; r < 573; r++) p < (s = u[2 * u[2 * (n = e.heap[r]) + 1] + 1] + 1) && (s = p, m++), u[2 * n + 1] = s, h < n || (e.bl_count[s]++, a = 0, c <= n && (a = d[n - c]), o = u[2 * n], e.opt_len += o * (s + a), l && (e.static_len += o * (f[2 * n + 1] + a)));

                                                      if (0 !== m) {
                                                        do {
                                                          for (s = p - 1; 0 === e.bl_count[s];) s--;

                                                          e.bl_count[s]--, e.bl_count[s + 1] += 2, e.bl_count[p]--, m -= 2;
                                                        } while (0 < m);

                                                        for (s = p; 0 !== s; s--) for (n = e.bl_count[s]; 0 !== n;) h < (i = e.heap[--r]) || (u[2 * i + 1] !== s && (e.opt_len += (s - u[2 * i + 1]) * u[2 * i], u[2 * i + 1] = s), n--);
                                                      }
                                                    }(e, t), E(s, h, e.bl_count);
                                                  }

                                                  function D(e, t, r) {
                                                    var n,
                                                        i,
                                                        s = -1,
                                                        a = t[1],
                                                        o = 0,
                                                        u = 7,
                                                        h = 4;

                                                    for (0 === a && (u = 138, h = 3), t[2 * (r + 1) + 1] = 65535, n = 0; n <= r; n++) i = a, a = t[2 * (n + 1) + 1], ++o < u && i === a || (o < h ? e.bl_tree[2 * i] += o : 0 !== i ? (i !== s && e.bl_tree[2 * i]++, e.bl_tree[32]++) : o <= 10 ? e.bl_tree[34]++ : e.bl_tree[36]++, s = i, h = (o = 0) === a ? (u = 138, 3) : i === a ? (u = 6, 3) : (u = 7, 4));
                                                  }

                                                  function F(e, t, r) {
                                                    var n,
                                                        i,
                                                        s = -1,
                                                        a = t[1],
                                                        o = 0,
                                                        u = 7,
                                                        h = 4;

                                                    for (0 === a && (u = 138, h = 3), n = 0; n <= r; n++) if (i = a, a = t[2 * (n + 1) + 1], !(++o < u && i === a)) {
                                                      if (o < h) for (; z(e, i, e.bl_tree), 0 != --o;);else 0 !== i ? (i !== s && (z(e, i, e.bl_tree), o--), z(e, 16, e.bl_tree), S(e, o - 3, 2)) : o <= 10 ? (z(e, 17, e.bl_tree), S(e, o - 3, 3)) : (z(e, 18, e.bl_tree), S(e, o - 11, 7));
                                                      s = i, h = (o = 0) === a ? (u = 138, 3) : i === a ? (u = 6, 3) : (u = 7, 4);
                                                    }
                                                  }

                                                  n(w);
                                                  var N = !1;

                                                  function U(e, t, r, n) {
                                                    var i, s, a;
                                                    S(e, 0 + (n ? 1 : 0), 3), s = t, a = r, I(i = e), x(i, a), x(i, ~a), o.arraySet(i.pending_buf, i.window, s, a, i.pending), i.pending += a;
                                                  }

                                                  r._tr_init = function (e) {
                                                    N || (function () {
                                                      var e,
                                                          t,
                                                          r,
                                                          n,
                                                          i,
                                                          s = new Array(_ + 1);

                                                      for (n = r = 0; n < 28; n++) for (m[n] = r, e = 0; e < 1 << u[n]; e++) p[r++] = n;

                                                      for (p[r - 1] = n, n = i = 0; n < 16; n++) for (w[n] = i, e = 0; e < 1 << h[n]; e++) c[i++] = n;

                                                      for (i >>= 7; n < 30; n++) for (w[n] = i << 7, e = 0; e < 1 << h[n] - 7; e++) c[256 + i++] = n;

                                                      for (t = 0; t <= _; t++) s[t] = 0;

                                                      for (e = 0; e <= 143;) l[2 * e + 1] = 8, e++, s[8]++;

                                                      for (; e <= 255;) l[2 * e + 1] = 9, e++, s[9]++;

                                                      for (; e <= 279;) l[2 * e + 1] = 7, e++, s[7]++;

                                                      for (; e <= 287;) l[2 * e + 1] = 8, e++, s[8]++;

                                                      for (E(l, 287, s), e = 0; e < 30; e++) d[2 * e + 1] = 5, d[2 * e] = C(e, 5);

                                                      g = new y(l, u, 257, 286, _), v = new y(d, h, 0, 30, _), b = new y(new Array(0), a, 0, 19, 7);
                                                    }(), N = !0), e.l_desc = new s(e.dyn_ltree, g), e.d_desc = new s(e.dyn_dtree, v), e.bl_desc = new s(e.bl_tree, b), e.bi_buf = 0, e.bi_valid = 0, A(e);
                                                  }, r._tr_stored_block = U, r._tr_flush_block = function (e, t, r, n) {
                                                    var i,
                                                        s,
                                                        a = 0;
                                                    0 < e.level ? (2 === e.strm.data_type && (e.strm.data_type = function (e) {
                                                      var t,
                                                          r = 4093624447;

                                                      for (t = 0; t <= 31; t++, r >>>= 1) if (1 & r && 0 !== e.dyn_ltree[2 * t]) return 0;

                                                      if (0 !== e.dyn_ltree[18] || 0 !== e.dyn_ltree[20] || 0 !== e.dyn_ltree[26]) return 1;

                                                      for (t = 32; t < 256; t++) if (0 !== e.dyn_ltree[2 * t]) return 1;

                                                      return 0;
                                                    }(e)), T(e, e.l_desc), T(e, e.d_desc), a = function (e) {
                                                      var t;

                                                      for (D(e, e.dyn_ltree, e.l_desc.max_code), D(e, e.dyn_dtree, e.d_desc.max_code), T(e, e.bl_desc), t = 18; 3 <= t && 0 === e.bl_tree[2 * f[t] + 1]; t--);

                                                      return e.opt_len += 3 * (t + 1) + 5 + 5 + 4, t;
                                                    }(e), i = e.opt_len + 3 + 7 >>> 3, (s = e.static_len + 3 + 7 >>> 3) <= i && (i = s)) : i = s = r + 5, r + 4 <= i && -1 !== t ? U(e, t, r, n) : 4 === e.strategy || s === i ? (S(e, 2 + (n ? 1 : 0), 3), R(e, l, d)) : (S(e, 4 + (n ? 1 : 0), 3), function (e, t, r, n) {
                                                      var i;

                                                      for (S(e, t - 257, 5), S(e, r - 1, 5), S(e, n - 4, 4), i = 0; i < n; i++) S(e, e.bl_tree[2 * f[i] + 1], 3);

                                                      F(e, e.dyn_ltree, t - 1), F(e, e.dyn_dtree, r - 1);
                                                    }(e, e.l_desc.max_code + 1, e.d_desc.max_code + 1, a + 1), R(e, e.dyn_ltree, e.dyn_dtree)), A(e), n && I(e);
                                                  }, r._tr_tally = function (e, t, r) {
                                                    return e.pending_buf[e.d_buf + 2 * e.last_lit] = t >>> 8 & 255, e.pending_buf[e.d_buf + 2 * e.last_lit + 1] = 255 & t, e.pending_buf[e.l_buf + e.last_lit] = 255 & r, e.last_lit++, 0 === t ? e.dyn_ltree[2 * r]++ : (e.matches++, t--, e.dyn_ltree[2 * (p[r] + 256 + 1)]++, e.dyn_dtree[2 * k(t)]++), e.last_lit === e.lit_bufsize - 1;
                                                  }, r._tr_align = function (e) {
                                                    var t;
                                                    S(e, 2, 3), z(e, 256, l), 16 === (t = e).bi_valid ? (x(t, t.bi_buf), t.bi_buf = 0, t.bi_valid = 0) : 8 <= t.bi_valid && (t.pending_buf[t.pending++] = 255 & t.bi_buf, t.bi_buf >>= 8, t.bi_valid -= 8);
                                                  };
                                                }, {
                                                  "../utils/common": 41
                                                }],
                                                53: [function (e, t, r) {

                                                  t.exports = function () {
                                                    this.input = null, this.next_in = 0, this.avail_in = 0, this.total_in = 0, this.output = null, this.next_out = 0, this.avail_out = 0, this.total_out = 0, this.msg = "", this.state = null, this.data_type = 2, this.adler = 0;
                                                  };
                                                }, {}],
                                                54: [function (e, t, r) {

                                                  t.exports = "function" == typeof setImmediate ? setImmediate : function () {
                                                    var e = [].slice.apply(arguments);
                                                    e.splice(1, 0, 0), setTimeout.apply(null, e);
                                                  };
                                                }, {}]
                                              }, {}, [10])(10);
                                            });
                                          }).call(this, void 0 !== r ? r : "undefined" != typeof self ? self : "undefined" != typeof window ? window : {});
                                        }, {}]
                                      }, {}, [1])(1);
                                    });
                                  }).call(this, void 0 !== r ? r : "undefined" != typeof self ? self : "undefined" != typeof window ? window : {});
                                }, {}]
                              }, {}, [1])(1);
                            });
                          }).call(this, void 0 !== r ? r : "undefined" != typeof self ? self : "undefined" != typeof window ? window : {});
                        }, {}]
                      }, {}, [1])(1);
                    });
                  }).call(this, "undefined" != typeof global ? global : "undefined" != typeof self ? self : "undefined" != typeof window ? window : {});
                }, {}]
              }, {}, [1])(1);
            });
          }).call(this, typeof commonjsGlobal$1 !== "undefined" ? commonjsGlobal$1 : typeof self !== "undefined" ? self : typeof window !== "undefined" ? window : {});
        }, {}]
      }, {}, [1])(1);
    });
  });

  var browser = createCommonjsModule$1(function (module, exports) {

    var getGlobal = function () {
      // the only reliable means to get the global object is
      // `Function('return this')()`
      // However, this causes CSP violations in Chrome apps.
      if (typeof self !== 'undefined') {
        return self;
      }

      if (typeof window !== 'undefined') {
        return window;
      }

      if (typeof global !== 'undefined') {
        return global;
      }

      throw new Error('unable to locate global object');
    };

    var global = getGlobal();
    module.exports = exports = global.fetch; // Needed for TypeScript and Webpack.

    if (global.fetch) {
      exports.default = global.fetch.bind(global);
    }

    exports.Headers = global.Headers;
    exports.Request = global.Request;
    exports.Response = global.Response;
  });

  var fetchArrayBuffer = async function fetchArrayBufferBrowser(url) {
    const result = await browser(url);
    return result.arrayBuffer();
  };

  const loadingPromises$1 = {};

  var loadCommercials = async function loadCommercials(options = {}) {
    const {
      url = 'https://couch.cheminfo.org/cheminfo-public/d2eb480198c80275a1d05dd3609414f9/upload/commercials.zip'
    } = options;

    if (!loadingPromises$1[url]) {
      loadingPromises$1[url] = fetchArrayBuffer(url);
    }

    const buffer = await loadingPromises$1[url];
    const jsZip = new jszip();
    let zip = await jsZip.loadAsync(buffer);
    let fileData = await zip.files['commercials.json'].async('string');
    let data = JSON.parse(fileData);
    data.sort((a, b) => a.em - b.em);
    return data;
  };

  var constants = {
    ELECTRON_MASS: 5.4857990907e-4
  };

  /**
   * Define static variable corresponding to the various Kinds of a molecular formula part.
   */

  var Kind = {
    BEGIN: 'begin',
    ATOM: 'atom',
    MULTIPLIER_RANGE: 'multiplierRange',
    ISOTOPE: 'isotope',
    ISOTOPE_RATIO: 'isotopeRatio',
    CHARGE: 'charge',
    SALT: 'salt',
    OPENING_PARENTHESIS: 'openingParenthesis',
    CLOSING_PARENTHESIS: 'closingParenthesis',
    PRE_MULTIPLIER: 'preMultiplier',
    MULTIPLIER: 'multiplier',
    TEXT: 'text',
    ANCHOR: 'anchor',
    COMMENT: 'comment'
  };

  /**
   * Parse a string to extract the charge
   * The charge may be in the form --, +++, +3, -2, 4+, 2-
   * @param {*} charge
   */

  var parseCharge = function parseCharge(charge) {
    charge = charge.replace(/[()]/g, '');
    let chargeNumber = 0;

    if (charge.match(/^[+-]+$/)) {
      for (let i = 0; i < charge.length; i++) {
        if (charge.charAt(i) === '+') chargeNumber++;else chargeNumber--;
      }
    } else if (charge.match(/^[0-9]+[+-]$/)) {
      chargeNumber = Number(charge.charAt(charge.length - 1) + charge.substring(0, charge.length - 1));
    } else {
      chargeNumber = Number(charge);
    }

    return chargeNumber;
  };

  /**
   * Parse a mf to an array of kind / value
   * @param {String} mf
   */


  var parse = function parse(mf) {
    return new MFParser().parse(mf);
  };

  class MFParser {
    parse(mf = '') {
      this.mf = mf;
      this.i = 0;
      this.result = [];
      let lastKind = Kind.BEGIN;

      while (this.i < mf.length) {
        if (this.result.length > 0 && this.result[this.result.length - 1].kind !== Kind.TEXT) {
          lastKind = this.result[this.result.length - 1].kind;
        }

        let char = mf.charAt(this.i);
        let ascii = mf.charCodeAt(this.i);
        let nextAscii = 0;
        if (this.i + 1 < mf.length) nextAscii = mf.charCodeAt(this.i + 1);

        if (ascii > 47 && ascii < 58 || char === '-' && nextAscii > 47 && nextAscii < 58) {
          // a number
          let value = this.getNumber(ascii);

          if (lastKind === Kind.SALT || lastKind === Kind.BEGIN || lastKind === Kind.OPENING_PARENTHESIS) {
            if (value.to) {
              throw new MFError(this.mf, this.i, 'Premultiplier may not contain a -');
            }

            this.result.push({
              kind: Kind.PRE_MULTIPLIER,
              value: value.from
            });
          } else if (lastKind === Kind.ANCHOR) {
            if (value.to) {
              throw new MFError(this.mf, this.i, 'Anchor ID may not contain -');
            }

            this.result[this.result.length - 1].value = value.from;
          } else {
            if (value.to) {
              this.result.push({
                kind: Kind.MULTIPLIER_RANGE,
                value: {
                  from: Math.min(value.from, value.to),
                  to: Math.max(value.from, value.to)
                }
              });
            } else {
              this.result.push({
                kind: Kind.MULTIPLIER,
                value: value.from
              });
            }
          }

          continue;
        } else if (char === '.') {
          // a point
          this.result.push({
            kind: Kind.SALT,
            value: char
          }); // it is not in a number otherwise it would have been taken before
          // it must be in a salt
        } else if (char === '#') {
          // an anchor
          this.result.push({
            kind: Kind.ANCHOR,
            value: 0
          }); // it is not in a number otherwise it would have been taken before
          // it must be in a salt
        } else if (ascii > 64 && ascii < 91) {
          // an uppercase = new atom
          let value = this.getAtom(ascii);
          this.result.push({
            kind: Kind.ATOM,
            value
          });
          continue;
        } else if (ascii > 96 && ascii < 123) {
          // a lowercase
          throw new MFError(this.mf, this.i, 'found a lowercase not following an uppercase');
        } else if (char === '(') {
          let charge = this.getParenthesisCharge(ascii);

          if (charge) {
            this.result.push({
              kind: Kind.CHARGE,
              value: charge
            });
          } else {
            this.result.push({
              kind: Kind.OPENING_PARENTHESIS,
              value: '('
            });
          }
        } else if (char === ')') {
          this.result.push({
            kind: Kind.CLOSING_PARENTHESIS,
            value: ')'
          });
        } else if (char === '[') {
          // defines an isotope
          let isotope = this.getIsotope(ascii);
          this.result.push({
            kind: Kind.ISOTOPE,
            value: isotope
          });
        } else if (char === ']') {
          throw new MFError(this.mf, this.i, 'should never meet an closing bracket not in isotopes');
        } else if (char === '{') {
          // can define an exotic isotopic ratio or mixtures of groups
          let isotopeRatio = this.getCurlyBracketIsotopeRatio(ascii);

          if (lastKind === Kind.ATOM) {
            let lastResult = this.result[this.result.length - 1];
            lastResult.kind = Kind.ISOTOPE_RATIO;
            lastResult.value = {
              atom: lastResult.value,
              ratio: isotopeRatio
            };
          } else {
            throw new MFError(this.mf, this.i, 'isotopic composition has to follow an atom');
          }
        } else if (char === '}') {
          throw new MFError(this.mf, this.i, 'found a unexpected closing curly bracket');
        } else if (char === '+') {
          // charge not in parenthesis
          let charge = this.getNonParenthesisCharge(ascii);
          this.result.push({
            kind: Kind.CHARGE,
            value: charge
          });
        } else if (char === '-') {
          // charge not in parenthesis
          let charge = this.getNonParenthesisCharge(ascii);
          this.result.push({
            kind: Kind.CHARGE,
            value: charge
          });
        } else if (char === '$') {
          // it is a comment after
          this.result.push({
            kind: Kind.COMMENT,
            value: this.mf.substring(this.i + 1)
          });
          break;
        } else {
          this.result.push({
            kind: Kind.TEXT,
            value: char
          });
        }

        this.i++;
      }

      this.checkParenthesis();
      return this.result;
    }

    checkParenthesis() {
      let counter = 0;

      for (let line of this.result) {
        if (line.kind === Kind.OPENING_PARENTHESIS) counter++;
        if (line.kind === Kind.CLOSING_PARENTHESIS) counter--;
      }

      if (counter !== 0) {
        throw new MFError(this.mf, this.i, 'number of opening and closing parenthesis not equal');
      }
    }

    getNumber(ascii) {
      let number = '';
      let previous;

      do {
        previous = ascii;
        number += String.fromCharCode(ascii);
        this.i++;
        ascii = this.mf.charCodeAt(this.i);
      } while (ascii > 47 && ascii < 58 || ascii === 46 || ascii === 45 || ascii === 47); // number . - /
      // we need to deal with the case there is a from / to


      if (previous === 46) this.i--;
      let indexOfDash = number.indexOf('-', 1);

      if (indexOfDash > -1) {
        return {
          from: parseNumberWithDivision(number.substr(0, indexOfDash)),
          to: parseNumberWithDivision(number.substr(indexOfDash + 1))
        };
      }

      return {
        from: parseNumberWithDivision(number)
      };
    }

    getAtom(ascii) {
      let atom = '';

      do {
        atom += String.fromCharCode(ascii);
        this.i++;
        ascii = this.mf.charCodeAt(this.i);
      } while (ascii > 96 && ascii < 123);

      return atom;
    }

    getIsotope(ascii) {
      // [13C]
      let substring = '';

      do {
        substring += String.fromCharCode(ascii);
        this.i++;
        ascii = this.mf.charCodeAt(this.i);
      } while (ascii !== 93 && this.i <= this.mf.length);

      let atom = substring.replace(/[^a-zA-Z]/g, '');
      let isotope = Number(substring.replace(/[^0-9]/g, ''));
      return {
        atom,
        isotope
      };
    }

    getCurlyBracketIsotopeRatio(ascii) {
      let substring = '';
      let first = true;

      do {
        if (!first) {
          substring += String.fromCharCode(ascii);
        } else {
          first = false;
        }

        this.i++;
        ascii = this.mf.charCodeAt(this.i);
      } while (ascii !== 125 && this.i <= this.mf.length); // closing curly bracket


      if (substring.match(/^[0-9,]+$/)) {
        return substring.split(',').map(a => Number(a));
      }

      throw new MFError(this.mf, this.i, 'Curly brackets should contain only number and comma');
    }

    getParenthesisCharge(ascii) {
      let substring = '';
      let begin = this.i;

      do {
        substring += String.fromCharCode(ascii);
        this.i++;
        ascii = this.mf.charCodeAt(this.i);
      } while (ascii !== 41 && this.i <= this.mf.length); // closing parenthesis


      if (substring.match(/^\([0-9+-]+$/)) {
        return parseCharge(substring.substring(1));
      } else {
        this.i = begin;
        return undefined;
      }
    }

    getNonParenthesisCharge(ascii) {
      let substring = '';

      do {
        substring += String.fromCharCode(ascii);
        this.i++;
        ascii = this.mf.charCodeAt(this.i);
      } while (ascii === 43 || ascii === 45 || ascii > 47 && ascii < 58);

      this.i--;
      return parseCharge(substring);
    }

  }

  class MFError extends SyntaxError {
    constructor(mf, i, message) {
      let text = `${message}\n\n${mf}\n${' '.repeat(i)}^`;
      super(text);
    }

  }

  function parseNumberWithDivision(string) {
    if (string.includes('/')) {
      let parts = string.split('/');

      if (parts.length !== 2) {
        throw new TypeError('Can not parse MF with number like: ', string);
      }

      return Number(parts[0]) / Number(parts[1]);
    } else {
      return Number(string);
    }
  }

  /**
   * Defines static variables corresponding to the various formatting possibilities
   */

  var Format = {
    SUBSCRIPT: 'subscript',
    SUPERSCRIPT: 'superscript',
    SUPERIMPOSE: 'superimpose',
    TEXT: 'text'
  };

  var formatCharge = function formatCharge(charge) {
    if (charge === 1) return '+';
    if (charge > 1) return `+${charge}`;
    if (charge < 0) return String(charge);
    return '';
  };

  /**
   * Converts an array of mf elements to an array of formatting information
   * @param {Array<Object>} result of the parse method
   */


  var toDisplay = function convertForDisplay(lines) {
    let results = [];
    let result = {};

    for (let line of lines) {
      switch (line.kind) {
        case Kind.MULTIPLIER:
          if (line.value !== 1) {
            result = {
              kind: Format.SUBSCRIPT,
              value: String(line.value)
            };
            results.push(result);
          }

          break;

        case Kind.MULTIPLIER_RANGE:
          result = {
            kind: Format.SUBSCRIPT,
            value: `${String(line.value.from)}-${line.value.to}`
          };
          results.push(result);
          break;

        case Kind.CHARGE:
          if (result.kind === Format.SUBSCRIPT) {
            result.kind = Format.SUPERIMPOSE;
            result.over = formatCharge(line.value);
            result.under = result.value;
            result.value = undefined;
          } else {
            result = {
              kind: Format.SUPERSCRIPT,
              value: formatCharge(line.value)
            };
            results.push(result);
          }

          break;

        case Kind.ISOTOPE:
          result = {
            kind: Format.SUPERSCRIPT,
            value: line.value.isotope
          };
          results.push(result);
          result = {
            kind: Format.TEXT,
            value: line.value.atom
          };
          results.push(result);
          break;

        case Kind.ISOTOPE_RATIO:
          if (result.kind === Format.TEXT) {
            result.value += line.value.atom;
          } else {
            result = {
              kind: Format.TEXT,
              value: line.value.atom
            };
            results.push(result);
          }

          result = {
            kind: Format.SUPERSCRIPT,
            value: `{${line.value.ratio.join(',')}}`
          };
          results.push(result);
          break;

        case Kind.SALT:
          if (result.kind === Format.TEXT) {
            result.value += ' • ';
          } else {
            result = {
              kind: Format.TEXT,
              value: ' • '
            };
            results.push(result);
          }

          break;

        default:
          if (result.kind === Format.TEXT) {
            result.value += line.value;
          } else {
            result = {
              kind: Format.TEXT,
              value: line.value
            };
            results.push(result);
          }

      }
    }

    return results;
  };

  var Style$1 = {
    SUPERIMPOSE: 'flex-direction: column;display: inline-flex;justify-content: center;text-align: left;vertical-align: middle;',
    SUPERIMPOSE_SUP_SUB: 'line-height: 1; font-size: 70%'
  };

  var toHtml = function getHtml(lines) {
    let html = [];

    for (let line of lines) {
      switch (line.kind) {
        case Format.SUBSCRIPT:
          html.push(`<sub>${line.value}</sub>`);
          break;

        case Format.SUPERSCRIPT:
          html.push(`<sup>${line.value}</sup>`);
          break;

        case Format.SUPERIMPOSE:
          html.push(`<span style="${Style$1.SUPERIMPOSE}">`);
          html.push(`<sup style="${Style$1.SUPERIMPOSE_SUP_SUB}">${line.over}</sup>`);
          html.push(`<sub style="${Style$1.SUPERIMPOSE_SUP_SUB}">${line.under}</sub>`);
          html.push('</span>');
          break;

        default:
          html.push(line.value);
      }
    }

    return html.join('');
  };

  var elements$5 = [
  	{
  		number: 1,
  		isotopes: [
  			{
  				nominal: 1,
  				mass: 1.00782503223,
  				abundance: 0.999885
  			},
  			{
  				nominal: 2,
  				mass: 2.01410177812,
  				abundance: 0.000115
  			},
  			{
  				nominal: 3,
  				mass: 3.0160492779
  			},
  			{
  				nominal: 4,
  				mass: 4.02643
  			},
  			{
  				nominal: 5,
  				mass: 5.035311
  			},
  			{
  				nominal: 6,
  				mass: 6.04496
  			},
  			{
  				nominal: 7,
  				mass: 7.0527
  			}
  		],
  		symbol: "H",
  		mass: 1.0079407540557772,
  		name: "Hydrogen",
  		monoisotopicMass: 1.00782503223
  	},
  	{
  		number: 2,
  		isotopes: [
  			{
  				nominal: 3,
  				mass: 3.0160293201,
  				abundance: 0.00000134
  			},
  			{
  				nominal: 4,
  				mass: 4.00260325413,
  				abundance: 0.99999866
  			},
  			{
  				nominal: 5,
  				mass: 5.012057
  			},
  			{
  				nominal: 6,
  				mass: 6.018885891
  			},
  			{
  				nominal: 7,
  				mass: 7.0279907
  			},
  			{
  				nominal: 8,
  				mass: 8.03393439
  			},
  			{
  				nominal: 9,
  				mass: 9.043946
  			},
  			{
  				nominal: 10,
  				mass: 10.05279
  			}
  		],
  		symbol: "He",
  		mass: 4.002601932120929,
  		name: "Helium",
  		monoisotopicMass: 4.00260325413
  	},
  	{
  		number: 3,
  		isotopes: [
  			{
  				nominal: 3,
  				mass: 3.0308
  			},
  			{
  				nominal: 4,
  				mass: 4.02719
  			},
  			{
  				nominal: 5,
  				mass: 5.012538
  			},
  			{
  				nominal: 6,
  				mass: 6.0151228874,
  				abundance: 0.0759
  			},
  			{
  				nominal: 7,
  				mass: 7.0160034366,
  				abundance: 0.9241
  			},
  			{
  				nominal: 8,
  				mass: 8.022486246
  			},
  			{
  				nominal: 9,
  				mass: 9.02679019
  			},
  			{
  				nominal: 10,
  				mass: 10.035483
  			},
  			{
  				nominal: 11,
  				mass: 11.04372358
  			},
  			{
  				nominal: 12,
  				mass: 12.052517
  			},
  			{
  				nominal: 13,
  				mass: 13.06263
  			}
  		],
  		symbol: "Li",
  		mass: 6.94003660291572,
  		name: "Lithium",
  		monoisotopicMass: 7.0160034366
  	},
  	{
  		number: 4,
  		isotopes: [
  			{
  				nominal: 5,
  				mass: 5.0399
  			},
  			{
  				nominal: 6,
  				mass: 6.0197264
  			},
  			{
  				nominal: 7,
  				mass: 7.016928717
  			},
  			{
  				nominal: 8,
  				mass: 8.005305102
  			},
  			{
  				nominal: 9,
  				mass: 9.012183065,
  				abundance: 1
  			},
  			{
  				nominal: 10,
  				mass: 10.013534695
  			},
  			{
  				nominal: 11,
  				mass: 11.02166108
  			},
  			{
  				nominal: 12,
  				mass: 12.0269221
  			},
  			{
  				nominal: 13,
  				mass: 13.036135
  			},
  			{
  				nominal: 14,
  				mass: 14.04289
  			},
  			{
  				nominal: 15,
  				mass: 15.05342
  			},
  			{
  				nominal: 16,
  				mass: 16.06167
  			}
  		],
  		symbol: "Be",
  		mass: 9.012183065,
  		name: "Beryllium",
  		monoisotopicMass: 9.012183065
  	},
  	{
  		number: 5,
  		isotopes: [
  			{
  				nominal: 6,
  				mass: 6.0508
  			},
  			{
  				nominal: 7,
  				mass: 7.029712
  			},
  			{
  				nominal: 8,
  				mass: 8.0246073
  			},
  			{
  				nominal: 9,
  				mass: 9.01332965
  			},
  			{
  				nominal: 10,
  				mass: 10.01293695,
  				abundance: 0.199
  			},
  			{
  				nominal: 11,
  				mass: 11.00930536,
  				abundance: 0.801
  			},
  			{
  				nominal: 12,
  				mass: 12.0143527
  			},
  			{
  				nominal: 13,
  				mass: 13.0177802
  			},
  			{
  				nominal: 14,
  				mass: 14.025404
  			},
  			{
  				nominal: 15,
  				mass: 15.031088
  			},
  			{
  				nominal: 16,
  				mass: 16.039842
  			},
  			{
  				nominal: 17,
  				mass: 17.04699
  			},
  			{
  				nominal: 18,
  				mass: 18.05566
  			},
  			{
  				nominal: 19,
  				mass: 19.0631
  			},
  			{
  				nominal: 20,
  				mass: 20.07207
  			},
  			{
  				nominal: 21,
  				mass: 21.08129
  			}
  		],
  		symbol: "B",
  		mass: 10.811028046410001,
  		name: "Boron",
  		monoisotopicMass: 11.00930536
  	},
  	{
  		number: 6,
  		isotopes: [
  			{
  				nominal: 8,
  				mass: 8.037643
  			},
  			{
  				nominal: 9,
  				mass: 9.0310372
  			},
  			{
  				nominal: 10,
  				mass: 10.01685331
  			},
  			{
  				nominal: 11,
  				mass: 11.0114336
  			},
  			{
  				nominal: 12,
  				mass: 12,
  				abundance: 0.9893
  			},
  			{
  				nominal: 13,
  				mass: 13.00335483507,
  				abundance: 0.0107
  			},
  			{
  				nominal: 14,
  				mass: 14.0032419884
  			},
  			{
  				nominal: 15,
  				mass: 15.01059926
  			},
  			{
  				nominal: 16,
  				mass: 16.0147013
  			},
  			{
  				nominal: 17,
  				mass: 17.022577
  			},
  			{
  				nominal: 18,
  				mass: 18.026751
  			},
  			{
  				nominal: 19,
  				mass: 19.0348
  			},
  			{
  				nominal: 20,
  				mass: 20.04032
  			},
  			{
  				nominal: 21,
  				mass: 21.049
  			},
  			{
  				nominal: 22,
  				mass: 22.05753
  			},
  			{
  				nominal: 23,
  				mass: 23.0689
  			}
  		],
  		symbol: "C",
  		mass: 12.010735896735248,
  		name: "Carbon",
  		monoisotopicMass: 12
  	},
  	{
  		number: 7,
  		isotopes: [
  			{
  				nominal: 10,
  				mass: 10.04165
  			},
  			{
  				nominal: 11,
  				mass: 11.026091
  			},
  			{
  				nominal: 12,
  				mass: 12.0186132
  			},
  			{
  				nominal: 13,
  				mass: 13.00573861
  			},
  			{
  				nominal: 14,
  				mass: 14.00307400443,
  				abundance: 0.99636
  			},
  			{
  				nominal: 15,
  				mass: 15.00010889888,
  				abundance: 0.00364
  			},
  			{
  				nominal: 16,
  				mass: 16.0061019
  			},
  			{
  				nominal: 17,
  				mass: 17.008449
  			},
  			{
  				nominal: 18,
  				mass: 18.014078
  			},
  			{
  				nominal: 19,
  				mass: 19.017022
  			},
  			{
  				nominal: 20,
  				mass: 20.023366
  			},
  			{
  				nominal: 21,
  				mass: 21.02711
  			},
  			{
  				nominal: 22,
  				mass: 22.03439
  			},
  			{
  				nominal: 23,
  				mass: 23.04114
  			},
  			{
  				nominal: 24,
  				mass: 24.05039
  			},
  			{
  				nominal: 25,
  				mass: 25.0601
  			}
  		],
  		symbol: "N",
  		mass: 14.006703211445798,
  		name: "Nitrogen",
  		monoisotopicMass: 14.00307400443
  	},
  	{
  		number: 8,
  		isotopes: [
  			{
  				nominal: 12,
  				mass: 12.034262
  			},
  			{
  				nominal: 13,
  				mass: 13.024815
  			},
  			{
  				nominal: 14,
  				mass: 14.00859636
  			},
  			{
  				nominal: 15,
  				mass: 15.00306562
  			},
  			{
  				nominal: 16,
  				mass: 15.99491461957,
  				abundance: 0.99757
  			},
  			{
  				nominal: 17,
  				mass: 16.9991317565,
  				abundance: 0.00038
  			},
  			{
  				nominal: 18,
  				mass: 17.99915961286,
  				abundance: 0.00205
  			},
  			{
  				nominal: 19,
  				mass: 19.003578
  			},
  			{
  				nominal: 20,
  				mass: 20.00407535
  			},
  			{
  				nominal: 21,
  				mass: 21.008655
  			},
  			{
  				nominal: 22,
  				mass: 22.009966
  			},
  			{
  				nominal: 23,
  				mass: 23.015696
  			},
  			{
  				nominal: 24,
  				mass: 24.01986
  			},
  			{
  				nominal: 25,
  				mass: 25.02936
  			},
  			{
  				nominal: 26,
  				mass: 26.03729
  			},
  			{
  				nominal: 27,
  				mass: 27.04772
  			},
  			{
  				nominal: 28,
  				mass: 28.05591
  			}
  		],
  		symbol: "O",
  		mass: 15.999404924318277,
  		name: "Oxygen",
  		monoisotopicMass: 15.99491461957
  	},
  	{
  		number: 9,
  		isotopes: [
  			{
  				nominal: 14,
  				mass: 14.034315
  			},
  			{
  				nominal: 15,
  				mass: 15.018043
  			},
  			{
  				nominal: 16,
  				mass: 16.0114657
  			},
  			{
  				nominal: 17,
  				mass: 17.00209524
  			},
  			{
  				nominal: 18,
  				mass: 18.00093733
  			},
  			{
  				nominal: 19,
  				mass: 18.99840316273,
  				abundance: 1
  			},
  			{
  				nominal: 20,
  				mass: 19.999981252
  			},
  			{
  				nominal: 21,
  				mass: 20.9999489
  			},
  			{
  				nominal: 22,
  				mass: 22.002999
  			},
  			{
  				nominal: 23,
  				mass: 23.003557
  			},
  			{
  				nominal: 24,
  				mass: 24.008115
  			},
  			{
  				nominal: 25,
  				mass: 25.012199
  			},
  			{
  				nominal: 26,
  				mass: 26.020038
  			},
  			{
  				nominal: 27,
  				mass: 27.02644
  			},
  			{
  				nominal: 28,
  				mass: 28.03534
  			},
  			{
  				nominal: 29,
  				mass: 29.04254
  			},
  			{
  				nominal: 30,
  				mass: 30.05165
  			},
  			{
  				nominal: 31,
  				mass: 31.05971
  			}
  		],
  		symbol: "F",
  		mass: 18.99840316273,
  		name: "Fluorine",
  		monoisotopicMass: 18.99840316273
  	},
  	{
  		number: 10,
  		isotopes: [
  			{
  				nominal: 16,
  				mass: 16.02575
  			},
  			{
  				nominal: 17,
  				mass: 17.01771396
  			},
  			{
  				nominal: 18,
  				mass: 18.0057087
  			},
  			{
  				nominal: 19,
  				mass: 19.00188091
  			},
  			{
  				nominal: 20,
  				mass: 19.9924401762,
  				abundance: 0.9048
  			},
  			{
  				nominal: 21,
  				mass: 20.993846685,
  				abundance: 0.0027
  			},
  			{
  				nominal: 22,
  				mass: 21.991385114,
  				abundance: 0.0925
  			},
  			{
  				nominal: 23,
  				mass: 22.99446691
  			},
  			{
  				nominal: 24,
  				mass: 23.99361065
  			},
  			{
  				nominal: 25,
  				mass: 24.997789
  			},
  			{
  				nominal: 26,
  				mass: 26.000515
  			},
  			{
  				nominal: 27,
  				mass: 27.007553
  			},
  			{
  				nominal: 28,
  				mass: 28.01212
  			},
  			{
  				nominal: 29,
  				mass: 29.01975
  			},
  			{
  				nominal: 30,
  				mass: 30.02473
  			},
  			{
  				nominal: 31,
  				mass: 31.0331
  			},
  			{
  				nominal: 32,
  				mass: 32.03972
  			},
  			{
  				nominal: 33,
  				mass: 33.04938
  			},
  			{
  				nominal: 34,
  				mass: 34.05673
  			}
  		],
  		symbol: "Ne",
  		mass: 20.18004638052026,
  		name: "Neon",
  		monoisotopicMass: 19.9924401762
  	},
  	{
  		number: 11,
  		isotopes: [
  			{
  				nominal: 18,
  				mass: 18.02688
  			},
  			{
  				nominal: 19,
  				mass: 19.01388
  			},
  			{
  				nominal: 20,
  				mass: 20.0073544
  			},
  			{
  				nominal: 21,
  				mass: 20.99765469
  			},
  			{
  				nominal: 22,
  				mass: 21.99443741
  			},
  			{
  				nominal: 23,
  				mass: 22.989769282,
  				abundance: 1
  			},
  			{
  				nominal: 24,
  				mass: 23.99096295
  			},
  			{
  				nominal: 25,
  				mass: 24.989954
  			},
  			{
  				nominal: 26,
  				mass: 25.9926346
  			},
  			{
  				nominal: 27,
  				mass: 26.9940765
  			},
  			{
  				nominal: 28,
  				mass: 27.998939
  			},
  			{
  				nominal: 29,
  				mass: 29.0028771
  			},
  			{
  				nominal: 30,
  				mass: 30.0090979
  			},
  			{
  				nominal: 31,
  				mass: 31.013163
  			},
  			{
  				nominal: 32,
  				mass: 32.02019
  			},
  			{
  				nominal: 33,
  				mass: 33.02573
  			},
  			{
  				nominal: 34,
  				mass: 34.03359
  			},
  			{
  				nominal: 35,
  				mass: 35.04062
  			},
  			{
  				nominal: 36,
  				mass: 36.04929
  			},
  			{
  				nominal: 37,
  				mass: 37.05705
  			}
  		],
  		symbol: "Na",
  		mass: 22.989769282,
  		name: "Sodium",
  		monoisotopicMass: 22.989769282
  	},
  	{
  		number: 12,
  		isotopes: [
  			{
  				nominal: 19,
  				mass: 19.034169
  			},
  			{
  				nominal: 20,
  				mass: 20.01885
  			},
  			{
  				nominal: 21,
  				mass: 21.011716
  			},
  			{
  				nominal: 22,
  				mass: 21.99957065
  			},
  			{
  				nominal: 23,
  				mass: 22.99412421
  			},
  			{
  				nominal: 24,
  				mass: 23.985041697,
  				abundance: 0.7899
  			},
  			{
  				nominal: 25,
  				mass: 24.985836976,
  				abundance: 0.1
  			},
  			{
  				nominal: 26,
  				mass: 25.982592968,
  				abundance: 0.1101
  			},
  			{
  				nominal: 27,
  				mass: 26.984340624
  			},
  			{
  				nominal: 28,
  				mass: 27.9838767
  			},
  			{
  				nominal: 29,
  				mass: 28.988617
  			},
  			{
  				nominal: 30,
  				mass: 29.9904629
  			},
  			{
  				nominal: 31,
  				mass: 30.996648
  			},
  			{
  				nominal: 32,
  				mass: 31.9991102
  			},
  			{
  				nominal: 33,
  				mass: 33.0053271
  			},
  			{
  				nominal: 34,
  				mass: 34.008935
  			},
  			{
  				nominal: 35,
  				mass: 35.01679
  			},
  			{
  				nominal: 36,
  				mass: 36.02188
  			},
  			{
  				nominal: 37,
  				mass: 37.03037
  			},
  			{
  				nominal: 38,
  				mass: 38.03658
  			},
  			{
  				nominal: 39,
  				mass: 39.04538
  			},
  			{
  				nominal: 40,
  				mass: 40.05218
  			}
  		],
  		symbol: "Mg",
  		mass: 24.3050516198371,
  		name: "Magnesium",
  		monoisotopicMass: 23.985041697
  	},
  	{
  		number: 13,
  		isotopes: [
  			{
  				nominal: 21,
  				mass: 21.02897
  			},
  			{
  				nominal: 22,
  				mass: 22.01954
  			},
  			{
  				nominal: 23,
  				mass: 23.00724435
  			},
  			{
  				nominal: 24,
  				mass: 23.9999489
  			},
  			{
  				nominal: 25,
  				mass: 24.9904281
  			},
  			{
  				nominal: 26,
  				mass: 25.986891904
  			},
  			{
  				nominal: 27,
  				mass: 26.98153853,
  				abundance: 1
  			},
  			{
  				nominal: 28,
  				mass: 27.98191021
  			},
  			{
  				nominal: 29,
  				mass: 28.9804565
  			},
  			{
  				nominal: 30,
  				mass: 29.98296
  			},
  			{
  				nominal: 31,
  				mass: 30.983945
  			},
  			{
  				nominal: 32,
  				mass: 31.988085
  			},
  			{
  				nominal: 33,
  				mass: 32.990909
  			},
  			{
  				nominal: 34,
  				mass: 33.996705
  			},
  			{
  				nominal: 35,
  				mass: 34.999764
  			},
  			{
  				nominal: 36,
  				mass: 36.00639
  			},
  			{
  				nominal: 37,
  				mass: 37.01053
  			},
  			{
  				nominal: 38,
  				mass: 38.0174
  			},
  			{
  				nominal: 39,
  				mass: 39.02254
  			},
  			{
  				nominal: 40,
  				mass: 40.03003
  			},
  			{
  				nominal: 41,
  				mass: 41.03638
  			},
  			{
  				nominal: 42,
  				mass: 42.04384
  			},
  			{
  				nominal: 43,
  				mass: 43.05147
  			}
  		],
  		symbol: "Al",
  		mass: 26.98153853,
  		name: "Aluminium",
  		monoisotopicMass: 26.98153853
  	},
  	{
  		number: 14,
  		isotopes: [
  			{
  				nominal: 22,
  				mass: 22.03579
  			},
  			{
  				nominal: 23,
  				mass: 23.02544
  			},
  			{
  				nominal: 24,
  				mass: 24.011535
  			},
  			{
  				nominal: 25,
  				mass: 25.004109
  			},
  			{
  				nominal: 26,
  				mass: 25.99233384
  			},
  			{
  				nominal: 27,
  				mass: 26.98670481
  			},
  			{
  				nominal: 28,
  				mass: 27.97692653465,
  				abundance: 0.92223
  			},
  			{
  				nominal: 29,
  				mass: 28.9764946649,
  				abundance: 0.04685
  			},
  			{
  				nominal: 30,
  				mass: 29.973770136,
  				abundance: 0.03092
  			},
  			{
  				nominal: 31,
  				mass: 30.975363194
  			},
  			{
  				nominal: 32,
  				mass: 31.97415154
  			},
  			{
  				nominal: 33,
  				mass: 32.97797696
  			},
  			{
  				nominal: 34,
  				mass: 33.978576
  			},
  			{
  				nominal: 35,
  				mass: 34.984583
  			},
  			{
  				nominal: 36,
  				mass: 35.986695
  			},
  			{
  				nominal: 37,
  				mass: 36.992921
  			},
  			{
  				nominal: 38,
  				mass: 37.995523
  			},
  			{
  				nominal: 39,
  				mass: 39.002491
  			},
  			{
  				nominal: 40,
  				mass: 40.00583
  			},
  			{
  				nominal: 41,
  				mass: 41.01301
  			},
  			{
  				nominal: 42,
  				mass: 42.01778
  			},
  			{
  				nominal: 43,
  				mass: 43.0248
  			},
  			{
  				nominal: 44,
  				mass: 44.03061
  			},
  			{
  				nominal: 45,
  				mass: 45.03995
  			}
  		],
  		symbol: "Si",
  		mass: 28.085498705705955,
  		name: "Silicon",
  		monoisotopicMass: 27.97692653465
  	},
  	{
  		number: 15,
  		isotopes: [
  			{
  				nominal: 24,
  				mass: 24.03577
  			},
  			{
  				nominal: 25,
  				mass: 25.02119
  			},
  			{
  				nominal: 26,
  				mass: 26.01178
  			},
  			{
  				nominal: 27,
  				mass: 26.999224
  			},
  			{
  				nominal: 28,
  				mass: 27.9923266
  			},
  			{
  				nominal: 29,
  				mass: 28.98180079
  			},
  			{
  				nominal: 30,
  				mass: 29.97831375
  			},
  			{
  				nominal: 31,
  				mass: 30.97376199842,
  				abundance: 1
  			},
  			{
  				nominal: 32,
  				mass: 31.973907643
  			},
  			{
  				nominal: 33,
  				mass: 32.9717257
  			},
  			{
  				nominal: 34,
  				mass: 33.97364589
  			},
  			{
  				nominal: 35,
  				mass: 34.9733141
  			},
  			{
  				nominal: 36,
  				mass: 35.97826
  			},
  			{
  				nominal: 37,
  				mass: 36.979607
  			},
  			{
  				nominal: 38,
  				mass: 37.984252
  			},
  			{
  				nominal: 39,
  				mass: 38.986227
  			},
  			{
  				nominal: 40,
  				mass: 39.99133
  			},
  			{
  				nominal: 41,
  				mass: 40.994654
  			},
  			{
  				nominal: 42,
  				mass: 42.00108
  			},
  			{
  				nominal: 43,
  				mass: 43.00502
  			},
  			{
  				nominal: 44,
  				mass: 44.01121
  			},
  			{
  				nominal: 45,
  				mass: 45.01645
  			},
  			{
  				nominal: 46,
  				mass: 46.02446
  			},
  			{
  				nominal: 47,
  				mass: 47.03139
  			}
  		],
  		symbol: "P",
  		mass: 30.97376199842,
  		name: "Phosphorus",
  		monoisotopicMass: 30.97376199842
  	},
  	{
  		number: 16,
  		isotopes: [
  			{
  				nominal: 26,
  				mass: 26.02907
  			},
  			{
  				nominal: 27,
  				mass: 27.01828
  			},
  			{
  				nominal: 28,
  				mass: 28.00437
  			},
  			{
  				nominal: 29,
  				mass: 28.996611
  			},
  			{
  				nominal: 30,
  				mass: 29.98490703
  			},
  			{
  				nominal: 31,
  				mass: 30.97955701
  			},
  			{
  				nominal: 32,
  				mass: 31.9720711744,
  				abundance: 0.9499
  			},
  			{
  				nominal: 33,
  				mass: 32.9714589098,
  				abundance: 0.0075
  			},
  			{
  				nominal: 34,
  				mass: 33.967867004,
  				abundance: 0.0425
  			},
  			{
  				nominal: 35,
  				mass: 34.96903231
  			},
  			{
  				nominal: 36,
  				mass: 35.96708071,
  				abundance: 0.0001
  			},
  			{
  				nominal: 37,
  				mass: 36.97112551
  			},
  			{
  				nominal: 38,
  				mass: 37.9711633
  			},
  			{
  				nominal: 39,
  				mass: 38.975134
  			},
  			{
  				nominal: 40,
  				mass: 39.9754826
  			},
  			{
  				nominal: 41,
  				mass: 40.9795935
  			},
  			{
  				nominal: 42,
  				mass: 41.9810651
  			},
  			{
  				nominal: 43,
  				mass: 42.9869076
  			},
  			{
  				nominal: 44,
  				mass: 43.9901188
  			},
  			{
  				nominal: 45,
  				mass: 44.99572
  			},
  			{
  				nominal: 46,
  				mass: 46.00004
  			},
  			{
  				nominal: 47,
  				mass: 47.00795
  			},
  			{
  				nominal: 48,
  				mass: 48.0137
  			},
  			{
  				nominal: 49,
  				mass: 49.02276
  			}
  		],
  		symbol: "S",
  		mass: 32.06478740612706,
  		name: "Sulfur",
  		monoisotopicMass: 31.9720711744
  	},
  	{
  		number: 17,
  		isotopes: [
  			{
  				nominal: 28,
  				mass: 28.02954
  			},
  			{
  				nominal: 29,
  				mass: 29.01478
  			},
  			{
  				nominal: 30,
  				mass: 30.00477
  			},
  			{
  				nominal: 31,
  				mass: 30.992414
  			},
  			{
  				nominal: 32,
  				mass: 31.98568464
  			},
  			{
  				nominal: 33,
  				mass: 32.97745199
  			},
  			{
  				nominal: 34,
  				mass: 33.973762485
  			},
  			{
  				nominal: 35,
  				mass: 34.968852682,
  				abundance: 0.7576
  			},
  			{
  				nominal: 36,
  				mass: 35.968306809
  			},
  			{
  				nominal: 37,
  				mass: 36.965902602,
  				abundance: 0.2424
  			},
  			{
  				nominal: 38,
  				mass: 37.96801044
  			},
  			{
  				nominal: 39,
  				mass: 38.9680082
  			},
  			{
  				nominal: 40,
  				mass: 39.970415
  			},
  			{
  				nominal: 41,
  				mass: 40.970685
  			},
  			{
  				nominal: 42,
  				mass: 41.97325
  			},
  			{
  				nominal: 43,
  				mass: 42.97389
  			},
  			{
  				nominal: 44,
  				mass: 43.97787
  			},
  			{
  				nominal: 45,
  				mass: 44.98029
  			},
  			{
  				nominal: 46,
  				mass: 45.98517
  			},
  			{
  				nominal: 47,
  				mass: 46.98916
  			},
  			{
  				nominal: 48,
  				mass: 47.99564
  			},
  			{
  				nominal: 49,
  				mass: 49.00123
  			},
  			{
  				nominal: 50,
  				mass: 50.00905
  			},
  			{
  				nominal: 51,
  				mass: 51.01554
  			}
  		],
  		symbol: "Cl",
  		mass: 35.452937582608,
  		name: "Chlorine",
  		monoisotopicMass: 34.968852682
  	},
  	{
  		number: 18,
  		isotopes: [
  			{
  				nominal: 30,
  				mass: 30.02307
  			},
  			{
  				nominal: 31,
  				mass: 31.01212
  			},
  			{
  				nominal: 32,
  				mass: 31.9976378
  			},
  			{
  				nominal: 33,
  				mass: 32.98992555
  			},
  			{
  				nominal: 34,
  				mass: 33.98027009
  			},
  			{
  				nominal: 35,
  				mass: 34.97525759
  			},
  			{
  				nominal: 36,
  				mass: 35.967545105,
  				abundance: 0.003336
  			},
  			{
  				nominal: 37,
  				mass: 36.96677633
  			},
  			{
  				nominal: 38,
  				mass: 37.96273211,
  				abundance: 0.000629
  			},
  			{
  				nominal: 39,
  				mass: 38.964313
  			},
  			{
  				nominal: 40,
  				mass: 39.9623831237,
  				abundance: 0.996035
  			},
  			{
  				nominal: 41,
  				mass: 40.96450057
  			},
  			{
  				nominal: 42,
  				mass: 41.9630457
  			},
  			{
  				nominal: 43,
  				mass: 42.9656361
  			},
  			{
  				nominal: 44,
  				mass: 43.9649238
  			},
  			{
  				nominal: 45,
  				mass: 44.96803973
  			},
  			{
  				nominal: 46,
  				mass: 45.968083
  			},
  			{
  				nominal: 47,
  				mass: 46.972935
  			},
  			{
  				nominal: 48,
  				mass: 47.97591
  			},
  			{
  				nominal: 49,
  				mass: 48.9819
  			},
  			{
  				nominal: 50,
  				mass: 49.98613
  			},
  			{
  				nominal: 51,
  				mass: 50.9937
  			},
  			{
  				nominal: 52,
  				mass: 51.99896
  			},
  			{
  				nominal: 53,
  				mass: 53.00729
  			}
  		],
  		symbol: "Ar",
  		mass: 39.947798563582005,
  		name: "Argon",
  		monoisotopicMass: 39.9623831237
  	},
  	{
  		number: 19,
  		isotopes: [
  			{
  				nominal: 32,
  				mass: 32.02265
  			},
  			{
  				nominal: 33,
  				mass: 33.00756
  			},
  			{
  				nominal: 34,
  				mass: 33.99869
  			},
  			{
  				nominal: 35,
  				mass: 34.98800541
  			},
  			{
  				nominal: 36,
  				mass: 35.98130201
  			},
  			{
  				nominal: 37,
  				mass: 36.97337589
  			},
  			{
  				nominal: 38,
  				mass: 37.96908112
  			},
  			{
  				nominal: 39,
  				mass: 38.9637064864,
  				abundance: 0.932581
  			},
  			{
  				nominal: 40,
  				mass: 39.963998166,
  				abundance: 0.000117
  			},
  			{
  				nominal: 41,
  				mass: 40.9618252579,
  				abundance: 0.067302
  			},
  			{
  				nominal: 42,
  				mass: 41.96240231
  			},
  			{
  				nominal: 43,
  				mass: 42.9607347
  			},
  			{
  				nominal: 44,
  				mass: 43.96158699
  			},
  			{
  				nominal: 45,
  				mass: 44.96069149
  			},
  			{
  				nominal: 46,
  				mass: 45.96198159
  			},
  			{
  				nominal: 47,
  				mass: 46.9616616
  			},
  			{
  				nominal: 48,
  				mass: 47.96534119
  			},
  			{
  				nominal: 49,
  				mass: 48.96821075
  			},
  			{
  				nominal: 50,
  				mass: 49.97238
  			},
  			{
  				nominal: 51,
  				mass: 50.975828
  			},
  			{
  				nominal: 52,
  				mass: 51.98224
  			},
  			{
  				nominal: 53,
  				mass: 52.98746
  			},
  			{
  				nominal: 54,
  				mass: 53.99463
  			},
  			{
  				nominal: 55,
  				mass: 55.00076
  			},
  			{
  				nominal: 56,
  				mass: 56.00851
  			}
  		],
  		symbol: "K",
  		mass: 39.098300910086,
  		name: "Potassium",
  		monoisotopicMass: 38.9637064864
  	},
  	{
  		number: 20,
  		isotopes: [
  			{
  				nominal: 34,
  				mass: 34.01487
  			},
  			{
  				nominal: 35,
  				mass: 35.00514
  			},
  			{
  				nominal: 36,
  				mass: 35.993074
  			},
  			{
  				nominal: 37,
  				mass: 36.98589785
  			},
  			{
  				nominal: 38,
  				mass: 37.97631922
  			},
  			{
  				nominal: 39,
  				mass: 38.97071081
  			},
  			{
  				nominal: 40,
  				mass: 39.962590863,
  				abundance: 0.96941
  			},
  			{
  				nominal: 41,
  				mass: 40.96227792
  			},
  			{
  				nominal: 42,
  				mass: 41.95861783,
  				abundance: 0.00647
  			},
  			{
  				nominal: 43,
  				mass: 42.95876644,
  				abundance: 0.00135
  			},
  			{
  				nominal: 44,
  				mass: 43.95548156,
  				abundance: 0.02086
  			},
  			{
  				nominal: 45,
  				mass: 44.95618635
  			},
  			{
  				nominal: 46,
  				mass: 45.953689,
  				abundance: 0.00004
  			},
  			{
  				nominal: 47,
  				mass: 46.9545424
  			},
  			{
  				nominal: 48,
  				mass: 47.95252276,
  				abundance: 0.00187
  			},
  			{
  				nominal: 49,
  				mass: 48.95566274
  			},
  			{
  				nominal: 50,
  				mass: 49.9574992
  			},
  			{
  				nominal: 51,
  				mass: 50.960989
  			},
  			{
  				nominal: 52,
  				mass: 51.963217
  			},
  			{
  				nominal: 53,
  				mass: 52.96945
  			},
  			{
  				nominal: 54,
  				mass: 53.9734
  			},
  			{
  				nominal: 55,
  				mass: 54.9803
  			},
  			{
  				nominal: 56,
  				mass: 55.98508
  			},
  			{
  				nominal: 57,
  				mass: 56.99262
  			},
  			{
  				nominal: 58,
  				mass: 57.99794
  			}
  		],
  		symbol: "Ca",
  		mass: 40.078022511017735,
  		name: "Calcium",
  		monoisotopicMass: 39.962590863
  	},
  	{
  		number: 21,
  		isotopes: [
  			{
  				nominal: 36,
  				mass: 36.01648
  			},
  			{
  				nominal: 37,
  				mass: 37.00374
  			},
  			{
  				nominal: 38,
  				mass: 37.99512
  			},
  			{
  				nominal: 39,
  				mass: 38.984785
  			},
  			{
  				nominal: 40,
  				mass: 39.9779673
  			},
  			{
  				nominal: 41,
  				mass: 40.969251105
  			},
  			{
  				nominal: 42,
  				mass: 41.96551653
  			},
  			{
  				nominal: 43,
  				mass: 42.9611505
  			},
  			{
  				nominal: 44,
  				mass: 43.9594029
  			},
  			{
  				nominal: 45,
  				mass: 44.95590828,
  				abundance: 1
  			},
  			{
  				nominal: 46,
  				mass: 45.95516826
  			},
  			{
  				nominal: 47,
  				mass: 46.9524037
  			},
  			{
  				nominal: 48,
  				mass: 47.9522236
  			},
  			{
  				nominal: 49,
  				mass: 48.9500146
  			},
  			{
  				nominal: 50,
  				mass: 49.952176
  			},
  			{
  				nominal: 51,
  				mass: 50.953592
  			},
  			{
  				nominal: 52,
  				mass: 51.95688
  			},
  			{
  				nominal: 53,
  				mass: 52.95909
  			},
  			{
  				nominal: 54,
  				mass: 53.96393
  			},
  			{
  				nominal: 55,
  				mass: 54.96782
  			},
  			{
  				nominal: 56,
  				mass: 55.97345
  			},
  			{
  				nominal: 57,
  				mass: 56.97777
  			},
  			{
  				nominal: 58,
  				mass: 57.98403
  			},
  			{
  				nominal: 59,
  				mass: 58.98894
  			},
  			{
  				nominal: 60,
  				mass: 59.99565
  			},
  			{
  				nominal: 61,
  				mass: 61.001
  			}
  		],
  		symbol: "Sc",
  		mass: 44.95590828,
  		name: "Scandium",
  		monoisotopicMass: 44.95590828
  	},
  	{
  		number: 22,
  		isotopes: [
  			{
  				nominal: 38,
  				mass: 38.01145
  			},
  			{
  				nominal: 39,
  				mass: 39.00236
  			},
  			{
  				nominal: 40,
  				mass: 39.9905
  			},
  			{
  				nominal: 41,
  				mass: 40.983148
  			},
  			{
  				nominal: 42,
  				mass: 41.97304903
  			},
  			{
  				nominal: 43,
  				mass: 42.9685225
  			},
  			{
  				nominal: 44,
  				mass: 43.95968995
  			},
  			{
  				nominal: 45,
  				mass: 44.95812198
  			},
  			{
  				nominal: 46,
  				mass: 45.95262772,
  				abundance: 0.0825
  			},
  			{
  				nominal: 47,
  				mass: 46.95175879,
  				abundance: 0.0744
  			},
  			{
  				nominal: 48,
  				mass: 47.94794198,
  				abundance: 0.7372
  			},
  			{
  				nominal: 49,
  				mass: 48.94786568,
  				abundance: 0.0541
  			},
  			{
  				nominal: 50,
  				mass: 49.94478689,
  				abundance: 0.0518
  			},
  			{
  				nominal: 51,
  				mass: 50.94661065
  			},
  			{
  				nominal: 52,
  				mass: 51.946893
  			},
  			{
  				nominal: 53,
  				mass: 52.94973
  			},
  			{
  				nominal: 54,
  				mass: 53.95105
  			},
  			{
  				nominal: 55,
  				mass: 54.95527
  			},
  			{
  				nominal: 56,
  				mass: 55.95791
  			},
  			{
  				nominal: 57,
  				mass: 56.96364
  			},
  			{
  				nominal: 58,
  				mass: 57.9666
  			},
  			{
  				nominal: 59,
  				mass: 58.97247
  			},
  			{
  				nominal: 60,
  				mass: 59.97603
  			},
  			{
  				nominal: 61,
  				mass: 60.98245
  			},
  			{
  				nominal: 62,
  				mass: 61.98651
  			},
  			{
  				nominal: 63,
  				mass: 62.99375
  			}
  		],
  		symbol: "Ti",
  		mass: 47.866744962721995,
  		name: "Titanium",
  		monoisotopicMass: 47.94794198
  	},
  	{
  		number: 23,
  		isotopes: [
  			{
  				nominal: 40,
  				mass: 40.01276
  			},
  			{
  				nominal: 41,
  				mass: 41.00021
  			},
  			{
  				nominal: 42,
  				mass: 41.99182
  			},
  			{
  				nominal: 43,
  				mass: 42.980766
  			},
  			{
  				nominal: 44,
  				mass: 43.97411
  			},
  			{
  				nominal: 45,
  				mass: 44.9657748
  			},
  			{
  				nominal: 46,
  				mass: 45.96019878
  			},
  			{
  				nominal: 47,
  				mass: 46.95490491
  			},
  			{
  				nominal: 48,
  				mass: 47.9522522
  			},
  			{
  				nominal: 49,
  				mass: 48.9485118
  			},
  			{
  				nominal: 50,
  				mass: 49.94715601,
  				abundance: 0.0025
  			},
  			{
  				nominal: 51,
  				mass: 50.94395704,
  				abundance: 0.9975
  			},
  			{
  				nominal: 52,
  				mass: 51.94477301
  			},
  			{
  				nominal: 53,
  				mass: 52.9443367
  			},
  			{
  				nominal: 54,
  				mass: 53.946439
  			},
  			{
  				nominal: 55,
  				mass: 54.94724
  			},
  			{
  				nominal: 56,
  				mass: 55.95048
  			},
  			{
  				nominal: 57,
  				mass: 56.95252
  			},
  			{
  				nominal: 58,
  				mass: 57.95672
  			},
  			{
  				nominal: 59,
  				mass: 58.95939
  			},
  			{
  				nominal: 60,
  				mass: 59.96431
  			},
  			{
  				nominal: 61,
  				mass: 60.96725
  			},
  			{
  				nominal: 62,
  				mass: 61.97265
  			},
  			{
  				nominal: 63,
  				mass: 62.97639
  			},
  			{
  				nominal: 64,
  				mass: 63.98264
  			},
  			{
  				nominal: 65,
  				mass: 64.9875
  			},
  			{
  				nominal: 66,
  				mass: 65.99398
  			}
  		],
  		symbol: "V",
  		mass: 50.941465037425004,
  		name: "Vanadium",
  		monoisotopicMass: 50.94395704
  	},
  	{
  		number: 24,
  		isotopes: [
  			{
  				nominal: 42,
  				mass: 42.0067
  			},
  			{
  				nominal: 43,
  				mass: 42.99753
  			},
  			{
  				nominal: 44,
  				mass: 43.98536
  			},
  			{
  				nominal: 45,
  				mass: 44.97905
  			},
  			{
  				nominal: 46,
  				mass: 45.968359
  			},
  			{
  				nominal: 47,
  				mass: 46.9628974
  			},
  			{
  				nominal: 48,
  				mass: 47.9540291
  			},
  			{
  				nominal: 49,
  				mass: 48.9513333
  			},
  			{
  				nominal: 50,
  				mass: 49.94604183,
  				abundance: 0.04345
  			},
  			{
  				nominal: 51,
  				mass: 50.94476502
  			},
  			{
  				nominal: 52,
  				mass: 51.94050623,
  				abundance: 0.83789
  			},
  			{
  				nominal: 53,
  				mass: 52.94064815,
  				abundance: 0.09501
  			},
  			{
  				nominal: 54,
  				mass: 53.93887916,
  				abundance: 0.02365
  			},
  			{
  				nominal: 55,
  				mass: 54.94083843
  			},
  			{
  				nominal: 56,
  				mass: 55.9406531
  			},
  			{
  				nominal: 57,
  				mass: 56.943613
  			},
  			{
  				nominal: 58,
  				mass: 57.94435
  			},
  			{
  				nominal: 59,
  				mass: 58.94859
  			},
  			{
  				nominal: 60,
  				mass: 59.95008
  			},
  			{
  				nominal: 61,
  				mass: 60.95442
  			},
  			{
  				nominal: 62,
  				mass: 61.9561
  			},
  			{
  				nominal: 63,
  				mass: 62.96165
  			},
  			{
  				nominal: 64,
  				mass: 63.96408
  			},
  			{
  				nominal: 65,
  				mass: 64.96996
  			},
  			{
  				nominal: 66,
  				mass: 65.97366
  			},
  			{
  				nominal: 67,
  				mass: 66.98016
  			},
  			{
  				nominal: 68,
  				mass: 67.98403
  			}
  		],
  		symbol: "Cr",
  		mass: 51.9961317554337,
  		name: "Chromium",
  		monoisotopicMass: 51.94050623
  	},
  	{
  		number: 25,
  		isotopes: [
  			{
  				nominal: 44,
  				mass: 44.00715
  			},
  			{
  				nominal: 45,
  				mass: 44.99449
  			},
  			{
  				nominal: 46,
  				mass: 45.98609
  			},
  			{
  				nominal: 47,
  				mass: 46.975775
  			},
  			{
  				nominal: 48,
  				mass: 47.96852
  			},
  			{
  				nominal: 49,
  				mass: 48.959595
  			},
  			{
  				nominal: 50,
  				mass: 49.95423778
  			},
  			{
  				nominal: 51,
  				mass: 50.94820847
  			},
  			{
  				nominal: 52,
  				mass: 51.9455639
  			},
  			{
  				nominal: 53,
  				mass: 52.94128889
  			},
  			{
  				nominal: 54,
  				mass: 53.9403576
  			},
  			{
  				nominal: 55,
  				mass: 54.93804391,
  				abundance: 1
  			},
  			{
  				nominal: 56,
  				mass: 55.93890369
  			},
  			{
  				nominal: 57,
  				mass: 56.9382861
  			},
  			{
  				nominal: 58,
  				mass: 57.9400666
  			},
  			{
  				nominal: 59,
  				mass: 58.9403911
  			},
  			{
  				nominal: 60,
  				mass: 59.9431366
  			},
  			{
  				nominal: 61,
  				mass: 60.9444525
  			},
  			{
  				nominal: 62,
  				mass: 61.94795
  			},
  			{
  				nominal: 63,
  				mass: 62.9496647
  			},
  			{
  				nominal: 64,
  				mass: 63.9538494
  			},
  			{
  				nominal: 65,
  				mass: 64.9560198
  			},
  			{
  				nominal: 66,
  				mass: 65.960547
  			},
  			{
  				nominal: 67,
  				mass: 66.96424
  			},
  			{
  				nominal: 68,
  				mass: 67.96962
  			},
  			{
  				nominal: 69,
  				mass: 68.97366
  			},
  			{
  				nominal: 70,
  				mass: 69.97937
  			},
  			{
  				nominal: 71,
  				mass: 70.98368
  			}
  		],
  		symbol: "Mn",
  		mass: 54.93804391,
  		name: "Manganese",
  		monoisotopicMass: 54.93804391
  	},
  	{
  		number: 26,
  		isotopes: [
  			{
  				nominal: 45,
  				mass: 45.01442
  			},
  			{
  				nominal: 46,
  				mass: 46.00063
  			},
  			{
  				nominal: 47,
  				mass: 46.99185
  			},
  			{
  				nominal: 48,
  				mass: 47.98023
  			},
  			{
  				nominal: 49,
  				mass: 48.973429
  			},
  			{
  				nominal: 50,
  				mass: 49.962975
  			},
  			{
  				nominal: 51,
  				mass: 50.956841
  			},
  			{
  				nominal: 52,
  				mass: 51.9481131
  			},
  			{
  				nominal: 53,
  				mass: 52.9453064
  			},
  			{
  				nominal: 54,
  				mass: 53.93960899,
  				abundance: 0.05845
  			},
  			{
  				nominal: 55,
  				mass: 54.93829199
  			},
  			{
  				nominal: 56,
  				mass: 55.93493633,
  				abundance: 0.91754
  			},
  			{
  				nominal: 57,
  				mass: 56.93539284,
  				abundance: 0.02119
  			},
  			{
  				nominal: 58,
  				mass: 57.93327443,
  				abundance: 0.00282
  			},
  			{
  				nominal: 59,
  				mass: 58.93487434
  			},
  			{
  				nominal: 60,
  				mass: 59.9340711
  			},
  			{
  				nominal: 61,
  				mass: 60.9367462
  			},
  			{
  				nominal: 62,
  				mass: 61.9367918
  			},
  			{
  				nominal: 63,
  				mass: 62.9402727
  			},
  			{
  				nominal: 64,
  				mass: 63.9409878
  			},
  			{
  				nominal: 65,
  				mass: 64.9450115
  			},
  			{
  				nominal: 66,
  				mass: 65.94625
  			},
  			{
  				nominal: 67,
  				mass: 66.95054
  			},
  			{
  				nominal: 68,
  				mass: 67.95295
  			},
  			{
  				nominal: 69,
  				mass: 68.95807
  			},
  			{
  				nominal: 70,
  				mass: 69.96102
  			},
  			{
  				nominal: 71,
  				mass: 70.96672
  			},
  			{
  				nominal: 72,
  				mass: 71.96983
  			},
  			{
  				nominal: 73,
  				mass: 72.97572
  			},
  			{
  				nominal: 74,
  				mass: 73.97935
  			}
  		],
  		symbol: "Fe",
  		mass: 55.845144433865904,
  		name: "Iron",
  		monoisotopicMass: 55.93493633
  	},
  	{
  		number: 27,
  		isotopes: [
  			{
  				nominal: 47,
  				mass: 47.01057
  			},
  			{
  				nominal: 48,
  				mass: 48.00093
  			},
  			{
  				nominal: 49,
  				mass: 48.98891
  			},
  			{
  				nominal: 50,
  				mass: 49.98091
  			},
  			{
  				nominal: 51,
  				mass: 50.970647
  			},
  			{
  				nominal: 52,
  				mass: 51.96351
  			},
  			{
  				nominal: 53,
  				mass: 52.9542041
  			},
  			{
  				nominal: 54,
  				mass: 53.94845987
  			},
  			{
  				nominal: 55,
  				mass: 54.9419972
  			},
  			{
  				nominal: 56,
  				mass: 55.9398388
  			},
  			{
  				nominal: 57,
  				mass: 56.93629057
  			},
  			{
  				nominal: 58,
  				mass: 57.9357521
  			},
  			{
  				nominal: 59,
  				mass: 58.93319429,
  				abundance: 1
  			},
  			{
  				nominal: 60,
  				mass: 59.9338163
  			},
  			{
  				nominal: 61,
  				mass: 60.93247662
  			},
  			{
  				nominal: 62,
  				mass: 61.934059
  			},
  			{
  				nominal: 63,
  				mass: 62.9336
  			},
  			{
  				nominal: 64,
  				mass: 63.935811
  			},
  			{
  				nominal: 65,
  				mass: 64.9364621
  			},
  			{
  				nominal: 66,
  				mass: 65.939443
  			},
  			{
  				nominal: 67,
  				mass: 66.9406096
  			},
  			{
  				nominal: 68,
  				mass: 67.94426
  			},
  			{
  				nominal: 69,
  				mass: 68.94614
  			},
  			{
  				nominal: 70,
  				mass: 69.94963
  			},
  			{
  				nominal: 71,
  				mass: 70.95237
  			},
  			{
  				nominal: 72,
  				mass: 71.95729
  			},
  			{
  				nominal: 73,
  				mass: 72.96039
  			},
  			{
  				nominal: 74,
  				mass: 73.96515
  			},
  			{
  				nominal: 75,
  				mass: 74.96876
  			},
  			{
  				nominal: 76,
  				mass: 75.97413
  			}
  		],
  		symbol: "Co",
  		mass: 58.93319429,
  		name: "Cobalt",
  		monoisotopicMass: 58.93319429
  	},
  	{
  		number: 28,
  		isotopes: [
  			{
  				nominal: 48,
  				mass: 48.01769
  			},
  			{
  				nominal: 49,
  				mass: 49.0077
  			},
  			{
  				nominal: 50,
  				mass: 49.99474
  			},
  			{
  				nominal: 51,
  				mass: 50.98611
  			},
  			{
  				nominal: 52,
  				mass: 51.9748
  			},
  			{
  				nominal: 53,
  				mass: 52.96819
  			},
  			{
  				nominal: 54,
  				mass: 53.957892
  			},
  			{
  				nominal: 55,
  				mass: 54.95133063
  			},
  			{
  				nominal: 56,
  				mass: 55.94212855
  			},
  			{
  				nominal: 57,
  				mass: 56.93979218
  			},
  			{
  				nominal: 58,
  				mass: 57.93534241,
  				abundance: 0.68077
  			},
  			{
  				nominal: 59,
  				mass: 58.9343462
  			},
  			{
  				nominal: 60,
  				mass: 59.93078588,
  				abundance: 0.26223
  			},
  			{
  				nominal: 61,
  				mass: 60.93105557,
  				abundance: 0.011399
  			},
  			{
  				nominal: 62,
  				mass: 61.92834537,
  				abundance: 0.036346
  			},
  			{
  				nominal: 63,
  				mass: 62.92966963
  			},
  			{
  				nominal: 64,
  				mass: 63.92796682,
  				abundance: 0.009255
  			},
  			{
  				nominal: 65,
  				mass: 64.93008517
  			},
  			{
  				nominal: 66,
  				mass: 65.9291393
  			},
  			{
  				nominal: 67,
  				mass: 66.9315694
  			},
  			{
  				nominal: 68,
  				mass: 67.9318688
  			},
  			{
  				nominal: 69,
  				mass: 68.9356103
  			},
  			{
  				nominal: 70,
  				mass: 69.9364313
  			},
  			{
  				nominal: 71,
  				mass: 70.940519
  			},
  			{
  				nominal: 72,
  				mass: 71.9417859
  			},
  			{
  				nominal: 73,
  				mass: 72.9462067
  			},
  			{
  				nominal: 74,
  				mass: 73.94798
  			},
  			{
  				nominal: 75,
  				mass: 74.9525
  			},
  			{
  				nominal: 76,
  				mass: 75.95533
  			},
  			{
  				nominal: 77,
  				mass: 76.96055
  			},
  			{
  				nominal: 78,
  				mass: 77.96336
  			},
  			{
  				nominal: 79,
  				mass: 78.97025
  			}
  		],
  		symbol: "Ni",
  		mass: 58.69334710994765,
  		name: "Nickel",
  		monoisotopicMass: 57.93534241
  	},
  	{
  		number: 29,
  		isotopes: [
  			{
  				nominal: 52,
  				mass: 51.99671
  			},
  			{
  				nominal: 53,
  				mass: 52.98459
  			},
  			{
  				nominal: 54,
  				mass: 53.97666
  			},
  			{
  				nominal: 55,
  				mass: 54.96604
  			},
  			{
  				nominal: 56,
  				mass: 55.95895
  			},
  			{
  				nominal: 57,
  				mass: 56.9492125
  			},
  			{
  				nominal: 58,
  				mass: 57.94453305
  			},
  			{
  				nominal: 59,
  				mass: 58.93949748
  			},
  			{
  				nominal: 60,
  				mass: 59.9373645
  			},
  			{
  				nominal: 61,
  				mass: 60.9334576
  			},
  			{
  				nominal: 62,
  				mass: 61.93259541
  			},
  			{
  				nominal: 63,
  				mass: 62.92959772,
  				abundance: 0.6915
  			},
  			{
  				nominal: 64,
  				mass: 63.92976434
  			},
  			{
  				nominal: 65,
  				mass: 64.9277897,
  				abundance: 0.3085
  			},
  			{
  				nominal: 66,
  				mass: 65.92886903
  			},
  			{
  				nominal: 67,
  				mass: 66.9277303
  			},
  			{
  				nominal: 68,
  				mass: 67.9296109
  			},
  			{
  				nominal: 69,
  				mass: 68.9294293
  			},
  			{
  				nominal: 70,
  				mass: 69.9323921
  			},
  			{
  				nominal: 71,
  				mass: 70.9326768
  			},
  			{
  				nominal: 72,
  				mass: 71.9358203
  			},
  			{
  				nominal: 73,
  				mass: 72.9366744
  			},
  			{
  				nominal: 74,
  				mass: 73.9398749
  			},
  			{
  				nominal: 75,
  				mass: 74.9415226
  			},
  			{
  				nominal: 76,
  				mass: 75.945275
  			},
  			{
  				nominal: 77,
  				mass: 76.94792
  			},
  			{
  				nominal: 78,
  				mass: 77.95223
  			},
  			{
  				nominal: 79,
  				mass: 78.95502
  			},
  			{
  				nominal: 80,
  				mass: 79.96089
  			},
  			{
  				nominal: 81,
  				mass: 80.96587
  			},
  			{
  				nominal: 82,
  				mass: 81.97244
  			}
  		],
  		symbol: "Cu",
  		mass: 63.54603994583,
  		name: "Copper",
  		monoisotopicMass: 62.92959772
  	},
  	{
  		number: 30,
  		isotopes: [
  			{
  				nominal: 54,
  				mass: 53.99204
  			},
  			{
  				nominal: 55,
  				mass: 54.98398
  			},
  			{
  				nominal: 56,
  				mass: 55.97254
  			},
  			{
  				nominal: 57,
  				mass: 56.96506
  			},
  			{
  				nominal: 58,
  				mass: 57.954591
  			},
  			{
  				nominal: 59,
  				mass: 58.94931266
  			},
  			{
  				nominal: 60,
  				mass: 59.9418421
  			},
  			{
  				nominal: 61,
  				mass: 60.939507
  			},
  			{
  				nominal: 62,
  				mass: 61.93433397
  			},
  			{
  				nominal: 63,
  				mass: 62.9332115
  			},
  			{
  				nominal: 64,
  				mass: 63.92914201,
  				abundance: 0.4917
  			},
  			{
  				nominal: 65,
  				mass: 64.92924077
  			},
  			{
  				nominal: 66,
  				mass: 65.92603381,
  				abundance: 0.2773
  			},
  			{
  				nominal: 67,
  				mass: 66.92712775,
  				abundance: 0.0404
  			},
  			{
  				nominal: 68,
  				mass: 67.92484455,
  				abundance: 0.1845
  			},
  			{
  				nominal: 69,
  				mass: 68.9265507
  			},
  			{
  				nominal: 70,
  				mass: 69.9253192,
  				abundance: 0.0061
  			},
  			{
  				nominal: 71,
  				mass: 70.9277196
  			},
  			{
  				nominal: 72,
  				mass: 71.9268428
  			},
  			{
  				nominal: 73,
  				mass: 72.9295826
  			},
  			{
  				nominal: 74,
  				mass: 73.9294073
  			},
  			{
  				nominal: 75,
  				mass: 74.9328402
  			},
  			{
  				nominal: 76,
  				mass: 75.933115
  			},
  			{
  				nominal: 77,
  				mass: 76.9368872
  			},
  			{
  				nominal: 78,
  				mass: 77.9382892
  			},
  			{
  				nominal: 79,
  				mass: 78.9426381
  			},
  			{
  				nominal: 80,
  				mass: 79.9445529
  			},
  			{
  				nominal: 81,
  				mass: 80.9504026
  			},
  			{
  				nominal: 82,
  				mass: 81.95426
  			},
  			{
  				nominal: 83,
  				mass: 82.96056
  			},
  			{
  				nominal: 84,
  				mass: 83.96521
  			},
  			{
  				nominal: 85,
  				mass: 84.97226
  			}
  		],
  		symbol: "Zn",
  		mass: 65.37778252952499,
  		name: "Zinc",
  		monoisotopicMass: 63.92914201
  	},
  	{
  		number: 31,
  		isotopes: [
  			{
  				nominal: 56,
  				mass: 55.99536
  			},
  			{
  				nominal: 57,
  				mass: 56.9832
  			},
  			{
  				nominal: 58,
  				mass: 57.97478
  			},
  			{
  				nominal: 59,
  				mass: 58.96353
  			},
  			{
  				nominal: 60,
  				mass: 59.95729
  			},
  			{
  				nominal: 61,
  				mass: 60.949399
  			},
  			{
  				nominal: 62,
  				mass: 61.94419025
  			},
  			{
  				nominal: 63,
  				mass: 62.9392942
  			},
  			{
  				nominal: 64,
  				mass: 63.9368404
  			},
  			{
  				nominal: 65,
  				mass: 64.93273459
  			},
  			{
  				nominal: 66,
  				mass: 65.9315894
  			},
  			{
  				nominal: 67,
  				mass: 66.9282025
  			},
  			{
  				nominal: 68,
  				mass: 67.9279805
  			},
  			{
  				nominal: 69,
  				mass: 68.9255735,
  				abundance: 0.60108
  			},
  			{
  				nominal: 70,
  				mass: 69.9260219
  			},
  			{
  				nominal: 71,
  				mass: 70.92470258,
  				abundance: 0.39892
  			},
  			{
  				nominal: 72,
  				mass: 71.92636747
  			},
  			{
  				nominal: 73,
  				mass: 72.9251747
  			},
  			{
  				nominal: 74,
  				mass: 73.9269457
  			},
  			{
  				nominal: 75,
  				mass: 74.9265002
  			},
  			{
  				nominal: 76,
  				mass: 75.9288276
  			},
  			{
  				nominal: 77,
  				mass: 76.9291543
  			},
  			{
  				nominal: 78,
  				mass: 77.9316088
  			},
  			{
  				nominal: 79,
  				mass: 78.9328523
  			},
  			{
  				nominal: 80,
  				mass: 79.9364208
  			},
  			{
  				nominal: 81,
  				mass: 80.9381338
  			},
  			{
  				nominal: 82,
  				mass: 81.9431765
  			},
  			{
  				nominal: 83,
  				mass: 82.9471203
  			},
  			{
  				nominal: 84,
  				mass: 83.95246
  			},
  			{
  				nominal: 85,
  				mass: 84.95699
  			},
  			{
  				nominal: 86,
  				mass: 85.96301
  			},
  			{
  				nominal: 87,
  				mass: 86.96824
  			}
  		],
  		symbol: "Ga",
  		mass: 69.7230660725936,
  		name: "Gallium",
  		monoisotopicMass: 68.9255735
  	},
  	{
  		number: 32,
  		isotopes: [
  			{
  				nominal: 58,
  				mass: 57.99172
  			},
  			{
  				nominal: 59,
  				mass: 58.98249
  			},
  			{
  				nominal: 60,
  				mass: 59.97036
  			},
  			{
  				nominal: 61,
  				mass: 60.96379
  			},
  			{
  				nominal: 62,
  				mass: 61.95502
  			},
  			{
  				nominal: 63,
  				mass: 62.949628
  			},
  			{
  				nominal: 64,
  				mass: 63.9416899
  			},
  			{
  				nominal: 65,
  				mass: 64.9393681
  			},
  			{
  				nominal: 66,
  				mass: 65.9338621
  			},
  			{
  				nominal: 67,
  				mass: 66.9327339
  			},
  			{
  				nominal: 68,
  				mass: 67.9280953
  			},
  			{
  				nominal: 69,
  				mass: 68.9279645
  			},
  			{
  				nominal: 70,
  				mass: 69.92424875,
  				abundance: 0.2057
  			},
  			{
  				nominal: 71,
  				mass: 70.92495233
  			},
  			{
  				nominal: 72,
  				mass: 71.922075826,
  				abundance: 0.2745
  			},
  			{
  				nominal: 73,
  				mass: 72.923458956,
  				abundance: 0.0775
  			},
  			{
  				nominal: 74,
  				mass: 73.921177761,
  				abundance: 0.365
  			},
  			{
  				nominal: 75,
  				mass: 74.92285837
  			},
  			{
  				nominal: 76,
  				mass: 75.921402726,
  				abundance: 0.0773
  			},
  			{
  				nominal: 77,
  				mass: 76.923549843
  			},
  			{
  				nominal: 78,
  				mass: 77.9228529
  			},
  			{
  				nominal: 79,
  				mass: 78.92536
  			},
  			{
  				nominal: 80,
  				mass: 79.9253508
  			},
  			{
  				nominal: 81,
  				mass: 80.9288329
  			},
  			{
  				nominal: 82,
  				mass: 81.929774
  			},
  			{
  				nominal: 83,
  				mass: 82.9345391
  			},
  			{
  				nominal: 84,
  				mass: 83.9375751
  			},
  			{
  				nominal: 85,
  				mass: 84.9429697
  			},
  			{
  				nominal: 86,
  				mass: 85.94658
  			},
  			{
  				nominal: 87,
  				mass: 86.95268
  			},
  			{
  				nominal: 88,
  				mass: 87.95691
  			},
  			{
  				nominal: 89,
  				mass: 88.96379
  			},
  			{
  				nominal: 90,
  				mass: 89.96863
  			}
  		],
  		symbol: "Ge",
  		mass: 72.6275501646868,
  		name: "Germanium",
  		monoisotopicMass: 73.921177761
  	},
  	{
  		number: 33,
  		isotopes: [
  			{
  				nominal: 60,
  				mass: 59.99388
  			},
  			{
  				nominal: 61,
  				mass: 60.98112
  			},
  			{
  				nominal: 62,
  				mass: 61.97361
  			},
  			{
  				nominal: 63,
  				mass: 62.9639
  			},
  			{
  				nominal: 64,
  				mass: 63.95743
  			},
  			{
  				nominal: 65,
  				mass: 64.949611
  			},
  			{
  				nominal: 66,
  				mass: 65.9441488
  			},
  			{
  				nominal: 67,
  				mass: 66.93925111
  			},
  			{
  				nominal: 68,
  				mass: 67.9367741
  			},
  			{
  				nominal: 69,
  				mass: 68.932246
  			},
  			{
  				nominal: 70,
  				mass: 69.930926
  			},
  			{
  				nominal: 71,
  				mass: 70.9271138
  			},
  			{
  				nominal: 72,
  				mass: 71.9267523
  			},
  			{
  				nominal: 73,
  				mass: 72.9238291
  			},
  			{
  				nominal: 74,
  				mass: 73.9239286
  			},
  			{
  				nominal: 75,
  				mass: 74.92159457,
  				abundance: 1
  			},
  			{
  				nominal: 76,
  				mass: 75.92239202
  			},
  			{
  				nominal: 77,
  				mass: 76.9206476
  			},
  			{
  				nominal: 78,
  				mass: 77.921828
  			},
  			{
  				nominal: 79,
  				mass: 78.9209484
  			},
  			{
  				nominal: 80,
  				mass: 79.9224746
  			},
  			{
  				nominal: 81,
  				mass: 80.9221323
  			},
  			{
  				nominal: 82,
  				mass: 81.9247412
  			},
  			{
  				nominal: 83,
  				mass: 82.9252069
  			},
  			{
  				nominal: 84,
  				mass: 83.9293033
  			},
  			{
  				nominal: 85,
  				mass: 84.9321637
  			},
  			{
  				nominal: 86,
  				mass: 85.9367015
  			},
  			{
  				nominal: 87,
  				mass: 86.9402917
  			},
  			{
  				nominal: 88,
  				mass: 87.94555
  			},
  			{
  				nominal: 89,
  				mass: 88.94976
  			},
  			{
  				nominal: 90,
  				mass: 89.95563
  			},
  			{
  				nominal: 91,
  				mass: 90.96039
  			},
  			{
  				nominal: 92,
  				mass: 91.96674
  			}
  		],
  		symbol: "As",
  		mass: 74.92159457,
  		name: "Arsenic",
  		monoisotopicMass: 74.92159457
  	},
  	{
  		number: 34,
  		isotopes: [
  			{
  				nominal: 64,
  				mass: 63.97109
  			},
  			{
  				nominal: 65,
  				mass: 64.9644
  			},
  			{
  				nominal: 66,
  				mass: 65.95559
  			},
  			{
  				nominal: 67,
  				mass: 66.949994
  			},
  			{
  				nominal: 68,
  				mass: 67.94182524
  			},
  			{
  				nominal: 69,
  				mass: 68.9394148
  			},
  			{
  				nominal: 70,
  				mass: 69.9335155
  			},
  			{
  				nominal: 71,
  				mass: 70.9322094
  			},
  			{
  				nominal: 72,
  				mass: 71.9271405
  			},
  			{
  				nominal: 73,
  				mass: 72.9267549
  			},
  			{
  				nominal: 74,
  				mass: 73.922475934,
  				abundance: 0.0089
  			},
  			{
  				nominal: 75,
  				mass: 74.92252287
  			},
  			{
  				nominal: 76,
  				mass: 75.919213704,
  				abundance: 0.0937
  			},
  			{
  				nominal: 77,
  				mass: 76.919914154,
  				abundance: 0.0763
  			},
  			{
  				nominal: 78,
  				mass: 77.91730928,
  				abundance: 0.2377
  			},
  			{
  				nominal: 79,
  				mass: 78.91849929
  			},
  			{
  				nominal: 80,
  				mass: 79.9165218,
  				abundance: 0.4961
  			},
  			{
  				nominal: 81,
  				mass: 80.917993
  			},
  			{
  				nominal: 82,
  				mass: 81.9166995,
  				abundance: 0.0873
  			},
  			{
  				nominal: 83,
  				mass: 82.9191186
  			},
  			{
  				nominal: 84,
  				mass: 83.9184668
  			},
  			{
  				nominal: 85,
  				mass: 84.9222608
  			},
  			{
  				nominal: 86,
  				mass: 85.9243117
  			},
  			{
  				nominal: 87,
  				mass: 86.9286886
  			},
  			{
  				nominal: 88,
  				mass: 87.9314175
  			},
  			{
  				nominal: 89,
  				mass: 88.9366691
  			},
  			{
  				nominal: 90,
  				mass: 89.9401
  			},
  			{
  				nominal: 91,
  				mass: 90.94596
  			},
  			{
  				nominal: 92,
  				mass: 91.94984
  			},
  			{
  				nominal: 93,
  				mass: 92.95629
  			},
  			{
  				nominal: 94,
  				mass: 93.96049
  			},
  			{
  				nominal: 95,
  				mass: 94.9673
  			}
  		],
  		symbol: "Se",
  		mass: 78.95938855701361,
  		name: "Selenium",
  		monoisotopicMass: 79.9165218
  	},
  	{
  		number: 35,
  		isotopes: [
  			{
  				nominal: 67,
  				mass: 66.96465
  			},
  			{
  				nominal: 68,
  				mass: 67.95873
  			},
  			{
  				nominal: 69,
  				mass: 68.950497
  			},
  			{
  				nominal: 70,
  				mass: 69.944792
  			},
  			{
  				nominal: 71,
  				mass: 70.9393422
  			},
  			{
  				nominal: 72,
  				mass: 71.9365886
  			},
  			{
  				nominal: 73,
  				mass: 72.9316715
  			},
  			{
  				nominal: 74,
  				mass: 73.9299102
  			},
  			{
  				nominal: 75,
  				mass: 74.9258105
  			},
  			{
  				nominal: 76,
  				mass: 75.924542
  			},
  			{
  				nominal: 77,
  				mass: 76.9213792
  			},
  			{
  				nominal: 78,
  				mass: 77.9211459
  			},
  			{
  				nominal: 79,
  				mass: 78.9183376,
  				abundance: 0.5069
  			},
  			{
  				nominal: 80,
  				mass: 79.9185298
  			},
  			{
  				nominal: 81,
  				mass: 80.9162897,
  				abundance: 0.4931
  			},
  			{
  				nominal: 82,
  				mass: 81.9168032
  			},
  			{
  				nominal: 83,
  				mass: 82.9151756
  			},
  			{
  				nominal: 84,
  				mass: 83.916496
  			},
  			{
  				nominal: 85,
  				mass: 84.9156458
  			},
  			{
  				nominal: 86,
  				mass: 85.9188054
  			},
  			{
  				nominal: 87,
  				mass: 86.920674
  			},
  			{
  				nominal: 88,
  				mass: 87.9240833
  			},
  			{
  				nominal: 89,
  				mass: 88.9267046
  			},
  			{
  				nominal: 90,
  				mass: 89.9312928
  			},
  			{
  				nominal: 91,
  				mass: 90.9343986
  			},
  			{
  				nominal: 92,
  				mass: 91.9396316
  			},
  			{
  				nominal: 93,
  				mass: 92.94313
  			},
  			{
  				nominal: 94,
  				mass: 93.9489
  			},
  			{
  				nominal: 95,
  				mass: 94.95301
  			},
  			{
  				nominal: 96,
  				mass: 95.95903
  			},
  			{
  				nominal: 97,
  				mass: 96.96344
  			},
  			{
  				nominal: 98,
  				mass: 97.96946
  			}
  		],
  		symbol: "Br",
  		mass: 79.90352778050999,
  		name: "Bromine",
  		monoisotopicMass: 78.9183376
  	},
  	{
  		number: 36,
  		isotopes: [
  			{
  				nominal: 69,
  				mass: 68.96518
  			},
  			{
  				nominal: 70,
  				mass: 69.95604
  			},
  			{
  				nominal: 71,
  				mass: 70.95027
  			},
  			{
  				nominal: 72,
  				mass: 71.9420924
  			},
  			{
  				nominal: 73,
  				mass: 72.9392892
  			},
  			{
  				nominal: 74,
  				mass: 73.933084
  			},
  			{
  				nominal: 75,
  				mass: 74.9309457
  			},
  			{
  				nominal: 76,
  				mass: 75.9259103
  			},
  			{
  				nominal: 77,
  				mass: 76.92467
  			},
  			{
  				nominal: 78,
  				mass: 77.92036494,
  				abundance: 0.00355
  			},
  			{
  				nominal: 79,
  				mass: 78.9200829
  			},
  			{
  				nominal: 80,
  				mass: 79.91637808,
  				abundance: 0.02286
  			},
  			{
  				nominal: 81,
  				mass: 80.9165912
  			},
  			{
  				nominal: 82,
  				mass: 81.91348273,
  				abundance: 0.11593
  			},
  			{
  				nominal: 83,
  				mass: 82.91412716,
  				abundance: 0.115
  			},
  			{
  				nominal: 84,
  				mass: 83.9114977282,
  				abundance: 0.56987
  			},
  			{
  				nominal: 85,
  				mass: 84.9125273
  			},
  			{
  				nominal: 86,
  				mass: 85.9106106269,
  				abundance: 0.17279
  			},
  			{
  				nominal: 87,
  				mass: 86.91335476
  			},
  			{
  				nominal: 88,
  				mass: 87.9144479
  			},
  			{
  				nominal: 89,
  				mass: 88.9178355
  			},
  			{
  				nominal: 90,
  				mass: 89.9195279
  			},
  			{
  				nominal: 91,
  				mass: 90.9238063
  			},
  			{
  				nominal: 92,
  				mass: 91.9261731
  			},
  			{
  				nominal: 93,
  				mass: 92.9311472
  			},
  			{
  				nominal: 94,
  				mass: 93.93414
  			},
  			{
  				nominal: 95,
  				mass: 94.939711
  			},
  			{
  				nominal: 96,
  				mass: 95.943017
  			},
  			{
  				nominal: 97,
  				mass: 96.94909
  			},
  			{
  				nominal: 98,
  				mass: 97.95243
  			},
  			{
  				nominal: 99,
  				mass: 98.95839
  			},
  			{
  				nominal: 100,
  				mass: 99.96237
  			},
  			{
  				nominal: 101,
  				mass: 100.96873
  			}
  		],
  		symbol: "Kr",
  		mass: 83.7979999953261,
  		name: "Krypton",
  		monoisotopicMass: 83.9114977282
  	},
  	{
  		number: 37,
  		isotopes: [
  			{
  				nominal: 71,
  				mass: 70.96532
  			},
  			{
  				nominal: 72,
  				mass: 71.95908
  			},
  			{
  				nominal: 73,
  				mass: 72.95053
  			},
  			{
  				nominal: 74,
  				mass: 73.9442659
  			},
  			{
  				nominal: 75,
  				mass: 74.9385732
  			},
  			{
  				nominal: 76,
  				mass: 75.935073
  			},
  			{
  				nominal: 77,
  				mass: 76.9304016
  			},
  			{
  				nominal: 78,
  				mass: 77.9281419
  			},
  			{
  				nominal: 79,
  				mass: 78.9239899
  			},
  			{
  				nominal: 80,
  				mass: 79.9225164
  			},
  			{
  				nominal: 81,
  				mass: 80.9189939
  			},
  			{
  				nominal: 82,
  				mass: 81.918209
  			},
  			{
  				nominal: 83,
  				mass: 82.9151142
  			},
  			{
  				nominal: 84,
  				mass: 83.9143752
  			},
  			{
  				nominal: 85,
  				mass: 84.9117897379,
  				abundance: 0.7217
  			},
  			{
  				nominal: 86,
  				mass: 85.91116743
  			},
  			{
  				nominal: 87,
  				mass: 86.909180531,
  				abundance: 0.2783
  			},
  			{
  				nominal: 88,
  				mass: 87.91131559
  			},
  			{
  				nominal: 89,
  				mass: 88.9122783
  			},
  			{
  				nominal: 90,
  				mass: 89.9147985
  			},
  			{
  				nominal: 91,
  				mass: 90.9165372
  			},
  			{
  				nominal: 92,
  				mass: 91.9197284
  			},
  			{
  				nominal: 93,
  				mass: 92.9220393
  			},
  			{
  				nominal: 94,
  				mass: 93.9263948
  			},
  			{
  				nominal: 95,
  				mass: 94.92926
  			},
  			{
  				nominal: 96,
  				mass: 95.9341334
  			},
  			{
  				nominal: 97,
  				mass: 96.9371771
  			},
  			{
  				nominal: 98,
  				mass: 97.9416869
  			},
  			{
  				nominal: 99,
  				mass: 98.94503
  			},
  			{
  				nominal: 100,
  				mass: 99.95003
  			},
  			{
  				nominal: 101,
  				mass: 100.95404
  			},
  			{
  				nominal: 102,
  				mass: 101.95952
  			},
  			{
  				nominal: 103,
  				mass: 102.96392
  			}
  		],
  		symbol: "Rb",
  		mass: 85.46766359561973,
  		name: "Rubidium",
  		monoisotopicMass: 84.9117897379
  	},
  	{
  		number: 38,
  		isotopes: [
  			{
  				nominal: 73,
  				mass: 72.9657
  			},
  			{
  				nominal: 74,
  				mass: 73.95617
  			},
  			{
  				nominal: 75,
  				mass: 74.94995
  			},
  			{
  				nominal: 76,
  				mass: 75.941763
  			},
  			{
  				nominal: 77,
  				mass: 76.9379455
  			},
  			{
  				nominal: 78,
  				mass: 77.93218
  			},
  			{
  				nominal: 79,
  				mass: 78.9297077
  			},
  			{
  				nominal: 80,
  				mass: 79.9245175
  			},
  			{
  				nominal: 81,
  				mass: 80.9232114
  			},
  			{
  				nominal: 82,
  				mass: 81.9183999
  			},
  			{
  				nominal: 83,
  				mass: 82.9175544
  			},
  			{
  				nominal: 84,
  				mass: 83.9134191,
  				abundance: 0.0056
  			},
  			{
  				nominal: 85,
  				mass: 84.912932
  			},
  			{
  				nominal: 86,
  				mass: 85.9092606,
  				abundance: 0.0986
  			},
  			{
  				nominal: 87,
  				mass: 86.9088775,
  				abundance: 0.07
  			},
  			{
  				nominal: 88,
  				mass: 87.9056125,
  				abundance: 0.8258
  			},
  			{
  				nominal: 89,
  				mass: 88.9074511
  			},
  			{
  				nominal: 90,
  				mass: 89.90773
  			},
  			{
  				nominal: 91,
  				mass: 90.9101954
  			},
  			{
  				nominal: 92,
  				mass: 91.9110382
  			},
  			{
  				nominal: 93,
  				mass: 92.9140242
  			},
  			{
  				nominal: 94,
  				mass: 93.9153556
  			},
  			{
  				nominal: 95,
  				mass: 94.9193529
  			},
  			{
  				nominal: 96,
  				mass: 95.9217066
  			},
  			{
  				nominal: 97,
  				mass: 96.926374
  			},
  			{
  				nominal: 98,
  				mass: 97.9286888
  			},
  			{
  				nominal: 99,
  				mass: 98.9328907
  			},
  			{
  				nominal: 100,
  				mass: 99.93577
  			},
  			{
  				nominal: 101,
  				mass: 100.940352
  			},
  			{
  				nominal: 102,
  				mass: 101.943791
  			},
  			{
  				nominal: 103,
  				mass: 102.94909
  			},
  			{
  				nominal: 104,
  				mass: 103.95265
  			},
  			{
  				nominal: 105,
  				mass: 104.95855
  			},
  			{
  				nominal: 106,
  				mass: 105.96265
  			},
  			{
  				nominal: 107,
  				mass: 106.96897
  			}
  		],
  		symbol: "Sr",
  		mass: 87.61664446962,
  		name: "Strontium",
  		monoisotopicMass: 87.9056125
  	},
  	{
  		number: 39,
  		isotopes: [
  			{
  				nominal: 76,
  				mass: 75.95856
  			},
  			{
  				nominal: 77,
  				mass: 76.949781
  			},
  			{
  				nominal: 78,
  				mass: 77.94361
  			},
  			{
  				nominal: 79,
  				mass: 78.93735
  			},
  			{
  				nominal: 80,
  				mass: 79.9343561
  			},
  			{
  				nominal: 81,
  				mass: 80.9294556
  			},
  			{
  				nominal: 82,
  				mass: 81.9269314
  			},
  			{
  				nominal: 83,
  				mass: 82.922485
  			},
  			{
  				nominal: 84,
  				mass: 83.9206721
  			},
  			{
  				nominal: 85,
  				mass: 84.916433
  			},
  			{
  				nominal: 86,
  				mass: 85.914886
  			},
  			{
  				nominal: 87,
  				mass: 86.9108761
  			},
  			{
  				nominal: 88,
  				mass: 87.9095016
  			},
  			{
  				nominal: 89,
  				mass: 88.9058403,
  				abundance: 1
  			},
  			{
  				nominal: 90,
  				mass: 89.9071439
  			},
  			{
  				nominal: 91,
  				mass: 90.9072974
  			},
  			{
  				nominal: 92,
  				mass: 91.9089451
  			},
  			{
  				nominal: 93,
  				mass: 92.909578
  			},
  			{
  				nominal: 94,
  				mass: 93.9115906
  			},
  			{
  				nominal: 95,
  				mass: 94.9128161
  			},
  			{
  				nominal: 96,
  				mass: 95.9158968
  			},
  			{
  				nominal: 97,
  				mass: 96.9182741
  			},
  			{
  				nominal: 98,
  				mass: 97.9223821
  			},
  			{
  				nominal: 99,
  				mass: 98.924148
  			},
  			{
  				nominal: 100,
  				mass: 99.927715
  			},
  			{
  				nominal: 101,
  				mass: 100.9301477
  			},
  			{
  				nominal: 102,
  				mass: 101.9343277
  			},
  			{
  				nominal: 103,
  				mass: 102.937243
  			},
  			{
  				nominal: 104,
  				mass: 103.94196
  			},
  			{
  				nominal: 105,
  				mass: 104.94544
  			},
  			{
  				nominal: 106,
  				mass: 105.95056
  			},
  			{
  				nominal: 107,
  				mass: 106.95452
  			},
  			{
  				nominal: 108,
  				mass: 107.95996
  			},
  			{
  				nominal: 109,
  				mass: 108.96436
  			}
  		],
  		symbol: "Y",
  		mass: 88.9058403,
  		name: "Yttrium",
  		monoisotopicMass: 88.9058403
  	},
  	{
  		number: 40,
  		isotopes: [
  			{
  				nominal: 78,
  				mass: 77.95566
  			},
  			{
  				nominal: 79,
  				mass: 78.94948
  			},
  			{
  				nominal: 80,
  				mass: 79.9404
  			},
  			{
  				nominal: 81,
  				mass: 80.93731
  			},
  			{
  				nominal: 82,
  				mass: 81.93135
  			},
  			{
  				nominal: 83,
  				mass: 82.9292421
  			},
  			{
  				nominal: 84,
  				mass: 83.9233269
  			},
  			{
  				nominal: 85,
  				mass: 84.9214444
  			},
  			{
  				nominal: 86,
  				mass: 85.9162972
  			},
  			{
  				nominal: 87,
  				mass: 86.914818
  			},
  			{
  				nominal: 88,
  				mass: 87.9102213
  			},
  			{
  				nominal: 89,
  				mass: 88.9088814
  			},
  			{
  				nominal: 90,
  				mass: 89.9046977,
  				abundance: 0.5145
  			},
  			{
  				nominal: 91,
  				mass: 90.9056396,
  				abundance: 0.1122
  			},
  			{
  				nominal: 92,
  				mass: 91.9050347,
  				abundance: 0.1715
  			},
  			{
  				nominal: 93,
  				mass: 92.9064699
  			},
  			{
  				nominal: 94,
  				mass: 93.9063108,
  				abundance: 0.1738
  			},
  			{
  				nominal: 95,
  				mass: 94.9080385
  			},
  			{
  				nominal: 96,
  				mass: 95.9082714,
  				abundance: 0.028
  			},
  			{
  				nominal: 97,
  				mass: 96.9109512
  			},
  			{
  				nominal: 98,
  				mass: 97.9127289
  			},
  			{
  				nominal: 99,
  				mass: 98.916667
  			},
  			{
  				nominal: 100,
  				mass: 99.9180006
  			},
  			{
  				nominal: 101,
  				mass: 100.921448
  			},
  			{
  				nominal: 102,
  				mass: 101.9231409
  			},
  			{
  				nominal: 103,
  				mass: 102.927191
  			},
  			{
  				nominal: 104,
  				mass: 103.929436
  			},
  			{
  				nominal: 105,
  				mass: 104.934008
  			},
  			{
  				nominal: 106,
  				mass: 105.93676
  			},
  			{
  				nominal: 107,
  				mass: 106.94174
  			},
  			{
  				nominal: 108,
  				mass: 107.94487
  			},
  			{
  				nominal: 109,
  				mass: 108.95041
  			},
  			{
  				nominal: 110,
  				mass: 109.95396
  			},
  			{
  				nominal: 111,
  				mass: 110.95968
  			},
  			{
  				nominal: 112,
  				mass: 111.9637
  			}
  		],
  		symbol: "Zr",
  		mass: 91.22364159706,
  		name: "Zirconium",
  		monoisotopicMass: 89.9046977
  	},
  	{
  		number: 41,
  		isotopes: [
  			{
  				nominal: 81,
  				mass: 80.9496
  			},
  			{
  				nominal: 82,
  				mass: 81.94396
  			},
  			{
  				nominal: 83,
  				mass: 82.93729
  			},
  			{
  				nominal: 84,
  				mass: 83.93449
  			},
  			{
  				nominal: 85,
  				mass: 84.9288458
  			},
  			{
  				nominal: 86,
  				mass: 85.9257828
  			},
  			{
  				nominal: 87,
  				mass: 86.9206937
  			},
  			{
  				nominal: 88,
  				mass: 87.918222
  			},
  			{
  				nominal: 89,
  				mass: 88.913445
  			},
  			{
  				nominal: 90,
  				mass: 89.9112584
  			},
  			{
  				nominal: 91,
  				mass: 90.9069897
  			},
  			{
  				nominal: 92,
  				mass: 91.9071881
  			},
  			{
  				nominal: 93,
  				mass: 92.906373,
  				abundance: 1
  			},
  			{
  				nominal: 94,
  				mass: 93.9072788
  			},
  			{
  				nominal: 95,
  				mass: 94.9068324
  			},
  			{
  				nominal: 96,
  				mass: 95.9080973
  			},
  			{
  				nominal: 97,
  				mass: 96.9080959
  			},
  			{
  				nominal: 98,
  				mass: 97.9103265
  			},
  			{
  				nominal: 99,
  				mass: 98.911613
  			},
  			{
  				nominal: 100,
  				mass: 99.9143276
  			},
  			{
  				nominal: 101,
  				mass: 100.9153103
  			},
  			{
  				nominal: 102,
  				mass: 101.9180772
  			},
  			{
  				nominal: 103,
  				mass: 102.9194572
  			},
  			{
  				nominal: 104,
  				mass: 103.9228925
  			},
  			{
  				nominal: 105,
  				mass: 104.9249465
  			},
  			{
  				nominal: 106,
  				mass: 105.9289317
  			},
  			{
  				nominal: 107,
  				mass: 106.9315937
  			},
  			{
  				nominal: 108,
  				mass: 107.9360748
  			},
  			{
  				nominal: 109,
  				mass: 108.93922
  			},
  			{
  				nominal: 110,
  				mass: 109.94403
  			},
  			{
  				nominal: 111,
  				mass: 110.94753
  			},
  			{
  				nominal: 112,
  				mass: 111.95247
  			},
  			{
  				nominal: 113,
  				mass: 112.95651
  			},
  			{
  				nominal: 114,
  				mass: 113.96201
  			},
  			{
  				nominal: 115,
  				mass: 114.96634
  			}
  		],
  		symbol: "Nb",
  		mass: 92.906373,
  		name: "Niobium",
  		monoisotopicMass: 92.906373
  	},
  	{
  		number: 42,
  		isotopes: [
  			{
  				nominal: 83,
  				mass: 82.94988
  			},
  			{
  				nominal: 84,
  				mass: 83.94149
  			},
  			{
  				nominal: 85,
  				mass: 84.938261
  			},
  			{
  				nominal: 86,
  				mass: 85.9311748
  			},
  			{
  				nominal: 87,
  				mass: 86.9281962
  			},
  			{
  				nominal: 88,
  				mass: 87.9219678
  			},
  			{
  				nominal: 89,
  				mass: 88.9194682
  			},
  			{
  				nominal: 90,
  				mass: 89.9139309
  			},
  			{
  				nominal: 91,
  				mass: 90.9117453
  			},
  			{
  				nominal: 92,
  				mass: 91.90680796,
  				abundance: 0.1453
  			},
  			{
  				nominal: 93,
  				mass: 92.90680958
  			},
  			{
  				nominal: 94,
  				mass: 93.9050849,
  				abundance: 0.0915
  			},
  			{
  				nominal: 95,
  				mass: 94.90583877,
  				abundance: 0.1584
  			},
  			{
  				nominal: 96,
  				mass: 95.90467612,
  				abundance: 0.1667
  			},
  			{
  				nominal: 97,
  				mass: 96.90601812,
  				abundance: 0.096
  			},
  			{
  				nominal: 98,
  				mass: 97.90540482,
  				abundance: 0.2439
  			},
  			{
  				nominal: 99,
  				mass: 98.90770851
  			},
  			{
  				nominal: 100,
  				mass: 99.9074718,
  				abundance: 0.0982
  			},
  			{
  				nominal: 101,
  				mass: 100.9103414
  			},
  			{
  				nominal: 102,
  				mass: 101.9102834
  			},
  			{
  				nominal: 103,
  				mass: 102.913079
  			},
  			{
  				nominal: 104,
  				mass: 103.9137344
  			},
  			{
  				nominal: 105,
  				mass: 104.916969
  			},
  			{
  				nominal: 106,
  				mass: 105.918259
  			},
  			{
  				nominal: 107,
  				mass: 106.922106
  			},
  			{
  				nominal: 108,
  				mass: 107.924033
  			},
  			{
  				nominal: 109,
  				mass: 108.928424
  			},
  			{
  				nominal: 110,
  				mass: 109.930704
  			},
  			{
  				nominal: 111,
  				mass: 110.935654
  			},
  			{
  				nominal: 112,
  				mass: 111.93831
  			},
  			{
  				nominal: 113,
  				mass: 112.94335
  			},
  			{
  				nominal: 114,
  				mass: 113.94653
  			},
  			{
  				nominal: 115,
  				mass: 114.95196
  			},
  			{
  				nominal: 116,
  				mass: 115.95545
  			},
  			{
  				nominal: 117,
  				mass: 116.96117
  			}
  		],
  		symbol: "Mo",
  		mass: 95.959788541188,
  		name: "Molybdenum",
  		monoisotopicMass: 97.90540482
  	},
  	{
  		number: 43,
  		isotopes: [
  			{
  				nominal: 85,
  				mass: 84.95058
  			},
  			{
  				nominal: 86,
  				mass: 85.94493
  			},
  			{
  				nominal: 87,
  				mass: 86.9380672
  			},
  			{
  				nominal: 88,
  				mass: 87.93378
  			},
  			{
  				nominal: 89,
  				mass: 88.9276487
  			},
  			{
  				nominal: 90,
  				mass: 89.9240739
  			},
  			{
  				nominal: 91,
  				mass: 90.9184254
  			},
  			{
  				nominal: 92,
  				mass: 91.9152698
  			},
  			{
  				nominal: 93,
  				mass: 92.910246
  			},
  			{
  				nominal: 94,
  				mass: 93.9096536
  			},
  			{
  				nominal: 95,
  				mass: 94.9076536
  			},
  			{
  				nominal: 96,
  				mass: 95.907868
  			},
  			{
  				nominal: 97,
  				mass: 96.9063667
  			},
  			{
  				nominal: 98,
  				mass: 97.9072124
  			},
  			{
  				nominal: 99,
  				mass: 98.9062508
  			},
  			{
  				nominal: 100,
  				mass: 99.9076539
  			},
  			{
  				nominal: 101,
  				mass: 100.907309
  			},
  			{
  				nominal: 102,
  				mass: 101.9092097
  			},
  			{
  				nominal: 103,
  				mass: 102.909176
  			},
  			{
  				nominal: 104,
  				mass: 103.911425
  			},
  			{
  				nominal: 105,
  				mass: 104.911655
  			},
  			{
  				nominal: 106,
  				mass: 105.914358
  			},
  			{
  				nominal: 107,
  				mass: 106.9154606
  			},
  			{
  				nominal: 108,
  				mass: 107.9184957
  			},
  			{
  				nominal: 109,
  				mass: 108.920256
  			},
  			{
  				nominal: 110,
  				mass: 109.923744
  			},
  			{
  				nominal: 111,
  				mass: 110.925901
  			},
  			{
  				nominal: 112,
  				mass: 111.9299458
  			},
  			{
  				nominal: 113,
  				mass: 112.932569
  			},
  			{
  				nominal: 114,
  				mass: 113.93691
  			},
  			{
  				nominal: 115,
  				mass: 114.93998
  			},
  			{
  				nominal: 116,
  				mass: 115.94476
  			},
  			{
  				nominal: 117,
  				mass: 116.94806
  			},
  			{
  				nominal: 118,
  				mass: 117.95299
  			},
  			{
  				nominal: 119,
  				mass: 118.95666
  			},
  			{
  				nominal: 120,
  				mass: 119.96187
  			}
  		],
  		symbol: "Tc",
  		mass: null,
  		name: "Technetium"
  	},
  	{
  		number: 44,
  		isotopes: [
  			{
  				nominal: 87,
  				mass: 86.95069
  			},
  			{
  				nominal: 88,
  				mass: 87.9416
  			},
  			{
  				nominal: 89,
  				mass: 88.93762
  			},
  			{
  				nominal: 90,
  				mass: 89.9303444
  			},
  			{
  				nominal: 91,
  				mass: 90.9267419
  			},
  			{
  				nominal: 92,
  				mass: 91.9202344
  			},
  			{
  				nominal: 93,
  				mass: 92.9171044
  			},
  			{
  				nominal: 94,
  				mass: 93.9113429
  			},
  			{
  				nominal: 95,
  				mass: 94.910406
  			},
  			{
  				nominal: 96,
  				mass: 95.90759025,
  				abundance: 0.0554
  			},
  			{
  				nominal: 97,
  				mass: 96.9075471
  			},
  			{
  				nominal: 98,
  				mass: 97.9052868,
  				abundance: 0.0187
  			},
  			{
  				nominal: 99,
  				mass: 98.9059341,
  				abundance: 0.1276
  			},
  			{
  				nominal: 100,
  				mass: 99.9042143,
  				abundance: 0.126
  			},
  			{
  				nominal: 101,
  				mass: 100.9055769,
  				abundance: 0.1706
  			},
  			{
  				nominal: 102,
  				mass: 101.9043441,
  				abundance: 0.3155
  			},
  			{
  				nominal: 103,
  				mass: 102.9063186
  			},
  			{
  				nominal: 104,
  				mass: 103.9054275,
  				abundance: 0.1862
  			},
  			{
  				nominal: 105,
  				mass: 104.9077476
  			},
  			{
  				nominal: 106,
  				mass: 105.9073291
  			},
  			{
  				nominal: 107,
  				mass: 106.909972
  			},
  			{
  				nominal: 108,
  				mass: 107.910188
  			},
  			{
  				nominal: 109,
  				mass: 108.913326
  			},
  			{
  				nominal: 110,
  				mass: 109.9140407
  			},
  			{
  				nominal: 111,
  				mass: 110.91757
  			},
  			{
  				nominal: 112,
  				mass: 111.918809
  			},
  			{
  				nominal: 113,
  				mass: 112.922844
  			},
  			{
  				nominal: 114,
  				mass: 113.9246136
  			},
  			{
  				nominal: 115,
  				mass: 114.92882
  			},
  			{
  				nominal: 116,
  				mass: 115.9312192
  			},
  			{
  				nominal: 117,
  				mass: 116.9361
  			},
  			{
  				nominal: 118,
  				mass: 117.93853
  			},
  			{
  				nominal: 119,
  				mass: 118.94357
  			},
  			{
  				nominal: 120,
  				mass: 119.94631
  			},
  			{
  				nominal: 121,
  				mass: 120.95164
  			},
  			{
  				nominal: 122,
  				mass: 121.95447
  			},
  			{
  				nominal: 123,
  				mass: 122.95989
  			},
  			{
  				nominal: 124,
  				mass: 123.96305
  			}
  		],
  		symbol: "Ru",
  		mass: 101.06494013916,
  		name: "Ruthenium",
  		monoisotopicMass: 101.9043441
  	},
  	{
  		number: 45,
  		isotopes: [
  			{
  				nominal: 89,
  				mass: 88.95058
  			},
  			{
  				nominal: 90,
  				mass: 89.94422
  			},
  			{
  				nominal: 91,
  				mass: 90.93688
  			},
  			{
  				nominal: 92,
  				mass: 91.9323677
  			},
  			{
  				nominal: 93,
  				mass: 92.9259128
  			},
  			{
  				nominal: 94,
  				mass: 93.9217305
  			},
  			{
  				nominal: 95,
  				mass: 94.9158979
  			},
  			{
  				nominal: 96,
  				mass: 95.914453
  			},
  			{
  				nominal: 97,
  				mass: 96.911329
  			},
  			{
  				nominal: 98,
  				mass: 97.910708
  			},
  			{
  				nominal: 99,
  				mass: 98.9081282
  			},
  			{
  				nominal: 100,
  				mass: 99.908117
  			},
  			{
  				nominal: 101,
  				mass: 100.9061606
  			},
  			{
  				nominal: 102,
  				mass: 101.9068374
  			},
  			{
  				nominal: 103,
  				mass: 102.905498,
  				abundance: 1
  			},
  			{
  				nominal: 104,
  				mass: 103.9066492
  			},
  			{
  				nominal: 105,
  				mass: 104.9056885
  			},
  			{
  				nominal: 106,
  				mass: 105.9072868
  			},
  			{
  				nominal: 107,
  				mass: 106.906748
  			},
  			{
  				nominal: 108,
  				mass: 107.908714
  			},
  			{
  				nominal: 109,
  				mass: 108.9087488
  			},
  			{
  				nominal: 110,
  				mass: 109.911079
  			},
  			{
  				nominal: 111,
  				mass: 110.9116423
  			},
  			{
  				nominal: 112,
  				mass: 111.914403
  			},
  			{
  				nominal: 113,
  				mass: 112.9154393
  			},
  			{
  				nominal: 114,
  				mass: 113.918718
  			},
  			{
  				nominal: 115,
  				mass: 114.9203116
  			},
  			{
  				nominal: 116,
  				mass: 115.924059
  			},
  			{
  				nominal: 117,
  				mass: 116.9260354
  			},
  			{
  				nominal: 118,
  				mass: 117.93034
  			},
  			{
  				nominal: 119,
  				mass: 118.932557
  			},
  			{
  				nominal: 120,
  				mass: 119.93686
  			},
  			{
  				nominal: 121,
  				mass: 120.93942
  			},
  			{
  				nominal: 122,
  				mass: 121.94399
  			},
  			{
  				nominal: 123,
  				mass: 122.94685
  			},
  			{
  				nominal: 124,
  				mass: 123.95151
  			},
  			{
  				nominal: 125,
  				mass: 124.95469
  			},
  			{
  				nominal: 126,
  				mass: 125.95946
  			}
  		],
  		symbol: "Rh",
  		mass: 102.905498,
  		name: "Rhodium",
  		monoisotopicMass: 102.905498
  	},
  	{
  		number: 46,
  		isotopes: [
  			{
  				nominal: 91,
  				mass: 90.95032
  			},
  			{
  				nominal: 92,
  				mass: 91.94088
  			},
  			{
  				nominal: 93,
  				mass: 92.93651
  			},
  			{
  				nominal: 94,
  				mass: 93.9290376
  			},
  			{
  				nominal: 95,
  				mass: 94.9248898
  			},
  			{
  				nominal: 96,
  				mass: 95.9182151
  			},
  			{
  				nominal: 97,
  				mass: 96.916472
  			},
  			{
  				nominal: 98,
  				mass: 97.9126983
  			},
  			{
  				nominal: 99,
  				mass: 98.9117748
  			},
  			{
  				nominal: 100,
  				mass: 99.908505
  			},
  			{
  				nominal: 101,
  				mass: 100.9082864
  			},
  			{
  				nominal: 102,
  				mass: 101.9056022,
  				abundance: 0.0102
  			},
  			{
  				nominal: 103,
  				mass: 102.9060809
  			},
  			{
  				nominal: 104,
  				mass: 103.9040305,
  				abundance: 0.1114
  			},
  			{
  				nominal: 105,
  				mass: 104.9050796,
  				abundance: 0.2233
  			},
  			{
  				nominal: 106,
  				mass: 105.9034804,
  				abundance: 0.2733
  			},
  			{
  				nominal: 107,
  				mass: 106.9051282
  			},
  			{
  				nominal: 108,
  				mass: 107.9038916,
  				abundance: 0.2646
  			},
  			{
  				nominal: 109,
  				mass: 108.9059504
  			},
  			{
  				nominal: 110,
  				mass: 109.9051722,
  				abundance: 0.1172
  			},
  			{
  				nominal: 111,
  				mass: 110.90768968
  			},
  			{
  				nominal: 112,
  				mass: 111.9073297
  			},
  			{
  				nominal: 113,
  				mass: 112.910261
  			},
  			{
  				nominal: 114,
  				mass: 113.9103686
  			},
  			{
  				nominal: 115,
  				mass: 114.913659
  			},
  			{
  				nominal: 116,
  				mass: 115.914297
  			},
  			{
  				nominal: 117,
  				mass: 116.9179547
  			},
  			{
  				nominal: 118,
  				mass: 117.9190667
  			},
  			{
  				nominal: 119,
  				mass: 118.9233402
  			},
  			{
  				nominal: 120,
  				mass: 119.9245511
  			},
  			{
  				nominal: 121,
  				mass: 120.9289503
  			},
  			{
  				nominal: 122,
  				mass: 121.930632
  			},
  			{
  				nominal: 123,
  				mass: 122.93514
  			},
  			{
  				nominal: 124,
  				mass: 123.93714
  			},
  			{
  				nominal: 125,
  				mass: 124.94179
  			},
  			{
  				nominal: 126,
  				mass: 125.94416
  			},
  			{
  				nominal: 127,
  				mass: 126.94907
  			},
  			{
  				nominal: 128,
  				mass: 127.95183
  			}
  		],
  		symbol: "Pd",
  		mass: 106.41532750734,
  		name: "Palladium",
  		monoisotopicMass: 105.9034804
  	},
  	{
  		number: 47,
  		isotopes: [
  			{
  				nominal: 93,
  				mass: 92.95033
  			},
  			{
  				nominal: 94,
  				mass: 93.94373
  			},
  			{
  				nominal: 95,
  				mass: 94.93602
  			},
  			{
  				nominal: 96,
  				mass: 95.930744
  			},
  			{
  				nominal: 97,
  				mass: 96.92397
  			},
  			{
  				nominal: 98,
  				mass: 97.92156
  			},
  			{
  				nominal: 99,
  				mass: 98.9176458
  			},
  			{
  				nominal: 100,
  				mass: 99.9161154
  			},
  			{
  				nominal: 101,
  				mass: 100.912684
  			},
  			{
  				nominal: 102,
  				mass: 101.9117047
  			},
  			{
  				nominal: 103,
  				mass: 102.9089631
  			},
  			{
  				nominal: 104,
  				mass: 103.9086239
  			},
  			{
  				nominal: 105,
  				mass: 104.9065256
  			},
  			{
  				nominal: 106,
  				mass: 105.9066636
  			},
  			{
  				nominal: 107,
  				mass: 106.9050916,
  				abundance: 0.51839
  			},
  			{
  				nominal: 108,
  				mass: 107.9059503
  			},
  			{
  				nominal: 109,
  				mass: 108.9047553,
  				abundance: 0.48161
  			},
  			{
  				nominal: 110,
  				mass: 109.9061102
  			},
  			{
  				nominal: 111,
  				mass: 110.9052959
  			},
  			{
  				nominal: 112,
  				mass: 111.9070486
  			},
  			{
  				nominal: 113,
  				mass: 112.906573
  			},
  			{
  				nominal: 114,
  				mass: 113.908823
  			},
  			{
  				nominal: 115,
  				mass: 114.908767
  			},
  			{
  				nominal: 116,
  				mass: 115.9113868
  			},
  			{
  				nominal: 117,
  				mass: 116.911774
  			},
  			{
  				nominal: 118,
  				mass: 117.9145955
  			},
  			{
  				nominal: 119,
  				mass: 118.91557
  			},
  			{
  				nominal: 120,
  				mass: 119.9187848
  			},
  			{
  				nominal: 121,
  				mass: 120.920125
  			},
  			{
  				nominal: 122,
  				mass: 121.923664
  			},
  			{
  				nominal: 123,
  				mass: 122.925337
  			},
  			{
  				nominal: 124,
  				mass: 123.92893
  			},
  			{
  				nominal: 125,
  				mass: 124.93105
  			},
  			{
  				nominal: 126,
  				mass: 125.93475
  			},
  			{
  				nominal: 127,
  				mass: 126.93711
  			},
  			{
  				nominal: 128,
  				mass: 127.94106
  			},
  			{
  				nominal: 129,
  				mass: 128.94395
  			},
  			{
  				nominal: 130,
  				mass: 129.9507
  			}
  		],
  		symbol: "Ag",
  		mass: 107.868149634557,
  		name: "Silver",
  		monoisotopicMass: 106.9050916
  	},
  	{
  		number: 48,
  		isotopes: [
  			{
  				nominal: 95,
  				mass: 94.94994
  			},
  			{
  				nominal: 96,
  				mass: 95.94034
  			},
  			{
  				nominal: 97,
  				mass: 96.9351
  			},
  			{
  				nominal: 98,
  				mass: 97.927389
  			},
  			{
  				nominal: 99,
  				mass: 98.9249258
  			},
  			{
  				nominal: 100,
  				mass: 99.9203488
  			},
  			{
  				nominal: 101,
  				mass: 100.9185862
  			},
  			{
  				nominal: 102,
  				mass: 101.914482
  			},
  			{
  				nominal: 103,
  				mass: 102.9134165
  			},
  			{
  				nominal: 104,
  				mass: 103.9098564
  			},
  			{
  				nominal: 105,
  				mass: 104.9094639
  			},
  			{
  				nominal: 106,
  				mass: 105.9064599,
  				abundance: 0.0125
  			},
  			{
  				nominal: 107,
  				mass: 106.9066121
  			},
  			{
  				nominal: 108,
  				mass: 107.9041834,
  				abundance: 0.0089
  			},
  			{
  				nominal: 109,
  				mass: 108.9049867
  			},
  			{
  				nominal: 110,
  				mass: 109.90300661,
  				abundance: 0.1249
  			},
  			{
  				nominal: 111,
  				mass: 110.90418287,
  				abundance: 0.128
  			},
  			{
  				nominal: 112,
  				mass: 111.90276287,
  				abundance: 0.2413
  			},
  			{
  				nominal: 113,
  				mass: 112.90440813,
  				abundance: 0.1222
  			},
  			{
  				nominal: 114,
  				mass: 113.90336509,
  				abundance: 0.2873
  			},
  			{
  				nominal: 115,
  				mass: 114.90543751
  			},
  			{
  				nominal: 116,
  				mass: 115.90476315,
  				abundance: 0.0749
  			},
  			{
  				nominal: 117,
  				mass: 116.907226
  			},
  			{
  				nominal: 118,
  				mass: 117.906922
  			},
  			{
  				nominal: 119,
  				mass: 118.909847
  			},
  			{
  				nominal: 120,
  				mass: 119.9098681
  			},
  			{
  				nominal: 121,
  				mass: 120.9129637
  			},
  			{
  				nominal: 122,
  				mass: 121.9134591
  			},
  			{
  				nominal: 123,
  				mass: 122.9168925
  			},
  			{
  				nominal: 124,
  				mass: 123.9176574
  			},
  			{
  				nominal: 125,
  				mass: 124.9212576
  			},
  			{
  				nominal: 126,
  				mass: 125.9224291
  			},
  			{
  				nominal: 127,
  				mass: 126.926472
  			},
  			{
  				nominal: 128,
  				mass: 127.9278129
  			},
  			{
  				nominal: 129,
  				mass: 128.93182
  			},
  			{
  				nominal: 130,
  				mass: 129.93394
  			},
  			{
  				nominal: 131,
  				mass: 130.9406
  			},
  			{
  				nominal: 132,
  				mass: 131.94604
  			},
  			{
  				nominal: 133,
  				mass: 132.95285
  			}
  		],
  		symbol: "Cd",
  		mass: 112.411557818268,
  		name: "Cadmium",
  		monoisotopicMass: 113.90336509
  	},
  	{
  		number: 49,
  		isotopes: [
  			{
  				nominal: 97,
  				mass: 96.94934
  			},
  			{
  				nominal: 98,
  				mass: 97.94214
  			},
  			{
  				nominal: 99,
  				mass: 98.93411
  			},
  			{
  				nominal: 100,
  				mass: 99.93096
  			},
  			{
  				nominal: 101,
  				mass: 100.92634
  			},
  			{
  				nominal: 102,
  				mass: 101.9241071
  			},
  			{
  				nominal: 103,
  				mass: 102.9198819
  			},
  			{
  				nominal: 104,
  				mass: 103.9182145
  			},
  			{
  				nominal: 105,
  				mass: 104.914502
  			},
  			{
  				nominal: 106,
  				mass: 105.913464
  			},
  			{
  				nominal: 107,
  				mass: 106.91029
  			},
  			{
  				nominal: 108,
  				mass: 107.9096935
  			},
  			{
  				nominal: 109,
  				mass: 108.9071514
  			},
  			{
  				nominal: 110,
  				mass: 109.90717
  			},
  			{
  				nominal: 111,
  				mass: 110.9051085
  			},
  			{
  				nominal: 112,
  				mass: 111.9055377
  			},
  			{
  				nominal: 113,
  				mass: 112.90406184,
  				abundance: 0.0429
  			},
  			{
  				nominal: 114,
  				mass: 113.90491791
  			},
  			{
  				nominal: 115,
  				mass: 114.903878776,
  				abundance: 0.9571
  			},
  			{
  				nominal: 116,
  				mass: 115.90525999
  			},
  			{
  				nominal: 117,
  				mass: 116.9045157
  			},
  			{
  				nominal: 118,
  				mass: 117.9063566
  			},
  			{
  				nominal: 119,
  				mass: 118.9058507
  			},
  			{
  				nominal: 120,
  				mass: 119.907967
  			},
  			{
  				nominal: 121,
  				mass: 120.907851
  			},
  			{
  				nominal: 122,
  				mass: 121.910281
  			},
  			{
  				nominal: 123,
  				mass: 122.910434
  			},
  			{
  				nominal: 124,
  				mass: 123.913182
  			},
  			{
  				nominal: 125,
  				mass: 124.913605
  			},
  			{
  				nominal: 126,
  				mass: 125.916507
  			},
  			{
  				nominal: 127,
  				mass: 126.917446
  			},
  			{
  				nominal: 128,
  				mass: 127.9204
  			},
  			{
  				nominal: 129,
  				mass: 128.9218053
  			},
  			{
  				nominal: 130,
  				mass: 129.924977
  			},
  			{
  				nominal: 131,
  				mass: 130.9269715
  			},
  			{
  				nominal: 132,
  				mass: 131.933001
  			},
  			{
  				nominal: 133,
  				mass: 132.93831
  			},
  			{
  				nominal: 134,
  				mass: 133.94454
  			},
  			{
  				nominal: 135,
  				mass: 134.95005
  			}
  		],
  		symbol: "In",
  		mass: 114.81808662944559,
  		name: "Indium",
  		monoisotopicMass: 114.903878776
  	},
  	{
  		number: 50,
  		isotopes: [
  			{
  				nominal: 99,
  				mass: 98.94853
  			},
  			{
  				nominal: 100,
  				mass: 99.9385
  			},
  			{
  				nominal: 101,
  				mass: 100.93526
  			},
  			{
  				nominal: 102,
  				mass: 101.93029
  			},
  			{
  				nominal: 103,
  				mass: 102.928105
  			},
  			{
  				nominal: 104,
  				mass: 103.9231052
  			},
  			{
  				nominal: 105,
  				mass: 104.9212684
  			},
  			{
  				nominal: 106,
  				mass: 105.9169574
  			},
  			{
  				nominal: 107,
  				mass: 106.9157137
  			},
  			{
  				nominal: 108,
  				mass: 107.9118943
  			},
  			{
  				nominal: 109,
  				mass: 108.9112921
  			},
  			{
  				nominal: 110,
  				mass: 109.907845
  			},
  			{
  				nominal: 111,
  				mass: 110.9077401
  			},
  			{
  				nominal: 112,
  				mass: 111.90482387,
  				abundance: 0.0097
  			},
  			{
  				nominal: 113,
  				mass: 112.9051757
  			},
  			{
  				nominal: 114,
  				mass: 113.9027827,
  				abundance: 0.0066
  			},
  			{
  				nominal: 115,
  				mass: 114.903344699,
  				abundance: 0.0034
  			},
  			{
  				nominal: 116,
  				mass: 115.9017428,
  				abundance: 0.1454
  			},
  			{
  				nominal: 117,
  				mass: 116.90295398,
  				abundance: 0.0768
  			},
  			{
  				nominal: 118,
  				mass: 117.90160657,
  				abundance: 0.2422
  			},
  			{
  				nominal: 119,
  				mass: 118.90331117,
  				abundance: 0.0859
  			},
  			{
  				nominal: 120,
  				mass: 119.90220163,
  				abundance: 0.3258
  			},
  			{
  				nominal: 121,
  				mass: 120.9042426
  			},
  			{
  				nominal: 122,
  				mass: 121.9034438,
  				abundance: 0.0463
  			},
  			{
  				nominal: 123,
  				mass: 122.9057252
  			},
  			{
  				nominal: 124,
  				mass: 123.9052766,
  				abundance: 0.0579
  			},
  			{
  				nominal: 125,
  				mass: 124.9077864
  			},
  			{
  				nominal: 126,
  				mass: 125.907659
  			},
  			{
  				nominal: 127,
  				mass: 126.91039
  			},
  			{
  				nominal: 128,
  				mass: 127.910507
  			},
  			{
  				nominal: 129,
  				mass: 128.913465
  			},
  			{
  				nominal: 130,
  				mass: 129.9139738
  			},
  			{
  				nominal: 131,
  				mass: 130.917045
  			},
  			{
  				nominal: 132,
  				mass: 131.9178267
  			},
  			{
  				nominal: 133,
  				mass: 132.9239134
  			},
  			{
  				nominal: 134,
  				mass: 133.9286821
  			},
  			{
  				nominal: 135,
  				mass: 134.9349086
  			},
  			{
  				nominal: 136,
  				mass: 135.93999
  			},
  			{
  				nominal: 137,
  				mass: 136.94655
  			},
  			{
  				nominal: 138,
  				mass: 137.95184
  			}
  		],
  		symbol: "Sn",
  		mass: 118.71011259301059,
  		name: "Tin",
  		monoisotopicMass: 119.90220163
  	},
  	{
  		number: 51,
  		isotopes: [
  			{
  				nominal: 103,
  				mass: 102.93969
  			},
  			{
  				nominal: 104,
  				mass: 103.93648
  			},
  			{
  				nominal: 105,
  				mass: 104.931276
  			},
  			{
  				nominal: 106,
  				mass: 105.928638
  			},
  			{
  				nominal: 107,
  				mass: 106.9241506
  			},
  			{
  				nominal: 108,
  				mass: 107.9222267
  			},
  			{
  				nominal: 109,
  				mass: 108.9181411
  			},
  			{
  				nominal: 110,
  				mass: 109.9168543
  			},
  			{
  				nominal: 111,
  				mass: 110.9132182
  			},
  			{
  				nominal: 112,
  				mass: 111.9124
  			},
  			{
  				nominal: 113,
  				mass: 112.909375
  			},
  			{
  				nominal: 114,
  				mass: 113.90929
  			},
  			{
  				nominal: 115,
  				mass: 114.906598
  			},
  			{
  				nominal: 116,
  				mass: 115.9067931
  			},
  			{
  				nominal: 117,
  				mass: 116.9048415
  			},
  			{
  				nominal: 118,
  				mass: 117.9055321
  			},
  			{
  				nominal: 119,
  				mass: 118.9039455
  			},
  			{
  				nominal: 120,
  				mass: 119.9050794
  			},
  			{
  				nominal: 121,
  				mass: 120.903812,
  				abundance: 0.5721
  			},
  			{
  				nominal: 122,
  				mass: 121.9051699
  			},
  			{
  				nominal: 123,
  				mass: 122.9042132,
  				abundance: 0.4279
  			},
  			{
  				nominal: 124,
  				mass: 123.905935
  			},
  			{
  				nominal: 125,
  				mass: 124.905253
  			},
  			{
  				nominal: 126,
  				mass: 125.907253
  			},
  			{
  				nominal: 127,
  				mass: 126.9069243
  			},
  			{
  				nominal: 128,
  				mass: 127.909146
  			},
  			{
  				nominal: 129,
  				mass: 128.909147
  			},
  			{
  				nominal: 130,
  				mass: 129.911662
  			},
  			{
  				nominal: 131,
  				mass: 130.9119888
  			},
  			{
  				nominal: 132,
  				mass: 131.9145077
  			},
  			{
  				nominal: 133,
  				mass: 132.9152732
  			},
  			{
  				nominal: 134,
  				mass: 133.9205357
  			},
  			{
  				nominal: 135,
  				mass: 134.9251851
  			},
  			{
  				nominal: 136,
  				mass: 135.9307459
  			},
  			{
  				nominal: 137,
  				mass: 136.93555
  			},
  			{
  				nominal: 138,
  				mass: 137.94145
  			},
  			{
  				nominal: 139,
  				mass: 138.94655
  			},
  			{
  				nominal: 140,
  				mass: 139.95283
  			}
  		],
  		symbol: "Sb",
  		mass: 121.75978367348,
  		name: "Antimony",
  		monoisotopicMass: 120.903812
  	},
  	{
  		number: 52,
  		isotopes: [
  			{
  				nominal: 105,
  				mass: 104.9433
  			},
  			{
  				nominal: 106,
  				mass: 105.9375
  			},
  			{
  				nominal: 107,
  				mass: 106.935012
  			},
  			{
  				nominal: 108,
  				mass: 107.9293805
  			},
  			{
  				nominal: 109,
  				mass: 108.9273045
  			},
  			{
  				nominal: 110,
  				mass: 109.9224581
  			},
  			{
  				nominal: 111,
  				mass: 110.9210006
  			},
  			{
  				nominal: 112,
  				mass: 111.9167279
  			},
  			{
  				nominal: 113,
  				mass: 112.915891
  			},
  			{
  				nominal: 114,
  				mass: 113.912089
  			},
  			{
  				nominal: 115,
  				mass: 114.911902
  			},
  			{
  				nominal: 116,
  				mass: 115.90846
  			},
  			{
  				nominal: 117,
  				mass: 116.908646
  			},
  			{
  				nominal: 118,
  				mass: 117.905854
  			},
  			{
  				nominal: 119,
  				mass: 118.9064071
  			},
  			{
  				nominal: 120,
  				mass: 119.9040593,
  				abundance: 0.0009
  			},
  			{
  				nominal: 121,
  				mass: 120.904944
  			},
  			{
  				nominal: 122,
  				mass: 121.9030435,
  				abundance: 0.0255
  			},
  			{
  				nominal: 123,
  				mass: 122.9042698,
  				abundance: 0.0089
  			},
  			{
  				nominal: 124,
  				mass: 123.9028171,
  				abundance: 0.0474
  			},
  			{
  				nominal: 125,
  				mass: 124.9044299,
  				abundance: 0.0707
  			},
  			{
  				nominal: 126,
  				mass: 125.9033109,
  				abundance: 0.1884
  			},
  			{
  				nominal: 127,
  				mass: 126.9052257
  			},
  			{
  				nominal: 128,
  				mass: 127.90446128,
  				abundance: 0.3174
  			},
  			{
  				nominal: 129,
  				mass: 128.90659646
  			},
  			{
  				nominal: 130,
  				mass: 129.906222748,
  				abundance: 0.3408
  			},
  			{
  				nominal: 131,
  				mass: 130.908522213
  			},
  			{
  				nominal: 132,
  				mass: 131.9085467
  			},
  			{
  				nominal: 133,
  				mass: 132.9109688
  			},
  			{
  				nominal: 134,
  				mass: 133.911394
  			},
  			{
  				nominal: 135,
  				mass: 134.9165557
  			},
  			{
  				nominal: 136,
  				mass: 135.9201006
  			},
  			{
  				nominal: 137,
  				mass: 136.9255989
  			},
  			{
  				nominal: 138,
  				mass: 137.9294722
  			},
  			{
  				nominal: 139,
  				mass: 138.9353672
  			},
  			{
  				nominal: 140,
  				mass: 139.939499
  			},
  			{
  				nominal: 141,
  				mass: 140.9458
  			},
  			{
  				nominal: 142,
  				mass: 141.95022
  			},
  			{
  				nominal: 143,
  				mass: 142.95676
  			}
  		],
  		symbol: "Te",
  		mass: 127.6031264846604,
  		name: "Tellurium",
  		monoisotopicMass: 129.906222748
  	},
  	{
  		number: 53,
  		isotopes: [
  			{
  				nominal: 107,
  				mass: 106.94678
  			},
  			{
  				nominal: 108,
  				mass: 107.94348
  			},
  			{
  				nominal: 109,
  				mass: 108.9380853
  			},
  			{
  				nominal: 110,
  				mass: 109.935089
  			},
  			{
  				nominal: 111,
  				mass: 110.9302692
  			},
  			{
  				nominal: 112,
  				mass: 111.928005
  			},
  			{
  				nominal: 113,
  				mass: 112.9236501
  			},
  			{
  				nominal: 114,
  				mass: 113.92185
  			},
  			{
  				nominal: 115,
  				mass: 114.918048
  			},
  			{
  				nominal: 116,
  				mass: 115.91681
  			},
  			{
  				nominal: 117,
  				mass: 116.913648
  			},
  			{
  				nominal: 118,
  				mass: 117.913074
  			},
  			{
  				nominal: 119,
  				mass: 118.910074
  			},
  			{
  				nominal: 120,
  				mass: 119.910087
  			},
  			{
  				nominal: 121,
  				mass: 120.9074051
  			},
  			{
  				nominal: 122,
  				mass: 121.9075888
  			},
  			{
  				nominal: 123,
  				mass: 122.9055885
  			},
  			{
  				nominal: 124,
  				mass: 123.906209
  			},
  			{
  				nominal: 125,
  				mass: 124.9046294
  			},
  			{
  				nominal: 126,
  				mass: 125.9056233
  			},
  			{
  				nominal: 127,
  				mass: 126.9044719,
  				abundance: 1
  			},
  			{
  				nominal: 128,
  				mass: 127.9058086
  			},
  			{
  				nominal: 129,
  				mass: 128.9049837
  			},
  			{
  				nominal: 130,
  				mass: 129.9066702
  			},
  			{
  				nominal: 131,
  				mass: 130.9061263
  			},
  			{
  				nominal: 132,
  				mass: 131.9079935
  			},
  			{
  				nominal: 133,
  				mass: 132.907797
  			},
  			{
  				nominal: 134,
  				mass: 133.9097588
  			},
  			{
  				nominal: 135,
  				mass: 134.9100488
  			},
  			{
  				nominal: 136,
  				mass: 135.914604
  			},
  			{
  				nominal: 137,
  				mass: 136.9180282
  			},
  			{
  				nominal: 138,
  				mass: 137.9227264
  			},
  			{
  				nominal: 139,
  				mass: 138.926506
  			},
  			{
  				nominal: 140,
  				mass: 139.93173
  			},
  			{
  				nominal: 141,
  				mass: 140.93569
  			},
  			{
  				nominal: 142,
  				mass: 141.9412
  			},
  			{
  				nominal: 143,
  				mass: 142.94565
  			},
  			{
  				nominal: 144,
  				mass: 143.95139
  			},
  			{
  				nominal: 145,
  				mass: 144.95605
  			}
  		],
  		symbol: "I",
  		mass: 126.9044719,
  		name: "Iodine",
  		monoisotopicMass: 126.9044719
  	},
  	{
  		number: 54,
  		isotopes: [
  			{
  				nominal: 109,
  				mass: 108.95043
  			},
  			{
  				nominal: 110,
  				mass: 109.94426
  			},
  			{
  				nominal: 111,
  				mass: 110.941607
  			},
  			{
  				nominal: 112,
  				mass: 111.935559
  			},
  			{
  				nominal: 113,
  				mass: 112.9332217
  			},
  			{
  				nominal: 114,
  				mass: 113.92798
  			},
  			{
  				nominal: 115,
  				mass: 114.926294
  			},
  			{
  				nominal: 116,
  				mass: 115.921581
  			},
  			{
  				nominal: 117,
  				mass: 116.920359
  			},
  			{
  				nominal: 118,
  				mass: 117.916179
  			},
  			{
  				nominal: 119,
  				mass: 118.915411
  			},
  			{
  				nominal: 120,
  				mass: 119.911784
  			},
  			{
  				nominal: 121,
  				mass: 120.911453
  			},
  			{
  				nominal: 122,
  				mass: 121.908368
  			},
  			{
  				nominal: 123,
  				mass: 122.908482
  			},
  			{
  				nominal: 124,
  				mass: 123.905892,
  				abundance: 0.000952
  			},
  			{
  				nominal: 125,
  				mass: 124.9063944
  			},
  			{
  				nominal: 126,
  				mass: 125.9042983,
  				abundance: 0.00089
  			},
  			{
  				nominal: 127,
  				mass: 126.9051829
  			},
  			{
  				nominal: 128,
  				mass: 127.903531,
  				abundance: 0.019102
  			},
  			{
  				nominal: 129,
  				mass: 128.9047808611,
  				abundance: 0.264006
  			},
  			{
  				nominal: 130,
  				mass: 129.903509349,
  				abundance: 0.04071
  			},
  			{
  				nominal: 131,
  				mass: 130.90508406,
  				abundance: 0.212324
  			},
  			{
  				nominal: 132,
  				mass: 131.9041550856,
  				abundance: 0.269086
  			},
  			{
  				nominal: 133,
  				mass: 132.9059108
  			},
  			{
  				nominal: 134,
  				mass: 133.90539466,
  				abundance: 0.104357
  			},
  			{
  				nominal: 135,
  				mass: 134.9072278
  			},
  			{
  				nominal: 136,
  				mass: 135.907214484,
  				abundance: 0.088573
  			},
  			{
  				nominal: 137,
  				mass: 136.91155778
  			},
  			{
  				nominal: 138,
  				mass: 137.9141463
  			},
  			{
  				nominal: 139,
  				mass: 138.9187922
  			},
  			{
  				nominal: 140,
  				mass: 139.9216458
  			},
  			{
  				nominal: 141,
  				mass: 140.9267872
  			},
  			{
  				nominal: 142,
  				mass: 141.9299731
  			},
  			{
  				nominal: 143,
  				mass: 142.9353696
  			},
  			{
  				nominal: 144,
  				mass: 143.9389451
  			},
  			{
  				nominal: 145,
  				mass: 144.94472
  			},
  			{
  				nominal: 146,
  				mass: 145.948518
  			},
  			{
  				nominal: 147,
  				mass: 146.95426
  			},
  			{
  				nominal: 148,
  				mass: 147.95813
  			}
  		],
  		symbol: "Xe",
  		mass: 131.29276144779053,
  		name: "Xenon",
  		monoisotopicMass: 131.9041550856
  	},
  	{
  		number: 55,
  		isotopes: [
  			{
  				nominal: 112,
  				mass: 111.950309
  			},
  			{
  				nominal: 113,
  				mass: 112.9444291
  			},
  			{
  				nominal: 114,
  				mass: 113.941296
  			},
  			{
  				nominal: 115,
  				mass: 114.93591
  			},
  			{
  				nominal: 116,
  				mass: 115.93337
  			},
  			{
  				nominal: 117,
  				mass: 116.928617
  			},
  			{
  				nominal: 118,
  				mass: 117.92656
  			},
  			{
  				nominal: 119,
  				mass: 118.922377
  			},
  			{
  				nominal: 120,
  				mass: 119.920677
  			},
  			{
  				nominal: 121,
  				mass: 120.917227
  			},
  			{
  				nominal: 122,
  				mass: 121.916108
  			},
  			{
  				nominal: 123,
  				mass: 122.912996
  			},
  			{
  				nominal: 124,
  				mass: 123.9122578
  			},
  			{
  				nominal: 125,
  				mass: 124.909728
  			},
  			{
  				nominal: 126,
  				mass: 125.909446
  			},
  			{
  				nominal: 127,
  				mass: 126.9074174
  			},
  			{
  				nominal: 128,
  				mass: 127.9077487
  			},
  			{
  				nominal: 129,
  				mass: 128.9060657
  			},
  			{
  				nominal: 130,
  				mass: 129.9067093
  			},
  			{
  				nominal: 131,
  				mass: 130.9054649
  			},
  			{
  				nominal: 132,
  				mass: 131.9064339
  			},
  			{
  				nominal: 133,
  				mass: 132.905451961,
  				abundance: 1
  			},
  			{
  				nominal: 134,
  				mass: 133.906718503
  			},
  			{
  				nominal: 135,
  				mass: 134.905977
  			},
  			{
  				nominal: 136,
  				mass: 135.9073114
  			},
  			{
  				nominal: 137,
  				mass: 136.90708923
  			},
  			{
  				nominal: 138,
  				mass: 137.9110171
  			},
  			{
  				nominal: 139,
  				mass: 138.9133638
  			},
  			{
  				nominal: 140,
  				mass: 139.9172831
  			},
  			{
  				nominal: 141,
  				mass: 140.9200455
  			},
  			{
  				nominal: 142,
  				mass: 141.924296
  			},
  			{
  				nominal: 143,
  				mass: 142.927349
  			},
  			{
  				nominal: 144,
  				mass: 143.932076
  			},
  			{
  				nominal: 145,
  				mass: 144.935527
  			},
  			{
  				nominal: 146,
  				mass: 145.940344
  			},
  			{
  				nominal: 147,
  				mass: 146.944156
  			},
  			{
  				nominal: 148,
  				mass: 147.94923
  			},
  			{
  				nominal: 149,
  				mass: 148.95302
  			},
  			{
  				nominal: 150,
  				mass: 149.95833
  			},
  			{
  				nominal: 151,
  				mass: 150.96258
  			}
  		],
  		symbol: "Cs",
  		mass: 132.905451961,
  		name: "Caesium",
  		monoisotopicMass: 132.905451961
  	},
  	{
  		number: 56,
  		isotopes: [
  			{
  				nominal: 114,
  				mass: 113.95066
  			},
  			{
  				nominal: 115,
  				mass: 114.94737
  			},
  			{
  				nominal: 116,
  				mass: 115.94128
  			},
  			{
  				nominal: 117,
  				mass: 116.93814
  			},
  			{
  				nominal: 118,
  				mass: 117.93306
  			},
  			{
  				nominal: 119,
  				mass: 118.93066
  			},
  			{
  				nominal: 120,
  				mass: 119.92605
  			},
  			{
  				nominal: 121,
  				mass: 120.92405
  			},
  			{
  				nominal: 122,
  				mass: 121.919904
  			},
  			{
  				nominal: 123,
  				mass: 122.918781
  			},
  			{
  				nominal: 124,
  				mass: 123.915094
  			},
  			{
  				nominal: 125,
  				mass: 124.914472
  			},
  			{
  				nominal: 126,
  				mass: 125.91125
  			},
  			{
  				nominal: 127,
  				mass: 126.911091
  			},
  			{
  				nominal: 128,
  				mass: 127.908342
  			},
  			{
  				nominal: 129,
  				mass: 128.908681
  			},
  			{
  				nominal: 130,
  				mass: 129.9063207,
  				abundance: 0.00106
  			},
  			{
  				nominal: 131,
  				mass: 130.906941
  			},
  			{
  				nominal: 132,
  				mass: 131.9050611,
  				abundance: 0.00101
  			},
  			{
  				nominal: 133,
  				mass: 132.9060074
  			},
  			{
  				nominal: 134,
  				mass: 133.90450818,
  				abundance: 0.02417
  			},
  			{
  				nominal: 135,
  				mass: 134.90568838,
  				abundance: 0.06592
  			},
  			{
  				nominal: 136,
  				mass: 135.90457573,
  				abundance: 0.07854
  			},
  			{
  				nominal: 137,
  				mass: 136.90582714,
  				abundance: 0.11232
  			},
  			{
  				nominal: 138,
  				mass: 137.905247,
  				abundance: 0.71698
  			},
  			{
  				nominal: 139,
  				mass: 138.9088411
  			},
  			{
  				nominal: 140,
  				mass: 139.9106057
  			},
  			{
  				nominal: 141,
  				mass: 140.9144033
  			},
  			{
  				nominal: 142,
  				mass: 141.9164324
  			},
  			{
  				nominal: 143,
  				mass: 142.9206253
  			},
  			{
  				nominal: 144,
  				mass: 143.9229549
  			},
  			{
  				nominal: 145,
  				mass: 144.9275184
  			},
  			{
  				nominal: 146,
  				mass: 145.930284
  			},
  			{
  				nominal: 147,
  				mass: 146.935304
  			},
  			{
  				nominal: 148,
  				mass: 147.938171
  			},
  			{
  				nominal: 149,
  				mass: 148.94308
  			},
  			{
  				nominal: 150,
  				mass: 149.94605
  			},
  			{
  				nominal: 151,
  				mass: 150.95127
  			},
  			{
  				nominal: 152,
  				mass: 151.95481
  			},
  			{
  				nominal: 153,
  				mass: 152.96036
  			}
  		],
  		symbol: "Ba",
  		mass: 137.3268916286322,
  		name: "Barium",
  		monoisotopicMass: 137.905247
  	},
  	{
  		number: 57,
  		isotopes: [
  			{
  				nominal: 116,
  				mass: 115.9563
  			},
  			{
  				nominal: 117,
  				mass: 116.94999
  			},
  			{
  				nominal: 118,
  				mass: 117.94673
  			},
  			{
  				nominal: 119,
  				mass: 118.94099
  			},
  			{
  				nominal: 120,
  				mass: 119.93807
  			},
  			{
  				nominal: 121,
  				mass: 120.93315
  			},
  			{
  				nominal: 122,
  				mass: 121.93071
  			},
  			{
  				nominal: 123,
  				mass: 122.9263
  			},
  			{
  				nominal: 124,
  				mass: 123.924574
  			},
  			{
  				nominal: 125,
  				mass: 124.920816
  			},
  			{
  				nominal: 126,
  				mass: 125.919513
  			},
  			{
  				nominal: 127,
  				mass: 126.916375
  			},
  			{
  				nominal: 128,
  				mass: 127.915592
  			},
  			{
  				nominal: 129,
  				mass: 128.912694
  			},
  			{
  				nominal: 130,
  				mass: 129.912369
  			},
  			{
  				nominal: 131,
  				mass: 130.91007
  			},
  			{
  				nominal: 132,
  				mass: 131.910119
  			},
  			{
  				nominal: 133,
  				mass: 132.908218
  			},
  			{
  				nominal: 134,
  				mass: 133.908514
  			},
  			{
  				nominal: 135,
  				mass: 134.906984
  			},
  			{
  				nominal: 136,
  				mass: 135.907635
  			},
  			{
  				nominal: 137,
  				mass: 136.9064504
  			},
  			{
  				nominal: 138,
  				mass: 137.9071149,
  				abundance: 0.0008881
  			},
  			{
  				nominal: 139,
  				mass: 138.9063563,
  				abundance: 0.9991119
  			},
  			{
  				nominal: 140,
  				mass: 139.9094806
  			},
  			{
  				nominal: 141,
  				mass: 140.910966
  			},
  			{
  				nominal: 142,
  				mass: 141.9140909
  			},
  			{
  				nominal: 143,
  				mass: 142.9160795
  			},
  			{
  				nominal: 144,
  				mass: 143.919646
  			},
  			{
  				nominal: 145,
  				mass: 144.921808
  			},
  			{
  				nominal: 146,
  				mass: 145.925875
  			},
  			{
  				nominal: 147,
  				mass: 146.928418
  			},
  			{
  				nominal: 148,
  				mass: 147.932679
  			},
  			{
  				nominal: 149,
  				mass: 148.93535
  			},
  			{
  				nominal: 150,
  				mass: 149.93947
  			},
  			{
  				nominal: 151,
  				mass: 150.94232
  			},
  			{
  				nominal: 152,
  				mass: 151.94682
  			},
  			{
  				nominal: 153,
  				mass: 152.95036
  			},
  			{
  				nominal: 154,
  				mass: 153.95517
  			},
  			{
  				nominal: 155,
  				mass: 154.95901
  			}
  		],
  		symbol: "La",
  		mass: 138.90546887371266,
  		name: "Lanthanum",
  		monoisotopicMass: 138.9063563
  	},
  	{
  		number: 58,
  		isotopes: [
  			{
  				nominal: 119,
  				mass: 118.95271
  			},
  			{
  				nominal: 120,
  				mass: 119.94654
  			},
  			{
  				nominal: 121,
  				mass: 120.94335
  			},
  			{
  				nominal: 122,
  				mass: 121.93787
  			},
  			{
  				nominal: 123,
  				mass: 122.93528
  			},
  			{
  				nominal: 124,
  				mass: 123.93031
  			},
  			{
  				nominal: 125,
  				mass: 124.92844
  			},
  			{
  				nominal: 126,
  				mass: 125.923971
  			},
  			{
  				nominal: 127,
  				mass: 126.922727
  			},
  			{
  				nominal: 128,
  				mass: 127.918911
  			},
  			{
  				nominal: 129,
  				mass: 128.918102
  			},
  			{
  				nominal: 130,
  				mass: 129.914736
  			},
  			{
  				nominal: 131,
  				mass: 130.914429
  			},
  			{
  				nominal: 132,
  				mass: 131.911464
  			},
  			{
  				nominal: 133,
  				mass: 132.91152
  			},
  			{
  				nominal: 134,
  				mass: 133.908928
  			},
  			{
  				nominal: 135,
  				mass: 134.909161
  			},
  			{
  				nominal: 136,
  				mass: 135.90712921,
  				abundance: 0.00185
  			},
  			{
  				nominal: 137,
  				mass: 136.90776236
  			},
  			{
  				nominal: 138,
  				mass: 137.905991,
  				abundance: 0.00251
  			},
  			{
  				nominal: 139,
  				mass: 138.9066551
  			},
  			{
  				nominal: 140,
  				mass: 139.9054431,
  				abundance: 0.8845
  			},
  			{
  				nominal: 141,
  				mass: 140.9082807
  			},
  			{
  				nominal: 142,
  				mass: 141.9092504,
  				abundance: 0.11114
  			},
  			{
  				nominal: 143,
  				mass: 142.9123921
  			},
  			{
  				nominal: 144,
  				mass: 143.9136529
  			},
  			{
  				nominal: 145,
  				mass: 144.917265
  			},
  			{
  				nominal: 146,
  				mass: 145.918802
  			},
  			{
  				nominal: 147,
  				mass: 146.9226899
  			},
  			{
  				nominal: 148,
  				mass: 147.924424
  			},
  			{
  				nominal: 149,
  				mass: 148.928427
  			},
  			{
  				nominal: 150,
  				mass: 149.930384
  			},
  			{
  				nominal: 151,
  				mass: 150.934272
  			},
  			{
  				nominal: 152,
  				mass: 151.9366
  			},
  			{
  				nominal: 153,
  				mass: 152.94093
  			},
  			{
  				nominal: 154,
  				mass: 153.9438
  			},
  			{
  				nominal: 155,
  				mass: 154.94855
  			},
  			{
  				nominal: 156,
  				mass: 155.95183
  			},
  			{
  				nominal: 157,
  				mass: 156.95705
  			}
  		],
  		symbol: "Ce",
  		mass: 140.1157307378545,
  		name: "Cerium",
  		monoisotopicMass: 139.9054431
  	},
  	{
  		number: 59,
  		isotopes: [
  			{
  				nominal: 121,
  				mass: 120.95532
  			},
  			{
  				nominal: 122,
  				mass: 121.95175
  			},
  			{
  				nominal: 123,
  				mass: 122.94596
  			},
  			{
  				nominal: 124,
  				mass: 123.94294
  			},
  			{
  				nominal: 125,
  				mass: 124.9377
  			},
  			{
  				nominal: 126,
  				mass: 125.93524
  			},
  			{
  				nominal: 127,
  				mass: 126.93071
  			},
  			{
  				nominal: 128,
  				mass: 127.928791
  			},
  			{
  				nominal: 129,
  				mass: 128.925095
  			},
  			{
  				nominal: 130,
  				mass: 129.92359
  			},
  			{
  				nominal: 131,
  				mass: 130.920235
  			},
  			{
  				nominal: 132,
  				mass: 131.919255
  			},
  			{
  				nominal: 133,
  				mass: 132.916331
  			},
  			{
  				nominal: 134,
  				mass: 133.915697
  			},
  			{
  				nominal: 135,
  				mass: 134.913112
  			},
  			{
  				nominal: 136,
  				mass: 135.912677
  			},
  			{
  				nominal: 137,
  				mass: 136.9106792
  			},
  			{
  				nominal: 138,
  				mass: 137.910754
  			},
  			{
  				nominal: 139,
  				mass: 138.9089408
  			},
  			{
  				nominal: 140,
  				mass: 139.9090803
  			},
  			{
  				nominal: 141,
  				mass: 140.9076576,
  				abundance: 1
  			},
  			{
  				nominal: 142,
  				mass: 141.9100496
  			},
  			{
  				nominal: 143,
  				mass: 142.9108228
  			},
  			{
  				nominal: 144,
  				mass: 143.9133109
  			},
  			{
  				nominal: 145,
  				mass: 144.9145182
  			},
  			{
  				nominal: 146,
  				mass: 145.91768
  			},
  			{
  				nominal: 147,
  				mass: 146.919008
  			},
  			{
  				nominal: 148,
  				mass: 147.92213
  			},
  			{
  				nominal: 149,
  				mass: 148.923736
  			},
  			{
  				nominal: 150,
  				mass: 149.9266765
  			},
  			{
  				nominal: 151,
  				mass: 150.928309
  			},
  			{
  				nominal: 152,
  				mass: 151.931553
  			},
  			{
  				nominal: 153,
  				mass: 152.933904
  			},
  			{
  				nominal: 154,
  				mass: 153.93753
  			},
  			{
  				nominal: 155,
  				mass: 154.940509
  			},
  			{
  				nominal: 156,
  				mass: 155.94464
  			},
  			{
  				nominal: 157,
  				mass: 156.94789
  			},
  			{
  				nominal: 158,
  				mass: 157.95241
  			},
  			{
  				nominal: 159,
  				mass: 158.95589
  			}
  		],
  		symbol: "Pr",
  		mass: 140.9076576,
  		name: "Praseodymium",
  		monoisotopicMass: 140.9076576
  	},
  	{
  		number: 60,
  		isotopes: [
  			{
  				nominal: 124,
  				mass: 123.9522
  			},
  			{
  				nominal: 125,
  				mass: 124.9489
  			},
  			{
  				nominal: 126,
  				mass: 125.94311
  			},
  			{
  				nominal: 127,
  				mass: 126.94038
  			},
  			{
  				nominal: 128,
  				mass: 127.93525
  			},
  			{
  				nominal: 129,
  				mass: 128.9331
  			},
  			{
  				nominal: 130,
  				mass: 129.928506
  			},
  			{
  				nominal: 131,
  				mass: 130.927248
  			},
  			{
  				nominal: 132,
  				mass: 131.923321
  			},
  			{
  				nominal: 133,
  				mass: 132.922348
  			},
  			{
  				nominal: 134,
  				mass: 133.91879
  			},
  			{
  				nominal: 135,
  				mass: 134.918181
  			},
  			{
  				nominal: 136,
  				mass: 135.914976
  			},
  			{
  				nominal: 137,
  				mass: 136.914562
  			},
  			{
  				nominal: 138,
  				mass: 137.91195
  			},
  			{
  				nominal: 139,
  				mass: 138.911954
  			},
  			{
  				nominal: 140,
  				mass: 139.90955
  			},
  			{
  				nominal: 141,
  				mass: 140.9096147
  			},
  			{
  				nominal: 142,
  				mass: 141.907729,
  				abundance: 0.27152
  			},
  			{
  				nominal: 143,
  				mass: 142.90982,
  				abundance: 0.12174
  			},
  			{
  				nominal: 144,
  				mass: 143.910093,
  				abundance: 0.23798
  			},
  			{
  				nominal: 145,
  				mass: 144.9125793,
  				abundance: 0.08293
  			},
  			{
  				nominal: 146,
  				mass: 145.9131226,
  				abundance: 0.17189
  			},
  			{
  				nominal: 147,
  				mass: 146.9161061
  			},
  			{
  				nominal: 148,
  				mass: 147.9168993,
  				abundance: 0.05756
  			},
  			{
  				nominal: 149,
  				mass: 148.9201548
  			},
  			{
  				nominal: 150,
  				mass: 149.9209022,
  				abundance: 0.05638
  			},
  			{
  				nominal: 151,
  				mass: 150.9238403
  			},
  			{
  				nominal: 152,
  				mass: 151.924692
  			},
  			{
  				nominal: 153,
  				mass: 152.927718
  			},
  			{
  				nominal: 154,
  				mass: 153.92948
  			},
  			{
  				nominal: 155,
  				mass: 154.9331357
  			},
  			{
  				nominal: 156,
  				mass: 155.93508
  			},
  			{
  				nominal: 157,
  				mass: 156.939386
  			},
  			{
  				nominal: 158,
  				mass: 157.94197
  			},
  			{
  				nominal: 159,
  				mass: 158.94653
  			},
  			{
  				nominal: 160,
  				mass: 159.9494
  			},
  			{
  				nominal: 161,
  				mass: 160.95428
  			}
  		],
  		symbol: "Nd",
  		mass: 144.241596031827,
  		name: "Neodymium",
  		monoisotopicMass: 141.907729
  	},
  	{
  		number: 61,
  		isotopes: [
  			{
  				nominal: 126,
  				mass: 125.95792
  			},
  			{
  				nominal: 127,
  				mass: 126.95192
  			},
  			{
  				nominal: 128,
  				mass: 127.9487
  			},
  			{
  				nominal: 129,
  				mass: 128.94323
  			},
  			{
  				nominal: 130,
  				mass: 129.94053
  			},
  			{
  				nominal: 131,
  				mass: 130.93567
  			},
  			{
  				nominal: 132,
  				mass: 131.93384
  			},
  			{
  				nominal: 133,
  				mass: 132.929782
  			},
  			{
  				nominal: 134,
  				mass: 133.928353
  			},
  			{
  				nominal: 135,
  				mass: 134.924823
  			},
  			{
  				nominal: 136,
  				mass: 135.923585
  			},
  			{
  				nominal: 137,
  				mass: 136.92048
  			},
  			{
  				nominal: 138,
  				mass: 137.919548
  			},
  			{
  				nominal: 139,
  				mass: 138.9168
  			},
  			{
  				nominal: 140,
  				mass: 139.91604
  			},
  			{
  				nominal: 141,
  				mass: 140.913555
  			},
  			{
  				nominal: 142,
  				mass: 141.91289
  			},
  			{
  				nominal: 143,
  				mass: 142.9109383
  			},
  			{
  				nominal: 144,
  				mass: 143.9125964
  			},
  			{
  				nominal: 145,
  				mass: 144.9127559
  			},
  			{
  				nominal: 146,
  				mass: 145.9147024
  			},
  			{
  				nominal: 147,
  				mass: 146.915145
  			},
  			{
  				nominal: 148,
  				mass: 147.9174819
  			},
  			{
  				nominal: 149,
  				mass: 148.9183423
  			},
  			{
  				nominal: 150,
  				mass: 149.920991
  			},
  			{
  				nominal: 151,
  				mass: 150.9212175
  			},
  			{
  				nominal: 152,
  				mass: 151.923506
  			},
  			{
  				nominal: 153,
  				mass: 152.9241567
  			},
  			{
  				nominal: 154,
  				mass: 153.926472
  			},
  			{
  				nominal: 155,
  				mass: 154.928137
  			},
  			{
  				nominal: 156,
  				mass: 155.9311175
  			},
  			{
  				nominal: 157,
  				mass: 156.9331214
  			},
  			{
  				nominal: 158,
  				mass: 157.936565
  			},
  			{
  				nominal: 159,
  				mass: 158.939287
  			},
  			{
  				nominal: 160,
  				mass: 159.9431
  			},
  			{
  				nominal: 161,
  				mass: 160.94607
  			},
  			{
  				nominal: 162,
  				mass: 161.95022
  			},
  			{
  				nominal: 163,
  				mass: 162.95357
  			}
  		],
  		symbol: "Pm",
  		mass: null,
  		name: "Promethium"
  	},
  	{
  		number: 62,
  		isotopes: [
  			{
  				nominal: 128,
  				mass: 127.95842
  			},
  			{
  				nominal: 129,
  				mass: 128.95476
  			},
  			{
  				nominal: 130,
  				mass: 129.949
  			},
  			{
  				nominal: 131,
  				mass: 130.94618
  			},
  			{
  				nominal: 132,
  				mass: 131.94087
  			},
  			{
  				nominal: 133,
  				mass: 132.93856
  			},
  			{
  				nominal: 134,
  				mass: 133.93411
  			},
  			{
  				nominal: 135,
  				mass: 134.93252
  			},
  			{
  				nominal: 136,
  				mass: 135.928276
  			},
  			{
  				nominal: 137,
  				mass: 136.926971
  			},
  			{
  				nominal: 138,
  				mass: 137.923244
  			},
  			{
  				nominal: 139,
  				mass: 138.922297
  			},
  			{
  				nominal: 140,
  				mass: 139.918995
  			},
  			{
  				nominal: 141,
  				mass: 140.9184816
  			},
  			{
  				nominal: 142,
  				mass: 141.9152044
  			},
  			{
  				nominal: 143,
  				mass: 142.9146353
  			},
  			{
  				nominal: 144,
  				mass: 143.9120065,
  				abundance: 0.0307
  			},
  			{
  				nominal: 145,
  				mass: 144.9134173
  			},
  			{
  				nominal: 146,
  				mass: 145.913047
  			},
  			{
  				nominal: 147,
  				mass: 146.9149044,
  				abundance: 0.1499
  			},
  			{
  				nominal: 148,
  				mass: 147.9148292,
  				abundance: 0.1124
  			},
  			{
  				nominal: 149,
  				mass: 148.9171921,
  				abundance: 0.1382
  			},
  			{
  				nominal: 150,
  				mass: 149.9172829,
  				abundance: 0.0738
  			},
  			{
  				nominal: 151,
  				mass: 150.9199398
  			},
  			{
  				nominal: 152,
  				mass: 151.9197397,
  				abundance: 0.2675
  			},
  			{
  				nominal: 153,
  				mass: 152.9221047
  			},
  			{
  				nominal: 154,
  				mass: 153.9222169,
  				abundance: 0.2275
  			},
  			{
  				nominal: 155,
  				mass: 154.9246477
  			},
  			{
  				nominal: 156,
  				mass: 155.925536
  			},
  			{
  				nominal: 157,
  				mass: 156.9284187
  			},
  			{
  				nominal: 158,
  				mass: 157.929951
  			},
  			{
  				nominal: 159,
  				mass: 158.9332172
  			},
  			{
  				nominal: 160,
  				mass: 159.9353353
  			},
  			{
  				nominal: 161,
  				mass: 160.9391602
  			},
  			{
  				nominal: 162,
  				mass: 161.94146
  			},
  			{
  				nominal: 163,
  				mass: 162.94555
  			},
  			{
  				nominal: 164,
  				mass: 163.94836
  			},
  			{
  				nominal: 165,
  				mass: 164.95297
  			}
  		],
  		symbol: "Sm",
  		mass: 150.36635571193,
  		name: "Samarium",
  		monoisotopicMass: 151.9197397
  	},
  	{
  		number: 63,
  		isotopes: [
  			{
  				nominal: 130,
  				mass: 129.96369
  			},
  			{
  				nominal: 131,
  				mass: 130.95784
  			},
  			{
  				nominal: 132,
  				mass: 131.95467
  			},
  			{
  				nominal: 133,
  				mass: 132.94929
  			},
  			{
  				nominal: 134,
  				mass: 133.9464
  			},
  			{
  				nominal: 135,
  				mass: 134.94187
  			},
  			{
  				nominal: 136,
  				mass: 135.93962
  			},
  			{
  				nominal: 137,
  				mass: 136.93546
  			},
  			{
  				nominal: 138,
  				mass: 137.933709
  			},
  			{
  				nominal: 139,
  				mass: 138.929792
  			},
  			{
  				nominal: 140,
  				mass: 139.928088
  			},
  			{
  				nominal: 141,
  				mass: 140.924932
  			},
  			{
  				nominal: 142,
  				mass: 141.923442
  			},
  			{
  				nominal: 143,
  				mass: 142.920299
  			},
  			{
  				nominal: 144,
  				mass: 143.91882
  			},
  			{
  				nominal: 145,
  				mass: 144.9162726
  			},
  			{
  				nominal: 146,
  				mass: 145.917211
  			},
  			{
  				nominal: 147,
  				mass: 146.9167527
  			},
  			{
  				nominal: 148,
  				mass: 147.918089
  			},
  			{
  				nominal: 149,
  				mass: 148.9179378
  			},
  			{
  				nominal: 150,
  				mass: 149.9197077
  			},
  			{
  				nominal: 151,
  				mass: 150.9198578,
  				abundance: 0.4781
  			},
  			{
  				nominal: 152,
  				mass: 151.9217522
  			},
  			{
  				nominal: 153,
  				mass: 152.921238,
  				abundance: 0.5219
  			},
  			{
  				nominal: 154,
  				mass: 153.922987
  			},
  			{
  				nominal: 155,
  				mass: 154.9229011
  			},
  			{
  				nominal: 156,
  				mass: 155.9247605
  			},
  			{
  				nominal: 157,
  				mass: 156.9254334
  			},
  			{
  				nominal: 158,
  				mass: 157.927799
  			},
  			{
  				nominal: 159,
  				mass: 158.9291001
  			},
  			{
  				nominal: 160,
  				mass: 159.931851
  			},
  			{
  				nominal: 161,
  				mass: 160.933664
  			},
  			{
  				nominal: 162,
  				mass: 161.936989
  			},
  			{
  				nominal: 163,
  				mass: 162.939196
  			},
  			{
  				nominal: 164,
  				mass: 163.94274
  			},
  			{
  				nominal: 165,
  				mass: 164.94559
  			},
  			{
  				nominal: 166,
  				mass: 165.94962
  			},
  			{
  				nominal: 167,
  				mass: 166.95289
  			}
  		],
  		symbol: "Eu",
  		mass: 151.96437812637998,
  		name: "Europium",
  		monoisotopicMass: 152.921238
  	},
  	{
  		number: 64,
  		isotopes: [
  			{
  				nominal: 133,
  				mass: 132.96133
  			},
  			{
  				nominal: 134,
  				mass: 133.95566
  			},
  			{
  				nominal: 135,
  				mass: 134.95245
  			},
  			{
  				nominal: 136,
  				mass: 135.9473
  			},
  			{
  				nominal: 137,
  				mass: 136.94502
  			},
  			{
  				nominal: 138,
  				mass: 137.94025
  			},
  			{
  				nominal: 139,
  				mass: 138.93813
  			},
  			{
  				nominal: 140,
  				mass: 139.933674
  			},
  			{
  				nominal: 141,
  				mass: 140.932126
  			},
  			{
  				nominal: 142,
  				mass: 141.928116
  			},
  			{
  				nominal: 143,
  				mass: 142.92675
  			},
  			{
  				nominal: 144,
  				mass: 143.922963
  			},
  			{
  				nominal: 145,
  				mass: 144.921713
  			},
  			{
  				nominal: 146,
  				mass: 145.9183188
  			},
  			{
  				nominal: 147,
  				mass: 146.9191014
  			},
  			{
  				nominal: 148,
  				mass: 147.9181215
  			},
  			{
  				nominal: 149,
  				mass: 148.9193481
  			},
  			{
  				nominal: 150,
  				mass: 149.9186644
  			},
  			{
  				nominal: 151,
  				mass: 150.920356
  			},
  			{
  				nominal: 152,
  				mass: 151.9197995,
  				abundance: 0.002
  			},
  			{
  				nominal: 153,
  				mass: 152.921758
  			},
  			{
  				nominal: 154,
  				mass: 153.9208741,
  				abundance: 0.0218
  			},
  			{
  				nominal: 155,
  				mass: 154.9226305,
  				abundance: 0.148
  			},
  			{
  				nominal: 156,
  				mass: 155.9221312,
  				abundance: 0.2047
  			},
  			{
  				nominal: 157,
  				mass: 156.9239686,
  				abundance: 0.1565
  			},
  			{
  				nominal: 158,
  				mass: 157.9241123,
  				abundance: 0.2484
  			},
  			{
  				nominal: 159,
  				mass: 158.926397
  			},
  			{
  				nominal: 160,
  				mass: 159.9270624,
  				abundance: 0.2186
  			},
  			{
  				nominal: 161,
  				mass: 160.9296775
  			},
  			{
  				nominal: 162,
  				mass: 161.930993
  			},
  			{
  				nominal: 163,
  				mass: 162.9341769
  			},
  			{
  				nominal: 164,
  				mass: 163.93583
  			},
  			{
  				nominal: 165,
  				mass: 164.93936
  			},
  			{
  				nominal: 166,
  				mass: 165.94146
  			},
  			{
  				nominal: 167,
  				mass: 166.94545
  			},
  			{
  				nominal: 168,
  				mass: 167.94808
  			},
  			{
  				nominal: 169,
  				mass: 168.9526
  			}
  		],
  		symbol: "Gd",
  		mass: 157.25213064687998,
  		name: "Gadolinium",
  		monoisotopicMass: 157.9241123
  	},
  	{
  		number: 65,
  		isotopes: [
  			{
  				nominal: 135,
  				mass: 134.96476
  			},
  			{
  				nominal: 136,
  				mass: 135.96129
  			},
  			{
  				nominal: 137,
  				mass: 136.95602
  			},
  			{
  				nominal: 138,
  				mass: 137.95312
  			},
  			{
  				nominal: 139,
  				mass: 138.94833
  			},
  			{
  				nominal: 140,
  				mass: 139.94581
  			},
  			{
  				nominal: 141,
  				mass: 140.94145
  			},
  			{
  				nominal: 142,
  				mass: 141.93928
  			},
  			{
  				nominal: 143,
  				mass: 142.935137
  			},
  			{
  				nominal: 144,
  				mass: 143.933045
  			},
  			{
  				nominal: 145,
  				mass: 144.92882
  			},
  			{
  				nominal: 146,
  				mass: 145.927253
  			},
  			{
  				nominal: 147,
  				mass: 146.9240548
  			},
  			{
  				nominal: 148,
  				mass: 147.924282
  			},
  			{
  				nominal: 149,
  				mass: 148.9232535
  			},
  			{
  				nominal: 150,
  				mass: 149.9236649
  			},
  			{
  				nominal: 151,
  				mass: 150.9231096
  			},
  			{
  				nominal: 152,
  				mass: 151.924083
  			},
  			{
  				nominal: 153,
  				mass: 152.9234424
  			},
  			{
  				nominal: 154,
  				mass: 153.924685
  			},
  			{
  				nominal: 155,
  				mass: 154.923511
  			},
  			{
  				nominal: 156,
  				mass: 155.9247552
  			},
  			{
  				nominal: 157,
  				mass: 156.924033
  			},
  			{
  				nominal: 158,
  				mass: 157.9254209
  			},
  			{
  				nominal: 159,
  				mass: 158.9253547,
  				abundance: 1
  			},
  			{
  				nominal: 160,
  				mass: 159.9271756
  			},
  			{
  				nominal: 161,
  				mass: 160.9275778
  			},
  			{
  				nominal: 162,
  				mass: 161.929495
  			},
  			{
  				nominal: 163,
  				mass: 162.9306547
  			},
  			{
  				nominal: 164,
  				mass: 163.93336
  			},
  			{
  				nominal: 165,
  				mass: 164.93498
  			},
  			{
  				nominal: 166,
  				mass: 165.93786
  			},
  			{
  				nominal: 167,
  				mass: 166.93996
  			},
  			{
  				nominal: 168,
  				mass: 167.9434
  			},
  			{
  				nominal: 169,
  				mass: 168.94597
  			},
  			{
  				nominal: 170,
  				mass: 169.94984
  			},
  			{
  				nominal: 171,
  				mass: 170.95273
  			}
  		],
  		symbol: "Tb",
  		mass: 158.9253547,
  		name: "Terbium",
  		monoisotopicMass: 158.9253547
  	},
  	{
  		number: 66,
  		isotopes: [
  			{
  				nominal: 138,
  				mass: 137.9625
  			},
  			{
  				nominal: 139,
  				mass: 138.95959
  			},
  			{
  				nominal: 140,
  				mass: 139.95402
  			},
  			{
  				nominal: 141,
  				mass: 140.95128
  			},
  			{
  				nominal: 142,
  				mass: 141.94619
  			},
  			{
  				nominal: 143,
  				mass: 142.943994
  			},
  			{
  				nominal: 144,
  				mass: 143.9392695
  			},
  			{
  				nominal: 145,
  				mass: 144.937474
  			},
  			{
  				nominal: 146,
  				mass: 145.9328445
  			},
  			{
  				nominal: 147,
  				mass: 146.9310827
  			},
  			{
  				nominal: 148,
  				mass: 147.927157
  			},
  			{
  				nominal: 149,
  				mass: 148.927322
  			},
  			{
  				nominal: 150,
  				mass: 149.9255933
  			},
  			{
  				nominal: 151,
  				mass: 150.9261916
  			},
  			{
  				nominal: 152,
  				mass: 151.9247253
  			},
  			{
  				nominal: 153,
  				mass: 152.9257724
  			},
  			{
  				nominal: 154,
  				mass: 153.9244293
  			},
  			{
  				nominal: 155,
  				mass: 154.925759
  			},
  			{
  				nominal: 156,
  				mass: 155.9242847,
  				abundance: 0.00056
  			},
  			{
  				nominal: 157,
  				mass: 156.9254707
  			},
  			{
  				nominal: 158,
  				mass: 157.9244159,
  				abundance: 0.00095
  			},
  			{
  				nominal: 159,
  				mass: 158.925747
  			},
  			{
  				nominal: 160,
  				mass: 159.9252046,
  				abundance: 0.02329
  			},
  			{
  				nominal: 161,
  				mass: 160.9269405,
  				abundance: 0.18889
  			},
  			{
  				nominal: 162,
  				mass: 161.9268056,
  				abundance: 0.25475
  			},
  			{
  				nominal: 163,
  				mass: 162.9287383,
  				abundance: 0.24896
  			},
  			{
  				nominal: 164,
  				mass: 163.9291819,
  				abundance: 0.2826
  			},
  			{
  				nominal: 165,
  				mass: 164.9317105
  			},
  			{
  				nominal: 166,
  				mass: 165.9328139
  			},
  			{
  				nominal: 167,
  				mass: 166.935661
  			},
  			{
  				nominal: 168,
  				mass: 167.93713
  			},
  			{
  				nominal: 169,
  				mass: 168.94031
  			},
  			{
  				nominal: 170,
  				mass: 169.94239
  			},
  			{
  				nominal: 171,
  				mass: 170.94612
  			},
  			{
  				nominal: 172,
  				mass: 171.94846
  			},
  			{
  				nominal: 173,
  				mass: 172.95283
  			}
  		],
  		symbol: "Dy",
  		mass: 162.499472819424,
  		name: "Dysprosium",
  		monoisotopicMass: 163.9291819
  	},
  	{
  		number: 67,
  		isotopes: [
  			{
  				nominal: 140,
  				mass: 139.96859
  			},
  			{
  				nominal: 141,
  				mass: 140.96311
  			},
  			{
  				nominal: 142,
  				mass: 141.96001
  			},
  			{
  				nominal: 143,
  				mass: 142.95486
  			},
  			{
  				nominal: 144,
  				mass: 143.9521097
  			},
  			{
  				nominal: 145,
  				mass: 144.9472674
  			},
  			{
  				nominal: 146,
  				mass: 145.9449935
  			},
  			{
  				nominal: 147,
  				mass: 146.9401423
  			},
  			{
  				nominal: 148,
  				mass: 147.937744
  			},
  			{
  				nominal: 149,
  				mass: 148.933803
  			},
  			{
  				nominal: 150,
  				mass: 149.933498
  			},
  			{
  				nominal: 151,
  				mass: 150.9316983
  			},
  			{
  				nominal: 152,
  				mass: 151.931724
  			},
  			{
  				nominal: 153,
  				mass: 152.9302064
  			},
  			{
  				nominal: 154,
  				mass: 153.9306068
  			},
  			{
  				nominal: 155,
  				mass: 154.929104
  			},
  			{
  				nominal: 156,
  				mass: 155.929706
  			},
  			{
  				nominal: 157,
  				mass: 156.928254
  			},
  			{
  				nominal: 158,
  				mass: 157.928946
  			},
  			{
  				nominal: 159,
  				mass: 158.9277197
  			},
  			{
  				nominal: 160,
  				mass: 159.928737
  			},
  			{
  				nominal: 161,
  				mass: 160.9278615
  			},
  			{
  				nominal: 162,
  				mass: 161.9291023
  			},
  			{
  				nominal: 163,
  				mass: 162.928741
  			},
  			{
  				nominal: 164,
  				mass: 163.9302403
  			},
  			{
  				nominal: 165,
  				mass: 164.9303288,
  				abundance: 1
  			},
  			{
  				nominal: 166,
  				mass: 165.9322909
  			},
  			{
  				nominal: 167,
  				mass: 166.9331385
  			},
  			{
  				nominal: 168,
  				mass: 167.935522
  			},
  			{
  				nominal: 169,
  				mass: 168.936878
  			},
  			{
  				nominal: 170,
  				mass: 169.939625
  			},
  			{
  				nominal: 171,
  				mass: 170.94147
  			},
  			{
  				nominal: 172,
  				mass: 171.94473
  			},
  			{
  				nominal: 173,
  				mass: 172.94702
  			},
  			{
  				nominal: 174,
  				mass: 173.95095
  			},
  			{
  				nominal: 175,
  				mass: 174.95362
  			}
  		],
  		symbol: "Ho",
  		mass: 164.9303288,
  		name: "Holmium",
  		monoisotopicMass: 164.9303288
  	},
  	{
  		number: 68,
  		isotopes: [
  			{
  				nominal: 142,
  				mass: 141.9701
  			},
  			{
  				nominal: 143,
  				mass: 142.96662
  			},
  			{
  				nominal: 144,
  				mass: 143.9607
  			},
  			{
  				nominal: 145,
  				mass: 144.95805
  			},
  			{
  				nominal: 146,
  				mass: 145.9524184
  			},
  			{
  				nominal: 147,
  				mass: 146.949964
  			},
  			{
  				nominal: 148,
  				mass: 147.944735
  			},
  			{
  				nominal: 149,
  				mass: 148.942306
  			},
  			{
  				nominal: 150,
  				mass: 149.937916
  			},
  			{
  				nominal: 151,
  				mass: 150.937449
  			},
  			{
  				nominal: 152,
  				mass: 151.935057
  			},
  			{
  				nominal: 153,
  				mass: 152.93508
  			},
  			{
  				nominal: 154,
  				mass: 153.9327908
  			},
  			{
  				nominal: 155,
  				mass: 154.9332159
  			},
  			{
  				nominal: 156,
  				mass: 155.931067
  			},
  			{
  				nominal: 157,
  				mass: 156.931949
  			},
  			{
  				nominal: 158,
  				mass: 157.929893
  			},
  			{
  				nominal: 159,
  				mass: 158.9306918
  			},
  			{
  				nominal: 160,
  				mass: 159.929077
  			},
  			{
  				nominal: 161,
  				mass: 160.9300046
  			},
  			{
  				nominal: 162,
  				mass: 161.9287884,
  				abundance: 0.00139
  			},
  			{
  				nominal: 163,
  				mass: 162.9300408
  			},
  			{
  				nominal: 164,
  				mass: 163.9292088,
  				abundance: 0.01601
  			},
  			{
  				nominal: 165,
  				mass: 164.9307345
  			},
  			{
  				nominal: 166,
  				mass: 165.9302995,
  				abundance: 0.33503
  			},
  			{
  				nominal: 167,
  				mass: 166.9320546,
  				abundance: 0.22869
  			},
  			{
  				nominal: 168,
  				mass: 167.9323767,
  				abundance: 0.26978
  			},
  			{
  				nominal: 169,
  				mass: 168.9345968
  			},
  			{
  				nominal: 170,
  				mass: 169.9354702,
  				abundance: 0.1491
  			},
  			{
  				nominal: 171,
  				mass: 170.9380357
  			},
  			{
  				nominal: 172,
  				mass: 171.9393619
  			},
  			{
  				nominal: 173,
  				mass: 172.9424
  			},
  			{
  				nominal: 174,
  				mass: 173.94423
  			},
  			{
  				nominal: 175,
  				mass: 174.94777
  			},
  			{
  				nominal: 176,
  				mass: 175.94994
  			},
  			{
  				nominal: 177,
  				mass: 176.95399
  			}
  		],
  		symbol: "Er",
  		mass: 167.259082649669,
  		name: "Erbium",
  		monoisotopicMass: 165.9302995
  	},
  	{
  		number: 69,
  		isotopes: [
  			{
  				nominal: 144,
  				mass: 143.97628
  			},
  			{
  				nominal: 145,
  				mass: 144.97039
  			},
  			{
  				nominal: 146,
  				mass: 145.96684
  			},
  			{
  				nominal: 147,
  				mass: 146.9613799
  			},
  			{
  				nominal: 148,
  				mass: 147.958384
  			},
  			{
  				nominal: 149,
  				mass: 148.95289
  			},
  			{
  				nominal: 150,
  				mass: 149.95009
  			},
  			{
  				nominal: 151,
  				mass: 150.945488
  			},
  			{
  				nominal: 152,
  				mass: 151.944422
  			},
  			{
  				nominal: 153,
  				mass: 152.94204
  			},
  			{
  				nominal: 154,
  				mass: 153.94157
  			},
  			{
  				nominal: 155,
  				mass: 154.93921
  			},
  			{
  				nominal: 156,
  				mass: 155.938992
  			},
  			{
  				nominal: 157,
  				mass: 156.936944
  			},
  			{
  				nominal: 158,
  				mass: 157.93698
  			},
  			{
  				nominal: 159,
  				mass: 158.934975
  			},
  			{
  				nominal: 160,
  				mass: 159.935263
  			},
  			{
  				nominal: 161,
  				mass: 160.933549
  			},
  			{
  				nominal: 162,
  				mass: 161.934002
  			},
  			{
  				nominal: 163,
  				mass: 162.9326592
  			},
  			{
  				nominal: 164,
  				mass: 163.933544
  			},
  			{
  				nominal: 165,
  				mass: 164.9324431
  			},
  			{
  				nominal: 166,
  				mass: 165.933561
  			},
  			{
  				nominal: 167,
  				mass: 166.9328562
  			},
  			{
  				nominal: 168,
  				mass: 167.9341774
  			},
  			{
  				nominal: 169,
  				mass: 168.9342179,
  				abundance: 1
  			},
  			{
  				nominal: 170,
  				mass: 169.935806
  			},
  			{
  				nominal: 171,
  				mass: 170.9364339
  			},
  			{
  				nominal: 172,
  				mass: 171.9384055
  			},
  			{
  				nominal: 173,
  				mass: 172.9396084
  			},
  			{
  				nominal: 174,
  				mass: 173.942173
  			},
  			{
  				nominal: 175,
  				mass: 174.943841
  			},
  			{
  				nominal: 176,
  				mass: 175.947
  			},
  			{
  				nominal: 177,
  				mass: 176.94904
  			},
  			{
  				nominal: 178,
  				mass: 177.95264
  			},
  			{
  				nominal: 179,
  				mass: 178.95534
  			}
  		],
  		symbol: "Tm",
  		mass: 168.9342179,
  		name: "Thulium",
  		monoisotopicMass: 168.9342179
  	},
  	{
  		number: 70,
  		isotopes: [
  			{
  				nominal: 148,
  				mass: 147.96758
  			},
  			{
  				nominal: 149,
  				mass: 148.96436
  			},
  			{
  				nominal: 150,
  				mass: 149.95852
  			},
  			{
  				nominal: 151,
  				mass: 150.9554
  			},
  			{
  				nominal: 152,
  				mass: 151.95027
  			},
  			{
  				nominal: 153,
  				mass: 152.94932
  			},
  			{
  				nominal: 154,
  				mass: 153.946396
  			},
  			{
  				nominal: 155,
  				mass: 154.945783
  			},
  			{
  				nominal: 156,
  				mass: 155.942825
  			},
  			{
  				nominal: 157,
  				mass: 156.942645
  			},
  			{
  				nominal: 158,
  				mass: 157.9398705
  			},
  			{
  				nominal: 159,
  				mass: 158.940055
  			},
  			{
  				nominal: 160,
  				mass: 159.937557
  			},
  			{
  				nominal: 161,
  				mass: 160.937907
  			},
  			{
  				nominal: 162,
  				mass: 161.935774
  			},
  			{
  				nominal: 163,
  				mass: 162.93634
  			},
  			{
  				nominal: 164,
  				mass: 163.934495
  			},
  			{
  				nominal: 165,
  				mass: 164.93527
  			},
  			{
  				nominal: 166,
  				mass: 165.9338747
  			},
  			{
  				nominal: 167,
  				mass: 166.934953
  			},
  			{
  				nominal: 168,
  				mass: 167.9338896,
  				abundance: 0.00123
  			},
  			{
  				nominal: 169,
  				mass: 168.9351825
  			},
  			{
  				nominal: 170,
  				mass: 169.9347664,
  				abundance: 0.02982
  			},
  			{
  				nominal: 171,
  				mass: 170.9363302,
  				abundance: 0.1409
  			},
  			{
  				nominal: 172,
  				mass: 171.9363859,
  				abundance: 0.2168
  			},
  			{
  				nominal: 173,
  				mass: 172.9382151,
  				abundance: 0.16103
  			},
  			{
  				nominal: 174,
  				mass: 173.9388664,
  				abundance: 0.32026
  			},
  			{
  				nominal: 175,
  				mass: 174.9412808
  			},
  			{
  				nominal: 176,
  				mass: 175.9425764,
  				abundance: 0.12996
  			},
  			{
  				nominal: 177,
  				mass: 176.9452656
  			},
  			{
  				nominal: 178,
  				mass: 177.946651
  			},
  			{
  				nominal: 179,
  				mass: 178.95004
  			},
  			{
  				nominal: 180,
  				mass: 179.95212
  			},
  			{
  				nominal: 181,
  				mass: 180.95589
  			}
  		],
  		symbol: "Yb",
  		mass: 173.05415016631702,
  		name: "Ytterbium",
  		monoisotopicMass: 173.9388664
  	},
  	{
  		number: 71,
  		isotopes: [
  			{
  				nominal: 150,
  				mass: 149.97355
  			},
  			{
  				nominal: 151,
  				mass: 150.96768
  			},
  			{
  				nominal: 152,
  				mass: 151.96412
  			},
  			{
  				nominal: 153,
  				mass: 152.95875
  			},
  			{
  				nominal: 154,
  				mass: 153.95736
  			},
  			{
  				nominal: 155,
  				mass: 154.954321
  			},
  			{
  				nominal: 156,
  				mass: 155.953033
  			},
  			{
  				nominal: 157,
  				mass: 156.950127
  			},
  			{
  				nominal: 158,
  				mass: 157.949316
  			},
  			{
  				nominal: 159,
  				mass: 158.946636
  			},
  			{
  				nominal: 160,
  				mass: 159.946033
  			},
  			{
  				nominal: 161,
  				mass: 160.943572
  			},
  			{
  				nominal: 162,
  				mass: 161.943283
  			},
  			{
  				nominal: 163,
  				mass: 162.941179
  			},
  			{
  				nominal: 164,
  				mass: 163.941339
  			},
  			{
  				nominal: 165,
  				mass: 164.939407
  			},
  			{
  				nominal: 166,
  				mass: 165.939859
  			},
  			{
  				nominal: 167,
  				mass: 166.93827
  			},
  			{
  				nominal: 168,
  				mass: 167.938736
  			},
  			{
  				nominal: 169,
  				mass: 168.9376441
  			},
  			{
  				nominal: 170,
  				mass: 169.938478
  			},
  			{
  				nominal: 171,
  				mass: 170.937917
  			},
  			{
  				nominal: 172,
  				mass: 171.9390891
  			},
  			{
  				nominal: 173,
  				mass: 172.938934
  			},
  			{
  				nominal: 174,
  				mass: 173.9403409
  			},
  			{
  				nominal: 175,
  				mass: 174.9407752,
  				abundance: 0.97401
  			},
  			{
  				nominal: 176,
  				mass: 175.9426897,
  				abundance: 0.02599
  			},
  			{
  				nominal: 177,
  				mass: 176.9437615
  			},
  			{
  				nominal: 178,
  				mass: 177.945958
  			},
  			{
  				nominal: 179,
  				mass: 178.9473309
  			},
  			{
  				nominal: 180,
  				mass: 179.949888
  			},
  			{
  				nominal: 181,
  				mass: 180.95191
  			},
  			{
  				nominal: 182,
  				mass: 181.95504
  			},
  			{
  				nominal: 183,
  				mass: 182.957363
  			},
  			{
  				nominal: 184,
  				mass: 183.96091
  			},
  			{
  				nominal: 185,
  				mass: 184.96362
  			}
  		],
  		symbol: "Lu",
  		mass: 174.96681495785498,
  		name: "Lutetium",
  		monoisotopicMass: 174.9407752
  	},
  	{
  		number: 72,
  		isotopes: [
  			{
  				nominal: 153,
  				mass: 152.97069
  			},
  			{
  				nominal: 154,
  				mass: 153.96486
  			},
  			{
  				nominal: 155,
  				mass: 154.96311
  			},
  			{
  				nominal: 156,
  				mass: 155.95935
  			},
  			{
  				nominal: 157,
  				mass: 156.95824
  			},
  			{
  				nominal: 158,
  				mass: 157.954801
  			},
  			{
  				nominal: 159,
  				mass: 158.953996
  			},
  			{
  				nominal: 160,
  				mass: 159.950691
  			},
  			{
  				nominal: 161,
  				mass: 160.950278
  			},
  			{
  				nominal: 162,
  				mass: 161.9472148
  			},
  			{
  				nominal: 163,
  				mass: 162.947113
  			},
  			{
  				nominal: 164,
  				mass: 163.944371
  			},
  			{
  				nominal: 165,
  				mass: 164.944567
  			},
  			{
  				nominal: 166,
  				mass: 165.94218
  			},
  			{
  				nominal: 167,
  				mass: 166.9426
  			},
  			{
  				nominal: 168,
  				mass: 167.940568
  			},
  			{
  				nominal: 169,
  				mass: 168.941259
  			},
  			{
  				nominal: 170,
  				mass: 169.939609
  			},
  			{
  				nominal: 171,
  				mass: 170.940492
  			},
  			{
  				nominal: 172,
  				mass: 171.93945
  			},
  			{
  				nominal: 173,
  				mass: 172.940513
  			},
  			{
  				nominal: 174,
  				mass: 173.9400461,
  				abundance: 0.0016
  			},
  			{
  				nominal: 175,
  				mass: 174.9415092
  			},
  			{
  				nominal: 176,
  				mass: 175.9414076,
  				abundance: 0.0526
  			},
  			{
  				nominal: 177,
  				mass: 176.9432277,
  				abundance: 0.186
  			},
  			{
  				nominal: 178,
  				mass: 177.9437058,
  				abundance: 0.2728
  			},
  			{
  				nominal: 179,
  				mass: 178.9458232,
  				abundance: 0.1362
  			},
  			{
  				nominal: 180,
  				mass: 179.946557,
  				abundance: 0.3508
  			},
  			{
  				nominal: 181,
  				mass: 180.9491083
  			},
  			{
  				nominal: 182,
  				mass: 181.9505612
  			},
  			{
  				nominal: 183,
  				mass: 182.95353
  			},
  			{
  				nominal: 184,
  				mass: 183.955446
  			},
  			{
  				nominal: 185,
  				mass: 184.958862
  			},
  			{
  				nominal: 186,
  				mass: 185.960897
  			},
  			{
  				nominal: 187,
  				mass: 186.96477
  			},
  			{
  				nominal: 188,
  				mass: 187.96685
  			},
  			{
  				nominal: 189,
  				mass: 188.97084
  			}
  		],
  		symbol: "Hf",
  		mass: 178.4849787234,
  		name: "Hafnium",
  		monoisotopicMass: 179.946557
  	},
  	{
  		number: 73,
  		isotopes: [
  			{
  				nominal: 155,
  				mass: 154.97424
  			},
  			{
  				nominal: 156,
  				mass: 155.97203
  			},
  			{
  				nominal: 157,
  				mass: 156.96818
  			},
  			{
  				nominal: 158,
  				mass: 157.96654
  			},
  			{
  				nominal: 159,
  				mass: 158.963023
  			},
  			{
  				nominal: 160,
  				mass: 159.961488
  			},
  			{
  				nominal: 161,
  				mass: 160.958452
  			},
  			{
  				nominal: 162,
  				mass: 161.957294
  			},
  			{
  				nominal: 163,
  				mass: 162.954337
  			},
  			{
  				nominal: 164,
  				mass: 163.953534
  			},
  			{
  				nominal: 165,
  				mass: 164.950781
  			},
  			{
  				nominal: 166,
  				mass: 165.950512
  			},
  			{
  				nominal: 167,
  				mass: 166.948093
  			},
  			{
  				nominal: 168,
  				mass: 167.948047
  			},
  			{
  				nominal: 169,
  				mass: 168.946011
  			},
  			{
  				nominal: 170,
  				mass: 169.946175
  			},
  			{
  				nominal: 171,
  				mass: 170.944476
  			},
  			{
  				nominal: 172,
  				mass: 171.944895
  			},
  			{
  				nominal: 173,
  				mass: 172.94375
  			},
  			{
  				nominal: 174,
  				mass: 173.944454
  			},
  			{
  				nominal: 175,
  				mass: 174.943737
  			},
  			{
  				nominal: 176,
  				mass: 175.944857
  			},
  			{
  				nominal: 177,
  				mass: 176.9444795
  			},
  			{
  				nominal: 178,
  				mass: 177.945678
  			},
  			{
  				nominal: 179,
  				mass: 178.9459366
  			},
  			{
  				nominal: 180,
  				mass: 179.9474648,
  				abundance: 0.0001201
  			},
  			{
  				nominal: 181,
  				mass: 180.9479958,
  				abundance: 0.9998799
  			},
  			{
  				nominal: 182,
  				mass: 181.9501519
  			},
  			{
  				nominal: 183,
  				mass: 182.9513726
  			},
  			{
  				nominal: 184,
  				mass: 183.954008
  			},
  			{
  				nominal: 185,
  				mass: 184.955559
  			},
  			{
  				nominal: 186,
  				mass: 185.958551
  			},
  			{
  				nominal: 187,
  				mass: 186.960386
  			},
  			{
  				nominal: 188,
  				mass: 187.963916
  			},
  			{
  				nominal: 189,
  				mass: 188.96583
  			},
  			{
  				nominal: 190,
  				mass: 189.96939
  			},
  			{
  				nominal: 191,
  				mass: 190.97156
  			},
  			{
  				nominal: 192,
  				mass: 191.97514
  			}
  		],
  		symbol: "Ta",
  		mass: 180.9478756362269,
  		name: "Tantalum",
  		monoisotopicMass: 180.9479958
  	},
  	{
  		number: 74,
  		isotopes: [
  			{
  				nominal: 157,
  				mass: 156.97884
  			},
  			{
  				nominal: 158,
  				mass: 157.97456
  			},
  			{
  				nominal: 159,
  				mass: 158.97264
  			},
  			{
  				nominal: 160,
  				mass: 159.96846
  			},
  			{
  				nominal: 161,
  				mass: 160.9672
  			},
  			{
  				nominal: 162,
  				mass: 161.963499
  			},
  			{
  				nominal: 163,
  				mass: 162.962524
  			},
  			{
  				nominal: 164,
  				mass: 163.958961
  			},
  			{
  				nominal: 165,
  				mass: 164.958281
  			},
  			{
  				nominal: 166,
  				mass: 165.955031
  			},
  			{
  				nominal: 167,
  				mass: 166.954805
  			},
  			{
  				nominal: 168,
  				mass: 167.951806
  			},
  			{
  				nominal: 169,
  				mass: 168.951779
  			},
  			{
  				nominal: 170,
  				mass: 169.949232
  			},
  			{
  				nominal: 171,
  				mass: 170.949451
  			},
  			{
  				nominal: 172,
  				mass: 171.947292
  			},
  			{
  				nominal: 173,
  				mass: 172.947689
  			},
  			{
  				nominal: 174,
  				mass: 173.946079
  			},
  			{
  				nominal: 175,
  				mass: 174.946717
  			},
  			{
  				nominal: 176,
  				mass: 175.945634
  			},
  			{
  				nominal: 177,
  				mass: 176.946643
  			},
  			{
  				nominal: 178,
  				mass: 177.945883
  			},
  			{
  				nominal: 179,
  				mass: 178.947077
  			},
  			{
  				nominal: 180,
  				mass: 179.9467108,
  				abundance: 0.0012
  			},
  			{
  				nominal: 181,
  				mass: 180.9481978
  			},
  			{
  				nominal: 182,
  				mass: 181.94820394,
  				abundance: 0.265
  			},
  			{
  				nominal: 183,
  				mass: 182.95022275,
  				abundance: 0.1431
  			},
  			{
  				nominal: 184,
  				mass: 183.95093092,
  				abundance: 0.3064
  			},
  			{
  				nominal: 185,
  				mass: 184.95341897
  			},
  			{
  				nominal: 186,
  				mass: 185.9543628,
  				abundance: 0.2843
  			},
  			{
  				nominal: 187,
  				mass: 186.9571588
  			},
  			{
  				nominal: 188,
  				mass: 187.9584862
  			},
  			{
  				nominal: 189,
  				mass: 188.961763
  			},
  			{
  				nominal: 190,
  				mass: 189.963091
  			},
  			{
  				nominal: 191,
  				mass: 190.966531
  			},
  			{
  				nominal: 192,
  				mass: 191.96817
  			},
  			{
  				nominal: 193,
  				mass: 192.97178
  			},
  			{
  				nominal: 194,
  				mass: 193.97367
  			}
  		],
  		symbol: "W",
  		mass: 183.841777550513,
  		name: "Tungsten",
  		monoisotopicMass: 183.95093092
  	},
  	{
  		number: 75,
  		isotopes: [
  			{
  				nominal: 159,
  				mass: 158.98418
  			},
  			{
  				nominal: 160,
  				mass: 159.98182
  			},
  			{
  				nominal: 161,
  				mass: 160.97757
  			},
  			{
  				nominal: 162,
  				mass: 161.97584
  			},
  			{
  				nominal: 163,
  				mass: 162.97208
  			},
  			{
  				nominal: 164,
  				mass: 163.970453
  			},
  			{
  				nominal: 165,
  				mass: 164.967103
  			},
  			{
  				nominal: 166,
  				mass: 165.965761
  			},
  			{
  				nominal: 167,
  				mass: 166.962595
  			},
  			{
  				nominal: 168,
  				mass: 167.961573
  			},
  			{
  				nominal: 169,
  				mass: 168.958766
  			},
  			{
  				nominal: 170,
  				mass: 169.95822
  			},
  			{
  				nominal: 171,
  				mass: 170.955716
  			},
  			{
  				nominal: 172,
  				mass: 171.95542
  			},
  			{
  				nominal: 173,
  				mass: 172.953243
  			},
  			{
  				nominal: 174,
  				mass: 173.953115
  			},
  			{
  				nominal: 175,
  				mass: 174.951381
  			},
  			{
  				nominal: 176,
  				mass: 175.951623
  			},
  			{
  				nominal: 177,
  				mass: 176.950328
  			},
  			{
  				nominal: 178,
  				mass: 177.950989
  			},
  			{
  				nominal: 179,
  				mass: 178.949989
  			},
  			{
  				nominal: 180,
  				mass: 179.950792
  			},
  			{
  				nominal: 181,
  				mass: 180.950058
  			},
  			{
  				nominal: 182,
  				mass: 181.95121
  			},
  			{
  				nominal: 183,
  				mass: 182.9508196
  			},
  			{
  				nominal: 184,
  				mass: 183.9525228
  			},
  			{
  				nominal: 185,
  				mass: 184.9529545,
  				abundance: 0.374
  			},
  			{
  				nominal: 186,
  				mass: 185.9549856
  			},
  			{
  				nominal: 187,
  				mass: 186.9557501,
  				abundance: 0.626
  			},
  			{
  				nominal: 188,
  				mass: 187.9581115
  			},
  			{
  				nominal: 189,
  				mass: 188.959226
  			},
  			{
  				nominal: 190,
  				mass: 189.961744
  			},
  			{
  				nominal: 191,
  				mass: 190.963122
  			},
  			{
  				nominal: 192,
  				mass: 191.966088
  			},
  			{
  				nominal: 193,
  				mass: 192.967541
  			},
  			{
  				nominal: 194,
  				mass: 193.97076
  			},
  			{
  				nominal: 195,
  				mass: 194.97254
  			},
  			{
  				nominal: 196,
  				mass: 195.9758
  			},
  			{
  				nominal: 197,
  				mass: 196.97799
  			},
  			{
  				nominal: 198,
  				mass: 197.9816
  			}
  		],
  		symbol: "Re",
  		mass: 186.20670454560002,
  		name: "Rhenium",
  		monoisotopicMass: 186.9557501
  	},
  	{
  		number: 76,
  		isotopes: [
  			{
  				nominal: 161,
  				mass: 160.98903
  			},
  			{
  				nominal: 162,
  				mass: 161.98443
  			},
  			{
  				nominal: 163,
  				mass: 162.98241
  			},
  			{
  				nominal: 164,
  				mass: 163.97802
  			},
  			{
  				nominal: 165,
  				mass: 164.9766
  			},
  			{
  				nominal: 166,
  				mass: 165.972692
  			},
  			{
  				nominal: 167,
  				mass: 166.971549
  			},
  			{
  				nominal: 168,
  				mass: 167.967808
  			},
  			{
  				nominal: 169,
  				mass: 168.967018
  			},
  			{
  				nominal: 170,
  				mass: 169.963578
  			},
  			{
  				nominal: 171,
  				mass: 170.963174
  			},
  			{
  				nominal: 172,
  				mass: 171.960017
  			},
  			{
  				nominal: 173,
  				mass: 172.959808
  			},
  			{
  				nominal: 174,
  				mass: 173.957064
  			},
  			{
  				nominal: 175,
  				mass: 174.956945
  			},
  			{
  				nominal: 176,
  				mass: 175.954806
  			},
  			{
  				nominal: 177,
  				mass: 176.954966
  			},
  			{
  				nominal: 178,
  				mass: 177.953254
  			},
  			{
  				nominal: 179,
  				mass: 178.953817
  			},
  			{
  				nominal: 180,
  				mass: 179.952375
  			},
  			{
  				nominal: 181,
  				mass: 180.953247
  			},
  			{
  				nominal: 182,
  				mass: 181.95211
  			},
  			{
  				nominal: 183,
  				mass: 182.953125
  			},
  			{
  				nominal: 184,
  				mass: 183.9524885,
  				abundance: 0.0002
  			},
  			{
  				nominal: 185,
  				mass: 184.9540417
  			},
  			{
  				nominal: 186,
  				mass: 185.953835,
  				abundance: 0.0159
  			},
  			{
  				nominal: 187,
  				mass: 186.9557474,
  				abundance: 0.0196
  			},
  			{
  				nominal: 188,
  				mass: 187.9558352,
  				abundance: 0.1324
  			},
  			{
  				nominal: 189,
  				mass: 188.9581442,
  				abundance: 0.1615
  			},
  			{
  				nominal: 190,
  				mass: 189.9584437,
  				abundance: 0.2626
  			},
  			{
  				nominal: 191,
  				mass: 190.9609264
  			},
  			{
  				nominal: 192,
  				mass: 191.961477,
  				abundance: 0.4078
  			},
  			{
  				nominal: 193,
  				mass: 192.9641479
  			},
  			{
  				nominal: 194,
  				mass: 193.9651772
  			},
  			{
  				nominal: 195,
  				mass: 194.968318
  			},
  			{
  				nominal: 196,
  				mass: 195.969641
  			},
  			{
  				nominal: 197,
  				mass: 196.97283
  			},
  			{
  				nominal: 198,
  				mass: 197.97441
  			},
  			{
  				nominal: 199,
  				mass: 198.97801
  			},
  			{
  				nominal: 200,
  				mass: 199.97984
  			},
  			{
  				nominal: 201,
  				mass: 200.98364
  			},
  			{
  				nominal: 202,
  				mass: 201.98595
  			}
  		],
  		symbol: "Os",
  		mass: 190.22485962823998,
  		name: "Osmium",
  		monoisotopicMass: 191.961477
  	},
  	{
  		number: 77,
  		isotopes: [
  			{
  				nominal: 164,
  				mass: 163.99191
  			},
  			{
  				nominal: 165,
  				mass: 164.9875
  			},
  			{
  				nominal: 166,
  				mass: 165.98566
  			},
  			{
  				nominal: 167,
  				mass: 166.981666
  			},
  			{
  				nominal: 168,
  				mass: 167.979907
  			},
  			{
  				nominal: 169,
  				mass: 168.976298
  			},
  			{
  				nominal: 170,
  				mass: 169.974922
  			},
  			{
  				nominal: 171,
  				mass: 170.97164
  			},
  			{
  				nominal: 172,
  				mass: 171.970607
  			},
  			{
  				nominal: 173,
  				mass: 172.967506
  			},
  			{
  				nominal: 174,
  				mass: 173.966861
  			},
  			{
  				nominal: 175,
  				mass: 174.96415
  			},
  			{
  				nominal: 176,
  				mass: 175.96365
  			},
  			{
  				nominal: 177,
  				mass: 176.961301
  			},
  			{
  				nominal: 178,
  				mass: 177.961082
  			},
  			{
  				nominal: 179,
  				mass: 178.95912
  			},
  			{
  				nominal: 180,
  				mass: 179.959229
  			},
  			{
  				nominal: 181,
  				mass: 180.957625
  			},
  			{
  				nominal: 182,
  				mass: 181.958076
  			},
  			{
  				nominal: 183,
  				mass: 182.95684
  			},
  			{
  				nominal: 184,
  				mass: 183.957476
  			},
  			{
  				nominal: 185,
  				mass: 184.956698
  			},
  			{
  				nominal: 186,
  				mass: 185.957944
  			},
  			{
  				nominal: 187,
  				mass: 186.957542
  			},
  			{
  				nominal: 188,
  				mass: 187.958828
  			},
  			{
  				nominal: 189,
  				mass: 188.958715
  			},
  			{
  				nominal: 190,
  				mass: 189.9605412
  			},
  			{
  				nominal: 191,
  				mass: 190.9605893,
  				abundance: 0.373
  			},
  			{
  				nominal: 192,
  				mass: 191.9626002
  			},
  			{
  				nominal: 193,
  				mass: 192.9629216,
  				abundance: 0.627
  			},
  			{
  				nominal: 194,
  				mass: 193.9650735
  			},
  			{
  				nominal: 195,
  				mass: 194.9659747
  			},
  			{
  				nominal: 196,
  				mass: 195.968397
  			},
  			{
  				nominal: 197,
  				mass: 196.969655
  			},
  			{
  				nominal: 198,
  				mass: 197.97228
  			},
  			{
  				nominal: 199,
  				mass: 198.973805
  			},
  			{
  				nominal: 200,
  				mass: 199.9768
  			},
  			{
  				nominal: 201,
  				mass: 200.97864
  			},
  			{
  				nominal: 202,
  				mass: 201.98199
  			},
  			{
  				nominal: 203,
  				mass: 202.98423
  			},
  			{
  				nominal: 204,
  				mass: 203.9896
  			}
  		],
  		symbol: "Ir",
  		mass: 192.2160516521,
  		name: "Iridium",
  		monoisotopicMass: 192.9629216
  	},
  	{
  		number: 78,
  		isotopes: [
  			{
  				nominal: 166,
  				mass: 165.99486
  			},
  			{
  				nominal: 167,
  				mass: 166.99269
  			},
  			{
  				nominal: 168,
  				mass: 167.98813
  			},
  			{
  				nominal: 169,
  				mass: 168.98657
  			},
  			{
  				nominal: 170,
  				mass: 169.982496
  			},
  			{
  				nominal: 171,
  				mass: 170.981245
  			},
  			{
  				nominal: 172,
  				mass: 171.977351
  			},
  			{
  				nominal: 173,
  				mass: 172.976443
  			},
  			{
  				nominal: 174,
  				mass: 173.97282
  			},
  			{
  				nominal: 175,
  				mass: 174.97241
  			},
  			{
  				nominal: 176,
  				mass: 175.968938
  			},
  			{
  				nominal: 177,
  				mass: 176.96847
  			},
  			{
  				nominal: 178,
  				mass: 177.96565
  			},
  			{
  				nominal: 179,
  				mass: 178.965359
  			},
  			{
  				nominal: 180,
  				mass: 179.963032
  			},
  			{
  				nominal: 181,
  				mass: 180.963098
  			},
  			{
  				nominal: 182,
  				mass: 181.961172
  			},
  			{
  				nominal: 183,
  				mass: 182.961597
  			},
  			{
  				nominal: 184,
  				mass: 183.959915
  			},
  			{
  				nominal: 185,
  				mass: 184.960614
  			},
  			{
  				nominal: 186,
  				mass: 185.959351
  			},
  			{
  				nominal: 187,
  				mass: 186.960617
  			},
  			{
  				nominal: 188,
  				mass: 187.9593889
  			},
  			{
  				nominal: 189,
  				mass: 188.960831
  			},
  			{
  				nominal: 190,
  				mass: 189.9599297,
  				abundance: 0.00012
  			},
  			{
  				nominal: 191,
  				mass: 190.9616729
  			},
  			{
  				nominal: 192,
  				mass: 191.9610387,
  				abundance: 0.00782
  			},
  			{
  				nominal: 193,
  				mass: 192.9629824
  			},
  			{
  				nominal: 194,
  				mass: 193.9626809,
  				abundance: 0.3286
  			},
  			{
  				nominal: 195,
  				mass: 194.9647917,
  				abundance: 0.3378
  			},
  			{
  				nominal: 196,
  				mass: 195.96495209,
  				abundance: 0.2521
  			},
  			{
  				nominal: 197,
  				mass: 196.96734069
  			},
  			{
  				nominal: 198,
  				mass: 197.9678949,
  				abundance: 0.07356
  			},
  			{
  				nominal: 199,
  				mass: 198.9705952
  			},
  			{
  				nominal: 200,
  				mass: 199.971443
  			},
  			{
  				nominal: 201,
  				mass: 200.974513
  			},
  			{
  				nominal: 202,
  				mass: 201.975639
  			},
  			{
  				nominal: 203,
  				mass: 202.97893
  			},
  			{
  				nominal: 204,
  				mass: 203.98076
  			},
  			{
  				nominal: 205,
  				mass: 204.98608
  			},
  			{
  				nominal: 206,
  				mass: 205.98966
  			}
  		],
  		symbol: "Pt",
  		mass: 195.084456864931,
  		name: "Platinum",
  		monoisotopicMass: 194.9647917
  	},
  	{
  		number: 79,
  		isotopes: [
  			{
  				nominal: 169,
  				mass: 168.99808
  			},
  			{
  				nominal: 170,
  				mass: 169.99597
  			},
  			{
  				nominal: 171,
  				mass: 170.991876
  			},
  			{
  				nominal: 172,
  				mass: 171.989942
  			},
  			{
  				nominal: 173,
  				mass: 172.986241
  			},
  			{
  				nominal: 174,
  				mass: 173.984717
  			},
  			{
  				nominal: 175,
  				mass: 174.981304
  			},
  			{
  				nominal: 176,
  				mass: 175.98025
  			},
  			{
  				nominal: 177,
  				mass: 176.97687
  			},
  			{
  				nominal: 178,
  				mass: 177.976032
  			},
  			{
  				nominal: 179,
  				mass: 178.973174
  			},
  			{
  				nominal: 180,
  				mass: 179.972523
  			},
  			{
  				nominal: 181,
  				mass: 180.970079
  			},
  			{
  				nominal: 182,
  				mass: 181.969618
  			},
  			{
  				nominal: 183,
  				mass: 182.967591
  			},
  			{
  				nominal: 184,
  				mass: 183.967452
  			},
  			{
  				nominal: 185,
  				mass: 184.96579
  			},
  			{
  				nominal: 186,
  				mass: 185.965953
  			},
  			{
  				nominal: 187,
  				mass: 186.964543
  			},
  			{
  				nominal: 188,
  				mass: 187.965349
  			},
  			{
  				nominal: 189,
  				mass: 188.963948
  			},
  			{
  				nominal: 190,
  				mass: 189.964698
  			},
  			{
  				nominal: 191,
  				mass: 190.963702
  			},
  			{
  				nominal: 192,
  				mass: 191.964814
  			},
  			{
  				nominal: 193,
  				mass: 192.9641373
  			},
  			{
  				nominal: 194,
  				mass: 193.9654178
  			},
  			{
  				nominal: 195,
  				mass: 194.9650352
  			},
  			{
  				nominal: 196,
  				mass: 195.9665699
  			},
  			{
  				nominal: 197,
  				mass: 196.96656879,
  				abundance: 1
  			},
  			{
  				nominal: 198,
  				mass: 197.96824242
  			},
  			{
  				nominal: 199,
  				mass: 198.96876528
  			},
  			{
  				nominal: 200,
  				mass: 199.970756
  			},
  			{
  				nominal: 201,
  				mass: 200.9716575
  			},
  			{
  				nominal: 202,
  				mass: 201.973856
  			},
  			{
  				nominal: 203,
  				mass: 202.9751544
  			},
  			{
  				nominal: 204,
  				mass: 203.97783
  			},
  			{
  				nominal: 205,
  				mass: 204.97985
  			},
  			{
  				nominal: 206,
  				mass: 205.98474
  			},
  			{
  				nominal: 207,
  				mass: 206.9884
  			},
  			{
  				nominal: 208,
  				mass: 207.99345
  			},
  			{
  				nominal: 209,
  				mass: 208.99735
  			},
  			{
  				nominal: 210,
  				mass: 210.0025
  			}
  		],
  		symbol: "Au",
  		mass: 196.96656879,
  		name: "Gold",
  		monoisotopicMass: 196.96656879
  	},
  	{
  		number: 80,
  		isotopes: [
  			{
  				nominal: 171,
  				mass: 171.00353
  			},
  			{
  				nominal: 172,
  				mass: 171.99881
  			},
  			{
  				nominal: 173,
  				mass: 172.99709
  			},
  			{
  				nominal: 174,
  				mass: 173.992865
  			},
  			{
  				nominal: 175,
  				mass: 174.991441
  			},
  			{
  				nominal: 176,
  				mass: 175.987361
  			},
  			{
  				nominal: 177,
  				mass: 176.986277
  			},
  			{
  				nominal: 178,
  				mass: 177.982484
  			},
  			{
  				nominal: 179,
  				mass: 178.981831
  			},
  			{
  				nominal: 180,
  				mass: 179.97826
  			},
  			{
  				nominal: 181,
  				mass: 180.977819
  			},
  			{
  				nominal: 182,
  				mass: 181.974689
  			},
  			{
  				nominal: 183,
  				mass: 182.9744448
  			},
  			{
  				nominal: 184,
  				mass: 183.971714
  			},
  			{
  				nominal: 185,
  				mass: 184.971899
  			},
  			{
  				nominal: 186,
  				mass: 185.969362
  			},
  			{
  				nominal: 187,
  				mass: 186.969814
  			},
  			{
  				nominal: 188,
  				mass: 187.967567
  			},
  			{
  				nominal: 189,
  				mass: 188.968195
  			},
  			{
  				nominal: 190,
  				mass: 189.966323
  			},
  			{
  				nominal: 191,
  				mass: 190.967157
  			},
  			{
  				nominal: 192,
  				mass: 191.965635
  			},
  			{
  				nominal: 193,
  				mass: 192.966653
  			},
  			{
  				nominal: 194,
  				mass: 193.9654491
  			},
  			{
  				nominal: 195,
  				mass: 194.966721
  			},
  			{
  				nominal: 196,
  				mass: 195.9658326,
  				abundance: 0.0015
  			},
  			{
  				nominal: 197,
  				mass: 196.9672128
  			},
  			{
  				nominal: 198,
  				mass: 197.9667686,
  				abundance: 0.0997
  			},
  			{
  				nominal: 199,
  				mass: 198.96828064,
  				abundance: 0.1687
  			},
  			{
  				nominal: 200,
  				mass: 199.96832659,
  				abundance: 0.231
  			},
  			{
  				nominal: 201,
  				mass: 200.97030284,
  				abundance: 0.1318
  			},
  			{
  				nominal: 202,
  				mass: 201.9706434,
  				abundance: 0.2986
  			},
  			{
  				nominal: 203,
  				mass: 202.9728728
  			},
  			{
  				nominal: 204,
  				mass: 203.97349398,
  				abundance: 0.0687
  			},
  			{
  				nominal: 205,
  				mass: 204.9760734
  			},
  			{
  				nominal: 206,
  				mass: 205.977514
  			},
  			{
  				nominal: 207,
  				mass: 206.9823
  			},
  			{
  				nominal: 208,
  				mass: 207.985759
  			},
  			{
  				nominal: 209,
  				mass: 208.99072
  			},
  			{
  				nominal: 210,
  				mass: 209.99424
  			},
  			{
  				nominal: 211,
  				mass: 210.99933
  			},
  			{
  				nominal: 212,
  				mass: 212.00296
  			},
  			{
  				nominal: 213,
  				mass: 213.00823
  			},
  			{
  				nominal: 214,
  				mass: 214.012
  			},
  			{
  				nominal: 215,
  				mass: 215.0174
  			},
  			{
  				nominal: 216,
  				mass: 216.02132
  			}
  		],
  		symbol: "Hg",
  		mass: 200.59916703455602,
  		name: "Mercury",
  		monoisotopicMass: 201.9706434
  	},
  	{
  		number: 81,
  		isotopes: [
  			{
  				nominal: 176,
  				mass: 176.000624
  			},
  			{
  				nominal: 177,
  				mass: 176.996431
  			},
  			{
  				nominal: 178,
  				mass: 177.99485
  			},
  			{
  				nominal: 179,
  				mass: 178.991111
  			},
  			{
  				nominal: 180,
  				mass: 179.990057
  			},
  			{
  				nominal: 181,
  				mass: 180.98626
  			},
  			{
  				nominal: 182,
  				mass: 181.985713
  			},
  			{
  				nominal: 183,
  				mass: 182.982193
  			},
  			{
  				nominal: 184,
  				mass: 183.981886
  			},
  			{
  				nominal: 185,
  				mass: 184.978789
  			},
  			{
  				nominal: 186,
  				mass: 185.978651
  			},
  			{
  				nominal: 187,
  				mass: 186.9759063
  			},
  			{
  				nominal: 188,
  				mass: 187.976021
  			},
  			{
  				nominal: 189,
  				mass: 188.973588
  			},
  			{
  				nominal: 190,
  				mass: 189.973828
  			},
  			{
  				nominal: 191,
  				mass: 190.9717842
  			},
  			{
  				nominal: 192,
  				mass: 191.972225
  			},
  			{
  				nominal: 193,
  				mass: 192.970502
  			},
  			{
  				nominal: 194,
  				mass: 193.971081
  			},
  			{
  				nominal: 195,
  				mass: 194.969774
  			},
  			{
  				nominal: 196,
  				mass: 195.970481
  			},
  			{
  				nominal: 197,
  				mass: 196.969576
  			},
  			{
  				nominal: 198,
  				mass: 197.970483
  			},
  			{
  				nominal: 199,
  				mass: 198.969877
  			},
  			{
  				nominal: 200,
  				mass: 199.9709633
  			},
  			{
  				nominal: 201,
  				mass: 200.970822
  			},
  			{
  				nominal: 202,
  				mass: 201.972102
  			},
  			{
  				nominal: 203,
  				mass: 202.9723446,
  				abundance: 0.2952
  			},
  			{
  				nominal: 204,
  				mass: 203.9738639
  			},
  			{
  				nominal: 205,
  				mass: 204.9744278,
  				abundance: 0.7048
  			},
  			{
  				nominal: 206,
  				mass: 205.9761106
  			},
  			{
  				nominal: 207,
  				mass: 206.9774197
  			},
  			{
  				nominal: 208,
  				mass: 207.982019
  			},
  			{
  				nominal: 209,
  				mass: 208.9853594
  			},
  			{
  				nominal: 210,
  				mass: 209.990074
  			},
  			{
  				nominal: 211,
  				mass: 210.993475
  			},
  			{
  				nominal: 212,
  				mass: 211.99834
  			},
  			{
  				nominal: 213,
  				mass: 213.001915
  			},
  			{
  				nominal: 214,
  				mass: 214.00694
  			},
  			{
  				nominal: 215,
  				mass: 215.01064
  			},
  			{
  				nominal: 216,
  				mass: 216.0158
  			},
  			{
  				nominal: 217,
  				mass: 217.01966
  			},
  			{
  				nominal: 218,
  				mass: 218.02479
  			}
  		],
  		symbol: "Tl",
  		mass: 204.38341283936,
  		name: "Thallium",
  		monoisotopicMass: 204.9744278
  	},
  	{
  		number: 82,
  		isotopes: [
  			{
  				nominal: 178,
  				mass: 178.003831
  			},
  			{
  				nominal: 179,
  				mass: 179.002201
  			},
  			{
  				nominal: 180,
  				mass: 179.997928
  			},
  			{
  				nominal: 181,
  				mass: 180.996653
  			},
  			{
  				nominal: 182,
  				mass: 181.992672
  			},
  			{
  				nominal: 183,
  				mass: 182.991872
  			},
  			{
  				nominal: 184,
  				mass: 183.988136
  			},
  			{
  				nominal: 185,
  				mass: 184.98761
  			},
  			{
  				nominal: 186,
  				mass: 185.984238
  			},
  			{
  				nominal: 187,
  				mass: 186.9839109
  			},
  			{
  				nominal: 188,
  				mass: 187.980875
  			},
  			{
  				nominal: 189,
  				mass: 188.980807
  			},
  			{
  				nominal: 190,
  				mass: 189.978082
  			},
  			{
  				nominal: 191,
  				mass: 190.978276
  			},
  			{
  				nominal: 192,
  				mass: 191.975775
  			},
  			{
  				nominal: 193,
  				mass: 192.976173
  			},
  			{
  				nominal: 194,
  				mass: 193.974012
  			},
  			{
  				nominal: 195,
  				mass: 194.974543
  			},
  			{
  				nominal: 196,
  				mass: 195.972774
  			},
  			{
  				nominal: 197,
  				mass: 196.9734312
  			},
  			{
  				nominal: 198,
  				mass: 197.972034
  			},
  			{
  				nominal: 199,
  				mass: 198.972913
  			},
  			{
  				nominal: 200,
  				mass: 199.971819
  			},
  			{
  				nominal: 201,
  				mass: 200.972883
  			},
  			{
  				nominal: 202,
  				mass: 201.972152
  			},
  			{
  				nominal: 203,
  				mass: 202.9733911
  			},
  			{
  				nominal: 204,
  				mass: 203.973044,
  				abundance: 0.014
  			},
  			{
  				nominal: 205,
  				mass: 204.9744822
  			},
  			{
  				nominal: 206,
  				mass: 205.9744657,
  				abundance: 0.241
  			},
  			{
  				nominal: 207,
  				mass: 206.9758973,
  				abundance: 0.221
  			},
  			{
  				nominal: 208,
  				mass: 207.9766525,
  				abundance: 0.524
  			},
  			{
  				nominal: 209,
  				mass: 208.9810905
  			},
  			{
  				nominal: 210,
  				mass: 209.9841889
  			},
  			{
  				nominal: 211,
  				mass: 210.9887371
  			},
  			{
  				nominal: 212,
  				mass: 211.9918977
  			},
  			{
  				nominal: 213,
  				mass: 212.9965629
  			},
  			{
  				nominal: 214,
  				mass: 213.9998059
  			},
  			{
  				nominal: 215,
  				mass: 215.00474
  			},
  			{
  				nominal: 216,
  				mass: 216.00803
  			},
  			{
  				nominal: 217,
  				mass: 217.01314
  			},
  			{
  				nominal: 218,
  				mass: 218.01659
  			},
  			{
  				nominal: 219,
  				mass: 219.02177
  			},
  			{
  				nominal: 220,
  				mass: 220.02541
  			}
  		],
  		symbol: "Pb",
  		mass: 207.216908063,
  		name: "Lead",
  		monoisotopicMass: 207.9766525
  	},
  	{
  		number: 83,
  		isotopes: [
  			{
  				nominal: 184,
  				mass: 184.001275
  			},
  			{
  				nominal: 185,
  				mass: 184.9976
  			},
  			{
  				nominal: 186,
  				mass: 185.996644
  			},
  			{
  				nominal: 187,
  				mass: 186.993147
  			},
  			{
  				nominal: 188,
  				mass: 187.992287
  			},
  			{
  				nominal: 189,
  				mass: 188.989195
  			},
  			{
  				nominal: 190,
  				mass: 189.988622
  			},
  			{
  				nominal: 191,
  				mass: 190.9857866
  			},
  			{
  				nominal: 192,
  				mass: 191.985469
  			},
  			{
  				nominal: 193,
  				mass: 192.98296
  			},
  			{
  				nominal: 194,
  				mass: 193.982785
  			},
  			{
  				nominal: 195,
  				mass: 194.9806488
  			},
  			{
  				nominal: 196,
  				mass: 195.980667
  			},
  			{
  				nominal: 197,
  				mass: 196.9788651
  			},
  			{
  				nominal: 198,
  				mass: 197.979206
  			},
  			{
  				nominal: 199,
  				mass: 198.977673
  			},
  			{
  				nominal: 200,
  				mass: 199.978131
  			},
  			{
  				nominal: 201,
  				mass: 200.97701
  			},
  			{
  				nominal: 202,
  				mass: 201.977734
  			},
  			{
  				nominal: 203,
  				mass: 202.976893
  			},
  			{
  				nominal: 204,
  				mass: 203.9778361
  			},
  			{
  				nominal: 205,
  				mass: 204.9773867
  			},
  			{
  				nominal: 206,
  				mass: 205.9784993
  			},
  			{
  				nominal: 207,
  				mass: 206.978471
  			},
  			{
  				nominal: 208,
  				mass: 207.9797425
  			},
  			{
  				nominal: 209,
  				mass: 208.9803991,
  				abundance: 1
  			},
  			{
  				nominal: 210,
  				mass: 209.9841207
  			},
  			{
  				nominal: 211,
  				mass: 210.9872697
  			},
  			{
  				nominal: 212,
  				mass: 211.991286
  			},
  			{
  				nominal: 213,
  				mass: 212.9943851
  			},
  			{
  				nominal: 214,
  				mass: 213.998712
  			},
  			{
  				nominal: 215,
  				mass: 215.00177
  			},
  			{
  				nominal: 216,
  				mass: 216.006306
  			},
  			{
  				nominal: 217,
  				mass: 217.009372
  			},
  			{
  				nominal: 218,
  				mass: 218.014188
  			},
  			{
  				nominal: 219,
  				mass: 219.01748
  			},
  			{
  				nominal: 220,
  				mass: 220.02235
  			},
  			{
  				nominal: 221,
  				mass: 221.02587
  			},
  			{
  				nominal: 222,
  				mass: 222.03078
  			},
  			{
  				nominal: 223,
  				mass: 223.0345
  			},
  			{
  				nominal: 224,
  				mass: 224.03947
  			}
  		],
  		symbol: "Bi",
  		mass: 208.9803991,
  		name: "Bismuth",
  		monoisotopicMass: 208.9803991
  	},
  	{
  		number: 84,
  		isotopes: [
  			{
  				nominal: 186,
  				mass: 186.004393
  			},
  			{
  				nominal: 187,
  				mass: 187.003041
  			},
  			{
  				nominal: 188,
  				mass: 187.999416
  			},
  			{
  				nominal: 189,
  				mass: 188.998473
  			},
  			{
  				nominal: 190,
  				mass: 189.995101
  			},
  			{
  				nominal: 191,
  				mass: 190.9945585
  			},
  			{
  				nominal: 192,
  				mass: 191.991336
  			},
  			{
  				nominal: 193,
  				mass: 192.991026
  			},
  			{
  				nominal: 194,
  				mass: 193.988186
  			},
  			{
  				nominal: 195,
  				mass: 194.988126
  			},
  			{
  				nominal: 196,
  				mass: 195.985526
  			},
  			{
  				nominal: 197,
  				mass: 196.98566
  			},
  			{
  				nominal: 198,
  				mass: 197.983389
  			},
  			{
  				nominal: 199,
  				mass: 198.983667
  			},
  			{
  				nominal: 200,
  				mass: 199.981799
  			},
  			{
  				nominal: 201,
  				mass: 200.9822598
  			},
  			{
  				nominal: 202,
  				mass: 201.980758
  			},
  			{
  				nominal: 203,
  				mass: 202.9814161
  			},
  			{
  				nominal: 204,
  				mass: 203.98031
  			},
  			{
  				nominal: 205,
  				mass: 204.981203
  			},
  			{
  				nominal: 206,
  				mass: 205.980474
  			},
  			{
  				nominal: 207,
  				mass: 206.9815938
  			},
  			{
  				nominal: 208,
  				mass: 207.9812461
  			},
  			{
  				nominal: 209,
  				mass: 208.9824308
  			},
  			{
  				nominal: 210,
  				mass: 209.9828741
  			},
  			{
  				nominal: 211,
  				mass: 210.9866536
  			},
  			{
  				nominal: 212,
  				mass: 211.9888684
  			},
  			{
  				nominal: 213,
  				mass: 212.9928576
  			},
  			{
  				nominal: 214,
  				mass: 213.9952017
  			},
  			{
  				nominal: 215,
  				mass: 214.9994201
  			},
  			{
  				nominal: 216,
  				mass: 216.0019152
  			},
  			{
  				nominal: 217,
  				mass: 217.0063182
  			},
  			{
  				nominal: 218,
  				mass: 218.0089735
  			},
  			{
  				nominal: 219,
  				mass: 219.013614
  			},
  			{
  				nominal: 220,
  				mass: 220.016386
  			},
  			{
  				nominal: 221,
  				mass: 221.021228
  			},
  			{
  				nominal: 222,
  				mass: 222.02414
  			},
  			{
  				nominal: 223,
  				mass: 223.02907
  			},
  			{
  				nominal: 224,
  				mass: 224.03211
  			},
  			{
  				nominal: 225,
  				mass: 225.03707
  			},
  			{
  				nominal: 226,
  				mass: 226.04031
  			},
  			{
  				nominal: 227,
  				mass: 227.04539
  			}
  		],
  		symbol: "Po",
  		mass: null,
  		name: "Polonium"
  	},
  	{
  		number: 85,
  		isotopes: [
  			{
  				nominal: 191,
  				mass: 191.004148
  			},
  			{
  				nominal: 192,
  				mass: 192.003152
  			},
  			{
  				nominal: 193,
  				mass: 192.999927
  			},
  			{
  				nominal: 194,
  				mass: 193.999236
  			},
  			{
  				nominal: 195,
  				mass: 194.9962685
  			},
  			{
  				nominal: 196,
  				mass: 195.9958
  			},
  			{
  				nominal: 197,
  				mass: 196.993189
  			},
  			{
  				nominal: 198,
  				mass: 197.992784
  			},
  			{
  				nominal: 199,
  				mass: 198.9905277
  			},
  			{
  				nominal: 200,
  				mass: 199.990351
  			},
  			{
  				nominal: 201,
  				mass: 200.9884171
  			},
  			{
  				nominal: 202,
  				mass: 201.98863
  			},
  			{
  				nominal: 203,
  				mass: 202.986943
  			},
  			{
  				nominal: 204,
  				mass: 203.987251
  			},
  			{
  				nominal: 205,
  				mass: 204.986076
  			},
  			{
  				nominal: 206,
  				mass: 205.986657
  			},
  			{
  				nominal: 207,
  				mass: 206.9858
  			},
  			{
  				nominal: 208,
  				mass: 207.9866133
  			},
  			{
  				nominal: 209,
  				mass: 208.9861702
  			},
  			{
  				nominal: 210,
  				mass: 209.9871479
  			},
  			{
  				nominal: 211,
  				mass: 210.9874966
  			},
  			{
  				nominal: 212,
  				mass: 211.9907377
  			},
  			{
  				nominal: 213,
  				mass: 212.992937
  			},
  			{
  				nominal: 214,
  				mass: 213.9963721
  			},
  			{
  				nominal: 215,
  				mass: 214.9986528
  			},
  			{
  				nominal: 216,
  				mass: 216.0024236
  			},
  			{
  				nominal: 217,
  				mass: 217.0047192
  			},
  			{
  				nominal: 218,
  				mass: 218.008695
  			},
  			{
  				nominal: 219,
  				mass: 219.0111618
  			},
  			{
  				nominal: 220,
  				mass: 220.015433
  			},
  			{
  				nominal: 221,
  				mass: 221.018017
  			},
  			{
  				nominal: 222,
  				mass: 222.022494
  			},
  			{
  				nominal: 223,
  				mass: 223.025151
  			},
  			{
  				nominal: 224,
  				mass: 224.029749
  			},
  			{
  				nominal: 225,
  				mass: 225.03263
  			},
  			{
  				nominal: 226,
  				mass: 226.03716
  			},
  			{
  				nominal: 227,
  				mass: 227.04024
  			},
  			{
  				nominal: 228,
  				mass: 228.04475
  			},
  			{
  				nominal: 229,
  				mass: 229.04812
  			}
  		],
  		symbol: "At",
  		mass: null,
  		name: "Astatine"
  	},
  	{
  		number: 86,
  		isotopes: [
  			{
  				nominal: 193,
  				mass: 193.009708
  			},
  			{
  				nominal: 194,
  				mass: 194.006144
  			},
  			{
  				nominal: 195,
  				mass: 195.005422
  			},
  			{
  				nominal: 196,
  				mass: 196.002116
  			},
  			{
  				nominal: 197,
  				mass: 197.001585
  			},
  			{
  				nominal: 198,
  				mass: 197.998679
  			},
  			{
  				nominal: 199,
  				mass: 198.99839
  			},
  			{
  				nominal: 200,
  				mass: 199.99569
  			},
  			{
  				nominal: 201,
  				mass: 200.995628
  			},
  			{
  				nominal: 202,
  				mass: 201.993264
  			},
  			{
  				nominal: 203,
  				mass: 202.993388
  			},
  			{
  				nominal: 204,
  				mass: 203.99143
  			},
  			{
  				nominal: 205,
  				mass: 204.991719
  			},
  			{
  				nominal: 206,
  				mass: 205.990214
  			},
  			{
  				nominal: 207,
  				mass: 206.9907303
  			},
  			{
  				nominal: 208,
  				mass: 207.989635
  			},
  			{
  				nominal: 209,
  				mass: 208.990415
  			},
  			{
  				nominal: 210,
  				mass: 209.9896891
  			},
  			{
  				nominal: 211,
  				mass: 210.9906011
  			},
  			{
  				nominal: 212,
  				mass: 211.9907039
  			},
  			{
  				nominal: 213,
  				mass: 212.9938831
  			},
  			{
  				nominal: 214,
  				mass: 213.995363
  			},
  			{
  				nominal: 215,
  				mass: 214.9987459
  			},
  			{
  				nominal: 216,
  				mass: 216.0002719
  			},
  			{
  				nominal: 217,
  				mass: 217.003928
  			},
  			{
  				nominal: 218,
  				mass: 218.0056016
  			},
  			{
  				nominal: 219,
  				mass: 219.0094804
  			},
  			{
  				nominal: 220,
  				mass: 220.0113941
  			},
  			{
  				nominal: 221,
  				mass: 221.0155371
  			},
  			{
  				nominal: 222,
  				mass: 222.0175782
  			},
  			{
  				nominal: 223,
  				mass: 223.0218893
  			},
  			{
  				nominal: 224,
  				mass: 224.024096
  			},
  			{
  				nominal: 225,
  				mass: 225.028486
  			},
  			{
  				nominal: 226,
  				mass: 226.030861
  			},
  			{
  				nominal: 227,
  				mass: 227.035304
  			},
  			{
  				nominal: 228,
  				mass: 228.037835
  			},
  			{
  				nominal: 229,
  				mass: 229.042257
  			},
  			{
  				nominal: 230,
  				mass: 230.04514
  			},
  			{
  				nominal: 231,
  				mass: 231.04987
  			}
  		],
  		symbol: "Rn",
  		mass: null,
  		name: "Radon"
  	},
  	{
  		number: 87,
  		isotopes: [
  			{
  				nominal: 199,
  				mass: 199.007259
  			},
  			{
  				nominal: 200,
  				mass: 200.006586
  			},
  			{
  				nominal: 201,
  				mass: 201.003867
  			},
  			{
  				nominal: 202,
  				mass: 202.00332
  			},
  			{
  				nominal: 203,
  				mass: 203.0009407
  			},
  			{
  				nominal: 204,
  				mass: 204.000652
  			},
  			{
  				nominal: 205,
  				mass: 204.9985939
  			},
  			{
  				nominal: 206,
  				mass: 205.998666
  			},
  			{
  				nominal: 207,
  				mass: 206.996946
  			},
  			{
  				nominal: 208,
  				mass: 207.997138
  			},
  			{
  				nominal: 209,
  				mass: 208.995955
  			},
  			{
  				nominal: 210,
  				mass: 209.996422
  			},
  			{
  				nominal: 211,
  				mass: 210.995556
  			},
  			{
  				nominal: 212,
  				mass: 211.9962257
  			},
  			{
  				nominal: 213,
  				mass: 212.996186
  			},
  			{
  				nominal: 214,
  				mass: 213.9989713
  			},
  			{
  				nominal: 215,
  				mass: 215.0003418
  			},
  			{
  				nominal: 216,
  				mass: 216.0031899
  			},
  			{
  				nominal: 217,
  				mass: 217.0046323
  			},
  			{
  				nominal: 218,
  				mass: 218.0075787
  			},
  			{
  				nominal: 219,
  				mass: 219.0092524
  			},
  			{
  				nominal: 220,
  				mass: 220.0123277
  			},
  			{
  				nominal: 221,
  				mass: 221.0142552
  			},
  			{
  				nominal: 222,
  				mass: 222.017552
  			},
  			{
  				nominal: 223,
  				mass: 223.019736
  			},
  			{
  				nominal: 224,
  				mass: 224.023398
  			},
  			{
  				nominal: 225,
  				mass: 225.025573
  			},
  			{
  				nominal: 226,
  				mass: 226.029566
  			},
  			{
  				nominal: 227,
  				mass: 227.031869
  			},
  			{
  				nominal: 228,
  				mass: 228.035823
  			},
  			{
  				nominal: 229,
  				mass: 229.038298
  			},
  			{
  				nominal: 230,
  				mass: 230.042416
  			},
  			{
  				nominal: 231,
  				mass: 231.045158
  			},
  			{
  				nominal: 232,
  				mass: 232.04937
  			},
  			{
  				nominal: 233,
  				mass: 233.05264
  			}
  		],
  		symbol: "Fr",
  		mass: null,
  		name: "Francium"
  	},
  	{
  		number: 88,
  		isotopes: [
  			{
  				nominal: 201,
  				mass: 201.01271
  			},
  			{
  				nominal: 202,
  				mass: 202.00976
  			},
  			{
  				nominal: 203,
  				mass: 203.009304
  			},
  			{
  				nominal: 204,
  				mass: 204.006492
  			},
  			{
  				nominal: 205,
  				mass: 205.006268
  			},
  			{
  				nominal: 206,
  				mass: 206.003828
  			},
  			{
  				nominal: 207,
  				mass: 207.003799
  			},
  			{
  				nominal: 208,
  				mass: 208.001841
  			},
  			{
  				nominal: 209,
  				mass: 209.00199
  			},
  			{
  				nominal: 210,
  				mass: 210.000494
  			},
  			{
  				nominal: 211,
  				mass: 211.0008932
  			},
  			{
  				nominal: 212,
  				mass: 211.999787
  			},
  			{
  				nominal: 213,
  				mass: 213.000384
  			},
  			{
  				nominal: 214,
  				mass: 214.0000997
  			},
  			{
  				nominal: 215,
  				mass: 215.0027204
  			},
  			{
  				nominal: 216,
  				mass: 216.0035334
  			},
  			{
  				nominal: 217,
  				mass: 217.0063207
  			},
  			{
  				nominal: 218,
  				mass: 218.007141
  			},
  			{
  				nominal: 219,
  				mass: 219.0100855
  			},
  			{
  				nominal: 220,
  				mass: 220.0110259
  			},
  			{
  				nominal: 221,
  				mass: 221.0139177
  			},
  			{
  				nominal: 222,
  				mass: 222.0153748
  			},
  			{
  				nominal: 223,
  				mass: 223.0185023
  			},
  			{
  				nominal: 224,
  				mass: 224.020212
  			},
  			{
  				nominal: 225,
  				mass: 225.0236119
  			},
  			{
  				nominal: 226,
  				mass: 226.0254103
  			},
  			{
  				nominal: 227,
  				mass: 227.0291783
  			},
  			{
  				nominal: 228,
  				mass: 228.0310707
  			},
  			{
  				nominal: 229,
  				mass: 229.034942
  			},
  			{
  				nominal: 230,
  				mass: 230.037055
  			},
  			{
  				nominal: 231,
  				mass: 231.041027
  			},
  			{
  				nominal: 232,
  				mass: 232.0434753
  			},
  			{
  				nominal: 233,
  				mass: 233.047582
  			},
  			{
  				nominal: 234,
  				mass: 234.050342
  			},
  			{
  				nominal: 235,
  				mass: 235.05497
  			}
  		],
  		symbol: "Ra",
  		mass: null,
  		name: "Radium"
  	},
  	{
  		number: 89,
  		isotopes: [
  			{
  				nominal: 206,
  				mass: 206.014452
  			},
  			{
  				nominal: 207,
  				mass: 207.011966
  			},
  			{
  				nominal: 208,
  				mass: 208.01155
  			},
  			{
  				nominal: 209,
  				mass: 209.009495
  			},
  			{
  				nominal: 210,
  				mass: 210.009436
  			},
  			{
  				nominal: 211,
  				mass: 211.007732
  			},
  			{
  				nominal: 212,
  				mass: 212.007813
  			},
  			{
  				nominal: 213,
  				mass: 213.006609
  			},
  			{
  				nominal: 214,
  				mass: 214.006918
  			},
  			{
  				nominal: 215,
  				mass: 215.006475
  			},
  			{
  				nominal: 216,
  				mass: 216.008743
  			},
  			{
  				nominal: 217,
  				mass: 217.009344
  			},
  			{
  				nominal: 218,
  				mass: 218.011642
  			},
  			{
  				nominal: 219,
  				mass: 219.012421
  			},
  			{
  				nominal: 220,
  				mass: 220.0147549
  			},
  			{
  				nominal: 221,
  				mass: 221.015592
  			},
  			{
  				nominal: 222,
  				mass: 222.0178442
  			},
  			{
  				nominal: 223,
  				mass: 223.0191377
  			},
  			{
  				nominal: 224,
  				mass: 224.0217232
  			},
  			{
  				nominal: 225,
  				mass: 225.02323
  			},
  			{
  				nominal: 226,
  				mass: 226.0260984
  			},
  			{
  				nominal: 227,
  				mass: 227.0277523
  			},
  			{
  				nominal: 228,
  				mass: 228.0310215
  			},
  			{
  				nominal: 229,
  				mass: 229.032956
  			},
  			{
  				nominal: 230,
  				mass: 230.036327
  			},
  			{
  				nominal: 231,
  				mass: 231.038393
  			},
  			{
  				nominal: 232,
  				mass: 232.042034
  			},
  			{
  				nominal: 233,
  				mass: 233.044346
  			},
  			{
  				nominal: 234,
  				mass: 234.048139
  			},
  			{
  				nominal: 235,
  				mass: 235.05084
  			},
  			{
  				nominal: 236,
  				mass: 236.054988
  			},
  			{
  				nominal: 237,
  				mass: 237.05827
  			}
  		],
  		symbol: "Ac",
  		mass: null,
  		name: "Actinium"
  	},
  	{
  		number: 90,
  		isotopes: [
  			{
  				nominal: 208,
  				mass: 208.0179
  			},
  			{
  				nominal: 209,
  				mass: 209.017753
  			},
  			{
  				nominal: 210,
  				mass: 210.015094
  			},
  			{
  				nominal: 211,
  				mass: 211.014929
  			},
  			{
  				nominal: 212,
  				mass: 212.012988
  			},
  			{
  				nominal: 213,
  				mass: 213.013009
  			},
  			{
  				nominal: 214,
  				mass: 214.0115
  			},
  			{
  				nominal: 215,
  				mass: 215.0117248
  			},
  			{
  				nominal: 216,
  				mass: 216.011056
  			},
  			{
  				nominal: 217,
  				mass: 217.013117
  			},
  			{
  				nominal: 218,
  				mass: 218.013276
  			},
  			{
  				nominal: 219,
  				mass: 219.015537
  			},
  			{
  				nominal: 220,
  				mass: 220.015748
  			},
  			{
  				nominal: 221,
  				mass: 221.018184
  			},
  			{
  				nominal: 222,
  				mass: 222.018469
  			},
  			{
  				nominal: 223,
  				mass: 223.0208119
  			},
  			{
  				nominal: 224,
  				mass: 224.021464
  			},
  			{
  				nominal: 225,
  				mass: 225.0239514
  			},
  			{
  				nominal: 226,
  				mass: 226.0249034
  			},
  			{
  				nominal: 227,
  				mass: 227.0277042
  			},
  			{
  				nominal: 228,
  				mass: 228.0287413
  			},
  			{
  				nominal: 229,
  				mass: 229.0317627
  			},
  			{
  				nominal: 230,
  				mass: 230.0331341
  			},
  			{
  				nominal: 231,
  				mass: 231.0363046
  			},
  			{
  				nominal: 232,
  				mass: 232.0380558,
  				abundance: 1
  			},
  			{
  				nominal: 233,
  				mass: 233.0415823
  			},
  			{
  				nominal: 234,
  				mass: 234.0436014
  			},
  			{
  				nominal: 235,
  				mass: 235.047255
  			},
  			{
  				nominal: 236,
  				mass: 236.049657
  			},
  			{
  				nominal: 237,
  				mass: 237.053629
  			},
  			{
  				nominal: 238,
  				mass: 238.0565
  			},
  			{
  				nominal: 239,
  				mass: 239.06077
  			}
  		],
  		symbol: "Th",
  		mass: 232.0380558,
  		name: "Thorium",
  		monoisotopicMass: 232.0380558
  	},
  	{
  		number: 91,
  		isotopes: [
  			{
  				nominal: 212,
  				mass: 212.023203
  			},
  			{
  				nominal: 213,
  				mass: 213.021109
  			},
  			{
  				nominal: 214,
  				mass: 214.020918
  			},
  			{
  				nominal: 215,
  				mass: 215.019183
  			},
  			{
  				nominal: 216,
  				mass: 216.019109
  			},
  			{
  				nominal: 217,
  				mass: 217.018325
  			},
  			{
  				nominal: 218,
  				mass: 218.020059
  			},
  			{
  				nominal: 219,
  				mass: 219.019904
  			},
  			{
  				nominal: 220,
  				mass: 220.021705
  			},
  			{
  				nominal: 221,
  				mass: 221.021875
  			},
  			{
  				nominal: 222,
  				mass: 222.023784
  			},
  			{
  				nominal: 223,
  				mass: 223.023963
  			},
  			{
  				nominal: 224,
  				mass: 224.0256176
  			},
  			{
  				nominal: 225,
  				mass: 225.026131
  			},
  			{
  				nominal: 226,
  				mass: 226.027948
  			},
  			{
  				nominal: 227,
  				mass: 227.0288054
  			},
  			{
  				nominal: 228,
  				mass: 228.0310517
  			},
  			{
  				nominal: 229,
  				mass: 229.0320972
  			},
  			{
  				nominal: 230,
  				mass: 230.034541
  			},
  			{
  				nominal: 231,
  				mass: 231.0358842,
  				abundance: 1
  			},
  			{
  				nominal: 232,
  				mass: 232.0385917
  			},
  			{
  				nominal: 233,
  				mass: 233.0402472
  			},
  			{
  				nominal: 234,
  				mass: 234.0433072
  			},
  			{
  				nominal: 235,
  				mass: 235.045399
  			},
  			{
  				nominal: 236,
  				mass: 236.048668
  			},
  			{
  				nominal: 237,
  				mass: 237.051023
  			},
  			{
  				nominal: 238,
  				mass: 238.054637
  			},
  			{
  				nominal: 239,
  				mass: 239.05726
  			},
  			{
  				nominal: 240,
  				mass: 240.06098
  			},
  			{
  				nominal: 241,
  				mass: 241.06408
  			}
  		],
  		symbol: "Pa",
  		mass: 231.0358842,
  		name: "Protactinium",
  		monoisotopicMass: 231.0358842
  	},
  	{
  		number: 92,
  		isotopes: [
  			{
  				nominal: 217,
  				mass: 217.02466
  			},
  			{
  				nominal: 218,
  				mass: 218.023523
  			},
  			{
  				nominal: 219,
  				mass: 219.024999
  			},
  			{
  				nominal: 220,
  				mass: 220.02462
  			},
  			{
  				nominal: 221,
  				mass: 221.02628
  			},
  			{
  				nominal: 222,
  				mass: 222.026
  			},
  			{
  				nominal: 223,
  				mass: 223.027739
  			},
  			{
  				nominal: 224,
  				mass: 224.027605
  			},
  			{
  				nominal: 225,
  				mass: 225.029391
  			},
  			{
  				nominal: 226,
  				mass: 226.029339
  			},
  			{
  				nominal: 227,
  				mass: 227.031157
  			},
  			{
  				nominal: 228,
  				mass: 228.031371
  			},
  			{
  				nominal: 229,
  				mass: 229.0335063
  			},
  			{
  				nominal: 230,
  				mass: 230.0339401
  			},
  			{
  				nominal: 231,
  				mass: 231.0362939
  			},
  			{
  				nominal: 232,
  				mass: 232.0371563
  			},
  			{
  				nominal: 233,
  				mass: 233.0396355
  			},
  			{
  				nominal: 234,
  				mass: 234.0409523,
  				abundance: 0.000054
  			},
  			{
  				nominal: 235,
  				mass: 235.0439301,
  				abundance: 0.007204
  			},
  			{
  				nominal: 236,
  				mass: 236.0455682
  			},
  			{
  				nominal: 237,
  				mass: 237.0487304
  			},
  			{
  				nominal: 238,
  				mass: 238.0507884,
  				abundance: 0.992742
  			},
  			{
  				nominal: 239,
  				mass: 239.0542935
  			},
  			{
  				nominal: 240,
  				mass: 240.0565934
  			},
  			{
  				nominal: 241,
  				mass: 241.06033
  			},
  			{
  				nominal: 242,
  				mass: 242.06293
  			},
  			{
  				nominal: 243,
  				mass: 243.06699
  			}
  		],
  		symbol: "U",
  		mass: 238.0289104616574,
  		name: "Uranium",
  		monoisotopicMass: 238.0507884
  	},
  	{
  		number: 93,
  		isotopes: [
  			{
  				nominal: 219,
  				mass: 219.03143
  			},
  			{
  				nominal: 220,
  				mass: 220.03254
  			},
  			{
  				nominal: 221,
  				mass: 221.03204
  			},
  			{
  				nominal: 222,
  				mass: 222.0333
  			},
  			{
  				nominal: 223,
  				mass: 223.03285
  			},
  			{
  				nominal: 224,
  				mass: 224.03422
  			},
  			{
  				nominal: 225,
  				mass: 225.033911
  			},
  			{
  				nominal: 226,
  				mass: 226.035188
  			},
  			{
  				nominal: 227,
  				mass: 227.034957
  			},
  			{
  				nominal: 228,
  				mass: 228.036067
  			},
  			{
  				nominal: 229,
  				mass: 229.036264
  			},
  			{
  				nominal: 230,
  				mass: 230.037828
  			},
  			{
  				nominal: 231,
  				mass: 231.038245
  			},
  			{
  				nominal: 232,
  				mass: 232.04011
  			},
  			{
  				nominal: 233,
  				mass: 233.040741
  			},
  			{
  				nominal: 234,
  				mass: 234.0428953
  			},
  			{
  				nominal: 235,
  				mass: 235.0440635
  			},
  			{
  				nominal: 236,
  				mass: 236.04657
  			},
  			{
  				nominal: 237,
  				mass: 237.0481736
  			},
  			{
  				nominal: 238,
  				mass: 238.0509466
  			},
  			{
  				nominal: 239,
  				mass: 239.0529392
  			},
  			{
  				nominal: 240,
  				mass: 240.056165
  			},
  			{
  				nominal: 241,
  				mass: 241.058253
  			},
  			{
  				nominal: 242,
  				mass: 242.06164
  			},
  			{
  				nominal: 243,
  				mass: 243.06428
  			},
  			{
  				nominal: 244,
  				mass: 244.06785
  			},
  			{
  				nominal: 245,
  				mass: 245.0708
  			}
  		],
  		symbol: "Np",
  		mass: null,
  		name: "Neptunium"
  	},
  	{
  		number: 94,
  		isotopes: [
  			{
  				nominal: 228,
  				mass: 228.038732
  			},
  			{
  				nominal: 229,
  				mass: 229.040144
  			},
  			{
  				nominal: 230,
  				mass: 230.03965
  			},
  			{
  				nominal: 231,
  				mass: 231.041102
  			},
  			{
  				nominal: 232,
  				mass: 232.041185
  			},
  			{
  				nominal: 233,
  				mass: 233.042998
  			},
  			{
  				nominal: 234,
  				mass: 234.0433174
  			},
  			{
  				nominal: 235,
  				mass: 235.045286
  			},
  			{
  				nominal: 236,
  				mass: 236.0460581
  			},
  			{
  				nominal: 237,
  				mass: 237.0484098
  			},
  			{
  				nominal: 238,
  				mass: 238.0495601
  			},
  			{
  				nominal: 239,
  				mass: 239.0521636
  			},
  			{
  				nominal: 240,
  				mass: 240.0538138
  			},
  			{
  				nominal: 241,
  				mass: 241.0568517
  			},
  			{
  				nominal: 242,
  				mass: 242.0587428
  			},
  			{
  				nominal: 243,
  				mass: 243.0620036
  			},
  			{
  				nominal: 244,
  				mass: 244.0642053
  			},
  			{
  				nominal: 245,
  				mass: 245.067826
  			},
  			{
  				nominal: 246,
  				mass: 246.070205
  			},
  			{
  				nominal: 247,
  				mass: 247.07419
  			}
  		],
  		symbol: "Pu",
  		mass: null,
  		name: "Plutonium"
  	},
  	{
  		number: 95,
  		isotopes: [
  			{
  				nominal: 230,
  				mass: 230.04609
  			},
  			{
  				nominal: 231,
  				mass: 231.04556
  			},
  			{
  				nominal: 232,
  				mass: 232.04645
  			},
  			{
  				nominal: 233,
  				mass: 233.04644
  			},
  			{
  				nominal: 234,
  				mass: 234.04773
  			},
  			{
  				nominal: 235,
  				mass: 235.047908
  			},
  			{
  				nominal: 236,
  				mass: 236.04943
  			},
  			{
  				nominal: 237,
  				mass: 237.049996
  			},
  			{
  				nominal: 238,
  				mass: 238.051985
  			},
  			{
  				nominal: 239,
  				mass: 239.0530247
  			},
  			{
  				nominal: 240,
  				mass: 240.0553
  			},
  			{
  				nominal: 241,
  				mass: 241.0568293
  			},
  			{
  				nominal: 242,
  				mass: 242.0595494
  			},
  			{
  				nominal: 243,
  				mass: 243.0613813
  			},
  			{
  				nominal: 244,
  				mass: 244.0642851
  			},
  			{
  				nominal: 245,
  				mass: 245.0664548
  			},
  			{
  				nominal: 246,
  				mass: 246.069775
  			},
  			{
  				nominal: 247,
  				mass: 247.07209
  			},
  			{
  				nominal: 248,
  				mass: 248.07575
  			},
  			{
  				nominal: 249,
  				mass: 249.07848
  			}
  		],
  		symbol: "Am",
  		name: "Americium",
  		mass: null
  	},
  	{
  		number: 96,
  		isotopes: [
  			{
  				nominal: 232,
  				mass: 232.04982
  			},
  			{
  				nominal: 233,
  				mass: 233.05077
  			},
  			{
  				nominal: 234,
  				mass: 234.05016
  			},
  			{
  				nominal: 235,
  				mass: 235.05154
  			},
  			{
  				nominal: 236,
  				mass: 236.051374
  			},
  			{
  				nominal: 237,
  				mass: 237.052869
  			},
  			{
  				nominal: 238,
  				mass: 238.053081
  			},
  			{
  				nominal: 239,
  				mass: 239.05491
  			},
  			{
  				nominal: 240,
  				mass: 240.0555297
  			},
  			{
  				nominal: 241,
  				mass: 241.0576532
  			},
  			{
  				nominal: 242,
  				mass: 242.058836
  			},
  			{
  				nominal: 243,
  				mass: 243.0613893
  			},
  			{
  				nominal: 244,
  				mass: 244.0627528
  			},
  			{
  				nominal: 245,
  				mass: 245.0654915
  			},
  			{
  				nominal: 246,
  				mass: 246.0672238
  			},
  			{
  				nominal: 247,
  				mass: 247.0703541
  			},
  			{
  				nominal: 248,
  				mass: 248.0723499
  			},
  			{
  				nominal: 249,
  				mass: 249.0759548
  			},
  			{
  				nominal: 250,
  				mass: 250.078358
  			},
  			{
  				nominal: 251,
  				mass: 251.082286
  			},
  			{
  				nominal: 252,
  				mass: 252.08487
  			}
  		],
  		symbol: "Cm",
  		name: "Curium",
  		mass: null
  	},
  	{
  		number: 97,
  		isotopes: [
  			{
  				nominal: 234,
  				mass: 234.05727
  			},
  			{
  				nominal: 235,
  				mass: 235.05658
  			},
  			{
  				nominal: 236,
  				mass: 236.05748
  			},
  			{
  				nominal: 237,
  				mass: 237.0571
  			},
  			{
  				nominal: 238,
  				mass: 238.0582
  			},
  			{
  				nominal: 239,
  				mass: 239.05824
  			},
  			{
  				nominal: 240,
  				mass: 240.05976
  			},
  			{
  				nominal: 241,
  				mass: 241.06016
  			},
  			{
  				nominal: 242,
  				mass: 242.06198
  			},
  			{
  				nominal: 243,
  				mass: 243.0630078
  			},
  			{
  				nominal: 244,
  				mass: 244.065181
  			},
  			{
  				nominal: 245,
  				mass: 245.0663618
  			},
  			{
  				nominal: 246,
  				mass: 246.068673
  			},
  			{
  				nominal: 247,
  				mass: 247.0703073
  			},
  			{
  				nominal: 248,
  				mass: 248.073088
  			},
  			{
  				nominal: 249,
  				mass: 249.0749877
  			},
  			{
  				nominal: 250,
  				mass: 250.0783167
  			},
  			{
  				nominal: 251,
  				mass: 251.080762
  			},
  			{
  				nominal: 252,
  				mass: 252.08431
  			},
  			{
  				nominal: 253,
  				mass: 253.08688
  			},
  			{
  				nominal: 254,
  				mass: 254.0906
  			}
  		],
  		symbol: "Bk",
  		name: "Berkelium",
  		mass: null
  	},
  	{
  		number: 98,
  		isotopes: [
  			{
  				nominal: 237,
  				mass: 237.062198
  			},
  			{
  				nominal: 238,
  				mass: 238.06149
  			},
  			{
  				nominal: 239,
  				mass: 239.06253
  			},
  			{
  				nominal: 240,
  				mass: 240.062256
  			},
  			{
  				nominal: 241,
  				mass: 241.06369
  			},
  			{
  				nominal: 242,
  				mass: 242.063754
  			},
  			{
  				nominal: 243,
  				mass: 243.06548
  			},
  			{
  				nominal: 244,
  				mass: 244.0660008
  			},
  			{
  				nominal: 245,
  				mass: 245.0680487
  			},
  			{
  				nominal: 246,
  				mass: 246.0688055
  			},
  			{
  				nominal: 247,
  				mass: 247.070965
  			},
  			{
  				nominal: 248,
  				mass: 248.0721851
  			},
  			{
  				nominal: 249,
  				mass: 249.0748539
  			},
  			{
  				nominal: 250,
  				mass: 250.0764062
  			},
  			{
  				nominal: 251,
  				mass: 251.0795886
  			},
  			{
  				nominal: 252,
  				mass: 252.0816272
  			},
  			{
  				nominal: 253,
  				mass: 253.0851345
  			},
  			{
  				nominal: 254,
  				mass: 254.087324
  			},
  			{
  				nominal: 255,
  				mass: 255.09105
  			},
  			{
  				nominal: 256,
  				mass: 256.09344
  			}
  		],
  		symbol: "Cf",
  		name: "Californium",
  		mass: null
  	},
  	{
  		number: 99,
  		isotopes: [
  			{
  				nominal: 239,
  				mass: 239.06823
  			},
  			{
  				nominal: 240,
  				mass: 240.06892
  			},
  			{
  				nominal: 241,
  				mass: 241.06856
  			},
  			{
  				nominal: 242,
  				mass: 242.06957
  			},
  			{
  				nominal: 243,
  				mass: 243.06951
  			},
  			{
  				nominal: 244,
  				mass: 244.07088
  			},
  			{
  				nominal: 245,
  				mass: 245.07125
  			},
  			{
  				nominal: 246,
  				mass: 246.0729
  			},
  			{
  				nominal: 247,
  				mass: 247.073622
  			},
  			{
  				nominal: 248,
  				mass: 248.075471
  			},
  			{
  				nominal: 249,
  				mass: 249.076411
  			},
  			{
  				nominal: 250,
  				mass: 250.07861
  			},
  			{
  				nominal: 251,
  				mass: 251.0799936
  			},
  			{
  				nominal: 252,
  				mass: 252.08298
  			},
  			{
  				nominal: 253,
  				mass: 253.0848257
  			},
  			{
  				nominal: 254,
  				mass: 254.0880222
  			},
  			{
  				nominal: 255,
  				mass: 255.090275
  			},
  			{
  				nominal: 256,
  				mass: 256.0936
  			},
  			{
  				nominal: 257,
  				mass: 257.09598
  			},
  			{
  				nominal: 258,
  				mass: 258.09952
  			}
  		],
  		symbol: "Es",
  		name: "Einsteinium",
  		mass: null
  	},
  	{
  		number: 100,
  		isotopes: [
  			{
  				nominal: 241,
  				mass: 241.07421
  			},
  			{
  				nominal: 242,
  				mass: 242.07343
  			},
  			{
  				nominal: 243,
  				mass: 243.07446
  			},
  			{
  				nominal: 244,
  				mass: 244.07404
  			},
  			{
  				nominal: 245,
  				mass: 245.07535
  			},
  			{
  				nominal: 246,
  				mass: 246.07535
  			},
  			{
  				nominal: 247,
  				mass: 247.07694
  			},
  			{
  				nominal: 248,
  				mass: 248.0771865
  			},
  			{
  				nominal: 249,
  				mass: 249.0789275
  			},
  			{
  				nominal: 250,
  				mass: 250.079521
  			},
  			{
  				nominal: 251,
  				mass: 251.08154
  			},
  			{
  				nominal: 252,
  				mass: 252.0824671
  			},
  			{
  				nominal: 253,
  				mass: 253.0851846
  			},
  			{
  				nominal: 254,
  				mass: 254.0868544
  			},
  			{
  				nominal: 255,
  				mass: 255.089964
  			},
  			{
  				nominal: 256,
  				mass: 256.0917745
  			},
  			{
  				nominal: 257,
  				mass: 257.0951061
  			},
  			{
  				nominal: 258,
  				mass: 258.09708
  			},
  			{
  				nominal: 259,
  				mass: 259.1006
  			},
  			{
  				nominal: 260,
  				mass: 260.10281
  			}
  		],
  		symbol: "Fm",
  		name: "Fermium",
  		mass: null
  	},
  	{
  		number: 101,
  		isotopes: [
  			{
  				nominal: 245,
  				mass: 245.08081
  			},
  			{
  				nominal: 246,
  				mass: 246.08171
  			},
  			{
  				nominal: 247,
  				mass: 247.08152
  			},
  			{
  				nominal: 248,
  				mass: 248.08282
  			},
  			{
  				nominal: 249,
  				mass: 249.08291
  			},
  			{
  				nominal: 250,
  				mass: 250.08441
  			},
  			{
  				nominal: 251,
  				mass: 251.084774
  			},
  			{
  				nominal: 252,
  				mass: 252.08643
  			},
  			{
  				nominal: 253,
  				mass: 253.087144
  			},
  			{
  				nominal: 254,
  				mass: 254.08959
  			},
  			{
  				nominal: 255,
  				mass: 255.0910841
  			},
  			{
  				nominal: 256,
  				mass: 256.09389
  			},
  			{
  				nominal: 257,
  				mass: 257.0955424
  			},
  			{
  				nominal: 258,
  				mass: 258.0984315
  			},
  			{
  				nominal: 259,
  				mass: 259.10051
  			},
  			{
  				nominal: 260,
  				mass: 260.10365
  			},
  			{
  				nominal: 261,
  				mass: 261.10583
  			},
  			{
  				nominal: 262,
  				mass: 262.1091
  			}
  		],
  		symbol: "Md",
  		name: "Mendelevium",
  		mass: null
  	},
  	{
  		number: 102,
  		isotopes: [
  			{
  				nominal: 248,
  				mass: 248.08655
  			},
  			{
  				nominal: 249,
  				mass: 249.0878
  			},
  			{
  				nominal: 250,
  				mass: 250.08756
  			},
  			{
  				nominal: 251,
  				mass: 251.08894
  			},
  			{
  				nominal: 252,
  				mass: 252.088967
  			},
  			{
  				nominal: 253,
  				mass: 253.0905641
  			},
  			{
  				nominal: 254,
  				mass: 254.090956
  			},
  			{
  				nominal: 255,
  				mass: 255.093191
  			},
  			{
  				nominal: 256,
  				mass: 256.0942829
  			},
  			{
  				nominal: 257,
  				mass: 257.0968878
  			},
  			{
  				nominal: 258,
  				mass: 258.09821
  			},
  			{
  				nominal: 259,
  				mass: 259.10103
  			},
  			{
  				nominal: 260,
  				mass: 260.10264
  			},
  			{
  				nominal: 261,
  				mass: 261.1057
  			},
  			{
  				nominal: 262,
  				mass: 262.10746
  			},
  			{
  				nominal: 263,
  				mass: 263.11071
  			},
  			{
  				nominal: 264,
  				mass: 264.11273
  			}
  		],
  		symbol: "No",
  		name: "Nobelium",
  		mass: null
  	},
  	{
  		number: 103,
  		isotopes: [
  			{
  				nominal: 251,
  				mass: 251.09418
  			},
  			{
  				nominal: 252,
  				mass: 252.09526
  			},
  			{
  				nominal: 253,
  				mass: 253.09509
  			},
  			{
  				nominal: 254,
  				mass: 254.09648
  			},
  			{
  				nominal: 255,
  				mass: 255.096562
  			},
  			{
  				nominal: 256,
  				mass: 256.098494
  			},
  			{
  				nominal: 257,
  				mass: 257.099418
  			},
  			{
  				nominal: 258,
  				mass: 258.10176
  			},
  			{
  				nominal: 259,
  				mass: 259.102902
  			},
  			{
  				nominal: 260,
  				mass: 260.1055
  			},
  			{
  				nominal: 261,
  				mass: 261.10688
  			},
  			{
  				nominal: 262,
  				mass: 262.10961
  			},
  			{
  				nominal: 263,
  				mass: 263.11136
  			},
  			{
  				nominal: 264,
  				mass: 264.1142
  			},
  			{
  				nominal: 265,
  				mass: 265.11619
  			},
  			{
  				nominal: 266,
  				mass: 266.11983
  			}
  		],
  		symbol: "Lr",
  		name: "Lawrencium",
  		mass: null
  	},
  	{
  		number: 104,
  		isotopes: [
  			{
  				nominal: 253,
  				mass: 253.10044
  			},
  			{
  				nominal: 254,
  				mass: 254.10005
  			},
  			{
  				nominal: 255,
  				mass: 255.10127
  			},
  			{
  				nominal: 256,
  				mass: 256.101152
  			},
  			{
  				nominal: 257,
  				mass: 257.102918
  			},
  			{
  				nominal: 258,
  				mass: 258.103428
  			},
  			{
  				nominal: 259,
  				mass: 259.105596
  			},
  			{
  				nominal: 260,
  				mass: 260.10644
  			},
  			{
  				nominal: 261,
  				mass: 261.108773
  			},
  			{
  				nominal: 262,
  				mass: 262.10992
  			},
  			{
  				nominal: 263,
  				mass: 263.11249
  			},
  			{
  				nominal: 264,
  				mass: 264.11388
  			},
  			{
  				nominal: 265,
  				mass: 265.11668
  			},
  			{
  				nominal: 266,
  				mass: 266.11817
  			},
  			{
  				nominal: 267,
  				mass: 267.12179
  			},
  			{
  				nominal: 268,
  				mass: 268.12397
  			}
  		],
  		symbol: "Rf",
  		name: "Rutherfordium",
  		mass: null
  	},
  	{
  		number: 105,
  		isotopes: [
  			{
  				nominal: 255,
  				mass: 255.10707
  			},
  			{
  				nominal: 256,
  				mass: 256.10789
  			},
  			{
  				nominal: 257,
  				mass: 257.10758
  			},
  			{
  				nominal: 258,
  				mass: 258.10928
  			},
  			{
  				nominal: 259,
  				mass: 259.109492
  			},
  			{
  				nominal: 260,
  				mass: 260.1113
  			},
  			{
  				nominal: 261,
  				mass: 261.11192
  			},
  			{
  				nominal: 262,
  				mass: 262.11407
  			},
  			{
  				nominal: 263,
  				mass: 263.11499
  			},
  			{
  				nominal: 264,
  				mass: 264.11741
  			},
  			{
  				nominal: 265,
  				mass: 265.11861
  			},
  			{
  				nominal: 266,
  				mass: 266.12103
  			},
  			{
  				nominal: 267,
  				mass: 267.12247
  			},
  			{
  				nominal: 268,
  				mass: 268.12567
  			},
  			{
  				nominal: 269,
  				mass: 269.12791
  			},
  			{
  				nominal: 270,
  				mass: 270.13136
  			}
  		],
  		symbol: "Db",
  		name: "Dubnium",
  		mass: null
  	},
  	{
  		number: 106,
  		isotopes: [
  			{
  				nominal: 258,
  				mass: 258.11298
  			},
  			{
  				nominal: 259,
  				mass: 259.1144
  			},
  			{
  				nominal: 260,
  				mass: 260.114384
  			},
  			{
  				nominal: 261,
  				mass: 261.115949
  			},
  			{
  				nominal: 262,
  				mass: 262.116337
  			},
  			{
  				nominal: 263,
  				mass: 263.11829
  			},
  			{
  				nominal: 264,
  				mass: 264.11893
  			},
  			{
  				nominal: 265,
  				mass: 265.12109
  			},
  			{
  				nominal: 266,
  				mass: 266.12198
  			},
  			{
  				nominal: 267,
  				mass: 267.12436
  			},
  			{
  				nominal: 268,
  				mass: 268.12539
  			},
  			{
  				nominal: 269,
  				mass: 269.12863
  			},
  			{
  				nominal: 270,
  				mass: 270.13043
  			},
  			{
  				nominal: 271,
  				mass: 271.13393
  			},
  			{
  				nominal: 272,
  				mass: 272.13589
  			},
  			{
  				nominal: 273,
  				mass: 273.13958
  			}
  		],
  		symbol: "Sg",
  		name: "Seaborgium",
  		mass: null
  	},
  	{
  		number: 107,
  		isotopes: [
  			{
  				nominal: 260,
  				mass: 260.12166
  			},
  			{
  				nominal: 261,
  				mass: 261.12145
  			},
  			{
  				nominal: 262,
  				mass: 262.12297
  			},
  			{
  				nominal: 263,
  				mass: 263.12292
  			},
  			{
  				nominal: 264,
  				mass: 264.12459
  			},
  			{
  				nominal: 265,
  				mass: 265.12491
  			},
  			{
  				nominal: 266,
  				mass: 266.12679
  			},
  			{
  				nominal: 267,
  				mass: 267.1275
  			},
  			{
  				nominal: 268,
  				mass: 268.12969
  			},
  			{
  				nominal: 269,
  				mass: 269.13042
  			},
  			{
  				nominal: 270,
  				mass: 270.13336
  			},
  			{
  				nominal: 271,
  				mass: 271.13526
  			},
  			{
  				nominal: 272,
  				mass: 272.13826
  			},
  			{
  				nominal: 273,
  				mass: 273.14024
  			},
  			{
  				nominal: 274,
  				mass: 274.14355
  			},
  			{
  				nominal: 275,
  				mass: 275.14567
  			}
  		],
  		symbol: "Bh",
  		name: "Bohrium",
  		mass: null
  	},
  	{
  		number: 108,
  		isotopes: [
  			{
  				nominal: 263,
  				mass: 263.12852
  			},
  			{
  				nominal: 264,
  				mass: 264.128357
  			},
  			{
  				nominal: 265,
  				mass: 265.129793
  			},
  			{
  				nominal: 266,
  				mass: 266.130046
  			},
  			{
  				nominal: 267,
  				mass: 267.13167
  			},
  			{
  				nominal: 268,
  				mass: 268.13186
  			},
  			{
  				nominal: 269,
  				mass: 269.13375
  			},
  			{
  				nominal: 270,
  				mass: 270.13429
  			},
  			{
  				nominal: 271,
  				mass: 271.13717
  			},
  			{
  				nominal: 272,
  				mass: 272.1385
  			},
  			{
  				nominal: 273,
  				mass: 273.14168
  			},
  			{
  				nominal: 274,
  				mass: 274.1433
  			},
  			{
  				nominal: 275,
  				mass: 275.14667
  			},
  			{
  				nominal: 276,
  				mass: 276.14846
  			},
  			{
  				nominal: 277,
  				mass: 277.1519
  			}
  		],
  		symbol: "Hs",
  		name: "Hassium",
  		mass: null
  	},
  	{
  		number: 109,
  		isotopes: [
  			{
  				nominal: 265,
  				mass: 265.136
  			},
  			{
  				nominal: 266,
  				mass: 266.13737
  			},
  			{
  				nominal: 267,
  				mass: 267.13719
  			},
  			{
  				nominal: 268,
  				mass: 268.13865
  			},
  			{
  				nominal: 269,
  				mass: 269.13882
  			},
  			{
  				nominal: 270,
  				mass: 270.14033
  			},
  			{
  				nominal: 271,
  				mass: 271.14074
  			},
  			{
  				nominal: 272,
  				mass: 272.14341
  			},
  			{
  				nominal: 273,
  				mass: 273.1444
  			},
  			{
  				nominal: 274,
  				mass: 274.14724
  			},
  			{
  				nominal: 275,
  				mass: 275.14882
  			},
  			{
  				nominal: 276,
  				mass: 276.15159
  			},
  			{
  				nominal: 277,
  				mass: 277.15327
  			},
  			{
  				nominal: 278,
  				mass: 278.15631
  			},
  			{
  				nominal: 279,
  				mass: 279.15808
  			}
  		],
  		symbol: "Mt",
  		name: "Meitnerium",
  		mass: null
  	},
  	{
  		number: 110,
  		isotopes: [
  			{
  				nominal: 267,
  				mass: 267.14377
  			},
  			{
  				nominal: 268,
  				mass: 268.14348
  			},
  			{
  				nominal: 269,
  				mass: 269.144752
  			},
  			{
  				nominal: 270,
  				mass: 270.144584
  			},
  			{
  				nominal: 271,
  				mass: 271.14595
  			},
  			{
  				nominal: 272,
  				mass: 272.14602
  			},
  			{
  				nominal: 273,
  				mass: 273.14856
  			},
  			{
  				nominal: 274,
  				mass: 274.14941
  			},
  			{
  				nominal: 275,
  				mass: 275.15203
  			},
  			{
  				nominal: 276,
  				mass: 276.15303
  			},
  			{
  				nominal: 277,
  				mass: 277.15591
  			},
  			{
  				nominal: 278,
  				mass: 278.15704
  			},
  			{
  				nominal: 279,
  				mass: 279.1601
  			},
  			{
  				nominal: 280,
  				mass: 280.16131
  			},
  			{
  				nominal: 281,
  				mass: 281.16451
  			}
  		],
  		symbol: "Ds",
  		name: "Darmstadtium",
  		mass: null
  	},
  	{
  		number: 111,
  		isotopes: [
  			{
  				nominal: 272,
  				mass: 272.15327
  			},
  			{
  				nominal: 273,
  				mass: 273.15313
  			},
  			{
  				nominal: 274,
  				mass: 274.15525
  			},
  			{
  				nominal: 275,
  				mass: 275.15594
  			},
  			{
  				nominal: 276,
  				mass: 276.15833
  			},
  			{
  				nominal: 277,
  				mass: 277.15907
  			},
  			{
  				nominal: 278,
  				mass: 278.16149
  			},
  			{
  				nominal: 279,
  				mass: 279.16272
  			},
  			{
  				nominal: 280,
  				mass: 280.16514
  			},
  			{
  				nominal: 281,
  				mass: 281.16636
  			},
  			{
  				nominal: 282,
  				mass: 282.16912
  			},
  			{
  				nominal: 283,
  				mass: 283.17054
  			}
  		],
  		symbol: "Rg",
  		name: "Roentgenium",
  		mass: null
  	},
  	{
  		number: 112,
  		isotopes: [
  			{
  				nominal: 276,
  				mass: 276.16141
  			},
  			{
  				nominal: 277,
  				mass: 277.16364
  			},
  			{
  				nominal: 278,
  				mass: 278.16416
  			},
  			{
  				nominal: 279,
  				mass: 279.16654
  			},
  			{
  				nominal: 280,
  				mass: 280.16715
  			},
  			{
  				nominal: 281,
  				mass: 281.16975
  			},
  			{
  				nominal: 282,
  				mass: 282.1705
  			},
  			{
  				nominal: 283,
  				mass: 283.17327
  			},
  			{
  				nominal: 284,
  				mass: 284.17416
  			},
  			{
  				nominal: 285,
  				mass: 285.17712
  			}
  		],
  		symbol: "Cn",
  		name: "Copernicium",
  		mass: null
  	},
  	{
  		number: 113,
  		isotopes: [
  			{
  				nominal: 278,
  				mass: 278.17058
  			},
  			{
  				nominal: 279,
  				mass: 279.17095
  			},
  			{
  				nominal: 280,
  				mass: 280.17293
  			},
  			{
  				nominal: 281,
  				mass: 281.17348
  			},
  			{
  				nominal: 282,
  				mass: 282.17567
  			},
  			{
  				nominal: 283,
  				mass: 283.17657
  			},
  			{
  				nominal: 284,
  				mass: 284.17873
  			},
  			{
  				nominal: 285,
  				mass: 285.17973
  			},
  			{
  				nominal: 286,
  				mass: 286.18221
  			},
  			{
  				nominal: 287,
  				mass: 287.18339
  			}
  		],
  		symbol: "Nh",
  		name: "Nihonium",
  		mass: null
  	},
  	{
  		number: 114,
  		isotopes: [
  			{
  				nominal: 285,
  				mass: 285.18364
  			},
  			{
  				nominal: 286,
  				mass: 286.18423
  			},
  			{
  				nominal: 287,
  				mass: 287.18678
  			},
  			{
  				nominal: 288,
  				mass: 288.18757
  			},
  			{
  				nominal: 289,
  				mass: 289.19042
  			}
  		],
  		symbol: "Fl",
  		name: "Flerovium",
  		mass: null
  	},
  	{
  		number: 115,
  		isotopes: [
  			{
  				nominal: 287,
  				mass: 287.1907
  			},
  			{
  				nominal: 288,
  				mass: 288.19274
  			},
  			{
  				nominal: 289,
  				mass: 289.19363
  			},
  			{
  				nominal: 290,
  				mass: 290.19598
  			},
  			{
  				nominal: 291,
  				mass: 291.19707
  			}
  		],
  		symbol: "Mc",
  		name: "Moscovium",
  		mass: null
  	},
  	{
  		number: 116,
  		isotopes: [
  			{
  				nominal: 289,
  				mass: 289.19816
  			},
  			{
  				nominal: 290,
  				mass: 290.19864
  			},
  			{
  				nominal: 291,
  				mass: 291.20108
  			},
  			{
  				nominal: 292,
  				mass: 292.20174
  			},
  			{
  				nominal: 293,
  				mass: 293.20449
  			}
  		],
  		symbol: "Lv",
  		name: "Livermorium",
  		mass: null
  	},
  	{
  		number: 117,
  		isotopes: [
  			{
  				nominal: 291,
  				mass: 291.20553
  			},
  			{
  				nominal: 292,
  				mass: 292.20746
  			},
  			{
  				nominal: 293,
  				mass: 293.20824
  			},
  			{
  				nominal: 294,
  				mass: 294.21046
  			}
  		],
  		symbol: "Ts",
  		name: "Teennessine",
  		mass: null
  	},
  	{
  		number: 118,
  		isotopes: [
  			{
  				nominal: 293,
  				mass: 293.21356
  			},
  			{
  				nominal: 294,
  				mass: 294.21392
  			},
  			{
  				nominal: 295,
  				mass: 295.21624
  			}
  		],
  		symbol: "Og",
  		name: "Oganesson",
  		mass: null
  	}
  ];

  const elements$4 = JSON.parse(JSON.stringify(elements$5));
  elements$4.forEach(element => {
    element.isotopes = element.isotopes.filter(i => i.abundance > 0);
  });
  var elementsAndStableIsotopes = elements$4;

  let elementsAndStableIsotopesObject = {};
  elementsAndStableIsotopes.forEach(element => {
    elementsAndStableIsotopesObject[element.symbol] = element;
  });
  var elementsAndStableIsotopesObject_1 = elementsAndStableIsotopesObject;

  const elements$3 = Object.keys(elementsAndStableIsotopesObject_1).sort((a, b) => b.length - a.length);
  /**
   * Ensure that the mf has been entered with capital letters and not only lowercase
   * If there is only lowercase we try to capitalize the mf
   * @param {string} mf
   */

  function capitalize$1(mf) {
    for (let i = 0; i < mf.length; i++) {
      if (mf.charCodeAt(i) > 64 && mf.charCodeAt(i) < 91) {
        return mf;
      }
    }

    let parts = mf.replace(/([a-z]*)([^a-z]*)/g, '$1 $2 ').split(/ +/);

    for (let i = 0; i < parts.length; i++) {
      if (parts[i].match(/^[a-z]$/)) {
        parts[i] = parts[i].toUpperCase();
      } else if (parts[i].match(/^[a-z]+$/)) {
        let newPart = '';

        for (let j = 0; j < parts[i].length; j++) {
          let two = parts[i].substr(j, 2);
          let one = parts[i].charAt(j).toUpperCase();

          if (['c', 'h', 'o', 'n'].includes(two.charAt(0)) && ['h', 'o', 'n'].includes(two.charAt(1))) {
            newPart += two.toUpperCase();
            j++;
          } else {
            two = two.charAt(0).toUpperCase() + two.charAt(1);

            if (elements$3.includes(two)) {
              newPart += two;
              j++;
            } else {
              if (elements$3.includes(one)) {
                newPart += one;
              } else {
                return mf;
              }
            }
          }
        }

        parts[i] = newPart;
      }
    }

    return parts.join('');
  }

  var ensureCase = capitalize$1;

  const data$2 = elements$5.map(element => ({
    number: element.number,
    symbol: element.symbol,
    mass: element.mass,
    name: element.name,
    monoisotopicMass: element.monoisotopicMass
  }));
  var elements_1 = data$2;

  var elementsAndIsotopes = elements$5;

  let elementsAndIsotopesObject$1 = {};
  elements$5.forEach(element => {
    elementsAndIsotopesObject$1[element.symbol] = element;
  });
  var elementsAndIsotopesObject_1 = elementsAndIsotopesObject$1;

  let elementsObject$1 = {};
  elements_1.forEach(element => {
    elementsObject$1[element.symbol] = element;
  });
  var elementsObject_1 = elementsObject$1;

  const {
    ELECTRON_MASS: ELECTRON_MASS$6
  } = constants;
  var src$j = {
    elements: elements_1,
    elementsObject: elementsObject_1,
    elementsAndIsotopes,
    elementsAndIsotopesObject: elementsAndIsotopesObject_1,
    elementsAndStableIsotopes,
    elementsAndStableIsotopesObject: elementsAndStableIsotopesObject_1,
    ELECTRON_MASS: ELECTRON_MASS$6
  };

  var groups$2 = [{
    "symbol": "Abu",
    "name": "2-Aminobutyric acid diradical",
    "mf": "C4H7NO",
    "ocl": {
      "value": "dazHPBPOEgEInVZjcH@",
      "coordinates": "!Bb@I~@Ha}_c~H@m]}bGt"
    },
    "mass": 85.10463700109551,
    "monoisotopicMass": 85.05276384961,
    "unsaturation": 2,
    "elements": [{
      "symbol": "C",
      "number": 4
    }, {
      "symbol": "H",
      "number": 7
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Acet",
    "name": "Acetyl",
    "mf": "C2H3O",
    "ocl": {
      "value": "gCaHDEeIi`@",
      "coordinates": "!BbOq~@Ha}"
    },
    "mass": 43.04469897995611,
    "monoisotopicMass": 43.01838971626,
    "unsaturation": 1,
    "elements": [{
      "symbol": "C",
      "number": 2
    }, {
      "symbol": "H",
      "number": 3
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Acm",
    "name": "Acetamidomethyl",
    "mf": "C3H6NO",
    "ocl": {
      "value": "gGYHDPliJuS@@",
      "coordinates": "!BbOrH_Xc|_`BH_P"
    },
    "mass": 72.08596035030448,
    "monoisotopicMass": 72.04493881738,
    "unsaturation": 1,
    "elements": [{
      "symbol": "C",
      "number": 3
    }, {
      "symbol": "H",
      "number": 6
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Adao",
    "name": "Adamantyloxy",
    "mf": "C10H15O",
    "ocl": {
      "value": "dc\\H`HAYRVeV^dUGZjjjj@@",
      "coordinates": "!B]BOXN`EP}CdB\\tbZ@Ijh~hRELdOBBp"
    },
    "mass": 151.2258752025074,
    "monoisotopicMass": 151.11229010302,
    "unsaturation": 5,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 15
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Aib",
    "name": "alpha-Aminoisobutyric acid diradical",
    "mf": "C4H7NO",
    "ocl": {
      "value": "dazHPBPOGgEInfZj@@",
      "coordinates": "!Bb@I~@Ha}b@K|uwwWbGt"
    },
    "mass": 85.10463700109551,
    "monoisotopicMass": 85.05276384961,
    "unsaturation": 2,
    "elements": [{
      "symbol": "C",
      "number": 4
    }, {
      "symbol": "H",
      "number": 7
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Ala",
    "name": "Alanine diradical",
    "mf": "C3H5NO",
    "kind": "aa",
    "oneLetter": "A",
    "alternativeOneLetter": "α",
    "ocl": {
      "value": "gNyDBaxmqR[fZjZ@",
      "coordinates": "!BbOr~@H`}bOr~Wxb}"
    },
    "mass": 71.07801959624871,
    "monoisotopicMass": 71.03711378515,
    "unsaturation": 2,
    "elements": [{
      "symbol": "C",
      "number": 3
    }, {
      "symbol": "H",
      "number": 5
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Arg",
    "name": "Arginine diradical",
    "mf": "C6H12N4O",
    "kind": "aa",
    "oneLetter": "R",
    "alternativeOneLetter": "ρ",
    "ocl": {
      "value": "dkLhPBgSPOEgEInWUijjihr@@",
      "coordinates": "!Bb@I~@Ha}_c~H@m]}bGvHHa}b@I~@Ha}"
    },
    "mass": 156.18592219918227,
    "monoisotopicMass": 156.10111102405,
    "unsaturation": 4,
    "elements": [{
      "symbol": "C",
      "number": 6
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 4
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Argp",
    "name": "Arginine triradical",
    "mf": "C6H11N4O",
    "ocl": {
      "value": "dglhpHpil@gWDEI[UYZfjji`T@",
      "coordinates": "!BbGvHGx@bGvH@ha}bOrH_Wxb@KW_Wx@bGt"
    },
    "mass": 155.1779814451265,
    "monoisotopicMass": 155.09328599182,
    "unsaturation": 5,
    "elements": [{
      "symbol": "C",
      "number": 6
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "N",
      "number": 4
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Asn",
    "name": "Asparagine diradical",
    "mf": "C4H6N2O2",
    "kind": "aa",
    "oneLetter": "N",
    "alternativeOneLetter": "η",
    "ocl": {
      "value": "deeDPBeACqYqR[ezZjZL`@",
      "coordinates": "!BbGu~Ox`B_`BH_X`Bb@I~@Ha}"
    },
    "mass": 114.10280438280381,
    "monoisotopicMass": 114.04292744137999,
    "unsaturation": 4,
    "elements": [{
      "symbol": "C",
      "number": 4
    }, {
      "symbol": "H",
      "number": 6
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Asnp",
    "name": "Asparagine triradical",
    "mf": "C4H5N2O2",
    "ocl": {
      "value": "dmUDpH[E@IEqgqRVvVijjXi@@",
      "coordinates": "!Bb@JH_Wxb@JH_Wxb@KW_Wx@bGt"
    },
    "mass": 113.09486362874803,
    "monoisotopicMass": 113.03510240915,
    "unsaturation": 5,
    "elements": [{
      "symbol": "C",
      "number": 4
    }, {
      "symbol": "H",
      "number": 5
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Asp",
    "name": "Aspartic acid diradical",
    "mf": "C4H5NO3",
    "kind": "aa",
    "oneLetter": "D",
    "alternativeOneLetter": "δ",
    "ocl": {
      "value": "defLPBPYCqYqR[ezZjZL`@",
      "coordinates": "!BbGu~Ox`B_`BH_X`Bb@I~@Ha}"
    },
    "mass": 115.08756534162052,
    "monoisotopicMass": 115.02694302429,
    "unsaturation": 4,
    "elements": [{
      "symbol": "C",
      "number": 4
    }, {
      "symbol": "H",
      "number": 5
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 3
    }]
  }, {
    "symbol": "Aspp",
    "name": "Aspartic acid triradical",
    "mf": "C4H4NO3",
    "ocl": {
      "value": "dmVLpFcE@IEqgqRVvVijjXi@@",
      "coordinates": "!Bb@JH_Wxb@JH_Wxb@KW_Wx@bGt"
    },
    "mass": 114.07962458756472,
    "monoisotopicMass": 114.01911799206,
    "unsaturation": 5,
    "elements": [{
      "symbol": "C",
      "number": 4
    }, {
      "symbol": "H",
      "number": 4
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 3
    }]
  }, {
    "symbol": "Asu",
    "name": "alpha-Aminosuberic acid diradical",
    "mf": "C8H13NO3",
    "ocl": {
      "value": "dgnLPBP{CqYqR[euVfjjihr@@",
      "coordinates": "!BbGu~Ox`B_`BH_Xc|bOrH_X`BbGvHGx@bGt"
    },
    "mass": 171.19403496100773,
    "monoisotopicMass": 171.08954328213002,
    "unsaturation": 4,
    "elements": [{
      "symbol": "C",
      "number": 8
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 3
    }]
  }, {
    "symbol": "Asup",
    "name": "alpha-Aminosuberic acid triradical",
    "mf": "C8H12NO3",
    "ocl": {
      "value": "do^LpEcG@IMqoqRVuUejZjjibT@",
      "coordinates": "!BbOrH_Wxb@JH_Xc|bGvHHa}_c~H@m]}_`BH_P"
    },
    "mass": 170.18609420695194,
    "monoisotopicMass": 170.0817182499,
    "unsaturation": 5,
    "elements": [{
      "symbol": "C",
      "number": 8
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 3
    }]
  }, {
    "symbol": "Boc",
    "name": "t-Butoxycarbonyl",
    "mf": "C5H9O2",
    "ocl": {
      "value": "daxD`DpEeImjZj@@",
      "coordinates": "!B|Ou~_A||Ow}mC}_O@"
    },
    "mass": 101.12395611881479,
    "monoisotopicMass": 101.06025452921,
    "unsaturation": 1,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 9
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Bom",
    "name": "Benzyloxymethyl",
    "mf": "C8H9O",
    "ocl": {
      "value": "deTH`DAYRUYTYj`@@@",
      "coordinates": "!B|Gsp__A||Owp_Gy|Gwp_Wy"
    },
    "mass": 121.15675888470227,
    "monoisotopicMass": 121.06533990964,
    "unsaturation": 7,
    "elements": [{
      "symbol": "C",
      "number": 8
    }, {
      "symbol": "H",
      "number": 9
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Brz",
    "name": "2-Bromobenzyloxycarbonyl",
    "mf": "C8H6BrO2",
    "ocl": {
      "value": "dcLDPDpEd\\QImYgWYjB@@@",
      "coordinates": "!Bb@I~@Hb}b@JH_X`B_c}~@Hb}bGu~Op"
    },
    "mass": 214.03586932736317,
    "monoisotopicMass": 212.95511703252,
    "unsaturation": 9,
    "elements": [{
      "symbol": "C",
      "number": 8
    }, {
      "symbol": "H",
      "number": 6
    }, {
      "symbol": "Br",
      "number": 1
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Bu",
    "name": "Butyl",
    "mf": "C4H9",
    "ocl": {
      "value": "gJPH@liJuP@",
      "coordinates": "!B@Fp@XpAl@FL"
    },
    "mass": 57.114410373442986,
    "monoisotopicMass": 57.07042529007,
    "unsaturation": -1,
    "elements": [{
      "symbol": "C",
      "number": 4
    }, {
      "symbol": "H",
      "number": 9
    }]
  }, {
    "symbol": "Bum",
    "name": "t-Butoxymethyl",
    "mf": "C5H11O",
    "ocl": {
      "value": "gNqHDEeIVjj`@",
      "coordinates": "!B@FL@[@AcXs|@Xvp@"
    },
    "mass": 87.14043270260808,
    "monoisotopicMass": 87.08098997409999,
    "unsaturation": -1,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Bz",
    "name": "Benzoyl",
    "mf": "C7H5O",
    "ocl": {
      "value": "didH`DAYR[e^FX@@@@",
      "coordinates": "!BbOq~@Ha}b@I~Oxa}bGu~Op"
    },
    "mass": 105.1142599717439,
    "monoisotopicMass": 105.03403978072,
    "unsaturation": 9,
    "elements": [{
      "symbol": "C",
      "number": 7
    }, {
      "symbol": "H",
      "number": 5
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Bzl",
    "name": "Benzyl",
    "mf": "C7H7",
    "ocl": {
      "value": "daD@`@VTeeVz`@@@",
      "coordinates": "!B|Gsp_A|_gp_A}_g|"
    },
    "mass": 91.13073655553718,
    "monoisotopicMass": 91.05477522561,
    "unsaturation": 7,
    "elements": [{
      "symbol": "C",
      "number": 7
    }, {
      "symbol": "H",
      "number": 7
    }]
  }, {
    "symbol": "Bn",
    "name": "Benzyl",
    "mf": "C7H7",
    "ocl": {
      "value": "daD@`@VTeeVz`@@@",
      "coordinates": "!B|Gsp_A|_gp_A}_g|"
    },
    "mass": 91.13073655553718,
    "monoisotopicMass": 91.05477522561,
    "unsaturation": 7,
    "elements": [{
      "symbol": "C",
      "number": 7
    }, {
      "symbol": "H",
      "number": 7
    }]
  }, {
    "symbol": "Bzlo",
    "name": "Benzyloxy",
    "mf": "C7H7O",
    "ocl": {
      "value": "didH`HAYRUe^Fh@@@@",
      "coordinates": "!B|Gwp_OC}|Gq~_A}|Gu~_p"
    },
    "mass": 107.13014147985547,
    "monoisotopicMass": 107.04968984518,
    "unsaturation": 7,
    "elements": [{
      "symbol": "C",
      "number": 7
    }, {
      "symbol": "H",
      "number": 7
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Cha",
    "name": "beta-Cyclohexylalanine diradical",
    "mf": "C9H15NO",
    "ocl": {
      "value": "dknHPBPOEgEInWe]NZjjjcH@",
      "coordinates": "!Bb@I~@Ha}_c~H@m]}bGvH@gxbGvH_Wx"
    },
    "mass": 153.22184251721796,
    "monoisotopicMass": 153.11536410745,
    "unsaturation": 4,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 15
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Chxo",
    "name": "Cyclohexyloxy",
    "mf": "C6H11O",
    "ocl": {
      "value": "daDH`HAYRVU[jjj@@",
      "coordinates": "!B|Gsp_A|_gp_A}_g|"
    },
    "mass": 99.15116859934332,
    "monoisotopicMass": 99.08098997409999,
    "unsaturation": 1,
    "elements": [{
      "symbol": "C",
      "number": 6
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Cit",
    "name": "Citrulline diradical",
    "mf": "C6H11N3O2",
    "ocl": {
      "value": "dkODPBdttOEgEInWUijjihr@@",
      "coordinates": "!Bb@I~@Ha}_c~H@m]}bGvHHa}b@I~@Ha}"
    },
    "mass": 157.170683157999,
    "monoisotopicMass": 157.08512660696,
    "unsaturation": 4,
    "elements": [{
      "symbol": "C",
      "number": 6
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Citp",
    "name": "Citrulline triradical",
    "mf": "C6H10N3O2",
    "ocl": {
      "value": "dgoDpHJ\\l@gWDEI[UYZfjji`T@",
      "coordinates": "!BbGvHGx@bGvH@ha}bOrH_Wxb@KW_Wx@bGt"
    },
    "mass": 156.16274240394318,
    "monoisotopicMass": 156.07730157473,
    "unsaturation": 5,
    "elements": [{
      "symbol": "C",
      "number": 6
    }, {
      "symbol": "H",
      "number": 10
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Clz",
    "name": "2-Chlorobenzyloxycarbonyl",
    "mf": "C8H6ClO2",
    "ocl": {
      "value": "dcLDPDpEdXaImYgWYjB@@@",
      "coordinates": "!Bb@I~@Hb}b@JH_X`B_c}~@Hb}bGu~Op"
    },
    "mass": 169.58527912946118,
    "monoisotopicMass": 169.00563211451998,
    "unsaturation": 9,
    "elements": [{
      "symbol": "C",
      "number": 8
    }, {
      "symbol": "H",
      "number": 6
    }, {
      "symbol": "Cl",
      "number": 1
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Cp",
    "name": "Cyclopentadienyl",
    "mf": "C5H5",
    "ocl": {
      "value": "gFpH@liLimRp@",
      "coordinates": "!B\\OtPThyEGl@fP"
    },
    "mass": 65.09338325395512,
    "monoisotopicMass": 65.03912516115,
    "unsaturation": 5,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 5
    }]
  }, {
    "symbol": "Cys",
    "name": "Cysteine diradical",
    "mf": "C3H5NOS",
    "kind": "aa",
    "oneLetter": "C",
    "alternativeOneLetter": "ς",
    "ocl": {
      "value": "dazHpBPOEgG`aInVZjcH@",
      "coordinates": "!Bb@I~@Ha}_c~H@m]}bGt"
    },
    "mass": 103.14280700237578,
    "monoisotopicMass": 103.00918495955,
    "unsaturation": 2,
    "elements": [{
      "symbol": "C",
      "number": 3
    }, {
      "symbol": "H",
      "number": 5
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Cysp",
    "name": "Cysteine triradical",
    "mf": "C3H4NOS",
    "ocl": {
      "value": "diFHHBD@f@agGoEIVVjjfLP@",
      "coordinates": "!BbGvHHa}_c~HM]}_`BH_P"
    },
    "mass": 102.13486624831998,
    "monoisotopicMass": 102.00135992732,
    "unsaturation": 3,
    "elements": [{
      "symbol": "C",
      "number": 3
    }, {
      "symbol": "H",
      "number": 4
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "D",
    "name": "Deuterium",
    "mf": "[2H]",
    "ocl": {
      "value": "eFAAYhBLCEH@",
      "coordinates": "!B@FL"
    },
    "mass": 2.01410177812,
    "monoisotopicMass": 2.01410177812,
    "unsaturation": -1,
    "elements": [{
      "symbol": "H",
      "number": 1,
      "isotope": 2
    }]
  }, {
    "symbol": "Dde",
    "name": "Dde",
    "mf": "C10H13O2",
    "ocl": {
      "value": "dklD`FDEgHhihicIVZfZj@@",
      "coordinates": "!Bb@I~@Ha}upJH@m]}_`BH_Wx@b@I}bOrH"
    },
    "mass": 165.20939861871415,
    "monoisotopicMass": 165.09155465812998,
    "unsaturation": 7,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Dnp",
    "name": "2,4-Dinitrophenyl",
    "mf": "C6H3N2O4",
    "ocl": {
      "value": "dkmB`bWatpVRd^VS{HhheEUFfBAbX@@",
      "coordinates": "!B_c~H_]]}b@I~Owx_`BH_]]}_c~H_]]}"
    },
    "mass": 167.09926376274353,
    "monoisotopicMass": 167.00928158383,
    "unsaturation": 11,
    "elements": [{
      "symbol": "C",
      "number": 6
    }, {
      "symbol": "H",
      "number": 3
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 4
    }]
  }, {
    "symbol": "Et",
    "name": "Ethyl",
    "mf": "C2H5",
    "ocl": {
      "value": "eMBAYRZ@",
      "coordinates": "!B@Fp@Xp"
    },
    "mass": 29.061175563749384,
    "monoisotopicMass": 29.03912516115,
    "unsaturation": -1,
    "elements": [{
      "symbol": "C",
      "number": 2
    }, {
      "symbol": "H",
      "number": 5
    }]
  }, {
    "symbol": "Fmoc",
    "name": "Fluorenylmethoxycarbonyl",
    "mf": "C15H11O2",
    "ocl": {
      "value": "fde@b@DX@liMkLrjxeVCzLuT@@@P@@@",
      "coordinates": "!BbOq~@Ha}bOrH_]ARcm}Tv~i`pAeKv|@fpB[j[~iozfAKvp"
    },
    "mass": 223.24719659427882,
    "monoisotopicMass": 223.07590459367,
    "unsaturation": 19,
    "elements": [{
      "symbol": "C",
      "number": 15
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "For",
    "name": "Formyl",
    "mf": "CHO",
    "ocl": {
      "value": "eMJDVTfP@",
      "coordinates": "!B@Fp@Xp"
    },
    "mass": 29.018081575109303,
    "monoisotopicMass": 29.0027396518,
    "unsaturation": 1,
    "elements": [{
      "symbol": "C",
      "number": 1
    }, {
      "symbol": "H",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Gln",
    "name": "Glutamine diradical",
    "mf": "C5H8N2O2",
    "kind": "aa",
    "oneLetter": "Q",
    "alternativeOneLetter": "ξ",
    "ocl": {
      "value": "dmUDPBUICqYqR[evfjihr@@",
      "coordinates": "!Bb@I~@Ha}_c~H@m]}bGvHGx@bGt"
    },
    "mass": 128.12942178765059,
    "monoisotopicMass": 128.05857750584,
    "unsaturation": 4,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 8
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Glnp",
    "name": "Glutamine triradical",
    "mf": "C5H7N2O2",
    "ocl": {
      "value": "dcuDpH{MAYeqWqRVuejZjiad@",
      "coordinates": "!BbGvHGx@bGvH@ha}_c~HM]}_`BH_P"
    },
    "mass": 127.12148103359483,
    "monoisotopicMass": 127.05075247361,
    "unsaturation": 5,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 7
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Glp",
    "name": "Pyroglutamine",
    "mf": "C5H5NO2",
    "ocl": {
      "value": "deVDPBRP|V\\TfygxYjjZL`@",
      "coordinates": "!Bb@I~@Ha}tEJNwr[@UMo@FXBN"
    },
    "mass": 111.09889631403748,
    "monoisotopicMass": 111.03202840472,
    "unsaturation": 6,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 5
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Glu",
    "name": "Glutamic acid diradical",
    "mf": "C5H7NO3",
    "kind": "aa",
    "oneLetter": "E",
    "alternativeOneLetter": "ε",
    "ocl": {
      "value": "dmVLPBRUCqYqR[evfjihr@@",
      "coordinates": "!Bb@I~@Ha}_c~H@m]}bGvHGx@bGt"
    },
    "mass": 129.11418274646732,
    "monoisotopicMass": 129.04259308875,
    "unsaturation": 4,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 7
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 3
    }]
  }, {
    "symbol": "Glup",
    "name": "Glutamic acid triradical",
    "mf": "C5H6NO3",
    "ocl": {
      "value": "dcvLpNcM@IeqWqRVuejZjiad@",
      "coordinates": "!BbGvHGx@bGvH@ha}_c~HM]}_`BH_P"
    },
    "mass": 128.10624199241153,
    "monoisotopicMass": 128.03476805652002,
    "unsaturation": 5,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 6
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 3
    }]
  }, {
    "symbol": "Gly",
    "name": "Glycine diradical",
    "mf": "C2H3NO",
    "kind": "aa",
    "oneLetter": "G",
    "alternativeOneLetter": "γ",
    "ocl": {
      "value": "gGYDBaxuqR[Yj@@",
      "coordinates": "!BbOq~@Ha}bOrH_P"
    },
    "mass": 57.051402191401905,
    "monoisotopicMass": 57.021463720689994,
    "unsaturation": 2,
    "elements": [{
      "symbol": "C",
      "number": 2
    }, {
      "symbol": "H",
      "number": 3
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Hci",
    "name": "Homocitrulline diradical",
    "mf": "C7H13N3O2",
    "ocl": {
      "value": "dgoDPBVtLOEgEInWUZZjjfcH@",
      "coordinates": "!BbGu~Ox`B_`BH_Xc|bOrH_X`BbGvHGx@bGt"
    },
    "mass": 171.19730056284578,
    "monoisotopicMass": 171.10077667142,
    "unsaturation": 4,
    "elements": [{
      "symbol": "C",
      "number": 7
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Hcip",
    "name": "Homocitrulline triradical",
    "mf": "C7H12N3O2",
    "ocl": {
      "value": "do_DpHI\\\\EdwFEI[UVVijjjfIP@",
      "coordinates": "!BbOrH_Wxb@JH_Xc|bGvHHa}_c~H@m]}_`BH_P"
    },
    "mass": 170.18935980879002,
    "monoisotopicMass": 170.09295163918998,
    "unsaturation": 5,
    "elements": [{
      "symbol": "C",
      "number": 7
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "His",
    "name": "Histidine diradical",
    "mf": "C6H7N3O",
    "kind": "aa",
    "oneLetter": "H",
    "alternativeOneLetter": "ζ",
    "ocl": {
      "value": "dcOHPBGTCqYqR[eyUvZjejL`@",
      "coordinates": "!Bb@I~@Ha}_c~H@m]}bGwPTh{_UMo@FP"
    },
    "mass": 137.13951521745759,
    "monoisotopicMass": 137.05891185847,
    "unsaturation": 8,
    "elements": [{
      "symbol": "C",
      "number": 6
    }, {
      "symbol": "H",
      "number": 7
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Hisp",
    "name": "Histidine triradical",
    "mf": "C6H6N3O",
    "ocl": {
      "value": "dkoHpHHSAYUqwqRY]YXjjVjihy@@",
      "coordinates": "!BTmA}bL@fUHRN`H`BbGu~Ox`Buwu~@Ha}"
    },
    "mass": 136.13157446340182,
    "monoisotopicMass": 136.05108682624,
    "unsaturation": 9,
    "elements": [{
      "symbol": "C",
      "number": 6
    }, {
      "symbol": "H",
      "number": 6
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Hser",
    "name": "Homoserine diradical",
    "mf": "C4H7NO2",
    "ocl": {
      "value": "diFDPBPP|V\\Tfy^Zjhr@@",
      "coordinates": "!BbGu~Ox`B_`BH_X`Bb@JH_P"
    },
    "mass": 101.10404192541378,
    "monoisotopicMass": 101.04767846918,
    "unsaturation": 2,
    "elements": [{
      "symbol": "C",
      "number": 4
    }, {
      "symbol": "H",
      "number": 7
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Hserp",
    "name": "Homoserine triradical",
    "mf": "C4H6NO2",
    "ocl": {
      "value": "defDpJbPV^\\Q|TeVVjji`d@",
      "coordinates": "!Bb@JH_X`BbGu~Oxc|uwu~@Ha}"
    },
    "mass": 100.09610117135801,
    "monoisotopicMass": 100.03985343695001,
    "unsaturation": 3,
    "elements": [{
      "symbol": "C",
      "number": 4
    }, {
      "symbol": "H",
      "number": 6
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Hyp",
    "name": "Hydroxyproline diradical",
    "mf": "C5H7NO2",
    "ocl": {
      "value": "deVDPBRP|V\\\\bfbbOCMUUIdE@@",
      "coordinates": "!Bb@I~@Ha}tEJNwr[@UMo@FUJO"
    },
    "mass": 113.11477782214904,
    "monoisotopicMass": 113.04767846918,
    "unsaturation": 4,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 7
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Hypp",
    "name": "Hydroxyproline triradical",
    "mf": "C5H6NO2",
    "ocl": {
      "value": "dmvDpJaPB^\\Y|TeeWjZjjidRL`@",
      "coordinates": "!BBOpH_UARcc}TNtBY@HyRSpCQDr\\"
    },
    "mass": 112.10683706809326,
    "monoisotopicMass": 112.03985343695001,
    "unsaturation": 5,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 6
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Ile",
    "name": "Isoleucine diradical",
    "mf": "C6H11NO",
    "kind": "aa",
    "oneLetter": "I",
    "alternativeOneLetter": "ι",
    "ocl": {
      "value": "defHPBPOEgEInVyjjdrT`@",
      "coordinates": "!BbGu~Oxc|_`BH_Xc|b@I~Oxa}"
    },
    "mass": 113.15787181078912,
    "monoisotopicMass": 113.08406397853,
    "unsaturation": 2,
    "elements": [{
      "symbol": "C",
      "number": 6
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Ivdde",
    "name": "1-[4,4-dimethyl-2,6-dioxocyclohexylidene)-3-methylbutyl",
    "mf": "C13H19O2",
    "ocl": {
      "value": "f`a@b@NR@lyEEDhhigEVfjYjj`@@",
      "coordinates": "!BbOq~@Ha}urHGxuwu~@Ha}_`CW_Xa}bOq}b@JH"
    },
    "mass": 207.28925083325453,
    "monoisotopicMass": 207.13850485151,
    "unsaturation": 7,
    "elements": [{
      "symbol": "C",
      "number": 13
    }, {
      "symbol": "H",
      "number": 19
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Leu",
    "name": "Leucine diradical",
    "mf": "C6H11NO",
    "kind": "aa",
    "oneLetter": "L",
    "alternativeOneLetter": "λ",
    "ocl": {
      "value": "defHPBPOEgEInWijjhr@@",
      "coordinates": "!BbGu~Ox`B_`BH_X`Bb@I~@Ha}"
    },
    "mass": 113.15787181078912,
    "monoisotopicMass": 113.08406397853,
    "unsaturation": 2,
    "elements": [{
      "symbol": "C",
      "number": 6
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Lys",
    "name": "Lysine diradical",
    "mf": "C6H12N2O",
    "kind": "aa",
    "oneLetter": "K",
    "alternativeOneLetter": "κ",
    "ocl": {
      "value": "dmUHPBU@|V\\Tfy]YjjjL`@",
      "coordinates": "!BbGu~Ox`B_`BHoX`Bb@JH_X`BbKt"
    },
    "mass": 128.17251577629068,
    "monoisotopicMass": 128.09496301519,
    "unsaturation": 2,
    "elements": [{
      "symbol": "C",
      "number": 6
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Lysp",
    "name": "Lysine triradical",
    "mf": "C6H11N2O",
    "ocl": {
      "value": "dcuHpH{PVY\\U|TeUYZjjjXY@@",
      "coordinates": "!Bb@JH_X`BbGvH@ha}_c~H@m]}_`BH_P"
    },
    "mass": 127.16457502223491,
    "monoisotopicMass": 127.08713798295999,
    "unsaturation": 3,
    "elements": [{
      "symbol": "C",
      "number": 6
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Mbh",
    "name": "4,4'-Dimethoxybenzhydryl",
    "mf": "C15H15O2",
    "ocl": {
      "value": "fdy@b@G^@liLsJkzlcZmT@@@UP@@@",
      "coordinates": "!BbGvHGx_`BH_Xa}uwvHHc|_c}~Oxa}uwvHGxbGwW_P"
    },
    "mass": 227.27895961050194,
    "monoisotopicMass": 227.10720472258998,
    "unsaturation": 15,
    "elements": [{
      "symbol": "C",
      "number": 15
    }, {
      "symbol": "H",
      "number": 15
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Me",
    "name": "Methyl",
    "mf": "CH3",
    "ocl": {
      "value": "eFBAYc@@",
      "coordinates": "!B@FL"
    },
    "mass": 15.03455815890258,
    "monoisotopicMass": 15.02347509669,
    "unsaturation": -1,
    "elements": [{
      "symbol": "C",
      "number": 1
    }, {
      "symbol": "H",
      "number": 3
    }]
  }, {
    "symbol": "Mebzl",
    "name": "4-Methylbenzyl",
    "mf": "C8H9",
    "ocl": {
      "value": "did@`@VTee]nh@H@@",
      "coordinates": "!B|Gsp__A|_gp_C}_gp_P"
    },
    "mass": 105.15735396038399,
    "monoisotopicMass": 105.07042529007,
    "unsaturation": 7,
    "elements": [{
      "symbol": "C",
      "number": 8
    }, {
      "symbol": "H",
      "number": 9
    }]
  }, {
    "symbol": "Meobzl",
    "name": "4-Methoxybenzyl",
    "mf": "C8H9O",
    "ocl": {
      "value": "deTH`AAYRVUunh@J@@",
      "coordinates": "!B|Gsp__A|_gp_A}_gp_Wy"
    },
    "mass": 121.15675888470227,
    "monoisotopicMass": 121.06533990964,
    "unsaturation": 7,
    "elements": [{
      "symbol": "C",
      "number": 8
    }, {
      "symbol": "H",
      "number": 9
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Met",
    "name": "Methionine diradical",
    "mf": "C5H9NOS",
    "kind": "aa",
    "oneLetter": "M",
    "alternativeOneLetter": "μ",
    "ocl": {
      "value": "defHpBPOEgDPaInWYjjhr@@",
      "coordinates": "!Bb@I~@Ha}_c~H@m]}bGvHHa}"
    },
    "mass": 131.19604181206938,
    "monoisotopicMass": 131.04048508847,
    "unsaturation": 2,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 9
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Mmt",
    "name": "4-Methoxytrityl",
    "mf": "C20H17O",
    "ocl": {
      "value": "ffcAB@B`V\\bdTTTRRRVvIhnRGMT@@@@AP@@@",
      "coordinates": "!BbKvHM^}_c}~@Hb}dXWHb}j|nHHc|AqOWoWxJV^Ho]\\BuwvHHb}"
    },
    "mass": 273.3491156779715,
    "monoisotopicMass": 273.12794016748,
    "unsaturation": 23,
    "elements": [{
      "symbol": "C",
      "number": 20
    }, {
      "symbol": "H",
      "number": 17
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Mtc",
    "name": "2,2,5,7,8-pentamethylchroman-6-sulphonyl",
    "mf": "C14H19O3S",
    "ocl": {
      "value": "fleAa@DX\\AY`DYEHXhhilmiKW`rpDQUUD@@",
      "coordinates": "!BbGtBbGwWbGvHGxbGu~@Ha}uwu~Ox`B_c~H_Xa}b@H@_osW"
    },
    "mass": 267.36417906043516,
    "monoisotopicMass": 267.10549064548,
    "unsaturation": 9,
    "elements": [{
      "symbol": "C",
      "number": 14
    }, {
      "symbol": "H",
      "number": 19
    }, {
      "symbol": "O",
      "number": 3
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Mtr",
    "name": "4-Methoxy-2,3,6-trimethylbenzenesulphonyl",
    "mf": "C10H13O3S",
    "ocl": {
      "value": "do|LPDrpVXBLbdLTTTngYXBHj@@",
      "coordinates": "!BbOq}b@KWb@I~@Ha}bOsWHc|_c~H_Wx@b@JH_P"
    },
    "mass": 213.27359094915948,
    "monoisotopicMass": 213.05854045209998,
    "unsaturation": 7,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "O",
      "number": 3
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Mts",
    "name": "Mesitylene-2-sulphonyl",
    "mf": "C9H11O2S",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 183.24756861999438,
    "monoisotopicMass": 183.04797576807,
    "unsaturation": 7,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "O",
      "number": 2
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Mtt",
    "name": "4-Methyltrityl",
    "mf": "C20H17",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 257.3497107536532,
    "monoisotopicMass": 257.13302554791,
    "unsaturation": 23,
    "elements": [{
      "symbol": "C",
      "number": 20
    }, {
      "symbol": "H",
      "number": 17
    }]
  }, {
    "symbol": "Nle",
    "name": "Norleucine diradical",
    "mf": "C6H11NO",
    "ocl": {
      "value": "defHPBPOEgEInWYjjhr@@",
      "coordinates": "!Bb@I~@Ha}_c~H@m]}bGvHHa}"
    },
    "mass": 113.15787181078912,
    "monoisotopicMass": 113.08406397853,
    "unsaturation": 2,
    "elements": [{
      "symbol": "C",
      "number": 6
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Npys",
    "name": "3-Nitro-2-pyridinesulphenyl",
    "mf": "C5H3N2O2S",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 155.1545054234988,
    "monoisotopicMass": 154.99152351908998,
    "unsaturation": 9,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 3
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 2
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Nva",
    "name": "Norvaline diradical",
    "mf": "C5H9NO",
    "ocl": {
      "value": "diFHPBPOEgEInWfjjL`@",
      "coordinates": "!BbGu~Ox`B_`BH_X`Bb@JH_P"
    },
    "mass": 99.13125440594231,
    "monoisotopicMass": 99.06841391407,
    "unsaturation": 2,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 9
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Odmab",
    "name": "Odmab",
    "mf": "C20H26NO3",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 328.4260955245558,
    "monoisotopicMass": 328.19126870111995,
    "unsaturation": 15,
    "elements": [{
      "symbol": "C",
      "number": 20
    }, {
      "symbol": "H",
      "number": 26
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 3
    }]
  }, {
    "symbol": "Orn",
    "name": "Ornithine diradical",
    "mf": "C5H10N2O",
    "ocl": {
      "value": "deeHPBe@|V\\Tfy]fjjcH@",
      "coordinates": "!Bb@I~@Ha}_c~H@m]}bGvHHa}"
    },
    "mass": 114.14589837144388,
    "monoisotopicMass": 114.07931295072999,
    "unsaturation": 2,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 10
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Ornp",
    "name": "Ornithine triradical",
    "mf": "C5H9N2O",
    "ocl": {
      "value": "dmUHpHYPBQ\\Y|TeUejjjfJP@",
      "coordinates": "!BbGvHHa}b@JH_Wxb@KW_Wx@bGt"
    },
    "mass": 113.13795761738811,
    "monoisotopicMass": 113.0714879185,
    "unsaturation": 3,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 9
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Pbf",
    "name": "2,2,4,6,7-pentamethyldihydrobenzofurane-5-sulfonyl",
    "mf": "C13H17O3S",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 253.33756165558833,
    "monoisotopicMass": 253.08984058101998,
    "unsaturation": 9,
    "elements": [{
      "symbol": "C",
      "number": 13
    }, {
      "symbol": "H",
      "number": 17
    }, {
      "symbol": "O",
      "number": 3
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Pen",
    "name": "Penicillamine diradical",
    "mf": "C5H9NOS",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 131.19604181206938,
    "monoisotopicMass": 131.04048508847,
    "unsaturation": 2,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 9
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Penp",
    "name": "Penicillamine triradical",
    "mf": "C5H8NOS",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 130.1881010580136,
    "monoisotopicMass": 130.03266005624,
    "unsaturation": 3,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 8
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Ph",
    "name": "Phenyl",
    "mf": "C6H5",
    "ocl": {
      "value": "gOpH@liLkW@@@@",
      "coordinates": "!B|Owp_Gy|OwpWy"
    },
    "mass": 77.10411915069038,
    "monoisotopicMass": 77.03912516115,
    "unsaturation": 7,
    "elements": [{
      "symbol": "C",
      "number": 6
    }, {
      "symbol": "H",
      "number": 5
    }]
  }, {
    "symbol": "Phe",
    "name": "Phenylalanine diradical",
    "mf": "C9H9NO",
    "kind": "aa",
    "oneLetter": "F",
    "alternativeOneLetter": "φ",
    "ocl": {
      "value": "dknHPBPOEgEInWe]NZj@@cH@",
      "coordinates": "!Bb@I~@Ha}_c~H@m]}bGvH@gxbGvH_Wx"
    },
    "mass": 147.1741979928833,
    "monoisotopicMass": 147.06841391407002,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 9
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Phepcl",
    "name": "4-Chlorophenylalanine diradical",
    "mf": "C9H8ClNO",
    "ocl": {
      "value": "dg^HpBPOEgFxaInWe_Sfj`@bL`@",
      "coordinates": "!BbOq~@Ha}_c~H@m]}bGvH@gxbGvH_WxbGt"
    },
    "mass": 181.6191948214355,
    "monoisotopicMass": 181.02944156384,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 8
    }, {
      "symbol": "Cl",
      "number": 1
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Pmc",
    "name": "2,2,5,7,8-Pentamethylchroman-6-sulphonyl",
    "mf": "C14H19O3S",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 267.36417906043516,
    "monoisotopicMass": 267.10549064548,
    "unsaturation": 9,
    "elements": [{
      "symbol": "C",
      "number": 14
    }, {
      "symbol": "H",
      "number": 19
    }, {
      "symbol": "O",
      "number": 3
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Pro",
    "name": "Proline diradical",
    "mf": "C5H7NO",
    "kind": "aa",
    "oneLetter": "P",
    "alternativeOneLetter": "π",
    "ocl": {
      "value": "difHPBPOEgEInYxYjjhr@@",
      "coordinates": "!Bb@I~@Ha}tEJNwr[@UMo@FP"
    },
    "mass": 97.11537289783075,
    "monoisotopicMass": 97.05276384961,
    "unsaturation": 4,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 7
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Pyr",
    "name": "Pyroglutamine",
    "mf": "C5H5NO2",
    "ocl": {
      "value": "deVDPBRP|V\\TfygxYjjZL`@",
      "coordinates": "!Bb@I~@Ha}tEJNwr[@UMo@FXBN"
    },
    "mass": 111.09889631403748,
    "monoisotopicMass": 111.03202840472,
    "unsaturation": 6,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 5
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Sar",
    "name": "Sarcosine diradical",
    "mf": "C3H5NO",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 71.07801959624871,
    "monoisotopicMass": 71.03711378515,
    "unsaturation": 2,
    "elements": [{
      "symbol": "C",
      "number": 3
    }, {
      "symbol": "H",
      "number": 5
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Ser",
    "name": "Serine diradical",
    "mf": "C3H5NO2",
    "kind": "aa",
    "oneLetter": "S",
    "alternativeOneLetter": "σ",
    "ocl": {
      "value": "dazDPBS`|V\\TfyYjjL`@",
      "coordinates": "!Bb@I~@Ha}_c~H@m]}bGt"
    },
    "mass": 87.07742452056698,
    "monoisotopicMass": 87.03202840472,
    "unsaturation": 2,
    "elements": [{
      "symbol": "C",
      "number": 3
    }, {
      "symbol": "H",
      "number": 5
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Serp",
    "name": "Serine triradical",
    "mf": "C3H4NO2",
    "ocl": {
      "value": "diFDpB`PBV\\^|TeYZjjXq@@",
      "coordinates": "!BbGvHHa}_c~HM]}_`BH_P"
    },
    "mass": 86.06948376651121,
    "monoisotopicMass": 86.02420337249,
    "unsaturation": 3,
    "elements": [{
      "symbol": "C",
      "number": 3
    }, {
      "symbol": "H",
      "number": 4
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Sta",
    "name": "Statine diradical",
    "mf": "C8H15NO2",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 157.210511544801,
    "monoisotopicMass": 157.11027872702002,
    "unsaturation": 2,
    "elements": [{
      "symbol": "C",
      "number": 8
    }, {
      "symbol": "H",
      "number": 15
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Stap",
    "name": "Statine triradical",
    "mf": "C8H14NO2",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 156.2025707907452,
    "monoisotopicMass": 156.10245369479,
    "unsaturation": 3,
    "elements": [{
      "symbol": "C",
      "number": 8
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Tacm",
    "name": "Trimethylacetamidomethyl",
    "mf": "C6H12NO",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 114.16581256484488,
    "monoisotopicMass": 114.09188901076,
    "unsaturation": 1,
    "elements": [{
      "symbol": "C",
      "number": 6
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Tbdms",
    "name": "t-Butyldimethylsilyl",
    "mf": "C6H15Si",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 115.2690253969541,
    "monoisotopicMass": 115.09430201810001,
    "unsaturation": -1,
    "elements": [{
      "symbol": "C",
      "number": 6
    }, {
      "symbol": "H",
      "number": 15
    }, {
      "symbol": "Si",
      "number": 1
    }]
  }, {
    "symbol": "Tbu",
    "name": "t-Butyl",
    "mf": "C4H9",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 57.114410373442986,
    "monoisotopicMass": 57.07042529007,
    "unsaturation": -1,
    "elements": [{
      "symbol": "C",
      "number": 4
    }, {
      "symbol": "H",
      "number": 9
    }]
  }, {
    "symbol": "Tbuo",
    "name": "t-Butoxy",
    "mf": "C4H9O",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 73.11381529776126,
    "monoisotopicMass": 73.06533990964,
    "unsaturation": -1,
    "elements": [{
      "symbol": "C",
      "number": 4
    }, {
      "symbol": "H",
      "number": 9
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Tbuthio",
    "name": "t-Butylthio",
    "mf": "C4H9S",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 89.17919777957005,
    "monoisotopicMass": 89.04249646446999,
    "unsaturation": -1,
    "elements": [{
      "symbol": "C",
      "number": 4
    }, {
      "symbol": "H",
      "number": 9
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Tfa",
    "name": "Trifluoroacetyl",
    "mf": "C2F3O",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 97.01608620597878,
    "monoisotopicMass": 96.99012410776,
    "unsaturation": 1,
    "elements": [{
      "symbol": "C",
      "number": 2
    }, {
      "symbol": "F",
      "number": 3
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Thr",
    "name": "Threonine diradical",
    "mf": "C4H7NO2",
    "kind": "aa",
    "oneLetter": "T",
    "alternativeOneLetter": "τ",
    "ocl": {
      "value": "d@"
    },
    "mass": 101.10404192541378,
    "monoisotopicMass": 101.04767846918,
    "unsaturation": 2,
    "elements": [{
      "symbol": "C",
      "number": 4
    }, {
      "symbol": "H",
      "number": 7
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Thrp",
    "name": "Threonine triradical",
    "mf": "C4H6NO2",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 100.09610117135801,
    "monoisotopicMass": 100.03985343695001,
    "unsaturation": 3,
    "elements": [{
      "symbol": "C",
      "number": 4
    }, {
      "symbol": "H",
      "number": 6
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Tfsi",
    "name": "(Bis)(trifluoromethanesulfonyl)imide",
    "mf": "C2F6NO4S2",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 280.1457884908235,
    "monoisotopicMass": 279.91729380789,
    "unsaturation": -1,
    "elements": [{
      "symbol": "C",
      "number": 2
    }, {
      "symbol": "F",
      "number": 6
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 4
    }, {
      "symbol": "S",
      "number": 2
    }]
  }, {
    "symbol": "Tips",
    "name": "Triisopropylsilyl",
    "mf": "C9H21Si",
    "ocl": {
      "value": "dmT@P@VX\\DffYjjjh@@",
      "coordinates": "!B_a@gHb\\]FBIuWxP^zi~KwxPFAt"
    },
    "mass": 157.34887761149452,
    "monoisotopicMass": 157.14125221148,
    "unsaturation": -1,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 21
    }, {
      "symbol": "Si",
      "number": 1
    }]
  }, {
    "symbol": "Tms",
    "name": "Trimethylsilyl",
    "mf": "C3H9Si",
    "ocl": {
      "value": "gJPD@lqpRZj`@",
      "coordinates": "!BbOq~@GxbGt"
    },
    "mass": 73.1891731824137,
    "monoisotopicMass": 73.04735182472,
    "unsaturation": -1,
    "elements": [{
      "symbol": "C",
      "number": 3
    }, {
      "symbol": "H",
      "number": 9
    }, {
      "symbol": "Si",
      "number": 1
    }]
  }, {
    "symbol": "Tos",
    "name": "Tosyl",
    "mf": "C7H7O2S",
    "ocl": {
      "value": "dmtDPDpEf@cHiCDeafV@B@@",
      "coordinates": "!B|Ou||Ovw|Gwp_Gy|GwpWy|Gt"
    },
    "mass": 155.1943338103008,
    "monoisotopicMass": 155.01667563914998,
    "unsaturation": 7,
    "elements": [{
      "symbol": "C",
      "number": 7
    }, {
      "symbol": "H",
      "number": 7
    }, {
      "symbol": "O",
      "number": 2
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Trp",
    "name": "Tryptophan diradical",
    "mf": "C11H10N2O",
    "kind": "aa",
    "oneLetter": "W",
    "alternativeOneLetter": "ω",
    "ocl": {
      "value": "f`qQA@BFPCqXxiMr|rnhsoSUTa@QCD@@",
      "coordinates": "!BbOq~@Ha}_c~H@m]}bGwPTh{_UMojXL@YpB[@Ini`"
    },
    "mass": 186.21031375185538,
    "monoisotopicMass": 186.07931295073,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 10
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Trpp",
    "name": "Tryptophan triradical",
    "mf": "C11H9N2O",
    "ocl": {
      "value": "fhiQC@HFB@I\\x~|TfYU_ebLDjhDHjibFd@",
      "coordinates": "!BTmA}bL@fUHR_Ihz@iVBeXHc|grZH_WxbOsW_Wx@bGt"
    },
    "mass": 185.20237299779959,
    "monoisotopicMass": 185.07148791850003,
    "unsaturation": 15,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 9
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Trt",
    "name": "Trityl",
    "mf": "C19H15",
    "ocl": {
      "value": "fbm@B@@KJSSLrjkyhnRGMT@@@@@@@@",
      "coordinates": "!BrHI~PGy_rMvW@l`BQCvWw\\bBAg}~PGy@]i}~W|c]cNwH`i_]_e|"
    },
    "mass": 243.32309334880637,
    "monoisotopicMass": 243.11737548345,
    "unsaturation": 23,
    "elements": [{
      "symbol": "C",
      "number": 19
    }, {
      "symbol": "H",
      "number": 15
    }]
  }, {
    "symbol": "Tyr",
    "name": "Tyrosine diradical",
    "mf": "C9H9NO2",
    "kind": "aa",
    "oneLetter": "Y",
    "alternativeOneLetter": "ψ",
    "ocl": {
      "value": "dg^DPBRp|V\\Tfy^U}NZj@BHr@@",
      "coordinates": "!BbOq~@Ha}_c~H@m]}bGvH@gxbGvH_WxbGt"
    },
    "mass": 163.1736029172016,
    "monoisotopicMass": 163.06332853364,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 9
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Tyrp",
    "name": "Tyrosine triradical",
    "mf": "C9H8NO2",
    "ocl": {
      "value": "do~DpEapBS\\[|Tee]YYnh@JjdbT@",
      "coordinates": "!B_`BHGx@bGvH@h`BbKvH@ha}_c~H@m]}_`BHoP"
    },
    "mass": 162.16566216314578,
    "monoisotopicMass": 162.05550350141,
    "unsaturation": 11,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 8
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Val",
    "name": "Valine",
    "mf": "C5H9NO",
    "kind": "aa",
    "oneLetter": "V",
    "alternativeOneLetter": "ν",
    "ocl": {
      "value": "diFHPBPOEgEInVfjjL`@",
      "coordinates": "!Bb@I~@Ha}_c~H@m]}_`BH_P"
    },
    "mass": 99.13125440594231,
    "monoisotopicMass": 99.06841391407,
    "unsaturation": 2,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 9
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Valoh",
    "name": "beta-Hydroxyvaline diradical",
    "mf": "C5H9NO2",
    "ocl": {
      "value": "defDPBS`|V\\TfyZfjjcH@",
      "coordinates": "!Bb@I~@Ha}b@I~Oxa}Owy~OpA~"
    },
    "mass": 115.13065933026058,
    "monoisotopicMass": 115.06332853364,
    "unsaturation": 2,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 9
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Valohp",
    "name": "beta-Hydroxyvaline triradical",
    "mf": "C5H8NO2",
    "ocl": {
      "value": "dmVDpFaPBQ\\Y|\\bTbaTjjjXq@@",
      "coordinates": "!BbGvHHa}_Xc|bGxb@KW_Wx@bGt"
    },
    "mass": 114.1227185762048,
    "monoisotopicMass": 114.05550350141002,
    "unsaturation": 3,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 8
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Xan",
    "name": "Xanthyl",
    "mf": "C13H9O",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 181.21043836837848,
    "monoisotopicMass": 181.06533990964002,
    "unsaturation": 17,
    "elements": [{
      "symbol": "C",
      "number": 13
    }, {
      "symbol": "H",
      "number": 9
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Xle",
    "name": "Leucine or Isoleucine diradical",
    "mf": "C6H11NO",
    "kind": "aa",
    "oneLetter": "J",
    "mass": 113.15787181078912,
    "monoisotopicMass": 113.08406397853,
    "unsaturation": 2,
    "elements": [{
      "symbol": "C",
      "number": 6
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Z",
    "name": "Benzyloxycarbonyl",
    "mf": "C8H7O2",
    "ocl": {
      "value": "dmtD`DpEeImYVUfh@@@@",
      "coordinates": "!Bb@I~@Ha}b@JH_Xc|_c~H_Xa}_c|"
    },
    "mass": 135.14028230090898,
    "monoisotopicMass": 135.04460446475,
    "unsaturation": 9,
    "elements": [{
      "symbol": "C",
      "number": 8
    }, {
      "symbol": "H",
      "number": 7
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Damp",
    "name": "Desoxyadenosine monophosphate diradical",
    "mf": "C10H12N5O5P",
    "kind": "DNAp",
    "oneLetter": "A",
    "alternativeOneLetter": "α",
    "ocl": {
      "value": "fnsiS@IASUlJB]xGbkplxyDhhldhiEEUeSdTekUUUULBATXPlKd@@",
      "coordinates": "!Bqc}{JxyO|XoSWC}W]poGQ\\Ou}]rmx\\Ou}]{qpza|qb}MJwlk^sFO|X"
    },
    "mass": 313.2069506932622,
    "monoisotopicMass": 313.05760550518,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 5
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Dcmp",
    "name": "Desoxycytidine monophosphate diradical",
    "mf": "C9H12N3O6P",
    "kind": "DNAp",
    "oneLetter": "C",
    "alternativeOneLetter": "ς",
    "ocl": {
      "value": "fjmps@IQKB`g^BCqUxV\\\\bTTVRTTbb^iqNZjjjifVkBEa\\`@",
      "coordinates": "!Bqc}{JxyO|XoSWA}_W]poGQ\\GuMKuMh\\Gu}]{qpSF]tWQTvatP"
    },
    "mass": 289.18221329795364,
    "monoisotopicMass": 289.04637211589,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 6
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Dgmp",
    "name": "Desoxyguanosine monophosphate diradical",
    "mf": "C10H12N5O6P",
    "kind": "DNAp",
    "oneLetter": "G",
    "alternativeOneLetter": "γ",
    "ocl": {
      "value": "fakhs@IASUlJB]{hOEWaYqrIQQYIQRJJkQTyEIZuUUUSRtsUaBpnP@",
      "coordinates": "!Bqc}{JxyO|XoSWA}W]poGQ\\Gu}]rmx\\Ou}]{qpza|qb}MJwlk^sFza|q`"
    },
    "mass": 329.20635561758047,
    "monoisotopicMass": 329.05252012475,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 6
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Dtmp",
    "name": "Desoxythymidine monophosphate diradical",
    "mf": "C10H13N2O7P",
    "kind": "DNAp",
    "oneLetter": "T",
    "alternativeOneLetter": "τ",
    "ocl": {
      "value": "ff}Qs@IQaPSoAjCqUxV\\\\bTTVRTTbbZUNIsUUUULsSVDKBy@@",
      "coordinates": "!Bqc}{JxyO|XoSWC}_W]poGQ\\GuMKuMh\\Gu}]{qpSF]tWQTvaSZGQ"
    },
    "mass": 304.1935916616171,
    "monoisotopicMass": 304.04603776326,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Dump",
    "name": "Desoxyuridine monophosphate diradical",
    "mf": "C9H11N2O7P",
    "kind": "DNAp",
    "oneLetter": "U",
    "alternativeOneLetter": "υ",
    "ocl": {
      "value": "fjmQs@IQaPSoAJCqUxV\\\\bTTVRTTbb^iqNZjjjifYkBEa\\`@",
      "coordinates": "!Bqc}{JxyO|XoSWA}_W]poGQ\\GuMKuMh\\Gu}]{qpSF]tWQTvatP"
    },
    "mass": 290.1669742567703,
    "monoisotopicMass": 290.0303876988,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Drmp",
    "name": "Desoxyribose monophosphate diradical",
    "mf": "C5H7O5P",
    "kind": "DNAp",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 178.08005138207807,
    "monoisotopicMass": 178.00311032188,
    "unsaturation": 4,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 7
    }, {
      "symbol": "O",
      "number": 5
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Dadp",
    "name": "Desoxyadenosine diphosphate diradical",
    "mf": "C10H13N5O8P2",
    "kind": "DNApp",
    "oneLetter": "A",
    "ocl": {
      "value": "fmwhH`IASM\\JBl{wQ`|U^F_AkbdlsjsSOoRtyEMYuUUUM@pSEQaBpnP@",
      "coordinates": "!BIi[Rx{_grZOSXa}_]^H@mQbGu}utnDM^HGwWzf~_Ih}M_`AKvto[_`@_`A~grZ_I`"
    },
    "mass": 393.1868682186928,
    "monoisotopicMass": 393.02393639454,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 2
    }]
  }, {
    "symbol": "Dcdp",
    "name": "Desoxycytidine diphosphate diradical",
    "mf": "C9H13N3O9P2",
    "kind": "DNApp",
    "oneLetter": "C",
    "ocl": {
      "value": "fikqH`IQGB`kN|EoP^JoCOaUqrIQQYIQRJKGRJgDejjjjZYfZkBEa\\`@",
      "coordinates": "!BIi[Rx{_grZOSXa}_]^H@mQbGuMcqLX@m^H@gwWKB__t]Q_`@SFGx@Owx@_mQ"
    },
    "mass": 369.16213082338425,
    "monoisotopicMass": 369.01270300525005,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 9
    }, {
      "symbol": "P",
      "number": 2
    }]
  }, {
    "symbol": "Dgdp",
    "name": "Desoxyguanosine diphosphate diradical",
    "mf": "C10H13N5O9P2",
    "kind": "DNApp",
    "oneLetter": "G",
    "ocl": {
      "value": "fcoiH`IASM\\JBl{wQ{Axj|L~CWEIYgUff_^fZ\\bflzjjjfiZifZlHVEr@@",
      "coordinates": "!BIi[Rx{_grZOSXa}_]^H@mQbGu}utnD@m^H@gwWzf~_Ih}M_`AKvto[_`@_`A~gr[j[y|f"
    },
    "mass": 409.186273143011,
    "monoisotopicMass": 409.01885101411,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 9
    }, {
      "symbol": "P",
      "number": 2
    }]
  }, {
    "symbol": "Dtdp",
    "name": "Desoxythymidine diphosphate diradical",
    "mf": "C10H14N2O10P2",
    "kind": "DNApp",
    "oneLetter": "T",
    "ocl": {
      "value": "fe{Ph`IQaPUg^Ct\\p^JoCO`uqrIQQYIQRJKEJQTxdmUUUSSMTsVDKBy@@",
      "coordinates": "!BIi[Rx{_grZOSXa}_]^HMQbGuMcqLX@m^H@gwWKB__t]Q_`@SFALX_`@_`A~w}D"
    },
    "mass": 384.1735091870477,
    "monoisotopicMass": 384.01236865262,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 2
    }]
  }, {
    "symbol": "Dudp",
    "name": "Desoxyuridine diphosphate diradical",
    "mf": "C9H12N2O10P2",
    "kind": "DNApp",
    "oneLetter": "U",
    "ocl": {
      "value": "fikPh`IQaPUg^Bwhp^JoCOaUqrIQQYIQRJKGRJgDejjjjZYjYkBEa\\`@",
      "coordinates": "!BIi[Rx{_grZOSXa}_]^H@mQbGuMcqLX@m^H@gwWKB__t]Q_`@SFGx@Owx@_mQ"
    },
    "mass": 370.1468917822009,
    "monoisotopicMass": 369.99671858816,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 2
    }]
  }, {
    "symbol": "Datp",
    "name": "Desoxyadenosine triphosphate diradical",
    "mf": "C10H14N5O11P3",
    "kind": "DNAppp",
    "oneLetter": "A",
    "ocl": {
      "value": "eohZMJ@I@diehJAKGOFnakg`OESpr|Mo@yqrIQQYIQRJKYZQKVRcbIJjZjjjihFAhjZcAAXKb@@",
      "coordinates": "!BIi[Rx{_grZOSXa}_]^H@mQbGu}utnDM^H@gwWzf~_Ih}M_`AKvto[@hcW@`A~grZ_Igx@_`@@_c}~"
    },
    "mass": 473.16678574412344,
    "monoisotopicMass": 472.9902672839,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 11
    }, {
      "symbol": "P",
      "number": 3
    }]
  }, {
    "symbol": "Dctp",
    "name": "Desoxycytidine triphosphate diradical",
    "mf": "C9H14N3O12P3",
    "kind": "DNAppp",
    "oneLetter": "C",
    "ocl": {
      "value": "fkopZ`IQGB`kN|Fk^{NCqUxY|I~BwGHeEEdeEHhl]HlYJ\\RVjjjiifVjfkBEa\\`@",
      "coordinates": "!BIi[Rx{_grZOSXa}_]^H@mQbGuMcqLX@m^H@gwWKB__t]Q_`@SFOrHupH@_mQ_`A~@@A~Owx"
    },
    "mass": 449.14204834881485,
    "monoisotopicMass": 448.97903389461004,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 12
    }, {
      "symbol": "P",
      "number": 3
    }]
  }, {
    "symbol": "Dgtp",
    "name": "Desoxyguanosine triphosphate diradical",
    "mf": "C10H14N5O12P3",
    "kind": "DNAppp",
    "oneLetter": "G",
    "ocl": {
      "value": "e`TZCJ@I@diehJAKGOFnamgo`OESpr|CoByqrIQQYIQRJKYZQQYrT\\QIUSUUUUMRuMLtuVBBpWD@@",
      "coordinates": "!BIi[Rx{_grZOSXa}_]^H@mQbGu}utnD@m^H@gwWzf~_Ih}M_`AKvto[@hcW@`A~gr[j[y|f_`A~@@A~Owx"
    },
    "mass": 489.16619066844174,
    "monoisotopicMass": 488.98518190347005,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 12
    }, {
      "symbol": "P",
      "number": 3
    }]
  }, {
    "symbol": "Dttp",
    "name": "Desoxythymidine triphosphate diradical",
    "mf": "C10H15N2O13P3",
    "kind": "DNAppp",
    "oneLetter": "T",
    "ocl": {
      "value": "fgQZ`IQaPUg^BwhygnCqUxY|E~FwGHeEEdeEHhlTiDSISbRuUUUMLuMMMVDKBy@@",
      "coordinates": "!BIi[Rx{_grZOSXa}_]^H@mQbGuMcqLX@m^H@gwWKB__t]Q_`@SFALXHcW@`A~w}E~@Gx@@Gx_`"
    },
    "mass": 464.15342671247834,
    "monoisotopicMass": 463.97869954198,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 15
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 13
    }, {
      "symbol": "P",
      "number": 3
    }]
  }, {
    "symbol": "Dutp",
    "name": "Desoxyuridine triphosphate diradical",
    "mf": "C9H13N2O13P3",
    "kind": "DNAppp",
    "oneLetter": "U",
    "ocl": {
      "value": "fkoQZ`IQaPUg^CUoQ{NCqUxY|I~BwGHeEEdeEHhl]HlYJ\\RVjjjiiffffkBEa\\`@",
      "coordinates": "!BIi[Rx{_grZOSXa}_]^H@mQbGuMcqLX@m^H@gwWKB__t]Q_`@SFOrHupH@_mQ_`A~@@A~Owx"
    },
    "mass": 450.1268093076315,
    "monoisotopicMass": 449.96304947752,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 13
    }, {
      "symbol": "P",
      "number": 3
    }]
  }, {
    "symbol": "Dade",
    "name": "Desoxyadenosine diradical",
    "mf": "C10H11N5O2",
    "kind": "DNA",
    "oneLetter": "A",
    "ocl": {
      "value": "fluha@IF]ELJ@|QNJRsN|rntyYpXuUUTBATXPlKd@@",
      "coordinates": "!B\\KqpQARcg|T^|X@@Id`zeHo@Ie}]vaLcg|T^qAMDDvN_xy"
    },
    "mass": 233.22703316783156,
    "monoisotopicMass": 233.09127461582,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Dcyt",
    "name": "Desoxycytidine diradical",
    "mf": "C9H11N3O3",
    "kind": "DNA",
    "oneLetter": "C",
    "ocl": {
      "value": "fhiqa@IVCBa`^HgEIYg^Y~gG^jjjiejpaXWH@",
      "coordinates": "!BBOpH_UARcc}TN|Y@PIe`zeIO@MDSIrpXTd}RSqLgTd|"
    },
    "mass": 209.202295772523,
    "monoisotopicMass": 209.08004122653,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 3
    }]
  }, {
    "symbol": "Dgua",
    "name": "Desoxyguanosine diradical",
    "mf": "C10H11N5O3",
    "kind": "DNA",
    "oneLetter": "G",
    "ocl": {
      "value": "fbmia@IF]ELJYAxb\\Tef]ye^Z\\lxLZjjjeZfkBEa\\`@",
      "coordinates": "!B\\KqpQARcg|T^|X@@Id`zeHo@Ie}]vaLcg|T^qAMDDvN_vaLcg|"
    },
    "mass": 249.22643809214986,
    "monoisotopicMass": 249.08618923539,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 3
    }]
  }, {
    "symbol": "Dthy",
    "name": "Desoxythymidine diradical",
    "mf": "C10H12N2O4",
    "kind": "DNA",
    "oneLetter": "T",
    "ocl": {
      "value": "fdyPQ@IVaPtP^HgEIYg^YuiqwjjjjYikBEa\\`@",
      "coordinates": "!BBOpH_UARcc}TN|Y@PIe`zeIO@MDSIrpXTd}RSqLgDr]RSp"
    },
    "mass": 224.2136741361865,
    "monoisotopicMass": 224.07970687390002,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 4
    }]
  }, {
    "symbol": "Dura",
    "name": "Desoxyuridine diradical",
    "mf": "C9H10N2O4",
    "kind": "DNA",
    "oneLetter": "U",
    "ocl": {
      "value": "fhiPQ@IVaPpP^HgEIYg^Y~gG^jjjifZpaXWH@",
      "coordinates": "!BBOpH_UARcc}TN|Y@PIe`zeIO@MDSIrpXTd}RSqLgTd|"
    },
    "mass": 210.1870567313397,
    "monoisotopicMass": 210.06405680944,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 10
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 4
    }]
  }, {
    "symbol": "Amp",
    "name": "Adenosine monophosphate diradical",
    "mf": "C10H12N5O6P",
    "kind": "RNAp",
    "oneLetter": "A",
    "alternativeOneLetter": "α",
    "ocl": {
      "value": "fakhs@INBwlJ\\TgHOFwaEqrIQQSYQJIRIMLyxMVuUUUPLpEPQDqBId@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtPCQ@@"
    },
    "mass": 329.20635561758047,
    "monoisotopicMass": 329.05252012475,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 6
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Cmp",
    "name": "Cytidine monophosphate diradical",
    "mf": "C9H12N3O7P",
    "kind": "RNAp",
    "oneLetter": "C",
    "alternativeOneLetter": "ς",
    "ocl": {
      "value": "ff}qs@I^kBgENSdGc[pbxyDhhilheDiLv\\BVjjjjYfZbHfHQL`@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFDuP"
    },
    "mass": 305.1816182222719,
    "monoisotopicMass": 305.04128673546,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Gmp",
    "name": "Guanosine monophosphate diradical",
    "mf": "C10H12N5O7P",
    "kind": "RNAp",
    "oneLetter": "G",
    "alternativeOneLetter": "γ",
    "ocl": {
      "value": "fi{is@INBwlJ\\TgHp^MoBKcdRbbfrbTRdR\\SN^CUmUUUUKMSMQDSDHfP@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtP@gD}D@"
    },
    "mass": 345.20576054189877,
    "monoisotopicMass": 345.04743474432,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Tmp",
    "name": "Thymidine monophosphate diradical",
    "mf": "C10H13N2O8P",
    "kind": "RNAp",
    "oneLetter": "T",
    "alternativeOneLetter": "τ",
    "ocl": {
      "value": "fncPK@I^aSbgIrtGc[pbxyDhhilheDiLjs`RuUUUSLuMDQLPbY@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@"
    },
    "mass": 320.1929965859354,
    "monoisotopicMass": 320.04095238282997,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Ump",
    "name": "Uridine monophosphate diradical",
    "mf": "C9H11N2O8P",
    "kind": "RNAp",
    "oneLetter": "U",
    "alternativeOneLetter": "υ",
    "ocl": {
      "value": "ff}PK@I^aSbgIsTGc[pbxyDhhilheDiLv\\BVjjjjYffbHfHQL`@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFDuP"
    },
    "mass": 306.1663791810886,
    "monoisotopicMass": 306.02530231837,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Rmp",
    "name": "Ribose monophosphate diradical",
    "mf": "C5H7O6P",
    "kind": "RNAp",
    "ocl": {
      "value": "d@",
      "coordinates": ""
    },
    "mass": 194.07945630639637,
    "monoisotopicMass": 193.99802494145,
    "unsaturation": 4,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 7
    }, {
      "symbol": "O",
      "number": 6
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Adp",
    "name": "Adenosine diphosphate diradical",
    "mf": "C10H13N5O9P2",
    "kind": "RNApp",
    "oneLetter": "A",
    "ocl": {
      "value": "fcoiH`INCt\\J\\UENU{Axv|F~DwGHeEEMeDheHd\\eHsg`u{UUUU@mAEMPQDqBId@@",
      "coordinates": "!BvuPfpDnDtEK_t_rHtXBH_TwPbOr_IorHbGtgD}F@RxPBuxc|_]^OTh}R_`CQ`MF@_`@_`A~"
    },
    "mass": 409.186273143011,
    "monoisotopicMass": 409.01885101411,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 9
    }, {
      "symbol": "P",
      "number": 2
    }]
  }, {
    "symbol": "Cdp",
    "name": "Cytidine diphosphate diradical",
    "mf": "C9H13N3O10P2",
    "kind": "RNApp",
    "oneLetter": "C",
    "ocl": {
      "value": "fe{ph`I^[BgENbgHy`|[^E_CkcdRbbfrbTRdqrdYpIZjjjieijZbHfHQL`@",
      "coordinates": "!BvuPfpDnDtEK_t_rHtXBH_TwPb@K_cbpXbKtSItwPS]@Bux`Bo]\\lFGx@S]A~@C}~@Gx"
    },
    "mass": 385.1615357477025,
    "monoisotopicMass": 385.00761762482,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 2
    }]
  }, {
    "symbol": "Gdp",
    "name": "Guanosine diphosphate diradical",
    "mf": "C10H13N5O10P2",
    "kind": "RNApp",
    "oneLetter": "G",
    "ocl": {
      "value": "fkhh`INCt\\J\\UENY{NCqmxM|EnNQJJJ[JIQJQHzIRLyxM^uUUUTkUSLuQDSDHfP@",
      "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@J_I`JHbGtgD}F@RxPBux`B_]^OTh}R_`CQ`B\\StXA~@C}~@Gx"
    },
    "mass": 425.1856780673293,
    "monoisotopicMass": 425.01376563368,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 2
    }]
  }, {
    "symbol": "Tdp",
    "name": "Thymidine diphosphate diradical",
    "mf": "C10H14N2O11P2",
    "kind": "RNApp",
    "oneLetter": "T",
    "ocl": {
      "value": "fmgQh`I^aSbgQSglu`|[^C_@[bdls^rruo}LxDmUUUTruTsTQDqBId@@",
      "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPbOs_cbpXbGtSItwPS]@Bux`B_]\\lFBpX_`AMtGx@Owx@_`"
    },
    "mass": 400.172914111366,
    "monoisotopicMass": 400.00728327219,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 11
    }, {
      "symbol": "P",
      "number": 2
    }]
  }, {
    "symbol": "Udp",
    "name": "Uridine diphosphate diradical",
    "mf": "C9H12N2O11P2",
    "kind": "RNApp",
    "oneLetter": "U",
    "ocl": {
      "value": "fe{Qh`I^aSbgQSehy`|[^E_CkcdRbbfrbTRdqrdYpIZjjjiejfZbHfHQL`@",
      "coordinates": "!BvuPfpDnDtEK_t_rHtXBH_TwPb@K_cbpXbKtSItwPS]@Bux`Bo]\\lFGx@S]A~@C}~@Gx"
    },
    "mass": 386.14629670651925,
    "monoisotopicMass": 385.99163320773005,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 11
    }, {
      "symbol": "P",
      "number": 2
    }]
  }, {
    "symbol": "Atp",
    "name": "Adenosine triphosphate diradical",
    "mf": "C10H14N5O12P3",
    "kind": "RNAppp",
    "oneLetter": "A",
    "ocl": {
      "value": "e`TZCJ@IG@nahJNEHdliemgo`OFspZ|CoByqrIQQSYQJIRIGIRWRL\\^AU]UUUUPKPQMTuABDpaBX`@",
      "coordinates": "!BvuPfpDnDtEK_t_rHtXBH_TwPbOr_IorHbGtgD}F@RxS|uxc|_]^OTh}R_`CQ`MF@@hcW@A~_`A~@@A~Owx"
    },
    "mass": 489.16619066844174,
    "monoisotopicMass": 488.98518190347005,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 12
    }, {
      "symbol": "P",
      "number": 3
    }]
  }, {
    "symbol": "Ctp",
    "name": "Cytidine triphosphate diradical",
    "mf": "C9H14N3O13P3",
    "kind": "RNAppp",
    "oneLetter": "C",
    "ocl": {
      "value": "fgqZ`I^[BgENbgOQsO\\Gc[pkxK|MnNQJJJ[JIQJSGJPzQg@ejjjjfVffjZbHfHQL`@",
      "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@K_cbpXbGtSItwPS]C|ux`B_]\\lFGx@S]@BbM\\B@Gy~@Gx@@Gx_`"
    },
    "mass": 465.1414532731331,
    "monoisotopicMass": 464.97394851418,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 13
    }, {
      "symbol": "P",
      "number": 3
    }]
  }, {
    "symbol": "Gtp",
    "name": "Guanosine triphosphate diradical",
    "mf": "C10H14N5O13P3",
    "kind": "RNAppp",
    "oneLetter": "G",
    "ocl": {
      "value": "eh\\ZKJ@IG@nahJNEHdliemco`POFspZ|KoAyqrIQQSYQJIRIGQJQzQccpJkjjjjjeZjYZijbDIaBDq@@",
      "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@J_I`JHbGtgD}F@RxPBux`B_]^OTh}R_`CQ`B\\StX@BbM_|@Gy~@Gx@@Gx_`"
    },
    "mass": 505.16559559276,
    "monoisotopicMass": 504.98009652304,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 13
    }, {
      "symbol": "P",
      "number": 3
    }]
  }, {
    "symbol": "Ttp",
    "name": "Thymidine triphosphate diradical",
    "mf": "C10H15N2O14P3",
    "kind": "RNAppp",
    "oneLetter": "T",
    "ocl": {
      "value": "eo`TGJ@IOHJNEGHdlnaiekg`OFspZ|Mo@yqrIQQSYQJIRY[ZPzQc`HjjjjjYZjVjZbDIaBDq@@",
      "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@K_cbpXbGtSItwPS]@Bux`B_]\\lFBpX_`AMt@JHupH@_gx@_`@@_c}~"
    },
    "mass": 480.15283163679663,
    "monoisotopicMass": 479.97361416155,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 15
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 14
    }, {
      "symbol": "P",
      "number": 3
    }]
  }, {
    "symbol": "Utp",
    "name": "Uridine triphosphate diradical",
    "mf": "C9H13N2O14P3",
    "kind": "RNAppp",
    "oneLetter": "U",
    "ocl": {
      "value": "fgPz`I^aSbgQSeoQsO\\Gc[pkxK|MnNQJJJ[JIQJSGJPzQg@ejjjjfVjVjZbHfHQL`@",
      "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@K_cbpXbGtSItwPS]C|ux`B_]\\lFGx@S]@BbM\\B@Gy~@Gx@@Gx_`"
    },
    "mass": 466.12621423194986,
    "monoisotopicMass": 465.95796409709004,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 14
    }, {
      "symbol": "P",
      "number": 3
    }]
  }, {
    "symbol": "Ade",
    "name": "Adenosine diradical",
    "mf": "C10H11N5O3",
    "kind": "RNA",
    "oneLetter": "A",
    "ocl": {
      "value": "fbmia@IV|gLJ\\Axj\\Tef[vyWV\\]zJZjjj`PJ`bIbDSH@",
      "coordinates": "!BBOpH_UARccFPEP{PId{RpBN[~i|BEP{iVA@fUARU@QTADBYPId"
    },
    "mass": 249.22643809214986,
    "monoisotopicMass": 249.08618923539,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 3
    }]
  }, {
    "symbol": "Cyt",
    "name": "Cytidine diradical",
    "mf": "C9H11N3O4",
    "kind": "RNA",
    "oneLetter": "C",
    "ocl": {
      "value": "fdypQ@INcBgK@|UNJRsM{\\~sg`uUUULmQDSDHfP@",
      "coordinates": "!BBOpH_UARccFPEP{PId{RpBN[~iRTBpgDq`@c`BNKB\\@c`"
    },
    "mass": 225.20170069684127,
    "monoisotopicMass": 225.0749558461,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 4
    }]
  }, {
    "symbol": "Gua",
    "name": "Guanosine diradical",
    "mf": "C10H11N5O4",
    "kind": "RNA",
    "oneLetter": "G",
    "ocl": {
      "value": "fj}hQ@IV|gLJ\\JCqTxiKLwmroKNN}EMUUUTkTuDQLPbY@@",
      "coordinates": "!BBOpH_UARccFPEP{PId{RpBN[~k|BEP{iVA@fUARU@QTADBYiVA@fP"
    },
    "mass": 265.22584301646816,
    "monoisotopicMass": 265.08110385496,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 4
    }]
  }, {
    "symbol": "Thy",
    "name": "Thymidine diradical",
    "mf": "C10H12N2O5",
    "kind": "RNA",
    "oneLetter": "T",
    "ocl": {
      "value": "fleQQ@INaSed`|UNJRsM{\\zlyxMUUUSMMDQLPbY@@",
      "coordinates": "!BBOpH_UARccFPEP{PId{RpBN[~iRTBpgDq`@c`BNKB\\lIpBN"
    },
    "mass": 240.21307906050478,
    "monoisotopicMass": 240.07462149347,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 5
    }]
  }, {
    "symbol": "Ura",
    "name": "Uridine diradical",
    "mf": "C9H10N2O5",
    "kind": "RNA",
    "oneLetter": "U",
    "ocl": {
      "value": "fdyQQ@INaSeh`|UNJRsM{\\~sg`uUUULsQDSDHfP@",
      "coordinates": "!BBOpH_UARccFPEP{PId{RpBN[~iRTBpgDq`@c`BNKB\\@c`"
    },
    "mass": 226.18646165565798,
    "monoisotopicMass": 226.05897142901,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 10
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 5
    }]
  }, {
    "symbol": "Dam",
    "name": "1,2′-O-dimethyladenosine monophosphate diradical 01A",
    "mf": "C12H16N5O6P",
    "kind": "NucleotideP",
    "oneLetter": "œ",
    "ocl": {
      "value": "feghs@E^ct\\J\\udhOEw`eqrIQQQKZIQJQIiLxFK^uUUUUKLtuQDSDHfP@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuKQ@MD@SuH"
    },
    "mass": 357.2595904272741,
    "monoisotopicMass": 357.08382025367,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 6
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Dgm",
    "name": "1,2′-O-dimethylguanosine monophosphate diradical 01G",
    "mf": "C12H16N5O7P",
    "kind": "NucleotideP",
    "oneLetter": "ε",
    "ocl": {
      "value": "fmwis@E^ct\\J\\udlp^KoAKcdRbbbVtRbTbSbSNAbwmUUUURsMSUDQLPbY@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuKQ@B\\StPAOT`"
    },
    "mass": 373.2589953515923,
    "monoisotopicMass": 373.07873487324,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Dim",
    "name": "1,2′-O-dimethylinosine monophosphate diradical 019A",
    "mf": "C12O7N4H15P",
    "kind": "NucleotideP",
    "oneLetter": "ξ",
    "ocl": {
      "value": "fegIs@E^cvENZrTXOEw`eqrIQQQKZIQJQIiLxFK^uUUUUKLtuQDSDHfP@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuKQ@MD@SuH"
    },
    "mass": 358.2443513860907,
    "monoisotopicMass": 358.06783583658,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 15
    }, {
      "symbol": "N",
      "number": 4
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Tia",
    "name": "2- methylthiomethylenethio-N6-isopentenyl-adenosine monophosphate diradical",
    "mf": "C17H24N5O6PS2",
    "kind": "NucleotideP",
    "oneLetter": "£",
    "ocl": {
      "value": "eh\\ZFJ@IG@nahJNEDl`OFspb\\V`cXHrIQQSYQJIRINIYIKQccpJkjjjjjAfBJjfjBDIaBDq@@",
      "coordinates": "!BpBYTvxBNFY|bEJObGvOS\\@Yt]~DUEJOctu~@Ha}`HzOSTwPTh~H@hc|_`BH_Xa}b@JH@gx@bGvH@h`B_`BH_P"
    },
    "mass": 489.50637075565066,
    "monoisotopicMass": 489.09056286031,
    "unsaturation": 16,
    "elements": [{
      "symbol": "C",
      "number": 17
    }, {
      "symbol": "H",
      "number": 24
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 6
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 2
    }]
  }, {
    "symbol": "Mhc",
    "name": "2′‐O‐Methyl-5-hydroxymethylcytidine monophosphate diradical",
    "mf": "C11H16N3O8P",
    "kind": "NucleotideP",
    "oneLetter": "¡",
    "ocl": {
      "value": "fikpK@EA{BgM^rTXOEw`eqrIQQQKZIQJSJigHujjjjifYjkBHQL`@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@HoTuOSU@HC~NKA`HoQLgSUAMT@a}oS_|BBpXKAaMT@CQ"
    },
    "mass": 349.2342579562838,
    "monoisotopicMass": 349.06750148395,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Odg",
    "name": "N2,2′-O-dimethylguanosine monophosphate diradical 02G",
    "mf": "C12H16N5O7P",
    "kind": "NucleotideP",
    "oneLetter": "γ",
    "ocl": {
      "value": "fmwis@E^ct\\J\\udlp^KoAKcdRbbbVtRbTbSbsNAbwmUUUURsMSUDQLPbY@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuKQ@B\\StPAOT`"
    },
    "mass": 373.2589953515923,
    "monoisotopicMass": 373.07873487324,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Ntg",
    "name": "N2,N2,2′-O-trimethylguanosine monophosphate diradical 022G",
    "mf": "C13H18N5O7P",
    "kind": "NucleotideP",
    "oneLetter": "|",
    "ocl": {
      "value": "fcois@E^ct\\J\\udlp^KoAKcdRbbbVtRbTbSbTYpLVcjjjjjVYjZjHbXaDr@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuHgD}D@tPBNOt}R"
    },
    "mass": 387.2856127564392,
    "monoisotopicMass": 387.0943849377,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 13
    }, {
      "symbol": "H",
      "number": 18
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Otg",
    "name": "N2,7,2′-O-trimethylguanosine monophosphate diradical 027G",
    "mf": "C13H20N5O7P",
    "kind": "NucleotideP",
    "oneLetter": "æ",
    "ocl": {
      "value": "fcoisBE^bN\\J\\udjp^KoAKcFU}dRbbbVtRbTbRlQYpLVcjjjjjVYjjjHbXaDr@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuKFPMD@IqOQ@D}R"
    },
    "mass": 389.30149426455074,
    "monoisotopicMass": 389.11003500216,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 13
    }, {
      "symbol": "H",
      "number": 20
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Rya",
    "name": "2′-O-ribosyladenosine monophosphate diradical 00A",
    "mf": "C15H20N5O9P",
    "kind": "NucleotideP",
    "oneLetter": "^",
    "ocl": {
      "value": "e`\\ZIL@DaegobFAIO@hlm`OGSp\\\\\\bbbfrRbdTT\\rbRQUCDQTrusuUUUUMUU@pET@@@",
      "coordinates": "!BIlAKaMARw}DBbMF@bGuMtHc|KAbH_ZU`@GzH_WwW@h`XKFjKB_jXB\\SiVA`zmG_Irp_hQKctvOSR\\lIrp"
    },
    "mass": 445.3217759066577,
    "monoisotopicMass": 445.09986424130005,
    "unsaturation": 16,
    "elements": [{
      "symbol": "C",
      "number": 15
    }, {
      "symbol": "H",
      "number": 20
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 9
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Ryg",
    "name": "2′-O-ribosylguanosine monophosphate diradical 00G",
    "mf": "C15H20N5O10P",
    "kind": "NucleotideP",
    "oneLetter": "ℑ",
    "ocl": {
      "value": "ehRZEL@DaegobFAIO@hlm`POGSp\\\\\\bbbfrRbdTT\\rbRQbhXbJfVn^jjjjijjjVZfj@@@",
      "coordinates": "!BIlAKaMARw}DBbMF@bGuMtH`BKAbH_ZU`@GzH_WwW@h`XKFjKB_jXB\\SiVA`zmG_Irp_hQKctvOSR\\lt]|gK@"
    },
    "mass": 461.321180830976,
    "monoisotopicMass": 461.09477886087,
    "unsaturation": 16,
    "elements": [{
      "symbol": "C",
      "number": 15
    }, {
      "symbol": "H",
      "number": 20
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Tmu",
    "name": "2-thio-2′-O-methyluridine monophosphate diradical 02U",
    "mf": "C10H13N2O7PS",
    "kind": "NucleotideP",
    "oneLetter": "∏",
    "ocl": {
      "value": "fncQp`EAaSfleZCq]x^BDnNQJJJI[QJIRYlyFmUUUULsSQDSDHfP@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXSU@"
    },
    "mass": 336.25837906774416,
    "monoisotopicMass": 336.01810893766003,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Dmut",
    "name": "3,2′-O-dimethyluridine monophosphate diradical 03U",
    "mf": "C11H15N2O8P",
    "kind": "NucleotideP",
    "oneLetter": "σ",
    "ocl": {
      "value": "fasPK@EAaSfoYKtGb{pRxyDhhhemDheIhv\\cVjjjjfYjZHbXaDr@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXOxyMT@"
    },
    "mass": 334.2196139907822,
    "monoisotopicMass": 334.05660244729,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 15
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Amc",
    "name": "N4-acetyl-2′-O-methylcytidine monophosphate diradical 042C",
    "mf": "C12H16N3O8P",
    "kind": "NucleotideP",
    "oneLetter": "ℵ",
    "ocl": {
      "value": "fe{pK@EA[BgM^rTXOEw`eqrIQQQKZIQJSMJLyFmUUUULsMMQDSDHfP@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXSU@caLgSU@"
    },
    "mass": 361.244993853019,
    "monoisotopicMass": 361.06750148395,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Tmc",
    "name": "N4,N4,2′-O-trimethylcytidine monophosphate diradical 044C",
    "mf": "C12H18N3O7P",
    "kind": "NucleotideP",
    "oneLetter": "β",
    "ocl": {
      "value": "fikqs@EA[BgM^rTGb{pRxyDhhhemDheIfhsdZuUUUTsLuTQDqBId@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXSU@cbpX"
    },
    "mass": 347.2614704368123,
    "monoisotopicMass": 347.08823692884005,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 18
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Dcy",
    "name": "N4,2′-O-dimethylcytidine monophosphate diradical 04C",
    "mf": "C11H16N3O7P",
    "kind": "NucleotideP",
    "oneLetter": "λ",
    "ocl": {
      "value": "fasqs@EA[BgM^rTGb{pRxyDhhhemDheIff\\cVjjjjfYfjHbXaDr@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXSU@lF@"
    },
    "mass": 333.23485303196554,
    "monoisotopicMass": 333.07258686438,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Muo",
    "name": "2′-O-methyluridine 5-oxyacetic acid methyl ester monophosphate diradical 0503U",
    "mf": "C13H17N2O11P",
    "kind": "NucleotideP",
    "oneLetter": "Ͽ",
    "ocl": {
      "value": "fkoQk@EAaSfoYJwj}`|W^BWGHeEEDmheDiLjlif\\cVjjjjfYjZZhbIbDSH@",
      "coordinates": "!BKAb@tURD@m\\YpMAMpBYMcvjbOplIwx@bGuMc}\\Bb@JH@dvOcuKPSXa}bGvHH`BbGu~Oxc|bGt"
    },
    "mass": 408.25518206531905,
    "monoisotopicMass": 408.05699637046,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 13
    }, {
      "symbol": "H",
      "number": 17
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 11
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Xmu",
    "name": "5-carboxymethylaminomethyl-2′-O-methyluridine monophosphate diradical 051U",
    "mf": "C13H18N3O10P",
    "kind": "NucleotideP",
    "oneLetter": ")",
    "ocl": {
      "value": "fkopk@EAGBgM^rWns`|W^BWGHeEEDmheDiLjleF\\cVjjjjfYjZfhbIbDSH@",
      "coordinates": "!BKAb@tURD@m\\YpMAMpBYMcvjb@HlIwx@bGuMc}\\Bb@JH@dvOcuKPSXa}bGvH@h`BbGvH@gx@bKt"
    },
    "mass": 407.2704211065024,
    "monoisotopicMass": 407.07298078755,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 13
    }, {
      "symbol": "H",
      "number": 18
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Mmu",
    "name": "5-methoxycarbonylmethyl-2′-O-methyluridine monophosphate diradical 0521U",
    "mf": "C13H17N2O10P",
    "kind": "NucleotideP",
    "oneLetter": "∩",
    "ocl": {
      "value": "fcwPk@EAaSfoYKvZp^KoAKcdRbbbVtRbTfUVfYrMZjjjjYfifjHbXaDr@@",
      "coordinates": "!BS]@lFJU`@Gyoza`lzf@lI}mK_`B@cm\\Bb@HlI}]}_`A~@BpgIqLXKH`Bb@I~@Ha}_c~HHa}"
    },
    "mass": 392.2557771410008,
    "monoisotopicMass": 392.06208175089,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 13
    }, {
      "symbol": "H",
      "number": 17
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Cue",
    "name": "5-(carboxyhydroxymethyl)-2′-O-methyluridine methyl ester monophosphate diradical 0522U",
    "mf": "C13H17N2O11P",
    "kind": "NucleotideP",
    "oneLetter": "∩",
    "ocl": {
      "value": "fkoQk@EAaSfoYKtZ}`|W^BWGHeEEDmheDiLjhYf\\cVjjjjfYjZZhbIbDSH@",
      "coordinates": "!BS]@lFJU`@Gyoza`lzf@lI}mK_`B@cm\\Bb@HlI}]}_`A~@BpgIqLXKH`Bb@I~@Gx@bGu~Oxc|bGt"
    },
    "mass": 408.25518206531905,
    "monoisotopicMass": 408.05699637046,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 13
    }, {
      "symbol": "H",
      "number": 17
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 11
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Cyu",
    "name": "5-carbamoylmethyl-2′-O-methyluridine monophosphate diradical 053U",
    "mf": "C12H16N3O9P",
    "kind": "NucleotideP",
    "oneLetter": "~",
    "ocl": {
      "value": "fmgqK@EAWBgM^rWlp^KoAKcdRbbbVtRbTfUVcNQkUUUUSLuLuDQLPbY@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXKAaMTDuPOxxlF@"
    },
    "mass": 377.24439877733727,
    "monoisotopicMass": 377.06241610352,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 9
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Ipu",
    "name": "5-(isopentenylaminomethyl)-2′-O-methyluridine monophosphate diradical 0583U",
    "mf": "C16H24N3O8P",
    "kind": "NucleotideP",
    "oneLetter": "¼",
    "ocl": {
      "value": "fgpK@EAGBgM^rWhOEw`eqrIQQQKZIQJSJkIJLyFmUUUULsTuMTQDqBId@@",
      "coordinates": "!BS]@lFJU`@Gyoza`lzf@lI}mK_`B@cm\\Bb@HlI}]}_`A~@BpgIqLXKH`Bb@I~@Ha}b@JH_Xc|_`BH_P"
    },
    "mass": 417.35146347240624,
    "monoisotopicMass": 417.13010174179004,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 16
    }, {
      "symbol": "H",
      "number": 24
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Mcy",
    "name": "5,2′-O-dimethylcytidine monophosphate diradical monophosphate diradical 05C",
    "mf": "C11H16N3O7P",
    "kind": "NucleotideP",
    "oneLetter": "τ",
    "ocl": {
      "value": "fasqs@EA{BgM^rTGb{pRxyDhhhemDheIeV\\cVjjjjfYfjHbXaDr@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXKAaMT@"
    },
    "mass": 333.23485303196554,
    "monoisotopicMass": 333.07258686438,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Dmuf",
    "name": "5,2′-O-dimethyluridine monophosphate diradical 05U",
    "mf": "C11H15N2O8P",
    "kind": "NucleotideP",
    "oneLetter": "\\",
    "ocl": {
      "value": "fasPK@EAaSfoYKtGb{pRxyDhhhemDheIeV\\cVjjjjfYjZHbXaDr@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXKAaMT@"
    },
    "mass": 334.2196139907822,
    "monoisotopicMass": 334.05660244729,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 15
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Tma",
    "name": "N6,N6,2′-O-trimethyladenosine monophosphate diradical 066A",
    "mf": "C13H18N5O6P",
    "kind": "NucleotideP",
    "oneLetter": "η",
    "ocl": {
      "value": "fmwhs@E^ct\\J\\udhOEw`eqrIQQQKZIQJQIkQg@q[vjjjj`Y`JjBHfHQL`@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuKQ@MD@SuHXK@"
    },
    "mass": 371.2862078321209,
    "monoisotopicMass": 371.09947031813005,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 13
    }, {
      "symbol": "H",
      "number": 18
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 6
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Nda",
    "name": "N6,2′-O-dimethyladenosine monophosphate diradical 06A",
    "mf": "C12H16N5O6P",
    "kind": "NucleotideP",
    "oneLetter": "χ",
    "ocl": {
      "value": "feghs@E^ct\\J\\udhOEw`eqrIQQQKZIQJQIkLxFK^uUUUTCLAUADSDHfP@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuKQ@MD@FBp"
    },
    "mass": 357.2595904272741,
    "monoisotopicMass": 357.08382025367,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 6
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Fmc",
    "name": "5-formyl-2′-O-methylcytidine monophosphate diradical 071C",
    "mf": "C10H12N3O8P",
    "kind": "NucleotideP",
    "oneLetter": "°",
    "ocl": {
      "value": "faspK@I^[BgENSghOFwaEqrIQQSYQJIRYULxDmUUUTsLttQDqBId@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@@tP"
    },
    "mass": 333.1917590433254,
    "monoisotopicMass": 333.03620135502996,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Omi",
    "name": "2′-O-methylinosine monophosphate diradical 09A",
    "mf": "C11H13N4O7P",
    "kind": "NucleotideP",
    "oneLetter": "≤",
    "ocl": {
      "value": "fi{Is@E^cvENZrTXOEw`eqrIQQQKZIQJQIig@q[vjjjjiYffhbIbDSH@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuKQ@MD@"
    },
    "mass": 344.21773398124395,
    "monoisotopicMass": 344.05218577211997,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 4
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Opu",
    "name": "2′-O-methylpseudouridine monophosphate diradical 09U",
    "mf": "C10H13N2O8P",
    "kind": "NucleotideP",
    "oneLetter": "Z",
    "ocl": {
      "value": "fncPK@@qaSfoYJtGb{pRxyDhhhemDheIfsdZuUUULuMMDQLPbY@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXSU@"
    },
    "mass": 320.1929965859354,
    "monoisotopicMass": 320.04095238282997,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Oma",
    "name": "2′-O-methyladenosine monophosphate diradical 0A",
    "mf": "C11H14N5O6P",
    "kind": "NucleotideP",
    "oneLetter": ":",
    "ocl": {
      "value": "fi{hs@E^ct\\J\\udhOEw`eqrIQQQKZIQJQIig@q[vjjjj`Y`J`bIbDSH@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuKQ@MD@"
    },
    "mass": 343.2329730224273,
    "monoisotopicMass": 343.06817018921,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 6
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Omc",
    "name": "2′-O-methylcytidine monophosphate diradical 0C",
    "mf": "C10H14N3O7P",
    "kind": "NucleotideP",
    "oneLetter": "B",
    "ocl": {
      "value": "fncqs@EA[BgM^rTGb{pRxyDhhhemDheIfsdZuUUUTsLuDQLPbY@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXSU@"
    },
    "mass": 319.2082356271187,
    "monoisotopicMass": 319.05693679992004,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Omg",
    "name": "2′-O-methylguanosine monophosphate diradical 0G",
    "mf": "C11H14N5O7P",
    "kind": "NucleotideP",
    "oneLetter": "#",
    "ocl": {
      "value": "fegis@E^ct\\J\\udlp^KoAKcdRbbbVtRbTbSbYpLV}jjjjjVYjZbHfHQL`@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@HoTuOSU@HEF@`H_R\\StPAKA@a}_S_|BD}RSuKQ@B\\StP@"
    },
    "mass": 359.23237794674554,
    "monoisotopicMass": 359.06308480878,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Omu",
    "name": "2′-O-methyluridinemonophosphate diradical 0U",
    "mf": "C10H13N2O8P",
    "kind": "NucleotideP",
    "oneLetter": "J",
    "ocl": {
      "value": "fncPK@EAaSfoYJtGb{pRxyDhhhemDheIfsdZuUUUTsMMDQLPbY@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuOSU@HC~NKA`H_QLgSUAMT@a}_S_|BBpXSU@"
    },
    "mass": 320.1929965859354,
    "monoisotopicMass": 320.04095238282997,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Cdg",
    "name": "7-cyano-7-deazaguanosine monophosphate diradical 100G",
    "mf": "C12H12N5O7P",
    "kind": "NucleotideP",
    "oneLetter": "φ",
    "ocl": {
      "value": "fmwis@INzM\\J\\TgLp^MoBKcdRbbfrbTRdRUbSN^CWmUUUUKLuSuDQLPbY@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RqdCQ@B\\StPCFP@"
    },
    "mass": 369.22723233536925,
    "monoisotopicMass": 369.04743474432,
    "unsaturation": 18,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Azg",
    "name": "7-aminomethyl-7-deazaguanosine monophosphate diradical 101G",
    "mf": "C12H16N5O7P",
    "kind": "NucleotideP",
    "oneLetter": "∉",
    "ocl": {
      "value": "fmwis@INzM\\J\\TgLp^MoBKcdRbbfrbTRdRUbSN^CWmUUUUKLuSUDQLPbY@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RqdCQ@B\\StPA`z`"
    },
    "mass": 373.2589953515923,
    "monoisotopicMass": 373.07873487324,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Eqo",
    "name": "epoxyqueuosine monophosphate diradical 102G",
    "mf": "C17H22N5O10P",
    "kind": "NucleotideP",
    "oneLetter": "ς",
    "ocl": {
      "value": "el^ZEL@IGNaehJNEDlig`TPOFspb\\\\bTTTvTRbTbRlRjbbfXx|Bjz~aAajjjjiYfjZjjjjHPfDHSD@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RqdCQ@B\\StP@{ULY@TEIKA@a}tPA}BOpHAEP"
    },
    "mass": 487.3585341325581,
    "monoisotopicMass": 487.11042892533,
    "unsaturation": 18,
    "elements": [{
      "symbol": "C",
      "number": 17
    }, {
      "symbol": "H",
      "number": 22
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Aes",
    "name": "archaeosine monophosphate diradical 103G",
    "mf": "C12H15N6O7P",
    "kind": "NucleotideP",
    "oneLetter": "(",
    "ocl": {
      "value": "fcoYs@INzM^xTxiNY`|[^DWGHeEEMeDheHdkDhsg`u{UUUURsMTmTQDqBId@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RqdCQ@B\\StP@{UFCj"
    },
    "mass": 386.2577578089824,
    "monoisotopicMass": 386.07398384544,
    "unsaturation": 16,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 15
    }, {
      "symbol": "N",
      "number": 6
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Goq",
    "name": "galactosyl-queuosine monophosphate diradical 104G",
    "mf": "C23H32N5O14P",
    "kind": "NucleotideP",
    "oneLetter": "9",
    "ocl": {
      "value": "ekXzGL@IGNaehJNEDliod\\VU]SPOFspb\\\\bTTTvTRbTbRlRjbTrTrbfRXx|Bjz^AyEjjjjiYfjZijjjjjjbDIaBDq@@",
      "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@J_IorHbGtgD}F@RxRH_WwW@hbOTh}RIlCQ`B\\StXC[UB[@RxSPT`JHbGwQ`H`BaEQ~@Ha}bOq~Ox`BbGu~@Ha}bOrH@`"
    },
    "mass": 633.4999767508004,
    "monoisotopicMass": 633.16833772591,
    "unsaturation": 20,
    "elements": [{
      "symbol": "C",
      "number": 23
    }, {
      "symbol": "H",
      "number": 32
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 14
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Gaq",
    "name": "glutamyl-queuosine monophosphate diradical105G",
    "mf": "C22H29N6O12P",
    "kind": "NucleotideP",
    "oneLetter": "⊄",
    "ocl": {
      "value": "emWVCL@IGNaejXJNEDlioh\\YUPOFspb\\\\bTTTvTRbTbRlRjbTJTtrTXx|Bjz^AjjjjiYfjZijfjfjbDIaBDq@@",
      "coordinates": "!BTmB@c`JHUMmMtL@YtEHYgxQTaDoQ`L@YFY|gKMARH`Ygy|fpAfN`Hz@`H{PTb\\ltEIRtHBNHaTv|@YFYPTha}b@I~@Ha}_c~H@ha}bOq~@Ha}"
    },
    "mass": 600.473311954707,
    "monoisotopicMass": 600.15810739451,
    "unsaturation": 22,
    "elements": [{
      "symbol": "C",
      "number": 22
    }, {
      "symbol": "H",
      "number": 29
    }, {
      "symbol": "N",
      "number": 6
    }, {
      "symbol": "O",
      "number": 12
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Moq",
    "name": "mannosyl-queuosine monophosphate diradical 106G",
    "mf": "C23H32N5O14P",
    "kind": "NucleotideP",
    "oneLetter": "8",
    "ocl": {
      "value": "ekXzGL@IGNaehJNEDliod\\VU]SPOFspb\\\\bTTTvTRbTbRlRjbTrTrbfRXx|Bjz^AyEjjjjiYfjZijjjjjjbDIaBDq@@",
      "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@J_IorHbGtgD}F@RxRH_WwW@hbOTh}RIlCQ`B\\StXC[UB[@RxSPT`JHbGwQ`H`BaEQ~@Ha}bOq~Ox`BbGu~@Ha}bOrH@`"
    },
    "mass": 633.4999767508004,
    "monoisotopicMass": 633.16833772591,
    "unsaturation": 20,
    "elements": [{
      "symbol": "C",
      "number": 23
    }, {
      "symbol": "H",
      "number": 32
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 14
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Qus",
    "name": "queuosine monophosphate diradical 10G",
    "mf": "C17H22N5O9P",
    "kind": "NucleotideP",
    "oneLetter": "Q",
    "ocl": {
      "value": "edZZIL@IGNaehJNEDliohPOFspb\\\\bTTTvTRbTbRlRjbTKGG`UWSpMUUUUKLuSUMUTPaLHPfH@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPHoWtw@aOTd}RqdCQ@B\\StP@{ULY@RpQPTopHBGwQ@@QT"
    },
    "mass": 471.35912920823984,
    "monoisotopicMass": 471.11551430576,
    "unsaturation": 18,
    "elements": [{
      "symbol": "C",
      "number": 17
    }, {
      "symbol": "H",
      "number": 22
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 9
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Cpo",
    "name": "1-methyl-3-(3-amino-3-carboxypropyl)pseudouridine monophosphate diradical 1309U",
    "mf": "C14H20N3O10P",
    "kind": "NucleotideP",
    "oneLetter": "α",
    "ocl": {
      "value": "fgpk@OAWBgENSgi{`|[^DWGHeEEMeDheIhjbihs`RuUUTsTuSUMQDSDHfP@",
      "coordinates": "!BTh|SI~ioOwy`iR\\SiV|SFGxw}FH_]]}DqbH@gx_c|SFA`lIqOW_Xa}uwu~Ox`BbGu~Ox`B_`BH_P"
    },
    "mass": 421.2970385113492,
    "monoisotopicMass": 421.08863085201,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 14
    }, {
      "symbol": "H",
      "number": 20
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Mls",
    "name": "1-methylinosine monophosphate diradical 19A",
    "mf": "C11H13N4O7P",
    "kind": "NucleotideP",
    "oneLetter": "O",
    "ocl": {
      "value": "fi{Is@INBvENJSghOFwaEqrIQQSYQJIRIMIgOAjvjjjjefZZhbIbDSH@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtPCQ@D}R"
    },
    "mass": 344.21773398124395,
    "monoisotopicMass": 344.05218577211997,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 4
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Mpu",
    "name": "1-methylpseudouridine monophosphate diradical 19U",
    "mf": "C10H13N2O8P",
    "kind": "NucleotideP",
    "oneLetter": "]",
    "ocl": {
      "value": "fncPK@OAaSbgIrtGc[pbxyDhhilheDiLjs`RuUUTsTuMDQLPbY@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@"
    },
    "mass": 320.1929965859354,
    "monoisotopicMass": 320.04095238282997,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Mad",
    "name": "1-methyladenosine monophosphate diradical 1A",
    "mf": "C11H14N5O6P",
    "kind": "NucleotideP",
    "oneLetter": "\"",
    "ocl": {
      "value": "fi{hs@INBwlJ\\TgHOFwaEqrIQQSYQJIRIMIgOAjvjjjjefZZhbIbDSH@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtPCQ@D}R"
    },
    "mass": 343.2329730224273,
    "monoisotopicMass": 343.06817018921,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 6
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Mgs",
    "name": "1-methylguanosine monophosphate diradical 1G",
    "mf": "C11H14N5O7P",
    "kind": "NucleotideP",
    "oneLetter": "K",
    "ocl": {
      "value": "fegis@INBwlJ\\TgHp^MoBKcdRbbfrbTRdR\\RYspZmjjjjiYfijbHfHQL`@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtP@gD}D@SuH"
    },
    "mass": 359.23237794674554,
    "monoisotopicMass": 359.06308480878,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Msu",
    "name": "5-aminomethyl-2-selenouridine monophosphate diradical 20510U",
    "mf": "C10H14N3O7PSe",
    "kind": "NucleotideP",
    "oneLetter": "π",
    "ocl": {
      "value": "fasqp`I^{BgEIrtGc[p\\bQ\\\\bTTTvTRbTfUSNAKUUUULsTuDQLPbY@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@@tP"
    },
    "mass": 398.1676241841323,
    "monoisotopicMass": 398.97345859992004,
    "unsaturation": null,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "Se",
      "number": 1
    }]
  }, {
    "symbol": "Mse",
    "name": "5-methylaminomethyl-2-selenouridine monophosphate diradical 20511U",
    "mf": "C11H16N3O7PSe",
    "kind": "NucleotideP",
    "oneLetter": "≅",
    "ocl": {
      "value": "fikqp`I^{BgEIrtGc[p\\bQ\\\\bTTTvTRbTfUVYpIZjjjifZfjHbXaDr@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSUAMTBpX"
    },
    "mass": 412.19424158897914,
    "monoisotopicMass": 412.98910866438,
    "unsaturation": null,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "Se",
      "number": 1
    }]
  }, {
    "symbol": "Cse",
    "name": "5-carboxymethylaminomethyl-2-selenouridine monophosphate diradical 2051U",
    "mf": "C12H16N3O9PSe",
    "kind": "NucleotideP",
    "oneLetter": "⊥",
    "ocl": {
      "value": "fcwqH`I^{BgEIru^p^MoArIEqrIQQSYQJIRYUYJLxDmUUUTsMSTuDQLPbY@@",
      "coordinates": "!BKAb@tURDM\\YpMAMpBYMcx`BKB]~@Ha}SXW@h`Bb@IMcx}RtDvH_Xa}b@JH@ha}b@I~@Ha}"
    },
    "mass": 456.20378733435086,
    "monoisotopicMass": 456.97893790352,
    "unsaturation": null,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 9
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "Se",
      "number": 1
    }]
  }, {
    "symbol": "Agm",
    "name": "agmatidine monophosphate diradical 20C",
    "mf": "C14H26N7O6P",
    "kind": "NucleotideP",
    "oneLetter": "¿",
    "ocl": {
      "value": "fgxs@I^BuY{piqR\\`|[^DWGHeEEMeDeEHmUddhsgbuUUUSTuUUMIDSDHfP@",
      "coordinates": "!BDqc_tTnD_]\\fpH}MgrYRc}_|Dr_W_Wx@ThWM_|bOqRc}ARctu~@Gx@urH@gx@b@I~@H`BbGu~@@"
    },
    "mass": 419.373876184194,
    "monoisotopicMass": 419.16821858483,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 14
    }, {
      "symbol": "H",
      "number": 26
    }, {
      "symbol": "N",
      "number": 7
    }, {
      "symbol": "O",
      "number": 6
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Sou",
    "name": "2-selenouridine monophosphate diradical 20U",
    "mf": "C9H11N2O7PSe",
    "kind": "NucleotideP",
    "oneLetter": "ω",
    "ocl": {
      "value": "ff}Qp`I^aSbdyjCqmxNQHnNQJJJ[JIQJSMg@ejjjjfYihbIbDSH@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFDuP"
    },
    "mass": 369.1263628137839,
    "monoisotopicMass": 369.9469094988,
    "unsaturation": null,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "Se",
      "number": 1
    }]
  }, {
    "symbol": "Agu",
    "name": "5-aminomethyl-2-geranylthiouridine monophosphate diradical 21510U",
    "mf": "C20H30N3O7PS",
    "kind": "NucleotideP",
    "oneLetter": "Δ",
    "ocl": {
      "value": "ed\\\\NB@IOIhJNEDla`OFsp\\BHgGHeEEMeDheHdjdcEdhqpEUUUUURsUSMTuQBDpaBXdDt@",
      "coordinates": "!BDr__cdo[_X`fgx}RgqeRtM]}Dqa~O}\\BTmBH_]]}uwuRtMAMcuI~O}\\BupJH_]]}_`A~Oxa}uwu~Oxa}_cW_Xa}"
    },
    "mass": 487.5074340654907,
    "monoisotopicMass": 487.15420849000003,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 20
    }, {
      "symbol": "H",
      "number": 30
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Mgu",
    "name": "5-methylaminomethyl-2-geranylthiouridine monophosphate diradical 21511U",
    "mf": "C21H32N3O7PS",
    "kind": "NucleotideP",
    "oneLetter": "h",
    "ocl": {
      "value": "elR\\NB@IOIhJNEDla`OFsp\\BHgGHeEEMeDheHdjdlileFN@jjjjjjVZjYjijbDIaBDqHIh",
      "coordinates": "!BTv^cbn{__@fw|}RwqeRdK]}Tva~_{_|TiCp_[]}mwuRdIAMsuI~_{]|mwsp_[]}mwu~_{_||Gvw_Wy|Gu~_{]}|Gt"
    },
    "mass": 501.5340514703375,
    "monoisotopicMass": 501.16985855446006,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 21
    }, {
      "symbol": "H",
      "number": 32
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Cgu",
    "name": "5-carboxymethylaminomethyl-2-geranylthiouridine monophosphate diradical 2151U",
    "mf": "C22H32N3O9PS",
    "kind": "NucleotideP",
    "oneLetter": "f",
    "ocl": {
      "value": "ef^\\IB@IOIhJNEDla`XPOFsp\\BHgGHeEEMeDheHdjdlhehbhqpEUUUUURsUSMUMMTPaLHPfIAM@",
      "coordinates": "!BTv^cbn{_@fw|}RwqeRdK]}Tva~_{]|TiCp[_}muRdIAMsuI~_{]|mwsp_[]}mwu~_{]||Gvw_[_}_g}~_{]||Ou~_{]}|Gt"
    },
    "mass": 545.5435972157093,
    "monoisotopicMass": 545.1596877935999,
    "unsaturation": 16,
    "elements": [{
      "symbol": "C",
      "number": 22
    }, {
      "symbol": "H",
      "number": 32
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 9
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Mha",
    "name": "2-methylthio-N6-(cis-hydroxyisopentenyl) adenosine monophosphate diradical 2160A",
    "mf": "C16H22N5O7PS",
    "kind": "NucleotideP",
    "oneLetter": "≠",
    "ocl": {
      "value": "e`TZNB@IG@nahJNEDlo`OFspb\\V`cHeEEMeDheHdxeleDqqxEUuUUUU@sAETuTDHSBDIbP[P",
      "coordinates": "!BzfC@IeKPaDn}bHCQb@KQwuRDFALYpHCQt]WHc|TmCQw}~N`ME~@Gx@upJH@h`B_`BH_X`BbGvHGxbGt"
    },
    "mass": 459.41437086899504,
    "monoisotopicMass": 459.09775624102,
    "unsaturation": 16,
    "elements": [{
      "symbol": "C",
      "number": 16
    }, {
      "symbol": "H",
      "number": 22
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Mpa",
    "name": "2-methylthio-N6-isopentenyladenosine monophosphate diradical 2161A",
    "mf": "C16H22N5O6PS",
    "kind": "NucleotideP",
    "oneLetter": "*",
    "ocl": {
      "value": "eohZFB@IG@nahJNEDl`OFspb\\V`cHeEEMeDheHdxeleFNO@jnjjjjhFXHjfjBDIaBDq@@",
      "coordinates": "!BpBYTvxBNFY|bEJObGvOS\\@Yt]~DUEJOctu~@Ha}`HzOSTwPTh~H@h`B_`BH_Xa}bOrH@ha}b@I~@Ha}"
    },
    "mass": 443.4149659446768,
    "monoisotopicMass": 443.10284162145,
    "unsaturation": 16,
    "elements": [{
      "symbol": "C",
      "number": 16
    }, {
      "symbol": "H",
      "number": 22
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 6
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Mca",
    "name": "2-methylthio-N6-threonylcarbamoyladenosine monophosphate diradical 2162A",
    "mf": "C16H21N6O10PS",
    "kind": "NucleotideP",
    "oneLetter": "[",
    "ocl": {
      "value": "ebVVEB@IG@nachJNEDlm`XTPOFspb\\V`cHeEEMeDheHdxemLhhhqqxEUuUUUU@sAESUMUABDpaBX`@",
      "coordinates": "!BzfC@IeKPaDn}bHCQb@KQwuRDFALYpHCQt]W@h`BTmCQw}~N`ME~@Gx@upJH@h`B_`BH_Wxb@JH_WxbOrHo]^}_`BH_P"
    },
    "mass": 520.4113480993399,
    "monoisotopicMass": 520.07774907193,
    "unsaturation": 18,
    "elements": [{
      "symbol": "C",
      "number": 16
    }, {
      "symbol": "H",
      "number": 21
    }, {
      "symbol": "N",
      "number": 6
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Mva",
    "name": "2-methylthio-N6-hydroxynorvalylcarbamoyladenosine monophosphate diradical 2163A",
    "mf": "C17H23N6O10PS",
    "kind": "NucleotideP",
    "oneLetter": "≈",
    "ocl": {
      "value": "ej^VEB@IG@nachJNEDlm`XTPOFspb\\V`cHeEEMeDheHdxemLhhiVNO@jnjjjjhFXHjZijjBDIaBDq@@",
      "coordinates": "!BpBYTvxBNFY|BbEJObGvOS\\@Yt]~DUEJOctu~@Ha}`HzOSTwPTh~H@h`B_`BH_Xa}bOrH@gx@bGvHGx@bGwW@h`B_c~H@ha}"
    },
    "mass": 534.4379655041866,
    "monoisotopicMass": 534.09339913639,
    "unsaturation": 18,
    "elements": [{
      "symbol": "C",
      "number": 17
    }, {
      "symbol": "H",
      "number": 23
    }, {
      "symbol": "N",
      "number": 6
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Mya",
    "name": "2-methylthio cyclic N6-threonylcarbamoyladenosine monophosphate diradical 2164A",
    "mf": "C17H20N5O9PS",
    "kind": "NucleotideP",
    "oneLetter": "ÿ",
    "ocl": {
      "value": "elVZIB@IG@nkhJNEDlcghPOFspb\\V`cHeEEMeDheHdxeihiUFNO@jnkojjjjhFXHjfZjbHPfDHSD@@",
      "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@J_IorHbGtgD}F@RxRH_WwW@hbOTh}RIqOQ`MF@cuKW@hQTcttfpL@YS]@BbGvH@Gx"
    },
    "mass": 501.4080351062552,
    "monoisotopicMass": 501.07193541570007,
    "unsaturation": 20,
    "elements": [{
      "symbol": "C",
      "number": 17
    }, {
      "symbol": "H",
      "number": 20
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 9
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Hta",
    "name": "hydroxy-N6-threonylcarbamoyladenosine monophosphate diradical 2165A",
    "mf": "C15H19N6O11P",
    "kind": "NucleotideP",
    "oneLetter": "«",
    "ocl": {
      "value": "elZVML@IG@fnehJNEDligo`TPOFspb\\\\bTTTvTRbTbSVTrbbeXx|BjZjjjj`Y`JZijjBDIaBDq@@",
      "coordinates": "!BpBYTvxBNFY|bEJObGvOS\\@Yt]~DUEJOctu~@Ha}`HzOSTwPTh~HH`BbGvH_Xc|_`BH_Xc|_`BH_]_|bOq~Oxc|bGt"
    },
    "mass": 490.31934821268436,
    "monoisotopicMass": 490.08494245264,
    "unsaturation": 18,
    "elements": [{
      "symbol": "C",
      "number": 15
    }, {
      "symbol": "H",
      "number": 19
    }, {
      "symbol": "N",
      "number": 6
    }, {
      "symbol": "O",
      "number": 11
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Lyd",
    "name": "2-lysidine monophosphate diradical 21C",
    "mf": "C15H24N5O8P",
    "kind": "NucleotideP",
    "oneLetter": "}",
    "ocl": {
      "value": "eo`ZAL@IGOFmhJNEDlkg`OFspb\\\\bTTTvTRbTbSVRTtXxBJjjjjfYjZjfhaBXPaLP@",
      "coordinates": "!BTh|SI~ioOwy`iR\\SiV|SFGxw}FH_]]}Dqa~Oxc|_c|SFA`lIqOW_Xa}_c~HHa}bOrH_WxbOq~@Ha}"
    },
    "mass": 433.3541339985626,
    "monoisotopicMass": 433.13624975064994,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 15
    }, {
      "symbol": "H",
      "number": 24
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Gtu",
    "name": "2-geranylthiouridine monophosphate diradical 21U",
    "mf": "C19H27N2O7PS",
    "kind": "NucleotideP",
    "oneLetter": "Γ",
    "ocl": {
      "value": "e`XTNB@IOHJNEDln`OFsp\\BHgGHeEEMeDheHdtmEdhqpEUUUUURsUKUMTPaLHPfIAu@",
      "coordinates": "!BTv^cbn{__@fw|}RwqeRdK]}Tva~_{_|TiCp_[]}mwuRdIAMsuI~_{]||Gvw_Wy|Gvw_Wy|Gu~_{]}|Gt"
    },
    "mass": 458.46617269514235,
    "monoisotopicMass": 458.12765938888003,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 19
    }, {
      "symbol": "H",
      "number": 27
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Tyg",
    "name": "N2,N2,7-trimethylguanosine cap monophosphate diradical (cap TMG) 2279553N",
    "mf": "C13H20N5O10P2",
    "kind": "NucleotideP",
    "oneLetter": "¶",
    "ocl": {
      "value": "e`TZEBHIG@aihJNEHdleck`OFspz|MgDJTef[vVVe_gifNO@jijjjjjUijifjhaBXPaLP@",
      "coordinates": "!BvuPfpDnDtEK_t_rHtXBH_TwPbOr_I`JHbGtgD}F@RxS|uxc|_]^OTh}RIlA~@B\\StXCQ`Gx@Owx@_h{_cuH"
    },
    "mass": 468.2734710359255,
    "monoisotopicMass": 468.06854085929,
    "unsaturation": 13,
    "elements": [{
      "symbol": "C",
      "number": 13
    }, {
      "symbol": "H",
      "number": 20
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 2
    }]
  }, {
    "symbol": "Tmg",
    "name": "N2,N2,7-trimethylguanosine monophosphate diradical 227G",
    "mf": "C13H20N5O7P",
    "kind": "NucleotideP",
    "oneLetter": "∠",
    "ocl": {
      "value": "fcoisBINCt\\J\\TgLp^MoBKbFY}dRbbfrbTRdRUbtYspZcjjjjiYfjjjHbXaDr@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}Rqd@gD}D@tPBNOt}R"
    },
    "mass": 389.30149426455074,
    "monoisotopicMass": 389.11003500216,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 13
    }, {
      "symbol": "H",
      "number": 20
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Dgu",
    "name": "N2,N2-dimethylguanosine monophosphate diradical 22G",
    "mf": "C12H16N5O7P",
    "kind": "NucleotideP",
    "oneLetter": "R",
    "ocl": {
      "value": "fmwis@INBwlJ\\TgHp^MoBKcdRbbfrbTRdR\\RcN^CWmUUUUKLuMUDQLPbY@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RIqOQ@MD@cc}OT`"
    },
    "mass": 373.2589953515923,
    "monoisotopicMass": 373.07873487324,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Atu",
    "name": "5-aminomethyl-2-thiouridine monophosphate diradical 2510U",
    "mf": "C10H14N3O7PS",
    "kind": "NucleotideP",
    "oneLetter": "∫",
    "ocl": {
      "value": "fasqp`I^{BgEIrtGc[p\\DQ\\\\bTTTvTRbTfUSNAKUUUULsTuDQLPbY@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@@tP"
    },
    "mass": 351.27302303324575,
    "monoisotopicMass": 351.02900797432005,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Mou",
    "name": "5-methylaminomethyl-2-thiouridine monophosphate diradical 2511U",
    "mf": "C11H16N3O7PS",
    "kind": "NucleotideP",
    "oneLetter": "S",
    "ocl": {
      "value": "fikqp`I^{BgEIrtGc[p\\DQ\\\\bTTTvTRbTfUVYpIZjjjifZfjHbXaDr@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSUAMTBpX"
    },
    "mass": 365.2996404380926,
    "monoisotopicMass": 365.04465803878,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Ctu",
    "name": "5-carboxymethylaminomethyl-2-thiouridine monophosphate diradical 251U",
    "mf": "C12H16N3O9PS",
    "kind": "NucleotideP",
    "oneLetter": "$",
    "ocl": {
      "value": "fcwqH`I^{BgEIru^p^MoApQEqrIQQSYQJIRYUYJLxDmUUUTsMSTuDQLPbY@@",
      "coordinates": "!BKAb@tURDM\\YpMAMpBYMcx`BKB]~@Ha}SXW@h`Bb@IMcx}RtDvH_Xa}b@JH@ha}b@I~@Ha}"
    },
    "mass": 409.3091861834643,
    "monoisotopicMass": 409.03448727792,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 9
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Myu",
    "name": "5-methoxycarbonylmethyl-2-thiouridine monophosphate diradical 2521U",
    "mf": "C12H15N2O9PS",
    "kind": "NucleotideP",
    "oneLetter": "3",
    "ocl": {
      "value": "fmgQH`I^aSbdyZNXOFw`xHbxyDhhilheDiLjmLs`RuUUUSLuLuQDSDHfP@",
      "coordinates": "!BS]@lFJU`@Gyoza`lzf@lIwx@`H{WHc|KB_W_Wx@_`@lIr\\SFBrH@h`B_`BH_WxbOrH_P"
    },
    "mass": 394.2945422179627,
    "monoisotopicMass": 394.02358824126003,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 15
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 9
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Cou",
    "name": "5-carbamoylmethyl-2-thiouridine monophosphate diradical 253U",
    "mf": "C11H14N3O8PS",
    "kind": "NucleotideP",
    "oneLetter": "l",
    "ocl": {
      "value": "fe{pH`I^gBgEIrtXOFw`xHbxyDhhilheDiLjmF\\BVjjjjYfifhbIbDSH@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSUAMTC~NKA`"
    },
    "mass": 379.2831638542993,
    "monoisotopicMass": 379.02392259389,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Cau",
    "name": "5-carboxymethyl-2-thiouridine monophosphate diradical 2540U",
    "mf": "C11H13N2O9PS",
    "kind": "NucleotideP",
    "oneLetter": "℘",
    "ocl": {
      "value": "fe{QH`I^aSbdyZNXOFw`xHbxyDhhilheDiLjmF\\BVjjjjYfifhbIbDSH@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSUAMTC~NKA`"
    },
    "mass": 380.26792481311594,
    "monoisotopicMass": 380.00793817680005,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 9
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Tau",
    "name": "5-taurinomethyl-2-thiouridine monophosphate diradical 254U",
    "mf": "C12H18N3O10PS2",
    "kind": "NucleotideP",
    "oneLetter": "∃",
    "ocl": {
      "value": "fgpj`I^{BgEIrwY{`|[^C`bKblHrIQQSYQJIRYUYIRLxDmUUUTsMSUKTQDqBId@@",
      "coordinates": "!BKAb@tURD@m\\YpMAMpBYMcx`BKB]~@Ha}SXW@h`Bb@IMcx}RtDvH_Xa}b@JH@ha}b@JH__rH_]^H_P"
    },
    "mass": 459.3892600220213,
    "monoisotopicMass": 459.01712313635005,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 18
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 2
    }]
  }, {
    "symbol": "Itu",
    "name": "5-(isopentenylaminomethyl)-2-thiouridine monophosphate diradical 2583U",
    "mf": "C15H22N3O7PS",
    "kind": "NucleotideP",
    "oneLetter": "½",
    "ocl": {
      "value": "fkoqp`I^{BgEIrtGc[p\\DQ\\\\bTTTvTRbTfUVRTYpIZjjjifZfijbHfHQL`@",
      "coordinates": "!BS]@lFJU`@Gyoza`lzf@lIwx@`H{W@h`BKB_W_Wx@_`@lIr\\SFBrH@h`B_`BH_Xc|bGvH@gx@bGt"
    },
    "mass": 419.3902285493682,
    "monoisotopicMass": 419.09160823216,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 15
    }, {
      "symbol": "H",
      "number": 22
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Mth",
    "name": "5-methyl-2-thiouridine monophosphate diradical 25U",
    "mf": "C10H13N2O7PS",
    "kind": "NucleotideP",
    "oneLetter": "F",
    "ocl": {
      "value": "fncQp`I^aSbdyZCqmxNBHnNQJJJ[JIQJSJlxDmUUUTsMSQDSDHfP@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@"
    },
    "mass": 336.25837906774416,
    "monoisotopicMass": 336.01810893766003,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Dmg",
    "name": "N2,7-dimethylguanosine monophosphate diradical 27G",
    "mf": "C12H18N5O7P",
    "kind": "NucleotideP",
    "oneLetter": "∨",
    "ocl": {
      "value": "fmwisBINCt\\J\\TgLp^MoBKbFY}dRbbfrbTRdRUbKN^CWmUUUUKLuUUDQLPbY@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RqdCQ@B\\StPAOT`"
    },
    "mass": 375.27487685970397,
    "monoisotopicMass": 375.0943849377,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 18
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Dma",
    "name": "2,8-dimethyladenosine monophosphate diradical 28A",
    "mf": "C12H16N5O6P",
    "kind": "NucleotideP",
    "oneLetter": "±",
    "ocl": {
      "value": "feghs@INCv\\J\\UdhOFw`eqrIQQSYQJJJQKqLyxK^uUUUPMLAUADSDHfP@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpP@c`a}_S_|BD}RSuKQ@B\\StP@"
    },
    "mass": 357.2595904272741,
    "monoisotopicMass": 357.08382025367,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 6
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Mas",
    "name": "2-methyladenosine monophosphate diradical 2A",
    "mf": "C11H14N5O6P",
    "kind": "NucleotideP",
    "oneLetter": "/",
    "ocl": {
      "value": "fi{hs@INBt\\J\\TgHOFwaEqrIQQSYQJIRINIgOAjvjjjjAf@j`bIbDSH@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtP@gD}D@"
    },
    "mass": 343.2329730224273,
    "monoisotopicMass": 343.06817018921,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 6
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Tcy",
    "name": "2-thiocytidine monophosphate diradical 2C",
    "mf": "C9H12N3O6PS",
    "kind": "NucleotideP",
    "oneLetter": "%",
    "ocl": {
      "value": "ff}pp`I^kBgEIrCqmxNBHnNQJJJ[JIQJSMg@ejjjjfYfhbIbDSH@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFDuP"
    },
    "mass": 321.2470007040807,
    "monoisotopicMass": 321.01844329029,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 6
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Nmg",
    "name": "N2-methylguanosine monophosphate diradical 2G",
    "mf": "C11H14N5O7P",
    "kind": "NucleotideP",
    "oneLetter": "L",
    "ocl": {
      "value": "fegis@INBwlJ\\TgHp^MoBKcdRbbfrbTRdR\\VYspZmjjjjiYfijbHfHQL`@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtP@gD}D@SuH"
    },
    "mass": 359.23237794674554,
    "monoisotopicMass": 359.06308480878,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Thu",
    "name": "2-thiouridine monophosphate diradical 2U",
    "mf": "C9H11N2O7PS",
    "kind": "NucleotideP",
    "oneLetter": "2",
    "ocl": {
      "value": "ff}Qp`I^aSbdyjCqmxNBHnNQJJJ[JIQJSMg@ejjjjfYihbIbDSH@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFDuP"
    },
    "mass": 322.2317616628973,
    "monoisotopicMass": 322.0024588732,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Ahu",
    "name": "3-(3-amino-3-carboxypropyl)-5,6-dihydrouridine monophosphate diradical 308U",
    "mf": "C13H20N3O10P",
    "kind": "NucleotideP",
    "oneLetter": "Ð",
    "ocl": {
      "value": "fkopk@I^gBgENSens`|[^DWGHeEEMeDheIhueMF\\BVjjjjZfijfhbIbDSH@",
      "coordinates": "!BTh|SI~ioOwy`iR\\SiV|SFGxw}FH_]]}DqbH@gx_c|SFA`lIqOW_Xa}_c~HHa}_c~H@gx@bGt"
    },
    "mass": 409.28630261461393,
    "monoisotopicMass": 409.08863085201,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 13
    }, {
      "symbol": "H",
      "number": 20
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "App",
    "name": "3-(3-amino-3-carboxypropyl)pseudouridine monophosphate diradical 309U",
    "mf": "C13H18N3O10P",
    "kind": "NucleotideP",
    "oneLetter": "Þ",
    "ocl": {
      "value": "fkopk@OAgBgENSens`|[^DWGHeEEMeDheIhueMF\\BVjjjfZfijfhbIbDSH@",
      "coordinates": "!BTh|SI~ioOwy`iR\\SiV|SFGxw}FH_]]}DqbH@gx_c|SFA`lIqOW_Xa}_c~HHa}_c~H@gx@bGt"
    },
    "mass": 407.2704211065024,
    "monoisotopicMass": 407.07298078755,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 13
    }, {
      "symbol": "H",
      "number": 18
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Apu",
    "name": "3-(3-amino-3-carboxypropyl)uridine monophosphate diradical 30U",
    "mf": "C13H18N3O10P",
    "kind": "NucleotideP",
    "oneLetter": "X",
    "ocl": {
      "value": "fkopk@I^gBgENSens`|[^DWGHeEEMeDheIhueMF\\BVjjjjYfijfhbIbDSH@",
      "coordinates": "!BTh|SI~ioOwy`iR\\SiV|SFGxw}FH_]]}DqbH@gx_c|SFA`lIqOW_Xa}_c~HHa}_c~H@gx@bGt"
    },
    "mass": 407.2704211065024,
    "monoisotopicMass": 407.07298078755,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 13
    }, {
      "symbol": "H",
      "number": 18
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Mws",
    "name": "methylwyosine monophosphate diradical 342G",
    "mf": "C15H18N5O7P",
    "kind": "NucleotideP",
    "oneLetter": "∑",
    "ocl": {
      "value": "eghZNL@IG@nahJNEDli`OFspb\\\\bTTTvTRbTbb\\rVSGG`SPrvuUUUUKMTsUUIBDpaBX`@",
      "coordinates": "!B_`CW@mF@ctvDUI|fRxPYgtwP[zV_IorHFY|gD}F@RxPYg|@YgrZOTh{_cuJOS]F@tXAKaI|fw}EMt@"
    },
    "mass": 411.3070845499097,
    "monoisotopicMass": 411.0943849377,
    "unsaturation": 18,
    "elements": [{
      "symbol": "C",
      "number": 15
    }, {
      "symbol": "H",
      "number": 18
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Uhw",
    "name": "undermodified hydroxywybutosine monophosphate diradical 3470G",
    "mf": "C18H23N6O10P",
    "kind": "NucleotideP",
    "oneLetter": "š",
    "ocl": {
      "value": "ejQVEL@IG@nahXJNEDliolRPOFspb\\\\bTTTvTRbTbb\\rVVTttXx|BZFVvjjjjiYjfZjjfjRDIaBDq@@",
      "coordinates": "!BKB^@ceS[H`Yg}ARpAeMtHa}KAcPTh{_S]CjXES[pAeMtH}MtEK@IdnDpBXBbES[UMo@F]ARaERH_X`B_`BH_WxbOq~@Ha}"
    },
    "mass": 514.3839139947949,
    "monoisotopicMass": 514.12132796199,
    "unsaturation": 20,
    "elements": [{
      "symbol": "C",
      "number": 18
    }, {
      "symbol": "H",
      "number": 23
    }, {
      "symbol": "N",
      "number": 6
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Acw",
    "name": "7-aminocarboxypropylwyosine monophosphate diradical 347G",
    "mf": "C18H23N6O9P",
    "kind": "NucleotideP",
    "oneLetter": "Ω",
    "ocl": {
      "value": "eb^VIL@IG@na`XJNEDlid\\POFspb\\\\bTTTvTRbTbb\\rVRrfcGG`SPrvuUUUUKMTsUUSUIBDpaBX`@",
      "coordinates": "!BDr]RcwwWpAg_tUS[cm~DUAf_XJUTvx}MaEP@_gwWcm~DUDnDUMo|urH@m_@FWwW_]^NwuS[bGtYgx`BbGu~Ox`B_`BH_P"
    },
    "mass": 498.38450907047655,
    "monoisotopicMass": 498.12641334242,
    "unsaturation": 20,
    "elements": [{
      "symbol": "C",
      "number": 18
    }, {
      "symbol": "H",
      "number": 23
    }, {
      "symbol": "N",
      "number": 6
    }, {
      "symbol": "O",
      "number": 9
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Hwy",
    "name": "methylated undermodified hydroxywybutosine monophosphate diradical 3480G",
    "mf": "C19H25N6O10P",
    "kind": "NucleotideP",
    "oneLetter": "y",
    "ocl": {
      "value": "efYVEL@IG@nahXJNEDliolRPOFspb\\\\bTTTvTRbTbb\\rVVTttsGG`SPrvuUUUUKMTsUUTuTdHSBDIb@@",
      "coordinates": "!B`HyRtL@f_XbDRxz@UHS_chc|S]BN`MAMwxyKaL@fUHS_cmG_chCjXI|YzfA}bL@fpBYTaHz@F\\BH@gx@upJH@ha}_`CWHc|_`@"
    },
    "mass": 528.4105313996416,
    "monoisotopicMass": 528.1369780264499,
    "unsaturation": 20,
    "elements": [{
      "symbol": "C",
      "number": 19
    }, {
      "symbol": "H",
      "number": 25
    }, {
      "symbol": "N",
      "number": 6
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Hwb",
    "name": "hydroxywybutosine monophosphate diradical 34830G",
    "mf": "C21H27N6O12P",
    "kind": "NucleotideP",
    "oneLetter": "⊆",
    "ocl": {
      "value": "ee[VCL@IG@nahXJNEDliobZV^POFspb\\\\bTTTvTRbTbb\\rVVTtRbfsGG`SPrvuUUUUKMTsUUULuUIBDpaBX`@",
      "coordinates": "!BKB^@ceS[@h`Yg}ARpAeMtHa}KAcPTh{_S]CjXES[pAeMtH}MtEK@IdnDpBXBbES[UMo@F]ARaERH_X`B_`BH_X`B_c~H_]]}bGu~Ox`B_c~H_P"
    },
    "mass": 586.4466945498602,
    "monoisotopicMass": 586.14245733005,
    "unsaturation": 22,
    "elements": [{
      "symbol": "C",
      "number": 21
    }, {
      "symbol": "H",
      "number": 27
    }, {
      "symbol": "N",
      "number": 6
    }, {
      "symbol": "O",
      "number": 12
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Pwb",
    "name": "peroxywybutosine monophosphate diradical 34832G",
    "mf": "C21H27N6O13P",
    "kind": "NucleotideP",
    "oneLetter": "W",
    "ocl": {
      "value": "emWVKL@IG@nadXJNEDliohZV^QPOFspb\\\\bTTTvTRbTbb\\rVVTRfTTvXx|BZFVvjjjjiYjfZjjjYjjRDIaBDq@@",
      "coordinates": "!BKB^@ceS[@h`Yg}ARpAeMtHa}KAcPTh{_S]CjXES[pAeMtH}MtEK@IdnDpB[|bES[UMo@F]ARaERH_X`B_`BH_X`Bb@I~Oxa}uwvH_Wxb@I~Oxa}"
    },
    "mass": 602.4460994741785,
    "monoisotopicMass": 602.1373719496199,
    "unsaturation": 22,
    "elements": [{
      "symbol": "C",
      "number": 21
    }, {
      "symbol": "H",
      "number": 27
    }, {
      "symbol": "N",
      "number": 6
    }, {
      "symbol": "O",
      "number": 13
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Wyb",
    "name": "wybutosine monophosphate diradical 3483G",
    "mf": "C21H27N6O11P",
    "kind": "NucleotideP",
    "oneLetter": "Y",
    "ocl": {
      "value": "eiSVML@IG@na`XJNEDlilRZVPOFspb\\\\bTTTvTRbTbb\\rVVRbTTvXx|BZFVvjjjjiYjfZjjfZjdaBXPaLP@",
      "coordinates": "!BsJ\\@ciP{@`YWuARPAeMT@a}sNaPThxSUCjhIP{PAeMTD}MTEI@IllDPB[|BIP{eCm@FUARAIPH_Pc|BGtHGzBGtw_Pa}_k|HGzBGt"
    },
    "mass": 570.4472896255419,
    "monoisotopicMass": 570.14754271048,
    "unsaturation": 22,
    "elements": [{
      "symbol": "C",
      "number": 21
    }, {
      "symbol": "H",
      "number": 27
    }, {
      "symbol": "N",
      "number": 6
    }, {
      "symbol": "O",
      "number": 11
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Awo",
    "name": "7-aminocarboxypropylwyosine methyl ester monophosphate diradical 348G",
    "mf": "C19H25N6O9P",
    "kind": "NucleotideP",
    "oneLetter": "⇑",
    "ocl": {
      "value": "ejQVIL@IG@na`XJNEDlid\\POFspb\\\\bTTTvTRbTbb\\rVVRffXx|BZFVvjjjjiYjfZjjZjRDIaBDq@@",
      "coordinates": "!B`HyRtL@f_XbDRxz@UHS_ch`BS]BN`MAMwxyKaL@fUHS_cmG_chCjXI|YzfA}bL@fpBYTaHz@F\\BHHa}bOq~@Ha}_c~H@ha}"
    },
    "mass": 512.4111264753234,
    "monoisotopicMass": 512.14206340688,
    "unsaturation": 20,
    "elements": [{
      "symbol": "C",
      "number": 19
    }, {
      "symbol": "H",
      "number": 25
    }, {
      "symbol": "N",
      "number": 6
    }, {
      "symbol": "O",
      "number": 9
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Wyo",
    "name": "wyosine monophosphate diradical 34G",
    "mf": "C14H16N5O7P",
    "kind": "NucleotideP",
    "oneLetter": "€",
    "ocl": {
      "value": "ek`ZNL@IG@nahJNEDli`OFspb\\\\bTTTvTRbTbb\\rVXx|BZFVvjjjjiYjfZjdaBXPaLP@",
      "coordinates": "!B_`CWMF@ctvDUI|fRxPYgtwP[zV_IorHFY|gD}F@RxPYg|@YgrZOTh{_cuJOS]F@tXAKaI|fw}D"
    },
    "mass": 397.2804671450629,
    "monoisotopicMass": 397.07873487324,
    "unsaturation": 18,
    "elements": [{
      "symbol": "C",
      "number": 14
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Pdu",
    "name": "3-methylpseudouridine monophosphate diradical 39U",
    "mf": "C10H13N2O8P",
    "kind": "NucleotideP",
    "oneLetter": "κ",
    "ocl": {
      "value": "fncPK@OAaSbgIrtGc[pbxyDhhilheDiMFs`RuUUTsTuMDQLPbY@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@HoWtw@`lFC~NSU@"
    },
    "mass": 320.1929965859354,
    "monoisotopicMass": 320.04095238282997,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Mri",
    "name": "3-methyluridine monophosphate diradical 3U",
    "mf": "C10H13N2O8P",
    "kind": "NucleotideP",
    "oneLetter": "δ",
    "ocl": {
      "value": "fncPK@I^aSbgIrtGc[pbxyDhhilheDiMFs`RuUUUSLuMDQLPbY@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@HoWtw@`lFC~NSU@"
    },
    "mass": 320.1929965859354,
    "monoisotopicMass": 320.04095238282997,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Acc",
    "name": "N4-acetylcytidine monophosphate diradical 42C",
    "mf": "C11H14N3O8P",
    "kind": "NucleotideP",
    "oneLetter": "M",
    "ocl": {
      "value": "fikpK@I^kBgENSghOFwaEqrIQQSYQJIRYiQg@ejjjjfYffhbIbDSH@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFDuPOxxSItuP"
    },
    "mass": 347.21837644817225,
    "monoisotopicMass": 347.05185141949,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Iws",
    "name": "isowyosine monophosphate diradical 42G",
    "mf": "C14H16N5O7P",
    "kind": "NucleotideP",
    "oneLetter": "⊇",
    "ocl": {
      "value": "ek`ZNL@IG@fnhJNEDla`OFspb\\\\bTTTvTRbTbSbRrXx|BjzfVjjjjiYjYjjdaBXPaLP@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RStwQ@MD@RpQ_qcQSU@"
    },
    "mass": 397.2804671450629,
    "monoisotopicMass": 397.07873487324,
    "unsaturation": 18,
    "elements": [{
      "symbol": "C",
      "number": 14
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Dmc",
    "name": "N4,N4-dimethylcytidine monophosphate diradical 44C",
    "mf": "C11H16N3O7P",
    "kind": "NucleotideP",
    "oneLetter": "μ",
    "ocl": {
      "value": "fasqs@I^kBgENSdGc[pbxyDhhilheDiLuF\\BVjjjjYfZjHbXaDr@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBGpcbpXBGtSItuPSU@H_Wtw@`lFDuPOxxlF@"
    },
    "mass": 333.23485303196554,
    "monoisotopicMass": 333.07258686438,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Adw",
    "name": "7-aminocarboxypropyl-demethylwyosine monophosphate diradical 47G",
    "mf": "C17H21N6O9P",
    "kind": "NucleotideP",
    "oneLetter": "¥",
    "ocl": {
      "value": "elVVIL@IG@fnohJNEDlahTPOFspb\\\\bTTTvTRbTbSbRrrTtXx|BjzfVjjjjiYjYjjijdaBXPaLP@",
      "coordinates": "!B`MERc|@Y_]^DUH{_UMo_tXa}SXPTh{_w}GjXES[pAg_t]F@cm@Il@f@haTvuS[pAgPThQTbGvH@ha}_c~HGx@bGt"
    },
    "mass": 484.3578916656298,
    "monoisotopicMass": 484.1107632779601,
    "unsaturation": 20,
    "elements": [{
      "symbol": "C",
      "number": 17
    }, {
      "symbol": "H",
      "number": 21
    }, {
      "symbol": "N",
      "number": 6
    }, {
      "symbol": "O",
      "number": 9
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Ncd",
    "name": "N4-methylcytidine monophosphate diradical 4C",
    "mf": "C10H14N3O7P",
    "kind": "NucleotideP",
    "oneLetter": "ν",
    "ocl": {
      "value": "fncqs@I^kBgENSdGc[pbxyDhhilheDiLts`RuUUUSLsUDQLPbY@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFDuPKA`"
    },
    "mass": 319.2082356271187,
    "monoisotopicMass": 319.05693679992004,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Dmw",
    "name": "4-demethylwyosine monophosphate diradical 4G",
    "mf": "C13H14N5O7P",
    "kind": "NucleotideP",
    "oneLetter": "†",
    "ocl": {
      "value": "fcis@INBwlJ\\TgHp^MoBKcdRbbfrbTRdR\\RVYspZ}fnjjjjefifjiHbXaDr@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RStwQ@MD@RpQ_qcQ"
    },
    "mass": 383.253849740216,
    "monoisotopicMass": 383.06308480878,
    "unsaturation": 18,
    "elements": [{
      "symbol": "C",
      "number": 13
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Mtu",
    "name": "5-methoxyuridine monophosphate diradical 501U",
    "mf": "C10H13N2O9P",
    "kind": "NucleotideP",
    "oneLetter": "5",
    "ocl": {
      "value": "fasQK@I^aSbgIsUhOFwaEqrIQQSYQJIRYULxDmUUUTsMSTQDqBId@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@@tP"
    },
    "mass": 336.19240151025366,
    "monoisotopicMass": 336.03586700240004,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 9
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Uoa",
    "name": "uridine 5-oxyacetic acid monophosphate diradical 502U",
    "mf": "C11H13N2O11P",
    "kind": "NucleotideP",
    "oneLetter": "V",
    "ocl": {
      "value": "fmgQk@I^aSbgIsUlu`|[^DWGHeEEMeDheIeUeF\\BVjjjjYfiijHbXaDr@@",
      "coordinates": "!BS]@lFJU`@Gyoza`lzf@lIwx@`H{WHc|KB_W_Wx@_`@lIr\\SFBrHHc|_`BH_Xc|_`BH_P"
    },
    "mass": 380.2019472556255,
    "monoisotopicMass": 380.02569624154,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 11
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Ume",
    "name": "uridine 5-oxyacetic acid methyl ester monophosphate diradical 503U",
    "mf": "C12H15N2O11P",
    "kind": "NucleotideP",
    "oneLetter": "υ",
    "ocl": {
      "value": "fcwQk@I^aSbgIsUlu`|[^DWGHeEEMeDheIeUeLs`RuUUUSLuMMTQDqBId@@",
      "coordinates": "!BKAb@tURDM\\YpMAMpBYMcxc|KB]~@Ha}SXWHc|bOqMcx}RtDvH_Xa}bOrH@ha}_c~HHa}"
    },
    "mass": 394.2285646604723,
    "monoisotopicMass": 394.041346306,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 15
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 11
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Hxc",
    "name": "5-hydroxycytidine monophosphate diradical 50C",
    "mf": "C9H12N3O8P",
    "kind": "NucleotideP",
    "oneLetter": "Ç",
    "ocl": {
      "value": "fncpK@I^[BgENSfhOFwaEqrIQQSYQJIRYUg@ejjjjfYfjHbXaDr@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@"
    },
    "mass": 321.1810231465902,
    "monoisotopicMass": 321.03620135502996,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Hxu",
    "name": "5-hydroxyuridine monophosphate diradical 50U",
    "mf": "C9H11N2O9P",
    "kind": "NucleotideP",
    "oneLetter": "∝",
    "ocl": {
      "value": "fncQK@I^aSbgIsUhOFwaEqrIQQSYQJIRYUg@ejjjjfYjZHbXaDr@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@"
    },
    "mass": 322.1657841054069,
    "monoisotopicMass": 322.02021693794,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 9
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Amu",
    "name": "5-aminomethyluridine monophosphate diradical 510U",
    "mf": "C10H14N3O8P",
    "kind": "NucleotideP",
    "oneLetter": "∪",
    "ocl": {
      "value": "faspK@I^{BgENSehOFwaEqrIQQSYQJIRYULxDmUUUTsMSTQDqBId@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@@tP"
    },
    "mass": 335.207640551437,
    "monoisotopicMass": 335.05185141949,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Mau",
    "name": "5-methylaminomethyluridine monophosphate diradical 511U",
    "mf": "C11H16N3O8P",
    "kind": "NucleotideP",
    "oneLetter": "{",
    "ocl": {
      "value": "fikpK@I^{BgENSehOFwaEqrIQQSYQJIRYUYg@ejjjjfYjZhbIbDSH@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSUAMTBpX"
    },
    "mass": 349.2342579562838,
    "monoisotopicMass": 349.06750148395,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Hmc",
    "name": "5-hydroxymethylcytidine monophosphate diradical 51C",
    "mf": "C10H14N3O8P",
    "kind": "NucleotideP",
    "oneLetter": "∅",
    "ocl": {
      "value": "faspK@I^[BgENSghOFwaEqrIQQSYQJIRYULxDmUUUTsLuTQDqBId@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@@tP"
    },
    "mass": 335.207640551437,
    "monoisotopicMass": 335.05185141949,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Cur",
    "name": "5-carboxymethylaminomethyluridine monophosphate diradical 51U",
    "mf": "C12H16N3O10P",
    "kind": "NucleotideP",
    "oneLetter": "!",
    "ocl": {
      "value": "fcwpk@I^{BgENSej}`|[^DWGHeEEMeDheIeUdhs`RuUUUSLuMSTQDqBId@@",
      "coordinates": "!BKAb@tURDM\\YpMAMpBYMcx`BKB]~@Ha}SXW@h`Bb@IMcx}RtDvH_Xa}b@JH@ha}b@I~@Ha}"
    },
    "mass": 393.24380370165557,
    "monoisotopicMass": 393.05733072309,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Chr",
    "name": "5-carboxyhydroxymethyluridine monophosphate diradical 520U",
    "mf": "C11H13N2O11P",
    "kind": "NucleotideP",
    "oneLetter": "≥",
    "ocl": {
      "value": "fmgQk@I^aSbgIrwlu`|[^DWGHeEEMeDheIeUCF\\BVjjjjYfiijHbXaDr@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@@tTuPOxxlF@"
    },
    "mass": 380.2019472556255,
    "monoisotopicMass": 380.02569624154,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 11
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Mcu",
    "name": "5-methoxycarbonylmethyluridine monophosphate diradical 521U",
    "mf": "C12H15N2O10P",
    "kind": "NucleotideP",
    "oneLetter": "1",
    "ocl": {
      "value": "fmgPk@I^aSbgIrt\\p^MoBKcdRbbfrbTRdrjtsNAKUUUULsTsUDQLPbY@@",
      "coordinates": "!BS]@lFJU`@Gyoza`lzf@lIwx@`H{WHc|KB_W_Wx@_`@lIr\\SFBrH@h`B_`BH_WxbOrH_P"
    },
    "mass": 378.229159736154,
    "monoisotopicMass": 378.04643168643,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 15
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Hme",
    "name": "5-(carboxyhydroxymethyl)uridine methyl ester monophosphate diradical 522U",
    "mf": "C12H15N2O11P",
    "kind": "NucleotideP",
    "oneLetter": ",",
    "ocl": {
      "value": "fcwQk@I^aSbgIrwlu`|[^DWGHeEEMeDheIeUCLs`RuUUUSLuMMTQDqBId@@",
      "coordinates": "!BS]@lFJU`@Gyoza`lzf@lIwx@`H{WHc|KB_W_Wx@_`@lIr\\SFBrHHc|_`A~@Ha}_c~H@ha}"
    },
    "mass": 394.2285646604723,
    "monoisotopicMass": 394.041346306,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 15
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 11
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Cxu",
    "name": "5-carboxymethyluridine monophosphate diradical 52U",
    "mf": "C11H13N2O10P",
    "kind": "NucleotideP",
    "oneLetter": "◊",
    "ocl": {
      "value": "fe{Pk@I^aSbgIrt\\p^MoBKcdRbbfrbTRdrjtYpIZjjjifZfZbHfHQL`@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSUAMTC~NKA`"
    },
    "mass": 364.2025423313072,
    "monoisotopicMass": 364.03078162197,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Hmu",
    "name": "5-carbamoylhydroxymethyluridine monophosphate diradical 531U",
    "mf": "C11H14N3O10P",
    "kind": "NucleotideP",
    "oneLetter": "r",
    "ocl": {
      "value": "fmgpk@I^WBgENSeoY`|[^DWGHeEEMeDheIeUCF\\BVjjjjYfiijHbXaDr@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@@tTuPOxxlF@"
    },
    "mass": 379.21718629680873,
    "monoisotopicMass": 379.04168065863,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Ymu",
    "name": "5-carbamoylmethyluridine monophosphate diradical 53U",
    "mf": "C11H14N3O9P",
    "kind": "NucleotideP",
    "oneLetter": "&",
    "ocl": {
      "value": "fe{qK@I^gBgENSehp^MoBKcdRbbfrbTRdrjtYpIZjjjifZfZbHfHQL`@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSUAMTC~NKA`"
    },
    "mass": 363.2177813724905,
    "monoisotopicMass": 363.04676603906006,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 9
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Ttu",
    "name": "5-taurinomethyluridine monophosphate diradical 54U",
    "mf": "C12H18N3O11PS",
    "kind": "NucleotideP",
    "oneLetter": "Ê",
    "ocl": {
      "value": "fgqh`I^{BgENSenswAxv|HnJpcHeEEMeDheIeUdeHs`RuUUUSLuMTmQDSDHfP@",
      "coordinates": "!BKAb@tURD@m\\YpMAMpBYMcx`BKB]~@Ha}SXW@h`Bb@IMcx}RtDvH_Xa}b@JH@ha}b@JH__rH_]^H_P"
    },
    "mass": 443.32387754021244,
    "monoisotopicMass": 443.03996658152005,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 18
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 11
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Cmu",
    "name": "5-cyanomethyluridine monophosphate diradical 55U",
    "mf": "C11H12N3O8P",
    "kind": "NucleotideP",
    "oneLetter": "Ѷ",
    "ocl": {
      "value": "fikpK@I^GBgENSehOFwaEqrIQQSYQJIRYUYg@ejjjjfYj[hbIbDSH@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@@tPCQ"
    },
    "mass": 345.20249494006066,
    "monoisotopicMass": 345.03620135502996,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Pyu",
    "name": "5-(isopentenylaminomethyl)uridine monophosphate diradical 583U",
    "mf": "C15H22N3O8P",
    "kind": "NucleotideP",
    "oneLetter": "¾",
    "ocl": {
      "value": "fkopK@I^{BgENSehOFwaEqrIQQSYQJIRYUYIQg@ejjjjfYjZfjHbXaDr@@",
      "coordinates": "!BS]@lFJU`@Gyoza`lzf@lIwx@`H{W@h`BKB_W_Wx@_`@lIr\\SFBrH@h`B_`BH_Xc|bGvH@gx@bGt"
    },
    "mass": 403.32484606755946,
    "monoisotopicMass": 403.11445167733,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 15
    }, {
      "symbol": "H",
      "number": 22
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Mdu",
    "name": "5-methyldihydrouridine monophosphate diradical 58U",
    "mf": "C10H15N2O8P",
    "kind": "NucleotideP",
    "oneLetter": "ρ",
    "ocl": {
      "value": "fncPK@I^aSbgIrtGc[pbxyDhhilheDiLjs`RuUUUSTuMDQLPbY@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@"
    },
    "mass": 322.20887809404695,
    "monoisotopicMass": 322.05660244729,
    "unsaturation": 8,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 15
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Mcd",
    "name": "5-methylcytidine monophosphate diradical 5C",
    "mf": "C10H14N3O7P",
    "kind": "NucleotideP",
    "oneLetter": "?",
    "ocl": {
      "value": "fncqs@I^[BgENSdGc[pbxyDhhilheDiLjs`RuUUUSLsUDQLPbY@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@"
    },
    "mass": 319.2082356271187,
    "monoisotopicMass": 319.05693679992004,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Hia",
    "name": "N6-(cis-hydroxyisopentenyl)adenosine monophosphate diradical 60A",
    "mf": "C15H20N5O7P",
    "kind": "NucleotideP",
    "oneLetter": "`",
    "ocl": {
      "value": "eg`ZNL@IG@fnhJNEDlk`OFspb\\\\bTTTvTRbTbSVRTSGG`USUUUUTCLATuTDHSBDIbPSH",
      "coordinates": "!BzfC@IeKPaDn}bHCQb@KQwuRDFALYpHCQt]W@h`BTmCQw}~N`ME~@Gx@b@JH@ha}bOrH_Wxb@JH_P"
    },
    "mass": 413.3229660580212,
    "monoisotopicMass": 413.11003500216003,
    "unsaturation": 16,
    "elements": [{
      "symbol": "C",
      "number": 15
    }, {
      "symbol": "H",
      "number": 20
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Mta",
    "name": "2-methylthio-N6-methyladenosine monophosphate diradical 621A",
    "mf": "C12H16N5O6PS",
    "kind": "NucleotideP",
    "oneLetter": "∞",
    "ocl": {
      "value": "fmwhp`CQstZLDxipEfGa[qZDYEIlheDdhXdmDmKR\\u{MUUUU@aEUAFPTdmH@",
      "coordinates": "!BBGw|B@a}_S\\H@a}TEJNOuP{Ntm@fPBN[~iRSpHUCneXDBYTEITAEPDiVA@fTBYU@Sj[p"
    },
    "mass": 389.3243778334011,
    "monoisotopicMass": 389.05589142807,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 6
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Tca",
    "name": "N6-threonylcarbamoyladenosine monophosphate diradical 62A",
    "mf": "C15H19N6O10P",
    "kind": "NucleotideP",
    "oneLetter": "6",
    "ocl": {
      "value": "edRVEL@IG@fnehJNEDligo`POFspb\\\\bTTTvTRbTbSVTrbbcGG`USUUUUTCLASUMUABDpaBX`@",
      "coordinates": "!BzfC@IeKPaDn}bHCQbOsQwuRDFALYpHCQt]W@h`BTmCQw}~N`ME~@Gx@b@JH@ha}_c~H@ha}_c~H@ha}uwu~@Ha}"
    },
    "mass": 474.31994328836606,
    "monoisotopicMass": 474.09002783307,
    "unsaturation": 18,
    "elements": [{
      "symbol": "C",
      "number": 15
    }, {
      "symbol": "H",
      "number": 19
    }, {
      "symbol": "N",
      "number": 6
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Hva",
    "name": "N6-hydroxynorvalylcarbamoyladenosine monophosphate diradical 63A",
    "mf": "C16H21N6O10P",
    "kind": "NucleotideP",
    "oneLetter": "√",
    "ocl": {
      "value": "elZVIB@IG@fnehJNDligo`POEQql|HgGHeEEMeDheHdueLhhiVNO@jfjjjjhFXBfjZj`aBXPaLP@",
      "coordinates": "!BpBYTvxBNFY|bEJObGvOS\\@Yt]~DUEJOctu~@Ha}`HzOSTwPTh~HH`BbGvH_Xc|_`BH_Xc|_`BH_]_|bOq~Oxc|bGt"
    },
    "mass": 488.34656069321284,
    "monoisotopicMass": 488.10567789753003,
    "unsaturation": 18,
    "elements": [{
      "symbol": "C",
      "number": 16
    }, {
      "symbol": "H",
      "number": 21
    }, {
      "symbol": "N",
      "number": 6
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Aya",
    "name": "N6-acetyladenosine monophosphate diradical 64A",
    "mf": "C12H14N5O7P",
    "kind": "NucleotideP",
    "oneLetter": "⇓",
    "ocl": {
      "value": "fmwis@INBwlJ\\TgLp^MoBKcdRbbfrbTRdRZrcN^CUmUUUTCLASTDQLPbY@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtPCQ@D}RIqOQ@@"
    },
    "mass": 371.2431138434808,
    "monoisotopicMass": 371.06308480878,
    "unsaturation": 16,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Gca",
    "name": "N6-glycinylcarbamoyladenosine monophosphate diradical 65A",
    "mf": "C13H15N6O9P",
    "kind": "NucleotideP",
    "oneLetter": "≡",
    "ocl": {
      "value": "eohVIL@IG@fnehJNEDlikg`OFspb\\\\bTTTvTRbTbSVTrTXx|BjZjjjj`Y`JZfhHPfDHSD@@",
      "coordinates": "!BzfC@IeKPaDn}bHCQb@KQwuRDFALYpHCQt]W@h`BTmCQw}~N`ME~@Gx@bOrHHa}_c~H@ha}bOq~@Ha}"
    },
    "mass": 430.2673035543541,
    "monoisotopicMass": 430.06381308458,
    "unsaturation": 18,
    "elements": [{
      "symbol": "C",
      "number": 13
    }, {
      "symbol": "H",
      "number": 15
    }, {
      "symbol": "N",
      "number": 6
    }, {
      "symbol": "O",
      "number": 9
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Tya",
    "name": "N6-methyl-N6-threonylcarbamoyladenosinemonophosphate diradical 662A",
    "mf": "C16H21N6O10P",
    "kind": "NucleotideP",
    "oneLetter": "E",
    "ocl": {
      "value": "elZVEL@IG@fnmhJNEDleo`XPOFspb\\\\bTTTvTRbTbSVbaTTTXx|BjZjjjj`Y`JfjZjBDIaBDq@@",
      "coordinates": "!BzfC@IeKPaDn}bHCQb@KQwuRDFALYpHCQt]W@h`BTmCQw}~N`ME~@Gx@bOrHHa}_`A~Ox`BbGu~Ox`BbGwW_Wx@bGt"
    },
    "mass": 488.34656069321284,
    "monoisotopicMass": 488.10567789753003,
    "unsaturation": 18,
    "elements": [{
      "symbol": "C",
      "number": 16
    }, {
      "symbol": "H",
      "number": 21
    }, {
      "symbol": "N",
      "number": 6
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Nna",
    "name": "N6,N6-dimethyladenosine monophosphate diradical 66A",
    "mf": "C12H16N5O6P",
    "kind": "NucleotideP",
    "oneLetter": "ζ",
    "ocl": {
      "value": "feghs@INBwlJ\\TgHOFwaEqrIQQSYQJIRIMZLyxMVuUUUPLpEUADSDHfP@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtPCQ@D}RFBp"
    },
    "mass": 357.2595904272741,
    "monoisotopicMass": 357.08382025367,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 6
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Fya",
    "name": "N6-formyladenosine monophosphate diradical 67A",
    "mf": "C11H12N5O7P",
    "kind": "NucleotideP",
    "oneLetter": "Ϩ",
    "ocl": {
      "value": "fegis@INBwlJ\\TgLp^MoBKcdRbbfrbTRdRZrYspZmjjjj`Y`JZBHfHQL`@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtPCQ@D}RtP@"
    },
    "mass": 357.216496438634,
    "monoisotopicMass": 357.04743474432,
    "unsaturation": 16,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Hma",
    "name": "N6-hydroxymethyladenosine monophosphate diradical 68A",
    "mf": "C11H14N5O7P",
    "kind": "NucleotideP",
    "oneLetter": "Ϫ",
    "ocl": {
      "value": "fegis@INBwlJ\\TgLp^MoBKcdRbbfrbTRdRZrYspZmjjjj`Y`JjBHfHQL`@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtPCQ@D}RtP@"
    },
    "mass": 359.23237794674554,
    "monoisotopicMass": 359.06308480878,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Cca",
    "name": "cyclic N6-threonylcarbamoyladenosine monophosphate diradical 69A",
    "mf": "C15H17N6O9P",
    "kind": "NucleotideP",
    "oneLetter": "e",
    "ocl": {
      "value": "ehRVIL@IG@fnehJNEDliko`OFspb\\\\bTTTvTRbTbSVTRRtXx|BjZvNjjjj`Y`IjfjbHPfDHSD`z`",
      "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@J_IorHbGtgD}F@RxRH_WwW@hbOTh}RtXCQ`A`l_`A`iVCjKAcjX@A~@h`Bup"
    },
    "mass": 456.30465685593623,
    "monoisotopicMass": 456.07946314904,
    "unsaturation": 20,
    "elements": [{
      "symbol": "C",
      "number": 15
    }, {
      "symbol": "H",
      "number": 17
    }, {
      "symbol": "N",
      "number": 6
    }, {
      "symbol": "O",
      "number": 9
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Fcy",
    "name": "5-formylcytidine monophosphate diradical71C",
    "mf": "C10H12N3O8P",
    "kind": "NucleotideP",
    "oneLetter": ">",
    "ocl": {
      "value": "faspK@I^[BgENSghOFwaEqrIQQSYQJIRYULxDmUUUTsLttQDqBId@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFBpXSU@@tP"
    },
    "mass": 333.1917590433254,
    "monoisotopicMass": 333.03620135502996,
    "unsaturation": 12,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 12
    }, {
      "symbol": "N",
      "number": 3
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Tur",
    "name": "4-thiouridine monophosphate diradical 74U",
    "mf": "C9H11N2O7PS",
    "kind": "NucleotideP",
    "oneLetter": "4",
    "ocl": {
      "value": "ff}Qp`I^aSbgIrCqmxQ\\ZaFQJJJ[JIQJSMg@ejjjjfYihbIbDSH@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFDuP"
    },
    "mass": 322.2317616628973,
    "monoisotopicMass": 322.0024588732,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Meg",
    "name": "7-methylguanosine monophosphate diradical 7G",
    "mf": "C11H15N5O7P",
    "kind": "NucleotideP",
    "oneLetter": "7",
    "ocl": {
      "value": "fegisDINCt\\J\\TgLp^MoBKbF\\bTTTvTRbTbRlSN^CWmUUUUKLuSTQDqBId@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RqdCQ@B\\StP@"
    },
    "mass": 360.2403187008013,
    "monoisotopicMass": 360.07090984101,
    "unsaturation": 13,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 15
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Mea",
    "name": "8-methyladenosine monophosphate diradical 8A",
    "mf": "C11H14N5O6P",
    "kind": "NucleotideP",
    "oneLetter": "â",
    "ocl": {
      "value": "fi{hs@INCt\\J\\UdhOFw`eqrIQQSYQJJJQKigOA[vjjjjAi`J`bIbDSH@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpP@c`a}_S_|BD}RSuKQ@MD@"
    },
    "mass": 343.2329730224273,
    "monoisotopicMass": 343.06817018921,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 14
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 6
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Dhu",
    "name": "dihydrouridine monophosphate diradical 8U",
    "mf": "C9H13N2O8P",
    "kind": "NucleotideP",
    "oneLetter": "D",
    "ocl": {
      "value": "ff}PK@I^aSbgIsTGc[pbxyDhhilheDiLv\\BVjjjjZffbHfHQL`@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFDuP"
    },
    "mass": 308.1822606892002,
    "monoisotopicMass": 308.04095238282997,
    "unsaturation": 8,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Ins",
    "name": "inosine monophosphate diradical 9A",
    "mf": "C10H11N4O7P",
    "kind": "NucleotideP",
    "oneLetter": "I",
    "ocl": {
      "value": "fakIs@INBvENJSghOFwaEqrIQQSYQJIRIMLyxMVuUUUTlsSTQDqBId@@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOq_qopHBGtgD}D@RpPH_Wtw@aOTd}RtPCQ@@"
    },
    "mass": 330.1911165763972,
    "monoisotopicMass": 330.03653570766,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "N",
      "number": 4
    }, {
      "symbol": "O",
      "number": 7
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Pis",
    "name": "pseudouridine monophosphate diradical 9U",
    "mf": "C9H11N2O8P",
    "kind": "NucleotideP",
    "oneLetter": "P",
    "ocl": {
      "value": "ff}PK@OAaSbgIsTGc[pbxyDhhilheDiLv\\BVjjjfZffbHfHQL`@",
      "coordinates": "!BNuSFPDlDTEHt_pHtP@H_TuPBOpcbpXBGtSItuPSU@H_Wtw@`lFDuP"
    },
    "mass": 306.1663791810886,
    "monoisotopicMass": 306.02530231837,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 9
    }, {
      "symbol": "H",
      "number": 11
    }, {
      "symbol": "N",
      "number": 2
    }, {
      "symbol": "O",
      "number": 8
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Pqb",
    "name": "preQ0base 100G diradical (base)",
    "mf": "C7H5N5O",
    "kind": "Nucleotide",
    "oneLetter": "ψ",
    "ocl": {
      "value": "dk^h@DxYLLbbTTRekiujYj^`@",
      "coordinates": "!B|Gwp_Gy|Gwp_[lk_gp_Ag_wrYRs}|f"
    },
    "mass": 175.1477760289729,
    "monoisotopicMass": 175.04940980287,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 7
    }, {
      "symbol": "H",
      "number": 5
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Pqg",
    "name": "preQ1base 101G diradical (base)",
    "mf": "C7H9N5O",
    "kind": "Nucleotide",
    "oneLetter": "∇",
    "ocl": {
      "value": "dk^h@DxYLLbbTTRckiUjYij`@",
      "coordinates": "!BWyfe[tlDWye_fXx@RpRe[wtHSuHH@a}"
    },
    "mass": 179.179539045196,
    "monoisotopicMass": 179.08070993179,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 7
    }, {
      "symbol": "H",
      "number": 9
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Qba",
    "name": "Qbase 10G diradical (base)",
    "mf": "C12H15N5O3",
    "kind": "Nucleotide",
    "oneLetter": "∴",
    "ocl": {
      "value": "fbmi`@D\\EHpHyrJIQQJMJIPtyIPTmSMMUMUP@@",
      "coordinates": "!BRpQ_f^i`RpQKAEARzfA_f_pHtP@H_Pc|BGuPThxUCl{RtBYTd|"
    },
    "mass": 277.27967290184347,
    "monoisotopicMass": 277.11748936431,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 15
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 3
    }]
  }, {
    "symbol": "Dgc",
    "name": "N2,7-dimethylguanosine cap (cap DMG) diradical 279553N",
    "mf": "C12H18N5O11P2",
    "kind": "Nucleotide",
    "oneLetter": "®",
    "ocl": {
      "value": "e`TZMBHIG@aihJNEHdlemck`OFspz|OgDJ\\bTTTvTRbTbRvbtfKGG`UPuUUUUJtuTmUTPaLHPfH@@",
      "coordinates": "!BvuPfpDnDtEK_t_rHtXBH_TwPbOr_IorHbGtgD}F@RxS|uxc|_]^OTh}RIlBH_]F@IqOQ`@A~_c|bH}RbGt"
    },
    "mass": 470.24625855539705,
    "monoisotopicMass": 470.04780541440005,
    "unsaturation": 13,
    "elements": [{
      "symbol": "C",
      "number": 12
    }, {
      "symbol": "H",
      "number": 18
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 11
    }, {
      "symbol": "P",
      "number": 2
    }]
  }, {
    "symbol": "Dpa",
    "name": "5′-(3′-dephosphoacetyl-CoA) diradical 4155N",
    "mf": "C23H35N7O16P3S",
    "kind": "Nucleotide",
    "oneLetter": "♣",
    "ocl": {
      "value": "elz~@jDCHlemnSTLBAEKBjfckgbV]XpEfCpB|IoCtHZy{lbdvbbfrbTRdRNRdnTbefRTrRTdTRrFVfjjjj`V`bZjjfjZjZ`bbLSaRP@",
      "coordinates": "!BvtmKaMmKUMlfgto[tDw_cosWt]~H@dvObGv_F_sWbOpgKMG_R}m}bHa}HbOSX}M_cQw}G_OwzH_[wW_c~H_Wx@G{|bM]}bGvHGxbGu~Oxa}bOq~Oxa}_c~H_WxuwvH_P"
    },
    "mass": 790.5483266874629,
    "monoisotopicMass": 790.1073852418399,
    "unsaturation": 21,
    "elements": [{
      "symbol": "C",
      "number": 23
    }, {
      "symbol": "H",
      "number": 35
    }, {
      "symbol": "N",
      "number": 7
    }, {
      "symbol": "O",
      "number": 16
    }, {
      "symbol": "P",
      "number": 3
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Dpm",
    "name": "5′-(3′-dephosphomalonyl-CoA) diradical 4255N",
    "mf": "C24H35N7O18P3S",
    "kind": "Nucleotide",
    "oneLetter": "♥",
    "ocl": {
      "value": "efq~DjDCHlemnSTLBAEKBjfckgbV]XrzpEfCpB|IoCtHZy{lbdvbbfrbTRdRNRdnTbefRTrRTrdbbVPrtuUUUTBtDSUUTuSUSSTDTQb\\JR@@",
      "coordinates": "!BIlB_Ib[@pAe`zni`FALSF@A~FBq~OrpXbGveX@A~_c~OTa`lzf@_ha}_]_Q`MF@bOpXKA`loXbH__rHb@JHoX`B@m]}uwx@bGu~Ox`BbKvH@ha}_c~H@hb}b@JH_Xc|_`BH_X`B_`BHoP"
    },
    "mass": 834.5578724328346,
    "monoisotopicMass": 834.0972144809799,
    "unsaturation": 23,
    "elements": [{
      "symbol": "C",
      "number": 24
    }, {
      "symbol": "H",
      "number": 35
    }, {
      "symbol": "N",
      "number": 7
    }, {
      "symbol": "O",
      "number": 18
    }, {
      "symbol": "P",
      "number": 3
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Dsc",
    "name": "5′-(3′-dephosphosuccinyl-CoA) radical 4355N",
    "mf": "C25H37N7O18P3S",
    "kind": "Nucleotide",
    "oneLetter": "♦",
    "ocl": {
      "value": "eny~DjDCHlemnSTLBAEKBjfckgbV]XzvpOFCpB|IoCtHZy{lbdvbbfrbTRdRNRdnTbefRTrRTrTdTRrFVfjjjj`V`bZjjfjZjZfhHhcDxTd@@",
      "coordinates": "!B[~kjXFjiV[Ry|fcm}MtGwWctvH_]Q_c}KaGwWbGvN`H}MgrX@_gx@h`gKB\\lbGvOSX}M@m^H@gwWbGvH@ha}_Xc|bGxb@I~@Ha}b@JH_X`B_`BH_X`BbGvH@ha}_c~H@ha}b@I~@Ha}"
    },
    "mass": 848.5844898376815,
    "monoisotopicMass": 848.11286454544,
    "unsaturation": 23,
    "elements": [{
      "symbol": "C",
      "number": 25
    }, {
      "symbol": "H",
      "number": 37
    }, {
      "symbol": "N",
      "number": 7
    }, {
      "symbol": "O",
      "number": 18
    }, {
      "symbol": "P",
      "number": 3
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Dpc",
    "name": "5′-(3′-dephospho-CoA) radical 455N",
    "mf": "C21H32N7O13P2S",
    "kind": "Nucleotide",
    "oneLetter": "♠",
    "ocl": {
      "value": "ek_^KBDIG@nabYXJNEHdliemh\\QPEfspZ|CPcKmnrIQQSYQJIRIGIRWJQRsIJYIccpJkjjjjjAZBIjjjZijjBDIaBDq@@",
      "coordinates": "!B[zW[UI|YchAMc{vHcuJH@m~NbGuKvwvHb@JNwx}Rgqe}bHa}@h`gDr\\Sb@JOTh}R@m]~@@A~b@I~@H`B_X`_hb}_`CW@h`B_`BH@gx@upJH@gx@b@I~@@"
    },
    "mass": 684.5310558604504,
    "monoisotopicMass": 684.1254042880199,
    "unsaturation": 19,
    "elements": [{
      "symbol": "C",
      "number": 21
    }, {
      "symbol": "H",
      "number": 32
    }, {
      "symbol": "N",
      "number": 7
    }, {
      "symbol": "O",
      "number": 13
    }, {
      "symbol": "P",
      "number": 2
    }, {
      "symbol": "S",
      "number": 1
    }]
  }, {
    "symbol": "Dpe",
    "name": "5′-diphosphate end 552N",
    "mf": "O3P",
    "kind": "Nucleotide",
    "oneLetter": "ϒ",
    "ocl": {
      "value": "gJQdebGF^Dx|duK@@",
      "coordinates": "!BbOq~@GxbGt"
    },
    "mass": 78.97197677137483,
    "monoisotopicMass": 78.95850585713,
    "unsaturation": 1,
    "elements": [{
      "symbol": "O",
      "number": 3
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Mgc",
    "name": "7-methylguanosine cap (cap 0) diradical 79553N",
    "mf": "C11H16N5O11P2",
    "kind": "Nucleotide",
    "oneLetter": "©",
    "ocl": {
      "value": "eohZMBHIG@aihJNEHdlemck`OFspz|GgDJ\\bTTTvTRbTbRvbtcXx|BjFjjjjiVfjejjHPfDHSD@@",
      "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@J_I`JHbGtgD}F@RxPBux`B_]^OTh}RIlBH_]F@IqOQ`@A~_c|BbHa}"
    },
    "mass": 456.2196411505502,
    "monoisotopicMass": 456.03215534994,
    "unsaturation": 13,
    "elements": [{
      "symbol": "C",
      "number": 11
    }, {
      "symbol": "H",
      "number": 16
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 11
    }, {
      "symbol": "P",
      "number": 2
    }]
  }, {
    "symbol": "Gyy",
    "name": "guanylylated 5′ end (cap G) diradical 9553N",
    "mf": "C10H13N5O10P2",
    "kind": "Nucleotide",
    "oneLetter": "ϑ",
    "ocl": {
      "value": "fkhh`INCt\\J\\UENY{NCqmxM|EnNQJJJ[JIQJQHzIRLyxM^uUUUTkSULuQDSDHfP@",
      "coordinates": "!BvuPfpDnDtEK_tPJHtXBH_TwPb@J_I`JHbGtgD}F@RxPBux`B_]^OTh}R_`CQ`B\\StXA~@C}~@Gx"
    },
    "mass": 425.1856780673293,
    "monoisotopicMass": 425.01376563368,
    "unsaturation": 14,
    "elements": [{
      "symbol": "C",
      "number": 10
    }, {
      "symbol": "H",
      "number": 13
    }, {
      "symbol": "N",
      "number": 5
    }, {
      "symbol": "O",
      "number": 10
    }, {
      "symbol": "P",
      "number": 2
    }]
  }, {
    "symbol": "Furp",
    "name": "furan phosphate radical",
    "mf": "C5H6O4P",
    "kind": "RNAp",
    "oneLetter": "⬠",
    "ocl": {
      "value": "dmtBPDpnAYcpRZ}eeYjii@@",
      "coordinates": "!BNvw|Vso|kUl{[So|PPAGuU\\z`pP"
    },
    "mass": 161.072705703704,
    "monoisotopicMass": 161.00037067008,
    "unsaturation": 5,
    "elements": [{
      "symbol": "C",
      "number": 5
    }, {
      "symbol": "H",
      "number": 6
    }, {
      "symbol": "O",
      "number": 4
    }, {
      "symbol": "P",
      "number": 1
    }]
  }, {
    "symbol": "Phg",
    "ocl": {
      "value": "dcNHPBPOEgEInVuWYj`@@@"
    },
    "name": "Phenyl glycine diradical",
    "mf": "C8H7NO",
    "kind": "aa",
    "mass": 133.1475805880365,
    "monoisotopicMass": 133.05276384961002,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 8
    }, {
      "symbol": "H",
      "number": 7
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 1
    }]
  }, {
    "symbol": "Hpg",
    "ocl": {
      "value": "dknDPBPp|V\\Tfy[WWYj`@`@@",
      "coordinates": "!Bb@K~@Hc}b@K~_xc}bOvw?_y??g~w?_y?"
    },
    "name": "Hydroxy phenyl glycine diradical",
    "mf": "C8H7NO2",
    "kind": "aa",
    "mass": 149.14698551235477,
    "monoisotopicMass": 149.04767846918,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 8
    }, {
      "symbol": "H",
      "number": 7
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 2
    }]
  }, {
    "symbol": "Dpg",
    "ocl": {
      "value": "dg^LPBS[CqYqR[emYTyj`BH@@",
      "coordinates": "!BbOq~@Ha}bOq~Oxa}bGwW_Wx?_c}H_]]}M?t"
    },
    "name": "Dihydroxy phenyl glycine diradical",
    "mf": "C8H7NO3",
    "kind": "aa",
    "mass": 165.14639043667304,
    "monoisotopicMass": 165.04259308875,
    "unsaturation": 10,
    "elements": [{
      "symbol": "C",
      "number": 8
    }, {
      "symbol": "H",
      "number": 7
    }, {
      "symbol": "N",
      "number": 1
    }, {
      "symbol": "O",
      "number": 3
    }]
  }];

  const groupsObject = {};
  groups$2.forEach(e => {
    if (groupsObject[e.symbol]) {
      console.log('The symbol ' + e.symbol + ' is used more than once');
    }

    groupsObject[e.symbol] = e;
  });
  var groupsObject_1 = groupsObject;

  const elements$2 = src$j.elementsAndStableIsotopesObject;

  function getIsotopeRatioInfo(value) {
    let result = {
      mass: 0,
      monoisotopicMass: 0
    };
    let element = elements$2[value.atom];
    if (!element) throw new Error(`Element not found: ${value.atom}`);
    let isotopesArray = element.isotopes;
    let ratios = normalize$4(value.ratio);
    let max = Math.max(...ratios);

    if (ratios.length > isotopesArray.length) {
      throw new Error(`the number of specified ratios is bigger that the number of stable isotopes: ${value.atom}`);
    }

    for (let i = 0; i < ratios.length; i++) {
      result.mass += ratios[i] * isotopesArray[i].mass;

      if (max === ratios[i] && result.monoisotopicMass === 0) {
        result.monoisotopicMass = isotopesArray[i].mass;
      }
    }

    return result;
  }

  function normalize$4(array) {
    let sum = array.reduce((prev, current) => prev + current, 0);
    return array.map(a => a / sum);
  }

  var getIsotopeRatioInfo_1 = getIsotopeRatioInfo;

  const isotopes = {};
  Object.keys(elementsAndIsotopesObject_1).forEach(key => {
    let e = elementsAndIsotopesObject_1[key];
    e.isotopes.forEach(i => {
      isotopes[i.nominal + key] = {
        abundance: i.abundance,
        mass: i.mass
      };
    });
  });
  var getIsotopesObject = isotopes;

  const elements$1 = src$j.elementsObject;
  /**
   *
   * @param {*} parts
   * @param {*} [options={}]
   */

  var getEA = function getEA(parts) {
    let results = {};

    for (let part of parts) {
      for (let line of part) {
        switch (line.kind) {
          case Kind.ISOTOPE:
            {
              let isotope = getIsotopesObject[line.value.isotope + line.value.atom];

              if (!isotope) {
                throw new Error(`Unknown isotope: ${line.value.isotope}${line.value.atom}`);
              }

              addMass(results, line.value.atom, isotope.mass * line.multiplier);
              break;
            }

          case Kind.ISOTOPE_RATIO:
            {
              let isotopeRatioInfo = getIsotopeRatioInfo_1(line.value);
              addMass(results, line.value.atom, isotopeRatioInfo.mass * line.multiplier);
              break;
            }

          case Kind.ATOM:
            {
              let element = elements$1[line.value];

              if (!element) {
                element = groupsObject_1[line.value];
                if (!element) throw Error(`Unknown element: ${line.value}`); // need to explode group ????
              }

              addMass(results, line.value, element.mass * line.multiplier);
              break;
            }

          case Kind.CHARGE:
            break;

          default:
            throw new Error('partToMF unhandled Kind: ', line.kind);
        }
      }
    }

    let eas = [];
    let sum = 0;

    for (let key in results) {
      sum += results[key];
      eas.push({
        element: key,
        mass: results[key]
      });
    }

    eas.forEach(ea => {
      ea.ratio = ea.mass / sum;
    });
    return eas;
  };

  function addMass(results, atom, mass) {
    if (!results[atom]) results[atom] = 0;
    results[atom] += mass;
  }

  const {
    elementsObject,
    elementsAndIsotopesObject
  } = src$j;
  /**
   *
   * @param {*} parts
   * @param {*} [options={}]
   */

  var getElements = function getElements(parts) {
    const elements = [];

    for (const part of parts) {
      for (const line of part) {
        let number = line.multiplier;

        switch (line.kind) {
          case Kind.ATOM:
            {
              let symbol = line.value;
              let element = elementsObject[symbol];

              if (!element) {
                throw new Error(`element unknown: ${symbol} - ${line}`);
              }

              addElement(elements, {
                symbol,
                number
              });
              break;
            }

          case Kind.ISOTOPE:
            {
              let element = elementsAndIsotopesObject[line.value.atom];

              if (!element) {
                throw new Error(`element unknown: ${part.value.atom} - ${line}`);
              }

              let isotope = element.isotopes.filter(a => a.nominal === line.value.isotope)[0];

              if (!isotope) {
                throw new Error(`isotope unknown: ${line.value.isotope} - ${line}`);
              }

              addElement(elements, {
                symbol: line.value.atom,
                number,
                isotope: line.value.isotope
              });
              break;
            }

          default:
            throw new Error(`unknown type: ${line.kind}`);
        }
      }
    }

    return elements;
  };

  function addElement(elements, newElement) {
    for (let element of elements) {
      if (element.symbol === newElement.symbol && element.isotope === newElement.isotope) {
        element.number += newElement.number;
        return;
      }
    }

    elements.push(newElement);
  }

  var unsaturationsObject = {
    O: 0,
    N: 1,
    H: -1,
    Na: -1,
    K: -1,
    Li: -1,
    Ca: -2,
    C: 2,
    F: -1,
    Si: 2,
    Cl: -1,
    Br: -1,
    I: -1,
    S: 0,
    P: 1
  };

  /**
   * Convert a MF part to an array of atoms
   * This procedure will suppress the isotopes !
   * This is mainly used to make queries
   */


  var partToAtoms = function partToAtoms(part) {
    let atoms = {};

    for (let line of part) {
      switch (line.kind) {
        case Kind.ISOTOPE:
          if (!atoms[line.value.atom]) atoms[line.value.atom] = 0;
          atoms[line.value.atom] += line.multiplier;
          break;

        case Kind.ISOTOPE_RATIO:
          if (!atoms[line.value.atom]) atoms[line.value.atom] = 0;
          atoms[line.value.atom] += line.multiplier;
          break;

        case Kind.ATOM:
          if (!atoms[line.value]) atoms[line.value] = 0;
          atoms[line.value] += line.multiplier;
          break;

        case Kind.CHARGE:
          break;

        case Kind.ANCHOR:
          break;

        default:
          throw new Error('partToMF unhandled Kind: ', line.kind);
      }
    }

    return atoms;
  };

  var partToMF = function partToMF(part, options = {}) {
    let mf = [];

    for (let line of part) {
      switch (line.kind) {
        case Kind.ISOTOPE:
          if (line.multiplier !== 0)