/**
 * file-collection - Library to manage large amount of files coming from different sources
 * @version v2.0.0
 * @link https://github.com/cheminfo/file-collection#readme
 * @license MIT
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.FileCollection = {}));
})(this, (function (exports) { 'use strict';

    class CachedFileItem {
      fileItem;
      textCache;
      arrayBufferCache;
      constructor(fileItem) {
        this.fileItem = fileItem;
      }
      async text() {
        if (this.arrayBufferCache) {
          return new TextDecoder().decode(this.arrayBufferCache);
        }
        return this.textCache = this.textCache || (await this.fileItem.text());
      }
      async arrayBuffer() {
        return this.arrayBufferCache = this.arrayBufferCache || (await this.fileItem.arrayBuffer());
      }
      stream() {
        return this.fileItem.stream(); // no cache for stream !
      }
      get lastModified() {
        return this.fileItem.lastModified;
      }
      get name() {
        return this.fileItem.name;
      }
      get relativePath() {
        return this.fileItem.relativePath;
      }
      get size() {
        return this.fileItem.size;
      }
      get sourceUUID() {
        return this.fileItem.sourceUUID;
      }
    }

    function getNameInfo(name) {
      const url = new URL(name, 'ium:/');
      return {
        url,
        name: url.pathname.split('/').pop(),
        relativePath: url.pathname
      };
    }

    async function appendArrayBuffer(fileCollection, relativePath, arrayBuffer, options = {}) {
      const source = await getExtendedSourceFromArrayBuffer(relativePath, arrayBuffer, options);
      await fileCollection.appendExtendedSource(source);
    }
    async function getExtendedSourceFromArrayBuffer(originalRelativePath, arrayBuffer, options = {}) {
      const {
        name,
        relativePath
      } = getNameInfo(originalRelativePath);
      const blob = new Blob([await arrayBuffer], {
        type: 'application/octet-stream'
      });
      return {
        uuid: crypto.randomUUID(),
        relativePath,
        name,
        lastModified: options.dateModified || Date.now(),
        baseURL: 'ium:/',
        text: () => blob.text(),
        stream: () => blob.stream(),
        arrayBuffer: () => blob.arrayBuffer()
      };
    }

    async function appendFileList(fileCollection, fileList) {
      const promises = [];
      for (const file of fileList) {
        const source = {
          uuid: crypto.randomUUID(),
          name: file.name,
          size: file.size,
          baseURL: 'ium:/',
          //@ts-expect-error We allow file.path as alternative to webkitRelativePath
          relativePath: file.webkitRelativePath || file.path || file.name,
          lastModified: file.lastModified,
          text: () => file.text(),
          arrayBuffer: () => file.arrayBuffer(),
          stream: () => file.stream()
        };
        promises.push(fileCollection.appendExtendedSource(source));
      }
      await Promise.all(promises);
    }

    function appendPath() {
      throw new Error('This method is not available in the browser');
    }

    /**
     * Utility function that allows to filter files from a FileCollection ignore by default the dotFiles
     * @param fileCollection
     * @param relativePath
     * @param options
     * @returns
     */
    function shouldAddItem(relativePath, options = {}) {
      const {
        ignoreDotfiles = true
      } = options;
      if (ignoreDotfiles) {
        return relativePath.split('/').filter(part => part.startsWith('.')).length === 0;
      }
      return true;
    }

    function sourceItemToExtendedSourceItem(entry, alternativeBaseURL) {
      let baseURL = entry.baseURL || alternativeBaseURL;
      if (!baseURL) {
        if (!globalThis.location?.href) {
          throw new Error(`We could not find a baseURL for ${entry.relativePath}`);
        } else {
          baseURL = globalThis.location.href;
        }
      }
      const fileURL = new URL(entry.relativePath, baseURL);
      const source = {
        uuid: crypto.randomUUID(),
        name: entry.relativePath.split('/').pop() || '',
        size: entry.size,
        baseURL,
        relativePath: entry.relativePath,
        lastModified: entry.lastModified,
        text: async () => {
          const response = await fetch(fileURL.toString());
          return response.text();
        },
        arrayBuffer: async () => {
          const response = await fetch(fileURL.toString());
          return response.arrayBuffer();
        },
        // @ts-expect-error Should contain stream
        stream: async () => {
          const response = await fetch(fileURL.toString());
          //@ts-expect-error Should contain stream
          return response.stream();
        }
      };
      return source;
    }

    async function appendSource(fileCollection, source, options = {}) {
      const {
        filter
      } = fileCollection;
      const {
        entries,
        baseURL
      } = source;
      const promises = [];
      for (const entry of entries) {
        const {
          relativePath
        } = entry;
        if (!shouldAddItem(relativePath, filter)) continue;
        const alternativeBaseURL = baseURL || options.baseURL;
        const source = sourceItemToExtendedSourceItem(entry, alternativeBaseURL);
        promises.push(fileCollection.appendExtendedSource(source));
      }
      await Promise.all(promises);
    }

    async function appendText(fileCollection, relativePath, text, options = {}) {
      const source = await getExtendedSourceFromText(relativePath, text, options);
      await fileCollection.appendExtendedSource(source);
    }
    async function getExtendedSourceFromText(originalRelativePath, text, options = {}) {
      const {
        relativePath,
        name
      } = getNameInfo(originalRelativePath);
      const blob = new Blob([await text], {
        type: 'text/plain'
      });
      return {
        uuid: crypto.randomUUID(),
        relativePath,
        name,
        lastModified: options.dateModified || Date.now(),
        baseURL: 'ium:/',
        text: () => blob.text(),
        stream: () => blob.stream(),
        arrayBuffer: () => blob.arrayBuffer()
      };
    }

    async function appendWebSource(fileCollection, webSourceURL, options = {}) {
      const response = await fetch(webSourceURL);
      const webSource = await response.json();
      return appendSource(fileCollection, webSource, {
        baseURL: webSourceURL,
        ...options
      });
    }

    var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

    function getDefaultExportFromCjs (x) {
    	return x && x.__esModule && Object.prototype.hasOwnProperty.call(x, 'default') ? x['default'] : x;
    }

    function commonjsRequire(path) {
    	throw new Error('Could not dynamically require "' + path + '". Please configure the dynamicRequireTargets or/and ignoreDynamicRequires option of @rollup/plugin-commonjs appropriately for this require call to work.');
    }

    var jszip_min = {exports: {}};

    /*!

    JSZip v3.10.1 - A JavaScript class for generating and reading zip files
    <http://stuartk.com/jszip>

    (c) 2009-2016 Stuart Knightley <stuart [at] stuartk.com>
    Dual licenced under the MIT license or GPLv3. See https://raw.github.com/Stuk/jszip/main/LICENSE.markdown.

    JSZip uses the library pako released under the MIT license :
    https://github.com/nodeca/pako/blob/main/LICENSE
    */
    (function (module, exports) {
      !function (e) {
        module.exports = e();
      }(function () {
        return function s(a, o, h) {
          function u(r, e) {
            if (!o[r]) {
              if (!a[r]) {
                var t = "function" == typeof commonjsRequire && commonjsRequire;
                if (!e && t) return t(r, true);
                if (l) return l(r, true);
                var n = new Error("Cannot find module '" + r + "'");
                throw n.code = "MODULE_NOT_FOUND", n;
              }
              var i = o[r] = {
                exports: {}
              };
              a[r][0].call(i.exports, function (e) {
                var t = a[r][1][e];
                return u(t || e);
              }, i, i.exports, s, a, o, h);
            }
            return o[r].exports;
          }
          for (var l = "function" == typeof commonjsRequire && commonjsRequire, e = 0; e < h.length; e++) u(h[e]);
          return u;
        }({
          1: [function (e, t, r) {

            var d = e("./utils"),
              c = e("./support"),
              p = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";
            r.encode = function (e) {
              for (var t, r, n, i, s, a, o, h = [], u = 0, l = e.length, f = l, c = "string" !== d.getTypeOf(e); u < e.length;) f = l - u, n = c ? (t = e[u++], r = u < l ? e[u++] : 0, u < l ? e[u++] : 0) : (t = e.charCodeAt(u++), r = u < l ? e.charCodeAt(u++) : 0, u < l ? e.charCodeAt(u++) : 0), i = t >> 2, s = (3 & t) << 4 | r >> 4, a = 1 < f ? (15 & r) << 2 | n >> 6 : 64, o = 2 < f ? 63 & n : 64, h.push(p.charAt(i) + p.charAt(s) + p.charAt(a) + p.charAt(o));
              return h.join("");
            }, r.decode = function (e) {
              var t,
                r,
                n,
                i,
                s,
                a,
                o = 0,
                h = 0,
                u = "data:";
              if (e.substr(0, u.length) === u) throw new Error("Invalid base64 input, it looks like a data url.");
              var l,
                f = 3 * (e = e.replace(/[^A-Za-z0-9+/=]/g, "")).length / 4;
              if (e.charAt(e.length - 1) === p.charAt(64) && f--, e.charAt(e.length - 2) === p.charAt(64) && f--, f % 1 != 0) throw new Error("Invalid base64 input, bad content length.");
              for (l = c.uint8array ? new Uint8Array(0 | f) : new Array(0 | f); o < e.length;) t = p.indexOf(e.charAt(o++)) << 2 | (i = p.indexOf(e.charAt(o++))) >> 4, r = (15 & i) << 4 | (s = p.indexOf(e.charAt(o++))) >> 2, n = (3 & s) << 6 | (a = p.indexOf(e.charAt(o++))), l[h++] = t, 64 !== s && (l[h++] = r), 64 !== a && (l[h++] = n);
              return l;
            };
          }, {
            "./support": 30,
            "./utils": 32
          }],
          2: [function (e, t, r) {

            var n = e("./external"),
              i = e("./stream/DataWorker"),
              s = e("./stream/Crc32Probe"),
              a = e("./stream/DataLengthProbe");
            function o(e, t, r, n, i) {
              this.compressedSize = e, this.uncompressedSize = t, this.crc32 = r, this.compression = n, this.compressedContent = i;
            }
            o.prototype = {
              getContentWorker: function () {
                var e = new i(n.Promise.resolve(this.compressedContent)).pipe(this.compression.uncompressWorker()).pipe(new a("data_length")),
                  t = this;
                return e.on("end", function () {
                  if (this.streamInfo.data_length !== t.uncompressedSize) throw new Error("Bug : uncompressed data size mismatch");
                }), e;
              },
              getCompressedWorker: function () {
                return new i(n.Promise.resolve(this.compressedContent)).withStreamInfo("compressedSize", this.compressedSize).withStreamInfo("uncompressedSize", this.uncompressedSize).withStreamInfo("crc32", this.crc32).withStreamInfo("compression", this.compression);
              }
            }, o.createWorkerFrom = function (e, t, r) {
              return e.pipe(new s()).pipe(new a("uncompressedSize")).pipe(t.compressWorker(r)).pipe(new a("compressedSize")).withStreamInfo("compression", t);
            }, t.exports = o;
          }, {
            "./external": 6,
            "./stream/Crc32Probe": 25,
            "./stream/DataLengthProbe": 26,
            "./stream/DataWorker": 27
          }],
          3: [function (e, t, r) {

            var n = e("./stream/GenericWorker");
            r.STORE = {
              magic: "\0\0",
              compressWorker: function () {
                return new n("STORE compression");
              },
              uncompressWorker: function () {
                return new n("STORE decompression");
              }
            }, r.DEFLATE = e("./flate");
          }, {
            "./flate": 7,
            "./stream/GenericWorker": 28
          }],
          4: [function (e, t, r) {

            var n = e("./utils");
            var o = function () {
              for (var e, t = [], r = 0; r < 256; r++) {
                e = r;
                for (var n = 0; n < 8; n++) e = 1 & e ? 3988292384 ^ e >>> 1 : e >>> 1;
                t[r] = e;
              }
              return t;
            }();
            t.exports = function (e, t) {
              return void 0 !== e && e.length ? "string" !== n.getTypeOf(e) ? function (e, t, r, n) {
                var i = o,
                  s = n + r;
                e ^= -1;
                for (var a = n; a < s; a++) e = e >>> 8 ^ i[255 & (e ^ t[a])];
                return -1 ^ e;
              }(0 | t, e, e.length, 0) : function (e, t, r, n) {
                var i = o,
                  s = n + r;
                e ^= -1;
                for (var a = n; a < s; a++) e = e >>> 8 ^ i[255 & (e ^ t.charCodeAt(a))];
                return -1 ^ e;
              }(0 | t, e, e.length, 0) : 0;
            };
          }, {
            "./utils": 32
          }],
          5: [function (e, t, r) {

            r.base64 = false, r.binary = false, r.dir = false, r.createFolders = true, r.date = null, r.compression = null, r.compressionOptions = null, r.comment = null, r.unixPermissions = null, r.dosPermissions = null;
          }, {}],
          6: [function (e, t, r) {

            var n = null;
            n = "undefined" != typeof Promise ? Promise : e("lie"), t.exports = {
              Promise: n
            };
          }, {
            lie: 37
          }],
          7: [function (e, t, r) {

            var n = "undefined" != typeof Uint8Array && "undefined" != typeof Uint16Array && "undefined" != typeof Uint32Array,
              i = e("pako"),
              s = e("./utils"),
              a = e("./stream/GenericWorker"),
              o = n ? "uint8array" : "array";
            function h(e, t) {
              a.call(this, "FlateWorker/" + e), this._pako = null, this._pakoAction = e, this._pakoOptions = t, this.meta = {};
            }
            r.magic = "\b\0", s.inherits(h, a), h.prototype.processChunk = function (e) {
              this.meta = e.meta, null === this._pako && this._createPako(), this._pako.push(s.transformTo(o, e.data), false);
            }, h.prototype.flush = function () {
              a.prototype.flush.call(this), null === this._pako && this._createPako(), this._pako.push([], true);
            }, h.prototype.cleanUp = function () {
              a.prototype.cleanUp.call(this), this._pako = null;
            }, h.prototype._createPako = function () {
              this._pako = new i[this._pakoAction]({
                raw: true,
                level: this._pakoOptions.level || -1
              });
              var t = this;
              this._pako.onData = function (e) {
                t.push({
                  data: e,
                  meta: t.meta
                });
              };
            }, r.compressWorker = function (e) {
              return new h("Deflate", e);
            }, r.uncompressWorker = function () {
              return new h("Inflate", {});
            };
          }, {
            "./stream/GenericWorker": 28,
            "./utils": 32,
            pako: 38
          }],
          8: [function (e, t, r) {

            function A(e, t) {
              var r,
                n = "";
              for (r = 0; r < t; r++) n += String.fromCharCode(255 & e), e >>>= 8;
              return n;
            }
            function n(e, t, r, n, i, s) {
              var a,
                o,
                h = e.file,
                u = e.compression,
                l = s !== O.utf8encode,
                f = I.transformTo("string", s(h.name)),
                c = I.transformTo("string", O.utf8encode(h.name)),
                d = h.comment,
                p = I.transformTo("string", s(d)),
                m = I.transformTo("string", O.utf8encode(d)),
                _ = c.length !== h.name.length,
                g = m.length !== d.length,
                b = "",
                v = "",
                y = "",
                w = h.dir,
                k = h.date,
                x = {
                  crc32: 0,
                  compressedSize: 0,
                  uncompressedSize: 0
                };
              t && !r || (x.crc32 = e.crc32, x.compressedSize = e.compressedSize, x.uncompressedSize = e.uncompressedSize);
              var S = 0;
              t && (S |= 8), l || !_ && !g || (S |= 2048);
              var z = 0,
                C = 0;
              w && (z |= 16), "UNIX" === i ? (C = 798, z |= function (e, t) {
                var r = e;
                return e || (r = t ? 16893 : 33204), (65535 & r) << 16;
              }(h.unixPermissions, w)) : (C = 20, z |= function (e) {
                return 63 & (e || 0);
              }(h.dosPermissions)), a = k.getUTCHours(), a <<= 6, a |= k.getUTCMinutes(), a <<= 5, a |= k.getUTCSeconds() / 2, o = k.getUTCFullYear() - 1980, o <<= 4, o |= k.getUTCMonth() + 1, o <<= 5, o |= k.getUTCDate(), _ && (v = A(1, 1) + A(B(f), 4) + c, b += "up" + A(v.length, 2) + v), g && (y = A(1, 1) + A(B(p), 4) + m, b += "uc" + A(y.length, 2) + y);
              var E = "";
              return E += "\n\0", E += A(S, 2), E += u.magic, E += A(a, 2), E += A(o, 2), E += A(x.crc32, 4), E += A(x.compressedSize, 4), E += A(x.uncompressedSize, 4), E += A(f.length, 2), E += A(b.length, 2), {
                fileRecord: R.LOCAL_FILE_HEADER + E + f + b,
                dirRecord: R.CENTRAL_FILE_HEADER + A(C, 2) + E + A(p.length, 2) + "\0\0\0\0" + A(z, 4) + A(n, 4) + f + b + p
              };
            }
            var I = e("../utils"),
              i = e("../stream/GenericWorker"),
              O = e("../utf8"),
              B = e("../crc32"),
              R = e("../signature");
            function s(e, t, r, n) {
              i.call(this, "ZipFileWorker"), this.bytesWritten = 0, this.zipComment = t, this.zipPlatform = r, this.encodeFileName = n, this.streamFiles = e, this.accumulate = false, this.contentBuffer = [], this.dirRecords = [], this.currentSourceOffset = 0, this.entriesCount = 0, this.currentFile = null, this._sources = [];
            }
            I.inherits(s, i), s.prototype.push = function (e) {
              var t = e.meta.percent || 0,
                r = this.entriesCount,
                n = this._sources.length;
              this.accumulate ? this.contentBuffer.push(e) : (this.bytesWritten += e.data.length, i.prototype.push.call(this, {
                data: e.data,
                meta: {
                  currentFile: this.currentFile,
                  percent: r ? (t + 100 * (r - n - 1)) / r : 100
                }
              }));
            }, s.prototype.openedSource = function (e) {
              this.currentSourceOffset = this.bytesWritten, this.currentFile = e.file.name;
              var t = this.streamFiles && !e.file.dir;
              if (t) {
                var r = n(e, t, false, this.currentSourceOffset, this.zipPlatform, this.encodeFileName);
                this.push({
                  data: r.fileRecord,
                  meta: {
                    percent: 0
                  }
                });
              } else this.accumulate = true;
            }, s.prototype.closedSource = function (e) {
              this.accumulate = false;
              var t = this.streamFiles && !e.file.dir,
                r = n(e, t, true, this.currentSourceOffset, this.zipPlatform, this.encodeFileName);
              if (this.dirRecords.push(r.dirRecord), t) this.push({
                data: function (e) {
                  return R.DATA_DESCRIPTOR + A(e.crc32, 4) + A(e.compressedSize, 4) + A(e.uncompressedSize, 4);
                }(e),
                meta: {
                  percent: 100
                }
              });else for (this.push({
                data: r.fileRecord,
                meta: {
                  percent: 0
                }
              }); this.contentBuffer.length;) this.push(this.contentBuffer.shift());
              this.currentFile = null;
            }, s.prototype.flush = function () {
              for (var e = this.bytesWritten, t = 0; t < this.dirRecords.length; t++) this.push({
                data: this.dirRecords[t],
                meta: {
                  percent: 100
                }
              });
              var r = this.bytesWritten - e,
                n = function (e, t, r, n, i) {
                  var s = I.transformTo("string", i(n));
                  return R.CENTRAL_DIRECTORY_END + "\0\0\0\0" + A(e, 2) + A(e, 2) + A(t, 4) + A(r, 4) + A(s.length, 2) + s;
                }(this.dirRecords.length, r, e, this.zipComment, this.encodeFileName);
              this.push({
                data: n,
                meta: {
                  percent: 100
                }
              });
            }, s.prototype.prepareNextSource = function () {
              this.previous = this._sources.shift(), this.openedSource(this.previous.streamInfo), this.isPaused ? this.previous.pause() : this.previous.resume();
            }, s.prototype.registerPrevious = function (e) {
              this._sources.push(e);
              var t = this;
              return e.on("data", function (e) {
                t.processChunk(e);
              }), e.on("end", function () {
                t.closedSource(t.previous.streamInfo), t._sources.length ? t.prepareNextSource() : t.end();
              }), e.on("error", function (e) {
                t.error(e);
              }), this;
            }, s.prototype.resume = function () {
              return !!i.prototype.resume.call(this) && (!this.previous && this._sources.length ? (this.prepareNextSource(), true) : this.previous || this._sources.length || this.generatedError ? void 0 : (this.end(), true));
            }, s.prototype.error = function (e) {
              var t = this._sources;
              if (!i.prototype.error.call(this, e)) return false;
              for (var r = 0; r < t.length; r++) try {
                t[r].error(e);
              } catch (e) {}
              return true;
            }, s.prototype.lock = function () {
              i.prototype.lock.call(this);
              for (var e = this._sources, t = 0; t < e.length; t++) e[t].lock();
            }, t.exports = s;
          }, {
            "../crc32": 4,
            "../signature": 23,
            "../stream/GenericWorker": 28,
            "../utf8": 31,
            "../utils": 32
          }],
          9: [function (e, t, r) {

            var u = e("../compressions"),
              n = e("./ZipFileWorker");
            r.generateWorker = function (e, a, t) {
              var o = new n(a.streamFiles, t, a.platform, a.encodeFileName),
                h = 0;
              try {
                e.forEach(function (e, t) {
                  h++;
                  var r = function (e, t) {
                      var r = e || t,
                        n = u[r];
                      if (!n) throw new Error(r + " is not a valid compression method !");
                      return n;
                    }(t.options.compression, a.compression),
                    n = t.options.compressionOptions || a.compressionOptions || {},
                    i = t.dir,
                    s = t.date;
                  t._compressWorker(r, n).withStreamInfo("file", {
                    name: e,
                    dir: i,
                    date: s,
                    comment: t.comment || "",
                    unixPermissions: t.unixPermissions,
                    dosPermissions: t.dosPermissions
                  }).pipe(o);
                }), o.entriesCount = h;
              } catch (e) {
                o.error(e);
              }
              return o;
            };
          }, {
            "../compressions": 3,
            "./ZipFileWorker": 8
          }],
          10: [function (e, t, r) {

            function n() {
              if (!(this instanceof n)) return new n();
              if (arguments.length) throw new Error("The constructor with parameters has been removed in JSZip 3.0, please check the upgrade guide.");
              this.files = Object.create(null), this.comment = null, this.root = "", this.clone = function () {
                var e = new n();
                for (var t in this) "function" != typeof this[t] && (e[t] = this[t]);
                return e;
              };
            }
            (n.prototype = e("./object")).loadAsync = e("./load"), n.support = e("./support"), n.defaults = e("./defaults"), n.version = "3.10.1", n.loadAsync = function (e, t) {
              return new n().loadAsync(e, t);
            }, n.external = e("./external"), t.exports = n;
          }, {
            "./defaults": 5,
            "./external": 6,
            "./load": 11,
            "./object": 15,
            "./support": 30
          }],
          11: [function (e, t, r) {

            var u = e("./utils"),
              i = e("./external"),
              n = e("./utf8"),
              s = e("./zipEntries"),
              a = e("./stream/Crc32Probe"),
              l = e("./nodejsUtils");
            function f(n) {
              return new i.Promise(function (e, t) {
                var r = n.decompressed.getContentWorker().pipe(new a());
                r.on("error", function (e) {
                  t(e);
                }).on("end", function () {
                  r.streamInfo.crc32 !== n.decompressed.crc32 ? t(new Error("Corrupted zip : CRC32 mismatch")) : e();
                }).resume();
              });
            }
            t.exports = function (e, o) {
              var h = this;
              return o = u.extend(o || {}, {
                base64: false,
                checkCRC32: false,
                optimizedBinaryString: false,
                createFolders: false,
                decodeFileName: n.utf8decode
              }), l.isNode && l.isStream(e) ? i.Promise.reject(new Error("JSZip can't accept a stream when loading a zip file.")) : u.prepareContent("the loaded zip file", e, true, o.optimizedBinaryString, o.base64).then(function (e) {
                var t = new s(o);
                return t.load(e), t;
              }).then(function (e) {
                var t = [i.Promise.resolve(e)],
                  r = e.files;
                if (o.checkCRC32) for (var n = 0; n < r.length; n++) t.push(f(r[n]));
                return i.Promise.all(t);
              }).then(function (e) {
                for (var t = e.shift(), r = t.files, n = 0; n < r.length; n++) {
                  var i = r[n],
                    s = i.fileNameStr,
                    a = u.resolve(i.fileNameStr);
                  h.file(a, i.decompressed, {
                    binary: true,
                    optimizedBinaryString: true,
                    date: i.date,
                    dir: i.dir,
                    comment: i.fileCommentStr.length ? i.fileCommentStr : null,
                    unixPermissions: i.unixPermissions,
                    dosPermissions: i.dosPermissions,
                    createFolders: o.createFolders
                  }), i.dir || (h.file(a).unsafeOriginalName = s);
                }
                return t.zipComment.length && (h.comment = t.zipComment), h;
              });
            };
          }, {
            "./external": 6,
            "./nodejsUtils": 14,
            "./stream/Crc32Probe": 25,
            "./utf8": 31,
            "./utils": 32,
            "./zipEntries": 33
          }],
          12: [function (e, t, r) {

            var n = e("../utils"),
              i = e("../stream/GenericWorker");
            function s(e, t) {
              i.call(this, "Nodejs stream input adapter for " + e), this._upstreamEnded = false, this._bindStream(t);
            }
            n.inherits(s, i), s.prototype._bindStream = function (e) {
              var t = this;
              (this._stream = e).pause(), e.on("data", function (e) {
                t.push({
                  data: e,
                  meta: {
                    percent: 0
                  }
                });
              }).on("error", function (e) {
                t.isPaused ? this.generatedError = e : t.error(e);
              }).on("end", function () {
                t.isPaused ? t._upstreamEnded = true : t.end();
              });
            }, s.prototype.pause = function () {
              return !!i.prototype.pause.call(this) && (this._stream.pause(), true);
            }, s.prototype.resume = function () {
              return !!i.prototype.resume.call(this) && (this._upstreamEnded ? this.end() : this._stream.resume(), true);
            }, t.exports = s;
          }, {
            "../stream/GenericWorker": 28,
            "../utils": 32
          }],
          13: [function (e, t, r) {

            var i = e("readable-stream").Readable;
            function n(e, t, r) {
              i.call(this, t), this._helper = e;
              var n = this;
              e.on("data", function (e, t) {
                n.push(e) || n._helper.pause(), r && r(t);
              }).on("error", function (e) {
                n.emit("error", e);
              }).on("end", function () {
                n.push(null);
              });
            }
            e("../utils").inherits(n, i), n.prototype._read = function () {
              this._helper.resume();
            }, t.exports = n;
          }, {
            "../utils": 32,
            "readable-stream": 16
          }],
          14: [function (e, t, r) {

            t.exports = {
              isNode: "undefined" != typeof Buffer,
              newBufferFrom: function (e, t) {
                if (Buffer.from && Buffer.from !== Uint8Array.from) return Buffer.from(e, t);
                if ("number" == typeof e) throw new Error('The "data" argument must not be a number');
                return new Buffer(e, t);
              },
              allocBuffer: function (e) {
                if (Buffer.alloc) return Buffer.alloc(e);
                var t = new Buffer(e);
                return t.fill(0), t;
              },
              isBuffer: function (e) {
                return Buffer.isBuffer(e);
              },
              isStream: function (e) {
                return e && "function" == typeof e.on && "function" == typeof e.pause && "function" == typeof e.resume;
              }
            };
          }, {}],
          15: [function (e, t, r) {

            function s(e, t, r) {
              var n,
                i = u.getTypeOf(t),
                s = u.extend(r || {}, f);
              s.date = s.date || new Date(), null !== s.compression && (s.compression = s.compression.toUpperCase()), "string" == typeof s.unixPermissions && (s.unixPermissions = parseInt(s.unixPermissions, 8)), s.unixPermissions && 16384 & s.unixPermissions && (s.dir = true), s.dosPermissions && 16 & s.dosPermissions && (s.dir = true), s.dir && (e = g(e)), s.createFolders && (n = _(e)) && b.call(this, n, true);
              var a = "string" === i && false === s.binary && false === s.base64;
              r && void 0 !== r.binary || (s.binary = !a), (t instanceof c && 0 === t.uncompressedSize || s.dir || !t || 0 === t.length) && (s.base64 = false, s.binary = true, t = "", s.compression = "STORE", i = "string");
              var o = null;
              o = t instanceof c || t instanceof l ? t : p.isNode && p.isStream(t) ? new m(e, t) : u.prepareContent(e, t, s.binary, s.optimizedBinaryString, s.base64);
              var h = new d(e, o, s);
              this.files[e] = h;
            }
            var i = e("./utf8"),
              u = e("./utils"),
              l = e("./stream/GenericWorker"),
              a = e("./stream/StreamHelper"),
              f = e("./defaults"),
              c = e("./compressedObject"),
              d = e("./zipObject"),
              o = e("./generate"),
              p = e("./nodejsUtils"),
              m = e("./nodejs/NodejsStreamInputAdapter"),
              _ = function (e) {
                "/" === e.slice(-1) && (e = e.substring(0, e.length - 1));
                var t = e.lastIndexOf("/");
                return 0 < t ? e.substring(0, t) : "";
              },
              g = function (e) {
                return "/" !== e.slice(-1) && (e += "/"), e;
              },
              b = function (e, t) {
                return t = void 0 !== t ? t : f.createFolders, e = g(e), this.files[e] || s.call(this, e, null, {
                  dir: true,
                  createFolders: t
                }), this.files[e];
              };
            function h(e) {
              return "[object RegExp]" === Object.prototype.toString.call(e);
            }
            var n = {
              load: function () {
                throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.");
              },
              forEach: function (e) {
                var t, r, n;
                for (t in this.files) n = this.files[t], (r = t.slice(this.root.length, t.length)) && t.slice(0, this.root.length) === this.root && e(r, n);
              },
              filter: function (r) {
                var n = [];
                return this.forEach(function (e, t) {
                  r(e, t) && n.push(t);
                }), n;
              },
              file: function (e, t, r) {
                if (1 !== arguments.length) return e = this.root + e, s.call(this, e, t, r), this;
                if (h(e)) {
                  var n = e;
                  return this.filter(function (e, t) {
                    return !t.dir && n.test(e);
                  });
                }
                var i = this.files[this.root + e];
                return i && !i.dir ? i : null;
              },
              folder: function (r) {
                if (!r) return this;
                if (h(r)) return this.filter(function (e, t) {
                  return t.dir && r.test(e);
                });
                var e = this.root + r,
                  t = b.call(this, e),
                  n = this.clone();
                return n.root = t.name, n;
              },
              remove: function (r) {
                r = this.root + r;
                var e = this.files[r];
                if (e || ("/" !== r.slice(-1) && (r += "/"), e = this.files[r]), e && !e.dir) delete this.files[r];else for (var t = this.filter(function (e, t) {
                    return t.name.slice(0, r.length) === r;
                  }), n = 0; n < t.length; n++) delete this.files[t[n].name];
                return this;
              },
              generate: function () {
                throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.");
              },
              generateInternalStream: function (e) {
                var t,
                  r = {};
                try {
                  if ((r = u.extend(e || {}, {
                    streamFiles: !1,
                    compression: "STORE",
                    compressionOptions: null,
                    type: "",
                    platform: "DOS",
                    comment: null,
                    mimeType: "application/zip",
                    encodeFileName: i.utf8encode
                  })).type = r.type.toLowerCase(), r.compression = r.compression.toUpperCase(), "binarystring" === r.type && (r.type = "string"), !r.type) throw new Error("No output type specified.");
                  u.checkSupport(r.type), "darwin" !== r.platform && "freebsd" !== r.platform && "linux" !== r.platform && "sunos" !== r.platform || (r.platform = "UNIX"), "win32" === r.platform && (r.platform = "DOS");
                  var n = r.comment || this.comment || "";
                  t = o.generateWorker(this, r, n);
                } catch (e) {
                  (t = new l("error")).error(e);
                }
                return new a(t, r.type || "string", r.mimeType);
              },
              generateAsync: function (e, t) {
                return this.generateInternalStream(e).accumulate(t);
              },
              generateNodeStream: function (e, t) {
                return (e = e || {}).type || (e.type = "nodebuffer"), this.generateInternalStream(e).toNodejsStream(t);
              }
            };
            t.exports = n;
          }, {
            "./compressedObject": 2,
            "./defaults": 5,
            "./generate": 9,
            "./nodejs/NodejsStreamInputAdapter": 12,
            "./nodejsUtils": 14,
            "./stream/GenericWorker": 28,
            "./stream/StreamHelper": 29,
            "./utf8": 31,
            "./utils": 32,
            "./zipObject": 35
          }],
          16: [function (e, t, r) {

            t.exports = e("stream");
          }, {
            stream: void 0
          }],
          17: [function (e, t, r) {

            var n = e("./DataReader");
            function i(e) {
              n.call(this, e);
              for (var t = 0; t < this.data.length; t++) e[t] = 255 & e[t];
            }
            e("../utils").inherits(i, n), i.prototype.byteAt = function (e) {
              return this.data[this.zero + e];
            }, i.prototype.lastIndexOfSignature = function (e) {
              for (var t = e.charCodeAt(0), r = e.charCodeAt(1), n = e.charCodeAt(2), i = e.charCodeAt(3), s = this.length - 4; 0 <= s; --s) if (this.data[s] === t && this.data[s + 1] === r && this.data[s + 2] === n && this.data[s + 3] === i) return s - this.zero;
              return -1;
            }, i.prototype.readAndCheckSignature = function (e) {
              var t = e.charCodeAt(0),
                r = e.charCodeAt(1),
                n = e.charCodeAt(2),
                i = e.charCodeAt(3),
                s = this.readData(4);
              return t === s[0] && r === s[1] && n === s[2] && i === s[3];
            }, i.prototype.readData = function (e) {
              if (this.checkOffset(e), 0 === e) return [];
              var t = this.data.slice(this.zero + this.index, this.zero + this.index + e);
              return this.index += e, t;
            }, t.exports = i;
          }, {
            "../utils": 32,
            "./DataReader": 18
          }],
          18: [function (e, t, r) {

            var n = e("../utils");
            function i(e) {
              this.data = e, this.length = e.length, this.index = 0, this.zero = 0;
            }
            i.prototype = {
              checkOffset: function (e) {
                this.checkIndex(this.index + e);
              },
              checkIndex: function (e) {
                if (this.length < this.zero + e || e < 0) throw new Error("End of data reached (data length = " + this.length + ", asked index = " + e + "). Corrupted zip ?");
              },
              setIndex: function (e) {
                this.checkIndex(e), this.index = e;
              },
              skip: function (e) {
                this.setIndex(this.index + e);
              },
              byteAt: function () {},
              readInt: function (e) {
                var t,
                  r = 0;
                for (this.checkOffset(e), t = this.index + e - 1; t >= this.index; t--) r = (r << 8) + this.byteAt(t);
                return this.index += e, r;
              },
              readString: function (e) {
                return n.transformTo("string", this.readData(e));
              },
              readData: function () {},
              lastIndexOfSignature: function () {},
              readAndCheckSignature: function () {},
              readDate: function () {
                var e = this.readInt(4);
                return new Date(Date.UTC(1980 + (e >> 25 & 127), (e >> 21 & 15) - 1, e >> 16 & 31, e >> 11 & 31, e >> 5 & 63, (31 & e) << 1));
              }
            }, t.exports = i;
          }, {
            "../utils": 32
          }],
          19: [function (e, t, r) {

            var n = e("./Uint8ArrayReader");
            function i(e) {
              n.call(this, e);
            }
            e("../utils").inherits(i, n), i.prototype.readData = function (e) {
              this.checkOffset(e);
              var t = this.data.slice(this.zero + this.index, this.zero + this.index + e);
              return this.index += e, t;
            }, t.exports = i;
          }, {
            "../utils": 32,
            "./Uint8ArrayReader": 21
          }],
          20: [function (e, t, r) {

            var n = e("./DataReader");
            function i(e) {
              n.call(this, e);
            }
            e("../utils").inherits(i, n), i.prototype.byteAt = function (e) {
              return this.data.charCodeAt(this.zero + e);
            }, i.prototype.lastIndexOfSignature = function (e) {
              return this.data.lastIndexOf(e) - this.zero;
            }, i.prototype.readAndCheckSignature = function (e) {
              return e === this.readData(4);
            }, i.prototype.readData = function (e) {
              this.checkOffset(e);
              var t = this.data.slice(this.zero + this.index, this.zero + this.index + e);
              return this.index += e, t;
            }, t.exports = i;
          }, {
            "../utils": 32,
            "./DataReader": 18
          }],
          21: [function (e, t, r) {

            var n = e("./ArrayReader");
            function i(e) {
              n.call(this, e);
            }
            e("../utils").inherits(i, n), i.prototype.readData = function (e) {
              if (this.checkOffset(e), 0 === e) return new Uint8Array(0);
              var t = this.data.subarray(this.zero + this.index, this.zero + this.index + e);
              return this.index += e, t;
            }, t.exports = i;
          }, {
            "../utils": 32,
            "./ArrayReader": 17
          }],
          22: [function (e, t, r) {

            var n = e("../utils"),
              i = e("../support"),
              s = e("./ArrayReader"),
              a = e("./StringReader"),
              o = e("./NodeBufferReader"),
              h = e("./Uint8ArrayReader");
            t.exports = function (e) {
              var t = n.getTypeOf(e);
              return n.checkSupport(t), "string" !== t || i.uint8array ? "nodebuffer" === t ? new o(e) : i.uint8array ? new h(n.transformTo("uint8array", e)) : new s(n.transformTo("array", e)) : new a(e);
            };
          }, {
            "../support": 30,
            "../utils": 32,
            "./ArrayReader": 17,
            "./NodeBufferReader": 19,
            "./StringReader": 20,
            "./Uint8ArrayReader": 21
          }],
          23: [function (e, t, r) {

            r.LOCAL_FILE_HEADER = "PK", r.CENTRAL_FILE_HEADER = "PK", r.CENTRAL_DIRECTORY_END = "PK", r.ZIP64_CENTRAL_DIRECTORY_LOCATOR = "PK", r.ZIP64_CENTRAL_DIRECTORY_END = "PK", r.DATA_DESCRIPTOR = "PK\b";
          }, {}],
          24: [function (e, t, r) {

            var n = e("./GenericWorker"),
              i = e("../utils");
            function s(e) {
              n.call(this, "ConvertWorker to " + e), this.destType = e;
            }
            i.inherits(s, n), s.prototype.processChunk = function (e) {
              this.push({
                data: i.transformTo(this.destType, e.data),
                meta: e.meta
              });
            }, t.exports = s;
          }, {
            "../utils": 32,
            "./GenericWorker": 28
          }],
          25: [function (e, t, r) {

            var n = e("./GenericWorker"),
              i = e("../crc32");
            function s() {
              n.call(this, "Crc32Probe"), this.withStreamInfo("crc32", 0);
            }
            e("../utils").inherits(s, n), s.prototype.processChunk = function (e) {
              this.streamInfo.crc32 = i(e.data, this.streamInfo.crc32 || 0), this.push(e);
            }, t.exports = s;
          }, {
            "../crc32": 4,
            "../utils": 32,
            "./GenericWorker": 28
          }],
          26: [function (e, t, r) {

            var n = e("../utils"),
              i = e("./GenericWorker");
            function s(e) {
              i.call(this, "DataLengthProbe for " + e), this.propName = e, this.withStreamInfo(e, 0);
            }
            n.inherits(s, i), s.prototype.processChunk = function (e) {
              if (e) {
                var t = this.streamInfo[this.propName] || 0;
                this.streamInfo[this.propName] = t + e.data.length;
              }
              i.prototype.processChunk.call(this, e);
            }, t.exports = s;
          }, {
            "../utils": 32,
            "./GenericWorker": 28
          }],
          27: [function (e, t, r) {

            var n = e("../utils"),
              i = e("./GenericWorker");
            function s(e) {
              i.call(this, "DataWorker");
              var t = this;
              this.dataIsReady = false, this.index = 0, this.max = 0, this.data = null, this.type = "", this._tickScheduled = false, e.then(function (e) {
                t.dataIsReady = true, t.data = e, t.max = e && e.length || 0, t.type = n.getTypeOf(e), t.isPaused || t._tickAndRepeat();
              }, function (e) {
                t.error(e);
              });
            }
            n.inherits(s, i), s.prototype.cleanUp = function () {
              i.prototype.cleanUp.call(this), this.data = null;
            }, s.prototype.resume = function () {
              return !!i.prototype.resume.call(this) && (!this._tickScheduled && this.dataIsReady && (this._tickScheduled = true, n.delay(this._tickAndRepeat, [], this)), true);
            }, s.prototype._tickAndRepeat = function () {
              this._tickScheduled = false, this.isPaused || this.isFinished || (this._tick(), this.isFinished || (n.delay(this._tickAndRepeat, [], this), this._tickScheduled = true));
            }, s.prototype._tick = function () {
              if (this.isPaused || this.isFinished) return false;
              var e = null,
                t = Math.min(this.max, this.index + 16384);
              if (this.index >= this.max) return this.end();
              switch (this.type) {
                case "string":
                  e = this.data.substring(this.index, t);
                  break;
                case "uint8array":
                  e = this.data.subarray(this.index, t);
                  break;
                case "array":
                case "nodebuffer":
                  e = this.data.slice(this.index, t);
              }
              return this.index = t, this.push({
                data: e,
                meta: {
                  percent: this.max ? this.index / this.max * 100 : 0
                }
              });
            }, t.exports = s;
          }, {
            "../utils": 32,
            "./GenericWorker": 28
          }],
          28: [function (e, t, r) {

            function n(e) {
              this.name = e || "default", this.streamInfo = {}, this.generatedError = null, this.extraStreamInfo = {}, this.isPaused = true, this.isFinished = false, this.isLocked = false, this._listeners = {
                data: [],
                end: [],
                error: []
              }, this.previous = null;
            }
            n.prototype = {
              push: function (e) {
                this.emit("data", e);
              },
              end: function () {
                if (this.isFinished) return false;
                this.flush();
                try {
                  this.emit("end"), this.cleanUp(), this.isFinished = !0;
                } catch (e) {
                  this.emit("error", e);
                }
                return true;
              },
              error: function (e) {
                return !this.isFinished && (this.isPaused ? this.generatedError = e : (this.isFinished = true, this.emit("error", e), this.previous && this.previous.error(e), this.cleanUp()), true);
              },
              on: function (e, t) {
                return this._listeners[e].push(t), this;
              },
              cleanUp: function () {
                this.streamInfo = this.generatedError = this.extraStreamInfo = null, this._listeners = [];
              },
              emit: function (e, t) {
                if (this._listeners[e]) for (var r = 0; r < this._listeners[e].length; r++) this._listeners[e][r].call(this, t);
              },
              pipe: function (e) {
                return e.registerPrevious(this);
              },
              registerPrevious: function (e) {
                if (this.isLocked) throw new Error("The stream '" + this + "' has already been used.");
                this.streamInfo = e.streamInfo, this.mergeStreamInfo(), this.previous = e;
                var t = this;
                return e.on("data", function (e) {
                  t.processChunk(e);
                }), e.on("end", function () {
                  t.end();
                }), e.on("error", function (e) {
                  t.error(e);
                }), this;
              },
              pause: function () {
                return !this.isPaused && !this.isFinished && (this.isPaused = true, this.previous && this.previous.pause(), true);
              },
              resume: function () {
                if (!this.isPaused || this.isFinished) return false;
                var e = this.isPaused = false;
                return this.generatedError && (this.error(this.generatedError), e = true), this.previous && this.previous.resume(), !e;
              },
              flush: function () {},
              processChunk: function (e) {
                this.push(e);
              },
              withStreamInfo: function (e, t) {
                return this.extraStreamInfo[e] = t, this.mergeStreamInfo(), this;
              },
              mergeStreamInfo: function () {
                for (var e in this.extraStreamInfo) Object.prototype.hasOwnProperty.call(this.extraStreamInfo, e) && (this.streamInfo[e] = this.extraStreamInfo[e]);
              },
              lock: function () {
                if (this.isLocked) throw new Error("The stream '" + this + "' has already been used.");
                this.isLocked = true, this.previous && this.previous.lock();
              },
              toString: function () {
                var e = "Worker " + this.name;
                return this.previous ? this.previous + " -> " + e : e;
              }
            }, t.exports = n;
          }, {}],
          29: [function (e, t, r) {

            var h = e("../utils"),
              i = e("./ConvertWorker"),
              s = e("./GenericWorker"),
              u = e("../base64"),
              n = e("../support"),
              a = e("../external"),
              o = null;
            if (n.nodestream) try {
              o = e("../nodejs/NodejsStreamOutputAdapter");
            } catch (e) {}
            function l(e, o) {
              return new a.Promise(function (t, r) {
                var n = [],
                  i = e._internalType,
                  s = e._outputType,
                  a = e._mimeType;
                e.on("data", function (e, t) {
                  n.push(e), o && o(t);
                }).on("error", function (e) {
                  n = [], r(e);
                }).on("end", function () {
                  try {
                    var e = function (e, t, r) {
                      switch (e) {
                        case "blob":
                          return h.newBlob(h.transformTo("arraybuffer", t), r);
                        case "base64":
                          return u.encode(t);
                        default:
                          return h.transformTo(e, t);
                      }
                    }(s, function (e, t) {
                      var r,
                        n = 0,
                        i = null,
                        s = 0;
                      for (r = 0; r < t.length; r++) s += t[r].length;
                      switch (e) {
                        case "string":
                          return t.join("");
                        case "array":
                          return Array.prototype.concat.apply([], t);
                        case "uint8array":
                          for (i = new Uint8Array(s), r = 0; r < t.length; r++) i.set(t[r], n), n += t[r].length;
                          return i;
                        case "nodebuffer":
                          return Buffer.concat(t);
                        default:
                          throw new Error("concat : unsupported type '" + e + "'");
                      }
                    }(i, n), a);
                    t(e);
                  } catch (e) {
                    r(e);
                  }
                  n = [];
                }).resume();
              });
            }
            function f(e, t, r) {
              var n = t;
              switch (t) {
                case "blob":
                case "arraybuffer":
                  n = "uint8array";
                  break;
                case "base64":
                  n = "string";
              }
              try {
                this._internalType = n, this._outputType = t, this._mimeType = r, h.checkSupport(n), this._worker = e.pipe(new i(n)), e.lock();
              } catch (e) {
                this._worker = new s("error"), this._worker.error(e);
              }
            }
            f.prototype = {
              accumulate: function (e) {
                return l(this, e);
              },
              on: function (e, t) {
                var r = this;
                return "data" === e ? this._worker.on(e, function (e) {
                  t.call(r, e.data, e.meta);
                }) : this._worker.on(e, function () {
                  h.delay(t, arguments, r);
                }), this;
              },
              resume: function () {
                return h.delay(this._worker.resume, [], this._worker), this;
              },
              pause: function () {
                return this._worker.pause(), this;
              },
              toNodejsStream: function (e) {
                if (h.checkSupport("nodestream"), "nodebuffer" !== this._outputType) throw new Error(this._outputType + " is not supported by this method");
                return new o(this, {
                  objectMode: "nodebuffer" !== this._outputType
                }, e);
              }
            }, t.exports = f;
          }, {
            "../base64": 1,
            "../external": 6,
            "../nodejs/NodejsStreamOutputAdapter": 13,
            "../support": 30,
            "../utils": 32,
            "./ConvertWorker": 24,
            "./GenericWorker": 28
          }],
          30: [function (e, t, r) {

            if (r.base64 = true, r.array = true, r.string = true, r.arraybuffer = "undefined" != typeof ArrayBuffer && "undefined" != typeof Uint8Array, r.nodebuffer = "undefined" != typeof Buffer, r.uint8array = "undefined" != typeof Uint8Array, "undefined" == typeof ArrayBuffer) r.blob = false;else {
              var n = new ArrayBuffer(0);
              try {
                r.blob = 0 === new Blob([n], {
                  type: "application/zip"
                }).size;
              } catch (e) {
                try {
                  var i = new (self.BlobBuilder || self.WebKitBlobBuilder || self.MozBlobBuilder || self.MSBlobBuilder)();
                  i.append(n), r.blob = 0 === i.getBlob("application/zip").size;
                } catch (e) {
                  r.blob = false;
                }
              }
            }
            try {
              r.nodestream = !!e("readable-stream").Readable;
            } catch (e) {
              r.nodestream = false;
            }
          }, {
            "readable-stream": 16
          }],
          31: [function (e, t, s) {

            for (var o = e("./utils"), h = e("./support"), r = e("./nodejsUtils"), n = e("./stream/GenericWorker"), u = new Array(256), i = 0; i < 256; i++) u[i] = 252 <= i ? 6 : 248 <= i ? 5 : 240 <= i ? 4 : 224 <= i ? 3 : 192 <= i ? 2 : 1;
            u[254] = u[254] = 1;
            function a() {
              n.call(this, "utf-8 decode"), this.leftOver = null;
            }
            function l() {
              n.call(this, "utf-8 encode");
            }
            s.utf8encode = function (e) {
              return h.nodebuffer ? r.newBufferFrom(e, "utf-8") : function (e) {
                var t,
                  r,
                  n,
                  i,
                  s,
                  a = e.length,
                  o = 0;
                for (i = 0; i < a; i++) 55296 == (64512 & (r = e.charCodeAt(i))) && i + 1 < a && 56320 == (64512 & (n = e.charCodeAt(i + 1))) && (r = 65536 + (r - 55296 << 10) + (n - 56320), i++), o += r < 128 ? 1 : r < 2048 ? 2 : r < 65536 ? 3 : 4;
                for (t = h.uint8array ? new Uint8Array(o) : new Array(o), i = s = 0; s < o; i++) 55296 == (64512 & (r = e.charCodeAt(i))) && i + 1 < a && 56320 == (64512 & (n = e.charCodeAt(i + 1))) && (r = 65536 + (r - 55296 << 10) + (n - 56320), i++), r < 128 ? t[s++] = r : (r < 2048 ? t[s++] = 192 | r >>> 6 : (r < 65536 ? t[s++] = 224 | r >>> 12 : (t[s++] = 240 | r >>> 18, t[s++] = 128 | r >>> 12 & 63), t[s++] = 128 | r >>> 6 & 63), t[s++] = 128 | 63 & r);
                return t;
              }(e);
            }, s.utf8decode = function (e) {
              return h.nodebuffer ? o.transformTo("nodebuffer", e).toString("utf-8") : function (e) {
                var t,
                  r,
                  n,
                  i,
                  s = e.length,
                  a = new Array(2 * s);
                for (t = r = 0; t < s;) if ((n = e[t++]) < 128) a[r++] = n;else if (4 < (i = u[n])) a[r++] = 65533, t += i - 1;else {
                  for (n &= 2 === i ? 31 : 3 === i ? 15 : 7; 1 < i && t < s;) n = n << 6 | 63 & e[t++], i--;
                  1 < i ? a[r++] = 65533 : n < 65536 ? a[r++] = n : (n -= 65536, a[r++] = 55296 | n >> 10 & 1023, a[r++] = 56320 | 1023 & n);
                }
                return a.length !== r && (a.subarray ? a = a.subarray(0, r) : a.length = r), o.applyFromCharCode(a);
              }(e = o.transformTo(h.uint8array ? "uint8array" : "array", e));
            }, o.inherits(a, n), a.prototype.processChunk = function (e) {
              var t = o.transformTo(h.uint8array ? "uint8array" : "array", e.data);
              if (this.leftOver && this.leftOver.length) {
                if (h.uint8array) {
                  var r = t;
                  (t = new Uint8Array(r.length + this.leftOver.length)).set(this.leftOver, 0), t.set(r, this.leftOver.length);
                } else t = this.leftOver.concat(t);
                this.leftOver = null;
              }
              var n = function (e, t) {
                  var r;
                  for ((t = t || e.length) > e.length && (t = e.length), r = t - 1; 0 <= r && 128 == (192 & e[r]);) r--;
                  return r < 0 ? t : 0 === r ? t : r + u[e[r]] > t ? r : t;
                }(t),
                i = t;
              n !== t.length && (h.uint8array ? (i = t.subarray(0, n), this.leftOver = t.subarray(n, t.length)) : (i = t.slice(0, n), this.leftOver = t.slice(n, t.length))), this.push({
                data: s.utf8decode(i),
                meta: e.meta
              });
            }, a.prototype.flush = function () {
              this.leftOver && this.leftOver.length && (this.push({
                data: s.utf8decode(this.leftOver),
                meta: {}
              }), this.leftOver = null);
            }, s.Utf8DecodeWorker = a, o.inherits(l, n), l.prototype.processChunk = function (e) {
              this.push({
                data: s.utf8encode(e.data),
                meta: e.meta
              });
            }, s.Utf8EncodeWorker = l;
          }, {
            "./nodejsUtils": 14,
            "./stream/GenericWorker": 28,
            "./support": 30,
            "./utils": 32
          }],
          32: [function (e, t, a) {

            var o = e("./support"),
              h = e("./base64"),
              r = e("./nodejsUtils"),
              u = e("./external");
            function n(e) {
              return e;
            }
            function l(e, t) {
              for (var r = 0; r < e.length; ++r) t[r] = 255 & e.charCodeAt(r);
              return t;
            }
            e("setimmediate"), a.newBlob = function (t, r) {
              a.checkSupport("blob");
              try {
                return new Blob([t], {
                  type: r
                });
              } catch (e) {
                try {
                  var n = new (self.BlobBuilder || self.WebKitBlobBuilder || self.MozBlobBuilder || self.MSBlobBuilder)();
                  return n.append(t), n.getBlob(r);
                } catch (e) {
                  throw new Error("Bug : can't construct the Blob.");
                }
              }
            };
            var i = {
              stringifyByChunk: function (e, t, r) {
                var n = [],
                  i = 0,
                  s = e.length;
                if (s <= r) return String.fromCharCode.apply(null, e);
                for (; i < s;) "array" === t || "nodebuffer" === t ? n.push(String.fromCharCode.apply(null, e.slice(i, Math.min(i + r, s)))) : n.push(String.fromCharCode.apply(null, e.subarray(i, Math.min(i + r, s)))), i += r;
                return n.join("");
              },
              stringifyByChar: function (e) {
                for (var t = "", r = 0; r < e.length; r++) t += String.fromCharCode(e[r]);
                return t;
              },
              applyCanBeUsed: {
                uint8array: function () {
                  try {
                    return o.uint8array && 1 === String.fromCharCode.apply(null, new Uint8Array(1)).length;
                  } catch (e) {
                    return false;
                  }
                }(),
                nodebuffer: function () {
                  try {
                    return o.nodebuffer && 1 === String.fromCharCode.apply(null, r.allocBuffer(1)).length;
                  } catch (e) {
                    return false;
                  }
                }()
              }
            };
            function s(e) {
              var t = 65536,
                r = a.getTypeOf(e),
                n = true;
              if ("uint8array" === r ? n = i.applyCanBeUsed.uint8array : "nodebuffer" === r && (n = i.applyCanBeUsed.nodebuffer), n) for (; 1 < t;) try {
                return i.stringifyByChunk(e, r, t);
              } catch (e) {
                t = Math.floor(t / 2);
              }
              return i.stringifyByChar(e);
            }
            function f(e, t) {
              for (var r = 0; r < e.length; r++) t[r] = e[r];
              return t;
            }
            a.applyFromCharCode = s;
            var c = {};
            c.string = {
              string: n,
              array: function (e) {
                return l(e, new Array(e.length));
              },
              arraybuffer: function (e) {
                return c.string.uint8array(e).buffer;
              },
              uint8array: function (e) {
                return l(e, new Uint8Array(e.length));
              },
              nodebuffer: function (e) {
                return l(e, r.allocBuffer(e.length));
              }
            }, c.array = {
              string: s,
              array: n,
              arraybuffer: function (e) {
                return new Uint8Array(e).buffer;
              },
              uint8array: function (e) {
                return new Uint8Array(e);
              },
              nodebuffer: function (e) {
                return r.newBufferFrom(e);
              }
            }, c.arraybuffer = {
              string: function (e) {
                return s(new Uint8Array(e));
              },
              array: function (e) {
                return f(new Uint8Array(e), new Array(e.byteLength));
              },
              arraybuffer: n,
              uint8array: function (e) {
                return new Uint8Array(e);
              },
              nodebuffer: function (e) {
                return r.newBufferFrom(new Uint8Array(e));
              }
            }, c.uint8array = {
              string: s,
              array: function (e) {
                return f(e, new Array(e.length));
              },
              arraybuffer: function (e) {
                return e.buffer;
              },
              uint8array: n,
              nodebuffer: function (e) {
                return r.newBufferFrom(e);
              }
            }, c.nodebuffer = {
              string: s,
              array: function (e) {
                return f(e, new Array(e.length));
              },
              arraybuffer: function (e) {
                return c.nodebuffer.uint8array(e).buffer;
              },
              uint8array: function (e) {
                return f(e, new Uint8Array(e.length));
              },
              nodebuffer: n
            }, a.transformTo = function (e, t) {
              if (t = t || "", !e) return t;
              a.checkSupport(e);
              var r = a.getTypeOf(t);
              return c[r][e](t);
            }, a.resolve = function (e) {
              for (var t = e.split("/"), r = [], n = 0; n < t.length; n++) {
                var i = t[n];
                "." === i || "" === i && 0 !== n && n !== t.length - 1 || (".." === i ? r.pop() : r.push(i));
              }
              return r.join("/");
            }, a.getTypeOf = function (e) {
              return "string" == typeof e ? "string" : "[object Array]" === Object.prototype.toString.call(e) ? "array" : o.nodebuffer && r.isBuffer(e) ? "nodebuffer" : o.uint8array && e instanceof Uint8Array ? "uint8array" : o.arraybuffer && e instanceof ArrayBuffer ? "arraybuffer" : void 0;
            }, a.checkSupport = function (e) {
              if (!o[e.toLowerCase()]) throw new Error(e + " is not supported by this platform");
            }, a.MAX_VALUE_16BITS = 65535, a.MAX_VALUE_32BITS = -1, a.pretty = function (e) {
              var t,
                r,
                n = "";
              for (r = 0; r < (e || "").length; r++) n += "\\x" + ((t = e.charCodeAt(r)) < 16 ? "0" : "") + t.toString(16).toUpperCase();
              return n;
            }, a.delay = function (e, t, r) {
              setImmediate(function () {
                e.apply(r || null, t || []);
              });
            }, a.inherits = function (e, t) {
              function r() {}
              r.prototype = t.prototype, e.prototype = new r();
            }, a.extend = function () {
              var e,
                t,
                r = {};
              for (e = 0; e < arguments.length; e++) for (t in arguments[e]) Object.prototype.hasOwnProperty.call(arguments[e], t) && void 0 === r[t] && (r[t] = arguments[e][t]);
              return r;
            }, a.prepareContent = function (r, e, n, i, s) {
              return u.Promise.resolve(e).then(function (n) {
                return o.blob && (n instanceof Blob || -1 !== ["[object File]", "[object Blob]"].indexOf(Object.prototype.toString.call(n))) && "undefined" != typeof FileReader ? new u.Promise(function (t, r) {
                  var e = new FileReader();
                  e.onload = function (e) {
                    t(e.target.result);
                  }, e.onerror = function (e) {
                    r(e.target.error);
                  }, e.readAsArrayBuffer(n);
                }) : n;
              }).then(function (e) {
                var t = a.getTypeOf(e);
                return t ? ("arraybuffer" === t ? e = a.transformTo("uint8array", e) : "string" === t && (s ? e = h.decode(e) : n && true !== i && (e = function (e) {
                  return l(e, o.uint8array ? new Uint8Array(e.length) : new Array(e.length));
                }(e))), e) : u.Promise.reject(new Error("Can't read the data of '" + r + "'. Is it in a supported JavaScript type (String, Blob, ArrayBuffer, etc) ?"));
              });
            };
          }, {
            "./base64": 1,
            "./external": 6,
            "./nodejsUtils": 14,
            "./support": 30,
            setimmediate: 54
          }],
          33: [function (e, t, r) {

            var n = e("./reader/readerFor"),
              i = e("./utils"),
              s = e("./signature"),
              a = e("./zipEntry"),
              o = e("./support");
            function h(e) {
              this.files = [], this.loadOptions = e;
            }
            h.prototype = {
              checkSignature: function (e) {
                if (!this.reader.readAndCheckSignature(e)) {
                  this.reader.index -= 4;
                  var t = this.reader.readString(4);
                  throw new Error("Corrupted zip or bug: unexpected signature (" + i.pretty(t) + ", expected " + i.pretty(e) + ")");
                }
              },
              isSignature: function (e, t) {
                var r = this.reader.index;
                this.reader.setIndex(e);
                var n = this.reader.readString(4) === t;
                return this.reader.setIndex(r), n;
              },
              readBlockEndOfCentral: function () {
                this.diskNumber = this.reader.readInt(2), this.diskWithCentralDirStart = this.reader.readInt(2), this.centralDirRecordsOnThisDisk = this.reader.readInt(2), this.centralDirRecords = this.reader.readInt(2), this.centralDirSize = this.reader.readInt(4), this.centralDirOffset = this.reader.readInt(4), this.zipCommentLength = this.reader.readInt(2);
                var e = this.reader.readData(this.zipCommentLength),
                  t = o.uint8array ? "uint8array" : "array",
                  r = i.transformTo(t, e);
                this.zipComment = this.loadOptions.decodeFileName(r);
              },
              readBlockZip64EndOfCentral: function () {
                this.zip64EndOfCentralSize = this.reader.readInt(8), this.reader.skip(4), this.diskNumber = this.reader.readInt(4), this.diskWithCentralDirStart = this.reader.readInt(4), this.centralDirRecordsOnThisDisk = this.reader.readInt(8), this.centralDirRecords = this.reader.readInt(8), this.centralDirSize = this.reader.readInt(8), this.centralDirOffset = this.reader.readInt(8), this.zip64ExtensibleData = {};
                for (var e, t, r, n = this.zip64EndOfCentralSize - 44; 0 < n;) e = this.reader.readInt(2), t = this.reader.readInt(4), r = this.reader.readData(t), this.zip64ExtensibleData[e] = {
                  id: e,
                  length: t,
                  value: r
                };
              },
              readBlockZip64EndOfCentralLocator: function () {
                if (this.diskWithZip64CentralDirStart = this.reader.readInt(4), this.relativeOffsetEndOfZip64CentralDir = this.reader.readInt(8), this.disksCount = this.reader.readInt(4), 1 < this.disksCount) throw new Error("Multi-volumes zip are not supported");
              },
              readLocalFiles: function () {
                var e, t;
                for (e = 0; e < this.files.length; e++) t = this.files[e], this.reader.setIndex(t.localHeaderOffset), this.checkSignature(s.LOCAL_FILE_HEADER), t.readLocalPart(this.reader), t.handleUTF8(), t.processAttributes();
              },
              readCentralDir: function () {
                var e;
                for (this.reader.setIndex(this.centralDirOffset); this.reader.readAndCheckSignature(s.CENTRAL_FILE_HEADER);) (e = new a({
                  zip64: this.zip64
                }, this.loadOptions)).readCentralPart(this.reader), this.files.push(e);
                if (this.centralDirRecords !== this.files.length && 0 !== this.centralDirRecords && 0 === this.files.length) throw new Error("Corrupted zip or bug: expected " + this.centralDirRecords + " records in central dir, got " + this.files.length);
              },
              readEndOfCentral: function () {
                var e = this.reader.lastIndexOfSignature(s.CENTRAL_DIRECTORY_END);
                if (e < 0) throw !this.isSignature(0, s.LOCAL_FILE_HEADER) ? new Error("Can't find end of central directory : is this a zip file ? If it is, see https://stuk.github.io/jszip/documentation/howto/read_zip.html") : new Error("Corrupted zip: can't find end of central directory");
                this.reader.setIndex(e);
                var t = e;
                if (this.checkSignature(s.CENTRAL_DIRECTORY_END), this.readBlockEndOfCentral(), this.diskNumber === i.MAX_VALUE_16BITS || this.diskWithCentralDirStart === i.MAX_VALUE_16BITS || this.centralDirRecordsOnThisDisk === i.MAX_VALUE_16BITS || this.centralDirRecords === i.MAX_VALUE_16BITS || this.centralDirSize === i.MAX_VALUE_32BITS || this.centralDirOffset === i.MAX_VALUE_32BITS) {
                  if (this.zip64 = true, (e = this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR)) < 0) throw new Error("Corrupted zip: can't find the ZIP64 end of central directory locator");
                  if (this.reader.setIndex(e), this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR), this.readBlockZip64EndOfCentralLocator(), !this.isSignature(this.relativeOffsetEndOfZip64CentralDir, s.ZIP64_CENTRAL_DIRECTORY_END) && (this.relativeOffsetEndOfZip64CentralDir = this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_END), this.relativeOffsetEndOfZip64CentralDir < 0)) throw new Error("Corrupted zip: can't find the ZIP64 end of central directory");
                  this.reader.setIndex(this.relativeOffsetEndOfZip64CentralDir), this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_END), this.readBlockZip64EndOfCentral();
                }
                var r = this.centralDirOffset + this.centralDirSize;
                this.zip64 && (r += 20, r += 12 + this.zip64EndOfCentralSize);
                var n = t - r;
                if (0 < n) this.isSignature(t, s.CENTRAL_FILE_HEADER) || (this.reader.zero = n);else if (n < 0) throw new Error("Corrupted zip: missing " + Math.abs(n) + " bytes.");
              },
              prepareReader: function (e) {
                this.reader = n(e);
              },
              load: function (e) {
                this.prepareReader(e), this.readEndOfCentral(), this.readCentralDir(), this.readLocalFiles();
              }
            }, t.exports = h;
          }, {
            "./reader/readerFor": 22,
            "./signature": 23,
            "./support": 30,
            "./utils": 32,
            "./zipEntry": 34
          }],
          34: [function (e, t, r) {

            var n = e("./reader/readerFor"),
              s = e("./utils"),
              i = e("./compressedObject"),
              a = e("./crc32"),
              o = e("./utf8"),
              h = e("./compressions"),
              u = e("./support");
            function l(e, t) {
              this.options = e, this.loadOptions = t;
            }
            l.prototype = {
              isEncrypted: function () {
                return 1 == (1 & this.bitFlag);
              },
              useUTF8: function () {
                return 2048 == (2048 & this.bitFlag);
              },
              readLocalPart: function (e) {
                var t, r;
                if (e.skip(22), this.fileNameLength = e.readInt(2), r = e.readInt(2), this.fileName = e.readData(this.fileNameLength), e.skip(r), -1 === this.compressedSize || -1 === this.uncompressedSize) throw new Error("Bug or corrupted zip : didn't get enough information from the central directory (compressedSize === -1 || uncompressedSize === -1)");
                if (null === (t = function (e) {
                  for (var t in h) if (Object.prototype.hasOwnProperty.call(h, t) && h[t].magic === e) return h[t];
                  return null;
                }(this.compressionMethod))) throw new Error("Corrupted zip : compression " + s.pretty(this.compressionMethod) + " unknown (inner file : " + s.transformTo("string", this.fileName) + ")");
                this.decompressed = new i(this.compressedSize, this.uncompressedSize, this.crc32, t, e.readData(this.compressedSize));
              },
              readCentralPart: function (e) {
                this.versionMadeBy = e.readInt(2), e.skip(2), this.bitFlag = e.readInt(2), this.compressionMethod = e.readString(2), this.date = e.readDate(), this.crc32 = e.readInt(4), this.compressedSize = e.readInt(4), this.uncompressedSize = e.readInt(4);
                var t = e.readInt(2);
                if (this.extraFieldsLength = e.readInt(2), this.fileCommentLength = e.readInt(2), this.diskNumberStart = e.readInt(2), this.internalFileAttributes = e.readInt(2), this.externalFileAttributes = e.readInt(4), this.localHeaderOffset = e.readInt(4), this.isEncrypted()) throw new Error("Encrypted zip are not supported");
                e.skip(t), this.readExtraFields(e), this.parseZIP64ExtraField(e), this.fileComment = e.readData(this.fileCommentLength);
              },
              processAttributes: function () {
                this.unixPermissions = null, this.dosPermissions = null;
                var e = this.versionMadeBy >> 8;
                this.dir = !!(16 & this.externalFileAttributes), 0 == e && (this.dosPermissions = 63 & this.externalFileAttributes), 3 == e && (this.unixPermissions = this.externalFileAttributes >> 16 & 65535), this.dir || "/" !== this.fileNameStr.slice(-1) || (this.dir = true);
              },
              parseZIP64ExtraField: function () {
                if (this.extraFields[1]) {
                  var e = n(this.extraFields[1].value);
                  this.uncompressedSize === s.MAX_VALUE_32BITS && (this.uncompressedSize = e.readInt(8)), this.compressedSize === s.MAX_VALUE_32BITS && (this.compressedSize = e.readInt(8)), this.localHeaderOffset === s.MAX_VALUE_32BITS && (this.localHeaderOffset = e.readInt(8)), this.diskNumberStart === s.MAX_VALUE_32BITS && (this.diskNumberStart = e.readInt(4));
                }
              },
              readExtraFields: function (e) {
                var t,
                  r,
                  n,
                  i = e.index + this.extraFieldsLength;
                for (this.extraFields || (this.extraFields = {}); e.index + 4 < i;) t = e.readInt(2), r = e.readInt(2), n = e.readData(r), this.extraFields[t] = {
                  id: t,
                  length: r,
                  value: n
                };
                e.setIndex(i);
              },
              handleUTF8: function () {
                var e = u.uint8array ? "uint8array" : "array";
                if (this.useUTF8()) this.fileNameStr = o.utf8decode(this.fileName), this.fileCommentStr = o.utf8decode(this.fileComment);else {
                  var t = this.findExtraFieldUnicodePath();
                  if (null !== t) this.fileNameStr = t;else {
                    var r = s.transformTo(e, this.fileName);
                    this.fileNameStr = this.loadOptions.decodeFileName(r);
                  }
                  var n = this.findExtraFieldUnicodeComment();
                  if (null !== n) this.fileCommentStr = n;else {
                    var i = s.transformTo(e, this.fileComment);
                    this.fileCommentStr = this.loadOptions.decodeFileName(i);
                  }
                }
              },
              findExtraFieldUnicodePath: function () {
                var e = this.extraFields[28789];
                if (e) {
                  var t = n(e.value);
                  return 1 !== t.readInt(1) ? null : a(this.fileName) !== t.readInt(4) ? null : o.utf8decode(t.readData(e.length - 5));
                }
                return null;
              },
              findExtraFieldUnicodeComment: function () {
                var e = this.extraFields[25461];
                if (e) {
                  var t = n(e.value);
                  return 1 !== t.readInt(1) ? null : a(this.fileComment) !== t.readInt(4) ? null : o.utf8decode(t.readData(e.length - 5));
                }
                return null;
              }
            }, t.exports = l;
          }, {
            "./compressedObject": 2,
            "./compressions": 3,
            "./crc32": 4,
            "./reader/readerFor": 22,
            "./support": 30,
            "./utf8": 31,
            "./utils": 32
          }],
          35: [function (e, t, r) {

            function n(e, t, r) {
              this.name = e, this.dir = r.dir, this.date = r.date, this.comment = r.comment, this.unixPermissions = r.unixPermissions, this.dosPermissions = r.dosPermissions, this._data = t, this._dataBinary = r.binary, this.options = {
                compression: r.compression,
                compressionOptions: r.compressionOptions
              };
            }
            var s = e("./stream/StreamHelper"),
              i = e("./stream/DataWorker"),
              a = e("./utf8"),
              o = e("./compressedObject"),
              h = e("./stream/GenericWorker");
            n.prototype = {
              internalStream: function (e) {
                var t = null,
                  r = "string";
                try {
                  if (!e) throw new Error("No output type specified.");
                  var n = "string" === (r = e.toLowerCase()) || "text" === r;
                  "binarystring" !== r && "text" !== r || (r = "string"), t = this._decompressWorker();
                  var i = !this._dataBinary;
                  i && !n && (t = t.pipe(new a.Utf8EncodeWorker())), !i && n && (t = t.pipe(new a.Utf8DecodeWorker()));
                } catch (e) {
                  (t = new h("error")).error(e);
                }
                return new s(t, r, "");
              },
              async: function (e, t) {
                return this.internalStream(e).accumulate(t);
              },
              nodeStream: function (e, t) {
                return this.internalStream(e || "nodebuffer").toNodejsStream(t);
              },
              _compressWorker: function (e, t) {
                if (this._data instanceof o && this._data.compression.magic === e.magic) return this._data.getCompressedWorker();
                var r = this._decompressWorker();
                return this._dataBinary || (r = r.pipe(new a.Utf8EncodeWorker())), o.createWorkerFrom(r, e, t);
              },
              _decompressWorker: function () {
                return this._data instanceof o ? this._data.getContentWorker() : this._data instanceof h ? this._data : new i(this._data);
              }
            };
            for (var u = ["asText", "asBinary", "asNodeBuffer", "asUint8Array", "asArrayBuffer"], l = function () {
                throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.");
              }, f = 0; f < u.length; f++) n.prototype[u[f]] = l;
            t.exports = n;
          }, {
            "./compressedObject": 2,
            "./stream/DataWorker": 27,
            "./stream/GenericWorker": 28,
            "./stream/StreamHelper": 29,
            "./utf8": 31
          }],
          36: [function (e, l, t) {
            (function (t) {

              var r,
                n,
                e = t.MutationObserver || t.WebKitMutationObserver;
              if (e) {
                var i = 0,
                  s = new e(u),
                  a = t.document.createTextNode("");
                s.observe(a, {
                  characterData: true
                }), r = function () {
                  a.data = i = ++i % 2;
                };
              } else if (t.setImmediate || void 0 === t.MessageChannel) r = "document" in t && "onreadystatechange" in t.document.createElement("script") ? function () {
                var e = t.document.createElement("script");
                e.onreadystatechange = function () {
                  u(), e.onreadystatechange = null, e.parentNode.removeChild(e), e = null;
                }, t.document.documentElement.appendChild(e);
              } : function () {
                setTimeout(u, 0);
              };else {
                var o = new t.MessageChannel();
                o.port1.onmessage = u, r = function () {
                  o.port2.postMessage(0);
                };
              }
              var h = [];
              function u() {
                var e, t;
                n = true;
                for (var r = h.length; r;) {
                  for (t = h, h = [], e = -1; ++e < r;) t[e]();
                  r = h.length;
                }
                n = false;
              }
              l.exports = function (e) {
                1 !== h.push(e) || n || r();
              };
            }).call(this, "undefined" != typeof commonjsGlobal ? commonjsGlobal : "undefined" != typeof self ? self : "undefined" != typeof window ? window : {});
          }, {}],
          37: [function (e, t, r) {

            var i = e("immediate");
            function u() {}
            var l = {},
              s = ["REJECTED"],
              a = ["FULFILLED"],
              n = ["PENDING"];
            function o(e) {
              if ("function" != typeof e) throw new TypeError("resolver must be a function");
              this.state = n, this.queue = [], this.outcome = void 0, e !== u && d(this, e);
            }
            function h(e, t, r) {
              this.promise = e, "function" == typeof t && (this.onFulfilled = t, this.callFulfilled = this.otherCallFulfilled), "function" == typeof r && (this.onRejected = r, this.callRejected = this.otherCallRejected);
            }
            function f(t, r, n) {
              i(function () {
                var e;
                try {
                  e = r(n);
                } catch (e) {
                  return l.reject(t, e);
                }
                e === t ? l.reject(t, new TypeError("Cannot resolve promise with itself")) : l.resolve(t, e);
              });
            }
            function c(e) {
              var t = e && e.then;
              if (e && ("object" == typeof e || "function" == typeof e) && "function" == typeof t) return function () {
                t.apply(e, arguments);
              };
            }
            function d(t, e) {
              var r = false;
              function n(e) {
                r || (r = true, l.reject(t, e));
              }
              function i(e) {
                r || (r = true, l.resolve(t, e));
              }
              var s = p(function () {
                e(i, n);
              });
              "error" === s.status && n(s.value);
            }
            function p(e, t) {
              var r = {};
              try {
                r.value = e(t), r.status = "success";
              } catch (e) {
                r.status = "error", r.value = e;
              }
              return r;
            }
            (t.exports = o).prototype.finally = function (t) {
              if ("function" != typeof t) return this;
              var r = this.constructor;
              return this.then(function (e) {
                return r.resolve(t()).then(function () {
                  return e;
                });
              }, function (e) {
                return r.resolve(t()).then(function () {
                  throw e;
                });
              });
            }, o.prototype.catch = function (e) {
              return this.then(null, e);
            }, o.prototype.then = function (e, t) {
              if ("function" != typeof e && this.state === a || "function" != typeof t && this.state === s) return this;
              var r = new this.constructor(u);
              this.state !== n ? f(r, this.state === a ? e : t, this.outcome) : this.queue.push(new h(r, e, t));
              return r;
            }, h.prototype.callFulfilled = function (e) {
              l.resolve(this.promise, e);
            }, h.prototype.otherCallFulfilled = function (e) {
              f(this.promise, this.onFulfilled, e);
            }, h.prototype.callRejected = function (e) {
              l.reject(this.promise, e);
            }, h.prototype.otherCallRejected = function (e) {
              f(this.promise, this.onRejected, e);
            }, l.resolve = function (e, t) {
              var r = p(c, t);
              if ("error" === r.status) return l.reject(e, r.value);
              var n = r.value;
              if (n) d(e, n);else {
                e.state = a, e.outcome = t;
                for (var i = -1, s = e.queue.length; ++i < s;) e.queue[i].callFulfilled(t);
              }
              return e;
            }, l.reject = function (e, t) {
              e.state = s, e.outcome = t;
              for (var r = -1, n = e.queue.length; ++r < n;) e.queue[r].callRejected(t);
              return e;
            }, o.resolve = function (e) {
              if (e instanceof this) return e;
              return l.resolve(new this(u), e);
            }, o.reject = function (e) {
              var t = new this(u);
              return l.reject(t, e);
            }, o.all = function (e) {
              var r = this;
              if ("[object Array]" !== Object.prototype.toString.call(e)) return this.reject(new TypeError("must be an array"));
              var n = e.length,
                i = false;
              if (!n) return this.resolve([]);
              var s = new Array(n),
                a = 0,
                t = -1,
                o = new this(u);
              for (; ++t < n;) h(e[t], t);
              return o;
              function h(e, t) {
                r.resolve(e).then(function (e) {
                  s[t] = e, ++a !== n || i || (i = true, l.resolve(o, s));
                }, function (e) {
                  i || (i = true, l.reject(o, e));
                });
              }
            }, o.race = function (e) {
              var t = this;
              if ("[object Array]" !== Object.prototype.toString.call(e)) return this.reject(new TypeError("must be an array"));
              var r = e.length,
                n = false;
              if (!r) return this.resolve([]);
              var i = -1,
                s = new this(u);
              for (; ++i < r;) a = e[i], t.resolve(a).then(function (e) {
                n || (n = true, l.resolve(s, e));
              }, function (e) {
                n || (n = true, l.reject(s, e));
              });
              var a;
              return s;
            };
          }, {
            immediate: 36
          }],
          38: [function (e, t, r) {

            var n = {};
            (0, e("./lib/utils/common").assign)(n, e("./lib/deflate"), e("./lib/inflate"), e("./lib/zlib/constants")), t.exports = n;
          }, {
            "./lib/deflate": 39,
            "./lib/inflate": 40,
            "./lib/utils/common": 41,
            "./lib/zlib/constants": 44
          }],
          39: [function (e, t, r) {

            var a = e("./zlib/deflate"),
              o = e("./utils/common"),
              h = e("./utils/strings"),
              i = e("./zlib/messages"),
              s = e("./zlib/zstream"),
              u = Object.prototype.toString,
              l = 0,
              f = -1,
              c = 0,
              d = 8;
            function p(e) {
              if (!(this instanceof p)) return new p(e);
              this.options = o.assign({
                level: f,
                method: d,
                chunkSize: 16384,
                windowBits: 15,
                memLevel: 8,
                strategy: c,
                to: ""
              }, e || {});
              var t = this.options;
              t.raw && 0 < t.windowBits ? t.windowBits = -t.windowBits : t.gzip && 0 < t.windowBits && t.windowBits < 16 && (t.windowBits += 16), this.err = 0, this.msg = "", this.ended = false, this.chunks = [], this.strm = new s(), this.strm.avail_out = 0;
              var r = a.deflateInit2(this.strm, t.level, t.method, t.windowBits, t.memLevel, t.strategy);
              if (r !== l) throw new Error(i[r]);
              if (t.header && a.deflateSetHeader(this.strm, t.header), t.dictionary) {
                var n;
                if (n = "string" == typeof t.dictionary ? h.string2buf(t.dictionary) : "[object ArrayBuffer]" === u.call(t.dictionary) ? new Uint8Array(t.dictionary) : t.dictionary, (r = a.deflateSetDictionary(this.strm, n)) !== l) throw new Error(i[r]);
                this._dict_set = true;
              }
            }
            function n(e, t) {
              var r = new p(t);
              if (r.push(e, true), r.err) throw r.msg || i[r.err];
              return r.result;
            }
            p.prototype.push = function (e, t) {
              var r,
                n,
                i = this.strm,
                s = this.options.chunkSize;
              if (this.ended) return false;
              n = t === ~~t ? t : true === t ? 4 : 0, "string" == typeof e ? i.input = h.string2buf(e) : "[object ArrayBuffer]" === u.call(e) ? i.input = new Uint8Array(e) : i.input = e, i.next_in = 0, i.avail_in = i.input.length;
              do {
                if (0 === i.avail_out && (i.output = new o.Buf8(s), i.next_out = 0, i.avail_out = s), 1 !== (r = a.deflate(i, n)) && r !== l) return this.onEnd(r), !(this.ended = true);
                0 !== i.avail_out && (0 !== i.avail_in || 4 !== n && 2 !== n) || ("string" === this.options.to ? this.onData(h.buf2binstring(o.shrinkBuf(i.output, i.next_out))) : this.onData(o.shrinkBuf(i.output, i.next_out)));
              } while ((0 < i.avail_in || 0 === i.avail_out) && 1 !== r);
              return 4 === n ? (r = a.deflateEnd(this.strm), this.onEnd(r), this.ended = true, r === l) : 2 !== n || (this.onEnd(l), !(i.avail_out = 0));
            }, p.prototype.onData = function (e) {
              this.chunks.push(e);
            }, p.prototype.onEnd = function (e) {
              e === l && ("string" === this.options.to ? this.result = this.chunks.join("") : this.result = o.flattenChunks(this.chunks)), this.chunks = [], this.err = e, this.msg = this.strm.msg;
            }, r.Deflate = p, r.deflate = n, r.deflateRaw = function (e, t) {
              return (t = t || {}).raw = true, n(e, t);
            }, r.gzip = function (e, t) {
              return (t = t || {}).gzip = true, n(e, t);
            };
          }, {
            "./utils/common": 41,
            "./utils/strings": 42,
            "./zlib/deflate": 46,
            "./zlib/messages": 51,
            "./zlib/zstream": 53
          }],
          40: [function (e, t, r) {

            var c = e("./zlib/inflate"),
              d = e("./utils/common"),
              p = e("./utils/strings"),
              m = e("./zlib/constants"),
              n = e("./zlib/messages"),
              i = e("./zlib/zstream"),
              s = e("./zlib/gzheader"),
              _ = Object.prototype.toString;
            function a(e) {
              if (!(this instanceof a)) return new a(e);
              this.options = d.assign({
                chunkSize: 16384,
                windowBits: 0,
                to: ""
              }, e || {});
              var t = this.options;
              t.raw && 0 <= t.windowBits && t.windowBits < 16 && (t.windowBits = -t.windowBits, 0 === t.windowBits && (t.windowBits = -15)), !(0 <= t.windowBits && t.windowBits < 16) || e && e.windowBits || (t.windowBits += 32), 15 < t.windowBits && t.windowBits < 48 && 0 == (15 & t.windowBits) && (t.windowBits |= 15), this.err = 0, this.msg = "", this.ended = false, this.chunks = [], this.strm = new i(), this.strm.avail_out = 0;
              var r = c.inflateInit2(this.strm, t.windowBits);
              if (r !== m.Z_OK) throw new Error(n[r]);
              this.header = new s(), c.inflateGetHeader(this.strm, this.header);
            }
            function o(e, t) {
              var r = new a(t);
              if (r.push(e, true), r.err) throw r.msg || n[r.err];
              return r.result;
            }
            a.prototype.push = function (e, t) {
              var r,
                n,
                i,
                s,
                a,
                o,
                h = this.strm,
                u = this.options.chunkSize,
                l = this.options.dictionary,
                f = false;
              if (this.ended) return false;
              n = t === ~~t ? t : true === t ? m.Z_FINISH : m.Z_NO_FLUSH, "string" == typeof e ? h.input = p.binstring2buf(e) : "[object ArrayBuffer]" === _.call(e) ? h.input = new Uint8Array(e) : h.input = e, h.next_in = 0, h.avail_in = h.input.length;
              do {
                if (0 === h.avail_out && (h.output = new d.Buf8(u), h.next_out = 0, h.avail_out = u), (r = c.inflate(h, m.Z_NO_FLUSH)) === m.Z_NEED_DICT && l && (o = "string" == typeof l ? p.string2buf(l) : "[object ArrayBuffer]" === _.call(l) ? new Uint8Array(l) : l, r = c.inflateSetDictionary(this.strm, o)), r === m.Z_BUF_ERROR && true === f && (r = m.Z_OK, f = false), r !== m.Z_STREAM_END && r !== m.Z_OK) return this.onEnd(r), !(this.ended = true);
                h.next_out && (0 !== h.avail_out && r !== m.Z_STREAM_END && (0 !== h.avail_in || n !== m.Z_FINISH && n !== m.Z_SYNC_FLUSH) || ("string" === this.options.to ? (i = p.utf8border(h.output, h.next_out), s = h.next_out - i, a = p.buf2string(h.output, i), h.next_out = s, h.avail_out = u - s, s && d.arraySet(h.output, h.output, i, s, 0), this.onData(a)) : this.onData(d.shrinkBuf(h.output, h.next_out)))), 0 === h.avail_in && 0 === h.avail_out && (f = true);
              } while ((0 < h.avail_in || 0 === h.avail_out) && r !== m.Z_STREAM_END);
              return r === m.Z_STREAM_END && (n = m.Z_FINISH), n === m.Z_FINISH ? (r = c.inflateEnd(this.strm), this.onEnd(r), this.ended = true, r === m.Z_OK) : n !== m.Z_SYNC_FLUSH || (this.onEnd(m.Z_OK), !(h.avail_out = 0));
            }, a.prototype.onData = function (e) {
              this.chunks.push(e);
            }, a.prototype.onEnd = function (e) {
              e === m.Z_OK && ("string" === this.options.to ? this.result = this.chunks.join("") : this.result = d.flattenChunks(this.chunks)), this.chunks = [], this.err = e, this.msg = this.strm.msg;
            }, r.Inflate = a, r.inflate = o, r.inflateRaw = function (e, t) {
              return (t = t || {}).raw = true, o(e, t);
            }, r.ungzip = o;
          }, {
            "./utils/common": 41,
            "./utils/strings": 42,
            "./zlib/constants": 44,
            "./zlib/gzheader": 47,
            "./zlib/inflate": 49,
            "./zlib/messages": 51,
            "./zlib/zstream": 53
          }],
          41: [function (e, t, r) {

            var n = "undefined" != typeof Uint8Array && "undefined" != typeof Uint16Array && "undefined" != typeof Int32Array;
            r.assign = function (e) {
              for (var t = Array.prototype.slice.call(arguments, 1); t.length;) {
                var r = t.shift();
                if (r) {
                  if ("object" != typeof r) throw new TypeError(r + "must be non-object");
                  for (var n in r) r.hasOwnProperty(n) && (e[n] = r[n]);
                }
              }
              return e;
            }, r.shrinkBuf = function (e, t) {
              return e.length === t ? e : e.subarray ? e.subarray(0, t) : (e.length = t, e);
            };
            var i = {
                arraySet: function (e, t, r, n, i) {
                  if (t.subarray && e.subarray) e.set(t.subarray(r, r + n), i);else for (var s = 0; s < n; s++) e[i + s] = t[r + s];
                },
                flattenChunks: function (e) {
                  var t, r, n, i, s, a;
                  for (t = n = 0, r = e.length; t < r; t++) n += e[t].length;
                  for (a = new Uint8Array(n), t = i = 0, r = e.length; t < r; t++) s = e[t], a.set(s, i), i += s.length;
                  return a;
                }
              },
              s = {
                arraySet: function (e, t, r, n, i) {
                  for (var s = 0; s < n; s++) e[i + s] = t[r + s];
                },
                flattenChunks: function (e) {
                  return [].concat.apply([], e);
                }
              };
            r.setTyped = function (e) {
              e ? (r.Buf8 = Uint8Array, r.Buf16 = Uint16Array, r.Buf32 = Int32Array, r.assign(r, i)) : (r.Buf8 = Array, r.Buf16 = Array, r.Buf32 = Array, r.assign(r, s));
            }, r.setTyped(n);
          }, {}],
          42: [function (e, t, r) {

            var h = e("./common"),
              i = true,
              s = true;
            try {
              String.fromCharCode.apply(null, [0]);
            } catch (e) {
              i = false;
            }
            try {
              String.fromCharCode.apply(null, new Uint8Array(1));
            } catch (e) {
              s = false;
            }
            for (var u = new h.Buf8(256), n = 0; n < 256; n++) u[n] = 252 <= n ? 6 : 248 <= n ? 5 : 240 <= n ? 4 : 224 <= n ? 3 : 192 <= n ? 2 : 1;
            function l(e, t) {
              if (t < 65537 && (e.subarray && s || !e.subarray && i)) return String.fromCharCode.apply(null, h.shrinkBuf(e, t));
              for (var r = "", n = 0; n < t; n++) r += String.fromCharCode(e[n]);
              return r;
            }
            u[254] = u[254] = 1, r.string2buf = function (e) {
              var t,
                r,
                n,
                i,
                s,
                a = e.length,
                o = 0;
              for (i = 0; i < a; i++) 55296 == (64512 & (r = e.charCodeAt(i))) && i + 1 < a && 56320 == (64512 & (n = e.charCodeAt(i + 1))) && (r = 65536 + (r - 55296 << 10) + (n - 56320), i++), o += r < 128 ? 1 : r < 2048 ? 2 : r < 65536 ? 3 : 4;
              for (t = new h.Buf8(o), i = s = 0; s < o; i++) 55296 == (64512 & (r = e.charCodeAt(i))) && i + 1 < a && 56320 == (64512 & (n = e.charCodeAt(i + 1))) && (r = 65536 + (r - 55296 << 10) + (n - 56320), i++), r < 128 ? t[s++] = r : (r < 2048 ? t[s++] = 192 | r >>> 6 : (r < 65536 ? t[s++] = 224 | r >>> 12 : (t[s++] = 240 | r >>> 18, t[s++] = 128 | r >>> 12 & 63), t[s++] = 128 | r >>> 6 & 63), t[s++] = 128 | 63 & r);
              return t;
            }, r.buf2binstring = function (e) {
              return l(e, e.length);
            }, r.binstring2buf = function (e) {
              for (var t = new h.Buf8(e.length), r = 0, n = t.length; r < n; r++) t[r] = e.charCodeAt(r);
              return t;
            }, r.buf2string = function (e, t) {
              var r,
                n,
                i,
                s,
                a = t || e.length,
                o = new Array(2 * a);
              for (r = n = 0; r < a;) if ((i = e[r++]) < 128) o[n++] = i;else if (4 < (s = u[i])) o[n++] = 65533, r += s - 1;else {
                for (i &= 2 === s ? 31 : 3 === s ? 15 : 7; 1 < s && r < a;) i = i << 6 | 63 & e[r++], s--;
                1 < s ? o[n++] = 65533 : i < 65536 ? o[n++] = i : (i -= 65536, o[n++] = 55296 | i >> 10 & 1023, o[n++] = 56320 | 1023 & i);
              }
              return l(o, n);
            }, r.utf8border = function (e, t) {
              var r;
              for ((t = t || e.length) > e.length && (t = e.length), r = t - 1; 0 <= r && 128 == (192 & e[r]);) r--;
              return r < 0 ? t : 0 === r ? t : r + u[e[r]] > t ? r : t;
            };
          }, {
            "./common": 41
          }],
          43: [function (e, t, r) {

            t.exports = function (e, t, r, n) {
              for (var i = 65535 & e | 0, s = e >>> 16 & 65535 | 0, a = 0; 0 !== r;) {
                for (r -= a = 2e3 < r ? 2e3 : r; s = s + (i = i + t[n++] | 0) | 0, --a;);
                i %= 65521, s %= 65521;
              }
              return i | s << 16 | 0;
            };
          }, {}],
          44: [function (e, t, r) {

            t.exports = {
              Z_NO_FLUSH: 0,
              Z_PARTIAL_FLUSH: 1,
              Z_SYNC_FLUSH: 2,
              Z_FULL_FLUSH: 3,
              Z_FINISH: 4,
              Z_BLOCK: 5,
              Z_TREES: 6,
              Z_OK: 0,
              Z_STREAM_END: 1,
              Z_NEED_DICT: 2,
              Z_ERRNO: -1,
              Z_STREAM_ERROR: -2,
              Z_DATA_ERROR: -3,
              Z_BUF_ERROR: -5,
              Z_NO_COMPRESSION: 0,
              Z_BEST_SPEED: 1,
              Z_BEST_COMPRESSION: 9,
              Z_DEFAULT_COMPRESSION: -1,
              Z_FILTERED: 1,
              Z_HUFFMAN_ONLY: 2,
              Z_RLE: 3,
              Z_FIXED: 4,
              Z_DEFAULT_STRATEGY: 0,
              Z_BINARY: 0,
              Z_TEXT: 1,
              Z_UNKNOWN: 2,
              Z_DEFLATED: 8
            };
          }, {}],
          45: [function (e, t, r) {

            var o = function () {
              for (var e, t = [], r = 0; r < 256; r++) {
                e = r;
                for (var n = 0; n < 8; n++) e = 1 & e ? 3988292384 ^ e >>> 1 : e >>> 1;
                t[r] = e;
              }
              return t;
            }();
            t.exports = function (e, t, r, n) {
              var i = o,
                s = n + r;
              e ^= -1;
              for (var a = n; a < s; a++) e = e >>> 8 ^ i[255 & (e ^ t[a])];
              return -1 ^ e;
            };
          }, {}],
          46: [function (e, t, r) {

            var h,
              c = e("../utils/common"),
              u = e("./trees"),
              d = e("./adler32"),
              p = e("./crc32"),
              n = e("./messages"),
              l = 0,
              f = 4,
              m = 0,
              _ = -2,
              g = -1,
              b = 4,
              i = 2,
              v = 8,
              y = 9,
              s = 286,
              a = 30,
              o = 19,
              w = 2 * s + 1,
              k = 15,
              x = 3,
              S = 258,
              z = S + x + 1,
              C = 42,
              E = 113,
              A = 1,
              I = 2,
              O = 3,
              B = 4;
            function R(e, t) {
              return e.msg = n[t], t;
            }
            function T(e) {
              return (e << 1) - (4 < e ? 9 : 0);
            }
            function D(e) {
              for (var t = e.length; 0 <= --t;) e[t] = 0;
            }
            function F(e) {
              var t = e.state,
                r = t.pending;
              r > e.avail_out && (r = e.avail_out), 0 !== r && (c.arraySet(e.output, t.pending_buf, t.pending_out, r, e.next_out), e.next_out += r, t.pending_out += r, e.total_out += r, e.avail_out -= r, t.pending -= r, 0 === t.pending && (t.pending_out = 0));
            }
            function N(e, t) {
              u._tr_flush_block(e, 0 <= e.block_start ? e.block_start : -1, e.strstart - e.block_start, t), e.block_start = e.strstart, F(e.strm);
            }
            function U(e, t) {
              e.pending_buf[e.pending++] = t;
            }
            function P(e, t) {
              e.pending_buf[e.pending++] = t >>> 8 & 255, e.pending_buf[e.pending++] = 255 & t;
            }
            function L(e, t) {
              var r,
                n,
                i = e.max_chain_length,
                s = e.strstart,
                a = e.prev_length,
                o = e.nice_match,
                h = e.strstart > e.w_size - z ? e.strstart - (e.w_size - z) : 0,
                u = e.window,
                l = e.w_mask,
                f = e.prev,
                c = e.strstart + S,
                d = u[s + a - 1],
                p = u[s + a];
              e.prev_length >= e.good_match && (i >>= 2), o > e.lookahead && (o = e.lookahead);
              do {
                if (u[(r = t) + a] === p && u[r + a - 1] === d && u[r] === u[s] && u[++r] === u[s + 1]) {
                  s += 2, r++;
                  do {} while (u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && s < c);
                  if (n = S - (c - s), s = c - S, a < n) {
                    if (e.match_start = t, o <= (a = n)) break;
                    d = u[s + a - 1], p = u[s + a];
                  }
                }
              } while ((t = f[t & l]) > h && 0 != --i);
              return a <= e.lookahead ? a : e.lookahead;
            }
            function j(e) {
              var t,
                r,
                n,
                i,
                s,
                a,
                o,
                h,
                u,
                l,
                f = e.w_size;
              do {
                if (i = e.window_size - e.lookahead - e.strstart, e.strstart >= f + (f - z)) {
                  for (c.arraySet(e.window, e.window, f, f, 0), e.match_start -= f, e.strstart -= f, e.block_start -= f, t = r = e.hash_size; n = e.head[--t], e.head[t] = f <= n ? n - f : 0, --r;);
                  for (t = r = f; n = e.prev[--t], e.prev[t] = f <= n ? n - f : 0, --r;);
                  i += f;
                }
                if (0 === e.strm.avail_in) break;
                if (a = e.strm, o = e.window, h = e.strstart + e.lookahead, u = i, l = void 0, l = a.avail_in, u < l && (l = u), r = 0 === l ? 0 : (a.avail_in -= l, c.arraySet(o, a.input, a.next_in, l, h), 1 === a.state.wrap ? a.adler = d(a.adler, o, l, h) : 2 === a.state.wrap && (a.adler = p(a.adler, o, l, h)), a.next_in += l, a.total_in += l, l), e.lookahead += r, e.lookahead + e.insert >= x) for (s = e.strstart - e.insert, e.ins_h = e.window[s], e.ins_h = (e.ins_h << e.hash_shift ^ e.window[s + 1]) & e.hash_mask; e.insert && (e.ins_h = (e.ins_h << e.hash_shift ^ e.window[s + x - 1]) & e.hash_mask, e.prev[s & e.w_mask] = e.head[e.ins_h], e.head[e.ins_h] = s, s++, e.insert--, !(e.lookahead + e.insert < x)););
              } while (e.lookahead < z && 0 !== e.strm.avail_in);
            }
            function Z(e, t) {
              for (var r, n;;) {
                if (e.lookahead < z) {
                  if (j(e), e.lookahead < z && t === l) return A;
                  if (0 === e.lookahead) break;
                }
                if (r = 0, e.lookahead >= x && (e.ins_h = (e.ins_h << e.hash_shift ^ e.window[e.strstart + x - 1]) & e.hash_mask, r = e.prev[e.strstart & e.w_mask] = e.head[e.ins_h], e.head[e.ins_h] = e.strstart), 0 !== r && e.strstart - r <= e.w_size - z && (e.match_length = L(e, r)), e.match_length >= x) {
                  if (n = u._tr_tally(e, e.strstart - e.match_start, e.match_length - x), e.lookahead -= e.match_length, e.match_length <= e.max_lazy_match && e.lookahead >= x) {
                    for (e.match_length--; e.strstart++, e.ins_h = (e.ins_h << e.hash_shift ^ e.window[e.strstart + x - 1]) & e.hash_mask, r = e.prev[e.strstart & e.w_mask] = e.head[e.ins_h], e.head[e.ins_h] = e.strstart, 0 != --e.match_length;);
                    e.strstart++;
                  } else e.strstart += e.match_length, e.match_length = 0, e.ins_h = e.window[e.strstart], e.ins_h = (e.ins_h << e.hash_shift ^ e.window[e.strstart + 1]) & e.hash_mask;
                } else n = u._tr_tally(e, 0, e.window[e.strstart]), e.lookahead--, e.strstart++;
                if (n && (N(e, false), 0 === e.strm.avail_out)) return A;
              }
              return e.insert = e.strstart < x - 1 ? e.strstart : x - 1, t === f ? (N(e, true), 0 === e.strm.avail_out ? O : B) : e.last_lit && (N(e, false), 0 === e.strm.avail_out) ? A : I;
            }
            function W(e, t) {
              for (var r, n, i;;) {
                if (e.lookahead < z) {
                  if (j(e), e.lookahead < z && t === l) return A;
                  if (0 === e.lookahead) break;
                }
                if (r = 0, e.lookahead >= x && (e.ins_h = (e.ins_h << e.hash_shift ^ e.window[e.strstart + x - 1]) & e.hash_mask, r = e.prev[e.strstart & e.w_mask] = e.head[e.ins_h], e.head[e.ins_h] = e.strstart), e.prev_length = e.match_length, e.prev_match = e.match_start, e.match_length = x - 1, 0 !== r && e.prev_length < e.max_lazy_match && e.strstart - r <= e.w_size - z && (e.match_length = L(e, r), e.match_length <= 5 && (1 === e.strategy || e.match_length === x && 4096 < e.strstart - e.match_start) && (e.match_length = x - 1)), e.prev_length >= x && e.match_length <= e.prev_length) {
                  for (i = e.strstart + e.lookahead - x, n = u._tr_tally(e, e.strstart - 1 - e.prev_match, e.prev_length - x), e.lookahead -= e.prev_length - 1, e.prev_length -= 2; ++e.strstart <= i && (e.ins_h = (e.ins_h << e.hash_shift ^ e.window[e.strstart + x - 1]) & e.hash_mask, r = e.prev[e.strstart & e.w_mask] = e.head[e.ins_h], e.head[e.ins_h] = e.strstart), 0 != --e.prev_length;);
                  if (e.match_available = 0, e.match_length = x - 1, e.strstart++, n && (N(e, false), 0 === e.strm.avail_out)) return A;
                } else if (e.match_available) {
                  if ((n = u._tr_tally(e, 0, e.window[e.strstart - 1])) && N(e, false), e.strstart++, e.lookahead--, 0 === e.strm.avail_out) return A;
                } else e.match_available = 1, e.strstart++, e.lookahead--;
              }
              return e.match_available && (n = u._tr_tally(e, 0, e.window[e.strstart - 1]), e.match_available = 0), e.insert = e.strstart < x - 1 ? e.strstart : x - 1, t === f ? (N(e, true), 0 === e.strm.avail_out ? O : B) : e.last_lit && (N(e, false), 0 === e.strm.avail_out) ? A : I;
            }
            function M(e, t, r, n, i) {
              this.good_length = e, this.max_lazy = t, this.nice_length = r, this.max_chain = n, this.func = i;
            }
            function H() {
              this.strm = null, this.status = 0, this.pending_buf = null, this.pending_buf_size = 0, this.pending_out = 0, this.pending = 0, this.wrap = 0, this.gzhead = null, this.gzindex = 0, this.method = v, this.last_flush = -1, this.w_size = 0, this.w_bits = 0, this.w_mask = 0, this.window = null, this.window_size = 0, this.prev = null, this.head = null, this.ins_h = 0, this.hash_size = 0, this.hash_bits = 0, this.hash_mask = 0, this.hash_shift = 0, this.block_start = 0, this.match_length = 0, this.prev_match = 0, this.match_available = 0, this.strstart = 0, this.match_start = 0, this.lookahead = 0, this.prev_length = 0, this.max_chain_length = 0, this.max_lazy_match = 0, this.level = 0, this.strategy = 0, this.good_match = 0, this.nice_match = 0, this.dyn_ltree = new c.Buf16(2 * w), this.dyn_dtree = new c.Buf16(2 * (2 * a + 1)), this.bl_tree = new c.Buf16(2 * (2 * o + 1)), D(this.dyn_ltree), D(this.dyn_dtree), D(this.bl_tree), this.l_desc = null, this.d_desc = null, this.bl_desc = null, this.bl_count = new c.Buf16(k + 1), this.heap = new c.Buf16(2 * s + 1), D(this.heap), this.heap_len = 0, this.heap_max = 0, this.depth = new c.Buf16(2 * s + 1), D(this.depth), this.l_buf = 0, this.lit_bufsize = 0, this.last_lit = 0, this.d_buf = 0, this.opt_len = 0, this.static_len = 0, this.matches = 0, this.insert = 0, this.bi_buf = 0, this.bi_valid = 0;
            }
            function G(e) {
              var t;
              return e && e.state ? (e.total_in = e.total_out = 0, e.data_type = i, (t = e.state).pending = 0, t.pending_out = 0, t.wrap < 0 && (t.wrap = -t.wrap), t.status = t.wrap ? C : E, e.adler = 2 === t.wrap ? 0 : 1, t.last_flush = l, u._tr_init(t), m) : R(e, _);
            }
            function K(e) {
              var t = G(e);
              return t === m && function (e) {
                e.window_size = 2 * e.w_size, D(e.head), e.max_lazy_match = h[e.level].max_lazy, e.good_match = h[e.level].good_length, e.nice_match = h[e.level].nice_length, e.max_chain_length = h[e.level].max_chain, e.strstart = 0, e.block_start = 0, e.lookahead = 0, e.insert = 0, e.match_length = e.prev_length = x - 1, e.match_available = 0, e.ins_h = 0;
              }(e.state), t;
            }
            function Y(e, t, r, n, i, s) {
              if (!e) return _;
              var a = 1;
              if (t === g && (t = 6), n < 0 ? (a = 0, n = -n) : 15 < n && (a = 2, n -= 16), i < 1 || y < i || r !== v || n < 8 || 15 < n || t < 0 || 9 < t || s < 0 || b < s) return R(e, _);
              8 === n && (n = 9);
              var o = new H();
              return (e.state = o).strm = e, o.wrap = a, o.gzhead = null, o.w_bits = n, o.w_size = 1 << o.w_bits, o.w_mask = o.w_size - 1, o.hash_bits = i + 7, o.hash_size = 1 << o.hash_bits, o.hash_mask = o.hash_size - 1, o.hash_shift = ~~((o.hash_bits + x - 1) / x), o.window = new c.Buf8(2 * o.w_size), o.head = new c.Buf16(o.hash_size), o.prev = new c.Buf16(o.w_size), o.lit_bufsize = 1 << i + 6, o.pending_buf_size = 4 * o.lit_bufsize, o.pending_buf = new c.Buf8(o.pending_buf_size), o.d_buf = 1 * o.lit_bufsize, o.l_buf = 3 * o.lit_bufsize, o.level = t, o.strategy = s, o.method = r, K(e);
            }
            h = [new M(0, 0, 0, 0, function (e, t) {
              var r = 65535;
              for (r > e.pending_buf_size - 5 && (r = e.pending_buf_size - 5);;) {
                if (e.lookahead <= 1) {
                  if (j(e), 0 === e.lookahead && t === l) return A;
                  if (0 === e.lookahead) break;
                }
                e.strstart += e.lookahead, e.lookahead = 0;
                var n = e.block_start + r;
                if ((0 === e.strstart || e.strstart >= n) && (e.lookahead = e.strstart - n, e.strstart = n, N(e, false), 0 === e.strm.avail_out)) return A;
                if (e.strstart - e.block_start >= e.w_size - z && (N(e, false), 0 === e.strm.avail_out)) return A;
              }
              return e.insert = 0, t === f ? (N(e, true), 0 === e.strm.avail_out ? O : B) : (e.strstart > e.block_start && (N(e, false), e.strm.avail_out), A);
            }), new M(4, 4, 8, 4, Z), new M(4, 5, 16, 8, Z), new M(4, 6, 32, 32, Z), new M(4, 4, 16, 16, W), new M(8, 16, 32, 32, W), new M(8, 16, 128, 128, W), new M(8, 32, 128, 256, W), new M(32, 128, 258, 1024, W), new M(32, 258, 258, 4096, W)], r.deflateInit = function (e, t) {
              return Y(e, t, v, 15, 8, 0);
            }, r.deflateInit2 = Y, r.deflateReset = K, r.deflateResetKeep = G, r.deflateSetHeader = function (e, t) {
              return e && e.state ? 2 !== e.state.wrap ? _ : (e.state.gzhead = t, m) : _;
            }, r.deflate = function (e, t) {
              var r, n, i, s;
              if (!e || !e.state || 5 < t || t < 0) return e ? R(e, _) : _;
              if (n = e.state, !e.output || !e.input && 0 !== e.avail_in || 666 === n.status && t !== f) return R(e, 0 === e.avail_out ? -5 : _);
              if (n.strm = e, r = n.last_flush, n.last_flush = t, n.status === C) if (2 === n.wrap) e.adler = 0, U(n, 31), U(n, 139), U(n, 8), n.gzhead ? (U(n, (n.gzhead.text ? 1 : 0) + (n.gzhead.hcrc ? 2 : 0) + (n.gzhead.extra ? 4 : 0) + (n.gzhead.name ? 8 : 0) + (n.gzhead.comment ? 16 : 0)), U(n, 255 & n.gzhead.time), U(n, n.gzhead.time >> 8 & 255), U(n, n.gzhead.time >> 16 & 255), U(n, n.gzhead.time >> 24 & 255), U(n, 9 === n.level ? 2 : 2 <= n.strategy || n.level < 2 ? 4 : 0), U(n, 255 & n.gzhead.os), n.gzhead.extra && n.gzhead.extra.length && (U(n, 255 & n.gzhead.extra.length), U(n, n.gzhead.extra.length >> 8 & 255)), n.gzhead.hcrc && (e.adler = p(e.adler, n.pending_buf, n.pending, 0)), n.gzindex = 0, n.status = 69) : (U(n, 0), U(n, 0), U(n, 0), U(n, 0), U(n, 0), U(n, 9 === n.level ? 2 : 2 <= n.strategy || n.level < 2 ? 4 : 0), U(n, 3), n.status = E);else {
                var a = v + (n.w_bits - 8 << 4) << 8;
                a |= (2 <= n.strategy || n.level < 2 ? 0 : n.level < 6 ? 1 : 6 === n.level ? 2 : 3) << 6, 0 !== n.strstart && (a |= 32), a += 31 - a % 31, n.status = E, P(n, a), 0 !== n.strstart && (P(n, e.adler >>> 16), P(n, 65535 & e.adler)), e.adler = 1;
              }
              if (69 === n.status) if (n.gzhead.extra) {
                for (i = n.pending; n.gzindex < (65535 & n.gzhead.extra.length) && (n.pending !== n.pending_buf_size || (n.gzhead.hcrc && n.pending > i && (e.adler = p(e.adler, n.pending_buf, n.pending - i, i)), F(e), i = n.pending, n.pending !== n.pending_buf_size));) U(n, 255 & n.gzhead.extra[n.gzindex]), n.gzindex++;
                n.gzhead.hcrc && n.pending > i && (e.adler = p(e.adler, n.pending_buf, n.pending - i, i)), n.gzindex === n.gzhead.extra.length && (n.gzindex = 0, n.status = 73);
              } else n.status = 73;
              if (73 === n.status) if (n.gzhead.name) {
                i = n.pending;
                do {
                  if (n.pending === n.pending_buf_size && (n.gzhead.hcrc && n.pending > i && (e.adler = p(e.adler, n.pending_buf, n.pending - i, i)), F(e), i = n.pending, n.pending === n.pending_buf_size)) {
                    s = 1;
                    break;
                  }
                  s = n.gzindex < n.gzhead.name.length ? 255 & n.gzhead.name.charCodeAt(n.gzindex++) : 0, U(n, s);
                } while (0 !== s);
                n.gzhead.hcrc && n.pending > i && (e.adler = p(e.adler, n.pending_buf, n.pending - i, i)), 0 === s && (n.gzindex = 0, n.status = 91);
              } else n.status = 91;
              if (91 === n.status) if (n.gzhead.comment) {
                i = n.pending;
                do {
                  if (n.pending === n.pending_buf_size && (n.gzhead.hcrc && n.pending > i && (e.adler = p(e.adler, n.pending_buf, n.pending - i, i)), F(e), i = n.pending, n.pending === n.pending_buf_size)) {
                    s = 1;
                    break;
                  }
                  s = n.gzindex < n.gzhead.comment.length ? 255 & n.gzhead.comment.charCodeAt(n.gzindex++) : 0, U(n, s);
                } while (0 !== s);
                n.gzhead.hcrc && n.pending > i && (e.adler = p(e.adler, n.pending_buf, n.pending - i, i)), 0 === s && (n.status = 103);
              } else n.status = 103;
              if (103 === n.status && (n.gzhead.hcrc ? (n.pending + 2 > n.pending_buf_size && F(e), n.pending + 2 <= n.pending_buf_size && (U(n, 255 & e.adler), U(n, e.adler >> 8 & 255), e.adler = 0, n.status = E)) : n.status = E), 0 !== n.pending) {
                if (F(e), 0 === e.avail_out) return n.last_flush = -1, m;
              } else if (0 === e.avail_in && T(t) <= T(r) && t !== f) return R(e, -5);
              if (666 === n.status && 0 !== e.avail_in) return R(e, -5);
              if (0 !== e.avail_in || 0 !== n.lookahead || t !== l && 666 !== n.status) {
                var o = 2 === n.strategy ? function (e, t) {
                  for (var r;;) {
                    if (0 === e.lookahead && (j(e), 0 === e.lookahead)) {
                      if (t === l) return A;
                      break;
                    }
                    if (e.match_length = 0, r = u._tr_tally(e, 0, e.window[e.strstart]), e.lookahead--, e.strstart++, r && (N(e, false), 0 === e.strm.avail_out)) return A;
                  }
                  return e.insert = 0, t === f ? (N(e, true), 0 === e.strm.avail_out ? O : B) : e.last_lit && (N(e, false), 0 === e.strm.avail_out) ? A : I;
                }(n, t) : 3 === n.strategy ? function (e, t) {
                  for (var r, n, i, s, a = e.window;;) {
                    if (e.lookahead <= S) {
                      if (j(e), e.lookahead <= S && t === l) return A;
                      if (0 === e.lookahead) break;
                    }
                    if (e.match_length = 0, e.lookahead >= x && 0 < e.strstart && (n = a[i = e.strstart - 1]) === a[++i] && n === a[++i] && n === a[++i]) {
                      s = e.strstart + S;
                      do {} while (n === a[++i] && n === a[++i] && n === a[++i] && n === a[++i] && n === a[++i] && n === a[++i] && n === a[++i] && n === a[++i] && i < s);
                      e.match_length = S - (s - i), e.match_length > e.lookahead && (e.match_length = e.lookahead);
                    }
                    if (e.match_length >= x ? (r = u._tr_tally(e, 1, e.match_length - x), e.lookahead -= e.match_length, e.strstart += e.match_length, e.match_length = 0) : (r = u._tr_tally(e, 0, e.window[e.strstart]), e.lookahead--, e.strstart++), r && (N(e, false), 0 === e.strm.avail_out)) return A;
                  }
                  return e.insert = 0, t === f ? (N(e, true), 0 === e.strm.avail_out ? O : B) : e.last_lit && (N(e, false), 0 === e.strm.avail_out) ? A : I;
                }(n, t) : h[n.level].func(n, t);
                if (o !== O && o !== B || (n.status = 666), o === A || o === O) return 0 === e.avail_out && (n.last_flush = -1), m;
                if (o === I && (1 === t ? u._tr_align(n) : 5 !== t && (u._tr_stored_block(n, 0, 0, false), 3 === t && (D(n.head), 0 === n.lookahead && (n.strstart = 0, n.block_start = 0, n.insert = 0))), F(e), 0 === e.avail_out)) return n.last_flush = -1, m;
              }
              return t !== f ? m : n.wrap <= 0 ? 1 : (2 === n.wrap ? (U(n, 255 & e.adler), U(n, e.adler >> 8 & 255), U(n, e.adler >> 16 & 255), U(n, e.adler >> 24 & 255), U(n, 255 & e.total_in), U(n, e.total_in >> 8 & 255), U(n, e.total_in >> 16 & 255), U(n, e.total_in >> 24 & 255)) : (P(n, e.adler >>> 16), P(n, 65535 & e.adler)), F(e), 0 < n.wrap && (n.wrap = -n.wrap), 0 !== n.pending ? m : 1);
            }, r.deflateEnd = function (e) {
              var t;
              return e && e.state ? (t = e.state.status) !== C && 69 !== t && 73 !== t && 91 !== t && 103 !== t && t !== E && 666 !== t ? R(e, _) : (e.state = null, t === E ? R(e, -3) : m) : _;
            }, r.deflateSetDictionary = function (e, t) {
              var r,
                n,
                i,
                s,
                a,
                o,
                h,
                u,
                l = t.length;
              if (!e || !e.state) return _;
              if (2 === (s = (r = e.state).wrap) || 1 === s && r.status !== C || r.lookahead) return _;
              for (1 === s && (e.adler = d(e.adler, t, l, 0)), r.wrap = 0, l >= r.w_size && (0 === s && (D(r.head), r.strstart = 0, r.block_start = 0, r.insert = 0), u = new c.Buf8(r.w_size), c.arraySet(u, t, l - r.w_size, r.w_size, 0), t = u, l = r.w_size), a = e.avail_in, o = e.next_in, h = e.input, e.avail_in = l, e.next_in = 0, e.input = t, j(r); r.lookahead >= x;) {
                for (n = r.strstart, i = r.lookahead - (x - 1); r.ins_h = (r.ins_h << r.hash_shift ^ r.window[n + x - 1]) & r.hash_mask, r.prev[n & r.w_mask] = r.head[r.ins_h], r.head[r.ins_h] = n, n++, --i;);
                r.strstart = n, r.lookahead = x - 1, j(r);
              }
              return r.strstart += r.lookahead, r.block_start = r.strstart, r.insert = r.lookahead, r.lookahead = 0, r.match_length = r.prev_length = x - 1, r.match_available = 0, e.next_in = o, e.input = h, e.avail_in = a, r.wrap = s, m;
            }, r.deflateInfo = "pako deflate (from Nodeca project)";
          }, {
            "../utils/common": 41,
            "./adler32": 43,
            "./crc32": 45,
            "./messages": 51,
            "./trees": 52
          }],
          47: [function (e, t, r) {

            t.exports = function () {
              this.text = 0, this.time = 0, this.xflags = 0, this.os = 0, this.extra = null, this.extra_len = 0, this.name = "", this.comment = "", this.hcrc = 0, this.done = false;
            };
          }, {}],
          48: [function (e, t, r) {

            t.exports = function (e, t) {
              var r, n, i, s, a, o, h, u, l, f, c, d, p, m, _, g, b, v, y, w, k, x, S, z, C;
              r = e.state, n = e.next_in, z = e.input, i = n + (e.avail_in - 5), s = e.next_out, C = e.output, a = s - (t - e.avail_out), o = s + (e.avail_out - 257), h = r.dmax, u = r.wsize, l = r.whave, f = r.wnext, c = r.window, d = r.hold, p = r.bits, m = r.lencode, _ = r.distcode, g = (1 << r.lenbits) - 1, b = (1 << r.distbits) - 1;
              e: do {
                p < 15 && (d += z[n++] << p, p += 8, d += z[n++] << p, p += 8), v = m[d & g];
                t: for (;;) {
                  if (d >>>= y = v >>> 24, p -= y, 0 === (y = v >>> 16 & 255)) C[s++] = 65535 & v;else {
                    if (!(16 & y)) {
                      if (0 == (64 & y)) {
                        v = m[(65535 & v) + (d & (1 << y) - 1)];
                        continue t;
                      }
                      if (32 & y) {
                        r.mode = 12;
                        break e;
                      }
                      e.msg = "invalid literal/length code", r.mode = 30;
                      break e;
                    }
                    w = 65535 & v, (y &= 15) && (p < y && (d += z[n++] << p, p += 8), w += d & (1 << y) - 1, d >>>= y, p -= y), p < 15 && (d += z[n++] << p, p += 8, d += z[n++] << p, p += 8), v = _[d & b];
                    r: for (;;) {
                      if (d >>>= y = v >>> 24, p -= y, !(16 & (y = v >>> 16 & 255))) {
                        if (0 == (64 & y)) {
                          v = _[(65535 & v) + (d & (1 << y) - 1)];
                          continue r;
                        }
                        e.msg = "invalid distance code", r.mode = 30;
                        break e;
                      }
                      if (k = 65535 & v, p < (y &= 15) && (d += z[n++] << p, (p += 8) < y && (d += z[n++] << p, p += 8)), h < (k += d & (1 << y) - 1)) {
                        e.msg = "invalid distance too far back", r.mode = 30;
                        break e;
                      }
                      if (d >>>= y, p -= y, (y = s - a) < k) {
                        if (l < (y = k - y) && r.sane) {
                          e.msg = "invalid distance too far back", r.mode = 30;
                          break e;
                        }
                        if (S = c, (x = 0) === f) {
                          if (x += u - y, y < w) {
                            for (w -= y; C[s++] = c[x++], --y;);
                            x = s - k, S = C;
                          }
                        } else if (f < y) {
                          if (x += u + f - y, (y -= f) < w) {
                            for (w -= y; C[s++] = c[x++], --y;);
                            if (x = 0, f < w) {
                              for (w -= y = f; C[s++] = c[x++], --y;);
                              x = s - k, S = C;
                            }
                          }
                        } else if (x += f - y, y < w) {
                          for (w -= y; C[s++] = c[x++], --y;);
                          x = s - k, S = C;
                        }
                        for (; 2 < w;) C[s++] = S[x++], C[s++] = S[x++], C[s++] = S[x++], w -= 3;
                        w && (C[s++] = S[x++], 1 < w && (C[s++] = S[x++]));
                      } else {
                        for (x = s - k; C[s++] = C[x++], C[s++] = C[x++], C[s++] = C[x++], 2 < (w -= 3););
                        w && (C[s++] = C[x++], 1 < w && (C[s++] = C[x++]));
                      }
                      break;
                    }
                  }
                  break;
                }
              } while (n < i && s < o);
              n -= w = p >> 3, d &= (1 << (p -= w << 3)) - 1, e.next_in = n, e.next_out = s, e.avail_in = n < i ? i - n + 5 : 5 - (n - i), e.avail_out = s < o ? o - s + 257 : 257 - (s - o), r.hold = d, r.bits = p;
            };
          }, {}],
          49: [function (e, t, r) {

            var I = e("../utils/common"),
              O = e("./adler32"),
              B = e("./crc32"),
              R = e("./inffast"),
              T = e("./inftrees"),
              D = 1,
              F = 2,
              N = 0,
              U = -2,
              P = 1,
              n = 852,
              i = 592;
            function L(e) {
              return (e >>> 24 & 255) + (e >>> 8 & 65280) + ((65280 & e) << 8) + ((255 & e) << 24);
            }
            function s() {
              this.mode = 0, this.last = false, this.wrap = 0, this.havedict = false, this.flags = 0, this.dmax = 0, this.check = 0, this.total = 0, this.head = null, this.wbits = 0, this.wsize = 0, this.whave = 0, this.wnext = 0, this.window = null, this.hold = 0, this.bits = 0, this.length = 0, this.offset = 0, this.extra = 0, this.lencode = null, this.distcode = null, this.lenbits = 0, this.distbits = 0, this.ncode = 0, this.nlen = 0, this.ndist = 0, this.have = 0, this.next = null, this.lens = new I.Buf16(320), this.work = new I.Buf16(288), this.lendyn = null, this.distdyn = null, this.sane = 0, this.back = 0, this.was = 0;
            }
            function a(e) {
              var t;
              return e && e.state ? (t = e.state, e.total_in = e.total_out = t.total = 0, e.msg = "", t.wrap && (e.adler = 1 & t.wrap), t.mode = P, t.last = 0, t.havedict = 0, t.dmax = 32768, t.head = null, t.hold = 0, t.bits = 0, t.lencode = t.lendyn = new I.Buf32(n), t.distcode = t.distdyn = new I.Buf32(i), t.sane = 1, t.back = -1, N) : U;
            }
            function o(e) {
              var t;
              return e && e.state ? ((t = e.state).wsize = 0, t.whave = 0, t.wnext = 0, a(e)) : U;
            }
            function h(e, t) {
              var r, n;
              return e && e.state ? (n = e.state, t < 0 ? (r = 0, t = -t) : (r = 1 + (t >> 4), t < 48 && (t &= 15)), t && (t < 8 || 15 < t) ? U : (null !== n.window && n.wbits !== t && (n.window = null), n.wrap = r, n.wbits = t, o(e))) : U;
            }
            function u(e, t) {
              var r, n;
              return e ? (n = new s(), (e.state = n).window = null, (r = h(e, t)) !== N && (e.state = null), r) : U;
            }
            var l,
              f,
              c = true;
            function j(e) {
              if (c) {
                var t;
                for (l = new I.Buf32(512), f = new I.Buf32(32), t = 0; t < 144;) e.lens[t++] = 8;
                for (; t < 256;) e.lens[t++] = 9;
                for (; t < 280;) e.lens[t++] = 7;
                for (; t < 288;) e.lens[t++] = 8;
                for (T(D, e.lens, 0, 288, l, 0, e.work, {
                  bits: 9
                }), t = 0; t < 32;) e.lens[t++] = 5;
                T(F, e.lens, 0, 32, f, 0, e.work, {
                  bits: 5
                }), c = false;
              }
              e.lencode = l, e.lenbits = 9, e.distcode = f, e.distbits = 5;
            }
            function Z(e, t, r, n) {
              var i,
                s = e.state;
              return null === s.window && (s.wsize = 1 << s.wbits, s.wnext = 0, s.whave = 0, s.window = new I.Buf8(s.wsize)), n >= s.wsize ? (I.arraySet(s.window, t, r - s.wsize, s.wsize, 0), s.wnext = 0, s.whave = s.wsize) : (n < (i = s.wsize - s.wnext) && (i = n), I.arraySet(s.window, t, r - n, i, s.wnext), (n -= i) ? (I.arraySet(s.window, t, r - n, n, 0), s.wnext = n, s.whave = s.wsize) : (s.wnext += i, s.wnext === s.wsize && (s.wnext = 0), s.whave < s.wsize && (s.whave += i))), 0;
            }
            r.inflateReset = o, r.inflateReset2 = h, r.inflateResetKeep = a, r.inflateInit = function (e) {
              return u(e, 15);
            }, r.inflateInit2 = u, r.inflate = function (e, t) {
              var r,
                n,
                i,
                s,
                a,
                o,
                h,
                u,
                l,
                f,
                c,
                d,
                p,
                m,
                _,
                g,
                b,
                v,
                y,
                w,
                k,
                x,
                S,
                z,
                C = 0,
                E = new I.Buf8(4),
                A = [16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15];
              if (!e || !e.state || !e.output || !e.input && 0 !== e.avail_in) return U;
              12 === (r = e.state).mode && (r.mode = 13), a = e.next_out, i = e.output, h = e.avail_out, s = e.next_in, n = e.input, o = e.avail_in, u = r.hold, l = r.bits, f = o, c = h, x = N;
              e: for (;;) switch (r.mode) {
                case P:
                  if (0 === r.wrap) {
                    r.mode = 13;
                    break;
                  }
                  for (; l < 16;) {
                    if (0 === o) break e;
                    o--, u += n[s++] << l, l += 8;
                  }
                  if (2 & r.wrap && 35615 === u) {
                    E[r.check = 0] = 255 & u, E[1] = u >>> 8 & 255, r.check = B(r.check, E, 2, 0), l = u = 0, r.mode = 2;
                    break;
                  }
                  if (r.flags = 0, r.head && (r.head.done = false), !(1 & r.wrap) || (((255 & u) << 8) + (u >> 8)) % 31) {
                    e.msg = "incorrect header check", r.mode = 30;
                    break;
                  }
                  if (8 != (15 & u)) {
                    e.msg = "unknown compression method", r.mode = 30;
                    break;
                  }
                  if (l -= 4, k = 8 + (15 & (u >>>= 4)), 0 === r.wbits) r.wbits = k;else if (k > r.wbits) {
                    e.msg = "invalid window size", r.mode = 30;
                    break;
                  }
                  r.dmax = 1 << k, e.adler = r.check = 1, r.mode = 512 & u ? 10 : 12, l = u = 0;
                  break;
                case 2:
                  for (; l < 16;) {
                    if (0 === o) break e;
                    o--, u += n[s++] << l, l += 8;
                  }
                  if (r.flags = u, 8 != (255 & r.flags)) {
                    e.msg = "unknown compression method", r.mode = 30;
                    break;
                  }
                  if (57344 & r.flags) {
                    e.msg = "unknown header flags set", r.mode = 30;
                    break;
                  }
                  r.head && (r.head.text = u >> 8 & 1), 512 & r.flags && (E[0] = 255 & u, E[1] = u >>> 8 & 255, r.check = B(r.check, E, 2, 0)), l = u = 0, r.mode = 3;
                case 3:
                  for (; l < 32;) {
                    if (0 === o) break e;
                    o--, u += n[s++] << l, l += 8;
                  }
                  r.head && (r.head.time = u), 512 & r.flags && (E[0] = 255 & u, E[1] = u >>> 8 & 255, E[2] = u >>> 16 & 255, E[3] = u >>> 24 & 255, r.check = B(r.check, E, 4, 0)), l = u = 0, r.mode = 4;
                case 4:
                  for (; l < 16;) {
                    if (0 === o) break e;
                    o--, u += n[s++] << l, l += 8;
                  }
                  r.head && (r.head.xflags = 255 & u, r.head.os = u >> 8), 512 & r.flags && (E[0] = 255 & u, E[1] = u >>> 8 & 255, r.check = B(r.check, E, 2, 0)), l = u = 0, r.mode = 5;
                case 5:
                  if (1024 & r.flags) {
                    for (; l < 16;) {
                      if (0 === o) break e;
                      o--, u += n[s++] << l, l += 8;
                    }
                    r.length = u, r.head && (r.head.extra_len = u), 512 & r.flags && (E[0] = 255 & u, E[1] = u >>> 8 & 255, r.check = B(r.check, E, 2, 0)), l = u = 0;
                  } else r.head && (r.head.extra = null);
                  r.mode = 6;
                case 6:
                  if (1024 & r.flags && (o < (d = r.length) && (d = o), d && (r.head && (k = r.head.extra_len - r.length, r.head.extra || (r.head.extra = new Array(r.head.extra_len)), I.arraySet(r.head.extra, n, s, d, k)), 512 & r.flags && (r.check = B(r.check, n, d, s)), o -= d, s += d, r.length -= d), r.length)) break e;
                  r.length = 0, r.mode = 7;
                case 7:
                  if (2048 & r.flags) {
                    if (0 === o) break e;
                    for (d = 0; k = n[s + d++], r.head && k && r.length < 65536 && (r.head.name += String.fromCharCode(k)), k && d < o;);
                    if (512 & r.flags && (r.check = B(r.check, n, d, s)), o -= d, s += d, k) break e;
                  } else r.head && (r.head.name = null);
                  r.length = 0, r.mode = 8;
                case 8:
                  if (4096 & r.flags) {
                    if (0 === o) break e;
                    for (d = 0; k = n[s + d++], r.head && k && r.length < 65536 && (r.head.comment += String.fromCharCode(k)), k && d < o;);
                    if (512 & r.flags && (r.check = B(r.check, n, d, s)), o -= d, s += d, k) break e;
                  } else r.head && (r.head.comment = null);
                  r.mode = 9;
                case 9:
                  if (512 & r.flags) {
                    for (; l < 16;) {
                      if (0 === o) break e;
                      o--, u += n[s++] << l, l += 8;
                    }
                    if (u !== (65535 & r.check)) {
                      e.msg = "header crc mismatch", r.mode = 30;
                      break;
                    }
                    l = u = 0;
                  }
                  r.head && (r.head.hcrc = r.flags >> 9 & 1, r.head.done = true), e.adler = r.check = 0, r.mode = 12;
                  break;
                case 10:
                  for (; l < 32;) {
                    if (0 === o) break e;
                    o--, u += n[s++] << l, l += 8;
                  }
                  e.adler = r.check = L(u), l = u = 0, r.mode = 11;
                case 11:
                  if (0 === r.havedict) return e.next_out = a, e.avail_out = h, e.next_in = s, e.avail_in = o, r.hold = u, r.bits = l, 2;
                  e.adler = r.check = 1, r.mode = 12;
                case 12:
                  if (5 === t || 6 === t) break e;
                case 13:
                  if (r.last) {
                    u >>>= 7 & l, l -= 7 & l, r.mode = 27;
                    break;
                  }
                  for (; l < 3;) {
                    if (0 === o) break e;
                    o--, u += n[s++] << l, l += 8;
                  }
                  switch (r.last = 1 & u, l -= 1, 3 & (u >>>= 1)) {
                    case 0:
                      r.mode = 14;
                      break;
                    case 1:
                      if (j(r), r.mode = 20, 6 !== t) break;
                      u >>>= 2, l -= 2;
                      break e;
                    case 2:
                      r.mode = 17;
                      break;
                    case 3:
                      e.msg = "invalid block type", r.mode = 30;
                  }
                  u >>>= 2, l -= 2;
                  break;
                case 14:
                  for (u >>>= 7 & l, l -= 7 & l; l < 32;) {
                    if (0 === o) break e;
                    o--, u += n[s++] << l, l += 8;
                  }
                  if ((65535 & u) != (u >>> 16 ^ 65535)) {
                    e.msg = "invalid stored block lengths", r.mode = 30;
                    break;
                  }
                  if (r.length = 65535 & u, l = u = 0, r.mode = 15, 6 === t) break e;
                case 15:
                  r.mode = 16;
                case 16:
                  if (d = r.length) {
                    if (o < d && (d = o), h < d && (d = h), 0 === d) break e;
                    I.arraySet(i, n, s, d, a), o -= d, s += d, h -= d, a += d, r.length -= d;
                    break;
                  }
                  r.mode = 12;
                  break;
                case 17:
                  for (; l < 14;) {
                    if (0 === o) break e;
                    o--, u += n[s++] << l, l += 8;
                  }
                  if (r.nlen = 257 + (31 & u), u >>>= 5, l -= 5, r.ndist = 1 + (31 & u), u >>>= 5, l -= 5, r.ncode = 4 + (15 & u), u >>>= 4, l -= 4, 286 < r.nlen || 30 < r.ndist) {
                    e.msg = "too many length or distance symbols", r.mode = 30;
                    break;
                  }
                  r.have = 0, r.mode = 18;
                case 18:
                  for (; r.have < r.ncode;) {
                    for (; l < 3;) {
                      if (0 === o) break e;
                      o--, u += n[s++] << l, l += 8;
                    }
                    r.lens[A[r.have++]] = 7 & u, u >>>= 3, l -= 3;
                  }
                  for (; r.have < 19;) r.lens[A[r.have++]] = 0;
                  if (r.lencode = r.lendyn, r.lenbits = 7, S = {
                    bits: r.lenbits
                  }, x = T(0, r.lens, 0, 19, r.lencode, 0, r.work, S), r.lenbits = S.bits, x) {
                    e.msg = "invalid code lengths set", r.mode = 30;
                    break;
                  }
                  r.have = 0, r.mode = 19;
                case 19:
                  for (; r.have < r.nlen + r.ndist;) {
                    for (; g = (C = r.lencode[u & (1 << r.lenbits) - 1]) >>> 16 & 255, b = 65535 & C, !((_ = C >>> 24) <= l);) {
                      if (0 === o) break e;
                      o--, u += n[s++] << l, l += 8;
                    }
                    if (b < 16) u >>>= _, l -= _, r.lens[r.have++] = b;else {
                      if (16 === b) {
                        for (z = _ + 2; l < z;) {
                          if (0 === o) break e;
                          o--, u += n[s++] << l, l += 8;
                        }
                        if (u >>>= _, l -= _, 0 === r.have) {
                          e.msg = "invalid bit length repeat", r.mode = 30;
                          break;
                        }
                        k = r.lens[r.have - 1], d = 3 + (3 & u), u >>>= 2, l -= 2;
                      } else if (17 === b) {
                        for (z = _ + 3; l < z;) {
                          if (0 === o) break e;
                          o--, u += n[s++] << l, l += 8;
                        }
                        l -= _, k = 0, d = 3 + (7 & (u >>>= _)), u >>>= 3, l -= 3;
                      } else {
                        for (z = _ + 7; l < z;) {
                          if (0 === o) break e;
                          o--, u += n[s++] << l, l += 8;
                        }
                        l -= _, k = 0, d = 11 + (127 & (u >>>= _)), u >>>= 7, l -= 7;
                      }
                      if (r.have + d > r.nlen + r.ndist) {
                        e.msg = "invalid bit length repeat", r.mode = 30;
                        break;
                      }
                      for (; d--;) r.lens[r.have++] = k;
                    }
                  }
                  if (30 === r.mode) break;
                  if (0 === r.lens[256]) {
                    e.msg = "invalid code -- missing end-of-block", r.mode = 30;
                    break;
                  }
                  if (r.lenbits = 9, S = {
                    bits: r.lenbits
                  }, x = T(D, r.lens, 0, r.nlen, r.lencode, 0, r.work, S), r.lenbits = S.bits, x) {
                    e.msg = "invalid literal/lengths set", r.mode = 30;
                    break;
                  }
                  if (r.distbits = 6, r.distcode = r.distdyn, S = {
                    bits: r.distbits
                  }, x = T(F, r.lens, r.nlen, r.ndist, r.distcode, 0, r.work, S), r.distbits = S.bits, x) {
                    e.msg = "invalid distances set", r.mode = 30;
                    break;
                  }
                  if (r.mode = 20, 6 === t) break e;
                case 20:
                  r.mode = 21;
                case 21:
                  if (6 <= o && 258 <= h) {
                    e.next_out = a, e.avail_out = h, e.next_in = s, e.avail_in = o, r.hold = u, r.bits = l, R(e, c), a = e.next_out, i = e.output, h = e.avail_out, s = e.next_in, n = e.input, o = e.avail_in, u = r.hold, l = r.bits, 12 === r.mode && (r.back = -1);
                    break;
                  }
                  for (r.back = 0; g = (C = r.lencode[u & (1 << r.lenbits) - 1]) >>> 16 & 255, b = 65535 & C, !((_ = C >>> 24) <= l);) {
                    if (0 === o) break e;
                    o--, u += n[s++] << l, l += 8;
                  }
                  if (g && 0 == (240 & g)) {
                    for (v = _, y = g, w = b; g = (C = r.lencode[w + ((u & (1 << v + y) - 1) >> v)]) >>> 16 & 255, b = 65535 & C, !(v + (_ = C >>> 24) <= l);) {
                      if (0 === o) break e;
                      o--, u += n[s++] << l, l += 8;
                    }
                    u >>>= v, l -= v, r.back += v;
                  }
                  if (u >>>= _, l -= _, r.back += _, r.length = b, 0 === g) {
                    r.mode = 26;
                    break;
                  }
                  if (32 & g) {
                    r.back = -1, r.mode = 12;
                    break;
                  }
                  if (64 & g) {
                    e.msg = "invalid literal/length code", r.mode = 30;
                    break;
                  }
                  r.extra = 15 & g, r.mode = 22;
                case 22:
                  if (r.extra) {
                    for (z = r.extra; l < z;) {
                      if (0 === o) break e;
                      o--, u += n[s++] << l, l += 8;
                    }
                    r.length += u & (1 << r.extra) - 1, u >>>= r.extra, l -= r.extra, r.back += r.extra;
                  }
                  r.was = r.length, r.mode = 23;
                case 23:
                  for (; g = (C = r.distcode[u & (1 << r.distbits) - 1]) >>> 16 & 255, b = 65535 & C, !((_ = C >>> 24) <= l);) {
                    if (0 === o) break e;
                    o--, u += n[s++] << l, l += 8;
                  }
                  if (0 == (240 & g)) {
                    for (v = _, y = g, w = b; g = (C = r.distcode[w + ((u & (1 << v + y) - 1) >> v)]) >>> 16 & 255, b = 65535 & C, !(v + (_ = C >>> 24) <= l);) {
                      if (0 === o) break e;
                      o--, u += n[s++] << l, l += 8;
                    }
                    u >>>= v, l -= v, r.back += v;
                  }
                  if (u >>>= _, l -= _, r.back += _, 64 & g) {
                    e.msg = "invalid distance code", r.mode = 30;
                    break;
                  }
                  r.offset = b, r.extra = 15 & g, r.mode = 24;
                case 24:
                  if (r.extra) {
                    for (z = r.extra; l < z;) {
                      if (0 === o) break e;
                      o--, u += n[s++] << l, l += 8;
                    }
                    r.offset += u & (1 << r.extra) - 1, u >>>= r.extra, l -= r.extra, r.back += r.extra;
                  }
                  if (r.offset > r.dmax) {
                    e.msg = "invalid distance too far back", r.mode = 30;
                    break;
                  }
                  r.mode = 25;
                case 25:
                  if (0 === h) break e;
                  if (d = c - h, r.offset > d) {
                    if ((d = r.offset - d) > r.whave && r.sane) {
                      e.msg = "invalid distance too far back", r.mode = 30;
                      break;
                    }
                    p = d > r.wnext ? (d -= r.wnext, r.wsize - d) : r.wnext - d, d > r.length && (d = r.length), m = r.window;
                  } else m = i, p = a - r.offset, d = r.length;
                  for (h < d && (d = h), h -= d, r.length -= d; i[a++] = m[p++], --d;);
                  0 === r.length && (r.mode = 21);
                  break;
                case 26:
                  if (0 === h) break e;
                  i[a++] = r.length, h--, r.mode = 21;
                  break;
                case 27:
                  if (r.wrap) {
                    for (; l < 32;) {
                      if (0 === o) break e;
                      o--, u |= n[s++] << l, l += 8;
                    }
                    if (c -= h, e.total_out += c, r.total += c, c && (e.adler = r.check = r.flags ? B(r.check, i, c, a - c) : O(r.check, i, c, a - c)), c = h, (r.flags ? u : L(u)) !== r.check) {
                      e.msg = "incorrect data check", r.mode = 30;
                      break;
                    }
                    l = u = 0;
                  }
                  r.mode = 28;
                case 28:
                  if (r.wrap && r.flags) {
                    for (; l < 32;) {
                      if (0 === o) break e;
                      o--, u += n[s++] << l, l += 8;
                    }
                    if (u !== (4294967295 & r.total)) {
                      e.msg = "incorrect length check", r.mode = 30;
                      break;
                    }
                    l = u = 0;
                  }
                  r.mode = 29;
                case 29:
                  x = 1;
                  break e;
                case 30:
                  x = -3;
                  break e;
                case 31:
                  return -4;
                case 32:
                default:
                  return U;
              }
              return e.next_out = a, e.avail_out = h, e.next_in = s, e.avail_in = o, r.hold = u, r.bits = l, (r.wsize || c !== e.avail_out && r.mode < 30 && (r.mode < 27 || 4 !== t)) && Z(e, e.output, e.next_out, c - e.avail_out) ? (r.mode = 31, -4) : (f -= e.avail_in, c -= e.avail_out, e.total_in += f, e.total_out += c, r.total += c, r.wrap && c && (e.adler = r.check = r.flags ? B(r.check, i, c, e.next_out - c) : O(r.check, i, c, e.next_out - c)), e.data_type = r.bits + (r.last ? 64 : 0) + (12 === r.mode ? 128 : 0) + (20 === r.mode || 15 === r.mode ? 256 : 0), (0 == f && 0 === c || 4 === t) && x === N && (x = -5), x);
            }, r.inflateEnd = function (e) {
              if (!e || !e.state) return U;
              var t = e.state;
              return t.window && (t.window = null), e.state = null, N;
            }, r.inflateGetHeader = function (e, t) {
              var r;
              return e && e.state ? 0 == (2 & (r = e.state).wrap) ? U : ((r.head = t).done = false, N) : U;
            }, r.inflateSetDictionary = function (e, t) {
              var r,
                n = t.length;
              return e && e.state ? 0 !== (r = e.state).wrap && 11 !== r.mode ? U : 11 === r.mode && O(1, t, n, 0) !== r.check ? -3 : Z(e, t, n, n) ? (r.mode = 31, -4) : (r.havedict = 1, N) : U;
            }, r.inflateInfo = "pako inflate (from Nodeca project)";
          }, {
            "../utils/common": 41,
            "./adler32": 43,
            "./crc32": 45,
            "./inffast": 48,
            "./inftrees": 50
          }],
          50: [function (e, t, r) {

            var D = e("../utils/common"),
              F = [3, 4, 5, 6, 7, 8, 9, 10, 11, 13, 15, 17, 19, 23, 27, 31, 35, 43, 51, 59, 67, 83, 99, 115, 131, 163, 195, 227, 258, 0, 0],
              N = [16, 16, 16, 16, 16, 16, 16, 16, 17, 17, 17, 17, 18, 18, 18, 18, 19, 19, 19, 19, 20, 20, 20, 20, 21, 21, 21, 21, 16, 72, 78],
              U = [1, 2, 3, 4, 5, 7, 9, 13, 17, 25, 33, 49, 65, 97, 129, 193, 257, 385, 513, 769, 1025, 1537, 2049, 3073, 4097, 6145, 8193, 12289, 16385, 24577, 0, 0],
              P = [16, 16, 16, 16, 17, 17, 18, 18, 19, 19, 20, 20, 21, 21, 22, 22, 23, 23, 24, 24, 25, 25, 26, 26, 27, 27, 28, 28, 29, 29, 64, 64];
            t.exports = function (e, t, r, n, i, s, a, o) {
              var h,
                u,
                l,
                f,
                c,
                d,
                p,
                m,
                _,
                g = o.bits,
                b = 0,
                v = 0,
                y = 0,
                w = 0,
                k = 0,
                x = 0,
                S = 0,
                z = 0,
                C = 0,
                E = 0,
                A = null,
                I = 0,
                O = new D.Buf16(16),
                B = new D.Buf16(16),
                R = null,
                T = 0;
              for (b = 0; b <= 15; b++) O[b] = 0;
              for (v = 0; v < n; v++) O[t[r + v]]++;
              for (k = g, w = 15; 1 <= w && 0 === O[w]; w--);
              if (w < k && (k = w), 0 === w) return i[s++] = 20971520, i[s++] = 20971520, o.bits = 1, 0;
              for (y = 1; y < w && 0 === O[y]; y++);
              for (k < y && (k = y), b = z = 1; b <= 15; b++) if (z <<= 1, (z -= O[b]) < 0) return -1;
              if (0 < z && (0 === e || 1 !== w)) return -1;
              for (B[1] = 0, b = 1; b < 15; b++) B[b + 1] = B[b] + O[b];
              for (v = 0; v < n; v++) 0 !== t[r + v] && (a[B[t[r + v]]++] = v);
              if (d = 0 === e ? (A = R = a, 19) : 1 === e ? (A = F, I -= 257, R = N, T -= 257, 256) : (A = U, R = P, -1), b = y, c = s, S = v = E = 0, l = -1, f = (C = 1 << (x = k)) - 1, 1 === e && 852 < C || 2 === e && 592 < C) return 1;
              for (;;) {
                for (p = b - S, _ = a[v] < d ? (m = 0, a[v]) : a[v] > d ? (m = R[T + a[v]], A[I + a[v]]) : (m = 96, 0), h = 1 << b - S, y = u = 1 << x; i[c + (E >> S) + (u -= h)] = p << 24 | m << 16 | _ | 0, 0 !== u;);
                for (h = 1 << b - 1; E & h;) h >>= 1;
                if (0 !== h ? (E &= h - 1, E += h) : E = 0, v++, 0 == --O[b]) {
                  if (b === w) break;
                  b = t[r + a[v]];
                }
                if (k < b && (E & f) !== l) {
                  for (0 === S && (S = k), c += y, z = 1 << (x = b - S); x + S < w && !((z -= O[x + S]) <= 0);) x++, z <<= 1;
                  if (C += 1 << x, 1 === e && 852 < C || 2 === e && 592 < C) return 1;
                  i[l = E & f] = k << 24 | x << 16 | c - s | 0;
                }
              }
              return 0 !== E && (i[c + E] = b - S << 24 | 64 << 16 | 0), o.bits = k, 0;
            };
          }, {
            "../utils/common": 41
          }],
          51: [function (e, t, r) {

            t.exports = {
              2: "need dictionary",
              1: "stream end",
              0: "",
              "-1": "file error",
              "-2": "stream error",
              "-3": "data error",
              "-4": "insufficient memory",
              "-5": "buffer error",
              "-6": "incompatible version"
            };
          }, {}],
          52: [function (e, t, r) {

            var i = e("../utils/common"),
              o = 0,
              h = 1;
            function n(e) {
              for (var t = e.length; 0 <= --t;) e[t] = 0;
            }
            var s = 0,
              a = 29,
              u = 256,
              l = u + 1 + a,
              f = 30,
              c = 19,
              _ = 2 * l + 1,
              g = 15,
              d = 16,
              p = 7,
              m = 256,
              b = 16,
              v = 17,
              y = 18,
              w = [0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 5, 5, 5, 5, 0],
              k = [0, 0, 0, 0, 1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9, 10, 10, 11, 11, 12, 12, 13, 13],
              x = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 3, 7],
              S = [16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15],
              z = new Array(2 * (l + 2));
            n(z);
            var C = new Array(2 * f);
            n(C);
            var E = new Array(512);
            n(E);
            var A = new Array(256);
            n(A);
            var I = new Array(a);
            n(I);
            var O,
              B,
              R,
              T = new Array(f);
            function D(e, t, r, n, i) {
              this.static_tree = e, this.extra_bits = t, this.extra_base = r, this.elems = n, this.max_length = i, this.has_stree = e && e.length;
            }
            function F(e, t) {
              this.dyn_tree = e, this.max_code = 0, this.stat_desc = t;
            }
            function N(e) {
              return e < 256 ? E[e] : E[256 + (e >>> 7)];
            }
            function U(e, t) {
              e.pending_buf[e.pending++] = 255 & t, e.pending_buf[e.pending++] = t >>> 8 & 255;
            }
            function P(e, t, r) {
              e.bi_valid > d - r ? (e.bi_buf |= t << e.bi_valid & 65535, U(e, e.bi_buf), e.bi_buf = t >> d - e.bi_valid, e.bi_valid += r - d) : (e.bi_buf |= t << e.bi_valid & 65535, e.bi_valid += r);
            }
            function L(e, t, r) {
              P(e, r[2 * t], r[2 * t + 1]);
            }
            function j(e, t) {
              for (var r = 0; r |= 1 & e, e >>>= 1, r <<= 1, 0 < --t;);
              return r >>> 1;
            }
            function Z(e, t, r) {
              var n,
                i,
                s = new Array(g + 1),
                a = 0;
              for (n = 1; n <= g; n++) s[n] = a = a + r[n - 1] << 1;
              for (i = 0; i <= t; i++) {
                var o = e[2 * i + 1];
                0 !== o && (e[2 * i] = j(s[o]++, o));
              }
            }
            function W(e) {
              var t;
              for (t = 0; t < l; t++) e.dyn_ltree[2 * t] = 0;
              for (t = 0; t < f; t++) e.dyn_dtree[2 * t] = 0;
              for (t = 0; t < c; t++) e.bl_tree[2 * t] = 0;
              e.dyn_ltree[2 * m] = 1, e.opt_len = e.static_len = 0, e.last_lit = e.matches = 0;
            }
            function M(e) {
              8 < e.bi_valid ? U(e, e.bi_buf) : 0 < e.bi_valid && (e.pending_buf[e.pending++] = e.bi_buf), e.bi_buf = 0, e.bi_valid = 0;
            }
            function H(e, t, r, n) {
              var i = 2 * t,
                s = 2 * r;
              return e[i] < e[s] || e[i] === e[s] && n[t] <= n[r];
            }
            function G(e, t, r) {
              for (var n = e.heap[r], i = r << 1; i <= e.heap_len && (i < e.heap_len && H(t, e.heap[i + 1], e.heap[i], e.depth) && i++, !H(t, n, e.heap[i], e.depth));) e.heap[r] = e.heap[i], r = i, i <<= 1;
              e.heap[r] = n;
            }
            function K(e, t, r) {
              var n,
                i,
                s,
                a,
                o = 0;
              if (0 !== e.last_lit) for (; n = e.pending_buf[e.d_buf + 2 * o] << 8 | e.pending_buf[e.d_buf + 2 * o + 1], i = e.pending_buf[e.l_buf + o], o++, 0 === n ? L(e, i, t) : (L(e, (s = A[i]) + u + 1, t), 0 !== (a = w[s]) && P(e, i -= I[s], a), L(e, s = N(--n), r), 0 !== (a = k[s]) && P(e, n -= T[s], a)), o < e.last_lit;);
              L(e, m, t);
            }
            function Y(e, t) {
              var r,
                n,
                i,
                s = t.dyn_tree,
                a = t.stat_desc.static_tree,
                o = t.stat_desc.has_stree,
                h = t.stat_desc.elems,
                u = -1;
              for (e.heap_len = 0, e.heap_max = _, r = 0; r < h; r++) 0 !== s[2 * r] ? (e.heap[++e.heap_len] = u = r, e.depth[r] = 0) : s[2 * r + 1] = 0;
              for (; e.heap_len < 2;) s[2 * (i = e.heap[++e.heap_len] = u < 2 ? ++u : 0)] = 1, e.depth[i] = 0, e.opt_len--, o && (e.static_len -= a[2 * i + 1]);
              for (t.max_code = u, r = e.heap_len >> 1; 1 <= r; r--) G(e, s, r);
              for (i = h; r = e.heap[1], e.heap[1] = e.heap[e.heap_len--], G(e, s, 1), n = e.heap[1], e.heap[--e.heap_max] = r, e.heap[--e.heap_max] = n, s[2 * i] = s[2 * r] + s[2 * n], e.depth[i] = (e.depth[r] >= e.depth[n] ? e.depth[r] : e.depth[n]) + 1, s[2 * r + 1] = s[2 * n + 1] = i, e.heap[1] = i++, G(e, s, 1), 2 <= e.heap_len;);
              e.heap[--e.heap_max] = e.heap[1], function (e, t) {
                var r,
                  n,
                  i,
                  s,
                  a,
                  o,
                  h = t.dyn_tree,
                  u = t.max_code,
                  l = t.stat_desc.static_tree,
                  f = t.stat_desc.has_stree,
                  c = t.stat_desc.extra_bits,
                  d = t.stat_desc.extra_base,
                  p = t.stat_desc.max_length,
                  m = 0;
                for (s = 0; s <= g; s++) e.bl_count[s] = 0;
                for (h[2 * e.heap[e.heap_max] + 1] = 0, r = e.heap_max + 1; r < _; r++) p < (s = h[2 * h[2 * (n = e.heap[r]) + 1] + 1] + 1) && (s = p, m++), h[2 * n + 1] = s, u < n || (e.bl_count[s]++, a = 0, d <= n && (a = c[n - d]), o = h[2 * n], e.opt_len += o * (s + a), f && (e.static_len += o * (l[2 * n + 1] + a)));
                if (0 !== m) {
                  do {
                    for (s = p - 1; 0 === e.bl_count[s];) s--;
                    e.bl_count[s]--, e.bl_count[s + 1] += 2, e.bl_count[p]--, m -= 2;
                  } while (0 < m);
                  for (s = p; 0 !== s; s--) for (n = e.bl_count[s]; 0 !== n;) u < (i = e.heap[--r]) || (h[2 * i + 1] !== s && (e.opt_len += (s - h[2 * i + 1]) * h[2 * i], h[2 * i + 1] = s), n--);
                }
              }(e, t), Z(s, u, e.bl_count);
            }
            function X(e, t, r) {
              var n,
                i,
                s = -1,
                a = t[1],
                o = 0,
                h = 7,
                u = 4;
              for (0 === a && (h = 138, u = 3), t[2 * (r + 1) + 1] = 65535, n = 0; n <= r; n++) i = a, a = t[2 * (n + 1) + 1], ++o < h && i === a || (o < u ? e.bl_tree[2 * i] += o : 0 !== i ? (i !== s && e.bl_tree[2 * i]++, e.bl_tree[2 * b]++) : o <= 10 ? e.bl_tree[2 * v]++ : e.bl_tree[2 * y]++, s = i, u = (o = 0) === a ? (h = 138, 3) : i === a ? (h = 6, 3) : (h = 7, 4));
            }
            function V(e, t, r) {
              var n,
                i,
                s = -1,
                a = t[1],
                o = 0,
                h = 7,
                u = 4;
              for (0 === a && (h = 138, u = 3), n = 0; n <= r; n++) if (i = a, a = t[2 * (n + 1) + 1], !(++o < h && i === a)) {
                if (o < u) for (; L(e, i, e.bl_tree), 0 != --o;);else 0 !== i ? (i !== s && (L(e, i, e.bl_tree), o--), L(e, b, e.bl_tree), P(e, o - 3, 2)) : o <= 10 ? (L(e, v, e.bl_tree), P(e, o - 3, 3)) : (L(e, y, e.bl_tree), P(e, o - 11, 7));
                s = i, u = (o = 0) === a ? (h = 138, 3) : i === a ? (h = 6, 3) : (h = 7, 4);
              }
            }
            n(T);
            var q = false;
            function J(e, t, r, n) {
              P(e, (s << 1) + (n ? 1 : 0), 3), function (e, t, r, n) {
                M(e), (U(e, r), U(e, ~r)), i.arraySet(e.pending_buf, e.window, t, r, e.pending), e.pending += r;
              }(e, t, r);
            }
            r._tr_init = function (e) {
              q || (function () {
                var e,
                  t,
                  r,
                  n,
                  i,
                  s = new Array(g + 1);
                for (n = r = 0; n < a - 1; n++) for (I[n] = r, e = 0; e < 1 << w[n]; e++) A[r++] = n;
                for (A[r - 1] = n, n = i = 0; n < 16; n++) for (T[n] = i, e = 0; e < 1 << k[n]; e++) E[i++] = n;
                for (i >>= 7; n < f; n++) for (T[n] = i << 7, e = 0; e < 1 << k[n] - 7; e++) E[256 + i++] = n;
                for (t = 0; t <= g; t++) s[t] = 0;
                for (e = 0; e <= 143;) z[2 * e + 1] = 8, e++, s[8]++;
                for (; e <= 255;) z[2 * e + 1] = 9, e++, s[9]++;
                for (; e <= 279;) z[2 * e + 1] = 7, e++, s[7]++;
                for (; e <= 287;) z[2 * e + 1] = 8, e++, s[8]++;
                for (Z(z, l + 1, s), e = 0; e < f; e++) C[2 * e + 1] = 5, C[2 * e] = j(e, 5);
                O = new D(z, w, u + 1, l, g), B = new D(C, k, 0, f, g), R = new D(new Array(0), x, 0, c, p);
              }(), q = true), e.l_desc = new F(e.dyn_ltree, O), e.d_desc = new F(e.dyn_dtree, B), e.bl_desc = new F(e.bl_tree, R), e.bi_buf = 0, e.bi_valid = 0, W(e);
            }, r._tr_stored_block = J, r._tr_flush_block = function (e, t, r, n) {
              var i,
                s,
                a = 0;
              0 < e.level ? (2 === e.strm.data_type && (e.strm.data_type = function (e) {
                var t,
                  r = 4093624447;
                for (t = 0; t <= 31; t++, r >>>= 1) if (1 & r && 0 !== e.dyn_ltree[2 * t]) return o;
                if (0 !== e.dyn_ltree[18] || 0 !== e.dyn_ltree[20] || 0 !== e.dyn_ltree[26]) return h;
                for (t = 32; t < u; t++) if (0 !== e.dyn_ltree[2 * t]) return h;
                return o;
              }(e)), Y(e, e.l_desc), Y(e, e.d_desc), a = function (e) {
                var t;
                for (X(e, e.dyn_ltree, e.l_desc.max_code), X(e, e.dyn_dtree, e.d_desc.max_code), Y(e, e.bl_desc), t = c - 1; 3 <= t && 0 === e.bl_tree[2 * S[t] + 1]; t--);
                return e.opt_len += 3 * (t + 1) + 5 + 5 + 4, t;
              }(e), i = e.opt_len + 3 + 7 >>> 3, (s = e.static_len + 3 + 7 >>> 3) <= i && (i = s)) : i = s = r + 5, r + 4 <= i && -1 !== t ? J(e, t, r, n) : 4 === e.strategy || s === i ? (P(e, 2 + (n ? 1 : 0), 3), K(e, z, C)) : (P(e, 4 + (n ? 1 : 0), 3), function (e, t, r, n) {
                var i;
                for (P(e, t - 257, 5), P(e, r - 1, 5), P(e, n - 4, 4), i = 0; i < n; i++) P(e, e.bl_tree[2 * S[i] + 1], 3);
                V(e, e.dyn_ltree, t - 1), V(e, e.dyn_dtree, r - 1);
              }(e, e.l_desc.max_code + 1, e.d_desc.max_code + 1, a + 1), K(e, e.dyn_ltree, e.dyn_dtree)), W(e), n && M(e);
            }, r._tr_tally = function (e, t, r) {
              return e.pending_buf[e.d_buf + 2 * e.last_lit] = t >>> 8 & 255, e.pending_buf[e.d_buf + 2 * e.last_lit + 1] = 255 & t, e.pending_buf[e.l_buf + e.last_lit] = 255 & r, e.last_lit++, 0 === t ? e.dyn_ltree[2 * r]++ : (e.matches++, t--, e.dyn_ltree[2 * (A[r] + u + 1)]++, e.dyn_dtree[2 * N(t)]++), e.last_lit === e.lit_bufsize - 1;
            }, r._tr_align = function (e) {
              P(e, 2, 3), L(e, m, z), function (e) {
                16 === e.bi_valid ? (U(e, e.bi_buf), e.bi_buf = 0, e.bi_valid = 0) : 8 <= e.bi_valid && (e.pending_buf[e.pending++] = 255 & e.bi_buf, e.bi_buf >>= 8, e.bi_valid -= 8);
              }(e);
            };
          }, {
            "../utils/common": 41
          }],
          53: [function (e, t, r) {

            t.exports = function () {
              this.input = null, this.next_in = 0, this.avail_in = 0, this.total_in = 0, this.output = null, this.next_out = 0, this.avail_out = 0, this.total_out = 0, this.msg = "", this.state = null, this.data_type = 2, this.adler = 0;
            };
          }, {}],
          54: [function (e, t, r) {
            (function (e) {
              !function (r, n) {

                if (!r.setImmediate) {
                  var i,
                    s,
                    t,
                    a,
                    o = 1,
                    h = {},
                    u = false,
                    l = r.document,
                    e = Object.getPrototypeOf && Object.getPrototypeOf(r);
                  e = e && e.setTimeout ? e : r, i = "[object process]" === {}.toString.call(r.process) ? function (e) {
                    process.nextTick(function () {
                      c(e);
                    });
                  } : function () {
                    if (r.postMessage && !r.importScripts) {
                      var e = true,
                        t = r.onmessage;
                      return r.onmessage = function () {
                        e = false;
                      }, r.postMessage("", "*"), r.onmessage = t, e;
                    }
                  }() ? (a = "setImmediate$" + Math.random() + "$", r.addEventListener ? r.addEventListener("message", d, false) : r.attachEvent("onmessage", d), function (e) {
                    r.postMessage(a + e, "*");
                  }) : r.MessageChannel ? ((t = new MessageChannel()).port1.onmessage = function (e) {
                    c(e.data);
                  }, function (e) {
                    t.port2.postMessage(e);
                  }) : l && "onreadystatechange" in l.createElement("script") ? (s = l.documentElement, function (e) {
                    var t = l.createElement("script");
                    t.onreadystatechange = function () {
                      c(e), t.onreadystatechange = null, s.removeChild(t), t = null;
                    }, s.appendChild(t);
                  }) : function (e) {
                    setTimeout(c, 0, e);
                  }, e.setImmediate = function (e) {
                    "function" != typeof e && (e = new Function("" + e));
                    for (var t = new Array(arguments.length - 1), r = 0; r < t.length; r++) t[r] = arguments[r + 1];
                    var n = {
                      callback: e,
                      args: t
                    };
                    return h[o] = n, i(o), o++;
                  }, e.clearImmediate = f;
                }
                function f(e) {
                  delete h[e];
                }
                function c(e) {
                  if (u) setTimeout(c, 0, e);else {
                    var t = h[e];
                    if (t) {
                      u = true;
                      try {
                        !function (e) {
                          var t = e.callback,
                            r = e.args;
                          switch (r.length) {
                            case 0:
                              t();
                              break;
                            case 1:
                              t(r[0]);
                              break;
                            case 2:
                              t(r[0], r[1]);
                              break;
                            case 3:
                              t(r[0], r[1], r[2]);
                              break;
                            default:
                              t.apply(n, r);
                          }
                        }(t);
                      } finally {
                        f(e), u = false;
                      }
                    }
                  }
                }
                function d(e) {
                  e.source === r && "string" == typeof e.data && 0 === e.data.indexOf(a) && c(+e.data.slice(a.length));
                }
              }("undefined" == typeof self ? void 0 === e ? this : e : self);
            }).call(this, "undefined" != typeof commonjsGlobal ? commonjsGlobal : "undefined" != typeof self ? self : "undefined" != typeof window ? window : {});
          }, {}]
        }, {}, [10])(10);
      });
    })(jszip_min);
    var jszip_minExports = jszip_min.exports;
    var JSZip = /*@__PURE__*/getDefaultExportFromCjs(jszip_minExports);

    async function fromIum(zipContent) {
      const jsZip = new JSZip();
      const zip = await jsZip.loadAsync(zipContent);
      if (!zip.files['/index.json']) {
        throw new Error('Invalid IUM file: missing index.json');
      }
      const index = JSON.parse(await zip.files['/index.json'].async('text'));
      const fileCollection = new FileCollection(index.options);
      const promises = [];
      for (const source of index.sources) {
        const url = new URL(source.relativePath, source.baseURL);
        if (url.protocol === 'ium:') {
          const zipEntry = zip.files[`/data${url.pathname}`];
          if (!zipEntry) {
            throw new Error(`Invalid IUM file: missing ${url.pathname}`);
          }
          promises.push(fileCollection.appendArrayBuffer(url.pathname, zipEntry.async('arraybuffer')));
        } else {
          promises.push(fileCollection.appendExtendedSource(sourceItemToExtendedSourceItem(source, undefined)));
          // should come from the web
        }
      }
      await Promise.all(promises);
      return fileCollection;
    }

    /**
     * This method will zip a file collection and return the zip as an ArrayBuffer
     * @param fileCollecrtion
     * @param fileCollection
     * @param options
     * @returns
     */
    async function toIum(fileCollection, options = {}) {
      const jsZip = new JSZip();
      const {
        includeData = true
      } = options;
      const sources = [];
      const promises = [];
      for (const source of fileCollection.sources) {
        const newSource = {
          relativePath: source.relativePath,
          baseURL: source.baseURL,
          name: source.name,
          lastModified: source.lastModified,
          size: source.size
        };
        sources.push(newSource);
        if (includeData || source.baseURL === 'ium:/') {
          newSource.baseURL = 'ium:/';
          const url = new URL(`data/${source.relativePath}`, newSource.baseURL);
          promises.push((async () => {
            jsZip.file(url.pathname, await source.arrayBuffer());
          })());
        }
      }
      await Promise.all(promises);
      const index = {
        options: fileCollection.options,
        sources
      };
      const url = new URL('index.json', 'ium:/');
      jsZip.file(url.pathname, JSON.stringify(index, null, 2));
      return jsZip.generateAsync({
        type: 'uint8array'
      });
    }

    function convertExtendedSourceToFile(source) {
      const stream = source.stream ? () => source.stream?.() : () => {
        return new ReadableStream({
          start: controller => {
            void source.arrayBuffer().then(arrayBuffer => {
              controller.enqueue(arrayBuffer);
              controller.close();
            });
          }
        });
      };
      return {
        sourceUUID: source.uuid,
        relativePath: source.relativePath,
        name: source.name,
        lastModified: source.lastModified,
        size: source.size,
        baseURL: source.baseURL,
        arrayBuffer: () => source.arrayBuffer(),
        text: () => source.text(),
        stream
      };
    }

    /*! pako 2.1.0 https://github.com/nodeca/pako @license (MIT AND Zlib) */
    // (C) 1995-2013 Jean-loup Gailly and Mark Adler
    // (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
    //
    // This software is provided 'as-is', without any express or implied
    // warranty. In no event will the authors be held liable for any damages
    // arising from the use of this software.
    //
    // Permission is granted to anyone to use this software for any purpose,
    // including commercial applications, and to alter it and redistribute it
    // freely, subject to the following restrictions:
    //
    // 1. The origin of this software must not be misrepresented; you must not
    //   claim that you wrote the original software. If you use this software
    //   in a product, an acknowledgment in the product documentation would be
    //   appreciated but is not required.
    // 2. Altered source versions must be plainly marked as such, and must not be
    //   misrepresented as being the original software.
    // 3. This notice may not be removed or altered from any source distribution.


    /*============================================================================*/

    function zero$1(buf) {
      let len = buf.length;
      while (--len >= 0) {
        buf[len] = 0;
      }
    }
    /* The three kinds of block type */

    const MIN_MATCH$1 = 3;
    const MAX_MATCH$1 = 258;
    /* The minimum and maximum match lengths */

    // From deflate.h
    /* ===========================================================================
     * Internal compression state.
     */

    const LENGTH_CODES$1 = 29;
    /* number of length codes, not counting the special END_BLOCK code */

    const LITERALS$1 = 256;
    /* number of literal bytes 0..255 */

    const L_CODES$1 = LITERALS$1 + 1 + LENGTH_CODES$1;
    /* number of Literal or Length codes, including the END_BLOCK code */

    const D_CODES$1 = 30;
    /* eslint-enable comma-spacing,array-bracket-spacing */

    /* The lengths of the bit length codes are sent in order of decreasing
     * probability, to avoid transmitting the lengths for unused bit length codes.
     */

    /* ===========================================================================
     * Local data. These are initialized only once.
     */

    // We pre-fill arrays with 0 to avoid uninitialized gaps

    const DIST_CODE_LEN = 512; /* see definition of array dist_code below */

    // !!!! Use flat array instead of structure, Freq = i*2, Len = i*2+1
    const static_ltree = new Array((L_CODES$1 + 2) * 2);
    zero$1(static_ltree);
    /* The static literal tree. Since the bit lengths are imposed, there is no
     * need for the L_CODES extra codes used during heap construction. However
     * The codes 286 and 287 are needed to build a canonical tree (see _tr_init
     * below).
     */

    const static_dtree = new Array(D_CODES$1 * 2);
    zero$1(static_dtree);
    /* The static distance tree. (Actually a trivial tree since all codes use
     * 5 bits.)
     */

    const _dist_code = new Array(DIST_CODE_LEN);
    zero$1(_dist_code);
    /* Distance codes. The first 256 values correspond to the distances
     * 3 .. 258, the last 256 values correspond to the top 8 bits of
     * the 15 bit distances.
     */

    const _length_code = new Array(MAX_MATCH$1 - MIN_MATCH$1 + 1);
    zero$1(_length_code);
    /* length code for each normalized match length (0 == MIN_MATCH) */

    const base_length = new Array(LENGTH_CODES$1);
    zero$1(base_length);
    /* First normalized length for each code (0 = MIN_MATCH) */

    const base_dist = new Array(D_CODES$1);
    zero$1(base_dist);

    // Note: adler32 takes 12% for level 0 and 2% for level 6.
    // It isn't worth it to make additional optimizations as in original.
    // Small size is preferable.

    // (C) 1995-2013 Jean-loup Gailly and Mark Adler
    // (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
    //
    // This software is provided 'as-is', without any express or implied
    // warranty. In no event will the authors be held liable for any damages
    // arising from the use of this software.
    //
    // Permission is granted to anyone to use this software for any purpose,
    // including commercial applications, and to alter it and redistribute it
    // freely, subject to the following restrictions:
    //
    // 1. The origin of this software must not be misrepresented; you must not
    //   claim that you wrote the original software. If you use this software
    //   in a product, an acknowledgment in the product documentation would be
    //   appreciated but is not required.
    // 2. Altered source versions must be plainly marked as such, and must not be
    //   misrepresented as being the original software.
    // 3. This notice may not be removed or altered from any source distribution.

    const adler32 = (adler, buf, len, pos) => {
      let s1 = adler & 0xffff | 0,
        s2 = adler >>> 16 & 0xffff | 0,
        n = 0;
      while (len !== 0) {
        // Set limit ~ twice less than 5552, to keep
        // s2 in 31-bits, because we force signed ints.
        // in other case %= will fail.
        n = len > 2000 ? 2000 : len;
        len -= n;
        do {
          s1 = s1 + buf[pos++] | 0;
          s2 = s2 + s1 | 0;
        } while (--n);
        s1 %= 65521;
        s2 %= 65521;
      }
      return s1 | s2 << 16 | 0;
    };
    var adler32_1 = adler32;

    // Note: we can't get significant speed boost here.
    // So write code to minimize size - no pregenerated tables
    // and array tools dependencies.

    // (C) 1995-2013 Jean-loup Gailly and Mark Adler
    // (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
    //
    // This software is provided 'as-is', without any express or implied
    // warranty. In no event will the authors be held liable for any damages
    // arising from the use of this software.
    //
    // Permission is granted to anyone to use this software for any purpose,
    // including commercial applications, and to alter it and redistribute it
    // freely, subject to the following restrictions:
    //
    // 1. The origin of this software must not be misrepresented; you must not
    //   claim that you wrote the original software. If you use this software
    //   in a product, an acknowledgment in the product documentation would be
    //   appreciated but is not required.
    // 2. Altered source versions must be plainly marked as such, and must not be
    //   misrepresented as being the original software.
    // 3. This notice may not be removed or altered from any source distribution.

    // Use ordinary array, since untyped makes no boost here
    const makeTable = () => {
      let c,
        table = [];
      for (var n = 0; n < 256; n++) {
        c = n;
        for (var k = 0; k < 8; k++) {
          c = c & 1 ? 0xEDB88320 ^ c >>> 1 : c >>> 1;
        }
        table[n] = c;
      }
      return table;
    };

    // Create table on load. Just 255 signed longs. Not a problem.
    const crcTable = new Uint32Array(makeTable());
    const crc32 = (crc, buf, len, pos) => {
      const t = crcTable;
      const end = pos + len;
      crc ^= -1;
      for (let i = pos; i < end; i++) {
        crc = crc >>> 8 ^ t[(crc ^ buf[i]) & 0xFF];
      }
      return crc ^ -1; // >>> 0;
    };
    var crc32_1 = crc32;

    // (C) 1995-2013 Jean-loup Gailly and Mark Adler
    // (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
    //
    // This software is provided 'as-is', without any express or implied
    // warranty. In no event will the authors be held liable for any damages
    // arising from the use of this software.
    //
    // Permission is granted to anyone to use this software for any purpose,
    // including commercial applications, and to alter it and redistribute it
    // freely, subject to the following restrictions:
    //
    // 1. The origin of this software must not be misrepresented; you must not
    //   claim that you wrote the original software. If you use this software
    //   in a product, an acknowledgment in the product documentation would be
    //   appreciated but is not required.
    // 2. Altered source versions must be plainly marked as such, and must not be
    //   misrepresented as being the original software.
    // 3. This notice may not be removed or altered from any source distribution.

    var messages = {
      2: 'need dictionary',
      /* Z_NEED_DICT       2  */
      1: 'stream end',
      /* Z_STREAM_END      1  */
      0: '',
      /* Z_OK              0  */
      '-1': 'file error',
      /* Z_ERRNO         (-1) */
      '-2': 'stream error',
      /* Z_STREAM_ERROR  (-2) */
      '-3': 'data error',
      /* Z_DATA_ERROR    (-3) */
      '-4': 'insufficient memory',
      /* Z_MEM_ERROR     (-4) */
      '-5': 'buffer error',
      /* Z_BUF_ERROR     (-5) */
      '-6': 'incompatible version' /* Z_VERSION_ERROR (-6) */
    };

    // (C) 1995-2013 Jean-loup Gailly and Mark Adler
    // (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
    //
    // This software is provided 'as-is', without any express or implied
    // warranty. In no event will the authors be held liable for any damages
    // arising from the use of this software.
    //
    // Permission is granted to anyone to use this software for any purpose,
    // including commercial applications, and to alter it and redistribute it
    // freely, subject to the following restrictions:
    //
    // 1. The origin of this software must not be misrepresented; you must not
    //   claim that you wrote the original software. If you use this software
    //   in a product, an acknowledgment in the product documentation would be
    //   appreciated but is not required.
    // 2. Altered source versions must be plainly marked as such, and must not be
    //   misrepresented as being the original software.
    // 3. This notice may not be removed or altered from any source distribution.

    var constants$2 = {
      /* Allowed flush values; see deflate() and inflate() below for details */
      Z_NO_FLUSH: 0,
      Z_FINISH: 4,
      Z_BLOCK: 5,
      Z_TREES: 6,
      /* Return codes for the compression/decompression functions. Negative values
      * are errors, positive values are used for special but normal events.
      */
      Z_OK: 0,
      Z_STREAM_END: 1,
      Z_NEED_DICT: 2,
      Z_STREAM_ERROR: -2,
      Z_DATA_ERROR: -3,
      Z_MEM_ERROR: -4,
      Z_BUF_ERROR: -5,
      /* The deflate compression method */
      Z_DEFLATED: 8
      //Z_NULL:                 null // Use -1 or null inline, depending on var type
    };
    const _has = (obj, key) => {
      return Object.prototype.hasOwnProperty.call(obj, key);
    };
    var assign = function (obj /*from1, from2, from3, ...*/) {
      const sources = Array.prototype.slice.call(arguments, 1);
      while (sources.length) {
        const source = sources.shift();
        if (!source) {
          continue;
        }
        if (typeof source !== 'object') {
          throw new TypeError(source + 'must be non-object');
        }
        for (const p in source) {
          if (_has(source, p)) {
            obj[p] = source[p];
          }
        }
      }
      return obj;
    };

    // Join array of chunks to single array.
    var flattenChunks = chunks => {
      // calculate data length
      let len = 0;
      for (let i = 0, l = chunks.length; i < l; i++) {
        len += chunks[i].length;
      }

      // join chunks
      const result = new Uint8Array(len);
      for (let i = 0, pos = 0, l = chunks.length; i < l; i++) {
        let chunk = chunks[i];
        result.set(chunk, pos);
        pos += chunk.length;
      }
      return result;
    };
    var common = {
      assign: assign,
      flattenChunks: flattenChunks
    };

    // String encode/decode helpers

    // Quick check if we can use fast array to bin string conversion
    //
    // - apply(Array) can fail on Android 2.2
    // - apply(Uint8Array) can fail on iOS 5.1 Safari
    //
    let STR_APPLY_UIA_OK = true;
    try {
      String.fromCharCode.apply(null, new Uint8Array(1));
    } catch (__) {
      STR_APPLY_UIA_OK = false;
    }

    // Table with utf8 lengths (calculated by first byte of sequence)
    // Note, that 5 & 6-byte values and some 4-byte values can not be represented in JS,
    // because max possible codepoint is 0x10ffff
    const _utf8len = new Uint8Array(256);
    for (let q = 0; q < 256; q++) {
      _utf8len[q] = q >= 252 ? 6 : q >= 248 ? 5 : q >= 240 ? 4 : q >= 224 ? 3 : q >= 192 ? 2 : 1;
    }
    _utf8len[254] = _utf8len[254] = 1; // Invalid sequence start

    // convert string to array (typed, when possible)
    var string2buf = str => {
      if (typeof TextEncoder === 'function' && TextEncoder.prototype.encode) {
        return new TextEncoder().encode(str);
      }
      let buf,
        c,
        c2,
        m_pos,
        i,
        str_len = str.length,
        buf_len = 0;

      // count binary size
      for (m_pos = 0; m_pos < str_len; m_pos++) {
        c = str.charCodeAt(m_pos);
        if ((c & 0xfc00) === 0xd800 && m_pos + 1 < str_len) {
          c2 = str.charCodeAt(m_pos + 1);
          if ((c2 & 0xfc00) === 0xdc00) {
            c = 0x10000 + (c - 0xd800 << 10) + (c2 - 0xdc00);
            m_pos++;
          }
        }
        buf_len += c < 0x80 ? 1 : c < 0x800 ? 2 : c < 0x10000 ? 3 : 4;
      }

      // allocate buffer
      buf = new Uint8Array(buf_len);

      // convert
      for (i = 0, m_pos = 0; i < buf_len; m_pos++) {
        c = str.charCodeAt(m_pos);
        if ((c & 0xfc00) === 0xd800 && m_pos + 1 < str_len) {
          c2 = str.charCodeAt(m_pos + 1);
          if ((c2 & 0xfc00) === 0xdc00) {
            c = 0x10000 + (c - 0xd800 << 10) + (c2 - 0xdc00);
            m_pos++;
          }
        }
        if (c < 0x80) {
          /* one byte */
          buf[i++] = c;
        } else if (c < 0x800) {
          /* two bytes */
          buf[i++] = 0xC0 | c >>> 6;
          buf[i++] = 0x80 | c & 0x3f;
        } else if (c < 0x10000) {
          /* three bytes */
          buf[i++] = 0xE0 | c >>> 12;
          buf[i++] = 0x80 | c >>> 6 & 0x3f;
          buf[i++] = 0x80 | c & 0x3f;
        } else {
          /* four bytes */
          buf[i++] = 0xf0 | c >>> 18;
          buf[i++] = 0x80 | c >>> 12 & 0x3f;
          buf[i++] = 0x80 | c >>> 6 & 0x3f;
          buf[i++] = 0x80 | c & 0x3f;
        }
      }
      return buf;
    };

    // Helper
    const buf2binstring = (buf, len) => {
      // On Chrome, the arguments in a function call that are allowed is `65534`.
      // If the length of the buffer is smaller than that, we can use this optimization,
      // otherwise we will take a slower path.
      if (len < 65534) {
        if (buf.subarray && STR_APPLY_UIA_OK) {
          return String.fromCharCode.apply(null, buf.length === len ? buf : buf.subarray(0, len));
        }
      }
      let result = '';
      for (let i = 0; i < len; i++) {
        result += String.fromCharCode(buf[i]);
      }
      return result;
    };

    // convert array to string
    var buf2string = (buf, max) => {
      const len = max || buf.length;
      if (typeof TextDecoder === 'function' && TextDecoder.prototype.decode) {
        return new TextDecoder().decode(buf.subarray(0, max));
      }
      let i, out;

      // Reserve max possible length (2 words per char)
      // NB: by unknown reasons, Array is significantly faster for
      //     String.fromCharCode.apply than Uint16Array.
      const utf16buf = new Array(len * 2);
      for (out = 0, i = 0; i < len;) {
        let c = buf[i++];
        // quick process ascii
        if (c < 0x80) {
          utf16buf[out++] = c;
          continue;
        }
        let c_len = _utf8len[c];
        // skip 5 & 6 byte codes
        if (c_len > 4) {
          utf16buf[out++] = 0xfffd;
          i += c_len - 1;
          continue;
        }

        // apply mask on first byte
        c &= c_len === 2 ? 0x1f : c_len === 3 ? 0x0f : 0x07;
        // join the rest
        while (c_len > 1 && i < len) {
          c = c << 6 | buf[i++] & 0x3f;
          c_len--;
        }

        // terminated by end of string?
        if (c_len > 1) {
          utf16buf[out++] = 0xfffd;
          continue;
        }
        if (c < 0x10000) {
          utf16buf[out++] = c;
        } else {
          c -= 0x10000;
          utf16buf[out++] = 0xd800 | c >> 10 & 0x3ff;
          utf16buf[out++] = 0xdc00 | c & 0x3ff;
        }
      }
      return buf2binstring(utf16buf, out);
    };

    // Calculate max possible position in utf8 buffer,
    // that will not break sequence. If that's not possible
    // - (very small limits) return max size as is.
    //
    // buf[] - utf8 bytes array
    // max   - length limit (mandatory);
    var utf8border = (buf, max) => {
      max = max || buf.length;
      if (max > buf.length) {
        max = buf.length;
      }

      // go back from last position, until start of sequence found
      let pos = max - 1;
      while (pos >= 0 && (buf[pos] & 0xC0) === 0x80) {
        pos--;
      }

      // Very small and broken sequence,
      // return max, because we should return something anyway.
      if (pos < 0) {
        return max;
      }

      // If we came to start of buffer - that means buffer is too small,
      // return max too.
      if (pos === 0) {
        return max;
      }
      return pos + _utf8len[buf[pos]] > max ? pos : max;
    };
    var strings = {
      string2buf: string2buf,
      buf2string: buf2string,
      utf8border: utf8border
    };

    // (C) 1995-2013 Jean-loup Gailly and Mark Adler
    // (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
    //
    // This software is provided 'as-is', without any express or implied
    // warranty. In no event will the authors be held liable for any damages
    // arising from the use of this software.
    //
    // Permission is granted to anyone to use this software for any purpose,
    // including commercial applications, and to alter it and redistribute it
    // freely, subject to the following restrictions:
    //
    // 1. The origin of this software must not be misrepresented; you must not
    //   claim that you wrote the original software. If you use this software
    //   in a product, an acknowledgment in the product documentation would be
    //   appreciated but is not required.
    // 2. Altered source versions must be plainly marked as such, and must not be
    //   misrepresented as being the original software.
    // 3. This notice may not be removed or altered from any source distribution.

    function ZStream() {
      /* next input byte */
      this.input = null; // JS specific, because we have no pointers
      this.next_in = 0;
      /* number of bytes available at input */
      this.avail_in = 0;
      /* total number of input bytes read so far */
      this.total_in = 0;
      /* next output byte should be put there */
      this.output = null; // JS specific, because we have no pointers
      this.next_out = 0;
      /* remaining free space at output */
      this.avail_out = 0;
      /* total number of bytes output so far */
      this.total_out = 0;
      /* last error message, NULL if no error */
      this.msg = '' /*Z_NULL*/;
      /* not visible by applications */
      this.state = null;
      /* best guess about the data type: binary or text */
      this.data_type = 2 /*Z_UNKNOWN*/;
      /* adler32 value of the uncompressed data */
      this.adler = 0;
    }
    var zstream = ZStream;

    // (C) 1995-2013 Jean-loup Gailly and Mark Adler
    // (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
    //
    // This software is provided 'as-is', without any express or implied
    // warranty. In no event will the authors be held liable for any damages
    // arising from the use of this software.
    //
    // Permission is granted to anyone to use this software for any purpose,
    // including commercial applications, and to alter it and redistribute it
    // freely, subject to the following restrictions:
    //
    // 1. The origin of this software must not be misrepresented; you must not
    //   claim that you wrote the original software. If you use this software
    //   in a product, an acknowledgment in the product documentation would be
    //   appreciated but is not required.
    // 2. Altered source versions must be plainly marked as such, and must not be
    //   misrepresented as being the original software.
    // 3. This notice may not be removed or altered from any source distribution.

    // See state defs from inflate.js
    const BAD$1 = 16209; /* got a data error -- remain here until reset */
    const TYPE$1 = 16191; /* i: waiting for type bits, including last-flag bit */

    /*
       Decode literal, length, and distance codes and write out the resulting
       literal and match bytes until either not enough input or output is
       available, an end-of-block is encountered, or a data error is encountered.
       When large enough input and output buffers are supplied to inflate(), for
       example, a 16K input buffer and a 64K output buffer, more than 95% of the
       inflate execution time is spent in this routine.

       Entry assumptions:

            state.mode === LEN
            strm.avail_in >= 6
            strm.avail_out >= 258
            start >= strm.avail_out
            state.bits < 8

       On return, state.mode is one of:

            LEN -- ran out of enough output space or enough available input
            TYPE -- reached end of block code, inflate() to interpret next block
            BAD -- error in block data

       Notes:

        - The maximum input bits used by a length/distance pair is 15 bits for the
          length code, 5 bits for the length extra, 15 bits for the distance code,
          and 13 bits for the distance extra.  This totals 48 bits, or six bytes.
          Therefore if strm.avail_in >= 6, then there is enough input to avoid
          checking for available input while decoding.

        - The maximum bytes that a single length/distance pair can output is 258
          bytes, which is the maximum length that can be coded.  inflate_fast()
          requires strm.avail_out >= 258 for each loop to avoid checking for
          output space.
     */
    var inffast = function inflate_fast(strm, start) {
      let _in; /* local strm.input */
      let last; /* have enough input while in < last */
      let _out; /* local strm.output */
      let beg; /* inflate()'s initial strm.output */
      let end; /* while out < end, enough space available */
      //#ifdef INFLATE_STRICT
      let dmax; /* maximum distance from zlib header */
      //#endif
      let wsize; /* window size or zero if not using window */
      let whave; /* valid bytes in the window */
      let wnext; /* window write index */
      // Use `s_window` instead `window`, avoid conflict with instrumentation tools
      let s_window; /* allocated sliding window, if wsize != 0 */
      let hold; /* local strm.hold */
      let bits; /* local strm.bits */
      let lcode; /* local strm.lencode */
      let dcode; /* local strm.distcode */
      let lmask; /* mask for first level of length codes */
      let dmask; /* mask for first level of distance codes */
      let here; /* retrieved table entry */
      let op; /* code bits, operation, extra bits, or */
      /*  window position, window bytes to copy */
      let len; /* match length, unused bytes */
      let dist; /* match distance */
      let from; /* where to copy match from */
      let from_source;
      let input, output; // JS specific, because we have no pointers

      /* copy state to local variables */
      const state = strm.state;
      //here = state.here;
      _in = strm.next_in;
      input = strm.input;
      last = _in + (strm.avail_in - 5);
      _out = strm.next_out;
      output = strm.output;
      beg = _out - (start - strm.avail_out);
      end = _out + (strm.avail_out - 257);
      //#ifdef INFLATE_STRICT
      dmax = state.dmax;
      //#endif
      wsize = state.wsize;
      whave = state.whave;
      wnext = state.wnext;
      s_window = state.window;
      hold = state.hold;
      bits = state.bits;
      lcode = state.lencode;
      dcode = state.distcode;
      lmask = (1 << state.lenbits) - 1;
      dmask = (1 << state.distbits) - 1;

      /* decode literals and length/distances until end-of-block or not enough
         input data or output space */

      top: do {
        if (bits < 15) {
          hold += input[_in++] << bits;
          bits += 8;
          hold += input[_in++] << bits;
          bits += 8;
        }
        here = lcode[hold & lmask];
        dolen: for (;;) {
          // Goto emulation
          op = here >>> 24 /*here.bits*/;
          hold >>>= op;
          bits -= op;
          op = here >>> 16 & 0xff /*here.op*/;
          if (op === 0) {
            /* literal */
            //Tracevv((stderr, here.val >= 0x20 && here.val < 0x7f ?
            //        "inflate:         literal '%c'\n" :
            //        "inflate:         literal 0x%02x\n", here.val));
            output[_out++] = here & 0xffff /*here.val*/;
          } else if (op & 16) {
            /* length base */
            len = here & 0xffff /*here.val*/;
            op &= 15; /* number of extra bits */
            if (op) {
              if (bits < op) {
                hold += input[_in++] << bits;
                bits += 8;
              }
              len += hold & (1 << op) - 1;
              hold >>>= op;
              bits -= op;
            }
            //Tracevv((stderr, "inflate:         length %u\n", len));
            if (bits < 15) {
              hold += input[_in++] << bits;
              bits += 8;
              hold += input[_in++] << bits;
              bits += 8;
            }
            here = dcode[hold & dmask];
            dodist: for (;;) {
              // goto emulation
              op = here >>> 24 /*here.bits*/;
              hold >>>= op;
              bits -= op;
              op = here >>> 16 & 0xff /*here.op*/;
              if (op & 16) {
                /* distance base */
                dist = here & 0xffff /*here.val*/;
                op &= 15; /* number of extra bits */
                if (bits < op) {
                  hold += input[_in++] << bits;
                  bits += 8;
                  if (bits < op) {
                    hold += input[_in++] << bits;
                    bits += 8;
                  }
                }
                dist += hold & (1 << op) - 1;
                //#ifdef INFLATE_STRICT
                if (dist > dmax) {
                  strm.msg = 'invalid distance too far back';
                  state.mode = BAD$1;
                  break top;
                }
                //#endif
                hold >>>= op;
                bits -= op;
                //Tracevv((stderr, "inflate:         distance %u\n", dist));
                op = _out - beg; /* max distance in output */
                if (dist > op) {
                  /* see if copy from window */
                  op = dist - op; /* distance back in window */
                  if (op > whave) {
                    if (state.sane) {
                      strm.msg = 'invalid distance too far back';
                      state.mode = BAD$1;
                      break top;
                    }

                    // (!) This block is disabled in zlib defaults,
                    // don't enable it for binary compatibility
                    //#ifdef INFLATE_ALLOW_INVALID_DISTANCE_TOOFAR_ARRR
                    //                if (len <= op - whave) {
                    //                  do {
                    //                    output[_out++] = 0;
                    //                  } while (--len);
                    //                  continue top;
                    //                }
                    //                len -= op - whave;
                    //                do {
                    //                  output[_out++] = 0;
                    //                } while (--op > whave);
                    //                if (op === 0) {
                    //                  from = _out - dist;
                    //                  do {
                    //                    output[_out++] = output[from++];
                    //                  } while (--len);
                    //                  continue top;
                    //                }
                    //#endif
                  }
                  from = 0; // window index
                  from_source = s_window;
                  if (wnext === 0) {
                    /* very common case */
                    from += wsize - op;
                    if (op < len) {
                      /* some from window */
                      len -= op;
                      do {
                        output[_out++] = s_window[from++];
                      } while (--op);
                      from = _out - dist; /* rest from output */
                      from_source = output;
                    }
                  } else if (wnext < op) {
                    /* wrap around window */
                    from += wsize + wnext - op;
                    op -= wnext;
                    if (op < len) {
                      /* some from end of window */
                      len -= op;
                      do {
                        output[_out++] = s_window[from++];
                      } while (--op);
                      from = 0;
                      if (wnext < len) {
                        /* some from start of window */
                        op = wnext;
                        len -= op;
                        do {
                          output[_out++] = s_window[from++];
                        } while (--op);
                        from = _out - dist; /* rest from output */
                        from_source = output;
                      }
                    }
                  } else {
                    /* contiguous in window */
                    from += wnext - op;
                    if (op < len) {
                      /* some from window */
                      len -= op;
                      do {
                        output[_out++] = s_window[from++];
                      } while (--op);
                      from = _out - dist; /* rest from output */
                      from_source = output;
                    }
                  }
                  while (len > 2) {
                    output[_out++] = from_source[from++];
                    output[_out++] = from_source[from++];
                    output[_out++] = from_source[from++];
                    len -= 3;
                  }
                  if (len) {
                    output[_out++] = from_source[from++];
                    if (len > 1) {
                      output[_out++] = from_source[from++];
                    }
                  }
                } else {
                  from = _out - dist; /* copy direct from output */
                  do {
                    /* minimum length is three */
                    output[_out++] = output[from++];
                    output[_out++] = output[from++];
                    output[_out++] = output[from++];
                    len -= 3;
                  } while (len > 2);
                  if (len) {
                    output[_out++] = output[from++];
                    if (len > 1) {
                      output[_out++] = output[from++];
                    }
                  }
                }
              } else if ((op & 64) === 0) {
                /* 2nd level distance code */
                here = dcode[(here & 0xffff /*here.val*/) + (hold & (1 << op) - 1)];
                continue dodist;
              } else {
                strm.msg = 'invalid distance code';
                state.mode = BAD$1;
                break top;
              }
              break; // need to emulate goto via "continue"
            }
          } else if ((op & 64) === 0) {
            /* 2nd level length code */
            here = lcode[(here & 0xffff /*here.val*/) + (hold & (1 << op) - 1)];
            continue dolen;
          } else if (op & 32) {
            /* end-of-block */
            //Tracevv((stderr, "inflate:         end of block\n"));
            state.mode = TYPE$1;
            break top;
          } else {
            strm.msg = 'invalid literal/length code';
            state.mode = BAD$1;
            break top;
          }
          break; // need to emulate goto via "continue"
        }
      } while (_in < last && _out < end);

      /* return unused bytes (on entry, bits < 8, so in won't go too far back) */
      len = bits >> 3;
      _in -= len;
      bits -= len << 3;
      hold &= (1 << bits) - 1;

      /* update state and return */
      strm.next_in = _in;
      strm.next_out = _out;
      strm.avail_in = _in < last ? 5 + (last - _in) : 5 - (_in - last);
      strm.avail_out = _out < end ? 257 + (end - _out) : 257 - (_out - end);
      state.hold = hold;
      state.bits = bits;
      return;
    };

    // (C) 1995-2013 Jean-loup Gailly and Mark Adler
    // (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
    //
    // This software is provided 'as-is', without any express or implied
    // warranty. In no event will the authors be held liable for any damages
    // arising from the use of this software.
    //
    // Permission is granted to anyone to use this software for any purpose,
    // including commercial applications, and to alter it and redistribute it
    // freely, subject to the following restrictions:
    //
    // 1. The origin of this software must not be misrepresented; you must not
    //   claim that you wrote the original software. If you use this software
    //   in a product, an acknowledgment in the product documentation would be
    //   appreciated but is not required.
    // 2. Altered source versions must be plainly marked as such, and must not be
    //   misrepresented as being the original software.
    // 3. This notice may not be removed or altered from any source distribution.

    const MAXBITS = 15;
    const ENOUGH_LENS$1 = 852;
    const ENOUGH_DISTS$1 = 592;
    //const ENOUGH = (ENOUGH_LENS+ENOUGH_DISTS);

    const CODES$1 = 0;
    const LENS$1 = 1;
    const DISTS$1 = 2;
    const lbase = new Uint16Array([/* Length codes 257..285 base */
    3, 4, 5, 6, 7, 8, 9, 10, 11, 13, 15, 17, 19, 23, 27, 31, 35, 43, 51, 59, 67, 83, 99, 115, 131, 163, 195, 227, 258, 0, 0]);
    const lext = new Uint8Array([/* Length codes 257..285 extra */
    16, 16, 16, 16, 16, 16, 16, 16, 17, 17, 17, 17, 18, 18, 18, 18, 19, 19, 19, 19, 20, 20, 20, 20, 21, 21, 21, 21, 16, 72, 78]);
    const dbase = new Uint16Array([/* Distance codes 0..29 base */
    1, 2, 3, 4, 5, 7, 9, 13, 17, 25, 33, 49, 65, 97, 129, 193, 257, 385, 513, 769, 1025, 1537, 2049, 3073, 4097, 6145, 8193, 12289, 16385, 24577, 0, 0]);
    const dext = new Uint8Array([/* Distance codes 0..29 extra */
    16, 16, 16, 16, 17, 17, 18, 18, 19, 19, 20, 20, 21, 21, 22, 22, 23, 23, 24, 24, 25, 25, 26, 26, 27, 27, 28, 28, 29, 29, 64, 64]);
    const inflate_table = (type, lens, lens_index, codes, table, table_index, work, opts) => {
      const bits = opts.bits;
      //here = opts.here; /* table entry for duplication */

      let len = 0; /* a code's length in bits */
      let sym = 0; /* index of code symbols */
      let min = 0,
        max = 0; /* minimum and maximum code lengths */
      let root = 0; /* number of index bits for root table */
      let curr = 0; /* number of index bits for current table */
      let drop = 0; /* code bits to drop for sub-table */
      let left = 0; /* number of prefix codes available */
      let used = 0; /* code entries in table used */
      let huff = 0; /* Huffman code */
      let incr; /* for incrementing code, index */
      let fill; /* index for replicating entries */
      let low; /* low bits for current root entry */
      let mask; /* mask for low root bits */
      let next; /* next available space in table */
      let base = null; /* base value table to use */
      //  let shoextra;    /* extra bits table to use */
      let match; /* use base and extra for symbol >= match */
      const count = new Uint16Array(MAXBITS + 1); //[MAXBITS+1];    /* number of codes of each length */
      const offs = new Uint16Array(MAXBITS + 1); //[MAXBITS+1];     /* offsets in table for each length */
      let extra = null;
      let here_bits, here_op, here_val;

      /*
       Process a set of code lengths to create a canonical Huffman code.  The
       code lengths are lens[0..codes-1].  Each length corresponds to the
       symbols 0..codes-1.  The Huffman code is generated by first sorting the
       symbols by length from short to long, and retaining the symbol order
       for codes with equal lengths.  Then the code starts with all zero bits
       for the first code of the shortest length, and the codes are integer
       increments for the same length, and zeros are appended as the length
       increases.  For the deflate format, these bits are stored backwards
       from their more natural integer increment ordering, and so when the
       decoding tables are built in the large loop below, the integer codes
       are incremented backwards.
        This routine assumes, but does not check, that all of the entries in
       lens[] are in the range 0..MAXBITS.  The caller must assure this.
       1..MAXBITS is interpreted as that code length.  zero means that that
       symbol does not occur in this code.
        The codes are sorted by computing a count of codes for each length,
       creating from that a table of starting indices for each length in the
       sorted table, and then entering the symbols in order in the sorted
       table.  The sorted table is work[], with that space being provided by
       the caller.
        The length counts are used for other purposes as well, i.e. finding
       the minimum and maximum length codes, determining if there are any
       codes at all, checking for a valid set of lengths, and looking ahead
       at length counts to determine sub-table sizes when building the
       decoding tables.
       */

      /* accumulate lengths for codes (assumes lens[] all in 0..MAXBITS) */
      for (len = 0; len <= MAXBITS; len++) {
        count[len] = 0;
      }
      for (sym = 0; sym < codes; sym++) {
        count[lens[lens_index + sym]]++;
      }

      /* bound code lengths, force root to be within code lengths */
      root = bits;
      for (max = MAXBITS; max >= 1; max--) {
        if (count[max] !== 0) {
          break;
        }
      }
      if (root > max) {
        root = max;
      }
      if (max === 0) {
        /* no symbols to code at all */
        //table.op[opts.table_index] = 64;  //here.op = (var char)64;    /* invalid code marker */
        //table.bits[opts.table_index] = 1;   //here.bits = (var char)1;
        //table.val[opts.table_index++] = 0;   //here.val = (var short)0;
        table[table_index++] = 1 << 24 | 64 << 16 | 0;

        //table.op[opts.table_index] = 64;
        //table.bits[opts.table_index] = 1;
        //table.val[opts.table_index++] = 0;
        table[table_index++] = 1 << 24 | 64 << 16 | 0;
        opts.bits = 1;
        return 0; /* no symbols, but wait for decoding to report error */
      }
      for (min = 1; min < max; min++) {
        if (count[min] !== 0) {
          break;
        }
      }
      if (root < min) {
        root = min;
      }

      /* check for an over-subscribed or incomplete set of lengths */
      left = 1;
      for (len = 1; len <= MAXBITS; len++) {
        left <<= 1;
        left -= count[len];
        if (left < 0) {
          return -1;
        } /* over-subscribed */
      }
      if (left > 0 && (type === CODES$1 || max !== 1)) {
        return -1; /* incomplete set */
      }

      /* generate offsets into symbol table for each length for sorting */
      offs[1] = 0;
      for (len = 1; len < MAXBITS; len++) {
        offs[len + 1] = offs[len] + count[len];
      }

      /* sort symbols by length, by symbol order within each length */
      for (sym = 0; sym < codes; sym++) {
        if (lens[lens_index + sym] !== 0) {
          work[offs[lens[lens_index + sym]]++] = sym;
        }
      }

      /*
       Create and fill in decoding tables.  In this loop, the table being
       filled is at next and has curr index bits.  The code being used is huff
       with length len.  That code is converted to an index by dropping drop
       bits off of the bottom.  For codes where len is less than drop + curr,
       those top drop + curr - len bits are incremented through all values to
       fill the table with replicated entries.
        root is the number of index bits for the root table.  When len exceeds
       root, sub-tables are created pointed to by the root entry with an index
       of the low root bits of huff.  This is saved in low to check for when a
       new sub-table should be started.  drop is zero when the root table is
       being filled, and drop is root when sub-tables are being filled.
        When a new sub-table is needed, it is necessary to look ahead in the
       code lengths to determine what size sub-table is needed.  The length
       counts are used for this, and so count[] is decremented as codes are
       entered in the tables.
        used keeps track of how many table entries have been allocated from the
       provided *table space.  It is checked for LENS and DIST tables against
       the constants ENOUGH_LENS and ENOUGH_DISTS to guard against changes in
       the initial root table size constants.  See the comments in inftrees.h
       for more information.
        sym increments through all symbols, and the loop terminates when
       all codes of length max, i.e. all codes, have been processed.  This
       routine permits incomplete codes, so another loop after this one fills
       in the rest of the decoding tables with invalid code markers.
       */

      /* set up for code type */
      // poor man optimization - use if-else instead of switch,
      // to avoid deopts in old v8
      if (type === CODES$1) {
        base = extra = work; /* dummy value--not used */
        match = 20;
      } else if (type === LENS$1) {
        base = lbase;
        extra = lext;
        match = 257;
      } else {
        /* DISTS */
        base = dbase;
        extra = dext;
        match = 0;
      }

      /* initialize opts for loop */
      huff = 0; /* starting code */
      sym = 0; /* starting code symbol */
      len = min; /* starting code length */
      next = table_index; /* current table to fill in */
      curr = root; /* current table index bits */
      drop = 0; /* current bits to drop from code for index */
      low = -1; /* trigger new sub-table when len > root */
      used = 1 << root; /* use root table entries */
      mask = used - 1; /* mask for comparing low */

      /* check available table space */
      if (type === LENS$1 && used > ENOUGH_LENS$1 || type === DISTS$1 && used > ENOUGH_DISTS$1) {
        return 1;
      }

      /* process all codes and make table entries */
      for (;;) {
        /* create table entry */
        here_bits = len - drop;
        if (work[sym] + 1 < match) {
          here_op = 0;
          here_val = work[sym];
        } else if (work[sym] >= match) {
          here_op = extra[work[sym] - match];
          here_val = base[work[sym] - match];
        } else {
          here_op = 32 + 64; /* end of block */
          here_val = 0;
        }

        /* replicate for those indices with low len bits equal to huff */
        incr = 1 << len - drop;
        fill = 1 << curr;
        min = fill; /* save offset to next table */
        do {
          fill -= incr;
          table[next + (huff >> drop) + fill] = here_bits << 24 | here_op << 16 | here_val | 0;
        } while (fill !== 0);

        /* backwards increment the len-bit code huff */
        incr = 1 << len - 1;
        while (huff & incr) {
          incr >>= 1;
        }
        if (incr !== 0) {
          huff &= incr - 1;
          huff += incr;
        } else {
          huff = 0;
        }

        /* go to next symbol, update count, len */
        sym++;
        if (--count[len] === 0) {
          if (len === max) {
            break;
          }
          len = lens[lens_index + work[sym]];
        }

        /* create new sub-table if needed */
        if (len > root && (huff & mask) !== low) {
          /* if first time, transition to sub-tables */
          if (drop === 0) {
            drop = root;
          }

          /* increment past last table */
          next += min; /* here min is 1 << curr */

          /* determine length of next table */
          curr = len - drop;
          left = 1 << curr;
          while (curr + drop < max) {
            left -= count[curr + drop];
            if (left <= 0) {
              break;
            }
            curr++;
            left <<= 1;
          }

          /* check for enough space */
          used += 1 << curr;
          if (type === LENS$1 && used > ENOUGH_LENS$1 || type === DISTS$1 && used > ENOUGH_DISTS$1) {
            return 1;
          }

          /* point entry in root table to sub-table */
          low = huff & mask;
          /*table.op[low] = curr;
          table.bits[low] = root;
          table.val[low] = next - opts.table_index;*/
          table[low] = root << 24 | curr << 16 | next - table_index | 0;
        }
      }

      /* fill in remaining table entry if code is incomplete (guaranteed to have
       at most one remaining entry, since if the code is incomplete, the
       maximum code length that was allowed to get this far is one bit) */
      if (huff !== 0) {
        //table.op[next + huff] = 64;            /* invalid code marker */
        //table.bits[next + huff] = len - drop;
        //table.val[next + huff] = 0;
        table[next + huff] = len - drop << 24 | 64 << 16 | 0;
      }

      /* set return parameters */
      //opts.table_index += used;
      opts.bits = root;
      return 0;
    };
    var inftrees = inflate_table;

    // (C) 1995-2013 Jean-loup Gailly and Mark Adler
    // (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
    //
    // This software is provided 'as-is', without any express or implied
    // warranty. In no event will the authors be held liable for any damages
    // arising from the use of this software.
    //
    // Permission is granted to anyone to use this software for any purpose,
    // including commercial applications, and to alter it and redistribute it
    // freely, subject to the following restrictions:
    //
    // 1. The origin of this software must not be misrepresented; you must not
    //   claim that you wrote the original software. If you use this software
    //   in a product, an acknowledgment in the product documentation would be
    //   appreciated but is not required.
    // 2. Altered source versions must be plainly marked as such, and must not be
    //   misrepresented as being the original software.
    // 3. This notice may not be removed or altered from any source distribution.

    const CODES = 0;
    const LENS = 1;
    const DISTS = 2;

    /* Public constants ==========================================================*/
    /* ===========================================================================*/

    const {
      Z_FINISH: Z_FINISH$1,
      Z_BLOCK,
      Z_TREES,
      Z_OK: Z_OK$1,
      Z_STREAM_END: Z_STREAM_END$1,
      Z_NEED_DICT: Z_NEED_DICT$1,
      Z_STREAM_ERROR: Z_STREAM_ERROR$1,
      Z_DATA_ERROR: Z_DATA_ERROR$1,
      Z_MEM_ERROR: Z_MEM_ERROR$1,
      Z_BUF_ERROR,
      Z_DEFLATED
    } = constants$2;

    /* STATES ====================================================================*/
    /* ===========================================================================*/

    const HEAD = 16180; /* i: waiting for magic header */
    const FLAGS = 16181; /* i: waiting for method and flags (gzip) */
    const TIME = 16182; /* i: waiting for modification time (gzip) */
    const OS = 16183; /* i: waiting for extra flags and operating system (gzip) */
    const EXLEN = 16184; /* i: waiting for extra length (gzip) */
    const EXTRA = 16185; /* i: waiting for extra bytes (gzip) */
    const NAME = 16186; /* i: waiting for end of file name (gzip) */
    const COMMENT = 16187; /* i: waiting for end of comment (gzip) */
    const HCRC = 16188; /* i: waiting for header crc (gzip) */
    const DICTID = 16189; /* i: waiting for dictionary check value */
    const DICT = 16190; /* waiting for inflateSetDictionary() call */
    const TYPE = 16191; /* i: waiting for type bits, including last-flag bit */
    const TYPEDO = 16192; /* i: same, but skip check to exit inflate on new block */
    const STORED = 16193; /* i: waiting for stored size (length and complement) */
    const COPY_ = 16194; /* i/o: same as COPY below, but only first time in */
    const COPY = 16195; /* i/o: waiting for input or output to copy stored block */
    const TABLE = 16196; /* i: waiting for dynamic block table lengths */
    const LENLENS = 16197; /* i: waiting for code length code lengths */
    const CODELENS = 16198; /* i: waiting for length/lit and distance code lengths */
    const LEN_ = 16199; /* i: same as LEN below, but only first time in */
    const LEN = 16200; /* i: waiting for length/lit/eob code */
    const LENEXT = 16201; /* i: waiting for length extra bits */
    const DIST = 16202; /* i: waiting for distance code */
    const DISTEXT = 16203; /* i: waiting for distance extra bits */
    const MATCH = 16204; /* o: waiting for output space to copy string */
    const LIT = 16205; /* o: waiting for output space to write literal */
    const CHECK = 16206; /* i: waiting for 32-bit check value */
    const LENGTH = 16207; /* i: waiting for 32-bit length (gzip) */
    const DONE = 16208; /* finished check, done -- remain here until reset */
    const BAD = 16209; /* got a data error -- remain here until reset */
    const MEM = 16210; /* got an inflate() memory error -- remain here until reset */
    const SYNC = 16211; /* looking for synchronization bytes to restart inflate() */

    /* ===========================================================================*/

    const ENOUGH_LENS = 852;
    const ENOUGH_DISTS = 592;
    //const ENOUGH =  (ENOUGH_LENS+ENOUGH_DISTS);

    const MAX_WBITS = 15;
    /* 32K LZ77 window */
    const DEF_WBITS = MAX_WBITS;
    const zswap32 = q => {
      return (q >>> 24 & 0xff) + (q >>> 8 & 0xff00) + ((q & 0xff00) << 8) + ((q & 0xff) << 24);
    };
    function InflateState() {
      this.strm = null; /* pointer back to this zlib stream */
      this.mode = 0; /* current inflate mode */
      this.last = false; /* true if processing last block */
      this.wrap = 0; /* bit 0 true for zlib, bit 1 true for gzip,
                        bit 2 true to validate check value */
      this.havedict = false; /* true if dictionary provided */
      this.flags = 0; /* gzip header method and flags (0 if zlib), or
                         -1 if raw or no header yet */
      this.dmax = 0; /* zlib header max distance (INFLATE_STRICT) */
      this.check = 0; /* protected copy of check value */
      this.total = 0; /* protected copy of output count */
      // TODO: may be {}
      this.head = null; /* where to save gzip header information */

      /* sliding window */
      this.wbits = 0; /* log base 2 of requested window size */
      this.wsize = 0; /* window size or zero if not using window */
      this.whave = 0; /* valid bytes in the window */
      this.wnext = 0; /* window write index */
      this.window = null; /* allocated sliding window, if needed */

      /* bit accumulator */
      this.hold = 0; /* input bit accumulator */
      this.bits = 0; /* number of bits in "in" */

      /* for string and stored block copying */
      this.length = 0; /* literal or length of data to copy */
      this.offset = 0; /* distance back to copy string from */

      /* for table and code decoding */
      this.extra = 0; /* extra bits needed */

      /* fixed and dynamic code tables */
      this.lencode = null; /* starting table for length/literal codes */
      this.distcode = null; /* starting table for distance codes */
      this.lenbits = 0; /* index bits for lencode */
      this.distbits = 0; /* index bits for distcode */

      /* dynamic table building */
      this.ncode = 0; /* number of code length code lengths */
      this.nlen = 0; /* number of length code lengths */
      this.ndist = 0; /* number of distance code lengths */
      this.have = 0; /* number of code lengths in lens[] */
      this.next = null; /* next available space in codes[] */

      this.lens = new Uint16Array(320); /* temporary storage for code lengths */
      this.work = new Uint16Array(288); /* work area for code table building */

      /*
       because we don't have pointers in js, we use lencode and distcode directly
       as buffers so we don't need codes
      */
      //this.codes = new Int32Array(ENOUGH);       /* space for code tables */
      this.lendyn = null; /* dynamic table for length/literal codes (JS specific) */
      this.distdyn = null; /* dynamic table for distance codes (JS specific) */
      this.sane = 0; /* if false, allow invalid distance too far */
      this.back = 0; /* bits back of last unprocessed length/lit */
      this.was = 0; /* initial length of match */
    }
    const inflateStateCheck = strm => {
      if (!strm) {
        return 1;
      }
      const state = strm.state;
      if (!state || state.strm !== strm || state.mode < HEAD || state.mode > SYNC) {
        return 1;
      }
      return 0;
    };
    const inflateResetKeep = strm => {
      if (inflateStateCheck(strm)) {
        return Z_STREAM_ERROR$1;
      }
      const state = strm.state;
      strm.total_in = strm.total_out = state.total = 0;
      strm.msg = ''; /*Z_NULL*/
      if (state.wrap) {
        /* to support ill-conceived Java test suite */
        strm.adler = state.wrap & 1;
      }
      state.mode = HEAD;
      state.last = 0;
      state.havedict = 0;
      state.flags = -1;
      state.dmax = 32768;
      state.head = null /*Z_NULL*/;
      state.hold = 0;
      state.bits = 0;
      //state.lencode = state.distcode = state.next = state.codes;
      state.lencode = state.lendyn = new Int32Array(ENOUGH_LENS);
      state.distcode = state.distdyn = new Int32Array(ENOUGH_DISTS);
      state.sane = 1;
      state.back = -1;
      //Tracev((stderr, "inflate: reset\n"));
      return Z_OK$1;
    };
    const inflateReset = strm => {
      if (inflateStateCheck(strm)) {
        return Z_STREAM_ERROR$1;
      }
      const state = strm.state;
      state.wsize = 0;
      state.whave = 0;
      state.wnext = 0;
      return inflateResetKeep(strm);
    };
    const inflateReset2 = (strm, windowBits) => {
      let wrap;

      /* get the state */
      if (inflateStateCheck(strm)) {
        return Z_STREAM_ERROR$1;
      }
      const state = strm.state;

      /* extract wrap request from windowBits parameter */
      if (windowBits < 0) {
        wrap = 0;
        windowBits = -windowBits;
      } else {
        wrap = (windowBits >> 4) + 5;
        if (windowBits < 48) {
          windowBits &= 15;
        }
      }

      /* set number of window bits, free window if different */
      if (windowBits && (windowBits < 8 || windowBits > 15)) {
        return Z_STREAM_ERROR$1;
      }
      if (state.window !== null && state.wbits !== windowBits) {
        state.window = null;
      }

      /* update state and reset the rest of it */
      state.wrap = wrap;
      state.wbits = windowBits;
      return inflateReset(strm);
    };
    const inflateInit2 = (strm, windowBits) => {
      if (!strm) {
        return Z_STREAM_ERROR$1;
      }
      //strm.msg = Z_NULL;                 /* in case we return an error */

      const state = new InflateState();

      //if (state === Z_NULL) return Z_MEM_ERROR;
      //Tracev((stderr, "inflate: allocated\n"));
      strm.state = state;
      state.strm = strm;
      state.window = null /*Z_NULL*/;
      state.mode = HEAD; /* to pass state test in inflateReset2() */
      const ret = inflateReset2(strm, windowBits);
      if (ret !== Z_OK$1) {
        strm.state = null /*Z_NULL*/;
      }
      return ret;
    };
    const inflateInit = strm => {
      return inflateInit2(strm, DEF_WBITS);
    };

    /*
     Return state with length and distance decoding tables and index sizes set to
     fixed code decoding.  Normally this returns fixed tables from inffixed.h.
     If BUILDFIXED is defined, then instead this routine builds the tables the
     first time it's called, and returns those tables the first time and
     thereafter.  This reduces the size of the code by about 2K bytes, in
     exchange for a little execution time.  However, BUILDFIXED should not be
     used for threaded applications, since the rewriting of the tables and virgin
     may not be thread-safe.
     */
    let virgin = true;
    let lenfix, distfix; // We have no pointers in JS, so keep tables separate

    const fixedtables = state => {
      /* build fixed huffman tables if first call (may not be thread safe) */
      if (virgin) {
        lenfix = new Int32Array(512);
        distfix = new Int32Array(32);

        /* literal/length table */
        let sym = 0;
        while (sym < 144) {
          state.lens[sym++] = 8;
        }
        while (sym < 256) {
          state.lens[sym++] = 9;
        }
        while (sym < 280) {
          state.lens[sym++] = 7;
        }
        while (sym < 288) {
          state.lens[sym++] = 8;
        }
        inftrees(LENS, state.lens, 0, 288, lenfix, 0, state.work, {
          bits: 9
        });

        /* distance table */
        sym = 0;
        while (sym < 32) {
          state.lens[sym++] = 5;
        }
        inftrees(DISTS, state.lens, 0, 32, distfix, 0, state.work, {
          bits: 5
        });

        /* do this just once */
        virgin = false;
      }
      state.lencode = lenfix;
      state.lenbits = 9;
      state.distcode = distfix;
      state.distbits = 5;
    };

    /*
     Update the window with the last wsize (normally 32K) bytes written before
     returning.  If window does not exist yet, create it.  This is only called
     when a window is already in use, or when output has been written during this
     inflate call, but the end of the deflate stream has not been reached yet.
     It is also called to create a window for dictionary data when a dictionary
     is loaded.

     Providing output buffers larger than 32K to inflate() should provide a speed
     advantage, since only the last 32K of output is copied to the sliding window
     upon return from inflate(), and since all distances after the first 32K of
     output will fall in the output data, making match copies simpler and faster.
     The advantage may be dependent on the size of the processor's data caches.
     */
    const updatewindow = (strm, src, end, copy) => {
      let dist;
      const state = strm.state;

      /* if it hasn't been done already, allocate space for the window */
      if (state.window === null) {
        state.wsize = 1 << state.wbits;
        state.wnext = 0;
        state.whave = 0;
        state.window = new Uint8Array(state.wsize);
      }

      /* copy state->wsize or less output bytes into the circular window */
      if (copy >= state.wsize) {
        state.window.set(src.subarray(end - state.wsize, end), 0);
        state.wnext = 0;
        state.whave = state.wsize;
      } else {
        dist = state.wsize - state.wnext;
        if (dist > copy) {
          dist = copy;
        }
        //zmemcpy(state->window + state->wnext, end - copy, dist);
        state.window.set(src.subarray(end - copy, end - copy + dist), state.wnext);
        copy -= dist;
        if (copy) {
          //zmemcpy(state->window, end - copy, copy);
          state.window.set(src.subarray(end - copy, end), 0);
          state.wnext = copy;
          state.whave = state.wsize;
        } else {
          state.wnext += dist;
          if (state.wnext === state.wsize) {
            state.wnext = 0;
          }
          if (state.whave < state.wsize) {
            state.whave += dist;
          }
        }
      }
      return 0;
    };
    const inflate$2 = (strm, flush) => {
      let state;
      let input, output; // input/output buffers
      let next; /* next input INDEX */
      let put; /* next output INDEX */
      let have, left; /* available input and output */
      let hold; /* bit buffer */
      let bits; /* bits in bit buffer */
      let _in, _out; /* save starting available input and output */
      let copy; /* number of stored or match bytes to copy */
      let from; /* where to copy match bytes from */
      let from_source;
      let here = 0; /* current decoding table entry */
      let here_bits, here_op, here_val; // paked "here" denormalized (JS specific)
      //let last;                   /* parent table entry */
      let last_bits, last_op, last_val; // paked "last" denormalized (JS specific)
      let len; /* length to copy for repeats, bits to drop */
      let ret; /* return code */
      const hbuf = new Uint8Array(4); /* buffer for gzip header crc calculation */
      let opts;
      let n; // temporary variable for NEED_BITS

      const order = /* permutation of code lengths */
      new Uint8Array([16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15]);
      if (inflateStateCheck(strm) || !strm.output || !strm.input && strm.avail_in !== 0) {
        return Z_STREAM_ERROR$1;
      }
      state = strm.state;
      if (state.mode === TYPE) {
        state.mode = TYPEDO;
      } /* skip check */

      //--- LOAD() ---
      put = strm.next_out;
      output = strm.output;
      left = strm.avail_out;
      next = strm.next_in;
      input = strm.input;
      have = strm.avail_in;
      hold = state.hold;
      bits = state.bits;
      //---

      _in = have;
      _out = left;
      ret = Z_OK$1;
      inf_leave:
      // goto emulation
      for (;;) {
        switch (state.mode) {
          case HEAD:
            if (state.wrap === 0) {
              state.mode = TYPEDO;
              break;
            }
            //=== NEEDBITS(16);
            while (bits < 16) {
              if (have === 0) {
                break inf_leave;
              }
              have--;
              hold += input[next++] << bits;
              bits += 8;
            }
            //===//
            if (state.wrap & 2 && hold === 0x8b1f) {
              /* gzip header */
              if (state.wbits === 0) {
                state.wbits = 15;
              }
              state.check = 0 /*crc32(0L, Z_NULL, 0)*/;
              //=== CRC2(state.check, hold);
              hbuf[0] = hold & 0xff;
              hbuf[1] = hold >>> 8 & 0xff;
              state.check = crc32_1(state.check, hbuf, 2, 0);
              //===//

              //=== INITBITS();
              hold = 0;
              bits = 0;
              //===//
              state.mode = FLAGS;
              break;
            }
            if (state.head) {
              state.head.done = false;
            }
            if (!(state.wrap & 1) || /* check if zlib header allowed */
            (((hold & 0xff /*BITS(8)*/) << 8) + (hold >> 8)) % 31) {
              strm.msg = 'incorrect header check';
              state.mode = BAD;
              break;
            }
            if ((hold & 0x0f /*BITS(4)*/) !== Z_DEFLATED) {
              strm.msg = 'unknown compression method';
              state.mode = BAD;
              break;
            }
            //--- DROPBITS(4) ---//
            hold >>>= 4;
            bits -= 4;
            //---//
            len = (hold & 0x0f /*BITS(4)*/) + 8;
            if (state.wbits === 0) {
              state.wbits = len;
            }
            if (len > 15 || len > state.wbits) {
              strm.msg = 'invalid window size';
              state.mode = BAD;
              break;
            }

            // !!! pako patch. Force use `options.windowBits` if passed.
            // Required to always use max window size by default.
            state.dmax = 1 << state.wbits;
            //state.dmax = 1 << len;

            state.flags = 0; /* indicate zlib header */
            //Tracev((stderr, "inflate:   zlib header ok\n"));
            strm.adler = state.check = 1 /*adler32(0L, Z_NULL, 0)*/;
            state.mode = hold & 0x200 ? DICTID : TYPE;
            //=== INITBITS();
            hold = 0;
            bits = 0;
            //===//
            break;
          case FLAGS:
            //=== NEEDBITS(16); */
            while (bits < 16) {
              if (have === 0) {
                break inf_leave;
              }
              have--;
              hold += input[next++] << bits;
              bits += 8;
            }
            //===//
            state.flags = hold;
            if ((state.flags & 0xff) !== Z_DEFLATED) {
              strm.msg = 'unknown compression method';
              state.mode = BAD;
              break;
            }
            if (state.flags & 0xe000) {
              strm.msg = 'unknown header flags set';
              state.mode = BAD;
              break;
            }
            if (state.head) {
              state.head.text = hold >> 8 & 1;
            }
            if (state.flags & 0x0200 && state.wrap & 4) {
              //=== CRC2(state.check, hold);
              hbuf[0] = hold & 0xff;
              hbuf[1] = hold >>> 8 & 0xff;
              state.check = crc32_1(state.check, hbuf, 2, 0);
              //===//
            }
            //=== INITBITS();
            hold = 0;
            bits = 0;
            //===//
            state.mode = TIME;
          /* falls through */
          case TIME:
            //=== NEEDBITS(32); */
            while (bits < 32) {
              if (have === 0) {
                break inf_leave;
              }
              have--;
              hold += input[next++] << bits;
              bits += 8;
            }
            //===//
            if (state.head) {
              state.head.time = hold;
            }
            if (state.flags & 0x0200 && state.wrap & 4) {
              //=== CRC4(state.check, hold)
              hbuf[0] = hold & 0xff;
              hbuf[1] = hold >>> 8 & 0xff;
              hbuf[2] = hold >>> 16 & 0xff;
              hbuf[3] = hold >>> 24 & 0xff;
              state.check = crc32_1(state.check, hbuf, 4, 0);
              //===
            }
            //=== INITBITS();
            hold = 0;
            bits = 0;
            //===//
            state.mode = OS;
          /* falls through */
          case OS:
            //=== NEEDBITS(16); */
            while (bits < 16) {
              if (have === 0) {
                break inf_leave;
              }
              have--;
              hold += input[next++] << bits;
              bits += 8;
            }
            //===//
            if (state.head) {
              state.head.xflags = hold & 0xff;
              state.head.os = hold >> 8;
            }
            if (state.flags & 0x0200 && state.wrap & 4) {
              //=== CRC2(state.check, hold);
              hbuf[0] = hold & 0xff;
              hbuf[1] = hold >>> 8 & 0xff;
              state.check = crc32_1(state.check, hbuf, 2, 0);
              //===//
            }
            //=== INITBITS();
            hold = 0;
            bits = 0;
            //===//
            state.mode = EXLEN;
          /* falls through */
          case EXLEN:
            if (state.flags & 0x0400) {
              //=== NEEDBITS(16); */
              while (bits < 16) {
                if (have === 0) {
                  break inf_leave;
                }
                have--;
                hold += input[next++] << bits;
                bits += 8;
              }
              //===//
              state.length = hold;
              if (state.head) {
                state.head.extra_len = hold;
              }
              if (state.flags & 0x0200 && state.wrap & 4) {
                //=== CRC2(state.check, hold);
                hbuf[0] = hold & 0xff;
                hbuf[1] = hold >>> 8 & 0xff;
                state.check = crc32_1(state.check, hbuf, 2, 0);
                //===//
              }
              //=== INITBITS();
              hold = 0;
              bits = 0;
              //===//
            } else if (state.head) {
              state.head.extra = null /*Z_NULL*/;
            }
            state.mode = EXTRA;
          /* falls through */
          case EXTRA:
            if (state.flags & 0x0400) {
              copy = state.length;
              if (copy > have) {
                copy = have;
              }
              if (copy) {
                if (state.head) {
                  len = state.head.extra_len - state.length;
                  if (!state.head.extra) {
                    // Use untyped array for more convenient processing later
                    state.head.extra = new Uint8Array(state.head.extra_len);
                  }
                  state.head.extra.set(input.subarray(next,
                  // extra field is limited to 65536 bytes
                  // - no need for additional size check
                  next + copy), /*len + copy > state.head.extra_max - len ? state.head.extra_max : copy,*/
                  len);
                  //zmemcpy(state.head.extra + len, next,
                  //        len + copy > state.head.extra_max ?
                  //        state.head.extra_max - len : copy);
                }
                if (state.flags & 0x0200 && state.wrap & 4) {
                  state.check = crc32_1(state.check, input, copy, next);
                }
                have -= copy;
                next += copy;
                state.length -= copy;
              }
              if (state.length) {
                break inf_leave;
              }
            }
            state.length = 0;
            state.mode = NAME;
          /* falls through */
          case NAME:
            if (state.flags & 0x0800) {
              if (have === 0) {
                break inf_leave;
              }
              copy = 0;
              do {
                // TODO: 2 or 1 bytes?
                len = input[next + copy++];
                /* use constant limit because in js we should not preallocate memory */
                if (state.head && len && state.length < 65536 /*state.head.name_max*/) {
                  state.head.name += String.fromCharCode(len);
                }
              } while (len && copy < have);
              if (state.flags & 0x0200 && state.wrap & 4) {
                state.check = crc32_1(state.check, input, copy, next);
              }
              have -= copy;
              next += copy;
              if (len) {
                break inf_leave;
              }
            } else if (state.head) {
              state.head.name = null;
            }
            state.length = 0;
            state.mode = COMMENT;
          /* falls through */
          case COMMENT:
            if (state.flags & 0x1000) {
              if (have === 0) {
                break inf_leave;
              }
              copy = 0;
              do {
                len = input[next + copy++];
                /* use constant limit because in js we should not preallocate memory */
                if (state.head && len && state.length < 65536 /*state.head.comm_max*/) {
                  state.head.comment += String.fromCharCode(len);
                }
              } while (len && copy < have);
              if (state.flags & 0x0200 && state.wrap & 4) {
                state.check = crc32_1(state.check, input, copy, next);
              }
              have -= copy;
              next += copy;
              if (len) {
                break inf_leave;
              }
            } else if (state.head) {
              state.head.comment = null;
            }
            state.mode = HCRC;
          /* falls through */
          case HCRC:
            if (state.flags & 0x0200) {
              //=== NEEDBITS(16); */
              while (bits < 16) {
                if (have === 0) {
                  break inf_leave;
                }
                have--;
                hold += input[next++] << bits;
                bits += 8;
              }
              //===//
              if (state.wrap & 4 && hold !== (state.check & 0xffff)) {
                strm.msg = 'header crc mismatch';
                state.mode = BAD;
                break;
              }
              //=== INITBITS();
              hold = 0;
              bits = 0;
              //===//
            }
            if (state.head) {
              state.head.hcrc = state.flags >> 9 & 1;
              state.head.done = true;
            }
            strm.adler = state.check = 0;
            state.mode = TYPE;
            break;
          case DICTID:
            //=== NEEDBITS(32); */
            while (bits < 32) {
              if (have === 0) {
                break inf_leave;
              }
              have--;
              hold += input[next++] << bits;
              bits += 8;
            }
            //===//
            strm.adler = state.check = zswap32(hold);
            //=== INITBITS();
            hold = 0;
            bits = 0;
            //===//
            state.mode = DICT;
          /* falls through */
          case DICT:
            if (state.havedict === 0) {
              //--- RESTORE() ---
              strm.next_out = put;
              strm.avail_out = left;
              strm.next_in = next;
              strm.avail_in = have;
              state.hold = hold;
              state.bits = bits;
              //---
              return Z_NEED_DICT$1;
            }
            strm.adler = state.check = 1 /*adler32(0L, Z_NULL, 0)*/;
            state.mode = TYPE;
          /* falls through */
          case TYPE:
            if (flush === Z_BLOCK || flush === Z_TREES) {
              break inf_leave;
            }
          /* falls through */
          case TYPEDO:
            if (state.last) {
              //--- BYTEBITS() ---//
              hold >>>= bits & 7;
              bits -= bits & 7;
              //---//
              state.mode = CHECK;
              break;
            }
            //=== NEEDBITS(3); */
            while (bits < 3) {
              if (have === 0) {
                break inf_leave;
              }
              have--;
              hold += input[next++] << bits;
              bits += 8;
            }
            //===//
            state.last = hold & 0x01 /*BITS(1)*/;
            //--- DROPBITS(1) ---//
            hold >>>= 1;
            bits -= 1;
            //---//

            switch (hold & 0x03 /*BITS(2)*/) {
              case 0:
                /* stored block */
                //Tracev((stderr, "inflate:     stored block%s\n",
                //        state.last ? " (last)" : ""));
                state.mode = STORED;
                break;
              case 1:
                /* fixed block */
                fixedtables(state);
                //Tracev((stderr, "inflate:     fixed codes block%s\n",
                //        state.last ? " (last)" : ""));
                state.mode = LEN_; /* decode codes */
                if (flush === Z_TREES) {
                  //--- DROPBITS(2) ---//
                  hold >>>= 2;
                  bits -= 2;
                  //---//
                  break inf_leave;
                }
                break;
              case 2:
                /* dynamic block */
                //Tracev((stderr, "inflate:     dynamic codes block%s\n",
                //        state.last ? " (last)" : ""));
                state.mode = TABLE;
                break;
              case 3:
                strm.msg = 'invalid block type';
                state.mode = BAD;
            }
            //--- DROPBITS(2) ---//
            hold >>>= 2;
            bits -= 2;
            //---//
            break;
          case STORED:
            //--- BYTEBITS() ---// /* go to byte boundary */
            hold >>>= bits & 7;
            bits -= bits & 7;
            //---//
            //=== NEEDBITS(32); */
            while (bits < 32) {
              if (have === 0) {
                break inf_leave;
              }
              have--;
              hold += input[next++] << bits;
              bits += 8;
            }
            //===//
            if ((hold & 0xffff) !== (hold >>> 16 ^ 0xffff)) {
              strm.msg = 'invalid stored block lengths';
              state.mode = BAD;
              break;
            }
            state.length = hold & 0xffff;
            //Tracev((stderr, "inflate:       stored length %u\n",
            //        state.length));
            //=== INITBITS();
            hold = 0;
            bits = 0;
            //===//
            state.mode = COPY_;
            if (flush === Z_TREES) {
              break inf_leave;
            }
          /* falls through */
          case COPY_:
            state.mode = COPY;
          /* falls through */
          case COPY:
            copy = state.length;
            if (copy) {
              if (copy > have) {
                copy = have;
              }
              if (copy > left) {
                copy = left;
              }
              if (copy === 0) {
                break inf_leave;
              }
              //--- zmemcpy(put, next, copy); ---
              output.set(input.subarray(next, next + copy), put);
              //---//
              have -= copy;
              next += copy;
              left -= copy;
              put += copy;
              state.length -= copy;
              break;
            }
            //Tracev((stderr, "inflate:       stored end\n"));
            state.mode = TYPE;
            break;
          case TABLE:
            //=== NEEDBITS(14); */
            while (bits < 14) {
              if (have === 0) {
                break inf_leave;
              }
              have--;
              hold += input[next++] << bits;
              bits += 8;
            }
            //===//
            state.nlen = (hold & 0x1f /*BITS(5)*/) + 257;
            //--- DROPBITS(5) ---//
            hold >>>= 5;
            bits -= 5;
            //---//
            state.ndist = (hold & 0x1f /*BITS(5)*/) + 1;
            //--- DROPBITS(5) ---//
            hold >>>= 5;
            bits -= 5;
            //---//
            state.ncode = (hold & 0x0f /*BITS(4)*/) + 4;
            //--- DROPBITS(4) ---//
            hold >>>= 4;
            bits -= 4;
            //---//
            //#ifndef PKZIP_BUG_WORKAROUND
            if (state.nlen > 286 || state.ndist > 30) {
              strm.msg = 'too many length or distance symbols';
              state.mode = BAD;
              break;
            }
            //#endif
            //Tracev((stderr, "inflate:       table sizes ok\n"));
            state.have = 0;
            state.mode = LENLENS;
          /* falls through */
          case LENLENS:
            while (state.have < state.ncode) {
              //=== NEEDBITS(3);
              while (bits < 3) {
                if (have === 0) {
                  break inf_leave;
                }
                have--;
                hold += input[next++] << bits;
                bits += 8;
              }
              //===//
              state.lens[order[state.have++]] = hold & 0x07; //BITS(3);
              //--- DROPBITS(3) ---//
              hold >>>= 3;
              bits -= 3;
              //---//
            }
            while (state.have < 19) {
              state.lens[order[state.have++]] = 0;
            }
            // We have separate tables & no pointers. 2 commented lines below not needed.
            //state.next = state.codes;
            //state.lencode = state.next;
            // Switch to use dynamic table
            state.lencode = state.lendyn;
            state.lenbits = 7;
            opts = {
              bits: state.lenbits
            };
            ret = inftrees(CODES, state.lens, 0, 19, state.lencode, 0, state.work, opts);
            state.lenbits = opts.bits;
            if (ret) {
              strm.msg = 'invalid code lengths set';
              state.mode = BAD;
              break;
            }
            //Tracev((stderr, "inflate:       code lengths ok\n"));
            state.have = 0;
            state.mode = CODELENS;
          /* falls through */
          case CODELENS:
            while (state.have < state.nlen + state.ndist) {
              for (;;) {
                here = state.lencode[hold & (1 << state.lenbits) - 1]; /*BITS(state.lenbits)*/
                here_bits = here >>> 24;
                here_op = here >>> 16 & 0xff;
                here_val = here & 0xffff;
                if (here_bits <= bits) {
                  break;
                }
                //--- PULLBYTE() ---//
                if (have === 0) {
                  break inf_leave;
                }
                have--;
                hold += input[next++] << bits;
                bits += 8;
                //---//
              }
              if (here_val < 16) {
                //--- DROPBITS(here.bits) ---//
                hold >>>= here_bits;
                bits -= here_bits;
                //---//
                state.lens[state.have++] = here_val;
              } else {
                if (here_val === 16) {
                  //=== NEEDBITS(here.bits + 2);
                  n = here_bits + 2;
                  while (bits < n) {
                    if (have === 0) {
                      break inf_leave;
                    }
                    have--;
                    hold += input[next++] << bits;
                    bits += 8;
                  }
                  //===//
                  //--- DROPBITS(here.bits) ---//
                  hold >>>= here_bits;
                  bits -= here_bits;
                  //---//
                  if (state.have === 0) {
                    strm.msg = 'invalid bit length repeat';
                    state.mode = BAD;
                    break;
                  }
                  len = state.lens[state.have - 1];
                  copy = 3 + (hold & 0x03); //BITS(2);
                  //--- DROPBITS(2) ---//
                  hold >>>= 2;
                  bits -= 2;
                  //---//
                } else if (here_val === 17) {
                  //=== NEEDBITS(here.bits + 3);
                  n = here_bits + 3;
                  while (bits < n) {
                    if (have === 0) {
                      break inf_leave;
                    }
                    have--;
                    hold += input[next++] << bits;
                    bits += 8;
                  }
                  //===//
                  //--- DROPBITS(here.bits) ---//
                  hold >>>= here_bits;
                  bits -= here_bits;
                  //---//
                  len = 0;
                  copy = 3 + (hold & 0x07); //BITS(3);
                  //--- DROPBITS(3) ---//
                  hold >>>= 3;
                  bits -= 3;
                  //---//
                } else {
                  //=== NEEDBITS(here.bits + 7);
                  n = here_bits + 7;
                  while (bits < n) {
                    if (have === 0) {
                      break inf_leave;
                    }
                    have--;
                    hold += input[next++] << bits;
                    bits += 8;
                  }
                  //===//
                  //--- DROPBITS(here.bits) ---//
                  hold >>>= here_bits;
                  bits -= here_bits;
                  //---//
                  len = 0;
                  copy = 11 + (hold & 0x7f); //BITS(7);
                  //--- DROPBITS(7) ---//
                  hold >>>= 7;
                  bits -= 7;
                  //---//
                }
                if (state.have + copy > state.nlen + state.ndist) {
                  strm.msg = 'invalid bit length repeat';
                  state.mode = BAD;
                  break;
                }
                while (copy--) {
                  state.lens[state.have++] = len;
                }
              }
            }

            /* handle error breaks in while */
            if (state.mode === BAD) {
              break;
            }

            /* check for end-of-block code (better have one) */
            if (state.lens[256] === 0) {
              strm.msg = 'invalid code -- missing end-of-block';
              state.mode = BAD;
              break;
            }

            /* build code tables -- note: do not change the lenbits or distbits
               values here (9 and 6) without reading the comments in inftrees.h
               concerning the ENOUGH constants, which depend on those values */
            state.lenbits = 9;
            opts = {
              bits: state.lenbits
            };
            ret = inftrees(LENS, state.lens, 0, state.nlen, state.lencode, 0, state.work, opts);
            // We have separate tables & no pointers. 2 commented lines below not needed.
            // state.next_index = opts.table_index;
            state.lenbits = opts.bits;
            // state.lencode = state.next;

            if (ret) {
              strm.msg = 'invalid literal/lengths set';
              state.mode = BAD;
              break;
            }
            state.distbits = 6;
            //state.distcode.copy(state.codes);
            // Switch to use dynamic table
            state.distcode = state.distdyn;
            opts = {
              bits: state.distbits
            };
            ret = inftrees(DISTS, state.lens, state.nlen, state.ndist, state.distcode, 0, state.work, opts);
            // We have separate tables & no pointers. 2 commented lines below not needed.
            // state.next_index = opts.table_index;
            state.distbits = opts.bits;
            // state.distcode = state.next;

            if (ret) {
              strm.msg = 'invalid distances set';
              state.mode = BAD;
              break;
            }
            //Tracev((stderr, 'inflate:       codes ok\n'));
            state.mode = LEN_;
            if (flush === Z_TREES) {
              break inf_leave;
            }
          /* falls through */
          case LEN_:
            state.mode = LEN;
          /* falls through */
          case LEN:
            if (have >= 6 && left >= 258) {
              //--- RESTORE() ---
              strm.next_out = put;
              strm.avail_out = left;
              strm.next_in = next;
              strm.avail_in = have;
              state.hold = hold;
              state.bits = bits;
              //---
              inffast(strm, _out);
              //--- LOAD() ---
              put = strm.next_out;
              output = strm.output;
              left = strm.avail_out;
              next = strm.next_in;
              input = strm.input;
              have = strm.avail_in;
              hold = state.hold;
              bits = state.bits;
              //---

              if (state.mode === TYPE) {
                state.back = -1;
              }
              break;
            }
            state.back = 0;
            for (;;) {
              here = state.lencode[hold & (1 << state.lenbits) - 1]; /*BITS(state.lenbits)*/
              here_bits = here >>> 24;
              here_op = here >>> 16 & 0xff;
              here_val = here & 0xffff;
              if (here_bits <= bits) {
                break;
              }
              //--- PULLBYTE() ---//
              if (have === 0) {
                break inf_leave;
              }
              have--;
              hold += input[next++] << bits;
              bits += 8;
              //---//
            }
            if (here_op && (here_op & 0xf0) === 0) {
              last_bits = here_bits;
              last_op = here_op;
              last_val = here_val;
              for (;;) {
                here = state.lencode[last_val + ((hold & (1 << last_bits + last_op) - 1 /*BITS(last.bits + last.op)*/) >> last_bits)];
                here_bits = here >>> 24;
                here_op = here >>> 16 & 0xff;
                here_val = here & 0xffff;
                if (last_bits + here_bits <= bits) {
                  break;
                }
                //--- PULLBYTE() ---//
                if (have === 0) {
                  break inf_leave;
                }
                have--;
                hold += input[next++] << bits;
                bits += 8;
                //---//
              }
              //--- DROPBITS(last.bits) ---//
              hold >>>= last_bits;
              bits -= last_bits;
              //---//
              state.back += last_bits;
            }
            //--- DROPBITS(here.bits) ---//
            hold >>>= here_bits;
            bits -= here_bits;
            //---//
            state.back += here_bits;
            state.length = here_val;
            if (here_op === 0) {
              //Tracevv((stderr, here.val >= 0x20 && here.val < 0x7f ?
              //        "inflate:         literal '%c'\n" :
              //        "inflate:         literal 0x%02x\n", here.val));
              state.mode = LIT;
              break;
            }
            if (here_op & 32) {
              //Tracevv((stderr, "inflate:         end of block\n"));
              state.back = -1;
              state.mode = TYPE;
              break;
            }
            if (here_op & 64) {
              strm.msg = 'invalid literal/length code';
              state.mode = BAD;
              break;
            }
            state.extra = here_op & 15;
            state.mode = LENEXT;
          /* falls through */
          case LENEXT:
            if (state.extra) {
              //=== NEEDBITS(state.extra);
              n = state.extra;
              while (bits < n) {
                if (have === 0) {
                  break inf_leave;
                }
                have--;
                hold += input[next++] << bits;
                bits += 8;
              }
              //===//
              state.length += hold & (1 << state.extra) - 1 /*BITS(state.extra)*/;
              //--- DROPBITS(state.extra) ---//
              hold >>>= state.extra;
              bits -= state.extra;
              //---//
              state.back += state.extra;
            }
            //Tracevv((stderr, "inflate:         length %u\n", state.length));
            state.was = state.length;
            state.mode = DIST;
          /* falls through */
          case DIST:
            for (;;) {
              here = state.distcode[hold & (1 << state.distbits) - 1]; /*BITS(state.distbits)*/
              here_bits = here >>> 24;
              here_op = here >>> 16 & 0xff;
              here_val = here & 0xffff;
              if (here_bits <= bits) {
                break;
              }
              //--- PULLBYTE() ---//
              if (have === 0) {
                break inf_leave;
              }
              have--;
              hold += input[next++] << bits;
              bits += 8;
              //---//
            }
            if ((here_op & 0xf0) === 0) {
              last_bits = here_bits;
              last_op = here_op;
              last_val = here_val;
              for (;;) {
                here = state.distcode[last_val + ((hold & (1 << last_bits + last_op) - 1 /*BITS(last.bits + last.op)*/) >> last_bits)];
                here_bits = here >>> 24;
                here_op = here >>> 16 & 0xff;
                here_val = here & 0xffff;
                if (last_bits + here_bits <= bits) {
                  break;
                }
                //--- PULLBYTE() ---//
                if (have === 0) {
                  break inf_leave;
                }
                have--;
                hold += input[next++] << bits;
                bits += 8;
                //---//
              }
              //--- DROPBITS(last.bits) ---//
              hold >>>= last_bits;
              bits -= last_bits;
              //---//
              state.back += last_bits;
            }
            //--- DROPBITS(here.bits) ---//
            hold >>>= here_bits;
            bits -= here_bits;
            //---//
            state.back += here_bits;
            if (here_op & 64) {
              strm.msg = 'invalid distance code';
              state.mode = BAD;
              break;
            }
            state.offset = here_val;
            state.extra = here_op & 15;
            state.mode = DISTEXT;
          /* falls through */
          case DISTEXT:
            if (state.extra) {
              //=== NEEDBITS(state.extra);
              n = state.extra;
              while (bits < n) {
                if (have === 0) {
                  break inf_leave;
                }
                have--;
                hold += input[next++] << bits;
                bits += 8;
              }
              //===//
              state.offset += hold & (1 << state.extra) - 1 /*BITS(state.extra)*/;
              //--- DROPBITS(state.extra) ---//
              hold >>>= state.extra;
              bits -= state.extra;
              //---//
              state.back += state.extra;
            }
            //#ifdef INFLATE_STRICT
            if (state.offset > state.dmax) {
              strm.msg = 'invalid distance too far back';
              state.mode = BAD;
              break;
            }
            //#endif
            //Tracevv((stderr, "inflate:         distance %u\n", state.offset));
            state.mode = MATCH;
          /* falls through */
          case MATCH:
            if (left === 0) {
              break inf_leave;
            }
            copy = _out - left;
            if (state.offset > copy) {
              /* copy from window */
              copy = state.offset - copy;
              if (copy > state.whave) {
                if (state.sane) {
                  strm.msg = 'invalid distance too far back';
                  state.mode = BAD;
                  break;
                }
                // (!) This block is disabled in zlib defaults,
                // don't enable it for binary compatibility
                //#ifdef INFLATE_ALLOW_INVALID_DISTANCE_TOOFAR_ARRR
                //          Trace((stderr, "inflate.c too far\n"));
                //          copy -= state.whave;
                //          if (copy > state.length) { copy = state.length; }
                //          if (copy > left) { copy = left; }
                //          left -= copy;
                //          state.length -= copy;
                //          do {
                //            output[put++] = 0;
                //          } while (--copy);
                //          if (state.length === 0) { state.mode = LEN; }
                //          break;
                //#endif
              }
              if (copy > state.wnext) {
                copy -= state.wnext;
                from = state.wsize - copy;
              } else {
                from = state.wnext - copy;
              }
              if (copy > state.length) {
                copy = state.length;
              }
              from_source = state.window;
            } else {
              /* copy from output */
              from_source = output;
              from = put - state.offset;
              copy = state.length;
            }
            if (copy > left) {
              copy = left;
            }
            left -= copy;
            state.length -= copy;
            do {
              output[put++] = from_source[from++];
            } while (--copy);
            if (state.length === 0) {
              state.mode = LEN;
            }
            break;
          case LIT:
            if (left === 0) {
              break inf_leave;
            }
            output[put++] = state.length;
            left--;
            state.mode = LEN;
            break;
          case CHECK:
            if (state.wrap) {
              //=== NEEDBITS(32);
              while (bits < 32) {
                if (have === 0) {
                  break inf_leave;
                }
                have--;
                // Use '|' instead of '+' to make sure that result is signed
                hold |= input[next++] << bits;
                bits += 8;
              }
              //===//
              _out -= left;
              strm.total_out += _out;
              state.total += _out;
              if (state.wrap & 4 && _out) {
                strm.adler = state.check = /*UPDATE_CHECK(state.check, put - _out, _out);*/
                state.flags ? crc32_1(state.check, output, _out, put - _out) : adler32_1(state.check, output, _out, put - _out);
              }
              _out = left;
              // NB: crc32 stored as signed 32-bit int, zswap32 returns signed too
              if (state.wrap & 4 && (state.flags ? hold : zswap32(hold)) !== state.check) {
                strm.msg = 'incorrect data check';
                state.mode = BAD;
                break;
              }
              //=== INITBITS();
              hold = 0;
              bits = 0;
              //===//
              //Tracev((stderr, "inflate:   check matches trailer\n"));
            }
            state.mode = LENGTH;
          /* falls through */
          case LENGTH:
            if (state.wrap && state.flags) {
              //=== NEEDBITS(32);
              while (bits < 32) {
                if (have === 0) {
                  break inf_leave;
                }
                have--;
                hold += input[next++] << bits;
                bits += 8;
              }
              //===//
              if (state.wrap & 4 && hold !== (state.total & 0xffffffff)) {
                strm.msg = 'incorrect length check';
                state.mode = BAD;
                break;
              }
              //=== INITBITS();
              hold = 0;
              bits = 0;
              //===//
              //Tracev((stderr, "inflate:   length matches trailer\n"));
            }
            state.mode = DONE;
          /* falls through */
          case DONE:
            ret = Z_STREAM_END$1;
            break inf_leave;
          case BAD:
            ret = Z_DATA_ERROR$1;
            break inf_leave;
          case MEM:
            return Z_MEM_ERROR$1;
          case SYNC:
          /* falls through */
          default:
            return Z_STREAM_ERROR$1;
        }
      }

      // inf_leave <- here is real place for "goto inf_leave", emulated via "break inf_leave"

      /*
         Return from inflate(), updating the total counts and the check value.
         If there was no progress during the inflate() call, return a buffer
         error.  Call updatewindow() to create and/or update the window state.
         Note: a memory error from inflate() is non-recoverable.
       */

      //--- RESTORE() ---
      strm.next_out = put;
      strm.avail_out = left;
      strm.next_in = next;
      strm.avail_in = have;
      state.hold = hold;
      state.bits = bits;
      //---

      if (state.wsize || _out !== strm.avail_out && state.mode < BAD && (state.mode < CHECK || flush !== Z_FINISH$1)) {
        if (updatewindow(strm, strm.output, strm.next_out, _out - strm.avail_out)) ;
      }
      _in -= strm.avail_in;
      _out -= strm.avail_out;
      strm.total_in += _in;
      strm.total_out += _out;
      state.total += _out;
      if (state.wrap & 4 && _out) {
        strm.adler = state.check = /*UPDATE_CHECK(state.check, strm.next_out - _out, _out);*/
        state.flags ? crc32_1(state.check, output, _out, strm.next_out - _out) : adler32_1(state.check, output, _out, strm.next_out - _out);
      }
      strm.data_type = state.bits + (state.last ? 64 : 0) + (state.mode === TYPE ? 128 : 0) + (state.mode === LEN_ || state.mode === COPY_ ? 256 : 0);
      if ((_in === 0 && _out === 0 || flush === Z_FINISH$1) && ret === Z_OK$1) {
        ret = Z_BUF_ERROR;
      }
      return ret;
    };
    const inflateEnd = strm => {
      if (inflateStateCheck(strm)) {
        return Z_STREAM_ERROR$1;
      }
      let state = strm.state;
      if (state.window) {
        state.window = null;
      }
      strm.state = null;
      return Z_OK$1;
    };
    const inflateGetHeader = (strm, head) => {
      /* check state */
      if (inflateStateCheck(strm)) {
        return Z_STREAM_ERROR$1;
      }
      const state = strm.state;
      if ((state.wrap & 2) === 0) {
        return Z_STREAM_ERROR$1;
      }

      /* save header structure */
      state.head = head;
      head.done = false;
      return Z_OK$1;
    };
    const inflateSetDictionary = (strm, dictionary) => {
      const dictLength = dictionary.length;
      let state;
      let dictid;
      let ret;

      /* check state */
      if (inflateStateCheck(strm)) {
        return Z_STREAM_ERROR$1;
      }
      state = strm.state;
      if (state.wrap !== 0 && state.mode !== DICT) {
        return Z_STREAM_ERROR$1;
      }

      /* check for correct dictionary identifier */
      if (state.mode === DICT) {
        dictid = 1; /* adler32(0, null, 0)*/
        /* dictid = adler32(dictid, dictionary, dictLength); */
        dictid = adler32_1(dictid, dictionary, dictLength, 0);
        if (dictid !== state.check) {
          return Z_DATA_ERROR$1;
        }
      }
      /* copy dictionary to window using updatewindow(), which will amend the
       existing dictionary if appropriate */
      ret = updatewindow(strm, dictionary, dictLength, dictLength);
      if (ret) {
        state.mode = MEM;
        return Z_MEM_ERROR$1;
      }
      state.havedict = 1;
      // Tracev((stderr, "inflate:   dictionary set\n"));
      return Z_OK$1;
    };
    var inflateReset_1 = inflateReset;
    var inflateReset2_1 = inflateReset2;
    var inflateResetKeep_1 = inflateResetKeep;
    var inflateInit_1 = inflateInit;
    var inflateInit2_1 = inflateInit2;
    var inflate_2$1 = inflate$2;
    var inflateEnd_1 = inflateEnd;
    var inflateGetHeader_1 = inflateGetHeader;
    var inflateSetDictionary_1 = inflateSetDictionary;
    var inflateInfo = 'pako inflate (from Nodeca project)';

    /* Not implemented
    module.exports.inflateCodesUsed = inflateCodesUsed;
    module.exports.inflateCopy = inflateCopy;
    module.exports.inflateGetDictionary = inflateGetDictionary;
    module.exports.inflateMark = inflateMark;
    module.exports.inflatePrime = inflatePrime;
    module.exports.inflateSync = inflateSync;
    module.exports.inflateSyncPoint = inflateSyncPoint;
    module.exports.inflateUndermine = inflateUndermine;
    module.exports.inflateValidate = inflateValidate;
    */

    var inflate_1$2 = {
      inflateReset: inflateReset_1,
      inflateReset2: inflateReset2_1,
      inflateResetKeep: inflateResetKeep_1,
      inflateInit: inflateInit_1,
      inflateInit2: inflateInit2_1,
      inflate: inflate_2$1,
      inflateEnd: inflateEnd_1,
      inflateGetHeader: inflateGetHeader_1,
      inflateSetDictionary: inflateSetDictionary_1,
      inflateInfo: inflateInfo
    };

    // (C) 1995-2013 Jean-loup Gailly and Mark Adler
    // (C) 2014-2017 Vitaly Puzrin and Andrey Tupitsin
    //
    // This software is provided 'as-is', without any express or implied
    // warranty. In no event will the authors be held liable for any damages
    // arising from the use of this software.
    //
    // Permission is granted to anyone to use this software for any purpose,
    // including commercial applications, and to alter it and redistribute it
    // freely, subject to the following restrictions:
    //
    // 1. The origin of this software must not be misrepresented; you must not
    //   claim that you wrote the original software. If you use this software
    //   in a product, an acknowledgment in the product documentation would be
    //   appreciated but is not required.
    // 2. Altered source versions must be plainly marked as such, and must not be
    //   misrepresented as being the original software.
    // 3. This notice may not be removed or altered from any source distribution.

    function GZheader() {
      /* true if compressed data believed to be text */
      this.text = 0;
      /* modification time */
      this.time = 0;
      /* extra flags (not used when writing a gzip file) */
      this.xflags = 0;
      /* operating system */
      this.os = 0;
      /* pointer to extra field or Z_NULL if none */
      this.extra = null;
      /* extra field length (valid if extra != Z_NULL) */
      this.extra_len = 0; // Actually, we don't need it in JS,
      // but leave for few code modifications

      //
      // Setup limits is not necessary because in js we should not preallocate memory
      // for inflate use constant limit in 65536 bytes
      //

      /* space at extra (only when reading header) */
      // this.extra_max  = 0;
      /* pointer to zero-terminated file name or Z_NULL */
      this.name = '';
      /* space at name (only when reading header) */
      // this.name_max   = 0;
      /* pointer to zero-terminated comment or Z_NULL */
      this.comment = '';
      /* space at comment (only when reading header) */
      // this.comm_max   = 0;
      /* true if there was or will be a header crc */
      this.hcrc = 0;
      /* true when done reading gzip header (not used when writing a gzip file) */
      this.done = false;
    }
    var gzheader = GZheader;
    const toString = Object.prototype.toString;

    /* Public constants ==========================================================*/
    /* ===========================================================================*/

    const {
      Z_NO_FLUSH,
      Z_FINISH,
      Z_OK,
      Z_STREAM_END,
      Z_NEED_DICT,
      Z_STREAM_ERROR,
      Z_DATA_ERROR,
      Z_MEM_ERROR
    } = constants$2;

    /* ===========================================================================*/

    /**
     * class Inflate
     *
     * Generic JS-style wrapper for zlib calls. If you don't need
     * streaming behaviour - use more simple functions: [[inflate]]
     * and [[inflateRaw]].
     **/

    /* internal
     * inflate.chunks -> Array
     *
     * Chunks of output data, if [[Inflate#onData]] not overridden.
     **/

    /**
     * Inflate.result -> Uint8Array|String
     *
     * Uncompressed result, generated by default [[Inflate#onData]]
     * and [[Inflate#onEnd]] handlers. Filled after you push last chunk
     * (call [[Inflate#push]] with `Z_FINISH` / `true` param).
     **/

    /**
     * Inflate.err -> Number
     *
     * Error code after inflate finished. 0 (Z_OK) on success.
     * Should be checked if broken data possible.
     **/

    /**
     * Inflate.msg -> String
     *
     * Error message, if [[Inflate.err]] != 0
     **/

    /**
     * new Inflate(options)
     * - options (Object): zlib inflate options.
     *
     * Creates new inflator instance with specified params. Throws exception
     * on bad params. Supported options:
     *
     * - `windowBits`
     * - `dictionary`
     *
     * [http://zlib.net/manual.html#Advanced](http://zlib.net/manual.html#Advanced)
     * for more information on these.
     *
     * Additional options, for internal needs:
     *
     * - `chunkSize` - size of generated data chunks (16K by default)
     * - `raw` (Boolean) - do raw inflate
     * - `to` (String) - if equal to 'string', then result will be converted
     *   from utf8 to utf16 (javascript) string. When string output requested,
     *   chunk length can differ from `chunkSize`, depending on content.
     *
     * By default, when no options set, autodetect deflate/gzip data format via
     * wrapper header.
     *
     * ##### Example:
     *
     * ```javascript
     * const pako = require('pako')
     * const chunk1 = new Uint8Array([1,2,3,4,5,6,7,8,9])
     * const chunk2 = new Uint8Array([10,11,12,13,14,15,16,17,18,19]);
     *
     * const inflate = new pako.Inflate({ level: 3});
     *
     * inflate.push(chunk1, false);
     * inflate.push(chunk2, true);  // true -> last chunk
     *
     * if (inflate.err) { throw new Error(inflate.err); }
     *
     * console.log(inflate.result);
     * ```
     **/
    function Inflate$1(options) {
      this.options = common.assign({
        chunkSize: 1024 * 64,
        windowBits: 15,
        to: ''
      }, options || {});
      const opt = this.options;

      // Force window size for `raw` data, if not set directly,
      // because we have no header for autodetect.
      if (opt.raw && opt.windowBits >= 0 && opt.windowBits < 16) {
        opt.windowBits = -opt.windowBits;
        if (opt.windowBits === 0) {
          opt.windowBits = -15;
        }
      }

      // If `windowBits` not defined (and mode not raw) - set autodetect flag for gzip/deflate
      if (opt.windowBits >= 0 && opt.windowBits < 16 && !(options && options.windowBits)) {
        opt.windowBits += 32;
      }

      // Gzip header has no info about windows size, we can do autodetect only
      // for deflate. So, if window size not set, force it to max when gzip possible
      if (opt.windowBits > 15 && opt.windowBits < 48) {
        // bit 3 (16) -> gzipped data
        // bit 4 (32) -> autodetect gzip/deflate
        if ((opt.windowBits & 15) === 0) {
          opt.windowBits |= 15;
        }
      }
      this.err = 0; // error code, if happens (0 = Z_OK)
      this.msg = ''; // error message
      this.ended = false; // used to avoid multiple onEnd() calls
      this.chunks = []; // chunks of compressed data

      this.strm = new zstream();
      this.strm.avail_out = 0;
      let status = inflate_1$2.inflateInit2(this.strm, opt.windowBits);
      if (status !== Z_OK) {
        throw new Error(messages[status]);
      }
      this.header = new gzheader();
      inflate_1$2.inflateGetHeader(this.strm, this.header);

      // Setup dictionary
      if (opt.dictionary) {
        // Convert data if needed
        if (typeof opt.dictionary === 'string') {
          opt.dictionary = strings.string2buf(opt.dictionary);
        } else if (toString.call(opt.dictionary) === '[object ArrayBuffer]') {
          opt.dictionary = new Uint8Array(opt.dictionary);
        }
        if (opt.raw) {
          //In raw mode we need to set the dictionary early
          status = inflate_1$2.inflateSetDictionary(this.strm, opt.dictionary);
          if (status !== Z_OK) {
            throw new Error(messages[status]);
          }
        }
      }
    }

    /**
     * Inflate#push(data[, flush_mode]) -> Boolean
     * - data (Uint8Array|ArrayBuffer): input data
     * - flush_mode (Number|Boolean): 0..6 for corresponding Z_NO_FLUSH..Z_TREE
     *   flush modes. See constants. Skipped or `false` means Z_NO_FLUSH,
     *   `true` means Z_FINISH.
     *
     * Sends input data to inflate pipe, generating [[Inflate#onData]] calls with
     * new output chunks. Returns `true` on success. If end of stream detected,
     * [[Inflate#onEnd]] will be called.
     *
     * `flush_mode` is not needed for normal operation, because end of stream
     * detected automatically. You may try to use it for advanced things, but
     * this functionality was not tested.
     *
     * On fail call [[Inflate#onEnd]] with error code and return false.
     *
     * ##### Example
     *
     * ```javascript
     * push(chunk, false); // push one of data chunks
     * ...
     * push(chunk, true);  // push last chunk
     * ```
     **/
    Inflate$1.prototype.push = function (data, flush_mode) {
      const strm = this.strm;
      const chunkSize = this.options.chunkSize;
      const dictionary = this.options.dictionary;
      let status, _flush_mode, last_avail_out;
      if (this.ended) return false;
      if (flush_mode === ~~flush_mode) _flush_mode = flush_mode;else _flush_mode = flush_mode === true ? Z_FINISH : Z_NO_FLUSH;

      // Convert data if needed
      if (toString.call(data) === '[object ArrayBuffer]') {
        strm.input = new Uint8Array(data);
      } else {
        strm.input = data;
      }
      strm.next_in = 0;
      strm.avail_in = strm.input.length;
      for (;;) {
        if (strm.avail_out === 0) {
          strm.output = new Uint8Array(chunkSize);
          strm.next_out = 0;
          strm.avail_out = chunkSize;
        }
        status = inflate_1$2.inflate(strm, _flush_mode);
        if (status === Z_NEED_DICT && dictionary) {
          status = inflate_1$2.inflateSetDictionary(strm, dictionary);
          if (status === Z_OK) {
            status = inflate_1$2.inflate(strm, _flush_mode);
          } else if (status === Z_DATA_ERROR) {
            // Replace code with more verbose
            status = Z_NEED_DICT;
          }
        }

        // Skip snyc markers if more data follows and not raw mode
        while (strm.avail_in > 0 && status === Z_STREAM_END && strm.state.wrap > 0 && data[strm.next_in] !== 0) {
          inflate_1$2.inflateReset(strm);
          status = inflate_1$2.inflate(strm, _flush_mode);
        }
        switch (status) {
          case Z_STREAM_ERROR:
          case Z_DATA_ERROR:
          case Z_NEED_DICT:
          case Z_MEM_ERROR:
            this.onEnd(status);
            this.ended = true;
            return false;
        }

        // Remember real `avail_out` value, because we may patch out buffer content
        // to align utf8 strings boundaries.
        last_avail_out = strm.avail_out;
        if (strm.next_out) {
          if (strm.avail_out === 0 || status === Z_STREAM_END) {
            if (this.options.to === 'string') {
              let next_out_utf8 = strings.utf8border(strm.output, strm.next_out);
              let tail = strm.next_out - next_out_utf8;
              let utf8str = strings.buf2string(strm.output, next_out_utf8);

              // move tail & realign counters
              strm.next_out = tail;
              strm.avail_out = chunkSize - tail;
              if (tail) strm.output.set(strm.output.subarray(next_out_utf8, next_out_utf8 + tail), 0);
              this.onData(utf8str);
            } else {
              this.onData(strm.output.length === strm.next_out ? strm.output : strm.output.subarray(0, strm.next_out));
            }
          }
        }

        // Must repeat iteration if out buffer is full
        if (status === Z_OK && last_avail_out === 0) continue;

        // Finalize if end of stream reached.
        if (status === Z_STREAM_END) {
          status = inflate_1$2.inflateEnd(this.strm);
          this.onEnd(status);
          this.ended = true;
          return true;
        }
        if (strm.avail_in === 0) break;
      }
      return true;
    };

    /**
     * Inflate#onData(chunk) -> Void
     * - chunk (Uint8Array|String): output data. When string output requested,
     *   each chunk will be string.
     *
     * By default, stores data blocks in `chunks[]` property and glue
     * those in `onEnd`. Override this handler, if you need another behaviour.
     **/
    Inflate$1.prototype.onData = function (chunk) {
      this.chunks.push(chunk);
    };

    /**
     * Inflate#onEnd(status) -> Void
     * - status (Number): inflate status. 0 (Z_OK) on success,
     *   other if not.
     *
     * Called either after you tell inflate that the input stream is
     * complete (Z_FINISH). By default - join collected chunks,
     * free memory and fill `results` / `err` properties.
     **/
    Inflate$1.prototype.onEnd = function (status) {
      // On success - join
      if (status === Z_OK) {
        if (this.options.to === 'string') {
          this.result = this.chunks.join('');
        } else {
          this.result = common.flattenChunks(this.chunks);
        }
      }
      this.chunks = [];
      this.err = status;
      this.msg = this.strm.msg;
    };

    /**
     * inflate(data[, options]) -> Uint8Array|String
     * - data (Uint8Array|ArrayBuffer): input data to decompress.
     * - options (Object): zlib inflate options.
     *
     * Decompress `data` with inflate/ungzip and `options`. Autodetect
     * format via wrapper header by default. That's why we don't provide
     * separate `ungzip` method.
     *
     * Supported options are:
     *
     * - windowBits
     *
     * [http://zlib.net/manual.html#Advanced](http://zlib.net/manual.html#Advanced)
     * for more information.
     *
     * Sugar (options):
     *
     * - `raw` (Boolean) - say that we work with raw stream, if you don't wish to specify
     *   negative windowBits implicitly.
     * - `to` (String) - if equal to 'string', then result will be converted
     *   from utf8 to utf16 (javascript) string. When string output requested,
     *   chunk length can differ from `chunkSize`, depending on content.
     *
     *
     * ##### Example:
     *
     * ```javascript
     * const pako = require('pako');
     * const input = pako.deflate(new Uint8Array([1,2,3,4,5,6,7,8,9]));
     * let output;
     *
     * try {
     *   output = pako.inflate(input);
     * } catch (err) {
     *   console.log(err);
     * }
     * ```
     **/
    function inflate$1(input, options) {
      const inflator = new Inflate$1(options);
      inflator.push(input);

      // That will never happens, if you don't cheat with options :)
      if (inflator.err) throw inflator.msg || messages[inflator.err];
      return inflator.result;
    }
    var ungzip$1 = inflate$1;
    var inflate_1$1 = {
      ungzip: ungzip$1};
    const {
      ungzip
    } = inflate_1$1;
    var ungzip_1 = ungzip;

    function ungzipStream(file) {
      return new ReadableStream({
        start(controller) {
          void file.arrayBuffer().then(arrayBuffer => ungzip_1(new Uint8Array(arrayBuffer))).then(arrayBuffer => {
            controller.enqueue(arrayBuffer);
            controller.close();
          });
        }
      });
    }

    /**
     * Some files in the fileItems may actually be gzip. This method will ungzip those files.
     * The method will actually not really ungzip the files but decompress them if you need.
     * During this process the extension .gz will be removed
     * @param fileItems
     * @param fileItem
     * @param options
     * @returns
     */
    async function fileItemUngzip(fileItem, options = {}) {
      const {
        ungzip: ungzipOptions = {},
        logger
      } = options;
      let {
        gzipExtensions = ['gz']
      } = ungzipOptions;
      gzipExtensions = gzipExtensions.map(extension => extension.toLowerCase());
      const extension = fileItem.name.replace(/^.*\./, '').toLowerCase();
      if (!gzipExtensions.includes(extension)) {
        return fileItem;
      }
      if (!(await isGzip(fileItem))) {
        if (logger) {
          logger.info(`Could not ungzip the following file: ${fileItem.relativePath}`);
        }
        return fileItem;
      }
      return {
        sourceUUID: fileItem.sourceUUID,
        parent: fileItem,
        name: fileItem.name.replace(/\.[^.]+$/, ''),
        size: fileItem.size,
        relativePath: fileItem.relativePath.replace(/\.[^.]+$/, ''),
        lastModified: fileItem.lastModified,
        text: () => {
          return fileItem.arrayBuffer().then(arrayBuffer => {
            const decoder = new TextDecoder('utf8');
            return decoder.decode(ungzip_1(new Uint8Array(arrayBuffer)));
          });
        },
        arrayBuffer: () => {
          return fileItem.arrayBuffer().then(arrayBuffer => ungzip_1(new Uint8Array(arrayBuffer)));
        },
        // @ts-expect-error feature is too new
        stream: () => {
          return ungzipStream(fileItem);
        }
      };
    }
    async function isGzip(file) {
      const buffer = await file.arrayBuffer();
      if (buffer.byteLength < 2) return false;
      const bytes = new Uint8Array(buffer);
      return bytes[0] === 0x1f && bytes[1] === 0x8b;
    }

    async function fileItemsFromZip(zipContent, sourceUUID, options = {}) {
      const jsZip = new JSZip();
      const zip = await jsZip.loadAsync(zipContent);
      const fileItems = [];
      for (const key in zip.files) {
        const entry = zip.files[key];
        if (entry.dir) continue;
        if (!shouldAddItem(entry.name, options.filter)) continue;
        const item = {
          name: entry.name.replace(/^.*\//, ''),
          sourceUUID,
          relativePath: entry.name,
          lastModified: entry.date.getTime(),
          // @ts-expect-error _data is not exposed because missing for folder   but it is really there
          size: entry._data.uncompressedSize,
          text: () => {
            return entry.async('text');
          },
          arrayBuffer: () => {
            return entry.async('arraybuffer');
          },
          stream: () => {
            return new ReadableStream({
              start(controller) {
                void entry.async('arraybuffer').then(arrayBuffer => {
                  controller.enqueue(arrayBuffer);
                  controller.close();
                });
              }
            });
          }
        };
        fileItems.push(item);
      }
      return fileItems;
    }

    /**
     * Some files in the fileItems may actually be zip. This method will unzip those files.
     * The method will actually not really unzip the files but only add them in the fileItems.
     * Unzipping will only take place when you want to actually retrieve the data.
     * @param fileItems
     * @param fileItem
     * @param options
     * @returns
     */
    async function fileItemUnzip(fileItem, options = {}) {
      const {
        unzip = {},
        filter = {},
        logger
      } = options;
      let {
        zipExtensions = ['zip']
      } = unzip;
      const {
        recursive = true
      } = unzip;
      zipExtensions = zipExtensions.map(extension => extension.toLowerCase());
      const extension = fileItem.name.replace(/^.*\./, '').toLowerCase();
      if (!fileItem.sourceUUID) {
        throw new Error('fileItem.sourceUUID is not defined');
      }
      if (!zipExtensions.includes(extension)) {
        return [fileItem];
      }
      const buffer = await fileItem.arrayBuffer();
      if (!isZip(buffer)) {
        if (logger) {
          logger.info(`Could not unzip the following file: ${fileItem.relativePath}`);
        }
        return [fileItem];
      }
      const zipFileItems = await fileItemsFromZip(buffer, fileItem.sourceUUID, options);
      const fileItems = [];
      for (const zipEntry of zipFileItems) {
        zipEntry.parent = fileItem;
        zipEntry.relativePath = `${fileItem.relativePath}/${zipEntry.relativePath}`;
        zipEntry.sourceUUID = fileItem.sourceUUID;
        if (recursive) {
          // todo could improve this by using a promise.all
          // eslint-disable-next-line no-await-in-loop
          fileItems.push(...(await expandAndFilter(zipEntry, options)));
        } else if (shouldAddItem(zipEntry.relativePath, filter)) {
          fileItems.push(zipEntry);
        }
      }
      return fileItems;
    }
    function isZip(buffer) {
      if (buffer.byteLength < 5) return false;
      const bytes = new Uint8Array(buffer);
      return bytes[0] === 0x50 && bytes[1] === 0x4b && (bytes[2] === 0x03 || bytes[2] === 0x05 || bytes[2] === 0x07) && (bytes[3] === 0x04 || bytes[3] === 0x06 || bytes[3] === 0x08);
    }

    /**
     * Utility function that allows to expand gzip and zip files without really expanding them
     * @param fileCollection
     * @param originalFileItem
     * @param options
     * @returns
     */
    async function expandAndFilter(originalFileItem, options = {}) {
      const {
        filter = {}
      } = options;
      if (!shouldAddItem(originalFileItem.relativePath, filter)) {
        return [];
      }
      const fileItem = await fileItemUngzip(originalFileItem, options);
      const fileItems = await fileItemUnzip(fileItem, options);
      for (const fileItem of fileItems) {
        fileItem.sourceUUID = originalFileItem.sourceUUID;
      }
      return fileItems;
    }

    class FileCollection {
      files;
      sources;
      options;
      filter;
      cache;
      constructor(options = {}) {
        this.options = options;
        this.filter = options.filter || {};
        this.files = [];
        this.sources = [];
        this.cache = options.cache ?? false;
      }
      /**
       * This is unexpected to be used directly
       * @param source
       * @private
       */
      async appendExtendedSource(source) {
        if (!shouldAddItem(source.relativePath, this.filter)) return;
        this.sources.push(source);
        const sourceFile = convertExtendedSourceToFile(source);
        const files = await expandAndFilter(sourceFile, this.options);
        for (const file of files) {
          if (this.files.some(existing => existing.relativePath === file.relativePath)) {
            throw new Error(`Duplicate relativePath: ${file.relativePath}`);
          }
          if (this.cache) {
            this.files.push(new CachedFileItem(file));
          } else {
            this.files.push(file);
          }
        }
      }
      /**
       * Save an object to the collection. This method will convert typed array to normal array and will
       * replace potentially existing file with the same name.
       * @param key
       * @param value
       * @returns
       */
      set(key, value) {
        this.removeFile(key);
        const string = JSON.stringify(value, (key, value) => ArrayBuffer.isView(value) ? Array.from(value) : value);
        return this.appendText(key, string);
      }
      removeFile(originalRelativePath) {
        const {
          relativePath
        } = getNameInfo(originalRelativePath);
        const index = this.files.findIndex(f => f.relativePath === relativePath);
        let removedFile;
        if (index !== -1) {
          const sourceUUID = this.files[index].sourceUUID;
          removedFile = this.files.splice(index, 1)[0];
          // any other files with the same sourceUUID?
          if (this.files.some(f => f.sourceUUID === sourceUUID)) return;
          // delete the source
          const sourceIndex = this.sources.findIndex(s => s.uuid === sourceUUID);
          if (sourceIndex !== -1) {
            this.sources.splice(sourceIndex, 1);
          } else {
            throw new Error(`Source not found for UUID: ${sourceUUID}`);
          }
        }
        return removedFile;
      }
      async get(key) {
        const {
          relativePath
        } = getNameInfo(key);
        const file = this.files.find(f => f.relativePath === relativePath);
        if (!file) {
          throw new Error(`Key not found: ${key}`);
        }
        const string = await file.text();
        return JSON.parse(string);
      }
      appendWebSource(webSourceURL, options = {}) {
        return appendWebSource(this, webSourceURL, options);
      }
      appendFileList(fileList) {
        return appendFileList(this, fileList);
      }
      appendSource(webSource, options = {}) {
        return appendSource(this, webSource, options);
      }
      /**
       * This method can only be used from nodejs and will throw an error in the browser
       * @param path
       * @param options
       * @param options.keepBasename
       */
      appendPath(path, options = {}) {
        return appendPath();
      }
      appendText(relativePath, text, options = {}) {
        return appendText(this, relativePath, text, options);
      }
      appendArrayBuffer(relativePath, arrayBuffer, options = {}) {
        return appendArrayBuffer(this, relativePath, arrayBuffer, options);
      }
      toIum(options = {}) {
        this.alphabetical();
        return toIum(this, options);
      }
      static async fromIum(ium) {
        const fileCollection = await fromIum(ium);
        fileCollection.alphabetical();
        return fileCollection;
      }
      alphabetical() {
        this.sources.sort((a, b) => a.relativePath < b.relativePath ? -1 : 1);
        this.files.sort((a, b) => a.relativePath < b.relativePath ? -1 : 1);
      }
      [Symbol.iterator]() {
        return this.files.values();
      }
    }

    exports.FileCollection = FileCollection;

}));
//# sourceMappingURL=file-collection.js.map
