/*
 * check-cas-number v2.0.0
 * Check the validity of a CAS number (RN)
 * https://github.com/cheminfo/check-cas-number#readme
 *
 * Licensed under the MIT license.
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.CheckCasNumber = {}));
})(this, (function (exports) { 'use strict';

    /**
     * Checks if a CAS number is valid.
     * A CAS number has the format XXXXXXXX-XX-X where X is a digit. The string can only contain the CAS number.
     * @param cas
     * @param options
     * @returns
     */
    function check(cas, options = {}) {
      const {
        logger
      } = options;
      if (!cas?.match(/[0-9]{2,8}-[0-9]{2}-[0-9]/)) {
        logger?.error(`Invalid CAS number: ${cas}, the format does not match: XXXXXXXX-XX-X.`);
        return false;
      }
      let sum = 0;
      const digits = cas.replaceAll('-', '');
      for (let i = digits.length - 2; i >= 0; i--) {
        // @ts-expect-error we are sure that digits[i] exists
        sum = sum + Number.parseInt(digits[i], 10) * (digits.length - i - 1);
      }
      // is it larger than 50000 ?
      if (Number.parseInt(cas.replaceAll('-', ''), 10) < 50000) {
        logger?.error(`Invalid CAS number: ${cas}, the number is smaller than 50-00-0.`);
        return false;
      }
      // does the checkdigit match ?
      const checkDigit = sum % 10;
      if (checkDigit !== Number.parseInt(cas.slice(-1), 10)) {
        logger?.error(`Invalid CAS number: ${cas}, the check digit does not match. Expected: ${checkDigit} but got: ${cas.slice(-1)}.`);
        return false;
      }
      return true;
    }

    /**
     * This method extracts the first CAS number from a string. A cas number is something
     * defined as matching the regex [0-9]{2,8}-[0-9]{2}-[0-9].
     * It returns the CAS number if found, or null otherwise.
     * A CAS number has the format XXXXXXXX-XX-X where X is a digit.
     * @param string
     * @param options
     * @returns
     */
    function extract(string, options = {}) {
      const cas = string.match(/[0-9]{2,8}-[0-9]{2}-[0-9]/)?.[0];
      if (!cas) return null;
      if (check(cas, options)) return cas;
      return null;
    }

    /**
     * This method extracts the first CAS number from a string. A cas number is something
     * defined as matching the regex [0-9]{2,8}-[0-9]{2}-[0-9].
     * It returns the CAS number if found, or null otherwise.
     * A CAS number has the format XXXXXXXX-XX-X where X is a digit.
     * @param string
     * @param options
     * @returns
     */
    function extractAll(string, options = {}) {
      const cas = [...string.matchAll(/[0-9]{2,8}-[0-9]{2}-[0-9]/g)];
      return cas.filter(item => check(item[0], options)).map(item => item[0]);
    }

    exports.check = check;
    exports.extract = extract;
    exports.extractAll = extractAll;

}));
//# sourceMappingURL=check-cas-number.umd.js.map
