/**
 * ml-spectra-processing - Various method to process spectra
 * @version v6.1.0
 * @link https://github.com/mljs/spectra-processing#readme
 * @license MIT
 */
(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
  typeof define === 'function' && define.amd ? define(['exports'], factory) :
  (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.SpectraProcessing = {}));
}(this, (function (exports) { 'use strict';

  /**
   * This function returns an array with absolute values
   * @param {Array<Number>} array
   * @return {Number}
   */
  function xAbsolute(array) {
    let tmpArray = array.slice();

    for (let i = 0; i < tmpArray.length; i++) {
      tmpArray[i] = Math.abs(tmpArray[i]);
    }

    return tmpArray;
  }

  const toString = Object.prototype.toString;
  function isAnyArray(object) {
    return toString.call(object).endsWith('Array]');
  }

  var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

  function createCommonjsModule(fn) {
    var module = { exports: {} };
  	return fn(module, module.exports), module.exports;
  }

  var medianQuickselect_min = createCommonjsModule(function (module) {
    (function () {
      function a(d) {
        for (var e = 0, f = d.length - 1, g = void 0, h = void 0, i = void 0, j = c(e, f); !0;) {
          if (f <= e) return d[j];
          if (f == e + 1) return d[e] > d[f] && b(d, e, f), d[j];

          for (g = c(e, f), d[g] > d[f] && b(d, g, f), d[e] > d[f] && b(d, e, f), d[g] > d[e] && b(d, g, e), b(d, g, e + 1), h = e + 1, i = f; !0;) {
            do h++; while (d[e] > d[h]);

            do i--; while (d[i] > d[e]);

            if (i < h) break;
            b(d, h, i);
          }

          b(d, e, i), i <= j && (e = h), i >= j && (f = i - 1);
        }
      }

      var b = function b(d, e, f) {
        var _ref;

        return _ref = [d[f], d[e]], d[e] = _ref[0], d[f] = _ref[1], _ref;
      },
          c = function c(d, e) {
        return ~~((d + e) / 2);
      };

      module.exports ? module.exports = a : window.median = a;
    })();
  });

  function median(input) {
    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    return medianQuickselect_min(input.slice());
  }

  /**
   * This function calculates the median after taking the reimAbsolute values of the points
   * @param {Array<Number>} array - the array that will be rotated
   * @return {Number}
   */

  function xAbsoluteMedian(array) {
    return median(xAbsolute(array));
  }

  /**
   * This function xAdd the first array by the second array or a constant value to each element of the first array
   * @param {Array<Number>} array1 - the array that will be rotated
   * @param {Array|Number} array2
   * @return {Array}
   */

  function xAdd(array1, array2) {
    let isConstant = false;
    let constant;

    if (isAnyArray(array2)) {
      if (array1.length !== array2.length) {
        throw new Error('sub: size of array1 and array2 must be identical');
      }
    } else {
      isConstant = true;
      constant = Number(array2);
    }

    let array3 = new Array(array1.length);

    if (isConstant) {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] + constant;
      }
    } else {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] + array2[i];
      }
    }

    return array3;
  }

  /**
   * This function xMultiply the first array by the second array or a constant value to each element of the first array
   * @param {Array} array1 - the array that will be rotated
   * @param {Array|Number} array2
   * @return {Float64Array}
   */

  function xMultiply(array1, array2) {
    let isConstant = false;
    let constant;

    if (isAnyArray(array2)) {
      if (array1.length !== array2.length) {
        throw new Error('sub: size of array1 and array2 must be identical');
      }
    } else {
      isConstant = true;
      constant = Number(array2);
    }

    let array3 = new Float64Array(array1.length);

    if (isConstant) {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] * constant;
      }
    } else {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] * array2[i];
      }
    }

    return array3;
  }

  function xDotProduct(A, B) {
    let g = xMultiply(A, B);
    let result = 0;

    for (let i = 0; i < A.length; i++) {
      result += g[i];
    }

    return result;
  }

  /**
   * Calculates the cross-correlation between 2 vectors
   * @param {Array<Number>} [A] - fixed array
   * @param {Array<Number>} [B] - sweeping array
   * @param {object} [options={}]
   * @param {number} [options.tau=1] - sweep increment size (in number of points, min=1, max=A.length)
   * @param {number} [options.lag=A.length - 1] - scalar lag parameter
   */

  function xCrossCorrelation(A, B, options = {}) {
    let {
      tau = 1,
      lag = A.length - 1
    } = options;
    let result = new Float64Array(1 + 2 * lag / tau);

    if (A.length === B.length) {
      let n = B.length;
      let g = new Float64Array(2 * n);
      let q = new Float64Array(2 * n);

      for (let i = 0; i < n; i++) {
        q[n + i] = B[i];
      }

      for (let i = n * 2 - (tau - 1); i > 0; i -= tau) {
        let k = 0;

        for (let j = i; j < n * 2; j++) {
          g[k] = q[j];
          k++;
        }

        let w = [];

        for (let l = 0; l < n; l++) {
          w[l] = g[l];
        }

        result[(k - (n - lag)) / tau] = xDotProduct(A, w);
      }
    }

    return result;
  }

  /**
   * Calculates the auto-correlation of a vector
   * @param {Array<Number>} [A] - the array that will be fixed
   * @param {object} [options={}]
   * @param {number} [options.tau=1] - sweep increment size (in number of points, min=1, max=A.length)
   * @param {number} [options.lag=A.length - 1] - scalar lag parameter
   */

  function xAutoCorrelation(A, options = {}) {
    return xCrossCorrelation(A, A, options);
  }

  /**
   * This function xSubtract the first array by the second array or a constant value from each element of the first array
   * @param {Array<Number>} array1 - the array that will be rotated
   * @return {object}
   */
  function xBoxPlot(array) {
    array = array.slice(0).sort((a, b) => a - b);

    if (array.length < 5) {
      throw Error('xBoxPlot: can not calculate info if array contains less than 3 elements');
    }

    let info = {
      Q1: 0.0,
      Q2: 0.0,
      Q3: 0.0,
      min: array[0],
      max: array[array.length - 1]
    };
    let q1max, q3min;

    if (array.length % 2 === 1) {
      // odd
      let middle = (array.length - 1) / 2;
      info.Q2 = array[middle];
      q1max = middle - 1;
      q3min = middle + 1;
    } else {
      // even
      q3min = array.length / 2;
      q1max = q3min - 1;
      info.Q2 = (array[q1max] + array[q3min]) / 2;
    }

    if (q1max % 2 === 0) {
      info.Q1 = array[q1max / 2];
      info.Q3 = array[(array.length + q3min - 1) / 2];
    } else {
      info.Q1 = (array[(q1max + 1) / 2] + array[(q1max - 1) / 2]) / 2;
      let middleOver = (array.length + q3min) / 2;
      info.Q3 = (array[middleOver] + array[middleOver - 1]) / 2;
    }

    return info;
  }

  /**

  /**
   * Calculates the correlation between 2 vectors
   * https://en.wikipedia.org/wiki/Correlation_and_dependence
   *
   * @param {Array<Number>} [A] - the array that will be rotated
   * @param {Array<Number>} [B]
   * @return {Array}
   */
  function xCorrelation(A, B) {
    let n = A.length;
    let sumA = 0;
    let sumA2 = 0;
    let sumB = 0;
    let sumB2 = 0;
    let sumAB = 0;

    for (let i = 0; i < n; i++) {
      let a = A[i];
      let b = B[i];
      sumA += a;
      sumA2 += a ** 2;
      sumB += b;
      sumB2 += b ** 2;
      sumAB += a * b;
    }

    return (n * sumAB - sumA * sumB) / (Math.sqrt(n * sumA2 - sumA ** 2) * Math.sqrt(n * sumB2 - sumB ** 2));
  }

  /**
   * Calculate a new array of the same size that is the cumulative values
   * @param {Array<number>} isArray
   * @returns {Array<number}
   */

  function xCumulative(array) {
    if (!isAnyArray(array)) {
      throw new TypeError('input must be an array');
    }

    let newArray = new Float64Array(array.length);
    if (array.length < 1) return newArray;
    newArray[0] = array[0];

    for (let i = 1; i < array.length; i++) {
      newArray[i] = newArray[i - 1] + array[i];
    }

    return newArray;
  }

  /**
   * This function divide the first array by the second array or a constant value to each element of the first array
   * @param {Array<Number>} array1 - the array that will be rotated
   * @param {Array<Number>|Number} array2
   * @return {Array}
   */

  function xDivide(array1, array2) {
    let isConstant = false;
    let constant;

    if (isAnyArray(array2)) {
      if (array1.length !== array2.length) {
        throw new Error('sub: size of array1 and array2 must be identical');
      }
    } else {
      isConstant = true;
      constant = Number(array2);
    }

    let array3 = new Array(array1.length);

    if (isConstant) {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] / constant;
      }
    } else {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] / array2[i];
      }
    }

    return array3;
  }

  /**
   * Returns the closest index of a `target` in an ordered array
   * @param {array<Number>} array
   * @param {number} target
   */
  function xFindClosestIndex(array, target) {
    let low = 0;
    let high = array.length - 1;
    let middle = 0;

    while (high - low > 1) {
      middle = low + (high - low >> 1);

      if (array[middle] < target) {
        low = middle;
      } else if (array[middle] > target) {
        high = middle;
      } else {
        return middle;
      }
    }

    if (low < array.length - 1) {
      if (Math.abs(target - array[low]) < Math.abs(array[low + 1] - target)) {
        return low;
      } else {
        return low + 1;
      }
    } else {
      return low;
    }
  }

  /**
   * Returns an object with {fromIndex, toIndex} for a specific from / to
   * @param {array} x
   * @param {object} [options={}]
   * @param {number} [options.from] - First value for xyIntegration in the X scale
   * @param {number} [options.fromIndex=0] - First point for xyIntegration
   * @param {number} [options.to] - Last value for xyIntegration in the X scale
   * @param {number} [options.toIndex=x.length-1] - Last point for xyIntegration
   */

  function xGetFromToIndex(x, options = {}) {
    let {
      fromIndex,
      toIndex,
      from,
      to
    } = options;

    if (fromIndex === undefined) {
      if (from !== undefined) {
        fromIndex = xFindClosestIndex(x, from);
      } else {
        fromIndex = 0;
      }
    }

    if (toIndex === undefined) {
      if (to !== undefined) {
        toIndex = xFindClosestIndex(x, to);
      } else {
        toIndex = x.length - 1;
      }
    }

    if (fromIndex > toIndex) [fromIndex, toIndex] = [toIndex, fromIndex];
    return {
      fromIndex,
      toIndex
    };
  }

  /**
   *  Returns the targetIndex
   * @param {array} [x]
   * @param {object} [options={}]
   * @param {number} [options.target]
   * @param {number} [options.targetIndex=0]
   * @param {number}
   */

  function xGetTargetIndex(x, options = {}) {
    let {
      target,
      targetIndex
    } = options;

    if (targetIndex === undefined) {
      if (target !== undefined) {
        return xFindClosestIndex(x, target);
      } else {
        return 0;
      }
    }

    return targetIndex;
  }

  function _typeof(obj) {
    "@babel/helpers - typeof";

    if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") {
      _typeof = function (obj) {
        return typeof obj;
      };
    } else {
      _typeof = function (obj) {
        return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj;
      };
    }

    return _typeof(obj);
  }
  /**
   * Fill an array with sequential numbers
   * @param {Array<number>} [input] - optional destination array (if not provided a new array will be created)
   * @param {object} [options={}]
   * @param {number} [options.from=0] - first value in the array
   * @param {number} [options.to=10] - last value in the array
   * @param {number} [options.size=input.length] - size of the array (if not provided calculated from step)
   * @param {number} [options.step] - if not provided calculated from size
   * @return {Array<number>}
   */


  function sequentialFill() {
    var input = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (_typeof(input) === 'object' && !isAnyArray(input)) {
      options = input;
      input = [];
    }

    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    var _options = options,
        _options$from = _options.from,
        from = _options$from === void 0 ? 0 : _options$from,
        _options$to = _options.to,
        to = _options$to === void 0 ? 10 : _options$to,
        _options$size = _options.size,
        size = _options$size === void 0 ? input.length : _options$size,
        step = _options.step;

    if (size !== 0 && step) {
      throw new Error('step is defined by the array size');
    }

    if (!size) {
      if (step) {
        size = Math.floor((to - from) / step) + 1;
      } else {
        size = to - from + 1;
      }
    }

    if (!step && size) {
      step = (to - from) / (size - 1);
    }

    if (Array.isArray(input)) {
      // only works with normal array
      input.length = 0;

      for (var i = 0; i < size; i++) {
        input.push(from);
        from += step;
      }
    } else {
      if (input.length !== size) {
        throw new Error('sequentialFill typed array must have the correct length');
      }

      for (var _i = 0; _i < size; _i++) {
        input[_i] = from;
        from += step;
      }
    }

    return input;
  }

  /**
   * Checks if input is valdi
   * @param {Array<number>} input

   */

  function xCheck(input) {
    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }
  }
  function xCheckLengths(array1, array2) {
    if (array1.length !== array2.length) {
      throw new TypeError('Length of array1 and array2 must be identical');
    }
  }

  /**
   * Computes the maximal value of an array of values
   * @param {Array<number>} array
   * @param {object} [options={}]
   * @param {number} [options.fromIndex=0] - First point for xyIntegration
   * @param {number} [options.toIndex=x.length-1] - Last point for xyIntegration
   * @return {number}
   */

  function xMaxValue(array, options = {}) {
    xCheck(array);
    const {
      fromIndex = 0,
      toIndex = array.length - 1
    } = options;
    let maxValue = array[fromIndex];

    for (let i = fromIndex + 1; i <= toIndex; i++) {
      if (array[i] > maxValue) {
        maxValue = array[i];
      }
    }

    return maxValue;
  }

  /**
   * Computes the minimal value of an array of values
   * @param {Array<number>} array
   * @param {object} [options={}]
   * @param {number} [options.fromIndex=0] - First point for xyIntegration
   * @param {number} [options.toIndex=x.length-1] - Last point for xyIntegration
   * @return {number}
   */

  function xMinValue(array, options = {}) {
    xCheck(array);
    const {
      fromIndex = 0,
      toIndex = array.length - 1
    } = options;
    let minValue = array[fromIndex];

    for (let i = fromIndex + 1; i <= toIndex; i++) {
      if (array[i] < minValue) {
        minValue = array[i];
      }
    }

    return minValue;
  }

  /**
   * Calculates an histogram of defined number of slots
   * @param {array} [array] Array containing values
   * @param {number} [options.nbSlots=256] Number of slots
   * @param {number} [options.min=minValue] Minimum value to calculate used to calculate slot size
   * @param {number} [options.max=maxValue] Maximal value to calculate used to calculate slot size
   * @param {number} [options.logBaseX] We can first apply a log on x axis
   * @param {number} [options.logBaseY] We can apply a log on the resulting histogram
   * @param {number} [options.centerX=true] Center the X value. We will enlarge the first and
   * @param {DataXY} [options.histogram={x:[], y:[]}] Previously existing histogram to continue to fill
   * @return {DataXY} {x,y} of the histogram
   */

  function xHistogram(array, options = {}) {
    xCheck(array);
    let histogram = options.histogram;
    const {
      centerX = true,
      nbSlots = histogram === undefined ? 256 : histogram.x.length,
      logBaseX,
      logBaseY
    } = options;

    if (logBaseX) {
      array = array.slice();
      const logOfBase = Math.log10(logBaseX);

      for (let i = 0; i < array.length; i++) {
        array[i] = Math.log10(array[i]) / logOfBase;
      }
    }

    const {
      min = xMinValue(array),
      max = xMaxValue(array)
    } = options;
    const slotSize = (max - min) / (nbSlots + Number.EPSILON);
    const y = histogram === undefined ? new Uint32Array(nbSlots) : histogram.y;
    const x = histogram === undefined ? sequentialFill({
      from: min + (centerX ? slotSize / 2 : 0),
      to: max - (centerX ? slotSize / 2 : 0),
      size: nbSlots
    }) : histogram.x;

    for (let i = 0; i < array.length; i++) {
      const index = Math.max(Math.min((array[i] - min - Number.EPSILON) / slotSize >> 0, nbSlots - 1), 0);
      y[index]++;
    }

    if (logBaseY) {
      const logOfBase = Math.log10(logBaseY);

      for (let i = 0; i < y.length; i++) {
        y[i] = Math.log10(y[i]) / logOfBase;
      }
    }

    return {
      x,
      y
    };
  }

  /**
   * Returns true if x is monotone
   * @param {Array} array
   * @return {boolean}
   */
  function xIsMonotone(array) {
    if (array.length <= 2) {
      return true;
    }

    if (array[0] === array[1]) {
      // maybe a constant series
      for (let i = 1; i < array.length - 1; i++) {
        if (array[i] !== array[i + 1]) return false;
      }

      return true;
    }

    if (array[0] < array[array.length - 1]) {
      for (let i = 0; i < array.length - 1; i++) {
        if (array[i] >= array[i + 1]) return false;
      }
    } else {
      for (let i = 0; i < array.length - 1; i++) {
        if (array[i] <= array[i + 1]) return false;
      }
    }

    return true;
  }

  /**
   * Computes the index of the maximum of the given values
   * @param {Array<number>} array
   * @return {number}
   */

  function xMaxIndex(array) {
    xCheck(array);
    let maxIndex = 0;

    for (let i = 1; i < array.length; i++) {
      if (array[i] > array[maxIndex]) {
        maxIndex = i;
      }
    }

    return maxIndex;
  }

  /**
   * Computes the maximal value of an array of values
   * @param {Array<number>} array
   * @param {object} [options={}]
   * @param {number} [options.fromIndex=0] - First point for xyIntegration
   * @param {number} [options.toIndex=x.length-1] - Last point for xyIntegration
   * @return {number}
   */

  function xMean(array, options = {}) {
    xCheck(array);
    const {
      fromIndex = 0,
      toIndex = array.length - 1
    } = options;
    let sumValue = array[fromIndex];

    for (let i = fromIndex + 1; i <= toIndex; i++) {
      sumValue += array[i];
    }

    return sumValue / (toIndex - fromIndex + 1);
  }

  /**
   * Computes the index of the minimum of the given values
   * @param {Array<number>} array
   * @return {number}
   */

  function xMinIndex(array) {
    xCheck(array);
    let minIndex = 0;

    for (let i = 1; i < array.length; i++) {
      if (array[i] < array[minIndex]) {
        minIndex = i;
      }
    }

    return minIndex;
  }

  var d3Array = createCommonjsModule(function (module, exports) {
    (function (global, factory) {
      factory(exports) ;
    })(commonjsGlobal, function (exports) {

      function ascending(a, b) {
        return a < b ? -1 : a > b ? 1 : a >= b ? 0 : NaN;
      }

      function bisector(compare) {
        if (compare.length === 1) compare = ascendingComparator(compare);
        return {
          left: function (a, x, lo, hi) {
            if (lo == null) lo = 0;
            if (hi == null) hi = a.length;

            while (lo < hi) {
              var mid = lo + hi >>> 1;
              if (compare(a[mid], x) < 0) lo = mid + 1;else hi = mid;
            }

            return lo;
          },
          right: function (a, x, lo, hi) {
            if (lo == null) lo = 0;
            if (hi == null) hi = a.length;

            while (lo < hi) {
              var mid = lo + hi >>> 1;
              if (compare(a[mid], x) > 0) hi = mid;else lo = mid + 1;
            }

            return lo;
          }
        };
      }

      function ascendingComparator(f) {
        return function (d, x) {
          return ascending(f(d), x);
        };
      }

      var ascendingBisect = bisector(ascending);
      var bisectRight = ascendingBisect.right;
      var bisectLeft = ascendingBisect.left;

      function descending(a, b) {
        return b < a ? -1 : b > a ? 1 : b >= a ? 0 : NaN;
      }

      function number$1(x) {
        return x === null ? NaN : +x;
      }

      function variance(array, f) {
        var n = array.length,
            m = 0,
            a,
            d,
            s = 0,
            i = -1,
            j = 0;

        if (f == null) {
          while (++i < n) {
            if (!isNaN(a = number$1(array[i]))) {
              d = a - m;
              m += d / ++j;
              s += d * (a - m);
            }
          }
        } else {
          while (++i < n) {
            if (!isNaN(a = number$1(f(array[i], i, array)))) {
              d = a - m;
              m += d / ++j;
              s += d * (a - m);
            }
          }
        }

        if (j > 1) return s / (j - 1);
      }

      function deviation(array, f) {
        var v = variance(array, f);
        return v ? Math.sqrt(v) : v;
      }

      function extent(array, f) {
        var i = -1,
            n = array.length,
            a,
            b,
            c;

        if (f == null) {
          while (++i < n) if ((b = array[i]) != null && b >= b) {
            a = c = b;
            break;
          }

          while (++i < n) if ((b = array[i]) != null) {
            if (a > b) a = b;
            if (c < b) c = b;
          }
        } else {
          while (++i < n) if ((b = f(array[i], i, array)) != null && b >= b) {
            a = c = b;
            break;
          }

          while (++i < n) if ((b = f(array[i], i, array)) != null) {
            if (a > b) a = b;
            if (c < b) c = b;
          }
        }

        return [a, c];
      }

      function constant(x) {
        return function () {
          return x;
        };
      }

      function identity(x) {
        return x;
      }

      function range(start, stop, step) {
        start = +start, stop = +stop, step = (n = arguments.length) < 2 ? (stop = start, start = 0, 1) : n < 3 ? 1 : +step;
        var i = -1,
            n = Math.max(0, Math.ceil((stop - start) / step)) | 0,
            range = new Array(n);

        while (++i < n) {
          range[i] = start + i * step;
        }

        return range;
      }

      var e10 = Math.sqrt(50);
      var e5 = Math.sqrt(10);
      var e2 = Math.sqrt(2);

      function ticks(start, stop, count) {
        var step = tickStep(start, stop, count);
        return range(Math.ceil(start / step) * step, Math.floor(stop / step) * step + step / 2, // inclusive
        step);
      }

      function tickStep(start, stop, count) {
        var step0 = Math.abs(stop - start) / Math.max(0, count),
            step1 = Math.pow(10, Math.floor(Math.log(step0) / Math.LN10)),
            error = step0 / step1;
        if (error >= e10) step1 *= 10;else if (error >= e5) step1 *= 5;else if (error >= e2) step1 *= 2;
        return stop < start ? -step1 : step1;
      }

      function sturges(values) {
        return Math.ceil(Math.log(values.length) / Math.LN2) + 1;
      }

      function number(x) {
        return +x;
      }

      function histogram() {
        var value = identity,
            domain = extent,
            threshold = sturges;

        function histogram(data) {
          var i,
              n = data.length,
              x,
              values = new Array(n); // Coerce values to numbers.

          for (i = 0; i < n; ++i) {
            values[i] = +value(data[i], i, data);
          }

          var xz = domain(values),
              x0 = +xz[0],
              x1 = +xz[1],
              tz = threshold(values, x0, x1); // Convert number of thresholds into uniform thresholds.

          if (!Array.isArray(tz)) tz = ticks(x0, x1, +tz); // Coerce thresholds to numbers, ignoring any outside the domain.

          var m = tz.length;

          for (i = 0; i < m; ++i) tz[i] = +tz[i];

          while (tz[0] <= x0) tz.shift(), --m;

          while (tz[m - 1] >= x1) tz.pop(), --m;

          var bins = new Array(m + 1),
              bin; // Initialize bins.

          for (i = 0; i <= m; ++i) {
            bin = bins[i] = [];
            bin.x0 = i > 0 ? tz[i - 1] : x0;
            bin.x1 = i < m ? tz[i] : x1;
          } // Assign data to bins by value, ignoring any outside the domain.


          for (i = 0; i < n; ++i) {
            x = values[i];

            if (x0 <= x && x <= x1) {
              bins[bisectRight(tz, x, 0, m)].push(data[i]);
            }
          }

          return bins;
        }

        histogram.value = function (_) {
          return arguments.length ? (value = typeof _ === "function" ? _ : constant(+_), histogram) : value;
        };

        histogram.domain = function (_) {
          return arguments.length ? (domain = typeof _ === "function" ? _ : constant([+_[0], +_[1]]), histogram) : domain;
        };

        histogram.thresholds = function (_) {
          if (!arguments.length) return threshold;
          threshold = typeof _ === "function" ? _ : Array.isArray(_) ? constant(Array.prototype.map.call(_, number)) : constant(+_);
          return histogram;
        };

        return histogram;
      }

      function quantile(array, p, f) {
        if (f == null) f = number$1;
        if (!(n = array.length)) return;
        if ((p = +p) <= 0 || n < 2) return +f(array[0], 0, array);
        if (p >= 1) return +f(array[n - 1], n - 1, array);
        var n,
            h = (n - 1) * p,
            i = Math.floor(h),
            a = +f(array[i], i, array),
            b = +f(array[i + 1], i + 1, array);
        return a + (b - a) * (h - i);
      }

      function freedmanDiaconis(values, min, max) {
        values.sort(ascending);
        return Math.ceil((max - min) / (2 * (quantile(values, 0.75) - quantile(values, 0.25)) * Math.pow(values.length, -1 / 3)));
      }

      function scott(values, min, max) {
        return Math.ceil((max - min) / (3.5 * deviation(values) * Math.pow(values.length, -1 / 3)));
      }

      function max(array, f) {
        var i = -1,
            n = array.length,
            a,
            b;

        if (f == null) {
          while (++i < n) if ((b = array[i]) != null && b >= b) {
            a = b;
            break;
          }

          while (++i < n) if ((b = array[i]) != null && b > a) a = b;
        } else {
          while (++i < n) if ((b = f(array[i], i, array)) != null && b >= b) {
            a = b;
            break;
          }

          while (++i < n) if ((b = f(array[i], i, array)) != null && b > a) a = b;
        }

        return a;
      }

      function mean(array, f) {
        var s = 0,
            n = array.length,
            a,
            i = -1,
            j = n;

        if (f == null) {
          while (++i < n) if (!isNaN(a = number$1(array[i]))) s += a;else --j;
        } else {
          while (++i < n) if (!isNaN(a = number$1(f(array[i], i, array)))) s += a;else --j;
        }

        if (j) return s / j;
      }

      function median(array, f) {
        var numbers = [],
            n = array.length,
            a,
            i = -1;

        if (f == null) {
          while (++i < n) if (!isNaN(a = number$1(array[i]))) numbers.push(a);
        } else {
          while (++i < n) if (!isNaN(a = number$1(f(array[i], i, array)))) numbers.push(a);
        }

        return quantile(numbers.sort(ascending), 0.5);
      }

      function merge(arrays) {
        var n = arrays.length,
            m,
            i = -1,
            j = 0,
            merged,
            array;

        while (++i < n) j += arrays[i].length;

        merged = new Array(j);

        while (--n >= 0) {
          array = arrays[n];
          m = array.length;

          while (--m >= 0) {
            merged[--j] = array[m];
          }
        }

        return merged;
      }

      function min(array, f) {
        var i = -1,
            n = array.length,
            a,
            b;

        if (f == null) {
          while (++i < n) if ((b = array[i]) != null && b >= b) {
            a = b;
            break;
          }

          while (++i < n) if ((b = array[i]) != null && a > b) a = b;
        } else {
          while (++i < n) if ((b = f(array[i], i, array)) != null && b >= b) {
            a = b;
            break;
          }

          while (++i < n) if ((b = f(array[i], i, array)) != null && a > b) a = b;
        }

        return a;
      }

      function pairs(array) {
        var i = 0,
            n = array.length - 1,
            p = array[0],
            pairs = new Array(n < 0 ? 0 : n);

        while (i < n) pairs[i] = [p, p = array[++i]];

        return pairs;
      }

      function permute(array, indexes) {
        var i = indexes.length,
            permutes = new Array(i);

        while (i--) permutes[i] = array[indexes[i]];

        return permutes;
      }

      function scan(array, compare) {
        if (!(n = array.length)) return;
        var i = 0,
            n,
            j = 0,
            xi,
            xj = array[j];
        if (!compare) compare = ascending;

        while (++i < n) if (compare(xi = array[i], xj) < 0 || compare(xj, xj) !== 0) xj = xi, j = i;

        if (compare(xj, xj) === 0) return j;
      }

      function shuffle(array, i0, i1) {
        var m = (i1 == null ? array.length : i1) - (i0 = i0 == null ? 0 : +i0),
            t,
            i;

        while (m) {
          i = Math.random() * m-- | 0;
          t = array[m + i0];
          array[m + i0] = array[i + i0];
          array[i + i0] = t;
        }

        return array;
      }

      function sum(array, f) {
        var s = 0,
            n = array.length,
            a,
            i = -1;

        if (f == null) {
          while (++i < n) if (a = +array[i]) s += a; // Note: zero and null are equivalent.

        } else {
          while (++i < n) if (a = +f(array[i], i, array)) s += a;
        }

        return s;
      }

      function transpose(matrix) {
        if (!(n = matrix.length)) return [];

        for (var i = -1, m = min(matrix, length), transpose = new Array(m); ++i < m;) {
          for (var j = -1, n, row = transpose[i] = new Array(n); ++j < n;) {
            row[j] = matrix[j][i];
          }
        }

        return transpose;
      }

      function length(d) {
        return d.length;
      }

      function zip() {
        return transpose(arguments);
      }

      var version = "0.7.1";
      exports.version = version;
      exports.bisect = bisectRight;
      exports.bisectRight = bisectRight;
      exports.bisectLeft = bisectLeft;
      exports.ascending = ascending;
      exports.bisector = bisector;
      exports.descending = descending;
      exports.deviation = deviation;
      exports.extent = extent;
      exports.histogram = histogram;
      exports.thresholdFreedmanDiaconis = freedmanDiaconis;
      exports.thresholdScott = scott;
      exports.thresholdSturges = sturges;
      exports.max = max;
      exports.mean = mean;
      exports.median = median;
      exports.merge = merge;
      exports.min = min;
      exports.pairs = pairs;
      exports.permute = permute;
      exports.quantile = quantile;
      exports.range = range;
      exports.scan = scan;
      exports.shuffle = shuffle;
      exports.sum = sum;
      exports.ticks = ticks;
      exports.tickStep = tickStep;
      exports.transpose = transpose;
      exports.variance = variance;
      exports.zip = zip;
    });
  });

  const {
    bisectRight
  } = d3Array;

  const quincunx = (u, v, w, q) => {
    const n = u.length - 1;
    u[0] = 0;
    v[0] = 0;
    w[0] = 0;
    v[1] = v[1] / u[1];
    w[1] = w[1] / u[1];

    for (let i = 2; i < n; ++i) {
      u[i] = u[i] - u[i - 2] * w[i - 2] * w[i - 2] - u[i - 1] * v[i - 1] * v[i - 1];
      v[i] = (v[i] - u[i - 1] * v[i - 1] * w[i - 1]) / u[i];
      w[i] = w[i] / u[i];
    }

    for (let i = 2; i < n; ++i) {
      q[i] = q[i] - v[i - 1] * q[i - 1] - w[i - 2] * q[i - 2];
    }

    for (let i = 1; i < n; ++i) {
      q[i] = q[i] / u[i];
    }

    q[n - 2] = q[n - 2] - v[n - 2] * q[n - 1];

    for (let i = n - 3; i > 0; --i) {
      q[i] = q[i] - v[i] * q[i + 1] - w[i] * q[i + 2];
    }
  };

  const smoothingSpline = (x, y, sigma, lambda) => {
    const n = x.length - 1;
    const h = new Array(n + 1);
    const r = new Array(n + 1);
    const f = new Array(n + 1);
    const p = new Array(n + 1);
    const q = new Array(n + 1);
    const u = new Array(n + 1);
    const v = new Array(n + 1);
    const w = new Array(n + 1);
    const params = x.map(() => [0, 0, 0, 0]);
    params.pop();
    const mu = 2 * (1 - lambda) / (3 * lambda);

    for (let i = 0; i < n; ++i) {
      h[i] = x[i + 1] - x[i];
      r[i] = 3 / h[i];
    }

    q[0] = 0;

    for (let i = 1; i < n; ++i) {
      f[i] = -(r[i - 1] + r[i]);
      p[i] = 2 * (x[i + 1] - x[i - 1]);
      q[i] = 3 * (y[i + 1] - y[i]) / h[i] - 3 * (y[i] - y[i - 1]) / h[i - 1];
    }

    q[n] = 0;

    for (let i = 1; i < n; ++i) {
      u[i] = r[i - 1] * r[i - 1] * sigma[i - 1] + f[i] * f[i] * sigma[i] + r[i] * r[i] * sigma[i + 1];
      u[i] = mu * u[i] + p[i];
    }

    for (let i = 1; i < n - 1; ++i) {
      v[i] = f[i] * r[i] * sigma[i] + r[i] * f[i + 1] * sigma[i + 1];
      v[i] = mu * v[i] + h[i];
    }

    for (let i = 1; i < n - 2; ++i) {
      w[i] = mu * r[i] * r[i + 1] * sigma[i + 1];
    }

    quincunx(u, v, w, q);
    params[0][3] = y[0] - mu * r[0] * q[1] * sigma[0];
    params[1][3] = y[1] - mu * (f[1] * q[1] + r[1] * q[2]) * sigma[0];
    params[0][0] = q[1] / (3 * h[0]);
    params[0][1] = 0;
    params[0][2] = (params[1][3] - params[0][3]) / h[0] - q[1] * h[0] / 3;
    r[0] = 0;

    for (let i = 1; i < n; ++i) {
      params[i][0] = (q[i + 1] - q[i]) / (3 * h[i]);
      params[i][1] = q[i];
      params[i][2] = (q[i] + q[i - 1]) * h[i - 1] + params[i - 1][2];
      params[i][3] = r[i - 1] * q[i - 1] + f[i] * q[i] + r[i] * q[i + 1];
      params[i][3] = y[i] - mu * params[i][3] * sigma[i];
    }

    return params;
  };

  class SplineInterpolator {
    constructor(xIn, yIn, lambda = 1) {
      const indices = xIn.map((_, i) => i);
      indices.sort((i, j) => xIn[i] - xIn[j]);
      const x = indices.map(i => xIn[i]);
      const y = indices.map(i => yIn[i]);
      const n = indices.length;
      const sigma = indices.map(() => 1);
      this.n = n;
      this.x = x;
      this.y = y;
      this.params = smoothingSpline(x, y, sigma, lambda);
    }

    interpolate(v) {
      if (v === this.x[this.n - 1]) {
        return this.y[this.n - 1];
      }

      const i = Math.min(Math.max(0, bisectRight(this.x, v) - 1), this.n - 2);
      const [a, b, c, d] = this.params[i];
      v = v - this.x[i];
      return a * v * v * v + b * v * v + c * v + d;
    }

    max(step = 100) {
      const xStart = this.x[0];
      const xStop = this.x[this.n - 1];
      const delta = (xStop - xStart) / step;
      let maxValue = -Infinity;

      for (let i = 0, x = xStart; i < step; ++i, x += delta) {
        const y = this.interpolate(x);

        if (y > maxValue) {
          maxValue = y;
        }
      }

      return maxValue;
    }

    min(step = 100) {
      const xStart = this.x[0];
      const xStop = this.x[this.n - 1];
      const delta = (xStop - xStart) / step;
      let minValue = Infinity;

      for (let i = 0, x = xStart; i < step; ++i, x += delta) {
        const y = this.interpolate(x);

        if (y < minValue) {
          minValue = y;
        }
      }

      return minValue;
    }

    domain() {
      return [this.x[0], this.x[this.x.length - 1]];
    }

    range() {
      return [this.min(), this.max()];
    }

    curve(nInterval, domain = null) {
      domain = domain || this.domain();
      const delta = (domain[1] - domain[0]) / (nInterval - 1);
      const vals = new Array(nInterval);

      for (let i = 0; i < nInterval; ++i) {
        const x = delta * i + domain[0];
        vals[i] = [x, this.interpolate(x)];
      }

      return vals;
    }

  }

  var splineInterpolator = SplineInterpolator;

  /* eslint-disable no-loss-of-precision */

  /*
  Adapted from: https://github.com/compute-io/erfcinv/blob/aa116e23883839359e310ad41a7c42f72815fc1e/lib/number.js

  The MIT License (MIT)

  Copyright (c) 2014-2015 The Compute.io Authors. All rights reserved.

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in all
  copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.


  Boost Software License - Version 1.0 - August 17th, 2003

  Permission is hereby granted, free of charge, to any person or organization obtaining a copy of the software and accompanying documentation covered by this license (the "Software") to use, reproduce, display, distribute, execute, and transmit the Software, and to prepare derivative works of the Software, and to permit third-parties to whom the Software is furnished to do so, all subject to the following:

  The copyright notices in the Software and this entire statement, including the above license grant, this restriction and the following disclaimer, must be included in all copies of the Software, in whole or in part, and all derivative works of the Software, unless such copies or derivative works are solely in the form of machine-executable object code generated by a source language processor.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
  */
  // Coefficients for erfcinv on [0, 0.5]:
  const Y1 = 8.91314744949340820313e-2;
  const P1 = [-5.38772965071242932965e-3, 8.22687874676915743155e-3, 2.19878681111168899165e-2, -3.65637971411762664006e-2, -1.26926147662974029034e-2, 3.34806625409744615033e-2, -8.36874819741736770379e-3, -5.08781949658280665617e-4];
  const Q1 = [8.86216390456424707504e-4, -2.33393759374190016776e-3, 7.95283687341571680018e-2, -5.27396382340099713954e-2, -7.1228902341542847553e-1, 6.62328840472002992063e-1, 1.56221558398423026363, -1.56574558234175846809, -9.70005043303290640362e-1, 1]; // Coefficients for erfcinv for 0.5 > 1-x >= 0:

  const Y2 = 2.249481201171875;
  const P2 = [-3.67192254707729348546, 2.11294655448340526258e1, 1.7445385985570866523e1, -4.46382324441786960818e1, -1.88510648058714251895e1, 1.76447298408374015486e1, 8.37050328343119927838, 1.05264680699391713268e-1, -2.02433508355938759655e-1];
  const Q2 = [1.72114765761200282724, -2.26436933413139721736e1, 1.08268667355460159008e1, 4.85609213108739935468e1, -2.01432634680485188801e1, -2.86608180499800029974e1, 3.9713437953343869095, 6.24264124854247537712, 1]; // Coefficients for erfcinv for sqrt( -log(1-x)):

  const Y3 = 8.07220458984375e-1;
  const P3 = [-6.81149956853776992068e-10, 2.85225331782217055858e-8, -6.79465575181126350155e-7, 2.14558995388805277169e-3, 2.90157910005329060432e-2, 1.42869534408157156766e-1, 3.37785538912035898924e-1, 3.87079738972604337464e-1, 1.17030156341995252019e-1, -1.63794047193317060787e-1, -1.31102781679951906451e-1];
  const Q3 = [1.105924229346489121e-2, 1.52264338295331783612e-1, 8.48854343457902036425e-1, 2.59301921623620271374, 4.77846592945843778382, 5.38168345707006855425, 3.46625407242567245975, 1];
  const Y4 = 9.3995571136474609375e-1;
  const P4 = [2.66339227425782031962e-12, -2.30404776911882601748e-10, 4.60469890584317994083e-6, 1.57544617424960554631e-4, 1.87123492819559223345e-3, 9.50804701325919603619e-3, 1.85573306514231072324e-2, -2.22426529213447927281e-3, -3.50353787183177984712e-2];
  const Q4 = [7.64675292302794483503e-5, 2.63861676657015992959e-3, 3.41589143670947727934e-2, 2.20091105764131249824e-1, 7.62059164553623404043e-1, 1.3653349817554063097, 1];
  const Y5 = 9.8362827301025390625e-1;
  const P5 = [9.9055709973310326855e-17, -2.81128735628831791805e-14, 4.62596163522878599135e-9, 4.49696789927706453732e-7, 1.49624783758342370182e-5, 2.09386317487588078668e-4, 1.05628862152492910091e-3, -1.12951438745580278863e-3, -1.67431005076633737133e-2];
  const Q5 = [2.82243172016108031869e-7, 2.75335474764726041141e-5, 9.64011807005165528527e-4, 1.60746087093676504695e-2, 1.38151865749083321638e-1, 5.91429344886417493481e-1, 1];

  function polyval(c, x) {
    let p = 0;

    for (const coef of c) {
      p = p * x + coef;
    }

    return p;
  }
  /**
   * Calculates a rational approximation.
   *
   * @private
   * @param {Number} x
   * @param {Number} v
   * @param {Array} P - array of polynomial coefficients
   * @param {Array} Q - array of polynomial coefficients
   * @param {Number} Y
   * @returns {Number} rational approximation
   */


  function calc(x, v, P, Q, Y) {
    const s = x - v;
    const r = polyval(P, s) / polyval(Q, s);
    return Y * x + r * x;
  }
  /**
   * Evaluates the complementary inverse error function for an input value.
   *
   * @private
   * @param {Number} x - input value
   * @returns {Number} evaluated complementary inverse error function
   */


  function erfcinv(x) {
    let sign = false;
    let val;
    let q;
    let g;
    let r; // [1] Special cases...
    // NaN:

    if (Number.isNaN(x)) {
      return NaN;
    } // x not on the interval: [0,2]


    if (x < 0 || x > 2) {
      throw new RangeError(`erfcinv()::invalid input argument. Value must be on the interval [0,2]. Value: \`${x}\`.`);
    }

    if (x === 0) {
      return Number.POSITIVE_INFINITY;
    }

    if (x === 2) {
      return Number.NEGATIVE_INFINITY;
    }

    if (x === 1) {
      return 0;
    } // [2] Get the sign and make use of `erfc` reflection formula: `erfc(-z)=2 - erfc(z)`...


    if (x > 1) {
      q = 2 - x;
      x = 1 - q;
      sign = true;
    } else {
      q = x;
      x = 1 - x;
    } // [3] |x| <= 0.5


    if (x <= 0.5) {
      g = x * (x + 10);
      r = polyval(P1, x) / polyval(Q1, x);
      val = g * Y1 + g * r;
      return sign ? -val : val;
    } // [4] 1-|x| >= 0.25


    if (q >= 0.25) {
      g = Math.sqrt(-2 * Math.log(q));
      q = q - 0.25;
      r = polyval(P2, q) / polyval(Q2, q);
      val = g / (Y2 + r);
      return sign ? -val : val;
    }

    q = Math.sqrt(-Math.log(q)); // [5] q < 3

    if (q < 3) {
      return calc(q, 1.125, P3, Q3, Y3);
    } // [6] q < 6


    if (q < 6) {
      return calc(q, 3, P4, Q4, Y4);
    } // Note that the smallest number in JavaScript is 5e-324. Math.sqrt( -Math.log( 5e-324 ) ) ~27.2844


    return calc(q, 6, P5, Q5, Y5); // Note that in the boost library, they are able to go to much smaller values, as 128 bit long doubles support ~1e-5000; something which JavaScript does not natively support.
  }

  function rayleighCdf(x, sigma = 1) {
    if (x < 0) {
      return 0;
    }

    return -Math.expm1(-Math.pow(x, 2) / (2 * Math.pow(sigma, 2)));
  }

  /**
   * Determine noise level by san plot methodology (https://doi.org/10.1002/mrc.4882)
   * @param {Array} data - real or magnitude spectra data.
   * @param {object} [options = {}]
   * @param {array} [options.mask] - boolean array to filter data, if the i-th element is true then the i-th element of the distribution will be ignored.
   * @param {number} [options.scaleFactor=1] - factor to scale the data input[i]*=scaleFactor.
   * @param {number} [options.cutOff] - percent of positive signal distribution where the noise level will be determined, if it is not defined the program calculate it.
   * @param {number} [options.factorStd=5] - factor times std to determine what will be marked as signals.
   * @param {boolean} [options.refine=true] - if true the noise level will be recalculated get out the signals using factorStd.
   * @param {boolean} [options.fixOffset=true] - If the baseline is correct, the midpoint of distribution should be zero. if true, the distribution will be centered.
   */

  function xNoiseSanPlot(data, options = {}) {
    const {
      mask,
      cutOff,
      refine = true,
      magnitudeMode = false,
      scaleFactor = 1,
      factorStd = 5,
      fixOffset = true
    } = options;
    let input;

    if (Array.isArray(mask) && mask.length === data.length) {
      input = data.filter((_e, i) => !mask[i]);
    } else {
      input = data.slice();
    }

    if (scaleFactor > 1) {
      for (let i = 0; i < input.length; i++) {
        input[i] *= scaleFactor;
      }
    }

    input.sort((a, b) => b - a);

    if (fixOffset && !magnitudeMode) {
      let medianIndex = Math.floor(input.length / 2);
      let median = 0.5 * (input[medianIndex] + input[medianIndex + 1]);

      for (let i = 0; i < input.length; i++) {
        input[i] -= median;
      }
    }

    let firstNegativeValueIndex = input.findIndex(e => e < 0);
    let lastPositiveValueIndex = firstNegativeValueIndex - 1;

    for (let i = lastPositiveValueIndex; i >= 0; i--) {
      if (input[i] > 0) {
        lastPositiveValueIndex = i;
        break;
      }
    }

    let signPositive = new Float64Array(input.slice(0, lastPositiveValueIndex + 1));
    let signNegative = new Float64Array(input.slice(firstNegativeValueIndex));
    let cutOffDist = cutOff || determineCutOff(signPositive, {
      magnitudeMode
    });
    let pIndex = Math.floor(signPositive.length * cutOffDist);
    let initialNoiseLevelPositive = signPositive[pIndex];
    let skyPoint = signPositive[0];
    let initialNoiseLevelNegative;

    if (signNegative.length > 0) {
      let nIndex = Math.floor(signNegative.length * (1 - cutOffDist));
      initialNoiseLevelNegative = -1 * signNegative[nIndex];
    } else {
      initialNoiseLevelNegative = 0;
    }

    let noiseLevelPositive = initialNoiseLevelPositive;
    let noiseLevelNegative = initialNoiseLevelNegative;
    let cloneSignPositive = signPositive.slice();
    let cloneSignNegative = signNegative.slice();
    let cutOffSignalsIndexPlus = 0;
    let cutOffSignalsIndexNeg = 2;

    if (refine) {
      let cutOffSignals = noiseLevelPositive * factorStd;
      cutOffSignalsIndexPlus = signPositive.findIndex(e => e < cutOffSignals);

      if (cutOffSignalsIndexPlus > -1) {
        cloneSignPositive = signPositive.slice(cutOffSignalsIndexPlus);
        noiseLevelPositive = cloneSignPositive[Math.floor(cloneSignPositive.length * cutOffDist)];
      }

      cutOffSignals = noiseLevelNegative * factorStd;
      cutOffSignalsIndexNeg = signNegative.findIndex(e => e < cutOffSignals);

      if (cutOffSignalsIndexNeg > -1) {
        cloneSignNegative = signNegative.slice(cutOffSignalsIndexNeg);
        noiseLevelNegative = cloneSignPositive[Math.floor(cloneSignNegative.length * (1 - cutOffDist))];
      }
    }

    let correctionFactor = -simpleNormInv(cutOffDist / 2, {
      magnitudeMode
    });
    initialNoiseLevelPositive = initialNoiseLevelPositive / correctionFactor;
    initialNoiseLevelNegative = initialNoiseLevelNegative / correctionFactor;
    let effectiveCutOffDist, refinedCorrectionFactor;

    if (refine && cutOffSignalsIndexPlus > -1) {
      effectiveCutOffDist = (cutOffDist * cloneSignPositive.length + cutOffSignalsIndexPlus) / (cloneSignPositive.length + cutOffSignalsIndexPlus);
      refinedCorrectionFactor = -1 * simpleNormInv(effectiveCutOffDist / 2, {
        magnitudeMode
      });
      noiseLevelPositive /= refinedCorrectionFactor;

      if (cutOffSignalsIndexNeg > -1) {
        effectiveCutOffDist = (cutOffDist * cloneSignNegative.length + cutOffSignalsIndexNeg) / (cloneSignNegative.length + cutOffSignalsIndexNeg);
        refinedCorrectionFactor = -1 * simpleNormInv(effectiveCutOffDist / 2, {
          magnitudeMode
        });

        if (noiseLevelNegative !== 0) {
          noiseLevelNegative /= refinedCorrectionFactor;
        }
      }
    } else {
      noiseLevelPositive /= correctionFactor;
      noiseLevelNegative /= correctionFactor;
    }

    return {
      positive: noiseLevelPositive,
      negative: noiseLevelNegative,
      snr: skyPoint / noiseLevelPositive
    };
  }

  function determineCutOff(signPositive, options = {}) {
    let {
      magnitudeMode = false,
      considerList = {
        from: 0.5,
        step: 0.1,
        to: 0.9
      }
    } = options; //generate a list of values for

    let cutOff = [];
    let indexMax = signPositive.length - 1;

    for (let i = 0.01; i <= 0.99; i += 0.01) {
      let index = Math.round(indexMax * i);
      let value = -signPositive[index] / simpleNormInv([i / 2], {
        magnitudeMode
      });
      cutOff.push([i, value]);
    }

    let minKi = Number.MAX_SAFE_INTEGER;
    let {
      from,
      to,
      step
    } = considerList;
    let delta = step / 2;
    let whereToCutStat = 0.5;

    for (let i = from; i <= to; i += step) {
      let floor = i - delta;
      let top = i + delta;
      let elementsOfCutOff = cutOff.filter(e => e[0] < top && e[0] > floor);
      let averageValue = elementsOfCutOff.reduce((a, b) => a + Math.abs(b[1]), 0);
      let kiSqrt = 0;

      for (let j = 0; j < elementsOfCutOff.length; j++) {
        kiSqrt += Math.pow(elementsOfCutOff[j][1] - averageValue, 2);
      }

      if (kiSqrt < minKi) {
        minKi = kiSqrt;
        whereToCutStat = i;
      }
    }

    return whereToCutStat;
  }

  function simpleNormInv(data, options = {}) {
    const {
      magnitudeMode = false
    } = options;
    if (!Array.isArray(data)) data = [data];
    let from = 0;
    let to = 2;
    let step = 0.01;
    let xTraining = createArray(from, to, step);
    let result = new Float64Array(data.length);
    let yTraining = new Float64Array(xTraining.length);

    if (magnitudeMode) {
      let factor = 1;

      for (let i = 0; i < yTraining.length; i++) {
        let finalInput = xTraining[i] * factor;
        yTraining[i] = 1 - rayleighCdf(finalInput);
      }

      let interp = new splineInterpolator(xTraining, yTraining);

      for (let i = 0; i < result.length; i++) {
        let yValue = 2 * data[i];
        result[i] = -1 * interp.interpolate(yValue);
      }
    } else {
      for (let i = 0; i < result.length; i++) {
        result[i] = -1 * Math.SQRT2 * erfcinv(2 * data[i]);
      }
    }

    return result.length === 1 ? result[0] : result;
  }

  function createArray(from, to, step) {
    let result = new Float32Array(Math.abs((from - to) / step + 1));

    for (let i = 0; i < result.length; i++) {
      result[i] = from + i * step;
    }

    return Array.from(result);
  }

  /**
   * This function calculate the norm of a vector
   * @example xNorm([3, 4]) -> 5
   * @param {Array} array - the array that will be rotated
   * @return {number} calculated norm
   */
  function xNorm(array) {
    let result = 0;

    for (let i = 0; i < array.length; i++) {
      result += array[i] ** 2;
    }

    return Math.sqrt(result);
  }

  function sum$1(input) {
    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    var sumValue = 0;

    for (var i = 0; i < input.length; i++) {
      sumValue += input[i];
    }

    return sumValue;
  }

  function mean(input) {
    return sum$1(input) / input.length;
  }

  function variance(values) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (!isAnyArray(values)) {
      throw new TypeError('input must be an array');
    }

    var _options$unbiased = options.unbiased,
        unbiased = _options$unbiased === void 0 ? true : _options$unbiased,
        _options$mean = options.mean,
        mean$1 = _options$mean === void 0 ? mean(values) : _options$mean;
    var sqrError = 0;

    for (var i = 0; i < values.length; i++) {
      var x = values[i] - mean$1;
      sqrError += x * x;
    }

    if (unbiased) {
      return sqrError / (values.length - 1);
    } else {
      return sqrError / values.length;
    }
  }

  function standardDeviation(values) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    return Math.sqrt(variance(values, options));
  }

  /**
   * Pareto scaling, which uses the square root of standard deviation as the scaling factor, circumvents the amplification of noise by retaining a small portion of magnitude information.
   * Noda, I. (2008). Scaling techniques to enhance two-dimensional correlation spectra. Journal of Molecular Structure, 883, 216-227.
   * DOI: 10.1016/j.molstruc.2007.12.026
   * @param {Array<Number>} array
   * @return {Array}
   */

  function xParetoNormalization(array) {
    xCheck(array);
    let result = [];
    const sqrtSD = Math.sqrt(standardDeviation(array));

    for (let item of array) {
      result.push(item / sqrtSD);
    }

    return result;
  }

  /**
   * This function pads an array
   * @param {Array} array - the array that will be padded
   * @param {object} [options={}]
   * @param {string} [options.algorithm=''] '', value, circular, duplicate
   * @param {number} [options.size=0] padding size before first element and after last element
   * @param {number} [options.value=0] value to use for padding (if algorithm='value')
   * @return {Array}
   */

  function xPadding(array, options = {}) {
    const {
      size = 0,
      value = 0,
      algorithm = ''
    } = options;
    xCheck(array);

    if (!algorithm) {
      if (array instanceof Float64Array) {
        return array.slice();
      } else {
        return Float64Array.from(array);
      }
    }

    let result = new Float64Array(array.length + size * 2);

    for (let i = 0; i < array.length; i++) {
      result[i + size] = array[i];
    }

    let fromEnd = size + array.length;
    let toEnd = 2 * size + array.length;

    switch (algorithm.toLowerCase()) {
      case 'value':
        for (let i = 0; i < size; i++) {
          result[i] = value;
        }

        for (let i = fromEnd; i < toEnd; i++) {
          result[i] = value;
        }

        break;

      case 'duplicate':
        for (let i = 0; i < size; i++) {
          result[i] = array[0];
        }

        for (let i = fromEnd; i < toEnd; i++) {
          result[i] = array[array.length - 1];
        }

        break;

      case 'circular':
        for (let i = 0; i < size; i++) {
          result[i] = array[(array.length - size % array.length + i) % array.length];
        }

        for (let i = 0; i < size; i++) {
          result[i + fromEnd] = array[i % array.length];
        }

        break;

      default:
        throw Error('xPadding: unknown algorithm');
    }

    return result;
  }

  /**
   * This function performs a circular shift to a new array
   * Positive values of shifts will shift to the right and negative values will do to the left
   * @example xRotate([1,2,3,4],1) -> [4,1,2,3]
   * @example xRotate([1,2,3,4],-1) -> [2,3,4,1]
   * @param {Array} array - the array that will be rotated
   * @param {number} shift
   * @return {Array}
   */
  function xRotate(array, shift) {
    shift = shift % array.length;
    if (shift < 0) shift += array.length;
    let result = new Float64Array(array.length);
    result.set(array.slice(array.length - shift));
    result.set(array.slice(0, array.length - shift), shift);
    return result;
  }

  /**
   * This function calculates a rolling average
   * @param {Array<Number>} array - the array that will be rotated
   * @param {function} fct callback function that from an array returns a value.
   * @param {object} [options={}]
   * @param {number} [options.window=5] rolling window
   * @param {string} [options.padding.size=0] none, value, circular, duplicate
   * @param {string} [options.padding.algorithm='value'] none, value, circular, duplicate
   * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
   * @return {Array<Number>}
   */

  function xRolling(array, fct, options = {}) {
    xCheck(array);
    if (typeof fct !== 'function') throw Error('fct has to be a function');
    const {
      window = 5,
      padding = {}
    } = options;
    const {
      size = window - 1,
      algorithm,
      value
    } = padding;
    array = xPadding(array, {
      size,
      algorithm,
      value
    }); // ensure we get a copy and it is float64

    const newArray = [];

    for (let i = 0; i < array.length - window + 1; i++) {
      let subArray = new Float64Array(array.buffer, i * 8, window); // we will send a view to the original buffer

      newArray.push(fct(subArray));
    }

    return newArray;
  }

  /**
   * This function calculates a rolling average
   * @param {Array<Number>} array - the array that will be rotated
   * @param {object} [options={}]
   * @param {number} [options.window=5] rolling window
   * @param {string} [options.padding.size=window-1] none, value, circular, duplicate
   * @param {string} [options.padding.algorithm=''] none, value, circular, duplicate
   * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
   * @return {Array<Number>}
   */

  function xRollingAverage(array, options = {}) {
    return xRolling(array, mean, options);
  }

  /**
   * This function calculates a rolling average
   * @param {Array<Number>} array - the array that will be rotated
   * @param {object} [options={}]
   * @param {number} [options.window=5] rolling window
   * @param {string} [options.padding.size=window-1] none, value, circular, duplicate
   * @param {string} [options.padding.algorithm=''] none, value, circular, duplicate
   * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
   * @return {Array<Number>}
   */

  function xRollingMedian(array, options = {}) {
    return xRolling(array, median, options);
  }

  function min(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;

    if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
      throw new Error('fromIndex must be a positive integer smaller than length');
    }

    if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
      throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
    }

    var minValue = input[fromIndex];

    for (var i = fromIndex + 1; i < toIndex; i++) {
      if (input[i] < minValue) minValue = input[i];
    }

    return minValue;
  }

  /**
   * This function calculates a minimum within a rolling window
   * @param {Array<Number>} array - the array that will be rotated
   * @param {object} [options={}]
   * @param {number} [options.window=5] rolling window
   * @param {string} [options.padding.size=window-1] none, value, circular, duplicate
   * @param {string} [options.padding.algorithm=''] none, value, circular, duplicate
   * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
   * @return {Array<Number>}
   */

  function xRollingMin(array, options = {}) {
    return xRolling(array, min, options);
  }

  function max(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }

    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    var _options$fromIndex = options.fromIndex,
        fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
        _options$toIndex = options.toIndex,
        toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;

    if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
      throw new Error('fromIndex must be a positive integer smaller than length');
    }

    if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
      throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
    }

    var maxValue = input[fromIndex];

    for (var i = fromIndex + 1; i < toIndex; i++) {
      if (input[i] > maxValue) maxValue = input[i];
    }

    return maxValue;
  }

  /**
   * This function calculates a maximum within a rolling window
   * @param {Array<Number>} array - the array that will be rotated
   * @param {object} [options={}]
   * @param {number} [options.window=5] rolling window
   * @param {string} [options.padding.size=window-1] none, value, circular, duplicate
   * @param {string} [options.padding.algorithm=''] none, value, circular, duplicate
   * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
   * @return {Array<Number>}
   */

  function xRollingMax(array, options = {}) {
    return xRolling(array, max, options);
  }

  /**
   * This function xSubtract the first array by the second array or a constant value from each element of the first array
   * @param {Array} array1 - the array that will be rotated
   * @param {Array|Number} array2
   * @return {Array}
   */

  function xSubtract(array1, array2) {
    let isConstant = false;
    let constant;

    if (isAnyArray(array2)) {
      if (array1.length !== array2.length) {
        throw new Error('sub: size of array1 and array2 must be identical');
      }
    } else {
      isConstant = true;
      constant = Number(array2);
    }

    let array3 = new Array(array1.length);

    if (isConstant) {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] - constant;
      }
    } else {
      for (let i = 0; i < array1.length; i++) {
        array3[i] = array1[i] - array2[i];
      }
    }

    return array3;
  }

  /**
   * Calculate the sum of the values
   * @param {DataXY} [array={}] - Object that contains property x (an ordered increasing array) and y (an array)
   * @param {object} [options={}]
   * @param {number} [options.fromIndex=0] - First point for xSum
   * @param {number} [options.toIndex=x.length-1] - Last point for xSum
   * @return {number} xSum value on the specified range
   */

  function xSum(array, options = {}) {
    const {
      fromIndex = 0,
      toIndex = array.length - 1
    } = options;
    xCheck(array);
    let sumValue = array[fromIndex];

    for (let i = fromIndex + 1; i <= toIndex; i++) {
      sumValue += array[i];
    }

    return sumValue;
  }

  /**
   * This function calculates the mean absolute error
   * @param {Array<number>} array1 -
   * @param {Array<number>} array2
   * @return {Number}
   */

  function xMeanAbsoluteError(array1, array2) {
    xCheckLengths(array1, array2);
    let sum = 0;

    for (let i = 0; i < array1.length; i++) {
      sum += Math.abs(array1[i] - array2[i]);
    }

    return sum / array1.length;
  }

  /**
   * This function calculates the mean squared error
   * @param {Array<number>} array1 -
   * @param {Array<number>} array2
   * @return {Number}
   */

  function xMeanSquaredError(array1, array2) {
    xCheckLengths(array1, array2);
    let sum = 0;

    for (let i = 0; i < array1.length; i++) {
      sum += Math.pow(array1[i] - array2[i], 2);
    }

    return sum / array1.length;
  }

  /**
   * Calculates reimAbsolute value of a complex spectrum
   * @param {object} [reim] - An object of kind {re:[], im:[]}.
   * @return {Float64Array}
   */
  function reimAbsolute(data) {
    const length = data.re.length;
    const re = data.re;
    const im = data.im;
    const newArray = new Float64Array(length);

    for (let i = 0; i < length; i++) {
      newArray[i] = Math.sqrt(re[i] ** 2 + im[i] ** 2);
    }

    return newArray;
  }

  /**
   * Phase correction filter
   * @param {object} reim - An object of kind {re:[], im:[]}
   * @param {number} [phi0=0] - Angle in radians for zero order phase correction
   * @param {number} [phi1=0] - Angle in radians for first order phase correction
   * @return {object} returns a new object {re:[], im:[]}
   */
  function reimPhaseCorrection(data, phi0, phi1) {
    phi0 = Number.isFinite(phi0) ? phi0 : 0;
    phi1 = Number.isFinite(phi1) ? phi1 : 0;
    const re = data.re;
    const im = data.im;
    const length = data.re.length;
    const delta = phi1 / length;
    const alpha = 2 * Math.pow(Math.sin(delta / 2), 2);
    const beta = Math.sin(delta);
    let cosTheta = Math.cos(phi0);
    let sinTheta = Math.sin(phi0);
    const newRe = new Float64Array(length);
    const newIm = new Float64Array(length);

    for (let i = 0; i < length; i++) {
      newRe[i] = re[i] * cosTheta - im[i] * sinTheta;
      newIm[i] = re[i] * sinTheta + im[i] * cosTheta; // calculate angles i+1 from i

      let newCosTheta = cosTheta - (alpha * cosTheta + beta * sinTheta);
      let newSinTheta = sinTheta - (alpha * sinTheta - beta * cosTheta);
      cosTheta = newCosTheta;
      sinTheta = newSinTheta;
    }

    return {
      re: newRe,
      im: newIm
    };
  }

  /**
   * Implementation of the algorithm for automatic phase correction: A robust, general automatic phase
   * correction algorithm for high-resolution NMR data. 10.1002/mrc.4586
   * @param {object} data - { re, im } real and imaginary data.
   * @param {object} options -
   * @param {Number} options.minRegSize - min number of points to auto phase a region.
   * @param {Number} options.maxDistanceToJoin - max distance between regions (in number of points) to join two regions
   * @param {boolean} options.magnitudeMode - if true it uses magnitude spectrum.boolean
   * @param {Number} options.factorNoise - scale the cutoff like factorStd * noiseLevel.
   */

  const defaultOptions = {
    minRegSize: 30,
    maxDistanceToJoin: 256,
    magnitudeMode: true,
    factorNoise: 3
  };
  function reimAutoPhaseCorrection(data, options = {}) {
    const {
      re,
      im
    } = data;
    const length = re.length;
    options = Object.assign(defaultOptions, options);
    const {
      magnitudeMode,
      minRegSize
    } = options;
    let magnitudeData = magnitudeMode ? reimAbsolute(data) : re;
    let ds = holoborodko(magnitudeData);
    let peaksDs = robustBaseLineRegionsDetection(ds, options);
    let peaksSp = robustBaseLineRegionsDetection(magnitudeData, options);
    let finalPeaks = new Array(length);

    for (let i = 0; i < length; i++) {
      finalPeaks[i] = peaksSp[i] & peaksDs[i];
    } // Once the regions are detected, we auto phase each of them separately.
    // TODO: This part can be put inside a function


    let i = -1;
    let x0 = 0;
    let res = [];

    while (i < length) {
      //phase first region
      let reTmp = [];
      let imTmp = []; //Look for the first 1 in the array

      while (!finalPeaks[++i] && i < length) {
        //TODO: Add some extra points(0.1 ppm) at rigth and left sides of the region.
        x0 = i;
      }

      for (; finalPeaks[i] && i < length; i++) {
        reTmp.push(re[i]);
        imTmp.push(im[i]);
        i++;
      }

      if (reTmp.length > minRegSize) {
        res.push(autoPhaseRegion(reTmp, imTmp, x0));
      }
    } // TODO: Still some corrections needed. In the paper they remove the outlayers interatively
    // until they can perform a regression witout bad points. Can someone help here?


    let [ph1, ph0] = weightedLinearRegression(res.map(r => r.x0 / length), res.map(r => r.ph0), res.map(r => r.area / 1e11));
    let phased = reimPhaseCorrection({
      re,
      im
    }, ph0 * Math.PI / 180, ph1 * Math.PI / 180);
    return {
      data: phased,
      ph0,
      ph1
    };
  }

  function autoPhaseRegion(re, im, x0) {
    let start = -180;
    let stop = 180;
    let nSteps = 6;
    let maxSteps = 5;
    let bestAng = 0;
    let minArea = Number.MAX_SAFE_INTEGER;

    while (maxSteps > 0) {
      let dAng = (stop - start) / (nSteps + 1);

      for (let i = start; i <= stop; i += dAng) {
        let phased = reimPhaseCorrection({
          re,
          im
        }, toRadians(i), 0);
        let negArea = getNegArea(phased.re);

        if (negArea < minArea) {
          [minArea, bestAng] = [negArea, i];
        }
      }

      start = bestAng - dAng;
      stop = bestAng + dAng;
      maxSteps--;
    } // Calculate the area for the best angle


    let phased = reimPhaseCorrection({
      re,
      im
    }, toRadians(bestAng), 0);
    let area = 0;
    let sumX = 0;

    for (let j = 0; j < re.length; j++) {
      area += phased.re[j];
      sumX += phased.re[j] * (j + x0);
    }

    return {
      ph0: bestAng,
      area,
      x0: sumX / area
    };
  }

  function holoborodko(s) {
    let dk = new Float64Array(s.length);

    for (let i = 5; i < s.length - 5; i++) {
      dk[i] = (42 * (s[i + 1] - s[i - 1]) + 48 * (s[i + 2] - s[i - 2]) + 27 * (s[i + 3] + s[i - 3]) + 8 * (s[i + 4] - s[i - 4]) + s[i + 5] - s[i - 5]) / 512;
    } //Fill the borders


    for (let i = 0; i < 5; i++) {
      dk[i] = dk[5];
      dk[s.length - i - 1] = dk[s.length - 6];
    }

    return dk;
  }

  function robustBaseLineRegionsDetection(s, options) {
    const {
      maxDistanceToJoin,
      magnitudeMode,
      factorNoise
    } = options;
    let mask = new Array(s.length);

    for (let i = 0; i < s.length; i++) {
      mask[i] = false;
    }

    let change = true;

    while (change) {
      let noiseLevel = xNoiseSanPlot(s, {
        magnitudeMode
      });
      let cutOff = factorNoise * noiseLevel.positive;
      change = false;

      for (let i = 0; i < s.length; i++) {
        if (Math.abs(s[i]) > cutOff && !mask[i]) {
          change = true;
          mask[i] = true;
        }
      }
    } // Clean up mask by merging peaks blocks, separated by just a few points(4??).


    let count = 0;
    let prev = 0;

    for (let i = 0; i < s.length; i++) {
      if (!mask[i]) {
        count++;
      } else {
        if (count < maxDistanceToJoin) {
          for (let j = 0; j <= count; j++) {
            mask[prev + j] = true;
          }
        }

        while (mask[++i] && i < s.length);

        prev = i;
        count = 0;
      }
    }

    return mask;
  }

  function weightedLinearRegression(x, y, w) {
    let sxtw = 0;
    let swx = 0;
    let sw = 0;
    let sxtwy = 0;
    let swy = 0;

    for (let i = 0; i < x.length; i++) {
      sxtw += x[i] * x[i] * w[i];
      swx += x[i] * w[i];
      sw += w[i];
      sxtwy += x[i] * w[i] * y[i];
      swy += w[i] * y[i];
    }
    /* Just to know what is the matrix system that we solve
     let Mx=[[sxtw, swx], [swx, sw]];
     let My=[[sxtwy], [swy]];
    */
    //Mx inverse


    let detMx = sxtw * sw - swx * swx;
    let inMx = [[sw / detMx, -swx / detMx], [-swx / detMx, sxtw / detMx]];
    return [inMx[0][0] * sxtwy + inMx[0][1] * swy, inMx[1][0] * sxtwy + inMx[1][1] * swy];
  }

  const toRadians = degree => degree * Math.PI / 180;

  const getNegArea = data => {
    let area = 0;

    for (let i = 0; i < data.length; i++) {
      if (data[i] < 0) area -= data[i];
    }

    return area;
  };

  function FFT(size) {
    this.size = size | 0;
    if (this.size <= 1 || (this.size & this.size - 1) !== 0) throw new Error('FFT size must be a power of two and bigger than 1');
    this._csize = size << 1; // NOTE: Use of `var` is intentional for old V8 versions

    var table = new Array(this.size * 2);

    for (var i = 0; i < table.length; i += 2) {
      const angle = Math.PI * i / this.size;
      table[i] = Math.cos(angle);
      table[i + 1] = -Math.sin(angle);
    }

    this.table = table; // Find size's power of two

    var power = 0;

    for (var t = 1; this.size > t; t <<= 1) power++; // Calculate initial step's width:
    //   * If we are full radix-4 - it is 2x smaller to give inital len=8
    //   * Otherwise it is the same as `power` to give len=4


    this._width = power % 2 === 0 ? power - 1 : power; // Pre-compute bit-reversal patterns

    this._bitrev = new Array(1 << this._width);

    for (var j = 0; j < this._bitrev.length; j++) {
      this._bitrev[j] = 0;

      for (var shift = 0; shift < this._width; shift += 2) {
        var revShift = this._width - shift - 2;
        this._bitrev[j] |= (j >>> shift & 3) << revShift;
      }
    }

    this._out = null;
    this._data = null;
    this._inv = 0;
  }

  var fft = FFT;

  FFT.prototype.fromComplexArray = function fromComplexArray(complex, storage) {
    var res = storage || new Array(complex.length >>> 1);

    for (var i = 0; i < complex.length; i += 2) res[i >>> 1] = complex[i];

    return res;
  };

  FFT.prototype.createComplexArray = function createComplexArray() {
    const res = new Array(this._csize);

    for (var i = 0; i < res.length; i++) res[i] = 0;

    return res;
  };

  FFT.prototype.toComplexArray = function toComplexArray(input, storage) {
    var res = storage || this.createComplexArray();

    for (var i = 0; i < res.length; i += 2) {
      res[i] = input[i >>> 1];
      res[i + 1] = 0;
    }

    return res;
  };

  FFT.prototype.completeSpectrum = function completeSpectrum(spectrum) {
    var size = this._csize;
    var half = size >>> 1;

    for (var i = 2; i < half; i += 2) {
      spectrum[size - i] = spectrum[i];
      spectrum[size - i + 1] = -spectrum[i + 1];
    }
  };

  FFT.prototype.transform = function transform(out, data) {
    if (out === data) throw new Error('Input and output buffers must be different');
    this._out = out;
    this._data = data;
    this._inv = 0;

    this._transform4();

    this._out = null;
    this._data = null;
  };

  FFT.prototype.realTransform = function realTransform(out, data) {
    if (out === data) throw new Error('Input and output buffers must be different');
    this._out = out;
    this._data = data;
    this._inv = 0;

    this._realTransform4();

    this._out = null;
    this._data = null;
  };

  FFT.prototype.inverseTransform = function inverseTransform(out, data) {
    if (out === data) throw new Error('Input and output buffers must be different');
    this._out = out;
    this._data = data;
    this._inv = 1;

    this._transform4();

    for (var i = 0; i < out.length; i++) out[i] /= this.size;

    this._out = null;
    this._data = null;
  }; // radix-4 implementation
  //
  // NOTE: Uses of `var` are intentional for older V8 version that do not
  // support both `let compound assignments` and `const phi`


  FFT.prototype._transform4 = function _transform4() {
    var out = this._out;
    var size = this._csize; // Initial step (permute and transform)

    var width = this._width;
    var step = 1 << width;
    var len = size / step << 1;
    var outOff;
    var t;
    var bitrev = this._bitrev;

    if (len === 4) {
      for (outOff = 0, t = 0; outOff < size; outOff += len, t++) {
        const off = bitrev[t];

        this._singleTransform2(outOff, off, step);
      }
    } else {
      // len === 8
      for (outOff = 0, t = 0; outOff < size; outOff += len, t++) {
        const off = bitrev[t];

        this._singleTransform4(outOff, off, step);
      }
    } // Loop through steps in decreasing order


    var inv = this._inv ? -1 : 1;
    var table = this.table;

    for (step >>= 2; step >= 2; step >>= 2) {
      len = size / step << 1;
      var quarterLen = len >>> 2; // Loop through offsets in the data

      for (outOff = 0; outOff < size; outOff += len) {
        // Full case
        var limit = outOff + quarterLen;

        for (var i = outOff, k = 0; i < limit; i += 2, k += step) {
          const A = i;
          const B = A + quarterLen;
          const C = B + quarterLen;
          const D = C + quarterLen; // Original values

          const Ar = out[A];
          const Ai = out[A + 1];
          const Br = out[B];
          const Bi = out[B + 1];
          const Cr = out[C];
          const Ci = out[C + 1];
          const Dr = out[D];
          const Di = out[D + 1]; // Middle values

          const MAr = Ar;
          const MAi = Ai;
          const tableBr = table[k];
          const tableBi = inv * table[k + 1];
          const MBr = Br * tableBr - Bi * tableBi;
          const MBi = Br * tableBi + Bi * tableBr;
          const tableCr = table[2 * k];
          const tableCi = inv * table[2 * k + 1];
          const MCr = Cr * tableCr - Ci * tableCi;
          const MCi = Cr * tableCi + Ci * tableCr;
          const tableDr = table[3 * k];
          const tableDi = inv * table[3 * k + 1];
          const MDr = Dr * tableDr - Di * tableDi;
          const MDi = Dr * tableDi + Di * tableDr; // Pre-Final values

          const T0r = MAr + MCr;
          const T0i = MAi + MCi;
          const T1r = MAr - MCr;
          const T1i = MAi - MCi;
          const T2r = MBr + MDr;
          const T2i = MBi + MDi;
          const T3r = inv * (MBr - MDr);
          const T3i = inv * (MBi - MDi); // Final values

          const FAr = T0r + T2r;
          const FAi = T0i + T2i;
          const FCr = T0r - T2r;
          const FCi = T0i - T2i;
          const FBr = T1r + T3i;
          const FBi = T1i - T3r;
          const FDr = T1r - T3i;
          const FDi = T1i + T3r;
          out[A] = FAr;
          out[A + 1] = FAi;
          out[B] = FBr;
          out[B + 1] = FBi;
          out[C] = FCr;
          out[C + 1] = FCi;
          out[D] = FDr;
          out[D + 1] = FDi;
        }
      }
    }
  }; // radix-2 implementation
  //
  // NOTE: Only called for len=4


  FFT.prototype._singleTransform2 = function _singleTransform2(outOff, off, step) {
    const out = this._out;
    const data = this._data;
    const evenR = data[off];
    const evenI = data[off + 1];
    const oddR = data[off + step];
    const oddI = data[off + step + 1];
    const leftR = evenR + oddR;
    const leftI = evenI + oddI;
    const rightR = evenR - oddR;
    const rightI = evenI - oddI;
    out[outOff] = leftR;
    out[outOff + 1] = leftI;
    out[outOff + 2] = rightR;
    out[outOff + 3] = rightI;
  }; // radix-4
  //
  // NOTE: Only called for len=8


  FFT.prototype._singleTransform4 = function _singleTransform4(outOff, off, step) {
    const out = this._out;
    const data = this._data;
    const inv = this._inv ? -1 : 1;
    const step2 = step * 2;
    const step3 = step * 3; // Original values

    const Ar = data[off];
    const Ai = data[off + 1];
    const Br = data[off + step];
    const Bi = data[off + step + 1];
    const Cr = data[off + step2];
    const Ci = data[off + step2 + 1];
    const Dr = data[off + step3];
    const Di = data[off + step3 + 1]; // Pre-Final values

    const T0r = Ar + Cr;
    const T0i = Ai + Ci;
    const T1r = Ar - Cr;
    const T1i = Ai - Ci;
    const T2r = Br + Dr;
    const T2i = Bi + Di;
    const T3r = inv * (Br - Dr);
    const T3i = inv * (Bi - Di); // Final values

    const FAr = T0r + T2r;
    const FAi = T0i + T2i;
    const FBr = T1r + T3i;
    const FBi = T1i - T3r;
    const FCr = T0r - T2r;
    const FCi = T0i - T2i;
    const FDr = T1r - T3i;
    const FDi = T1i + T3r;
    out[outOff] = FAr;
    out[outOff + 1] = FAi;
    out[outOff + 2] = FBr;
    out[outOff + 3] = FBi;
    out[outOff + 4] = FCr;
    out[outOff + 5] = FCi;
    out[outOff + 6] = FDr;
    out[outOff + 7] = FDi;
  }; // Real input radix-4 implementation


  FFT.prototype._realTransform4 = function _realTransform4() {
    var out = this._out;
    var size = this._csize; // Initial step (permute and transform)

    var width = this._width;
    var step = 1 << width;
    var len = size / step << 1;
    var outOff;
    var t;
    var bitrev = this._bitrev;

    if (len === 4) {
      for (outOff = 0, t = 0; outOff < size; outOff += len, t++) {
        const off = bitrev[t];

        this._singleRealTransform2(outOff, off >>> 1, step >>> 1);
      }
    } else {
      // len === 8
      for (outOff = 0, t = 0; outOff < size; outOff += len, t++) {
        const off = bitrev[t];

        this._singleRealTransform4(outOff, off >>> 1, step >>> 1);
      }
    } // Loop through steps in decreasing order


    var inv = this._inv ? -1 : 1;
    var table = this.table;

    for (step >>= 2; step >= 2; step >>= 2) {
      len = size / step << 1;
      var halfLen = len >>> 1;
      var quarterLen = halfLen >>> 1;
      var hquarterLen = quarterLen >>> 1; // Loop through offsets in the data

      for (outOff = 0; outOff < size; outOff += len) {
        for (var i = 0, k = 0; i <= hquarterLen; i += 2, k += step) {
          var A = outOff + i;
          var B = A + quarterLen;
          var C = B + quarterLen;
          var D = C + quarterLen; // Original values

          var Ar = out[A];
          var Ai = out[A + 1];
          var Br = out[B];
          var Bi = out[B + 1];
          var Cr = out[C];
          var Ci = out[C + 1];
          var Dr = out[D];
          var Di = out[D + 1]; // Middle values

          var MAr = Ar;
          var MAi = Ai;
          var tableBr = table[k];
          var tableBi = inv * table[k + 1];
          var MBr = Br * tableBr - Bi * tableBi;
          var MBi = Br * tableBi + Bi * tableBr;
          var tableCr = table[2 * k];
          var tableCi = inv * table[2 * k + 1];
          var MCr = Cr * tableCr - Ci * tableCi;
          var MCi = Cr * tableCi + Ci * tableCr;
          var tableDr = table[3 * k];
          var tableDi = inv * table[3 * k + 1];
          var MDr = Dr * tableDr - Di * tableDi;
          var MDi = Dr * tableDi + Di * tableDr; // Pre-Final values

          var T0r = MAr + MCr;
          var T0i = MAi + MCi;
          var T1r = MAr - MCr;
          var T1i = MAi - MCi;
          var T2r = MBr + MDr;
          var T2i = MBi + MDi;
          var T3r = inv * (MBr - MDr);
          var T3i = inv * (MBi - MDi); // Final values

          var FAr = T0r + T2r;
          var FAi = T0i + T2i;
          var FBr = T1r + T3i;
          var FBi = T1i - T3r;
          out[A] = FAr;
          out[A + 1] = FAi;
          out[B] = FBr;
          out[B + 1] = FBi; // Output final middle point

          if (i === 0) {
            var FCr = T0r - T2r;
            var FCi = T0i - T2i;
            out[C] = FCr;
            out[C + 1] = FCi;
            continue;
          } // Do not overwrite ourselves


          if (i === hquarterLen) continue; // In the flipped case:
          // MAi = -MAi
          // MBr=-MBi, MBi=-MBr
          // MCr=-MCr
          // MDr=MDi, MDi=MDr

          var ST0r = T1r;
          var ST0i = -T1i;
          var ST1r = T0r;
          var ST1i = -T0i;
          var ST2r = -inv * T3i;
          var ST2i = -inv * T3r;
          var ST3r = -inv * T2i;
          var ST3i = -inv * T2r;
          var SFAr = ST0r + ST2r;
          var SFAi = ST0i + ST2i;
          var SFBr = ST1r + ST3i;
          var SFBi = ST1i - ST3r;
          var SA = outOff + quarterLen - i;
          var SB = outOff + halfLen - i;
          out[SA] = SFAr;
          out[SA + 1] = SFAi;
          out[SB] = SFBr;
          out[SB + 1] = SFBi;
        }
      }
    }
  }; // radix-2 implementation
  //
  // NOTE: Only called for len=4


  FFT.prototype._singleRealTransform2 = function _singleRealTransform2(outOff, off, step) {
    const out = this._out;
    const data = this._data;
    const evenR = data[off];
    const oddR = data[off + step];
    const leftR = evenR + oddR;
    const rightR = evenR - oddR;
    out[outOff] = leftR;
    out[outOff + 1] = 0;
    out[outOff + 2] = rightR;
    out[outOff + 3] = 0;
  }; // radix-4
  //
  // NOTE: Only called for len=8


  FFT.prototype._singleRealTransform4 = function _singleRealTransform4(outOff, off, step) {
    const out = this._out;
    const data = this._data;
    const inv = this._inv ? -1 : 1;
    const step2 = step * 2;
    const step3 = step * 3; // Original values

    const Ar = data[off];
    const Br = data[off + step];
    const Cr = data[off + step2];
    const Dr = data[off + step3]; // Pre-Final values

    const T0r = Ar + Cr;
    const T1r = Ar - Cr;
    const T2r = Br + Dr;
    const T3r = inv * (Br - Dr); // Final values

    const FAr = T0r + T2r;
    const FBr = T1r;
    const FBi = -T3r;
    const FCr = T0r - T2r;
    const FDr = T1r;
    const FDi = T3r;
    out[outOff] = FAr;
    out[outOff + 1] = 0;
    out[outOff + 2] = FBr;
    out[outOff + 3] = FBi;
    out[outOff + 4] = FCr;
    out[outOff + 5] = 0;
    out[outOff + 6] = FDr;
    out[outOff + 7] = FDi;
  };

  function reimFFT(data, options = {}) {
    const {
      inverse = false,
      applyZeroShift = false
    } = options;
    let {
      re,
      im
    } = data;
    const size = re.length;
    const csize = size << 1;
    let complexArray = new Float64Array(csize);

    for (let i = 0; i < csize; i += 2) {
      complexArray[i] = re[i >>> 1];
      complexArray[i + 1] = im[i >>> 1];
    }

    let fft$1 = new fft(size);
    let output = new Float64Array(csize);

    if (inverse) {
      if (applyZeroShift) complexArray = zeroShift(complexArray, true);
      fft$1.inverseTransform(output, complexArray);
    } else {
      fft$1.transform(output, complexArray);
      if (applyZeroShift) output = zeroShift(output);
    }

    let newRe = new Float64Array(size);
    let newIm = new Float64Array(size);

    for (let i = 0; i < csize; i += 2) {
      newRe[i >>> 1] = output[i];
      newIm[i >>> 1] = output[i + 1];
    }

    return {
      re: newRe,
      im: newIm
    };
  }

  const zeroShift = (data, inverse) => {
    let middle = inverse ? Math.ceil(data.length / 2) : Math.floor(data.length / 2);
    return xRotate(data, middle);
  };

  /**
   * This function make a zero filling to re and im part.
   * @param {object} data Object of kind {x:[], re:[], im:[]}.
   * @param {number} totalLength - final number of points
   * @return {SD}
   */
  function xreimZeroFilling(data, totalLength) {
    let length = data.x.length;
    if (totalLength === 0 || length === totalLength) return data;

    if (length > totalLength) {
      return {
        x: data.x.slice(0, totalLength),
        re: data.re.slice(0, totalLength),
        im: data.im.slice(0, totalLength)
      };
    }

    const x = data.x;
    const re = data.re;
    const im = data.im;
    const newX = new Float64Array(totalLength);
    const newRE = new Float64Array(totalLength);
    const newIM = new Float64Array(totalLength);

    for (let i = 0; i < length; i++) {
      newX[i] = x[i];
      newRE[i] = re[i];
      newIM[i] = im[i];
    }

    const deltaX = (x[x.length - 1] - x[0]) / (length - 1);

    for (let i = length; i < totalLength; i++) {
      newX[i] = newX[i - 1] + deltaX;
    }

    return {
      x: newX,
      re: newRE,
      im: newIM
    };
  }

  /**
   * Sort object of array, x has to be monotone.
   * @param {object} data Object of kind {x:[], re:[], im:[]}.
   * @return {SD}
   */
  function xreimSortX(data) {
    const {
      x,
      re,
      im
    } = data;

    if (x.length !== re.length || x.length !== im.length) {
      throw TypeError('xreimSortX: length of x, re and im must be identical');
    }

    if (x.length < 2 || x[0] < x[1]) return data;
    return {
      x: x.slice(0).reverse(),
      re: re.slice(0).reverse(),
      im: im.slice(0).reverse()
    };
  }

  /**
   * Throw an error in no an object of x,y arrays
   * @param {DataXY} [data={}]
   */

  function xyCheck(data = {}) {
    if (!isAnyArray(data.x) || !isAnyArray(data.y)) {
      throw new Error('Data must be an object of x and y arrays');
    }

    if (data.x.length !== data.y.length) {
      throw new Error('The x and y arrays mush have the same length');
    }
  }

  /**
   * Join x / y values when difference in X is closer than delta.
   * When joining, y values are summed and x values are weighted average
   * @param {DataXY} [data={}] - Object that contains property x (an ordered increasing array) and y (an array)
   * @param {object} [options={}]
   * @param {number|function} [options.delta=1] The range in which the two x values of the spectra must be to be placed on the same line. It may also be a function that allows to change `delta` depending on the X values of the spectrum
   * @return {DataXY} An object with the xyIntegration function
   */

  function xyJoinX(data = {}, options = {}) {
    xyCheck(data);
    const {
      delta = 1
    } = options;
    const deltaIsFunction = typeof delta === 'function';
    const x = Array.from(data.x);
    const y = Array.from(data.y);

    if (x.length < 2) {
      return {
        x,
        y
      };
    }

    let position = 0;

    for (let i = 1; i < x.length; i++) {
      let difference = x[i] - x[i - 1];
      let currentDelta = deltaIsFunction ? delta((x[i] + x[i - 1]) / 2) : delta;

      if (difference <= currentDelta) {
        // we join
        if (y[position] !== 0 || y[i] !== 0) {
          x[position] = (x[position] * y[position] + x[i] * y[i]) / (y[position] + y[i]);
          y[position] += y[i];
        }
      } else {
        position++;
        x[position] = x[i];
        y[position] = y[i];
      }
    }

    x.length = position + 1;
    y.length = position + 1;
    return {
      x,
      y
    };
  }

  function getSlots(spectra, options = {}) {
    const {
      delta = 1
    } = options;
    const deltaIsFunction = typeof delta === 'function';
    let possibleXs = Float64Array.from([].concat(...spectra.map(spectrum => spectrum.x))).sort();

    if (possibleXs.length < 1) {
      throw new Error('xyArrayMerge can not process empty arrays');
    }

    let currentSlot = {
      from: possibleXs[0],
      to: possibleXs[0],
      average: possibleXs[0],
      sum: possibleXs[0],
      number: 1
    };
    let slots = [currentSlot];

    for (let i = 1; i < possibleXs.length; i++) {
      let currentDelta = deltaIsFunction ? delta(possibleXs[i]) : delta;

      if (possibleXs[i] - currentSlot.to <= currentDelta) {
        currentSlot.to = possibleXs[i];
        currentSlot.number++;
        currentSlot.sum += possibleXs[i];
        currentSlot.average = currentSlot.sum / currentSlot.number;
      } else {
        currentSlot = {
          from: possibleXs[i],
          to: possibleXs[i],
          average: possibleXs[i],
          sum: possibleXs[i],
          number: 1
        };
        slots.push(currentSlot);
      }
    }

    return slots;
  }

  /**
   * Aligns spectra
   * @param {Array<DataXY>} spectra
   * @param {object} [options={}]
   * @param {number|function} [options.delta=1] The range in which the two x values of the spectra must be to be placed on the same line. It may also be a function that allows to change `delta` depending on the X values of the spectrum
   */

  function xyArrayAlign(spectra, options = {}) {
    const {
      delta = 1
    } = options; // we start by checking that the spectra don't have peaks too close and we simplify them

    spectra = spectra.map(spectrum => xyJoinX(spectrum, {
      delta
    }));
    const slots = getSlots(spectra, options);
    let x = Float64Array.from(slots.map(slot => slot.average));
    let ys = new Array(spectra.length).fill().map(() => new Float64Array(x.length));
    let positions = new Uint32Array(spectra.length);

    for (let i = 0; i < slots.length; i++) {
      let slot = slots[i];

      for (let j = 0; j < spectra.length; j++) {
        let spectrum = spectra[j];

        while (positions[j] < spectrum.x.length && spectrum.x[positions[j]] <= slot.to) {
          ys[j][i] += spectrum.y[positions[j]];
          positions[j]++;
        }
      }
    }

    return {
      x,
      ys
    };
  }

  /**
   * Merge DataXY
   * We have an array of DataXY and the goal is to merge all the values that are the closest possible
   * @param {Array<DataXY>} spectra
   * @param {object} [options={}]
   * @param {number|function} [options.delta=1] The range in which the two x values of the spectra must be to be placed on the same line. It may also be a function that allows to change `delta` depending on the X values of the spectrum
   */

  function xyArrayMerge(spectra, options = {}) {
    const {
      delta = 1
    } = options; // we start by checking that the spectra don't have peaks too close and we simplify them

    spectra = spectra.map(spectrum => xyJoinX(spectrum, {
      delta
    })); // at first we will calculate the X values (simple mean)

    let slots = getSlots(spectra, options);
    let x = Float64Array.from(slots.map(slot => slot.average));
    let y = new Float64Array(x.length);
    let positions = new Uint32Array(spectra.length);

    for (let i = 0; i < slots.length; i++) {
      let slot = slots[i];

      for (let j = 0; j < spectra.length; j++) {
        let spectrum = spectra[j];

        while (positions[j] < spectrum.x.length && spectrum.x[positions[j]] <= slot.to) {
          y[i] += spectrum.y[positions[j]];
          positions[j]++;
        }
      }
    }

    return {
      x,
      y
    };
  }

  /**
   * xyAlign will align data of two spectra by verifying wether x values are in a certain range (`delta`).
   * The two spectra should not have two consecutive x values which difference is
   * smaller than `delta` to achieve good results!
   * @param {DataXY} data1 First spectrum data
   * @param {DataXY} data2 Second spectrum data
   * @param {object} [options={}]
   * @param {number|function} [options.delta=1] The range in which the two x values of the spectra must be to be placed on the same line. It may also be a function that allows to change `delta` depending on the X values of the spectrum
   * @param {boolean} [options.common=true] If `true`, only the data considered as common to both spectra is kept. If `false`, the data y arrays are completed with zeroes where no common values are found
   * @param {string} [options.x='x1'] Defines what x values should be kept (`x1` : spectrum 1 x values, `x2` spectrum 2 x values, `weighted`: weighted average of both spectra x values)
   */
  function xyAlign(data1, data2, options = {}) {
    const {
      delta = 1,
      common = true,
      x = 'x1'
    } = options;
    let result = {
      x: [],
      y1: [],
      y2: []
    };
    let i = 0;
    let j = 0;
    let length1 = data1.x.length;
    let length2 = data2.x.length;

    while (i < length1 && j < length2) {
      let maxDiff = 0;

      if (typeof delta === 'function') {
        let mean = (data1.x[i] + data2.x[j]) / 2; // is this a good thing to do?

        maxDiff = delta(mean);
      } else {
        maxDiff = delta;
      }

      let difference = data1.x[i] - data2.x[j];

      if (Math.abs(difference) > maxDiff) {
        if (difference > 0) {
          if (!common) {
            result.x.push(data2.x[j]);
            result.y1.push(0);
            result.y2.push(data2.y[j]);

            if (j === length2 - 1) {
              while (i < length1) {
                result.x.push(data1.x[i]);
                result.y1.push(data1.y[i]);
                result.y2.push(0);
                i++;
              }
            }
          } // console.log({ i, j }, result);


          j++;
        } else {
          if (!common) {
            result.x.push(data1.x[i]);
            result.y1.push(data1.y[i]);
            result.y2.push(0);

            if (i === length1 - 1) {
              while (j < length2) {
                result.x.push(data2.x[j]);
                result.y1.push(0);
                result.y2.push(data2.y[j]);
                j++;
              }
            }
          } // console.log({ i, j }, result);


          i++;
        }
      } else {
        let weightedX = (data1.x[i] * data1.y[i] + data2.x[j] * data2.y[j]) / (data1.y[i] + data2.y[j]);

        switch (x) {
          case 'x1':
            result.x.push(data1.x[i]);
            break;

          case 'x2':
            result.x.push(data2.x[j]);
            break;

          case 'weighted':
            result.x.push(weightedX);
            break;

          default:
            throw new Error(`Error: Unknown x option value: ${x}`);
        }

        result.y1.push(data1.y[i]);
        result.y2.push(data2.y[j]); // console.log({ i, j }, result);

        i++;
        j++;
      }
    }

    return result;
  }

  /**
   * Finds the max y value in a range and return a {x,y} point
   * @param {DataXY} [data={}] - Object that contains property x (an ordered increasing array) and y (an array)
   * @param {object} [options={}]
   * @param {number} [options.from] - First value for xyIntegration in the X scale
   * @param {number} [options.fromIndex=0] - First point for xyIntegration
   * @param {number} [options.to] - Last value for xyIntegration in the X scale
   * @param {number} [options.toIndex=x.length-1] - Last point for xyIntegration
   * @return {object}
   */

  function xyMaxYPoint(data = {}, options = {}) {
    xyCheck(data);
    const {
      x,
      y
    } = data;
    if (x.length < 2) return 0;
    const {
      fromIndex,
      toIndex
    } = xGetFromToIndex(x, options);
    let current = {
      x: x[fromIndex],
      y: y[fromIndex],
      index: fromIndex
    };

    for (let i = fromIndex; i <= toIndex; i++) {
      if (y[i] > current.y) current = {
        x: x[i],
        y: y[i],
        index: i
      };
    }

    return current;
  }

  /**
   * Cumulative Distribution Statistics
   * @param {DataXY} [data] array of points {x,y}
   * @returns {object} x0, x25, x50, x75, x100, mode (x for maxY)
   */

  const STEPS = [0.25, 0.5, 0.75];
  function xyCumulativeDistributionStatistics(data) {
    xyCheck(data);
    const {
      x,
      y
    } = data;

    if (x.length === 0) {
      throw new Error('xyCumulativeDistributionStatistics: Array length must be greater than 0');
    }

    const cumulativeSum = xCumulative(y);
    const maxY = max(cumulativeSum);

    for (let i = 0; i < cumulativeSum.length; i++) {
      cumulativeSum[i] /= maxY;
    }

    const result = {}; // need to find the x values closest to STEPS/100

    result.x0 = x[0];
    result.x100 = x[x.length - 1];
    let currentStep = 0;

    breakPoint: for (let i = 1; i < cumulativeSum.length; i++) {
      while (STEPS[currentStep] < cumulativeSum[i]) {
        result[`x${STEPS[currentStep] * 100}`] = x[i - 1] + (x[i] - x[i - 1]) * ((STEPS[currentStep] - cumulativeSum[i - 1]) / (cumulativeSum[i] - cumulativeSum[i - 1]));
        currentStep++;
        if (currentStep === STEPS.length) break breakPoint;
      }
    }

    result.xMode = xyMaxYPoint(data).x;
    let sumXY = 0;
    let sumY = 0;

    for (let i = 0; i < x.length; i++) {
      sumXY += x[i] * y[i];
      sumY += y[i];
    }

    result.xMean = sumXY / sumY;
    return result;
  }

  /**
   * Filters x,y values to allow strictly growing values in x axis.
   * @param {DataXY} [data={}] - Object that contains property x (an ordered increasing array) and y (an array)
   * @return {DataXY}
   */

  function xyEnsureGrowingX(data = {}) {
    xyCheck(data);
    const x = Array.from(data.x);
    const y = Array.from(data.y);
    let prevX = -Infinity;
    let ansX = [];
    let ansY = [];

    for (let index = 0; index < x.length; index++) {
      if (prevX < x[index]) {
        ansX.push(x[index]);
        ansY.push(y[index]);
        prevX = x[index];
      }
    }

    return {
      x: ansX,
      y: ansY
    };
  }

  /**
   * Normalize an array of zones:
   * - ensure than from < to
   * - merge overlapping zones
   * @param {Array<Zone>} [zones=[]]
   * @param {object} [options={}]
   * @param {number} [options.from=Number.MIN_VALUE]
   * @param {number} [options.to=Number.MAX_VALUE]
   */
  function zonesNormalize(zones = [], options = {}) {
    if (zones.length === 0) return [];
    zones = JSON.parse(JSON.stringify(zones)).map(zone => zone.from > zone.to ? {
      from: zone.to,
      to: zone.from
    } : zone);
    let {
      from = Number.NEGATIVE_INFINITY,
      to = Number.POSITIVE_INFINITY
    } = options;

    if (from > to) {
      [from, to] = [to, from];
    }

    zones = zones.sort((a, b) => {
      if (a.from !== b.from) return a.from - b.from;
      return a.to - b.to;
    });
    zones.forEach(zone => {
      if (from > zone.from) zone.from = from;
      if (to < zone.to) zone.to = to;
    });
    zones = zones.filter(zone => zone.from <= zone.to);
    if (zones.length === 0) return [];
    let currentZone = zones[0];
    let result = [currentZone];

    for (let zone of zones) {
      if (zone.from <= currentZone.to) {
        currentZone.to = zone.to;
      } else {
        currentZone = zone;
        result.push(currentZone);
      }
    }

    return result;
  }

  /**
   * xyExtract zones from a XY data
   * @param {DataXY} [data={}] - Object that contains property x (an ordered increasing array) and y (an array)
   * @param {object} [options={}]
   * @param {Array} [options.zones=[]]
   * @return {Array} Array of points
   */

  function xyExtract(data = {}, options = {}) {
    xyCheck(data);
    const {
      x,
      y
    } = data;
    let {
      zones
    } = options;
    zones = zonesNormalize(zones);
    if (!Array.isArray(zones) || zones.length === 0) return data;
    let newX = [];
    let newY = [];
    let currentZone = zones[0];
    let position = 0;

    loop: for (let i = 0; i < x.length; i++) {
      while (currentZone.to < x[i]) {
        position++;
        currentZone = zones[position];

        if (!currentZone) {
          i = x.length;
          break loop;
        }
      }

      if (x[i] >= currentZone.from) {
        newX.push(x[i]);
        newY.push(y[i]);
      }
    }

    return {
      x: newX,
      y: newY
    };
  }

  /**
   * Filter out all the points for which x <= 0. Useful to display log scale data
   * @param {DataXY} [data={}]
   * @return {{x:[],y:[]}} An object with the filtered data
   */

  function xyFilterXPositive(data = {}) {
    xyCheck(data);
    const {
      x,
      y
    } = data;
    const newX = [];
    const newY = [];

    for (let i = 0; i < x.length; i++) {
      if (x[i] > 0) {
        newX.push(x[i]);
        newY.push(y[i]);
      }
    }

    return {
      x: newX,
      y: newY
    };
  }

  /**
   * Returns the numberMaxPoints points with the bigger y.
   * @param {DataXY} data - Object that contains property x (an ordered increasing array) and y (an array)
   * @param {number} numberMaxPoints Number of points to keep
   * @returns {object} The points filtered to keep the `numberMaxPoints` most intense points of the input
   */

  function xyGetNMaxY(data, numberMaxPoints) {
    xyCheck(data);

    if (data.x.length <= numberMaxPoints) {
      return data;
    } else {
      let newX = new Array(numberMaxPoints);
      let newY = new Array(numberMaxPoints); // slice() is used to make a copy of the array, because sort() is IPM

      let threshold = data.y.slice().sort((a, b) => b - a)[numberMaxPoints - 1];
      let index = 0;

      for (let i = 0; i < data.x.length; i++) {
        if (data.y[i] >= threshold) {
          newX[index] = data.x[i];
          newY[index] = data.y[i];
          index++;
        }

        if (index === numberMaxPoints) {
          return {
            x: newX,
            y: newY
          };
        }
      }
    }
  }

  /**
   * Order object of array, x has to be monotone.
   * Ensure x is growing
   * @param {DataXY} data Object of kind {x:[], y:[]}.
   * @return {SD}
   */
  function xyGrowingX(data) {
    const {
      x,
      y
    } = data;

    if (x.length !== y.length) {
      throw TypeError('sortX: length of x and y must be identical');
    }

    if (x.length < 2 || x[0] < x[1]) return data;
    return {
      x: x.slice(0).reverse(),
      y: y.slice(0).reverse()
    };
  }

  /**
   * Generate a X / Y of the xyIntegral
   * @param {DataXY} [data={}] - Object that contains property x (an ordered increasing array) and y (an array)
   * @param {object} [options={}]
   * @param {number} [options.from] - First value for xyIntegration in the X scale
   * @param {number} [options.fromIndex=0] - First point for xyIntegration
   * @param {number} [options.to] - Last value for xyIntegration in the X scale
   * @param {number} [options.toIndex=x.length-1] - Last point for xyIntegration
   * @param {boolean} [options.reverse=false] - Integrate from the larger value to the smallest value
   * @return {{x:[],y:[]}} An object with the xyIntegration function
   */

  function xyIntegral(data = {}, options = {}) {
    const {
      reverse = false
    } = options;
    xyCheck(data);
    const {
      x,
      y
    } = data;
    if (x.length < 2) return 0;
    const {
      fromIndex,
      toIndex
    } = xGetFromToIndex(x, options);
    let xyIntegration = 0;
    let currentxyIntegral;

    if (reverse) {
      currentxyIntegral = {
        x: [x[toIndex]],
        y: [0]
      };

      for (let i = toIndex; i > fromIndex; i--) {
        xyIntegration += (x[i] - x[i - 1]) * (y[i - 1] + y[i]) / 2;
        currentxyIntegral.x.push(x[i - 1]);
        currentxyIntegral.y.push(xyIntegration);
      }

      currentxyIntegral.x.reverse();
      currentxyIntegral.y.reverse();
    } else {
      currentxyIntegral = {
        x: [x[fromIndex]],
        y: [0]
      };

      for (let i = fromIndex; i < toIndex; i++) {
        xyIntegration += (x[i + 1] - x[i]) * (y[i + 1] + y[i]) / 2;
        currentxyIntegral.x.push(x[i + 1]);
        currentxyIntegral.y.push(xyIntegration);
      }
    }

    return currentxyIntegral;
  }

  /**
   * Calculate integration
   * @param {DataXY} [data={}] - Object that contains property x (an ordered increasing array) and y (an array)
   * @param {object} [options={}]
   * @param {number} [options.from] - First value for xyIntegration in the X scale
   * @param {number} [options.fromIndex=0] - First point for xyIntegration
   * @param {number} [options.to] - Last value for xyIntegration in the X scale
   * @param {number} [options.toIndex=x.length-1] - Last point for xyIntegration
   * @return {number} xyIntegration value on the specified range
   */

  function xyIntegration(data = {}, options = {}) {
    xyCheck(data);
    const {
      x,
      y
    } = data;
    if (x.length < 2) return 0;
    const {
      fromIndex,
      toIndex
    } = xGetFromToIndex(x, options);
    let currentxyIntegration = 0;

    for (let i = fromIndex; i < toIndex; i++) {
      currentxyIntegration += (x[i + 1] - x[i]) * (y[i + 1] + y[i]) / 2;
    }

    return currentxyIntegration;
  }

  /**
   * Find the closest maximum going up hill
   * @param {DataXY} [data={}] - Object that contains property x (an ordered increasing array) and y (an array)
   * @param {object} [options={}]
   * @param {number} [options.target]
   * @param {number} [options.targetIndex=0]
   * @return {{x,y,xIndex}} An object with the x/y value
   */

  function xyMaxClosestYPoint(data, options = {}) {
    xyCheck(data);
    const {
      x,
      y
    } = data;
    let {
      target,
      targetIndex
    } = options;

    if (targetIndex === undefined) {
      if (target !== undefined) {
        targetIndex = xFindClosestIndex(x, target);
      } else {
        targetIndex = 0;
      }
    }

    let previousIndex = Number.MIN_SAFE_INTEGER;
    let currentIndex = targetIndex;
    let xyMaxY = y[targetIndex];

    while (currentIndex !== previousIndex) {
      previousIndex = currentIndex;

      if (currentIndex > 0 && y[currentIndex - 1] > xyMaxY) {
        currentIndex--;
      } else if (currentIndex < x.length - 1 && y[currentIndex + 1] > xyMaxY) {
        currentIndex++;
      }

      xyMaxY = y[currentIndex];
    }

    return {
      x: x[currentIndex],
      y: y[currentIndex],
      index: currentIndex
    };
  }

  /**
   * Finds the max value in a zone
   * @param {DataXY} [data={}] - Object that contains property x (an ordered increasing array) and y (an array)
   * @param {object} [options={}]
   * @param {number} [options.from] - First value for xyIntegration in the X scale
   * @param {number} [options.fromIndex=0] - First point for xyIntegration
   * @param {number} [options.to] - Last value for xyIntegration in the X scale
   * @param {number} [options.toIndex=x.length-1] - Last point for xyIntegration
   * @return {number} Max y on the specified range
   */

  function xyMaxY(data = {}, options = {}) {
    xyCheck(data);
    const {
      x,
      y
    } = data;
    if (x.length < 2) return 0;
    const {
      fromIndex,
      toIndex
    } = xGetFromToIndex(x, options);
    let currentxyMaxY = y[fromIndex];

    for (let i = fromIndex; i <= toIndex; i++) {
      if (y[i] > currentxyMaxY) currentxyMaxY = y[i];
    }

    return currentxyMaxY;
  }

  /**
   * Finds all the max values
   * If the values are equal the middle
   * of the equal part will be the position of the signal!
   *
   * @param {DataXY} [data={}] - Object that contains property x (an ordered increasing array) and y (an array)
   * @return {Array} Array of points
   */

  function xyMaximaY(data = {}) {
    xyCheck(data);
    const {
      x,
      y
    } = data;
    if (x.length < 3) return [];
    let maxima = [];
    let startEqualIndex = -1;

    for (let i = 1; i < x.length - 1; i++) {
      if (y[i - 1] < y[i] && y[i + 1] < y[i]) {
        maxima.push({
          x: x[i],
          y: y[i],
          index: i
        });
      } else if (y[i - 1] < y[i] && y[i + 1] === y[i]) {
        startEqualIndex = i;
      } else if (y[i - 1] === y[i] && y[i + 1] < y[i]) {
        let index = (i + startEqualIndex) / 2 >> 0;
        maxima.push({
          x: x[index],
          y: y[index],
          index
        });
      }
    }

    return maxima;
  }

  /**
   * Find the closest minimum going down hill
   * @param {DataXY} [data={}] - Object that contains property x (an ordered increasing array) and y (an array)
   * @param {object} [options={}]
   * @param {number} [options.target]
   * @param {number} [options.targetIndex=0]
   * @return {{x,y,xIndex}} An object with the x/y value
   */

  function xyMinClosestYPoint(data, options = {}) {
    xyCheck(data);
    const {
      x,
      y
    } = data;
    let {
      target,
      targetIndex
    } = options;

    if (targetIndex === undefined) {
      if (target !== undefined) {
        targetIndex = xFindClosestIndex(x, target);
      } else {
        targetIndex = 0;
      }
    }

    let previousIndex = Number.MIN_SAFE_INTEGER;
    let currentIndex = targetIndex;
    let minY = y[targetIndex];

    while (currentIndex !== previousIndex) {
      previousIndex = currentIndex;

      if (currentIndex > 0 && y[currentIndex - 1] < minY) {
        currentIndex--;
      } else if (currentIndex < x.length - 1 && y[currentIndex + 1] < minY) {
        currentIndex++;
      }

      minY = y[currentIndex];
    }

    return {
      x: x[currentIndex],
      y: y[currentIndex],
      index: currentIndex
    };
  }

  /**
   * Finds the max y value in a range and return a {x,y} point
   * @param {DataXY} [data={}] - Object that contains property x (an ordered increasing array) and y (an array)
   * @param {object} [options={}]
   * @param {number} [options.from] - First value for xyIntegration in the X scale
   * @param {number} [options.fromIndex=0] - First point for xyIntegration
   * @param {number} [options.to] - Last value for xyIntegration in the X scale
   * @param {number} [options.toIndex=x.length-1] - Last point for xyIntegration
   * @return {object}
   */

  function xyMinYPoint(data = {}, options = {}) {
    xyCheck(data);
    const {
      x,
      y
    } = data;
    if (x.length < 2) return 0;
    const {
      fromIndex,
      toIndex
    } = xGetFromToIndex(x, options);
    let current = {
      x: x[fromIndex],
      y: y[fromIndex],
      index: fromIndex
    };

    for (let i = fromIndex; i <= toIndex; i++) {
      if (y[i] < current.y) current = {
        x: x[i],
        y: y[i],
        index: i
      };
    }

    return current;
  }

  /**
   * Finds all the min values
   * If the values are equal the middle
   * of the equal part will be the position of the signal!
   *
   * @param {DataXY} [data={}] - Object that contains property x (an ordered increasing array) and y (an array)
   * @return {Array} Array of points
   */

  function xyMinimaY(data = {}) {
    xyCheck(data);
    const {
      x,
      y
    } = data;
    if (x.length < 3) return [];
    let maxima = [];
    let startEqualIndex = -1;

    for (let i = 1; i < x.length - 1; i++) {
      if (y[i - 1] > y[i] && y[i + 1] > y[i]) {
        maxima.push({
          x: x[i],
          y: y[i],
          index: i
        });
      } else if (y[i - 1] > y[i] && y[i + 1] === y[i]) {
        startEqualIndex = i;
      } else if (y[i - 1] === y[i] && y[i + 1] > y[i]) {
        let index = (i + startEqualIndex) / 2 >> 0;
        maxima.push({
          x: x[index],
          y: y[index],
          index
        });
      }
    }

    return maxima;
  }

  /**
   * Returns an information about a signal
   *
   * We expect ordered data and equidistant X axis
   * You can use the method helper if required:
   * ML.ArrayPoints.uniqueX
   * ML.ArrayPoints.sortX
   * ML.ArrayPoints.equallySpaced
   *
   * @param {object} [data={}] - Object that contains property x (an ordered increasing array) and y (an array)
   * @param {object} [options={}]
   * @param {number} [options.target]
   * @param {number} [options.targetIndex]
   * @return {object} Information about signal
   */

  function xyPeakInfo(data = {}, options = {}) {
    xyCheck(data);
    const {
      x,
      y
    } = data;
    if (x.length < 3) return undefined;
    let {
      targetIndex,
      target
    } = options;

    if (targetIndex === undefined) {
      if (target !== undefined) {
        targetIndex = xFindClosestIndex(x, target);
      }
    }

    if (targetIndex === undefined) {
      throw new Error('xyPeakInfo: need to specify target or targetIndex');
    }

    let i = targetIndex;
    let currentDiff = y[i] - y[i + 1];
    let multiplier = currentDiff < 0 ? -1 : 1;
    currentDiff *= multiplier;

    while (i < x.length - 1) {
      i++;
      let newDiff = (y[i] - y[i + 1]) * multiplier;
      if (newDiff < currentDiff) break;
      currentDiff = newDiff;
    }

    let after = {
      x: x[i],
      y: y[i]
    };
    i = targetIndex;
    currentDiff = (y[i] - y[i - 1]) * multiplier;

    while (i > 1) {
      i--;
      let newDiff = (y[i] - y[i - 1]) * multiplier;
      if (newDiff < currentDiff) break;
      currentDiff = newDiff;
    }

    let before = {
      x: x[i],
      y: y[i]
    };
    return {
      inflectionBefore: before,
      inflectionAfter: after,
      extrema: {
        x: x[targetIndex],
        y: y[targetIndex]
      },
      inflectionMiddle: {
        x: (before.x + after.x) / 2,
        y: (before.y + after.y) / 2
      },
      width: Math.abs(before.x - after.x)
    };
  }

  /**
   * Find the closest minimum going down hill
   * @param {object} [data={}] - Object that contains property x (an ordered increasing array) and y (an array)
   * @param {object} [options={}]
   * @param {number} [options.target]
   * @param {number} [options.targetIndex=0]
   * @return {{x,y,xIndex}} An object with the x/y value
   */

  function xyRealMaxYPoint(data, options = {}) {
    xyCheck(data);
    const {
      x,
      y
    } = data;
    const targetIndex = xGetTargetIndex(x, options); // interpolation to a sin() function

    if (y[targetIndex - 1] > 0 && y[targetIndex + 1] > 0 && y[targetIndex] >= y[targetIndex - 1] && y[targetIndex] >= y[targetIndex + 1]) {
      let alpha = 20 * Math.log10(y[targetIndex - 1]);
      let beta = 20 * Math.log10(y[targetIndex]);
      let gamma = 20 * Math.log10(y[targetIndex + 1]);
      let p = 0.5 * (alpha - gamma) / (alpha - 2 * beta + gamma);
      return {
        x: x[targetIndex] + (x[targetIndex] - x[targetIndex - 1]) * p,
        y: y[targetIndex] - 0.25 * (y[targetIndex - 1] - y[targetIndex + 1]) * p,
        index: targetIndex
      };
    } else {
      return {
        x: x[targetIndex],
        y: y[targetIndex],
        index: targetIndex
      };
    }
  }

  function xyRealMinYPoint(data, options = {}) {
    xyCheck(data);
    const {
      x,
      y
    } = data;
    const targetIndex = xGetTargetIndex(x, options); // interpolation to a sin() function

    if (y[targetIndex - 1] < 0 && y[targetIndex + 1] < 0 && y[targetIndex] <= y[targetIndex - 1] && y[targetIndex] <= y[targetIndex + 1]) {
      let alpha = 20 * Math.log10(-y[targetIndex - 1]);
      let beta = 20 * Math.log10(-y[targetIndex]);
      let gamma = 20 * Math.log10(-y[targetIndex + 1]);
      let p = 0.5 * (alpha - gamma) / (alpha - 2 * beta + gamma);
      return {
        x: x[targetIndex] + (x[targetIndex] - x[targetIndex - 1]) * p,
        y: y[targetIndex] - 0.25 * (y[targetIndex - 1] - y[targetIndex + 1]) * p,
        index: targetIndex
      };
    } else {
      return {
        x: x[targetIndex],
        y: y[targetIndex],
        index: targetIndex
      };
    }
  }

  /**
   * xyReduce the number of points while keeping visually the same noise. Practical to
   * display many spectra as SVG.
   * SHOULD NOT BE USED FOR DATA PROCESSING !!!
   * You should rather use ml-xy-equally-spaced to make further processing
   * @param {DataXY} [data={}] - Object that contains property x (an ordered increasing array) and y (an array)
   * @param {object} [options={}]
   * @param {number} [options.from=x[0]]
   * @param {number} [options.to=x[x.length-1]]
   * @param {number} [options.nbPoints=4001] Number of points
   * @param {number} [options.zones=[]] Array of zones to keep (from/to object)
   * @param {number} [options.optimize=false] If optimize we may have less than nbPoints at the end
   */

  function xyReduce(data, options = {}) {
    xyCheck(data);
    const {
      x,
      y
    } = data;
    let {
      from = x[0],
      to = x[x.length - 1],
      nbPoints = 4001,
      optimize = false,
      zones = []
    } = options;
    zones = zonesNormalize(zones, {
      from,
      to
    });
    if (zones.length === 0) zones = [{
      from,
      to
    }]; // we take everything
    // for each zone we should know the first index, the last index and the number of points

    let totalPoints = 0;

    for (let zone of zones) {
      zone.fromIndex = xFindClosestIndex(x, zone.from);
      zone.toIndex = xFindClosestIndex(x, zone.to);

      if (zone.fromIndex > 0 && x[zone.fromIndex] > zone.from) {
        zone.fromIndex--;
      }

      if (zone.toIndex < x.length - 1 && x[zone.toIndex] < zone.to) {
        zone.toIndex++;
      }

      zone.nbPoints = zone.toIndex - zone.fromIndex + 1;
      totalPoints += zone.nbPoints;
    } // we calculate the number of points per zone that we should keep


    if (totalPoints > nbPoints) {
      // need to xyReduce number of points
      let ratio = nbPoints / totalPoints;
      let currentTotal = 0;

      for (let i = 0; i < zones.length - 1; i++) {
        const zone = zones[i];
        zone.nbPoints = Math.round(zone.nbPoints * ratio);
        currentTotal += zone.nbPoints;
      }

      zones[zones.length - 1].nbPoints = nbPoints - currentTotal;
    } else {
      let newX = new Float64Array(totalPoints);
      let newY = new Float64Array(totalPoints);
      let index = 0;

      for (let zone of zones) {
        for (let i = zone.fromIndex; i < zone.toIndex + 1; i++) {
          newX[index] = x[i];
          newY[index] = y[i];
          index++;
        }
      }

      return {
        x: newX,
        y: newY
      };
    }

    let newX = [];
    let newY = [];

    for (let zone of zones) {
      if (!zone.nbPoints) continue;
      appendFromTo(zone.fromIndex, zone.toIndex, zone.nbPoints);
    }

    return {
      x: newX,
      y: newY
    };

    function appendFromTo(fromIndex, toIndex, zoneNbPoints) {
      if (zoneNbPoints === 1) {
        newX.push(x[Math.round((toIndex - fromIndex) / 2)]);
        newY.push(y[Math.round((toIndex - fromIndex) / 2)]);
        return;
      }

      if (zoneNbPoints === 2) {
        newX.push(x[fromIndex], x[toIndex]);
        newY.push(y[fromIndex], y[toIndex]);
        return;
      }

      newX.push(x[fromIndex]);
      newY.push(y[fromIndex]);
      let minY = Number.MAX_VALUE;
      let xyMaxY = Number.MIN_VALUE;

      if (zoneNbPoints % 2 === 0) {
        zoneNbPoints = zoneNbPoints / 2 + 1;
      } else {
        zoneNbPoints = (zoneNbPoints - 1) / 2 + 1;
      } // we will need to make some kind of min / max because there are too many points
      // we will always keep the first point and the last point


      let slot = (x[toIndex] - x[fromIndex]) / (zoneNbPoints - 1);
      let currentX = x[fromIndex] + slot;
      let first = true;

      for (let i = fromIndex + 1; i <= toIndex; i++) {
        if (first) {
          minY = y[i];
          xyMaxY = y[i];
          first = false;
        } else {
          if (y[i] < minY) minY = y[i];
          if (y[i] > xyMaxY) xyMaxY = y[i];
        }

        if (x[i] >= currentX || i === toIndex) {
          if (optimize) {
            if (minY > newY[newX.length - 1]) ; else if (xyMaxY < newY[newX.length - 1]) {
              // we can skip the intermediate value
              xyMaxY = minY;
            } else {
              newX.push(currentX - slot / 2);
              newY.push(minY);
            }
          } else {
            newX.push(currentX - slot / 2);
            newY.push(minY);
          }

          newX.push(currentX);
          newY.push(xyMaxY);
          currentX += slot;
          first = true;
        }
      }
    }
  }

  /**
   * This function calculates a rolling average
   *
   * This methods will recalculate the x values by using xRollingAverage
   * @param {ArrayPoints} [points] array of points {x,y}
   * @param {object} [options={}]
   * @param {number} [options.window=5] rolling window
   * @param {function} [fct] callback function that from an array returns a value.
   * @param {string} [options.padding.size=0] none, value, circular, duplicate
   * @param {string} [options.padding.algorithm='value'] none, value, circular, duplicate
   * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
   * @return {Array<Number>}
   */

  function xyRolling(points, fct, options = {}) {
    let {
      x,
      y
    } = points;
    y = xRolling(y, fct, options);

    if (x.length !== y.length) {
      x = xRollingAverage(x, options);
    }

    return {
      x,
      y
    };
  }

  /**
   *
   * @param {DataXY} [data] array of points {x,y}
   * @returns {DataPoints}
   */

  function xyToXYObject(data) {
    xyCheck(data);
    const {
      x,
      y
    } = data;
    let objectArray = [];

    for (let i = 0; i < x.length; i++) {
      objectArray.push({
        x: x[i],
        y: y[i]
      });
    }

    return objectArray;
  }

  /**
   * Convert a DataXY to an array of array containing x,y
   * @param {DataXY} [data] array of points {x,y}
   * @returns {Array<Array<number,number>>}
   */

  function xyToXYArray(data) {
    xyCheck(data);
    const {
      x,
      y
    } = data;
    let objectArray = [];

    for (let i = 0; i < x.length; i++) {
      objectArray.push([x[i], y[i]]);
    }

    return objectArray;
  }

  const GAUSSIAN_EXP_FACTOR = -4 * Math.LN2;
  const ROOT_PI_OVER_LN2 = Math.sqrt(Math.PI / Math.LN2);
  const ROOT_THREE = Math.sqrt(3);
  const ROOT_2LN2 = Math.sqrt(2 * Math.LN2);
  const ROOT_2LN2_MINUS_ONE = Math.sqrt(2 * Math.LN2) - 1;

  // https://en.wikipedia.org/wiki/Error_function#Inverse_functions
  // This code yields to a good approximation
  // If needed a better implementation using polynomial can be found on https://en.wikipedia.org/wiki/Error_function#Inverse_functions
  function erfinv(x) {
    let a = 0.147;
    if (x === 0) return 0;
    let ln1MinusXSqrd = Math.log(1 - x * x);
    let lnEtcBy2Plus2 = ln1MinusXSqrd / 2 + 2 / (Math.PI * a);
    let firstSqrt = Math.sqrt(lnEtcBy2Plus2 ** 2 - ln1MinusXSqrd / a);
    let secondSqrt = Math.sqrt(firstSqrt - lnEtcBy2Plus2);
    return secondSqrt * (x > 0 ? 1 : -1);
  }

  class Gaussian {
    /**
     * @param {object} [options = {}]
     * @param {number} [options.height=4*LN2/(PI*FWHM)] Define the height of the peak, by default area=1 (normalized)
     * @param {number} [options.fwhm = 500] - Full Width at Half Maximum in the number of points in FWHM.
     * @param {number} [options.sd] - Standard deviation, if it's defined options.fwhm will be ignored and the value will be computed sd * Math.sqrt(8 * Math.LN2);
     */
    constructor(options = {}) {
      this.fwhm = options.sd ? Gaussian.widthToFWHM(2 * options.sd) : options.fwhm ? options.fwhm : 500;
      this.height = options.height === undefined ? Math.sqrt(-GAUSSIAN_EXP_FACTOR / Math.PI) / this.fwhm : options.height;
    }
    /**
     * Calculate a gaussian shape
     * @param {object} [options = {}]
     * @param {number} [options.factor = 6] - Number of time to take fwhm to calculate length. Default covers 99.99 % of area.
     * @param {number} [options.length = fwhm * factor + 1] - total number of points to calculate
     * @return {Float64Array} y values
     */


    getData(options = {}) {
      let {
        length,
        factor = this.getFactor()
      } = options;

      if (!length) {
        length = Math.min(Math.ceil(this.fwhm * factor), Math.pow(2, 25) - 1);
        if (length % 2 === 0) length++;
      }

      const center = (length - 1) / 2;
      const data = new Float64Array(length);

      for (let i = 0; i <= center; i++) {
        data[i] = this.fct(i - center) * this.height;
        data[length - 1 - i] = data[i];
      }

      return data;
    }
    /**
     * Return a parameterized function of a gaussian shape (see README for equation).
     * @param {number} x - x value to calculate.
     * @returns {number} - the y value of gaussian with the current parameters.
     */


    fct(x) {
      return Gaussian.fct(x, this.fwhm);
    }
    /**
     * Calculate the number of times FWHM allows to reach a specific area coverage
     * @param {number} [area=0.9999]
     * @returns {number}
     */


    getFactor(area = 0.9999) {
      return Gaussian.getFactor(area);
    }
    /**
     * Calculate the area of the shape.
     * @returns {number} - returns the area.
     */


    getArea() {
      return Gaussian.getArea(this.fwhm, {
        height: this.height
      });
    }
    /**
     * Compute the value of Full Width at Half Maximum (FWHM) from the width between the inflection points.
     * //https://mathworld.wolfram.com/GaussianFunction.html
     * @param {number} width - Width between the inflection points
     * @returns {number} fwhm
     */


    widthToFWHM(width) {
      //https://mathworld.wolfram.com/GaussianFunction.html
      return Gaussian.widthToFWHM(width);
    }
    /**
     * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
     * //https://mathworld.wolfram.com/GaussianFunction.html
     * @param {number} fwhm - Full Width at Half Maximum.
     * @returns {number} width
     */


    fwhmToWidth(fwhm = this.fwhm) {
      return Gaussian.fwhmToWidth(fwhm);
    }
    /**
     * set a new full width at half maximum
     * @param {number} fwhm - full width at half maximum
     */


    setFWHM(fwhm) {
      this.fwhm = fwhm;
    }
    /**
     * set a new height
     * @param {number} height - The maximal intensity of the shape.
     */


    setHeight(height) {
      this.height = height;
    }

  }
  /**
   * Return a parameterized function of a gaussian shape (see README for equation).
   * @param {number} x - x value to calculate.
   * @param {number} fwhm - full width half maximum
   * @returns {number} - the y value of gaussian with the current parameters.
   */

  Gaussian.fct = function fct(x, fwhm = 500) {
    return Math.exp(GAUSSIAN_EXP_FACTOR * Math.pow(x / fwhm, 2));
  };
  /**
   * Compute the value of Full Width at Half Maximum (FWHM) from the width between the inflection points.
   * //https://mathworld.wolfram.com/GaussianFunction.html
   * @param {number} width - Width between the inflection points
   * @returns {number} fwhm
   */


  Gaussian.widthToFWHM = function widthToFWHM(width) {
    return width * ROOT_2LN2;
  };
  /**
   * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
   * //https://mathworld.wolfram.com/GaussianFunction.html
   * @param {number} fwhm - Full Width at Half Maximum.
   * @returns {number} width
   */


  Gaussian.fwhmToWidth = function fwhmToWidth(fwhm) {
    return fwhm / ROOT_2LN2;
  };
  /**
   * Calculate the area of a specific shape.
   * @param {number} fwhm - Full width at half maximum.
   * @param {object} [options = {}] - options.
   * @param {number} [options.height = 1] - Maximum y value of the shape.
   * @returns {number} - returns the area of the specific shape and parameters.
   */


  Gaussian.getArea = function getArea(fwhm, options = {}) {
    let {
      height = 1
    } = options;
    return height * ROOT_PI_OVER_LN2 * fwhm / 2;
  };
  /**
   * Calculate the number of times FWHM allows to reach a specific area coverage.
   * @param {number} [area=0.9999]
   * @returns {number}
   */


  Gaussian.getFactor = function getFactor(area = 0.9999) {
    return Math.sqrt(2) * erfinv(area);
  };

  class Lorentzian {
    /**
     * @param {object} [options = {}]
     * @param {number} [options.height=2/(PI*FWHM)] Define the height of the peak, by default area=1 (normalized)
     * @param {number} [options.fwhm = 500] - Full Width at Half Maximum in the number of points in FWHM.
     * @param {number} [options.sd] - Standard deviation, if it's defined options.fwhm will be ignored and the value will be computed sd * Math.sqrt(8 * Math.LN2);
     */
    constructor(options = {}) {
      this.fwhm = options.fwhm === undefined ? 500 : options.fwhm;
      this.height = options.height === undefined ? 2 / Math.PI / this.fwhm : options.height;
    }
    /**
     * Calculate a lorentzian shape
     * @param {object} [options = {}]
     * @param {number} [options.factor = Math.tan(Math.PI * (0.9999 - 0.5))] - Number of time to take fwhm to calculate length. Default covers 99.99 % of area.
     * @param {number} [options.length = fwhm * factor + 1] - total number of points to calculate
     * @return {Float64Array} y values
     */


    getData(options = {}) {
      let {
        length,
        factor = this.getFactor()
      } = options;

      if (!length) {
        length = Math.min(Math.ceil(this.fwhm * factor), Math.pow(2, 25) - 1);
        if (length % 2 === 0) length++;
      }

      const center = (length - 1) / 2;
      const data = new Float64Array(length);

      for (let i = 0; i <= center; i++) {
        data[i] = this.fct(i - center) * this.height;
        data[length - 1 - i] = data[i];
      }

      return data;
    }
    /**
     * Return a parameterized function of a lorentzian shape (see README for equation).
     * @param {number} x - x value to calculate.
     * @returns {number} - the y value of lorentzian with the current parameters.
     */


    fct(x) {
      return Lorentzian.fct(x, this.fwhm);
    }
    /**
     * Calculate the number of times FWHM allows to reach a specific area coverage
     * @param {number} [area=0.9999]
     * @returns {number}
     */


    getFactor(area = 0.9999) {
      return Lorentzian.getFactor(area);
    }
    /**
     * Calculate the area of the shape.
     * @returns {number} - returns the area.
     */


    getArea() {
      return Lorentzian.getArea(this.fwhm, {
        height: this.height
      });
    }
    /**
     * Compute the value of width between the inflection points of a specific shape from Full Width at Half Maximum (FWHM).
     * //https://mathworld.wolfram.com/LorentzianFunction.html
     * @param {number} [fwhm] - Full Width at Half Maximum.
     * @returns {number} width between the inflection points
     */


    fwhmToWidth(fwhm = this.fwhm) {
      return Lorentzian.fwhmToWidth(fwhm);
    }
    /**
     * Compute the value of Full Width at Half Maximum (FWHM) of a specific shape from the width between the inflection points.
     * //https://mathworld.wolfram.com/LorentzianFunction.html
     * @param {number} [width] Width between the inflection points
     * @returns {number} fwhm
     */


    widthToFWHM(width) {
      return Lorentzian.widthToFWHM(width);
    }
    /**
     * set a new full width at half maximum
     * @param {number} fwhm - full width at half maximum
     */


    setFWHM(fwhm) {
      this.fwhm = fwhm;
    }
    /**
     * set a new height
     * @param {number} height - The maximal intensity of the shape.
     */


    setHeight(height) {
      this.height = height;
    }

  }
  /**
   * Return a parameterized function of a gaussian shape (see README for equation).
   * @param {number} x - x value to calculate.
   * @param {number} fwhm - full width half maximum
   * @returns {number} - the y value of gaussian with the current parameters.
   */

  Lorentzian.fct = function fct(x, fwhm) {
    const squareFWHM = fwhm * fwhm;
    return squareFWHM / (4 * Math.pow(x, 2) + squareFWHM);
  };
  /**
   * Compute the value of width between the inflection points of a specific shape from Full Width at Half Maximum (FWHM).
   * //https://mathworld.wolfram.com/LorentzianFunction.html
   * @param {number} [fwhm] - Full Width at Half Maximum.
   * @returns {number} width between the inflection points
   */


  Lorentzian.fwhmToWidth = function fwhmToWidth(fwhm) {
    return fwhm / ROOT_THREE;
  };
  /**
   * Compute the value of Full Width at Half Maximum (FWHM) of a specific shape from the width between the inflection points.
   * //https://mathworld.wolfram.com/LorentzianFunction.html
   * @param {number} [width] Width between the inflection points
   * @returns {number} fwhm
   */


  Lorentzian.widthToFWHM = function widthToFWHM(width) {
    return width * ROOT_THREE;
  };
  /**
   * Calculate the area of a specific shape.
   * @param {number} fwhm - Full width at half maximum.
   * @param {*} [options = {}] - options.
   * @param {number} [options.height = 1] - Maximum y value of the shape.
   * @returns {number} - returns the area of the specific shape and parameters.
   */


  Lorentzian.getArea = function getArea(fwhm, options = {}) {
    let {
      height = 1
    } = options;
    return height * Math.PI * fwhm / 2;
  };
  /**
   * Calculate the number of times FWHM allows to reach a specific area coverage
   * @param {number} [area=0.9999]
   * @returns {number}
   */


  Lorentzian.getFactor = function getFactor(area = 0.9999) {
    return 2 * Math.tan(Math.PI * (area - 0.5));
  };

  class PseudoVoigt {
    /**
     * @param {object} [options={}]
     * @param {number} [options.height=1/(mu*FWHM/sqrt(4*LN2/PI)+(1-mu)*fwhm*PI*0.5)] Define the height of the peak, by default area=1 (normalized)
     * @param {number} [options.fwhm=500] - Full Width at Half Maximum in the number of points in FWHM.
     * @param {number} [options.mu=0.5] - ratio of gaussian contribution.
     */
    constructor(options = {}) {
      this.mu = options.mu === undefined ? 0.5 : options.mu;
      this.fwhm = options.fwhm === undefined ? 500 : options.fwhm;
      this.height = options.height === undefined ? 1 / (this.mu / Math.sqrt(-GAUSSIAN_EXP_FACTOR / Math.PI) * this.fwhm + (1 - this.mu) * this.fwhm * Math.PI / 2) : options.height;
    }
    /**
     * Calculate a linear combination of gaussian and lorentzian function width an same full width at half maximum
     * @param { object } [options = {}]
     * @param { number } [options.factor = 2 * Math.tan(Math.PI * (0.9999 - 0.5))] - Number of time to take fwhm in the calculation of the length.Default covers 99.99 % of area.
     * @param { number } [options.length = fwhm * factor + 1] - total number of points to calculate
     * @return { object } - { fwhm, data<Float64Array>} - An with the number of points at half maximum and the array of y values covering the 99.99 % of the area.
     */


    getData(options = {}) {
      let {
        length,
        factor = this.getFactor()
      } = options;

      if (!length) {
        length = Math.ceil(this.fwhm * factor);
        if (length % 2 === 0) length++;
      }

      const center = (length - 1) / 2;
      let data = new Float64Array(length);

      for (let i = 0; i <= center; i++) {
        data[i] = this.fct(i - center) * this.height;
        data[length - 1 - i] = data[i];
      }

      return data;
    }
    /**
     * Return a parameterized function of a linear combination of Gaussian and Lorentzian shapes where the full width at half maximum are the same for both kind of shapes (see README for equation).
     * @param {number} [x] x value to calculate.
     * @returns {number} - the y value of a pseudo voigt with the current parameters.
     */


    fct(x) {
      return PseudoVoigt.fct(x, this.fwhm, this.mu);
    }
    /**
     * Calculate the number of times FWHM allows to reach a specific area coverage
     * @param {number} [area=0.9999] - required area to be coverage
     * @param {number} [mu=this.mu] - ratio of gaussian contribution.
     * @returns {number}
     */


    getFactor(area = 0.9999, mu = this.mu) {
      return PseudoVoigt.getFactor(area, mu);
    }
    /**
     * Calculate the area of the shape.
     * @returns {number} - returns the area.
     */


    getArea() {
      return PseudoVoigt.getArea(this.fwhm, {
        height: this.height,
        mu: this.mu
      });
    }
    /**
     * Compute the value of Full Width at Half Maximum (FMHM) from width between the inflection points.
     * @param {number} width - width between the inflection points
     * @param {number} [mu = 0.5] - ratio of gaussian contribution.
     * @returns {number} Full Width at Half Maximum (FMHM).
     */


    widthToFWHM(width, mu) {
      return PseudoVoigt.widthToFWHM(width, mu);
    }
    /**
     * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
     * @param {number} fwhm - Full Width at Half Maximum.
     * @param {number} [mu] - ratio of gaussian contribution.
     * @returns {number} width between the inflection points.
     */


    fwhmToWidth(fwhm = this.fwhm, mu = this.mu) {
      return PseudoVoigt.fwhmToWidth(fwhm, mu);
    }
    /**
     * set a new full width at half maximum
     * @param {number} fwhm - full width at half maximum
     */


    setFWHM(fwhm) {
      this.fwhm = fwhm;
    }
    /**
     * set a new height
     * @param {number} height - The maximal intensity of the shape.
     */


    setHeight(height) {
      this.height = height;
    }
    /**
     * set a new mu
     * @param {number} mu - ratio of gaussian contribution.
     */


    setMu(mu) {
      this.mu = mu;
    }

  }
  /**
   * Return a parameterized function of a gaussian shape (see README for equation).
   * @param {number} x - x value to calculate.
   * @param {number} fwhm - full width half maximum
   * @param {number} [mu=0.5] - ratio of gaussian contribution.
   * @returns {number} - the y value of gaussian with the current parameters.
   */

  PseudoVoigt.fct = function fct(x, fwhm, mu = 0.5) {
    return (1 - mu) * Lorentzian.fct(x, fwhm) + mu * Gaussian.fct(x, fwhm);
  };
  /**
   * Compute the value of Full Width at Half Maximum (FMHM) from width between the inflection points.
   * @param {number} width - width between the inflection points
   * @param {number} [mu = 0.5] - ratio of gaussian contribution.
   * @returns {number} Full Width at Half Maximum (FMHM).
   */


  PseudoVoigt.widthToFWHM = function widthToFWHM(width, mu = 0.5) {
    return width * (mu * ROOT_2LN2_MINUS_ONE + 1);
  };
  /**
   * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
   * @param {number} fwhm - Full Width at Half Maximum.
   * @param {number} [mu = 0.5] - ratio of gaussian contribution.
   * @returns {number} width between the inflection points.
   */


  PseudoVoigt.fwhmToWidth = function fwhmToWidth(fwhm, mu = 0.5) {
    return fwhm / (mu * ROOT_2LN2_MINUS_ONE + 1);
  };
  /**
   * Calculate the area of a specific shape.
   * @param {number} fwhm - Full width at half maximum.
   * @param {*} [options = {}] - options.
   * @param {number} [options.height = 1] - Maximum y value of the shape.
   * @param {number} [options.mu = 0.5] - ratio of gaussian contribution.
   * @returns {number} - returns the area of the specific shape and parameters.
   */


  PseudoVoigt.getArea = function getArea(fwhm, options = {}) {
    let {
      height = 1,
      mu = 0.5
    } = options;
    return fwhm * height * (mu * ROOT_PI_OVER_LN2 + (1 - mu) * Math.PI) / 2;
  };
  /**
   * Calculate the number of times FWHM allows to reach a specific area coverage
   * @param {number} [area=0.9999] - required area to be coverage
   * @param {number} [mu=this.mu] - ratio of gaussian contribution.
   * @returns {number}
   */


  PseudoVoigt.getFactor = function getFactor(area = 0.9999, mu = 0.5) {
    return mu < 1 ? Lorentzian.getFactor(area) : Gaussian.getFactor(area);
  };

  let axis = ['x', 'y'];
  class Gaussian2D {
    /**
     * @param {object} [options = {}]
     * @param {number} [options.height=4*LN2/(PI*xFWHM*yFWHM)] Define the height of the peak, by default area=1 (normalized).
     * @param {number} [options.fwhm = 500] - Full Width at Half Maximum in the number of points in FWHM used if x or y has not the fwhm property.
     * @param {object} [options.x] - Options for x axis.
     * @param {number} [options.x.fwhm = fwhm] - Full Width at Half Maximum in the number of points in FWHM for x axis.
     * @param {number} [options.x.sd] - Standard deviation for x axis, if it's defined options.x.fwhm will be ignored and the value will be computed sd * Math.sqrt(8 * Math.LN2);
     * @param {object} [options.y] - Options for y axis.
     * @param {number} [options.y.fwhm = fwhm] - Full Width at Half Maximum in the number of points in FWHM for y axis.
     * @param {number} [options.y.sd] - Standard deviation for y axis, if it's defined options.y.fwhm will be ignored and the value will be computed sd * Math.sqrt(8 * Math.LN2);
     */
    constructor(options = {}) {
      let {
        fwhm: globalFWHM = 500
      } = options;

      for (let i of axis) {
        let fwhm;

        if (!options[i]) {
          fwhm = globalFWHM;
        } else {
          fwhm = options[i].sd ? Gaussian2D.widthToFWHM(2 * options[i].sd) : options[i].fwhm || globalFWHM;
        }

        this[i] = {
          fwhm
        };
      }

      this.height = options.height === undefined ? -GAUSSIAN_EXP_FACTOR / Math.PI / this.x.fwhm / this.y.fwhm : options.height;
    }
    /**
     * Calculate a Gaussian2D shape
     * @param {object} [options = {}]
     * @param {number} [options.factor] - Number of time to take fwhm to calculate length. Default covers 99.99 % of area.
     * @param {object} [options.x] - parameter for x axis.
     * @param {number} [options.x.length=fwhm*factor+1] - length on x axis.
     * @param {number} [options.x.factor=factor] - Number of time to take fwhm to calculate length. Default covers 99.99 % of area.
     * @param {object} [options.y] - parameter for y axis.
     * @param {number} [options.y.length=fwhm*factor+1] - length on y axis.
     * @param {number} [options.y.factor=factor] - Number of time to take fwhm to calculate length. Default covers 99.99 % of area.
     * @return {Array<Float64Array>} - z values.
     */


    getData(options = {}) {
      let {
        x = {},
        y = {},
        factor = this.getFactor(),
        length
      } = options;
      let xLength = x.length || length;

      if (!xLength) {
        let {
          factor: xFactor = factor
        } = x;
        xLength = Math.min(Math.ceil(this.x.fwhm * xFactor), Math.pow(2, 25) - 1);
        if (xLength % 2 === 0) xLength++;
      }

      let yLength = y.length || length;

      if (!yLength) {
        let {
          factor: yFactor = factor
        } = y;
        yLength = Math.min(Math.ceil(this.y.fwhm * yFactor), Math.pow(2, 25) - 1);
        if (yLength % 2 === 0) yLength++;
      }

      const xCenter = (xLength - 1) / 2;
      const yCenter = (yLength - 1) / 2;
      const data = new Array(xLength);

      for (let i = 0; i < xLength; i++) {
        data[i] = new Array(yLength);
      }

      for (let i = 0; i < xLength; i++) {
        for (let j = 0; j < yLength; j++) {
          data[i][j] = this.fct(i - xCenter, j - yCenter) * this.height;
        }
      }

      return data;
    }
    /**
     * Return the intensity value of a 2D gaussian shape (see README for equation).
     * @param {number} x - x value to calculate.
     * @param {number} y - y value to calculate.
     * @returns {number} - the z value of bi-dimensional gaussian with the current parameters.
     */


    fct(x, y) {
      return Gaussian2D.fct(x, y, this.x.fwhm, this.y.fwhm);
    }
    /**
     * Calculate the number of times FWHM allows to reach a specific volume coverage.
     * @param {number} [volume=0.9999]
     * @returns {number}
     */


    getFactor(volume = 0.9999) {
      return Gaussian2D.getFactor(volume);
    }
    /**
     * Calculate the volume of the shape.
     * @returns {number} - returns the volume.
     */


    getVolume() {
      return Gaussian2D.getVolume(this.x.fwhm, this.y.fwhm, {
        height: this.height
      });
    }
    /**
     * Compute the value of Full Width at Half Maximum (FWHM) from the width between the inflection points.
     * //https://mathworld.wolfram.com/Gaussian2DFunction.html
     * @param {number} width - Width between the inflection points
     * @returns {number} fwhm
     */


    widthToFWHM(width) {
      //https://mathworld.wolfram.com/Gaussian2DFunction.html
      return Gaussian2D.widthToFWHM(width);
    }
    /**
     * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
     * //https://mathworld.wolfram.com/Gaussian2DFunction.html
     * @param {number} fwhm - Full Width at Half Maximum.
     * @returns {number} width
     */


    fwhmToWidth(fwhm = this.x.fwhm) {
      return Gaussian2D.fwhmToWidth(fwhm);
    }
    /**
     * set a new full width at half maximum
     * @param {number} fwhm - full width at half maximum
     * @param {string|Array<string>} axisLabel - label of axis, if it is undefined fwhm is set to both axis.
     */


    setFWHM(fwhm, axisLabel) {
      if (!axisLabel) axisLabel = axis;
      if (!Array.isArray(axisLabel)) axisLabel = [axisLabel];

      for (let i of axisLabel) {
        let axisName = i.toLowerCase();

        if (axisName !== 'y' && axisName !== 'x') {
          throw new Error('axis label should be x or y');
        }

        this[axisName].fwhm = fwhm;
      }
    }
    /**
     * set a new height
     * @param {number} height - The maximal intensity of the shape.
     */


    setHeight(height) {
      this.height = height;
    }

  }
  /**
   * Return a parameterized function of a Gaussian2D shape (see README for equation).
   * @param {number} x - x value to calculate.
   * @param {number} y - y value to calculate.
   * @param {number} fwhmX - full width half maximum in the x axis.
   * @param {number} fwhmY - full width half maximum in the y axis.
   * @returns {number} - the z value of bi-dimensional gaussian with the current parameters.
   */

  Gaussian2D.fct = function fct(x, y, xFWHM = 500, yFWHM = 500) {
    return Math.exp(GAUSSIAN_EXP_FACTOR * (Math.pow(x / xFWHM, 2) + Math.pow(y / yFWHM, 2)));
  };
  /**
   * Compute the value of Full Width at Half Maximum (FWHM) from the width between the inflection points.
   * //https://mathworld.wolfram.com/Gaussian2DFunction.html
   * @param {number} width - Width between the inflection points
   * @returns {number} fwhm
   */


  Gaussian2D.widthToFWHM = function widthToFWHM(width) {
    return width * ROOT_2LN2;
  };
  /**
   * Compute the value of width between the inflection points from Full Width at Half Maximum (FWHM).
   * //https://mathworld.wolfram.com/Gaussian2DFunction.html
   * @param {number} fwhm - Full Width at Half Maximum.
   * @returns {number} width
   */


  Gaussian2D.fwhmToWidth = function fwhmToWidth(fwhm) {
    return fwhm / ROOT_2LN2;
  };
  /**
   * Calculate the volume of a specific shape.
   * @param {number} xFWHM - Full width at half maximum for x axis.
   * @param {number} yFWHM - Full width at half maximum for y axis.
   * @param {object} [options = {}] - options.
   * @param {number} [options.height = 1] - Maximum z value of the shape.
   * @returns {number} - returns the area of the specific shape and parameters.
   */


  Gaussian2D.getVolume = function getVolume(xFWHM, yFWHM, options = {}) {
    let {
      height = 1
    } = options;
    return height * Math.PI * xFWHM * yFWHM / Math.LN2 / 4;
  };
  /**@TODO look for a better factor
   * Calculate the number of times FWHM allows to reach a specific volume coverage.
   * @param {number} [volume=0.9999]
   * @returns {number}
   */


  Gaussian2D.getFactor = function getFactor(volume = 0.9999) {
    return Math.sqrt(2) * erfinv(volume);
  };

  function getShapeGenerator(options) {
    let {
      kind = 'Gaussian',
      options: shapeOptions
    } = options;

    switch (kind.toLowerCase().replace(/[^a-z^0-9]/g, '')) {
      case 'gaussian':
        return new Gaussian(shapeOptions);

      case 'lorentzian':
        return new Lorentzian(shapeOptions);

      case 'pseudovoigt':
        return new PseudoVoigt(shapeOptions);

      case 'gaussian2d':
        return new Gaussian2D(shapeOptions);

      default:
        throw new Error(`Unknown kind: ${kind}`);
    }
  }

  /**
   * Apply Savitzky Golay algorithm
   * @param {array} [ys] Array of y values
   * @param {array|number} [xs] Array of X or deltaX
   * @param {object} [options={}]
   * @param {number} [options.windowSize=9]
   * @param {number} [options.derivative=0]
   * @param {number} [options.polynomial=3]
   * @return {array} Array containing the new ys (same length)
   */
  function SavitzkyGolay(ys, xs, options = {}) {
    let {
      windowSize = 9,
      derivative = 0,
      polynomial = 3
    } = options;

    if (windowSize % 2 === 0 || windowSize < 5 || !Number.isInteger(windowSize)) {
      throw new RangeError('Invalid window size (should be odd and at least 5 integer number)');
    }

    if (windowSize > ys.length) {
      throw new RangeError(`Window size is higher than the data length ${windowSize}>${ys.length}`);
    }

    if (derivative < 0 || !Number.isInteger(derivative)) {
      throw new RangeError('Derivative should be a positive integer');
    }

    if (polynomial < 1 || !Number.isInteger(polynomial)) {
      throw new RangeError('Polynomial should be a positive integer');
    }

    if (polynomial >= 6) {
      // eslint-disable-next-line no-console
      console.warn('You should not use polynomial grade higher than 5 if you are' + ' not sure that your data arises from such a model. Possible polynomial oscillation problems');
    }

    let half = Math.floor(windowSize / 2);
    let np = ys.length;
    let ans = new Array(np);
    let weights = fullWeights(windowSize, polynomial, derivative);
    let hs = 0;
    let constantH = true;

    if (Array.isArray(xs)) {
      constantH = false;
    } else {
      hs = Math.pow(xs, derivative);
    } //For the borders


    for (let i = 0; i < half; i++) {
      let wg1 = weights[half - i - 1];
      let wg2 = weights[half + i + 1];
      let d1 = 0;
      let d2 = 0;

      for (let l = 0; l < windowSize; l++) {
        d1 += wg1[l] * ys[l];
        d2 += wg2[l] * ys[np - windowSize + l];
      }

      if (constantH) {
        ans[half - i - 1] = d1 / hs;
        ans[np - half + i] = d2 / hs;
      } else {
        hs = getHs(xs, half - i - 1, half, derivative);
        ans[half - i - 1] = d1 / hs;
        hs = getHs(xs, np - half + i, half, derivative);
        ans[np - half + i] = d2 / hs;
      }
    } //For the internal points


    let wg = weights[half];

    for (let i = windowSize; i <= np; i++) {
      let d = 0;

      for (let l = 0; l < windowSize; l++) d += wg[l] * ys[l + i - windowSize];

      if (!constantH) hs = getHs(xs, i - half - 1, half, derivative);
      ans[i - half - 1] = d / hs;
    }

    return ans;
  }

  function getHs(h, center, half, derivative) {
    let hs = 0;
    let count = 0;

    for (let i = center - half; i < center + half; i++) {
      if (i >= 0 && i < h.length - 1) {
        hs += h[i + 1] - h[i];
        count++;
      }
    }

    return Math.pow(hs / count, derivative);
  }

  function GramPoly(i, m, k, s) {
    let Grampoly = 0;

    if (k > 0) {
      Grampoly = (4 * k - 2) / (k * (2 * m - k + 1)) * (i * GramPoly(i, m, k - 1, s) + s * GramPoly(i, m, k - 1, s - 1)) - (k - 1) * (2 * m + k) / (k * (2 * m - k + 1)) * GramPoly(i, m, k - 2, s);
    } else {
      if (k === 0 && s === 0) {
        Grampoly = 1;
      } else {
        Grampoly = 0;
      }
    }

    return Grampoly;
  }

  function GenFact(a, b) {
    let gf = 1;

    if (a >= b) {
      for (let j = a - b + 1; j <= a; j++) {
        gf *= j;
      }
    }

    return gf;
  }

  function Weight(i, t, m, n, s) {
    let sum = 0;

    for (let k = 0; k <= n; k++) {
      //console.log(k);
      sum += (2 * k + 1) * (GenFact(2 * m, k) / GenFact(2 * m + k + 1, k + 1)) * GramPoly(i, m, k, 0) * GramPoly(t, m, k, s);
    }

    return sum;
  }
  /**
   *
   * @param m  Number of points
   * @param n  Polynomial grade
   * @param s  Derivative
   */


  function fullWeights(m, n, s) {
    let weights = new Array(m);
    let np = Math.floor(m / 2);

    for (let t = -np; t <= np; t++) {
      weights[t + np] = new Array(m);

      for (let j = -np; j <= np; j++) {
        weights[t + np][j + np] = Weight(j, t, np, n, s);
      }
    }

    return weights;
  }
  /*function entropy(data,h,options){
      var trend = SavitzkyGolay(data,h,trendOptions);
      var copy = new Array(data.length);
      var sum = 0;
      var max = 0;
      for(var i=0;i<data.length;i++){
          copy[i] = data[i]-trend[i];
      }

      sum/=data.length;
      console.log(sum+" "+max);
      console.log(stat.array.standardDeviation(copy));
      console.log(Math.abs(stat.array.mean(copy))/stat.array.standardDeviation(copy));
      return sum;

  }



  function guessWindowSize(data, h){
      console.log("entropy "+entropy(data,h,trendOptions));
      return 5;
  }
  */

  /**
   * Global spectra deconvolution
   * @param {object} data - Object data with x and y arrays
   * @param {Array<number>} [data.x] - Independent variable
   * @param {Array<number>} [data.y] - Dependent variable
   * @param {object} [options={}] - Options object
   * @param {object} [options.shape={}] - Object that specified the kind of shape to calculate the FWHM instead of width between inflection points. see https://mljs.github.io/peak-shape-generator/#inflectionpointswidthtofwhm
   * @param {object} [options.shape.kind='gaussian']
   * @param {object} [options.shape.options={}]
   * @param {object} [options.sgOptions] - Options object for Savitzky-Golay filter. See https://github.com/mljs/savitzky-golay-generalized#options
   * @param {number} [options.sgOptions.windowSize = 9] - points to use in the approximations
   * @param {number} [options.sgOptions.polynomial = 3] - degree of the polynomial to use in the approximations
   * @param {number} [options.minMaxRatio = 0.00025] - Threshold to determine if a given peak should be considered as a noise
   * @param {number} [options.broadRatio = 0.00] - If `broadRatio` is higher than 0, then all the peaks which second derivative
   * smaller than `broadRatio * maxAbsSecondDerivative` will be marked with the soft mask equal to true.
   * @param {number} [options.noiseLevel = 0] - Noise threshold in spectrum units
   * @param {boolean} [options.maxCriteria = true] - Peaks are local maximum(true) or minimum(false)
   * @param {boolean} [options.smoothY = true] - Select the peak intensities from a smoothed version of the independent variables
   * @param {boolean} [options.realTopDetection = false] - Use a quadratic optimizations with the peak and its 3 closest neighbors
   * to determine the true x,y values of the peak?
   * @param {number} [options.heightFactor = 0] - Factor to multiply the calculated height (usually 2)
   * @param {number} [options.derivativeThreshold = -1] - Filters based on the amplitude of the first derivative
   * @return {Array<object>}
   */

  function gsd(data, options = {}) {
    let {
      noiseLevel,
      sgOptions = {
        windowSize: 9,
        polynomial: 3
      },
      shape = {},
      smoothY = true,
      heightFactor = 0,
      broadRatio = 0.0,
      maxCriteria = true,
      minMaxRatio = 0.00025,
      derivativeThreshold = -1,
      realTopDetection = false
    } = options;
    let {
      y: yIn,
      x
    } = data;
    const y = yIn.slice();
    let equalSpaced = isEqualSpaced(x);

    if (noiseLevel === undefined) {
      noiseLevel = equalSpaced ? getNoiseLevel(y) : 0;
    }

    const yCorrection = {
      m: 1,
      b: noiseLevel
    };

    if (!maxCriteria) {
      yCorrection.m = -1;
      yCorrection.b *= -1;
    }

    for (let i = 0; i < y.length; i++) {
      y[i] = yCorrection.m * y[i] - yCorrection.b;
    }

    for (let i = 0; i < y.length; i++) {
      if (y[i] < 0) {
        y[i] = 0;
      }
    } // If the max difference between delta x is less than 5%, then,
    // we can assume it to be equally spaced variable


    let yData = y;
    let dY, ddY;
    const {
      windowSize,
      polynomial
    } = sgOptions;

    if (equalSpaced) {
      if (smoothY) {
        yData = SavitzkyGolay(y, x[1] - x[0], {
          windowSize,
          polynomial,
          derivative: 0
        });
      }

      dY = SavitzkyGolay(y, x[1] - x[0], {
        windowSize,
        polynomial,
        derivative: 1
      });
      ddY = SavitzkyGolay(y, x[1] - x[0], {
        windowSize,
        polynomial,
        derivative: 2
      });
    } else {
      if (smoothY) {
        yData = SavitzkyGolay(y, x, {
          windowSize,
          polynomial,
          derivative: 0
        });
      }

      dY = SavitzkyGolay(y, x, {
        windowSize,
        polynomial,
        derivative: 1
      });
      ddY = SavitzkyGolay(y, x, {
        windowSize,
        polynomial,
        derivative: 2
      });
    }

    const xData = x;
    const dX = x[1] - x[0];
    let maxDdy = 0;
    let maxY = 0;

    for (let i = 0; i < yData.length; i++) {
      if (Math.abs(ddY[i]) > maxDdy) {
        maxDdy = Math.abs(ddY[i]);
      }

      if (Math.abs(yData[i]) > maxY) {
        maxY = Math.abs(yData[i]);
      }
    }

    let lastMax = null;
    let lastMin = null;
    let minddY = [];
    let intervalL = [];
    let intervalR = [];
    let broadMask = []; // By the intermediate value theorem We cannot find 2 consecutive maximum or minimum

    for (let i = 1; i < yData.length - 1; ++i) {
      // filter based on derivativeThreshold
      // console.log('pasa', y[i], dY[i], ddY[i]);
      if (Math.abs(dY[i]) > derivativeThreshold) {
        // Minimum in first derivative
        if (dY[i] < dY[i - 1] && dY[i] <= dY[i + 1] || dY[i] <= dY[i - 1] && dY[i] < dY[i + 1]) {
          lastMin = {
            x: xData[i],
            index: i
          };

          if (dX > 0 && lastMax !== null) {
            intervalL.push(lastMax);
            intervalR.push(lastMin);
          }
        } // Maximum in first derivative


        if (dY[i] >= dY[i - 1] && dY[i] > dY[i + 1] || dY[i] > dY[i - 1] && dY[i] >= dY[i + 1]) {
          lastMax = {
            x: xData[i],
            index: i
          };

          if (dX < 0 && lastMin !== null) {
            intervalL.push(lastMax);
            intervalR.push(lastMin);
          }
        }
      } // Minimum in second derivative


      if (ddY[i] < ddY[i - 1] && ddY[i] < ddY[i + 1]) {
        minddY.push(i);
        broadMask.push(Math.abs(ddY[i]) <= broadRatio * maxDdy);
      }
    }

    let widthProcessor = shape.kind ? getShapeGenerator(shape.kind, shape.options).widthToFWHM : x => x;
    let signals = [];
    let lastK = -1;
    let possible, frequency, distanceJ, minDistance, gettingCloser;

    for (let j = 0; j < minddY.length; ++j) {
      frequency = xData[minddY[j]];
      possible = -1;
      let k = lastK + 1;
      minDistance = Number.MAX_VALUE;
      distanceJ = 0;
      gettingCloser = true;

      while (possible === -1 && k < intervalL.length && gettingCloser) {
        distanceJ = Math.abs(frequency - (intervalL[k].x + intervalR[k].x) / 2); // Still getting closer?

        if (distanceJ < minDistance) {
          minDistance = distanceJ;
        } else {
          gettingCloser = false;
        }

        if (distanceJ < Math.abs(intervalL[k].x - intervalR[k].x) / 2) {
          possible = k;
          lastK = k;
        }

        ++k;
      }

      if (possible !== -1) {
        if (Math.abs(yData[minddY[j]]) > minMaxRatio * maxY) {
          let width = Math.abs(intervalR[possible].x - intervalL[possible].x);
          signals.push({
            index: minddY[j],
            x: frequency,
            y: (yData[minddY[j]] + yCorrection.b) / yCorrection.m,
            width: widthProcessor(width),
            soft: broadMask[j]
          });
          signals[signals.length - 1].left = intervalL[possible];
          signals[signals.length - 1].right = intervalR[possible];

          if (heightFactor) {
            let yLeft = yData[intervalL[possible].index];
            let yRight = yData[intervalR[possible].index];
            signals[signals.length - 1].height = heightFactor * (signals[signals.length - 1].y - (yLeft + yRight) / 2);
          }
        }
      }
    }

    if (realTopDetection) {
      determineRealTop(signals, xData, yData);
    } // Correct the values to fit the original spectra data


    for (let j = 0; j < signals.length; j++) {
      signals[j].base = noiseLevel;
    }

    signals.sort(function (a, b) {
      return a.x - b.x;
    });
    return signals;
  }

  const isEqualSpaced = x => {
    let tmp;
    let maxDx = 0;
    let minDx = Number.MAX_SAFE_INTEGER;

    for (let i = 0; i < x.length - 1; ++i) {
      tmp = Math.abs(x[i + 1] - x[i]);

      if (tmp < minDx) {
        minDx = tmp;
      }

      if (tmp > maxDx) {
        maxDx = tmp;
      }
    }

    return (maxDx - minDx) / maxDx < 0.05;
  };

  const getNoiseLevel = y => {
    let mean = 0;
    let stddev = 0;
    let length = y.length;

    for (let i = 0; i < length; ++i) {
      mean += y[i];
    }

    mean /= length;
    let averageDeviations = new Array(length);

    for (let i = 0; i < length; ++i) {
      averageDeviations[i] = Math.abs(y[i] - mean);
    }

    averageDeviations.sort((a, b) => a - b);

    if (length % 2 === 1) {
      stddev = averageDeviations[(length - 1) / 2] / 0.6745;
    } else {
      stddev = 0.5 * (averageDeviations[length / 2] + averageDeviations[length / 2 - 1]) / 0.6745;
    }

    return stddev;
  };

  const determineRealTop = (peakList, x, y) => {
    let alpha, beta, gamma, p, currentPoint;

    for (let j = 0; j < peakList.length; j++) {
      currentPoint = peakList[j].index; // peakList[j][2];
      // The detected peak could be moved 1 or 2 units to left or right.

      if (y[currentPoint - 1] >= y[currentPoint - 2] && y[currentPoint - 1] >= y[currentPoint]) {
        currentPoint--;
      } else {
        if (y[currentPoint + 1] >= y[currentPoint] && y[currentPoint + 1] >= y[currentPoint + 2]) {
          currentPoint++;
        } else {
          if (y[currentPoint - 2] >= y[currentPoint - 3] && y[currentPoint - 2] >= y[currentPoint - 1]) {
            currentPoint -= 2;
          } else {
            if (y[currentPoint + 2] >= y[currentPoint + 1] && y[currentPoint + 2] >= y[currentPoint + 3]) {
              currentPoint += 2;
            }
          }
        }
      } // interpolation to a sin() function


      if (y[currentPoint - 1] > 0 && y[currentPoint + 1] > 0 && y[currentPoint] >= y[currentPoint - 1] && y[currentPoint] >= y[currentPoint + 1] && (y[currentPoint] !== y[currentPoint - 1] || y[currentPoint] !== y[currentPoint + 1])) {
        alpha = 20 * Math.log10(y[currentPoint - 1]);
        beta = 20 * Math.log10(y[currentPoint]);
        gamma = 20 * Math.log10(y[currentPoint + 1]);
        p = 0.5 * (alpha - gamma) / (alpha - 2 * beta + gamma); // console.log(alpha, beta, gamma, `p: ${p}`);
        // console.log(x[currentPoint]+" "+tmp+" "+currentPoint);

        peakList[j].x = x[currentPoint] + (x[currentPoint] - x[currentPoint - 1]) * p;
        peakList[j].y = y[currentPoint] - 0.25 * (y[currentPoint - 1] - y[currentPoint + 1]) * p;
      }
    }
  };

  function rescale(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    } else if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }

    var output;

    if (options.output !== undefined) {
      if (!isAnyArray(options.output)) {
        throw new TypeError('output option must be an array if specified');
      }

      output = options.output;
    } else {
      output = new Array(input.length);
    }

    var currentMin = min(input);
    var currentMax = max(input);

    if (currentMin === currentMax) {
      throw new RangeError('minimum and maximum input values are equal. Cannot rescale a constant array');
    }

    var _options$min = options.min,
        minValue = _options$min === void 0 ? options.autoMinMax ? currentMin : 0 : _options$min,
        _options$max = options.max,
        maxValue = _options$max === void 0 ? options.autoMinMax ? currentMax : 1 : _options$max;

    if (minValue >= maxValue) {
      throw new RangeError('min option must be smaller than max option');
    }

    var factor = (maxValue - minValue) / (currentMax - currentMin);

    for (var i = 0; i < input.length; i++) {
      output[i] = (input[i] - currentMin) * factor + minValue;
    }

    return output;
  }

  const indent = ' '.repeat(2);
  const indentData = ' '.repeat(4);
  function inspectMatrix() {
    return inspectMatrixWithOptions(this);
  }
  function inspectMatrixWithOptions(matrix, options = {}) {
    const {
      maxRows = 15,
      maxColumns = 10,
      maxNumSize = 8
    } = options;
    return `${matrix.constructor.name} {
${indent}[
${indentData}${inspectData(matrix, maxRows, maxColumns, maxNumSize)}
${indent}]
${indent}rows: ${matrix.rows}
${indent}columns: ${matrix.columns}
}`;
  }

  function inspectData(matrix, maxRows, maxColumns, maxNumSize) {
    const {
      rows,
      columns
    } = matrix;
    const maxI = Math.min(rows, maxRows);
    const maxJ = Math.min(columns, maxColumns);
    const result = [];

    for (let i = 0; i < maxI; i++) {
      let line = [];

      for (let j = 0; j < maxJ; j++) {
        line.push(formatNumber(matrix.get(i, j), maxNumSize));
      }

      result.push(`${line.join(' ')}`);
    }

    if (maxJ !== columns) {
      result[result.length - 1] += ` ... ${columns - maxColumns} more columns`;
    }

    if (maxI !== rows) {
      result.push(`... ${rows - maxRows} more rows`);
    }

    return result.join(`\n${indentData}`);
  }

  function formatNumber(num, maxNumSize) {
    const numStr = String(num);

    if (numStr.length <= maxNumSize) {
      return numStr.padEnd(maxNumSize, ' ');
    }

    const precise = num.toPrecision(maxNumSize - 2);

    if (precise.length <= maxNumSize) {
      return precise;
    }

    const exponential = num.toExponential(maxNumSize - 2);
    const eIndex = exponential.indexOf('e');
    const e = exponential.slice(eIndex);
    return exponential.slice(0, maxNumSize - e.length) + e;
  }

  function installMathOperations(AbstractMatrix, Matrix) {
    AbstractMatrix.prototype.add = function add(value) {
      if (typeof value === 'number') return this.addS(value);
      return this.addM(value);
    };

    AbstractMatrix.prototype.addS = function addS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.addM = function addM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.add = function add(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.add(value);
    };

    AbstractMatrix.prototype.sub = function sub(value) {
      if (typeof value === 'number') return this.subS(value);
      return this.subM(value);
    };

    AbstractMatrix.prototype.subS = function subS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.subM = function subM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.sub = function sub(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sub(value);
    };

    AbstractMatrix.prototype.subtract = AbstractMatrix.prototype.sub;
    AbstractMatrix.prototype.subtractS = AbstractMatrix.prototype.subS;
    AbstractMatrix.prototype.subtractM = AbstractMatrix.prototype.subM;
    AbstractMatrix.subtract = AbstractMatrix.sub;

    AbstractMatrix.prototype.mul = function mul(value) {
      if (typeof value === 'number') return this.mulS(value);
      return this.mulM(value);
    };

    AbstractMatrix.prototype.mulS = function mulS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.mulM = function mulM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.mul = function mul(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.mul(value);
    };

    AbstractMatrix.prototype.multiply = AbstractMatrix.prototype.mul;
    AbstractMatrix.prototype.multiplyS = AbstractMatrix.prototype.mulS;
    AbstractMatrix.prototype.multiplyM = AbstractMatrix.prototype.mulM;
    AbstractMatrix.multiply = AbstractMatrix.mul;

    AbstractMatrix.prototype.div = function div(value) {
      if (typeof value === 'number') return this.divS(value);
      return this.divM(value);
    };

    AbstractMatrix.prototype.divS = function divS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.divM = function divM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.div = function div(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.div(value);
    };

    AbstractMatrix.prototype.divide = AbstractMatrix.prototype.div;
    AbstractMatrix.prototype.divideS = AbstractMatrix.prototype.divS;
    AbstractMatrix.prototype.divideM = AbstractMatrix.prototype.divM;
    AbstractMatrix.divide = AbstractMatrix.div;

    AbstractMatrix.prototype.mod = function mod(value) {
      if (typeof value === 'number') return this.modS(value);
      return this.modM(value);
    };

    AbstractMatrix.prototype.modS = function modS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) % value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.modM = function modM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) % matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.mod = function mod(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.mod(value);
    };

    AbstractMatrix.prototype.modulus = AbstractMatrix.prototype.mod;
    AbstractMatrix.prototype.modulusS = AbstractMatrix.prototype.modS;
    AbstractMatrix.prototype.modulusM = AbstractMatrix.prototype.modM;
    AbstractMatrix.modulus = AbstractMatrix.mod;

    AbstractMatrix.prototype.and = function and(value) {
      if (typeof value === 'number') return this.andS(value);
      return this.andM(value);
    };

    AbstractMatrix.prototype.andS = function andS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) & value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.andM = function andM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) & matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.and = function and(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.and(value);
    };

    AbstractMatrix.prototype.or = function or(value) {
      if (typeof value === 'number') return this.orS(value);
      return this.orM(value);
    };

    AbstractMatrix.prototype.orS = function orS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) | value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.orM = function orM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) | matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.or = function or(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.or(value);
    };

    AbstractMatrix.prototype.xor = function xor(value) {
      if (typeof value === 'number') return this.xorS(value);
      return this.xorM(value);
    };

    AbstractMatrix.prototype.xorS = function xorS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) ^ value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.xorM = function xorM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) ^ matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.xor = function xor(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.xor(value);
    };

    AbstractMatrix.prototype.leftShift = function leftShift(value) {
      if (typeof value === 'number') return this.leftShiftS(value);
      return this.leftShiftM(value);
    };

    AbstractMatrix.prototype.leftShiftS = function leftShiftS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) << value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.leftShiftM = function leftShiftM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) << matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.leftShift = function leftShift(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.leftShift(value);
    };

    AbstractMatrix.prototype.signPropagatingRightShift = function signPropagatingRightShift(value) {
      if (typeof value === 'number') return this.signPropagatingRightShiftS(value);
      return this.signPropagatingRightShiftM(value);
    };

    AbstractMatrix.prototype.signPropagatingRightShiftS = function signPropagatingRightShiftS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >> value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.signPropagatingRightShiftM = function signPropagatingRightShiftM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >> matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.signPropagatingRightShift = function signPropagatingRightShift(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.signPropagatingRightShift(value);
    };

    AbstractMatrix.prototype.rightShift = function rightShift(value) {
      if (typeof value === 'number') return this.rightShiftS(value);
      return this.rightShiftM(value);
    };

    AbstractMatrix.prototype.rightShiftS = function rightShiftS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >>> value);
        }
      }

      return this;
    };

    AbstractMatrix.prototype.rightShiftM = function rightShiftM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >>> matrix.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.rightShift = function rightShift(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.rightShift(value);
    };

    AbstractMatrix.prototype.zeroFillRightShift = AbstractMatrix.prototype.rightShift;
    AbstractMatrix.prototype.zeroFillRightShiftS = AbstractMatrix.prototype.rightShiftS;
    AbstractMatrix.prototype.zeroFillRightShiftM = AbstractMatrix.prototype.rightShiftM;
    AbstractMatrix.zeroFillRightShift = AbstractMatrix.rightShift;

    AbstractMatrix.prototype.not = function not() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, ~this.get(i, j));
        }
      }

      return this;
    };

    AbstractMatrix.not = function not(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.not();
    };

    AbstractMatrix.prototype.abs = function abs() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.abs(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.abs = function abs(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.abs();
    };

    AbstractMatrix.prototype.acos = function acos() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.acos(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.acos = function acos(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.acos();
    };

    AbstractMatrix.prototype.acosh = function acosh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.acosh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.acosh = function acosh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.acosh();
    };

    AbstractMatrix.prototype.asin = function asin() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.asin(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.asin = function asin(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.asin();
    };

    AbstractMatrix.prototype.asinh = function asinh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.asinh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.asinh = function asinh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.asinh();
    };

    AbstractMatrix.prototype.atan = function atan() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.atan(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.atan = function atan(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.atan();
    };

    AbstractMatrix.prototype.atanh = function atanh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.atanh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.atanh = function atanh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.atanh();
    };

    AbstractMatrix.prototype.cbrt = function cbrt() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.cbrt(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.cbrt = function cbrt(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.cbrt();
    };

    AbstractMatrix.prototype.ceil = function ceil() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.ceil(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.ceil = function ceil(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.ceil();
    };

    AbstractMatrix.prototype.clz32 = function clz32() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.clz32(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.clz32 = function clz32(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.clz32();
    };

    AbstractMatrix.prototype.cos = function cos() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.cos(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.cos = function cos(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.cos();
    };

    AbstractMatrix.prototype.cosh = function cosh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.cosh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.cosh = function cosh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.cosh();
    };

    AbstractMatrix.prototype.exp = function exp() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.exp(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.exp = function exp(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.exp();
    };

    AbstractMatrix.prototype.expm1 = function expm1() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.expm1(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.expm1 = function expm1(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.expm1();
    };

    AbstractMatrix.prototype.floor = function floor() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.floor(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.floor = function floor(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.floor();
    };

    AbstractMatrix.prototype.fround = function fround() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.fround(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.fround = function fround(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.fround();
    };

    AbstractMatrix.prototype.log = function log() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.log = function log(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log();
    };

    AbstractMatrix.prototype.log1p = function log1p() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log1p(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.log1p = function log1p(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log1p();
    };

    AbstractMatrix.prototype.log10 = function log10() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log10(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.log10 = function log10(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log10();
    };

    AbstractMatrix.prototype.log2 = function log2() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log2(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.log2 = function log2(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log2();
    };

    AbstractMatrix.prototype.round = function round() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.round(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.round = function round(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.round();
    };

    AbstractMatrix.prototype.sign = function sign() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sign(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.sign = function sign(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sign();
    };

    AbstractMatrix.prototype.sin = function sin() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sin(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.sin = function sin(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sin();
    };

    AbstractMatrix.prototype.sinh = function sinh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sinh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.sinh = function sinh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sinh();
    };

    AbstractMatrix.prototype.sqrt = function sqrt() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sqrt(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.sqrt = function sqrt(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sqrt();
    };

    AbstractMatrix.prototype.tan = function tan() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.tan(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.tan = function tan(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.tan();
    };

    AbstractMatrix.prototype.tanh = function tanh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.tanh(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.tanh = function tanh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.tanh();
    };

    AbstractMatrix.prototype.trunc = function trunc() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.trunc(this.get(i, j)));
        }
      }

      return this;
    };

    AbstractMatrix.trunc = function trunc(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.trunc();
    };

    AbstractMatrix.pow = function pow(matrix, arg0) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.pow(arg0);
    };

    AbstractMatrix.prototype.pow = function pow(value) {
      if (typeof value === 'number') return this.powS(value);
      return this.powM(value);
    };

    AbstractMatrix.prototype.powS = function powS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.pow(this.get(i, j), value));
        }
      }

      return this;
    };

    AbstractMatrix.prototype.powM = function powM(matrix) {
      matrix = Matrix.checkMatrix(matrix);

      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.pow(this.get(i, j), matrix.get(i, j)));
        }
      }

      return this;
    };
  }

  /**
   * @private
   * Check that a row index is not out of bounds
   * @param {Matrix} matrix
   * @param {number} index
   * @param {boolean} [outer]
   */
  function checkRowIndex(matrix, index, outer) {
    let max = outer ? matrix.rows : matrix.rows - 1;

    if (index < 0 || index > max) {
      throw new RangeError('Row index out of range');
    }
  }
  /**
   * @private
   * Check that a column index is not out of bounds
   * @param {Matrix} matrix
   * @param {number} index
   * @param {boolean} [outer]
   */

  function checkColumnIndex(matrix, index, outer) {
    let max = outer ? matrix.columns : matrix.columns - 1;

    if (index < 0 || index > max) {
      throw new RangeError('Column index out of range');
    }
  }
  /**
   * @private
   * Check that the provided vector is an array with the right length
   * @param {Matrix} matrix
   * @param {Array|Matrix} vector
   * @return {Array}
   * @throws {RangeError}
   */

  function checkRowVector(matrix, vector) {
    if (vector.to1DArray) {
      vector = vector.to1DArray();
    }

    if (vector.length !== matrix.columns) {
      throw new RangeError('vector size must be the same as the number of columns');
    }

    return vector;
  }
  /**
   * @private
   * Check that the provided vector is an array with the right length
   * @param {Matrix} matrix
   * @param {Array|Matrix} vector
   * @return {Array}
   * @throws {RangeError}
   */

  function checkColumnVector(matrix, vector) {
    if (vector.to1DArray) {
      vector = vector.to1DArray();
    }

    if (vector.length !== matrix.rows) {
      throw new RangeError('vector size must be the same as the number of rows');
    }

    return vector;
  }
  function checkIndices(matrix, rowIndices, columnIndices) {
    return {
      row: checkRowIndices(matrix, rowIndices),
      column: checkColumnIndices(matrix, columnIndices)
    };
  }
  function checkRowIndices(matrix, rowIndices) {
    if (typeof rowIndices !== 'object') {
      throw new TypeError('unexpected type for row indices');
    }

    let rowOut = rowIndices.some(r => {
      return r < 0 || r >= matrix.rows;
    });

    if (rowOut) {
      throw new RangeError('row indices are out of range');
    }

    if (!Array.isArray(rowIndices)) rowIndices = Array.from(rowIndices);
    return rowIndices;
  }
  function checkColumnIndices(matrix, columnIndices) {
    if (typeof columnIndices !== 'object') {
      throw new TypeError('unexpected type for column indices');
    }

    let columnOut = columnIndices.some(c => {
      return c < 0 || c >= matrix.columns;
    });

    if (columnOut) {
      throw new RangeError('column indices are out of range');
    }

    if (!Array.isArray(columnIndices)) columnIndices = Array.from(columnIndices);
    return columnIndices;
  }
  function checkRange(matrix, startRow, endRow, startColumn, endColumn) {
    if (arguments.length !== 5) {
      throw new RangeError('expected 4 arguments');
    }

    checkNumber('startRow', startRow);
    checkNumber('endRow', endRow);
    checkNumber('startColumn', startColumn);
    checkNumber('endColumn', endColumn);

    if (startRow > endRow || startColumn > endColumn || startRow < 0 || startRow >= matrix.rows || endRow < 0 || endRow >= matrix.rows || startColumn < 0 || startColumn >= matrix.columns || endColumn < 0 || endColumn >= matrix.columns) {
      throw new RangeError('Submatrix indices are out of range');
    }
  }
  function newArray(length, value = 0) {
    let array = [];

    for (let i = 0; i < length; i++) {
      array.push(value);
    }

    return array;
  }

  function checkNumber(name, value) {
    if (typeof value !== 'number') {
      throw new TypeError(`${name} must be a number`);
    }
  }

  function checkNonEmpty(matrix) {
    if (matrix.isEmpty()) {
      throw new Error('Empty matrix has no elements to index');
    }
  }

  function sumByRow(matrix) {
    let sum = newArray(matrix.rows);

    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[i] += matrix.get(i, j);
      }
    }

    return sum;
  }
  function sumByColumn(matrix) {
    let sum = newArray(matrix.columns);

    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[j] += matrix.get(i, j);
      }
    }

    return sum;
  }
  function sumAll(matrix) {
    let v = 0;

    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        v += matrix.get(i, j);
      }
    }

    return v;
  }
  function productByRow(matrix) {
    let sum = newArray(matrix.rows, 1);

    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[i] *= matrix.get(i, j);
      }
    }

    return sum;
  }
  function productByColumn(matrix) {
    let sum = newArray(matrix.columns, 1);

    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[j] *= matrix.get(i, j);
      }
    }

    return sum;
  }
  function productAll(matrix) {
    let v = 1;

    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        v *= matrix.get(i, j);
      }
    }

    return v;
  }
  function varianceByRow(matrix, unbiased, mean) {
    const rows = matrix.rows;
    const cols = matrix.columns;
    const variance = [];

    for (let i = 0; i < rows; i++) {
      let sum1 = 0;
      let sum2 = 0;
      let x = 0;

      for (let j = 0; j < cols; j++) {
        x = matrix.get(i, j) - mean[i];
        sum1 += x;
        sum2 += x * x;
      }

      if (unbiased) {
        variance.push((sum2 - sum1 * sum1 / cols) / (cols - 1));
      } else {
        variance.push((sum2 - sum1 * sum1 / cols) / cols);
      }
    }

    return variance;
  }
  function varianceByColumn(matrix, unbiased, mean) {
    const rows = matrix.rows;
    const cols = matrix.columns;
    const variance = [];

    for (let j = 0; j < cols; j++) {
      let sum1 = 0;
      let sum2 = 0;
      let x = 0;

      for (let i = 0; i < rows; i++) {
        x = matrix.get(i, j) - mean[j];
        sum1 += x;
        sum2 += x * x;
      }

      if (unbiased) {
        variance.push((sum2 - sum1 * sum1 / rows) / (rows - 1));
      } else {
        variance.push((sum2 - sum1 * sum1 / rows) / rows);
      }
    }

    return variance;
  }
  function varianceAll(matrix, unbiased, mean) {
    const rows = matrix.rows;
    const cols = matrix.columns;
    const size = rows * cols;
    let sum1 = 0;
    let sum2 = 0;
    let x = 0;

    for (let i = 0; i < rows; i++) {
      for (let j = 0; j < cols; j++) {
        x = matrix.get(i, j) - mean;
        sum1 += x;
        sum2 += x * x;
      }
    }

    if (unbiased) {
      return (sum2 - sum1 * sum1 / size) / (size - 1);
    } else {
      return (sum2 - sum1 * sum1 / size) / size;
    }
  }
  function centerByRow(matrix, mean) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) - mean[i]);
      }
    }
  }
  function centerByColumn(matrix, mean) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) - mean[j]);
      }
    }
  }
  function centerAll(matrix, mean) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) - mean);
      }
    }
  }
  function getScaleByRow(matrix) {
    const scale = [];

    for (let i = 0; i < matrix.rows; i++) {
      let sum = 0;

      for (let j = 0; j < matrix.columns; j++) {
        sum += Math.pow(matrix.get(i, j), 2) / (matrix.columns - 1);
      }

      scale.push(Math.sqrt(sum));
    }

    return scale;
  }
  function scaleByRow(matrix, scale) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) / scale[i]);
      }
    }
  }
  function getScaleByColumn(matrix) {
    const scale = [];

    for (let j = 0; j < matrix.columns; j++) {
      let sum = 0;

      for (let i = 0; i < matrix.rows; i++) {
        sum += Math.pow(matrix.get(i, j), 2) / (matrix.rows - 1);
      }

      scale.push(Math.sqrt(sum));
    }

    return scale;
  }
  function scaleByColumn(matrix, scale) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) / scale[j]);
      }
    }
  }
  function getScaleAll(matrix) {
    const divider = matrix.size - 1;
    let sum = 0;

    for (let j = 0; j < matrix.columns; j++) {
      for (let i = 0; i < matrix.rows; i++) {
        sum += Math.pow(matrix.get(i, j), 2) / divider;
      }
    }

    return Math.sqrt(sum);
  }
  function scaleAll(matrix, scale) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) / scale);
      }
    }
  }

  class AbstractMatrix {
    static from1DArray(newRows, newColumns, newData) {
      let length = newRows * newColumns;

      if (length !== newData.length) {
        throw new RangeError('data length does not match given dimensions');
      }

      let newMatrix = new Matrix(newRows, newColumns);

      for (let row = 0; row < newRows; row++) {
        for (let column = 0; column < newColumns; column++) {
          newMatrix.set(row, column, newData[row * newColumns + column]);
        }
      }

      return newMatrix;
    }

    static rowVector(newData) {
      let vector = new Matrix(1, newData.length);

      for (let i = 0; i < newData.length; i++) {
        vector.set(0, i, newData[i]);
      }

      return vector;
    }

    static columnVector(newData) {
      let vector = new Matrix(newData.length, 1);

      for (let i = 0; i < newData.length; i++) {
        vector.set(i, 0, newData[i]);
      }

      return vector;
    }

    static zeros(rows, columns) {
      return new Matrix(rows, columns);
    }

    static ones(rows, columns) {
      return new Matrix(rows, columns).fill(1);
    }

    static rand(rows, columns, options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        random = Math.random
      } = options;
      let matrix = new Matrix(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          matrix.set(i, j, random());
        }
      }

      return matrix;
    }

    static randInt(rows, columns, options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        min = 0,
        max = 1000,
        random = Math.random
      } = options;
      if (!Number.isInteger(min)) throw new TypeError('min must be an integer');
      if (!Number.isInteger(max)) throw new TypeError('max must be an integer');
      if (min >= max) throw new RangeError('min must be smaller than max');
      let interval = max - min;
      let matrix = new Matrix(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          let value = min + Math.round(random() * interval);
          matrix.set(i, j, value);
        }
      }

      return matrix;
    }

    static eye(rows, columns, value) {
      if (columns === undefined) columns = rows;
      if (value === undefined) value = 1;
      let min = Math.min(rows, columns);
      let matrix = this.zeros(rows, columns);

      for (let i = 0; i < min; i++) {
        matrix.set(i, i, value);
      }

      return matrix;
    }

    static diag(data, rows, columns) {
      let l = data.length;
      if (rows === undefined) rows = l;
      if (columns === undefined) columns = rows;
      let min = Math.min(l, rows, columns);
      let matrix = this.zeros(rows, columns);

      for (let i = 0; i < min; i++) {
        matrix.set(i, i, data[i]);
      }

      return matrix;
    }

    static min(matrix1, matrix2) {
      matrix1 = this.checkMatrix(matrix1);
      matrix2 = this.checkMatrix(matrix2);
      let rows = matrix1.rows;
      let columns = matrix1.columns;
      let result = new Matrix(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          result.set(i, j, Math.min(matrix1.get(i, j), matrix2.get(i, j)));
        }
      }

      return result;
    }

    static max(matrix1, matrix2) {
      matrix1 = this.checkMatrix(matrix1);
      matrix2 = this.checkMatrix(matrix2);
      let rows = matrix1.rows;
      let columns = matrix1.columns;
      let result = new this(rows, columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          result.set(i, j, Math.max(matrix1.get(i, j), matrix2.get(i, j)));
        }
      }

      return result;
    }

    static checkMatrix(value) {
      return AbstractMatrix.isMatrix(value) ? value : new Matrix(value);
    }

    static isMatrix(value) {
      return value != null && value.klass === 'Matrix';
    }

    get size() {
      return this.rows * this.columns;
    }

    apply(callback) {
      if (typeof callback !== 'function') {
        throw new TypeError('callback must be a function');
      }

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          callback.call(this, i, j);
        }
      }

      return this;
    }

    to1DArray() {
      let array = [];

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          array.push(this.get(i, j));
        }
      }

      return array;
    }

    to2DArray() {
      let copy = [];

      for (let i = 0; i < this.rows; i++) {
        copy.push([]);

        for (let j = 0; j < this.columns; j++) {
          copy[i].push(this.get(i, j));
        }
      }

      return copy;
    }

    toJSON() {
      return this.to2DArray();
    }

    isRowVector() {
      return this.rows === 1;
    }

    isColumnVector() {
      return this.columns === 1;
    }

    isVector() {
      return this.rows === 1 || this.columns === 1;
    }

    isSquare() {
      return this.rows === this.columns;
    }

    isEmpty() {
      return this.rows === 0 || this.columns === 0;
    }

    isSymmetric() {
      if (this.isSquare()) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j <= i; j++) {
            if (this.get(i, j) !== this.get(j, i)) {
              return false;
            }
          }
        }

        return true;
      }

      return false;
    }

    isEchelonForm() {
      let i = 0;
      let j = 0;
      let previousColumn = -1;
      let isEchelonForm = true;
      let checked = false;

      while (i < this.rows && isEchelonForm) {
        j = 0;
        checked = false;

        while (j < this.columns && checked === false) {
          if (this.get(i, j) === 0) {
            j++;
          } else if (this.get(i, j) === 1 && j > previousColumn) {
            checked = true;
            previousColumn = j;
          } else {
            isEchelonForm = false;
            checked = true;
          }
        }

        i++;
      }

      return isEchelonForm;
    }

    isReducedEchelonForm() {
      let i = 0;
      let j = 0;
      let previousColumn = -1;
      let isReducedEchelonForm = true;
      let checked = false;

      while (i < this.rows && isReducedEchelonForm) {
        j = 0;
        checked = false;

        while (j < this.columns && checked === false) {
          if (this.get(i, j) === 0) {
            j++;
          } else if (this.get(i, j) === 1 && j > previousColumn) {
            checked = true;
            previousColumn = j;
          } else {
            isReducedEchelonForm = false;
            checked = true;
          }
        }

        for (let k = j + 1; k < this.rows; k++) {
          if (this.get(i, k) !== 0) {
            isReducedEchelonForm = false;
          }
        }

        i++;
      }

      return isReducedEchelonForm;
    }

    echelonForm() {
      let result = this.clone();
      let h = 0;
      let k = 0;

      while (h < result.rows && k < result.columns) {
        let iMax = h;

        for (let i = h; i < result.rows; i++) {
          if (result.get(i, k) > result.get(iMax, k)) {
            iMax = i;
          }
        }

        if (result.get(iMax, k) === 0) {
          k++;
        } else {
          result.swapRows(h, iMax);
          let tmp = result.get(h, k);

          for (let j = k; j < result.columns; j++) {
            result.set(h, j, result.get(h, j) / tmp);
          }

          for (let i = h + 1; i < result.rows; i++) {
            let factor = result.get(i, k) / result.get(h, k);
            result.set(i, k, 0);

            for (let j = k + 1; j < result.columns; j++) {
              result.set(i, j, result.get(i, j) - result.get(h, j) * factor);
            }
          }

          h++;
          k++;
        }
      }

      return result;
    }

    reducedEchelonForm() {
      let result = this.echelonForm();
      let m = result.columns;
      let n = result.rows;
      let h = n - 1;

      while (h >= 0) {
        if (result.maxRow(h) === 0) {
          h--;
        } else {
          let p = 0;
          let pivot = false;

          while (p < n && pivot === false) {
            if (result.get(h, p) === 1) {
              pivot = true;
            } else {
              p++;
            }
          }

          for (let i = 0; i < h; i++) {
            let factor = result.get(i, p);

            for (let j = p; j < m; j++) {
              let tmp = result.get(i, j) - factor * result.get(h, j);
              result.set(i, j, tmp);
            }
          }

          h--;
        }
      }

      return result;
    }

    set() {
      throw new Error('set method is unimplemented');
    }

    get() {
      throw new Error('get method is unimplemented');
    }

    repeat(options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        rows = 1,
        columns = 1
      } = options;

      if (!Number.isInteger(rows) || rows <= 0) {
        throw new TypeError('rows must be a positive integer');
      }

      if (!Number.isInteger(columns) || columns <= 0) {
        throw new TypeError('columns must be a positive integer');
      }

      let matrix = new Matrix(this.rows * rows, this.columns * columns);

      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          matrix.setSubMatrix(this, this.rows * i, this.columns * j);
        }
      }

      return matrix;
    }

    fill(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, value);
        }
      }

      return this;
    }

    neg() {
      return this.mulS(-1);
    }

    getRow(index) {
      checkRowIndex(this, index);
      let row = [];

      for (let i = 0; i < this.columns; i++) {
        row.push(this.get(index, i));
      }

      return row;
    }

    getRowVector(index) {
      return Matrix.rowVector(this.getRow(index));
    }

    setRow(index, array) {
      checkRowIndex(this, index);
      array = checkRowVector(this, array);

      for (let i = 0; i < this.columns; i++) {
        this.set(index, i, array[i]);
      }

      return this;
    }

    swapRows(row1, row2) {
      checkRowIndex(this, row1);
      checkRowIndex(this, row2);

      for (let i = 0; i < this.columns; i++) {
        let temp = this.get(row1, i);
        this.set(row1, i, this.get(row2, i));
        this.set(row2, i, temp);
      }

      return this;
    }

    getColumn(index) {
      checkColumnIndex(this, index);
      let column = [];

      for (let i = 0; i < this.rows; i++) {
        column.push(this.get(i, index));
      }

      return column;
    }

    getColumnVector(index) {
      return Matrix.columnVector(this.getColumn(index));
    }

    setColumn(index, array) {
      checkColumnIndex(this, index);
      array = checkColumnVector(this, array);

      for (let i = 0; i < this.rows; i++) {
        this.set(i, index, array[i]);
      }

      return this;
    }

    swapColumns(column1, column2) {
      checkColumnIndex(this, column1);
      checkColumnIndex(this, column2);

      for (let i = 0; i < this.rows; i++) {
        let temp = this.get(i, column1);
        this.set(i, column1, this.get(i, column2));
        this.set(i, column2, temp);
      }

      return this;
    }

    addRowVector(vector) {
      vector = checkRowVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + vector[j]);
        }
      }

      return this;
    }

    subRowVector(vector) {
      vector = checkRowVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - vector[j]);
        }
      }

      return this;
    }

    mulRowVector(vector) {
      vector = checkRowVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * vector[j]);
        }
      }

      return this;
    }

    divRowVector(vector) {
      vector = checkRowVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / vector[j]);
        }
      }

      return this;
    }

    addColumnVector(vector) {
      vector = checkColumnVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + vector[i]);
        }
      }

      return this;
    }

    subColumnVector(vector) {
      vector = checkColumnVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - vector[i]);
        }
      }

      return this;
    }

    mulColumnVector(vector) {
      vector = checkColumnVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * vector[i]);
        }
      }

      return this;
    }

    divColumnVector(vector) {
      vector = checkColumnVector(this, vector);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / vector[i]);
        }
      }

      return this;
    }

    mulRow(index, value) {
      checkRowIndex(this, index);

      for (let i = 0; i < this.columns; i++) {
        this.set(index, i, this.get(index, i) * value);
      }

      return this;
    }

    mulColumn(index, value) {
      checkColumnIndex(this, index);

      for (let i = 0; i < this.rows; i++) {
        this.set(i, index, this.get(i, index) * value);
      }

      return this;
    }

    max() {
      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(0, 0);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) > v) {
            v = this.get(i, j);
          }
        }
      }

      return v;
    }

    maxIndex() {
      checkNonEmpty(this);
      let v = this.get(0, 0);
      let idx = [0, 0];

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) > v) {
            v = this.get(i, j);
            idx[0] = i;
            idx[1] = j;
          }
        }
      }

      return idx;
    }

    min() {
      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(0, 0);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) < v) {
            v = this.get(i, j);
          }
        }
      }

      return v;
    }

    minIndex() {
      checkNonEmpty(this);
      let v = this.get(0, 0);
      let idx = [0, 0];

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) < v) {
            v = this.get(i, j);
            idx[0] = i;
            idx[1] = j;
          }
        }
      }

      return idx;
    }

    maxRow(row) {
      checkRowIndex(this, row);

      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(row, 0);

      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) > v) {
          v = this.get(row, i);
        }
      }

      return v;
    }

    maxRowIndex(row) {
      checkRowIndex(this, row);
      checkNonEmpty(this);
      let v = this.get(row, 0);
      let idx = [row, 0];

      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) > v) {
          v = this.get(row, i);
          idx[1] = i;
        }
      }

      return idx;
    }

    minRow(row) {
      checkRowIndex(this, row);

      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(row, 0);

      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) < v) {
          v = this.get(row, i);
        }
      }

      return v;
    }

    minRowIndex(row) {
      checkRowIndex(this, row);
      checkNonEmpty(this);
      let v = this.get(row, 0);
      let idx = [row, 0];

      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) < v) {
          v = this.get(row, i);
          idx[1] = i;
        }
      }

      return idx;
    }

    maxColumn(column) {
      checkColumnIndex(this, column);

      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(0, column);

      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) > v) {
          v = this.get(i, column);
        }
      }

      return v;
    }

    maxColumnIndex(column) {
      checkColumnIndex(this, column);
      checkNonEmpty(this);
      let v = this.get(0, column);
      let idx = [0, column];

      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) > v) {
          v = this.get(i, column);
          idx[0] = i;
        }
      }

      return idx;
    }

    minColumn(column) {
      checkColumnIndex(this, column);

      if (this.isEmpty()) {
        return NaN;
      }

      let v = this.get(0, column);

      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) < v) {
          v = this.get(i, column);
        }
      }

      return v;
    }

    minColumnIndex(column) {
      checkColumnIndex(this, column);
      checkNonEmpty(this);
      let v = this.get(0, column);
      let idx = [0, column];

      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) < v) {
          v = this.get(i, column);
          idx[0] = i;
        }
      }

      return idx;
    }

    diag() {
      let min = Math.min(this.rows, this.columns);
      let diag = [];

      for (let i = 0; i < min; i++) {
        diag.push(this.get(i, i));
      }

      return diag;
    }

    norm(type = 'frobenius') {
      let result = 0;

      if (type === 'max') {
        return this.max();
      } else if (type === 'frobenius') {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            result = result + this.get(i, j) * this.get(i, j);
          }
        }

        return Math.sqrt(result);
      } else {
        throw new RangeError(`unknown norm type: ${type}`);
      }
    }

    cumulativeSum() {
      let sum = 0;

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          sum += this.get(i, j);
          this.set(i, j, sum);
        }
      }

      return this;
    }

    dot(vector2) {
      if (AbstractMatrix.isMatrix(vector2)) vector2 = vector2.to1DArray();
      let vector1 = this.to1DArray();

      if (vector1.length !== vector2.length) {
        throw new RangeError('vectors do not have the same size');
      }

      let dot = 0;

      for (let i = 0; i < vector1.length; i++) {
        dot += vector1[i] * vector2[i];
      }

      return dot;
    }

    mmul(other) {
      other = Matrix.checkMatrix(other);
      let m = this.rows;
      let n = this.columns;
      let p = other.columns;
      let result = new Matrix(m, p);
      let Bcolj = new Float64Array(n);

      for (let j = 0; j < p; j++) {
        for (let k = 0; k < n; k++) {
          Bcolj[k] = other.get(k, j);
        }

        for (let i = 0; i < m; i++) {
          let s = 0;

          for (let k = 0; k < n; k++) {
            s += this.get(i, k) * Bcolj[k];
          }

          result.set(i, j, s);
        }
      }

      return result;
    }

    strassen2x2(other) {
      other = Matrix.checkMatrix(other);
      let result = new Matrix(2, 2);
      const a11 = this.get(0, 0);
      const b11 = other.get(0, 0);
      const a12 = this.get(0, 1);
      const b12 = other.get(0, 1);
      const a21 = this.get(1, 0);
      const b21 = other.get(1, 0);
      const a22 = this.get(1, 1);
      const b22 = other.get(1, 1); // Compute intermediate values.

      const m1 = (a11 + a22) * (b11 + b22);
      const m2 = (a21 + a22) * b11;
      const m3 = a11 * (b12 - b22);
      const m4 = a22 * (b21 - b11);
      const m5 = (a11 + a12) * b22;
      const m6 = (a21 - a11) * (b11 + b12);
      const m7 = (a12 - a22) * (b21 + b22); // Combine intermediate values into the output.

      const c00 = m1 + m4 - m5 + m7;
      const c01 = m3 + m5;
      const c10 = m2 + m4;
      const c11 = m1 - m2 + m3 + m6;
      result.set(0, 0, c00);
      result.set(0, 1, c01);
      result.set(1, 0, c10);
      result.set(1, 1, c11);
      return result;
    }

    strassen3x3(other) {
      other = Matrix.checkMatrix(other);
      let result = new Matrix(3, 3);
      const a00 = this.get(0, 0);
      const a01 = this.get(0, 1);
      const a02 = this.get(0, 2);
      const a10 = this.get(1, 0);
      const a11 = this.get(1, 1);
      const a12 = this.get(1, 2);
      const a20 = this.get(2, 0);
      const a21 = this.get(2, 1);
      const a22 = this.get(2, 2);
      const b00 = other.get(0, 0);
      const b01 = other.get(0, 1);
      const b02 = other.get(0, 2);
      const b10 = other.get(1, 0);
      const b11 = other.get(1, 1);
      const b12 = other.get(1, 2);
      const b20 = other.get(2, 0);
      const b21 = other.get(2, 1);
      const b22 = other.get(2, 2);
      const m1 = (a00 + a01 + a02 - a10 - a11 - a21 - a22) * b11;
      const m2 = (a00 - a10) * (-b01 + b11);
      const m3 = a11 * (-b00 + b01 + b10 - b11 - b12 - b20 + b22);
      const m4 = (-a00 + a10 + a11) * (b00 - b01 + b11);
      const m5 = (a10 + a11) * (-b00 + b01);
      const m6 = a00 * b00;
      const m7 = (-a00 + a20 + a21) * (b00 - b02 + b12);
      const m8 = (-a00 + a20) * (b02 - b12);
      const m9 = (a20 + a21) * (-b00 + b02);
      const m10 = (a00 + a01 + a02 - a11 - a12 - a20 - a21) * b12;
      const m11 = a21 * (-b00 + b02 + b10 - b11 - b12 - b20 + b21);
      const m12 = (-a02 + a21 + a22) * (b11 + b20 - b21);
      const m13 = (a02 - a22) * (b11 - b21);
      const m14 = a02 * b20;
      const m15 = (a21 + a22) * (-b20 + b21);
      const m16 = (-a02 + a11 + a12) * (b12 + b20 - b22);
      const m17 = (a02 - a12) * (b12 - b22);
      const m18 = (a11 + a12) * (-b20 + b22);
      const m19 = a01 * b10;
      const m20 = a12 * b21;
      const m21 = a10 * b02;
      const m22 = a20 * b01;
      const m23 = a22 * b22;
      const c00 = m6 + m14 + m19;
      const c01 = m1 + m4 + m5 + m6 + m12 + m14 + m15;
      const c02 = m6 + m7 + m9 + m10 + m14 + m16 + m18;
      const c10 = m2 + m3 + m4 + m6 + m14 + m16 + m17;
      const c11 = m2 + m4 + m5 + m6 + m20;
      const c12 = m14 + m16 + m17 + m18 + m21;
      const c20 = m6 + m7 + m8 + m11 + m12 + m13 + m14;
      const c21 = m12 + m13 + m14 + m15 + m22;
      const c22 = m6 + m7 + m8 + m9 + m23;
      result.set(0, 0, c00);
      result.set(0, 1, c01);
      result.set(0, 2, c02);
      result.set(1, 0, c10);
      result.set(1, 1, c11);
      result.set(1, 2, c12);
      result.set(2, 0, c20);
      result.set(2, 1, c21);
      result.set(2, 2, c22);
      return result;
    }

    mmulStrassen(y) {
      y = Matrix.checkMatrix(y);
      let x = this.clone();
      let r1 = x.rows;
      let c1 = x.columns;
      let r2 = y.rows;
      let c2 = y.columns;

      if (c1 !== r2) {
        // eslint-disable-next-line no-console
        console.warn(`Multiplying ${r1} x ${c1} and ${r2} x ${c2} matrix: dimensions do not match.`);
      } // Put a matrix into the top left of a matrix of zeros.
      // `rows` and `cols` are the dimensions of the output matrix.


      function embed(mat, rows, cols) {
        let r = mat.rows;
        let c = mat.columns;

        if (r === rows && c === cols) {
          return mat;
        } else {
          let resultat = AbstractMatrix.zeros(rows, cols);
          resultat = resultat.setSubMatrix(mat, 0, 0);
          return resultat;
        }
      } // Make sure both matrices are the same size.
      // This is exclusively for simplicity:
      // this algorithm can be implemented with matrices of different sizes.


      let r = Math.max(r1, r2);
      let c = Math.max(c1, c2);
      x = embed(x, r, c);
      y = embed(y, r, c); // Our recursive multiplication function.

      function blockMult(a, b, rows, cols) {
        // For small matrices, resort to naive multiplication.
        if (rows <= 512 || cols <= 512) {
          return a.mmul(b); // a is equivalent to this
        } // Apply dynamic padding.


        if (rows % 2 === 1 && cols % 2 === 1) {
          a = embed(a, rows + 1, cols + 1);
          b = embed(b, rows + 1, cols + 1);
        } else if (rows % 2 === 1) {
          a = embed(a, rows + 1, cols);
          b = embed(b, rows + 1, cols);
        } else if (cols % 2 === 1) {
          a = embed(a, rows, cols + 1);
          b = embed(b, rows, cols + 1);
        }

        let halfRows = parseInt(a.rows / 2, 10);
        let halfCols = parseInt(a.columns / 2, 10); // Subdivide input matrices.

        let a11 = a.subMatrix(0, halfRows - 1, 0, halfCols - 1);
        let b11 = b.subMatrix(0, halfRows - 1, 0, halfCols - 1);
        let a12 = a.subMatrix(0, halfRows - 1, halfCols, a.columns - 1);
        let b12 = b.subMatrix(0, halfRows - 1, halfCols, b.columns - 1);
        let a21 = a.subMatrix(halfRows, a.rows - 1, 0, halfCols - 1);
        let b21 = b.subMatrix(halfRows, b.rows - 1, 0, halfCols - 1);
        let a22 = a.subMatrix(halfRows, a.rows - 1, halfCols, a.columns - 1);
        let b22 = b.subMatrix(halfRows, b.rows - 1, halfCols, b.columns - 1); // Compute intermediate values.

        let m1 = blockMult(AbstractMatrix.add(a11, a22), AbstractMatrix.add(b11, b22), halfRows, halfCols);
        let m2 = blockMult(AbstractMatrix.add(a21, a22), b11, halfRows, halfCols);
        let m3 = blockMult(a11, AbstractMatrix.sub(b12, b22), halfRows, halfCols);
        let m4 = blockMult(a22, AbstractMatrix.sub(b21, b11), halfRows, halfCols);
        let m5 = blockMult(AbstractMatrix.add(a11, a12), b22, halfRows, halfCols);
        let m6 = blockMult(AbstractMatrix.sub(a21, a11), AbstractMatrix.add(b11, b12), halfRows, halfCols);
        let m7 = blockMult(AbstractMatrix.sub(a12, a22), AbstractMatrix.add(b21, b22), halfRows, halfCols); // Combine intermediate values into the output.

        let c11 = AbstractMatrix.add(m1, m4);
        c11.sub(m5);
        c11.add(m7);
        let c12 = AbstractMatrix.add(m3, m5);
        let c21 = AbstractMatrix.add(m2, m4);
        let c22 = AbstractMatrix.sub(m1, m2);
        c22.add(m3);
        c22.add(m6); // Crop output to the desired size (undo dynamic padding).

        let resultat = AbstractMatrix.zeros(2 * c11.rows, 2 * c11.columns);
        resultat = resultat.setSubMatrix(c11, 0, 0);
        resultat = resultat.setSubMatrix(c12, c11.rows, 0);
        resultat = resultat.setSubMatrix(c21, 0, c11.columns);
        resultat = resultat.setSubMatrix(c22, c11.rows, c11.columns);
        return resultat.subMatrix(0, rows - 1, 0, cols - 1);
      }

      return blockMult(x, y, r, c);
    }

    scaleRows(options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        min = 0,
        max = 1
      } = options;
      if (!Number.isFinite(min)) throw new TypeError('min must be a number');
      if (!Number.isFinite(max)) throw new TypeError('max must be a number');
      if (min >= max) throw new RangeError('min must be smaller than max');
      let newMatrix = new Matrix(this.rows, this.columns);

      for (let i = 0; i < this.rows; i++) {
        const row = this.getRow(i);

        if (row.length > 0) {
          rescale(row, {
            min,
            max,
            output: row
          });
        }

        newMatrix.setRow(i, row);
      }

      return newMatrix;
    }

    scaleColumns(options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        min = 0,
        max = 1
      } = options;
      if (!Number.isFinite(min)) throw new TypeError('min must be a number');
      if (!Number.isFinite(max)) throw new TypeError('max must be a number');
      if (min >= max) throw new RangeError('min must be smaller than max');
      let newMatrix = new Matrix(this.rows, this.columns);

      for (let i = 0; i < this.columns; i++) {
        const column = this.getColumn(i);

        if (column.length) {
          rescale(column, {
            min: min,
            max: max,
            output: column
          });
        }

        newMatrix.setColumn(i, column);
      }

      return newMatrix;
    }

    flipRows() {
      const middle = Math.ceil(this.columns / 2);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < middle; j++) {
          let first = this.get(i, j);
          let last = this.get(i, this.columns - 1 - j);
          this.set(i, j, last);
          this.set(i, this.columns - 1 - j, first);
        }
      }

      return this;
    }

    flipColumns() {
      const middle = Math.ceil(this.rows / 2);

      for (let j = 0; j < this.columns; j++) {
        for (let i = 0; i < middle; i++) {
          let first = this.get(i, j);
          let last = this.get(this.rows - 1 - i, j);
          this.set(i, j, last);
          this.set(this.rows - 1 - i, j, first);
        }
      }

      return this;
    }

    kroneckerProduct(other) {
      other = Matrix.checkMatrix(other);
      let m = this.rows;
      let n = this.columns;
      let p = other.rows;
      let q = other.columns;
      let result = new Matrix(m * p, n * q);

      for (let i = 0; i < m; i++) {
        for (let j = 0; j < n; j++) {
          for (let k = 0; k < p; k++) {
            for (let l = 0; l < q; l++) {
              result.set(p * i + k, q * j + l, this.get(i, j) * other.get(k, l));
            }
          }
        }
      }

      return result;
    }

    kroneckerSum(other) {
      other = Matrix.checkMatrix(other);

      if (!this.isSquare() || !other.isSquare()) {
        throw new Error('Kronecker Sum needs two Square Matrices');
      }

      let m = this.rows;
      let n = other.rows;
      let AxI = this.kroneckerProduct(Matrix.eye(n, n));
      let IxB = Matrix.eye(m, m).kroneckerProduct(other);
      return AxI.add(IxB);
    }

    transpose() {
      let result = new Matrix(this.columns, this.rows);

      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          result.set(j, i, this.get(i, j));
        }
      }

      return result;
    }

    sortRows(compareFunction = compareNumbers) {
      for (let i = 0; i < this.rows; i++) {
        this.setRow(i, this.getRow(i).sort(compareFunction));
      }

      return this;
    }

    sortColumns(compareFunction = compareNumbers) {
      for (let i = 0; i < this.columns; i++) {
        this.setColumn(i, this.getColumn(i).sort(compareFunction));
      }

      return this;
    }

    subMatrix(startRow, endRow, startColumn, endColumn) {
      checkRange(this, startRow, endRow, startColumn, endColumn);
      let newMatrix = new Matrix(endRow - startRow + 1, endColumn - startColumn + 1);

      for (let i = startRow; i <= endRow; i++) {
        for (let j = startColumn; j <= endColumn; j++) {
          newMatrix.set(i - startRow, j - startColumn, this.get(i, j));
        }
      }

      return newMatrix;
    }

    subMatrixRow(indices, startColumn, endColumn) {
      if (startColumn === undefined) startColumn = 0;
      if (endColumn === undefined) endColumn = this.columns - 1;

      if (startColumn > endColumn || startColumn < 0 || startColumn >= this.columns || endColumn < 0 || endColumn >= this.columns) {
        throw new RangeError('Argument out of range');
      }

      let newMatrix = new Matrix(indices.length, endColumn - startColumn + 1);

      for (let i = 0; i < indices.length; i++) {
        for (let j = startColumn; j <= endColumn; j++) {
          if (indices[i] < 0 || indices[i] >= this.rows) {
            throw new RangeError(`Row index out of range: ${indices[i]}`);
          }

          newMatrix.set(i, j - startColumn, this.get(indices[i], j));
        }
      }

      return newMatrix;
    }

    subMatrixColumn(indices, startRow, endRow) {
      if (startRow === undefined) startRow = 0;
      if (endRow === undefined) endRow = this.rows - 1;

      if (startRow > endRow || startRow < 0 || startRow >= this.rows || endRow < 0 || endRow >= this.rows) {
        throw new RangeError('Argument out of range');
      }

      let newMatrix = new Matrix(endRow - startRow + 1, indices.length);

      for (let i = 0; i < indices.length; i++) {
        for (let j = startRow; j <= endRow; j++) {
          if (indices[i] < 0 || indices[i] >= this.columns) {
            throw new RangeError(`Column index out of range: ${indices[i]}`);
          }

          newMatrix.set(j - startRow, i, this.get(j, indices[i]));
        }
      }

      return newMatrix;
    }

    setSubMatrix(matrix, startRow, startColumn) {
      matrix = Matrix.checkMatrix(matrix);

      if (matrix.isEmpty()) {
        return this;
      }

      let endRow = startRow + matrix.rows - 1;
      let endColumn = startColumn + matrix.columns - 1;
      checkRange(this, startRow, endRow, startColumn, endColumn);

      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          this.set(startRow + i, startColumn + j, matrix.get(i, j));
        }
      }

      return this;
    }

    selection(rowIndices, columnIndices) {
      let indices = checkIndices(this, rowIndices, columnIndices);
      let newMatrix = new Matrix(rowIndices.length, columnIndices.length);

      for (let i = 0; i < indices.row.length; i++) {
        let rowIndex = indices.row[i];

        for (let j = 0; j < indices.column.length; j++) {
          let columnIndex = indices.column[j];
          newMatrix.set(i, j, this.get(rowIndex, columnIndex));
        }
      }

      return newMatrix;
    }

    trace() {
      let min = Math.min(this.rows, this.columns);
      let trace = 0;

      for (let i = 0; i < min; i++) {
        trace += this.get(i, i);
      }

      return trace;
    }

    clone() {
      let newMatrix = new Matrix(this.rows, this.columns);

      for (let row = 0; row < this.rows; row++) {
        for (let column = 0; column < this.columns; column++) {
          newMatrix.set(row, column, this.get(row, column));
        }
      }

      return newMatrix;
    }

    sum(by) {
      switch (by) {
        case 'row':
          return sumByRow(this);

        case 'column':
          return sumByColumn(this);

        case undefined:
          return sumAll(this);

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    product(by) {
      switch (by) {
        case 'row':
          return productByRow(this);

        case 'column':
          return productByColumn(this);

        case undefined:
          return productAll(this);

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    mean(by) {
      const sum = this.sum(by);

      switch (by) {
        case 'row':
          {
            for (let i = 0; i < this.rows; i++) {
              sum[i] /= this.columns;
            }

            return sum;
          }

        case 'column':
          {
            for (let i = 0; i < this.columns; i++) {
              sum[i] /= this.rows;
            }

            return sum;
          }

        case undefined:
          return sum / this.size;

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    variance(by, options = {}) {
      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }

      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        unbiased = true,
        mean = this.mean(by)
      } = options;

      if (typeof unbiased !== 'boolean') {
        throw new TypeError('unbiased must be a boolean');
      }

      switch (by) {
        case 'row':
          {
            if (!Array.isArray(mean)) {
              throw new TypeError('mean must be an array');
            }

            return varianceByRow(this, unbiased, mean);
          }

        case 'column':
          {
            if (!Array.isArray(mean)) {
              throw new TypeError('mean must be an array');
            }

            return varianceByColumn(this, unbiased, mean);
          }

        case undefined:
          {
            if (typeof mean !== 'number') {
              throw new TypeError('mean must be a number');
            }

            return varianceAll(this, unbiased, mean);
          }

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    standardDeviation(by, options) {
      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }

      const variance = this.variance(by, options);

      if (by === undefined) {
        return Math.sqrt(variance);
      } else {
        for (let i = 0; i < variance.length; i++) {
          variance[i] = Math.sqrt(variance[i]);
        }

        return variance;
      }
    }

    center(by, options = {}) {
      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }

      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      const {
        center = this.mean(by)
      } = options;

      switch (by) {
        case 'row':
          {
            if (!Array.isArray(center)) {
              throw new TypeError('center must be an array');
            }

            centerByRow(this, center);
            return this;
          }

        case 'column':
          {
            if (!Array.isArray(center)) {
              throw new TypeError('center must be an array');
            }

            centerByColumn(this, center);
            return this;
          }

        case undefined:
          {
            if (typeof center !== 'number') {
              throw new TypeError('center must be a number');
            }

            centerAll(this, center);
            return this;
          }

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    scale(by, options = {}) {
      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }

      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }

      let scale = options.scale;

      switch (by) {
        case 'row':
          {
            if (scale === undefined) {
              scale = getScaleByRow(this);
            } else if (!Array.isArray(scale)) {
              throw new TypeError('scale must be an array');
            }

            scaleByRow(this, scale);
            return this;
          }

        case 'column':
          {
            if (scale === undefined) {
              scale = getScaleByColumn(this);
            } else if (!Array.isArray(scale)) {
              throw new TypeError('scale must be an array');
            }

            scaleByColumn(this, scale);
            return this;
          }

        case undefined:
          {
            if (scale === undefined) {
              scale = getScaleAll(this);
            } else if (typeof scale !== 'number') {
              throw new TypeError('scale must be a number');
            }

            scaleAll(this, scale);
            return this;
          }

        default:
          throw new Error(`invalid option: ${by}`);
      }
    }

    toString(options) {
      return inspectMatrixWithOptions(this, options);
    }

  }
  AbstractMatrix.prototype.klass = 'Matrix';

  if (typeof Symbol !== 'undefined') {
    AbstractMatrix.prototype[Symbol.for('nodejs.util.inspect.custom')] = inspectMatrix;
  }

  function compareNumbers(a, b) {
    return a - b;
  } // Synonyms


  AbstractMatrix.random = AbstractMatrix.rand;
  AbstractMatrix.randomInt = AbstractMatrix.randInt;
  AbstractMatrix.diagonal = AbstractMatrix.diag;
  AbstractMatrix.prototype.diagonal = AbstractMatrix.prototype.diag;
  AbstractMatrix.identity = AbstractMatrix.eye;
  AbstractMatrix.prototype.negate = AbstractMatrix.prototype.neg;
  AbstractMatrix.prototype.tensorProduct = AbstractMatrix.prototype.kroneckerProduct;
  class Matrix extends AbstractMatrix {
    constructor(nRows, nColumns) {
      super();

      if (Matrix.isMatrix(nRows)) {
        // eslint-disable-next-line no-constructor-return
        return nRows.clone();
      } else if (Number.isInteger(nRows) && nRows >= 0) {
        // Create an empty matrix
        this.data = [];

        if (Number.isInteger(nColumns) && nColumns >= 0) {
          for (let i = 0; i < nRows; i++) {
            this.data.push(new Float64Array(nColumns));
          }
        } else {
          throw new TypeError('nColumns must be a positive integer');
        }
      } else if (Array.isArray(nRows)) {
        // Copy the values from the 2D array
        const arrayData = nRows;
        nRows = arrayData.length;
        nColumns = nRows ? arrayData[0].length : 0;

        if (typeof nColumns !== 'number') {
          throw new TypeError('Data must be a 2D array with at least one element');
        }

        this.data = [];

        for (let i = 0; i < nRows; i++) {
          if (arrayData[i].length !== nColumns) {
            throw new RangeError('Inconsistent array dimensions');
          }

          this.data.push(Float64Array.from(arrayData[i]));
        }
      } else {
        throw new TypeError('First argument must be a positive number or an array');
      }

      this.rows = nRows;
      this.columns = nColumns;
    }

    set(rowIndex, columnIndex, value) {
      this.data[rowIndex][columnIndex] = value;
      return this;
    }

    get(rowIndex, columnIndex) {
      return this.data[rowIndex][columnIndex];
    }

    removeRow(index) {
      checkRowIndex(this, index);
      this.data.splice(index, 1);
      this.rows -= 1;
      return this;
    }

    addRow(index, array) {
      if (array === undefined) {
        array = index;
        index = this.rows;
      }

      checkRowIndex(this, index, true);
      array = Float64Array.from(checkRowVector(this, array));
      this.data.splice(index, 0, array);
      this.rows += 1;
      return this;
    }

    removeColumn(index) {
      checkColumnIndex(this, index);

      for (let i = 0; i < this.rows; i++) {
        const newRow = new Float64Array(this.columns - 1);

        for (let j = 0; j < index; j++) {
          newRow[j] = this.data[i][j];
        }

        for (let j = index + 1; j < this.columns; j++) {
          newRow[j - 1] = this.data[i][j];
        }

        this.data[i] = newRow;
      }

      this.columns -= 1;
      return this;
    }

    addColumn(index, array) {
      if (typeof array === 'undefined') {
        array = index;
        index = this.columns;
      }

      checkColumnIndex(this, index, true);
      array = checkColumnVector(this, array);

      for (let i = 0; i < this.rows; i++) {
        const newRow = new Float64Array(this.columns + 1);
        let j = 0;

        for (; j < index; j++) {
          newRow[j] = this.data[i][j];
        }

        newRow[j++] = array[i];

        for (; j < this.columns + 1; j++) {
          newRow[j] = this.data[i][j - 1];
        }

        this.data[i] = newRow;
      }

      this.columns += 1;
      return this;
    }

  }
  installMathOperations(AbstractMatrix, Matrix);

  /**
   * Calibrates the data based on a range and means of peaks in this range
   * Based on a range we will make a peak picking using global spectra deconvolution
   * The selected nbPeaks will then be taken into account to calculate an average X value.
   * The difference between the targetX and the averageX value will be returned
   * @param {DataXY} [data] array of points {x,y}
   * @param {Object} [range={}]
   * @param {number} [range.from] - Beginning of the range where the interest signal is localed
   * @param {number} [range.to] - End of the range where the interest signal is localed
   * @param {Object} [options={}]
   * @param {number} [options.nbPeaks=1] Number of peaks to consider to calculate mean (sorted by height)
   * @param {number} [options.targetX=0] Expected value for the mean of the peaks position
   * @param {number} [options.gsd={}] GSD options. You may check options here: http://mljs.github.io/global-spectral-deconvolution/#gsd
   * @param {number} [options.gsd.minMaxRatio=0.2] - GSD Threshold to determine if a given peak should be considered as a noise.
   * @returns {number} difference between targetX and currentX
   */

  function xyCalibrate(data, range = {}, options = {}) {
    const {
      targetX = 0,
      nbPeaks = 1,
      gsd: gsdOptions = {
        minMaxRatio: 0.1,
        realTopDetection: true,
        smoothY: true,
        sgOptions: {
          windowSize: 7,
          polynomial: 3
        }
      }
    } = options;
    let {
      from,
      to
    } = range;
    if (from === undefined || to === undefined) return 0;
    const fromIndex = xFindClosestIndex(data.x, from);
    const toIndex = xFindClosestIndex(data.x, to);
    const sliceddata = {
      x: data.x.slice(fromIndex, toIndex),
      y: data.y.slice(fromIndex, toIndex)
    };
    let peaks = gsd(sliceddata, gsdOptions).sort((a, b) => b.y - a.y).slice(0, nbPeaks);
    if (peaks.length === 0) return 0;
    const middle = mean(peaks.map(peak => peak.x));
    return targetX - middle;
  }

  /**
   *This function performs a quick sort of the x array while transforming the y array to preserve the coordinates.
   * @param {DataXY} [data] Object that contains property x (Array) and y (Array)
   */
  function xySortX(data) {
    const {
      x,
      y
    } = data;
    let xyObject = x.map((val, index) => ({
      x: val,
      y: y[index]
    })).sort((a, b) => a.x - b.x);
    let response = {
      x: [],
      y: []
    };

    for (let i = 0; i < x.length; i++) {
      response.x.push(xyObject[i].x);
      response.y.push(xyObject[i].y);
    }

    return response;
  }

  /**
   * Ensure x values are unique
   * @param {DataXY} [data] Object that contains property x (Array) and y (Array)
   * @param {Object} [options={}] Object containing a property algorithm (can be 'sum' or 'average', the latter being the default value), and a property isSorted (boolean indicating if the x-array is sorted).
   * @param {string} [options.algorithm='average'] either 'average' or 'sum'
   * @param {boolean} [options.isSorted=true] if false the DataXY has to be sorted first
   * @returns {DataXY}
   */

  function xyUniqueX(data, options = {}) {
    xyCheck(data);
    const {
      algorithm = 'average',
      isSorted = true
    } = options;

    if (!isSorted) {
      data = xySortX(data);
    }

    switch (algorithm) {
      case 'average':
        return average(data);

      case 'sum':
        return sum(data);

      default:
        throw new Error(`xyUniqueX: unknown algorithm: ${algorithm}`);
    }
  }

  function average(data) {
    let x = [];
    let y = [];
    let cumulativeY = data.y[0];
    let divider = 1;

    for (let i = 1; i < data.x.length; i++) {
      if (!(data.x[i] === data.x[i - 1])) {
        x.push(data.x[i - 1]);
        y.push(cumulativeY / divider);
        cumulativeY = 0;
        divider = 0;
      }

      cumulativeY += data.y[i];
      divider++;
    }

    x.push(data.x[data.x.length - 1]);
    y.push(cumulativeY / divider);
    return {
      x,
      y
    };
  }

  function sum(data) {
    let x = [];
    let y = [];
    let cumulativeY = data.y[0];

    for (let i = 1; i < data.x.length; i++) {
      if (!(data.x[i] === data.x[i - 1])) {
        x.push(data.x[i - 1]);
        y.push(cumulativeY);
        cumulativeY = 0;
      }

      cumulativeY += data.y[i];
    }

    x.push(data.x[data.x.length - 1]);
    y.push(cumulativeY);
    return {
      x,
      y
    };
  }

  /**
   * Throw an error in no an object of x,y arrays
   * @param {ArrayPoints} [points=[]]
   */
  function xyObjectCheck(points = []) {
    if (!Array.isArray(points)) {
      throw new Error('ArrayPoints must be an array of {x,y} object');
    }

    if (points.length > 0 && (points[0].x === undefined || points[0].y === undefined)) {
      throw new Error('ArrayPoints must be an array of {x,y} object');
    }
  }

  /**
   * Finds the max x value and return a {x,y,index} point
   * @param {DataXY} [points=[]] - Object that contains property x (an ordered increasing array) and y (an array)
   * @return {object}
   */

  function xyObjectMaxXPoint(points = []) {
    xyObjectCheck(points);
    if (points.length < 1) return {};
    let current = {
      x: points[0].x,
      y: points[0].y,
      index: 0
    };

    for (let i = 1; i < points.length; i++) {
      if (points[i].x > current.x) {
        current = {
          x: points[i].x,
          y: points[i].y,
          index: i
        };
      }
    }

    return current;
  }

  /**
   * Finds the min x value and return a {x,y,index} point
   * @param {DataXY} [points=[]] - Object that contains property x (an ordered increasing array) and y (an array)
   * @return {object}
   */

  function xyObjectMinXPoint(points = []) {
    xyObjectCheck(points);
    if (points.length < 1) return {};
    let current = {
      x: points[0].x,
      y: points[0].y,
      index: 0
    };

    for (let i = 1; i < points.length; i++) {
      if (points[i].x < current.x) {
        current = {
          x: points[i].x,
          y: points[i].y,
          index: i
        };
      }
    }

    return current;
  }

  /**
   * Filter the array by taking the higher points (max y value) and only
   * keep one per slot.
   * There are 2 different slots, the smallest one will have the
   * new property `close` to true
   * @param {array} points - array of all the points
   * @param {object} [options={}]
   * @param {number} [options.from] - min X value of the window to consider
   * @param {number} [options.to] - max X value of the window to consider
   * @param {number} [options.limit=20] - max number of points
   * @param {number} [options.threshold=0.01] - minimal intensity compare to more intense point
   * @param {number} [options.numberSlots=10] - define the number of slots and indirectly the slot width
   * @param {number} [options.numberCloseSlots=50]
   * @returns {array} - copy of points with 'close' property
   */

  function xyObjectBestPoints(points, options = {}) {
    const {
      from = xyObjectMinXPoint(points).x,
      to = xyObjectMaxXPoint(points).x,
      limit = 20,
      threshold = 0.01,
      numberCloseSlots = 50,
      numberSlots = 10
    } = options;
    let slot = (to - from) / numberSlots;
    let closeSlot = (to - from) / numberCloseSlots;
    let selected = points.filter(point => point.x >= from && point.x <= to).map(point => {
      return {
        point,
        monoisotopic: false
      };
    });
    selected = selected.sort((a, b) => {
      if (a.monoisotopic && !b.monoisotopic) return -1;
      if (b.monoisotopic && !a.monoisotopic) return 1;
      return b.point.y - a.point.y;
    });
    let toReturn = [];
    if (selected.length === 0) return [];
    let minY = selected[0].point.y * threshold;

    peakLoop: for (let item of selected) {
      if (item.point.y < minY) {
        if (item.monoisotopic) {
          continue;
        } else {
          break;
        }
      }

      let close = false;

      for (let existing of toReturn) {
        if (Math.abs(existing.x - item.point.x) < closeSlot) {
          continue peakLoop;
        }

        if (Math.abs(existing.x - item.point.x) < slot) {
          close = true;
        }
      }

      let newPeak = JSON.parse(JSON.stringify(item.point));
      newPeak.close = close;
      toReturn.push(newPeak);
      if (toReturn.length === limit) break;
    }

    return toReturn.sort((a, b) => a.x - b.x);
  }

  /**
   *
   * @param {ArrayPoints} [points] array of growing points {x,y}
   * @param {object} [options={}]
   * @param {object} [xError=Number.EPSILON] limit to join the data
   */
  function xyObjectJoinX(points, options = {}) {
    const {
      xError = Number.EPSILON
    } = options; // when we join we will use the center of mass

    let result = [];
    let current = {
      x: Number.MIN_SAFE_INTEGER,
      y: 0
    };

    for (let point of points) {
      if (point.x - current.x <= xError) {
        // weighted sum
        if (current.y !== 0 || point.y !== 0) {
          current.x = point.y / (current.y + point.y) * (point.x - current.x) + current.x;
          current.y += point.y;
        }
      } else {
        current = {
          x: point.x,
          y: point.y
        };
        result.push(current);
      }
    }

    return result;
  }

  /**
   * Finds the max y value and return a {x,y,index} point
   * @param {DataXY} [points=[]] - Object that contains property x (an ordered increasing array) and y (an array)
   * @return {object}
   */

  function xyObjectMaxYPoint(points = []) {
    xyObjectCheck(points);
    if (points.length < 1) return {};
    let current = {
      x: points[0].x,
      y: points[0].y,
      index: 0
    };

    for (let i = 1; i < points.length; i++) {
      if (points[i].y > current.y) {
        current = {
          x: points[i].x,
          y: points[i].y,
          index: i
        };
      }
    }

    return current;
  }

  /**
   * Finds the min y value and return a {x,y,index} point
   * @param {DataXY} [points=[]] - Object that contains property x (an ordered increasing array) and y (an array)
   * @return {object}
   */

  function xyObjectMinYPoint(points = []) {
    xyObjectCheck(points);
    if (points.length < 1) return {};
    let current = {
      x: points[0].x,
      y: points[0].y,
      index: 0
    };

    for (let i = 1; i < points.length; i++) {
      if (points[i].y < current.y) {
        current = {
          x: points[i].x,
          y: points[i].y,
          index: i
        };
      }
    }

    return current;
  }

  /**
   *
   * @param {ArrayPoints} [points] array of growing points {x,y}
   * @param {object} [options={}]
   * @param {object} [slotWidth=1] limit to join the data
   */
  function xyObjectSlotX(points, options = {}) {
    const {
      slotWidth = 1
    } = options;
    const halfSlot = slotWidth / 2; // when we join we will use the center of mass

    let result = [];
    let current = {
      x: Number.MIN_VALUE,
      y: 0
    };

    for (let point of points) {
      let slot = point.x - (point.x + halfSlot) % slotWidth + halfSlot;

      if (Math.abs(current.x - slot) > Number.EPSILON) {
        current = {
          x: slot,
          y: 0
        };
        result.push(current);
      }

      current.y += point.y;
    }

    return result;
  }

  /**
   * Sorts an array of points
   * @param {ArrayPoints} [points] array of points {x,y}
   */
  function xyObjectSortX(points) {
    return points.sort((a, b) => a.x - b.x);
  }

  /**
   *
   * @param {ArrayPoints} [points] array of points {x,y}
   */
  function xyObjectToXY(points) {
    return {
      x: points.map(entry => entry.x),
      y: points.map(entry => entry.y)
    };
  }

  function zoneToX(zone, size) {
    const {
      from,
      to
    } = zone;
    let array = new Float64Array(size);
    let step = (to - from) / (size - 1);

    for (let i = 0; i < size; i++) {
      array[i] = from + step * i;
    }

    return array;
  }

  /**
   * Center mean of columns
   * @param {Array<Array<Number>>} [matrix] - matrix [rows][cols].
   */
  function matrixCenterZMean(matrix) {
    const nbRows = matrix.length;
    const nbColumns = matrix[0].length;
    const newMatrix = new Array(nbRows);

    for (let row = 0; row < nbRows; row++) {
      newMatrix[row] = new Float64Array(nbColumns);
    }

    for (let column = 0; column < nbColumns; column++) {
      let mean = 0;

      for (let row = 0; row < nbRows; row++) {
        mean += matrix[row][column];
      }

      mean /= nbRows;

      for (let row = 0; row < nbRows; row++) {
        newMatrix[row][column] = matrix[row][column] - mean;
      }
    }

    return newMatrix;
  }

  /**
   * Get min and max Z
   * @param {Array<Array<Number>>} [matrix] - matrix [rows][cols].
   *
   */
  function matrixMinMaxZ(matrix) {
    if (matrix.length === 0 || matrix[0].length === 0) {
      return {
        min: undefined,
        max: undefined
      };
    }

    const nbRows = matrix.length;
    const nbColumns = matrix[0].length;
    let min = matrix[0][0];
    let max = matrix[0][0];

    for (let column = 0; column < nbColumns; column++) {
      for (let row = 0; row < nbRows; row++) {
        if (matrix[row][column] < min) min = matrix[row][column];
        if (matrix[row][column] > max) max = matrix[row][column];
      }
    }

    return {
      min,
      max
    };
  }

  /**
  import { matrixMinMaxZ } from './matrixMinMaxZ';
  import { xHistogram } from '../x/xHistogram';
   * Calculates an histogram of defined number of slots
   * @param {Array<Array<Number>>} [matrix] - matrix [rows][cols].
   * @param {number} [options.nbSlots=256] Number of slots
   * @param {number} [options.min=minValue] Minimum value to calculate used to calculate slot size
   * @param {number} [options.max=maxValue] Maximal value to calculate used to calculate slot size
   * @param {number} [options.logBaseX] We can first apply a log on x axi
   * @param {number} [options.logBaseY] We can apply a log on the resulting histogra
   * @param {number} [options.centerX=true] Center the X value. We will enlarge the first and
   * @return {DataXY} {x,y} of the histogram
   *
   */

  function matrixHistogram(matrix, options = {}) {
    const {
      logBaseY
    } = options;
    options = JSON.parse(JSON.stringify(options));
    delete options.logBaseY;

    if (matrix.length === 0 || matrix[0].length === 0) {
      throw new Error('matrixHistogram: matrix should have at least one column and one row');
    }

    if (options.min === undefined || options.max === undefined) {
      const minMax = matrixMinMaxZ(matrix);
      if (options.min === undefined) options.min = minMax.min;
      if (options.max === undefined) options.max = minMax.max;
    }

    let histogram = xHistogram(matrix[0], options);
    options.histogram = histogram;
    const nbRows = matrix.length;

    for (let row = 1; row < nbRows; row++) {
      xHistogram(matrix[row], options);
    }

    const y = histogram.y;

    if (logBaseY) {
      const logOfBase = Math.log10(logBaseY);

      for (let i = 0; i < y.length; i++) {
        y[i] = Math.log10(y[i]) / logOfBase;
      }
    }

    return histogram;
  }

  /**
   * Performs a Probabilistic quotient normalization (PQN) over the dataset to account dilution based in median spectrum.
   * Dieterle, F., Ross, A., Schlotterbeck, G., & Senn, H. (2006). Probabilistic quotient normalization as robust method to account for dilution of complex biological mixtures. Application in 1H NMR metabonomics. Analytical chemistry, 78(13), 4281-4290.
   * DOI: 10.1021/ac051632c
   * @param {Array<Array<Number>>} [matrix] - matrix [rows][cols].
   * @param {Object} [options={}]
   * @param {number} [options.max=100] - Normalization integral constant.
   * @return {Object} { data, medianOfQuotients }.
   * data: Normalized dataset.
   * medianOfQuotients: The median of quotients of each variables.
   */

  function matrixPQN(matrix, options = {}) {
    const {
      max = 100
    } = options;
    matrix = new Matrix(matrix);

    for (let i = 0; i < matrix.rows; i++) {
      const normalizationFactor = matrix.getRowVector(i).norm() / max;
      const row = matrix.getRowVector(i).div(normalizationFactor);
      matrix.setRow(i, row);
    }

    let referenceSpectrum = [];

    for (let i = 0; i < matrix.columns; i++) {
      const currentVariable = matrix.getColumn(i);
      referenceSpectrum.push(median(currentVariable));
    }

    let medianOfQuotients = [];

    for (let i = 0; i < matrix.columns; i++) {
      let quotients = matrix.getColumnVector(i).div(referenceSpectrum[i]);
      medianOfQuotients.push(median(quotients.getColumn(0)));
    }

    for (let i = 0; i < matrix.rows; i++) {
      matrix.mulRow(i, 1 / medianOfQuotients[i]);
    }

    return {
      data: matrix.to2DArray(),
      medianOfQuotients: medianOfQuotients
    };
  }

  /**
   * Rescale columns
   * @param {Array<Array<Number>>} [matrix] - matrix [rows][cols].
   * @param {object} [options={}]
   * @param {object} [options.min=0]
   * @param {object} [options.max=1]
   *
   */
  function matrixZRescale(matrix, options = {}) {
    const {
      min = 0,
      max = 1
    } = options;
    const nbRows = matrix.length;
    const nbColumns = matrix[0].length;
    const newMatrix = new Array(nbRows);

    for (let row = 0; row < nbRows; row++) {
      newMatrix[row] = new Float64Array(nbColumns);
    }

    for (let column = 0; column < nbColumns; column++) {
      let currentMin = matrix[0][column];
      let currentMax = matrix[0][column];

      for (let row = 1; row < nbRows; row++) {
        if (matrix[row][column] < currentMin) currentMin = matrix[row][column];
        if (matrix[row][column] > currentMax) currentMax = matrix[row][column];
      }

      const factor = (max - min) / (currentMax - currentMin);

      for (let row = 0; row < nbRows; row++) {
        newMatrix[row][column] = (matrix[row][column] - currentMin) * factor + min;
      }
    }

    return newMatrix;
  }

  /**
   * Create an array with sequential numbers between from and to of length
   * @param {number} [options.from=0]
   * @param {number} [options.to=1]
   * @param {number} [options.length=1001]
   */
  function createSequentialArray(options = {}) {
    const {
      from = 0,
      to = 1,
      length = 1000
    } = options;
    const array = new Float64Array(length);
    let step = (to - from) / (array.length - 1);

    for (let i = 0; i < array.length; i++) {
      array[i] = from + step * i;
    }

    return array;
  }

  exports.createSequentialArray = createSequentialArray;
  exports.matrixCenterZMean = matrixCenterZMean;
  exports.matrixHistogram = matrixHistogram;
  exports.matrixMinMaxZ = matrixMinMaxZ;
  exports.matrixPQN = matrixPQN;
  exports.matrixZRescale = matrixZRescale;
  exports.reimAbsolute = reimAbsolute;
  exports.reimAutoPhaseCorrection = reimAutoPhaseCorrection;
  exports.reimFFT = reimFFT;
  exports.reimPhaseCorrection = reimPhaseCorrection;
  exports.xAbsolute = xAbsolute;
  exports.xAbsoluteMedian = xAbsoluteMedian;
  exports.xAdd = xAdd;
  exports.xAutoCorrelation = xAutoCorrelation;
  exports.xBoxPlot = xBoxPlot;
  exports.xCorrelation = xCorrelation;
  exports.xCrossCorrelation = xCrossCorrelation;
  exports.xCumulative = xCumulative;
  exports.xDivide = xDivide;
  exports.xDotProduct = xDotProduct;
  exports.xFindClosestIndex = xFindClosestIndex;
  exports.xGetFromToIndex = xGetFromToIndex;
  exports.xGetTargetIndex = xGetTargetIndex;
  exports.xHistogram = xHistogram;
  exports.xIsMonotone = xIsMonotone;
  exports.xMaxIndex = xMaxIndex;
  exports.xMaxValue = xMaxValue;
  exports.xMean = xMean;
  exports.xMeanAbsoluteError = xMeanAbsoluteError;
  exports.xMeanSquaredError = xMeanSquaredError;
  exports.xMinIndex = xMinIndex;
  exports.xMinValue = xMinValue;
  exports.xMultiply = xMultiply;
  exports.xNoiseSanPlot = xNoiseSanPlot;
  exports.xNorm = xNorm;
  exports.xPadding = xPadding;
  exports.xParetoNormalization = xParetoNormalization;
  exports.xRolling = xRolling;
  exports.xRollingAverage = xRollingAverage;
  exports.xRollingMax = xRollingMax;
  exports.xRollingMedian = xRollingMedian;
  exports.xRollingMin = xRollingMin;
  exports.xRotate = xRotate;
  exports.xSubtract = xSubtract;
  exports.xSum = xSum;
  exports.xreimSortX = xreimSortX;
  exports.xreimZeroFilling = xreimZeroFilling;
  exports.xyAlign = xyAlign;
  exports.xyArrayAlign = xyArrayAlign;
  exports.xyArrayMerge = xyArrayMerge;
  exports.xyCalibrate = xyCalibrate;
  exports.xyCheck = xyCheck;
  exports.xyCumulativeDistributionStatistics = xyCumulativeDistributionStatistics;
  exports.xyEnsureGrowingX = xyEnsureGrowingX;
  exports.xyExtract = xyExtract;
  exports.xyFilterXPositive = xyFilterXPositive;
  exports.xyGetNMaxY = xyGetNMaxY;
  exports.xyGrowingX = xyGrowingX;
  exports.xyIntegral = xyIntegral;
  exports.xyIntegration = xyIntegration;
  exports.xyJoinX = xyJoinX;
  exports.xyMaxClosestYPoint = xyMaxClosestYPoint;
  exports.xyMaxY = xyMaxY;
  exports.xyMaxYPoint = xyMaxYPoint;
  exports.xyMaximaY = xyMaximaY;
  exports.xyMinClosestYPoint = xyMinClosestYPoint;
  exports.xyMinYPoint = xyMinYPoint;
  exports.xyMinimaY = xyMinimaY;
  exports.xyObjectBestPoints = xyObjectBestPoints;
  exports.xyObjectJoinX = xyObjectJoinX;
  exports.xyObjectMaxXPoint = xyObjectMaxXPoint;
  exports.xyObjectMaxYPoint = xyObjectMaxYPoint;
  exports.xyObjectMinXPoint = xyObjectMinXPoint;
  exports.xyObjectMinYPoint = xyObjectMinYPoint;
  exports.xyObjectSlotX = xyObjectSlotX;
  exports.xyObjectSortX = xyObjectSortX;
  exports.xyObjectToXY = xyObjectToXY;
  exports.xyPeakInfo = xyPeakInfo;
  exports.xyRealMaxYPoint = xyRealMaxYPoint;
  exports.xyRealMinYPoint = xyRealMinYPoint;
  exports.xyReduce = xyReduce;
  exports.xyRolling = xyRolling;
  exports.xySortX = xySortX;
  exports.xyToXYArray = xyToXYArray;
  exports.xyToXYObject = xyToXYObject;
  exports.xyUniqueX = xyUniqueX;
  exports.zoneToX = zoneToX;
  exports.zonesNormalize = zonesNormalize;

  Object.defineProperty(exports, '__esModule', { value: true });

})));
//# sourceMappingURL=ml-spectra-processing.js.map
