/**
 * fifo-logger - Simple event logger for the browser and node.js
 * @version v0.7.0
 * @link https://github.com/cheminfo/fifo-logger#readme
 * @license MIT
 */
(function (global, factory) {
	typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
	typeof define === 'function' && define.amd ? define(['exports'], factory) :
	(global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.FifoLogger = {}));
})(this, (function (exports) { 'use strict';

	var IDX = 256,
	  HEX = [],
	  BUFFER;
	while (IDX--) HEX[IDX] = (IDX + 256).toString(16).substring(1);
	function v4() {
	  var i = 0,
	    num,
	    out = '';
	  if (!BUFFER || IDX + 16 > 256) {
	    BUFFER = Array(i = 256);
	    while (i--) BUFFER[i] = 256 * Math.random() | 0;
	    i = IDX = 0;
	  }
	  for (; i < 16; i++) {
	    num = BUFFER[IDX + i];
	    if (i == 6) out += HEX[num & 15 | 64];else if (i == 8) out += HEX[num & 63 | 128];else out += HEX[num];
	    if (i & 1 && i > 1 && i < 11) out += '-';
	  }
	  IDX++;
	  return out;
	}

	var e = class extends EventTarget {
	  dispatchTypedEvent(s, t) {
	    return super.dispatchEvent(t);
	  }
	};

	// compatibility for CustomEvent in Node 16 and 18
	// todo should be removed in 2026 ...
	// https://stackoverflow.com/questions/69791262/how-do-i-send-a-customevent-via-a-node-16-eventtarget
	class CustomEvent extends Event {
	  constructor(message, data) {
	    super(message, data);
	    this.detail = data.detail;
	  }
	}

	class LogEvent extends CustomEvent {
	  constructor(data) {
	    super('log', {
	      detail: data
	    });
	  }
	}

	const levels = {
	  values: {
	    fatal: 60,
	    error: 50,
	    warn: 40,
	    info: 30,
	    debug: 20,
	    trace: 10,
	    silent: 0
	  },
	  labels: {
	    0: 'silent',
	    10: 'trace',
	    20: 'debug',
	    30: 'info',
	    40: 'warn',
	    50: 'error',
	    60: 'fatal'
	  }
	};

	var __classPrivateFieldGet = undefined && undefined.__classPrivateFieldGet || function (receiver, state, kind, f) {
	  if (kind === "a" && !f) throw new TypeError("Private accessor was defined without a getter");
	  if (typeof state === "function" ? receiver !== state || !f : !state.has(receiver)) throw new TypeError("Cannot read private member from an object whose class did not declare it");
	  return kind === "m" ? f : kind === "a" ? f.call(receiver) : f ? f.value : state.get(receiver);
	};
	var _FifoLogger_instances, _FifoLogger_addEvent;
	/**
	 * A FIFO logger that stores the last events in an array.
	 */
	class FifoLogger extends e {
	  constructor(options = {}) {
	    super();
	    _FifoLogger_instances.add(this);
	    this.lastID = {
	      id: 0
	    };
	    this.initialOptions = options;
	    this.uuids = [v4()];
	    this.events = [];
	    this.level = options.level || 'info';
	    this.levelAsNumber = levels.values[this.level];
	    this.limit = options.limit ?? 1000;
	    this.bindings = options.bindings ?? {};
	    this.onChange = options.onChange;
	  }
	  setLevel(level) {
	    this.level = level;
	    this.levelAsNumber = levels.values[level];
	  }
	  setLimit(limit) {
	    this.limit = limit;
	    this.checkSize();
	  }
	  checkSize() {
	    if (this.events.length > this.limit) {
	      this.events.splice(0, this.events.length - this.limit);
	    }
	  }
	  /**
	   * Remove events from the current logger and its children.
	   * @param options
	   */
	  clear() {
	    for (let i = this.events.length - 1; i >= 0; i--) {
	      if (this.events[i].uuids.includes(this.uuids[0])) {
	        this.events.splice(i, 1);
	      }
	    }
	    this.onChange?.(undefined, this.events, {
	      depth: this.uuids.length
	    });
	  }
	  getLogs(options = {}) {
	    const {
	      level,
	      minLevel,
	      includeChildren
	    } = options;
	    let logs = this.events.slice();
	    if (includeChildren) {
	      logs = logs.filter(log => log.uuids.includes(this.uuids[0]));
	    } else {
	      logs = logs.filter(log => log.uuids[0] === this.uuids[0]);
	    }
	    if (level) {
	      const levelNumber = Number(levels.values[level]);
	      if (Number.isNaN(levelNumber)) {
	        throw new Error('Invalid level');
	      }
	      logs = logs.filter(log => log.level === levelNumber);
	    }
	    if (minLevel) {
	      const levelNumber = Number(levels.values[minLevel]);
	      if (Number.isNaN(levelNumber)) {
	        throw new Error('Invalid level');
	      }
	      logs = logs.filter(log => log.level >= levelNumber);
	    }
	    return logs;
	  }
	  /**
	   * @param bindings: an object of key-value pairs to include in log lines as properties.
	   * @param options: an options object that will override child logger inherited options.
	   */
	  child(bindings) {
	    const newFifoLogger = new FifoLogger(this.initialOptions);
	    newFifoLogger.events = this.events;
	    newFifoLogger.uuids = [v4(), ...this.uuids];
	    newFifoLogger.lastID = this.lastID;
	    newFifoLogger.bindings = {
	      ...this.bindings,
	      ...bindings
	    };
	    return newFifoLogger;
	  }
	  trace(value, message) {
	    __classPrivateFieldGet(this, _FifoLogger_instances, "m", _FifoLogger_addEvent).call(this, levels.values.trace, value, message);
	  }
	  debug(value, message) {
	    __classPrivateFieldGet(this, _FifoLogger_instances, "m", _FifoLogger_addEvent).call(this, levels.values.debug, value, message);
	  }
	  info(value, message) {
	    __classPrivateFieldGet(this, _FifoLogger_instances, "m", _FifoLogger_addEvent).call(this, levels.values.info, value, message);
	  }
	  warn(value, message) {
	    __classPrivateFieldGet(this, _FifoLogger_instances, "m", _FifoLogger_addEvent).call(this, levels.values.warn, value, message);
	  }
	  error(value, message) {
	    __classPrivateFieldGet(this, _FifoLogger_instances, "m", _FifoLogger_addEvent).call(this, levels.values.error, value, message);
	  }
	  fatal(value, message) {
	    __classPrivateFieldGet(this, _FifoLogger_instances, "m", _FifoLogger_addEvent).call(this, levels.values.fatal, value, message);
	  }
	}
	_FifoLogger_instances = new WeakSet(), _FifoLogger_addEvent = function _FifoLogger_addEvent(level, value, message) {
	  if (level < this.levelAsNumber) return;
	  const event = {
	    id: ++this.lastID.id,
	    level,
	    levelLabel: levels.labels[level],
	    time: Date.now(),
	    uuids: this.uuids,
	    message: ''
	  };
	  if (value instanceof Error) {
	    event.message = value.toString();
	    event.error = value;
	    event.meta = {
	      ...this.bindings
	    };
	  } else if (message !== undefined && typeof value === 'object') {
	    event.message = message;
	    event.meta = {
	      ...this.bindings,
	      ...value
	    };
	  } else if (message === undefined && typeof value === 'string') {
	    event.message = value;
	    event.meta = {
	      ...this.bindings
	    };
	  } else {
	    throw new Error('Invalid arguments');
	  }
	  this.events.push(event);
	  this.checkSize();
	  this.dispatchTypedEvent('log', new LogEvent({
	    log: event,
	    logs: this.events,
	    info: {
	      depth: this.uuids.length
	    }
	  }));
	  if (this.onChange) {
	    this.onChange(event, this.events, {
	      depth: this.uuids.length
	    });
	  }
	};

	exports.FifoLogger = FifoLogger;
	exports.LogEvent = LogEvent;

	Object.defineProperty(exports, '__esModule', { value: true });

}));
//# sourceMappingURL=fifo-logger.js.map
