/**
 * spectrum-generator - generate a spectrum from discrete peaks
 * @version v8.0.1
 * @link https://github.com/cheminfo/spectrum-generator#readme
 * @license MIT
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.spectrumGenerator = {}));
})(this, (function (exports) { 'use strict';

    const GAUSSIAN_EXP_FACTOR = -4 * Math.LN2;
    const ROOT_PI_OVER_LN2 = Math.sqrt(Math.PI / Math.LN2);
    const ROOT_THREE = Math.sqrt(3);
    const ROOT_2LN2 = Math.sqrt(2 * Math.LN2);
    const ROOT_2LN2_MINUS_ONE = Math.sqrt(2 * Math.LN2) - 1;

    // https://en.wikipedia.org/wiki/Error_function#Inverse_functions
    // This code yields to a good approximation
    // If needed a better implementation using polynomial can be found on https://en.wikipedia.org/wiki/Error_function#Inverse_functions
    function erfinv(x) {
      let a = 0.147;
      if (x === 0) return 0;
      let ln1MinusXSqrd = Math.log(1 - x * x);
      let lnEtcBy2Plus2 = ln1MinusXSqrd / 2 + 2 / (Math.PI * a);
      let firstSqrt = Math.sqrt(lnEtcBy2Plus2 ** 2 - ln1MinusXSqrd / a);
      let secondSqrt = Math.sqrt(firstSqrt - lnEtcBy2Plus2);
      return secondSqrt * (x > 0 ? 1 : -1);
    }

    class Gaussian {
      constructor() {
        let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
        const {
          fwhm = 500,
          sd
        } = options;
        this.fwhm = sd ? gaussianWidthToFWHM(2 * sd) : fwhm;
      }

      fwhmToWidth() {
        let fwhm = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : this.fwhm;
        return gaussianFwhmToWidth(fwhm);
      }

      widthToFWHM(width) {
        return gaussianWidthToFWHM(width);
      }

      fct(x) {
        return gaussianFct(x, this.fwhm);
      }

      getArea() {
        let height = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : calculateGaussianHeight({
          fwhm: this.fwhm
        });
        return getGaussianArea({
          fwhm: this.fwhm,
          height
        });
      }

      getFactor(area) {
        return getGaussianFactor(area);
      }

      getData() {
        let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
        return getGaussianData(this, options);
      }

      calculateHeight() {
        let area = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        return calculateGaussianHeight({
          fwhm: this.fwhm,
          area
        });
      }

      getParameters() {
        return ['fwhm'];
      }

    }
    function calculateGaussianHeight(options) {
      let {
        fwhm = 500,
        area = 1,
        sd
      } = options;
      if (sd) fwhm = gaussianWidthToFWHM(2 * sd);
      return 2 * area / ROOT_PI_OVER_LN2 / fwhm;
    }
    function gaussianFct(x, fwhm) {
      return Math.exp(GAUSSIAN_EXP_FACTOR * Math.pow(x / fwhm, 2));
    }
    function gaussianWidthToFWHM(width) {
      return width * ROOT_2LN2;
    }
    function gaussianFwhmToWidth(fwhm) {
      return fwhm / ROOT_2LN2;
    }
    function getGaussianArea(options) {
      let {
        fwhm = 500,
        sd,
        height = 1
      } = options;
      if (sd) fwhm = gaussianWidthToFWHM(2 * sd);
      return height * ROOT_PI_OVER_LN2 * fwhm / 2;
    }
    function getGaussianFactor() {
      let area = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 0.9999;
      return Math.sqrt(2) * erfinv(area);
    }
    function getGaussianData() {
      let shape = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      let {
        fwhm = 500,
        sd
      } = shape;
      if (sd) fwhm = gaussianWidthToFWHM(2 * sd);
      let {
        length,
        factor = getGaussianFactor(),
        height = calculateGaussianHeight({
          fwhm
        })
      } = options;

      if (!length) {
        length = Math.min(Math.ceil(fwhm * factor), Math.pow(2, 25) - 1);
        if (length % 2 === 0) length++;
      }

      const center = (length - 1) / 2;
      const data = new Float64Array(length);

      for (let i = 0; i <= center; i++) {
        data[i] = gaussianFct(i - center, fwhm) * height;
        data[length - 1 - i] = data[i];
      }

      return data;
    }

    class Lorentzian {
      constructor() {
        let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
        const {
          fwhm = 500
        } = options;
        this.fwhm = fwhm;
      }

      fwhmToWidth() {
        let fwhm = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : this.fwhm;
        return lorentzianFwhmToWidth(fwhm);
      }

      widthToFWHM(width) {
        return lorentzianWidthToFWHM(width);
      }

      fct(x) {
        return lorentzianFct(x, this.fwhm);
      }

      getArea() {
        let height = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        return getLorentzianArea({
          fwhm: this.fwhm,
          height
        });
      }

      getFactor(area) {
        return getLorentzianFactor(area);
      }

      getData() {
        let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
        return getLorentzianData(this, options);
      }

      calculateHeight() {
        let area = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        return calculateLorentzianHeight({
          fwhm: this.fwhm,
          area
        });
      }

      getParameters() {
        return ['fwhm'];
      }

    }
    const calculateLorentzianHeight = _ref => {
      let {
        fwhm = 1,
        area = 1
      } = _ref;
      return 2 * area / Math.PI / fwhm;
    };
    const lorentzianFct = (x, fwhm) => {
      return Math.pow(fwhm, 2) / (4 * Math.pow(x, 2) + Math.pow(fwhm, 2));
    };
    const lorentzianWidthToFWHM = width => {
      return width * ROOT_THREE;
    };
    const lorentzianFwhmToWidth = fwhm => {
      return fwhm / ROOT_THREE;
    };
    const getLorentzianArea = options => {
      const {
        fwhm = 500,
        height = 1
      } = options;
      return height * Math.PI * fwhm / 2;
    };
    const getLorentzianFactor = function () {
      let area = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 0.9999;
      return 2 * Math.tan(Math.PI * (area - 0.5));
    };
    const getLorentzianData = function () {
      let shape = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      let {
        fwhm = 500
      } = shape;
      let {
        length,
        factor = getLorentzianFactor(),
        height = calculateLorentzianHeight({
          fwhm,
          area: 1
        })
      } = options;

      if (!length) {
        length = Math.min(Math.ceil(fwhm * factor), Math.pow(2, 25) - 1);
        if (length % 2 === 0) length++;
      }

      const center = (length - 1) / 2;
      const data = new Float64Array(length);

      for (let i = 0; i <= center; i++) {
        data[i] = lorentzianFct(i - center, fwhm) * height;
        data[length - 1 - i] = data[i];
      }

      return data;
    };

    class PseudoVoigt {
      constructor() {
        let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
        const {
          fwhm = 500,
          mu = 0.5
        } = options;
        this.mu = mu;
        this.fwhm = fwhm;
      }

      fwhmToWidth() {
        let fwhm = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : this.fwhm;
        let mu = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : this.mu;
        return pseudoVoigtFwhmToWidth(fwhm, mu);
      }

      widthToFWHM(width) {
        let mu = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : this.mu;
        return pseudoVoigtWidthToFWHM(width, mu);
      }

      fct(x) {
        return pseudoVoigtFct(x, this.fwhm, this.mu);
      }

      getArea() {
        let height = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        return getPseudoVoigtArea({
          fwhm: this.fwhm,
          height,
          mu: this.mu
        });
      }

      getFactor(area) {
        return getPseudoVoigtFactor(area);
      }

      getData() {
        let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
        const {
          length,
          factor,
          height = calculatePseudoVoigtHeight({
            fwhm: this.fwhm,
            mu: this.mu,
            area: 1
          })
        } = options;
        return getPseudoVoigtData(this, {
          factor,
          length,
          height
        });
      }

      calculateHeight() {
        let area = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        return calculatePseudoVoigtHeight({
          fwhm: this.fwhm,
          mu: this.mu,
          area
        });
      }

      getParameters() {
        return ['fwhm', 'mu'];
      }

    }
    const calculatePseudoVoigtHeight = function () {
      let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      let {
        fwhm = 1,
        mu = 0.5,
        area = 1
      } = options;
      return 2 * area / (fwhm * (mu * ROOT_PI_OVER_LN2 + (1 - mu) * Math.PI));
    };
    const pseudoVoigtFct = (x, fwhm, mu) => {
      return (1 - mu) * lorentzianFct(x, fwhm) + mu * gaussianFct(x, fwhm);
    };
    const pseudoVoigtWidthToFWHM = function (width) {
      let mu = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0.5;
      return width * (mu * ROOT_2LN2_MINUS_ONE + 1);
    };
    const pseudoVoigtFwhmToWidth = function (fwhm) {
      let mu = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0.5;
      return fwhm / (mu * ROOT_2LN2_MINUS_ONE + 1);
    };
    const getPseudoVoigtArea = options => {
      const {
        fwhm = 500,
        height = 1,
        mu = 0.5
      } = options;
      return fwhm * height * (mu * ROOT_PI_OVER_LN2 + (1 - mu) * Math.PI) / 2;
    };
    const getPseudoVoigtFactor = function () {
      let area = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 0.9999;
      let mu = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0.5;
      return mu < 1 ? getLorentzianFactor(area) : getGaussianFactor(area);
    };
    const getPseudoVoigtData = function () {
      let shape = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      let {
        fwhm = 500,
        mu = 0.5
      } = shape;
      let {
        length,
        factor = getPseudoVoigtFactor(0.999, mu),
        height = calculatePseudoVoigtHeight({
          fwhm,
          mu,
          area: 1
        })
      } = options;

      if (!height) {
        height = 1 / (mu / Math.sqrt(-GAUSSIAN_EXP_FACTOR / Math.PI) * fwhm + (1 - mu) * fwhm * Math.PI / 2);
      }

      if (!length) {
        length = Math.min(Math.ceil(fwhm * factor), Math.pow(2, 25) - 1);
        if (length % 2 === 0) length++;
      }

      const center = (length - 1) / 2;
      const data = new Float64Array(length);

      for (let i = 0; i <= center; i++) {
        data[i] = pseudoVoigtFct(i - center, fwhm, mu) * height;
        data[length - 1 - i] = data[i];
      }

      return data;
    };

    class Gaussian2D {
      constructor() {
        let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
        let {
          fwhm = 50,
          sd
        } = options;
        fwhm = ensureFWHM2D(fwhm, sd);
        this.fwhmX = fwhm.x;
        this.fwhmY = fwhm.y;
      }

      fct(x, y) {
        return gaussian2DFct(x, y, this.fwhmX, this.fwhmY);
      }

      getData() {
        let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
        return getGaussian2DData({
          fwhm: {
            x: this.fwhmX,
            y: this.fwhmY
          }
        }, options);
      }

      getFactor() {
        let volume = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        return getGaussianFactor(volume);
      }

      getVolume() {
        let height = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : calculateGaussian2DHeight({
          fwhm: {
            x: this.fwhmX,
            y: this.fwhmY
          },
          volume: 1
        });
        return getGaussian2DVolume({
          fwhm: {
            x: this.fwhmX,
            y: this.fwhmY
          },
          height
        });
      }

      widthToFWHM(width) {
        return gaussianWidthToFWHM(width);
      }

      fwhmToWidth(fwhm) {
        return gaussianFwhmToWidth(fwhm);
      }

      calculateHeight() {
        let volume = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        return calculateGaussian2DHeight({
          volume,
          fwhm: {
            x: this.fwhmX,
            y: this.fwhmY
          }
        });
      }

      set fwhm(fwhm) {
        fwhm = ensureXYNumber$1(fwhm);
        this.fwhmX = fwhm.x;
        this.fwhmY = fwhm.y;
      }

    }
    const gaussian2DFct = (x, y, xFWHM, yFWHM) => {
      return Math.exp(GAUSSIAN_EXP_FACTOR * (Math.pow(x / xFWHM, 2) + Math.pow(y / yFWHM, 2)));
    };
    const getGaussian2DData = function (shape) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      let {
        fwhm = 50,
        sd
      } = shape;
      fwhm = ensureFWHM2D(fwhm, sd);
      let {
        factor = getGaussianFactor(),
        length = {
          x: 0,
          y: 0
        },
        height = calculateGaussian2DHeight({
          fwhm,
          volume: 1
        })
      } = options;
      factor = ensureXYNumber$1(factor);
      length = ensureXYNumber$1(length);

      for (const axis of ['x', 'y']) {
        if (!length[axis]) {
          length[axis] = Math.min(Math.ceil(fwhm[axis] * factor[axis]), Math.pow(2, 25) - 1);
          if (length[axis] % 2 === 0) length[axis]++;
        }
      }

      const xCenter = (length.x - 1) / 2;
      const yCenter = (length.y - 1) / 2;
      const data = new Array(length.x);

      for (let i = 0; i < length.x; i++) {
        data[i] = new Float64Array(length.y);
      }

      for (let i = 0; i < length.x; i++) {
        for (let j = 0; j < length.y; j++) {
          data[i][j] = gaussian2DFct(i - xCenter, j - yCenter, fwhm.x, fwhm.y) * height;
        }
      }

      return data;
    };
    const calculateGaussian2DHeight = function () {
      let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      let {
        volume = 1,
        fwhm = 50,
        sd
      } = options;
      fwhm = ensureFWHM2D(fwhm, sd);
      return volume * Math.LN2 * 4 / (Math.PI * fwhm.y * fwhm.x);
    };
    const getGaussian2DVolume = function () {
      let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      let {
        fwhm = 50,
        height = 1,
        sd
      } = options;
      fwhm = ensureFWHM2D(fwhm, sd);
      return height * Math.PI * fwhm.y * fwhm.x / Math.LN2 / 4;
    };

    function ensureXYNumber$1(input) {
      return typeof input !== 'object' ? {
        x: input,
        y: input
      } : { ...input
      };
    }

    function ensureFWHM2D(fwhm, sd) {
      if (sd !== undefined) {
        let sdObject = ensureXYNumber$1(sd);
        return {
          x: gaussianWidthToFWHM(2 * sdObject.x),
          y: gaussianWidthToFWHM(2 * sdObject.y)
        };
      } else if (fwhm !== undefined) {
        return ensureXYNumber$1(fwhm);
      } else {
        throw new Error('ensureFWHM2D must have either fwhm or sd defined');
      }
    }

    /**
     * Generate a instance of a specific kind of shape.
     */

    function getShape1D(shape) {
      const {
        kind
      } = shape;

      switch (kind) {
        case 'gaussian':
          return new Gaussian(shape);

        case 'lorentzian':
          return new Lorentzian(shape);

        case 'pseudoVoigt':
          return new PseudoVoigt(shape);

        default:
          {
            throw Error(`Unknown distribution ${kind}`);
          }
      }
    }

    /**
     * Generate a instance of a specific kind of shape.
     */

    function getShape2D(shape) {
      const {
        kind
      } = shape;

      switch (kind) {
        case 'gaussian':
          return new Gaussian2D(shape);

        default:
          {
            const unHandled = kind; // eslint-disable-next-line @typescript-eslint/restrict-template-expressions

            throw Error(`Unknown distribution ${unHandled}`);
          }
      }
    }

    function addBaseline(data, baselineFct) {
      if (!baselineFct) return data;
      let xs = data.x;
      let ys = data.y;

      for (let i = 0; i < xs.length; i++) {
        ys[i] += baselineFct(xs[i]);
      }

      return data;
    }

    const toString = Object.prototype.toString;
    /**
     * Checks if an object is an instance of an Array (array or typed array).
     *
     * @param {any} value - Object to check.
     * @returns {boolean} True if the object is an array.
     */

    function isAnyArray(value) {
      return toString.call(value).endsWith('Array]');
    }

    /**
     * This function xAdd the first array by the second array or a constant value to each element of the first array
     *
     * @param array1 - the first array
     * @param array2 - the second array or number
     */

    function xAdd(array1, array2) {
      let isConstant = false;
      let constant = 0;

      if (isAnyArray(array2)) {
        if (array1.length !== array2.length) {
          throw new Error('xAdd: size of array1 and array2 must be identical');
        }
      } else {
        isConstant = true;
        constant = array2;
      }

      let array3 = new Float64Array(array1.length);

      if (isConstant) {
        for (let i = 0; i < array1.length; i++) {
          array3[i] = array1[i] + constant;
        }
      } else {
        for (let i = 0; i < array1.length; i++) {
          array3[i] = array1[i] + array2[i];
        }
      }

      return array3;
    }

    /**
     * Checks if input is of type array
     *
     * @param input - input
     */

    function xCheck(input) {
      if (!isAnyArray(input)) {
        throw new TypeError('input must be an array');
      }

      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
    }

    /**
     * Computes the maximal value of an array of values
     *
     * @param array - array of numbers
     * @param options - options
     */

    function xMaxValue(array) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      xCheck(array);
      const {
        fromIndex = 0,
        toIndex = array.length - 1
      } = options;
      let maxValue = array[fromIndex];

      for (let i = fromIndex + 1; i <= toIndex; i++) {
        if (array[i] > maxValue) {
          maxValue = array[i];
        }
      }

      return maxValue;
    }

    function matrixCheck(data) {
      if (data.length === 0 || data[0].length === 0) {
        throw RangeError('matrix should contain data');
      }

      const firstLength = data[0].length;

      for (let i = 1; i < data.length; i++) {
        if (data[i].length !== firstLength) {
          throw new RangeError('All rows should has the same length');
        }
      }
    }

    /**
     * Get min and max Z
     *
     * @param matrix - matrix [rows][cols].
     */

    function matrixMinMaxZ(matrix) {
      matrixCheck(matrix);
      const nbRows = matrix.length;
      const nbColumns = matrix[0].length;
      let min = matrix[0][0];
      let max = matrix[0][0];

      for (let column = 0; column < nbColumns; column++) {
        for (let row = 0; row < nbRows; row++) {
          if (matrix[row][column] < min) min = matrix[row][column];
          if (matrix[row][column] > max) max = matrix[row][column];
        }
      }

      return {
        min,
        max
      };
    }

    const LOOP = 8;
    const FLOAT_MUL = 1 / 16777216;
    const sh1 = 15;
    const sh2 = 18;
    const sh3 = 11;

    function multiply_uint32(n, m) {
      n >>>= 0;
      m >>>= 0;
      const nlo = n & 0xffff;
      const nhi = n - nlo;
      return (nhi * m >>> 0) + nlo * m >>> 0;
    }

    class XSadd {
      constructor() {
        let seed = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : Date.now();
        this.state = new Uint32Array(4);
        this.init(seed);
        this.random = this.getFloat.bind(this);
      }
      /**
       * Returns a 32-bit integer r (0 <= r < 2^32)
       */


      getUint32() {
        this.nextState();
        return this.state[3] + this.state[2] >>> 0;
      }
      /**
       * Returns a floating point number r (0.0 <= r < 1.0)
       */


      getFloat() {
        return (this.getUint32() >>> 8) * FLOAT_MUL;
      }

      init(seed) {
        if (!Number.isInteger(seed)) {
          throw new TypeError('seed must be an integer');
        }

        this.state[0] = seed;
        this.state[1] = 0;
        this.state[2] = 0;
        this.state[3] = 0;

        for (let i = 1; i < LOOP; i++) {
          this.state[i & 3] ^= i + multiply_uint32(1812433253, this.state[i - 1 & 3] ^ this.state[i - 1 & 3] >>> 30 >>> 0) >>> 0;
        }

        this.periodCertification();

        for (let i = 0; i < LOOP; i++) {
          this.nextState();
        }
      }

      periodCertification() {
        if (this.state[0] === 0 && this.state[1] === 0 && this.state[2] === 0 && this.state[3] === 0) {
          this.state[0] = 88; // X

          this.state[1] = 83; // S

          this.state[2] = 65; // A

          this.state[3] = 68; // D
        }
      }

      nextState() {
        let t = this.state[0];
        t ^= t << sh1;
        t ^= t >>> sh2;
        t ^= this.state[3] << sh3;
        this.state[0] = this.state[1];
        this.state[1] = this.state[2];
        this.state[2] = this.state[3];
        this.state[3] = t;
      }

    }

    /**
     * Create a random array of numbers of a specific length
     *
     * @return - array of random floats normally distributed
     */

    let spare;
    let hasSpare = false;
    function createRandomArray() {
      let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      let {
        mean = 0,
        standardDeviation = 1,
        length = 1000,
        range = 1,
        seed,
        distribution = 'normal'
      } = options;
      const generator = new XSadd(seed);
      let returnArray = new Float64Array(length);

      switch (distribution) {
        case 'normal':
          for (let i = 0; i < length; i++) {
            returnArray[i] = generateGaussian(mean, standardDeviation, generator);
          }

          break;

        case 'uniform':
          for (let i = 0; i < length; i++) {
            returnArray[i] = (generator.random() - 0.5) * range + mean;
          }

          break;

        default:
          // eslint-disable-next-line @typescript-eslint/restrict-template-expressions
          throw new Error(`unknown distribution: ${distribution}`);
      }

      return returnArray;
    }

    function generateGaussian(mean, standardDeviation, generator) {
      let val, u, v, s;

      if (hasSpare) {
        hasSpare = false;
        val = spare * standardDeviation + mean;
      } else {
        do {
          u = generator.random() * 2 - 1;
          v = generator.random() * 2 - 1;
          s = u * u + v * v;
        } while (s >= 1 || s === 0);

        s = Math.sqrt(-2.0 * Math.log(s) / s);
        spare = v * s;
        hasSpare = true;
        val = mean + standardDeviation * u * s;
      }

      return val;
    }

    function addNoise(data) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        seed = 0,
        distribution = 'normal',
        percent = 1
      } = options;
      const range = xMaxValue(data.y) * percent / 100;
      const noise = createRandomArray({
        distribution,
        seed,
        mean: 0,
        standardDeviation: range,
        range,
        length: data.x.length
      });
      data.y = xAdd(data.y, noise);
      return data;
    }

    class SpectrumGenerator {
      constructor() {
        let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
        const {
          from = 0,
          to = 1000,
          nbPoints = 10001,
          peakWidthFct,
          shape = {
            kind: 'gaussian',
            fwhm: 5
          }
        } = options;
        this.from = from;
        this.to = to;
        this.nbPoints = nbPoints;
        this.interval = (this.to - this.from) / (this.nbPoints - 1);
        this.peakWidthFct = peakWidthFct;
        this.maxPeakHeight = Number.MIN_SAFE_INTEGER;
        this.data = {
          x: new Float64Array(this.nbPoints),
          y: new Float64Array(this.nbPoints)
        };
        let shapeGenerator = getShape1D(shape);
        this.shape = shapeGenerator;
        assertNumber$1(this.from, 'from');
        assertNumber$1(this.to, 'to');
        assertInteger$1(this.nbPoints, 'nbPoints');

        if (this.to <= this.from) {
          throw new RangeError('to option must be larger than from');
        }

        if (this.peakWidthFct && typeof this.peakWidthFct !== 'function') {
          throw new TypeError('peakWidthFct option must be a function');
        }

        this.reset();
      }
      /**
       * Add a series of peaks to the spectrum.
       * @param peaks - Peaks to add.
       */


      addPeaks(peaks, options) {
        if (!Array.isArray(peaks) && (typeof peaks !== 'object' || peaks.x === undefined || peaks.y === undefined || !Array.isArray(peaks.x) || !Array.isArray(peaks.y) || peaks.x.length !== peaks.y.length)) {
          throw new TypeError('peaks must be an array or an object containing x[] and y[]');
        }

        if (Array.isArray(peaks)) {
          for (const peak of peaks) {
            this.addPeak(peak, options);
          }
        } else {
          for (let i = 0; i < peaks.x.length; i++) {
            this.addPeak([peaks.x[i], peaks.y[i]], options);
          }
        }
      }
      /**
       * Add a single peak to the spectrum.
       * A peak may be either defined as [x,y,fwhm,...] or as {x, y, shape}
       * @param peak
       * @param options
       */


      addPeak(peak) {
        let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

        if (Array.isArray(peak) && peak.length < 2) {
          throw new Error('peak must be an array with two (or three) values or an object with {x,y,width?}');
        }

        if (!Array.isArray(peak) && (peak.x === undefined || peak.y === undefined)) {
          throw new Error('peak must be an array with two (or three) values or an object with {x,y,width?}');
        }

        let xPosition;
        let intensity;
        let peakFWHM;
        let peakWidth;
        let peakShapeOptions;

        if (Array.isArray(peak)) {
          [xPosition, intensity, peakFWHM, peakShapeOptions] = peak;
        } else {
          xPosition = peak.x;
          intensity = peak.y;
          peakWidth = peak.width;
          peakShapeOptions = peak.shape;
        }

        if (intensity > this.maxPeakHeight) this.maxPeakHeight = intensity;
        let {
          shape: shapeOptions
        } = options;

        if (peakShapeOptions) {
          shapeOptions = shapeOptions ? { ...shapeOptions,
            ...peakShapeOptions
          } : peakShapeOptions;
        }

        if (shapeOptions) {
          this.shape = getShape1D(shapeOptions);
        }

        let {
          widthLeft,
          widthRight
        } = options;
        /*
         if we don't force the fwhm we just take the one from the shape
         however we have many way to force it:
         - use [x,y,fwhm]
         - define `width` that will be converted to fwhm
         - define `widthLeft` and `widthRight` to define asymmetric peaks
         - have a callback `peakWidthFct`
         This should evolve in the future because we will not always have `fwhm`
         */

        const fwhm = peakFWHM !== undefined ? peakFWHM : peakWidth ? this.shape.widthToFWHM(peakWidth) : this.peakWidthFct ? this.peakWidthFct(xPosition) : this.shape.fwhm;
        if (!widthLeft) widthLeft = fwhm;
        if (!widthRight) widthRight = fwhm;

        if (!widthLeft || !widthRight) {
          throw new Error('Width left or right is undefined or zero');
        }

        let factor = options.factor === undefined ? this.shape.getFactor() : options.factor;
        const firstValue = xPosition - widthLeft / 2 * factor;
        const lastValue = xPosition + widthRight / 2 * factor;
        const firstPoint = Math.max(0, Math.floor((firstValue - this.from) / this.interval));
        const lastPoint = Math.min(this.nbPoints - 1, Math.ceil((lastValue - this.from) / this.interval));
        const middlePoint = Math.round((xPosition - this.from) / this.interval); // PEAK SHAPE MAY BE ASYMMETRC (widthLeft and widthRight) !
        // we calculate the left part of the shape

        this.shape.fwhm = widthLeft;

        for (let index = firstPoint; index < Math.max(middlePoint, 0); index++) {
          this.data.y[index] += intensity * this.shape.fct(this.data.x[index] - xPosition);
        } // we calculate the right part of the gaussian


        this.shape.fwhm = widthRight;

        for (let index = Math.min(middlePoint, lastPoint); index <= lastPoint; index++) {
          this.data.y[index] += intensity * this.shape.fct(this.data.x[index] - xPosition);
        }
      }
      /**
       * Add a baseline to the spectrum.
       * @param baselineFct - Mathematical function producing the baseline you want.
       */


      addBaseline(baselineFct) {
        addBaseline(this.data, baselineFct);
        return this;
      }
      /**
       * Add noise to the spectrum.
       *
       * @param percent - Noise's amplitude in percents of the spectrum max value. Default: 1.
       */


      addNoise(options) {
        addNoise(this.data, options);
        return this;
      }
      /**
       * Get the generated spectrum.
       */


      getSpectrum() {
        let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

        if (typeof options === 'boolean') {
          options = {
            copy: options
          };
        }

        const {
          copy = true,
          threshold = 0
        } = options;

        if (threshold) {
          let minPeakHeight = this.maxPeakHeight * threshold;
          let x = [];
          let y = [];

          for (let i = 0; i < this.data.x.length; i++) {
            if (this.data.y[i] >= minPeakHeight) {
              x.push(this.data.x[i]);
              y.push(this.data.y[i]);
            }
          }

          return {
            x,
            y
          };
        }

        if (copy) {
          return {
            x: this.data.x.slice(),
            y: this.data.y.slice()
          };
        } else {
          return this.data;
        }
      }
      /**
       * Resets the generator with an empty spectrum.
       */


      reset() {
        const spectrum = this.data;

        for (let i = 0; i < this.nbPoints; i++) {
          spectrum.x[i] = this.from + i * this.interval;
        }

        return this;
      }

    }

    function assertInteger$1(value, name) {
      if (!Number.isInteger(value)) {
        throw new TypeError(`${name} option must be an integer`);
      }
    }

    function assertNumber$1(value, name) {
      if (!Number.isFinite(value)) {
        throw new TypeError(`${name} option must be a number`);
      }
    }
    /**
     * Generates a spectrum and returns it.
     * @param peaks - List of peaks to put in the spectrum.
     * @param options
     */


    function generateSpectrum(peaks) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        generator: generatorOptions,
        noise,
        baseline,
        threshold,
        peakOptions
      } = options;
      const generator = new SpectrumGenerator(generatorOptions);
      generator.addPeaks(peaks, peakOptions);
      if (baseline) generator.addBaseline(baseline);

      if (noise) {
        generator.addNoise(noise);
      }

      return generator.getSpectrum({
        threshold
      });
    }

    const axis2D = ['x', 'y'];
    const peakCoordinates = ['x', 'y', 'z'];

    const convertWidthToFWHM = (shape, width) => {
      const widthData = ensureXYNumber(width);

      for (let key of axis2D) {
        widthData[key] = shape.widthToFWHM(widthData[key]);
      }

      return widthData;
    };

    class Spectrum2DGenerator {
      constructor() {
        let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
        let {
          from = 0,
          to = 100,
          nbPoints = 1001,
          peakWidthFct = () => 5,
          shape = {
            kind: 'gaussian'
          }
        } = options;
        from = ensureXYNumber(from);
        to = ensureXYNumber(to);
        nbPoints = ensureXYNumber(nbPoints);

        for (const axis of axis2D) {
          assertNumber(from[axis], `from-${axis}`);
          assertNumber(to[axis], `to-${axis}`);
          assertInteger(nbPoints[axis], `nbPoints-${axis}`);
        }

        this.from = from;
        this.to = to;
        this.nbPoints = nbPoints;
        this.interval = calculeIntervals(from, to, nbPoints);
        this.peakWidthFct = peakWidthFct;
        this.maxPeakHeight = Number.MIN_SAFE_INTEGER;
        let shapeGenerator = getShape2D(shape);
        this.shape = shapeGenerator;
        this.data = {
          x: new Float64Array(nbPoints.x),
          y: new Float64Array(nbPoints.y),
          z: createMatrix(this.nbPoints)
        };

        for (const axis of axis2D) {
          if (this.to[axis] <= this.from[axis]) {
            throw new RangeError('to option must be larger than from');
          }
        }

        if (typeof this.peakWidthFct !== 'function') {
          throw new TypeError('peakWidthFct option must be a function');
        }

        this.reset();
      }

      addPeaks(peaks, options) {
        if (!Array.isArray(peaks) && (typeof peaks !== 'object' || peaks.x === undefined || peaks.y === undefined || !Array.isArray(peaks.x) || !Array.isArray(peaks.y) || peaks.x.length !== peaks.y.length)) {
          throw new TypeError('peaks must be an array or an object containing x[] and y[]');
        }

        if (Array.isArray(peaks)) {
          for (const peak of peaks) {
            this.addPeak(peak, options);
          }
        } else {
          let nbPeaks = peaks.x.length;

          for (const c of peakCoordinates) {
            if (peaks[c] && Array.isArray(peaks[c])) {
              if (nbPeaks !== peaks[c].length) {
                throw new Error('x, y, z should have the same length');
              }
            }
          }

          for (let i = 0; i < peaks.x.length; i++) {
            this.addPeak([peaks.x[i], peaks.y[i], peaks.z[i]], options);
          }
        }

        return this;
      }

      addPeak(peak) {
        let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

        if (Array.isArray(peak) && peak.length < 3) {
          throw new Error('peak must be an array with three (or four) values or an object with {x,y,z,width?}');
        }

        if (!Array.isArray(peak) && peakCoordinates.some(e => peak[e] === undefined)) {
          throw new Error('peak must be an array with three (or four) values or an object with {x,y,z,width?}');
        }

        let xPosition;
        let yPosition;
        let intensity;
        let peakFWHM;
        let peakWidth;
        let peakShapeOptions;

        if (Array.isArray(peak)) {
          [xPosition, yPosition, intensity, peakFWHM, peakShapeOptions] = peak;
        } else {
          xPosition = peak.x;
          yPosition = peak.y;
          intensity = peak.z;
          peakFWHM = peak.fwhm;
          peakWidth = peak.width;
          peakShapeOptions = peak.shape;
        }

        const position = {
          x: xPosition,
          y: yPosition
        };
        if (intensity > this.maxPeakHeight) this.maxPeakHeight = intensity;
        let {
          shape: shapeOptions,
          width
        } = options;

        if (peakShapeOptions) {
          shapeOptions = shapeOptions ? { ...shapeOptions,
            ...peakShapeOptions
          } : peakShapeOptions;
        }

        if (shapeOptions) {
          this.shape = getShape2D(shapeOptions);
        }

        let {
          fwhm = peakFWHM !== undefined ? peakFWHM : peakWidth ? convertWidthToFWHM(this.shape, peakWidth) : width ? convertWidthToFWHM(this.shape, width) : this.peakWidthFct(xPosition, yPosition)
        } = options;
        fwhm = ensureXYNumber(fwhm);
        let factor = options.factor === undefined ? this.shape.getFactor() : options.factor;
        factor = ensureXYNumber(factor);
        const firstPoint = {
          x: 0,
          y: 0
        };
        const lastPoint = {
          x: 0,
          y: 0
        };

        for (const axis of axis2D) {
          const first = position[axis] - fwhm[axis] / 2 * factor[axis];
          const last = position[axis] + fwhm[axis] / 2 * factor[axis];
          firstPoint[axis] = Math.max(0, Math.floor((first - this.from[axis]) / this.interval[axis]));
          lastPoint[axis] = Math.min(this.nbPoints[axis], Math.ceil((last - this.from[axis]) / this.interval[axis]));
        }

        this.shape.fwhm = fwhm;

        for (let xIndex = firstPoint.x; xIndex < lastPoint.x; xIndex++) {
          for (let yIndex = firstPoint.y; yIndex < lastPoint.y; yIndex++) {
            const value = intensity * this.shape.fct(this.data.x[xIndex] - position.x, this.data.y[yIndex] - position.y);

            if (value > 1e-6) {
              this.data.z[yIndex][xIndex] += value;
            }
          }
        }

        return this;
      }

      getSpectrum() {
        let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

        if (typeof options === 'boolean') {
          options = {
            copy: options
          };
        }

        const {
          copy = true
        } = options;
        let minMaxZ = matrixMinMaxZ(this.data.z);
        return {
          minX: this.from.x,
          maxX: this.to.x,
          maxY: this.to.y,
          minY: this.from.y,
          minZ: minMaxZ.min,
          maxZ: minMaxZ.max,
          z: copy ? this.data.z.slice() : this.data.z
        };
      }

      reset() {
        const spectrum = this.data;

        for (const axis of axis2D) {
          for (let i = 0; i < this.nbPoints[axis]; i++) {
            spectrum[axis][i] = this.from[axis] + i * this.interval[axis];
          }
        }

        for (let row of spectrum.z) {
          for (let j = 0; j < row.length; j++) {
            row[j] = 0;
          }
        }

        return this;
      }

    }
    function generateSpectrum2D(peaks) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        generator: generatorOptions,
        peaks: addPeaksOptions
      } = options;
      const generator = new Spectrum2DGenerator(generatorOptions);
      generator.addPeaks(peaks, addPeaksOptions);
      return generator.getSpectrum();
    }

    function ensureXYNumber(input) {
      return typeof input !== 'object' ? {
        x: input,
        y: input
      } : { ...input
      };
    }

    function calculeIntervals(from, to, nbPoints) {
      return {
        x: (to.x - from.x) / (nbPoints.x - 1),
        y: (to.y - from.y) / (nbPoints.y - 1)
      };
    }

    function assertInteger(value, name) {
      if (!Number.isInteger(value)) {
        throw new TypeError(`${name} option must be an integer`);
      }
    }

    function assertNumber(value, name) {
      if (!Number.isFinite(value)) {
        throw new TypeError(`${name} option must be a number`);
      }
    }

    function createMatrix(nbPoints) {
      const zMatrix = new Array(nbPoints.y);

      for (let i = 0; i < nbPoints.y; i++) {
        zMatrix[i] = new Float64Array(nbPoints.x);
      }

      return zMatrix;
    }

    exports.Spectrum2DGenerator = Spectrum2DGenerator;
    exports.SpectrumGenerator = SpectrumGenerator;
    exports.generateSpectrum = generateSpectrum;
    exports.generateSpectrum2D = generateSpectrum2D;

    Object.defineProperty(exports, '__esModule', { value: true });

}));
//# sourceMappingURL=spectrum-generator.js.map
