/**
 * nmr-parser - Read and convert any NMR file
 * @version v1.7.2
 * @link https://github.com/cheminfo/nmr-parser#readme
 * @license MIT
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.NMRparser = {}));
})(this, (function (exports) { 'use strict';

    const toString$1 = Object.prototype.toString;
    /**
     * Checks if an object is an instance of an Array (array or typed array).
     *
     * @param {any} value - Object to check.
     * @returns {boolean} True if the object is an array.
     */

    function isAnyArray$1(value) {
      return toString$1.call(value).endsWith('Array]');
    }

    /**
     * a number that correspond to a type of numeric
     * @typedef {number} numericType
     * @const
     */
    const numericTypeTable = {
      0: 'uint8',
      1: 'uint16',
      2: 'uint32',
      3: 'uint64',
      4: 'int8',
      5: 'int16',
      6: 'int32',
      7: 'int64',
      8: 'float32',
      9: 'float64',
      10: 'complex64',
      11: 'complex128'
    };
    /**
     * a number that corresponds to a type of quantity
     * @typedef {number} quantityType
     * @const
     */

    const quantityTypeTable = {
      0: 'scalar',
      1: 'vector',
      2: 'matrix',
      3: 'symetricMatrix',
      4: 'pixel'
    };

    /**
     * a class for dependent variable
     * @param {object || array} data - the dependent variable
     * @param {numericType} numericType - a number that correspond to a type of numeric used to store the components
     * @param {object} [options] - an object with options (name, unit, quantityName, componentLabels, sparseSampling, application, description)
     * @param {string} [options.name] - a name of the dependent variable
     * @param {string} [options.unit] - the unit of the dependent variable
     * @param {string} [options.quantityName] - a name of the quantity
     * @param {array} [options.componentLabels] - an array of labels for each component of the dependent variable
     * @return {object} - an dependent variable
     */

    function formatDependentVariable(data, numericType) {
      let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
      let {
        quantityType = 0,
        encoding = 'none',
        name = '',
        unit = '',
        quantityName = '',
        componentLabels = [],
        sparseSampling = {},
        from = 0,
        to = -1
      } = options;
      let components;

      if (isAnyArray$1(data)) {
        throw new Error('not yet implemented');
      } else if (Object.keys(data).length === 2) {
        components = fromReIm(data, from, to);
      }

      if (componentLabels.length === 0) {
        componentLabels = components.componentLabels;
      }

      return {
        type: 'internal',
        quantityType: quantityTypeTable[quantityType],
        numericType: numericTypeTable[numericType],
        encoding,
        name,
        unit,
        quantityName,
        componentLabels,
        sparseSampling,
        description: options.description || '',
        application: options.application || '',
        components: components.components,
        dataLength: components.dataLength
      };
    }
    /**
     * import object {re:[], im:[]} to component
     * @param {object} reIm - a reIm object to import
     * @param {number} from - lower limit
     * @param {number} to - upper limit
     * @return {array} - components
     */

    function fromReIm(reIm, from, to) {
      let dataLength = [];
      let componentLabels = [];
      let components = [];

      if (isAnyArray$1(reIm.re) & isAnyArray$1(reIm.im)) {
        if (typeof reIm.re[0] === 'number') {
          // if 1D
          dataLength[0] = setLengthComplex(from[0], to[0], reIm.re.length);
          let component = new Float64Array(dataLength[0]);

          for (let i = 0; i < dataLength[0]; i += 2) {
            let idx = i + from[0] * 2;
            component[i] = reIm.re[idx / 2];
            component[i + 1] = reIm.im[idx / 2];
          }

          components.push(component);
          componentLabels.push('complex');
        } else if (isAnyArray$1(reIm.re[0])) {
          // if 2D
          dataLength[0] = setLength(from[1], to[1], reIm.re.length);
          dataLength[1] = setLengthComplex(from[0], to[0], reIm.re[0].length);

          for (let j = 0; j < dataLength[0]; j++) {
            let component = new Float64Array(dataLength[1]);

            for (let i = 0; i < dataLength[1]; i += 2) {
              let idx = i + from[0] * 2;
              component[i] = reIm.re[j][idx / 2];
              component[i + 1] = reIm.im[j][idx / 2];
            }

            components.push(component);
          }
        } else {
          throw new Error('check your object');
        }
      } else if (isAnyArray$1(reIm.re.re)) {
        dataLength[0] = reIm.re.re.length * 2;
        let re = fromReIm(reIm.re, from, to).components;
        let im = fromReIm(reIm.im, from, to).components;

        for (let j = 0; j < dataLength[0] / 2; j++) {
          components.push(re[j]);
          components.push(im[j]);
        }
      } else {
        throw new Error('check the dimension or the type of data in your array');
      }

      return {
        dataLength,
        componentLabels,
        components
      };
    }

    function setLength(from, to, length) {
      if (to - from + 1 < length) {
        return to - from + 1;
      } else {
        return length;
      }
    }

    function setLengthComplex(from, to, length) {
      if (to - from + 1 < length) {
        return (to - from + 1) * 2;
      } else {
        return length * 2;
      }
    } // /**
    //  * add component to components from 1D array.
    //  * @param {array} array - a 1D or 2D array to import
    //  * @return {Float64Array} - component
    //  */
    // function add1DArray(array) {
    //   let component;
    //   component = new Float64Array(array.length);
    //   for (let i = 0; i < array.length; i++) {
    //     component[i] = array[i];
    //   }
    //   return component;
    // }
    // /**
    //  * import component to InternalDEPENDENTVAR class object from 1D or 2D array.
    //  * @param {array} array - a 1D or 2D array to import
    //  */
    // function fromArray(array) {
    //   this.dataLength[0] = array.length;
    //   if (typeof array[0] === 'number') {
    //     this.components = [this.add1DArray(array)];
    //   } else if (Array.isArray(array[0])) {
    //     this.dataLength[1] = array[0].length;
    //     for (let j = 0; j < this.dataLength[1]; j++) {
    //       this.components.push(this.add1DArray(array[j]));
    //     }
    //   } else {
    //     throw new Error('check the dimension or the type of data in your array');
    //   }
    //   return this;
    // }

    /**
     *
     * @param {*} label
     * @param {*} count
     * @param {*} increment
     * @param {*} options
     */
    function formatLinearDimension(label, count, increment) {
      let options = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : {};
      return {
        label: String(label),
        count: Number(count),
        increment: increment,
        type: 'linear',
        description: String(options.description) || '',
        application: options.application || {},
        coordinatesOffset: options.coordinatesOffset || 0,
        originOffset: options.originOffset || 0,
        quantityName: String(options.quantityName) || '',
        reciprocal: options.reciprocal || {},
        period: options.period || 0,
        complexFFT: options.complexFFT || false
      };
    }

    // eslint-disable-next-line import/no-unassigned-import
    const decoder = new TextDecoder('utf-8');
    function decode(bytes) {
      return decoder.decode(bytes);
    }
    const encoder = new TextEncoder();
    function encode(str) {
      return encoder.encode(str);
    }

    const defaultByteLength = 1024 * 8;
    class IOBuffer {
      /**
       * @param data - The data to construct the IOBuffer with.
       * If data is a number, it will be the new buffer's length<br>
       * If data is `undefined`, the buffer will be initialized with a default length of 8Kb<br>
       * If data is an ArrayBuffer, SharedArrayBuffer, an ArrayBufferView (Typed Array), an IOBuffer instance,
       * or a Node.js Buffer, a view will be created over the underlying ArrayBuffer.
       * @param options
       */
      constructor() {
        let data = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : defaultByteLength;
        let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
        let dataIsGiven = false;

        if (typeof data === 'number') {
          data = new ArrayBuffer(data);
        } else {
          dataIsGiven = true;
          this.lastWrittenByte = data.byteLength;
        }

        const offset = options.offset ? options.offset >>> 0 : 0;
        const byteLength = data.byteLength - offset;
        let dvOffset = offset;

        if (ArrayBuffer.isView(data) || data instanceof IOBuffer) {
          if (data.byteLength !== data.buffer.byteLength) {
            dvOffset = data.byteOffset + offset;
          }

          data = data.buffer;
        }

        if (dataIsGiven) {
          this.lastWrittenByte = byteLength;
        } else {
          this.lastWrittenByte = 0;
        }

        this.buffer = data;
        this.length = byteLength;
        this.byteLength = byteLength;
        this.byteOffset = dvOffset;
        this.offset = 0;
        this.littleEndian = true;
        this._data = new DataView(this.buffer, dvOffset, byteLength);
        this._mark = 0;
        this._marks = [];
      }
      /**
       * Checks if the memory allocated to the buffer is sufficient to store more
       * bytes after the offset.
       * @param byteLength - The needed memory in bytes.
       * @returns `true` if there is sufficient space and `false` otherwise.
       */


      available() {
        let byteLength = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        return this.offset + byteLength <= this.length;
      }
      /**
       * Check if little-endian mode is used for reading and writing multi-byte
       * values.
       * @returns `true` if little-endian mode is used, `false` otherwise.
       */


      isLittleEndian() {
        return this.littleEndian;
      }
      /**
       * Set little-endian mode for reading and writing multi-byte values.
       */


      setLittleEndian() {
        this.littleEndian = true;
        return this;
      }
      /**
       * Check if big-endian mode is used for reading and writing multi-byte values.
       * @returns `true` if big-endian mode is used, `false` otherwise.
       */


      isBigEndian() {
        return !this.littleEndian;
      }
      /**
       * Switches to big-endian mode for reading and writing multi-byte values.
       */


      setBigEndian() {
        this.littleEndian = false;
        return this;
      }
      /**
       * Move the pointer n bytes forward.
       * @param n - Number of bytes to skip.
       */


      skip() {
        let n = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        this.offset += n;
        return this;
      }
      /**
       * Move the pointer to the given offset.
       * @param offset
       */


      seek(offset) {
        this.offset = offset;
        return this;
      }
      /**
       * Store the current pointer offset.
       * @see {@link IOBuffer#reset}
       */


      mark() {
        this._mark = this.offset;
        return this;
      }
      /**
       * Move the pointer back to the last pointer offset set by mark.
       * @see {@link IOBuffer#mark}
       */


      reset() {
        this.offset = this._mark;
        return this;
      }
      /**
       * Push the current pointer offset to the mark stack.
       * @see {@link IOBuffer#popMark}
       */


      pushMark() {
        this._marks.push(this.offset);

        return this;
      }
      /**
       * Pop the last pointer offset from the mark stack, and set the current
       * pointer offset to the popped value.
       * @see {@link IOBuffer#pushMark}
       */


      popMark() {
        const offset = this._marks.pop();

        if (offset === undefined) {
          throw new Error('Mark stack empty');
        }

        this.seek(offset);
        return this;
      }
      /**
       * Move the pointer offset back to 0.
       */


      rewind() {
        this.offset = 0;
        return this;
      }
      /**
       * Make sure the buffer has sufficient memory to write a given byteLength at
       * the current pointer offset.
       * If the buffer's memory is insufficient, this method will create a new
       * buffer (a copy) with a length that is twice (byteLength + current offset).
       * @param byteLength
       */


      ensureAvailable() {
        let byteLength = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;

        if (!this.available(byteLength)) {
          const lengthNeeded = this.offset + byteLength;
          const newLength = lengthNeeded * 2;
          const newArray = new Uint8Array(newLength);
          newArray.set(new Uint8Array(this.buffer));
          this.buffer = newArray.buffer;
          this.length = this.byteLength = newLength;
          this._data = new DataView(this.buffer);
        }

        return this;
      }
      /**
       * Read a byte and return false if the byte's value is 0, or true otherwise.
       * Moves pointer forward by one byte.
       */


      readBoolean() {
        return this.readUint8() !== 0;
      }
      /**
       * Read a signed 8-bit integer and move pointer forward by 1 byte.
       */


      readInt8() {
        return this._data.getInt8(this.offset++);
      }
      /**
       * Read an unsigned 8-bit integer and move pointer forward by 1 byte.
       */


      readUint8() {
        return this._data.getUint8(this.offset++);
      }
      /**
       * Alias for {@link IOBuffer#readUint8}.
       */


      readByte() {
        return this.readUint8();
      }
      /**
       * Read `n` bytes and move pointer forward by `n` bytes.
       */


      readBytes() {
        let n = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        const bytes = new Uint8Array(n);

        for (let i = 0; i < n; i++) {
          bytes[i] = this.readByte();
        }

        return bytes;
      }
      /**
       * Read a 16-bit signed integer and move pointer forward by 2 bytes.
       */


      readInt16() {
        const value = this._data.getInt16(this.offset, this.littleEndian);

        this.offset += 2;
        return value;
      }
      /**
       * Read a 16-bit unsigned integer and move pointer forward by 2 bytes.
       */


      readUint16() {
        const value = this._data.getUint16(this.offset, this.littleEndian);

        this.offset += 2;
        return value;
      }
      /**
       * Read a 32-bit signed integer and move pointer forward by 4 bytes.
       */


      readInt32() {
        const value = this._data.getInt32(this.offset, this.littleEndian);

        this.offset += 4;
        return value;
      }
      /**
       * Read a 32-bit unsigned integer and move pointer forward by 4 bytes.
       */


      readUint32() {
        const value = this._data.getUint32(this.offset, this.littleEndian);

        this.offset += 4;
        return value;
      }
      /**
       * Read a 32-bit floating number and move pointer forward by 4 bytes.
       */


      readFloat32() {
        const value = this._data.getFloat32(this.offset, this.littleEndian);

        this.offset += 4;
        return value;
      }
      /**
       * Read a 64-bit floating number and move pointer forward by 8 bytes.
       */


      readFloat64() {
        const value = this._data.getFloat64(this.offset, this.littleEndian);

        this.offset += 8;
        return value;
      }
      /**
       * Read a 64-bit signed integer number and move pointer forward by 8 bytes.
       */


      readBigInt64() {
        const value = this._data.getBigInt64(this.offset, this.littleEndian);

        this.offset += 8;
        return value;
      }
      /**
       * Read a 64-bit unsigned integer number and move pointer forward by 8 bytes.
       */


      readBigUint64() {
        const value = this._data.getBigUint64(this.offset, this.littleEndian);

        this.offset += 8;
        return value;
      }
      /**
       * Read a 1-byte ASCII character and move pointer forward by 1 byte.
       */


      readChar() {
        return String.fromCharCode(this.readInt8());
      }
      /**
       * Read `n` 1-byte ASCII characters and move pointer forward by `n` bytes.
       */


      readChars() {
        let n = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        let result = '';

        for (let i = 0; i < n; i++) {
          result += this.readChar();
        }

        return result;
      }
      /**
       * Read the next `n` bytes, return a UTF-8 decoded string and move pointer
       * forward by `n` bytes.
       */


      readUtf8() {
        let n = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        return decode(this.readBytes(n));
      }
      /**
       * Write 0xff if the passed value is truthy, 0x00 otherwise and move pointer
       * forward by 1 byte.
       */


      writeBoolean(value) {
        this.writeUint8(value ? 0xff : 0x00);
        return this;
      }
      /**
       * Write `value` as an 8-bit signed integer and move pointer forward by 1 byte.
       */


      writeInt8(value) {
        this.ensureAvailable(1);

        this._data.setInt8(this.offset++, value);

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as an 8-bit unsigned integer and move pointer forward by 1
       * byte.
       */


      writeUint8(value) {
        this.ensureAvailable(1);

        this._data.setUint8(this.offset++, value);

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * An alias for {@link IOBuffer#writeUint8}.
       */


      writeByte(value) {
        return this.writeUint8(value);
      }
      /**
       * Write all elements of `bytes` as uint8 values and move pointer forward by
       * `bytes.length` bytes.
       */


      writeBytes(bytes) {
        this.ensureAvailable(bytes.length);

        for (let i = 0; i < bytes.length; i++) {
          this._data.setUint8(this.offset++, bytes[i]);
        }

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 16-bit signed integer and move pointer forward by 2
       * bytes.
       */


      writeInt16(value) {
        this.ensureAvailable(2);

        this._data.setInt16(this.offset, value, this.littleEndian);

        this.offset += 2;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 16-bit unsigned integer and move pointer forward by 2
       * bytes.
       */


      writeUint16(value) {
        this.ensureAvailable(2);

        this._data.setUint16(this.offset, value, this.littleEndian);

        this.offset += 2;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 32-bit signed integer and move pointer forward by 4
       * bytes.
       */


      writeInt32(value) {
        this.ensureAvailable(4);

        this._data.setInt32(this.offset, value, this.littleEndian);

        this.offset += 4;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 32-bit unsigned integer and move pointer forward by 4
       * bytes.
       */


      writeUint32(value) {
        this.ensureAvailable(4);

        this._data.setUint32(this.offset, value, this.littleEndian);

        this.offset += 4;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 32-bit floating number and move pointer forward by 4
       * bytes.
       */


      writeFloat32(value) {
        this.ensureAvailable(4);

        this._data.setFloat32(this.offset, value, this.littleEndian);

        this.offset += 4;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 64-bit floating number and move pointer forward by 8
       * bytes.
       */


      writeFloat64(value) {
        this.ensureAvailable(8);

        this._data.setFloat64(this.offset, value, this.littleEndian);

        this.offset += 8;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 64-bit signed bigint and move pointer forward by 8
       * bytes.
       */


      writeBigInt64(value) {
        this.ensureAvailable(8);

        this._data.setBigInt64(this.offset, value, this.littleEndian);

        this.offset += 8;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 64-bit unsigned bigint and move pointer forward by 8
       * bytes.
       */


      writeBigUint64(value) {
        this.ensureAvailable(8);

        this._data.setBigUint64(this.offset, value, this.littleEndian);

        this.offset += 8;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write the charCode of `str`'s first character as an 8-bit unsigned integer
       * and move pointer forward by 1 byte.
       */


      writeChar(str) {
        return this.writeUint8(str.charCodeAt(0));
      }
      /**
       * Write the charCodes of all `str`'s characters as 8-bit unsigned integers
       * and move pointer forward by `str.length` bytes.
       */


      writeChars(str) {
        for (let i = 0; i < str.length; i++) {
          this.writeUint8(str.charCodeAt(i));
        }

        return this;
      }
      /**
       * UTF-8 encode and write `str` to the current pointer offset and move pointer
       * forward according to the encoded length.
       */


      writeUtf8(str) {
        return this.writeBytes(encode(str));
      }
      /**
       * Export a Uint8Array view of the internal buffer.
       * The view starts at the byte offset and its length
       * is calculated to stop at the last written byte or the original length.
       */


      toArray() {
        return new Uint8Array(this.buffer, this.byteOffset, this.lastWrittenByte);
      }
      /**
       * Update the last written byte offset
       * @private
       */


      _updateLastWrittenByte() {
        if (this.offset > this.lastWrittenByte) {
          this.lastWrittenByte = this.offset;
        }
      }

    }

    const endianness = {
      0: 'bigEndian',
      1: 'littleEndian'
    };
    const instrumentTable = {
      0: 'NONE',
      1: 'GSX',
      2: 'ALPHA',
      3: 'ECLIPSE',
      4: 'MASS_SPEC',
      5: 'COMPILER',
      6: 'OTHER_NMR',
      7: 'UNKNOWN',
      8: 'GEMINI',
      9: 'UNITY',
      10: 'ASPECT',
      11: 'UX',
      12: 'FELIX',
      13: 'LAMBDA',
      14: 'GE_1280',
      15: 'GE_OMEGA',
      16: 'CHEMAGNETICS',
      17: 'CDFF',
      18: 'GALACTIC',
      19: 'TRIAD',
      20: 'GENERIC_NMR',
      21: 'GAMMA',
      22: 'JCAMP_DX',
      23: 'AMX',
      24: 'DMX',
      25: 'ECA',
      26: 'ALICE',
      27: 'NMR_PIPE',
      28: 'SIMPSON'
    };
    const dataTypeTable = {
      0: '64Bit Float',
      1: '32Bit Float',
      2: 'Reserved',
      3: 'Reserved'
    };
    const dataFormatTable = {
      1: 'One_D',
      2: 'Two_D',
      3: 'Three_D',
      4: 'Four_D',
      5: 'Five_D',
      6: 'Six_D',
      7: 'Seven_D',
      8: 'Eight_D',
      9: 'not for NMR data formats',
      10: 'not for NMR data formats',
      11: 'not for NMR data formats',
      12: 'Small_Two_D',
      13: 'Small_Three_D',
      14: 'Small_Four_D'
    };
    const dataAxisTypeTable = {
      0: 'None',
      //Axis is not used.
      1: 'Real',
      //Axis has real data only, no imaginary.
      2: 'TPPI',
      3: 'Complex',
      4: 'Real_Complex',

      /* Axis should be accessed as complex when it is the major axis,
                accessed as real otherwise.  This is only valid when all axes in
                use have this setting.*/
      5: 'Envelope'
      /* Behaves the same way as a Real_Complex dimension but the data
            has different meaning.  Instead of being treated as real and
            imaginary parts of a complex number, the data should be treated as minimum and maximum parts of a projection.  This is used
            for the data that results from an envelope projection.*/

    };
    const prefixTable = {
      '-8': 'Yotta',
      '-6': 'Exa',
      '-7': 'Zetta',
      '-5': 'Pecta',
      '-4': 'Tera',
      '-3': 'Giga',
      '-2': 'Mega',
      '-1': 'Kilo',
      0: 'None',
      1: 'Milli',
      2: 'Micro',
      3: 'Nano',
      4: 'Pico',
      5: 'Femto',
      6: 'Atto',
      7: 'Zepto'
    };
    const unitPrefixTable = {
      Yotta: 24,
      Exa: 21,
      Zetta: 18,
      Pecta: 15,
      Tera: 12,
      Giga: 9,
      Mega: 6,
      Kilo: 3,
      None: 0,
      Milli: -3,
      Micro: -6,
      Nano: -9,
      Pico: -12,
      Femto: -15,
      Atto: -18,
      Zepto: -21
    };
    const baseTable = {
      0: 'None',
      1: 'Abundance',
      2: 'Ampere',
      3: 'Candela',
      4: 'Celsius',
      5: 'Coulomb',
      6: 'Degree',
      7: 'Electronvolt',
      8: 'Farad',
      9: 'Sievert',
      10: 'Gram',
      11: 'Gray',
      12: 'Henry',
      13: 'Hertz',
      14: 'Kelvin',
      15: 'Joule',
      16: 'Liter',
      17: 'Lumen',
      18: 'Lux',
      19: 'Meter',
      20: 'Mole',
      21: 'Newton',
      22: 'Ohm',
      23: 'Pascal',
      24: 'Percent',
      25: 'Point',
      26: 'Ppm',
      27: 'Radian',
      28: 'Second',
      29: 'Siemens',
      30: 'Steradian',
      31: 'Tesla',
      32: 'Volt',
      33: 'Watt',
      34: 'Weber',
      35: 'Decibel',
      36: 'Dalton',
      37: 'Thompson',
      38: 'Ugeneric',
      // Treated as None, but never displayed',
      39: 'LPercent ',
      // Treated as percent for display, but different for comparison',
      40: 'PPT',
      // Parts per trillion (Private, do not use)',
      41: 'PPB ',
      // Parts per billion (Private, do not use)',
      42: 'Index'
    };
    const dataAxisRangedTable = {
      0: 'Ranged',

      /* The ruler for the axis ranges from Data_Axis_Start[n] to
            Data_Axis_Stop[n] with a step function of
                (Data_Axis_Stop[n] - Data_Axis_Start[n]) /
                (Data_Offset_Stop[n] - Data_Offset_Start[n]) */
      1: 'Listed',
      // (deprecated)

      /* The ruler for the axis is a list of doubles stored in the
            List Section.  Values in the ruler may be anything.*/
      2: 'Sparse',

      /*The ruler for the axis is a list of doubles stored in the
            List Section.  Values in the rulers must be strictly monotonically
            increasing or decreasing.*/
      3: 'Listed'
      /* The ruler for the axis is a list of doubles stored in the
            List Section.  Values in the rulers do not fit definition of Sparse.*/

    };
    const valueTypeTable = {
      0: 'String',
      1: 'Integer',
      2: 'Float',
      3: 'Complex',
      4: 'Infinity'
    };

    function getPar(param, searchStr) {
      return param.paramArray.find(o => o.name === searchStr) || '';
    }
    function getDigitalFilter(param) {
      const orders = param.paramArray.find(e => e.name === 'orders');
      const factors = param.paramArray.find(e => e.name === 'factors');
      const sweep = param.paramArray.find(e => e.name === 'x_sweep');
      const acqTime = param.paramArray.find(e => e.name === 'x_acq_time');
      const nbPoints = param.paramArray.find(e => e.name === 'x_points');
      const shouldStop = [orders, factors, sweep, acqTime, nbPoints].some(e => e === undefined);

      if (shouldStop) {
        return;
      }

      const s = parseInt(orders.value.slice(0, 1), 10);
      const jump = orders.value.slice(1).length / s;
      let arg = 0;
      let factorNumber = new Int8Array(s);
      let offsetO = 1;
      let offsetF = 0;

      for (let i = 0; i < s; i++) {
        factorNumber[i] = parseInt(factors.value.slice(offsetF, offsetF + 1), 10);
        offsetF += 1;
      }

      for (let i = 0; i < s; i++) {
        let productorial = 1;

        for (let j = i; j < s; j++) {
          productorial *= factorNumber[j];
        }

        arg += (parseInt(orders.value.slice(offsetO, offsetO + jump), 10) - 1) / productorial;
        offsetO += jump;
      }

      arg /= 2;
      const delaySec = arg / sweep.value;
      return delaySec / acqTime.value * (nbPoints.value - 1);
    }
    function getMagnitude(param, searchStr) {
      let par = getPar(param, searchStr) || 'NA';

      if (par === 'NA') {
        return {
          magnitude: 'NA',
          unit: 'NA'
        };
      }

      let unit = par.unit[0].base;
      let unitMult = unitPrefixTable[par.unit[0].prefix];
      let magnitude = par.value * 10 ** unitMult;
      return {
        magnitude,
        unit
      };
    }
    function getUnit(buffer, size) {
      let unit = [];

      for (let i = 0; i < size; i++) {
        let byte = buffer.readByte();
        let prefix = prefixTable[byte >> 4];
        let power = byte & 0b00001111;
        let base = baseTable[buffer.readInt8()];
        unit.push({
          prefix,
          power,
          base
        });
      }

      return unit;
    }
    function getString(buffer, size) {
      let string = [];

      for (let i = 0; i < size; i++) {
        let char = buffer.readChar();

        if (char !== '\u0000') {
          string.push(char);
        }
      }

      return string.join('');
    }
    function getParamName(buffer, size) {
      let string = [];

      for (let i = 0; i < size; i++) {
        let char = buffer.readChar();

        if (char !== ' ') {
          string.push(char);
        }
      }

      return string.join('');
    }
    function getArray(buffer, size, format) {
      let double = [];

      for (let i = 0; i < size; i++) {
        switch (format) {
          case 'readUint32':
            double.push(buffer.readUint32());
            break;

          case 'readFloat64':
            double.push(buffer.readFloat64());
            break;

          case 'readFloat32':
            double.push(buffer.readFloat32());
            break;

          case 'readUint8':
            double.push(buffer.readUint8());
            break;

          case 'readBoolean':
            double.push(buffer.readBoolean());
            break;
        }
      }

      return double;
    }

    /**
     * A parser for 1D and 2D JDL NMR Files
     * @param {ArrayBuffer} buffer - a buffer object containing the JDL file
     * @return {Object} - an Object with converted data
     */

    function parseJEOL(buffer) {
      let ioBuffer = new IOBuffer(buffer);
      ioBuffer.setBigEndian(); // read header section

      let byte;
      let header = {};
      let byteArray = [];
      header.fileIdentifier = ioBuffer.readChars(8);
      header.endian = endianness[ioBuffer.readInt8()];
      header.majorVersion = ioBuffer.readUint8();
      header.minorVersion = ioBuffer.readUint16();
      header.dataDimensionNumber = ioBuffer.readUint8();
      header.dataDimensionExist = ioBuffer.readByte().toString(2).split('').map(x => Boolean(Number(x)));
      byte = ioBuffer.readByte();
      header.dataType = dataTypeTable[byte >> 6];
      header.dataFormat = dataFormatTable[byte & 0b00111111];
      header.dataInstrument = instrumentTable[ioBuffer.readInt8()];
      header.translate = getArray(ioBuffer, 8, 'readUint8');
      header.dataAxisType = getArray(ioBuffer, 8, 'readUint8').map(x => dataAxisTypeTable[x]);
      header.dataUnits = getUnit(ioBuffer, 8);
      header.title = getString(ioBuffer, 124);

      for (byte in getArray(ioBuffer, 4, 'readUint8')) {
        byteArray.push(dataAxisRangedTable[byte >> 4]);
        byteArray.push(dataAxisRangedTable[byte & 0b00001111]);
      }

      header.dataAxisRanged = byteArray;
      header.dataPoints = getArray(ioBuffer, 8, 'readUint32');
      header.dataOffsetStart = getArray(ioBuffer, 8, 'readUint32');
      header.dataOffsetStop = getArray(ioBuffer, 8, 'readUint32');
      header.dataAxisStart = getArray(ioBuffer, 8, 'readFloat64');
      header.dataAxisStop = getArray(ioBuffer, 8, 'readFloat64');
      byteArray = new Uint8Array(4);

      for (let i = 0; i < 4; i++) {
        byteArray[i] = ioBuffer.readByte();
      }

      let year = 1990 + (byteArray[0] >> 1);
      let month = (byteArray[0] << 3 & 0b00001000) + (byteArray[1] >> 5);
      let day = byteArray[2] & 0b00011111;
      header.creationTime = {
        year,
        month,
        day
      };

      for (let i = 0; i < 4; i++) {
        byteArray[i] = ioBuffer.readByte();
      }

      year = 1990 + (byteArray[0] >> 1);
      month = (byteArray[0] << 3 & 0b00001000) + (byteArray[1] >> 5);
      day = byteArray[2] & 0b00011111;
      header.revisionTime = {
        year,
        month,
        day
      };
      header.nodeName = getString(ioBuffer, 16);
      header.site = getString(ioBuffer, 128);
      header.author = getString(ioBuffer, 128);
      header.comment = getString(ioBuffer, 128);
      let dataAxisTitles = [];

      for (let i = 0; i < 8; i++) {
        dataAxisTitles.push(getString(ioBuffer, 32));
      }

      header.dataAxisTitles = dataAxisTitles;
      header.baseFreq = getArray(ioBuffer, 8, 'readFloat64');
      header.zeroPoint = getArray(ioBuffer, 8, 'readFloat64');
      header.reversed = getArray(ioBuffer, 8, 'readBoolean');
      ioBuffer.skip(3);
      header.annotationOK = Boolean(ioBuffer.readByte() >> 7);
      header.historyUsed = ioBuffer.readUint32();
      header.historyLength = ioBuffer.readUint32();
      header.paramStart = ioBuffer.readUint32();
      header.paramLength = ioBuffer.readUint32();
      header.ListStart = getArray(ioBuffer, 8, 'readUint32');
      header.ListLength = getArray(ioBuffer, 8, 'readUint32');
      header.dataStart = ioBuffer.readUint32();
      header.dataLength = ioBuffer.readUint32() << 32 | ioBuffer.readUint32();
      header.contextStart = ioBuffer.readUint32() << 32 | ioBuffer.readUint32();
      header.contextLength = ioBuffer.readUint32();
      header.annoteStart = ioBuffer.readUint32() << 32 | ioBuffer.readUint32();
      header.annoteLength = ioBuffer.readUint32();
      header.totalSize = ioBuffer.readUint32() << 32 | ioBuffer.readUint32();
      header.unitLocation = getArray(ioBuffer, 8, 'readUint8');
      let compoundUnit = [];

      for (let i = 0; i < 2; i++) {
        let unit = [];
        let scaler = ioBuffer.readInt16();

        for (let j = 0; j < 5; j++) {
          byte = ioBuffer.readInt16();
          unit.push(byte);
        }

        compoundUnit.push({
          scaler,
          unit
        });
      }

      header.compoundUnit = compoundUnit; // section parameters (param header and array)

      if (header.endian === 'littleEndian') {
        ioBuffer.setLittleEndian();
      }

      ioBuffer.seek(header.paramStart);
      let parameters = {
        parameterSize: ioBuffer.readUint32(),
        lowIndex: ioBuffer.readUint32(),
        highIndex: ioBuffer.readUint32(),
        totalSize: ioBuffer.readUint32()
      };
      let paramArray = [];

      for (let p = 0; p < parameters.highIndex + 1; p++) {
        ioBuffer.skip(4);
        let scaler = ioBuffer.readInt16();
        let unit = getUnit(ioBuffer, 5);
        ioBuffer.skip(16);
        let valueType = valueTypeTable[ioBuffer.readInt32()];
        ioBuffer.seek(ioBuffer.offset - 20);
        let value;

        switch (valueType) {
          case 'String':
            value = getParamName(ioBuffer, 16);
            break;

          case 'Integer':
            value = ioBuffer.readInt32();
            ioBuffer.skip(12);
            break;

          case 'Float':
            value = ioBuffer.readFloat64();
            ioBuffer.skip(8);
            break;

          case 'Complex':
            value.Real = ioBuffer.readFloat64();
            value.Imag = ioBuffer.readFloat64();
            break;

          case 'Infinity':
            value = ioBuffer.readInt32();
            ioBuffer.skip(12);
            break;

          default:
            ioBuffer.skip(16);
            break;
        }

        ioBuffer.skip(4);
        let name = getParamName(ioBuffer, 28);
        paramArray.push({
          name: name.toLowerCase(),
          scaler,
          unit,
          value,
          valueType
        });
      }

      parameters.paramArray = paramArray; // data section

      ioBuffer.seek(header.dataStart);

      if (header.endian === 'littleEndian') {
        ioBuffer.setLittleEndian();
      }

      let data = {};
      let dataSectionCount = 1;
      let realComplex = 0;

      for (let type of header.dataAxisType) {
        if (type === 'Real_Complex' && realComplex === 0) {
          dataSectionCount += 1;
          realComplex += 1;
        }

        if (type === 'Complex') {
          dataSectionCount *= 2;
        }
      }

      if (header.dataFormat !== 'One_D' && header.dataFormat !== 'Two_D') {
        throw new Error('Only One_D and two_D data formats are implemented yet');
      }

      if (header.dataFormat === 'One_D') {
        for (let s = 0; s < dataSectionCount; s++) {
          let section;

          if (header.dataType === '32Bit Float') {
            section = getArray(ioBuffer, header.dataPoints[0], 'readFloat32');
          } else if (header.dataType === '64Bit Float') {
            section = getArray(ioBuffer, header.dataPoints[0], 'readFloat64');
          }

          if (s === 0) data.re = section;
          if (s === 1) data.im = section;
        }
      }

      if (header.dataFormat === 'Two_D') {
        let me = 32;
        let dim1 = header.dataPoints[0];
        let dim2 = header.dataPoints[1]; // console.log(
        // `dim1: ${dim1},
        // dim2: ${dim2},
        // total: ${dim1 * dim2},
        // total(byte): ${dim1 * dim2 * 8},
        // total(length): ${dim1 * dim2 * 8 * dataSectionCount}
        // m size: ${dim1 / me} / ${dim2 / me}`,
        // );

        let I = dim2 / me;
        let J = dim1 / me;

        for (let s = 0; s < dataSectionCount; s++) {
          let section;

          for (let i = 0; i < I; i++) {
            let row = [];

            for (let j = 0; j < J; j++) {
              for (let k = 0; k < me; k++) {
                if (j === 0) {
                  if (header.dataType === '32Bit Float') {
                    row[k] = getArray(ioBuffer, me, 'readFloat32');
                  } else if (header.dataType === '64Bit Float') {
                    row[k] = getArray(ioBuffer, me, 'readFloat64');
                  }
                } else {
                  if (header.dataType === '32Bit Float') {
                    row[k] = row[k].concat(getArray(ioBuffer, me, 'readFloat32'));
                  } else if (header.dataType === '64Bit Float') {
                    row[k] = row[k].concat(getArray(ioBuffer, me, 'readFloat64'));
                  }
                }
              }
            }

            if (i === 0) {
              section = row;
            } else {
              section = section.concat(row);
            }
          }

          if (dataSectionCount === 2) {
            if (s === 0) data.re = section;
            if (s === 1) data.im = section;
          }

          if (dataSectionCount === 4) {
            if (s === 0) {
              data.re = {};
              data.re.re = section;
            }

            if (s === 1) data.re.im = section;

            if (s === 2) {
              data.im = {};
              data.im.re = section;
            }

            if (s === 3) data.im.im = section;
          }
        }
      } // format output


      let nucleus = [];
      let acquisitionTime = [];
      let spectralWidth = [];
      let spectralWidthClipped = [];
      let resolution = [];
      let originFrequency = [];
      let frequencyOffset = [];
      let dataUnits = [];

      if (header.dataFormat === 'One_D' || header.dataFormat === 'Two_D') {
        nucleus.push(getPar(parameters, 'x_domain').value);
        acquisitionTime.push(getMagnitude(parameters, 'x_acq_time'));
        spectralWidth.push(getMagnitude(parameters, 'x_sweep'));
        spectralWidthClipped.push(getMagnitude(parameters, 'x_sweep_clipped'));
        resolution.push(getMagnitude(parameters, 'x_resolution'));
        originFrequency.push(getMagnitude(parameters, 'x_freq'));
        frequencyOffset.push(getMagnitude(parameters, 'x_offset'));
        dataUnits.push(header.dataUnits[0].base);
      }

      if (header.dataFormat === 'Two_D') {
        nucleus.push(getPar(parameters, 'y_domain').value);
        acquisitionTime.push(getMagnitude(parameters, 'y_acq_time'));
        spectralWidth.push(getMagnitude(parameters, 'y_sweep'));
        resolution.push(getMagnitude(parameters, 'y_resolution'));
        originFrequency.push(getMagnitude(parameters, 'y_freq'));
        frequencyOffset.push(getMagnitude(parameters, 'y_offset'));
        dataUnits.push(header.dataUnits[1].base);
      }

      let digest = {
        info: {
          sampleName: getPar(parameters, 'sample_id').value,
          creationTime: header.creationTime,
          revisionTime: header.revisionTime,
          author: header.author,
          comment: header.comment,
          solvent: getPar(parameters, 'solvent').value,
          temperature: getMagnitude(parameters, 'temp_get'),
          probeName: getPar(parameters, 'probe_id').value,
          fieldStrength: getMagnitude(parameters, 'field_strength'),
          experiment: getPar(parameters, 'experiment').value,
          dimension: header.dataDimensionNumber,
          nucleus,
          pulseStrength90: getMagnitude(parameters, 'x90'),
          numberOfScans: getPar(parameters, 'scans').value,
          relaxationTime: getMagnitude(parameters, 'relaxation_delay'),
          dataPoints: header.dataPoints.slice(0, header.dataDimensionNumber),
          dataOffsetStart: header.dataOffsetStart,
          dataOffsetStop: header.dataOffsetStop,
          dataUnits: dataUnits,
          dataSections: Object.keys(data),
          originFrequency,
          frequencyOffset,
          acquisitionTime,
          spectralWidth,
          spectralWidthClipped,
          dataAxisStart: header.dataAxisStart,
          dataAxisStop: header.dataAxisStop,
          resolution: resolution,
          decimationRate: getPar(parameters, 'decimation_rate').value,
          paramList: JSON.stringify(parameters.paramArray.map(par => par.name))
        },
        headers: header,
        parameters: parameters,
        data: data
      };
      digest.info.digitalFilter = getDigitalFilter(parameters);
      return digest;
    }

    // sources:
    // https://en.wikipedia.org/wiki/Gyromagnetic_ratio
    // TODO: #13 can we have a better source and more digits ? @jwist
    const gyromagneticRatio = {
      '1H': 267.52218744e6,
      '2H': 41.065e6,
      '3H': 285.3508e6,
      '3He': -203.789e6,
      '7Li': 103.962e6,
      '13C': 67.28284e6,
      '14N': 19.331e6,
      '15N': -27.116e6,
      '17O': -36.264e6,
      '19F': 251.662e6,
      '23Na': 70.761e6,
      '27Al': 69.763e6,
      '29Si': -53.19e6,
      '31P': 108.291e6,
      '57Fe': 8.681e6,
      '63Cu': 71.118e6,
      '67Zn': 16.767e6,
      '129Xe': -73.997e6
    };

    var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

    function commonjsRequire (path) {
    	throw new Error('Could not dynamically require "' + path + '". Please configure the dynamicRequireTargets or/and ignoreDynamicRequires option of @rollup/plugin-commonjs appropriately for this require call to work.');
    }

    /**
     * Ensure that the data is string. If it is an ArrayBuffer it will be converted to string using TextDecoder.
     * @param blob
     * @param options
     * @returns
     */
    function ensureString(blob) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

      if (typeof blob === 'string') {
        return blob;
      }

      if (ArrayBuffer.isView(blob) || blob instanceof ArrayBuffer) {
        const {
          encoding = guessEncoding(blob)
        } = options;
        const decoder = new TextDecoder(encoding);
        return decoder.decode(blob);
      }

      throw new TypeError(`blob must be a string, ArrayBuffer or ArrayBufferView`);
    }

    function guessEncoding(blob) {
      const uint8 = ArrayBuffer.isView(blob) ? new Uint8Array(blob.buffer, blob.byteOffset, blob.byteLength) : new Uint8Array(blob);

      if (uint8.length >= 2) {
        if (uint8[0] === 0xfe && uint8[1] === 0xff) {
          return 'utf-16be';
        }

        if (uint8[0] === 0xff && uint8[1] === 0xfe) {
          return 'utf-16le';
        }
      }

      return 'utf-8';
    }

    const toString = Object.prototype.toString;
    function isAnyArray(object) {
      return toString.call(object).endsWith('Array]');
    }

    var medianQuickselect_min = {exports: {}};

    (function (module) {
      (function () {
        function a(d) {
          for (var e = 0, f = d.length - 1, g = void 0, h = void 0, i = void 0, j = c(e, f); !0;) {
            if (f <= e) return d[j];
            if (f == e + 1) return d[e] > d[f] && b(d, e, f), d[j];

            for (g = c(e, f), d[g] > d[f] && b(d, g, f), d[e] > d[f] && b(d, e, f), d[g] > d[e] && b(d, g, e), b(d, g, e + 1), h = e + 1, i = f; !0;) {
              do h++; while (d[e] > d[h]);

              do i--; while (d[i] > d[e]);

              if (i < h) break;
              b(d, h, i);
            }

            b(d, e, i), i <= j && (e = h), i >= j && (f = i - 1);
          }
        }

        var b = function b(d, e, f) {
          var _ref;

          return _ref = [d[f], d[e]], d[e] = _ref[0], d[f] = _ref[1], _ref;
        },
            c = function c(d, e) {
          return ~~((d + e) / 2);
        };

        module.exports ? module.exports = a : window.median = a;
      })();
    })(medianQuickselect_min);

    var quickSelectMedian = medianQuickselect_min.exports;

    function median(input) {
      if (!isAnyArray(input)) {
        throw new TypeError('input must be an array');
      }

      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }

      return quickSelectMedian(input.slice());
    }

    var name = "nmr-parser";
    var version = "1.7.2";
    var description = "Read and convert any NMR file";
    var main = "lib/index.js";
    var module = "src/index.js";
    var files = [
    	"lib",
    	"src"
    ];
    var scripts = {
    	eslint: "eslint src",
    	"eslint-fix": "npm run eslint -- --fix",
    	compile: "rollup -c",
    	prepublishOnly: "npm run compile",
    	test: "npm run compile && npm run test-coverage && npm run eslint",
    	"test-coverage": "jest --coverage",
    	"test-only": "jest",
    	build: "cheminfo-build --entry src/index.js --root NMRparser"
    };
    var repository = {
    	type: "git",
    	url: "git+https://github.com/cheminfo/nmr-parser.git"
    };
    var keywords = [
    	"nmr",
    	"magnetic resonance",
    	"parser",
    	"bruker",
    	"JEOL",
    	"CSDM",
    	"data analysis"
    ];
    var author = "Julien Wist";
    var license = "MIT";
    var bugs = {
    	url: "https://github.com/cheminfo/nmr-parser/issues"
    };
    var homepage = "https://github.com/cheminfo/nmr-parser#readme";
    var jest = {
    	testEnvironment: "node"
    };
    var prettier = {
    	arrowParens: "always",
    	semi: true,
    	singleQuote: true,
    	tabWidth: 2,
    	trailingComma: "all"
    };
    var devDependencies = {
    	"@babel/plugin-transform-modules-commonjs": "^7.16.8",
    	"@rollup/plugin-json": "^4.1.0",
    	"@types/jest": "^27.4.0",
    	"bruker-data-test": "^0.1.0",
    	"cheminfo-build": "^1.1.11",
    	eslint: "^8.7.0",
    	"eslint-config-cheminfo": "^7.2.1",
    	"jcamp-data-test": "^0.0.5",
    	"jeol-data-test": "^0.2.3",
    	jest: "^27.4.7",
    	prettier: "^2.5.1",
    	rollup: "^2.65.0"
    };
    var dependencies = {
    	brukerconverter: "^3.5.0",
    	"is-any-array": "^2.0.0",
    	jcampconverter: "^8.2.4",
    	jeolconverter: "^1.0.1",
    	"nmr-processing": "^6.0.2"
    };
    var packageJson = {
    	name: name,
    	version: version,
    	description: description,
    	main: main,
    	module: module,
    	files: files,
    	scripts: scripts,
    	repository: repository,
    	keywords: keywords,
    	author: author,
    	license: license,
    	bugs: bugs,
    	homepage: homepage,
    	jest: jest,
    	prettier: prettier,
    	devDependencies: devDependencies,
    	dependencies: dependencies
    };

    function toKeyValue(object) {
      let newObject = {};

      for (let key in object) {
        if (typeof object[key] !== 'string') {
          newObject[key] = JSON.stringify(object[key]);
        } else {
          newObject[key] = object[key];
        }
      }

      return newObject;
    }

    function fromJEOL(buffer) {
      let parsedData = parseJEOL(buffer);
      let info = parsedData.info;
      let headers = parsedData.headers;
      let parameters = parsedData.parameters;
      let paramArray = Object.assign({}, parameters.paramArray);
      delete parameters.paramArray;
      let data = parsedData.data; // curation of parameters

      let newInfo = {};
      newInfo.title = `title: ${headers.title} / comment: ${headers.comment} / author:${headers.author} / site: ${headers.site}`;
      newInfo.nucleus = info.nucleus.map(x => {
        if (x === 'Proton') {
          x = '1H';
        }

        if (x === 'Carbon13') {
          x = '13C';
        }

        if (x === 'Nitrogen15') {
          x = '15N';
        }

        return x;
      });
      newInfo.sampleName = info.sampleName;
      newInfo.date = JSON.stringify(info.creationTime);
      newInfo.author = info.author; //newInfo.comment = info.comment;

      newInfo.solvent = info.solvent;
      newInfo.temperature = info.temperature.magnitude;
      newInfo.probeName = info.probeName || '';
      newInfo.fieldStrength = info.fieldStrength.magnitude;
      let gyromagneticRatioConstants = newInfo.nucleus.map(nucleus => gyromagneticRatio[nucleus]);
      newInfo.baseFrequency = gyromagneticRatioConstants.map(gmr => info.fieldStrength.magnitude * gmr / (2 * Math.PI * 1e6));
      newInfo.pulseSequence = info.experiment;
      newInfo.temperature = info.temperature.unit.toLowerCase() === 'celsius' ? 273.15 + info.temperature.magnitude : info.temperature.magnitude;
      newInfo.digitalFilter = info.digitalFilter;
      newInfo.pulseStrength90 = 1 / (4 * info.pulseStrength90.magnitude);
      newInfo.numberOfScans = info.numberOfScans;
      newInfo.relaxationTime = info.relaxationTime.magnitude;
      newInfo.isComplex = info.dataSections.includes('im');
      newInfo.isFid = info.dataUnits[0] === 'Second';
      newInfo.isFt = info.dataUnits[0] === 'Ppm';
      newInfo.dimension = info.dimension;
      const dimension = newInfo.dimension;
      newInfo.originFrequency = info.originFrequency.map(d => d.magnitude / 1e6).slice(0, dimension);
      newInfo.numberOfPoints = info.dataPoints.slice(0, 1);
      newInfo.frequencyOffset = info.frequencyOffset.map((f, i) => f.magnitude * newInfo.baseFrequency[i]).slice(0, dimension);
      newInfo.acquisitionTime = info.acquisitionTime.map(a => a.magnitude).slice(0, dimension);
      newInfo.spectralWidth = info.spectralWidth.map((sw, i) => sw.magnitude / info.originFrequency[i].magnitude * 1e6).slice(0, dimension); // set options for dimensions

      let dimensions = [];
      let options = {};
      let increment;

      for (let d = 0; d < info.dimension; d++) {
        increment = {
          magnitude: info.acquisitionTime[d].magnitude / (info.dataPoints[d] - 1),
          unit: 's'
        };

        if (info.dataUnits[d] === 'Second') {
          options.quantityName = 'time';
          options.originOffset = {
            magnitude: 0,
            unit: 's'
          };

          if (d === 0) {
            options.coordinatesOffset = {
              magnitude: info.digitalFilter * increment.magnitude,
              unit: 's'
            };
          } else {
            options.coordinatesOffset = {
              magnitude: 0,
              unit: 's'
            };
          }

          options.reciprocal = {
            originOffset: {
              magnitude: info.originFrequency[d].magnitude,
              unit: 'Hz'
            },
            quantityName: 'frequency',
            coordinatesOffset: {
              magnitude: info.frequencyOffset[d].magnitude * info.originFrequency[d].magnitude / 1000000,
              unit: 'Hz'
            }
          };
        } else if (info.dataUnits[d] === 'Ppm') {
          options.quantityName = 'frequency';
          let origin = info.originFrequency[d].magnitude;
          options.originOffset = {
            magnitude: origin,
            unit: 'Hz'
          };
          let firstPoint = info.dataOffsetStart[0];
          let lastPoint = info.dataOffsetStop[0];
          let dataLength = lastPoint - firstPoint + 1;
          let spectralWidth = info.spectralWidth[d].magnitude;
          let incr = spectralWidth / info.dataPoints[d];
          increment = {
            magnitude: incr,
            unit: 'Hz'
          };
          let offset = info.dataAxisStop[0] * origin / 1000000;
          options.coordinatesOffset = {
            magnitude: offset,
            unit: 'Hz'
          }; // after increment is computed with whole frequency
          // and original number of points, we recast the
          // number of point for export

          if (dataLength < info.dataPoints[d]) {
            info.dataPoints[d] = dataLength;
          }
        }

        if (d === 0) {
          options.description = 'direct dimension';
        } else {
          options.description = 'indirect dimension';
        }

        dimensions.push(formatLinearDimension(headers.dataAxisTitles[d], info.dataPoints[d], increment, options));
      } // set options for dependentVariable


      options = {
        unit: 'none',
        quantityName: 'relative intensity',
        from: info.dataOffsetStart,
        to: info.dataOffsetStop
      };
      let dependentVariables = [];
      dependentVariables.push(formatDependentVariable(data, 11, options));
      let description = Object.assign({}, newInfo);
      delete description.paramList;
      description.metadata = Object.assign({}, toKeyValue(headers), toKeyValue(parameters), toKeyValue(paramArray));
      let dataStructure = {
        timeStamp: Date.now(),
        version: packageJson.version,
        description,
        tags: ['magnetic resonance'].concat(newInfo.nucleus),
        application: {
          spectralWidthClipped: info.spectralWidthClipped[0].magnitude / newInfo.baseFrequency[0]
        },
        dimensions: dimensions,
        dependentVariables: dependentVariables
      };
      return [dataStructure];
    }

    var jszip_min = {exports: {}};

    /*!

    JSZip v3.7.1 - A JavaScript class for generating and reading zip files
    <http://stuartk.com/jszip>

    (c) 2009-2016 Stuart Knightley <stuart [at] stuartk.com>
    Dual licenced under the MIT license or GPLv3. See https://raw.github.com/Stuk/jszip/master/LICENSE.markdown.

    JSZip uses the library pako released under the MIT license :
    https://github.com/nodeca/pako/blob/master/LICENSE
    */

    (function (module, exports) {
      !function (t) {
        module.exports = t();
      }(function () {
        return function s(a, o, h) {
          function u(r, t) {
            if (!o[r]) {
              if (!a[r]) {
                var e = "function" == typeof commonjsRequire && commonjsRequire;
                if (!t && e) return e(r, !0);
                if (l) return l(r, !0);
                var i = new Error("Cannot find module '" + r + "'");
                throw i.code = "MODULE_NOT_FOUND", i;
              }

              var n = o[r] = {
                exports: {}
              };
              a[r][0].call(n.exports, function (t) {
                var e = a[r][1][t];
                return u(e || t);
              }, n, n.exports, s, a, o, h);
            }

            return o[r].exports;
          }

          for (var l = "function" == typeof commonjsRequire && commonjsRequire, t = 0; t < h.length; t++) u(h[t]);

          return u;
        }({
          1: [function (t, e, r) {

            var c = t("./utils"),
                d = t("./support"),
                p = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";
            r.encode = function (t) {
              for (var e, r, i, n, s, a, o, h = [], u = 0, l = t.length, f = l, d = "string" !== c.getTypeOf(t); u < t.length;) f = l - u, i = d ? (e = t[u++], r = u < l ? t[u++] : 0, u < l ? t[u++] : 0) : (e = t.charCodeAt(u++), r = u < l ? t.charCodeAt(u++) : 0, u < l ? t.charCodeAt(u++) : 0), n = e >> 2, s = (3 & e) << 4 | r >> 4, a = 1 < f ? (15 & r) << 2 | i >> 6 : 64, o = 2 < f ? 63 & i : 64, h.push(p.charAt(n) + p.charAt(s) + p.charAt(a) + p.charAt(o));

              return h.join("");
            }, r.decode = function (t) {
              var e,
                  r,
                  i,
                  n,
                  s,
                  a,
                  o = 0,
                  h = 0,
                  u = "data:";
              if (t.substr(0, u.length) === u) throw new Error("Invalid base64 input, it looks like a data url.");
              var l,
                  f = 3 * (t = t.replace(/[^A-Za-z0-9\+\/\=]/g, "")).length / 4;
              if (t.charAt(t.length - 1) === p.charAt(64) && f--, t.charAt(t.length - 2) === p.charAt(64) && f--, f % 1 != 0) throw new Error("Invalid base64 input, bad content length.");

              for (l = d.uint8array ? new Uint8Array(0 | f) : new Array(0 | f); o < t.length;) e = p.indexOf(t.charAt(o++)) << 2 | (n = p.indexOf(t.charAt(o++))) >> 4, r = (15 & n) << 4 | (s = p.indexOf(t.charAt(o++))) >> 2, i = (3 & s) << 6 | (a = p.indexOf(t.charAt(o++))), l[h++] = e, 64 !== s && (l[h++] = r), 64 !== a && (l[h++] = i);

              return l;
            };
          }, {
            "./support": 30,
            "./utils": 32
          }],
          2: [function (t, e, r) {

            var i = t("./external"),
                n = t("./stream/DataWorker"),
                s = t("./stream/Crc32Probe"),
                a = t("./stream/DataLengthProbe");

            function o(t, e, r, i, n) {
              this.compressedSize = t, this.uncompressedSize = e, this.crc32 = r, this.compression = i, this.compressedContent = n;
            }

            o.prototype = {
              getContentWorker: function () {
                var t = new n(i.Promise.resolve(this.compressedContent)).pipe(this.compression.uncompressWorker()).pipe(new a("data_length")),
                    e = this;
                return t.on("end", function () {
                  if (this.streamInfo.data_length !== e.uncompressedSize) throw new Error("Bug : uncompressed data size mismatch");
                }), t;
              },
              getCompressedWorker: function () {
                return new n(i.Promise.resolve(this.compressedContent)).withStreamInfo("compressedSize", this.compressedSize).withStreamInfo("uncompressedSize", this.uncompressedSize).withStreamInfo("crc32", this.crc32).withStreamInfo("compression", this.compression);
              }
            }, o.createWorkerFrom = function (t, e, r) {
              return t.pipe(new s()).pipe(new a("uncompressedSize")).pipe(e.compressWorker(r)).pipe(new a("compressedSize")).withStreamInfo("compression", e);
            }, e.exports = o;
          }, {
            "./external": 6,
            "./stream/Crc32Probe": 25,
            "./stream/DataLengthProbe": 26,
            "./stream/DataWorker": 27
          }],
          3: [function (t, e, r) {

            var i = t("./stream/GenericWorker");
            r.STORE = {
              magic: "\0\0",
              compressWorker: function (t) {
                return new i("STORE compression");
              },
              uncompressWorker: function () {
                return new i("STORE decompression");
              }
            }, r.DEFLATE = t("./flate");
          }, {
            "./flate": 7,
            "./stream/GenericWorker": 28
          }],
          4: [function (t, e, r) {

            var i = t("./utils");

            var o = function () {
              for (var t, e = [], r = 0; r < 256; r++) {
                t = r;

                for (var i = 0; i < 8; i++) t = 1 & t ? 3988292384 ^ t >>> 1 : t >>> 1;

                e[r] = t;
              }

              return e;
            }();

            e.exports = function (t, e) {
              return void 0 !== t && t.length ? "string" !== i.getTypeOf(t) ? function (t, e, r, i) {
                var n = o,
                    s = i + r;
                t ^= -1;

                for (var a = i; a < s; a++) t = t >>> 8 ^ n[255 & (t ^ e[a])];

                return -1 ^ t;
              }(0 | e, t, t.length, 0) : function (t, e, r, i) {
                var n = o,
                    s = i + r;
                t ^= -1;

                for (var a = i; a < s; a++) t = t >>> 8 ^ n[255 & (t ^ e.charCodeAt(a))];

                return -1 ^ t;
              }(0 | e, t, t.length, 0) : 0;
            };
          }, {
            "./utils": 32
          }],
          5: [function (t, e, r) {

            r.base64 = !1, r.binary = !1, r.dir = !1, r.createFolders = !0, r.date = null, r.compression = null, r.compressionOptions = null, r.comment = null, r.unixPermissions = null, r.dosPermissions = null;
          }, {}],
          6: [function (t, e, r) {

            var i = null;
            i = "undefined" != typeof Promise ? Promise : t("lie"), e.exports = {
              Promise: i
            };
          }, {
            lie: 37
          }],
          7: [function (t, e, r) {

            var i = "undefined" != typeof Uint8Array && "undefined" != typeof Uint16Array && "undefined" != typeof Uint32Array,
                n = t("pako"),
                s = t("./utils"),
                a = t("./stream/GenericWorker"),
                o = i ? "uint8array" : "array";

            function h(t, e) {
              a.call(this, "FlateWorker/" + t), this._pako = null, this._pakoAction = t, this._pakoOptions = e, this.meta = {};
            }

            r.magic = "\b\0", s.inherits(h, a), h.prototype.processChunk = function (t) {
              this.meta = t.meta, null === this._pako && this._createPako(), this._pako.push(s.transformTo(o, t.data), !1);
            }, h.prototype.flush = function () {
              a.prototype.flush.call(this), null === this._pako && this._createPako(), this._pako.push([], !0);
            }, h.prototype.cleanUp = function () {
              a.prototype.cleanUp.call(this), this._pako = null;
            }, h.prototype._createPako = function () {
              this._pako = new n[this._pakoAction]({
                raw: !0,
                level: this._pakoOptions.level || -1
              });
              var e = this;

              this._pako.onData = function (t) {
                e.push({
                  data: t,
                  meta: e.meta
                });
              };
            }, r.compressWorker = function (t) {
              return new h("Deflate", t);
            }, r.uncompressWorker = function () {
              return new h("Inflate", {});
            };
          }, {
            "./stream/GenericWorker": 28,
            "./utils": 32,
            pako: 38
          }],
          8: [function (t, e, r) {

            function A(t, e) {
              var r,
                  i = "";

              for (r = 0; r < e; r++) i += String.fromCharCode(255 & t), t >>>= 8;

              return i;
            }

            function i(t, e, r, i, n, s) {
              var a,
                  o,
                  h = t.file,
                  u = t.compression,
                  l = s !== O.utf8encode,
                  f = I.transformTo("string", s(h.name)),
                  d = I.transformTo("string", O.utf8encode(h.name)),
                  c = h.comment,
                  p = I.transformTo("string", s(c)),
                  m = I.transformTo("string", O.utf8encode(c)),
                  _ = d.length !== h.name.length,
                  g = m.length !== c.length,
                  b = "",
                  v = "",
                  y = "",
                  w = h.dir,
                  k = h.date,
                  x = {
                crc32: 0,
                compressedSize: 0,
                uncompressedSize: 0
              };

              e && !r || (x.crc32 = t.crc32, x.compressedSize = t.compressedSize, x.uncompressedSize = t.uncompressedSize);
              var S = 0;
              e && (S |= 8), l || !_ && !g || (S |= 2048);
              var z = 0,
                  C = 0;
              w && (z |= 16), "UNIX" === n ? (C = 798, z |= function (t, e) {
                var r = t;
                return t || (r = e ? 16893 : 33204), (65535 & r) << 16;
              }(h.unixPermissions, w)) : (C = 20, z |= function (t) {
                return 63 & (t || 0);
              }(h.dosPermissions)), a = k.getUTCHours(), a <<= 6, a |= k.getUTCMinutes(), a <<= 5, a |= k.getUTCSeconds() / 2, o = k.getUTCFullYear() - 1980, o <<= 4, o |= k.getUTCMonth() + 1, o <<= 5, o |= k.getUTCDate(), _ && (v = A(1, 1) + A(B(f), 4) + d, b += "up" + A(v.length, 2) + v), g && (y = A(1, 1) + A(B(p), 4) + m, b += "uc" + A(y.length, 2) + y);
              var E = "";
              return E += "\n\0", E += A(S, 2), E += u.magic, E += A(a, 2), E += A(o, 2), E += A(x.crc32, 4), E += A(x.compressedSize, 4), E += A(x.uncompressedSize, 4), E += A(f.length, 2), E += A(b.length, 2), {
                fileRecord: R.LOCAL_FILE_HEADER + E + f + b,
                dirRecord: R.CENTRAL_FILE_HEADER + A(C, 2) + E + A(p.length, 2) + "\0\0\0\0" + A(z, 4) + A(i, 4) + f + b + p
              };
            }

            var I = t("../utils"),
                n = t("../stream/GenericWorker"),
                O = t("../utf8"),
                B = t("../crc32"),
                R = t("../signature");

            function s(t, e, r, i) {
              n.call(this, "ZipFileWorker"), this.bytesWritten = 0, this.zipComment = e, this.zipPlatform = r, this.encodeFileName = i, this.streamFiles = t, this.accumulate = !1, this.contentBuffer = [], this.dirRecords = [], this.currentSourceOffset = 0, this.entriesCount = 0, this.currentFile = null, this._sources = [];
            }

            I.inherits(s, n), s.prototype.push = function (t) {
              var e = t.meta.percent || 0,
                  r = this.entriesCount,
                  i = this._sources.length;
              this.accumulate ? this.contentBuffer.push(t) : (this.bytesWritten += t.data.length, n.prototype.push.call(this, {
                data: t.data,
                meta: {
                  currentFile: this.currentFile,
                  percent: r ? (e + 100 * (r - i - 1)) / r : 100
                }
              }));
            }, s.prototype.openedSource = function (t) {
              this.currentSourceOffset = this.bytesWritten, this.currentFile = t.file.name;
              var e = this.streamFiles && !t.file.dir;

              if (e) {
                var r = i(t, e, !1, this.currentSourceOffset, this.zipPlatform, this.encodeFileName);
                this.push({
                  data: r.fileRecord,
                  meta: {
                    percent: 0
                  }
                });
              } else this.accumulate = !0;
            }, s.prototype.closedSource = function (t) {
              this.accumulate = !1;
              var e = this.streamFiles && !t.file.dir,
                  r = i(t, e, !0, this.currentSourceOffset, this.zipPlatform, this.encodeFileName);
              if (this.dirRecords.push(r.dirRecord), e) this.push({
                data: function (t) {
                  return R.DATA_DESCRIPTOR + A(t.crc32, 4) + A(t.compressedSize, 4) + A(t.uncompressedSize, 4);
                }(t),
                meta: {
                  percent: 100
                }
              });else for (this.push({
                data: r.fileRecord,
                meta: {
                  percent: 0
                }
              }); this.contentBuffer.length;) this.push(this.contentBuffer.shift());
              this.currentFile = null;
            }, s.prototype.flush = function () {
              for (var t = this.bytesWritten, e = 0; e < this.dirRecords.length; e++) this.push({
                data: this.dirRecords[e],
                meta: {
                  percent: 100
                }
              });

              var r = this.bytesWritten - t,
                  i = function (t, e, r, i, n) {
                var s = I.transformTo("string", n(i));
                return R.CENTRAL_DIRECTORY_END + "\0\0\0\0" + A(t, 2) + A(t, 2) + A(e, 4) + A(r, 4) + A(s.length, 2) + s;
              }(this.dirRecords.length, r, t, this.zipComment, this.encodeFileName);

              this.push({
                data: i,
                meta: {
                  percent: 100
                }
              });
            }, s.prototype.prepareNextSource = function () {
              this.previous = this._sources.shift(), this.openedSource(this.previous.streamInfo), this.isPaused ? this.previous.pause() : this.previous.resume();
            }, s.prototype.registerPrevious = function (t) {
              this._sources.push(t);

              var e = this;
              return t.on("data", function (t) {
                e.processChunk(t);
              }), t.on("end", function () {
                e.closedSource(e.previous.streamInfo), e._sources.length ? e.prepareNextSource() : e.end();
              }), t.on("error", function (t) {
                e.error(t);
              }), this;
            }, s.prototype.resume = function () {
              return !!n.prototype.resume.call(this) && (!this.previous && this._sources.length ? (this.prepareNextSource(), !0) : this.previous || this._sources.length || this.generatedError ? void 0 : (this.end(), !0));
            }, s.prototype.error = function (t) {
              var e = this._sources;
              if (!n.prototype.error.call(this, t)) return !1;

              for (var r = 0; r < e.length; r++) try {
                e[r].error(t);
              } catch (t) {}

              return !0;
            }, s.prototype.lock = function () {
              n.prototype.lock.call(this);

              for (var t = this._sources, e = 0; e < t.length; e++) t[e].lock();
            }, e.exports = s;
          }, {
            "../crc32": 4,
            "../signature": 23,
            "../stream/GenericWorker": 28,
            "../utf8": 31,
            "../utils": 32
          }],
          9: [function (t, e, r) {

            var u = t("../compressions"),
                i = t("./ZipFileWorker");

            r.generateWorker = function (t, a, e) {
              var o = new i(a.streamFiles, e, a.platform, a.encodeFileName),
                  h = 0;

              try {
                t.forEach(function (t, e) {
                  h++;

                  var r = function (t, e) {
                    var r = t || e,
                        i = u[r];
                    if (!i) throw new Error(r + " is not a valid compression method !");
                    return i;
                  }(e.options.compression, a.compression),
                      i = e.options.compressionOptions || a.compressionOptions || {},
                      n = e.dir,
                      s = e.date;

                  e._compressWorker(r, i).withStreamInfo("file", {
                    name: t,
                    dir: n,
                    date: s,
                    comment: e.comment || "",
                    unixPermissions: e.unixPermissions,
                    dosPermissions: e.dosPermissions
                  }).pipe(o);
                }), o.entriesCount = h;
              } catch (t) {
                o.error(t);
              }

              return o;
            };
          }, {
            "../compressions": 3,
            "./ZipFileWorker": 8
          }],
          10: [function (t, e, r) {

            function i() {
              if (!(this instanceof i)) return new i();
              if (arguments.length) throw new Error("The constructor with parameters has been removed in JSZip 3.0, please check the upgrade guide.");
              this.files = Object.create(null), this.comment = null, this.root = "", this.clone = function () {
                var t = new i();

                for (var e in this) "function" != typeof this[e] && (t[e] = this[e]);

                return t;
              };
            }

            (i.prototype = t("./object")).loadAsync = t("./load"), i.support = t("./support"), i.defaults = t("./defaults"), i.version = "3.7.1", i.loadAsync = function (t, e) {
              return new i().loadAsync(t, e);
            }, i.external = t("./external"), e.exports = i;
          }, {
            "./defaults": 5,
            "./external": 6,
            "./load": 11,
            "./object": 15,
            "./support": 30
          }],
          11: [function (t, e, r) {

            var i = t("./utils"),
                n = t("./external"),
                o = t("./utf8"),
                h = t("./zipEntries"),
                s = t("./stream/Crc32Probe"),
                u = t("./nodejsUtils");

            function l(i) {
              return new n.Promise(function (t, e) {
                var r = i.decompressed.getContentWorker().pipe(new s());
                r.on("error", function (t) {
                  e(t);
                }).on("end", function () {
                  r.streamInfo.crc32 !== i.decompressed.crc32 ? e(new Error("Corrupted zip : CRC32 mismatch")) : t();
                }).resume();
              });
            }

            e.exports = function (t, s) {
              var a = this;
              return s = i.extend(s || {}, {
                base64: !1,
                checkCRC32: !1,
                optimizedBinaryString: !1,
                createFolders: !1,
                decodeFileName: o.utf8decode
              }), u.isNode && u.isStream(t) ? n.Promise.reject(new Error("JSZip can't accept a stream when loading a zip file.")) : i.prepareContent("the loaded zip file", t, !0, s.optimizedBinaryString, s.base64).then(function (t) {
                var e = new h(s);
                return e.load(t), e;
              }).then(function (t) {
                var e = [n.Promise.resolve(t)],
                    r = t.files;
                if (s.checkCRC32) for (var i = 0; i < r.length; i++) e.push(l(r[i]));
                return n.Promise.all(e);
              }).then(function (t) {
                for (var e = t.shift(), r = e.files, i = 0; i < r.length; i++) {
                  var n = r[i];
                  a.file(n.fileNameStr, n.decompressed, {
                    binary: !0,
                    optimizedBinaryString: !0,
                    date: n.date,
                    dir: n.dir,
                    comment: n.fileCommentStr.length ? n.fileCommentStr : null,
                    unixPermissions: n.unixPermissions,
                    dosPermissions: n.dosPermissions,
                    createFolders: s.createFolders
                  });
                }

                return e.zipComment.length && (a.comment = e.zipComment), a;
              });
            };
          }, {
            "./external": 6,
            "./nodejsUtils": 14,
            "./stream/Crc32Probe": 25,
            "./utf8": 31,
            "./utils": 32,
            "./zipEntries": 33
          }],
          12: [function (t, e, r) {

            var i = t("../utils"),
                n = t("../stream/GenericWorker");

            function s(t, e) {
              n.call(this, "Nodejs stream input adapter for " + t), this._upstreamEnded = !1, this._bindStream(e);
            }

            i.inherits(s, n), s.prototype._bindStream = function (t) {
              var e = this;
              (this._stream = t).pause(), t.on("data", function (t) {
                e.push({
                  data: t,
                  meta: {
                    percent: 0
                  }
                });
              }).on("error", function (t) {
                e.isPaused ? this.generatedError = t : e.error(t);
              }).on("end", function () {
                e.isPaused ? e._upstreamEnded = !0 : e.end();
              });
            }, s.prototype.pause = function () {
              return !!n.prototype.pause.call(this) && (this._stream.pause(), !0);
            }, s.prototype.resume = function () {
              return !!n.prototype.resume.call(this) && (this._upstreamEnded ? this.end() : this._stream.resume(), !0);
            }, e.exports = s;
          }, {
            "../stream/GenericWorker": 28,
            "../utils": 32
          }],
          13: [function (t, e, r) {

            var n = t("readable-stream").Readable;

            function i(t, e, r) {
              n.call(this, e), this._helper = t;
              var i = this;
              t.on("data", function (t, e) {
                i.push(t) || i._helper.pause(), r && r(e);
              }).on("error", function (t) {
                i.emit("error", t);
              }).on("end", function () {
                i.push(null);
              });
            }

            t("../utils").inherits(i, n), i.prototype._read = function () {
              this._helper.resume();
            }, e.exports = i;
          }, {
            "../utils": 32,
            "readable-stream": 16
          }],
          14: [function (t, e, r) {

            e.exports = {
              isNode: "undefined" != typeof Buffer,
              newBufferFrom: function (t, e) {
                if (Buffer.from && Buffer.from !== Uint8Array.from) return Buffer.from(t, e);
                if ("number" == typeof t) throw new Error('The "data" argument must not be a number');
                return new Buffer(t, e);
              },
              allocBuffer: function (t) {
                if (Buffer.alloc) return Buffer.alloc(t);
                var e = new Buffer(t);
                return e.fill(0), e;
              },
              isBuffer: function (t) {
                return Buffer.isBuffer(t);
              },
              isStream: function (t) {
                return t && "function" == typeof t.on && "function" == typeof t.pause && "function" == typeof t.resume;
              }
            };
          }, {}],
          15: [function (t, e, r) {

            function s(t, e, r) {
              var i,
                  n = u.getTypeOf(e),
                  s = u.extend(r || {}, f);
              s.date = s.date || new Date(), null !== s.compression && (s.compression = s.compression.toUpperCase()), "string" == typeof s.unixPermissions && (s.unixPermissions = parseInt(s.unixPermissions, 8)), s.unixPermissions && 16384 & s.unixPermissions && (s.dir = !0), s.dosPermissions && 16 & s.dosPermissions && (s.dir = !0), s.dir && (t = g(t)), s.createFolders && (i = _(t)) && b.call(this, i, !0);
              var a = "string" === n && !1 === s.binary && !1 === s.base64;
              r && void 0 !== r.binary || (s.binary = !a), (e instanceof d && 0 === e.uncompressedSize || s.dir || !e || 0 === e.length) && (s.base64 = !1, s.binary = !0, e = "", s.compression = "STORE", n = "string");
              var o = null;
              o = e instanceof d || e instanceof l ? e : p.isNode && p.isStream(e) ? new m(t, e) : u.prepareContent(t, e, s.binary, s.optimizedBinaryString, s.base64);
              var h = new c(t, o, s);
              this.files[t] = h;
            }

            var n = t("./utf8"),
                u = t("./utils"),
                l = t("./stream/GenericWorker"),
                a = t("./stream/StreamHelper"),
                f = t("./defaults"),
                d = t("./compressedObject"),
                c = t("./zipObject"),
                o = t("./generate"),
                p = t("./nodejsUtils"),
                m = t("./nodejs/NodejsStreamInputAdapter"),
                _ = function (t) {
              "/" === t.slice(-1) && (t = t.substring(0, t.length - 1));
              var e = t.lastIndexOf("/");
              return 0 < e ? t.substring(0, e) : "";
            },
                g = function (t) {
              return "/" !== t.slice(-1) && (t += "/"), t;
            },
                b = function (t, e) {
              return e = void 0 !== e ? e : f.createFolders, t = g(t), this.files[t] || s.call(this, t, null, {
                dir: !0,
                createFolders: e
              }), this.files[t];
            };

            function h(t) {
              return "[object RegExp]" === Object.prototype.toString.call(t);
            }

            var i = {
              load: function () {
                throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.");
              },
              forEach: function (t) {
                var e, r, i;

                for (e in this.files) i = this.files[e], (r = e.slice(this.root.length, e.length)) && e.slice(0, this.root.length) === this.root && t(r, i);
              },
              filter: function (r) {
                var i = [];
                return this.forEach(function (t, e) {
                  r(t, e) && i.push(e);
                }), i;
              },
              file: function (t, e, r) {
                if (1 !== arguments.length) return t = this.root + t, s.call(this, t, e, r), this;

                if (h(t)) {
                  var i = t;
                  return this.filter(function (t, e) {
                    return !e.dir && i.test(t);
                  });
                }

                var n = this.files[this.root + t];
                return n && !n.dir ? n : null;
              },
              folder: function (r) {
                if (!r) return this;
                if (h(r)) return this.filter(function (t, e) {
                  return e.dir && r.test(t);
                });
                var t = this.root + r,
                    e = b.call(this, t),
                    i = this.clone();
                return i.root = e.name, i;
              },
              remove: function (r) {
                r = this.root + r;
                var t = this.files[r];
                if (t || ("/" !== r.slice(-1) && (r += "/"), t = this.files[r]), t && !t.dir) delete this.files[r];else for (var e = this.filter(function (t, e) {
                  return e.name.slice(0, r.length) === r;
                }), i = 0; i < e.length; i++) delete this.files[e[i].name];
                return this;
              },
              generate: function (t) {
                throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.");
              },
              generateInternalStream: function (t) {
                var e,
                    r = {};

                try {
                  if ((r = u.extend(t || {}, {
                    streamFiles: !1,
                    compression: "STORE",
                    compressionOptions: null,
                    type: "",
                    platform: "DOS",
                    comment: null,
                    mimeType: "application/zip",
                    encodeFileName: n.utf8encode
                  })).type = r.type.toLowerCase(), r.compression = r.compression.toUpperCase(), "binarystring" === r.type && (r.type = "string"), !r.type) throw new Error("No output type specified.");
                  u.checkSupport(r.type), "darwin" !== r.platform && "freebsd" !== r.platform && "linux" !== r.platform && "sunos" !== r.platform || (r.platform = "UNIX"), "win32" === r.platform && (r.platform = "DOS");
                  var i = r.comment || this.comment || "";
                  e = o.generateWorker(this, r, i);
                } catch (t) {
                  (e = new l("error")).error(t);
                }

                return new a(e, r.type || "string", r.mimeType);
              },
              generateAsync: function (t, e) {
                return this.generateInternalStream(t).accumulate(e);
              },
              generateNodeStream: function (t, e) {
                return (t = t || {}).type || (t.type = "nodebuffer"), this.generateInternalStream(t).toNodejsStream(e);
              }
            };
            e.exports = i;
          }, {
            "./compressedObject": 2,
            "./defaults": 5,
            "./generate": 9,
            "./nodejs/NodejsStreamInputAdapter": 12,
            "./nodejsUtils": 14,
            "./stream/GenericWorker": 28,
            "./stream/StreamHelper": 29,
            "./utf8": 31,
            "./utils": 32,
            "./zipObject": 35
          }],
          16: [function (t, e, r) {
            e.exports = t("stream");
          }, {
            stream: void 0
          }],
          17: [function (t, e, r) {

            var i = t("./DataReader");

            function n(t) {
              i.call(this, t);

              for (var e = 0; e < this.data.length; e++) t[e] = 255 & t[e];
            }

            t("../utils").inherits(n, i), n.prototype.byteAt = function (t) {
              return this.data[this.zero + t];
            }, n.prototype.lastIndexOfSignature = function (t) {
              for (var e = t.charCodeAt(0), r = t.charCodeAt(1), i = t.charCodeAt(2), n = t.charCodeAt(3), s = this.length - 4; 0 <= s; --s) if (this.data[s] === e && this.data[s + 1] === r && this.data[s + 2] === i && this.data[s + 3] === n) return s - this.zero;

              return -1;
            }, n.prototype.readAndCheckSignature = function (t) {
              var e = t.charCodeAt(0),
                  r = t.charCodeAt(1),
                  i = t.charCodeAt(2),
                  n = t.charCodeAt(3),
                  s = this.readData(4);
              return e === s[0] && r === s[1] && i === s[2] && n === s[3];
            }, n.prototype.readData = function (t) {
              if (this.checkOffset(t), 0 === t) return [];
              var e = this.data.slice(this.zero + this.index, this.zero + this.index + t);
              return this.index += t, e;
            }, e.exports = n;
          }, {
            "../utils": 32,
            "./DataReader": 18
          }],
          18: [function (t, e, r) {

            var i = t("../utils");

            function n(t) {
              this.data = t, this.length = t.length, this.index = 0, this.zero = 0;
            }

            n.prototype = {
              checkOffset: function (t) {
                this.checkIndex(this.index + t);
              },
              checkIndex: function (t) {
                if (this.length < this.zero + t || t < 0) throw new Error("End of data reached (data length = " + this.length + ", asked index = " + t + "). Corrupted zip ?");
              },
              setIndex: function (t) {
                this.checkIndex(t), this.index = t;
              },
              skip: function (t) {
                this.setIndex(this.index + t);
              },
              byteAt: function (t) {},
              readInt: function (t) {
                var e,
                    r = 0;

                for (this.checkOffset(t), e = this.index + t - 1; e >= this.index; e--) r = (r << 8) + this.byteAt(e);

                return this.index += t, r;
              },
              readString: function (t) {
                return i.transformTo("string", this.readData(t));
              },
              readData: function (t) {},
              lastIndexOfSignature: function (t) {},
              readAndCheckSignature: function (t) {},
              readDate: function () {
                var t = this.readInt(4);
                return new Date(Date.UTC(1980 + (t >> 25 & 127), (t >> 21 & 15) - 1, t >> 16 & 31, t >> 11 & 31, t >> 5 & 63, (31 & t) << 1));
              }
            }, e.exports = n;
          }, {
            "../utils": 32
          }],
          19: [function (t, e, r) {

            var i = t("./Uint8ArrayReader");

            function n(t) {
              i.call(this, t);
            }

            t("../utils").inherits(n, i), n.prototype.readData = function (t) {
              this.checkOffset(t);
              var e = this.data.slice(this.zero + this.index, this.zero + this.index + t);
              return this.index += t, e;
            }, e.exports = n;
          }, {
            "../utils": 32,
            "./Uint8ArrayReader": 21
          }],
          20: [function (t, e, r) {

            var i = t("./DataReader");

            function n(t) {
              i.call(this, t);
            }

            t("../utils").inherits(n, i), n.prototype.byteAt = function (t) {
              return this.data.charCodeAt(this.zero + t);
            }, n.prototype.lastIndexOfSignature = function (t) {
              return this.data.lastIndexOf(t) - this.zero;
            }, n.prototype.readAndCheckSignature = function (t) {
              return t === this.readData(4);
            }, n.prototype.readData = function (t) {
              this.checkOffset(t);
              var e = this.data.slice(this.zero + this.index, this.zero + this.index + t);
              return this.index += t, e;
            }, e.exports = n;
          }, {
            "../utils": 32,
            "./DataReader": 18
          }],
          21: [function (t, e, r) {

            var i = t("./ArrayReader");

            function n(t) {
              i.call(this, t);
            }

            t("../utils").inherits(n, i), n.prototype.readData = function (t) {
              if (this.checkOffset(t), 0 === t) return new Uint8Array(0);
              var e = this.data.subarray(this.zero + this.index, this.zero + this.index + t);
              return this.index += t, e;
            }, e.exports = n;
          }, {
            "../utils": 32,
            "./ArrayReader": 17
          }],
          22: [function (t, e, r) {

            var i = t("../utils"),
                n = t("../support"),
                s = t("./ArrayReader"),
                a = t("./StringReader"),
                o = t("./NodeBufferReader"),
                h = t("./Uint8ArrayReader");

            e.exports = function (t) {
              var e = i.getTypeOf(t);
              return i.checkSupport(e), "string" !== e || n.uint8array ? "nodebuffer" === e ? new o(t) : n.uint8array ? new h(i.transformTo("uint8array", t)) : new s(i.transformTo("array", t)) : new a(t);
            };
          }, {
            "../support": 30,
            "../utils": 32,
            "./ArrayReader": 17,
            "./NodeBufferReader": 19,
            "./StringReader": 20,
            "./Uint8ArrayReader": 21
          }],
          23: [function (t, e, r) {

            r.LOCAL_FILE_HEADER = "PK", r.CENTRAL_FILE_HEADER = "PK", r.CENTRAL_DIRECTORY_END = "PK", r.ZIP64_CENTRAL_DIRECTORY_LOCATOR = "PK", r.ZIP64_CENTRAL_DIRECTORY_END = "PK", r.DATA_DESCRIPTOR = "PK\b";
          }, {}],
          24: [function (t, e, r) {

            var i = t("./GenericWorker"),
                n = t("../utils");

            function s(t) {
              i.call(this, "ConvertWorker to " + t), this.destType = t;
            }

            n.inherits(s, i), s.prototype.processChunk = function (t) {
              this.push({
                data: n.transformTo(this.destType, t.data),
                meta: t.meta
              });
            }, e.exports = s;
          }, {
            "../utils": 32,
            "./GenericWorker": 28
          }],
          25: [function (t, e, r) {

            var i = t("./GenericWorker"),
                n = t("../crc32");

            function s() {
              i.call(this, "Crc32Probe"), this.withStreamInfo("crc32", 0);
            }

            t("../utils").inherits(s, i), s.prototype.processChunk = function (t) {
              this.streamInfo.crc32 = n(t.data, this.streamInfo.crc32 || 0), this.push(t);
            }, e.exports = s;
          }, {
            "../crc32": 4,
            "../utils": 32,
            "./GenericWorker": 28
          }],
          26: [function (t, e, r) {

            var i = t("../utils"),
                n = t("./GenericWorker");

            function s(t) {
              n.call(this, "DataLengthProbe for " + t), this.propName = t, this.withStreamInfo(t, 0);
            }

            i.inherits(s, n), s.prototype.processChunk = function (t) {
              if (t) {
                var e = this.streamInfo[this.propName] || 0;
                this.streamInfo[this.propName] = e + t.data.length;
              }

              n.prototype.processChunk.call(this, t);
            }, e.exports = s;
          }, {
            "../utils": 32,
            "./GenericWorker": 28
          }],
          27: [function (t, e, r) {

            var i = t("../utils"),
                n = t("./GenericWorker");

            function s(t) {
              n.call(this, "DataWorker");
              var e = this;
              this.dataIsReady = !1, this.index = 0, this.max = 0, this.data = null, this.type = "", this._tickScheduled = !1, t.then(function (t) {
                e.dataIsReady = !0, e.data = t, e.max = t && t.length || 0, e.type = i.getTypeOf(t), e.isPaused || e._tickAndRepeat();
              }, function (t) {
                e.error(t);
              });
            }

            i.inherits(s, n), s.prototype.cleanUp = function () {
              n.prototype.cleanUp.call(this), this.data = null;
            }, s.prototype.resume = function () {
              return !!n.prototype.resume.call(this) && (!this._tickScheduled && this.dataIsReady && (this._tickScheduled = !0, i.delay(this._tickAndRepeat, [], this)), !0);
            }, s.prototype._tickAndRepeat = function () {
              this._tickScheduled = !1, this.isPaused || this.isFinished || (this._tick(), this.isFinished || (i.delay(this._tickAndRepeat, [], this), this._tickScheduled = !0));
            }, s.prototype._tick = function () {
              if (this.isPaused || this.isFinished) return !1;
              var t = null,
                  e = Math.min(this.max, this.index + 16384);
              if (this.index >= this.max) return this.end();

              switch (this.type) {
                case "string":
                  t = this.data.substring(this.index, e);
                  break;

                case "uint8array":
                  t = this.data.subarray(this.index, e);
                  break;

                case "array":
                case "nodebuffer":
                  t = this.data.slice(this.index, e);
              }

              return this.index = e, this.push({
                data: t,
                meta: {
                  percent: this.max ? this.index / this.max * 100 : 0
                }
              });
            }, e.exports = s;
          }, {
            "../utils": 32,
            "./GenericWorker": 28
          }],
          28: [function (t, e, r) {

            function i(t) {
              this.name = t || "default", this.streamInfo = {}, this.generatedError = null, this.extraStreamInfo = {}, this.isPaused = !0, this.isFinished = !1, this.isLocked = !1, this._listeners = {
                data: [],
                end: [],
                error: []
              }, this.previous = null;
            }

            i.prototype = {
              push: function (t) {
                this.emit("data", t);
              },
              end: function () {
                if (this.isFinished) return !1;
                this.flush();

                try {
                  this.emit("end"), this.cleanUp(), this.isFinished = !0;
                } catch (t) {
                  this.emit("error", t);
                }

                return !0;
              },
              error: function (t) {
                return !this.isFinished && (this.isPaused ? this.generatedError = t : (this.isFinished = !0, this.emit("error", t), this.previous && this.previous.error(t), this.cleanUp()), !0);
              },
              on: function (t, e) {
                return this._listeners[t].push(e), this;
              },
              cleanUp: function () {
                this.streamInfo = this.generatedError = this.extraStreamInfo = null, this._listeners = [];
              },
              emit: function (t, e) {
                if (this._listeners[t]) for (var r = 0; r < this._listeners[t].length; r++) this._listeners[t][r].call(this, e);
              },
              pipe: function (t) {
                return t.registerPrevious(this);
              },
              registerPrevious: function (t) {
                if (this.isLocked) throw new Error("The stream '" + this + "' has already been used.");
                this.streamInfo = t.streamInfo, this.mergeStreamInfo(), this.previous = t;
                var e = this;
                return t.on("data", function (t) {
                  e.processChunk(t);
                }), t.on("end", function () {
                  e.end();
                }), t.on("error", function (t) {
                  e.error(t);
                }), this;
              },
              pause: function () {
                return !this.isPaused && !this.isFinished && (this.isPaused = !0, this.previous && this.previous.pause(), !0);
              },
              resume: function () {
                if (!this.isPaused || this.isFinished) return !1;
                var t = this.isPaused = !1;
                return this.generatedError && (this.error(this.generatedError), t = !0), this.previous && this.previous.resume(), !t;
              },
              flush: function () {},
              processChunk: function (t) {
                this.push(t);
              },
              withStreamInfo: function (t, e) {
                return this.extraStreamInfo[t] = e, this.mergeStreamInfo(), this;
              },
              mergeStreamInfo: function () {
                for (var t in this.extraStreamInfo) this.extraStreamInfo.hasOwnProperty(t) && (this.streamInfo[t] = this.extraStreamInfo[t]);
              },
              lock: function () {
                if (this.isLocked) throw new Error("The stream '" + this + "' has already been used.");
                this.isLocked = !0, this.previous && this.previous.lock();
              },
              toString: function () {
                var t = "Worker " + this.name;
                return this.previous ? this.previous + " -> " + t : t;
              }
            }, e.exports = i;
          }, {}],
          29: [function (t, e, r) {

            var h = t("../utils"),
                n = t("./ConvertWorker"),
                s = t("./GenericWorker"),
                u = t("../base64"),
                i = t("../support"),
                a = t("../external"),
                o = null;
            if (i.nodestream) try {
              o = t("../nodejs/NodejsStreamOutputAdapter");
            } catch (t) {}

            function l(t, o) {
              return new a.Promise(function (e, r) {
                var i = [],
                    n = t._internalType,
                    s = t._outputType,
                    a = t._mimeType;
                t.on("data", function (t, e) {
                  i.push(t), o && o(e);
                }).on("error", function (t) {
                  i = [], r(t);
                }).on("end", function () {
                  try {
                    var t = function (t, e, r) {
                      switch (t) {
                        case "blob":
                          return h.newBlob(h.transformTo("arraybuffer", e), r);

                        case "base64":
                          return u.encode(e);

                        default:
                          return h.transformTo(t, e);
                      }
                    }(s, function (t, e) {
                      var r,
                          i = 0,
                          n = null,
                          s = 0;

                      for (r = 0; r < e.length; r++) s += e[r].length;

                      switch (t) {
                        case "string":
                          return e.join("");

                        case "array":
                          return Array.prototype.concat.apply([], e);

                        case "uint8array":
                          for (n = new Uint8Array(s), r = 0; r < e.length; r++) n.set(e[r], i), i += e[r].length;

                          return n;

                        case "nodebuffer":
                          return Buffer.concat(e);

                        default:
                          throw new Error("concat : unsupported type '" + t + "'");
                      }
                    }(n, i), a);

                    e(t);
                  } catch (t) {
                    r(t);
                  }

                  i = [];
                }).resume();
              });
            }

            function f(t, e, r) {
              var i = e;

              switch (e) {
                case "blob":
                case "arraybuffer":
                  i = "uint8array";
                  break;

                case "base64":
                  i = "string";
              }

              try {
                this._internalType = i, this._outputType = e, this._mimeType = r, h.checkSupport(i), this._worker = t.pipe(new n(i)), t.lock();
              } catch (t) {
                this._worker = new s("error"), this._worker.error(t);
              }
            }

            f.prototype = {
              accumulate: function (t) {
                return l(this, t);
              },
              on: function (t, e) {
                var r = this;
                return "data" === t ? this._worker.on(t, function (t) {
                  e.call(r, t.data, t.meta);
                }) : this._worker.on(t, function () {
                  h.delay(e, arguments, r);
                }), this;
              },
              resume: function () {
                return h.delay(this._worker.resume, [], this._worker), this;
              },
              pause: function () {
                return this._worker.pause(), this;
              },
              toNodejsStream: function (t) {
                if (h.checkSupport("nodestream"), "nodebuffer" !== this._outputType) throw new Error(this._outputType + " is not supported by this method");
                return new o(this, {
                  objectMode: "nodebuffer" !== this._outputType
                }, t);
              }
            }, e.exports = f;
          }, {
            "../base64": 1,
            "../external": 6,
            "../nodejs/NodejsStreamOutputAdapter": 13,
            "../support": 30,
            "../utils": 32,
            "./ConvertWorker": 24,
            "./GenericWorker": 28
          }],
          30: [function (t, e, r) {

            if (r.base64 = !0, r.array = !0, r.string = !0, r.arraybuffer = "undefined" != typeof ArrayBuffer && "undefined" != typeof Uint8Array, r.nodebuffer = "undefined" != typeof Buffer, r.uint8array = "undefined" != typeof Uint8Array, "undefined" == typeof ArrayBuffer) r.blob = !1;else {
              var i = new ArrayBuffer(0);

              try {
                r.blob = 0 === new Blob([i], {
                  type: "application/zip"
                }).size;
              } catch (t) {
                try {
                  var n = new (self.BlobBuilder || self.WebKitBlobBuilder || self.MozBlobBuilder || self.MSBlobBuilder)();
                  n.append(i), r.blob = 0 === n.getBlob("application/zip").size;
                } catch (t) {
                  r.blob = !1;
                }
              }
            }

            try {
              r.nodestream = !!t("readable-stream").Readable;
            } catch (t) {
              r.nodestream = !1;
            }
          }, {
            "readable-stream": 16
          }],
          31: [function (t, e, s) {

            for (var o = t("./utils"), h = t("./support"), r = t("./nodejsUtils"), i = t("./stream/GenericWorker"), u = new Array(256), n = 0; n < 256; n++) u[n] = 252 <= n ? 6 : 248 <= n ? 5 : 240 <= n ? 4 : 224 <= n ? 3 : 192 <= n ? 2 : 1;

            u[254] = u[254] = 1;

            function a() {
              i.call(this, "utf-8 decode"), this.leftOver = null;
            }

            function l() {
              i.call(this, "utf-8 encode");
            }

            s.utf8encode = function (t) {
              return h.nodebuffer ? r.newBufferFrom(t, "utf-8") : function (t) {
                var e,
                    r,
                    i,
                    n,
                    s,
                    a = t.length,
                    o = 0;

                for (n = 0; n < a; n++) 55296 == (64512 & (r = t.charCodeAt(n))) && n + 1 < a && 56320 == (64512 & (i = t.charCodeAt(n + 1))) && (r = 65536 + (r - 55296 << 10) + (i - 56320), n++), o += r < 128 ? 1 : r < 2048 ? 2 : r < 65536 ? 3 : 4;

                for (e = h.uint8array ? new Uint8Array(o) : new Array(o), n = s = 0; s < o; n++) 55296 == (64512 & (r = t.charCodeAt(n))) && n + 1 < a && 56320 == (64512 & (i = t.charCodeAt(n + 1))) && (r = 65536 + (r - 55296 << 10) + (i - 56320), n++), r < 128 ? e[s++] = r : (r < 2048 ? e[s++] = 192 | r >>> 6 : (r < 65536 ? e[s++] = 224 | r >>> 12 : (e[s++] = 240 | r >>> 18, e[s++] = 128 | r >>> 12 & 63), e[s++] = 128 | r >>> 6 & 63), e[s++] = 128 | 63 & r);

                return e;
              }(t);
            }, s.utf8decode = function (t) {
              return h.nodebuffer ? o.transformTo("nodebuffer", t).toString("utf-8") : function (t) {
                var e,
                    r,
                    i,
                    n,
                    s = t.length,
                    a = new Array(2 * s);

                for (e = r = 0; e < s;) if ((i = t[e++]) < 128) a[r++] = i;else if (4 < (n = u[i])) a[r++] = 65533, e += n - 1;else {
                  for (i &= 2 === n ? 31 : 3 === n ? 15 : 7; 1 < n && e < s;) i = i << 6 | 63 & t[e++], n--;

                  1 < n ? a[r++] = 65533 : i < 65536 ? a[r++] = i : (i -= 65536, a[r++] = 55296 | i >> 10 & 1023, a[r++] = 56320 | 1023 & i);
                }

                return a.length !== r && (a.subarray ? a = a.subarray(0, r) : a.length = r), o.applyFromCharCode(a);
              }(t = o.transformTo(h.uint8array ? "uint8array" : "array", t));
            }, o.inherits(a, i), a.prototype.processChunk = function (t) {
              var e = o.transformTo(h.uint8array ? "uint8array" : "array", t.data);

              if (this.leftOver && this.leftOver.length) {
                if (h.uint8array) {
                  var r = e;
                  (e = new Uint8Array(r.length + this.leftOver.length)).set(this.leftOver, 0), e.set(r, this.leftOver.length);
                } else e = this.leftOver.concat(e);

                this.leftOver = null;
              }

              var i = function (t, e) {
                var r;

                for ((e = e || t.length) > t.length && (e = t.length), r = e - 1; 0 <= r && 128 == (192 & t[r]);) r--;

                return r < 0 ? e : 0 === r ? e : r + u[t[r]] > e ? r : e;
              }(e),
                  n = e;

              i !== e.length && (h.uint8array ? (n = e.subarray(0, i), this.leftOver = e.subarray(i, e.length)) : (n = e.slice(0, i), this.leftOver = e.slice(i, e.length))), this.push({
                data: s.utf8decode(n),
                meta: t.meta
              });
            }, a.prototype.flush = function () {
              this.leftOver && this.leftOver.length && (this.push({
                data: s.utf8decode(this.leftOver),
                meta: {}
              }), this.leftOver = null);
            }, s.Utf8DecodeWorker = a, o.inherits(l, i), l.prototype.processChunk = function (t) {
              this.push({
                data: s.utf8encode(t.data),
                meta: t.meta
              });
            }, s.Utf8EncodeWorker = l;
          }, {
            "./nodejsUtils": 14,
            "./stream/GenericWorker": 28,
            "./support": 30,
            "./utils": 32
          }],
          32: [function (t, e, a) {

            var o = t("./support"),
                h = t("./base64"),
                r = t("./nodejsUtils"),
                i = t("set-immediate-shim"),
                u = t("./external");

            function n(t) {
              return t;
            }

            function l(t, e) {
              for (var r = 0; r < t.length; ++r) e[r] = 255 & t.charCodeAt(r);

              return e;
            }

            a.newBlob = function (e, r) {
              a.checkSupport("blob");

              try {
                return new Blob([e], {
                  type: r
                });
              } catch (t) {
                try {
                  var i = new (self.BlobBuilder || self.WebKitBlobBuilder || self.MozBlobBuilder || self.MSBlobBuilder)();
                  return i.append(e), i.getBlob(r);
                } catch (t) {
                  throw new Error("Bug : can't construct the Blob.");
                }
              }
            };

            var s = {
              stringifyByChunk: function (t, e, r) {
                var i = [],
                    n = 0,
                    s = t.length;
                if (s <= r) return String.fromCharCode.apply(null, t);

                for (; n < s;) "array" === e || "nodebuffer" === e ? i.push(String.fromCharCode.apply(null, t.slice(n, Math.min(n + r, s)))) : i.push(String.fromCharCode.apply(null, t.subarray(n, Math.min(n + r, s)))), n += r;

                return i.join("");
              },
              stringifyByChar: function (t) {
                for (var e = "", r = 0; r < t.length; r++) e += String.fromCharCode(t[r]);

                return e;
              },
              applyCanBeUsed: {
                uint8array: function () {
                  try {
                    return o.uint8array && 1 === String.fromCharCode.apply(null, new Uint8Array(1)).length;
                  } catch (t) {
                    return !1;
                  }
                }(),
                nodebuffer: function () {
                  try {
                    return o.nodebuffer && 1 === String.fromCharCode.apply(null, r.allocBuffer(1)).length;
                  } catch (t) {
                    return !1;
                  }
                }()
              }
            };

            function f(t) {
              var e = 65536,
                  r = a.getTypeOf(t),
                  i = !0;
              if ("uint8array" === r ? i = s.applyCanBeUsed.uint8array : "nodebuffer" === r && (i = s.applyCanBeUsed.nodebuffer), i) for (; 1 < e;) try {
                return s.stringifyByChunk(t, r, e);
              } catch (t) {
                e = Math.floor(e / 2);
              }
              return s.stringifyByChar(t);
            }

            function d(t, e) {
              for (var r = 0; r < t.length; r++) e[r] = t[r];

              return e;
            }

            a.applyFromCharCode = f;
            var c = {};
            c.string = {
              string: n,
              array: function (t) {
                return l(t, new Array(t.length));
              },
              arraybuffer: function (t) {
                return c.string.uint8array(t).buffer;
              },
              uint8array: function (t) {
                return l(t, new Uint8Array(t.length));
              },
              nodebuffer: function (t) {
                return l(t, r.allocBuffer(t.length));
              }
            }, c.array = {
              string: f,
              array: n,
              arraybuffer: function (t) {
                return new Uint8Array(t).buffer;
              },
              uint8array: function (t) {
                return new Uint8Array(t);
              },
              nodebuffer: function (t) {
                return r.newBufferFrom(t);
              }
            }, c.arraybuffer = {
              string: function (t) {
                return f(new Uint8Array(t));
              },
              array: function (t) {
                return d(new Uint8Array(t), new Array(t.byteLength));
              },
              arraybuffer: n,
              uint8array: function (t) {
                return new Uint8Array(t);
              },
              nodebuffer: function (t) {
                return r.newBufferFrom(new Uint8Array(t));
              }
            }, c.uint8array = {
              string: f,
              array: function (t) {
                return d(t, new Array(t.length));
              },
              arraybuffer: function (t) {
                return t.buffer;
              },
              uint8array: n,
              nodebuffer: function (t) {
                return r.newBufferFrom(t);
              }
            }, c.nodebuffer = {
              string: f,
              array: function (t) {
                return d(t, new Array(t.length));
              },
              arraybuffer: function (t) {
                return c.nodebuffer.uint8array(t).buffer;
              },
              uint8array: function (t) {
                return d(t, new Uint8Array(t.length));
              },
              nodebuffer: n
            }, a.transformTo = function (t, e) {
              if (e = e || "", !t) return e;
              a.checkSupport(t);
              var r = a.getTypeOf(e);
              return c[r][t](e);
            }, a.getTypeOf = function (t) {
              return "string" == typeof t ? "string" : "[object Array]" === Object.prototype.toString.call(t) ? "array" : o.nodebuffer && r.isBuffer(t) ? "nodebuffer" : o.uint8array && t instanceof Uint8Array ? "uint8array" : o.arraybuffer && t instanceof ArrayBuffer ? "arraybuffer" : void 0;
            }, a.checkSupport = function (t) {
              if (!o[t.toLowerCase()]) throw new Error(t + " is not supported by this platform");
            }, a.MAX_VALUE_16BITS = 65535, a.MAX_VALUE_32BITS = -1, a.pretty = function (t) {
              var e,
                  r,
                  i = "";

              for (r = 0; r < (t || "").length; r++) i += "\\x" + ((e = t.charCodeAt(r)) < 16 ? "0" : "") + e.toString(16).toUpperCase();

              return i;
            }, a.delay = function (t, e, r) {
              i(function () {
                t.apply(r || null, e || []);
              });
            }, a.inherits = function (t, e) {
              function r() {}

              r.prototype = e.prototype, t.prototype = new r();
            }, a.extend = function () {
              var t,
                  e,
                  r = {};

              for (t = 0; t < arguments.length; t++) for (e in arguments[t]) arguments[t].hasOwnProperty(e) && void 0 === r[e] && (r[e] = arguments[t][e]);

              return r;
            }, a.prepareContent = function (r, t, i, n, s) {
              return u.Promise.resolve(t).then(function (i) {
                return o.blob && (i instanceof Blob || -1 !== ["[object File]", "[object Blob]"].indexOf(Object.prototype.toString.call(i))) && "undefined" != typeof FileReader ? new u.Promise(function (e, r) {
                  var t = new FileReader();
                  t.onload = function (t) {
                    e(t.target.result);
                  }, t.onerror = function (t) {
                    r(t.target.error);
                  }, t.readAsArrayBuffer(i);
                }) : i;
              }).then(function (t) {
                var e = a.getTypeOf(t);
                return e ? ("arraybuffer" === e ? t = a.transformTo("uint8array", t) : "string" === e && (s ? t = h.decode(t) : i && !0 !== n && (t = function (t) {
                  return l(t, o.uint8array ? new Uint8Array(t.length) : new Array(t.length));
                }(t))), t) : u.Promise.reject(new Error("Can't read the data of '" + r + "'. Is it in a supported JavaScript type (String, Blob, ArrayBuffer, etc) ?"));
              });
            };
          }, {
            "./base64": 1,
            "./external": 6,
            "./nodejsUtils": 14,
            "./support": 30,
            "set-immediate-shim": 54
          }],
          33: [function (t, e, r) {

            var i = t("./reader/readerFor"),
                n = t("./utils"),
                s = t("./signature"),
                a = t("./zipEntry"),
                o = (t("./utf8"), t("./support"));

            function h(t) {
              this.files = [], this.loadOptions = t;
            }

            h.prototype = {
              checkSignature: function (t) {
                if (!this.reader.readAndCheckSignature(t)) {
                  this.reader.index -= 4;
                  var e = this.reader.readString(4);
                  throw new Error("Corrupted zip or bug: unexpected signature (" + n.pretty(e) + ", expected " + n.pretty(t) + ")");
                }
              },
              isSignature: function (t, e) {
                var r = this.reader.index;
                this.reader.setIndex(t);
                var i = this.reader.readString(4) === e;
                return this.reader.setIndex(r), i;
              },
              readBlockEndOfCentral: function () {
                this.diskNumber = this.reader.readInt(2), this.diskWithCentralDirStart = this.reader.readInt(2), this.centralDirRecordsOnThisDisk = this.reader.readInt(2), this.centralDirRecords = this.reader.readInt(2), this.centralDirSize = this.reader.readInt(4), this.centralDirOffset = this.reader.readInt(4), this.zipCommentLength = this.reader.readInt(2);
                var t = this.reader.readData(this.zipCommentLength),
                    e = o.uint8array ? "uint8array" : "array",
                    r = n.transformTo(e, t);
                this.zipComment = this.loadOptions.decodeFileName(r);
              },
              readBlockZip64EndOfCentral: function () {
                this.zip64EndOfCentralSize = this.reader.readInt(8), this.reader.skip(4), this.diskNumber = this.reader.readInt(4), this.diskWithCentralDirStart = this.reader.readInt(4), this.centralDirRecordsOnThisDisk = this.reader.readInt(8), this.centralDirRecords = this.reader.readInt(8), this.centralDirSize = this.reader.readInt(8), this.centralDirOffset = this.reader.readInt(8), this.zip64ExtensibleData = {};

                for (var t, e, r, i = this.zip64EndOfCentralSize - 44; 0 < i;) t = this.reader.readInt(2), e = this.reader.readInt(4), r = this.reader.readData(e), this.zip64ExtensibleData[t] = {
                  id: t,
                  length: e,
                  value: r
                };
              },
              readBlockZip64EndOfCentralLocator: function () {
                if (this.diskWithZip64CentralDirStart = this.reader.readInt(4), this.relativeOffsetEndOfZip64CentralDir = this.reader.readInt(8), this.disksCount = this.reader.readInt(4), 1 < this.disksCount) throw new Error("Multi-volumes zip are not supported");
              },
              readLocalFiles: function () {
                var t, e;

                for (t = 0; t < this.files.length; t++) e = this.files[t], this.reader.setIndex(e.localHeaderOffset), this.checkSignature(s.LOCAL_FILE_HEADER), e.readLocalPart(this.reader), e.handleUTF8(), e.processAttributes();
              },
              readCentralDir: function () {
                var t;

                for (this.reader.setIndex(this.centralDirOffset); this.reader.readAndCheckSignature(s.CENTRAL_FILE_HEADER);) (t = new a({
                  zip64: this.zip64
                }, this.loadOptions)).readCentralPart(this.reader), this.files.push(t);

                if (this.centralDirRecords !== this.files.length && 0 !== this.centralDirRecords && 0 === this.files.length) throw new Error("Corrupted zip or bug: expected " + this.centralDirRecords + " records in central dir, got " + this.files.length);
              },
              readEndOfCentral: function () {
                var t = this.reader.lastIndexOfSignature(s.CENTRAL_DIRECTORY_END);
                if (t < 0) throw !this.isSignature(0, s.LOCAL_FILE_HEADER) ? new Error("Can't find end of central directory : is this a zip file ? If it is, see https://stuk.github.io/jszip/documentation/howto/read_zip.html") : new Error("Corrupted zip: can't find end of central directory");
                this.reader.setIndex(t);
                var e = t;

                if (this.checkSignature(s.CENTRAL_DIRECTORY_END), this.readBlockEndOfCentral(), this.diskNumber === n.MAX_VALUE_16BITS || this.diskWithCentralDirStart === n.MAX_VALUE_16BITS || this.centralDirRecordsOnThisDisk === n.MAX_VALUE_16BITS || this.centralDirRecords === n.MAX_VALUE_16BITS || this.centralDirSize === n.MAX_VALUE_32BITS || this.centralDirOffset === n.MAX_VALUE_32BITS) {
                  if (this.zip64 = !0, (t = this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR)) < 0) throw new Error("Corrupted zip: can't find the ZIP64 end of central directory locator");
                  if (this.reader.setIndex(t), this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR), this.readBlockZip64EndOfCentralLocator(), !this.isSignature(this.relativeOffsetEndOfZip64CentralDir, s.ZIP64_CENTRAL_DIRECTORY_END) && (this.relativeOffsetEndOfZip64CentralDir = this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_END), this.relativeOffsetEndOfZip64CentralDir < 0)) throw new Error("Corrupted zip: can't find the ZIP64 end of central directory");
                  this.reader.setIndex(this.relativeOffsetEndOfZip64CentralDir), this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_END), this.readBlockZip64EndOfCentral();
                }

                var r = this.centralDirOffset + this.centralDirSize;
                this.zip64 && (r += 20, r += 12 + this.zip64EndOfCentralSize);
                var i = e - r;
                if (0 < i) this.isSignature(e, s.CENTRAL_FILE_HEADER) || (this.reader.zero = i);else if (i < 0) throw new Error("Corrupted zip: missing " + Math.abs(i) + " bytes.");
              },
              prepareReader: function (t) {
                this.reader = i(t);
              },
              load: function (t) {
                this.prepareReader(t), this.readEndOfCentral(), this.readCentralDir(), this.readLocalFiles();
              }
            }, e.exports = h;
          }, {
            "./reader/readerFor": 22,
            "./signature": 23,
            "./support": 30,
            "./utf8": 31,
            "./utils": 32,
            "./zipEntry": 34
          }],
          34: [function (t, e, r) {

            var i = t("./reader/readerFor"),
                s = t("./utils"),
                n = t("./compressedObject"),
                a = t("./crc32"),
                o = t("./utf8"),
                h = t("./compressions"),
                u = t("./support");

            function l(t, e) {
              this.options = t, this.loadOptions = e;
            }

            l.prototype = {
              isEncrypted: function () {
                return 1 == (1 & this.bitFlag);
              },
              useUTF8: function () {
                return 2048 == (2048 & this.bitFlag);
              },
              readLocalPart: function (t) {
                var e, r;
                if (t.skip(22), this.fileNameLength = t.readInt(2), r = t.readInt(2), this.fileName = t.readData(this.fileNameLength), t.skip(r), -1 === this.compressedSize || -1 === this.uncompressedSize) throw new Error("Bug or corrupted zip : didn't get enough information from the central directory (compressedSize === -1 || uncompressedSize === -1)");
                if (null === (e = function (t) {
                  for (var e in h) if (h.hasOwnProperty(e) && h[e].magic === t) return h[e];

                  return null;
                }(this.compressionMethod))) throw new Error("Corrupted zip : compression " + s.pretty(this.compressionMethod) + " unknown (inner file : " + s.transformTo("string", this.fileName) + ")");
                this.decompressed = new n(this.compressedSize, this.uncompressedSize, this.crc32, e, t.readData(this.compressedSize));
              },
              readCentralPart: function (t) {
                this.versionMadeBy = t.readInt(2), t.skip(2), this.bitFlag = t.readInt(2), this.compressionMethod = t.readString(2), this.date = t.readDate(), this.crc32 = t.readInt(4), this.compressedSize = t.readInt(4), this.uncompressedSize = t.readInt(4);
                var e = t.readInt(2);
                if (this.extraFieldsLength = t.readInt(2), this.fileCommentLength = t.readInt(2), this.diskNumberStart = t.readInt(2), this.internalFileAttributes = t.readInt(2), this.externalFileAttributes = t.readInt(4), this.localHeaderOffset = t.readInt(4), this.isEncrypted()) throw new Error("Encrypted zip are not supported");
                t.skip(e), this.readExtraFields(t), this.parseZIP64ExtraField(t), this.fileComment = t.readData(this.fileCommentLength);
              },
              processAttributes: function () {
                this.unixPermissions = null, this.dosPermissions = null;
                var t = this.versionMadeBy >> 8;
                this.dir = !!(16 & this.externalFileAttributes), 0 == t && (this.dosPermissions = 63 & this.externalFileAttributes), 3 == t && (this.unixPermissions = this.externalFileAttributes >> 16 & 65535), this.dir || "/" !== this.fileNameStr.slice(-1) || (this.dir = !0);
              },
              parseZIP64ExtraField: function (t) {
                if (this.extraFields[1]) {
                  var e = i(this.extraFields[1].value);
                  this.uncompressedSize === s.MAX_VALUE_32BITS && (this.uncompressedSize = e.readInt(8)), this.compressedSize === s.MAX_VALUE_32BITS && (this.compressedSize = e.readInt(8)), this.localHeaderOffset === s.MAX_VALUE_32BITS && (this.localHeaderOffset = e.readInt(8)), this.diskNumberStart === s.MAX_VALUE_32BITS && (this.diskNumberStart = e.readInt(4));
                }
              },
              readExtraFields: function (t) {
                var e,
                    r,
                    i,
                    n = t.index + this.extraFieldsLength;

                for (this.extraFields || (this.extraFields = {}); t.index + 4 < n;) e = t.readInt(2), r = t.readInt(2), i = t.readData(r), this.extraFields[e] = {
                  id: e,
                  length: r,
                  value: i
                };

                t.setIndex(n);
              },
              handleUTF8: function () {
                var t = u.uint8array ? "uint8array" : "array";
                if (this.useUTF8()) this.fileNameStr = o.utf8decode(this.fileName), this.fileCommentStr = o.utf8decode(this.fileComment);else {
                  var e = this.findExtraFieldUnicodePath();
                  if (null !== e) this.fileNameStr = e;else {
                    var r = s.transformTo(t, this.fileName);
                    this.fileNameStr = this.loadOptions.decodeFileName(r);
                  }
                  var i = this.findExtraFieldUnicodeComment();
                  if (null !== i) this.fileCommentStr = i;else {
                    var n = s.transformTo(t, this.fileComment);
                    this.fileCommentStr = this.loadOptions.decodeFileName(n);
                  }
                }
              },
              findExtraFieldUnicodePath: function () {
                var t = this.extraFields[28789];

                if (t) {
                  var e = i(t.value);
                  return 1 !== e.readInt(1) ? null : a(this.fileName) !== e.readInt(4) ? null : o.utf8decode(e.readData(t.length - 5));
                }

                return null;
              },
              findExtraFieldUnicodeComment: function () {
                var t = this.extraFields[25461];

                if (t) {
                  var e = i(t.value);
                  return 1 !== e.readInt(1) ? null : a(this.fileComment) !== e.readInt(4) ? null : o.utf8decode(e.readData(t.length - 5));
                }

                return null;
              }
            }, e.exports = l;
          }, {
            "./compressedObject": 2,
            "./compressions": 3,
            "./crc32": 4,
            "./reader/readerFor": 22,
            "./support": 30,
            "./utf8": 31,
            "./utils": 32
          }],
          35: [function (t, e, r) {

            function i(t, e, r) {
              this.name = t, this.dir = r.dir, this.date = r.date, this.comment = r.comment, this.unixPermissions = r.unixPermissions, this.dosPermissions = r.dosPermissions, this._data = e, this._dataBinary = r.binary, this.options = {
                compression: r.compression,
                compressionOptions: r.compressionOptions
              };
            }

            var s = t("./stream/StreamHelper"),
                n = t("./stream/DataWorker"),
                a = t("./utf8"),
                o = t("./compressedObject"),
                h = t("./stream/GenericWorker");
            i.prototype = {
              internalStream: function (t) {
                var e = null,
                    r = "string";

                try {
                  if (!t) throw new Error("No output type specified.");
                  var i = "string" === (r = t.toLowerCase()) || "text" === r;
                  "binarystring" !== r && "text" !== r || (r = "string"), e = this._decompressWorker();
                  var n = !this._dataBinary;
                  n && !i && (e = e.pipe(new a.Utf8EncodeWorker())), !n && i && (e = e.pipe(new a.Utf8DecodeWorker()));
                } catch (t) {
                  (e = new h("error")).error(t);
                }

                return new s(e, r, "");
              },
              async: function (t, e) {
                return this.internalStream(t).accumulate(e);
              },
              nodeStream: function (t, e) {
                return this.internalStream(t || "nodebuffer").toNodejsStream(e);
              },
              _compressWorker: function (t, e) {
                if (this._data instanceof o && this._data.compression.magic === t.magic) return this._data.getCompressedWorker();

                var r = this._decompressWorker();

                return this._dataBinary || (r = r.pipe(new a.Utf8EncodeWorker())), o.createWorkerFrom(r, t, e);
              },
              _decompressWorker: function () {
                return this._data instanceof o ? this._data.getContentWorker() : this._data instanceof h ? this._data : new n(this._data);
              }
            };

            for (var u = ["asText", "asBinary", "asNodeBuffer", "asUint8Array", "asArrayBuffer"], l = function () {
              throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.");
            }, f = 0; f < u.length; f++) i.prototype[u[f]] = l;

            e.exports = i;
          }, {
            "./compressedObject": 2,
            "./stream/DataWorker": 27,
            "./stream/GenericWorker": 28,
            "./stream/StreamHelper": 29,
            "./utf8": 31
          }],
          36: [function (t, l, e) {
            (function (e) {

              var r,
                  i,
                  t = e.MutationObserver || e.WebKitMutationObserver;

              if (t) {
                var n = 0,
                    s = new t(u),
                    a = e.document.createTextNode("");
                s.observe(a, {
                  characterData: !0
                }), r = function () {
                  a.data = n = ++n % 2;
                };
              } else if (e.setImmediate || void 0 === e.MessageChannel) r = "document" in e && "onreadystatechange" in e.document.createElement("script") ? function () {
                var t = e.document.createElement("script");
                t.onreadystatechange = function () {
                  u(), t.onreadystatechange = null, t.parentNode.removeChild(t), t = null;
                }, e.document.documentElement.appendChild(t);
              } : function () {
                setTimeout(u, 0);
              };else {
                var o = new e.MessageChannel();
                o.port1.onmessage = u, r = function () {
                  o.port2.postMessage(0);
                };
              }

              var h = [];

              function u() {
                var t, e;
                i = !0;

                for (var r = h.length; r;) {
                  for (e = h, h = [], t = -1; ++t < r;) e[t]();

                  r = h.length;
                }

                i = !1;
              }

              l.exports = function (t) {
                1 !== h.push(t) || i || r();
              };
            }).call(this, "undefined" != typeof commonjsGlobal ? commonjsGlobal : "undefined" != typeof self ? self : "undefined" != typeof window ? window : {});
          }, {}],
          37: [function (t, e, r) {

            var n = t("immediate");

            function u() {}

            var l = {},
                s = ["REJECTED"],
                a = ["FULFILLED"],
                i = ["PENDING"];

            function o(t) {
              if ("function" != typeof t) throw new TypeError("resolver must be a function");
              this.state = i, this.queue = [], this.outcome = void 0, t !== u && c(this, t);
            }

            function h(t, e, r) {
              this.promise = t, "function" == typeof e && (this.onFulfilled = e, this.callFulfilled = this.otherCallFulfilled), "function" == typeof r && (this.onRejected = r, this.callRejected = this.otherCallRejected);
            }

            function f(e, r, i) {
              n(function () {
                var t;

                try {
                  t = r(i);
                } catch (t) {
                  return l.reject(e, t);
                }

                t === e ? l.reject(e, new TypeError("Cannot resolve promise with itself")) : l.resolve(e, t);
              });
            }

            function d(t) {
              var e = t && t.then;
              if (t && ("object" == typeof t || "function" == typeof t) && "function" == typeof e) return function () {
                e.apply(t, arguments);
              };
            }

            function c(e, t) {
              var r = !1;

              function i(t) {
                r || (r = !0, l.reject(e, t));
              }

              function n(t) {
                r || (r = !0, l.resolve(e, t));
              }

              var s = p(function () {
                t(n, i);
              });
              "error" === s.status && i(s.value);
            }

            function p(t, e) {
              var r = {};

              try {
                r.value = t(e), r.status = "success";
              } catch (t) {
                r.status = "error", r.value = t;
              }

              return r;
            }

            (e.exports = o).prototype.finally = function (e) {
              if ("function" != typeof e) return this;
              var r = this.constructor;
              return this.then(function (t) {
                return r.resolve(e()).then(function () {
                  return t;
                });
              }, function (t) {
                return r.resolve(e()).then(function () {
                  throw t;
                });
              });
            }, o.prototype.catch = function (t) {
              return this.then(null, t);
            }, o.prototype.then = function (t, e) {
              if ("function" != typeof t && this.state === a || "function" != typeof e && this.state === s) return this;
              var r = new this.constructor(u);
              this.state !== i ? f(r, this.state === a ? t : e, this.outcome) : this.queue.push(new h(r, t, e));
              return r;
            }, h.prototype.callFulfilled = function (t) {
              l.resolve(this.promise, t);
            }, h.prototype.otherCallFulfilled = function (t) {
              f(this.promise, this.onFulfilled, t);
            }, h.prototype.callRejected = function (t) {
              l.reject(this.promise, t);
            }, h.prototype.otherCallRejected = function (t) {
              f(this.promise, this.onRejected, t);
            }, l.resolve = function (t, e) {
              var r = p(d, e);
              if ("error" === r.status) return l.reject(t, r.value);
              var i = r.value;
              if (i) c(t, i);else {
                t.state = a, t.outcome = e;

                for (var n = -1, s = t.queue.length; ++n < s;) t.queue[n].callFulfilled(e);
              }
              return t;
            }, l.reject = function (t, e) {
              t.state = s, t.outcome = e;

              for (var r = -1, i = t.queue.length; ++r < i;) t.queue[r].callRejected(e);

              return t;
            }, o.resolve = function (t) {
              if (t instanceof this) return t;
              return l.resolve(new this(u), t);
            }, o.reject = function (t) {
              var e = new this(u);
              return l.reject(e, t);
            }, o.all = function (t) {
              var r = this;
              if ("[object Array]" !== Object.prototype.toString.call(t)) return this.reject(new TypeError("must be an array"));
              var i = t.length,
                  n = !1;
              if (!i) return this.resolve([]);
              var s = new Array(i),
                  a = 0,
                  e = -1,
                  o = new this(u);

              for (; ++e < i;) h(t[e], e);

              return o;

              function h(t, e) {
                r.resolve(t).then(function (t) {
                  s[e] = t, ++a !== i || n || (n = !0, l.resolve(o, s));
                }, function (t) {
                  n || (n = !0, l.reject(o, t));
                });
              }
            }, o.race = function (t) {
              var e = this;
              if ("[object Array]" !== Object.prototype.toString.call(t)) return this.reject(new TypeError("must be an array"));
              var r = t.length,
                  i = !1;
              if (!r) return this.resolve([]);
              var n = -1,
                  s = new this(u);

              for (; ++n < r;) a = t[n], e.resolve(a).then(function (t) {
                i || (i = !0, l.resolve(s, t));
              }, function (t) {
                i || (i = !0, l.reject(s, t));
              });

              var a;
              return s;
            };
          }, {
            immediate: 36
          }],
          38: [function (t, e, r) {

            var i = {};
            (0, t("./lib/utils/common").assign)(i, t("./lib/deflate"), t("./lib/inflate"), t("./lib/zlib/constants")), e.exports = i;
          }, {
            "./lib/deflate": 39,
            "./lib/inflate": 40,
            "./lib/utils/common": 41,
            "./lib/zlib/constants": 44
          }],
          39: [function (t, e, r) {

            var a = t("./zlib/deflate"),
                o = t("./utils/common"),
                h = t("./utils/strings"),
                n = t("./zlib/messages"),
                s = t("./zlib/zstream"),
                u = Object.prototype.toString,
                l = 0,
                f = -1,
                d = 0,
                c = 8;

            function p(t) {
              if (!(this instanceof p)) return new p(t);
              this.options = o.assign({
                level: f,
                method: c,
                chunkSize: 16384,
                windowBits: 15,
                memLevel: 8,
                strategy: d,
                to: ""
              }, t || {});
              var e = this.options;
              e.raw && 0 < e.windowBits ? e.windowBits = -e.windowBits : e.gzip && 0 < e.windowBits && e.windowBits < 16 && (e.windowBits += 16), this.err = 0, this.msg = "", this.ended = !1, this.chunks = [], this.strm = new s(), this.strm.avail_out = 0;
              var r = a.deflateInit2(this.strm, e.level, e.method, e.windowBits, e.memLevel, e.strategy);
              if (r !== l) throw new Error(n[r]);

              if (e.header && a.deflateSetHeader(this.strm, e.header), e.dictionary) {
                var i;
                if (i = "string" == typeof e.dictionary ? h.string2buf(e.dictionary) : "[object ArrayBuffer]" === u.call(e.dictionary) ? new Uint8Array(e.dictionary) : e.dictionary, (r = a.deflateSetDictionary(this.strm, i)) !== l) throw new Error(n[r]);
                this._dict_set = !0;
              }
            }

            function i(t, e) {
              var r = new p(e);
              if (r.push(t, !0), r.err) throw r.msg || n[r.err];
              return r.result;
            }

            p.prototype.push = function (t, e) {
              var r,
                  i,
                  n = this.strm,
                  s = this.options.chunkSize;
              if (this.ended) return !1;
              i = e === ~~e ? e : !0 === e ? 4 : 0, "string" == typeof t ? n.input = h.string2buf(t) : "[object ArrayBuffer]" === u.call(t) ? n.input = new Uint8Array(t) : n.input = t, n.next_in = 0, n.avail_in = n.input.length;

              do {
                if (0 === n.avail_out && (n.output = new o.Buf8(s), n.next_out = 0, n.avail_out = s), 1 !== (r = a.deflate(n, i)) && r !== l) return this.onEnd(r), !(this.ended = !0);
                0 !== n.avail_out && (0 !== n.avail_in || 4 !== i && 2 !== i) || ("string" === this.options.to ? this.onData(h.buf2binstring(o.shrinkBuf(n.output, n.next_out))) : this.onData(o.shrinkBuf(n.output, n.next_out)));
              } while ((0 < n.avail_in || 0 === n.avail_out) && 1 !== r);

              return 4 === i ? (r = a.deflateEnd(this.strm), this.onEnd(r), this.ended = !0, r === l) : 2 !== i || (this.onEnd(l), !(n.avail_out = 0));
            }, p.prototype.onData = function (t) {
              this.chunks.push(t);
            }, p.prototype.onEnd = function (t) {
              t === l && ("string" === this.options.to ? this.result = this.chunks.join("") : this.result = o.flattenChunks(this.chunks)), this.chunks = [], this.err = t, this.msg = this.strm.msg;
            }, r.Deflate = p, r.deflate = i, r.deflateRaw = function (t, e) {
              return (e = e || {}).raw = !0, i(t, e);
            }, r.gzip = function (t, e) {
              return (e = e || {}).gzip = !0, i(t, e);
            };
          }, {
            "./utils/common": 41,
            "./utils/strings": 42,
            "./zlib/deflate": 46,
            "./zlib/messages": 51,
            "./zlib/zstream": 53
          }],
          40: [function (t, e, r) {

            var d = t("./zlib/inflate"),
                c = t("./utils/common"),
                p = t("./utils/strings"),
                m = t("./zlib/constants"),
                i = t("./zlib/messages"),
                n = t("./zlib/zstream"),
                s = t("./zlib/gzheader"),
                _ = Object.prototype.toString;

            function a(t) {
              if (!(this instanceof a)) return new a(t);
              this.options = c.assign({
                chunkSize: 16384,
                windowBits: 0,
                to: ""
              }, t || {});
              var e = this.options;
              e.raw && 0 <= e.windowBits && e.windowBits < 16 && (e.windowBits = -e.windowBits, 0 === e.windowBits && (e.windowBits = -15)), !(0 <= e.windowBits && e.windowBits < 16) || t && t.windowBits || (e.windowBits += 32), 15 < e.windowBits && e.windowBits < 48 && 0 == (15 & e.windowBits) && (e.windowBits |= 15), this.err = 0, this.msg = "", this.ended = !1, this.chunks = [], this.strm = new n(), this.strm.avail_out = 0;
              var r = d.inflateInit2(this.strm, e.windowBits);
              if (r !== m.Z_OK) throw new Error(i[r]);
              this.header = new s(), d.inflateGetHeader(this.strm, this.header);
            }

            function o(t, e) {
              var r = new a(e);
              if (r.push(t, !0), r.err) throw r.msg || i[r.err];
              return r.result;
            }

            a.prototype.push = function (t, e) {
              var r,
                  i,
                  n,
                  s,
                  a,
                  o,
                  h = this.strm,
                  u = this.options.chunkSize,
                  l = this.options.dictionary,
                  f = !1;
              if (this.ended) return !1;
              i = e === ~~e ? e : !0 === e ? m.Z_FINISH : m.Z_NO_FLUSH, "string" == typeof t ? h.input = p.binstring2buf(t) : "[object ArrayBuffer]" === _.call(t) ? h.input = new Uint8Array(t) : h.input = t, h.next_in = 0, h.avail_in = h.input.length;

              do {
                if (0 === h.avail_out && (h.output = new c.Buf8(u), h.next_out = 0, h.avail_out = u), (r = d.inflate(h, m.Z_NO_FLUSH)) === m.Z_NEED_DICT && l && (o = "string" == typeof l ? p.string2buf(l) : "[object ArrayBuffer]" === _.call(l) ? new Uint8Array(l) : l, r = d.inflateSetDictionary(this.strm, o)), r === m.Z_BUF_ERROR && !0 === f && (r = m.Z_OK, f = !1), r !== m.Z_STREAM_END && r !== m.Z_OK) return this.onEnd(r), !(this.ended = !0);
                h.next_out && (0 !== h.avail_out && r !== m.Z_STREAM_END && (0 !== h.avail_in || i !== m.Z_FINISH && i !== m.Z_SYNC_FLUSH) || ("string" === this.options.to ? (n = p.utf8border(h.output, h.next_out), s = h.next_out - n, a = p.buf2string(h.output, n), h.next_out = s, h.avail_out = u - s, s && c.arraySet(h.output, h.output, n, s, 0), this.onData(a)) : this.onData(c.shrinkBuf(h.output, h.next_out)))), 0 === h.avail_in && 0 === h.avail_out && (f = !0);
              } while ((0 < h.avail_in || 0 === h.avail_out) && r !== m.Z_STREAM_END);

              return r === m.Z_STREAM_END && (i = m.Z_FINISH), i === m.Z_FINISH ? (r = d.inflateEnd(this.strm), this.onEnd(r), this.ended = !0, r === m.Z_OK) : i !== m.Z_SYNC_FLUSH || (this.onEnd(m.Z_OK), !(h.avail_out = 0));
            }, a.prototype.onData = function (t) {
              this.chunks.push(t);
            }, a.prototype.onEnd = function (t) {
              t === m.Z_OK && ("string" === this.options.to ? this.result = this.chunks.join("") : this.result = c.flattenChunks(this.chunks)), this.chunks = [], this.err = t, this.msg = this.strm.msg;
            }, r.Inflate = a, r.inflate = o, r.inflateRaw = function (t, e) {
              return (e = e || {}).raw = !0, o(t, e);
            }, r.ungzip = o;
          }, {
            "./utils/common": 41,
            "./utils/strings": 42,
            "./zlib/constants": 44,
            "./zlib/gzheader": 47,
            "./zlib/inflate": 49,
            "./zlib/messages": 51,
            "./zlib/zstream": 53
          }],
          41: [function (t, e, r) {

            var i = "undefined" != typeof Uint8Array && "undefined" != typeof Uint16Array && "undefined" != typeof Int32Array;
            r.assign = function (t) {
              for (var e = Array.prototype.slice.call(arguments, 1); e.length;) {
                var r = e.shift();

                if (r) {
                  if ("object" != typeof r) throw new TypeError(r + "must be non-object");

                  for (var i in r) r.hasOwnProperty(i) && (t[i] = r[i]);
                }
              }

              return t;
            }, r.shrinkBuf = function (t, e) {
              return t.length === e ? t : t.subarray ? t.subarray(0, e) : (t.length = e, t);
            };
            var n = {
              arraySet: function (t, e, r, i, n) {
                if (e.subarray && t.subarray) t.set(e.subarray(r, r + i), n);else for (var s = 0; s < i; s++) t[n + s] = e[r + s];
              },
              flattenChunks: function (t) {
                var e, r, i, n, s, a;

                for (e = i = 0, r = t.length; e < r; e++) i += t[e].length;

                for (a = new Uint8Array(i), e = n = 0, r = t.length; e < r; e++) s = t[e], a.set(s, n), n += s.length;

                return a;
              }
            },
                s = {
              arraySet: function (t, e, r, i, n) {
                for (var s = 0; s < i; s++) t[n + s] = e[r + s];
              },
              flattenChunks: function (t) {
                return [].concat.apply([], t);
              }
            };
            r.setTyped = function (t) {
              t ? (r.Buf8 = Uint8Array, r.Buf16 = Uint16Array, r.Buf32 = Int32Array, r.assign(r, n)) : (r.Buf8 = Array, r.Buf16 = Array, r.Buf32 = Array, r.assign(r, s));
            }, r.setTyped(i);
          }, {}],
          42: [function (t, e, r) {

            var h = t("./common"),
                n = !0,
                s = !0;

            try {
              String.fromCharCode.apply(null, [0]);
            } catch (t) {
              n = !1;
            }

            try {
              String.fromCharCode.apply(null, new Uint8Array(1));
            } catch (t) {
              s = !1;
            }

            for (var u = new h.Buf8(256), i = 0; i < 256; i++) u[i] = 252 <= i ? 6 : 248 <= i ? 5 : 240 <= i ? 4 : 224 <= i ? 3 : 192 <= i ? 2 : 1;

            function l(t, e) {
              if (e < 65537 && (t.subarray && s || !t.subarray && n)) return String.fromCharCode.apply(null, h.shrinkBuf(t, e));

              for (var r = "", i = 0; i < e; i++) r += String.fromCharCode(t[i]);

              return r;
            }

            u[254] = u[254] = 1, r.string2buf = function (t) {
              var e,
                  r,
                  i,
                  n,
                  s,
                  a = t.length,
                  o = 0;

              for (n = 0; n < a; n++) 55296 == (64512 & (r = t.charCodeAt(n))) && n + 1 < a && 56320 == (64512 & (i = t.charCodeAt(n + 1))) && (r = 65536 + (r - 55296 << 10) + (i - 56320), n++), o += r < 128 ? 1 : r < 2048 ? 2 : r < 65536 ? 3 : 4;

              for (e = new h.Buf8(o), n = s = 0; s < o; n++) 55296 == (64512 & (r = t.charCodeAt(n))) && n + 1 < a && 56320 == (64512 & (i = t.charCodeAt(n + 1))) && (r = 65536 + (r - 55296 << 10) + (i - 56320), n++), r < 128 ? e[s++] = r : (r < 2048 ? e[s++] = 192 | r >>> 6 : (r < 65536 ? e[s++] = 224 | r >>> 12 : (e[s++] = 240 | r >>> 18, e[s++] = 128 | r >>> 12 & 63), e[s++] = 128 | r >>> 6 & 63), e[s++] = 128 | 63 & r);

              return e;
            }, r.buf2binstring = function (t) {
              return l(t, t.length);
            }, r.binstring2buf = function (t) {
              for (var e = new h.Buf8(t.length), r = 0, i = e.length; r < i; r++) e[r] = t.charCodeAt(r);

              return e;
            }, r.buf2string = function (t, e) {
              var r,
                  i,
                  n,
                  s,
                  a = e || t.length,
                  o = new Array(2 * a);

              for (r = i = 0; r < a;) if ((n = t[r++]) < 128) o[i++] = n;else if (4 < (s = u[n])) o[i++] = 65533, r += s - 1;else {
                for (n &= 2 === s ? 31 : 3 === s ? 15 : 7; 1 < s && r < a;) n = n << 6 | 63 & t[r++], s--;

                1 < s ? o[i++] = 65533 : n < 65536 ? o[i++] = n : (n -= 65536, o[i++] = 55296 | n >> 10 & 1023, o[i++] = 56320 | 1023 & n);
              }

              return l(o, i);
            }, r.utf8border = function (t, e) {
              var r;

              for ((e = e || t.length) > t.length && (e = t.length), r = e - 1; 0 <= r && 128 == (192 & t[r]);) r--;

              return r < 0 ? e : 0 === r ? e : r + u[t[r]] > e ? r : e;
            };
          }, {
            "./common": 41
          }],
          43: [function (t, e, r) {

            e.exports = function (t, e, r, i) {
              for (var n = 65535 & t | 0, s = t >>> 16 & 65535 | 0, a = 0; 0 !== r;) {
                for (r -= a = 2e3 < r ? 2e3 : r; s = s + (n = n + e[i++] | 0) | 0, --a;);

                n %= 65521, s %= 65521;
              }

              return n | s << 16 | 0;
            };
          }, {}],
          44: [function (t, e, r) {

            e.exports = {
              Z_NO_FLUSH: 0,
              Z_PARTIAL_FLUSH: 1,
              Z_SYNC_FLUSH: 2,
              Z_FULL_FLUSH: 3,
              Z_FINISH: 4,
              Z_BLOCK: 5,
              Z_TREES: 6,
              Z_OK: 0,
              Z_STREAM_END: 1,
              Z_NEED_DICT: 2,
              Z_ERRNO: -1,
              Z_STREAM_ERROR: -2,
              Z_DATA_ERROR: -3,
              Z_BUF_ERROR: -5,
              Z_NO_COMPRESSION: 0,
              Z_BEST_SPEED: 1,
              Z_BEST_COMPRESSION: 9,
              Z_DEFAULT_COMPRESSION: -1,
              Z_FILTERED: 1,
              Z_HUFFMAN_ONLY: 2,
              Z_RLE: 3,
              Z_FIXED: 4,
              Z_DEFAULT_STRATEGY: 0,
              Z_BINARY: 0,
              Z_TEXT: 1,
              Z_UNKNOWN: 2,
              Z_DEFLATED: 8
            };
          }, {}],
          45: [function (t, e, r) {

            var o = function () {
              for (var t, e = [], r = 0; r < 256; r++) {
                t = r;

                for (var i = 0; i < 8; i++) t = 1 & t ? 3988292384 ^ t >>> 1 : t >>> 1;

                e[r] = t;
              }

              return e;
            }();

            e.exports = function (t, e, r, i) {
              var n = o,
                  s = i + r;
              t ^= -1;

              for (var a = i; a < s; a++) t = t >>> 8 ^ n[255 & (t ^ e[a])];

              return -1 ^ t;
            };
          }, {}],
          46: [function (t, e, r) {

            var h,
                d = t("../utils/common"),
                u = t("./trees"),
                c = t("./adler32"),
                p = t("./crc32"),
                i = t("./messages"),
                l = 0,
                f = 4,
                m = 0,
                _ = -2,
                g = -1,
                b = 4,
                n = 2,
                v = 8,
                y = 9,
                s = 286,
                a = 30,
                o = 19,
                w = 2 * s + 1,
                k = 15,
                x = 3,
                S = 258,
                z = S + x + 1,
                C = 42,
                E = 113,
                A = 1,
                I = 2,
                O = 3,
                B = 4;

            function R(t, e) {
              return t.msg = i[e], e;
            }

            function T(t) {
              return (t << 1) - (4 < t ? 9 : 0);
            }

            function D(t) {
              for (var e = t.length; 0 <= --e;) t[e] = 0;
            }

            function F(t) {
              var e = t.state,
                  r = e.pending;
              r > t.avail_out && (r = t.avail_out), 0 !== r && (d.arraySet(t.output, e.pending_buf, e.pending_out, r, t.next_out), t.next_out += r, e.pending_out += r, t.total_out += r, t.avail_out -= r, e.pending -= r, 0 === e.pending && (e.pending_out = 0));
            }

            function N(t, e) {
              u._tr_flush_block(t, 0 <= t.block_start ? t.block_start : -1, t.strstart - t.block_start, e), t.block_start = t.strstart, F(t.strm);
            }

            function U(t, e) {
              t.pending_buf[t.pending++] = e;
            }

            function P(t, e) {
              t.pending_buf[t.pending++] = e >>> 8 & 255, t.pending_buf[t.pending++] = 255 & e;
            }

            function L(t, e) {
              var r,
                  i,
                  n = t.max_chain_length,
                  s = t.strstart,
                  a = t.prev_length,
                  o = t.nice_match,
                  h = t.strstart > t.w_size - z ? t.strstart - (t.w_size - z) : 0,
                  u = t.window,
                  l = t.w_mask,
                  f = t.prev,
                  d = t.strstart + S,
                  c = u[s + a - 1],
                  p = u[s + a];
              t.prev_length >= t.good_match && (n >>= 2), o > t.lookahead && (o = t.lookahead);

              do {
                if (u[(r = e) + a] === p && u[r + a - 1] === c && u[r] === u[s] && u[++r] === u[s + 1]) {
                  s += 2, r++;

                  do {} while (u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && s < d);

                  if (i = S - (d - s), s = d - S, a < i) {
                    if (t.match_start = e, o <= (a = i)) break;
                    c = u[s + a - 1], p = u[s + a];
                  }
                }
              } while ((e = f[e & l]) > h && 0 != --n);

              return a <= t.lookahead ? a : t.lookahead;
            }

            function j(t) {
              var e,
                  r,
                  i,
                  n,
                  s,
                  a,
                  o,
                  h,
                  u,
                  l,
                  f = t.w_size;

              do {
                if (n = t.window_size - t.lookahead - t.strstart, t.strstart >= f + (f - z)) {
                  for (d.arraySet(t.window, t.window, f, f, 0), t.match_start -= f, t.strstart -= f, t.block_start -= f, e = r = t.hash_size; i = t.head[--e], t.head[e] = f <= i ? i - f : 0, --r;);

                  for (e = r = f; i = t.prev[--e], t.prev[e] = f <= i ? i - f : 0, --r;);

                  n += f;
                }

                if (0 === t.strm.avail_in) break;
                if (a = t.strm, o = t.window, h = t.strstart + t.lookahead, u = n, l = void 0, l = a.avail_in, u < l && (l = u), r = 0 === l ? 0 : (a.avail_in -= l, d.arraySet(o, a.input, a.next_in, l, h), 1 === a.state.wrap ? a.adler = c(a.adler, o, l, h) : 2 === a.state.wrap && (a.adler = p(a.adler, o, l, h)), a.next_in += l, a.total_in += l, l), t.lookahead += r, t.lookahead + t.insert >= x) for (s = t.strstart - t.insert, t.ins_h = t.window[s], t.ins_h = (t.ins_h << t.hash_shift ^ t.window[s + 1]) & t.hash_mask; t.insert && (t.ins_h = (t.ins_h << t.hash_shift ^ t.window[s + x - 1]) & t.hash_mask, t.prev[s & t.w_mask] = t.head[t.ins_h], t.head[t.ins_h] = s, s++, t.insert--, !(t.lookahead + t.insert < x)););
              } while (t.lookahead < z && 0 !== t.strm.avail_in);
            }

            function Z(t, e) {
              for (var r, i;;) {
                if (t.lookahead < z) {
                  if (j(t), t.lookahead < z && e === l) return A;
                  if (0 === t.lookahead) break;
                }

                if (r = 0, t.lookahead >= x && (t.ins_h = (t.ins_h << t.hash_shift ^ t.window[t.strstart + x - 1]) & t.hash_mask, r = t.prev[t.strstart & t.w_mask] = t.head[t.ins_h], t.head[t.ins_h] = t.strstart), 0 !== r && t.strstart - r <= t.w_size - z && (t.match_length = L(t, r)), t.match_length >= x) {
                  if (i = u._tr_tally(t, t.strstart - t.match_start, t.match_length - x), t.lookahead -= t.match_length, t.match_length <= t.max_lazy_match && t.lookahead >= x) {
                    for (t.match_length--; t.strstart++, t.ins_h = (t.ins_h << t.hash_shift ^ t.window[t.strstart + x - 1]) & t.hash_mask, r = t.prev[t.strstart & t.w_mask] = t.head[t.ins_h], t.head[t.ins_h] = t.strstart, 0 != --t.match_length;);

                    t.strstart++;
                  } else t.strstart += t.match_length, t.match_length = 0, t.ins_h = t.window[t.strstart], t.ins_h = (t.ins_h << t.hash_shift ^ t.window[t.strstart + 1]) & t.hash_mask;
                } else i = u._tr_tally(t, 0, t.window[t.strstart]), t.lookahead--, t.strstart++;
                if (i && (N(t, !1), 0 === t.strm.avail_out)) return A;
              }

              return t.insert = t.strstart < x - 1 ? t.strstart : x - 1, e === f ? (N(t, !0), 0 === t.strm.avail_out ? O : B) : t.last_lit && (N(t, !1), 0 === t.strm.avail_out) ? A : I;
            }

            function W(t, e) {
              for (var r, i, n;;) {
                if (t.lookahead < z) {
                  if (j(t), t.lookahead < z && e === l) return A;
                  if (0 === t.lookahead) break;
                }

                if (r = 0, t.lookahead >= x && (t.ins_h = (t.ins_h << t.hash_shift ^ t.window[t.strstart + x - 1]) & t.hash_mask, r = t.prev[t.strstart & t.w_mask] = t.head[t.ins_h], t.head[t.ins_h] = t.strstart), t.prev_length = t.match_length, t.prev_match = t.match_start, t.match_length = x - 1, 0 !== r && t.prev_length < t.max_lazy_match && t.strstart - r <= t.w_size - z && (t.match_length = L(t, r), t.match_length <= 5 && (1 === t.strategy || t.match_length === x && 4096 < t.strstart - t.match_start) && (t.match_length = x - 1)), t.prev_length >= x && t.match_length <= t.prev_length) {
                  for (n = t.strstart + t.lookahead - x, i = u._tr_tally(t, t.strstart - 1 - t.prev_match, t.prev_length - x), t.lookahead -= t.prev_length - 1, t.prev_length -= 2; ++t.strstart <= n && (t.ins_h = (t.ins_h << t.hash_shift ^ t.window[t.strstart + x - 1]) & t.hash_mask, r = t.prev[t.strstart & t.w_mask] = t.head[t.ins_h], t.head[t.ins_h] = t.strstart), 0 != --t.prev_length;);

                  if (t.match_available = 0, t.match_length = x - 1, t.strstart++, i && (N(t, !1), 0 === t.strm.avail_out)) return A;
                } else if (t.match_available) {
                  if ((i = u._tr_tally(t, 0, t.window[t.strstart - 1])) && N(t, !1), t.strstart++, t.lookahead--, 0 === t.strm.avail_out) return A;
                } else t.match_available = 1, t.strstart++, t.lookahead--;
              }

              return t.match_available && (i = u._tr_tally(t, 0, t.window[t.strstart - 1]), t.match_available = 0), t.insert = t.strstart < x - 1 ? t.strstart : x - 1, e === f ? (N(t, !0), 0 === t.strm.avail_out ? O : B) : t.last_lit && (N(t, !1), 0 === t.strm.avail_out) ? A : I;
            }

            function M(t, e, r, i, n) {
              this.good_length = t, this.max_lazy = e, this.nice_length = r, this.max_chain = i, this.func = n;
            }

            function H() {
              this.strm = null, this.status = 0, this.pending_buf = null, this.pending_buf_size = 0, this.pending_out = 0, this.pending = 0, this.wrap = 0, this.gzhead = null, this.gzindex = 0, this.method = v, this.last_flush = -1, this.w_size = 0, this.w_bits = 0, this.w_mask = 0, this.window = null, this.window_size = 0, this.prev = null, this.head = null, this.ins_h = 0, this.hash_size = 0, this.hash_bits = 0, this.hash_mask = 0, this.hash_shift = 0, this.block_start = 0, this.match_length = 0, this.prev_match = 0, this.match_available = 0, this.strstart = 0, this.match_start = 0, this.lookahead = 0, this.prev_length = 0, this.max_chain_length = 0, this.max_lazy_match = 0, this.level = 0, this.strategy = 0, this.good_match = 0, this.nice_match = 0, this.dyn_ltree = new d.Buf16(2 * w), this.dyn_dtree = new d.Buf16(2 * (2 * a + 1)), this.bl_tree = new d.Buf16(2 * (2 * o + 1)), D(this.dyn_ltree), D(this.dyn_dtree), D(this.bl_tree), this.l_desc = null, this.d_desc = null, this.bl_desc = null, this.bl_count = new d.Buf16(k + 1), this.heap = new d.Buf16(2 * s + 1), D(this.heap), this.heap_len = 0, this.heap_max = 0, this.depth = new d.Buf16(2 * s + 1), D(this.depth), this.l_buf = 0, this.lit_bufsize = 0, this.last_lit = 0, this.d_buf = 0, this.opt_len = 0, this.static_len = 0, this.matches = 0, this.insert = 0, this.bi_buf = 0, this.bi_valid = 0;
            }

            function G(t) {
              var e;
              return t && t.state ? (t.total_in = t.total_out = 0, t.data_type = n, (e = t.state).pending = 0, e.pending_out = 0, e.wrap < 0 && (e.wrap = -e.wrap), e.status = e.wrap ? C : E, t.adler = 2 === e.wrap ? 0 : 1, e.last_flush = l, u._tr_init(e), m) : R(t, _);
            }

            function K(t) {
              var e = G(t);
              return e === m && function (t) {
                t.window_size = 2 * t.w_size, D(t.head), t.max_lazy_match = h[t.level].max_lazy, t.good_match = h[t.level].good_length, t.nice_match = h[t.level].nice_length, t.max_chain_length = h[t.level].max_chain, t.strstart = 0, t.block_start = 0, t.lookahead = 0, t.insert = 0, t.match_length = t.prev_length = x - 1, t.match_available = 0, t.ins_h = 0;
              }(t.state), e;
            }

            function Y(t, e, r, i, n, s) {
              if (!t) return _;
              var a = 1;
              if (e === g && (e = 6), i < 0 ? (a = 0, i = -i) : 15 < i && (a = 2, i -= 16), n < 1 || y < n || r !== v || i < 8 || 15 < i || e < 0 || 9 < e || s < 0 || b < s) return R(t, _);
              8 === i && (i = 9);
              var o = new H();
              return (t.state = o).strm = t, o.wrap = a, o.gzhead = null, o.w_bits = i, o.w_size = 1 << o.w_bits, o.w_mask = o.w_size - 1, o.hash_bits = n + 7, o.hash_size = 1 << o.hash_bits, o.hash_mask = o.hash_size - 1, o.hash_shift = ~~((o.hash_bits + x - 1) / x), o.window = new d.Buf8(2 * o.w_size), o.head = new d.Buf16(o.hash_size), o.prev = new d.Buf16(o.w_size), o.lit_bufsize = 1 << n + 6, o.pending_buf_size = 4 * o.lit_bufsize, o.pending_buf = new d.Buf8(o.pending_buf_size), o.d_buf = 1 * o.lit_bufsize, o.l_buf = 3 * o.lit_bufsize, o.level = e, o.strategy = s, o.method = r, K(t);
            }

            h = [new M(0, 0, 0, 0, function (t, e) {
              var r = 65535;

              for (r > t.pending_buf_size - 5 && (r = t.pending_buf_size - 5);;) {
                if (t.lookahead <= 1) {
                  if (j(t), 0 === t.lookahead && e === l) return A;
                  if (0 === t.lookahead) break;
                }

                t.strstart += t.lookahead, t.lookahead = 0;
                var i = t.block_start + r;
                if ((0 === t.strstart || t.strstart >= i) && (t.lookahead = t.strstart - i, t.strstart = i, N(t, !1), 0 === t.strm.avail_out)) return A;
                if (t.strstart - t.block_start >= t.w_size - z && (N(t, !1), 0 === t.strm.avail_out)) return A;
              }

              return t.insert = 0, e === f ? (N(t, !0), 0 === t.strm.avail_out ? O : B) : (t.strstart > t.block_start && (N(t, !1), t.strm.avail_out), A);
            }), new M(4, 4, 8, 4, Z), new M(4, 5, 16, 8, Z), new M(4, 6, 32, 32, Z), new M(4, 4, 16, 16, W), new M(8, 16, 32, 32, W), new M(8, 16, 128, 128, W), new M(8, 32, 128, 256, W), new M(32, 128, 258, 1024, W), new M(32, 258, 258, 4096, W)], r.deflateInit = function (t, e) {
              return Y(t, e, v, 15, 8, 0);
            }, r.deflateInit2 = Y, r.deflateReset = K, r.deflateResetKeep = G, r.deflateSetHeader = function (t, e) {
              return t && t.state ? 2 !== t.state.wrap ? _ : (t.state.gzhead = e, m) : _;
            }, r.deflate = function (t, e) {
              var r, i, n, s;
              if (!t || !t.state || 5 < e || e < 0) return t ? R(t, _) : _;
              if (i = t.state, !t.output || !t.input && 0 !== t.avail_in || 666 === i.status && e !== f) return R(t, 0 === t.avail_out ? -5 : _);
              if (i.strm = t, r = i.last_flush, i.last_flush = e, i.status === C) if (2 === i.wrap) t.adler = 0, U(i, 31), U(i, 139), U(i, 8), i.gzhead ? (U(i, (i.gzhead.text ? 1 : 0) + (i.gzhead.hcrc ? 2 : 0) + (i.gzhead.extra ? 4 : 0) + (i.gzhead.name ? 8 : 0) + (i.gzhead.comment ? 16 : 0)), U(i, 255 & i.gzhead.time), U(i, i.gzhead.time >> 8 & 255), U(i, i.gzhead.time >> 16 & 255), U(i, i.gzhead.time >> 24 & 255), U(i, 9 === i.level ? 2 : 2 <= i.strategy || i.level < 2 ? 4 : 0), U(i, 255 & i.gzhead.os), i.gzhead.extra && i.gzhead.extra.length && (U(i, 255 & i.gzhead.extra.length), U(i, i.gzhead.extra.length >> 8 & 255)), i.gzhead.hcrc && (t.adler = p(t.adler, i.pending_buf, i.pending, 0)), i.gzindex = 0, i.status = 69) : (U(i, 0), U(i, 0), U(i, 0), U(i, 0), U(i, 0), U(i, 9 === i.level ? 2 : 2 <= i.strategy || i.level < 2 ? 4 : 0), U(i, 3), i.status = E);else {
                var a = v + (i.w_bits - 8 << 4) << 8;
                a |= (2 <= i.strategy || i.level < 2 ? 0 : i.level < 6 ? 1 : 6 === i.level ? 2 : 3) << 6, 0 !== i.strstart && (a |= 32), a += 31 - a % 31, i.status = E, P(i, a), 0 !== i.strstart && (P(i, t.adler >>> 16), P(i, 65535 & t.adler)), t.adler = 1;
              }
              if (69 === i.status) if (i.gzhead.extra) {
                for (n = i.pending; i.gzindex < (65535 & i.gzhead.extra.length) && (i.pending !== i.pending_buf_size || (i.gzhead.hcrc && i.pending > n && (t.adler = p(t.adler, i.pending_buf, i.pending - n, n)), F(t), n = i.pending, i.pending !== i.pending_buf_size));) U(i, 255 & i.gzhead.extra[i.gzindex]), i.gzindex++;

                i.gzhead.hcrc && i.pending > n && (t.adler = p(t.adler, i.pending_buf, i.pending - n, n)), i.gzindex === i.gzhead.extra.length && (i.gzindex = 0, i.status = 73);
              } else i.status = 73;
              if (73 === i.status) if (i.gzhead.name) {
                n = i.pending;

                do {
                  if (i.pending === i.pending_buf_size && (i.gzhead.hcrc && i.pending > n && (t.adler = p(t.adler, i.pending_buf, i.pending - n, n)), F(t), n = i.pending, i.pending === i.pending_buf_size)) {
                    s = 1;
                    break;
                  }

                  s = i.gzindex < i.gzhead.name.length ? 255 & i.gzhead.name.charCodeAt(i.gzindex++) : 0, U(i, s);
                } while (0 !== s);

                i.gzhead.hcrc && i.pending > n && (t.adler = p(t.adler, i.pending_buf, i.pending - n, n)), 0 === s && (i.gzindex = 0, i.status = 91);
              } else i.status = 91;
              if (91 === i.status) if (i.gzhead.comment) {
                n = i.pending;

                do {
                  if (i.pending === i.pending_buf_size && (i.gzhead.hcrc && i.pending > n && (t.adler = p(t.adler, i.pending_buf, i.pending - n, n)), F(t), n = i.pending, i.pending === i.pending_buf_size)) {
                    s = 1;
                    break;
                  }

                  s = i.gzindex < i.gzhead.comment.length ? 255 & i.gzhead.comment.charCodeAt(i.gzindex++) : 0, U(i, s);
                } while (0 !== s);

                i.gzhead.hcrc && i.pending > n && (t.adler = p(t.adler, i.pending_buf, i.pending - n, n)), 0 === s && (i.status = 103);
              } else i.status = 103;

              if (103 === i.status && (i.gzhead.hcrc ? (i.pending + 2 > i.pending_buf_size && F(t), i.pending + 2 <= i.pending_buf_size && (U(i, 255 & t.adler), U(i, t.adler >> 8 & 255), t.adler = 0, i.status = E)) : i.status = E), 0 !== i.pending) {
                if (F(t), 0 === t.avail_out) return i.last_flush = -1, m;
              } else if (0 === t.avail_in && T(e) <= T(r) && e !== f) return R(t, -5);

              if (666 === i.status && 0 !== t.avail_in) return R(t, -5);

              if (0 !== t.avail_in || 0 !== i.lookahead || e !== l && 666 !== i.status) {
                var o = 2 === i.strategy ? function (t, e) {
                  for (var r;;) {
                    if (0 === t.lookahead && (j(t), 0 === t.lookahead)) {
                      if (e === l) return A;
                      break;
                    }

                    if (t.match_length = 0, r = u._tr_tally(t, 0, t.window[t.strstart]), t.lookahead--, t.strstart++, r && (N(t, !1), 0 === t.strm.avail_out)) return A;
                  }

                  return t.insert = 0, e === f ? (N(t, !0), 0 === t.strm.avail_out ? O : B) : t.last_lit && (N(t, !1), 0 === t.strm.avail_out) ? A : I;
                }(i, e) : 3 === i.strategy ? function (t, e) {
                  for (var r, i, n, s, a = t.window;;) {
                    if (t.lookahead <= S) {
                      if (j(t), t.lookahead <= S && e === l) return A;
                      if (0 === t.lookahead) break;
                    }

                    if (t.match_length = 0, t.lookahead >= x && 0 < t.strstart && (i = a[n = t.strstart - 1]) === a[++n] && i === a[++n] && i === a[++n]) {
                      s = t.strstart + S;

                      do {} while (i === a[++n] && i === a[++n] && i === a[++n] && i === a[++n] && i === a[++n] && i === a[++n] && i === a[++n] && i === a[++n] && n < s);

                      t.match_length = S - (s - n), t.match_length > t.lookahead && (t.match_length = t.lookahead);
                    }

                    if (t.match_length >= x ? (r = u._tr_tally(t, 1, t.match_length - x), t.lookahead -= t.match_length, t.strstart += t.match_length, t.match_length = 0) : (r = u._tr_tally(t, 0, t.window[t.strstart]), t.lookahead--, t.strstart++), r && (N(t, !1), 0 === t.strm.avail_out)) return A;
                  }

                  return t.insert = 0, e === f ? (N(t, !0), 0 === t.strm.avail_out ? O : B) : t.last_lit && (N(t, !1), 0 === t.strm.avail_out) ? A : I;
                }(i, e) : h[i.level].func(i, e);
                if (o !== O && o !== B || (i.status = 666), o === A || o === O) return 0 === t.avail_out && (i.last_flush = -1), m;
                if (o === I && (1 === e ? u._tr_align(i) : 5 !== e && (u._tr_stored_block(i, 0, 0, !1), 3 === e && (D(i.head), 0 === i.lookahead && (i.strstart = 0, i.block_start = 0, i.insert = 0))), F(t), 0 === t.avail_out)) return i.last_flush = -1, m;
              }

              return e !== f ? m : i.wrap <= 0 ? 1 : (2 === i.wrap ? (U(i, 255 & t.adler), U(i, t.adler >> 8 & 255), U(i, t.adler >> 16 & 255), U(i, t.adler >> 24 & 255), U(i, 255 & t.total_in), U(i, t.total_in >> 8 & 255), U(i, t.total_in >> 16 & 255), U(i, t.total_in >> 24 & 255)) : (P(i, t.adler >>> 16), P(i, 65535 & t.adler)), F(t), 0 < i.wrap && (i.wrap = -i.wrap), 0 !== i.pending ? m : 1);
            }, r.deflateEnd = function (t) {
              var e;
              return t && t.state ? (e = t.state.status) !== C && 69 !== e && 73 !== e && 91 !== e && 103 !== e && e !== E && 666 !== e ? R(t, _) : (t.state = null, e === E ? R(t, -3) : m) : _;
            }, r.deflateSetDictionary = function (t, e) {
              var r,
                  i,
                  n,
                  s,
                  a,
                  o,
                  h,
                  u,
                  l = e.length;
              if (!t || !t.state) return _;
              if (2 === (s = (r = t.state).wrap) || 1 === s && r.status !== C || r.lookahead) return _;

              for (1 === s && (t.adler = c(t.adler, e, l, 0)), r.wrap = 0, l >= r.w_size && (0 === s && (D(r.head), r.strstart = 0, r.block_start = 0, r.insert = 0), u = new d.Buf8(r.w_size), d.arraySet(u, e, l - r.w_size, r.w_size, 0), e = u, l = r.w_size), a = t.avail_in, o = t.next_in, h = t.input, t.avail_in = l, t.next_in = 0, t.input = e, j(r); r.lookahead >= x;) {
                for (i = r.strstart, n = r.lookahead - (x - 1); r.ins_h = (r.ins_h << r.hash_shift ^ r.window[i + x - 1]) & r.hash_mask, r.prev[i & r.w_mask] = r.head[r.ins_h], r.head[r.ins_h] = i, i++, --n;);

                r.strstart = i, r.lookahead = x - 1, j(r);
              }

              return r.strstart += r.lookahead, r.block_start = r.strstart, r.insert = r.lookahead, r.lookahead = 0, r.match_length = r.prev_length = x - 1, r.match_available = 0, t.next_in = o, t.input = h, t.avail_in = a, r.wrap = s, m;
            }, r.deflateInfo = "pako deflate (from Nodeca project)";
          }, {
            "../utils/common": 41,
            "./adler32": 43,
            "./crc32": 45,
            "./messages": 51,
            "./trees": 52
          }],
          47: [function (t, e, r) {

            e.exports = function () {
              this.text = 0, this.time = 0, this.xflags = 0, this.os = 0, this.extra = null, this.extra_len = 0, this.name = "", this.comment = "", this.hcrc = 0, this.done = !1;
            };
          }, {}],
          48: [function (t, e, r) {

            e.exports = function (t, e) {
              var r, i, n, s, a, o, h, u, l, f, d, c, p, m, _, g, b, v, y, w, k, x, S, z, C;

              r = t.state, i = t.next_in, z = t.input, n = i + (t.avail_in - 5), s = t.next_out, C = t.output, a = s - (e - t.avail_out), o = s + (t.avail_out - 257), h = r.dmax, u = r.wsize, l = r.whave, f = r.wnext, d = r.window, c = r.hold, p = r.bits, m = r.lencode, _ = r.distcode, g = (1 << r.lenbits) - 1, b = (1 << r.distbits) - 1;

              t: do {
                p < 15 && (c += z[i++] << p, p += 8, c += z[i++] << p, p += 8), v = m[c & g];

                e: for (;;) {
                  if (c >>>= y = v >>> 24, p -= y, 0 === (y = v >>> 16 & 255)) C[s++] = 65535 & v;else {
                    if (!(16 & y)) {
                      if (0 == (64 & y)) {
                        v = m[(65535 & v) + (c & (1 << y) - 1)];
                        continue e;
                      }

                      if (32 & y) {
                        r.mode = 12;
                        break t;
                      }

                      t.msg = "invalid literal/length code", r.mode = 30;
                      break t;
                    }

                    w = 65535 & v, (y &= 15) && (p < y && (c += z[i++] << p, p += 8), w += c & (1 << y) - 1, c >>>= y, p -= y), p < 15 && (c += z[i++] << p, p += 8, c += z[i++] << p, p += 8), v = _[c & b];

                    r: for (;;) {
                      if (c >>>= y = v >>> 24, p -= y, !(16 & (y = v >>> 16 & 255))) {
                        if (0 == (64 & y)) {
                          v = _[(65535 & v) + (c & (1 << y) - 1)];
                          continue r;
                        }

                        t.msg = "invalid distance code", r.mode = 30;
                        break t;
                      }

                      if (k = 65535 & v, p < (y &= 15) && (c += z[i++] << p, (p += 8) < y && (c += z[i++] << p, p += 8)), h < (k += c & (1 << y) - 1)) {
                        t.msg = "invalid distance too far back", r.mode = 30;
                        break t;
                      }

                      if (c >>>= y, p -= y, (y = s - a) < k) {
                        if (l < (y = k - y) && r.sane) {
                          t.msg = "invalid distance too far back", r.mode = 30;
                          break t;
                        }

                        if (S = d, (x = 0) === f) {
                          if (x += u - y, y < w) {
                            for (w -= y; C[s++] = d[x++], --y;);

                            x = s - k, S = C;
                          }
                        } else if (f < y) {
                          if (x += u + f - y, (y -= f) < w) {
                            for (w -= y; C[s++] = d[x++], --y;);

                            if (x = 0, f < w) {
                              for (w -= y = f; C[s++] = d[x++], --y;);

                              x = s - k, S = C;
                            }
                          }
                        } else if (x += f - y, y < w) {
                          for (w -= y; C[s++] = d[x++], --y;);

                          x = s - k, S = C;
                        }

                        for (; 2 < w;) C[s++] = S[x++], C[s++] = S[x++], C[s++] = S[x++], w -= 3;

                        w && (C[s++] = S[x++], 1 < w && (C[s++] = S[x++]));
                      } else {
                        for (x = s - k; C[s++] = C[x++], C[s++] = C[x++], C[s++] = C[x++], 2 < (w -= 3););

                        w && (C[s++] = C[x++], 1 < w && (C[s++] = C[x++]));
                      }

                      break;
                    }
                  }
                  break;
                }
              } while (i < n && s < o);

              i -= w = p >> 3, c &= (1 << (p -= w << 3)) - 1, t.next_in = i, t.next_out = s, t.avail_in = i < n ? n - i + 5 : 5 - (i - n), t.avail_out = s < o ? o - s + 257 : 257 - (s - o), r.hold = c, r.bits = p;
            };
          }, {}],
          49: [function (t, e, r) {

            var I = t("../utils/common"),
                O = t("./adler32"),
                B = t("./crc32"),
                R = t("./inffast"),
                T = t("./inftrees"),
                D = 1,
                F = 2,
                N = 0,
                U = -2,
                P = 1,
                i = 852,
                n = 592;

            function L(t) {
              return (t >>> 24 & 255) + (t >>> 8 & 65280) + ((65280 & t) << 8) + ((255 & t) << 24);
            }

            function s() {
              this.mode = 0, this.last = !1, this.wrap = 0, this.havedict = !1, this.flags = 0, this.dmax = 0, this.check = 0, this.total = 0, this.head = null, this.wbits = 0, this.wsize = 0, this.whave = 0, this.wnext = 0, this.window = null, this.hold = 0, this.bits = 0, this.length = 0, this.offset = 0, this.extra = 0, this.lencode = null, this.distcode = null, this.lenbits = 0, this.distbits = 0, this.ncode = 0, this.nlen = 0, this.ndist = 0, this.have = 0, this.next = null, this.lens = new I.Buf16(320), this.work = new I.Buf16(288), this.lendyn = null, this.distdyn = null, this.sane = 0, this.back = 0, this.was = 0;
            }

            function a(t) {
              var e;
              return t && t.state ? (e = t.state, t.total_in = t.total_out = e.total = 0, t.msg = "", e.wrap && (t.adler = 1 & e.wrap), e.mode = P, e.last = 0, e.havedict = 0, e.dmax = 32768, e.head = null, e.hold = 0, e.bits = 0, e.lencode = e.lendyn = new I.Buf32(i), e.distcode = e.distdyn = new I.Buf32(n), e.sane = 1, e.back = -1, N) : U;
            }

            function o(t) {
              var e;
              return t && t.state ? ((e = t.state).wsize = 0, e.whave = 0, e.wnext = 0, a(t)) : U;
            }

            function h(t, e) {
              var r, i;
              return t && t.state ? (i = t.state, e < 0 ? (r = 0, e = -e) : (r = 1 + (e >> 4), e < 48 && (e &= 15)), e && (e < 8 || 15 < e) ? U : (null !== i.window && i.wbits !== e && (i.window = null), i.wrap = r, i.wbits = e, o(t))) : U;
            }

            function u(t, e) {
              var r, i;
              return t ? (i = new s(), (t.state = i).window = null, (r = h(t, e)) !== N && (t.state = null), r) : U;
            }

            var l,
                f,
                d = !0;

            function j(t) {
              if (d) {
                var e;

                for (l = new I.Buf32(512), f = new I.Buf32(32), e = 0; e < 144;) t.lens[e++] = 8;

                for (; e < 256;) t.lens[e++] = 9;

                for (; e < 280;) t.lens[e++] = 7;

                for (; e < 288;) t.lens[e++] = 8;

                for (T(D, t.lens, 0, 288, l, 0, t.work, {
                  bits: 9
                }), e = 0; e < 32;) t.lens[e++] = 5;

                T(F, t.lens, 0, 32, f, 0, t.work, {
                  bits: 5
                }), d = !1;
              }

              t.lencode = l, t.lenbits = 9, t.distcode = f, t.distbits = 5;
            }

            function Z(t, e, r, i) {
              var n,
                  s = t.state;
              return null === s.window && (s.wsize = 1 << s.wbits, s.wnext = 0, s.whave = 0, s.window = new I.Buf8(s.wsize)), i >= s.wsize ? (I.arraySet(s.window, e, r - s.wsize, s.wsize, 0), s.wnext = 0, s.whave = s.wsize) : (i < (n = s.wsize - s.wnext) && (n = i), I.arraySet(s.window, e, r - i, n, s.wnext), (i -= n) ? (I.arraySet(s.window, e, r - i, i, 0), s.wnext = i, s.whave = s.wsize) : (s.wnext += n, s.wnext === s.wsize && (s.wnext = 0), s.whave < s.wsize && (s.whave += n))), 0;
            }

            r.inflateReset = o, r.inflateReset2 = h, r.inflateResetKeep = a, r.inflateInit = function (t) {
              return u(t, 15);
            }, r.inflateInit2 = u, r.inflate = function (t, e) {
              var r,
                  i,
                  n,
                  s,
                  a,
                  o,
                  h,
                  u,
                  l,
                  f,
                  d,
                  c,
                  p,
                  m,
                  _,
                  g,
                  b,
                  v,
                  y,
                  w,
                  k,
                  x,
                  S,
                  z,
                  C = 0,
                  E = new I.Buf8(4),
                  A = [16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15];

              if (!t || !t.state || !t.output || !t.input && 0 !== t.avail_in) return U;
              12 === (r = t.state).mode && (r.mode = 13), a = t.next_out, n = t.output, h = t.avail_out, s = t.next_in, i = t.input, o = t.avail_in, u = r.hold, l = r.bits, f = o, d = h, x = N;

              t: for (;;) switch (r.mode) {
                case P:
                  if (0 === r.wrap) {
                    r.mode = 13;
                    break;
                  }

                  for (; l < 16;) {
                    if (0 === o) break t;
                    o--, u += i[s++] << l, l += 8;
                  }

                  if (2 & r.wrap && 35615 === u) {
                    E[r.check = 0] = 255 & u, E[1] = u >>> 8 & 255, r.check = B(r.check, E, 2, 0), l = u = 0, r.mode = 2;
                    break;
                  }

                  if (r.flags = 0, r.head && (r.head.done = !1), !(1 & r.wrap) || (((255 & u) << 8) + (u >> 8)) % 31) {
                    t.msg = "incorrect header check", r.mode = 30;
                    break;
                  }

                  if (8 != (15 & u)) {
                    t.msg = "unknown compression method", r.mode = 30;
                    break;
                  }

                  if (l -= 4, k = 8 + (15 & (u >>>= 4)), 0 === r.wbits) r.wbits = k;else if (k > r.wbits) {
                    t.msg = "invalid window size", r.mode = 30;
                    break;
                  }
                  r.dmax = 1 << k, t.adler = r.check = 1, r.mode = 512 & u ? 10 : 12, l = u = 0;
                  break;

                case 2:
                  for (; l < 16;) {
                    if (0 === o) break t;
                    o--, u += i[s++] << l, l += 8;
                  }

                  if (r.flags = u, 8 != (255 & r.flags)) {
                    t.msg = "unknown compression method", r.mode = 30;
                    break;
                  }

                  if (57344 & r.flags) {
                    t.msg = "unknown header flags set", r.mode = 30;
                    break;
                  }

                  r.head && (r.head.text = u >> 8 & 1), 512 & r.flags && (E[0] = 255 & u, E[1] = u >>> 8 & 255, r.check = B(r.check, E, 2, 0)), l = u = 0, r.mode = 3;

                case 3:
                  for (; l < 32;) {
                    if (0 === o) break t;
                    o--, u += i[s++] << l, l += 8;
                  }

                  r.head && (r.head.time = u), 512 & r.flags && (E[0] = 255 & u, E[1] = u >>> 8 & 255, E[2] = u >>> 16 & 255, E[3] = u >>> 24 & 255, r.check = B(r.check, E, 4, 0)), l = u = 0, r.mode = 4;

                case 4:
                  for (; l < 16;) {
                    if (0 === o) break t;
                    o--, u += i[s++] << l, l += 8;
                  }

                  r.head && (r.head.xflags = 255 & u, r.head.os = u >> 8), 512 & r.flags && (E[0] = 255 & u, E[1] = u >>> 8 & 255, r.check = B(r.check, E, 2, 0)), l = u = 0, r.mode = 5;

                case 5:
                  if (1024 & r.flags) {
                    for (; l < 16;) {
                      if (0 === o) break t;
                      o--, u += i[s++] << l, l += 8;
                    }

                    r.length = u, r.head && (r.head.extra_len = u), 512 & r.flags && (E[0] = 255 & u, E[1] = u >>> 8 & 255, r.check = B(r.check, E, 2, 0)), l = u = 0;
                  } else r.head && (r.head.extra = null);

                  r.mode = 6;

                case 6:
                  if (1024 & r.flags && (o < (c = r.length) && (c = o), c && (r.head && (k = r.head.extra_len - r.length, r.head.extra || (r.head.extra = new Array(r.head.extra_len)), I.arraySet(r.head.extra, i, s, c, k)), 512 & r.flags && (r.check = B(r.check, i, c, s)), o -= c, s += c, r.length -= c), r.length)) break t;
                  r.length = 0, r.mode = 7;

                case 7:
                  if (2048 & r.flags) {
                    if (0 === o) break t;

                    for (c = 0; k = i[s + c++], r.head && k && r.length < 65536 && (r.head.name += String.fromCharCode(k)), k && c < o;);

                    if (512 & r.flags && (r.check = B(r.check, i, c, s)), o -= c, s += c, k) break t;
                  } else r.head && (r.head.name = null);

                  r.length = 0, r.mode = 8;

                case 8:
                  if (4096 & r.flags) {
                    if (0 === o) break t;

                    for (c = 0; k = i[s + c++], r.head && k && r.length < 65536 && (r.head.comment += String.fromCharCode(k)), k && c < o;);

                    if (512 & r.flags && (r.check = B(r.check, i, c, s)), o -= c, s += c, k) break t;
                  } else r.head && (r.head.comment = null);

                  r.mode = 9;

                case 9:
                  if (512 & r.flags) {
                    for (; l < 16;) {
                      if (0 === o) break t;
                      o--, u += i[s++] << l, l += 8;
                    }

                    if (u !== (65535 & r.check)) {
                      t.msg = "header crc mismatch", r.mode = 30;
                      break;
                    }

                    l = u = 0;
                  }

                  r.head && (r.head.hcrc = r.flags >> 9 & 1, r.head.done = !0), t.adler = r.check = 0, r.mode = 12;
                  break;

                case 10:
                  for (; l < 32;) {
                    if (0 === o) break t;
                    o--, u += i[s++] << l, l += 8;
                  }

                  t.adler = r.check = L(u), l = u = 0, r.mode = 11;

                case 11:
                  if (0 === r.havedict) return t.next_out = a, t.avail_out = h, t.next_in = s, t.avail_in = o, r.hold = u, r.bits = l, 2;
                  t.adler = r.check = 1, r.mode = 12;

                case 12:
                  if (5 === e || 6 === e) break t;

                case 13:
                  if (r.last) {
                    u >>>= 7 & l, l -= 7 & l, r.mode = 27;
                    break;
                  }

                  for (; l < 3;) {
                    if (0 === o) break t;
                    o--, u += i[s++] << l, l += 8;
                  }

                  switch (r.last = 1 & u, l -= 1, 3 & (u >>>= 1)) {
                    case 0:
                      r.mode = 14;
                      break;

                    case 1:
                      if (j(r), r.mode = 20, 6 !== e) break;
                      u >>>= 2, l -= 2;
                      break t;

                    case 2:
                      r.mode = 17;
                      break;

                    case 3:
                      t.msg = "invalid block type", r.mode = 30;
                  }

                  u >>>= 2, l -= 2;
                  break;

                case 14:
                  for (u >>>= 7 & l, l -= 7 & l; l < 32;) {
                    if (0 === o) break t;
                    o--, u += i[s++] << l, l += 8;
                  }

                  if ((65535 & u) != (u >>> 16 ^ 65535)) {
                    t.msg = "invalid stored block lengths", r.mode = 30;
                    break;
                  }

                  if (r.length = 65535 & u, l = u = 0, r.mode = 15, 6 === e) break t;

                case 15:
                  r.mode = 16;

                case 16:
                  if (c = r.length) {
                    if (o < c && (c = o), h < c && (c = h), 0 === c) break t;
                    I.arraySet(n, i, s, c, a), o -= c, s += c, h -= c, a += c, r.length -= c;
                    break;
                  }

                  r.mode = 12;
                  break;

                case 17:
                  for (; l < 14;) {
                    if (0 === o) break t;
                    o--, u += i[s++] << l, l += 8;
                  }

                  if (r.nlen = 257 + (31 & u), u >>>= 5, l -= 5, r.ndist = 1 + (31 & u), u >>>= 5, l -= 5, r.ncode = 4 + (15 & u), u >>>= 4, l -= 4, 286 < r.nlen || 30 < r.ndist) {
                    t.msg = "too many length or distance symbols", r.mode = 30;
                    break;
                  }

                  r.have = 0, r.mode = 18;

                case 18:
                  for (; r.have < r.ncode;) {
                    for (; l < 3;) {
                      if (0 === o) break t;
                      o--, u += i[s++] << l, l += 8;
                    }

                    r.lens[A[r.have++]] = 7 & u, u >>>= 3, l -= 3;
                  }

                  for (; r.have < 19;) r.lens[A[r.have++]] = 0;

                  if (r.lencode = r.lendyn, r.lenbits = 7, S = {
                    bits: r.lenbits
                  }, x = T(0, r.lens, 0, 19, r.lencode, 0, r.work, S), r.lenbits = S.bits, x) {
                    t.msg = "invalid code lengths set", r.mode = 30;
                    break;
                  }

                  r.have = 0, r.mode = 19;

                case 19:
                  for (; r.have < r.nlen + r.ndist;) {
                    for (; g = (C = r.lencode[u & (1 << r.lenbits) - 1]) >>> 16 & 255, b = 65535 & C, !((_ = C >>> 24) <= l);) {
                      if (0 === o) break t;
                      o--, u += i[s++] << l, l += 8;
                    }

                    if (b < 16) u >>>= _, l -= _, r.lens[r.have++] = b;else {
                      if (16 === b) {
                        for (z = _ + 2; l < z;) {
                          if (0 === o) break t;
                          o--, u += i[s++] << l, l += 8;
                        }

                        if (u >>>= _, l -= _, 0 === r.have) {
                          t.msg = "invalid bit length repeat", r.mode = 30;
                          break;
                        }

                        k = r.lens[r.have - 1], c = 3 + (3 & u), u >>>= 2, l -= 2;
                      } else if (17 === b) {
                        for (z = _ + 3; l < z;) {
                          if (0 === o) break t;
                          o--, u += i[s++] << l, l += 8;
                        }

                        l -= _, k = 0, c = 3 + (7 & (u >>>= _)), u >>>= 3, l -= 3;
                      } else {
                        for (z = _ + 7; l < z;) {
                          if (0 === o) break t;
                          o--, u += i[s++] << l, l += 8;
                        }

                        l -= _, k = 0, c = 11 + (127 & (u >>>= _)), u >>>= 7, l -= 7;
                      }

                      if (r.have + c > r.nlen + r.ndist) {
                        t.msg = "invalid bit length repeat", r.mode = 30;
                        break;
                      }

                      for (; c--;) r.lens[r.have++] = k;
                    }
                  }

                  if (30 === r.mode) break;

                  if (0 === r.lens[256]) {
                    t.msg = "invalid code -- missing end-of-block", r.mode = 30;
                    break;
                  }

                  if (r.lenbits = 9, S = {
                    bits: r.lenbits
                  }, x = T(D, r.lens, 0, r.nlen, r.lencode, 0, r.work, S), r.lenbits = S.bits, x) {
                    t.msg = "invalid literal/lengths set", r.mode = 30;
                    break;
                  }

                  if (r.distbits = 6, r.distcode = r.distdyn, S = {
                    bits: r.distbits
                  }, x = T(F, r.lens, r.nlen, r.ndist, r.distcode, 0, r.work, S), r.distbits = S.bits, x) {
                    t.msg = "invalid distances set", r.mode = 30;
                    break;
                  }

                  if (r.mode = 20, 6 === e) break t;

                case 20:
                  r.mode = 21;

                case 21:
                  if (6 <= o && 258 <= h) {
                    t.next_out = a, t.avail_out = h, t.next_in = s, t.avail_in = o, r.hold = u, r.bits = l, R(t, d), a = t.next_out, n = t.output, h = t.avail_out, s = t.next_in, i = t.input, o = t.avail_in, u = r.hold, l = r.bits, 12 === r.mode && (r.back = -1);
                    break;
                  }

                  for (r.back = 0; g = (C = r.lencode[u & (1 << r.lenbits) - 1]) >>> 16 & 255, b = 65535 & C, !((_ = C >>> 24) <= l);) {
                    if (0 === o) break t;
                    o--, u += i[s++] << l, l += 8;
                  }

                  if (g && 0 == (240 & g)) {
                    for (v = _, y = g, w = b; g = (C = r.lencode[w + ((u & (1 << v + y) - 1) >> v)]) >>> 16 & 255, b = 65535 & C, !(v + (_ = C >>> 24) <= l);) {
                      if (0 === o) break t;
                      o--, u += i[s++] << l, l += 8;
                    }

                    u >>>= v, l -= v, r.back += v;
                  }

                  if (u >>>= _, l -= _, r.back += _, r.length = b, 0 === g) {
                    r.mode = 26;
                    break;
                  }

                  if (32 & g) {
                    r.back = -1, r.mode = 12;
                    break;
                  }

                  if (64 & g) {
                    t.msg = "invalid literal/length code", r.mode = 30;
                    break;
                  }

                  r.extra = 15 & g, r.mode = 22;

                case 22:
                  if (r.extra) {
                    for (z = r.extra; l < z;) {
                      if (0 === o) break t;
                      o--, u += i[s++] << l, l += 8;
                    }

                    r.length += u & (1 << r.extra) - 1, u >>>= r.extra, l -= r.extra, r.back += r.extra;
                  }

                  r.was = r.length, r.mode = 23;

                case 23:
                  for (; g = (C = r.distcode[u & (1 << r.distbits) - 1]) >>> 16 & 255, b = 65535 & C, !((_ = C >>> 24) <= l);) {
                    if (0 === o) break t;
                    o--, u += i[s++] << l, l += 8;
                  }

                  if (0 == (240 & g)) {
                    for (v = _, y = g, w = b; g = (C = r.distcode[w + ((u & (1 << v + y) - 1) >> v)]) >>> 16 & 255, b = 65535 & C, !(v + (_ = C >>> 24) <= l);) {
                      if (0 === o) break t;
                      o--, u += i[s++] << l, l += 8;
                    }

                    u >>>= v, l -= v, r.back += v;
                  }

                  if (u >>>= _, l -= _, r.back += _, 64 & g) {
                    t.msg = "invalid distance code", r.mode = 30;
                    break;
                  }

                  r.offset = b, r.extra = 15 & g, r.mode = 24;

                case 24:
                  if (r.extra) {
                    for (z = r.extra; l < z;) {
                      if (0 === o) break t;
                      o--, u += i[s++] << l, l += 8;
                    }

                    r.offset += u & (1 << r.extra) - 1, u >>>= r.extra, l -= r.extra, r.back += r.extra;
                  }

                  if (r.offset > r.dmax) {
                    t.msg = "invalid distance too far back", r.mode = 30;
                    break;
                  }

                  r.mode = 25;

                case 25:
                  if (0 === h) break t;

                  if (c = d - h, r.offset > c) {
                    if ((c = r.offset - c) > r.whave && r.sane) {
                      t.msg = "invalid distance too far back", r.mode = 30;
                      break;
                    }

                    p = c > r.wnext ? (c -= r.wnext, r.wsize - c) : r.wnext - c, c > r.length && (c = r.length), m = r.window;
                  } else m = n, p = a - r.offset, c = r.length;

                  for (h < c && (c = h), h -= c, r.length -= c; n[a++] = m[p++], --c;);

                  0 === r.length && (r.mode = 21);
                  break;

                case 26:
                  if (0 === h) break t;
                  n[a++] = r.length, h--, r.mode = 21;
                  break;

                case 27:
                  if (r.wrap) {
                    for (; l < 32;) {
                      if (0 === o) break t;
                      o--, u |= i[s++] << l, l += 8;
                    }

                    if (d -= h, t.total_out += d, r.total += d, d && (t.adler = r.check = r.flags ? B(r.check, n, d, a - d) : O(r.check, n, d, a - d)), d = h, (r.flags ? u : L(u)) !== r.check) {
                      t.msg = "incorrect data check", r.mode = 30;
                      break;
                    }

                    l = u = 0;
                  }

                  r.mode = 28;

                case 28:
                  if (r.wrap && r.flags) {
                    for (; l < 32;) {
                      if (0 === o) break t;
                      o--, u += i[s++] << l, l += 8;
                    }

                    if (u !== (4294967295 & r.total)) {
                      t.msg = "incorrect length check", r.mode = 30;
                      break;
                    }

                    l = u = 0;
                  }

                  r.mode = 29;

                case 29:
                  x = 1;
                  break t;

                case 30:
                  x = -3;
                  break t;

                case 31:
                  return -4;

                case 32:
                default:
                  return U;
              }

              return t.next_out = a, t.avail_out = h, t.next_in = s, t.avail_in = o, r.hold = u, r.bits = l, (r.wsize || d !== t.avail_out && r.mode < 30 && (r.mode < 27 || 4 !== e)) && Z(t, t.output, t.next_out, d - t.avail_out) ? (r.mode = 31, -4) : (f -= t.avail_in, d -= t.avail_out, t.total_in += f, t.total_out += d, r.total += d, r.wrap && d && (t.adler = r.check = r.flags ? B(r.check, n, d, t.next_out - d) : O(r.check, n, d, t.next_out - d)), t.data_type = r.bits + (r.last ? 64 : 0) + (12 === r.mode ? 128 : 0) + (20 === r.mode || 15 === r.mode ? 256 : 0), (0 == f && 0 === d || 4 === e) && x === N && (x = -5), x);
            }, r.inflateEnd = function (t) {
              if (!t || !t.state) return U;
              var e = t.state;
              return e.window && (e.window = null), t.state = null, N;
            }, r.inflateGetHeader = function (t, e) {
              var r;
              return t && t.state ? 0 == (2 & (r = t.state).wrap) ? U : ((r.head = e).done = !1, N) : U;
            }, r.inflateSetDictionary = function (t, e) {
              var r,
                  i = e.length;
              return t && t.state ? 0 !== (r = t.state).wrap && 11 !== r.mode ? U : 11 === r.mode && O(1, e, i, 0) !== r.check ? -3 : Z(t, e, i, i) ? (r.mode = 31, -4) : (r.havedict = 1, N) : U;
            }, r.inflateInfo = "pako inflate (from Nodeca project)";
          }, {
            "../utils/common": 41,
            "./adler32": 43,
            "./crc32": 45,
            "./inffast": 48,
            "./inftrees": 50
          }],
          50: [function (t, e, r) {

            var D = t("../utils/common"),
                F = [3, 4, 5, 6, 7, 8, 9, 10, 11, 13, 15, 17, 19, 23, 27, 31, 35, 43, 51, 59, 67, 83, 99, 115, 131, 163, 195, 227, 258, 0, 0],
                N = [16, 16, 16, 16, 16, 16, 16, 16, 17, 17, 17, 17, 18, 18, 18, 18, 19, 19, 19, 19, 20, 20, 20, 20, 21, 21, 21, 21, 16, 72, 78],
                U = [1, 2, 3, 4, 5, 7, 9, 13, 17, 25, 33, 49, 65, 97, 129, 193, 257, 385, 513, 769, 1025, 1537, 2049, 3073, 4097, 6145, 8193, 12289, 16385, 24577, 0, 0],
                P = [16, 16, 16, 16, 17, 17, 18, 18, 19, 19, 20, 20, 21, 21, 22, 22, 23, 23, 24, 24, 25, 25, 26, 26, 27, 27, 28, 28, 29, 29, 64, 64];

            e.exports = function (t, e, r, i, n, s, a, o) {
              var h,
                  u,
                  l,
                  f,
                  d,
                  c,
                  p,
                  m,
                  _,
                  g = o.bits,
                  b = 0,
                  v = 0,
                  y = 0,
                  w = 0,
                  k = 0,
                  x = 0,
                  S = 0,
                  z = 0,
                  C = 0,
                  E = 0,
                  A = null,
                  I = 0,
                  O = new D.Buf16(16),
                  B = new D.Buf16(16),
                  R = null,
                  T = 0;

              for (b = 0; b <= 15; b++) O[b] = 0;

              for (v = 0; v < i; v++) O[e[r + v]]++;

              for (k = g, w = 15; 1 <= w && 0 === O[w]; w--);

              if (w < k && (k = w), 0 === w) return n[s++] = 20971520, n[s++] = 20971520, o.bits = 1, 0;

              for (y = 1; y < w && 0 === O[y]; y++);

              for (k < y && (k = y), b = z = 1; b <= 15; b++) if (z <<= 1, (z -= O[b]) < 0) return -1;

              if (0 < z && (0 === t || 1 !== w)) return -1;

              for (B[1] = 0, b = 1; b < 15; b++) B[b + 1] = B[b] + O[b];

              for (v = 0; v < i; v++) 0 !== e[r + v] && (a[B[e[r + v]]++] = v);

              if (c = 0 === t ? (A = R = a, 19) : 1 === t ? (A = F, I -= 257, R = N, T -= 257, 256) : (A = U, R = P, -1), b = y, d = s, S = v = E = 0, l = -1, f = (C = 1 << (x = k)) - 1, 1 === t && 852 < C || 2 === t && 592 < C) return 1;

              for (;;) {
                for (p = b - S, _ = a[v] < c ? (m = 0, a[v]) : a[v] > c ? (m = R[T + a[v]], A[I + a[v]]) : (m = 96, 0), h = 1 << b - S, y = u = 1 << x; n[d + (E >> S) + (u -= h)] = p << 24 | m << 16 | _ | 0, 0 !== u;);

                for (h = 1 << b - 1; E & h;) h >>= 1;

                if (0 !== h ? (E &= h - 1, E += h) : E = 0, v++, 0 == --O[b]) {
                  if (b === w) break;
                  b = e[r + a[v]];
                }

                if (k < b && (E & f) !== l) {
                  for (0 === S && (S = k), d += y, z = 1 << (x = b - S); x + S < w && !((z -= O[x + S]) <= 0);) x++, z <<= 1;

                  if (C += 1 << x, 1 === t && 852 < C || 2 === t && 592 < C) return 1;
                  n[l = E & f] = k << 24 | x << 16 | d - s | 0;
                }
              }

              return 0 !== E && (n[d + E] = b - S << 24 | 64 << 16 | 0), o.bits = k, 0;
            };
          }, {
            "../utils/common": 41
          }],
          51: [function (t, e, r) {

            e.exports = {
              2: "need dictionary",
              1: "stream end",
              0: "",
              "-1": "file error",
              "-2": "stream error",
              "-3": "data error",
              "-4": "insufficient memory",
              "-5": "buffer error",
              "-6": "incompatible version"
            };
          }, {}],
          52: [function (t, e, r) {

            var n = t("../utils/common"),
                o = 0,
                h = 1;

            function i(t) {
              for (var e = t.length; 0 <= --e;) t[e] = 0;
            }

            var s = 0,
                a = 29,
                u = 256,
                l = u + 1 + a,
                f = 30,
                d = 19,
                _ = 2 * l + 1,
                g = 15,
                c = 16,
                p = 7,
                m = 256,
                b = 16,
                v = 17,
                y = 18,
                w = [0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 5, 5, 5, 5, 0],
                k = [0, 0, 0, 0, 1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9, 10, 10, 11, 11, 12, 12, 13, 13],
                x = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 3, 7],
                S = [16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15],
                z = new Array(2 * (l + 2));

            i(z);
            var C = new Array(2 * f);
            i(C);
            var E = new Array(512);
            i(E);
            var A = new Array(256);
            i(A);
            var I = new Array(a);
            i(I);
            var O,
                B,
                R,
                T = new Array(f);

            function D(t, e, r, i, n) {
              this.static_tree = t, this.extra_bits = e, this.extra_base = r, this.elems = i, this.max_length = n, this.has_stree = t && t.length;
            }

            function F(t, e) {
              this.dyn_tree = t, this.max_code = 0, this.stat_desc = e;
            }

            function N(t) {
              return t < 256 ? E[t] : E[256 + (t >>> 7)];
            }

            function U(t, e) {
              t.pending_buf[t.pending++] = 255 & e, t.pending_buf[t.pending++] = e >>> 8 & 255;
            }

            function P(t, e, r) {
              t.bi_valid > c - r ? (t.bi_buf |= e << t.bi_valid & 65535, U(t, t.bi_buf), t.bi_buf = e >> c - t.bi_valid, t.bi_valid += r - c) : (t.bi_buf |= e << t.bi_valid & 65535, t.bi_valid += r);
            }

            function L(t, e, r) {
              P(t, r[2 * e], r[2 * e + 1]);
            }

            function j(t, e) {
              for (var r = 0; r |= 1 & t, t >>>= 1, r <<= 1, 0 < --e;);

              return r >>> 1;
            }

            function Z(t, e, r) {
              var i,
                  n,
                  s = new Array(g + 1),
                  a = 0;

              for (i = 1; i <= g; i++) s[i] = a = a + r[i - 1] << 1;

              for (n = 0; n <= e; n++) {
                var o = t[2 * n + 1];
                0 !== o && (t[2 * n] = j(s[o]++, o));
              }
            }

            function W(t) {
              var e;

              for (e = 0; e < l; e++) t.dyn_ltree[2 * e] = 0;

              for (e = 0; e < f; e++) t.dyn_dtree[2 * e] = 0;

              for (e = 0; e < d; e++) t.bl_tree[2 * e] = 0;

              t.dyn_ltree[2 * m] = 1, t.opt_len = t.static_len = 0, t.last_lit = t.matches = 0;
            }

            function M(t) {
              8 < t.bi_valid ? U(t, t.bi_buf) : 0 < t.bi_valid && (t.pending_buf[t.pending++] = t.bi_buf), t.bi_buf = 0, t.bi_valid = 0;
            }

            function H(t, e, r, i) {
              var n = 2 * e,
                  s = 2 * r;
              return t[n] < t[s] || t[n] === t[s] && i[e] <= i[r];
            }

            function G(t, e, r) {
              for (var i = t.heap[r], n = r << 1; n <= t.heap_len && (n < t.heap_len && H(e, t.heap[n + 1], t.heap[n], t.depth) && n++, !H(e, i, t.heap[n], t.depth));) t.heap[r] = t.heap[n], r = n, n <<= 1;

              t.heap[r] = i;
            }

            function K(t, e, r) {
              var i,
                  n,
                  s,
                  a,
                  o = 0;
              if (0 !== t.last_lit) for (; i = t.pending_buf[t.d_buf + 2 * o] << 8 | t.pending_buf[t.d_buf + 2 * o + 1], n = t.pending_buf[t.l_buf + o], o++, 0 === i ? L(t, n, e) : (L(t, (s = A[n]) + u + 1, e), 0 !== (a = w[s]) && P(t, n -= I[s], a), L(t, s = N(--i), r), 0 !== (a = k[s]) && P(t, i -= T[s], a)), o < t.last_lit;);
              L(t, m, e);
            }

            function Y(t, e) {
              var r,
                  i,
                  n,
                  s = e.dyn_tree,
                  a = e.stat_desc.static_tree,
                  o = e.stat_desc.has_stree,
                  h = e.stat_desc.elems,
                  u = -1;

              for (t.heap_len = 0, t.heap_max = _, r = 0; r < h; r++) 0 !== s[2 * r] ? (t.heap[++t.heap_len] = u = r, t.depth[r] = 0) : s[2 * r + 1] = 0;

              for (; t.heap_len < 2;) s[2 * (n = t.heap[++t.heap_len] = u < 2 ? ++u : 0)] = 1, t.depth[n] = 0, t.opt_len--, o && (t.static_len -= a[2 * n + 1]);

              for (e.max_code = u, r = t.heap_len >> 1; 1 <= r; r--) G(t, s, r);

              for (n = h; r = t.heap[1], t.heap[1] = t.heap[t.heap_len--], G(t, s, 1), i = t.heap[1], t.heap[--t.heap_max] = r, t.heap[--t.heap_max] = i, s[2 * n] = s[2 * r] + s[2 * i], t.depth[n] = (t.depth[r] >= t.depth[i] ? t.depth[r] : t.depth[i]) + 1, s[2 * r + 1] = s[2 * i + 1] = n, t.heap[1] = n++, G(t, s, 1), 2 <= t.heap_len;);

              t.heap[--t.heap_max] = t.heap[1], function (t, e) {
                var r,
                    i,
                    n,
                    s,
                    a,
                    o,
                    h = e.dyn_tree,
                    u = e.max_code,
                    l = e.stat_desc.static_tree,
                    f = e.stat_desc.has_stree,
                    d = e.stat_desc.extra_bits,
                    c = e.stat_desc.extra_base,
                    p = e.stat_desc.max_length,
                    m = 0;

                for (s = 0; s <= g; s++) t.bl_count[s] = 0;

                for (h[2 * t.heap[t.heap_max] + 1] = 0, r = t.heap_max + 1; r < _; r++) p < (s = h[2 * h[2 * (i = t.heap[r]) + 1] + 1] + 1) && (s = p, m++), h[2 * i + 1] = s, u < i || (t.bl_count[s]++, a = 0, c <= i && (a = d[i - c]), o = h[2 * i], t.opt_len += o * (s + a), f && (t.static_len += o * (l[2 * i + 1] + a)));

                if (0 !== m) {
                  do {
                    for (s = p - 1; 0 === t.bl_count[s];) s--;

                    t.bl_count[s]--, t.bl_count[s + 1] += 2, t.bl_count[p]--, m -= 2;
                  } while (0 < m);

                  for (s = p; 0 !== s; s--) for (i = t.bl_count[s]; 0 !== i;) u < (n = t.heap[--r]) || (h[2 * n + 1] !== s && (t.opt_len += (s - h[2 * n + 1]) * h[2 * n], h[2 * n + 1] = s), i--);
                }
              }(t, e), Z(s, u, t.bl_count);
            }

            function X(t, e, r) {
              var i,
                  n,
                  s = -1,
                  a = e[1],
                  o = 0,
                  h = 7,
                  u = 4;

              for (0 === a && (h = 138, u = 3), e[2 * (r + 1) + 1] = 65535, i = 0; i <= r; i++) n = a, a = e[2 * (i + 1) + 1], ++o < h && n === a || (o < u ? t.bl_tree[2 * n] += o : 0 !== n ? (n !== s && t.bl_tree[2 * n]++, t.bl_tree[2 * b]++) : o <= 10 ? t.bl_tree[2 * v]++ : t.bl_tree[2 * y]++, s = n, u = (o = 0) === a ? (h = 138, 3) : n === a ? (h = 6, 3) : (h = 7, 4));
            }

            function V(t, e, r) {
              var i,
                  n,
                  s = -1,
                  a = e[1],
                  o = 0,
                  h = 7,
                  u = 4;

              for (0 === a && (h = 138, u = 3), i = 0; i <= r; i++) if (n = a, a = e[2 * (i + 1) + 1], !(++o < h && n === a)) {
                if (o < u) for (; L(t, n, t.bl_tree), 0 != --o;);else 0 !== n ? (n !== s && (L(t, n, t.bl_tree), o--), L(t, b, t.bl_tree), P(t, o - 3, 2)) : o <= 10 ? (L(t, v, t.bl_tree), P(t, o - 3, 3)) : (L(t, y, t.bl_tree), P(t, o - 11, 7));
                s = n, u = (o = 0) === a ? (h = 138, 3) : n === a ? (h = 6, 3) : (h = 7, 4);
              }
            }

            i(T);
            var q = !1;

            function J(t, e, r, i) {
              P(t, (s << 1) + (i ? 1 : 0), 3), function (t, e, r, i) {
                M(t), i && (U(t, r), U(t, ~r)), n.arraySet(t.pending_buf, t.window, e, r, t.pending), t.pending += r;
              }(t, e, r, !0);
            }

            r._tr_init = function (t) {
              q || (function () {
                var t,
                    e,
                    r,
                    i,
                    n,
                    s = new Array(g + 1);

                for (i = r = 0; i < a - 1; i++) for (I[i] = r, t = 0; t < 1 << w[i]; t++) A[r++] = i;

                for (A[r - 1] = i, i = n = 0; i < 16; i++) for (T[i] = n, t = 0; t < 1 << k[i]; t++) E[n++] = i;

                for (n >>= 7; i < f; i++) for (T[i] = n << 7, t = 0; t < 1 << k[i] - 7; t++) E[256 + n++] = i;

                for (e = 0; e <= g; e++) s[e] = 0;

                for (t = 0; t <= 143;) z[2 * t + 1] = 8, t++, s[8]++;

                for (; t <= 255;) z[2 * t + 1] = 9, t++, s[9]++;

                for (; t <= 279;) z[2 * t + 1] = 7, t++, s[7]++;

                for (; t <= 287;) z[2 * t + 1] = 8, t++, s[8]++;

                for (Z(z, l + 1, s), t = 0; t < f; t++) C[2 * t + 1] = 5, C[2 * t] = j(t, 5);

                O = new D(z, w, u + 1, l, g), B = new D(C, k, 0, f, g), R = new D(new Array(0), x, 0, d, p);
              }(), q = !0), t.l_desc = new F(t.dyn_ltree, O), t.d_desc = new F(t.dyn_dtree, B), t.bl_desc = new F(t.bl_tree, R), t.bi_buf = 0, t.bi_valid = 0, W(t);
            }, r._tr_stored_block = J, r._tr_flush_block = function (t, e, r, i) {
              var n,
                  s,
                  a = 0;
              0 < t.level ? (2 === t.strm.data_type && (t.strm.data_type = function (t) {
                var e,
                    r = 4093624447;

                for (e = 0; e <= 31; e++, r >>>= 1) if (1 & r && 0 !== t.dyn_ltree[2 * e]) return o;

                if (0 !== t.dyn_ltree[18] || 0 !== t.dyn_ltree[20] || 0 !== t.dyn_ltree[26]) return h;

                for (e = 32; e < u; e++) if (0 !== t.dyn_ltree[2 * e]) return h;

                return o;
              }(t)), Y(t, t.l_desc), Y(t, t.d_desc), a = function (t) {
                var e;

                for (X(t, t.dyn_ltree, t.l_desc.max_code), X(t, t.dyn_dtree, t.d_desc.max_code), Y(t, t.bl_desc), e = d - 1; 3 <= e && 0 === t.bl_tree[2 * S[e] + 1]; e--);

                return t.opt_len += 3 * (e + 1) + 5 + 5 + 4, e;
              }(t), n = t.opt_len + 3 + 7 >>> 3, (s = t.static_len + 3 + 7 >>> 3) <= n && (n = s)) : n = s = r + 5, r + 4 <= n && -1 !== e ? J(t, e, r, i) : 4 === t.strategy || s === n ? (P(t, 2 + (i ? 1 : 0), 3), K(t, z, C)) : (P(t, 4 + (i ? 1 : 0), 3), function (t, e, r, i) {
                var n;

                for (P(t, e - 257, 5), P(t, r - 1, 5), P(t, i - 4, 4), n = 0; n < i; n++) P(t, t.bl_tree[2 * S[n] + 1], 3);

                V(t, t.dyn_ltree, e - 1), V(t, t.dyn_dtree, r - 1);
              }(t, t.l_desc.max_code + 1, t.d_desc.max_code + 1, a + 1), K(t, t.dyn_ltree, t.dyn_dtree)), W(t), i && M(t);
            }, r._tr_tally = function (t, e, r) {
              return t.pending_buf[t.d_buf + 2 * t.last_lit] = e >>> 8 & 255, t.pending_buf[t.d_buf + 2 * t.last_lit + 1] = 255 & e, t.pending_buf[t.l_buf + t.last_lit] = 255 & r, t.last_lit++, 0 === e ? t.dyn_ltree[2 * r]++ : (t.matches++, e--, t.dyn_ltree[2 * (A[r] + u + 1)]++, t.dyn_dtree[2 * N(e)]++), t.last_lit === t.lit_bufsize - 1;
            }, r._tr_align = function (t) {
              P(t, 2, 3), L(t, m, z), function (t) {
                16 === t.bi_valid ? (U(t, t.bi_buf), t.bi_buf = 0, t.bi_valid = 0) : 8 <= t.bi_valid && (t.pending_buf[t.pending++] = 255 & t.bi_buf, t.bi_buf >>= 8, t.bi_valid -= 8);
              }(t);
            };
          }, {
            "../utils/common": 41
          }],
          53: [function (t, e, r) {

            e.exports = function () {
              this.input = null, this.next_in = 0, this.avail_in = 0, this.total_in = 0, this.output = null, this.next_out = 0, this.avail_out = 0, this.total_out = 0, this.msg = "", this.state = null, this.data_type = 2, this.adler = 0;
            };
          }, {}],
          54: [function (t, e, r) {

            e.exports = "function" == typeof setImmediate ? setImmediate : function () {
              var t = [].slice.apply(arguments);
              t.splice(1, 0, 0), setTimeout.apply(null, t);
            };
          }, {}]
        }, {}, [10])(10);
      });
    })(jszip_min);

    var JSZip = jszip_min.exports;

    /**
     * Those functions should disappear if add2D becomes accessible in jcampconvert
     * @param spectra
     * @returns {{z: Array, minX: *, maxX: *, minY: *, maxY: *, minZ: *, maxZ: *, noise: number}}
     */
    function convertTo3DZ$1(spectra) {
      let noise = 0;
      let minZ = spectra[0].data[0];
      let maxZ = minZ;
      let ySize = spectra.length;
      let xSize = spectra[0].data.length / 2;
      let z = new Array(ySize);

      for (let i = 0; i < ySize; i++) {
        z[i] = new Float64Array(xSize);

        for (let j = 0; j < xSize; j++) {
          z[i][j] = spectra[i].data[j * 2 + 1];
          if (z[i][j] < minZ) minZ = spectra[i].data[j * 2 + 1];
          if (z[i][j] > maxZ) maxZ = spectra[i].data[j * 2 + 1];

          if (i !== 0 && j !== 0) {
            noise += Math.abs(z[i][j] - z[i][j - 1]) + Math.abs(z[i][j] - z[i - 1][j]);
          }
        }
      }

      const firstX = spectra[0].data[0];
      const lastX = spectra[0].data[spectra[0].data.length - 2]; // has to be -2 because it is a 1D array [x,y,x,y,...]

      const firstY = spectra[0].pageValue;
      const lastY = spectra[ySize - 1].pageValue; // Because the min / max value are the only information about the matrix if we invert
      // min and max we need to invert the array

      if (firstX > lastX) {
        for (let spectrum of z) {
          spectrum.reverse();
        }
      }

      if (firstY > lastY) {
        z.reverse();
      }

      let minMaxX = firstX < lastX ? {
        minX: firstX,
        maxX: lastX
      } : {
        minX: lastX,
        maxX: firstX
      };
      let minMaxY = firstY < lastY ? {
        minY: firstY,
        maxY: lastY
      } : {
        minY: lastY,
        maxY: firstY
      };
      return {
        z,
        minZ: minZ,
        maxZ: maxZ,
        ...minMaxX,
        ...minMaxY,
        noise: noise / ((ySize - 1) * (xSize - 1) * 2)
      };
    }

    function generateContourLines$1(zData) {
      let noise = zData.noise;
      let z = zData.z;
      let contourLevels = [];
      let nbLevels = 7;
      let povarHeight = new Float32Array(4);
      let isOver = [];
      let nbSubSpectra = z.length;
      let nbPovars = z[0].length;
      let pAx, pAy, pBx, pBy;
      let x0 = zData.minX;
      let xN = zData.maxX;
      let dx = (xN - x0) / (nbPovars - 1);
      let y0 = zData.minY;
      let yN = zData.maxY;
      let dy = (yN - y0) / (nbSubSpectra - 1);
      let minZ = zData.minZ;
      let maxZ = zData.maxZ; // System.out.prvarln('y0 '+y0+' yN '+yN);
      // -------------------------
      // Povars attribution
      //
      // 0----1
      // |  / |
      // | /  |
      // 2----3
      //
      // ---------------------d------

      let lineZValue;

      for (let level = 0; level < nbLevels * 2; level++) {
        // multiply by 2 for positif and negatif
        let contourLevel = {};
        contourLevels.push(contourLevel);
        let side = level % 2;

        if (side === 0) {
          lineZValue = (maxZ - 5 * noise) * Math.exp(level / 2 - nbLevels) + 5 * noise;
        } else {
          lineZValue = -(maxZ - 5 * noise) * Math.exp(level / 2 - nbLevels) - 5 * noise;
        }

        let lines = [];
        contourLevel.zValue = lineZValue;
        contourLevel.lines = lines;
        if (lineZValue <= minZ || lineZValue >= maxZ) continue;

        for (let iSubSpectra = 0; iSubSpectra < nbSubSpectra - 1; iSubSpectra++) {
          for (let povar = 0; povar < nbPovars - 1; povar++) {
            povarHeight[0] = z[iSubSpectra][povar];
            povarHeight[1] = z[iSubSpectra][povar + 1];
            povarHeight[2] = z[iSubSpectra + 1][povar];
            povarHeight[3] = z[iSubSpectra + 1][povar + 1];

            for (let i = 0; i < 4; i++) {
              isOver[i] = povarHeight[i] > lineZValue;
            } // Example povar0 is over the plane and povar1 and
            // povar2 are below, we find the varersections and add
            // the segment


            if (isOver[0] !== isOver[1] && isOver[0] !== isOver[2]) {
              pAx = povar + (lineZValue - povarHeight[0]) / (povarHeight[1] - povarHeight[0]);
              pAy = iSubSpectra;
              pBx = povar;
              pBy = iSubSpectra + (lineZValue - povarHeight[0]) / (povarHeight[2] - povarHeight[0]);
              lines.push(pAx * dx + x0, pAy * dy + y0, pBx * dx + x0, pBy * dy + y0);
            }

            if (isOver[3] !== isOver[1] && isOver[3] !== isOver[2]) {
              pAx = povar + 1;
              pAy = iSubSpectra + 1 - (lineZValue - povarHeight[3]) / (povarHeight[1] - povarHeight[3]);
              pBx = povar + 1 - (lineZValue - povarHeight[3]) / (povarHeight[2] - povarHeight[3]);
              pBy = iSubSpectra + 1;
              lines.push(pAx * dx + x0, pAy * dy + y0, pBx * dx + x0, pBy * dy + y0);
            } // test around the diagonal


            if (isOver[1] !== isOver[2]) {
              pAx = povar + 1 - (lineZValue - povarHeight[1]) / (povarHeight[2] - povarHeight[1]);
              pAy = iSubSpectra + (lineZValue - povarHeight[1]) / (povarHeight[2] - povarHeight[1]);

              if (isOver[1] !== isOver[0]) {
                pBx = povar + 1 - (lineZValue - povarHeight[1]) / (povarHeight[0] - povarHeight[1]);
                pBy = iSubSpectra;
                lines.push(pAx * dx + x0, pAy * dy + y0, pBx * dx + x0, pBy * dy + y0);
              }

              if (isOver[2] !== isOver[0]) {
                pBx = povar;
                pBy = iSubSpectra + 1 - (lineZValue - povarHeight[2]) / (povarHeight[0] - povarHeight[2]);
                lines.push(pAx * dx + x0, pAy * dy + y0, pBx * dx + x0, pBy * dy + y0);
              }

              if (isOver[1] !== isOver[3]) {
                pBx = povar + 1;
                pBy = iSubSpectra + (lineZValue - povarHeight[1]) / (povarHeight[3] - povarHeight[1]);
                lines.push(pAx * dx + x0, pAy * dy + y0, pBx * dx + x0, pBy * dy + y0);
              }

              if (isOver[2] !== isOver[3]) {
                pBx = povar + (lineZValue - povarHeight[2]) / (povarHeight[3] - povarHeight[2]);
                pBy = iSubSpectra + 1;
                lines.push(pAx * dx + x0, pAy * dy + y0, pBx * dx + x0, pBy * dy + y0);
              }
            }
          }
        }
      }

      return {
        minX: zData.minX,
        maxX: zData.maxX,
        minY: zData.minY,
        maxY: zData.maxY,
        segments: contourLevels
      };
    }

    function add2D$1(result, options) {
      let zData = convertTo3DZ$1(result.spectra);

      if (!options.noContours) {
        result.contourLines = generateContourLines$1(zData);
      }

      result.minMax = zData;
    }

    /**
     * Dynamically type a string
     * @param {string} value String to dynamically type
     * @returns {boolean|string|number}
     */
    function parseString(value) {
      if (value.length === 4 || value.length === 5) {
        let lowercase = value.toLowerCase();
        if (lowercase === 'true') return true;
        if (lowercase === 'false') return false;
      }

      let number = Number(value);

      if (number === 0 && !value.includes('0')) {
        return value;
      }

      if (!Number.isNaN(number)) return number;
      return value;
    }

    const GC_MS_FIELDS = ['TIC', '.RIC', 'SCANNUMBER'];
    function complexChromatogram(result) {
      let spectra = result.spectra;
      let length = spectra.length;
      let chromatogram = {
        times: new Array(length),
        series: {
          ms: {
            dimension: 2,
            data: new Array(length)
          }
        }
      };
      let existingGCMSFields = [];

      for (let i = 0; i < GC_MS_FIELDS.length; i++) {
        let label = convertMSFieldToLabel(GC_MS_FIELDS[i]);

        if (spectra[0][label]) {
          existingGCMSFields.push(label);
          chromatogram.series[label] = {
            dimension: 1,
            data: new Array(length)
          };
        }
      }

      for (let i = 0; i < length; i++) {
        let spectrum = spectra[i];
        chromatogram.times[i] = spectrum.pageValue;

        for (let j = 0; j < existingGCMSFields.length; j++) {
          chromatogram.series[existingGCMSFields[j]].data[i] = Number(spectrum[existingGCMSFields[j]]);
        }

        if (spectrum.data) {
          chromatogram.series.ms.data[i] = [spectrum.data.x, spectrum.data.y];
        }
      }

      result.chromatogram = chromatogram;
    }
    function isMSField(canonicDataLabel) {
      return GC_MS_FIELDS.indexOf(canonicDataLabel) !== -1;
    }
    function convertMSFieldToLabel(value) {
      return value.toLowerCase().replace(/[^a-z0-9]/g, '');
    }

    function convertToFloatArray$1(stringArray) {
      let floatArray = [];

      for (let i = 0; i < stringArray.length; i++) {
        floatArray.push(Number(stringArray[i]));
      }

      return floatArray;
    }

    function fastParseXYData(spectrum, value) {
      // TODO need to deal with result
      //  console.log(value);
      // we check if deltaX is defined otherwise we calculate it
      let yFactor = spectrum.yFactor;
      let deltaX = spectrum.deltaX;
      spectrum.isXYdata = true;
      let currentData = {
        x: [],
        y: []
      };
      spectrum.data = currentData;
      let currentX = spectrum.firstX;
      let currentY = spectrum.firstY; // we skip the first line
      //

      let endLine = false;
      let ascii;
      let i = 0;

      for (; i < value.length; i++) {
        ascii = value.charCodeAt(i);

        if (ascii === 13 || ascii === 10) {
          endLine = true;
        } else {
          if (endLine) break;
        }
      } // we proceed taking the i after the first line


      let newLine = true;
      let isDifference = false;
      let isLastDifference = false;
      let lastDifference = 0;
      let isDuplicate = false;
      let inComment = false;
      let currentValue = 0; // can be a difference or a duplicate

      let lastValue = 0; // must be the real last value

      let isNegative = false;
      let inValue = false;
      let skipFirstValue = false;
      let decimalPosition = 0;

      for (; i <= value.length; i++) {
        if (i === value.length) ascii = 13;else ascii = value.charCodeAt(i);

        if (inComment) {
          // we should ignore the text if we are after $$
          if (ascii === 13 || ascii === 10) {
            newLine = true;
            inComment = false;
          }
        } else {
          // when is it a new value ?
          // when it is not a digit, . or comma
          // it is a number that is either new or we continue
          if (ascii <= 57 && ascii >= 48) {
            // a number
            inValue = true;

            if (decimalPosition > 0) {
              currentValue += (ascii - 48) / Math.pow(10, decimalPosition++);
            } else {
              currentValue *= 10;
              currentValue += ascii - 48;
            }
          } else if (ascii === 44 || ascii === 46) {
            // a "," or "."
            inValue = true;
            decimalPosition++;
          } else {
            if (inValue) {
              // need to process the previous value
              if (newLine) {
                newLine = false; // we don't check the X value
                // console.log("NEW LINE",isDifference, lastDifference);
                // if new line and lastDifference, the first value is just a check !
                // that we don't check ...

                if (isLastDifference) skipFirstValue = true;
              } else {
                // need to deal with duplicate and differences
                if (skipFirstValue) {
                  skipFirstValue = false;
                } else {
                  if (isDifference) {
                    lastDifference = isNegative ? 0 - currentValue : currentValue;
                    isLastDifference = true;
                    isDifference = false;
                  } else if (!isDuplicate) {
                    lastValue = isNegative ? 0 - currentValue : currentValue;
                  }

                  let duplicate = isDuplicate ? currentValue - 1 : 1;

                  for (let j = 0; j < duplicate; j++) {
                    if (isLastDifference) {
                      currentY += lastDifference;
                    } else {
                      currentY = lastValue;
                    }

                    currentData.x.push(currentX);
                    currentData.y.push(currentY * yFactor);
                    currentX += deltaX;
                  }
                }
              }

              isNegative = false;
              currentValue = 0;
              decimalPosition = 0;
              inValue = false;
              isDuplicate = false;
            } // positive SQZ digits @ A B C D E F G H I (ascii 64-73)


            if (ascii < 74 && ascii > 63) {
              inValue = true;
              isLastDifference = false;
              currentValue = ascii - 64;
            } else if (ascii > 96 && ascii < 106) {
              // negative SQZ digits a b c d e f g h i (ascii 97-105)
              inValue = true;
              isLastDifference = false;
              currentValue = ascii - 96;
              isNegative = true;
            } else if (ascii === 115) {
              // DUP digits S T U V W X Y Z s (ascii 83-90, 115)
              inValue = true;
              isDuplicate = true;
              currentValue = 9;
            } else if (ascii > 82 && ascii < 91) {
              inValue = true;
              isDuplicate = true;
              currentValue = ascii - 82;
            } else if (ascii > 73 && ascii < 83) {
              // positive DIF digits % J K L M N O P Q R (ascii 37, 74-82)
              inValue = true;
              isDifference = true;
              currentValue = ascii - 73;
            } else if (ascii > 105 && ascii < 115) {
              // negative DIF digits j k l m n o p q r (ascii 106-114)
              inValue = true;
              isDifference = true;
              currentValue = ascii - 105;
              isNegative = true;
            } else if (ascii === 36 && value.charCodeAt(i + 1) === 36) {
              // $ sign, we need to check the next one
              inValue = true;
              inComment = true;
            } else if (ascii === 37) {
              // positive DIF digits % J K L M N O P Q R (ascii 37, 74-82)
              inValue = true;
              isDifference = true;
              currentValue = 0;
              isNegative = false;
            } else if (ascii === 45) {
              // a "-"
              // check if after there is a number, decimal or comma
              let ascii2 = value.charCodeAt(i + 1);

              if (ascii2 >= 48 && ascii2 <= 57 || ascii2 === 44 || ascii2 === 46) {
                inValue = true;
                if (!newLine) isLastDifference = false;
                isNegative = true;
              }
            } else if (ascii === 13 || ascii === 10) {
              newLine = true;
              inComment = false;
            } // and now analyse the details ... space or tabulation
            // if "+" we just don't care

          }
        }
      }
    }

    const removeCommentRegExp = /\$\$.*/;
    const peakTableSplitRegExp = /[,\t ]+/;
    function parsePeakTable(spectrum, value, result) {
      spectrum.isPeaktable = true;

      if (!spectrum.variables || Object.keys(spectrum.variables) === 2) {
        parseXY(spectrum, value, result);
      } else {
        parseXYZ(spectrum, value, result);
      } // we will add the data in the variables


      if (spectrum.variables) {
        for (let key in spectrum.variables) {
          spectrum.variables[key].data = spectrum.data[key];
        }
      }
    }

    function parseXY(spectrum, value, result) {
      let currentData = {
        x: [],
        y: []
      };
      spectrum.data = currentData; // counts for around 20% of the time

      let lines = value.split(/,? *,?[;\r\n]+ */);

      for (let i = 1; i < lines.length; i++) {
        let values = lines[i].trim().replace(removeCommentRegExp, '').split(peakTableSplitRegExp);

        if (values.length % 2 === 0) {
          for (let j = 0; j < values.length; j = j + 2) {
            // takes around 40% of the time to add and parse the 2 values nearly exclusively because of Number
            currentData.x.push(Number(values[j]) * spectrum.xFactor);
            currentData.y.push(Number(values[j + 1]) * spectrum.yFactor);
          }
        } else {
          result.logs.push(`Format error: ${values}`);
        }
      }
    }

    function parseXYZ(spectrum, value, result) {
      let currentData = {};
      let variables = Object.keys(spectrum.variables);
      let numberOfVariables = variables.length;
      variables.forEach(variable => currentData[variable] = []);
      spectrum.data = currentData; // counts for around 20% of the time

      let lines = value.split(/,? *,?[;\r\n]+ */);

      for (let i = 1; i < lines.length; i++) {
        let values = lines[i].trim().replace(removeCommentRegExp, '').split(peakTableSplitRegExp);

        if (values.length % numberOfVariables === 0) {
          for (let j = 0; j < values.length; j++) {
            // todo should try to find a xFactor (y, ...)
            currentData[variables[j % numberOfVariables]].push(Number(values[j]));
          }
        } else {
          result.logs.push(`Format error: ${values}`);
        }
      }
    }

    function parseXYA(spectrum, value) {
      let removeSymbolRegExp = /(\(+|\)+|<+|>+|\s+)/g;
      spectrum.isXYAdata = true;
      let values;
      let currentData = {
        x: [],
        y: []
      };
      spectrum.data = currentData;
      let lines = value.split(/,? *,?[;\r\n]+ */);

      for (let i = 1; i < lines.length; i++) {
        values = lines[i].trim().replace(removeSymbolRegExp, '').split(',');
        currentData.x.push(Number(values[0]));
        currentData.y.push(Number(values[1]));
      }
    }

    function convertTo3DZ(spectra) {
      let minZ = spectra[0].data.y[0];
      let maxZ = minZ;
      let ySize = spectra.length;
      let xSize = spectra[0].data.x.length;
      let z = new Array(ySize);

      for (let i = 0; i < ySize; i++) {
        z[i] = spectra[i].data.y;

        for (let j = 0; j < xSize; j++) {
          let value = z[i][j];
          if (value < minZ) minZ = value;
          if (value > maxZ) maxZ = value;
        }
      }

      const firstX = spectra[0].data.x[0];
      const lastX = spectra[0].data.x[spectra[0].data.x.length - 1]; // has to be -2 because it is a 1D array [x,y,x,y,...]

      const firstY = spectra[0].pageValue;
      const lastY = spectra[ySize - 1].pageValue; // Because the min / max value are the only information about the matrix if we invert
      // min and max we need to invert the array

      if (firstX > lastX) {
        for (let spectrum of z) {
          spectrum.reverse();
        }
      }

      if (firstY > lastY) {
        z.reverse();
      }

      const medians = [];

      for (let i = 0; i < z.length; i++) {
        const row = Float64Array.from(z[i]);

        for (let i = 0; i < row.length; i++) {
          if (row[i] < 0) row[i] = -row[i];
        }

        medians.push(median(row));
      }

      const median$1 = median(medians);
      return {
        z: z,
        minX: Math.min(firstX, lastX),
        maxX: Math.max(firstX, lastX),
        minY: Math.min(firstY, lastY),
        maxY: Math.max(firstY, lastY),
        minZ: minZ,
        maxZ: maxZ,
        noise: median$1
      };
    }

    function generateContourLines(zData, options) {
      let noise = zData.noise;
      let z = zData.z;
      let povarHeight0, povarHeight1, povarHeight2, povarHeight3;
      let isOver0, isOver1, isOver2, isOver3;
      let nbSubSpectra = z.length;
      let nbPovars = z[0].length;
      let pAx, pAy, pBx, pBy;
      let x0 = zData.minX;
      let xN = zData.maxX;
      let dx = (xN - x0) / (nbPovars - 1);
      let y0 = zData.minY;
      let yN = zData.maxY;
      let dy = (yN - y0) / (nbSubSpectra - 1);
      let minZ = zData.minZ;
      let maxZ = zData.maxZ; // System.out.prvarln('y0 '+y0+' yN '+yN);
      // -------------------------
      // Povars attribution
      //
      // 0----1
      // |  / |
      // | /  |
      // 2----3
      //
      // ---------------------d------

      let iter = options.nbContourLevels * 2;
      let contourLevels = new Array(iter);
      let lineZValue;

      for (let level = 0; level < iter; level++) {
        // multiply by 2 for positif and negatif
        let contourLevel = {};
        contourLevels[level] = contourLevel;
        let side = level % 2;
        let factor = (maxZ - options.noiseMultiplier * noise) * Math.exp((level >> 1) - options.nbContourLevels);

        if (side === 0) {
          lineZValue = factor + options.noiseMultiplier * noise;
        } else {
          lineZValue = 0 - factor - options.noiseMultiplier * noise;
        }

        let lines = [];
        contourLevel.zValue = lineZValue;
        contourLevel.lines = lines;
        if (lineZValue <= minZ || lineZValue >= maxZ) continue;

        for (let iSubSpectra = 0; iSubSpectra < nbSubSpectra - 1; iSubSpectra++) {
          let subSpectra = z[iSubSpectra];
          let subSpectraAfter = z[iSubSpectra + 1];

          for (let povar = 0; povar < nbPovars - 1; povar++) {
            povarHeight0 = subSpectra[povar];
            povarHeight1 = subSpectra[povar + 1];
            povarHeight2 = subSpectraAfter[povar];
            povarHeight3 = subSpectraAfter[povar + 1];
            isOver0 = povarHeight0 > lineZValue;
            isOver1 = povarHeight1 > lineZValue;
            isOver2 = povarHeight2 > lineZValue;
            isOver3 = povarHeight3 > lineZValue; // Example povar0 is over the plane and povar1 and
            // povar2 are below, we find the varersections and add
            // the segment

            if (isOver0 !== isOver1 && isOver0 !== isOver2) {
              pAx = povar + (lineZValue - povarHeight0) / (povarHeight1 - povarHeight0);
              pAy = iSubSpectra;
              pBx = povar;
              pBy = iSubSpectra + (lineZValue - povarHeight0) / (povarHeight2 - povarHeight0);
              lines.push(pAx * dx + x0);
              lines.push(pAy * dy + y0);
              lines.push(pBx * dx + x0);
              lines.push(pBy * dy + y0);
            } // remove push does not help !!!!


            if (isOver3 !== isOver1 && isOver3 !== isOver2) {
              pAx = povar + 1;
              pAy = iSubSpectra + 1 - (lineZValue - povarHeight3) / (povarHeight1 - povarHeight3);
              pBx = povar + 1 - (lineZValue - povarHeight3) / (povarHeight2 - povarHeight3);
              pBy = iSubSpectra + 1;
              lines.push(pAx * dx + x0);
              lines.push(pAy * dy + y0);
              lines.push(pBx * dx + x0);
              lines.push(pBy * dy + y0);
            } // test around the diagonal


            if (isOver1 !== isOver2) {
              pAx = (povar + 1 - (lineZValue - povarHeight1) / (povarHeight2 - povarHeight1)) * dx + x0;
              pAy = (iSubSpectra + (lineZValue - povarHeight1) / (povarHeight2 - povarHeight1)) * dy + y0;

              if (isOver1 !== isOver0) {
                pBx = povar + 1 - (lineZValue - povarHeight1) / (povarHeight0 - povarHeight1);
                pBy = iSubSpectra;
                lines.push(pAx);
                lines.push(pAy);
                lines.push(pBx * dx + x0);
                lines.push(pBy * dy + y0);
              }

              if (isOver2 !== isOver0) {
                pBx = povar;
                pBy = iSubSpectra + 1 - (lineZValue - povarHeight2) / (povarHeight0 - povarHeight2);
                lines.push(pAx);
                lines.push(pAy);
                lines.push(pBx * dx + x0);
                lines.push(pBy * dy + y0);
              }

              if (isOver1 !== isOver3) {
                pBx = povar + 1;
                pBy = iSubSpectra + (lineZValue - povarHeight1) / (povarHeight3 - povarHeight1);
                lines.push(pAx);
                lines.push(pAy);
                lines.push(pBx * dx + x0);
                lines.push(pBy * dy + y0);
              }

              if (isOver2 !== isOver3) {
                pBx = povar + (lineZValue - povarHeight2) / (povarHeight3 - povarHeight2);
                pBy = iSubSpectra + 1;
                lines.push(pAx);
                lines.push(pAy);
                lines.push(pBx * dx + x0);
                lines.push(pBy * dy + y0);
              }
            }
          }
        }
      }

      return {
        minX: zData.minX,
        maxX: zData.maxX,
        minY: zData.minY,
        maxY: zData.maxY,
        segments: contourLevels
      };
    }

    function add2D(result, options) {
      let zData = convertTo3DZ(result.spectra);

      if (!options.noContour) {
        result.contourLines = generateContourLines(zData, options);
        delete zData.z;
      }

      result.minMax = zData;
    }

    function postProcessingNMR(entriesFlat) {
      // specific NMR functions
      for (let entry of entriesFlat) {
        let observeFrequency = 0;
        let shiftOffsetVal = 0;

        for (let spectrum of entry.spectra) {
          if (entry.ntuples && entry.ntuples.symbol) {
            if (!observeFrequency && spectrum.observeFrequency) {
              observeFrequency = spectrum.observeFrequency;
            }

            if (!shiftOffsetVal && spectrum.shiftOffsetVal) {
              shiftOffsetVal = spectrum.shiftOffsetVal;
            }
          } else {
            observeFrequency = spectrum.observeFrequency;
            shiftOffsetVal = spectrum.shiftOffsetVal;
          }

          if (observeFrequency) {
            if (spectrum.xUnits && spectrum.xUnits.toUpperCase().includes('HZ')) {
              spectrum.xUnits = 'PPM';
              spectrum.xFactor = spectrum.xFactor / observeFrequency;
              spectrum.firstX = spectrum.firstX / observeFrequency;
              spectrum.lastX = spectrum.lastX / observeFrequency;
              spectrum.deltaX = spectrum.deltaX / observeFrequency;

              for (let i = 0; i < spectrum.data.x.length; i++) {
                spectrum.data.x[i] /= observeFrequency;
              }
            }
          }

          if (shiftOffsetVal) {
            let shift = spectrum.firstX - shiftOffsetVal;
            spectrum.firstX = spectrum.firstX - shift;
            spectrum.lastX = spectrum.lastX - shift;

            for (let i = 0; i < spectrum.data.x.length; i++) {
              spectrum.data.x[i] -= shift;
            }
          } // we will check if some nucleus are missing ...


          if (entry.ntuples && entry.ntuples.nucleus && entry.ntuples.symbol) {
            for (let i = 0; i < entry.ntuples.nucleus.length; i++) {
              let symbol = entry.ntuples.symbol[i];
              let nucleus = entry.ntuples.nucleus[i];

              if (symbol.startsWith('F') && !nucleus) {
                if (symbol === 'F1') {
                  // if F1 is defined we will use F2
                  if (entry.tmp.$NUC2) {
                    entry.ntuples.nucleus[i] = entry.tmp.$NUC2;
                  } else {
                    let f2index = entry.ntuples.symbol.indexOf('F2');

                    if (f2index && entry.ntuples.nucleus[f2index]) {
                      entry.ntuples.nucleus[i] = entry.ntuples.nucleus[f2index];
                    }
                  }
                }

                if (symbol === 'F2') entry.ntuples.nucleus[i] = entry.tmp.$NUC1;
              }

              if (symbol === 'F2') {
                entry.yType = entry.ntuples.nucleus[0];
              }
            }
          }

          if (observeFrequency && entry.ntuples && entry.ntuples.symbol && entry.ntuples.nucleus) {
            let unit = '';
            let pageSymbolIndex = entry.ntuples.symbol.indexOf(spectrum.pageSymbol);

            if (entry.ntuples.units && entry.ntuples.units[pageSymbolIndex]) {
              unit = entry.ntuples.units[pageSymbolIndex];
            }

            if (unit !== 'PPM') {
              if (pageSymbolIndex !== 0) {
                throw Error('Not sure about this ntuples format');
              }

              let ratio0 = gyromagneticRatio[entry.ntuples.nucleus[0]];
              let ratio1 = gyromagneticRatio[entry.ntuples.nucleus[1]];

              if (!ratio0 || !ratio1) {
                throw Error('Problem with determination of gyromagnetic ratio');
              }

              let ratio = ratio0 / ratio1 * observeFrequency;
              spectrum.pageValue /= ratio;
            }
          }
        }
      }
    }

    function profiling(result, action, options) {
      if (result.profiling) {
        result.profiling.push({
          action,
          time: Date.now() - options.start
        });
      }
    }

    function simpleChromatogram(result) {
      let data = result.spectra[0].data;
      result.chromatogram = {
        times: data.x.slice(),
        series: {
          intensity: {
            dimension: 1,
            data: data.y.slice()
          }
        }
      };
    }

    function postProcessing(entriesFlat, result, options) {
      // converting Hz to ppm
      postProcessingNMR(entriesFlat);

      for (let entry of entriesFlat) {
        if (Object.keys(entry.ntuples).length > 0) {
          let newNtuples = [];
          let keys = Object.keys(entry.ntuples);

          for (let i = 0; i < keys.length; i++) {
            let key = keys[i];
            let values = entry.ntuples[key];

            for (let j = 0; j < values.length; j++) {
              if (!newNtuples[j]) newNtuples[j] = {};
              newNtuples[j][key] = values[j];
            }
          }

          entry.ntuples = newNtuples;
        }

        if (entry.twoD && options.wantXY) {
          add2D(entry, options);
          profiling(result, 'Finished countour plot calculation', options);

          if (!options.keepSpectra) {
            delete entry.spectra;
          }
        } // maybe it is a GC (HPLC) / MS. In this case we add a new format


        if (options.chromatogram) {
          if (entry.spectra.length > 1) {
            complexChromatogram(entry);
          } else {
            simpleChromatogram(entry);
          }

          profiling(result, 'Finished chromatogram calculation', options);
        }

        delete entry.tmp;
      }
    }

    function prepareNtuplesDatatable(currentEntry, spectrum, kind) {
      let xIndex = -1;
      let yIndex = -1;
      let firstVariable = '';
      let secondVariable = '';

      if (kind.indexOf('++') > 0) {
        firstVariable = kind.replace(/.*\(([a-zA-Z0-9]+)\+\+.*/, '$1');
        secondVariable = kind.replace(/.*\.\.([a-zA-Z0-9]+).*/, '$1');
      } else {
        kind = kind.replace(/[^a-zA-Z]/g, '');
        firstVariable = kind.charAt(0);
        secondVariable = kind.charAt(1);
        spectrum.variables = {};

        for (let symbol of kind) {
          let lowerCaseSymbol = symbol.toLowerCase();
          let index = currentEntry.ntuples.symbol.indexOf(symbol);
          if (index === -1) throw Error(`Symbol undefined: ${symbol}`);
          spectrum.variables[lowerCaseSymbol] = {};

          for (let key in currentEntry.ntuples) {
            if (currentEntry.ntuples[key][index]) {
              spectrum.variables[lowerCaseSymbol][key.replace(/^var/, '')] = currentEntry.ntuples[key][index];
            }
          }
        }
      }

      xIndex = currentEntry.ntuples.symbol.indexOf(firstVariable);
      yIndex = currentEntry.ntuples.symbol.indexOf(secondVariable);
      if (xIndex === -1) xIndex = 0;
      if (yIndex === -1) yIndex = 0;

      if (currentEntry.ntuples.first) {
        if (currentEntry.ntuples.first.length > xIndex) {
          spectrum.firstX = currentEntry.ntuples.first[xIndex];
        }

        if (currentEntry.ntuples.first.length > yIndex) {
          spectrum.firstY = currentEntry.ntuples.first[yIndex];
        }
      }

      if (currentEntry.ntuples.last) {
        if (currentEntry.ntuples.last.length > xIndex) {
          spectrum.lastX = currentEntry.ntuples.last[xIndex];
        }

        if (currentEntry.ntuples.last.length > yIndex) {
          spectrum.lastY = currentEntry.ntuples.last[yIndex];
        }
      }

      if (currentEntry.ntuples.vardim && currentEntry.ntuples.vardim.length > xIndex) {
        spectrum.nbPoints = currentEntry.ntuples.vardim[xIndex];
      }

      if (currentEntry.ntuples.factor) {
        if (currentEntry.ntuples.factor.length > xIndex) {
          spectrum.xFactor = currentEntry.ntuples.factor[xIndex];
        }

        if (currentEntry.ntuples.factor.length > yIndex) {
          spectrum.yFactor = currentEntry.ntuples.factor[yIndex];
        }
      }

      if (currentEntry.ntuples.units) {
        if (currentEntry.ntuples.units.length > xIndex) {
          if (currentEntry.ntuples.varname && currentEntry.ntuples.varname[xIndex]) {
            spectrum.xUnits = `${currentEntry.ntuples.varname[xIndex]} [${currentEntry.ntuples.units[xIndex]}]`;
          } else {
            spectrum.xUnits = currentEntry.ntuples.units[xIndex];
          }
        }

        if (currentEntry.ntuples.units.length > yIndex) {
          if (currentEntry.ntuples.varname && currentEntry.ntuples.varname[yIndex]) {
            spectrum.yUnits = `${currentEntry.ntuples.varname[yIndex]} [${currentEntry.ntuples.units[yIndex]}]`;
          } else {
            spectrum.yUnits = currentEntry.ntuples.units[yIndex];
          }
        }
      }
    }

    function prepareSpectrum(spectrum) {
      if (!spectrum.xFactor) spectrum.xFactor = 1;
      if (!spectrum.yFactor) spectrum.yFactor = 1;
    }

    const ntuplesSeparatorRegExp = /[ \t]*,[ \t]*/;
    const defaultOptions$1 = {
      keepRecordsRegExp: /^$/,
      canonicDataLabels: true,
      canonicMetadataLabels: false,
      dynamicTyping: true,
      withoutXY: false,
      chromatogram: false,
      keepSpectra: false,
      noContour: false,
      nbContourLevels: 7,
      noiseMultiplier: 5,
      profiling: false
    };
    /**
     *
     * @typedef {object} ConvertOptions
     * @property {RegExp} [options.keepRecordsRegExp=/^$/] - By default we don't keep meta information.
     * @property {boolean} [options.canonicDataLabels=true] - Canonize the Labels (uppercase without symbol).
     * @property {boolean} [options.canonicMetadataLabels=false] - Canonize the metadata Labels (uppercase without symbol).
     * @property {boolean} [options.dynamicTyping=false] - Convert numbers to Number.
     * @property {boolean} [options.withoutXY=false] - Remove the XY data.
     * @property {boolean} [options.chromatogram=false] - Special post-processing for GC / HPLC / MS.
     * @property {boolean} [options.keepSpectra=false] - Force to keep the spectra in case of 2D.
     * @property {boolean} [options.noContour=false] - Don't calculate countour in case of 2D.
     * @property {number} [options.nbContourLevels=7] - Number of positive / negative contour levels to calculate.
     * @property {number} [options.noiseMultiplier=5] - Define for 2D the level as 5 times the median as default.
     * @property {boolean} [options.profiling=false] - Add profiling information.
     */

    /**
     *
     * @typedef {object} Ntuples
     * @property {string[]} [varname]
     * @property {string[]} [symbol]
     * @property {string[]} [vartype]
     * @property {string[]} [varform]
     * @property {number[]} [vardim]
     * @property {string[]} [units]
     * @property {number[]} [factor]
     * @property {number[]} [first]
     * @property {number[]} [last]
     * @property {number[]} [min]
     * @property {number[]} [max]
     * @property {string[]} [nucleus]
     */

    /**
     * @typedef { Record<string, any> } Spectrum
     * @property {Record<string, number[]>} [data]
     * @property {number} [firstX]
     * @property {number} [lastX]
     * @property {number} [deltaX]
     * @property {number} [yFactor]
     * @property {number} [xFactor]
     * @property {number} [nbPoints]
     */

    /**
     *
     * @typedef {object} Entry
     * @property {Spectrum[]} spectra
     * @property {Ntuples} ntuples
     * @property {object} meta
     * @property {object} tmp
     * @property {string} [title]
     * @property {string} [dataType]
     * @property {string} [dataClass]
     * @property {boolean} [twoD]
     */

    /**
     *
     * @typedef { object } ConvertResult
     * @property { object[] | boolean } profiling
     * @property { string[] } logs
     * @property { object[] } entries
     * @property { Entry[] } flatten
     */

    /**
     * Parse a jcamp.
     *
     * @param {string|ArrayBuffer|Uint8Array} jcamp
     * @param {ConvertOptions} [options]
     * @returns {ConvertResult}
     */

    function convert(jcamp) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      jcamp = ensureString(jcamp);
      options = { ...defaultOptions$1,
        ...options
      };
      options.wantXY = !options.withoutXY;
      options.start = Date.now();
      let entriesFlat = [];
      let result = {
        profiling: options.profiling ? [] : false,
        logs: [],
        entries: []
      };
      let tmpResult = {
        children: []
      };
      let currentEntry = tmpResult;
      let parentsStack = [];
      let spectrum = {};

      if (typeof jcamp !== 'string') {
        throw new TypeError('the JCAMP should be a string');
      }

      profiling(result, 'Before split to LDRS', options);
      let ldrs = jcamp.replace(/[\r\n]+##/g, '\n##').split('\n##');
      profiling(result, 'Split to LDRS', options);
      if (ldrs[0]) ldrs[0] = ldrs[0].replace(/^[\r\n ]*##/, '');

      for (let ldr of ldrs) {
        // This is a new LDR
        let position = ldr.indexOf('=');
        let dataLabel = position > 0 ? ldr.substring(0, position) : ldr;
        let dataValue = position > 0 ? ldr.substring(position + 1).trim() : '';
        let canonicDataLabel = dataLabel.replace(/[_ -]/g, '').toUpperCase();

        if (canonicDataLabel === 'DATATABLE') {
          let endLine = dataValue.indexOf('\n');
          if (endLine === -1) endLine = dataValue.indexOf('\r');

          if (endLine > 0) {
            // ##DATA TABLE= (X++(I..I)), XYDATA
            // We need to find the variables
            let infos = dataValue.substring(0, endLine).split(/[ ,;\t]+/);
            prepareNtuplesDatatable(currentEntry, spectrum, infos[0]);
            spectrum.datatable = infos[0];

            if (infos[1] && infos[1].indexOf('PEAKS') > -1) {
              canonicDataLabel = 'PEAKTABLE';
            } else if (infos[1] && (infos[1].indexOf('XYDATA') || infos[0].indexOf('++') > 0)) {
              canonicDataLabel = 'XYDATA';
              spectrum.deltaX = (spectrum.lastX - spectrum.firstX) / (spectrum.nbPoints - 1);
            }
          }
        }

        if (canonicDataLabel === 'XYDATA') {
          if (options.wantXY) {
            prepareSpectrum(spectrum); // well apparently we should still consider it is a PEAK TABLE if there are no '++' after

            if (dataValue.match(/.*\+\+.*/)) {
              // ex: (X++(Y..Y))
              spectrum.deltaX = (spectrum.lastX - spectrum.firstX) / (spectrum.nbPoints - 1);
              fastParseXYData(spectrum, dataValue);
            } else {
              parsePeakTable(spectrum, dataValue, result);
            }

            currentEntry.spectra.push(spectrum);
            spectrum = {};
          }

          continue;
        } else if (canonicDataLabel === 'PEAKTABLE') {
          if (options.wantXY) {
            prepareSpectrum(spectrum);
            parsePeakTable(spectrum, dataValue, result);
            currentEntry.spectra.push(spectrum);
            spectrum = {};
          }

          continue;
        }

        if (canonicDataLabel === 'PEAKASSIGNMENTS') {
          if (options.wantXY) {
            if (dataValue.match(/.*(XYA).*/)) {
              // ex: (XYA)
              parseXYA(spectrum, dataValue);
            }

            currentEntry.spectra.push(spectrum);
            spectrum = {};
          }

          continue;
        }

        if (canonicDataLabel === 'TITLE') {
          let parentEntry = currentEntry;

          if (!parentEntry.children) {
            parentEntry.children = [];
          }

          currentEntry = {
            spectra: [],
            ntuples: {},
            info: {},
            meta: {},
            tmp: {} // tmp information we need to keep for postprocessing

          };
          parentEntry.children.push(currentEntry);
          parentsStack.push(parentEntry);
          entriesFlat.push(currentEntry);
          currentEntry.title = dataValue;
        } else if (canonicDataLabel === 'DATATYPE') {
          currentEntry.dataType = dataValue;

          if (dataValue.match(/(^nd|\snd\s)/i)) {
            currentEntry.twoD = true;
          }
        } else if (canonicDataLabel === 'NTUPLES') {
          if (dataValue.match(/(^nd|\snd\s)/i)) {
            currentEntry.twoD = true;
          }
        } else if (canonicDataLabel === 'DATACLASS') {
          currentEntry.dataClass = dataValue;
        } else if (canonicDataLabel === 'XUNITS') {
          spectrum.xUnits = dataValue;
        } else if (canonicDataLabel === 'YUNITS') {
          spectrum.yUnits = dataValue;
        } else if (canonicDataLabel === 'FIRSTX') {
          spectrum.firstX = Number(dataValue);
        } else if (canonicDataLabel === 'LASTX') {
          spectrum.lastX = Number(dataValue);
        } else if (canonicDataLabel === 'FIRSTY') {
          spectrum.firstY = Number(dataValue);
        } else if (canonicDataLabel === 'LASTY') {
          spectrum.lastY = Number(dataValue);
        } else if (canonicDataLabel === 'NPOINTS') {
          spectrum.nbPoints = Number(dataValue);
        } else if (canonicDataLabel === 'XFACTOR') {
          spectrum.xFactor = Number(dataValue);
        } else if (canonicDataLabel === 'YFACTOR') {
          spectrum.yFactor = Number(dataValue);
        } else if (canonicDataLabel === 'MAXX') {
          spectrum.maxX = Number(dataValue);
        } else if (canonicDataLabel === 'MINX') {
          spectrum.minX = Number(dataValue);
        } else if (canonicDataLabel === 'MAXY') {
          spectrum.maxY = Number(dataValue);
        } else if (canonicDataLabel === 'MINY') {
          spectrum.minY = Number(dataValue);
        } else if (canonicDataLabel === 'DELTAX') {
          spectrum.deltaX = Number(dataValue);
        } else if (canonicDataLabel === '.OBSERVEFREQUENCY' || canonicDataLabel === '$SFO1') {
          if (!spectrum.observeFrequency) {
            spectrum.observeFrequency = Number(dataValue);
          }
        } else if (canonicDataLabel === '.OBSERVENUCLEUS') {
          if (!spectrum.xType) {
            currentEntry.xType = dataValue.replace(/[^a-zA-Z0-9]/g, '');
          }
        } else if (canonicDataLabel === '$OFFSET') {
          // OFFSET for Bruker spectra
          currentEntry.shiftOffsetNum = 0;

          if (!spectrum.shiftOffsetVal) {
            spectrum.shiftOffsetVal = Number(dataValue);
          }
        } else if (canonicDataLabel === '$REFERENCEPOINT') ; else if (canonicDataLabel === 'VARNAME') {
          currentEntry.ntuples.varname = dataValue.split(ntuplesSeparatorRegExp);
        } else if (canonicDataLabel === 'SYMBOL') {
          currentEntry.ntuples.symbol = dataValue.split(ntuplesSeparatorRegExp);
        } else if (canonicDataLabel === 'VARTYPE') {
          currentEntry.ntuples.vartype = dataValue.split(ntuplesSeparatorRegExp);
        } else if (canonicDataLabel === 'VARFORM') {
          currentEntry.ntuples.varform = dataValue.split(ntuplesSeparatorRegExp);
        } else if (canonicDataLabel === 'VARDIM') {
          currentEntry.ntuples.vardim = convertToFloatArray$1(dataValue.split(ntuplesSeparatorRegExp));
        } else if (canonicDataLabel === 'UNITS') {
          currentEntry.ntuples.units = dataValue.split(ntuplesSeparatorRegExp);
        } else if (canonicDataLabel === 'FACTOR') {
          currentEntry.ntuples.factor = convertToFloatArray$1(dataValue.split(ntuplesSeparatorRegExp));
        } else if (canonicDataLabel === 'FIRST') {
          currentEntry.ntuples.first = convertToFloatArray$1(dataValue.split(ntuplesSeparatorRegExp));
        } else if (canonicDataLabel === 'LAST') {
          currentEntry.ntuples.last = convertToFloatArray$1(dataValue.split(ntuplesSeparatorRegExp));
        } else if (canonicDataLabel === 'MIN') {
          currentEntry.ntuples.min = convertToFloatArray$1(dataValue.split(ntuplesSeparatorRegExp));
        } else if (canonicDataLabel === 'MAX') {
          currentEntry.ntuples.max = convertToFloatArray$1(dataValue.split(ntuplesSeparatorRegExp));
        } else if (canonicDataLabel === '.NUCLEUS') {
          if (currentEntry.ntuples) {
            currentEntry.ntuples.nucleus = dataValue.split(ntuplesSeparatorRegExp);
          }
        } else if (canonicDataLabel === 'PAGE') {
          spectrum.page = dataValue.trim();
          spectrum.pageValue = Number(dataValue.replace(/^.*=/, ''));
          spectrum.pageSymbol = spectrum.page.replace(/[=].*/, '');
        } else if (canonicDataLabel === 'RETENTIONTIME') {
          spectrum.pageValue = Number(dataValue);
        } else if (isMSField(canonicDataLabel)) {
          spectrum[convertMSFieldToLabel(canonicDataLabel)] = dataValue;
        } else if (canonicDataLabel === 'SAMPLEDESCRIPTION') {
          spectrum.sampleDescription = dataValue;
        } else if (canonicDataLabel.startsWith('$NUC')) {
          if (!currentEntry.tmp[canonicDataLabel] && !dataValue.includes('off')) {
            currentEntry.tmp[canonicDataLabel] = dataValue.replace(/[<>]/g, '');
          }
        } else if (canonicDataLabel === 'END') {
          currentEntry = parentsStack.pop();
        }

        if (currentEntry && currentEntry.info && currentEntry.meta && canonicDataLabel.match(options.keepRecordsRegExp)) {
          let value = dataValue.trim();
          let target, label;

          if (dataLabel.startsWith('$')) {
            label = options.canonicMetadataLabels ? canonicDataLabel.substring(1) : dataLabel.substring(1);
            target = currentEntry.meta;
          } else {
            label = options.canonicDataLabels ? canonicDataLabel : dataLabel;
            target = currentEntry.info;
          }

          if (options.dynamicTyping) {
            value = parseString(value);
          }

          if (target[label]) {
            if (!Array.isArray(target[label])) {
              target[label] = [target[label]];
            }

            target[label].push(value);
          } else {
            target[label] = value;
          }
        }
      }

      profiling(result, 'Finished parsing', options);
      postProcessing(entriesFlat, result, options);
      profiling(result, 'Total time', options);
      /*
      if (result.children && result.children.length>0) {
        result = { ...result, ...result.children[0] };
      }
      */

      result.entries = tmpResult.children;
      result.flatten = entriesFlat;
      return result;
    }

    function parseData(file, options) {
      let {
        keepRecordsRegExp = /.*/
      } = options;
      let result = convert(file, {
        keepRecordsRegExp: keepRecordsRegExp
      });
      return result.flatten.length === 0 ? {} : result.flatten[0];
    }

    function ensureIOBuffer(data) {
      if (data instanceof Array || data instanceof Uint8Array) {
        data = new ArrayBuffer(data);
      }

      if (data instanceof ArrayBuffer) {
        return new IOBuffer(data);
      }

      return data;
    }

    function setFIDSpectrumData(file, spectra) {
      file = ensureIOBuffer(file);
      let td = parseInt(spectra.meta.TD[0], 10);
      let SW_H = parseFloat(spectra.meta.SW_h[0]);
      let SF = parseFloat(spectra.meta.SFO1[0]);
      spectra.meta.DATATYPE = 'NMR FID';
      let DW = 1 / (2 * SW_H);
      let AQ = td * DW;
      let endian = parseInt(spectra.meta.BYTORDA, 10);
      endian = endian ? 0 : 1;

      if (endian) {
        file.setLittleEndian();
      } else {
        file.setBigEndian();
      }

      let nbSubSpectra = spectra.meta.nbSubSpectra ? spectra.meta.nbSubSpectra : 1;
      spectra.spectra = new Array(nbSubSpectra);

      for (let j = 0; j < nbSubSpectra / 2; j++) {
        let toSave = {
          dataType: 'NMR FID',
          dataTable: '(X++(R..R))',
          nbPoints: td,
          firstX: 0,
          lastX: AQ,
          nucleus: spectra.meta.NUC1,
          xUnit: 'Sec',
          yUnit: 'Arbitrary',
          data: [new Float64Array(2 * td)],
          isXYdata: true,
          observeFrequency: SF,
          title: spectra.meta.TITLE,
          deltaX: DW
        };
        spectra.spectra[j * 2] = toSave;
        toSave = {
          dataType: 'NMR FID',
          dataTable: '(X++(I..I))',
          nbPoints: td,
          firstX: 0,
          lastX: AQ,
          nucleus: spectra.meta.NUC1,
          xUnit: 'Sec',
          yUnit: 'Arbitrary',
          data: new Float64Array(2 * td),
          isXYdata: true,
          directFrequency: SF,
          title: spectra.meta.TITLE,
          deltaX: DW
        };
        spectra.spectra[j * 2 + 1] = toSave;
        let i = 0;
        let x = 0;

        for (; file.available(8) && i < td; i++, x = i * DW) {
          let y = file.readInt32();

          if (y === null || isNaN(y)) {
            y = 0;
          }

          spectra.spectra[j * 2].data[2 * i + 1] = y;
          spectra.spectra[j * 2].data[2 * i] = x;
          y = file.readInt32();

          if (y === null || isNaN(y)) {
            y = 0;
          }

          spectra.spectra[j * 2 + 1].data[2 * i + 1] = y;
          spectra.spectra[j * 2 + 1].data[2 * i] = x;
        }

        for (; i < td; i++, x = i * DW) {
          spectra.spectra[j * 2].data[2 * i + 1] = 0;
          spectra.spectra[j * 2].data[2 * i] = x;
          spectra.spectra[j * 2 + 1].data[2 * i + 1] = 0;
          spectra.spectra[j * 2 + 1].data[2 * i] = x;
        }
      }
    }

    function setXYSpectrumData(file, spectra, real) {
      file = ensureIOBuffer(file);
      let td = parseInt(spectra.meta.SI, 10);
      let swP = parseFloat(spectra.meta.SW_p);
      let sf = parseFloat(spectra.meta.SF);
      let bf = sf;
      let offset = spectra.shiftOffsetVal || parseFloat(spectra.meta.OFFSET);
      spectra.meta.observeFrequency = sf;
      spectra.meta.brukerReference = bf;
      spectra.meta.DATATYPE = 'NMR Spectrum';
      let endian = parseInt(spectra.meta.BYTORDP, 10);
      endian = endian ? 0 : 1;
      let nbSubSpectra = spectra.meta.nbSubSpectra ? spectra.meta.nbSubSpectra : 1;

      if (endian) {
        file.setLittleEndian();
      } else {
        file.setBigEndian();
      }

      for (let i = 0; i < nbSubSpectra; i++) {
        let toSave = {
          dataType: 'NMR Spectrum',
          dataTable: '(X++(R..R))',
          nbPoints: td,
          firstX: offset,
          lastX: offset - swP / sf,
          xUnit: 'PPM',
          yUnit: 'Arbitrary',
          data: new Float64Array(td * 2),
          isXYdata: true,
          observeFrequency: sf,
          title: spectra.meta.TITLE,
          deltaX: -(swP / sf) / (td - 1)
        };
        let x = offset;
        let deltaX = toSave.deltaX;

        if (real) {
          for (let k = 0; k < td; ++k) {
            toSave.data[2 * k] = x;
            toSave.data[2 * k + 1] = file.readInt32();

            if (toSave.data[2 * k + 1] === null || isNaN(toSave.data[2 * k + 1])) {
              toSave.data[2 * k + 1] = 0;
            }

            x += deltaX;
          }
        } else {
          for (let k = 0; k < td; ++k) {
            toSave.data[2 * k] = x;
            toSave.data[2 * k + 1] = file.readInt32();

            if (toSave.data[2 * k + 1] === null || isNaN(toSave.data[2 * k + 1])) {
              toSave.data[2 * k + 1] = 0;
            }

            x += deltaX;
          }
        }

        spectra.spectra.push(toSave);
      }
    }

    function convert1D(files, options) {
      let result = parseData(files.procs || '', options);
      let temp = parseData(files.acqus || '', options);

      if (!Object.keys(result).length) {
        result = temp;
      }

      for (let key in result.meta) {
        result.meta[key] = [result.meta[key]];
      }

      for (let currKey in temp.meta) {
        if (result.meta[currKey] === undefined) {
          result.meta[currKey] = [temp.meta[currKey]];
        }
      }

      if (files['1r'] || files['1i']) {
        if (files['1r']) {
          setXYSpectrumData(files['1r'], result, true);
        }

        if (files['1i']) {
          setXYSpectrumData(files['1i'], result, false);
        }
      } else if (files.fid) {
        setFIDSpectrumData(files.fid, result);
      }

      return result;
    }

    function mergeMetadata(main, complement) {
      for (let key in complement.meta) {
        if (main.meta[key]) {
          if (!Array.isArray(main.meta[key])) {
            main.meta[key] = [main.meta[key]];
          }

          main.meta[key].push(complement.meta[key]);
        } else if (main.meta[key] === undefined) {
          main.meta[key] = [complement.meta[key]];
        }
      }

      return main;
    }

    function convert2D(files, options) {
      let temp, temp2, result;

      if (files.proc2s && files.procs) {
        result = parseData(files.procs, options);
        temp = parseData(files.proc2s, options);
        result = mergeMetadata(result, temp);
      }

      temp = parseData(files.acqus, options);
      temp2 = parseData(files.acqu2s, options);
      temp = mergeMetadata(temp, temp2);

      if (!result) {
        result = temp;
      }

      for (let key in temp.meta) {
        if (result.meta[key] === undefined) {
          result.meta[key] = temp.meta[key];
        }
      }

      for (let key in result.meta) {
        if (!Array.isArray(result.meta[key])) {
          result.meta[key] = [result.meta[key]];
        }
      }

      result.meta.nbSubSpectra = files['2rr'] ? parseInt(result.meta.SI[1], 10) : parseInt(result.meta.TD[1], 10);

      if (!result.meta.SW_p) {
        // eslint-disable-next-line camelcase
        result.meta.SW_p = result.meta.SW_h;
      }

      if (!result.meta.SF) {
        result.meta.SF = result.meta.SFO1;
      }

      let firstY, lastY, xOffset, yOffset;

      if (files['2rr']) {
        let sf = parseFloat(result.meta.SF[1]);
        let swP = parseFloat(result.meta.SW_p[1] || result.meta.SW[1]);
        yOffset = parseFloat(result.meta.OFFSET[1]);
        xOffset = parseFloat(result.meta.OFFSET[0]);
        firstY = yOffset;
        lastY = yOffset - swP / sf;
        result.meta.firstY = firstY;
        result.meta.lastY = lastY;
        setXYSpectrumData(files['2rr'], result, true);
      } else if (files.ser) {
        firstY = 0;
        lastY = result.meta.nbSubSpectra;
        let xWindowSize = parseFloat(result.meta.SW[0]);
        let yWindowSize = parseFloat(result.meta.SW[1]);
        let xTransmitterFrequency = parseFloat(result.meta.SFO1[0]);
        let yTransmitterFrequency = parseFloat(result.meta.SFO1[1]);
        let xTransmitterFrequencyOffset = parseFloat(result.meta.O1[0]);
        let yTransmitterFrequencyOffset = parseFloat(result.meta.O1[1]);
        xOffset = xTransmitterFrequencyOffset / xTransmitterFrequency + xWindowSize / 2;
        yOffset = yTransmitterFrequencyOffset / yTransmitterFrequency + yWindowSize / 2;
        setFIDSpectrumData(files.ser, result);
      }

      let pageValue = firstY;
      let nbSubSpectra = result.meta.nbSubSpectra;
      let deltaY = (lastY - firstY) / (nbSubSpectra - 1);

      for (let i = 0; i < nbSubSpectra; i++) {
        pageValue += deltaY;
        result.spectra[i].pageValue = pageValue;
      }

      let {
        NUC1: nuc1,
        AXNUC: axnuc,
        SF: sf
      } = result.meta;
      const nucleus = axnuc ? axnuc : nuc1 ? nuc1 : [];
      result.info['2D_Y_NUCLEUS'] = nucleus[1];
      result.info['2D_X_NUCLEUS'] = nucleus[0];
      result.info['2D_Y_FRECUENCY'] = sf[1];
      result.info['2D_X_FRECUENCY'] = sf[0];
      result.info['2D_Y_OFFSET'] = yOffset;
      result.info['2D_X_OFFSET'] = xOffset;
      result.info.twoD = result.twoD = true;
      return result;
    }

    /**
     * Extract information and data from bruker files.
     * @param {object} brukerFiles - Needed bruker files to parse raw data.
     * @param {object} [options = {}] - options.
     * @param {boolean} [options.xy] - if true, spectra data is a object with x and y
     * @param {RegExp} [options.keepRecordsRegExp='\/.*\/'] - regular expresion to parse the metadata of the spectrum.
     * @param {boolean} [options.noContours=false] - if true the contour data is not generated.
     * @returns
     */

    function convertFolder(brukerFiles) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      let start = new Date();
      let result;

      if (brukerFiles.ser || brukerFiles['2rr']) {
        result = convert2D(brukerFiles, options);
      } else if (brukerFiles['1r'] || brukerFiles['1i'] || brukerFiles.fid) {
        result = convert1D(brukerFiles, options);
      } else {
        throw new RangeError('The current files are invalid');
      } //normalizing info


      result.meta.DATE = parseFloat(result.meta.DATE);

      if (result.meta.GRPDLY) {
        result.meta.GRPDLY = parseFloat(result.meta.GRPDLY);
        result.meta.DSPFVS = parseFloat(result.meta.DSPFVS);
        result.meta.DECIM = parseFloat(result.meta.DECIM);
      }

      for (let key in result.meta) {
        if (!Array.isArray(result.meta[key])) {
          continue;
        }

        if (result.meta[key].length === 1) {
          result.meta[key] = result.meta[key][0];
        } else if (typeof result.meta[key][0] === 'string' && result.meta[key][0].indexOf('(0..') > -1) {
          result.meta[key] = result.meta[key][0];
        }
      }

      if (result.twoD) {
        add2D$1(result, options);

        if (result.profiling) {
          result.profiling.push({
            action: 'Finished countour plot calculation',
            time: new Date() - start
          });
        }

        if (!options.keepSpectra) {
          delete result.spectra;
        }
      }

      let spectra = result.spectra;

      if (options.xy && !!spectra) {
        //the spectraData should not be a oneD Array but an object with x and y
        if (spectra.length > 0) {
          for (let i = 0; i < spectra.length; i++) {
            let spectrum = spectra[i];

            if (spectrum.data.length) {
              let data = spectrum.data;
              let newData = {
                x: new Float64Array(data.length / 2),
                y: new Float64Array(data.length / 2)
              };

              for (let k = 0; k < data.length; k = k + 2) {
                newData.x[k / 2] = data[k];
                newData.y[k / 2] = data[k + 1];
              }

              spectrum.data = newData;
            }
          }
        }
      }

      const {
        file
      } = brukerFiles;

      if (file) {
        result.source = {
          file
        };
      }

      return result;
    }

    function extractFilePaths(pathSpectrum) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      let {
        zipFiles
      } = options;
      const expnoCheck = pathSpectrum.replace(/([.*\w+/]*)([0-9]+\/)[pdata|fid|ser]\/*.*/, '$1$2');
      const procnoCheck = pathSpectrum.match('pdata') ? pathSpectrum.replace(/([.*\w+/]*[0-9]+\/pdata\/[0-9]+\/)*.*/, '$1') : false;
      let filePaths = [];

      for (let file in zipFiles) {
        if (file.endsWith('/')) continue;
        let parts = file.split('/');
        let expno = parts.indexOf('pdata') > 0 ? file.slice(0, file.indexOf('pdata')) : parts[parts.length - 2].match(/^[0-9]+$/) ? file.slice(0, file.lastIndexOf('/') + 1) : null;
        if (expnoCheck !== expno) continue;

        if (file.match('pdata')) {
          let procno = file.slice(0, file.lastIndexOf('/') + 1);

          if (!procnoCheck) {
            if (file.match(/[1|2]+[i|r]+[i|r]*/)) continue;
          } else if (procnoCheck !== procno) {
            continue;
          }
        }

        filePaths.push(file);
      }

      return filePaths;
    }
    async function extractSingleSpectrumZip(pathSpectrum) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      let {
        zipFiles
      } = options;
      const filePaths = extractFilePaths(pathSpectrum, {
        zipFiles
      });
      let zipFolder = new JSZip();

      for (let file of filePaths) {
        zipFolder.file(file, await zipFiles[file].async('arraybuffer'));
      }

      return {
        extension: 'zip',
        name: pathSpectrum,
        binary: await zipFolder.generateAsync({
          type: 'uint8array',
          compression: 'DEFLATE',
          compressionOptions: {
            level: 9
          }
        })
      };
    }

    const BINARY = 1;
    const TEXT = 2;
    /**
     * Load a zip file using jszip and looking for folders contaning bruker files to parse.
     * @param {*} zipFile - binary or base64 file of the bruker zip file.
     * @param {object} [options={}] - options for jszip and convertFolder functions.
     * @param {boolean} [options.keepOriginal=false] - options to keep a smaller zip file per spectrum.
     * @returns {Promise} - Array of spectrum parsed from fid, ser, 2rr, 1r bruker files.
     */

    function convertZip(zipFile) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        keepOriginal = false
      } = options;
      const jsZip = new JSZip();
      return jsZip.loadAsync(zipFile, options).then(zip => {
        let files = {
          ser: BINARY,
          fid: BINARY,
          acqus: TEXT,
          acqu2s: TEXT,
          procs: TEXT,
          proc2s: TEXT,
          '1r': BINARY,
          '1i': BINARY,
          '2rr': BINARY
        };
        let folders = zip.filter(function (relativePath) {
          if (relativePath.match('__MACOSX')) return false;

          if (relativePath.endsWith('ser') || relativePath.endsWith('fid') || relativePath.endsWith('1r') || relativePath.endsWith('2rr')) {
            return true;
          }

          return false;
        });
        let spectra = new Array(folders.length);

        for (let i = 0; i < folders.length; ++i) {
          let promises = [];
          let name = folders[i].name;
          name = name.substr(0, name.lastIndexOf('/') + 1);
          promises.push(name);
          let currFolder = zip.folder(name);
          let currFiles = currFolder.filter(function (relativePath) {
            return files[relativePath] ? true : false;
          });

          if (name.indexOf('pdata') >= 0) {
            promises.push('acqus');
            promises.push(zip.file(name.replace(/pdata\/[0-9]+\//, 'acqus')).async('string'));
            let acqu2s = zip.file(name.replace(/pdata\/[0-9]+\//, 'acqu2s'));

            if (acqu2s) {
              promises.push('acqu2s');
              promises.push(acqu2s.async('string'));
            }
          }

          for (let j = 0; j < currFiles.length; ++j) {
            let idx = currFiles[j].name.lastIndexOf('/');
            name = currFiles[j].name.substr(idx + 1);
            promises.push(name);

            if (files[name] === BINARY) {
              promises.push(currFiles[j].async('arraybuffer'));
            } else {
              promises.push(currFiles[j].async('string'));
            }
          }

          if (keepOriginal) {
            promises.push('file');
            promises.push(extractSingleSpectrumZip(folders[i].name, {
              zipFiles: zip.files
            }));
          }

          spectra[i] = Promise.all(promises).then(result => {
            let brukerFiles = {};

            for (let k = 1; k < result.length; k += 2) {
              name = result[k];
              brukerFiles[name] = result[k + 1];
            }

            return {
              filename: result[0],
              value: convertFolder(brukerFiles, options)
            };
          });
        }

        return Promise.all(spectra);
      });
    }

    function convertToFloatArray(data) {
      if (isAnyArray$1(data[0])) {
        return data.map(e => Float64Array.from(e));
      } else if (isAnyArray$1(data)) {
        return Float64Array.from(data);
      } else if (typeof data === 'object') {
        let keys = Object.keys(data);

        for (let key of keys) {
          data[key] = convertToFloatArray(data[key]);
        }

        return data;
      }

      return data;
    }

    /* eslint-disable no-loss-of-precision */

    /**
     * Returns the group delay for old Bruker NMR spectra
     * @param {number} gspfvs
     * @param {number} decim
     * @return {number}
     */
    function getDigitalFilterParameters(grpdly, dspfvs, decim) {
      let value;

      if (grpdly > 0) {
        value = Number(grpdly);
      } else {
        if (dspfvs > 14) {
          value = 0;
        } else {
          if (!brukerDspTable[dspfvs]) {
            throw new Error('dspfvs not in lookup table');
          } else {
            const dspfvsList = brukerDspTable[dspfvs];
            if (!dspfvsList[decim]) throw new Error('decim not in lookup table');
            value = dspfvsList[decim];
          }
        }
      }

      return value;
    }
    const brukerDspTable = {
      10: {
        2: 44.75,
        3: 33.5,
        4: 66.625,
        6: 59.083333333333333,
        8: 68.5625,
        12: 60.375,
        16: 69.53125,
        24: 61.020833333333333,
        32: 70.015625,
        48: 61.34375,
        64: 70.2578125,
        96: 61.505208333333333,
        128: 70.37890625,
        192: 61.5859375,
        256: 70.439453125,
        384: 61.626302083333333,
        512: 70.4697265625,
        768: 61.646484375,
        1024: 70.48486328125,
        1536: 61.656575520833333,
        2048: 70.492431640625
      },
      11: {
        2: 46,
        3: 36.5,
        4: 48,
        6: 50.166666666666667,
        8: 53.25,
        12: 69.5,
        16: 72.25,
        24: 70.166666666666667,
        32: 72.75,
        48: 70.5,
        64: 73,
        96: 70.666666666666667,
        128: 72.5,
        192: 71.333333333333333,
        256: 72.25,
        384: 71.666666666666667,
        512: 72.125,
        768: 71.833333333333333,
        1024: 72.0625,
        1536: 71.916666666666667,
        2048: 72.03125
      },
      12: {
        2: 46,
        3: 36.5,
        4: 48,
        6: 50.166666666666667,
        8: 53.25,
        12: 69.5,
        16: 71.625,
        24: 70.166666666666667,
        32: 72.125,
        48: 70.5,
        64: 72.375,
        96: 70.666666666666667,
        128: 72.5,
        192: 71.333333333333333,
        256: 72.25,
        384: 71.666666666666667,
        512: 72.125,
        768: 71.833333333333333,
        1024: 72.0625,
        1536: 71.916666666666667,
        2048: 72.03125
      },
      13: {
        2: 2.75,
        3: 2.8333333333333333,
        4: 2.875,
        6: 2.9166666666666667,
        8: 2.9375,
        12: 2.9583333333333333,
        16: 2.96875,
        24: 2.9791666666666667,
        32: 2.984375,
        48: 2.9895833333333333,
        64: 2.9921875,
        96: 2.9947916666666667
      }
    };

    function getNucleusFromMetadata(metaData, info, subfix) {
      let nucleus = [];

      if (metaData[`${subfix}AXNUC`]) {
        nucleus = metaData[`${subfix}AXNUC`];
        if (!Array.isArray(nucleus)) nucleus = [nucleus];
        nucleus = checkForNucleus(nucleus);
      }

      if (nucleus.length < 1 && metaData[`${subfix}NUC1`]) {
        nucleus = metaData[`${subfix}NUC1`];
        if (!Array.isArray(nucleus)) nucleus = [nucleus];
        nucleus = checkForNucleus(nucleus);
      }

      if (nucleus.length === 0) {
        if (metaData['.NUCLEUS']) {
          nucleus = metaData['.NUCLEUS'].split(',').map(nuc => nuc.trim());
        } else if (metaData['.OBSERVENUCLEUS']) {
          nucleus = [metaData['.OBSERVENUCLEUS'].replace(/[^A-Za-z0-9]/g, '')];
        } else {
          nucleus = getNucleusFrom2DExperiment(info.experiment);
        }

        nucleus = checkForNucleus(nucleus);
      }

      if (metaData['2D_X_NUCLEUS'] && metaData['2D_Y_NUCLEUS']) {
        nucleus = [metaData['2D_X_NUCLEUS'].replace(/[^A-Za-z0-9]/g, ''), metaData['2D_Y_NUCLEUS'].replace(/[^A-Za-z0-9]/g, '')];
      }

      return nucleus;
    }
    /**
     * Returns a list of likely nuclei based on an experiment string
     * This is really an hypothesis and should not be used
     * @param {string} experiment
     * @return {string[]}
     */

    function getNucleusFrom2DExperiment(experiment) {
      if (typeof experiment !== 'string') {
        return [];
      }

      experiment = experiment.toLowerCase();

      if (experiment.includes('jres')) {
        return ['1H', 'Hz'];
      }

      if (experiment.includes('hmbc') || experiment.includes('hsqc')) {
        return ['1H', '13C'];
      }

      if (experiment.includes('cosy') || experiment.includes('tocsy')) {
        return ['1H', '1H'];
      }

      return [];
    }

    function checkForNucleus(nucleus) {
      nucleus = nucleus.map(value => value.replace(/[^A-Za-z0-9]/g, '').replace('NA', '').replace('off', ''));
      let beforeLength = nucleus.length;
      nucleus = nucleus.filter(value => value);
      return nucleus.length !== beforeLength ? [] : nucleus;
    }

    /**
     * Returns an experiment string based on a pulse sequence
     * @param {string} pulse
     * @return {string}
     */
    function getSpectrumType() {
      let meta = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      let info = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
      const {
        subfix = ''
      } = options;
      if (meta === null) meta = {};
      if (typeof meta === 'string') meta = {
        pulseSequence: meta
      };
      let spectyp = info[`${subfix}SPECTYP`];
      spectyp = (Array.isArray(spectyp) ? spectyp[0] : spectyp || '').replace(/^<(.*)>$/, '$1') // eslint-disable-line prefer-named-capture-group
      .toLowerCase();
      if (spectyp.length > 0 && spectyp !== 'undefined') return spectyp;
      let pulse = Array.isArray(meta.pulseSequence) ? meta.pulseSequence[0] : meta.pulseSequence || '';

      if (typeof pulse !== 'string') {
        return meta.dimension ? `${meta.dimension}d` : '';
      }

      pulse = pulse.toLowerCase();

      if (pulse.includes('zg') || pulse.includes('single_pulse_dec') || pulse.includes('udeft')) {
        return '1d';
      }

      if (pulse.includes('hsqct') || pulse.includes('invi') && (pulse.includes('ml') || pulse.includes('di'))) {
        return 'hsqctocsy';
      }

      if (pulse.includes('hsqc') || pulse.includes('invi')) {
        return 'hsqc';
      }

      if (pulse.includes('hmbc') || pulse.includes('inv4') && pulse.includes('lp')) {
        return 'hmbc';
      }

      if (pulse.includes('hmqc')) {
        return 'hmqc';
      }

      if (pulse.includes('cosy')) {
        return 'cosy';
      }

      if (pulse.includes('jres')) {
        return 'jres';
      }

      if (pulse.includes('tocsy') || pulse.includes('mlev') || pulse.includes('dipsi')) {
        return 'tocsy';
      }

      if (pulse.includes('noesy')) {
        return 'noesy';
      }

      if (pulse.includes('roesy')) {
        return 'roesy';
      }

      if (pulse.includes('dept')) {
        return 'dept';
      }

      if (pulse.includes('jmod') || pulse.includes('apt')) {
        return 'aptjmod';
      }

      if (pulse.includes('inad')) {
        return 'inadequate';
      }

      if (pulse.includes('adeq')) {
        return 'adequate';
      }

      return meta.dimension ? `${meta.dimension}d` : '';
    }

    function getInfoFromJCAMP(metaData) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        subfix = ''
      } = options;
      const info = {
        dimension: 0,
        nucleus: [],
        isFid: false,
        isFt: false,
        isComplex: false
      };
      let metadataString = JSON.stringify(metaData);
      const separator = metadataString.match('\r\n') ? '\r\n' : '\n';
      let {
        JCAMPDX: jcampdx = '',
        ORIGIN: origin = ''
      } = metaData;
      let creator = String(jcampdx).toLowerCase() + origin.toLowerCase();

      if (creator.includes('mestre') || creator.includes('nova')) {
        creator = 'mnova';
      }

      if (creator === 'mnova') {
        if (metaData.LONGDATE) {
          info.date = metaData.LONGDATE;
        }
      }

      info.nucleus = getNucleusFromMetadata(metaData, info, subfix);
      info.dimension = info.nucleus.length;
      maybeAdd(info, 'title', metaData.TITLE);
      maybeAdd(info, 'solvent', metaData['.SOLVENTNAME']);
      maybeAdd(info, 'temperature', metaData[`${subfix}TE`] || metaData['.TE']);
      maybeAdd(info, 'type', metaData.DATATYPE);

      if (info.type) {
        let typeLowerCase = info.type[0].toUpperCase();

        if (typeLowerCase.indexOf('FID') >= 0) {
          info.isFid = true;
          info.isComplex = true;
        } else if (typeLowerCase.indexOf('SPECTRUM') >= 0) {
          info.isFt = true;
          info.isComplex = true;
        }
      }

      maybeAdd(info, 'pulseSequence', metaData['.PULSESEQUENCE'] || metaData['.PULPROG'] || metaData[`${subfix}PULPROG`]);
      maybeAdd(info, 'experiment', getSpectrumType(info, metaData, {
        subfix
      }));
      maybeAdd(info, 'originFrequency', metaData['.OBSERVEFREQUENCY']);

      if (creator !== 'mnova' && creator !== 'mestre') {
        const gyromagneticRatioConst = gyromagneticRatio[info.nucleus[0]];
        maybeAdd(info, 'probeName', metaData[`${subfix}PROBHD`]);
        maybeAdd(info, 'originFrequency', metaData[`${subfix}SFO1`]);
        maybeAdd(info, 'baseFrequency', metaData[`${subfix}BF1`]);

        if (!['baseFrequency', 'originFrequency'].some(e => !info[e])) {
          const {
            baseFrequency,
            originFrequency
          } = info;
          let fieldStrength = 2 * Math.PI * (baseFrequency[0] / gyromagneticRatioConst) * 1e6;
          let frequencyOffset = baseFrequency.map((bf, i) => (originFrequency[i] - bf) * 1e6);
          maybeAdd(info, 'fieldStrength', fieldStrength);
          maybeAdd(info, 'frequencyOffset', frequencyOffset);
        }

        maybeAdd(info, 'spectralWidth', metaData[`${subfix}SW`]);
        maybeAdd(info, 'numberOfPoints', metaData[`${subfix}TD`]);
        const numberOfPoints = info.numberOfPoints;
        maybeAdd(info, 'sampleName', metaData[`${subfix}NAME`]);

        if (metaData[`${subfix}FNTYPE`] !== undefined) {
          maybeAdd(info, 'acquisitionMode', parseInt(metaData[`${subfix}FNTYPE`], 10));
        }

        let varName = metaData[`${subfix}VARNAME`] ? metaData[`${subfix}VARNAME`].split(',')[0] : '';

        if (varName === 'TIME') {
          let value = typeof metaData.LAST === 'string' || metaData.LAST instanceof String ? metaData.LAST.replace(' ', '').split(',')[0] : metaData.LAST;
          maybeAdd(info, 'acquisitionTime', Number(value));
        }

        if (!info.acquisitionTime) {
          if (!['numberOfPoints', 'spectralWidth'].some(e => !info[e])) {
            const {
              spectralWidth,
              originFrequency
            } = info;
            maybeAdd(info, 'acquisitionTime', Number((numberOfPoints[0] - 1) / (2 * spectralWidth[0] * originFrequency[0])));
          }
        }

        if (metaData[`${subfix}P`]) {
          let pulseStrength = 1e6 / (metaData[`${subfix}P`].split(separator)[1].split(' ')[1] * 4);
          maybeAdd(info, 'pulseStrength90', pulseStrength);
        }

        if (metaData[`${subfix}D`]) {
          let relaxationTime = metaData[`${subfix}D`].split(separator)[1].split(' ')[1];
          maybeAdd(info, 'relaxationTime', Number(relaxationTime));
        }

        maybeAdd(info, 'numberOfScans', Number(metaData[`${subfix}NS`]));
        let increment;

        if (!['numberOfPoints', 'spectralWidth'].some(e => !info[e])) {
          const {
            spectralWidth,
            numberOfPoints
          } = info;

          if (info.isFid) {
            maybeAdd(info, 'groupDelay', metaData[`${subfix}GRPDLY`] || 0);
            maybeAdd(info, 'DSPFVS', metaData[`${subfix}DSPFVS`]);
            maybeAdd(info, 'DECIM', metaData[`${subfix}DECIM`]);

            if (!['groupDelay', 'DSPFVS', 'DECIM'].some(e => !info[e])) {
              let {
                groupDelay,
                DSPFVS,
                DECIM
              } = info;
              let digitalFilterParameters = getDigitalFilterParameters(groupDelay[0], DSPFVS[0], DECIM[0]);
              maybeAdd(info, 'digitalFilter', digitalFilterParameters);
            }

            increment = numberOfPoints.map(nb => {
              return info.acquisitionTime[0] / (nb - 1);
            });
          } else {
            increment = numberOfPoints.map((nb, i) => {
              return spectralWidth[i] / (nb - 1);
            });
          }
        }

        maybeAdd(info, 'increment', increment);

        if (metaData[`${subfix}DATE`]) {
          info.date = new Date(parseInt(metaData[`${subfix}DATE`], 10) * 1000).toISOString();
        }

        if (!info.solvent) {
          maybeAdd(info, 'solvent', Array.isArray(metaData[`${subfix}SOLVENT`]) ? metaData[`${subfix}SOLVENT`][0] : metaData[`${subfix}SOLVENT`]);
        }
      }

      if (metaData.SYMBOL) {
        let symbols = metaData.SYMBOL.split(/[, ]+/);

        if (symbols.includes('R') && symbols.includes('I')) {
          info.isComplex = true;
        }
      }

      for (let key in info) {
        if (info[key].length === 1) info[key] = info[key][0];
      }

      if (!Array.isArray(info.nucleus)) info.nucleus = [info.nucleus];
      return info;
    }

    function maybeAdd(obj, name, value) {
      if (value !== undefined) {
        if (Array.isArray(value)) {
          obj[name] = value.map(v => {
            return removeUnless(v);
          });
        } else {
          obj[name] = [removeUnless(value)];
        }
      }
    }

    function removeUnless(value) {
      if (typeof value === 'string') {
        if (value.startsWith('<') && value.endsWith('>')) {
          value = value.substring(1, value.length - 1);
        }

        value = value.trim();
      }

      return value;
    }

    const defaultOptions = {
      noContour: true,
      xy: true,
      keepRecordsRegExp: /.*/,
      profiling: true
    };
    async function fromBruker(zipFile) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      let parseData = await convertZip(zipFile, { ...defaultOptions,
        ...options
      });
      let dataStructure = [];

      for (let element of parseData) {
        let entry = element.value;
        let metadata = Object.assign({}, entry.info, entry.meta);
        let info = getInfoFromJCAMP(metadata);
        if (info.experiment === 'wobble_curve') continue;
        let dimensions = [];
        let dependentVariables = [];
        let dependentVariable = {};

        if (info.dimension === 1) {
          for (let i = 0; i < entry.spectra.length; i++) {
            let data = entry.spectra[i].data;
            data = convertToFloatArray(data);
          }

          dependentVariable.components = entry.spectra;
        } else if (info.dimension === 2) {
          entry.minMax.z = convertToFloatArray(entry.minMax.z);
          dependentVariable.components = entry.minMax;
        }

        let dimension = {
          increment: info.increment,
          numberOfPoints: info.numberOfPoints
        };

        if (info.fid) {
          dimension.coordinatesOffset = {
            magnitude: -info.digitalFilter * info.increment,
            units: 'second'
          };
        } else {
          dimension.coordinatesOffset = {
            magnitude: info.frequencyOffset / info.baseFrequency - 0.5 * info.spectraWidth,
            units: 'ppm'
          };
        }

        dimensions.push(dimension);
        dependentVariables.push(dependentVariable);
        const {
          source
        } = entry;
        dataStructure.push({
          dimensions,
          dependentVariables,
          source,
          info: info,
          meta: metadata,
          timeStamp: new Date().valueOf(),
          version: packageJson.version
        });
      }

      return dataStructure;
    }

    const expectedTypes = ['ndnmrspectrum', 'ndnmrfid', 'nmrspectrum', 'nmrfid'];
    function fromJCAMP(buffer) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        noContour = true,
        xy = true,
        keepRecordsRegExp = /.*/,
        profiling = true
      } = options;
      let parsedData = convert(buffer, {
        noContour,
        xy,
        keepRecordsRegExp,
        profiling
      });
      let dataStructure = [];
      let entries = parsedData.flatten;

      for (let entry of entries) {
        if (!isSpectraData(entry)) continue;

        if (entry.spectra && entry.spectra.length > 0 || entry.minMax) {
          let metadata = Object.assign({}, entry.info, entry.meta);
          let info = getInfoFromJCAMP(metadata);
          if (info.experiment === 'wobble_curve') continue;
          let dimensions = [];
          let dependentVariables = [];
          let dependentVariable = {};

          if (info.dimension === 1) {
            for (let i = 0; i < entry.spectra.length; i++) {
              let data = entry.spectra[i].data;
              data = convertToFloatArray(data);
            }

            dependentVariable.components = entry.spectra;
          } else if (info.dimension === 2) {
            entry.minMax.z = convertToFloatArray(entry.minMax.z);
            dependentVariable.components = entry.minMax;
          }

          let dimension = {
            increment: info.increment,
            numberOfPoints: info.numberOfPoints
          };

          if (info.fid) {
            dimension.coordinatesOffset = {
              magnitude: -info.digitalFilter * info.increment,
              units: 'second'
            };
          } else {
            dimension.coordinatesOffset = {
              magnitude: info.frequencyOffset / info.baseFrequency - 0.5 * info.spectraWidth,
              units: 'ppm'
            };
          }

          dimensions.push(dimension);
          dependentVariables.push(dependentVariable);
          const data = {
            dimensions,
            dependentVariables,
            info,
            meta: metadata,
            timeStamp: new Date().valueOf(),
            version: packageJson.version
          };
          dataStructure.push(data);
        }
      }

      return dataStructure;
    }

    function isSpectraData(entry) {
      const {
        dataType = '',
        dataClass = ''
      } = entry;
      const inputDataType = dataType.replace(/\s/g, '').toLowerCase();
      const inputDataClass = dataClass.replace(/\s/g, '').toLowerCase();
      return expectedTypes.some(type => type === inputDataType) && inputDataClass !== 'peak table';
    }

    exports.formatDependentVariable = formatDependentVariable;
    exports.formatLinearDimension = formatLinearDimension;
    exports.fromBruker = fromBruker;
    exports.fromJCAMP = fromJCAMP;
    exports.fromJEOL = fromJEOL;

    Object.defineProperty(exports, '__esModule', { value: true });

}));
//# sourceMappingURL=nmr-parser.js.map
