/**
 * openchemlib-utils - Various utilities that extends openchemlib-js like HOSE codes or diastereotopic IDs
 * @version v2.7.1
 * @link https://github.com/cheminfo/openchemlib-utils#readme
 * @license MIT
 */
(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
  typeof define === 'function' && define.amd ? define(['exports'], factory) :
  (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.OCLUtils = {}));
})(this, (function (exports) { 'use strict';

  let xAtomicNumber = 0;

  /**
   * Tag an atom to be able to visualize it
   * @param {import('openchemlib').Molecule} molecule
   * @param {number} iAtom
   */
  function tagAtom(molecule, iAtom) {
    let customLabel = `${molecule.getAtomLabel(iAtom)}*`;
    molecule.setAtomCustomLabel(iAtom, customLabel);
    if (molecule.getAtomicNo(iAtom) === 1) {
      molecule.setAtomicNo(iAtom, getXAtomicNumber(molecule));
    } else {
      // we can not use X because we would have problems with valencies if it is
      // expanded hydrogens or not
      // we can not only use a custom label because it does not count for the canonisation
      molecule.setAtomMass(iAtom, molecule.getAtomMass(iAtom) + 5);
    }
    return customLabel;
  }
  function getXAtomicNumber(molecule) {
    if (!xAtomicNumber) {
      const OCL = molecule.getOCL();
      xAtomicNumber = OCL.Molecule.getAtomicNoFromLabel('X', OCL.Molecule.cPseudoAtomX);
    }
    return xAtomicNumber;
  }

  /**
   * Add either missing chirality of diastereotopic missing chirality
   * The problem is that sometimes we need to add chiral bond that was not planned because it is the same group
   * This is the case for example for the valine where the 2 C of the methyl groups are diastereotopic
   * @param {import('openchemlib').Molecule} molecule
   * @param {object} [options={}]
   * @param {number} [options.esrType=cESRTypeAnd]
   */
  function addDiastereotopicMissingChirality(molecule, options = {}) {
    const {
      Molecule
    } = molecule.getOCL();
    const {
      esrType = Molecule.cESRTypeAnd
    } = options;
    for (let iAtom = 0; iAtom < molecule.getAllAtoms(); iAtom++) {
      let tempMolecule = molecule.getCompactCopy();
      tagAtom(tempMolecule, iAtom);
      // After copy, helpers must be recalculated
      tempMolecule.ensureHelperArrays(Molecule.cHelperBitsStereo);
      // We need to have >0 and not >1 because there could be unspecified chirality in racemate

      for (let i = 0; i < tempMolecule.getAtoms(); i++) {
        // changed from from handling below; TLS 9.Nov.2015
        if (tempMolecule.isAtomStereoCenter(i) && tempMolecule.getStereoBond(i) === -1) {
          let stereoBond = tempMolecule.getAtomPreferredStereoBond(i);
          if (stereoBond !== -1) {
            molecule.setBondType(stereoBond, Molecule.cBondTypeUp);
            if (molecule.getBondAtom(1, stereoBond) === i) {
              let connAtom = molecule.getBondAtom(0, stereoBond);
              molecule.setBondAtom(0, stereoBond, i);
              molecule.setBondAtom(1, stereoBond, connAtom);
            }
            // To me it seems that we have to add all stereo centers into AND group 0. TLS 9.Nov.2015
            molecule.setAtomESR(i, esrType, 0);
          }
        }
      }
    }
  }

  /**
   *
   * @param {import('openchemlib').Molecule} [molecule] An instance of a molecule
   * @param {object} [options={}]
   * @param {object} [options.OCL] openchemlib library
   */
  function makeRacemic(molecule) {
    const {
      Molecule
    } = molecule.getOCL();

    // if we don't calculate this we have 2 epimers
    molecule.ensureHelperArrays(Molecule.cHelperCIP);

    // we need to make one group "AND" for chiral (to force to racemic, this means diastereotopic and not enantiotopic)
    for (let i = 0; i < molecule.getAllAtoms(); i++) {
      if (molecule.getAtomParity(i) !== Molecule.cAtomParityNone) {
        molecule.setAtomESR(i, Molecule.cESRTypeAnd, 0); // changed to group 0; TLS 9.Nov.2015
      }
    }
  }

  /**
   * Returns an array of diastereotopic ID (as oclCode)
   * @param {import('openchemlib').Molecule} molecule
   */
  function getDiastereotopicAtomIDs(molecule) {
    const OCL = molecule.getOCL();
    addDiastereotopicMissingChirality(molecule);
    let numberAtoms = molecule.getAllAtoms();
    let ids = [];
    for (let iAtom = 0; iAtom < numberAtoms; iAtom++) {
      let tempMolecule = molecule.getCompactCopy();
      tagAtom(tempMolecule, iAtom);
      makeRacemic(tempMolecule);
      // We need to ensure the helper array in order to get correctly the result of racemisation
      ids[iAtom] = tempMolecule.getCanonizedIDCode(OCL.Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS);
    }
    return ids;
  }

  /**
   *
   * @param {import('openchemlib').Molecule} molecule
   */
  function getDiastereotopicAtomIDsAndH(molecule) {
    const OCL = molecule.getOCL();
    molecule = molecule.getCompactCopy();
    molecule.addImplicitHydrogens();
    // TODO Temporary code ???
    molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
    const diaIDs = getDiastereotopicAtomIDs(molecule);
    const newDiaIDs = [];
    for (let i = 0; i < diaIDs.length; i++) {
      const diaID = diaIDs[i];
      const newDiaID = {
        oclID: diaID,
        hydrogenOCLIDs: [],
        nbHydrogens: 0
      };
      if (molecule.getAtomicNo(i) === 1) {
        const atom = molecule.getConnAtom(i, 0);
        newDiaID.heavyAtom = diaIDs[atom];
      }
      for (let j = 0; j < molecule.getAllConnAtoms(i); j++) {
        const atom = molecule.getConnAtom(i, j);
        if (molecule.getAtomicNo(atom) === 1) {
          newDiaID.nbHydrogens++;
          if (newDiaID.hydrogenOCLIDs.indexOf(diaIDs[atom]) === -1) {
            newDiaID.hydrogenOCLIDs.push(diaIDs[atom]);
          }
        }
      }
      newDiaIDs.push(newDiaID);
    }
    return newDiaIDs;
  }

  /**
   * Returns a SVG
   * @param {*} molecule
   * @param {*} [options={}]
   */
  function toDiastereotopicSVG(molecule, options = {}) {
    let {
      width = 300,
      height = 200,
      prefix = 'ocl',
      heavyAtomHydrogen = false
    } = options;
    let svg = options.svg;
    let diaIDs = [];
    let hydrogenInfo = {};
    getDiastereotopicAtomIDsAndH(molecule).forEach(line => {
      hydrogenInfo[line.oclID] = line;
    });
    if (heavyAtomHydrogen) {
      for (let i = 0; i < molecule.getAtoms(); i++) {
        diaIDs.push([]);
      }
      let groupedDiaIDs = molecule.getGroupedDiastereotopicAtomIDs();
      groupedDiaIDs.forEach(diaID => {
        if (hydrogenInfo[diaID.oclID] && hydrogenInfo[diaID.oclID].nbHydrogens > 0) {
          diaID.atoms.forEach(atom => {
            hydrogenInfo[diaID.oclID].hydrogenOCLIDs.forEach(id => {
              if (!diaIDs[atom * 1].includes(id)) diaIDs[atom].push(id);
            });
          });
        }
      });
    } else {
      diaIDs = molecule.getDiastereotopicAtomIDs().map(a => [a]);
    }
    if (!svg) svg = molecule.toSVG(width, height, prefix);
    svg = svg.replace(/Atom:[0-9]+"/g, value => {
      let atom = value.replace(/[^0-9]/g, '');
      return `${value} data-diaid="${diaIDs[atom].join(',')}"`;
    });
    return svg;
  }

  function groupDiastereotopicAtomIDs(diaIDs, molecule, options = {}) {
    const {
      atomLabel
    } = options;
    const diaIDsObject = {};
    for (let i = 0; i < diaIDs.length; i++) {
      if (!atomLabel || molecule.getAtomLabel(i) === atomLabel) {
        let diaID = diaIDs[i];
        if (!diaIDsObject[diaID]) {
          diaIDsObject[diaID] = {
            counter: 0,
            atoms: [],
            oclID: diaID,
            atomLabel: molecule.getAtomLabel(i)
          };
        }
        diaIDsObject[diaID].counter++;
        diaIDsObject[diaID].atoms.push(i);
      }
    }
    return Object.keys(diaIDsObject).map(key => diaIDsObject[key]);
  }

  /**
   * This function groups the diasterotopic atomIds of the molecule based on equivalence of atoms. The output object contains
   * a set of chemically equivalent atoms(element.atoms) and the groups of magnetically equivalent atoms (element.magneticGroups)
   * @param {import('openchemlib').Molecule} molecule
   * @param {object} [options={}]
   * @param {string} [options.atomLabel] Select atoms of the given atomLabel. By default it returns all the explicit atoms in the molecule
   * @returns {Array}
   */

  function getGroupedDiastereotopicAtomIDs(molecule, options = {}) {
    let diaIDs = getDiastereotopicAtomIDs(molecule);
    return groupDiastereotopicAtomIDs(diaIDs, molecule, options);
  }

  /**
   * Parse a molfile and returns an object containing the molecule, the map and the diaIDs
   * The map allows to reload properties assigned to the atom molfile
   * Pelase take care than numbering of atoms starts at 0 !
   * @param {import('openchemlib')} OCL - openchemlib library
   * @param {string} molfile
   * @returns
   */
  function getDiastereotopicAtomIDsFromMolfile(OCL, molfile) {
    const {
      map,
      molecule
    } = OCL.Molecule.fromMolfileWithAtomMap(molfile);
    const diaIDsArray = getDiastereotopicAtomIDsAndH(molecule);
    const diaIDs = {};
    for (let i = 0; i < map.length; i++) {
      diaIDs[map[i]] = {
        source: map[i],
        destination: i,
        ...diaIDsArray[i]
      };
    }
    return {
      map: diaIDs,
      molecule,
      diaIDs: diaIDsArray
    };
  }

  /**
   * Check if a specific atom is a sp3 carbon
   * @param {import('openchemlib').Molecule} molecule
   * @param {number} atomID
   */

  function isCsp3(molecule, atomID) {
    if (molecule.getAtomicNo(atomID) !== 6) return false;
    if (molecule.getAtomCharge(atomID) !== 0) return false;
    if (molecule.getImplicitHydrogens(atomID) + molecule.getConnAtoms(atomID) !== 4) {
      return false;
    }
    return true;
  }

  const FULL_HOSE_CODE = 1;
  const HOSE_CODE_CUT_C_SP3_SP3 = 2;

  /**
   * Returns the hose code for a specific atom number
   * @param {import('openchemlib').Molecule} originalMolecule - The OCL molecule to be fragmented
   * @param {number[]} rootAtoms
   * @param {object} [options={}]
   * @param {boolean} [options.isTagged] Specify is the atoms are already tagged
   * @param {number} [options.minSphereSize=0] Smallest hose code sphere
   * @param {number} [options.maxSphereSize=4] Largest hose code sphere
   * @param {number} [options.kind=FULL_HOSE_CODE] Kind of hose code, default usual sphere
   */
  function getHoseCodesForAtoms(originalMolecule, rootAtoms = [], options = {}) {
    const OCL = originalMolecule.getOCL();
    const {
      minSphereSize = 0,
      maxSphereSize = 4,
      kind = FULL_HOSE_CODE,
      isTagged = false
    } = options;
    const molecule = originalMolecule.getCompactCopy();
    if (!isTagged) {
      const tags = [];
      for (let i = 0; i < rootAtoms.length; i++) {
        let rootAtom = rootAtoms[i];
        tags.push(tagAtom(molecule, rootAtom));
        molecule.addImplicitHydrogens();
        molecule.addMissingChirality();
        molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
        // because ensuring helper reorder atoms we need to look again for it
      }

      rootAtoms.length = 0;
      for (let j = 0; j < molecule.getAllAtoms(); j++) {
        if (tags.includes(molecule.getAtomCustomLabel(j))) {
          rootAtoms.push(j);
        }
      }
    }
    let fragment = new OCL.Molecule(0, 0);
    let results = [];
    let min = 0;
    let max = 0;
    let atomMask = new Array(molecule.getAllAtoms());
    let atomList = new Array(molecule.getAllAtoms());
    for (let sphere = 0; sphere <= maxSphereSize; sphere++) {
      if (max === 0) {
        for (let rootAtom of rootAtoms) {
          atomList[max] = rootAtom;
          atomMask[rootAtom] = true;
          max++;
        }
      } else {
        let newMax = max;
        for (let i = min; i < max; i++) {
          let atom = atomList[i];
          for (let j = 0; j < molecule.getAllConnAtoms(atom); j++) {
            let connAtom = molecule.getConnAtom(atom, j);
            if (!atomMask[connAtom]) {
              switch (kind) {
                case FULL_HOSE_CODE:
                  atomMask[connAtom] = true;
                  atomList[newMax++] = connAtom;
                  break;
                case HOSE_CODE_CUT_C_SP3_SP3:
                  if (!(isCsp3(molecule, atom) && isCsp3(molecule, connAtom))) {
                    atomMask[connAtom] = true;
                    atomList[newMax++] = connAtom;
                  }
                  break;
                default:
                  throw new Error('getHoseCoesForAtom unknown kind');
              }
            }
          }
        }
        min = max;
        max = newMax;
      }
      molecule.copyMoleculeByAtoms(fragment, atomMask, true, null);
      if (sphere >= minSphereSize) {
        makeRacemic(fragment);
        results.push(fragment.getCanonizedIDCode(OCL.Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS));
      }
    }
    return results;
  }

  /**
   * Returns the hose code for a specific atom number
   * @param {import('openchemlib').Molecule} originalMolecule
   * @param {number} rootAtom
   * @param {object} [options={}]
   * @param {boolean} [options.isTagged] Specify is the atom is already tagged
   * @param {number} [options.minSphereSize=0] Smallest hose code sphere
   * @param {number} [options.maxSphereSize=4] Largest hose code sphere
   * @param {number} [options.kind=FULL_HOSE_CODE] Kind of hose code, default usual sphere
   */
  function getHoseCodesForAtom(originalMolecule, rootAtom, options = {}) {
    return getHoseCodesForAtoms(originalMolecule, [rootAtom], options);
  }

  /**
   * Returns the hose code for a specific marked atom
   * @param {import('openchemlib').Molecule} diastereotopicID
   * @param {object} options
   */

  function getHoseCodesFromDiastereotopicID(molecule, options = {}) {
    molecule.addImplicitHydrogens();
    molecule.addMissingChirality();

    // One of the atom has to be marked !
    let atomID = -1;
    for (let i = 0; i < molecule.getAllAtoms(); i++) {
      // we need to find the marked atom
      const atomCustomLabel = molecule.getAtomCustomLabel(i);
      if (atomCustomLabel != null && atomCustomLabel.endsWith('*')) {
        atomID = i;
        break;
      }
    }
    if (atomID >= 0) {
      options.isTagged = true;
      return getHoseCodesForAtom(molecule, atomID, options);
    }
    return undefined;
  }

  /**
   * Returns an array containing one entry per atom containing
   * diaID and hose code
   * @param {import('openchemlib').Molecule} molecule
   * @param {object} options
   */

  function getHoseCodesAndDiastereotopicIDs(molecule, options) {
    const diaIDs = getDiastereotopicAtomIDs(molecule).map(oclID => ({
      oclID
    }));
    const OCL = molecule.getOCL();
    // TODO: seems like a very slow approach
    diaIDs.forEach(diaID => {
      const hoses = getHoseCodesFromDiastereotopicID(OCL.Molecule.fromIDCode(diaID.oclID), options);
      diaID.hoses = [];
      let sphere = 0;
      for (const hose of hoses) {
        diaID.hoses.push({
          sphere: sphere++,
          oclID: hose
        });
      }
    });
    return diaIDs;
  }

  let fragment$1;

  /**
   * Returns the hose code for a specific atom number
   * @param {import('openchemlib').Molecule} molecule
   */
  function getHoseCodesForPath(molecule, from, to, maxLength) {
    const OCL = molecule.getOCL();
    const originalFrom = from;
    const originalTo = to;
    molecule = molecule.getCompactCopy();
    let originalAtoms = []; // path before renumbering
    molecule.getPath(originalAtoms, from, to, maxLength + 1);
    let torsion;
    if (originalAtoms.length === 4) {
      torsion = molecule.calculateTorsion(originalAtoms);
    }
    const tag1 = tagAtom(molecule, from);
    const tag2 = tagAtom(molecule, to);
    molecule.addImplicitHydrogens();
    molecule.addMissingChirality();
    molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
    from = -1;
    to = -1;
    for (let i = 0; i < molecule.getAllAtoms(); i++) {
      if (tag1 === tag2) {
        if (molecule.getAtomCustomLabel(i) === tag1) {
          if (from === -1) {
            from = i;
          } else {
            to = i;
          }
        }
      } else {
        if (tag1 === molecule.getAtomCustomLabel(i)) {
          from = i;
        }
        if (tag2 === molecule.getAtomCustomLabel(i)) {
          to = i;
        }
      }
    }
    if (!fragment$1) fragment$1 = new OCL.Molecule(0, 0);
    let atoms = [];
    molecule.getPath(atoms, from, to, maxLength + 1);
    let min = 0;
    let max = 0;
    let atomMask = new Array(molecule.getAllAtoms()).fill(false);
    let atomList = new Array(molecule.getAllAtoms()).fill(-1);
    let hoses = [];
    for (let sphere = 0; sphere <= 2; sphere++) {
      if (max === 0) {
        for (let atom of atoms) {
          atomMask[atom] = true;
          atomList[max++] = atom;
        }
      } else {
        let newMax = max;
        for (let i = min; i < max; i++) {
          let atom = atomList[i];
          for (let j = 0; j < molecule.getAllConnAtoms(atom); j++) {
            let connAtom = molecule.getConnAtom(atom, j);
            if (!atomMask[connAtom]) {
              atomMask[connAtom] = true;
              atomList[newMax++] = connAtom;
            }
          }
        }
        min = max;
        max = newMax;
      }
      let atomMap = [];
      molecule.copyMoleculeByAtoms(fragment$1, atomMask, true, atomMap);
      makeRacemic(fragment$1);
      let oclID = fragment$1.getCanonizedIDCode(OCL.Molecule.CANONIZER_ENCODE_ATOM_CUSTOM_LABELS);
      hoses.push({
        sphere,
        oclID
      });
    }
    return {
      atoms: originalAtoms,
      from: originalFrom,
      to: originalTo,
      torsion,
      hoses,
      length: originalAtoms.length - 1
    };
  }

  const MAX_R = 10;

  /**
   * Generate molecules and calculate predicted properties form a list of smiles and fragments
   * @param {string} [coreSmiles]
   * @param {array} [fragments] Array of {smiles,R1,R2,...}
   * @param {import('openchemlib')} OCL - openchemlib library
   * @param {object} [options={}]
   * @param {function} [options.onStep] method to execute each new molecules
   * @param {boolean} [options.complexity] returns only the number of molecules to evaluate
   * @return {Promise} promise that resolves to molecules or complexity as a number
   */
  async function combineSmiles(coreSmiles, fragments, OCL, options = {}) {
    const {
      complexity = false
    } = options;
    const core = getCore(coreSmiles);
    const rGroups = getRGroups(core, fragments);
    if (complexity) {
      return getComplexity(rGroups);
    }
    return generate(core, rGroups, OCL, options);
  }
  function getComplexity(rGroups) {
    let complexity = 1;
    for (let rGroup of rGroups) {
      complexity *= rGroup.smiles.length;
    }
    return complexity;
  }
  async function generate(core, rGroups, OCL, options = {}) {
    const {
      onStep
    } = options;
    const molecules = {};
    const sizes = new Array(rGroups.length);
    const currents = new Array(rGroups.length);
    for (let i = 0; i < rGroups.length; i++) {
      sizes[i] = rGroups[i].smiles.length - 1;
      currents[i] = 0;
    }
    let position = 0;
    let counter = 0;
    while (true) {
      counter++;
      while (position < currents.length) {
        if (currents[position] < sizes[position]) {
          if (onStep) {
            await onStep(counter);
          }
          appendMolecule(molecules, core, rGroups, currents, OCL);
          currents[position]++;
          for (let i = 0; i < position; i++) {
            currents[i] = 0;
          }
          position = 0;
        } else {
          position++;
        }
      }
      if (position = currents.length) {
        if (onStep) {
          await onStep(counter);
        }
        appendMolecule(molecules, core, rGroups, currents, OCL);
        break;
      }
    }
    return Object.keys(molecules).map(key => molecules[key]).sort((m1, m2) => m1.mw - m2.mw);
  }
  function appendMolecule(molecules, core, rGroups, currents, OCL) {
    let newSmiles = core.smiles;
    for (let i = 0; i < currents.length; i++) {
      newSmiles += `.${rGroups[i].smiles[currents[i]]}`;
    }
    const currentMol = OCL.Molecule.fromSmiles(newSmiles);
    const idCode = currentMol.getIDCode();
    if (!molecules[idCode]) {
      let molecule = {};
      molecules[idCode] = molecule;
      molecule.smiles = currentMol.toSmiles();
      molecule.combinedSmiles = newSmiles;
      molecule.idCode = idCode;
      molecule.molfile = currentMol.toMolfile();
      const props = new OCL.MoleculeProperties(currentMol);
      molecule.nbHAcceptor = props.acceptorCount;
      molecule.nbHDonor = props.donorCount;
      molecule.logP = props.logP;
      molecule.logS = props.logS;
      molecule.PSA = props.polarSurfaceArea;
      molecule.nbRottable = props.rotatableBondCount;
      molecule.nbStereoCenter = props.stereoCenterCount;
      let mf = currentMol.getMolecularFormula();
      molecule.mf = mf.formula;
      molecule.mw = mf.relativeWeight;
    }
  }
  function getCore(coreSmiles) {
    let core = {
      originalSmiles: coreSmiles,
      smiles: coreSmiles.replace(/\[R(?<group>[1-4])\]/g, '%5$<group>')
    };
    for (let i = 0; i < MAX_R; i++) {
      if (core.originalSmiles.indexOf(`[R${i}]`) > -1) core[`R${i}`] = true;
    }
    return core;
  }
  function getRGroups(core, fragments) {
    let rGroups = {};
    for (const fragment of fragments) {
      if (fragment.smiles) {
        const smiles = updateRPosition(fragment.smiles);
        for (let i = 0; i < MAX_R; i++) {
          if (core[`R${i}`]) {
            // we only consider the R that are in the core
            if (fragment[`R${i}`]) {
              if (!rGroups[`R${i}`]) {
                rGroups[`R${i}`] = {
                  group: `R${i}`,
                  smiles: []
                };
              }
              rGroups[`R${i}`].smiles.push(smiles.replace(/\[R\]/, `(%5${i})`));
            }
          }
        }
      }
    }
    return Object.keys(rGroups).map(key => rGroups[key]);
  }
  function updateRPosition(smiles) {
    // R group should not be at the beginning
    if (smiles.indexOf('[R]') !== 0) return smiles;
    if (smiles.length === 3) return '[H][R]';
    // we are in trouble ... we need to move the R
    let newSmiles = smiles.replace('[R]', '');
    // we need to check where we should put the R group
    let level = 0;
    for (let j = 0; j < newSmiles.length; j++) {
      let currentChar = newSmiles.charAt(j);
      let currentSubstring = newSmiles.substr(j);
      if (currentChar === '(') {
        level++;
      } else if (currentChar === ')') {
        level--;
      } else if (level === 0) {
        if (currentSubstring.match(/^[a-z]/)) {
          return `${newSmiles.substr(0, j + 1)}([R])${newSmiles.substr(j + 1)}`;
        } else if (currentSubstring.match(/^[A-Z][a-z]/)) {
          return `${newSmiles.substr(0, j + 2)}([R])${newSmiles.substr(j + 2)}`;
        } else if (currentSubstring.match(/^[A-Z]/)) {
          return `${newSmiles.substr(0, j + 1)}([R])${newSmiles.substr(j + 1)}`;
        }
      }
    }
    return smiles;
  }

  /**
   * Returns various information about atoms in the molecule
   * @param {import('openchemlib').Molecule} [molecule]
   */
  function getAtomsInfo(molecule) {
    const OCL = molecule.getOCL();
    molecule.ensureHelperArrays(OCL.Molecule.cHelperRings);
    let diaIDs = getDiastereotopicAtomIDs(molecule);
    let results = [];
    for (let i = 0; i < diaIDs.length; i++) {
      let result = {
        oclID: diaIDs[i],
        extra: {
          singleBonds: 0,
          doubleBonds: 0,
          tripleBonds: 0,
          aromaticBonds: 0,
          cnoHybridation: 0 // should be 1 (sp), 2 (sp2) or 3 (sp3)
        }
      };

      let extra = result.extra;
      results.push(result);
      result.abnormalValence = molecule.getAtomAbnormalValence(i); // -1 is normal otherwise specified
      result.charge = molecule.getAtomCharge(i);
      result.cipParity = molecule.getAtomCIPParity(i);
      result.color = molecule.getAtomColor(i);
      result.customLabel = molecule.getAtomCustomLabel(i);
      //        result.esrGroup=molecule.getAtomESRGroup(i);
      //        result.esrType=molecule.getAtomESRType(i);
      result.atomicNo = molecule.getAtomicNo(i);
      result.label = molecule.getAtomLabel(i);
      //        result.list=molecule.getAtomList(i);
      //        result.listString=molecule.getAtomListString(i);
      //        result.mapNo=molecule.getAtomMapNo(i);
      result.mass = molecule.getAtomMass(i);
      //        result.parity=molecule.getAtomParity(i);
      //        result.pi=molecule.getAtomPi(i);
      //        result.preferredStereoBond=molecule.getAtomPreferredStereoBond(i);
      //        result.queryFeatures=molecule.getAtomQueryFeatures(i);
      result.radical = molecule.getAtomRadical(i);
      result.ringBondCount = molecule.getAtomRingBondCount(i);
      //        result.ringCount=molecule.getAtomRingCount(i);
      result.ringSize = molecule.getAtomRingSize(i);
      result.x = molecule.getAtomX(i);
      result.y = molecule.getAtomY(i);
      result.z = molecule.getAtomZ(i);
      result.allHydrogens = molecule.getAllHydrogens(i);
      result.connAtoms = molecule.getConnAtoms(i);
      result.allConnAtoms = molecule.getAllConnAtoms(i);
      result.implicitHydrogens = result.allHydrogens + result.connAtoms - result.allConnAtoms;
      result.isAromatic = molecule.isAromaticAtom(i);
      result.isAllylic = molecule.isAllylicAtom(i);
      result.isStereoCenter = molecule.isAtomStereoCenter(i);
      result.isRing = molecule.isRingAtom(i);
      result.isSmallRing = molecule.isSmallRingAtom(i);
      result.isStabilized = molecule.isStabilizedAtom(i);

      // todo HACK to circumvent bug in OCL that consider than an hydrogen is connected to itself
      result.extra.singleBonds = result.atomicNo === 1 ? 0 : result.implicitHydrogens;
      for (let j = 0; j < molecule.getAllConnAtoms(i); j++) {
        let bond = molecule.getConnBond(i, j);
        let bondOrder = molecule.getBondOrder(bond);
        if (molecule.isAromaticBond(bond)) {
          extra.aromaticBonds++;
        } else if (bondOrder === 1) {
          // not an hydrogen
          extra.singleBonds++;
        } else if (bondOrder === 2) {
          extra.doubleBonds++;
        } else if (bondOrder === 3) {
          extra.tripleBonds++;
        }
      }
      result.extra.totalBonds = result.extra.singleBonds + result.extra.doubleBonds + result.extra.tripleBonds + result.extra.aromaticBonds;
      if (result.atomicNo === 6) {
        result.extra.cnoHybridation = result.extra.totalBonds - 1;
      } else if (result.atomicNo === 7) {
        result.extra.cnoHybridation = result.extra.totalBonds;
      } else if (result.atomicNo === 8) {
        result.extra.cnoHybridation = result.extra.totalBonds + 1;
      } else if (result.atomicNo === 1) {
        let connectedAtom = molecule.getAllConnAtoms(i) === 0 ? 0 : molecule.getAtomicNo(molecule.getConnAtom(i, 0));
        result.extra.hydrogenOnAtomicNo = connectedAtom;
        if (connectedAtom === 7 || connectedAtom === 8) {
          result.extra.labileHydrogen = true;
        }
      }
    }
    return results;
  }

  // eslint-disable-next-line @typescript-eslint/unbound-method
  const toString = Object.prototype.toString;
  /**
   * Checks if an object is an instance of an Array (array or typed array, except those that contain bigint values).
   *
   * @param value - Object to check.
   * @returns True if the object is an array or a typed array.
   */
  function isAnyArray(value) {
    const tag = toString.call(value);
    return tag.endsWith('Array]') && !tag.includes('Big');
  }

  function max(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }
    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }
    var _options$fromIndex = options.fromIndex,
      fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
      _options$toIndex = options.toIndex,
      toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;
    if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
      throw new Error('fromIndex must be a positive integer smaller than length');
    }
    if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
      throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
    }
    var maxValue = input[fromIndex];
    for (var i = fromIndex + 1; i < toIndex; i++) {
      if (input[i] > maxValue) maxValue = input[i];
    }
    return maxValue;
  }

  function min(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    }
    if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }
    var _options$fromIndex = options.fromIndex,
      fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
      _options$toIndex = options.toIndex,
      toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;
    if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
      throw new Error('fromIndex must be a positive integer smaller than length');
    }
    if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
      throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
    }
    var minValue = input[fromIndex];
    for (var i = fromIndex + 1; i < toIndex; i++) {
      if (input[i] < minValue) minValue = input[i];
    }
    return minValue;
  }

  function rescale(input) {
    var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    if (!isAnyArray(input)) {
      throw new TypeError('input must be an array');
    } else if (input.length === 0) {
      throw new TypeError('input must not be empty');
    }
    var output;
    if (options.output !== undefined) {
      if (!isAnyArray(options.output)) {
        throw new TypeError('output option must be an array if specified');
      }
      output = options.output;
    } else {
      output = new Array(input.length);
    }
    var currentMin = min(input);
    var currentMax = max(input);
    if (currentMin === currentMax) {
      throw new RangeError('minimum and maximum input values are equal. Cannot rescale a constant array');
    }
    var _options$min = options.min,
      minValue = _options$min === void 0 ? options.autoMinMax ? currentMin : 0 : _options$min,
      _options$max = options.max,
      maxValue = _options$max === void 0 ? options.autoMinMax ? currentMax : 1 : _options$max;
    if (minValue >= maxValue) {
      throw new RangeError('min option must be smaller than max option');
    }
    var factor = (maxValue - minValue) / (currentMax - currentMin);
    for (var i = 0; i < input.length; i++) {
      output[i] = (input[i] - currentMin) * factor + minValue;
    }
    return output;
  }

  const indent = ' '.repeat(2);
  const indentData = ' '.repeat(4);
  function inspectMatrix() {
    return inspectMatrixWithOptions(this);
  }
  function inspectMatrixWithOptions(matrix, options = {}) {
    const {
      maxRows = 15,
      maxColumns = 10,
      maxNumSize = 8,
      padMinus = 'auto'
    } = options;
    return `${matrix.constructor.name} {
${indent}[
${indentData}${inspectData(matrix, maxRows, maxColumns, maxNumSize, padMinus)}
${indent}]
${indent}rows: ${matrix.rows}
${indent}columns: ${matrix.columns}
}`;
  }
  function inspectData(matrix, maxRows, maxColumns, maxNumSize, padMinus) {
    const {
      rows,
      columns
    } = matrix;
    const maxI = Math.min(rows, maxRows);
    const maxJ = Math.min(columns, maxColumns);
    const result = [];
    if (padMinus === 'auto') {
      padMinus = false;
      loop: for (let i = 0; i < maxI; i++) {
        for (let j = 0; j < maxJ; j++) {
          if (matrix.get(i, j) < 0) {
            padMinus = true;
            break loop;
          }
        }
      }
    }
    for (let i = 0; i < maxI; i++) {
      let line = [];
      for (let j = 0; j < maxJ; j++) {
        line.push(formatNumber(matrix.get(i, j), maxNumSize, padMinus));
      }
      result.push(`${line.join(' ')}`);
    }
    if (maxJ !== columns) {
      result[result.length - 1] += ` ... ${columns - maxColumns} more columns`;
    }
    if (maxI !== rows) {
      result.push(`... ${rows - maxRows} more rows`);
    }
    return result.join(`\n${indentData}`);
  }
  function formatNumber(num, maxNumSize, padMinus) {
    return (num >= 0 && padMinus ? ` ${formatNumber2(num, maxNumSize - 1)}` : formatNumber2(num, maxNumSize)).padEnd(maxNumSize);
  }
  function formatNumber2(num, len) {
    // small.length numbers should be as is
    let str = num.toString();
    if (str.length <= len) return str;

    // (7)'0.00123' is better then (7)'1.23e-2'
    // (8)'0.000123' is worse then (7)'1.23e-3',
    let fix = num.toFixed(len);
    if (fix.length > len) {
      fix = num.toFixed(Math.max(0, len - (fix.length - len)));
    }
    if (fix.length <= len && !fix.startsWith('0.000') && !fix.startsWith('-0.000')) {
      return fix;
    }

    // well, if it's still too long the user should've used longer numbers
    let exp = num.toExponential(len);
    if (exp.length > len) {
      exp = num.toExponential(Math.max(0, len - (exp.length - len)));
    }
    return exp.slice(0);
  }

  function installMathOperations(AbstractMatrix, Matrix) {
    AbstractMatrix.prototype.add = function add(value) {
      if (typeof value === 'number') return this.addS(value);
      return this.addM(value);
    };
    AbstractMatrix.prototype.addS = function addS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + value);
        }
      }
      return this;
    };
    AbstractMatrix.prototype.addM = function addM(matrix) {
      matrix = Matrix.checkMatrix(matrix);
      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + matrix.get(i, j));
        }
      }
      return this;
    };
    AbstractMatrix.add = function add(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.add(value);
    };
    AbstractMatrix.prototype.sub = function sub(value) {
      if (typeof value === 'number') return this.subS(value);
      return this.subM(value);
    };
    AbstractMatrix.prototype.subS = function subS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - value);
        }
      }
      return this;
    };
    AbstractMatrix.prototype.subM = function subM(matrix) {
      matrix = Matrix.checkMatrix(matrix);
      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - matrix.get(i, j));
        }
      }
      return this;
    };
    AbstractMatrix.sub = function sub(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sub(value);
    };
    AbstractMatrix.prototype.subtract = AbstractMatrix.prototype.sub;
    AbstractMatrix.prototype.subtractS = AbstractMatrix.prototype.subS;
    AbstractMatrix.prototype.subtractM = AbstractMatrix.prototype.subM;
    AbstractMatrix.subtract = AbstractMatrix.sub;
    AbstractMatrix.prototype.mul = function mul(value) {
      if (typeof value === 'number') return this.mulS(value);
      return this.mulM(value);
    };
    AbstractMatrix.prototype.mulS = function mulS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * value);
        }
      }
      return this;
    };
    AbstractMatrix.prototype.mulM = function mulM(matrix) {
      matrix = Matrix.checkMatrix(matrix);
      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * matrix.get(i, j));
        }
      }
      return this;
    };
    AbstractMatrix.mul = function mul(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.mul(value);
    };
    AbstractMatrix.prototype.multiply = AbstractMatrix.prototype.mul;
    AbstractMatrix.prototype.multiplyS = AbstractMatrix.prototype.mulS;
    AbstractMatrix.prototype.multiplyM = AbstractMatrix.prototype.mulM;
    AbstractMatrix.multiply = AbstractMatrix.mul;
    AbstractMatrix.prototype.div = function div(value) {
      if (typeof value === 'number') return this.divS(value);
      return this.divM(value);
    };
    AbstractMatrix.prototype.divS = function divS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / value);
        }
      }
      return this;
    };
    AbstractMatrix.prototype.divM = function divM(matrix) {
      matrix = Matrix.checkMatrix(matrix);
      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / matrix.get(i, j));
        }
      }
      return this;
    };
    AbstractMatrix.div = function div(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.div(value);
    };
    AbstractMatrix.prototype.divide = AbstractMatrix.prototype.div;
    AbstractMatrix.prototype.divideS = AbstractMatrix.prototype.divS;
    AbstractMatrix.prototype.divideM = AbstractMatrix.prototype.divM;
    AbstractMatrix.divide = AbstractMatrix.div;
    AbstractMatrix.prototype.mod = function mod(value) {
      if (typeof value === 'number') return this.modS(value);
      return this.modM(value);
    };
    AbstractMatrix.prototype.modS = function modS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) % value);
        }
      }
      return this;
    };
    AbstractMatrix.prototype.modM = function modM(matrix) {
      matrix = Matrix.checkMatrix(matrix);
      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) % matrix.get(i, j));
        }
      }
      return this;
    };
    AbstractMatrix.mod = function mod(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.mod(value);
    };
    AbstractMatrix.prototype.modulus = AbstractMatrix.prototype.mod;
    AbstractMatrix.prototype.modulusS = AbstractMatrix.prototype.modS;
    AbstractMatrix.prototype.modulusM = AbstractMatrix.prototype.modM;
    AbstractMatrix.modulus = AbstractMatrix.mod;
    AbstractMatrix.prototype.and = function and(value) {
      if (typeof value === 'number') return this.andS(value);
      return this.andM(value);
    };
    AbstractMatrix.prototype.andS = function andS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) & value);
        }
      }
      return this;
    };
    AbstractMatrix.prototype.andM = function andM(matrix) {
      matrix = Matrix.checkMatrix(matrix);
      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) & matrix.get(i, j));
        }
      }
      return this;
    };
    AbstractMatrix.and = function and(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.and(value);
    };
    AbstractMatrix.prototype.or = function or(value) {
      if (typeof value === 'number') return this.orS(value);
      return this.orM(value);
    };
    AbstractMatrix.prototype.orS = function orS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) | value);
        }
      }
      return this;
    };
    AbstractMatrix.prototype.orM = function orM(matrix) {
      matrix = Matrix.checkMatrix(matrix);
      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) | matrix.get(i, j));
        }
      }
      return this;
    };
    AbstractMatrix.or = function or(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.or(value);
    };
    AbstractMatrix.prototype.xor = function xor(value) {
      if (typeof value === 'number') return this.xorS(value);
      return this.xorM(value);
    };
    AbstractMatrix.prototype.xorS = function xorS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) ^ value);
        }
      }
      return this;
    };
    AbstractMatrix.prototype.xorM = function xorM(matrix) {
      matrix = Matrix.checkMatrix(matrix);
      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) ^ matrix.get(i, j));
        }
      }
      return this;
    };
    AbstractMatrix.xor = function xor(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.xor(value);
    };
    AbstractMatrix.prototype.leftShift = function leftShift(value) {
      if (typeof value === 'number') return this.leftShiftS(value);
      return this.leftShiftM(value);
    };
    AbstractMatrix.prototype.leftShiftS = function leftShiftS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) << value);
        }
      }
      return this;
    };
    AbstractMatrix.prototype.leftShiftM = function leftShiftM(matrix) {
      matrix = Matrix.checkMatrix(matrix);
      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) << matrix.get(i, j));
        }
      }
      return this;
    };
    AbstractMatrix.leftShift = function leftShift(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.leftShift(value);
    };
    AbstractMatrix.prototype.signPropagatingRightShift = function signPropagatingRightShift(value) {
      if (typeof value === 'number') return this.signPropagatingRightShiftS(value);
      return this.signPropagatingRightShiftM(value);
    };
    AbstractMatrix.prototype.signPropagatingRightShiftS = function signPropagatingRightShiftS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >> value);
        }
      }
      return this;
    };
    AbstractMatrix.prototype.signPropagatingRightShiftM = function signPropagatingRightShiftM(matrix) {
      matrix = Matrix.checkMatrix(matrix);
      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >> matrix.get(i, j));
        }
      }
      return this;
    };
    AbstractMatrix.signPropagatingRightShift = function signPropagatingRightShift(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.signPropagatingRightShift(value);
    };
    AbstractMatrix.prototype.rightShift = function rightShift(value) {
      if (typeof value === 'number') return this.rightShiftS(value);
      return this.rightShiftM(value);
    };
    AbstractMatrix.prototype.rightShiftS = function rightShiftS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >>> value);
        }
      }
      return this;
    };
    AbstractMatrix.prototype.rightShiftM = function rightShiftM(matrix) {
      matrix = Matrix.checkMatrix(matrix);
      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) >>> matrix.get(i, j));
        }
      }
      return this;
    };
    AbstractMatrix.rightShift = function rightShift(matrix, value) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.rightShift(value);
    };
    AbstractMatrix.prototype.zeroFillRightShift = AbstractMatrix.prototype.rightShift;
    AbstractMatrix.prototype.zeroFillRightShiftS = AbstractMatrix.prototype.rightShiftS;
    AbstractMatrix.prototype.zeroFillRightShiftM = AbstractMatrix.prototype.rightShiftM;
    AbstractMatrix.zeroFillRightShift = AbstractMatrix.rightShift;
    AbstractMatrix.prototype.not = function not() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, ~this.get(i, j));
        }
      }
      return this;
    };
    AbstractMatrix.not = function not(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.not();
    };
    AbstractMatrix.prototype.abs = function abs() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.abs(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.abs = function abs(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.abs();
    };
    AbstractMatrix.prototype.acos = function acos() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.acos(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.acos = function acos(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.acos();
    };
    AbstractMatrix.prototype.acosh = function acosh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.acosh(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.acosh = function acosh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.acosh();
    };
    AbstractMatrix.prototype.asin = function asin() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.asin(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.asin = function asin(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.asin();
    };
    AbstractMatrix.prototype.asinh = function asinh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.asinh(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.asinh = function asinh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.asinh();
    };
    AbstractMatrix.prototype.atan = function atan() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.atan(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.atan = function atan(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.atan();
    };
    AbstractMatrix.prototype.atanh = function atanh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.atanh(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.atanh = function atanh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.atanh();
    };
    AbstractMatrix.prototype.cbrt = function cbrt() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.cbrt(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.cbrt = function cbrt(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.cbrt();
    };
    AbstractMatrix.prototype.ceil = function ceil() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.ceil(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.ceil = function ceil(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.ceil();
    };
    AbstractMatrix.prototype.clz32 = function clz32() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.clz32(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.clz32 = function clz32(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.clz32();
    };
    AbstractMatrix.prototype.cos = function cos() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.cos(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.cos = function cos(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.cos();
    };
    AbstractMatrix.prototype.cosh = function cosh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.cosh(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.cosh = function cosh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.cosh();
    };
    AbstractMatrix.prototype.exp = function exp() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.exp(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.exp = function exp(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.exp();
    };
    AbstractMatrix.prototype.expm1 = function expm1() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.expm1(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.expm1 = function expm1(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.expm1();
    };
    AbstractMatrix.prototype.floor = function floor() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.floor(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.floor = function floor(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.floor();
    };
    AbstractMatrix.prototype.fround = function fround() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.fround(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.fround = function fround(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.fround();
    };
    AbstractMatrix.prototype.log = function log() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.log = function log(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log();
    };
    AbstractMatrix.prototype.log1p = function log1p() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log1p(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.log1p = function log1p(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log1p();
    };
    AbstractMatrix.prototype.log10 = function log10() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log10(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.log10 = function log10(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log10();
    };
    AbstractMatrix.prototype.log2 = function log2() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.log2(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.log2 = function log2(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.log2();
    };
    AbstractMatrix.prototype.round = function round() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.round(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.round = function round(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.round();
    };
    AbstractMatrix.prototype.sign = function sign() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sign(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.sign = function sign(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sign();
    };
    AbstractMatrix.prototype.sin = function sin() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sin(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.sin = function sin(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sin();
    };
    AbstractMatrix.prototype.sinh = function sinh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sinh(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.sinh = function sinh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sinh();
    };
    AbstractMatrix.prototype.sqrt = function sqrt() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.sqrt(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.sqrt = function sqrt(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.sqrt();
    };
    AbstractMatrix.prototype.tan = function tan() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.tan(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.tan = function tan(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.tan();
    };
    AbstractMatrix.prototype.tanh = function tanh() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.tanh(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.tanh = function tanh(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.tanh();
    };
    AbstractMatrix.prototype.trunc = function trunc() {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.trunc(this.get(i, j)));
        }
      }
      return this;
    };
    AbstractMatrix.trunc = function trunc(matrix) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.trunc();
    };
    AbstractMatrix.pow = function pow(matrix, arg0) {
      const newMatrix = new Matrix(matrix);
      return newMatrix.pow(arg0);
    };
    AbstractMatrix.prototype.pow = function pow(value) {
      if (typeof value === 'number') return this.powS(value);
      return this.powM(value);
    };
    AbstractMatrix.prototype.powS = function powS(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.pow(this.get(i, j), value));
        }
      }
      return this;
    };
    AbstractMatrix.prototype.powM = function powM(matrix) {
      matrix = Matrix.checkMatrix(matrix);
      if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
        throw new RangeError('Matrices dimensions must be equal');
      }
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, Math.pow(this.get(i, j), matrix.get(i, j)));
        }
      }
      return this;
    };
  }

  /**
   * @private
   * Check that a row index is not out of bounds
   * @param {Matrix} matrix
   * @param {number} index
   * @param {boolean} [outer]
   */
  function checkRowIndex(matrix, index, outer) {
    let max = outer ? matrix.rows : matrix.rows - 1;
    if (index < 0 || index > max) {
      throw new RangeError('Row index out of range');
    }
  }

  /**
   * @private
   * Check that a column index is not out of bounds
   * @param {Matrix} matrix
   * @param {number} index
   * @param {boolean} [outer]
   */
  function checkColumnIndex(matrix, index, outer) {
    let max = outer ? matrix.columns : matrix.columns - 1;
    if (index < 0 || index > max) {
      throw new RangeError('Column index out of range');
    }
  }

  /**
   * @private
   * Check that the provided vector is an array with the right length
   * @param {Matrix} matrix
   * @param {Array|Matrix} vector
   * @return {Array}
   * @throws {RangeError}
   */
  function checkRowVector(matrix, vector) {
    if (vector.to1DArray) {
      vector = vector.to1DArray();
    }
    if (vector.length !== matrix.columns) {
      throw new RangeError('vector size must be the same as the number of columns');
    }
    return vector;
  }

  /**
   * @private
   * Check that the provided vector is an array with the right length
   * @param {Matrix} matrix
   * @param {Array|Matrix} vector
   * @return {Array}
   * @throws {RangeError}
   */
  function checkColumnVector(matrix, vector) {
    if (vector.to1DArray) {
      vector = vector.to1DArray();
    }
    if (vector.length !== matrix.rows) {
      throw new RangeError('vector size must be the same as the number of rows');
    }
    return vector;
  }
  function checkRowIndices(matrix, rowIndices) {
    if (!isAnyArray(rowIndices)) {
      throw new TypeError('row indices must be an array');
    }
    for (let i = 0; i < rowIndices.length; i++) {
      if (rowIndices[i] < 0 || rowIndices[i] >= matrix.rows) {
        throw new RangeError('row indices are out of range');
      }
    }
  }
  function checkColumnIndices(matrix, columnIndices) {
    if (!isAnyArray(columnIndices)) {
      throw new TypeError('column indices must be an array');
    }
    for (let i = 0; i < columnIndices.length; i++) {
      if (columnIndices[i] < 0 || columnIndices[i] >= matrix.columns) {
        throw new RangeError('column indices are out of range');
      }
    }
  }
  function checkRange(matrix, startRow, endRow, startColumn, endColumn) {
    if (arguments.length !== 5) {
      throw new RangeError('expected 4 arguments');
    }
    checkNumber('startRow', startRow);
    checkNumber('endRow', endRow);
    checkNumber('startColumn', startColumn);
    checkNumber('endColumn', endColumn);
    if (startRow > endRow || startColumn > endColumn || startRow < 0 || startRow >= matrix.rows || endRow < 0 || endRow >= matrix.rows || startColumn < 0 || startColumn >= matrix.columns || endColumn < 0 || endColumn >= matrix.columns) {
      throw new RangeError('Submatrix indices are out of range');
    }
  }
  function newArray(length, value = 0) {
    let array = [];
    for (let i = 0; i < length; i++) {
      array.push(value);
    }
    return array;
  }
  function checkNumber(name, value) {
    if (typeof value !== 'number') {
      throw new TypeError(`${name} must be a number`);
    }
  }
  function checkNonEmpty(matrix) {
    if (matrix.isEmpty()) {
      throw new Error('Empty matrix has no elements to index');
    }
  }

  function sumByRow(matrix) {
    let sum = newArray(matrix.rows);
    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[i] += matrix.get(i, j);
      }
    }
    return sum;
  }
  function sumByColumn(matrix) {
    let sum = newArray(matrix.columns);
    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[j] += matrix.get(i, j);
      }
    }
    return sum;
  }
  function sumAll(matrix) {
    let v = 0;
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        v += matrix.get(i, j);
      }
    }
    return v;
  }
  function productByRow(matrix) {
    let sum = newArray(matrix.rows, 1);
    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[i] *= matrix.get(i, j);
      }
    }
    return sum;
  }
  function productByColumn(matrix) {
    let sum = newArray(matrix.columns, 1);
    for (let i = 0; i < matrix.rows; ++i) {
      for (let j = 0; j < matrix.columns; ++j) {
        sum[j] *= matrix.get(i, j);
      }
    }
    return sum;
  }
  function productAll(matrix) {
    let v = 1;
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        v *= matrix.get(i, j);
      }
    }
    return v;
  }
  function varianceByRow(matrix, unbiased, mean) {
    const rows = matrix.rows;
    const cols = matrix.columns;
    const variance = [];
    for (let i = 0; i < rows; i++) {
      let sum1 = 0;
      let sum2 = 0;
      let x = 0;
      for (let j = 0; j < cols; j++) {
        x = matrix.get(i, j) - mean[i];
        sum1 += x;
        sum2 += x * x;
      }
      if (unbiased) {
        variance.push((sum2 - sum1 * sum1 / cols) / (cols - 1));
      } else {
        variance.push((sum2 - sum1 * sum1 / cols) / cols);
      }
    }
    return variance;
  }
  function varianceByColumn(matrix, unbiased, mean) {
    const rows = matrix.rows;
    const cols = matrix.columns;
    const variance = [];
    for (let j = 0; j < cols; j++) {
      let sum1 = 0;
      let sum2 = 0;
      let x = 0;
      for (let i = 0; i < rows; i++) {
        x = matrix.get(i, j) - mean[j];
        sum1 += x;
        sum2 += x * x;
      }
      if (unbiased) {
        variance.push((sum2 - sum1 * sum1 / rows) / (rows - 1));
      } else {
        variance.push((sum2 - sum1 * sum1 / rows) / rows);
      }
    }
    return variance;
  }
  function varianceAll(matrix, unbiased, mean) {
    const rows = matrix.rows;
    const cols = matrix.columns;
    const size = rows * cols;
    let sum1 = 0;
    let sum2 = 0;
    let x = 0;
    for (let i = 0; i < rows; i++) {
      for (let j = 0; j < cols; j++) {
        x = matrix.get(i, j) - mean;
        sum1 += x;
        sum2 += x * x;
      }
    }
    if (unbiased) {
      return (sum2 - sum1 * sum1 / size) / (size - 1);
    } else {
      return (sum2 - sum1 * sum1 / size) / size;
    }
  }
  function centerByRow(matrix, mean) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) - mean[i]);
      }
    }
  }
  function centerByColumn(matrix, mean) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) - mean[j]);
      }
    }
  }
  function centerAll(matrix, mean) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) - mean);
      }
    }
  }
  function getScaleByRow(matrix) {
    const scale = [];
    for (let i = 0; i < matrix.rows; i++) {
      let sum = 0;
      for (let j = 0; j < matrix.columns; j++) {
        sum += Math.pow(matrix.get(i, j), 2) / (matrix.columns - 1);
      }
      scale.push(Math.sqrt(sum));
    }
    return scale;
  }
  function scaleByRow(matrix, scale) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) / scale[i]);
      }
    }
  }
  function getScaleByColumn(matrix) {
    const scale = [];
    for (let j = 0; j < matrix.columns; j++) {
      let sum = 0;
      for (let i = 0; i < matrix.rows; i++) {
        sum += Math.pow(matrix.get(i, j), 2) / (matrix.rows - 1);
      }
      scale.push(Math.sqrt(sum));
    }
    return scale;
  }
  function scaleByColumn(matrix, scale) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) / scale[j]);
      }
    }
  }
  function getScaleAll(matrix) {
    const divider = matrix.size - 1;
    let sum = 0;
    for (let j = 0; j < matrix.columns; j++) {
      for (let i = 0; i < matrix.rows; i++) {
        sum += Math.pow(matrix.get(i, j), 2) / divider;
      }
    }
    return Math.sqrt(sum);
  }
  function scaleAll(matrix, scale) {
    for (let i = 0; i < matrix.rows; i++) {
      for (let j = 0; j < matrix.columns; j++) {
        matrix.set(i, j, matrix.get(i, j) / scale);
      }
    }
  }

  class AbstractMatrix {
    static from1DArray(newRows, newColumns, newData) {
      let length = newRows * newColumns;
      if (length !== newData.length) {
        throw new RangeError('data length does not match given dimensions');
      }
      let newMatrix = new Matrix(newRows, newColumns);
      for (let row = 0; row < newRows; row++) {
        for (let column = 0; column < newColumns; column++) {
          newMatrix.set(row, column, newData[row * newColumns + column]);
        }
      }
      return newMatrix;
    }
    static rowVector(newData) {
      let vector = new Matrix(1, newData.length);
      for (let i = 0; i < newData.length; i++) {
        vector.set(0, i, newData[i]);
      }
      return vector;
    }
    static columnVector(newData) {
      let vector = new Matrix(newData.length, 1);
      for (let i = 0; i < newData.length; i++) {
        vector.set(i, 0, newData[i]);
      }
      return vector;
    }
    static zeros(rows, columns) {
      return new Matrix(rows, columns);
    }
    static ones(rows, columns) {
      return new Matrix(rows, columns).fill(1);
    }
    static rand(rows, columns, options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }
      const {
        random = Math.random
      } = options;
      let matrix = new Matrix(rows, columns);
      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          matrix.set(i, j, random());
        }
      }
      return matrix;
    }
    static randInt(rows, columns, options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }
      const {
        min = 0,
        max = 1000,
        random = Math.random
      } = options;
      if (!Number.isInteger(min)) throw new TypeError('min must be an integer');
      if (!Number.isInteger(max)) throw new TypeError('max must be an integer');
      if (min >= max) throw new RangeError('min must be smaller than max');
      let interval = max - min;
      let matrix = new Matrix(rows, columns);
      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          let value = min + Math.round(random() * interval);
          matrix.set(i, j, value);
        }
      }
      return matrix;
    }
    static eye(rows, columns, value) {
      if (columns === undefined) columns = rows;
      if (value === undefined) value = 1;
      let min = Math.min(rows, columns);
      let matrix = this.zeros(rows, columns);
      for (let i = 0; i < min; i++) {
        matrix.set(i, i, value);
      }
      return matrix;
    }
    static diag(data, rows, columns) {
      let l = data.length;
      if (rows === undefined) rows = l;
      if (columns === undefined) columns = rows;
      let min = Math.min(l, rows, columns);
      let matrix = this.zeros(rows, columns);
      for (let i = 0; i < min; i++) {
        matrix.set(i, i, data[i]);
      }
      return matrix;
    }
    static min(matrix1, matrix2) {
      matrix1 = this.checkMatrix(matrix1);
      matrix2 = this.checkMatrix(matrix2);
      let rows = matrix1.rows;
      let columns = matrix1.columns;
      let result = new Matrix(rows, columns);
      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          result.set(i, j, Math.min(matrix1.get(i, j), matrix2.get(i, j)));
        }
      }
      return result;
    }
    static max(matrix1, matrix2) {
      matrix1 = this.checkMatrix(matrix1);
      matrix2 = this.checkMatrix(matrix2);
      let rows = matrix1.rows;
      let columns = matrix1.columns;
      let result = new this(rows, columns);
      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          result.set(i, j, Math.max(matrix1.get(i, j), matrix2.get(i, j)));
        }
      }
      return result;
    }
    static checkMatrix(value) {
      return AbstractMatrix.isMatrix(value) ? value : new Matrix(value);
    }
    static isMatrix(value) {
      return value != null && value.klass === 'Matrix';
    }
    get size() {
      return this.rows * this.columns;
    }
    apply(callback) {
      if (typeof callback !== 'function') {
        throw new TypeError('callback must be a function');
      }
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          callback.call(this, i, j);
        }
      }
      return this;
    }
    to1DArray() {
      let array = [];
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          array.push(this.get(i, j));
        }
      }
      return array;
    }
    to2DArray() {
      let copy = [];
      for (let i = 0; i < this.rows; i++) {
        copy.push([]);
        for (let j = 0; j < this.columns; j++) {
          copy[i].push(this.get(i, j));
        }
      }
      return copy;
    }
    toJSON() {
      return this.to2DArray();
    }
    isRowVector() {
      return this.rows === 1;
    }
    isColumnVector() {
      return this.columns === 1;
    }
    isVector() {
      return this.rows === 1 || this.columns === 1;
    }
    isSquare() {
      return this.rows === this.columns;
    }
    isEmpty() {
      return this.rows === 0 || this.columns === 0;
    }
    isSymmetric() {
      if (this.isSquare()) {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j <= i; j++) {
            if (this.get(i, j) !== this.get(j, i)) {
              return false;
            }
          }
        }
        return true;
      }
      return false;
    }
    isEchelonForm() {
      let i = 0;
      let j = 0;
      let previousColumn = -1;
      let isEchelonForm = true;
      let checked = false;
      while (i < this.rows && isEchelonForm) {
        j = 0;
        checked = false;
        while (j < this.columns && checked === false) {
          if (this.get(i, j) === 0) {
            j++;
          } else if (this.get(i, j) === 1 && j > previousColumn) {
            checked = true;
            previousColumn = j;
          } else {
            isEchelonForm = false;
            checked = true;
          }
        }
        i++;
      }
      return isEchelonForm;
    }
    isReducedEchelonForm() {
      let i = 0;
      let j = 0;
      let previousColumn = -1;
      let isReducedEchelonForm = true;
      let checked = false;
      while (i < this.rows && isReducedEchelonForm) {
        j = 0;
        checked = false;
        while (j < this.columns && checked === false) {
          if (this.get(i, j) === 0) {
            j++;
          } else if (this.get(i, j) === 1 && j > previousColumn) {
            checked = true;
            previousColumn = j;
          } else {
            isReducedEchelonForm = false;
            checked = true;
          }
        }
        for (let k = j + 1; k < this.rows; k++) {
          if (this.get(i, k) !== 0) {
            isReducedEchelonForm = false;
          }
        }
        i++;
      }
      return isReducedEchelonForm;
    }
    echelonForm() {
      let result = this.clone();
      let h = 0;
      let k = 0;
      while (h < result.rows && k < result.columns) {
        let iMax = h;
        for (let i = h; i < result.rows; i++) {
          if (result.get(i, k) > result.get(iMax, k)) {
            iMax = i;
          }
        }
        if (result.get(iMax, k) === 0) {
          k++;
        } else {
          result.swapRows(h, iMax);
          let tmp = result.get(h, k);
          for (let j = k; j < result.columns; j++) {
            result.set(h, j, result.get(h, j) / tmp);
          }
          for (let i = h + 1; i < result.rows; i++) {
            let factor = result.get(i, k) / result.get(h, k);
            result.set(i, k, 0);
            for (let j = k + 1; j < result.columns; j++) {
              result.set(i, j, result.get(i, j) - result.get(h, j) * factor);
            }
          }
          h++;
          k++;
        }
      }
      return result;
    }
    reducedEchelonForm() {
      let result = this.echelonForm();
      let m = result.columns;
      let n = result.rows;
      let h = n - 1;
      while (h >= 0) {
        if (result.maxRow(h) === 0) {
          h--;
        } else {
          let p = 0;
          let pivot = false;
          while (p < n && pivot === false) {
            if (result.get(h, p) === 1) {
              pivot = true;
            } else {
              p++;
            }
          }
          for (let i = 0; i < h; i++) {
            let factor = result.get(i, p);
            for (let j = p; j < m; j++) {
              let tmp = result.get(i, j) - factor * result.get(h, j);
              result.set(i, j, tmp);
            }
          }
          h--;
        }
      }
      return result;
    }
    set() {
      throw new Error('set method is unimplemented');
    }
    get() {
      throw new Error('get method is unimplemented');
    }
    repeat(options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }
      const {
        rows = 1,
        columns = 1
      } = options;
      if (!Number.isInteger(rows) || rows <= 0) {
        throw new TypeError('rows must be a positive integer');
      }
      if (!Number.isInteger(columns) || columns <= 0) {
        throw new TypeError('columns must be a positive integer');
      }
      let matrix = new Matrix(this.rows * rows, this.columns * columns);
      for (let i = 0; i < rows; i++) {
        for (let j = 0; j < columns; j++) {
          matrix.setSubMatrix(this, this.rows * i, this.columns * j);
        }
      }
      return matrix;
    }
    fill(value) {
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, value);
        }
      }
      return this;
    }
    neg() {
      return this.mulS(-1);
    }
    getRow(index) {
      checkRowIndex(this, index);
      let row = [];
      for (let i = 0; i < this.columns; i++) {
        row.push(this.get(index, i));
      }
      return row;
    }
    getRowVector(index) {
      return Matrix.rowVector(this.getRow(index));
    }
    setRow(index, array) {
      checkRowIndex(this, index);
      array = checkRowVector(this, array);
      for (let i = 0; i < this.columns; i++) {
        this.set(index, i, array[i]);
      }
      return this;
    }
    swapRows(row1, row2) {
      checkRowIndex(this, row1);
      checkRowIndex(this, row2);
      for (let i = 0; i < this.columns; i++) {
        let temp = this.get(row1, i);
        this.set(row1, i, this.get(row2, i));
        this.set(row2, i, temp);
      }
      return this;
    }
    getColumn(index) {
      checkColumnIndex(this, index);
      let column = [];
      for (let i = 0; i < this.rows; i++) {
        column.push(this.get(i, index));
      }
      return column;
    }
    getColumnVector(index) {
      return Matrix.columnVector(this.getColumn(index));
    }
    setColumn(index, array) {
      checkColumnIndex(this, index);
      array = checkColumnVector(this, array);
      for (let i = 0; i < this.rows; i++) {
        this.set(i, index, array[i]);
      }
      return this;
    }
    swapColumns(column1, column2) {
      checkColumnIndex(this, column1);
      checkColumnIndex(this, column2);
      for (let i = 0; i < this.rows; i++) {
        let temp = this.get(i, column1);
        this.set(i, column1, this.get(i, column2));
        this.set(i, column2, temp);
      }
      return this;
    }
    addRowVector(vector) {
      vector = checkRowVector(this, vector);
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + vector[j]);
        }
      }
      return this;
    }
    subRowVector(vector) {
      vector = checkRowVector(this, vector);
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - vector[j]);
        }
      }
      return this;
    }
    mulRowVector(vector) {
      vector = checkRowVector(this, vector);
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * vector[j]);
        }
      }
      return this;
    }
    divRowVector(vector) {
      vector = checkRowVector(this, vector);
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / vector[j]);
        }
      }
      return this;
    }
    addColumnVector(vector) {
      vector = checkColumnVector(this, vector);
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) + vector[i]);
        }
      }
      return this;
    }
    subColumnVector(vector) {
      vector = checkColumnVector(this, vector);
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) - vector[i]);
        }
      }
      return this;
    }
    mulColumnVector(vector) {
      vector = checkColumnVector(this, vector);
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) * vector[i]);
        }
      }
      return this;
    }
    divColumnVector(vector) {
      vector = checkColumnVector(this, vector);
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          this.set(i, j, this.get(i, j) / vector[i]);
        }
      }
      return this;
    }
    mulRow(index, value) {
      checkRowIndex(this, index);
      for (let i = 0; i < this.columns; i++) {
        this.set(index, i, this.get(index, i) * value);
      }
      return this;
    }
    mulColumn(index, value) {
      checkColumnIndex(this, index);
      for (let i = 0; i < this.rows; i++) {
        this.set(i, index, this.get(i, index) * value);
      }
      return this;
    }
    max(by) {
      if (this.isEmpty()) {
        return NaN;
      }
      switch (by) {
        case 'row':
          {
            const max = new Array(this.rows).fill(Number.NEGATIVE_INFINITY);
            for (let row = 0; row < this.rows; row++) {
              for (let column = 0; column < this.columns; column++) {
                if (this.get(row, column) > max[row]) {
                  max[row] = this.get(row, column);
                }
              }
            }
            return max;
          }
        case 'column':
          {
            const max = new Array(this.columns).fill(Number.NEGATIVE_INFINITY);
            for (let row = 0; row < this.rows; row++) {
              for (let column = 0; column < this.columns; column++) {
                if (this.get(row, column) > max[column]) {
                  max[column] = this.get(row, column);
                }
              }
            }
            return max;
          }
        case undefined:
          {
            let max = this.get(0, 0);
            for (let row = 0; row < this.rows; row++) {
              for (let column = 0; column < this.columns; column++) {
                if (this.get(row, column) > max) {
                  max = this.get(row, column);
                }
              }
            }
            return max;
          }
        default:
          throw new Error(`invalid option: ${by}`);
      }
    }
    maxIndex() {
      checkNonEmpty(this);
      let v = this.get(0, 0);
      let idx = [0, 0];
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) > v) {
            v = this.get(i, j);
            idx[0] = i;
            idx[1] = j;
          }
        }
      }
      return idx;
    }
    min(by) {
      if (this.isEmpty()) {
        return NaN;
      }
      switch (by) {
        case 'row':
          {
            const min = new Array(this.rows).fill(Number.POSITIVE_INFINITY);
            for (let row = 0; row < this.rows; row++) {
              for (let column = 0; column < this.columns; column++) {
                if (this.get(row, column) < min[row]) {
                  min[row] = this.get(row, column);
                }
              }
            }
            return min;
          }
        case 'column':
          {
            const min = new Array(this.columns).fill(Number.POSITIVE_INFINITY);
            for (let row = 0; row < this.rows; row++) {
              for (let column = 0; column < this.columns; column++) {
                if (this.get(row, column) < min[column]) {
                  min[column] = this.get(row, column);
                }
              }
            }
            return min;
          }
        case undefined:
          {
            let min = this.get(0, 0);
            for (let row = 0; row < this.rows; row++) {
              for (let column = 0; column < this.columns; column++) {
                if (this.get(row, column) < min) {
                  min = this.get(row, column);
                }
              }
            }
            return min;
          }
        default:
          throw new Error(`invalid option: ${by}`);
      }
    }
    minIndex() {
      checkNonEmpty(this);
      let v = this.get(0, 0);
      let idx = [0, 0];
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          if (this.get(i, j) < v) {
            v = this.get(i, j);
            idx[0] = i;
            idx[1] = j;
          }
        }
      }
      return idx;
    }
    maxRow(row) {
      checkRowIndex(this, row);
      if (this.isEmpty()) {
        return NaN;
      }
      let v = this.get(row, 0);
      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) > v) {
          v = this.get(row, i);
        }
      }
      return v;
    }
    maxRowIndex(row) {
      checkRowIndex(this, row);
      checkNonEmpty(this);
      let v = this.get(row, 0);
      let idx = [row, 0];
      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) > v) {
          v = this.get(row, i);
          idx[1] = i;
        }
      }
      return idx;
    }
    minRow(row) {
      checkRowIndex(this, row);
      if (this.isEmpty()) {
        return NaN;
      }
      let v = this.get(row, 0);
      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) < v) {
          v = this.get(row, i);
        }
      }
      return v;
    }
    minRowIndex(row) {
      checkRowIndex(this, row);
      checkNonEmpty(this);
      let v = this.get(row, 0);
      let idx = [row, 0];
      for (let i = 1; i < this.columns; i++) {
        if (this.get(row, i) < v) {
          v = this.get(row, i);
          idx[1] = i;
        }
      }
      return idx;
    }
    maxColumn(column) {
      checkColumnIndex(this, column);
      if (this.isEmpty()) {
        return NaN;
      }
      let v = this.get(0, column);
      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) > v) {
          v = this.get(i, column);
        }
      }
      return v;
    }
    maxColumnIndex(column) {
      checkColumnIndex(this, column);
      checkNonEmpty(this);
      let v = this.get(0, column);
      let idx = [0, column];
      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) > v) {
          v = this.get(i, column);
          idx[0] = i;
        }
      }
      return idx;
    }
    minColumn(column) {
      checkColumnIndex(this, column);
      if (this.isEmpty()) {
        return NaN;
      }
      let v = this.get(0, column);
      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) < v) {
          v = this.get(i, column);
        }
      }
      return v;
    }
    minColumnIndex(column) {
      checkColumnIndex(this, column);
      checkNonEmpty(this);
      let v = this.get(0, column);
      let idx = [0, column];
      for (let i = 1; i < this.rows; i++) {
        if (this.get(i, column) < v) {
          v = this.get(i, column);
          idx[0] = i;
        }
      }
      return idx;
    }
    diag() {
      let min = Math.min(this.rows, this.columns);
      let diag = [];
      for (let i = 0; i < min; i++) {
        diag.push(this.get(i, i));
      }
      return diag;
    }
    norm(type = 'frobenius') {
      let result = 0;
      if (type === 'max') {
        return this.max();
      } else if (type === 'frobenius') {
        for (let i = 0; i < this.rows; i++) {
          for (let j = 0; j < this.columns; j++) {
            result = result + this.get(i, j) * this.get(i, j);
          }
        }
        return Math.sqrt(result);
      } else {
        throw new RangeError(`unknown norm type: ${type}`);
      }
    }
    cumulativeSum() {
      let sum = 0;
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          sum += this.get(i, j);
          this.set(i, j, sum);
        }
      }
      return this;
    }
    dot(vector2) {
      if (AbstractMatrix.isMatrix(vector2)) vector2 = vector2.to1DArray();
      let vector1 = this.to1DArray();
      if (vector1.length !== vector2.length) {
        throw new RangeError('vectors do not have the same size');
      }
      let dot = 0;
      for (let i = 0; i < vector1.length; i++) {
        dot += vector1[i] * vector2[i];
      }
      return dot;
    }
    mmul(other) {
      other = Matrix.checkMatrix(other);
      let m = this.rows;
      let n = this.columns;
      let p = other.columns;
      let result = new Matrix(m, p);
      let Bcolj = new Float64Array(n);
      for (let j = 0; j < p; j++) {
        for (let k = 0; k < n; k++) {
          Bcolj[k] = other.get(k, j);
        }
        for (let i = 0; i < m; i++) {
          let s = 0;
          for (let k = 0; k < n; k++) {
            s += this.get(i, k) * Bcolj[k];
          }
          result.set(i, j, s);
        }
      }
      return result;
    }
    strassen2x2(other) {
      other = Matrix.checkMatrix(other);
      let result = new Matrix(2, 2);
      const a11 = this.get(0, 0);
      const b11 = other.get(0, 0);
      const a12 = this.get(0, 1);
      const b12 = other.get(0, 1);
      const a21 = this.get(1, 0);
      const b21 = other.get(1, 0);
      const a22 = this.get(1, 1);
      const b22 = other.get(1, 1);

      // Compute intermediate values.
      const m1 = (a11 + a22) * (b11 + b22);
      const m2 = (a21 + a22) * b11;
      const m3 = a11 * (b12 - b22);
      const m4 = a22 * (b21 - b11);
      const m5 = (a11 + a12) * b22;
      const m6 = (a21 - a11) * (b11 + b12);
      const m7 = (a12 - a22) * (b21 + b22);

      // Combine intermediate values into the output.
      const c00 = m1 + m4 - m5 + m7;
      const c01 = m3 + m5;
      const c10 = m2 + m4;
      const c11 = m1 - m2 + m3 + m6;
      result.set(0, 0, c00);
      result.set(0, 1, c01);
      result.set(1, 0, c10);
      result.set(1, 1, c11);
      return result;
    }
    strassen3x3(other) {
      other = Matrix.checkMatrix(other);
      let result = new Matrix(3, 3);
      const a00 = this.get(0, 0);
      const a01 = this.get(0, 1);
      const a02 = this.get(0, 2);
      const a10 = this.get(1, 0);
      const a11 = this.get(1, 1);
      const a12 = this.get(1, 2);
      const a20 = this.get(2, 0);
      const a21 = this.get(2, 1);
      const a22 = this.get(2, 2);
      const b00 = other.get(0, 0);
      const b01 = other.get(0, 1);
      const b02 = other.get(0, 2);
      const b10 = other.get(1, 0);
      const b11 = other.get(1, 1);
      const b12 = other.get(1, 2);
      const b20 = other.get(2, 0);
      const b21 = other.get(2, 1);
      const b22 = other.get(2, 2);
      const m1 = (a00 + a01 + a02 - a10 - a11 - a21 - a22) * b11;
      const m2 = (a00 - a10) * (-b01 + b11);
      const m3 = a11 * (-b00 + b01 + b10 - b11 - b12 - b20 + b22);
      const m4 = (-a00 + a10 + a11) * (b00 - b01 + b11);
      const m5 = (a10 + a11) * (-b00 + b01);
      const m6 = a00 * b00;
      const m7 = (-a00 + a20 + a21) * (b00 - b02 + b12);
      const m8 = (-a00 + a20) * (b02 - b12);
      const m9 = (a20 + a21) * (-b00 + b02);
      const m10 = (a00 + a01 + a02 - a11 - a12 - a20 - a21) * b12;
      const m11 = a21 * (-b00 + b02 + b10 - b11 - b12 - b20 + b21);
      const m12 = (-a02 + a21 + a22) * (b11 + b20 - b21);
      const m13 = (a02 - a22) * (b11 - b21);
      const m14 = a02 * b20;
      const m15 = (a21 + a22) * (-b20 + b21);
      const m16 = (-a02 + a11 + a12) * (b12 + b20 - b22);
      const m17 = (a02 - a12) * (b12 - b22);
      const m18 = (a11 + a12) * (-b20 + b22);
      const m19 = a01 * b10;
      const m20 = a12 * b21;
      const m21 = a10 * b02;
      const m22 = a20 * b01;
      const m23 = a22 * b22;
      const c00 = m6 + m14 + m19;
      const c01 = m1 + m4 + m5 + m6 + m12 + m14 + m15;
      const c02 = m6 + m7 + m9 + m10 + m14 + m16 + m18;
      const c10 = m2 + m3 + m4 + m6 + m14 + m16 + m17;
      const c11 = m2 + m4 + m5 + m6 + m20;
      const c12 = m14 + m16 + m17 + m18 + m21;
      const c20 = m6 + m7 + m8 + m11 + m12 + m13 + m14;
      const c21 = m12 + m13 + m14 + m15 + m22;
      const c22 = m6 + m7 + m8 + m9 + m23;
      result.set(0, 0, c00);
      result.set(0, 1, c01);
      result.set(0, 2, c02);
      result.set(1, 0, c10);
      result.set(1, 1, c11);
      result.set(1, 2, c12);
      result.set(2, 0, c20);
      result.set(2, 1, c21);
      result.set(2, 2, c22);
      return result;
    }
    mmulStrassen(y) {
      y = Matrix.checkMatrix(y);
      let x = this.clone();
      let r1 = x.rows;
      let c1 = x.columns;
      let r2 = y.rows;
      let c2 = y.columns;
      if (c1 !== r2) {
        // eslint-disable-next-line no-console
        console.warn(`Multiplying ${r1} x ${c1} and ${r2} x ${c2} matrix: dimensions do not match.`);
      }

      // Put a matrix into the top left of a matrix of zeros.
      // `rows` and `cols` are the dimensions of the output matrix.
      function embed(mat, rows, cols) {
        let r = mat.rows;
        let c = mat.columns;
        if (r === rows && c === cols) {
          return mat;
        } else {
          let resultat = AbstractMatrix.zeros(rows, cols);
          resultat = resultat.setSubMatrix(mat, 0, 0);
          return resultat;
        }
      }

      // Make sure both matrices are the same size.
      // This is exclusively for simplicity:
      // this algorithm can be implemented with matrices of different sizes.

      let r = Math.max(r1, r2);
      let c = Math.max(c1, c2);
      x = embed(x, r, c);
      y = embed(y, r, c);

      // Our recursive multiplication function.
      function blockMult(a, b, rows, cols) {
        // For small matrices, resort to naive multiplication.
        if (rows <= 512 || cols <= 512) {
          return a.mmul(b); // a is equivalent to this
        }

        // Apply dynamic padding.
        if (rows % 2 === 1 && cols % 2 === 1) {
          a = embed(a, rows + 1, cols + 1);
          b = embed(b, rows + 1, cols + 1);
        } else if (rows % 2 === 1) {
          a = embed(a, rows + 1, cols);
          b = embed(b, rows + 1, cols);
        } else if (cols % 2 === 1) {
          a = embed(a, rows, cols + 1);
          b = embed(b, rows, cols + 1);
        }
        let halfRows = parseInt(a.rows / 2, 10);
        let halfCols = parseInt(a.columns / 2, 10);
        // Subdivide input matrices.
        let a11 = a.subMatrix(0, halfRows - 1, 0, halfCols - 1);
        let b11 = b.subMatrix(0, halfRows - 1, 0, halfCols - 1);
        let a12 = a.subMatrix(0, halfRows - 1, halfCols, a.columns - 1);
        let b12 = b.subMatrix(0, halfRows - 1, halfCols, b.columns - 1);
        let a21 = a.subMatrix(halfRows, a.rows - 1, 0, halfCols - 1);
        let b21 = b.subMatrix(halfRows, b.rows - 1, 0, halfCols - 1);
        let a22 = a.subMatrix(halfRows, a.rows - 1, halfCols, a.columns - 1);
        let b22 = b.subMatrix(halfRows, b.rows - 1, halfCols, b.columns - 1);

        // Compute intermediate values.
        let m1 = blockMult(AbstractMatrix.add(a11, a22), AbstractMatrix.add(b11, b22), halfRows, halfCols);
        let m2 = blockMult(AbstractMatrix.add(a21, a22), b11, halfRows, halfCols);
        let m3 = blockMult(a11, AbstractMatrix.sub(b12, b22), halfRows, halfCols);
        let m4 = blockMult(a22, AbstractMatrix.sub(b21, b11), halfRows, halfCols);
        let m5 = blockMult(AbstractMatrix.add(a11, a12), b22, halfRows, halfCols);
        let m6 = blockMult(AbstractMatrix.sub(a21, a11), AbstractMatrix.add(b11, b12), halfRows, halfCols);
        let m7 = blockMult(AbstractMatrix.sub(a12, a22), AbstractMatrix.add(b21, b22), halfRows, halfCols);

        // Combine intermediate values into the output.
        let c11 = AbstractMatrix.add(m1, m4);
        c11.sub(m5);
        c11.add(m7);
        let c12 = AbstractMatrix.add(m3, m5);
        let c21 = AbstractMatrix.add(m2, m4);
        let c22 = AbstractMatrix.sub(m1, m2);
        c22.add(m3);
        c22.add(m6);

        // Crop output to the desired size (undo dynamic padding).
        let resultat = AbstractMatrix.zeros(2 * c11.rows, 2 * c11.columns);
        resultat = resultat.setSubMatrix(c11, 0, 0);
        resultat = resultat.setSubMatrix(c12, c11.rows, 0);
        resultat = resultat.setSubMatrix(c21, 0, c11.columns);
        resultat = resultat.setSubMatrix(c22, c11.rows, c11.columns);
        return resultat.subMatrix(0, rows - 1, 0, cols - 1);
      }
      return blockMult(x, y, r, c);
    }
    scaleRows(options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }
      const {
        min = 0,
        max = 1
      } = options;
      if (!Number.isFinite(min)) throw new TypeError('min must be a number');
      if (!Number.isFinite(max)) throw new TypeError('max must be a number');
      if (min >= max) throw new RangeError('min must be smaller than max');
      let newMatrix = new Matrix(this.rows, this.columns);
      for (let i = 0; i < this.rows; i++) {
        const row = this.getRow(i);
        if (row.length > 0) {
          rescale(row, {
            min,
            max,
            output: row
          });
        }
        newMatrix.setRow(i, row);
      }
      return newMatrix;
    }
    scaleColumns(options = {}) {
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }
      const {
        min = 0,
        max = 1
      } = options;
      if (!Number.isFinite(min)) throw new TypeError('min must be a number');
      if (!Number.isFinite(max)) throw new TypeError('max must be a number');
      if (min >= max) throw new RangeError('min must be smaller than max');
      let newMatrix = new Matrix(this.rows, this.columns);
      for (let i = 0; i < this.columns; i++) {
        const column = this.getColumn(i);
        if (column.length) {
          rescale(column, {
            min: min,
            max: max,
            output: column
          });
        }
        newMatrix.setColumn(i, column);
      }
      return newMatrix;
    }
    flipRows() {
      const middle = Math.ceil(this.columns / 2);
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < middle; j++) {
          let first = this.get(i, j);
          let last = this.get(i, this.columns - 1 - j);
          this.set(i, j, last);
          this.set(i, this.columns - 1 - j, first);
        }
      }
      return this;
    }
    flipColumns() {
      const middle = Math.ceil(this.rows / 2);
      for (let j = 0; j < this.columns; j++) {
        for (let i = 0; i < middle; i++) {
          let first = this.get(i, j);
          let last = this.get(this.rows - 1 - i, j);
          this.set(i, j, last);
          this.set(this.rows - 1 - i, j, first);
        }
      }
      return this;
    }
    kroneckerProduct(other) {
      other = Matrix.checkMatrix(other);
      let m = this.rows;
      let n = this.columns;
      let p = other.rows;
      let q = other.columns;
      let result = new Matrix(m * p, n * q);
      for (let i = 0; i < m; i++) {
        for (let j = 0; j < n; j++) {
          for (let k = 0; k < p; k++) {
            for (let l = 0; l < q; l++) {
              result.set(p * i + k, q * j + l, this.get(i, j) * other.get(k, l));
            }
          }
        }
      }
      return result;
    }
    kroneckerSum(other) {
      other = Matrix.checkMatrix(other);
      if (!this.isSquare() || !other.isSquare()) {
        throw new Error('Kronecker Sum needs two Square Matrices');
      }
      let m = this.rows;
      let n = other.rows;
      let AxI = this.kroneckerProduct(Matrix.eye(n, n));
      let IxB = Matrix.eye(m, m).kroneckerProduct(other);
      return AxI.add(IxB);
    }
    transpose() {
      let result = new Matrix(this.columns, this.rows);
      for (let i = 0; i < this.rows; i++) {
        for (let j = 0; j < this.columns; j++) {
          result.set(j, i, this.get(i, j));
        }
      }
      return result;
    }
    sortRows(compareFunction = compareNumbers) {
      for (let i = 0; i < this.rows; i++) {
        this.setRow(i, this.getRow(i).sort(compareFunction));
      }
      return this;
    }
    sortColumns(compareFunction = compareNumbers) {
      for (let i = 0; i < this.columns; i++) {
        this.setColumn(i, this.getColumn(i).sort(compareFunction));
      }
      return this;
    }
    subMatrix(startRow, endRow, startColumn, endColumn) {
      checkRange(this, startRow, endRow, startColumn, endColumn);
      let newMatrix = new Matrix(endRow - startRow + 1, endColumn - startColumn + 1);
      for (let i = startRow; i <= endRow; i++) {
        for (let j = startColumn; j <= endColumn; j++) {
          newMatrix.set(i - startRow, j - startColumn, this.get(i, j));
        }
      }
      return newMatrix;
    }
    subMatrixRow(indices, startColumn, endColumn) {
      if (startColumn === undefined) startColumn = 0;
      if (endColumn === undefined) endColumn = this.columns - 1;
      if (startColumn > endColumn || startColumn < 0 || startColumn >= this.columns || endColumn < 0 || endColumn >= this.columns) {
        throw new RangeError('Argument out of range');
      }
      let newMatrix = new Matrix(indices.length, endColumn - startColumn + 1);
      for (let i = 0; i < indices.length; i++) {
        for (let j = startColumn; j <= endColumn; j++) {
          if (indices[i] < 0 || indices[i] >= this.rows) {
            throw new RangeError(`Row index out of range: ${indices[i]}`);
          }
          newMatrix.set(i, j - startColumn, this.get(indices[i], j));
        }
      }
      return newMatrix;
    }
    subMatrixColumn(indices, startRow, endRow) {
      if (startRow === undefined) startRow = 0;
      if (endRow === undefined) endRow = this.rows - 1;
      if (startRow > endRow || startRow < 0 || startRow >= this.rows || endRow < 0 || endRow >= this.rows) {
        throw new RangeError('Argument out of range');
      }
      let newMatrix = new Matrix(endRow - startRow + 1, indices.length);
      for (let i = 0; i < indices.length; i++) {
        for (let j = startRow; j <= endRow; j++) {
          if (indices[i] < 0 || indices[i] >= this.columns) {
            throw new RangeError(`Column index out of range: ${indices[i]}`);
          }
          newMatrix.set(j - startRow, i, this.get(j, indices[i]));
        }
      }
      return newMatrix;
    }
    setSubMatrix(matrix, startRow, startColumn) {
      matrix = Matrix.checkMatrix(matrix);
      if (matrix.isEmpty()) {
        return this;
      }
      let endRow = startRow + matrix.rows - 1;
      let endColumn = startColumn + matrix.columns - 1;
      checkRange(this, startRow, endRow, startColumn, endColumn);
      for (let i = 0; i < matrix.rows; i++) {
        for (let j = 0; j < matrix.columns; j++) {
          this.set(startRow + i, startColumn + j, matrix.get(i, j));
        }
      }
      return this;
    }
    selection(rowIndices, columnIndices) {
      checkRowIndices(this, rowIndices);
      checkColumnIndices(this, columnIndices);
      let newMatrix = new Matrix(rowIndices.length, columnIndices.length);
      for (let i = 0; i < rowIndices.length; i++) {
        let rowIndex = rowIndices[i];
        for (let j = 0; j < columnIndices.length; j++) {
          let columnIndex = columnIndices[j];
          newMatrix.set(i, j, this.get(rowIndex, columnIndex));
        }
      }
      return newMatrix;
    }
    trace() {
      let min = Math.min(this.rows, this.columns);
      let trace = 0;
      for (let i = 0; i < min; i++) {
        trace += this.get(i, i);
      }
      return trace;
    }
    clone() {
      let newMatrix = new Matrix(this.rows, this.columns);
      for (let row = 0; row < this.rows; row++) {
        for (let column = 0; column < this.columns; column++) {
          newMatrix.set(row, column, this.get(row, column));
        }
      }
      return newMatrix;
    }
    sum(by) {
      switch (by) {
        case 'row':
          return sumByRow(this);
        case 'column':
          return sumByColumn(this);
        case undefined:
          return sumAll(this);
        default:
          throw new Error(`invalid option: ${by}`);
      }
    }
    product(by) {
      switch (by) {
        case 'row':
          return productByRow(this);
        case 'column':
          return productByColumn(this);
        case undefined:
          return productAll(this);
        default:
          throw new Error(`invalid option: ${by}`);
      }
    }
    mean(by) {
      const sum = this.sum(by);
      switch (by) {
        case 'row':
          {
            for (let i = 0; i < this.rows; i++) {
              sum[i] /= this.columns;
            }
            return sum;
          }
        case 'column':
          {
            for (let i = 0; i < this.columns; i++) {
              sum[i] /= this.rows;
            }
            return sum;
          }
        case undefined:
          return sum / this.size;
        default:
          throw new Error(`invalid option: ${by}`);
      }
    }
    variance(by, options = {}) {
      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }
      const {
        unbiased = true,
        mean = this.mean(by)
      } = options;
      if (typeof unbiased !== 'boolean') {
        throw new TypeError('unbiased must be a boolean');
      }
      switch (by) {
        case 'row':
          {
            if (!isAnyArray(mean)) {
              throw new TypeError('mean must be an array');
            }
            return varianceByRow(this, unbiased, mean);
          }
        case 'column':
          {
            if (!isAnyArray(mean)) {
              throw new TypeError('mean must be an array');
            }
            return varianceByColumn(this, unbiased, mean);
          }
        case undefined:
          {
            if (typeof mean !== 'number') {
              throw new TypeError('mean must be a number');
            }
            return varianceAll(this, unbiased, mean);
          }
        default:
          throw new Error(`invalid option: ${by}`);
      }
    }
    standardDeviation(by, options) {
      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }
      const variance = this.variance(by, options);
      if (by === undefined) {
        return Math.sqrt(variance);
      } else {
        for (let i = 0; i < variance.length; i++) {
          variance[i] = Math.sqrt(variance[i]);
        }
        return variance;
      }
    }
    center(by, options = {}) {
      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }
      const {
        center = this.mean(by)
      } = options;
      switch (by) {
        case 'row':
          {
            if (!isAnyArray(center)) {
              throw new TypeError('center must be an array');
            }
            centerByRow(this, center);
            return this;
          }
        case 'column':
          {
            if (!isAnyArray(center)) {
              throw new TypeError('center must be an array');
            }
            centerByColumn(this, center);
            return this;
          }
        case undefined:
          {
            if (typeof center !== 'number') {
              throw new TypeError('center must be a number');
            }
            centerAll(this, center);
            return this;
          }
        default:
          throw new Error(`invalid option: ${by}`);
      }
    }
    scale(by, options = {}) {
      if (typeof by === 'object') {
        options = by;
        by = undefined;
      }
      if (typeof options !== 'object') {
        throw new TypeError('options must be an object');
      }
      let scale = options.scale;
      switch (by) {
        case 'row':
          {
            if (scale === undefined) {
              scale = getScaleByRow(this);
            } else if (!isAnyArray(scale)) {
              throw new TypeError('scale must be an array');
            }
            scaleByRow(this, scale);
            return this;
          }
        case 'column':
          {
            if (scale === undefined) {
              scale = getScaleByColumn(this);
            } else if (!isAnyArray(scale)) {
              throw new TypeError('scale must be an array');
            }
            scaleByColumn(this, scale);
            return this;
          }
        case undefined:
          {
            if (scale === undefined) {
              scale = getScaleAll(this);
            } else if (typeof scale !== 'number') {
              throw new TypeError('scale must be a number');
            }
            scaleAll(this, scale);
            return this;
          }
        default:
          throw new Error(`invalid option: ${by}`);
      }
    }
    toString(options) {
      return inspectMatrixWithOptions(this, options);
    }
  }
  AbstractMatrix.prototype.klass = 'Matrix';
  if (typeof Symbol !== 'undefined') {
    AbstractMatrix.prototype[Symbol.for('nodejs.util.inspect.custom')] = inspectMatrix;
  }
  function compareNumbers(a, b) {
    return a - b;
  }
  function isArrayOfNumbers(array) {
    return array.every(element => {
      return typeof element === 'number';
    });
  }

  // Synonyms
  AbstractMatrix.random = AbstractMatrix.rand;
  AbstractMatrix.randomInt = AbstractMatrix.randInt;
  AbstractMatrix.diagonal = AbstractMatrix.diag;
  AbstractMatrix.prototype.diagonal = AbstractMatrix.prototype.diag;
  AbstractMatrix.identity = AbstractMatrix.eye;
  AbstractMatrix.prototype.negate = AbstractMatrix.prototype.neg;
  AbstractMatrix.prototype.tensorProduct = AbstractMatrix.prototype.kroneckerProduct;
  class Matrix extends AbstractMatrix {
    constructor(nRows, nColumns) {
      super();
      if (Matrix.isMatrix(nRows)) {
        // eslint-disable-next-line no-constructor-return
        return nRows.clone();
      } else if (Number.isInteger(nRows) && nRows >= 0) {
        // Create an empty matrix
        this.data = [];
        if (Number.isInteger(nColumns) && nColumns >= 0) {
          for (let i = 0; i < nRows; i++) {
            this.data.push(new Float64Array(nColumns));
          }
        } else {
          throw new TypeError('nColumns must be a positive integer');
        }
      } else if (isAnyArray(nRows)) {
        // Copy the values from the 2D array
        const arrayData = nRows;
        nRows = arrayData.length;
        nColumns = nRows ? arrayData[0].length : 0;
        if (typeof nColumns !== 'number') {
          throw new TypeError('Data must be a 2D array with at least one element');
        }
        this.data = [];
        for (let i = 0; i < nRows; i++) {
          if (arrayData[i].length !== nColumns) {
            throw new RangeError('Inconsistent array dimensions');
          }
          if (!isArrayOfNumbers(arrayData[i])) {
            throw new TypeError('Input data contains non-numeric values');
          }
          this.data.push(Float64Array.from(arrayData[i]));
        }
      } else {
        throw new TypeError('First argument must be a positive number or an array');
      }
      this.rows = nRows;
      this.columns = nColumns;
    }
    set(rowIndex, columnIndex, value) {
      this.data[rowIndex][columnIndex] = value;
      return this;
    }
    get(rowIndex, columnIndex) {
      return this.data[rowIndex][columnIndex];
    }
    removeRow(index) {
      checkRowIndex(this, index);
      this.data.splice(index, 1);
      this.rows -= 1;
      return this;
    }
    addRow(index, array) {
      if (array === undefined) {
        array = index;
        index = this.rows;
      }
      checkRowIndex(this, index, true);
      array = Float64Array.from(checkRowVector(this, array));
      this.data.splice(index, 0, array);
      this.rows += 1;
      return this;
    }
    removeColumn(index) {
      checkColumnIndex(this, index);
      for (let i = 0; i < this.rows; i++) {
        const newRow = new Float64Array(this.columns - 1);
        for (let j = 0; j < index; j++) {
          newRow[j] = this.data[i][j];
        }
        for (let j = index + 1; j < this.columns; j++) {
          newRow[j - 1] = this.data[i][j];
        }
        this.data[i] = newRow;
      }
      this.columns -= 1;
      return this;
    }
    addColumn(index, array) {
      if (typeof array === 'undefined') {
        array = index;
        index = this.columns;
      }
      checkColumnIndex(this, index, true);
      array = checkColumnVector(this, array);
      for (let i = 0; i < this.rows; i++) {
        const newRow = new Float64Array(this.columns + 1);
        let j = 0;
        for (; j < index; j++) {
          newRow[j] = this.data[i][j];
        }
        newRow[j++] = array[i];
        for (; j < this.columns + 1; j++) {
          newRow[j] = this.data[i][j - 1];
        }
        this.data[i] = newRow;
      }
      this.columns += 1;
      return this;
    }
  }
  installMathOperations(AbstractMatrix, Matrix);

  /**
   * Algorithm that finds the shortest distance from one node to the other
   * @param {Matrix} adjMatrix - A squared adjacency matrix
   * @return {Matrix} - Distance from a node to the other, -1 if the node is unreachable
   */
  function floydWarshall(adjMatrix) {
    if (Matrix.isMatrix(adjMatrix) && adjMatrix.columns !== adjMatrix.rows) {
      throw new TypeError('The adjacency matrix should be squared');
    }
    const numVertices = adjMatrix.columns;
    let distMatrix = new Matrix(numVertices, numVertices);
    distMatrix.apply((row, column) => {
      // principal diagonal is 0
      if (row === column) {
        distMatrix.set(row, column, 0);
      } else {
        let val = adjMatrix.get(row, column);
        if (val || Object.is(val, -0)) {
          // edges values remain the same
          distMatrix.set(row, column, val);
        } else {
          // 0 values become infinity
          distMatrix.set(row, column, Number.POSITIVE_INFINITY);
        }
      }
    });
    for (let k = 0; k < numVertices; ++k) {
      for (let i = 0; i < numVertices; ++i) {
        for (let j = 0; j < numVertices; ++j) {
          let dist = distMatrix.get(i, k) + distMatrix.get(k, j);
          if (distMatrix.get(i, j) > dist) {
            distMatrix.set(i, j, dist);
          }
        }
      }
    }

    // When there's no connection the value is -1
    distMatrix.apply((row, column) => {
      if (distMatrix.get(row, column) === Number.POSITIVE_INFINITY) {
        distMatrix.set(row, column, -1);
      }
    });
    return distMatrix;
  }

  /**
   * Returns a connectivity matrix
   * @param {import('openchemlib').Molecule} molecule
   * @param {object} [options={}]
   * @param {boolean} [options.pathLength=false] get the path length between atoms
   * @param {boolean} [options.mass=false] set the nominal mass of the atoms on diagonal
   * @param {boolean} [options.atomicNo=false] set the atomic number of the atom on diagonal
   * @param {boolean} [options.negativeAtomicNo=false] set the atomic number * -1 of the atom on diagonal
   * @param {boolean} [options.sdt=false] set 1, 2 or 3 depending if single, double or triple bond
   * @param {boolean} [options.sdta=false] set 1, 2, 3 or 4 depending if single, double, triple or aromatic  bond
   */
  function getConnectivityMatrix(molecule, options = {}) {
    const OCL = molecule.getOCL();
    molecule.ensureHelperArrays(OCL.Molecule.cHelperNeighbours);
    let nbAtoms = molecule.getAllAtoms();
    let result = new Array(nbAtoms).fill();
    result = result.map(() => new Array(nbAtoms).fill(0));
    if (!options.pathLength) {
      if (options.atomicNo) {
        for (let i = 0; i < nbAtoms; i++) {
          result[i][i] = molecule.getAtomicNo(i);
        }
      } else if (options.negativeAtomicNo) {
        for (let i = 0; i < nbAtoms; i++) {
          result[i][i] = -molecule.getAtomicNo(i);
        }
      } else if (options.mass) {
        for (let i = 0; i < nbAtoms; i++) {
          result[i][i] = OCL.Molecule.cRoundedMass[molecule.getAtomicNo(i)];
        }
      } else {
        for (let i = 0; i < nbAtoms; i++) {
          result[i][i] = 1;
        }
      }
    }
    if (options.sdt) {
      for (let i = 0; i < nbAtoms; i++) {
        let l = molecule.getAllConnAtoms(i);
        for (let j = 0; j < l; j++) {
          result[i][molecule.getConnAtom(i, j)] = molecule.getConnBondOrder(i, j);
        }
      }
    } else if (options.sdta) {
      for (let i = 0; i < nbAtoms; i++) {
        let l = molecule.getAllConnAtoms(i);
        for (let j = 0; j < l; j++) {
          let bondNumber = molecule.getConnBond(i, j);
          if (molecule.isAromaticBond(bondNumber)) {
            result[i][molecule.getConnAtom(i, j)] = 4;
          } else {
            result[i][molecule.getConnAtom(i, j)] = molecule.getConnBondOrder(i, j);
          }
        }
      }
    } else {
      for (let i = 0; i < nbAtoms; i++) {
        let l = molecule.getAllConnAtoms(i);
        for (let j = 0; j < l; j++) {
          result[i][molecule.getConnAtom(i, j)] = 1;
        }
      }
    }
    if (options.pathLength) {
      result = floydWarshall(new Matrix(result)).to2DArray();
    }
    return result;
  }

  /**
   * Implementation of the Hill system for sorting atoms
   * https://en.wikipedia.org/wiki/Chemical_formula#Hill_system
   * @param {string} a - first atom to compare
   * @param {string} b - second atom to compare
   * @returns
   */

  function atomSorter(a, b) {
    if (a === b) return 0;
    if (a === 'C') return -1;
    if (b === 'C') return 1;
    if (a === 'H') return -1;
    if (b === 'H') return 1;
    if (a < b) return -1;
    return 1;
  }

  /**
   * Calculate the molecular formula in 'chemcalc' notation taking into account fragments, isotopes and charges
   * @param {OCL.Molecule} [molecule] an instance of OCL.Molecule
   * @returns {object}
   */

  function getMF(molecule) {
    let entries = molecule.getFragments();
    let result = {};
    let parts = [];
    let allAtoms = [];
    entries.forEach(entry => {
      let mf = getFragmentMF(entry, allAtoms);
      parts.push(mf);
    });
    let counts = {};
    for (let part of parts) {
      if (!counts[part]) counts[part] = 0;
      counts[part]++;
    }
    parts = [];
    for (let key of Object.keys(counts).sort()) {
      if (counts[key] > 1) {
        parts.push(counts[key] + key);
      } else {
        parts.push(key);
      }
    }
    result.parts = parts;
    result.mf = toMFString(allAtoms);
    return result;
  }
  function getFragmentMF(molecule, allAtoms) {
    let atoms = [];
    for (let i = 0; i < molecule.getAllAtoms(); i++) {
      let atom = {};
      atom.charge = molecule.getAtomCharge(i);
      atom.label = molecule.getAtomLabel(i);
      atom.mass = molecule.getAtomMass(i);
      atom.implicitHydrogens = molecule.getImplicitHydrogens(i);
      atoms.push(atom);
      allAtoms.push(atom);
    }
    return toMFString(atoms);
  }
  function toMFString(atoms) {
    let charge = 0;
    let mfs = {};
    for (let atom of atoms) {
      let label = atom.label;
      charge += atom.charge;
      if (atom.mass) {
        label = `[${atom.mass}${label}]`;
      }
      let mfAtom = mfs[label];
      if (!mfAtom) {
        mfs[label] = 0;
      }
      mfs[label] += 1;
      if (atom.implicitHydrogens) {
        if (!mfs.H) mfs.H = 0;
        mfs.H += atom.implicitHydrogens;
      }
    }
    let mf = '';
    let keys = Object.keys(mfs).sort(atomSorter);
    for (let key of keys) {
      mf += key;
      if (mfs[key] > 1) mf += mfs[key];
    }
    if (charge > 0) {
      mf += `(+${charge > 1 ? charge : ''})`;
    } else if (charge < 0) {
      mf += `(${charge < -1 ? charge : '-'})`;
    }
    return mf;
  }

  /**
   * Calculate the molecular formula in 'chemcalc' notation taking into account fragments, isotopes and charges
   * @param {OCL.Molecule} [molecule] an instance of OCL.Molecule
   * @returns {object}
   */

  function getAtoms(molecule) {
    let entries = molecule.getFragments();
    const atoms = {};
    const result = {
      atoms,
      parts: []
    };
    entries.forEach(entry => {
      const part = {};
      result.parts.push(part);
      appendAtomPart(entry, atoms, part);
    });
    return result;
  }
  function appendAtomPart(molecule, atoms, part) {
    for (let i = 0; i < molecule.getAllAtoms(); i++) {
      let label = molecule.getAtomLabel(i);
      if (!atoms[label]) {
        atoms[label] = 0;
      }
      atoms[label] += 1;
      if (!part[label]) {
        part[label] = 0;
      }
      part[label] += 1;
      let implicitHydrogens = molecule.getImplicitHydrogens(i);
      if (implicitHydrogens) {
        if (!atoms.H) {
          atoms.H = 0;
        }
        atoms.H += implicitHydrogens;
        if (!part.H) {
          part.H = 0;
        }
        part.H += implicitHydrogens;
      }
    }
  }

  /**
   * Return the number of Hydroxyl groups in a molecule or fragment
   * @param {import('openchemlib').Molecule} molecule
   * @returns {number} 'Number of Hydroxyl groups'
   */

  function nbOH(molecule) {
    let counter = 0;
    for (let i = 0; i < molecule.getAllAtoms(); i++) {
      if (molecule.getAtomicNo(i) === 6) {
        let carbonyl = false;
        let hydroxyl = false;
        let carbonOrHydrogen = true;
        for (let neighbour = 0; neighbour < molecule.getConnAtoms(i); neighbour++) {
          const neighbourAtom = molecule.getConnAtom(i, neighbour);
          const neighbourBond = molecule.getConnBond(i, neighbour);
          if (molecule.getAtomicNo(neighbourAtom) === 8) {
            if (molecule.getBondOrder(neighbourBond) === 1 && molecule.getAllHydrogens(neighbourAtom) > 0) {
              // If there is more than a Hydroxyl in the same carbon atome they are not couted as Hydroxyl groups
              if (hydroxyl) {
                hydroxyl = false;
                break;
              }
              hydroxyl = true;
            } else if (molecule.getBondOrder(neighbourBond) === 2) {
              // If there is Carbonyl group on the same carbon atom it is not couted as Hydroxyl group
              carbonyl = true;
            }
          } else if (
          // If there is not at least one carbon or hydrogen as neighbour atom it is not counted as Hydroxyl group
          molecule.getAtomicNo(neighbourAtom) !== 6 && molecule.getAtomicNo(neighbourAtom) !== 1) {
            carbonOrHydrogen = false;
          }
        }
        if (carbonyl === false && hydroxyl && carbonOrHydrogen) counter++;
      }
    }
    return counter;
  }

  /**
   * Return the number of Carboxyl groups in a molecule or fragment
   * @param {import('openchemlib').Molecule} molecule
   * @returns {number} 'Number of Carboxyl groups'
   */

  function nbCOOH(molecule) {
    let counter = 0;
    for (let i = 0; i < molecule.getAllAtoms(); i++) {
      if (molecule.getAtomicNo(i) === 6) {
        let carbonyl = false;
        let hydroxyl = false;
        let carbonOrHydrogen = true;
        for (let neighbour = 0; neighbour < molecule.getConnAtoms(i); neighbour++) {
          const neighbourAtom = molecule.getConnAtom(i, neighbour);
          const neighbourBond = molecule.getConnBond(i, neighbour);
          if (molecule.getAtomicNo(neighbourAtom) === 8) {
            if (molecule.getBondOrder(neighbourBond) === 1 && molecule.getAllHydrogens(neighbourAtom) > 0) {
              // If there is more than a Hydroxyl in the same carbon atom it is not couted as Carboxyl group
              if (hydroxyl) {
                hydroxyl = false;
                break;
              }
              hydroxyl = true;
            } else if (molecule.getBondOrder(neighbourBond) === 2) {
              // If there is more than one carbonyl in the same carbon atom it is not count as Carboxyl group
              if (carbonyl) {
                carbonyl = false;
                break;
              }
              carbonyl = true;
            }
          } else if (
          // If there is not at least one carbon or hydrogen as neighbour atom it is not counted as Carboxyl group
          molecule.getAtomicNo(neighbourAtom) !== 6 && molecule.getAtomicNo(neighbourAtom) !== 1) {
            carbonOrHydrogen = false;
          }
        }
        if (carbonyl && hydroxyl && carbonOrHydrogen) counter++;
      }
    }
    return counter;
  }

  /**
   * Return the number of Carbonyl groups in a molecule or fragment
   * @param {import('openchemlib').Molecule} molecule
   * @returns {number} 'Number of Carbonyl groups'
   */

  function nbCHO(molecule) {
    let counter = 0;
    for (let i = 0; i < molecule.getAllAtoms(); i++) {
      // if there is not at least one hydrogen in the carbon atom there can not be a carbonyl group
      if (molecule.getAtomicNo(i) === 6 && molecule.getAllHydrogens(i) > 0) {
        let carbonyl = false;
        let carbonOrHydrogen = true;
        for (let neighbour = 0; neighbour < molecule.getConnAtoms(i); neighbour++) {
          const neighbourAtom = molecule.getConnAtom(i, neighbour);
          const neighbourBond = molecule.getConnBond(i, neighbour);
          if (molecule.getAtomicNo(neighbourAtom) === 8) {
            if (molecule.getBondOrder(neighbourBond) === 2) {
              // If there is more than one carbonyl group on the same carbon atom they are not counted as carbonyl groups
              if (carbonyl) {
                carbonyl = false;
                break;
              }
              carbonyl = true;
            }
          } else if (
          // If there is not at least one carbon or hydrogen as neighbour atom it is not counted as Carbonyl group
          molecule.getAtomicNo(neighbourAtom) !== 6 && molecule.getAtomicNo(neighbourAtom) !== 1) {
            carbonOrHydrogen = false;
          }
        }
        if (carbonyl && carbonOrHydrogen) counter++;
      }
    }
    return counter;
  }

  /**
   * Return the number of Primary amine groups in a molecule or fragment
   * @param {import('openchemlib').Molecule} molecule
   * @returns {number} 'Number of Primary amine groups'
   */

  function nbNH2(molecule) {
    let counter = 0;
    for (let i = 0; i < molecule.getAllAtoms(); i++) {
      if (molecule.getAtomicNo(i) === 6) {
        let amine = false;
        let carbonOrHydrogen = true;
        for (let neighbour = 0; neighbour < molecule.getConnAtoms(i); neighbour++) {
          const neighbourAtom = molecule.getConnAtom(i, neighbour);
          const neighbourBond = molecule.getConnBond(i, neighbour);
          if (molecule.getAtomicNo(neighbourAtom) === 7 && molecule.getBondOrder(neighbourBond) === 1 && molecule.getAllHydrogens(neighbourAtom) > 1) {
            // If there is more than a Primary amine in the same carbon atom they are not couted as Primary amines groups
            if (amine) {
              amine = false;
              break;
            }
            amine = true;
          } else if (
          // If there is not at least one carbon or hydrogen as neighbour atom it is not counted as Primary amine group
          molecule.getAtomicNo(neighbourAtom) !== 6 && molecule.getAtomicNo(neighbourAtom) !== 1) {
            carbonOrHydrogen = false;
          }
        }
        if (amine && carbonOrHydrogen) counter++;
      }
    }
    return counter;
  }

  /**
   * Return the number of Nitrile groups in a molecule or fragment
   * @param {import('openchemlib').Molecule} molecule
   * @returns {number} 'Number of Nitrile groups'
   */

  function nbCN(molecule) {
    let counter = 0;
    for (let i = 0; i < molecule.getAllAtoms(); i++) {
      if (molecule.getAtomicNo(i) === 6) {
        let cn = false;
        let carbonOrHydrogen = true;
        for (let neighbour = 0; neighbour < molecule.getConnAtoms(i); neighbour++) {
          const neighbourAtom = molecule.getConnAtom(i, neighbour);
          const neighbourBond = molecule.getConnBond(i, neighbour);
          if (molecule.getAtomicNo(neighbourAtom) === 7 && molecule.getBondOrder(neighbourBond) === 3) {
            // If there is more than one Nitrile group in the same carbon atome they are not counted as Nitrile groups
            if (cn) {
              cn = false;
              break;
            }
            cn = true;
          } else if (
          // If there is not at least one carbon or hydrogen as neighbour atom it is not counted as Nitrile group
          molecule.getAtomicNo(neighbourAtom) !== 6 && molecule.getAtomicNo(neighbourAtom) !== 1) {
            carbonOrHydrogen = false;
          }
        }
        if (cn && carbonOrHydrogen) counter++;
      }
    }
    return counter;
  }

  /**
   * Return the number of labile protons being either on O, N, Br, Cl, F, I or S
   * @param {import('openchemlib').Molecule} molecule
   * @param {object} [options={}]
   * @param {Array<number>} [options.atomicNumbers=[7, 8, 9, 16, 17, 35, 53]] - atomic numbers of the labile protons
   * @returns {number} 'Number of labile protons'
   */

  function nbLabileH(molecule, options = {}) {
    const {
      atomicNumbers = [7, 8, 9, 16, 17, 35, 53]
    } = options;
    let counter = 0;
    for (let i = 0; i < molecule.getAllAtoms(); i++) {
      if (atomicNumbers.includes(molecule.getAtomicNo(i))) {
        counter += molecule.getAllHydrogens(i);
      }
    }
    return counter;
  }

  /**
   * Calculates the path between 2 atoms
   * @param {import('openchemlib').Molecule} molecule
   * @param {number} from - index of the first atom
   * @param {number} to - index of the end atom
   * @param {number} maxLength - maximal length of the path
   */
  function getPathAndTorsion(molecule, from, to, maxLength) {
    let originalAtoms = []; // path before renumbering
    molecule.getPath(originalAtoms, from, to, maxLength + 1);
    let torsion;
    if (originalAtoms.length === 4) {
      torsion = molecule.calculateTorsion(originalAtoms);
    }
    return {
      atoms: originalAtoms,
      from,
      to,
      torsion,
      length: originalAtoms.length - 1
    };
  }

  let fragment;

  /**
   *
   * @param {import('openchemlib').Molecule} molecule
   * @param {object} [options={}]
   * @param {string} [options.fromLabel='H']
   * @param {string} [options.toLabel='H']
   * @param {number} [options.minLength=1]
   * @param {number} [options.maxLength=4]
   * @param {boolean} [options.withHOSES=false]

   */
  function getPathsInfo(molecule, options = {}) {
    const {
      fromLabel = 'H',
      toLabel = 'H',
      minLength = 1,
      maxLength = 4,
      withHOSES = false
    } = options;
    const OCL = molecule.getOCL();
    if (!fragment) {
      fragment = new OCL.Molecule(0, 0);
    }
    let fromAtomicNumber = OCL.Molecule.getAtomicNoFromLabel(fromLabel);
    let toAtomicNumber = OCL.Molecule.getAtomicNoFromLabel(toLabel);

    // we need to find all the atoms 'fromLabel' and 'toLabel'
    let atomsInfo = getAtomsInfo(molecule);
    let pathLengthMatrix = getConnectivityMatrix(molecule, {
      pathLength: true
    });
    for (let from = 0; from < molecule.getAllAtoms(); from++) {
      atomsInfo[from].paths = [];
      for (let to = 0; to < molecule.getAllAtoms(); to++) {
        if (from !== to) {
          if (molecule.getAtomicNo(from) === fromAtomicNumber) {
            if (molecule.getAtomicNo(to) === toAtomicNumber) {
              let pathLength = pathLengthMatrix[from][to];
              if (pathLength >= minLength && pathLength <= maxLength) {
                if (withHOSES) {
                  atomsInfo[from].paths.push(getHoseCodesForPath(molecule, from, to, pathLength));
                } else {
                  atomsInfo[from].paths.push(getPathAndTorsion(molecule, from, to, pathLength));
                }
              }
            }
          }
        }
      }
    }
    return atomsInfo;
  }

  /**
   * Get the shortest path between each pair of atoms in the molecule
   * @param {import('openchemlib').Molecule} molecule
   * @param {object} [options={}]
   * @param {string} [opions.fromLabel='H']
   * @param {string} [opions.toLabel='H']
   * @param {string} [opions.maxLength=4]
   * @returns {Array<Array>} A matrix containing on each cell (i,j) the shortest path from atom i to atom j
   */
  function getShortestPaths(molecule, options = {}) {
    const OCL = molecule.getOCL();
    const {
      fromLabel = '',
      toLabel = '',
      maxLength = 3
    } = options;
    let fromAtomicNumber = OCL.Molecule.getAtomicNoFromLabel(fromLabel);
    let toAtomicNumber = OCL.Molecule.getAtomicNoFromLabel(toLabel);
    const nbAtoms = molecule.getAllAtoms();
    let allShortestPaths = new Array(nbAtoms);
    for (let i = 0; i < nbAtoms; i++) {
      allShortestPaths[i] = new Array(nbAtoms);
    }
    for (let from = 0; from < nbAtoms; from++) {
      allShortestPaths[from][from] = [from];
      for (let to = from + 1; to < nbAtoms; to++) {
        if ((fromAtomicNumber === 0 || molecule.getAtomicNo(from) === fromAtomicNumber) && (toAtomicNumber === 0 || molecule.getAtomicNo(to) === toAtomicNumber)) {
          let path = [];
          molecule.getPath(path, from, to, maxLength);
          if (path.length) {
            allShortestPaths[from][to] = path.slice();
            allShortestPaths[to][from] = path.reverse();
          } else {
            allShortestPaths[from][to] = null;
            allShortestPaths[to][from] = null;
          }
        } else {
          allShortestPaths[from][to] = null;
          allShortestPaths[to][from] = null;
        }
      }
    }
    return allShortestPaths;
  }

  /*
      https://tools.ietf.org/html/rfc3629

      UTF8-char = UTF8-1 / UTF8-2 / UTF8-3 / UTF8-4

      UTF8-1    = %x00-7F

      UTF8-2    = %xC2-DF UTF8-tail

      UTF8-3    = %xE0 %xA0-BF UTF8-tail
                  %xE1-EC 2( UTF8-tail )
                  %xED %x80-9F UTF8-tail
                  %xEE-EF 2( UTF8-tail )

      UTF8-4    = %xF0 %x90-BF 2( UTF8-tail )
                  %xF1-F3 3( UTF8-tail )
                  %xF4 %x80-8F 2( UTF8-tail )

      UTF8-tail = %x80-BF
  */
  /**
   * Check if a Node.js Buffer or Uint8Array is UTF-8.
   */
  function isUtf8(buf) {
    if (!buf) {
      return false;
    }
    var i = 0;
    var len = buf.length;
    while (i < len) {
      // UTF8-1 = %x00-7F
      if (buf[i] <= 0x7F) {
        i++;
        continue;
      }
      // UTF8-2 = %xC2-DF UTF8-tail
      if (buf[i] >= 0xC2 && buf[i] <= 0xDF) {
        // if(buf[i + 1] >= 0x80 && buf[i + 1] <= 0xBF) {
        if (buf[i + 1] >> 6 === 2) {
          i += 2;
          continue;
        } else {
          return false;
        }
      }
      // UTF8-3 = %xE0 %xA0-BF UTF8-tail
      // UTF8-3 = %xED %x80-9F UTF8-tail
      if ((buf[i] === 0xE0 && buf[i + 1] >= 0xA0 && buf[i + 1] <= 0xBF || buf[i] === 0xED && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x9F) && buf[i + 2] >> 6 === 2) {
        i += 3;
        continue;
      }
      // UTF8-3 = %xE1-EC 2( UTF8-tail )
      // UTF8-3 = %xEE-EF 2( UTF8-tail )
      if ((buf[i] >= 0xE1 && buf[i] <= 0xEC || buf[i] >= 0xEE && buf[i] <= 0xEF) && buf[i + 1] >> 6 === 2 && buf[i + 2] >> 6 === 2) {
        i += 3;
        continue;
      }
      // UTF8-4 = %xF0 %x90-BF 2( UTF8-tail )
      //          %xF1-F3 3( UTF8-tail )
      //          %xF4 %x80-8F 2( UTF8-tail )
      if ((buf[i] === 0xF0 && buf[i + 1] >= 0x90 && buf[i + 1] <= 0xBF || buf[i] >= 0xF1 && buf[i] <= 0xF3 && buf[i + 1] >> 6 === 2 || buf[i] === 0xF4 && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x8F) && buf[i + 2] >> 6 === 2 && buf[i + 3] >> 6 === 2) {
        i += 4;
        continue;
      }
      return false;
    }
    return true;
  }

  /**
   * Ensure that the data is string. If it is an ArrayBuffer it will be converted to string using TextDecoder.
   * @param blob
   * @param options
   * @returns
   */
  function ensureString(blob, options = {}) {
    if (typeof blob === 'string') {
      return blob;
    }
    if (ArrayBuffer.isView(blob) || blob instanceof ArrayBuffer) {
      const {
        encoding = guessEncoding(blob)
      } = options;
      const decoder = new TextDecoder(encoding);
      return decoder.decode(blob);
    }
    throw new TypeError(`blob must be a string, ArrayBuffer or ArrayBufferView`);
  }
  function guessEncoding(blob) {
    const uint8 = ArrayBuffer.isView(blob) ? new Uint8Array(blob.buffer, blob.byteOffset, blob.byteLength) : new Uint8Array(blob);
    if (uint8.length >= 2) {
      if (uint8[0] === 0xfe && uint8[1] === 0xff) {
        return 'utf-16be';
      }
      if (uint8[0] === 0xff && uint8[1] === 0xfe) {
        return 'utf-16le';
      }
    }
    //@ts-expect-error an ArrayBuffer is also ok
    if (!isUtf8(blob)) return 'latin1';
    return 'utf-8';
  }

  var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

  var papaparse_min = {exports: {}};

  /* @license
  Papa Parse
  v5.4.1
  https://github.com/mholt/PapaParse
  License: MIT
  */
  (function (module, exports) {
    !function (e, t) {
      module.exports = t() ;
    }(commonjsGlobal, function s() {

      var f = "undefined" != typeof self ? self : "undefined" != typeof window ? window : void 0 !== f ? f : {};
      var n = !f.document && !!f.postMessage,
        o = f.IS_PAPA_WORKER || !1,
        a = {},
        u = 0,
        b = {
          parse: function (e, t) {
            var r = (t = t || {}).dynamicTyping || !1;
            J(r) && (t.dynamicTypingFunction = r, r = {});
            if (t.dynamicTyping = r, t.transform = !!J(t.transform) && t.transform, t.worker && b.WORKERS_SUPPORTED) {
              var i = function () {
                if (!b.WORKERS_SUPPORTED) return !1;
                var e = (r = f.URL || f.webkitURL || null, i = s.toString(), b.BLOB_URL || (b.BLOB_URL = r.createObjectURL(new Blob(["var global = (function() { if (typeof self !== 'undefined') { return self; } if (typeof window !== 'undefined') { return window; } if (typeof global !== 'undefined') { return global; } return {}; })(); global.IS_PAPA_WORKER=true; ", "(", i, ")();"], {
                    type: "text/javascript"
                  })))),
                  t = new f.Worker(e);
                var r, i;
                return t.onmessage = _, t.id = u++, a[t.id] = t;
              }();
              return i.userStep = t.step, i.userChunk = t.chunk, i.userComplete = t.complete, i.userError = t.error, t.step = J(t.step), t.chunk = J(t.chunk), t.complete = J(t.complete), t.error = J(t.error), delete t.worker, void i.postMessage({
                input: e,
                config: t,
                workerId: i.id
              });
            }
            var n = null;
            b.NODE_STREAM_INPUT, "string" == typeof e ? (e = function (e) {
              if (65279 === e.charCodeAt(0)) return e.slice(1);
              return e;
            }(e), n = t.download ? new l(t) : new p(t)) : !0 === e.readable && J(e.read) && J(e.on) ? n = new g(t) : (f.File && e instanceof File || e instanceof Object) && (n = new c(t));
            return n.stream(e);
          },
          unparse: function (e, t) {
            var n = !1,
              _ = !0,
              m = ",",
              y = "\r\n",
              s = '"',
              a = s + s,
              r = !1,
              i = null,
              o = !1;
            !function () {
              if ("object" != typeof t) return;
              "string" != typeof t.delimiter || b.BAD_DELIMITERS.filter(function (e) {
                return -1 !== t.delimiter.indexOf(e);
              }).length || (m = t.delimiter);
              ("boolean" == typeof t.quotes || "function" == typeof t.quotes || Array.isArray(t.quotes)) && (n = t.quotes);
              "boolean" != typeof t.skipEmptyLines && "string" != typeof t.skipEmptyLines || (r = t.skipEmptyLines);
              "string" == typeof t.newline && (y = t.newline);
              "string" == typeof t.quoteChar && (s = t.quoteChar);
              "boolean" == typeof t.header && (_ = t.header);
              if (Array.isArray(t.columns)) {
                if (0 === t.columns.length) throw new Error("Option columns is empty");
                i = t.columns;
              }
              void 0 !== t.escapeChar && (a = t.escapeChar + s);
              ("boolean" == typeof t.escapeFormulae || t.escapeFormulae instanceof RegExp) && (o = t.escapeFormulae instanceof RegExp ? t.escapeFormulae : /^[=+\-@\t\r].*$/);
            }();
            var u = new RegExp(Q(s), "g");
            "string" == typeof e && (e = JSON.parse(e));
            if (Array.isArray(e)) {
              if (!e.length || Array.isArray(e[0])) return h(null, e, r);
              if ("object" == typeof e[0]) return h(i || Object.keys(e[0]), e, r);
            } else if ("object" == typeof e) return "string" == typeof e.data && (e.data = JSON.parse(e.data)), Array.isArray(e.data) && (e.fields || (e.fields = e.meta && e.meta.fields || i), e.fields || (e.fields = Array.isArray(e.data[0]) ? e.fields : "object" == typeof e.data[0] ? Object.keys(e.data[0]) : []), Array.isArray(e.data[0]) || "object" == typeof e.data[0] || (e.data = [e.data])), h(e.fields || [], e.data || [], r);
            throw new Error("Unable to serialize unrecognized input");
            function h(e, t, r) {
              var i = "";
              "string" == typeof e && (e = JSON.parse(e)), "string" == typeof t && (t = JSON.parse(t));
              var n = Array.isArray(e) && 0 < e.length,
                s = !Array.isArray(t[0]);
              if (n && _) {
                for (var a = 0; a < e.length; a++) 0 < a && (i += m), i += v(e[a], a);
                0 < t.length && (i += y);
              }
              for (var o = 0; o < t.length; o++) {
                var u = n ? e.length : t[o].length,
                  h = !1,
                  f = n ? 0 === Object.keys(t[o]).length : 0 === t[o].length;
                if (r && !n && (h = "greedy" === r ? "" === t[o].join("").trim() : 1 === t[o].length && 0 === t[o][0].length), "greedy" === r && n) {
                  for (var d = [], l = 0; l < u; l++) {
                    var c = s ? e[l] : l;
                    d.push(t[o][c]);
                  }
                  h = "" === d.join("").trim();
                }
                if (!h) {
                  for (var p = 0; p < u; p++) {
                    0 < p && !f && (i += m);
                    var g = n && s ? e[p] : p;
                    i += v(t[o][g], p);
                  }
                  o < t.length - 1 && (!r || 0 < u && !f) && (i += y);
                }
              }
              return i;
            }
            function v(e, t) {
              if (null == e) return "";
              if (e.constructor === Date) return JSON.stringify(e).slice(1, 25);
              var r = !1;
              o && "string" == typeof e && o.test(e) && (e = "'" + e, r = !0);
              var i = e.toString().replace(u, a);
              return (r = r || !0 === n || "function" == typeof n && n(e, t) || Array.isArray(n) && n[t] || function (e, t) {
                for (var r = 0; r < t.length; r++) if (-1 < e.indexOf(t[r])) return !0;
                return !1;
              }(i, b.BAD_DELIMITERS) || -1 < i.indexOf(m) || " " === i.charAt(0) || " " === i.charAt(i.length - 1)) ? s + i + s : i;
            }
          }
        };
      if (b.RECORD_SEP = String.fromCharCode(30), b.UNIT_SEP = String.fromCharCode(31), b.BYTE_ORDER_MARK = "\ufeff", b.BAD_DELIMITERS = ["\r", "\n", '"', b.BYTE_ORDER_MARK], b.WORKERS_SUPPORTED = !n && !!f.Worker, b.NODE_STREAM_INPUT = 1, b.LocalChunkSize = 10485760, b.RemoteChunkSize = 5242880, b.DefaultDelimiter = ",", b.Parser = E, b.ParserHandle = r, b.NetworkStreamer = l, b.FileStreamer = c, b.StringStreamer = p, b.ReadableStreamStreamer = g, f.jQuery) {
        var d = f.jQuery;
        d.fn.parse = function (o) {
          var r = o.config || {},
            u = [];
          return this.each(function (e) {
            if (!("INPUT" === d(this).prop("tagName").toUpperCase() && "file" === d(this).attr("type").toLowerCase() && f.FileReader) || !this.files || 0 === this.files.length) return !0;
            for (var t = 0; t < this.files.length; t++) u.push({
              file: this.files[t],
              inputElem: this,
              instanceConfig: d.extend({}, r)
            });
          }), e(), this;
          function e() {
            if (0 !== u.length) {
              var e,
                t,
                r,
                i,
                n = u[0];
              if (J(o.before)) {
                var s = o.before(n.file, n.inputElem);
                if ("object" == typeof s) {
                  if ("abort" === s.action) return e = "AbortError", t = n.file, r = n.inputElem, i = s.reason, void (J(o.error) && o.error({
                    name: e
                  }, t, r, i));
                  if ("skip" === s.action) return void h();
                  "object" == typeof s.config && (n.instanceConfig = d.extend(n.instanceConfig, s.config));
                } else if ("skip" === s) return void h();
              }
              var a = n.instanceConfig.complete;
              n.instanceConfig.complete = function (e) {
                J(a) && a(e, n.file, n.inputElem), h();
              }, b.parse(n.file, n.instanceConfig);
            } else J(o.complete) && o.complete();
          }
          function h() {
            u.splice(0, 1), e();
          }
        };
      }
      function h(e) {
        this._handle = null, this._finished = !1, this._completed = !1, this._halted = !1, this._input = null, this._baseIndex = 0, this._partialLine = "", this._rowCount = 0, this._start = 0, this._nextChunk = null, this.isFirstChunk = !0, this._completeResults = {
          data: [],
          errors: [],
          meta: {}
        }, function (e) {
          var t = w(e);
          t.chunkSize = parseInt(t.chunkSize), e.step || e.chunk || (t.chunkSize = null);
          this._handle = new r(t), (this._handle.streamer = this)._config = t;
        }.call(this, e), this.parseChunk = function (e, t) {
          if (this.isFirstChunk && J(this._config.beforeFirstChunk)) {
            var r = this._config.beforeFirstChunk(e);
            void 0 !== r && (e = r);
          }
          this.isFirstChunk = !1, this._halted = !1;
          var i = this._partialLine + e;
          this._partialLine = "";
          var n = this._handle.parse(i, this._baseIndex, !this._finished);
          if (!this._handle.paused() && !this._handle.aborted()) {
            var s = n.meta.cursor;
            this._finished || (this._partialLine = i.substring(s - this._baseIndex), this._baseIndex = s), n && n.data && (this._rowCount += n.data.length);
            var a = this._finished || this._config.preview && this._rowCount >= this._config.preview;
            if (o) f.postMessage({
              results: n,
              workerId: b.WORKER_ID,
              finished: a
            });else if (J(this._config.chunk) && !t) {
              if (this._config.chunk(n, this._handle), this._handle.paused() || this._handle.aborted()) return void (this._halted = !0);
              n = void 0, this._completeResults = void 0;
            }
            return this._config.step || this._config.chunk || (this._completeResults.data = this._completeResults.data.concat(n.data), this._completeResults.errors = this._completeResults.errors.concat(n.errors), this._completeResults.meta = n.meta), this._completed || !a || !J(this._config.complete) || n && n.meta.aborted || (this._config.complete(this._completeResults, this._input), this._completed = !0), a || n && n.meta.paused || this._nextChunk(), n;
          }
          this._halted = !0;
        }, this._sendError = function (e) {
          J(this._config.error) ? this._config.error(e) : o && this._config.error && f.postMessage({
            workerId: b.WORKER_ID,
            error: e,
            finished: !1
          });
        };
      }
      function l(e) {
        var i;
        (e = e || {}).chunkSize || (e.chunkSize = b.RemoteChunkSize), h.call(this, e), this._nextChunk = n ? function () {
          this._readChunk(), this._chunkLoaded();
        } : function () {
          this._readChunk();
        }, this.stream = function (e) {
          this._input = e, this._nextChunk();
        }, this._readChunk = function () {
          if (this._finished) this._chunkLoaded();else {
            if (i = new XMLHttpRequest(), this._config.withCredentials && (i.withCredentials = this._config.withCredentials), n || (i.onload = v(this._chunkLoaded, this), i.onerror = v(this._chunkError, this)), i.open(this._config.downloadRequestBody ? "POST" : "GET", this._input, !n), this._config.downloadRequestHeaders) {
              var e = this._config.downloadRequestHeaders;
              for (var t in e) i.setRequestHeader(t, e[t]);
            }
            if (this._config.chunkSize) {
              var r = this._start + this._config.chunkSize - 1;
              i.setRequestHeader("Range", "bytes=" + this._start + "-" + r);
            }
            try {
              i.send(this._config.downloadRequestBody);
            } catch (e) {
              this._chunkError(e.message);
            }
            n && 0 === i.status && this._chunkError();
          }
        }, this._chunkLoaded = function () {
          4 === i.readyState && (i.status < 200 || 400 <= i.status ? this._chunkError() : (this._start += this._config.chunkSize ? this._config.chunkSize : i.responseText.length, this._finished = !this._config.chunkSize || this._start >= function (e) {
            var t = e.getResponseHeader("Content-Range");
            if (null === t) return -1;
            return parseInt(t.substring(t.lastIndexOf("/") + 1));
          }(i), this.parseChunk(i.responseText)));
        }, this._chunkError = function (e) {
          var t = i.statusText || e;
          this._sendError(new Error(t));
        };
      }
      function c(e) {
        var i, n;
        (e = e || {}).chunkSize || (e.chunkSize = b.LocalChunkSize), h.call(this, e);
        var s = "undefined" != typeof FileReader;
        this.stream = function (e) {
          this._input = e, n = e.slice || e.webkitSlice || e.mozSlice, s ? ((i = new FileReader()).onload = v(this._chunkLoaded, this), i.onerror = v(this._chunkError, this)) : i = new FileReaderSync(), this._nextChunk();
        }, this._nextChunk = function () {
          this._finished || this._config.preview && !(this._rowCount < this._config.preview) || this._readChunk();
        }, this._readChunk = function () {
          var e = this._input;
          if (this._config.chunkSize) {
            var t = Math.min(this._start + this._config.chunkSize, this._input.size);
            e = n.call(e, this._start, t);
          }
          var r = i.readAsText(e, this._config.encoding);
          s || this._chunkLoaded({
            target: {
              result: r
            }
          });
        }, this._chunkLoaded = function (e) {
          this._start += this._config.chunkSize, this._finished = !this._config.chunkSize || this._start >= this._input.size, this.parseChunk(e.target.result);
        }, this._chunkError = function () {
          this._sendError(i.error);
        };
      }
      function p(e) {
        var r;
        h.call(this, e = e || {}), this.stream = function (e) {
          return r = e, this._nextChunk();
        }, this._nextChunk = function () {
          if (!this._finished) {
            var e,
              t = this._config.chunkSize;
            return t ? (e = r.substring(0, t), r = r.substring(t)) : (e = r, r = ""), this._finished = !r, this.parseChunk(e);
          }
        };
      }
      function g(e) {
        h.call(this, e = e || {});
        var t = [],
          r = !0,
          i = !1;
        this.pause = function () {
          h.prototype.pause.apply(this, arguments), this._input.pause();
        }, this.resume = function () {
          h.prototype.resume.apply(this, arguments), this._input.resume();
        }, this.stream = function (e) {
          this._input = e, this._input.on("data", this._streamData), this._input.on("end", this._streamEnd), this._input.on("error", this._streamError);
        }, this._checkIsFinished = function () {
          i && 1 === t.length && (this._finished = !0);
        }, this._nextChunk = function () {
          this._checkIsFinished(), t.length ? this.parseChunk(t.shift()) : r = !0;
        }, this._streamData = v(function (e) {
          try {
            t.push("string" == typeof e ? e : e.toString(this._config.encoding)), r && (r = !1, this._checkIsFinished(), this.parseChunk(t.shift()));
          } catch (e) {
            this._streamError(e);
          }
        }, this), this._streamError = v(function (e) {
          this._streamCleanUp(), this._sendError(e);
        }, this), this._streamEnd = v(function () {
          this._streamCleanUp(), i = !0, this._streamData("");
        }, this), this._streamCleanUp = v(function () {
          this._input.removeListener("data", this._streamData), this._input.removeListener("end", this._streamEnd), this._input.removeListener("error", this._streamError);
        }, this);
      }
      function r(m) {
        var a,
          o,
          u,
          i = Math.pow(2, 53),
          n = -i,
          s = /^\s*-?(\d+\.?|\.\d+|\d+\.\d+)([eE][-+]?\d+)?\s*$/,
          h = /^((\d{4}-[01]\d-[0-3]\dT[0-2]\d:[0-5]\d:[0-5]\d\.\d+([+-][0-2]\d:[0-5]\d|Z))|(\d{4}-[01]\d-[0-3]\dT[0-2]\d:[0-5]\d:[0-5]\d([+-][0-2]\d:[0-5]\d|Z))|(\d{4}-[01]\d-[0-3]\dT[0-2]\d:[0-5]\d([+-][0-2]\d:[0-5]\d|Z)))$/,
          t = this,
          r = 0,
          f = 0,
          d = !1,
          e = !1,
          l = [],
          c = {
            data: [],
            errors: [],
            meta: {}
          };
        if (J(m.step)) {
          var p = m.step;
          m.step = function (e) {
            if (c = e, _()) g();else {
              if (g(), 0 === c.data.length) return;
              r += e.data.length, m.preview && r > m.preview ? o.abort() : (c.data = c.data[0], p(c, t));
            }
          };
        }
        function y(e) {
          return "greedy" === m.skipEmptyLines ? "" === e.join("").trim() : 1 === e.length && 0 === e[0].length;
        }
        function g() {
          return c && u && (k("Delimiter", "UndetectableDelimiter", "Unable to auto-detect delimiting character; defaulted to '" + b.DefaultDelimiter + "'"), u = !1), m.skipEmptyLines && (c.data = c.data.filter(function (e) {
            return !y(e);
          })), _() && function () {
            if (!c) return;
            function e(e, t) {
              J(m.transformHeader) && (e = m.transformHeader(e, t)), l.push(e);
            }
            if (Array.isArray(c.data[0])) {
              for (var t = 0; _() && t < c.data.length; t++) c.data[t].forEach(e);
              c.data.splice(0, 1);
            } else c.data.forEach(e);
          }(), function () {
            if (!c || !m.header && !m.dynamicTyping && !m.transform) return c;
            function e(e, t) {
              var r,
                i = m.header ? {} : [];
              for (r = 0; r < e.length; r++) {
                var n = r,
                  s = e[r];
                m.header && (n = r >= l.length ? "__parsed_extra" : l[r]), m.transform && (s = m.transform(s, n)), s = v(n, s), "__parsed_extra" === n ? (i[n] = i[n] || [], i[n].push(s)) : i[n] = s;
              }
              return m.header && (r > l.length ? k("FieldMismatch", "TooManyFields", "Too many fields: expected " + l.length + " fields but parsed " + r, f + t) : r < l.length && k("FieldMismatch", "TooFewFields", "Too few fields: expected " + l.length + " fields but parsed " + r, f + t)), i;
            }
            var t = 1;
            !c.data.length || Array.isArray(c.data[0]) ? (c.data = c.data.map(e), t = c.data.length) : c.data = e(c.data, 0);
            m.header && c.meta && (c.meta.fields = l);
            return f += t, c;
          }();
        }
        function _() {
          return m.header && 0 === l.length;
        }
        function v(e, t) {
          return r = e, m.dynamicTypingFunction && void 0 === m.dynamicTyping[r] && (m.dynamicTyping[r] = m.dynamicTypingFunction(r)), !0 === (m.dynamicTyping[r] || m.dynamicTyping) ? "true" === t || "TRUE" === t || "false" !== t && "FALSE" !== t && (function (e) {
            if (s.test(e)) {
              var t = parseFloat(e);
              if (n < t && t < i) return !0;
            }
            return !1;
          }(t) ? parseFloat(t) : h.test(t) ? new Date(t) : "" === t ? null : t) : t;
          var r;
        }
        function k(e, t, r, i) {
          var n = {
            type: e,
            code: t,
            message: r
          };
          void 0 !== i && (n.row = i), c.errors.push(n);
        }
        this.parse = function (e, t, r) {
          var i = m.quoteChar || '"';
          if (m.newline || (m.newline = function (e, t) {
            e = e.substring(0, 1048576);
            var r = new RegExp(Q(t) + "([^]*?)" + Q(t), "gm"),
              i = (e = e.replace(r, "")).split("\r"),
              n = e.split("\n"),
              s = 1 < n.length && n[0].length < i[0].length;
            if (1 === i.length || s) return "\n";
            for (var a = 0, o = 0; o < i.length; o++) "\n" === i[o][0] && a++;
            return a >= i.length / 2 ? "\r\n" : "\r";
          }(e, i)), u = !1, m.delimiter) J(m.delimiter) && (m.delimiter = m.delimiter(e), c.meta.delimiter = m.delimiter);else {
            var n = function (e, t, r, i, n) {
              var s, a, o, u;
              n = n || [",", "\t", "|", ";", b.RECORD_SEP, b.UNIT_SEP];
              for (var h = 0; h < n.length; h++) {
                var f = n[h],
                  d = 0,
                  l = 0,
                  c = 0;
                o = void 0;
                for (var p = new E({
                    comments: i,
                    delimiter: f,
                    newline: t,
                    preview: 10
                  }).parse(e), g = 0; g < p.data.length; g++) if (r && y(p.data[g])) c++;else {
                  var _ = p.data[g].length;
                  l += _, void 0 !== o ? 0 < _ && (d += Math.abs(_ - o), o = _) : o = _;
                }
                0 < p.data.length && (l /= p.data.length - c), (void 0 === a || d <= a) && (void 0 === u || u < l) && 1.99 < l && (a = d, s = f, u = l);
              }
              return {
                successful: !!(m.delimiter = s),
                bestDelimiter: s
              };
            }(e, m.newline, m.skipEmptyLines, m.comments, m.delimitersToGuess);
            n.successful ? m.delimiter = n.bestDelimiter : (u = !0, m.delimiter = b.DefaultDelimiter), c.meta.delimiter = m.delimiter;
          }
          var s = w(m);
          return m.preview && m.header && s.preview++, a = e, o = new E(s), c = o.parse(a, t, r), g(), d ? {
            meta: {
              paused: !0
            }
          } : c || {
            meta: {
              paused: !1
            }
          };
        }, this.paused = function () {
          return d;
        }, this.pause = function () {
          d = !0, o.abort(), a = J(m.chunk) ? "" : a.substring(o.getCharIndex());
        }, this.resume = function () {
          t.streamer._halted ? (d = !1, t.streamer.parseChunk(a, !0)) : setTimeout(t.resume, 3);
        }, this.aborted = function () {
          return e;
        }, this.abort = function () {
          e = !0, o.abort(), c.meta.aborted = !0, J(m.complete) && m.complete(c), a = "";
        };
      }
      function Q(e) {
        return e.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
      }
      function E(j) {
        var z,
          M = (j = j || {}).delimiter,
          P = j.newline,
          U = j.comments,
          q = j.step,
          N = j.preview,
          B = j.fastMode,
          K = z = void 0 === j.quoteChar || null === j.quoteChar ? '"' : j.quoteChar;
        if (void 0 !== j.escapeChar && (K = j.escapeChar), ("string" != typeof M || -1 < b.BAD_DELIMITERS.indexOf(M)) && (M = ","), U === M) throw new Error("Comment character same as delimiter");
        !0 === U ? U = "#" : ("string" != typeof U || -1 < b.BAD_DELIMITERS.indexOf(U)) && (U = !1), "\n" !== P && "\r" !== P && "\r\n" !== P && (P = "\n");
        var W = 0,
          H = !1;
        this.parse = function (i, t, r) {
          if ("string" != typeof i) throw new Error("Input must be a string");
          var n = i.length,
            e = M.length,
            s = P.length,
            a = U.length,
            o = J(q),
            u = [],
            h = [],
            f = [],
            d = W = 0;
          if (!i) return L();
          if (j.header && !t) {
            var l = i.split(P)[0].split(M),
              c = [],
              p = {},
              g = !1;
            for (var _ in l) {
              var m = l[_];
              J(j.transformHeader) && (m = j.transformHeader(m, _));
              var y = m,
                v = p[m] || 0;
              for (0 < v && (g = !0, y = m + "_" + v), p[m] = v + 1; c.includes(y);) y = y + "_" + v;
              c.push(y);
            }
            if (g) {
              var k = i.split(P);
              k[0] = c.join(M), i = k.join(P);
            }
          }
          if (B || !1 !== B && -1 === i.indexOf(z)) {
            for (var b = i.split(P), E = 0; E < b.length; E++) {
              if (f = b[E], W += f.length, E !== b.length - 1) W += P.length;else if (r) return L();
              if (!U || f.substring(0, a) !== U) {
                if (o) {
                  if (u = [], I(f.split(M)), F(), H) return L();
                } else I(f.split(M));
                if (N && N <= E) return u = u.slice(0, N), L(!0);
              }
            }
            return L();
          }
          for (var w = i.indexOf(M, W), R = i.indexOf(P, W), C = new RegExp(Q(K) + Q(z), "g"), S = i.indexOf(z, W);;) if (i[W] !== z) {
            if (U && 0 === f.length && i.substring(W, W + a) === U) {
              if (-1 === R) return L();
              W = R + s, R = i.indexOf(P, W), w = i.indexOf(M, W);
            } else if (-1 !== w && (w < R || -1 === R)) f.push(i.substring(W, w)), W = w + e, w = i.indexOf(M, W);else {
              if (-1 === R) break;
              if (f.push(i.substring(W, R)), D(R + s), o && (F(), H)) return L();
              if (N && u.length >= N) return L(!0);
            }
          } else for (S = W, W++;;) {
            if (-1 === (S = i.indexOf(z, S + 1))) return r || h.push({
              type: "Quotes",
              code: "MissingQuotes",
              message: "Quoted field unterminated",
              row: u.length,
              index: W
            }), T();
            if (S === n - 1) return T(i.substring(W, S).replace(C, z));
            if (z !== K || i[S + 1] !== K) {
              if (z === K || 0 === S || i[S - 1] !== K) {
                -1 !== w && w < S + 1 && (w = i.indexOf(M, S + 1)), -1 !== R && R < S + 1 && (R = i.indexOf(P, S + 1));
                var O = A(-1 === R ? w : Math.min(w, R));
                if (i.substr(S + 1 + O, e) === M) {
                  f.push(i.substring(W, S).replace(C, z)), i[W = S + 1 + O + e] !== z && (S = i.indexOf(z, W)), w = i.indexOf(M, W), R = i.indexOf(P, W);
                  break;
                }
                var x = A(R);
                if (i.substring(S + 1 + x, S + 1 + x + s) === P) {
                  if (f.push(i.substring(W, S).replace(C, z)), D(S + 1 + x + s), w = i.indexOf(M, W), S = i.indexOf(z, W), o && (F(), H)) return L();
                  if (N && u.length >= N) return L(!0);
                  break;
                }
                h.push({
                  type: "Quotes",
                  code: "InvalidQuotes",
                  message: "Trailing quote on quoted field is malformed",
                  row: u.length,
                  index: W
                }), S++;
              }
            } else S++;
          }
          return T();
          function I(e) {
            u.push(e), d = W;
          }
          function A(e) {
            var t = 0;
            if (-1 !== e) {
              var r = i.substring(S + 1, e);
              r && "" === r.trim() && (t = r.length);
            }
            return t;
          }
          function T(e) {
            return r || (void 0 === e && (e = i.substring(W)), f.push(e), W = n, I(f), o && F()), L();
          }
          function D(e) {
            W = e, I(f), f = [], R = i.indexOf(P, W);
          }
          function L(e) {
            return {
              data: u,
              errors: h,
              meta: {
                delimiter: M,
                linebreak: P,
                aborted: H,
                truncated: !!e,
                cursor: d + (t || 0)
              }
            };
          }
          function F() {
            q(L()), u = [], h = [];
          }
        }, this.abort = function () {
          H = !0;
        }, this.getCharIndex = function () {
          return W;
        };
      }
      function _(e) {
        var t = e.data,
          r = a[t.workerId],
          i = !1;
        if (t.error) r.userError(t.error, t.file);else if (t.results && t.results.data) {
          var n = {
            abort: function () {
              i = !0, m(t.workerId, {
                data: [],
                errors: [],
                meta: {
                  aborted: !0
                }
              });
            },
            pause: y,
            resume: y
          };
          if (J(r.userStep)) {
            for (var s = 0; s < t.results.data.length && (r.userStep({
              data: t.results.data[s],
              errors: t.results.errors,
              meta: t.results.meta
            }, n), !i); s++);
            delete t.results;
          } else J(r.userChunk) && (r.userChunk(t.results, n, t.file), delete t.results);
        }
        t.finished && !i && m(t.workerId, t.results);
      }
      function m(e, t) {
        var r = a[e];
        J(r.userComplete) && r.userComplete(t), r.terminate(), delete a[e];
      }
      function y() {
        throw new Error("Not implemented.");
      }
      function w(e) {
        if ("object" != typeof e || null === e) return e;
        var t = Array.isArray(e) ? [] : {};
        for (var r in e) t[r] = w(e[r]);
        return t;
      }
      function v(e, t) {
        return function () {
          e.apply(t, arguments);
        };
      }
      function J(e) {
        return "function" == typeof e;
      }
      return o && (f.onmessage = function (e) {
        var t = e.data;
        void 0 === b.WORKER_ID && t && (b.WORKER_ID = t.workerId);
        if ("string" == typeof t.input) f.postMessage({
          workerId: b.WORKER_ID,
          results: b.parse(t.input, t.config),
          finished: !0
        });else if (f.File && t.input instanceof File || t.input instanceof Object) {
          var r = b.parse(t.input, t.config);
          r && f.postMessage({
            workerId: b.WORKER_ID,
            results: r,
            finished: !0
          });
        }
      }), (l.prototype = Object.create(h.prototype)).constructor = l, (c.prototype = Object.create(h.prototype)).constructor = c, (p.prototype = Object.create(p.prototype)).constructor = p, (g.prototype = Object.create(h.prototype)).constructor = g, b;
    });
  })(papaparse_min);
  var Papa = papaparse_min.exports;

  function getMoleculeCreators(Molecule) {
    const fields = new Map();
    fields.set('oclid', Molecule.fromIDCode);
    fields.set('idcode', Molecule.fromIDCode);
    fields.set('smiles', Molecule.fromSmiles);
    fields.set('molfile', Molecule.fromMolfile);
    return fields;
  }

  const defaultCSVOptions = {
    header: true,
    dynamicTyping: true,
    skipEmptyLines: true
  };
  async function appendCSV(moleculesDB, csv, options = {}) {
    const {
      onStep
    } = options;
    csv = ensureString(csv);
    const moleculeCreators = getMoleculeCreators(moleculesDB.OCL.Molecule);
    if (typeof csv !== 'string') {
      throw new TypeError('csv must be a string');
    }
    options = {
      ...defaultCSVOptions,
      ...options
    };
    const parsed = Papa.parse(csv, options);
    const fields = parsed.meta.fields;
    const stats = new Array(fields.length);
    const firstElement = parsed.data[0];
    let moleculeCreator, moleculeField;
    for (let i = 0; i < fields.length; i++) {
      stats[i] = {
        label: fields[i],
        isNumeric: typeof firstElement[fields[i]] === 'number'
      };
      const lowerField = fields[i].toLowerCase();
      if (moleculeCreators.has(lowerField)) {
        moleculeCreator = moleculeCreators.get(lowerField);
        moleculeField = fields[i];
      }
    }
    if (!moleculeCreator) {
      throw new Error('this document does not contain any molecule field');
    }
    moleculesDB.statistics = stats;
    for (let i = 0; i < parsed.data.length; i++) {
      moleculesDB.pushEntry(moleculeCreator(parsed.data[i][moleculeField]), parsed.data[i]);
      if (onStep) {
        await onStep(i + 1, parsed.data.length);
      }
    }
  }

  function appendColor(moleculesDB, options = {}) {
    const {
      dataLabel,
      propertyLabel,
      minValue,
      maxValue,
      minHue = 0,
      maxHue = 360,
      saturation = 65,
      lightness = 65,
      colorLabel = 'color'
    } = options;
    const db = moleculesDB.getDB();
    let values;
    if (dataLabel) {
      values = db.map(result => result.data.map(datum => ({
        value: datum[dataLabel],
        data: datum
      }))).flat();
    } else if (propertyLabel) {
      values = db.map(result => result.data.map(datum => ({
        value: result.properties[propertyLabel],
        data: datum
      }))).flat();
    } else {
      values = db.map(result => result.data.map(datum => ({
        value: undefined,
        data: datum
      }))).flat();
    }
    if (minValue !== undefined) {
      values = values.forEach(value => {
        if (value.value !== undefined && value.value < minValue) {
          value.value = minValue;
        }
      });
    }
    if (maxValue !== undefined) {
      values = values.forEach(value => {
        if (value.value !== undefined && value.value > maxValue) {
          value.value = maxValue;
        }
      });
    }
    const definedValues = values.filter(value => value.value !== undefined);
    const min = Math.min(...definedValues.map(value => value.value));
    const max = Math.max(...definedValues.map(value => value.value));
    for (let value of values) {
      if (value.value !== undefined) {
        value.data[colorLabel] = `hsl(${Math.floor((value.value - min) / (max - min) * (maxHue - minHue) + minHue)},${saturation}%,${lightness}%)`;
      } else {
        value.data.color = 'black';
      }
    }
  }

  function getEntriesBoundaries(string, substring, eol) {
    const res = [];
    let previous = 0;
    let next = 0;
    while (next !== -1) {
      next = string.indexOf(substring, previous);
      if (next !== -1) {
        res.push([previous, next]);
        const nextMatch = string.indexOf(eol, next + substring.length);
        if (nextMatch === -1) {
          next = -1;
        } else {
          previous = nextMatch + eol.length;
          next = previous;
        }
      } else {
        res.push([previous, string.length]);
      }
    }
    return res;
  }

  function getMolecule(sdfPart, labels, currentLabels, options) {
    let parts = sdfPart.split(`${options.eol}>`);
    if (parts.length === 0 || parts[0].length <= 5) return;
    let molecule = {};
    molecule.molfile = parts[0] + options.eol;
    for (let j = 1; j < parts.length; j++) {
      let lines = parts[j].split(options.eol);
      let from = lines[0].indexOf('<');
      let to = lines[0].indexOf('>');
      let label = lines[0].substring(from + 1, to);
      currentLabels.push(label);
      if (!labels[label]) {
        labels[label] = {
          counter: 0,
          isNumeric: options.dynamicTyping,
          keep: false
        };
        if ((!options.exclude || options.exclude.indexOf(label) === -1) && (!options.include || options.include.indexOf(label) > -1)) {
          labels[label].keep = true;
          if (options.modifiers[label]) {
            labels[label].modifier = options.modifiers[label];
          }
          if (options.forEach[label]) {
            labels[label].forEach = options.forEach[label];
          }
        }
      }
      if (labels[label].keep) {
        for (let k = 1; k < lines.length - 1; k++) {
          if (molecule[label]) {
            molecule[label] += options.eol + lines[k];
          } else {
            molecule[label] = lines[k];
          }
        }
        if (labels[label].modifier) {
          let modifiedValue = labels[label].modifier(molecule[label]);
          if (modifiedValue === undefined || modifiedValue === null) {
            delete molecule[label];
          } else {
            molecule[label] = modifiedValue;
          }
        }
        if (labels[label].isNumeric) {
          if (!isFinite(molecule[label]) || molecule[label].match(/^0[0-9]/)) {
            labels[label].isNumeric = false;
          }
        }
      }
    }
    return molecule;
  }

  /**
   *  Parse a SDF file
   * @param {string|ArrayBuffer|Uint8Array} sdf SDF file to parse
   * @param {object} [options={}]
   * @param {string[]} [options.include] List of fields to include
   * @param {string[]} [options.exclude] List of fields to exclude
   * @param {Function} [options.filter] Callback allowing to filter the molecules
   * @param {boolean} [options.dynamicTyping] Dynamically type the data
   * @param {object} [options.modifiers] Object containing callbacks to apply on some specific fields
   * @param {boolean} [options.mixedEOL=false] Set to true if you know there is a mixture between \r\n and \n
   * @param {string} [options.eol] Specify the end of line character. Default will be the one found in the file
   */
  function parse(sdf, options = {}) {
    options = {
      ...options
    };
    if (options.modifiers === undefined) options.modifiers = {};
    if (options.forEach === undefined) options.forEach = {};
    if (options.dynamicTyping === undefined) options.dynamicTyping = true;
    sdf = ensureString(sdf);
    if (typeof sdf !== 'string') {
      throw new TypeError('Parameter "sdf" must be a string');
    }
    if (options.eol === undefined) {
      options.eol = '\n';
      if (options.mixedEOL) {
        sdf = sdf.replace(/\r\n/g, '\n');
        sdf = sdf.replace(/\r/g, '\n');
      } else {
        // we will find the delimiter in order to be much faster and not use regular expression
        let header = sdf.substr(0, 1000);
        if (header.indexOf('\r\n') > -1) {
          options.eol = '\r\n';
        } else if (header.indexOf('\r') > -1) {
          options.eol = '\r';
        }
      }
    }
    let entriesBoundaries = getEntriesBoundaries(sdf, `${options.eol}$$$$`, options.eol);
    let molecules = [];
    let labels = {};
    let start = Date.now();
    for (let i = 0; i < entriesBoundaries.length; i++) {
      let sdfPart = sdf.substring(...entriesBoundaries[i]);
      let currentLabels = [];
      const molecule = getMolecule(sdfPart, labels, currentLabels, options);
      if (!molecule) continue;
      if (!options.filter || options.filter(molecule)) {
        molecules.push(molecule);
        // only now we can increase the counter
        for (let j = 0; j < currentLabels.length; j++) {
          labels[currentLabels[j]].counter++;
        }
      }
    }
    // all numeric fields should be converted to numbers
    for (let label in labels) {
      let currentLabel = labels[label];
      if (currentLabel.isNumeric) {
        currentLabel.minValue = Infinity;
        currentLabel.maxValue = -Infinity;
        for (let j = 0; j < molecules.length; j++) {
          if (molecules[j][label]) {
            let value = parseFloat(molecules[j][label]);
            molecules[j][label] = value;
            if (value > currentLabel.maxValue) {
              currentLabel.maxValue = value;
            }
            if (value < currentLabel.minValue) {
              currentLabel.minValue = value;
            }
          }
        }
      }
    }

    // we check that a label is in all the records
    for (let key in labels) {
      if (labels[key].counter === molecules.length) {
        labels[key].always = true;
      } else {
        labels[key].always = false;
      }
    }
    let statistics = [];
    for (let key in labels) {
      let statistic = labels[key];
      statistic.label = key;
      statistics.push(statistic);
    }
    return {
      time: Date.now() - start,
      molecules,
      labels: Object.keys(labels),
      statistics
    };
  }

  async function appendSDF(moleculesDB, sdf, options = {}) {
    const {
      onStep
    } = options;
    sdf = ensureString(sdf);
    if (typeof sdf !== 'string') {
      throw new TypeError('sdf must be a string');
    }
    const parsed = parse(sdf);
    moleculesDB.statistics = parsed.statistics;
    for (let i = 0; i < parsed.molecules.length; i++) {
      const molecule = parsed.molecules[i];
      moleculesDB.pushEntry(moleculesDB.OCL.Molecule.fromMolfile(molecule.molfile), molecule);
      if (onStep) {
        await onStep(i + 1, parsed.molecules.length);
      }
    }
  }

  async function appendSmilesList(moleculesDB, text, options = {}) {
    const {
      onStep
    } = options;
    text = ensureString(text);
    if (typeof text !== 'string') {
      throw new TypeError('text must be a string');
    }
    const smilesArray = text.split(/\r?\n/).map(line => line.trim()).filter(line => line);
    for (let i = 0; i < smilesArray.length; i++) {
      const oneSmiles = smilesArray[i];
      moleculesDB.pushEntry(moleculesDB.OCL.Molecule.fromSmiles(oneSmiles));
      if (onStep) {
        await onStep(i + 1, smilesArray.length);
      }
    }
  }

  /**
   *
   * @param {MoleculesDB} moleculesDB
   * @param {import('openchemlib').Molecule} molecule
   * @param {object} data
   * @param {object} [moleculeInfo]
   * @param {string} [moleculeInfo.idCode]
   * @param {number[]} [moleculeInfo.index]
   */

  function pushEntry(moleculesDB, molecule, data = {}, moleculeInfo = {}) {
    // the following line could be the source of problems if the idCode version
    // changes

    let moleculeIDCode = moleculeInfo.idCode ? moleculeInfo.idCode : molecule.getIDCode();
    let entry = moleculesDB.db[moleculeIDCode];
    if (!entry) {
      // a new molecule
      entry = {
        molecule,
        properties: {},
        data: [],
        idCode: moleculeIDCode
      };
      moleculesDB.db[moleculeIDCode] = entry;

      // ensure helper arrays needed for substructure search
      molecule.ensureHelperArrays(moleculesDB.OCL.Molecule.cHelperRings);
      if (!moleculeInfo.index) {
        entry.index = molecule.getIndex();
      } else {
        entry.index = moleculeInfo.index;
      }
      let molecularFormula;
      if (!moleculeInfo.mw) {
        molecularFormula = molecule.getMolecularFormula();
        entry.properties.mw = molecularFormula.relativeWeight;
      } else {
        entry.properties.mw = moleculeInfo.mw;
      }
      if (moleculesDB.computeProperties) {
        if (!molecularFormula) {
          molecularFormula = molecule.getMolecularFormula();
        }
        const properties = new moleculesDB.OCL.MoleculeProperties(molecule);
        entry.properties.em = molecularFormula.absoluteWeight;
        entry.properties.mf = molecularFormula.formula;
        entry.properties.acceptorCount = properties.acceptorCount;
        entry.properties.donorCount = properties.donorCount;
        entry.properties.logP = properties.logP;
        entry.properties.logS = properties.logS;
        entry.properties.polarSurfaceArea = properties.polarSurfaceArea;
        entry.properties.rotatableBondCount = properties.rotatableBondCount;
        entry.properties.stereoCenterCount = properties.stereoCenterCount;
      }
    }
    entry.data.push(data);
  }

  function pushMoleculeInfo(moleculesDB, moleculeInfo, data = {}) {
    if (typeof moleculeInfo !== 'object') {
      throw new Error('pushMoleculeInfo requires an object as first parameter');
    }
    const Molecule = moleculesDB.OCL.Molecule;
    let molecule;
    if (moleculeInfo.molfile) {
      molecule = Molecule.fromMolfile(moleculeInfo.molfile);
    }
    if (moleculeInfo.smiles) molecule = Molecule.fromSmiles(moleculeInfo.smiles);
    if (moleculeInfo.idCode) {
      if (moleculesDB.db[moleculeInfo.idCode]) {
        molecule = moleculesDB.db[moleculeInfo.idCode].molecule;
      } else {
        molecule = Molecule.fromIDCode(moleculeInfo.idCode, moleculeInfo.coordinates || false);
      }
    }
    if (molecule) {
      moleculesDB.pushEntry(molecule, data, moleculeInfo);
    }
  }

  async function noWait() {
    return new Promise(resolve => {
      if (typeof setImmediate === 'function') {
        setImmediate(() => resolve());
      } else {
        // didn't find a better way to do it in the browser
        setTimeout(() => resolve(), 0);
      }
    });
  }

  class AbortError extends Error {
    name = 'AbortError';
    code = 20;
  }
  function getQuery(moleculesDB, query, options) {
    const {
      format = 'idCode'
    } = options;
    if (typeof query === 'string') {
      const moleculeCreators = getMoleculeCreators(moleculesDB.OCL.Molecule);
      query = moleculeCreators.get(format.toLowerCase())(query);
    } else if (!(query instanceof moleculesDB.OCL.Molecule)) {
      throw new TypeError('toSearch must be a Molecule or string');
    }
    return query;
  }
  function search(moleculesDB, query = '', options = {}) {
    const {
      mode = 'substructure'
    } = options;
    query = getQuery(moleculesDB, query, options);
    let result;
    switch (mode.toLowerCase()) {
      case 'exact':
        result = exactSearch(moleculesDB, query);
        break;
      case 'substructure':
        result = subStructureSearch(moleculesDB, query);
        break;
      case 'similarity':
        result = similaritySearch(moleculesDB, query);
        break;
      default:
        throw new Error(`unknown search mode: ${options.mode}`);
    }
    return processResult(result, options);
  }
  async function searchAsync(moleculesDB, query = '', options = {}) {
    const {
      mode = 'substructure'
    } = options;
    query = getQuery(moleculesDB, query, options);
    let result;
    switch (mode.toLowerCase()) {
      case 'exact':
        result = exactSearch(moleculesDB, query);
        break;
      case 'substructure':
        result = await subStructureSearchAsync(moleculesDB, query, options);
        break;
      case 'similarity':
        result = similaritySearch(moleculesDB, query);
        break;
      default:
        throw new Error(`unknown search mode: ${options.mode}`);
    }
    return processResult(result, options);
  }
  function exactSearch(moleculesDB, query) {
    const queryIDCode = query.getIDCode();
    let searchResult = moleculesDB.db[queryIDCode] ? [moleculesDB.db[queryIDCode]] : [];
    return searchResult;
  }
  function substructureSearchBegin(moleculesDB, query) {
    let resetFragment = false;
    if (!query.isFragment()) {
      resetFragment = true;
      query.setFragment(true);
    }
    const queryMW = getMW(query);
    const searchResult = [];
    if (query.getAllAtoms() === 0) {
      for (let idCode in moleculesDB.db) {
        searchResult.push(moleculesDB.db[idCode]);
      }
    }
    return {
      resetFragment,
      queryMW,
      searchResult
    };
  }
  function substructureSearchEnd(searchResult, queryMW, resetFragment, query) {
    searchResult.sort((a, b) => {
      return Math.abs(queryMW - a.properties.mw) - Math.abs(queryMW - b.properties.mw);
    });
    if (resetFragment) {
      query.setFragment(false);
    }
    return searchResult;
  }
  function subStructureSearch(moleculesDB, query) {
    const {
      resetFragment,
      queryMW,
      searchResult
    } = substructureSearchBegin(moleculesDB, query);
    if (searchResult.length === 0) {
      const queryIndex = query.getIndex();
      const searcher = moleculesDB.searcher;
      searcher.setFragment(query, queryIndex);
      for (let idCode in moleculesDB.db) {
        let entry = moleculesDB.db[idCode];
        searcher.setMolecule(entry.molecule, entry.index);
        if (searcher.isFragmentInMolecule()) {
          searchResult.push(entry);
        }
      }
    }
    return substructureSearchEnd(searchResult, queryMW, resetFragment, query);
  }
  async function subStructureSearchAsync(moleculesDB, query, options = {}) {
    const {
      interval = 100,
      onStep,
      controller
    } = options;
    let shouldAbort = false;
    if (controller) {
      const abortEventListener = () => {
        shouldAbort = true;
      };
      controller.signal.addEventListener('abort', abortEventListener);
    }
    const {
      resetFragment,
      queryMW,
      searchResult
    } = substructureSearchBegin(moleculesDB, query);
    let begin = performance.now();
    if (searchResult.length === 0) {
      const queryIndex = query.getIndex();
      const searcher = moleculesDB.searcher;
      searcher.setFragment(query, queryIndex);
      let index = 0;
      let length = Object.keys(moleculesDB.db).length;
      for (let idCode in moleculesDB.db) {
        if (shouldAbort) {
          throw new AbortError('Query aborted');
        }
        let entry = moleculesDB.db[idCode];
        searcher.setMolecule(entry.molecule, entry.index);
        if (searcher.isFragmentInMolecule()) {
          searchResult.push(entry);
        }
        if ((onStep || controller) && performance.now() - begin >= interval) {
          begin = performance.now();
          if (onStep) {
            onStep(index, length);
          }
          if (controller && !onStep) {
            await noWait();
          }
        }
        index++;
      }
    }
    return substructureSearchEnd(searchResult, queryMW, resetFragment, query);
  }
  function similaritySearch(moleculesDB, query) {
    const queryIndex = query.getIndex();
    const queryMW = getMW(query);
    const queryIdCode = query.getIDCode();
    const searchResult = [];
    let similarity;
    for (let idCode in moleculesDB.db) {
      let entry = moleculesDB.db[idCode];
      if (entry.idCode === queryIdCode) {
        similarity = Number.MAX_SAFE_INTEGER;
      } else {
        similarity = moleculesDB.OCL.SSSearcherWithIndex.getSimilarityTanimoto(queryIndex, entry.index) * 1000000 - Math.abs(queryMW - entry.properties.mw) / 10000;
      }
      searchResult.push({
        similarity,
        entry
      });
    }
    searchResult.sort((a, b) => {
      return b.similarity - a.similarity;
    });
    return searchResult.map(entry => entry.entry);
  }
  function getMW(query) {
    let copy = query.getCompactCopy();
    copy.setFragment(false);
    return copy.getMolecularFormula().relativeWeight;
  }
  function processResult(entries, options = {}) {
    const {
      flattenResult = true,
      keepMolecule = false,
      limit = Number.MAX_SAFE_INTEGER
    } = options;
    let results = [];
    if (flattenResult) {
      for (let entry of entries) {
        for (let data of entry.data) {
          results.push({
            data,
            idCode: entry.idCode,
            properties: entry.properties,
            molecule: keepMolecule ? entry.molecule : undefined
          });
        }
      }
    } else {
      for (let entry of entries) {
        results.push({
          data: entry.data,
          idCode: entry.idCode,
          properties: entry.properties,
          molecule: keepMolecule ? entry.molecule : undefined
        });
      }
    }
    if (limit < results.length) results.length = limit;
    return results;
  }

  /*
      this.db is an object with properties 'oclID' that has as value
      an object that contains the following properties:
      * molecule: an OCL molecule instance
      * index: OCL index used for substructure searching
      * properties: all the calculates properties
      * data: array containing free data associated with this molecule
    */

  class MoleculesDB {
    /**
     *
     * @param {import('openchemlib')} OCL - openchemlib library
     * @param {object} [options={}]
     * @param {boolean} [options.computeProperties=false]
     */
    constructor(OCL, options = {}) {
      const {
        computeProperties = false
      } = options;
      this.OCL = OCL;
      this.db = {};
      this.statistics = null;
      this.computeProperties = computeProperties;
      this.searcher = new OCL.SSSearcherWithIndex();
    }

    /**
     * append to the current database a CSV file
     * @param {string|ArrayBuffer} csv - text file containing the comma separated value file
     * @param {object} [options={}]
     * @param {boolean} [options.header=true]
     * @param {boolean} [options.dynamicTyping=true]
     * @param {boolean} [options.skipEmptyLines=true]
     * @param {function} [options.onStep] call back to execute after each molecule
     */

    appendCSV(csv, options) {
      return appendCSV(this, csv, {
        computeProperties: this.computeProperties,
        ...options
      });
    }

    /**
     * Append a SDF to the current database
     * @param {string|ArrayBuffer} sdf - text file containing the sdf
     * @param {object} [options={}]
     * @param {function} [options.onStep] call back to execute after each molecule
     * @returns {DB}
     */

    appendSDF(sdf, options) {
      return appendSDF(this, sdf, {
        computeProperties: this.computeProperties,
        ...options
      });
    }

    /**
     * Append a SDF to the current database
     * @param {string|ArrayBuffer} smiles - text file containing a list of smiles
     * @param {object} [options={}]
     * @param {function} [options.onStep] call back to execute after each molecule
     * @returns {DB}
     */

    appendSmilesList(text, options) {
      return appendSmilesList(this, text, {
        computeProperties: this.computeProperties,
        ...options
      });
    }

    /**
     * Add a molecule to the current database
     * @param {import('openchemlib').Molecule} molecule
     * @param {object} [data={}]
     * @param {object} [moleculeInfo={}] may contain precalculated index and mw
     */

    pushEntry(molecule, data, moleculeInfo) {
      pushEntry(this, molecule, data, moleculeInfo);
    }

    /**
     * Add an entry in the database
     * @param {object} moleculeInfo - a molecule as a JSON that may contain the following properties: molfile, smiles, idCode, mf, index
     * @param {object} [data={}]
     */
    pushMoleculeInfo(moleculeInfo, data) {
      return pushMoleculeInfo(this, moleculeInfo, data);
    }

    /**
     * Search in a MoleculesDB
     * Inside the database all the same molecules are group together
     * @param {string|OCL.Molecule} [query] smiles, molfile, oclCode or instance of Molecule to look for
     * @param {object} [options={}]
     * @param {string} [options.format='idCode'] - query is in the format 'smiles', 'oclid' or 'molfile'
     * @param {string} [options.mode='substructure'] - search by 'substructure', 'exact' or 'similarity'
     * @param {boolean} [options.flattenResult=true] - The database group the data for the same product. This allows to flatten the result
     * @param {boolean} [options.keepMolecule=false] - keep the OCL.Molecule object in the result
     * @param {number} [options.limit=Number.MAX_SAFE_INTEGER] - maximal number of result
     * @return {Array} array of object of the type {(molecule), idCode, data, properties}
     */
    search(query, options) {
      return search(this, query, options);
    }

    /**
     * Search in a MoleculesDB
     * Inside the database all the same molecules are group together
     * @param {string|OCL.Molecule} [query] smiles, molfile, oclCode or instance of Molecule to look for
     * @param {object} [options={}]
     * @param {string} [options.format='idCode'] - query is in the format 'smiles', 'oclid' or 'molfile'
     * @param {string} [options.mode='substructure'] - search by 'substructure', 'exact' or 'similarity'
     * @param {boolean} [options.flattenResult=true] - The database group the data for the same product. This allows to flatten the result
     * @param {boolean} [options.keepMolecule=false] - keep the OCL.Molecule object in the result
     * @param {number} [options.limit=Number.MAX_SAFE_INTEGER] - maximal number of result
     * @param {number} [options.interval=100] - interval in ms to call the onStep callback
     * @param {function} [options.onStep] - callback to execute after each interval
     * @param {AbortController} [options.controler] - callback to execute to check if the search should be aborted
     * @return {Promise<Array>} array of object of the type {(molecule), idCode, data, properties}
     */
    searchAsync(query, options) {
      return searchAsync(this, query, options);
    }

    /**
     * Returns an array with the current database
     * @returns
     */
    getDB() {
      return Object.keys(this.db).map(key => this.db[key]);
    }

    /**
     * Append the property `data.color` to each entry based on a data or property label
     * {object} [options={}]
     * {string} [options.dataLabel] name of the property from `data` to use
     * {string} [options.propertyLabel] name of the property from `properties` to use
     * {number} [options.colorLabel='color'] name of the property to add in data that will contain the color
     * {number} [options.minValue]
     * {number} [options.maxValue]
     * {number} [options.minHue=0]
     * {number} [options.maxHue=360]
     * {number} [options.saturation=65] percent of color saturation
     * {number} [options.lightness=65] percent of color lightness
     */
    appendColor(options) {
      appendColor(this, options);
    }
  }

  function getAtomFeatures(originalMolecule, options = {}) {
    const OCL = originalMolecule.getOCL();
    const {
      sphere = 1
    } = options;
    const fragment = new OCL.Molecule(0, 0);
    const results = [];
    for (let rootAtom = 0; rootAtom < originalMolecule.getAllAtoms(); rootAtom++) {
      let min = 0;
      let max = 0;
      let atomMask = new Array(originalMolecule.getAtoms());
      let atomList = new Array(originalMolecule.getAtoms());
      const molecule = originalMolecule.getCompactCopy();
      for (let currentSphere = 0; currentSphere <= sphere; currentSphere++) {
        if (max === 0) {
          atomList[max] = rootAtom;
          atomMask[rootAtom] = true;
          max++;
        } else {
          let newMax = max;
          for (let i = min; i < max; i++) {
            let atom = atomList[i];
            for (let j = 0; j < molecule.getAllConnAtoms(atom); j++) {
              let connAtom = molecule.getConnAtom(atom, j);
              if (!atomMask[connAtom]) {
                atomMask[connAtom] = true;
                atomList[newMax++] = connAtom;
              }
            }
          }
          min = max;
          max = newMax;
        }
        molecule.copyMoleculeByAtoms(fragment, atomMask, true, null);
        if (currentSphere === sphere) {
          makeRacemic(fragment);
          results.push(fragment.getCanonizedIDCode());
        }
      }
    }
    const atoms = {};
    for (let result of results) {
      if (!atoms[result]) {
        atoms[result] = 1;
      } else {
        atoms[result]++;
      }
    }
    return atoms;
  }

  function toVisualizerMolfile(molecule, options = {}) {
    const {
      diastereotopic,
      heavyAtomHydrogen
    } = options;
    let highlight = [];
    let atoms = {};
    if (diastereotopic) {
      let hydrogenInfo = {};
      let extendedIDs = getDiastereotopicAtomIDsAndH(molecule);
      for (let line of extendedIDs) {
        hydrogenInfo[line.oclID] = line;
      }
      let diaIDs = getGroupedDiastereotopicAtomIDs(molecule);
      for (const diaID of diaIDs) {
        atoms[diaID.oclID] = diaID.atoms;
        highlight.push(diaID.oclID);
        if (heavyAtomHydrogen) {
          if (hydrogenInfo[diaID.oclID] && hydrogenInfo[diaID.oclID].nbHydrogens > 0) {
            for (let id of hydrogenInfo[diaID.oclID].hydrogenOCLIDs) {
              highlight.push(id);
              atoms[id] = diaID.atoms;
            }
          }
        }
      }
    } else {
      let size = molecule.getAllAtoms();
      highlight = new Array(size).fill(0).map((a, index) => index);
      atoms = highlight.map(a => [a]);
    }
    let molfile = {
      type: 'mol2d',
      value: molecule.toMolfile(),
      _highlight: highlight,
      _atoms: atoms
    };
    return molfile;
  }

  function getParts(text) {
    const lines = text.split(/\r?\n/);
    let parts = {
      data: []
    };
    let currentPart = parts.data;
    let currentLabel = '';
    for (let line of lines) {
      if (line.startsWith('</')) {
        // close existing part
        if (!currentLabel === line.slice(2, -1)) {
          throw new Error('This should not happen');
        }
        currentLabel = '';
        currentPart = parts.data;
      } else if (line.startsWith('<') && !line.includes('=')) {
        // open new part
        if (currentLabel) {
          throw new Error('This should not happen');
        }
        currentLabel = line.slice(1, -1);
        const target = getCamelCase(currentLabel);
        parts[target] = [];
        currentPart = parts[target];
      } else if (currentLabel) {
        // add line to current part
        currentPart.push(line);
      } else {
        //data lines
        currentPart.push(line);
      }
    }
    return parts;
  }

  function parseColumnbProperties(lines) {
    lines = lines.map(line => {
      const [key, value] = line.slice(1, -1).split('=');
      return {
        key,
        value: value.slice(1, -1)
      };
    });
    const columnProperties = {};
    let currentColumnName = '';
    for (let line of lines) {
      switch (line.key) {
        case 'columnName':
          currentColumnName = line.value;
          columnProperties[currentColumnName] = {};
          break;
        case 'columnProperty':
          {
            if (!currentColumnName) {
              throw new Error('This should not happen');
            }
            const [key, value] = line.value.split('\t');
            columnProperties[currentColumnName][key] = value;
          }
          break;
        default:
          throw new Error('This should not happen');
      }
    }
    for (let key in columnProperties) {
      const columnPropery = columnProperties[key];
      if (columnProperties[key].parent) {
        const target = columnProperties[columnPropery.parent];
        if (!target) {
          throw new Error('Parent column not found');
        }
        if (!target.related) {
          target.related = {};
        }
        target.related[columnPropery.specialType] = key;
      }
    }
    return columnProperties;
  }

  function parseData(lines, options = {}) {
    lines = lines.filter(line => !line.match(/^\s*$/));
    const {
      columnProperties = {}
    } = options;
    const headers = lines.shift().split('\t').map(header => {
      if (columnProperties[header]) {
        return {
          label: header,
          ...columnProperties[header]
        };
      }
      return {
        label: header
      };
    });
    const entries = [];
    const rawEntries = [];
    for (let line of lines) {
      const fields = line.split('\t');
      const rawEntry = {};
      headers.forEach((header, index) => {
        rawEntry[header.label] = fields[index];
      });
      rawEntries.push(rawEntry);
      const entry = {};
      headers.forEach(header => {
        if (header.parent) return;
        entry[header.label] = valueEhnhancer(header, rawEntry);
      });
      entries.push(entry);
    }
    return {
      entries,
      rawEntries
    };
  }
  function valueEhnhancer(header, rawEntry) {
    if (header?.specialType === 'rxncode') {
      return `${rawEntry[header.label]}#${rawEntry[header.related.atomMapping]}#${rawEntry[header.related.idcoordinates2D]}`;
    }
    return rawEntry[header.label];
  }

  /*
  entry.rxnCode =
  */

  /**
   * Convert a DataWarrior database into a JSON object
   * @param {string} text
   * @returns
   */
  function parseDwar(text) {
    text = ensureString(text);
    const parts = getParts(text);
    improveParts(parts);
    return parts;
  }
  function getCamelCase(name) {
    return name.replace(/[ -][a-z]/g, string => string[1].toUpperCase());
  }
  function improveParts(parts) {
    for (let key in parts) {
      switch (key) {
        case 'columnProperties':
          parts[key] = parseColumnbProperties(parts[key]);
          break;
        case 'data':
          break;
        default:
          parts[key] = parseDefault(parts[key]);
      }
    }
    const data = parseData(parts.data, {
      columnProperties: parts.columnProperties
    });
    parts.data = data.entries;
    parts.rawData = data.rawEntries;
  }
  function parseDefault(lines) {
    const result = {};
    for (let line of lines) {
      const [key, value] = line.slice(1, -1).split('=');
      result[key] = value.slice(1, -1);
    }
    return result;
  }

  function fragmentAcyclicSingleBonds(molecule) {
    const OCL = molecule.getOCL();
    let atoms = [];
    for (let i = 0; i < molecule.getAllAtoms(); i++) {
      let atom = {};
      atoms.push(atom);
      atom.i = i;
      atom.links = []; // we will store connected atoms of broken bonds
    }

    let bonds = [];
    for (let i = 0; i < molecule.getAllBonds(); i++) {
      let bond = {};
      bonds.push(bond);
      bond.i = i;
      bond.order = molecule.getBondOrder(i);
      bond.atom1 = molecule.getBondAtom(0, i);
      bond.atom2 = molecule.getBondAtom(1, i);
      bond.type = molecule.getBondType(i);
      bond.isAromatic = molecule.isAromaticBond(i);
      bond.isRingBond = molecule.isRingBond(i);
      if (!bond.isAromatic && (bond.type & 0b11) === 1 && !bond.isRingBond) {
        bond.selected = true;
        atoms[bond.atom1].links.push(bond.atom2);
        atoms[bond.atom2].links.push(bond.atom1);
      }
    }

    //  console.log(bonds);

    let brokenMolecule = molecule.getCompactCopy();
    for (let bond of bonds) {
      if (bond.selected) {
        brokenMolecule.markBondForDeletion(bond.i);
      }
    }
    brokenMolecule.deleteMarkedAtomsAndBonds();
    let fragmentMap = [];
    let nbFragments = brokenMolecule.getFragmentNumbers(fragmentMap);
    let results = [];
    for (let i = 0; i < nbFragments; i++) {
      let result = {};
      result.atomMap = [];
      let includeAtom = fragmentMap.map(id => {
        return id === i;
      });
      let fragment = new OCL.Molecule(0, 0);
      let atomMap = [];
      brokenMolecule.copyMoleculeByAtoms(fragment, includeAtom, false, atomMap);
      // we will add some R groups at the level of the broken bonds
      for (let j = 0; j < atomMap.length; j++) {
        if (atomMap[j] > -1) {
          result.atomMap.push(j);
          if (atoms[j].links.length > 0) {
            for (let k = 0; k < atoms[j].links.length; k++) {
              fragment.addBond(atomMap[j], fragment.addAtom(154), 1);
            }
          }
        }
      }
      fragment.setFragment(false);
      result.idCode = fragment.getIDCode();
      result.mf = getMF(fragment).mf.replace(/R[1-9]?/, '');
      results.push(result);
    }
    return results;
  }

  /**
   *
   * @param {import('openchemlib').Molecule} molecule
   * @param {Map} moleculesInfo
   * @returns
   */
  function getInfo(molecule, moleculesInfo) {
    if (moleculesInfo.has(molecule)) {
      return moleculesInfo.get(molecule);
    }
    const reactantInfo = {
      molfile: molecule.toMolfile(),
      idCode: molecule.getIDCode(),
      mf: getMF(molecule).mf
    };
    moleculesInfo.set(molecule, reactantInfo);
    return reactantInfo;
  }

  /**
   * @description apply one reaction to one reactant
   * @param {*} reactants either a molecule or an array of molecules
   * @param {Array<Object>} reactions rxnCode of the reaction
   * @param {Object} options options to apply the reaction
   * @param {number} options.currentDepth current depth of the recursion
   * @param {number} options.maxDepth max depth of the recursion
   * @param {Map} options.moleculesInfo map of molecules info
   * @param {Set} options.processedMolecules set of processed molecules
   * @param {Array} options.trees array of trees of previous recursions
   * @param {*} options.OCL OCL object
   * @returns {Array} array of results
   */
  function applyOneReactantReaction(reactants, reactions, options) {
    const {
      currentDepth,
      maxDepth,
      moleculesInfo,
      processedMolecules,
      trees
    } = options;
    const todoNextDepth = [];
    // if the current depth is greater than the max depth, we stop the recursion and return an empty array
    if (currentDepth >= maxDepth) return [];
    // if the reactants is not an array, we make it an array
    if (!Array.isArray(reactants)) {
      reactants = [reactants];
    }
    const {
      OCL
    } = options;
    for (const reactant of reactants) {
      const idCode = reactant.getIDCode();
      // check if reactant is charged
      let isReactantCharged = reactant.canonizeCharge(true);
      // check if the reactant has already been processed
      if (processedMolecules.has(idCode)) {
        continue;
      } else {
        processedMolecules.add(idCode);
      }
      for (const reaction of reactions) {
        // if the reaction need to be charged and the reactant is not charged, we continue to the next reaction
        // this is useful for charge remote reactions
        if (reaction.needToBeCharged && !isReactantCharged) {
          continue;
        }
        const reactor = new OCL.Reactor(reaction.oclReaction);
        // isMatching is true if the reactant is matching the reaction else we continue to the next reaction
        const isMatching = Boolean(reactor.setReactant(0, reactant));
        if (isMatching) {
          // get the products of the reaction
          const oneReactionProducts = reactor.getProducts();
          for (let i = 0; i < oneReactionProducts.length; i++) {
            const products = [];
            for (let j = 0; j < oneReactionProducts[i].length; j++) {
              // get the info of the product (molfile, idCode, mf)
              const moleculeInfo = getInfo(oneReactionProducts[i][j], moleculesInfo);
              // if the product has not been processed yet, we add it to the list of products and we add it to the list of todoNextDepth
              if (!processedMolecules.has(moleculeInfo.idCode)) {
                const product = {
                  ...moleculeInfo,
                  children: []
                };
                products.push(product);
                todoNextDepth.push(() => {
                  return applyOneReactantReaction(oneReactionProducts[i][j], reactions, {
                    ...options,
                    currentDepth: options.currentDepth + 1,
                    trees: product.children
                  });
                });
              }
            }
            // if there is at least one product, we add the reaction to the results
            if (products.length > 0) {
              // eslint-disable-next-line no-unused-vars
              const {
                oclReaction,
                needToBeCharged,
                ...reactionWithoutOCL
              } = reaction;
              const oneReaction = {
                reaction: reactionWithoutOCL,
                reactant: getInfo(reactant, moleculesInfo),
                products
              };
              trees.push(oneReaction);
            }
          }
        }
      }
    }
    // by returning todoNextDepth, we make sure that the recursion will continue
    return todoNextDepth;
  }

  /**
   * @description Trim the tree of reactions to keep only the paths to the product
   * @param {string} idCode idCode of the product
   * @param {Object} tree Tree of reactions
   * @param {Array} reactions Array of rxnCode of reactions
   */

  function trimTree(idCode, tree, reactions) {
    markProduct(idCode, tree);
    cutBranches(tree, reactions);
  }

  /**
   * @description For a given idCode, mark the products that contain it with a flag true and the others with a flag false
   * @param {string} idCode idCode of the product
   * @param {Object} tree Current branch of the tree of reactions
   */
  function markProduct(idCode, tree) {
    for (const product of tree.products) {
      product.flag = product.idCode === idCode;
      if (product.children.length > 0) {
        for (const child of product.children) {
          markProduct(idCode, child);
        }
      }
    }
  }

  /**
   * @description Check if the child branch has a flag true in its products
   * @param {Object} tree Current branch of the tree of reactions
   * @returns {boolean} true if the tree has a flag true in its products
   */
  function childHasFlag(tree) {
    for (const product of tree.products) {
      if (product.flag) {
        return true;
      }
      if (product.children.length > 0) {
        for (const child of product.children) {
          if (childHasFlag(child)) {
            return true;
          }
        }
      }
    }
    return false;
  }

  /**
   *@description Cut the branches of the tree that don't have a flag true in their products
   * @param {Object} tree Current branch of the tree of reactions
   * @param {Array} reactions Array of rxnCode of reactions
   */
  function cutBranches(tree, reactions) {
    reactions.push(tree.reaction.rxnCode);
    for (const product of tree.products) {
      if (product.flag) {
        product.children = [];
      }
      if (product.children.length > 0) {
        for (let child of product.children) {
          const hasFlag = childHasFlag(child);
          if (!hasFlag) {
            child.products = [];
          }
          cutBranches(child, reactions);
        }
        product.children = product.children.filter(child => child.products.length > 0);
      }
    }
  }

  /**
   * @description Group the trees by product idCode
   * @param {Array} trees Trees of reactions
   * @returns {Array} Array of products with their corresponding trees and reactions
   */
  function groupTreesByProducts(trees) {
    let results = {};
    for (const tree of trees) {
      let copyTree = JSON.parse(JSON.stringify(tree));
      groupProductTrees(copyTree, results, tree);
    }
    return Object.values(results);
  }

  /**
   * @description For a given tree, recursively group the branches leading to a each idCode
   * @param {Object} currentBranch Current recursive branch of the tree of reactions
   * @param {Object} results Object with the branches grouped by idCode
   * @param {Object} originalBranch Original tree of reactions (not modified)
   */
  function groupProductTrees(currentBranch, results, originalBranch) {
    for (let product of currentBranch.products) {
      // This way is faster than structuredClone
      let copyBranch = JSON.parse(JSON.stringify(originalBranch));
      let reactions = [];
      // Trim the tree to get all branches leading to the idCode of the product
      trimTree(product.idCode, copyBranch, reactions);
      let nbReactions = reactions.length;
      if (results[product.idCode] === undefined) {
        results[product.idCode] = {
          idCode: product.idCode,
          mf: product.mf,
          trees: [copyBranch],
          reactions,
          minSteps: nbReactions
        };
      } else {
        results[product.idCode].trees.push(copyBranch);
        if (nbReactions < results[product.idCode].minSteps) {
          results[product.idCode].minSteps = nbReactions;
          results[product.idCode].reactions = reactions;
        }
      }
      if (product.children.length > 0) {
        for (let child of product.children) {
          groupProductTrees(child, results, originalBranch);
        }
      }
    }
  }

  /**
   * Create reaction trees of products based on reactions and reactants
   * @param {import('openchemlib').Molecule[]} reactants
   * @param {Array} reactions array of reactions objects with rxnCode, label and needChargeToReact
   * @param {object} options options to apply the reaction
   * @param {number} [options.maxDepth=10] max depth of the recursion
   * @returns {Object} The returned object has two properties:
   * - trees: the tree of reactions
   * - products: reactions trees grouped by product idCode
   */
  function applyReactions(reactants, reactions, options = {}) {
    // Reaction are applied recursively until maximal tree depth is reached (default 10)
    const {
      maxDepth = 10
    } = options;
    const moleculesInfo = new Map();
    const processedMolecules = new Set();
    if (!reactants.length) {
      throw new Error('Can not extract OCL because there is no reactants');
    }
    // get the OCL object from the first reactant
    const OCL = reactants[0].getOCL();
    reactions = appendOCLReaction(reactions, OCL);
    const trees = [];
    // Start the recursion by applying the first level of reactions
    let todoCurrentLevel = applyOneReactantReaction(reactants, reactions, {
      OCL,
      currentDepth: 0,
      moleculesInfo,
      processedMolecules,
      maxDepth,
      trees
    });
    do {
      const nexts = [];
      for (const todo of todoCurrentLevel) {
        nexts.push(todo());
      }
      todoCurrentLevel = nexts.flat();
    } while (todoCurrentLevel.length > 0);
    const products = groupTreesByProducts(trees);
    return {
      trees,
      products
    };
  }

  /**
   * @description Append the OCL reaction to the reaction object
   * @param {Array} reactions array of reactions objects with rxnCode and label
   * @param {Object} OCL OCL object
   * @returns {Array} array of reactions objects with rxnCode, label and oclReaction (a decoded version of rxnCode reaction)
   */
  function appendOCLReaction(reactions, OCL) {
    reactions = JSON.parse(JSON.stringify(reactions)).filter(reaction => reaction.rxnCode);
    for (const reaction of reactions) {
      reaction.oclReaction = OCL.ReactionEncoder.decode(reaction.rxnCode);
    }
    return reactions;
  }

  exports.FULL_HOSE_CODE = FULL_HOSE_CODE;
  exports.HOSE_CODE_CUT_C_SP3_SP3 = HOSE_CODE_CUT_C_SP3_SP3;
  exports.MoleculesDB = MoleculesDB;
  exports.addDiastereotopicMissingChirality = addDiastereotopicMissingChirality;
  exports.applyReactions = applyReactions;
  exports.combineSmiles = combineSmiles;
  exports.fragmentAcyclicSingleBonds = fragmentAcyclicSingleBonds;
  exports.getAtomFeatures = getAtomFeatures;
  exports.getAtoms = getAtoms;
  exports.getAtomsInfo = getAtomsInfo;
  exports.getCamelCase = getCamelCase;
  exports.getConnectivityMatrix = getConnectivityMatrix;
  exports.getDiastereotopicAtomIDs = getDiastereotopicAtomIDs;
  exports.getDiastereotopicAtomIDsAndH = getDiastereotopicAtomIDsAndH;
  exports.getDiastereotopicAtomIDsFromMolfile = getDiastereotopicAtomIDsFromMolfile;
  exports.getGroupedDiastereotopicAtomIDs = getGroupedDiastereotopicAtomIDs;
  exports.getHoseCodesAndDiastereotopicIDs = getHoseCodesAndDiastereotopicIDs;
  exports.getHoseCodesForAtom = getHoseCodesForAtom;
  exports.getHoseCodesForAtoms = getHoseCodesForAtoms;
  exports.getHoseCodesForPath = getHoseCodesForPath;
  exports.getHoseCodesFromDiastereotopicID = getHoseCodesFromDiastereotopicID;
  exports.getMF = getMF;
  exports.getPathAndTorsion = getPathAndTorsion;
  exports.getPathsInfo = getPathsInfo;
  exports.getShortestPaths = getShortestPaths;
  exports.groupDiastereotopicAtomIDs = groupDiastereotopicAtomIDs;
  exports.isCsp3 = isCsp3;
  exports.makeRacemic = makeRacemic;
  exports.nbCHO = nbCHO;
  exports.nbCN = nbCN;
  exports.nbCOOH = nbCOOH;
  exports.nbLabileH = nbLabileH;
  exports.nbNH2 = nbNH2;
  exports.nbOH = nbOH;
  exports.parseDwar = parseDwar;
  exports.tagAtom = tagAtom;
  exports.toDiastereotopicSVG = toDiastereotopicSVG;
  exports.toVisualizerMolfile = toVisualizerMolfile;

  Object.defineProperty(exports, '__esModule', { value: true });

}));
//# sourceMappingURL=openchemlib-utils.js.map
