/**
 * spectra-processor
 * @version v2.10.2
 * @link https://github.com/cheminfo/spectra-processor#readme
 * @license MIT
 */
(function (global, factory) {
	typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
	typeof define === 'function' && define.amd ? define(['exports'], factory) :
	(global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.spectraProcessor = {}));
})(this, (function (exports) { 'use strict';

	var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

	function commonjsRequire (path) {
		throw new Error('Could not dynamically require "' + path + '". Please configure the dynamicRequireTargets or/and ignoreDynamicRequires option of @rollup/plugin-commonjs appropriately for this require call to work.');
	}

	const toString = Object.prototype.toString;
	/**
	 * Checks if an object is an instance of an Array (array or typed array).
	 *
	 * @param {any} value - Object to check.
	 * @returns {boolean} True if the object is an array.
	 */

	function isAnyArray(value) {
	  return toString.call(value).endsWith('Array]');
	}

	var medianQuickselect_min = {exports: {}};

	(function (module) {
	  (function () {
	    function a(d) {
	      for (var e = 0, f = d.length - 1, g = void 0, h = void 0, i = void 0, j = c(e, f); !0;) {
	        if (f <= e) return d[j];
	        if (f == e + 1) return d[e] > d[f] && b(d, e, f), d[j];

	        for (g = c(e, f), d[g] > d[f] && b(d, g, f), d[e] > d[f] && b(d, e, f), d[g] > d[e] && b(d, g, e), b(d, g, e + 1), h = e + 1, i = f; !0;) {
	          do h++; while (d[e] > d[h]);

	          do i--; while (d[i] > d[e]);

	          if (i < h) break;
	          b(d, h, i);
	        }

	        b(d, e, i), i <= j && (e = h), i >= j && (f = i - 1);
	      }
	    }

	    var b = function b(d, e, f) {
	      var _ref;

	      return _ref = [d[f], d[e]], d[e] = _ref[0], d[f] = _ref[1], _ref;
	    },
	        c = function c(d, e) {
	      return ~~((d + e) / 2);
	    };

	    module.exports ? module.exports = a : window.median = a;
	  })();
	})(medianQuickselect_min);

	var quickSelectMedian = medianQuickselect_min.exports;

	/**
	 * Calculates the median of an array
	 *
	 * @param input - Array containing values
	 * @returns - median
	 */

	function xMedian(input) {
	  if (!isAnyArray(input)) {
	    throw new TypeError('input must be an array');
	  }

	  if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  } // we need to slice because the order of elements is changed in the quickselect
	  // https://github.com/mad-gooze/median-quickselect


	  return quickSelectMedian(input.slice());
	}

	/**
	 * This function xAdd the first array by the second array or a constant value to each element of the first array
	 *
	 * @param array1 - the first array
	 * @param array2 - the second array or number
	 */

	function xAdd(array1, array2) {
	  let isConstant = false;
	  let constant = 0;

	  if (isAnyArray(array2)) {
	    if (array1.length !== array2.length) {
	      throw new Error('xAdd: size of array1 and array2 must be identical');
	    }
	  } else {
	    isConstant = true;
	    constant = array2;
	  }

	  let array3 = new Float64Array(array1.length);

	  if (isConstant) {
	    for (let i = 0; i < array1.length; i++) {
	      array3[i] = array1[i] + constant;
	    }
	  } else {
	    for (let i = 0; i < array1.length; i++) {
	      array3[i] = array1[i] + array2[i];
	    }
	  }

	  return array3;
	}

	/**
	 * This function
	 * @param output - undefined or a new array
	 * @param length - length of the output array
	 * @returns
	 */

	function getOutputArray(output, length) {
	  if (output !== undefined) {
	    if (!isAnyArray(output)) {
	      throw new TypeError('output option must be an array if specified');
	    }

	    if (output.length !== length) {
	      throw new TypeError('the output array does not have the correct length');
	    }

	    return output;
	  } else {
	    return new Float64Array(length);
	  }
	}

	/**
	 * This function xMultiply the first array by the second array or a constant value to each element of the first array
	 *
	 * @param array1 - first array
	 * @param array2 - second array
	 * @param options - options
	 */

	function xMultiply(array1, array2) {
	  let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
	  let isConstant = false;
	  let constant = 0;

	  if (isAnyArray(array2)) {
	    if (array1.length !== array2.length) {
	      throw new Error('xMultiply: size of array1 and array2 must be identical');
	    }
	  } else {
	    isConstant = true;
	    constant = Number(array2);
	  }

	  let array3 = getOutputArray(options.output, array1.length);

	  if (isConstant) {
	    for (let i = 0; i < array1.length; i++) {
	      array3[i] = array1[i] * constant;
	    }
	  } else {
	    for (let i = 0; i < array1.length; i++) {
	      array3[i] = array1[i] * array2[i];
	    }
	  }

	  return array3;
	}

	/**
	 * Calculating the box plot of the array
	 *
	 * @param array - data
	 */
	function xBoxPlot(array) {
	  array = array.slice(0).sort((a, b) => a - b);

	  if (array.length < 5) {
	    throw Error('xBoxPlot: can not calculate info if array contains less than 3 elements');
	  }

	  let info = {
	    Q1: 0.0,
	    Q2: 0.0,
	    Q3: 0.0,
	    min: array[0],
	    max: array[array.length - 1]
	  };
	  let q1max, q3min;

	  if (array.length % 2 === 1) {
	    // odd
	    let middle = (array.length - 1) / 2;
	    info.Q2 = array[middle];
	    q1max = middle - 1;
	    q3min = middle + 1;
	  } else {
	    // even
	    q3min = array.length / 2;
	    q1max = q3min - 1;
	    info.Q2 = (array[q1max] + array[q3min]) / 2;
	  }

	  if (q1max % 2 === 0) {
	    info.Q1 = array[q1max / 2];
	    info.Q3 = array[(array.length + q3min - 1) / 2];
	  } else {
	    info.Q1 = (array[(q1max + 1) / 2] + array[(q1max - 1) / 2]) / 2;
	    let middleOver = (array.length + q3min) / 2;
	    info.Q3 = (array[middleOver] + array[middleOver - 1]) / 2;
	  }

	  return info;
	}

	/**
	 * Checks if input is of type array
	 *
	 * @param input - input
	 */

	function xCheck(input) {
	  if (!isAnyArray(input)) {
	    throw new TypeError('input must be an array');
	  }

	  if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }
	}

	/**
	 * Calculates the correlation between 2 vectors
	 * https://en.wikipedia.org/wiki/Correlation_and_dependence
	 *
	 * @param A - first array
	 * @param B - sencond array
	 */
	function xCorrelation(A, B) {
	  let n = A.length;
	  let sumA = 0;
	  let sumA2 = 0;
	  let sumB = 0;
	  let sumB2 = 0;
	  let sumAB = 0;

	  for (let i = 0; i < n; i++) {
	    let a = A[i];
	    let b = B[i];
	    sumA += a;
	    sumA2 += a ** 2;
	    sumB += b;
	    sumB2 += b ** 2;
	    sumAB += a * b;
	  }

	  return (n * sumAB - sumA * sumB) / (Math.sqrt(n * sumA2 - sumA ** 2) * Math.sqrt(n * sumB2 - sumB ** 2));
	}

	/**
	 * Returns the closest index of a `target`
	 *
	 * @param array - array of numbers
	 * @param target - target
	 * @returns - closest index
	 */
	function xFindClosestIndex(array, target) {
	  let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
	  const {
	    sorted = true
	  } = options;

	  if (sorted) {
	    let low = 0;
	    let high = array.length - 1;
	    let middle = 0;

	    while (high - low > 1) {
	      middle = low + (high - low >> 1);

	      if (array[middle] < target) {
	        low = middle;
	      } else if (array[middle] > target) {
	        high = middle;
	      } else {
	        return middle;
	      }
	    }

	    if (low < array.length - 1) {
	      if (Math.abs(target - array[low]) < Math.abs(array[low + 1] - target)) {
	        return low;
	      } else {
	        return low + 1;
	      }
	    } else {
	      return low;
	    }
	  } else {
	    let index = 0;
	    let diff = Number.POSITIVE_INFINITY;

	    for (let i = 0; i < array.length; i++) {
	      const currentDiff = Math.abs(array[i] - target);

	      if (currentDiff < diff) {
	        diff = currentDiff;
	        index = i;
	      }
	    }

	    return index;
	  }
	}

	/**
	 * Returns an object with {fromIndex, toIndex} for a specific from / to
	 *
	 * @param x - array of numbers
	 * @param options - Options
	 */

	function xGetFromToIndex(x) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  let {
	    fromIndex,
	    toIndex,
	    from,
	    to
	  } = options;

	  if (fromIndex === undefined) {
	    if (from !== undefined) {
	      fromIndex = xFindClosestIndex(x, from);
	    } else {
	      fromIndex = 0;
	    }
	  }

	  if (toIndex === undefined) {
	    if (to !== undefined) {
	      toIndex = xFindClosestIndex(x, to);
	    } else {
	      toIndex = x.length - 1;
	    }
	  }

	  if (fromIndex < 0) fromIndex = 0;
	  if (toIndex < 0) toIndex = 0;
	  if (fromIndex >= x.length) fromIndex = x.length - 1;
	  if (toIndex >= x.length) toIndex = x.length - 1;
	  if (fromIndex > toIndex) [fromIndex, toIndex] = [toIndex, fromIndex];
	  return {
	    fromIndex,
	    toIndex
	  };
	}

	/**
	 * Computes the maximal value of an array of values
	 *
	 * @param array - array of numbers
	 * @param options - options
	 */

	function xMaxValue(array) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  xCheck(array);
	  const {
	    fromIndex = 0,
	    toIndex = array.length - 1
	  } = options;
	  let maxValue = array[fromIndex];

	  for (let i = fromIndex + 1; i <= toIndex; i++) {
	    if (array[i] > maxValue) {
	      maxValue = array[i];
	    }
	  }

	  return maxValue;
	}

	/**
	 * Computes the minimal value of an array of values
	 *
	 * @param array - array of numbers
	 * @param options - options
	 */

	function xMinValue(array) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  xCheck(array);
	  const {
	    fromIndex = 0,
	    toIndex = array.length - 1
	  } = options;
	  let minValue = array[fromIndex];

	  for (let i = fromIndex + 1; i <= toIndex; i++) {
	    if (array[i] < minValue) {
	      minValue = array[i];
	    }
	  }

	  return minValue;
	}

	/**
	 * Check if the values are separated always by the same difference
	 *
	 * @param array - Monotone growing array of number
	 */
	function xIsEquallySpaced(array) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  if (array.length < 3) return true;
	  const {
	    tolerance = 0.05
	  } = options;
	  let maxDx = 0;
	  let minDx = Number.MAX_SAFE_INTEGER;

	  for (let i = 0; i < array.length - 1; ++i) {
	    let absoluteDifference = array[i + 1] - array[i];

	    if (absoluteDifference < minDx) {
	      minDx = absoluteDifference;
	    }

	    if (absoluteDifference > maxDx) {
	      maxDx = absoluteDifference;
	    }
	  }

	  return (maxDx - minDx) / maxDx < tolerance;
	}

	/**
	 * Returns true if x is monotone
	 *
	 * @param array - array of numbers
	 */
	function xIsMonotone(array) {
	  if (array.length <= 2) {
	    return true;
	  }

	  if (array[0] === array[1]) {
	    // maybe a constant series
	    for (let i = 1; i < array.length - 1; i++) {
	      if (array[i] !== array[i + 1]) return false;
	    }

	    return true;
	  }

	  if (array[0] < array[array.length - 1]) {
	    for (let i = 0; i < array.length - 1; i++) {
	      if (array[i] >= array[i + 1]) return false;
	    }
	  } else {
	    for (let i = 0; i < array.length - 1; i++) {
	      if (array[i] <= array[i + 1]) return false;
	    }
	  }

	  return true;
	}

	/**
	 * Returns true if x is monotone
	 *
	 * @param array - array of numbers
	 */
	function xIsMonotoneIncreasing(array) {
	  if (array.length < 2) {
	    return true;
	  }

	  for (let i = 0; i < array.length - 1; i++) {
	    if (array[i] >= array[i + 1]) return false;
	  }

	  return true;
	}

	/**
	 * Computes the mean value of an array of values
	 *
	 * @param array - array of numbers
	 * @param options - options
	 */

	function xMean(array) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  xCheck(array);
	  const {
	    fromIndex = 0,
	    toIndex = array.length - 1
	  } = options;
	  let sumValue = array[fromIndex];

	  for (let i = fromIndex + 1; i <= toIndex; i++) {
	    sumValue += array[i];
	  }

	  return sumValue / (toIndex - fromIndex + 1);
	}

	/**
	 * This function calculates the median absolute deviation (MAD)
	 * https://en.wikipedia.org/wiki/Median_absolute_deviation
	 * @param array
	 */

	function xMedianAbsoluteDeviation(array) {
	  const median = xMedian(array);
	  const averageDeviations = new Float64Array(array.length);

	  for (let i = 0; i < array.length; i++) {
	    averageDeviations[i] = Math.abs(array[i] - median);
	  }

	  return {
	    median,
	    mad: xMedian(averageDeviations)
	  };
	}

	/**
	 * Determine noise level using MAD https://en.wikipedia.org/wiki/Median_absolute_deviation
	 * Constant to convert mad to sd calculated using https://www.wolframalpha.com/input?i=sqrt%282%29+inverse+erf%280.5%29
	 * This assumes a gaussian distribution of the noise
	 * @param array
	 * @returns noise level corresponding to one standard deviation
	 */

	function xNoiseStandardDeviation(array) {
	  const {
	    mad,
	    median
	  } = xMedianAbsoluteDeviation(array);
	  return {
	    sd: mad / 0.6744897501960817,
	    mad,
	    median
	  };
	}

	/**
	 * Calculate the sum of the values
	 *
	 * @param array - Object that contains property x (an ordered increasing array) and y (an array).
	 * @param options - Options.
	 * @returns XSum value on the specified range.
	 */

	function xSum(array) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    fromIndex = 0,
	    toIndex = array.length - 1
	  } = options;
	  xCheck(array);
	  let sumValue = array[fromIndex];

	  for (let i = fromIndex + 1; i <= toIndex; i++) {
	    sumValue += array[i];
	  }

	  return sumValue;
	}

	/** Finds the variance of the data
	 *
	 * @param values - the values of the array
	 * @param options - options
	 * @returns variance
	 */

	function xVariance(values) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	  if (!isAnyArray(values)) {
	    throw new TypeError('input must be an array');
	  }

	  const {
	    unbiased = true,
	    mean = xMean(values)
	  } = options;
	  let sqrError = 0;

	  for (let i = 0; i < values.length; i++) {
	    let x = values[i] - mean;
	    sqrError += x * x;
	  }

	  if (unbiased) {
	    return sqrError / (values.length - 1);
	  } else {
	    return sqrError / values.length;
	  }
	}

	/** Finds the standard deviation for the data at hand
	 *
	 * @param values - values in the data
	 * @param options - options
	 * @returns standard deviation
	 */

	function xStandardDeviation(values) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  return Math.sqrt(xVariance(values, options));
	}

	/**
	 * Pareto scaling, which uses the square root of standard deviation as the scaling factor, circumvents the amplification of noise by retaining a small portion of magnitude information.
	 * Noda, I. (2008). Scaling techniques to enhance two-dimensional correlation spectra. Journal of Molecular Structure, 883, 216-227.
	 * DOI: 10.1016/j.molstruc.2007.12.026
	 *
	 * @param array - array of number
	 */

	function xParetoNormalization(array) {
	  xCheck(array);
	  let result = [];
	  const sqrtSD = Math.sqrt(xStandardDeviation(array));

	  for (let item of array) {
	    result.push(item / sqrtSD);
	  }

	  return result;
	}

	/**
	 * This function xSubtract the first array by the second array or a constant value from each element of the first array
	 *
	 * @param array1 - the array that will be rotated
	 * @param array2 - second array or number
	 * @returns array after subtraction
	 */

	function xSubtract(array1, array2) {
	  let isConstant = false;
	  let constant = 0;

	  if (isAnyArray(array2)) {
	    if (array1.length !== array2.length) {
	      throw new Error('xSubtract: size of array1 and array2 must be identical');
	    }
	  } else {
	    isConstant = true;
	    constant = Number(array2);
	  }

	  let array3 = new Float64Array(array1.length);

	  if (isConstant) {
	    for (let i = 0; i < array1.length; i++) {
	      array3[i] = array1[i] - constant;
	    }
	  } else {
	    for (let i = 0; i < array1.length; i++) {
	      array3[i] = array1[i] - array2[i];
	    }
	  }

	  return array3;
	}

	/**
	 * Throw an error in no an object of x,y arrays
	 *
	 * @param data - array of points {x,y,z}
	 */

	function xyCheck(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    minLength
	  } = options;

	  if (typeof data !== 'object' || !isAnyArray(data.x) || !isAnyArray(data.y)) {
	    throw new Error('Data must be an object of x and y arrays');
	  }

	  if (data.x.length !== data.y.length) {
	    throw new Error('The x and y arrays must have the same length');
	  }

	  if (minLength) {
	    if (data.x.length < minLength) {
	      throw new Error(`data.x must have a length of at least ${minLength}`);
	    }
	  }
	}

	/**
	 * Finds the max y value in a range and return a {x,y} point
	 *
	 * @param data - Object that contains property x (an ordered increasing array) and y (an array)
	 * @param options Options
	 */

	function xyMaxYPoint(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  xyCheck(data, {
	    minLength: 1
	  });
	  const {
	    x,
	    y
	  } = data;

	  if (x.length === 1) {
	    return {
	      x: x[0],
	      y: y[0],
	      index: 0
	    };
	  }

	  const {
	    fromIndex,
	    toIndex
	  } = xGetFromToIndex(x, options);
	  let current = {
	    x: x[fromIndex],
	    y: y[fromIndex],
	    index: fromIndex
	  };

	  for (let i = fromIndex; i <= toIndex; i++) {
	    if (y[i] > current.y) current = {
	      x: x[i],
	      y: y[i],
	      index: i
	    };
	  }

	  return current;
	}

	/**
	 * Filters x,y values to allow strictly growing values in x axis.
	 *
	 * @param data - Object that contains property x (an ordered increasing array) and y (an array).
	 */

	function xyEnsureGrowingX(data) {
	  xyCheck(data);
	  if (xIsMonotoneIncreasing(data.x)) return data;
	  const x = Array.from(data.x);
	  const y = Array.from(data.y);
	  let prevX = Number.NEGATIVE_INFINITY;
	  let currentIndex = 0;

	  for (let index = 0; index < x.length; index++) {
	    if (prevX < x[index]) {
	      if (currentIndex < index) {
	        x[currentIndex] = x[index];
	        y[currentIndex] = y[index];
	      }

	      currentIndex++;
	      prevX = x[index];
	    }
	  }

	  x.length = currentIndex;
	  y.length = currentIndex;
	  return {
	    x,
	    y
	  };
	}

	/**
	 * Calculate integration
	 *
	 * @param data - Object that contains property x (an ordered increasing array) and y (an array)
	 * @param options - Options
	 * @returns - xyIntegration value on the specified range
	 */

	function xyIntegration(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  xyCheck(data, {
	    minLength: 1
	  });
	  const {
	    x,
	    y
	  } = data;
	  if (x.length === 1) return 0;
	  const {
	    fromIndex,
	    toIndex
	  } = xGetFromToIndex(x, options);
	  let currentxyIntegration = 0;

	  for (let i = fromIndex; i < toIndex; i++) {
	    currentxyIntegration += (x[i + 1] - x[i]) * (y[i + 1] + y[i]) / 2;
	  }

	  return currentxyIntegration;
	}

	/**
	 * Finds the min y value in a range and return a {x,y} point
	 *
	 * @param data - Object that contains property x (an ordered increasing array) and y (an array)
	 * @param options - Options
	 */

	function xyMinYPoint(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  xyCheck(data, {
	    minLength: 1
	  });
	  const {
	    x,
	    y
	  } = data;
	  if (x.length === 1) return {
	    x: x[0],
	    y: y[0],
	    index: 0
	  };
	  const {
	    fromIndex,
	    toIndex
	  } = xGetFromToIndex(x, options);
	  let current = {
	    x: x[fromIndex],
	    y: y[fromIndex],
	    index: fromIndex
	  };

	  for (let i = fromIndex; i <= toIndex; i++) {
	    if (y[i] < current.y) current = {
	      x: x[i],
	      y: y[i],
	      index: i
	    };
	  }

	  return current;
	}

	/**
	 * Center mean of columns
	 *
	 * @param matrix - matrix [rows][cols]
	 */
	function matrixCenterZMean(matrix) {
	  const nbRows = matrix.length;
	  const nbColumns = matrix[0].length;
	  const newMatrix = new Array(nbRows);

	  for (let row = 0; row < nbRows; row++) {
	    newMatrix[row] = new Float64Array(nbColumns);
	  }

	  for (let column = 0; column < nbColumns; column++) {
	    let mean = 0;

	    for (let row = 0; row < nbRows; row++) {
	      mean += matrix[row][column];
	    }

	    mean /= nbRows;

	    for (let row = 0; row < nbRows; row++) {
	      newMatrix[row][column] = matrix[row][column] - mean;
	    }
	  }

	  return newMatrix;
	}

	function max$1(input) {
	  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	  if (!isAnyArray(input)) {
	    throw new TypeError('input must be an array');
	  }

	  if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }

	  var _options$fromIndex = options.fromIndex,
	      fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
	      _options$toIndex = options.toIndex,
	      toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;

	  if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
	    throw new Error('fromIndex must be a positive integer smaller than length');
	  }

	  if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
	    throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
	  }

	  var maxValue = input[fromIndex];

	  for (var i = fromIndex + 1; i < toIndex; i++) {
	    if (input[i] > maxValue) maxValue = input[i];
	  }

	  return maxValue;
	}

	function min$1(input) {
	  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	  if (!isAnyArray(input)) {
	    throw new TypeError('input must be an array');
	  }

	  if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }

	  var _options$fromIndex = options.fromIndex,
	      fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
	      _options$toIndex = options.toIndex,
	      toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;

	  if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
	    throw new Error('fromIndex must be a positive integer smaller than length');
	  }

	  if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
	    throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
	  }

	  var minValue = input[fromIndex];

	  for (var i = fromIndex + 1; i < toIndex; i++) {
	    if (input[i] < minValue) minValue = input[i];
	  }

	  return minValue;
	}

	function rescale(input) {
	  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	  if (!isAnyArray(input)) {
	    throw new TypeError('input must be an array');
	  } else if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }

	  var output;

	  if (options.output !== undefined) {
	    if (!isAnyArray(options.output)) {
	      throw new TypeError('output option must be an array if specified');
	    }

	    output = options.output;
	  } else {
	    output = new Array(input.length);
	  }

	  var currentMin = min$1(input);
	  var currentMax = max$1(input);

	  if (currentMin === currentMax) {
	    throw new RangeError('minimum and maximum input values are equal. Cannot rescale a constant array');
	  }

	  var _options$min = options.min,
	      minValue = _options$min === void 0 ? options.autoMinMax ? currentMin : 0 : _options$min,
	      _options$max = options.max,
	      maxValue = _options$max === void 0 ? options.autoMinMax ? currentMax : 1 : _options$max;

	  if (minValue >= maxValue) {
	    throw new RangeError('min option must be smaller than max option');
	  }

	  var factor = (maxValue - minValue) / (currentMax - currentMin);

	  for (var i = 0; i < input.length; i++) {
	    output[i] = (input[i] - currentMin) * factor + minValue;
	  }

	  return output;
	}

	const indent = ' '.repeat(2);
	const indentData = ' '.repeat(4);
	function inspectMatrix() {
	  return inspectMatrixWithOptions(this);
	}
	function inspectMatrixWithOptions(matrix) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    maxRows = 15,
	    maxColumns = 10,
	    maxNumSize = 8
	  } = options;
	  return `${matrix.constructor.name} {
${indent}[
${indentData}${inspectData(matrix, maxRows, maxColumns, maxNumSize)}
${indent}]
${indent}rows: ${matrix.rows}
${indent}columns: ${matrix.columns}
}`;
	}

	function inspectData(matrix, maxRows, maxColumns, maxNumSize) {
	  const {
	    rows,
	    columns
	  } = matrix;
	  const maxI = Math.min(rows, maxRows);
	  const maxJ = Math.min(columns, maxColumns);
	  const result = [];

	  for (let i = 0; i < maxI; i++) {
	    let line = [];

	    for (let j = 0; j < maxJ; j++) {
	      line.push(formatNumber(matrix.get(i, j), maxNumSize));
	    }

	    result.push(`${line.join(' ')}`);
	  }

	  if (maxJ !== columns) {
	    result[result.length - 1] += ` ... ${columns - maxColumns} more columns`;
	  }

	  if (maxI !== rows) {
	    result.push(`... ${rows - maxRows} more rows`);
	  }

	  return result.join(`\n${indentData}`);
	}

	function formatNumber(num, maxNumSize) {
	  const numStr = String(num);

	  if (numStr.length <= maxNumSize) {
	    return numStr.padEnd(maxNumSize, ' ');
	  }

	  const precise = num.toPrecision(maxNumSize - 2);

	  if (precise.length <= maxNumSize) {
	    return precise;
	  }

	  const exponential = num.toExponential(maxNumSize - 2);
	  const eIndex = exponential.indexOf('e');
	  const e = exponential.slice(eIndex);
	  return exponential.slice(0, maxNumSize - e.length) + e;
	}

	function installMathOperations(AbstractMatrix, Matrix) {
	  AbstractMatrix.prototype.add = function add(value) {
	    if (typeof value === 'number') return this.addS(value);
	    return this.addM(value);
	  };

	  AbstractMatrix.prototype.addS = function addS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) + value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.addM = function addM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) + matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.add = function add(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.add(value);
	  };

	  AbstractMatrix.prototype.sub = function sub(value) {
	    if (typeof value === 'number') return this.subS(value);
	    return this.subM(value);
	  };

	  AbstractMatrix.prototype.subS = function subS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) - value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.subM = function subM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) - matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.sub = function sub(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.sub(value);
	  };

	  AbstractMatrix.prototype.subtract = AbstractMatrix.prototype.sub;
	  AbstractMatrix.prototype.subtractS = AbstractMatrix.prototype.subS;
	  AbstractMatrix.prototype.subtractM = AbstractMatrix.prototype.subM;
	  AbstractMatrix.subtract = AbstractMatrix.sub;

	  AbstractMatrix.prototype.mul = function mul(value) {
	    if (typeof value === 'number') return this.mulS(value);
	    return this.mulM(value);
	  };

	  AbstractMatrix.prototype.mulS = function mulS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) * value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.mulM = function mulM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) * matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.mul = function mul(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.mul(value);
	  };

	  AbstractMatrix.prototype.multiply = AbstractMatrix.prototype.mul;
	  AbstractMatrix.prototype.multiplyS = AbstractMatrix.prototype.mulS;
	  AbstractMatrix.prototype.multiplyM = AbstractMatrix.prototype.mulM;
	  AbstractMatrix.multiply = AbstractMatrix.mul;

	  AbstractMatrix.prototype.div = function div(value) {
	    if (typeof value === 'number') return this.divS(value);
	    return this.divM(value);
	  };

	  AbstractMatrix.prototype.divS = function divS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) / value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.divM = function divM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) / matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.div = function div(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.div(value);
	  };

	  AbstractMatrix.prototype.divide = AbstractMatrix.prototype.div;
	  AbstractMatrix.prototype.divideS = AbstractMatrix.prototype.divS;
	  AbstractMatrix.prototype.divideM = AbstractMatrix.prototype.divM;
	  AbstractMatrix.divide = AbstractMatrix.div;

	  AbstractMatrix.prototype.mod = function mod(value) {
	    if (typeof value === 'number') return this.modS(value);
	    return this.modM(value);
	  };

	  AbstractMatrix.prototype.modS = function modS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) % value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.modM = function modM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) % matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.mod = function mod(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.mod(value);
	  };

	  AbstractMatrix.prototype.modulus = AbstractMatrix.prototype.mod;
	  AbstractMatrix.prototype.modulusS = AbstractMatrix.prototype.modS;
	  AbstractMatrix.prototype.modulusM = AbstractMatrix.prototype.modM;
	  AbstractMatrix.modulus = AbstractMatrix.mod;

	  AbstractMatrix.prototype.and = function and(value) {
	    if (typeof value === 'number') return this.andS(value);
	    return this.andM(value);
	  };

	  AbstractMatrix.prototype.andS = function andS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) & value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.andM = function andM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) & matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.and = function and(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.and(value);
	  };

	  AbstractMatrix.prototype.or = function or(value) {
	    if (typeof value === 'number') return this.orS(value);
	    return this.orM(value);
	  };

	  AbstractMatrix.prototype.orS = function orS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) | value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.orM = function orM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) | matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.or = function or(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.or(value);
	  };

	  AbstractMatrix.prototype.xor = function xor(value) {
	    if (typeof value === 'number') return this.xorS(value);
	    return this.xorM(value);
	  };

	  AbstractMatrix.prototype.xorS = function xorS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) ^ value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.xorM = function xorM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) ^ matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.xor = function xor(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.xor(value);
	  };

	  AbstractMatrix.prototype.leftShift = function leftShift(value) {
	    if (typeof value === 'number') return this.leftShiftS(value);
	    return this.leftShiftM(value);
	  };

	  AbstractMatrix.prototype.leftShiftS = function leftShiftS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) << value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.leftShiftM = function leftShiftM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) << matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.leftShift = function leftShift(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.leftShift(value);
	  };

	  AbstractMatrix.prototype.signPropagatingRightShift = function signPropagatingRightShift(value) {
	    if (typeof value === 'number') return this.signPropagatingRightShiftS(value);
	    return this.signPropagatingRightShiftM(value);
	  };

	  AbstractMatrix.prototype.signPropagatingRightShiftS = function signPropagatingRightShiftS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) >> value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.signPropagatingRightShiftM = function signPropagatingRightShiftM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) >> matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.signPropagatingRightShift = function signPropagatingRightShift(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.signPropagatingRightShift(value);
	  };

	  AbstractMatrix.prototype.rightShift = function rightShift(value) {
	    if (typeof value === 'number') return this.rightShiftS(value);
	    return this.rightShiftM(value);
	  };

	  AbstractMatrix.prototype.rightShiftS = function rightShiftS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) >>> value);
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.rightShiftM = function rightShiftM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) >>> matrix.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.rightShift = function rightShift(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.rightShift(value);
	  };

	  AbstractMatrix.prototype.zeroFillRightShift = AbstractMatrix.prototype.rightShift;
	  AbstractMatrix.prototype.zeroFillRightShiftS = AbstractMatrix.prototype.rightShiftS;
	  AbstractMatrix.prototype.zeroFillRightShiftM = AbstractMatrix.prototype.rightShiftM;
	  AbstractMatrix.zeroFillRightShift = AbstractMatrix.rightShift;

	  AbstractMatrix.prototype.not = function not() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, ~this.get(i, j));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.not = function not(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.not();
	  };

	  AbstractMatrix.prototype.abs = function abs() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.abs(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.abs = function abs(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.abs();
	  };

	  AbstractMatrix.prototype.acos = function acos() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.acos(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.acos = function acos(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.acos();
	  };

	  AbstractMatrix.prototype.acosh = function acosh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.acosh(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.acosh = function acosh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.acosh();
	  };

	  AbstractMatrix.prototype.asin = function asin() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.asin(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.asin = function asin(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.asin();
	  };

	  AbstractMatrix.prototype.asinh = function asinh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.asinh(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.asinh = function asinh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.asinh();
	  };

	  AbstractMatrix.prototype.atan = function atan() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.atan(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.atan = function atan(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.atan();
	  };

	  AbstractMatrix.prototype.atanh = function atanh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.atanh(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.atanh = function atanh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.atanh();
	  };

	  AbstractMatrix.prototype.cbrt = function cbrt() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.cbrt(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.cbrt = function cbrt(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.cbrt();
	  };

	  AbstractMatrix.prototype.ceil = function ceil() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.ceil(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.ceil = function ceil(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.ceil();
	  };

	  AbstractMatrix.prototype.clz32 = function clz32() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.clz32(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.clz32 = function clz32(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.clz32();
	  };

	  AbstractMatrix.prototype.cos = function cos() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.cos(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.cos = function cos(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.cos();
	  };

	  AbstractMatrix.prototype.cosh = function cosh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.cosh(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.cosh = function cosh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.cosh();
	  };

	  AbstractMatrix.prototype.exp = function exp() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.exp(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.exp = function exp(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.exp();
	  };

	  AbstractMatrix.prototype.expm1 = function expm1() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.expm1(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.expm1 = function expm1(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.expm1();
	  };

	  AbstractMatrix.prototype.floor = function floor() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.floor(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.floor = function floor(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.floor();
	  };

	  AbstractMatrix.prototype.fround = function fround() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.fround(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.fround = function fround(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.fround();
	  };

	  AbstractMatrix.prototype.log = function log() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.log(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.log = function log(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.log();
	  };

	  AbstractMatrix.prototype.log1p = function log1p() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.log1p(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.log1p = function log1p(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.log1p();
	  };

	  AbstractMatrix.prototype.log10 = function log10() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.log10(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.log10 = function log10(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.log10();
	  };

	  AbstractMatrix.prototype.log2 = function log2() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.log2(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.log2 = function log2(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.log2();
	  };

	  AbstractMatrix.prototype.round = function round() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.round(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.round = function round(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.round();
	  };

	  AbstractMatrix.prototype.sign = function sign() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.sign(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.sign = function sign(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.sign();
	  };

	  AbstractMatrix.prototype.sin = function sin() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.sin(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.sin = function sin(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.sin();
	  };

	  AbstractMatrix.prototype.sinh = function sinh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.sinh(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.sinh = function sinh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.sinh();
	  };

	  AbstractMatrix.prototype.sqrt = function sqrt() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.sqrt(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.sqrt = function sqrt(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.sqrt();
	  };

	  AbstractMatrix.prototype.tan = function tan() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.tan(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.tan = function tan(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.tan();
	  };

	  AbstractMatrix.prototype.tanh = function tanh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.tanh(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.tanh = function tanh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.tanh();
	  };

	  AbstractMatrix.prototype.trunc = function trunc() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.trunc(this.get(i, j)));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.trunc = function trunc(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.trunc();
	  };

	  AbstractMatrix.pow = function pow(matrix, arg0) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.pow(arg0);
	  };

	  AbstractMatrix.prototype.pow = function pow(value) {
	    if (typeof value === 'number') return this.powS(value);
	    return this.powM(value);
	  };

	  AbstractMatrix.prototype.powS = function powS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.pow(this.get(i, j), value));
	      }
	    }

	    return this;
	  };

	  AbstractMatrix.prototype.powM = function powM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.pow(this.get(i, j), matrix.get(i, j)));
	      }
	    }

	    return this;
	  };
	}

	/**
	 * @private
	 * Check that a row index is not out of bounds
	 * @param {Matrix} matrix
	 * @param {number} index
	 * @param {boolean} [outer]
	 */

	function checkRowIndex(matrix, index, outer) {
	  let max = outer ? matrix.rows : matrix.rows - 1;

	  if (index < 0 || index > max) {
	    throw new RangeError('Row index out of range');
	  }
	}
	/**
	 * @private
	 * Check that a column index is not out of bounds
	 * @param {Matrix} matrix
	 * @param {number} index
	 * @param {boolean} [outer]
	 */

	function checkColumnIndex(matrix, index, outer) {
	  let max = outer ? matrix.columns : matrix.columns - 1;

	  if (index < 0 || index > max) {
	    throw new RangeError('Column index out of range');
	  }
	}
	/**
	 * @private
	 * Check that the provided vector is an array with the right length
	 * @param {Matrix} matrix
	 * @param {Array|Matrix} vector
	 * @return {Array}
	 * @throws {RangeError}
	 */

	function checkRowVector(matrix, vector) {
	  if (vector.to1DArray) {
	    vector = vector.to1DArray();
	  }

	  if (vector.length !== matrix.columns) {
	    throw new RangeError('vector size must be the same as the number of columns');
	  }

	  return vector;
	}
	/**
	 * @private
	 * Check that the provided vector is an array with the right length
	 * @param {Matrix} matrix
	 * @param {Array|Matrix} vector
	 * @return {Array}
	 * @throws {RangeError}
	 */

	function checkColumnVector(matrix, vector) {
	  if (vector.to1DArray) {
	    vector = vector.to1DArray();
	  }

	  if (vector.length !== matrix.rows) {
	    throw new RangeError('vector size must be the same as the number of rows');
	  }

	  return vector;
	}
	function checkRowIndices(matrix, rowIndices) {
	  if (!isAnyArray(rowIndices)) {
	    throw new TypeError('row indices must be an array');
	  }

	  for (let i = 0; i < rowIndices.length; i++) {
	    if (rowIndices[i] < 0 || rowIndices[i] >= matrix.rows) {
	      throw new RangeError('row indices are out of range');
	    }
	  }
	}
	function checkColumnIndices(matrix, columnIndices) {
	  if (!isAnyArray(columnIndices)) {
	    throw new TypeError('column indices must be an array');
	  }

	  for (let i = 0; i < columnIndices.length; i++) {
	    if (columnIndices[i] < 0 || columnIndices[i] >= matrix.columns) {
	      throw new RangeError('column indices are out of range');
	    }
	  }
	}
	function checkRange(matrix, startRow, endRow, startColumn, endColumn) {
	  if (arguments.length !== 5) {
	    throw new RangeError('expected 4 arguments');
	  }

	  checkNumber('startRow', startRow);
	  checkNumber('endRow', endRow);
	  checkNumber('startColumn', startColumn);
	  checkNumber('endColumn', endColumn);

	  if (startRow > endRow || startColumn > endColumn || startRow < 0 || startRow >= matrix.rows || endRow < 0 || endRow >= matrix.rows || startColumn < 0 || startColumn >= matrix.columns || endColumn < 0 || endColumn >= matrix.columns) {
	    throw new RangeError('Submatrix indices are out of range');
	  }
	}
	function newArray(length) {
	  let value = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 0;
	  let array = [];

	  for (let i = 0; i < length; i++) {
	    array.push(value);
	  }

	  return array;
	}

	function checkNumber(name, value) {
	  if (typeof value !== 'number') {
	    throw new TypeError(`${name} must be a number`);
	  }
	}

	function checkNonEmpty(matrix) {
	  if (matrix.isEmpty()) {
	    throw new Error('Empty matrix has no elements to index');
	  }
	}

	function sumByRow(matrix) {
	  let sum = newArray(matrix.rows);

	  for (let i = 0; i < matrix.rows; ++i) {
	    for (let j = 0; j < matrix.columns; ++j) {
	      sum[i] += matrix.get(i, j);
	    }
	  }

	  return sum;
	}
	function sumByColumn(matrix) {
	  let sum = newArray(matrix.columns);

	  for (let i = 0; i < matrix.rows; ++i) {
	    for (let j = 0; j < matrix.columns; ++j) {
	      sum[j] += matrix.get(i, j);
	    }
	  }

	  return sum;
	}
	function sumAll(matrix) {
	  let v = 0;

	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      v += matrix.get(i, j);
	    }
	  }

	  return v;
	}
	function productByRow(matrix) {
	  let sum = newArray(matrix.rows, 1);

	  for (let i = 0; i < matrix.rows; ++i) {
	    for (let j = 0; j < matrix.columns; ++j) {
	      sum[i] *= matrix.get(i, j);
	    }
	  }

	  return sum;
	}
	function productByColumn(matrix) {
	  let sum = newArray(matrix.columns, 1);

	  for (let i = 0; i < matrix.rows; ++i) {
	    for (let j = 0; j < matrix.columns; ++j) {
	      sum[j] *= matrix.get(i, j);
	    }
	  }

	  return sum;
	}
	function productAll(matrix) {
	  let v = 1;

	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      v *= matrix.get(i, j);
	    }
	  }

	  return v;
	}
	function varianceByRow(matrix, unbiased, mean) {
	  const rows = matrix.rows;
	  const cols = matrix.columns;
	  const variance = [];

	  for (let i = 0; i < rows; i++) {
	    let sum1 = 0;
	    let sum2 = 0;
	    let x = 0;

	    for (let j = 0; j < cols; j++) {
	      x = matrix.get(i, j) - mean[i];
	      sum1 += x;
	      sum2 += x * x;
	    }

	    if (unbiased) {
	      variance.push((sum2 - sum1 * sum1 / cols) / (cols - 1));
	    } else {
	      variance.push((sum2 - sum1 * sum1 / cols) / cols);
	    }
	  }

	  return variance;
	}
	function varianceByColumn(matrix, unbiased, mean) {
	  const rows = matrix.rows;
	  const cols = matrix.columns;
	  const variance = [];

	  for (let j = 0; j < cols; j++) {
	    let sum1 = 0;
	    let sum2 = 0;
	    let x = 0;

	    for (let i = 0; i < rows; i++) {
	      x = matrix.get(i, j) - mean[j];
	      sum1 += x;
	      sum2 += x * x;
	    }

	    if (unbiased) {
	      variance.push((sum2 - sum1 * sum1 / rows) / (rows - 1));
	    } else {
	      variance.push((sum2 - sum1 * sum1 / rows) / rows);
	    }
	  }

	  return variance;
	}
	function varianceAll(matrix, unbiased, mean) {
	  const rows = matrix.rows;
	  const cols = matrix.columns;
	  const size = rows * cols;
	  let sum1 = 0;
	  let sum2 = 0;
	  let x = 0;

	  for (let i = 0; i < rows; i++) {
	    for (let j = 0; j < cols; j++) {
	      x = matrix.get(i, j) - mean;
	      sum1 += x;
	      sum2 += x * x;
	    }
	  }

	  if (unbiased) {
	    return (sum2 - sum1 * sum1 / size) / (size - 1);
	  } else {
	    return (sum2 - sum1 * sum1 / size) / size;
	  }
	}
	function centerByRow(matrix, mean) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) - mean[i]);
	    }
	  }
	}
	function centerByColumn(matrix, mean) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) - mean[j]);
	    }
	  }
	}
	function centerAll(matrix, mean) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) - mean);
	    }
	  }
	}
	function getScaleByRow(matrix) {
	  const scale = [];

	  for (let i = 0; i < matrix.rows; i++) {
	    let sum = 0;

	    for (let j = 0; j < matrix.columns; j++) {
	      sum += Math.pow(matrix.get(i, j), 2) / (matrix.columns - 1);
	    }

	    scale.push(Math.sqrt(sum));
	  }

	  return scale;
	}
	function scaleByRow(matrix, scale) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) / scale[i]);
	    }
	  }
	}
	function getScaleByColumn(matrix) {
	  const scale = [];

	  for (let j = 0; j < matrix.columns; j++) {
	    let sum = 0;

	    for (let i = 0; i < matrix.rows; i++) {
	      sum += Math.pow(matrix.get(i, j), 2) / (matrix.rows - 1);
	    }

	    scale.push(Math.sqrt(sum));
	  }

	  return scale;
	}
	function scaleByColumn(matrix, scale) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) / scale[j]);
	    }
	  }
	}
	function getScaleAll(matrix) {
	  const divider = matrix.size - 1;
	  let sum = 0;

	  for (let j = 0; j < matrix.columns; j++) {
	    for (let i = 0; i < matrix.rows; i++) {
	      sum += Math.pow(matrix.get(i, j), 2) / divider;
	    }
	  }

	  return Math.sqrt(sum);
	}
	function scaleAll(matrix, scale) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) / scale);
	    }
	  }
	}

	class AbstractMatrix {
	  static from1DArray(newRows, newColumns, newData) {
	    let length = newRows * newColumns;

	    if (length !== newData.length) {
	      throw new RangeError('data length does not match given dimensions');
	    }

	    let newMatrix = new Matrix(newRows, newColumns);

	    for (let row = 0; row < newRows; row++) {
	      for (let column = 0; column < newColumns; column++) {
	        newMatrix.set(row, column, newData[row * newColumns + column]);
	      }
	    }

	    return newMatrix;
	  }

	  static rowVector(newData) {
	    let vector = new Matrix(1, newData.length);

	    for (let i = 0; i < newData.length; i++) {
	      vector.set(0, i, newData[i]);
	    }

	    return vector;
	  }

	  static columnVector(newData) {
	    let vector = new Matrix(newData.length, 1);

	    for (let i = 0; i < newData.length; i++) {
	      vector.set(i, 0, newData[i]);
	    }

	    return vector;
	  }

	  static zeros(rows, columns) {
	    return new Matrix(rows, columns);
	  }

	  static ones(rows, columns) {
	    return new Matrix(rows, columns).fill(1);
	  }

	  static rand(rows, columns) {
	    let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }

	    const {
	      random = Math.random
	    } = options;
	    let matrix = new Matrix(rows, columns);

	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        matrix.set(i, j, random());
	      }
	    }

	    return matrix;
	  }

	  static randInt(rows, columns) {
	    let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }

	    const {
	      min = 0,
	      max = 1000,
	      random = Math.random
	    } = options;
	    if (!Number.isInteger(min)) throw new TypeError('min must be an integer');
	    if (!Number.isInteger(max)) throw new TypeError('max must be an integer');
	    if (min >= max) throw new RangeError('min must be smaller than max');
	    let interval = max - min;
	    let matrix = new Matrix(rows, columns);

	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        let value = min + Math.round(random() * interval);
	        matrix.set(i, j, value);
	      }
	    }

	    return matrix;
	  }

	  static eye(rows, columns, value) {
	    if (columns === undefined) columns = rows;
	    if (value === undefined) value = 1;
	    let min = Math.min(rows, columns);
	    let matrix = this.zeros(rows, columns);

	    for (let i = 0; i < min; i++) {
	      matrix.set(i, i, value);
	    }

	    return matrix;
	  }

	  static diag(data, rows, columns) {
	    let l = data.length;
	    if (rows === undefined) rows = l;
	    if (columns === undefined) columns = rows;
	    let min = Math.min(l, rows, columns);
	    let matrix = this.zeros(rows, columns);

	    for (let i = 0; i < min; i++) {
	      matrix.set(i, i, data[i]);
	    }

	    return matrix;
	  }

	  static min(matrix1, matrix2) {
	    matrix1 = this.checkMatrix(matrix1);
	    matrix2 = this.checkMatrix(matrix2);
	    let rows = matrix1.rows;
	    let columns = matrix1.columns;
	    let result = new Matrix(rows, columns);

	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        result.set(i, j, Math.min(matrix1.get(i, j), matrix2.get(i, j)));
	      }
	    }

	    return result;
	  }

	  static max(matrix1, matrix2) {
	    matrix1 = this.checkMatrix(matrix1);
	    matrix2 = this.checkMatrix(matrix2);
	    let rows = matrix1.rows;
	    let columns = matrix1.columns;
	    let result = new this(rows, columns);

	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        result.set(i, j, Math.max(matrix1.get(i, j), matrix2.get(i, j)));
	      }
	    }

	    return result;
	  }

	  static checkMatrix(value) {
	    return AbstractMatrix.isMatrix(value) ? value : new Matrix(value);
	  }

	  static isMatrix(value) {
	    return value != null && value.klass === 'Matrix';
	  }

	  get size() {
	    return this.rows * this.columns;
	  }

	  apply(callback) {
	    if (typeof callback !== 'function') {
	      throw new TypeError('callback must be a function');
	    }

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        callback.call(this, i, j);
	      }
	    }

	    return this;
	  }

	  to1DArray() {
	    let array = [];

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        array.push(this.get(i, j));
	      }
	    }

	    return array;
	  }

	  to2DArray() {
	    let copy = [];

	    for (let i = 0; i < this.rows; i++) {
	      copy.push([]);

	      for (let j = 0; j < this.columns; j++) {
	        copy[i].push(this.get(i, j));
	      }
	    }

	    return copy;
	  }

	  toJSON() {
	    return this.to2DArray();
	  }

	  isRowVector() {
	    return this.rows === 1;
	  }

	  isColumnVector() {
	    return this.columns === 1;
	  }

	  isVector() {
	    return this.rows === 1 || this.columns === 1;
	  }

	  isSquare() {
	    return this.rows === this.columns;
	  }

	  isEmpty() {
	    return this.rows === 0 || this.columns === 0;
	  }

	  isSymmetric() {
	    if (this.isSquare()) {
	      for (let i = 0; i < this.rows; i++) {
	        for (let j = 0; j <= i; j++) {
	          if (this.get(i, j) !== this.get(j, i)) {
	            return false;
	          }
	        }
	      }

	      return true;
	    }

	    return false;
	  }

	  isEchelonForm() {
	    let i = 0;
	    let j = 0;
	    let previousColumn = -1;
	    let isEchelonForm = true;
	    let checked = false;

	    while (i < this.rows && isEchelonForm) {
	      j = 0;
	      checked = false;

	      while (j < this.columns && checked === false) {
	        if (this.get(i, j) === 0) {
	          j++;
	        } else if (this.get(i, j) === 1 && j > previousColumn) {
	          checked = true;
	          previousColumn = j;
	        } else {
	          isEchelonForm = false;
	          checked = true;
	        }
	      }

	      i++;
	    }

	    return isEchelonForm;
	  }

	  isReducedEchelonForm() {
	    let i = 0;
	    let j = 0;
	    let previousColumn = -1;
	    let isReducedEchelonForm = true;
	    let checked = false;

	    while (i < this.rows && isReducedEchelonForm) {
	      j = 0;
	      checked = false;

	      while (j < this.columns && checked === false) {
	        if (this.get(i, j) === 0) {
	          j++;
	        } else if (this.get(i, j) === 1 && j > previousColumn) {
	          checked = true;
	          previousColumn = j;
	        } else {
	          isReducedEchelonForm = false;
	          checked = true;
	        }
	      }

	      for (let k = j + 1; k < this.rows; k++) {
	        if (this.get(i, k) !== 0) {
	          isReducedEchelonForm = false;
	        }
	      }

	      i++;
	    }

	    return isReducedEchelonForm;
	  }

	  echelonForm() {
	    let result = this.clone();
	    let h = 0;
	    let k = 0;

	    while (h < result.rows && k < result.columns) {
	      let iMax = h;

	      for (let i = h; i < result.rows; i++) {
	        if (result.get(i, k) > result.get(iMax, k)) {
	          iMax = i;
	        }
	      }

	      if (result.get(iMax, k) === 0) {
	        k++;
	      } else {
	        result.swapRows(h, iMax);
	        let tmp = result.get(h, k);

	        for (let j = k; j < result.columns; j++) {
	          result.set(h, j, result.get(h, j) / tmp);
	        }

	        for (let i = h + 1; i < result.rows; i++) {
	          let factor = result.get(i, k) / result.get(h, k);
	          result.set(i, k, 0);

	          for (let j = k + 1; j < result.columns; j++) {
	            result.set(i, j, result.get(i, j) - result.get(h, j) * factor);
	          }
	        }

	        h++;
	        k++;
	      }
	    }

	    return result;
	  }

	  reducedEchelonForm() {
	    let result = this.echelonForm();
	    let m = result.columns;
	    let n = result.rows;
	    let h = n - 1;

	    while (h >= 0) {
	      if (result.maxRow(h) === 0) {
	        h--;
	      } else {
	        let p = 0;
	        let pivot = false;

	        while (p < n && pivot === false) {
	          if (result.get(h, p) === 1) {
	            pivot = true;
	          } else {
	            p++;
	          }
	        }

	        for (let i = 0; i < h; i++) {
	          let factor = result.get(i, p);

	          for (let j = p; j < m; j++) {
	            let tmp = result.get(i, j) - factor * result.get(h, j);
	            result.set(i, j, tmp);
	          }
	        }

	        h--;
	      }
	    }

	    return result;
	  }

	  set() {
	    throw new Error('set method is unimplemented');
	  }

	  get() {
	    throw new Error('get method is unimplemented');
	  }

	  repeat() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }

	    const {
	      rows = 1,
	      columns = 1
	    } = options;

	    if (!Number.isInteger(rows) || rows <= 0) {
	      throw new TypeError('rows must be a positive integer');
	    }

	    if (!Number.isInteger(columns) || columns <= 0) {
	      throw new TypeError('columns must be a positive integer');
	    }

	    let matrix = new Matrix(this.rows * rows, this.columns * columns);

	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        matrix.setSubMatrix(this, this.rows * i, this.columns * j);
	      }
	    }

	    return matrix;
	  }

	  fill(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, value);
	      }
	    }

	    return this;
	  }

	  neg() {
	    return this.mulS(-1);
	  }

	  getRow(index) {
	    checkRowIndex(this, index);
	    let row = [];

	    for (let i = 0; i < this.columns; i++) {
	      row.push(this.get(index, i));
	    }

	    return row;
	  }

	  getRowVector(index) {
	    return Matrix.rowVector(this.getRow(index));
	  }

	  setRow(index, array) {
	    checkRowIndex(this, index);
	    array = checkRowVector(this, array);

	    for (let i = 0; i < this.columns; i++) {
	      this.set(index, i, array[i]);
	    }

	    return this;
	  }

	  swapRows(row1, row2) {
	    checkRowIndex(this, row1);
	    checkRowIndex(this, row2);

	    for (let i = 0; i < this.columns; i++) {
	      let temp = this.get(row1, i);
	      this.set(row1, i, this.get(row2, i));
	      this.set(row2, i, temp);
	    }

	    return this;
	  }

	  getColumn(index) {
	    checkColumnIndex(this, index);
	    let column = [];

	    for (let i = 0; i < this.rows; i++) {
	      column.push(this.get(i, index));
	    }

	    return column;
	  }

	  getColumnVector(index) {
	    return Matrix.columnVector(this.getColumn(index));
	  }

	  setColumn(index, array) {
	    checkColumnIndex(this, index);
	    array = checkColumnVector(this, array);

	    for (let i = 0; i < this.rows; i++) {
	      this.set(i, index, array[i]);
	    }

	    return this;
	  }

	  swapColumns(column1, column2) {
	    checkColumnIndex(this, column1);
	    checkColumnIndex(this, column2);

	    for (let i = 0; i < this.rows; i++) {
	      let temp = this.get(i, column1);
	      this.set(i, column1, this.get(i, column2));
	      this.set(i, column2, temp);
	    }

	    return this;
	  }

	  addRowVector(vector) {
	    vector = checkRowVector(this, vector);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) + vector[j]);
	      }
	    }

	    return this;
	  }

	  subRowVector(vector) {
	    vector = checkRowVector(this, vector);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) - vector[j]);
	      }
	    }

	    return this;
	  }

	  mulRowVector(vector) {
	    vector = checkRowVector(this, vector);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) * vector[j]);
	      }
	    }

	    return this;
	  }

	  divRowVector(vector) {
	    vector = checkRowVector(this, vector);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) / vector[j]);
	      }
	    }

	    return this;
	  }

	  addColumnVector(vector) {
	    vector = checkColumnVector(this, vector);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) + vector[i]);
	      }
	    }

	    return this;
	  }

	  subColumnVector(vector) {
	    vector = checkColumnVector(this, vector);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) - vector[i]);
	      }
	    }

	    return this;
	  }

	  mulColumnVector(vector) {
	    vector = checkColumnVector(this, vector);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) * vector[i]);
	      }
	    }

	    return this;
	  }

	  divColumnVector(vector) {
	    vector = checkColumnVector(this, vector);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) / vector[i]);
	      }
	    }

	    return this;
	  }

	  mulRow(index, value) {
	    checkRowIndex(this, index);

	    for (let i = 0; i < this.columns; i++) {
	      this.set(index, i, this.get(index, i) * value);
	    }

	    return this;
	  }

	  mulColumn(index, value) {
	    checkColumnIndex(this, index);

	    for (let i = 0; i < this.rows; i++) {
	      this.set(i, index, this.get(i, index) * value);
	    }

	    return this;
	  }

	  max() {
	    if (this.isEmpty()) {
	      return NaN;
	    }

	    let v = this.get(0, 0);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        if (this.get(i, j) > v) {
	          v = this.get(i, j);
	        }
	      }
	    }

	    return v;
	  }

	  maxIndex() {
	    checkNonEmpty(this);
	    let v = this.get(0, 0);
	    let idx = [0, 0];

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        if (this.get(i, j) > v) {
	          v = this.get(i, j);
	          idx[0] = i;
	          idx[1] = j;
	        }
	      }
	    }

	    return idx;
	  }

	  min() {
	    if (this.isEmpty()) {
	      return NaN;
	    }

	    let v = this.get(0, 0);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        if (this.get(i, j) < v) {
	          v = this.get(i, j);
	        }
	      }
	    }

	    return v;
	  }

	  minIndex() {
	    checkNonEmpty(this);
	    let v = this.get(0, 0);
	    let idx = [0, 0];

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        if (this.get(i, j) < v) {
	          v = this.get(i, j);
	          idx[0] = i;
	          idx[1] = j;
	        }
	      }
	    }

	    return idx;
	  }

	  maxRow(row) {
	    checkRowIndex(this, row);

	    if (this.isEmpty()) {
	      return NaN;
	    }

	    let v = this.get(row, 0);

	    for (let i = 1; i < this.columns; i++) {
	      if (this.get(row, i) > v) {
	        v = this.get(row, i);
	      }
	    }

	    return v;
	  }

	  maxRowIndex(row) {
	    checkRowIndex(this, row);
	    checkNonEmpty(this);
	    let v = this.get(row, 0);
	    let idx = [row, 0];

	    for (let i = 1; i < this.columns; i++) {
	      if (this.get(row, i) > v) {
	        v = this.get(row, i);
	        idx[1] = i;
	      }
	    }

	    return idx;
	  }

	  minRow(row) {
	    checkRowIndex(this, row);

	    if (this.isEmpty()) {
	      return NaN;
	    }

	    let v = this.get(row, 0);

	    for (let i = 1; i < this.columns; i++) {
	      if (this.get(row, i) < v) {
	        v = this.get(row, i);
	      }
	    }

	    return v;
	  }

	  minRowIndex(row) {
	    checkRowIndex(this, row);
	    checkNonEmpty(this);
	    let v = this.get(row, 0);
	    let idx = [row, 0];

	    for (let i = 1; i < this.columns; i++) {
	      if (this.get(row, i) < v) {
	        v = this.get(row, i);
	        idx[1] = i;
	      }
	    }

	    return idx;
	  }

	  maxColumn(column) {
	    checkColumnIndex(this, column);

	    if (this.isEmpty()) {
	      return NaN;
	    }

	    let v = this.get(0, column);

	    for (let i = 1; i < this.rows; i++) {
	      if (this.get(i, column) > v) {
	        v = this.get(i, column);
	      }
	    }

	    return v;
	  }

	  maxColumnIndex(column) {
	    checkColumnIndex(this, column);
	    checkNonEmpty(this);
	    let v = this.get(0, column);
	    let idx = [0, column];

	    for (let i = 1; i < this.rows; i++) {
	      if (this.get(i, column) > v) {
	        v = this.get(i, column);
	        idx[0] = i;
	      }
	    }

	    return idx;
	  }

	  minColumn(column) {
	    checkColumnIndex(this, column);

	    if (this.isEmpty()) {
	      return NaN;
	    }

	    let v = this.get(0, column);

	    for (let i = 1; i < this.rows; i++) {
	      if (this.get(i, column) < v) {
	        v = this.get(i, column);
	      }
	    }

	    return v;
	  }

	  minColumnIndex(column) {
	    checkColumnIndex(this, column);
	    checkNonEmpty(this);
	    let v = this.get(0, column);
	    let idx = [0, column];

	    for (let i = 1; i < this.rows; i++) {
	      if (this.get(i, column) < v) {
	        v = this.get(i, column);
	        idx[0] = i;
	      }
	    }

	    return idx;
	  }

	  diag() {
	    let min = Math.min(this.rows, this.columns);
	    let diag = [];

	    for (let i = 0; i < min; i++) {
	      diag.push(this.get(i, i));
	    }

	    return diag;
	  }

	  norm() {
	    let type = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 'frobenius';
	    let result = 0;

	    if (type === 'max') {
	      return this.max();
	    } else if (type === 'frobenius') {
	      for (let i = 0; i < this.rows; i++) {
	        for (let j = 0; j < this.columns; j++) {
	          result = result + this.get(i, j) * this.get(i, j);
	        }
	      }

	      return Math.sqrt(result);
	    } else {
	      throw new RangeError(`unknown norm type: ${type}`);
	    }
	  }

	  cumulativeSum() {
	    let sum = 0;

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        sum += this.get(i, j);
	        this.set(i, j, sum);
	      }
	    }

	    return this;
	  }

	  dot(vector2) {
	    if (AbstractMatrix.isMatrix(vector2)) vector2 = vector2.to1DArray();
	    let vector1 = this.to1DArray();

	    if (vector1.length !== vector2.length) {
	      throw new RangeError('vectors do not have the same size');
	    }

	    let dot = 0;

	    for (let i = 0; i < vector1.length; i++) {
	      dot += vector1[i] * vector2[i];
	    }

	    return dot;
	  }

	  mmul(other) {
	    other = Matrix.checkMatrix(other);
	    let m = this.rows;
	    let n = this.columns;
	    let p = other.columns;
	    let result = new Matrix(m, p);
	    let Bcolj = new Float64Array(n);

	    for (let j = 0; j < p; j++) {
	      for (let k = 0; k < n; k++) {
	        Bcolj[k] = other.get(k, j);
	      }

	      for (let i = 0; i < m; i++) {
	        let s = 0;

	        for (let k = 0; k < n; k++) {
	          s += this.get(i, k) * Bcolj[k];
	        }

	        result.set(i, j, s);
	      }
	    }

	    return result;
	  }

	  strassen2x2(other) {
	    other = Matrix.checkMatrix(other);
	    let result = new Matrix(2, 2);
	    const a11 = this.get(0, 0);
	    const b11 = other.get(0, 0);
	    const a12 = this.get(0, 1);
	    const b12 = other.get(0, 1);
	    const a21 = this.get(1, 0);
	    const b21 = other.get(1, 0);
	    const a22 = this.get(1, 1);
	    const b22 = other.get(1, 1); // Compute intermediate values.

	    const m1 = (a11 + a22) * (b11 + b22);
	    const m2 = (a21 + a22) * b11;
	    const m3 = a11 * (b12 - b22);
	    const m4 = a22 * (b21 - b11);
	    const m5 = (a11 + a12) * b22;
	    const m6 = (a21 - a11) * (b11 + b12);
	    const m7 = (a12 - a22) * (b21 + b22); // Combine intermediate values into the output.

	    const c00 = m1 + m4 - m5 + m7;
	    const c01 = m3 + m5;
	    const c10 = m2 + m4;
	    const c11 = m1 - m2 + m3 + m6;
	    result.set(0, 0, c00);
	    result.set(0, 1, c01);
	    result.set(1, 0, c10);
	    result.set(1, 1, c11);
	    return result;
	  }

	  strassen3x3(other) {
	    other = Matrix.checkMatrix(other);
	    let result = new Matrix(3, 3);
	    const a00 = this.get(0, 0);
	    const a01 = this.get(0, 1);
	    const a02 = this.get(0, 2);
	    const a10 = this.get(1, 0);
	    const a11 = this.get(1, 1);
	    const a12 = this.get(1, 2);
	    const a20 = this.get(2, 0);
	    const a21 = this.get(2, 1);
	    const a22 = this.get(2, 2);
	    const b00 = other.get(0, 0);
	    const b01 = other.get(0, 1);
	    const b02 = other.get(0, 2);
	    const b10 = other.get(1, 0);
	    const b11 = other.get(1, 1);
	    const b12 = other.get(1, 2);
	    const b20 = other.get(2, 0);
	    const b21 = other.get(2, 1);
	    const b22 = other.get(2, 2);
	    const m1 = (a00 + a01 + a02 - a10 - a11 - a21 - a22) * b11;
	    const m2 = (a00 - a10) * (-b01 + b11);
	    const m3 = a11 * (-b00 + b01 + b10 - b11 - b12 - b20 + b22);
	    const m4 = (-a00 + a10 + a11) * (b00 - b01 + b11);
	    const m5 = (a10 + a11) * (-b00 + b01);
	    const m6 = a00 * b00;
	    const m7 = (-a00 + a20 + a21) * (b00 - b02 + b12);
	    const m8 = (-a00 + a20) * (b02 - b12);
	    const m9 = (a20 + a21) * (-b00 + b02);
	    const m10 = (a00 + a01 + a02 - a11 - a12 - a20 - a21) * b12;
	    const m11 = a21 * (-b00 + b02 + b10 - b11 - b12 - b20 + b21);
	    const m12 = (-a02 + a21 + a22) * (b11 + b20 - b21);
	    const m13 = (a02 - a22) * (b11 - b21);
	    const m14 = a02 * b20;
	    const m15 = (a21 + a22) * (-b20 + b21);
	    const m16 = (-a02 + a11 + a12) * (b12 + b20 - b22);
	    const m17 = (a02 - a12) * (b12 - b22);
	    const m18 = (a11 + a12) * (-b20 + b22);
	    const m19 = a01 * b10;
	    const m20 = a12 * b21;
	    const m21 = a10 * b02;
	    const m22 = a20 * b01;
	    const m23 = a22 * b22;
	    const c00 = m6 + m14 + m19;
	    const c01 = m1 + m4 + m5 + m6 + m12 + m14 + m15;
	    const c02 = m6 + m7 + m9 + m10 + m14 + m16 + m18;
	    const c10 = m2 + m3 + m4 + m6 + m14 + m16 + m17;
	    const c11 = m2 + m4 + m5 + m6 + m20;
	    const c12 = m14 + m16 + m17 + m18 + m21;
	    const c20 = m6 + m7 + m8 + m11 + m12 + m13 + m14;
	    const c21 = m12 + m13 + m14 + m15 + m22;
	    const c22 = m6 + m7 + m8 + m9 + m23;
	    result.set(0, 0, c00);
	    result.set(0, 1, c01);
	    result.set(0, 2, c02);
	    result.set(1, 0, c10);
	    result.set(1, 1, c11);
	    result.set(1, 2, c12);
	    result.set(2, 0, c20);
	    result.set(2, 1, c21);
	    result.set(2, 2, c22);
	    return result;
	  }

	  mmulStrassen(y) {
	    y = Matrix.checkMatrix(y);
	    let x = this.clone();
	    let r1 = x.rows;
	    let c1 = x.columns;
	    let r2 = y.rows;
	    let c2 = y.columns;

	    if (c1 !== r2) {
	      // eslint-disable-next-line no-console
	      console.warn(`Multiplying ${r1} x ${c1} and ${r2} x ${c2} matrix: dimensions do not match.`);
	    } // Put a matrix into the top left of a matrix of zeros.
	    // `rows` and `cols` are the dimensions of the output matrix.


	    function embed(mat, rows, cols) {
	      let r = mat.rows;
	      let c = mat.columns;

	      if (r === rows && c === cols) {
	        return mat;
	      } else {
	        let resultat = AbstractMatrix.zeros(rows, cols);
	        resultat = resultat.setSubMatrix(mat, 0, 0);
	        return resultat;
	      }
	    } // Make sure both matrices are the same size.
	    // This is exclusively for simplicity:
	    // this algorithm can be implemented with matrices of different sizes.


	    let r = Math.max(r1, r2);
	    let c = Math.max(c1, c2);
	    x = embed(x, r, c);
	    y = embed(y, r, c); // Our recursive multiplication function.

	    function blockMult(a, b, rows, cols) {
	      // For small matrices, resort to naive multiplication.
	      if (rows <= 512 || cols <= 512) {
	        return a.mmul(b); // a is equivalent to this
	      } // Apply dynamic padding.


	      if (rows % 2 === 1 && cols % 2 === 1) {
	        a = embed(a, rows + 1, cols + 1);
	        b = embed(b, rows + 1, cols + 1);
	      } else if (rows % 2 === 1) {
	        a = embed(a, rows + 1, cols);
	        b = embed(b, rows + 1, cols);
	      } else if (cols % 2 === 1) {
	        a = embed(a, rows, cols + 1);
	        b = embed(b, rows, cols + 1);
	      }

	      let halfRows = parseInt(a.rows / 2, 10);
	      let halfCols = parseInt(a.columns / 2, 10); // Subdivide input matrices.

	      let a11 = a.subMatrix(0, halfRows - 1, 0, halfCols - 1);
	      let b11 = b.subMatrix(0, halfRows - 1, 0, halfCols - 1);
	      let a12 = a.subMatrix(0, halfRows - 1, halfCols, a.columns - 1);
	      let b12 = b.subMatrix(0, halfRows - 1, halfCols, b.columns - 1);
	      let a21 = a.subMatrix(halfRows, a.rows - 1, 0, halfCols - 1);
	      let b21 = b.subMatrix(halfRows, b.rows - 1, 0, halfCols - 1);
	      let a22 = a.subMatrix(halfRows, a.rows - 1, halfCols, a.columns - 1);
	      let b22 = b.subMatrix(halfRows, b.rows - 1, halfCols, b.columns - 1); // Compute intermediate values.

	      let m1 = blockMult(AbstractMatrix.add(a11, a22), AbstractMatrix.add(b11, b22), halfRows, halfCols);
	      let m2 = blockMult(AbstractMatrix.add(a21, a22), b11, halfRows, halfCols);
	      let m3 = blockMult(a11, AbstractMatrix.sub(b12, b22), halfRows, halfCols);
	      let m4 = blockMult(a22, AbstractMatrix.sub(b21, b11), halfRows, halfCols);
	      let m5 = blockMult(AbstractMatrix.add(a11, a12), b22, halfRows, halfCols);
	      let m6 = blockMult(AbstractMatrix.sub(a21, a11), AbstractMatrix.add(b11, b12), halfRows, halfCols);
	      let m7 = blockMult(AbstractMatrix.sub(a12, a22), AbstractMatrix.add(b21, b22), halfRows, halfCols); // Combine intermediate values into the output.

	      let c11 = AbstractMatrix.add(m1, m4);
	      c11.sub(m5);
	      c11.add(m7);
	      let c12 = AbstractMatrix.add(m3, m5);
	      let c21 = AbstractMatrix.add(m2, m4);
	      let c22 = AbstractMatrix.sub(m1, m2);
	      c22.add(m3);
	      c22.add(m6); // Crop output to the desired size (undo dynamic padding).

	      let resultat = AbstractMatrix.zeros(2 * c11.rows, 2 * c11.columns);
	      resultat = resultat.setSubMatrix(c11, 0, 0);
	      resultat = resultat.setSubMatrix(c12, c11.rows, 0);
	      resultat = resultat.setSubMatrix(c21, 0, c11.columns);
	      resultat = resultat.setSubMatrix(c22, c11.rows, c11.columns);
	      return resultat.subMatrix(0, rows - 1, 0, cols - 1);
	    }

	    return blockMult(x, y, r, c);
	  }

	  scaleRows() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }

	    const {
	      min = 0,
	      max = 1
	    } = options;
	    if (!Number.isFinite(min)) throw new TypeError('min must be a number');
	    if (!Number.isFinite(max)) throw new TypeError('max must be a number');
	    if (min >= max) throw new RangeError('min must be smaller than max');
	    let newMatrix = new Matrix(this.rows, this.columns);

	    for (let i = 0; i < this.rows; i++) {
	      const row = this.getRow(i);

	      if (row.length > 0) {
	        rescale(row, {
	          min,
	          max,
	          output: row
	        });
	      }

	      newMatrix.setRow(i, row);
	    }

	    return newMatrix;
	  }

	  scaleColumns() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }

	    const {
	      min = 0,
	      max = 1
	    } = options;
	    if (!Number.isFinite(min)) throw new TypeError('min must be a number');
	    if (!Number.isFinite(max)) throw new TypeError('max must be a number');
	    if (min >= max) throw new RangeError('min must be smaller than max');
	    let newMatrix = new Matrix(this.rows, this.columns);

	    for (let i = 0; i < this.columns; i++) {
	      const column = this.getColumn(i);

	      if (column.length) {
	        rescale(column, {
	          min: min,
	          max: max,
	          output: column
	        });
	      }

	      newMatrix.setColumn(i, column);
	    }

	    return newMatrix;
	  }

	  flipRows() {
	    const middle = Math.ceil(this.columns / 2);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < middle; j++) {
	        let first = this.get(i, j);
	        let last = this.get(i, this.columns - 1 - j);
	        this.set(i, j, last);
	        this.set(i, this.columns - 1 - j, first);
	      }
	    }

	    return this;
	  }

	  flipColumns() {
	    const middle = Math.ceil(this.rows / 2);

	    for (let j = 0; j < this.columns; j++) {
	      for (let i = 0; i < middle; i++) {
	        let first = this.get(i, j);
	        let last = this.get(this.rows - 1 - i, j);
	        this.set(i, j, last);
	        this.set(this.rows - 1 - i, j, first);
	      }
	    }

	    return this;
	  }

	  kroneckerProduct(other) {
	    other = Matrix.checkMatrix(other);
	    let m = this.rows;
	    let n = this.columns;
	    let p = other.rows;
	    let q = other.columns;
	    let result = new Matrix(m * p, n * q);

	    for (let i = 0; i < m; i++) {
	      for (let j = 0; j < n; j++) {
	        for (let k = 0; k < p; k++) {
	          for (let l = 0; l < q; l++) {
	            result.set(p * i + k, q * j + l, this.get(i, j) * other.get(k, l));
	          }
	        }
	      }
	    }

	    return result;
	  }

	  kroneckerSum(other) {
	    other = Matrix.checkMatrix(other);

	    if (!this.isSquare() || !other.isSquare()) {
	      throw new Error('Kronecker Sum needs two Square Matrices');
	    }

	    let m = this.rows;
	    let n = other.rows;
	    let AxI = this.kroneckerProduct(Matrix.eye(n, n));
	    let IxB = Matrix.eye(m, m).kroneckerProduct(other);
	    return AxI.add(IxB);
	  }

	  transpose() {
	    let result = new Matrix(this.columns, this.rows);

	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        result.set(j, i, this.get(i, j));
	      }
	    }

	    return result;
	  }

	  sortRows() {
	    let compareFunction = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : compareNumbers;

	    for (let i = 0; i < this.rows; i++) {
	      this.setRow(i, this.getRow(i).sort(compareFunction));
	    }

	    return this;
	  }

	  sortColumns() {
	    let compareFunction = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : compareNumbers;

	    for (let i = 0; i < this.columns; i++) {
	      this.setColumn(i, this.getColumn(i).sort(compareFunction));
	    }

	    return this;
	  }

	  subMatrix(startRow, endRow, startColumn, endColumn) {
	    checkRange(this, startRow, endRow, startColumn, endColumn);
	    let newMatrix = new Matrix(endRow - startRow + 1, endColumn - startColumn + 1);

	    for (let i = startRow; i <= endRow; i++) {
	      for (let j = startColumn; j <= endColumn; j++) {
	        newMatrix.set(i - startRow, j - startColumn, this.get(i, j));
	      }
	    }

	    return newMatrix;
	  }

	  subMatrixRow(indices, startColumn, endColumn) {
	    if (startColumn === undefined) startColumn = 0;
	    if (endColumn === undefined) endColumn = this.columns - 1;

	    if (startColumn > endColumn || startColumn < 0 || startColumn >= this.columns || endColumn < 0 || endColumn >= this.columns) {
	      throw new RangeError('Argument out of range');
	    }

	    let newMatrix = new Matrix(indices.length, endColumn - startColumn + 1);

	    for (let i = 0; i < indices.length; i++) {
	      for (let j = startColumn; j <= endColumn; j++) {
	        if (indices[i] < 0 || indices[i] >= this.rows) {
	          throw new RangeError(`Row index out of range: ${indices[i]}`);
	        }

	        newMatrix.set(i, j - startColumn, this.get(indices[i], j));
	      }
	    }

	    return newMatrix;
	  }

	  subMatrixColumn(indices, startRow, endRow) {
	    if (startRow === undefined) startRow = 0;
	    if (endRow === undefined) endRow = this.rows - 1;

	    if (startRow > endRow || startRow < 0 || startRow >= this.rows || endRow < 0 || endRow >= this.rows) {
	      throw new RangeError('Argument out of range');
	    }

	    let newMatrix = new Matrix(endRow - startRow + 1, indices.length);

	    for (let i = 0; i < indices.length; i++) {
	      for (let j = startRow; j <= endRow; j++) {
	        if (indices[i] < 0 || indices[i] >= this.columns) {
	          throw new RangeError(`Column index out of range: ${indices[i]}`);
	        }

	        newMatrix.set(j - startRow, i, this.get(j, indices[i]));
	      }
	    }

	    return newMatrix;
	  }

	  setSubMatrix(matrix, startRow, startColumn) {
	    matrix = Matrix.checkMatrix(matrix);

	    if (matrix.isEmpty()) {
	      return this;
	    }

	    let endRow = startRow + matrix.rows - 1;
	    let endColumn = startColumn + matrix.columns - 1;
	    checkRange(this, startRow, endRow, startColumn, endColumn);

	    for (let i = 0; i < matrix.rows; i++) {
	      for (let j = 0; j < matrix.columns; j++) {
	        this.set(startRow + i, startColumn + j, matrix.get(i, j));
	      }
	    }

	    return this;
	  }

	  selection(rowIndices, columnIndices) {
	    checkRowIndices(this, rowIndices);
	    checkColumnIndices(this, columnIndices);
	    let newMatrix = new Matrix(rowIndices.length, columnIndices.length);

	    for (let i = 0; i < rowIndices.length; i++) {
	      let rowIndex = rowIndices[i];

	      for (let j = 0; j < columnIndices.length; j++) {
	        let columnIndex = columnIndices[j];
	        newMatrix.set(i, j, this.get(rowIndex, columnIndex));
	      }
	    }

	    return newMatrix;
	  }

	  trace() {
	    let min = Math.min(this.rows, this.columns);
	    let trace = 0;

	    for (let i = 0; i < min; i++) {
	      trace += this.get(i, i);
	    }

	    return trace;
	  }

	  clone() {
	    let newMatrix = new Matrix(this.rows, this.columns);

	    for (let row = 0; row < this.rows; row++) {
	      for (let column = 0; column < this.columns; column++) {
	        newMatrix.set(row, column, this.get(row, column));
	      }
	    }

	    return newMatrix;
	  }

	  sum(by) {
	    switch (by) {
	      case 'row':
	        return sumByRow(this);

	      case 'column':
	        return sumByColumn(this);

	      case undefined:
	        return sumAll(this);

	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }

	  product(by) {
	    switch (by) {
	      case 'row':
	        return productByRow(this);

	      case 'column':
	        return productByColumn(this);

	      case undefined:
	        return productAll(this);

	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }

	  mean(by) {
	    const sum = this.sum(by);

	    switch (by) {
	      case 'row':
	        {
	          for (let i = 0; i < this.rows; i++) {
	            sum[i] /= this.columns;
	          }

	          return sum;
	        }

	      case 'column':
	        {
	          for (let i = 0; i < this.columns; i++) {
	            sum[i] /= this.rows;
	          }

	          return sum;
	        }

	      case undefined:
	        return sum / this.size;

	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }

	  variance(by) {
	    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	    if (typeof by === 'object') {
	      options = by;
	      by = undefined;
	    }

	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }

	    const {
	      unbiased = true,
	      mean = this.mean(by)
	    } = options;

	    if (typeof unbiased !== 'boolean') {
	      throw new TypeError('unbiased must be a boolean');
	    }

	    switch (by) {
	      case 'row':
	        {
	          if (!isAnyArray(mean)) {
	            throw new TypeError('mean must be an array');
	          }

	          return varianceByRow(this, unbiased, mean);
	        }

	      case 'column':
	        {
	          if (!isAnyArray(mean)) {
	            throw new TypeError('mean must be an array');
	          }

	          return varianceByColumn(this, unbiased, mean);
	        }

	      case undefined:
	        {
	          if (typeof mean !== 'number') {
	            throw new TypeError('mean must be a number');
	          }

	          return varianceAll(this, unbiased, mean);
	        }

	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }

	  standardDeviation(by, options) {
	    if (typeof by === 'object') {
	      options = by;
	      by = undefined;
	    }

	    const variance = this.variance(by, options);

	    if (by === undefined) {
	      return Math.sqrt(variance);
	    } else {
	      for (let i = 0; i < variance.length; i++) {
	        variance[i] = Math.sqrt(variance[i]);
	      }

	      return variance;
	    }
	  }

	  center(by) {
	    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	    if (typeof by === 'object') {
	      options = by;
	      by = undefined;
	    }

	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }

	    const {
	      center = this.mean(by)
	    } = options;

	    switch (by) {
	      case 'row':
	        {
	          if (!isAnyArray(center)) {
	            throw new TypeError('center must be an array');
	          }

	          centerByRow(this, center);
	          return this;
	        }

	      case 'column':
	        {
	          if (!isAnyArray(center)) {
	            throw new TypeError('center must be an array');
	          }

	          centerByColumn(this, center);
	          return this;
	        }

	      case undefined:
	        {
	          if (typeof center !== 'number') {
	            throw new TypeError('center must be a number');
	          }

	          centerAll(this, center);
	          return this;
	        }

	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }

	  scale(by) {
	    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	    if (typeof by === 'object') {
	      options = by;
	      by = undefined;
	    }

	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }

	    let scale = options.scale;

	    switch (by) {
	      case 'row':
	        {
	          if (scale === undefined) {
	            scale = getScaleByRow(this);
	          } else if (!isAnyArray(scale)) {
	            throw new TypeError('scale must be an array');
	          }

	          scaleByRow(this, scale);
	          return this;
	        }

	      case 'column':
	        {
	          if (scale === undefined) {
	            scale = getScaleByColumn(this);
	          } else if (!isAnyArray(scale)) {
	            throw new TypeError('scale must be an array');
	          }

	          scaleByColumn(this, scale);
	          return this;
	        }

	      case undefined:
	        {
	          if (scale === undefined) {
	            scale = getScaleAll(this);
	          } else if (typeof scale !== 'number') {
	            throw new TypeError('scale must be a number');
	          }

	          scaleAll(this, scale);
	          return this;
	        }

	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }

	  toString(options) {
	    return inspectMatrixWithOptions(this, options);
	  }

	}
	AbstractMatrix.prototype.klass = 'Matrix';

	if (typeof Symbol !== 'undefined') {
	  AbstractMatrix.prototype[Symbol.for('nodejs.util.inspect.custom')] = inspectMatrix;
	}

	function compareNumbers(a, b) {
	  return a - b;
	} // Synonyms


	AbstractMatrix.random = AbstractMatrix.rand;
	AbstractMatrix.randomInt = AbstractMatrix.randInt;
	AbstractMatrix.diagonal = AbstractMatrix.diag;
	AbstractMatrix.prototype.diagonal = AbstractMatrix.prototype.diag;
	AbstractMatrix.identity = AbstractMatrix.eye;
	AbstractMatrix.prototype.negate = AbstractMatrix.prototype.neg;
	AbstractMatrix.prototype.tensorProduct = AbstractMatrix.prototype.kroneckerProduct;
	class Matrix extends AbstractMatrix {
	  constructor(nRows, nColumns) {
	    super();

	    if (Matrix.isMatrix(nRows)) {
	      // eslint-disable-next-line no-constructor-return
	      return nRows.clone();
	    } else if (Number.isInteger(nRows) && nRows >= 0) {
	      // Create an empty matrix
	      this.data = [];

	      if (Number.isInteger(nColumns) && nColumns >= 0) {
	        for (let i = 0; i < nRows; i++) {
	          this.data.push(new Float64Array(nColumns));
	        }
	      } else {
	        throw new TypeError('nColumns must be a positive integer');
	      }
	    } else if (isAnyArray(nRows)) {
	      // Copy the values from the 2D array
	      const arrayData = nRows;
	      nRows = arrayData.length;
	      nColumns = nRows ? arrayData[0].length : 0;

	      if (typeof nColumns !== 'number') {
	        throw new TypeError('Data must be a 2D array with at least one element');
	      }

	      this.data = [];

	      for (let i = 0; i < nRows; i++) {
	        if (arrayData[i].length !== nColumns) {
	          throw new RangeError('Inconsistent array dimensions');
	        }

	        this.data.push(Float64Array.from(arrayData[i]));
	      }
	    } else {
	      throw new TypeError('First argument must be a positive number or an array');
	    }

	    this.rows = nRows;
	    this.columns = nColumns;
	  }

	  set(rowIndex, columnIndex, value) {
	    this.data[rowIndex][columnIndex] = value;
	    return this;
	  }

	  get(rowIndex, columnIndex) {
	    return this.data[rowIndex][columnIndex];
	  }

	  removeRow(index) {
	    checkRowIndex(this, index);
	    this.data.splice(index, 1);
	    this.rows -= 1;
	    return this;
	  }

	  addRow(index, array) {
	    if (array === undefined) {
	      array = index;
	      index = this.rows;
	    }

	    checkRowIndex(this, index, true);
	    array = Float64Array.from(checkRowVector(this, array));
	    this.data.splice(index, 0, array);
	    this.rows += 1;
	    return this;
	  }

	  removeColumn(index) {
	    checkColumnIndex(this, index);

	    for (let i = 0; i < this.rows; i++) {
	      const newRow = new Float64Array(this.columns - 1);

	      for (let j = 0; j < index; j++) {
	        newRow[j] = this.data[i][j];
	      }

	      for (let j = index + 1; j < this.columns; j++) {
	        newRow[j - 1] = this.data[i][j];
	      }

	      this.data[i] = newRow;
	    }

	    this.columns -= 1;
	    return this;
	  }

	  addColumn(index, array) {
	    if (typeof array === 'undefined') {
	      array = index;
	      index = this.columns;
	    }

	    checkColumnIndex(this, index, true);
	    array = checkColumnVector(this, array);

	    for (let i = 0; i < this.rows; i++) {
	      const newRow = new Float64Array(this.columns + 1);
	      let j = 0;

	      for (; j < index; j++) {
	        newRow[j] = this.data[i][j];
	      }

	      newRow[j++] = array[i];

	      for (; j < this.columns + 1; j++) {
	        newRow[j] = this.data[i][j - 1];
	      }

	      this.data[i] = newRow;
	    }

	    this.columns += 1;
	    return this;
	  }

	}
	installMathOperations(AbstractMatrix, Matrix);

	/**
	 * Performs a Probabilistic quotient normalization (PQN) over the dataset to account dilution based in median spectrum.
	 * Dieterle, F., Ross, A., Schlotterbeck, G., & Senn, H. (2006). Probabilistic quotient normalization as robust method to account for dilution of complex biological mixtures. Application in 1H NMR metabonomics. Analytical chemistry, 78(13), 4281-4290.
	 * DOI: 10.1021/ac051632c
	 *
	 * @param matrix - matrix [rows][cols].
	 * @param options - options
	 * @returns - {data: Normalized dataset, medianOfQuotients: The median of quotients of each variables}
	 */

	function matrixPQN(matrix) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    max = 100
	  } = options;
	  let matrixB = new Matrix(matrix);

	  for (let i = 0; i < matrixB.rows; i++) {
	    const normalizationFactor = matrixB.getRowVector(i).norm('frobenius') / max;
	    const row = matrixB.getRowVector(i).div(normalizationFactor);
	    matrixB.setRow(i, row);
	  }

	  let referenceSpectrum = [];

	  for (let i = 0; i < matrixB.columns; i++) {
	    const currentVariable = matrixB.getColumn(i);
	    referenceSpectrum.push(xMedian(currentVariable));
	  }

	  let medianOfQuotients = [];

	  for (let i = 0; i < matrixB.columns; i++) {
	    let quotients = matrixB.getColumnVector(i).div(referenceSpectrum[i]);
	    medianOfQuotients.push(xMedian(quotients.getColumn(0)));
	  }

	  for (let i = 0; i < matrixB.rows; i++) {
	    matrixB.mulRow(i, 1 / medianOfQuotients[i]);
	  }

	  return {
	    data: matrixB.to2DArray(),
	    medianOfQuotients: medianOfQuotients
	  };
	}

	/**
	 * Rescale columns
	 *
	 * @param matrix - matrix [rows][cols].
	 * @param options - Options
	 */
	function matrixZRescale(matrix) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    min = 0,
	    max = 1
	  } = options;
	  const nbRows = matrix.length;
	  const nbColumns = matrix[0].length;
	  const newMatrix = new Array(nbRows);

	  for (let row = 0; row < nbRows; row++) {
	    newMatrix[row] = new Float64Array(nbColumns);
	  }

	  for (let column = 0; column < nbColumns; column++) {
	    let currentMin = matrix[0][column];
	    let currentMax = matrix[0][column];

	    for (let row = 1; row < nbRows; row++) {
	      if (matrix[row][column] < currentMin) currentMin = matrix[row][column];
	      if (matrix[row][column] > currentMax) currentMax = matrix[row][column];
	    }

	    const factor = (max - min) / (currentMax - currentMin);

	    for (let row = 0; row < nbRows; row++) {
	      newMatrix[row][column] = (matrix[row][column] - currentMin) * factor + min;
	    }
	  }

	  return newMatrix;
	}

	var chroma$1 = {exports: {}};

	/**
	 * chroma.js - JavaScript library for color conversions
	 *
	 * Copyright (c) 2011-2019, Gregor Aisch
	 * All rights reserved.
	 *
	 * Redistribution and use in source and binary forms, with or without
	 * modification, are permitted provided that the following conditions are met:
	 *
	 * 1. Redistributions of source code must retain the above copyright notice, this
	 * list of conditions and the following disclaimer.
	 *
	 * 2. Redistributions in binary form must reproduce the above copyright notice,
	 * this list of conditions and the following disclaimer in the documentation
	 * and/or other materials provided with the distribution.
	 *
	 * 3. The name Gregor Aisch may not be used to endorse or promote products
	 * derived from this software without specific prior written permission.
	 *
	 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
	 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
	 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
	 * DISCLAIMED. IN NO EVENT SHALL GREGOR AISCH OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
	 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
	 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
	 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
	 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
	 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
	 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
	 *
	 * -------------------------------------------------------
	 *
	 * chroma.js includes colors from colorbrewer2.org, which are released under
	 * the following license:
	 *
	 * Copyright (c) 2002 Cynthia Brewer, Mark Harrower,
	 * and The Pennsylvania State University.
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 * http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 *
	 * ------------------------------------------------------
	 *
	 * Named colors are taken from X11 Color Names.
	 * http://www.w3.org/TR/css3-color/#svg-color
	 *
	 * @preserve
	 */

	(function (module, exports) {
	  (function (global, factory) {
	    module.exports = factory() ;
	  })(commonjsGlobal, function () {

	    var limit$2 = function (x, min, max) {
	      if (min === void 0) min = 0;
	      if (max === void 0) max = 1;
	      return x < min ? min : x > max ? max : x;
	    };

	    var limit$1 = limit$2;

	    var clip_rgb$3 = function (rgb) {
	      rgb._clipped = false;
	      rgb._unclipped = rgb.slice(0);

	      for (var i = 0; i <= 3; i++) {
	        if (i < 3) {
	          if (rgb[i] < 0 || rgb[i] > 255) {
	            rgb._clipped = true;
	          }

	          rgb[i] = limit$1(rgb[i], 0, 255);
	        } else if (i === 3) {
	          rgb[i] = limit$1(rgb[i], 0, 1);
	        }
	      }

	      return rgb;
	    }; // ported from jQuery's $.type


	    var classToType = {};

	    for (var i$1 = 0, list$1 = ['Boolean', 'Number', 'String', 'Function', 'Array', 'Date', 'RegExp', 'Undefined', 'Null']; i$1 < list$1.length; i$1 += 1) {
	      var name = list$1[i$1];
	      classToType["[object " + name + "]"] = name.toLowerCase();
	    }

	    var type$p = function (obj) {
	      return classToType[Object.prototype.toString.call(obj)] || "object";
	    };

	    var type$o = type$p;

	    var unpack$B = function (args, keyOrder) {
	      if (keyOrder === void 0) keyOrder = null; // if called with more than 3 arguments, we return the arguments

	      if (args.length >= 3) {
	        return Array.prototype.slice.call(args);
	      } // with less than 3 args we check if first arg is object
	      // and use the keyOrder string to extract and sort properties


	      if (type$o(args[0]) == 'object' && keyOrder) {
	        return keyOrder.split('').filter(function (k) {
	          return args[0][k] !== undefined;
	        }).map(function (k) {
	          return args[0][k];
	        });
	      } // otherwise we just return the first argument
	      // (which we suppose is an array of args)


	      return args[0];
	    };

	    var type$n = type$p;

	    var last$4 = function (args) {
	      if (args.length < 2) {
	        return null;
	      }

	      var l = args.length - 1;

	      if (type$n(args[l]) == 'string') {
	        return args[l].toLowerCase();
	      }

	      return null;
	    };

	    var PI$2 = Math.PI;
	    var utils = {
	      clip_rgb: clip_rgb$3,
	      limit: limit$2,
	      type: type$p,
	      unpack: unpack$B,
	      last: last$4,
	      PI: PI$2,
	      TWOPI: PI$2 * 2,
	      PITHIRD: PI$2 / 3,
	      DEG2RAD: PI$2 / 180,
	      RAD2DEG: 180 / PI$2
	    };
	    var input$h = {
	      format: {},
	      autodetect: []
	    };
	    var last$3 = utils.last;
	    var clip_rgb$2 = utils.clip_rgb;
	    var type$m = utils.type;
	    var _input = input$h;

	    var Color$D = function Color() {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      var me = this;

	      if (type$m(args[0]) === 'object' && args[0].constructor && args[0].constructor === this.constructor) {
	        // the argument is already a Color instance
	        return args[0];
	      } // last argument could be the mode


	      var mode = last$3(args);
	      var autodetect = false;

	      if (!mode) {
	        autodetect = true;

	        if (!_input.sorted) {
	          _input.autodetect = _input.autodetect.sort(function (a, b) {
	            return b.p - a.p;
	          });
	          _input.sorted = true;
	        } // auto-detect format


	        for (var i = 0, list = _input.autodetect; i < list.length; i += 1) {
	          var chk = list[i];
	          mode = chk.test.apply(chk, args);

	          if (mode) {
	            break;
	          }
	        }
	      }

	      if (_input.format[mode]) {
	        var rgb = _input.format[mode].apply(null, autodetect ? args : args.slice(0, -1));

	        me._rgb = clip_rgb$2(rgb);
	      } else {
	        throw new Error('unknown format: ' + args);
	      } // add alpha channel


	      if (me._rgb.length === 3) {
	        me._rgb.push(1);
	      }
	    };

	    Color$D.prototype.toString = function toString() {
	      if (type$m(this.hex) == 'function') {
	        return this.hex();
	      }

	      return "[" + this._rgb.join(',') + "]";
	    };

	    var Color_1 = Color$D;

	    var chroma$k = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      return new (Function.prototype.bind.apply(chroma$k.Color, [null].concat(args)))();
	    };

	    chroma$k.Color = Color_1;
	    chroma$k.version = '2.4.2';
	    var chroma_1 = chroma$k;
	    var unpack$A = utils.unpack;
	    var max$2 = Math.max;

	    var rgb2cmyk$1 = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      var ref = unpack$A(args, 'rgb');
	      var r = ref[0];
	      var g = ref[1];
	      var b = ref[2];
	      r = r / 255;
	      g = g / 255;
	      b = b / 255;
	      var k = 1 - max$2(r, max$2(g, b));
	      var f = k < 1 ? 1 / (1 - k) : 0;
	      var c = (1 - r - k) * f;
	      var m = (1 - g - k) * f;
	      var y = (1 - b - k) * f;
	      return [c, m, y, k];
	    };

	    var rgb2cmyk_1 = rgb2cmyk$1;
	    var unpack$z = utils.unpack;

	    var cmyk2rgb = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      args = unpack$z(args, 'cmyk');
	      var c = args[0];
	      var m = args[1];
	      var y = args[2];
	      var k = args[3];
	      var alpha = args.length > 4 ? args[4] : 1;

	      if (k === 1) {
	        return [0, 0, 0, alpha];
	      }

	      return [c >= 1 ? 0 : 255 * (1 - c) * (1 - k), // r
	      m >= 1 ? 0 : 255 * (1 - m) * (1 - k), // g
	      y >= 1 ? 0 : 255 * (1 - y) * (1 - k), // b
	      alpha];
	    };

	    var cmyk2rgb_1 = cmyk2rgb;
	    var chroma$j = chroma_1;
	    var Color$C = Color_1;
	    var input$g = input$h;
	    var unpack$y = utils.unpack;
	    var type$l = utils.type;
	    var rgb2cmyk = rgb2cmyk_1;

	    Color$C.prototype.cmyk = function () {
	      return rgb2cmyk(this._rgb);
	    };

	    chroma$j.cmyk = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      return new (Function.prototype.bind.apply(Color$C, [null].concat(args, ['cmyk'])))();
	    };

	    input$g.format.cmyk = cmyk2rgb_1;
	    input$g.autodetect.push({
	      p: 2,
	      test: function () {
	        var args = [],
	            len = arguments.length;

	        while (len--) args[len] = arguments[len];

	        args = unpack$y(args, 'cmyk');

	        if (type$l(args) === 'array' && args.length === 4) {
	          return 'cmyk';
	        }
	      }
	    });
	    var unpack$x = utils.unpack;
	    var last$2 = utils.last;

	    var rnd = function (a) {
	      return Math.round(a * 100) / 100;
	    };
	    /*
	     * supported arguments:
	     * - hsl2css(h,s,l)
	     * - hsl2css(h,s,l,a)
	     * - hsl2css([h,s,l], mode)
	     * - hsl2css([h,s,l,a], mode)
	     * - hsl2css({h,s,l,a}, mode)
	     */


	    var hsl2css$1 = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      var hsla = unpack$x(args, 'hsla');
	      var mode = last$2(args) || 'lsa';
	      hsla[0] = rnd(hsla[0] || 0);
	      hsla[1] = rnd(hsla[1] * 100) + '%';
	      hsla[2] = rnd(hsla[2] * 100) + '%';

	      if (mode === 'hsla' || hsla.length > 3 && hsla[3] < 1) {
	        hsla[3] = hsla.length > 3 ? hsla[3] : 1;
	        mode = 'hsla';
	      } else {
	        hsla.length = 3;
	      }

	      return mode + "(" + hsla.join(',') + ")";
	    };

	    var hsl2css_1 = hsl2css$1;
	    var unpack$w = utils.unpack;
	    /*
	     * supported arguments:
	     * - rgb2hsl(r,g,b)
	     * - rgb2hsl(r,g,b,a)
	     * - rgb2hsl([r,g,b])
	     * - rgb2hsl([r,g,b,a])
	     * - rgb2hsl({r,g,b,a})
	     */

	    var rgb2hsl$3 = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      args = unpack$w(args, 'rgba');
	      var r = args[0];
	      var g = args[1];
	      var b = args[2];
	      r /= 255;
	      g /= 255;
	      b /= 255;
	      var min = Math.min(r, g, b);
	      var max = Math.max(r, g, b);
	      var l = (max + min) / 2;
	      var s, h;

	      if (max === min) {
	        s = 0;
	        h = Number.NaN;
	      } else {
	        s = l < 0.5 ? (max - min) / (max + min) : (max - min) / (2 - max - min);
	      }

	      if (r == max) {
	        h = (g - b) / (max - min);
	      } else if (g == max) {
	        h = 2 + (b - r) / (max - min);
	      } else if (b == max) {
	        h = 4 + (r - g) / (max - min);
	      }

	      h *= 60;

	      if (h < 0) {
	        h += 360;
	      }

	      if (args.length > 3 && args[3] !== undefined) {
	        return [h, s, l, args[3]];
	      }

	      return [h, s, l];
	    };

	    var rgb2hsl_1 = rgb2hsl$3;
	    var unpack$v = utils.unpack;
	    var last$1 = utils.last;
	    var hsl2css = hsl2css_1;
	    var rgb2hsl$2 = rgb2hsl_1;
	    var round$6 = Math.round;
	    /*
	     * supported arguments:
	     * - rgb2css(r,g,b)
	     * - rgb2css(r,g,b,a)
	     * - rgb2css([r,g,b], mode)
	     * - rgb2css([r,g,b,a], mode)
	     * - rgb2css({r,g,b,a}, mode)
	     */

	    var rgb2css$1 = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      var rgba = unpack$v(args, 'rgba');
	      var mode = last$1(args) || 'rgb';

	      if (mode.substr(0, 3) == 'hsl') {
	        return hsl2css(rgb2hsl$2(rgba), mode);
	      }

	      rgba[0] = round$6(rgba[0]);
	      rgba[1] = round$6(rgba[1]);
	      rgba[2] = round$6(rgba[2]);

	      if (mode === 'rgba' || rgba.length > 3 && rgba[3] < 1) {
	        rgba[3] = rgba.length > 3 ? rgba[3] : 1;
	        mode = 'rgba';
	      }

	      return mode + "(" + rgba.slice(0, mode === 'rgb' ? 3 : 4).join(',') + ")";
	    };

	    var rgb2css_1 = rgb2css$1;
	    var unpack$u = utils.unpack;
	    var round$5 = Math.round;

	    var hsl2rgb$1 = function () {
	      var assign;
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      args = unpack$u(args, 'hsl');
	      var h = args[0];
	      var s = args[1];
	      var l = args[2];
	      var r, g, b;

	      if (s === 0) {
	        r = g = b = l * 255;
	      } else {
	        var t3 = [0, 0, 0];
	        var c = [0, 0, 0];
	        var t2 = l < 0.5 ? l * (1 + s) : l + s - l * s;
	        var t1 = 2 * l - t2;
	        var h_ = h / 360;
	        t3[0] = h_ + 1 / 3;
	        t3[1] = h_;
	        t3[2] = h_ - 1 / 3;

	        for (var i = 0; i < 3; i++) {
	          if (t3[i] < 0) {
	            t3[i] += 1;
	          }

	          if (t3[i] > 1) {
	            t3[i] -= 1;
	          }

	          if (6 * t3[i] < 1) {
	            c[i] = t1 + (t2 - t1) * 6 * t3[i];
	          } else if (2 * t3[i] < 1) {
	            c[i] = t2;
	          } else if (3 * t3[i] < 2) {
	            c[i] = t1 + (t2 - t1) * (2 / 3 - t3[i]) * 6;
	          } else {
	            c[i] = t1;
	          }
	        }

	        assign = [round$5(c[0] * 255), round$5(c[1] * 255), round$5(c[2] * 255)], r = assign[0], g = assign[1], b = assign[2];
	      }

	      if (args.length > 3) {
	        // keep alpha channel
	        return [r, g, b, args[3]];
	      }

	      return [r, g, b, 1];
	    };

	    var hsl2rgb_1 = hsl2rgb$1;
	    var hsl2rgb = hsl2rgb_1;
	    var input$f = input$h;
	    var RE_RGB = /^rgb\(\s*(-?\d+),\s*(-?\d+)\s*,\s*(-?\d+)\s*\)$/;
	    var RE_RGBA = /^rgba\(\s*(-?\d+),\s*(-?\d+)\s*,\s*(-?\d+)\s*,\s*([01]|[01]?\.\d+)\)$/;
	    var RE_RGB_PCT = /^rgb\(\s*(-?\d+(?:\.\d+)?)%,\s*(-?\d+(?:\.\d+)?)%\s*,\s*(-?\d+(?:\.\d+)?)%\s*\)$/;
	    var RE_RGBA_PCT = /^rgba\(\s*(-?\d+(?:\.\d+)?)%,\s*(-?\d+(?:\.\d+)?)%\s*,\s*(-?\d+(?:\.\d+)?)%\s*,\s*([01]|[01]?\.\d+)\)$/;
	    var RE_HSL = /^hsl\(\s*(-?\d+(?:\.\d+)?),\s*(-?\d+(?:\.\d+)?)%\s*,\s*(-?\d+(?:\.\d+)?)%\s*\)$/;
	    var RE_HSLA = /^hsla\(\s*(-?\d+(?:\.\d+)?),\s*(-?\d+(?:\.\d+)?)%\s*,\s*(-?\d+(?:\.\d+)?)%\s*,\s*([01]|[01]?\.\d+)\)$/;
	    var round$4 = Math.round;

	    var css2rgb$1 = function (css) {
	      css = css.toLowerCase().trim();
	      var m;

	      if (input$f.format.named) {
	        try {
	          return input$f.format.named(css);
	        } catch (e) {// eslint-disable-next-line
	        }
	      } // rgb(250,20,0)


	      if (m = css.match(RE_RGB)) {
	        var rgb = m.slice(1, 4);

	        for (var i = 0; i < 3; i++) {
	          rgb[i] = +rgb[i];
	        }

	        rgb[3] = 1; // default alpha

	        return rgb;
	      } // rgba(250,20,0,0.4)


	      if (m = css.match(RE_RGBA)) {
	        var rgb$1 = m.slice(1, 5);

	        for (var i$1 = 0; i$1 < 4; i$1++) {
	          rgb$1[i$1] = +rgb$1[i$1];
	        }

	        return rgb$1;
	      } // rgb(100%,0%,0%)


	      if (m = css.match(RE_RGB_PCT)) {
	        var rgb$2 = m.slice(1, 4);

	        for (var i$2 = 0; i$2 < 3; i$2++) {
	          rgb$2[i$2] = round$4(rgb$2[i$2] * 2.55);
	        }

	        rgb$2[3] = 1; // default alpha

	        return rgb$2;
	      } // rgba(100%,0%,0%,0.4)


	      if (m = css.match(RE_RGBA_PCT)) {
	        var rgb$3 = m.slice(1, 5);

	        for (var i$3 = 0; i$3 < 3; i$3++) {
	          rgb$3[i$3] = round$4(rgb$3[i$3] * 2.55);
	        }

	        rgb$3[3] = +rgb$3[3];
	        return rgb$3;
	      } // hsl(0,100%,50%)


	      if (m = css.match(RE_HSL)) {
	        var hsl = m.slice(1, 4);
	        hsl[1] *= 0.01;
	        hsl[2] *= 0.01;
	        var rgb$4 = hsl2rgb(hsl);
	        rgb$4[3] = 1;
	        return rgb$4;
	      } // hsla(0,100%,50%,0.5)


	      if (m = css.match(RE_HSLA)) {
	        var hsl$1 = m.slice(1, 4);
	        hsl$1[1] *= 0.01;
	        hsl$1[2] *= 0.01;
	        var rgb$5 = hsl2rgb(hsl$1);
	        rgb$5[3] = +m[4]; // default alpha = 1

	        return rgb$5;
	      }
	    };

	    css2rgb$1.test = function (s) {
	      return RE_RGB.test(s) || RE_RGBA.test(s) || RE_RGB_PCT.test(s) || RE_RGBA_PCT.test(s) || RE_HSL.test(s) || RE_HSLA.test(s);
	    };

	    var css2rgb_1 = css2rgb$1;
	    var chroma$i = chroma_1;
	    var Color$B = Color_1;
	    var input$e = input$h;
	    var type$k = utils.type;
	    var rgb2css = rgb2css_1;
	    var css2rgb = css2rgb_1;

	    Color$B.prototype.css = function (mode) {
	      return rgb2css(this._rgb, mode);
	    };

	    chroma$i.css = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      return new (Function.prototype.bind.apply(Color$B, [null].concat(args, ['css'])))();
	    };

	    input$e.format.css = css2rgb;
	    input$e.autodetect.push({
	      p: 5,
	      test: function (h) {
	        var rest = [],
	            len = arguments.length - 1;

	        while (len-- > 0) rest[len] = arguments[len + 1];

	        if (!rest.length && type$k(h) === 'string' && css2rgb.test(h)) {
	          return 'css';
	        }
	      }
	    });
	    var Color$A = Color_1;
	    var chroma$h = chroma_1;
	    var input$d = input$h;
	    var unpack$t = utils.unpack;

	    input$d.format.gl = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      var rgb = unpack$t(args, 'rgba');
	      rgb[0] *= 255;
	      rgb[1] *= 255;
	      rgb[2] *= 255;
	      return rgb;
	    };

	    chroma$h.gl = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      return new (Function.prototype.bind.apply(Color$A, [null].concat(args, ['gl'])))();
	    };

	    Color$A.prototype.gl = function () {
	      var rgb = this._rgb;
	      return [rgb[0] / 255, rgb[1] / 255, rgb[2] / 255, rgb[3]];
	    };

	    var unpack$s = utils.unpack;

	    var rgb2hcg$1 = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      var ref = unpack$s(args, 'rgb');
	      var r = ref[0];
	      var g = ref[1];
	      var b = ref[2];
	      var min = Math.min(r, g, b);
	      var max = Math.max(r, g, b);
	      var delta = max - min;
	      var c = delta * 100 / 255;

	      var _g = min / (255 - delta) * 100;

	      var h;

	      if (delta === 0) {
	        h = Number.NaN;
	      } else {
	        if (r === max) {
	          h = (g - b) / delta;
	        }

	        if (g === max) {
	          h = 2 + (b - r) / delta;
	        }

	        if (b === max) {
	          h = 4 + (r - g) / delta;
	        }

	        h *= 60;

	        if (h < 0) {
	          h += 360;
	        }
	      }

	      return [h, c, _g];
	    };

	    var rgb2hcg_1 = rgb2hcg$1;
	    var unpack$r = utils.unpack;
	    var floor$3 = Math.floor;
	    /*
	     * this is basically just HSV with some minor tweaks
	     *
	     * hue.. [0..360]
	     * chroma .. [0..1]
	     * grayness .. [0..1]
	     */

	    var hcg2rgb = function () {
	      var assign, assign$1, assign$2, assign$3, assign$4, assign$5;
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      args = unpack$r(args, 'hcg');
	      var h = args[0];
	      var c = args[1];
	      var _g = args[2];
	      var r, g, b;
	      _g = _g * 255;

	      var _c = c * 255;

	      if (c === 0) {
	        r = g = b = _g;
	      } else {
	        if (h === 360) {
	          h = 0;
	        }

	        if (h > 360) {
	          h -= 360;
	        }

	        if (h < 0) {
	          h += 360;
	        }

	        h /= 60;
	        var i = floor$3(h);
	        var f = h - i;
	        var p = _g * (1 - c);
	        var q = p + _c * (1 - f);
	        var t = p + _c * f;
	        var v = p + _c;

	        switch (i) {
	          case 0:
	            assign = [v, t, p], r = assign[0], g = assign[1], b = assign[2];
	            break;

	          case 1:
	            assign$1 = [q, v, p], r = assign$1[0], g = assign$1[1], b = assign$1[2];
	            break;

	          case 2:
	            assign$2 = [p, v, t], r = assign$2[0], g = assign$2[1], b = assign$2[2];
	            break;

	          case 3:
	            assign$3 = [p, q, v], r = assign$3[0], g = assign$3[1], b = assign$3[2];
	            break;

	          case 4:
	            assign$4 = [t, p, v], r = assign$4[0], g = assign$4[1], b = assign$4[2];
	            break;

	          case 5:
	            assign$5 = [v, p, q], r = assign$5[0], g = assign$5[1], b = assign$5[2];
	            break;
	        }
	      }

	      return [r, g, b, args.length > 3 ? args[3] : 1];
	    };

	    var hcg2rgb_1 = hcg2rgb;
	    var unpack$q = utils.unpack;
	    var type$j = utils.type;
	    var chroma$g = chroma_1;
	    var Color$z = Color_1;
	    var input$c = input$h;
	    var rgb2hcg = rgb2hcg_1;

	    Color$z.prototype.hcg = function () {
	      return rgb2hcg(this._rgb);
	    };

	    chroma$g.hcg = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      return new (Function.prototype.bind.apply(Color$z, [null].concat(args, ['hcg'])))();
	    };

	    input$c.format.hcg = hcg2rgb_1;
	    input$c.autodetect.push({
	      p: 1,
	      test: function () {
	        var args = [],
	            len = arguments.length;

	        while (len--) args[len] = arguments[len];

	        args = unpack$q(args, 'hcg');

	        if (type$j(args) === 'array' && args.length === 3) {
	          return 'hcg';
	        }
	      }
	    });
	    var unpack$p = utils.unpack;
	    var last = utils.last;
	    var round$3 = Math.round;

	    var rgb2hex$2 = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      var ref = unpack$p(args, 'rgba');
	      var r = ref[0];
	      var g = ref[1];
	      var b = ref[2];
	      var a = ref[3];
	      var mode = last(args) || 'auto';

	      if (a === undefined) {
	        a = 1;
	      }

	      if (mode === 'auto') {
	        mode = a < 1 ? 'rgba' : 'rgb';
	      }

	      r = round$3(r);
	      g = round$3(g);
	      b = round$3(b);
	      var u = r << 16 | g << 8 | b;
	      var str = "000000" + u.toString(16); //#.toUpperCase();

	      str = str.substr(str.length - 6);
	      var hxa = '0' + round$3(a * 255).toString(16);
	      hxa = hxa.substr(hxa.length - 2);

	      switch (mode.toLowerCase()) {
	        case 'rgba':
	          return "#" + str + hxa;

	        case 'argb':
	          return "#" + hxa + str;

	        default:
	          return "#" + str;
	      }
	    };

	    var rgb2hex_1 = rgb2hex$2;
	    var RE_HEX = /^#?([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/;
	    var RE_HEXA = /^#?([A-Fa-f0-9]{8}|[A-Fa-f0-9]{4})$/;

	    var hex2rgb$1 = function (hex) {
	      if (hex.match(RE_HEX)) {
	        // remove optional leading #
	        if (hex.length === 4 || hex.length === 7) {
	          hex = hex.substr(1);
	        } // expand short-notation to full six-digit


	        if (hex.length === 3) {
	          hex = hex.split('');
	          hex = hex[0] + hex[0] + hex[1] + hex[1] + hex[2] + hex[2];
	        }

	        var u = parseInt(hex, 16);
	        var r = u >> 16;
	        var g = u >> 8 & 0xFF;
	        var b = u & 0xFF;
	        return [r, g, b, 1];
	      } // match rgba hex format, eg #FF000077


	      if (hex.match(RE_HEXA)) {
	        if (hex.length === 5 || hex.length === 9) {
	          // remove optional leading #
	          hex = hex.substr(1);
	        } // expand short-notation to full eight-digit


	        if (hex.length === 4) {
	          hex = hex.split('');
	          hex = hex[0] + hex[0] + hex[1] + hex[1] + hex[2] + hex[2] + hex[3] + hex[3];
	        }

	        var u$1 = parseInt(hex, 16);
	        var r$1 = u$1 >> 24 & 0xFF;
	        var g$1 = u$1 >> 16 & 0xFF;
	        var b$1 = u$1 >> 8 & 0xFF;
	        var a = Math.round((u$1 & 0xFF) / 0xFF * 100) / 100;
	        return [r$1, g$1, b$1, a];
	      } // we used to check for css colors here
	      // if _input.css? and rgb = _input.css hex
	      //     return rgb


	      throw new Error("unknown hex color: " + hex);
	    };

	    var hex2rgb_1 = hex2rgb$1;
	    var chroma$f = chroma_1;
	    var Color$y = Color_1;
	    var type$i = utils.type;
	    var input$b = input$h;
	    var rgb2hex$1 = rgb2hex_1;

	    Color$y.prototype.hex = function (mode) {
	      return rgb2hex$1(this._rgb, mode);
	    };

	    chroma$f.hex = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      return new (Function.prototype.bind.apply(Color$y, [null].concat(args, ['hex'])))();
	    };

	    input$b.format.hex = hex2rgb_1;
	    input$b.autodetect.push({
	      p: 4,
	      test: function (h) {
	        var rest = [],
	            len = arguments.length - 1;

	        while (len-- > 0) rest[len] = arguments[len + 1];

	        if (!rest.length && type$i(h) === 'string' && [3, 4, 5, 6, 7, 8, 9].indexOf(h.length) >= 0) {
	          return 'hex';
	        }
	      }
	    });
	    var unpack$o = utils.unpack;
	    var TWOPI$2 = utils.TWOPI;
	    var min$2 = Math.min;
	    var sqrt$4 = Math.sqrt;
	    var acos = Math.acos;

	    var rgb2hsi$1 = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];
	      /*
	      borrowed from here:
	      http://hummer.stanford.edu/museinfo/doc/examples/humdrum/keyscape2/rgb2hsi.cpp
	      */


	      var ref = unpack$o(args, 'rgb');
	      var r = ref[0];
	      var g = ref[1];
	      var b = ref[2];
	      r /= 255;
	      g /= 255;
	      b /= 255;
	      var h;
	      var min_ = min$2(r, g, b);
	      var i = (r + g + b) / 3;
	      var s = i > 0 ? 1 - min_ / i : 0;

	      if (s === 0) {
	        h = NaN;
	      } else {
	        h = (r - g + (r - b)) / 2;
	        h /= sqrt$4((r - g) * (r - g) + (r - b) * (g - b));
	        h = acos(h);

	        if (b > g) {
	          h = TWOPI$2 - h;
	        }

	        h /= TWOPI$2;
	      }

	      return [h * 360, s, i];
	    };

	    var rgb2hsi_1 = rgb2hsi$1;
	    var unpack$n = utils.unpack;
	    var limit = utils.limit;
	    var TWOPI$1 = utils.TWOPI;
	    var PITHIRD = utils.PITHIRD;
	    var cos$4 = Math.cos;
	    /*
	     * hue [0..360]
	     * saturation [0..1]
	     * intensity [0..1]
	     */

	    var hsi2rgb = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];
	      /*
	      borrowed from here:
	      http://hummer.stanford.edu/museinfo/doc/examples/humdrum/keyscape2/hsi2rgb.cpp
	      */


	      args = unpack$n(args, 'hsi');
	      var h = args[0];
	      var s = args[1];
	      var i = args[2];
	      var r, g, b;

	      if (isNaN(h)) {
	        h = 0;
	      }

	      if (isNaN(s)) {
	        s = 0;
	      } // normalize hue


	      if (h > 360) {
	        h -= 360;
	      }

	      if (h < 0) {
	        h += 360;
	      }

	      h /= 360;

	      if (h < 1 / 3) {
	        b = (1 - s) / 3;
	        r = (1 + s * cos$4(TWOPI$1 * h) / cos$4(PITHIRD - TWOPI$1 * h)) / 3;
	        g = 1 - (b + r);
	      } else if (h < 2 / 3) {
	        h -= 1 / 3;
	        r = (1 - s) / 3;
	        g = (1 + s * cos$4(TWOPI$1 * h) / cos$4(PITHIRD - TWOPI$1 * h)) / 3;
	        b = 1 - (r + g);
	      } else {
	        h -= 2 / 3;
	        g = (1 - s) / 3;
	        b = (1 + s * cos$4(TWOPI$1 * h) / cos$4(PITHIRD - TWOPI$1 * h)) / 3;
	        r = 1 - (g + b);
	      }

	      r = limit(i * r * 3);
	      g = limit(i * g * 3);
	      b = limit(i * b * 3);
	      return [r * 255, g * 255, b * 255, args.length > 3 ? args[3] : 1];
	    };

	    var hsi2rgb_1 = hsi2rgb;
	    var unpack$m = utils.unpack;
	    var type$h = utils.type;
	    var chroma$e = chroma_1;
	    var Color$x = Color_1;
	    var input$a = input$h;
	    var rgb2hsi = rgb2hsi_1;

	    Color$x.prototype.hsi = function () {
	      return rgb2hsi(this._rgb);
	    };

	    chroma$e.hsi = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      return new (Function.prototype.bind.apply(Color$x, [null].concat(args, ['hsi'])))();
	    };

	    input$a.format.hsi = hsi2rgb_1;
	    input$a.autodetect.push({
	      p: 2,
	      test: function () {
	        var args = [],
	            len = arguments.length;

	        while (len--) args[len] = arguments[len];

	        args = unpack$m(args, 'hsi');

	        if (type$h(args) === 'array' && args.length === 3) {
	          return 'hsi';
	        }
	      }
	    });
	    var unpack$l = utils.unpack;
	    var type$g = utils.type;
	    var chroma$d = chroma_1;
	    var Color$w = Color_1;
	    var input$9 = input$h;
	    var rgb2hsl$1 = rgb2hsl_1;

	    Color$w.prototype.hsl = function () {
	      return rgb2hsl$1(this._rgb);
	    };

	    chroma$d.hsl = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      return new (Function.prototype.bind.apply(Color$w, [null].concat(args, ['hsl'])))();
	    };

	    input$9.format.hsl = hsl2rgb_1;
	    input$9.autodetect.push({
	      p: 2,
	      test: function () {
	        var args = [],
	            len = arguments.length;

	        while (len--) args[len] = arguments[len];

	        args = unpack$l(args, 'hsl');

	        if (type$g(args) === 'array' && args.length === 3) {
	          return 'hsl';
	        }
	      }
	    });
	    var unpack$k = utils.unpack;
	    var min$1 = Math.min;
	    var max$1 = Math.max;
	    /*
	     * supported arguments:
	     * - rgb2hsv(r,g,b)
	     * - rgb2hsv([r,g,b])
	     * - rgb2hsv({r,g,b})
	     */

	    var rgb2hsl = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      args = unpack$k(args, 'rgb');
	      var r = args[0];
	      var g = args[1];
	      var b = args[2];
	      var min_ = min$1(r, g, b);
	      var max_ = max$1(r, g, b);
	      var delta = max_ - min_;
	      var h, s, v;
	      v = max_ / 255.0;

	      if (max_ === 0) {
	        h = Number.NaN;
	        s = 0;
	      } else {
	        s = delta / max_;

	        if (r === max_) {
	          h = (g - b) / delta;
	        }

	        if (g === max_) {
	          h = 2 + (b - r) / delta;
	        }

	        if (b === max_) {
	          h = 4 + (r - g) / delta;
	        }

	        h *= 60;

	        if (h < 0) {
	          h += 360;
	        }
	      }

	      return [h, s, v];
	    };

	    var rgb2hsv$1 = rgb2hsl;
	    var unpack$j = utils.unpack;
	    var floor$2 = Math.floor;

	    var hsv2rgb = function () {
	      var assign, assign$1, assign$2, assign$3, assign$4, assign$5;
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      args = unpack$j(args, 'hsv');
	      var h = args[0];
	      var s = args[1];
	      var v = args[2];
	      var r, g, b;
	      v *= 255;

	      if (s === 0) {
	        r = g = b = v;
	      } else {
	        if (h === 360) {
	          h = 0;
	        }

	        if (h > 360) {
	          h -= 360;
	        }

	        if (h < 0) {
	          h += 360;
	        }

	        h /= 60;
	        var i = floor$2(h);
	        var f = h - i;
	        var p = v * (1 - s);
	        var q = v * (1 - s * f);
	        var t = v * (1 - s * (1 - f));

	        switch (i) {
	          case 0:
	            assign = [v, t, p], r = assign[0], g = assign[1], b = assign[2];
	            break;

	          case 1:
	            assign$1 = [q, v, p], r = assign$1[0], g = assign$1[1], b = assign$1[2];
	            break;

	          case 2:
	            assign$2 = [p, v, t], r = assign$2[0], g = assign$2[1], b = assign$2[2];
	            break;

	          case 3:
	            assign$3 = [p, q, v], r = assign$3[0], g = assign$3[1], b = assign$3[2];
	            break;

	          case 4:
	            assign$4 = [t, p, v], r = assign$4[0], g = assign$4[1], b = assign$4[2];
	            break;

	          case 5:
	            assign$5 = [v, p, q], r = assign$5[0], g = assign$5[1], b = assign$5[2];
	            break;
	        }
	      }

	      return [r, g, b, args.length > 3 ? args[3] : 1];
	    };

	    var hsv2rgb_1 = hsv2rgb;
	    var unpack$i = utils.unpack;
	    var type$f = utils.type;
	    var chroma$c = chroma_1;
	    var Color$v = Color_1;
	    var input$8 = input$h;
	    var rgb2hsv = rgb2hsv$1;

	    Color$v.prototype.hsv = function () {
	      return rgb2hsv(this._rgb);
	    };

	    chroma$c.hsv = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      return new (Function.prototype.bind.apply(Color$v, [null].concat(args, ['hsv'])))();
	    };

	    input$8.format.hsv = hsv2rgb_1;
	    input$8.autodetect.push({
	      p: 2,
	      test: function () {
	        var args = [],
	            len = arguments.length;

	        while (len--) args[len] = arguments[len];

	        args = unpack$i(args, 'hsv');

	        if (type$f(args) === 'array' && args.length === 3) {
	          return 'hsv';
	        }
	      }
	    });
	    var labConstants = {
	      // Corresponds roughly to RGB brighter/darker
	      Kn: 18,
	      // D65 standard referent
	      Xn: 0.950470,
	      Yn: 1,
	      Zn: 1.088830,
	      t0: 0.137931034,
	      // 4 / 29
	      t1: 0.206896552,
	      // 6 / 29
	      t2: 0.12841855,
	      // 3 * t1 * t1
	      t3: 0.008856452 // t1 * t1 * t1

	    };
	    var LAB_CONSTANTS$3 = labConstants;
	    var unpack$h = utils.unpack;
	    var pow$a = Math.pow;

	    var rgb2lab$2 = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      var ref = unpack$h(args, 'rgb');
	      var r = ref[0];
	      var g = ref[1];
	      var b = ref[2];
	      var ref$1 = rgb2xyz(r, g, b);
	      var x = ref$1[0];
	      var y = ref$1[1];
	      var z = ref$1[2];
	      var l = 116 * y - 16;
	      return [l < 0 ? 0 : l, 500 * (x - y), 200 * (y - z)];
	    };

	    var rgb_xyz = function (r) {
	      if ((r /= 255) <= 0.04045) {
	        return r / 12.92;
	      }

	      return pow$a((r + 0.055) / 1.055, 2.4);
	    };

	    var xyz_lab = function (t) {
	      if (t > LAB_CONSTANTS$3.t3) {
	        return pow$a(t, 1 / 3);
	      }

	      return t / LAB_CONSTANTS$3.t2 + LAB_CONSTANTS$3.t0;
	    };

	    var rgb2xyz = function (r, g, b) {
	      r = rgb_xyz(r);
	      g = rgb_xyz(g);
	      b = rgb_xyz(b);
	      var x = xyz_lab((0.4124564 * r + 0.3575761 * g + 0.1804375 * b) / LAB_CONSTANTS$3.Xn);
	      var y = xyz_lab((0.2126729 * r + 0.7151522 * g + 0.0721750 * b) / LAB_CONSTANTS$3.Yn);
	      var z = xyz_lab((0.0193339 * r + 0.1191920 * g + 0.9503041 * b) / LAB_CONSTANTS$3.Zn);
	      return [x, y, z];
	    };

	    var rgb2lab_1 = rgb2lab$2;
	    var LAB_CONSTANTS$2 = labConstants;
	    var unpack$g = utils.unpack;
	    var pow$9 = Math.pow;
	    /*
	     * L* [0..100]
	     * a [-100..100]
	     * b [-100..100]
	     */

	    var lab2rgb$1 = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      args = unpack$g(args, 'lab');
	      var l = args[0];
	      var a = args[1];
	      var b = args[2];
	      var x, y, z, r, g, b_;
	      y = (l + 16) / 116;
	      x = isNaN(a) ? y : y + a / 500;
	      z = isNaN(b) ? y : y - b / 200;
	      y = LAB_CONSTANTS$2.Yn * lab_xyz(y);
	      x = LAB_CONSTANTS$2.Xn * lab_xyz(x);
	      z = LAB_CONSTANTS$2.Zn * lab_xyz(z);
	      r = xyz_rgb(3.2404542 * x - 1.5371385 * y - 0.4985314 * z); // D65 -> sRGB

	      g = xyz_rgb(-0.9692660 * x + 1.8760108 * y + 0.0415560 * z);
	      b_ = xyz_rgb(0.0556434 * x - 0.2040259 * y + 1.0572252 * z);
	      return [r, g, b_, args.length > 3 ? args[3] : 1];
	    };

	    var xyz_rgb = function (r) {
	      return 255 * (r <= 0.00304 ? 12.92 * r : 1.055 * pow$9(r, 1 / 2.4) - 0.055);
	    };

	    var lab_xyz = function (t) {
	      return t > LAB_CONSTANTS$2.t1 ? t * t * t : LAB_CONSTANTS$2.t2 * (t - LAB_CONSTANTS$2.t0);
	    };

	    var lab2rgb_1 = lab2rgb$1;
	    var unpack$f = utils.unpack;
	    var type$e = utils.type;
	    var chroma$b = chroma_1;
	    var Color$u = Color_1;
	    var input$7 = input$h;
	    var rgb2lab$1 = rgb2lab_1;

	    Color$u.prototype.lab = function () {
	      return rgb2lab$1(this._rgb);
	    };

	    chroma$b.lab = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      return new (Function.prototype.bind.apply(Color$u, [null].concat(args, ['lab'])))();
	    };

	    input$7.format.lab = lab2rgb_1;
	    input$7.autodetect.push({
	      p: 2,
	      test: function () {
	        var args = [],
	            len = arguments.length;

	        while (len--) args[len] = arguments[len];

	        args = unpack$f(args, 'lab');

	        if (type$e(args) === 'array' && args.length === 3) {
	          return 'lab';
	        }
	      }
	    });
	    var unpack$e = utils.unpack;
	    var RAD2DEG = utils.RAD2DEG;
	    var sqrt$3 = Math.sqrt;
	    var atan2$2 = Math.atan2;
	    var round$2 = Math.round;

	    var lab2lch$2 = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      var ref = unpack$e(args, 'lab');
	      var l = ref[0];
	      var a = ref[1];
	      var b = ref[2];
	      var c = sqrt$3(a * a + b * b);
	      var h = (atan2$2(b, a) * RAD2DEG + 360) % 360;

	      if (round$2(c * 10000) === 0) {
	        h = Number.NaN;
	      }

	      return [l, c, h];
	    };

	    var lab2lch_1 = lab2lch$2;
	    var unpack$d = utils.unpack;
	    var rgb2lab = rgb2lab_1;
	    var lab2lch$1 = lab2lch_1;

	    var rgb2lch$1 = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      var ref = unpack$d(args, 'rgb');
	      var r = ref[0];
	      var g = ref[1];
	      var b = ref[2];
	      var ref$1 = rgb2lab(r, g, b);
	      var l = ref$1[0];
	      var a = ref$1[1];
	      var b_ = ref$1[2];
	      return lab2lch$1(l, a, b_);
	    };

	    var rgb2lch_1 = rgb2lch$1;
	    var unpack$c = utils.unpack;
	    var DEG2RAD = utils.DEG2RAD;
	    var sin$3 = Math.sin;
	    var cos$3 = Math.cos;

	    var lch2lab$2 = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];
	      /*
	      Convert from a qualitative parameter h and a quantitative parameter l to a 24-bit pixel.
	      These formulas were invented by David Dalrymple to obtain maximum contrast without going
	      out of gamut if the parameters are in the range 0-1.
	       A saturation multiplier was added by Gregor Aisch
	      */


	      var ref = unpack$c(args, 'lch');
	      var l = ref[0];
	      var c = ref[1];
	      var h = ref[2];

	      if (isNaN(h)) {
	        h = 0;
	      }

	      h = h * DEG2RAD;
	      return [l, cos$3(h) * c, sin$3(h) * c];
	    };

	    var lch2lab_1 = lch2lab$2;
	    var unpack$b = utils.unpack;
	    var lch2lab$1 = lch2lab_1;
	    var lab2rgb = lab2rgb_1;

	    var lch2rgb$1 = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      args = unpack$b(args, 'lch');
	      var l = args[0];
	      var c = args[1];
	      var h = args[2];
	      var ref = lch2lab$1(l, c, h);
	      var L = ref[0];
	      var a = ref[1];
	      var b_ = ref[2];
	      var ref$1 = lab2rgb(L, a, b_);
	      var r = ref$1[0];
	      var g = ref$1[1];
	      var b = ref$1[2];
	      return [r, g, b, args.length > 3 ? args[3] : 1];
	    };

	    var lch2rgb_1 = lch2rgb$1;
	    var unpack$a = utils.unpack;
	    var lch2rgb = lch2rgb_1;

	    var hcl2rgb = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      var hcl = unpack$a(args, 'hcl').reverse();
	      return lch2rgb.apply(void 0, hcl);
	    };

	    var hcl2rgb_1 = hcl2rgb;
	    var unpack$9 = utils.unpack;
	    var type$d = utils.type;
	    var chroma$a = chroma_1;
	    var Color$t = Color_1;
	    var input$6 = input$h;
	    var rgb2lch = rgb2lch_1;

	    Color$t.prototype.lch = function () {
	      return rgb2lch(this._rgb);
	    };

	    Color$t.prototype.hcl = function () {
	      return rgb2lch(this._rgb).reverse();
	    };

	    chroma$a.lch = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      return new (Function.prototype.bind.apply(Color$t, [null].concat(args, ['lch'])))();
	    };

	    chroma$a.hcl = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      return new (Function.prototype.bind.apply(Color$t, [null].concat(args, ['hcl'])))();
	    };

	    input$6.format.lch = lch2rgb_1;
	    input$6.format.hcl = hcl2rgb_1;
	    ['lch', 'hcl'].forEach(function (m) {
	      return input$6.autodetect.push({
	        p: 2,
	        test: function () {
	          var args = [],
	              len = arguments.length;

	          while (len--) args[len] = arguments[len];

	          args = unpack$9(args, m);

	          if (type$d(args) === 'array' && args.length === 3) {
	            return m;
	          }
	        }
	      });
	    });
	    /**
	    	X11 color names
	     	http://www.w3.org/TR/css3-color/#svg-color
	    */

	    var w3cx11$1 = {
	      aliceblue: '#f0f8ff',
	      antiquewhite: '#faebd7',
	      aqua: '#00ffff',
	      aquamarine: '#7fffd4',
	      azure: '#f0ffff',
	      beige: '#f5f5dc',
	      bisque: '#ffe4c4',
	      black: '#000000',
	      blanchedalmond: '#ffebcd',
	      blue: '#0000ff',
	      blueviolet: '#8a2be2',
	      brown: '#a52a2a',
	      burlywood: '#deb887',
	      cadetblue: '#5f9ea0',
	      chartreuse: '#7fff00',
	      chocolate: '#d2691e',
	      coral: '#ff7f50',
	      cornflower: '#6495ed',
	      cornflowerblue: '#6495ed',
	      cornsilk: '#fff8dc',
	      crimson: '#dc143c',
	      cyan: '#00ffff',
	      darkblue: '#00008b',
	      darkcyan: '#008b8b',
	      darkgoldenrod: '#b8860b',
	      darkgray: '#a9a9a9',
	      darkgreen: '#006400',
	      darkgrey: '#a9a9a9',
	      darkkhaki: '#bdb76b',
	      darkmagenta: '#8b008b',
	      darkolivegreen: '#556b2f',
	      darkorange: '#ff8c00',
	      darkorchid: '#9932cc',
	      darkred: '#8b0000',
	      darksalmon: '#e9967a',
	      darkseagreen: '#8fbc8f',
	      darkslateblue: '#483d8b',
	      darkslategray: '#2f4f4f',
	      darkslategrey: '#2f4f4f',
	      darkturquoise: '#00ced1',
	      darkviolet: '#9400d3',
	      deeppink: '#ff1493',
	      deepskyblue: '#00bfff',
	      dimgray: '#696969',
	      dimgrey: '#696969',
	      dodgerblue: '#1e90ff',
	      firebrick: '#b22222',
	      floralwhite: '#fffaf0',
	      forestgreen: '#228b22',
	      fuchsia: '#ff00ff',
	      gainsboro: '#dcdcdc',
	      ghostwhite: '#f8f8ff',
	      gold: '#ffd700',
	      goldenrod: '#daa520',
	      gray: '#808080',
	      green: '#008000',
	      greenyellow: '#adff2f',
	      grey: '#808080',
	      honeydew: '#f0fff0',
	      hotpink: '#ff69b4',
	      indianred: '#cd5c5c',
	      indigo: '#4b0082',
	      ivory: '#fffff0',
	      khaki: '#f0e68c',
	      laserlemon: '#ffff54',
	      lavender: '#e6e6fa',
	      lavenderblush: '#fff0f5',
	      lawngreen: '#7cfc00',
	      lemonchiffon: '#fffacd',
	      lightblue: '#add8e6',
	      lightcoral: '#f08080',
	      lightcyan: '#e0ffff',
	      lightgoldenrod: '#fafad2',
	      lightgoldenrodyellow: '#fafad2',
	      lightgray: '#d3d3d3',
	      lightgreen: '#90ee90',
	      lightgrey: '#d3d3d3',
	      lightpink: '#ffb6c1',
	      lightsalmon: '#ffa07a',
	      lightseagreen: '#20b2aa',
	      lightskyblue: '#87cefa',
	      lightslategray: '#778899',
	      lightslategrey: '#778899',
	      lightsteelblue: '#b0c4de',
	      lightyellow: '#ffffe0',
	      lime: '#00ff00',
	      limegreen: '#32cd32',
	      linen: '#faf0e6',
	      magenta: '#ff00ff',
	      maroon: '#800000',
	      maroon2: '#7f0000',
	      maroon3: '#b03060',
	      mediumaquamarine: '#66cdaa',
	      mediumblue: '#0000cd',
	      mediumorchid: '#ba55d3',
	      mediumpurple: '#9370db',
	      mediumseagreen: '#3cb371',
	      mediumslateblue: '#7b68ee',
	      mediumspringgreen: '#00fa9a',
	      mediumturquoise: '#48d1cc',
	      mediumvioletred: '#c71585',
	      midnightblue: '#191970',
	      mintcream: '#f5fffa',
	      mistyrose: '#ffe4e1',
	      moccasin: '#ffe4b5',
	      navajowhite: '#ffdead',
	      navy: '#000080',
	      oldlace: '#fdf5e6',
	      olive: '#808000',
	      olivedrab: '#6b8e23',
	      orange: '#ffa500',
	      orangered: '#ff4500',
	      orchid: '#da70d6',
	      palegoldenrod: '#eee8aa',
	      palegreen: '#98fb98',
	      paleturquoise: '#afeeee',
	      palevioletred: '#db7093',
	      papayawhip: '#ffefd5',
	      peachpuff: '#ffdab9',
	      peru: '#cd853f',
	      pink: '#ffc0cb',
	      plum: '#dda0dd',
	      powderblue: '#b0e0e6',
	      purple: '#800080',
	      purple2: '#7f007f',
	      purple3: '#a020f0',
	      rebeccapurple: '#663399',
	      red: '#ff0000',
	      rosybrown: '#bc8f8f',
	      royalblue: '#4169e1',
	      saddlebrown: '#8b4513',
	      salmon: '#fa8072',
	      sandybrown: '#f4a460',
	      seagreen: '#2e8b57',
	      seashell: '#fff5ee',
	      sienna: '#a0522d',
	      silver: '#c0c0c0',
	      skyblue: '#87ceeb',
	      slateblue: '#6a5acd',
	      slategray: '#708090',
	      slategrey: '#708090',
	      snow: '#fffafa',
	      springgreen: '#00ff7f',
	      steelblue: '#4682b4',
	      tan: '#d2b48c',
	      teal: '#008080',
	      thistle: '#d8bfd8',
	      tomato: '#ff6347',
	      turquoise: '#40e0d0',
	      violet: '#ee82ee',
	      wheat: '#f5deb3',
	      white: '#ffffff',
	      whitesmoke: '#f5f5f5',
	      yellow: '#ffff00',
	      yellowgreen: '#9acd32'
	    };
	    var w3cx11_1 = w3cx11$1;
	    var Color$s = Color_1;
	    var input$5 = input$h;
	    var type$c = utils.type;
	    var w3cx11 = w3cx11_1;
	    var hex2rgb = hex2rgb_1;
	    var rgb2hex = rgb2hex_1;

	    Color$s.prototype.name = function () {
	      var hex = rgb2hex(this._rgb, 'rgb');

	      for (var i = 0, list = Object.keys(w3cx11); i < list.length; i += 1) {
	        var n = list[i];

	        if (w3cx11[n] === hex) {
	          return n.toLowerCase();
	        }
	      }

	      return hex;
	    };

	    input$5.format.named = function (name) {
	      name = name.toLowerCase();

	      if (w3cx11[name]) {
	        return hex2rgb(w3cx11[name]);
	      }

	      throw new Error('unknown color name: ' + name);
	    };

	    input$5.autodetect.push({
	      p: 5,
	      test: function (h) {
	        var rest = [],
	            len = arguments.length - 1;

	        while (len-- > 0) rest[len] = arguments[len + 1];

	        if (!rest.length && type$c(h) === 'string' && w3cx11[h.toLowerCase()]) {
	          return 'named';
	        }
	      }
	    });
	    var unpack$8 = utils.unpack;

	    var rgb2num$1 = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      var ref = unpack$8(args, 'rgb');
	      var r = ref[0];
	      var g = ref[1];
	      var b = ref[2];
	      return (r << 16) + (g << 8) + b;
	    };

	    var rgb2num_1 = rgb2num$1;
	    var type$b = utils.type;

	    var num2rgb = function (num) {
	      if (type$b(num) == "number" && num >= 0 && num <= 0xFFFFFF) {
	        var r = num >> 16;
	        var g = num >> 8 & 0xFF;
	        var b = num & 0xFF;
	        return [r, g, b, 1];
	      }

	      throw new Error("unknown num color: " + num);
	    };

	    var num2rgb_1 = num2rgb;
	    var chroma$9 = chroma_1;
	    var Color$r = Color_1;
	    var input$4 = input$h;
	    var type$a = utils.type;
	    var rgb2num = rgb2num_1;

	    Color$r.prototype.num = function () {
	      return rgb2num(this._rgb);
	    };

	    chroma$9.num = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      return new (Function.prototype.bind.apply(Color$r, [null].concat(args, ['num'])))();
	    };

	    input$4.format.num = num2rgb_1;
	    input$4.autodetect.push({
	      p: 5,
	      test: function () {
	        var args = [],
	            len = arguments.length;

	        while (len--) args[len] = arguments[len];

	        if (args.length === 1 && type$a(args[0]) === 'number' && args[0] >= 0 && args[0] <= 0xFFFFFF) {
	          return 'num';
	        }
	      }
	    });
	    var chroma$8 = chroma_1;
	    var Color$q = Color_1;
	    var input$3 = input$h;
	    var unpack$7 = utils.unpack;
	    var type$9 = utils.type;
	    var round$1 = Math.round;

	    Color$q.prototype.rgb = function (rnd) {
	      if (rnd === void 0) rnd = true;

	      if (rnd === false) {
	        return this._rgb.slice(0, 3);
	      }

	      return this._rgb.slice(0, 3).map(round$1);
	    };

	    Color$q.prototype.rgba = function (rnd) {
	      if (rnd === void 0) rnd = true;
	      return this._rgb.slice(0, 4).map(function (v, i) {
	        return i < 3 ? rnd === false ? v : round$1(v) : v;
	      });
	    };

	    chroma$8.rgb = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      return new (Function.prototype.bind.apply(Color$q, [null].concat(args, ['rgb'])))();
	    };

	    input$3.format.rgb = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      var rgba = unpack$7(args, 'rgba');

	      if (rgba[3] === undefined) {
	        rgba[3] = 1;
	      }

	      return rgba;
	    };

	    input$3.autodetect.push({
	      p: 3,
	      test: function () {
	        var args = [],
	            len = arguments.length;

	        while (len--) args[len] = arguments[len];

	        args = unpack$7(args, 'rgba');

	        if (type$9(args) === 'array' && (args.length === 3 || args.length === 4 && type$9(args[3]) == 'number' && args[3] >= 0 && args[3] <= 1)) {
	          return 'rgb';
	        }
	      }
	    });
	    /*
	     * Based on implementation by Neil Bartlett
	     * https://github.com/neilbartlett/color-temperature
	     */

	    var log$1 = Math.log;

	    var temperature2rgb$1 = function (kelvin) {
	      var temp = kelvin / 100;
	      var r, g, b;

	      if (temp < 66) {
	        r = 255;
	        g = temp < 6 ? 0 : -155.25485562709179 - 0.44596950469579133 * (g = temp - 2) + 104.49216199393888 * log$1(g);
	        b = temp < 20 ? 0 : -254.76935184120902 + 0.8274096064007395 * (b = temp - 10) + 115.67994401066147 * log$1(b);
	      } else {
	        r = 351.97690566805693 + 0.114206453784165 * (r = temp - 55) - 40.25366309332127 * log$1(r);
	        g = 325.4494125711974 + 0.07943456536662342 * (g = temp - 50) - 28.0852963507957 * log$1(g);
	        b = 255;
	      }

	      return [r, g, b, 1];
	    };

	    var temperature2rgb_1 = temperature2rgb$1;
	    /*
	     * Based on implementation by Neil Bartlett
	     * https://github.com/neilbartlett/color-temperature
	     **/

	    var temperature2rgb = temperature2rgb_1;
	    var unpack$6 = utils.unpack;
	    var round = Math.round;

	    var rgb2temperature$1 = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      var rgb = unpack$6(args, 'rgb');
	      var r = rgb[0],
	          b = rgb[2];
	      var minTemp = 1000;
	      var maxTemp = 40000;
	      var eps = 0.4;
	      var temp;

	      while (maxTemp - minTemp > eps) {
	        temp = (maxTemp + minTemp) * 0.5;
	        var rgb$1 = temperature2rgb(temp);

	        if (rgb$1[2] / rgb$1[0] >= b / r) {
	          maxTemp = temp;
	        } else {
	          minTemp = temp;
	        }
	      }

	      return round(temp);
	    };

	    var rgb2temperature_1 = rgb2temperature$1;
	    var chroma$7 = chroma_1;
	    var Color$p = Color_1;
	    var input$2 = input$h;
	    var rgb2temperature = rgb2temperature_1;

	    Color$p.prototype.temp = Color$p.prototype.kelvin = Color$p.prototype.temperature = function () {
	      return rgb2temperature(this._rgb);
	    };

	    chroma$7.temp = chroma$7.kelvin = chroma$7.temperature = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      return new (Function.prototype.bind.apply(Color$p, [null].concat(args, ['temp'])))();
	    };

	    input$2.format.temp = input$2.format.kelvin = input$2.format.temperature = temperature2rgb_1;
	    var unpack$5 = utils.unpack;
	    var cbrt = Math.cbrt;
	    var pow$8 = Math.pow;
	    var sign$1 = Math.sign;

	    var rgb2oklab$2 = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len]; // OKLab color space implementation taken from
	      // https://bottosson.github.io/posts/oklab/


	      var ref = unpack$5(args, 'rgb');
	      var r = ref[0];
	      var g = ref[1];
	      var b = ref[2];
	      var ref$1 = [rgb2lrgb(r / 255), rgb2lrgb(g / 255), rgb2lrgb(b / 255)];
	      var lr = ref$1[0];
	      var lg = ref$1[1];
	      var lb = ref$1[2];
	      var l = cbrt(0.4122214708 * lr + 0.5363325363 * lg + 0.0514459929 * lb);
	      var m = cbrt(0.2119034982 * lr + 0.6806995451 * lg + 0.1073969566 * lb);
	      var s = cbrt(0.0883024619 * lr + 0.2817188376 * lg + 0.6299787005 * lb);
	      return [0.2104542553 * l + 0.793617785 * m - 0.0040720468 * s, 1.9779984951 * l - 2.428592205 * m + 0.4505937099 * s, 0.0259040371 * l + 0.7827717662 * m - 0.808675766 * s];
	    };

	    var rgb2oklab_1 = rgb2oklab$2;

	    function rgb2lrgb(c) {
	      var abs = Math.abs(c);

	      if (abs < 0.04045) {
	        return c / 12.92;
	      }

	      return (sign$1(c) || 1) * pow$8((abs + 0.055) / 1.055, 2.4);
	    }

	    var unpack$4 = utils.unpack;
	    var pow$7 = Math.pow;
	    var sign = Math.sign;
	    /*
	     * L* [0..100]
	     * a [-100..100]
	     * b [-100..100]
	     */

	    var oklab2rgb$1 = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      args = unpack$4(args, 'lab');
	      var L = args[0];
	      var a = args[1];
	      var b = args[2];
	      var l = pow$7(L + 0.3963377774 * a + 0.2158037573 * b, 3);
	      var m = pow$7(L - 0.1055613458 * a - 0.0638541728 * b, 3);
	      var s = pow$7(L - 0.0894841775 * a - 1.291485548 * b, 3);
	      return [255 * lrgb2rgb(+4.0767416621 * l - 3.3077115913 * m + 0.2309699292 * s), 255 * lrgb2rgb(-1.2684380046 * l + 2.6097574011 * m - 0.3413193965 * s), 255 * lrgb2rgb(-0.0041960863 * l - 0.7034186147 * m + 1.707614701 * s), args.length > 3 ? args[3] : 1];
	    };

	    var oklab2rgb_1 = oklab2rgb$1;

	    function lrgb2rgb(c) {
	      var abs = Math.abs(c);

	      if (abs > 0.0031308) {
	        return (sign(c) || 1) * (1.055 * pow$7(abs, 1 / 2.4) - 0.055);
	      }

	      return c * 12.92;
	    }

	    var unpack$3 = utils.unpack;
	    var type$8 = utils.type;
	    var chroma$6 = chroma_1;
	    var Color$o = Color_1;
	    var input$1 = input$h;
	    var rgb2oklab$1 = rgb2oklab_1;

	    Color$o.prototype.oklab = function () {
	      return rgb2oklab$1(this._rgb);
	    };

	    chroma$6.oklab = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      return new (Function.prototype.bind.apply(Color$o, [null].concat(args, ['oklab'])))();
	    };

	    input$1.format.oklab = oklab2rgb_1;
	    input$1.autodetect.push({
	      p: 3,
	      test: function () {
	        var args = [],
	            len = arguments.length;

	        while (len--) args[len] = arguments[len];

	        args = unpack$3(args, 'oklab');

	        if (type$8(args) === 'array' && args.length === 3) {
	          return 'oklab';
	        }
	      }
	    });
	    var unpack$2 = utils.unpack;
	    var rgb2oklab = rgb2oklab_1;
	    var lab2lch = lab2lch_1;

	    var rgb2oklch$1 = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      var ref = unpack$2(args, 'rgb');
	      var r = ref[0];
	      var g = ref[1];
	      var b = ref[2];
	      var ref$1 = rgb2oklab(r, g, b);
	      var l = ref$1[0];
	      var a = ref$1[1];
	      var b_ = ref$1[2];
	      return lab2lch(l, a, b_);
	    };

	    var rgb2oklch_1 = rgb2oklch$1;
	    var unpack$1 = utils.unpack;
	    var lch2lab = lch2lab_1;
	    var oklab2rgb = oklab2rgb_1;

	    var oklch2rgb = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      args = unpack$1(args, 'lch');
	      var l = args[0];
	      var c = args[1];
	      var h = args[2];
	      var ref = lch2lab(l, c, h);
	      var L = ref[0];
	      var a = ref[1];
	      var b_ = ref[2];
	      var ref$1 = oklab2rgb(L, a, b_);
	      var r = ref$1[0];
	      var g = ref$1[1];
	      var b = ref$1[2];
	      return [r, g, b, args.length > 3 ? args[3] : 1];
	    };

	    var oklch2rgb_1 = oklch2rgb;
	    var unpack = utils.unpack;
	    var type$7 = utils.type;
	    var chroma$5 = chroma_1;
	    var Color$n = Color_1;
	    var input = input$h;
	    var rgb2oklch = rgb2oklch_1;

	    Color$n.prototype.oklch = function () {
	      return rgb2oklch(this._rgb);
	    };

	    chroma$5.oklch = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      return new (Function.prototype.bind.apply(Color$n, [null].concat(args, ['oklch'])))();
	    };

	    input.format.oklch = oklch2rgb_1;
	    input.autodetect.push({
	      p: 3,
	      test: function () {
	        var args = [],
	            len = arguments.length;

	        while (len--) args[len] = arguments[len];

	        args = unpack(args, 'oklch');

	        if (type$7(args) === 'array' && args.length === 3) {
	          return 'oklch';
	        }
	      }
	    });
	    var Color$m = Color_1;
	    var type$6 = utils.type;

	    Color$m.prototype.alpha = function (a, mutate) {
	      if (mutate === void 0) mutate = false;

	      if (a !== undefined && type$6(a) === 'number') {
	        if (mutate) {
	          this._rgb[3] = a;
	          return this;
	        }

	        return new Color$m([this._rgb[0], this._rgb[1], this._rgb[2], a], 'rgb');
	      }

	      return this._rgb[3];
	    };

	    var Color$l = Color_1;

	    Color$l.prototype.clipped = function () {
	      return this._rgb._clipped || false;
	    };

	    var Color$k = Color_1;
	    var LAB_CONSTANTS$1 = labConstants;

	    Color$k.prototype.darken = function (amount) {
	      if (amount === void 0) amount = 1;
	      var me = this;
	      var lab = me.lab();
	      lab[0] -= LAB_CONSTANTS$1.Kn * amount;
	      return new Color$k(lab, 'lab').alpha(me.alpha(), true);
	    };

	    Color$k.prototype.brighten = function (amount) {
	      if (amount === void 0) amount = 1;
	      return this.darken(-amount);
	    };

	    Color$k.prototype.darker = Color$k.prototype.darken;
	    Color$k.prototype.brighter = Color$k.prototype.brighten;
	    var Color$j = Color_1;

	    Color$j.prototype.get = function (mc) {
	      var ref = mc.split('.');
	      var mode = ref[0];
	      var channel = ref[1];
	      var src = this[mode]();

	      if (channel) {
	        var i = mode.indexOf(channel) - (mode.substr(0, 2) === 'ok' ? 2 : 0);

	        if (i > -1) {
	          return src[i];
	        }

	        throw new Error("unknown channel " + channel + " in mode " + mode);
	      } else {
	        return src;
	      }
	    };

	    var Color$i = Color_1;
	    var type$5 = utils.type;
	    var pow$6 = Math.pow;
	    var EPS = 1e-7;
	    var MAX_ITER = 20;

	    Color$i.prototype.luminance = function (lum) {
	      if (lum !== undefined && type$5(lum) === 'number') {
	        if (lum === 0) {
	          // return pure black
	          return new Color$i([0, 0, 0, this._rgb[3]], 'rgb');
	        }

	        if (lum === 1) {
	          // return pure white
	          return new Color$i([255, 255, 255, this._rgb[3]], 'rgb');
	        } // compute new color using...


	        var cur_lum = this.luminance();
	        var mode = 'rgb';
	        var max_iter = MAX_ITER;

	        var test = function (low, high) {
	          var mid = low.interpolate(high, 0.5, mode);
	          var lm = mid.luminance();

	          if (Math.abs(lum - lm) < EPS || !max_iter--) {
	            // close enough
	            return mid;
	          }

	          return lm > lum ? test(low, mid) : test(mid, high);
	        };

	        var rgb = (cur_lum > lum ? test(new Color$i([0, 0, 0]), this) : test(this, new Color$i([255, 255, 255]))).rgb();
	        return new Color$i(rgb.concat([this._rgb[3]]));
	      }

	      return rgb2luminance.apply(void 0, this._rgb.slice(0, 3));
	    };

	    var rgb2luminance = function (r, g, b) {
	      // relative luminance
	      // see http://www.w3.org/TR/2008/REC-WCAG20-20081211/#relativeluminancedef
	      r = luminance_x(r);
	      g = luminance_x(g);
	      b = luminance_x(b);
	      return 0.2126 * r + 0.7152 * g + 0.0722 * b;
	    };

	    var luminance_x = function (x) {
	      x /= 255;
	      return x <= 0.03928 ? x / 12.92 : pow$6((x + 0.055) / 1.055, 2.4);
	    };

	    var interpolator$1 = {};
	    var Color$h = Color_1;
	    var type$4 = utils.type;
	    var interpolator = interpolator$1;

	    var mix$1 = function (col1, col2, f) {
	      if (f === void 0) f = 0.5;
	      var rest = [],
	          len = arguments.length - 3;

	      while (len-- > 0) rest[len] = arguments[len + 3];

	      var mode = rest[0] || 'lrgb';

	      if (!interpolator[mode] && !rest.length) {
	        // fall back to the first supported mode
	        mode = Object.keys(interpolator)[0];
	      }

	      if (!interpolator[mode]) {
	        throw new Error("interpolation mode " + mode + " is not defined");
	      }

	      if (type$4(col1) !== 'object') {
	        col1 = new Color$h(col1);
	      }

	      if (type$4(col2) !== 'object') {
	        col2 = new Color$h(col2);
	      }

	      return interpolator[mode](col1, col2, f).alpha(col1.alpha() + f * (col2.alpha() - col1.alpha()));
	    };

	    var Color$g = Color_1;
	    var mix = mix$1;

	    Color$g.prototype.mix = Color$g.prototype.interpolate = function (col2, f) {
	      if (f === void 0) f = 0.5;
	      var rest = [],
	          len = arguments.length - 2;

	      while (len-- > 0) rest[len] = arguments[len + 2];

	      return mix.apply(void 0, [this, col2, f].concat(rest));
	    };

	    var Color$f = Color_1;

	    Color$f.prototype.premultiply = function (mutate) {
	      if (mutate === void 0) mutate = false;
	      var rgb = this._rgb;
	      var a = rgb[3];

	      if (mutate) {
	        this._rgb = [rgb[0] * a, rgb[1] * a, rgb[2] * a, a];
	        return this;
	      } else {
	        return new Color$f([rgb[0] * a, rgb[1] * a, rgb[2] * a, a], 'rgb');
	      }
	    };

	    var Color$e = Color_1;
	    var LAB_CONSTANTS = labConstants;

	    Color$e.prototype.saturate = function (amount) {
	      if (amount === void 0) amount = 1;
	      var me = this;
	      var lch = me.lch();
	      lch[1] += LAB_CONSTANTS.Kn * amount;

	      if (lch[1] < 0) {
	        lch[1] = 0;
	      }

	      return new Color$e(lch, 'lch').alpha(me.alpha(), true);
	    };

	    Color$e.prototype.desaturate = function (amount) {
	      if (amount === void 0) amount = 1;
	      return this.saturate(-amount);
	    };

	    var Color$d = Color_1;
	    var type$3 = utils.type;

	    Color$d.prototype.set = function (mc, value, mutate) {
	      if (mutate === void 0) mutate = false;
	      var ref = mc.split('.');
	      var mode = ref[0];
	      var channel = ref[1];
	      var src = this[mode]();

	      if (channel) {
	        var i = mode.indexOf(channel) - (mode.substr(0, 2) === 'ok' ? 2 : 0);

	        if (i > -1) {
	          if (type$3(value) == 'string') {
	            switch (value.charAt(0)) {
	              case '+':
	                src[i] += +value;
	                break;

	              case '-':
	                src[i] += +value;
	                break;

	              case '*':
	                src[i] *= +value.substr(1);
	                break;

	              case '/':
	                src[i] /= +value.substr(1);
	                break;

	              default:
	                src[i] = +value;
	            }
	          } else if (type$3(value) === 'number') {
	            src[i] = value;
	          } else {
	            throw new Error("unsupported value for Color.set");
	          }

	          var out = new Color$d(src, mode);

	          if (mutate) {
	            this._rgb = out._rgb;
	            return this;
	          }

	          return out;
	        }

	        throw new Error("unknown channel " + channel + " in mode " + mode);
	      } else {
	        return src;
	      }
	    };

	    var Color$c = Color_1;

	    var rgb = function (col1, col2, f) {
	      var xyz0 = col1._rgb;
	      var xyz1 = col2._rgb;
	      return new Color$c(xyz0[0] + f * (xyz1[0] - xyz0[0]), xyz0[1] + f * (xyz1[1] - xyz0[1]), xyz0[2] + f * (xyz1[2] - xyz0[2]), 'rgb');
	    }; // register interpolator


	    interpolator$1.rgb = rgb;
	    var Color$b = Color_1;
	    var sqrt$2 = Math.sqrt;
	    var pow$5 = Math.pow;

	    var lrgb = function (col1, col2, f) {
	      var ref = col1._rgb;
	      var x1 = ref[0];
	      var y1 = ref[1];
	      var z1 = ref[2];
	      var ref$1 = col2._rgb;
	      var x2 = ref$1[0];
	      var y2 = ref$1[1];
	      var z2 = ref$1[2];
	      return new Color$b(sqrt$2(pow$5(x1, 2) * (1 - f) + pow$5(x2, 2) * f), sqrt$2(pow$5(y1, 2) * (1 - f) + pow$5(y2, 2) * f), sqrt$2(pow$5(z1, 2) * (1 - f) + pow$5(z2, 2) * f), 'rgb');
	    }; // register interpolator


	    interpolator$1.lrgb = lrgb;
	    var Color$a = Color_1;

	    var lab = function (col1, col2, f) {
	      var xyz0 = col1.lab();
	      var xyz1 = col2.lab();
	      return new Color$a(xyz0[0] + f * (xyz1[0] - xyz0[0]), xyz0[1] + f * (xyz1[1] - xyz0[1]), xyz0[2] + f * (xyz1[2] - xyz0[2]), 'lab');
	    }; // register interpolator


	    interpolator$1.lab = lab;
	    var Color$9 = Color_1;

	    var _hsx = function (col1, col2, f, m) {
	      var assign, assign$1;
	      var xyz0, xyz1;

	      if (m === 'hsl') {
	        xyz0 = col1.hsl();
	        xyz1 = col2.hsl();
	      } else if (m === 'hsv') {
	        xyz0 = col1.hsv();
	        xyz1 = col2.hsv();
	      } else if (m === 'hcg') {
	        xyz0 = col1.hcg();
	        xyz1 = col2.hcg();
	      } else if (m === 'hsi') {
	        xyz0 = col1.hsi();
	        xyz1 = col2.hsi();
	      } else if (m === 'lch' || m === 'hcl') {
	        m = 'hcl';
	        xyz0 = col1.hcl();
	        xyz1 = col2.hcl();
	      } else if (m === 'oklch') {
	        xyz0 = col1.oklch().reverse();
	        xyz1 = col2.oklch().reverse();
	      }

	      var hue0, hue1, sat0, sat1, lbv0, lbv1;

	      if (m.substr(0, 1) === 'h' || m === 'oklch') {
	        assign = xyz0, hue0 = assign[0], sat0 = assign[1], lbv0 = assign[2];
	        assign$1 = xyz1, hue1 = assign$1[0], sat1 = assign$1[1], lbv1 = assign$1[2];
	      }

	      var sat, hue, lbv, dh;

	      if (!isNaN(hue0) && !isNaN(hue1)) {
	        // both colors have hue
	        if (hue1 > hue0 && hue1 - hue0 > 180) {
	          dh = hue1 - (hue0 + 360);
	        } else if (hue1 < hue0 && hue0 - hue1 > 180) {
	          dh = hue1 + 360 - hue0;
	        } else {
	          dh = hue1 - hue0;
	        }

	        hue = hue0 + f * dh;
	      } else if (!isNaN(hue0)) {
	        hue = hue0;

	        if ((lbv1 == 1 || lbv1 == 0) && m != 'hsv') {
	          sat = sat0;
	        }
	      } else if (!isNaN(hue1)) {
	        hue = hue1;

	        if ((lbv0 == 1 || lbv0 == 0) && m != 'hsv') {
	          sat = sat1;
	        }
	      } else {
	        hue = Number.NaN;
	      }

	      if (sat === undefined) {
	        sat = sat0 + f * (sat1 - sat0);
	      }

	      lbv = lbv0 + f * (lbv1 - lbv0);
	      return m === 'oklch' ? new Color$9([lbv, sat, hue], m) : new Color$9([hue, sat, lbv], m);
	    };

	    var interpolate_hsx$5 = _hsx;

	    var lch = function (col1, col2, f) {
	      return interpolate_hsx$5(col1, col2, f, 'lch');
	    }; // register interpolator


	    interpolator$1.lch = lch;
	    interpolator$1.hcl = lch;
	    var Color$8 = Color_1;

	    var num = function (col1, col2, f) {
	      var c1 = col1.num();
	      var c2 = col2.num();
	      return new Color$8(c1 + f * (c2 - c1), 'num');
	    }; // register interpolator


	    interpolator$1.num = num;
	    var interpolate_hsx$4 = _hsx;

	    var hcg = function (col1, col2, f) {
	      return interpolate_hsx$4(col1, col2, f, 'hcg');
	    }; // register interpolator


	    interpolator$1.hcg = hcg;
	    var interpolate_hsx$3 = _hsx;

	    var hsi = function (col1, col2, f) {
	      return interpolate_hsx$3(col1, col2, f, 'hsi');
	    }; // register interpolator


	    interpolator$1.hsi = hsi;
	    var interpolate_hsx$2 = _hsx;

	    var hsl = function (col1, col2, f) {
	      return interpolate_hsx$2(col1, col2, f, 'hsl');
	    }; // register interpolator


	    interpolator$1.hsl = hsl;
	    var interpolate_hsx$1 = _hsx;

	    var hsv = function (col1, col2, f) {
	      return interpolate_hsx$1(col1, col2, f, 'hsv');
	    }; // register interpolator


	    interpolator$1.hsv = hsv;
	    var Color$7 = Color_1;

	    var oklab = function (col1, col2, f) {
	      var xyz0 = col1.oklab();
	      var xyz1 = col2.oklab();
	      return new Color$7(xyz0[0] + f * (xyz1[0] - xyz0[0]), xyz0[1] + f * (xyz1[1] - xyz0[1]), xyz0[2] + f * (xyz1[2] - xyz0[2]), 'oklab');
	    }; // register interpolator


	    interpolator$1.oklab = oklab;
	    var interpolate_hsx = _hsx;

	    var oklch = function (col1, col2, f) {
	      return interpolate_hsx(col1, col2, f, 'oklch');
	    }; // register interpolator


	    interpolator$1.oklch = oklch;
	    var Color$6 = Color_1;
	    var clip_rgb$1 = utils.clip_rgb;
	    var pow$4 = Math.pow;
	    var sqrt$1 = Math.sqrt;
	    var PI$1 = Math.PI;
	    var cos$2 = Math.cos;
	    var sin$2 = Math.sin;
	    var atan2$1 = Math.atan2;

	    var average = function (colors, mode, weights) {
	      if (mode === void 0) mode = 'lrgb';
	      if (weights === void 0) weights = null;
	      var l = colors.length;

	      if (!weights) {
	        weights = Array.from(new Array(l)).map(function () {
	          return 1;
	        });
	      } // normalize weights


	      var k = l / weights.reduce(function (a, b) {
	        return a + b;
	      });
	      weights.forEach(function (w, i) {
	        weights[i] *= k;
	      }); // convert colors to Color objects

	      colors = colors.map(function (c) {
	        return new Color$6(c);
	      });

	      if (mode === 'lrgb') {
	        return _average_lrgb(colors, weights);
	      }

	      var first = colors.shift();
	      var xyz = first.get(mode);
	      var cnt = [];
	      var dx = 0;
	      var dy = 0; // initial color

	      for (var i = 0; i < xyz.length; i++) {
	        xyz[i] = (xyz[i] || 0) * weights[0];
	        cnt.push(isNaN(xyz[i]) ? 0 : weights[0]);

	        if (mode.charAt(i) === 'h' && !isNaN(xyz[i])) {
	          var A = xyz[i] / 180 * PI$1;
	          dx += cos$2(A) * weights[0];
	          dy += sin$2(A) * weights[0];
	        }
	      }

	      var alpha = first.alpha() * weights[0];
	      colors.forEach(function (c, ci) {
	        var xyz2 = c.get(mode);
	        alpha += c.alpha() * weights[ci + 1];

	        for (var i = 0; i < xyz.length; i++) {
	          if (!isNaN(xyz2[i])) {
	            cnt[i] += weights[ci + 1];

	            if (mode.charAt(i) === 'h') {
	              var A = xyz2[i] / 180 * PI$1;
	              dx += cos$2(A) * weights[ci + 1];
	              dy += sin$2(A) * weights[ci + 1];
	            } else {
	              xyz[i] += xyz2[i] * weights[ci + 1];
	            }
	          }
	        }
	      });

	      for (var i$1 = 0; i$1 < xyz.length; i$1++) {
	        if (mode.charAt(i$1) === 'h') {
	          var A$1 = atan2$1(dy / cnt[i$1], dx / cnt[i$1]) / PI$1 * 180;

	          while (A$1 < 0) {
	            A$1 += 360;
	          }

	          while (A$1 >= 360) {
	            A$1 -= 360;
	          }

	          xyz[i$1] = A$1;
	        } else {
	          xyz[i$1] = xyz[i$1] / cnt[i$1];
	        }
	      }

	      alpha /= l;
	      return new Color$6(xyz, mode).alpha(alpha > 0.99999 ? 1 : alpha, true);
	    };

	    var _average_lrgb = function (colors, weights) {
	      var l = colors.length;
	      var xyz = [0, 0, 0, 0];

	      for (var i = 0; i < colors.length; i++) {
	        var col = colors[i];
	        var f = weights[i] / l;
	        var rgb = col._rgb;
	        xyz[0] += pow$4(rgb[0], 2) * f;
	        xyz[1] += pow$4(rgb[1], 2) * f;
	        xyz[2] += pow$4(rgb[2], 2) * f;
	        xyz[3] += rgb[3] * f;
	      }

	      xyz[0] = sqrt$1(xyz[0]);
	      xyz[1] = sqrt$1(xyz[1]);
	      xyz[2] = sqrt$1(xyz[2]);

	      if (xyz[3] > 0.9999999) {
	        xyz[3] = 1;
	      }

	      return new Color$6(clip_rgb$1(xyz));
	    }; // minimal multi-purpose interface
	    // @requires utils color analyze


	    var chroma$4 = chroma_1;
	    var type$2 = utils.type;
	    var pow$3 = Math.pow;

	    var scale$2 = function (colors) {
	      // constructor
	      var _mode = 'rgb';

	      var _nacol = chroma$4('#ccc');

	      var _spread = 0; // const _fixed = false;

	      var _domain = [0, 1];
	      var _pos = [];
	      var _padding = [0, 0];
	      var _classes = false;
	      var _colors = [];
	      var _out = false;
	      var _min = 0;
	      var _max = 1;
	      var _correctLightness = false;
	      var _colorCache = {};
	      var _useCache = true;
	      var _gamma = 1; // private methods

	      var setColors = function (colors) {
	        colors = colors || ['#fff', '#000'];

	        if (colors && type$2(colors) === 'string' && chroma$4.brewer && chroma$4.brewer[colors.toLowerCase()]) {
	          colors = chroma$4.brewer[colors.toLowerCase()];
	        }

	        if (type$2(colors) === 'array') {
	          // handle single color
	          if (colors.length === 1) {
	            colors = [colors[0], colors[0]];
	          } // make a copy of the colors


	          colors = colors.slice(0); // convert to chroma classes

	          for (var c = 0; c < colors.length; c++) {
	            colors[c] = chroma$4(colors[c]);
	          } // auto-fill color position


	          _pos.length = 0;

	          for (var c$1 = 0; c$1 < colors.length; c$1++) {
	            _pos.push(c$1 / (colors.length - 1));
	          }
	        }

	        resetCache();
	        return _colors = colors;
	      };

	      var getClass = function (value) {
	        if (_classes != null) {
	          var n = _classes.length - 1;
	          var i = 0;

	          while (i < n && value >= _classes[i]) {
	            i++;
	          }

	          return i - 1;
	        }

	        return 0;
	      };

	      var tMapLightness = function (t) {
	        return t;
	      };

	      var tMapDomain = function (t) {
	        return t;
	      }; // const classifyValue = function(value) {
	      //     let val = value;
	      //     if (_classes.length > 2) {
	      //         const n = _classes.length-1;
	      //         const i = getClass(value);
	      //         const minc = _classes[0] + ((_classes[1]-_classes[0]) * (0 + (_spread * 0.5)));  // center of 1st class
	      //         const maxc = _classes[n-1] + ((_classes[n]-_classes[n-1]) * (1 - (_spread * 0.5)));  // center of last class
	      //         val = _min + ((((_classes[i] + ((_classes[i+1] - _classes[i]) * 0.5)) - minc) / (maxc-minc)) * (_max - _min));
	      //     }
	      //     return val;
	      // };


	      var getColor = function (val, bypassMap) {
	        var col, t;

	        if (bypassMap == null) {
	          bypassMap = false;
	        }

	        if (isNaN(val) || val === null) {
	          return _nacol;
	        }

	        if (!bypassMap) {
	          if (_classes && _classes.length > 2) {
	            // find the class
	            var c = getClass(val);
	            t = c / (_classes.length - 2);
	          } else if (_max !== _min) {
	            // just interpolate between min/max
	            t = (val - _min) / (_max - _min);
	          } else {
	            t = 1;
	          }
	        } else {
	          t = val;
	        } // domain map


	        t = tMapDomain(t);

	        if (!bypassMap) {
	          t = tMapLightness(t); // lightness correction
	        }

	        if (_gamma !== 1) {
	          t = pow$3(t, _gamma);
	        }

	        t = _padding[0] + t * (1 - _padding[0] - _padding[1]);
	        t = Math.min(1, Math.max(0, t));
	        var k = Math.floor(t * 10000);

	        if (_useCache && _colorCache[k]) {
	          col = _colorCache[k];
	        } else {
	          if (type$2(_colors) === 'array') {
	            //for i in [0.._pos.length-1]
	            for (var i = 0; i < _pos.length; i++) {
	              var p = _pos[i];

	              if (t <= p) {
	                col = _colors[i];
	                break;
	              }

	              if (t >= p && i === _pos.length - 1) {
	                col = _colors[i];
	                break;
	              }

	              if (t > p && t < _pos[i + 1]) {
	                t = (t - p) / (_pos[i + 1] - p);
	                col = chroma$4.interpolate(_colors[i], _colors[i + 1], t, _mode);
	                break;
	              }
	            }
	          } else if (type$2(_colors) === 'function') {
	            col = _colors(t);
	          }

	          if (_useCache) {
	            _colorCache[k] = col;
	          }
	        }

	        return col;
	      };

	      var resetCache = function () {
	        return _colorCache = {};
	      };

	      setColors(colors); // public interface

	      var f = function (v) {
	        var c = chroma$4(getColor(v));

	        if (_out && c[_out]) {
	          return c[_out]();
	        } else {
	          return c;
	        }
	      };

	      f.classes = function (classes) {
	        if (classes != null) {
	          if (type$2(classes) === 'array') {
	            _classes = classes;
	            _domain = [classes[0], classes[classes.length - 1]];
	          } else {
	            var d = chroma$4.analyze(_domain);

	            if (classes === 0) {
	              _classes = [d.min, d.max];
	            } else {
	              _classes = chroma$4.limits(d, 'e', classes);
	            }
	          }

	          return f;
	        }

	        return _classes;
	      };

	      f.domain = function (domain) {
	        if (!arguments.length) {
	          return _domain;
	        }

	        _min = domain[0];
	        _max = domain[domain.length - 1];
	        _pos = [];
	        var k = _colors.length;

	        if (domain.length === k && _min !== _max) {
	          // update positions
	          for (var i = 0, list = Array.from(domain); i < list.length; i += 1) {
	            var d = list[i];

	            _pos.push((d - _min) / (_max - _min));
	          }
	        } else {
	          for (var c = 0; c < k; c++) {
	            _pos.push(c / (k - 1));
	          }

	          if (domain.length > 2) {
	            // set domain map
	            var tOut = domain.map(function (d, i) {
	              return i / (domain.length - 1);
	            });
	            var tBreaks = domain.map(function (d) {
	              return (d - _min) / (_max - _min);
	            });

	            if (!tBreaks.every(function (val, i) {
	              return tOut[i] === val;
	            })) {
	              tMapDomain = function (t) {
	                if (t <= 0 || t >= 1) {
	                  return t;
	                }

	                var i = 0;

	                while (t >= tBreaks[i + 1]) {
	                  i++;
	                }

	                var f = (t - tBreaks[i]) / (tBreaks[i + 1] - tBreaks[i]);
	                var out = tOut[i] + f * (tOut[i + 1] - tOut[i]);
	                return out;
	              };
	            }
	          }
	        }

	        _domain = [_min, _max];
	        return f;
	      };

	      f.mode = function (_m) {
	        if (!arguments.length) {
	          return _mode;
	        }

	        _mode = _m;
	        resetCache();
	        return f;
	      };

	      f.range = function (colors, _pos) {
	        setColors(colors);
	        return f;
	      };

	      f.out = function (_o) {
	        _out = _o;
	        return f;
	      };

	      f.spread = function (val) {
	        if (!arguments.length) {
	          return _spread;
	        }

	        _spread = val;
	        return f;
	      };

	      f.correctLightness = function (v) {
	        if (v == null) {
	          v = true;
	        }

	        _correctLightness = v;
	        resetCache();

	        if (_correctLightness) {
	          tMapLightness = function (t) {
	            var L0 = getColor(0, true).lab()[0];
	            var L1 = getColor(1, true).lab()[0];
	            var pol = L0 > L1;
	            var L_actual = getColor(t, true).lab()[0];
	            var L_ideal = L0 + (L1 - L0) * t;
	            var L_diff = L_actual - L_ideal;
	            var t0 = 0;
	            var t1 = 1;
	            var max_iter = 20;

	            while (Math.abs(L_diff) > 1e-2 && max_iter-- > 0) {
	              (function () {
	                if (pol) {
	                  L_diff *= -1;
	                }

	                if (L_diff < 0) {
	                  t0 = t;
	                  t += (t1 - t) * 0.5;
	                } else {
	                  t1 = t;
	                  t += (t0 - t) * 0.5;
	                }

	                L_actual = getColor(t, true).lab()[0];
	                return L_diff = L_actual - L_ideal;
	              })();
	            }

	            return t;
	          };
	        } else {
	          tMapLightness = function (t) {
	            return t;
	          };
	        }

	        return f;
	      };

	      f.padding = function (p) {
	        if (p != null) {
	          if (type$2(p) === 'number') {
	            p = [p, p];
	          }

	          _padding = p;
	          return f;
	        } else {
	          return _padding;
	        }
	      };

	      f.colors = function (numColors, out) {
	        // If no arguments are given, return the original colors that were provided
	        if (arguments.length < 2) {
	          out = 'hex';
	        }

	        var result = [];

	        if (arguments.length === 0) {
	          result = _colors.slice(0);
	        } else if (numColors === 1) {
	          result = [f(0.5)];
	        } else if (numColors > 1) {
	          var dm = _domain[0];
	          var dd = _domain[1] - dm;
	          result = __range__(0, numColors, false).map(function (i) {
	            return f(dm + i / (numColors - 1) * dd);
	          });
	        } else {
	          // returns all colors based on the defined classes
	          colors = [];
	          var samples = [];

	          if (_classes && _classes.length > 2) {
	            for (var i = 1, end = _classes.length, asc = 1 <= end; asc ? i < end : i > end; asc ? i++ : i--) {
	              samples.push((_classes[i - 1] + _classes[i]) * 0.5);
	            }
	          } else {
	            samples = _domain;
	          }

	          result = samples.map(function (v) {
	            return f(v);
	          });
	        }

	        if (chroma$4[out]) {
	          result = result.map(function (c) {
	            return c[out]();
	          });
	        }

	        return result;
	      };

	      f.cache = function (c) {
	        if (c != null) {
	          _useCache = c;
	          return f;
	        } else {
	          return _useCache;
	        }
	      };

	      f.gamma = function (g) {
	        if (g != null) {
	          _gamma = g;
	          return f;
	        } else {
	          return _gamma;
	        }
	      };

	      f.nodata = function (d) {
	        if (d != null) {
	          _nacol = chroma$4(d);
	          return f;
	        } else {
	          return _nacol;
	        }
	      };

	      return f;
	    };

	    function __range__(left, right, inclusive) {
	      var range = [];
	      var ascending = left < right;
	      var end = !inclusive ? right : ascending ? right + 1 : right - 1;

	      for (var i = left; ascending ? i < end : i > end; ascending ? i++ : i--) {
	        range.push(i);
	      }

	      return range;
	    } //
	    // interpolates between a set of colors uzing a bezier spline
	    //
	    // @requires utils lab


	    var Color$5 = Color_1;
	    var scale$1 = scale$2; // nth row of the pascal triangle

	    var binom_row = function (n) {
	      var row = [1, 1];

	      for (var i = 1; i < n; i++) {
	        var newrow = [1];

	        for (var j = 1; j <= row.length; j++) {
	          newrow[j] = (row[j] || 0) + row[j - 1];
	        }

	        row = newrow;
	      }

	      return row;
	    };

	    var bezier = function (colors) {
	      var assign, assign$1, assign$2;
	      var I, lab0, lab1, lab2;
	      colors = colors.map(function (c) {
	        return new Color$5(c);
	      });

	      if (colors.length === 2) {
	        // linear interpolation
	        assign = colors.map(function (c) {
	          return c.lab();
	        }), lab0 = assign[0], lab1 = assign[1];

	        I = function (t) {
	          var lab = [0, 1, 2].map(function (i) {
	            return lab0[i] + t * (lab1[i] - lab0[i]);
	          });
	          return new Color$5(lab, 'lab');
	        };
	      } else if (colors.length === 3) {
	        // quadratic bezier interpolation
	        assign$1 = colors.map(function (c) {
	          return c.lab();
	        }), lab0 = assign$1[0], lab1 = assign$1[1], lab2 = assign$1[2];

	        I = function (t) {
	          var lab = [0, 1, 2].map(function (i) {
	            return (1 - t) * (1 - t) * lab0[i] + 2 * (1 - t) * t * lab1[i] + t * t * lab2[i];
	          });
	          return new Color$5(lab, 'lab');
	        };
	      } else if (colors.length === 4) {
	        // cubic bezier interpolation
	        var lab3;
	        assign$2 = colors.map(function (c) {
	          return c.lab();
	        }), lab0 = assign$2[0], lab1 = assign$2[1], lab2 = assign$2[2], lab3 = assign$2[3];

	        I = function (t) {
	          var lab = [0, 1, 2].map(function (i) {
	            return (1 - t) * (1 - t) * (1 - t) * lab0[i] + 3 * (1 - t) * (1 - t) * t * lab1[i] + 3 * (1 - t) * t * t * lab2[i] + t * t * t * lab3[i];
	          });
	          return new Color$5(lab, 'lab');
	        };
	      } else if (colors.length >= 5) {
	        // general case (degree n bezier)
	        var labs, row, n;
	        labs = colors.map(function (c) {
	          return c.lab();
	        });
	        n = colors.length - 1;
	        row = binom_row(n);

	        I = function (t) {
	          var u = 1 - t;
	          var lab = [0, 1, 2].map(function (i) {
	            return labs.reduce(function (sum, el, j) {
	              return sum + row[j] * Math.pow(u, n - j) * Math.pow(t, j) * el[i];
	            }, 0);
	          });
	          return new Color$5(lab, 'lab');
	        };
	      } else {
	        throw new RangeError("No point in running bezier with only one color.");
	      }

	      return I;
	    };

	    var bezier_1 = function (colors) {
	      var f = bezier(colors);

	      f.scale = function () {
	        return scale$1(f);
	      };

	      return f;
	    };
	    /*
	     * interpolates between a set of colors uzing a bezier spline
	     * blend mode formulas taken from http://www.venture-ware.com/kevin/coding/lets-learn-math-photoshop-blend-modes/
	     */


	    var chroma$3 = chroma_1;

	    var blend = function (bottom, top, mode) {
	      if (!blend[mode]) {
	        throw new Error('unknown blend mode ' + mode);
	      }

	      return blend[mode](bottom, top);
	    };

	    var blend_f = function (f) {
	      return function (bottom, top) {
	        var c0 = chroma$3(top).rgb();
	        var c1 = chroma$3(bottom).rgb();
	        return chroma$3.rgb(f(c0, c1));
	      };
	    };

	    var each = function (f) {
	      return function (c0, c1) {
	        var out = [];
	        out[0] = f(c0[0], c1[0]);
	        out[1] = f(c0[1], c1[1]);
	        out[2] = f(c0[2], c1[2]);
	        return out;
	      };
	    };

	    var normal = function (a) {
	      return a;
	    };

	    var multiply = function (a, b) {
	      return a * b / 255;
	    };

	    var darken = function (a, b) {
	      return a > b ? b : a;
	    };

	    var lighten = function (a, b) {
	      return a > b ? a : b;
	    };

	    var screen = function (a, b) {
	      return 255 * (1 - (1 - a / 255) * (1 - b / 255));
	    };

	    var overlay = function (a, b) {
	      return b < 128 ? 2 * a * b / 255 : 255 * (1 - 2 * (1 - a / 255) * (1 - b / 255));
	    };

	    var burn = function (a, b) {
	      return 255 * (1 - (1 - b / 255) / (a / 255));
	    };

	    var dodge = function (a, b) {
	      if (a === 255) {
	        return 255;
	      }

	      a = 255 * (b / 255) / (1 - a / 255);
	      return a > 255 ? 255 : a;
	    }; // # add = (a,b) ->
	    // #     if (a + b > 255) then 255 else a + b


	    blend.normal = blend_f(each(normal));
	    blend.multiply = blend_f(each(multiply));
	    blend.screen = blend_f(each(screen));
	    blend.overlay = blend_f(each(overlay));
	    blend.darken = blend_f(each(darken));
	    blend.lighten = blend_f(each(lighten));
	    blend.dodge = blend_f(each(dodge));
	    blend.burn = blend_f(each(burn)); // blend.add = blend_f(each(add));

	    var blend_1 = blend; // cubehelix interpolation
	    // based on D.A. Green "A colour scheme for the display of astronomical intensity images"
	    // http://astron-soc.in/bulletin/11June/289392011.pdf

	    var type$1 = utils.type;
	    var clip_rgb = utils.clip_rgb;
	    var TWOPI = utils.TWOPI;
	    var pow$2 = Math.pow;
	    var sin$1 = Math.sin;
	    var cos$1 = Math.cos;
	    var chroma$2 = chroma_1;

	    var cubehelix = function (start, rotations, hue, gamma, lightness) {
	      if (start === void 0) start = 300;
	      if (rotations === void 0) rotations = -1.5;
	      if (hue === void 0) hue = 1;
	      if (gamma === void 0) gamma = 1;
	      if (lightness === void 0) lightness = [0, 1];
	      var dh = 0,
	          dl;

	      if (type$1(lightness) === 'array') {
	        dl = lightness[1] - lightness[0];
	      } else {
	        dl = 0;
	        lightness = [lightness, lightness];
	      }

	      var f = function (fract) {
	        var a = TWOPI * ((start + 120) / 360 + rotations * fract);
	        var l = pow$2(lightness[0] + dl * fract, gamma);
	        var h = dh !== 0 ? hue[0] + fract * dh : hue;
	        var amp = h * l * (1 - l) / 2;
	        var cos_a = cos$1(a);
	        var sin_a = sin$1(a);
	        var r = l + amp * (-0.14861 * cos_a + 1.78277 * sin_a);
	        var g = l + amp * (-0.29227 * cos_a - 0.90649 * sin_a);
	        var b = l + amp * (+1.97294 * cos_a);
	        return chroma$2(clip_rgb([r * 255, g * 255, b * 255, 1]));
	      };

	      f.start = function (s) {
	        if (s == null) {
	          return start;
	        }

	        start = s;
	        return f;
	      };

	      f.rotations = function (r) {
	        if (r == null) {
	          return rotations;
	        }

	        rotations = r;
	        return f;
	      };

	      f.gamma = function (g) {
	        if (g == null) {
	          return gamma;
	        }

	        gamma = g;
	        return f;
	      };

	      f.hue = function (h) {
	        if (h == null) {
	          return hue;
	        }

	        hue = h;

	        if (type$1(hue) === 'array') {
	          dh = hue[1] - hue[0];

	          if (dh === 0) {
	            hue = hue[1];
	          }
	        } else {
	          dh = 0;
	        }

	        return f;
	      };

	      f.lightness = function (h) {
	        if (h == null) {
	          return lightness;
	        }

	        if (type$1(h) === 'array') {
	          lightness = h;
	          dl = h[1] - h[0];
	        } else {
	          lightness = [h, h];
	          dl = 0;
	        }

	        return f;
	      };

	      f.scale = function () {
	        return chroma$2.scale(f);
	      };

	      f.hue(hue);
	      return f;
	    };

	    var Color$4 = Color_1;
	    var digits = '0123456789abcdef';
	    var floor$1 = Math.floor;
	    var random = Math.random;

	    var random_1 = function () {
	      var code = '#';

	      for (var i = 0; i < 6; i++) {
	        code += digits.charAt(floor$1(random() * 16));
	      }

	      return new Color$4(code, 'hex');
	    };

	    var type = type$p;
	    var log = Math.log;
	    var pow$1 = Math.pow;
	    var floor = Math.floor;
	    var abs$1 = Math.abs;

	    var analyze = function (data, key) {
	      if (key === void 0) key = null;
	      var r = {
	        min: Number.MAX_VALUE,
	        max: Number.MAX_VALUE * -1,
	        sum: 0,
	        values: [],
	        count: 0
	      };

	      if (type(data) === 'object') {
	        data = Object.values(data);
	      }

	      data.forEach(function (val) {
	        if (key && type(val) === 'object') {
	          val = val[key];
	        }

	        if (val !== undefined && val !== null && !isNaN(val)) {
	          r.values.push(val);
	          r.sum += val;

	          if (val < r.min) {
	            r.min = val;
	          }

	          if (val > r.max) {
	            r.max = val;
	          }

	          r.count += 1;
	        }
	      });
	      r.domain = [r.min, r.max];

	      r.limits = function (mode, num) {
	        return limits(r, mode, num);
	      };

	      return r;
	    };

	    var limits = function (data, mode, num) {
	      if (mode === void 0) mode = 'equal';
	      if (num === void 0) num = 7;

	      if (type(data) == 'array') {
	        data = analyze(data);
	      }

	      var min = data.min;
	      var max = data.max;
	      var values = data.values.sort(function (a, b) {
	        return a - b;
	      });

	      if (num === 1) {
	        return [min, max];
	      }

	      var limits = [];

	      if (mode.substr(0, 1) === 'c') {
	        // continuous
	        limits.push(min);
	        limits.push(max);
	      }

	      if (mode.substr(0, 1) === 'e') {
	        // equal interval
	        limits.push(min);

	        for (var i = 1; i < num; i++) {
	          limits.push(min + i / num * (max - min));
	        }

	        limits.push(max);
	      } else if (mode.substr(0, 1) === 'l') {
	        // log scale
	        if (min <= 0) {
	          throw new Error('Logarithmic scales are only possible for values > 0');
	        }

	        var min_log = Math.LOG10E * log(min);
	        var max_log = Math.LOG10E * log(max);
	        limits.push(min);

	        for (var i$1 = 1; i$1 < num; i$1++) {
	          limits.push(pow$1(10, min_log + i$1 / num * (max_log - min_log)));
	        }

	        limits.push(max);
	      } else if (mode.substr(0, 1) === 'q') {
	        // quantile scale
	        limits.push(min);

	        for (var i$2 = 1; i$2 < num; i$2++) {
	          var p = (values.length - 1) * i$2 / num;
	          var pb = floor(p);

	          if (pb === p) {
	            limits.push(values[pb]);
	          } else {
	            // p > pb
	            var pr = p - pb;
	            limits.push(values[pb] * (1 - pr) + values[pb + 1] * pr);
	          }
	        }

	        limits.push(max);
	      } else if (mode.substr(0, 1) === 'k') {
	        // k-means clustering

	        /*
	        implementation based on
	        http://code.google.com/p/figue/source/browse/trunk/figue.js#336
	        simplified for 1-d input values
	        */
	        var cluster;
	        var n = values.length;
	        var assignments = new Array(n);
	        var clusterSizes = new Array(num);
	        var repeat = true;
	        var nb_iters = 0;
	        var centroids = null; // get seed values

	        centroids = [];
	        centroids.push(min);

	        for (var i$3 = 1; i$3 < num; i$3++) {
	          centroids.push(min + i$3 / num * (max - min));
	        }

	        centroids.push(max);

	        while (repeat) {
	          // assignment step
	          for (var j = 0; j < num; j++) {
	            clusterSizes[j] = 0;
	          }

	          for (var i$4 = 0; i$4 < n; i$4++) {
	            var value = values[i$4];
	            var mindist = Number.MAX_VALUE;
	            var best = void 0;

	            for (var j$1 = 0; j$1 < num; j$1++) {
	              var dist = abs$1(centroids[j$1] - value);

	              if (dist < mindist) {
	                mindist = dist;
	                best = j$1;
	              }

	              clusterSizes[best]++;
	              assignments[i$4] = best;
	            }
	          } // update centroids step


	          var newCentroids = new Array(num);

	          for (var j$2 = 0; j$2 < num; j$2++) {
	            newCentroids[j$2] = null;
	          }

	          for (var i$5 = 0; i$5 < n; i$5++) {
	            cluster = assignments[i$5];

	            if (newCentroids[cluster] === null) {
	              newCentroids[cluster] = values[i$5];
	            } else {
	              newCentroids[cluster] += values[i$5];
	            }
	          }

	          for (var j$3 = 0; j$3 < num; j$3++) {
	            newCentroids[j$3] *= 1 / clusterSizes[j$3];
	          } // check convergence


	          repeat = false;

	          for (var j$4 = 0; j$4 < num; j$4++) {
	            if (newCentroids[j$4] !== centroids[j$4]) {
	              repeat = true;
	              break;
	            }
	          }

	          centroids = newCentroids;
	          nb_iters++;

	          if (nb_iters > 200) {
	            repeat = false;
	          }
	        } // finished k-means clustering
	        // the next part is borrowed from gabrielflor.it


	        var kClusters = {};

	        for (var j$5 = 0; j$5 < num; j$5++) {
	          kClusters[j$5] = [];
	        }

	        for (var i$6 = 0; i$6 < n; i$6++) {
	          cluster = assignments[i$6];
	          kClusters[cluster].push(values[i$6]);
	        }

	        var tmpKMeansBreaks = [];

	        for (var j$6 = 0; j$6 < num; j$6++) {
	          tmpKMeansBreaks.push(kClusters[j$6][0]);
	          tmpKMeansBreaks.push(kClusters[j$6][kClusters[j$6].length - 1]);
	        }

	        tmpKMeansBreaks = tmpKMeansBreaks.sort(function (a, b) {
	          return a - b;
	        });
	        limits.push(tmpKMeansBreaks[0]);

	        for (var i$7 = 1; i$7 < tmpKMeansBreaks.length; i$7 += 2) {
	          var v = tmpKMeansBreaks[i$7];

	          if (!isNaN(v) && limits.indexOf(v) === -1) {
	            limits.push(v);
	          }
	        }
	      }

	      return limits;
	    };

	    var analyze_1 = {
	      analyze: analyze,
	      limits: limits
	    };
	    var Color$3 = Color_1;

	    var contrast = function (a, b) {
	      // WCAG contrast ratio
	      // see http://www.w3.org/TR/2008/REC-WCAG20-20081211/#contrast-ratiodef
	      a = new Color$3(a);
	      b = new Color$3(b);
	      var l1 = a.luminance();
	      var l2 = b.luminance();
	      return l1 > l2 ? (l1 + 0.05) / (l2 + 0.05) : (l2 + 0.05) / (l1 + 0.05);
	    };

	    var Color$2 = Color_1;
	    var sqrt = Math.sqrt;
	    var pow = Math.pow;
	    var min = Math.min;
	    var max = Math.max;
	    var atan2 = Math.atan2;
	    var abs = Math.abs;
	    var cos = Math.cos;
	    var sin = Math.sin;
	    var exp = Math.exp;
	    var PI = Math.PI;

	    var deltaE = function (a, b, Kl, Kc, Kh) {
	      if (Kl === void 0) Kl = 1;
	      if (Kc === void 0) Kc = 1;
	      if (Kh === void 0) Kh = 1; // Delta E (CIE 2000)
	      // see http://www.brucelindbloom.com/index.html?Eqn_DeltaE_CIE2000.html

	      var rad2deg = function (rad) {
	        return 360 * rad / (2 * PI);
	      };

	      var deg2rad = function (deg) {
	        return 2 * PI * deg / 360;
	      };

	      a = new Color$2(a);
	      b = new Color$2(b);
	      var ref = Array.from(a.lab());
	      var L1 = ref[0];
	      var a1 = ref[1];
	      var b1 = ref[2];
	      var ref$1 = Array.from(b.lab());
	      var L2 = ref$1[0];
	      var a2 = ref$1[1];
	      var b2 = ref$1[2];
	      var avgL = (L1 + L2) / 2;
	      var C1 = sqrt(pow(a1, 2) + pow(b1, 2));
	      var C2 = sqrt(pow(a2, 2) + pow(b2, 2));
	      var avgC = (C1 + C2) / 2;
	      var G = 0.5 * (1 - sqrt(pow(avgC, 7) / (pow(avgC, 7) + pow(25, 7))));
	      var a1p = a1 * (1 + G);
	      var a2p = a2 * (1 + G);
	      var C1p = sqrt(pow(a1p, 2) + pow(b1, 2));
	      var C2p = sqrt(pow(a2p, 2) + pow(b2, 2));
	      var avgCp = (C1p + C2p) / 2;
	      var arctan1 = rad2deg(atan2(b1, a1p));
	      var arctan2 = rad2deg(atan2(b2, a2p));
	      var h1p = arctan1 >= 0 ? arctan1 : arctan1 + 360;
	      var h2p = arctan2 >= 0 ? arctan2 : arctan2 + 360;
	      var avgHp = abs(h1p - h2p) > 180 ? (h1p + h2p + 360) / 2 : (h1p + h2p) / 2;
	      var T = 1 - 0.17 * cos(deg2rad(avgHp - 30)) + 0.24 * cos(deg2rad(2 * avgHp)) + 0.32 * cos(deg2rad(3 * avgHp + 6)) - 0.2 * cos(deg2rad(4 * avgHp - 63));
	      var deltaHp = h2p - h1p;
	      deltaHp = abs(deltaHp) <= 180 ? deltaHp : h2p <= h1p ? deltaHp + 360 : deltaHp - 360;
	      deltaHp = 2 * sqrt(C1p * C2p) * sin(deg2rad(deltaHp) / 2);
	      var deltaL = L2 - L1;
	      var deltaCp = C2p - C1p;
	      var sl = 1 + 0.015 * pow(avgL - 50, 2) / sqrt(20 + pow(avgL - 50, 2));
	      var sc = 1 + 0.045 * avgCp;
	      var sh = 1 + 0.015 * avgCp * T;
	      var deltaTheta = 30 * exp(-pow((avgHp - 275) / 25, 2));
	      var Rc = 2 * sqrt(pow(avgCp, 7) / (pow(avgCp, 7) + pow(25, 7)));
	      var Rt = -Rc * sin(2 * deg2rad(deltaTheta));
	      var result = sqrt(pow(deltaL / (Kl * sl), 2) + pow(deltaCp / (Kc * sc), 2) + pow(deltaHp / (Kh * sh), 2) + Rt * (deltaCp / (Kc * sc)) * (deltaHp / (Kh * sh)));
	      return max(0, min(100, result));
	    };

	    var Color$1 = Color_1; // simple Euclidean distance

	    var distance = function (a, b, mode) {
	      if (mode === void 0) mode = 'lab'; // Delta E (CIE 1976)
	      // see http://www.brucelindbloom.com/index.html?Equations.html

	      a = new Color$1(a);
	      b = new Color$1(b);
	      var l1 = a.get(mode);
	      var l2 = b.get(mode);
	      var sum_sq = 0;

	      for (var i in l1) {
	        var d = (l1[i] || 0) - (l2[i] || 0);
	        sum_sq += d * d;
	      }

	      return Math.sqrt(sum_sq);
	    };

	    var Color = Color_1;

	    var valid = function () {
	      var args = [],
	          len = arguments.length;

	      while (len--) args[len] = arguments[len];

	      try {
	        new (Function.prototype.bind.apply(Color, [null].concat(args)))();
	        return true;
	      } catch (e) {
	        return false;
	      }
	    }; // some pre-defined color scales:


	    var chroma$1 = chroma_1;
	    var scale = scale$2;
	    var scales = {
	      cool: function cool() {
	        return scale([chroma$1.hsl(180, 1, .9), chroma$1.hsl(250, .7, .4)]);
	      },
	      hot: function hot() {
	        return scale(['#000', '#f00', '#ff0', '#fff']).mode('rgb');
	      }
	    };
	    /**
	        ColorBrewer colors for chroma.js
	         Copyright (c) 2002 Cynthia Brewer, Mark Harrower, and The
	        Pennsylvania State University.
	         Licensed under the Apache License, Version 2.0 (the "License");
	        you may not use this file except in compliance with the License.
	        You may obtain a copy of the License at
	        http://www.apache.org/licenses/LICENSE-2.0
	         Unless required by applicable law or agreed to in writing, software distributed
	        under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
	        CONDITIONS OF ANY KIND, either express or implied. See the License for the
	        specific language governing permissions and limitations under the License.
	    */

	    var colorbrewer = {
	      // sequential
	      OrRd: ['#fff7ec', '#fee8c8', '#fdd49e', '#fdbb84', '#fc8d59', '#ef6548', '#d7301f', '#b30000', '#7f0000'],
	      PuBu: ['#fff7fb', '#ece7f2', '#d0d1e6', '#a6bddb', '#74a9cf', '#3690c0', '#0570b0', '#045a8d', '#023858'],
	      BuPu: ['#f7fcfd', '#e0ecf4', '#bfd3e6', '#9ebcda', '#8c96c6', '#8c6bb1', '#88419d', '#810f7c', '#4d004b'],
	      Oranges: ['#fff5eb', '#fee6ce', '#fdd0a2', '#fdae6b', '#fd8d3c', '#f16913', '#d94801', '#a63603', '#7f2704'],
	      BuGn: ['#f7fcfd', '#e5f5f9', '#ccece6', '#99d8c9', '#66c2a4', '#41ae76', '#238b45', '#006d2c', '#00441b'],
	      YlOrBr: ['#ffffe5', '#fff7bc', '#fee391', '#fec44f', '#fe9929', '#ec7014', '#cc4c02', '#993404', '#662506'],
	      YlGn: ['#ffffe5', '#f7fcb9', '#d9f0a3', '#addd8e', '#78c679', '#41ab5d', '#238443', '#006837', '#004529'],
	      Reds: ['#fff5f0', '#fee0d2', '#fcbba1', '#fc9272', '#fb6a4a', '#ef3b2c', '#cb181d', '#a50f15', '#67000d'],
	      RdPu: ['#fff7f3', '#fde0dd', '#fcc5c0', '#fa9fb5', '#f768a1', '#dd3497', '#ae017e', '#7a0177', '#49006a'],
	      Greens: ['#f7fcf5', '#e5f5e0', '#c7e9c0', '#a1d99b', '#74c476', '#41ab5d', '#238b45', '#006d2c', '#00441b'],
	      YlGnBu: ['#ffffd9', '#edf8b1', '#c7e9b4', '#7fcdbb', '#41b6c4', '#1d91c0', '#225ea8', '#253494', '#081d58'],
	      Purples: ['#fcfbfd', '#efedf5', '#dadaeb', '#bcbddc', '#9e9ac8', '#807dba', '#6a51a3', '#54278f', '#3f007d'],
	      GnBu: ['#f7fcf0', '#e0f3db', '#ccebc5', '#a8ddb5', '#7bccc4', '#4eb3d3', '#2b8cbe', '#0868ac', '#084081'],
	      Greys: ['#ffffff', '#f0f0f0', '#d9d9d9', '#bdbdbd', '#969696', '#737373', '#525252', '#252525', '#000000'],
	      YlOrRd: ['#ffffcc', '#ffeda0', '#fed976', '#feb24c', '#fd8d3c', '#fc4e2a', '#e31a1c', '#bd0026', '#800026'],
	      PuRd: ['#f7f4f9', '#e7e1ef', '#d4b9da', '#c994c7', '#df65b0', '#e7298a', '#ce1256', '#980043', '#67001f'],
	      Blues: ['#f7fbff', '#deebf7', '#c6dbef', '#9ecae1', '#6baed6', '#4292c6', '#2171b5', '#08519c', '#08306b'],
	      PuBuGn: ['#fff7fb', '#ece2f0', '#d0d1e6', '#a6bddb', '#67a9cf', '#3690c0', '#02818a', '#016c59', '#014636'],
	      Viridis: ['#440154', '#482777', '#3f4a8a', '#31678e', '#26838f', '#1f9d8a', '#6cce5a', '#b6de2b', '#fee825'],
	      // diverging
	      Spectral: ['#9e0142', '#d53e4f', '#f46d43', '#fdae61', '#fee08b', '#ffffbf', '#e6f598', '#abdda4', '#66c2a5', '#3288bd', '#5e4fa2'],
	      RdYlGn: ['#a50026', '#d73027', '#f46d43', '#fdae61', '#fee08b', '#ffffbf', '#d9ef8b', '#a6d96a', '#66bd63', '#1a9850', '#006837'],
	      RdBu: ['#67001f', '#b2182b', '#d6604d', '#f4a582', '#fddbc7', '#f7f7f7', '#d1e5f0', '#92c5de', '#4393c3', '#2166ac', '#053061'],
	      PiYG: ['#8e0152', '#c51b7d', '#de77ae', '#f1b6da', '#fde0ef', '#f7f7f7', '#e6f5d0', '#b8e186', '#7fbc41', '#4d9221', '#276419'],
	      PRGn: ['#40004b', '#762a83', '#9970ab', '#c2a5cf', '#e7d4e8', '#f7f7f7', '#d9f0d3', '#a6dba0', '#5aae61', '#1b7837', '#00441b'],
	      RdYlBu: ['#a50026', '#d73027', '#f46d43', '#fdae61', '#fee090', '#ffffbf', '#e0f3f8', '#abd9e9', '#74add1', '#4575b4', '#313695'],
	      BrBG: ['#543005', '#8c510a', '#bf812d', '#dfc27d', '#f6e8c3', '#f5f5f5', '#c7eae5', '#80cdc1', '#35978f', '#01665e', '#003c30'],
	      RdGy: ['#67001f', '#b2182b', '#d6604d', '#f4a582', '#fddbc7', '#ffffff', '#e0e0e0', '#bababa', '#878787', '#4d4d4d', '#1a1a1a'],
	      PuOr: ['#7f3b08', '#b35806', '#e08214', '#fdb863', '#fee0b6', '#f7f7f7', '#d8daeb', '#b2abd2', '#8073ac', '#542788', '#2d004b'],
	      // qualitative
	      Set2: ['#66c2a5', '#fc8d62', '#8da0cb', '#e78ac3', '#a6d854', '#ffd92f', '#e5c494', '#b3b3b3'],
	      Accent: ['#7fc97f', '#beaed4', '#fdc086', '#ffff99', '#386cb0', '#f0027f', '#bf5b17', '#666666'],
	      Set1: ['#e41a1c', '#377eb8', '#4daf4a', '#984ea3', '#ff7f00', '#ffff33', '#a65628', '#f781bf', '#999999'],
	      Set3: ['#8dd3c7', '#ffffb3', '#bebada', '#fb8072', '#80b1d3', '#fdb462', '#b3de69', '#fccde5', '#d9d9d9', '#bc80bd', '#ccebc5', '#ffed6f'],
	      Dark2: ['#1b9e77', '#d95f02', '#7570b3', '#e7298a', '#66a61e', '#e6ab02', '#a6761d', '#666666'],
	      Paired: ['#a6cee3', '#1f78b4', '#b2df8a', '#33a02c', '#fb9a99', '#e31a1c', '#fdbf6f', '#ff7f00', '#cab2d6', '#6a3d9a', '#ffff99', '#b15928'],
	      Pastel2: ['#b3e2cd', '#fdcdac', '#cbd5e8', '#f4cae4', '#e6f5c9', '#fff2ae', '#f1e2cc', '#cccccc'],
	      Pastel1: ['#fbb4ae', '#b3cde3', '#ccebc5', '#decbe4', '#fed9a6', '#ffffcc', '#e5d8bd', '#fddaec', '#f2f2f2']
	    }; // add lowercase aliases for case-insensitive matches

	    for (var i = 0, list = Object.keys(colorbrewer); i < list.length; i += 1) {
	      var key = list[i];
	      colorbrewer[key.toLowerCase()] = colorbrewer[key];
	    }

	    var colorbrewer_1 = colorbrewer;
	    var chroma = chroma_1; // feel free to comment out anything to rollup
	    // a smaller chroma.js built
	    // io --> convert colors
	    // operators --> modify existing Colors
	    // interpolators
	    // generators -- > create new colors

	    chroma.average = average;
	    chroma.bezier = bezier_1;
	    chroma.blend = blend_1;
	    chroma.cubehelix = cubehelix;
	    chroma.mix = chroma.interpolate = mix$1;
	    chroma.random = random_1;
	    chroma.scale = scale$2; // other utility methods

	    chroma.analyze = analyze_1.analyze;
	    chroma.contrast = contrast;
	    chroma.deltaE = deltaE;
	    chroma.distance = distance;
	    chroma.limits = analyze_1.limits;
	    chroma.valid = valid; // scale

	    chroma.scales = scales; // colors

	    chroma.colors = w3cx11_1;
	    chroma.brewer = colorbrewer_1;
	    var chroma_js = chroma;
	    return chroma_js;
	  });
	})(chroma$1);

	var chroma = chroma$1.exports;

	function getZones(from, to) {
	  let exclusions = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : [];

	  if (from > to) {
	    [from, to] = [to, from];
	  } // in exclusions from and to have to be defined


	  exclusions = exclusions.filter(exclusion => exclusion.from !== undefined && exclusion.to !== undefined);
	  exclusions = JSON.parse(JSON.stringify(exclusions)); // we ensure that from before to

	  exclusions.forEach(exclusion => {
	    if (exclusion.from > exclusion.to) {
	      [exclusion.to, exclusion.from] = [exclusion.from, exclusion.to];
	    }
	  });
	  exclusions.sort((a, b) => a.from - b.from); // we will rework the exclusions in order to remove overlap and outside range (from / to)

	  exclusions.forEach(exclusion => {
	    if (exclusion.from < from) exclusion.from = from;
	    if (exclusion.to > to) exclusion.to = to;
	  });

	  for (let i = 0; i < exclusions.length - 1; i++) {
	    if (exclusions[i].to > exclusions[i + 1].from) {
	      exclusions[i].to = exclusions[i + 1].from;
	    }
	  }

	  exclusions = exclusions.filter(exclusion => exclusion.from < exclusion.to);

	  if (!exclusions || exclusions.length === 0) {
	    return [{
	      from,
	      to
	    }];
	  }

	  let zones = [];
	  let currentFrom = from;

	  for (let exclusion of exclusions) {
	    if (currentFrom < exclusion.from) {
	      zones.push({
	        from: currentFrom,
	        to: exclusion.from
	      });
	    }

	    currentFrom = exclusion.to;
	  }

	  if (currentFrom < to) {
	    zones.push({
	      from: currentFrom,
	      to: to
	    });
	  }

	  return zones;
	}

	/**
	 * Filter an array x/y based on various criteria
	 * x points are expected to be sorted
	 *
	 * @param {object} points
	 * @param {object} [options={}]
	 * @param {array} [options.from]
	 * @param {array} [options.to]
	 * @param {array} [options.exclusions=[]]
	 * @return {{x: Array<number>, y: Array<number>}}
	 */

	function filterX(points) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    x,
	    y
	  } = points;
	  const {
	    from = x[0],
	    to = x[x.length - 1],
	    exclusions = []
	  } = options;
	  let zones = getZones(from, to, exclusions);
	  let currentZoneIndex = 0;
	  let newX = [];
	  let newY = [];
	  let position = 0;

	  while (position < x.length) {
	    if (x[position] <= zones[currentZoneIndex].to && x[position] >= zones[currentZoneIndex].from) {
	      newX.push(x[position]);
	      newY.push(y[position]);
	    } else {
	      if (x[position] > zones[currentZoneIndex].to) {
	        currentZoneIndex++;
	        if (!zones[currentZoneIndex]) break;
	      }
	    }

	    position++;
	  }

	  return {
	    x: newX,
	    y: newY
	  };
	}

	/**
	 * Retrieve a chart with autocorrelation color
	 * @param {SpectraProcessor} [spectraProcessor]
	 * @param {number} [index]
	 * @param {object} [options={}]
	 * @param {Array} [options.autocorrelation] precalculated autocorrelation {x,y}
	 * @param {Array} [options.maxDataPoints=]
	 * @param {Array} [options.ids] ids of the spectra to select, by default all
	 * @param {array} [options.xFilter.from]
	 * @param {array} [options.xFilter.to]
	 * @param {array} [options.xFilter.exclusions=[]]
	 */

	function getAutocorrelationChart(spectraProcessor, index) {
	  let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
	  const {
	    autocorrelation = spectraProcessor.getAutocorrelation(index, options),
	    xFilter,
	    ids
	  } = options;
	  let max = max$1(autocorrelation.y);
	  let min = min$1(autocorrelation.y);
	  let colorCallback = chroma.scale(['blue', 'cyan', 'yellow', 'red']).domain([min, max]).mode('lch');
	  let colorScale = autocorrelation.y.map(y => `rgb(${colorCallback(y).rgb().join()})`);
	  let mean = spectraProcessor.getMeanData({
	    ids
	  });

	  if (xFilter) {
	    mean = filterX({
	      x: mean.x,
	      y: mean.y
	    }, xFilter);
	  }

	  let colorSpectrum = {
	    type: 'color',
	    x: mean.x,
	    y: mean.y,
	    color: colorScale,
	    styles: {
	      unselected: {
	        lineWidth: 1,
	        lineStyle: 1
	      },
	      selected: {
	        lineWidth: 3,
	        lineStyle: 1
	      }
	    }
	  };
	  return colorSpectrum;
	}

	/**
	 * @private
	 * @param {*} spectra
	 */

	function getBoxPlotData(normalizedData) {
	  const {
	    x,
	    matrix
	  } = normalizedData;
	  let nbRow = matrix.length;
	  let nbColumn = matrix[0].length;
	  let result;
	  let aColumn = new Float64Array(nbRow);

	  for (let column = 0; column < nbColumn; column++) {
	    for (let row = 0; row < nbRow; row++) {
	      aColumn[row] = matrix[row][column];
	    }

	    let info = xBoxPlot(aColumn);

	    if (!result) {
	      result = {
	        x
	      };

	      for (let key in info) {
	        result[key] = [];
	      }
	    }

	    for (let key in info) {
	      result[key].push(info[key]);
	    }
	  }

	  return result;
	}

	function getBoxPlotChart(normalizedData) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    q13FillColor = '#000',
	    q13FillOpacity = 0.3,
	    q2StrokeColor = '',
	    q2StrokeWidth = 3,
	    minMaxFillColor = '#000',
	    minMaxFillOpacity = 0.15
	  } = options;
	  let annotations = [];
	  const boxPlotData = getBoxPlotData(normalizedData);

	  if (q13FillOpacity && q13FillColor) {
	    let q13 = [];

	    for (let i = 0; i < boxPlotData.x.length; i++) {
	      q13.push({
	        x: boxPlotData.x[i],
	        y: boxPlotData.Q1[i]
	      });
	    }

	    for (let i = boxPlotData.x.length - 1; i >= 0; i--) {
	      q13.push({
	        x: boxPlotData.x[i],
	        y: boxPlotData.Q3[i]
	      });
	    }

	    annotations.push({
	      type: 'polygon',
	      layer: 0,
	      properties: {
	        fillColor: q13FillColor,
	        fillOpacity: q13FillOpacity,
	        strokeWidth: 0.0000001,
	        position: q13
	      }
	    });
	  }

	  if (minMaxFillColor && minMaxFillOpacity) {
	    let minMax = [];

	    for (let i = 0; i < boxPlotData.x.length; i++) {
	      minMax.push({
	        x: boxPlotData.x[i],
	        y: boxPlotData.min[i]
	      });
	    }

	    for (let i = boxPlotData.x.length - 1; i >= 0; i--) {
	      minMax.push({
	        x: boxPlotData.x[i],
	        y: boxPlotData.max[i]
	      });
	    }

	    annotations.push({
	      type: 'polygon',
	      layer: 0,
	      properties: {
	        fillColor: minMaxFillColor,
	        fillOpacity: minMaxFillOpacity,
	        strokeWidth: 0.0000001,
	        strokeColor: '#FFF',
	        position: minMax
	      }
	    });
	  }

	  let colorSpectrum = {
	    type: 'color',
	    data: {
	      x: boxPlotData.x,
	      y: boxPlotData.Q2,
	      color: q2StrokeColor ? new Array(boxPlotData.x.length).fill(q2StrokeColor) : getColors(normalizedData.matrix)
	    },
	    styles: {
	      unselected: {
	        lineWidth: q2StrokeWidth,
	        lineStyle: 1
	      },
	      selected: {
	        lineWidth: q2StrokeWidth,
	        lineStyle: 1
	      }
	    },
	    annotations
	  };
	  return colorSpectrum;
	}

	function getColors(dataset) {
	  const matrix = new Matrix(dataset);
	  const stdevs = matrix.standardDeviation('column');
	  const min = min$1(stdevs);
	  const max = max$1(stdevs);
	  const colorCallback = chroma.scale(['blue', 'cyan', 'yellow', 'red']).domain([min, max]).mode('lch');
	  const colors = stdevs.map(y => `rgb(${colorCallback(y).rgb().join()})`);
	  return colors;
	}

	function addChartDataStyle(data, spectrum) {
	  data.styles = {
	    unselected: {
	      lineColor: spectrum.meta.color || 'darkgrey',
	      lineWidth: 1,
	      lineStyle: 1
	    },
	    selected: {
	      lineColor: spectrum.meta.color || 'darkgrey',
	      lineWidth: 3,
	      lineStyle: 1
	    }
	  };
	  data.label = spectrum.meta.id || spectrum.id;
	}

	/**
	 * Retrieve a chart with selected original data
	 * @param {object} [options={}]
	 * @param {Array} [options.ids] List of spectra ids, by default all
	 * @param {Array} [options.yFactor=1]
	 */

	function getChart(spectra) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    ids,
	    yFactor,
	    xFilter = {}
	  } = options;
	  let chart = {
	    data: []
	  };

	  for (let spectrum of spectra) {
	    if (!ids || ids.includes(spectrum.id)) {
	      let data = spectrum.getData({
	        yFactor,
	        xFilter
	      });
	      addChartDataStyle(data, spectrum);
	      chart.data.push(data);
	    }
	  }

	  return chart;
	}

	function getNormalizationAnnotations() {
	  let filter = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	  let {
	    exclusions = []
	  } = filter;
	  let annotations = [];
	  exclusions = exclusions.filter(exclusion => !exclusion.ignore);
	  annotations = exclusions.map(exclusion => {
	    let annotation = {
	      type: 'rect',
	      position: [{
	        x: exclusion.from,
	        y: '0px'
	      }, {
	        x: exclusion.to,
	        y: '2000px'
	      }],
	      strokeWidth: 0,
	      fillColor: 'rgba(255,255,224,1)'
	    };
	    return annotation;
	  });

	  if (filter.from !== undefined) {
	    annotations.push({
	      type: 'rect',
	      position: [{
	        x: Number.MIN_SAFE_INTEGER,
	        y: '0px'
	      }, {
	        x: filter.from,
	        y: '2000px'
	      }],
	      strokeWidth: 0,
	      fillColor: 'rgba(255,255,224,1)'
	    });
	  }

	  if (filter.to !== undefined) {
	    annotations.push({
	      type: 'rect',
	      position: [{
	        x: filter.to,
	        y: '0px'
	      }, {
	        x: Number.MAX_SAFE_INTEGER,
	        y: '2000px'
	      }],
	      strokeWidth: 0,
	      fillColor: 'rgba(255,255,224,1)'
	    });
	  }

	  return annotations;
	}

	/**
	 * @param {Array<Spectrum>} spectra
	 * @param {object} [options={}]
	 * @param {object} [options.xFilter={}]
	 * @param {array} [options.xFilter.from]
	 * @param {array} [options.xFilter.to]
	 * @param {array} [options.xFilter.exclusions=[]]
	 */

	function getNormalizedChart(spectra) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    xFilter
	  } = options;
	  let chart = {
	    data: []
	  };

	  for (let spectrum of spectra) {
	    let data = spectrum.normalized;

	    if (xFilter) {
	      data = filterX({
	        x: data.x,
	        y: data.y
	      }, xFilter);
	    }

	    addChartDataStyle(data, spectrum);
	    chart.data.push(data);
	  }

	  return chart;
	}

	/**
	 *
	 * @param {SpectraProcessor} spectraProcessor
	 * @param {object}   [options={}] scale spectra based on various parameters
	 * @param {Array}    [options.ids] ids of selected spectra
	 * @param {Array}    [options.filters=[]] Array of object containing {name:'', options:''}
	 * @param {object}   [options.scale={}] object containing the options for rescaling
	 * @param {string}   [options.scale.targetID=spectra[0].id]
	 * @param {string}   [options.scale.method='max'] min, max, integration, minMax
	 * @param {Array}    [options.scale.range] from - to to apply the method and rescale
	 * @param {boolean}  [options.scale.relative=false]
	 */

	function getPostProcessedChart(spectraProcessor) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  let scaled = spectraProcessor.getPostProcessedData(options);
	  let chart = {
	    data: []
	  };
	  if (!scaled || !scaled.matrix) return chart;

	  for (let i = 0; i < scaled.matrix.length; i++) {
	    let data = {
	      x: scaled.x,
	      y: Array.from(scaled.matrix[i]) // need to ensure not a typed array

	    };
	    addChartDataStyle(data, {
	      meta: scaled.meta[i],
	      id: scaled.ids[i]
	    });
	    chart.data.push(data);
	  }

	  return chart;
	}

	/**
	 * @private
	 * @param {*} spectra
	 * @param {object} [options={}]
	 * @param {Array} [options.ids] ids of selected spectra
	 */
	function getNormalizedData(spectra) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    ids
	  } = options;
	  let matrix = [];
	  let meta = [];
	  let currentIDs = [];
	  let x = [];

	  if (Array.isArray(spectra) && spectra.length > 0) {
	    for (let spectrum of spectra) {
	      if (!ids || ids.includes(spectrum.id)) {
	        currentIDs.push(spectrum.id);
	        matrix.push(spectrum.normalized.y);
	        meta.push(spectrum.meta);
	      }
	    }

	    x = spectra[0].normalized.x;
	  }

	  return {
	    ids: currentIDs,
	    matrix,
	    meta,
	    x
	  };
	}

	function getTrackAnnotation(spectra, index) {
	  let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
	  const {
	    ids,
	    showSpectrumID = true,
	    sortY = true,
	    limit = 20
	  } = options;
	  let annotations = [];
	  let normalized = getNormalizedData(spectra, {
	    ids
	  });
	  if (normalized.ids.length === 0) return annotations;
	  let line = 0; // header containing X coordinate

	  annotations.push({
	    type: 'line',
	    position: [{
	      x: `${70}px`,
	      y: `${15 + 15 * line}px`
	    }, {
	      x: `${85}px`,
	      y: `${15 + 15 * line}px`
	    }],
	    strokeWidth: 0.0000001,
	    label: {
	      size: 16,
	      text: `x: ${normalized.x[index].toPrecision(6)}`,
	      position: {
	        x: `${130}px`,
	        y: `${20 + 15 * line}px`
	      }
	    }
	  });
	  line++;
	  let peaks = [];

	  for (let i = 0; i < normalized.ids.length; i++) {
	    peaks.push({
	      id: normalized.ids[i],
	      meta: normalized.meta[i],
	      y: normalized.matrix[i][index]
	    });
	  }

	  if (sortY) {
	    peaks = peaks.sort((a, b) => b.y - a.y);
	  }

	  if (limit) {
	    peaks = peaks.slice(0, limit);
	  }

	  for (let {
	    id,
	    meta,
	    y
	  } of peaks) {
	    annotations.push({
	      type: 'line',
	      position: [{
	        x: `${70}px`,
	        y: `${15 + 15 * line}px`
	      }, {
	        x: `${85}px`,
	        y: `${15 + 15 * line}px`
	      }],
	      strokeColor: meta.color,
	      strokeWidth: 2,
	      label: {
	        text: `${y.toPrecision(4)}${showSpectrumID ? ` - ${id}` : ''}`,
	        position: {
	          x: `${90}px`,
	          y: `${20 + 15 * line}px`
	        }
	      }
	    });
	    line++;
	  }

	  return annotations;
	}

	/**
	 * @private
	 * @param {Array<object>} [metadata]
	 * @param {object} [options={}] scale spectra based on various parameters
	 * @param {string} [options.propertyName="category"]
	 */
	function getCategoriesStats(metadata) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    propertyName = 'category'
	  } = options;
	  let categories = {};
	  let classNumber = 0;

	  for (let metadatum of metadata) {
	    let value = metadatum[propertyName];

	    if (!categories[value]) {
	      categories[value] = {
	        classNumber: classNumber++,
	        counter: 0
	      };
	    }

	    categories[value].counter++;
	  }

	  return categories;
	}

	/**
	 * @private
	 * @param {Array<object>} [metadata]
	 * @param {object} [options={}] scale spectra based on various parameters
	 * @param {string} [options.propertyName="category"]
	 */
	function getClassLabels(metadata) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    propertyName = 'category'
	  } = options;
	  const categories = [];

	  for (const metadatum of metadata) {
	    categories.push(metadatum[propertyName]);
	  }

	  return categories;
	}

	/**
	 * @private
	 * @param {Array<object>} [metadata]
	 * @param {object} [options={}] scale spectra based on various parameters
	 * @param {string} [options.propertyName="category"]
	 */

	function getClasses(metadata) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    propertyName = 'category'
	  } = options;
	  const categoriesStats = getCategoriesStats(metadata, options);
	  const result = new Array(metadata.length);

	  for (let i = 0; i < metadata.length; i++) {
	    result[i] = categoriesStats[metadata[i][propertyName]].classNumber;
	  }

	  return result;
	}

	/**
	 * @private
	 * @param {*} spectra
	 * @param {object} options
	 */
	function getMetadata(spectra) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    ids
	  } = options;
	  let metadata = [];

	  if (Array.isArray(spectra) && spectra.length > 0) {
	    for (let spectrum of spectra) {
	      if (!ids || ids.includes(spectrum.id)) {
	        metadata.push(spectrum.meta);
	      }
	    }
	  }

	  return metadata;
	}

	/**
	 * Ensure that the data is string. If it is an ArrayBuffer it will be converted to string using TextDecoder.
	 * @param blob
	 * @param options
	 * @returns
	 */
	function ensureString(blob) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	  if (typeof blob === 'string') {
	    return blob;
	  }

	  if (ArrayBuffer.isView(blob) || blob instanceof ArrayBuffer) {
	    const {
	      encoding = guessEncoding(blob)
	    } = options;
	    const decoder = new TextDecoder(encoding);
	    return decoder.decode(blob);
	  }

	  throw new TypeError(`blob must be a string, ArrayBuffer or ArrayBufferView`);
	}

	function guessEncoding(blob) {
	  const uint8 = ArrayBuffer.isView(blob) ? new Uint8Array(blob.buffer, blob.byteOffset, blob.byteLength) : new Uint8Array(blob);

	  if (uint8.length >= 2) {
	    if (uint8[0] === 0xfe && uint8[1] === 0xff) {
	      return 'utf-16be';
	    }

	    if (uint8[0] === 0xff && uint8[1] === 0xfe) {
	      return 'utf-16le';
	    }
	  }

	  return 'utf-8';
	}

	/**
	 * Dynamically type a string
	 * @param {string} value String to dynamically type
	 * @returns {boolean|string|number}
	 */
	function parseString(value) {
	  if (value.length === 4 || value.length === 5) {
	    let lowercase = value.toLowerCase();
	    if (lowercase === 'true') return true;
	    if (lowercase === 'false') return false;
	  }

	  let number = Number(value);

	  if (number === 0 && !value.includes('0')) {
	    return value;
	  }

	  if (!Number.isNaN(number)) return number;
	  return value;
	}

	const GC_MS_FIELDS = ['TIC', '.RIC', 'SCANNUMBER'];
	function complexChromatogram(result) {
	  let spectra = result.spectra;
	  let length = spectra.length;
	  let chromatogram = {
	    times: new Array(length),
	    series: {
	      ms: {
	        dimension: 2,
	        data: new Array(length)
	      }
	    }
	  };
	  let existingGCMSFields = [];

	  for (let i = 0; i < GC_MS_FIELDS.length; i++) {
	    let label = convertMSFieldToLabel(GC_MS_FIELDS[i]);

	    if (spectra[0][label]) {
	      existingGCMSFields.push(label);
	      chromatogram.series[label] = {
	        dimension: 1,
	        data: new Array(length)
	      };
	    }
	  }

	  for (let i = 0; i < length; i++) {
	    let spectrum = spectra[i];
	    chromatogram.times[i] = spectrum.pageValue;

	    for (let j = 0; j < existingGCMSFields.length; j++) {
	      chromatogram.series[existingGCMSFields[j]].data[i] = Number(spectrum[existingGCMSFields[j]]);
	    }

	    if (spectrum.data) {
	      chromatogram.series.ms.data[i] = [spectrum.data.x, spectrum.data.y];
	    }
	  }

	  result.chromatogram = chromatogram;
	}
	function isMSField(canonicDataLabel) {
	  return GC_MS_FIELDS.indexOf(canonicDataLabel) !== -1;
	}
	function convertMSFieldToLabel(value) {
	  return value.toLowerCase().replace(/[^a-z0-9]/g, '');
	}

	function convertToFloatArray(stringArray) {
	  let floatArray = [];

	  for (let i = 0; i < stringArray.length; i++) {
	    floatArray.push(Number(stringArray[i]));
	  }

	  return floatArray;
	}

	function fastParseXYData(spectrum, value) {
	  // TODO need to deal with result
	  //  console.log(value);
	  // we check if deltaX is defined otherwise we calculate it
	  let yFactor = spectrum.yFactor;
	  let deltaX = spectrum.deltaX;
	  spectrum.isXYdata = true;
	  let currentData = {
	    x: [],
	    y: []
	  };
	  spectrum.data = currentData;
	  let currentX = spectrum.firstX;
	  let currentY = spectrum.firstY; // we skip the first line
	  //

	  let endLine = false;
	  let ascii;
	  let i = 0;

	  for (; i < value.length; i++) {
	    ascii = value.charCodeAt(i);

	    if (ascii === 13 || ascii === 10) {
	      endLine = true;
	    } else {
	      if (endLine) break;
	    }
	  } // we proceed taking the i after the first line


	  let newLine = true;
	  let isDifference = false;
	  let isLastDifference = false;
	  let lastDifference = 0;
	  let isDuplicate = false;
	  let inComment = false;
	  let currentValue = 0; // can be a difference or a duplicate

	  let lastValue = 0; // must be the real last value

	  let isNegative = false;
	  let inValue = false;
	  let skipFirstValue = false;
	  let decimalPosition = 0;

	  for (; i <= value.length; i++) {
	    if (i === value.length) ascii = 13;else ascii = value.charCodeAt(i);

	    if (inComment) {
	      // we should ignore the text if we are after $$
	      if (ascii === 13 || ascii === 10) {
	        newLine = true;
	        inComment = false;
	      }
	    } else {
	      // when is it a new value ?
	      // when it is not a digit, . or comma
	      // it is a number that is either new or we continue
	      if (ascii <= 57 && ascii >= 48) {
	        // a number
	        inValue = true;

	        if (decimalPosition > 0) {
	          currentValue += (ascii - 48) / Math.pow(10, decimalPosition++);
	        } else {
	          currentValue *= 10;
	          currentValue += ascii - 48;
	        }
	      } else if (ascii === 44 || ascii === 46) {
	        // a "," or "."
	        inValue = true;
	        decimalPosition++;
	      } else {
	        if (inValue) {
	          // need to process the previous value
	          if (newLine) {
	            newLine = false; // we don't check the X value
	            // console.log("NEW LINE",isDifference, lastDifference);
	            // if new line and lastDifference, the first value is just a check !
	            // that we don't check ...

	            if (isLastDifference) skipFirstValue = true;
	          } else {
	            // need to deal with duplicate and differences
	            if (skipFirstValue) {
	              skipFirstValue = false;
	            } else {
	              if (isDifference) {
	                lastDifference = isNegative ? 0 - currentValue : currentValue;
	                isLastDifference = true;
	                isDifference = false;
	              } else if (!isDuplicate) {
	                lastValue = isNegative ? 0 - currentValue : currentValue;
	              }

	              let duplicate = isDuplicate ? currentValue - 1 : 1;

	              for (let j = 0; j < duplicate; j++) {
	                if (isLastDifference) {
	                  currentY += lastDifference;
	                } else {
	                  currentY = lastValue;
	                }

	                currentData.x.push(currentX);
	                currentData.y.push(currentY * yFactor);
	                currentX += deltaX;
	              }
	            }
	          }

	          isNegative = false;
	          currentValue = 0;
	          decimalPosition = 0;
	          inValue = false;
	          isDuplicate = false;
	        } // positive SQZ digits @ A B C D E F G H I (ascii 64-73)


	        if (ascii < 74 && ascii > 63) {
	          inValue = true;
	          isLastDifference = false;
	          currentValue = ascii - 64;
	        } else if (ascii > 96 && ascii < 106) {
	          // negative SQZ digits a b c d e f g h i (ascii 97-105)
	          inValue = true;
	          isLastDifference = false;
	          currentValue = ascii - 96;
	          isNegative = true;
	        } else if (ascii === 115) {
	          // DUP digits S T U V W X Y Z s (ascii 83-90, 115)
	          inValue = true;
	          isDuplicate = true;
	          currentValue = 9;
	        } else if (ascii > 82 && ascii < 91) {
	          inValue = true;
	          isDuplicate = true;
	          currentValue = ascii - 82;
	        } else if (ascii > 73 && ascii < 83) {
	          // positive DIF digits % J K L M N O P Q R (ascii 37, 74-82)
	          inValue = true;
	          isDifference = true;
	          currentValue = ascii - 73;
	        } else if (ascii > 105 && ascii < 115) {
	          // negative DIF digits j k l m n o p q r (ascii 106-114)
	          inValue = true;
	          isDifference = true;
	          currentValue = ascii - 105;
	          isNegative = true;
	        } else if (ascii === 36 && value.charCodeAt(i + 1) === 36) {
	          // $ sign, we need to check the next one
	          inValue = true;
	          inComment = true;
	        } else if (ascii === 37) {
	          // positive DIF digits % J K L M N O P Q R (ascii 37, 74-82)
	          inValue = true;
	          isDifference = true;
	          currentValue = 0;
	          isNegative = false;
	        } else if (ascii === 45) {
	          // a "-"
	          // check if after there is a number, decimal or comma
	          let ascii2 = value.charCodeAt(i + 1);

	          if (ascii2 >= 48 && ascii2 <= 57 || ascii2 === 44 || ascii2 === 46) {
	            inValue = true;
	            if (!newLine) isLastDifference = false;
	            isNegative = true;
	          }
	        } else if (ascii === 13 || ascii === 10) {
	          newLine = true;
	          inComment = false;
	        } // and now analyse the details ... space or tabulation
	        // if "+" we just don't care

	      }
	    }
	  }
	}

	const removeCommentRegExp = /\$\$.*/;
	const peakTableSplitRegExp = /[,\t ]+/;
	function parsePeakTable(spectrum, value, result) {
	  spectrum.isPeaktable = true;

	  if (!spectrum.variables || Object.keys(spectrum.variables) === 2) {
	    parseXY$1(spectrum, value, result);
	  } else {
	    parseXYZ(spectrum, value, result);
	  } // we will add the data in the variables


	  if (spectrum.variables) {
	    for (let key in spectrum.variables) {
	      spectrum.variables[key].data = spectrum.data[key];
	    }
	  }
	}

	function parseXY$1(spectrum, value, result) {
	  let currentData = {
	    x: [],
	    y: []
	  };
	  spectrum.data = currentData; // counts for around 20% of the time

	  let lines = value.split(/,? *,?[;\r\n]+ */);

	  for (let i = 1; i < lines.length; i++) {
	    let values = lines[i].trim().replace(removeCommentRegExp, '').split(peakTableSplitRegExp);

	    if (values.length % 2 === 0) {
	      for (let j = 0; j < values.length; j = j + 2) {
	        // takes around 40% of the time to add and parse the 2 values nearly exclusively because of Number
	        currentData.x.push(Number(values[j]) * spectrum.xFactor);
	        currentData.y.push(Number(values[j + 1]) * spectrum.yFactor);
	      }
	    } else {
	      result.logs.push(`Format error: ${values}`);
	    }
	  }
	}

	function parseXYZ(spectrum, value, result) {
	  let currentData = {};
	  let variables = Object.keys(spectrum.variables);
	  let numberOfVariables = variables.length;
	  variables.forEach(variable => currentData[variable] = []);
	  spectrum.data = currentData; // counts for around 20% of the time

	  let lines = value.split(/,? *,?[;\r\n]+ */);

	  for (let i = 1; i < lines.length; i++) {
	    let values = lines[i].trim().replace(removeCommentRegExp, '').split(peakTableSplitRegExp);

	    if (values.length % numberOfVariables === 0) {
	      for (let j = 0; j < values.length; j++) {
	        // todo should try to find a xFactor (y, ...)
	        currentData[variables[j % numberOfVariables]].push(Number(values[j]));
	      }
	    } else {
	      result.logs.push(`Format error: ${values}`);
	    }
	  }
	}

	function parseXYA(spectrum, value) {
	  let removeSymbolRegExp = /(\(+|\)+|<+|>+|\s+)/g;
	  spectrum.isXYAdata = true;
	  let values;
	  let currentData = {
	    x: [],
	    y: []
	  };
	  spectrum.data = currentData;
	  let lines = value.split(/,? *,?[;\r\n]+ */);

	  for (let i = 1; i < lines.length; i++) {
	    values = lines[i].trim().replace(removeSymbolRegExp, '').split(',');
	    currentData.x.push(Number(values[0]));
	    currentData.y.push(Number(values[1]));
	  }
	}

	function median(input) {
	  if (!isAnyArray(input)) {
	    throw new TypeError('input must be an array');
	  }

	  if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }

	  return quickSelectMedian(input.slice());
	}

	function convertTo3DZ(spectra) {
	  let minZ = spectra[0].data.y[0];
	  let maxZ = minZ;
	  let ySize = spectra.length;
	  let xSize = spectra[0].data.x.length;
	  let z = new Array(ySize);

	  for (let i = 0; i < ySize; i++) {
	    z[i] = spectra[i].data.y;

	    for (let j = 0; j < xSize; j++) {
	      let value = z[i][j];
	      if (value < minZ) minZ = value;
	      if (value > maxZ) maxZ = value;
	    }
	  }

	  const firstX = spectra[0].data.x[0];
	  const lastX = spectra[0].data.x[spectra[0].data.x.length - 1]; // has to be -2 because it is a 1D array [x,y,x,y,...]

	  const firstY = spectra[0].pageValue;
	  const lastY = spectra[ySize - 1].pageValue; // Because the min / max value are the only information about the matrix if we invert
	  // min and max we need to invert the array

	  if (firstX > lastX) {
	    for (let spectrum of z) {
	      spectrum.reverse();
	    }
	  }

	  if (firstY > lastY) {
	    z.reverse();
	  }

	  const medians = [];

	  for (let i = 0; i < z.length; i++) {
	    const row = Float64Array.from(z[i]);

	    for (let i = 0; i < row.length; i++) {
	      if (row[i] < 0) row[i] = -row[i];
	    }

	    medians.push(median(row));
	  }

	  const median$1 = median(medians);
	  return {
	    z: z,
	    minX: Math.min(firstX, lastX),
	    maxX: Math.max(firstX, lastX),
	    minY: Math.min(firstY, lastY),
	    maxY: Math.max(firstY, lastY),
	    minZ: minZ,
	    maxZ: maxZ,
	    noise: median$1
	  };
	}

	function generateContourLines(zData, options) {
	  let noise = zData.noise;
	  let z = zData.z;
	  let povarHeight0, povarHeight1, povarHeight2, povarHeight3;
	  let isOver0, isOver1, isOver2, isOver3;
	  let nbSubSpectra = z.length;
	  let nbPovars = z[0].length;
	  let pAx, pAy, pBx, pBy;
	  let x0 = zData.minX;
	  let xN = zData.maxX;
	  let dx = (xN - x0) / (nbPovars - 1);
	  let y0 = zData.minY;
	  let yN = zData.maxY;
	  let dy = (yN - y0) / (nbSubSpectra - 1);
	  let minZ = zData.minZ;
	  let maxZ = zData.maxZ; // System.out.prvarln('y0 '+y0+' yN '+yN);
	  // -------------------------
	  // Povars attribution
	  //
	  // 0----1
	  // |  / |
	  // | /  |
	  // 2----3
	  //
	  // ---------------------d------

	  let iter = options.nbContourLevels * 2;
	  let contourLevels = new Array(iter);
	  let lineZValue;

	  for (let level = 0; level < iter; level++) {
	    // multiply by 2 for positif and negatif
	    let contourLevel = {};
	    contourLevels[level] = contourLevel;
	    let side = level % 2;
	    let factor = (maxZ - options.noiseMultiplier * noise) * Math.exp((level >> 1) - options.nbContourLevels);

	    if (side === 0) {
	      lineZValue = factor + options.noiseMultiplier * noise;
	    } else {
	      lineZValue = 0 - factor - options.noiseMultiplier * noise;
	    }

	    let lines = [];
	    contourLevel.zValue = lineZValue;
	    contourLevel.lines = lines;
	    if (lineZValue <= minZ || lineZValue >= maxZ) continue;

	    for (let iSubSpectra = 0; iSubSpectra < nbSubSpectra - 1; iSubSpectra++) {
	      let subSpectra = z[iSubSpectra];
	      let subSpectraAfter = z[iSubSpectra + 1];

	      for (let povar = 0; povar < nbPovars - 1; povar++) {
	        povarHeight0 = subSpectra[povar];
	        povarHeight1 = subSpectra[povar + 1];
	        povarHeight2 = subSpectraAfter[povar];
	        povarHeight3 = subSpectraAfter[povar + 1];
	        isOver0 = povarHeight0 > lineZValue;
	        isOver1 = povarHeight1 > lineZValue;
	        isOver2 = povarHeight2 > lineZValue;
	        isOver3 = povarHeight3 > lineZValue; // Example povar0 is over the plane and povar1 and
	        // povar2 are below, we find the varersections and add
	        // the segment

	        if (isOver0 !== isOver1 && isOver0 !== isOver2) {
	          pAx = povar + (lineZValue - povarHeight0) / (povarHeight1 - povarHeight0);
	          pAy = iSubSpectra;
	          pBx = povar;
	          pBy = iSubSpectra + (lineZValue - povarHeight0) / (povarHeight2 - povarHeight0);
	          lines.push(pAx * dx + x0);
	          lines.push(pAy * dy + y0);
	          lines.push(pBx * dx + x0);
	          lines.push(pBy * dy + y0);
	        } // remove push does not help !!!!


	        if (isOver3 !== isOver1 && isOver3 !== isOver2) {
	          pAx = povar + 1;
	          pAy = iSubSpectra + 1 - (lineZValue - povarHeight3) / (povarHeight1 - povarHeight3);
	          pBx = povar + 1 - (lineZValue - povarHeight3) / (povarHeight2 - povarHeight3);
	          pBy = iSubSpectra + 1;
	          lines.push(pAx * dx + x0);
	          lines.push(pAy * dy + y0);
	          lines.push(pBx * dx + x0);
	          lines.push(pBy * dy + y0);
	        } // test around the diagonal


	        if (isOver1 !== isOver2) {
	          pAx = (povar + 1 - (lineZValue - povarHeight1) / (povarHeight2 - povarHeight1)) * dx + x0;
	          pAy = (iSubSpectra + (lineZValue - povarHeight1) / (povarHeight2 - povarHeight1)) * dy + y0;

	          if (isOver1 !== isOver0) {
	            pBx = povar + 1 - (lineZValue - povarHeight1) / (povarHeight0 - povarHeight1);
	            pBy = iSubSpectra;
	            lines.push(pAx);
	            lines.push(pAy);
	            lines.push(pBx * dx + x0);
	            lines.push(pBy * dy + y0);
	          }

	          if (isOver2 !== isOver0) {
	            pBx = povar;
	            pBy = iSubSpectra + 1 - (lineZValue - povarHeight2) / (povarHeight0 - povarHeight2);
	            lines.push(pAx);
	            lines.push(pAy);
	            lines.push(pBx * dx + x0);
	            lines.push(pBy * dy + y0);
	          }

	          if (isOver1 !== isOver3) {
	            pBx = povar + 1;
	            pBy = iSubSpectra + (lineZValue - povarHeight1) / (povarHeight3 - povarHeight1);
	            lines.push(pAx);
	            lines.push(pAy);
	            lines.push(pBx * dx + x0);
	            lines.push(pBy * dy + y0);
	          }

	          if (isOver2 !== isOver3) {
	            pBx = povar + (lineZValue - povarHeight2) / (povarHeight3 - povarHeight2);
	            pBy = iSubSpectra + 1;
	            lines.push(pAx);
	            lines.push(pAy);
	            lines.push(pBx * dx + x0);
	            lines.push(pBy * dy + y0);
	          }
	        }
	      }
	    }
	  }

	  return {
	    minX: zData.minX,
	    maxX: zData.maxX,
	    minY: zData.minY,
	    maxY: zData.maxY,
	    segments: contourLevels
	  };
	}

	function add2D(result, options) {
	  let zData = convertTo3DZ(result.spectra);

	  if (!options.noContour) {
	    result.contourLines = generateContourLines(zData, options);
	    delete zData.z;
	  }

	  result.minMax = zData;
	}

	// sources:
	// https://en.wikipedia.org/wiki/Gyromagnetic_ratio
	// TODO: #13 can we have a better source and more digits ? @jwist
	const gyromagneticRatio = {
	  '1H': 267.52218744e6,
	  '2H': 41.065e6,
	  '3H': 285.3508e6,
	  '3He': -203.789e6,
	  '7Li': 103.962e6,
	  '13C': 67.28284e6,
	  '14N': 19.331e6,
	  '15N': -27.116e6,
	  '17O': -36.264e6,
	  '19F': 251.662e6,
	  '23Na': 70.761e6,
	  '27Al': 69.763e6,
	  '29Si': -53.19e6,
	  '31P': 108.291e6,
	  '57Fe': 8.681e6,
	  '63Cu': 71.118e6,
	  '67Zn': 16.767e6,
	  '129Xe': -73.997e6
	};

	function sum(input) {
	  if (!isAnyArray(input)) {
	    throw new TypeError('input must be an array');
	  }

	  if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }

	  var sumValue = 0;

	  for (var i = 0; i < input.length; i++) {
	    sumValue += input[i];
	  }

	  return sumValue;
	}

	function mean(input) {
	  return sum(input) / input.length;
	}

	function _typeof(obj) {
	  "@babel/helpers - typeof";

	  if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") {
	    _typeof = function (obj) {
	      return typeof obj;
	    };
	  } else {
	    _typeof = function (obj) {
	      return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj;
	    };
	  }

	  return _typeof(obj);
	}
	/**
	 * Fill an array with sequential numbers
	 * @param {Array<number>} [input] - optional destination array (if not provided a new array will be created)
	 * @param {object} [options={}]
	 * @param {number} [options.from=0] - first value in the array
	 * @param {number} [options.to=10] - last value in the array
	 * @param {number} [options.size=input.length] - size of the array (if not provided calculated from step)
	 * @param {number} [options.step] - if not provided calculated from size
	 * @return {Array<number>}
	 */


	function sequentialFill() {
	  var input = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];
	  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	  if (_typeof(input) === 'object' && !isAnyArray(input)) {
	    options = input;
	    input = [];
	  }

	  if (!isAnyArray(input)) {
	    throw new TypeError('input must be an array');
	  }

	  var _options = options,
	      _options$from = _options.from,
	      from = _options$from === void 0 ? 0 : _options$from,
	      _options$to = _options.to,
	      to = _options$to === void 0 ? 10 : _options$to,
	      _options$size = _options.size,
	      size = _options$size === void 0 ? input.length : _options$size,
	      step = _options.step;

	  if (size !== 0 && step) {
	    throw new Error('step is defined by the array size');
	  }

	  if (!size) {
	    if (step) {
	      size = Math.floor((to - from) / step) + 1;
	    } else {
	      size = to - from + 1;
	    }
	  }

	  if (!step && size) {
	    step = (to - from) / (size - 1);
	  }

	  if (Array.isArray(input)) {
	    // only works with normal array
	    input.length = 0;

	    for (var i = 0; i < size; i++) {
	      input.push(from);
	      from += step;
	    }
	  } else {
	    if (input.length !== size) {
	      throw new Error('sequentialFill typed array must have the correct length');
	    }

	    for (var _i = 0; _i < size; _i++) {
	      input[_i] = from;
	      from += step;
	    }
	  }

	  return input;
	}

	/**
	 * Apply Savitzky Golay algorithm
	 * @param [ys] Array of y values
	 * @param [xs] Array of X or deltaX
	 * @return  Array containing the new ys (same length)
	 */

	function sgg(ys, xs) {
	  let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
	  let {
	    windowSize = 9,
	    derivative = 0,
	    polynomial = 3
	  } = options;

	  if (windowSize % 2 === 0 || windowSize < 5 || !Number.isInteger(windowSize)) {
	    throw new RangeError('Invalid window size (should be odd and at least 5 integer number)');
	  }

	  if (!isAnyArray(ys)) {
	    throw new TypeError('Y values must be an array');
	  }

	  if (typeof xs === 'undefined') {
	    throw new TypeError('X must be defined');
	  }

	  if (windowSize > ys.length) {
	    throw new RangeError(`Window size is higher than the data length ${windowSize}>${ys.length}`);
	  }

	  if (derivative < 0 || !Number.isInteger(derivative)) {
	    throw new RangeError('Derivative should be a positive integer');
	  }

	  if (polynomial < 1 || !Number.isInteger(polynomial)) {
	    throw new RangeError('Polynomial should be a positive integer');
	  }

	  if (polynomial >= 6) {
	    // eslint-disable-next-line no-console
	    console.warn('You should not use polynomial grade higher than 5 if you are' + ' not sure that your data arises from such a model. Possible polynomial oscillation problems');
	  }

	  let half = Math.floor(windowSize / 2);
	  let np = ys.length;
	  let ans = new Float64Array(np);
	  let weights = fullWeights(windowSize, polynomial, derivative);
	  let hs = 0;
	  let constantH = true;

	  if (isAnyArray(xs)) {
	    constantH = false;
	  } else {
	    hs = Math.pow(xs, derivative);
	  } //For the borders


	  for (let i = 0; i < half; i++) {
	    let wg1 = weights[half - i - 1];
	    let wg2 = weights[half + i + 1];
	    let d1 = 0;
	    let d2 = 0;

	    for (let l = 0; l < windowSize; l++) {
	      d1 += wg1[l] * ys[l];
	      d2 += wg2[l] * ys[np - windowSize + l];
	    }

	    if (constantH) {
	      ans[half - i - 1] = d1 / hs;
	      ans[np - half + i] = d2 / hs;
	    } else {
	      hs = getHs(xs, half - i - 1, half, derivative);
	      ans[half - i - 1] = d1 / hs;
	      hs = getHs(xs, np - half + i, half, derivative);
	      ans[np - half + i] = d2 / hs;
	    }
	  } //For the internal points


	  let wg = weights[half];

	  for (let i = windowSize; i <= np; i++) {
	    let d = 0;

	    for (let l = 0; l < windowSize; l++) d += wg[l] * ys[l + i - windowSize];

	    if (!constantH) {
	      hs = getHs(xs, i - half - 1, half, derivative);
	    }

	    ans[i - half - 1] = d / hs;
	  }

	  return ans;
	}

	function getHs(h, center, half, derivative) {
	  let hs = 0;
	  let count = 0;

	  for (let i = center - half; i < center + half; i++) {
	    if (i >= 0 && i < h.length - 1) {
	      hs += h[i + 1] - h[i];
	      count++;
	    }
	  }

	  return Math.pow(hs / count, derivative);
	}

	function gramPoly(i, m, k, s) {
	  let Grampoly = 0;

	  if (k > 0) {
	    Grampoly = (4 * k - 2) / (k * (2 * m - k + 1)) * (i * gramPoly(i, m, k - 1, s) + s * gramPoly(i, m, k - 1, s - 1)) - (k - 1) * (2 * m + k) / (k * (2 * m - k + 1)) * gramPoly(i, m, k - 2, s);
	  } else {
	    if (k === 0 && s === 0) {
	      Grampoly = 1;
	    } else {
	      Grampoly = 0;
	    }
	  }

	  return Grampoly;
	}

	function genFact(a, b) {
	  let gf = 1;

	  if (a >= b) {
	    for (let j = a - b + 1; j <= a; j++) {
	      gf *= j;
	    }
	  }

	  return gf;
	}

	function weight(i, t, m, n, s) {
	  let sum = 0;

	  for (let k = 0; k <= n; k++) {
	    sum += (2 * k + 1) * (genFact(2 * m, k) / genFact(2 * m + k + 1, k + 1)) * gramPoly(i, m, k, 0) * gramPoly(t, m, k, s);
	  }

	  return sum;
	}
	/**
	 * @private
	 * @param m  Number of points
	 * @param n  Polynomial grade
	 * @param s  Derivative
	 */


	function fullWeights(m, n, s) {
	  let weights = new Array(m);
	  let np = Math.floor(m / 2);

	  for (let t = -np; t <= np; t++) {
	    weights[t + np] = new Float64Array(m);

	    for (let j = -np; j <= np; j++) {
	      weights[t + np][j + np] = weight(j, t, np, n, s);
	    }
	  }

	  return weights;
	}

	/**
	 * Correction of the x and y coordinates using a quadratic optimizations with the peak and its 3 closest neighbors to determine the true x,y values of the peak.
	 * This process is done in place and is very fast.
	 * @param data
	 * @param peaks
	 */
	function optimizeTop(data, peaks) {
	  const {
	    x,
	    y
	  } = data;

	  for (const peak of peaks) {
	    let currentIndex = peak.index; // The detected peak could be moved 1 or 2 units to left or right.

	    if (y[currentIndex - 1] >= y[currentIndex - 2] && y[currentIndex - 1] >= y[currentIndex]) {
	      currentIndex--;
	    } else {
	      if (y[currentIndex + 1] >= y[currentIndex] && y[currentIndex + 1] >= y[currentIndex + 2]) {
	        currentIndex++;
	      } else {
	        if (y[currentIndex - 2] >= y[currentIndex - 3] && y[currentIndex - 2] >= y[currentIndex - 1]) {
	          currentIndex -= 2;
	        } else {
	          if (y[currentIndex + 2] >= y[currentIndex + 1] && y[currentIndex + 2] >= y[currentIndex + 3]) {
	            currentIndex += 2;
	          }
	        }
	      }
	    } // interpolation to a sin() function


	    if (y[currentIndex - 1] > 0 && y[currentIndex + 1] > 0 && y[currentIndex] >= y[currentIndex - 1] && y[currentIndex] >= y[currentIndex + 1] && (y[currentIndex] !== y[currentIndex - 1] || y[currentIndex] !== y[currentIndex + 1])) {
	      let alpha = 20 * Math.log10(y[currentIndex - 1]);
	      let beta = 20 * Math.log10(y[currentIndex]);
	      let gamma = 20 * Math.log10(y[currentIndex + 1]);
	      let p = 0.5 * (alpha - gamma) / (alpha - 2 * beta + gamma);
	      peak.x = x[currentIndex] + (x[currentIndex] - x[currentIndex - 1]) * p;
	      peak.y = y[currentIndex] - 0.25 * (y[currentIndex - 1] - y[currentIndex + 1]) * p;
	    }
	  }
	}

	/**
	 * Global spectra deconvolution
	 * @param  data - Object data with x and y arrays. Values in x has to be growing
	 * @param {number} [options.broadRatio = 0.00] - If `broadRatio` is higher than 0, then all the peaks which second derivative
	 * smaller than `broadRatio * maxAbsSecondDerivative` will be marked with the soft mask equal to true.

	 */

	function gsd(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  let {
	    sgOptions = {
	      windowSize: 9,
	      polynomial: 3
	    },
	    noiseLevel,
	    smoothY = false,
	    maxCriteria = true,
	    minMaxRatio = 0.00025,
	    realTopDetection = false
	  } = options;
	  let {
	    x,
	    y
	  } = data;

	  if (!xIsMonotoneIncreasing(x)) {
	    throw new Error('GSD only accepts monotone increasing x values');
	  } //rescale;


	  y = y.slice(); // If the max difference between delta x is less than 5%, then,
	  // we can assume it to be equally spaced variable

	  let equallySpaced = xIsEquallySpaced(x);

	  if (noiseLevel === undefined) {
	    if (equallySpaced) {
	      const noiseInfo = xNoiseStandardDeviation(y);

	      if (maxCriteria) {
	        noiseLevel = noiseInfo.median + 1.5 * noiseInfo.sd;
	      } else {
	        noiseLevel = -noiseInfo.median + 1.5 * noiseInfo.sd;
	      }
	    } else {
	      noiseLevel = 0;
	    }
	  } else {
	    if (maxCriteria === false) {
	      noiseLevel *= -1;
	    }
	  }

	  if (maxCriteria === false) {
	    for (let i = 0; i < y.length; i++) {
	      y[i] *= -1;
	    }
	  }

	  if (noiseLevel !== undefined) {
	    for (let i = 0; i < y.length; i++) {
	      if (y[i] < noiseLevel) {
	        y[i] = noiseLevel;
	      }
	    }
	  }

	  let yData = y;
	  let dY, ddY;
	  const {
	    windowSize,
	    polynomial
	  } = sgOptions;

	  if (equallySpaced) {
	    if (smoothY) {
	      yData = sgg(y, x[1] - x[0], {
	        windowSize,
	        polynomial,
	        derivative: 0
	      });
	    }

	    dY = sgg(y, x[1] - x[0], {
	      windowSize,
	      polynomial,
	      derivative: 1
	    });
	    ddY = sgg(y, x[1] - x[0], {
	      windowSize,
	      polynomial,
	      derivative: 2
	    });
	  } else {
	    if (smoothY) {
	      yData = sgg(y, x, {
	        windowSize,
	        polynomial,
	        derivative: 0
	      });
	    }

	    dY = sgg(y, x, {
	      windowSize,
	      polynomial,
	      derivative: 1
	    });
	    ddY = sgg(y, x, {
	      windowSize,
	      polynomial,
	      derivative: 2
	    });
	  }

	  const minY = xMinValue(yData);
	  const maxY = xMaxValue(yData);
	  if (minY > maxY || minY === maxY) return [];
	  const yThreshold = minY + (maxY - minY) * minMaxRatio;
	  const dX = x[1] - x[0];
	  let lastMax = null;
	  let lastMin = null;
	  let minddY = [];
	  let intervalL = [];
	  let intervalR = []; // By the intermediate value theorem We cannot find 2 consecutive maximum or minimum

	  for (let i = 1; i < yData.length - 1; ++i) {
	    if (dY[i] < dY[i - 1] && dY[i] <= dY[i + 1] || dY[i] <= dY[i - 1] && dY[i] < dY[i + 1]) {
	      lastMin = {
	        x: x[i],
	        index: i
	      };

	      if (dX > 0 && lastMax !== null) {
	        intervalL.push(lastMax);
	        intervalR.push(lastMin);
	      }
	    } // Maximum in first derivative


	    if (dY[i] >= dY[i - 1] && dY[i] > dY[i + 1] || dY[i] > dY[i - 1] && dY[i] >= dY[i + 1]) {
	      lastMax = {
	        x: x[i],
	        index: i
	      };

	      if (dX < 0 && lastMin !== null) {
	        intervalL.push(lastMax);
	        intervalR.push(lastMin);
	      }
	    } // Minimum in second derivative


	    if (ddY[i] < ddY[i - 1] && ddY[i] < ddY[i + 1]) {
	      minddY.push(i);
	    }
	  }

	  let lastK = -1;
	  const peaks = [];

	  for (const minddYIndex of minddY) {
	    let deltaX = x[minddYIndex];
	    let possible = -1;
	    let k = lastK + 1;
	    let minDistance = Number.POSITIVE_INFINITY;
	    let currentDistance = 0;

	    while (possible === -1 && k < intervalL.length) {
	      currentDistance = Math.abs(deltaX - (intervalL[k].x + intervalR[k].x) / 2);

	      if (currentDistance < (intervalR[k].x - intervalL[k].x) / 2) {
	        possible = k;
	        lastK = k;
	      }

	      ++k; // Not getting closer?

	      if (currentDistance >= minDistance) {
	        break;
	      }

	      minDistance = currentDistance;
	    }

	    if (possible !== -1) {
	      if (yData[minddYIndex] > yThreshold) {
	        let width = Math.abs(intervalR[possible].x - intervalL[possible].x);
	        peaks.push({
	          x: deltaX,
	          y: yData[minddYIndex],
	          width: width,
	          index: minddYIndex,
	          ddY: ddY[minddYIndex],
	          inflectionPoints: {
	            from: intervalL[possible],
	            to: intervalR[possible]
	          }
	        });
	      }
	    }
	  }

	  if (realTopDetection) {
	    optimizeTop({
	      x,
	      y: yData
	    }, peaks);
	  }

	  peaks.forEach(peak => {
	    if (!maxCriteria) {
	      peak.y *= -1;
	      peak.ddY = peak.ddY * -1;
	    }
	  });
	  peaks.sort((a, b) => {
	    return a.x - b.x;
	  });
	  return peaks;
	}

	function postProcessingNMR(entriesFlat) {
	  // specific NMR functions
	  for (let entry of entriesFlat) {
	    let observeFrequency = 0;
	    let shiftOffsetVal = 0;

	    for (let spectrum of entry.spectra) {
	      if (entry.ntuples && entry.ntuples.symbol) {
	        if (!observeFrequency && spectrum.observeFrequency) {
	          observeFrequency = spectrum.observeFrequency;
	        }

	        if (!shiftOffsetVal && spectrum.shiftOffsetVal) {
	          shiftOffsetVal = spectrum.shiftOffsetVal;
	        }
	      } else {
	        observeFrequency = spectrum.observeFrequency;
	        shiftOffsetVal = spectrum.shiftOffsetVal;
	      }

	      if (observeFrequency) {
	        if (spectrum.xUnits && spectrum.xUnits.toUpperCase().includes('HZ')) {
	          spectrum.xUnits = 'PPM';
	          spectrum.xFactor = spectrum.xFactor / observeFrequency;
	          spectrum.firstX = spectrum.firstX / observeFrequency;
	          spectrum.lastX = spectrum.lastX / observeFrequency;
	          spectrum.deltaX = spectrum.deltaX / observeFrequency;

	          for (let i = 0; i < spectrum.data.x.length; i++) {
	            spectrum.data.x[i] /= observeFrequency;
	          }
	        }
	      }

	      if (shiftOffsetVal) {
	        let shift = spectrum.firstX - shiftOffsetVal;
	        spectrum.firstX = spectrum.firstX - shift;
	        spectrum.lastX = spectrum.lastX - shift;

	        for (let i = 0; i < spectrum.data.x.length; i++) {
	          spectrum.data.x[i] -= shift;
	        }
	      } // we will check if some nucleus are missing ...


	      if (entry.ntuples && entry.ntuples.nucleus && entry.ntuples.symbol) {
	        for (let i = 0; i < entry.ntuples.nucleus.length; i++) {
	          let symbol = entry.ntuples.symbol[i];
	          let nucleus = entry.ntuples.nucleus[i];

	          if (symbol.startsWith('F') && !nucleus) {
	            if (symbol === 'F1') {
	              // if F1 is defined we will use F2
	              if (entry.tmp.$NUC2) {
	                entry.ntuples.nucleus[i] = entry.tmp.$NUC2;
	              } else {
	                let f2index = entry.ntuples.symbol.indexOf('F2');

	                if (f2index && entry.ntuples.nucleus[f2index]) {
	                  entry.ntuples.nucleus[i] = entry.ntuples.nucleus[f2index];
	                }
	              }
	            }

	            if (symbol === 'F2') entry.ntuples.nucleus[i] = entry.tmp.$NUC1;
	          }

	          if (symbol === 'F2') {
	            entry.yType = entry.ntuples.nucleus[0];
	          }
	        }
	      }

	      if (observeFrequency && entry.ntuples && entry.ntuples.symbol && entry.ntuples.nucleus) {
	        let unit = '';
	        let pageSymbolIndex = entry.ntuples.symbol.indexOf(spectrum.pageSymbol);

	        if (entry.ntuples.units && entry.ntuples.units[pageSymbolIndex]) {
	          unit = entry.ntuples.units[pageSymbolIndex];
	        }

	        if (unit !== 'PPM') {
	          if (pageSymbolIndex !== 0) {
	            throw Error('Not sure about this ntuples format');
	          }

	          let ratio0 = gyromagneticRatio[entry.ntuples.nucleus[0]];
	          let ratio1 = gyromagneticRatio[entry.ntuples.nucleus[1]];

	          if (!ratio0 || !ratio1) {
	            throw Error('Problem with determination of gyromagnetic ratio');
	          }

	          let ratio = ratio0 / ratio1 * observeFrequency;
	          spectrum.pageValue /= ratio;
	        }
	      }
	    }
	  }
	}

	function profiling(result, action, options) {
	  if (result.profiling) {
	    result.profiling.push({
	      action,
	      time: Date.now() - options.start
	    });
	  }
	}

	function simpleChromatogram(result) {
	  let data = result.spectra[0].data;
	  result.chromatogram = {
	    times: data.x.slice(),
	    series: {
	      intensity: {
	        dimension: 1,
	        data: data.y.slice()
	      }
	    }
	  };
	}

	function postProcessing(entriesFlat, result, options) {
	  // converting Hz to ppm
	  postProcessingNMR(entriesFlat);

	  for (let entry of entriesFlat) {
	    if (Object.keys(entry.ntuples).length > 0) {
	      let newNtuples = [];
	      let keys = Object.keys(entry.ntuples);

	      for (let i = 0; i < keys.length; i++) {
	        let key = keys[i];
	        let values = entry.ntuples[key];

	        for (let j = 0; j < values.length; j++) {
	          if (!newNtuples[j]) newNtuples[j] = {};
	          newNtuples[j][key] = values[j];
	        }
	      }

	      entry.ntuples = newNtuples;
	    }

	    if (entry.twoD && options.wantXY) {
	      add2D(entry, options);
	      profiling(result, 'Finished countour plot calculation', options);

	      if (!options.keepSpectra) {
	        delete entry.spectra;
	      }
	    } // maybe it is a GC (HPLC) / MS. In this case we add a new format


	    if (options.chromatogram) {
	      if (entry.spectra.length > 1) {
	        complexChromatogram(entry);
	      } else {
	        simpleChromatogram(entry);
	      }

	      profiling(result, 'Finished chromatogram calculation', options);
	    }

	    delete entry.tmp;
	  }
	}

	function prepareNtuplesDatatable(currentEntry, spectrum, kind) {
	  let xIndex = -1;
	  let yIndex = -1;
	  let firstVariable = '';
	  let secondVariable = '';

	  if (kind.indexOf('++') > 0) {
	    firstVariable = kind.replace(/.*\(([a-zA-Z0-9]+)\+\+.*/, '$1');
	    secondVariable = kind.replace(/.*\.\.([a-zA-Z0-9]+).*/, '$1');
	  } else {
	    kind = kind.replace(/[^a-zA-Z]/g, '');
	    firstVariable = kind.charAt(0);
	    secondVariable = kind.charAt(1);
	    spectrum.variables = {};

	    for (let symbol of kind) {
	      let lowerCaseSymbol = symbol.toLowerCase();
	      let index = currentEntry.ntuples.symbol.indexOf(symbol);
	      if (index === -1) throw Error(`Symbol undefined: ${symbol}`);
	      spectrum.variables[lowerCaseSymbol] = {};

	      for (let key in currentEntry.ntuples) {
	        if (currentEntry.ntuples[key][index]) {
	          spectrum.variables[lowerCaseSymbol][key.replace(/^var/, '')] = currentEntry.ntuples[key][index];
	        }
	      }
	    }
	  }

	  xIndex = currentEntry.ntuples.symbol.indexOf(firstVariable);
	  yIndex = currentEntry.ntuples.symbol.indexOf(secondVariable);
	  if (xIndex === -1) xIndex = 0;
	  if (yIndex === -1) yIndex = 0;

	  if (currentEntry.ntuples.first) {
	    if (currentEntry.ntuples.first.length > xIndex) {
	      spectrum.firstX = currentEntry.ntuples.first[xIndex];
	    }

	    if (currentEntry.ntuples.first.length > yIndex) {
	      spectrum.firstY = currentEntry.ntuples.first[yIndex];
	    }
	  }

	  if (currentEntry.ntuples.last) {
	    if (currentEntry.ntuples.last.length > xIndex) {
	      spectrum.lastX = currentEntry.ntuples.last[xIndex];
	    }

	    if (currentEntry.ntuples.last.length > yIndex) {
	      spectrum.lastY = currentEntry.ntuples.last[yIndex];
	    }
	  }

	  if (currentEntry.ntuples.vardim && currentEntry.ntuples.vardim.length > xIndex) {
	    spectrum.nbPoints = currentEntry.ntuples.vardim[xIndex];
	  }

	  if (currentEntry.ntuples.factor) {
	    if (currentEntry.ntuples.factor.length > xIndex) {
	      spectrum.xFactor = currentEntry.ntuples.factor[xIndex];
	    }

	    if (currentEntry.ntuples.factor.length > yIndex) {
	      spectrum.yFactor = currentEntry.ntuples.factor[yIndex];
	    }
	  }

	  if (currentEntry.ntuples.units) {
	    if (currentEntry.ntuples.units.length > xIndex) {
	      if (currentEntry.ntuples.varname && currentEntry.ntuples.varname[xIndex]) {
	        spectrum.xUnits = `${currentEntry.ntuples.varname[xIndex]} [${currentEntry.ntuples.units[xIndex]}]`;
	      } else {
	        spectrum.xUnits = currentEntry.ntuples.units[xIndex];
	      }
	    }

	    if (currentEntry.ntuples.units.length > yIndex) {
	      if (currentEntry.ntuples.varname && currentEntry.ntuples.varname[yIndex]) {
	        spectrum.yUnits = `${currentEntry.ntuples.varname[yIndex]} [${currentEntry.ntuples.units[yIndex]}]`;
	      } else {
	        spectrum.yUnits = currentEntry.ntuples.units[yIndex];
	      }
	    }
	  }
	}

	function prepareSpectrum(spectrum) {
	  if (!spectrum.xFactor) spectrum.xFactor = 1;
	  if (!spectrum.yFactor) spectrum.yFactor = 1;
	}

	const ntuplesSeparatorRegExp = /[ \t]*,[ \t]*/;
	const defaultOptions = {
	  keepRecordsRegExp: /^$/,
	  canonicDataLabels: true,
	  canonicMetadataLabels: false,
	  dynamicTyping: true,
	  withoutXY: false,
	  chromatogram: false,
	  keepSpectra: false,
	  noContour: false,
	  nbContourLevels: 7,
	  noiseMultiplier: 5,
	  profiling: false
	};
	/**
	 *
	 * @typedef {object} ConvertOptions
	 * @property {RegExp} [options.keepRecordsRegExp=/^$/] - By default we don't keep meta information.
	 * @property {boolean} [options.canonicDataLabels=true] - Canonize the Labels (uppercase without symbol).
	 * @property {boolean} [options.canonicMetadataLabels=false] - Canonize the metadata Labels (uppercase without symbol).
	 * @property {boolean} [options.dynamicTyping=false] - Convert numbers to Number.
	 * @property {boolean} [options.withoutXY=false] - Remove the XY data.
	 * @property {boolean} [options.chromatogram=false] - Special post-processing for GC / HPLC / MS.
	 * @property {boolean} [options.keepSpectra=false] - Force to keep the spectra in case of 2D.
	 * @property {boolean} [options.noContour=false] - Don't calculate countour in case of 2D.
	 * @property {number} [options.nbContourLevels=7] - Number of positive / negative contour levels to calculate.
	 * @property {number} [options.noiseMultiplier=5] - Define for 2D the level as 5 times the median as default.
	 * @property {boolean} [options.profiling=false] - Add profiling information.
	 */

	/**
	 *
	 * @typedef {object} Ntuples
	 * @property {string[]} [varname]
	 * @property {string[]} [symbol]
	 * @property {string[]} [vartype]
	 * @property {string[]} [varform]
	 * @property {number[]} [vardim]
	 * @property {string[]} [units]
	 * @property {number[]} [factor]
	 * @property {number[]} [first]
	 * @property {number[]} [last]
	 * @property {number[]} [min]
	 * @property {number[]} [max]
	 * @property {string[]} [nucleus]
	 */

	/**
	 * @typedef { Record<string, any> } Spectrum
	 * @property {Record<string, number[]>} [data]
	 * @property {number} [firstX]
	 * @property {number} [lastX]
	 * @property {number} [deltaX]
	 * @property {number} [yFactor]
	 * @property {number} [xFactor]
	 * @property {number} [nbPoints]
	 */

	/**
	 *
	 * @typedef {object} Entry
	 * @property {Spectrum[]} spectra
	 * @property {Ntuples} ntuples
	 * @property {object} meta
	 * @property {object} tmp
	 * @property {string} [title]
	 * @property {string} [dataType]
	 * @property {string} [dataClass]
	 * @property {boolean} [twoD]
	 */

	/**
	 *
	 * @typedef { object } ConvertResult
	 * @property { object[] | boolean } profiling
	 * @property { string[] } logs
	 * @property { object[] } entries
	 * @property { Entry[] } flatten
	 */

	/**
	 * Parse a jcamp.
	 *
	 * @param {string|ArrayBuffer|Uint8Array} jcamp
	 * @param {ConvertOptions} [options]
	 * @returns {ConvertResult}
	 */

	function convert(jcamp) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  jcamp = ensureString(jcamp);
	  options = { ...defaultOptions,
	    ...options
	  };
	  options.wantXY = !options.withoutXY;
	  options.start = Date.now();
	  let entriesFlat = [];
	  let result = {
	    profiling: options.profiling ? [] : false,
	    logs: [],
	    entries: []
	  };
	  let tmpResult = {
	    children: []
	  };
	  let currentEntry = tmpResult;
	  let parentsStack = [];
	  let spectrum = {};

	  if (typeof jcamp !== 'string') {
	    throw new TypeError('the JCAMP should be a string');
	  }

	  profiling(result, 'Before split to LDRS', options);
	  let ldrs = jcamp.replace(/[\r\n]+##/g, '\n##').split('\n##');
	  profiling(result, 'Split to LDRS', options);
	  if (ldrs[0]) ldrs[0] = ldrs[0].replace(/^[\r\n ]*##/, '');

	  for (let ldr of ldrs) {
	    // This is a new LDR
	    let position = ldr.indexOf('=');
	    let dataLabel = position > 0 ? ldr.substring(0, position) : ldr;
	    let dataValue = position > 0 ? ldr.substring(position + 1).trim() : '';
	    let canonicDataLabel = dataLabel.replace(/[_ -]/g, '').toUpperCase();

	    if (canonicDataLabel === 'DATATABLE') {
	      let endLine = dataValue.indexOf('\n');
	      if (endLine === -1) endLine = dataValue.indexOf('\r');

	      if (endLine > 0) {
	        // ##DATA TABLE= (X++(I..I)), XYDATA
	        // We need to find the variables
	        let infos = dataValue.substring(0, endLine).split(/[ ,;\t]+/);
	        prepareNtuplesDatatable(currentEntry, spectrum, infos[0]);
	        spectrum.datatable = infos[0];

	        if (infos[1] && infos[1].indexOf('PEAKS') > -1) {
	          canonicDataLabel = 'PEAKTABLE';
	        } else if (infos[1] && (infos[1].indexOf('XYDATA') || infos[0].indexOf('++') > 0)) {
	          canonicDataLabel = 'XYDATA';
	          spectrum.deltaX = (spectrum.lastX - spectrum.firstX) / (spectrum.nbPoints - 1);
	        }
	      }
	    }

	    if (canonicDataLabel === 'XYDATA') {
	      if (options.wantXY) {
	        prepareSpectrum(spectrum); // well apparently we should still consider it is a PEAK TABLE if there are no '++' after

	        if (dataValue.match(/.*\+\+.*/)) {
	          // ex: (X++(Y..Y))
	          spectrum.deltaX = (spectrum.lastX - spectrum.firstX) / (spectrum.nbPoints - 1);
	          fastParseXYData(spectrum, dataValue);
	        } else {
	          parsePeakTable(spectrum, dataValue, result);
	        }

	        currentEntry.spectra.push(spectrum);
	        spectrum = {};
	      }

	      continue;
	    } else if (canonicDataLabel === 'PEAKTABLE') {
	      if (options.wantXY) {
	        prepareSpectrum(spectrum);
	        parsePeakTable(spectrum, dataValue, result);
	        currentEntry.spectra.push(spectrum);
	        spectrum = {};
	      }

	      continue;
	    }

	    if (canonicDataLabel === 'PEAKASSIGNMENTS') {
	      if (options.wantXY) {
	        if (dataValue.match(/.*(XYA).*/)) {
	          // ex: (XYA)
	          parseXYA(spectrum, dataValue);
	        }

	        currentEntry.spectra.push(spectrum);
	        spectrum = {};
	      }

	      continue;
	    }

	    if (canonicDataLabel === 'TITLE') {
	      let parentEntry = currentEntry;

	      if (!parentEntry.children) {
	        parentEntry.children = [];
	      }

	      currentEntry = {
	        spectra: [],
	        ntuples: {},
	        info: {},
	        meta: {},
	        tmp: {} // tmp information we need to keep for postprocessing

	      };
	      parentEntry.children.push(currentEntry);
	      parentsStack.push(parentEntry);
	      entriesFlat.push(currentEntry);
	      currentEntry.title = dataValue;
	    } else if (canonicDataLabel === 'DATATYPE') {
	      currentEntry.dataType = dataValue;

	      if (dataValue.match(/(^nd|\snd\s)/i)) {
	        currentEntry.twoD = true;
	      }
	    } else if (canonicDataLabel === 'NTUPLES') {
	      if (dataValue.match(/(^nd|\snd\s)/i)) {
	        currentEntry.twoD = true;
	      }
	    } else if (canonicDataLabel === 'DATACLASS') {
	      currentEntry.dataClass = dataValue;
	    } else if (canonicDataLabel === 'XUNITS') {
	      spectrum.xUnits = dataValue;
	    } else if (canonicDataLabel === 'YUNITS') {
	      spectrum.yUnits = dataValue;
	    } else if (canonicDataLabel === 'FIRSTX') {
	      spectrum.firstX = Number(dataValue);
	    } else if (canonicDataLabel === 'LASTX') {
	      spectrum.lastX = Number(dataValue);
	    } else if (canonicDataLabel === 'FIRSTY') {
	      spectrum.firstY = Number(dataValue);
	    } else if (canonicDataLabel === 'LASTY') {
	      spectrum.lastY = Number(dataValue);
	    } else if (canonicDataLabel === 'NPOINTS') {
	      spectrum.nbPoints = Number(dataValue);
	    } else if (canonicDataLabel === 'XFACTOR') {
	      spectrum.xFactor = Number(dataValue);
	    } else if (canonicDataLabel === 'YFACTOR') {
	      spectrum.yFactor = Number(dataValue);
	    } else if (canonicDataLabel === 'MAXX') {
	      spectrum.maxX = Number(dataValue);
	    } else if (canonicDataLabel === 'MINX') {
	      spectrum.minX = Number(dataValue);
	    } else if (canonicDataLabel === 'MAXY') {
	      spectrum.maxY = Number(dataValue);
	    } else if (canonicDataLabel === 'MINY') {
	      spectrum.minY = Number(dataValue);
	    } else if (canonicDataLabel === 'DELTAX') {
	      spectrum.deltaX = Number(dataValue);
	    } else if (canonicDataLabel === '.OBSERVEFREQUENCY' || canonicDataLabel === '$SFO1') {
	      if (!spectrum.observeFrequency) {
	        spectrum.observeFrequency = Number(dataValue);
	      }
	    } else if (canonicDataLabel === '.OBSERVENUCLEUS') {
	      if (!spectrum.xType) {
	        currentEntry.xType = dataValue.replace(/[^a-zA-Z0-9]/g, '');
	      }
	    } else if (canonicDataLabel === '$OFFSET') {
	      // OFFSET for Bruker spectra
	      currentEntry.shiftOffsetNum = 0;

	      if (!spectrum.shiftOffsetVal) {
	        spectrum.shiftOffsetVal = Number(dataValue);
	      }
	    } else if (canonicDataLabel === '$REFERENCEPOINT') ; else if (canonicDataLabel === 'VARNAME') {
	      currentEntry.ntuples.varname = dataValue.split(ntuplesSeparatorRegExp);
	    } else if (canonicDataLabel === 'SYMBOL') {
	      currentEntry.ntuples.symbol = dataValue.split(ntuplesSeparatorRegExp);
	    } else if (canonicDataLabel === 'VARTYPE') {
	      currentEntry.ntuples.vartype = dataValue.split(ntuplesSeparatorRegExp);
	    } else if (canonicDataLabel === 'VARFORM') {
	      currentEntry.ntuples.varform = dataValue.split(ntuplesSeparatorRegExp);
	    } else if (canonicDataLabel === 'VARDIM') {
	      currentEntry.ntuples.vardim = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
	    } else if (canonicDataLabel === 'UNITS') {
	      currentEntry.ntuples.units = dataValue.split(ntuplesSeparatorRegExp);
	    } else if (canonicDataLabel === 'FACTOR') {
	      currentEntry.ntuples.factor = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
	    } else if (canonicDataLabel === 'FIRST') {
	      currentEntry.ntuples.first = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
	    } else if (canonicDataLabel === 'LAST') {
	      currentEntry.ntuples.last = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
	    } else if (canonicDataLabel === 'MIN') {
	      currentEntry.ntuples.min = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
	    } else if (canonicDataLabel === 'MAX') {
	      currentEntry.ntuples.max = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
	    } else if (canonicDataLabel === '.NUCLEUS') {
	      if (currentEntry.ntuples) {
	        currentEntry.ntuples.nucleus = dataValue.split(ntuplesSeparatorRegExp);
	      }
	    } else if (canonicDataLabel === 'PAGE') {
	      spectrum.page = dataValue.trim();
	      spectrum.pageValue = Number(dataValue.replace(/^.*=/, ''));
	      spectrum.pageSymbol = spectrum.page.replace(/[=].*/, '');
	    } else if (canonicDataLabel === 'RETENTIONTIME') {
	      spectrum.pageValue = Number(dataValue);
	    } else if (isMSField(canonicDataLabel)) {
	      spectrum[convertMSFieldToLabel(canonicDataLabel)] = dataValue;
	    } else if (canonicDataLabel === 'SAMPLEDESCRIPTION') {
	      spectrum.sampleDescription = dataValue;
	    } else if (canonicDataLabel.startsWith('$NUC')) {
	      if (!currentEntry.tmp[canonicDataLabel] && !dataValue.includes('off')) {
	        currentEntry.tmp[canonicDataLabel] = dataValue.replace(/[<>]/g, '');
	      }
	    } else if (canonicDataLabel === 'END') {
	      currentEntry = parentsStack.pop();
	    }

	    if (currentEntry && currentEntry.info && currentEntry.meta && canonicDataLabel.match(options.keepRecordsRegExp)) {
	      let value = dataValue.trim();
	      let target, label;

	      if (dataLabel.startsWith('$')) {
	        label = options.canonicMetadataLabels ? canonicDataLabel.substring(1) : dataLabel.substring(1);
	        target = currentEntry.meta;
	      } else {
	        label = options.canonicDataLabels ? canonicDataLabel : dataLabel;
	        target = currentEntry.info;
	      }

	      if (options.dynamicTyping) {
	        value = parseString(value);
	      }

	      if (target[label]) {
	        if (!Array.isArray(target[label])) {
	          target[label] = [target[label]];
	        }

	        target[label].push(value);
	      } else {
	        target[label] = value;
	      }
	    }
	  }

	  profiling(result, 'Finished parsing', options);
	  postProcessing(entriesFlat, result, options);
	  profiling(result, 'Total time', options);
	  /*
	  if (result.children && result.children.length>0) {
	    result = { ...result, ...result.children[0] };
	  }
	  */

	  result.entries = tmpResult.children;
	  result.flatten = entriesFlat;
	  return result;
	}

	function getJcampKind(data) {
	  let dataType = data.dataType.toLowerCase();
	  let yUnits = data.spectra[0].yUnits.toLowerCase();

	  if (dataType.match(/infrared/)) {
	    if (yUnits.match(/absorbance/)) {
	      return IR_ABSORBANCE;
	    } else {
	      return IR_TRANSMITTANCE;
	    }
	  }

	  return undefined;
	}
	const IR_TRANSMITTANCE = {
	  normalization: {},
	  importation: {
	    converter: transmittance => -Math.log10(transmittance)
	  },
	  kind: 'Infrared',
	  display: {
	    xLabel: 'wavelength [cm-1]',
	    xInverted: true,
	    yLabel: 'Absorbance'
	  }
	};
	const IR_ABSORBANCE = {
	  normalization: {},
	  kind: 'Infrared',
	  display: {
	    xLabel: 'wavelength [cm-1]',
	    xInverted: true,
	    yLabel: 'Absorbance'
	  }
	};

	/**
	 * Create a spectrum from a jcamp
	 * @param {string} jcampText - String containing the JCAMP data
	 * @return {Spectrum} - new instance of Spectrum with the provided data
	 */

	function jcamp(jcampText) {
	  const parsed = convert(jcampText, {
	    xy: true,
	    keepRecordsRegExp: /TITLE/
	  }).flatten[0];
	  const kind = getJcampKind(parsed);
	  const data = parsed.spectra[0].data;
	  const meta = parsed.info; // we convert the data

	  if (kind && kind.importation && kind.importation.converter) {
	    data.y = data.y.map(kind.importation.converter);
	  }

	  return {
	    data,
	    kind,
	    meta
	  };
	}

	/**
	 * Creates a g
	 * @param {string} text - String containing the text data
	 * @param {object} [options={}]
	 * @param {string} [options.fs='\t'] Field separator
	 * @return {object} - {matrix, data, x, ids}
	 */
	function matrix(text) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const lines = text.split(/[\r\n]+/).filter(value => value);
	  const {
	    fs = '\t'
	  } = options;
	  let currentMatrix = [];
	  let ids = [];
	  let meta = [];
	  let x = [];
	  let headers = lines[0].split(fs);
	  let labels = [];

	  for (let i = 0; i < headers.length; i++) {
	    let header = headers[i];

	    if (isNaN(header)) {
	      labels[i] = header;
	    } else {
	      x = headers.slice(i).map(value => Number(value));
	      break;
	    }
	  }

	  for (let i = 1; i < lines.length; i++) {
	    let line = lines[i];
	    let parts = line.split('\t');
	    ids.push(parts[0]);
	    let oneMeta = {};
	    meta.push(oneMeta);

	    for (let j = 1; j < parts.length; j++) {
	      if (j < labels.length) {
	        oneMeta[labels[j]] = parts[j];
	      } else {
	        currentMatrix.push(parts.slice(labels.length).map(value => Number(value)));
	        break;
	      }
	    }
	  }

	  return {
	    x,
	    meta,
	    matrix: currentMatrix,
	    ids
	  };
	}

	/**
	 * In place modification of the 2 arrays to make X unique and sum the Y if X has the same value
	 * @param {object} [points={}] : Object of points contains property x (an array) and y (an array)
	 * @return points
	 */
	function uniqueX() {
	  let points = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	  const {
	    x,
	    y
	  } = points;
	  if (x.length < 2) return;

	  if (x.length !== y.length) {
	    throw new Error('The X and Y arrays mush have the same length');
	  }

	  let current = x[0];
	  let counter = 0;

	  for (let i = 1; i < x.length; i++) {
	    if (current !== x[i]) {
	      counter++;
	      current = x[i];
	      x[counter] = x[i];

	      if (i !== counter) {
	        y[counter] = 0;
	      }
	    }

	    if (i !== counter) {
	      y[counter] += y[i];
	    }
	  }

	  x.length = counter + 1;
	  y.length = counter + 1;
	}

	/**
	 * General internal parsing function
	 *
	 * @param text - Csv or tsv strings.
	 * @param options - Parsing options
	 * @returns parsed text file with column information
	 */

	function parse(text) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  let {
	    rescale = false,
	    uniqueX: uniqueX$1 = false,
	    xColumn = 0,
	    yColumn = 1,
	    bestGuess = false,
	    //@ts-expect-error old library used this property and we want to throw an error so that people are forced to migrate
	    keepInfo,
	    numberColumns = Number.MAX_SAFE_INTEGER,
	    maxNumberColumns = Number.MAX_SAFE_INTEGER,
	    minNumberColumns = 2
	  } = options;

	  if (keepInfo !== undefined) {
	    throw new Error('keepInfo has been deprecated, pelase use the new method parseXYAndKeepInfo');
	  }

	  text = ensureString(text);
	  maxNumberColumns = Math.max(maxNumberColumns, xColumn + 1, yColumn + 1);
	  minNumberColumns = Math.max(xColumn + 1, yColumn + 1, minNumberColumns);
	  let lines = text.split(/[\r\n]+/);
	  let matrix = [];
	  let info = [];
	  let position = 0;
	  lines.forEach(line => {
	    line = line.trim(); // we will consider only lines that contains only numbers

	    if (/[0-9]+/.test(line) && /^[0-9eE,;. \t+-]+$/.test(line)) {
	      let fields = line.split(/,[; \t]+|[; \t]+/);

	      if (fields.length === 1) {
	        fields = line.split(/[,; \t]+/);
	      }

	      if (fields && fields.length >= minNumberColumns && // we filter lines that have not enough or too many columns
	      fields.length <= maxNumberColumns) {
	        matrix.push(fields.map(value => parseFloat(value.replace(',', '.'))));
	        position++;
	      }
	    } else if (line) {
	      info.push({
	        position,
	        value: line
	      });
	    }
	  });

	  if (bestGuess) {
	    if (matrix[0] && matrix[0].length === 3 && options.xColumn === undefined && options.yColumn === undefined) {
	      // is the first column a seuqnetial number ?
	      let skipFirstColumn = true;

	      for (let i = 0; i < matrix.length - 1; i++) {
	        if (Math.abs(matrix[i][0] - matrix[i + 1][0]) !== 1) {
	          skipFirstColumn = false;
	        }
	      }

	      if (skipFirstColumn) {
	        xColumn = 1;
	        yColumn = 2;
	      }
	    }

	    if (matrix[0] && matrix[0].length > 3) {
	      const xs = [];

	      for (let row of matrix) {
	        for (let i = xColumn; i < row.length; i += 2) {
	          xs.push(row[i]);
	        }
	      }

	      if (xIsMonotone(xs)) {
	        numberColumns = 2;
	      }
	    }
	  }

	  if (numberColumns) {
	    const newMatrix = [];

	    for (const row of matrix) {
	      for (let i = 0; i < row.length; i += numberColumns) {
	        newMatrix.push(row.slice(i, i + numberColumns));
	      }
	    }

	    matrix = newMatrix;
	  }

	  const result = {
	    x: matrix.map(row => row[xColumn]),
	    y: matrix.map(row => row[yColumn])
	  };

	  if (uniqueX$1) {
	    uniqueX(result);
	  }

	  if (rescale) {
	    let maxY = max$1(result.y);

	    for (let i = 0; i < result.y.length; i++) {
	      result.y[i] /= maxY;
	    }
	  }

	  return {
	    info,
	    data: result
	  };
	}

	/**
	 * Parse a text-file and convert it to an object {x:[], y:[]}
	 *
	 * @param text - Csv or tsv strings.
	 * @param options - Parsing options
	 * @returns - The parsed data
	 */

	function parseXY(text) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  return parse(text, options).data;
	}

	/**
	 * Create a spectrum from a text file
	 * @param {string} value - String containing the JCAMP data
	 * @return {Spectrum} - new instance of Spectrum with the provided data
	 */

	function text(value) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    kind,
	    parserOptions = {}
	  } = options;
	  const data = parseXY(value, parserOptions); // we convert the data

	  if (kind && kind.importation && kind.importation.converter) {
	    data.y = data.y.map(kind.importation.converter);
	  }

	  return {
	    data
	  };
	}

	/**
	 * @private
	 */

	function getAutocorrelation(normalized, index) {
	  let matrix = normalized.matrix;
	  let nbRow = matrix.length;
	  let nbColumn = matrix[0].length;
	  let array1 = new Float64Array(nbRow);
	  let array2 = new Float64Array(nbRow);
	  let result = new Array(nbColumn);

	  for (let j = 0; j < nbRow; j++) {
	    array1[j] = matrix[j][index];
	  }

	  for (let i = 0; i < nbColumn; i++) {
	    for (let j = 0; j < nbRow; j++) {
	      array2[j] = matrix[j][i];
	    }

	    result[i] = xCorrelation(array1, array2);
	  }

	  return {
	    x: normalized.x,
	    y: result
	  };
	}

	/**
	 * @private
	 */

	function getMeanData(normalized) {
	  let matrix = new Matrix(normalized.matrix);
	  return {
	    x: normalized.x,
	    y: matrix.mean('column')
	  };
	}

	function convertToText(data) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  let {
	    fs = '\t',
	    rs = '\n'
	  } = options;
	  let {
	    matrix,
	    meta,
	    ids,
	    x
	  } = data;
	  let allKeysObject = {};

	  for (let metum of meta) {
	    if (metum) {
	      for (let key of Object.keys(metum)) {
	        let type = typeof metum[key];

	        if (type === 'number' || type === 'string' || type === 'boolean') {
	          allKeysObject[key] = true;
	        }
	      }
	    }
	  }

	  let allKeys = Object.keys(allKeysObject);
	  let lines = [];
	  let line = [];
	  line.push('id', ...allKeys, ...x);
	  lines.push(line.join(fs));

	  for (let i = 0; i < ids.length; i++) {
	    line = [];
	    line.push(ids[i]);

	    for (let key of allKeys) {
	      line.push(meta[i][key]);
	    }

	    line.push(...matrix[i]);
	    lines.push(line.join(fs));
	  }

	  return lines.join(rs);
	}

	/**
	 * @private
	 * @param {*} spectra
	 * @param {object} [options={}]
	 * @param {string} [options.fs='\t'] field separator
	 * @param {string} [options.rs='\n'] record (line) separator
	 */

	function getNormalizedText(spectra) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  let {
	    fs = '\t',
	    rs = '\n'
	  } = options;
	  return convertToText(getNormalizedData(spectra), {
	    rs,
	    fs
	  });
	}

	var object_hash = {exports: {}};

	(function (module, exports) {
	  !function (e) {
	    module.exports = e() ;
	  }(function () {
	    return function r(o, i, u) {
	      function s(n, e) {
	        if (!i[n]) {
	          if (!o[n]) {
	            var t = "function" == typeof commonjsRequire && commonjsRequire;
	            if (!e && t) return t(n, !0);
	            if (a) return a(n, !0);
	            throw new Error("Cannot find module '" + n + "'");
	          }

	          e = i[n] = {
	            exports: {}
	          };
	          o[n][0].call(e.exports, function (e) {
	            var t = o[n][1][e];
	            return s(t || e);
	          }, e, e.exports, r, o, i, u);
	        }

	        return i[n].exports;
	      }

	      for (var a = "function" == typeof commonjsRequire && commonjsRequire, e = 0; e < u.length; e++) s(u[e]);

	      return s;
	    }({
	      1: [function (w, b, m) {
	        !function (e, n, s, c, d, h, p, g, y) {

	          var r = w("crypto");

	          function t(e, t) {
	            t = u(e, t);
	            var n;
	            return void 0 === (n = "passthrough" !== t.algorithm ? r.createHash(t.algorithm) : new l()).write && (n.write = n.update, n.end = n.update), f(t, n).dispatch(e), n.update || n.end(""), n.digest ? n.digest("buffer" === t.encoding ? void 0 : t.encoding) : (e = n.read(), "buffer" !== t.encoding ? e.toString(t.encoding) : e);
	          }

	          (m = b.exports = t).sha1 = function (e) {
	            return t(e);
	          }, m.keys = function (e) {
	            return t(e, {
	              excludeValues: !0,
	              algorithm: "sha1",
	              encoding: "hex"
	            });
	          }, m.MD5 = function (e) {
	            return t(e, {
	              algorithm: "md5",
	              encoding: "hex"
	            });
	          }, m.keysMD5 = function (e) {
	            return t(e, {
	              algorithm: "md5",
	              encoding: "hex",
	              excludeValues: !0
	            });
	          };
	          var o = r.getHashes ? r.getHashes().slice() : ["sha1", "md5"],
	              i = (o.push("passthrough"), ["buffer", "hex", "binary", "base64"]);

	          function u(e, t) {
	            var n = {};
	            if (n.algorithm = (t = t || {}).algorithm || "sha1", n.encoding = t.encoding || "hex", n.excludeValues = !!t.excludeValues, n.algorithm = n.algorithm.toLowerCase(), n.encoding = n.encoding.toLowerCase(), n.ignoreUnknown = !0 === t.ignoreUnknown, n.respectType = !1 !== t.respectType, n.respectFunctionNames = !1 !== t.respectFunctionNames, n.respectFunctionProperties = !1 !== t.respectFunctionProperties, n.unorderedArrays = !0 === t.unorderedArrays, n.unorderedSets = !1 !== t.unorderedSets, n.unorderedObjects = !1 !== t.unorderedObjects, n.replacer = t.replacer || void 0, n.excludeKeys = t.excludeKeys || void 0, void 0 === e) throw new Error("Object argument required.");

	            for (var r = 0; r < o.length; ++r) o[r].toLowerCase() === n.algorithm.toLowerCase() && (n.algorithm = o[r]);

	            if (-1 === o.indexOf(n.algorithm)) throw new Error('Algorithm "' + n.algorithm + '"  not supported. supported values: ' + o.join(", "));
	            if (-1 === i.indexOf(n.encoding) && "passthrough" !== n.algorithm) throw new Error('Encoding "' + n.encoding + '"  not supported. supported values: ' + i.join(", "));
	            return n;
	          }

	          function a(e) {
	            if ("function" == typeof e) return null != /^function\s+\w*\s*\(\s*\)\s*{\s+\[native code\]\s+}$/i.exec(Function.prototype.toString.call(e));
	          }

	          function f(o, t, i) {
	            i = i || [];

	            function u(e) {
	              return t.update ? t.update(e, "utf8") : t.write(e, "utf8");
	            }

	            return {
	              dispatch: function (e) {
	                return this["_" + (null === (e = o.replacer ? o.replacer(e) : e) ? "null" : typeof e)](e);
	              },
	              _object: function (t) {
	                var n,
	                    e = Object.prototype.toString.call(t),
	                    r = /\[object (.*)\]/i.exec(e);
	                r = (r = r ? r[1] : "unknown:[" + e + "]").toLowerCase();
	                if (0 <= (e = i.indexOf(t))) return this.dispatch("[CIRCULAR:" + e + "]");
	                if (i.push(t), void 0 !== s && s.isBuffer && s.isBuffer(t)) return u("buffer:"), u(t);
	                if ("object" === r || "function" === r || "asyncfunction" === r) return e = Object.keys(t), o.unorderedObjects && (e = e.sort()), !1 === o.respectType || a(t) || e.splice(0, 0, "prototype", "__proto__", "constructor"), o.excludeKeys && (e = e.filter(function (e) {
	                  return !o.excludeKeys(e);
	                })), u("object:" + e.length + ":"), n = this, e.forEach(function (e) {
	                  n.dispatch(e), u(":"), o.excludeValues || n.dispatch(t[e]), u(",");
	                });

	                if (!this["_" + r]) {
	                  if (o.ignoreUnknown) return u("[" + r + "]");
	                  throw new Error('Unknown object type "' + r + '"');
	                }

	                this["_" + r](t);
	              },
	              _array: function (e, t) {
	                t = void 0 !== t ? t : !1 !== o.unorderedArrays;
	                var n = this;
	                if (u("array:" + e.length + ":"), !t || e.length <= 1) return e.forEach(function (e) {
	                  return n.dispatch(e);
	                });
	                var r = [],
	                    t = e.map(function (e) {
	                  var t = new l(),
	                      n = i.slice();
	                  return f(o, t, n).dispatch(e), r = r.concat(n.slice(i.length)), t.read().toString();
	                });
	                return i = i.concat(r), t.sort(), this._array(t, !1);
	              },
	              _date: function (e) {
	                return u("date:" + e.toJSON());
	              },
	              _symbol: function (e) {
	                return u("symbol:" + e.toString());
	              },
	              _error: function (e) {
	                return u("error:" + e.toString());
	              },
	              _boolean: function (e) {
	                return u("bool:" + e.toString());
	              },
	              _string: function (e) {
	                u("string:" + e.length + ":"), u(e.toString());
	              },
	              _function: function (e) {
	                u("fn:"), a(e) ? this.dispatch("[native]") : this.dispatch(e.toString()), !1 !== o.respectFunctionNames && this.dispatch("function-name:" + String(e.name)), o.respectFunctionProperties && this._object(e);
	              },
	              _number: function (e) {
	                return u("number:" + e.toString());
	              },
	              _xml: function (e) {
	                return u("xml:" + e.toString());
	              },
	              _null: function () {
	                return u("Null");
	              },
	              _undefined: function () {
	                return u("Undefined");
	              },
	              _regexp: function (e) {
	                return u("regex:" + e.toString());
	              },
	              _uint8array: function (e) {
	                return u("uint8array:"), this.dispatch(Array.prototype.slice.call(e));
	              },
	              _uint8clampedarray: function (e) {
	                return u("uint8clampedarray:"), this.dispatch(Array.prototype.slice.call(e));
	              },
	              _int8array: function (e) {
	                return u("int8array:"), this.dispatch(Array.prototype.slice.call(e));
	              },
	              _uint16array: function (e) {
	                return u("uint16array:"), this.dispatch(Array.prototype.slice.call(e));
	              },
	              _int16array: function (e) {
	                return u("int16array:"), this.dispatch(Array.prototype.slice.call(e));
	              },
	              _uint32array: function (e) {
	                return u("uint32array:"), this.dispatch(Array.prototype.slice.call(e));
	              },
	              _int32array: function (e) {
	                return u("int32array:"), this.dispatch(Array.prototype.slice.call(e));
	              },
	              _float32array: function (e) {
	                return u("float32array:"), this.dispatch(Array.prototype.slice.call(e));
	              },
	              _float64array: function (e) {
	                return u("float64array:"), this.dispatch(Array.prototype.slice.call(e));
	              },
	              _arraybuffer: function (e) {
	                return u("arraybuffer:"), this.dispatch(new Uint8Array(e));
	              },
	              _url: function (e) {
	                return u("url:" + e.toString());
	              },
	              _map: function (e) {
	                u("map:");
	                e = Array.from(e);
	                return this._array(e, !1 !== o.unorderedSets);
	              },
	              _set: function (e) {
	                u("set:");
	                e = Array.from(e);
	                return this._array(e, !1 !== o.unorderedSets);
	              },
	              _file: function (e) {
	                return u("file:"), this.dispatch([e.name, e.size, e.type, e.lastModfied]);
	              },
	              _blob: function () {
	                if (o.ignoreUnknown) return u("[blob]");
	                throw Error('Hashing Blob objects is currently not supported\n(see https://github.com/puleos/object-hash/issues/26)\nUse "options.replacer" or "options.ignoreUnknown"\n');
	              },
	              _domwindow: function () {
	                return u("domwindow");
	              },
	              _bigint: function (e) {
	                return u("bigint:" + e.toString());
	              },
	              _process: function () {
	                return u("process");
	              },
	              _timer: function () {
	                return u("timer");
	              },
	              _pipe: function () {
	                return u("pipe");
	              },
	              _tcp: function () {
	                return u("tcp");
	              },
	              _udp: function () {
	                return u("udp");
	              },
	              _tty: function () {
	                return u("tty");
	              },
	              _statwatcher: function () {
	                return u("statwatcher");
	              },
	              _securecontext: function () {
	                return u("securecontext");
	              },
	              _connection: function () {
	                return u("connection");
	              },
	              _zlib: function () {
	                return u("zlib");
	              },
	              _context: function () {
	                return u("context");
	              },
	              _nodescript: function () {
	                return u("nodescript");
	              },
	              _httpparser: function () {
	                return u("httpparser");
	              },
	              _dataview: function () {
	                return u("dataview");
	              },
	              _signal: function () {
	                return u("signal");
	              },
	              _fsevent: function () {
	                return u("fsevent");
	              },
	              _tlswrap: function () {
	                return u("tlswrap");
	              }
	            };
	          }

	          function l() {
	            return {
	              buf: "",
	              write: function (e) {
	                this.buf += e;
	              },
	              end: function (e) {
	                this.buf += e;
	              },
	              read: function () {
	                return this.buf;
	              }
	            };
	          }

	          m.writeToStream = function (e, t, n) {
	            return void 0 === n && (n = t, t = {}), f(t = u(e, t), n).dispatch(e);
	          };
	        }.call(this, w("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, w("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/fake_9a5aa49d.js", "/");
	      }, {
	        buffer: 3,
	        crypto: 5,
	        lYpoI2: 11
	      }],
	      2: [function (e, t, f) {
	        !function (e, t, n, r, o, i, u, s, a) {
	          !function (e) {

	            var a = "undefined" != typeof Uint8Array ? Uint8Array : Array,
	                t = "+".charCodeAt(0),
	                n = "/".charCodeAt(0),
	                r = "0".charCodeAt(0),
	                o = "a".charCodeAt(0),
	                i = "A".charCodeAt(0),
	                u = "-".charCodeAt(0),
	                s = "_".charCodeAt(0);

	            function f(e) {
	              e = e.charCodeAt(0);
	              return e === t || e === u ? 62 : e === n || e === s ? 63 : e < r ? -1 : e < r + 10 ? e - r + 26 + 26 : e < i + 26 ? e - i : e < o + 26 ? e - o + 26 : void 0;
	            }

	            e.toByteArray = function (e) {
	              var t, n;
	              if (0 < e.length % 4) throw new Error("Invalid string. Length must be a multiple of 4");
	              var r = e.length,
	                  r = "=" === e.charAt(r - 2) ? 2 : "=" === e.charAt(r - 1) ? 1 : 0,
	                  o = new a(3 * e.length / 4 - r),
	                  i = 0 < r ? e.length - 4 : e.length,
	                  u = 0;

	              function s(e) {
	                o[u++] = e;
	              }

	              for (t = 0; t < i; t += 4, 0) s((16711680 & (n = f(e.charAt(t)) << 18 | f(e.charAt(t + 1)) << 12 | f(e.charAt(t + 2)) << 6 | f(e.charAt(t + 3)))) >> 16), s((65280 & n) >> 8), s(255 & n);

	              return 2 == r ? s(255 & (n = f(e.charAt(t)) << 2 | f(e.charAt(t + 1)) >> 4)) : 1 == r && (s((n = f(e.charAt(t)) << 10 | f(e.charAt(t + 1)) << 4 | f(e.charAt(t + 2)) >> 2) >> 8 & 255), s(255 & n)), o;
	            }, e.fromByteArray = function (e) {
	              var t,
	                  n,
	                  r,
	                  o,
	                  i = e.length % 3,
	                  u = "";

	              function s(e) {
	                return "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/".charAt(e);
	              }

	              for (t = 0, r = e.length - i; t < r; t += 3) n = (e[t] << 16) + (e[t + 1] << 8) + e[t + 2], u += s((o = n) >> 18 & 63) + s(o >> 12 & 63) + s(o >> 6 & 63) + s(63 & o);

	              switch (i) {
	                case 1:
	                  u = (u += s((n = e[e.length - 1]) >> 2)) + s(n << 4 & 63) + "==";
	                  break;

	                case 2:
	                  u = (u = (u += s((n = (e[e.length - 2] << 8) + e[e.length - 1]) >> 10)) + s(n >> 4 & 63)) + s(n << 2 & 63) + "=";
	              }

	              return u;
	            };
	          }(void 0 === f ? this.base64js = {} : f);
	        }.call(this, e("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, e("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/base64-js/lib/b64.js", "/node_modules/gulp-browserify/node_modules/base64-js/lib");
	      }, {
	        buffer: 3,
	        lYpoI2: 11
	      }],
	      3: [function (O, e, H) {
	        !function (e, n, f, r, h, p, g, y, w) {
	          var a = O("base64-js"),
	              i = O("ieee754");

	          function f(e, t, n) {
	            if (!(this instanceof f)) return new f(e, t, n);
	            var r,
	                o,
	                i,
	                u,
	                s = typeof e;
	            if ("base64" === t && "string" == s) for (e = (u = e).trim ? u.trim() : u.replace(/^\s+|\s+$/g, ""); e.length % 4 != 0;) e += "=";
	            if ("number" == s) r = j(e);else if ("string" == s) r = f.byteLength(e, t);else {
	              if ("object" != s) throw new Error("First argument needs to be a number, array or string.");
	              r = j(e.length);
	            }
	            if (f._useTypedArrays ? o = f._augment(new Uint8Array(r)) : ((o = this).length = r, o._isBuffer = !0), f._useTypedArrays && "number" == typeof e.byteLength) o._set(e);else if (C(u = e) || f.isBuffer(u) || u && "object" == typeof u && "number" == typeof u.length) for (i = 0; i < r; i++) f.isBuffer(e) ? o[i] = e.readUInt8(i) : o[i] = e[i];else if ("string" == s) o.write(e, 0, t);else if ("number" == s && !f._useTypedArrays && !n) for (i = 0; i < r; i++) o[i] = 0;
	            return o;
	          }

	          function b(e, t, n, r) {
	            return f._charsWritten = c(function (e) {
	              for (var t = [], n = 0; n < e.length; n++) t.push(255 & e.charCodeAt(n));

	              return t;
	            }(t), e, n, r);
	          }

	          function m(e, t, n, r) {
	            return f._charsWritten = c(function (e) {
	              for (var t, n, r = [], o = 0; o < e.length; o++) n = e.charCodeAt(o), t = n >> 8, n = n % 256, r.push(n), r.push(t);

	              return r;
	            }(t), e, n, r);
	          }

	          function v(e, t, n) {
	            var r = "";
	            n = Math.min(e.length, n);

	            for (var o = t; o < n; o++) r += String.fromCharCode(e[o]);

	            return r;
	          }

	          function o(e, t, n, r) {
	            r || (d("boolean" == typeof n, "missing or invalid endian"), d(null != t, "missing offset"), d(t + 1 < e.length, "Trying to read beyond buffer length"));
	            var o,
	                r = e.length;
	            if (!(r <= t)) return n ? (o = e[t], t + 1 < r && (o |= e[t + 1] << 8)) : (o = e[t] << 8, t + 1 < r && (o |= e[t + 1])), o;
	          }

	          function u(e, t, n, r) {
	            r || (d("boolean" == typeof n, "missing or invalid endian"), d(null != t, "missing offset"), d(t + 3 < e.length, "Trying to read beyond buffer length"));
	            var o,
	                r = e.length;
	            if (!(r <= t)) return n ? (t + 2 < r && (o = e[t + 2] << 16), t + 1 < r && (o |= e[t + 1] << 8), o |= e[t], t + 3 < r && (o += e[t + 3] << 24 >>> 0)) : (t + 1 < r && (o = e[t + 1] << 16), t + 2 < r && (o |= e[t + 2] << 8), t + 3 < r && (o |= e[t + 3]), o += e[t] << 24 >>> 0), o;
	          }

	          function _(e, t, n, r) {
	            if (r || (d("boolean" == typeof n, "missing or invalid endian"), d(null != t, "missing offset"), d(t + 1 < e.length, "Trying to read beyond buffer length")), !(e.length <= t)) return r = o(e, t, n, !0), 32768 & r ? -1 * (65535 - r + 1) : r;
	          }

	          function E(e, t, n, r) {
	            if (r || (d("boolean" == typeof n, "missing or invalid endian"), d(null != t, "missing offset"), d(t + 3 < e.length, "Trying to read beyond buffer length")), !(e.length <= t)) return r = u(e, t, n, !0), 2147483648 & r ? -1 * (4294967295 - r + 1) : r;
	          }

	          function I(e, t, n, r) {
	            return r || (d("boolean" == typeof n, "missing or invalid endian"), d(t + 3 < e.length, "Trying to read beyond buffer length")), i.read(e, t, n, 23, 4);
	          }

	          function A(e, t, n, r) {
	            return r || (d("boolean" == typeof n, "missing or invalid endian"), d(t + 7 < e.length, "Trying to read beyond buffer length")), i.read(e, t, n, 52, 8);
	          }

	          function s(e, t, n, r, o) {
	            o || (d(null != t, "missing value"), d("boolean" == typeof r, "missing or invalid endian"), d(null != n, "missing offset"), d(n + 1 < e.length, "trying to write beyond buffer length"), Y(t, 65535));
	            o = e.length;
	            if (!(o <= n)) for (var i = 0, u = Math.min(o - n, 2); i < u; i++) e[n + i] = (t & 255 << 8 * (r ? i : 1 - i)) >>> 8 * (r ? i : 1 - i);
	          }

	          function l(e, t, n, r, o) {
	            o || (d(null != t, "missing value"), d("boolean" == typeof r, "missing or invalid endian"), d(null != n, "missing offset"), d(n + 3 < e.length, "trying to write beyond buffer length"), Y(t, 4294967295));
	            o = e.length;
	            if (!(o <= n)) for (var i = 0, u = Math.min(o - n, 4); i < u; i++) e[n + i] = t >>> 8 * (r ? i : 3 - i) & 255;
	          }

	          function B(e, t, n, r, o) {
	            o || (d(null != t, "missing value"), d("boolean" == typeof r, "missing or invalid endian"), d(null != n, "missing offset"), d(n + 1 < e.length, "Trying to write beyond buffer length"), F(t, 32767, -32768)), e.length <= n || s(e, 0 <= t ? t : 65535 + t + 1, n, r, o);
	          }

	          function L(e, t, n, r, o) {
	            o || (d(null != t, "missing value"), d("boolean" == typeof r, "missing or invalid endian"), d(null != n, "missing offset"), d(n + 3 < e.length, "Trying to write beyond buffer length"), F(t, 2147483647, -2147483648)), e.length <= n || l(e, 0 <= t ? t : 4294967295 + t + 1, n, r, o);
	          }

	          function U(e, t, n, r, o) {
	            o || (d(null != t, "missing value"), d("boolean" == typeof r, "missing or invalid endian"), d(null != n, "missing offset"), d(n + 3 < e.length, "Trying to write beyond buffer length"), D(t, 34028234663852886e22, -34028234663852886e22)), e.length <= n || i.write(e, t, n, r, 23, 4);
	          }

	          function x(e, t, n, r, o) {
	            o || (d(null != t, "missing value"), d("boolean" == typeof r, "missing or invalid endian"), d(null != n, "missing offset"), d(n + 7 < e.length, "Trying to write beyond buffer length"), D(t, 17976931348623157e292, -17976931348623157e292)), e.length <= n || i.write(e, t, n, r, 52, 8);
	          }

	          H.Buffer = f, H.SlowBuffer = f, H.INSPECT_MAX_BYTES = 50, f.poolSize = 8192, f._useTypedArrays = function () {
	            try {
	              var e = new ArrayBuffer(0),
	                  t = new Uint8Array(e);
	              return t.foo = function () {
	                return 42;
	              }, 42 === t.foo() && "function" == typeof t.subarray;
	            } catch (e) {
	              return !1;
	            }
	          }(), f.isEncoding = function (e) {
	            switch (String(e).toLowerCase()) {
	              case "hex":
	              case "utf8":
	              case "utf-8":
	              case "ascii":
	              case "binary":
	              case "base64":
	              case "raw":
	              case "ucs2":
	              case "ucs-2":
	              case "utf16le":
	              case "utf-16le":
	                return !0;

	              default:
	                return !1;
	            }
	          }, f.isBuffer = function (e) {
	            return !(null == e || !e._isBuffer);
	          }, f.byteLength = function (e, t) {
	            var n;

	            switch (e += "", t || "utf8") {
	              case "hex":
	                n = e.length / 2;
	                break;

	              case "utf8":
	              case "utf-8":
	                n = T(e).length;
	                break;

	              case "ascii":
	              case "binary":
	              case "raw":
	                n = e.length;
	                break;

	              case "base64":
	                n = M(e).length;
	                break;

	              case "ucs2":
	              case "ucs-2":
	              case "utf16le":
	              case "utf-16le":
	                n = 2 * e.length;
	                break;

	              default:
	                throw new Error("Unknown encoding");
	            }

	            return n;
	          }, f.concat = function (e, t) {
	            if (d(C(e), "Usage: Buffer.concat(list, [totalLength])\nlist should be an Array."), 0 === e.length) return new f(0);
	            if (1 === e.length) return e[0];
	            if ("number" != typeof t) for (o = t = 0; o < e.length; o++) t += e[o].length;

	            for (var n = new f(t), r = 0, o = 0; o < e.length; o++) {
	              var i = e[o];
	              i.copy(n, r), r += i.length;
	            }

	            return n;
	          }, f.prototype.write = function (e, t, n, r) {
	            isFinite(t) ? isFinite(n) || (r = n, n = void 0) : (a = r, r = t, t = n, n = a), t = Number(t) || 0;
	            var o,
	                i,
	                u,
	                s,
	                a = this.length - t;

	            switch ((!n || a < (n = Number(n))) && (n = a), r = String(r || "utf8").toLowerCase()) {
	              case "hex":
	                o = function (e, t, n, r) {
	                  n = Number(n) || 0;
	                  var o = e.length - n;
	                  (!r || o < (r = Number(r))) && (r = o), d((o = t.length) % 2 == 0, "Invalid hex string"), o / 2 < r && (r = o / 2);

	                  for (var i = 0; i < r; i++) {
	                    var u = parseInt(t.substr(2 * i, 2), 16);
	                    d(!isNaN(u), "Invalid hex string"), e[n + i] = u;
	                  }

	                  return f._charsWritten = 2 * i, i;
	                }(this, e, t, n);

	                break;

	              case "utf8":
	              case "utf-8":
	                i = this, u = t, s = n, o = f._charsWritten = c(T(e), i, u, s);
	                break;

	              case "ascii":
	              case "binary":
	                o = b(this, e, t, n);
	                break;

	              case "base64":
	                i = this, u = t, s = n, o = f._charsWritten = c(M(e), i, u, s);
	                break;

	              case "ucs2":
	              case "ucs-2":
	              case "utf16le":
	              case "utf-16le":
	                o = m(this, e, t, n);
	                break;

	              default:
	                throw new Error("Unknown encoding");
	            }

	            return o;
	          }, f.prototype.toString = function (e, t, n) {
	            var r,
	                o,
	                i,
	                u,
	                s = this;
	            if (e = String(e || "utf8").toLowerCase(), t = Number(t) || 0, (n = void 0 !== n ? Number(n) : s.length) === t) return "";

	            switch (e) {
	              case "hex":
	                r = function (e, t, n) {
	                  var r = e.length;
	                  (!t || t < 0) && (t = 0);
	                  (!n || n < 0 || r < n) && (n = r);

	                  for (var o = "", i = t; i < n; i++) o += k(e[i]);

	                  return o;
	                }(s, t, n);

	                break;

	              case "utf8":
	              case "utf-8":
	                r = function (e, t, n) {
	                  var r = "",
	                      o = "";
	                  n = Math.min(e.length, n);

	                  for (var i = t; i < n; i++) e[i] <= 127 ? (r += N(o) + String.fromCharCode(e[i]), o = "") : o += "%" + e[i].toString(16);

	                  return r + N(o);
	                }(s, t, n);

	                break;

	              case "ascii":
	              case "binary":
	                r = v(s, t, n);
	                break;

	              case "base64":
	                o = s, u = n, r = 0 === (i = t) && u === o.length ? a.fromByteArray(o) : a.fromByteArray(o.slice(i, u));
	                break;

	              case "ucs2":
	              case "ucs-2":
	              case "utf16le":
	              case "utf-16le":
	                r = function (e, t, n) {
	                  for (var r = e.slice(t, n), o = "", i = 0; i < r.length; i += 2) o += String.fromCharCode(r[i] + 256 * r[i + 1]);

	                  return o;
	                }(s, t, n);

	                break;

	              default:
	                throw new Error("Unknown encoding");
	            }

	            return r;
	          }, f.prototype.toJSON = function () {
	            return {
	              type: "Buffer",
	              data: Array.prototype.slice.call(this._arr || this, 0)
	            };
	          }, f.prototype.copy = function (e, t, n, r) {
	            if (t = t || 0, (r = r || 0 === r ? r : this.length) !== (n = n || 0) && 0 !== e.length && 0 !== this.length) {
	              d(n <= r, "sourceEnd < sourceStart"), d(0 <= t && t < e.length, "targetStart out of bounds"), d(0 <= n && n < this.length, "sourceStart out of bounds"), d(0 <= r && r <= this.length, "sourceEnd out of bounds"), r > this.length && (r = this.length);
	              var o = (r = e.length - t < r - n ? e.length - t + n : r) - n;
	              if (o < 100 || !f._useTypedArrays) for (var i = 0; i < o; i++) e[i + t] = this[i + n];else e._set(this.subarray(n, n + o), t);
	            }
	          }, f.prototype.slice = function (e, t) {
	            var n = this.length;
	            if (e = S(e, n, 0), t = S(t, n, n), f._useTypedArrays) return f._augment(this.subarray(e, t));

	            for (var r = t - e, o = new f(r, void 0, !0), i = 0; i < r; i++) o[i] = this[i + e];

	            return o;
	          }, f.prototype.get = function (e) {
	            return console.log(".get() is deprecated. Access using array indexes instead."), this.readUInt8(e);
	          }, f.prototype.set = function (e, t) {
	            return console.log(".set() is deprecated. Access using array indexes instead."), this.writeUInt8(e, t);
	          }, f.prototype.readUInt8 = function (e, t) {
	            if (t || (d(null != e, "missing offset"), d(e < this.length, "Trying to read beyond buffer length")), !(e >= this.length)) return this[e];
	          }, f.prototype.readUInt16LE = function (e, t) {
	            return o(this, e, !0, t);
	          }, f.prototype.readUInt16BE = function (e, t) {
	            return o(this, e, !1, t);
	          }, f.prototype.readUInt32LE = function (e, t) {
	            return u(this, e, !0, t);
	          }, f.prototype.readUInt32BE = function (e, t) {
	            return u(this, e, !1, t);
	          }, f.prototype.readInt8 = function (e, t) {
	            if (t || (d(null != e, "missing offset"), d(e < this.length, "Trying to read beyond buffer length")), !(e >= this.length)) return 128 & this[e] ? -1 * (255 - this[e] + 1) : this[e];
	          }, f.prototype.readInt16LE = function (e, t) {
	            return _(this, e, !0, t);
	          }, f.prototype.readInt16BE = function (e, t) {
	            return _(this, e, !1, t);
	          }, f.prototype.readInt32LE = function (e, t) {
	            return E(this, e, !0, t);
	          }, f.prototype.readInt32BE = function (e, t) {
	            return E(this, e, !1, t);
	          }, f.prototype.readFloatLE = function (e, t) {
	            return I(this, e, !0, t);
	          }, f.prototype.readFloatBE = function (e, t) {
	            return I(this, e, !1, t);
	          }, f.prototype.readDoubleLE = function (e, t) {
	            return A(this, e, !0, t);
	          }, f.prototype.readDoubleBE = function (e, t) {
	            return A(this, e, !1, t);
	          }, f.prototype.writeUInt8 = function (e, t, n) {
	            n || (d(null != e, "missing value"), d(null != t, "missing offset"), d(t < this.length, "trying to write beyond buffer length"), Y(e, 255)), t >= this.length || (this[t] = e);
	          }, f.prototype.writeUInt16LE = function (e, t, n) {
	            s(this, e, t, !0, n);
	          }, f.prototype.writeUInt16BE = function (e, t, n) {
	            s(this, e, t, !1, n);
	          }, f.prototype.writeUInt32LE = function (e, t, n) {
	            l(this, e, t, !0, n);
	          }, f.prototype.writeUInt32BE = function (e, t, n) {
	            l(this, e, t, !1, n);
	          }, f.prototype.writeInt8 = function (e, t, n) {
	            n || (d(null != e, "missing value"), d(null != t, "missing offset"), d(t < this.length, "Trying to write beyond buffer length"), F(e, 127, -128)), t >= this.length || (0 <= e ? this.writeUInt8(e, t, n) : this.writeUInt8(255 + e + 1, t, n));
	          }, f.prototype.writeInt16LE = function (e, t, n) {
	            B(this, e, t, !0, n);
	          }, f.prototype.writeInt16BE = function (e, t, n) {
	            B(this, e, t, !1, n);
	          }, f.prototype.writeInt32LE = function (e, t, n) {
	            L(this, e, t, !0, n);
	          }, f.prototype.writeInt32BE = function (e, t, n) {
	            L(this, e, t, !1, n);
	          }, f.prototype.writeFloatLE = function (e, t, n) {
	            U(this, e, t, !0, n);
	          }, f.prototype.writeFloatBE = function (e, t, n) {
	            U(this, e, t, !1, n);
	          }, f.prototype.writeDoubleLE = function (e, t, n) {
	            x(this, e, t, !0, n);
	          }, f.prototype.writeDoubleBE = function (e, t, n) {
	            x(this, e, t, !1, n);
	          }, f.prototype.fill = function (e, t, n) {
	            if (t = t || 0, n = n || this.length, d("number" == typeof (e = "string" == typeof (e = e || 0) ? e.charCodeAt(0) : e) && !isNaN(e), "value is not a number"), d(t <= n, "end < start"), n !== t && 0 !== this.length) {
	              d(0 <= t && t < this.length, "start out of bounds"), d(0 <= n && n <= this.length, "end out of bounds");

	              for (var r = t; r < n; r++) this[r] = e;
	            }
	          }, f.prototype.inspect = function () {
	            for (var e = [], t = this.length, n = 0; n < t; n++) if (e[n] = k(this[n]), n === H.INSPECT_MAX_BYTES) {
	              e[n + 1] = "...";
	              break;
	            }

	            return "<Buffer " + e.join(" ") + ">";
	          }, f.prototype.toArrayBuffer = function () {
	            if ("undefined" == typeof Uint8Array) throw new Error("Buffer.toArrayBuffer not supported in this browser");
	            if (f._useTypedArrays) return new f(this).buffer;

	            for (var e = new Uint8Array(this.length), t = 0, n = e.length; t < n; t += 1) e[t] = this[t];

	            return e.buffer;
	          };
	          var t = f.prototype;

	          function S(e, t, n) {
	            return "number" != typeof e ? n : t <= (e = ~~e) ? t : 0 <= e || 0 <= (e += t) ? e : 0;
	          }

	          function j(e) {
	            return (e = ~~Math.ceil(+e)) < 0 ? 0 : e;
	          }

	          function C(e) {
	            return (Array.isArray || function (e) {
	              return "[object Array]" === Object.prototype.toString.call(e);
	            })(e);
	          }

	          function k(e) {
	            return e < 16 ? "0" + e.toString(16) : e.toString(16);
	          }

	          function T(e) {
	            for (var t = [], n = 0; n < e.length; n++) {
	              var r = e.charCodeAt(n);
	              if (r <= 127) t.push(e.charCodeAt(n));else for (var o = n, i = (55296 <= r && r <= 57343 && n++, encodeURIComponent(e.slice(o, n + 1)).substr(1).split("%")), u = 0; u < i.length; u++) t.push(parseInt(i[u], 16));
	            }

	            return t;
	          }

	          function M(e) {
	            return a.toByteArray(e);
	          }

	          function c(e, t, n, r) {
	            for (var o = 0; o < r && !(o + n >= t.length || o >= e.length); o++) t[o + n] = e[o];

	            return o;
	          }

	          function N(e) {
	            try {
	              return decodeURIComponent(e);
	            } catch (e) {
	              return String.fromCharCode(65533);
	            }
	          }

	          function Y(e, t) {
	            d("number" == typeof e, "cannot write a non-number as a number"), d(0 <= e, "specified a negative value for writing an unsigned value"), d(e <= t, "value is larger than maximum value for type"), d(Math.floor(e) === e, "value has a fractional component");
	          }

	          function F(e, t, n) {
	            d("number" == typeof e, "cannot write a non-number as a number"), d(e <= t, "value larger than maximum allowed value"), d(n <= e, "value smaller than minimum allowed value"), d(Math.floor(e) === e, "value has a fractional component");
	          }

	          function D(e, t, n) {
	            d("number" == typeof e, "cannot write a non-number as a number"), d(e <= t, "value larger than maximum allowed value"), d(n <= e, "value smaller than minimum allowed value");
	          }

	          function d(e, t) {
	            if (!e) throw new Error(t || "Failed assertion");
	          }

	          f._augment = function (e) {
	            return e._isBuffer = !0, e._get = e.get, e._set = e.set, e.get = t.get, e.set = t.set, e.write = t.write, e.toString = t.toString, e.toLocaleString = t.toString, e.toJSON = t.toJSON, e.copy = t.copy, e.slice = t.slice, e.readUInt8 = t.readUInt8, e.readUInt16LE = t.readUInt16LE, e.readUInt16BE = t.readUInt16BE, e.readUInt32LE = t.readUInt32LE, e.readUInt32BE = t.readUInt32BE, e.readInt8 = t.readInt8, e.readInt16LE = t.readInt16LE, e.readInt16BE = t.readInt16BE, e.readInt32LE = t.readInt32LE, e.readInt32BE = t.readInt32BE, e.readFloatLE = t.readFloatLE, e.readFloatBE = t.readFloatBE, e.readDoubleLE = t.readDoubleLE, e.readDoubleBE = t.readDoubleBE, e.writeUInt8 = t.writeUInt8, e.writeUInt16LE = t.writeUInt16LE, e.writeUInt16BE = t.writeUInt16BE, e.writeUInt32LE = t.writeUInt32LE, e.writeUInt32BE = t.writeUInt32BE, e.writeInt8 = t.writeInt8, e.writeInt16LE = t.writeInt16LE, e.writeInt16BE = t.writeInt16BE, e.writeInt32LE = t.writeInt32LE, e.writeInt32BE = t.writeInt32BE, e.writeFloatLE = t.writeFloatLE, e.writeFloatBE = t.writeFloatBE, e.writeDoubleLE = t.writeDoubleLE, e.writeDoubleBE = t.writeDoubleBE, e.fill = t.fill, e.inspect = t.inspect, e.toArrayBuffer = t.toArrayBuffer, e;
	          };
	        }.call(this, O("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, O("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/buffer/index.js", "/node_modules/gulp-browserify/node_modules/buffer");
	      }, {
	        "base64-js": 2,
	        buffer: 3,
	        ieee754: 10,
	        lYpoI2: 11
	      }],
	      4: [function (c, d, e) {
	        !function (e, t, a, n, r, o, i, u, s) {
	          var a = c("buffer").Buffer,
	              f = 4,
	              l = new a(f);
	          l.fill(0);
	          d.exports = {
	            hash: function (e, t, n, r) {
	              for (var o = t(function (e, t) {
	                e.length % f != 0 && (n = e.length + (f - e.length % f), e = a.concat([e, l], n));

	                for (var n, r = [], o = t ? e.readInt32BE : e.readInt32LE, i = 0; i < e.length; i += f) r.push(o.call(e, i));

	                return r;
	              }(e = a.isBuffer(e) ? e : new a(e), r), 8 * e.length), t = r, i = new a(n), u = t ? i.writeInt32BE : i.writeInt32LE, s = 0; s < o.length; s++) u.call(i, o[s], 4 * s, !0);

	              return i;
	            }
	          };
	        }.call(this, c("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, c("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/crypto-browserify/helpers.js", "/node_modules/gulp-browserify/node_modules/crypto-browserify");
	      }, {
	        buffer: 3,
	        lYpoI2: 11
	      }],
	      5: [function (v, e, _) {
	        !function (l, c, u, d, h, p, g, y, w) {
	          var u = v("buffer").Buffer,
	              e = v("./sha"),
	              t = v("./sha256"),
	              n = v("./rng"),
	              b = {
	            sha1: e,
	            sha256: t,
	            md5: v("./md5")
	          },
	              s = 64,
	              a = new u(s);

	          function r(e, n) {
	            var r = b[e = e || "sha1"],
	                o = [];
	            return r || i("algorithm:", e, "is not yet supported"), {
	              update: function (e) {
	                return u.isBuffer(e) || (e = new u(e)), o.push(e), e.length, this;
	              },
	              digest: function (e) {
	                var t = u.concat(o),
	                    t = n ? function (e, t, n) {
	                  u.isBuffer(t) || (t = new u(t)), u.isBuffer(n) || (n = new u(n)), t.length > s ? t = e(t) : t.length < s && (t = u.concat([t, a], s));

	                  for (var r = new u(s), o = new u(s), i = 0; i < s; i++) r[i] = 54 ^ t[i], o[i] = 92 ^ t[i];

	                  return n = e(u.concat([r, n])), e(u.concat([o, n]));
	                }(r, n, t) : r(t);
	                return o = null, e ? t.toString(e) : t;
	              }
	            };
	          }

	          function i() {
	            var e = [].slice.call(arguments).join(" ");
	            throw new Error([e, "we accept pull requests", "http://github.com/dominictarr/crypto-browserify"].join("\n"));
	          }

	          a.fill(0), _.createHash = function (e) {
	            return r(e);
	          }, _.createHmac = r, _.randomBytes = function (e, t) {
	            if (!t || !t.call) return new u(n(e));

	            try {
	              t.call(this, void 0, new u(n(e)));
	            } catch (e) {
	              t(e);
	            }
	          };

	          var o,
	              f = ["createCredentials", "createCipher", "createCipheriv", "createDecipher", "createDecipheriv", "createSign", "createVerify", "createDiffieHellman", "pbkdf2"],
	              m = function (e) {
	            _[e] = function () {
	              i("sorry,", e, "is not implemented yet");
	            };
	          };

	          for (o in f) m(f[o]);
	        }.call(this, v("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, v("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/crypto-browserify/index.js", "/node_modules/gulp-browserify/node_modules/crypto-browserify");
	      }, {
	        "./md5": 6,
	        "./rng": 7,
	        "./sha": 8,
	        "./sha256": 9,
	        buffer: 3,
	        lYpoI2: 11
	      }],
	      6: [function (w, b, e) {
	        !function (e, r, o, i, u, a, f, l, y) {
	          var t = w("./helpers");

	          function n(e, t) {
	            e[t >> 5] |= 128 << t % 32, e[14 + (t + 64 >>> 9 << 4)] = t;

	            for (var n = 1732584193, r = -271733879, o = -1732584194, i = 271733878, u = 0; u < e.length; u += 16) {
	              var s = n,
	                  a = r,
	                  f = o,
	                  l = i,
	                  n = c(n, r, o, i, e[u + 0], 7, -680876936),
	                  i = c(i, n, r, o, e[u + 1], 12, -389564586),
	                  o = c(o, i, n, r, e[u + 2], 17, 606105819),
	                  r = c(r, o, i, n, e[u + 3], 22, -1044525330);
	              n = c(n, r, o, i, e[u + 4], 7, -176418897), i = c(i, n, r, o, e[u + 5], 12, 1200080426), o = c(o, i, n, r, e[u + 6], 17, -1473231341), r = c(r, o, i, n, e[u + 7], 22, -45705983), n = c(n, r, o, i, e[u + 8], 7, 1770035416), i = c(i, n, r, o, e[u + 9], 12, -1958414417), o = c(o, i, n, r, e[u + 10], 17, -42063), r = c(r, o, i, n, e[u + 11], 22, -1990404162), n = c(n, r, o, i, e[u + 12], 7, 1804603682), i = c(i, n, r, o, e[u + 13], 12, -40341101), o = c(o, i, n, r, e[u + 14], 17, -1502002290), n = d(n, r = c(r, o, i, n, e[u + 15], 22, 1236535329), o, i, e[u + 1], 5, -165796510), i = d(i, n, r, o, e[u + 6], 9, -1069501632), o = d(o, i, n, r, e[u + 11], 14, 643717713), r = d(r, o, i, n, e[u + 0], 20, -373897302), n = d(n, r, o, i, e[u + 5], 5, -701558691), i = d(i, n, r, o, e[u + 10], 9, 38016083), o = d(o, i, n, r, e[u + 15], 14, -660478335), r = d(r, o, i, n, e[u + 4], 20, -405537848), n = d(n, r, o, i, e[u + 9], 5, 568446438), i = d(i, n, r, o, e[u + 14], 9, -1019803690), o = d(o, i, n, r, e[u + 3], 14, -187363961), r = d(r, o, i, n, e[u + 8], 20, 1163531501), n = d(n, r, o, i, e[u + 13], 5, -1444681467), i = d(i, n, r, o, e[u + 2], 9, -51403784), o = d(o, i, n, r, e[u + 7], 14, 1735328473), n = h(n, r = d(r, o, i, n, e[u + 12], 20, -1926607734), o, i, e[u + 5], 4, -378558), i = h(i, n, r, o, e[u + 8], 11, -2022574463), o = h(o, i, n, r, e[u + 11], 16, 1839030562), r = h(r, o, i, n, e[u + 14], 23, -35309556), n = h(n, r, o, i, e[u + 1], 4, -1530992060), i = h(i, n, r, o, e[u + 4], 11, 1272893353), o = h(o, i, n, r, e[u + 7], 16, -155497632), r = h(r, o, i, n, e[u + 10], 23, -1094730640), n = h(n, r, o, i, e[u + 13], 4, 681279174), i = h(i, n, r, o, e[u + 0], 11, -358537222), o = h(o, i, n, r, e[u + 3], 16, -722521979), r = h(r, o, i, n, e[u + 6], 23, 76029189), n = h(n, r, o, i, e[u + 9], 4, -640364487), i = h(i, n, r, o, e[u + 12], 11, -421815835), o = h(o, i, n, r, e[u + 15], 16, 530742520), n = p(n, r = h(r, o, i, n, e[u + 2], 23, -995338651), o, i, e[u + 0], 6, -198630844), i = p(i, n, r, o, e[u + 7], 10, 1126891415), o = p(o, i, n, r, e[u + 14], 15, -1416354905), r = p(r, o, i, n, e[u + 5], 21, -57434055), n = p(n, r, o, i, e[u + 12], 6, 1700485571), i = p(i, n, r, o, e[u + 3], 10, -1894986606), o = p(o, i, n, r, e[u + 10], 15, -1051523), r = p(r, o, i, n, e[u + 1], 21, -2054922799), n = p(n, r, o, i, e[u + 8], 6, 1873313359), i = p(i, n, r, o, e[u + 15], 10, -30611744), o = p(o, i, n, r, e[u + 6], 15, -1560198380), r = p(r, o, i, n, e[u + 13], 21, 1309151649), n = p(n, r, o, i, e[u + 4], 6, -145523070), i = p(i, n, r, o, e[u + 11], 10, -1120210379), o = p(o, i, n, r, e[u + 2], 15, 718787259), r = p(r, o, i, n, e[u + 9], 21, -343485551), n = g(n, s), r = g(r, a), o = g(o, f), i = g(i, l);
	            }

	            return Array(n, r, o, i);
	          }

	          function s(e, t, n, r, o, i) {
	            return g((t = g(g(t, e), g(r, i))) << o | t >>> 32 - o, n);
	          }

	          function c(e, t, n, r, o, i, u) {
	            return s(t & n | ~t & r, e, t, o, i, u);
	          }

	          function d(e, t, n, r, o, i, u) {
	            return s(t & r | n & ~r, e, t, o, i, u);
	          }

	          function h(e, t, n, r, o, i, u) {
	            return s(t ^ n ^ r, e, t, o, i, u);
	          }

	          function p(e, t, n, r, o, i, u) {
	            return s(n ^ (t | ~r), e, t, o, i, u);
	          }

	          function g(e, t) {
	            var n = (65535 & e) + (65535 & t);
	            return (e >> 16) + (t >> 16) + (n >> 16) << 16 | 65535 & n;
	          }

	          b.exports = function (e) {
	            return t.hash(e, n, 16);
	          };
	        }.call(this, w("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, w("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/crypto-browserify/md5.js", "/node_modules/gulp-browserify/node_modules/crypto-browserify");
	      }, {
	        "./helpers": 4,
	        buffer: 3,
	        lYpoI2: 11
	      }],
	      7: [function (e, l, t) {
	        !function (e, t, n, r, o, i, u, s, f) {

	          l.exports = function (e) {
	            for (var t, n = new Array(e), r = 0; r < e; r++) 0 == (3 & r) && (t = 4294967296 * Math.random()), n[r] = t >>> ((3 & r) << 3) & 255;

	            return n;
	          };
	        }.call(this, e("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, e("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/crypto-browserify/rng.js", "/node_modules/gulp-browserify/node_modules/crypto-browserify");
	      }, {
	        buffer: 3,
	        lYpoI2: 11
	      }],
	      8: [function (c, d, e) {
	        !function (e, t, n, r, o, s, a, f, l) {
	          var i = c("./helpers");

	          function u(l, c) {
	            l[c >> 5] |= 128 << 24 - c % 32, l[15 + (c + 64 >> 9 << 4)] = c;

	            for (var e, t, n, r = Array(80), o = 1732584193, i = -271733879, u = -1732584194, s = 271733878, d = -1009589776, h = 0; h < l.length; h += 16) {
	              for (var p = o, g = i, y = u, w = s, b = d, a = 0; a < 80; a++) {
	                r[a] = a < 16 ? l[h + a] : v(r[a - 3] ^ r[a - 8] ^ r[a - 14] ^ r[a - 16], 1);
	                var f = m(m(v(o, 5), (f = i, t = u, n = s, (e = a) < 20 ? f & t | ~f & n : !(e < 40) && e < 60 ? f & t | f & n | t & n : f ^ t ^ n)), m(m(d, r[a]), (e = a) < 20 ? 1518500249 : e < 40 ? 1859775393 : e < 60 ? -1894007588 : -899497514)),
	                    d = s,
	                    s = u,
	                    u = v(i, 30),
	                    i = o,
	                    o = f;
	              }

	              o = m(o, p), i = m(i, g), u = m(u, y), s = m(s, w), d = m(d, b);
	            }

	            return Array(o, i, u, s, d);
	          }

	          function m(e, t) {
	            var n = (65535 & e) + (65535 & t);
	            return (e >> 16) + (t >> 16) + (n >> 16) << 16 | 65535 & n;
	          }

	          function v(e, t) {
	            return e << t | e >>> 32 - t;
	          }

	          d.exports = function (e) {
	            return i.hash(e, u, 20, !0);
	          };
	        }.call(this, c("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, c("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/crypto-browserify/sha.js", "/node_modules/gulp-browserify/node_modules/crypto-browserify");
	      }, {
	        "./helpers": 4,
	        buffer: 3,
	        lYpoI2: 11
	      }],
	      9: [function (c, d, e) {
	        !function (e, t, n, r, u, s, a, f, l) {
	          function b(e, t) {
	            var n = (65535 & e) + (65535 & t);
	            return (e >> 16) + (t >> 16) + (n >> 16) << 16 | 65535 & n;
	          }

	          function o(e, l) {
	            var c,
	                d = new Array(1116352408, 1899447441, 3049323471, 3921009573, 961987163, 1508970993, 2453635748, 2870763221, 3624381080, 310598401, 607225278, 1426881987, 1925078388, 2162078206, 2614888103, 3248222580, 3835390401, 4022224774, 264347078, 604807628, 770255983, 1249150122, 1555081692, 1996064986, 2554220882, 2821834349, 2952996808, 3210313671, 3336571891, 3584528711, 113926993, 338241895, 666307205, 773529912, 1294757372, 1396182291, 1695183700, 1986661051, 2177026350, 2456956037, 2730485921, 2820302411, 3259730800, 3345764771, 3516065817, 3600352804, 4094571909, 275423344, 430227734, 506948616, 659060556, 883997877, 958139571, 1322822218, 1537002063, 1747873779, 1955562222, 2024104815, 2227730452, 2361852424, 2428436474, 2756734187, 3204031479, 3329325298),
	                t = new Array(1779033703, 3144134277, 1013904242, 2773480762, 1359893119, 2600822924, 528734635, 1541459225),
	                n = new Array(64);
	            e[l >> 5] |= 128 << 24 - l % 32, e[15 + (l + 64 >> 9 << 4)] = l;

	            for (var r, o, h = 0; h < e.length; h += 16) {
	              for (var i = t[0], u = t[1], s = t[2], p = t[3], a = t[4], g = t[5], y = t[6], w = t[7], f = 0; f < 64; f++) n[f] = f < 16 ? e[f + h] : b(b(b((o = n[f - 2], m(o, 17) ^ m(o, 19) ^ v(o, 10)), n[f - 7]), (o = n[f - 15], m(o, 7) ^ m(o, 18) ^ v(o, 3))), n[f - 16]), c = b(b(b(b(w, m(o = a, 6) ^ m(o, 11) ^ m(o, 25)), a & g ^ ~a & y), d[f]), n[f]), r = b(m(r = i, 2) ^ m(r, 13) ^ m(r, 22), i & u ^ i & s ^ u & s), w = y, y = g, g = a, a = b(p, c), p = s, s = u, u = i, i = b(c, r);

	              t[0] = b(i, t[0]), t[1] = b(u, t[1]), t[2] = b(s, t[2]), t[3] = b(p, t[3]), t[4] = b(a, t[4]), t[5] = b(g, t[5]), t[6] = b(y, t[6]), t[7] = b(w, t[7]);
	            }

	            return t;
	          }

	          var i = c("./helpers"),
	              m = function (e, t) {
	            return e >>> t | e << 32 - t;
	          },
	              v = function (e, t) {
	            return e >>> t;
	          };

	          d.exports = function (e) {
	            return i.hash(e, o, 32, !0);
	          };
	        }.call(this, c("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, c("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/crypto-browserify/sha256.js", "/node_modules/gulp-browserify/node_modules/crypto-browserify");
	      }, {
	        "./helpers": 4,
	        buffer: 3,
	        lYpoI2: 11
	      }],
	      10: [function (e, t, f) {
	        !function (e, t, n, r, o, i, u, s, a) {
	          f.read = function (e, t, n, r, o) {
	            var i,
	                u,
	                l = 8 * o - r - 1,
	                c = (1 << l) - 1,
	                d = c >> 1,
	                s = -7,
	                a = n ? o - 1 : 0,
	                f = n ? -1 : 1,
	                o = e[t + a];

	            for (a += f, i = o & (1 << -s) - 1, o >>= -s, s += l; 0 < s; i = 256 * i + e[t + a], a += f, s -= 8);

	            for (u = i & (1 << -s) - 1, i >>= -s, s += r; 0 < s; u = 256 * u + e[t + a], a += f, s -= 8);

	            if (0 === i) i = 1 - d;else {
	              if (i === c) return u ? NaN : 1 / 0 * (o ? -1 : 1);
	              u += Math.pow(2, r), i -= d;
	            }
	            return (o ? -1 : 1) * u * Math.pow(2, i - r);
	          }, f.write = function (e, t, l, n, r, c) {
	            var o,
	                i,
	                u = 8 * c - r - 1,
	                s = (1 << u) - 1,
	                a = s >> 1,
	                d = 23 === r ? Math.pow(2, -24) - Math.pow(2, -77) : 0,
	                f = n ? 0 : c - 1,
	                h = n ? 1 : -1,
	                c = t < 0 || 0 === t && 1 / t < 0 ? 1 : 0;

	            for (t = Math.abs(t), isNaN(t) || t === 1 / 0 ? (i = isNaN(t) ? 1 : 0, o = s) : (o = Math.floor(Math.log(t) / Math.LN2), t * (n = Math.pow(2, -o)) < 1 && (o--, n *= 2), 2 <= (t += 1 <= o + a ? d / n : d * Math.pow(2, 1 - a)) * n && (o++, n /= 2), s <= o + a ? (i = 0, o = s) : 1 <= o + a ? (i = (t * n - 1) * Math.pow(2, r), o += a) : (i = t * Math.pow(2, a - 1) * Math.pow(2, r), o = 0)); 8 <= r; e[l + f] = 255 & i, f += h, i /= 256, r -= 8);

	            for (o = o << r | i, u += r; 0 < u; e[l + f] = 255 & o, f += h, o /= 256, u -= 8);

	            e[l + f - h] |= 128 * c;
	          };
	        }.call(this, e("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, e("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/ieee754/index.js", "/node_modules/gulp-browserify/node_modules/ieee754");
	      }, {
	        buffer: 3,
	        lYpoI2: 11
	      }],
	      11: [function (e, h, t) {
	        !function (e, t, n, r, o, f, l, c, d) {
	          var i, u, s;

	          function a() {}

	          (e = h.exports = {}).nextTick = (u = "undefined" != typeof window && window.setImmediate, s = "undefined" != typeof window && window.postMessage && window.addEventListener, u ? function (e) {
	            return window.setImmediate(e);
	          } : s ? (i = [], window.addEventListener("message", function (e) {
	            var t = e.source;
	            t !== window && null !== t || "process-tick" !== e.data || (e.stopPropagation(), 0 < i.length && i.shift()());
	          }, !0), function (e) {
	            i.push(e), window.postMessage("process-tick", "*");
	          }) : function (e) {
	            setTimeout(e, 0);
	          }), e.title = "browser", e.browser = !0, e.env = {}, e.argv = [], e.on = a, e.addListener = a, e.once = a, e.off = a, e.removeListener = a, e.removeAllListeners = a, e.emit = a, e.binding = function (e) {
	            throw new Error("process.binding is not supported");
	          }, e.cwd = function () {
	            return "/";
	          }, e.chdir = function (e) {
	            throw new Error("process.chdir is not supported");
	          };
	        }.call(this, e("lYpoI2"), "undefined" != typeof self ? self : "undefined" != typeof window ? window : {}, e("buffer").Buffer, arguments[3], arguments[4], arguments[5], arguments[6], "/node_modules/gulp-browserify/node_modules/process/browser.js", "/node_modules/gulp-browserify/node_modules/process");
	      }, {
	        buffer: 3,
	        lYpoI2: 11
	      }]
	    }, {}, [1])(1);
	  });
	})(object_hash);

	var hash = object_hash.exports;

	function getFromToIndex(xs, range) {
	  let {
	    from,
	    to
	  } = range;

	  if (from === undefined) {
	    from = xs[0];
	  }

	  if (to === undefined) {
	    to = xs[xs.length - 1];
	  }

	  return {
	    fromIndex: xFindClosestIndex(xs, from),
	    toIndex: xFindClosestIndex(xs, to)
	  };
	}

	function integration(matrix, normalizedTarget) {
	  let range = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
	  let fromToIndex = getFromToIndex(normalizedTarget.x, range);
	  let targetValue = xSum(normalizedTarget.y, fromToIndex);
	  let values = matrix.map(row => xSum(row, fromToIndex));

	  for (let i = 0; i < matrix.length; i++) {
	    let factor = targetValue / values[i];
	    matrix[i] = xMultiply(matrix[i], factor);
	  }
	}

	function max(matrix, normalizedTarget) {
	  let range = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
	  let fromToIndex = getFromToIndex(normalizedTarget.x, range);
	  let targetValue = xMaxValue(normalizedTarget.y, fromToIndex);
	  let values = matrix.map(row => xMaxValue(row, fromToIndex));

	  for (let i = 0; i < matrix.length; i++) {
	    let factor = targetValue / values[i];
	    matrix[i] = xMultiply(matrix[i], factor);
	  }
	}

	function min(matrix, normalizedTarget) {
	  let range = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
	  let fromToIndex = getFromToIndex(normalizedTarget.x, range);
	  let targetValue = xMinValue(normalizedTarget.y, fromToIndex);
	  let values = matrix.map(row => xMinValue(row, fromToIndex));

	  for (let i = 0; i < matrix.length; i++) {
	    let factor = targetValue / values[i];
	    matrix[i] = xMultiply(matrix[i], factor);
	  }
	}

	function minMax(matrix, normalizedTarget) {
	  let range = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
	  let fromToIndex = getFromToIndex(normalizedTarget.x, range);
	  let targetValue = {
	    min: xMinValue(normalizedTarget.y, fromToIndex),
	    max: xMaxValue(normalizedTarget.y, fromToIndex)
	  };
	  let deltaTarget = targetValue.max - targetValue.min;
	  let minTarget = targetValue.min;
	  let values = matrix.map(row => {
	    return {
	      min: xMinValue(row, fromToIndex),
	      max: xMaxValue(row, fromToIndex)
	    };
	  });

	  for (let i = 0; i < matrix.length; i++) {
	    let deltaSource = values[i].max - values[i].min;
	    let minSource = values[i].min;
	    let newData = [];

	    for (let j = 0; j < normalizedTarget.y.length; j++) {
	      newData.push((matrix[i][j] - minSource) / deltaSource * deltaTarget + minTarget);
	    }

	    matrix[i] = newData;
	  }
	}

	/**
	 * Allows to calculate relative intensity between normalized spectra
	 * @param {SpectraProcessor} spectraProcessor
	 * @param {object}   [options={}] scale spectra based on various parameters
	 * @param {Array}    [options.ids] ids of selected spectra
	 * @param {Array}    [options.filters=[]] Array of object containing {name:'', options:''}
	 * @param {object}   [options.scale={}] object containing the options for rescaling
	 * @param {string}   [options.scale.targetID=spectra[0].id]
	 * @param {string}   [options.scale.method='max'] min, max, integration, minMax
	 * @param {Array}    [options.scale.range] from - to to apply the method and rescale
	 * @param {boolean}  [options.scale.relative=false]
	 * @param {Array}    [options.ranges] Array of object containing {from:'', to:'', label:''}
	 * @param {Array}    [options.calculations] Array of object containing {label:'', formula:''}
	 * @returns {object} { ids:[], matrix:[Array], meta:[object], x:[], ranges:[object] }
	 */

	let cache = {};
	function getPostProcessedData(spectraProcessor) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	  /**
	   * could implement a cache if all the options are identical and the normalized data is identical as well
	   * in order ot check if the normalized data are identical we should check if the normalized array of all the spectra are identical
	   * Because we don't make in-place modification when creating normalized array we can check if the 'pointer' to the object
	   * is identical
	   */
	  const optionsHash = hash(options);
	  if (!spectraProcessor.spectra || !spectraProcessor.spectra[0]) return {};
	  const {
	    scale = {},
	    ids,
	    ranges,
	    calculations,
	    filters = []
	  } = options;
	  const {
	    range,
	    targetID,
	    relative,
	    method = ''
	  } = scale;
	  let spectra = spectraProcessor.getSpectra(ids); // are we able to reuse the cache ?
	  // we can if the normalized information didn't change and optionsHash is the same

	  if (cache.optionsHash === optionsHash) {
	    let validCache = true;

	    for (let spectrum of spectra) {
	      if (!cache.weakMap.get(spectrum.normalized)) validCache = false;
	    }

	    if (validCache) return cache;
	  }

	  const weakMap = new WeakMap();

	  for (let spectrum of spectra) {
	    weakMap.set(spectrum.normalized, true);
	  }

	  let normalizedData = getNormalizedData(spectra);

	  for (let filter of filters) {
	    switch (filter.name) {
	      case 'pqn':
	        {
	          normalizedData.matrix = matrixPQN(normalizedData.matrix, filter.options).data;
	          break;
	        }

	      case 'centerMean':
	        {
	          normalizedData.matrix = matrixCenterZMean(normalizedData.matrix);
	          break;
	        }

	      case 'rescale':
	        {
	          normalizedData.matrix = matrixZRescale(normalizedData.matrix, filter.options);
	          break;
	        }

	      case '':
	      case undefined:
	        break;

	      default:
	        throw new Error(`Unknown matrix filter name: ${filter.name}`);
	    }
	  }

	  let normalizedTarget = targetID ? spectraProcessor.getSpectrum(targetID).normalized : spectraProcessor.spectra[0].normalized;

	  if (method) {
	    switch (method.toLowerCase()) {
	      case 'min':
	        min(normalizedData.matrix, normalizedTarget, range);
	        break;

	      case 'max':
	        max(normalizedData.matrix, normalizedTarget, range);
	        break;

	      case 'minmax':
	        minMax(normalizedData.matrix, normalizedTarget, range);
	        break;

	      case 'integration':
	        integration(normalizedData.matrix, normalizedTarget, range);
	        break;

	      default:
	        throw new Error(`getPostProcessedData: unknown method: ${method}`);
	    }
	  }

	  if (relative) {
	    for (let i = 0; i < normalizedData.matrix.length; i++) {
	      normalizedData.matrix[i] = xSubtract(normalizedData.matrix[i], normalizedTarget.y);
	    }
	  }

	  if (ranges) {
	    normalizedData.ranges = [];

	    for (let i = 0; i < normalizedData.matrix.length; i++) {
	      let rangesCopy = JSON.parse(JSON.stringify(ranges));
	      let yNormalized = normalizedData.matrix[i];
	      let resultRanges = {};
	      normalizedData.ranges.push(resultRanges);

	      for (let currentRange of rangesCopy) {
	        if (currentRange.label) {
	          let fromToIndex = getFromToIndex(normalizedTarget.x, currentRange);
	          let deltaX = normalizedTarget.x[1] - normalizedTarget.x[0];
	          currentRange.integration = xSum(yNormalized, fromToIndex) * deltaX;
	          currentRange.maxPoint = xyMaxYPoint({
	            x: normalizedData.x,
	            y: yNormalized
	          }, fromToIndex);
	          resultRanges[currentRange.label] = currentRange;
	        }
	      }
	    }
	  }

	  if (calculations && normalizedData.ranges) {
	    normalizedData.calculations = normalizedData.ranges.map(() => {
	      return {};
	    });
	    const parameters = Object.keys(normalizedData.ranges[0]);

	    for (let calculation of calculations) {
	      // eslint-disable-next-line no-new-func
	      const callback = new Function(...parameters, `return ${calculation.formula}`);

	      for (let i = 0; i < normalizedData.ranges.length; i++) {
	        let oneRanges = normalizedData.ranges[i];
	        let values = parameters.map(key => oneRanges[key].integration);
	        normalizedData.calculations[i][calculation.label] = callback(...values);
	      }
	    }
	  }

	  cache = { ...normalizedData,
	    optionsHash,
	    weakMap
	  };
	  return cache;
	}

	/**
	 * @private
	 * @param {SpectraProcessor} spectraProcessor
	 * @param {object} [options={}]
	 * @param {string} [options.fs='\t'] field separator
	 * @param {string} [options.rs='\n'] record (line) separator
	 * @param {object} [options.postProcessing={}] post processing options
	 */

	function getPostProcessedText(spectraProcessor) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  let {
	    fs = '\t',
	    rs = '\n',
	    postProcessing: postProcessingOptions = {}
	  } = options;
	  return convertToText(getPostProcessedData(spectraProcessor, postProcessingOptions), {
	    rs,
	    fs
	  });
	}

	var array = {};

	(function (exports) {

	  function compareNumbers(a, b) {
	    return a - b;
	  }
	  /**
	   * Computes the sum of the given values
	   * @param {Array} values
	   * @returns {number}
	   */


	  exports.sum = function sum(values) {
	    var sum = 0;

	    for (var i = 0; i < values.length; i++) {
	      sum += values[i];
	    }

	    return sum;
	  };
	  /**
	   * Computes the maximum of the given values
	   * @param {Array} values
	   * @returns {number}
	   */


	  exports.max = function max(values) {
	    var max = values[0];
	    var l = values.length;

	    for (var i = 1; i < l; i++) {
	      if (values[i] > max) max = values[i];
	    }

	    return max;
	  };
	  /**
	   * Computes the minimum of the given values
	   * @param {Array} values
	   * @returns {number}
	   */


	  exports.min = function min(values) {
	    var min = values[0];
	    var l = values.length;

	    for (var i = 1; i < l; i++) {
	      if (values[i] < min) min = values[i];
	    }

	    return min;
	  };
	  /**
	   * Computes the min and max of the given values
	   * @param {Array} values
	   * @returns {{min: number, max: number}}
	   */


	  exports.minMax = function minMax(values) {
	    var min = values[0];
	    var max = values[0];
	    var l = values.length;

	    for (var i = 1; i < l; i++) {
	      if (values[i] < min) min = values[i];
	      if (values[i] > max) max = values[i];
	    }

	    return {
	      min: min,
	      max: max
	    };
	  };
	  /**
	   * Computes the arithmetic mean of the given values
	   * @param {Array} values
	   * @returns {number}
	   */


	  exports.arithmeticMean = function arithmeticMean(values) {
	    var sum = 0;
	    var l = values.length;

	    for (var i = 0; i < l; i++) {
	      sum += values[i];
	    }

	    return sum / l;
	  };
	  /**
	   * {@link arithmeticMean}
	   */


	  exports.mean = exports.arithmeticMean;
	  /**
	   * Computes the geometric mean of the given values
	   * @param {Array} values
	   * @returns {number}
	   */

	  exports.geometricMean = function geometricMean(values) {
	    var mul = 1;
	    var l = values.length;

	    for (var i = 0; i < l; i++) {
	      mul *= values[i];
	    }

	    return Math.pow(mul, 1 / l);
	  };
	  /**
	   * Computes the mean of the log of the given values
	   * If the return value is exponentiated, it gives the same result as the
	   * geometric mean.
	   * @param {Array} values
	   * @returns {number}
	   */


	  exports.logMean = function logMean(values) {
	    var lnsum = 0;
	    var l = values.length;

	    for (var i = 0; i < l; i++) {
	      lnsum += Math.log(values[i]);
	    }

	    return lnsum / l;
	  };
	  /**
	   * Computes the weighted grand mean for a list of means and sample sizes
	   * @param {Array} means - Mean values for each set of samples
	   * @param {Array} samples - Number of original values for each set of samples
	   * @returns {number}
	   */


	  exports.grandMean = function grandMean(means, samples) {
	    var sum = 0;
	    var n = 0;
	    var l = means.length;

	    for (var i = 0; i < l; i++) {
	      sum += samples[i] * means[i];
	      n += samples[i];
	    }

	    return sum / n;
	  };
	  /**
	   * Computes the truncated mean of the given values using a given percentage
	   * @param {Array} values
	   * @param {number} percent - The percentage of values to keep (range: [0,1])
	   * @param {boolean} [alreadySorted=false]
	   * @returns {number}
	   */


	  exports.truncatedMean = function truncatedMean(values, percent, alreadySorted) {
	    if (alreadySorted === undefined) alreadySorted = false;

	    if (!alreadySorted) {
	      values = [].concat(values).sort(compareNumbers);
	    }

	    var l = values.length;
	    var k = Math.floor(l * percent);
	    var sum = 0;

	    for (var i = k; i < l - k; i++) {
	      sum += values[i];
	    }

	    return sum / (l - 2 * k);
	  };
	  /**
	   * Computes the harmonic mean of the given values
	   * @param {Array} values
	   * @returns {number}
	   */


	  exports.harmonicMean = function harmonicMean(values) {
	    var sum = 0;
	    var l = values.length;

	    for (var i = 0; i < l; i++) {
	      if (values[i] === 0) {
	        throw new RangeError('value at index ' + i + 'is zero');
	      }

	      sum += 1 / values[i];
	    }

	    return l / sum;
	  };
	  /**
	   * Computes the contraharmonic mean of the given values
	   * @param {Array} values
	   * @returns {number}
	   */


	  exports.contraHarmonicMean = function contraHarmonicMean(values) {
	    var r1 = 0;
	    var r2 = 0;
	    var l = values.length;

	    for (var i = 0; i < l; i++) {
	      r1 += values[i] * values[i];
	      r2 += values[i];
	    }

	    if (r2 < 0) {
	      throw new RangeError('sum of values is negative');
	    }

	    return r1 / r2;
	  };
	  /**
	   * Computes the median of the given values
	   * @param {Array} values
	   * @param {boolean} [alreadySorted=false]
	   * @returns {number}
	   */


	  exports.median = function median(values, alreadySorted) {
	    if (alreadySorted === undefined) alreadySorted = false;

	    if (!alreadySorted) {
	      values = [].concat(values).sort(compareNumbers);
	    }

	    var l = values.length;
	    var half = Math.floor(l / 2);

	    if (l % 2 === 0) {
	      return (values[half - 1] + values[half]) * 0.5;
	    } else {
	      return values[half];
	    }
	  };
	  /**
	   * Computes the variance of the given values
	   * @param {Array} values
	   * @param {boolean} [unbiased=true] - if true, divide by (n-1); if false, divide by n.
	   * @returns {number}
	   */


	  exports.variance = function variance(values, unbiased) {
	    if (unbiased === undefined) unbiased = true;
	    var theMean = exports.mean(values);
	    var theVariance = 0;
	    var l = values.length;

	    for (var i = 0; i < l; i++) {
	      var x = values[i] - theMean;
	      theVariance += x * x;
	    }

	    if (unbiased) {
	      return theVariance / (l - 1);
	    } else {
	      return theVariance / l;
	    }
	  };
	  /**
	   * Computes the standard deviation of the given values
	   * @param {Array} values
	   * @param {boolean} [unbiased=true] - if true, divide by (n-1); if false, divide by n.
	   * @returns {number}
	   */


	  exports.standardDeviation = function standardDeviation(values, unbiased) {
	    return Math.sqrt(exports.variance(values, unbiased));
	  };

	  exports.standardError = function standardError(values) {
	    return exports.standardDeviation(values) / Math.sqrt(values.length);
	  };
	  /**
	   * IEEE Transactions on biomedical engineering, vol. 52, no. 1, january 2005, p. 76-
	   * Calculate the standard deviation via the Median of the absolute deviation
	   *  The formula for the standard deviation only holds for Gaussian random variables.
	   * @returns {{mean: number, stdev: number}}
	   */


	  exports.robustMeanAndStdev = function robustMeanAndStdev(y) {
	    var mean = 0,
	        stdev = 0;
	    var length = y.length,
	        i = 0;

	    for (i = 0; i < length; i++) {
	      mean += y[i];
	    }

	    mean /= length;
	    var averageDeviations = new Array(length);

	    for (i = 0; i < length; i++) averageDeviations[i] = Math.abs(y[i] - mean);

	    averageDeviations.sort(compareNumbers);

	    if (length % 2 === 1) {
	      stdev = averageDeviations[(length - 1) / 2] / 0.6745;
	    } else {
	      stdev = 0.5 * (averageDeviations[length / 2] + averageDeviations[length / 2 - 1]) / 0.6745;
	    }

	    return {
	      mean: mean,
	      stdev: stdev
	    };
	  };

	  exports.quartiles = function quartiles(values, alreadySorted) {
	    if (typeof alreadySorted === 'undefined') alreadySorted = false;

	    if (!alreadySorted) {
	      values = [].concat(values).sort(compareNumbers);
	    }

	    var quart = values.length / 4;
	    var q1 = values[Math.ceil(quart) - 1];
	    var q2 = exports.median(values, true);
	    var q3 = values[Math.ceil(quart * 3) - 1];
	    return {
	      q1: q1,
	      q2: q2,
	      q3: q3
	    };
	  };

	  exports.pooledStandardDeviation = function pooledStandardDeviation(samples, unbiased) {
	    return Math.sqrt(exports.pooledVariance(samples, unbiased));
	  };

	  exports.pooledVariance = function pooledVariance(samples, unbiased) {
	    if (typeof unbiased === 'undefined') unbiased = true;
	    var sum = 0;
	    var length = 0,
	        l = samples.length;

	    for (var i = 0; i < l; i++) {
	      var values = samples[i];
	      var vari = exports.variance(values);
	      sum += (values.length - 1) * vari;
	      if (unbiased) length += values.length - 1;else length += values.length;
	    }

	    return sum / length;
	  };

	  exports.mode = function mode(values) {
	    var l = values.length,
	        itemCount = new Array(l),
	        i;

	    for (i = 0; i < l; i++) {
	      itemCount[i] = 0;
	    }

	    var itemArray = new Array(l);
	    var count = 0;

	    for (i = 0; i < l; i++) {
	      var index = itemArray.indexOf(values[i]);
	      if (index >= 0) itemCount[index]++;else {
	        itemArray[count] = values[i];
	        itemCount[count] = 1;
	        count++;
	      }
	    }

	    var maxValue = 0,
	        maxIndex = 0;

	    for (i = 0; i < count; i++) {
	      if (itemCount[i] > maxValue) {
	        maxValue = itemCount[i];
	        maxIndex = i;
	      }
	    }

	    return itemArray[maxIndex];
	  };

	  exports.covariance = function covariance(vector1, vector2, unbiased) {
	    if (typeof unbiased === 'undefined') unbiased = true;
	    var mean1 = exports.mean(vector1);
	    var mean2 = exports.mean(vector2);
	    if (vector1.length !== vector2.length) throw 'Vectors do not have the same dimensions';
	    var cov = 0,
	        l = vector1.length;

	    for (var i = 0; i < l; i++) {
	      var x = vector1[i] - mean1;
	      var y = vector2[i] - mean2;
	      cov += x * y;
	    }

	    if (unbiased) return cov / (l - 1);else return cov / l;
	  };

	  exports.skewness = function skewness(values, unbiased) {
	    if (typeof unbiased === 'undefined') unbiased = true;
	    var theMean = exports.mean(values);
	    var s2 = 0,
	        s3 = 0,
	        l = values.length;

	    for (var i = 0; i < l; i++) {
	      var dev = values[i] - theMean;
	      s2 += dev * dev;
	      s3 += dev * dev * dev;
	    }

	    var m2 = s2 / l;
	    var m3 = s3 / l;
	    var g = m3 / Math.pow(m2, 3 / 2.0);

	    if (unbiased) {
	      var a = Math.sqrt(l * (l - 1));
	      var b = l - 2;
	      return a / b * g;
	    } else {
	      return g;
	    }
	  };

	  exports.kurtosis = function kurtosis(values, unbiased) {
	    if (typeof unbiased === 'undefined') unbiased = true;
	    var theMean = exports.mean(values);
	    var n = values.length,
	        s2 = 0,
	        s4 = 0;

	    for (var i = 0; i < n; i++) {
	      var dev = values[i] - theMean;
	      s2 += dev * dev;
	      s4 += dev * dev * dev * dev;
	    }

	    var m2 = s2 / n;
	    var m4 = s4 / n;

	    if (unbiased) {
	      var v = s2 / (n - 1);
	      var a = n * (n + 1) / ((n - 1) * (n - 2) * (n - 3));
	      var b = s4 / (v * v);
	      var c = (n - 1) * (n - 1) / ((n - 2) * (n - 3));
	      return a * b - 3 * c;
	    } else {
	      return m4 / (m2 * m2) - 3;
	    }
	  };

	  exports.entropy = function entropy(values, eps) {
	    if (typeof eps === 'undefined') eps = 0;
	    var sum = 0,
	        l = values.length;

	    for (var i = 0; i < l; i++) sum += values[i] * Math.log(values[i] + eps);

	    return -sum;
	  };

	  exports.weightedMean = function weightedMean(values, weights) {
	    var sum = 0,
	        l = values.length;

	    for (var i = 0; i < l; i++) sum += values[i] * weights[i];

	    return sum;
	  };

	  exports.weightedStandardDeviation = function weightedStandardDeviation(values, weights) {
	    return Math.sqrt(exports.weightedVariance(values, weights));
	  };

	  exports.weightedVariance = function weightedVariance(values, weights) {
	    var theMean = exports.weightedMean(values, weights);
	    var vari = 0,
	        l = values.length;
	    var a = 0,
	        b = 0;

	    for (var i = 0; i < l; i++) {
	      var z = values[i] - theMean;
	      var w = weights[i];
	      vari += w * (z * z);
	      b += w;
	      a += w * w;
	    }

	    return vari * (b / (b * b - a));
	  };

	  exports.center = function center(values, inPlace) {
	    if (typeof inPlace === 'undefined') inPlace = false;
	    var result = values;
	    if (!inPlace) result = [].concat(values);
	    var theMean = exports.mean(result),
	        l = result.length;

	    for (var i = 0; i < l; i++) result[i] -= theMean;
	  };

	  exports.standardize = function standardize(values, standardDev, inPlace) {
	    if (typeof standardDev === 'undefined') standardDev = exports.standardDeviation(values);
	    if (typeof inPlace === 'undefined') inPlace = false;
	    var l = values.length;
	    var result = inPlace ? values : new Array(l);

	    for (var i = 0; i < l; i++) result[i] = values[i] / standardDev;

	    return result;
	  };

	  exports.cumulativeSum = function cumulativeSum(array) {
	    var l = array.length;
	    var result = new Array(l);
	    result[0] = array[0];

	    for (var i = 1; i < l; i++) result[i] = result[i - 1] + array[i];

	    return result;
	  };
	})(array);

	/**
	 *
	 * @private
	 * @param {Spectrum} spectrum
	 * @param {object} [options.xFilter={}]
	 * @param {array} [options.xFilter.from]
	 * @param {array} [options.xFilter.to]
	 * @param {array} [options.xFilter.exclusions=[]]
	 * @param {array} [options.yFactor=1]
	 */

	function getData(spectrum) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  const {
	    xFilter = {},
	    yFactor = 1
	  } = options;
	  let data = {
	    x: spectrum.x,
	    y: spectrum.y
	  };

	  if (xFilter) {
	    data = filterX({
	      x: spectrum.x,
	      y: spectrum.y
	    }, xFilter);
	  }

	  if (yFactor && yFactor !== 1) {
	    data.y = data.y.map(y => y * yFactor);
	  }

	  return data;
	}

	function norm(input) {
	  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  var _options$algorithm = options.algorithm,
	      algorithm = _options$algorithm === void 0 ? 'absolute' : _options$algorithm,
	      _options$sumValue = options.sumValue,
	      sumValue = _options$sumValue === void 0 ? 1 : _options$sumValue,
	      _options$maxValue = options.maxValue,
	      maxValue = _options$maxValue === void 0 ? 1 : _options$maxValue;

	  if (!isAnyArray(input)) {
	    throw new Error('input must be an array');
	  }

	  var output;

	  if (options.output !== undefined) {
	    if (!isAnyArray(options.output)) {
	      throw new TypeError('output option must be an array if specified');
	    }

	    output = options.output;
	  } else {
	    output = new Array(input.length);
	  }

	  if (input.length === 0) {
	    throw new Error('input must not be empty');
	  }

	  switch (algorithm.toLowerCase()) {
	    case 'absolute':
	      {
	        var absoluteSumValue = absoluteSum(input) / sumValue;
	        if (absoluteSumValue === 0) return input.slice(0);

	        for (var i = 0; i < input.length; i++) {
	          output[i] = input[i] / absoluteSumValue;
	        }

	        return output;
	      }

	    case 'max':
	      {
	        var currentMaxValue = max$1(input);
	        if (currentMaxValue === 0) return input.slice(0);
	        var factor = maxValue / currentMaxValue;

	        for (var _i = 0; _i < input.length; _i++) {
	          output[_i] = input[_i] * factor;
	        }

	        return output;
	      }

	    case 'sum':
	      {
	        var sumFactor = sum(input) / sumValue;
	        if (sumFactor === 0) return input.slice(0);

	        for (var _i2 = 0; _i2 < input.length; _i2++) {
	          output[_i2] = input[_i2] / sumFactor;
	        }

	        return output;
	      }

	    default:
	      throw new Error("norm: unknown algorithm: ".concat(algorithm));
	  }
	}

	function absoluteSum(input) {
	  var sumValue = 0;

	  for (var i = 0; i < input.length; i++) {
	    sumValue += Math.abs(input[i]);
	  }

	  return sumValue;
	}

	/**
	 * Normalize an array of zones:
	 * - ensure than from < to
	 * - merge overlapping zones
	 *
	 * The method will always check if from if lower than to and will swap if required.
	 * @param {Array} [zones=[]]
	 * @param {object} [options={}]
	 * @param {number} [options.from=Number.NEGATIVE_INFINITY] Specify min value of a zone
	 * @param {number} [options.to=Number.POSITIVE_INFINITY] Specify max value of a zone
	 */
	function normalize() {
	  let zones = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  if (zones.length === 0) return [];
	  let {
	    from = Number.NEGATIVE_INFINITY,
	    to = Number.POSITIVE_INFINITY
	  } = options;
	  if (from > to) [from, to] = [to, from];
	  zones = JSON.parse(JSON.stringify(zones)).map(zone => zone.from > zone.to ? {
	    from: zone.to,
	    to: zone.from
	  } : zone);
	  zones = zones.sort((a, b) => {
	    if (a.from !== b.from) return a.from - b.from;
	    return a.to - b.to;
	  });
	  zones.forEach(zone => {
	    if (from > zone.from) zone.from = from;
	    if (to < zone.to) zone.to = to;
	  });
	  zones = zones.filter(zone => zone.from <= zone.to);
	  if (zones.length === 0) return [];
	  let currentZone = zones[0];
	  let result = [currentZone];

	  for (let i = 1; i < zones.length; i++) {
	    let zone = zones[i];

	    if (zone.from <= currentZone.to) {
	      currentZone.to = zone.to;
	    } else {
	      currentZone = zone;
	      result.push(currentZone);
	    }
	  }

	  return result;
	}

	/**
	 * Convert an array of exclusions and keep only from / to
	 *
	 * The method will always check if from if lower than to and will swap if required.
	 * @param {Array} [exclusions=[]]
	 * @param {object} [options={}]
	 * @param {number} [options.from=Number.NEGATIVE_INFINITY] Specify min value of zones (after inversion)
	 * @param {number} [options.to=Number.POSITIVE_INFINITY] Specify max value of zones (after inversion)
	 */

	function invert() {
	  let exclusions = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  let {
	    from = Number.NEGATIVE_INFINITY,
	    to = Number.POSITIVE_INFINITY
	  } = options;
	  if (from > to) [from, to] = [to, from];
	  exclusions = normalize(exclusions, {
	    from,
	    to
	  });
	  if (exclusions.length === 0) return [{
	    from,
	    to
	  }];
	  let zones = [];

	  for (let i = 0; i < exclusions.length; i++) {
	    let exclusion = exclusions[i];
	    let nextExclusion = exclusions[i + 1];

	    if (i === 0) {
	      if (exclusion.from > from) {
	        zones.push({
	          from,
	          to: exclusion.from
	        });
	      }
	    }

	    if (i === exclusions.length - 1) {
	      if (exclusion.to < to) {
	        zones.push({
	          from: exclusion.to,
	          to
	        });
	      }
	    } else {
	      zones.push({
	        from: exclusion.to,
	        to: nextExclusion.from
	      });
	    }
	  }

	  return zones;
	}

	/**
	 * Add the number of points per zone to reach a specified total
	 * @param {Array} [zones=[]]
	 * @param {number} [numberOfPoints] Total number of points to distribute between zones
	 * @param {object} [options={}]
	 * @param {number} [options.from=Number.NEGATIVE_INFINITY] Specify min value of a zone
	 * @param {number} [options.to=Number.POSITIVE_INFINITY] Specify max value of a zone
	 */

	function zonesWithPoints(zones, numberOfPoints) {
	  let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
	  if (zones.length === 0) return zones;
	  zones = normalize(zones, options);
	  const totalSize = zones.reduce((previous, current) => {
	    return previous + (current.to - current.from);
	  }, 0);
	  let unitsPerPoint = totalSize / numberOfPoints;
	  let currentTotal = 0;

	  for (let i = 0; i < zones.length - 1; i++) {
	    let zone = zones[i];
	    zone.numberOfPoints = Math.min(Math.round((zone.to - zone.from) / unitsPerPoint), numberOfPoints - currentTotal);
	    currentTotal += zone.numberOfPoints;
	  }

	  zones[zones.length - 1].numberOfPoints = numberOfPoints - currentTotal;
	  return zones;
	}

	/**
	 * function that retrieves the getEquallySpacedData with the variant "slot"
	 *
	 * @param {Array<number>} x
	 * @param {Array<number>} y
	 * @param {number} from - Initial point
	 * @param {number} to - Final point
	 * @param {number} numberOfPoints
	 * @return {Array} - Array of y's equally spaced with the variant "slot"
	 */
	function equallySpacedSlot(x, y, from, to, numberOfPoints) {
	  let xLength = x.length;
	  let step = (to - from) / (numberOfPoints > 1 ? numberOfPoints - 1 : 1);
	  let halfStep = step / 2;
	  let lastStep = x[x.length - 1] - x[x.length - 2];
	  let start = from - halfStep;
	  let output = new Array(numberOfPoints); // Init main variables

	  let min = start;
	  let max = start + step;
	  let previousX = -Number.MAX_VALUE;
	  let previousY = 0;
	  let nextX = x[0];
	  let nextY = y[0];
	  let frontOutsideSpectra = 0;
	  let backOutsideSpectra = true;
	  let currentValue = 0; // for slot algorithm

	  let currentPoints = 0;
	  let i = 1; // index of input

	  let j = 0; // index of output

	  main: while (true) {
	    if (previousX >= nextX) throw new Error('x must be a growing series');

	    while (previousX - max > 0) {
	      // no overlap with original point, just consume current value
	      if (backOutsideSpectra) {
	        currentPoints++;
	        backOutsideSpectra = false;
	      }

	      output[j] = currentPoints <= 0 ? 0 : currentValue / currentPoints;
	      j++;

	      if (j === numberOfPoints) {
	        break main;
	      }

	      min = max;
	      max += step;
	      currentValue = 0;
	      currentPoints = 0;
	    }

	    if (previousX > min) {
	      currentValue += previousY;
	      currentPoints++;
	    }

	    if (previousX === -Number.MAX_VALUE || frontOutsideSpectra > 1) {
	      currentPoints--;
	    }

	    previousX = nextX;
	    previousY = nextY;

	    if (i < xLength) {
	      nextX = x[i];
	      nextY = y[i];
	      i++;
	    } else {
	      nextX += lastStep;
	      nextY = 0;
	      frontOutsideSpectra++;
	    }
	  }

	  return output;
	}

	/**
	 * Function that calculates the integral of the line between two
	 * x-coordinates, given the slope and intercept of the line.
	 * @param {number} x0
	 * @param {number} x1
	 * @param {number} slope
	 * @param {number} intercept
	 * @return {number} integral value.
	 */
	function integral(x0, x1, slope, intercept) {
	  return 0.5 * slope * x1 * x1 + intercept * x1 - (0.5 * slope * x0 * x0 + intercept * x0);
	}

	/**
	 * function that retrieves the getEquallySpacedData with the variant "smooth"
	 *
	 * @param {Array<number>} x
	 * @param {Array<number>} y
	 * @param {number} from - Initial point
	 * @param {number} to - Final point
	 * @param {number} numberOfPoints
	 * @return {Array} - Array of y's equally spaced with the variant "smooth"
	 */

	function equallySpacedSmooth(x, y, from, to, numberOfPoints) {
	  let xLength = x.length;
	  let step = (to - from) / (numberOfPoints > 1 ? numberOfPoints - 1 : 1);
	  let halfStep = step / 2;
	  let output = new Array(numberOfPoints);
	  let initialOriginalStep = x[1] - x[0];
	  let lastOriginalStep = x[xLength - 1] - x[xLength - 2]; // Init main variables

	  let min = from - halfStep;
	  let max = from + halfStep;
	  let previousX = Number.MIN_SAFE_INTEGER;
	  let previousY = 0;
	  let nextX = x[0] - initialOriginalStep;
	  let nextY = 0;
	  let currentValue = 0;
	  let slope = 0;
	  let intercept = 0;
	  let sumAtMin = 0;
	  let sumAtMax = 0;
	  let i = 0; // index of input

	  let j = 0; // index of output

	  function getSlope(x0, y0, x1, y1) {
	    return (y1 - y0) / (x1 - x0);
	  }

	  let add = 0;

	  main: while (true) {
	    if (previousX >= nextX) throw new Error('x must be a growing series');

	    if (previousX <= min && min <= nextX) {
	      add = integral(0, min - previousX, slope, previousY);
	      sumAtMin = currentValue + add;
	    }

	    while (nextX - max >= 0) {
	      // no overlap with original point, just consume current value
	      add = integral(0, max - previousX, slope, previousY);
	      sumAtMax = currentValue + add;
	      output[j++] = (sumAtMax - sumAtMin) / step;

	      if (j === numberOfPoints) {
	        break main;
	      }

	      min = max;
	      max += step;
	      sumAtMin = sumAtMax;
	    }

	    currentValue += integral(previousX, nextX, slope, intercept);
	    previousX = nextX;
	    previousY = nextY;

	    if (i < xLength) {
	      nextX = x[i];
	      nextY = y[i];
	      i++;
	    } else if (i === xLength) {
	      nextX += lastOriginalStep;
	      nextY = 0;
	    }

	    slope = getSlope(previousX, previousY, nextX, nextY);
	    intercept = -slope * previousX + previousY;
	  }

	  return output;
	}

	/**
	 * Function that returns a Number array of equally spaced numberOfPoints
	 * containing a representation of intensities of the spectra arguments x
	 * and y.
	 *
	 * The options parameter contains an object in the following form:
	 * from: starting point
	 * to: last point
	 * numberOfPoints: number of points between from and to
	 * variant: "slot" or "smooth" - smooth is the default option
	 *
	 * The slot variant consist that each point in the new array is calculated
	 * averaging the existing points between the slot that belongs to the current
	 * value. The smooth variant is the same but takes the integral of the range
	 * of the slot and divide by the step size between two points in the new array.
	 *
	 * If exclusions zone are present, zones are ignored !
	 * @param {object} [arrayXY={}] - object containing 2 properties x and y (both an array)
	 * @param {object} [options={}]
	 * @param {number} [options.from=x[0]]
	 * @param {number} [options.to=x[x.length-1]]
	 * @param {string} [options.variant='smooth']
	 * @param {number} [options.numberOfPoints=100]
	 * @param {Array} [options.exclusions=[]] array of from / to that should be skipped for the generation of the points
	 * @param {Array} [options.zones=[]] array of from / to that should be kept
	 * @return {object<x: Array, y:Array>} new object with x / y array with the equally spaced data.
	 */

	function equallySpaced() {
	  let arrayXY = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  let {
	    x,
	    y
	  } = arrayXY;
	  let xLength = x.length;
	  let reverse = false;

	  if (x.length > 1 && x[0] > x[1]) {
	    x = x.slice().reverse();
	    y = y.slice().reverse();
	    reverse = true;
	  }

	  let {
	    from = x[0],
	    to = x[xLength - 1],
	    variant = 'smooth',
	    numberOfPoints = 100,
	    exclusions = [],
	    zones = []
	  } = options;

	  if (xLength !== y.length) {
	    throw new RangeError("the x and y vector doesn't have the same size.");
	  }

	  if (typeof from !== 'number' || isNaN(from)) {
	    throw new RangeError("'from' option must be a number");
	  }

	  if (typeof to !== 'number' || isNaN(to)) {
	    throw new RangeError("'to' option must be a number");
	  }

	  if (typeof numberOfPoints !== 'number' || isNaN(numberOfPoints)) {
	    throw new RangeError("'numberOfPoints' option must be a number");
	  }

	  if (numberOfPoints < 2) {
	    throw new RangeError("'numberOfPoints' option must be greater than 1");
	  }

	  if (zones.length === 0) {
	    zones = invert(exclusions, {
	      from,
	      to
	    });
	  }

	  zones = zonesWithPoints(zones, numberOfPoints, {
	    from,
	    to
	  });
	  let xResult = [];
	  let yResult = [];

	  for (let zone of zones) {
	    let zoneResult = processZone(x, y, zone.from, zone.to, zone.numberOfPoints, variant);
	    xResult = xResult.concat(zoneResult.x);
	    yResult = yResult.concat(zoneResult.y);
	  }

	  if (reverse) {
	    if (from < to) {
	      return {
	        x: xResult.reverse(),
	        y: yResult.reverse()
	      };
	    } else {
	      return {
	        x: xResult,
	        y: yResult
	      };
	    }
	  } else {
	    if (from < to) {
	      return {
	        x: xResult,
	        y: yResult
	      };
	    } else {
	      return {
	        x: xResult.reverse(),
	        y: yResult.reverse()
	      };
	    }
	  }
	}

	function processZone(x, y, from, to, numberOfPoints, variant) {
	  if (numberOfPoints < 1) {
	    throw new RangeError('the number of points must be at least 1');
	  }

	  let output = variant === 'slot' ? equallySpacedSlot(x, y, from, to, numberOfPoints) : equallySpacedSmooth(x, y, from, to, numberOfPoints);
	  return {
	    x: sequentialFill({
	      from,
	      to,
	      size: numberOfPoints
	    }),
	    y: output
	  };
	}

	/**
	 * Calibrates the data based on a range and means of peaks in this range
	 * Based on a range we will make a peak picking using global spectra deconvolution
	 * The selected nbPeaks will then be taken into account to calculate an average X value.
	 * The difference between the targetX and the averageX value will be returned
	 *
	 * @param {DataXY} [data] array of points {x,y}
	 * @param {object} [range={}]
	 * @param {number} [range.from] - Beginning of the range where the interest signal is localed
	 * @param {number} [range.to] - End of the range where the interest signal is localed
	 * @param {object} [options={}]
	 * @param {number} [options.nbPeaks=1] Number of peaks to consider to calculate mean (sorted by height)
	 * @param {number} [options.targetX=0] Expected value for the mean of the peaks position
	 * @param {number} [options.gsd={}] GSD options. You may check options here: http://mljs.github.io/global-spectral-deconvolution/#gsd
	 * @param {number} [options.gsd.minMaxRatio=0.2] - GSD Threshold to determine if a given peak should be considered as a noise.
	 * @returns {number} difference between targetX and currentX
	 */

	/**
	 * @param data
	 * @param range
	 * @param options
	 */

	function xyCalibrate(data) {
	  let range = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
	  const {
	    targetX = 0,
	    nbPeaks = 1,
	    gsd: gsdOptions = {
	      minMaxRatio: 0.1,
	      realTopDetection: true,
	      smoothY: true,
	      sgOptions: {
	        windowSize: 7,
	        polynomial: 3
	      }
	    }
	  } = options;
	  let {
	    from,
	    to
	  } = range;
	  if (from === undefined || to === undefined) return 0;
	  const fromIndex = xFindClosestIndex(data.x, from);
	  const toIndex = xFindClosestIndex(data.x, to);
	  const slicedData = {
	    x: data.x.slice(fromIndex, toIndex),
	    y: data.y.slice(fromIndex, toIndex)
	  };
	  let peaks = gsd(slicedData, gsdOptions).sort((a, b) => b.y - a.y).slice(0, nbPeaks);
	  if (peaks.length === 0) return 0;
	  const middle = mean(peaks.map(peak => peak.x));
	  return targetX - middle;
	}

	/**
	 *
	 * @private
	 * @param {DataXY} spectrum
	 * @param {object} [options={}]
	 * @param {number} [options.from=x[0]]
	 * @param {number} [options.to=x[x.length-1]]
	 * @param {number} [options.numberOfPoints=1024]
	 * @param {array} [options.filters=[]]
	 * @param {array} [options.exclusions=[]]
	 */

	function getNormalized(spectrum) {
	  let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	  if (!isAnyArray(spectrum.x) || !isAnyArray(spectrum.y)) {
	    throw new Error('getNormalized: Can not get normalized data, missing original data.');
	  }

	  let xs = spectrum.x;
	  let {
	    from = xs[0],
	    to = xs[xs.length - 1],
	    numberOfPoints = 1024,
	    filters = [],
	    exclusions = []
	  } = options;
	  let ys = spectrum.y.slice(0);

	  for (let filter of filters) {
	    let filterOptions = filter.options;

	    switch (filter.name) {
	      case 'centerMean':
	        {
	          let mean = array.mean(spectrum.y);

	          for (let i = 0; i < ys.length; i++) {
	            ys[i] -= mean;
	          }

	          break;
	        }

	      case 'scaleSD':
	        {
	          let std = array.standardDeviation(spectrum.y);

	          for (let i = 0; i < ys.length; i++) {
	            ys[i] /= std;
	          }

	          break;
	        }

	      case 'pareto':
	        {
	          ys = xParetoNormalization(ys);
	          break;
	        }

	      case 'normalize':
	        {
	          ys = norm(ys);
	          break;
	        }

	      case 'rescale':
	        {
	          ys = rescale(ys);
	          break;
	        }

	      case 'calibrate':
	        {
	          if (filterOptions.from === undefined || filterOptions.to === undefined) {
	            break;
	          }

	          let shift = xyCalibrate({
	            x: xs,
	            y: ys
	          }, {
	            from: filterOptions.from,
	            to: filterOptions.to
	          }, {
	            nbPeaks: filterOptions.nbPeaks,
	            targetX: filterOptions.targetX
	          });
	          xs = xAdd(xs, shift);
	          break;
	        }

	      case 'growingX':
	        {
	          const growing = xyEnsureGrowingX({
	            x: xs,
	            y: ys
	          });
	          xs = growing.x;
	          ys = growing.y;
	          break;
	        }

	      case 'sgg':
	        {
	          ys = sgg(ys, xs, filterOptions);
	          break;
	        }

	      case 'xFunction':
	        {
	          xs = applyArrayFunction(xs, 'x', filterOptions.function);
	          break;
	        }

	      case 'yFunction':
	        {
	          ys = applyArrayFunction(ys, 'y', filterOptions.function);
	          break;
	        }

	      case '':
	      case undefined:
	        break;

	      default:
	        throw new Error(`Unknown filter kind: ${filter.name}`);
	    }
	  }

	  const allowedBoundary = {
	    x: {
	      min: xs[0],
	      max: xs[xs.length - 1]
	    },
	    y: array.minMax(ys)
	  };
	  let data = equallySpaced({
	    x: xs,
	    y: ys
	  }, {
	    from,
	    to,
	    numberOfPoints,
	    exclusions
	  });
	  return {
	    data,
	    allowedBoundary
	  };
	}

	function applyArrayFunction(array, variableLabel, fctString) {
	  if (!fctString) fctString = variableLabel;
	  fctString = fctString.replace(/(^|\W)([a-z]{2,})/g, '$1Math.$2').replace(/Math\.Math/g, 'Math'); // eslint-disable-next-line no-new-func

	  let fct = new Function(variableLabel, `return Number(${fctString})`);
	  array = array.slice();

	  for (let i = 0; i < array.length; i++) {
	    array[i] = fct(array[i]);

	    if (isNaN(array[i])) {
	      throw Error(`The callback ${fctString} does not return a number: ${array[i]}`);
	    }
	  }

	  return array;
	}

	/**
	 * @private
	 * @param {*} spectrum
	 * @param {*} ranges
	 */

	function updateRangesInfo(spectrum) {
	  let ranges = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : [];
	  spectrum.ranges = {};

	  for (let range of ranges) {
	    range = JSON.parse(JSON.stringify(range));
	    spectrum.ranges[range.label] = range;
	    let fromToIndex = {
	      fromIndex: xFindClosestIndex(spectrum.normalized.x, range.from),
	      toIndex: xFindClosestIndex(spectrum.normalized.x, range.to)
	    };
	    range.integration = xyIntegration(spectrum.normalized, fromToIndex);
	    range.maxPoint = xyMaxYPoint(spectrum.normalized, fromToIndex);
	    range.minPoint = xyMinYPoint(spectrum.normalized, fromToIndex);
	    range.x = spectrum.normalized.x.slice(fromToIndex.fromIndex, fromToIndex.toIndex + 1);
	    range.y = spectrum.normalized.y.slice(fromToIndex.fromIndex, fromToIndex.toIndex + 1);
	  }
	}

	/**
	 * Class allowing manipulate one IR spectrum
	 * @class spectrum
	 * @param {object} [json={}] - object containing a spectrum
	 * @param {Array} [json.x=[]] - x values
	 * @param {Array} [json.y=[]] - y values
	 */

	class Spectrum {
	  /**
	   *
	   * @param {array} x
	   * @param {array} y
	   * @param {string} id
	   * @param {object} [options={}]
	   * @param {object} [options.meta={}]
	   * @param {object} [options.normalization]
	   * @param {object} [options.normalized]
	   */
	  constructor(x, y, id) {
	    let options = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : {};
	    const {
	      meta = {},
	      normalization = {},
	      normalized
	    } = options;

	    if (x && x.length > 1 && x[0] > x[x.length - 1]) {
	      this.x = x.slice().reverse();
	      this.y = y.slice().reverse();
	    } else {
	      this.x = x || [];
	      this.y = y || [];
	    }

	    this.id = id;
	    this.meta = meta;
	    this.normalizedBoundary = {
	      x: {
	        min: 0,
	        max: 0
	      },
	      y: {
	        min: 0,
	        max: 0
	      }
	    };

	    if (normalized) {
	      this.normalized = normalized;
	      this.updateNormalizedBoundary();
	      this.normalizedAllowedBoundary = this.normalizedBoundary;
	    } else {
	      this.updateNormalization(normalization);
	    }

	    this.updateMemory();
	  }

	  get minX() {
	    return this.x[0];
	  }

	  get maxX() {
	    return this.x[this.x.length - 1];
	  }

	  updateMemory() {
	    let original = (this.x && this.x.length || 0) * 16;
	    let normalized = this.normalized.x.length * 16;
	    this.memory = {
	      original,
	      normalized,
	      total: original + normalized
	    };
	  }

	  removeOriginal() {
	    this.x = undefined;
	    this.y = undefined;
	    this.updateMemory();
	  }

	  getXY() {
	    if (!Array.isArray(this.x) || !Array.isArray(this.y)) {
	      throw new Error('Can not get normalized data');
	    }

	    return {
	      x: this.x,
	      y: this.y
	    };
	  }

	  updateRangesInfo(ranges) {
	    updateRangesInfo(this, ranges);
	  }

	  updateNormalization(normalization) {
	    const result = getNormalized(this, normalization);
	    this.normalized = result.data;
	    this.normalizedAllowedBoundary = result.allowedBoundary;
	    this.ranges = {};
	    this.updateMemory();
	    this.updateNormalizedBoundary();
	  }

	  getData(options) {
	    return getData(this, options);
	  }

	  updateNormalizedBoundary() {
	    this.normalizedBoundary.x = {
	      min: this.normalized.x[0],
	      max: this.normalized.x[this.normalized.x.length - 1]
	    };
	    this.normalizedBoundary.y = array.minMax(this.normalized.y);
	  }

	}

	class SpectraProcessor {
	  /**
	   * Manager a large number of spectra with the possibility to normalize the data
	   * and skip the original data.
	   * @param {object} [options={}]
	   * @param {boolean} [options.maxMemory=64M]
	   * @param {object} [options.normalization={}] options to normalize the spectra before comparison
	   * @param {number} [options.normalization.from]
	   * @param {number} [options.normalization.to]
	   * @param {number} [options.normalization.numberOfPoints]
	   * @param {array<object>} [options.normalization.filters]
	   * @param {string} [options.normalization.filters.X.name]
	   * @param {object} [options.normalization.filters.X.options]
	   * @param {array<object>} [options.normalization.exclusions]
	   * @param {string} [options.normalization.exclusions.X.from]
	   * @param {object} [options.normalization.exclusions.X.to]
	   */
	  constructor() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	    this.normalization = options.normalization;
	    this.maxMemory = options.maxMemory || 256 * 1024 * 1024;
	    this.keepOriginal = true;
	    this.spectra = [];
	  }

	  getNormalizationAnnotations() {
	    return getNormalizationAnnotations(this.normalization, this.getNormalizedBoundary());
	  }
	  /**
	   * Recalculate the normalized data using the stored original data if available
	   * This will throw an error in the original data is not present
	   * @param {number} [normalization.from]
	   * @param {number} [normalization.to]
	   * @param {number} [normalization.numberOfPoints]
	   * @param {array<object>} [normalization.filters]
	   * @param {string} [normalization.filters.X.name]
	   * @param {object} [normalization.filters.X.options]
	   * @param {array<object>} [normalization.exclusions]
	   * @param {string} [normalization.exclusions.X.from]
	   * @param {object} [normalization.exclusions.X.to]
	   */


	  setNormalization() {
	    let normalization = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

	    if (JSON.stringify(this.normalization) === JSON.stringify(normalization)) {
	      return;
	    }

	    this.normalization = normalization;

	    for (let spectrum of this.spectra) {
	      spectrum.updateNormalization(this.normalization);
	    }
	  }

	  getNormalization() {
	    return this.normalization;
	  }
	  /**
	   * Returns an object {x:[], y:[]} containing the autocorrelation for the
	   * specified index
	   * @param {integer} [index|undefined] x index of the spectrum to autocorrelate
	   * @param {object} [options={}]
	   * @param {array} [options.ids=[]] list of ids, by default all spectra
	   * @param {array} [options.x] x value if index is undefined
	   */


	  getAutocorrelation(index) {
	    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	    const {
	      x
	    } = options;
	    const normalizedData = this.getNormalizedData(options);

	    if (index === undefined && x !== undefined) {
	      index = xFindClosestIndex(normalizedData.x, x);
	    }

	    return getAutocorrelation(normalizedData, index);
	  }
	  /**
	   * Returns a {x:[], y:[]} containg the average of specified spectra
	   * @param {object} [options={}]
	   * @param {array} [options.ids=[]] list of ids, by default all spectra
	   */


	  getMeanData(options) {
	    return getMeanData(this.getNormalizedData(options));
	  }
	  /**
	   * Returns an object contains 4 parameters with the normalized data
	   * @returns {object} { ids:[], matrix:[Array], meta:[object], x:[] }
	   * @param {object} [options={}]
	   * @param {Array} [options.ids] List of spectra ids to export, by default all
	   */


	  getNormalizedData() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	    const {
	      ids
	    } = options;
	    let spectra = this.getSpectra(ids);
	    return getNormalizedData(spectra);
	  }
	  /**
	   * Returns a tab separated value containing the normalized data
	   * @param {object} [options={}]
	   * @param {Array} [options.ids] List of spectra ids to export, by default all
	   * @param {string} [options.fs='\t'] field separator
	   * @param {string} [options.rs='\n'] record (line) separator
	   * @returns {string}
	   */


	  getNormalizedText() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	    const {
	      ids
	    } = options;
	    let spectra = this.getSpectra(ids);
	    return getNormalizedText(spectra, options);
	  }
	  /**
	   * Returns a tab separated value containing the post processed data
	   * @param {object} [options={}]
	   * @param {Array} [options.ids] List of spectra ids to export, by default all
	   * @param {string} [options.fs='\t'] field separator
	   * @param {string} [options.rs='\n'] record (line) separator
	   * @param {object} [postProcessing={}]
	   * @returns {string}
	   */


	  getPostProcessedText() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	    return getPostProcessedText(this, options);
	  }

	  getMinMaxX() {
	    let min = Number.MAX_VALUE;
	    let max = Number.MIN_VALUE;

	    for (let spectrum of this.spectra) {
	      if (spectrum.minX < min) min = spectrum.minX;
	      if (spectrum.maxX > max) max = spectrum.maxX;
	    }

	    return {
	      min,
	      max
	    };
	  }
	  /**
	    * Returns an object contains 4 parameters with the scaled data
	   * @param {object} [options={}] scale spectra based on various parameters
	   * @param {object} [options.range] from - to
	   * @param {Array} [options.ids] ids of selected spectra, by default all
	   * @param {string} [options.targetID=spectra[0].id]
	   * @param {string} [options.method='max'] min, max, range, minMax
	   * @param {boolean} [options.relative=false]
	   * @returns {object} { ids:[], matrix:[Array], meta:[object], x:[] }
	   */


	  getPostProcessedData(options) {
	    return getPostProcessedData(this, options);
	  }
	  /**
	   * Add from text
	   * By default TITLE from the jcamp will be in the meta information
	   * @param {string} text
	   * @param {object} [options={}]
	   * @param {object} [options.parserOptions={}] XY parser options
	   * @param {object} [options.meta={}]
	   * @param {string} [options.meta.color]
	   * @param {object} [options.id={}]
	   * @param {object} [options.kind]
	   * @param {boolean} [options.force=false] replace existing spectrum (same ID)
	   */


	  addFromText(text$1) {
	    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	    if (options.force !== true && options.id && this.contains(options.id)) {
	      return;
	    }

	    let parsed = text(text$1, options);
	    let meta = { ...parsed.meta,
	      ...(options.meta || {})
	    };
	    this.addFromData(parsed.data, {
	      meta,
	      id: options.id
	    });
	  }
	  /**
	   * Add jcamp
	   * By default TITLE from the jcamp will be in the meta information
	   * @param {string} jcamp
	   * @param {object} [options={}]
	   * @param {object} [options.meta={}]
	   * @param {string} [options.meta.color]
	   * @param {object} [options.id={}]
	   * @param {boolean} [options.force=false] replace existing spectrum (same ID)
	   */


	  addFromJcamp(jcamp$1) {
	    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

	    if (options.force !== true && options.id && this.contains(options.id)) {
	      return;
	    }

	    let parsed = jcamp(jcamp$1);
	    let meta = { ...parsed.meta,
	      ...(options.meta || {})
	    };
	    this.addFromData(parsed.data, {
	      meta,
	      id: options.id
	    });
	  }

	  updateRangesInfo(options) {
	    for (let spectrum of this.spectra) {
	      spectrum.updateRangesInfo(options);
	    }
	  }
	  /**
	   * Returns the metadata for a set of spectra
	   * @param {object} [options={}]
	   * @param {Array} [options.ids] ids of the spectra to select, by default all
	   * @returns {object}
	   */


	  getMetadata() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	    const {
	      ids
	    } = options;
	    return getMetadata(this.getSpectra(ids));
	  }
	  /**
	   *
	   * @param {object} [options={}]
	   * @param {Array} [options.ids] ids of the spectra to select, by default all
	   * @param {string} [options.propertyName="category"]
	   */


	  getClasses(options) {
	    return getClasses(this.getMetadata(options), options);
	  }
	  /**
	   *
	   * @param {object} [options={}]
	   * @param {Array} [options.ids] ids of the spectra to select, by default all
	   * @param {string} [options.propertyName="category"]
	   */


	  getClassLabels(options) {
	    return getClassLabels(this.getMetadata(options), options);
	  }
	  /**
	   *
	   * @param {object} [options={}]
	   * @param {string} [options.propertyName="category"]
	   */


	  getCategoriesStats(options) {
	    return getCategoriesStats(this.getMetadata(), options);
	  }
	  /**
	   * Add a spectrum based on the data
	   * @param {object} data {x, y}}
	   * @param {object} [options={}]
	   * @param {object} [options.meta={}]
	   * @param {object} [options.id]
	   * @param {object} [options.normalization={}]
	   * @param {object} [options.normalized]
	   * @return {Spectrum}
	   */


	  addFromData(data) {
	    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	    if (this.spectra.length === 0) this.keepOriginal = true;
	    const id = options.id || Math.random().toString(36).substring(2, 10);
	    let index = this.getSpectrumIndex(id);
	    if (index === undefined) index = this.spectra.length;
	    let spectrum = new Spectrum(data.x, data.y, id, {
	      meta: options.meta,
	      normalized: options.normalized,
	      normalization: this.normalization
	    });
	    this.spectra[index] = spectrum;

	    if (!this.keepOriginal) {
	      spectrum.removeOriginal();
	    } else {
	      let memoryInfo = this.getMemoryInfo();

	      if (memoryInfo.total > this.maxMemory) {
	        this.keepOriginal = false;
	        this.removeOriginals();
	      }
	    }
	  }

	  removeOriginals() {
	    for (let spectrum of this.spectra) {
	      spectrum.removeOriginal();
	    }
	  }
	  /**
	   * Remove the spectrum from the SpectraProcessor for the specified id
	   * @param {string} id
	   */


	  removeSpectrum(id) {
	    let index = this.getSpectrumIndex(id);
	    if (index === undefined) return undefined;
	    return this.spectra.splice(index, 1);
	  }
	  /**
	   * Remove all the spectra not present in the list
	   * @param {Array} [ids] Array of ids of the spectra to keep
	   */


	  removeSpectraNotIn(ids) {
	    let currentIDs = this.spectra.map(spectrum => spectrum.id);

	    for (let id of currentIDs) {
	      if (!ids.includes(id)) {
	        this.removeSpectrum(id);
	      }
	    }
	  }
	  /**
	   * Checks if the ID of a spectrum exists in the SpectraProcessor
	   * @param {string} id
	   */


	  contains(id) {
	    return !isNaN(this.getSpectrumIndex(id));
	  }
	  /**
	   * Returns the index of the spectrum in the spectra array
	   * @param {string} id
	   * @returns {number}
	   */


	  getSpectrumIndex(id) {
	    if (!id) return undefined;

	    for (let i = 0; i < this.spectra.length; i++) {
	      let spectrum = this.spectra[i];
	      if (spectrum.id === id) return i;
	    }

	    return undefined;
	  }
	  /**
	   * Returns an array of all the ids
	   * @returns {array}
	   */


	  getIDs() {
	    return this.spectra.map(spectrum => spectrum.id);
	  }
	  /**
	   * Returns an array of spectrum from their ids
	   * @param {Array} ids
	   * @returns {Array<Spectrum}
	   */


	  getSpectra(ids) {
	    if (!ids || !Array.isArray(ids)) return this.spectra;
	    let spectra = [];

	    for (let id of ids) {
	      let index = this.getSpectrumIndex(id);

	      if (index !== undefined) {
	        spectra.push(this.spectra[index]);
	      }
	    }

	    return spectra;
	  }
	  /**
	   * Returns the index of the spectrum in the spectra array
	   * @param {string} id
	   * @returns {number}
	   */


	  getSpectrum(id) {
	    let index = this.getSpectrumIndex(id);
	    if (index === undefined) return undefined;
	    return this.spectra[index];
	  }
	  /**
	   * Returns a JSGraph chart object for all the spectra
	   * @returns {object}
	   */


	  getChart(options) {
	    return getChart(this.spectra, options);
	  }
	  /**
	   * Returns a JSGraph chart object for all the spectra
	   * @param {object} [options={}]
	   * @param {Array} [options.ids] ids of the spectra to select, by default all
	   * @returns {object}
	   */


	  getAutocorrelationChart(index, options) {
	    return getAutocorrelationChart(this, index, options);
	  }
	  /**
	   * Returns a JSGraph annotation object for the specified index
	   * @param {object} [options={}]
	   * @param {array} [options.ids=[]]
	   * @param {array} [options.showSpectrumID=true]
	   * @param {array} [options.sortY=true]
	   * @param {object} [options.limit=20]
	   * ids, showSpectrumID = true, sort = true, limit = 20
	   * @returns {object}
	   */


	  getTrackAnnotation(index, options) {
	    return getTrackAnnotation(this.spectra, index, options);
	  }
	  /**
	   * Returns a JSGraph annotation object for box plot
	   * @param {object} [options={}]
	   * @param {array} [options.ids=[]] list of ids, by default all spectra
	   * @returns {object}
	   */


	  getBoxPlotChart() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	    const normalizedData = this.getNormalizedData(options);
	    return getBoxPlotChart(normalizedData, options);
	  }
	  /**
	   * Returns boxplot information
	   * @param {object} [options={}]
	   * @param {array} [options.ids=[]] list of ids, by default all spectra
	   * @returns {object}
	   */


	  getBoxPlotData() {
	    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	    const normalizedData = this.getNormalizedData(options);
	    return getBoxPlotData(normalizedData);
	  }
	  /**
	   * Returns a JSGraph chart object for all the normalized spectra
	   * @param {object} [options={}]
	   * @param {Array} [options.ids] ids of the spectra to select, by default all
	   * @returns {object}
	   */


	  getNormalizedChart(options) {
	    const {
	      ids
	    } = options;
	    let spectra = this.getSpectra(ids);
	    return getNormalizedChart(spectra, options);
	  }
	  /**
	   * Returns a JSGraph chart object for all the scaled normalized spectra
	   * @param {object} [options={}]
	   * @param {Array} [options.ids] ids of the spectra to select, by default all
	   * @param {object} [options.range] from - to
	   * @param {string} [options.targetID=spectra[0].id]
	   * @param {string} [options.method='max'] min, max, range, minMax
	   * @param {boolean} [options.relative=false]
	   * @returns {object}
	   */


	  getPostProcessedChart(options) {
	    return getPostProcessedChart(this, options);
	  }

	  getMemoryInfo() {
	    let memoryInfo = {
	      original: 0,
	      normalized: 0,
	      total: 0
	    };
	    this.spectra.forEach(spectrum => {
	      let memory = spectrum.memory;
	      memoryInfo.original += memory.original;
	      memoryInfo.normalized += memory.normalized;
	      memoryInfo.total += memory.total;
	    });
	    memoryInfo.keepOriginal = this.keepOriginal;
	    memoryInfo.maxMemory = this.maxMemory;
	    return memoryInfo;
	  }

	  getNormalizedBoundary() {
	    let boundary = {
	      x: {
	        min: Number.MAX_VALUE,
	        max: Number.MIN_VALUE
	      },
	      y: {
	        min: Number.MAX_VALUE,
	        max: Number.MIN_VALUE
	      }
	    };

	    for (let spectrum of this.spectra) {
	      if (spectrum.normalizedBoundary.x.min < boundary.x.min) {
	        boundary.x.min = spectrum.normalizedBoundary.x.min;
	      }

	      if (spectrum.normalizedBoundary.x.max > boundary.x.max) {
	        boundary.x.max = spectrum.normalizedBoundary.x.max;
	      }

	      if (spectrum.normalizedBoundary.y.min < boundary.y.min) {
	        boundary.y.min = spectrum.normalizedBoundary.y.min;
	      }

	      if (spectrum.normalizedBoundary.y.max > boundary.y.max) {
	        boundary.y.max = spectrum.normalizedBoundary.y.max;
	      }
	    }

	    return boundary;
	  }
	  /**
	   * We provide the allowed from / to after normalization
	   * For the X axis we return the smallest common values
	   * For the Y axis we return the largest min / max
	   */


	  getNormalizedCommonBoundary() {
	    let boundary = {
	      x: {
	        min: Number.NEGATIVE_INFINITY,
	        max: Number.POSITIVE_INFINITY
	      },
	      y: {
	        min: Number.POSITIVE_INFINITY,
	        max: Number.NEGATIVE_INFINITY
	      }
	    };

	    for (let spectrum of this.spectra) {
	      if (spectrum.normalizedAllowedBoundary.x.min > boundary.x.min) {
	        boundary.x.min = spectrum.normalizedAllowedBoundary.x.min;
	      }

	      if (spectrum.normalizedAllowedBoundary.x.max < boundary.x.max) {
	        boundary.x.max = spectrum.normalizedAllowedBoundary.x.max;
	      }

	      if (spectrum.normalizedAllowedBoundary.y.min < boundary.y.min) {
	        boundary.y.min = spectrum.normalizedAllowedBoundary.y.min;
	      }

	      if (spectrum.normalizedAllowedBoundary.y.max > boundary.y.max) {
	        boundary.y.max = spectrum.normalizedAllowedBoundary.y.max;
	      }
	    }

	    return boundary;
	  }
	  /**
	   * Create SpectraProcessor from normalized TSV
	   * @param {string} text
	   * @param {object} [options={}]
	   * @param {object} [options.separator='\t']
	   */


	  static fromNormalizedMatrix(text) {
	    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	    let parsed = matrix(text, options);

	    if (!parsed) {
	      throw new Error('Can not parse TSV file');
	    }

	    let spectraProcessor = new SpectraProcessor();
	    spectraProcessor.setNormalization({
	      from: parsed.x[0],
	      to: parsed.x[parsed.x.length - 1],
	      numberOfPoints: parsed.x.length
	    });

	    for (let i = 0; i < parsed.ids.length; i++) {
	      spectraProcessor.addFromData({
	        x: [],
	        y: []
	      }, {
	        normalized: {
	          x: parsed.x,
	          y: parsed.matrix[i]
	        },
	        id: parsed.ids[i],
	        meta: parsed.meta[i]
	      });
	    }

	    spectraProcessor.keepOriginal = false;
	    return spectraProcessor;
	  }

	}

	function getFilterAnnotations() {
	  let filter = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
	  let {
	    exclusions = []
	  } = filter;
	  let annotations = [];
	  exclusions = exclusions.filter(exclusion => !exclusion.ignore);
	  annotations = exclusions.map(exclusion => {
	    let annotation = {
	      type: 'rect',
	      position: [{
	        x: exclusion.from,
	        y: '0px'
	      }, {
	        x: exclusion.to,
	        y: '2000px'
	      }],
	      strokeWidth: 0,
	      fillColor: 'rgba(255,255,128,1)'
	    };
	    return annotation;
	  });

	  if (filter.from !== undefined) {
	    annotations.push({
	      type: 'rect',
	      position: [{
	        x: Number.MIN_SAFE_INTEGER,
	        y: '0px'
	      }, {
	        x: filter.from,
	        y: '2000px'
	      }],
	      strokeWidth: 0,
	      fillColor: 'rgba(255,255,224,1)'
	    });
	  }

	  if (filter.to !== undefined) {
	    annotations.push({
	      type: 'rect',
	      position: [{
	        x: filter.to,
	        y: '0px'
	      }, {
	        x: Number.MAX_SAFE_INTEGER,
	        y: '2000px'
	      }],
	      strokeWidth: 0,
	      fillColor: 'rgba(255,255,224,1)'
	    });
	  }

	  return annotations;
	}

	exports.SpectraProcessor = SpectraProcessor;
	exports.getFilterAnnotations = getFilterAnnotations;
	exports.xFindClosestIndex = xFindClosestIndex;

	Object.defineProperty(exports, '__esModule', { value: true });

}));
//# sourceMappingURL=spectra-processor.js.map
