/**
 * nmr-parser - Read and convert any NMR file
 * @version v1.9.0
 * @link https://github.com/cheminfo/nmr-parser#readme
 * @license MIT
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.NMRparser = {}));
})(this, (function (exports) { 'use strict';

    const toString = Object.prototype.toString;
    /**
     * Checks if an object is an instance of an Array (array or typed array).
     *
     * @param {any} value - Object to check.
     * @returns {boolean} True if the object is an array.
     */

    function isAnyArray(value) {
      return toString.call(value).endsWith('Array]');
    }

    /**
     * a number that correspond to a type of numeric
     * @typedef {number} numericType
     * @const
     */
    const numericTypeTable = {
      0: 'uint8',
      1: 'uint16',
      2: 'uint32',
      3: 'uint64',
      4: 'int8',
      5: 'int16',
      6: 'int32',
      7: 'int64',
      8: 'float32',
      9: 'float64',
      10: 'complex64',
      11: 'complex128'
    };
    /**
     * a number that corresponds to a type of quantity
     * @typedef {number} quantityType
     * @const
     */

    const quantityTypeTable = {
      0: 'scalar',
      1: 'vector',
      2: 'matrix',
      3: 'symetricMatrix',
      4: 'pixel'
    };

    /**
     * a class for dependent variable
     * @param {object || array} data - the dependent variable
     * @param {numericType} numericType - a number that correspond to a type of numeric used to store the components
     * @param {object} [options] - an object with options (name, unit, quantityName, componentLabels, sparseSampling, application, description)
     * @param {string} [options.name] - a name of the dependent variable
     * @param {string} [options.unit] - the unit of the dependent variable
     * @param {string} [options.quantityName] - a name of the quantity
     * @param {array} [options.componentLabels] - an array of labels for each component of the dependent variable
     * @return {object} - an dependent variable
     */

    function formatDependentVariable(data, numericType) {
      let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
      let {
        quantityType = 0,
        encoding = 'none',
        name = '',
        unit = '',
        quantityName = '',
        componentLabels = [],
        sparseSampling = {},
        from = 0,
        to = -1
      } = options;
      let components;

      if (isAnyArray(data)) {
        throw new Error('not yet implemented');
      } else if (Object.keys(data).length === 2) {
        components = fromReIm(data, from, to);
      }

      if (componentLabels.length === 0) {
        componentLabels = components.componentLabels;
      }

      return {
        type: 'internal',
        quantityType: quantityTypeTable[quantityType],
        numericType: numericTypeTable[numericType],
        encoding,
        name,
        unit,
        quantityName,
        componentLabels,
        sparseSampling,
        description: options.description || '',
        application: options.application || '',
        components: components.components,
        dataLength: components.dataLength
      };
    }
    /**
     * import object {re:[], im:[]} to component
     * @param {object} reIm - a reIm object to import
     * @param {number} from - lower limit
     * @param {number} to - upper limit
     * @return {array} - components
     */

    function fromReIm(reIm, from, to) {
      let dataLength = [];
      let componentLabels = [];
      let components = [];

      if (isAnyArray(reIm.re) & isAnyArray(reIm.im)) {
        if (typeof reIm.re[0] === 'number') {
          // if 1D
          dataLength[0] = setLengthComplex(from[0], to[0], reIm.re.length);
          let component = new Float64Array(dataLength[0]);

          for (let i = 0; i < dataLength[0]; i += 2) {
            let idx = i + from[0] * 2;
            component[i] = reIm.re[idx / 2];
            component[i + 1] = reIm.im[idx / 2];
          }

          components.push(component);
          componentLabels.push('complex');
        } else if (isAnyArray(reIm.re[0])) {
          // if 2D
          dataLength[0] = setLength(from[1], to[1], reIm.re.length);
          dataLength[1] = setLengthComplex(from[0], to[0], reIm.re[0].length);

          for (let j = 0; j < dataLength[0]; j++) {
            let component = new Float64Array(dataLength[1]);

            for (let i = 0; i < dataLength[1]; i += 2) {
              let idx = i + from[0] * 2;
              component[i] = reIm.re[j][idx / 2];
              component[i + 1] = reIm.im[j][idx / 2];
            }

            components.push(component);
          }
        } else {
          throw new Error('check your object');
        }
      } else if (isAnyArray(reIm.re.re)) {
        dataLength[0] = reIm.re.re.length * 2;
        let re = fromReIm(reIm.re, from, to).components;
        let im = fromReIm(reIm.im, from, to).components;

        for (let j = 0; j < dataLength[0] / 2; j++) {
          components.push(re[j]);
          components.push(im[j]);
        }
      } else {
        throw new Error('check the dimension or the type of data in your array');
      }

      return {
        dataLength,
        componentLabels,
        components
      };
    }

    function setLength(from, to, length) {
      if (to - from + 1 < length) {
        return to - from + 1;
      } else {
        return length;
      }
    }

    function setLengthComplex(from, to, length) {
      if (to - from + 1 < length) {
        return (to - from + 1) * 2;
      } else {
        return length * 2;
      }
    } // /**
    //  * add component to components from 1D array.
    //  * @param {array} array - a 1D or 2D array to import
    //  * @return {Float64Array} - component
    //  */
    // function add1DArray(array) {
    //   let component;
    //   component = new Float64Array(array.length);
    //   for (let i = 0; i < array.length; i++) {
    //     component[i] = array[i];
    //   }
    //   return component;
    // }
    // /**
    //  * import component to InternalDEPENDENTVAR class object from 1D or 2D array.
    //  * @param {array} array - a 1D or 2D array to import
    //  */
    // function fromArray(array) {
    //   this.dataLength[0] = array.length;
    //   if (typeof array[0] === 'number') {
    //     this.components = [this.add1DArray(array)];
    //   } else if (Array.isArray(array[0])) {
    //     this.dataLength[1] = array[0].length;
    //     for (let j = 0; j < this.dataLength[1]; j++) {
    //       this.components.push(this.add1DArray(array[j]));
    //     }
    //   } else {
    //     throw new Error('check the dimension or the type of data in your array');
    //   }
    //   return this;
    // }

    /**
     *
     * @param {*} label
     * @param {*} count
     * @param {*} increment
     * @param {*} options
     */
    function formatLinearDimension(label, count, increment) {
      let options = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : {};
      return {
        label: String(label),
        count: Number(count),
        increment: increment,
        type: 'linear',
        description: String(options.description) || '',
        application: options.application || {},
        coordinatesOffset: options.coordinatesOffset || 0,
        originOffset: options.originOffset || 0,
        quantityName: String(options.quantityName) || '',
        reciprocal: options.reciprocal || {},
        period: options.period || 0,
        complexFFT: options.complexFFT || false
      };
    }

    // eslint-disable-next-line import/no-unassigned-import
    const decoder = new TextDecoder('utf-8');
    function decode(bytes) {
      return decoder.decode(bytes);
    }
    const encoder = new TextEncoder();
    function encode(str) {
      return encoder.encode(str);
    }

    const defaultByteLength = 1024 * 8;
    class IOBuffer {
      /**
       * @param data - The data to construct the IOBuffer with.
       * If data is a number, it will be the new buffer's length<br>
       * If data is `undefined`, the buffer will be initialized with a default length of 8Kb<br>
       * If data is an ArrayBuffer, SharedArrayBuffer, an ArrayBufferView (Typed Array), an IOBuffer instance,
       * or a Node.js Buffer, a view will be created over the underlying ArrayBuffer.
       * @param options
       */
      constructor() {
        let data = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : defaultByteLength;
        let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
        let dataIsGiven = false;

        if (typeof data === 'number') {
          data = new ArrayBuffer(data);
        } else {
          dataIsGiven = true;
          this.lastWrittenByte = data.byteLength;
        }

        const offset = options.offset ? options.offset >>> 0 : 0;
        const byteLength = data.byteLength - offset;
        let dvOffset = offset;

        if (ArrayBuffer.isView(data) || data instanceof IOBuffer) {
          if (data.byteLength !== data.buffer.byteLength) {
            dvOffset = data.byteOffset + offset;
          }

          data = data.buffer;
        }

        if (dataIsGiven) {
          this.lastWrittenByte = byteLength;
        } else {
          this.lastWrittenByte = 0;
        }

        this.buffer = data;
        this.length = byteLength;
        this.byteLength = byteLength;
        this.byteOffset = dvOffset;
        this.offset = 0;
        this.littleEndian = true;
        this._data = new DataView(this.buffer, dvOffset, byteLength);
        this._mark = 0;
        this._marks = [];
      }
      /**
       * Checks if the memory allocated to the buffer is sufficient to store more
       * bytes after the offset.
       * @param byteLength - The needed memory in bytes.
       * @returns `true` if there is sufficient space and `false` otherwise.
       */


      available() {
        let byteLength = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        return this.offset + byteLength <= this.length;
      }
      /**
       * Check if little-endian mode is used for reading and writing multi-byte
       * values.
       * @returns `true` if little-endian mode is used, `false` otherwise.
       */


      isLittleEndian() {
        return this.littleEndian;
      }
      /**
       * Set little-endian mode for reading and writing multi-byte values.
       */


      setLittleEndian() {
        this.littleEndian = true;
        return this;
      }
      /**
       * Check if big-endian mode is used for reading and writing multi-byte values.
       * @returns `true` if big-endian mode is used, `false` otherwise.
       */


      isBigEndian() {
        return !this.littleEndian;
      }
      /**
       * Switches to big-endian mode for reading and writing multi-byte values.
       */


      setBigEndian() {
        this.littleEndian = false;
        return this;
      }
      /**
       * Move the pointer n bytes forward.
       * @param n - Number of bytes to skip.
       */


      skip() {
        let n = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        this.offset += n;
        return this;
      }
      /**
       * Move the pointer to the given offset.
       * @param offset
       */


      seek(offset) {
        this.offset = offset;
        return this;
      }
      /**
       * Store the current pointer offset.
       * @see {@link IOBuffer#reset}
       */


      mark() {
        this._mark = this.offset;
        return this;
      }
      /**
       * Move the pointer back to the last pointer offset set by mark.
       * @see {@link IOBuffer#mark}
       */


      reset() {
        this.offset = this._mark;
        return this;
      }
      /**
       * Push the current pointer offset to the mark stack.
       * @see {@link IOBuffer#popMark}
       */


      pushMark() {
        this._marks.push(this.offset);

        return this;
      }
      /**
       * Pop the last pointer offset from the mark stack, and set the current
       * pointer offset to the popped value.
       * @see {@link IOBuffer#pushMark}
       */


      popMark() {
        const offset = this._marks.pop();

        if (offset === undefined) {
          throw new Error('Mark stack empty');
        }

        this.seek(offset);
        return this;
      }
      /**
       * Move the pointer offset back to 0.
       */


      rewind() {
        this.offset = 0;
        return this;
      }
      /**
       * Make sure the buffer has sufficient memory to write a given byteLength at
       * the current pointer offset.
       * If the buffer's memory is insufficient, this method will create a new
       * buffer (a copy) with a length that is twice (byteLength + current offset).
       * @param byteLength
       */


      ensureAvailable() {
        let byteLength = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;

        if (!this.available(byteLength)) {
          const lengthNeeded = this.offset + byteLength;
          const newLength = lengthNeeded * 2;
          const newArray = new Uint8Array(newLength);
          newArray.set(new Uint8Array(this.buffer));
          this.buffer = newArray.buffer;
          this.length = this.byteLength = newLength;
          this._data = new DataView(this.buffer);
        }

        return this;
      }
      /**
       * Read a byte and return false if the byte's value is 0, or true otherwise.
       * Moves pointer forward by one byte.
       */


      readBoolean() {
        return this.readUint8() !== 0;
      }
      /**
       * Read a signed 8-bit integer and move pointer forward by 1 byte.
       */


      readInt8() {
        return this._data.getInt8(this.offset++);
      }
      /**
       * Read an unsigned 8-bit integer and move pointer forward by 1 byte.
       */


      readUint8() {
        return this._data.getUint8(this.offset++);
      }
      /**
       * Alias for {@link IOBuffer#readUint8}.
       */


      readByte() {
        return this.readUint8();
      }
      /**
       * Read `n` bytes and move pointer forward by `n` bytes.
       */


      readBytes() {
        let n = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        const bytes = new Uint8Array(n);

        for (let i = 0; i < n; i++) {
          bytes[i] = this.readByte();
        }

        return bytes;
      }
      /**
       * Read a 16-bit signed integer and move pointer forward by 2 bytes.
       */


      readInt16() {
        const value = this._data.getInt16(this.offset, this.littleEndian);

        this.offset += 2;
        return value;
      }
      /**
       * Read a 16-bit unsigned integer and move pointer forward by 2 bytes.
       */


      readUint16() {
        const value = this._data.getUint16(this.offset, this.littleEndian);

        this.offset += 2;
        return value;
      }
      /**
       * Read a 32-bit signed integer and move pointer forward by 4 bytes.
       */


      readInt32() {
        const value = this._data.getInt32(this.offset, this.littleEndian);

        this.offset += 4;
        return value;
      }
      /**
       * Read a 32-bit unsigned integer and move pointer forward by 4 bytes.
       */


      readUint32() {
        const value = this._data.getUint32(this.offset, this.littleEndian);

        this.offset += 4;
        return value;
      }
      /**
       * Read a 32-bit floating number and move pointer forward by 4 bytes.
       */


      readFloat32() {
        const value = this._data.getFloat32(this.offset, this.littleEndian);

        this.offset += 4;
        return value;
      }
      /**
       * Read a 64-bit floating number and move pointer forward by 8 bytes.
       */


      readFloat64() {
        const value = this._data.getFloat64(this.offset, this.littleEndian);

        this.offset += 8;
        return value;
      }
      /**
       * Read a 64-bit signed integer number and move pointer forward by 8 bytes.
       */


      readBigInt64() {
        const value = this._data.getBigInt64(this.offset, this.littleEndian);

        this.offset += 8;
        return value;
      }
      /**
       * Read a 64-bit unsigned integer number and move pointer forward by 8 bytes.
       */


      readBigUint64() {
        const value = this._data.getBigUint64(this.offset, this.littleEndian);

        this.offset += 8;
        return value;
      }
      /**
       * Read a 1-byte ASCII character and move pointer forward by 1 byte.
       */


      readChar() {
        return String.fromCharCode(this.readInt8());
      }
      /**
       * Read `n` 1-byte ASCII characters and move pointer forward by `n` bytes.
       */


      readChars() {
        let n = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        let result = '';

        for (let i = 0; i < n; i++) {
          result += this.readChar();
        }

        return result;
      }
      /**
       * Read the next `n` bytes, return a UTF-8 decoded string and move pointer
       * forward by `n` bytes.
       */


      readUtf8() {
        let n = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1;
        return decode(this.readBytes(n));
      }
      /**
       * Write 0xff if the passed value is truthy, 0x00 otherwise and move pointer
       * forward by 1 byte.
       */


      writeBoolean(value) {
        this.writeUint8(value ? 0xff : 0x00);
        return this;
      }
      /**
       * Write `value` as an 8-bit signed integer and move pointer forward by 1 byte.
       */


      writeInt8(value) {
        this.ensureAvailable(1);

        this._data.setInt8(this.offset++, value);

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as an 8-bit unsigned integer and move pointer forward by 1
       * byte.
       */


      writeUint8(value) {
        this.ensureAvailable(1);

        this._data.setUint8(this.offset++, value);

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * An alias for {@link IOBuffer#writeUint8}.
       */


      writeByte(value) {
        return this.writeUint8(value);
      }
      /**
       * Write all elements of `bytes` as uint8 values and move pointer forward by
       * `bytes.length` bytes.
       */


      writeBytes(bytes) {
        this.ensureAvailable(bytes.length);

        for (let i = 0; i < bytes.length; i++) {
          this._data.setUint8(this.offset++, bytes[i]);
        }

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 16-bit signed integer and move pointer forward by 2
       * bytes.
       */


      writeInt16(value) {
        this.ensureAvailable(2);

        this._data.setInt16(this.offset, value, this.littleEndian);

        this.offset += 2;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 16-bit unsigned integer and move pointer forward by 2
       * bytes.
       */


      writeUint16(value) {
        this.ensureAvailable(2);

        this._data.setUint16(this.offset, value, this.littleEndian);

        this.offset += 2;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 32-bit signed integer and move pointer forward by 4
       * bytes.
       */


      writeInt32(value) {
        this.ensureAvailable(4);

        this._data.setInt32(this.offset, value, this.littleEndian);

        this.offset += 4;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 32-bit unsigned integer and move pointer forward by 4
       * bytes.
       */


      writeUint32(value) {
        this.ensureAvailable(4);

        this._data.setUint32(this.offset, value, this.littleEndian);

        this.offset += 4;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 32-bit floating number and move pointer forward by 4
       * bytes.
       */


      writeFloat32(value) {
        this.ensureAvailable(4);

        this._data.setFloat32(this.offset, value, this.littleEndian);

        this.offset += 4;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 64-bit floating number and move pointer forward by 8
       * bytes.
       */


      writeFloat64(value) {
        this.ensureAvailable(8);

        this._data.setFloat64(this.offset, value, this.littleEndian);

        this.offset += 8;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 64-bit signed bigint and move pointer forward by 8
       * bytes.
       */


      writeBigInt64(value) {
        this.ensureAvailable(8);

        this._data.setBigInt64(this.offset, value, this.littleEndian);

        this.offset += 8;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write `value` as a 64-bit unsigned bigint and move pointer forward by 8
       * bytes.
       */


      writeBigUint64(value) {
        this.ensureAvailable(8);

        this._data.setBigUint64(this.offset, value, this.littleEndian);

        this.offset += 8;

        this._updateLastWrittenByte();

        return this;
      }
      /**
       * Write the charCode of `str`'s first character as an 8-bit unsigned integer
       * and move pointer forward by 1 byte.
       */


      writeChar(str) {
        return this.writeUint8(str.charCodeAt(0));
      }
      /**
       * Write the charCodes of all `str`'s characters as 8-bit unsigned integers
       * and move pointer forward by `str.length` bytes.
       */


      writeChars(str) {
        for (let i = 0; i < str.length; i++) {
          this.writeUint8(str.charCodeAt(i));
        }

        return this;
      }
      /**
       * UTF-8 encode and write `str` to the current pointer offset and move pointer
       * forward according to the encoded length.
       */


      writeUtf8(str) {
        return this.writeBytes(encode(str));
      }
      /**
       * Export a Uint8Array view of the internal buffer.
       * The view starts at the byte offset and its length
       * is calculated to stop at the last written byte or the original length.
       */


      toArray() {
        return new Uint8Array(this.buffer, this.byteOffset, this.lastWrittenByte);
      }
      /**
       * Update the last written byte offset
       * @private
       */


      _updateLastWrittenByte() {
        if (this.offset > this.lastWrittenByte) {
          this.lastWrittenByte = this.offset;
        }
      }

    }

    const endianness = {
      0: 'bigEndian',
      1: 'littleEndian'
    };
    const instrumentTable = {
      0: 'NONE',
      1: 'GSX',
      2: 'ALPHA',
      3: 'ECLIPSE',
      4: 'MASS_SPEC',
      5: 'COMPILER',
      6: 'OTHER_NMR',
      7: 'UNKNOWN',
      8: 'GEMINI',
      9: 'UNITY',
      10: 'ASPECT',
      11: 'UX',
      12: 'FELIX',
      13: 'LAMBDA',
      14: 'GE_1280',
      15: 'GE_OMEGA',
      16: 'CHEMAGNETICS',
      17: 'CDFF',
      18: 'GALACTIC',
      19: 'TRIAD',
      20: 'GENERIC_NMR',
      21: 'GAMMA',
      22: 'JCAMP_DX',
      23: 'AMX',
      24: 'DMX',
      25: 'ECA',
      26: 'ALICE',
      27: 'NMR_PIPE',
      28: 'SIMPSON'
    };
    const dataTypeTable = {
      0: '64Bit Float',
      1: '32Bit Float',
      2: 'Reserved',
      3: 'Reserved'
    };
    const dataFormatTable = {
      1: 'One_D',
      2: 'Two_D',
      3: 'Three_D',
      4: 'Four_D',
      5: 'Five_D',
      6: 'Six_D',
      7: 'Seven_D',
      8: 'Eight_D',
      9: 'not for NMR data formats',
      10: 'not for NMR data formats',
      11: 'not for NMR data formats',
      12: 'Small_Two_D',
      13: 'Small_Three_D',
      14: 'Small_Four_D'
    };
    const dataAxisTypeTable = {
      0: 'None',
      //Axis is not used.
      1: 'Real',
      //Axis has real data only, no imaginary.
      2: 'TPPI',
      3: 'Complex',
      4: 'Real_Complex',

      /* Axis should be accessed as complex when it is the major axis,
                accessed as real otherwise.  This is only valid when all axes in
                use have this setting.*/
      5: 'Envelope'
      /* Behaves the same way as a Real_Complex dimension but the data
            has different meaning.  Instead of being treated as real and
            imaginary parts of a complex number, the data should be treated as minimum and maximum parts of a projection.  This is used
            for the data that results from an envelope projection.*/

    };
    const prefixTable = {
      '-8': 'Yotta',
      '-6': 'Exa',
      '-7': 'Zetta',
      '-5': 'Pecta',
      '-4': 'Tera',
      '-3': 'Giga',
      '-2': 'Mega',
      '-1': 'Kilo',
      0: 'None',
      1: 'Milli',
      2: 'Micro',
      3: 'Nano',
      4: 'Pico',
      5: 'Femto',
      6: 'Atto',
      7: 'Zepto'
    };
    const unitPrefixTable = {
      Yotta: 24,
      Exa: 21,
      Zetta: 18,
      Pecta: 15,
      Tera: 12,
      Giga: 9,
      Mega: 6,
      Kilo: 3,
      None: 0,
      Milli: -3,
      Micro: -6,
      Nano: -9,
      Pico: -12,
      Femto: -15,
      Atto: -18,
      Zepto: -21
    };
    const baseTable = {
      0: 'None',
      1: 'Abundance',
      2: 'Ampere',
      3: 'Candela',
      4: 'Celsius',
      5: 'Coulomb',
      6: 'Degree',
      7: 'Electronvolt',
      8: 'Farad',
      9: 'Sievert',
      10: 'Gram',
      11: 'Gray',
      12: 'Henry',
      13: 'Hertz',
      14: 'Kelvin',
      15: 'Joule',
      16: 'Liter',
      17: 'Lumen',
      18: 'Lux',
      19: 'Meter',
      20: 'Mole',
      21: 'Newton',
      22: 'Ohm',
      23: 'Pascal',
      24: 'Percent',
      25: 'Point',
      26: 'Ppm',
      27: 'Radian',
      28: 'Second',
      29: 'Siemens',
      30: 'Steradian',
      31: 'Tesla',
      32: 'Volt',
      33: 'Watt',
      34: 'Weber',
      35: 'Decibel',
      36: 'Dalton',
      37: 'Thompson',
      38: 'Ugeneric',
      // Treated as None, but never displayed',
      39: 'LPercent ',
      // Treated as percent for display, but different for comparison',
      40: 'PPT',
      // Parts per trillion (Private, do not use)',
      41: 'PPB ',
      // Parts per billion (Private, do not use)',
      42: 'Index'
    };
    const dataAxisRangedTable = {
      0: 'Ranged',

      /* The ruler for the axis ranges from Data_Axis_Start[n] to
            Data_Axis_Stop[n] with a step function of
                (Data_Axis_Stop[n] - Data_Axis_Start[n]) /
                (Data_Offset_Stop[n] - Data_Offset_Start[n]) */
      1: 'Listed',
      // (deprecated)

      /* The ruler for the axis is a list of doubles stored in the
            List Section.  Values in the ruler may be anything.*/
      2: 'Sparse',

      /*The ruler for the axis is a list of doubles stored in the
            List Section.  Values in the rulers must be strictly monotonically
            increasing or decreasing.*/
      3: 'Listed'
      /* The ruler for the axis is a list of doubles stored in the
            List Section.  Values in the rulers do not fit definition of Sparse.*/

    };
    const valueTypeTable = {
      0: 'String',
      1: 'Integer',
      2: 'Float',
      3: 'Complex',
      4: 'Infinity'
    };

    function getPar(param, searchStr) {
      return param.paramArray.find(o => o.name === searchStr) || '';
    }
    function getDigitalFilter(param) {
      const orders = param.paramArray.find(e => e.name === 'orders');
      const factors = param.paramArray.find(e => e.name === 'factors');
      const sweep = param.paramArray.find(e => e.name === 'x_sweep');
      const acqTime = param.paramArray.find(e => e.name === 'x_acq_time');
      const nbPoints = param.paramArray.find(e => e.name === 'x_points');
      const shouldStop = [orders, factors, sweep, acqTime, nbPoints].some(e => e === undefined);

      if (shouldStop) {
        return;
      }

      const s = parseInt(orders.value.slice(0, 1), 10);
      const jump = orders.value.slice(1).length / s;
      let arg = 0;
      let factorNumber = new Int8Array(s);
      let offsetO = 1;
      let offsetF = 0;

      for (let i = 0; i < s; i++) {
        factorNumber[i] = parseInt(factors.value.slice(offsetF, offsetF + 1), 10);
        offsetF += 1;
      }

      for (let i = 0; i < s; i++) {
        let productorial = 1;

        for (let j = i; j < s; j++) {
          productorial *= factorNumber[j];
        }

        arg += (parseInt(orders.value.slice(offsetO, offsetO + jump), 10) - 1) / productorial;
        offsetO += jump;
      }

      arg /= 2;
      const delaySec = arg / sweep.value;
      return delaySec / acqTime.value * (nbPoints.value - 1);
    }
    function getMagnitude(param, searchStr) {
      let par = getPar(param, searchStr) || 'NA';

      if (par === 'NA') {
        return {
          magnitude: 'NA',
          unit: 'NA'
        };
      }

      let unit = par.unit[0].base;
      let unitMult = unitPrefixTable[par.unit[0].prefix];
      let magnitude = par.value * 10 ** unitMult;
      return {
        magnitude,
        unit
      };
    }
    function getUnit(buffer, size) {
      let unit = [];

      for (let i = 0; i < size; i++) {
        let byte = buffer.readByte();
        let prefix = prefixTable[byte >> 4];
        let power = byte & 0b00001111;
        let base = baseTable[buffer.readInt8()];
        unit.push({
          prefix,
          power,
          base
        });
      }

      return unit;
    }
    function getString(buffer, size) {
      let string = [];

      for (let i = 0; i < size; i++) {
        let char = buffer.readChar();

        if (char !== '\u0000') {
          string.push(char);
        }
      }

      return string.join('');
    }
    function getParamName(buffer, size) {
      let string = [];

      for (let i = 0; i < size; i++) {
        let char = buffer.readChar();

        if (char !== ' ') {
          string.push(char);
        }
      }

      return string.join('');
    }
    function getArray(buffer, size, format) {
      let double = [];

      for (let i = 0; i < size; i++) {
        switch (format) {
          case 'readUint32':
            double.push(buffer.readUint32());
            break;

          case 'readFloat64':
            double.push(buffer.readFloat64());
            break;

          case 'readFloat32':
            double.push(buffer.readFloat32());
            break;

          case 'readUint8':
            double.push(buffer.readUint8());
            break;

          case 'readBoolean':
            double.push(buffer.readBoolean());
            break;
        }
      }

      return double;
    }

    /**
     * A parser for 1D and 2D JDL NMR Files
     * @param {ArrayBuffer} buffer - a buffer object containing the JDL file
     * @return {Object} - an Object with converted data
     */

    function parseJEOL(buffer) {
      let ioBuffer = new IOBuffer(buffer);
      ioBuffer.setBigEndian(); // read header section

      let byte;
      let header = {};
      let byteArray = [];
      header.fileIdentifier = ioBuffer.readChars(8);
      header.endian = endianness[ioBuffer.readInt8()];
      header.majorVersion = ioBuffer.readUint8();
      header.minorVersion = ioBuffer.readUint16();
      header.dataDimensionNumber = ioBuffer.readUint8();
      header.dataDimensionExist = ioBuffer.readByte().toString(2).split('').map(x => Boolean(Number(x)));
      byte = ioBuffer.readByte();
      header.dataType = dataTypeTable[byte >> 6];
      header.dataFormat = dataFormatTable[byte & 0b00111111];
      header.dataInstrument = instrumentTable[ioBuffer.readInt8()];
      header.translate = getArray(ioBuffer, 8, 'readUint8');
      header.dataAxisType = getArray(ioBuffer, 8, 'readUint8').map(x => dataAxisTypeTable[x]);
      header.dataUnits = getUnit(ioBuffer, 8);
      header.title = getString(ioBuffer, 124);

      for (byte in getArray(ioBuffer, 4, 'readUint8')) {
        byteArray.push(dataAxisRangedTable[byte >> 4]);
        byteArray.push(dataAxisRangedTable[byte & 0b00001111]);
      }

      header.dataAxisRanged = byteArray;
      header.dataPoints = getArray(ioBuffer, 8, 'readUint32');
      header.dataOffsetStart = getArray(ioBuffer, 8, 'readUint32');
      header.dataOffsetStop = getArray(ioBuffer, 8, 'readUint32');
      header.dataAxisStart = getArray(ioBuffer, 8, 'readFloat64');
      header.dataAxisStop = getArray(ioBuffer, 8, 'readFloat64');
      byteArray = new Uint8Array(4);

      for (let i = 0; i < 4; i++) {
        byteArray[i] = ioBuffer.readByte();
      }

      let year = 1990 + (byteArray[0] >> 1);
      let month = (byteArray[0] << 3 & 0b00001000) + (byteArray[1] >> 5);
      let day = byteArray[2] & 0b00011111;
      header.creationTime = {
        year,
        month,
        day
      };

      for (let i = 0; i < 4; i++) {
        byteArray[i] = ioBuffer.readByte();
      }

      year = 1990 + (byteArray[0] >> 1);
      month = (byteArray[0] << 3 & 0b00001000) + (byteArray[1] >> 5);
      day = byteArray[2] & 0b00011111;
      header.revisionTime = {
        year,
        month,
        day
      };
      header.nodeName = getString(ioBuffer, 16);
      header.site = getString(ioBuffer, 128);
      header.author = getString(ioBuffer, 128);
      header.comment = getString(ioBuffer, 128);
      let dataAxisTitles = [];

      for (let i = 0; i < 8; i++) {
        dataAxisTitles.push(getString(ioBuffer, 32));
      }

      header.dataAxisTitles = dataAxisTitles;
      header.baseFreq = getArray(ioBuffer, 8, 'readFloat64');
      header.zeroPoint = getArray(ioBuffer, 8, 'readFloat64');
      header.reversed = getArray(ioBuffer, 8, 'readBoolean');
      ioBuffer.skip(3);
      header.annotationOK = Boolean(ioBuffer.readByte() >> 7);
      header.historyUsed = ioBuffer.readUint32();
      header.historyLength = ioBuffer.readUint32();
      header.paramStart = ioBuffer.readUint32();
      header.paramLength = ioBuffer.readUint32();
      header.ListStart = getArray(ioBuffer, 8, 'readUint32');
      header.ListLength = getArray(ioBuffer, 8, 'readUint32');
      header.dataStart = ioBuffer.readUint32();
      header.dataLength = ioBuffer.readUint32() << 32 | ioBuffer.readUint32();
      header.contextStart = ioBuffer.readUint32() << 32 | ioBuffer.readUint32();
      header.contextLength = ioBuffer.readUint32();
      header.annoteStart = ioBuffer.readUint32() << 32 | ioBuffer.readUint32();
      header.annoteLength = ioBuffer.readUint32();
      header.totalSize = ioBuffer.readUint32() << 32 | ioBuffer.readUint32();
      header.unitLocation = getArray(ioBuffer, 8, 'readUint8');
      let compoundUnit = [];

      for (let i = 0; i < 2; i++) {
        let unit = [];
        let scaler = ioBuffer.readInt16();

        for (let j = 0; j < 5; j++) {
          byte = ioBuffer.readInt16();
          unit.push(byte);
        }

        compoundUnit.push({
          scaler,
          unit
        });
      }

      header.compoundUnit = compoundUnit; // section parameters (param header and array)

      if (header.endian === 'littleEndian') {
        ioBuffer.setLittleEndian();
      }

      ioBuffer.seek(header.paramStart);
      let parameters = {
        parameterSize: ioBuffer.readUint32(),
        lowIndex: ioBuffer.readUint32(),
        highIndex: ioBuffer.readUint32(),
        totalSize: ioBuffer.readUint32()
      };
      let paramArray = [];

      for (let p = 0; p < parameters.highIndex + 1; p++) {
        ioBuffer.skip(4);
        let scaler = ioBuffer.readInt16();
        let unit = getUnit(ioBuffer, 5);
        ioBuffer.skip(16);
        let valueType = valueTypeTable[ioBuffer.readInt32()];
        ioBuffer.seek(ioBuffer.offset - 20);
        let value;

        switch (valueType) {
          case 'String':
            value = getParamName(ioBuffer, 16);
            break;

          case 'Integer':
            value = ioBuffer.readInt32();
            ioBuffer.skip(12);
            break;

          case 'Float':
            value = ioBuffer.readFloat64();
            ioBuffer.skip(8);
            break;

          case 'Complex':
            value.Real = ioBuffer.readFloat64();
            value.Imag = ioBuffer.readFloat64();
            break;

          case 'Infinity':
            value = ioBuffer.readInt32();
            ioBuffer.skip(12);
            break;

          default:
            ioBuffer.skip(16);
            break;
        }

        ioBuffer.skip(4);
        let name = getParamName(ioBuffer, 28);
        paramArray.push({
          name: name.toLowerCase(),
          scaler,
          unit,
          value,
          valueType
        });
      }

      parameters.paramArray = paramArray; // data section

      ioBuffer.seek(header.dataStart);

      if (header.endian === 'littleEndian') {
        ioBuffer.setLittleEndian();
      }

      let data = {};
      let dataSectionCount = 1;
      let realComplex = 0;

      for (let type of header.dataAxisType) {
        if (type === 'Real_Complex' && realComplex === 0) {
          dataSectionCount += 1;
          realComplex += 1;
        }

        if (type === 'Complex') {
          dataSectionCount *= 2;
        }
      }

      if (header.dataFormat !== 'One_D' && header.dataFormat !== 'Two_D') {
        throw new Error('Only One_D and two_D data formats are implemented yet');
      }

      if (header.dataFormat === 'One_D') {
        for (let s = 0; s < dataSectionCount; s++) {
          let section;

          if (header.dataType === '32Bit Float') {
            section = getArray(ioBuffer, header.dataPoints[0], 'readFloat32');
          } else if (header.dataType === '64Bit Float') {
            section = getArray(ioBuffer, header.dataPoints[0], 'readFloat64');
          }

          if (s === 0) data.re = section;
          if (s === 1) data.im = section;
        }
      }

      if (header.dataFormat === 'Two_D') {
        let me = 32;
        let dim1 = header.dataPoints[0];
        let dim2 = header.dataPoints[1]; // console.log(
        // `dim1: ${dim1},
        // dim2: ${dim2},
        // total: ${dim1 * dim2},
        // total(byte): ${dim1 * dim2 * 8},
        // total(length): ${dim1 * dim2 * 8 * dataSectionCount}
        // m size: ${dim1 / me} / ${dim2 / me}`,
        // );

        let I = dim2 / me;
        let J = dim1 / me;

        for (let s = 0; s < dataSectionCount; s++) {
          let section;

          for (let i = 0; i < I; i++) {
            let row = [];

            for (let j = 0; j < J; j++) {
              for (let k = 0; k < me; k++) {
                if (j === 0) {
                  if (header.dataType === '32Bit Float') {
                    row[k] = getArray(ioBuffer, me, 'readFloat32');
                  } else if (header.dataType === '64Bit Float') {
                    row[k] = getArray(ioBuffer, me, 'readFloat64');
                  }
                } else {
                  if (header.dataType === '32Bit Float') {
                    row[k] = row[k].concat(getArray(ioBuffer, me, 'readFloat32'));
                  } else if (header.dataType === '64Bit Float') {
                    row[k] = row[k].concat(getArray(ioBuffer, me, 'readFloat64'));
                  }
                }
              }
            }

            if (i === 0) {
              section = row;
            } else {
              section = section.concat(row);
            }
          }

          if (dataSectionCount === 2) {
            if (s === 0) data.re = section;
            if (s === 1) data.im = section;
          }

          if (dataSectionCount === 4) {
            if (s === 0) {
              data.re = {};
              data.re.re = section;
            }

            if (s === 1) data.re.im = section;

            if (s === 2) {
              data.im = {};
              data.im.re = section;
            }

            if (s === 3) data.im.im = section;
          }
        }
      } // format output


      let nucleus = [];
      let acquisitionTime = [];
      let spectralWidth = [];
      let spectralWidthClipped = [];
      let resolution = [];
      let originFrequency = [];
      let frequencyOffset = [];
      let dataUnits = [];

      if (header.dataFormat === 'One_D' || header.dataFormat === 'Two_D') {
        nucleus.push(getPar(parameters, 'x_domain').value);
        acquisitionTime.push(getMagnitude(parameters, 'x_acq_time'));
        spectralWidth.push(getMagnitude(parameters, 'x_sweep'));
        spectralWidthClipped.push(getMagnitude(parameters, 'x_sweep_clipped'));
        resolution.push(getMagnitude(parameters, 'x_resolution'));
        originFrequency.push(getMagnitude(parameters, 'x_freq'));
        frequencyOffset.push(getMagnitude(parameters, 'x_offset'));
        dataUnits.push(header.dataUnits[0].base);
      }

      if (header.dataFormat === 'Two_D') {
        nucleus.push(getPar(parameters, 'y_domain').value);
        acquisitionTime.push(getMagnitude(parameters, 'y_acq_time'));
        spectralWidth.push(getMagnitude(parameters, 'y_sweep'));
        resolution.push(getMagnitude(parameters, 'y_resolution'));
        originFrequency.push(getMagnitude(parameters, 'y_freq'));
        frequencyOffset.push(getMagnitude(parameters, 'y_offset'));
        dataUnits.push(header.dataUnits[1].base);
      }

      let digest = {
        info: {
          sampleName: getPar(parameters, 'sample_id').value,
          creationTime: header.creationTime,
          revisionTime: header.revisionTime,
          author: header.author,
          comment: header.comment,
          solvent: getPar(parameters, 'solvent').value,
          temperature: getMagnitude(parameters, 'temp_get'),
          probeName: getPar(parameters, 'probe_id').value,
          fieldStrength: getMagnitude(parameters, 'field_strength'),
          experiment: getPar(parameters, 'experiment').value,
          dimension: header.dataDimensionNumber,
          nucleus,
          pulseStrength90: getMagnitude(parameters, 'x90'),
          numberOfScans: getPar(parameters, 'scans').value,
          relaxationTime: getMagnitude(parameters, 'relaxation_delay'),
          dataPoints: header.dataPoints.slice(0, header.dataDimensionNumber),
          dataOffsetStart: header.dataOffsetStart,
          dataOffsetStop: header.dataOffsetStop,
          dataUnits: dataUnits,
          dataSections: Object.keys(data),
          originFrequency,
          frequencyOffset,
          acquisitionTime,
          spectralWidth,
          spectralWidthClipped,
          dataAxisStart: header.dataAxisStart,
          dataAxisStop: header.dataAxisStop,
          resolution: resolution,
          decimationRate: getPar(parameters, 'decimation_rate').value,
          paramList: JSON.stringify(parameters.paramArray.map(par => par.name))
        },
        headers: header,
        parameters: parameters,
        data: data
      };
      digest.info.digitalFilter = getDigitalFilter(parameters);
      return digest;
    }

    const gyromagneticRatio$1 = {
      '1H': 267.52218744e6,
      '2H': 41.065e6,
      '3H': 285.3508e6,
      '3He': -203.789e6,
      '7Li': 103.962e6,
      '13C': 67.28284e6,
      '14N': 19.331e6,
      '15N': -27.116e6,
      '17O': -36.264e6,
      '19F': 251.662e6,
      '23Na': 70.761e6,
      '27Al': 69.763e6,
      '29Si': -53.19e6,
      '31P': 108.291e6,
      '57Fe': 8.681e6,
      '63Cu': 71.118e6,
      '67Zn': 16.767e6,
      '129Xe': -73.997e6
    };

    var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

    function commonjsRequire (path) {
    	throw new Error('Could not dynamically require "' + path + '". Please configure the dynamicRequireTargets or/and ignoreDynamicRequires option of @rollup/plugin-commonjs appropriately for this require call to work.');
    }

    /*
        https://tools.ietf.org/html/rfc3629

        UTF8-char = UTF8-1 / UTF8-2 / UTF8-3 / UTF8-4

        UTF8-1    = %x00-7F

        UTF8-2    = %xC2-DF UTF8-tail

        UTF8-3    = %xE0 %xA0-BF UTF8-tail
                    %xE1-EC 2( UTF8-tail )
                    %xED %x80-9F UTF8-tail
                    %xEE-EF 2( UTF8-tail )

        UTF8-4    = %xF0 %x90-BF 2( UTF8-tail )
                    %xF1-F3 3( UTF8-tail )
                    %xF4 %x80-8F 2( UTF8-tail )

        UTF8-tail = %x80-BF
    */

    /**
     * Check if a Node.js Buffer or Uint8Array is UTF-8.
     */
    function isUtf8(buf) {
      if (!buf) {
        return false;
      }

      var i = 0;
      var len = buf.length;

      while (i < len) {
        // UTF8-1 = %x00-7F
        if (buf[i] <= 0x7F) {
          i++;
          continue;
        } // UTF8-2 = %xC2-DF UTF8-tail


        if (buf[i] >= 0xC2 && buf[i] <= 0xDF) {
          // if(buf[i + 1] >= 0x80 && buf[i + 1] <= 0xBF) {
          if (buf[i + 1] >> 6 === 2) {
            i += 2;
            continue;
          } else {
            return false;
          }
        } // UTF8-3 = %xE0 %xA0-BF UTF8-tail
        // UTF8-3 = %xED %x80-9F UTF8-tail


        if ((buf[i] === 0xE0 && buf[i + 1] >= 0xA0 && buf[i + 1] <= 0xBF || buf[i] === 0xED && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x9F) && buf[i + 2] >> 6 === 2) {
          i += 3;
          continue;
        } // UTF8-3 = %xE1-EC 2( UTF8-tail )
        // UTF8-3 = %xEE-EF 2( UTF8-tail )


        if ((buf[i] >= 0xE1 && buf[i] <= 0xEC || buf[i] >= 0xEE && buf[i] <= 0xEF) && buf[i + 1] >> 6 === 2 && buf[i + 2] >> 6 === 2) {
          i += 3;
          continue;
        } // UTF8-4 = %xF0 %x90-BF 2( UTF8-tail )
        //          %xF1-F3 3( UTF8-tail )
        //          %xF4 %x80-8F 2( UTF8-tail )


        if ((buf[i] === 0xF0 && buf[i + 1] >= 0x90 && buf[i + 1] <= 0xBF || buf[i] >= 0xF1 && buf[i] <= 0xF3 && buf[i + 1] >> 6 === 2 || buf[i] === 0xF4 && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x8F) && buf[i + 2] >> 6 === 2 && buf[i + 3] >> 6 === 2) {
          i += 4;
          continue;
        }

        return false;
      }

      return true;
    }

    /**
     * Ensure that the data is string. If it is an ArrayBuffer it will be converted to string using TextDecoder.
     * @param blob
     * @param options
     * @returns
     */

    function ensureString(blob) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

      if (typeof blob === 'string') {
        return blob;
      }

      if (ArrayBuffer.isView(blob) || blob instanceof ArrayBuffer) {
        const {
          encoding = guessEncoding(blob)
        } = options;
        const decoder = new TextDecoder(encoding);
        return decoder.decode(blob);
      }

      throw new TypeError(`blob must be a string, ArrayBuffer or ArrayBufferView`);
    }

    function guessEncoding(blob) {
      const uint8 = ArrayBuffer.isView(blob) ? new Uint8Array(blob.buffer, blob.byteOffset, blob.byteLength) : new Uint8Array(blob);

      if (uint8.length >= 2) {
        if (uint8[0] === 0xfe && uint8[1] === 0xff) {
          return 'utf-16be';
        }

        if (uint8[0] === 0xff && uint8[1] === 0xfe) {
          return 'utf-16le';
        }
      } //@ts-expect-error an ArrayBuffer is also ok


      if (!isUtf8(blob)) return 'latin1';
      return 'utf-8';
    }

    var medianQuickselect_min = {exports: {}};

    (function (module) {
      (function () {
        function a(d) {
          for (var e = 0, f = d.length - 1, g = void 0, h = void 0, i = void 0, j = c(e, f); !0;) {
            if (f <= e) return d[j];
            if (f == e + 1) return d[e] > d[f] && b(d, e, f), d[j];

            for (g = c(e, f), d[g] > d[f] && b(d, g, f), d[e] > d[f] && b(d, e, f), d[g] > d[e] && b(d, g, e), b(d, g, e + 1), h = e + 1, i = f; !0;) {
              do h++; while (d[e] > d[h]);

              do i--; while (d[i] > d[e]);

              if (i < h) break;
              b(d, h, i);
            }

            b(d, e, i), i <= j && (e = h), i >= j && (f = i - 1);
          }
        }

        var b = function b(d, e, f) {
          var _ref;

          return _ref = [d[f], d[e]], d[e] = _ref[0], d[f] = _ref[1], _ref;
        },
            c = function c(d, e) {
          return ~~((d + e) / 2);
        };

        module.exports ? module.exports = a : window.median = a;
      })();
    })(medianQuickselect_min);

    var quickSelectMedian = medianQuickselect_min.exports;

    function matrixCheck(data) {
      if (data.length === 0 || data[0].length === 0) {
        throw new RangeError('matrix should contain data');
      }

      const firstLength = data[0].length;

      for (let i = 1; i < data.length; i++) {
        if (data[i].length !== firstLength) {
          throw new RangeError('All rows should has the same length');
        }
      }
    }

    /**
     * Get min and max Z
     *
     * @param matrix - matrix [rows][cols].
     */

    function matrixMinMaxZ(matrix) {
      matrixCheck(matrix);
      const nbRows = matrix.length;
      const nbColumns = matrix[0].length;
      let min = matrix[0][0];
      let max = matrix[0][0];

      for (let column = 0; column < nbColumns; column++) {
        for (let row = 0; row < nbRows; row++) {
          if (matrix[row][column] < min) min = matrix[row][column];
          if (matrix[row][column] > max) max = matrix[row][column];
        }
      }

      return {
        min,
        max
      };
    }

    /**
     * Create an array with numbers starting from "from" with step "step" of length "length"
     *
     * @param options - options
     * @return - array of distributed numbers with step "step" from "from"
     */
    function createStepArray() {
      let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      let {
        from = 0,
        step = 1,
        length = 1000
      } = options;
      const array = new Float64Array(length);
      let index = 0;

      while (index < length) {
        array[index] = from + step * index;
        index++;
      }

      return array;
    }

    var name = "nmr-parser";
    var version = "1.9.0";
    var description = "Read and convert any NMR file";
    var main = "lib/index.js";
    var module = "src/index.js";
    var files = [
    	"lib",
    	"src"
    ];
    var scripts = {
    	eslint: "eslint src",
    	"eslint-fix": "npm run eslint -- --fix",
    	compile: "rollup -c",
    	prepublishOnly: "npm run compile",
    	test: "npm run compile && npm run test-coverage && npm run eslint",
    	"test-coverage": "jest --coverage",
    	"test-only": "jest",
    	build: "cheminfo-build --entry src/index.js --root NMRparser"
    };
    var repository = {
    	type: "git",
    	url: "git+https://github.com/cheminfo/nmr-parser.git"
    };
    var keywords = [
    	"nmr",
    	"magnetic resonance",
    	"parser",
    	"bruker",
    	"JEOL",
    	"CSDM",
    	"data analysis"
    ];
    var author = "Julien Wist";
    var license = "MIT";
    var bugs = {
    	url: "https://github.com/cheminfo/nmr-parser/issues"
    };
    var homepage = "https://github.com/cheminfo/nmr-parser#readme";
    var jest = {
    	testEnvironment: "node"
    };
    var prettier = {
    	arrowParens: "always",
    	semi: true,
    	singleQuote: true,
    	tabWidth: 2,
    	trailingComma: "all"
    };
    var devDependencies = {
    	"@babel/plugin-transform-modules-commonjs": "^7.17.9",
    	"@rollup/plugin-json": "^4.1.0",
    	"@types/jest": "^27.5.0",
    	"bruker-data-test": "^0.2.1",
    	"cheminfo-build": "^1.1.11",
    	eslint: "^8.15.0",
    	"eslint-config-cheminfo": "^7.3.0",
    	"jcamp-data-test": "^0.0.5",
    	"jeol-data-test": "^0.2.3",
    	jest: "^28.1.0",
    	prettier: "^2.6.2",
    	rollup: "^2.72.1"
    };
    var dependencies = {
    	brukerconverter: "^4.1.0",
    	"filelist-from": "^0.3.0",
    	"filelist-utils": "^0.5.0",
    	"is-any-array": "^2.0.0",
    	jcampconverter: "^9.0.1",
    	jeolconverter: "^1.0.1",
    	"nmr-processing": "^8.0.0"
    };
    var packageJson = {
    	name: name,
    	version: version,
    	description: description,
    	main: main,
    	module: module,
    	files: files,
    	scripts: scripts,
    	repository: repository,
    	keywords: keywords,
    	author: author,
    	license: license,
    	bugs: bugs,
    	homepage: homepage,
    	jest: jest,
    	prettier: prettier,
    	devDependencies: devDependencies,
    	dependencies: dependencies
    };

    function toKeyValue(object) {
      let newObject = {};

      for (let key in object) {
        if (typeof object[key] !== 'string') {
          newObject[key] = JSON.stringify(object[key]);
        } else {
          newObject[key] = object[key];
        }
      }

      return newObject;
    }

    function fromJEOL(buffer) {
      let parsedData = parseJEOL(buffer);
      let info = parsedData.info;
      let headers = parsedData.headers;
      let parameters = parsedData.parameters;
      let paramArray = Object.assign({}, parameters.paramArray);
      delete parameters.paramArray;
      let data = parsedData.data; // curation of parameters

      let newInfo = {};
      newInfo.title = `title: ${headers.title} / comment: ${headers.comment} / author:${headers.author} / site: ${headers.site}`;
      newInfo.nucleus = info.nucleus.map(x => {
        if (x === 'Proton') {
          x = '1H';
        }

        if (x === 'Carbon13') {
          x = '13C';
        }

        if (x === 'Nitrogen15') {
          x = '15N';
        }

        return x;
      });
      newInfo.sampleName = info.sampleName;
      newInfo.date = JSON.stringify(info.creationTime);
      newInfo.author = info.author; //newInfo.comment = info.comment;

      newInfo.solvent = info.solvent;
      newInfo.temperature = info.temperature.magnitude;
      newInfo.probeName = info.probeName || '';
      newInfo.fieldStrength = info.fieldStrength.magnitude;
      let gyromagneticRatioConstants = newInfo.nucleus.map(nucleus => gyromagneticRatio$1[nucleus]);
      newInfo.baseFrequency = gyromagneticRatioConstants.map(gmr => info.fieldStrength.magnitude * gmr / (2 * Math.PI * 1e6));
      newInfo.pulseSequence = info.experiment;
      newInfo.temperature = info.temperature.unit.toLowerCase() === 'celsius' ? 273.15 + info.temperature.magnitude : info.temperature.magnitude;
      newInfo.digitalFilter = info.digitalFilter;
      newInfo.pulseStrength90 = 1 / (4 * info.pulseStrength90.magnitude);
      newInfo.numberOfScans = info.numberOfScans;
      newInfo.relaxationTime = info.relaxationTime.magnitude;
      newInfo.isComplex = info.dataSections.includes('im');
      newInfo.isFid = info.dataUnits[0] === 'Second';
      newInfo.isFt = info.dataUnits[0] === 'Ppm';
      newInfo.dimension = info.dimension;
      const dimension = newInfo.dimension;
      newInfo.originFrequency = info.originFrequency.map(d => d.magnitude / 1e6).slice(0, dimension);
      newInfo.numberOfPoints = info.dataPoints.slice(0, 1);
      newInfo.frequencyOffset = info.frequencyOffset.map((f, i) => f.magnitude * newInfo.baseFrequency[i]).slice(0, dimension);
      newInfo.acquisitionTime = info.acquisitionTime.map(a => a.magnitude).slice(0, dimension);
      newInfo.spectralWidth = info.spectralWidth.map((sw, i) => sw.magnitude / info.originFrequency[i].magnitude * 1e6).slice(0, dimension); // set options for dimensions

      let dimensions = [];
      let options = {};
      let increment;

      for (let d = 0; d < info.dimension; d++) {
        increment = {
          magnitude: info.acquisitionTime[d].magnitude / (info.dataPoints[d] - 1),
          unit: 's'
        };

        if (info.dataUnits[d] === 'Second') {
          options.quantityName = 'time';
          options.originOffset = {
            magnitude: 0,
            unit: 's'
          };

          if (d === 0) {
            options.coordinatesOffset = {
              magnitude: info.digitalFilter * increment.magnitude,
              unit: 's'
            };
          } else {
            options.coordinatesOffset = {
              magnitude: 0,
              unit: 's'
            };
          }

          options.reciprocal = {
            originOffset: {
              magnitude: info.originFrequency[d].magnitude,
              unit: 'Hz'
            },
            quantityName: 'frequency',
            coordinatesOffset: {
              magnitude: info.frequencyOffset[d].magnitude * info.originFrequency[d].magnitude / 1000000,
              unit: 'Hz'
            }
          };
        } else if (info.dataUnits[d] === 'Ppm') {
          options.quantityName = 'frequency';
          let origin = info.originFrequency[d].magnitude;
          options.originOffset = {
            magnitude: origin,
            unit: 'Hz'
          };
          let firstPoint = info.dataOffsetStart[0];
          let lastPoint = info.dataOffsetStop[0];
          let dataLength = lastPoint - firstPoint + 1;
          let spectralWidth = info.spectralWidth[d].magnitude;
          let incr = spectralWidth / info.dataPoints[d];
          increment = {
            magnitude: incr,
            unit: 'Hz'
          };
          let offset = info.dataAxisStop[0] * origin / 1000000;
          options.coordinatesOffset = {
            magnitude: offset,
            unit: 'Hz'
          }; // after increment is computed with whole frequency
          // and original number of points, we recast the
          // number of point for export

          if (dataLength < info.dataPoints[d]) {
            info.dataPoints[d] = dataLength;
          }
        }

        if (d === 0) {
          options.description = 'direct dimension';
        } else {
          options.description = 'indirect dimension';
        }

        dimensions.push(formatLinearDimension(headers.dataAxisTitles[d], info.dataPoints[d], increment, options));
      } // set options for dependentVariable


      options = {
        unit: 'none',
        quantityName: 'relative intensity',
        from: info.dataOffsetStart,
        to: info.dataOffsetStop
      };
      let dependentVariables = [];
      dependentVariables.push(formatDependentVariable(data, 11, options));
      let description = Object.assign({}, newInfo);
      delete description.paramList;
      description.metadata = Object.assign({}, toKeyValue(headers), toKeyValue(parameters), toKeyValue(paramArray));
      let dataStructure = {
        timeStamp: Date.now(),
        version: packageJson.version,
        description,
        tags: ['magnetic resonance'].concat(newInfo.nucleus),
        application: {
          spectralWidthClipped: info.spectralWidthClipped[0].magnitude / newInfo.baseFrequency[0]
        },
        dimensions: dimensions,
        dependentVariables: dependentVariables
      };
      return [dataStructure];
    }

    /**
     * Retrieve the list of files for further process
     * @param {File[]} fileList
     * @param {object} [options={}]
     * @param {number|number[]} [options.processingNumber] - processing number to select, default the smallest number
     * @param {number|number[]} [options.experimentNumber] - experiment number to select, default all
     * @param {boolean} [options.ignoreFID=false] - should we ignore fid
     * @param {boolean} [options.ignoreFT=false] - should we ignore the ft transformed fid
     * @param {boolean} [options.ignore1D=false] - should we ignore 1D spectra
     * @param {boolean} [options.ignore2D=false] - should we ignore 2D spectra
     * @param {boolean} [options.onlyFirstProcessedData=true] - should we take only the first processed data (procno).
     */
    function groupByExperiments(fileList) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      let {
        processingNumber,
        experimentNumber,
        ignoreFID = false,
        ignoreFT = false,
        ignore1D = false,
        ignore2D = false,
        onlyFirstProcessedData = true
      } = options;

      if (typeof processingNumber === 'number') {
        processingNumber = [processingNumber];
      }

      if (typeof experimentNumber === 'number') {
        experimentNumber = [experimentNumber];
      }

      const experiments = {};

      for (let file of fileList) {
        let currentProcessingNo;
        let currentExperimentNo;
        let experiment;
        let name;
        let id;
        const parts = file.webkitRelativePath.split('/');

        if (file.webkitRelativePath.match(/\/pdata\/[0-9]+\/.*$/)) {
          currentProcessingNo = Number(parts[parts.length - 2]);
          currentExperimentNo = Number(parts[parts.length - 4]);
          name = parts[parts.length - 5];
          id = parts.slice(0, -3).join('/');
        } else if (file.webkitRelativePath.match(/[0-9]+\/.*$/)) {
          currentExperimentNo = Number(parts[parts.length - 2]);
          name = parts[parts.length - 3] || parts[parts.length - 2];
          id = parts.slice(0, -1).join('/');
        } else {
          continue;
        }

        if (experimentNumber && !experimentNumber.includes(currentExperimentNo)) {
          continue;
        }

        if (!experiments[id]) {
          experiments[id] = {
            name: name,
            expno: currentExperimentNo,
            processedData: {},
            fileList: []
          };
        }

        experiment = experiments[id];

        if (currentProcessingNo) {
          if (!experiment.processedData[currentProcessingNo]) {
            experiment.processedData[currentProcessingNo] = {
              fileList: [],
              name: name,
              expno: currentExperimentNo
            };
          }

          const processedData = experiment.processedData[currentProcessingNo];
          processedData.fileList.push(file);

          if (file.name.match(/^(1r|1i|2rr|procs|proc2s)$/)) {
            processedData[file.name] = file;
          }

          if (file.name === '2rr') {
            processedData.is2D = true;
            processedData.isFT = true;
          }

          if (file.name === '1r') {
            processedData.is1D = true;
            processedData.isFT = true;
          }
        } else {
          experiment.fileList.push(file);

          if (file.name.match(/^(ser|fid|acqus|acqu2s)$/)) {
            experiment[file.name] = file;
          }

          if (file.name === 'ser') {
            experiment.is2D = true;
            experiment.isFID = true;
          }

          if (file.name === 'fid') {
            experiment.is1D = true;
            experiment.isFID = true;
          }
        }
      }

      if (processingNumber) {
        // we need to deal with the list of processingNumber that is specified
        for (const key in experiments) {
          const processedData = experiments[key].processedData;
          const newProcessedData = {};

          for (let key in processedData) {
            if (!processingNumber.includes(parseInt(key, 10))) continue;
            newProcessedData[key] = processedData[key];
          }

          experiments[key].processedData = newProcessedData;
        }
      } else if (onlyFirstProcessedData) {
        // take the smallest processing and remove all the other
        for (const key in experiments) {
          const processedData = experiments[key].processedData;
          const firstProcessedNumber = Object.keys(processedData).sort((a, b) => Number(a) - Number(b))[0];

          if (firstProcessedNumber !== undefined) {
            experiments[key].processedData = {
              firstProcessedNumber: processedData[firstProcessedNumber]
            };
          }
        }
      } // console.log('experiments', experiments)
      // We convert the object to an array
      // and before filtering we will move all the processedData


      let experimentsArray = [];

      for (let key in experiments) {
        const experiment = { ...experiments[key]
        };
        const processed = experiment.processedData;
        delete experiment.processedData;

        if (experiment.ser || experiment.fid) {
          if (Object.keys(processed).length > 0) {
            const firstProcessed = processed[Object.keys(processed)[0]];

            if (firstProcessed.procs) {
              experiment.fileList.push(firstProcessed.procs);
              experiment.procs = firstProcessed.procs;
            }

            if (firstProcessed.proc2s) {
              experiment.fileList.push(firstProcessed.proc2s);
              experiment.proc2s = firstProcessed.proc2s;
            }
          }

          experimentsArray.push(experiment);
        }

        for (let processedKey in processed) {
          const oneProcessed = processed[processedKey];

          if (oneProcessed['1r'] || oneProcessed['2rr']) {
            if (experiment.acqus) {
              oneProcessed.fileList.push(experiment.acqus);
            }

            if (experiment.acqu2s) {
              oneProcessed.fileList.push(experiment.acqu2s);
            }

            experimentsArray.push({
              acqus: experiment.acqus,
              acqu2s: experiment.acqu2s,
              ...oneProcessed
            });
          }
        }
      } // we can not easily filter


      if (ignoreFID) {
        experimentsArray = experimentsArray.filter(item => !item.isFID);
      }

      if (ignoreFT) {
        experimentsArray = experimentsArray.filter(item => !item.isFT);
      }

      if (ignore1D) {
        experimentsArray = experimentsArray.filter(item => !item.is1D);
      }

      if (ignore2D) {
        experimentsArray = experimentsArray.filter(item => !item.is2D);
      } // console.log(experimentsArray)


      return experimentsArray;
    }

    function joinInfoMeta(target, toAppend) {
      for (let key in toAppend.meta) {
        if (target.meta[key] === undefined) {
          target.meta[key] = toAppend.meta[key];
        }
      }

      for (let key in toAppend.info) {
        if (target.info[key] === undefined) {
          target.info[key] = toAppend.info[key];
        }
      }

      for (let key in target.meta) {
        if (!Array.isArray(target.meta[key])) {
          target.meta[key] = [target.meta[key]];
        }
      }
    }

    /**
     * Dynamically type a string
     * @param {string} value String to dynamically type
     * @returns {boolean|string|number}
     */
    function parseString(value) {
      if (value.length === 4 || value.length === 5) {
        let lowercase = value.toLowerCase();
        if (lowercase === 'true') return true;
        if (lowercase === 'false') return false;
      }

      let number = Number(value);

      if (number === 0 && !value.includes('0')) {
        return value;
      }

      if (!Number.isNaN(number)) return number;
      return value;
    }

    const GC_MS_FIELDS$1 = ['TIC', '.RIC', 'SCANNUMBER'];
    function complexChromatogram$1(result) {
      let spectra = result.spectra;
      let length = spectra.length;
      let chromatogram = {
        times: new Array(length),
        series: {
          ms: {
            dimension: 2,
            data: new Array(length)
          }
        }
      };
      let existingGCMSFields = [];

      for (let i = 0; i < GC_MS_FIELDS$1.length; i++) {
        let label = convertMSFieldToLabel$1(GC_MS_FIELDS$1[i]);

        if (spectra[0][label]) {
          existingGCMSFields.push(label);
          chromatogram.series[label] = {
            dimension: 1,
            data: new Array(length)
          };
        }
      }

      for (let i = 0; i < length; i++) {
        let spectrum = spectra[i];
        chromatogram.times[i] = spectrum.pageValue;

        for (let j = 0; j < existingGCMSFields.length; j++) {
          chromatogram.series[existingGCMSFields[j]].data[i] = Number(spectrum[existingGCMSFields[j]]);
        }

        if (spectrum.data) {
          chromatogram.series.ms.data[i] = [spectrum.data.x, spectrum.data.y];
        }
      }

      result.chromatogram = chromatogram;
    }
    function isMSField$1(canonicDataLabel) {
      return GC_MS_FIELDS$1.indexOf(canonicDataLabel) !== -1;
    }
    function convertMSFieldToLabel$1(value) {
      return value.toLowerCase().replace(/[^a-z0-9]/g, '');
    }

    function convertToFloatArray$2(stringArray) {
      let floatArray = [];

      for (let i = 0; i < stringArray.length; i++) {
        floatArray.push(Number(stringArray[i]));
      }

      return floatArray;
    }

    function fastParseXYData$1(spectrum, value) {
      // TODO need to deal with result
      //  console.log(value);
      // we check if deltaX is defined otherwise we calculate it
      let yFactor = spectrum.yFactor;
      let deltaX = spectrum.deltaX;
      spectrum.isXYdata = true;
      let currentData = {
        x: [],
        y: []
      };
      spectrum.data = currentData;
      let currentX = spectrum.firstX;
      let currentY = spectrum.firstY; // we skip the first line
      //

      let endLine = false;
      let ascii;
      let i = 0;

      for (; i < value.length; i++) {
        ascii = value.charCodeAt(i);

        if (ascii === 13 || ascii === 10) {
          endLine = true;
        } else {
          if (endLine) break;
        }
      } // we proceed taking the i after the first line


      let newLine = true;
      let isDifference = false;
      let isLastDifference = false;
      let lastDifference = 0;
      let isDuplicate = false;
      let inComment = false;
      let currentValue = 0; // can be a difference or a duplicate

      let lastValue = 0; // must be the real last value

      let isNegative = false;
      let inValue = false;
      let skipFirstValue = false;
      let decimalPosition = 0;

      for (; i <= value.length; i++) {
        if (i === value.length) ascii = 13;else ascii = value.charCodeAt(i);

        if (inComment) {
          // we should ignore the text if we are after $$
          if (ascii === 13 || ascii === 10) {
            newLine = true;
            inComment = false;
          }
        } else {
          // when is it a new value ?
          // when it is not a digit, . or comma
          // it is a number that is either new or we continue
          if (ascii <= 57 && ascii >= 48) {
            // a number
            inValue = true;

            if (decimalPosition > 0) {
              currentValue += (ascii - 48) / Math.pow(10, decimalPosition++);
            } else {
              currentValue *= 10;
              currentValue += ascii - 48;
            }
          } else if (ascii === 44 || ascii === 46) {
            // a "," or "."
            inValue = true;
            decimalPosition++;
          } else {
            if (inValue) {
              // need to process the previous value
              if (newLine) {
                newLine = false; // we don't check the X value
                // console.log("NEW LINE",isDifference, lastDifference);
                // if new line and lastDifference, the first value is just a check !
                // that we don't check ...

                if (isLastDifference) skipFirstValue = true;
              } else {
                // need to deal with duplicate and differences
                if (skipFirstValue) {
                  skipFirstValue = false;
                } else {
                  if (isDifference) {
                    lastDifference = isNegative ? 0 - currentValue : currentValue;
                    isLastDifference = true;
                    isDifference = false;
                  } else if (!isDuplicate) {
                    lastValue = isNegative ? 0 - currentValue : currentValue;
                  }

                  let duplicate = isDuplicate ? currentValue - 1 : 1;

                  for (let j = 0; j < duplicate; j++) {
                    if (isLastDifference) {
                      currentY += lastDifference;
                    } else {
                      currentY = lastValue;
                    }

                    currentData.x.push(currentX);
                    currentData.y.push(currentY * yFactor);
                    currentX += deltaX;
                  }
                }
              }

              isNegative = false;
              currentValue = 0;
              decimalPosition = 0;
              inValue = false;
              isDuplicate = false;
            } // positive SQZ digits @ A B C D E F G H I (ascii 64-73)


            if (ascii < 74 && ascii > 63) {
              inValue = true;
              isLastDifference = false;
              currentValue = ascii - 64;
            } else if (ascii > 96 && ascii < 106) {
              // negative SQZ digits a b c d e f g h i (ascii 97-105)
              inValue = true;
              isLastDifference = false;
              currentValue = ascii - 96;
              isNegative = true;
            } else if (ascii === 115) {
              // DUP digits S T U V W X Y Z s (ascii 83-90, 115)
              inValue = true;
              isDuplicate = true;
              currentValue = 9;
            } else if (ascii > 82 && ascii < 91) {
              inValue = true;
              isDuplicate = true;
              currentValue = ascii - 82;
            } else if (ascii > 73 && ascii < 83) {
              // positive DIF digits % J K L M N O P Q R (ascii 37, 74-82)
              inValue = true;
              isDifference = true;
              currentValue = ascii - 73;
            } else if (ascii > 105 && ascii < 115) {
              // negative DIF digits j k l m n o p q r (ascii 106-114)
              inValue = true;
              isDifference = true;
              currentValue = ascii - 105;
              isNegative = true;
            } else if (ascii === 36 && value.charCodeAt(i + 1) === 36) {
              // $ sign, we need to check the next one
              inValue = true;
              inComment = true;
            } else if (ascii === 37) {
              // positive DIF digits % J K L M N O P Q R (ascii 37, 74-82)
              inValue = true;
              isDifference = true;
              currentValue = 0;
              isNegative = false;
            } else if (ascii === 45) {
              // a "-"
              // check if after there is a number, decimal or comma
              let ascii2 = value.charCodeAt(i + 1);

              if (ascii2 >= 48 && ascii2 <= 57 || ascii2 === 44 || ascii2 === 46) {
                inValue = true;
                if (!newLine) isLastDifference = false;
                isNegative = true;
              }
            } else if (ascii === 13 || ascii === 10) {
              newLine = true;
              inComment = false;
            } // and now analyse the details ... space or tabulation
            // if "+" we just don't care

          }
        }
      }
    }

    const removeCommentRegExp$1 = /\$\$.*/;
    const peakTableSplitRegExp$1 = /[,\t ]+/;
    function parsePeakTable$1(spectrum, value, result) {
      spectrum.isPeaktable = true;

      if (!spectrum.variables || Object.keys(spectrum.variables) === 2) {
        parseXY$1(spectrum, value, result);
      } else {
        parseXYZ$1(spectrum, value, result);
      } // we will add the data in the variables


      if (spectrum.variables) {
        for (let key in spectrum.variables) {
          spectrum.variables[key].data = spectrum.data[key];
        }
      }
    }

    function parseXY$1(spectrum, value, result) {
      let currentData = {
        x: [],
        y: []
      };
      spectrum.data = currentData; // counts for around 20% of the time

      let lines = value.split(/,? *,?[;\r\n]+ */);

      for (let i = 1; i < lines.length; i++) {
        let values = lines[i].trim().replace(removeCommentRegExp$1, '').split(peakTableSplitRegExp$1);

        if (values.length % 2 === 0) {
          for (let j = 0; j < values.length; j = j + 2) {
            // takes around 40% of the time to add and parse the 2 values nearly exclusively because of Number
            currentData.x.push(Number(values[j]) * spectrum.xFactor);
            currentData.y.push(Number(values[j + 1]) * spectrum.yFactor);
          }
        } else {
          result.logs.push(`Format error: ${values}`);
        }
      }
    }

    function parseXYZ$1(spectrum, value, result) {
      let currentData = {};
      let variables = Object.keys(spectrum.variables);
      let numberOfVariables = variables.length;
      variables.forEach(variable => currentData[variable] = []);
      spectrum.data = currentData; // counts for around 20% of the time

      let lines = value.split(/,? *,?[;\r\n]+ */);

      for (let i = 1; i < lines.length; i++) {
        let values = lines[i].trim().replace(removeCommentRegExp$1, '').split(peakTableSplitRegExp$1);

        if (values.length % numberOfVariables === 0) {
          for (let j = 0; j < values.length; j++) {
            // todo should try to find a xFactor (y, ...)
            currentData[variables[j % numberOfVariables]].push(Number(values[j]));
          }
        } else {
          result.logs.push(`Format error: ${values}`);
        }
      }
    }

    function parseXYA$1(spectrum, value) {
      let removeSymbolRegExp = /(\(+|\)+|<+|>+|\s+)/g;
      spectrum.isXYAdata = true;
      let values;
      let currentData = {
        x: [],
        y: []
      };
      spectrum.data = currentData;
      let lines = value.split(/,? *,?[;\r\n]+ */);

      for (let i = 1; i < lines.length; i++) {
        values = lines[i].trim().replace(removeSymbolRegExp, '').split(',');
        currentData.x.push(Number(values[0]));
        currentData.y.push(Number(values[1]));
      }
    }

    function median(input) {
      if (!isAnyArray(input)) {
        throw new TypeError('input must be an array');
      }

      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }

      return quickSelectMedian(input.slice());
    }

    function convertTo3DZ$2(spectra) {
      let minZ = spectra[0].data.y[0];
      let maxZ = minZ;
      let ySize = spectra.length;
      let xSize = spectra[0].data.x.length;
      let z = new Array(ySize);

      for (let i = 0; i < ySize; i++) {
        z[i] = spectra[i].data.y;

        for (let j = 0; j < xSize; j++) {
          let value = z[i][j];
          if (value < minZ) minZ = value;
          if (value > maxZ) maxZ = value;
        }
      }

      const firstX = spectra[0].data.x[0];
      const lastX = spectra[0].data.x[spectra[0].data.x.length - 1]; // has to be -2 because it is a 1D array [x,y,x,y,...]

      const firstY = spectra[0].pageValue;
      const lastY = spectra[ySize - 1].pageValue; // Because the min / max value are the only information about the matrix if we invert
      // min and max we need to invert the array

      if (firstX > lastX) {
        for (let spectrum of z) {
          spectrum.reverse();
        }
      }

      if (firstY > lastY) {
        z.reverse();
      }

      const medians = [];

      for (let i = 0; i < z.length; i++) {
        const row = Float64Array.from(z[i]);

        for (let i = 0; i < row.length; i++) {
          if (row[i] < 0) row[i] = -row[i];
        }

        medians.push(median(row));
      }

      const median$1 = median(medians);
      return {
        z: z,
        minX: Math.min(firstX, lastX),
        maxX: Math.max(firstX, lastX),
        minY: Math.min(firstY, lastY),
        maxY: Math.max(firstY, lastY),
        minZ: minZ,
        maxZ: maxZ,
        noise: median$1
      };
    }

    function generateContourLines$1(zData, options) {
      let noise = zData.noise;
      let z = zData.z;
      let povarHeight0, povarHeight1, povarHeight2, povarHeight3;
      let isOver0, isOver1, isOver2, isOver3;
      let nbSubSpectra = z.length;
      let nbPovars = z[0].length;
      let pAx, pAy, pBx, pBy;
      let x0 = zData.minX;
      let xN = zData.maxX;
      let dx = (xN - x0) / (nbPovars - 1);
      let y0 = zData.minY;
      let yN = zData.maxY;
      let dy = (yN - y0) / (nbSubSpectra - 1);
      let minZ = zData.minZ;
      let maxZ = zData.maxZ; // System.out.prvarln('y0 '+y0+' yN '+yN);
      // -------------------------
      // Povars attribution
      //
      // 0----1
      // |  / |
      // | /  |
      // 2----3
      //
      // ---------------------d------

      let iter = options.nbContourLevels * 2;
      let contourLevels = new Array(iter);
      let lineZValue;

      for (let level = 0; level < iter; level++) {
        // multiply by 2 for positif and negatif
        let contourLevel = {};
        contourLevels[level] = contourLevel;
        let side = level % 2;
        let factor = (maxZ - options.noiseMultiplier * noise) * Math.exp((level >> 1) - options.nbContourLevels);

        if (side === 0) {
          lineZValue = factor + options.noiseMultiplier * noise;
        } else {
          lineZValue = 0 - factor - options.noiseMultiplier * noise;
        }

        let lines = [];
        contourLevel.zValue = lineZValue;
        contourLevel.lines = lines;
        if (lineZValue <= minZ || lineZValue >= maxZ) continue;

        for (let iSubSpectra = 0; iSubSpectra < nbSubSpectra - 1; iSubSpectra++) {
          let subSpectra = z[iSubSpectra];
          let subSpectraAfter = z[iSubSpectra + 1];

          for (let povar = 0; povar < nbPovars - 1; povar++) {
            povarHeight0 = subSpectra[povar];
            povarHeight1 = subSpectra[povar + 1];
            povarHeight2 = subSpectraAfter[povar];
            povarHeight3 = subSpectraAfter[povar + 1];
            isOver0 = povarHeight0 > lineZValue;
            isOver1 = povarHeight1 > lineZValue;
            isOver2 = povarHeight2 > lineZValue;
            isOver3 = povarHeight3 > lineZValue; // Example povar0 is over the plane and povar1 and
            // povar2 are below, we find the varersections and add
            // the segment

            if (isOver0 !== isOver1 && isOver0 !== isOver2) {
              pAx = povar + (lineZValue - povarHeight0) / (povarHeight1 - povarHeight0);
              pAy = iSubSpectra;
              pBx = povar;
              pBy = iSubSpectra + (lineZValue - povarHeight0) / (povarHeight2 - povarHeight0);
              lines.push(pAx * dx + x0);
              lines.push(pAy * dy + y0);
              lines.push(pBx * dx + x0);
              lines.push(pBy * dy + y0);
            } // remove push does not help !!!!


            if (isOver3 !== isOver1 && isOver3 !== isOver2) {
              pAx = povar + 1;
              pAy = iSubSpectra + 1 - (lineZValue - povarHeight3) / (povarHeight1 - povarHeight3);
              pBx = povar + 1 - (lineZValue - povarHeight3) / (povarHeight2 - povarHeight3);
              pBy = iSubSpectra + 1;
              lines.push(pAx * dx + x0);
              lines.push(pAy * dy + y0);
              lines.push(pBx * dx + x0);
              lines.push(pBy * dy + y0);
            } // test around the diagonal


            if (isOver1 !== isOver2) {
              pAx = (povar + 1 - (lineZValue - povarHeight1) / (povarHeight2 - povarHeight1)) * dx + x0;
              pAy = (iSubSpectra + (lineZValue - povarHeight1) / (povarHeight2 - povarHeight1)) * dy + y0;

              if (isOver1 !== isOver0) {
                pBx = povar + 1 - (lineZValue - povarHeight1) / (povarHeight0 - povarHeight1);
                pBy = iSubSpectra;
                lines.push(pAx);
                lines.push(pAy);
                lines.push(pBx * dx + x0);
                lines.push(pBy * dy + y0);
              }

              if (isOver2 !== isOver0) {
                pBx = povar;
                pBy = iSubSpectra + 1 - (lineZValue - povarHeight2) / (povarHeight0 - povarHeight2);
                lines.push(pAx);
                lines.push(pAy);
                lines.push(pBx * dx + x0);
                lines.push(pBy * dy + y0);
              }

              if (isOver1 !== isOver3) {
                pBx = povar + 1;
                pBy = iSubSpectra + (lineZValue - povarHeight1) / (povarHeight3 - povarHeight1);
                lines.push(pAx);
                lines.push(pAy);
                lines.push(pBx * dx + x0);
                lines.push(pBy * dy + y0);
              }

              if (isOver2 !== isOver3) {
                pBx = povar + (lineZValue - povarHeight2) / (povarHeight3 - povarHeight2);
                pBy = iSubSpectra + 1;
                lines.push(pAx);
                lines.push(pAy);
                lines.push(pBx * dx + x0);
                lines.push(pBy * dy + y0);
              }
            }
          }
        }
      }

      return {
        minX: zData.minX,
        maxX: zData.maxX,
        minY: zData.minY,
        maxY: zData.maxY,
        segments: contourLevels
      };
    }

    function add2D$1(result, options) {
      let zData = convertTo3DZ$2(result.spectra);

      if (!options.noContour) {
        result.contourLines = generateContourLines$1(zData, options);
        delete zData.z;
      }

      result.minMax = zData;
    }

    // sources:
    // https://en.wikipedia.org/wiki/Gyromagnetic_ratio
    const gyromagneticRatio = {
      '1H': 267.52218744e6,
      '2H': 41.065e6,
      '3H': 285.3508e6,
      '3He': -203.789e6,
      '7Li': 103.962e6,
      '13C': 67.28284e6,
      '14N': 19.331e6,
      '15N': -27.116e6,
      '17O': -36.264e6,
      '19F': 251.662e6,
      '23Na': 70.761e6,
      '27Al': 69.763e6,
      '29Si': -53.19e6,
      '31P': 108.291e6,
      '57Fe': 8.681e6,
      '63Cu': 71.118e6,
      '67Zn': 16.767e6,
      '129Xe': -73.997e6
    };

    function postProcessingNMR$1(entriesFlat) {
      // specific NMR functions
      for (let entry of entriesFlat) {
        let observeFrequency = 0;
        let shiftOffsetVal = 0;

        for (let spectrum of entry.spectra) {
          if (entry.ntuples && entry.ntuples.symbol) {
            if (!observeFrequency && spectrum.observeFrequency) {
              observeFrequency = spectrum.observeFrequency;
            }

            if (!shiftOffsetVal && spectrum.shiftOffsetVal) {
              shiftOffsetVal = spectrum.shiftOffsetVal;
            }
          } else {
            observeFrequency = spectrum.observeFrequency;
            shiftOffsetVal = spectrum.shiftOffsetVal;
          }

          if (observeFrequency) {
            if (spectrum.xUnits && spectrum.xUnits.toUpperCase().includes('HZ')) {
              spectrum.xUnits = 'PPM';
              spectrum.xFactor = spectrum.xFactor / observeFrequency;
              spectrum.firstX = spectrum.firstX / observeFrequency;
              spectrum.lastX = spectrum.lastX / observeFrequency;
              spectrum.deltaX = spectrum.deltaX / observeFrequency;

              for (let i = 0; i < spectrum.data.x.length; i++) {
                spectrum.data.x[i] /= observeFrequency;
              }
            }
          }

          if (shiftOffsetVal) {
            let shift = spectrum.firstX - shiftOffsetVal;
            spectrum.firstX = spectrum.firstX - shift;
            spectrum.lastX = spectrum.lastX - shift;

            for (let i = 0; i < spectrum.data.x.length; i++) {
              spectrum.data.x[i] -= shift;
            }
          } // we will check if some nucleus are missing ...


          if (entry.ntuples && entry.ntuples.nucleus && entry.ntuples.symbol) {
            for (let i = 0; i < entry.ntuples.nucleus.length; i++) {
              let symbol = entry.ntuples.symbol[i];
              let nucleus = entry.ntuples.nucleus[i];

              if (symbol.startsWith('F') && !nucleus) {
                if (symbol === 'F1') {
                  // if F1 is defined we will use F2
                  if (entry.tmp.$NUC2) {
                    entry.ntuples.nucleus[i] = entry.tmp.$NUC2;
                  } else {
                    let f2index = entry.ntuples.symbol.indexOf('F2');

                    if (f2index && entry.ntuples.nucleus[f2index]) {
                      entry.ntuples.nucleus[i] = entry.ntuples.nucleus[f2index];
                    }
                  }
                }

                if (symbol === 'F2') entry.ntuples.nucleus[i] = entry.tmp.$NUC1;
              }

              if (symbol === 'F2') {
                entry.yType = entry.ntuples.nucleus[0];
              }
            }
          }

          if (observeFrequency && entry.ntuples && entry.ntuples.symbol && entry.ntuples.nucleus) {
            let unit = '';
            let pageSymbolIndex = entry.ntuples.symbol.indexOf(spectrum.pageSymbol);

            if (entry.ntuples.units && entry.ntuples.units[pageSymbolIndex]) {
              unit = entry.ntuples.units[pageSymbolIndex];
            }

            if (unit !== 'PPM') {
              if (pageSymbolIndex !== 0) {
                throw Error('Not sure about this ntuples format');
              }

              let ratio0 = gyromagneticRatio[entry.ntuples.nucleus[0]];
              let ratio1 = gyromagneticRatio[entry.ntuples.nucleus[1]];

              if (!ratio0 || !ratio1) {
                throw Error('Problem with determination of gyromagnetic ratio');
              }

              let ratio = ratio0 / ratio1 * observeFrequency;
              spectrum.pageValue /= ratio;
            }
          }
        }
      }
    }

    function profiling$1(result, action, options) {
      if (result.profiling) {
        result.profiling.push({
          action,
          time: Date.now() - options.start
        });
      }
    }

    function simpleChromatogram$1(result) {
      let data = result.spectra[0].data;
      result.chromatogram = {
        times: data.x.slice(),
        series: {
          intensity: {
            dimension: 1,
            data: data.y.slice()
          }
        }
      };
    }

    function postProcessing$1(entriesFlat, result, options) {
      // converting Hz to ppm
      postProcessingNMR$1(entriesFlat);

      for (let entry of entriesFlat) {
        if (Object.keys(entry.ntuples).length > 0) {
          let newNtuples = [];
          let keys = Object.keys(entry.ntuples);

          for (let i = 0; i < keys.length; i++) {
            let key = keys[i];
            let values = entry.ntuples[key];

            for (let j = 0; j < values.length; j++) {
              if (!newNtuples[j]) newNtuples[j] = {};
              newNtuples[j][key] = values[j];
            }
          }

          entry.ntuples = newNtuples;
        }

        if (entry.twoD && options.wantXY) {
          add2D$1(entry, options);
          profiling$1(result, 'Finished countour plot calculation', options);

          if (!options.keepSpectra) {
            delete entry.spectra;
          }
        } // maybe it is a GC (HPLC) / MS. In this case we add a new format


        if (options.chromatogram) {
          if (entry.spectra.length > 1) {
            complexChromatogram$1(entry);
          } else {
            simpleChromatogram$1(entry);
          }

          profiling$1(result, 'Finished chromatogram calculation', options);
        }

        delete entry.tmp;
      }
    }

    function prepareNtuplesDatatable$1(currentEntry, spectrum, kind) {
      let xIndex = -1;
      let yIndex = -1;
      let firstVariable = '';
      let secondVariable = '';

      if (kind.indexOf('++') > 0) {
        firstVariable = kind.replace(/.*\(([a-zA-Z0-9]+)\+\+.*/, '$1');
        secondVariable = kind.replace(/.*\.\.([a-zA-Z0-9]+).*/, '$1');
      } else {
        kind = kind.replace(/[^a-zA-Z]/g, '');
        firstVariable = kind.charAt(0);
        secondVariable = kind.charAt(1);
        spectrum.variables = {};

        for (let symbol of kind) {
          let lowerCaseSymbol = symbol.toLowerCase();
          let index = currentEntry.ntuples.symbol.indexOf(symbol);
          if (index === -1) throw Error(`Symbol undefined: ${symbol}`);
          spectrum.variables[lowerCaseSymbol] = {};

          for (let key in currentEntry.ntuples) {
            if (currentEntry.ntuples[key][index]) {
              spectrum.variables[lowerCaseSymbol][key.replace(/^var/, '')] = currentEntry.ntuples[key][index];
            }
          }
        }
      }

      xIndex = currentEntry.ntuples.symbol.indexOf(firstVariable);
      yIndex = currentEntry.ntuples.symbol.indexOf(secondVariable);
      if (xIndex === -1) xIndex = 0;
      if (yIndex === -1) yIndex = 0;

      if (currentEntry.ntuples.first) {
        if (currentEntry.ntuples.first.length > xIndex) {
          spectrum.firstX = currentEntry.ntuples.first[xIndex];
        }

        if (currentEntry.ntuples.first.length > yIndex) {
          spectrum.firstY = currentEntry.ntuples.first[yIndex];
        }
      }

      if (currentEntry.ntuples.last) {
        if (currentEntry.ntuples.last.length > xIndex) {
          spectrum.lastX = currentEntry.ntuples.last[xIndex];
        }

        if (currentEntry.ntuples.last.length > yIndex) {
          spectrum.lastY = currentEntry.ntuples.last[yIndex];
        }
      }

      if (currentEntry.ntuples.vardim && currentEntry.ntuples.vardim.length > xIndex) {
        spectrum.nbPoints = currentEntry.ntuples.vardim[xIndex];
      }

      if (currentEntry.ntuples.factor) {
        if (currentEntry.ntuples.factor.length > xIndex) {
          spectrum.xFactor = currentEntry.ntuples.factor[xIndex];
        }

        if (currentEntry.ntuples.factor.length > yIndex) {
          spectrum.yFactor = currentEntry.ntuples.factor[yIndex];
        }
      }

      if (currentEntry.ntuples.units) {
        if (currentEntry.ntuples.units.length > xIndex) {
          if (currentEntry.ntuples.varname && currentEntry.ntuples.varname[xIndex]) {
            spectrum.xUnits = `${currentEntry.ntuples.varname[xIndex]} [${currentEntry.ntuples.units[xIndex]}]`;
          } else {
            spectrum.xUnits = currentEntry.ntuples.units[xIndex];
          }
        }

        if (currentEntry.ntuples.units.length > yIndex) {
          if (currentEntry.ntuples.varname && currentEntry.ntuples.varname[yIndex]) {
            spectrum.yUnits = `${currentEntry.ntuples.varname[yIndex]} [${currentEntry.ntuples.units[yIndex]}]`;
          } else {
            spectrum.yUnits = currentEntry.ntuples.units[yIndex];
          }
        }
      }
    }

    function prepareSpectrum$1(spectrum) {
      if (!spectrum.xFactor) spectrum.xFactor = 1;
      if (!spectrum.yFactor) spectrum.yFactor = 1;
    }

    const ntuplesSeparatorRegExp$1 = /[ \t]*,[ \t]*/;
    const defaultOptions$2 = {
      keepRecordsRegExp: /^$/,
      canonicDataLabels: true,
      canonicMetadataLabels: false,
      dynamicTyping: true,
      withoutXY: false,
      chromatogram: false,
      keepSpectra: false,
      noContour: false,
      nbContourLevels: 7,
      noiseMultiplier: 5,
      profiling: false
    };
    /**
     *
     * @typedef {object} ConvertOptions
     * @property {RegExp} [options.keepRecordsRegExp=/^$/] - By default we don't keep meta information.
     * @property {boolean} [options.canonicDataLabels=true] - Canonize the Labels (uppercase without symbol).
     * @property {boolean} [options.canonicMetadataLabels=false] - Canonize the metadata Labels (uppercase without symbol).
     * @property {boolean} [options.dynamicTyping=false] - Convert numbers to Number.
     * @property {boolean} [options.withoutXY=false] - Remove the XY data.
     * @property {boolean} [options.chromatogram=false] - Special post-processing for GC / HPLC / MS.
     * @property {boolean} [options.keepSpectra=false] - Force to keep the spectra in case of 2D.
     * @property {boolean} [options.noContour=false] - Don't calculate countour in case of 2D.
     * @property {number} [options.nbContourLevels=7] - Number of positive / negative contour levels to calculate.
     * @property {number} [options.noiseMultiplier=5] - Define for 2D the level as 5 times the median as default.
     * @property {boolean} [options.profiling=false] - Add profiling information.
     */

    /**
     *
     * @typedef {object} Ntuples
     * @property {string[]} [varname]
     * @property {string[]} [symbol]
     * @property {string[]} [vartype]
     * @property {string[]} [varform]
     * @property {number[]} [vardim]
     * @property {string[]} [units]
     * @property {number[]} [factor]
     * @property {number[]} [first]
     * @property {number[]} [last]
     * @property {number[]} [min]
     * @property {number[]} [max]
     * @property {string[]} [nucleus]
     */

    /**
     * @typedef { Record<string, any> } Spectrum
     * @property {Record<string, number[]>} [data]
     * @property {number} [firstX]
     * @property {number} [lastX]
     * @property {number} [deltaX]
     * @property {number} [yFactor]
     * @property {number} [xFactor]
     * @property {number} [nbPoints]
     */

    /**
     *
     * @typedef {object} Entry
     * @property {Spectrum[]} spectra
     * @property {Ntuples} ntuples
     * @property {object} meta
     * @property {object} tmp
     * @property {string} [title]
     * @property {string} [dataType]
     * @property {string} [dataClass]
     * @property {boolean} [twoD]
     */

    /**
     *
     * @typedef { object } ConvertResult
     * @property { object[] | boolean } profiling
     * @property { string[] } logs
     * @property { object[] } entries
     * @property { Entry[] } flatten
     */

    /**
     * Parse a jcamp.
     *
     * @param {string|ArrayBuffer|Uint8Array} jcamp
     * @param {ConvertOptions} [options]
     * @returns {ConvertResult}
     */

    function convert$1(jcamp) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      jcamp = ensureString(jcamp);
      options = { ...defaultOptions$2,
        ...options
      };
      options.wantXY = !options.withoutXY;
      options.start = Date.now();
      let entriesFlat = [];
      let result = {
        profiling: options.profiling ? [] : false,
        logs: [],
        entries: []
      };
      let tmpResult = {
        children: []
      };
      let currentEntry = tmpResult;
      let parentsStack = [];
      let spectrum = {};

      if (typeof jcamp !== 'string') {
        throw new TypeError('the JCAMP should be a string');
      }

      profiling$1(result, 'Before split to LDRS', options);
      let ldrs = jcamp.replace(/[\r\n]+##/g, '\n##').split('\n##');
      profiling$1(result, 'Split to LDRS', options);
      if (ldrs[0]) ldrs[0] = ldrs[0].replace(/^[\r\n ]*##/, '');

      for (let ldr of ldrs) {
        // This is a new LDR
        let position = ldr.indexOf('=');
        let dataLabel = position > 0 ? ldr.substring(0, position) : ldr;
        let dataValue = position > 0 ? ldr.substring(position + 1).trim() : '';
        let canonicDataLabel = dataLabel.replace(/[_ -]/g, '').toUpperCase();

        if (canonicDataLabel === 'DATATABLE') {
          let endLine = dataValue.indexOf('\n');
          if (endLine === -1) endLine = dataValue.indexOf('\r');

          if (endLine > 0) {
            // ##DATA TABLE= (X++(I..I)), XYDATA
            // We need to find the variables
            let infos = dataValue.substring(0, endLine).split(/[ ,;\t]+/);
            prepareNtuplesDatatable$1(currentEntry, spectrum, infos[0]);
            spectrum.datatable = infos[0];

            if (infos[1] && infos[1].indexOf('PEAKS') > -1) {
              canonicDataLabel = 'PEAKTABLE';
            } else if (infos[1] && (infos[1].indexOf('XYDATA') || infos[0].indexOf('++') > 0)) {
              canonicDataLabel = 'XYDATA';

              if (spectrum.nbPoints) {
                spectrum.deltaX = (spectrum.lastX - spectrum.firstX) / (spectrum.nbPoints - 1);
              }
            }
          }
        }

        if (canonicDataLabel === 'XYDATA') {
          if (options.wantXY) {
            prepareSpectrum$1(spectrum); // well apparently we should still consider it is a PEAK TABLE if there are no '++' after

            if (dataValue.match(/.*\+\+.*/)) {
              // ex: (X++(Y..Y))
              if (spectrum.nbPoints) {
                spectrum.deltaX = (spectrum.lastX - spectrum.firstX) / (spectrum.nbPoints - 1);
              }

              fastParseXYData$1(spectrum, dataValue);
            } else {
              parsePeakTable$1(spectrum, dataValue, result);
            }

            currentEntry.spectra.push(spectrum);
            spectrum = {};
          }

          continue;
        } else if (canonicDataLabel === 'PEAKTABLE') {
          if (options.wantXY) {
            prepareSpectrum$1(spectrum);
            parsePeakTable$1(spectrum, dataValue, result);
            currentEntry.spectra.push(spectrum);
            spectrum = {};
          }

          continue;
        }

        if (canonicDataLabel === 'PEAKASSIGNMENTS') {
          if (options.wantXY) {
            if (dataValue.match(/.*(XYA).*/)) {
              // ex: (XYA)
              parseXYA$1(spectrum, dataValue);
            }

            currentEntry.spectra.push(spectrum);
            spectrum = {};
          }

          continue;
        }

        if (canonicDataLabel === 'TITLE') {
          let parentEntry = currentEntry;

          if (!parentEntry.children) {
            parentEntry.children = [];
          }

          currentEntry = {
            spectra: [],
            ntuples: {},
            info: {},
            meta: {},
            tmp: {} // tmp information we need to keep for postprocessing

          };
          parentEntry.children.push(currentEntry);
          parentsStack.push(parentEntry);
          entriesFlat.push(currentEntry);
          currentEntry.title = dataValue;
        } else if (canonicDataLabel === 'DATATYPE') {
          currentEntry.dataType = dataValue;

          if (dataValue.match(/(^nd|\snd\s)/i)) {
            currentEntry.twoD = true;
          }
        } else if (canonicDataLabel === 'NTUPLES') {
          if (dataValue.match(/(^nd|\snd\s)/i)) {
            currentEntry.twoD = true;
          }
        } else if (canonicDataLabel === 'DATACLASS') {
          currentEntry.dataClass = dataValue;
        } else if (canonicDataLabel === 'XUNITS') {
          spectrum.xUnits = dataValue;
        } else if (canonicDataLabel === 'YUNITS') {
          spectrum.yUnits = dataValue;
        } else if (canonicDataLabel === 'FIRSTX') {
          spectrum.firstX = Number(dataValue);
        } else if (canonicDataLabel === 'LASTX') {
          spectrum.lastX = Number(dataValue);
        } else if (canonicDataLabel === 'FIRSTY') {
          spectrum.firstY = Number(dataValue);
        } else if (canonicDataLabel === 'LASTY') {
          spectrum.lastY = Number(dataValue);
        } else if (canonicDataLabel === 'NPOINTS') {
          spectrum.nbPoints = Number(dataValue);
        } else if (canonicDataLabel === 'XFACTOR') {
          spectrum.xFactor = Number(dataValue);
        } else if (canonicDataLabel === 'YFACTOR') {
          spectrum.yFactor = Number(dataValue);
        } else if (canonicDataLabel === 'MAXX') {
          spectrum.maxX = Number(dataValue);
        } else if (canonicDataLabel === 'MINX') {
          spectrum.minX = Number(dataValue);
        } else if (canonicDataLabel === 'MAXY') {
          spectrum.maxY = Number(dataValue);
        } else if (canonicDataLabel === 'MINY') {
          spectrum.minY = Number(dataValue);
        } else if (canonicDataLabel === 'DELTAX') {
          spectrum.deltaX = Number(dataValue);
        } else if (canonicDataLabel === '.OBSERVEFREQUENCY' || canonicDataLabel === '$SFO1') {
          if (!spectrum.observeFrequency) {
            spectrum.observeFrequency = Number(dataValue);
          }
        } else if (canonicDataLabel === '.OBSERVENUCLEUS') {
          if (!spectrum.xType) {
            currentEntry.xType = dataValue.replace(/[^a-zA-Z0-9]/g, '');
          }
        } else if (canonicDataLabel === '$OFFSET') {
          // OFFSET for Bruker spectra
          currentEntry.shiftOffsetNum = 0;

          if (!spectrum.shiftOffsetVal) {
            spectrum.shiftOffsetVal = Number(dataValue);
          }
        } else if (canonicDataLabel === '$REFERENCEPOINT') ; else if (canonicDataLabel === 'VARNAME') {
          currentEntry.ntuples.varname = dataValue.split(ntuplesSeparatorRegExp$1);
        } else if (canonicDataLabel === 'SYMBOL') {
          currentEntry.ntuples.symbol = dataValue.split(ntuplesSeparatorRegExp$1);
        } else if (canonicDataLabel === 'VARTYPE') {
          currentEntry.ntuples.vartype = dataValue.split(ntuplesSeparatorRegExp$1);
        } else if (canonicDataLabel === 'VARFORM') {
          currentEntry.ntuples.varform = dataValue.split(ntuplesSeparatorRegExp$1);
        } else if (canonicDataLabel === 'VARDIM') {
          currentEntry.ntuples.vardim = convertToFloatArray$2(dataValue.split(ntuplesSeparatorRegExp$1));
        } else if (canonicDataLabel === 'UNITS') {
          currentEntry.ntuples.units = dataValue.split(ntuplesSeparatorRegExp$1);
        } else if (canonicDataLabel === 'FACTOR') {
          currentEntry.ntuples.factor = convertToFloatArray$2(dataValue.split(ntuplesSeparatorRegExp$1));
        } else if (canonicDataLabel === 'FIRST') {
          currentEntry.ntuples.first = convertToFloatArray$2(dataValue.split(ntuplesSeparatorRegExp$1));
        } else if (canonicDataLabel === 'LAST') {
          currentEntry.ntuples.last = convertToFloatArray$2(dataValue.split(ntuplesSeparatorRegExp$1));
        } else if (canonicDataLabel === 'MIN') {
          currentEntry.ntuples.min = convertToFloatArray$2(dataValue.split(ntuplesSeparatorRegExp$1));
        } else if (canonicDataLabel === 'MAX') {
          currentEntry.ntuples.max = convertToFloatArray$2(dataValue.split(ntuplesSeparatorRegExp$1));
        } else if (canonicDataLabel === '.NUCLEUS') {
          if (currentEntry.ntuples) {
            currentEntry.ntuples.nucleus = dataValue.split(ntuplesSeparatorRegExp$1);
          }
        } else if (canonicDataLabel === 'PAGE') {
          spectrum.page = dataValue.trim();
          spectrum.pageValue = Number(dataValue.replace(/^.*=/, ''));
          spectrum.pageSymbol = spectrum.page.replace(/[=].*/, '');
        } else if (canonicDataLabel === 'RETENTIONTIME') {
          spectrum.pageValue = Number(dataValue);
        } else if (isMSField$1(canonicDataLabel)) {
          spectrum[convertMSFieldToLabel$1(canonicDataLabel)] = dataValue;
        } else if (canonicDataLabel === 'SAMPLEDESCRIPTION') {
          spectrum.sampleDescription = dataValue;
        } else if (canonicDataLabel.startsWith('$NUC')) {
          if (!currentEntry.tmp[canonicDataLabel] && !dataValue.includes('off')) {
            currentEntry.tmp[canonicDataLabel] = dataValue.replace(/[<>]/g, '');
          }
        } else if (canonicDataLabel === 'END') {
          currentEntry = parentsStack.pop();
        }

        if (currentEntry && currentEntry.info && currentEntry.meta && canonicDataLabel.match(options.keepRecordsRegExp)) {
          let value = dataValue.trim();
          let target, label;

          if (dataLabel.startsWith('$')) {
            label = options.canonicMetadataLabels ? canonicDataLabel.substring(1) : dataLabel.substring(1);
            target = currentEntry.meta;
          } else {
            label = options.canonicDataLabels ? canonicDataLabel : dataLabel;
            target = currentEntry.info;
          }

          if (options.dynamicTyping) {
            value = parseString(value);
          }

          if (target[label]) {
            if (!Array.isArray(target[label])) {
              target[label] = [target[label]];
            }

            target[label].push(value);
          } else {
            target[label] = value;
          }
        }
      }

      profiling$1(result, 'Finished parsing', options);
      postProcessing$1(entriesFlat, result, options);
      profiling$1(result, 'Total time', options);
      /*
      if (result.children && result.children.length>0) {
        result = { ...result, ...result.children[0] };
      }
      */

      result.entries = tmpResult.children;
      result.flatten = entriesFlat;
      return result;
    }

    /**
     *
     * @param {File} file
     * @param {object} [options={}]
     * @param {RegExp} [options.keepRecordsRegExp={}]
     * @returns
     */

    async function parseData(file, options) {
      if (!file) return {
        info: {},
        meta: {}
      };
      const data = await file.text();
      let {
        keepRecordsRegExp = /.*/
      } = options;
      let result = convert$1(data, {
        keepRecordsRegExp
      });
      const firstSpectrum = result.flatten.length === 0 ? {} : result.flatten[0];
      return firstSpectrum;
    }

    async function setFIDSpectrumData(file, spectra) {
      let td = parseInt(spectra.meta.TD[0], 10);
      let ioBuffer = new IOBuffer(await file.arrayBuffer());

      if (td * spectra.meta.nbSubSpectra * 4 > ioBuffer.length) {
        throw new RangeError(`td: ${td} and nbSubSpectra: ${nbSubSpectra} exceed the buffer size`);
      }

      let SW_H = Number(spectra.meta.SW_h[0]);
      let SF = Number(spectra.meta.SFO1[0]);
      spectra.meta.DATATYPE = 'NMR FID';
      let DW = 1 / (2 * SW_H);
      let AQ = td * DW;
      let endian = parseInt(spectra.meta.BYTORDA, 10);
      endian = endian ? 0 : 1;

      if (endian) {
        ioBuffer.setLittleEndian();
      } else {
        ioBuffer.setBigEndian();
      }

      let nbSubSpectra = spectra.meta.nbSubSpectra ? spectra.meta.nbSubSpectra / 2 : 1;
      spectra.spectra = new Array(nbSubSpectra);
      const stopReading = td / 2;

      for (let j = 0; j < nbSubSpectra; j++) {
        let toSave = {
          dataType: 'NMR FID',
          dataTable: '(X++(R..R))',
          nbPoints: td,
          firstX: 0,
          lastX: AQ,
          nucleus: spectra.meta.NUC1,
          xUnit: 'Sec',
          yUnit: 'Arbitrary',
          data: {
            x: createStepArray({
              length: td,
              step: DW
            }),
            re: new Float64Array(td),
            im: new Float64Array(td)
          },
          isXYdata: true,
          observeFrequency: SF,
          title: spectra.meta.TITLE,
          deltaX: DW
        };
        spectra.spectra[j] = toSave;

        for (let i = 0; i < stopReading; i++) {
          spectra.spectra[j].data.re[i] = ioBuffer.readInt32();
          spectra.spectra[j].data.im[i] = ioBuffer.readInt32();
        }
      }
    }

    async function setXYSpectrumData(file, spectra) {
      let ioBufferReal = file.re ? new IOBuffer(await file.re.arrayBuffer()) : null;
      let ioBufferImaginary = file.im ? new IOBuffer(await file.im.arrayBuffer()) : null;
      let td = getDirectParameter(spectra.meta.SI);
      let swP = getDirectParameter(spectra.meta.SW_p);
      let sf = getDirectParameter(spectra.meta.SF);
      let bf = sf;
      let offset = getDirectParameter(spectra.shiftOffsetVal || spectra.meta.OFFSET);
      spectra.meta.observeFrequency = sf;
      spectra.meta.brukerReference = bf;
      spectra.meta.DATATYPE = 'NMR Spectrum';
      let endian = parseInt(spectra.meta.BYTORDP, 10);
      endian = endian ? 0 : 1;
      let nbSubSpectra = spectra.meta.nbSubSpectra ? spectra.meta.nbSubSpectra : 1;

      if (endian) {
        if (file.re) ioBufferReal.setLittleEndian();
        if (file.im) ioBufferImaginary.setLittleEndian();
      } else {
        if (file.re) ioBufferReal.setBigEndian();
        if (file.im) ioBufferImaginary.setBigEndian();
      }

      for (let i = 0; i < nbSubSpectra; i++) {
        let toSave = {
          dataType: 'NMR Spectrum',
          nbPoints: td,
          firstX: offset,
          lastX: offset - swP / sf,
          xUnit: 'PPM',
          yUnit: 'Arbitrary',
          isXYdata: true,
          observeFrequency: sf,
          title: spectra.meta.TITLE,
          deltaX: -(swP / sf) / (td - 1)
        };
        let deltaX = toSave.deltaX;
        let x = new Float64Array(td);
        let re = new Float64Array(td);
        let im = file.im ? new Float64Array(td) : null;

        if (im) {
          for (let k = 0; k < td; ++k) {
            x[k] = offset + k * deltaX;
            re[k] = ioBufferReal.readInt32();
            im[k] = ioBufferImaginary.readInt32();
          }
        } else {
          for (let k = 0; k < td; ++k) {
            x[k] = offset + k * deltaX;
            re[k] = ioBufferReal.readInt32();
          }
        }

        toSave.data = im ? {
          x,
          re,
          im
        } : {
          x,
          re
        };
        spectra.spectra.push(toSave);
      }
    }

    function getDirectParameter(meta) {
      return Number(isAnyArray(meta) ? meta[0] : meta);
    }

    async function convert1D(files, options) {
      const result = await parseData(files.procs, options);
      const acqus = await parseData(files.acqus, options);
      joinInfoMeta(result, acqus);

      if (files['1r'] || files['1i']) {
        await setXYSpectrumData({
          re: files['1r'],
          im: files['1i']
        }, result);
      } else if (files.fid) {
        await setFIDSpectrumData(files.fid, result);
      }

      return result;
    }

    function mergeMetadata(main, complement) {
      for (let key in complement.meta) {
        if (main.meta[key]) {
          if (!Array.isArray(main.meta[key])) {
            main.meta[key] = [main.meta[key]];
          }

          main.meta[key].push(complement.meta[key]);
        } else if (main.meta[key] === undefined) {
          main.meta[key] = [complement.meta[key]];
        }
      }

      return main;
    }

    async function convert2D(files, options) {
      const result = mergeMetadata(await parseData(files.procs, options), await parseData(files.proc2s, options));
      const acqus = mergeMetadata(await parseData(files.acqus, options), await parseData(files.acqu2s, options));
      joinInfoMeta(result, acqus);
      result.meta.nbSubSpectra = files['2rr'] ? parseInt(result.meta.SI[1], 10) : parseInt(result.meta.TD[1], 10);

      if (!result.meta.SW_p) {
        // eslint-disable-next-line camelcase
        result.meta.SW_p = result.meta.SW_h;
      }

      if (!result.meta.SF) {
        result.meta.SF = result.meta.SFO1;
      }

      let firstY, lastY, xOffset, yOffset;

      if (files['2rr']) {
        let sf = Number(result.meta.SF[1]);
        let swP = Number(result.meta.SW_p[1] || result.meta.SW[1]);
        yOffset = Number(result.meta.OFFSET[1]);
        xOffset = Number(result.meta.OFFSET[0]);
        firstY = yOffset;
        lastY = yOffset - swP / sf;
        result.meta.firstY = firstY;
        result.meta.lastY = lastY;
        await setXYSpectrumData({
          re: files['2rr']
        }, result);
      } else if (files.ser) {
        firstY = 0;
        lastY = result.meta.nbSubSpectra;
        let xWindowSize = Number(result.meta.SW[0]);
        let yWindowSize = Number(result.meta.SW[1]);
        let xTransmitterFrequency = Number(result.meta.SFO1[0]);
        let yTransmitterFrequency = Number(result.meta.SFO1[1]);
        let xTransmitterFrequencyOffset = Number(result.meta.O1[0]);
        let yTransmitterFrequencyOffset = Number(result.meta.O1[1]);
        xOffset = xTransmitterFrequencyOffset / xTransmitterFrequency + xWindowSize / 2;
        yOffset = yTransmitterFrequencyOffset / yTransmitterFrequency + yWindowSize / 2;
        await setFIDSpectrumData(files.ser, result);
      }

      let pageValue = firstY;
      let nbSubSpectra = result.meta.nbSubSpectra / (files['2rr'] ? 1 : 2);
      let deltaY = (lastY - firstY) / (nbSubSpectra - 1);

      for (let i = 0; i < nbSubSpectra; i++) {
        pageValue += deltaY;
        result.spectra[i].pageValue = pageValue;
      }

      let {
        NUC1: nuc1,
        AXNUC: axnuc,
        SF: sf
      } = result.meta;
      const nucleus = axnuc ? axnuc : nuc1 ? nuc1 : [];
      result.info['2D_Y_NUCLEUS'] = nucleus[1];
      result.info['2D_X_NUCLEUS'] = nucleus[0];
      result.info['2D_Y_FRECUENCY'] = sf[1];
      result.info['2D_X_FRECUENCY'] = sf[0];
      result.info['2D_Y_OFFSET'] = yOffset;
      result.info['2D_X_OFFSET'] = xOffset;
      result.info.twoD = result.twoD = true;
      return result;
    }

    /**
     * Those functions should disappear if add2D becomes accessible in jcampconvert
     * @param spectra
     * @returns {{z: Array, minX: *, maxX: *, minY: *, maxY: *, minZ: *, maxZ: *, noise: number}}
     */

    function convertTo3DZ$1(spectra) {
      let ySize = spectra.length;
      let xSize = spectra[0].data.re.length;
      let z = new Array(ySize);

      for (let i = 0; i < ySize; i++) {
        z[i] = new Float64Array(spectra[i].data.re);
      }

      const firstX = spectra[0].data.x[0];
      const lastX = spectra[0].data.x[xSize - 1];
      const firstY = spectra[0].pageValue;
      const lastY = spectra[ySize - 1].pageValue; // Because the min / max value are the only information about the matrix if we invert
      // min and max we need to invert the array

      if (firstX > lastX) {
        for (let spectrum of z) {
          spectrum.reverse();
        }
      }

      if (firstY > lastY) {
        z.reverse();
      }

      let minMaxX = firstX < lastX ? {
        minX: firstX,
        maxX: lastX
      } : {
        minX: lastX,
        maxX: firstX
      };
      let minMaxY = firstY < lastY ? {
        minY: firstY,
        maxY: lastY
      } : {
        minY: lastY,
        maxY: firstY
      };
      const {
        min: minZ,
        max: maxZ
      } = matrixMinMaxZ(z);
      return {
        z,
        minZ,
        maxZ,
        ...minMaxX,
        ...minMaxY
      };
    }

    /**
     * Extract information and data from bruker files.
     * @param {File[]} brukerFiles - Needed bruker files to parse raw data.
     * @param {object} [options = {}] - options.
     * @param {boolean} [options.xy] - if true, spectra data is a object with x and y
     * @param {boolean} [options.keepSpectra=false] - for 2D should we keep the spectra or just the matrix ?
     * @param {RegExp} [options.keepRecordsRegExp='\/.*\/'] - regular expresion to parse the metadata of the spectrum.
     * @returns
     */

    async function convertOneExperiment(brukerFiles) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      let result;

      if (brukerFiles.ser || brukerFiles['2rr']) {
        result = await convert2D(brukerFiles, options);
      } else if (brukerFiles['1r'] || brukerFiles['1i'] || brukerFiles.fid) {
        result = await convert1D(brukerFiles, options);
      } else {
        throw new RangeError('The current files are invalid');
      }

      result.source = {};

      for (let key in brukerFiles) {
        if ((typeof brukerFiles[key]).match(/number|string|boolean/)) {
          result.source[key] = brukerFiles[key];
        } // todo we could as well keep the FileList at this level if
        // we want to keep the original data

      } //normalizing info


      result.meta.DATE = Number(result.meta.DATE);

      if (result.meta.GRPDLY) {
        result.meta.GRPDLY = Number(result.meta.GRPDLY);
        result.meta.DSPFVS = Number(result.meta.DSPFVS);
        result.meta.DECIM = Number(result.meta.DECIM);
      }

      for (let key in result.meta) {
        if (!Array.isArray(result.meta[key])) {
          continue;
        }

        if (result.meta[key].length === 1) {
          result.meta[key] = result.meta[key][0];
        } else if (typeof result.meta[key][0] === 'string' && result.meta[key][0].indexOf('(0..') > -1) {
          result.meta[key] = result.meta[key][0];
        }
      }

      if (result.twoD) {
        result.minMax = convertTo3DZ$1(result.spectra);

        if (!options.keepSpectra) {
          delete result.spectra;
        }
      }

      return result;
    }

    /**
     *
     * @param {*} fileList
     * @param {object} [options={}]
     * @param {object} [options.filter]
     * @param {object} [options.converter]
     * @returns
     */

    async function convertFileList(fileList) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        filter,
        converter
      } = options;
      const results = [];
      const experiments = groupByExperiments(fileList, filter);

      for (let experiment of experiments) {
        const result = await convertOneExperiment(experiment, converter);

        if (result) {
          results.push(result);
        }
      }

      return results;
    }

    var jszip_min = {exports: {}};

    /*!

    JSZip v3.10.0 - A JavaScript class for generating and reading zip files
    <http://stuartk.com/jszip>

    (c) 2009-2016 Stuart Knightley <stuart [at] stuartk.com>
    Dual licenced under the MIT license or GPLv3. See https://raw.github.com/Stuk/jszip/main/LICENSE.markdown.

    JSZip uses the library pako released under the MIT license :
    https://github.com/nodeca/pako/blob/main/LICENSE
    */

    (function (module, exports) {
      !function (e) {
        module.exports = e();
      }(function () {
        return function s(a, o, h) {
          function u(r, e) {
            if (!o[r]) {
              if (!a[r]) {
                var t = "function" == typeof commonjsRequire && commonjsRequire;
                if (!e && t) return t(r, !0);
                if (l) return l(r, !0);
                var n = new Error("Cannot find module '" + r + "'");
                throw n.code = "MODULE_NOT_FOUND", n;
              }

              var i = o[r] = {
                exports: {}
              };
              a[r][0].call(i.exports, function (e) {
                var t = a[r][1][e];
                return u(t || e);
              }, i, i.exports, s, a, o, h);
            }

            return o[r].exports;
          }

          for (var l = "function" == typeof commonjsRequire && commonjsRequire, e = 0; e < h.length; e++) u(h[e]);

          return u;
        }({
          1: [function (e, t, r) {

            var d = e("./utils"),
                c = e("./support"),
                p = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";
            r.encode = function (e) {
              for (var t, r, n, i, s, a, o, h = [], u = 0, l = e.length, f = l, c = "string" !== d.getTypeOf(e); u < e.length;) f = l - u, n = c ? (t = e[u++], r = u < l ? e[u++] : 0, u < l ? e[u++] : 0) : (t = e.charCodeAt(u++), r = u < l ? e.charCodeAt(u++) : 0, u < l ? e.charCodeAt(u++) : 0), i = t >> 2, s = (3 & t) << 4 | r >> 4, a = 1 < f ? (15 & r) << 2 | n >> 6 : 64, o = 2 < f ? 63 & n : 64, h.push(p.charAt(i) + p.charAt(s) + p.charAt(a) + p.charAt(o));

              return h.join("");
            }, r.decode = function (e) {
              var t,
                  r,
                  n,
                  i,
                  s,
                  a,
                  o = 0,
                  h = 0,
                  u = "data:";
              if (e.substr(0, u.length) === u) throw new Error("Invalid base64 input, it looks like a data url.");
              var l,
                  f = 3 * (e = e.replace(/[^A-Za-z0-9\+\/\=]/g, "")).length / 4;
              if (e.charAt(e.length - 1) === p.charAt(64) && f--, e.charAt(e.length - 2) === p.charAt(64) && f--, f % 1 != 0) throw new Error("Invalid base64 input, bad content length.");

              for (l = c.uint8array ? new Uint8Array(0 | f) : new Array(0 | f); o < e.length;) t = p.indexOf(e.charAt(o++)) << 2 | (i = p.indexOf(e.charAt(o++))) >> 4, r = (15 & i) << 4 | (s = p.indexOf(e.charAt(o++))) >> 2, n = (3 & s) << 6 | (a = p.indexOf(e.charAt(o++))), l[h++] = t, 64 !== s && (l[h++] = r), 64 !== a && (l[h++] = n);

              return l;
            };
          }, {
            "./support": 30,
            "./utils": 32
          }],
          2: [function (e, t, r) {

            var n = e("./external"),
                i = e("./stream/DataWorker"),
                s = e("./stream/Crc32Probe"),
                a = e("./stream/DataLengthProbe");

            function o(e, t, r, n, i) {
              this.compressedSize = e, this.uncompressedSize = t, this.crc32 = r, this.compression = n, this.compressedContent = i;
            }

            o.prototype = {
              getContentWorker: function () {
                var e = new i(n.Promise.resolve(this.compressedContent)).pipe(this.compression.uncompressWorker()).pipe(new a("data_length")),
                    t = this;
                return e.on("end", function () {
                  if (this.streamInfo.data_length !== t.uncompressedSize) throw new Error("Bug : uncompressed data size mismatch");
                }), e;
              },
              getCompressedWorker: function () {
                return new i(n.Promise.resolve(this.compressedContent)).withStreamInfo("compressedSize", this.compressedSize).withStreamInfo("uncompressedSize", this.uncompressedSize).withStreamInfo("crc32", this.crc32).withStreamInfo("compression", this.compression);
              }
            }, o.createWorkerFrom = function (e, t, r) {
              return e.pipe(new s()).pipe(new a("uncompressedSize")).pipe(t.compressWorker(r)).pipe(new a("compressedSize")).withStreamInfo("compression", t);
            }, t.exports = o;
          }, {
            "./external": 6,
            "./stream/Crc32Probe": 25,
            "./stream/DataLengthProbe": 26,
            "./stream/DataWorker": 27
          }],
          3: [function (e, t, r) {

            var n = e("./stream/GenericWorker");
            r.STORE = {
              magic: "\0\0",
              compressWorker: function (e) {
                return new n("STORE compression");
              },
              uncompressWorker: function () {
                return new n("STORE decompression");
              }
            }, r.DEFLATE = e("./flate");
          }, {
            "./flate": 7,
            "./stream/GenericWorker": 28
          }],
          4: [function (e, t, r) {

            var n = e("./utils");

            var o = function () {
              for (var e, t = [], r = 0; r < 256; r++) {
                e = r;

                for (var n = 0; n < 8; n++) e = 1 & e ? 3988292384 ^ e >>> 1 : e >>> 1;

                t[r] = e;
              }

              return t;
            }();

            t.exports = function (e, t) {
              return void 0 !== e && e.length ? "string" !== n.getTypeOf(e) ? function (e, t, r, n) {
                var i = o,
                    s = n + r;
                e ^= -1;

                for (var a = n; a < s; a++) e = e >>> 8 ^ i[255 & (e ^ t[a])];

                return -1 ^ e;
              }(0 | t, e, e.length, 0) : function (e, t, r, n) {
                var i = o,
                    s = n + r;
                e ^= -1;

                for (var a = n; a < s; a++) e = e >>> 8 ^ i[255 & (e ^ t.charCodeAt(a))];

                return -1 ^ e;
              }(0 | t, e, e.length, 0) : 0;
            };
          }, {
            "./utils": 32
          }],
          5: [function (e, t, r) {

            r.base64 = !1, r.binary = !1, r.dir = !1, r.createFolders = !0, r.date = null, r.compression = null, r.compressionOptions = null, r.comment = null, r.unixPermissions = null, r.dosPermissions = null;
          }, {}],
          6: [function (e, t, r) {

            var n = null;
            n = "undefined" != typeof Promise ? Promise : e("lie"), t.exports = {
              Promise: n
            };
          }, {
            lie: 37
          }],
          7: [function (e, t, r) {

            var n = "undefined" != typeof Uint8Array && "undefined" != typeof Uint16Array && "undefined" != typeof Uint32Array,
                i = e("pako"),
                s = e("./utils"),
                a = e("./stream/GenericWorker"),
                o = n ? "uint8array" : "array";

            function h(e, t) {
              a.call(this, "FlateWorker/" + e), this._pako = null, this._pakoAction = e, this._pakoOptions = t, this.meta = {};
            }

            r.magic = "\b\0", s.inherits(h, a), h.prototype.processChunk = function (e) {
              this.meta = e.meta, null === this._pako && this._createPako(), this._pako.push(s.transformTo(o, e.data), !1);
            }, h.prototype.flush = function () {
              a.prototype.flush.call(this), null === this._pako && this._createPako(), this._pako.push([], !0);
            }, h.prototype.cleanUp = function () {
              a.prototype.cleanUp.call(this), this._pako = null;
            }, h.prototype._createPako = function () {
              this._pako = new i[this._pakoAction]({
                raw: !0,
                level: this._pakoOptions.level || -1
              });
              var t = this;

              this._pako.onData = function (e) {
                t.push({
                  data: e,
                  meta: t.meta
                });
              };
            }, r.compressWorker = function (e) {
              return new h("Deflate", e);
            }, r.uncompressWorker = function () {
              return new h("Inflate", {});
            };
          }, {
            "./stream/GenericWorker": 28,
            "./utils": 32,
            pako: 38
          }],
          8: [function (e, t, r) {

            function A(e, t) {
              var r,
                  n = "";

              for (r = 0; r < t; r++) n += String.fromCharCode(255 & e), e >>>= 8;

              return n;
            }

            function n(e, t, r, n, i, s) {
              var a,
                  o,
                  h = e.file,
                  u = e.compression,
                  l = s !== O.utf8encode,
                  f = I.transformTo("string", s(h.name)),
                  c = I.transformTo("string", O.utf8encode(h.name)),
                  d = h.comment,
                  p = I.transformTo("string", s(d)),
                  m = I.transformTo("string", O.utf8encode(d)),
                  _ = c.length !== h.name.length,
                  g = m.length !== d.length,
                  b = "",
                  v = "",
                  y = "",
                  w = h.dir,
                  k = h.date,
                  x = {
                crc32: 0,
                compressedSize: 0,
                uncompressedSize: 0
              };

              t && !r || (x.crc32 = e.crc32, x.compressedSize = e.compressedSize, x.uncompressedSize = e.uncompressedSize);
              var S = 0;
              t && (S |= 8), l || !_ && !g || (S |= 2048);
              var z = 0,
                  C = 0;
              w && (z |= 16), "UNIX" === i ? (C = 798, z |= function (e, t) {
                var r = e;
                return e || (r = t ? 16893 : 33204), (65535 & r) << 16;
              }(h.unixPermissions, w)) : (C = 20, z |= function (e) {
                return 63 & (e || 0);
              }(h.dosPermissions)), a = k.getUTCHours(), a <<= 6, a |= k.getUTCMinutes(), a <<= 5, a |= k.getUTCSeconds() / 2, o = k.getUTCFullYear() - 1980, o <<= 4, o |= k.getUTCMonth() + 1, o <<= 5, o |= k.getUTCDate(), _ && (v = A(1, 1) + A(B(f), 4) + c, b += "up" + A(v.length, 2) + v), g && (y = A(1, 1) + A(B(p), 4) + m, b += "uc" + A(y.length, 2) + y);
              var E = "";
              return E += "\n\0", E += A(S, 2), E += u.magic, E += A(a, 2), E += A(o, 2), E += A(x.crc32, 4), E += A(x.compressedSize, 4), E += A(x.uncompressedSize, 4), E += A(f.length, 2), E += A(b.length, 2), {
                fileRecord: R.LOCAL_FILE_HEADER + E + f + b,
                dirRecord: R.CENTRAL_FILE_HEADER + A(C, 2) + E + A(p.length, 2) + "\0\0\0\0" + A(z, 4) + A(n, 4) + f + b + p
              };
            }

            var I = e("../utils"),
                i = e("../stream/GenericWorker"),
                O = e("../utf8"),
                B = e("../crc32"),
                R = e("../signature");

            function s(e, t, r, n) {
              i.call(this, "ZipFileWorker"), this.bytesWritten = 0, this.zipComment = t, this.zipPlatform = r, this.encodeFileName = n, this.streamFiles = e, this.accumulate = !1, this.contentBuffer = [], this.dirRecords = [], this.currentSourceOffset = 0, this.entriesCount = 0, this.currentFile = null, this._sources = [];
            }

            I.inherits(s, i), s.prototype.push = function (e) {
              var t = e.meta.percent || 0,
                  r = this.entriesCount,
                  n = this._sources.length;
              this.accumulate ? this.contentBuffer.push(e) : (this.bytesWritten += e.data.length, i.prototype.push.call(this, {
                data: e.data,
                meta: {
                  currentFile: this.currentFile,
                  percent: r ? (t + 100 * (r - n - 1)) / r : 100
                }
              }));
            }, s.prototype.openedSource = function (e) {
              this.currentSourceOffset = this.bytesWritten, this.currentFile = e.file.name;
              var t = this.streamFiles && !e.file.dir;

              if (t) {
                var r = n(e, t, !1, this.currentSourceOffset, this.zipPlatform, this.encodeFileName);
                this.push({
                  data: r.fileRecord,
                  meta: {
                    percent: 0
                  }
                });
              } else this.accumulate = !0;
            }, s.prototype.closedSource = function (e) {
              this.accumulate = !1;
              var t = this.streamFiles && !e.file.dir,
                  r = n(e, t, !0, this.currentSourceOffset, this.zipPlatform, this.encodeFileName);
              if (this.dirRecords.push(r.dirRecord), t) this.push({
                data: function (e) {
                  return R.DATA_DESCRIPTOR + A(e.crc32, 4) + A(e.compressedSize, 4) + A(e.uncompressedSize, 4);
                }(e),
                meta: {
                  percent: 100
                }
              });else for (this.push({
                data: r.fileRecord,
                meta: {
                  percent: 0
                }
              }); this.contentBuffer.length;) this.push(this.contentBuffer.shift());
              this.currentFile = null;
            }, s.prototype.flush = function () {
              for (var e = this.bytesWritten, t = 0; t < this.dirRecords.length; t++) this.push({
                data: this.dirRecords[t],
                meta: {
                  percent: 100
                }
              });

              var r = this.bytesWritten - e,
                  n = function (e, t, r, n, i) {
                var s = I.transformTo("string", i(n));
                return R.CENTRAL_DIRECTORY_END + "\0\0\0\0" + A(e, 2) + A(e, 2) + A(t, 4) + A(r, 4) + A(s.length, 2) + s;
              }(this.dirRecords.length, r, e, this.zipComment, this.encodeFileName);

              this.push({
                data: n,
                meta: {
                  percent: 100
                }
              });
            }, s.prototype.prepareNextSource = function () {
              this.previous = this._sources.shift(), this.openedSource(this.previous.streamInfo), this.isPaused ? this.previous.pause() : this.previous.resume();
            }, s.prototype.registerPrevious = function (e) {
              this._sources.push(e);

              var t = this;
              return e.on("data", function (e) {
                t.processChunk(e);
              }), e.on("end", function () {
                t.closedSource(t.previous.streamInfo), t._sources.length ? t.prepareNextSource() : t.end();
              }), e.on("error", function (e) {
                t.error(e);
              }), this;
            }, s.prototype.resume = function () {
              return !!i.prototype.resume.call(this) && (!this.previous && this._sources.length ? (this.prepareNextSource(), !0) : this.previous || this._sources.length || this.generatedError ? void 0 : (this.end(), !0));
            }, s.prototype.error = function (e) {
              var t = this._sources;
              if (!i.prototype.error.call(this, e)) return !1;

              for (var r = 0; r < t.length; r++) try {
                t[r].error(e);
              } catch (e) {}

              return !0;
            }, s.prototype.lock = function () {
              i.prototype.lock.call(this);

              for (var e = this._sources, t = 0; t < e.length; t++) e[t].lock();
            }, t.exports = s;
          }, {
            "../crc32": 4,
            "../signature": 23,
            "../stream/GenericWorker": 28,
            "../utf8": 31,
            "../utils": 32
          }],
          9: [function (e, t, r) {

            var u = e("../compressions"),
                n = e("./ZipFileWorker");

            r.generateWorker = function (e, a, t) {
              var o = new n(a.streamFiles, t, a.platform, a.encodeFileName),
                  h = 0;

              try {
                e.forEach(function (e, t) {
                  h++;

                  var r = function (e, t) {
                    var r = e || t,
                        n = u[r];
                    if (!n) throw new Error(r + " is not a valid compression method !");
                    return n;
                  }(t.options.compression, a.compression),
                      n = t.options.compressionOptions || a.compressionOptions || {},
                      i = t.dir,
                      s = t.date;

                  t._compressWorker(r, n).withStreamInfo("file", {
                    name: e,
                    dir: i,
                    date: s,
                    comment: t.comment || "",
                    unixPermissions: t.unixPermissions,
                    dosPermissions: t.dosPermissions
                  }).pipe(o);
                }), o.entriesCount = h;
              } catch (e) {
                o.error(e);
              }

              return o;
            };
          }, {
            "../compressions": 3,
            "./ZipFileWorker": 8
          }],
          10: [function (e, t, r) {

            function n() {
              if (!(this instanceof n)) return new n();
              if (arguments.length) throw new Error("The constructor with parameters has been removed in JSZip 3.0, please check the upgrade guide.");
              this.files = Object.create(null), this.comment = null, this.root = "", this.clone = function () {
                var e = new n();

                for (var t in this) "function" != typeof this[t] && (e[t] = this[t]);

                return e;
              };
            }

            (n.prototype = e("./object")).loadAsync = e("./load"), n.support = e("./support"), n.defaults = e("./defaults"), n.version = "3.10.0", n.loadAsync = function (e, t) {
              return new n().loadAsync(e, t);
            }, n.external = e("./external"), t.exports = n;
          }, {
            "./defaults": 5,
            "./external": 6,
            "./load": 11,
            "./object": 15,
            "./support": 30
          }],
          11: [function (e, t, r) {

            var u = e("./utils"),
                i = e("./external"),
                n = e("./utf8"),
                s = e("./zipEntries"),
                a = e("./stream/Crc32Probe"),
                l = e("./nodejsUtils");

            function f(n) {
              return new i.Promise(function (e, t) {
                var r = n.decompressed.getContentWorker().pipe(new a());
                r.on("error", function (e) {
                  t(e);
                }).on("end", function () {
                  r.streamInfo.crc32 !== n.decompressed.crc32 ? t(new Error("Corrupted zip : CRC32 mismatch")) : e();
                }).resume();
              });
            }

            t.exports = function (e, o) {
              var h = this;
              return o = u.extend(o || {}, {
                base64: !1,
                checkCRC32: !1,
                optimizedBinaryString: !1,
                createFolders: !1,
                decodeFileName: n.utf8decode
              }), l.isNode && l.isStream(e) ? i.Promise.reject(new Error("JSZip can't accept a stream when loading a zip file.")) : u.prepareContent("the loaded zip file", e, !0, o.optimizedBinaryString, o.base64).then(function (e) {
                var t = new s(o);
                return t.load(e), t;
              }).then(function (e) {
                var t = [i.Promise.resolve(e)],
                    r = e.files;
                if (o.checkCRC32) for (var n = 0; n < r.length; n++) t.push(f(r[n]));
                return i.Promise.all(t);
              }).then(function (e) {
                for (var t = e.shift(), r = t.files, n = 0; n < r.length; n++) {
                  var i = r[n],
                      s = i.fileNameStr,
                      a = u.resolve(i.fileNameStr);
                  h.file(a, i.decompressed, {
                    binary: !0,
                    optimizedBinaryString: !0,
                    date: i.date,
                    dir: i.dir,
                    comment: i.fileCommentStr.length ? i.fileCommentStr : null,
                    unixPermissions: i.unixPermissions,
                    dosPermissions: i.dosPermissions,
                    createFolders: o.createFolders
                  }), i.dir || (h.file(a).unsafeOriginalName = s);
                }

                return t.zipComment.length && (h.comment = t.zipComment), h;
              });
            };
          }, {
            "./external": 6,
            "./nodejsUtils": 14,
            "./stream/Crc32Probe": 25,
            "./utf8": 31,
            "./utils": 32,
            "./zipEntries": 33
          }],
          12: [function (e, t, r) {

            var n = e("../utils"),
                i = e("../stream/GenericWorker");

            function s(e, t) {
              i.call(this, "Nodejs stream input adapter for " + e), this._upstreamEnded = !1, this._bindStream(t);
            }

            n.inherits(s, i), s.prototype._bindStream = function (e) {
              var t = this;
              (this._stream = e).pause(), e.on("data", function (e) {
                t.push({
                  data: e,
                  meta: {
                    percent: 0
                  }
                });
              }).on("error", function (e) {
                t.isPaused ? this.generatedError = e : t.error(e);
              }).on("end", function () {
                t.isPaused ? t._upstreamEnded = !0 : t.end();
              });
            }, s.prototype.pause = function () {
              return !!i.prototype.pause.call(this) && (this._stream.pause(), !0);
            }, s.prototype.resume = function () {
              return !!i.prototype.resume.call(this) && (this._upstreamEnded ? this.end() : this._stream.resume(), !0);
            }, t.exports = s;
          }, {
            "../stream/GenericWorker": 28,
            "../utils": 32
          }],
          13: [function (e, t, r) {

            var i = e("readable-stream").Readable;

            function n(e, t, r) {
              i.call(this, t), this._helper = e;
              var n = this;
              e.on("data", function (e, t) {
                n.push(e) || n._helper.pause(), r && r(t);
              }).on("error", function (e) {
                n.emit("error", e);
              }).on("end", function () {
                n.push(null);
              });
            }

            e("../utils").inherits(n, i), n.prototype._read = function () {
              this._helper.resume();
            }, t.exports = n;
          }, {
            "../utils": 32,
            "readable-stream": 16
          }],
          14: [function (e, t, r) {

            t.exports = {
              isNode: "undefined" != typeof Buffer,
              newBufferFrom: function (e, t) {
                if (Buffer.from && Buffer.from !== Uint8Array.from) return Buffer.from(e, t);
                if ("number" == typeof e) throw new Error('The "data" argument must not be a number');
                return new Buffer(e, t);
              },
              allocBuffer: function (e) {
                if (Buffer.alloc) return Buffer.alloc(e);
                var t = new Buffer(e);
                return t.fill(0), t;
              },
              isBuffer: function (e) {
                return Buffer.isBuffer(e);
              },
              isStream: function (e) {
                return e && "function" == typeof e.on && "function" == typeof e.pause && "function" == typeof e.resume;
              }
            };
          }, {}],
          15: [function (e, t, r) {

            function s(e, t, r) {
              var n,
                  i = u.getTypeOf(t),
                  s = u.extend(r || {}, f);
              s.date = s.date || new Date(), null !== s.compression && (s.compression = s.compression.toUpperCase()), "string" == typeof s.unixPermissions && (s.unixPermissions = parseInt(s.unixPermissions, 8)), s.unixPermissions && 16384 & s.unixPermissions && (s.dir = !0), s.dosPermissions && 16 & s.dosPermissions && (s.dir = !0), s.dir && (e = g(e)), s.createFolders && (n = _(e)) && b.call(this, n, !0);
              var a = "string" === i && !1 === s.binary && !1 === s.base64;
              r && void 0 !== r.binary || (s.binary = !a), (t instanceof c && 0 === t.uncompressedSize || s.dir || !t || 0 === t.length) && (s.base64 = !1, s.binary = !0, t = "", s.compression = "STORE", i = "string");
              var o = null;
              o = t instanceof c || t instanceof l ? t : p.isNode && p.isStream(t) ? new m(e, t) : u.prepareContent(e, t, s.binary, s.optimizedBinaryString, s.base64);
              var h = new d(e, o, s);
              this.files[e] = h;
            }

            var i = e("./utf8"),
                u = e("./utils"),
                l = e("./stream/GenericWorker"),
                a = e("./stream/StreamHelper"),
                f = e("./defaults"),
                c = e("./compressedObject"),
                d = e("./zipObject"),
                o = e("./generate"),
                p = e("./nodejsUtils"),
                m = e("./nodejs/NodejsStreamInputAdapter"),
                _ = function (e) {
              "/" === e.slice(-1) && (e = e.substring(0, e.length - 1));
              var t = e.lastIndexOf("/");
              return 0 < t ? e.substring(0, t) : "";
            },
                g = function (e) {
              return "/" !== e.slice(-1) && (e += "/"), e;
            },
                b = function (e, t) {
              return t = void 0 !== t ? t : f.createFolders, e = g(e), this.files[e] || s.call(this, e, null, {
                dir: !0,
                createFolders: t
              }), this.files[e];
            };

            function h(e) {
              return "[object RegExp]" === Object.prototype.toString.call(e);
            }

            var n = {
              load: function () {
                throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.");
              },
              forEach: function (e) {
                var t, r, n;

                for (t in this.files) n = this.files[t], (r = t.slice(this.root.length, t.length)) && t.slice(0, this.root.length) === this.root && e(r, n);
              },
              filter: function (r) {
                var n = [];
                return this.forEach(function (e, t) {
                  r(e, t) && n.push(t);
                }), n;
              },
              file: function (e, t, r) {
                if (1 !== arguments.length) return e = this.root + e, s.call(this, e, t, r), this;

                if (h(e)) {
                  var n = e;
                  return this.filter(function (e, t) {
                    return !t.dir && n.test(e);
                  });
                }

                var i = this.files[this.root + e];
                return i && !i.dir ? i : null;
              },
              folder: function (r) {
                if (!r) return this;
                if (h(r)) return this.filter(function (e, t) {
                  return t.dir && r.test(e);
                });
                var e = this.root + r,
                    t = b.call(this, e),
                    n = this.clone();
                return n.root = t.name, n;
              },
              remove: function (r) {
                r = this.root + r;
                var e = this.files[r];
                if (e || ("/" !== r.slice(-1) && (r += "/"), e = this.files[r]), e && !e.dir) delete this.files[r];else for (var t = this.filter(function (e, t) {
                  return t.name.slice(0, r.length) === r;
                }), n = 0; n < t.length; n++) delete this.files[t[n].name];
                return this;
              },
              generate: function (e) {
                throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.");
              },
              generateInternalStream: function (e) {
                var t,
                    r = {};

                try {
                  if ((r = u.extend(e || {}, {
                    streamFiles: !1,
                    compression: "STORE",
                    compressionOptions: null,
                    type: "",
                    platform: "DOS",
                    comment: null,
                    mimeType: "application/zip",
                    encodeFileName: i.utf8encode
                  })).type = r.type.toLowerCase(), r.compression = r.compression.toUpperCase(), "binarystring" === r.type && (r.type = "string"), !r.type) throw new Error("No output type specified.");
                  u.checkSupport(r.type), "darwin" !== r.platform && "freebsd" !== r.platform && "linux" !== r.platform && "sunos" !== r.platform || (r.platform = "UNIX"), "win32" === r.platform && (r.platform = "DOS");
                  var n = r.comment || this.comment || "";
                  t = o.generateWorker(this, r, n);
                } catch (e) {
                  (t = new l("error")).error(e);
                }

                return new a(t, r.type || "string", r.mimeType);
              },
              generateAsync: function (e, t) {
                return this.generateInternalStream(e).accumulate(t);
              },
              generateNodeStream: function (e, t) {
                return (e = e || {}).type || (e.type = "nodebuffer"), this.generateInternalStream(e).toNodejsStream(t);
              }
            };
            t.exports = n;
          }, {
            "./compressedObject": 2,
            "./defaults": 5,
            "./generate": 9,
            "./nodejs/NodejsStreamInputAdapter": 12,
            "./nodejsUtils": 14,
            "./stream/GenericWorker": 28,
            "./stream/StreamHelper": 29,
            "./utf8": 31,
            "./utils": 32,
            "./zipObject": 35
          }],
          16: [function (e, t, r) {
            t.exports = e("stream");
          }, {
            stream: void 0
          }],
          17: [function (e, t, r) {

            var n = e("./DataReader");

            function i(e) {
              n.call(this, e);

              for (var t = 0; t < this.data.length; t++) e[t] = 255 & e[t];
            }

            e("../utils").inherits(i, n), i.prototype.byteAt = function (e) {
              return this.data[this.zero + e];
            }, i.prototype.lastIndexOfSignature = function (e) {
              for (var t = e.charCodeAt(0), r = e.charCodeAt(1), n = e.charCodeAt(2), i = e.charCodeAt(3), s = this.length - 4; 0 <= s; --s) if (this.data[s] === t && this.data[s + 1] === r && this.data[s + 2] === n && this.data[s + 3] === i) return s - this.zero;

              return -1;
            }, i.prototype.readAndCheckSignature = function (e) {
              var t = e.charCodeAt(0),
                  r = e.charCodeAt(1),
                  n = e.charCodeAt(2),
                  i = e.charCodeAt(3),
                  s = this.readData(4);
              return t === s[0] && r === s[1] && n === s[2] && i === s[3];
            }, i.prototype.readData = function (e) {
              if (this.checkOffset(e), 0 === e) return [];
              var t = this.data.slice(this.zero + this.index, this.zero + this.index + e);
              return this.index += e, t;
            }, t.exports = i;
          }, {
            "../utils": 32,
            "./DataReader": 18
          }],
          18: [function (e, t, r) {

            var n = e("../utils");

            function i(e) {
              this.data = e, this.length = e.length, this.index = 0, this.zero = 0;
            }

            i.prototype = {
              checkOffset: function (e) {
                this.checkIndex(this.index + e);
              },
              checkIndex: function (e) {
                if (this.length < this.zero + e || e < 0) throw new Error("End of data reached (data length = " + this.length + ", asked index = " + e + "). Corrupted zip ?");
              },
              setIndex: function (e) {
                this.checkIndex(e), this.index = e;
              },
              skip: function (e) {
                this.setIndex(this.index + e);
              },
              byteAt: function (e) {},
              readInt: function (e) {
                var t,
                    r = 0;

                for (this.checkOffset(e), t = this.index + e - 1; t >= this.index; t--) r = (r << 8) + this.byteAt(t);

                return this.index += e, r;
              },
              readString: function (e) {
                return n.transformTo("string", this.readData(e));
              },
              readData: function (e) {},
              lastIndexOfSignature: function (e) {},
              readAndCheckSignature: function (e) {},
              readDate: function () {
                var e = this.readInt(4);
                return new Date(Date.UTC(1980 + (e >> 25 & 127), (e >> 21 & 15) - 1, e >> 16 & 31, e >> 11 & 31, e >> 5 & 63, (31 & e) << 1));
              }
            }, t.exports = i;
          }, {
            "../utils": 32
          }],
          19: [function (e, t, r) {

            var n = e("./Uint8ArrayReader");

            function i(e) {
              n.call(this, e);
            }

            e("../utils").inherits(i, n), i.prototype.readData = function (e) {
              this.checkOffset(e);
              var t = this.data.slice(this.zero + this.index, this.zero + this.index + e);
              return this.index += e, t;
            }, t.exports = i;
          }, {
            "../utils": 32,
            "./Uint8ArrayReader": 21
          }],
          20: [function (e, t, r) {

            var n = e("./DataReader");

            function i(e) {
              n.call(this, e);
            }

            e("../utils").inherits(i, n), i.prototype.byteAt = function (e) {
              return this.data.charCodeAt(this.zero + e);
            }, i.prototype.lastIndexOfSignature = function (e) {
              return this.data.lastIndexOf(e) - this.zero;
            }, i.prototype.readAndCheckSignature = function (e) {
              return e === this.readData(4);
            }, i.prototype.readData = function (e) {
              this.checkOffset(e);
              var t = this.data.slice(this.zero + this.index, this.zero + this.index + e);
              return this.index += e, t;
            }, t.exports = i;
          }, {
            "../utils": 32,
            "./DataReader": 18
          }],
          21: [function (e, t, r) {

            var n = e("./ArrayReader");

            function i(e) {
              n.call(this, e);
            }

            e("../utils").inherits(i, n), i.prototype.readData = function (e) {
              if (this.checkOffset(e), 0 === e) return new Uint8Array(0);
              var t = this.data.subarray(this.zero + this.index, this.zero + this.index + e);
              return this.index += e, t;
            }, t.exports = i;
          }, {
            "../utils": 32,
            "./ArrayReader": 17
          }],
          22: [function (e, t, r) {

            var n = e("../utils"),
                i = e("../support"),
                s = e("./ArrayReader"),
                a = e("./StringReader"),
                o = e("./NodeBufferReader"),
                h = e("./Uint8ArrayReader");

            t.exports = function (e) {
              var t = n.getTypeOf(e);
              return n.checkSupport(t), "string" !== t || i.uint8array ? "nodebuffer" === t ? new o(e) : i.uint8array ? new h(n.transformTo("uint8array", e)) : new s(n.transformTo("array", e)) : new a(e);
            };
          }, {
            "../support": 30,
            "../utils": 32,
            "./ArrayReader": 17,
            "./NodeBufferReader": 19,
            "./StringReader": 20,
            "./Uint8ArrayReader": 21
          }],
          23: [function (e, t, r) {

            r.LOCAL_FILE_HEADER = "PK", r.CENTRAL_FILE_HEADER = "PK", r.CENTRAL_DIRECTORY_END = "PK", r.ZIP64_CENTRAL_DIRECTORY_LOCATOR = "PK", r.ZIP64_CENTRAL_DIRECTORY_END = "PK", r.DATA_DESCRIPTOR = "PK\b";
          }, {}],
          24: [function (e, t, r) {

            var n = e("./GenericWorker"),
                i = e("../utils");

            function s(e) {
              n.call(this, "ConvertWorker to " + e), this.destType = e;
            }

            i.inherits(s, n), s.prototype.processChunk = function (e) {
              this.push({
                data: i.transformTo(this.destType, e.data),
                meta: e.meta
              });
            }, t.exports = s;
          }, {
            "../utils": 32,
            "./GenericWorker": 28
          }],
          25: [function (e, t, r) {

            var n = e("./GenericWorker"),
                i = e("../crc32");

            function s() {
              n.call(this, "Crc32Probe"), this.withStreamInfo("crc32", 0);
            }

            e("../utils").inherits(s, n), s.prototype.processChunk = function (e) {
              this.streamInfo.crc32 = i(e.data, this.streamInfo.crc32 || 0), this.push(e);
            }, t.exports = s;
          }, {
            "../crc32": 4,
            "../utils": 32,
            "./GenericWorker": 28
          }],
          26: [function (e, t, r) {

            var n = e("../utils"),
                i = e("./GenericWorker");

            function s(e) {
              i.call(this, "DataLengthProbe for " + e), this.propName = e, this.withStreamInfo(e, 0);
            }

            n.inherits(s, i), s.prototype.processChunk = function (e) {
              if (e) {
                var t = this.streamInfo[this.propName] || 0;
                this.streamInfo[this.propName] = t + e.data.length;
              }

              i.prototype.processChunk.call(this, e);
            }, t.exports = s;
          }, {
            "../utils": 32,
            "./GenericWorker": 28
          }],
          27: [function (e, t, r) {

            var n = e("../utils"),
                i = e("./GenericWorker");

            function s(e) {
              i.call(this, "DataWorker");
              var t = this;
              this.dataIsReady = !1, this.index = 0, this.max = 0, this.data = null, this.type = "", this._tickScheduled = !1, e.then(function (e) {
                t.dataIsReady = !0, t.data = e, t.max = e && e.length || 0, t.type = n.getTypeOf(e), t.isPaused || t._tickAndRepeat();
              }, function (e) {
                t.error(e);
              });
            }

            n.inherits(s, i), s.prototype.cleanUp = function () {
              i.prototype.cleanUp.call(this), this.data = null;
            }, s.prototype.resume = function () {
              return !!i.prototype.resume.call(this) && (!this._tickScheduled && this.dataIsReady && (this._tickScheduled = !0, n.delay(this._tickAndRepeat, [], this)), !0);
            }, s.prototype._tickAndRepeat = function () {
              this._tickScheduled = !1, this.isPaused || this.isFinished || (this._tick(), this.isFinished || (n.delay(this._tickAndRepeat, [], this), this._tickScheduled = !0));
            }, s.prototype._tick = function () {
              if (this.isPaused || this.isFinished) return !1;
              var e = null,
                  t = Math.min(this.max, this.index + 16384);
              if (this.index >= this.max) return this.end();

              switch (this.type) {
                case "string":
                  e = this.data.substring(this.index, t);
                  break;

                case "uint8array":
                  e = this.data.subarray(this.index, t);
                  break;

                case "array":
                case "nodebuffer":
                  e = this.data.slice(this.index, t);
              }

              return this.index = t, this.push({
                data: e,
                meta: {
                  percent: this.max ? this.index / this.max * 100 : 0
                }
              });
            }, t.exports = s;
          }, {
            "../utils": 32,
            "./GenericWorker": 28
          }],
          28: [function (e, t, r) {

            function n(e) {
              this.name = e || "default", this.streamInfo = {}, this.generatedError = null, this.extraStreamInfo = {}, this.isPaused = !0, this.isFinished = !1, this.isLocked = !1, this._listeners = {
                data: [],
                end: [],
                error: []
              }, this.previous = null;
            }

            n.prototype = {
              push: function (e) {
                this.emit("data", e);
              },
              end: function () {
                if (this.isFinished) return !1;
                this.flush();

                try {
                  this.emit("end"), this.cleanUp(), this.isFinished = !0;
                } catch (e) {
                  this.emit("error", e);
                }

                return !0;
              },
              error: function (e) {
                return !this.isFinished && (this.isPaused ? this.generatedError = e : (this.isFinished = !0, this.emit("error", e), this.previous && this.previous.error(e), this.cleanUp()), !0);
              },
              on: function (e, t) {
                return this._listeners[e].push(t), this;
              },
              cleanUp: function () {
                this.streamInfo = this.generatedError = this.extraStreamInfo = null, this._listeners = [];
              },
              emit: function (e, t) {
                if (this._listeners[e]) for (var r = 0; r < this._listeners[e].length; r++) this._listeners[e][r].call(this, t);
              },
              pipe: function (e) {
                return e.registerPrevious(this);
              },
              registerPrevious: function (e) {
                if (this.isLocked) throw new Error("The stream '" + this + "' has already been used.");
                this.streamInfo = e.streamInfo, this.mergeStreamInfo(), this.previous = e;
                var t = this;
                return e.on("data", function (e) {
                  t.processChunk(e);
                }), e.on("end", function () {
                  t.end();
                }), e.on("error", function (e) {
                  t.error(e);
                }), this;
              },
              pause: function () {
                return !this.isPaused && !this.isFinished && (this.isPaused = !0, this.previous && this.previous.pause(), !0);
              },
              resume: function () {
                if (!this.isPaused || this.isFinished) return !1;
                var e = this.isPaused = !1;
                return this.generatedError && (this.error(this.generatedError), e = !0), this.previous && this.previous.resume(), !e;
              },
              flush: function () {},
              processChunk: function (e) {
                this.push(e);
              },
              withStreamInfo: function (e, t) {
                return this.extraStreamInfo[e] = t, this.mergeStreamInfo(), this;
              },
              mergeStreamInfo: function () {
                for (var e in this.extraStreamInfo) this.extraStreamInfo.hasOwnProperty(e) && (this.streamInfo[e] = this.extraStreamInfo[e]);
              },
              lock: function () {
                if (this.isLocked) throw new Error("The stream '" + this + "' has already been used.");
                this.isLocked = !0, this.previous && this.previous.lock();
              },
              toString: function () {
                var e = "Worker " + this.name;
                return this.previous ? this.previous + " -> " + e : e;
              }
            }, t.exports = n;
          }, {}],
          29: [function (e, t, r) {

            var h = e("../utils"),
                i = e("./ConvertWorker"),
                s = e("./GenericWorker"),
                u = e("../base64"),
                n = e("../support"),
                a = e("../external"),
                o = null;
            if (n.nodestream) try {
              o = e("../nodejs/NodejsStreamOutputAdapter");
            } catch (e) {}

            function l(e, o) {
              return new a.Promise(function (t, r) {
                var n = [],
                    i = e._internalType,
                    s = e._outputType,
                    a = e._mimeType;
                e.on("data", function (e, t) {
                  n.push(e), o && o(t);
                }).on("error", function (e) {
                  n = [], r(e);
                }).on("end", function () {
                  try {
                    var e = function (e, t, r) {
                      switch (e) {
                        case "blob":
                          return h.newBlob(h.transformTo("arraybuffer", t), r);

                        case "base64":
                          return u.encode(t);

                        default:
                          return h.transformTo(e, t);
                      }
                    }(s, function (e, t) {
                      var r,
                          n = 0,
                          i = null,
                          s = 0;

                      for (r = 0; r < t.length; r++) s += t[r].length;

                      switch (e) {
                        case "string":
                          return t.join("");

                        case "array":
                          return Array.prototype.concat.apply([], t);

                        case "uint8array":
                          for (i = new Uint8Array(s), r = 0; r < t.length; r++) i.set(t[r], n), n += t[r].length;

                          return i;

                        case "nodebuffer":
                          return Buffer.concat(t);

                        default:
                          throw new Error("concat : unsupported type '" + e + "'");
                      }
                    }(i, n), a);

                    t(e);
                  } catch (e) {
                    r(e);
                  }

                  n = [];
                }).resume();
              });
            }

            function f(e, t, r) {
              var n = t;

              switch (t) {
                case "blob":
                case "arraybuffer":
                  n = "uint8array";
                  break;

                case "base64":
                  n = "string";
              }

              try {
                this._internalType = n, this._outputType = t, this._mimeType = r, h.checkSupport(n), this._worker = e.pipe(new i(n)), e.lock();
              } catch (e) {
                this._worker = new s("error"), this._worker.error(e);
              }
            }

            f.prototype = {
              accumulate: function (e) {
                return l(this, e);
              },
              on: function (e, t) {
                var r = this;
                return "data" === e ? this._worker.on(e, function (e) {
                  t.call(r, e.data, e.meta);
                }) : this._worker.on(e, function () {
                  h.delay(t, arguments, r);
                }), this;
              },
              resume: function () {
                return h.delay(this._worker.resume, [], this._worker), this;
              },
              pause: function () {
                return this._worker.pause(), this;
              },
              toNodejsStream: function (e) {
                if (h.checkSupport("nodestream"), "nodebuffer" !== this._outputType) throw new Error(this._outputType + " is not supported by this method");
                return new o(this, {
                  objectMode: "nodebuffer" !== this._outputType
                }, e);
              }
            }, t.exports = f;
          }, {
            "../base64": 1,
            "../external": 6,
            "../nodejs/NodejsStreamOutputAdapter": 13,
            "../support": 30,
            "../utils": 32,
            "./ConvertWorker": 24,
            "./GenericWorker": 28
          }],
          30: [function (e, t, r) {

            if (r.base64 = !0, r.array = !0, r.string = !0, r.arraybuffer = "undefined" != typeof ArrayBuffer && "undefined" != typeof Uint8Array, r.nodebuffer = "undefined" != typeof Buffer, r.uint8array = "undefined" != typeof Uint8Array, "undefined" == typeof ArrayBuffer) r.blob = !1;else {
              var n = new ArrayBuffer(0);

              try {
                r.blob = 0 === new Blob([n], {
                  type: "application/zip"
                }).size;
              } catch (e) {
                try {
                  var i = new (self.BlobBuilder || self.WebKitBlobBuilder || self.MozBlobBuilder || self.MSBlobBuilder)();
                  i.append(n), r.blob = 0 === i.getBlob("application/zip").size;
                } catch (e) {
                  r.blob = !1;
                }
              }
            }

            try {
              r.nodestream = !!e("readable-stream").Readable;
            } catch (e) {
              r.nodestream = !1;
            }
          }, {
            "readable-stream": 16
          }],
          31: [function (e, t, s) {

            for (var o = e("./utils"), h = e("./support"), r = e("./nodejsUtils"), n = e("./stream/GenericWorker"), u = new Array(256), i = 0; i < 256; i++) u[i] = 252 <= i ? 6 : 248 <= i ? 5 : 240 <= i ? 4 : 224 <= i ? 3 : 192 <= i ? 2 : 1;

            u[254] = u[254] = 1;

            function a() {
              n.call(this, "utf-8 decode"), this.leftOver = null;
            }

            function l() {
              n.call(this, "utf-8 encode");
            }

            s.utf8encode = function (e) {
              return h.nodebuffer ? r.newBufferFrom(e, "utf-8") : function (e) {
                var t,
                    r,
                    n,
                    i,
                    s,
                    a = e.length,
                    o = 0;

                for (i = 0; i < a; i++) 55296 == (64512 & (r = e.charCodeAt(i))) && i + 1 < a && 56320 == (64512 & (n = e.charCodeAt(i + 1))) && (r = 65536 + (r - 55296 << 10) + (n - 56320), i++), o += r < 128 ? 1 : r < 2048 ? 2 : r < 65536 ? 3 : 4;

                for (t = h.uint8array ? new Uint8Array(o) : new Array(o), i = s = 0; s < o; i++) 55296 == (64512 & (r = e.charCodeAt(i))) && i + 1 < a && 56320 == (64512 & (n = e.charCodeAt(i + 1))) && (r = 65536 + (r - 55296 << 10) + (n - 56320), i++), r < 128 ? t[s++] = r : (r < 2048 ? t[s++] = 192 | r >>> 6 : (r < 65536 ? t[s++] = 224 | r >>> 12 : (t[s++] = 240 | r >>> 18, t[s++] = 128 | r >>> 12 & 63), t[s++] = 128 | r >>> 6 & 63), t[s++] = 128 | 63 & r);

                return t;
              }(e);
            }, s.utf8decode = function (e) {
              return h.nodebuffer ? o.transformTo("nodebuffer", e).toString("utf-8") : function (e) {
                var t,
                    r,
                    n,
                    i,
                    s = e.length,
                    a = new Array(2 * s);

                for (t = r = 0; t < s;) if ((n = e[t++]) < 128) a[r++] = n;else if (4 < (i = u[n])) a[r++] = 65533, t += i - 1;else {
                  for (n &= 2 === i ? 31 : 3 === i ? 15 : 7; 1 < i && t < s;) n = n << 6 | 63 & e[t++], i--;

                  1 < i ? a[r++] = 65533 : n < 65536 ? a[r++] = n : (n -= 65536, a[r++] = 55296 | n >> 10 & 1023, a[r++] = 56320 | 1023 & n);
                }

                return a.length !== r && (a.subarray ? a = a.subarray(0, r) : a.length = r), o.applyFromCharCode(a);
              }(e = o.transformTo(h.uint8array ? "uint8array" : "array", e));
            }, o.inherits(a, n), a.prototype.processChunk = function (e) {
              var t = o.transformTo(h.uint8array ? "uint8array" : "array", e.data);

              if (this.leftOver && this.leftOver.length) {
                if (h.uint8array) {
                  var r = t;
                  (t = new Uint8Array(r.length + this.leftOver.length)).set(this.leftOver, 0), t.set(r, this.leftOver.length);
                } else t = this.leftOver.concat(t);

                this.leftOver = null;
              }

              var n = function (e, t) {
                var r;

                for ((t = t || e.length) > e.length && (t = e.length), r = t - 1; 0 <= r && 128 == (192 & e[r]);) r--;

                return r < 0 ? t : 0 === r ? t : r + u[e[r]] > t ? r : t;
              }(t),
                  i = t;

              n !== t.length && (h.uint8array ? (i = t.subarray(0, n), this.leftOver = t.subarray(n, t.length)) : (i = t.slice(0, n), this.leftOver = t.slice(n, t.length))), this.push({
                data: s.utf8decode(i),
                meta: e.meta
              });
            }, a.prototype.flush = function () {
              this.leftOver && this.leftOver.length && (this.push({
                data: s.utf8decode(this.leftOver),
                meta: {}
              }), this.leftOver = null);
            }, s.Utf8DecodeWorker = a, o.inherits(l, n), l.prototype.processChunk = function (e) {
              this.push({
                data: s.utf8encode(e.data),
                meta: e.meta
              });
            }, s.Utf8EncodeWorker = l;
          }, {
            "./nodejsUtils": 14,
            "./stream/GenericWorker": 28,
            "./support": 30,
            "./utils": 32
          }],
          32: [function (e, t, a) {

            var o = e("./support"),
                h = e("./base64"),
                r = e("./nodejsUtils"),
                u = e("./external");

            function n(e) {
              return e;
            }

            function l(e, t) {
              for (var r = 0; r < e.length; ++r) t[r] = 255 & e.charCodeAt(r);

              return t;
            }

            e("setimmediate"), a.newBlob = function (t, r) {
              a.checkSupport("blob");

              try {
                return new Blob([t], {
                  type: r
                });
              } catch (e) {
                try {
                  var n = new (self.BlobBuilder || self.WebKitBlobBuilder || self.MozBlobBuilder || self.MSBlobBuilder)();
                  return n.append(t), n.getBlob(r);
                } catch (e) {
                  throw new Error("Bug : can't construct the Blob.");
                }
              }
            };
            var i = {
              stringifyByChunk: function (e, t, r) {
                var n = [],
                    i = 0,
                    s = e.length;
                if (s <= r) return String.fromCharCode.apply(null, e);

                for (; i < s;) "array" === t || "nodebuffer" === t ? n.push(String.fromCharCode.apply(null, e.slice(i, Math.min(i + r, s)))) : n.push(String.fromCharCode.apply(null, e.subarray(i, Math.min(i + r, s)))), i += r;

                return n.join("");
              },
              stringifyByChar: function (e) {
                for (var t = "", r = 0; r < e.length; r++) t += String.fromCharCode(e[r]);

                return t;
              },
              applyCanBeUsed: {
                uint8array: function () {
                  try {
                    return o.uint8array && 1 === String.fromCharCode.apply(null, new Uint8Array(1)).length;
                  } catch (e) {
                    return !1;
                  }
                }(),
                nodebuffer: function () {
                  try {
                    return o.nodebuffer && 1 === String.fromCharCode.apply(null, r.allocBuffer(1)).length;
                  } catch (e) {
                    return !1;
                  }
                }()
              }
            };

            function s(e) {
              var t = 65536,
                  r = a.getTypeOf(e),
                  n = !0;
              if ("uint8array" === r ? n = i.applyCanBeUsed.uint8array : "nodebuffer" === r && (n = i.applyCanBeUsed.nodebuffer), n) for (; 1 < t;) try {
                return i.stringifyByChunk(e, r, t);
              } catch (e) {
                t = Math.floor(t / 2);
              }
              return i.stringifyByChar(e);
            }

            function f(e, t) {
              for (var r = 0; r < e.length; r++) t[r] = e[r];

              return t;
            }

            a.applyFromCharCode = s;
            var c = {};
            c.string = {
              string: n,
              array: function (e) {
                return l(e, new Array(e.length));
              },
              arraybuffer: function (e) {
                return c.string.uint8array(e).buffer;
              },
              uint8array: function (e) {
                return l(e, new Uint8Array(e.length));
              },
              nodebuffer: function (e) {
                return l(e, r.allocBuffer(e.length));
              }
            }, c.array = {
              string: s,
              array: n,
              arraybuffer: function (e) {
                return new Uint8Array(e).buffer;
              },
              uint8array: function (e) {
                return new Uint8Array(e);
              },
              nodebuffer: function (e) {
                return r.newBufferFrom(e);
              }
            }, c.arraybuffer = {
              string: function (e) {
                return s(new Uint8Array(e));
              },
              array: function (e) {
                return f(new Uint8Array(e), new Array(e.byteLength));
              },
              arraybuffer: n,
              uint8array: function (e) {
                return new Uint8Array(e);
              },
              nodebuffer: function (e) {
                return r.newBufferFrom(new Uint8Array(e));
              }
            }, c.uint8array = {
              string: s,
              array: function (e) {
                return f(e, new Array(e.length));
              },
              arraybuffer: function (e) {
                return e.buffer;
              },
              uint8array: n,
              nodebuffer: function (e) {
                return r.newBufferFrom(e);
              }
            }, c.nodebuffer = {
              string: s,
              array: function (e) {
                return f(e, new Array(e.length));
              },
              arraybuffer: function (e) {
                return c.nodebuffer.uint8array(e).buffer;
              },
              uint8array: function (e) {
                return f(e, new Uint8Array(e.length));
              },
              nodebuffer: n
            }, a.transformTo = function (e, t) {
              if (t = t || "", !e) return t;
              a.checkSupport(e);
              var r = a.getTypeOf(t);
              return c[r][e](t);
            }, a.resolve = function (e) {
              for (var t = e.split("/"), r = [], n = 0; n < t.length; n++) {
                var i = t[n];
                "." === i || "" === i && 0 !== n && n !== t.length - 1 || (".." === i ? r.pop() : r.push(i));
              }

              return r.join("/");
            }, a.getTypeOf = function (e) {
              return "string" == typeof e ? "string" : "[object Array]" === Object.prototype.toString.call(e) ? "array" : o.nodebuffer && r.isBuffer(e) ? "nodebuffer" : o.uint8array && e instanceof Uint8Array ? "uint8array" : o.arraybuffer && e instanceof ArrayBuffer ? "arraybuffer" : void 0;
            }, a.checkSupport = function (e) {
              if (!o[e.toLowerCase()]) throw new Error(e + " is not supported by this platform");
            }, a.MAX_VALUE_16BITS = 65535, a.MAX_VALUE_32BITS = -1, a.pretty = function (e) {
              var t,
                  r,
                  n = "";

              for (r = 0; r < (e || "").length; r++) n += "\\x" + ((t = e.charCodeAt(r)) < 16 ? "0" : "") + t.toString(16).toUpperCase();

              return n;
            }, a.delay = function (e, t, r) {
              setImmediate(function () {
                e.apply(r || null, t || []);
              });
            }, a.inherits = function (e, t) {
              function r() {}

              r.prototype = t.prototype, e.prototype = new r();
            }, a.extend = function () {
              var e,
                  t,
                  r = {};

              for (e = 0; e < arguments.length; e++) for (t in arguments[e]) arguments[e].hasOwnProperty(t) && void 0 === r[t] && (r[t] = arguments[e][t]);

              return r;
            }, a.prepareContent = function (r, e, n, i, s) {
              return u.Promise.resolve(e).then(function (n) {
                return o.blob && (n instanceof Blob || -1 !== ["[object File]", "[object Blob]"].indexOf(Object.prototype.toString.call(n))) && "undefined" != typeof FileReader ? new u.Promise(function (t, r) {
                  var e = new FileReader();
                  e.onload = function (e) {
                    t(e.target.result);
                  }, e.onerror = function (e) {
                    r(e.target.error);
                  }, e.readAsArrayBuffer(n);
                }) : n;
              }).then(function (e) {
                var t = a.getTypeOf(e);
                return t ? ("arraybuffer" === t ? e = a.transformTo("uint8array", e) : "string" === t && (s ? e = h.decode(e) : n && !0 !== i && (e = function (e) {
                  return l(e, o.uint8array ? new Uint8Array(e.length) : new Array(e.length));
                }(e))), e) : u.Promise.reject(new Error("Can't read the data of '" + r + "'. Is it in a supported JavaScript type (String, Blob, ArrayBuffer, etc) ?"));
              });
            };
          }, {
            "./base64": 1,
            "./external": 6,
            "./nodejsUtils": 14,
            "./support": 30,
            setimmediate: 54
          }],
          33: [function (e, t, r) {

            var n = e("./reader/readerFor"),
                i = e("./utils"),
                s = e("./signature"),
                a = e("./zipEntry"),
                o = (e("./utf8"), e("./support"));

            function h(e) {
              this.files = [], this.loadOptions = e;
            }

            h.prototype = {
              checkSignature: function (e) {
                if (!this.reader.readAndCheckSignature(e)) {
                  this.reader.index -= 4;
                  var t = this.reader.readString(4);
                  throw new Error("Corrupted zip or bug: unexpected signature (" + i.pretty(t) + ", expected " + i.pretty(e) + ")");
                }
              },
              isSignature: function (e, t) {
                var r = this.reader.index;
                this.reader.setIndex(e);
                var n = this.reader.readString(4) === t;
                return this.reader.setIndex(r), n;
              },
              readBlockEndOfCentral: function () {
                this.diskNumber = this.reader.readInt(2), this.diskWithCentralDirStart = this.reader.readInt(2), this.centralDirRecordsOnThisDisk = this.reader.readInt(2), this.centralDirRecords = this.reader.readInt(2), this.centralDirSize = this.reader.readInt(4), this.centralDirOffset = this.reader.readInt(4), this.zipCommentLength = this.reader.readInt(2);
                var e = this.reader.readData(this.zipCommentLength),
                    t = o.uint8array ? "uint8array" : "array",
                    r = i.transformTo(t, e);
                this.zipComment = this.loadOptions.decodeFileName(r);
              },
              readBlockZip64EndOfCentral: function () {
                this.zip64EndOfCentralSize = this.reader.readInt(8), this.reader.skip(4), this.diskNumber = this.reader.readInt(4), this.diskWithCentralDirStart = this.reader.readInt(4), this.centralDirRecordsOnThisDisk = this.reader.readInt(8), this.centralDirRecords = this.reader.readInt(8), this.centralDirSize = this.reader.readInt(8), this.centralDirOffset = this.reader.readInt(8), this.zip64ExtensibleData = {};

                for (var e, t, r, n = this.zip64EndOfCentralSize - 44; 0 < n;) e = this.reader.readInt(2), t = this.reader.readInt(4), r = this.reader.readData(t), this.zip64ExtensibleData[e] = {
                  id: e,
                  length: t,
                  value: r
                };
              },
              readBlockZip64EndOfCentralLocator: function () {
                if (this.diskWithZip64CentralDirStart = this.reader.readInt(4), this.relativeOffsetEndOfZip64CentralDir = this.reader.readInt(8), this.disksCount = this.reader.readInt(4), 1 < this.disksCount) throw new Error("Multi-volumes zip are not supported");
              },
              readLocalFiles: function () {
                var e, t;

                for (e = 0; e < this.files.length; e++) t = this.files[e], this.reader.setIndex(t.localHeaderOffset), this.checkSignature(s.LOCAL_FILE_HEADER), t.readLocalPart(this.reader), t.handleUTF8(), t.processAttributes();
              },
              readCentralDir: function () {
                var e;

                for (this.reader.setIndex(this.centralDirOffset); this.reader.readAndCheckSignature(s.CENTRAL_FILE_HEADER);) (e = new a({
                  zip64: this.zip64
                }, this.loadOptions)).readCentralPart(this.reader), this.files.push(e);

                if (this.centralDirRecords !== this.files.length && 0 !== this.centralDirRecords && 0 === this.files.length) throw new Error("Corrupted zip or bug: expected " + this.centralDirRecords + " records in central dir, got " + this.files.length);
              },
              readEndOfCentral: function () {
                var e = this.reader.lastIndexOfSignature(s.CENTRAL_DIRECTORY_END);
                if (e < 0) throw !this.isSignature(0, s.LOCAL_FILE_HEADER) ? new Error("Can't find end of central directory : is this a zip file ? If it is, see https://stuk.github.io/jszip/documentation/howto/read_zip.html") : new Error("Corrupted zip: can't find end of central directory");
                this.reader.setIndex(e);
                var t = e;

                if (this.checkSignature(s.CENTRAL_DIRECTORY_END), this.readBlockEndOfCentral(), this.diskNumber === i.MAX_VALUE_16BITS || this.diskWithCentralDirStart === i.MAX_VALUE_16BITS || this.centralDirRecordsOnThisDisk === i.MAX_VALUE_16BITS || this.centralDirRecords === i.MAX_VALUE_16BITS || this.centralDirSize === i.MAX_VALUE_32BITS || this.centralDirOffset === i.MAX_VALUE_32BITS) {
                  if (this.zip64 = !0, (e = this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR)) < 0) throw new Error("Corrupted zip: can't find the ZIP64 end of central directory locator");
                  if (this.reader.setIndex(e), this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR), this.readBlockZip64EndOfCentralLocator(), !this.isSignature(this.relativeOffsetEndOfZip64CentralDir, s.ZIP64_CENTRAL_DIRECTORY_END) && (this.relativeOffsetEndOfZip64CentralDir = this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_END), this.relativeOffsetEndOfZip64CentralDir < 0)) throw new Error("Corrupted zip: can't find the ZIP64 end of central directory");
                  this.reader.setIndex(this.relativeOffsetEndOfZip64CentralDir), this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_END), this.readBlockZip64EndOfCentral();
                }

                var r = this.centralDirOffset + this.centralDirSize;
                this.zip64 && (r += 20, r += 12 + this.zip64EndOfCentralSize);
                var n = t - r;
                if (0 < n) this.isSignature(t, s.CENTRAL_FILE_HEADER) || (this.reader.zero = n);else if (n < 0) throw new Error("Corrupted zip: missing " + Math.abs(n) + " bytes.");
              },
              prepareReader: function (e) {
                this.reader = n(e);
              },
              load: function (e) {
                this.prepareReader(e), this.readEndOfCentral(), this.readCentralDir(), this.readLocalFiles();
              }
            }, t.exports = h;
          }, {
            "./reader/readerFor": 22,
            "./signature": 23,
            "./support": 30,
            "./utf8": 31,
            "./utils": 32,
            "./zipEntry": 34
          }],
          34: [function (e, t, r) {

            var n = e("./reader/readerFor"),
                s = e("./utils"),
                i = e("./compressedObject"),
                a = e("./crc32"),
                o = e("./utf8"),
                h = e("./compressions"),
                u = e("./support");

            function l(e, t) {
              this.options = e, this.loadOptions = t;
            }

            l.prototype = {
              isEncrypted: function () {
                return 1 == (1 & this.bitFlag);
              },
              useUTF8: function () {
                return 2048 == (2048 & this.bitFlag);
              },
              readLocalPart: function (e) {
                var t, r;
                if (e.skip(22), this.fileNameLength = e.readInt(2), r = e.readInt(2), this.fileName = e.readData(this.fileNameLength), e.skip(r), -1 === this.compressedSize || -1 === this.uncompressedSize) throw new Error("Bug or corrupted zip : didn't get enough information from the central directory (compressedSize === -1 || uncompressedSize === -1)");
                if (null === (t = function (e) {
                  for (var t in h) if (h.hasOwnProperty(t) && h[t].magic === e) return h[t];

                  return null;
                }(this.compressionMethod))) throw new Error("Corrupted zip : compression " + s.pretty(this.compressionMethod) + " unknown (inner file : " + s.transformTo("string", this.fileName) + ")");
                this.decompressed = new i(this.compressedSize, this.uncompressedSize, this.crc32, t, e.readData(this.compressedSize));
              },
              readCentralPart: function (e) {
                this.versionMadeBy = e.readInt(2), e.skip(2), this.bitFlag = e.readInt(2), this.compressionMethod = e.readString(2), this.date = e.readDate(), this.crc32 = e.readInt(4), this.compressedSize = e.readInt(4), this.uncompressedSize = e.readInt(4);
                var t = e.readInt(2);
                if (this.extraFieldsLength = e.readInt(2), this.fileCommentLength = e.readInt(2), this.diskNumberStart = e.readInt(2), this.internalFileAttributes = e.readInt(2), this.externalFileAttributes = e.readInt(4), this.localHeaderOffset = e.readInt(4), this.isEncrypted()) throw new Error("Encrypted zip are not supported");
                e.skip(t), this.readExtraFields(e), this.parseZIP64ExtraField(e), this.fileComment = e.readData(this.fileCommentLength);
              },
              processAttributes: function () {
                this.unixPermissions = null, this.dosPermissions = null;
                var e = this.versionMadeBy >> 8;
                this.dir = !!(16 & this.externalFileAttributes), 0 == e && (this.dosPermissions = 63 & this.externalFileAttributes), 3 == e && (this.unixPermissions = this.externalFileAttributes >> 16 & 65535), this.dir || "/" !== this.fileNameStr.slice(-1) || (this.dir = !0);
              },
              parseZIP64ExtraField: function (e) {
                if (this.extraFields[1]) {
                  var t = n(this.extraFields[1].value);
                  this.uncompressedSize === s.MAX_VALUE_32BITS && (this.uncompressedSize = t.readInt(8)), this.compressedSize === s.MAX_VALUE_32BITS && (this.compressedSize = t.readInt(8)), this.localHeaderOffset === s.MAX_VALUE_32BITS && (this.localHeaderOffset = t.readInt(8)), this.diskNumberStart === s.MAX_VALUE_32BITS && (this.diskNumberStart = t.readInt(4));
                }
              },
              readExtraFields: function (e) {
                var t,
                    r,
                    n,
                    i = e.index + this.extraFieldsLength;

                for (this.extraFields || (this.extraFields = {}); e.index + 4 < i;) t = e.readInt(2), r = e.readInt(2), n = e.readData(r), this.extraFields[t] = {
                  id: t,
                  length: r,
                  value: n
                };

                e.setIndex(i);
              },
              handleUTF8: function () {
                var e = u.uint8array ? "uint8array" : "array";
                if (this.useUTF8()) this.fileNameStr = o.utf8decode(this.fileName), this.fileCommentStr = o.utf8decode(this.fileComment);else {
                  var t = this.findExtraFieldUnicodePath();
                  if (null !== t) this.fileNameStr = t;else {
                    var r = s.transformTo(e, this.fileName);
                    this.fileNameStr = this.loadOptions.decodeFileName(r);
                  }
                  var n = this.findExtraFieldUnicodeComment();
                  if (null !== n) this.fileCommentStr = n;else {
                    var i = s.transformTo(e, this.fileComment);
                    this.fileCommentStr = this.loadOptions.decodeFileName(i);
                  }
                }
              },
              findExtraFieldUnicodePath: function () {
                var e = this.extraFields[28789];

                if (e) {
                  var t = n(e.value);
                  return 1 !== t.readInt(1) ? null : a(this.fileName) !== t.readInt(4) ? null : o.utf8decode(t.readData(e.length - 5));
                }

                return null;
              },
              findExtraFieldUnicodeComment: function () {
                var e = this.extraFields[25461];

                if (e) {
                  var t = n(e.value);
                  return 1 !== t.readInt(1) ? null : a(this.fileComment) !== t.readInt(4) ? null : o.utf8decode(t.readData(e.length - 5));
                }

                return null;
              }
            }, t.exports = l;
          }, {
            "./compressedObject": 2,
            "./compressions": 3,
            "./crc32": 4,
            "./reader/readerFor": 22,
            "./support": 30,
            "./utf8": 31,
            "./utils": 32
          }],
          35: [function (e, t, r) {

            function n(e, t, r) {
              this.name = e, this.dir = r.dir, this.date = r.date, this.comment = r.comment, this.unixPermissions = r.unixPermissions, this.dosPermissions = r.dosPermissions, this._data = t, this._dataBinary = r.binary, this.options = {
                compression: r.compression,
                compressionOptions: r.compressionOptions
              };
            }

            var s = e("./stream/StreamHelper"),
                i = e("./stream/DataWorker"),
                a = e("./utf8"),
                o = e("./compressedObject"),
                h = e("./stream/GenericWorker");
            n.prototype = {
              internalStream: function (e) {
                var t = null,
                    r = "string";

                try {
                  if (!e) throw new Error("No output type specified.");
                  var n = "string" === (r = e.toLowerCase()) || "text" === r;
                  "binarystring" !== r && "text" !== r || (r = "string"), t = this._decompressWorker();
                  var i = !this._dataBinary;
                  i && !n && (t = t.pipe(new a.Utf8EncodeWorker())), !i && n && (t = t.pipe(new a.Utf8DecodeWorker()));
                } catch (e) {
                  (t = new h("error")).error(e);
                }

                return new s(t, r, "");
              },
              async: function (e, t) {
                return this.internalStream(e).accumulate(t);
              },
              nodeStream: function (e, t) {
                return this.internalStream(e || "nodebuffer").toNodejsStream(t);
              },
              _compressWorker: function (e, t) {
                if (this._data instanceof o && this._data.compression.magic === e.magic) return this._data.getCompressedWorker();

                var r = this._decompressWorker();

                return this._dataBinary || (r = r.pipe(new a.Utf8EncodeWorker())), o.createWorkerFrom(r, e, t);
              },
              _decompressWorker: function () {
                return this._data instanceof o ? this._data.getContentWorker() : this._data instanceof h ? this._data : new i(this._data);
              }
            };

            for (var u = ["asText", "asBinary", "asNodeBuffer", "asUint8Array", "asArrayBuffer"], l = function () {
              throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.");
            }, f = 0; f < u.length; f++) n.prototype[u[f]] = l;

            t.exports = n;
          }, {
            "./compressedObject": 2,
            "./stream/DataWorker": 27,
            "./stream/GenericWorker": 28,
            "./stream/StreamHelper": 29,
            "./utf8": 31
          }],
          36: [function (e, l, t) {
            (function (t) {

              var r,
                  n,
                  e = t.MutationObserver || t.WebKitMutationObserver;

              if (e) {
                var i = 0,
                    s = new e(u),
                    a = t.document.createTextNode("");
                s.observe(a, {
                  characterData: !0
                }), r = function () {
                  a.data = i = ++i % 2;
                };
              } else if (t.setImmediate || void 0 === t.MessageChannel) r = "document" in t && "onreadystatechange" in t.document.createElement("script") ? function () {
                var e = t.document.createElement("script");
                e.onreadystatechange = function () {
                  u(), e.onreadystatechange = null, e.parentNode.removeChild(e), e = null;
                }, t.document.documentElement.appendChild(e);
              } : function () {
                setTimeout(u, 0);
              };else {
                var o = new t.MessageChannel();
                o.port1.onmessage = u, r = function () {
                  o.port2.postMessage(0);
                };
              }

              var h = [];

              function u() {
                var e, t;
                n = !0;

                for (var r = h.length; r;) {
                  for (t = h, h = [], e = -1; ++e < r;) t[e]();

                  r = h.length;
                }

                n = !1;
              }

              l.exports = function (e) {
                1 !== h.push(e) || n || r();
              };
            }).call(this, "undefined" != typeof commonjsGlobal ? commonjsGlobal : "undefined" != typeof self ? self : "undefined" != typeof window ? window : {});
          }, {}],
          37: [function (e, t, r) {

            var i = e("immediate");

            function u() {}

            var l = {},
                s = ["REJECTED"],
                a = ["FULFILLED"],
                n = ["PENDING"];

            function o(e) {
              if ("function" != typeof e) throw new TypeError("resolver must be a function");
              this.state = n, this.queue = [], this.outcome = void 0, e !== u && d(this, e);
            }

            function h(e, t, r) {
              this.promise = e, "function" == typeof t && (this.onFulfilled = t, this.callFulfilled = this.otherCallFulfilled), "function" == typeof r && (this.onRejected = r, this.callRejected = this.otherCallRejected);
            }

            function f(t, r, n) {
              i(function () {
                var e;

                try {
                  e = r(n);
                } catch (e) {
                  return l.reject(t, e);
                }

                e === t ? l.reject(t, new TypeError("Cannot resolve promise with itself")) : l.resolve(t, e);
              });
            }

            function c(e) {
              var t = e && e.then;
              if (e && ("object" == typeof e || "function" == typeof e) && "function" == typeof t) return function () {
                t.apply(e, arguments);
              };
            }

            function d(t, e) {
              var r = !1;

              function n(e) {
                r || (r = !0, l.reject(t, e));
              }

              function i(e) {
                r || (r = !0, l.resolve(t, e));
              }

              var s = p(function () {
                e(i, n);
              });
              "error" === s.status && n(s.value);
            }

            function p(e, t) {
              var r = {};

              try {
                r.value = e(t), r.status = "success";
              } catch (e) {
                r.status = "error", r.value = e;
              }

              return r;
            }

            (t.exports = o).prototype.finally = function (t) {
              if ("function" != typeof t) return this;
              var r = this.constructor;
              return this.then(function (e) {
                return r.resolve(t()).then(function () {
                  return e;
                });
              }, function (e) {
                return r.resolve(t()).then(function () {
                  throw e;
                });
              });
            }, o.prototype.catch = function (e) {
              return this.then(null, e);
            }, o.prototype.then = function (e, t) {
              if ("function" != typeof e && this.state === a || "function" != typeof t && this.state === s) return this;
              var r = new this.constructor(u);
              this.state !== n ? f(r, this.state === a ? e : t, this.outcome) : this.queue.push(new h(r, e, t));
              return r;
            }, h.prototype.callFulfilled = function (e) {
              l.resolve(this.promise, e);
            }, h.prototype.otherCallFulfilled = function (e) {
              f(this.promise, this.onFulfilled, e);
            }, h.prototype.callRejected = function (e) {
              l.reject(this.promise, e);
            }, h.prototype.otherCallRejected = function (e) {
              f(this.promise, this.onRejected, e);
            }, l.resolve = function (e, t) {
              var r = p(c, t);
              if ("error" === r.status) return l.reject(e, r.value);
              var n = r.value;
              if (n) d(e, n);else {
                e.state = a, e.outcome = t;

                for (var i = -1, s = e.queue.length; ++i < s;) e.queue[i].callFulfilled(t);
              }
              return e;
            }, l.reject = function (e, t) {
              e.state = s, e.outcome = t;

              for (var r = -1, n = e.queue.length; ++r < n;) e.queue[r].callRejected(t);

              return e;
            }, o.resolve = function (e) {
              if (e instanceof this) return e;
              return l.resolve(new this(u), e);
            }, o.reject = function (e) {
              var t = new this(u);
              return l.reject(t, e);
            }, o.all = function (e) {
              var r = this;
              if ("[object Array]" !== Object.prototype.toString.call(e)) return this.reject(new TypeError("must be an array"));
              var n = e.length,
                  i = !1;
              if (!n) return this.resolve([]);
              var s = new Array(n),
                  a = 0,
                  t = -1,
                  o = new this(u);

              for (; ++t < n;) h(e[t], t);

              return o;

              function h(e, t) {
                r.resolve(e).then(function (e) {
                  s[t] = e, ++a !== n || i || (i = !0, l.resolve(o, s));
                }, function (e) {
                  i || (i = !0, l.reject(o, e));
                });
              }
            }, o.race = function (e) {
              var t = this;
              if ("[object Array]" !== Object.prototype.toString.call(e)) return this.reject(new TypeError("must be an array"));
              var r = e.length,
                  n = !1;
              if (!r) return this.resolve([]);
              var i = -1,
                  s = new this(u);

              for (; ++i < r;) a = e[i], t.resolve(a).then(function (e) {
                n || (n = !0, l.resolve(s, e));
              }, function (e) {
                n || (n = !0, l.reject(s, e));
              });

              var a;
              return s;
            };
          }, {
            immediate: 36
          }],
          38: [function (e, t, r) {

            var n = {};
            (0, e("./lib/utils/common").assign)(n, e("./lib/deflate"), e("./lib/inflate"), e("./lib/zlib/constants")), t.exports = n;
          }, {
            "./lib/deflate": 39,
            "./lib/inflate": 40,
            "./lib/utils/common": 41,
            "./lib/zlib/constants": 44
          }],
          39: [function (e, t, r) {

            var a = e("./zlib/deflate"),
                o = e("./utils/common"),
                h = e("./utils/strings"),
                i = e("./zlib/messages"),
                s = e("./zlib/zstream"),
                u = Object.prototype.toString,
                l = 0,
                f = -1,
                c = 0,
                d = 8;

            function p(e) {
              if (!(this instanceof p)) return new p(e);
              this.options = o.assign({
                level: f,
                method: d,
                chunkSize: 16384,
                windowBits: 15,
                memLevel: 8,
                strategy: c,
                to: ""
              }, e || {});
              var t = this.options;
              t.raw && 0 < t.windowBits ? t.windowBits = -t.windowBits : t.gzip && 0 < t.windowBits && t.windowBits < 16 && (t.windowBits += 16), this.err = 0, this.msg = "", this.ended = !1, this.chunks = [], this.strm = new s(), this.strm.avail_out = 0;
              var r = a.deflateInit2(this.strm, t.level, t.method, t.windowBits, t.memLevel, t.strategy);
              if (r !== l) throw new Error(i[r]);

              if (t.header && a.deflateSetHeader(this.strm, t.header), t.dictionary) {
                var n;
                if (n = "string" == typeof t.dictionary ? h.string2buf(t.dictionary) : "[object ArrayBuffer]" === u.call(t.dictionary) ? new Uint8Array(t.dictionary) : t.dictionary, (r = a.deflateSetDictionary(this.strm, n)) !== l) throw new Error(i[r]);
                this._dict_set = !0;
              }
            }

            function n(e, t) {
              var r = new p(t);
              if (r.push(e, !0), r.err) throw r.msg || i[r.err];
              return r.result;
            }

            p.prototype.push = function (e, t) {
              var r,
                  n,
                  i = this.strm,
                  s = this.options.chunkSize;
              if (this.ended) return !1;
              n = t === ~~t ? t : !0 === t ? 4 : 0, "string" == typeof e ? i.input = h.string2buf(e) : "[object ArrayBuffer]" === u.call(e) ? i.input = new Uint8Array(e) : i.input = e, i.next_in = 0, i.avail_in = i.input.length;

              do {
                if (0 === i.avail_out && (i.output = new o.Buf8(s), i.next_out = 0, i.avail_out = s), 1 !== (r = a.deflate(i, n)) && r !== l) return this.onEnd(r), !(this.ended = !0);
                0 !== i.avail_out && (0 !== i.avail_in || 4 !== n && 2 !== n) || ("string" === this.options.to ? this.onData(h.buf2binstring(o.shrinkBuf(i.output, i.next_out))) : this.onData(o.shrinkBuf(i.output, i.next_out)));
              } while ((0 < i.avail_in || 0 === i.avail_out) && 1 !== r);

              return 4 === n ? (r = a.deflateEnd(this.strm), this.onEnd(r), this.ended = !0, r === l) : 2 !== n || (this.onEnd(l), !(i.avail_out = 0));
            }, p.prototype.onData = function (e) {
              this.chunks.push(e);
            }, p.prototype.onEnd = function (e) {
              e === l && ("string" === this.options.to ? this.result = this.chunks.join("") : this.result = o.flattenChunks(this.chunks)), this.chunks = [], this.err = e, this.msg = this.strm.msg;
            }, r.Deflate = p, r.deflate = n, r.deflateRaw = function (e, t) {
              return (t = t || {}).raw = !0, n(e, t);
            }, r.gzip = function (e, t) {
              return (t = t || {}).gzip = !0, n(e, t);
            };
          }, {
            "./utils/common": 41,
            "./utils/strings": 42,
            "./zlib/deflate": 46,
            "./zlib/messages": 51,
            "./zlib/zstream": 53
          }],
          40: [function (e, t, r) {

            var c = e("./zlib/inflate"),
                d = e("./utils/common"),
                p = e("./utils/strings"),
                m = e("./zlib/constants"),
                n = e("./zlib/messages"),
                i = e("./zlib/zstream"),
                s = e("./zlib/gzheader"),
                _ = Object.prototype.toString;

            function a(e) {
              if (!(this instanceof a)) return new a(e);
              this.options = d.assign({
                chunkSize: 16384,
                windowBits: 0,
                to: ""
              }, e || {});
              var t = this.options;
              t.raw && 0 <= t.windowBits && t.windowBits < 16 && (t.windowBits = -t.windowBits, 0 === t.windowBits && (t.windowBits = -15)), !(0 <= t.windowBits && t.windowBits < 16) || e && e.windowBits || (t.windowBits += 32), 15 < t.windowBits && t.windowBits < 48 && 0 == (15 & t.windowBits) && (t.windowBits |= 15), this.err = 0, this.msg = "", this.ended = !1, this.chunks = [], this.strm = new i(), this.strm.avail_out = 0;
              var r = c.inflateInit2(this.strm, t.windowBits);
              if (r !== m.Z_OK) throw new Error(n[r]);
              this.header = new s(), c.inflateGetHeader(this.strm, this.header);
            }

            function o(e, t) {
              var r = new a(t);
              if (r.push(e, !0), r.err) throw r.msg || n[r.err];
              return r.result;
            }

            a.prototype.push = function (e, t) {
              var r,
                  n,
                  i,
                  s,
                  a,
                  o,
                  h = this.strm,
                  u = this.options.chunkSize,
                  l = this.options.dictionary,
                  f = !1;
              if (this.ended) return !1;
              n = t === ~~t ? t : !0 === t ? m.Z_FINISH : m.Z_NO_FLUSH, "string" == typeof e ? h.input = p.binstring2buf(e) : "[object ArrayBuffer]" === _.call(e) ? h.input = new Uint8Array(e) : h.input = e, h.next_in = 0, h.avail_in = h.input.length;

              do {
                if (0 === h.avail_out && (h.output = new d.Buf8(u), h.next_out = 0, h.avail_out = u), (r = c.inflate(h, m.Z_NO_FLUSH)) === m.Z_NEED_DICT && l && (o = "string" == typeof l ? p.string2buf(l) : "[object ArrayBuffer]" === _.call(l) ? new Uint8Array(l) : l, r = c.inflateSetDictionary(this.strm, o)), r === m.Z_BUF_ERROR && !0 === f && (r = m.Z_OK, f = !1), r !== m.Z_STREAM_END && r !== m.Z_OK) return this.onEnd(r), !(this.ended = !0);
                h.next_out && (0 !== h.avail_out && r !== m.Z_STREAM_END && (0 !== h.avail_in || n !== m.Z_FINISH && n !== m.Z_SYNC_FLUSH) || ("string" === this.options.to ? (i = p.utf8border(h.output, h.next_out), s = h.next_out - i, a = p.buf2string(h.output, i), h.next_out = s, h.avail_out = u - s, s && d.arraySet(h.output, h.output, i, s, 0), this.onData(a)) : this.onData(d.shrinkBuf(h.output, h.next_out)))), 0 === h.avail_in && 0 === h.avail_out && (f = !0);
              } while ((0 < h.avail_in || 0 === h.avail_out) && r !== m.Z_STREAM_END);

              return r === m.Z_STREAM_END && (n = m.Z_FINISH), n === m.Z_FINISH ? (r = c.inflateEnd(this.strm), this.onEnd(r), this.ended = !0, r === m.Z_OK) : n !== m.Z_SYNC_FLUSH || (this.onEnd(m.Z_OK), !(h.avail_out = 0));
            }, a.prototype.onData = function (e) {
              this.chunks.push(e);
            }, a.prototype.onEnd = function (e) {
              e === m.Z_OK && ("string" === this.options.to ? this.result = this.chunks.join("") : this.result = d.flattenChunks(this.chunks)), this.chunks = [], this.err = e, this.msg = this.strm.msg;
            }, r.Inflate = a, r.inflate = o, r.inflateRaw = function (e, t) {
              return (t = t || {}).raw = !0, o(e, t);
            }, r.ungzip = o;
          }, {
            "./utils/common": 41,
            "./utils/strings": 42,
            "./zlib/constants": 44,
            "./zlib/gzheader": 47,
            "./zlib/inflate": 49,
            "./zlib/messages": 51,
            "./zlib/zstream": 53
          }],
          41: [function (e, t, r) {

            var n = "undefined" != typeof Uint8Array && "undefined" != typeof Uint16Array && "undefined" != typeof Int32Array;
            r.assign = function (e) {
              for (var t = Array.prototype.slice.call(arguments, 1); t.length;) {
                var r = t.shift();

                if (r) {
                  if ("object" != typeof r) throw new TypeError(r + "must be non-object");

                  for (var n in r) r.hasOwnProperty(n) && (e[n] = r[n]);
                }
              }

              return e;
            }, r.shrinkBuf = function (e, t) {
              return e.length === t ? e : e.subarray ? e.subarray(0, t) : (e.length = t, e);
            };
            var i = {
              arraySet: function (e, t, r, n, i) {
                if (t.subarray && e.subarray) e.set(t.subarray(r, r + n), i);else for (var s = 0; s < n; s++) e[i + s] = t[r + s];
              },
              flattenChunks: function (e) {
                var t, r, n, i, s, a;

                for (t = n = 0, r = e.length; t < r; t++) n += e[t].length;

                for (a = new Uint8Array(n), t = i = 0, r = e.length; t < r; t++) s = e[t], a.set(s, i), i += s.length;

                return a;
              }
            },
                s = {
              arraySet: function (e, t, r, n, i) {
                for (var s = 0; s < n; s++) e[i + s] = t[r + s];
              },
              flattenChunks: function (e) {
                return [].concat.apply([], e);
              }
            };
            r.setTyped = function (e) {
              e ? (r.Buf8 = Uint8Array, r.Buf16 = Uint16Array, r.Buf32 = Int32Array, r.assign(r, i)) : (r.Buf8 = Array, r.Buf16 = Array, r.Buf32 = Array, r.assign(r, s));
            }, r.setTyped(n);
          }, {}],
          42: [function (e, t, r) {

            var h = e("./common"),
                i = !0,
                s = !0;

            try {
              String.fromCharCode.apply(null, [0]);
            } catch (e) {
              i = !1;
            }

            try {
              String.fromCharCode.apply(null, new Uint8Array(1));
            } catch (e) {
              s = !1;
            }

            for (var u = new h.Buf8(256), n = 0; n < 256; n++) u[n] = 252 <= n ? 6 : 248 <= n ? 5 : 240 <= n ? 4 : 224 <= n ? 3 : 192 <= n ? 2 : 1;

            function l(e, t) {
              if (t < 65537 && (e.subarray && s || !e.subarray && i)) return String.fromCharCode.apply(null, h.shrinkBuf(e, t));

              for (var r = "", n = 0; n < t; n++) r += String.fromCharCode(e[n]);

              return r;
            }

            u[254] = u[254] = 1, r.string2buf = function (e) {
              var t,
                  r,
                  n,
                  i,
                  s,
                  a = e.length,
                  o = 0;

              for (i = 0; i < a; i++) 55296 == (64512 & (r = e.charCodeAt(i))) && i + 1 < a && 56320 == (64512 & (n = e.charCodeAt(i + 1))) && (r = 65536 + (r - 55296 << 10) + (n - 56320), i++), o += r < 128 ? 1 : r < 2048 ? 2 : r < 65536 ? 3 : 4;

              for (t = new h.Buf8(o), i = s = 0; s < o; i++) 55296 == (64512 & (r = e.charCodeAt(i))) && i + 1 < a && 56320 == (64512 & (n = e.charCodeAt(i + 1))) && (r = 65536 + (r - 55296 << 10) + (n - 56320), i++), r < 128 ? t[s++] = r : (r < 2048 ? t[s++] = 192 | r >>> 6 : (r < 65536 ? t[s++] = 224 | r >>> 12 : (t[s++] = 240 | r >>> 18, t[s++] = 128 | r >>> 12 & 63), t[s++] = 128 | r >>> 6 & 63), t[s++] = 128 | 63 & r);

              return t;
            }, r.buf2binstring = function (e) {
              return l(e, e.length);
            }, r.binstring2buf = function (e) {
              for (var t = new h.Buf8(e.length), r = 0, n = t.length; r < n; r++) t[r] = e.charCodeAt(r);

              return t;
            }, r.buf2string = function (e, t) {
              var r,
                  n,
                  i,
                  s,
                  a = t || e.length,
                  o = new Array(2 * a);

              for (r = n = 0; r < a;) if ((i = e[r++]) < 128) o[n++] = i;else if (4 < (s = u[i])) o[n++] = 65533, r += s - 1;else {
                for (i &= 2 === s ? 31 : 3 === s ? 15 : 7; 1 < s && r < a;) i = i << 6 | 63 & e[r++], s--;

                1 < s ? o[n++] = 65533 : i < 65536 ? o[n++] = i : (i -= 65536, o[n++] = 55296 | i >> 10 & 1023, o[n++] = 56320 | 1023 & i);
              }

              return l(o, n);
            }, r.utf8border = function (e, t) {
              var r;

              for ((t = t || e.length) > e.length && (t = e.length), r = t - 1; 0 <= r && 128 == (192 & e[r]);) r--;

              return r < 0 ? t : 0 === r ? t : r + u[e[r]] > t ? r : t;
            };
          }, {
            "./common": 41
          }],
          43: [function (e, t, r) {

            t.exports = function (e, t, r, n) {
              for (var i = 65535 & e | 0, s = e >>> 16 & 65535 | 0, a = 0; 0 !== r;) {
                for (r -= a = 2e3 < r ? 2e3 : r; s = s + (i = i + t[n++] | 0) | 0, --a;);

                i %= 65521, s %= 65521;
              }

              return i | s << 16 | 0;
            };
          }, {}],
          44: [function (e, t, r) {

            t.exports = {
              Z_NO_FLUSH: 0,
              Z_PARTIAL_FLUSH: 1,
              Z_SYNC_FLUSH: 2,
              Z_FULL_FLUSH: 3,
              Z_FINISH: 4,
              Z_BLOCK: 5,
              Z_TREES: 6,
              Z_OK: 0,
              Z_STREAM_END: 1,
              Z_NEED_DICT: 2,
              Z_ERRNO: -1,
              Z_STREAM_ERROR: -2,
              Z_DATA_ERROR: -3,
              Z_BUF_ERROR: -5,
              Z_NO_COMPRESSION: 0,
              Z_BEST_SPEED: 1,
              Z_BEST_COMPRESSION: 9,
              Z_DEFAULT_COMPRESSION: -1,
              Z_FILTERED: 1,
              Z_HUFFMAN_ONLY: 2,
              Z_RLE: 3,
              Z_FIXED: 4,
              Z_DEFAULT_STRATEGY: 0,
              Z_BINARY: 0,
              Z_TEXT: 1,
              Z_UNKNOWN: 2,
              Z_DEFLATED: 8
            };
          }, {}],
          45: [function (e, t, r) {

            var o = function () {
              for (var e, t = [], r = 0; r < 256; r++) {
                e = r;

                for (var n = 0; n < 8; n++) e = 1 & e ? 3988292384 ^ e >>> 1 : e >>> 1;

                t[r] = e;
              }

              return t;
            }();

            t.exports = function (e, t, r, n) {
              var i = o,
                  s = n + r;
              e ^= -1;

              for (var a = n; a < s; a++) e = e >>> 8 ^ i[255 & (e ^ t[a])];

              return -1 ^ e;
            };
          }, {}],
          46: [function (e, t, r) {

            var h,
                c = e("../utils/common"),
                u = e("./trees"),
                d = e("./adler32"),
                p = e("./crc32"),
                n = e("./messages"),
                l = 0,
                f = 4,
                m = 0,
                _ = -2,
                g = -1,
                b = 4,
                i = 2,
                v = 8,
                y = 9,
                s = 286,
                a = 30,
                o = 19,
                w = 2 * s + 1,
                k = 15,
                x = 3,
                S = 258,
                z = S + x + 1,
                C = 42,
                E = 113,
                A = 1,
                I = 2,
                O = 3,
                B = 4;

            function R(e, t) {
              return e.msg = n[t], t;
            }

            function T(e) {
              return (e << 1) - (4 < e ? 9 : 0);
            }

            function D(e) {
              for (var t = e.length; 0 <= --t;) e[t] = 0;
            }

            function F(e) {
              var t = e.state,
                  r = t.pending;
              r > e.avail_out && (r = e.avail_out), 0 !== r && (c.arraySet(e.output, t.pending_buf, t.pending_out, r, e.next_out), e.next_out += r, t.pending_out += r, e.total_out += r, e.avail_out -= r, t.pending -= r, 0 === t.pending && (t.pending_out = 0));
            }

            function N(e, t) {
              u._tr_flush_block(e, 0 <= e.block_start ? e.block_start : -1, e.strstart - e.block_start, t), e.block_start = e.strstart, F(e.strm);
            }

            function U(e, t) {
              e.pending_buf[e.pending++] = t;
            }

            function P(e, t) {
              e.pending_buf[e.pending++] = t >>> 8 & 255, e.pending_buf[e.pending++] = 255 & t;
            }

            function L(e, t) {
              var r,
                  n,
                  i = e.max_chain_length,
                  s = e.strstart,
                  a = e.prev_length,
                  o = e.nice_match,
                  h = e.strstart > e.w_size - z ? e.strstart - (e.w_size - z) : 0,
                  u = e.window,
                  l = e.w_mask,
                  f = e.prev,
                  c = e.strstart + S,
                  d = u[s + a - 1],
                  p = u[s + a];
              e.prev_length >= e.good_match && (i >>= 2), o > e.lookahead && (o = e.lookahead);

              do {
                if (u[(r = t) + a] === p && u[r + a - 1] === d && u[r] === u[s] && u[++r] === u[s + 1]) {
                  s += 2, r++;

                  do {} while (u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && u[++s] === u[++r] && s < c);

                  if (n = S - (c - s), s = c - S, a < n) {
                    if (e.match_start = t, o <= (a = n)) break;
                    d = u[s + a - 1], p = u[s + a];
                  }
                }
              } while ((t = f[t & l]) > h && 0 != --i);

              return a <= e.lookahead ? a : e.lookahead;
            }

            function j(e) {
              var t,
                  r,
                  n,
                  i,
                  s,
                  a,
                  o,
                  h,
                  u,
                  l,
                  f = e.w_size;

              do {
                if (i = e.window_size - e.lookahead - e.strstart, e.strstart >= f + (f - z)) {
                  for (c.arraySet(e.window, e.window, f, f, 0), e.match_start -= f, e.strstart -= f, e.block_start -= f, t = r = e.hash_size; n = e.head[--t], e.head[t] = f <= n ? n - f : 0, --r;);

                  for (t = r = f; n = e.prev[--t], e.prev[t] = f <= n ? n - f : 0, --r;);

                  i += f;
                }

                if (0 === e.strm.avail_in) break;
                if (a = e.strm, o = e.window, h = e.strstart + e.lookahead, u = i, l = void 0, l = a.avail_in, u < l && (l = u), r = 0 === l ? 0 : (a.avail_in -= l, c.arraySet(o, a.input, a.next_in, l, h), 1 === a.state.wrap ? a.adler = d(a.adler, o, l, h) : 2 === a.state.wrap && (a.adler = p(a.adler, o, l, h)), a.next_in += l, a.total_in += l, l), e.lookahead += r, e.lookahead + e.insert >= x) for (s = e.strstart - e.insert, e.ins_h = e.window[s], e.ins_h = (e.ins_h << e.hash_shift ^ e.window[s + 1]) & e.hash_mask; e.insert && (e.ins_h = (e.ins_h << e.hash_shift ^ e.window[s + x - 1]) & e.hash_mask, e.prev[s & e.w_mask] = e.head[e.ins_h], e.head[e.ins_h] = s, s++, e.insert--, !(e.lookahead + e.insert < x)););
              } while (e.lookahead < z && 0 !== e.strm.avail_in);
            }

            function Z(e, t) {
              for (var r, n;;) {
                if (e.lookahead < z) {
                  if (j(e), e.lookahead < z && t === l) return A;
                  if (0 === e.lookahead) break;
                }

                if (r = 0, e.lookahead >= x && (e.ins_h = (e.ins_h << e.hash_shift ^ e.window[e.strstart + x - 1]) & e.hash_mask, r = e.prev[e.strstart & e.w_mask] = e.head[e.ins_h], e.head[e.ins_h] = e.strstart), 0 !== r && e.strstart - r <= e.w_size - z && (e.match_length = L(e, r)), e.match_length >= x) {
                  if (n = u._tr_tally(e, e.strstart - e.match_start, e.match_length - x), e.lookahead -= e.match_length, e.match_length <= e.max_lazy_match && e.lookahead >= x) {
                    for (e.match_length--; e.strstart++, e.ins_h = (e.ins_h << e.hash_shift ^ e.window[e.strstart + x - 1]) & e.hash_mask, r = e.prev[e.strstart & e.w_mask] = e.head[e.ins_h], e.head[e.ins_h] = e.strstart, 0 != --e.match_length;);

                    e.strstart++;
                  } else e.strstart += e.match_length, e.match_length = 0, e.ins_h = e.window[e.strstart], e.ins_h = (e.ins_h << e.hash_shift ^ e.window[e.strstart + 1]) & e.hash_mask;
                } else n = u._tr_tally(e, 0, e.window[e.strstart]), e.lookahead--, e.strstart++;
                if (n && (N(e, !1), 0 === e.strm.avail_out)) return A;
              }

              return e.insert = e.strstart < x - 1 ? e.strstart : x - 1, t === f ? (N(e, !0), 0 === e.strm.avail_out ? O : B) : e.last_lit && (N(e, !1), 0 === e.strm.avail_out) ? A : I;
            }

            function W(e, t) {
              for (var r, n, i;;) {
                if (e.lookahead < z) {
                  if (j(e), e.lookahead < z && t === l) return A;
                  if (0 === e.lookahead) break;
                }

                if (r = 0, e.lookahead >= x && (e.ins_h = (e.ins_h << e.hash_shift ^ e.window[e.strstart + x - 1]) & e.hash_mask, r = e.prev[e.strstart & e.w_mask] = e.head[e.ins_h], e.head[e.ins_h] = e.strstart), e.prev_length = e.match_length, e.prev_match = e.match_start, e.match_length = x - 1, 0 !== r && e.prev_length < e.max_lazy_match && e.strstart - r <= e.w_size - z && (e.match_length = L(e, r), e.match_length <= 5 && (1 === e.strategy || e.match_length === x && 4096 < e.strstart - e.match_start) && (e.match_length = x - 1)), e.prev_length >= x && e.match_length <= e.prev_length) {
                  for (i = e.strstart + e.lookahead - x, n = u._tr_tally(e, e.strstart - 1 - e.prev_match, e.prev_length - x), e.lookahead -= e.prev_length - 1, e.prev_length -= 2; ++e.strstart <= i && (e.ins_h = (e.ins_h << e.hash_shift ^ e.window[e.strstart + x - 1]) & e.hash_mask, r = e.prev[e.strstart & e.w_mask] = e.head[e.ins_h], e.head[e.ins_h] = e.strstart), 0 != --e.prev_length;);

                  if (e.match_available = 0, e.match_length = x - 1, e.strstart++, n && (N(e, !1), 0 === e.strm.avail_out)) return A;
                } else if (e.match_available) {
                  if ((n = u._tr_tally(e, 0, e.window[e.strstart - 1])) && N(e, !1), e.strstart++, e.lookahead--, 0 === e.strm.avail_out) return A;
                } else e.match_available = 1, e.strstart++, e.lookahead--;
              }

              return e.match_available && (n = u._tr_tally(e, 0, e.window[e.strstart - 1]), e.match_available = 0), e.insert = e.strstart < x - 1 ? e.strstart : x - 1, t === f ? (N(e, !0), 0 === e.strm.avail_out ? O : B) : e.last_lit && (N(e, !1), 0 === e.strm.avail_out) ? A : I;
            }

            function M(e, t, r, n, i) {
              this.good_length = e, this.max_lazy = t, this.nice_length = r, this.max_chain = n, this.func = i;
            }

            function H() {
              this.strm = null, this.status = 0, this.pending_buf = null, this.pending_buf_size = 0, this.pending_out = 0, this.pending = 0, this.wrap = 0, this.gzhead = null, this.gzindex = 0, this.method = v, this.last_flush = -1, this.w_size = 0, this.w_bits = 0, this.w_mask = 0, this.window = null, this.window_size = 0, this.prev = null, this.head = null, this.ins_h = 0, this.hash_size = 0, this.hash_bits = 0, this.hash_mask = 0, this.hash_shift = 0, this.block_start = 0, this.match_length = 0, this.prev_match = 0, this.match_available = 0, this.strstart = 0, this.match_start = 0, this.lookahead = 0, this.prev_length = 0, this.max_chain_length = 0, this.max_lazy_match = 0, this.level = 0, this.strategy = 0, this.good_match = 0, this.nice_match = 0, this.dyn_ltree = new c.Buf16(2 * w), this.dyn_dtree = new c.Buf16(2 * (2 * a + 1)), this.bl_tree = new c.Buf16(2 * (2 * o + 1)), D(this.dyn_ltree), D(this.dyn_dtree), D(this.bl_tree), this.l_desc = null, this.d_desc = null, this.bl_desc = null, this.bl_count = new c.Buf16(k + 1), this.heap = new c.Buf16(2 * s + 1), D(this.heap), this.heap_len = 0, this.heap_max = 0, this.depth = new c.Buf16(2 * s + 1), D(this.depth), this.l_buf = 0, this.lit_bufsize = 0, this.last_lit = 0, this.d_buf = 0, this.opt_len = 0, this.static_len = 0, this.matches = 0, this.insert = 0, this.bi_buf = 0, this.bi_valid = 0;
            }

            function G(e) {
              var t;
              return e && e.state ? (e.total_in = e.total_out = 0, e.data_type = i, (t = e.state).pending = 0, t.pending_out = 0, t.wrap < 0 && (t.wrap = -t.wrap), t.status = t.wrap ? C : E, e.adler = 2 === t.wrap ? 0 : 1, t.last_flush = l, u._tr_init(t), m) : R(e, _);
            }

            function K(e) {
              var t = G(e);
              return t === m && function (e) {
                e.window_size = 2 * e.w_size, D(e.head), e.max_lazy_match = h[e.level].max_lazy, e.good_match = h[e.level].good_length, e.nice_match = h[e.level].nice_length, e.max_chain_length = h[e.level].max_chain, e.strstart = 0, e.block_start = 0, e.lookahead = 0, e.insert = 0, e.match_length = e.prev_length = x - 1, e.match_available = 0, e.ins_h = 0;
              }(e.state), t;
            }

            function Y(e, t, r, n, i, s) {
              if (!e) return _;
              var a = 1;
              if (t === g && (t = 6), n < 0 ? (a = 0, n = -n) : 15 < n && (a = 2, n -= 16), i < 1 || y < i || r !== v || n < 8 || 15 < n || t < 0 || 9 < t || s < 0 || b < s) return R(e, _);
              8 === n && (n = 9);
              var o = new H();
              return (e.state = o).strm = e, o.wrap = a, o.gzhead = null, o.w_bits = n, o.w_size = 1 << o.w_bits, o.w_mask = o.w_size - 1, o.hash_bits = i + 7, o.hash_size = 1 << o.hash_bits, o.hash_mask = o.hash_size - 1, o.hash_shift = ~~((o.hash_bits + x - 1) / x), o.window = new c.Buf8(2 * o.w_size), o.head = new c.Buf16(o.hash_size), o.prev = new c.Buf16(o.w_size), o.lit_bufsize = 1 << i + 6, o.pending_buf_size = 4 * o.lit_bufsize, o.pending_buf = new c.Buf8(o.pending_buf_size), o.d_buf = 1 * o.lit_bufsize, o.l_buf = 3 * o.lit_bufsize, o.level = t, o.strategy = s, o.method = r, K(e);
            }

            h = [new M(0, 0, 0, 0, function (e, t) {
              var r = 65535;

              for (r > e.pending_buf_size - 5 && (r = e.pending_buf_size - 5);;) {
                if (e.lookahead <= 1) {
                  if (j(e), 0 === e.lookahead && t === l) return A;
                  if (0 === e.lookahead) break;
                }

                e.strstart += e.lookahead, e.lookahead = 0;
                var n = e.block_start + r;
                if ((0 === e.strstart || e.strstart >= n) && (e.lookahead = e.strstart - n, e.strstart = n, N(e, !1), 0 === e.strm.avail_out)) return A;
                if (e.strstart - e.block_start >= e.w_size - z && (N(e, !1), 0 === e.strm.avail_out)) return A;
              }

              return e.insert = 0, t === f ? (N(e, !0), 0 === e.strm.avail_out ? O : B) : (e.strstart > e.block_start && (N(e, !1), e.strm.avail_out), A);
            }), new M(4, 4, 8, 4, Z), new M(4, 5, 16, 8, Z), new M(4, 6, 32, 32, Z), new M(4, 4, 16, 16, W), new M(8, 16, 32, 32, W), new M(8, 16, 128, 128, W), new M(8, 32, 128, 256, W), new M(32, 128, 258, 1024, W), new M(32, 258, 258, 4096, W)], r.deflateInit = function (e, t) {
              return Y(e, t, v, 15, 8, 0);
            }, r.deflateInit2 = Y, r.deflateReset = K, r.deflateResetKeep = G, r.deflateSetHeader = function (e, t) {
              return e && e.state ? 2 !== e.state.wrap ? _ : (e.state.gzhead = t, m) : _;
            }, r.deflate = function (e, t) {
              var r, n, i, s;
              if (!e || !e.state || 5 < t || t < 0) return e ? R(e, _) : _;
              if (n = e.state, !e.output || !e.input && 0 !== e.avail_in || 666 === n.status && t !== f) return R(e, 0 === e.avail_out ? -5 : _);
              if (n.strm = e, r = n.last_flush, n.last_flush = t, n.status === C) if (2 === n.wrap) e.adler = 0, U(n, 31), U(n, 139), U(n, 8), n.gzhead ? (U(n, (n.gzhead.text ? 1 : 0) + (n.gzhead.hcrc ? 2 : 0) + (n.gzhead.extra ? 4 : 0) + (n.gzhead.name ? 8 : 0) + (n.gzhead.comment ? 16 : 0)), U(n, 255 & n.gzhead.time), U(n, n.gzhead.time >> 8 & 255), U(n, n.gzhead.time >> 16 & 255), U(n, n.gzhead.time >> 24 & 255), U(n, 9 === n.level ? 2 : 2 <= n.strategy || n.level < 2 ? 4 : 0), U(n, 255 & n.gzhead.os), n.gzhead.extra && n.gzhead.extra.length && (U(n, 255 & n.gzhead.extra.length), U(n, n.gzhead.extra.length >> 8 & 255)), n.gzhead.hcrc && (e.adler = p(e.adler, n.pending_buf, n.pending, 0)), n.gzindex = 0, n.status = 69) : (U(n, 0), U(n, 0), U(n, 0), U(n, 0), U(n, 0), U(n, 9 === n.level ? 2 : 2 <= n.strategy || n.level < 2 ? 4 : 0), U(n, 3), n.status = E);else {
                var a = v + (n.w_bits - 8 << 4) << 8;
                a |= (2 <= n.strategy || n.level < 2 ? 0 : n.level < 6 ? 1 : 6 === n.level ? 2 : 3) << 6, 0 !== n.strstart && (a |= 32), a += 31 - a % 31, n.status = E, P(n, a), 0 !== n.strstart && (P(n, e.adler >>> 16), P(n, 65535 & e.adler)), e.adler = 1;
              }
              if (69 === n.status) if (n.gzhead.extra) {
                for (i = n.pending; n.gzindex < (65535 & n.gzhead.extra.length) && (n.pending !== n.pending_buf_size || (n.gzhead.hcrc && n.pending > i && (e.adler = p(e.adler, n.pending_buf, n.pending - i, i)), F(e), i = n.pending, n.pending !== n.pending_buf_size));) U(n, 255 & n.gzhead.extra[n.gzindex]), n.gzindex++;

                n.gzhead.hcrc && n.pending > i && (e.adler = p(e.adler, n.pending_buf, n.pending - i, i)), n.gzindex === n.gzhead.extra.length && (n.gzindex = 0, n.status = 73);
              } else n.status = 73;
              if (73 === n.status) if (n.gzhead.name) {
                i = n.pending;

                do {
                  if (n.pending === n.pending_buf_size && (n.gzhead.hcrc && n.pending > i && (e.adler = p(e.adler, n.pending_buf, n.pending - i, i)), F(e), i = n.pending, n.pending === n.pending_buf_size)) {
                    s = 1;
                    break;
                  }

                  s = n.gzindex < n.gzhead.name.length ? 255 & n.gzhead.name.charCodeAt(n.gzindex++) : 0, U(n, s);
                } while (0 !== s);

                n.gzhead.hcrc && n.pending > i && (e.adler = p(e.adler, n.pending_buf, n.pending - i, i)), 0 === s && (n.gzindex = 0, n.status = 91);
              } else n.status = 91;
              if (91 === n.status) if (n.gzhead.comment) {
                i = n.pending;

                do {
                  if (n.pending === n.pending_buf_size && (n.gzhead.hcrc && n.pending > i && (e.adler = p(e.adler, n.pending_buf, n.pending - i, i)), F(e), i = n.pending, n.pending === n.pending_buf_size)) {
                    s = 1;
                    break;
                  }

                  s = n.gzindex < n.gzhead.comment.length ? 255 & n.gzhead.comment.charCodeAt(n.gzindex++) : 0, U(n, s);
                } while (0 !== s);

                n.gzhead.hcrc && n.pending > i && (e.adler = p(e.adler, n.pending_buf, n.pending - i, i)), 0 === s && (n.status = 103);
              } else n.status = 103;

              if (103 === n.status && (n.gzhead.hcrc ? (n.pending + 2 > n.pending_buf_size && F(e), n.pending + 2 <= n.pending_buf_size && (U(n, 255 & e.adler), U(n, e.adler >> 8 & 255), e.adler = 0, n.status = E)) : n.status = E), 0 !== n.pending) {
                if (F(e), 0 === e.avail_out) return n.last_flush = -1, m;
              } else if (0 === e.avail_in && T(t) <= T(r) && t !== f) return R(e, -5);

              if (666 === n.status && 0 !== e.avail_in) return R(e, -5);

              if (0 !== e.avail_in || 0 !== n.lookahead || t !== l && 666 !== n.status) {
                var o = 2 === n.strategy ? function (e, t) {
                  for (var r;;) {
                    if (0 === e.lookahead && (j(e), 0 === e.lookahead)) {
                      if (t === l) return A;
                      break;
                    }

                    if (e.match_length = 0, r = u._tr_tally(e, 0, e.window[e.strstart]), e.lookahead--, e.strstart++, r && (N(e, !1), 0 === e.strm.avail_out)) return A;
                  }

                  return e.insert = 0, t === f ? (N(e, !0), 0 === e.strm.avail_out ? O : B) : e.last_lit && (N(e, !1), 0 === e.strm.avail_out) ? A : I;
                }(n, t) : 3 === n.strategy ? function (e, t) {
                  for (var r, n, i, s, a = e.window;;) {
                    if (e.lookahead <= S) {
                      if (j(e), e.lookahead <= S && t === l) return A;
                      if (0 === e.lookahead) break;
                    }

                    if (e.match_length = 0, e.lookahead >= x && 0 < e.strstart && (n = a[i = e.strstart - 1]) === a[++i] && n === a[++i] && n === a[++i]) {
                      s = e.strstart + S;

                      do {} while (n === a[++i] && n === a[++i] && n === a[++i] && n === a[++i] && n === a[++i] && n === a[++i] && n === a[++i] && n === a[++i] && i < s);

                      e.match_length = S - (s - i), e.match_length > e.lookahead && (e.match_length = e.lookahead);
                    }

                    if (e.match_length >= x ? (r = u._tr_tally(e, 1, e.match_length - x), e.lookahead -= e.match_length, e.strstart += e.match_length, e.match_length = 0) : (r = u._tr_tally(e, 0, e.window[e.strstart]), e.lookahead--, e.strstart++), r && (N(e, !1), 0 === e.strm.avail_out)) return A;
                  }

                  return e.insert = 0, t === f ? (N(e, !0), 0 === e.strm.avail_out ? O : B) : e.last_lit && (N(e, !1), 0 === e.strm.avail_out) ? A : I;
                }(n, t) : h[n.level].func(n, t);
                if (o !== O && o !== B || (n.status = 666), o === A || o === O) return 0 === e.avail_out && (n.last_flush = -1), m;
                if (o === I && (1 === t ? u._tr_align(n) : 5 !== t && (u._tr_stored_block(n, 0, 0, !1), 3 === t && (D(n.head), 0 === n.lookahead && (n.strstart = 0, n.block_start = 0, n.insert = 0))), F(e), 0 === e.avail_out)) return n.last_flush = -1, m;
              }

              return t !== f ? m : n.wrap <= 0 ? 1 : (2 === n.wrap ? (U(n, 255 & e.adler), U(n, e.adler >> 8 & 255), U(n, e.adler >> 16 & 255), U(n, e.adler >> 24 & 255), U(n, 255 & e.total_in), U(n, e.total_in >> 8 & 255), U(n, e.total_in >> 16 & 255), U(n, e.total_in >> 24 & 255)) : (P(n, e.adler >>> 16), P(n, 65535 & e.adler)), F(e), 0 < n.wrap && (n.wrap = -n.wrap), 0 !== n.pending ? m : 1);
            }, r.deflateEnd = function (e) {
              var t;
              return e && e.state ? (t = e.state.status) !== C && 69 !== t && 73 !== t && 91 !== t && 103 !== t && t !== E && 666 !== t ? R(e, _) : (e.state = null, t === E ? R(e, -3) : m) : _;
            }, r.deflateSetDictionary = function (e, t) {
              var r,
                  n,
                  i,
                  s,
                  a,
                  o,
                  h,
                  u,
                  l = t.length;
              if (!e || !e.state) return _;
              if (2 === (s = (r = e.state).wrap) || 1 === s && r.status !== C || r.lookahead) return _;

              for (1 === s && (e.adler = d(e.adler, t, l, 0)), r.wrap = 0, l >= r.w_size && (0 === s && (D(r.head), r.strstart = 0, r.block_start = 0, r.insert = 0), u = new c.Buf8(r.w_size), c.arraySet(u, t, l - r.w_size, r.w_size, 0), t = u, l = r.w_size), a = e.avail_in, o = e.next_in, h = e.input, e.avail_in = l, e.next_in = 0, e.input = t, j(r); r.lookahead >= x;) {
                for (n = r.strstart, i = r.lookahead - (x - 1); r.ins_h = (r.ins_h << r.hash_shift ^ r.window[n + x - 1]) & r.hash_mask, r.prev[n & r.w_mask] = r.head[r.ins_h], r.head[r.ins_h] = n, n++, --i;);

                r.strstart = n, r.lookahead = x - 1, j(r);
              }

              return r.strstart += r.lookahead, r.block_start = r.strstart, r.insert = r.lookahead, r.lookahead = 0, r.match_length = r.prev_length = x - 1, r.match_available = 0, e.next_in = o, e.input = h, e.avail_in = a, r.wrap = s, m;
            }, r.deflateInfo = "pako deflate (from Nodeca project)";
          }, {
            "../utils/common": 41,
            "./adler32": 43,
            "./crc32": 45,
            "./messages": 51,
            "./trees": 52
          }],
          47: [function (e, t, r) {

            t.exports = function () {
              this.text = 0, this.time = 0, this.xflags = 0, this.os = 0, this.extra = null, this.extra_len = 0, this.name = "", this.comment = "", this.hcrc = 0, this.done = !1;
            };
          }, {}],
          48: [function (e, t, r) {

            t.exports = function (e, t) {
              var r, n, i, s, a, o, h, u, l, f, c, d, p, m, _, g, b, v, y, w, k, x, S, z, C;

              r = e.state, n = e.next_in, z = e.input, i = n + (e.avail_in - 5), s = e.next_out, C = e.output, a = s - (t - e.avail_out), o = s + (e.avail_out - 257), h = r.dmax, u = r.wsize, l = r.whave, f = r.wnext, c = r.window, d = r.hold, p = r.bits, m = r.lencode, _ = r.distcode, g = (1 << r.lenbits) - 1, b = (1 << r.distbits) - 1;

              e: do {
                p < 15 && (d += z[n++] << p, p += 8, d += z[n++] << p, p += 8), v = m[d & g];

                t: for (;;) {
                  if (d >>>= y = v >>> 24, p -= y, 0 === (y = v >>> 16 & 255)) C[s++] = 65535 & v;else {
                    if (!(16 & y)) {
                      if (0 == (64 & y)) {
                        v = m[(65535 & v) + (d & (1 << y) - 1)];
                        continue t;
                      }

                      if (32 & y) {
                        r.mode = 12;
                        break e;
                      }

                      e.msg = "invalid literal/length code", r.mode = 30;
                      break e;
                    }

                    w = 65535 & v, (y &= 15) && (p < y && (d += z[n++] << p, p += 8), w += d & (1 << y) - 1, d >>>= y, p -= y), p < 15 && (d += z[n++] << p, p += 8, d += z[n++] << p, p += 8), v = _[d & b];

                    r: for (;;) {
                      if (d >>>= y = v >>> 24, p -= y, !(16 & (y = v >>> 16 & 255))) {
                        if (0 == (64 & y)) {
                          v = _[(65535 & v) + (d & (1 << y) - 1)];
                          continue r;
                        }

                        e.msg = "invalid distance code", r.mode = 30;
                        break e;
                      }

                      if (k = 65535 & v, p < (y &= 15) && (d += z[n++] << p, (p += 8) < y && (d += z[n++] << p, p += 8)), h < (k += d & (1 << y) - 1)) {
                        e.msg = "invalid distance too far back", r.mode = 30;
                        break e;
                      }

                      if (d >>>= y, p -= y, (y = s - a) < k) {
                        if (l < (y = k - y) && r.sane) {
                          e.msg = "invalid distance too far back", r.mode = 30;
                          break e;
                        }

                        if (S = c, (x = 0) === f) {
                          if (x += u - y, y < w) {
                            for (w -= y; C[s++] = c[x++], --y;);

                            x = s - k, S = C;
                          }
                        } else if (f < y) {
                          if (x += u + f - y, (y -= f) < w) {
                            for (w -= y; C[s++] = c[x++], --y;);

                            if (x = 0, f < w) {
                              for (w -= y = f; C[s++] = c[x++], --y;);

                              x = s - k, S = C;
                            }
                          }
                        } else if (x += f - y, y < w) {
                          for (w -= y; C[s++] = c[x++], --y;);

                          x = s - k, S = C;
                        }

                        for (; 2 < w;) C[s++] = S[x++], C[s++] = S[x++], C[s++] = S[x++], w -= 3;

                        w && (C[s++] = S[x++], 1 < w && (C[s++] = S[x++]));
                      } else {
                        for (x = s - k; C[s++] = C[x++], C[s++] = C[x++], C[s++] = C[x++], 2 < (w -= 3););

                        w && (C[s++] = C[x++], 1 < w && (C[s++] = C[x++]));
                      }

                      break;
                    }
                  }
                  break;
                }
              } while (n < i && s < o);

              n -= w = p >> 3, d &= (1 << (p -= w << 3)) - 1, e.next_in = n, e.next_out = s, e.avail_in = n < i ? i - n + 5 : 5 - (n - i), e.avail_out = s < o ? o - s + 257 : 257 - (s - o), r.hold = d, r.bits = p;
            };
          }, {}],
          49: [function (e, t, r) {

            var I = e("../utils/common"),
                O = e("./adler32"),
                B = e("./crc32"),
                R = e("./inffast"),
                T = e("./inftrees"),
                D = 1,
                F = 2,
                N = 0,
                U = -2,
                P = 1,
                n = 852,
                i = 592;

            function L(e) {
              return (e >>> 24 & 255) + (e >>> 8 & 65280) + ((65280 & e) << 8) + ((255 & e) << 24);
            }

            function s() {
              this.mode = 0, this.last = !1, this.wrap = 0, this.havedict = !1, this.flags = 0, this.dmax = 0, this.check = 0, this.total = 0, this.head = null, this.wbits = 0, this.wsize = 0, this.whave = 0, this.wnext = 0, this.window = null, this.hold = 0, this.bits = 0, this.length = 0, this.offset = 0, this.extra = 0, this.lencode = null, this.distcode = null, this.lenbits = 0, this.distbits = 0, this.ncode = 0, this.nlen = 0, this.ndist = 0, this.have = 0, this.next = null, this.lens = new I.Buf16(320), this.work = new I.Buf16(288), this.lendyn = null, this.distdyn = null, this.sane = 0, this.back = 0, this.was = 0;
            }

            function a(e) {
              var t;
              return e && e.state ? (t = e.state, e.total_in = e.total_out = t.total = 0, e.msg = "", t.wrap && (e.adler = 1 & t.wrap), t.mode = P, t.last = 0, t.havedict = 0, t.dmax = 32768, t.head = null, t.hold = 0, t.bits = 0, t.lencode = t.lendyn = new I.Buf32(n), t.distcode = t.distdyn = new I.Buf32(i), t.sane = 1, t.back = -1, N) : U;
            }

            function o(e) {
              var t;
              return e && e.state ? ((t = e.state).wsize = 0, t.whave = 0, t.wnext = 0, a(e)) : U;
            }

            function h(e, t) {
              var r, n;
              return e && e.state ? (n = e.state, t < 0 ? (r = 0, t = -t) : (r = 1 + (t >> 4), t < 48 && (t &= 15)), t && (t < 8 || 15 < t) ? U : (null !== n.window && n.wbits !== t && (n.window = null), n.wrap = r, n.wbits = t, o(e))) : U;
            }

            function u(e, t) {
              var r, n;
              return e ? (n = new s(), (e.state = n).window = null, (r = h(e, t)) !== N && (e.state = null), r) : U;
            }

            var l,
                f,
                c = !0;

            function j(e) {
              if (c) {
                var t;

                for (l = new I.Buf32(512), f = new I.Buf32(32), t = 0; t < 144;) e.lens[t++] = 8;

                for (; t < 256;) e.lens[t++] = 9;

                for (; t < 280;) e.lens[t++] = 7;

                for (; t < 288;) e.lens[t++] = 8;

                for (T(D, e.lens, 0, 288, l, 0, e.work, {
                  bits: 9
                }), t = 0; t < 32;) e.lens[t++] = 5;

                T(F, e.lens, 0, 32, f, 0, e.work, {
                  bits: 5
                }), c = !1;
              }

              e.lencode = l, e.lenbits = 9, e.distcode = f, e.distbits = 5;
            }

            function Z(e, t, r, n) {
              var i,
                  s = e.state;
              return null === s.window && (s.wsize = 1 << s.wbits, s.wnext = 0, s.whave = 0, s.window = new I.Buf8(s.wsize)), n >= s.wsize ? (I.arraySet(s.window, t, r - s.wsize, s.wsize, 0), s.wnext = 0, s.whave = s.wsize) : (n < (i = s.wsize - s.wnext) && (i = n), I.arraySet(s.window, t, r - n, i, s.wnext), (n -= i) ? (I.arraySet(s.window, t, r - n, n, 0), s.wnext = n, s.whave = s.wsize) : (s.wnext += i, s.wnext === s.wsize && (s.wnext = 0), s.whave < s.wsize && (s.whave += i))), 0;
            }

            r.inflateReset = o, r.inflateReset2 = h, r.inflateResetKeep = a, r.inflateInit = function (e) {
              return u(e, 15);
            }, r.inflateInit2 = u, r.inflate = function (e, t) {
              var r,
                  n,
                  i,
                  s,
                  a,
                  o,
                  h,
                  u,
                  l,
                  f,
                  c,
                  d,
                  p,
                  m,
                  _,
                  g,
                  b,
                  v,
                  y,
                  w,
                  k,
                  x,
                  S,
                  z,
                  C = 0,
                  E = new I.Buf8(4),
                  A = [16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15];

              if (!e || !e.state || !e.output || !e.input && 0 !== e.avail_in) return U;
              12 === (r = e.state).mode && (r.mode = 13), a = e.next_out, i = e.output, h = e.avail_out, s = e.next_in, n = e.input, o = e.avail_in, u = r.hold, l = r.bits, f = o, c = h, x = N;

              e: for (;;) switch (r.mode) {
                case P:
                  if (0 === r.wrap) {
                    r.mode = 13;
                    break;
                  }

                  for (; l < 16;) {
                    if (0 === o) break e;
                    o--, u += n[s++] << l, l += 8;
                  }

                  if (2 & r.wrap && 35615 === u) {
                    E[r.check = 0] = 255 & u, E[1] = u >>> 8 & 255, r.check = B(r.check, E, 2, 0), l = u = 0, r.mode = 2;
                    break;
                  }

                  if (r.flags = 0, r.head && (r.head.done = !1), !(1 & r.wrap) || (((255 & u) << 8) + (u >> 8)) % 31) {
                    e.msg = "incorrect header check", r.mode = 30;
                    break;
                  }

                  if (8 != (15 & u)) {
                    e.msg = "unknown compression method", r.mode = 30;
                    break;
                  }

                  if (l -= 4, k = 8 + (15 & (u >>>= 4)), 0 === r.wbits) r.wbits = k;else if (k > r.wbits) {
                    e.msg = "invalid window size", r.mode = 30;
                    break;
                  }
                  r.dmax = 1 << k, e.adler = r.check = 1, r.mode = 512 & u ? 10 : 12, l = u = 0;
                  break;

                case 2:
                  for (; l < 16;) {
                    if (0 === o) break e;
                    o--, u += n[s++] << l, l += 8;
                  }

                  if (r.flags = u, 8 != (255 & r.flags)) {
                    e.msg = "unknown compression method", r.mode = 30;
                    break;
                  }

                  if (57344 & r.flags) {
                    e.msg = "unknown header flags set", r.mode = 30;
                    break;
                  }

                  r.head && (r.head.text = u >> 8 & 1), 512 & r.flags && (E[0] = 255 & u, E[1] = u >>> 8 & 255, r.check = B(r.check, E, 2, 0)), l = u = 0, r.mode = 3;

                case 3:
                  for (; l < 32;) {
                    if (0 === o) break e;
                    o--, u += n[s++] << l, l += 8;
                  }

                  r.head && (r.head.time = u), 512 & r.flags && (E[0] = 255 & u, E[1] = u >>> 8 & 255, E[2] = u >>> 16 & 255, E[3] = u >>> 24 & 255, r.check = B(r.check, E, 4, 0)), l = u = 0, r.mode = 4;

                case 4:
                  for (; l < 16;) {
                    if (0 === o) break e;
                    o--, u += n[s++] << l, l += 8;
                  }

                  r.head && (r.head.xflags = 255 & u, r.head.os = u >> 8), 512 & r.flags && (E[0] = 255 & u, E[1] = u >>> 8 & 255, r.check = B(r.check, E, 2, 0)), l = u = 0, r.mode = 5;

                case 5:
                  if (1024 & r.flags) {
                    for (; l < 16;) {
                      if (0 === o) break e;
                      o--, u += n[s++] << l, l += 8;
                    }

                    r.length = u, r.head && (r.head.extra_len = u), 512 & r.flags && (E[0] = 255 & u, E[1] = u >>> 8 & 255, r.check = B(r.check, E, 2, 0)), l = u = 0;
                  } else r.head && (r.head.extra = null);

                  r.mode = 6;

                case 6:
                  if (1024 & r.flags && (o < (d = r.length) && (d = o), d && (r.head && (k = r.head.extra_len - r.length, r.head.extra || (r.head.extra = new Array(r.head.extra_len)), I.arraySet(r.head.extra, n, s, d, k)), 512 & r.flags && (r.check = B(r.check, n, d, s)), o -= d, s += d, r.length -= d), r.length)) break e;
                  r.length = 0, r.mode = 7;

                case 7:
                  if (2048 & r.flags) {
                    if (0 === o) break e;

                    for (d = 0; k = n[s + d++], r.head && k && r.length < 65536 && (r.head.name += String.fromCharCode(k)), k && d < o;);

                    if (512 & r.flags && (r.check = B(r.check, n, d, s)), o -= d, s += d, k) break e;
                  } else r.head && (r.head.name = null);

                  r.length = 0, r.mode = 8;

                case 8:
                  if (4096 & r.flags) {
                    if (0 === o) break e;

                    for (d = 0; k = n[s + d++], r.head && k && r.length < 65536 && (r.head.comment += String.fromCharCode(k)), k && d < o;);

                    if (512 & r.flags && (r.check = B(r.check, n, d, s)), o -= d, s += d, k) break e;
                  } else r.head && (r.head.comment = null);

                  r.mode = 9;

                case 9:
                  if (512 & r.flags) {
                    for (; l < 16;) {
                      if (0 === o) break e;
                      o--, u += n[s++] << l, l += 8;
                    }

                    if (u !== (65535 & r.check)) {
                      e.msg = "header crc mismatch", r.mode = 30;
                      break;
                    }

                    l = u = 0;
                  }

                  r.head && (r.head.hcrc = r.flags >> 9 & 1, r.head.done = !0), e.adler = r.check = 0, r.mode = 12;
                  break;

                case 10:
                  for (; l < 32;) {
                    if (0 === o) break e;
                    o--, u += n[s++] << l, l += 8;
                  }

                  e.adler = r.check = L(u), l = u = 0, r.mode = 11;

                case 11:
                  if (0 === r.havedict) return e.next_out = a, e.avail_out = h, e.next_in = s, e.avail_in = o, r.hold = u, r.bits = l, 2;
                  e.adler = r.check = 1, r.mode = 12;

                case 12:
                  if (5 === t || 6 === t) break e;

                case 13:
                  if (r.last) {
                    u >>>= 7 & l, l -= 7 & l, r.mode = 27;
                    break;
                  }

                  for (; l < 3;) {
                    if (0 === o) break e;
                    o--, u += n[s++] << l, l += 8;
                  }

                  switch (r.last = 1 & u, l -= 1, 3 & (u >>>= 1)) {
                    case 0:
                      r.mode = 14;
                      break;

                    case 1:
                      if (j(r), r.mode = 20, 6 !== t) break;
                      u >>>= 2, l -= 2;
                      break e;

                    case 2:
                      r.mode = 17;
                      break;

                    case 3:
                      e.msg = "invalid block type", r.mode = 30;
                  }

                  u >>>= 2, l -= 2;
                  break;

                case 14:
                  for (u >>>= 7 & l, l -= 7 & l; l < 32;) {
                    if (0 === o) break e;
                    o--, u += n[s++] << l, l += 8;
                  }

                  if ((65535 & u) != (u >>> 16 ^ 65535)) {
                    e.msg = "invalid stored block lengths", r.mode = 30;
                    break;
                  }

                  if (r.length = 65535 & u, l = u = 0, r.mode = 15, 6 === t) break e;

                case 15:
                  r.mode = 16;

                case 16:
                  if (d = r.length) {
                    if (o < d && (d = o), h < d && (d = h), 0 === d) break e;
                    I.arraySet(i, n, s, d, a), o -= d, s += d, h -= d, a += d, r.length -= d;
                    break;
                  }

                  r.mode = 12;
                  break;

                case 17:
                  for (; l < 14;) {
                    if (0 === o) break e;
                    o--, u += n[s++] << l, l += 8;
                  }

                  if (r.nlen = 257 + (31 & u), u >>>= 5, l -= 5, r.ndist = 1 + (31 & u), u >>>= 5, l -= 5, r.ncode = 4 + (15 & u), u >>>= 4, l -= 4, 286 < r.nlen || 30 < r.ndist) {
                    e.msg = "too many length or distance symbols", r.mode = 30;
                    break;
                  }

                  r.have = 0, r.mode = 18;

                case 18:
                  for (; r.have < r.ncode;) {
                    for (; l < 3;) {
                      if (0 === o) break e;
                      o--, u += n[s++] << l, l += 8;
                    }

                    r.lens[A[r.have++]] = 7 & u, u >>>= 3, l -= 3;
                  }

                  for (; r.have < 19;) r.lens[A[r.have++]] = 0;

                  if (r.lencode = r.lendyn, r.lenbits = 7, S = {
                    bits: r.lenbits
                  }, x = T(0, r.lens, 0, 19, r.lencode, 0, r.work, S), r.lenbits = S.bits, x) {
                    e.msg = "invalid code lengths set", r.mode = 30;
                    break;
                  }

                  r.have = 0, r.mode = 19;

                case 19:
                  for (; r.have < r.nlen + r.ndist;) {
                    for (; g = (C = r.lencode[u & (1 << r.lenbits) - 1]) >>> 16 & 255, b = 65535 & C, !((_ = C >>> 24) <= l);) {
                      if (0 === o) break e;
                      o--, u += n[s++] << l, l += 8;
                    }

                    if (b < 16) u >>>= _, l -= _, r.lens[r.have++] = b;else {
                      if (16 === b) {
                        for (z = _ + 2; l < z;) {
                          if (0 === o) break e;
                          o--, u += n[s++] << l, l += 8;
                        }

                        if (u >>>= _, l -= _, 0 === r.have) {
                          e.msg = "invalid bit length repeat", r.mode = 30;
                          break;
                        }

                        k = r.lens[r.have - 1], d = 3 + (3 & u), u >>>= 2, l -= 2;
                      } else if (17 === b) {
                        for (z = _ + 3; l < z;) {
                          if (0 === o) break e;
                          o--, u += n[s++] << l, l += 8;
                        }

                        l -= _, k = 0, d = 3 + (7 & (u >>>= _)), u >>>= 3, l -= 3;
                      } else {
                        for (z = _ + 7; l < z;) {
                          if (0 === o) break e;
                          o--, u += n[s++] << l, l += 8;
                        }

                        l -= _, k = 0, d = 11 + (127 & (u >>>= _)), u >>>= 7, l -= 7;
                      }

                      if (r.have + d > r.nlen + r.ndist) {
                        e.msg = "invalid bit length repeat", r.mode = 30;
                        break;
                      }

                      for (; d--;) r.lens[r.have++] = k;
                    }
                  }

                  if (30 === r.mode) break;

                  if (0 === r.lens[256]) {
                    e.msg = "invalid code -- missing end-of-block", r.mode = 30;
                    break;
                  }

                  if (r.lenbits = 9, S = {
                    bits: r.lenbits
                  }, x = T(D, r.lens, 0, r.nlen, r.lencode, 0, r.work, S), r.lenbits = S.bits, x) {
                    e.msg = "invalid literal/lengths set", r.mode = 30;
                    break;
                  }

                  if (r.distbits = 6, r.distcode = r.distdyn, S = {
                    bits: r.distbits
                  }, x = T(F, r.lens, r.nlen, r.ndist, r.distcode, 0, r.work, S), r.distbits = S.bits, x) {
                    e.msg = "invalid distances set", r.mode = 30;
                    break;
                  }

                  if (r.mode = 20, 6 === t) break e;

                case 20:
                  r.mode = 21;

                case 21:
                  if (6 <= o && 258 <= h) {
                    e.next_out = a, e.avail_out = h, e.next_in = s, e.avail_in = o, r.hold = u, r.bits = l, R(e, c), a = e.next_out, i = e.output, h = e.avail_out, s = e.next_in, n = e.input, o = e.avail_in, u = r.hold, l = r.bits, 12 === r.mode && (r.back = -1);
                    break;
                  }

                  for (r.back = 0; g = (C = r.lencode[u & (1 << r.lenbits) - 1]) >>> 16 & 255, b = 65535 & C, !((_ = C >>> 24) <= l);) {
                    if (0 === o) break e;
                    o--, u += n[s++] << l, l += 8;
                  }

                  if (g && 0 == (240 & g)) {
                    for (v = _, y = g, w = b; g = (C = r.lencode[w + ((u & (1 << v + y) - 1) >> v)]) >>> 16 & 255, b = 65535 & C, !(v + (_ = C >>> 24) <= l);) {
                      if (0 === o) break e;
                      o--, u += n[s++] << l, l += 8;
                    }

                    u >>>= v, l -= v, r.back += v;
                  }

                  if (u >>>= _, l -= _, r.back += _, r.length = b, 0 === g) {
                    r.mode = 26;
                    break;
                  }

                  if (32 & g) {
                    r.back = -1, r.mode = 12;
                    break;
                  }

                  if (64 & g) {
                    e.msg = "invalid literal/length code", r.mode = 30;
                    break;
                  }

                  r.extra = 15 & g, r.mode = 22;

                case 22:
                  if (r.extra) {
                    for (z = r.extra; l < z;) {
                      if (0 === o) break e;
                      o--, u += n[s++] << l, l += 8;
                    }

                    r.length += u & (1 << r.extra) - 1, u >>>= r.extra, l -= r.extra, r.back += r.extra;
                  }

                  r.was = r.length, r.mode = 23;

                case 23:
                  for (; g = (C = r.distcode[u & (1 << r.distbits) - 1]) >>> 16 & 255, b = 65535 & C, !((_ = C >>> 24) <= l);) {
                    if (0 === o) break e;
                    o--, u += n[s++] << l, l += 8;
                  }

                  if (0 == (240 & g)) {
                    for (v = _, y = g, w = b; g = (C = r.distcode[w + ((u & (1 << v + y) - 1) >> v)]) >>> 16 & 255, b = 65535 & C, !(v + (_ = C >>> 24) <= l);) {
                      if (0 === o) break e;
                      o--, u += n[s++] << l, l += 8;
                    }

                    u >>>= v, l -= v, r.back += v;
                  }

                  if (u >>>= _, l -= _, r.back += _, 64 & g) {
                    e.msg = "invalid distance code", r.mode = 30;
                    break;
                  }

                  r.offset = b, r.extra = 15 & g, r.mode = 24;

                case 24:
                  if (r.extra) {
                    for (z = r.extra; l < z;) {
                      if (0 === o) break e;
                      o--, u += n[s++] << l, l += 8;
                    }

                    r.offset += u & (1 << r.extra) - 1, u >>>= r.extra, l -= r.extra, r.back += r.extra;
                  }

                  if (r.offset > r.dmax) {
                    e.msg = "invalid distance too far back", r.mode = 30;
                    break;
                  }

                  r.mode = 25;

                case 25:
                  if (0 === h) break e;

                  if (d = c - h, r.offset > d) {
                    if ((d = r.offset - d) > r.whave && r.sane) {
                      e.msg = "invalid distance too far back", r.mode = 30;
                      break;
                    }

                    p = d > r.wnext ? (d -= r.wnext, r.wsize - d) : r.wnext - d, d > r.length && (d = r.length), m = r.window;
                  } else m = i, p = a - r.offset, d = r.length;

                  for (h < d && (d = h), h -= d, r.length -= d; i[a++] = m[p++], --d;);

                  0 === r.length && (r.mode = 21);
                  break;

                case 26:
                  if (0 === h) break e;
                  i[a++] = r.length, h--, r.mode = 21;
                  break;

                case 27:
                  if (r.wrap) {
                    for (; l < 32;) {
                      if (0 === o) break e;
                      o--, u |= n[s++] << l, l += 8;
                    }

                    if (c -= h, e.total_out += c, r.total += c, c && (e.adler = r.check = r.flags ? B(r.check, i, c, a - c) : O(r.check, i, c, a - c)), c = h, (r.flags ? u : L(u)) !== r.check) {
                      e.msg = "incorrect data check", r.mode = 30;
                      break;
                    }

                    l = u = 0;
                  }

                  r.mode = 28;

                case 28:
                  if (r.wrap && r.flags) {
                    for (; l < 32;) {
                      if (0 === o) break e;
                      o--, u += n[s++] << l, l += 8;
                    }

                    if (u !== (4294967295 & r.total)) {
                      e.msg = "incorrect length check", r.mode = 30;
                      break;
                    }

                    l = u = 0;
                  }

                  r.mode = 29;

                case 29:
                  x = 1;
                  break e;

                case 30:
                  x = -3;
                  break e;

                case 31:
                  return -4;

                case 32:
                default:
                  return U;
              }

              return e.next_out = a, e.avail_out = h, e.next_in = s, e.avail_in = o, r.hold = u, r.bits = l, (r.wsize || c !== e.avail_out && r.mode < 30 && (r.mode < 27 || 4 !== t)) && Z(e, e.output, e.next_out, c - e.avail_out) ? (r.mode = 31, -4) : (f -= e.avail_in, c -= e.avail_out, e.total_in += f, e.total_out += c, r.total += c, r.wrap && c && (e.adler = r.check = r.flags ? B(r.check, i, c, e.next_out - c) : O(r.check, i, c, e.next_out - c)), e.data_type = r.bits + (r.last ? 64 : 0) + (12 === r.mode ? 128 : 0) + (20 === r.mode || 15 === r.mode ? 256 : 0), (0 == f && 0 === c || 4 === t) && x === N && (x = -5), x);
            }, r.inflateEnd = function (e) {
              if (!e || !e.state) return U;
              var t = e.state;
              return t.window && (t.window = null), e.state = null, N;
            }, r.inflateGetHeader = function (e, t) {
              var r;
              return e && e.state ? 0 == (2 & (r = e.state).wrap) ? U : ((r.head = t).done = !1, N) : U;
            }, r.inflateSetDictionary = function (e, t) {
              var r,
                  n = t.length;
              return e && e.state ? 0 !== (r = e.state).wrap && 11 !== r.mode ? U : 11 === r.mode && O(1, t, n, 0) !== r.check ? -3 : Z(e, t, n, n) ? (r.mode = 31, -4) : (r.havedict = 1, N) : U;
            }, r.inflateInfo = "pako inflate (from Nodeca project)";
          }, {
            "../utils/common": 41,
            "./adler32": 43,
            "./crc32": 45,
            "./inffast": 48,
            "./inftrees": 50
          }],
          50: [function (e, t, r) {

            var D = e("../utils/common"),
                F = [3, 4, 5, 6, 7, 8, 9, 10, 11, 13, 15, 17, 19, 23, 27, 31, 35, 43, 51, 59, 67, 83, 99, 115, 131, 163, 195, 227, 258, 0, 0],
                N = [16, 16, 16, 16, 16, 16, 16, 16, 17, 17, 17, 17, 18, 18, 18, 18, 19, 19, 19, 19, 20, 20, 20, 20, 21, 21, 21, 21, 16, 72, 78],
                U = [1, 2, 3, 4, 5, 7, 9, 13, 17, 25, 33, 49, 65, 97, 129, 193, 257, 385, 513, 769, 1025, 1537, 2049, 3073, 4097, 6145, 8193, 12289, 16385, 24577, 0, 0],
                P = [16, 16, 16, 16, 17, 17, 18, 18, 19, 19, 20, 20, 21, 21, 22, 22, 23, 23, 24, 24, 25, 25, 26, 26, 27, 27, 28, 28, 29, 29, 64, 64];

            t.exports = function (e, t, r, n, i, s, a, o) {
              var h,
                  u,
                  l,
                  f,
                  c,
                  d,
                  p,
                  m,
                  _,
                  g = o.bits,
                  b = 0,
                  v = 0,
                  y = 0,
                  w = 0,
                  k = 0,
                  x = 0,
                  S = 0,
                  z = 0,
                  C = 0,
                  E = 0,
                  A = null,
                  I = 0,
                  O = new D.Buf16(16),
                  B = new D.Buf16(16),
                  R = null,
                  T = 0;

              for (b = 0; b <= 15; b++) O[b] = 0;

              for (v = 0; v < n; v++) O[t[r + v]]++;

              for (k = g, w = 15; 1 <= w && 0 === O[w]; w--);

              if (w < k && (k = w), 0 === w) return i[s++] = 20971520, i[s++] = 20971520, o.bits = 1, 0;

              for (y = 1; y < w && 0 === O[y]; y++);

              for (k < y && (k = y), b = z = 1; b <= 15; b++) if (z <<= 1, (z -= O[b]) < 0) return -1;

              if (0 < z && (0 === e || 1 !== w)) return -1;

              for (B[1] = 0, b = 1; b < 15; b++) B[b + 1] = B[b] + O[b];

              for (v = 0; v < n; v++) 0 !== t[r + v] && (a[B[t[r + v]]++] = v);

              if (d = 0 === e ? (A = R = a, 19) : 1 === e ? (A = F, I -= 257, R = N, T -= 257, 256) : (A = U, R = P, -1), b = y, c = s, S = v = E = 0, l = -1, f = (C = 1 << (x = k)) - 1, 1 === e && 852 < C || 2 === e && 592 < C) return 1;

              for (;;) {
                for (p = b - S, _ = a[v] < d ? (m = 0, a[v]) : a[v] > d ? (m = R[T + a[v]], A[I + a[v]]) : (m = 96, 0), h = 1 << b - S, y = u = 1 << x; i[c + (E >> S) + (u -= h)] = p << 24 | m << 16 | _ | 0, 0 !== u;);

                for (h = 1 << b - 1; E & h;) h >>= 1;

                if (0 !== h ? (E &= h - 1, E += h) : E = 0, v++, 0 == --O[b]) {
                  if (b === w) break;
                  b = t[r + a[v]];
                }

                if (k < b && (E & f) !== l) {
                  for (0 === S && (S = k), c += y, z = 1 << (x = b - S); x + S < w && !((z -= O[x + S]) <= 0);) x++, z <<= 1;

                  if (C += 1 << x, 1 === e && 852 < C || 2 === e && 592 < C) return 1;
                  i[l = E & f] = k << 24 | x << 16 | c - s | 0;
                }
              }

              return 0 !== E && (i[c + E] = b - S << 24 | 64 << 16 | 0), o.bits = k, 0;
            };
          }, {
            "../utils/common": 41
          }],
          51: [function (e, t, r) {

            t.exports = {
              2: "need dictionary",
              1: "stream end",
              0: "",
              "-1": "file error",
              "-2": "stream error",
              "-3": "data error",
              "-4": "insufficient memory",
              "-5": "buffer error",
              "-6": "incompatible version"
            };
          }, {}],
          52: [function (e, t, r) {

            var i = e("../utils/common"),
                o = 0,
                h = 1;

            function n(e) {
              for (var t = e.length; 0 <= --t;) e[t] = 0;
            }

            var s = 0,
                a = 29,
                u = 256,
                l = u + 1 + a,
                f = 30,
                c = 19,
                _ = 2 * l + 1,
                g = 15,
                d = 16,
                p = 7,
                m = 256,
                b = 16,
                v = 17,
                y = 18,
                w = [0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 5, 5, 5, 5, 0],
                k = [0, 0, 0, 0, 1, 1, 2, 2, 3, 3, 4, 4, 5, 5, 6, 6, 7, 7, 8, 8, 9, 9, 10, 10, 11, 11, 12, 12, 13, 13],
                x = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 3, 7],
                S = [16, 17, 18, 0, 8, 7, 9, 6, 10, 5, 11, 4, 12, 3, 13, 2, 14, 1, 15],
                z = new Array(2 * (l + 2));

            n(z);
            var C = new Array(2 * f);
            n(C);
            var E = new Array(512);
            n(E);
            var A = new Array(256);
            n(A);
            var I = new Array(a);
            n(I);
            var O,
                B,
                R,
                T = new Array(f);

            function D(e, t, r, n, i) {
              this.static_tree = e, this.extra_bits = t, this.extra_base = r, this.elems = n, this.max_length = i, this.has_stree = e && e.length;
            }

            function F(e, t) {
              this.dyn_tree = e, this.max_code = 0, this.stat_desc = t;
            }

            function N(e) {
              return e < 256 ? E[e] : E[256 + (e >>> 7)];
            }

            function U(e, t) {
              e.pending_buf[e.pending++] = 255 & t, e.pending_buf[e.pending++] = t >>> 8 & 255;
            }

            function P(e, t, r) {
              e.bi_valid > d - r ? (e.bi_buf |= t << e.bi_valid & 65535, U(e, e.bi_buf), e.bi_buf = t >> d - e.bi_valid, e.bi_valid += r - d) : (e.bi_buf |= t << e.bi_valid & 65535, e.bi_valid += r);
            }

            function L(e, t, r) {
              P(e, r[2 * t], r[2 * t + 1]);
            }

            function j(e, t) {
              for (var r = 0; r |= 1 & e, e >>>= 1, r <<= 1, 0 < --t;);

              return r >>> 1;
            }

            function Z(e, t, r) {
              var n,
                  i,
                  s = new Array(g + 1),
                  a = 0;

              for (n = 1; n <= g; n++) s[n] = a = a + r[n - 1] << 1;

              for (i = 0; i <= t; i++) {
                var o = e[2 * i + 1];
                0 !== o && (e[2 * i] = j(s[o]++, o));
              }
            }

            function W(e) {
              var t;

              for (t = 0; t < l; t++) e.dyn_ltree[2 * t] = 0;

              for (t = 0; t < f; t++) e.dyn_dtree[2 * t] = 0;

              for (t = 0; t < c; t++) e.bl_tree[2 * t] = 0;

              e.dyn_ltree[2 * m] = 1, e.opt_len = e.static_len = 0, e.last_lit = e.matches = 0;
            }

            function M(e) {
              8 < e.bi_valid ? U(e, e.bi_buf) : 0 < e.bi_valid && (e.pending_buf[e.pending++] = e.bi_buf), e.bi_buf = 0, e.bi_valid = 0;
            }

            function H(e, t, r, n) {
              var i = 2 * t,
                  s = 2 * r;
              return e[i] < e[s] || e[i] === e[s] && n[t] <= n[r];
            }

            function G(e, t, r) {
              for (var n = e.heap[r], i = r << 1; i <= e.heap_len && (i < e.heap_len && H(t, e.heap[i + 1], e.heap[i], e.depth) && i++, !H(t, n, e.heap[i], e.depth));) e.heap[r] = e.heap[i], r = i, i <<= 1;

              e.heap[r] = n;
            }

            function K(e, t, r) {
              var n,
                  i,
                  s,
                  a,
                  o = 0;
              if (0 !== e.last_lit) for (; n = e.pending_buf[e.d_buf + 2 * o] << 8 | e.pending_buf[e.d_buf + 2 * o + 1], i = e.pending_buf[e.l_buf + o], o++, 0 === n ? L(e, i, t) : (L(e, (s = A[i]) + u + 1, t), 0 !== (a = w[s]) && P(e, i -= I[s], a), L(e, s = N(--n), r), 0 !== (a = k[s]) && P(e, n -= T[s], a)), o < e.last_lit;);
              L(e, m, t);
            }

            function Y(e, t) {
              var r,
                  n,
                  i,
                  s = t.dyn_tree,
                  a = t.stat_desc.static_tree,
                  o = t.stat_desc.has_stree,
                  h = t.stat_desc.elems,
                  u = -1;

              for (e.heap_len = 0, e.heap_max = _, r = 0; r < h; r++) 0 !== s[2 * r] ? (e.heap[++e.heap_len] = u = r, e.depth[r] = 0) : s[2 * r + 1] = 0;

              for (; e.heap_len < 2;) s[2 * (i = e.heap[++e.heap_len] = u < 2 ? ++u : 0)] = 1, e.depth[i] = 0, e.opt_len--, o && (e.static_len -= a[2 * i + 1]);

              for (t.max_code = u, r = e.heap_len >> 1; 1 <= r; r--) G(e, s, r);

              for (i = h; r = e.heap[1], e.heap[1] = e.heap[e.heap_len--], G(e, s, 1), n = e.heap[1], e.heap[--e.heap_max] = r, e.heap[--e.heap_max] = n, s[2 * i] = s[2 * r] + s[2 * n], e.depth[i] = (e.depth[r] >= e.depth[n] ? e.depth[r] : e.depth[n]) + 1, s[2 * r + 1] = s[2 * n + 1] = i, e.heap[1] = i++, G(e, s, 1), 2 <= e.heap_len;);

              e.heap[--e.heap_max] = e.heap[1], function (e, t) {
                var r,
                    n,
                    i,
                    s,
                    a,
                    o,
                    h = t.dyn_tree,
                    u = t.max_code,
                    l = t.stat_desc.static_tree,
                    f = t.stat_desc.has_stree,
                    c = t.stat_desc.extra_bits,
                    d = t.stat_desc.extra_base,
                    p = t.stat_desc.max_length,
                    m = 0;

                for (s = 0; s <= g; s++) e.bl_count[s] = 0;

                for (h[2 * e.heap[e.heap_max] + 1] = 0, r = e.heap_max + 1; r < _; r++) p < (s = h[2 * h[2 * (n = e.heap[r]) + 1] + 1] + 1) && (s = p, m++), h[2 * n + 1] = s, u < n || (e.bl_count[s]++, a = 0, d <= n && (a = c[n - d]), o = h[2 * n], e.opt_len += o * (s + a), f && (e.static_len += o * (l[2 * n + 1] + a)));

                if (0 !== m) {
                  do {
                    for (s = p - 1; 0 === e.bl_count[s];) s--;

                    e.bl_count[s]--, e.bl_count[s + 1] += 2, e.bl_count[p]--, m -= 2;
                  } while (0 < m);

                  for (s = p; 0 !== s; s--) for (n = e.bl_count[s]; 0 !== n;) u < (i = e.heap[--r]) || (h[2 * i + 1] !== s && (e.opt_len += (s - h[2 * i + 1]) * h[2 * i], h[2 * i + 1] = s), n--);
                }
              }(e, t), Z(s, u, e.bl_count);
            }

            function X(e, t, r) {
              var n,
                  i,
                  s = -1,
                  a = t[1],
                  o = 0,
                  h = 7,
                  u = 4;

              for (0 === a && (h = 138, u = 3), t[2 * (r + 1) + 1] = 65535, n = 0; n <= r; n++) i = a, a = t[2 * (n + 1) + 1], ++o < h && i === a || (o < u ? e.bl_tree[2 * i] += o : 0 !== i ? (i !== s && e.bl_tree[2 * i]++, e.bl_tree[2 * b]++) : o <= 10 ? e.bl_tree[2 * v]++ : e.bl_tree[2 * y]++, s = i, u = (o = 0) === a ? (h = 138, 3) : i === a ? (h = 6, 3) : (h = 7, 4));
            }

            function V(e, t, r) {
              var n,
                  i,
                  s = -1,
                  a = t[1],
                  o = 0,
                  h = 7,
                  u = 4;

              for (0 === a && (h = 138, u = 3), n = 0; n <= r; n++) if (i = a, a = t[2 * (n + 1) + 1], !(++o < h && i === a)) {
                if (o < u) for (; L(e, i, e.bl_tree), 0 != --o;);else 0 !== i ? (i !== s && (L(e, i, e.bl_tree), o--), L(e, b, e.bl_tree), P(e, o - 3, 2)) : o <= 10 ? (L(e, v, e.bl_tree), P(e, o - 3, 3)) : (L(e, y, e.bl_tree), P(e, o - 11, 7));
                s = i, u = (o = 0) === a ? (h = 138, 3) : i === a ? (h = 6, 3) : (h = 7, 4);
              }
            }

            n(T);
            var q = !1;

            function J(e, t, r, n) {
              P(e, (s << 1) + (n ? 1 : 0), 3), function (e, t, r, n) {
                M(e), n && (U(e, r), U(e, ~r)), i.arraySet(e.pending_buf, e.window, t, r, e.pending), e.pending += r;
              }(e, t, r, !0);
            }

            r._tr_init = function (e) {
              q || (function () {
                var e,
                    t,
                    r,
                    n,
                    i,
                    s = new Array(g + 1);

                for (n = r = 0; n < a - 1; n++) for (I[n] = r, e = 0; e < 1 << w[n]; e++) A[r++] = n;

                for (A[r - 1] = n, n = i = 0; n < 16; n++) for (T[n] = i, e = 0; e < 1 << k[n]; e++) E[i++] = n;

                for (i >>= 7; n < f; n++) for (T[n] = i << 7, e = 0; e < 1 << k[n] - 7; e++) E[256 + i++] = n;

                for (t = 0; t <= g; t++) s[t] = 0;

                for (e = 0; e <= 143;) z[2 * e + 1] = 8, e++, s[8]++;

                for (; e <= 255;) z[2 * e + 1] = 9, e++, s[9]++;

                for (; e <= 279;) z[2 * e + 1] = 7, e++, s[7]++;

                for (; e <= 287;) z[2 * e + 1] = 8, e++, s[8]++;

                for (Z(z, l + 1, s), e = 0; e < f; e++) C[2 * e + 1] = 5, C[2 * e] = j(e, 5);

                O = new D(z, w, u + 1, l, g), B = new D(C, k, 0, f, g), R = new D(new Array(0), x, 0, c, p);
              }(), q = !0), e.l_desc = new F(e.dyn_ltree, O), e.d_desc = new F(e.dyn_dtree, B), e.bl_desc = new F(e.bl_tree, R), e.bi_buf = 0, e.bi_valid = 0, W(e);
            }, r._tr_stored_block = J, r._tr_flush_block = function (e, t, r, n) {
              var i,
                  s,
                  a = 0;
              0 < e.level ? (2 === e.strm.data_type && (e.strm.data_type = function (e) {
                var t,
                    r = 4093624447;

                for (t = 0; t <= 31; t++, r >>>= 1) if (1 & r && 0 !== e.dyn_ltree[2 * t]) return o;

                if (0 !== e.dyn_ltree[18] || 0 !== e.dyn_ltree[20] || 0 !== e.dyn_ltree[26]) return h;

                for (t = 32; t < u; t++) if (0 !== e.dyn_ltree[2 * t]) return h;

                return o;
              }(e)), Y(e, e.l_desc), Y(e, e.d_desc), a = function (e) {
                var t;

                for (X(e, e.dyn_ltree, e.l_desc.max_code), X(e, e.dyn_dtree, e.d_desc.max_code), Y(e, e.bl_desc), t = c - 1; 3 <= t && 0 === e.bl_tree[2 * S[t] + 1]; t--);

                return e.opt_len += 3 * (t + 1) + 5 + 5 + 4, t;
              }(e), i = e.opt_len + 3 + 7 >>> 3, (s = e.static_len + 3 + 7 >>> 3) <= i && (i = s)) : i = s = r + 5, r + 4 <= i && -1 !== t ? J(e, t, r, n) : 4 === e.strategy || s === i ? (P(e, 2 + (n ? 1 : 0), 3), K(e, z, C)) : (P(e, 4 + (n ? 1 : 0), 3), function (e, t, r, n) {
                var i;

                for (P(e, t - 257, 5), P(e, r - 1, 5), P(e, n - 4, 4), i = 0; i < n; i++) P(e, e.bl_tree[2 * S[i] + 1], 3);

                V(e, e.dyn_ltree, t - 1), V(e, e.dyn_dtree, r - 1);
              }(e, e.l_desc.max_code + 1, e.d_desc.max_code + 1, a + 1), K(e, e.dyn_ltree, e.dyn_dtree)), W(e), n && M(e);
            }, r._tr_tally = function (e, t, r) {
              return e.pending_buf[e.d_buf + 2 * e.last_lit] = t >>> 8 & 255, e.pending_buf[e.d_buf + 2 * e.last_lit + 1] = 255 & t, e.pending_buf[e.l_buf + e.last_lit] = 255 & r, e.last_lit++, 0 === t ? e.dyn_ltree[2 * r]++ : (e.matches++, t--, e.dyn_ltree[2 * (A[r] + u + 1)]++, e.dyn_dtree[2 * N(t)]++), e.last_lit === e.lit_bufsize - 1;
            }, r._tr_align = function (e) {
              P(e, 2, 3), L(e, m, z), function (e) {
                16 === e.bi_valid ? (U(e, e.bi_buf), e.bi_buf = 0, e.bi_valid = 0) : 8 <= e.bi_valid && (e.pending_buf[e.pending++] = 255 & e.bi_buf, e.bi_buf >>= 8, e.bi_valid -= 8);
              }(e);
            };
          }, {
            "../utils/common": 41
          }],
          53: [function (e, t, r) {

            t.exports = function () {
              this.input = null, this.next_in = 0, this.avail_in = 0, this.total_in = 0, this.output = null, this.next_out = 0, this.avail_out = 0, this.total_out = 0, this.msg = "", this.state = null, this.data_type = 2, this.adler = 0;
            };
          }, {}],
          54: [function (e, t, r) {
            (function (e) {
              !function (r, n) {

                if (!r.setImmediate) {
                  var i,
                      s,
                      t,
                      a,
                      o = 1,
                      h = {},
                      u = !1,
                      l = r.document,
                      e = Object.getPrototypeOf && Object.getPrototypeOf(r);
                  e = e && e.setTimeout ? e : r, i = "[object process]" === {}.toString.call(r.process) ? function (e) {
                    process.nextTick(function () {
                      c(e);
                    });
                  } : function () {
                    if (r.postMessage && !r.importScripts) {
                      var e = !0,
                          t = r.onmessage;
                      return r.onmessage = function () {
                        e = !1;
                      }, r.postMessage("", "*"), r.onmessage = t, e;
                    }
                  }() ? (a = "setImmediate$" + Math.random() + "$", r.addEventListener ? r.addEventListener("message", d, !1) : r.attachEvent("onmessage", d), function (e) {
                    r.postMessage(a + e, "*");
                  }) : r.MessageChannel ? ((t = new MessageChannel()).port1.onmessage = function (e) {
                    c(e.data);
                  }, function (e) {
                    t.port2.postMessage(e);
                  }) : l && "onreadystatechange" in l.createElement("script") ? (s = l.documentElement, function (e) {
                    var t = l.createElement("script");
                    t.onreadystatechange = function () {
                      c(e), t.onreadystatechange = null, s.removeChild(t), t = null;
                    }, s.appendChild(t);
                  }) : function (e) {
                    setTimeout(c, 0, e);
                  }, e.setImmediate = function (e) {
                    "function" != typeof e && (e = new Function("" + e));

                    for (var t = new Array(arguments.length - 1), r = 0; r < t.length; r++) t[r] = arguments[r + 1];

                    var n = {
                      callback: e,
                      args: t
                    };
                    return h[o] = n, i(o), o++;
                  }, e.clearImmediate = f;
                }

                function f(e) {
                  delete h[e];
                }

                function c(e) {
                  if (u) setTimeout(c, 0, e);else {
                    var t = h[e];

                    if (t) {
                      u = !0;

                      try {
                        !function (e) {
                          var t = e.callback,
                              r = e.args;

                          switch (r.length) {
                            case 0:
                              t();
                              break;

                            case 1:
                              t(r[0]);
                              break;

                            case 2:
                              t(r[0], r[1]);
                              break;

                            case 3:
                              t(r[0], r[1], r[2]);
                              break;

                            default:
                              t.apply(n, r);
                          }
                        }(t);
                      } finally {
                        f(e), u = !1;
                      }
                    }
                  }
                }

                function d(e) {
                  e.source === r && "string" == typeof e.data && 0 === e.data.indexOf(a) && c(+e.data.slice(a.length));
                }
              }("undefined" == typeof self ? void 0 === e ? this : e : self);
            }).call(this, "undefined" != typeof commonjsGlobal ? commonjsGlobal : "undefined" != typeof self ? self : "undefined" != typeof window ? window : {});
          }, {}]
        }, {}, [10])(10);
      });
    })(jszip_min);

    var JSZip = jszip_min.exports;

    /**
     * Create a FileList from a zip
     * @param zipContent
     * @returns
     */

    async function fileListFromZip(zipContent) {
      const jsZip = new JSZip();
      const zip = await jsZip.loadAsync(zipContent);
      const fileList = [];

      for (let key in zip.files) {
        const entry = zip.files[key];
        if (entry.dir) continue;
        fileList.push({
          name: entry.name.replace(/^.*\//, ''),
          webkitRelativePath: entry.name,
          lastModified: entry.date.getTime(),
          // @ts-expect-error _data is not exposed because missing for folder but it is really there
          size: entry._data.uncompressedSize,
          text: () => {
            return entry.async('text');
          },
          arrayBuffer: () => {
            return entry.async('arraybuffer');
          }
        });
      }

      return fileList;
    }

    function convertToFloatArray$1(data) {
      if (isAnyArray(data[0])) {
        return data.map(e => Float64Array.from(e));
      } else if (isAnyArray(data)) {
        return Float64Array.from(data);
      } else if (typeof data === 'object') {
        let keys = Object.keys(data);

        for (let key of keys) {
          data[key] = convertToFloatArray$1(data[key]);
        }

        return data;
      }

      return data;
    }

    /* eslint-disable no-loss-of-precision */

    /**
     * Returns the group delay for old Bruker NMR spectra
     * @param {number} gspfvs
     * @param {number} decim
     * @return {number}
     */
    function getDigitalFilterParameters(grpdly, dspfvs, decim) {
      let value;

      if (grpdly > 0) {
        value = Number(grpdly);
      } else {
        if (dspfvs > 14) {
          value = 0;
        } else {
          if (!brukerDspTable[dspfvs]) {
            throw new Error('dspfvs not in lookup table');
          } else {
            const dspfvsList = brukerDspTable[dspfvs];
            if (!dspfvsList[decim]) throw new Error('decim not in lookup table');
            value = dspfvsList[decim];
          }
        }
      }

      return value;
    }
    const brukerDspTable = {
      10: {
        2: 44.75,
        3: 33.5,
        4: 66.625,
        6: 59.083333333333333,
        8: 68.5625,
        12: 60.375,
        16: 69.53125,
        24: 61.020833333333333,
        32: 70.015625,
        48: 61.34375,
        64: 70.2578125,
        96: 61.505208333333333,
        128: 70.37890625,
        192: 61.5859375,
        256: 70.439453125,
        384: 61.626302083333333,
        512: 70.4697265625,
        768: 61.646484375,
        1024: 70.48486328125,
        1536: 61.656575520833333,
        2048: 70.492431640625
      },
      11: {
        2: 46,
        3: 36.5,
        4: 48,
        6: 50.166666666666667,
        8: 53.25,
        12: 69.5,
        16: 72.25,
        24: 70.166666666666667,
        32: 72.75,
        48: 70.5,
        64: 73,
        96: 70.666666666666667,
        128: 72.5,
        192: 71.333333333333333,
        256: 72.25,
        384: 71.666666666666667,
        512: 72.125,
        768: 71.833333333333333,
        1024: 72.0625,
        1536: 71.916666666666667,
        2048: 72.03125
      },
      12: {
        2: 46,
        3: 36.5,
        4: 48,
        6: 50.166666666666667,
        8: 53.25,
        12: 69.5,
        16: 71.625,
        24: 70.166666666666667,
        32: 72.125,
        48: 70.5,
        64: 72.375,
        96: 70.666666666666667,
        128: 72.5,
        192: 71.333333333333333,
        256: 72.25,
        384: 71.666666666666667,
        512: 72.125,
        768: 71.833333333333333,
        1024: 72.0625,
        1536: 71.916666666666667,
        2048: 72.03125
      },
      13: {
        2: 2.75,
        3: 2.8333333333333333,
        4: 2.875,
        6: 2.9166666666666667,
        8: 2.9375,
        12: 2.9583333333333333,
        16: 2.96875,
        24: 2.9791666666666667,
        32: 2.984375,
        48: 2.9895833333333333,
        64: 2.9921875,
        96: 2.9947916666666667
      }
    };

    function getNucleusFromMetadata(metaData, info, subfix) {
      let nucleus = [];

      if (metaData[`${subfix}AXNUC`]) {
        nucleus = metaData[`${subfix}AXNUC`];
        if (!Array.isArray(nucleus)) nucleus = [nucleus];
        nucleus = checkForNucleus(nucleus);
      }

      if (nucleus.length < 1 && metaData[`${subfix}NUC1`]) {
        nucleus = metaData[`${subfix}NUC1`];
        if (!Array.isArray(nucleus)) nucleus = [nucleus];
        nucleus = checkForNucleus(nucleus);
      }

      if (nucleus.length === 0) {
        if (metaData['.NUCLEUS']) {
          nucleus = metaData['.NUCLEUS'].split(',').map(nuc => nuc.trim());
        } else if (metaData['.OBSERVENUCLEUS']) {
          nucleus = [metaData['.OBSERVENUCLEUS'].replace(/[^A-Za-z0-9]/g, '')];
        } else {
          nucleus = getNucleusFrom2DExperiment(info.experiment);
        }

        nucleus = checkForNucleus(nucleus);
      }

      if (metaData['2D_X_NUCLEUS'] && metaData['2D_Y_NUCLEUS']) {
        nucleus = [metaData['2D_X_NUCLEUS'].replace(/[^A-Za-z0-9]/g, ''), metaData['2D_Y_NUCLEUS'].replace(/[^A-Za-z0-9]/g, '')];
      }

      return nucleus;
    }
    /**
     * Returns a list of likely nuclei based on an experiment string
     * This is really an hypothesis and should not be used
     * @param {string} experiment
     * @return {string[]}
     */

    function getNucleusFrom2DExperiment(experiment) {
      if (typeof experiment !== 'string') {
        return [];
      }

      experiment = experiment.toLowerCase();

      if (experiment.includes('jres')) {
        return ['1H', 'Hz'];
      }

      if (experiment.includes('hmbc') || experiment.includes('hsqc')) {
        return ['1H', '13C'];
      }

      if (experiment.includes('cosy') || experiment.includes('tocsy')) {
        return ['1H', '1H'];
      }

      return [];
    }

    function checkForNucleus(nucleus) {
      nucleus = nucleus.map(value => value.replace(/[^A-Za-z0-9]/g, '').replace('NA', '').replace('off', ''));
      let beforeLength = nucleus.length;
      nucleus = nucleus.filter(value => value);
      return nucleus.length !== beforeLength ? [] : nucleus;
    }

    /**
     * Returns an experiment string based on a pulse sequence
     * @param {string} pulse
     * @return {string}
     */
    function getSpectrumType() {
      let meta = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
      let info = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
      const {
        subfix = ''
      } = options;
      if (meta === null) meta = {};
      if (typeof meta === 'string') meta = {
        pulseSequence: meta
      };
      let spectyp = info[`${subfix}SPECTYP`];
      spectyp = (Array.isArray(spectyp) ? spectyp[0] : spectyp || '').replace(/^<(.*)>$/, '$1') // eslint-disable-line prefer-named-capture-group
      .toLowerCase();
      if (spectyp.length > 0 && spectyp !== 'undefined') return spectyp;
      let pulse = Array.isArray(meta.pulseSequence) ? meta.pulseSequence[0] : meta.pulseSequence || '';

      if (typeof pulse !== 'string') {
        return meta.dimension ? `${meta.dimension}d` : '';
      }

      pulse = pulse.toLowerCase();

      if (pulse.includes('zg') || pulse.includes('single_pulse_dec') || pulse.includes('udeft')) {
        return '1d';
      }

      if (pulse.includes('hsqct') || pulse.includes('invi') && (pulse.includes('ml') || pulse.includes('di'))) {
        return 'hsqctocsy';
      }

      if (pulse.includes('hsqc') || pulse.includes('invi')) {
        return 'hsqc';
      }

      if (pulse.includes('hmbc') || pulse.includes('inv4') && pulse.includes('lp')) {
        return 'hmbc';
      }

      if (pulse.includes('hmqc')) {
        return 'hmqc';
      }

      if (pulse.includes('cosy')) {
        return 'cosy';
      }

      if (pulse.includes('jres')) {
        return 'jres';
      }

      if (pulse.includes('tocsy') || pulse.includes('mlev') || pulse.includes('dipsi')) {
        return 'tocsy';
      }

      if (pulse.includes('noesy')) {
        return 'noesy';
      }

      if (pulse.includes('roesy')) {
        return 'roesy';
      }

      if (pulse.includes('dept')) {
        return 'dept';
      }

      if (pulse.includes('jmod') || pulse.includes('apt')) {
        return 'aptjmod';
      }

      if (pulse.includes('inad')) {
        return 'inadequate';
      }

      if (pulse.includes('adeq')) {
        return 'adequate';
      }

      return meta.dimension ? `${meta.dimension}d` : '';
    }

    function getInfoFromJCAMP(metaData) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        subfix = ''
      } = options;
      const info = {
        dimension: 0,
        nucleus: [],
        isFid: false,
        isFt: false,
        isComplex: false
      };
      let metadataString = JSON.stringify(metaData);
      const separator = metadataString.match('\r\n') ? '\r\n' : '\n';
      let {
        JCAMPDX: jcampdx = '',
        ORIGIN: origin = ''
      } = metaData;
      let creator = String(jcampdx).toLowerCase() + origin.toLowerCase();

      if (creator.includes('mestre') || creator.includes('nova')) {
        creator = 'mnova';
      }

      if (creator === 'mnova') {
        if (metaData.LONGDATE) {
          info.date = metaData.LONGDATE;
        }
      }

      info.nucleus = getNucleusFromMetadata(metaData, info, subfix);
      info.dimension = info.nucleus.length;
      maybeAdd(info, 'title', metaData.TITLE);
      maybeAdd(info, 'solvent', metaData['.SOLVENTNAME']);
      maybeAdd(info, 'temperature', metaData[`${subfix}TE`] || metaData['.TE']);
      maybeAdd(info, 'type', metaData.DATATYPE);

      if (info.type) {
        let typeLowerCase = info.type[0].toUpperCase();

        if (typeLowerCase.indexOf('FID') >= 0) {
          info.isFid = true;
          info.isComplex = true;
        } else if (typeLowerCase.indexOf('SPECTRUM') >= 0) {
          info.isFt = true;
          info.isComplex = true;
        }
      }

      maybeAdd(info, 'pulseSequence', metaData['.PULSESEQUENCE'] || metaData['.PULPROG'] || metaData[`${subfix}PULPROG`]);
      maybeAdd(info, 'experiment', getSpectrumType(info, metaData, {
        subfix
      }));
      maybeAddNumber(info, 'originFrequency', metaData['.OBSERVEFREQUENCY']);

      if (creator !== 'mnova' && creator !== 'mestre') {
        const gyromagneticRatioConst = gyromagneticRatio$1[info.nucleus[0]];
        maybeAdd(info, 'probeName', metaData[`${subfix}PROBHD`]);
        maybeAdd(info, 'originFrequency', metaData[`${subfix}SFO1`]);
        maybeAdd(info, 'baseFrequency', metaData[`${subfix}BF1`]);

        if (!('baseFrequency' in info) && 'originFrequency' in info) {
          maybeAdd(info, 'baseFrequency', info.originFrequency);
        }

        if (!['baseFrequency', 'originFrequency'].some(e => !info[e])) {
          const {
            baseFrequency,
            originFrequency
          } = info;
          let fieldStrength = 2 * Math.PI * (baseFrequency[0] / gyromagneticRatioConst) * 1e6;
          let frequencyOffset = baseFrequency.map((bf, i) => (originFrequency[i] - bf) * 1e6);
          maybeAdd(info, 'fieldStrength', fieldStrength);
          maybeAdd(info, 'frequencyOffset', frequencyOffset);
        }

        maybeAddNumber(info, 'spectralWidth', metaData[`${subfix}SW`]);
        maybeAddNumber(info, 'spectralWidth', metaData[`${subfix}QM_SPECTRAL_WIDTH`]);
        maybeAdd(info, 'numberOfPoints', metaData[`${subfix}TD`]);
        const numberOfPoints = info.numberOfPoints;
        maybeAdd(info, 'sampleName', metaData[`${subfix}NAME`]);

        if (metaData[`${subfix}FNTYPE`] !== undefined) {
          maybeAdd(info, 'acquisitionMode', parseInt(metaData[`${subfix}FNTYPE`], 10));
        }

        let varName = metaData[`${subfix}VARNAME`] ? metaData[`${subfix}VARNAME`].split(',')[0] : '';

        if (varName === 'TIME') {
          let value = typeof metaData.LAST === 'string' || metaData.LAST instanceof String ? metaData.LAST.replace(' ', '').split(',')[0] : metaData.LAST;
          maybeAddNumber(info, 'acquisitionTime', value);
        }

        if (!info.acquisitionTime) {
          if (!['numberOfPoints', 'spectralWidth'].some(e => !info[e])) {
            const {
              spectralWidth,
              originFrequency
            } = info;
            maybeAdd(info, 'acquisitionTime', Number((numberOfPoints[0] - 1) / (2 * spectralWidth[0] * originFrequency[0])));
          }
        }

        if (metaData[`${subfix}P`]) {
          let pulseStrength = 1e6 / (metaData[`${subfix}P`].split(separator)[1].split(' ')[1] * 4);
          maybeAdd(info, 'pulseStrength90', pulseStrength);
        }

        if (metaData[`${subfix}D`]) {
          let relaxationTime = metaData[`${subfix}D`].split(separator)[1].split(' ')[1];
          maybeAddNumber(info, 'relaxationTime', relaxationTime);
        }

        maybeAddNumber(info, 'numberOfScans', metaData[`${subfix}NS`]);
        maybeAddNumber(info, 'numberOfScans', metaData[`${subfix}QM_NSCANS`]);
        let increment;

        if (!['numberOfPoints', 'spectralWidth'].some(e => !info[e])) {
          const {
            spectralWidth,
            numberOfPoints
          } = info;

          if (info.isFid) {
            maybeAdd(info, 'groupDelay', metaData[`${subfix}GRPDLY`] || 0);
            maybeAdd(info, 'DSPFVS', metaData[`${subfix}DSPFVS`]);
            maybeAdd(info, 'DECIM', metaData[`${subfix}DECIM`]);

            if (!['groupDelay', 'DSPFVS', 'DECIM'].some(e => !info[e])) {
              let {
                groupDelay,
                DSPFVS,
                DECIM
              } = info;
              let digitalFilterParameters = getDigitalFilterParameters(groupDelay[0], DSPFVS[0], DECIM[0]);
              maybeAdd(info, 'digitalFilter', digitalFilterParameters);
            }

            increment = numberOfPoints.map(nb => {
              return info.acquisitionTime[0] / (nb - 1);
            });
          } else {
            increment = numberOfPoints.map((nb, i) => {
              return spectralWidth[i] / (nb - 1);
            });
          }
        }

        maybeAdd(info, 'increment', increment);

        if (metaData[`${subfix}DATE`]) {
          info.date = new Date(parseInt(metaData[`${subfix}DATE`], 10) * 1000).toISOString();
        }

        if (!info.solvent) {
          maybeAdd(info, 'solvent', Array.isArray(metaData[`${subfix}SOLVENT`]) ? metaData[`${subfix}SOLVENT`][0] : metaData[`${subfix}SOLVENT`]);
        }
      }

      if (metaData.SYMBOL) {
        let symbols = metaData.SYMBOL.split(/[, ]+/);

        if (symbols.includes('R') && symbols.includes('I')) {
          info.isComplex = true;
        }
      }

      for (let key in info) {
        if (info[key].length === 1) info[key] = info[key][0];
      }

      if (!Array.isArray(info.nucleus)) info.nucleus = [info.nucleus];
      return info;
    }

    function maybeAddNumber(obj, name, value) {
      if (value === undefined) return;

      if (typeof value === 'string') {
        value = Number(value.replace(/\$.*/, ''));
      }

      maybeAdd(obj, name, value);
    }

    function maybeAdd(obj, name, value) {
      if (value === undefined) return;

      if (Array.isArray(value)) {
        obj[name] = value.map(cleanValue);
      } else {
        obj[name] = [cleanValue(value)];
      }
    }

    function cleanValue(value) {
      if (typeof value === 'string') {
        if (value.startsWith('<') && value.endsWith('>')) {
          value = value.substring(1, value.length - 1);
        }

        value = value.trim();
      }

      return value;
    }

    const defaultOptions$1 = {
      converter: {
        xy: true,
        noContour: true,
        keepRecordsRegExp: /.*/,
        profiling: true
      }
    };
    async function fromBruker(zipFile) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const fileList = await fileListFromZip(zipFile);
      let parseData = await convertFileList(fileList, { ...defaultOptions$1,
        ...options
      });
      let dataStructure = [];

      for (let entry of parseData) {
        let metadata = Object.assign({}, entry.info, entry.meta);
        let info = getInfoFromJCAMP(metadata);
        let dimensions = [];
        let dependentVariables = [];
        let dependentVariable = {};

        if (info.dimension === 1) {
          dependentVariable.components = entry.spectra;
        } else if (info.dimension === 2) {
          entry.minMax.z = convertToFloatArray$1(entry.minMax.z);
          dependentVariable.components = entry.minMax;
        }

        let dimension = {
          increment: info.increment,
          numberOfPoints: info.numberOfPoints
        };

        if (info.fid) {
          dimension.coordinatesOffset = {
            magnitude: -info.digitalFilter * info.increment,
            units: 'second'
          };
        } else {
          dimension.coordinatesOffset = {
            magnitude: info.frequencyOffset / info.baseFrequency - 0.5 * info.spectraWidth,
            units: 'ppm'
          };
        }

        dimensions.push(dimension);
        dependentVariables.push(dependentVariable);
        const {
          source
        } = entry;
        dataStructure.push({
          dimensions,
          dependentVariables,
          source,
          info: info,
          meta: metadata,
          timeStamp: new Date().valueOf(),
          version: packageJson.version
        });
      }

      return dataStructure;
    }

    const GC_MS_FIELDS = ['TIC', '.RIC', 'SCANNUMBER'];
    function complexChromatogram(result) {
      let spectra = result.spectra;
      let length = spectra.length;
      let chromatogram = {
        times: new Array(length),
        series: {
          ms: {
            dimension: 2,
            data: new Array(length)
          }
        }
      };
      let existingGCMSFields = [];

      for (let i = 0; i < GC_MS_FIELDS.length; i++) {
        let label = convertMSFieldToLabel(GC_MS_FIELDS[i]);

        if (spectra[0][label]) {
          existingGCMSFields.push(label);
          chromatogram.series[label] = {
            dimension: 1,
            data: new Array(length)
          };
        }
      }

      for (let i = 0; i < length; i++) {
        let spectrum = spectra[i];
        chromatogram.times[i] = spectrum.pageValue;

        for (let j = 0; j < existingGCMSFields.length; j++) {
          chromatogram.series[existingGCMSFields[j]].data[i] = Number(spectrum[existingGCMSFields[j]]);
        }

        if (spectrum.data) {
          chromatogram.series.ms.data[i] = [spectrum.data.x, spectrum.data.y];
        }
      }

      result.chromatogram = chromatogram;
    }
    function isMSField(canonicDataLabel) {
      return GC_MS_FIELDS.indexOf(canonicDataLabel) !== -1;
    }
    function convertMSFieldToLabel(value) {
      return value.toLowerCase().replace(/[^a-z0-9]/g, '');
    }

    function convertToFloatArray(stringArray) {
      let floatArray = [];

      for (let i = 0; i < stringArray.length; i++) {
        floatArray.push(Number(stringArray[i]));
      }

      return floatArray;
    }

    function fastParseXYData(spectrum, value) {
      // TODO need to deal with result
      //  console.log(value);
      // we check if deltaX is defined otherwise we calculate it
      let yFactor = spectrum.yFactor;
      let deltaX = spectrum.deltaX;
      spectrum.isXYdata = true;
      let currentData = {
        x: [],
        y: []
      };
      spectrum.data = currentData;
      let currentX = spectrum.firstX;
      let currentY = spectrum.firstY; // we skip the first line
      //

      let endLine = false;
      let ascii;
      let i = 0;

      for (; i < value.length; i++) {
        ascii = value.charCodeAt(i);

        if (ascii === 13 || ascii === 10) {
          endLine = true;
        } else {
          if (endLine) break;
        }
      } // we proceed taking the i after the first line


      let newLine = true;
      let isDifference = false;
      let isLastDifference = false;
      let lastDifference = 0;
      let isDuplicate = false;
      let inComment = false;
      let currentValue = 0; // can be a difference or a duplicate

      let lastValue = 0; // must be the real last value

      let isNegative = false;
      let inValue = false;
      let skipFirstValue = false;
      let decimalPosition = 0;

      for (; i <= value.length; i++) {
        if (i === value.length) ascii = 13;else ascii = value.charCodeAt(i);

        if (inComment) {
          // we should ignore the text if we are after $$
          if (ascii === 13 || ascii === 10) {
            newLine = true;
            inComment = false;
          }
        } else {
          // when is it a new value ?
          // when it is not a digit, . or comma
          // it is a number that is either new or we continue
          if (ascii <= 57 && ascii >= 48) {
            // a number
            inValue = true;

            if (decimalPosition > 0) {
              currentValue += (ascii - 48) / Math.pow(10, decimalPosition++);
            } else {
              currentValue *= 10;
              currentValue += ascii - 48;
            }
          } else if (ascii === 44 || ascii === 46) {
            // a "," or "."
            inValue = true;
            decimalPosition++;
          } else {
            if (inValue) {
              // need to process the previous value
              if (newLine) {
                newLine = false; // we don't check the X value
                // console.log("NEW LINE",isDifference, lastDifference);
                // if new line and lastDifference, the first value is just a check !
                // that we don't check ...

                if (isLastDifference) skipFirstValue = true;
              } else {
                // need to deal with duplicate and differences
                if (skipFirstValue) {
                  skipFirstValue = false;
                } else {
                  if (isDifference) {
                    lastDifference = isNegative ? 0 - currentValue : currentValue;
                    isLastDifference = true;
                    isDifference = false;
                  } else if (!isDuplicate) {
                    lastValue = isNegative ? 0 - currentValue : currentValue;
                  }

                  let duplicate = isDuplicate ? currentValue - 1 : 1;

                  for (let j = 0; j < duplicate; j++) {
                    if (isLastDifference) {
                      currentY += lastDifference;
                    } else {
                      currentY = lastValue;
                    }

                    currentData.x.push(currentX);
                    currentData.y.push(currentY * yFactor);
                    currentX += deltaX;
                  }
                }
              }

              isNegative = false;
              currentValue = 0;
              decimalPosition = 0;
              inValue = false;
              isDuplicate = false;
            } // positive SQZ digits @ A B C D E F G H I (ascii 64-73)


            if (ascii < 74 && ascii > 63) {
              inValue = true;
              isLastDifference = false;
              currentValue = ascii - 64;
            } else if (ascii > 96 && ascii < 106) {
              // negative SQZ digits a b c d e f g h i (ascii 97-105)
              inValue = true;
              isLastDifference = false;
              currentValue = ascii - 96;
              isNegative = true;
            } else if (ascii === 115) {
              // DUP digits S T U V W X Y Z s (ascii 83-90, 115)
              inValue = true;
              isDuplicate = true;
              currentValue = 9;
            } else if (ascii > 82 && ascii < 91) {
              inValue = true;
              isDuplicate = true;
              currentValue = ascii - 82;
            } else if (ascii > 73 && ascii < 83) {
              // positive DIF digits % J K L M N O P Q R (ascii 37, 74-82)
              inValue = true;
              isDifference = true;
              currentValue = ascii - 73;
            } else if (ascii > 105 && ascii < 115) {
              // negative DIF digits j k l m n o p q r (ascii 106-114)
              inValue = true;
              isDifference = true;
              currentValue = ascii - 105;
              isNegative = true;
            } else if (ascii === 36 && value.charCodeAt(i + 1) === 36) {
              // $ sign, we need to check the next one
              inValue = true;
              inComment = true;
            } else if (ascii === 37) {
              // positive DIF digits % J K L M N O P Q R (ascii 37, 74-82)
              inValue = true;
              isDifference = true;
              currentValue = 0;
              isNegative = false;
            } else if (ascii === 45) {
              // a "-"
              // check if after there is a number, decimal or comma
              let ascii2 = value.charCodeAt(i + 1);

              if (ascii2 >= 48 && ascii2 <= 57 || ascii2 === 44 || ascii2 === 46) {
                inValue = true;
                if (!newLine) isLastDifference = false;
                isNegative = true;
              }
            } else if (ascii === 13 || ascii === 10) {
              newLine = true;
              inComment = false;
            } // and now analyse the details ... space or tabulation
            // if "+" we just don't care

          }
        }
      }
    }

    const removeCommentRegExp = /\$\$.*/;
    const peakTableSplitRegExp = /[,\t ]+/;
    function parsePeakTable(spectrum, value, result) {
      spectrum.isPeaktable = true;

      if (!spectrum.variables || Object.keys(spectrum.variables) === 2) {
        parseXY(spectrum, value, result);
      } else {
        parseXYZ(spectrum, value, result);
      } // we will add the data in the variables


      if (spectrum.variables) {
        for (let key in spectrum.variables) {
          spectrum.variables[key].data = spectrum.data[key];
        }
      }
    }

    function parseXY(spectrum, value, result) {
      let currentData = {
        x: [],
        y: []
      };
      spectrum.data = currentData; // counts for around 20% of the time

      let lines = value.split(/,? *,?[;\r\n]+ */);

      for (let i = 1; i < lines.length; i++) {
        let values = lines[i].trim().replace(removeCommentRegExp, '').split(peakTableSplitRegExp);

        if (values.length % 2 === 0) {
          for (let j = 0; j < values.length; j = j + 2) {
            // takes around 40% of the time to add and parse the 2 values nearly exclusively because of Number
            currentData.x.push(Number(values[j]) * spectrum.xFactor);
            currentData.y.push(Number(values[j + 1]) * spectrum.yFactor);
          }
        } else {
          result.logs.push(`Format error: ${values}`);
        }
      }
    }

    function parseXYZ(spectrum, value, result) {
      let currentData = {};
      let variables = Object.keys(spectrum.variables);
      let numberOfVariables = variables.length;
      variables.forEach(variable => currentData[variable] = []);
      spectrum.data = currentData; // counts for around 20% of the time

      let lines = value.split(/,? *,?[;\r\n]+ */);

      for (let i = 1; i < lines.length; i++) {
        let values = lines[i].trim().replace(removeCommentRegExp, '').split(peakTableSplitRegExp);

        if (values.length % numberOfVariables === 0) {
          for (let j = 0; j < values.length; j++) {
            // todo should try to find a xFactor (y, ...)
            currentData[variables[j % numberOfVariables]].push(Number(values[j]));
          }
        } else {
          result.logs.push(`Format error: ${values}`);
        }
      }
    }

    function parseXYA(spectrum, value) {
      let removeSymbolRegExp = /(\(+|\)+|<+|>+|\s+)/g;
      spectrum.isXYAdata = true;
      let values;
      let currentData = {
        x: [],
        y: []
      };
      spectrum.data = currentData;
      let lines = value.split(/,? *,?[;\r\n]+ */);

      for (let i = 1; i < lines.length; i++) {
        values = lines[i].trim().replace(removeSymbolRegExp, '').split(',');
        currentData.x.push(Number(values[0]));
        currentData.y.push(Number(values[1]));
      }
    }

    function convertTo3DZ(spectra) {
      let minZ = spectra[0].data.y[0];
      let maxZ = minZ;
      let ySize = spectra.length;
      let xSize = spectra[0].data.x.length;
      let z = new Array(ySize);

      for (let i = 0; i < ySize; i++) {
        z[i] = spectra[i].data.y;

        for (let j = 0; j < xSize; j++) {
          let value = z[i][j];
          if (value < minZ) minZ = value;
          if (value > maxZ) maxZ = value;
        }
      }

      const firstX = spectra[0].data.x[0];
      const lastX = spectra[0].data.x[spectra[0].data.x.length - 1]; // has to be -2 because it is a 1D array [x,y,x,y,...]

      const firstY = spectra[0].pageValue;
      const lastY = spectra[ySize - 1].pageValue; // Because the min / max value are the only information about the matrix if we invert
      // min and max we need to invert the array

      if (firstX > lastX) {
        for (let spectrum of z) {
          spectrum.reverse();
        }
      }

      if (firstY > lastY) {
        z.reverse();
      }

      const medians = [];

      for (let i = 0; i < z.length; i++) {
        const row = Float64Array.from(z[i]);

        for (let i = 0; i < row.length; i++) {
          if (row[i] < 0) row[i] = -row[i];
        }

        medians.push(median(row));
      }

      const median$1 = median(medians);
      return {
        z: z,
        minX: Math.min(firstX, lastX),
        maxX: Math.max(firstX, lastX),
        minY: Math.min(firstY, lastY),
        maxY: Math.max(firstY, lastY),
        minZ: minZ,
        maxZ: maxZ,
        noise: median$1
      };
    }

    function generateContourLines(zData, options) {
      let noise = zData.noise;
      let z = zData.z;
      let povarHeight0, povarHeight1, povarHeight2, povarHeight3;
      let isOver0, isOver1, isOver2, isOver3;
      let nbSubSpectra = z.length;
      let nbPovars = z[0].length;
      let pAx, pAy, pBx, pBy;
      let x0 = zData.minX;
      let xN = zData.maxX;
      let dx = (xN - x0) / (nbPovars - 1);
      let y0 = zData.minY;
      let yN = zData.maxY;
      let dy = (yN - y0) / (nbSubSpectra - 1);
      let minZ = zData.minZ;
      let maxZ = zData.maxZ; // System.out.prvarln('y0 '+y0+' yN '+yN);
      // -------------------------
      // Povars attribution
      //
      // 0----1
      // |  / |
      // | /  |
      // 2----3
      //
      // ---------------------d------

      let iter = options.nbContourLevels * 2;
      let contourLevels = new Array(iter);
      let lineZValue;

      for (let level = 0; level < iter; level++) {
        // multiply by 2 for positif and negatif
        let contourLevel = {};
        contourLevels[level] = contourLevel;
        let side = level % 2;
        let factor = (maxZ - options.noiseMultiplier * noise) * Math.exp((level >> 1) - options.nbContourLevels);

        if (side === 0) {
          lineZValue = factor + options.noiseMultiplier * noise;
        } else {
          lineZValue = 0 - factor - options.noiseMultiplier * noise;
        }

        let lines = [];
        contourLevel.zValue = lineZValue;
        contourLevel.lines = lines;
        if (lineZValue <= minZ || lineZValue >= maxZ) continue;

        for (let iSubSpectra = 0; iSubSpectra < nbSubSpectra - 1; iSubSpectra++) {
          let subSpectra = z[iSubSpectra];
          let subSpectraAfter = z[iSubSpectra + 1];

          for (let povar = 0; povar < nbPovars - 1; povar++) {
            povarHeight0 = subSpectra[povar];
            povarHeight1 = subSpectra[povar + 1];
            povarHeight2 = subSpectraAfter[povar];
            povarHeight3 = subSpectraAfter[povar + 1];
            isOver0 = povarHeight0 > lineZValue;
            isOver1 = povarHeight1 > lineZValue;
            isOver2 = povarHeight2 > lineZValue;
            isOver3 = povarHeight3 > lineZValue; // Example povar0 is over the plane and povar1 and
            // povar2 are below, we find the varersections and add
            // the segment

            if (isOver0 !== isOver1 && isOver0 !== isOver2) {
              pAx = povar + (lineZValue - povarHeight0) / (povarHeight1 - povarHeight0);
              pAy = iSubSpectra;
              pBx = povar;
              pBy = iSubSpectra + (lineZValue - povarHeight0) / (povarHeight2 - povarHeight0);
              lines.push(pAx * dx + x0);
              lines.push(pAy * dy + y0);
              lines.push(pBx * dx + x0);
              lines.push(pBy * dy + y0);
            } // remove push does not help !!!!


            if (isOver3 !== isOver1 && isOver3 !== isOver2) {
              pAx = povar + 1;
              pAy = iSubSpectra + 1 - (lineZValue - povarHeight3) / (povarHeight1 - povarHeight3);
              pBx = povar + 1 - (lineZValue - povarHeight3) / (povarHeight2 - povarHeight3);
              pBy = iSubSpectra + 1;
              lines.push(pAx * dx + x0);
              lines.push(pAy * dy + y0);
              lines.push(pBx * dx + x0);
              lines.push(pBy * dy + y0);
            } // test around the diagonal


            if (isOver1 !== isOver2) {
              pAx = (povar + 1 - (lineZValue - povarHeight1) / (povarHeight2 - povarHeight1)) * dx + x0;
              pAy = (iSubSpectra + (lineZValue - povarHeight1) / (povarHeight2 - povarHeight1)) * dy + y0;

              if (isOver1 !== isOver0) {
                pBx = povar + 1 - (lineZValue - povarHeight1) / (povarHeight0 - povarHeight1);
                pBy = iSubSpectra;
                lines.push(pAx);
                lines.push(pAy);
                lines.push(pBx * dx + x0);
                lines.push(pBy * dy + y0);
              }

              if (isOver2 !== isOver0) {
                pBx = povar;
                pBy = iSubSpectra + 1 - (lineZValue - povarHeight2) / (povarHeight0 - povarHeight2);
                lines.push(pAx);
                lines.push(pAy);
                lines.push(pBx * dx + x0);
                lines.push(pBy * dy + y0);
              }

              if (isOver1 !== isOver3) {
                pBx = povar + 1;
                pBy = iSubSpectra + (lineZValue - povarHeight1) / (povarHeight3 - povarHeight1);
                lines.push(pAx);
                lines.push(pAy);
                lines.push(pBx * dx + x0);
                lines.push(pBy * dy + y0);
              }

              if (isOver2 !== isOver3) {
                pBx = povar + (lineZValue - povarHeight2) / (povarHeight3 - povarHeight2);
                pBy = iSubSpectra + 1;
                lines.push(pAx);
                lines.push(pAy);
                lines.push(pBx * dx + x0);
                lines.push(pBy * dy + y0);
              }
            }
          }
        }
      }

      return {
        minX: zData.minX,
        maxX: zData.maxX,
        minY: zData.minY,
        maxY: zData.maxY,
        segments: contourLevels
      };
    }

    function add2D(result, options) {
      let zData = convertTo3DZ(result.spectra);

      if (!options.noContour) {
        result.contourLines = generateContourLines(zData, options);
        delete zData.z;
      }

      result.minMax = zData;
    }

    function postProcessingNMR(entriesFlat) {
      // specific NMR functions
      for (let entry of entriesFlat) {
        let observeFrequency = 0;
        let shiftOffsetVal = 0;

        for (let spectrum of entry.spectra) {
          if (entry.ntuples && entry.ntuples.symbol) {
            if (!observeFrequency && spectrum.observeFrequency) {
              observeFrequency = spectrum.observeFrequency;
            }

            if (!shiftOffsetVal && spectrum.shiftOffsetVal) {
              shiftOffsetVal = spectrum.shiftOffsetVal;
            }
          } else {
            observeFrequency = spectrum.observeFrequency;
            shiftOffsetVal = spectrum.shiftOffsetVal;
          }

          if (observeFrequency) {
            if (spectrum.xUnits && spectrum.xUnits.toUpperCase().includes('HZ')) {
              spectrum.xUnits = 'PPM';
              spectrum.xFactor = spectrum.xFactor / observeFrequency;
              spectrum.firstX = spectrum.firstX / observeFrequency;
              spectrum.lastX = spectrum.lastX / observeFrequency;
              spectrum.deltaX = spectrum.deltaX / observeFrequency;

              for (let i = 0; i < spectrum.data.x.length; i++) {
                spectrum.data.x[i] /= observeFrequency;
              }
            }
          }

          if (shiftOffsetVal) {
            let shift = spectrum.firstX - shiftOffsetVal;
            spectrum.firstX = spectrum.firstX - shift;
            spectrum.lastX = spectrum.lastX - shift;

            for (let i = 0; i < spectrum.data.x.length; i++) {
              spectrum.data.x[i] -= shift;
            }
          } // we will check if some nucleus are missing ...


          if (entry.ntuples && entry.ntuples.nucleus && entry.ntuples.symbol) {
            for (let i = 0; i < entry.ntuples.nucleus.length; i++) {
              let symbol = entry.ntuples.symbol[i];
              let nucleus = entry.ntuples.nucleus[i];

              if (symbol.startsWith('F') && !nucleus) {
                if (symbol === 'F1') {
                  // if F1 is defined we will use F2
                  if (entry.tmp.$NUC2) {
                    entry.ntuples.nucleus[i] = entry.tmp.$NUC2;
                  } else {
                    let f2index = entry.ntuples.symbol.indexOf('F2');

                    if (f2index && entry.ntuples.nucleus[f2index]) {
                      entry.ntuples.nucleus[i] = entry.ntuples.nucleus[f2index];
                    }
                  }
                }

                if (symbol === 'F2') entry.ntuples.nucleus[i] = entry.tmp.$NUC1;
              }

              if (symbol === 'F2') {
                entry.yType = entry.ntuples.nucleus[0];
              }
            }
          }

          if (observeFrequency && entry.ntuples && entry.ntuples.symbol && entry.ntuples.nucleus) {
            let unit = '';
            let pageSymbolIndex = entry.ntuples.symbol.indexOf(spectrum.pageSymbol);

            if (entry.ntuples.units && entry.ntuples.units[pageSymbolIndex]) {
              unit = entry.ntuples.units[pageSymbolIndex];
            }

            if (unit !== 'PPM') {
              if (pageSymbolIndex !== 0) {
                throw Error('Not sure about this ntuples format');
              }

              let ratio0 = gyromagneticRatio$1[entry.ntuples.nucleus[0]];
              let ratio1 = gyromagneticRatio$1[entry.ntuples.nucleus[1]];

              if (!ratio0 || !ratio1) {
                throw Error('Problem with determination of gyromagnetic ratio');
              }

              let ratio = ratio0 / ratio1 * observeFrequency;
              spectrum.pageValue /= ratio;
            }
          }
        }
      }
    }

    function profiling(result, action, options) {
      if (result.profiling) {
        result.profiling.push({
          action,
          time: Date.now() - options.start
        });
      }
    }

    function simpleChromatogram(result) {
      let data = result.spectra[0].data;
      result.chromatogram = {
        times: data.x.slice(),
        series: {
          intensity: {
            dimension: 1,
            data: data.y.slice()
          }
        }
      };
    }

    function postProcessing(entriesFlat, result, options) {
      // converting Hz to ppm
      postProcessingNMR(entriesFlat);

      for (let entry of entriesFlat) {
        if (Object.keys(entry.ntuples).length > 0) {
          let newNtuples = [];
          let keys = Object.keys(entry.ntuples);

          for (let i = 0; i < keys.length; i++) {
            let key = keys[i];
            let values = entry.ntuples[key];

            for (let j = 0; j < values.length; j++) {
              if (!newNtuples[j]) newNtuples[j] = {};
              newNtuples[j][key] = values[j];
            }
          }

          entry.ntuples = newNtuples;
        }

        if (entry.twoD && options.wantXY) {
          add2D(entry, options);
          profiling(result, 'Finished countour plot calculation', options);

          if (!options.keepSpectra) {
            delete entry.spectra;
          }
        } // maybe it is a GC (HPLC) / MS. In this case we add a new format


        if (options.chromatogram) {
          if (entry.spectra.length > 1) {
            complexChromatogram(entry);
          } else {
            simpleChromatogram(entry);
          }

          profiling(result, 'Finished chromatogram calculation', options);
        }

        delete entry.tmp;
      }
    }

    function prepareNtuplesDatatable(currentEntry, spectrum, kind) {
      let xIndex = -1;
      let yIndex = -1;
      let firstVariable = '';
      let secondVariable = '';

      if (kind.indexOf('++') > 0) {
        firstVariable = kind.replace(/.*\(([a-zA-Z0-9]+)\+\+.*/, '$1');
        secondVariable = kind.replace(/.*\.\.([a-zA-Z0-9]+).*/, '$1');
      } else {
        kind = kind.replace(/[^a-zA-Z]/g, '');
        firstVariable = kind.charAt(0);
        secondVariable = kind.charAt(1);
        spectrum.variables = {};

        for (let symbol of kind) {
          let lowerCaseSymbol = symbol.toLowerCase();
          let index = currentEntry.ntuples.symbol.indexOf(symbol);
          if (index === -1) throw Error(`Symbol undefined: ${symbol}`);
          spectrum.variables[lowerCaseSymbol] = {};

          for (let key in currentEntry.ntuples) {
            if (currentEntry.ntuples[key][index]) {
              spectrum.variables[lowerCaseSymbol][key.replace(/^var/, '')] = currentEntry.ntuples[key][index];
            }
          }
        }
      }

      xIndex = currentEntry.ntuples.symbol.indexOf(firstVariable);
      yIndex = currentEntry.ntuples.symbol.indexOf(secondVariable);
      if (xIndex === -1) xIndex = 0;
      if (yIndex === -1) yIndex = 0;

      if (currentEntry.ntuples.first) {
        if (currentEntry.ntuples.first.length > xIndex) {
          spectrum.firstX = currentEntry.ntuples.first[xIndex];
        }

        if (currentEntry.ntuples.first.length > yIndex) {
          spectrum.firstY = currentEntry.ntuples.first[yIndex];
        }
      }

      if (currentEntry.ntuples.last) {
        if (currentEntry.ntuples.last.length > xIndex) {
          spectrum.lastX = currentEntry.ntuples.last[xIndex];
        }

        if (currentEntry.ntuples.last.length > yIndex) {
          spectrum.lastY = currentEntry.ntuples.last[yIndex];
        }
      }

      if (currentEntry.ntuples.vardim && currentEntry.ntuples.vardim.length > xIndex) {
        spectrum.nbPoints = currentEntry.ntuples.vardim[xIndex];
      }

      if (currentEntry.ntuples.factor) {
        if (currentEntry.ntuples.factor.length > xIndex) {
          spectrum.xFactor = currentEntry.ntuples.factor[xIndex];
        }

        if (currentEntry.ntuples.factor.length > yIndex) {
          spectrum.yFactor = currentEntry.ntuples.factor[yIndex];
        }
      }

      if (currentEntry.ntuples.units) {
        if (currentEntry.ntuples.units.length > xIndex) {
          if (currentEntry.ntuples.varname && currentEntry.ntuples.varname[xIndex]) {
            spectrum.xUnits = `${currentEntry.ntuples.varname[xIndex]} [${currentEntry.ntuples.units[xIndex]}]`;
          } else {
            spectrum.xUnits = currentEntry.ntuples.units[xIndex];
          }
        }

        if (currentEntry.ntuples.units.length > yIndex) {
          if (currentEntry.ntuples.varname && currentEntry.ntuples.varname[yIndex]) {
            spectrum.yUnits = `${currentEntry.ntuples.varname[yIndex]} [${currentEntry.ntuples.units[yIndex]}]`;
          } else {
            spectrum.yUnits = currentEntry.ntuples.units[yIndex];
          }
        }
      }
    }

    function prepareSpectrum(spectrum) {
      if (!spectrum.xFactor) spectrum.xFactor = 1;
      if (!spectrum.yFactor) spectrum.yFactor = 1;
    }

    const ntuplesSeparatorRegExp = /[ \t]*,[ \t]*/;
    const defaultOptions = {
      keepRecordsRegExp: /^$/,
      canonicDataLabels: true,
      canonicMetadataLabels: false,
      dynamicTyping: true,
      withoutXY: false,
      chromatogram: false,
      keepSpectra: false,
      noContour: false,
      nbContourLevels: 7,
      noiseMultiplier: 5,
      profiling: false
    };
    /**
     *
     * @typedef {object} ConvertOptions
     * @property {RegExp} [options.keepRecordsRegExp=/^$/] - By default we don't keep meta information.
     * @property {boolean} [options.canonicDataLabels=true] - Canonize the Labels (uppercase without symbol).
     * @property {boolean} [options.canonicMetadataLabels=false] - Canonize the metadata Labels (uppercase without symbol).
     * @property {boolean} [options.dynamicTyping=false] - Convert numbers to Number.
     * @property {boolean} [options.withoutXY=false] - Remove the XY data.
     * @property {boolean} [options.chromatogram=false] - Special post-processing for GC / HPLC / MS.
     * @property {boolean} [options.keepSpectra=false] - Force to keep the spectra in case of 2D.
     * @property {boolean} [options.noContour=false] - Don't calculate countour in case of 2D.
     * @property {number} [options.nbContourLevels=7] - Number of positive / negative contour levels to calculate.
     * @property {number} [options.noiseMultiplier=5] - Define for 2D the level as 5 times the median as default.
     * @property {boolean} [options.profiling=false] - Add profiling information.
     */

    /**
     *
     * @typedef {object} Ntuples
     * @property {string[]} [varname]
     * @property {string[]} [symbol]
     * @property {string[]} [vartype]
     * @property {string[]} [varform]
     * @property {number[]} [vardim]
     * @property {string[]} [units]
     * @property {number[]} [factor]
     * @property {number[]} [first]
     * @property {number[]} [last]
     * @property {number[]} [min]
     * @property {number[]} [max]
     * @property {string[]} [nucleus]
     */

    /**
     * @typedef { Record<string, any> } Spectrum
     * @property {Record<string, number[]>} [data]
     * @property {number} [firstX]
     * @property {number} [lastX]
     * @property {number} [deltaX]
     * @property {number} [yFactor]
     * @property {number} [xFactor]
     * @property {number} [nbPoints]
     */

    /**
     *
     * @typedef {object} Entry
     * @property {Spectrum[]} spectra
     * @property {Ntuples} ntuples
     * @property {object} meta
     * @property {object} tmp
     * @property {string} [title]
     * @property {string} [dataType]
     * @property {string} [dataClass]
     * @property {boolean} [twoD]
     */

    /**
     *
     * @typedef { object } ConvertResult
     * @property { object[] | boolean } profiling
     * @property { string[] } logs
     * @property { object[] } entries
     * @property { Entry[] } flatten
     */

    /**
     * Parse a jcamp.
     *
     * @param {string|ArrayBuffer|Uint8Array} jcamp
     * @param {ConvertOptions} [options]
     * @returns {ConvertResult}
     */

    function convert(jcamp) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      jcamp = ensureString(jcamp);
      options = { ...defaultOptions,
        ...options
      };
      options.wantXY = !options.withoutXY;
      options.start = Date.now();
      let entriesFlat = [];
      let result = {
        profiling: options.profiling ? [] : false,
        logs: [],
        entries: []
      };
      let tmpResult = {
        children: []
      };
      let currentEntry = tmpResult;
      let parentsStack = [];
      let spectrum = {};

      if (typeof jcamp !== 'string') {
        throw new TypeError('the JCAMP should be a string');
      }

      profiling(result, 'Before split to LDRS', options);
      let ldrs = jcamp.replace(/[\r\n]+##/g, '\n##').split('\n##');
      profiling(result, 'Split to LDRS', options);
      if (ldrs[0]) ldrs[0] = ldrs[0].replace(/^[\r\n ]*##/, '');

      for (let ldr of ldrs) {
        // This is a new LDR
        let position = ldr.indexOf('=');
        let dataLabel = position > 0 ? ldr.substring(0, position) : ldr;
        let dataValue = position > 0 ? ldr.substring(position + 1).trim() : '';
        let canonicDataLabel = dataLabel.replace(/[_ -]/g, '').toUpperCase();

        if (canonicDataLabel === 'DATATABLE') {
          let endLine = dataValue.indexOf('\n');
          if (endLine === -1) endLine = dataValue.indexOf('\r');

          if (endLine > 0) {
            // ##DATA TABLE= (X++(I..I)), XYDATA
            // We need to find the variables
            let infos = dataValue.substring(0, endLine).split(/[ ,;\t]+/);
            prepareNtuplesDatatable(currentEntry, spectrum, infos[0]);
            spectrum.datatable = infos[0];

            if (infos[1] && infos[1].indexOf('PEAKS') > -1) {
              canonicDataLabel = 'PEAKTABLE';
            } else if (infos[1] && (infos[1].indexOf('XYDATA') || infos[0].indexOf('++') > 0)) {
              canonicDataLabel = 'XYDATA';

              if (spectrum.nbPoints) {
                spectrum.deltaX = (spectrum.lastX - spectrum.firstX) / (spectrum.nbPoints - 1);
              }
            }
          }
        }

        if (canonicDataLabel === 'XYDATA') {
          if (options.wantXY) {
            prepareSpectrum(spectrum); // well apparently we should still consider it is a PEAK TABLE if there are no '++' after

            if (dataValue.match(/.*\+\+.*/)) {
              // ex: (X++(Y..Y))
              if (spectrum.nbPoints) {
                spectrum.deltaX = (spectrum.lastX - spectrum.firstX) / (spectrum.nbPoints - 1);
              }

              fastParseXYData(spectrum, dataValue);
            } else {
              parsePeakTable(spectrum, dataValue, result);
            }

            currentEntry.spectra.push(spectrum);
            spectrum = {};
          }

          continue;
        } else if (canonicDataLabel === 'PEAKTABLE') {
          if (options.wantXY) {
            prepareSpectrum(spectrum);
            parsePeakTable(spectrum, dataValue, result);
            currentEntry.spectra.push(spectrum);
            spectrum = {};
          }

          continue;
        }

        if (canonicDataLabel === 'PEAKASSIGNMENTS') {
          if (options.wantXY) {
            if (dataValue.match(/.*(XYA).*/)) {
              // ex: (XYA)
              parseXYA(spectrum, dataValue);
            }

            currentEntry.spectra.push(spectrum);
            spectrum = {};
          }

          continue;
        }

        if (canonicDataLabel === 'TITLE') {
          let parentEntry = currentEntry;

          if (!parentEntry.children) {
            parentEntry.children = [];
          }

          currentEntry = {
            spectra: [],
            ntuples: {},
            info: {},
            meta: {},
            tmp: {} // tmp information we need to keep for postprocessing

          };
          parentEntry.children.push(currentEntry);
          parentsStack.push(parentEntry);
          entriesFlat.push(currentEntry);
          currentEntry.title = dataValue;
        } else if (canonicDataLabel === 'DATATYPE') {
          currentEntry.dataType = dataValue;

          if (dataValue.match(/(^nd|\snd\s)/i)) {
            currentEntry.twoD = true;
          }
        } else if (canonicDataLabel === 'NTUPLES') {
          if (dataValue.match(/(^nd|\snd\s)/i)) {
            currentEntry.twoD = true;
          }
        } else if (canonicDataLabel === 'DATACLASS') {
          currentEntry.dataClass = dataValue;
        } else if (canonicDataLabel === 'XUNITS') {
          spectrum.xUnits = dataValue;
        } else if (canonicDataLabel === 'YUNITS') {
          spectrum.yUnits = dataValue;
        } else if (canonicDataLabel === 'FIRSTX') {
          spectrum.firstX = Number(dataValue);
        } else if (canonicDataLabel === 'LASTX') {
          spectrum.lastX = Number(dataValue);
        } else if (canonicDataLabel === 'FIRSTY') {
          spectrum.firstY = Number(dataValue);
        } else if (canonicDataLabel === 'LASTY') {
          spectrum.lastY = Number(dataValue);
        } else if (canonicDataLabel === 'NPOINTS') {
          spectrum.nbPoints = Number(dataValue);
        } else if (canonicDataLabel === 'XFACTOR') {
          spectrum.xFactor = Number(dataValue);
        } else if (canonicDataLabel === 'YFACTOR') {
          spectrum.yFactor = Number(dataValue);
        } else if (canonicDataLabel === 'MAXX') {
          spectrum.maxX = Number(dataValue);
        } else if (canonicDataLabel === 'MINX') {
          spectrum.minX = Number(dataValue);
        } else if (canonicDataLabel === 'MAXY') {
          spectrum.maxY = Number(dataValue);
        } else if (canonicDataLabel === 'MINY') {
          spectrum.minY = Number(dataValue);
        } else if (canonicDataLabel === 'DELTAX') {
          spectrum.deltaX = Number(dataValue);
        } else if (canonicDataLabel === '.OBSERVEFREQUENCY' || canonicDataLabel === '$SFO1') {
          if (!spectrum.observeFrequency) {
            spectrum.observeFrequency = Number(dataValue);
          }
        } else if (canonicDataLabel === '.OBSERVENUCLEUS') {
          if (!spectrum.xType) {
            currentEntry.xType = dataValue.replace(/[^a-zA-Z0-9]/g, '');
          }
        } else if (canonicDataLabel === '$OFFSET') {
          // OFFSET for Bruker spectra
          currentEntry.shiftOffsetNum = 0;

          if (!spectrum.shiftOffsetVal) {
            spectrum.shiftOffsetVal = Number(dataValue);
          }
        } else if (canonicDataLabel === '$REFERENCEPOINT') ; else if (canonicDataLabel === 'VARNAME') {
          currentEntry.ntuples.varname = dataValue.split(ntuplesSeparatorRegExp);
        } else if (canonicDataLabel === 'SYMBOL') {
          currentEntry.ntuples.symbol = dataValue.split(ntuplesSeparatorRegExp);
        } else if (canonicDataLabel === 'VARTYPE') {
          currentEntry.ntuples.vartype = dataValue.split(ntuplesSeparatorRegExp);
        } else if (canonicDataLabel === 'VARFORM') {
          currentEntry.ntuples.varform = dataValue.split(ntuplesSeparatorRegExp);
        } else if (canonicDataLabel === 'VARDIM') {
          currentEntry.ntuples.vardim = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
        } else if (canonicDataLabel === 'UNITS') {
          currentEntry.ntuples.units = dataValue.split(ntuplesSeparatorRegExp);
        } else if (canonicDataLabel === 'FACTOR') {
          currentEntry.ntuples.factor = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
        } else if (canonicDataLabel === 'FIRST') {
          currentEntry.ntuples.first = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
        } else if (canonicDataLabel === 'LAST') {
          currentEntry.ntuples.last = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
        } else if (canonicDataLabel === 'MIN') {
          currentEntry.ntuples.min = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
        } else if (canonicDataLabel === 'MAX') {
          currentEntry.ntuples.max = convertToFloatArray(dataValue.split(ntuplesSeparatorRegExp));
        } else if (canonicDataLabel === '.NUCLEUS') {
          if (currentEntry.ntuples) {
            currentEntry.ntuples.nucleus = dataValue.split(ntuplesSeparatorRegExp);
          }
        } else if (canonicDataLabel === 'PAGE') {
          spectrum.page = dataValue.trim();
          spectrum.pageValue = Number(dataValue.replace(/^.*=/, ''));
          spectrum.pageSymbol = spectrum.page.replace(/[=].*/, '');
        } else if (canonicDataLabel === 'RETENTIONTIME') {
          spectrum.pageValue = Number(dataValue);
        } else if (isMSField(canonicDataLabel)) {
          spectrum[convertMSFieldToLabel(canonicDataLabel)] = dataValue;
        } else if (canonicDataLabel === 'SAMPLEDESCRIPTION') {
          spectrum.sampleDescription = dataValue;
        } else if (canonicDataLabel.startsWith('$NUC')) {
          if (!currentEntry.tmp[canonicDataLabel] && !dataValue.includes('off')) {
            currentEntry.tmp[canonicDataLabel] = dataValue.replace(/[<>]/g, '');
          }
        } else if (canonicDataLabel === 'END') {
          currentEntry = parentsStack.pop();
        }

        if (currentEntry && currentEntry.info && currentEntry.meta && canonicDataLabel.match(options.keepRecordsRegExp)) {
          let value = dataValue.trim();
          let target, label;

          if (dataLabel.startsWith('$')) {
            label = options.canonicMetadataLabels ? canonicDataLabel.substring(1) : dataLabel.substring(1);
            target = currentEntry.meta;
          } else {
            label = options.canonicDataLabels ? canonicDataLabel : dataLabel;
            target = currentEntry.info;
          }

          if (options.dynamicTyping) {
            value = parseString(value);
          }

          if (target[label]) {
            if (!Array.isArray(target[label])) {
              target[label] = [target[label]];
            }

            target[label].push(value);
          } else {
            target[label] = value;
          }
        }
      }

      profiling(result, 'Finished parsing', options);
      postProcessing(entriesFlat, result, options);
      profiling(result, 'Total time', options);
      /*
      if (result.children && result.children.length>0) {
        result = { ...result, ...result.children[0] };
      }
      */

      result.entries = tmpResult.children;
      result.flatten = entriesFlat;
      return result;
    }

    const expectedTypes = ['ndnmrspectrum', 'ndnmrfid', 'nmrspectrum', 'nmrfid'];
    function fromJCAMP(buffer) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        noContour = true,
        xy = true,
        keepRecordsRegExp = /.*/,
        profiling = true
      } = options;
      let parsedData = convert(buffer, {
        noContour,
        xy,
        keepRecordsRegExp,
        profiling
      });
      let dataStructure = [];
      let entries = parsedData.flatten;

      for (let entry of entries) {
        if (!isSpectraData(entry)) continue;

        if (entry.spectra && entry.spectra.length > 0 || entry.minMax) {
          let metadata = Object.assign({}, entry.info, entry.meta);
          let info = getInfoFromJCAMP(metadata);
          if (info.experiment === 'wobble_curve') continue;
          let dimensions = [];
          let dependentVariables = [];
          let dependentVariable = {};

          if (info.dimension === 1) {
            for (let i = 0; i < entry.spectra.length; i++) {
              let data = entry.spectra[i].data;
              data = convertToFloatArray$1(data);
            }

            dependentVariable.components = entry.spectra;
          } else if (info.dimension === 2) {
            entry.minMax.z = convertToFloatArray$1(entry.minMax.z);
            dependentVariable.components = entry.minMax;
          }

          let dimension = {
            increment: info.increment,
            numberOfPoints: info.numberOfPoints
          };

          if (info.fid) {
            dimension.coordinatesOffset = {
              magnitude: -info.digitalFilter * info.increment,
              units: 'second'
            };
          } else {
            dimension.coordinatesOffset = {
              magnitude: info.frequencyOffset / info.baseFrequency - 0.5 * info.spectraWidth,
              units: 'ppm'
            };
          }

          dimensions.push(dimension);
          dependentVariables.push(dependentVariable);
          const data = {
            dimensions,
            dependentVariables,
            info,
            meta: metadata,
            timeStamp: new Date().valueOf(),
            version: packageJson.version
          };
          dataStructure.push(data);
        }
      }

      return dataStructure;
    }

    function isSpectraData(entry) {
      const {
        dataType = '',
        dataClass = ''
      } = entry;
      const inputDataType = dataType.replace(/\s/g, '').toLowerCase();
      const inputDataClass = dataClass.replace(/\s/g, '').toLowerCase();
      return expectedTypes.some(type => type === inputDataType) && inputDataClass !== 'peak table';
    }

    exports.formatDependentVariable = formatDependentVariable;
    exports.formatLinearDimension = formatLinearDimension;
    exports.fromBruker = fromBruker;
    exports.fromJCAMP = fromJCAMP;
    exports.fromJEOL = fromJEOL;

    Object.defineProperty(exports, '__esModule', { value: true });

}));
//# sourceMappingURL=nmr-parser.js.map
