/**
 * convert-to-jcamp - Convert strings into JCAMP
 * @version v5.4.4
 * @link https://github.com/cheminfo/convert-to-jcamp#readme
 * @license MIT
 */
(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.ConvertToJcamp = {}));
})(this, (function (exports) { 'use strict';

    const addInfoData = function (data) {
      let keys = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : Object.keys(data);
      let prefix = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : '##$';
      let header = '';
      for (const key of keys) {
        header += typeof data[key] === 'object' ? `${prefix}${key}=${JSON.stringify(data[key])}\n` : `${prefix}${key}=${data[key]}\n`;
      }
      return header;
    };

    const toString = Object.prototype.toString;
    /**
     * Checks if an object is an instance of an Array (array or typed array).
     *
     * @param {any} value - Object to check.
     * @returns {boolean} True if the object is an array.
     */
    function isAnyArray(value) {
      return toString.call(value).endsWith('Array]');
    }

    /**
     * This function
     * @param output - undefined or a new array
     * @param length - length of the output array
     * @returns
     */
    function getOutputArray(output, length) {
      if (output !== undefined) {
        if (!isAnyArray(output)) {
          throw new TypeError('output option must be an array if specified');
        }
        if (output.length !== length) {
          throw new TypeError('the output array does not have the correct length');
        }
        return output;
      } else {
        return new Float64Array(length);
      }
    }

    /**
     * This function xMultiply the first array by the second array or a constant value to each element of the first array
     *
     * @param array1 - first array
     * @param array2 - second array
     * @param options - options
     */
    function xMultiply(array1, array2) {
      let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
      let isConstant = false;
      let constant = 0;
      if (isAnyArray(array2)) {
        if (array1.length !== array2.length) {
          throw new Error('xMultiply: size of array1 and array2 must be identical');
        }
      } else {
        isConstant = true;
        constant = Number(array2);
      }
      let array3 = getOutputArray(options.output, array1.length);
      if (isConstant) {
        for (let i = 0; i < array1.length; i++) {
          array3[i] = array1[i] * constant;
        }
      } else {
        for (let i = 0; i < array1.length; i++) {
          array3[i] = array1[i] * array2[i];
        }
      }
      return array3;
    }

    /**
     * Checks if input is of type array
     *
     * @param input - input
     */
    function xCheck(input) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        minLength
      } = options;
      if (!isAnyArray(input)) {
        throw new TypeError('input must be an array');
      }
      if (input.length === 0) {
        throw new TypeError('input must not be empty');
      }
      //@ts-expect-error we already checked that input is an array
      if (minLength && input.length < minLength) {
        throw new Error(`input must have a length of at least ${minLength}`);
      }
    }

    /**
     * This function divide the first array by the second array or a constant value to each element of the first array
     *
     * @param array1 - first array
     * @param array2 - second array or number
     * @param options - options
     */
    function xDivide(array1, array2) {
      let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
      let isConstant = false;
      let constant = 0;
      if (isAnyArray(array2)) {
        if (array1.length !== array2.length) {
          throw new Error('xDivide: size of array1 and array2 must be identical');
        }
      } else {
        isConstant = true;
        constant = Number(array2);
      }
      let array3 = getOutputArray(options.output, array1.length);
      if (isConstant) {
        for (let i = 0; i < array1.length; i++) {
          array3[i] = array1[i] / constant;
        }
      } else {
        for (let i = 0; i < array1.length; i++) {
          array3[i] = array1[i] / array2[i];
        }
      }
      return array3;
    }

    /**
     * Return min and max values of an array
     *
     * @param array - array of number
     * @returns - Object with 2 properties, min and max
     */
    function xMinMaxValues(array) {
      xCheck(array);
      let min = array[0];
      let max = array[0];
      for (let value of array) {
        if (value < min) min = value;
        if (value > max) max = value;
      }
      return {
        min,
        max
      };
    }

    function matrixCheck(data) {
      if (data.length === 0 || data[0].length === 0) {
        throw new RangeError('matrix should contain data');
      }
      const firstLength = data[0].length;
      for (let i = 1; i < data.length; i++) {
        if (data[i].length !== firstLength) {
          throw new RangeError('All rows should has the same length');
        }
      }
    }

    /**
     * Get min and max Z
     *
     * @param matrix - matrix [rows][cols].
     */
    function matrixMinMaxZ(matrix) {
      matrixCheck(matrix);
      const nbRows = matrix.length;
      const nbColumns = matrix[0].length;
      let min = matrix[0][0];
      let max = matrix[0][0];
      for (let column = 0; column < nbColumns; column++) {
        for (let row = 0; row < nbRows; row++) {
          if (matrix[row][column] < min) min = matrix[row][column];
          if (matrix[row][column] > max) max = matrix[row][column];
        }
      }
      return {
        min,
        max
      };
    }

    function checkMatrix(data) {
      if (!isAnyArray(data) || !isAnyArray(data[0])) {
        throw new Error(`2D data should be a matrix`);
      }
    }

    function checkNumberOrArray(data) {
      if (!isAnyArray(data) || isAnyArray(data[0])) {
        throw new Error(`x and y data should be an array of numbers`);
      }
    }

    function getExtremeValues(data) {
      if (isAnyArray(data[0])) {
        checkMatrix(data);
        const firstRow = data[0];
        return {
          firstLast: {
            first: firstRow[0],
            last: data[data.length - 1][data[0].length - 1]
          },
          minMax: matrixMinMaxZ(data)
        };
      }
      checkNumberOrArray(data);
      return {
        firstLast: {
          first: data[0],
          last: data[data.length - 1]
        },
        minMax: xMinMaxValues(data)
      };
    }

    /**
     * Parse from a xyxy data array
     * @param variables - Variables to convert to jcamp
     * @param [options={}] - options that allows to add meta data in the jcamp
     * @return JCAMP-DX text file corresponding to the variables
     */
    function creatorNtuples(variables, options) {
      const {
        meta = {},
        info = {}
      } = options;
      const {
        title = '',
        owner = '',
        origin = '',
        dataType = ''
      } = info;
      const symbol = [];
      const varName = [];
      const varType = [];
      const varDim = [];
      const units = [];
      const first = [];
      const last = [];
      const min = [];
      const max = [];
      const keys = Object.keys(variables);
      for (let i = 0; i < keys.length; i++) {
        const key = keys[i];
        let variable = variables[key];
        if (!variable) continue;
        let name = variable?.label.replace(/ *\[.*/, '');
        let unit = variable?.label.replace(/.*\[(?<units>.*)\].*/, '$<units>');
        const {
          firstLast,
          minMax
        } = getExtremeValues(variable.data);
        symbol.push(variable.symbol || key);
        varName.push(name || key);
        varDim.push(variable.data.length);
        first.push(firstLast.first);
        last.push(firstLast.last);
        max.push(minMax.max);
        min.push(minMax.min);
        if (variable.isDependent !== undefined) {
          varType.push(variable.isDependent ? 'DEPENDENT' : 'INDEPENDENT');
        } else {
          varType.push(variable.isDependent !== undefined ? !variable.isDependent : i === 0 ? 'INDEPENDENT' : 'DEPENDENT');
        }
        units.push(variable.units || unit || '');
      }
      let header = `##TITLE=${title}
##JCAMP-DX=6.00
##DATA TYPE=${dataType}
##DATA CLASS= NTUPLES
##ORIGIN=${origin}
##OWNER=${owner}\n`;
      const infoKeys = Object.keys(info).filter(e => !['title', 'owner', 'origin', 'datatype'].includes(e.toLocaleLowerCase()));
      header += addInfoData(info, infoKeys, '##');
      header += addInfoData(meta);
      header += `##NTUPLES= ${dataType}
##VAR_NAME=  ${varName.join()}
##SYMBOL=    ${symbol.join()}
##VAR_TYPE=  ${varType.join()}
##VAR_DIM=   ${varDim.join()}
##UNITS=     ${units.join()}
##FIRST=     ${first.join()}
##LAST=      ${last.join()}
##MIN=       ${min.join()}
##MAX=       ${max.join()}
##PAGE= N=1\n`;
      header += `##DATA TABLE= (${symbol.join('')}..${symbol.join('')}), PEAKS\n`;
      for (let i = 0; i < variables.x.data.length; i++) {
        let point = [];
        for (let key of keys) {
          let variable = variables[key];
          if (!variable) continue;
          point.push(variable.data[i]);
        }
        header += `${point.join('\t')}\n`;
      }
      header += `##END NTUPLES= ${dataType}\n`;
      header += '##END=\n##END=';
      return header;
    }

    function getFactorNumber(minMax) {
      let maxValue = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 2 ** 31 - 1;
      let factor;
      if (minMax.min < 0) {
        if (minMax.max > 0) {
          factor = Math.max(-minMax.min, minMax.max) / maxValue;
        } else {
          factor = -minMax.min / maxValue;
        }
      } else {
        factor = minMax.max / maxValue;
      }
      return factor;
    }

    function getBestFactor(array) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        maxValue,
        factor,
        minMax
      } = options;
      if (factor !== undefined) {
        return factor;
      }
      // is there non integer number ?
      let onlyInteger = true;
      for (let y of array) {
        if (Math.round(y) !== y) {
          onlyInteger = false;
          break;
        }
      }
      if (onlyInteger) {
        return 1;
      }
      // we need to rescale the values
      // need to find the max and min values
      const extremeValues = minMax || xMinMaxValues(array);
      return getFactorNumber(extremeValues, maxValue);
    }

    /**
     * Reconvert number to original value
     * @param number Number used for computation
     * @param factor Multiplying factor
     * @returns Original value
     */
    function getNumber(number, factor) {
      if (factor !== 1) number /= factor;
      const rounded = Math.round(number);
      if (rounded !== number && Math.abs(rounded - number) <= Number.EPSILON) {
        return rounded;
      }
      return number;
    }

    function peakTableCreator(data) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        xFactor = 1,
        yFactor = 1
      } = options.info || {};
      let firstX = Number.POSITIVE_INFINITY;
      let lastX = Number.NEGATIVE_INFINITY;
      let firstY = Number.POSITIVE_INFINITY;
      let lastY = Number.NEGATIVE_INFINITY;
      let lines = [];
      for (let i = 0; i < data.x.length; i++) {
        let x = data.x[i];
        let y = data.y[i];
        if (firstX > x) {
          firstX = x;
        }
        if (lastX < x) {
          lastX = x;
        }
        if (firstY > y) {
          firstY = y;
        }
        if (lastY < y) {
          lastY = y;
        }
      }
      lines.push(`##FIRSTX=${firstX}`);
      lines.push(`##LASTX=${lastX}`);
      lines.push(`##FIRSTY=${firstY}`);
      lines.push(`##LASTY=${lastY}`);
      lines.push(`##XFACTOR=${xFactor}`);
      lines.push(`##YFACTOR=${yFactor}`);
      lines.push('##PEAK TABLE=(XY..XY)');
      for (let i = 0; i < data.x.length; i++) {
        lines.push(`${getNumber(data.x[i], xFactor)} ${getNumber(data.y[i], yFactor)}`);
      }
      return lines;
    }

    function rescaleAndEnsureInteger(data) {
      let factor = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 1;
      if (factor === 1) return data.map(value => Math.round(value));
      return xDivide(data, factor);
    }

    /**
     * class encodes a integer vector as a String in order to store it in a text file.
     * The algorithms used to encode the data are describe in:
     *            http://www.iupac.org/publications/pac/pdf/2001/pdf/7311x1765.pdf
     */
    const newLine = '\n';
    const pseudoDigits = [['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'], ['@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I'], ['@', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i'], ['%', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R'], ['%', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r'], [' ', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 's']];
    const SQZ_P = 1;
    const SQZ_N = 2;
    const DIF_P = 3;
    const DIF_N = 4;
    const DUP = 5;
    const maxLinelength = 100;
    /**
     * This function encodes the given vector. The xyEncoding format is specified by the
     * xyEncoding option
     * @param xyEncoding: ('FIX','SQZ','DIF','DIFDUP','CVS','PAC') Default 'DIFDUP'
     * @return {string}
     */
    function vectorEncoder(data, firstX, intervalX, xyEncoding) {
      switch (xyEncoding) {
        case 'FIX':
          return fixEncoding(data, firstX, intervalX);
        case 'SQZ':
          return squeezedEncoding(data, firstX, intervalX);
        case 'DIF':
          return differenceEncoding(data, firstX, intervalX);
        case 'DIFDUP':
          return differenceDuplicateEncoding(data, firstX, intervalX);
        case 'CSV':
          return commaSeparatedValuesEncoding(data, firstX, intervalX);
        case 'PAC':
          return packedEncoding(data, firstX, intervalX);
        default:
          return differenceEncoding(data, firstX, intervalX);
      }
    }
    /**
     * @private
     * No data compression used. The data is separated by a comma(',').
     */
    function commaSeparatedValuesEncoding(data, firstX, intervalX) {
      return fixEncoding(data, firstX, intervalX, ',');
    }
    /**
     * @private
     * No data compression used. The data is separated by the specified separator.
     */
    function fixEncoding(data, firstX, intervalX) {
      let separator = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : ' ';
      let outputData = '';
      let j = 0;
      let dataLength = data.length;
      while (j < dataLength - 7) {
        outputData += Math.ceil(firstX + j * intervalX);
        for (let i = 0; i < 8; i++) {
          outputData += `${separator}${data[j++]}`;
        }
        outputData += newLine;
      }
      if (j < dataLength) {
        // We add last numbers
        outputData += Math.ceil(firstX + j * intervalX);
        for (let i = j; i < dataLength; i++) {
          outputData += `${separator}${data[i]}`;
        }
      }
      return outputData;
    }
    /**
     * @private
     * No data compression used. The data is separated by the sign of the number.
     */
    function packedEncoding(data, firstX, intervalX) {
      let outputData = '';
      let j = 0;
      let dataLength = data.length;
      while (j < dataLength - 7) {
        outputData += Math.ceil(firstX + j * intervalX);
        for (let i = 0; i < 8; i++) {
          outputData += data[j] < 0 ? data[j++] : `+${data[j++]}`;
        }
        outputData += newLine;
      }
      if (j < dataLength) {
        // We add last numbers
        outputData += Math.ceil(firstX + j * intervalX);
        for (let i = j; i < dataLength; i++) {
          outputData += data[i] < 0 ? data[i] : `+${data[i]}`;
        }
      }
      return outputData;
    }
    /**
     * @private
     * Data compression is possible using the squeezed form (SQZ) in which the delimiter, the leading digit,
     * and sign are replaced by a pseudo-digit from Table 1. For example, the Y-values 30, 32 would be
     * represented as C0C2.
     */
    function squeezedEncoding(data, firstX, intervalX) {
      let outputData = '';
      // String outputData = new String();
      let j = 0;
      let dataLength = data.length;
      while (j < dataLength - 10) {
        outputData += Math.ceil(firstX + j * intervalX);
        for (let i = 0; i < 10; i++) {
          outputData += squeezedDigit(data[j++].toString());
        }
        outputData += newLine;
      }
      if (j < dataLength) {
        // We add last numbers
        outputData += Math.ceil(firstX + j * intervalX);
        for (let i = j; i < dataLength; i++) {
          outputData += squeezedDigit(data[i].toString());
        }
      }
      return outputData;
    }
    /**
     * @private
     * Duplicate suppression xyEncoding
     */
    function differenceDuplicateEncoding(data, firstX, intervalX) {
      let mult = 0;
      let index = 0;
      let charCount = 0;
      // We built a string where we store the encoded data.
      let encodedData = '';
      let encodedNumber = '';
      let temp = '';
      // We calculate the differences vector
      let diffData = new Array(data.length - 1);
      for (let i = 0; i < diffData.length; i++) {
        diffData[i] = data[i + 1] - data[i];
      }
      // We simulate a line carry
      let numDiff = diffData.length;
      while (index < numDiff) {
        if (charCount === 0) {
          // Start line
          encodedNumber = `${Math.ceil(firstX + index * intervalX)}${squeezedDigit(data[index].toString())}${differenceDigit(diffData[index].toString())}`;
          encodedData += encodedNumber;
          charCount += encodedNumber.length;
        } else if (diffData[index - 1] === diffData[index]) {
          // Try to insert next difference
          mult++;
        } else if (mult > 0) {
          // Now we know that it can be in line
          mult++;
          encodedNumber = duplicateDigit(mult.toString());
          encodedData += encodedNumber;
          charCount += encodedNumber.length;
          mult = 0;
          index--;
        } else {
          // Check if it fits, otherwise start a new line
          encodedNumber = differenceDigit(diffData[index].toString());
          if (encodedNumber.length + charCount < maxLinelength) {
            encodedData += encodedNumber;
            charCount += encodedNumber.length;
          } else {
            // start a new line
            encodedData += newLine;
            temp = `${Math.ceil(firstX + index * intervalX)}${squeezedDigit(data[index].toString())}${encodedNumber}`;
            encodedData += temp; // Each line start with first index number.
            charCount = temp.length;
          }
        }
        index++;
      }
      if (mult > 0) {
        encodedData += duplicateDigit((mult + 1).toString());
      }
      // We insert the last data from fid. It is done to control of data
      // The last line start with the number of datas in the fid.
      encodedData += `${newLine}${Math.ceil(firstX + index * intervalX)}${squeezedDigit(data[index].toString())}`;
      return encodedData;
    }
    /**
     * @private
     * Differential xyEncoding
     */
    function differenceEncoding(data, firstX, intervalX) {
      let index = 0;
      let charCount = 0;
      let i;
      let encodedData = '';
      let encodedNumber = '';
      let temp = '';
      // We calculate the differences vector
      let diffData = new Array(data.length - 1);
      for (i = 0; i < diffData.length; i++) {
        diffData[i] = data[i + 1] - data[i];
      }
      let numDiff = diffData.length;
      while (index < numDiff) {
        if (charCount === 0) {
          // We convert the first number.
          encodedNumber = `${Math.ceil(firstX + index * intervalX)}${squeezedDigit(data[index].toString())}${differenceDigit(diffData[index].toString())}`;
          encodedData += encodedNumber;
          charCount += encodedNumber.length;
        } else {
          encodedNumber = differenceDigit(diffData[index].toString());
          if (encodedNumber.length + charCount < maxLinelength) {
            encodedData += encodedNumber;
            charCount += encodedNumber.length;
          } else {
            encodedData += newLine;
            temp = `${Math.ceil(firstX + index * intervalX)}${squeezedDigit(data[index].toString())}${encodedNumber}`;
            encodedData += temp; // Each line start with first index number.
            charCount = temp.length;
          }
        }
        index++;
      }
      // We insert the last number from data. It is done to control of data
      encodedData += `${newLine}${Math.ceil(firstX + index * intervalX)}${squeezedDigit(data[index].toString())}`;
      return encodedData;
    }
    /**
     * @private
     * Convert number to the ZQZ format, using pseudo digits.
     */
    function squeezedDigit(num) {
      let sqzDigits = '';
      if (num.startsWith('-')) {
        sqzDigits += pseudoDigits[SQZ_N][num.charCodeAt(1) - 48];
        if (num.length > 2) {
          sqzDigits += num.substring(2);
        }
      } else {
        sqzDigits += pseudoDigits[SQZ_P][num.charCodeAt(0) - 48];
        if (num.length > 1) {
          sqzDigits += num.substring(1);
        }
      }
      return sqzDigits;
    }
    /**
     * Convert number to the DIF format, using pseudo digits.
     */
    function differenceDigit(num) {
      let diffDigits = '';
      if (num.startsWith('-')) {
        diffDigits += pseudoDigits[DIF_N][num.charCodeAt(1) - 48];
        if (num.length > 2) {
          diffDigits += num.substring(2);
        }
      } else {
        diffDigits += pseudoDigits[DIF_P][num.charCodeAt(0) - 48];
        if (num.length > 1) {
          diffDigits += num.substring(1);
        }
      }
      return diffDigits;
    }
    /**
     * Convert number to the DUP format, using pseudo digits.
     */
    function duplicateDigit(num) {
      let dupDigits = '';
      dupDigits += pseudoDigits[DUP][num.charCodeAt(0) - 48];
      if (num.length > 1) {
        dupDigits += num.substring(1);
      }
      return dupDigits;
    }

    function xyDataCreator(data) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        xyEncoding = 'DIF'
      } = options;
      const {
        xFactor = 1,
        yFactor = 1
      } = options.info || {};
      let firstX = data.x[0];
      let lastX = data.x[data.x.length - 1];
      let firstY = data.y[0];
      let lastY = data.y[data.y.length - 1];
      let nbPoints = data.x.length;
      let deltaX = (lastX - firstX) / (nbPoints - 1);
      let lines = [];
      lines.push(`##FIRSTX=${firstX}`);
      lines.push(`##LASTX=${lastX}`);
      lines.push(`##FIRSTY=${firstY}`);
      lines.push(`##LASTY=${lastY}`);
      lines.push(`##DELTAX=${deltaX}`);
      lines.push(`##XFACTOR=${xFactor}`);
      lines.push(`##YFACTOR=${yFactor}`);
      lines.push('##XYDATA=(X++(Y..Y))');
      let line = vectorEncoder(rescaleAndEnsureInteger(data.y, yFactor), firstX / xFactor, deltaX / xFactor, xyEncoding);
      if (line) lines.push(line);
      return lines;
    }

    const infoDefaultKeys = ['title', 'owner', 'origin', 'dataType', 'xUnits', 'yUnits', 'xFactor', 'yFactor'];
    /**
     * Create a jcamp
     * @param data object of array
     * @param [options={meta:{},info:{}} - metadata object
     * @returns JCAMP of the input
     */
    function fromJSON(data) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        meta = {},
        info = {},
        xyEncoding
      } = options;
      let {
        title = '',
        owner = '',
        origin = '',
        dataType = '',
        xUnits = '',
        yUnits = '',
        xFactor,
        yFactor
      } = info;
      data = {
        x: data.x,
        y: data.y
      };
      let header = `##TITLE=${title}
##JCAMP-DX=4.24
##DATA TYPE=${dataType}
##ORIGIN=${origin}
##OWNER=${owner}
##XUNITS=${xUnits}
##YUNITS=${yUnits}\n`;
      const infoKeys = Object.keys(info).filter(keys => !infoDefaultKeys.includes(keys));
      header += addInfoData(info, infoKeys, '##');
      header += addInfoData(meta);
      // we leave the header and utf8 fonts ${header.replace(/[^\t\n\x20-\x7F]/g, '')
      if (xyEncoding) {
        xFactor = getBestFactor(data.x, {
          factor: xFactor
        });
        yFactor = getBestFactor(data.y, {
          factor: yFactor
        });
        return `${header}##NPOINTS=${data.x.length}
${xyDataCreator(data, {
      info: {
        xFactor,
        yFactor
      },
      xyEncoding
    }).join('\n')}
##END=`;
      } else {
        if (xFactor === undefined) xFactor = 1;
        if (yFactor === undefined) yFactor = 1;
        if (xFactor !== 1) {
          //@ts-expect-error xFactor is always defined
          data.x = data.x.map(value => value / xFactor);
        }
        if (yFactor !== 1) {
          //@ts-expect-error yFactor is always defined
          data.y = data.y.map(value => value / yFactor);
        }
        return `${header}##NPOINTS=${data.x.length}
${peakTableCreator(data, {
      info: {
        xFactor,
        yFactor
      }
    }).join('\n')}
##END=`;
      }
    }

    /**
     * Create a jcamp from variables
     */
    function fromVariables( /** object of variables */
    variables) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        info = {},
        meta = {},
        forceNtuples = false
      } = options;
      let jcampOptions = {
        info,
        meta
      };
      let keys = Object.keys(variables).map(key => key.toLowerCase());
      if (!forceNtuples && keys.length === 2) {
        let x = variables.x;
        let xLabel = x.label || 'x';
        if (variables.x.units) {
          if (xLabel.includes(variables.x.units)) {
            jcampOptions.info.xUnits = xLabel;
          } else {
            jcampOptions.info.xUnits = `${xLabel} (${variables.x.units})`;
          }
        } else {
          jcampOptions.info.xUnits = xLabel;
        }
        let y = variables.y;
        let yLabel = y.label || 'y';
        if (variables.y.units) {
          if (yLabel.includes(variables.y.units)) {
            jcampOptions.info.xUnits = yLabel;
          } else {
            jcampOptions.info.yUnits = `${yLabel} (${variables.y.units})`;
          }
        } else {
          jcampOptions.info.yUnits = yLabel;
        }
        const xData = variables.x.data;
        const yData = variables.y.data;
        checkNumberOrArray(xData);
        checkNumberOrArray(yData);
        return fromJSON({
          x: xData,
          y: yData
        }, jcampOptions);
      } else {
        return creatorNtuples(variables, options);
      }
    }

    const ntuplesKeys = ['r', 'i'];
    function isNTuplesData(variables) {
      return 'r' in variables && 'i' in variables;
    }
    function isRealData(variables) {
      return 'r' in variables && !('i' in variables);
    }
    /**
     * Create a jcamp of 1D NMR data by variables x and y or x, r, i
     * @param variables - Variables to convert to jcamp
     * @param [options={}] - options that allows to add meta data in the jcamp
     * @return JCAMP-DX text file corresponding to the variables
     */
    function from1DNMRVariables(variables, options) {
      const {
        meta = {},
        info = {},
        xyEncoding = ''
      } = options;
      const factor = 'factor' in options ? {
        ...options.factor
      } : {};
      const {
        title = '',
        owner = '',
        origin = '',
        dataType = '',
        nucleus = info.nucleus,
        originFrequency = info['.OBSERVE FREQUENCY']
      } = info;
      if (!originFrequency) {
        throw new Error('.OBSERVE FREQUENCY is mandatory into the info object for nmr data');
      }
      const newInfo = {
        '.OBSERVE FREQUENCY': originFrequency,
        '.OBSERVE NUCLEUS': nucleus,
        ...info
      };
      const xVariable = variables.x;
      let xData = xVariable.data.slice();
      if (xVariable.units?.toLowerCase() === 'ppm') {
        xData = xMultiply(xData, originFrequency);
        xVariable.units = 'Hz';
      }
      const newMeta = {
        ...meta,
        OFFSET: xData[0] / originFrequency
      };
      let header = `##TITLE=${title}
##JCAMP-DX=6.00
##DATA TYPE= ${dataType}
##DATA CLASS= NTUPLES
##ORIGIN=${origin}
##OWNER=${owner}
##.SHIFT REFERENCE= INTERNAL, CDCl3, 1, ${xData[xData.length - 1] / originFrequency}\n`; //TOPSPIN use this LDR to generate x axis.
      const infoKeys = Object.keys(newInfo).filter(key => !['title', 'owner', 'origin', 'datatype'].includes(key.toLocaleLowerCase()));
      header += addInfoData(newInfo, infoKeys, '##');
      header += addInfoData(newMeta);
      const nbPoints = xData.length;
      const spectralWidth = xData[nbPoints - 1] - xData[0];
      const firstPoint = spectralWidth > 0 ? 0 : -spectralWidth;
      const lastPoint = spectralWidth > 0 ? spectralWidth : 0;
      const symbol = ['X'];
      const varDim = [nbPoints];
      const units = [xVariable.units];
      const varType = ['INDEPENDENT'];
      const factorArray = [spectralWidth / (nbPoints + 1)];
      const varName = [xVariable.label.replace(/ *\[.*/, '') || 'X'];
      const first = [firstPoint];
      const last = [lastPoint];
      const max = [Math.max(lastPoint, firstPoint)];
      const min = [Math.min(lastPoint, firstPoint)];
      for (const key of ntuplesKeys) {
        let variable = variables[key];
        if (!variable) {
          if (key !== 'i') {
            throw new Error(`variable ${key} is mandatory in real/imaginary data`);
          }
          continue;
        }
        let name = variable?.label.replace(/ *\[.*/, '');
        let unit = variable?.label.replace(/.*\[(?<units>.*)\].*/, '$<units>');
        const {
          firstLast,
          minMax
        } = getExtremeValues(variable.data);
        factor[key] = getBestFactor(variable.data, {
          factor: factor[key],
          minMax
        });
        const currentFactor = factor[key];
        factorArray.push(currentFactor || 1);
        symbol.push(variable.symbol || key);
        varName.push(name || key);
        varDim.push(variable.data.length);
        first.push(firstLast.first);
        last.push(firstLast.last);
        max.push(minMax.max);
        min.push(minMax.min);
        varType.push('DEPENDENT');
        units.push(variable.units || unit || '');
      }
      return isNTuplesData(variables) ? addNtuplesHeader(header, variables, {
        symbol,
        varName,
        varDim,
        first,
        last,
        min,
        max,
        units,
        factor,
        varType,
        factorArray
      }, newInfo) : isRealData(variables) ? addRealData(header, {
        xData,
        yData: variables.r.data,
        xyEncoding,
        info: {
          XUNITS: 'HZ',
          YUNITS: units[1],
          XFACTOR: factorArray[0],
          YFACTOR: factorArray[1],
          DELTAX: (xData[0] - xData[nbPoints - 1]) / (nbPoints + 1),
          FIRSTX: first[0],
          FIRSTY: first[1],
          LASTX: last[0],
          MAXY: max[1],
          MINY: min[1],
          NPOINTS: xData.length,
          XYDATA: '(X++(Y..Y))'
        }
      }) : header;
    }
    function addNtuplesHeader(header, variables, inputs, info) {
      const {
        dataType = ''
      } = info;
      const {
        symbol,
        varName,
        varDim,
        first,
        last,
        min,
        max,
        units,
        varType,
        factorArray,
        xyEncoding,
        factor
      } = inputs;
      header += `##NTUPLES= ${dataType}
##VAR_NAME=  ${varName.join()}
##SYMBOL=    ${symbol.join()}
##VAR_TYPE=  ${varType.join()}
##VAR_DIM=   ${varDim.join()}
##UNITS=     ${units.join()}
##FIRST=     ${first.join()}
##LAST=      ${last.join()}
##MAX=       ${max.join()}
##MIN=       ${min.join()}\n`;
      for (const key of ['r', 'i']) {
        const variable = variables[key];
        if (!variable) continue;
        checkNumberOrArray(variable.data);
        header += `##FACTOR=    ${factorArray.join()}\n`;
        header += `##PAGE= N=${key === 'r' ? 1 : 2}\n`;
        header += `##DATA TABLE= (X++(${key === 'r' ? 'R..R' : 'I..I'})), XYDATA\n`;
        header += vectorEncoder(rescaleAndEnsureInteger(variable.data, factor[key]), 0, 1, xyEncoding);
        header += '\n';
      }
      header += `##END NTUPLES= ${dataType}\n`;
      header += '##END=';
      return header;
    }
    function addRealData(header, options) {
      const {
        xData,
        yData,
        info,
        xyEncoding
      } = options;
      header += addInfoData(info, undefined, '##');
      return `${header}
${vectorEncoder(rescaleAndEnsureInteger(yData, info.YFACTOR), xData.length - 1, -1, xyEncoding)}
##END=`;
    }

    function getBestFactorMatrix(matrix) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        maxValue,
        factor,
        minMax
      } = options;
      if (factor !== undefined) {
        return factor;
      }
      // is there non integer number ?
      let onlyInteger = true;
      for (let row of matrix) {
        for (let y of row) {
          if (Math.round(y) !== y) {
            onlyInteger = false;
            break;
          }
        }
      }
      if (onlyInteger) {
        return 1;
      }
      // we need to rescale the values
      // need to find the max and min values
      const extremeValues = minMax || matrixMinMaxZ(matrix);
      return getFactorNumber(extremeValues, maxValue);
    }

    /**
     * Create a jcamp of 2D NMR data by variables. Currently only the convertion of processed data
     * is supported. The variables x and y are the direct (F2 in Bruker) and indirect (F1 in bruker),
     * data should be in ppm scale, the z variable is the intensity (dependent)
     */
    function from2DNMRVariables(variables) {
      let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
      const {
        info = {},
        meta = {},
        xyEncoding = 'DIFDUP'
      } = options;
      const factor = 'factor' in options ? {
        ...options.factor
      } : {};
      const {
        title = '',
        owner = '',
        origin = '',
        dataType = 'NMR SPECTRUM'
      } = info;
      const symbol = [];
      const varName = [];
      const varType = [];
      const varDim = [];
      const units = [];
      const first = [];
      const last = [];
      const min = [];
      const max = [];
      const factors = [];
      const keys = ['y', 'x', 'z'];
      for (let i = 0; i < keys.length; i++) {
        const key = keys[i];
        let variable = variables[key];
        if (!variable) throw new Error('variables x, y and z are mandatory');
        let name = variable?.label.replace(/ *\[.*/, '');
        let unit = variable?.label.replace(/.*\[(?<units>.*)\].*/, '$<units>');
        const {
          firstLast,
          minMax
        } = getExtremeValues(variable.data);
        if (!(key in factor)) factor[key] = calculateFactor(variable.data, minMax);
        symbol.push(variable.symbol || key);
        varName.push(name || key);
        varDim.push(variable.data.length);
        first.push(firstLast.first);
        last.push(firstLast.last);
        max.push(minMax.max);
        min.push(minMax.min);
        factors.push(factor[key]);
        if (variable.isDependent !== undefined) {
          varType.push(variable.isDependent ? 'DEPENDENT' : 'INDEPENDENT');
        } else {
          varType.push(variable.isDependent !== undefined ? !variable.isDependent : i === 0 ? 'INDEPENDENT' : 'DEPENDENT');
        }
        units.push(variable.units || unit || '');
      }
      let header = `##TITLE=${title}
##JCAMP-DX=6.00
##DATA TYPE=${dataType}
##DATA CLASS= NTUPLES
##ORIGIN=${origin}
##OWNER=${owner}\n`;
      const infoKeys = Object.keys(info).filter(e => !['title', 'owner', 'origin', 'dataType'].includes(e));
      header += addInfoData(info, infoKeys, '##');
      header += addInfoData(meta);
      let xData = variables.x.data;
      let yData = variables.y.data;
      checkNumberOrArray(xData);
      checkNumberOrArray(yData);
      checkMandatoryParameters(meta);
      const zData = variables.z?.data || [];
      checkMatrix(zData);
      const {
        direct,
        indirect,
        dependent
      } = getDimensionIndices(symbol);
      const nuc1 = String(meta.NUC1);
      const nuc2 = String(meta.NUC2);
      const nucleus = new Array(3);
      nucleus[direct] = nuc1;
      nucleus[indirect] = nuc2;
      const sfo1 = Number(meta.SFO1);
      const sfo2 = Number(meta.SFO2);
      const optionsScaleAndJoin = {
        indices: {
          direct,
          indirect
        },
        sfo1,
        sfo2
      };
      header += `##NTUPLES= ${dataType}
##VAR_NAME=  ${varName.join()}
##SYMBOL=    ${symbol.join()}
##VAR_TYPE=  ${varType.join()}
##VAR_DIM=   ${varDim.join()}
##.NUCLEUS=  ${nucleus.join()}
##UNITS=     ${units.join()}
##FACTOR=    ${factors.join()}
##FIRST=     ${scaleAndJoin(first, optionsScaleAndJoin)}
##LAST=      ${scaleAndJoin(last, optionsScaleAndJoin)}
##MIN=       ${scaleAndJoin(min, optionsScaleAndJoin)}
##MAX=       ${scaleAndJoin(max, optionsScaleAndJoin)}\n`;
      header += `##VAR_FORM= AFFN, AFFN, ASDF\n`;
      header += '##NUM DIM= 2\n';
      if (keys[direct] !== 'x') {
        [yData, xData] = [xData, yData];
      }
      const directSymbol = symbol[direct].toUpperCase();
      const indirectSymbol = symbol[indirect].toUpperCase();
      const firstY = yData[0] * sfo2;
      const lastY = yData[yData.length - 1] * sfo2;
      const firstX = xData[0];
      const lastX = xData[xData.length - 1];
      const deltaX = (lastX - firstX) / (xData.length - 1);
      const deltaY = (lastY - firstY) / (yData.length - 1);
      let firstData = new Float64Array(3);
      firstData[direct] = firstX * sfo1;
      firstData[indirect] = firstY;
      const yFactor = factor.y || 1;
      const xFactor = factor.x || 1;
      const zFactor = factor.z || 1;
      for (let index = 0; index < zData.length; index++) {
        firstData[dependent] = zData[index][0];
        header += `##PAGE= ${indirectSymbol}=${(firstY + deltaY * index) / yFactor}\n`;
        header += `##FIRST=  ${firstData.join()}\n`;
        header += `##DATA TABLE= (${directSymbol}++(Y..Y)), PROFILE\n`;
        header += vectorEncoder(rescaleAndEnsureInteger(zData[index], zFactor), firstX / xFactor, deltaX / xFactor, xyEncoding);
        header += '\n';
      }
      return header;
    }
    function scaleAndJoin(variable, options) {
      const {
        sfo1,
        sfo2
      } = options;
      const {
        direct,
        indirect
      } = options.indices;
      const copy = variable.slice();
      copy[direct] *= sfo1;
      copy[indirect] *= sfo2;
      return copy.join();
    }
    function getDimensionIndices(entry) {
      const symbol = entry.map(e => e.toUpperCase());
      const direct = symbol.includes('F2') ? symbol.indexOf('F2') : symbol.indexOf('T2');
      const indirect = symbol.includes('F1') ? symbol.indexOf('F1') : symbol.indexOf('T1');
      if (direct === -1 || indirect === -1) {
        throw new Error('F2/T2 and F1/T1 symbol should be defined for nD NMR SPECTRUM');
      }
      return {
        direct,
        indirect,
        dependent: 3 - direct - indirect
      };
    }
    function checkMandatoryParameters(meta) {
      const list = ['SFO1', 'SFO2', 'NUC1', 'NUC2'];
      for (const key of list) {
        if (!meta[key]) {
          throw new Error(`${key} in options.meta should be defined`);
        }
      }
    }
    function calculateFactor(data, minMax) {
      return isAnyArray(data[0]) ? getBestFactorMatrix(data, {
        minMax
      }) : getBestFactor(data, {
        minMax
      });
    }

    exports.from1DNMRVariables = from1DNMRVariables;
    exports.from2DNMRVariables = from2DNMRVariables;
    exports.fromJSON = fromJSON;
    exports.fromVariables = fromVariables;

    Object.defineProperty(exports, '__esModule', { value: true });

}));
//# sourceMappingURL=convert-to-jcamp.js.map
