/**
 * xps-analysis - XPS analysis
 * @version v0.10.0
 * @link https://github.com/cheminfo/xps-analysis#readme
 * @license MIT
 */
(function (global, factory) {
	typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
	typeof define === 'function' && define.amd ? define(['exports'], factory) :
	(global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.XPS = {}));
})(this, (function (exports) { 'use strict';

	var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

	function getAugmentedNamespace(n) {
		if (n.__esModule) return n;
		var a = Object.defineProperty({}, '__esModule', {value: true});
		Object.keys(n).forEach(function (k) {
			var d = Object.getOwnPropertyDescriptor(n, k);
			Object.defineProperty(a, k, d.get ? d : {
				enumerable: true,
				get: function () {
					return n[k];
				}
			});
		});
		return a;
	}

	/**
	 * Create an array with numbers between "from" and "to" of length "length"
	 *
	 * @param options - options
	 * @return - array of distributed numbers between "from" and "to"
	 */
	function createFromToArray(options = {}) {
	  const {
	    from = 0,
	    to = 1,
	    length = 1000,
	    includeFrom = true,
	    includeTo = true,
	    distribution = 'uniform'
	  } = options;
	  const array = new Float64Array(length);
	  let div = length;
	  if (includeFrom && includeTo) {
	    div = length - 1;
	  } else if (!includeFrom && includeTo || includeFrom && !includeTo) {
	    div = length;
	  } else if (!includeFrom && !includeTo) {
	    div = length + 1;
	  }
	  const delta = (to - from) / div;
	  if (distribution === 'uniform') {
	    if (includeFrom) {
	      let index = 0;
	      while (index < length) {
	        array[index] = from + delta * index;
	        index++;
	      }
	    } else {
	      let index = 0;
	      while (index < length) {
	        array[index] = from + delta * (index + 1);
	        index++;
	      }
	    }
	  } else if (distribution === 'log') {
	    const base = (to / from) ** (1 / div);
	    const firstExponent = Math.log(from) / Math.log(base);
	    if (includeFrom) {
	      let index = 0;
	      while (index < length) {
	        array[index] = base ** (firstExponent + index);
	        index++;
	      }
	    } else {
	      let index = 0;
	      while (index < length) {
	        array[index] = base ** (firstExponent + index + 1);
	        index++;
	      }
	    }
	  } else {
	    throw new Error('Please choose for the distribution either uniform or log. By default the distribution chosen is uniform.');
	  }
	  return array;
	}

	// eslint-disable-next-line @typescript-eslint/unbound-method
	const toString = Object.prototype.toString;
	/**
	 * Checks if an object is an instance of an Array (array or typed array, except those that contain bigint values).
	 *
	 * @param value - Object to check.
	 * @returns True if the object is an array or a typed array.
	 */
	function isAnyArray$1(value) {
	  const tag = toString.call(value);
	  return tag.endsWith('Array]') && !tag.includes('Big');
	}

	var libEsm = /*#__PURE__*/Object.freeze({
		__proto__: null,
		isAnyArray: isAnyArray$1
	});

	/**
	 * Calculates the median of an array
	 *
	 * @param input - Array containing values
	 * @returns - median
	 */
	function xMedian(input) {
	  if (!isAnyArray$1(input)) {
	    throw new TypeError('input must be an array');
	  }
	  if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }
	  const array = input.slice();
	  let low = 0;
	  let high = array.length - 1;
	  let middle = 0;
	  let currentLow = 0;
	  let currentHigh = 0;
	  const median = calcMiddle(low, high);
	  while (true) {
	    if (high <= low) {
	      return array[median];
	    }
	    if (high === low + 1) {
	      if (array[low] > array[high]) {
	        swap(array, low, high);
	      }
	      return array[median];
	    }
	    // Find median of low, middle and high items; swap into position low
	    middle = calcMiddle(low, high);
	    if (array[middle] > array[high]) swap(array, middle, high);
	    if (array[low] > array[high]) swap(array, low, high);
	    if (array[middle] > array[low]) swap(array, middle, low);
	    // Swap low item (now in position middle) into position (low+1)
	    swap(array, middle, low + 1);
	    // Nibble from each end towards middle, swapping items when stuck
	    currentLow = low + 1;
	    currentHigh = high;
	    while (true) {
	      do currentLow++; while (array[low] > array[currentLow]);
	      do currentHigh--; while (array[currentHigh] > array[low]);
	      if (currentHigh < currentLow) {
	        break;
	      }
	      swap(array, currentLow, currentHigh);
	    }
	    // Swap middle item (in position low) back into correct position
	    swap(array, low, currentHigh);
	    // Re-set active partition
	    if (currentHigh <= median) {
	      low = currentLow;
	    }
	    if (currentHigh >= median) {
	      high = currentHigh - 1;
	    }
	  }
	}
	function swap(array, i, j) {
	  const temp = array[j];
	  array[j] = array[i];
	  array[i] = temp;
	}
	function calcMiddle(i, j) {
	  return Math.floor((i + j) / 2);
	}

	/**
	 * This function xAdd the first array by the second array or a constant value to each element of the first array
	 *
	 * @param array1 - the first array
	 * @param array2 - the second array or number
	 */
	function xAdd(array1, array2) {
	  let isConstant = false;
	  let constant = 0;
	  if (isAnyArray$1(array2)) {
	    if (array1.length !== array2.length) {
	      throw new Error('xAdd: size of array1 and array2 must be identical');
	    }
	  } else {
	    isConstant = true;
	    constant = array2;
	  }
	  const array3 = new Float64Array(array1.length);
	  if (isConstant) {
	    for (let i = 0; i < array1.length; i++) {
	      array3[i] = array1[i] + constant;
	    }
	  } else {
	    for (let i = 0; i < array1.length; i++) {
	      array3[i] = array1[i] + array2[i];
	    }
	  }
	  return array3;
	}

	/**
	 * Will apply a function on each element of the array described as a string
	 * By default we will use as variable 'x'
	 * In front of sequence of lowercase we will add 'Math.'. This allows to write
	 * `sin(x) + cos(x)` and it will be replace internally by (x) => (Math.sin(x) + Math.cos(x))
	 * @param array
	 * @param options
	 * @returns
	 */
	function xApplyFunctionStr(array, options = {}) {
	  const {
	    variableLabel = 'x',
	    fctString = variableLabel
	  } = options;
	  const fct = new Function(variableLabel, `return Number(${fctString.replace(/(?<before>^|\W)(?<after>[\da-z]{2,}\()/g, '$<before>Math.$<after>').replace(/Math\.Math/g, 'Math')})`);
	  const toReturn = Float64Array.from(array);
	  for (let i = 0; i < array.length; i++) {
	    toReturn[i] = fct(array[i]);
	    if (Number.isNaN(toReturn[i])) {
	      throw new Error(`The callback ${fctString} does not return a number: ${array[i]}`);
	    }
	  }
	  return toReturn;
	}

	/**
	 * This function
	 * @param output - undefined or a new array
	 * @param length - length of the output array
	 * @returns
	 */
	function getOutputArray(output, length) {
	  if (typeof output !== 'undefined') {
	    if (!isAnyArray$1(output)) {
	      throw new TypeError('output option must be an array if specified');
	    }
	    if (output.length !== length) {
	      throw new TypeError('the output array does not have the correct length');
	    }
	    return output;
	  } else {
	    return new Float64Array(length);
	  }
	}

	/**
	 * Checks if input is of type array
	 *
	 * @param input - input
	 */
	function xCheck(input, options = {}) {
	  const {
	    minLength
	  } = options;
	  if (!isAnyArray$1(input)) {
	    throw new TypeError('input must be an array');
	  }
	  if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }
	  if (minLength && input.length < minLength) {
	    throw new Error(`input must have a length of at least ${minLength}`);
	  }
	}

	var matrix = {};

	var require$$0 = /*@__PURE__*/getAugmentedNamespace(libEsm);

	function max(input) {
	  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  if (!isAnyArray$1(input)) {
	    throw new TypeError('input must be an array');
	  }
	  if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }
	  var _options$fromIndex = options.fromIndex,
	    fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
	    _options$toIndex = options.toIndex,
	    toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;
	  if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
	    throw new Error('fromIndex must be a positive integer smaller than length');
	  }
	  if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
	    throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
	  }
	  var maxValue = input[fromIndex];
	  for (var i = fromIndex + 1; i < toIndex; i++) {
	    if (input[i] > maxValue) maxValue = input[i];
	  }
	  return maxValue;
	}

	function min(input) {
	  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  if (!isAnyArray$1(input)) {
	    throw new TypeError('input must be an array');
	  }
	  if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }
	  var _options$fromIndex = options.fromIndex,
	    fromIndex = _options$fromIndex === void 0 ? 0 : _options$fromIndex,
	    _options$toIndex = options.toIndex,
	    toIndex = _options$toIndex === void 0 ? input.length : _options$toIndex;
	  if (fromIndex < 0 || fromIndex >= input.length || !Number.isInteger(fromIndex)) {
	    throw new Error('fromIndex must be a positive integer smaller than length');
	  }
	  if (toIndex <= fromIndex || toIndex > input.length || !Number.isInteger(toIndex)) {
	    throw new Error('toIndex must be an integer greater than fromIndex and at most equal to length');
	  }
	  var minValue = input[fromIndex];
	  for (var i = fromIndex + 1; i < toIndex; i++) {
	    if (input[i] < minValue) minValue = input[i];
	  }
	  return minValue;
	}

	function rescale$2(input) {
	  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  if (!isAnyArray$1(input)) {
	    throw new TypeError('input must be an array');
	  } else if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }
	  var output;
	  if (options.output !== undefined) {
	    if (!isAnyArray$1(options.output)) {
	      throw new TypeError('output option must be an array if specified');
	    }
	    output = options.output;
	  } else {
	    output = new Array(input.length);
	  }
	  var currentMin = min(input);
	  var currentMax = max(input);
	  if (currentMin === currentMax) {
	    throw new RangeError('minimum and maximum input values are equal. Cannot rescale a constant array');
	  }
	  var _options$min = options.min,
	    minValue = _options$min === void 0 ? options.autoMinMax ? currentMin : 0 : _options$min,
	    _options$max = options.max,
	    maxValue = _options$max === void 0 ? options.autoMinMax ? currentMax : 1 : _options$max;
	  if (minValue >= maxValue) {
	    throw new RangeError('min option must be smaller than max option');
	  }
	  var factor = (maxValue - minValue) / (currentMax - currentMin);
	  for (var i = 0; i < input.length; i++) {
	    output[i] = (input[i] - currentMin) * factor + minValue;
	  }
	  return output;
	}

	var libEs6 = /*#__PURE__*/Object.freeze({
		__proto__: null,
		'default': rescale$2
	});

	var require$$1 = /*@__PURE__*/getAugmentedNamespace(libEs6);

	Object.defineProperty(matrix, '__esModule', {
	  value: true
	});
	var isAnyArray = require$$0;
	var rescale$1 = require$$1;
	const indent = ' '.repeat(2);
	const indentData = ' '.repeat(4);
	function inspectMatrix() {
	  return inspectMatrixWithOptions(this);
	}
	function inspectMatrixWithOptions(matrix, options = {}) {
	  const {
	    maxRows = 15,
	    maxColumns = 10,
	    maxNumSize = 8,
	    padMinus = 'auto'
	  } = options;
	  return `${matrix.constructor.name} {
${indent}[
${indentData}${inspectData(matrix, maxRows, maxColumns, maxNumSize, padMinus)}
${indent}]
${indent}rows: ${matrix.rows}
${indent}columns: ${matrix.columns}
}`;
	}
	function inspectData(matrix, maxRows, maxColumns, maxNumSize, padMinus) {
	  const {
	    rows,
	    columns
	  } = matrix;
	  const maxI = Math.min(rows, maxRows);
	  const maxJ = Math.min(columns, maxColumns);
	  const result = [];
	  if (padMinus === 'auto') {
	    padMinus = false;
	    loop: for (let i = 0; i < maxI; i++) {
	      for (let j = 0; j < maxJ; j++) {
	        if (matrix.get(i, j) < 0) {
	          padMinus = true;
	          break loop;
	        }
	      }
	    }
	  }
	  for (let i = 0; i < maxI; i++) {
	    let line = [];
	    for (let j = 0; j < maxJ; j++) {
	      line.push(formatNumber(matrix.get(i, j), maxNumSize, padMinus));
	    }
	    result.push(`${line.join(' ')}`);
	  }
	  if (maxJ !== columns) {
	    result[result.length - 1] += ` ... ${columns - maxColumns} more columns`;
	  }
	  if (maxI !== rows) {
	    result.push(`... ${rows - maxRows} more rows`);
	  }
	  return result.join(`\n${indentData}`);
	}
	function formatNumber(num, maxNumSize, padMinus) {
	  return (num >= 0 && padMinus ? ` ${formatNumber2(num, maxNumSize - 1)}` : formatNumber2(num, maxNumSize)).padEnd(maxNumSize);
	}
	function formatNumber2(num, len) {
	  // small.length numbers should be as is
	  let str = num.toString();
	  if (str.length <= len) return str;

	  // (7)'0.00123' is better then (7)'1.23e-2'
	  // (8)'0.000123' is worse then (7)'1.23e-3',
	  let fix = num.toFixed(len);
	  if (fix.length > len) {
	    fix = num.toFixed(Math.max(0, len - (fix.length - len)));
	  }
	  if (fix.length <= len && !fix.startsWith('0.000') && !fix.startsWith('-0.000')) {
	    return fix;
	  }

	  // well, if it's still too long the user should've used longer numbers
	  let exp = num.toExponential(len);
	  if (exp.length > len) {
	    exp = num.toExponential(Math.max(0, len - (exp.length - len)));
	  }
	  return exp.slice(0);
	}
	function installMathOperations(AbstractMatrix, Matrix) {
	  AbstractMatrix.prototype.add = function add(value) {
	    if (typeof value === 'number') return this.addS(value);
	    return this.addM(value);
	  };
	  AbstractMatrix.prototype.addS = function addS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) + value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.addM = function addM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) + matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.add = function add(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.add(value);
	  };
	  AbstractMatrix.prototype.sub = function sub(value) {
	    if (typeof value === 'number') return this.subS(value);
	    return this.subM(value);
	  };
	  AbstractMatrix.prototype.subS = function subS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) - value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.subM = function subM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) - matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.sub = function sub(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.sub(value);
	  };
	  AbstractMatrix.prototype.subtract = AbstractMatrix.prototype.sub;
	  AbstractMatrix.prototype.subtractS = AbstractMatrix.prototype.subS;
	  AbstractMatrix.prototype.subtractM = AbstractMatrix.prototype.subM;
	  AbstractMatrix.subtract = AbstractMatrix.sub;
	  AbstractMatrix.prototype.mul = function mul(value) {
	    if (typeof value === 'number') return this.mulS(value);
	    return this.mulM(value);
	  };
	  AbstractMatrix.prototype.mulS = function mulS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) * value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.mulM = function mulM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) * matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.mul = function mul(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.mul(value);
	  };
	  AbstractMatrix.prototype.multiply = AbstractMatrix.prototype.mul;
	  AbstractMatrix.prototype.multiplyS = AbstractMatrix.prototype.mulS;
	  AbstractMatrix.prototype.multiplyM = AbstractMatrix.prototype.mulM;
	  AbstractMatrix.multiply = AbstractMatrix.mul;
	  AbstractMatrix.prototype.div = function div(value) {
	    if (typeof value === 'number') return this.divS(value);
	    return this.divM(value);
	  };
	  AbstractMatrix.prototype.divS = function divS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) / value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.divM = function divM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) / matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.div = function div(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.div(value);
	  };
	  AbstractMatrix.prototype.divide = AbstractMatrix.prototype.div;
	  AbstractMatrix.prototype.divideS = AbstractMatrix.prototype.divS;
	  AbstractMatrix.prototype.divideM = AbstractMatrix.prototype.divM;
	  AbstractMatrix.divide = AbstractMatrix.div;
	  AbstractMatrix.prototype.mod = function mod(value) {
	    if (typeof value === 'number') return this.modS(value);
	    return this.modM(value);
	  };
	  AbstractMatrix.prototype.modS = function modS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) % value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.modM = function modM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) % matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.mod = function mod(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.mod(value);
	  };
	  AbstractMatrix.prototype.modulus = AbstractMatrix.prototype.mod;
	  AbstractMatrix.prototype.modulusS = AbstractMatrix.prototype.modS;
	  AbstractMatrix.prototype.modulusM = AbstractMatrix.prototype.modM;
	  AbstractMatrix.modulus = AbstractMatrix.mod;
	  AbstractMatrix.prototype.and = function and(value) {
	    if (typeof value === 'number') return this.andS(value);
	    return this.andM(value);
	  };
	  AbstractMatrix.prototype.andS = function andS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) & value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.andM = function andM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) & matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.and = function and(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.and(value);
	  };
	  AbstractMatrix.prototype.or = function or(value) {
	    if (typeof value === 'number') return this.orS(value);
	    return this.orM(value);
	  };
	  AbstractMatrix.prototype.orS = function orS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) | value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.orM = function orM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) | matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.or = function or(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.or(value);
	  };
	  AbstractMatrix.prototype.xor = function xor(value) {
	    if (typeof value === 'number') return this.xorS(value);
	    return this.xorM(value);
	  };
	  AbstractMatrix.prototype.xorS = function xorS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) ^ value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.xorM = function xorM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) ^ matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.xor = function xor(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.xor(value);
	  };
	  AbstractMatrix.prototype.leftShift = function leftShift(value) {
	    if (typeof value === 'number') return this.leftShiftS(value);
	    return this.leftShiftM(value);
	  };
	  AbstractMatrix.prototype.leftShiftS = function leftShiftS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) << value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.leftShiftM = function leftShiftM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) << matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.leftShift = function leftShift(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.leftShift(value);
	  };
	  AbstractMatrix.prototype.signPropagatingRightShift = function signPropagatingRightShift(value) {
	    if (typeof value === 'number') return this.signPropagatingRightShiftS(value);
	    return this.signPropagatingRightShiftM(value);
	  };
	  AbstractMatrix.prototype.signPropagatingRightShiftS = function signPropagatingRightShiftS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) >> value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.signPropagatingRightShiftM = function signPropagatingRightShiftM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) >> matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.signPropagatingRightShift = function signPropagatingRightShift(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.signPropagatingRightShift(value);
	  };
	  AbstractMatrix.prototype.rightShift = function rightShift(value) {
	    if (typeof value === 'number') return this.rightShiftS(value);
	    return this.rightShiftM(value);
	  };
	  AbstractMatrix.prototype.rightShiftS = function rightShiftS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) >>> value);
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.rightShiftM = function rightShiftM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) >>> matrix.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.rightShift = function rightShift(matrix, value) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.rightShift(value);
	  };
	  AbstractMatrix.prototype.zeroFillRightShift = AbstractMatrix.prototype.rightShift;
	  AbstractMatrix.prototype.zeroFillRightShiftS = AbstractMatrix.prototype.rightShiftS;
	  AbstractMatrix.prototype.zeroFillRightShiftM = AbstractMatrix.prototype.rightShiftM;
	  AbstractMatrix.zeroFillRightShift = AbstractMatrix.rightShift;
	  AbstractMatrix.prototype.not = function not() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, ~this.get(i, j));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.not = function not(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.not();
	  };
	  AbstractMatrix.prototype.abs = function abs() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.abs(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.abs = function abs(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.abs();
	  };
	  AbstractMatrix.prototype.acos = function acos() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.acos(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.acos = function acos(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.acos();
	  };
	  AbstractMatrix.prototype.acosh = function acosh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.acosh(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.acosh = function acosh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.acosh();
	  };
	  AbstractMatrix.prototype.asin = function asin() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.asin(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.asin = function asin(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.asin();
	  };
	  AbstractMatrix.prototype.asinh = function asinh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.asinh(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.asinh = function asinh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.asinh();
	  };
	  AbstractMatrix.prototype.atan = function atan() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.atan(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.atan = function atan(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.atan();
	  };
	  AbstractMatrix.prototype.atanh = function atanh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.atanh(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.atanh = function atanh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.atanh();
	  };
	  AbstractMatrix.prototype.cbrt = function cbrt() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.cbrt(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.cbrt = function cbrt(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.cbrt();
	  };
	  AbstractMatrix.prototype.ceil = function ceil() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.ceil(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.ceil = function ceil(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.ceil();
	  };
	  AbstractMatrix.prototype.clz32 = function clz32() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.clz32(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.clz32 = function clz32(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.clz32();
	  };
	  AbstractMatrix.prototype.cos = function cos() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.cos(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.cos = function cos(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.cos();
	  };
	  AbstractMatrix.prototype.cosh = function cosh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.cosh(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.cosh = function cosh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.cosh();
	  };
	  AbstractMatrix.prototype.exp = function exp() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.exp(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.exp = function exp(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.exp();
	  };
	  AbstractMatrix.prototype.expm1 = function expm1() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.expm1(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.expm1 = function expm1(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.expm1();
	  };
	  AbstractMatrix.prototype.floor = function floor() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.floor(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.floor = function floor(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.floor();
	  };
	  AbstractMatrix.prototype.fround = function fround() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.fround(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.fround = function fround(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.fround();
	  };
	  AbstractMatrix.prototype.log = function log() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.log(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.log = function log(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.log();
	  };
	  AbstractMatrix.prototype.log1p = function log1p() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.log1p(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.log1p = function log1p(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.log1p();
	  };
	  AbstractMatrix.prototype.log10 = function log10() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.log10(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.log10 = function log10(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.log10();
	  };
	  AbstractMatrix.prototype.log2 = function log2() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.log2(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.log2 = function log2(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.log2();
	  };
	  AbstractMatrix.prototype.round = function round() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.round(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.round = function round(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.round();
	  };
	  AbstractMatrix.prototype.sign = function sign() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.sign(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.sign = function sign(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.sign();
	  };
	  AbstractMatrix.prototype.sin = function sin() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.sin(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.sin = function sin(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.sin();
	  };
	  AbstractMatrix.prototype.sinh = function sinh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.sinh(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.sinh = function sinh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.sinh();
	  };
	  AbstractMatrix.prototype.sqrt = function sqrt() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.sqrt(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.sqrt = function sqrt(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.sqrt();
	  };
	  AbstractMatrix.prototype.tan = function tan() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.tan(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.tan = function tan(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.tan();
	  };
	  AbstractMatrix.prototype.tanh = function tanh() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.tanh(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.tanh = function tanh(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.tanh();
	  };
	  AbstractMatrix.prototype.trunc = function trunc() {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.trunc(this.get(i, j)));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.trunc = function trunc(matrix) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.trunc();
	  };
	  AbstractMatrix.pow = function pow(matrix, arg0) {
	    const newMatrix = new Matrix(matrix);
	    return newMatrix.pow(arg0);
	  };
	  AbstractMatrix.prototype.pow = function pow(value) {
	    if (typeof value === 'number') return this.powS(value);
	    return this.powM(value);
	  };
	  AbstractMatrix.prototype.powS = function powS(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.pow(this.get(i, j), value));
	      }
	    }
	    return this;
	  };
	  AbstractMatrix.prototype.powM = function powM(matrix) {
	    matrix = Matrix.checkMatrix(matrix);
	    if (this.rows !== matrix.rows || this.columns !== matrix.columns) {
	      throw new RangeError('Matrices dimensions must be equal');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, Math.pow(this.get(i, j), matrix.get(i, j)));
	      }
	    }
	    return this;
	  };
	}

	/**
	 * @private
	 * Check that a row index is not out of bounds
	 * @param {Matrix} matrix
	 * @param {number} index
	 * @param {boolean} [outer]
	 */
	function checkRowIndex(matrix, index, outer) {
	  let max = outer ? matrix.rows : matrix.rows - 1;
	  if (index < 0 || index > max) {
	    throw new RangeError('Row index out of range');
	  }
	}

	/**
	 * @private
	 * Check that a column index is not out of bounds
	 * @param {Matrix} matrix
	 * @param {number} index
	 * @param {boolean} [outer]
	 */
	function checkColumnIndex(matrix, index, outer) {
	  let max = outer ? matrix.columns : matrix.columns - 1;
	  if (index < 0 || index > max) {
	    throw new RangeError('Column index out of range');
	  }
	}

	/**
	 * @private
	 * Check that the provided vector is an array with the right length
	 * @param {Matrix} matrix
	 * @param {Array|Matrix} vector
	 * @return {Array}
	 * @throws {RangeError}
	 */
	function checkRowVector(matrix, vector) {
	  if (vector.to1DArray) {
	    vector = vector.to1DArray();
	  }
	  if (vector.length !== matrix.columns) {
	    throw new RangeError('vector size must be the same as the number of columns');
	  }
	  return vector;
	}

	/**
	 * @private
	 * Check that the provided vector is an array with the right length
	 * @param {Matrix} matrix
	 * @param {Array|Matrix} vector
	 * @return {Array}
	 * @throws {RangeError}
	 */
	function checkColumnVector(matrix, vector) {
	  if (vector.to1DArray) {
	    vector = vector.to1DArray();
	  }
	  if (vector.length !== matrix.rows) {
	    throw new RangeError('vector size must be the same as the number of rows');
	  }
	  return vector;
	}
	function checkRowIndices(matrix, rowIndices) {
	  if (!isAnyArray.isAnyArray(rowIndices)) {
	    throw new TypeError('row indices must be an array');
	  }
	  for (let i = 0; i < rowIndices.length; i++) {
	    if (rowIndices[i] < 0 || rowIndices[i] >= matrix.rows) {
	      throw new RangeError('row indices are out of range');
	    }
	  }
	}
	function checkColumnIndices(matrix, columnIndices) {
	  if (!isAnyArray.isAnyArray(columnIndices)) {
	    throw new TypeError('column indices must be an array');
	  }
	  for (let i = 0; i < columnIndices.length; i++) {
	    if (columnIndices[i] < 0 || columnIndices[i] >= matrix.columns) {
	      throw new RangeError('column indices are out of range');
	    }
	  }
	}
	function checkRange(matrix, startRow, endRow, startColumn, endColumn) {
	  if (arguments.length !== 5) {
	    throw new RangeError('expected 4 arguments');
	  }
	  checkNumber('startRow', startRow);
	  checkNumber('endRow', endRow);
	  checkNumber('startColumn', startColumn);
	  checkNumber('endColumn', endColumn);
	  if (startRow > endRow || startColumn > endColumn || startRow < 0 || startRow >= matrix.rows || endRow < 0 || endRow >= matrix.rows || startColumn < 0 || startColumn >= matrix.columns || endColumn < 0 || endColumn >= matrix.columns) {
	    throw new RangeError('Submatrix indices are out of range');
	  }
	}
	function newArray(length, value = 0) {
	  let array = [];
	  for (let i = 0; i < length; i++) {
	    array.push(value);
	  }
	  return array;
	}
	function checkNumber(name, value) {
	  if (typeof value !== 'number') {
	    throw new TypeError(`${name} must be a number`);
	  }
	}
	function checkNonEmpty(matrix) {
	  if (matrix.isEmpty()) {
	    throw new Error('Empty matrix has no elements to index');
	  }
	}
	function sumByRow(matrix) {
	  let sum = newArray(matrix.rows);
	  for (let i = 0; i < matrix.rows; ++i) {
	    for (let j = 0; j < matrix.columns; ++j) {
	      sum[i] += matrix.get(i, j);
	    }
	  }
	  return sum;
	}
	function sumByColumn(matrix) {
	  let sum = newArray(matrix.columns);
	  for (let i = 0; i < matrix.rows; ++i) {
	    for (let j = 0; j < matrix.columns; ++j) {
	      sum[j] += matrix.get(i, j);
	    }
	  }
	  return sum;
	}
	function sumAll(matrix) {
	  let v = 0;
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      v += matrix.get(i, j);
	    }
	  }
	  return v;
	}
	function productByRow(matrix) {
	  let sum = newArray(matrix.rows, 1);
	  for (let i = 0; i < matrix.rows; ++i) {
	    for (let j = 0; j < matrix.columns; ++j) {
	      sum[i] *= matrix.get(i, j);
	    }
	  }
	  return sum;
	}
	function productByColumn(matrix) {
	  let sum = newArray(matrix.columns, 1);
	  for (let i = 0; i < matrix.rows; ++i) {
	    for (let j = 0; j < matrix.columns; ++j) {
	      sum[j] *= matrix.get(i, j);
	    }
	  }
	  return sum;
	}
	function productAll(matrix) {
	  let v = 1;
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      v *= matrix.get(i, j);
	    }
	  }
	  return v;
	}
	function varianceByRow(matrix, unbiased, mean) {
	  const rows = matrix.rows;
	  const cols = matrix.columns;
	  const variance = [];
	  for (let i = 0; i < rows; i++) {
	    let sum1 = 0;
	    let sum2 = 0;
	    let x = 0;
	    for (let j = 0; j < cols; j++) {
	      x = matrix.get(i, j) - mean[i];
	      sum1 += x;
	      sum2 += x * x;
	    }
	    if (unbiased) {
	      variance.push((sum2 - sum1 * sum1 / cols) / (cols - 1));
	    } else {
	      variance.push((sum2 - sum1 * sum1 / cols) / cols);
	    }
	  }
	  return variance;
	}
	function varianceByColumn(matrix, unbiased, mean) {
	  const rows = matrix.rows;
	  const cols = matrix.columns;
	  const variance = [];
	  for (let j = 0; j < cols; j++) {
	    let sum1 = 0;
	    let sum2 = 0;
	    let x = 0;
	    for (let i = 0; i < rows; i++) {
	      x = matrix.get(i, j) - mean[j];
	      sum1 += x;
	      sum2 += x * x;
	    }
	    if (unbiased) {
	      variance.push((sum2 - sum1 * sum1 / rows) / (rows - 1));
	    } else {
	      variance.push((sum2 - sum1 * sum1 / rows) / rows);
	    }
	  }
	  return variance;
	}
	function varianceAll(matrix, unbiased, mean) {
	  const rows = matrix.rows;
	  const cols = matrix.columns;
	  const size = rows * cols;
	  let sum1 = 0;
	  let sum2 = 0;
	  let x = 0;
	  for (let i = 0; i < rows; i++) {
	    for (let j = 0; j < cols; j++) {
	      x = matrix.get(i, j) - mean;
	      sum1 += x;
	      sum2 += x * x;
	    }
	  }
	  if (unbiased) {
	    return (sum2 - sum1 * sum1 / size) / (size - 1);
	  } else {
	    return (sum2 - sum1 * sum1 / size) / size;
	  }
	}
	function centerByRow(matrix, mean) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) - mean[i]);
	    }
	  }
	}
	function centerByColumn(matrix, mean) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) - mean[j]);
	    }
	  }
	}
	function centerAll(matrix, mean) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) - mean);
	    }
	  }
	}
	function getScaleByRow(matrix) {
	  const scale = [];
	  for (let i = 0; i < matrix.rows; i++) {
	    let sum = 0;
	    for (let j = 0; j < matrix.columns; j++) {
	      sum += Math.pow(matrix.get(i, j), 2) / (matrix.columns - 1);
	    }
	    scale.push(Math.sqrt(sum));
	  }
	  return scale;
	}
	function scaleByRow(matrix, scale) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) / scale[i]);
	    }
	  }
	}
	function getScaleByColumn(matrix) {
	  const scale = [];
	  for (let j = 0; j < matrix.columns; j++) {
	    let sum = 0;
	    for (let i = 0; i < matrix.rows; i++) {
	      sum += Math.pow(matrix.get(i, j), 2) / (matrix.rows - 1);
	    }
	    scale.push(Math.sqrt(sum));
	  }
	  return scale;
	}
	function scaleByColumn(matrix, scale) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) / scale[j]);
	    }
	  }
	}
	function getScaleAll(matrix) {
	  const divider = matrix.size - 1;
	  let sum = 0;
	  for (let j = 0; j < matrix.columns; j++) {
	    for (let i = 0; i < matrix.rows; i++) {
	      sum += Math.pow(matrix.get(i, j), 2) / divider;
	    }
	  }
	  return Math.sqrt(sum);
	}
	function scaleAll(matrix, scale) {
	  for (let i = 0; i < matrix.rows; i++) {
	    for (let j = 0; j < matrix.columns; j++) {
	      matrix.set(i, j, matrix.get(i, j) / scale);
	    }
	  }
	}
	class AbstractMatrix {
	  static from1DArray(newRows, newColumns, newData) {
	    let length = newRows * newColumns;
	    if (length !== newData.length) {
	      throw new RangeError('data length does not match given dimensions');
	    }
	    let newMatrix = new Matrix$1(newRows, newColumns);
	    for (let row = 0; row < newRows; row++) {
	      for (let column = 0; column < newColumns; column++) {
	        newMatrix.set(row, column, newData[row * newColumns + column]);
	      }
	    }
	    return newMatrix;
	  }
	  static rowVector(newData) {
	    let vector = new Matrix$1(1, newData.length);
	    for (let i = 0; i < newData.length; i++) {
	      vector.set(0, i, newData[i]);
	    }
	    return vector;
	  }
	  static columnVector(newData) {
	    let vector = new Matrix$1(newData.length, 1);
	    for (let i = 0; i < newData.length; i++) {
	      vector.set(i, 0, newData[i]);
	    }
	    return vector;
	  }
	  static zeros(rows, columns) {
	    return new Matrix$1(rows, columns);
	  }
	  static ones(rows, columns) {
	    return new Matrix$1(rows, columns).fill(1);
	  }
	  static rand(rows, columns, options = {}) {
	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }
	    const {
	      random = Math.random
	    } = options;
	    let matrix = new Matrix$1(rows, columns);
	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        matrix.set(i, j, random());
	      }
	    }
	    return matrix;
	  }
	  static randInt(rows, columns, options = {}) {
	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }
	    const {
	      min = 0,
	      max = 1000,
	      random = Math.random
	    } = options;
	    if (!Number.isInteger(min)) throw new TypeError('min must be an integer');
	    if (!Number.isInteger(max)) throw new TypeError('max must be an integer');
	    if (min >= max) throw new RangeError('min must be smaller than max');
	    let interval = max - min;
	    let matrix = new Matrix$1(rows, columns);
	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        let value = min + Math.round(random() * interval);
	        matrix.set(i, j, value);
	      }
	    }
	    return matrix;
	  }
	  static eye(rows, columns, value) {
	    if (columns === undefined) columns = rows;
	    if (value === undefined) value = 1;
	    let min = Math.min(rows, columns);
	    let matrix = this.zeros(rows, columns);
	    for (let i = 0; i < min; i++) {
	      matrix.set(i, i, value);
	    }
	    return matrix;
	  }
	  static diag(data, rows, columns) {
	    let l = data.length;
	    if (rows === undefined) rows = l;
	    if (columns === undefined) columns = rows;
	    let min = Math.min(l, rows, columns);
	    let matrix = this.zeros(rows, columns);
	    for (let i = 0; i < min; i++) {
	      matrix.set(i, i, data[i]);
	    }
	    return matrix;
	  }
	  static min(matrix1, matrix2) {
	    matrix1 = this.checkMatrix(matrix1);
	    matrix2 = this.checkMatrix(matrix2);
	    let rows = matrix1.rows;
	    let columns = matrix1.columns;
	    let result = new Matrix$1(rows, columns);
	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        result.set(i, j, Math.min(matrix1.get(i, j), matrix2.get(i, j)));
	      }
	    }
	    return result;
	  }
	  static max(matrix1, matrix2) {
	    matrix1 = this.checkMatrix(matrix1);
	    matrix2 = this.checkMatrix(matrix2);
	    let rows = matrix1.rows;
	    let columns = matrix1.columns;
	    let result = new this(rows, columns);
	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        result.set(i, j, Math.max(matrix1.get(i, j), matrix2.get(i, j)));
	      }
	    }
	    return result;
	  }
	  static checkMatrix(value) {
	    return AbstractMatrix.isMatrix(value) ? value : new Matrix$1(value);
	  }
	  static isMatrix(value) {
	    return value != null && value.klass === 'Matrix';
	  }
	  get size() {
	    return this.rows * this.columns;
	  }
	  apply(callback) {
	    if (typeof callback !== 'function') {
	      throw new TypeError('callback must be a function');
	    }
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        callback.call(this, i, j);
	      }
	    }
	    return this;
	  }
	  to1DArray() {
	    let array = [];
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        array.push(this.get(i, j));
	      }
	    }
	    return array;
	  }
	  to2DArray() {
	    let copy = [];
	    for (let i = 0; i < this.rows; i++) {
	      copy.push([]);
	      for (let j = 0; j < this.columns; j++) {
	        copy[i].push(this.get(i, j));
	      }
	    }
	    return copy;
	  }
	  toJSON() {
	    return this.to2DArray();
	  }
	  isRowVector() {
	    return this.rows === 1;
	  }
	  isColumnVector() {
	    return this.columns === 1;
	  }
	  isVector() {
	    return this.rows === 1 || this.columns === 1;
	  }
	  isSquare() {
	    return this.rows === this.columns;
	  }
	  isEmpty() {
	    return this.rows === 0 || this.columns === 0;
	  }
	  isSymmetric() {
	    if (this.isSquare()) {
	      for (let i = 0; i < this.rows; i++) {
	        for (let j = 0; j <= i; j++) {
	          if (this.get(i, j) !== this.get(j, i)) {
	            return false;
	          }
	        }
	      }
	      return true;
	    }
	    return false;
	  }
	  isDistance() {
	    if (!this.isSymmetric()) return false;
	    for (let i = 0; i < this.rows; i++) {
	      if (this.get(i, i) !== 0) return false;
	    }
	    return true;
	  }
	  isEchelonForm() {
	    let i = 0;
	    let j = 0;
	    let previousColumn = -1;
	    let isEchelonForm = true;
	    let checked = false;
	    while (i < this.rows && isEchelonForm) {
	      j = 0;
	      checked = false;
	      while (j < this.columns && checked === false) {
	        if (this.get(i, j) === 0) {
	          j++;
	        } else if (this.get(i, j) === 1 && j > previousColumn) {
	          checked = true;
	          previousColumn = j;
	        } else {
	          isEchelonForm = false;
	          checked = true;
	        }
	      }
	      i++;
	    }
	    return isEchelonForm;
	  }
	  isReducedEchelonForm() {
	    let i = 0;
	    let j = 0;
	    let previousColumn = -1;
	    let isReducedEchelonForm = true;
	    let checked = false;
	    while (i < this.rows && isReducedEchelonForm) {
	      j = 0;
	      checked = false;
	      while (j < this.columns && checked === false) {
	        if (this.get(i, j) === 0) {
	          j++;
	        } else if (this.get(i, j) === 1 && j > previousColumn) {
	          checked = true;
	          previousColumn = j;
	        } else {
	          isReducedEchelonForm = false;
	          checked = true;
	        }
	      }
	      for (let k = j + 1; k < this.rows; k++) {
	        if (this.get(i, k) !== 0) {
	          isReducedEchelonForm = false;
	        }
	      }
	      i++;
	    }
	    return isReducedEchelonForm;
	  }
	  echelonForm() {
	    let result = this.clone();
	    let h = 0;
	    let k = 0;
	    while (h < result.rows && k < result.columns) {
	      let iMax = h;
	      for (let i = h; i < result.rows; i++) {
	        if (result.get(i, k) > result.get(iMax, k)) {
	          iMax = i;
	        }
	      }
	      if (result.get(iMax, k) === 0) {
	        k++;
	      } else {
	        result.swapRows(h, iMax);
	        let tmp = result.get(h, k);
	        for (let j = k; j < result.columns; j++) {
	          result.set(h, j, result.get(h, j) / tmp);
	        }
	        for (let i = h + 1; i < result.rows; i++) {
	          let factor = result.get(i, k) / result.get(h, k);
	          result.set(i, k, 0);
	          for (let j = k + 1; j < result.columns; j++) {
	            result.set(i, j, result.get(i, j) - result.get(h, j) * factor);
	          }
	        }
	        h++;
	        k++;
	      }
	    }
	    return result;
	  }
	  reducedEchelonForm() {
	    let result = this.echelonForm();
	    let m = result.columns;
	    let n = result.rows;
	    let h = n - 1;
	    while (h >= 0) {
	      if (result.maxRow(h) === 0) {
	        h--;
	      } else {
	        let p = 0;
	        let pivot = false;
	        while (p < n && pivot === false) {
	          if (result.get(h, p) === 1) {
	            pivot = true;
	          } else {
	            p++;
	          }
	        }
	        for (let i = 0; i < h; i++) {
	          let factor = result.get(i, p);
	          for (let j = p; j < m; j++) {
	            let tmp = result.get(i, j) - factor * result.get(h, j);
	            result.set(i, j, tmp);
	          }
	        }
	        h--;
	      }
	    }
	    return result;
	  }
	  set() {
	    throw new Error('set method is unimplemented');
	  }
	  get() {
	    throw new Error('get method is unimplemented');
	  }
	  repeat(options = {}) {
	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }
	    const {
	      rows = 1,
	      columns = 1
	    } = options;
	    if (!Number.isInteger(rows) || rows <= 0) {
	      throw new TypeError('rows must be a positive integer');
	    }
	    if (!Number.isInteger(columns) || columns <= 0) {
	      throw new TypeError('columns must be a positive integer');
	    }
	    let matrix = new Matrix$1(this.rows * rows, this.columns * columns);
	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        matrix.setSubMatrix(this, this.rows * i, this.columns * j);
	      }
	    }
	    return matrix;
	  }
	  fill(value) {
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, value);
	      }
	    }
	    return this;
	  }
	  neg() {
	    return this.mulS(-1);
	  }
	  getRow(index) {
	    checkRowIndex(this, index);
	    let row = [];
	    for (let i = 0; i < this.columns; i++) {
	      row.push(this.get(index, i));
	    }
	    return row;
	  }
	  getRowVector(index) {
	    return Matrix$1.rowVector(this.getRow(index));
	  }
	  setRow(index, array) {
	    checkRowIndex(this, index);
	    array = checkRowVector(this, array);
	    for (let i = 0; i < this.columns; i++) {
	      this.set(index, i, array[i]);
	    }
	    return this;
	  }
	  swapRows(row1, row2) {
	    checkRowIndex(this, row1);
	    checkRowIndex(this, row2);
	    for (let i = 0; i < this.columns; i++) {
	      let temp = this.get(row1, i);
	      this.set(row1, i, this.get(row2, i));
	      this.set(row2, i, temp);
	    }
	    return this;
	  }
	  getColumn(index) {
	    checkColumnIndex(this, index);
	    let column = [];
	    for (let i = 0; i < this.rows; i++) {
	      column.push(this.get(i, index));
	    }
	    return column;
	  }
	  getColumnVector(index) {
	    return Matrix$1.columnVector(this.getColumn(index));
	  }
	  setColumn(index, array) {
	    checkColumnIndex(this, index);
	    array = checkColumnVector(this, array);
	    for (let i = 0; i < this.rows; i++) {
	      this.set(i, index, array[i]);
	    }
	    return this;
	  }
	  swapColumns(column1, column2) {
	    checkColumnIndex(this, column1);
	    checkColumnIndex(this, column2);
	    for (let i = 0; i < this.rows; i++) {
	      let temp = this.get(i, column1);
	      this.set(i, column1, this.get(i, column2));
	      this.set(i, column2, temp);
	    }
	    return this;
	  }
	  addRowVector(vector) {
	    vector = checkRowVector(this, vector);
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) + vector[j]);
	      }
	    }
	    return this;
	  }
	  subRowVector(vector) {
	    vector = checkRowVector(this, vector);
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) - vector[j]);
	      }
	    }
	    return this;
	  }
	  mulRowVector(vector) {
	    vector = checkRowVector(this, vector);
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) * vector[j]);
	      }
	    }
	    return this;
	  }
	  divRowVector(vector) {
	    vector = checkRowVector(this, vector);
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) / vector[j]);
	      }
	    }
	    return this;
	  }
	  addColumnVector(vector) {
	    vector = checkColumnVector(this, vector);
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) + vector[i]);
	      }
	    }
	    return this;
	  }
	  subColumnVector(vector) {
	    vector = checkColumnVector(this, vector);
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) - vector[i]);
	      }
	    }
	    return this;
	  }
	  mulColumnVector(vector) {
	    vector = checkColumnVector(this, vector);
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) * vector[i]);
	      }
	    }
	    return this;
	  }
	  divColumnVector(vector) {
	    vector = checkColumnVector(this, vector);
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        this.set(i, j, this.get(i, j) / vector[i]);
	      }
	    }
	    return this;
	  }
	  mulRow(index, value) {
	    checkRowIndex(this, index);
	    for (let i = 0; i < this.columns; i++) {
	      this.set(index, i, this.get(index, i) * value);
	    }
	    return this;
	  }
	  mulColumn(index, value) {
	    checkColumnIndex(this, index);
	    for (let i = 0; i < this.rows; i++) {
	      this.set(i, index, this.get(i, index) * value);
	    }
	    return this;
	  }
	  max(by) {
	    if (this.isEmpty()) {
	      return NaN;
	    }
	    switch (by) {
	      case 'row':
	        {
	          const max = new Array(this.rows).fill(Number.NEGATIVE_INFINITY);
	          for (let row = 0; row < this.rows; row++) {
	            for (let column = 0; column < this.columns; column++) {
	              if (this.get(row, column) > max[row]) {
	                max[row] = this.get(row, column);
	              }
	            }
	          }
	          return max;
	        }
	      case 'column':
	        {
	          const max = new Array(this.columns).fill(Number.NEGATIVE_INFINITY);
	          for (let row = 0; row < this.rows; row++) {
	            for (let column = 0; column < this.columns; column++) {
	              if (this.get(row, column) > max[column]) {
	                max[column] = this.get(row, column);
	              }
	            }
	          }
	          return max;
	        }
	      case undefined:
	        {
	          let max = this.get(0, 0);
	          for (let row = 0; row < this.rows; row++) {
	            for (let column = 0; column < this.columns; column++) {
	              if (this.get(row, column) > max) {
	                max = this.get(row, column);
	              }
	            }
	          }
	          return max;
	        }
	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }
	  maxIndex() {
	    checkNonEmpty(this);
	    let v = this.get(0, 0);
	    let idx = [0, 0];
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        if (this.get(i, j) > v) {
	          v = this.get(i, j);
	          idx[0] = i;
	          idx[1] = j;
	        }
	      }
	    }
	    return idx;
	  }
	  min(by) {
	    if (this.isEmpty()) {
	      return NaN;
	    }
	    switch (by) {
	      case 'row':
	        {
	          const min = new Array(this.rows).fill(Number.POSITIVE_INFINITY);
	          for (let row = 0; row < this.rows; row++) {
	            for (let column = 0; column < this.columns; column++) {
	              if (this.get(row, column) < min[row]) {
	                min[row] = this.get(row, column);
	              }
	            }
	          }
	          return min;
	        }
	      case 'column':
	        {
	          const min = new Array(this.columns).fill(Number.POSITIVE_INFINITY);
	          for (let row = 0; row < this.rows; row++) {
	            for (let column = 0; column < this.columns; column++) {
	              if (this.get(row, column) < min[column]) {
	                min[column] = this.get(row, column);
	              }
	            }
	          }
	          return min;
	        }
	      case undefined:
	        {
	          let min = this.get(0, 0);
	          for (let row = 0; row < this.rows; row++) {
	            for (let column = 0; column < this.columns; column++) {
	              if (this.get(row, column) < min) {
	                min = this.get(row, column);
	              }
	            }
	          }
	          return min;
	        }
	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }
	  minIndex() {
	    checkNonEmpty(this);
	    let v = this.get(0, 0);
	    let idx = [0, 0];
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        if (this.get(i, j) < v) {
	          v = this.get(i, j);
	          idx[0] = i;
	          idx[1] = j;
	        }
	      }
	    }
	    return idx;
	  }
	  maxRow(row) {
	    checkRowIndex(this, row);
	    if (this.isEmpty()) {
	      return NaN;
	    }
	    let v = this.get(row, 0);
	    for (let i = 1; i < this.columns; i++) {
	      if (this.get(row, i) > v) {
	        v = this.get(row, i);
	      }
	    }
	    return v;
	  }
	  maxRowIndex(row) {
	    checkRowIndex(this, row);
	    checkNonEmpty(this);
	    let v = this.get(row, 0);
	    let idx = [row, 0];
	    for (let i = 1; i < this.columns; i++) {
	      if (this.get(row, i) > v) {
	        v = this.get(row, i);
	        idx[1] = i;
	      }
	    }
	    return idx;
	  }
	  minRow(row) {
	    checkRowIndex(this, row);
	    if (this.isEmpty()) {
	      return NaN;
	    }
	    let v = this.get(row, 0);
	    for (let i = 1; i < this.columns; i++) {
	      if (this.get(row, i) < v) {
	        v = this.get(row, i);
	      }
	    }
	    return v;
	  }
	  minRowIndex(row) {
	    checkRowIndex(this, row);
	    checkNonEmpty(this);
	    let v = this.get(row, 0);
	    let idx = [row, 0];
	    for (let i = 1; i < this.columns; i++) {
	      if (this.get(row, i) < v) {
	        v = this.get(row, i);
	        idx[1] = i;
	      }
	    }
	    return idx;
	  }
	  maxColumn(column) {
	    checkColumnIndex(this, column);
	    if (this.isEmpty()) {
	      return NaN;
	    }
	    let v = this.get(0, column);
	    for (let i = 1; i < this.rows; i++) {
	      if (this.get(i, column) > v) {
	        v = this.get(i, column);
	      }
	    }
	    return v;
	  }
	  maxColumnIndex(column) {
	    checkColumnIndex(this, column);
	    checkNonEmpty(this);
	    let v = this.get(0, column);
	    let idx = [0, column];
	    for (let i = 1; i < this.rows; i++) {
	      if (this.get(i, column) > v) {
	        v = this.get(i, column);
	        idx[0] = i;
	      }
	    }
	    return idx;
	  }
	  minColumn(column) {
	    checkColumnIndex(this, column);
	    if (this.isEmpty()) {
	      return NaN;
	    }
	    let v = this.get(0, column);
	    for (let i = 1; i < this.rows; i++) {
	      if (this.get(i, column) < v) {
	        v = this.get(i, column);
	      }
	    }
	    return v;
	  }
	  minColumnIndex(column) {
	    checkColumnIndex(this, column);
	    checkNonEmpty(this);
	    let v = this.get(0, column);
	    let idx = [0, column];
	    for (let i = 1; i < this.rows; i++) {
	      if (this.get(i, column) < v) {
	        v = this.get(i, column);
	        idx[0] = i;
	      }
	    }
	    return idx;
	  }
	  diag() {
	    let min = Math.min(this.rows, this.columns);
	    let diag = [];
	    for (let i = 0; i < min; i++) {
	      diag.push(this.get(i, i));
	    }
	    return diag;
	  }
	  norm(type = 'frobenius') {
	    switch (type) {
	      case 'max':
	        return this.max();
	      case 'frobenius':
	        return Math.sqrt(this.dot(this));
	      default:
	        throw new RangeError(`unknown norm type: ${type}`);
	    }
	  }
	  cumulativeSum() {
	    let sum = 0;
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        sum += this.get(i, j);
	        this.set(i, j, sum);
	      }
	    }
	    return this;
	  }
	  dot(vector2) {
	    if (AbstractMatrix.isMatrix(vector2)) vector2 = vector2.to1DArray();
	    let vector1 = this.to1DArray();
	    if (vector1.length !== vector2.length) {
	      throw new RangeError('vectors do not have the same size');
	    }
	    let dot = 0;
	    for (let i = 0; i < vector1.length; i++) {
	      dot += vector1[i] * vector2[i];
	    }
	    return dot;
	  }
	  mmul(other) {
	    other = Matrix$1.checkMatrix(other);
	    let m = this.rows;
	    let n = this.columns;
	    let p = other.columns;
	    let result = new Matrix$1(m, p);
	    let Bcolj = new Float64Array(n);
	    for (let j = 0; j < p; j++) {
	      for (let k = 0; k < n; k++) {
	        Bcolj[k] = other.get(k, j);
	      }
	      for (let i = 0; i < m; i++) {
	        let s = 0;
	        for (let k = 0; k < n; k++) {
	          s += this.get(i, k) * Bcolj[k];
	        }
	        result.set(i, j, s);
	      }
	    }
	    return result;
	  }
	  strassen2x2(other) {
	    other = Matrix$1.checkMatrix(other);
	    let result = new Matrix$1(2, 2);
	    const a11 = this.get(0, 0);
	    const b11 = other.get(0, 0);
	    const a12 = this.get(0, 1);
	    const b12 = other.get(0, 1);
	    const a21 = this.get(1, 0);
	    const b21 = other.get(1, 0);
	    const a22 = this.get(1, 1);
	    const b22 = other.get(1, 1);

	    // Compute intermediate values.
	    const m1 = (a11 + a22) * (b11 + b22);
	    const m2 = (a21 + a22) * b11;
	    const m3 = a11 * (b12 - b22);
	    const m4 = a22 * (b21 - b11);
	    const m5 = (a11 + a12) * b22;
	    const m6 = (a21 - a11) * (b11 + b12);
	    const m7 = (a12 - a22) * (b21 + b22);

	    // Combine intermediate values into the output.
	    const c00 = m1 + m4 - m5 + m7;
	    const c01 = m3 + m5;
	    const c10 = m2 + m4;
	    const c11 = m1 - m2 + m3 + m6;
	    result.set(0, 0, c00);
	    result.set(0, 1, c01);
	    result.set(1, 0, c10);
	    result.set(1, 1, c11);
	    return result;
	  }
	  strassen3x3(other) {
	    other = Matrix$1.checkMatrix(other);
	    let result = new Matrix$1(3, 3);
	    const a00 = this.get(0, 0);
	    const a01 = this.get(0, 1);
	    const a02 = this.get(0, 2);
	    const a10 = this.get(1, 0);
	    const a11 = this.get(1, 1);
	    const a12 = this.get(1, 2);
	    const a20 = this.get(2, 0);
	    const a21 = this.get(2, 1);
	    const a22 = this.get(2, 2);
	    const b00 = other.get(0, 0);
	    const b01 = other.get(0, 1);
	    const b02 = other.get(0, 2);
	    const b10 = other.get(1, 0);
	    const b11 = other.get(1, 1);
	    const b12 = other.get(1, 2);
	    const b20 = other.get(2, 0);
	    const b21 = other.get(2, 1);
	    const b22 = other.get(2, 2);
	    const m1 = (a00 + a01 + a02 - a10 - a11 - a21 - a22) * b11;
	    const m2 = (a00 - a10) * (-b01 + b11);
	    const m3 = a11 * (-b00 + b01 + b10 - b11 - b12 - b20 + b22);
	    const m4 = (-a00 + a10 + a11) * (b00 - b01 + b11);
	    const m5 = (a10 + a11) * (-b00 + b01);
	    const m6 = a00 * b00;
	    const m7 = (-a00 + a20 + a21) * (b00 - b02 + b12);
	    const m8 = (-a00 + a20) * (b02 - b12);
	    const m9 = (a20 + a21) * (-b00 + b02);
	    const m10 = (a00 + a01 + a02 - a11 - a12 - a20 - a21) * b12;
	    const m11 = a21 * (-b00 + b02 + b10 - b11 - b12 - b20 + b21);
	    const m12 = (-a02 + a21 + a22) * (b11 + b20 - b21);
	    const m13 = (a02 - a22) * (b11 - b21);
	    const m14 = a02 * b20;
	    const m15 = (a21 + a22) * (-b20 + b21);
	    const m16 = (-a02 + a11 + a12) * (b12 + b20 - b22);
	    const m17 = (a02 - a12) * (b12 - b22);
	    const m18 = (a11 + a12) * (-b20 + b22);
	    const m19 = a01 * b10;
	    const m20 = a12 * b21;
	    const m21 = a10 * b02;
	    const m22 = a20 * b01;
	    const m23 = a22 * b22;
	    const c00 = m6 + m14 + m19;
	    const c01 = m1 + m4 + m5 + m6 + m12 + m14 + m15;
	    const c02 = m6 + m7 + m9 + m10 + m14 + m16 + m18;
	    const c10 = m2 + m3 + m4 + m6 + m14 + m16 + m17;
	    const c11 = m2 + m4 + m5 + m6 + m20;
	    const c12 = m14 + m16 + m17 + m18 + m21;
	    const c20 = m6 + m7 + m8 + m11 + m12 + m13 + m14;
	    const c21 = m12 + m13 + m14 + m15 + m22;
	    const c22 = m6 + m7 + m8 + m9 + m23;
	    result.set(0, 0, c00);
	    result.set(0, 1, c01);
	    result.set(0, 2, c02);
	    result.set(1, 0, c10);
	    result.set(1, 1, c11);
	    result.set(1, 2, c12);
	    result.set(2, 0, c20);
	    result.set(2, 1, c21);
	    result.set(2, 2, c22);
	    return result;
	  }
	  mmulStrassen(y) {
	    y = Matrix$1.checkMatrix(y);
	    let x = this.clone();
	    let r1 = x.rows;
	    let c1 = x.columns;
	    let r2 = y.rows;
	    let c2 = y.columns;
	    if (c1 !== r2) {
	      // eslint-disable-next-line no-console
	      console.warn(`Multiplying ${r1} x ${c1} and ${r2} x ${c2} matrix: dimensions do not match.`);
	    }

	    // Put a matrix into the top left of a matrix of zeros.
	    // `rows` and `cols` are the dimensions of the output matrix.
	    function embed(mat, rows, cols) {
	      let r = mat.rows;
	      let c = mat.columns;
	      if (r === rows && c === cols) {
	        return mat;
	      } else {
	        let resultat = AbstractMatrix.zeros(rows, cols);
	        resultat = resultat.setSubMatrix(mat, 0, 0);
	        return resultat;
	      }
	    }

	    // Make sure both matrices are the same size.
	    // This is exclusively for simplicity:
	    // this algorithm can be implemented with matrices of different sizes.

	    let r = Math.max(r1, r2);
	    let c = Math.max(c1, c2);
	    x = embed(x, r, c);
	    y = embed(y, r, c);

	    // Our recursive multiplication function.
	    function blockMult(a, b, rows, cols) {
	      // For small matrices, resort to naive multiplication.
	      if (rows <= 512 || cols <= 512) {
	        return a.mmul(b); // a is equivalent to this
	      }

	      // Apply dynamic padding.
	      if (rows % 2 === 1 && cols % 2 === 1) {
	        a = embed(a, rows + 1, cols + 1);
	        b = embed(b, rows + 1, cols + 1);
	      } else if (rows % 2 === 1) {
	        a = embed(a, rows + 1, cols);
	        b = embed(b, rows + 1, cols);
	      } else if (cols % 2 === 1) {
	        a = embed(a, rows, cols + 1);
	        b = embed(b, rows, cols + 1);
	      }
	      let halfRows = parseInt(a.rows / 2, 10);
	      let halfCols = parseInt(a.columns / 2, 10);
	      // Subdivide input matrices.
	      let a11 = a.subMatrix(0, halfRows - 1, 0, halfCols - 1);
	      let b11 = b.subMatrix(0, halfRows - 1, 0, halfCols - 1);
	      let a12 = a.subMatrix(0, halfRows - 1, halfCols, a.columns - 1);
	      let b12 = b.subMatrix(0, halfRows - 1, halfCols, b.columns - 1);
	      let a21 = a.subMatrix(halfRows, a.rows - 1, 0, halfCols - 1);
	      let b21 = b.subMatrix(halfRows, b.rows - 1, 0, halfCols - 1);
	      let a22 = a.subMatrix(halfRows, a.rows - 1, halfCols, a.columns - 1);
	      let b22 = b.subMatrix(halfRows, b.rows - 1, halfCols, b.columns - 1);

	      // Compute intermediate values.
	      let m1 = blockMult(AbstractMatrix.add(a11, a22), AbstractMatrix.add(b11, b22), halfRows, halfCols);
	      let m2 = blockMult(AbstractMatrix.add(a21, a22), b11, halfRows, halfCols);
	      let m3 = blockMult(a11, AbstractMatrix.sub(b12, b22), halfRows, halfCols);
	      let m4 = blockMult(a22, AbstractMatrix.sub(b21, b11), halfRows, halfCols);
	      let m5 = blockMult(AbstractMatrix.add(a11, a12), b22, halfRows, halfCols);
	      let m6 = blockMult(AbstractMatrix.sub(a21, a11), AbstractMatrix.add(b11, b12), halfRows, halfCols);
	      let m7 = blockMult(AbstractMatrix.sub(a12, a22), AbstractMatrix.add(b21, b22), halfRows, halfCols);

	      // Combine intermediate values into the output.
	      let c11 = AbstractMatrix.add(m1, m4);
	      c11.sub(m5);
	      c11.add(m7);
	      let c12 = AbstractMatrix.add(m3, m5);
	      let c21 = AbstractMatrix.add(m2, m4);
	      let c22 = AbstractMatrix.sub(m1, m2);
	      c22.add(m3);
	      c22.add(m6);

	      // Crop output to the desired size (undo dynamic padding).
	      let result = AbstractMatrix.zeros(2 * c11.rows, 2 * c11.columns);
	      result = result.setSubMatrix(c11, 0, 0);
	      result = result.setSubMatrix(c12, c11.rows, 0);
	      result = result.setSubMatrix(c21, 0, c11.columns);
	      result = result.setSubMatrix(c22, c11.rows, c11.columns);
	      return result.subMatrix(0, rows - 1, 0, cols - 1);
	    }
	    return blockMult(x, y, r, c);
	  }
	  scaleRows(options = {}) {
	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }
	    const {
	      min = 0,
	      max = 1
	    } = options;
	    if (!Number.isFinite(min)) throw new TypeError('min must be a number');
	    if (!Number.isFinite(max)) throw new TypeError('max must be a number');
	    if (min >= max) throw new RangeError('min must be smaller than max');
	    let newMatrix = new Matrix$1(this.rows, this.columns);
	    for (let i = 0; i < this.rows; i++) {
	      const row = this.getRow(i);
	      if (row.length > 0) {
	        rescale$1(row, {
	          min,
	          max,
	          output: row
	        });
	      }
	      newMatrix.setRow(i, row);
	    }
	    return newMatrix;
	  }
	  scaleColumns(options = {}) {
	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }
	    const {
	      min = 0,
	      max = 1
	    } = options;
	    if (!Number.isFinite(min)) throw new TypeError('min must be a number');
	    if (!Number.isFinite(max)) throw new TypeError('max must be a number');
	    if (min >= max) throw new RangeError('min must be smaller than max');
	    let newMatrix = new Matrix$1(this.rows, this.columns);
	    for (let i = 0; i < this.columns; i++) {
	      const column = this.getColumn(i);
	      if (column.length) {
	        rescale$1(column, {
	          min,
	          max,
	          output: column
	        });
	      }
	      newMatrix.setColumn(i, column);
	    }
	    return newMatrix;
	  }
	  flipRows() {
	    const middle = Math.ceil(this.columns / 2);
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < middle; j++) {
	        let first = this.get(i, j);
	        let last = this.get(i, this.columns - 1 - j);
	        this.set(i, j, last);
	        this.set(i, this.columns - 1 - j, first);
	      }
	    }
	    return this;
	  }
	  flipColumns() {
	    const middle = Math.ceil(this.rows / 2);
	    for (let j = 0; j < this.columns; j++) {
	      for (let i = 0; i < middle; i++) {
	        let first = this.get(i, j);
	        let last = this.get(this.rows - 1 - i, j);
	        this.set(i, j, last);
	        this.set(this.rows - 1 - i, j, first);
	      }
	    }
	    return this;
	  }
	  kroneckerProduct(other) {
	    other = Matrix$1.checkMatrix(other);
	    let m = this.rows;
	    let n = this.columns;
	    let p = other.rows;
	    let q = other.columns;
	    let result = new Matrix$1(m * p, n * q);
	    for (let i = 0; i < m; i++) {
	      for (let j = 0; j < n; j++) {
	        for (let k = 0; k < p; k++) {
	          for (let l = 0; l < q; l++) {
	            result.set(p * i + k, q * j + l, this.get(i, j) * other.get(k, l));
	          }
	        }
	      }
	    }
	    return result;
	  }
	  kroneckerSum(other) {
	    other = Matrix$1.checkMatrix(other);
	    if (!this.isSquare() || !other.isSquare()) {
	      throw new Error('Kronecker Sum needs two Square Matrices');
	    }
	    let m = this.rows;
	    let n = other.rows;
	    let AxI = this.kroneckerProduct(Matrix$1.eye(n, n));
	    let IxB = Matrix$1.eye(m, m).kroneckerProduct(other);
	    return AxI.add(IxB);
	  }
	  transpose() {
	    let result = new Matrix$1(this.columns, this.rows);
	    for (let i = 0; i < this.rows; i++) {
	      for (let j = 0; j < this.columns; j++) {
	        result.set(j, i, this.get(i, j));
	      }
	    }
	    return result;
	  }
	  sortRows(compareFunction = compareNumbers) {
	    for (let i = 0; i < this.rows; i++) {
	      this.setRow(i, this.getRow(i).sort(compareFunction));
	    }
	    return this;
	  }
	  sortColumns(compareFunction = compareNumbers) {
	    for (let i = 0; i < this.columns; i++) {
	      this.setColumn(i, this.getColumn(i).sort(compareFunction));
	    }
	    return this;
	  }
	  subMatrix(startRow, endRow, startColumn, endColumn) {
	    checkRange(this, startRow, endRow, startColumn, endColumn);
	    let newMatrix = new Matrix$1(endRow - startRow + 1, endColumn - startColumn + 1);
	    for (let i = startRow; i <= endRow; i++) {
	      for (let j = startColumn; j <= endColumn; j++) {
	        newMatrix.set(i - startRow, j - startColumn, this.get(i, j));
	      }
	    }
	    return newMatrix;
	  }
	  subMatrixRow(indices, startColumn, endColumn) {
	    if (startColumn === undefined) startColumn = 0;
	    if (endColumn === undefined) endColumn = this.columns - 1;
	    if (startColumn > endColumn || startColumn < 0 || startColumn >= this.columns || endColumn < 0 || endColumn >= this.columns) {
	      throw new RangeError('Argument out of range');
	    }
	    let newMatrix = new Matrix$1(indices.length, endColumn - startColumn + 1);
	    for (let i = 0; i < indices.length; i++) {
	      for (let j = startColumn; j <= endColumn; j++) {
	        if (indices[i] < 0 || indices[i] >= this.rows) {
	          throw new RangeError(`Row index out of range: ${indices[i]}`);
	        }
	        newMatrix.set(i, j - startColumn, this.get(indices[i], j));
	      }
	    }
	    return newMatrix;
	  }
	  subMatrixColumn(indices, startRow, endRow) {
	    if (startRow === undefined) startRow = 0;
	    if (endRow === undefined) endRow = this.rows - 1;
	    if (startRow > endRow || startRow < 0 || startRow >= this.rows || endRow < 0 || endRow >= this.rows) {
	      throw new RangeError('Argument out of range');
	    }
	    let newMatrix = new Matrix$1(endRow - startRow + 1, indices.length);
	    for (let i = 0; i < indices.length; i++) {
	      for (let j = startRow; j <= endRow; j++) {
	        if (indices[i] < 0 || indices[i] >= this.columns) {
	          throw new RangeError(`Column index out of range: ${indices[i]}`);
	        }
	        newMatrix.set(j - startRow, i, this.get(j, indices[i]));
	      }
	    }
	    return newMatrix;
	  }
	  setSubMatrix(matrix, startRow, startColumn) {
	    matrix = Matrix$1.checkMatrix(matrix);
	    if (matrix.isEmpty()) {
	      return this;
	    }
	    let endRow = startRow + matrix.rows - 1;
	    let endColumn = startColumn + matrix.columns - 1;
	    checkRange(this, startRow, endRow, startColumn, endColumn);
	    for (let i = 0; i < matrix.rows; i++) {
	      for (let j = 0; j < matrix.columns; j++) {
	        this.set(startRow + i, startColumn + j, matrix.get(i, j));
	      }
	    }
	    return this;
	  }
	  selection(rowIndices, columnIndices) {
	    checkRowIndices(this, rowIndices);
	    checkColumnIndices(this, columnIndices);
	    let newMatrix = new Matrix$1(rowIndices.length, columnIndices.length);
	    for (let i = 0; i < rowIndices.length; i++) {
	      let rowIndex = rowIndices[i];
	      for (let j = 0; j < columnIndices.length; j++) {
	        let columnIndex = columnIndices[j];
	        newMatrix.set(i, j, this.get(rowIndex, columnIndex));
	      }
	    }
	    return newMatrix;
	  }
	  trace() {
	    let min = Math.min(this.rows, this.columns);
	    let trace = 0;
	    for (let i = 0; i < min; i++) {
	      trace += this.get(i, i);
	    }
	    return trace;
	  }
	  clone() {
	    return this.constructor.copy(this, new Matrix$1(this.rows, this.columns));
	  }

	  /**
	   * @template {AbstractMatrix} M
	   * @param {AbstractMatrix} from
	   * @param {M} to
	   * @return {M}
	   */
	  static copy(from, to) {
	    for (const [row, column, value] of from.entries()) {
	      to.set(row, column, value);
	    }
	    return to;
	  }
	  sum(by) {
	    switch (by) {
	      case 'row':
	        return sumByRow(this);
	      case 'column':
	        return sumByColumn(this);
	      case undefined:
	        return sumAll(this);
	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }
	  product(by) {
	    switch (by) {
	      case 'row':
	        return productByRow(this);
	      case 'column':
	        return productByColumn(this);
	      case undefined:
	        return productAll(this);
	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }
	  mean(by) {
	    const sum = this.sum(by);
	    switch (by) {
	      case 'row':
	        {
	          for (let i = 0; i < this.rows; i++) {
	            sum[i] /= this.columns;
	          }
	          return sum;
	        }
	      case 'column':
	        {
	          for (let i = 0; i < this.columns; i++) {
	            sum[i] /= this.rows;
	          }
	          return sum;
	        }
	      case undefined:
	        return sum / this.size;
	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }
	  variance(by, options = {}) {
	    if (typeof by === 'object') {
	      options = by;
	      by = undefined;
	    }
	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }
	    const {
	      unbiased = true,
	      mean = this.mean(by)
	    } = options;
	    if (typeof unbiased !== 'boolean') {
	      throw new TypeError('unbiased must be a boolean');
	    }
	    switch (by) {
	      case 'row':
	        {
	          if (!isAnyArray.isAnyArray(mean)) {
	            throw new TypeError('mean must be an array');
	          }
	          return varianceByRow(this, unbiased, mean);
	        }
	      case 'column':
	        {
	          if (!isAnyArray.isAnyArray(mean)) {
	            throw new TypeError('mean must be an array');
	          }
	          return varianceByColumn(this, unbiased, mean);
	        }
	      case undefined:
	        {
	          if (typeof mean !== 'number') {
	            throw new TypeError('mean must be a number');
	          }
	          return varianceAll(this, unbiased, mean);
	        }
	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }
	  standardDeviation(by, options) {
	    if (typeof by === 'object') {
	      options = by;
	      by = undefined;
	    }
	    const variance = this.variance(by, options);
	    if (by === undefined) {
	      return Math.sqrt(variance);
	    } else {
	      for (let i = 0; i < variance.length; i++) {
	        variance[i] = Math.sqrt(variance[i]);
	      }
	      return variance;
	    }
	  }
	  center(by, options = {}) {
	    if (typeof by === 'object') {
	      options = by;
	      by = undefined;
	    }
	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }
	    const {
	      center = this.mean(by)
	    } = options;
	    switch (by) {
	      case 'row':
	        {
	          if (!isAnyArray.isAnyArray(center)) {
	            throw new TypeError('center must be an array');
	          }
	          centerByRow(this, center);
	          return this;
	        }
	      case 'column':
	        {
	          if (!isAnyArray.isAnyArray(center)) {
	            throw new TypeError('center must be an array');
	          }
	          centerByColumn(this, center);
	          return this;
	        }
	      case undefined:
	        {
	          if (typeof center !== 'number') {
	            throw new TypeError('center must be a number');
	          }
	          centerAll(this, center);
	          return this;
	        }
	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }
	  scale(by, options = {}) {
	    if (typeof by === 'object') {
	      options = by;
	      by = undefined;
	    }
	    if (typeof options !== 'object') {
	      throw new TypeError('options must be an object');
	    }
	    let scale = options.scale;
	    switch (by) {
	      case 'row':
	        {
	          if (scale === undefined) {
	            scale = getScaleByRow(this);
	          } else if (!isAnyArray.isAnyArray(scale)) {
	            throw new TypeError('scale must be an array');
	          }
	          scaleByRow(this, scale);
	          return this;
	        }
	      case 'column':
	        {
	          if (scale === undefined) {
	            scale = getScaleByColumn(this);
	          } else if (!isAnyArray.isAnyArray(scale)) {
	            throw new TypeError('scale must be an array');
	          }
	          scaleByColumn(this, scale);
	          return this;
	        }
	      case undefined:
	        {
	          if (scale === undefined) {
	            scale = getScaleAll(this);
	          } else if (typeof scale !== 'number') {
	            throw new TypeError('scale must be a number');
	          }
	          scaleAll(this, scale);
	          return this;
	        }
	      default:
	        throw new Error(`invalid option: ${by}`);
	    }
	  }
	  toString(options) {
	    return inspectMatrixWithOptions(this, options);
	  }
	  [Symbol.iterator]() {
	    return this.entries();
	  }

	  /**
	   * iterator from left to right, from top to bottom
	   * yield [row, column, value]
	   * @returns {Generator<[number, number, number], void, *>}
	   */
	  *entries() {
	    for (let row = 0; row < this.rows; row++) {
	      for (let col = 0; col < this.columns; col++) {
	        yield [row, col, this.get(row, col)];
	      }
	    }
	  }

	  /**
	   * iterator from left to right, from top to bottom
	   * yield value
	   * @returns {Generator<number, void, *>}
	   */
	  *values() {
	    for (let row = 0; row < this.rows; row++) {
	      for (let col = 0; col < this.columns; col++) {
	        yield this.get(row, col);
	      }
	    }
	  }
	}
	AbstractMatrix.prototype.klass = 'Matrix';
	if (typeof Symbol !== 'undefined') {
	  AbstractMatrix.prototype[Symbol.for('nodejs.util.inspect.custom')] = inspectMatrix;
	}
	function compareNumbers(a, b) {
	  return a - b;
	}
	function isArrayOfNumbers(array) {
	  return array.every(element => {
	    return typeof element === 'number';
	  });
	}

	// Synonyms
	AbstractMatrix.random = AbstractMatrix.rand;
	AbstractMatrix.randomInt = AbstractMatrix.randInt;
	AbstractMatrix.diagonal = AbstractMatrix.diag;
	AbstractMatrix.prototype.diagonal = AbstractMatrix.prototype.diag;
	AbstractMatrix.identity = AbstractMatrix.eye;
	AbstractMatrix.prototype.negate = AbstractMatrix.prototype.neg;
	AbstractMatrix.prototype.tensorProduct = AbstractMatrix.prototype.kroneckerProduct;
	class Matrix$1 extends AbstractMatrix {
	  /**
	   * @type {Float64Array[]}
	   */
	  data;

	  /**
	   * Init an empty matrix
	   * @param {number} nRows
	   * @param {number} nColumns
	   */
	  #initData(nRows, nColumns) {
	    this.data = [];
	    if (Number.isInteger(nColumns) && nColumns >= 0) {
	      for (let i = 0; i < nRows; i++) {
	        this.data.push(new Float64Array(nColumns));
	      }
	    } else {
	      throw new TypeError('nColumns must be a positive integer');
	    }
	    this.rows = nRows;
	    this.columns = nColumns;
	  }
	  constructor(nRows, nColumns) {
	    super();
	    if (Matrix$1.isMatrix(nRows)) {
	      this.#initData(nRows.rows, nRows.columns);
	      Matrix$1.copy(nRows, this);
	    } else if (Number.isInteger(nRows) && nRows >= 0) {
	      this.#initData(nRows, nColumns);
	    } else if (isAnyArray.isAnyArray(nRows)) {
	      // Copy the values from the 2D array
	      const arrayData = nRows;
	      nRows = arrayData.length;
	      nColumns = nRows ? arrayData[0].length : 0;
	      if (typeof nColumns !== 'number') {
	        throw new TypeError('Data must be a 2D array with at least one element');
	      }
	      this.data = [];
	      for (let i = 0; i < nRows; i++) {
	        if (arrayData[i].length !== nColumns) {
	          throw new RangeError('Inconsistent array dimensions');
	        }
	        if (!isArrayOfNumbers(arrayData[i])) {
	          throw new TypeError('Input data contains non-numeric values');
	        }
	        this.data.push(Float64Array.from(arrayData[i]));
	      }
	      this.rows = nRows;
	      this.columns = nColumns;
	    } else {
	      throw new TypeError('First argument must be a positive number or an array');
	    }
	  }
	  set(rowIndex, columnIndex, value) {
	    this.data[rowIndex][columnIndex] = value;
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    return this.data[rowIndex][columnIndex];
	  }
	  removeRow(index) {
	    checkRowIndex(this, index);
	    this.data.splice(index, 1);
	    this.rows -= 1;
	    return this;
	  }
	  addRow(index, array) {
	    if (array === undefined) {
	      array = index;
	      index = this.rows;
	    }
	    checkRowIndex(this, index, true);
	    array = Float64Array.from(checkRowVector(this, array));
	    this.data.splice(index, 0, array);
	    this.rows += 1;
	    return this;
	  }
	  removeColumn(index) {
	    checkColumnIndex(this, index);
	    for (let i = 0; i < this.rows; i++) {
	      const newRow = new Float64Array(this.columns - 1);
	      for (let j = 0; j < index; j++) {
	        newRow[j] = this.data[i][j];
	      }
	      for (let j = index + 1; j < this.columns; j++) {
	        newRow[j - 1] = this.data[i][j];
	      }
	      this.data[i] = newRow;
	    }
	    this.columns -= 1;
	    return this;
	  }
	  addColumn(index, array) {
	    if (typeof array === 'undefined') {
	      array = index;
	      index = this.columns;
	    }
	    checkColumnIndex(this, index, true);
	    array = checkColumnVector(this, array);
	    for (let i = 0; i < this.rows; i++) {
	      const newRow = new Float64Array(this.columns + 1);
	      let j = 0;
	      for (; j < index; j++) {
	        newRow[j] = this.data[i][j];
	      }
	      newRow[j++] = array[i];
	      for (; j < this.columns + 1; j++) {
	        newRow[j] = this.data[i][j - 1];
	      }
	      this.data[i] = newRow;
	    }
	    this.columns += 1;
	    return this;
	  }
	}
	installMathOperations(AbstractMatrix, Matrix$1);

	/**
	 * @typedef {0 | 1 | number | boolean} Mask
	 */

	class SymmetricMatrix extends AbstractMatrix {
	  /** @type {Matrix} */
	  #matrix;
	  get size() {
	    return this.#matrix.size;
	  }
	  get rows() {
	    return this.#matrix.rows;
	  }
	  get columns() {
	    return this.#matrix.columns;
	  }
	  get diagonalSize() {
	    return this.rows;
	  }

	  /**
	   * not the same as matrix.isSymmetric()
	   * Here is to check if it's instanceof SymmetricMatrix without bundling issues
	   *
	   * @param value
	   * @returns {boolean}
	   */
	  static isSymmetricMatrix(value) {
	    return Matrix$1.isMatrix(value) && value.klassType === 'SymmetricMatrix';
	  }

	  /**
	   * @param diagonalSize
	   * @return {SymmetricMatrix}
	   */
	  static zeros(diagonalSize) {
	    return new this(diagonalSize);
	  }

	  /**
	   * @param diagonalSize
	   * @return {SymmetricMatrix}
	   */
	  static ones(diagonalSize) {
	    return new this(diagonalSize).fill(1);
	  }

	  /**
	   * @param {number | AbstractMatrix | ArrayLike<ArrayLike<number>>} diagonalSize
	   * @return {this}
	   */
	  constructor(diagonalSize) {
	    super();
	    if (Matrix$1.isMatrix(diagonalSize)) {
	      if (!diagonalSize.isSymmetric()) {
	        throw new TypeError('not symmetric data');
	      }
	      this.#matrix = Matrix$1.copy(diagonalSize, new Matrix$1(diagonalSize.rows, diagonalSize.rows));
	    } else if (Number.isInteger(diagonalSize) && diagonalSize >= 0) {
	      this.#matrix = new Matrix$1(diagonalSize, diagonalSize);
	    } else {
	      this.#matrix = new Matrix$1(diagonalSize);
	      if (!this.isSymmetric()) {
	        throw new TypeError('not symmetric data');
	      }
	    }
	  }
	  clone() {
	    const matrix = new SymmetricMatrix(this.diagonalSize);
	    for (const [row, col, value] of this.upperRightEntries()) {
	      matrix.set(row, col, value);
	    }
	    return matrix;
	  }
	  toMatrix() {
	    return new Matrix$1(this);
	  }
	  get(rowIndex, columnIndex) {
	    return this.#matrix.get(rowIndex, columnIndex);
	  }
	  set(rowIndex, columnIndex, value) {
	    // symmetric set
	    this.#matrix.set(rowIndex, columnIndex, value);
	    this.#matrix.set(columnIndex, rowIndex, value);
	    return this;
	  }
	  removeCross(index) {
	    // symmetric remove side
	    this.#matrix.removeRow(index);
	    this.#matrix.removeColumn(index);
	    return this;
	  }
	  addCross(index, array) {
	    if (array === undefined) {
	      array = index;
	      index = this.diagonalSize;
	    }
	    const row = array.slice();
	    row.splice(index, 1);
	    this.#matrix.addRow(index, row);
	    this.#matrix.addColumn(index, array);
	    return this;
	  }

	  /**
	   * @param {Mask[]} mask
	   */
	  applyMask(mask) {
	    if (mask.length !== this.diagonalSize) {
	      throw new RangeError('Mask size do not match with matrix size');
	    }

	    // prepare sides to remove from matrix from mask
	    /** @type {number[]} */
	    const sidesToRemove = [];
	    for (const [index, passthroughs] of mask.entries()) {
	      if (passthroughs) continue;
	      sidesToRemove.push(index);
	    }
	    // to remove from highest to lowest for no mutation shifting
	    sidesToRemove.reverse();

	    // remove sides
	    for (const sideIndex of sidesToRemove) {
	      this.removeCross(sideIndex);
	    }
	    return this;
	  }

	  /**
	   * Compact format upper-right corner of matrix
	   * iterate from left to right, from top to bottom.
	   *
	   * ```
	   *   A B C D
	   * A 1 2 3 4
	   * B 2 5 6 7
	   * C 3 6 8 9
	   * D 4 7 9 10
	   * ```
	   *
	   * will return compact 1D array `[1, 2, 3, 4, 5, 6, 7, 8, 9, 10]`
	   *
	   * length is S(i=0, n=sideSize) => 10 for a 4 sideSized matrix
	   *
	   * @returns {number[]}
	   */
	  toCompact() {
	    const {
	      diagonalSize
	    } = this;

	    /** @type {number[]} */
	    const compact = new Array(diagonalSize * (diagonalSize + 1) / 2);
	    for (let col = 0, row = 0, index = 0; index < compact.length; index++) {
	      compact[index] = this.get(row, col);
	      if (++col >= diagonalSize) col = ++row;
	    }
	    return compact;
	  }

	  /**
	   * @param {number[]} compact
	   * @return {SymmetricMatrix}
	   */
	  static fromCompact(compact) {
	    const compactSize = compact.length;
	    // compactSize = (sideSize * (sideSize + 1)) / 2
	    // https://mathsolver.microsoft.com/fr/solve-problem/y%20%3D%20%20x%20%60cdot%20%20%20%60frac%7B%20%20%60left(%20x%2B1%20%20%60right)%20%20%20%20%7D%7B%202%20%20%7D
	    // sideSize = (Sqrt(8 × compactSize + 1) - 1) / 2
	    const diagonalSize = (Math.sqrt(8 * compactSize + 1) - 1) / 2;
	    if (!Number.isInteger(diagonalSize)) {
	      throw new TypeError(`This array is not a compact representation of a Symmetric Matrix, ${JSON.stringify(compact)}`);
	    }
	    const matrix = new SymmetricMatrix(diagonalSize);
	    for (let col = 0, row = 0, index = 0; index < compactSize; index++) {
	      matrix.set(col, row, compact[index]);
	      if (++col >= diagonalSize) col = ++row;
	    }
	    return matrix;
	  }

	  /**
	   * half iterator upper-right-corner from left to right, from top to bottom
	   * yield [row, column, value]
	   *
	   * @returns {Generator<[number, number, number], void, *>}
	   */
	  *upperRightEntries() {
	    for (let row = 0, col = 0; row < this.diagonalSize; void 0) {
	      const value = this.get(row, col);
	      yield [row, col, value];

	      // at the end of row, move cursor to next row at diagonal position
	      if (++col >= this.diagonalSize) col = ++row;
	    }
	  }

	  /**
	   * half iterator upper-right-corner from left to right, from top to bottom
	   * yield value
	   *
	   * @returns {Generator<[number, number, number], void, *>}
	   */
	  *upperRightValues() {
	    for (let row = 0, col = 0; row < this.diagonalSize; void 0) {
	      const value = this.get(row, col);
	      yield value;

	      // at the end of row, move cursor to next row at diagonal position
	      if (++col >= this.diagonalSize) col = ++row;
	    }
	  }
	}
	SymmetricMatrix.prototype.klassType = 'SymmetricMatrix';
	class DistanceMatrix extends SymmetricMatrix {
	  /**
	   * not the same as matrix.isSymmetric()
	   * Here is to check if it's instanceof SymmetricMatrix without bundling issues
	   *
	   * @param value
	   * @returns {boolean}
	   */
	  static isDistanceMatrix(value) {
	    return SymmetricMatrix.isSymmetricMatrix(value) && value.klassSubType === 'DistanceMatrix';
	  }
	  constructor(sideSize) {
	    super(sideSize);
	    if (!this.isDistance()) {
	      throw new TypeError('Provided arguments do no produce a distance matrix');
	    }
	  }
	  set(rowIndex, columnIndex, value) {
	    // distance matrix diagonal is 0
	    if (rowIndex === columnIndex) value = 0;
	    return super.set(rowIndex, columnIndex, value);
	  }
	  addCross(index, array) {
	    if (array === undefined) {
	      array = index;
	      index = this.diagonalSize;
	    }

	    // ensure distance
	    array = array.slice();
	    array[index] = 0;
	    return super.addCross(index, array);
	  }
	  toSymmetricMatrix() {
	    return new SymmetricMatrix(this);
	  }
	  clone() {
	    const matrix = new DistanceMatrix(this.diagonalSize);
	    for (const [row, col, value] of this.upperRightEntries()) {
	      if (row === col) continue;
	      matrix.set(row, col, value);
	    }
	    return matrix;
	  }

	  /**
	   * Compact format upper-right corner of matrix
	   * no diagonal (only zeros)
	   * iterable from left to right, from top to bottom.
	   *
	   * ```
	   *   A B C D
	   * A 0 1 2 3
	   * B 1 0 4 5
	   * C 2 4 0 6
	   * D 3 5 6 0
	   * ```
	   *
	   * will return compact 1D array `[1, 2, 3, 4, 5, 6]`
	   *
	   * length is S(i=0, n=sideSize-1) => 6 for a 4 side sized matrix
	   *
	   * @returns {number[]}
	   */
	  toCompact() {
	    const {
	      diagonalSize
	    } = this;
	    const compactLength = (diagonalSize - 1) * diagonalSize / 2;

	    /** @type {number[]} */
	    const compact = new Array(compactLength);
	    for (let col = 1, row = 0, index = 0; index < compact.length; index++) {
	      compact[index] = this.get(row, col);
	      if (++col >= diagonalSize) col = ++row + 1;
	    }
	    return compact;
	  }

	  /**
	   * @param {number[]} compact
	   */
	  static fromCompact(compact) {
	    const compactSize = compact.length;
	    // compactSize = (sideSize * (sideSize - 1)) / 2
	    // sideSize = (Sqrt(8 × compactSize + 1) + 1) / 2
	    const diagonalSize = (Math.sqrt(8 * compactSize + 1) + 1) / 2;
	    if (!Number.isInteger(diagonalSize)) {
	      throw new TypeError(`This array is not a compact representation of a DistanceMatrix, ${JSON.stringify(compact)}`);
	    }
	    const matrix = new this(diagonalSize);
	    for (let col = 1, row = 0, index = 0; index < compactSize; index++) {
	      matrix.set(col, row, compact[index]);
	      if (++col >= diagonalSize) col = ++row + 1;
	    }
	    return matrix;
	  }
	}
	DistanceMatrix.prototype.klassSubType = 'DistanceMatrix';
	class BaseView extends AbstractMatrix {
	  constructor(matrix, rows, columns) {
	    super();
	    this.matrix = matrix;
	    this.rows = rows;
	    this.columns = columns;
	  }
	}
	class MatrixColumnView extends BaseView {
	  constructor(matrix, column) {
	    checkColumnIndex(matrix, column);
	    super(matrix, matrix.rows, 1);
	    this.column = column;
	  }
	  set(rowIndex, columnIndex, value) {
	    this.matrix.set(rowIndex, this.column, value);
	    return this;
	  }
	  get(rowIndex) {
	    return this.matrix.get(rowIndex, this.column);
	  }
	}
	class MatrixColumnSelectionView extends BaseView {
	  constructor(matrix, columnIndices) {
	    checkColumnIndices(matrix, columnIndices);
	    super(matrix, matrix.rows, columnIndices.length);
	    this.columnIndices = columnIndices;
	  }
	  set(rowIndex, columnIndex, value) {
	    this.matrix.set(rowIndex, this.columnIndices[columnIndex], value);
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    return this.matrix.get(rowIndex, this.columnIndices[columnIndex]);
	  }
	}
	class MatrixFlipColumnView extends BaseView {
	  constructor(matrix) {
	    super(matrix, matrix.rows, matrix.columns);
	  }
	  set(rowIndex, columnIndex, value) {
	    this.matrix.set(rowIndex, this.columns - columnIndex - 1, value);
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    return this.matrix.get(rowIndex, this.columns - columnIndex - 1);
	  }
	}
	class MatrixFlipRowView extends BaseView {
	  constructor(matrix) {
	    super(matrix, matrix.rows, matrix.columns);
	  }
	  set(rowIndex, columnIndex, value) {
	    this.matrix.set(this.rows - rowIndex - 1, columnIndex, value);
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    return this.matrix.get(this.rows - rowIndex - 1, columnIndex);
	  }
	}
	class MatrixRowView extends BaseView {
	  constructor(matrix, row) {
	    checkRowIndex(matrix, row);
	    super(matrix, 1, matrix.columns);
	    this.row = row;
	  }
	  set(rowIndex, columnIndex, value) {
	    this.matrix.set(this.row, columnIndex, value);
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    return this.matrix.get(this.row, columnIndex);
	  }
	}
	class MatrixRowSelectionView extends BaseView {
	  constructor(matrix, rowIndices) {
	    checkRowIndices(matrix, rowIndices);
	    super(matrix, rowIndices.length, matrix.columns);
	    this.rowIndices = rowIndices;
	  }
	  set(rowIndex, columnIndex, value) {
	    this.matrix.set(this.rowIndices[rowIndex], columnIndex, value);
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    return this.matrix.get(this.rowIndices[rowIndex], columnIndex);
	  }
	}
	class MatrixSelectionView extends BaseView {
	  constructor(matrix, rowIndices, columnIndices) {
	    checkRowIndices(matrix, rowIndices);
	    checkColumnIndices(matrix, columnIndices);
	    super(matrix, rowIndices.length, columnIndices.length);
	    this.rowIndices = rowIndices;
	    this.columnIndices = columnIndices;
	  }
	  set(rowIndex, columnIndex, value) {
	    this.matrix.set(this.rowIndices[rowIndex], this.columnIndices[columnIndex], value);
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    return this.matrix.get(this.rowIndices[rowIndex], this.columnIndices[columnIndex]);
	  }
	}
	class MatrixSubView extends BaseView {
	  constructor(matrix, startRow, endRow, startColumn, endColumn) {
	    checkRange(matrix, startRow, endRow, startColumn, endColumn);
	    super(matrix, endRow - startRow + 1, endColumn - startColumn + 1);
	    this.startRow = startRow;
	    this.startColumn = startColumn;
	  }
	  set(rowIndex, columnIndex, value) {
	    this.matrix.set(this.startRow + rowIndex, this.startColumn + columnIndex, value);
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    return this.matrix.get(this.startRow + rowIndex, this.startColumn + columnIndex);
	  }
	}
	class MatrixTransposeView$1 extends BaseView {
	  constructor(matrix) {
	    super(matrix, matrix.columns, matrix.rows);
	  }
	  set(rowIndex, columnIndex, value) {
	    this.matrix.set(columnIndex, rowIndex, value);
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    return this.matrix.get(columnIndex, rowIndex);
	  }
	}
	class WrapperMatrix1D extends AbstractMatrix {
	  constructor(data, options = {}) {
	    const {
	      rows = 1
	    } = options;
	    if (data.length % rows !== 0) {
	      throw new Error('the data length is not divisible by the number of rows');
	    }
	    super();
	    this.rows = rows;
	    this.columns = data.length / rows;
	    this.data = data;
	  }
	  set(rowIndex, columnIndex, value) {
	    let index = this._calculateIndex(rowIndex, columnIndex);
	    this.data[index] = value;
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    let index = this._calculateIndex(rowIndex, columnIndex);
	    return this.data[index];
	  }
	  _calculateIndex(row, column) {
	    return row * this.columns + column;
	  }
	}
	class WrapperMatrix2D extends AbstractMatrix {
	  constructor(data) {
	    super();
	    this.data = data;
	    this.rows = data.length;
	    this.columns = data[0].length;
	  }
	  set(rowIndex, columnIndex, value) {
	    this.data[rowIndex][columnIndex] = value;
	    return this;
	  }
	  get(rowIndex, columnIndex) {
	    return this.data[rowIndex][columnIndex];
	  }
	}
	function wrap(array, options) {
	  if (isAnyArray.isAnyArray(array)) {
	    if (array[0] && isAnyArray.isAnyArray(array[0])) {
	      return new WrapperMatrix2D(array);
	    } else {
	      return new WrapperMatrix1D(array, options);
	    }
	  } else {
	    throw new Error('the argument is not an array');
	  }
	}
	class LuDecomposition {
	  constructor(matrix) {
	    matrix = WrapperMatrix2D.checkMatrix(matrix);
	    let lu = matrix.clone();
	    let rows = lu.rows;
	    let columns = lu.columns;
	    let pivotVector = new Float64Array(rows);
	    let pivotSign = 1;
	    let i, j, k, p, s, t, v;
	    let LUcolj, kmax;
	    for (i = 0; i < rows; i++) {
	      pivotVector[i] = i;
	    }
	    LUcolj = new Float64Array(rows);
	    for (j = 0; j < columns; j++) {
	      for (i = 0; i < rows; i++) {
	        LUcolj[i] = lu.get(i, j);
	      }
	      for (i = 0; i < rows; i++) {
	        kmax = Math.min(i, j);
	        s = 0;
	        for (k = 0; k < kmax; k++) {
	          s += lu.get(i, k) * LUcolj[k];
	        }
	        LUcolj[i] -= s;
	        lu.set(i, j, LUcolj[i]);
	      }
	      p = j;
	      for (i = j + 1; i < rows; i++) {
	        if (Math.abs(LUcolj[i]) > Math.abs(LUcolj[p])) {
	          p = i;
	        }
	      }
	      if (p !== j) {
	        for (k = 0; k < columns; k++) {
	          t = lu.get(p, k);
	          lu.set(p, k, lu.get(j, k));
	          lu.set(j, k, t);
	        }
	        v = pivotVector[p];
	        pivotVector[p] = pivotVector[j];
	        pivotVector[j] = v;
	        pivotSign = -pivotSign;
	      }
	      if (j < rows && lu.get(j, j) !== 0) {
	        for (i = j + 1; i < rows; i++) {
	          lu.set(i, j, lu.get(i, j) / lu.get(j, j));
	        }
	      }
	    }
	    this.LU = lu;
	    this.pivotVector = pivotVector;
	    this.pivotSign = pivotSign;
	  }
	  isSingular() {
	    let data = this.LU;
	    let col = data.columns;
	    for (let j = 0; j < col; j++) {
	      if (data.get(j, j) === 0) {
	        return true;
	      }
	    }
	    return false;
	  }
	  solve(value) {
	    value = Matrix$1.checkMatrix(value);
	    let lu = this.LU;
	    let rows = lu.rows;
	    if (rows !== value.rows) {
	      throw new Error('Invalid matrix dimensions');
	    }
	    if (this.isSingular()) {
	      throw new Error('LU matrix is singular');
	    }
	    let count = value.columns;
	    let X = value.subMatrixRow(this.pivotVector, 0, count - 1);
	    let columns = lu.columns;
	    let i, j, k;
	    for (k = 0; k < columns; k++) {
	      for (i = k + 1; i < columns; i++) {
	        for (j = 0; j < count; j++) {
	          X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
	        }
	      }
	    }
	    for (k = columns - 1; k >= 0; k--) {
	      for (j = 0; j < count; j++) {
	        X.set(k, j, X.get(k, j) / lu.get(k, k));
	      }
	      for (i = 0; i < k; i++) {
	        for (j = 0; j < count; j++) {
	          X.set(i, j, X.get(i, j) - X.get(k, j) * lu.get(i, k));
	        }
	      }
	    }
	    return X;
	  }
	  get determinant() {
	    let data = this.LU;
	    if (!data.isSquare()) {
	      throw new Error('Matrix must be square');
	    }
	    let determinant = this.pivotSign;
	    let col = data.columns;
	    for (let j = 0; j < col; j++) {
	      determinant *= data.get(j, j);
	    }
	    return determinant;
	  }
	  get lowerTriangularMatrix() {
	    let data = this.LU;
	    let rows = data.rows;
	    let columns = data.columns;
	    let X = new Matrix$1(rows, columns);
	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        if (i > j) {
	          X.set(i, j, data.get(i, j));
	        } else if (i === j) {
	          X.set(i, j, 1);
	        } else {
	          X.set(i, j, 0);
	        }
	      }
	    }
	    return X;
	  }
	  get upperTriangularMatrix() {
	    let data = this.LU;
	    let rows = data.rows;
	    let columns = data.columns;
	    let X = new Matrix$1(rows, columns);
	    for (let i = 0; i < rows; i++) {
	      for (let j = 0; j < columns; j++) {
	        if (i <= j) {
	          X.set(i, j, data.get(i, j));
	        } else {
	          X.set(i, j, 0);
	        }
	      }
	    }
	    return X;
	  }
	  get pivotPermutationVector() {
	    return Array.from(this.pivotVector);
	  }
	}
	function hypotenuse(a, b) {
	  let r = 0;
	  if (Math.abs(a) > Math.abs(b)) {
	    r = b / a;
	    return Math.abs(a) * Math.sqrt(1 + r * r);
	  }
	  if (b !== 0) {
	    r = a / b;
	    return Math.abs(b) * Math.sqrt(1 + r * r);
	  }
	  return 0;
	}
	class QrDecomposition {
	  constructor(value) {
	    value = WrapperMatrix2D.checkMatrix(value);
	    let qr = value.clone();
	    let m = value.rows;
	    let n = value.columns;
	    let rdiag = new Float64Array(n);
	    let i, j, k, s;
	    for (k = 0; k < n; k++) {
	      let nrm = 0;
	      for (i = k; i < m; i++) {
	        nrm = hypotenuse(nrm, qr.get(i, k));
	      }
	      if (nrm !== 0) {
	        if (qr.get(k, k) < 0) {
	          nrm = -nrm;
	        }
	        for (i = k; i < m; i++) {
	          qr.set(i, k, qr.get(i, k) / nrm);
	        }
	        qr.set(k, k, qr.get(k, k) + 1);
	        for (j = k + 1; j < n; j++) {
	          s = 0;
	          for (i = k; i < m; i++) {
	            s += qr.get(i, k) * qr.get(i, j);
	          }
	          s = -s / qr.get(k, k);
	          for (i = k; i < m; i++) {
	            qr.set(i, j, qr.get(i, j) + s * qr.get(i, k));
	          }
	        }
	      }
	      rdiag[k] = -nrm;
	    }
	    this.QR = qr;
	    this.Rdiag = rdiag;
	  }
	  solve(value) {
	    value = Matrix$1.checkMatrix(value);
	    let qr = this.QR;
	    let m = qr.rows;
	    if (value.rows !== m) {
	      throw new Error('Matrix row dimensions must agree');
	    }
	    if (!this.isFullRank()) {
	      throw new Error('Matrix is rank deficient');
	    }
	    let count = value.columns;
	    let X = value.clone();
	    let n = qr.columns;
	    let i, j, k, s;
	    for (k = 0; k < n; k++) {
	      for (j = 0; j < count; j++) {
	        s = 0;
	        for (i = k; i < m; i++) {
	          s += qr.get(i, k) * X.get(i, j);
	        }
	        s = -s / qr.get(k, k);
	        for (i = k; i < m; i++) {
	          X.set(i, j, X.get(i, j) + s * qr.get(i, k));
	        }
	      }
	    }
	    for (k = n - 1; k >= 0; k--) {
	      for (j = 0; j < count; j++) {
	        X.set(k, j, X.get(k, j) / this.Rdiag[k]);
	      }
	      for (i = 0; i < k; i++) {
	        for (j = 0; j < count; j++) {
	          X.set(i, j, X.get(i, j) - X.get(k, j) * qr.get(i, k));
	        }
	      }
	    }
	    return X.subMatrix(0, n - 1, 0, count - 1);
	  }
	  isFullRank() {
	    let columns = this.QR.columns;
	    for (let i = 0; i < columns; i++) {
	      if (this.Rdiag[i] === 0) {
	        return false;
	      }
	    }
	    return true;
	  }
	  get upperTriangularMatrix() {
	    let qr = this.QR;
	    let n = qr.columns;
	    let X = new Matrix$1(n, n);
	    let i, j;
	    for (i = 0; i < n; i++) {
	      for (j = 0; j < n; j++) {
	        if (i < j) {
	          X.set(i, j, qr.get(i, j));
	        } else if (i === j) {
	          X.set(i, j, this.Rdiag[i]);
	        } else {
	          X.set(i, j, 0);
	        }
	      }
	    }
	    return X;
	  }
	  get orthogonalMatrix() {
	    let qr = this.QR;
	    let rows = qr.rows;
	    let columns = qr.columns;
	    let X = new Matrix$1(rows, columns);
	    let i, j, k, s;
	    for (k = columns - 1; k >= 0; k--) {
	      for (i = 0; i < rows; i++) {
	        X.set(i, k, 0);
	      }
	      X.set(k, k, 1);
	      for (j = k; j < columns; j++) {
	        if (qr.get(k, k) !== 0) {
	          s = 0;
	          for (i = k; i < rows; i++) {
	            s += qr.get(i, k) * X.get(i, j);
	          }
	          s = -s / qr.get(k, k);
	          for (i = k; i < rows; i++) {
	            X.set(i, j, X.get(i, j) + s * qr.get(i, k));
	          }
	        }
	      }
	    }
	    return X;
	  }
	}
	class SingularValueDecomposition {
	  constructor(value, options = {}) {
	    value = WrapperMatrix2D.checkMatrix(value);
	    if (value.isEmpty()) {
	      throw new Error('Matrix must be non-empty');
	    }
	    let m = value.rows;
	    let n = value.columns;
	    const {
	      computeLeftSingularVectors = true,
	      computeRightSingularVectors = true,
	      autoTranspose = false
	    } = options;
	    let wantu = Boolean(computeLeftSingularVectors);
	    let wantv = Boolean(computeRightSingularVectors);
	    let swapped = false;
	    let a;
	    if (m < n) {
	      if (!autoTranspose) {
	        a = value.clone();
	        // eslint-disable-next-line no-console
	        console.warn('Computing SVD on a matrix with more columns than rows. Consider enabling autoTranspose');
	      } else {
	        a = value.transpose();
	        m = a.rows;
	        n = a.columns;
	        swapped = true;
	        let aux = wantu;
	        wantu = wantv;
	        wantv = aux;
	      }
	    } else {
	      a = value.clone();
	    }
	    let nu = Math.min(m, n);
	    let ni = Math.min(m + 1, n);
	    let s = new Float64Array(ni);
	    let U = new Matrix$1(m, nu);
	    let V = new Matrix$1(n, n);
	    let e = new Float64Array(n);
	    let work = new Float64Array(m);
	    let si = new Float64Array(ni);
	    for (let i = 0; i < ni; i++) si[i] = i;
	    let nct = Math.min(m - 1, n);
	    let nrt = Math.max(0, Math.min(n - 2, m));
	    let mrc = Math.max(nct, nrt);
	    for (let k = 0; k < mrc; k++) {
	      if (k < nct) {
	        s[k] = 0;
	        for (let i = k; i < m; i++) {
	          s[k] = hypotenuse(s[k], a.get(i, k));
	        }
	        if (s[k] !== 0) {
	          if (a.get(k, k) < 0) {
	            s[k] = -s[k];
	          }
	          for (let i = k; i < m; i++) {
	            a.set(i, k, a.get(i, k) / s[k]);
	          }
	          a.set(k, k, a.get(k, k) + 1);
	        }
	        s[k] = -s[k];
	      }
	      for (let j = k + 1; j < n; j++) {
	        if (k < nct && s[k] !== 0) {
	          let t = 0;
	          for (let i = k; i < m; i++) {
	            t += a.get(i, k) * a.get(i, j);
	          }
	          t = -t / a.get(k, k);
	          for (let i = k; i < m; i++) {
	            a.set(i, j, a.get(i, j) + t * a.get(i, k));
	          }
	        }
	        e[j] = a.get(k, j);
	      }
	      if (wantu && k < nct) {
	        for (let i = k; i < m; i++) {
	          U.set(i, k, a.get(i, k));
	        }
	      }
	      if (k < nrt) {
	        e[k] = 0;
	        for (let i = k + 1; i < n; i++) {
	          e[k] = hypotenuse(e[k], e[i]);
	        }
	        if (e[k] !== 0) {
	          if (e[k + 1] < 0) {
	            e[k] = 0 - e[k];
	          }
	          for (let i = k + 1; i < n; i++) {
	            e[i] /= e[k];
	          }
	          e[k + 1] += 1;
	        }
	        e[k] = -e[k];
	        if (k + 1 < m && e[k] !== 0) {
	          for (let i = k + 1; i < m; i++) {
	            work[i] = 0;
	          }
	          for (let i = k + 1; i < m; i++) {
	            for (let j = k + 1; j < n; j++) {
	              work[i] += e[j] * a.get(i, j);
	            }
	          }
	          for (let j = k + 1; j < n; j++) {
	            let t = -e[j] / e[k + 1];
	            for (let i = k + 1; i < m; i++) {
	              a.set(i, j, a.get(i, j) + t * work[i]);
	            }
	          }
	        }
	        if (wantv) {
	          for (let i = k + 1; i < n; i++) {
	            V.set(i, k, e[i]);
	          }
	        }
	      }
	    }
	    let p = Math.min(n, m + 1);
	    if (nct < n) {
	      s[nct] = a.get(nct, nct);
	    }
	    if (m < p) {
	      s[p - 1] = 0;
	    }
	    if (nrt + 1 < p) {
	      e[nrt] = a.get(nrt, p - 1);
	    }
	    e[p - 1] = 0;
	    if (wantu) {
	      for (let j = nct; j < nu; j++) {
	        for (let i = 0; i < m; i++) {
	          U.set(i, j, 0);
	        }
	        U.set(j, j, 1);
	      }
	      for (let k = nct - 1; k >= 0; k--) {
	        if (s[k] !== 0) {
	          for (let j = k + 1; j < nu; j++) {
	            let t = 0;
	            for (let i = k; i < m; i++) {
	              t += U.get(i, k) * U.get(i, j);
	            }
	            t = -t / U.get(k, k);
	            for (let i = k; i < m; i++) {
	              U.set(i, j, U.get(i, j) + t * U.get(i, k));
	            }
	          }
	          for (let i = k; i < m; i++) {
	            U.set(i, k, -U.get(i, k));
	          }
	          U.set(k, k, 1 + U.get(k, k));
	          for (let i = 0; i < k - 1; i++) {
	            U.set(i, k, 0);
	          }
	        } else {
	          for (let i = 0; i < m; i++) {
	            U.set(i, k, 0);
	          }
	          U.set(k, k, 1);
	        }
	      }
	    }
	    if (wantv) {
	      for (let k = n - 1; k >= 0; k--) {
	        if (k < nrt && e[k] !== 0) {
	          for (let j = k + 1; j < n; j++) {
	            let t = 0;
	            for (let i = k + 1; i < n; i++) {
	              t += V.get(i, k) * V.get(i, j);
	            }
	            t = -t / V.get(k + 1, k);
	            for (let i = k + 1; i < n; i++) {
	              V.set(i, j, V.get(i, j) + t * V.get(i, k));
	            }
	          }
	        }
	        for (let i = 0; i < n; i++) {
	          V.set(i, k, 0);
	        }
	        V.set(k, k, 1);
	      }
	    }
	    let pp = p - 1;
	    let eps = Number.EPSILON;
	    while (p > 0) {
	      let k, kase;
	      for (k = p - 2; k >= -1; k--) {
	        if (k === -1) {
	          break;
	        }
	        const alpha = Number.MIN_VALUE + eps * Math.abs(s[k] + Math.abs(s[k + 1]));
	        if (Math.abs(e[k]) <= alpha || Number.isNaN(e[k])) {
	          e[k] = 0;
	          break;
	        }
	      }
	      if (k === p - 2) {
	        kase = 4;
	      } else {
	        let ks;
	        for (ks = p - 1; ks >= k; ks--) {
	          if (ks === k) {
	            break;
	          }
	          let t = (ks !== p ? Math.abs(e[ks]) : 0) + (ks !== k + 1 ? Math.abs(e[ks - 1]) : 0);
	          if (Math.abs(s[ks]) <= eps * t) {
	            s[ks] = 0;
	            break;
	          }
	        }
	        if (ks === k) {
	          kase = 3;
	        } else if (ks === p - 1) {
	          kase = 1;
	        } else {
	          kase = 2;
	          k = ks;
	        }
	      }
	      k++;
	      switch (kase) {
	        case 1:
	          {
	            let f = e[p - 2];
	            e[p - 2] = 0;
	            for (let j = p - 2; j >= k; j--) {
	              let t = hypotenuse(s[j], f);
	              let cs = s[j] / t;
	              let sn = f / t;
	              s[j] = t;
	              if (j !== k) {
	                f = -sn * e[j - 1];
	                e[j - 1] = cs * e[j - 1];
	              }
	              if (wantv) {
	                for (let i = 0; i < n; i++) {
	                  t = cs * V.get(i, j) + sn * V.get(i, p - 1);
	                  V.set(i, p - 1, -sn * V.get(i, j) + cs * V.get(i, p - 1));
	                  V.set(i, j, t);
	                }
	              }
	            }
	            break;
	          }
	        case 2:
	          {
	            let f = e[k - 1];
	            e[k - 1] = 0;
	            for (let j = k; j < p; j++) {
	              let t = hypotenuse(s[j], f);
	              let cs = s[j] / t;
	              let sn = f / t;
	              s[j] = t;
	              f = -sn * e[j];
	              e[j] = cs * e[j];
	              if (wantu) {
	                for (let i = 0; i < m; i++) {
	                  t = cs * U.get(i, j) + sn * U.get(i, k - 1);
	                  U.set(i, k - 1, -sn * U.get(i, j) + cs * U.get(i, k - 1));
	                  U.set(i, j, t);
	                }
	              }
	            }
	            break;
	          }
	        case 3:
	          {
	            const scale = Math.max(Math.abs(s[p - 1]), Math.abs(s[p - 2]), Math.abs(e[p - 2]), Math.abs(s[k]), Math.abs(e[k]));
	            const sp = s[p - 1] / scale;
	            const spm1 = s[p - 2] / scale;
	            const epm1 = e[p - 2] / scale;
	            const sk = s[k] / scale;
	            const ek = e[k] / scale;
	            const b = ((spm1 + sp) * (spm1 - sp) + epm1 * epm1) / 2;
	            const c = sp * epm1 * (sp * epm1);
	            let shift = 0;
	            if (b !== 0 || c !== 0) {
	              if (b < 0) {
	                shift = 0 - Math.sqrt(b * b + c);
	              } else {
	                shift = Math.sqrt(b * b + c);
	              }
	              shift = c / (b + shift);
	            }
	            let f = (sk + sp) * (sk - sp) + shift;
	            let g = sk * ek;
	            for (let j = k; j < p - 1; j++) {
	              let t = hypotenuse(f, g);
	              if (t === 0) t = Number.MIN_VALUE;
	              let cs = f / t;
	              let sn = g / t;
	              if (j !== k) {
	                e[j - 1] = t;
	              }
	              f = cs * s[j] + sn * e[j];
	              e[j] = cs * e[j] - sn * s[j];
	              g = sn * s[j + 1];
	              s[j + 1] = cs * s[j + 1];
	              if (wantv) {
	                for (let i = 0; i < n; i++) {
	                  t = cs * V.get(i, j) + sn * V.get(i, j + 1);
	                  V.set(i, j + 1, -sn * V.get(i, j) + cs * V.get(i, j + 1));
	                  V.set(i, j, t);
	                }
	              }
	              t = hypotenuse(f, g);
	              if (t === 0) t = Number.MIN_VALUE;
	              cs = f / t;
	              sn = g / t;
	              s[j] = t;
	              f = cs * e[j] + sn * s[j + 1];
	              s[j + 1] = -sn * e[j] + cs * s[j + 1];
	              g = sn * e[j + 1];
	              e[j + 1] = cs * e[j + 1];
	              if (wantu && j < m - 1) {
	                for (let i = 0; i < m; i++) {
	                  t = cs * U.get(i, j) + sn * U.get(i, j + 1);
	                  U.set(i, j + 1, -sn * U.get(i, j) + cs * U.get(i, j + 1));
	                  U.set(i, j, t);
	                }
	              }
	            }
	            e[p - 2] = f;
	            break;
	          }
	        case 4:
	          {
	            if (s[k] <= 0) {
	              s[k] = s[k] < 0 ? -s[k] : 0;
	              if (wantv) {
	                for (let i = 0; i <= pp; i++) {
	                  V.set(i, k, -V.get(i, k));
	                }
	              }
	            }
	            while (k < pp) {
	              if (s[k] >= s[k + 1]) {
	                break;
	              }
	              let t = s[k];
	              s[k] = s[k + 1];
	              s[k + 1] = t;
	              if (wantv && k < n - 1) {
	                for (let i = 0; i < n; i++) {
	                  t = V.get(i, k + 1);
	                  V.set(i, k + 1, V.get(i, k));
	                  V.set(i, k, t);
	                }
	              }
	              if (wantu && k < m - 1) {
	                for (let i = 0; i < m; i++) {
	                  t = U.get(i, k + 1);
	                  U.set(i, k + 1, U.get(i, k));
	                  U.set(i, k, t);
	                }
	              }
	              k++;
	            }
	            p--;
	            break;
	          }
	        // no default
	      }
	    }
	    if (swapped) {
	      let tmp = V;
	      V = U;
	      U = tmp;
	    }
	    this.m = m;
	    this.n = n;
	    this.s = s;
	    this.U = U;
	    this.V = V;
	  }
	  solve(value) {
	    let Y = value;
	    let e = this.threshold;
	    let scols = this.s.length;
	    let Ls = Matrix$1.zeros(scols, scols);
	    for (let i = 0; i < scols; i++) {
	      if (Math.abs(this.s[i]) <= e) {
	        Ls.set(i, i, 0);
	      } else {
	        Ls.set(i, i, 1 / this.s[i]);
	      }
	    }
	    let U = this.U;
	    let V = this.rightSingularVectors;
	    let VL = V.mmul(Ls);
	    let vrows = V.rows;
	    let urows = U.rows;
	    let VLU = Matrix$1.zeros(vrows, urows);
	    for (let i = 0; i < vrows; i++) {
	      for (let j = 0; j < urows; j++) {
	        let sum = 0;
	        for (let k = 0; k < scols; k++) {
	          sum += VL.get(i, k) * U.get(j, k);
	        }
	        VLU.set(i, j, sum);
	      }
	    }
	    return VLU.mmul(Y);
	  }
	  solveForDiagonal(value) {
	    return this.solve(Matrix$1.diag(value));
	  }
	  inverse() {
	    let V = this.V;
	    let e = this.threshold;
	    let vrows = V.rows;
	    let vcols = V.columns;
	    let X = new Matrix$1(vrows, this.s.length);
	    for (let i = 0; i < vrows; i++) {
	      for (let j = 0; j < vcols; j++) {
	        if (Math.abs(this.s[j]) > e) {
	          X.set(i, j, V.get(i, j) / this.s[j]);
	        }
	      }
	    }
	    let U = this.U;
	    let urows = U.rows;
	    let ucols = U.columns;
	    let Y = new Matrix$1(vrows, urows);
	    for (let i = 0; i < vrows; i++) {
	      for (let j = 0; j < urows; j++) {
	        let sum = 0;
	        for (let k = 0; k < ucols; k++) {
	          sum += X.get(i, k) * U.get(j, k);
	        }
	        Y.set(i, j, sum);
	      }
	    }
	    return Y;
	  }
	  get condition() {
	    return this.s[0] / this.s[Math.min(this.m, this.n) - 1];
	  }
	  get norm2() {
	    return this.s[0];
	  }
	  get rank() {
	    let tol = Math.max(this.m, this.n) * this.s[0] * Number.EPSILON;
	    let r = 0;
	    let s = this.s;
	    for (let i = 0, ii = s.length; i < ii; i++) {
	      if (s[i] > tol) {
	        r++;
	      }
	    }
	    return r;
	  }
	  get diagonal() {
	    return Array.from(this.s);
	  }
	  get threshold() {
	    return Number.EPSILON / 2 * Math.max(this.m, this.n) * this.s[0];
	  }
	  get leftSingularVectors() {
	    return this.U;
	  }
	  get rightSingularVectors() {
	    return this.V;
	  }
	  get diagonalMatrix() {
	    return Matrix$1.diag(this.s);
	  }
	}
	function inverse(matrix, useSVD = false) {
	  matrix = WrapperMatrix2D.checkMatrix(matrix);
	  if (useSVD) {
	    return new SingularValueDecomposition(matrix).inverse();
	  } else {
	    return solve$1(matrix, Matrix$1.eye(matrix.rows));
	  }
	}
	function solve$1(leftHandSide, rightHandSide, useSVD = false) {
	  leftHandSide = WrapperMatrix2D.checkMatrix(leftHandSide);
	  rightHandSide = WrapperMatrix2D.checkMatrix(rightHandSide);
	  if (useSVD) {
	    return new SingularValueDecomposition(leftHandSide).solve(rightHandSide);
	  } else {
	    return leftHandSide.isSquare() ? new LuDecomposition(leftHandSide).solve(rightHandSide) : new QrDecomposition(leftHandSide).solve(rightHandSide);
	  }
	}
	function determinant(matrix) {
	  matrix = Matrix$1.checkMatrix(matrix);
	  if (matrix.isSquare()) {
	    if (matrix.columns === 0) {
	      return 1;
	    }
	    let a, b, c, d;
	    if (matrix.columns === 2) {
	      // 2 x 2 matrix
	      a = matrix.get(0, 0);
	      b = matrix.get(0, 1);
	      c = matrix.get(1, 0);
	      d = matrix.get(1, 1);
	      return a * d - b * c;
	    } else if (matrix.columns === 3) {
	      // 3 x 3 matrix
	      let subMatrix0, subMatrix1, subMatrix2;
	      subMatrix0 = new MatrixSelectionView(matrix, [1, 2], [1, 2]);
	      subMatrix1 = new MatrixSelectionView(matrix, [1, 2], [0, 2]);
	      subMatrix2 = new MatrixSelectionView(matrix, [1, 2], [0, 1]);
	      a = matrix.get(0, 0);
	      b = matrix.get(0, 1);
	      c = matrix.get(0, 2);
	      return a * determinant(subMatrix0) - b * determinant(subMatrix1) + c * determinant(subMatrix2);
	    } else {
	      // general purpose determinant using the LU decomposition
	      return new LuDecomposition(matrix).determinant;
	    }
	  } else {
	    throw Error('determinant can only be calculated for a square matrix');
	  }
	}
	function xrange(n, exception) {
	  let range = [];
	  for (let i = 0; i < n; i++) {
	    if (i !== exception) {
	      range.push(i);
	    }
	  }
	  return range;
	}
	function dependenciesOneRow(error, matrix, index, thresholdValue = 10e-10, thresholdError = 10e-10) {
	  if (error > thresholdError) {
	    return new Array(matrix.rows + 1).fill(0);
	  } else {
	    let returnArray = matrix.addRow(index, [0]);
	    for (let i = 0; i < returnArray.rows; i++) {
	      if (Math.abs(returnArray.get(i, 0)) < thresholdValue) {
	        returnArray.set(i, 0, 0);
	      }
	    }
	    return returnArray.to1DArray();
	  }
	}
	function linearDependencies(matrix, options = {}) {
	  const {
	    thresholdValue = 10e-10,
	    thresholdError = 10e-10
	  } = options;
	  matrix = Matrix$1.checkMatrix(matrix);
	  let n = matrix.rows;
	  let results = new Matrix$1(n, n);
	  for (let i = 0; i < n; i++) {
	    let b = Matrix$1.columnVector(matrix.getRow(i));
	    let Abis = matrix.subMatrixRow(xrange(n, i)).transpose();
	    let svd = new SingularValueDecomposition(Abis);
	    let x = svd.solve(b);
	    let error = Matrix$1.sub(b, Abis.mmul(x)).abs().max();
	    results.setRow(i, dependenciesOneRow(error, x, i, thresholdValue, thresholdError));
	  }
	  return results;
	}
	function pseudoInverse(matrix, threshold = Number.EPSILON) {
	  matrix = Matrix$1.checkMatrix(matrix);
	  if (matrix.isEmpty()) {
	    // with a zero dimension, the pseudo-inverse is the transpose, since all 0xn and nx0 matrices are singular
	    // (0xn)*(nx0)*(0xn) = 0xn
	    // (nx0)*(0xn)*(nx0) = nx0
	    return matrix.transpose();
	  }
	  let svdSolution = new SingularValueDecomposition(matrix, {
	    autoTranspose: true
	  });
	  let U = svdSolution.leftSingularVectors;
	  let V = svdSolution.rightSingularVectors;
	  let s = svdSolution.diagonal;
	  for (let i = 0; i < s.length; i++) {
	    if (Math.abs(s[i]) > threshold) {
	      s[i] = 1.0 / s[i];
	    } else {
	      s[i] = 0.0;
	    }
	  }
	  return V.mmul(Matrix$1.diag(s).mmul(U.transpose()));
	}
	function covariance(xMatrix, yMatrix = xMatrix, options = {}) {
	  xMatrix = new Matrix$1(xMatrix);
	  let yIsSame = false;
	  if (typeof yMatrix === 'object' && !Matrix$1.isMatrix(yMatrix) && !isAnyArray.isAnyArray(yMatrix)) {
	    options = yMatrix;
	    yMatrix = xMatrix;
	    yIsSame = true;
	  } else {
	    yMatrix = new Matrix$1(yMatrix);
	  }
	  if (xMatrix.rows !== yMatrix.rows) {
	    throw new TypeError('Both matrices must have the same number of rows');
	  }
	  const {
	    center = true
	  } = options;
	  if (center) {
	    xMatrix = xMatrix.center('column');
	    if (!yIsSame) {
	      yMatrix = yMatrix.center('column');
	    }
	  }
	  const cov = xMatrix.transpose().mmul(yMatrix);
	  for (let i = 0; i < cov.rows; i++) {
	    for (let j = 0; j < cov.columns; j++) {
	      cov.set(i, j, cov.get(i, j) * (1 / (xMatrix.rows - 1)));
	    }
	  }
	  return cov;
	}
	function correlation(xMatrix, yMatrix = xMatrix, options = {}) {
	  xMatrix = new Matrix$1(xMatrix);
	  let yIsSame = false;
	  if (typeof yMatrix === 'object' && !Matrix$1.isMatrix(yMatrix) && !isAnyArray.isAnyArray(yMatrix)) {
	    options = yMatrix;
	    yMatrix = xMatrix;
	    yIsSame = true;
	  } else {
	    yMatrix = new Matrix$1(yMatrix);
	  }
	  if (xMatrix.rows !== yMatrix.rows) {
	    throw new TypeError('Both matrices must have the same number of rows');
	  }
	  const {
	    center = true,
	    scale = true
	  } = options;
	  if (center) {
	    xMatrix.center('column');
	    if (!yIsSame) {
	      yMatrix.center('column');
	    }
	  }
	  if (scale) {
	    xMatrix.scale('column');
	    if (!yIsSame) {
	      yMatrix.scale('column');
	    }
	  }
	  const sdx = xMatrix.standardDeviation('column', {
	    unbiased: true
	  });
	  const sdy = yIsSame ? sdx : yMatrix.standardDeviation('column', {
	    unbiased: true
	  });
	  const corr = xMatrix.transpose().mmul(yMatrix);
	  for (let i = 0; i < corr.rows; i++) {
	    for (let j = 0; j < corr.columns; j++) {
	      corr.set(i, j, corr.get(i, j) * (1 / (sdx[i] * sdy[j])) * (1 / (xMatrix.rows - 1)));
	    }
	  }
	  return corr;
	}
	class EigenvalueDecomposition {
	  constructor(matrix, options = {}) {
	    const {
	      assumeSymmetric = false
	    } = options;
	    matrix = WrapperMatrix2D.checkMatrix(matrix);
	    if (!matrix.isSquare()) {
	      throw new Error('Matrix is not a square matrix');
	    }
	    if (matrix.isEmpty()) {
	      throw new Error('Matrix must be non-empty');
	    }
	    let n = matrix.columns;
	    let V = new Matrix$1(n, n);
	    let d = new Float64Array(n);
	    let e = new Float64Array(n);
	    let value = matrix;
	    let i, j;
	    let isSymmetric = false;
	    if (assumeSymmetric) {
	      isSymmetric = true;
	    } else {
	      isSymmetric = matrix.isSymmetric();
	    }
	    if (isSymmetric) {
	      for (i = 0; i < n; i++) {
	        for (j = 0; j < n; j++) {
	          V.set(i, j, value.get(i, j));
	        }
	      }
	      tred2(n, e, d, V);
	      tql2(n, e, d, V);
	    } else {
	      let H = new Matrix$1(n, n);
	      let ort = new Float64Array(n);
	      for (j = 0; j < n; j++) {
	        for (i = 0; i < n; i++) {
	          H.set(i, j, value.get(i, j));
	        }
	      }
	      orthes(n, H, ort, V);
	      hqr2(n, e, d, V, H);
	    }
	    this.n = n;
	    this.e = e;
	    this.d = d;
	    this.V = V;
	  }
	  get realEigenvalues() {
	    return Array.from(this.d);
	  }
	  get imaginaryEigenvalues() {
	    return Array.from(this.e);
	  }
	  get eigenvectorMatrix() {
	    return this.V;
	  }
	  get diagonalMatrix() {
	    let n = this.n;
	    let e = this.e;
	    let d = this.d;
	    let X = new Matrix$1(n, n);
	    let i, j;
	    for (i = 0; i < n; i++) {
	      for (j = 0; j < n; j++) {
	        X.set(i, j, 0);
	      }
	      X.set(i, i, d[i]);
	      if (e[i] > 0) {
	        X.set(i, i + 1, e[i]);
	      } else if (e[i] < 0) {
	        X.set(i, i - 1, e[i]);
	      }
	    }
	    return X;
	  }
	}
	function tred2(n, e, d, V) {
	  let f, g, h, i, j, k, hh, scale;
	  for (j = 0; j < n; j++) {
	    d[j] = V.get(n - 1, j);
	  }
	  for (i = n - 1; i > 0; i--) {
	    scale = 0;
	    h = 0;
	    for (k = 0; k < i; k++) {
	      scale = scale + Math.abs(d[k]);
	    }
	    if (scale === 0) {
	      e[i] = d[i - 1];
	      for (j = 0; j < i; j++) {
	        d[j] = V.get(i - 1, j);
	        V.set(i, j, 0);
	        V.set(j, i, 0);
	      }
	    } else {
	      for (k = 0; k < i; k++) {
	        d[k] /= scale;
	        h += d[k] * d[k];
	      }
	      f = d[i - 1];
	      g = Math.sqrt(h);
	      if (f > 0) {
	        g = -g;
	      }
	      e[i] = scale * g;
	      h = h - f * g;
	      d[i - 1] = f - g;
	      for (j = 0; j < i; j++) {
	        e[j] = 0;
	      }
	      for (j = 0; j < i; j++) {
	        f = d[j];
	        V.set(j, i, f);
	        g = e[j] + V.get(j, j) * f;
	        for (k = j + 1; k <= i - 1; k++) {
	          g += V.get(k, j) * d[k];
	          e[k] += V.get(k, j) * f;
	        }
	        e[j] = g;
	      }
	      f = 0;
	      for (j = 0; j < i; j++) {
	        e[j] /= h;
	        f += e[j] * d[j];
	      }
	      hh = f / (h + h);
	      for (j = 0; j < i; j++) {
	        e[j] -= hh * d[j];
	      }
	      for (j = 0; j < i; j++) {
	        f = d[j];
	        g = e[j];
	        for (k = j; k <= i - 1; k++) {
	          V.set(k, j, V.get(k, j) - (f * e[k] + g * d[k]));
	        }
	        d[j] = V.get(i - 1, j);
	        V.set(i, j, 0);
	      }
	    }
	    d[i] = h;
	  }
	  for (i = 0; i < n - 1; i++) {
	    V.set(n - 1, i, V.get(i, i));
	    V.set(i, i, 1);
	    h = d[i + 1];
	    if (h !== 0) {
	      for (k = 0; k <= i; k++) {
	        d[k] = V.get(k, i + 1) / h;
	      }
	      for (j = 0; j <= i; j++) {
	        g = 0;
	        for (k = 0; k <= i; k++) {
	          g += V.get(k, i + 1) * V.get(k, j);
	        }
	        for (k = 0; k <= i; k++) {
	          V.set(k, j, V.get(k, j) - g * d[k]);
	        }
	      }
	    }
	    for (k = 0; k <= i; k++) {
	      V.set(k, i + 1, 0);
	    }
	  }
	  for (j = 0; j < n; j++) {
	    d[j] = V.get(n - 1, j);
	    V.set(n - 1, j, 0);
	  }
	  V.set(n - 1, n - 1, 1);
	  e[0] = 0;
	}
	function tql2(n, e, d, V) {
	  let g, h, i, j, k, l, m, p, r, dl1, c, c2, c3, el1, s, s2;
	  for (i = 1; i < n; i++) {
	    e[i - 1] = e[i];
	  }
	  e[n - 1] = 0;
	  let f = 0;
	  let tst1 = 0;
	  let eps = Number.EPSILON;
	  for (l = 0; l < n; l++) {
	    tst1 = Math.max(tst1, Math.abs(d[l]) + Math.abs(e[l]));
	    m = l;
	    while (m < n) {
	      if (Math.abs(e[m]) <= eps * tst1) {
	        break;
	      }
	      m++;
	    }
	    if (m > l) {
	      do {
	        g = d[l];
	        p = (d[l + 1] - g) / (2 * e[l]);
	        r = hypotenuse(p, 1);
	        if (p < 0) {
	          r = -r;
	        }
	        d[l] = e[l] / (p + r);
	        d[l + 1] = e[l] * (p + r);
	        dl1 = d[l + 1];
	        h = g - d[l];
	        for (i = l + 2; i < n; i++) {
	          d[i] -= h;
	        }
	        f = f + h;
	        p = d[m];
	        c = 1;
	        c2 = c;
	        c3 = c;
	        el1 = e[l + 1];
	        s = 0;
	        s2 = 0;
	        for (i = m - 1; i >= l; i--) {
	          c3 = c2;
	          c2 = c;
	          s2 = s;
	          g = c * e[i];
	          h = c * p;
	          r = hypotenuse(p, e[i]);
	          e[i + 1] = s * r;
	          s = e[i] / r;
	          c = p / r;
	          p = c * d[i] - s * g;
	          d[i + 1] = h + s * (c * g + s * d[i]);
	          for (k = 0; k < n; k++) {
	            h = V.get(k, i + 1);
	            V.set(k, i + 1, s * V.get(k, i) + c * h);
	            V.set(k, i, c * V.get(k, i) - s * h);
	          }
	        }
	        p = -s * s2 * c3 * el1 * e[l] / dl1;
	        e[l] = s * p;
	        d[l] = c * p;
	      } while (Math.abs(e[l]) > eps * tst1);
	    }
	    d[l] = d[l] + f;
	    e[l] = 0;
	  }
	  for (i = 0; i < n - 1; i++) {
	    k = i;
	    p = d[i];
	    for (j = i + 1; j < n; j++) {
	      if (d[j] < p) {
	        k = j;
	        p = d[j];
	      }
	    }
	    if (k !== i) {
	      d[k] = d[i];
	      d[i] = p;
	      for (j = 0; j < n; j++) {
	        p = V.get(j, i);
	        V.set(j, i, V.get(j, k));
	        V.set(j, k, p);
	      }
	    }
	  }
	}
	function orthes(n, H, ort, V) {
	  let low = 0;
	  let high = n - 1;
	  let f, g, h, i, j, m;
	  let scale;
	  for (m = low + 1; m <= high - 1; m++) {
	    scale = 0;
	    for (i = m; i <= high; i++) {
	      scale = scale + Math.abs(H.get(i, m - 1));
	    }
	    if (scale !== 0) {
	      h = 0;
	      for (i = high; i >= m; i--) {
	        ort[i] = H.get(i, m - 1) / scale;
	        h += ort[i] * ort[i];
	      }
	      g = Math.sqrt(h);
	      if (ort[m] > 0) {
	        g = -g;
	      }
	      h = h - ort[m] * g;
	      ort[m] = ort[m] - g;
	      for (j = m; j < n; j++) {
	        f = 0;
	        for (i = high; i >= m; i--) {
	          f += ort[i] * H.get(i, j);
	        }
	        f = f / h;
	        for (i = m; i <= high; i++) {
	          H.set(i, j, H.get(i, j) - f * ort[i]);
	        }
	      }
	      for (i = 0; i <= high; i++) {
	        f = 0;
	        for (j = high; j >= m; j--) {
	          f += ort[j] * H.get(i, j);
	        }
	        f = f / h;
	        for (j = m; j <= high; j++) {
	          H.set(i, j, H.get(i, j) - f * ort[j]);
	        }
	      }
	      ort[m] = scale * ort[m];
	      H.set(m, m - 1, scale * g);
	    }
	  }
	  for (i = 0; i < n; i++) {
	    for (j = 0; j < n; j++) {
	      V.set(i, j, i === j ? 1 : 0);
	    }
	  }
	  for (m = high - 1; m >= low + 1; m--) {
	    if (H.get(m, m - 1) !== 0) {
	      for (i = m + 1; i <= high; i++) {
	        ort[i] = H.get(i, m - 1);
	      }
	      for (j = m; j <= high; j++) {
	        g = 0;
	        for (i = m; i <= high; i++) {
	          g += ort[i] * V.get(i, j);
	        }
	        g = g / ort[m] / H.get(m, m - 1);
	        for (i = m; i <= high; i++) {
	          V.set(i, j, V.get(i, j) + g * ort[i]);
	        }
	      }
	    }
	  }
	}
	function hqr2(nn, e, d, V, H) {
	  let n = nn - 1;
	  let low = 0;
	  let high = nn - 1;
	  let eps = Number.EPSILON;
	  let exshift = 0;
	  let norm = 0;
	  let p = 0;
	  let q = 0;
	  let r = 0;
	  let s = 0;
	  let z = 0;
	  let iter = 0;
	  let i, j, k, l, m, t, w, x, y;
	  let ra, sa, vr, vi;
	  let notlast, cdivres;
	  for (i = 0; i < nn; i++) {
	    if (i < low || i > high) {
	      d[i] = H.get(i, i);
	      e[i] = 0;
	    }
	    for (j = Math.max(i - 1, 0); j < nn; j++) {
	      norm = norm + Math.abs(H.get(i, j));
	    }
	  }
	  while (n >= low) {
	    l = n;
	    while (l > low) {
	      s = Math.abs(H.get(l - 1, l - 1)) + Math.abs(H.get(l, l));
	      if (s === 0) {
	        s = norm;
	      }
	      if (Math.abs(H.get(l, l - 1)) < eps * s) {
	        break;
	      }
	      l--;
	    }
	    if (l === n) {
	      H.set(n, n, H.get(n, n) + exshift);
	      d[n] = H.get(n, n);
	      e[n] = 0;
	      n--;
	      iter = 0;
	    } else if (l === n - 1) {
	      w = H.get(n, n - 1) * H.get(n - 1, n);
	      p = (H.get(n - 1, n - 1) - H.get(n, n)) / 2;
	      q = p * p + w;
	      z = Math.sqrt(Math.abs(q));
	      H.set(n, n, H.get(n, n) + exshift);
	      H.set(n - 1, n - 1, H.get(n - 1, n - 1) + exshift);
	      x = H.get(n, n);
	      if (q >= 0) {
	        z = p >= 0 ? p + z : p - z;
	        d[n - 1] = x + z;
	        d[n] = d[n - 1];
	        if (z !== 0) {
	          d[n] = x - w / z;
	        }
	        e[n - 1] = 0;
	        e[n] = 0;
	        x = H.get(n, n - 1);
	        s = Math.abs(x) + Math.abs(z);
	        p = x / s;
	        q = z / s;
	        r = Math.sqrt(p * p + q * q);
	        p = p / r;
	        q = q / r;
	        for (j = n - 1; j < nn; j++) {
	          z = H.get(n - 1, j);
	          H.set(n - 1, j, q * z + p * H.get(n, j));
	          H.set(n, j, q * H.get(n, j) - p * z);
	        }
	        for (i = 0; i <= n; i++) {
	          z = H.get(i, n - 1);
	          H.set(i, n - 1, q * z + p * H.get(i, n));
	          H.set(i, n, q * H.get(i, n) - p * z);
	        }
	        for (i = low; i <= high; i++) {
	          z = V.get(i, n - 1);
	          V.set(i, n - 1, q * z + p * V.get(i, n));
	          V.set(i, n, q * V.get(i, n) - p * z);
	        }
	      } else {
	        d[n - 1] = x + p;
	        d[n] = x + p;
	        e[n - 1] = z;
	        e[n] = -z;
	      }
	      n = n - 2;
	      iter = 0;
	    } else {
	      x = H.get(n, n);
	      y = 0;
	      w = 0;
	      if (l < n) {
	        y = H.get(n - 1, n - 1);
	        w = H.get(n, n - 1) * H.get(n - 1, n);
	      }
	      if (iter === 10) {
	        exshift += x;
	        for (i = low; i <= n; i++) {
	          H.set(i, i, H.get(i, i) - x);
	        }
	        s = Math.abs(H.get(n, n - 1)) + Math.abs(H.get(n - 1, n - 2));
	        // eslint-disable-next-line no-multi-assign
	        x = y = 0.75 * s;
	        w = -0.4375 * s * s;
	      }
	      if (iter === 30) {
	        s = (y - x) / 2;
	        s = s * s + w;
	        if (s > 0) {
	          s = Math.sqrt(s);
	          if (y < x) {
	            s = -s;
	          }
	          s = x - w / ((y - x) / 2 + s);
	          for (i = low; i <= n; i++) {
	            H.set(i, i, H.get(i, i) - s);
	          }
	          exshift += s;
	          // eslint-disable-next-line no-multi-assign
	          x = y = w = 0.964;
	        }
	      }
	      iter = iter + 1;
	      m = n - 2;
	      while (m >= l) {
	        z = H.get(m, m);
	        r = x - z;
	        s = y - z;
	        p = (r * s - w) / H.get(m + 1, m) + H.get(m, m + 1);
	        q = H.get(m + 1, m + 1) - z - r - s;
	        r = H.get(m + 2, m + 1);
	        s = Math.abs(p) + Math.abs(q) + Math.abs(r);
	        p = p / s;
	        q = q / s;
	        r = r / s;
	        if (m === l) {
	          break;
	        }
	        if (Math.abs(H.get(m, m - 1)) * (Math.abs(q) + Math.abs(r)) < eps * (Math.abs(p) * (Math.abs(H.get(m - 1, m - 1)) + Math.abs(z) + Math.abs(H.get(m + 1, m + 1))))) {
	          break;
	        }
	        m--;
	      }
	      for (i = m + 2; i <= n; i++) {
	        H.set(i, i - 2, 0);
	        if (i > m + 2) {
	          H.set(i, i - 3, 0);
	        }
	      }
	      for (k = m; k <= n - 1; k++) {
	        notlast = k !== n - 1;
	        if (k !== m) {
	          p = H.get(k, k - 1);
	          q = H.get(k + 1, k - 1);
	          r = notlast ? H.get(k + 2, k - 1) : 0;
	          x = Math.abs(p) + Math.abs(q) + Math.abs(r);
	          if (x !== 0) {
	            p = p / x;
	            q = q / x;
	            r = r / x;
	          }
	        }
	        if (x === 0) {
	          break;
	        }
	        s = Math.sqrt(p * p + q * q + r * r);
	        if (p < 0) {
	          s = -s;
	        }
	        if (s !== 0) {
	          if (k !== m) {
	            H.set(k, k - 1, -s * x);
	          } else if (l !== m) {
	            H.set(k, k - 1, -H.get(k, k - 1));
	          }
	          p = p + s;
	          x = p / s;
	          y = q / s;
	          z = r / s;
	          q = q / p;
	          r = r / p;
	          for (j = k; j < nn; j++) {
	            p = H.get(k, j) + q * H.get(k + 1, j);
	            if (notlast) {
	              p = p + r * H.get(k + 2, j);
	              H.set(k + 2, j, H.get(k + 2, j) - p * z);
	            }
	            H.set(k, j, H.get(k, j) - p * x);
	            H.set(k + 1, j, H.get(k + 1, j) - p * y);
	          }
	          for (i = 0; i <= Math.min(n, k + 3); i++) {
	            p = x * H.get(i, k) + y * H.get(i, k + 1);
	            if (notlast) {
	              p = p + z * H.get(i, k + 2);
	              H.set(i, k + 2, H.get(i, k + 2) - p * r);
	            }
	            H.set(i, k, H.get(i, k) - p);
	            H.set(i, k + 1, H.get(i, k + 1) - p * q);
	          }
	          for (i = low; i <= high; i++) {
	            p = x * V.get(i, k) + y * V.get(i, k + 1);
	            if (notlast) {
	              p = p + z * V.get(i, k + 2);
	              V.set(i, k + 2, V.get(i, k + 2) - p * r);
	            }
	            V.set(i, k, V.get(i, k) - p);
	            V.set(i, k + 1, V.get(i, k + 1) - p * q);
	          }
	        }
	      }
	    }
	  }
	  if (norm === 0) {
	    return;
	  }
	  for (n = nn - 1; n >= 0; n--) {
	    p = d[n];
	    q = e[n];
	    if (q === 0) {
	      l = n;
	      H.set(n, n, 1);
	      for (i = n - 1; i >= 0; i--) {
	        w = H.get(i, i) - p;
	        r = 0;
	        for (j = l; j <= n; j++) {
	          r = r + H.get(i, j) * H.get(j, n);
	        }
	        if (e[i] < 0) {
	          z = w;
	          s = r;
	        } else {
	          l = i;
	          if (e[i] === 0) {
	            H.set(i, n, w !== 0 ? -r / w : -r / (eps * norm));
	          } else {
	            x = H.get(i, i + 1);
	            y = H.get(i + 1, i);
	            q = (d[i] - p) * (d[i] - p) + e[i] * e[i];
	            t = (x * s - z * r) / q;
	            H.set(i, n, t);
	            H.set(i + 1, n, Math.abs(x) > Math.abs(z) ? (-r - w * t) / x : (-s - y * t) / z);
	          }
	          t = Math.abs(H.get(i, n));
	          if (eps * t * t > 1) {
	            for (j = i; j <= n; j++) {
	              H.set(j, n, H.get(j, n) / t);
	            }
	          }
	        }
	      }
	    } else if (q < 0) {
	      l = n - 1;
	      if (Math.abs(H.get(n, n - 1)) > Math.abs(H.get(n - 1, n))) {
	        H.set(n - 1, n - 1, q / H.get(n, n - 1));
	        H.set(n - 1, n, -(H.get(n, n) - p) / H.get(n, n - 1));
	      } else {
	        cdivres = cdiv(0, -H.get(n - 1, n), H.get(n - 1, n - 1) - p, q);
	        H.set(n - 1, n - 1, cdivres[0]);
	        H.set(n - 1, n, cdivres[1]);
	      }
	      H.set(n, n - 1, 0);
	      H.set(n, n, 1);
	      for (i = n - 2; i >= 0; i--) {
	        ra = 0;
	        sa = 0;
	        for (j = l; j <= n; j++) {
	          ra = ra + H.get(i, j) * H.get(j, n - 1);
	          sa = sa + H.get(i, j) * H.get(j, n);
	        }
	        w = H.get(i, i) - p;
	        if (e[i] < 0) {
	          z = w;
	          r = ra;
	          s = sa;
	        } else {
	          l = i;
	          if (e[i] === 0) {
	            cdivres = cdiv(-ra, -sa, w, q);
	            H.set(i, n - 1, cdivres[0]);
	            H.set(i, n, cdivres[1]);
	          } else {
	            x = H.get(i, i + 1);
	            y = H.get(i + 1, i);
	            vr = (d[i] - p) * (d[i] - p) + e[i] * e[i] - q * q;
	            vi = (d[i] - p) * 2 * q;
	            if (vr === 0 && vi === 0) {
	              vr = eps * norm * (Math.abs(w) + Math.abs(q) + Math.abs(x) + Math.abs(y) + Math.abs(z));
	            }
	            cdivres = cdiv(x * r - z * ra + q * sa, x * s - z * sa - q * ra, vr, vi);
	            H.set(i, n - 1, cdivres[0]);
	            H.set(i, n, cdivres[1]);
	            if (Math.abs(x) > Math.abs(z) + Math.abs(q)) {
	              H.set(i + 1, n - 1, (-ra - w * H.get(i, n - 1) + q * H.get(i, n)) / x);
	              H.set(i + 1, n, (-sa - w * H.get(i, n) - q * H.get(i, n - 1)) / x);
	            } else {
	              cdivres = cdiv(-r - y * H.get(i, n - 1), -s - y * H.get(i, n), z, q);
	              H.set(i + 1, n - 1, cdivres[0]);
	              H.set(i + 1, n, cdivres[1]);
	            }
	          }
	          t = Math.max(Math.abs(H.get(i, n - 1)), Math.abs(H.get(i, n)));
	          if (eps * t * t > 1) {
	            for (j = i; j <= n; j++) {
	              H.set(j, n - 1, H.get(j, n - 1) / t);
	              H.set(j, n, H.get(j, n) / t);
	            }
	          }
	        }
	      }
	    }
	  }
	  for (i = 0; i < nn; i++) {
	    if (i < low || i > high) {
	      for (j = i; j < nn; j++) {
	        V.set(i, j, H.get(i, j));
	      }
	    }
	  }
	  for (j = nn - 1; j >= low; j--) {
	    for (i = low; i <= high; i++) {
	      z = 0;
	      for (k = low; k <= Math.min(j, high); k++) {
	        z = z + V.get(i, k) * H.get(k, j);
	      }
	      V.set(i, j, z);
	    }
	  }
	}
	function cdiv(xr, xi, yr, yi) {
	  let r, d;
	  if (Math.abs(yr) > Math.abs(yi)) {
	    r = yi / yr;
	    d = yr + r * yi;
	    return [(xr + r * xi) / d, (xi - r * xr) / d];
	  } else {
	    r = yr / yi;
	    d = yi + r * yr;
	    return [(r * xr + xi) / d, (r * xi - xr) / d];
	  }
	}
	class CholeskyDecomposition {
	  constructor(value) {
	    value = WrapperMatrix2D.checkMatrix(value);
	    if (!value.isSymmetric()) {
	      throw new Error('Matrix is not symmetric');
	    }
	    let a = value;
	    let dimension = a.rows;
	    let l = new Matrix$1(dimension, dimension);
	    let positiveDefinite = true;
	    let i, j, k;
	    for (j = 0; j < dimension; j++) {
	      let d = 0;
	      for (k = 0; k < j; k++) {
	        let s = 0;
	        for (i = 0; i < k; i++) {
	          s += l.get(k, i) * l.get(j, i);
	        }
	        s = (a.get(j, k) - s) / l.get(k, k);
	        l.set(j, k, s);
	        d = d + s * s;
	      }
	      d = a.get(j, j) - d;
	      positiveDefinite &= d > 0;
	      l.set(j, j, Math.sqrt(Math.max(d, 0)));
	      for (k = j + 1; k < dimension; k++) {
	        l.set(j, k, 0);
	      }
	    }
	    this.L = l;
	    this.positiveDefinite = Boolean(positiveDefinite);
	  }
	  isPositiveDefinite() {
	    return this.positiveDefinite;
	  }
	  solve(value) {
	    value = WrapperMatrix2D.checkMatrix(value);
	    let l = this.L;
	    let dimension = l.rows;
	    if (value.rows !== dimension) {
	      throw new Error('Matrix dimensions do not match');
	    }
	    if (this.isPositiveDefinite() === false) {
	      throw new Error('Matrix is not positive definite');
	    }
	    let count = value.columns;
	    let B = value.clone();
	    let i, j, k;
	    for (k = 0; k < dimension; k++) {
	      for (j = 0; j < count; j++) {
	        for (i = 0; i < k; i++) {
	          B.set(k, j, B.get(k, j) - B.get(i, j) * l.get(k, i));
	        }
	        B.set(k, j, B.get(k, j) / l.get(k, k));
	      }
	    }
	    for (k = dimension - 1; k >= 0; k--) {
	      for (j = 0; j < count; j++) {
	        for (i = k + 1; i < dimension; i++) {
	          B.set(k, j, B.get(k, j) - B.get(i, j) * l.get(i, k));
	        }
	        B.set(k, j, B.get(k, j) / l.get(k, k));
	      }
	    }
	    return B;
	  }
	  get lowerTriangularMatrix() {
	    return this.L;
	  }
	}
	class nipals {
	  constructor(X, options = {}) {
	    X = WrapperMatrix2D.checkMatrix(X);
	    let {
	      Y
	    } = options;
	    const {
	      scaleScores = false,
	      maxIterations = 1000,
	      terminationCriteria = 1e-10
	    } = options;
	    let u;
	    if (Y) {
	      if (isAnyArray.isAnyArray(Y) && typeof Y[0] === 'number') {
	        Y = Matrix$1.columnVector(Y);
	      } else {
	        Y = WrapperMatrix2D.checkMatrix(Y);
	      }
	      if (Y.rows !== X.rows) {
	        throw new Error('Y should have the same number of rows as X');
	      }
	      u = Y.getColumnVector(0);
	    } else {
	      u = X.getColumnVector(0);
	    }
	    let diff = 1;
	    let t, q, w, tOld;
	    for (let counter = 0; counter < maxIterations && diff > terminationCriteria; counter++) {
	      w = X.transpose().mmul(u).div(u.transpose().mmul(u).get(0, 0));
	      w = w.div(w.norm());
	      t = X.mmul(w).div(w.transpose().mmul(w).get(0, 0));
	      if (counter > 0) {
	        diff = t.clone().sub(tOld).pow(2).sum();
	      }
	      tOld = t.clone();
	      if (Y) {
	        q = Y.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
	        q = q.div(q.norm());
	        u = Y.mmul(q).div(q.transpose().mmul(q).get(0, 0));
	      } else {
	        u = t;
	      }
	    }
	    if (Y) {
	      let p = X.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
	      p = p.div(p.norm());
	      let xResidual = X.clone().sub(t.clone().mmul(p.transpose()));
	      let residual = u.transpose().mmul(t).div(t.transpose().mmul(t).get(0, 0));
	      let yResidual = Y.clone().sub(t.clone().mulS(residual.get(0, 0)).mmul(q.transpose()));
	      this.t = t;
	      this.p = p.transpose();
	      this.w = w.transpose();
	      this.q = q;
	      this.u = u;
	      this.s = t.transpose().mmul(t);
	      this.xResidual = xResidual;
	      this.yResidual = yResidual;
	      this.betas = residual;
	    } else {
	      this.w = w.transpose();
	      this.s = t.transpose().mmul(t).sqrt();
	      if (scaleScores) {
	        this.t = t.clone().div(this.s.get(0, 0));
	      } else {
	        this.t = t;
	      }
	      this.xResidual = X.sub(t.mmul(w.transpose()));
	    }
	  }
	}
	matrix.AbstractMatrix = AbstractMatrix;
	matrix.CHO = CholeskyDecomposition;
	matrix.CholeskyDecomposition = CholeskyDecomposition;
	matrix.DistanceMatrix = DistanceMatrix;
	matrix.EVD = EigenvalueDecomposition;
	matrix.EigenvalueDecomposition = EigenvalueDecomposition;
	matrix.LU = LuDecomposition;
	matrix.LuDecomposition = LuDecomposition;
	var Matrix_1 = matrix.Matrix = Matrix$1;
	matrix.MatrixColumnSelectionView = MatrixColumnSelectionView;
	matrix.MatrixColumnView = MatrixColumnView;
	matrix.MatrixFlipColumnView = MatrixFlipColumnView;
	matrix.MatrixFlipRowView = MatrixFlipRowView;
	matrix.MatrixRowSelectionView = MatrixRowSelectionView;
	matrix.MatrixRowView = MatrixRowView;
	matrix.MatrixSelectionView = MatrixSelectionView;
	matrix.MatrixSubView = MatrixSubView;
	var MatrixTransposeView_1 = matrix.MatrixTransposeView = MatrixTransposeView$1;
	matrix.NIPALS = nipals;
	matrix.Nipals = nipals;
	matrix.QR = QrDecomposition;
	matrix.QrDecomposition = QrDecomposition;
	matrix.SVD = SingularValueDecomposition;
	matrix.SingularValueDecomposition = SingularValueDecomposition;
	matrix.SymmetricMatrix = SymmetricMatrix;
	matrix.WrapperMatrix1D = WrapperMatrix1D;
	matrix.WrapperMatrix2D = WrapperMatrix2D;
	matrix.correlation = correlation;
	matrix.covariance = covariance;
	var _default = matrix.default = Matrix$1;
	matrix.determinant = determinant;
	matrix.inverse = inverse;
	matrix.linearDependencies = linearDependencies;
	matrix.pseudoInverse = pseudoInverse;
	var solve_1 = matrix.solve = solve$1;
	matrix.wrap = wrap;

	const Matrix = Matrix_1;
	const MatrixTransposeView = MatrixTransposeView_1;
	_default.Matrix ? _default.Matrix : Matrix_1;
	const solve = solve_1;

	/**
	 * This function divide the first array by the second array or a constant value to each element of the first array
	 *
	 * @param array1 - first array
	 * @param array2 - second array or number
	 * @param options - options
	 */
	function xDivide(array1, array2, options = {}) {
	  let isConstant = false;
	  let constant = 0;
	  if (isAnyArray$1(array2)) {
	    if (array1.length !== array2.length) {
	      throw new Error('xDivide: size of array1 and array2 must be identical');
	    }
	  } else {
	    isConstant = true;
	    constant = Number(array2);
	  }
	  const array3 = getOutputArray(options.output, array1.length);
	  if (isConstant) {
	    for (let i = 0; i < array1.length; i++) {
	      array3[i] = array1[i] / constant;
	    }
	  } else {
	    for (let i = 0; i < array1.length; i++) {
	      array3[i] = array1[i] / array2[i];
	    }
	  }
	  return array3;
	}

	/**
	 * Returns a copy of the data as Float64
	 *
	 * @param array - array of numbers
	 */
	function xEnsureFloat64(array) {
	  xCheck(array);
	  if (array instanceof Float64Array) {
	    return array.slice(0);
	  }
	  return Float64Array.from(array);
	}

	/**
	 * Returns the closest index of a `target`
	 *
	 * @param array - array of numbers
	 * @param target - target
	 * @returns - closest index
	 */
	function xFindClosestIndex(array, target, options = {}) {
	  const {
	    sorted = true
	  } = options;
	  if (sorted) {
	    let low = 0;
	    let high = array.length - 1;
	    let middle = 0;
	    while (high - low > 1) {
	      middle = low + (high - low >> 1);
	      if (array[middle] < target) {
	        low = middle;
	      } else if (array[middle] > target) {
	        high = middle;
	      } else {
	        return middle;
	      }
	    }
	    if (low < array.length - 1) {
	      if (Math.abs(target - array[low]) < Math.abs(array[low + 1] - target)) {
	        return low;
	      } else {
	        return low + 1;
	      }
	    } else {
	      return low;
	    }
	  } else {
	    let index = 0;
	    let diff = Number.POSITIVE_INFINITY;
	    for (let i = 0; i < array.length; i++) {
	      const currentDiff = Math.abs(array[i] - target);
	      if (currentDiff < diff) {
	        diff = currentDiff;
	        index = i;
	      }
	    }
	    return index;
	  }
	}

	/**
	 * Returns an object with {fromIndex, toIndex} for a specific from / to
	 *
	 * @param x - array of numbers
	 * @param options - Options
	 */
	function xGetFromToIndex(x, options = {}) {
	  let {
	    fromIndex,
	    toIndex
	  } = options;
	  const {
	    from,
	    to
	  } = options;
	  if (typeof fromIndex === 'undefined') {
	    if (typeof from !== 'undefined') {
	      fromIndex = xFindClosestIndex(x, from);
	    } else {
	      fromIndex = 0;
	    }
	  }
	  if (typeof toIndex === 'undefined') {
	    if (typeof to !== 'undefined') {
	      toIndex = xFindClosestIndex(x, to);
	    } else {
	      toIndex = x.length - 1;
	    }
	  }
	  if (fromIndex < 0) fromIndex = 0;
	  if (toIndex < 0) toIndex = 0;
	  if (fromIndex >= x.length) fromIndex = x.length - 1;
	  if (toIndex >= x.length) toIndex = x.length - 1;
	  if (fromIndex > toIndex) [fromIndex, toIndex] = [toIndex, fromIndex];
	  return {
	    fromIndex,
	    toIndex
	  };
	}

	/**
	 * Computes the maximal value of an array of values
	 *
	 * @param array - array of numbers
	 * @param options - options
	 */
	function xMaxValue(array, options = {}) {
	  xCheck(array);
	  const {
	    fromIndex,
	    toIndex
	  } = xGetFromToIndex(array, options);
	  let maxValue = array[fromIndex];
	  for (let i = fromIndex + 1; i <= toIndex; i++) {
	    if (array[i] > maxValue) {
	      maxValue = array[i];
	    }
	  }
	  return maxValue;
	}

	/**
	 * Computes the minimal value of an array of values
	 *
	 * @param array - array of numbers
	 * @param options - options
	 */
	function xMinValue(array, options = {}) {
	  xCheck(array);
	  const {
	    fromIndex,
	    toIndex
	  } = xGetFromToIndex(array, options);
	  let minValue = array[fromIndex];
	  for (let i = fromIndex + 1; i <= toIndex; i++) {
	    if (array[i] < minValue) {
	      minValue = array[i];
	    }
	  }
	  return minValue;
	}

	/**
	 * Check if the values are separated always by the same difference
	 *
	 * @param array - Monotone growing array of number
	 */
	function xIsEquallySpaced(array, options = {}) {
	  if (array.length < 3) return true;
	  const {
	    tolerance = 0.05
	  } = options;
	  let maxDx = 0;
	  let minDx = Number.MAX_SAFE_INTEGER;
	  for (let i = 0; i < array.length - 1; ++i) {
	    const absoluteDifference = array[i + 1] - array[i];
	    if (absoluteDifference < minDx) {
	      minDx = absoluteDifference;
	    }
	    if (absoluteDifference > maxDx) {
	      maxDx = absoluteDifference;
	    }
	  }
	  return (maxDx - minDx) / maxDx < tolerance;
	}

	/**
	 * Returns true if x is monotone
	 *
	 * @param array - array of numbers
	 */
	function xIsMonotonic(array) {
	  if (array.length <= 2) {
	    return 1;
	  }
	  if (array[0] === array[1]) {
	    // maybe a constant series
	    for (let i = 1; i < array.length - 1; i++) {
	      if (array[i] !== array[i + 1]) return 0;
	    }
	    return 1;
	  }
	  if (array[0] < array[array.length - 1]) {
	    for (let i = 0; i < array.length - 1; i++) {
	      if (array[i] >= array[i + 1]) return 0;
	    }
	    return 1;
	  } else {
	    for (let i = 0; i < array.length - 1; i++) {
	      if (array[i] <= array[i + 1]) return 0;
	    }
	    return -1;
	  }
	}

	/**
	 * Computes the mean value of an array of values
	 *
	 * @param array - array of numbers
	 * @param options - options
	 */
	function xMean(array, options = {}) {
	  xCheck(array);
	  const {
	    fromIndex,
	    toIndex
	  } = xGetFromToIndex(array, options);
	  let sumValue = array[fromIndex];
	  for (let i = fromIndex + 1; i <= toIndex; i++) {
	    sumValue += array[i];
	  }
	  return sumValue / (toIndex - fromIndex + 1);
	}

	/**
	 * This function calculates the median absolute deviation (MAD)
	 * https://en.wikipedia.org/wiki/Median_absolute_deviation
	 * @param array
	 */
	function xMedianAbsoluteDeviation(array) {
	  const median = xMedian(array);
	  const averageDeviations = new Float64Array(array.length);
	  for (let i = 0; i < array.length; i++) {
	    averageDeviations[i] = Math.abs(array[i] - median);
	  }
	  return {
	    median,
	    mad: xMedian(averageDeviations)
	  };
	}

	/**
	 * Return min and max values of an array
	 *
	 * @param array - array of number
	 * @returns - Object with 2 properties, min and max
	 */
	function xMinMaxValues(array) {
	  xCheck(array);
	  let min = array[0];
	  let max = array[0];
	  for (const value of array) {
	    if (value < min) min = value;
	    if (value > max) max = value;
	  }
	  return {
	    min,
	    max
	  };
	}

	/**
	 * Determine noise level using MAD https://en.wikipedia.org/wiki/Median_absolute_deviation
	 * Constant to convert mad to sd calculated using https://www.wolframalpha.com/input?i=sqrt%282%29+inverse+erf%280.5%29
	 * This assumes a gaussian distribution of the noise
	 * @param array
	 * @returns noise level corresponding to one standard deviation
	 */
	function xNoiseStandardDeviation(array) {
	  const {
	    mad,
	    median
	  } = xMedianAbsoluteDeviation(array);
	  return {
	    sd: mad / 0.6744897501960817,
	    mad,
	    median
	  };
	}

	/**
	 * Calculate the sum of the values
	 *
	 * @param array - Object that contains property x (an ordered increasing array) and y (an array).
	 * @param options - Options.
	 * @returns XSum value on the specified range.
	 */
	function xSum(array, options = {}) {
	  xCheck(array);
	  const {
	    fromIndex,
	    toIndex
	  } = xGetFromToIndex(array, options);
	  let sumValue = array[fromIndex];
	  for (let i = fromIndex + 1; i <= toIndex; i++) {
	    sumValue += array[i];
	  }
	  return sumValue;
	}

	/**
	 * Divides the data with either the sum, the absolute sum or the maximum of the data
	 * @param array - Array containing values
	 * @param options - options
	 * @returns - normalized data
	 */
	function xNormed(input, options = {}) {
	  const {
	    algorithm = 'absolute',
	    value = 1
	  } = options;
	  xCheck(input);
	  const output = getOutputArray(options.output, input.length);
	  if (input.length === 0) {
	    throw new Error('input must not be empty');
	  }
	  switch (algorithm.toLowerCase()) {
	    case 'absolute':
	      {
	        const absoluteSumValue = absoluteSum(input) / value;
	        if (absoluteSumValue === 0) {
	          throw new Error('xNormed: trying to divide by 0');
	        }
	        for (let i = 0; i < input.length; i++) {
	          output[i] = input[i] / absoluteSumValue;
	        }
	        return output;
	      }
	    case 'max':
	      {
	        const currentMaxValue = xMaxValue(input);
	        if (currentMaxValue === 0) {
	          throw new Error('xNormed: trying to divide by 0');
	        }
	        const factor = value / currentMaxValue;
	        for (let i = 0; i < input.length; i++) {
	          output[i] = input[i] * factor;
	        }
	        return output;
	      }
	    case 'sum':
	      {
	        const sumFactor = xSum(input) / value;
	        if (sumFactor === 0) {
	          throw new Error('xNormed: trying to divide by 0');
	        }
	        for (let i = 0; i < input.length; i++) {
	          output[i] = input[i] / sumFactor;
	        }
	        return output;
	      }
	    default:
	      throw new Error(`norm: unknown algorithm: ${algorithm}`);
	  }
	}
	function absoluteSum(input) {
	  let sumValue = 0;
	  for (let i = 0; i < input.length; i++) {
	    sumValue += Math.abs(input[i]);
	  }
	  return sumValue;
	}

	/**
	 * This function pads an array
	 *s
	 * @param array - the array that will be padded
	 * @param options - options
	 */
	function xPadding(array, options = {}) {
	  const {
	    size = 0,
	    value = 0,
	    algorithm = ''
	  } = options;
	  xCheck(array);
	  if (!algorithm) {
	    if (array instanceof Float64Array) {
	      return array.slice();
	    } else {
	      return Float64Array.from(array);
	    }
	  }
	  const result = new Float64Array(array.length + size * 2);
	  for (let i = 0; i < array.length; i++) {
	    result[i + size] = array[i];
	  }
	  const fromEnd = size + array.length;
	  const toEnd = 2 * size + array.length;
	  switch (algorithm.toLowerCase()) {
	    case 'value':
	      for (let i = 0; i < size; i++) {
	        result[i] = value;
	      }
	      for (let i = fromEnd; i < toEnd; i++) {
	        result[i] = value;
	      }
	      break;
	    case 'duplicate':
	      for (let i = 0; i < size; i++) {
	        result[i] = array[0];
	      }
	      for (let i = fromEnd; i < toEnd; i++) {
	        result[i] = array[array.length - 1];
	      }
	      break;
	    case 'circular':
	      for (let i = 0; i < size; i++) {
	        result[i] = array[(array.length - size % array.length + i) % array.length];
	      }
	      for (let i = 0; i < size; i++) {
	        result[i + fromEnd] = array[i % array.length];
	      }
	      break;
	    default:
	      throw new Error('xPadding: unknown algorithm');
	  }
	  return result;
	}

	/** Finds the variance of the data
	 *
	 * @param values - the values of the array
	 * @param options - options
	 * @returns variance
	 */
	function xVariance(values, options = {}) {
	  if (!isAnyArray$1(values)) {
	    throw new TypeError('input must be an array');
	  }
	  const {
	    unbiased = true,
	    mean = xMean(values)
	  } = options;
	  let sqrError = 0;
	  for (let i = 0; i < values.length; i++) {
	    const x = values[i] - mean;
	    sqrError += x * x;
	  }
	  if (unbiased) {
	    return sqrError / (values.length - 1);
	  } else {
	    return sqrError / values.length;
	  }
	}

	/** Finds the standard deviation for the data at hand
	 *
	 * @param values - values in the data
	 * @param options - options
	 * @returns standard deviation
	 */
	function xStandardDeviation(values, options = {}) {
	  return Math.sqrt(xVariance(values, options));
	}

	/**
	 * Pareto scaling, which uses the square root of standard deviation as the scaling factor, circumvents the amplification of noise by retaining a small portion of magnitude information.
	 * Noda, I. (2008). Scaling techniques to enhance two-dimensional correlation spectra. Journal of Molecular Structure, 883, 216-227.
	 * DOI: 10.1016/j.molstruc.2007.12.026
	 *
	 * @param array - array of number
	 */
	function xParetoNormalization(array) {
	  xCheck(array);
	  const result = [];
	  const sqrtSD = Math.sqrt(xStandardDeviation(array));
	  for (const item of array) {
	    result.push(item / sqrtSD);
	  }
	  return result;
	}

	/** Function used to rescale data
	 *
	 * @param input - input for the rescale
	 * @param options - options
	 * @returns rescaled data
	 */
	function xRescale(input, options = {}) {
	  xCheck(input);
	  const output = getOutputArray(options.output, input.length);
	  const currentMin = xMinValue(input);
	  const currentMax = xMaxValue(input);
	  if (currentMin === currentMax) {
	    throw new RangeError('minimum and maximum input values are equal. Cannot rescale a constant array');
	  }
	  const {
	    min = 0,
	    max = 1
	  } = options;
	  if (min >= max) {
	    throw new RangeError('min option must be smaller than max option');
	  }
	  const factor = (max - min) / (currentMax - currentMin);
	  for (let i = 0; i < input.length; i++) {
	    output[i] = (input[i] - currentMin) * factor + min;
	  }
	  return output;
	}

	/**
	 * This function calculates a rolling average
	 *
	 * @param array - array
	 * @param fct - callback function that from an array returns a value
	 * @param options - options
	 */
	function xRolling(array, fct, options = {}) {
	  xCheck(array);
	  if (typeof fct !== 'function') throw new Error('fct has to be a function');
	  const {
	    window = 5,
	    padding = {}
	  } = options;
	  const {
	    size = window - 1,
	    algorithm,
	    value
	  } = padding;
	  array = xPadding(array, {
	    size,
	    algorithm,
	    value
	  }); // ensure we get a copy and it is float64
	  const newArray = [];
	  for (let i = 0; i < array.length - window + 1; i++) {
	    // we will send a view to the original buffer
	    newArray.push(fct(array.subarray(i, i + window)));
	  }
	  return newArray;
	}

	/**
	 * This function calculates a rolling average
	 *
	 * @param array - array
	 * @param options - option
	 */
	function xRollingAverage(array, options = {}) {
	  return xRolling(array, xMean, options);
	}

	/**
	 * This function calculates a rolling average
	 *
	 * @param array - array
	 * @param options - options
	 */
	function xRollingMedian(array, options = {}) {
	  return xRolling(array, xMedian, options);
	}

	/**
	 * Throw an error in no an object of x,y arrays
	 *
	 * @param data - array of points {x,y,z}
	 */
	function xyCheck(data, options = {}) {
	  const {
	    minLength
	  } = options;
	  if (typeof data !== 'object' || !isAnyArray$1(data.x) || !isAnyArray$1(data.y)) {
	    throw new Error('Data must be an object of x and y arrays');
	  }
	  if (data.x.length !== data.y.length) {
	    throw new Error('The x and y arrays must have the same length');
	  }
	  if (minLength && data.x.length < minLength) {
	    throw new Error(`data.x must have a length of at least ${minLength}`);
	  }
	}

	/**
	 * Filters x,y values to allow strictly growing values in x axis.
	 *
	 * @param data - Object that contains property x (an ordered increasing array) and y (an array).
	 */
	function xyEnsureGrowingX(data) {
	  xyCheck(data);
	  if (xIsMonotonic(data.x) === 1) return data;
	  const x = Array.from(data.x);
	  const y = Array.from(data.y);
	  let prevX = Number.NEGATIVE_INFINITY;
	  let currentIndex = 0;
	  for (let index = 0; index < x.length; index++) {
	    if (prevX < x[index]) {
	      if (currentIndex < index) {
	        x[currentIndex] = x[index];
	        y[currentIndex] = y[index];
	      }
	      currentIndex++;
	      prevX = x[index];
	    }
	  }
	  x.length = currentIndex;
	  y.length = currentIndex;
	  return {
	    x,
	    y
	  };
	}

	/**
	 * Normalize an array of zones:
	 * - ensure than from < to
	 * - merge overlapping zones
	 * - deal with exclusions zones
	 * - if no zones is specified add one between -Infinity and +Infinity
	 * @param zones - array of zones
	 * @param options - options
	 * @returns array of zones
	 */
	function zonesNormalize(zones = [], options = {}) {
	  const {
	    exclusions = []
	  } = options;
	  let {
	    from = Number.NEGATIVE_INFINITY,
	    to = Number.POSITIVE_INFINITY
	  } = options;
	  if (from > to) [from, to] = [to, from];
	  zones = JSON.parse(JSON.stringify(zones)).map(zone => zone.from > zone.to ? {
	    from: zone.to,
	    to: zone.from
	  } : zone);
	  zones = zones.sort((a, b) => {
	    if (a.from !== b.from) return a.from - b.from;
	    return a.to - b.to;
	  });
	  if (zones.length === 0) {
	    zones.push({
	      from,
	      to
	    });
	  }
	  for (const zone of zones) {
	    if (from > zone.from) zone.from = from;
	    if (to < zone.to) zone.to = to;
	  }
	  zones = zones.filter(zone => zone.from <= zone.to);
	  if (zones.length === 0) return [];
	  let currentZone = zones[0];
	  const beforeExclusionsZones = [currentZone];
	  for (let i = 1; i < zones.length; i++) {
	    const zone = zones[i];
	    if (zone.from <= currentZone.to) {
	      if (currentZone.to < zone.to) {
	        currentZone.to = zone.to;
	      }
	    } else {
	      currentZone = zone;
	      beforeExclusionsZones.push(currentZone);
	    }
	  }
	  if (exclusions.length === 0) return beforeExclusionsZones;
	  const normalizedExclusions = zonesNormalize(exclusions);
	  let currentExclusionIndex = 0;
	  const results = [];
	  let counter = 0;
	  for (let zoneIndex = 0; zoneIndex < beforeExclusionsZones.length; zoneIndex++) {
	    if (counter++ > 5) break;
	    const zone = beforeExclusionsZones[zoneIndex];
	    if (currentExclusionIndex === normalizedExclusions.length) {
	      // we analysed all the exclusion zones
	      results.push(zone);
	      continue;
	    }
	    while (currentExclusionIndex < normalizedExclusions.length && normalizedExclusions[currentExclusionIndex].to <= zone.from) {
	      currentExclusionIndex++;
	    }
	    if (currentExclusionIndex === normalizedExclusions.length) {
	      // we analysed all the exclusion zones
	      results.push(zone);
	      continue;
	    }
	    if (zone.to < normalizedExclusions[currentExclusionIndex].from) {
	      // no problems, not yet in exclusion
	      results.push(zone);
	      continue;
	    }
	    if (normalizedExclusions[currentExclusionIndex].to >= zone.to) {
	      // could be totally excluded
	      if (normalizedExclusions[currentExclusionIndex].from <= zone.from) {
	        continue;
	      }
	      results.push({
	        from: normalizedExclusions[currentExclusionIndex].to,
	        to: zone.to
	      });
	    }
	    // we cut in the middle, we need to create more zones, annoying !
	    if (normalizedExclusions[currentExclusionIndex].from > zone.from) {
	      results.push({
	        from: zone.from,
	        to: normalizedExclusions[currentExclusionIndex].from
	      });
	    }
	    zone.from = normalizedExclusions[currentExclusionIndex].to;
	    zoneIndex--;
	  }
	  return results;
	}

	/**
	 * Add the number of points per zone to reach a specified total
	 *
	 * @param zones - array of zones
	 * @param numberOfPoints - total number of points to distribute between zones
	 * @param options - options
	 * @returns array of zones with points
	 */
	function zonesWithPoints(zones = [],
	/**
	 * total number of points to distribute between zones
	 * @default 10
	 */
	numberOfPoints = 10, options = {}) {
	  if (zones.length === 0) return zones;
	  const returnZones = zonesNormalize(zones, options);
	  const totalSize = returnZones.reduce((previous, current) => {
	    return previous + (current.to - current.from);
	  }, 0);
	  const unitsPerPoint = totalSize / numberOfPoints;
	  let currentTotal = 0;
	  for (let i = 0; i < returnZones.length - 1; i++) {
	    const tempZone = returnZones[i];
	    tempZone.numberOfPoints = Math.min(Math.round((tempZone.to - tempZone.from) / unitsPerPoint), numberOfPoints - currentTotal);
	    currentTotal += tempZone.numberOfPoints;
	  }
	  const zone = returnZones[returnZones.length - 1];
	  zone.numberOfPoints = numberOfPoints - currentTotal;
	  return returnZones;
	}

	/**
	 * function that retrieves the getEquallySpacedData with the variant "slot"
	 *
	 * @param x
	 * @param y
	 * @param from
	 * @param to
	 * @param numberOfPoints
	 * @return Array of y's equally spaced with the variant "slot"
	 */
	function equallySpacedSlot( /** x coordinates */
	x, /** y coordinates */
	y, /** from value */
	from, /** to value */
	to, /** number of points */
	numberOfPoints) {
	  const xLength = x.length;
	  const step = (to - from) / (numberOfPoints > 1 ? numberOfPoints - 1 : 1);
	  const halfStep = step / 2;
	  const lastStep = x[x.length - 1] - x[x.length - 2];
	  const start = from - halfStep;
	  // Changed Array to Float64Array
	  const output = new Float64Array(numberOfPoints);
	  // Init main variables
	  let min = start;
	  let max = start + step;
	  let previousX = -Number.MAX_VALUE;
	  let previousY = 0;
	  let nextX = x[0];
	  let nextY = y[0];
	  let frontOutsideSpectra = 0;
	  let backOutsideSpectra = true;
	  let currentValue = 0;
	  // for slot algorithm
	  let currentPoints = 0;
	  let i = 1; // index of input
	  let j = 0; // index of output
	  main: while (true) {
	    if (previousX >= nextX) throw new Error('x must be a growing series');
	    while (previousX - max > 0) {
	      // no overlap with original point, just consume current value
	      if (backOutsideSpectra) {
	        currentPoints++;
	        backOutsideSpectra = false;
	      }
	      output[j] = currentPoints <= 0 ? 0 : currentValue / currentPoints;
	      j++;
	      if (j === numberOfPoints) {
	        break main;
	      }
	      min = max;
	      max += step;
	      currentValue = 0;
	      currentPoints = 0;
	    }
	    if (previousX > min) {
	      currentValue += previousY;
	      currentPoints++;
	    }
	    if (previousX === -Number.MAX_VALUE || frontOutsideSpectra > 1) {
	      currentPoints--;
	    }
	    previousX = nextX;
	    previousY = nextY;
	    if (i < xLength) {
	      nextX = x[i];
	      nextY = y[i];
	      i++;
	    } else {
	      nextX += lastStep;
	      nextY = 0;
	      frontOutsideSpectra++;
	    }
	  }
	  return output;
	}

	/**
	 * Function that calculates the integral of the line between two
	 * x-coordinates, given the slope and intercept of the line.
	 * @param x0
	 * @param x1
	 * @param slope
	 * @param intercept
	 * @return integral value.
	 */
	function integral( /** first coordinate of point */
	x0, /** second coordinate of point */
	x1, /** slope of the line */
	slope, /** intercept of the line on the y axis */
	intercept) {
	  return 0.5 * slope * x1 * x1 + intercept * x1 - (0.5 * slope * x0 * x0 + intercept * x0);
	}

	/**
	 * function that retrieves the getEquallySpacedData with the variant "smooth"
	 *
	 * @param x
	 * @param y
	 * @param from
	 * @param to
	 * @param numberOfPoints
	 * @return - Array of y's equally spaced with the variant "smooth"
	 */
	function equallySpacedSmooth( /** x coordinates */
	x, /** y coordinates */
	y, /** from value */
	from, /** to value */
	to, /** number of points */
	numberOfPoints) {
	  const xLength = x.length;
	  const step = (to - from) / (numberOfPoints > 1 ? numberOfPoints - 1 : 1);
	  const halfStep = step / 2;
	  // Changed Array to Float64Array
	  const output = new Float64Array(numberOfPoints);
	  const initialOriginalStep = x[1] - x[0];
	  const lastOriginalStep = x[xLength - 1] - x[xLength - 2];
	  // Init main variables
	  let min = from - halfStep;
	  let max = from + halfStep;
	  let previousX = Number.MIN_SAFE_INTEGER;
	  let previousY = 0;
	  let nextX = x[0] - initialOriginalStep;
	  let nextY = 0;
	  let currentValue = 0;
	  let slope = 0;
	  let intercept = 0;
	  let sumAtMin = 0;
	  let sumAtMax = 0;
	  let i = 0; // index of input
	  let j = 0; // index of output
	  let add = 0;
	  main: while (true) {
	    if (previousX >= nextX) throw new Error('x must be a growing series');
	    if (previousX <= min && min <= nextX) {
	      add = integral(0, min - previousX, slope, previousY);
	      sumAtMin = currentValue + add;
	    }
	    while (nextX - max >= 0) {
	      // no overlap with original point, just consume current value
	      add = integral(0, max - previousX, slope, previousY);
	      sumAtMax = currentValue + add;
	      output[j++] = (sumAtMax - sumAtMin) / step;
	      if (j === numberOfPoints) {
	        break main;
	      }
	      min = max;
	      max += step;
	      sumAtMin = sumAtMax;
	    }
	    currentValue += integral(previousX, nextX, slope, intercept);
	    previousX = nextX;
	    previousY = nextY;
	    if (i < xLength) {
	      nextX = x[i];
	      nextY = y[i];
	      i++;
	    } else if (i === xLength) {
	      nextX += lastOriginalStep;
	      nextY = 0;
	    }
	    slope = getSlope(previousX, previousY, nextX, nextY);
	    intercept = -slope * previousX + previousY;
	  }
	  return output;
	}
	function getSlope(x0, y0, x1, y1) {
	  return (y1 - y0) / (x1 - x0);
	}

	/**
	 * Function that returns a Number array of equally spaced numberOfPoints
	 * containing a representation of intensities of the spectra arguments x
	 * and y.
	 *
	 * The options parameter contains an object in the following form:
	 * from: starting point
	 * to: last point
	 * numberOfPoints: number of points between from and to
	 * variant: "slot" or "smooth" - smooth is the default option
	 *
	 * The slot variant consist that each point in an array is calculated
	 * averaging the existing points between the slot that belongs to the current
	 * value. The smooth variant is the same but takes the integral of the range
	 * of the slot and divide by the step size between two points in an array.
	 *
	 * If exclusions zone are present, zones are ignored !
	 *
	 * @param data - object containing 2 properties x and y
	 * @param options - options
	 * @return new object with x / y array with the equally spaced data.
	 */
	function xyEquallySpaced(data, options = {}) {
	  const {
	    x,
	    y
	  } = data;
	  const xLength = x.length;
	  const {
	    from = x[0],
	    to = x[xLength - 1],
	    variant = 'smooth',
	    numberOfPoints = 100,
	    exclusions = [],
	    zones = [{
	      from,
	      to
	    }]
	  } = options;
	  if (from > to) {
	    throw new RangeError('from should be larger than to');
	  }
	  xyCheck(data);
	  if (numberOfPoints < 2) {
	    throw new RangeError("'numberOfPoints' option must be greater than 1");
	  }
	  const normalizedZones = zonesNormalize(zones, {
	    from,
	    to,
	    exclusions
	  });
	  const zonesWithPointsRes = zonesWithPoints(normalizedZones, numberOfPoints, {
	    from,
	    to
	  });
	  let xResult = [];
	  let yResult = [];
	  for (const zone of zonesWithPointsRes) {
	    if (!zone.numberOfPoints) {
	      zone.numberOfPoints = 0;
	    }
	    const zoneResult = processZone(Array.from(x), Array.from(y), zone.from, zone.to, zone.numberOfPoints, variant);
	    xResult = xResult.concat(zoneResult.x);
	    yResult = yResult.concat(zoneResult.y);
	  }
	  return {
	    x: xResult,
	    y: yResult
	  };
	}
	function processZone(x, y, from, to, numberOfPoints, variant) {
	  if (numberOfPoints < 1) {
	    throw new RangeError('the number of points must be at least 1');
	  }
	  const output = variant === 'slot' ? Array.from(equallySpacedSlot(x, y, from, to, numberOfPoints)) : Array.from(equallySpacedSmooth(x, y, from, to, numberOfPoints));
	  return {
	    x: Array.from(createFromToArray({
	      from,
	      to,
	      length: numberOfPoints
	    })),
	    y: output
	  };
	}

	/** Filter an array x/y based on various criteria x points are expected to be sorted
	 *
	 * @param data - object containing 2 properties x and y
	 * @param options - options
	 * @return filtered array
	 */
	function xyFilterX(data, options = {}) {
	  const {
	    x,
	    y
	  } = data;
	  const {
	    from = x[0],
	    to = x[x.length - 1],
	    zones = [{
	      from,
	      to
	    }],
	    exclusions = []
	  } = options;
	  const normalizedZones = zonesNormalize(zones, {
	    from,
	    to,
	    exclusions
	  });
	  let currentZoneIndex = 0;
	  const newX = [];
	  const newY = [];
	  let position = 0;
	  while (position < x.length) {
	    if (x[position] <= normalizedZones[currentZoneIndex].to && x[position] >= normalizedZones[currentZoneIndex].from) {
	      newX.push(x[position]);
	      newY.push(y[position]);
	    } else if (x[position] > normalizedZones[currentZoneIndex].to) {
	      currentZoneIndex++;
	      if (!normalizedZones[currentZoneIndex]) break;
	    }
	    position++;
	  }
	  return {
	    x: newX,
	    y: newY
	  };
	}

	/**
	 * Filter out all the points for which x <= 0. Useful to display log scale data
	 *
	 * @param data - data
	 * @returns - An object with the filtered data
	 */
	function xyFilterXPositive(data) {
	  xyCheck(data);
	  const {
	    x,
	    y
	  } = data;
	  const newX = [];
	  const newY = [];
	  if (typeof x === 'undefined' || typeof y === 'undefined') {
	    return {
	      x: newX,
	      y: newY
	    };
	  }
	  for (let i = 0; i < x.length; i++) {
	    if (x[i] > 0) {
	      newX.push(x[i]);
	      newY.push(y[i]);
	    }
	  }
	  return {
	    x: newX,
	    y: newY
	  };
	}

	function matrixCheck(data) {
	  if (data.length === 0 || data[0].length === 0) {
	    throw new RangeError('matrix should contain data');
	  }
	  const firstLength = data[0].length;
	  for (let i = 1; i < data.length; i++) {
	    if (data[i].length !== firstLength) {
	      throw new RangeError('All rows should has the same length');
	    }
	  }
	}

	/**
	 * Get min and max Z
	 *
	 * @param matrix - matrix [rows][cols].
	 */
	function matrixMinMaxZ(matrix) {
	  matrixCheck(matrix);
	  const nbRows = matrix.length;
	  const nbColumns = matrix[0].length;
	  let min = matrix[0][0];
	  let max = matrix[0][0];
	  for (let column = 0; column < nbColumns; column++) {
	    for (let row = 0; row < nbRows; row++) {
	      if (matrix[row][column] < min) min = matrix[row][column];
	      if (matrix[row][column] > max) max = matrix[row][column];
	    }
	  }
	  return {
	    min,
	    max
	  };
	}

	function addStyle(serie, spectrum, options = {}) {
	  let {
	    color = '#A9A9A9'
	  } = options;
	  const {
	    opacity = 1,
	    lineWidth = 1
	  } = options;
	  if (color.match(/#[0-9A-F]{6}$/i)) {
	    color = (color + (opacity * 255 >> 0).toString(16)).toUpperCase();
	  } else {
	    color = color.replace(/rgb ?\((.*)\)/, `rgba($1,${opacity})`);
	  }
	  serie.style = [{
	    name: 'unselected',
	    style: {
	      line: {
	        color,
	        width: lineWidth,
	        dash: 1
	      }
	    }
	  }, {
	    name: 'selected',
	    style: {
	      line: {
	        color,
	        width: lineWidth + 2,
	        dash: 1
	      }
	    }
	  }];
	  serie.name = spectrum.label || spectrum.id;
	}

	const COLORS = ['#FFB300', '#803E75', '#FF6800', '#A6BDD7', '#C10020', '#CEA262', '#817066', '#007D34', '#F6768E', '#00538A', '#FF7A5C', '#53377A', '#FF8E00', '#B32851', '#F4C800', '#7F180D', '#93AA00', '#593315', '#F13A13', '#232C16'];

	/**
	 * Generate a jsgraph chart format from an array of Analysis
	 */
	function getJSGraph(analyses, options = {}) {
	  const {
	    colors = COLORS,
	    opacities = [1],
	    linesWidth = [1],
	    selector = {},
	    normalization,
	    xAxis = {},
	    yAxis = {}
	  } = options;
	  const series = [];
	  let xLabel = xAxis.label;
	  let yLabel = yAxis.label;
	  let xUnits = xAxis.units;
	  let yUnits = yAxis.units;
	  for (let i = 0; i < analyses.length; i++) {
	    const analysis = analyses[i];
	    const spectra = analysis.getNormalizedSpectra({
	      selector,
	      normalization
	    });
	    if (spectra.length === 0) continue;
	    const firstSpectrum = spectra[0];
	    // todo: if many spectra are available and not xUnits / yUnits are specified we should ensure that all the3 spectra are compatible
	    if (!xLabel) xLabel = firstSpectrum.variables.x.label;
	    if (!yLabel) yLabel = firstSpectrum.variables.y.label;
	    if (!xUnits) xUnits = firstSpectrum.variables.x.units;
	    if (!yUnits) yUnits = firstSpectrum.variables.y.units;
	    for (const spectrum of spectra) {
	      const serie = {};
	      addStyle(serie, analysis, {
	        color: colors[i % colors.length],
	        opacity: opacities[i % opacities.length],
	        lineWidth: linesWidth[i % linesWidth.length]
	      });
	      serie.data = {
	        x: spectrum.variables.x.data,
	        y: spectrum.variables.y.data
	      };
	      serie.id = spectrum.id;
	      if (xAxis.logScale) {
	        serie.data = xyFilterXPositive(serie.data);
	      }
	      series.push(serie);
	    }
	  }
	  return {
	    axes: {
	      x: {
	        label: xLabel,
	        unit: xUnits,
	        unitWrapperBefore: '(',
	        unitWrapperAfter: ')',
	        flipped: false,
	        display: true,
	        ...xAxis
	      },
	      y: {
	        label: yLabel,
	        unit: yUnits,
	        unitWrapperBefore: '(',
	        unitWrapperAfter: ')',
	        flipped: false,
	        display: true,
	        ...yAxis
	      }
	    },
	    series
	  };
	}

	function getNormalizationAnnotations(filter = {}, boundary = {
	  y: {
	    min: '0px',
	    max: '2000px'
	  }
	}) {
	  let {
	    exclusions = []
	  } = filter;
	  let annotations = [];
	  exclusions = exclusions.filter(exclusion => !exclusion.ignore);
	  annotations = exclusions.map(exclusion => {
	    const annotation = {
	      type: 'rect',
	      position: [{
	        x: exclusion.from,
	        y: boundary.y.min
	      }, {
	        x: exclusion.to,
	        y: boundary.y.max
	      }],
	      strokeWidth: 0,
	      fillColor: 'rgba(255,255,224,1)'
	    };
	    return annotation;
	  });
	  if (filter.from !== undefined) {
	    annotations.push({
	      type: 'rect',
	      position: [{
	        x: Number.MIN_SAFE_INTEGER,
	        y: boundary.y.min
	      }, {
	        x: filter.from,
	        y: boundary.y.max
	      }],
	      strokeWidth: 0,
	      fillColor: 'rgba(255,255,224,1)'
	    });
	  }
	  if (filter.to !== undefined) {
	    annotations.push({
	      type: 'rect',
	      position: [{
	        x: filter.to,
	        y: boundary.y.min
	      }, {
	        x: Number.MAX_SAFE_INTEGER,
	        y: boundary.y.max
	      }],
	      strokeWidth: 0,
	      fillColor: 'rgba(255,255,224,1)'
	    });
	  }
	  return annotations;
	}

	function appendDistinctParameter(values, key, value) {
	  if (!values[key]) {
	    values[key] = {
	      key,
	      values: [],
	      count: 0
	    };
	  }
	  if (!values[key].values.includes(value)) {
	    values[key].values.push(value);
	  }
	  values[key].count++;
	}

	function appendDistinctValue(values, key) {
	  if (!values[key]) {
	    values[key] = {
	      key,
	      count: 0
	    };
	  }
	  values[key].count++;
	}

	class AnalysesManager {
	  analyses;
	  constructor() {
	    this.analyses = [];
	  }
	  addAnalysis(analysis) {
	    const index = this.getAnalysisIndex(analysis.id);
	    if (index === undefined) {
	      this.analyses.push(analysis);
	    } else {
	      this.analyses[index] = analysis;
	    }
	  }
	  getAnalyses(options = {}) {
	    const {
	      ids
	    } = options;
	    const analyses = [];
	    for (const analysis of this.analyses) {
	      if (!ids || ids.includes(analysis.id)) {
	        analyses.push(analysis);
	      }
	    }
	    return analyses;
	  }
	  getSpectra() {
	    const spectra = [];
	    for (const analysis of this.analyses) {
	      spectra.push(...analysis.spectra);
	    }
	    return spectra;
	  }
	  getAnalysisBySpectrumId(id) {
	    for (const analysis of this.analyses) {
	      for (const spectrum of analysis.spectra) {
	        if (spectrum.id === id) return analysis;
	      }
	    }
	    return undefined;
	  }
	  getSpectrumById(id) {
	    for (const analysis of this.analyses) {
	      for (const spectrum of analysis.spectra) {
	        if (spectrum.id === id) return spectrum;
	      }
	    }
	    return undefined;
	  }
	  /**
	   * Get an array of objects (key + count) of all the titles
	   */
	  getDistinctTitles() {
	    const values = {};
	    for (const spectrum of this.getSpectra()) {
	      if (spectrum.title) {
	        appendDistinctValue(values, spectrum.title);
	      }
	    }
	    return Object.keys(values).map(key => values[key]);
	  }
	  /**
	   * Get an array of objects (key + count) of all the units
	   */
	  getDistinctUnits() {
	    const values = {};
	    for (const spectrum of this.getSpectra()) {
	      if (spectrum.variables) {
	        for (const [, variable] of Object.entries(spectrum.variables)) {
	          if (variable.units) {
	            appendDistinctValue(values, variable.units);
	          }
	        }
	      }
	    }
	    return Object.keys(values).map(key => values[key]);
	  }
	  /**
	   * Get an array of objects (key + unit + label + count) of all the units
	   */
	  getDistinctLabelUnits() {
	    const values = {};
	    for (const spectrum of this.getSpectra()) {
	      if (spectrum.variables) {
	        for (const [, variable] of Object.entries(spectrum.variables)) {
	          const {
	            label,
	            units
	          } = normalizeLabelUnits(variable.label, variable.units);
	          const key = label + (units ? ` (${units})` : '');
	          if (key) {
	            if (!values[key]) {
	              values[key] = {
	                key,
	                units,
	                label,
	                count: 0
	              };
	            }
	            values[key].count++;
	          }
	        }
	      }
	    }
	    return Object.keys(values).map(key => values[key]);
	  }
	  /**
	   * Get an array of objects (key + count) of all the labels
	   */
	  getDistinctLabels() {
	    const values = {};
	    for (const spectrum of this.getSpectra()) {
	      if (spectrum.variables) {
	        for (const [, variable] of Object.entries(spectrum.variables)) {
	          appendDistinctValue(values, variable.label.replace(/\s+[[(].*$/, ''));
	        }
	      }
	    }
	    return Object.keys(values).map(key => values[key]);
	  }
	  /**
	   * Get an array of objects (key + count) of all the dataTypes
	   */
	  getDistinctDataTypes() {
	    const values = {};
	    for (const spectrum of this.getSpectra()) {
	      if (spectrum.dataType) {
	        appendDistinctValue(values, spectrum.dataType);
	      }
	    }
	    return Object.keys(values).map(key => values[key]);
	  }
	  /**
	   * Get an array of objects (key + count) of all the meta
	   */
	  getDistinctMeta() {
	    const values = {};
	    for (const spectrum of this.getSpectra()) {
	      if (spectrum.meta) {
	        for (const key in spectrum.meta) {
	          appendDistinctParameter(values, key, spectrum.meta[key]);
	        }
	      }
	    }
	    return Object.keys(values).map(key => values[key]);
	  }
	  removeAllAnalyses() {
	    this.analyses.splice(0);
	  }
	  /**
	   * Remove the analysis from the AnalysesManager for the specified id
	   */
	  removeAnalysis(id) {
	    const index = this.getAnalysisIndex(id);
	    if (index === undefined) return undefined;
	    return this.analyses.splice(index, 1);
	  }
	  /**
	   * Returns the index of the analysis in the analyses array
	   */
	  getAnalysisIndex(id) {
	    if (!id) return undefined;
	    for (let i = 0; i < this.analyses.length; i++) {
	      const analysis = this.analyses[i];
	      if (analysis.id === id) return i;
	    }
	    return undefined;
	  }
	  /**
	   * Checks if the ID of an analysis exists in the AnalysesManager
	   */
	  includes(id) {
	    const index = this.getAnalysisIndex(id);
	    return index === undefined ? false : !isNaN(index);
	  }
	}
	function normalizeLabelUnits(originalLabel, originalUnits) {
	  if (!originalLabel) {
	    return {
	      units: '',
	      label: ''
	    };
	  }
	  if (originalLabel.search(/[[(]]/) >= 0) {
	    const [units, label] = originalLabel.split(/\s*[[(]/);
	    return {
	      units: originalUnits || units,
	      label
	    };
	  }
	  return {
	    label: originalLabel,
	    units: originalUnits
	  };
	}

	/**
	 * Center the mean
	 * @param data
	 */
	function centerMean(data) {
	  const {
	    y
	  } = data;
	  const mean = xMean(y);
	  for (let i = 0; i < y.length; i++) {
	    y[i] -= mean;
	  }
	  return {
	    data
	  };
	}

	/**
	 * Center the median
	 * @param data
	 */
	function centerMedian(data) {
	  const {
	    y
	  } = data;
	  const median = xMedian(y);
	  for (let i = 0; i < y.length; i++) {
	    y[i] -= median;
	  }
	  return {
	    data
	  };
	}

	/**
	 * Filter that allows to
	 * @param data
	 * @param options
	 */
	function fromTo(data, options = {}) {
	  const {
	    fromIndex,
	    toIndex
	  } = xGetFromToIndex(data.x, options);
	  return {
	    data: {
	      x: data.x.subarray(fromIndex, toIndex + 1),
	      y: data.y.subarray(fromIndex, toIndex + 1)
	    }
	  };
	}

	/**
	 * Norm the Y values
	 * @param data
	 */
	function normed(data, options = {}) {
	  xNormed(data.y, {
	    ...options,
	    output: data.y
	  });
	  return {
	    data
	  };
	}

	/**
	 * Center the mean
	 * @param data
	 */
	function divideBySD(data) {
	  const {
	    y
	  } = data;
	  const sd = xStandardDeviation(y);
	  for (let i = 0; i < y.length; i++) {
	    y[i] /= sd;
	  }
	  return {
	    data
	  };
	}

	/**
	 * Center the mean
	 * @param data
	 */
	function rescale(data, options = {}) {
	  xRescale(data.y, {
	    ...options,
	    output: data.y
	  });
	  return {
	    data
	  };
	}

	/**
	 * Filter that allows to
	 * @param data
	 */
	function paretoNormalization(data) {
	  return {
	    data: {
	      x: data.x,
	      y: xParetoNormalization(data.y)
	    }
	  };
	}

	// Based on https://github.com/scijs/cholesky-solve

	/*
	The MIT License (MIT)

	Copyright (c) 2013 Eric Arnebäck

	Permission is hereby granted, free of charge, to any person obtaining a copy
	of this software and associated documentation files (the "Software"), to deal
	in the Software without restriction, including without limitation the rights
	to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
	copies of the Software, and to permit persons to whom the Software is
	furnished to do so, subject to the following conditions:

	The above copyright notice and this permission notice shall be included in
	all copies or substantial portions of the Software.

	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
	FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
	AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
	LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
	OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
	THE SOFTWARE.
	*/

	function ldlSymbolic(n /* A and L are n-by-n, where n >= 0 */, Ap /* input of size n + 1, not modified */, Ai /* input of size nz=Ap[n], not modified */, Lp /* output of size n + 1, not defined on input */, Parent /* output of size n, not defined on input */, Lnz /* output of size n, not defined on input */, Flag /* workspace of size n, not defn. on input or output */) {
	  let i, k, p, kk, p2;
	  for (k = 0; k < n; k++) {
	    /* L(k,:) pattern: all nodes reachable in etree from nz in A(0:k-1,k) */
	    Parent[k] = -1; /* parent of k is not yet known */
	    Flag[k] = k; /* mark node k as visited */
	    Lnz[k] = 0; /* count of nonzeros in column k of L */
	    kk = k; /* kth original, or permuted, column */
	    p2 = Ap[kk + 1];
	    for (p = Ap[kk]; p < p2; p++) {
	      /* A (i,k) is nonzero (original or permuted A) */
	      i = Ai[p];
	      if (i < k) {
	        /* follow path from i to root of etree, stop at flagged node */
	        for (; Flag[i] !== k; i = Parent[i]) {
	          /* find parent of i if not yet determined */
	          if (Parent[i] === -1) Parent[i] = k;
	          Lnz[i]++; /* L (k,i) is nonzero */
	          Flag[i] = k; /* mark i as visited */
	        }
	      }
	    }
	  }
	  /* construct Lp index array from Lnz column counts */
	  Lp[0] = 0;
	  for (k = 0; k < n; k++) {
	    Lp[k + 1] = Lp[k] + Lnz[k];
	  }
	}
	function ldlNumeric(n /* A and L are n-by-n, where n >= 0 */, Ap /* input of size n+1, not modified */, Ai /* input of size nz=Ap[n], not modified */, Ax /* input of size nz=Ap[n], not modified */, Lp /* input of size n+1, not modified */, Parent /* input of size n, not modified */, Lnz /* output of size n, not defn. on input */, Li /* output of size lnz=Lp[n], not defined on input */, Lx /* output of size lnz=Lp[n], not defined on input */, D /* output of size n, not defined on input */, Y /* workspace of size n, not defn. on input or output */, Pattern /* workspace of size n, not defn. on input or output */, Flag /* workspace of size n, not defn. on input or output */) {
	  let yi, lKi;
	  let i, k, p, kk, p2, len, top;
	  for (k = 0; k < n; k++) {
	    /* compute nonzero Pattern of kth row of L, in topological order */
	    Y[k] = 0.0; /* Y(0:k) is now all zero */
	    top = n; /* stack for pattern is empty */
	    Flag[k] = k; /* mark node k as visited */
	    Lnz[k] = 0; /* count of nonzeros in column k of L */
	    kk = k; /* kth original, or permuted, column */
	    p2 = Ap[kk + 1];
	    for (p = Ap[kk]; p < p2; p++) {
	      i = Ai[p]; /* get A(i,k) */
	      if (i <= k) {
	        Y[i] += Ax[p]; /* scatter A(i,k) into Y (sum duplicates) */
	        for (len = 0; Flag[i] !== k; i = Parent[i]) {
	          Pattern[len++] = i; /* L(k,i) is nonzero */
	          Flag[i] = k; /* mark i as visited */
	        }
	        while (len > 0) Pattern[--top] = Pattern[--len];
	      }
	    }
	    /* compute numerical values kth row of L (a sparse triangular solve) */
	    D[k] = Y[k]; /* get D(k,k) and clear Y(k) */
	    Y[k] = 0.0;
	    for (; top < n; top++) {
	      i = Pattern[top]; /* Pattern[top:n-1] is pattern of L(:,k) */
	      yi = Y[i]; /* get and clear Y(i) */
	      Y[i] = 0.0;
	      p2 = Lp[i] + Lnz[i];
	      for (p = Lp[i]; p < p2; p++) {
	        Y[Li[p]] -= Lx[p] * yi;
	      }
	      lKi = yi / D[i]; /* the nonzero entry L(k,i) */
	      D[k] -= lKi * yi;
	      Li[p] = k; /* store L(k,i) in column form of L */
	      Lx[p] = lKi;
	      Lnz[i]++; /* increment count of nonzeros in col i */
	    }
	    if (D[k] === 0.0) return k; /* failure, D(k,k) is zero */
	  }
	  return n; /* success, diagonal of D is all nonzero */
	}
	function ldlLsolve(n /* L is n-by-n, where n >= 0 */, X /* size n. right-hand-side on input, soln. on output */, Lp /* input of size n+1, not modified */, Li /* input of size lnz=Lp[n], not modified */, Lx /* input of size lnz=Lp[n], not modified */) {
	  let j, p, p2;
	  for (j = 0; j < n; j++) {
	    p2 = Lp[j + 1];
	    for (p = Lp[j]; p < p2; p++) {
	      X[Li[p]] -= Lx[p] * X[j];
	    }
	  }
	}
	function ldlDsolve(n /* D is n-by-n, where n >= 0 */, X /* size n. right-hand-side on input, soln. on output */, D /* input of size n, not modified */) {
	  let j;
	  for (j = 0; j < n; j++) {
	    X[j] /= D[j];
	  }
	}
	function ldlLTsolve(n /* L is n-by-n, where n >= 0 */, X /* size n. right-hand-side on input, soln. on output */, Lp /* input of size n+1, not modified */, Li /* input of size lnz=Lp[n], not modified */, Lx /* input of size lnz=Lp[n], not modified */) {
	  let j, p, p2;
	  for (j = n - 1; j >= 0; j--) {
	    p2 = Lp[j + 1];
	    for (p = Lp[j]; p < p2; p++) {
	      X[j] -= Lx[p] * X[Li[p]];
	    }
	  }
	}
	function ldlPerm(n /* size of X, B, and P */, X /* output of size n. */, B /* input of size n. */, P /* input permutation array of size n. */) {
	  let j;
	  for (j = 0; j < n; j++) {
	    X[j] = B[P[j]];
	  }
	}
	function ldlPermt(n /* size of X, B, and P */, X /* output of size n. */, B /* input of size n. */, P /* input permutation array of size n. */) {
	  let j;
	  for (j = 0; j < n; j++) {
	    X[P[j]] = B[j];
	  }
	}
	function prepare(M, n, P) {
	  // if a permutation was specified, apply it.
	  if (P) {
	    let Pinv = new Array(n);
	    for (let k = 0; k < n; k++) {
	      Pinv[P[k]] = k;
	    }
	    let Mt = []; // scratch memory
	    // Apply permutation. We make M into P*M*P^T
	    for (let a = 0; a < M.length; ++a) {
	      let ar = Pinv[M[a][0]];
	      let ac = Pinv[M[a][1]];

	      // we only store the upper-diagonal elements(since we assume matrix is symmetric, we only need to store these)
	      // if permuted element is below diagonal, we simply transpose it.
	      if (ac < ar) {
	        let t = ac;
	        ac = ar;
	        ar = t;
	      }
	      Mt[a] = [];
	      Mt[a][0] = ar;
	      Mt[a][1] = ac;
	      Mt[a][2] = M[a][2];
	    }
	    M = Mt; // copy scratch memory.
	  } else {
	    // if P argument is null, we just use an identity permutation.
	    P = [];
	    for (let i = 0; i < n; ++i) {
	      P[i] = i;
	    }
	  }

	  // The sparse matrix we are decomposing is A.
	  // Now we shall create A from M.
	  let Ap = new Array(n + 1);
	  let Ai = new Array(M.length);
	  let Ax = new Array(M.length);

	  // count number of non-zero elements in columns.
	  let LNZ = [];
	  for (let i = 0; i < n; ++i) {
	    LNZ[i] = 0;
	  }
	  for (let a = 0; a < M.length; ++a) {
	    LNZ[M[a][1]]++;
	  }
	  Ap[0] = 0;
	  for (let i = 0; i < n; ++i) {
	    Ap[i + 1] = Ap[i] + LNZ[i];
	  }
	  let coloffset = [];
	  for (let a = 0; a < n; ++a) {
	    coloffset[a] = 0;
	  }

	  // go through all elements in M, and add them to sparse matrix A.
	  for (let i = 0; i < M.length; ++i) {
	    let e = M[i];
	    let col = e[1];
	    let adr = Ap[col] + coloffset[col];
	    Ai[adr] = e[0];
	    Ax[adr] = e[2];
	    coloffset[col]++;
	  }
	  let D = new Array(n);
	  let Y = new Array(n);
	  let Lp = new Array(n + 1);
	  let Parent = new Array(n);
	  let Lnz = new Array(n);
	  let Flag = new Array(n);
	  let Pattern = new Array(n);
	  let bp1 = new Array(n);
	  let x = new Array(n);
	  let d;
	  ldlSymbolic(n, Ap, Ai, Lp, Parent, Lnz, Flag);
	  let Lx = new Array(Lp[n]);
	  let Li = new Array(Lp[n]);
	  d = ldlNumeric(n, Ap, Ai, Ax, Lp, Parent, Lnz, Li, Lx, D, Y, Pattern, Flag);
	  if (d === n) {
	    return b => {
	      ldlPerm(n, bp1, b, P);
	      ldlLsolve(n, bp1, Lp, Li, Lx);
	      ldlDsolve(n, bp1, D);
	      ldlLTsolve(n, bp1, Lp, Li, Lx);
	      ldlPermt(n, x, bp1, P);
	      return x;
	    };
	  } else {
	    return null;
	  }
	}

	var cuthillMckee_1 = cuthillMckee;
	function compareNum(a, b) {
	  return a - b;
	}
	function cuthillMckee(list, n) {
	  var adj = new Array(n);
	  var visited = new Array(n);
	  for (var i = 0; i < n; ++i) {
	    adj[i] = [];
	    visited[i] = false;
	  }
	  for (var i = 0; i < list.length; ++i) {
	    var l = list[i];
	    adj[l[0]].push(l[1]);
	  }
	  var toVisit = new Array(n);
	  var eol = 0;
	  var ptr = 0;
	  for (var i = 0; i < n; ++i) {
	    if (visited[i]) {
	      continue;
	    }
	    toVisit[eol++] = i;
	    visited[i] = true;
	    while (ptr < eol) {
	      var v = toVisit[ptr++];
	      var nbhd = adj[v];
	      nbhd.sort(compareNum);
	      for (var j = 0; j < nbhd.length; ++j) {
	        var u = nbhd[j];
	        if (visited[u]) {
	          continue;
	        }
	        visited[u] = true;
	        toVisit[eol++] = u;
	      }
	    }
	  }
	  var result = new Array(n);
	  for (var i = 0; i < n; ++i) {
	    result[toVisit[i]] = i;
	  }
	  return result;
	}
	var cuthillMckee$1 = cuthillMckee_1;

	const getClosestNumber = (array = [], goal = 0) => {
	  const closest = array.reduce((prev, curr) => {
	    return Math.abs(curr - goal) < Math.abs(prev - goal) ? curr : prev;
	  });
	  return closest;
	};
	const getCloseIndex = (array = [], goal = 0) => {
	  const closest = getClosestNumber(array, goal);
	  return array.indexOf(closest);
	};
	const updateSystem = (matrix, y, weights) => {
	  let nbPoints = y.length;
	  let l = nbPoints - 1;
	  let newMatrix = new Array(matrix.length);
	  let newVector = new Float64Array(nbPoints);
	  for (let i = 0; i < l; i++) {
	    let w = weights[i];
	    let diag = i * 2;
	    let next = diag + 1;
	    newMatrix[diag] = matrix[diag].slice();
	    newMatrix[next] = matrix[next].slice();
	    if (w === 0) {
	      newVector[i] = 0;
	    } else {
	      newVector[i] = y[i] * w;
	      newMatrix[diag][2] += w;
	    }
	  }
	  newVector[l] = y[l] * weights[l];
	  newMatrix[l * 2] = matrix[l * 2].slice();
	  newMatrix[l * 2][2] += weights[l];
	  return [newMatrix, newVector];
	};
	const getDeltaMatrix = (nbPoints, lambda) => {
	  let matrix = [];
	  let last = nbPoints - 1;
	  for (let i = 0; i < last; i++) {
	    matrix.push([i, i, lambda * 2]);
	    matrix.push([i + 1, i, -1 * lambda]);
	  }
	  matrix[0][2] = lambda;
	  matrix.push([last, last, lambda]);
	  return {
	    lowerTriangularNonZeros: matrix,
	    permutationEncodedArray: cuthillMckee$1(matrix, nbPoints)
	  };
	};

	/**
	 * Fit the baseline drift by iteratively changing weights of sum square error between the fitted baseline and original signals,
	 * for further information about the parameters you can get the [paper of airPLS](https://github.com/zmzhang/airPLS/blob/master/airPLS_manuscript.pdf)
	 * @param {Array<number>} x - x axis data useful when control points or zones are submitted
	 * @param {Array<number>} y - Original data
	 * @param {object} [options={}] - Options object
	 * @param {number} [options.maxIterations = 100] - Maximal number of iterations if the method does not reach the stop criterion
	 * @param {number} [options.tolerance = 0.001] - Factor of the sum of absolute value of original data, to compute stop criterion
	 * @param {Array<number>} [options.weights = [1,1,...]] - Initial weights vector, default each point has the same weight
	 * @param {number} [options.lambda = 100] - Factor of weights matrix in -> [I + lambda D'D]z = x
	 * @param {Array<number>} [options.controlPoints = []] - Array of x axis values to force that baseline cross those points.
	 * @param {Array<number>} [options.baseLineZones = []] - Array of x axis values (as from - to), to force that baseline cross those zones.
	 * @returns {{corrected: Array<number>, error: number, iteration: number, baseline: Array<number>}}
	 */
	function airPLS(x, y, options = {}) {
	  let {
	    maxIterations = 100,
	    lambda = 100,
	    tolerance = 0.001,
	    weights = new Array(y.length).fill(1),
	    controlPoints = [],
	    baseLineZones = []
	  } = options;
	  if (controlPoints.length > 0) {
	    controlPoints.forEach((e, i, arr) => arr[i] = getCloseIndex(x, e));
	  }
	  if (baseLineZones.length > 0) {
	    baseLineZones.forEach(range => {
	      let indexFrom = getCloseIndex(x, range.from);
	      let indexTo = getCloseIndex(x, range.to);
	      if (indexFrom > indexTo) [indexFrom, indexTo] = [indexTo, indexFrom];
	      for (let i = indexFrom; i < indexTo; i++) {
	        controlPoints.push(i);
	      }
	    });
	  }
	  let baseline, iteration;
	  let nbPoints = y.length;
	  let l = nbPoints - 1;
	  let sumNegDifferences = Number.MAX_SAFE_INTEGER;
	  let stopCriterion = tolerance * y.reduce((sum, e) => Math.abs(e) + sum, 0);
	  let {
	    lowerTriangularNonZeros,
	    permutationEncodedArray
	  } = getDeltaMatrix(nbPoints, lambda);
	  for (iteration = 0; iteration < maxIterations && Math.abs(sumNegDifferences) > stopCriterion; iteration++) {
	    let [leftHandSide, rightHandSide] = updateSystem(lowerTriangularNonZeros, y, weights);
	    let cho = prepare(leftHandSide, nbPoints, permutationEncodedArray);
	    baseline = cho(rightHandSide);
	    sumNegDifferences = 0;
	    let difference = y.map(calculateError);
	    let maxNegativeDiff = -1 * Number.MAX_SAFE_INTEGER;
	    for (let i = 1; i < l; i++) {
	      let diff = difference[i];
	      if (diff >= 0) {
	        weights[i] = 0;
	      } else {
	        weights[i] = Math.exp(iteration * diff / sumNegDifferences);
	        if (maxNegativeDiff < diff) maxNegativeDiff = diff;
	      }
	    }
	    let value = Math.exp(iteration * maxNegativeDiff / sumNegDifferences);
	    weights[0] = value;
	    weights[l] = value;
	    controlPoints.forEach(i => weights[i] = value);
	  }
	  return {
	    corrected: y.map((e, i) => e - baseline[i]),
	    baseline,
	    iteration,
	    error: sumNegDifferences
	  };
	  function calculateError(e, i) {
	    let diff = e - baseline[i];
	    if (diff < 0) sumNegDifferences += diff;
	    return diff;
	  }
	}

	function _typeof(obj) {
	  "@babel/helpers - typeof";

	  if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") {
	    _typeof = function (obj) {
	      return typeof obj;
	    };
	  } else {
	    _typeof = function (obj) {
	      return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj;
	    };
	  }
	  return _typeof(obj);
	}

	/**
	 * Fill an array with sequential numbers
	 * @param {Array<number>} [input] - optional destination array (if not provided a new array will be created)
	 * @param {object} [options={}]
	 * @param {number} [options.from=0] - first value in the array
	 * @param {number} [options.to=10] - last value in the array
	 * @param {number} [options.size=input.length] - size of the array (if not provided calculated from step)
	 * @param {number} [options.step] - if not provided calculated from size
	 * @return {Array<number>}
	 */

	function sequentialFill() {
	  var input = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];
	  var options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
	  if (_typeof(input) === 'object' && !isAnyArray$1(input)) {
	    options = input;
	    input = [];
	  }
	  if (!isAnyArray$1(input)) {
	    throw new TypeError('input must be an array');
	  }
	  var _options = options,
	    _options$from = _options.from,
	    from = _options$from === void 0 ? 0 : _options$from,
	    _options$to = _options.to,
	    to = _options$to === void 0 ? 10 : _options$to,
	    _options$size = _options.size,
	    size = _options$size === void 0 ? input.length : _options$size,
	    step = _options.step;
	  if (size !== 0 && step) {
	    throw new Error('step is defined by the array size');
	  }
	  if (!size) {
	    if (step) {
	      size = Math.floor((to - from) / step) + 1;
	    } else {
	      size = to - from + 1;
	    }
	  }
	  if (!step && size) {
	    step = (to - from) / (size - 1);
	  }
	  if (Array.isArray(input)) {
	    // only works with normal array
	    input.length = 0;
	    for (var i = 0; i < size; i++) {
	      input.push(from);
	      from += step;
	    }
	  } else {
	    if (input.length !== size) {
	      throw new Error('sequentialFill typed array must have the correct length');
	    }
	    for (var _i = 0; _i < size; _i++) {
	      input[_i] = from;
	      from += step;
	    }
	  }
	  return input;
	}

	/**
	 * Adaptive iteratively reweighted penalized least squares [1]
	 *
	 * This function calls ml-airpls
	 *
	 * References:
	 * [1] Zhang, Z.-M.; Chen, S.; Liang, Y.-Z.
	 * Baseline Correction Using Adaptive Iteratively Reweighted Penalized Least Squares.
	 * Analyst 2010, 135 (5), 1138–1146. https://doi.org/10.1039/B922045C.
	 * @export
	 * @param {Array<number>} ys
	 * @param {object} [options] - Options object
	 * @param {Array<number>} [options.x] Optional, Independent axis variable. If not specified, we use a linear grid
	 * @param {object} [options.regression] - Options for the regression
	 * @param {number} [options.regression.maxIterations = 100] - Maximum number of allowed iterations
	 * @param {function} [options.regression.§Regression = PolynomialRegression] - Regression class with a predict method
	 * @param {*} [options.regression.regressionOptions] - Options for regressionFunction
	 * @param {number} [options.regression.tolerance = 0.001] - Convergence error tolerance
	 * @returns {BaselineOutput}
	 */
	function airPLSBaseline$1(ys, options = {}) {
	  const numberPoints = ys.length;
	  let {
	    x,
	    regressionOptions
	  } = options;
	  if (!x) {
	    x = sequentialFill({
	      from: 0,
	      to: numberPoints - 1,
	      size: numberPoints
	    });
	  }
	  let output = airPLS(x, ys, regressionOptions);
	  return {
	    baseline: output.baseline,
	    correctedSpectrum: output.corrected
	  };
	}

	function maybeToPrecision(value, digits) {
	  if (value < 0) {
	    value = 0 - value;
	    if (typeof digits === 'number') {
	      return `- ${value.toPrecision(digits)}`;
	    } else {
	      return `- ${value.toString()}`;
	    }
	  } else {
	    if (typeof digits === 'number') {
	      return value.toPrecision(digits);
	    } else {
	      return value.toString();
	    }
	  }
	}

	function checkArraySize(x, y) {
	  if (!isAnyArray$1(x) || !isAnyArray$1(y)) {
	    throw new TypeError('x and y must be arrays');
	  }
	  if (x.length !== y.length) {
	    throw new RangeError('x and y arrays must have the same length');
	  }
	}

	class BaseRegression {
	  constructor() {
	    if (new.target === BaseRegression) {
	      throw new Error('BaseRegression must be subclassed');
	    }
	  }
	  predict(x) {
	    if (typeof x === 'number') {
	      return this._predict(x);
	    } else if (isAnyArray$1(x)) {
	      const y = [];
	      for (let i = 0; i < x.length; i++) {
	        y.push(this._predict(x[i]));
	      }
	      return y;
	    } else {
	      throw new TypeError('x must be a number or array');
	    }
	  }
	  _predict() {
	    throw new Error('_predict must be implemented');
	  }
	  train() {
	    // Do nothing for this package
	  }
	  toString() {
	    return '';
	  }
	  toLaTeX() {
	    return '';
	  }

	  /**
	   * Return the correlation coefficient of determination (r) and chi-square.
	   * @param {Array<number>} x
	   * @param {Array<number>} y
	   * @return {object}
	   */
	  score(x, y) {
	    if (!isAnyArray$1(x) || !isAnyArray$1(y) || x.length !== y.length) {
	      throw new Error('x and y must be arrays of the same length');
	    }
	    const n = x.length;
	    const y2 = new Array(n);
	    for (let i = 0; i < n; i++) {
	      y2[i] = this._predict(x[i]);
	    }
	    let xSum = 0;
	    let ySum = 0;
	    let chi2 = 0;
	    let rmsd = 0;
	    let xSquared = 0;
	    let ySquared = 0;
	    let xY = 0;
	    for (let i = 0; i < n; i++) {
	      xSum += y2[i];
	      ySum += y[i];
	      xSquared += y2[i] * y2[i];
	      ySquared += y[i] * y[i];
	      xY += y2[i] * y[i];
	      if (y[i] !== 0) {
	        chi2 += (y[i] - y2[i]) * (y[i] - y2[i]) / y[i];
	      }
	      rmsd += (y[i] - y2[i]) * (y[i] - y2[i]);
	    }
	    const r = (n * xY - xSum * ySum) / Math.sqrt((n * xSquared - xSum * xSum) * (n * ySquared - ySum * ySum));
	    return {
	      r: r,
	      r2: r * r,
	      chi2: chi2,
	      rmsd: Math.sqrt(rmsd / n)
	    };
	  }
	}

	class PolynomialRegression extends BaseRegression {
	  constructor(x, y, degree) {
	    super();
	    if (x === true) {
	      this.degree = y.degree;
	      this.powers = y.powers;
	      this.coefficients = y.coefficients;
	    } else {
	      checkArraySize(x, y);
	      regress(this, x, y, degree);
	    }
	  }
	  _predict(x) {
	    let y = 0;
	    for (let k = 0; k < this.powers.length; k++) {
	      y += this.coefficients[k] * Math.pow(x, this.powers[k]);
	    }
	    return y;
	  }
	  toJSON() {
	    return {
	      name: 'polynomialRegression',
	      degree: this.degree,
	      powers: this.powers,
	      coefficients: this.coefficients
	    };
	  }
	  toString(precision) {
	    return this._toFormula(precision, false);
	  }
	  toLaTeX(precision) {
	    return this._toFormula(precision, true);
	  }
	  _toFormula(precision, isLaTeX) {
	    let sup = '^';
	    let closeSup = '';
	    let times = ' * ';
	    if (isLaTeX) {
	      sup = '^{';
	      closeSup = '}';
	      times = '';
	    }
	    let fn = '';
	    let str = '';
	    for (let k = 0; k < this.coefficients.length; k++) {
	      str = '';
	      if (this.coefficients[k] !== 0) {
	        if (this.powers[k] === 0) {
	          str = maybeToPrecision(this.coefficients[k], precision);
	        } else {
	          if (this.powers[k] === 1) {
	            str = `${maybeToPrecision(this.coefficients[k], precision) + times}x`;
	          } else {
	            str = `${maybeToPrecision(this.coefficients[k], precision) + times}x${sup}${this.powers[k]}${closeSup}`;
	          }
	        }
	        if (this.coefficients[k] > 0 && k !== this.coefficients.length - 1) {
	          str = ` + ${str}`;
	        } else if (k !== this.coefficients.length - 1) {
	          str = ` ${str}`;
	        }
	      }
	      fn = str + fn;
	    }
	    if (fn.charAt(0) === '+') {
	      fn = fn.slice(1);
	    }
	    return `f(x) = ${fn}`;
	  }
	  static load(json) {
	    if (json.name !== 'polynomialRegression') {
	      throw new TypeError('not a polynomial regression model');
	    }
	    return new PolynomialRegression(true, json);
	  }
	}
	function regress(pr, x, y, degree) {
	  const n = x.length;
	  let powers;
	  if (Array.isArray(degree)) {
	    powers = degree;
	    degree = powers.length;
	  } else {
	    degree++;
	    powers = new Array(degree);
	    for (let k = 0; k < degree; k++) {
	      powers[k] = k;
	    }
	  }
	  const F = new Matrix(n, degree);
	  const Y = new Matrix([y]);
	  for (let k = 0; k < degree; k++) {
	    for (let i = 0; i < n; i++) {
	      if (powers[k] === 0) {
	        F.set(i, k, 1);
	      } else {
	        F.set(i, k, Math.pow(x[i], powers[k]));
	      }
	    }
	  }
	  const FT = new MatrixTransposeView(F);
	  const A = FT.mmul(F);
	  const B = FT.mmul(new MatrixTransposeView(Y));
	  pr.degree = degree - 1;
	  pr.powers = powers;
	  pr.coefficients = solve(A, B).to1DArray();
	}

	/**
	 * Iterative regression-based baseline correction
	 * @param {Array<number>} x - Independent axis variable
	 * @param {Array<number>} y - Dependent axis variable
	 * @param {object} [options] - Options object
	 * @param {number} [options.maxIterations = 100] - Maximum number of allowed iterations
	 * @param {function} [options.Regression = PolynomialRegression] - Regression class with a predict method
	 * @param {*} [options.regressionOptions] - Options for regressionFunction
	 * @param {number} [options.tolerance = 0.001] - Convergence error tolerance
	 * @return {{corrected: Array<number>, delta: number, iteration: number, baseline: Array<number>}}
	 */
	function baselineCorrectionRegression(x, y, options = {}) {
	  let {
	    maxIterations = 100,
	    Regression = PolynomialRegression,
	    regressionOptions,
	    tolerance = 0.001
	  } = options;
	  if (!regressionOptions && Regression === PolynomialRegression) {
	    regressionOptions = 3;
	  }
	  let baseline = y.slice();
	  let fitting = y.slice();
	  let oldFitting = y;
	  let iteration = 0;
	  let delta;
	  let regression;
	  while (iteration < maxIterations) {
	    // Calculate the fitting result
	    regression = new Regression(x, baseline, regressionOptions);
	    delta = 0;
	    for (let i = 0; i < baseline.length; i++) {
	      fitting[i] = regression.predict(x[i]);
	      if (baseline[i] > fitting[i]) {
	        baseline[i] = fitting[i];
	      }
	      delta += Math.abs((fitting[i] - oldFitting[i]) / oldFitting[i]);
	    }

	    // Stop criterion
	    if (delta < tolerance) {
	      break;
	    } else {
	      oldFitting = fitting.slice();
	      iteration++;
	    }
	  }

	  // removes baseline
	  let corrected = new Array(baseline.length);
	  for (let j = 0; j < baseline.length; j++) {
	    corrected[j] = y[j] - baseline[j];
	  }
	  return {
	    corrected,
	    delta,
	    iteration,
	    baseline,
	    regression
	  };
	}

	/**
	 * Iterative polynomial fitting [1]
	 *
	 * Implementation based on ml-baseline-correction-regression
	 *
	 * References:
	 * [1] Gan, F.; Ruan, G.; Mo, J.
	 * Baseline Correction by Improved Iterative Polynomial Fitting with Automatic Threshold.
	 *  Chemometrics and Intelligent Laboratory Systems 2006, 82 (1), 59–65.
	 * https://doi.org/10.1016/j.chemolab.2005.08.009.
	 * @export
	 * @param {Array<number>} ys
	 * @param {object} [options] - Options object
	 * @param {Array<number>} [options.x] Optional, Independent axis variable. If not specified, we use a linear grid
	 * @param {Object} [options.regression]
	 * @param {number} [options.regression.maxIterations = 100] - Maximum number of allowed iterations
	 * @param {Object} [options.regression]
	 * @param {function} [options.regression.Regression = PolynomialRegression] - Regression class with a predict method
	 * @param {Object} [options.regression.regressionOptions] - Options for regressionFunction
	 * @param {number} [options.regression.tolerance = 0.001] - Convergence error tolerance
	 * @returns {BaselineOutput}
	 */
	function iterativePolynomialBaseline$1(ys, options = {}) {
	  const numberPoints = ys.length;
	  let {
	    x,
	    regressionOptions
	  } = options;
	  if (!x) {
	    x = sequentialFill({
	      from: 0,
	      to: numberPoints - 1,
	      size: numberPoints
	    });
	  }
	  let output = baselineCorrectionRegression(x, ys, regressionOptions);
	  return {
	    baseline: output.baseline,
	    correctedSpectrum: output.corrected
	  };
	}

	/**

	 *
	 * @export
	 * @param {Array<number>} ys
	 * @param {Object} [options={}]
	 * @param {number} [options.window] rolling window size, defaults to 10% of the length of the spectrum
	 * @param {string} [options.padding.size=window-1] none, value, circular, duplicate
	 * @param {string} [options.padding.algorithm='duplicate'] none, value, circular, duplicate
	 * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
	 * @returns {BaselineOutput}
	 */
	function rollingAverageBaseline$1(ys, options = {}) {
	  let window = Math.max(Math.round(ys.length * 0.1), 2);
	  let defaults = {
	    window,
	    padding: {
	      size: window - 1,
	      algorithm: 'duplicate',
	      value: 0
	    }
	  };
	  let actualOptions = {
	    ...defaults,
	    ...options
	  };
	  let baseline = xRollingAverage(ys, actualOptions);
	  let corrected = new Float64Array(ys.length);
	  for (let i = 0; i < corrected.length; i++) {
	    corrected[i] = ys[i] - baseline[i];
	  }
	  return {
	    baseline,
	    correctedSpectrum: corrected
	  };
	}

	function rollingBall(spectrum, options = {}) {
	  if (!isAnyArray$1(spectrum)) {
	    throw new Error('Spectrum must be an array');
	  }
	  if (spectrum.length === 0) {
	    throw new TypeError('Spectrum must not be empty');
	  }
	  const numberPoints = spectrum.length;
	  const maxima = new Float64Array(numberPoints);
	  const minima = new Float64Array(numberPoints);
	  const baseline = new Float64Array(numberPoints);
	  // windowM 4 percent of spectrum length
	  // windowS 8 percent of spectrum length
	  const {
	    windowM = Math.round(numberPoints * 0.04),
	    windowS = Math.round(numberPoints * 0.08)
	  } = options;
	  // fi(1) in original paper
	  for (let i = 0; i < spectrum.length; i++) {
	    const windowLeft = Math.max(0, i - windowM);
	    const windowRight = Math.min(i + windowM + 1, spectrum.length);
	    minima[i] = xMinValue(spectrum, {
	      fromIndex: windowLeft,
	      toIndex: windowRight
	    });
	  }
	  // fi in original paper
	  for (let i = 0; i < minima.length; i++) {
	    const windowLeft = Math.max(0, i - windowM);
	    const windowRight = Math.min(i + windowM + 1, minima.length);
	    maxima[i] = xMaxValue(minima, {
	      fromIndex: windowLeft,
	      toIndex: windowRight
	    });
	  }
	  for (let i = 0; i < minima.length; i++) {
	    const windowLeft = Math.max(0, i - windowS);
	    const windowRight = Math.min(i + windowS + 1, maxima.length);
	    baseline[i] = xMean(maxima.subarray(windowLeft, windowRight));
	  }
	  return baseline;
	}

	/**
	 * Rolling ball baseline correction algorithm.
	 * From the abstract of (1):
	 * "This algorithm behaves equivalently to traditional polynomial backgrounds in simple spectra,
	 * [...] and is considerably more robust for multiple overlapping peaks, rapidly varying background [...]
	 *
	 * The baseline is the trace one gets by rolling a ball below a spectrum. Algorithm has three steps:
	 * Finding the minima in each window, find maxima among minima and then smooth over them by averaging.
	 *
	 * Algorithm described in (1), but in the implementation here the window width does not change.
	 *
	 * Reference:
	 * (1) Kneen, M. A.; Annegarn, H. J.
	 *     Algorithm for Fitting XRF, SEM and PIXE X-Ray Spectra Backgrounds.
	 *     Nuclear Instruments and Methods in Physics Research Section B: Beam Interactions with Materials and Atoms 1996, 109–110, 209–213.
	 *     https://doi.org/10.1016/0168-583X(95)00908-6.
	 * (2) Kristian Hovde Liland, Bjørn-Helge Mevik, Roberto Canteri: baseline.
	 *     https://cran.r-project.org/web/packages/baseline/index.html
	 *
	 * @export
	 * @param {Array<number>} ys
	 * @param {Object} [options={}]
	 * @param {Number} [options.windowM] - width of local window for minimization/maximization, defaults to 4% of the spectrum length
	 * @param {Number} [options.windowS] - width of local window for smoothing, defaults to 8% of the specturm length
	 * @returns {BaselineOutput}
	 */
	function rollingBallBaseline$1(ys, options = {}) {
	  const baseline = rollingBall(ys, options);
	  let corrected = new Float64Array(ys.length);
	  for (let i = 0; i < corrected.length; i++) {
	    corrected[i] = ys[i] - baseline[i];
	  }
	  return {
	    baseline,
	    correctedSpectrum: corrected
	  };
	}

	/**

	 *
	 * @export
	 * @param {Array<number>} ys
	 * @param {Object} [options={}]
	 * @param {number} [options.window] rolling window size, defaults to 10% of the length of the spectrum
	 * @param {string} [options.padding.size=window-1] none, value, circular, duplicate
	 * @param {string} [options.padding.algorithm='duplicate'] none, value, circular, duplicate
	 * @param {number} [options.padding.value=0] value to use for padding (if algorithm='value')
	 * @returns {BaselineOutput}
	 */
	function rollingMedianBaseline$1(ys, options = {}) {
	  let window = Math.max(Math.round(ys.length * 0.1), 2);
	  let defaults = {
	    window,
	    padding: {
	      size: window - 1,
	      algorithm: 'duplicate',
	      value: 0
	    }
	  };
	  let actualOptions = {
	    ...defaults,
	    ...options
	  };
	  let baseline = xRollingMedian(ys, actualOptions);
	  let corrected = new Float64Array(ys.length);
	  for (let i = 0; i < corrected.length; i++) {
	    corrected[i] = ys[i] - baseline[i];
	  }
	  return {
	    baseline,
	    correctedSpectrum: corrected
	  };
	}

	//@ts-expect-error no type definition for baselines
	/**
	 * @param data
	 */
	function airPLSBaseline(data) {
	  data.y = airPLSBaseline$1(data.y).correctedSpectrum;
	  return {
	    data
	  };
	}

	//@ts-expect-error no type definition for baselines
	/**
	 * @param data
	 */
	function iterativePolynomialBaseline(data) {
	  data.y = iterativePolynomialBaseline$1(data.y).correctedSpectrum;
	  return {
	    data
	  };
	}

	//@ts-expect-error no type definition for baselines
	/**
	 * @param data
	 */
	function rollingAverageBaseline(data) {
	  data.y = rollingAverageBaseline$1(data.y).correctedSpectrum;
	  return {
	    data
	  };
	}

	//@ts-expect-error no type definition for baselines
	/**
	 * @param data
	 */
	function rollingBallBaseline(data) {
	  data.y = rollingBallBaseline$1(data.y).correctedSpectrum;
	  return {
	    data
	  };
	}

	//@ts-expect-error no type definition for baselines
	/**
	 * @param data
	 */
	function rollingMedianBaseline(data) {
	  data.y = rollingMedianBaseline$1(data.y).correctedSpectrum;
	  return {
	    data
	  };
	}

	/**
	 * Apply Savitzky Golay algorithm
	 * @param [ys] Array of y values
	 * @param [xs] Array of X or deltaX
	 * @return  Array containing the new ys (same length)
	 */
	function sgg(ys, xs, options = {}) {
	  let {
	    windowSize = 9,
	    derivative = 0,
	    polynomial = 3
	  } = options;
	  if (windowSize % 2 === 0 || windowSize < 5 || !Number.isInteger(windowSize)) {
	    throw new RangeError('Invalid window size (should be odd and at least 5 integer number)');
	  }
	  if (!isAnyArray$1(ys)) {
	    throw new TypeError('Y values must be an array');
	  }
	  if (typeof xs === 'undefined') {
	    throw new TypeError('X must be defined');
	  }
	  if (windowSize > ys.length) {
	    throw new RangeError(`Window size is higher than the data length ${windowSize}>${ys.length}`);
	  }
	  if (derivative < 0 || !Number.isInteger(derivative)) {
	    throw new RangeError('Derivative should be a positive integer');
	  }
	  if (polynomial < 1 || !Number.isInteger(polynomial)) {
	    throw new RangeError('Polynomial should be a positive integer');
	  }
	  if (polynomial >= 6) {
	    // eslint-disable-next-line no-console
	    console.warn('You should not use polynomial grade higher than 5 if you are' + ' not sure that your data arises from such a model. Possible polynomial oscillation problems');
	  }
	  let half = Math.floor(windowSize / 2);
	  let np = ys.length;
	  let ans = new Float64Array(np);
	  let weights = fullWeights(windowSize, polynomial, derivative);
	  let hs = 0;
	  let constantH = true;
	  if (isAnyArray$1(xs)) {
	    constantH = false;
	  } else {
	    hs = Math.pow(xs, derivative);
	  }
	  //For the borders
	  for (let i = 0; i < half; i++) {
	    let wg1 = weights[half - i - 1];
	    let wg2 = weights[half + i + 1];
	    let d1 = 0;
	    let d2 = 0;
	    for (let l = 0; l < windowSize; l++) {
	      d1 += wg1[l] * ys[l];
	      d2 += wg2[l] * ys[np - windowSize + l];
	    }
	    if (constantH) {
	      ans[half - i - 1] = d1 / hs;
	      ans[np - half + i] = d2 / hs;
	    } else {
	      hs = getHs(xs, half - i - 1, half, derivative);
	      ans[half - i - 1] = d1 / hs;
	      hs = getHs(xs, np - half + i, half, derivative);
	      ans[np - half + i] = d2 / hs;
	    }
	  }
	  //For the internal points
	  let wg = weights[half];
	  for (let i = windowSize; i <= np; i++) {
	    let d = 0;
	    for (let l = 0; l < windowSize; l++) d += wg[l] * ys[l + i - windowSize];
	    if (!constantH) {
	      hs = getHs(xs, i - half - 1, half, derivative);
	    }
	    ans[i - half - 1] = d / hs;
	  }
	  return ans;
	}
	function getHs(h, center, half, derivative) {
	  let hs = 0;
	  let count = 0;
	  for (let i = center - half; i < center + half; i++) {
	    if (i >= 0 && i < h.length - 1) {
	      hs += h[i + 1] - h[i];
	      count++;
	    }
	  }
	  return Math.pow(hs / count, derivative);
	}
	function gramPoly(i, m, k, s) {
	  let Grampoly = 0;
	  if (k > 0) {
	    Grampoly = (4 * k - 2) / (k * (2 * m - k + 1)) * (i * gramPoly(i, m, k - 1, s) + s * gramPoly(i, m, k - 1, s - 1)) - (k - 1) * (2 * m + k) / (k * (2 * m - k + 1)) * gramPoly(i, m, k - 2, s);
	  } else {
	    if (k === 0 && s === 0) {
	      Grampoly = 1;
	    } else {
	      Grampoly = 0;
	    }
	  }
	  return Grampoly;
	}
	function genFact(a, b) {
	  let gf = 1;
	  if (a >= b) {
	    for (let j = a - b + 1; j <= a; j++) {
	      gf *= j;
	    }
	  }
	  return gf;
	}
	function weight(i, t, m, n, s) {
	  let sum = 0;
	  for (let k = 0; k <= n; k++) {
	    sum += (2 * k + 1) * (genFact(2 * m, k) / genFact(2 * m + k + 1, k + 1)) * gramPoly(i, m, k, 0) * gramPoly(t, m, k, s);
	  }
	  return sum;
	}
	/**
	 * @private
	 * @param m  Number of points
	 * @param n  Polynomial grade
	 * @param s  Derivative
	 */
	function fullWeights(m, n, s) {
	  let weights = new Array(m);
	  let np = Math.floor(m / 2);
	  for (let t = -np; t <= np; t++) {
	    weights[t + np] = new Float64Array(m);
	    for (let j = -np; j <= np; j++) {
	      weights[t + np][j + np] = weight(j, t, np, n, s);
	    }
	  }
	  return weights;
	}

	/**
	 * Calculate the first derivative using Savitzky–Golay filter.
	 * @param data
	 */
	function firstDerivative(data, options = {}) {
	  const {
	    x,
	    y
	  } = data;
	  return {
	    data: {
	      x,
	      y: sgg(y, x, {
	        ...options,
	        derivative: 1
	      })
	    }
	  };
	}

	/**
	 * Calculate the second derivative using Savitzky–Golay filter.
	 * @param data
	 */
	function secondDerivative(data, options = {}) {
	  const {
	    x,
	    y
	  } = data;
	  return {
	    data: {
	      x,
	      y: sgg(y, x, {
	        ...options,
	        derivative: 2
	      })
	    }
	  };
	}

	/**
	 * Calculate the third derivative using Savitzky–Golay filter.
	 * @param data
	 */
	function thirdDerivative(data, options = {}) {
	  const {
	    x,
	    y
	  } = data;
	  return {
	    data: {
	      x,
	      y: sgg(y, x, {
	        ...options,
	        derivative: 3
	      })
	    }
	  };
	}

	/**
	 * Apply the Savitzky Golay Generalized Filter
	 * @param data
	 */
	function savitzkyGolay(data, options = {}) {
	  const {
	    x,
	    y
	  } = data;
	  return {
	    data: {
	      x,
	      y: sgg(y, x, options)
	    }
	  };
	}

	/**
	 * Ensure X values are strictly monotonic increasing
	 * http://www-groups.mcs.st-andrews.ac.uk/~john/analysis/Lectures/L8.html
	 * @param data
	 */
	function ensureGrowing(data) {
	  return {
	    data: xyEnsureGrowingX(data)
	  };
	}

	/**
	 * Filter that allows to
	 * @param data
	 * @param options
	 */
	function equallySpaced(data, options = {}) {
	  return {
	    data: xyEquallySpaced(data, options)
	  };
	}

	/**
	 * Filter that allows to
	 * @param data
	 * @param options
	 */
	function filterX(data, options = {}) {
	  return {
	    data: xyFilterX(data, options)
	  };
	}

	var IDX = 256,
	  HEX = [],
	  BUFFER;
	while (IDX--) HEX[IDX] = (IDX + 256).toString(16).substring(1);
	function v4() {
	  var i = 0,
	    num,
	    out = '';
	  if (!BUFFER || IDX + 16 > 256) {
	    BUFFER = Array(i = 256);
	    while (i--) BUFFER[i] = 256 * Math.random() | 0;
	    i = IDX = 0;
	  }
	  for (; i < 16; i++) {
	    num = BUFFER[IDX + i];
	    if (i == 6) out += HEX[num & 15 | 64];else if (i == 8) out += HEX[num & 63 | 128];else out += HEX[num];
	    if (i & 1 && i > 1 && i < 11) out += '-';
	  }
	  IDX++;
	  return out;
	}

	/**
	 * Correction of the x and y coordinates using a quadratic optimizations with the peak and its 3 closest neighbors to determine the true x,y values of the peak.
	 * This process is done in place and is very fast.
	 * @param data
	 * @param peaks
	 */
	function optimizeTop(data, peaks) {
	  const {
	    x,
	    y
	  } = data;
	  for (const peak of peaks) {
	    let currentIndex = peak.index;
	    // The detected peak could be moved 1 or 2 units to left or right.
	    if (y[currentIndex - 1] >= y[currentIndex - 2] && y[currentIndex - 1] >= y[currentIndex]) {
	      currentIndex--;
	    } else if (y[currentIndex + 1] >= y[currentIndex] && y[currentIndex + 1] >= y[currentIndex + 2]) {
	      currentIndex++;
	    } else if (y[currentIndex - 2] >= y[currentIndex - 3] && y[currentIndex - 2] >= y[currentIndex - 1]) {
	      currentIndex -= 2;
	    } else if (y[currentIndex + 2] >= y[currentIndex + 1] && y[currentIndex + 2] >= y[currentIndex + 3]) {
	      currentIndex += 2;
	    }
	    // interpolation to a sin() function
	    if (y[currentIndex - 1] > 0 && y[currentIndex + 1] > 0 && y[currentIndex] >= y[currentIndex - 1] && y[currentIndex] >= y[currentIndex + 1] && (y[currentIndex] !== y[currentIndex - 1] || y[currentIndex] !== y[currentIndex + 1])) {
	      let alpha = 20 * Math.log10(y[currentIndex - 1]);
	      let beta = 20 * Math.log10(y[currentIndex]);
	      let gamma = 20 * Math.log10(y[currentIndex + 1]);
	      let p = 0.5 * (alpha - gamma) / (alpha - 2 * beta + gamma);
	      peak.x = x[currentIndex] + (x[currentIndex] - x[currentIndex - 1]) * p;
	      peak.y = y[currentIndex] - 0.25 * (y[currentIndex - 1] - y[currentIndex + 1]) * p;
	    }
	  }
	}

	/**
	 * Global spectra deconvolution
	 * @param  data - Object data with x and y arrays. Values in x has to be growing
	 * @param {number} [options.broadRatio = 0.00] - If `broadRatio` is higher than 0, then all the peaks which second derivative
	 * smaller than `broadRatio * maxAbsSecondDerivative` will be marked with the soft mask equal to true.

	 */
	function gsd(data, options = {}) {
	  let {
	    sgOptions = {
	      windowSize: 9,
	      polynomial: 3
	    },
	    noiseLevel,
	    smoothY = false,
	    maxCriteria = true,
	    minMaxRatio = 0.00025,
	    realTopDetection = false
	  } = options;
	  let {
	    x,
	    y
	  } = data;
	  if (xIsMonotonic(x) !== 1) {
	    throw new Error('GSD only accepts monotone increasing x values');
	  }
	  //rescale;
	  y = y.slice();
	  // If the max difference between delta x is less than 5%, then,
	  // we can assume it to be equally spaced variable
	  let equallySpaced = xIsEquallySpaced(x);
	  if (noiseLevel === undefined) {
	    if (equallySpaced) {
	      const noiseInfo = xNoiseStandardDeviation(y);
	      if (maxCriteria) {
	        noiseLevel = noiseInfo.median + 1.5 * noiseInfo.sd;
	      } else {
	        noiseLevel = -noiseInfo.median + 1.5 * noiseInfo.sd;
	      }
	    } else {
	      noiseLevel = 0;
	    }
	  } else if (!maxCriteria) {
	    noiseLevel *= -1;
	  }
	  if (!maxCriteria) {
	    for (let i = 0; i < y.length; i++) {
	      y[i] *= -1;
	    }
	  }
	  if (noiseLevel !== undefined) {
	    for (let i = 0; i < y.length; i++) {
	      if (y[i] < noiseLevel) {
	        y[i] = noiseLevel;
	      }
	    }
	  }
	  let yData = y;
	  let dY, ddY;
	  const {
	    windowSize,
	    polynomial
	  } = sgOptions;
	  if (equallySpaced) {
	    if (smoothY) {
	      yData = sgg(y, x[1] - x[0], {
	        windowSize,
	        polynomial,
	        derivative: 0
	      });
	    }
	    dY = sgg(y, x[1] - x[0], {
	      windowSize,
	      polynomial,
	      derivative: 1
	    });
	    ddY = sgg(y, x[1] - x[0], {
	      windowSize,
	      polynomial,
	      derivative: 2
	    });
	  } else {
	    if (smoothY) {
	      yData = sgg(y, x, {
	        windowSize,
	        polynomial,
	        derivative: 0
	      });
	    }
	    dY = sgg(y, x, {
	      windowSize,
	      polynomial,
	      derivative: 1
	    });
	    ddY = sgg(y, x, {
	      windowSize,
	      polynomial,
	      derivative: 2
	    });
	  }
	  const minY = xMinValue(yData);
	  const maxY = xMaxValue(yData);
	  if (minY > maxY || minY === maxY) return [];
	  const yThreshold = minY + (maxY - minY) * minMaxRatio;
	  const dX = x[1] - x[0];
	  let lastMax = null;
	  let lastMin = null;
	  let minddY = [];
	  let intervalL = [];
	  let intervalR = [];
	  // By the intermediate value theorem We cannot find 2 consecutive maximum or minimum
	  for (let i = 1; i < yData.length - 1; ++i) {
	    if (dY[i] < dY[i - 1] && dY[i] <= dY[i + 1] || dY[i] <= dY[i - 1] && dY[i] < dY[i + 1]) {
	      lastMin = {
	        x: x[i],
	        index: i
	      };
	      if (dX > 0 && lastMax !== null) {
	        intervalL.push(lastMax);
	        intervalR.push(lastMin);
	      }
	    }
	    // Maximum in first derivative
	    if (dY[i] >= dY[i - 1] && dY[i] > dY[i + 1] || dY[i] > dY[i - 1] && dY[i] >= dY[i + 1]) {
	      lastMax = {
	        x: x[i],
	        index: i
	      };
	      if (dX < 0 && lastMin !== null) {
	        intervalL.push(lastMax);
	        intervalR.push(lastMin);
	      }
	    }
	    // Minimum in second derivative
	    if (ddY[i] < ddY[i - 1] && ddY[i] < ddY[i + 1]) {
	      minddY.push(i);
	    }
	  }
	  let lastK = -1;
	  const peaks = [];
	  for (const minddYIndex of minddY) {
	    let deltaX = x[minddYIndex];
	    let possible = -1;
	    let k = lastK + 1;
	    let minDistance = Number.POSITIVE_INFINITY;
	    let currentDistance = 0;
	    while (possible === -1 && k < intervalL.length) {
	      currentDistance = Math.abs(deltaX - (intervalL[k].x + intervalR[k].x) / 2);
	      if (currentDistance < (intervalR[k].x - intervalL[k].x) / 2) {
	        possible = k;
	        lastK = k;
	      }
	      ++k;
	      // Not getting closer?
	      if (currentDistance >= minDistance) {
	        break;
	      }
	      minDistance = currentDistance;
	    }
	    if (possible !== -1) {
	      if (yData[minddYIndex] > yThreshold) {
	        let width = Math.abs(intervalR[possible].x - intervalL[possible].x);
	        peaks.push({
	          id: v4(),
	          x: deltaX,
	          y: yData[minddYIndex],
	          width,
	          index: minddYIndex,
	          ddY: ddY[minddYIndex],
	          inflectionPoints: {
	            from: intervalL[possible],
	            to: intervalR[possible]
	          }
	        });
	      }
	    }
	  }
	  if (realTopDetection) {
	    optimizeTop({
	      x,
	      y: yData
	    }, peaks);
	  }
	  peaks.forEach(peak => {
	    if (!maxCriteria) {
	      peak.y *= -1;
	      peak.ddY = peak.ddY * -1;
	    }
	  });
	  peaks.sort((a, b) => {
	    return a.x - b.x;
	  });
	  return peaks;
	}

	/**
	 * Filter that allows to calibrateX the x axis based on the presence of peaks
	 */
	function calibrateX(data, options = {}) {
	  const {
	    targetX = 0,
	    nbPeaks = 1,
	    from = data.x[0],
	    to = data.x[data.x.length - 1],
	    gsd: gsdOptions = {
	      minMaxRatio: 0.1,
	      realTopDetection: true,
	      smoothY: true,
	      sgOptions: {
	        windowSize: 7,
	        polynomial: 3
	      }
	    }
	  } = options;
	  const fromIndex = xFindClosestIndex(data.x, from);
	  const toIndex = xFindClosestIndex(data.x, to);
	  let peaks = gsd({
	    x: data.x.subarray(fromIndex, toIndex),
	    y: data.y.subarray(fromIndex, toIndex)
	  }, gsdOptions).sort((a, b) => b.y - a.y).slice(0, nbPeaks);
	  if (peaks.length < nbPeaks) return {
	    data
	  };
	  const middle = xMean(peaks.map(peak => peak.x));
	  return {
	    data: {
	      x: xAdd(data.x, targetX - middle),
	      y: data.y
	    }
	  };
	}

	/**
	 * Filter that allows to
	 * @param data
	 * @param options
	 */
	function xFunction(data, options = {}) {
	  return {
	    data: {
	      x: xApplyFunctionStr(data.x, {
	        variableLabel: 'x',
	        fctString: options.function
	      }),
	      y: data.y
	    }
	  };
	}

	/**
	 * Filter that allows to
	 * @param data
	 * @param options
	 */
	function yFunction(data, options = {}) {
	  return {
	    data: {
	      x: data.x,
	      y: xApplyFunctionStr(data.y, {
	        variableLabel: 'y',
	        fctString: options.function
	      })
	    }
	  };
	}

	var Filters = /*#__PURE__*/Object.freeze({
		__proto__: null,
		centerMean: centerMean,
		centerMedian: centerMedian,
		fromTo: fromTo,
		normed: normed,
		divideBySD: divideBySD,
		rescale: rescale,
		paretoNormalization: paretoNormalization,
		airPLSBaseline: airPLSBaseline,
		iterativePolynomialBaseline: iterativePolynomialBaseline,
		rollingAverageBaseline: rollingAverageBaseline,
		rollingBallBaseline: rollingBallBaseline,
		rollingMedianBaseline: rollingMedianBaseline,
		firstDerivative: firstDerivative,
		secondDerivative: secondDerivative,
		thirdDerivative: thirdDerivative,
		savitzkyGolay: savitzkyGolay,
		ensureGrowing: ensureGrowing,
		equallySpaced: equallySpaced,
		filterX: filterX,
		calibrateX: calibrateX,
		xFunction: xFunction,
		yFunction: yFunction
	});

	/**
	 * Apply filters on {x:[], y:[]}
	 * @returns A very important number
	 */
	function filterXY(data, filters) {
	  let result = {
	    data: {
	      x: xEnsureFloat64(data.x),
	      y: xEnsureFloat64(data.y)
	    }
	  };
	  const logs = [];
	  for (let filter of filters) {
	    const start = Date.now();
	    // eslint-disable-next-line import/namespace
	    const filterFct = Filters[filter.name];
	    if (!filterFct) {
	      throw new Error(`Unknown filter: ${filter.name}`);
	    }
	    // @ts-expect-error some method have options and some other ones don't have any options
	    result = filterFct(result.data, filter.options);
	    logs.push({
	      name: filter.name,
	      time: Date.now() - start
	    });
	  }
	  return {
	    logs,
	    data: result.data
	  };
	}

	function getNormalizedSpectrum(spectrum, options = {}) {
	  const data = {
	    x: spectrum.variables.x.data,
	    y: spectrum.variables.y.data
	  };
	  const newSpectrum = {
	    variables: {
	      x: {
	        data: spectrum.variables.x.data,
	        units: spectrum.variables.x.units,
	        label: spectrum.variables.x.label
	      },
	      y: {
	        data: spectrum.variables.y.data,
	        units: spectrum.variables.y.units,
	        label: spectrum.variables.y.label
	      }
	    }
	  };
	  if (spectrum.title) newSpectrum.title = spectrum.title;
	  if (spectrum.dataType) newSpectrum.dataType = spectrum.dataType;
	  if (spectrum.meta) newSpectrum.meta = spectrum.meta;
	  if (spectrum.id) newSpectrum.id = spectrum.id;
	  const {
	    from = spectrum.variables.x.min,
	    to = spectrum.variables.x.max,
	    numberOfPoints,
	    exclusions = [],
	    zones = []
	  } = options;
	  let {
	    filters = []
	  } = options;
	  filters = JSON.parse(JSON.stringify(filters));
	  if (numberOfPoints) {
	    filters.push({
	      name: 'equallySpaced',
	      options: {
	        from,
	        to,
	        exclusions,
	        zones,
	        numberOfPoints
	      }
	    });
	  } else {
	    filters.push({
	      name: 'filterX',
	      options: {
	        from,
	        to,
	        exclusions,
	        zones
	      }
	    });
	  }
	  const {
	    x,
	    y
	  } = filterXY(data, filters).data;
	  // filters change the y axis, we get rid of the units
	  // TODO we should deal correctly with this problem
	  if (filters.length > 1) {
	    newSpectrum.variables.y.units = '';
	    newSpectrum.variables.y.label = newSpectrum.variables.y.label?.replace(/\s*\[.*\]/, '');
	  }
	  newSpectrum.variables.x.data = x;
	  newSpectrum.variables.x.min = xMinValue(x);
	  newSpectrum.variables.x.max = xMaxValue(x);
	  newSpectrum.variables.x.isMonotone = xIsMonotonic(x) !== 0;
	  newSpectrum.variables.y.data = y;
	  newSpectrum.variables.y.min = xMinValue(y);
	  newSpectrum.variables.y.max = xMaxValue(y);
	  newSpectrum.variables.y.isMonotone = xIsMonotonic(y) !== 0;
	  return newSpectrum;
	}

	var quantities = {exports: {}};

	/*
	The MIT License (MIT)
	Copyright © 2006-2007 Kevin C. Olbrich
	Copyright © 2010-2016 LIM SAS (http://lim.eu) - Julien Sanchez

	Permission is hereby granted, free of charge, to any person obtaining a copy of
	this software and associated documentation files (the "Software"), to deal in
	the Software without restriction, including without limitation the rights to
	use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
	of the Software, and to permit persons to whom the Software is furnished to do
	so, subject to the following conditions:

	The above copyright notice and this permission notice shall be included in all
	copies or substantial portions of the Software.

	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
	FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
	AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
	LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
	OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
	SOFTWARE.
	*/
	(function (module, exports) {
	  (function (global, factory) {
	    module.exports = factory() ;
	  })(commonjsGlobal, function () {

	    /**
	     * Tests if a value is a string
	     *
	     * @param {*} value - Value to test
	     *
	     * @returns {boolean} true if value is a string, false otherwise
	     */
	    function isString(value) {
	      return typeof value === "string" || value instanceof String;
	    }

	    /*
	     * Prefer stricter Number.isFinite if currently supported.
	     * To be dropped when ES6 is finalized. Obsolete browsers will
	     * have to use ES6 polyfills.
	     */
	    var isFiniteImpl = Number.isFinite || window.isFinite;
	    /**
	     * Tests if a value is a number
	     *
	     * @param {*} value - Value to test
	     *
	     * @returns {boolean} true if value is a number, false otherwise
	     */
	    function isNumber(value) {
	      // Number.isFinite allows not to consider NaN or '1' as numbers
	      return isFiniteImpl(value);
	    }

	    /*
	     * Identity function
	     */
	    function identity(value) {
	      return value;
	    }

	    /**
	     * Returns unique strings from list
	     *
	     * @param {string[]} strings - array of strings
	     *
	     *
	     * @returns {string[]} a new array of strings without duplicates
	     */
	    function uniq(strings) {
	      var seen = {};
	      return strings.filter(function (item) {
	        return seen.hasOwnProperty(item) ? false : seen[item] = true;
	      });
	    }
	    function compareArray(array1, array2) {
	      if (array2.length !== array1.length) {
	        return false;
	      }
	      for (var i = 0; i < array1.length; i++) {
	        if (array2[i].compareArray) {
	          if (!array2[i].compareArray(array1[i])) {
	            return false;
	          }
	        }
	        if (array2[i] !== array1[i]) {
	          return false;
	        }
	      }
	      return true;
	    }
	    function assign(target, properties) {
	      Object.keys(properties).forEach(function (key) {
	        target[key] = properties[key];
	      });
	    }

	    /**
	     * Safely multiplies numbers while avoiding floating errors
	     * like 0.1 * 0.1 => 0.010000000000000002
	     *
	     * @param {...number} numbers - numbers to multiply
	     *
	     * @returns {number} result
	     */
	    function mulSafe() {
	      var result = 1,
	        decimals = 0;
	      for (var i = 0; i < arguments.length; i++) {
	        var arg = arguments[i];
	        decimals = decimals + getFractional(arg);
	        result *= arg;
	      }
	      return decimals !== 0 ? round(result, decimals) : result;
	    }

	    /**
	     * Safely divides two numbers while avoiding floating errors
	     * like 0.3 / 0.05 => 5.999999999999999
	     *
	     * @returns {number} result
	     * @param {number} num Numerator
	     * @param {number} den Denominator
	     */
	    function divSafe(num, den) {
	      if (den === 0) {
	        throw new Error("Divide by zero");
	      }
	      var factor = Math.pow(10, getFractional(den));
	      var invDen = factor / (factor * den);
	      return mulSafe(num, invDen);
	    }

	    /**
	     * Rounds value at the specified number of decimals
	     *
	     * @param {number} val - value to round
	     * @param {number} decimals - number of decimals
	     *
	     * @returns {number} rounded number
	     */
	    function round(val, decimals) {
	      return Math.round(val * Math.pow(10, decimals)) / Math.pow(10, decimals);
	    }
	    function getFractional(num) {
	      // Check for NaNs or Infinities
	      if (!isFinite(num)) {
	        return 0;
	      }

	      // Faster than parsing strings
	      // http://jsperf.com/count-decimals/2
	      var count = 0;
	      while (num % 1 !== 0) {
	        num *= 10;
	        count++;
	      }
	      return count;
	    }

	    /**
	     * Custom error type definition
	     * @constructor
	     */
	    function QtyError() {
	      var err;
	      if (!this) {
	        // Allows to instantiate QtyError without new()
	        err = Object.create(QtyError.prototype);
	        QtyError.apply(err, arguments);
	        return err;
	      }
	      err = Error.apply(this, arguments);
	      this.name = "QtyError";
	      this.message = err.message;
	      this.stack = err.stack;
	    }
	    QtyError.prototype = Object.create(Error.prototype, {
	      constructor: {
	        value: QtyError
	      }
	    });

	    /*
	     * Throws incompatible units error
	     * @param {string} left - units
	     * @param {string} right - units incompatible with first argument
	     * @throws "Incompatible units" error
	     */
	    function throwIncompatibleUnits(left, right) {
	      throw new QtyError("Incompatible units: " + left + " and " + right);
	    }
	    var UNITS = {
	      /* prefixes */
	      "<googol>": [["googol"], 1e100, "prefix"],
	      "<kibi>": [["Ki", "Kibi", "kibi"], Math.pow(2, 10), "prefix"],
	      "<mebi>": [["Mi", "Mebi", "mebi"], Math.pow(2, 20), "prefix"],
	      "<gibi>": [["Gi", "Gibi", "gibi"], Math.pow(2, 30), "prefix"],
	      "<tebi>": [["Ti", "Tebi", "tebi"], Math.pow(2, 40), "prefix"],
	      "<pebi>": [["Pi", "Pebi", "pebi"], Math.pow(2, 50), "prefix"],
	      "<exi>": [["Ei", "Exi", "exi"], Math.pow(2, 60), "prefix"],
	      "<zebi>": [["Zi", "Zebi", "zebi"], Math.pow(2, 70), "prefix"],
	      "<yebi>": [["Yi", "Yebi", "yebi"], Math.pow(2, 80), "prefix"],
	      "<yotta>": [["Y", "Yotta", "yotta"], 1e24, "prefix"],
	      "<zetta>": [["Z", "Zetta", "zetta"], 1e21, "prefix"],
	      "<exa>": [["E", "Exa", "exa"], 1e18, "prefix"],
	      "<peta>": [["P", "Peta", "peta"], 1e15, "prefix"],
	      "<tera>": [["T", "Tera", "tera"], 1e12, "prefix"],
	      "<giga>": [["G", "Giga", "giga"], 1e9, "prefix"],
	      "<mega>": [["M", "Mega", "mega"], 1e6, "prefix"],
	      "<kilo>": [["k", "kilo"], 1e3, "prefix"],
	      "<hecto>": [["h", "Hecto", "hecto"], 1e2, "prefix"],
	      "<deca>": [["da", "Deca", "deca", "deka"], 1e1, "prefix"],
	      "<deci>": [["d", "Deci", "deci"], 1e-1, "prefix"],
	      "<centi>": [["c", "Centi", "centi"], 1e-2, "prefix"],
	      "<milli>": [["m", "Milli", "milli"], 1e-3, "prefix"],
	      "<micro>": [["u", "\u03BC" /*µ as greek letter*/, "\u00B5" /*µ as micro sign*/, "Micro", "mc", "micro"], 1e-6, "prefix"],
	      "<nano>": [["n", "Nano", "nano"], 1e-9, "prefix"],
	      "<pico>": [["p", "Pico", "pico"], 1e-12, "prefix"],
	      "<femto>": [["f", "Femto", "femto"], 1e-15, "prefix"],
	      "<atto>": [["a", "Atto", "atto"], 1e-18, "prefix"],
	      "<zepto>": [["z", "Zepto", "zepto"], 1e-21, "prefix"],
	      "<yocto>": [["y", "Yocto", "yocto"], 1e-24, "prefix"],
	      "<1>": [["1", "<1>"], 1, ""],
	      /* length units */
	      "<meter>": [["m", "meter", "meters", "metre", "metres"], 1.0, "length", ["<meter>"]],
	      "<inch>": [["in", "inch", "inches", "\""], 0.0254, "length", ["<meter>"]],
	      "<foot>": [["ft", "foot", "feet", "'"], 0.3048, "length", ["<meter>"]],
	      "<yard>": [["yd", "yard", "yards"], 0.9144, "length", ["<meter>"]],
	      "<mile>": [["mi", "mile", "miles"], 1609.344, "length", ["<meter>"]],
	      "<naut-mile>": [["nmi", "naut-mile"], 1852, "length", ["<meter>"]],
	      "<league>": [["league", "leagues"], 4828, "length", ["<meter>"]],
	      "<furlong>": [["furlong", "furlongs"], 201.2, "length", ["<meter>"]],
	      "<rod>": [["rd", "rod", "rods"], 5.029, "length", ["<meter>"]],
	      "<mil>": [["mil", "mils"], 0.0000254, "length", ["<meter>"]],
	      "<angstrom>": [["ang", "angstrom", "angstroms"], 1e-10, "length", ["<meter>"]],
	      "<fathom>": [["fathom", "fathoms"], 1.829, "length", ["<meter>"]],
	      "<pica>": [["pica", "picas"], 0.00423333333, "length", ["<meter>"]],
	      "<point>": [["pt", "point", "points"], 0.000352777778, "length", ["<meter>"]],
	      "<redshift>": [["z", "red-shift", "redshift"], 1.302773e26, "length", ["<meter>"]],
	      "<AU>": [["AU", "astronomical-unit"], 149597900000, "length", ["<meter>"]],
	      "<light-second>": [["ls", "light-second"], 299792500, "length", ["<meter>"]],
	      "<light-minute>": [["lmin", "light-minute"], 17987550000, "length", ["<meter>"]],
	      "<light-year>": [["ly", "light-year"], 9460528000000000, "length", ["<meter>"]],
	      "<parsec>": [["pc", "parsec", "parsecs"], 30856780000000000, "length", ["<meter>"]],
	      "<datamile>": [["DM", "datamile"], 1828.8, "length", ["<meter>"]],
	      /* mass */
	      "<kilogram>": [["kg", "kilogram", "kilograms"], 1.0, "mass", ["<kilogram>"]],
	      "<AMU>": [["u", "AMU", "amu"], 1.660538921e-27, "mass", ["<kilogram>"]],
	      "<dalton>": [["Da", "Dalton", "Daltons", "dalton", "daltons"], 1.660538921e-27, "mass", ["<kilogram>"]],
	      "<slug>": [["slug", "slugs"], 14.5939029, "mass", ["<kilogram>"]],
	      "<short-ton>": [["tn", "ton", "short-ton"], 907.18474, "mass", ["<kilogram>"]],
	      "<metric-ton>": [["t", "tonne", "metric-ton"], 1000, "mass", ["<kilogram>"]],
	      "<carat>": [["ct", "carat", "carats"], 0.0002, "mass", ["<kilogram>"]],
	      "<pound>": [["lbs", "lb", "pound", "pounds", "#"], 0.45359237, "mass", ["<kilogram>"]],
	      "<ounce>": [["oz", "ounce", "ounces"], 0.0283495231, "mass", ["<kilogram>"]],
	      "<gram>": [["g", "gram", "grams", "gramme", "grammes"], 1e-3, "mass", ["<kilogram>"]],
	      "<grain>": [["grain", "grains", "gr"], 6.479891e-5, "mass", ["<kilogram>"]],
	      "<dram>": [["dram", "drams", "dr"], 0.0017718452, "mass", ["<kilogram>"]],
	      "<stone>": [["stone", "stones", "st"], 6.35029318, "mass", ["<kilogram>"]],
	      /* area */
	      "<hectare>": [["hectare"], 10000, "area", ["<meter>", "<meter>"]],
	      "<acre>": [["acre", "acres"], 4046.85642, "area", ["<meter>", "<meter>"]],
	      "<sqft>": [["sqft"], 1, "area", ["<foot>", "<foot>"]],
	      /* volume */
	      "<liter>": [["l", "L", "liter", "liters", "litre", "litres"], 0.001, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<gallon>": [["gal", "gallon", "gallons"], 0.0037854118, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<gallon-imp>": [["galimp", "gallon-imp", "gallons-imp"], 0.0045460900, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<quart>": [["qt", "quart", "quarts"], 0.00094635295, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<pint>": [["pt", "pint", "pints"], 0.000473176475, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<pint-imp>": [["ptimp", "pint-imp", "pints-imp"], 5.6826125e-4, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<cup>": [["cu", "cup", "cups"], 0.000236588238, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<fluid-ounce>": [["floz", "fluid-ounce", "fluid-ounces"], 2.95735297e-5, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<fluid-ounce-imp>": [["flozimp", "floz-imp", "fluid-ounce-imp", "fluid-ounces-imp"], 2.84130625e-5, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<tablespoon>": [["tb", "tbsp", "tbs", "tablespoon", "tablespoons"], 1.47867648e-5, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<teaspoon>": [["tsp", "teaspoon", "teaspoons"], 4.92892161e-6, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<bushel>": [["bu", "bsh", "bushel", "bushels"], 0.035239072, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<oilbarrel>": [["bbl", "oilbarrel", "oilbarrels", "oil-barrel", "oil-barrels"], 0.158987294928, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<beerbarrel>": [["bl", "bl-us", "beerbarrel", "beerbarrels", "beer-barrel", "beer-barrels"], 0.1173477658, "volume", ["<meter>", "<meter>", "<meter>"]],
	      "<beerbarrel-imp>": [["blimp", "bl-imp", "beerbarrel-imp", "beerbarrels-imp", "beer-barrel-imp", "beer-barrels-imp"], 0.16365924, "volume", ["<meter>", "<meter>", "<meter>"]],
	      /* speed */
	      "<kph>": [["kph"], 0.277777778, "speed", ["<meter>"], ["<second>"]],
	      "<mph>": [["mph"], 0.44704, "speed", ["<meter>"], ["<second>"]],
	      "<knot>": [["kt", "kn", "kts", "knot", "knots"], 0.514444444, "speed", ["<meter>"], ["<second>"]],
	      "<fps>": [["fps"], 0.3048, "speed", ["<meter>"], ["<second>"]],
	      /* acceleration */
	      "<gee>": [["gee"], 9.80665, "acceleration", ["<meter>"], ["<second>", "<second>"]],
	      "<Gal>": [["Gal"], 1e-2, "acceleration", ["<meter>"], ["<second>", "<second>"]],
	      /* temperature_difference */
	      "<kelvin>": [["degK", "kelvin"], 1.0, "temperature", ["<kelvin>"]],
	      "<celsius>": [["degC", "celsius", "celsius", "centigrade"], 1.0, "temperature", ["<kelvin>"]],
	      "<fahrenheit>": [["degF", "fahrenheit"], 5 / 9, "temperature", ["<kelvin>"]],
	      "<rankine>": [["degR", "rankine"], 5 / 9, "temperature", ["<kelvin>"]],
	      "<temp-K>": [["tempK", "temp-K"], 1.0, "temperature", ["<temp-K>"]],
	      "<temp-C>": [["tempC", "temp-C"], 1.0, "temperature", ["<temp-K>"]],
	      "<temp-F>": [["tempF", "temp-F"], 5 / 9, "temperature", ["<temp-K>"]],
	      "<temp-R>": [["tempR", "temp-R"], 5 / 9, "temperature", ["<temp-K>"]],
	      /* time */
	      "<second>": [["s", "sec", "secs", "second", "seconds"], 1.0, "time", ["<second>"]],
	      "<minute>": [["min", "mins", "minute", "minutes"], 60.0, "time", ["<second>"]],
	      "<hour>": [["h", "hr", "hrs", "hour", "hours"], 3600.0, "time", ["<second>"]],
	      "<day>": [["d", "day", "days"], 3600 * 24, "time", ["<second>"]],
	      "<week>": [["wk", "week", "weeks"], 7 * 3600 * 24, "time", ["<second>"]],
	      "<fortnight>": [["fortnight", "fortnights"], 1209600, "time", ["<second>"]],
	      "<year>": [["y", "yr", "year", "years", "annum"], 31556926, "time", ["<second>"]],
	      "<decade>": [["decade", "decades"], 315569260, "time", ["<second>"]],
	      "<century>": [["century", "centuries"], 3155692600, "time", ["<second>"]],
	      /* pressure */
	      "<pascal>": [["Pa", "pascal", "Pascal"], 1.0, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<bar>": [["bar", "bars"], 100000, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<mmHg>": [["mmHg"], 133.322368, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<inHg>": [["inHg"], 3386.3881472, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<torr>": [["torr"], 133.322368, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<atm>": [["atm", "ATM", "atmosphere", "atmospheres"], 101325, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<psi>": [["psi"], 6894.76, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<cmh2o>": [["cmH2O", "cmh2o"], 98.0638, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      "<inh2o>": [["inH2O", "inh2o"], 249.082052, "pressure", ["<kilogram>"], ["<meter>", "<second>", "<second>"]],
	      /* viscosity */
	      "<poise>": [["P", "poise"], 0.1, "viscosity", ["<kilogram>"], ["<meter>", "<second>"]],
	      "<stokes>": [["St", "stokes"], 1e-4, "viscosity", ["<meter>", "<meter>"], ["<second>"]],
	      /* substance */
	      "<mole>": [["mol", "mole"], 1.0, "substance", ["<mole>"]],
	      /* molar_concentration */
	      "<molar>": [["M", "molar"], 1000, "molar_concentration", ["<mole>"], ["<meter>", "<meter>", "<meter>"]],
	      "<wtpercent>": [["wt%", "wtpercent"], 10, "molar_concentration", ["<kilogram>"], ["<meter>", "<meter>", "<meter>"]],
	      /* activity */
	      "<katal>": [["kat", "katal", "Katal"], 1.0, "activity", ["<mole>"], ["<second>"]],
	      "<unit>": [["U", "enzUnit", "unit"], 16.667e-16, "activity", ["<mole>"], ["<second>"]],
	      /* capacitance */
	      "<farad>": [["F", "farad", "Farad"], 1.0, "capacitance", ["<second>", "<second>", "<second>", "<second>", "<ampere>", "<ampere>"], ["<meter>", "<meter>", "<kilogram>"]],
	      /* charge */
	      "<coulomb>": [["C", "coulomb", "Coulomb"], 1.0, "charge", ["<ampere>", "<second>"]],
	      "<Ah>": [["Ah"], 3600, "charge", ["<ampere>", "<second>"]],
	      /* current */
	      "<ampere>": [["A", "Ampere", "ampere", "amp", "amps"], 1.0, "current", ["<ampere>"]],
	      /* conductance */
	      "<siemens>": [["S", "Siemens", "siemens"], 1.0, "conductance", ["<second>", "<second>", "<second>", "<ampere>", "<ampere>"], ["<kilogram>", "<meter>", "<meter>"]],
	      /* inductance */
	      "<henry>": [["H", "Henry", "henry"], 1.0, "inductance", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<ampere>", "<ampere>"]],
	      /* potential */
	      "<volt>": [["V", "Volt", "volt", "volts"], 1.0, "potential", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<second>", "<ampere>"]],
	      /* resistance */
	      "<ohm>": [["Ohm", "ohm", "\u03A9" /*Ω as greek letter*/, "\u2126" /*Ω as ohm sign*/], 1.0, "resistance", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<second>", "<ampere>", "<ampere>"]],
	      /* magnetism */
	      "<weber>": [["Wb", "weber", "webers"], 1.0, "magnetism", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<ampere>"]],
	      "<tesla>": [["T", "tesla", "teslas"], 1.0, "magnetism", ["<kilogram>"], ["<second>", "<second>", "<ampere>"]],
	      "<gauss>": [["G", "gauss"], 1e-4, "magnetism", ["<kilogram>"], ["<second>", "<second>", "<ampere>"]],
	      "<maxwell>": [["Mx", "maxwell", "maxwells"], 1e-8, "magnetism", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>", "<ampere>"]],
	      "<oersted>": [["Oe", "oersted", "oersteds"], 250.0 / Math.PI, "magnetism", ["<ampere>"], ["<meter>"]],
	      /* energy */
	      "<joule>": [["J", "joule", "Joule", "joules", "Joules"], 1.0, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      "<erg>": [["erg", "ergs"], 1e-7, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      "<btu>": [["BTU", "btu", "BTUs"], 1055.056, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      "<calorie>": [["cal", "calorie", "calories"], 4.18400, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      "<Calorie>": [["Cal", "Calorie", "Calories"], 4184.00, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      "<therm-US>": [["th", "therm", "therms", "Therm", "therm-US"], 105480400, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      "<Wh>": [["Wh"], 3600, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      "<electronvolt>": [["eV", "electronvolt", "electronvolts"], 1.602176634E-19, "energy", ["<meter>", "<meter>", "<kilogram>"], ["<second>", "<second>"]],
	      /* force */
	      "<newton>": [["N", "Newton", "newton"], 1.0, "force", ["<kilogram>", "<meter>"], ["<second>", "<second>"]],
	      "<dyne>": [["dyn", "dyne"], 1e-5, "force", ["<kilogram>", "<meter>"], ["<second>", "<second>"]],
	      "<pound-force>": [["lbf", "pound-force"], 4.448222, "force", ["<kilogram>", "<meter>"], ["<second>", "<second>"]],
	      /* frequency */
	      "<hertz>": [["Hz", "hertz", "Hertz"], 1.0, "frequency", ["<1>"], ["<second>"]],
	      /* angle */
	      "<radian>": [["rad", "radian", "radians"], 1.0, "angle", ["<radian>"]],
	      "<degree>": [["deg", "degree", "degrees"], Math.PI / 180.0, "angle", ["<radian>"]],
	      "<arcminute>": [["arcmin", "arcminute", "arcminutes"], Math.PI / 10800.0, "angle", ["<radian>"]],
	      "<arcsecond>": [["arcsec", "arcsecond", "arcseconds"], Math.PI / 648000.0, "angle", ["<radian>"]],
	      "<gradian>": [["gon", "grad", "gradian", "grads"], Math.PI / 200.0, "angle", ["<radian>"]],
	      "<steradian>": [["sr", "steradian", "steradians"], 1.0, "solid_angle", ["<steradian>"]],
	      /* rotation */
	      "<rotation>": [["rotation"], 2.0 * Math.PI, "angle", ["<radian>"]],
	      "<rpm>": [["rpm"], 2.0 * Math.PI / 60.0, "angular_velocity", ["<radian>"], ["<second>"]],
	      /* information */
	      "<byte>": [["B", "byte", "bytes"], 1.0, "information", ["<byte>"]],
	      "<bit>": [["b", "bit", "bits"], 0.125, "information", ["<byte>"]],
	      /* information rate */
	      "<Bps>": [["Bps"], 1.0, "information_rate", ["<byte>"], ["<second>"]],
	      "<bps>": [["bps"], 0.125, "information_rate", ["<byte>"], ["<second>"]],
	      /* currency */
	      "<dollar>": [["USD", "dollar"], 1.0, "currency", ["<dollar>"]],
	      "<cents>": [["cents"], 0.01, "currency", ["<dollar>"]],
	      /* luminosity */
	      "<candela>": [["cd", "candela"], 1.0, "luminosity", ["<candela>"]],
	      "<lumen>": [["lm", "lumen"], 1.0, "luminous_power", ["<candela>", "<steradian>"]],
	      "<lux>": [["lux"], 1.0, "illuminance", ["<candela>", "<steradian>"], ["<meter>", "<meter>"]],
	      /* power */
	      "<watt>": [["W", "watt", "watts"], 1.0, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
	      "<volt-ampere>": [["VA", "volt-ampere"], 1.0, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
	      "<volt-ampere-reactive>": [["var", "Var", "VAr", "VAR", "volt-ampere-reactive"], 1.0, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
	      "<horsepower>": [["hp", "horsepower"], 745.699872, "power", ["<kilogram>", "<meter>", "<meter>"], ["<second>", "<second>", "<second>"]],
	      /* radiation */
	      "<gray>": [["Gy", "gray", "grays"], 1.0, "radiation", ["<meter>", "<meter>"], ["<second>", "<second>"]],
	      "<roentgen>": [["R", "roentgen"], 0.009330, "radiation", ["<meter>", "<meter>"], ["<second>", "<second>"]],
	      "<sievert>": [["Sv", "sievert", "sieverts"], 1.0, "radiation", ["<meter>", "<meter>"], ["<second>", "<second>"]],
	      "<becquerel>": [["Bq", "becquerel", "becquerels"], 1.0, "radiation", ["<1>"], ["<second>"]],
	      "<curie>": [["Ci", "curie", "curies"], 3.7e10, "radiation", ["<1>"], ["<second>"]],
	      /* rate */
	      "<cpm>": [["cpm"], 1.0 / 60.0, "rate", ["<count>"], ["<second>"]],
	      "<dpm>": [["dpm"], 1.0 / 60.0, "rate", ["<count>"], ["<second>"]],
	      "<bpm>": [["bpm"], 1.0 / 60.0, "rate", ["<count>"], ["<second>"]],
	      /* resolution / typography */
	      "<dot>": [["dot", "dots"], 1, "resolution", ["<each>"]],
	      "<pixel>": [["pixel", "px"], 1, "resolution", ["<each>"]],
	      "<ppi>": [["ppi"], 1, "resolution", ["<pixel>"], ["<inch>"]],
	      "<dpi>": [["dpi"], 1, "typography", ["<dot>"], ["<inch>"]],
	      /* other */
	      "<cell>": [["cells", "cell"], 1, "counting", ["<each>"]],
	      "<each>": [["each"], 1.0, "counting", ["<each>"]],
	      "<count>": [["count"], 1.0, "counting", ["<each>"]],
	      "<base-pair>": [["bp", "base-pair"], 1.0, "counting", ["<each>"]],
	      "<nucleotide>": [["nt", "nucleotide"], 1.0, "counting", ["<each>"]],
	      "<molecule>": [["molecule", "molecules"], 1.0, "counting", ["<1>"]],
	      "<dozen>": [["doz", "dz", "dozen"], 12.0, "prefix_only", ["<each>"]],
	      "<percent>": [["%", "percent"], 0.01, "prefix_only", ["<1>"]],
	      "<ppm>": [["ppm"], 1e-6, "prefix_only", ["<1>"]],
	      "<ppb>": [["ppb"], 1e-9, "prefix_only", ["<1>"]],
	      "<ppt>": [["ppt"], 1e-12, "prefix_only", ["<1>"]],
	      "<ppq>": [["ppq"], 1e-15, "prefix_only", ["<1>"]],
	      "<gross>": [["gr", "gross"], 144.0, "prefix_only", ["<dozen>", "<dozen>"]],
	      "<decibel>": [["dB", "decibel", "decibels"], 1.0, "logarithmic", ["<decibel>"]]
	    };
	    var BASE_UNITS = ["<meter>", "<kilogram>", "<second>", "<mole>", "<ampere>", "<radian>", "<kelvin>", "<temp-K>", "<byte>", "<dollar>", "<candela>", "<each>", "<steradian>", "<decibel>"];
	    var UNITY = "<1>";
	    var UNITY_ARRAY = [UNITY];

	    // Setup

	    /**
	     * Asserts unit definition is valid
	     *
	     * @param {string} unitDef - Name of unit to test
	     * @param {Object} definition - Definition of unit to test
	     *
	     * @returns {void}
	     * @throws {QtyError} if unit definition is not valid
	     */
	    function validateUnitDefinition(unitDef, definition) {
	      var scalar = definition[1];
	      var numerator = definition[3] || [];
	      var denominator = definition[4] || [];
	      if (!isNumber(scalar)) {
	        throw new QtyError(unitDef + ": Invalid unit definition. " + "'scalar' must be a number");
	      }
	      numerator.forEach(function (unit) {
	        if (UNITS[unit] === undefined) {
	          throw new QtyError(unitDef + ": Invalid unit definition. " + "Unit " + unit + " in 'numerator' is not recognized");
	        }
	      });
	      denominator.forEach(function (unit) {
	        if (UNITS[unit] === undefined) {
	          throw new QtyError(unitDef + ": Invalid unit definition. " + "Unit " + unit + " in 'denominator' is not recognized");
	        }
	      });
	    }
	    var PREFIX_VALUES = {};
	    var PREFIX_MAP = {};
	    var UNIT_VALUES = {};
	    var UNIT_MAP = {};
	    var OUTPUT_MAP = {};
	    for (var unitDef in UNITS) {
	      if (UNITS.hasOwnProperty(unitDef)) {
	        var definition = UNITS[unitDef];
	        if (definition[2] === "prefix") {
	          PREFIX_VALUES[unitDef] = definition[1];
	          for (var i = 0; i < definition[0].length; i++) {
	            PREFIX_MAP[definition[0][i]] = unitDef;
	          }
	        } else {
	          validateUnitDefinition(unitDef, definition);
	          UNIT_VALUES[unitDef] = {
	            scalar: definition[1],
	            numerator: definition[3],
	            denominator: definition[4]
	          };
	          for (var j = 0; j < definition[0].length; j++) {
	            UNIT_MAP[definition[0][j]] = unitDef;
	          }
	        }
	        OUTPUT_MAP[unitDef] = definition[0][0];
	      }
	    }

	    /**
	     * Returns a list of available units of kind
	     *
	     * @param {string} [kind] - kind of units
	     * @returns {array} names of units
	     * @throws {QtyError} if kind is unknown
	     */
	    function getUnits(kind) {
	      var i;
	      var units = [];
	      var unitKeys = Object.keys(UNITS);
	      if (typeof kind === "undefined") {
	        for (i = 0; i < unitKeys.length; i++) {
	          if (["", "prefix"].indexOf(UNITS[unitKeys[i]][2]) === -1) {
	            units.push(unitKeys[i].substr(1, unitKeys[i].length - 2));
	          }
	        }
	      } else if (this.getKinds().indexOf(kind) === -1) {
	        throw new QtyError("Kind not recognized");
	      } else {
	        for (i = 0; i < unitKeys.length; i++) {
	          if (UNITS[unitKeys[i]][2] === kind) {
	            units.push(unitKeys[i].substr(1, unitKeys[i].length - 2));
	          }
	        }
	      }
	      return units.sort(function (a, b) {
	        if (a.toLowerCase() < b.toLowerCase()) {
	          return -1;
	        }
	        if (a.toLowerCase() > b.toLowerCase()) {
	          return 1;
	        }
	        return 0;
	      });
	    }

	    /**
	     * Returns a list of alternative names for a unit
	     *
	     * @param {string} unitName - name of unit
	     * @returns {string[]} aliases for unit
	     * @throws {QtyError} if unit is unknown
	     */
	    function getAliases(unitName) {
	      if (!UNIT_MAP[unitName]) {
	        throw new QtyError("Unit not recognized");
	      }
	      return UNITS[UNIT_MAP[unitName]][0];
	    }
	    var SIGNATURE_VECTOR = ["length", "time", "temperature", "mass", "current", "substance", "luminosity", "currency", "information", "angle"];

	    /*
	    calculates the unit signature id for use in comparing compatible units and simplification
	    the signature is based on a simple classification of units and is based on the following publication
	     Novak, G.S., Jr. "Conversion of units of measurement", IEEE Transactions on Software Engineering,
	    21(8), Aug 1995, pp.651-661
	    doi://10.1109/32.403789
	    http://ieeexplore.ieee.org/Xplore/login.jsp?url=/iel1/32/9079/00403789.pdf?isnumber=9079&prod=JNL&arnumber=403789&arSt=651&ared=661&arAuthor=Novak%2C+G.S.%2C+Jr.
	    */
	    function unitSignature() {
	      if (this.signature) {
	        return this.signature;
	      }
	      var vector = unitSignatureVector.call(this);
	      for (var i = 0; i < vector.length; i++) {
	        vector[i] *= Math.pow(20, i);
	      }
	      return vector.reduce(function (previous, current) {
	        return previous + current;
	      }, 0);
	    }

	    // calculates the unit signature vector used by unit_signature
	    function unitSignatureVector() {
	      if (!this.isBase()) {
	        return unitSignatureVector.call(this.toBase());
	      }
	      var vector = new Array(SIGNATURE_VECTOR.length);
	      for (var i = 0; i < vector.length; i++) {
	        vector[i] = 0;
	      }
	      var r, n;
	      for (var j = 0; j < this.numerator.length; j++) {
	        if (r = UNITS[this.numerator[j]]) {
	          n = SIGNATURE_VECTOR.indexOf(r[2]);
	          if (n >= 0) {
	            vector[n] = vector[n] + 1;
	          }
	        }
	      }
	      for (var k = 0; k < this.denominator.length; k++) {
	        if (r = UNITS[this.denominator[k]]) {
	          n = SIGNATURE_VECTOR.indexOf(r[2]);
	          if (n >= 0) {
	            vector[n] = vector[n] - 1;
	          }
	        }
	      }
	      return vector;
	    }
	    var SIGN = "[+-]";
	    var INTEGER = "\\d+";
	    var SIGNED_INTEGER = SIGN + "?" + INTEGER;
	    var FRACTION = "\\." + INTEGER;
	    var FLOAT = "(?:" + INTEGER + "(?:" + FRACTION + ")?" + ")" + "|" + "(?:" + FRACTION + ")";
	    var EXPONENT = "[Ee]" + SIGNED_INTEGER;
	    var SCI_NUMBER = "(?:" + FLOAT + ")(?:" + EXPONENT + ")?";
	    var SIGNED_NUMBER = SIGN + "?\\s*" + SCI_NUMBER;
	    var QTY_STRING = "(" + SIGNED_NUMBER + ")?" + "\\s*([^/]*)(?:\/(.+))?";
	    var QTY_STRING_REGEX = new RegExp("^" + QTY_STRING + "$");
	    var POWER_OP = "\\^|\\*{2}";
	    // Allow unit powers representing scalar, length, area, volume; 4 is for some
	    // special case representations in SI base units.
	    var SAFE_POWER = "[01234]";
	    var TOP_REGEX = new RegExp("([^ \\*\\d]+?)(?:" + POWER_OP + ")?(-?" + SAFE_POWER + "(?![a-zA-Z]))");
	    var BOTTOM_REGEX = new RegExp("([^ \\*\\d]+?)(?:" + POWER_OP + ")?(" + SAFE_POWER + "(?![a-zA-Z]))");

	    /* parse a string into a unit object.
	     * Typical formats like :
	     * "5.6 kg*m/s^2"
	     * "5.6 kg*m*s^-2"
	     * "5.6 kilogram*meter*second^-2"
	     * "2.2 kPa"
	     * "37 degC"
	     * "1"  -- creates a unitless constant with value 1
	     * "GPa"  -- creates a unit with scalar 1 with units 'GPa'
	     * 6'4"  -- recognized as 6 feet + 4 inches
	     * 8 lbs 8 oz -- recognized as 8 lbs + 8 ounces
	     */
	    function parse(val) {
	      if (!isString(val)) {
	        val = val.toString();
	      }
	      val = val.trim();
	      var result = QTY_STRING_REGEX.exec(val);
	      if (!result) {
	        throw new QtyError(val + ": Quantity not recognized");
	      }
	      var scalarMatch = result[1];
	      if (scalarMatch) {
	        // Allow whitespaces between sign and scalar for loose parsing
	        scalarMatch = scalarMatch.replace(/\s/g, "");
	        this.scalar = parseFloat(scalarMatch);
	      } else {
	        this.scalar = 1;
	      }
	      var top = result[2];
	      var bottom = result[3];
	      var n, x, nx;
	      // TODO DRY me
	      while (result = TOP_REGEX.exec(top)) {
	        n = parseFloat(result[2]);
	        if (isNaN(n)) {
	          // Prevents infinite loops
	          throw new QtyError("Unit exponent is not a number");
	        }
	        // Disallow unrecognized unit even if exponent is 0
	        if (n === 0 && !UNIT_TEST_REGEX.test(result[1])) {
	          throw new QtyError("Unit not recognized");
	        }
	        x = result[1] + " ";
	        nx = "";
	        for (var i = 0; i < Math.abs(n); i++) {
	          nx += x;
	        }
	        if (n >= 0) {
	          top = top.replace(result[0], nx);
	        } else {
	          bottom = bottom ? bottom + nx : nx;
	          top = top.replace(result[0], "");
	        }
	      }
	      while (result = BOTTOM_REGEX.exec(bottom)) {
	        n = parseFloat(result[2]);
	        if (isNaN(n)) {
	          // Prevents infinite loops
	          throw new QtyError("Unit exponent is not a number");
	        }
	        // Disallow unrecognized unit even if exponent is 0
	        if (n === 0 && !UNIT_TEST_REGEX.test(result[1])) {
	          throw new QtyError("Unit not recognized");
	        }
	        x = result[1] + " ";
	        nx = "";
	        for (var j = 0; j < n; j++) {
	          nx += x;
	        }
	        bottom = bottom.replace(result[0], nx);
	      }
	      if (top) {
	        this.numerator = parseUnits(top.trim());
	      }
	      if (bottom) {
	        this.denominator = parseUnits(bottom.trim());
	      }
	    }
	    var PREFIX_REGEX = Object.keys(PREFIX_MAP).sort(function (a, b) {
	      return b.length - a.length;
	    }).join("|");
	    var UNIT_REGEX = Object.keys(UNIT_MAP).sort(function (a, b) {
	      return b.length - a.length;
	    }).join("|");
	    /*
	     * Minimal boundary regex to support units with Unicode characters
	     * \b only works for ASCII
	     */
	    var BOUNDARY_REGEX = "\\b|$";
	    var UNIT_MATCH = "(" + PREFIX_REGEX + ")??(" + UNIT_REGEX + ")(?:" + BOUNDARY_REGEX + ")";
	    var UNIT_TEST_REGEX = new RegExp("^\\s*(" + UNIT_MATCH + "[\\s\\*]*)+$");
	    var UNIT_MATCH_REGEX = new RegExp(UNIT_MATCH, "g"); // g flag for multiple occurences
	    var parsedUnitsCache = {};
	    /**
	     * Parses and converts units string to normalized unit array.
	     * Result is cached to speed up next calls.
	     *
	     * @param {string} units Units string
	     * @returns {string[]} Array of normalized units
	     *
	     * @example
	     * // Returns ["<second>", "<meter>", "<second>"]
	     * parseUnits("s m s");
	     *
	     */
	    function parseUnits(units) {
	      var cached = parsedUnitsCache[units];
	      if (cached) {
	        return cached;
	      }
	      var unitMatch,
	        normalizedUnits = [];

	      // Scan
	      if (!UNIT_TEST_REGEX.test(units)) {
	        throw new QtyError("Unit not recognized");
	      }
	      while (unitMatch = UNIT_MATCH_REGEX.exec(units)) {
	        normalizedUnits.push(unitMatch.slice(1));
	      }
	      normalizedUnits = normalizedUnits.map(function (item) {
	        return PREFIX_MAP[item[0]] ? [PREFIX_MAP[item[0]], UNIT_MAP[item[1]]] : [UNIT_MAP[item[1]]];
	      });

	      // Flatten and remove null elements
	      normalizedUnits = normalizedUnits.reduce(function (a, b) {
	        return a.concat(b);
	      }, []);
	      normalizedUnits = normalizedUnits.filter(function (item) {
	        return item;
	      });
	      parsedUnitsCache[units] = normalizedUnits;
	      return normalizedUnits;
	    }

	    /**
	     * Parses a string as a quantity
	     * @param {string} value - quantity as text
	     * @throws if value is not a string
	     * @returns {Qty|null} Parsed quantity or null if unrecognized
	     */
	    function globalParse(value) {
	      if (!isString(value)) {
	        throw new QtyError("Argument should be a string");
	      }
	      try {
	        return this(value);
	      } catch (e) {
	        return null;
	      }
	    }

	    /**
	     * Tests if a value is a Qty instance
	     *
	     * @param {*} value - Value to test
	     *
	     * @returns {boolean} true if value is a Qty instance, false otherwise
	     */
	    function isQty(value) {
	      return value instanceof Qty;
	    }
	    function Qty(initValue, initUnits) {
	      assertValidConstructorArgs.apply(null, arguments);
	      if (!isQty(this)) {
	        return new Qty(initValue, initUnits);
	      }
	      this.scalar = null;
	      this.baseScalar = null;
	      this.signature = null;
	      this._conversionCache = {};
	      this.numerator = UNITY_ARRAY;
	      this.denominator = UNITY_ARRAY;
	      if (isDefinitionObject(initValue)) {
	        this.scalar = initValue.scalar;
	        this.numerator = initValue.numerator && initValue.numerator.length !== 0 ? initValue.numerator : UNITY_ARRAY;
	        this.denominator = initValue.denominator && initValue.denominator.length !== 0 ? initValue.denominator : UNITY_ARRAY;
	      } else if (initUnits) {
	        parse.call(this, initUnits);
	        this.scalar = initValue;
	      } else {
	        parse.call(this, initValue);
	      }

	      // math with temperatures is very limited
	      if (this.denominator.join("*").indexOf("temp") >= 0) {
	        throw new QtyError("Cannot divide with temperatures");
	      }
	      if (this.numerator.join("*").indexOf("temp") >= 0) {
	        if (this.numerator.length > 1) {
	          throw new QtyError("Cannot multiply by temperatures");
	        }
	        if (!compareArray(this.denominator, UNITY_ARRAY)) {
	          throw new QtyError("Cannot divide with temperatures");
	        }
	      }
	      this.initValue = initValue;
	      updateBaseScalar.call(this);
	      if (this.isTemperature() && this.baseScalar < 0) {
	        throw new QtyError("Temperatures must not be less than absolute zero");
	      }
	    }
	    Qty.prototype = {
	      // Properly set up constructor
	      constructor: Qty
	    };

	    /**
	     * Asserts constructor arguments are valid
	     *
	     * @param {*} value - Value to test
	     * @param {string} [units] - Optional units when value is passed as a number
	     *
	     * @returns {void}
	     * @throws {QtyError} if constructor arguments are invalid
	     */
	    function assertValidConstructorArgs(value, units) {
	      if (units) {
	        if (!(isNumber(value) && isString(units))) {
	          throw new QtyError("Only number accepted as initialization value " + "when units are explicitly provided");
	        }
	      } else {
	        if (!(isString(value) || isNumber(value) || isQty(value) || isDefinitionObject(value))) {
	          throw new QtyError("Only string, number or quantity accepted as " + "single initialization value");
	        }
	      }
	    }

	    /**
	     * Tests if a value is a Qty definition object
	     *
	     * @param {*} value - Value to test
	     *
	     * @returns {boolean} true if value is a definition object, false otherwise
	     */
	    function isDefinitionObject(value) {
	      return value && typeof value === "object" && value.hasOwnProperty("scalar");
	    }
	    function updateBaseScalar() {
	      if (this.baseScalar) {
	        return this.baseScalar;
	      }
	      if (this.isBase()) {
	        this.baseScalar = this.scalar;
	        this.signature = unitSignature.call(this);
	      } else {
	        var base = this.toBase();
	        this.baseScalar = base.scalar;
	        this.signature = base.signature;
	      }
	    }
	    var KINDS = {
	      "-312078": "elastance",
	      "-312058": "resistance",
	      "-312038": "inductance",
	      "-152058": "potential",
	      "-152040": "magnetism",
	      "-152038": "magnetism",
	      "-7997": "specific_volume",
	      "-79": "snap",
	      "-59": "jolt",
	      "-39": "acceleration",
	      "-38": "radiation",
	      "-20": "frequency",
	      "-19": "speed",
	      "-18": "viscosity",
	      "-17": "volumetric_flow",
	      "-1": "wavenumber",
	      "0": "unitless",
	      "1": "length",
	      "2": "area",
	      "3": "volume",
	      "20": "time",
	      "400": "temperature",
	      "7941": "yank",
	      "7942": "power",
	      "7959": "pressure",
	      "7961": "force",
	      "7962": "energy",
	      "7979": "viscosity",
	      "7981": "momentum",
	      "7982": "angular_momentum",
	      "7997": "density",
	      "7998": "area_density",
	      "8000": "mass",
	      "152020": "radiation_exposure",
	      "159999": "magnetism",
	      "160000": "current",
	      "160020": "charge",
	      "312058": "conductance",
	      "312078": "capacitance",
	      "3199980": "activity",
	      "3199997": "molar_concentration",
	      "3200000": "substance",
	      "63999998": "illuminance",
	      "64000000": "luminous_power",
	      "1280000000": "currency",
	      "25599999980": "information_rate",
	      "25600000000": "information",
	      "511999999980": "angular_velocity",
	      "512000000000": "angle"
	    };

	    /**
	     * Returns the list of available well-known kinds of units, e.g.
	     * "radiation" or "length".
	     *
	     * @returns {string[]} names of kinds of units
	     */
	    function getKinds() {
	      return uniq(Object.keys(KINDS).map(function (knownSignature) {
	        return KINDS[knownSignature];
	      }));
	    }
	    Qty.prototype.kind = function () {
	      return KINDS[this.signature.toString()];
	    };
	    assign(Qty.prototype, {
	      isDegrees: function () {
	        // signature may not have been calculated yet
	        return (this.signature === null || this.signature === 400) && this.numerator.length === 1 && compareArray(this.denominator, UNITY_ARRAY) && (this.numerator[0].match(/<temp-[CFRK]>/) || this.numerator[0].match(/<(kelvin|celsius|rankine|fahrenheit)>/));
	      },
	      isTemperature: function () {
	        return this.isDegrees() && this.numerator[0].match(/<temp-[CFRK]>/);
	      }
	    });
	    function subtractTemperatures(lhs, rhs) {
	      var lhsUnits = lhs.units();
	      var rhsConverted = rhs.to(lhsUnits);
	      var dstDegrees = Qty(getDegreeUnits(lhsUnits));
	      return Qty({
	        "scalar": lhs.scalar - rhsConverted.scalar,
	        "numerator": dstDegrees.numerator,
	        "denominator": dstDegrees.denominator
	      });
	    }
	    function subtractTempDegrees(temp, deg) {
	      var tempDegrees = deg.to(getDegreeUnits(temp.units()));
	      return Qty({
	        "scalar": temp.scalar - tempDegrees.scalar,
	        "numerator": temp.numerator,
	        "denominator": temp.denominator
	      });
	    }
	    function addTempDegrees(temp, deg) {
	      var tempDegrees = deg.to(getDegreeUnits(temp.units()));
	      return Qty({
	        "scalar": temp.scalar + tempDegrees.scalar,
	        "numerator": temp.numerator,
	        "denominator": temp.denominator
	      });
	    }
	    function getDegreeUnits(units) {
	      if (units === "tempK") {
	        return "degK";
	      } else if (units === "tempC") {
	        return "degC";
	      } else if (units === "tempF") {
	        return "degF";
	      } else if (units === "tempR") {
	        return "degR";
	      } else {
	        throw new QtyError("Unknown type for temp conversion from: " + units);
	      }
	    }
	    function toDegrees(src, dst) {
	      var srcDegK = toDegK(src);
	      var dstUnits = dst.units();
	      var dstScalar;
	      if (dstUnits === "degK") {
	        dstScalar = srcDegK.scalar;
	      } else if (dstUnits === "degC") {
	        dstScalar = srcDegK.scalar;
	      } else if (dstUnits === "degF") {
	        dstScalar = srcDegK.scalar * 9 / 5;
	      } else if (dstUnits === "degR") {
	        dstScalar = srcDegK.scalar * 9 / 5;
	      } else {
	        throw new QtyError("Unknown type for degree conversion to: " + dstUnits);
	      }
	      return Qty({
	        "scalar": dstScalar,
	        "numerator": dst.numerator,
	        "denominator": dst.denominator
	      });
	    }
	    function toDegK(qty) {
	      var units = qty.units();
	      var q;
	      if (units.match(/(deg)[CFRK]/)) {
	        q = qty.baseScalar;
	      } else if (units === "tempK") {
	        q = qty.scalar;
	      } else if (units === "tempC") {
	        q = qty.scalar;
	      } else if (units === "tempF") {
	        q = qty.scalar * 5 / 9;
	      } else if (units === "tempR") {
	        q = qty.scalar * 5 / 9;
	      } else {
	        throw new QtyError("Unknown type for temp conversion from: " + units);
	      }
	      return Qty({
	        "scalar": q,
	        "numerator": ["<kelvin>"],
	        "denominator": UNITY_ARRAY
	      });
	    }
	    function toTemp(src, dst) {
	      var dstUnits = dst.units();
	      var dstScalar;
	      if (dstUnits === "tempK") {
	        dstScalar = src.baseScalar;
	      } else if (dstUnits === "tempC") {
	        dstScalar = src.baseScalar - 273.15;
	      } else if (dstUnits === "tempF") {
	        dstScalar = src.baseScalar * 9 / 5 - 459.67;
	      } else if (dstUnits === "tempR") {
	        dstScalar = src.baseScalar * 9 / 5;
	      } else {
	        throw new QtyError("Unknown type for temp conversion to: " + dstUnits);
	      }
	      return Qty({
	        "scalar": dstScalar,
	        "numerator": dst.numerator,
	        "denominator": dst.denominator
	      });
	    }
	    function toTempK(qty) {
	      var units = qty.units();
	      var q;
	      if (units.match(/(deg)[CFRK]/)) {
	        q = qty.baseScalar;
	      } else if (units === "tempK") {
	        q = qty.scalar;
	      } else if (units === "tempC") {
	        q = qty.scalar + 273.15;
	      } else if (units === "tempF") {
	        q = (qty.scalar + 459.67) * 5 / 9;
	      } else if (units === "tempR") {
	        q = qty.scalar * 5 / 9;
	      } else {
	        throw new QtyError("Unknown type for temp conversion from: " + units);
	      }
	      return Qty({
	        "scalar": q,
	        "numerator": ["<temp-K>"],
	        "denominator": UNITY_ARRAY
	      });
	    }
	    assign(Qty.prototype, {
	      /**
	       * Converts to other compatible units.
	       * Instance's converted quantities are cached for faster subsequent calls.
	       *
	       * @param {(string|Qty)} other - Target units as string or retrieved from
	       *                               other Qty instance (scalar is ignored)
	       *
	       * @returns {Qty} New converted Qty instance with target units
	       *
	       * @throws {QtyError} if target units are incompatible
	       *
	       * @example
	       * var weight = Qty("25 kg");
	       * weight.to("lb"); // => Qty("55.11556554621939 lbs");
	       * weight.to(Qty("3 g")); // => Qty("25000 g"); // scalar of passed Qty is ignored
	       */
	      to: function (other) {
	        var cached, target;
	        if (other === undefined || other === null) {
	          return this;
	        }
	        if (!isString(other)) {
	          return this.to(other.units());
	        }
	        cached = this._conversionCache[other];
	        if (cached) {
	          return cached;
	        }

	        // Instantiating target to normalize units
	        target = Qty(other);
	        if (target.units() === this.units()) {
	          return this;
	        }
	        if (!this.isCompatible(target)) {
	          if (this.isInverse(target)) {
	            target = this.inverse().to(other);
	          } else {
	            throwIncompatibleUnits(this.units(), target.units());
	          }
	        } else {
	          if (target.isTemperature()) {
	            target = toTemp(this, target);
	          } else if (target.isDegrees()) {
	            target = toDegrees(this, target);
	          } else {
	            var q = divSafe(this.baseScalar, target.baseScalar);
	            target = Qty({
	              "scalar": q,
	              "numerator": target.numerator,
	              "denominator": target.denominator
	            });
	          }
	        }
	        this._conversionCache[other] = target;
	        return target;
	      },
	      // convert to base SI units
	      // results of the conversion are cached so subsequent calls to this will be fast
	      toBase: function () {
	        if (this.isBase()) {
	          return this;
	        }
	        if (this.isTemperature()) {
	          return toTempK(this);
	        }
	        var cached = baseUnitCache[this.units()];
	        if (!cached) {
	          cached = toBaseUnits(this.numerator, this.denominator);
	          baseUnitCache[this.units()] = cached;
	        }
	        return cached.mul(this.scalar);
	      },
	      // Converts the unit back to a float if it is unitless.  Otherwise raises an exception
	      toFloat: function () {
	        if (this.isUnitless()) {
	          return this.scalar;
	        }
	        throw new QtyError("Can't convert to Float unless unitless.  Use Unit#scalar");
	      },
	      /**
	       * Returns the nearest multiple of quantity passed as
	       * precision
	       *
	       * @param {(Qty|string|number)} precQuantity - Quantity, string formated
	       *   quantity or number as expected precision
	       *
	       * @returns {Qty} Nearest multiple of precQuantity
	       *
	       * @example
	       * Qty('5.5 ft').toPrec('2 ft'); // returns 6 ft
	       * Qty('0.8 cu').toPrec('0.25 cu'); // returns 0.75 cu
	       * Qty('6.3782 m').toPrec('cm'); // returns 6.38 m
	       * Qty('1.146 MPa').toPrec('0.1 bar'); // returns 1.15 MPa
	       *
	       */
	      toPrec: function (precQuantity) {
	        if (isString(precQuantity)) {
	          precQuantity = Qty(precQuantity);
	        }
	        if (isNumber(precQuantity)) {
	          precQuantity = Qty(precQuantity + " " + this.units());
	        }
	        if (!this.isUnitless()) {
	          precQuantity = precQuantity.to(this.units());
	        } else if (!precQuantity.isUnitless()) {
	          throwIncompatibleUnits(this.units(), precQuantity.units());
	        }
	        if (precQuantity.scalar === 0) {
	          throw new QtyError("Divide by zero");
	        }
	        var precRoundedResult = mulSafe(Math.round(this.scalar / precQuantity.scalar), precQuantity.scalar);
	        return Qty(precRoundedResult + this.units());
	      }
	    });

	    /**
	     * Configures and returns a fast function to convert
	     * Number values from units to others.
	     * Useful to efficiently convert large array of values
	     * with same units into others with iterative methods.
	     * Does not take care of rounding issues.
	     *
	     * @param {string} srcUnits Units of values to convert
	     * @param {string} dstUnits Units to convert to
	     *
	     * @returns {Function} Converting function accepting Number value
	     *   and returning converted value
	     *
	     * @throws "Incompatible units" if units are incompatible
	     *
	     * @example
	     * // Converting large array of numbers with the same units
	     * // into other units
	     * var converter = Qty.swiftConverter("m/h", "ft/s");
	     * var convertedSerie = largeSerie.map(converter);
	     *
	     */
	    function swiftConverter(srcUnits, dstUnits) {
	      var srcQty = Qty(srcUnits);
	      var dstQty = Qty(dstUnits);
	      if (srcQty.eq(dstQty)) {
	        return identity;
	      }
	      var convert;
	      if (!srcQty.isTemperature()) {
	        convert = function (value) {
	          return value * srcQty.baseScalar / dstQty.baseScalar;
	        };
	      } else {
	        convert = function (value) {
	          // TODO Not optimized
	          return srcQty.mul(value).to(dstQty).scalar;
	        };
	      }
	      return function converter(value) {
	        var i, length, result;
	        if (!Array.isArray(value)) {
	          return convert(value);
	        } else {
	          length = value.length;
	          result = [];
	          for (i = 0; i < length; i++) {
	            result.push(convert(value[i]));
	          }
	          return result;
	        }
	      };
	    }
	    var baseUnitCache = {};
	    function toBaseUnits(numerator, denominator) {
	      var num = [];
	      var den = [];
	      var q = 1;
	      var unit;
	      for (var i = 0; i < numerator.length; i++) {
	        unit = numerator[i];
	        if (PREFIX_VALUES[unit]) {
	          // workaround to fix
	          // 0.1 * 0.1 => 0.010000000000000002
	          q = mulSafe(q, PREFIX_VALUES[unit]);
	        } else {
	          if (UNIT_VALUES[unit]) {
	            q *= UNIT_VALUES[unit].scalar;
	            if (UNIT_VALUES[unit].numerator) {
	              num.push(UNIT_VALUES[unit].numerator);
	            }
	            if (UNIT_VALUES[unit].denominator) {
	              den.push(UNIT_VALUES[unit].denominator);
	            }
	          }
	        }
	      }
	      for (var j = 0; j < denominator.length; j++) {
	        unit = denominator[j];
	        if (PREFIX_VALUES[unit]) {
	          q /= PREFIX_VALUES[unit];
	        } else {
	          if (UNIT_VALUES[unit]) {
	            q /= UNIT_VALUES[unit].scalar;
	            if (UNIT_VALUES[unit].numerator) {
	              den.push(UNIT_VALUES[unit].numerator);
	            }
	            if (UNIT_VALUES[unit].denominator) {
	              num.push(UNIT_VALUES[unit].denominator);
	            }
	          }
	        }
	      }

	      // Flatten
	      num = num.reduce(function (a, b) {
	        return a.concat(b);
	      }, []);
	      den = den.reduce(function (a, b) {
	        return a.concat(b);
	      }, []);
	      return Qty({
	        "scalar": q,
	        "numerator": num,
	        "denominator": den
	      });
	    }
	    Qty.parse = globalParse;
	    Qty.getUnits = getUnits;
	    Qty.getAliases = getAliases;
	    Qty.mulSafe = mulSafe;
	    Qty.divSafe = divSafe;
	    Qty.getKinds = getKinds;
	    Qty.swiftConverter = swiftConverter;
	    Qty.Error = QtyError;
	    assign(Qty.prototype, {
	      // Returns new instance with units of this
	      add: function (other) {
	        if (isString(other)) {
	          other = Qty(other);
	        }
	        if (!this.isCompatible(other)) {
	          throwIncompatibleUnits(this.units(), other.units());
	        }
	        if (this.isTemperature() && other.isTemperature()) {
	          throw new QtyError("Cannot add two temperatures");
	        } else if (this.isTemperature()) {
	          return addTempDegrees(this, other);
	        } else if (other.isTemperature()) {
	          return addTempDegrees(other, this);
	        }
	        return Qty({
	          "scalar": this.scalar + other.to(this).scalar,
	          "numerator": this.numerator,
	          "denominator": this.denominator
	        });
	      },
	      sub: function (other) {
	        if (isString(other)) {
	          other = Qty(other);
	        }
	        if (!this.isCompatible(other)) {
	          throwIncompatibleUnits(this.units(), other.units());
	        }
	        if (this.isTemperature() && other.isTemperature()) {
	          return subtractTemperatures(this, other);
	        } else if (this.isTemperature()) {
	          return subtractTempDegrees(this, other);
	        } else if (other.isTemperature()) {
	          throw new QtyError("Cannot subtract a temperature from a differential degree unit");
	        }
	        return Qty({
	          "scalar": this.scalar - other.to(this).scalar,
	          "numerator": this.numerator,
	          "denominator": this.denominator
	        });
	      },
	      mul: function (other) {
	        if (isNumber(other)) {
	          return Qty({
	            "scalar": mulSafe(this.scalar, other),
	            "numerator": this.numerator,
	            "denominator": this.denominator
	          });
	        } else if (isString(other)) {
	          other = Qty(other);
	        }
	        if ((this.isTemperature() || other.isTemperature()) && !(this.isUnitless() || other.isUnitless())) {
	          throw new QtyError("Cannot multiply by temperatures");
	        }

	        // Quantities should be multiplied with same units if compatible, with base units else
	        var op1 = this;
	        var op2 = other;

	        // so as not to confuse results, multiplication and division between temperature degrees will maintain original unit info in num/den
	        // multiplication and division between deg[CFRK] can never factor each other out, only themselves: "degK*degC/degC^2" == "degK/degC"
	        if (op1.isCompatible(op2) && op1.signature !== 400) {
	          op2 = op2.to(op1);
	        }
	        var numdenscale = cleanTerms(op1.numerator, op1.denominator, op2.numerator, op2.denominator);
	        return Qty({
	          "scalar": mulSafe(op1.scalar, op2.scalar, numdenscale[2]),
	          "numerator": numdenscale[0],
	          "denominator": numdenscale[1]
	        });
	      },
	      div: function (other) {
	        if (isNumber(other)) {
	          if (other === 0) {
	            throw new QtyError("Divide by zero");
	          }
	          return Qty({
	            "scalar": this.scalar / other,
	            "numerator": this.numerator,
	            "denominator": this.denominator
	          });
	        } else if (isString(other)) {
	          other = Qty(other);
	        }
	        if (other.scalar === 0) {
	          throw new QtyError("Divide by zero");
	        }
	        if (other.isTemperature()) {
	          throw new QtyError("Cannot divide with temperatures");
	        } else if (this.isTemperature() && !other.isUnitless()) {
	          throw new QtyError("Cannot divide with temperatures");
	        }

	        // Quantities should be multiplied with same units if compatible, with base units else
	        var op1 = this;
	        var op2 = other;

	        // so as not to confuse results, multiplication and division between temperature degrees will maintain original unit info in num/den
	        // multiplication and division between deg[CFRK] can never factor each other out, only themselves: "degK*degC/degC^2" == "degK/degC"
	        if (op1.isCompatible(op2) && op1.signature !== 400) {
	          op2 = op2.to(op1);
	        }
	        var numdenscale = cleanTerms(op1.numerator, op1.denominator, op2.denominator, op2.numerator);
	        return Qty({
	          "scalar": mulSafe(op1.scalar, numdenscale[2]) / op2.scalar,
	          "numerator": numdenscale[0],
	          "denominator": numdenscale[1]
	        });
	      },
	      // Returns a Qty that is the inverse of this Qty,
	      inverse: function () {
	        if (this.isTemperature()) {
	          throw new QtyError("Cannot divide with temperatures");
	        }
	        if (this.scalar === 0) {
	          throw new QtyError("Divide by zero");
	        }
	        return Qty({
	          "scalar": 1 / this.scalar,
	          "numerator": this.denominator,
	          "denominator": this.numerator
	        });
	      }
	    });
	    function cleanTerms(num1, den1, num2, den2) {
	      function notUnity(val) {
	        return val !== UNITY;
	      }
	      num1 = num1.filter(notUnity);
	      num2 = num2.filter(notUnity);
	      den1 = den1.filter(notUnity);
	      den2 = den2.filter(notUnity);
	      var combined = {};
	      function combineTerms(terms, direction) {
	        var k;
	        var prefix;
	        var prefixValue;
	        for (var i = 0; i < terms.length; i++) {
	          if (PREFIX_VALUES[terms[i]]) {
	            k = terms[i + 1];
	            prefix = terms[i];
	            prefixValue = PREFIX_VALUES[prefix];
	            i++;
	          } else {
	            k = terms[i];
	            prefix = null;
	            prefixValue = 1;
	          }
	          if (k && k !== UNITY) {
	            if (combined[k]) {
	              combined[k][0] += direction;
	              var combinedPrefixValue = combined[k][2] ? PREFIX_VALUES[combined[k][2]] : 1;
	              combined[k][direction === 1 ? 3 : 4] *= divSafe(prefixValue, combinedPrefixValue);
	            } else {
	              combined[k] = [direction, k, prefix, 1, 1];
	            }
	          }
	        }
	      }
	      combineTerms(num1, 1);
	      combineTerms(den1, -1);
	      combineTerms(num2, 1);
	      combineTerms(den2, -1);
	      var num = [];
	      var den = [];
	      var scale = 1;
	      for (var prop in combined) {
	        if (combined.hasOwnProperty(prop)) {
	          var item = combined[prop];
	          var n;
	          if (item[0] > 0) {
	            for (n = 0; n < item[0]; n++) {
	              num.push(item[2] === null ? item[1] : [item[2], item[1]]);
	            }
	          } else if (item[0] < 0) {
	            for (n = 0; n < -item[0]; n++) {
	              den.push(item[2] === null ? item[1] : [item[2], item[1]]);
	            }
	          }
	          scale *= divSafe(item[3], item[4]);
	        }
	      }
	      if (num.length === 0) {
	        num = UNITY_ARRAY;
	      }
	      if (den.length === 0) {
	        den = UNITY_ARRAY;
	      }

	      // Flatten
	      num = num.reduce(function (a, b) {
	        return a.concat(b);
	      }, []);
	      den = den.reduce(function (a, b) {
	        return a.concat(b);
	      }, []);
	      return [num, den, scale];
	    }
	    assign(Qty.prototype, {
	      eq: function (other) {
	        return this.compareTo(other) === 0;
	      },
	      lt: function (other) {
	        return this.compareTo(other) === -1;
	      },
	      lte: function (other) {
	        return this.eq(other) || this.lt(other);
	      },
	      gt: function (other) {
	        return this.compareTo(other) === 1;
	      },
	      gte: function (other) {
	        return this.eq(other) || this.gt(other);
	      },
	      // Compare two Qty objects. Throws an exception if they are not of compatible types.
	      // Comparisons are done based on the value of the quantity in base SI units.
	      //
	      // NOTE: We cannot compare inverses as that breaks the general compareTo contract:
	      //   if a.compareTo(b) < 0 then b.compareTo(a) > 0
	      //   if a.compareTo(b) == 0 then b.compareTo(a) == 0
	      //
	      //   Since "10S" == ".1ohm" (10 > .1) and "10ohm" == ".1S" (10 > .1)
	      //     Qty("10S").inverse().compareTo("10ohm") == -1
	      //     Qty("10ohm").inverse().compareTo("10S") == -1
	      //
	      //   If including inverses in the sort is needed, I suggest writing: Qty.sort(qtyArray,units)
	      compareTo: function (other) {
	        if (isString(other)) {
	          return this.compareTo(Qty(other));
	        }
	        if (!this.isCompatible(other)) {
	          throwIncompatibleUnits(this.units(), other.units());
	        }
	        if (this.baseScalar < other.baseScalar) {
	          return -1;
	        } else if (this.baseScalar === other.baseScalar) {
	          return 0;
	        } else if (this.baseScalar > other.baseScalar) {
	          return 1;
	        }
	      },
	      // Return true if quantities and units match
	      // Unit("100 cm").same(Unit("100 cm"))  # => true
	      // Unit("100 cm").same(Unit("1 m"))     # => false
	      same: function (other) {
	        return this.scalar === other.scalar && this.units() === other.units();
	      }
	    });
	    assign(Qty.prototype, {
	      // returns true if no associated units
	      // false, even if the units are "unitless" like 'radians, each, etc'
	      isUnitless: function () {
	        return [this.numerator, this.denominator].every(function (item) {
	          return compareArray(item, UNITY_ARRAY);
	        });
	      },
	      /*
	      check to see if units are compatible, but not the scalar part
	      this check is done by comparing signatures for performance reasons
	      if passed a string, it will create a unit object with the string and then do the comparison
	      this permits a syntax like:
	      unit =~ "mm"
	      if you want to do a regexp on the unit string do this ...
	      unit.units =~ /regexp/
	      */
	      isCompatible: function (other) {
	        if (isString(other)) {
	          return this.isCompatible(Qty(other));
	        }
	        if (!isQty(other)) {
	          return false;
	        }
	        if (other.signature !== undefined) {
	          return this.signature === other.signature;
	        } else {
	          return false;
	        }
	      },
	      /*
	      check to see if units are inverse of each other, but not the scalar part
	      this check is done by comparing signatures for performance reasons
	      if passed a string, it will create a unit object with the string and then do the comparison
	      this permits a syntax like:
	      unit =~ "mm"
	      if you want to do a regexp on the unit string do this ...
	      unit.units =~ /regexp/
	      */
	      isInverse: function (other) {
	        return this.inverse().isCompatible(other);
	      },
	      // Returns 'true' if the Unit is represented in base units
	      isBase: function () {
	        if (this._isBase !== undefined) {
	          return this._isBase;
	        }
	        if (this.isDegrees() && this.numerator[0].match(/<(kelvin|temp-K)>/)) {
	          this._isBase = true;
	          return this._isBase;
	        }
	        this.numerator.concat(this.denominator).forEach(function (item) {
	          if (item !== UNITY && BASE_UNITS.indexOf(item) === -1) {
	            this._isBase = false;
	          }
	        }, this);
	        if (this._isBase === false) {
	          return this._isBase;
	        }
	        this._isBase = true;
	        return this._isBase;
	      }
	    });
	    function NestedMap() {}
	    NestedMap.prototype.get = function (keys) {
	      // Allows to pass key1, key2, ... instead of [key1, key2, ...]
	      if (arguments.length > 1) {
	        // Slower with Firefox but faster with Chrome than
	        // Array.prototype.slice.call(arguments)
	        // See http://jsperf.com/array-apply-versus-array-prototype-slice-call
	        keys = Array.apply(null, arguments);
	      }
	      return keys.reduce(function (map, key, index) {
	        if (map) {
	          var childMap = map[key];
	          if (index === keys.length - 1) {
	            return childMap ? childMap.data : undefined;
	          } else {
	            return childMap;
	          }
	        }
	      }, this);
	    };
	    NestedMap.prototype.set = function (keys, value) {
	      if (arguments.length > 2) {
	        keys = Array.prototype.slice.call(arguments, 0, -1);
	        value = arguments[arguments.length - 1];
	      }
	      return keys.reduce(function (map, key, index) {
	        var childMap = map[key];
	        if (childMap === undefined) {
	          childMap = map[key] = {};
	        }
	        if (index === keys.length - 1) {
	          childMap.data = value;
	          return value;
	        } else {
	          return childMap;
	        }
	      }, this);
	    };

	    /**
	     * Default formatter
	     *
	     * @param {number} scalar - scalar value
	     * @param {string} units - units as string
	     *
	     * @returns {string} formatted result
	     */
	    function defaultFormatter(scalar, units) {
	      return (scalar + " " + units).trim();
	    }

	    /**
	     *
	     * Configurable Qty default formatter
	     *
	     * @type {function}
	     *
	     * @param {number} scalar
	     * @param {string} units
	     *
	     * @returns {string} formatted result
	     */
	    Qty.formatter = defaultFormatter;
	    assign(Qty.prototype, {
	      // returns the 'unit' part of the Unit object without the scalar
	      units: function () {
	        if (this._units !== undefined) {
	          return this._units;
	        }
	        var numIsUnity = compareArray(this.numerator, UNITY_ARRAY);
	        var denIsUnity = compareArray(this.denominator, UNITY_ARRAY);
	        if (numIsUnity && denIsUnity) {
	          this._units = "";
	          return this._units;
	        }
	        var numUnits = stringifyUnits(this.numerator);
	        var denUnits = stringifyUnits(this.denominator);
	        this._units = numUnits + (denIsUnity ? "" : "/" + denUnits);
	        return this._units;
	      },
	      /**
	       * Stringifies the quantity
	       * Deprecation notice: only units parameter is supported.
	       *
	       * @param {(number|string|Qty)} targetUnitsOrMaxDecimalsOrPrec -
	       *                              target units if string,
	       *                              max number of decimals if number,
	       *                              passed to #toPrec before converting if Qty
	       *
	       * @param {number=} maxDecimals - Maximum number of decimals of
	       *                                formatted output
	       *
	       * @returns {string} reparseable quantity as string
	       */
	      toString: function (targetUnitsOrMaxDecimalsOrPrec, maxDecimals) {
	        var targetUnits;
	        if (isNumber(targetUnitsOrMaxDecimalsOrPrec)) {
	          targetUnits = this.units();
	          maxDecimals = targetUnitsOrMaxDecimalsOrPrec;
	        } else if (isString(targetUnitsOrMaxDecimalsOrPrec)) {
	          targetUnits = targetUnitsOrMaxDecimalsOrPrec;
	        } else if (isQty(targetUnitsOrMaxDecimalsOrPrec)) {
	          return this.toPrec(targetUnitsOrMaxDecimalsOrPrec).toString(maxDecimals);
	        }
	        var out = this.to(targetUnits);
	        var outScalar = maxDecimals !== undefined ? round(out.scalar, maxDecimals) : out.scalar;
	        out = (outScalar + " " + out.units()).trim();
	        return out;
	      },
	      /**
	       * Format the quantity according to optional passed target units
	       * and formatter
	       *
	       * @param {string} [targetUnits=current units] -
	       *                 optional units to convert to before formatting
	       *
	       * @param {function} [formatter=Qty.formatter] -
	       *                   delegates formatting to formatter callback.
	       *                   formatter is called back with two parameters (scalar, units)
	       *                   and should return formatted result.
	       *                   If unspecified, formatting is delegated to default formatter
	       *                   set to Qty.formatter
	       *
	       * @example
	       * var roundingAndLocalizingFormatter = function(scalar, units) {
	       *   // localize or limit scalar to n max decimals for instance
	       *   // return formatted result
	       * };
	       * var qty = Qty('1.1234 m');
	       * qty.format(); // same units, default formatter => "1.234 m"
	       * qty.format("cm"); // converted to "cm", default formatter => "123.45 cm"
	       * qty.format(roundingAndLocalizingFormatter); // same units, custom formatter => "1,2 m"
	       * qty.format("cm", roundingAndLocalizingFormatter); // convert to "cm", custom formatter => "123,4 cm"
	       *
	       * @returns {string} quantity as string
	       */
	      format: function (targetUnits, formatter) {
	        if (arguments.length === 1) {
	          if (typeof targetUnits === "function") {
	            formatter = targetUnits;
	            targetUnits = undefined;
	          }
	        }
	        formatter = formatter || Qty.formatter;
	        var targetQty = this.to(targetUnits);
	        return formatter.call(this, targetQty.scalar, targetQty.units());
	      }
	    });
	    var stringifiedUnitsCache = new NestedMap();
	    /**
	     * Returns a string representing a normalized unit array
	     *
	     * @param {string[]} units Normalized unit array
	     * @returns {string} String representing passed normalized unit array and
	     *   suitable for output
	     *
	     */
	    function stringifyUnits(units) {
	      var stringified = stringifiedUnitsCache.get(units);
	      if (stringified) {
	        return stringified;
	      }
	      var isUnity = compareArray(units, UNITY_ARRAY);
	      if (isUnity) {
	        stringified = "1";
	      } else {
	        stringified = simplify(getOutputNames(units)).join("*");
	      }

	      // Cache result
	      stringifiedUnitsCache.set(units, stringified);
	      return stringified;
	    }
	    function getOutputNames(units) {
	      var unitNames = [],
	        token,
	        tokenNext;
	      for (var i = 0; i < units.length; i++) {
	        token = units[i];
	        tokenNext = units[i + 1];
	        if (PREFIX_VALUES[token]) {
	          unitNames.push(OUTPUT_MAP[token] + OUTPUT_MAP[tokenNext]);
	          i++;
	        } else {
	          unitNames.push(OUTPUT_MAP[token]);
	        }
	      }
	      return unitNames;
	    }
	    function simplify(units) {
	      // this turns ['s','m','s'] into ['s2','m']

	      var unitCounts = units.reduce(function (acc, unit) {
	        var unitCounter = acc[unit];
	        if (!unitCounter) {
	          acc.push(unitCounter = acc[unit] = [unit, 0]);
	        }
	        unitCounter[1]++;
	        return acc;
	      }, []);
	      return unitCounts.map(function (unitCount) {
	        return unitCount[0] + (unitCount[1] > 1 ? unitCount[1] : "");
	      });
	    }
	    Qty.version = "1.8.0";
	    return Qty;
	  });
	})(quantities);
	var Qty = quantities.exports;

	function convertUnit(array, fromUnit, toUnit) {
	  fromUnit = normalize(fromUnit);
	  toUnit = normalize(toUnit);
	  if (fromUnit === toUnit) return array;
	  const convert = Qty.swiftConverter(fromUnit, toUnit); // Configures converter
	  //@ts-expect-error convert does not allowed typed array but it works
	  return convert(array);
	}
	function normalize(unit) {
	  unit = unit.replace(/°C/g, 'tempC');
	  unit = unit.replace(/°F/g, 'tempF');
	  unit = unit.replace(/(^|\W)K(\W|$)/g, '$1tempK$2');
	  return unit;
	}

	const testRegExp = /^\/((?:\\\/|[^/])+)\/([migyu]{0,5})?$/;
	function ensureRegexp(string) {
	  if (typeof string !== 'string') return string;
	  const parts = testRegExp.exec(string);
	  if (parts) {
	    try {
	      return new RegExp(parts[1], parts[2]);
	    } catch (err) {
	      return stringToRegexp(string);
	    }
	  } else {
	    return stringToRegexp(string);
	  }
	}
	function stringToRegexp(string, flags = 'i') {
	  return new RegExp(string.replace(/[[\]\\{}()+*?.$^|]/g, match => `\\${match}`), flags);
	}

	function getConvertedVariable(variable, newUnits) {
	  const data = variable.units !== undefined && variable.units !== newUnits // would be nice if convertUnit would allow typedArray
	  ? convertUnit(Array.from(variable.data), variable.units, newUnits) : variable.data;
	  return {
	    units: newUnits,
	    label: variable.label.replace(`[${variable.units || ''}]`, `[${newUnits}]`),
	    data: data || [],
	    min: data ? xMinValue(data) : undefined,
	    max: data ? xMaxValue(data) : undefined,
	    isMonotone: xIsMonotonic(data) !== 0
	  };
	}

	/* eslint-disable @typescript-eslint/no-dynamic-delete */
	/**
	 * Retrieve the spectrum with only X/Y data that match all the selectors
	 * If more than one variable match the selector the 'x' or 'y' variable will be
	 * taken
	 */
	function getXYSpectra(spectra = [], selector = {}) {
	  const selectedSpectra = [];
	  if (spectra.length < 1) return selectedSpectra;
	  const {
	    variables,
	    units,
	    labels,
	    meta,
	    index
	  } = selector;
	  let {
	    dataType,
	    title,
	    xUnits,
	    yUnits,
	    xVariable = 'x',
	    yVariable = 'y',
	    xLabel,
	    yLabel
	  } = selector;
	  if (index !== undefined) {
	    return [spectra[index]];
	  }
	  if (dataType) {
	    dataType = ensureRegexp(dataType);
	  }
	  if (title) {
	    title = ensureRegexp(title);
	  }
	  if (units && !xUnits && !yUnits) [yUnits, xUnits] = units.split(/\s*vs\s*/);
	  if (labels && !xLabel && !yLabel) {
	    [yLabel, xLabel] = labels.split(/\s*vs\s*/);
	  }
	  if (variables) {
	    const parts = variables.split(/\s*vs\s*/);
	    if (parts.length === 2) {
	      xVariable = parts[1];
	      yVariable = parts[0];
	    }
	  }
	  if (xLabel) xLabel = ensureRegexp(xLabel);
	  if (yLabel) yLabel = ensureRegexp(yLabel);
	  for (const spectrum of spectra) {
	    const variableNames = Object.keys(spectrum.variables);
	    if (!(variableNames.length > 1)) continue;
	    // we filter on general spectrum information
	    if (dataType) {
	      if (!spectrum.dataType || !dataType.exec(spectrum.dataType)) {
	        continue;
	      }
	    }
	    if (title) {
	      if (!spectrum.title || !title.exec(spectrum.title)) {
	        continue;
	      }
	    }
	    if (meta && typeof meta === 'object') {
	      if (!spectrum.meta) continue;
	      for (const key in spectrum.meta) {
	        if (!spectrum.meta[key]) continue;
	        const value = ensureRegexp(spectrum.meta[key]);
	        if (!value.exec(spectrum.meta[key])) continue;
	      }
	    }
	    const x = getPossibleVariable(spectrum.variables, {
	      units: xUnits,
	      label: xLabel,
	      variableName: xVariable
	    });
	    const y = getPossibleVariable(spectrum.variables, {
	      units: yUnits,
	      label: yLabel,
	      variableName: yVariable
	    });
	    if (x && y) {
	      selectedSpectra.push({
	        title: spectrum.title,
	        dataType: spectrum.dataType,
	        meta: spectrum.meta,
	        variables: {
	          x,
	          y
	        },
	        id: spectrum.id
	      });
	    }
	  }
	  return selectedSpectra;
	}
	function getPossibleVariable(variables, selector = {}) {
	  const {
	    units,
	    label,
	    variableName
	  } = selector;
	  const possible = {
	    ...variables
	  };
	  let key;
	  if (units !== undefined) {
	    for (key in possible) {
	      const variable = variables[key];
	      let convertibleUnits = true;
	      try {
	        convertUnit(1, variable?.units || '', units);
	      } catch (e) {
	        convertibleUnits = false;
	      }
	      if (convertibleUnits && variable) {
	        possible[key] = getConvertedVariable(variable, units);
	      } else {
	        delete possible[key];
	      }
	    }
	  }
	  if (label !== undefined) {
	    const regexpLabel = ensureRegexp(label);
	    for (key in possible) {
	      if (!regexpLabel.exec(variables[key]?.label ?? '')) {
	        delete possible[key];
	      }
	    }
	  }
	  if (variableName !== undefined) {
	    if (possible[variableName]) return possible[variableName];
	    const upper = variableName.toUpperCase();
	    if (Object.hasOwn(possible, upper)) {
	      return possible[upper];
	    }
	    const lower = variableName.toLowerCase();
	    if (Object.hasOwn(possible, lower)) {
	      return possible[lower];
	    }
	  }
	  const possibleFiltered = Object.values(possible).filter(val => val !== undefined);
	  if (possibleFiltered.length > 0) {
	    return possibleFiltered[0];
	  }
	}

	/**
	 * Retrieve the spectrum with only X/Y data that match all the selectors
	 * If more than one variable match the selector the 'x' or 'y' variable will be
	 * taken
	 */
	function getXYSpectrum(spectra = [], selector = {}) {
	  const selectedSpectra = getXYSpectra(spectra, selector);
	  if (selectedSpectra.length === 0) return undefined;
	  return selectedSpectra[0];
	}

	/**
	 * Class allowing to store and manipulate an analysis.
	 * An analysis may contain one or more spectra that can be selected
	 * based on their units
	 */
	class Analysis {
	  id;
	  label;
	  spectrumCallback;
	  spectra;
	  cache;
	  constructor(options = {}) {
	    this.id = options.id || Math.random().toString(36).substring(2, 10);
	    this.label = options.label || this.id;
	    this.spectrumCallback = options.spectrumCallback;
	    this.spectra = [];
	    this.cache = {
	      spectrum: {},
	      spectra: {}
	    };
	  }
	  /**
	   * Add a spectrum in the internal spectra variable
	   */
	  pushSpectrum(variables, options = {}) {
	    this.spectra.push(standardizeData(variables, options, {
	      spectrumCallback: this.spectrumCallback
	    }));
	    this.cache = {
	      spectrum: {},
	      spectra: {}
	    };
	  }
	  /**
	   * Retrieve a Spectrum based on x/y units
	   */
	  getXYSpectrum(selector = {}) {
	    const id = JSON.stringify(selector);
	    if (!this.cache.spectrum[id]) {
	      this.cache.spectrum[id] = getXYSpectrum(this.spectra, selector);
	    }
	    return this.cache.spectrum[id];
	  }
	  /**
	   * Retrieve spectra matching selector
	   */
	  getXYSpectra(selector = {}) {
	    const id = JSON.stringify(selector);
	    if (!this.cache.spectra[id]) {
	      this.cache.spectra[id] = getXYSpectra(this.spectra, selector);
	    }
	    return this.cache.spectra[id];
	  }
	  /**
	   * Retrieve a xy object
	   * @param selector.units Units separated by vs like for example "g vs °C"
	   * @param selector.xUnits if undefined takes the first variable
	   * @param selector.yUnits if undefined takes the second variable
	   */
	  getXY(selector = {}) {
	    const spectrum = this.getXYSpectrum(selector);
	    if (!spectrum) return undefined;
	    return {
	      x: spectrum.variables.x.data,
	      y: spectrum.variables.y.data
	    };
	  }
	  /**
	   * Return the data object for specific x/y units with possibly some
	   * normalization options
	   * @param options.selector.xUnits // if undefined takes the first variable
	   * @param options.selector.yUnits // if undefined takes the second variable
	   */
	  getNormalizedSpectrum(options = {}) {
	    const {
	      normalization,
	      selector
	    } = options;
	    const spectrum = this.getXYSpectrum(selector);
	    if (!spectrum) return undefined;
	    return getNormalizedSpectrum(spectrum, normalization);
	  }
	  /**
	   */
	  getNormalizedSpectra(options = {}) {
	    const {
	      normalization,
	      selector
	    } = options;
	    const spectra = this.getXYSpectra(selector);
	    if (spectra.length === 0) return [];
	    const normalizedSpectra = [];
	    for (const spectrum of spectra) {
	      normalizedSpectra.push(getNormalizedSpectrum(spectrum, normalization));
	    }
	    return normalizedSpectra;
	  }
	  /**
	   * Returns the first spectrum. This method could be improved in the future
	   * @returns
	   */
	  getSpectrum() {
	    return this.spectra[0];
	  }
	  /**
	   * Returns the xLabel
	   * @param selector.xUnits // if undefined takes the first variable
	   * @param selector.yUnits // if undefined takes the second variable
	   */
	  getXLabel(selector) {
	    return this.getXYSpectrum(selector)?.variables.x.label;
	  }
	  /**
	   * Returns the yLabel
	   * @param selector.xUnits // if undefined takes the first variable
	   * @param selector.yUnits // if undefined takes the second variable
	   */
	  getYLabel(selector) {
	    return this.getXYSpectrum(selector)?.variables.y.label;
	  }
	}
	/**
	 * Internal function that ensure the order of x / y array
	 */
	function standardizeData(variables, options, analysisOptions) {
	  const {
	    meta = {},
	    dataType = '',
	    title = ''
	  } = options;
	  const {
	    spectrumCallback
	  } = analysisOptions;
	  if (spectrumCallback) {
	    spectrumCallback(variables);
	  }
	  const xVariable = variables.x;
	  const yVariable = variables.y;
	  if (!xVariable || !yVariable) {
	    throw Error('A spectrum must contain at least x and y variables');
	  }
	  if (!isAnyArray$1(xVariable.data) || !isAnyArray$1(yVariable.data)) {
	    throw Error('x and y variables must contain an array data');
	  }
	  const x = xVariable.data;
	  const reverse = x && x.length > 1 && x[0] > x[x.length - 1];
	  for (const [key, variable] of Object.entries(variables)) {
	    if (reverse) variable.data = variable.data.slice().reverse();
	    variable.label = variable.label || key;
	    if (variable.label.match(/^.*[([](?<units>.*)[)\]].*$/)) {
	      const units = variable.label.replace(/^.*[([](?<units>.*)[)\]].*$/, '$<units>');
	      if (!variable.units || variable.units === units) {
	        variable.units = units;
	        variable.label = variable.label.replace(/[([].*[)\]]/, '').trim();
	      }
	    }
	    variable.min = xMinValue(variable.data);
	    variable.max = xMaxValue(variable.data);
	    variable.isMonotone = xIsMonotonic(variable.data) !== 0;
	  }
	  return {
	    variables,
	    title,
	    dataType,
	    meta,
	    id: Math.random().toString(36).replace('0.', '')
	  };
	}

	/*
	    https://tools.ietf.org/html/rfc3629

	    UTF8-char = UTF8-1 / UTF8-2 / UTF8-3 / UTF8-4

	    UTF8-1    = %x00-7F

	    UTF8-2    = %xC2-DF UTF8-tail

	    UTF8-3    = %xE0 %xA0-BF UTF8-tail
	                %xE1-EC 2( UTF8-tail )
	                %xED %x80-9F UTF8-tail
	                %xEE-EF 2( UTF8-tail )

	    UTF8-4    = %xF0 %x90-BF 2( UTF8-tail )
	                %xF1-F3 3( UTF8-tail )
	                %xF4 %x80-8F 2( UTF8-tail )

	    UTF8-tail = %x80-BF
	*/
	/**
	 * Check if a Node.js Buffer or Uint8Array is UTF-8.
	 */
	function isUtf8(buf) {
	  if (!buf) {
	    return false;
	  }
	  var i = 0;
	  var len = buf.length;
	  while (i < len) {
	    // UTF8-1 = %x00-7F
	    if (buf[i] <= 0x7F) {
	      i++;
	      continue;
	    }
	    // UTF8-2 = %xC2-DF UTF8-tail
	    if (buf[i] >= 0xC2 && buf[i] <= 0xDF) {
	      // if(buf[i + 1] >= 0x80 && buf[i + 1] <= 0xBF) {
	      if (buf[i + 1] >> 6 === 2) {
	        i += 2;
	        continue;
	      } else {
	        return false;
	      }
	    }
	    // UTF8-3 = %xE0 %xA0-BF UTF8-tail
	    // UTF8-3 = %xED %x80-9F UTF8-tail
	    if ((buf[i] === 0xE0 && buf[i + 1] >= 0xA0 && buf[i + 1] <= 0xBF || buf[i] === 0xED && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x9F) && buf[i + 2] >> 6 === 2) {
	      i += 3;
	      continue;
	    }
	    // UTF8-3 = %xE1-EC 2( UTF8-tail )
	    // UTF8-3 = %xEE-EF 2( UTF8-tail )
	    if ((buf[i] >= 0xE1 && buf[i] <= 0xEC || buf[i] >= 0xEE && buf[i] <= 0xEF) && buf[i + 1] >> 6 === 2 && buf[i + 2] >> 6 === 2) {
	      i += 3;
	      continue;
	    }
	    // UTF8-4 = %xF0 %x90-BF 2( UTF8-tail )
	    //          %xF1-F3 3( UTF8-tail )
	    //          %xF4 %x80-8F 2( UTF8-tail )
	    if ((buf[i] === 0xF0 && buf[i + 1] >= 0x90 && buf[i + 1] <= 0xBF || buf[i] >= 0xF1 && buf[i] <= 0xF3 && buf[i + 1] >> 6 === 2 || buf[i] === 0xF4 && buf[i + 1] >= 0x80 && buf[i + 1] <= 0x8F) && buf[i + 2] >> 6 === 2 && buf[i + 3] >> 6 === 2) {
	      i += 4;
	      continue;
	    }
	    return false;
	  }
	  return true;
	}

	/**
	 * Ensure that the data is string. If it is an ArrayBuffer it will be converted to string using TextDecoder.
	 * @param blob
	 * @param options
	 * @returns
	 */
	function ensureString(blob, options = {}) {
	  if (typeof blob === 'string') {
	    return blob;
	  }
	  if (ArrayBuffer.isView(blob) || blob instanceof ArrayBuffer) {
	    const {
	      encoding = guessEncoding(blob)
	    } = options;
	    const decoder = new TextDecoder(encoding);
	    return decoder.decode(blob);
	  }
	  throw new TypeError(`blob must be a string, ArrayBuffer or ArrayBufferView`);
	}
	function guessEncoding(blob) {
	  const uint8 = ArrayBuffer.isView(blob) ? new Uint8Array(blob.buffer, blob.byteOffset, blob.byteLength) : new Uint8Array(blob);
	  if (uint8.length >= 2) {
	    if (uint8[0] === 0xfe && uint8[1] === 0xff) {
	      return 'utf-16be';
	    }
	    if (uint8[0] === 0xff && uint8[1] === 0xfe) {
	      return 'utf-16le';
	    }
	  }
	  //@ts-expect-error an ArrayBuffer is also ok
	  if (!isUtf8(blob)) return 'latin1';
	  return 'utf-8';
	}

	/**
	 * Dynamically type a string
	 * @param {string} value String to dynamically type
	 * @returns {boolean|string|number}
	 */
	function parseString(value) {
	  if (value.length === 4 || value.length === 5) {
	    let lowercase = value.toLowerCase();
	    if (lowercase === 'true') return true;
	    if (lowercase === 'false') return false;
	  }
	  let number = Number(value);
	  if (number === 0 && !value.includes('0')) {
	    return value;
	  }
	  if (!Number.isNaN(number)) return number;
	  return value;
	}

	const gyromagneticRatio = {
	  '1H': 267.52218744e6,
	  '2H': 41.065e6,
	  '3H': 285.3508e6,
	  '3He': -203.789e6,
	  '7Li': 103.962e6,
	  '13C': 67.28284e6,
	  '14N': 19.331e6,
	  '15N': -27.116e6,
	  '17O': -36.264e6,
	  '19F': 251.662e6,
	  '23Na': 70.761e6,
	  '27Al': 69.763e6,
	  '29Si': -53.19e6,
	  '31P': 108.291e6,
	  '57Fe': 8.681e6,
	  '63Cu': 71.118e6,
	  '67Zn': 16.767e6,
	  '129Xe': -73.997e6
	};
	function getGyromagneticRatio(nucleus) {
	  if (gyromagneticRatio[nucleus]) {
	    return gyromagneticRatio[nucleus];
	  }
	  nucleus = nucleus.toLowerCase();
	  if (nucleus === 'proton') return gyromagneticRatio['1H'];
	  // we try to use only the numbers
	  const nucleusNumber = nucleus.replace(/[^0-9]/g, '');
	  if (!nucleusNumber) return null;
	  for (const key in gyromagneticRatio) {
	    if (key.includes(nucleusNumber)) return gyromagneticRatio[key];
	  }
	  return null;
	}

	var medianQuickselect_min = {exports: {}};

	(function (module) {
	  (function () {
	    function a(d) {
	      for (var e = 0, f = d.length - 1, g = void 0, h = void 0, i = void 0, j = c(e, f); !0;) {
	        if (f <= e) return d[j];
	        if (f == e + 1) return d[e] > d[f] && b(d, e, f), d[j];
	        for (g = c(e, f), d[g] > d[f] && b(d, g, f), d[e] > d[f] && b(d, e, f), d[g] > d[e] && b(d, g, e), b(d, g, e + 1), h = e + 1, i = f; !0;) {
	          do h++; while (d[e] > d[h]);
	          do i--; while (d[i] > d[e]);
	          if (i < h) break;
	          b(d, h, i);
	        }
	        b(d, e, i), i <= j && (e = h), i >= j && (f = i - 1);
	      }
	    }
	    var b = function b(d, e, f) {
	        var _ref;
	        return _ref = [d[f], d[e]], d[e] = _ref[0], d[f] = _ref[1], _ref;
	      },
	      c = function c(d, e) {
	        return ~~((d + e) / 2);
	      };
	    module.exports ? module.exports = a : window.median = a;
	  })();
	})(medianQuickselect_min);
	var quickSelectMedian = medianQuickselect_min.exports;

	function median(input) {
	  if (!isAnyArray$1(input)) {
	    throw new TypeError('input must be an array');
	  }
	  if (input.length === 0) {
	    throw new TypeError('input must not be empty');
	  }
	  return quickSelectMedian(input.slice());
	}

	var P = /\s*\$\$.*/;
	function M(e) {
	  return e.replace(P, "");
	}
	var E = ["TIC", ".RIC", "SCANNUMBER"];
	function te(e) {
	  let a = e.spectra,
	    s = a.length,
	    i = {
	      times: new Array(s),
	      series: {
	        ms: {
	          dimension: 2,
	          data: new Array(s)
	        }
	      }
	    },
	    o = [];
	  for (let l = 0; l < E.length; l++) {
	    let r = I(E[l]);
	    a[0][r] && (o.push(r), i.series[r] = {
	      dimension: 1,
	      data: new Array(s)
	    });
	  }
	  for (let l = 0; l < s; l++) {
	    let r = a[l];
	    i.times[l] = r.pageValue;
	    for (let t = 0; t < o.length; t++) i.series[o[t]].data[l] = Number(r[o[t]]);
	    r.data && (i.series.ms.data[l] = [r.data.x, r.data.y]);
	  }
	  e.chromatogram = i;
	}
	function le(e) {
	  return E.indexOf(e) !== -1;
	}
	function I(e) {
	  return e.toLowerCase().replace(/[^a-z0-9]/g, "");
	}
	function Y(e) {
	  let a = [];
	  for (let s = 0; s < e.length; s++) a.push(Number(e[s]));
	  return a;
	}
	function U(e, a) {
	  let s = e.yFactor,
	    i = e.deltaX;
	  e.isXYdata = !0;
	  let o = {
	    x: [],
	    y: []
	  };
	  e.data = o;
	  let l = e.firstX,
	    r = e.firstY,
	    t = !1,
	    n,
	    p = 0;
	  for (; p < a.length; p++) if (n = a.charCodeAt(p), n === 13 || n === 10) t = !0;else if (t) break;
	  let c = !0,
	    d = !1,
	    u = !1,
	    f = 0,
	    m = !1,
	    h = !1,
	    g = 0,
	    A = 0,
	    N = !1,
	    y = !1,
	    X = !1,
	    T = 0;
	  for (; p <= a.length; p++) if (p === a.length ? n = 13 : n = a.charCodeAt(p), h) (n === 13 || n === 10) && (c = !0, h = !1);else if (n <= 57 && n >= 48) y = !0, T > 0 ? g += (n - 48) / 10 ** T++ : (g *= 10, g += n - 48);else if (n === 44 || n === 46) y = !0, T++;else {
	    if (y) {
	      if (c) c = !1, u && (X = !0);else if (X) X = !1;else {
	        d ? (f = N ? 0 - g : g, u = !0, d = !1) : m || (A = N ? 0 - g : g);
	        let O = m ? g - 1 : 1;
	        for (let v = 0; v < O; v++) u ? r += f : r = A, o.x.push(l), o.y.push(r * s), l += i;
	      }
	      N = !1, g = 0, T = 0, y = !1, m = !1;
	    }
	    if (n < 74 && n > 63) y = !0, u = !1, g = n - 64;else if (n > 96 && n < 106) y = !0, u = !1, g = n - 96, N = !0;else if (n === 115) y = !0, m = !0, g = 9;else if (n > 82 && n < 91) y = !0, m = !0, g = n - 82;else if (n > 73 && n < 83) y = !0, d = !0, g = n - 73;else if (n > 105 && n < 115) y = !0, d = !0, g = n - 105, N = !0;else if (n === 36 && a.charCodeAt(p + 1) === 36) y = !0, h = !0;else if (n === 37) y = !0, d = !0, g = 0, N = !1;else if (n === 45) {
	      let O = a.charCodeAt(p + 1);
	      (O >= 48 && O <= 57 || O === 44 || O === 46) && (y = !0, c || (u = !1), N = !0);
	    } else (n === 13 || n === 10) && (c = !0, h = !1);
	  }
	}
	var se = /[,\t ]+/;
	function D(e, a, s) {
	  if (e.isPeaktable = !0, !e.variables || Object.keys(e.variables) === 2 ? me(e, a, s) : ce(e, a, s), e.variables) for (let i in e.variables) e.variables[i].data = e.data[i];
	}
	function me(e, a, s) {
	  let i = {
	    x: [],
	    y: []
	  };
	  e.data = i;
	  let o = a.split(/,? *,?[;\r\n]+ */);
	  for (let l = 1; l < o.length; l++) {
	    let r = o[l].trim().replace(P, "").split(se);
	    if (r.length % 2 === 0) for (let t = 0; t < r.length; t = t + 2) i.x.push(Number(r[t]) * e.xFactor), i.y.push(Number(r[t + 1]) * e.yFactor);else s.logs.push(`Format error: ${r}`);
	  }
	}
	function ce(e, a, s) {
	  let i = {},
	    o = Object.keys(e.variables),
	    l = o.length;
	  o.forEach(t => i[t] = []), e.data = i;
	  let r = a.split(/,? *,?[;\r\n]+ */);
	  for (let t = 1; t < r.length; t++) {
	    let n = r[t].trim().replace(P, "").split(se);
	    if (n.length % l === 0) for (let p = 0; p < n.length; p++) i[o[p % l]].push(Number(n[p]));else s.logs.push(`Wrong number of columns: ${n}`);
	  }
	}
	function V(e, a) {
	  e.isXYAdata = !0;
	  let s = {};
	  e.data = s;
	  let i = a.split(/\r?\n/),
	    o = i[0].replace(/^.*?([A-Z]+).*$/, "$1").split("").map(l => l.toLowerCase());
	  for (let l = 1; l < i.length; l++) {
	    let r = i[l].replace(/^\((.*)\)$/, "$1").split(/ *, */);
	    for (let t = 0; t < o.length; t++) {
	      let n = r[t];
	      switch (o[t]) {
	        case "x":
	        case "y":
	        case "w":
	          n = Number.parseFloat(n);
	          break;
	        case "a":
	          n = n.replace(/^<(.*)>$/, "$1");
	          break;
	        case "m":
	          break;
	        default:
	          continue;
	      }
	      s[o[t]] || (s[o[t]] = []), s[o[t]].push(n);
	    }
	  }
	}
	function ie(e) {
	  return Array.isArray(e) ? e[0] : e;
	}
	function B(e) {
	  let {
	      spectra: a
	    } = e,
	    s = a[0].data.y[0],
	    i = s,
	    o = a.length,
	    l = a[0].data.x.length,
	    r = new Array(o);
	  for (let f = 0; f < o; f++) {
	    r[f] = a[f].data.y;
	    for (let m = 0; m < l; m++) {
	      let h = r[f][m];
	      h < s && (s = h), h > i && (i = h);
	    }
	  }
	  let t = a[0].data.x[0],
	    n = a[0].data.x[a[0].data.x.length - 1],
	    {
	      firstY: p,
	      lastY: c
	    } = he(e);
	  if (t > n) for (let f of r) f.reverse();
	  p > c && r.reverse();
	  let d = [];
	  for (let f = 0; f < r.length; f++) {
	    let m = Float64Array.from(r[f]);
	    for (let h = 0; h < m.length; h++) m[h] < 0 && (m[h] = -m[h]);
	    d.push(median(m));
	  }
	  let u = median(d);
	  return {
	    z: r,
	    minX: Math.min(t, n),
	    maxX: Math.max(t, n),
	    minY: Math.min(p, c),
	    maxY: Math.max(p, c),
	    minZ: s,
	    maxZ: i,
	    noise: u
	  };
	}
	function he(e) {
	  let {
	    spectra: a,
	    ntuples: s,
	    info: i
	  } = e;
	  if (s) for (let o = 0; o < s.length; o++) {
	    let {
	      symbol: l,
	      nucleus: r
	    } = s[o];
	    if (l.match(/[F|T]1/)) {
	      let t = ie(i[".OBSERVEFREQUENCY"]),
	        {
	          nucleus: n
	        } = s.find(m => m.symbol.match(/[F|T]2/));
	      if ([t, n, r].some(m => !m)) break;
	      let p = getGyromagneticRatio(n),
	        c = getGyromagneticRatio(r),
	        {
	          first: d,
	          last: u
	        } = s[o],
	        f = t * c / p;
	      return {
	        firstY: d / f,
	        lastY: u / f
	      };
	    }
	  }
	  return {
	    firstY: a[0].pageValue,
	    lastY: a[a.length - 1].pageValue
	  };
	}
	function k(e, a) {
	  let s = e.noise,
	    i = e.z,
	    o,
	    l,
	    r,
	    t,
	    n,
	    p,
	    c,
	    d,
	    u = i.length,
	    f = i[0].length,
	    m,
	    h,
	    g,
	    A,
	    N = e.minX,
	    X = (e.maxX - N) / (f - 1),
	    T = e.minY,
	    v = (e.maxY - T) / (u - 1),
	    oe = e.minZ,
	    K = e.maxZ,
	    Q = a.nbContourLevels * 2,
	    W = new Array(Q),
	    x;
	  for (let R = 0; R < Q; R++) {
	    let $ = {};
	    W[R] = $;
	    let fe = R % 2,
	      _ = (K - a.noiseMultiplier * s) * Math.exp((R >> 1) - a.nbContourLevels);
	    fe === 0 ? x = _ + a.noiseMultiplier * s : x = 0 - _ - a.noiseMultiplier * s;
	    let b = [];
	    if ($.zValue = x, $.lines = b, !(x <= oe || x >= K)) for (let C = 0; C < u - 1; C++) {
	      let z = i[C],
	        ee = i[C + 1];
	      for (let S = 0; S < f - 1; S++) o = z[S], l = z[S + 1], r = ee[S], t = ee[S + 1], n = o > x, p = l > x, c = r > x, d = t > x, n !== p && n !== c && (m = S + (x - o) / (l - o), h = C, g = S, A = C + (x - o) / (r - o), b.push(m * X + N), b.push(h * v + T), b.push(g * X + N), b.push(A * v + T)), d !== p && d !== c && (m = S + 1, h = C + 1 - (x - t) / (l - t), g = S + 1 - (x - t) / (r - t), A = C + 1, b.push(m * X + N), b.push(h * v + T), b.push(g * X + N), b.push(A * v + T)), p !== c && (m = (S + 1 - (x - l) / (r - l)) * X + N, h = (C + (x - l) / (r - l)) * v + T, p !== n && (g = S + 1 - (x - l) / (o - l), A = C, b.push(m), b.push(h), b.push(g * X + N), b.push(A * v + T)), c !== n && (g = S, A = C + 1 - (x - r) / (o - r), b.push(m), b.push(h), b.push(g * X + N), b.push(A * v + T)), p !== d && (g = S + 1, A = C + (x - l) / (t - l), b.push(m), b.push(h), b.push(g * X + N), b.push(A * v + T)), c !== d && (g = S + (x - r) / (t - r), A = C + 1, b.push(m), b.push(h), b.push(g * X + N), b.push(A * v + T)));
	    }
	  }
	  return {
	    minX: e.minX,
	    maxX: e.maxX,
	    minY: e.minY,
	    maxY: e.maxY,
	    segments: W
	  };
	}
	function j(e, a) {
	  let s = B(e);
	  a.noContour || (e.contourLines = k(s, a), delete s.z), e.minMax = s;
	}
	function J(e, a) {
	  for (let s of e) {
	    let i = 0,
	      o = 0;
	    for (let l of s.spectra) {
	      if (s.ntuples && s.ntuples.symbol ? (!i && l.observeFrequency && (i = l.observeFrequency), !o && l.shiftOffsetVal && (o = l.shiftOffsetVal)) : (i = l.observeFrequency, o = l.shiftOffsetVal), i && l.xUnits && l.xUnits.toUpperCase().includes("HZ")) {
	        l.xUnits = "PPM", l.xFactor = l.xFactor / i, l.firstX = l.firstX / i, l.lastX = l.lastX / i, l.deltaX = l.deltaX / i;
	        for (let r = 0; r < l.data.x.length; r++) l.data.x[r] /= i;
	      }
	      if (o && l.xUnits.toLowerCase().includes("ppm")) {
	        let r = l.firstX - o;
	        l.firstX = l.firstX - r, l.lastX = l.lastX - r;
	        for (let t = 0; t < l.data.x.length; t++) l.data.x[t] -= r;
	      }
	      if (s.ntuples && s.ntuples.nucleus && s.ntuples.symbol) for (let r = 0; r < s.ntuples.nucleus.length; r++) {
	        let t = s.ntuples.symbol[r],
	          n = s.ntuples.nucleus[r];
	        if (t.match(/^[F|T]/) && !n) {
	          if (t.match(/[F|T]1/)) if (s.tmp.$NUC2) s.ntuples.nucleus[r] = s.tmp.$NUC2;else {
	            let p = s.ntuples.symbol.indexOf(t.replace(/^([F|T]).*/, "$12"));
	            p && s.ntuples.nucleus[p] && (s.ntuples.nucleus[r] = s.ntuples.nucleus[p]);
	          }
	          t.match(/[F|T]2/) && (s.ntuples.nucleus[r] = s.tmp.$NUC1);
	        }
	        t.match(/[F|T]2/) && (s.yType = s.ntuples.nucleus[0], getGyromagneticRatio(s.xType) || (s.xType = s.ntuples.nucleus[1]));
	      }
	      if (i && s.ntuples && s.ntuples.symbol && s.ntuples.nucleus) {
	        let r = "",
	          t = s.ntuples.symbol.indexOf(l.pageSymbol);
	        if (s.ntuples.units && s.ntuples.units[t] && (r = s.ntuples.units[t]), r !== "PPM") {
	          if (t !== 0) {
	            let u = "Not sure about this ntuples format";
	            if (a) a.warn(u);else throw Error(u);
	          }
	          let {
	              nucleus: n
	            } = s.ntuples,
	            p = getGyromagneticRatio(n[0]),
	            c = getGyromagneticRatio(n[1]);
	          if (!p || !c) {
	            let u = `Problem with determination of gyromagnetic ratio for ${n.join("-")}`;
	            if (a) a.error(u);else throw Error(u);
	          }
	          let d = p / c * i;
	          l.pageValue /= d;
	        }
	      }
	    }
	  }
	}
	function L(e, a, s = {}) {
	  s.logger?.trace(a), e.profiling && e.profiling.push({
	    action: a,
	    time: Date.now() - s.start
	  });
	}
	function G(e) {
	  let a = e.spectra[0].data;
	  e.chromatogram = {
	    times: a.x.slice(),
	    series: {
	      intensity: {
	        dimension: 1,
	        data: a.y.slice()
	      }
	    }
	  };
	}
	function H(e, a, s) {
	  J(e, s.logger), de(e, s);
	  for (let i of e) {
	    if (Object.keys(i.ntuples).length > 0) {
	      let o = [],
	        l = Object.keys(i.ntuples);
	      for (let r = 0; r < l.length; r++) {
	        let t = l[r],
	          n = i.ntuples[t];
	        for (let p = 0; p < n.length; p++) o[p] || (o[p] = {}), o[p][t] = n[p];
	      }
	      i.ntuples = o;
	    }
	    i.twoD && s.wantXY && (j(i, s), L(a, "Finished countour plot calculation", s), s.keepSpectra || delete i.spectra), s.chromatogram && (i.spectra.length > 1 ? te(i) : G(i), L(a, "Finished chromatogram calculation", s)), delete i.tmp;
	  }
	}
	function de(e, a) {
	  for (let s of e) for (let i in s.meta) {
	    let o = s.meta[i];
	    if (typeof o == "string") {
	      if (o[0] === "{") {
	        if (o[o.length - 1] === "}") {
	          let l = o.slice(1, -1).split(/[,; ]+/).filter(r => r);
	          for (let r = 0; r < l.length; r++) s.meta[i + String(r)] = a.dynamicTyping ? parseString(l[r]) : l[r];
	        }
	      } else if (o[0] === "(") {
	        let l = o.split(/\r?\n/),
	          r = /^\((?<from>\d+)\.\.(?<to>\d+)\).*$/;
	        if (r.test(l[0])) {
	          let [t, n] = l[0].match(r).slice(1).map(Number),
	            p = l.slice(1).join(" ").split(/[,; ]+/).filter(c => c);
	          for (let c = t; c <= n; c++) a.dynamicTyping && typeof p[c - t] == "string" ? s.meta[i + String(c)] = parseString(p[c - t]) : s.meta[i + String(c)] = p[c - t];
	        }
	      }
	    }
	  }
	}
	function q(e, a, s) {
	  let i = -1,
	    o = -1,
	    l = "",
	    r = "";
	  if (s.indexOf("++") > 0) l = s.replace(/.*\(([a-zA-Z0-9]+)\+\+.*/, "$1"), r = s.replace(/.*\.\.([a-zA-Z0-9]+).*/, "$1");else {
	    s = s.replace(/[^a-zA-Z%]/g, ""), l = s.charAt(0), r = s.charAt(1), a.variables = {};
	    for (let t of s) {
	      let n = t.toLowerCase(),
	        p = e.ntuples.symbol.indexOf(t);
	      if (p === -1) throw Error(`Symbol undefined: ${t}`);
	      a.variables[n] = {};
	      for (let c in e.ntuples) e.ntuples[c][p] && (a.variables[n][c.replace(/^var/, "")] = e.ntuples[c][p]);
	    }
	  }
	  i = e.ntuples.symbol.indexOf(l), o = e.ntuples.symbol.indexOf(r), i === -1 && (i = 0), o === -1 && (o = 0), e.ntuples.first && (e.ntuples.first.length > i && (a.firstX = e.ntuples.first[i]), e.ntuples.first.length > o && (a.firstY = e.ntuples.first[o])), e.ntuples.last && (e.ntuples.last.length > i && (a.lastX = e.ntuples.last[i]), e.ntuples.last.length > o && (a.lastY = e.ntuples.last[o])), e.ntuples.vardim && e.ntuples.vardim.length > i && (a.nbPoints = e.ntuples.vardim[i]), e.ntuples.factor && (e.ntuples.factor.length > i && (a.xFactor = e.ntuples.factor[i]), e.ntuples.factor.length > o && (a.yFactor = e.ntuples.factor[o])), e.ntuples.units && (e.ntuples.units.length > i && (e.ntuples.varname && e.ntuples.varname[i] ? a.xUnits = `${e.ntuples.varname[i]} [${e.ntuples.units[i]}]` : a.xUnits = e.ntuples.units[i]), e.ntuples.units.length > o && (e.ntuples.varname && e.ntuples.varname[o] ? a.yUnits = `${e.ntuples.varname[o]} [${e.ntuples.units[o]}]` : a.yUnits = e.ntuples.units[o]));
	}
	function w(e) {
	  e.xFactor || (e.xFactor = 1), e.yFactor || (e.yFactor = 1);
	}
	var F = /[ \t]*,[ \t]*/,
	  xe = {
	    keepRecordsRegExp: /^$/,
	    canonicDataLabels: !0,
	    canonicMetadataLabels: !1,
	    dynamicTyping: !0,
	    withoutXY: !1,
	    noTrimRegExp: /^$/,
	    chromatogram: !1,
	    keepSpectra: !1,
	    noContour: !1,
	    nbContourLevels: 7,
	    noiseMultiplier: 5,
	    profiling: !1
	  };
	function Ne(e, a = {}) {
	  e = ensureString(e), a = {
	    ...xe,
	    ...a
	  }, a.logger?.debug("Starting jcamp conversion"), a.wantXY = !a.withoutXY, a.start = Date.now();
	  let s = [],
	    i = {
	      profiling: a.profiling ? [] : !1,
	      logs: [],
	      entries: []
	    },
	    o = {
	      children: []
	    },
	    l = o,
	    r = [],
	    t = {};
	  if (typeof e != "string") throw new TypeError("the JCAMP should be a string");
	  L(i, "Before split to LDRS", a);
	  let n = e.replace(/[\r\n]+##/g, `
##`).split(`
##`);
	  L(i, "Split to LDRS", a), n[0] && (n[0] = n[0].replace(/^[\r\n ]*##/, ""));
	  for (let p of n) {
	    let c = p.indexOf("="),
	      d = c > 0 ? p.substring(0, c) : p,
	      u = c > 0 ? d.match(a.noTrimRegExp) ? p.substring(c + 1) : p.substring(c + 1).trim() : "",
	      f = d.replace(/[_ -]/g, "").toUpperCase();
	    if (f === "DATATABLE") {
	      let m = u.indexOf(`
`);
	      if (m === -1 && (m = u.indexOf("\r")), m > 0) {
	        let h = u.substring(0, m).split(/[ ,;\t]+/);
	        q(l, t, h[0]), t.datatable = h[0], h[1] && h[1].indexOf("PEAKS") > -1 ? f = "PEAKTABLE" : h[1] && (h[1].indexOf("XYDATA") || h[0].indexOf("++") > 0) && (f = "XYDATA", t.nbPoints && (t.deltaX = (t.lastX - t.firstX) / (t.nbPoints - 1)));
	      }
	    }
	    if (f === "XYDATA") {
	      a.wantXY && (w(t), u.match(/.*\+\+.*/) ? (t.nbPoints && (t.deltaX = (t.lastX - t.firstX) / (t.nbPoints - 1)), U(t, u)) : D(t, u, i), l.spectra.push(t), t = {});
	      continue;
	    } else if (f === "PEAKTABLE") {
	      a.wantXY && (w(t), D(t, u, i), l.spectra.push(t), t = {});
	      continue;
	    }
	    if (f === "PEAKASSIGNMENTS") {
	      a.wantXY && (u.match(/.*([^A-Z]*).*/) && V(t, u), l.spectra.push(t), t = {});
	      continue;
	    }
	    if (f === "TITLE") {
	      let m = l;
	      m.children || (m.children = []), l = {
	        spectra: [],
	        ntuples: {},
	        info: {},
	        meta: {},
	        tmp: {}
	      }, m.children.push(l), r.push(m), s.push(l), l.title = u;
	    } else f === "DATATYPE" ? (l.dataType = u, u.match(/(^nd|\snd\s)/i) && (l.twoD = !0)) : f === "NTUPLES" ? u.match(/(^nd|\snd\s)/i) && (l.twoD = !0) : f === "DATACLASS" ? l.dataClass = u : f === "JCAMPDX" ? l.jcampDX = M(u) : f === "JCAMPCS" ? l.jcampCS = M(u) : f === "XUNITS" ? t.xUnits = u : f === "YUNITS" ? t.yUnits = u : f === "FIRSTX" ? t.firstX = Number(u) : f === "LASTX" ? t.lastX = Number(u) : f === "FIRSTY" ? t.firstY = Number(u) : f === "LASTY" ? t.lastY = Number(u) : f === "NPOINTS" ? t.nbPoints = Number(u) : f === "XFACTOR" ? t.xFactor = Number(u) : f === "YFACTOR" ? t.yFactor = Number(u) : f === "MAXX" ? t.maxX = Number(u) : f === "MINX" ? t.minX = Number(u) : f === "MAXY" ? t.maxY = Number(u) : f === "MINY" ? t.minY = Number(u) : f === "DELTAX" ? t.deltaX = Number(u) : f === ".OBSERVEFREQUENCY" || f === "$SFO1" ? t.observeFrequency || (t.observeFrequency = Number(u)) : f === ".OBSERVENUCLEUS" ? t.xType || (l.xType = u.replace(/[^a-zA-Z0-9]/g, "")) : f === "$OFFSET" ? (l.shiftOffsetNum = 0, t.shiftOffsetVal || (t.shiftOffsetVal = Number(u))) : f === "$REFERENCEPOINT" || (f === "VARNAME" ? l.ntuples.varname = u.split(F) : f === "SYMBOL" ? l.ntuples.symbol = u.split(F) : f === "VARTYPE" ? l.ntuples.vartype = u.split(F) : f === "VARFORM" ? l.ntuples.varform = u.split(F) : f === "VARDIM" ? l.ntuples.vardim = Y(u.split(F)) : f === "UNITS" ? l.ntuples.units = u.split(F) : f === "FACTOR" ? l.ntuples.factor = Y(u.split(F)) : f === "FIRST" ? l.ntuples.first = Y(u.split(F)) : f === "LAST" ? l.ntuples.last = Y(u.split(F)) : f === "MIN" ? l.ntuples.min = Y(u.split(F)) : f === "MAX" ? l.ntuples.max = Y(u.split(F)) : f === ".NUCLEUS" ? l.ntuples && (l.ntuples.nucleus = u.split(F).map(m => m.replace(/[^a-zA-Z0-9]/g, ""))) : f === "PAGE" ? (t.page = u.trim(), t.pageValue = Number(u.replace(/^.*=/, "")), t.pageSymbol = t.page.replace(/[=].*/, "")) : f === "RETENTIONTIME" ? t.pageValue = Number(u) : le(f) ? t[I(f)] = u : f === "SAMPLEDESCRIPTION" ? t.sampleDescription = u : f.startsWith("$NUC") ? !l.tmp[f] && !u.includes("off") && (l.tmp[f] = u.replace(/[<>]/g, "")) : f === "END" && (l = r.pop()));
	    if (l && l.info && l.meta && f.match(a.keepRecordsRegExp)) {
	      let m, h;
	      d.startsWith("$") ? (h = a.canonicMetadataLabels ? f.substring(1) : d.substring(1), m = l.meta) : (h = a.canonicDataLabels ? f : d, m = l.info), a.dynamicTyping && (u = parseString(u)), m[h] ? (Array.isArray(m[h]) || (m[h] = [m[h]]), m[h].push(u)) : m[h] = u;
	    }
	  }
	  if (L(i, "Finished parsing", a), H(s, i, a), L(i, "Total time", a), i.entries = o.children, i.flatten = s, a.logger) {
	    a.logger.debug("Finished jcamp conversion");
	    for (let p of i.flatten) a.logger.debug(`${p.dataType} - ${p.title}`);
	  }
	  return i;
	}

	/**
	 * Creates a new Analysis from a JCAMP string
	 * @param {string} jcamp - String containing the JCAMP data
	 * @param {object} [options={}]
	 * @param {object} [options.id=Math.random()]
	 * @param {string} [options.label=options.id] human redeable label
	 * @param {string} [options.spectrumCallback] a callback to apply on variables when creating spectrum
	 * @return {Analysis} - New class element with the given data
	 */
	function fromJcamp(jcamp, options = {}) {
	  const analysis = new Analysis(options);
	  addJcamp(analysis, jcamp);
	  return analysis;
	}
	function addJcamp(analysis, jcamp) {
	  const converted = Ne(jcamp, {
	    keepRecordsRegExp: /.*/
	  });
	  for (const entry of converted.flatten) {
	    if (!entry.spectra?.[0]) continue;
	    const currentSpectrum = entry.spectra[0];
	    // we ensure variables
	    if (!currentSpectrum.variables) {
	      const variables = {};
	      currentSpectrum.variables = variables;
	      variables.x = {
	        label: currentSpectrum.xUnits,
	        symbol: 'X',
	        data: currentSpectrum.data.x || currentSpectrum.data.X
	      };
	      variables.y = {
	        label: currentSpectrum.yUnits,
	        symbol: 'Y',
	        data: currentSpectrum.data.y || currentSpectrum.data.Y
	      };
	    } else {
	      for (const key in currentSpectrum.variables) {
	        const variable = currentSpectrum.variables[key];
	        if (variable.label) continue;
	        variable.label = variable.name || variable.symbol || key;
	        if (variable.units && !variable.label.includes(variable.units)) {
	          variable.label += ` [${variable.units}]`;
	        }
	      }
	    }
	    analysis.pushSpectrum(currentSpectrum.variables, {
	      dataType: entry.dataType,
	      title: entry.title,
	      meta: entry.meta
	    });
	  }
	}

	const addInfoData = (data, options = {}) => {
	  const {
	    keys = Object.keys(data),
	    prefix = '##$'
	  } = options;
	  let header = '';
	  for (const key of keys) {
	    header += typeof data[key] === 'object' ? `${prefix}${key}=${JSON.stringify(data[key])}\n` : `${prefix}${key}=${data[key]}\n`;
	  }
	  return header;
	};

	function checkMatrix(data) {
	  if (!isAnyArray$1(data) || !isAnyArray$1(data[0])) {
	    throw new Error(`2D data should be a matrix`);
	  }
	}

	function checkNumberOrArray(data) {
	  if (!isAnyArray$1(data) || isAnyArray$1(data[0])) {
	    throw new Error(`x and y data should be an array of numbers`);
	  }
	}

	function getExtremeValues(data) {
	  if (isAnyArray$1(data[0])) {
	    checkMatrix(data);
	    const firstRow = data[0];
	    return {
	      firstLast: {
	        first: firstRow[0],
	        last: data[data.length - 1][data[0].length - 1]
	      },
	      minMax: matrixMinMaxZ(data)
	    };
	  }
	  checkNumberOrArray(data);
	  return {
	    firstLast: {
	      first: data[0],
	      last: data[data.length - 1]
	    },
	    minMax: xMinMaxValues(data)
	  };
	}

	/**
	 * Parse from a xyxy data array
	 * @param variables - Variables to convert to jcamp
	 * @param [options={}] - options that allows to add meta data in the jcamp
	 * @return JCAMP-DX text file corresponding to the variables
	 */
	function creatorNtuples(variables, options) {
	  const {
	    meta = {},
	    info = {}
	  } = options;
	  const {
	    title = '',
	    owner = '',
	    origin = '',
	    dataType = '',
	    ...resInfo
	  } = info;
	  const symbol = [];
	  const varName = [];
	  const varType = [];
	  const varDim = [];
	  const units = [];
	  const first = [];
	  const last = [];
	  const min = [];
	  const max = [];
	  const keys = Object.keys(variables);
	  for (let i = 0; i < keys.length; i++) {
	    const key = keys[i];
	    const variable = variables[key];
	    if (!variable) continue;
	    const name = variable?.label.replace(/ *\[.*/, '');
	    const unit = variable?.label.replace(/.*\[(?<units>.*)\].*/, '$<units>');
	    const {
	      firstLast,
	      minMax
	    } = getExtremeValues(variable.data);
	    symbol.push(variable.symbol || key);
	    varName.push(name || key);
	    varDim.push(variable.data.length);
	    first.push(firstLast.first);
	    last.push(firstLast.last);
	    max.push(minMax.max);
	    min.push(minMax.min);
	    if (variable.isDependent !== undefined) {
	      varType.push(variable.isDependent ? 'DEPENDENT' : 'INDEPENDENT');
	    } else {
	      varType.push(variable.isDependent !== undefined ? !variable.isDependent : i === 0 ? 'INDEPENDENT' : 'DEPENDENT');
	    }
	    units.push(variable.units || unit || '');
	  }
	  let header = `##TITLE=${title}
##JCAMP-DX=6.00
##DATA TYPE=${dataType}
##DATA CLASS= NTUPLES
##ORIGIN=${origin}
##OWNER=${owner}\n`;
	  header += addInfoData(resInfo, {
	    prefix: '##'
	  });
	  header += addInfoData(meta);
	  header += `##NTUPLES= ${dataType}
##VAR_NAME=  ${varName.join()}
##SYMBOL=    ${symbol.join()}
##VAR_TYPE=  ${varType.join()}
##VAR_DIM=   ${varDim.join()}
##UNITS=     ${units.join()}
##FIRST=     ${first.join()}
##LAST=      ${last.join()}
##MIN=       ${min.join()}
##MAX=       ${max.join()}
##PAGE= N=1\n`;
	  header += `##DATA TABLE= (${symbol.join('')}..${symbol.join('')}), PEAKS\n`;
	  for (let i = 0; i < variables.x.data.length; i++) {
	    const point = [];
	    for (const key of keys) {
	      const variable = variables[key];
	      if (!variable) continue;
	      point.push(variable.data[i]);
	    }
	    header += `${point.join('\t')}\n`;
	  }
	  header += `##END NTUPLES= ${dataType}\n`;
	  header += '##END=\n##END=';
	  return header;
	}

	function getFactorNumber(minMax, maxValue = 2 ** 31 - 1) {
	  let factor;
	  if (minMax.min < 0) {
	    if (minMax.max > 0) {
	      factor = Math.max(-minMax.min, minMax.max) / maxValue;
	    } else {
	      factor = -minMax.min / maxValue;
	    }
	  } else {
	    factor = minMax.max / maxValue;
	  }
	  return factor;
	}

	function getBestFactor(array, options = {}) {
	  const {
	    maxValue,
	    factor,
	    minMax
	  } = options;
	  if (factor !== undefined) {
	    return factor;
	  }
	  // is there non integer number ?
	  let onlyInteger = true;
	  for (const y of array) {
	    if (Math.round(y) !== y) {
	      onlyInteger = false;
	      break;
	    }
	  }
	  if (onlyInteger) {
	    return 1;
	  }
	  // we need to rescale the values
	  // need to find the max and min values
	  const extremeValues = minMax || xMinMaxValues(array);
	  return getFactorNumber(extremeValues, maxValue);
	}

	/**
	 * Reconvert number to original value
	 * @param number Number used for computation
	 * @param factor Multiplying factor
	 * @returns Original value
	 */
	function getNumber(number, factor) {
	  if (factor !== 1) number /= factor;
	  const rounded = Math.round(number);
	  if (rounded !== number && Math.abs(rounded - number) <= Number.EPSILON) {
	    return rounded;
	  }
	  return number;
	}

	function peakTableCreator(data, options = {}) {
	  const {
	    xFactor = 1,
	    yFactor = 1
	  } = options.info || {};
	  let firstX = Number.POSITIVE_INFINITY;
	  let lastX = Number.NEGATIVE_INFINITY;
	  let firstY = Number.POSITIVE_INFINITY;
	  let lastY = Number.NEGATIVE_INFINITY;
	  const lines = [];
	  for (let i = 0; i < data.x.length; i++) {
	    const x = data.x[i];
	    const y = data.y[i];
	    if (firstX > x) {
	      firstX = x;
	    }
	    if (lastX < x) {
	      lastX = x;
	    }
	    if (firstY > y) {
	      firstY = y;
	    }
	    if (lastY < y) {
	      lastY = y;
	    }
	  }
	  lines.push(`##FIRSTX=${firstX}`);
	  lines.push(`##LASTX=${lastX}`);
	  lines.push(`##FIRSTY=${firstY}`);
	  lines.push(`##LASTY=${lastY}`);
	  lines.push(`##XFACTOR=${xFactor}`);
	  lines.push(`##YFACTOR=${yFactor}`);
	  lines.push('##PEAK TABLE=(XY..XY)');
	  for (let i = 0; i < data.x.length; i++) {
	    lines.push(`${getNumber(data.x[i], xFactor)} ${getNumber(data.y[i], yFactor)}`);
	  }
	  return lines;
	}

	function rescaleAndEnsureInteger(data, factor = 1) {
	  if (factor === 1) return data.map(value => Math.round(value));
	  return xDivide(data, factor).map(value => Math.round(value));
	}

	/**
	 * class encodes a integer vector as a String in order to store it in a text file.
	 * The algorithms used to encode the data are describe in:
	 *            http://www.iupac.org/publications/pac/pdf/2001/pdf/7311x1765.pdf
	 */
	const newLine = '\n';
	const pseudoDigits = [['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'], ['@', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I'], ['@', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i'], ['%', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R'], ['%', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r'], [' ', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 's']];
	const SQZ_P = 1;
	const SQZ_N = 2;
	const DIF_P = 3;
	const DIF_N = 4;
	const DUP = 5;
	const maxLinelength = 100;
	/**
	 * This function encodes the given vector. The xyEncoding format is specified by the
	 * xyEncoding option
	 * @param xyEncoding: ('FIX','SQZ','DIF','DIFDUP','CVS','PAC') Default 'DIFDUP'
	 * @return {string}
	 */
	function vectorEncoder(data, firstX, intervalX, xyEncoding) {
	  switch (xyEncoding) {
	    case 'FIX':
	      return fixEncoding(data, firstX, intervalX);
	    case 'SQZ':
	      return squeezedEncoding(data, firstX, intervalX);
	    case 'DIF':
	      return differenceEncoding(data, firstX, intervalX);
	    case 'DIFDUP':
	      return differenceDuplicateEncoding(data, firstX, intervalX);
	    case 'CSV':
	      return commaSeparatedValuesEncoding(data, firstX, intervalX);
	    case 'PAC':
	      return packedEncoding(data, firstX, intervalX);
	    default:
	      return differenceEncoding(data, firstX, intervalX);
	  }
	}
	/**
	 * @private
	 * No data compression used. The data is separated by a comma(',').
	 */
	function commaSeparatedValuesEncoding(data, firstX, intervalX) {
	  return fixEncoding(data, firstX, intervalX, ',');
	}
	/**
	 * @private
	 * No data compression used. The data is separated by the specified separator.
	 */
	function fixEncoding(data, firstX, intervalX, separator = ' ') {
	  let outputData = '';
	  let j = 0;
	  const dataLength = data.length;
	  while (j < dataLength - 7) {
	    outputData += Math.ceil(firstX + j * intervalX);
	    for (let i = 0; i < 8; i++) {
	      outputData += `${separator}${data[j++]}`;
	    }
	    outputData += newLine;
	  }
	  if (j < dataLength) {
	    // We add last numbers
	    outputData += Math.ceil(firstX + j * intervalX);
	    for (let i = j; i < dataLength; i++) {
	      outputData += `${separator}${data[i]}`;
	    }
	  }
	  return outputData;
	}
	/**
	 * @private
	 * No data compression used. The data is separated by the sign of the number.
	 */
	function packedEncoding(data, firstX, intervalX) {
	  let outputData = '';
	  let j = 0;
	  const dataLength = data.length;
	  while (j < dataLength - 7) {
	    outputData += Math.ceil(firstX + j * intervalX);
	    for (let i = 0; i < 8; i++) {
	      outputData += data[j] < 0 ? data[j++] : `+${data[j++]}`;
	    }
	    outputData += newLine;
	  }
	  if (j < dataLength) {
	    // We add last numbers
	    outputData += Math.ceil(firstX + j * intervalX);
	    for (let i = j; i < dataLength; i++) {
	      outputData += data[i] < 0 ? data[i] : `+${data[i]}`;
	    }
	  }
	  return outputData;
	}
	/**
	 * @private
	 * Data compression is possible using the squeezed form (SQZ) in which the delimiter, the leading digit,
	 * and sign are replaced by a pseudo-digit from Table 1. For example, the Y-values 30, 32 would be
	 * represented as C0C2.
	 */
	function squeezedEncoding(data, firstX, intervalX) {
	  let outputData = '';
	  // String outputData = new String();
	  let j = 0;
	  const dataLength = data.length;
	  while (j < dataLength - 10) {
	    outputData += Math.ceil(firstX + j * intervalX);
	    for (let i = 0; i < 10; i++) {
	      outputData += squeezedDigit(data[j++].toString());
	    }
	    outputData += newLine;
	  }
	  if (j < dataLength) {
	    // We add last numbers
	    outputData += Math.ceil(firstX + j * intervalX);
	    for (let i = j; i < dataLength; i++) {
	      outputData += squeezedDigit(data[i].toString());
	    }
	  }
	  return outputData;
	}
	/**
	 * @private
	 * Duplicate suppression xyEncoding
	 */
	function differenceDuplicateEncoding(data, firstX, intervalX) {
	  let mult = 0;
	  let index = 0;
	  let charCount = 0;
	  // We built a string where we store the encoded data.
	  let encodedData = '';
	  let encodedNumber = '';
	  let temp = '';
	  // We calculate the differences vector
	  const diffData = new Array(data.length - 1);
	  for (let i = 0; i < diffData.length; i++) {
	    diffData[i] = data[i + 1] - data[i];
	  }
	  // We simulate a line carry
	  const numDiff = diffData.length;
	  while (index < numDiff) {
	    if (charCount === 0) {
	      // Start line
	      encodedNumber = `${Math.ceil(firstX + index * intervalX)}${squeezedDigit(data[index].toString())}${differenceDigit(diffData[index].toString())}`;
	      encodedData += encodedNumber;
	      charCount += encodedNumber.length;
	    } else if (diffData[index - 1] === diffData[index]) {
	      // Try to insert next difference
	      mult++;
	    } else if (mult > 0) {
	      // Now we know that it can be in line
	      mult++;
	      encodedNumber = duplicateDigit(mult.toString());
	      encodedData += encodedNumber;
	      charCount += encodedNumber.length;
	      mult = 0;
	      index--;
	    } else {
	      // Check if it fits, otherwise start a new line
	      encodedNumber = differenceDigit(diffData[index].toString());
	      if (encodedNumber.length + charCount < maxLinelength) {
	        encodedData += encodedNumber;
	        charCount += encodedNumber.length;
	      } else {
	        // start a new line
	        encodedData += newLine;
	        temp = `${Math.ceil(firstX + index * intervalX)}${squeezedDigit(data[index].toString())}${encodedNumber}`;
	        encodedData += temp; // Each line start with first index number.
	        charCount = temp.length;
	      }
	    }
	    index++;
	  }
	  if (mult > 0) {
	    encodedData += duplicateDigit((mult + 1).toString());
	  }
	  // We insert the last data from fid. It is done to control of data
	  // The last line start with the number of datas in the fid.
	  encodedData += `${newLine}${Math.ceil(firstX + index * intervalX)}${squeezedDigit(data[index].toString())}`;
	  return encodedData;
	}
	/**
	 * @private
	 * Differential xyEncoding
	 */
	function differenceEncoding(data, firstX, intervalX) {
	  let index = 0;
	  let charCount = 0;
	  let i;
	  let encodedData = '';
	  let encodedNumber = '';
	  let temp = '';
	  // We calculate the differences vector
	  const diffData = new Array(data.length - 1);
	  for (i = 0; i < diffData.length; i++) {
	    diffData[i] = data[i + 1] - data[i];
	  }
	  const numDiff = diffData.length;
	  while (index < numDiff) {
	    if (charCount === 0) {
	      // We convert the first number.
	      encodedNumber = `${Math.ceil(firstX + index * intervalX)}${squeezedDigit(data[index].toString())}${differenceDigit(diffData[index].toString())}`;
	      encodedData += encodedNumber;
	      charCount += encodedNumber.length;
	    } else {
	      encodedNumber = differenceDigit(diffData[index].toString());
	      if (encodedNumber.length + charCount < maxLinelength) {
	        encodedData += encodedNumber;
	        charCount += encodedNumber.length;
	      } else {
	        encodedData += newLine;
	        temp = `${Math.ceil(firstX + index * intervalX)}${squeezedDigit(data[index].toString())}${encodedNumber}`;
	        encodedData += temp; // Each line start with first index number.
	        charCount = temp.length;
	      }
	    }
	    index++;
	  }
	  // We insert the last number from data. It is done to control of data
	  encodedData += `${newLine}${Math.ceil(firstX + index * intervalX)}${squeezedDigit(data[index].toString())}`;
	  return encodedData;
	}
	/**
	 * @private
	 * Convert number to the ZQZ format, using pseudo digits.
	 */
	function squeezedDigit(num) {
	  let sqzDigits = '';
	  if (num.startsWith('-')) {
	    sqzDigits += pseudoDigits[SQZ_N][num.charCodeAt(1) - 48];
	    if (num.length > 2) {
	      sqzDigits += num.substring(2);
	    }
	  } else {
	    sqzDigits += pseudoDigits[SQZ_P][num.charCodeAt(0) - 48];
	    if (num.length > 1) {
	      sqzDigits += num.substring(1);
	    }
	  }
	  return sqzDigits;
	}
	/**
	 * Convert number to the DIF format, using pseudo digits.
	 */
	function differenceDigit(num) {
	  let diffDigits = '';
	  if (num.startsWith('-')) {
	    diffDigits += pseudoDigits[DIF_N][num.charCodeAt(1) - 48];
	    if (num.length > 2) {
	      diffDigits += num.substring(2);
	    }
	  } else {
	    diffDigits += pseudoDigits[DIF_P][num.charCodeAt(0) - 48];
	    if (num.length > 1) {
	      diffDigits += num.substring(1);
	    }
	  }
	  return diffDigits;
	}
	/**
	 * Convert number to the DUP format, using pseudo digits.
	 */
	function duplicateDigit(num) {
	  let dupDigits = '';
	  dupDigits += pseudoDigits[DUP][num.charCodeAt(0) - 48];
	  if (num.length > 1) {
	    dupDigits += num.substring(1);
	  }
	  return dupDigits;
	}

	function xyDataCreator(data, options = {}) {
	  const {
	    xyEncoding = 'DIF'
	  } = options;
	  const {
	    xFactor = 1,
	    yFactor = 1
	  } = options.info || {};
	  const firstX = data.x[0];
	  const lastX = data.x[data.x.length - 1];
	  const firstY = data.y[0];
	  const lastY = data.y[data.y.length - 1];
	  const nbPoints = data.x.length;
	  const deltaX = (lastX - firstX) / (nbPoints - 1);
	  const lines = [];
	  lines.push(`##FIRSTX=${firstX}`);
	  lines.push(`##LASTX=${lastX}`);
	  lines.push(`##FIRSTY=${firstY}`);
	  lines.push(`##LASTY=${lastY}`);
	  lines.push(`##DELTAX=${deltaX}`);
	  lines.push(`##XFACTOR=${xFactor}`);
	  lines.push(`##YFACTOR=${yFactor}`);
	  lines.push('##XYDATA=(X++(Y..Y))');
	  const line = vectorEncoder(rescaleAndEnsureInteger(data.y, yFactor), firstX / xFactor, deltaX / xFactor, xyEncoding);
	  if (line) lines.push(line);
	  return lines;
	}

	/**
	 * Create a jcamp
	 * @param data object of array
	 * @param [options={meta:{},info:{}} - metadata object
	 * @returns JCAMP of the input
	 */
	function fromJSON(data, options = {}) {
	  const {
	    meta = {},
	    info = {},
	    xyEncoding
	  } = options;
	  const {
	    title = '',
	    owner = '',
	    origin = '',
	    dataType = '',
	    xUnits = '',
	    yUnits = '',
	    ...resInfo
	  } = info;
	  let {
	    xFactor,
	    yFactor
	  } = info;
	  data = {
	    x: data.x,
	    y: data.y
	  };
	  let header = `##TITLE=${title}
##JCAMP-DX=4.24
##DATA TYPE=${dataType}
##ORIGIN=${origin}
##OWNER=${owner}
##XUNITS=${xUnits}
##YUNITS=${yUnits}\n`;
	  header += addInfoData(resInfo, {
	    prefix: '##'
	  });
	  header += addInfoData(meta);
	  // we leave the header and utf8 fonts ${header.replace(/[^\t\n\x20-\x7F]/g, '')
	  if (xyEncoding) {
	    xFactor = getBestFactor(data.x, {
	      factor: xFactor
	    });
	    yFactor = getBestFactor(data.y, {
	      factor: yFactor
	    });
	    return `${header}##NPOINTS=${data.x.length}
${xyDataCreator(data, {
      info: {
        xFactor,
        yFactor
      },
      xyEncoding
    }).join('\n')}
##END=`;
	  } else {
	    if (xFactor === undefined) xFactor = 1;
	    if (yFactor === undefined) yFactor = 1;
	    if (xFactor !== 1) {
	      //@ts-expect-error xFactor is always defined
	      data.x = data.x.map(value => value / xFactor);
	    }
	    if (yFactor !== 1) {
	      //@ts-expect-error yFactor is always defined
	      data.y = data.y.map(value => value / yFactor);
	    }
	    return `${header}##NPOINTS=${data.x.length}
${peakTableCreator(data, {
      info: {
        xFactor,
        yFactor
      }
    }).join('\n')}
##END=`;
	  }
	}

	/**
	 * Create a jcamp from variables
	 */
	function fromVariables( /** object of variables */
	variables, options = {}) {
	  const {
	    info = {},
	    meta = {},
	    forceNtuples = false
	  } = options;
	  const jcampOptions = {
	    info,
	    meta
	  };
	  const keys = Object.keys(variables).map(key => key.toLowerCase());
	  if (!forceNtuples && keys.length === 2) {
	    const x = variables.x;
	    const xLabel = x.label || 'x';
	    if (variables.x.units) {
	      if (xLabel.includes(variables.x.units)) {
	        jcampOptions.info.xUnits = xLabel;
	      } else {
	        jcampOptions.info.xUnits = `${xLabel} (${variables.x.units})`;
	      }
	    } else {
	      jcampOptions.info.xUnits = xLabel;
	    }
	    const y = variables.y;
	    const yLabel = y.label || 'y';
	    if (variables.y.units) {
	      if (yLabel.includes(variables.y.units)) {
	        jcampOptions.info.xUnits = yLabel;
	      } else {
	        jcampOptions.info.yUnits = `${yLabel} (${variables.y.units})`;
	      }
	    } else {
	      jcampOptions.info.yUnits = yLabel;
	    }
	    const xData = variables.x.data;
	    const yData = variables.y.data;
	    checkNumberOrArray(xData);
	    checkNumberOrArray(yData);
	    return fromJSON({
	      x: xData,
	      y: yData
	    }, jcampOptions);
	  } else {
	    return creatorNtuples(variables, options);
	  }
	}

	function toJcamps(analysis, options = {}) {
	  const jcamps = [];
	  for (const spectrum of analysis.spectra) {
	    jcamps.push(getJcamp(spectrum, options));
	  }
	  return jcamps;
	}
	function getJcamp(spectrum, options) {
	  const {
	    info = {},
	    meta = {}
	  } = options;
	  const jcampOptions = {
	    options: {},
	    info: {
	      title: spectrum.title,
	      dataType: spectrum.dataType,
	      ...info
	    },
	    meta: {
	      ...spectrum.meta,
	      ...meta
	    }
	  };
	  return fromVariables(spectrum.variables, jcampOptions);
	}

	function toJcamp(analysis, options = {}) {
	  return toJcamps(analysis, options).join('\n');
	}

	const JSGraph$1 = {
	  getJSGraph,
	  getNormalizationAnnotations
	};

	/**
	 * @typedef {Object} Component
	 * @property {number} kineticEnergy
	 * @property {number} bindingEnergy
	 * @property {number} intensity
	 * @property {number} kind
	 * @property {number} assignment
	 */

	/**
	 * Creates annotations for jsgraph that allows to display the result of component picking
	 * @param {array<Peak>} components
	 * @param {object} [options={}]
	 * @param {string} [options.fillColor='green']
	 * @param {string} [options.strokeColor='red']
	 * @param {string} [options.showAssignment=true] Display the assignment
	 * @param {function} [options.createFct] (annotation, component) => {}: callback allowing to add properties
	 * @param {string} [options.mode='binding'] 'binding' or 'kinetic'
	 * @returns array
	 */

	function getAnnotations(components, options = {}) {
	  const {
	    fillColor = 'green',
	    strokeColor = 'red',
	    creationFct
	  } = options;
	  let annotations = components.map(component => {
	    let annotation = {
	      line: 1,
	      type: 'rect',
	      strokeColor,
	      strokeWidth: 0,
	      fillColor
	    };
	    if (creationFct) {
	      creationFct(annotation, component);
	    }
	    return getAnnotation(annotation, component, options);
	  });
	  return annotations;
	}
	function getAnnotation(annotation, component, options = {}) {
	  const {
	    showAssignment = true,
	    mode = 'binding'
	  } = options;
	  let labels = [];
	  let line = 0;
	  let energy = mode === 'kinetic' ? component.kineticEnergy.value : component.bindingEnergy.value;
	  let intensity = component.intensity || '20px';
	  if (showAssignment) {
	    labels.push({
	      text: component.assignment,
	      size: '18px',
	      anchor: 'middle',
	      color: 'darkred',
	      position: {
	        x: energy,
	        y: intensity,
	        dy: `${23 + line * 14}px`
	      }
	    });
	    line++;
	  }
	  annotation.labels = labels;
	  annotation.position = [{
	    x: energy,
	    y: intensity,
	    dy: '10px',
	    dx: '-1px'
	  }, {
	    x: energy,
	    y: intensity,
	    dy: '5px',
	    dx: '1px'
	  }];
	  return annotation;
	}

	/* eslint-disable no-unused-vars */
	function parse(text) {
	  const lines = text.split(/\r?\n/);
	  let pointer = 0;
	  let parsed = {
	    header: {},
	    blocks: [],
	    info: {}
	  };
	  pointer = parseHeader(lines, parsed, pointer);
	  for (let i = 0; i < parsed.info.nbBlocks; i++) {
	    pointer = parseBlock(lines, parsed, pointer);
	  }
	  return parsed;
	}
	function parseHeader(lines, parsed, pointer) {
	  const {
	    header,
	    info
	  } = parsed;
	  header['format identifier'] = lines[pointer++];
	  header['institution identifier'] = lines[pointer++];
	  header['instrument model identifier'] = lines[pointer++];
	  header['operator identifier'] = lines[pointer++];
	  header['experiment identifier'] = lines[pointer++];
	  info.nbComments = Number(lines[pointer++]);
	  header['number of lines in comment'] = info.nbComments;
	  const comments = [];
	  for (let i = 0; i < info.nbComments; i++) {
	    comments.push(lines[pointer++]);
	  }
	  header['comment line'] = comments.join('\n');
	  header['experiment mode'] = lines[pointer++];
	  header['scan mode'] = lines[pointer++];
	  if (['MAP', 'MAPD', 'NORM', 'SDP'].includes(header['experiment mode'])) {
	    header['number of spectral regions'] = Number(lines[pointer++]);
	  }
	  if (['MAP', 'MAPD'].includes(header['experiment mode'])) {
	    header['number of analysis positions'] = Number(lines[pointer++]);
	    header['number of discrete x coordinates available in full map'] = Number(lines[pointer++]);
	    header['number of discrete y coordinates available in full map'] = Number(lines[pointer++]);
	  }
	  info.nbExperimentVariables = Number(lines[pointer++]);
	  const experimentVariables = [];
	  header['number of experimental variables'] = info.nbExperimentVariables;
	  for (let i = 0; i < info.nbExperimentVariables; i++) {
	    experimentVariables.push({
	      label: lines[pointer++],
	      unit: lines[pointer++]
	    });
	  }
	  header.experimentVariables = experimentVariables;

	  /*
	    If the values of any of the block parameters are the same in all of the
	    blocks their values may be sent in the first block and then omitted
	    from all subsequent blocks.
	    - n > 0 : the parameters listed are to be included
	    - n < 0 : the parameters listed are to be excluded
	    - n = 0 : all parameters are to be given in all blocks
	    A complete block contains 40 parts.
	    */
	  info.nbEntriesInclusionExclusion = Number(lines[pointer++]);
	  header['number of entries in parameter inclusion or exclusion list'] = info.nbEntriesInclusionExclusion;
	  info.blockParametersincludes = new Array(40).fill(info.nbEntriesInclusionExclusion <= 0);
	  for (let i = 0; i < Math.abs(info.nbEntriesInclusionExclusion); i++) {
	    info.blockParametersincludes[Number(lines[pointer++]) + 1] = info.nbEntriesInclusionExclusion > 0;
	  }
	  header['number of manually entered items in block'] = Number(lines[pointer++]);
	  info.nbFutureUpgradeExperimentEntries = Number(lines[pointer++]);
	  header['number of future upgrade experiment entries'] = info.nbFutureUpgradeExperimentEntries;
	  const futureUpgradeExperimentEntries = [];
	  for (let i = 0; i < info.nbFutureUpgradeExperimentEntries; i++) {
	    futureUpgradeExperimentEntries.push({
	      label: lines[pointer++],
	      unit: lines[pointer++]
	    });
	  }
	  header.futureUpgradeExperimentEntries = futureUpgradeExperimentEntries;
	  if (info.nbFutureUpgradeExperimentEntries !== 0) {
	    throw Error('unsupported future upgrade experiment entries');
	  }
	  header['number of future upgrade block entries'] = Number(lines[pointer++]);
	  if (header['number of future upgrade block entries'] !== 0) {
	    throw Error('unsupported future upgrade block entries');
	  }
	  info.nbBlocks = Number(lines[pointer++]);
	  header['number of blocks'] = info.nbBlocks;
	  return pointer;
	}
	function parseBlock(lines, parsed, pointer) {
	  const {
	    blocks,
	    header,
	    info
	  } = parsed;
	  const firstBlock = blocks[0];
	  const includes = blocks.length === 0 ? new Array(40).fill(true) : info.blockParametersincludes;
	  const block = {};
	  block['block identifier'] = lines[pointer++];
	  block['sample identifier'] = lines[pointer++];
	  block['year in full'] = includes[0] ? Number(lines[pointer++]) : firstBlock['year in full'];
	  block.month = includes[1] ? Number(lines[pointer++]) : firstBlock.month;
	  block['day of month'] = includes[2] ? Number(lines[pointer++]) : firstBlock['day of month'];
	  block.hours = includes[3] ? Number(lines[pointer++]) : firstBlock.hours;
	  block.minutes = includes[4] ? Number(lines[pointer++]) : firstBlock.minutes;
	  block.seconds = includes[5] ? Number(lines[pointer++]) : firstBlock.seconds;
	  block['number of hours in advance of Greenwich Mean Time'] = includes[6] ? Number(lines[pointer++]) : firstBlock['number of hours in advance of Greenwich Mean Time'];
	  if (includes[7]) {
	    const nbComments = Number(lines[pointer++]);
	    block['number of lines in block comment'] = nbComments;
	    const comments = [];
	    for (let i = 0; i < nbComments; i++) {
	      comments.push(lines[pointer++]);
	    }
	    block.blockComment = comments.join('\n');
	  } else {
	    block['number of lines in block comment'] = firstBlock['number of lines in block comment'];
	    block.blockComment = firstBlock.blockComment;
	  }
	  block.technique = includes[8] ? lines[pointer++] : firstBlock.technique;
	  if (['MAP', 'MAPDP'].includes(header['experiment mode'])) {
	    block['x coordinate'] = includes[9] ? Number(lines[pointer++]) : firstBlock['x coordinate'];
	    block['y coordinate'] = includes[9] ? Number(lines[pointer++]) : firstBlock['y coordinate'];
	  }
	  if (includes[10]) {
	    let values = [];
	    for (let i = 0; i < header.experimentVariables.length; i++) {
	      values.push(lines[pointer++]);
	    }
	    block['value of experimental variable'] = values;
	  } else {
	    block['value of experimental variable'] = firstBlock['value of experimental variable'];
	  }
	  block['analysis source label'] = includes[11] ? lines[pointer++] : firstBlock['analysis source label'];
	  if (['MAPDP', 'MAPSVDP', 'SDP', 'SDPSV'].includes(header['experiment mode']) || ['SNMS energy spec', 'FABMS', 'FABMS energy spec', 'ISS', 'SIMS', 'SIMS energy spec', 'SNMS'].includes(block.technique)) {
	    block['sputtering ion or atom atomic number'] = includes[12] ? Number(lines[pointer++]) : firstBlock['sputtering ion or atom atomic number'];
	    block['number of atoms in sputtering ion or atom particle'] = includes[12] ? Number(lines[pointer++]) : firstBlock['number of atoms in sputtering ion or atom particle'];
	    block['sputtering ion or atom charge sign and number'] = includes[12] ? lines[pointer++] : firstBlock['sputtering ion or atom charge sign and number'];
	  }
	  block['analysis source characteristic energy'] = includes[13] ? Number(lines[pointer++]) : firstBlock['analysis source characteristic energy'];
	  block['analysis source strength'] = includes[14] ? Number(lines[pointer++]) : firstBlock['analysis source strength'];
	  block['analysis source beam width x'] = includes[15] ? Number(lines[pointer++]) : firstBlock['analysis source beam width x'];
	  block['analysis source beam width y'] = includes[15] ? Number(lines[pointer++]) : firstBlock['analysis source beam width y'];
	  if (['MAP', 'MAPDP', 'MAPSV', 'MAPSVDP', 'SEM'].includes(header['experiment mode'])) {
	    block['field of view x'] = includes[16] ? Number(lines[pointer++]) : firstBlock['field of view x'];
	    block['field of view y'] = includes[16] ? Number(lines[pointer++]) : firstBlock['field of view y'];
	  }
	  if (['SEM', 'MAPSV', 'MAPSVDP'].includes(header['experiment mode'])) {
	    block['first linescan start x coordinate'] = includes[16] ? Number(lines[pointer++]) : firstBlock['first linescan start x coordinate'];
	    block['first linescan start y coordinate'] = includes[16] ? Number(lines[pointer++]) : firstBlock['first linescan start y coordinate'];
	    block['first linescan finish x coordinate'] = includes[16] ? Number(lines[pointer++]) : firstBlock['first linescan finish x coordinate'];
	    block['first linescan finish y coordinate'] = includes[16] ? Number(lines[pointer++]) : firstBlock['first linescan finish y coordinate'];
	    block['last linescan start x coordinate'] = includes[16] ? Number(lines[pointer++]) : firstBlock['first linescan last x coordinate'];
	    block['last linescan start y coordinate'] = includes[16] ? Number(lines[pointer++]) : firstBlock['first linescan last y coordinate'];
	  }
	  block['analysis source polar angle of incidence'] = includes[18] ? Number(lines[pointer++]) : firstBlock['analysis source polar angle of incidence'];
	  block['analysis source azimuth'] = includes[19] ? Number(lines[pointer++]) : firstBlock['analysis source azimuth'];
	  block['analyser mode'] = includes[20] ? lines[pointer++] : firstBlock['analyser mode'];
	  block['analyser pass energy or retard ratio or mass resolution'] = includes[21] ? Number(lines[pointer++]) : firstBlock['analyser pass energy or retard ratio or mass resolution'];
	  if (block.technique === 'AES diff') {
	    block['differential width'] = includes[22] ? lines[pointer++] : firstBlock['differential width'];
	  }
	  block['magnification of analyser transfer lens'] = includes[23] ? Number(lines[pointer++]) : firstBlock['magnification of analyser transfer lens'];
	  block['analyser work function or acceptance energy of atom or ion'] = includes[24] ? Number(lines[pointer++]) : firstBlock['analyser work function or acceptance energy of atom or ion'];
	  block['target bias'] = includes[25] ? Number(lines[pointer++]) : firstBlock['target bias'];
	  block['analysis width x'] = includes[26] ? Number(lines[pointer++]) : firstBlock['analysis width x'];
	  block['analysis width y'] = includes[26] ? Number(lines[pointer++]) : firstBlock['analysis width y'];
	  block['analyser axis take off polar angle'] = includes[27] ? Number(lines[pointer++]) : firstBlock['analyser axis take off polar angle'];
	  block['analyser axis take off azimuth'] = includes[27] ? Number(lines[pointer++]) : firstBlock['analyser axis take off azimuth'];
	  block['species label'] = includes[28] ? lines[pointer++] : firstBlock['species label'];
	  block['transition or charge state label'] = includes[29] ? lines[pointer++] : firstBlock['transition or charge state label'];
	  block['charge of detected particle'] = includes[29] ? Number(lines[pointer++]) : firstBlock['charge of detected particle'];
	  if (header['scan mode'] !== 'REGULAR') {
	    throw Error('Only REGULAR scans are supported');
	  }
	  block['abscissa label'] = includes[30] ? lines[pointer++] : firstBlock['abscissa label'];
	  block['abscissa units'] = includes[30] ? lines[pointer++] : firstBlock['abscissa units'];
	  block['abscissa start'] = includes[30] ? Number(lines[pointer++]) : firstBlock['abscissa start'];
	  block['abscissa increment'] = includes[30] ? Number(lines[pointer++]) : firstBlock['abscissa increment'];
	  if (includes[31]) {
	    const nbCorrespondingVariables = Number(lines[pointer++]);
	    block['number of corresponding variables'] = nbCorrespondingVariables;
	    const correspondingVariables = [];
	    for (let i = 0; i < nbCorrespondingVariables; i++) {
	      correspondingVariables.push({
	        label: lines[pointer++],
	        unit: lines[pointer++],
	        array: []
	      });
	    }
	    block.correspondingVariables = correspondingVariables;
	  } else {
	    block['number of corresponding variables'] = firstBlock['number of corresponding variables'];
	    block.correspondingVariables = JSON.parse(JSON.stringify(firstBlock.correspondingVariables));
	    block.correspondingVariables.array = [];
	  }
	  block['signal mode'] = includes[32] ? lines[pointer++] : firstBlock['signal mode'];
	  block['signal collection time'] = includes[33] ? Number(lines[pointer++]) : firstBlock['signal collection time'];
	  block['number of scans to compile this block'] = includes[34] ? Number(lines[pointer++]) : firstBlock['number of scans to compile this block'];
	  block['signal time correction'] = includes[35] ? Number(lines[pointer++]) : firstBlock['signal time correction'];
	  if (['MAPDP', 'MAPSVDP', 'SDP', 'SDPSV'].includes(header['experiment mode']) && ['AES diff', 'AES dir', 'EDX', 'ELS', 'UPS', 'XPS', 'XRF'].includes(block.technique)) {
	    block['sputtering source energy'] = includes[36] ? Number(lines[pointer++]) : firstBlock['sputtering source energy'];
	    block['sputtering source beam current'] = includes[36] ? Number(lines[pointer++]) : firstBlock['sputtering source beam current'];
	    block['sputtering source width x'] = includes[36] ? Number(lines[pointer++]) : firstBlock['sputtering source width x'];
	    block['sputtering source width y'] = includes[36] ? Number(lines[pointer++]) : firstBlock['sputtering source width y'];
	    block['sputtering source polar angle of incidence'] = includes[36] ? Number(lines[pointer++]) : firstBlock['sputtering source polar angle of incidence'];
	    block['sputtering source azimuth'] = includes[36] ? Number(lines[pointer++]) : firstBlock['sputtering source azimuth'];
	    block['sputtering mode'] = includes[36] ? lines[pointer++] : firstBlock['sputtering mode'];
	  }
	  block['sample normal polar angle of tilt'] = includes[37] ? Number(lines[pointer++]) : firstBlock['sample normal polar angle of tilt'];
	  block['sample normal tilt azimuth'] = includes[37] ? Number(lines[pointer++]) : firstBlock['sample normal tilt azimuth'];
	  block['sample rotation angle'] = includes[38] ? Number(lines[pointer++]) : firstBlock['sample rotation angle'];
	  if (includes[39]) {
	    const nbAdditionalNumericalParameters = Number(lines[pointer++]);
	    block['number of additional numerical parameters'] = nbAdditionalNumericalParameters;
	    const additionalNumericalParameters = [];
	    for (let i = 0; i < nbAdditionalNumericalParameters; i++) {
	      additionalNumericalParameters.push({
	        label: lines[pointer++],
	        unit: lines[pointer++],
	        value: lines[pointer++]
	      });
	    }
	    block.additionalNumericalParameters = additionalNumericalParameters;
	  } else {
	    block['number of additional numerical parameters'] = firstBlock['number of additional numerical parameters'];
	    block.additionalNumericalParameters = firstBlock.additionalNumericalParameters;
	  }
	  block.nbOrdinateValues = Number(lines[pointer++]);
	  for (let correspondingVariable of block.correspondingVariables) {
	    correspondingVariable.minimumOrdinateValue = Number(lines[pointer++]);
	    correspondingVariable.maximumOrdinateValue = Number(lines[pointer++]);
	  }
	  for (let i = 0; i < block.nbOrdinateValues / block.correspondingVariables.length; i++) {
	    for (let correspondingVariable of block.correspondingVariables) {
	      correspondingVariable.array.push(Number(lines[pointer++]));
	    }
	  }
	  parsed.blocks.push(block);
	  return pointer;
	}

	function appendCalibration(calibrations, line) {
	  let calibration = {};
	  // Calib M = 281.1700 A = 284.8 BE AD
	  let fields = line.match(/Calib (M = .*) (A = [^ ]*) (.*)/);
	  if (!fields) {
	    throw new Error(`appendCalibration fails on: ${line}`);
	  }
	  calibrations.push(calibration);
	}

	function appendComponent(components, line) {
	  // CASA comp (*Mo 3d MoS2 2H*) (*LA(1.53,243)*) Area 230.36971 1e-020 2327991 -1 1 MFWHM 0.88528218 0.2 2 -1 1 Position 1257.22 1257.02 1257.22 -1 1 RSF 10.804667 MASS 95.9219 INDEX -1 (*Mo 3d*) CONST (**) UNCORRECTEDRSF 9.5
	  let component = {};
	  const componentRegex = new RegExp([/CASA comp /, /\((?<name>.*)\) /, /\((?<shape>[^ ]*)\) /, /(?<area>Area .*)/, /(?<fwhm>MFWHM .*)/, /(?<position>Position .*) /, /(?<rsf>RSF .*) /, /(?<mass>MASS .*) /, /(?<index>INDEX .*) /, /(?<const>CONST .*) /, /(?<uncorrectedRSF>UNCORRECTEDRSF.*)/].map(r => r.source).join(''));
	  let fields = line.match(componentRegex);
	  if (!fields) {
	    throw new Error(`appendComponent fails on: ${line}`);
	  }
	  component = {
	    name: fields.groups.name,
	    shape: parseShape(fields.groups.shape),
	    area: parseArea(fields.groups.area),
	    fwhm: parseFWHM(fields.groups.fwhm),
	    position: parsePosition(fields.groups.position),
	    rsf: parseRSF(fields.groups.rsf),
	    mass: parseMass(fields.groups.mass),
	    index: parseIndex(fields.groups.index),
	    const: parseConst(fields.groups.const),
	    uncorrectedRSF: parseUncorrectedRSF(fields.groups.uncorrectedRSF)
	  };
	  components.push(component);
	}
	function parseShape(value) {
	  let parts = value.replace(/[*(),]/g, ' ').trim().split(/ +/);
	  let options = {};
	  let kind;
	  switch (parts[0]) {
	    case 'LA':
	      kind = `${parts[0]}`;
	      options.asymmetry = Number(parts[1]);
	      options.extension = Number(parts[2]);
	      break;
	    case 'GL':
	      kind = `${parts[0]}`;
	      options.unknown = Number(parts[1]);
	      break;
	    default:
	      throw Error(`appendComponent: unknown shape: ${parts[0]}`);
	  }
	  return {
	    kind,
	    options
	  };
	}
	function parseArea(value) {
	  let parts = value.split(' ');
	  return {
	    value: Number(parts[1]),
	    from: Number(parts[2]),
	    to: Number(parts[3]),
	    unknown1: Number(parts[4]),
	    unknown2: Number(parts[5])
	  };
	}
	function parseFWHM(value) {
	  let parts = value.split(' ');
	  return {
	    value: Number(parts[1]),
	    from: Number(parts[2]),
	    to: Number(parts[3]),
	    unknown1: Number(parts[4]),
	    unknown2: Number(parts[5])
	  };
	}
	function parsePosition(value) {
	  let parts = value.split(' ');
	  return {
	    value: Number(parts[1]),
	    from: Number(parts[2]),
	    to: Number(parts[3]),
	    unknown1: Number(parts[4]),
	    unknown2: Number(parts[5])
	  };
	}
	function parseRSF(value) {
	  let parts = value.split(' ');
	  return Number(parts[1]);
	}
	function parseMass(value) {
	  let parts = value.split(' ');
	  return Number(parts[1]);
	}
	function parseIndex(value) {
	  let parts = value.split(' ');
	  return Number(parts[1]);
	}
	function parseConst(value) {
	  return value;
	} // We do not really know what this value means and hence just parse the string
	function parseUncorrectedRSF(value) {
	  let parts = value.split(' ');
	  return Number(parts[1]);
	}

	function appendRegion(regions, line) {
	  // CASA region (*Mo 3d*) (*Shirley*) 1249.3343 1262.7065 10.804667 2 0 0 392.54541 -450 0 0 (*Mo 3d*) 95.9219 0 9.5
	  let fields = line.match(/CASA region \(\*(?<name>.*)\*\) \(\*(?<backgroundKind>.*)\*\) (?<backgroundOptions>.*) \((?<comment>.*)\) (?<surface>.*)/);
	  if (!fields) {
	    throw new Error(`appendRegion fails on: ${line}`);
	  }
	  let region = {
	    name: fields.groups.name,
	    background: {
	      name: fields.groups.backgroundKind,
	      options: fields.groups.backgroundOptions
	    }
	  };
	  regions.push(region);
	}

	function parseCASA(text) {
	  const casa = {
	    regions: [],
	    components: [],
	    calibrations: []
	  };
	  const lines = text.split(/\r?\n/);
	  for (const line of lines) {
	    if (line.startsWith('CASA comp ')) {
	      appendComponent(casa.components, line);
	    }
	    if (line.startsWith('CASA region')) {
	      appendRegion(casa.regions, line);
	    }
	    if (line.startsWith('Calib')) {
	      appendCalibration(casa.calibrations, line);
	    }
	  }
	  return casa;
	}

	function mapComponents(parsedBlockComment, sourceEnergy = undefined, energyUnits = 'eV') {
	  const components = [];
	  if (parsedBlockComment.components) {
	    for (let component of parsedBlockComment.components) {
	      components.push({
	        kineticEnergy: {
	          value: component.position.value,
	          units: energyUnits
	        },
	        bindingEnergy: {
	          value: sourceEnergy - component.position.value,
	          units: energyUnits
	        },
	        assignment: component.name.match(/\*([^]+)\*/)[1],
	        type: component.shape.kind,
	        shapeParameters: {
	          gamma: component.fwhm.value
	        },
	        area: component.area.value
	      });
	    }
	  }
	  return components;
	}

	const XPS_REGEX = /(?<element>[a-zA-Z]{0,3})\s(?<shell>[1-6])(?<angularMomentum>[a-z])/;
	const AUGER_REGEX = /(?<element>[a-zA-Z]{0,3})\s(?<transition>[a-zA-Z]{0,3})/;
	function parseRegion(string) {
	  const result = {
	    name: string.trim(),
	    orbital: {
	      element: null,
	      shell: null,
	      angularMomentum: null
	    },
	    auger: {
	      element: null,
	      transition: null
	    }
	  };
	  let xpsMatches = string.match(XPS_REGEX);
	  let augerMatches = string.match(AUGER_REGEX);
	  if (xpsMatches && xpsMatches.groups.shell) {
	    result.orbital.element = xpsMatches.groups.element;
	    result.orbital.shell = parseInt(xpsMatches.groups.shell, 10);
	    result.orbital.angularMomentum = xpsMatches.groups.angularMomentum;
	  }
	  if (augerMatches && augerMatches.groups.transition) {
	    result.auger.element = augerMatches.groups.element;
	    result.auger.transition = augerMatches.groups.transition;
	  }
	  return result;
	}

	function getNormalizedMeta(meta = {}) {
	  const normalized = {};
	  normalized.region = parseRegion(meta['block identifier']);
	  const energyType = {};
	  energyType.kind = meta['abscissa label'].replace('energy', '').replace(/\s/g, '').toLowerCase();
	  energyType.units = meta['abscissa units'];
	  normalized.energyType = energyType;
	  const source = {};
	  source.label = meta['analysis source label'];
	  source.characteristicEnergy = {
	    value: meta['analysis source characteristic energy'],
	    units: 'eV'
	  };
	  source.beamWidthX = {
	    value: meta['analysis source beam width x'],
	    units: 'um'
	  };
	  source.beamWidthY = {
	    value: meta['analysis source beam width y'],
	    units: 'um'
	  };
	  normalized.analysisSource = source;
	  normalized.speciesLabel = meta['species label'];
	  normalized.components = mapComponents(parseCASA(meta.blockComment), source.characteristicEnergy.value);
	  const increment = meta['abscissa increment'];
	  normalized.from = meta['abscissa start'];
	  normalized.to = meta['abscissa start'] + increment * (meta.nbOrdinateValues - 1);
	  normalized.analyserMode = meta['analyser mode'];
	  return normalized;
	}

	/**
	 * Returns an Analysis from a VAMAS text file
	 * @param {arrayBuffer|string} [text] the vamas text file
	 */
	function fromVamas(text) {
	  text = ensureString(text);
	  let parsed = parse(text);
	  let header = parsed.header;
	  let blocks = parsed.blocks;
	  let title = header['experiment identifier'];
	  let analysis = new Analysis();
	  for (let block of blocks) {
	    let sourceEnergy = block['analysis source characteristic energy'];
	    let yVariable = block.correspondingVariables[0];
	    let yLabel = yVariable.label;
	    let yValues = yVariable.array;
	    let xLabel = block['abscissa label'];
	    let xUnits = block['abscissa units'];
	    let xStart = block['abscissa start'];
	    let xIncrement = block['abscissa increment'];
	    let xValues = new Float64Array(yValues.length);
	    for (let i = 0; i < yValues.length; i++) {
	      xValues[i] = xStart + xIncrement * i;
	    }
	    // currently we take the first corresponding variables

	    let meta = {};
	    for (let key in block) {
	      if (typeof block[key] === 'string' || typeof block[key] === 'number') {
	        meta[key] = block[key];
	      }
	    }
	    for (let key in header) {
	      if (typeof header[key] === 'string' || typeof header[key] === 'number') {
	        meta[key] = header[key];
	      }
	    }
	    meta.cheminfo = {
	      meta: getNormalizedMeta(meta)
	    };
	    const variables = {};
	    if (xLabel === 'Kinetic energy' && sourceEnergy) {
	      // we will calculate binding energy
	      variables.x = {
	        data: xValues.map(value => sourceEnergy - value).reverse(),
	        label: 'Binding energy',
	        units: xUnits,
	        type: 'DEPENDENT'
	      };
	      variables.y = {
	        data: yValues.reverse(),
	        label: yLabel,
	        type: 'DEPENDENT'
	      };
	      variables.k = {
	        data: xValues.reverse(),
	        label: xLabel,
	        units: xUnits,
	        type: 'INDEPENDENT'
	      };
	    } else {
	      variables.x = {
	        data: xValues,
	        label: xLabel,
	        units: xUnits,
	        type: 'INDEPENDENT'
	      };
	      variables.y = {
	        data: yValues,
	        label: yLabel,
	        type: 'DEPENDENT'
	      };
	    }
	    analysis.pushSpectrum(variables, {
	      dataType: 'XPS',
	      title,
	      meta
	    });
	  }
	  return analysis;
	}

	/**
	 * Allow
	 * @param {object} spectrum
	 * @param {object} [options={}]
	 * @param {object} [options.gsd={}] global spectra deconvolution options. Available options: http://mljs.github.io/global-spectral-deconvolution/#gsd
	 * @return {array} Array of peaks
	 */

	function peakPicking(spectrum, options = {}) {
	  const {
	    gsd: gsdOptions = {
	      realTopDetection: true
	    }
	  } = options;
	  let data = {
	    x: spectrum.variables.x.data,
	    y: spectrum.variables.y.data
	  };
	  let peaks = gsd(data, gsdOptions);
	  return peaks;
	}

	/**
	 * Array of object containing element, orbital and be
	 */
	const references = [{
	  element: 'H',
	  orbital: '1s',
	  be: 14
	}, {
	  element: 'He',
	  orbital: '1s',
	  be: 25
	}, {
	  element: 'Li',
	  orbital: '1s',
	  be: 55
	}, {
	  element: 'Be',
	  orbital: '1s',
	  be: 111
	}, {
	  element: 'B',
	  orbital: '1s',
	  be: 188
	}, {
	  element: 'C',
	  orbital: '1s',
	  be: 284
	}, {
	  element: 'N',
	  orbital: '1s',
	  be: 399
	}, {
	  element: 'O',
	  orbital: '1s',
	  be: 532
	}, {
	  element: 'F',
	  orbital: '1s',
	  be: 686
	}, {
	  element: 'Ne',
	  orbital: '1s',
	  be: 867
	}, {
	  element: 'Na',
	  orbital: '1s',
	  be: 1072
	}, {
	  element: 'O',
	  orbital: '2s',
	  be: 24
	}, {
	  element: 'F',
	  orbital: '2s',
	  be: 31
	}, {
	  element: 'Ne',
	  orbital: '2s',
	  be: 45
	}, {
	  element: 'Na',
	  orbital: '2s',
	  be: 63
	}, {
	  element: 'Mg',
	  orbital: '2s',
	  be: 89
	}, {
	  element: 'Al',
	  orbital: '2s',
	  be: 118
	}, {
	  element: 'Si',
	  orbital: '2s',
	  be: 149
	}, {
	  element: 'P',
	  orbital: '2s',
	  be: 189
	}, {
	  element: 'S',
	  orbital: '2s',
	  be: 229
	}, {
	  element: 'Cl',
	  orbital: '2s',
	  be: 270
	}, {
	  element: 'Ar',
	  orbital: '2s',
	  be: 320
	}, {
	  element: 'K',
	  orbital: '2s',
	  be: 377
	}, {
	  element: 'Ca',
	  orbital: '2s',
	  be: 438
	}, {
	  element: 'Sc',
	  orbital: '2s',
	  be: 500
	}, {
	  element: 'Ti',
	  orbital: '2s',
	  be: 564
	}, {
	  element: 'V',
	  orbital: '2s',
	  be: 628
	}, {
	  element: 'Cr',
	  orbital: '2s',
	  be: 695
	}, {
	  element: 'Mn',
	  orbital: '2s',
	  be: 769
	}, {
	  element: 'Fe',
	  orbital: '2s',
	  be: 846
	}, {
	  element: 'Co',
	  orbital: '2s',
	  be: 926
	}, {
	  element: 'Ni',
	  orbital: '2s',
	  be: 1008
	}, {
	  element: 'Cu',
	  orbital: '2s',
	  be: 1096
	}, {
	  element: 'Zn',
	  orbital: '2s',
	  be: 1194
	}, {
	  element: 'B',
	  orbital: '2p1/2',
	  be: 5
	}, {
	  element: 'C',
	  orbital: '2p1/2',
	  be: 7
	}, {
	  element: 'N',
	  orbital: '2p1/2',
	  be: 9
	}, {
	  element: 'O',
	  orbital: '2p1/2',
	  be: 7
	}, {
	  element: 'F',
	  orbital: '2p1/2',
	  be: 9
	}, {
	  element: 'Ne',
	  orbital: '2p1/2',
	  be: 18
	}, {
	  element: 'Na',
	  orbital: '2p1/2',
	  be: 31
	}, {
	  element: 'Mg',
	  orbital: '2p1/2',
	  be: 52
	}, {
	  element: 'Al',
	  orbital: '2p1/2',
	  be: 74
	}, {
	  element: 'Si',
	  orbital: '2p1/2',
	  be: 100
	}, {
	  element: 'P',
	  orbital: '2p1/2',
	  be: 136
	}, {
	  element: 'S',
	  orbital: '2p1/2',
	  be: 165
	}, {
	  element: 'Cl',
	  orbital: '2p1/2',
	  be: 202
	}, {
	  element: 'Ar',
	  orbital: '2p1/2',
	  be: 247
	}, {
	  element: 'K',
	  orbital: '2p1/2',
	  be: 297
	}, {
	  element: 'Ca',
	  orbital: '2p1/2',
	  be: 350
	}, {
	  element: 'Sc',
	  orbital: '2p1/2',
	  be: 407
	}, {
	  element: 'Ti',
	  orbital: '2p1/2',
	  be: 461
	}, {
	  element: 'V',
	  orbital: '2p1/2',
	  be: 520
	}, {
	  element: 'Cr',
	  orbital: '2p1/2',
	  be: 584
	}, {
	  element: 'Mn',
	  orbital: '2p1/2',
	  be: 652
	}, {
	  element: 'Fe',
	  orbital: '2p1/2',
	  be: 723
	}, {
	  element: 'Co',
	  orbital: '2p1/2',
	  be: 794
	}, {
	  element: 'Ni',
	  orbital: '2p1/2',
	  be: 872
	}, {
	  element: 'Cu',
	  orbital: '2p1/2',
	  be: 951
	}, {
	  element: 'Zn',
	  orbital: '2p1/2',
	  be: 1044
	}, {
	  element: 'Ga',
	  orbital: '2p1/2',
	  be: 1143
	}, {
	  element: 'Ge',
	  orbital: '2p1/2',
	  be: 1249
	}, {
	  element: 'B',
	  orbital: '2p3/2',
	  be: 5
	}, {
	  element: 'C',
	  orbital: '2p3/2',
	  be: 7
	}, {
	  element: 'N',
	  orbital: '2p3/2',
	  be: 9
	}, {
	  element: 'O',
	  orbital: '2p3/2',
	  be: 7
	}, {
	  element: 'F',
	  orbital: '2p3/2',
	  be: 9
	}, {
	  element: 'Ne',
	  orbital: '2p3/2',
	  be: 18
	}, {
	  element: 'Na',
	  orbital: '2p3/2',
	  be: 31
	}, {
	  element: 'Mg',
	  orbital: '2p3/2',
	  be: 52
	}, {
	  element: 'Al',
	  orbital: '2p3/2',
	  be: 73
	}, {
	  element: 'Si',
	  orbital: '2p3/2',
	  be: 99
	}, {
	  element: 'P',
	  orbital: '2p3/2',
	  be: 135
	}, {
	  element: 'S',
	  orbital: '2p3/2',
	  be: 164
	}, {
	  element: 'Cl',
	  orbital: '2p3/2',
	  be: 200
	}, {
	  element: 'Ar',
	  orbital: '2p3/2',
	  be: 245
	}, {
	  element: 'K',
	  orbital: '2p3/2',
	  be: 294
	}, {
	  element: 'Ca',
	  orbital: '2p3/2',
	  be: 347
	}, {
	  element: 'Sc',
	  orbital: '2p3/2',
	  be: 402
	}, {
	  element: 'Ti',
	  orbital: '2p3/2',
	  be: 455
	}, {
	  element: 'V',
	  orbital: '2p3/2',
	  be: 513
	}, {
	  element: 'Cr',
	  orbital: '2p3/2',
	  be: 575
	}, {
	  element: 'Mn',
	  orbital: '2p3/2',
	  be: 641
	}, {
	  element: 'Fe',
	  orbital: '2p3/2',
	  be: 710
	}, {
	  element: 'Co',
	  orbital: '2p3/2',
	  be: 779
	}, {
	  element: 'Ni',
	  orbital: '2p3/2',
	  be: 855
	}, {
	  element: 'Cu',
	  orbital: '2p3/2',
	  be: 931
	}, {
	  element: 'Zn',
	  orbital: '2p3/2',
	  be: 1021
	}, {
	  element: 'Ga',
	  orbital: '2p3/2',
	  be: 1116
	}, {
	  element: 'Ge',
	  orbital: '2p3/2',
	  be: 1217
	}, {
	  element: 'B',
	  orbital: '2p',
	  be: 5
	}, {
	  element: 'C',
	  orbital: '2p',
	  be: 7
	}, {
	  element: 'N',
	  orbital: '2p',
	  be: 9
	}, {
	  element: 'O',
	  orbital: '2p',
	  be: 7
	}, {
	  element: 'F',
	  orbital: '2p',
	  be: 9
	}, {
	  element: 'Ne',
	  orbital: '2p',
	  be: 18
	}, {
	  element: 'Na',
	  orbital: '2p',
	  be: 31
	}, {
	  element: 'Mg',
	  orbital: '2p',
	  be: 52
	}, {
	  element: 'Al',
	  orbital: '2p',
	  be: 73
	}, {
	  element: 'Si',
	  orbital: '2p',
	  be: 99
	}, {
	  element: 'P',
	  orbital: '2p',
	  be: 135
	}, {
	  element: 'S',
	  orbital: '2p',
	  be: 164
	}, {
	  element: 'Cl',
	  orbital: '2p',
	  be: 200
	}, {
	  element: 'Ar',
	  orbital: '2p',
	  be: 245
	}, {
	  element: 'K',
	  orbital: '2p',
	  be: 294
	}, {
	  element: 'Ca',
	  orbital: '2p',
	  be: 347
	}, {
	  element: 'Sc',
	  orbital: '2p',
	  be: 402
	}, {
	  element: 'Ti',
	  orbital: '2p',
	  be: 455
	}, {
	  element: 'V',
	  orbital: '2p',
	  be: 513
	}, {
	  element: 'Cr',
	  orbital: '2p',
	  be: 575
	}, {
	  element: 'Mn',
	  orbital: '2p',
	  be: 641
	}, {
	  element: 'Fe',
	  orbital: '2p',
	  be: 710
	}, {
	  element: 'Co',
	  orbital: '2p',
	  be: 779
	}, {
	  element: 'Ni',
	  orbital: '2p',
	  be: 855
	}, {
	  element: 'Cu',
	  orbital: '2p',
	  be: 931
	}, {
	  element: 'Zn',
	  orbital: '2p',
	  be: 1021
	}, {
	  element: 'Ga',
	  orbital: '2p',
	  be: 1116
	}, {
	  element: 'Ge',
	  orbital: '2p',
	  be: 1217
	}, {
	  element: 'Na',
	  orbital: '3s',
	  be: 1
	}, {
	  element: 'Mg',
	  orbital: '3s',
	  be: 2
	}, {
	  element: 'Al',
	  orbital: '3s',
	  be: 1
	}, {
	  element: 'Si',
	  orbital: '3s',
	  be: 8
	}, {
	  element: 'P',
	  orbital: '3s',
	  be: 16
	}, {
	  element: 'S',
	  orbital: '3s',
	  be: 16
	}, {
	  element: 'Cl',
	  orbital: '3s',
	  be: 18
	}, {
	  element: 'Ar',
	  orbital: '3s',
	  be: 25
	}, {
	  element: 'K',
	  orbital: '3s',
	  be: 34
	}, {
	  element: 'Ca',
	  orbital: '3s',
	  be: 44
	}, {
	  element: 'Sc',
	  orbital: '3s',
	  be: 54
	}, {
	  element: 'Ti',
	  orbital: '3s',
	  be: 59
	}, {
	  element: 'V',
	  orbital: '3s',
	  be: 66
	}, {
	  element: 'Cr',
	  orbital: '3s',
	  be: 74
	}, {
	  element: 'Mn',
	  orbital: '3s',
	  be: 84
	}, {
	  element: 'Fe',
	  orbital: '3s',
	  be: 95
	}, {
	  element: 'Co',
	  orbital: '3s',
	  be: 101
	}, {
	  element: 'Ni',
	  orbital: '3s',
	  be: 112
	}, {
	  element: 'Cu',
	  orbital: '3s',
	  be: 120
	}, {
	  element: 'Zn',
	  orbital: '3s',
	  be: 137
	}, {
	  element: 'Ga',
	  orbital: '3s',
	  be: 158
	}, {
	  element: 'Ge',
	  orbital: '3s',
	  be: 181
	}, {
	  element: 'As',
	  orbital: '3s',
	  be: 204
	}, {
	  element: 'Se',
	  orbital: '3s',
	  be: 232
	}, {
	  element: 'Br',
	  orbital: '3s',
	  be: 257
	}, {
	  element: 'Kr',
	  orbital: '3s',
	  be: 289
	}, {
	  element: 'Rb',
	  orbital: '3s',
	  be: 322
	}, {
	  element: 'Sr',
	  orbital: '3s',
	  be: 358
	}, {
	  element: 'Y',
	  orbital: '3s',
	  be: 395
	}, {
	  element: 'Zr',
	  orbital: '3s',
	  be: 431
	}, {
	  element: 'Nb',
	  orbital: '3s',
	  be: 469
	}, {
	  element: 'Mo',
	  orbital: '3s',
	  be: 505
	}, {
	  element: 'Tc',
	  orbital: '3s',
	  be: 544
	}, {
	  element: 'Ru',
	  orbital: '3s',
	  be: 585
	}, {
	  element: 'Rh',
	  orbital: '3s',
	  be: 627
	}, {
	  element: 'Pd',
	  orbital: '3s',
	  be: 670
	}, {
	  element: 'Ag',
	  orbital: '3s',
	  be: 717
	}, {
	  element: 'Cd',
	  orbital: '3s',
	  be: 770
	}, {
	  element: 'In',
	  orbital: '3s',
	  be: 826
	}, {
	  element: 'Sn',
	  orbital: '3s',
	  be: 884
	}, {
	  element: 'Sb',
	  orbital: '3s',
	  be: 944
	}, {
	  element: 'Te',
	  orbital: '3s',
	  be: 1006
	}, {
	  element: 'I',
	  orbital: '3s',
	  be: 1072
	}, {
	  element: 'Xe',
	  orbital: '3s',
	  be: 1145
	}, {
	  element: 'Cs',
	  orbital: '3s',
	  be: 1217
	}, {
	  element: 'Si',
	  orbital: '3p1/2',
	  be: 3
	}, {
	  element: 'P',
	  orbital: '3p1/2',
	  be: 10
	}, {
	  element: 'S',
	  orbital: '3p1/2',
	  be: 8
	}, {
	  element: 'Cl',
	  orbital: '3p1/2',
	  be: 7
	}, {
	  element: 'Ar',
	  orbital: '3p1/2',
	  be: 12
	}, {
	  element: 'K',
	  orbital: '3p1/2',
	  be: 18
	}, {
	  element: 'Ca',
	  orbital: '3p1/2',
	  be: 26
	}, {
	  element: 'Sc',
	  orbital: '3p1/2',
	  be: 32
	}, {
	  element: 'Ti',
	  orbital: '3p1/2',
	  be: 34
	}, {
	  element: 'V',
	  orbital: '3p1/2',
	  be: 38
	}, {
	  element: 'Cr',
	  orbital: '3p1/2',
	  be: 43
	}, {
	  element: 'Mn',
	  orbital: '3p1/2',
	  be: 49
	}, {
	  element: 'Fe',
	  orbital: '3p1/2',
	  be: 56
	}, {
	  element: 'Co',
	  orbital: '3p1/2',
	  be: 60
	}, {
	  element: 'Ni',
	  orbital: '3p1/2',
	  be: 68
	}, {
	  element: 'Cu',
	  orbital: '3p1/2',
	  be: 74
	}, {
	  element: 'Zn',
	  orbital: '3p1/2',
	  be: 87
	}, {
	  element: 'Ga',
	  orbital: '3p1/2',
	  be: 107
	}, {
	  element: 'Ge',
	  orbital: '3p1/2',
	  be: 129
	}, {
	  element: 'As',
	  orbital: '3p1/2',
	  be: 147
	}, {
	  element: 'Se',
	  orbital: '3p1/2',
	  be: 168
	}, {
	  element: 'Br',
	  orbital: '3p1/2',
	  be: 189
	}, {
	  element: 'Kr',
	  orbital: '3p1/2',
	  be: 223
	}, {
	  element: 'Rb',
	  orbital: '3p1/2',
	  be: 248
	}, {
	  element: 'Sr',
	  orbital: '3p1/2',
	  be: 280
	}, {
	  element: 'Y',
	  orbital: '3p1/2',
	  be: 313
	}, {
	  element: 'Zr',
	  orbital: '3p1/2',
	  be: 345
	}, {
	  element: 'Nb',
	  orbital: '3p1/2',
	  be: 379
	}, {
	  element: 'Mo',
	  orbital: '3p1/2',
	  be: 410
	}, {
	  element: 'Tc',
	  orbital: '3p1/2',
	  be: 445
	}, {
	  element: 'Ru',
	  orbital: '3p1/2',
	  be: 483
	}, {
	  element: 'Rh',
	  orbital: '3p1/2',
	  be: 521
	}, {
	  element: 'Pd',
	  orbital: '3p1/2',
	  be: 559
	}, {
	  element: 'Ag',
	  orbital: '3p1/2',
	  be: 602
	}, {
	  element: 'Cd',
	  orbital: '3p1/2',
	  be: 651
	}, {
	  element: 'In',
	  orbital: '3p1/2',
	  be: 702
	}, {
	  element: 'Sn',
	  orbital: '3p1/2',
	  be: 757
	}, {
	  element: 'Sb',
	  orbital: '3p1/2',
	  be: 812
	}, {
	  element: 'Te',
	  orbital: '3p1/2',
	  be: 870
	}, {
	  element: 'I',
	  orbital: '3p1/2',
	  be: 931
	}, {
	  element: 'Xe',
	  orbital: '3p1/2',
	  be: 999
	}, {
	  element: 'Cs',
	  orbital: '3p1/2',
	  be: 1065
	}, {
	  element: 'Ba',
	  orbital: '3p1/2',
	  be: 1137
	}, {
	  element: 'La',
	  orbital: '3p1/2',
	  be: 1205
	}, {
	  element: 'Si',
	  orbital: '3p3/2',
	  be: 3
	}, {
	  element: 'P',
	  orbital: '3p3/2',
	  be: 10
	}, {
	  element: 'S',
	  orbital: '3p3/2',
	  be: 8
	}, {
	  element: 'Cl',
	  orbital: '3p3/2',
	  be: 7
	}, {
	  element: 'Ar',
	  orbital: '3p3/2',
	  be: 12
	}, {
	  element: 'K',
	  orbital: '3p3/2',
	  be: 18
	}, {
	  element: 'Ca',
	  orbital: '3p3/2',
	  be: 26
	}, {
	  element: 'Sc',
	  orbital: '3p3/2',
	  be: 32
	}, {
	  element: 'Ti',
	  orbital: '3p3/2',
	  be: 34
	}, {
	  element: 'V',
	  orbital: '3p3/2',
	  be: 38
	}, {
	  element: 'Cr',
	  orbital: '3p3/2',
	  be: 43
	}, {
	  element: 'Mn',
	  orbital: '3p3/2',
	  be: 49
	}, {
	  element: 'Fe',
	  orbital: '3p3/2',
	  be: 56
	}, {
	  element: 'Co',
	  orbital: '3p3/2',
	  be: 60
	}, {
	  element: 'Ni',
	  orbital: '3p3/2',
	  be: 68
	}, {
	  element: 'Cu',
	  orbital: '3p3/2',
	  be: 74
	}, {
	  element: 'Zn',
	  orbital: '3p3/2',
	  be: 87
	}, {
	  element: 'Ga',
	  orbital: '3p3/2',
	  be: 103
	}, {
	  element: 'Ge',
	  orbital: '3p3/2',
	  be: 122
	}, {
	  element: 'As',
	  orbital: '3p3/2',
	  be: 141
	}, {
	  element: 'Se',
	  orbital: '3p3/2',
	  be: 162
	}, {
	  element: 'Br',
	  orbital: '3p3/2',
	  be: 182
	}, {
	  element: 'Kr',
	  orbital: '3p3/2',
	  be: 214
	}, {
	  element: 'Rb',
	  orbital: '3p3/2',
	  be: 239
	}, {
	  element: 'Sr',
	  orbital: '3p3/2',
	  be: 269
	}, {
	  element: 'Y',
	  orbital: '3p3/2',
	  be: 301
	}, {
	  element: 'Zr',
	  orbital: '3p3/2',
	  be: 331
	}, {
	  element: 'Nb',
	  orbital: '3p3/2',
	  be: 363
	}, {
	  element: 'Mo',
	  orbital: '3p3/2',
	  be: 393
	}, {
	  element: 'Tc',
	  orbital: '3p3/2',
	  be: 425
	}, {
	  element: 'Ru',
	  orbital: '3p3/2',
	  be: 461
	}, {
	  element: 'Rh',
	  orbital: '3p3/2',
	  be: 496
	}, {
	  element: 'Pd',
	  orbital: '3p3/2',
	  be: 531
	}, {
	  element: 'Ag',
	  orbital: '3p3/2',
	  be: 571
	}, {
	  element: 'Cd',
	  orbital: '3p3/2',
	  be: 617
	}, {
	  element: 'In',
	  orbital: '3p3/2',
	  be: 664
	}, {
	  element: 'Sn',
	  orbital: '3p3/2',
	  be: 715
	}, {
	  element: 'Sb',
	  orbital: '3p3/2',
	  be: 766
	}, {
	  element: 'Te',
	  orbital: '3p3/2',
	  be: 819
	}, {
	  element: 'I',
	  orbital: '3p3/2',
	  be: 875
	}, {
	  element: 'Xe',
	  orbital: '3p3/2',
	  be: 937
	}, {
	  element: 'Cs',
	  orbital: '3p3/2',
	  be: 998
	}, {
	  element: 'Ba',
	  orbital: '3p3/2',
	  be: 1063
	}, {
	  element: 'La',
	  orbital: '3p3/2',
	  be: 1124
	}, {
	  element: 'Ce',
	  orbital: '3p3/2',
	  be: 1186
	}, {
	  element: 'Pr',
	  orbital: '3p3/2',
	  be: 1243
	}, {
	  element: 'Si',
	  orbital: '3p',
	  be: 3
	}, {
	  element: 'P',
	  orbital: '3p',
	  be: 10
	}, {
	  element: 'S',
	  orbital: '3p',
	  be: 8
	}, {
	  element: 'Cl',
	  orbital: '3p',
	  be: 7
	}, {
	  element: 'Ar',
	  orbital: '3p',
	  be: 12
	}, {
	  element: 'K',
	  orbital: '3p',
	  be: 18
	}, {
	  element: 'Ca',
	  orbital: '3p',
	  be: 26
	}, {
	  element: 'Sc',
	  orbital: '3p',
	  be: 32
	}, {
	  element: 'Ti',
	  orbital: '3p',
	  be: 34
	}, {
	  element: 'V',
	  orbital: '3p',
	  be: 38
	}, {
	  element: 'Cr',
	  orbital: '3p',
	  be: 43
	}, {
	  element: 'Mn',
	  orbital: '3p',
	  be: 49
	}, {
	  element: 'Fe',
	  orbital: '3p',
	  be: 56
	}, {
	  element: 'Co',
	  orbital: '3p',
	  be: 60
	}, {
	  element: 'Ni',
	  orbital: '3p',
	  be: 68
	}, {
	  element: 'Cu',
	  orbital: '3p',
	  be: 74
	}, {
	  element: 'Zn',
	  orbital: '3p',
	  be: 87
	}, {
	  element: 'Ga',
	  orbital: '3p',
	  be: 103
	}, {
	  element: 'Ge',
	  orbital: '3p',
	  be: 122
	}, {
	  element: 'As',
	  orbital: '3p',
	  be: 141
	}, {
	  element: 'Se',
	  orbital: '3p',
	  be: 162
	}, {
	  element: 'Br',
	  orbital: '3p',
	  be: 182
	}, {
	  element: 'Kr',
	  orbital: '3p',
	  be: 214
	}, {
	  element: 'Rb',
	  orbital: '3p',
	  be: 239
	}, {
	  element: 'Sr',
	  orbital: '3p',
	  be: 269
	}, {
	  element: 'Y',
	  orbital: '3p',
	  be: 301
	}, {
	  element: 'Zr',
	  orbital: '3p',
	  be: 331
	}, {
	  element: 'Nb',
	  orbital: '3p',
	  be: 363
	}, {
	  element: 'Mo',
	  orbital: '3p',
	  be: 393
	}, {
	  element: 'Tc',
	  orbital: '3p',
	  be: 425
	}, {
	  element: 'Ru',
	  orbital: '3p',
	  be: 461
	}, {
	  element: 'Rh',
	  orbital: '3p',
	  be: 496
	}, {
	  element: 'Pd',
	  orbital: '3p',
	  be: 531
	}, {
	  element: 'Ag',
	  orbital: '3p',
	  be: 571
	}, {
	  element: 'Cd',
	  orbital: '3p',
	  be: 617
	}, {
	  element: 'In',
	  orbital: '3p',
	  be: 664
	}, {
	  element: 'Sn',
	  orbital: '3p',
	  be: 715
	}, {
	  element: 'Sb',
	  orbital: '3p',
	  be: 766
	}, {
	  element: 'Te',
	  orbital: '3p',
	  be: 819
	}, {
	  element: 'I',
	  orbital: '3p',
	  be: 875
	}, {
	  element: 'Xe',
	  orbital: '3p',
	  be: 937
	}, {
	  element: 'Cs',
	  orbital: '3p',
	  be: 998
	}, {
	  element: 'Ba',
	  orbital: '3p',
	  be: 1063
	}, {
	  element: 'La',
	  orbital: '3p',
	  be: 1124
	}, {
	  element: 'Ce',
	  orbital: '3p',
	  be: 1186
	}, {
	  element: 'Pr',
	  orbital: '3p',
	  be: 1243
	}, {
	  element: 'Sc',
	  orbital: '3d3/2',
	  be: 7
	}, {
	  element: 'Ti',
	  orbital: '3d3/2',
	  be: 3
	}, {
	  element: 'V',
	  orbital: '3d3/2',
	  be: 2
	}, {
	  element: 'Cr',
	  orbital: '3d3/2',
	  be: 2
	}, {
	  element: 'Mn',
	  orbital: '3d3/2',
	  be: 4
	}, {
	  element: 'Fe',
	  orbital: '3d3/2',
	  be: 6
	}, {
	  element: 'Co',
	  orbital: '3d3/2',
	  be: 3
	}, {
	  element: 'Ni',
	  orbital: '3d3/2',
	  be: 4
	}, {
	  element: 'Cu',
	  orbital: '3d3/2',
	  be: 2
	}, {
	  element: 'Zn',
	  orbital: '3d3/2',
	  be: 9
	}, {
	  element: 'Ga',
	  orbital: '3d3/2',
	  be: 18
	}, {
	  element: 'Ge',
	  orbital: '3d3/2',
	  be: 29
	}, {
	  element: 'As',
	  orbital: '3d3/2',
	  be: 41
	}, {
	  element: 'Se',
	  orbital: '3d3/2',
	  be: 57
	}, {
	  element: 'Br',
	  orbital: '3d3/2',
	  be: 70
	}, {
	  element: 'Kr',
	  orbital: '3d3/2',
	  be: 89
	}, {
	  element: 'Rb',
	  orbital: '3d3/2',
	  be: 112
	}, {
	  element: 'Sr',
	  orbital: '3d3/2',
	  be: 135
	}, {
	  element: 'Y',
	  orbital: '3d3/2',
	  be: 160
	}, {
	  element: 'Zr',
	  orbital: '3d3/2',
	  be: 183
	}, {
	  element: 'Nb',
	  orbital: '3d3/2',
	  be: 208
	}, {
	  element: 'Mo',
	  orbital: '3d3/2',
	  be: 230
	}, {
	  element: 'Tc',
	  orbital: '3d3/2',
	  be: 257
	}, {
	  element: 'Ru',
	  orbital: '3d3/2',
	  be: 284
	}, {
	  element: 'Rh',
	  orbital: '3d3/2',
	  be: 312
	}, {
	  element: 'Pd',
	  orbital: '3d3/2',
	  be: 340
	}, {
	  element: 'Ag',
	  orbital: '3d3/2',
	  be: 373
	}, {
	  element: 'Cd',
	  orbital: '3d3/2',
	  be: 411
	}, {
	  element: 'In',
	  orbital: '3d3/2',
	  be: 451
	}, {
	  element: 'Sn',
	  orbital: '3d3/2',
	  be: 494
	}, {
	  element: 'Sb',
	  orbital: '3d3/2',
	  be: 537
	}, {
	  element: 'Te',
	  orbital: '3d3/2',
	  be: 582
	}, {
	  element: 'I',
	  orbital: '3d3/2',
	  be: 631
	}, {
	  element: 'Xe',
	  orbital: '3d3/2',
	  be: 685
	}, {
	  element: 'Cs',
	  orbital: '3d3/2',
	  be: 740
	}, {
	  element: 'Ba',
	  orbital: '3d3/2',
	  be: 796
	}, {
	  element: 'La',
	  orbital: '3d3/2',
	  be: 849
	}, {
	  element: 'Ce',
	  orbital: '3d3/2',
	  be: 902
	}, {
	  element: 'Pr',
	  orbital: '3d3/2',
	  be: 951
	}, {
	  element: 'Nd',
	  orbital: '3d3/2',
	  be: 1000
	}, {
	  element: 'Pm',
	  orbital: '3d3/2',
	  be: 1052
	}, {
	  element: 'Sm',
	  orbital: '3d3/2',
	  be: 1107
	}, {
	  element: 'Eu',
	  orbital: '3d3/2',
	  be: 1161
	}, {
	  element: 'Gd',
	  orbital: '3d3/2',
	  be: 1218
	}, {
	  element: 'Sc',
	  orbital: '3d5/2',
	  be: 7
	}, {
	  element: 'Ti',
	  orbital: '3d5/2',
	  be: 3
	}, {
	  element: 'V',
	  orbital: '3d5/2',
	  be: 2
	}, {
	  element: 'Cr',
	  orbital: '3d5/2',
	  be: 2
	}, {
	  element: 'Mn',
	  orbital: '3d5/2',
	  be: 4
	}, {
	  element: 'Fe',
	  orbital: '3d5/2',
	  be: 6
	}, {
	  element: 'Co',
	  orbital: '3d5/2',
	  be: 3
	}, {
	  element: 'Ni',
	  orbital: '3d5/2',
	  be: 4
	}, {
	  element: 'Cu',
	  orbital: '3d5/2',
	  be: 2
	}, {
	  element: 'Zn',
	  orbital: '3d5/2',
	  be: 9
	}, {
	  element: 'Ga',
	  orbital: '3d5/2',
	  be: 18
	}, {
	  element: 'Ge',
	  orbital: '3d5/2',
	  be: 29
	}, {
	  element: 'As',
	  orbital: '3d5/2',
	  be: 41
	}, {
	  element: 'Se',
	  orbital: '3d5/2',
	  be: 57
	}, {
	  element: 'Br',
	  orbital: '3d5/2',
	  be: 69
	}, {
	  element: 'Kr',
	  orbital: '3d5/2',
	  be: 89
	}, {
	  element: 'Rb',
	  orbital: '3d5/2',
	  be: 111
	}, {
	  element: 'Sr',
	  orbital: '3d5/2',
	  be: 133
	}, {
	  element: 'Y',
	  orbital: '3d5/2',
	  be: 158
	}, {
	  element: 'Zr',
	  orbital: '3d5/2',
	  be: 180
	}, {
	  element: 'Nb',
	  orbital: '3d5/2',
	  be: 205
	}, {
	  element: 'Mo',
	  orbital: '3d5/2',
	  be: 227
	}, {
	  element: 'Tc',
	  orbital: '3d5/2',
	  be: 253
	}, {
	  element: 'Ru',
	  orbital: '3d5/2',
	  be: 279
	}, {
	  element: 'Rh',
	  orbital: '3d5/2',
	  be: 307
	}, {
	  element: 'Pd',
	  orbital: '3d5/2',
	  be: 335
	}, {
	  element: 'Ag',
	  orbital: '3d5/2',
	  be: 367
	}, {
	  element: 'Cd',
	  orbital: '3d5/2',
	  be: 404
	}, {
	  element: 'In',
	  orbital: '3d5/2',
	  be: 443
	}, {
	  element: 'Sn',
	  orbital: '3d5/2',
	  be: 485
	}, {
	  element: 'Sb',
	  orbital: '3d5/2',
	  be: 528
	}, {
	  element: 'Te',
	  orbital: '3d5/2',
	  be: 572
	}, {
	  element: 'I',
	  orbital: '3d5/2',
	  be: 620
	}, {
	  element: 'Xe',
	  orbital: '3d5/2',
	  be: 672
	}, {
	  element: 'Cs',
	  orbital: '3d5/2',
	  be: 726
	}, {
	  element: 'Ba',
	  orbital: '3d5/2',
	  be: 781
	}, {
	  element: 'La',
	  orbital: '3d5/2',
	  be: 832
	}, {
	  element: 'Ce',
	  orbital: '3d5/2',
	  be: 884
	}, {
	  element: 'Pr',
	  orbital: '3d5/2',
	  be: 931
	}, {
	  element: 'Nd',
	  orbital: '3d5/2',
	  be: 978
	}, {
	  element: 'Pm',
	  orbital: '3d5/2',
	  be: 1027
	}, {
	  element: 'Sm',
	  orbital: '3d5/2',
	  be: 1081
	}, {
	  element: 'Eu',
	  orbital: '3d5/2',
	  be: 1131
	}, {
	  element: 'Gd',
	  orbital: '3d5/2',
	  be: 1186
	}, {
	  element: 'Tb',
	  orbital: '3d5/2',
	  be: 1242
	}, {
	  element: 'Sc',
	  orbital: '3d',
	  be: 7
	}, {
	  element: 'Ti',
	  orbital: '3d',
	  be: 3
	}, {
	  element: 'V',
	  orbital: '3d',
	  be: 2
	}, {
	  element: 'Cr',
	  orbital: '3d',
	  be: 2
	}, {
	  element: 'Mn',
	  orbital: '3d',
	  be: 4
	}, {
	  element: 'Fe',
	  orbital: '3d',
	  be: 6
	}, {
	  element: 'Co',
	  orbital: '3d',
	  be: 3
	}, {
	  element: 'Ni',
	  orbital: '3d',
	  be: 4
	}, {
	  element: 'Cu',
	  orbital: '3d',
	  be: 2
	}, {
	  element: 'Zn',
	  orbital: '3d',
	  be: 9
	}, {
	  element: 'Ga',
	  orbital: '3d',
	  be: 18
	}, {
	  element: 'Ge',
	  orbital: '3d',
	  be: 29
	}, {
	  element: 'As',
	  orbital: '3d',
	  be: 41
	}, {
	  element: 'Se',
	  orbital: '3d',
	  be: 57
	}, {
	  element: 'Br',
	  orbital: '3d',
	  be: 69
	}, {
	  element: 'Kr',
	  orbital: '3d',
	  be: 89
	}, {
	  element: 'Rb',
	  orbital: '3d',
	  be: 111
	}, {
	  element: 'Sr',
	  orbital: '3d',
	  be: 133
	}, {
	  element: 'Y',
	  orbital: '3d',
	  be: 158
	}, {
	  element: 'Zr',
	  orbital: '3d',
	  be: 180
	}, {
	  element: 'Nb',
	  orbital: '3d',
	  be: 205
	}, {
	  element: 'Mo',
	  orbital: '3d',
	  be: 227
	}, {
	  element: 'Tc',
	  orbital: '3d',
	  be: 253
	}, {
	  element: 'Ru',
	  orbital: '3d',
	  be: 279
	}, {
	  element: 'Rh',
	  orbital: '3d',
	  be: 307
	}, {
	  element: 'Pd',
	  orbital: '3d',
	  be: 335
	}, {
	  element: 'Ag',
	  orbital: '3d',
	  be: 367
	}, {
	  element: 'Cd',
	  orbital: '3d',
	  be: 404
	}, {
	  element: 'In',
	  orbital: '3d',
	  be: 443
	}, {
	  element: 'Sn',
	  orbital: '3d',
	  be: 485
	}, {
	  element: 'Sb',
	  orbital: '3d',
	  be: 528
	}, {
	  element: 'Te',
	  orbital: '3d',
	  be: 572
	}, {
	  element: 'I',
	  orbital: '3d',
	  be: 620
	}, {
	  element: 'Xe',
	  orbital: '3d',
	  be: 672
	}, {
	  element: 'Cs',
	  orbital: '3d',
	  be: 726
	}, {
	  element: 'Ba',
	  orbital: '3d',
	  be: 781
	}, {
	  element: 'La',
	  orbital: '3d',
	  be: 832
	}, {
	  element: 'Ce',
	  orbital: '3d',
	  be: 884
	}, {
	  element: 'Pr',
	  orbital: '3d',
	  be: 931
	}, {
	  element: 'Nd',
	  orbital: '3d',
	  be: 978
	}, {
	  element: 'Pm',
	  orbital: '3d',
	  be: 1027
	}, {
	  element: 'Sm',
	  orbital: '3d',
	  be: 1081
	}, {
	  element: 'Eu',
	  orbital: '3d',
	  be: 1131
	}, {
	  element: 'Gd',
	  orbital: '3d',
	  be: 1186
	}, {
	  element: 'Tb',
	  orbital: '3d',
	  be: 1242
	}, {
	  element: 'Br',
	  orbital: '4s',
	  be: 27
	}, {
	  element: 'Kr',
	  orbital: '4s',
	  be: 24
	}, {
	  element: 'Rb',
	  orbital: '4s',
	  be: 30
	}, {
	  element: 'Sr',
	  orbital: '4s',
	  be: 38
	}, {
	  element: 'Y',
	  orbital: '4s',
	  be: 46
	}, {
	  element: 'Zr',
	  orbital: '4s',
	  be: 52
	}, {
	  element: 'Nb',
	  orbital: '4s',
	  be: 58
	}, {
	  element: 'Mo',
	  orbital: '4s',
	  be: 62
	}, {
	  element: 'Tc',
	  orbital: '4s',
	  be: 68
	}, {
	  element: 'Ru',
	  orbital: '4s',
	  be: 75
	}, {
	  element: 'Rh',
	  orbital: '4s',
	  be: 81
	}, {
	  element: 'Pd',
	  orbital: '4s',
	  be: 86
	}, {
	  element: 'Ag',
	  orbital: '4s',
	  be: 95
	}, {
	  element: 'Cd',
	  orbital: '4s',
	  be: 108
	}, {
	  element: 'In',
	  orbital: '4s',
	  be: 122
	}, {
	  element: 'Sn',
	  orbital: '4s',
	  be: 137
	}, {
	  element: 'Sb',
	  orbital: '4s',
	  be: 152
	}, {
	  element: 'Te',
	  orbital: '4s',
	  be: 168
	}, {
	  element: 'I',
	  orbital: '4s',
	  be: 186
	}, {
	  element: 'Xe',
	  orbital: '4s',
	  be: 208
	}, {
	  element: 'Cs',
	  orbital: '4s',
	  be: 231
	}, {
	  element: 'Ba',
	  orbital: '4s',
	  be: 253
	}, {
	  element: 'La',
	  orbital: '4s',
	  be: 271
	}, {
	  element: 'Ce',
	  orbital: '4s',
	  be: 290
	}, {
	  element: 'Pr',
	  orbital: '4s',
	  be: 305
	}, {
	  element: 'Nd',
	  orbital: '4s',
	  be: 316
	}, {
	  element: 'Pm',
	  orbital: '4s',
	  be: 331
	}, {
	  element: 'Sm',
	  orbital: '4s',
	  be: 347
	}, {
	  element: 'Eu',
	  orbital: '4s',
	  be: 360
	}, {
	  element: 'Gd',
	  orbital: '4s',
	  be: 376
	}, {
	  element: 'Tb',
	  orbital: '4s',
	  be: 398
	}, {
	  element: 'Dy',
	  orbital: '4s',
	  be: 416
	}, {
	  element: 'Ho',
	  orbital: '4s',
	  be: 436
	}, {
	  element: 'Er',
	  orbital: '4s',
	  be: 449
	}, {
	  element: 'Tm',
	  orbital: '4s',
	  be: 472
	}, {
	  element: 'Yb',
	  orbital: '4s',
	  be: 487
	}, {
	  element: 'Lu',
	  orbital: '4s',
	  be: 506
	}, {
	  element: 'Hf',
	  orbital: '4s',
	  be: 538
	}, {
	  element: 'Ta',
	  orbital: '4s',
	  be: 566
	}, {
	  element: 'W',
	  orbital: '4s',
	  be: 595
	}, {
	  element: 'Re',
	  orbital: '4s',
	  be: 625
	}, {
	  element: 'Os',
	  orbital: '4s',
	  be: 655
	}, {
	  element: 'Ir',
	  orbital: '4s',
	  be: 690
	}, {
	  element: 'Pt',
	  orbital: '4s',
	  be: 724
	}, {
	  element: 'Au',
	  orbital: '4s',
	  be: 759
	}, {
	  element: 'Hg',
	  orbital: '4s',
	  be: 800
	}, {
	  element: 'Tl',
	  orbital: '4s',
	  be: 846
	}, {
	  element: 'Pb',
	  orbital: '4s',
	  be: 894
	}, {
	  element: 'Bi',
	  orbital: '4s',
	  be: 939
	}, {
	  element: 'Po',
	  orbital: '4s',
	  be: 995
	}, {
	  element: 'At',
	  orbital: '4s',
	  be: 1042
	}, {
	  element: 'Rn',
	  orbital: '4s',
	  be: 1097
	}, {
	  element: 'Fr',
	  orbital: '4s',
	  be: 1153
	}, {
	  element: 'Ra',
	  orbital: '4s',
	  be: 1208
	}, {
	  element: 'U',
	  orbital: '4s',
	  be: 1043
	}, {
	  element: 'Ga',
	  orbital: '4p1/2',
	  be: 1
	}, {
	  element: 'Ge',
	  orbital: '4p1/2',
	  be: 3
	}, {
	  element: 'As',
	  orbital: '4p1/2',
	  be: 3
	}, {
	  element: 'Se',
	  orbital: '4p1/2',
	  be: 6
	}, {
	  element: 'Br',
	  orbital: '4p1/2',
	  be: 5
	}, {
	  element: 'Kr',
	  orbital: '4p1/2',
	  be: 11
	}, {
	  element: 'Rb',
	  orbital: '4p1/2',
	  be: 15
	}, {
	  element: 'Sr',
	  orbital: '4p1/2',
	  be: 20
	}, {
	  element: 'Y',
	  orbital: '4p1/2',
	  be: 26
	}, {
	  element: 'Zr',
	  orbital: '4p1/2',
	  be: 29
	}, {
	  element: 'Nb',
	  orbital: '4p1/2',
	  be: 34
	}, {
	  element: 'Mo',
	  orbital: '4p1/2',
	  be: 35
	}, {
	  element: 'Tc',
	  orbital: '4p1/2',
	  be: 39
	}, {
	  element: 'Ru',
	  orbital: '4p1/2',
	  be: 43
	}, {
	  element: 'Rh',
	  orbital: '4p1/2',
	  be: 48
	}, {
	  element: 'Pd',
	  orbital: '4p1/2',
	  be: 51
	}, {
	  element: 'Ag',
	  orbital: '4p1/2',
	  be: 62
	}, {
	  element: 'Cd',
	  orbital: '4p1/2',
	  be: 67
	}, {
	  element: 'In',
	  orbital: '4p1/2',
	  be: 77
	}, {
	  element: 'Sn',
	  orbital: '4p1/2',
	  be: 89
	}, {
	  element: 'Sb',
	  orbital: '4p1/2',
	  be: 99
	}, {
	  element: 'Te',
	  orbital: '4p1/2',
	  be: 110
	}, {
	  element: 'I',
	  orbital: '4p1/2',
	  be: 123
	}, {
	  element: 'Xe',
	  orbital: '4p1/2',
	  be: 147
	}, {
	  element: 'Cs',
	  orbital: '4p1/2',
	  be: 172
	}, {
	  element: 'Ba',
	  orbital: '4p1/2',
	  be: 192
	}, {
	  element: 'La',
	  orbital: '4p1/2',
	  be: 206
	}, {
	  element: 'Ce',
	  orbital: '4p1/2',
	  be: 224
	}, {
	  element: 'Pr',
	  orbital: '4p1/2',
	  be: 237
	}, {
	  element: 'Nd',
	  orbital: '4p1/2',
	  be: 244
	}, {
	  element: 'Pm',
	  orbital: '4p1/2',
	  be: 255
	}, {
	  element: 'Sm',
	  orbital: '4p1/2',
	  be: 267
	}, {
	  element: 'Eu',
	  orbital: '4p1/2',
	  be: 284
	}, {
	  element: 'Gd',
	  orbital: '4p1/2',
	  be: 289
	}, {
	  element: 'Tb',
	  orbital: '4p1/2',
	  be: 311
	}, {
	  element: 'Dy',
	  orbital: '4p1/2',
	  be: 332
	}, {
	  element: 'Ho',
	  orbital: '4p1/2',
	  be: 343
	}, {
	  element: 'Er',
	  orbital: '4p1/2',
	  be: 366
	}, {
	  element: 'Tm',
	  orbital: '4p1/2',
	  be: 386
	}, {
	  element: 'Yb',
	  orbital: '4p1/2',
	  be: 396
	}, {
	  element: 'Lu',
	  orbital: '4p1/2',
	  be: 410
	}, {
	  element: 'Hf',
	  orbital: '4p1/2',
	  be: 437
	}, {
	  element: 'Ta',
	  orbital: '4p1/2',
	  be: 465
	}, {
	  element: 'W',
	  orbital: '4p1/2',
	  be: 492
	}, {
	  element: 'Re',
	  orbital: '4p1/2',
	  be: 518
	}, {
	  element: 'Os',
	  orbital: '4p1/2',
	  be: 547
	}, {
	  element: 'Ir',
	  orbital: '4p1/2',
	  be: 577
	}, {
	  element: 'Pt',
	  orbital: '4p1/2',
	  be: 608
	}, {
	  element: 'Au',
	  orbital: '4p1/2',
	  be: 644
	}, {
	  element: 'Hg',
	  orbital: '4p1/2',
	  be: 677
	}, {
	  element: 'Tl',
	  orbital: '4p1/2',
	  be: 722
	}, {
	  element: 'Pb',
	  orbital: '4p1/2',
	  be: 764
	}, {
	  element: 'Bi',
	  orbital: '4p1/2',
	  be: 806
	}, {
	  element: 'Po',
	  orbital: '4p1/2',
	  be: 851
	}, {
	  element: 'At',
	  orbital: '4p1/2',
	  be: 886
	}, {
	  element: 'Rn',
	  orbital: '4p1/2',
	  be: 929
	}, {
	  element: 'Fr',
	  orbital: '4p1/2',
	  be: 980
	}, {
	  element: 'Ra',
	  orbital: '4p1/2',
	  be: 1058
	}, {
	  element: 'Ac',
	  orbital: '4p1/2',
	  be: 1080
	}, {
	  element: 'Th',
	  orbital: '4p1/2',
	  be: 1168
	}, {
	  element: 'Ga',
	  orbital: '4p3/2',
	  be: 1
	}, {
	  element: 'Ge',
	  orbital: '4p3/2',
	  be: 3
	}, {
	  element: 'As',
	  orbital: '4p3/2',
	  be: 3
	}, {
	  element: 'Se',
	  orbital: '4p3/2',
	  be: 6
	}, {
	  element: 'Br',
	  orbital: '4p3/2',
	  be: 5
	}, {
	  element: 'Kr',
	  orbital: '4p3/2',
	  be: 11
	}, {
	  element: 'Rb',
	  orbital: '4p3/2',
	  be: 14
	}, {
	  element: 'Sr',
	  orbital: '4p3/2',
	  be: 20
	}, {
	  element: 'Y',
	  orbital: '4p3/2',
	  be: 26
	}, {
	  element: 'Zr',
	  orbital: '4p3/2',
	  be: 29
	}, {
	  element: 'Nb',
	  orbital: '4p3/2',
	  be: 34
	}, {
	  element: 'Mo',
	  orbital: '4p3/2',
	  be: 35
	}, {
	  element: 'Tc',
	  orbital: '4p3/2',
	  be: 39
	}, {
	  element: 'Ru',
	  orbital: '4p3/2',
	  be: 43
	}, {
	  element: 'Rh',
	  orbital: '4p3/2',
	  be: 48
	}, {
	  element: 'Pd',
	  orbital: '4p3/2',
	  be: 56
	}, {
	  element: 'Ag',
	  orbital: '4p3/2',
	  be: 56
	}, {
	  element: 'Cd',
	  orbital: '4p3/2',
	  be: 67
	}, {
	  element: 'In',
	  orbital: '4p3/2',
	  be: 77
	}, {
	  element: 'Sn',
	  orbital: '4p3/2',
	  be: 89
	}, {
	  element: 'Sb',
	  orbital: '4p3/2',
	  be: 99
	}, {
	  element: 'Te',
	  orbital: '4p3/2',
	  be: 110
	}, {
	  element: 'I',
	  orbital: '4p3/2',
	  be: 123
	}, {
	  element: 'Xe',
	  orbital: '4p3/2',
	  be: 147
	}, {
	  element: 'Cs',
	  orbital: '4p3/2',
	  be: 162
	}, {
	  element: 'Ba',
	  orbital: '4p3/2',
	  be: 180
	}, {
	  element: 'La',
	  orbital: '4p3/2',
	  be: 192
	}, {
	  element: 'Ce',
	  orbital: '4p3/2',
	  be: 208
	}, {
	  element: 'Pr',
	  orbital: '4p3/2',
	  be: 218
	}, {
	  element: 'Nd',
	  orbital: '4p3/2',
	  be: 225
	}, {
	  element: 'Pm',
	  orbital: '4p3/2',
	  be: 237
	}, {
	  element: 'Sm',
	  orbital: '4p3/2',
	  be: 249
	}, {
	  element: 'Eu',
	  orbital: '4p3/2',
	  be: 257
	}, {
	  element: 'Gd',
	  orbital: '4p3/2',
	  be: 271
	}, {
	  element: 'Tb',
	  orbital: '4p3/2',
	  be: 286
	}, {
	  element: 'Dy',
	  orbital: '4p3/2',
	  be: 293
	}, {
	  element: 'Ho',
	  orbital: '4p3/2',
	  be: 306
	}, {
	  element: 'Er',
	  orbital: '4p3/2',
	  be: 320
	}, {
	  element: 'Tm',
	  orbital: '4p3/2',
	  be: 337
	}, {
	  element: 'Yb',
	  orbital: '4p3/2',
	  be: 343
	}, {
	  element: 'Lu',
	  orbital: '4p3/2',
	  be: 359
	}, {
	  element: 'Hf',
	  orbital: '4p3/2',
	  be: 380
	}, {
	  element: 'Ta',
	  orbital: '4p3/2',
	  be: 405
	}, {
	  element: 'W',
	  orbital: '4p3/2',
	  be: 426
	}, {
	  element: 'Re',
	  orbital: '4p3/2',
	  be: 445
	}, {
	  element: 'Os',
	  orbital: '4p3/2',
	  be: 469
	}, {
	  element: 'Ir',
	  orbital: '4p3/2',
	  be: 495
	}, {
	  element: 'Pt',
	  orbital: '4p3/2',
	  be: 519
	}, {
	  element: 'Au',
	  orbital: '4p3/2',
	  be: 546
	}, {
	  element: 'Hg',
	  orbital: '4p3/2',
	  be: 571
	}, {
	  element: 'Tl',
	  orbital: '4p3/2',
	  be: 609
	}, {
	  element: 'Pb',
	  orbital: '4p3/2',
	  be: 645
	}, {
	  element: 'Bi',
	  orbital: '4p3/2',
	  be: 679
	}, {
	  element: 'Po',
	  orbital: '4p3/2',
	  be: 705
	}, {
	  element: 'At',
	  orbital: '4p3/2',
	  be: 740
	}, {
	  element: 'Rn',
	  orbital: '4p3/2',
	  be: 768
	}, {
	  element: 'Fr',
	  orbital: '4p3/2',
	  be: 810
	}, {
	  element: 'Ra',
	  orbital: '4p3/2',
	  be: 879
	}, {
	  element: 'Ac',
	  orbital: '4p3/2',
	  be: 890
	}, {
	  element: 'Th',
	  orbital: '4p3/2',
	  be: 968
	}, {
	  element: 'Ga',
	  orbital: '4p',
	  be: 1
	}, {
	  element: 'Ge',
	  orbital: '4p',
	  be: 3
	}, {
	  element: 'As',
	  orbital: '4p',
	  be: 3
	}, {
	  element: 'Se',
	  orbital: '4p',
	  be: 6
	}, {
	  element: 'Br',
	  orbital: '4p',
	  be: 5
	}, {
	  element: 'Kr',
	  orbital: '4p',
	  be: 11
	}, {
	  element: 'Rb',
	  orbital: '4p',
	  be: 14
	}, {
	  element: 'Sr',
	  orbital: '4p',
	  be: 20
	}, {
	  element: 'Y',
	  orbital: '4p',
	  be: 26
	}, {
	  element: 'Zr',
	  orbital: '4p',
	  be: 29
	}, {
	  element: 'Nb',
	  orbital: '4p',
	  be: 34
	}, {
	  element: 'Mo',
	  orbital: '4p',
	  be: 35
	}, {
	  element: 'Tc',
	  orbital: '4p',
	  be: 39
	}, {
	  element: 'Ru',
	  orbital: '4p',
	  be: 43
	}, {
	  element: 'Rh',
	  orbital: '4p',
	  be: 48
	}, {
	  element: 'Pd',
	  orbital: '4p',
	  be: 56
	}, {
	  element: 'Ag',
	  orbital: '4p',
	  be: 56
	}, {
	  element: 'Cd',
	  orbital: '4p',
	  be: 67
	}, {
	  element: 'In',
	  orbital: '4p',
	  be: 77
	}, {
	  element: 'Sn',
	  orbital: '4p',
	  be: 89
	}, {
	  element: 'Sb',
	  orbital: '4p',
	  be: 99
	}, {
	  element: 'Te',
	  orbital: '4p',
	  be: 110
	}, {
	  element: 'I',
	  orbital: '4p',
	  be: 123
	}, {
	  element: 'Xe',
	  orbital: '4p',
	  be: 147
	}, {
	  element: 'Cs',
	  orbital: '4p',
	  be: 162
	}, {
	  element: 'Ba',
	  orbital: '4p',
	  be: 180
	}, {
	  element: 'La',
	  orbital: '4p',
	  be: 192
	}, {
	  element: 'Ce',
	  orbital: '4p',
	  be: 208
	}, {
	  element: 'Pr',
	  orbital: '4p',
	  be: 218
	}, {
	  element: 'Nd',
	  orbital: '4p',
	  be: 225
	}, {
	  element: 'Pm',
	  orbital: '4p',
	  be: 237
	}, {
	  element: 'Sm',
	  orbital: '4p',
	  be: 249
	}, {
	  element: 'Eu',
	  orbital: '4p',
	  be: 257
	}, {
	  element: 'Gd',
	  orbital: '4p',
	  be: 271
	}, {
	  element: 'Tb',
	  orbital: '4p',
	  be: 286
	}, {
	  element: 'Dy',
	  orbital: '4p',
	  be: 293
	}, {
	  element: 'Ho',
	  orbital: '4p',
	  be: 306
	}, {
	  element: 'Er',
	  orbital: '4p',
	  be: 320
	}, {
	  element: 'Tm',
	  orbital: '4p',
	  be: 337
	}, {
	  element: 'Yb',
	  orbital: '4p',
	  be: 343
	}, {
	  element: 'Lu',
	  orbital: '4p',
	  be: 359
	}, {
	  element: 'Hf',
	  orbital: '4p',
	  be: 380
	}, {
	  element: 'Ta',
	  orbital: '4p',
	  be: 405
	}, {
	  element: 'W',
	  orbital: '4p',
	  be: 426
	}, {
	  element: 'Re',
	  orbital: '4p',
	  be: 445
	}, {
	  element: 'Os',
	  orbital: '4p',
	  be: 469
	}, {
	  element: 'Ir',
	  orbital: '4p',
	  be: 495
	}, {
	  element: 'Pt',
	  orbital: '4p',
	  be: 519
	}, {
	  element: 'Au',
	  orbital: '4p',
	  be: 546
	}, {
	  element: 'Hg',
	  orbital: '4p',
	  be: 571
	}, {
	  element: 'Tl',
	  orbital: '4p',
	  be: 609
	}, {
	  element: 'Pb',
	  orbital: '4p',
	  be: 645
	}, {
	  element: 'Bi',
	  orbital: '4p',
	  be: 679
	}, {
	  element: 'Po',
	  orbital: '4p',
	  be: 705
	}, {
	  element: 'At',
	  orbital: '4p',
	  be: 740
	}, {
	  element: 'Rn',
	  orbital: '4p',
	  be: 768
	}, {
	  element: 'Fr',
	  orbital: '4p',
	  be: 810
	}, {
	  element: 'Ra',
	  orbital: '4p',
	  be: 879
	}, {
	  element: 'Ac',
	  orbital: '4p',
	  be: 890
	}, {
	  element: 'Th',
	  orbital: '4p',
	  be: 968
	}, {
	  element: 'Y',
	  orbital: '4d3/2',
	  be: 3
	}, {
	  element: 'Zr',
	  orbital: '4d3/2',
	  be: 3
	}, {
	  element: 'Nb',
	  orbital: '4d3/2',
	  be: 4
	}, {
	  element: 'Mo',
	  orbital: '4d3/2',
	  be: 2
	}, {
	  element: 'Tc',
	  orbital: '4d3/2',
	  be: 2
	}, {
	  element: 'Ru',
	  orbital: '4d3/2',
	  be: 2
	}, {
	  element: 'Rh',
	  orbital: '4d3/2',
	  be: 3
	}, {
	  element: 'Pd',
	  orbital: '4d3/2',
	  be: 1
	}, {
	  element: 'Ag',
	  orbital: '4d3/2',
	  be: 3
	}, {
	  element: 'Cd',
	  orbital: '4d3/2',
	  be: 9
	}, {
	  element: 'In',
	  orbital: '4d3/2',
	  be: 16
	}, {
	  element: 'Sn',
	  orbital: '4d3/2',
	  be: 24
	}, {
	  element: 'Sb',
	  orbital: '4d3/2',
	  be: 32
	}, {
	  element: 'Te',
	  orbital: '4d3/2',
	  be: 40
	}, {
	  element: 'I',
	  orbital: '4d3/2',
	  be: 50
	}, {
	  element: 'Xe',
	  orbital: '4d3/2',
	  be: 63
	}, {
	  element: 'Cs',
	  orbital: '4d3/2',
	  be: 79
	}, {
	  element: 'Ba',
	  orbital: '4d3/2',
	  be: 93
	}, {
	  element: 'La',
	  orbital: '4d3/2',
	  be: 99
	}, {
	  element: 'Ce',
	  orbital: '4d3/2',
	  be: 111
	}, {
	  element: 'Pr',
	  orbital: '4d3/2',
	  be: 114
	}, {
	  element: 'Nd',
	  orbital: '4d3/2',
	  be: 118
	}, {
	  element: 'Pm',
	  orbital: '4d3/2',
	  be: 121
	}, {
	  element: 'Sm',
	  orbital: '4d3/2',
	  be: 130
	}, {
	  element: 'Eu',
	  orbital: '4d3/2',
	  be: 134
	}, {
	  element: 'Gd',
	  orbital: '4d3/2',
	  be: 141
	}, {
	  element: 'Tb',
	  orbital: '4d3/2',
	  be: 148
	}, {
	  element: 'Dy',
	  orbital: '4d3/2',
	  be: 154
	}, {
	  element: 'Ho',
	  orbital: '4d3/2',
	  be: 161
	}, {
	  element: 'Er',
	  orbital: '4d3/2',
	  be: 177
	}, {
	  element: 'Tm',
	  orbital: '4d3/2',
	  be: 180
	}, {
	  element: 'Yb',
	  orbital: '4d3/2',
	  be: 197
	}, {
	  element: 'Lu',
	  orbital: '4d3/2',
	  be: 205
	}, {
	  element: 'Hf',
	  orbital: '4d3/2',
	  be: 224
	}, {
	  element: 'Ta',
	  orbital: '4d3/2',
	  be: 242
	}, {
	  element: 'W',
	  orbital: '4d3/2',
	  be: 259
	}, {
	  element: 'Re',
	  orbital: '4d3/2',
	  be: 274
	}, {
	  element: 'Os',
	  orbital: '4d3/2',
	  be: 290
	}, {
	  element: 'Ir',
	  orbital: '4d3/2',
	  be: 312
	}, {
	  element: 'Pt',
	  orbital: '4d3/2',
	  be: 331
	}, {
	  element: 'Au',
	  orbital: '4d3/2',
	  be: 352
	}, {
	  element: 'Hg',
	  orbital: '4d3/2',
	  be: 379
	}, {
	  element: 'Tl',
	  orbital: '4d3/2',
	  be: 407
	}, {
	  element: 'Pb',
	  orbital: '4d3/2',
	  be: 435
	}, {
	  element: 'Bi',
	  orbital: '4d3/2',
	  be: 464
	}, {
	  element: 'Po',
	  orbital: '4d3/2',
	  be: 500
	}, {
	  element: 'At',
	  orbital: '4d3/2',
	  be: 533
	}, {
	  element: 'Rn',
	  orbital: '4d3/2',
	  be: 567
	}, {
	  element: 'Fr',
	  orbital: '4d3/2',
	  be: 603
	}, {
	  element: 'Ra',
	  orbital: '4d3/2',
	  be: 636
	}, {
	  element: 'Ac',
	  orbital: '4d3/2',
	  be: 675
	}, {
	  element: 'Th',
	  orbital: '4d3/2',
	  be: 714
	}, {
	  element: 'U',
	  orbital: '4d3/2',
	  be: 781
	}, {
	  element: 'Y',
	  orbital: '4d5/2',
	  be: 3
	}, {
	  element: 'Zr',
	  orbital: '4d5/2',
	  be: 3
	}, {
	  element: 'Nb',
	  orbital: '4d5/2',
	  be: 4
	}, {
	  element: 'Mo',
	  orbital: '4d5/2',
	  be: 2
	}, {
	  element: 'Tc',
	  orbital: '4d5/2',
	  be: 2
	}, {
	  element: 'Ru',
	  orbital: '4d5/2',
	  be: 2
	}, {
	  element: 'Rh',
	  orbital: '4d5/2',
	  be: 3
	}, {
	  element: 'Pd',
	  orbital: '4d5/2',
	  be: 1
	}, {
	  element: 'Ag',
	  orbital: '4d5/2',
	  be: 3
	}, {
	  element: 'Cd',
	  orbital: '4d5/2',
	  be: 9
	}, {
	  element: 'In',
	  orbital: '4d5/2',
	  be: 16
	}, {
	  element: 'Sn',
	  orbital: '4d5/2',
	  be: 24
	}, {
	  element: 'Sb',
	  orbital: '4d5/2',
	  be: 32
	}, {
	  element: 'Te',
	  orbital: '4d5/2',
	  be: 40
	}, {
	  element: 'I',
	  orbital: '4d5/2',
	  be: 50
	}, {
	  element: 'Xe',
	  orbital: '4d5/2',
	  be: 63
	}, {
	  element: 'Cs',
	  orbital: '4d5/2',
	  be: 77
	}, {
	  element: 'Ba',
	  orbital: '4d5/2',
	  be: 90
	}, {
	  element: 'La',
	  orbital: '4d5/2',
	  be: 99
	}, {
	  element: 'Ce',
	  orbital: '4d5/2',
	  be: 111
	}, {
	  element: 'Pr',
	  orbital: '4d5/2',
	  be: 114
	}, {
	  element: 'Nd',
	  orbital: '4d5/2',
	  be: 118
	}, {
	  element: 'Pm',
	  orbital: '4d5/2',
	  be: 121
	}, {
	  element: 'Sm',
	  orbital: '4d5/2',
	  be: 130
	}, {
	  element: 'Eu',
	  orbital: '4d5/2',
	  be: 134
	}, {
	  element: 'Gd',
	  orbital: '4d5/2',
	  be: 141
	}, {
	  element: 'Tb',
	  orbital: '4d5/2',
	  be: 148
	}, {
	  element: 'Dy',
	  orbital: '4d5/2',
	  be: 154
	}, {
	  element: 'Ho',
	  orbital: '4d5/2',
	  be: 161
	}, {
	  element: 'Er',
	  orbital: '4d5/2',
	  be: 168
	}, {
	  element: 'Tm',
	  orbital: '4d5/2',
	  be: 180
	}, {
	  element: 'Yb',
	  orbital: '4d5/2',
	  be: 184
	}, {
	  element: 'Lu',
	  orbital: '4d5/2',
	  be: 195
	}, {
	  element: 'Hf',
	  orbital: '4d5/2',
	  be: 214
	}, {
	  element: 'Ta',
	  orbital: '4d5/2',
	  be: 230
	}, {
	  element: 'W',
	  orbital: '4d5/2',
	  be: 246
	}, {
	  element: 'Re',
	  orbital: '4d5/2',
	  be: 260
	}, {
	  element: 'Os',
	  orbital: '4d5/2',
	  be: 273
	}, {
	  element: 'Ir',
	  orbital: '4d5/2',
	  be: 295
	}, {
	  element: 'Pt',
	  orbital: '4d5/2',
	  be: 314
	}, {
	  element: 'Au',
	  orbital: '4d5/2',
	  be: 334
	}, {
	  element: 'Hg',
	  orbital: '4d5/2',
	  be: 360
	}, {
	  element: 'Tl',
	  orbital: '4d5/2',
	  be: 386
	}, {
	  element: 'Pb',
	  orbital: '4d5/2',
	  be: 413
	}, {
	  element: 'Bi',
	  orbital: '4d5/2',
	  be: 440
	}, {
	  element: 'Po',
	  orbital: '4d5/2',
	  be: 473
	}, {
	  element: 'At',
	  orbital: '4d5/2',
	  be: 507
	}, {
	  element: 'Rn',
	  orbital: '4d5/2',
	  be: 541
	}, {
	  element: 'Fr',
	  orbital: '4d5/2',
	  be: 577
	}, {
	  element: 'Ra',
	  orbital: '4d5/2',
	  be: 603
	}, {
	  element: 'Ac',
	  orbital: '4d5/2',
	  be: 639
	}, {
	  element: 'Th',
	  orbital: '4d5/2',
	  be: 677
	}, {
	  element: 'U',
	  orbital: '4d5/2',
	  be: 739
	}, {
	  element: 'Y',
	  orbital: '4d',
	  be: 3
	}, {
	  element: 'Zr',
	  orbital: '4d',
	  be: 3
	}, {
	  element: 'Nb',
	  orbital: '4d',
	  be: 4
	}, {
	  element: 'Mo',
	  orbital: '4d',
	  be: 2
	}, {
	  element: 'Tc',
	  orbital: '4d',
	  be: 2
	}, {
	  element: 'Ru',
	  orbital: '4d',
	  be: 2
	}, {
	  element: 'Rh',
	  orbital: '4d',
	  be: 3
	}, {
	  element: 'Pd',
	  orbital: '4d',
	  be: 1
	}, {
	  element: 'Ag',
	  orbital: '4d',
	  be: 3
	}, {
	  element: 'Cd',
	  orbital: '4d',
	  be: 9
	}, {
	  element: 'In',
	  orbital: '4d',
	  be: 16
	}, {
	  element: 'Sn',
	  orbital: '4d',
	  be: 24
	}, {
	  element: 'Sb',
	  orbital: '4d',
	  be: 32
	}, {
	  element: 'Te',
	  orbital: '4d',
	  be: 40
	}, {
	  element: 'I',
	  orbital: '4d',
	  be: 50
	}, {
	  element: 'Xe',
	  orbital: '4d',
	  be: 63
	}, {
	  element: 'Cs',
	  orbital: '4d',
	  be: 77
	}, {
	  element: 'Ba',
	  orbital: '4d',
	  be: 90
	}, {
	  element: 'La',
	  orbital: '4d',
	  be: 99
	}, {
	  element: 'Ce',
	  orbital: '4d',
	  be: 111
	}, {
	  element: 'Pr',
	  orbital: '4d',
	  be: 114
	}, {
	  element: 'Nd',
	  orbital: '4d',
	  be: 118
	}, {
	  element: 'Pm',
	  orbital: '4d',
	  be: 121
	}, {
	  element: 'Sm',
	  orbital: '4d',
	  be: 130
	}, {
	  element: 'Eu',
	  orbital: '4d',
	  be: 134
	}, {
	  element: 'Gd',
	  orbital: '4d',
	  be: 141
	}, {
	  element: 'Tb',
	  orbital: '4d',
	  be: 148
	}, {
	  element: 'Dy',
	  orbital: '4d',
	  be: 154
	}, {
	  element: 'Ho',
	  orbital: '4d',
	  be: 161
	}, {
	  element: 'Er',
	  orbital: '4d',
	  be: 168
	}, {
	  element: 'Tm',
	  orbital: '4d',
	  be: 180
	}, {
	  element: 'Yb',
	  orbital: '4d',
	  be: 184
	}, {
	  element: 'Lu',
	  orbital: '4d',
	  be: 195
	}, {
	  element: 'Hf',
	  orbital: '4d',
	  be: 214
	}, {
	  element: 'Ta',
	  orbital: '4d',
	  be: 230
	}, {
	  element: 'W',
	  orbital: '4d',
	  be: 246
	}, {
	  element: 'Re',
	  orbital: '4d',
	  be: 260
	}, {
	  element: 'Os',
	  orbital: '4d',
	  be: 273
	}, {
	  element: 'Ir',
	  orbital: '4d',
	  be: 295
	}, {
	  element: 'Pt',
	  orbital: '4d',
	  be: 314
	}, {
	  element: 'Au',
	  orbital: '4d',
	  be: 334
	}, {
	  element: 'Hg',
	  orbital: '4d',
	  be: 360
	}, {
	  element: 'Tl',
	  orbital: '4d',
	  be: 386
	}, {
	  element: 'Pb',
	  orbital: '4d',
	  be: 413
	}, {
	  element: 'Bi',
	  orbital: '4d',
	  be: 440
	}, {
	  element: 'Po',
	  orbital: '4d',
	  be: 473
	}, {
	  element: 'At',
	  orbital: '4d',
	  be: 507
	}, {
	  element: 'Rn',
	  orbital: '4d',
	  be: 541
	}, {
	  element: 'Fr',
	  orbital: '4d',
	  be: 577
	}, {
	  element: 'Ra',
	  orbital: '4d',
	  be: 603
	}, {
	  element: 'Ac',
	  orbital: '4d',
	  be: 639
	}, {
	  element: 'Th',
	  orbital: '4d',
	  be: 677
	}, {
	  element: 'U',
	  orbital: '4d',
	  be: 739
	}, {
	  element: 'Ce',
	  orbital: '4f5/2',
	  be: 1
	}, {
	  element: 'Pr',
	  orbital: '4f5/2',
	  be: 2
	}, {
	  element: 'Nd',
	  orbital: '4f5/2',
	  be: 2
	}, {
	  element: 'Pm',
	  orbital: '4f5/2',
	  be: 4
	}, {
	  element: 'Sm',
	  orbital: '4f5/2',
	  be: 7
	}, {
	  element: 'Tb',
	  orbital: '4f5/2',
	  be: 3
	}, {
	  element: 'Dy',
	  orbital: '4f5/2',
	  be: 4
	}, {
	  element: 'Ho',
	  orbital: '4f5/2',
	  be: 4
	}, {
	  element: 'Er',
	  orbital: '4f5/2',
	  be: 4
	}, {
	  element: 'Tm',
	  orbital: '4f5/2',
	  be: 5
	}, {
	  element: 'Yb',
	  orbital: '4f5/2',
	  be: 6
	}, {
	  element: 'Lu',
	  orbital: '4f5/2',
	  be: 7
	}, {
	  element: 'Hf',
	  orbital: '4f5/2',
	  be: 19
	}, {
	  element: 'Ta',
	  orbital: '4f5/2',
	  be: 27
	}, {
	  element: 'W',
	  orbital: '4f5/2',
	  be: 37
	}, {
	  element: 'Re',
	  orbital: '4f5/2',
	  be: 47
	}, {
	  element: 'Os',
	  orbital: '4f5/2',
	  be: 52
	}, {
	  element: 'Ir',
	  orbital: '4f5/2',
	  be: 63
	}, {
	  element: 'Pt',
	  orbital: '4f5/2',
	  be: 74
	}, {
	  element: 'Au',
	  orbital: '4f5/2',
	  be: 87
	}, {
	  element: 'Hg',
	  orbital: '4f5/2',
	  be: 103
	}, {
	  element: 'Tl',
	  orbital: '4f5/2',
	  be: 122
	}, {
	  element: 'Pb',
	  orbital: '4f5/2',
	  be: 143
	}, {
	  element: 'Bi',
	  orbital: '4f5/2',
	  be: 163
	}, {
	  element: 'Po',
	  orbital: '4f5/2',
	  be: 184
	}, {
	  element: 'At',
	  orbital: '4f5/2',
	  be: 210
	}, {
	  element: 'Rn',
	  orbital: '4f5/2',
	  be: 238
	}, {
	  element: 'Fr',
	  orbital: '4f5/2',
	  be: 268
	}, {
	  element: 'Ra',
	  orbital: '4f5/2',
	  be: 299
	}, {
	  element: 'Ac',
	  orbital: '4f5/2',
	  be: 319
	}, {
	  element: 'Th',
	  orbital: '4f5/2',
	  be: 344
	}, {
	  element: 'U',
	  orbital: '4f5/2',
	  be: 391
	}, {
	  element: 'Ce',
	  orbital: '4f7/2',
	  be: 1
	}, {
	  element: 'Pr',
	  orbital: '4f7/2',
	  be: 2
	}, {
	  element: 'Nd',
	  orbital: '4f7/2',
	  be: 2
	}, {
	  element: 'Pm',
	  orbital: '4f7/2',
	  be: 4
	}, {
	  element: 'Sm',
	  orbital: '4f7/2',
	  be: 7
	}, {
	  element: 'Tb',
	  orbital: '4f7/2',
	  be: 3
	}, {
	  element: 'Dy',
	  orbital: '4f7/2',
	  be: 4
	}, {
	  element: 'Ho',
	  orbital: '4f7/2',
	  be: 4
	}, {
	  element: 'Er',
	  orbital: '4f7/2',
	  be: 4
	}, {
	  element: 'Tm',
	  orbital: '4f7/2',
	  be: 5
	}, {
	  element: 'Yb',
	  orbital: '4f7/2',
	  be: 6
	}, {
	  element: 'Lu',
	  orbital: '4f7/2',
	  be: 7
	}, {
	  element: 'Hf',
	  orbital: '4f7/2',
	  be: 18
	}, {
	  element: 'Ta',
	  orbital: '4f7/2',
	  be: 25
	}, {
	  element: 'W',
	  orbital: '4f7/2',
	  be: 34
	}, {
	  element: 'Re',
	  orbital: '4f7/2',
	  be: 45
	}, {
	  element: 'Os',
	  orbital: '4f7/2',
	  be: 50
	}, {
	  element: 'Ir',
	  orbital: '4f7/2',
	  be: 60
	}, {
	  element: 'Pt',
	  orbital: '4f7/2',
	  be: 70
	}, {
	  element: 'Au',
	  orbital: '4f7/2',
	  be: 83
	}, {
	  element: 'Hg',
	  orbital: '4f7/2',
	  be: 99
	}, {
	  element: 'Tl',
	  orbital: '4f7/2',
	  be: 118
	}, {
	  element: 'Pb',
	  orbital: '4f7/2',
	  be: 138
	}, {
	  element: 'Bi',
	  orbital: '4f7/2',
	  be: 158
	}, {
	  element: 'Po',
	  orbital: '4f7/2',
	  be: 184
	}, {
	  element: 'At',
	  orbital: '4f7/2',
	  be: 210
	}, {
	  element: 'Rn',
	  orbital: '4f7/2',
	  be: 238
	}, {
	  element: 'Fr',
	  orbital: '4f7/2',
	  be: 268
	}, {
	  element: 'Ra',
	  orbital: '4f7/2',
	  be: 299
	}, {
	  element: 'Ac',
	  orbital: '4f7/2',
	  be: 319
	}, {
	  element: 'Th',
	  orbital: '4f7/2',
	  be: 335
	}, {
	  element: 'U',
	  orbital: '4f7/2',
	  be: 380
	}, {
	  element: 'Ce',
	  orbital: '4f',
	  be: 1
	}, {
	  element: 'Pr',
	  orbital: '4f',
	  be: 2
	}, {
	  element: 'Nd',
	  orbital: '4f',
	  be: 2
	}, {
	  element: 'Pm',
	  orbital: '4f',
	  be: 4
	}, {
	  element: 'Sm',
	  orbital: '4f',
	  be: 7
	}, {
	  element: 'Tb',
	  orbital: '4f',
	  be: 3
	}, {
	  element: 'Dy',
	  orbital: '4f',
	  be: 4
	}, {
	  element: 'Ho',
	  orbital: '4f',
	  be: 4
	}, {
	  element: 'Er',
	  orbital: '4f',
	  be: 4
	}, {
	  element: 'Tm',
	  orbital: '4f',
	  be: 5
	}, {
	  element: 'Yb',
	  orbital: '4f',
	  be: 6
	}, {
	  element: 'Lu',
	  orbital: '4f',
	  be: 7
	}, {
	  element: 'Hf',
	  orbital: '4f',
	  be: 18
	}, {
	  element: 'Ta',
	  orbital: '4f',
	  be: 25
	}, {
	  element: 'W',
	  orbital: '4f',
	  be: 34
	}, {
	  element: 'Re',
	  orbital: '4f',
	  be: 45
	}, {
	  element: 'Os',
	  orbital: '4f',
	  be: 50
	}, {
	  element: 'Ir',
	  orbital: '4f',
	  be: 60
	}, {
	  element: 'Pt',
	  orbital: '4f',
	  be: 70
	}, {
	  element: 'Au',
	  orbital: '4f',
	  be: 83
	}, {
	  element: 'Hg',
	  orbital: '4f',
	  be: 99
	}, {
	  element: 'Tl',
	  orbital: '4f',
	  be: 118
	}, {
	  element: 'Pb',
	  orbital: '4f',
	  be: 138
	}, {
	  element: 'Bi',
	  orbital: '4f',
	  be: 158
	}, {
	  element: 'Po',
	  orbital: '4f',
	  be: 184
	}, {
	  element: 'At',
	  orbital: '4f',
	  be: 210
	}, {
	  element: 'Rn',
	  orbital: '4f',
	  be: 238
	}, {
	  element: 'Fr',
	  orbital: '4f',
	  be: 268
	}, {
	  element: 'Ra',
	  orbital: '4f',
	  be: 299
	}, {
	  element: 'Ac',
	  orbital: '4f',
	  be: 319
	}, {
	  element: 'Th',
	  orbital: '4f',
	  be: 335
	}, {
	  element: 'U',
	  orbital: '4f',
	  be: 380
	}, {
	  element: 'Sn',
	  orbital: '5s',
	  be: 1
	}, {
	  element: 'Sb',
	  orbital: '5s',
	  be: 7
	}, {
	  element: 'Te',
	  orbital: '5s',
	  be: 12
	}, {
	  element: 'I',
	  orbital: '5s',
	  be: 14
	}, {
	  element: 'Xe',
	  orbital: '5s',
	  be: 18
	}, {
	  element: 'Cs',
	  orbital: '5s',
	  be: 23
	}, {
	  element: 'Ba',
	  orbital: '5s',
	  be: 40
	}, {
	  element: 'La',
	  orbital: '5s',
	  be: 33
	}, {
	  element: 'Ce',
	  orbital: '5s',
	  be: 38
	}, {
	  element: 'Pr',
	  orbital: '5s',
	  be: 38
	}, {
	  element: 'Nd',
	  orbital: '5s',
	  be: 38
	}, {
	  element: 'Pm',
	  orbital: '5s',
	  be: 38
	}, {
	  element: 'Sm',
	  orbital: '5s',
	  be: 39
	}, {
	  element: 'Eu',
	  orbital: '5s',
	  be: 32
	}, {
	  element: 'Gd',
	  orbital: '5s',
	  be: 36
	}, {
	  element: 'Tb',
	  orbital: '5s',
	  be: 40
	}, {
	  element: 'Dy',
	  orbital: '5s',
	  be: 63
	}, {
	  element: 'Ho',
	  orbital: '5s',
	  be: 51
	}, {
	  element: 'Er',
	  orbital: '5s',
	  be: 60
	}, {
	  element: 'Tm',
	  orbital: '5s',
	  be: 53
	}, {
	  element: 'Yb',
	  orbital: '5s',
	  be: 53
	}, {
	  element: 'Lu',
	  orbital: '5s',
	  be: 57
	}, {
	  element: 'Hf',
	  orbital: '5s',
	  be: 65
	}, {
	  element: 'Ta',
	  orbital: '5s',
	  be: 71
	}, {
	  element: 'W',
	  orbital: '5s',
	  be: 77
	}, {
	  element: 'Re',
	  orbital: '5s',
	  be: 83
	}, {
	  element: 'Os',
	  orbital: '5s',
	  be: 84
	}, {
	  element: 'Ir',
	  orbital: '5s',
	  be: 96
	}, {
	  element: 'Pt',
	  orbital: '5s',
	  be: 102
	}, {
	  element: 'Au',
	  orbital: '5s',
	  be: 108
	}, {
	  element: 'Hg',
	  orbital: '5s',
	  be: 120
	}, {
	  element: 'Tl',
	  orbital: '5s',
	  be: 137
	}, {
	  element: 'Pb',
	  orbital: '5s',
	  be: 148
	}, {
	  element: 'Bi',
	  orbital: '5s',
	  be: 160
	}, {
	  element: 'Po',
	  orbital: '5s',
	  be: 177
	}, {
	  element: 'At',
	  orbital: '5s',
	  be: 195
	}, {
	  element: 'Rn',
	  orbital: '5s',
	  be: 214
	}, {
	  element: 'Fr',
	  orbital: '5s',
	  be: 234
	}, {
	  element: 'Ra',
	  orbital: '5s',
	  be: 254
	}, {
	  element: 'Ac',
	  orbital: '5s',
	  be: 272
	}, {
	  element: 'Th',
	  orbital: '5s',
	  be: 290
	}, {
	  element: 'In',
	  orbital: '5p1/2',
	  be: 1
	}, {
	  element: 'Sn',
	  orbital: '5p1/2',
	  be: 1
	}, {
	  element: 'Sb',
	  orbital: '5p1/2',
	  be: 2
	}, {
	  element: 'Te',
	  orbital: '5p1/2',
	  be: 2
	}, {
	  element: 'I',
	  orbital: '5p1/2',
	  be: 3
	}, {
	  element: 'Xe',
	  orbital: '5p1/2',
	  be: 7
	}, {
	  element: 'Cs',
	  orbital: '5p1/2',
	  be: 13
	}, {
	  element: 'Ba',
	  orbital: '5p1/2',
	  be: 17
	}, {
	  element: 'La',
	  orbital: '5p1/2',
	  be: 15
	}, {
	  element: 'Ce',
	  orbital: '5p1/2',
	  be: 20
	}, {
	  element: 'Pr',
	  orbital: '5p1/2',
	  be: 23
	}, {
	  element: 'Nd',
	  orbital: '5p1/2',
	  be: 22
	}, {
	  element: 'Pm',
	  orbital: '5p1/2',
	  be: 22
	}, {
	  element: 'Sm',
	  orbital: '5p1/2',
	  be: 22
	}, {
	  element: 'Eu',
	  orbital: '5p1/2',
	  be: 22
	}, {
	  element: 'Gd',
	  orbital: '5p1/2',
	  be: 21
	}, {
	  element: 'Tb',
	  orbital: '5p1/2',
	  be: 26
	}, {
	  element: 'Dy',
	  orbital: '5p1/2',
	  be: 26
	}, {
	  element: 'Ho',
	  orbital: '5p1/2',
	  be: 20
	}, {
	  element: 'Er',
	  orbital: '5p1/2',
	  be: 29
	}, {
	  element: 'Tm',
	  orbital: '5p1/2',
	  be: 32
	}, {
	  element: 'Yb',
	  orbital: '5p1/2',
	  be: 23
	}, {
	  element: 'Lu',
	  orbital: '5p1/2',
	  be: 28
	}, {
	  element: 'Hf',
	  orbital: '5p1/2',
	  be: 38
	}, {
	  element: 'Ta',
	  orbital: '5p1/2',
	  be: 45
	}, {
	  element: 'W',
	  orbital: '5p1/2',
	  be: 47
	}, {
	  element: 'Re',
	  orbital: '5p1/2',
	  be: 46
	}, {
	  element: 'Os',
	  orbital: '5p1/2',
	  be: 58
	}, {
	  element: 'Ir',
	  orbital: '5p1/2',
	  be: 63
	}, {
	  element: 'Pt',
	  orbital: '5p1/2',
	  be: 66
	}, {
	  element: 'Au',
	  orbital: '5p1/2',
	  be: 72
	}, {
	  element: 'Hg',
	  orbital: '5p1/2',
	  be: 81
	}, {
	  element: 'Tl',
	  orbital: '5p1/2',
	  be: 100
	}, {
	  element: 'Pb',
	  orbital: '5p1/2',
	  be: 105
	}, {
	  element: 'Bi',
	  orbital: '5p1/2',
	  be: 117
	}, {
	  element: 'Po',
	  orbital: '5p1/2',
	  be: 132
	}, {
	  element: 'At',
	  orbital: '5p1/2',
	  be: 148
	}, {
	  element: 'Rn',
	  orbital: '5p1/2',
	  be: 164
	}, {
	  element: 'Fr',
	  orbital: '5p1/2',
	  be: 182
	}, {
	  element: 'Ra',
	  orbital: '5p1/2',
	  be: 200
	}, {
	  element: 'Ac',
	  orbital: '5p1/2',
	  be: 215
	}, {
	  element: 'Th',
	  orbital: '5p1/2',
	  be: 229
	}, {
	  element: 'In',
	  orbital: '5p3/2',
	  be: 1
	}, {
	  element: 'Sn',
	  orbital: '5p3/2',
	  be: 1
	}, {
	  element: 'Sb',
	  orbital: '5p3/2',
	  be: 2
	}, {
	  element: 'Te',
	  orbital: '5p3/2',
	  be: 2
	}, {
	  element: 'I',
	  orbital: '5p3/2',
	  be: 3
	}, {
	  element: 'Xe',
	  orbital: '5p3/2',
	  be: 7
	}, {
	  element: 'Cs',
	  orbital: '5p3/2',
	  be: 12
	}, {
	  element: 'Ba',
	  orbital: '5p3/2',
	  be: 15
	}, {
	  element: 'La',
	  orbital: '5p3/2',
	  be: 15
	}, {
	  element: 'Ce',
	  orbital: '5p3/2',
	  be: 20
	}, {
	  element: 'Pr',
	  orbital: '5p3/2',
	  be: 23
	}, {
	  element: 'Nd',
	  orbital: '5p3/2',
	  be: 22
	}, {
	  element: 'Pm',
	  orbital: '5p3/2',
	  be: 22
	}, {
	  element: 'Sm',
	  orbital: '5p3/2',
	  be: 22
	}, {
	  element: 'Eu',
	  orbital: '5p3/2',
	  be: 22
	}, {
	  element: 'Gd',
	  orbital: '5p3/2',
	  be: 21
	}, {
	  element: 'Tb',
	  orbital: '5p3/2',
	  be: 26
	}, {
	  element: 'Dy',
	  orbital: '5p3/2',
	  be: 26
	}, {
	  element: 'Ho',
	  orbital: '5p3/2',
	  be: 20
	}, {
	  element: 'Er',
	  orbital: '5p3/2',
	  be: 29
	}, {
	  element: 'Tm',
	  orbital: '5p3/2',
	  be: 32
	}, {
	  element: 'Yb',
	  orbital: '5p3/2',
	  be: 23
	}, {
	  element: 'Lu',
	  orbital: '5p3/2',
	  be: 28
	}, {
	  element: 'Hf',
	  orbital: '5p3/2',
	  be: 31
	}, {
	  element: 'Ta',
	  orbital: '5p3/2',
	  be: 37
	}, {
	  element: 'W',
	  orbital: '5p3/2',
	  be: 37
	}, {
	  element: 'Re',
	  orbital: '5p3/2',
	  be: 35
	}, {
	  element: 'Os',
	  orbital: '5p3/2',
	  be: 46
	}, {
	  element: 'Ir',
	  orbital: '5p3/2',
	  be: 51
	}, {
	  element: 'Pt',
	  orbital: '5p3/2',
	  be: 51
	}, {
	  element: 'Au',
	  orbital: '5p3/2',
	  be: 54
	}, {
	  element: 'Hg',
	  orbital: '5p3/2',
	  be: 58
	}, {
	  element: 'Tl',
	  orbital: '5p3/2',
	  be: 76
	}, {
	  element: 'Pb',
	  orbital: '5p3/2',
	  be: 86
	}, {
	  element: 'Bi',
	  orbital: '5p3/2',
	  be: 93
	}, {
	  element: 'Po',
	  orbital: '5p3/2',
	  be: 104
	}, {
	  element: 'At',
	  orbital: '5p3/2',
	  be: 115
	}, {
	  element: 'Rn',
	  orbital: '5p3/2',
	  be: 127
	}, {
	  element: 'Fr',
	  orbital: '5p3/2',
	  be: 140
	}, {
	  element: 'Ra',
	  orbital: '5p3/2',
	  be: 153
	}, {
	  element: 'Ac',
	  orbital: '5p3/2',
	  be: 167
	}, {
	  element: 'Th',
	  orbital: '5p3/2',
	  be: 182
	}, {
	  element: 'In',
	  orbital: '5p',
	  be: 1
	}, {
	  element: 'Sn',
	  orbital: '5p',
	  be: 1
	}, {
	  element: 'Sb',
	  orbital: '5p',
	  be: 2
	}, {
	  element: 'Te',
	  orbital: '5p',
	  be: 2
	}, {
	  element: 'I',
	  orbital: '5p',
	  be: 3
	}, {
	  element: 'Xe',
	  orbital: '5p',
	  be: 7
	}, {
	  element: 'Cs',
	  orbital: '5p',
	  be: 12
	}, {
	  element: 'Ba',
	  orbital: '5p',
	  be: 15
	}, {
	  element: 'La',
	  orbital: '5p',
	  be: 15
	}, {
	  element: 'Ce',
	  orbital: '5p',
	  be: 20
	}, {
	  element: 'Pr',
	  orbital: '5p',
	  be: 23
	}, {
	  element: 'Nd',
	  orbital: '5p',
	  be: 22
	}, {
	  element: 'Pm',
	  orbital: '5p',
	  be: 22
	}, {
	  element: 'Sm',
	  orbital: '5p',
	  be: 22
	}, {
	  element: 'Eu',
	  orbital: '5p',
	  be: 22
	}, {
	  element: 'Gd',
	  orbital: '5p',
	  be: 21
	}, {
	  element: 'Tb',
	  orbital: '5p',
	  be: 26
	}, {
	  element: 'Dy',
	  orbital: '5p',
	  be: 26
	}, {
	  element: 'Ho',
	  orbital: '5p',
	  be: 20
	}, {
	  element: 'Er',
	  orbital: '5p',
	  be: 29
	}, {
	  element: 'Tm',
	  orbital: '5p',
	  be: 32
	}, {
	  element: 'Yb',
	  orbital: '5p',
	  be: 23
	}, {
	  element: 'Lu',
	  orbital: '5p',
	  be: 28
	}, {
	  element: 'Hf',
	  orbital: '5p',
	  be: 31
	}, {
	  element: 'Ta',
	  orbital: '5p',
	  be: 37
	}, {
	  element: 'W',
	  orbital: '5p',
	  be: 37
	}, {
	  element: 'Re',
	  orbital: '5p',
	  be: 35
	}, {
	  element: 'Os',
	  orbital: '5p',
	  be: 46
	}, {
	  element: 'Ir',
	  orbital: '5p',
	  be: 51
	}, {
	  element: 'Pt',
	  orbital: '5p',
	  be: 51
	}, {
	  element: 'Au',
	  orbital: '5p',
	  be: 54
	}, {
	  element: 'Hg',
	  orbital: '5p',
	  be: 58
	}, {
	  element: 'Tl',
	  orbital: '5p',
	  be: 76
	}, {
	  element: 'Pb',
	  orbital: '5p',
	  be: 86
	}, {
	  element: 'Bi',
	  orbital: '5p',
	  be: 93
	}, {
	  element: 'Po',
	  orbital: '5p',
	  be: 104
	}, {
	  element: 'At',
	  orbital: '5p',
	  be: 115
	}, {
	  element: 'Rn',
	  orbital: '5p',
	  be: 127
	}, {
	  element: 'Fr',
	  orbital: '5p',
	  be: 140
	}, {
	  element: 'Ra',
	  orbital: '5p',
	  be: 153
	}, {
	  element: 'Ac',
	  orbital: '5p',
	  be: 167
	}, {
	  element: 'Th',
	  orbital: '5p',
	  be: 182
	}, {
	  element: 'Lu',
	  orbital: '5d3/2',
	  be: 5
	}, {
	  element: 'Hf',
	  orbital: '5d3/2',
	  be: 7
	}, {
	  element: 'Ta',
	  orbital: '5d3/2',
	  be: 6
	}, {
	  element: 'W',
	  orbital: '5d3/2',
	  be: 6
	}, {
	  element: 'Re',
	  orbital: '5d3/2',
	  be: 4
	}, {
	  element: 'Ir',
	  orbital: '5d3/2',
	  be: 4
	}, {
	  element: 'Pt',
	  orbital: '5d3/2',
	  be: 2
	}, {
	  element: 'Au',
	  orbital: '5d3/2',
	  be: 3
	}, {
	  element: 'Hg',
	  orbital: '5d3/2',
	  be: 7
	}, {
	  element: 'Tl',
	  orbital: '5d3/2',
	  be: 16
	}, {
	  element: 'Pb',
	  orbital: '5d3/2',
	  be: 22
	}, {
	  element: 'Bi',
	  orbital: '5d3/2',
	  be: 27
	}, {
	  element: 'Po',
	  orbital: '5d3/2',
	  be: 31
	}, {
	  element: 'At',
	  orbital: '5d3/2',
	  be: 40
	}, {
	  element: 'Rn',
	  orbital: '5d3/2',
	  be: 48
	}, {
	  element: 'Fr',
	  orbital: '5d3/2',
	  be: 58
	}, {
	  element: 'Ra',
	  orbital: '5d3/2',
	  be: 68
	}, {
	  element: 'Ac',
	  orbital: '5d3/2',
	  be: 80
	}, {
	  element: 'Th',
	  orbital: '5d3/2',
	  be: 95
	}, {
	  element: 'U',
	  orbital: '5d3/2',
	  be: 104
	}, {
	  element: 'Lu',
	  orbital: '5d5/2',
	  be: 5
	}, {
	  element: 'Hf',
	  orbital: '5d5/2',
	  be: 7
	}, {
	  element: 'Ta',
	  orbital: '5d5/2',
	  be: 6
	}, {
	  element: 'W',
	  orbital: '5d5/2',
	  be: 6
	}, {
	  element: 'Re',
	  orbital: '5d5/2',
	  be: 4
	}, {
	  element: 'Ir',
	  orbital: '5d5/2',
	  be: 4
	}, {
	  element: 'Pt',
	  orbital: '5d5/2',
	  be: 2
	}, {
	  element: 'Au',
	  orbital: '5d5/2',
	  be: 3
	}, {
	  element: 'Hg',
	  orbital: '5d5/2',
	  be: 7
	}, {
	  element: 'Tl',
	  orbital: '5d5/2',
	  be: 13
	}, {
	  element: 'Pb',
	  orbital: '5d5/2',
	  be: 20
	}, {
	  element: 'Bi',
	  orbital: '5d5/2',
	  be: 25
	}, {
	  element: 'Po',
	  orbital: '5d5/2',
	  be: 31
	}, {
	  element: 'At',
	  orbital: '5d5/2',
	  be: 40
	}, {
	  element: 'Rn',
	  orbital: '5d5/2',
	  be: 48
	}, {
	  element: 'Fr',
	  orbital: '5d5/2',
	  be: 58
	}, {
	  element: 'Ra',
	  orbital: '5d5/2',
	  be: 68
	}, {
	  element: 'Ac',
	  orbital: '5d5/2',
	  be: 80
	}, {
	  element: 'Th',
	  orbital: '5d5/2',
	  be: 88
	}, {
	  element: 'U',
	  orbital: '5d5/2',
	  be: 96
	}, {
	  element: 'Lu',
	  orbital: '5d',
	  be: 5
	}, {
	  element: 'Hf',
	  orbital: '5d',
	  be: 7
	}, {
	  element: 'Ta',
	  orbital: '5d',
	  be: 6
	}, {
	  element: 'W',
	  orbital: '5d',
	  be: 6
	}, {
	  element: 'Re',
	  orbital: '5d',
	  be: 4
	}, {
	  element: 'Ir',
	  orbital: '5d',
	  be: 4
	}, {
	  element: 'Pt',
	  orbital: '5d',
	  be: 2
	}, {
	  element: 'Au',
	  orbital: '5d',
	  be: 3
	}, {
	  element: 'Hg',
	  orbital: '5d',
	  be: 7
	}, {
	  element: 'Tl',
	  orbital: '5d',
	  be: 13
	}, {
	  element: 'Pb',
	  orbital: '5d',
	  be: 20
	}, {
	  element: 'Bi',
	  orbital: '5d',
	  be: 25
	}, {
	  element: 'Po',
	  orbital: '5d',
	  be: 31
	}, {
	  element: 'At',
	  orbital: '5d',
	  be: 40
	}, {
	  element: 'Rn',
	  orbital: '5d',
	  be: 48
	}, {
	  element: 'Fr',
	  orbital: '5d',
	  be: 58
	}, {
	  element: 'Ra',
	  orbital: '5d',
	  be: 68
	}, {
	  element: 'Ac',
	  orbital: '5d',
	  be: 80
	}, {
	  element: 'Th',
	  orbital: '5d',
	  be: 88
	}, {
	  element: 'U',
	  orbital: '5d',
	  be: 96
	}, {
	  element: 'Pb',
	  orbital: '6s',
	  be: 3
	}, {
	  element: 'Bi',
	  orbital: '6s',
	  be: 8
	}, {
	  element: 'Po',
	  orbital: '6s',
	  be: 12
	}, {
	  element: 'At',
	  orbital: '6s',
	  be: 18
	}, {
	  element: 'Rn',
	  orbital: '6s',
	  be: 26
	}, {
	  element: 'Fr',
	  orbital: '6s',
	  be: 34
	}, {
	  element: 'Ra',
	  orbital: '6s',
	  be: 44
	}, {
	  element: 'Th',
	  orbital: '6s',
	  be: 60
	}, {
	  element: 'Pb',
	  orbital: '6p1/2',
	  be: 1
	}, {
	  element: 'Bi',
	  orbital: '6p1/2',
	  be: 3
	}, {
	  element: 'Po',
	  orbital: '6p1/2',
	  be: 5
	}, {
	  element: 'At',
	  orbital: '6p1/2',
	  be: 8
	}, {
	  element: 'Rn',
	  orbital: '6p1/2',
	  be: 11
	}, {
	  element: 'Fr',
	  orbital: '6p1/2',
	  be: 15
	}, {
	  element: 'Ra',
	  orbital: '6p1/2',
	  be: 19
	}, {
	  element: 'Th',
	  orbital: '6p1/2',
	  be: 49
	}, {
	  element: 'U',
	  orbital: '6p1/2',
	  be: 28.9
	}, {
	  element: 'Pb',
	  orbital: '6p3/2',
	  be: 1
	}, {
	  element: 'Bi',
	  orbital: '6p3/2',
	  be: 3
	}, {
	  element: 'Po',
	  orbital: '6p3/2',
	  be: 5
	}, {
	  element: 'At',
	  orbital: '6p3/2',
	  be: 8
	}, {
	  element: 'Rn',
	  orbital: '6p3/2',
	  be: 11
	}, {
	  element: 'Fr',
	  orbital: '6p3/2',
	  be: 15
	}, {
	  element: 'Ra',
	  orbital: '6p3/2',
	  be: 19
	}, {
	  element: 'Th',
	  orbital: '6p3/2',
	  be: 43
	}, {
	  element: 'U',
	  orbital: '6p3/2',
	  be: 19
	}, {
	  element: 'Pb',
	  orbital: '6p',
	  be: 1
	}, {
	  element: 'Bi',
	  orbital: '6p',
	  be: 3
	}, {
	  element: 'Po',
	  orbital: '6p',
	  be: 5
	}, {
	  element: 'At',
	  orbital: '6p',
	  be: 8
	}, {
	  element: 'Rn',
	  orbital: '6p',
	  be: 11
	}, {
	  element: 'Fr',
	  orbital: '6p',
	  be: 15
	}, {
	  element: 'Ra',
	  orbital: '6p',
	  be: 19
	}, {
	  element: 'Th',
	  orbital: '6p',
	  be: 43
	}, {
	  element: 'U',
	  orbital: '6p',
	  be: 19
	}, {
	  element: 'H',
	  orbital: '1s',
	  be: 14
	}, {
	  element: 'He',
	  orbital: '1s',
	  be: 25
	}, {
	  element: 'Li',
	  orbital: '1s',
	  be: 55
	}, {
	  element: 'Be',
	  orbital: '1s',
	  be: 111
	}, {
	  element: 'B',
	  orbital: '1s',
	  be: 188
	}, {
	  element: 'C',
	  orbital: '1s',
	  be: 284
	}, {
	  element: 'N',
	  orbital: '1s',
	  be: 399
	}, {
	  element: 'O',
	  orbital: '1s',
	  be: 532
	}, {
	  element: 'F',
	  orbital: '1s',
	  be: 686
	}, {
	  element: 'Ne',
	  orbital: '1s',
	  be: 867
	}, {
	  element: 'Na',
	  orbital: '1s',
	  be: 1072
	}, {
	  element: 'O',
	  orbital: '2s',
	  be: 24
	}, {
	  element: 'F',
	  orbital: '2s',
	  be: 31
	}, {
	  element: 'Ne',
	  orbital: '2s',
	  be: 45
	}, {
	  element: 'Na',
	  orbital: '2s',
	  be: 63
	}, {
	  element: 'Mg',
	  orbital: '2s',
	  be: 89
	}, {
	  element: 'Al',
	  orbital: '2s',
	  be: 118
	}, {
	  element: 'Si',
	  orbital: '2s',
	  be: 149
	}, {
	  element: 'P',
	  orbital: '2s',
	  be: 189
	}, {
	  element: 'S',
	  orbital: '2s',
	  be: 229
	}, {
	  element: 'Cl',
	  orbital: '2s',
	  be: 270
	}, {
	  element: 'Ar',
	  orbital: '2s',
	  be: 320
	}, {
	  element: 'K',
	  orbital: '2s',
	  be: 377
	}, {
	  element: 'Ca',
	  orbital: '2s',
	  be: 438
	}, {
	  element: 'Sc',
	  orbital: '2s',
	  be: 500
	}, {
	  element: 'Ti',
	  orbital: '2s',
	  be: 564
	}, {
	  element: 'V',
	  orbital: '2s',
	  be: 628
	}, {
	  element: 'Cr',
	  orbital: '2s',
	  be: 695
	}, {
	  element: 'Mn',
	  orbital: '2s',
	  be: 769
	}, {
	  element: 'Fe',
	  orbital: '2s',
	  be: 846
	}, {
	  element: 'Co',
	  orbital: '2s',
	  be: 926
	}, {
	  element: 'Ni',
	  orbital: '2s',
	  be: 1008
	}, {
	  element: 'Cu',
	  orbital: '2s',
	  be: 1096
	}, {
	  element: 'Zn',
	  orbital: '2s',
	  be: 1194
	}, {
	  element: 'B',
	  orbital: '2p1/2',
	  be: 5
	}, {
	  element: 'C',
	  orbital: '2p1/2',
	  be: 7
	}, {
	  element: 'N',
	  orbital: '2p1/2',
	  be: 9
	}, {
	  element: 'O',
	  orbital: '2p1/2',
	  be: 7
	}, {
	  element: 'F',
	  orbital: '2p1/2',
	  be: 9
	}, {
	  element: 'Ne',
	  orbital: '2p1/2',
	  be: 18
	}, {
	  element: 'Na',
	  orbital: '2p1/2',
	  be: 31
	}, {
	  element: 'Mg',
	  orbital: '2p1/2',
	  be: 52
	}, {
	  element: 'Al',
	  orbital: '2p1/2',
	  be: 74
	}, {
	  element: 'Si',
	  orbital: '2p1/2',
	  be: 100
	}, {
	  element: 'P',
	  orbital: '2p1/2',
	  be: 136
	}, {
	  element: 'S',
	  orbital: '2p1/2',
	  be: 165
	}, {
	  element: 'Cl',
	  orbital: '2p1/2',
	  be: 202
	}, {
	  element: 'Ar',
	  orbital: '2p1/2',
	  be: 247
	}, {
	  element: 'K',
	  orbital: '2p1/2',
	  be: 297
	}, {
	  element: 'Ca',
	  orbital: '2p1/2',
	  be: 350
	}, {
	  element: 'Sc',
	  orbital: '2p1/2',
	  be: 407
	}, {
	  element: 'Ti',
	  orbital: '2p1/2',
	  be: 461
	}, {
	  element: 'V',
	  orbital: '2p1/2',
	  be: 520
	}, {
	  element: 'Cr',
	  orbital: '2p1/2',
	  be: 584
	}, {
	  element: 'Mn',
	  orbital: '2p1/2',
	  be: 652
	}, {
	  element: 'Fe',
	  orbital: '2p1/2',
	  be: 723
	}, {
	  element: 'Co',
	  orbital: '2p1/2',
	  be: 794
	}, {
	  element: 'Ni',
	  orbital: '2p1/2',
	  be: 872
	}, {
	  element: 'Cu',
	  orbital: '2p1/2',
	  be: 951
	}, {
	  element: 'Zn',
	  orbital: '2p1/2',
	  be: 1044
	}, {
	  element: 'Ga',
	  orbital: '2p1/2',
	  be: 1143
	}, {
	  element: 'B',
	  orbital: '2p3/2',
	  be: 5
	}, {
	  element: 'C',
	  orbital: '2p3/2',
	  be: 7
	}, {
	  element: 'N',
	  orbital: '2p3/2',
	  be: 9
	}, {
	  element: 'O',
	  orbital: '2p3/2',
	  be: 7
	}, {
	  element: 'F',
	  orbital: '2p3/2',
	  be: 9
	}, {
	  element: 'Ne',
	  orbital: '2p3/2',
	  be: 18
	}, {
	  element: 'Na',
	  orbital: '2p3/2',
	  be: 31
	}, {
	  element: 'Mg',
	  orbital: '2p3/2',
	  be: 52
	}, {
	  element: 'Al',
	  orbital: '2p3/2',
	  be: 73
	}, {
	  element: 'Si',
	  orbital: '2p3/2',
	  be: 99
	}, {
	  element: 'P',
	  orbital: '2p3/2',
	  be: 135
	}, {
	  element: 'S',
	  orbital: '2p3/2',
	  be: 164
	}, {
	  element: 'Cl',
	  orbital: '2p3/2',
	  be: 200
	}, {
	  element: 'Ar',
	  orbital: '2p3/2',
	  be: 245
	}, {
	  element: 'K',
	  orbital: '2p3/2',
	  be: 294
	}, {
	  element: 'Ca',
	  orbital: '2p3/2',
	  be: 347
	}, {
	  element: 'Sc',
	  orbital: '2p3/2',
	  be: 402
	}, {
	  element: 'Ti',
	  orbital: '2p3/2',
	  be: 455
	}, {
	  element: 'V',
	  orbital: '2p3/2',
	  be: 513
	}, {
	  element: 'Cr',
	  orbital: '2p3/2',
	  be: 575
	}, {
	  element: 'Mn',
	  orbital: '2p3/2',
	  be: 641
	}, {
	  element: 'Fe',
	  orbital: '2p3/2',
	  be: 710
	}, {
	  element: 'Co',
	  orbital: '2p3/2',
	  be: 779
	}, {
	  element: 'Ni',
	  orbital: '2p3/2',
	  be: 855
	}, {
	  element: 'Cu',
	  orbital: '2p3/2',
	  be: 931
	}, {
	  element: 'Zn',
	  orbital: '2p3/2',
	  be: 1021
	}, {
	  element: 'Ga',
	  orbital: '2p3/2',
	  be: 1116
	}, {
	  element: 'Ge',
	  orbital: '2p3/2',
	  be: 1217
	}, {
	  element: 'B',
	  orbital: '2p',
	  be: 5
	}, {
	  element: 'C',
	  orbital: '2p',
	  be: 7
	}, {
	  element: 'N',
	  orbital: '2p',
	  be: 9
	}, {
	  element: 'O',
	  orbital: '2p',
	  be: 7
	}, {
	  element: 'F',
	  orbital: '2p',
	  be: 9
	}, {
	  element: 'Ne',
	  orbital: '2p',
	  be: 18
	}, {
	  element: 'Na',
	  orbital: '2p',
	  be: 31
	}, {
	  element: 'Mg',
	  orbital: '2p',
	  be: 52
	}, {
	  element: 'Al',
	  orbital: '2p',
	  be: 73
	}, {
	  element: 'Si',
	  orbital: '2p',
	  be: 99
	}, {
	  element: 'P',
	  orbital: '2p',
	  be: 135
	}, {
	  element: 'S',
	  orbital: '2p',
	  be: 164
	}, {
	  element: 'Cl',
	  orbital: '2p',
	  be: 200
	}, {
	  element: 'Ar',
	  orbital: '2p',
	  be: 245
	}, {
	  element: 'K',
	  orbital: '2p',
	  be: 294
	}, {
	  element: 'Ca',
	  orbital: '2p',
	  be: 347
	}, {
	  element: 'Sc',
	  orbital: '2p',
	  be: 402
	}, {
	  element: 'Ti',
	  orbital: '2p',
	  be: 455
	}, {
	  element: 'V',
	  orbital: '2p',
	  be: 513
	}, {
	  element: 'Cr',
	  orbital: '2p',
	  be: 575
	}, {
	  element: 'Mn',
	  orbital: '2p',
	  be: 641
	}, {
	  element: 'Fe',
	  orbital: '2p',
	  be: 710
	}, {
	  element: 'Co',
	  orbital: '2p',
	  be: 779
	}, {
	  element: 'Ni',
	  orbital: '2p',
	  be: 855
	}, {
	  element: 'Cu',
	  orbital: '2p',
	  be: 931
	}, {
	  element: 'Zn',
	  orbital: '2p',
	  be: 1021
	}, {
	  element: 'Ga',
	  orbital: '2p',
	  be: 1116
	}, {
	  element: 'Ge',
	  orbital: '2p',
	  be: 1217
	}, {
	  element: 'Na',
	  orbital: '3s',
	  be: 1
	}, {
	  element: 'Mg',
	  orbital: '3s',
	  be: 2
	}, {
	  element: 'Al',
	  orbital: '3s',
	  be: 1
	}, {
	  element: 'Si',
	  orbital: '3s',
	  be: 8
	}, {
	  element: 'P',
	  orbital: '3s',
	  be: 16
	}, {
	  element: 'S',
	  orbital: '3s',
	  be: 16
	}, {
	  element: 'Cl',
	  orbital: '3s',
	  be: 18
	}, {
	  element: 'Ar',
	  orbital: '3s',
	  be: 25
	}, {
	  element: 'K',
	  orbital: '3s',
	  be: 34
	}, {
	  element: 'Ca',
	  orbital: '3s',
	  be: 44
	}, {
	  element: 'Sc',
	  orbital: '3s',
	  be: 54
	}, {
	  element: 'Ti',
	  orbital: '3s',
	  be: 59
	}, {
	  element: 'V',
	  orbital: '3s',
	  be: 66
	}, {
	  element: 'Cr',
	  orbital: '3s',
	  be: 74
	}, {
	  element: 'Mn',
	  orbital: '3s',
	  be: 84
	}, {
	  element: 'Fe',
	  orbital: '3s',
	  be: 95
	}, {
	  element: 'Co',
	  orbital: '3s',
	  be: 101
	}, {
	  element: 'Ni',
	  orbital: '3s',
	  be: 112
	}, {
	  element: 'Cu',
	  orbital: '3s',
	  be: 120
	}, {
	  element: 'Zn',
	  orbital: '3s',
	  be: 137
	}, {
	  element: 'Ga',
	  orbital: '3s',
	  be: 158
	}, {
	  element: 'Ge',
	  orbital: '3s',
	  be: 181
	}, {
	  element: 'As',
	  orbital: '3s',
	  be: 204
	}, {
	  element: 'Se',
	  orbital: '3s',
	  be: 232
	}, {
	  element: 'Br',
	  orbital: '3s',
	  be: 257
	}, {
	  element: 'Kr',
	  orbital: '3s',
	  be: 289
	}, {
	  element: 'Rb',
	  orbital: '3s',
	  be: 322
	}, {
	  element: 'Sr',
	  orbital: '3s',
	  be: 358
	}, {
	  element: 'Y',
	  orbital: '3s',
	  be: 395
	}, {
	  element: 'Zr',
	  orbital: '3s',
	  be: 431
	}, {
	  element: 'Nb',
	  orbital: '3s',
	  be: 469
	}, {
	  element: 'Mo',
	  orbital: '3s',
	  be: 505
	}, {
	  element: 'Tc',
	  orbital: '3s',
	  be: 544
	}, {
	  element: 'Ru',
	  orbital: '3s',
	  be: 585
	}, {
	  element: 'Rh',
	  orbital: '3s',
	  be: 627
	}, {
	  element: 'Pd',
	  orbital: '3s',
	  be: 670
	}, {
	  element: 'Ag',
	  orbital: '3s',
	  be: 717
	}, {
	  element: 'Cd',
	  orbital: '3s',
	  be: 770
	}, {
	  element: 'In',
	  orbital: '3s',
	  be: 826
	}, {
	  element: 'Sn',
	  orbital: '3s',
	  be: 884
	}, {
	  element: 'Sb',
	  orbital: '3s',
	  be: 944
	}, {
	  element: 'Te',
	  orbital: '3s',
	  be: 1006
	}, {
	  element: 'I',
	  orbital: '3s',
	  be: 1072
	}, {
	  element: 'Xe',
	  orbital: '3s',
	  be: 1145
	}, {
	  element: 'Cs',
	  orbital: '3s',
	  be: 1217
	}, {
	  element: 'Si',
	  orbital: '3p1/2',
	  be: 3
	}, {
	  element: 'P',
	  orbital: '3p1/2',
	  be: 10
	}, {
	  element: 'S',
	  orbital: '3p1/2',
	  be: 8
	}, {
	  element: 'Cl',
	  orbital: '3p1/2',
	  be: 7
	}, {
	  element: 'Ar',
	  orbital: '3p1/2',
	  be: 12
	}, {
	  element: 'K',
	  orbital: '3p1/2',
	  be: 18
	}, {
	  element: 'Ca',
	  orbital: '3p1/2',
	  be: 26
	}, {
	  element: 'Sc',
	  orbital: '3p1/2',
	  be: 32
	}, {
	  element: 'Ti',
	  orbital: '3p1/2',
	  be: 34
	}, {
	  element: 'V',
	  orbital: '3p1/2',
	  be: 38
	}, {
	  element: 'Cr',
	  orbital: '3p1/2',
	  be: 43
	}, {
	  element: 'Mn',
	  orbital: '3p1/2',
	  be: 49
	}, {
	  element: 'Fe',
	  orbital: '3p1/2',
	  be: 56
	}, {
	  element: 'Co',
	  orbital: '3p1/2',
	  be: 60
	}, {
	  element: 'Ni',
	  orbital: '3p1/2',
	  be: 68
	}, {
	  element: 'Cu',
	  orbital: '3p1/2',
	  be: 74
	}, {
	  element: 'Zn',
	  orbital: '3p1/2',
	  be: 87
	}, {
	  element: 'Ga',
	  orbital: '3p1/2',
	  be: 107
	}, {
	  element: 'Ge',
	  orbital: '3p1/2',
	  be: 129
	}, {
	  element: 'As',
	  orbital: '3p1/2',
	  be: 147
	}, {
	  element: 'Se',
	  orbital: '3p1/2',
	  be: 168
	}, {
	  element: 'Br',
	  orbital: '3p1/2',
	  be: 189
	}, {
	  element: 'Kr',
	  orbital: '3p1/2',
	  be: 223
	}, {
	  element: 'Rb',
	  orbital: '3p1/2',
	  be: 248
	}, {
	  element: 'Sr',
	  orbital: '3p1/2',
	  be: 280
	}, {
	  element: 'Y',
	  orbital: '3p1/2',
	  be: 313
	}, {
	  element: 'Zr',
	  orbital: '3p1/2',
	  be: 345
	}, {
	  element: 'Nb',
	  orbital: '3p1/2',
	  be: 379
	}, {
	  element: 'Mo',
	  orbital: '3p1/2',
	  be: 410
	}, {
	  element: 'Tc',
	  orbital: '3p1/2',
	  be: 445
	}, {
	  element: 'Ru',
	  orbital: '3p1/2',
	  be: 483
	}, {
	  element: 'Rh',
	  orbital: '3p1/2',
	  be: 521
	}, {
	  element: 'Pd',
	  orbital: '3p1/2',
	  be: 559
	}, {
	  element: 'Ag',
	  orbital: '3p1/2',
	  be: 602
	}, {
	  element: 'Cd',
	  orbital: '3p1/2',
	  be: 651
	}, {
	  element: 'In',
	  orbital: '3p1/2',
	  be: 702
	}, {
	  element: 'Sn',
	  orbital: '3p1/2',
	  be: 757
	}, {
	  element: 'Sb',
	  orbital: '3p1/2',
	  be: 812
	}, {
	  element: 'Te',
	  orbital: '3p1/2',
	  be: 870
	}, {
	  element: 'I',
	  orbital: '3p1/2',
	  be: 931
	}, {
	  element: 'Xe',
	  orbital: '3p1/2',
	  be: 999
	}, {
	  element: 'Cs',
	  orbital: '3p1/2',
	  be: 1065
	}, {
	  element: 'Ba',
	  orbital: '3p1/2',
	  be: 1137
	}, {
	  element: 'La',
	  orbital: '3p1/2',
	  be: 1205
	}, {
	  element: 'Si',
	  orbital: '3p3/2',
	  be: 3
	}, {
	  element: 'P',
	  orbital: '3p3/2',
	  be: 10
	}, {
	  element: 'S',
	  orbital: '3p3/2',
	  be: 8
	}, {
	  element: 'Cl',
	  orbital: '3p3/2',
	  be: 7
	}, {
	  element: 'Ar',
	  orbital: '3p3/2',
	  be: 12
	}, {
	  element: 'K',
	  orbital: '3p3/2',
	  be: 18
	}, {
	  element: 'Ca',
	  orbital: '3p3/2',
	  be: 26
	}, {
	  element: 'Sc',
	  orbital: '3p3/2',
	  be: 32
	}, {
	  element: 'Ti',
	  orbital: '3p3/2',
	  be: 34
	}, {
	  element: 'V',
	  orbital: '3p3/2',
	  be: 38
	}, {
	  element: 'Cr',
	  orbital: '3p3/2',
	  be: 43
	}, {
	  element: 'Mn',
	  orbital: '3p3/2',
	  be: 49
	}, {
	  element: 'Fe',
	  orbital: '3p3/2',
	  be: 56
	}, {
	  element: 'Co',
	  orbital: '3p3/2',
	  be: 60
	}, {
	  element: 'Ni',
	  orbital: '3p3/2',
	  be: 68
	}, {
	  element: 'Cu',
	  orbital: '3p3/2',
	  be: 74
	}, {
	  element: 'Zn',
	  orbital: '3p3/2',
	  be: 87
	}, {
	  element: 'Ga',
	  orbital: '3p3/2',
	  be: 103
	}, {
	  element: 'Ge',
	  orbital: '3p3/2',
	  be: 122
	}, {
	  element: 'As',
	  orbital: '3p3/2',
	  be: 141
	}, {
	  element: 'Se',
	  orbital: '3p3/2',
	  be: 162
	}, {
	  element: 'Br',
	  orbital: '3p3/2',
	  be: 182
	}, {
	  element: 'Kr',
	  orbital: '3p3/2',
	  be: 214
	}, {
	  element: 'Rb',
	  orbital: '3p3/2',
	  be: 239
	}, {
	  element: 'Sr',
	  orbital: '3p3/2',
	  be: 269
	}, {
	  element: 'Y',
	  orbital: '3p3/2',
	  be: 301
	}, {
	  element: 'Zr',
	  orbital: '3p3/2',
	  be: 331
	}, {
	  element: 'Nb',
	  orbital: '3p3/2',
	  be: 363
	}, {
	  element: 'Mo',
	  orbital: '3p3/2',
	  be: 393
	}, {
	  element: 'Tc',
	  orbital: '3p3/2',
	  be: 425
	}, {
	  element: 'Ru',
	  orbital: '3p3/2',
	  be: 461
	}, {
	  element: 'Rh',
	  orbital: '3p3/2',
	  be: 496
	}, {
	  element: 'Pd',
	  orbital: '3p3/2',
	  be: 531
	}, {
	  element: 'Ag',
	  orbital: '3p3/2',
	  be: 571
	}, {
	  element: 'Cd',
	  orbital: '3p3/2',
	  be: 617
	}, {
	  element: 'In',
	  orbital: '3p3/2',
	  be: 664
	}, {
	  element: 'Sn',
	  orbital: '3p3/2',
	  be: 715
	}, {
	  element: 'Sb',
	  orbital: '3p3/2',
	  be: 766
	}, {
	  element: 'Te',
	  orbital: '3p3/2',
	  be: 819
	}, {
	  element: 'I',
	  orbital: '3p3/2',
	  be: 875
	}, {
	  element: 'Xe',
	  orbital: '3p3/2',
	  be: 937
	}, {
	  element: 'Cs',
	  orbital: '3p3/2',
	  be: 998
	}, {
	  element: 'Ba',
	  orbital: '3p3/2',
	  be: 1063
	}, {
	  element: 'La',
	  orbital: '3p3/2',
	  be: 1124
	}, {
	  element: 'Ce',
	  orbital: '3p3/2',
	  be: 1186
	}, {
	  element: 'Pr',
	  orbital: '3p3/2',
	  be: 1243
	}, {
	  element: 'Si',
	  orbital: '3p',
	  be: 3
	}, {
	  element: 'P',
	  orbital: '3p',
	  be: 10
	}, {
	  element: 'S',
	  orbital: '3p',
	  be: 8
	}, {
	  element: 'Cl',
	  orbital: '3p',
	  be: 7
	}, {
	  element: 'Ar',
	  orbital: '3p',
	  be: 12
	}, {
	  element: 'K',
	  orbital: '3p',
	  be: 18
	}, {
	  element: 'Ca',
	  orbital: '3p',
	  be: 26
	}, {
	  element: 'Sc',
	  orbital: '3p',
	  be: 32
	}, {
	  element: 'Ti',
	  orbital: '3p',
	  be: 34
	}, {
	  element: 'V',
	  orbital: '3p',
	  be: 38
	}, {
	  element: 'Cr',
	  orbital: '3p',
	  be: 43
	}, {
	  element: 'Mn',
	  orbital: '3p',
	  be: 49
	}, {
	  element: 'Fe',
	  orbital: '3p',
	  be: 56
	}, {
	  element: 'Co',
	  orbital: '3p',
	  be: 60
	}, {
	  element: 'Ni',
	  orbital: '3p',
	  be: 68
	}, {
	  element: 'Cu',
	  orbital: '3p',
	  be: 74
	}, {
	  element: 'Zn',
	  orbital: '3p',
	  be: 87
	}, {
	  element: 'Ga',
	  orbital: '3p',
	  be: 103
	}, {
	  element: 'Ge',
	  orbital: '3p',
	  be: 122
	}, {
	  element: 'As',
	  orbital: '3p',
	  be: 141
	}, {
	  element: 'Se',
	  orbital: '3p',
	  be: 162
	}, {
	  element: 'Br',
	  orbital: '3p',
	  be: 182
	}, {
	  element: 'Kr',
	  orbital: '3p',
	  be: 214
	}, {
	  element: 'Rb',
	  orbital: '3p',
	  be: 239
	}, {
	  element: 'Sr',
	  orbital: '3p',
	  be: 269
	}, {
	  element: 'Y',
	  orbital: '3p',
	  be: 301
	}, {
	  element: 'Zr',
	  orbital: '3p',
	  be: 331
	}, {
	  element: 'Nb',
	  orbital: '3p',
	  be: 363
	}, {
	  element: 'Mo',
	  orbital: '3p',
	  be: 393
	}, {
	  element: 'Tc',
	  orbital: '3p',
	  be: 425
	}, {
	  element: 'Ru',
	  orbital: '3p',
	  be: 461
	}, {
	  element: 'Rh',
	  orbital: '3p',
	  be: 496
	}, {
	  element: 'Pd',
	  orbital: '3p',
	  be: 531
	}, {
	  element: 'Ag',
	  orbital: '3p',
	  be: 571
	}, {
	  element: 'Cd',
	  orbital: '3p',
	  be: 617
	}, {
	  element: 'In',
	  orbital: '3p',
	  be: 664
	}, {
	  element: 'Sn',
	  orbital: '3p',
	  be: 715
	}, {
	  element: 'Sb',
	  orbital: '3p',
	  be: 766
	}, {
	  element: 'Te',
	  orbital: '3p',
	  be: 819
	}, {
	  element: 'I',
	  orbital: '3p',
	  be: 875
	}, {
	  element: 'Xe',
	  orbital: '3p',
	  be: 937
	}, {
	  element: 'Cs',
	  orbital: '3p',
	  be: 998
	}, {
	  element: 'Ba',
	  orbital: '3p',
	  be: 1063
	}, {
	  element: 'La',
	  orbital: '3p',
	  be: 1124
	}, {
	  element: 'Ce',
	  orbital: '3p',
	  be: 1186
	}, {
	  element: 'Pr',
	  orbital: '3p',
	  be: 1243
	}, {
	  element: 'Sc',
	  orbital: '3d3/2',
	  be: 7
	}, {
	  element: 'Ti',
	  orbital: '3d3/2',
	  be: 3
	}, {
	  element: 'V',
	  orbital: '3d3/2',
	  be: 2
	}, {
	  element: 'Cr',
	  orbital: '3d3/2',
	  be: 2
	}, {
	  element: 'Mn',
	  orbital: '3d3/2',
	  be: 4
	}, {
	  element: 'Fe',
	  orbital: '3d3/2',
	  be: 6
	}, {
	  element: 'Co',
	  orbital: '3d3/2',
	  be: 3
	}, {
	  element: 'Ni',
	  orbital: '3d3/2',
	  be: 4
	}, {
	  element: 'Cu',
	  orbital: '3d3/2',
	  be: 2
	}, {
	  element: 'Zn',
	  orbital: '3d3/2',
	  be: 9
	}, {
	  element: 'Ga',
	  orbital: '3d3/2',
	  be: 18
	}, {
	  element: 'Ge',
	  orbital: '3d3/2',
	  be: 29
	}, {
	  element: 'As',
	  orbital: '3d3/2',
	  be: 41
	}, {
	  element: 'Se',
	  orbital: '3d3/2',
	  be: 57
	}, {
	  element: 'Br',
	  orbital: '3d3/2',
	  be: 70
	}, {
	  element: 'Kr',
	  orbital: '3d3/2',
	  be: 89
	}, {
	  element: 'Rb',
	  orbital: '3d3/2',
	  be: 112
	}, {
	  element: 'Sr',
	  orbital: '3d3/2',
	  be: 135
	}, {
	  element: 'Y',
	  orbital: '3d3/2',
	  be: 160
	}, {
	  element: 'Zr',
	  orbital: '3d3/2',
	  be: 183
	}, {
	  element: 'Nb',
	  orbital: '3d3/2',
	  be: 208
	}, {
	  element: 'Mo',
	  orbital: '3d3/2',
	  be: 230
	}, {
	  element: 'Tc',
	  orbital: '3d3/2',
	  be: 257
	}, {
	  element: 'Ru',
	  orbital: '3d3/2',
	  be: 284
	}, {
	  element: 'Rh',
	  orbital: '3d3/2',
	  be: 312
	}, {
	  element: 'Pd',
	  orbital: '3d3/2',
	  be: 340
	}, {
	  element: 'Ag',
	  orbital: '3d3/2',
	  be: 373
	}, {
	  element: 'Cd',
	  orbital: '3d3/2',
	  be: 411
	}, {
	  element: 'In',
	  orbital: '3d3/2',
	  be: 451
	}, {
	  element: 'Sn',
	  orbital: '3d3/2',
	  be: 494
	}, {
	  element: 'Sb',
	  orbital: '3d3/2',
	  be: 537
	}, {
	  element: 'Te',
	  orbital: '3d3/2',
	  be: 582
	}, {
	  element: 'I',
	  orbital: '3d3/2',
	  be: 631
	}, {
	  element: 'Xe',
	  orbital: '3d3/2',
	  be: 685
	}, {
	  element: 'Cs',
	  orbital: '3d3/2',
	  be: 740
	}, {
	  element: 'Ba',
	  orbital: '3d3/2',
	  be: 796
	}, {
	  element: 'La',
	  orbital: '3d3/2',
	  be: 849
	}, {
	  element: 'Ce',
	  orbital: '3d3/2',
	  be: 902
	}, {
	  element: 'Pr',
	  orbital: '3d3/2',
	  be: 951
	}, {
	  element: 'Nd',
	  orbital: '3d3/2',
	  be: 1000
	}, {
	  element: 'Pm',
	  orbital: '3d3/2',
	  be: 1052
	}, {
	  element: 'Sm',
	  orbital: '3d3/2',
	  be: 1107
	}, {
	  element: 'Eu',
	  orbital: '3d3/2',
	  be: 1161
	}, {
	  element: 'Gd',
	  orbital: '3d3/2',
	  be: 1218
	}, {
	  element: 'Sc',
	  orbital: '3d5/2',
	  be: 7
	}, {
	  element: 'Ti',
	  orbital: '3d5/2',
	  be: 3
	}, {
	  element: 'V',
	  orbital: '3d5/2',
	  be: 2
	}, {
	  element: 'Cr',
	  orbital: '3d5/2',
	  be: 2
	}, {
	  element: 'Mn',
	  orbital: '3d5/2',
	  be: 4
	}, {
	  element: 'Fe',
	  orbital: '3d5/2',
	  be: 6
	}, {
	  element: 'Co',
	  orbital: '3d5/2',
	  be: 3
	}, {
	  element: 'Ni',
	  orbital: '3d5/2',
	  be: 4
	}, {
	  element: 'Cu',
	  orbital: '3d5/2',
	  be: 2
	}, {
	  element: 'Zn',
	  orbital: '3d5/2',
	  be: 9
	}, {
	  element: 'Ga',
	  orbital: '3d5/2',
	  be: 18
	}, {
	  element: 'Ge',
	  orbital: '3d5/2',
	  be: 29
	}, {
	  element: 'As',
	  orbital: '3d5/2',
	  be: 41
	}, {
	  element: 'Se',
	  orbital: '3d5/2',
	  be: 57
	}, {
	  element: 'Br',
	  orbital: '3d5/2',
	  be: 69
	}, {
	  element: 'Kr',
	  orbital: '3d5/2',
	  be: 89
	}, {
	  element: 'Rb',
	  orbital: '3d5/2',
	  be: 111
	}, {
	  element: 'Sr',
	  orbital: '3d5/2',
	  be: 133
	}, {
	  element: 'Y',
	  orbital: '3d5/2',
	  be: 158
	}, {
	  element: 'Zr',
	  orbital: '3d5/2',
	  be: 180
	}, {
	  element: 'Nb',
	  orbital: '3d5/2',
	  be: 205
	}, {
	  element: 'Mo',
	  orbital: '3d5/2',
	  be: 227
	}, {
	  element: 'Tc',
	  orbital: '3d5/2',
	  be: 253
	}, {
	  element: 'Ru',
	  orbital: '3d5/2',
	  be: 279
	}, {
	  element: 'Rh',
	  orbital: '3d5/2',
	  be: 307
	}, {
	  element: 'Pd',
	  orbital: '3d5/2',
	  be: 335
	}, {
	  element: 'Ag',
	  orbital: '3d5/2',
	  be: 367
	}, {
	  element: 'Cd',
	  orbital: '3d5/2',
	  be: 404
	}, {
	  element: 'In',
	  orbital: '3d5/2',
	  be: 443
	}, {
	  element: 'Sn',
	  orbital: '3d5/2',
	  be: 485
	}, {
	  element: 'Sb',
	  orbital: '3d5/2',
	  be: 528
	}, {
	  element: 'Te',
	  orbital: '3d5/2',
	  be: 572
	}, {
	  element: 'I',
	  orbital: '3d5/2',
	  be: 620
	}, {
	  element: 'Xe',
	  orbital: '3d5/2',
	  be: 672
	}, {
	  element: 'Cs',
	  orbital: '3d5/2',
	  be: 726
	}, {
	  element: 'Ba',
	  orbital: '3d5/2',
	  be: 781
	}, {
	  element: 'La',
	  orbital: '3d5/2',
	  be: 832
	}, {
	  element: 'Ce',
	  orbital: '3d5/2',
	  be: 884
	}, {
	  element: 'Pr',
	  orbital: '3d5/2',
	  be: 931
	}, {
	  element: 'Nd',
	  orbital: '3d5/2',
	  be: 978
	}, {
	  element: 'Pm',
	  orbital: '3d5/2',
	  be: 1027
	}, {
	  element: 'Sm',
	  orbital: '3d5/2',
	  be: 1081
	}, {
	  element: 'Eu',
	  orbital: '3d5/2',
	  be: 1131
	}, {
	  element: 'Gd',
	  orbital: '3d5/2',
	  be: 1186
	}, {
	  element: 'Tb',
	  orbital: '3d5/2',
	  be: 1242
	}, {
	  element: 'Sc',
	  orbital: '3d',
	  be: 7
	}, {
	  element: 'Ti',
	  orbital: '3d',
	  be: 3
	}, {
	  element: 'V',
	  orbital: '3d',
	  be: 2
	}, {
	  element: 'Cr',
	  orbital: '3d',
	  be: 2
	}, {
	  element: 'Mn',
	  orbital: '3d',
	  be: 4
	}, {
	  element: 'Fe',
	  orbital: '3d',
	  be: 6
	}, {
	  element: 'Co',
	  orbital: '3d',
	  be: 3
	}, {
	  element: 'Ni',
	  orbital: '3d',
	  be: 4
	}, {
	  element: 'Cu',
	  orbital: '3d',
	  be: 2
	}, {
	  element: 'Zn',
	  orbital: '3d',
	  be: 9
	}, {
	  element: 'Ga',
	  orbital: '3d',
	  be: 18
	}, {
	  element: 'Ge',
	  orbital: '3d',
	  be: 29
	}, {
	  element: 'As',
	  orbital: '3d',
	  be: 41
	}, {
	  element: 'Se',
	  orbital: '3d',
	  be: 57
	}, {
	  element: 'Br',
	  orbital: '3d',
	  be: 69
	}, {
	  element: 'Kr',
	  orbital: '3d',
	  be: 89
	}, {
	  element: 'Rb',
	  orbital: '3d',
	  be: 111
	}, {
	  element: 'Sr',
	  orbital: '3d',
	  be: 133
	}, {
	  element: 'Y',
	  orbital: '3d',
	  be: 158
	}, {
	  element: 'Zr',
	  orbital: '3d',
	  be: 180
	}, {
	  element: 'Nb',
	  orbital: '3d',
	  be: 205
	}, {
	  element: 'Mo',
	  orbital: '3d',
	  be: 227
	}, {
	  element: 'Tc',
	  orbital: '3d',
	  be: 253
	}, {
	  element: 'Ru',
	  orbital: '3d',
	  be: 279
	}, {
	  element: 'Rh',
	  orbital: '3d',
	  be: 307
	}, {
	  element: 'Pd',
	  orbital: '3d',
	  be: 335
	}, {
	  element: 'Ag',
	  orbital: '3d',
	  be: 367
	}, {
	  element: 'Cd',
	  orbital: '3d',
	  be: 404
	}, {
	  element: 'In',
	  orbital: '3d',
	  be: 443
	}, {
	  element: 'Sn',
	  orbital: '3d',
	  be: 485
	}, {
	  element: 'Sb',
	  orbital: '3d',
	  be: 528
	}, {
	  element: 'Te',
	  orbital: '3d',
	  be: 572
	}, {
	  element: 'I',
	  orbital: '3d',
	  be: 620
	}, {
	  element: 'Xe',
	  orbital: '3d',
	  be: 672
	}, {
	  element: 'Cs',
	  orbital: '3d',
	  be: 726
	}, {
	  element: 'Ba',
	  orbital: '3d',
	  be: 781
	}, {
	  element: 'La',
	  orbital: '3d',
	  be: 832
	}, {
	  element: 'Ce',
	  orbital: '3d',
	  be: 884
	}, {
	  element: 'Pr',
	  orbital: '3d',
	  be: 931
	}, {
	  element: 'Nd',
	  orbital: '3d',
	  be: 978
	}, {
	  element: 'Pm',
	  orbital: '3d',
	  be: 1027
	}, {
	  element: 'Sm',
	  orbital: '3d',
	  be: 1081
	}, {
	  element: 'Eu',
	  orbital: '3d',
	  be: 1131
	}, {
	  element: 'Gd',
	  orbital: '3d',
	  be: 1186
	}, {
	  element: 'Tb',
	  orbital: '3d',
	  be: 1242
	}, {
	  element: 'Br',
	  orbital: '4s',
	  be: 27
	}, {
	  element: 'Kr',
	  orbital: '4s',
	  be: 24
	}, {
	  element: 'Rb',
	  orbital: '4s',
	  be: 30
	}, {
	  element: 'Sr',
	  orbital: '4s',
	  be: 38
	}, {
	  element: 'Y',
	  orbital: '4s',
	  be: 46
	}, {
	  element: 'Zr',
	  orbital: '4s',
	  be: 52
	}, {
	  element: 'Nb',
	  orbital: '4s',
	  be: 58
	}, {
	  element: 'Mo',
	  orbital: '4s',
	  be: 62
	}, {
	  element: 'Tc',
	  orbital: '4s',
	  be: 68
	}, {
	  element: 'Ru',
	  orbital: '4s',
	  be: 75
	}, {
	  element: 'Rh',
	  orbital: '4s',
	  be: 81
	}, {
	  element: 'Pd',
	  orbital: '4s',
	  be: 86
	}, {
	  element: 'Ag',
	  orbital: '4s',
	  be: 95
	}, {
	  element: 'Cd',
	  orbital: '4s',
	  be: 108
	}, {
	  element: 'In',
	  orbital: '4s',
	  be: 122
	}, {
	  element: 'Sn',
	  orbital: '4s',
	  be: 137
	}, {
	  element: 'Sb',
	  orbital: '4s',
	  be: 152
	}, {
	  element: 'Te',
	  orbital: '4s',
	  be: 168
	}, {
	  element: 'I',
	  orbital: '4s',
	  be: 186
	}, {
	  element: 'Xe',
	  orbital: '4s',
	  be: 208
	}, {
	  element: 'Cs',
	  orbital: '4s',
	  be: 231
	}, {
	  element: 'Ba',
	  orbital: '4s',
	  be: 253
	}, {
	  element: 'La',
	  orbital: '4s',
	  be: 271
	}, {
	  element: 'Ce',
	  orbital: '4s',
	  be: 290
	}, {
	  element: 'Pr',
	  orbital: '4s',
	  be: 305
	}, {
	  element: 'Nd',
	  orbital: '4s',
	  be: 316
	}, {
	  element: 'Pm',
	  orbital: '4s',
	  be: 331
	}, {
	  element: 'Sm',
	  orbital: '4s',
	  be: 347
	}, {
	  element: 'Eu',
	  orbital: '4s',
	  be: 360
	}, {
	  element: 'Gd',
	  orbital: '4s',
	  be: 376
	}, {
	  element: 'Tb',
	  orbital: '4s',
	  be: 398
	}, {
	  element: 'Dy',
	  orbital: '4s',
	  be: 416
	}, {
	  element: 'Ho',
	  orbital: '4s',
	  be: 436
	}, {
	  element: 'Er',
	  orbital: '4s',
	  be: 449
	}, {
	  element: 'Tm',
	  orbital: '4s',
	  be: 472
	}, {
	  element: 'Yb',
	  orbital: '4s',
	  be: 487
	}, {
	  element: 'Lu',
	  orbital: '4s',
	  be: 506
	}, {
	  element: 'Hf',
	  orbital: '4s',
	  be: 538
	}, {
	  element: 'Ta',
	  orbital: '4s',
	  be: 566
	}, {
	  element: 'W',
	  orbital: '4s',
	  be: 595
	}, {
	  element: 'Re',
	  orbital: '4s',
	  be: 625
	}, {
	  element: 'Os',
	  orbital: '4s',
	  be: 655
	}, {
	  element: 'Ir',
	  orbital: '4s',
	  be: 690
	}, {
	  element: 'Pt',
	  orbital: '4s',
	  be: 724
	}, {
	  element: 'Au',
	  orbital: '4s',
	  be: 759
	}, {
	  element: 'Hg',
	  orbital: '4s',
	  be: 800
	}, {
	  element: 'Tl',
	  orbital: '4s',
	  be: 846
	}, {
	  element: 'Pb',
	  orbital: '4s',
	  be: 894
	}, {
	  element: 'Bi',
	  orbital: '4s',
	  be: 939
	}, {
	  element: 'Po',
	  orbital: '4s',
	  be: 995
	}, {
	  element: 'At',
	  orbital: '4s',
	  be: 1042
	}, {
	  element: 'Rn',
	  orbital: '4s',
	  be: 1097
	}, {
	  element: 'Fr',
	  orbital: '4s',
	  be: 1153
	}, {
	  element: 'Ra',
	  orbital: '4s',
	  be: 1208
	}, {
	  element: 'Ga',
	  orbital: '4p1/2',
	  be: 1
	}, {
	  element: 'Ge',
	  orbital: '4p1/2',
	  be: 3
	}, {
	  element: 'As',
	  orbital: '4p1/2',
	  be: 3
	}, {
	  element: 'Se',
	  orbital: '4p1/2',
	  be: 6
	}, {
	  element: 'Br',
	  orbital: '4p1/2',
	  be: 5
	}, {
	  element: 'Kr',
	  orbital: '4p1/2',
	  be: 11
	}, {
	  element: 'Rb',
	  orbital: '4p1/2',
	  be: 15
	}, {
	  element: 'Sr',
	  orbital: '4p1/2',
	  be: 20
	}, {
	  element: 'Y',
	  orbital: '4p1/2',
	  be: 26
	}, {
	  element: 'Zr',
	  orbital: '4p1/2',
	  be: 29
	}, {
	  element: 'Nb',
	  orbital: '4p1/2',
	  be: 34
	}, {
	  element: 'Mo',
	  orbital: '4p1/2',
	  be: 35
	}, {
	  element: 'Tc',
	  orbital: '4p1/2',
	  be: 39
	}, {
	  element: 'Ru',
	  orbital: '4p1/2',
	  be: 43
	}, {
	  element: 'Rh',
	  orbital: '4p1/2',
	  be: 48
	}, {
	  element: 'Pd',
	  orbital: '4p1/2',
	  be: 51
	}, {
	  element: 'Ag',
	  orbital: '4p1/2',
	  be: 62
	}, {
	  element: 'Cd',
	  orbital: '4p1/2',
	  be: 67
	}, {
	  element: 'In',
	  orbital: '4p1/2',
	  be: 77
	}, {
	  element: 'Sn',
	  orbital: '4p1/2',
	  be: 89
	}, {
	  element: 'Sb',
	  orbital: '4p1/2',
	  be: 99
	}, {
	  element: 'Te',
	  orbital: '4p1/2',
	  be: 110
	}, {
	  element: 'I',
	  orbital: '4p1/2',
	  be: 123
	}, {
	  element: 'Xe',
	  orbital: '4p1/2',
	  be: 147
	}, {
	  element: 'Cs',
	  orbital: '4p1/2',
	  be: 172
	}, {
	  element: 'Ba',
	  orbital: '4p1/2',
	  be: 192
	}, {
	  element: 'La',
	  orbital: '4p1/2',
	  be: 206
	}, {
	  element: 'Ce',
	  orbital: '4p1/2',
	  be: 224
	}, {
	  element: 'Pr',
	  orbital: '4p1/2',
	  be: 237
	}, {
	  element: 'Nd',
	  orbital: '4p1/2',
	  be: 244
	}, {
	  element: 'Pm',
	  orbital: '4p1/2',
	  be: 255
	}, {
	  element: 'Sm',
	  orbital: '4p1/2',
	  be: 267
	}, {
	  element: 'Eu',
	  orbital: '4p1/2',
	  be: 284
	}, {
	  element: 'Gd',
	  orbital: '4p1/2',
	  be: 289
	}, {
	  element: 'Tb',
	  orbital: '4p1/2',
	  be: 311
	}, {
	  element: 'Dy',
	  orbital: '4p1/2',
	  be: 332
	}, {
	  element: 'Ho',
	  orbital: '4p1/2',
	  be: 343
	}, {
	  element: 'Er',
	  orbital: '4p1/2',
	  be: 366
	}, {
	  element: 'Tm',
	  orbital: '4p1/2',
	  be: 386
	}, {
	  element: 'Yb',
	  orbital: '4p1/2',
	  be: 396
	}, {
	  element: 'Lu',
	  orbital: '4p1/2',
	  be: 410
	}, {
	  element: 'Hf',
	  orbital: '4p1/2',
	  be: 437
	}, {
	  element: 'Ta',
	  orbital: '4p1/2',
	  be: 465
	}, {
	  element: 'W',
	  orbital: '4p1/2',
	  be: 492
	}, {
	  element: 'Re',
	  orbital: '4p1/2',
	  be: 518
	}, {
	  element: 'Os',
	  orbital: '4p1/2',
	  be: 547
	}, {
	  element: 'Ir',
	  orbital: '4p1/2',
	  be: 577
	}, {
	  element: 'Pt',
	  orbital: '4p1/2',
	  be: 608
	}, {
	  element: 'Au',
	  orbital: '4p1/2',
	  be: 644
	}, {
	  element: 'Hg',
	  orbital: '4p1/2',
	  be: 677
	}, {
	  element: 'Tl',
	  orbital: '4p1/2',
	  be: 722
	}, {
	  element: 'Pb',
	  orbital: '4p1/2',
	  be: 764
	}, {
	  element: 'Bi',
	  orbital: '4p1/2',
	  be: 806
	}, {
	  element: 'Po',
	  orbital: '4p1/2',
	  be: 851
	}, {
	  element: 'At',
	  orbital: '4p1/2',
	  be: 886
	}, {
	  element: 'Rn',
	  orbital: '4p1/2',
	  be: 929
	}, {
	  element: 'Fr',
	  orbital: '4p1/2',
	  be: 980
	}, {
	  element: 'Ra',
	  orbital: '4p1/2',
	  be: 1058
	}, {
	  element: 'Ac',
	  orbital: '4p1/2',
	  be: 1080
	}, {
	  element: 'Th',
	  orbital: '4p1/2',
	  be: 1168
	}, {
	  element: 'Ga',
	  orbital: '4p3/2',
	  be: 1
	}, {
	  element: 'Ge',
	  orbital: '4p3/2',
	  be: 3
	}, {
	  element: 'As',
	  orbital: '4p3/2',
	  be: 3
	}, {
	  element: 'Se',
	  orbital: '4p3/2',
	  be: 6
	}, {
	  element: 'Br',
	  orbital: '4p3/2',
	  be: 5
	}, {
	  element: 'Kr',
	  orbital: '4p3/2',
	  be: 11
	}, {
	  element: 'Rb',
	  orbital: '4p3/2',
	  be: 14
	}, {
	  element: 'Sr',
	  orbital: '4p3/2',
	  be: 20
	}, {
	  element: 'Y',
	  orbital: '4p3/2',
	  be: 26
	}, {
	  element: 'Zr',
	  orbital: '4p3/2',
	  be: 29
	}, {
	  element: 'Nb',
	  orbital: '4p3/2',
	  be: 34
	}, {
	  element: 'Mo',
	  orbital: '4p3/2',
	  be: 35
	}, {
	  element: 'Tc',
	  orbital: '4p3/2',
	  be: 39
	}, {
	  element: 'Ru',
	  orbital: '4p3/2',
	  be: 43
	}, {
	  element: 'Rh',
	  orbital: '4p3/2',
	  be: 48
	}, {
	  element: 'Pd',
	  orbital: '4p3/2',
	  be: 56
	}, {
	  element: 'Ag',
	  orbital: '4p3/2',
	  be: 56
	}, {
	  element: 'Cd',
	  orbital: '4p3/2',
	  be: 67
	}, {
	  element: 'In',
	  orbital: '4p3/2',
	  be: 77
	}, {
	  element: 'Sn',
	  orbital: '4p3/2',
	  be: 89
	}, {
	  element: 'Sb',
	  orbital: '4p3/2',
	  be: 99
	}, {
	  element: 'Te',
	  orbital: '4p3/2',
	  be: 110
	}, {
	  element: 'I',
	  orbital: '4p3/2',
	  be: 123
	}, {
	  element: 'Xe',
	  orbital: '4p3/2',
	  be: 147
	}, {
	  element: 'Cs',
	  orbital: '4p3/2',
	  be: 162
	}, {
	  element: 'Ba',
	  orbital: '4p3/2',
	  be: 180
	}, {
	  element: 'La',
	  orbital: '4p3/2',
	  be: 192
	}, {
	  element: 'Ce',
	  orbital: '4p3/2',
	  be: 208
	}, {
	  element: 'Pr',
	  orbital: '4p3/2',
	  be: 218
	}, {
	  element: 'Nd',
	  orbital: '4p3/2',
	  be: 225
	}, {
	  element: 'Pm',
	  orbital: '4p3/2',
	  be: 237
	}, {
	  element: 'Sm',
	  orbital: '4p3/2',
	  be: 249
	}, {
	  element: 'Eu',
	  orbital: '4p3/2',
	  be: 257
	}, {
	  element: 'Gd',
	  orbital: '4p3/2',
	  be: 271
	}, {
	  element: 'Tb',
	  orbital: '4p3/2',
	  be: 286
	}, {
	  element: 'Dy',
	  orbital: '4p3/2',
	  be: 293
	}, {
	  element: 'Ho',
	  orbital: '4p3/2',
	  be: 306
	}, {
	  element: 'Er',
	  orbital: '4p3/2',
	  be: 320
	}, {
	  element: 'Tm',
	  orbital: '4p3/2',
	  be: 337
	}, {
	  element: 'Yb',
	  orbital: '4p3/2',
	  be: 343
	}, {
	  element: 'Lu',
	  orbital: '4p3/2',
	  be: 359
	}, {
	  element: 'Hf',
	  orbital: '4p3/2',
	  be: 380
	}, {
	  element: 'Ta',
	  orbital: '4p3/2',
	  be: 405
	}, {
	  element: 'W',
	  orbital: '4p3/2',
	  be: 426
	}, {
	  element: 'Re',
	  orbital: '4p3/2',
	  be: 445
	}, {
	  element: 'Os',
	  orbital: '4p3/2',
	  be: 469
	}, {
	  element: 'Ir',
	  orbital: '4p3/2',
	  be: 495
	}, {
	  element: 'Pt',
	  orbital: '4p3/2',
	  be: 519
	}, {
	  element: 'Au',
	  orbital: '4p3/2',
	  be: 546
	}, {
	  element: 'Hg',
	  orbital: '4p3/2',
	  be: 571
	}, {
	  element: 'Tl',
	  orbital: '4p3/2',
	  be: 609
	}, {
	  element: 'Pb',
	  orbital: '4p3/2',
	  be: 645
	}, {
	  element: 'Bi',
	  orbital: '4p3/2',
	  be: 679
	}, {
	  element: 'Po',
	  orbital: '4p3/2',
	  be: 705
	}, {
	  element: 'At',
	  orbital: '4p3/2',
	  be: 740
	}, {
	  element: 'Rn',
	  orbital: '4p3/2',
	  be: 768
	}, {
	  element: 'Fr',
	  orbital: '4p3/2',
	  be: 810
	}, {
	  element: 'Ra',
	  orbital: '4p3/2',
	  be: 879
	}, {
	  element: 'Ac',
	  orbital: '4p3/2',
	  be: 890
	}, {
	  element: 'Th',
	  orbital: '4p3/2',
	  be: 968
	}, {
	  element: 'Ga',
	  orbital: '4p',
	  be: 1
	}, {
	  element: 'Ge',
	  orbital: '4p',
	  be: 3
	}, {
	  element: 'As',
	  orbital: '4p',
	  be: 3
	}, {
	  element: 'Se',
	  orbital: '4p',
	  be: 6
	}, {
	  element: 'Br',
	  orbital: '4p',
	  be: 5
	}, {
	  element: 'Kr',
	  orbital: '4p',
	  be: 11
	}, {
	  element: 'Rb',
	  orbital: '4p',
	  be: 14
	}, {
	  element: 'Sr',
	  orbital: '4p',
	  be: 20
	}, {
	  element: 'Y',
	  orbital: '4p',
	  be: 26
	}, {
	  element: 'Zr',
	  orbital: '4p',
	  be: 29
	}, {
	  element: 'Nb',
	  orbital: '4p',
	  be: 34
	}, {
	  element: 'Mo',
	  orbital: '4p',
	  be: 35
	}, {
	  element: 'Tc',
	  orbital: '4p',
	  be: 39
	}, {
	  element: 'Ru',
	  orbital: '4p',
	  be: 43
	}, {
	  element: 'Rh',
	  orbital: '4p',
	  be: 48
	}, {
	  element: 'Pd',
	  orbital: '4p',
	  be: 56
	}, {
	  element: 'Ag',
	  orbital: '4p',
	  be: 56
	}, {
	  element: 'Cd',
	  orbital: '4p',
	  be: 67
	}, {
	  element: 'In',
	  orbital: '4p',
	  be: 77
	}, {
	  element: 'Sn',
	  orbital: '4p',
	  be: 89
	}, {
	  element: 'Sb',
	  orbital: '4p',
	  be: 99
	}, {
	  element: 'Te',
	  orbital: '4p',
	  be: 110
	}, {
	  element: 'I',
	  orbital: '4p',
	  be: 123
	}, {
	  element: 'Xe',
	  orbital: '4p',
	  be: 147
	}, {
	  element: 'Cs',
	  orbital: '4p',
	  be: 162
	}, {
	  element: 'Ba',
	  orbital: '4p',
	  be: 180
	}, {
	  element: 'La',
	  orbital: '4p',
	  be: 192
	}, {
	  element: 'Ce',
	  orbital: '4p',
	  be: 208
	}, {
	  element: 'Pr',
	  orbital: '4p',
	  be: 218
	}, {
	  element: 'Nd',
	  orbital: '4p',
	  be: 225
	}, {
	  element: 'Pm',
	  orbital: '4p',
	  be: 237
	}, {
	  element: 'Sm',
	  orbital: '4p',
	  be: 249
	}, {
	  element: 'Eu',
	  orbital: '4p',
	  be: 257
	}, {
	  element: 'Gd',
	  orbital: '4p',
	  be: 271
	}, {
	  element: 'Tb',
	  orbital: '4p',
	  be: 286
	}, {
	  element: 'Dy',
	  orbital: '4p',
	  be: 293
	}, {
	  element: 'Ho',
	  orbital: '4p',
	  be: 306
	}, {
	  element: 'Er',
	  orbital: '4p',
	  be: 320
	}, {
	  element: 'Tm',
	  orbital: '4p',
	  be: 337
	}, {
	  element: 'Yb',
	  orbital: '4p',
	  be: 343
	}, {
	  element: 'Lu',
	  orbital: '4p',
	  be: 359
	}, {
	  element: 'Hf',
	  orbital: '4p',
	  be: 380
	}, {
	  element: 'Ta',
	  orbital: '4p',
	  be: 405
	}, {
	  element: 'W',
	  orbital: '4p',
	  be: 426
	}, {
	  element: 'Re',
	  orbital: '4p',
	  be: 445
	}, {
	  element: 'Os',
	  orbital: '4p',
	  be: 469
	}, {
	  element: 'Ir',
	  orbital: '4p',
	  be: 495
	}, {
	  element: 'Pt',
	  orbital: '4p',
	  be: 519
	}, {
	  element: 'Au',
	  orbital: '4p',
	  be: 546
	}, {
	  element: 'Hg',
	  orbital: '4p',
	  be: 571
	}, {
	  element: 'Tl',
	  orbital: '4p',
	  be: 609
	}, {
	  element: 'Pb',
	  orbital: '4p',
	  be: 645
	}, {
	  element: 'Bi',
	  orbital: '4p',
	  be: 679
	}, {
	  element: 'Po',
	  orbital: '4p',
	  be: 705
	}, {
	  element: 'At',
	  orbital: '4p',
	  be: 740
	}, {
	  element: 'Rn',
	  orbital: '4p',
	  be: 768
	}, {
	  element: 'Fr',
	  orbital: '4p',
	  be: 810
	}, {
	  element: 'Ra',
	  orbital: '4p',
	  be: 879
	}, {
	  element: 'Ac',
	  orbital: '4p',
	  be: 890
	}, {
	  element: 'Th',
	  orbital: '4p',
	  be: 968
	}, {
	  element: 'Y',
	  orbital: '4d3/2',
	  be: 3
	}, {
	  element: 'Zr',
	  orbital: '4d3/2',
	  be: 3
	}, {
	  element: 'Nb',
	  orbital: '4d3/2',
	  be: 4
	}, {
	  element: 'Mo',
	  orbital: '4d3/2',
	  be: 2
	}, {
	  element: 'Tc',
	  orbital: '4d3/2',
	  be: 2
	}, {
	  element: 'Ru',
	  orbital: '4d3/2',
	  be: 2
	}, {
	  element: 'Rh',
	  orbital: '4d3/2',
	  be: 3
	}, {
	  element: 'Pd',
	  orbital: '4d3/2',
	  be: 1
	}, {
	  element: 'Ag',
	  orbital: '4d3/2',
	  be: 3
	}, {
	  element: 'Cd',
	  orbital: '4d3/2',
	  be: 9
	}, {
	  element: 'In',
	  orbital: '4d3/2',
	  be: 16
	}, {
	  element: 'Sn',
	  orbital: '4d3/2',
	  be: 24
	}, {
	  element: 'Sb',
	  orbital: '4d3/2',
	  be: 32
	}, {
	  element: 'Te',
	  orbital: '4d3/2',
	  be: 40
	}, {
	  element: 'I',
	  orbital: '4d3/2',
	  be: 50
	}, {
	  element: 'Xe',
	  orbital: '4d3/2',
	  be: 63
	}, {
	  element: 'Cs',
	  orbital: '4d3/2',
	  be: 79
	}, {
	  element: 'Ba',
	  orbital: '4d3/2',
	  be: 93
	}, {
	  element: 'La',
	  orbital: '4d3/2',
	  be: 99
	}, {
	  element: 'Ce',
	  orbital: '4d3/2',
	  be: 111
	}, {
	  element: 'Pr',
	  orbital: '4d3/2',
	  be: 114
	}, {
	  element: 'Nd',
	  orbital: '4d3/2',
	  be: 118
	}, {
	  element: 'Pm',
	  orbital: '4d3/2',
	  be: 121
	}, {
	  element: 'Sm',
	  orbital: '4d3/2',
	  be: 130
	}, {
	  element: 'Eu',
	  orbital: '4d3/2',
	  be: 134
	}, {
	  element: 'Gd',
	  orbital: '4d3/2',
	  be: 141
	}, {
	  element: 'Tb',
	  orbital: '4d3/2',
	  be: 148
	}, {
	  element: 'Dy',
	  orbital: '4d3/2',
	  be: 154
	}, {
	  element: 'Ho',
	  orbital: '4d3/2',
	  be: 161
	}, {
	  element: 'Er',
	  orbital: '4d3/2',
	  be: 177
	}, {
	  element: 'Tm',
	  orbital: '4d3/2',
	  be: 180
	}, {
	  element: 'Yb',
	  orbital: '4d3/2',
	  be: 197
	}, {
	  element: 'Lu',
	  orbital: '4d3/2',
	  be: 205
	}, {
	  element: 'Hf',
	  orbital: '4d3/2',
	  be: 224
	}, {
	  element: 'Ta',
	  orbital: '4d3/2',
	  be: 242
	}, {
	  element: 'W',
	  orbital: '4d3/2',
	  be: 259
	}, {
	  element: 'Re',
	  orbital: '4d3/2',
	  be: 274
	}, {
	  element: 'Os',
	  orbital: '4d3/2',
	  be: 290
	}, {
	  element: 'Ir',
	  orbital: '4d3/2',
	  be: 312
	}, {
	  element: 'Pt',
	  orbital: '4d3/2',
	  be: 331
	}, {
	  element: 'Au',
	  orbital: '4d3/2',
	  be: 352
	}, {
	  element: 'Hg',
	  orbital: '4d3/2',
	  be: 379
	}, {
	  element: 'Tl',
	  orbital: '4d3/2',
	  be: 407
	}, {
	  element: 'Pb',
	  orbital: '4d3/2',
	  be: 435
	}, {
	  element: 'Bi',
	  orbital: '4d3/2',
	  be: 464
	}, {
	  element: 'Po',
	  orbital: '4d3/2',
	  be: 500
	}, {
	  element: 'At',
	  orbital: '4d3/2',
	  be: 533
	}, {
	  element: 'Rn',
	  orbital: '4d3/2',
	  be: 567
	}, {
	  element: 'Fr',
	  orbital: '4d3/2',
	  be: 603
	}, {
	  element: 'Ra',
	  orbital: '4d3/2',
	  be: 636
	}, {
	  element: 'Ac',
	  orbital: '4d3/2',
	  be: 675
	}, {
	  element: 'Th',
	  orbital: '4d3/2',
	  be: 714
	}, {
	  element: 'U',
	  orbital: '4d3/2',
	  be: 781
	}, {
	  element: 'Y',
	  orbital: '4d5/2',
	  be: 3
	}, {
	  element: 'Zr',
	  orbital: '4d5/2',
	  be: 3
	}, {
	  element: 'Nb',
	  orbital: '4d5/2',
	  be: 4
	}, {
	  element: 'Mo',
	  orbital: '4d5/2',
	  be: 2
	}, {
	  element: 'Tc',
	  orbital: '4d5/2',
	  be: 2
	}, {
	  element: 'Ru',
	  orbital: '4d5/2',
	  be: 2
	}, {
	  element: 'Rh',
	  orbital: '4d5/2',
	  be: 3
	}, {
	  element: 'Pd',
	  orbital: '4d5/2',
	  be: 1
	}, {
	  element: 'Ag',
	  orbital: '4d5/2',
	  be: 3
	}, {
	  element: 'Cd',
	  orbital: '4d5/2',
	  be: 9
	}, {
	  element: 'In',
	  orbital: '4d5/2',
	  be: 16
	}, {
	  element: 'Sn',
	  orbital: '4d5/2',
	  be: 24
	}, {
	  element: 'Sb',
	  orbital: '4d5/2',
	  be: 32
	}, {
	  element: 'Te',
	  orbital: '4d5/2',
	  be: 40
	}, {
	  element: 'I',
	  orbital: '4d5/2',
	  be: 50
	}, {
	  element: 'Xe',
	  orbital: '4d5/2',
	  be: 63
	}, {
	  element: 'Cs',
	  orbital: '4d5/2',
	  be: 77
	}, {
	  element: 'Ba',
	  orbital: '4d5/2',
	  be: 90
	}, {
	  element: 'La',
	  orbital: '4d5/2',
	  be: 99
	}, {
	  element: 'Ce',
	  orbital: '4d5/2',
	  be: 111
	}, {
	  element: 'Pr',
	  orbital: '4d5/2',
	  be: 114
	}, {
	  element: 'Nd',
	  orbital: '4d5/2',
	  be: 118
	}, {
	  element: 'Pm',
	  orbital: '4d5/2',
	  be: 121
	}, {
	  element: 'Sm',
	  orbital: '4d5/2',
	  be: 130
	}, {
	  element: 'Eu',
	  orbital: '4d5/2',
	  be: 134
	}, {
	  element: 'Gd',
	  orbital: '4d5/2',
	  be: 141
	}, {
	  element: 'Tb',
	  orbital: '4d5/2',
	  be: 148
	}, {
	  element: 'Dy',
	  orbital: '4d5/2',
	  be: 154
	}, {
	  element: 'Ho',
	  orbital: '4d5/2',
	  be: 161
	}, {
	  element: 'Er',
	  orbital: '4d5/2',
	  be: 168
	}, {
	  element: 'Tm',
	  orbital: '4d5/2',
	  be: 180
	}, {
	  element: 'Yb',
	  orbital: '4d5/2',
	  be: 184
	}, {
	  element: 'Lu',
	  orbital: '4d5/2',
	  be: 195
	}, {
	  element: 'Hf',
	  orbital: '4d5/2',
	  be: 214
	}, {
	  element: 'Ta',
	  orbital: '4d5/2',
	  be: 230
	}, {
	  element: 'W',
	  orbital: '4d5/2',
	  be: 246
	}, {
	  element: 'Re',
	  orbital: '4d5/2',
	  be: 260
	}, {
	  element: 'Os',
	  orbital: '4d5/2',
	  be: 273
	}, {
	  element: 'Ir',
	  orbital: '4d5/2',
	  be: 295
	}, {
	  element: 'Pt',
	  orbital: '4d5/2',
	  be: 314
	}, {
	  element: 'Au',
	  orbital: '4d5/2',
	  be: 334
	}, {
	  element: 'Hg',
	  orbital: '4d5/2',
	  be: 360
	}, {
	  element: 'Tl',
	  orbital: '4d5/2',
	  be: 386
	}, {
	  element: 'Pb',
	  orbital: '4d5/2',
	  be: 413
	}, {
	  element: 'Bi',
	  orbital: '4d5/2',
	  be: 440
	}, {
	  element: 'Po',
	  orbital: '4d5/2',
	  be: 473
	}, {
	  element: 'At',
	  orbital: '4d5/2',
	  be: 507
	}, {
	  element: 'Rn',
	  orbital: '4d5/2',
	  be: 541
	}, {
	  element: 'Fr',
	  orbital: '4d5/2',
	  be: 577
	}, {
	  element: 'Ra',
	  orbital: '4d5/2',
	  be: 603
	}, {
	  element: 'Ac',
	  orbital: '4d5/2',
	  be: 639
	}, {
	  element: 'Th',
	  orbital: '4d5/2',
	  be: 677
	}, {
	  element: 'U',
	  orbital: '4d5/2',
	  be: 739
	}, {
	  element: 'Y',
	  orbital: '4d',
	  be: 3
	}, {
	  element: 'Zr',
	  orbital: '4d',
	  be: 3
	}, {
	  element: 'Nb',
	  orbital: '4d',
	  be: 4
	}, {
	  element: 'Mo',
	  orbital: '4d',
	  be: 2
	}, {
	  element: 'Tc',
	  orbital: '4d',
	  be: 2
	}, {
	  element: 'Ru',
	  orbital: '4d',
	  be: 2
	}, {
	  element: 'Rh',
	  orbital: '4d',
	  be: 3
	}, {
	  element: 'Pd',
	  orbital: '4d',
	  be: 1
	}, {
	  element: 'Ag',
	  orbital: '4d',
	  be: 3
	}, {
	  element: 'Cd',
	  orbital: '4d',
	  be: 9
	}, {
	  element: 'In',
	  orbital: '4d',
	  be: 16
	}, {
	  element: 'Sn',
	  orbital: '4d',
	  be: 24
	}, {
	  element: 'Sb',
	  orbital: '4d',
	  be: 32
	}, {
	  element: 'Te',
	  orbital: '4d',
	  be: 40
	}, {
	  element: 'I',
	  orbital: '4d',
	  be: 50
	}, {
	  element: 'Xe',
	  orbital: '4d',
	  be: 63
	}, {
	  element: 'Cs',
	  orbital: '4d',
	  be: 77
	}, {
	  element: 'Ba',
	  orbital: '4d',
	  be: 90
	}, {
	  element: 'La',
	  orbital: '4d',
	  be: 99
	}, {
	  element: 'Ce',
	  orbital: '4d',
	  be: 111
	}, {
	  element: 'Pr',
	  orbital: '4d',
	  be: 114
	}, {
	  element: 'Nd',
	  orbital: '4d',
	  be: 118
	}, {
	  element: 'Pm',
	  orbital: '4d',
	  be: 121
	}, {
	  element: 'Sm',
	  orbital: '4d',
	  be: 130
	}, {
	  element: 'Eu',
	  orbital: '4d',
	  be: 134
	}, {
	  element: 'Gd',
	  orbital: '4d',
	  be: 141
	}, {
	  element: 'Tb',
	  orbital: '4d',
	  be: 148
	}, {
	  element: 'Dy',
	  orbital: '4d',
	  be: 154
	}, {
	  element: 'Ho',
	  orbital: '4d',
	  be: 161
	}, {
	  element: 'Er',
	  orbital: '4d',
	  be: 168
	}, {
	  element: 'Tm',
	  orbital: '4d',
	  be: 180
	}, {
	  element: 'Yb',
	  orbital: '4d',
	  be: 184
	}, {
	  element: 'Lu',
	  orbital: '4d',
	  be: 195
	}, {
	  element: 'Hf',
	  orbital: '4d',
	  be: 214
	}, {
	  element: 'Ta',
	  orbital: '4d',
	  be: 230
	}, {
	  element: 'W',
	  orbital: '4d',
	  be: 246
	}, {
	  element: 'Re',
	  orbital: '4d',
	  be: 260
	}, {
	  element: 'Os',
	  orbital: '4d',
	  be: 273
	}, {
	  element: 'Ir',
	  orbital: '4d',
	  be: 295
	}, {
	  element: 'Pt',
	  orbital: '4d',
	  be: 314
	}, {
	  element: 'Au',
	  orbital: '4d',
	  be: 334
	}, {
	  element: 'Hg',
	  orbital: '4d',
	  be: 360
	}, {
	  element: 'Tl',
	  orbital: '4d',
	  be: 386
	}, {
	  element: 'Pb',
	  orbital: '4d',
	  be: 413
	}, {
	  element: 'Bi',
	  orbital: '4d',
	  be: 440
	}, {
	  element: 'Po',
	  orbital: '4d',
	  be: 473
	}, {
	  element: 'At',
	  orbital: '4d',
	  be: 507
	}, {
	  element: 'Rn',
	  orbital: '4d',
	  be: 541
	}, {
	  element: 'Fr',
	  orbital: '4d',
	  be: 577
	}, {
	  element: 'Ra',
	  orbital: '4d',
	  be: 603
	}, {
	  element: 'Ac',
	  orbital: '4d',
	  be: 639
	}, {
	  element: 'Th',
	  orbital: '4d',
	  be: 677
	}, {
	  element: 'U',
	  orbital: '4d',
	  be: 739
	}, {
	  element: 'Ce',
	  orbital: '4f5/2',
	  be: 1
	}, {
	  element: 'Pr',
	  orbital: '4f5/2',
	  be: 2
	}, {
	  element: 'Nd',
	  orbital: '4f5/2',
	  be: 2
	}, {
	  element: 'Pm',
	  orbital: '4f5/2',
	  be: 4
	}, {
	  element: 'Sm',
	  orbital: '4f5/2',
	  be: 7
	}, {
	  element: 'Tb',
	  orbital: '4f5/2',
	  be: 3
	}, {
	  element: 'Dy',
	  orbital: '4f5/2',
	  be: 4
	}, {
	  element: 'Ho',
	  orbital: '4f5/2',
	  be: 4
	}, {
	  element: 'Er',
	  orbital: '4f5/2',
	  be: 4
	}, {
	  element: 'Tm',
	  orbital: '4f5/2',
	  be: 5
	}, {
	  element: 'Yb',
	  orbital: '4f5/2',
	  be: 6
	}, {
	  element: 'Lu',
	  orbital: '4f5/2',
	  be: 7
	}, {
	  element: 'Hf',
	  orbital: '4f5/2',
	  be: 19
	}, {
	  element: 'Ta',
	  orbital: '4f5/2',
	  be: 27
	}, {
	  element: 'W',
	  orbital: '4f5/2',
	  be: 37
	}, {
	  element: 'Re',
	  orbital: '4f5/2',
	  be: 47
	}, {
	  element: 'Os',
	  orbital: '4f5/2',
	  be: 52
	}, {
	  element: 'Ir',
	  orbital: '4f5/2',
	  be: 63
	}, {
	  element: 'Pt',
	  orbital: '4f5/2',
	  be: 74
	}, {
	  element: 'Au',
	  orbital: '4f5/2',
	  be: 87
	}, {
	  element: 'Hg',
	  orbital: '4f5/2',
	  be: 103
	}, {
	  element: 'Tl',
	  orbital: '4f5/2',
	  be: 122
	}, {
	  element: 'Pb',
	  orbital: '4f5/2',
	  be: 143
	}, {
	  element: 'Bi',
	  orbital: '4f5/2',
	  be: 163
	}, {
	  element: 'Po',
	  orbital: '4f5/2',
	  be: 184
	}, {
	  element: 'At',
	  orbital: '4f5/2',
	  be: 210
	}, {
	  element: 'Rn',
	  orbital: '4f5/2',
	  be: 238
	}, {
	  element: 'Fr',
	  orbital: '4f5/2',
	  be: 268
	}, {
	  element: 'Ra',
	  orbital: '4f5/2',
	  be: 299
	}, {
	  element: 'Ac',
	  orbital: '4f5/2',
	  be: 319
	}, {
	  element: 'Th',
	  orbital: '4f5/2',
	  be: 344
	}, {
	  element: 'U',
	  orbital: '4f5/2',
	  be: 391
	}, {
	  element: 'Ce',
	  orbital: '4f7/2',
	  be: 1
	}, {
	  element: 'Pr',
	  orbital: '4f7/2',
	  be: 2
	}, {
	  element: 'Nd',
	  orbital: '4f7/2',
	  be: 2
	}, {
	  element: 'Pm',
	  orbital: '4f7/2',
	  be: 4
	}, {
	  element: 'Sm',
	  orbital: '4f7/2',
	  be: 7
	}, {
	  element: 'Tb',
	  orbital: '4f7/2',
	  be: 3
	}, {
	  element: 'Dy',
	  orbital: '4f7/2',
	  be: 4
	}, {
	  element: 'Ho',
	  orbital: '4f7/2',
	  be: 4
	}, {
	  element: 'Er',
	  orbital: '4f7/2',
	  be: 4
	}, {
	  element: 'Tm',
	  orbital: '4f7/2',
	  be: 5
	}, {
	  element: 'Yb',
	  orbital: '4f7/2',
	  be: 6
	}, {
	  element: 'Lu',
	  orbital: '4f7/2',
	  be: 7
	}, {
	  element: 'Hf',
	  orbital: '4f7/2',
	  be: 18
	}, {
	  element: 'Ta',
	  orbital: '4f7/2',
	  be: 25
	}, {
	  element: 'W',
	  orbital: '4f7/2',
	  be: 34
	}, {
	  element: 'Re',
	  orbital: '4f7/2',
	  be: 45
	}, {
	  element: 'Os',
	  orbital: '4f7/2',
	  be: 50
	}, {
	  element: 'Ir',
	  orbital: '4f7/2',
	  be: 60
	}, {
	  element: 'Pt',
	  orbital: '4f7/2',
	  be: 70
	}, {
	  element: 'Au',
	  orbital: '4f7/2',
	  be: 83
	}, {
	  element: 'Hg',
	  orbital: '4f7/2',
	  be: 99
	}, {
	  element: 'Tl',
	  orbital: '4f7/2',
	  be: 118
	}, {
	  element: 'Pb',
	  orbital: '4f7/2',
	  be: 138
	}, {
	  element: 'Bi',
	  orbital: '4f7/2',
	  be: 158
	}, {
	  element: 'Po',
	  orbital: '4f7/2',
	  be: 184
	}, {
	  element: 'At',
	  orbital: '4f7/2',
	  be: 210
	}, {
	  element: 'Rn',
	  orbital: '4f7/2',
	  be: 238
	}, {
	  element: 'Fr',
	  orbital: '4f7/2',
	  be: 268
	}, {
	  element: 'Ra',
	  orbital: '4f7/2',
	  be: 299
	}, {
	  element: 'Ac',
	  orbital: '4f7/2',
	  be: 319
	}, {
	  element: 'Th',
	  orbital: '4f7/2',
	  be: 335
	}, {
	  element: 'U',
	  orbital: '4f7/2',
	  be: 380
	}, {
	  element: 'Ce',
	  orbital: '4f',
	  be: 1
	}, {
	  element: 'Pr',
	  orbital: '4f',
	  be: 2
	}, {
	  element: 'Nd',
	  orbital: '4f',
	  be: 2
	}, {
	  element: 'Pm',
	  orbital: '4f',
	  be: 4
	}, {
	  element: 'Sm',
	  orbital: '4f',
	  be: 7
	}, {
	  element: 'Tb',
	  orbital: '4f',
	  be: 3
	}, {
	  element: 'Dy',
	  orbital: '4f',
	  be: 4
	}, {
	  element: 'Ho',
	  orbital: '4f',
	  be: 4
	}, {
	  element: 'Er',
	  orbital: '4f',
	  be: 4
	}, {
	  element: 'Tm',
	  orbital: '4f',
	  be: 5
	}, {
	  element: 'Yb',
	  orbital: '4f',
	  be: 6
	}, {
	  element: 'Lu',
	  orbital: '4f',
	  be: 7
	}, {
	  element: 'Hf',
	  orbital: '4f',
	  be: 18
	}, {
	  element: 'Ta',
	  orbital: '4f',
	  be: 25
	}, {
	  element: 'W',
	  orbital: '4f',
	  be: 34
	}, {
	  element: 'Re',
	  orbital: '4f',
	  be: 45
	}, {
	  element: 'Os',
	  orbital: '4f',
	  be: 50
	}, {
	  element: 'Ir',
	  orbital: '4f',
	  be: 60
	}, {
	  element: 'Pt',
	  orbital: '4f',
	  be: 70
	}, {
	  element: 'Au',
	  orbital: '4f',
	  be: 83
	}, {
	  element: 'Hg',
	  orbital: '4f',
	  be: 99
	}, {
	  element: 'Tl',
	  orbital: '4f',
	  be: 118
	}, {
	  element: 'Pb',
	  orbital: '4f',
	  be: 138
	}, {
	  element: 'Bi',
	  orbital: '4f',
	  be: 158
	}, {
	  element: 'Po',
	  orbital: '4f',
	  be: 184
	}, {
	  element: 'At',
	  orbital: '4f',
	  be: 210
	}, {
	  element: 'Rn',
	  orbital: '4f',
	  be: 238
	}, {
	  element: 'Fr',
	  orbital: '4f',
	  be: 268
	}, {
	  element: 'Ra',
	  orbital: '4f',
	  be: 299
	}, {
	  element: 'Ac',
	  orbital: '4f',
	  be: 319
	}, {
	  element: 'Th',
	  orbital: '4f',
	  be: 335
	}, {
	  element: 'U',
	  orbital: '4f',
	  be: 380
	}, {
	  element: 'Sn',
	  orbital: '5s',
	  be: 1
	}, {
	  element: 'Sb',
	  orbital: '5s',
	  be: 7
	}, {
	  element: 'Te',
	  orbital: '5s',
	  be: 12
	}, {
	  element: 'I',
	  orbital: '5s',
	  be: 14
	}, {
	  element: 'Xe',
	  orbital: '5s',
	  be: 18
	}, {
	  element: 'Cs',
	  orbital: '5s',
	  be: 23
	}, {
	  element: 'Ba',
	  orbital: '5s',
	  be: 40
	}, {
	  element: 'La',
	  orbital: '5s',
	  be: 33
	}, {
	  element: 'Ce',
	  orbital: '5s',
	  be: 38
	}, {
	  element: 'Pr',
	  orbital: '5s',
	  be: 38
	}, {
	  element: 'Nd',
	  orbital: '5s',
	  be: 38
	}, {
	  element: 'Pm',
	  orbital: '5s',
	  be: 38
	}, {
	  element: 'Sm',
	  orbital: '5s',
	  be: 39
	}, {
	  element: 'Eu',
	  orbital: '5s',
	  be: 32
	}, {
	  element: 'Gd',
	  orbital: '5s',
	  be: 36
	}, {
	  element: 'Tb',
	  orbital: '5s',
	  be: 40
	}, {
	  element: 'Dy',
	  orbital: '5s',
	  be: 63
	}, {
	  element: 'Ho',
	  orbital: '5s',
	  be: 51
	}, {
	  element: 'Er',
	  orbital: '5s',
	  be: 60
	}, {
	  element: 'Tm',
	  orbital: '5s',
	  be: 53
	}, {
	  element: 'Yb',
	  orbital: '5s',
	  be: 53
	}, {
	  element: 'Lu',
	  orbital: '5s',
	  be: 57
	}, {
	  element: 'Hf',
	  orbital: '5s',
	  be: 65
	}, {
	  element: 'Ta',
	  orbital: '5s',
	  be: 71
	}, {
	  element: 'W',
	  orbital: '5s',
	  be: 77
	}, {
	  element: 'Re',
	  orbital: '5s',
	  be: 83
	}, {
	  element: 'Os',
	  orbital: '5s',
	  be: 84
	}, {
	  element: 'Ir',
	  orbital: '5s',
	  be: 96
	}, {
	  element: 'Pt',
	  orbital: '5s',
	  be: 102
	}, {
	  element: 'Au',
	  orbital: '5s',
	  be: 108
	}, {
	  element: 'Hg',
	  orbital: '5s',
	  be: 120
	}, {
	  element: 'Tl',
	  orbital: '5s',
	  be: 137
	}, {
	  element: 'Pb',
	  orbital: '5s',
	  be: 148
	}, {
	  element: 'Bi',
	  orbital: '5s',
	  be: 160
	}, {
	  element: 'Po',
	  orbital: '5s',
	  be: 177
	}, {
	  element: 'At',
	  orbital: '5s',
	  be: 195
	}, {
	  element: 'Rn',
	  orbital: '5s',
	  be: 214
	}, {
	  element: 'Fr',
	  orbital: '5s',
	  be: 234
	}, {
	  element: 'Ra',
	  orbital: '5s',
	  be: 254
	}, {
	  element: 'Ac',
	  orbital: '5s',
	  be: 272
	}, {
	  element: 'Th',
	  orbital: '5s',
	  be: 290
	}, {
	  element: 'In',
	  orbital: '5p1/2',
	  be: 1
	}, {
	  element: 'Sn',
	  orbital: '5p1/2',
	  be: 1
	}, {
	  element: 'Sb',
	  orbital: '5p1/2',
	  be: 2
	}, {
	  element: 'Te',
	  orbital: '5p1/2',
	  be: 2
	}, {
	  element: 'I',
	  orbital: '5p1/2',
	  be: 3
	}, {
	  element: 'Xe',
	  orbital: '5p1/2',
	  be: 7
	}, {
	  element: 'Cs',
	  orbital: '5p1/2',
	  be: 13
	}, {
	  element: 'Ba',
	  orbital: '5p1/2',
	  be: 17
	}, {
	  element: 'La',
	  orbital: '5p1/2',
	  be: 15
	}, {
	  element: 'Ce',
	  orbital: '5p1/2',
	  be: 20
	}, {
	  element: 'Pr',
	  orbital: '5p1/2',
	  be: 23
	}, {
	  element: 'Nd',
	  orbital: '5p1/2',
	  be: 22
	}, {
	  element: 'Pm',
	  orbital: '5p1/2',
	  be: 22
	}, {
	  element: 'Sm',
	  orbital: '5p1/2',
	  be: 22
	}, {
	  element: 'Eu',
	  orbital: '5p1/2',
	  be: 22
	}, {
	  element: 'Gd',
	  orbital: '5p1/2',
	  be: 21
	}, {
	  element: 'Tb',
	  orbital: '5p1/2',
	  be: 26
	}, {
	  element: 'Dy',
	  orbital: '5p1/2',
	  be: 26
	}, {
	  element: 'Ho',
	  orbital: '5p1/2',
	  be: 20
	}, {
	  element: 'Er',
	  orbital: '5p1/2',
	  be: 29
	}, {
	  element: 'Tm',
	  orbital: '5p1/2',
	  be: 32
	}, {
	  element: 'Yb',
	  orbital: '5p1/2',
	  be: 23
	}, {
	  element: 'Lu',
	  orbital: '5p1/2',
	  be: 28
	}, {
	  element: 'Hf',
	  orbital: '5p1/2',
	  be: 38
	}, {
	  element: 'Ta',
	  orbital: '5p1/2',
	  be: 45
	}, {
	  element: 'W',
	  orbital: '5p1/2',
	  be: 47
	}, {
	  element: 'Re',
	  orbital: '5p1/2',
	  be: 46
	}, {
	  element: 'Os',
	  orbital: '5p1/2',
	  be: 58
	}, {
	  element: 'Ir',
	  orbital: '5p1/2',
	  be: 63
	}, {
	  element: 'Pt',
	  orbital: '5p1/2',
	  be: 66
	}, {
	  element: 'Au',
	  orbital: '5p1/2',
	  be: 72
	}, {
	  element: 'Hg',
	  orbital: '5p1/2',
	  be: 81
	}, {
	  element: 'Tl',
	  orbital: '5p1/2',
	  be: 100
	}, {
	  element: 'Pb',
	  orbital: '5p1/2',
	  be: 105
	}, {
	  element: 'Bi',
	  orbital: '5p1/2',
	  be: 117
	}, {
	  element: 'Po',
	  orbital: '5p1/2',
	  be: 132
	}, {
	  element: 'At',
	  orbital: '5p1/2',
	  be: 148
	}, {
	  element: 'Rn',
	  orbital: '5p1/2',
	  be: 164
	}, {
	  element: 'Fr',
	  orbital: '5p1/2',
	  be: 182
	}, {
	  element: 'Ra',
	  orbital: '5p1/2',
	  be: 200
	}, {
	  element: 'Ac',
	  orbital: '5p1/2',
	  be: 215
	}, {
	  element: 'Th',
	  orbital: '5p1/2',
	  be: 229
	}, {
	  element: 'In',
	  orbital: '5p3/2',
	  be: 1
	}, {
	  element: 'Sn',
	  orbital: '5p3/2',
	  be: 1
	}, {
	  element: 'Sb',
	  orbital: '5p3/2',
	  be: 2
	}, {
	  element: 'Te',
	  orbital: '5p3/2',
	  be: 2
	}, {
	  element: 'I',
	  orbital: '5p3/2',
	  be: 3
	}, {
	  element: 'Xe',
	  orbital: '5p3/2',
	  be: 7
	}, {
	  element: 'Cs',
	  orbital: '5p3/2',
	  be: 12
	}, {
	  element: 'Ba',
	  orbital: '5p3/2',
	  be: 15
	}, {
	  element: 'La',
	  orbital: '5p3/2',
	  be: 15
	}, {
	  element: 'Ce',
	  orbital: '5p3/2',
	  be: 20
	}, {
	  element: 'Pr',
	  orbital: '5p3/2',
	  be: 23
	}, {
	  element: 'Nd',
	  orbital: '5p3/2',
	  be: 22
	}, {
	  element: 'Pm',
	  orbital: '5p3/2',
	  be: 22
	}, {
	  element: 'Sm',
	  orbital: '5p3/2',
	  be: 22
	}, {
	  element: 'Eu',
	  orbital: '5p3/2',
	  be: 22
	}, {
	  element: 'Gd',
	  orbital: '5p3/2',
	  be: 21
	}, {
	  element: 'Tb',
	  orbital: '5p3/2',
	  be: 26
	}, {
	  element: 'Dy',
	  orbital: '5p3/2',
	  be: 26
	}, {
	  element: 'Ho',
	  orbital: '5p3/2',
	  be: 20
	}, {
	  element: 'Er',
	  orbital: '5p3/2',
	  be: 29
	}, {
	  element: 'Tm',
	  orbital: '5p3/2',
	  be: 32
	}, {
	  element: 'Yb',
	  orbital: '5p3/2',
	  be: 23
	}, {
	  element: 'Lu',
	  orbital: '5p3/2',
	  be: 28
	}, {
	  element: 'Hf',
	  orbital: '5p3/2',
	  be: 31
	}, {
	  element: 'Ta',
	  orbital: '5p3/2',
	  be: 37
	}, {
	  element: 'W',
	  orbital: '5p3/2',
	  be: 37
	}, {
	  element: 'Re',
	  orbital: '5p3/2',
	  be: 35
	}, {
	  element: 'Os',
	  orbital: '5p3/2',
	  be: 46
	}, {
	  element: 'Ir',
	  orbital: '5p3/2',
	  be: 51
	}, {
	  element: 'Pt',
	  orbital: '5p3/2',
	  be: 51
	}, {
	  element: 'Au',
	  orbital: '5p3/2',
	  be: 54
	}, {
	  element: 'Hg',
	  orbital: '5p3/2',
	  be: 58
	}, {
	  element: 'Tl',
	  orbital: '5p3/2',
	  be: 76
	}, {
	  element: 'Pb',
	  orbital: '5p3/2',
	  be: 86
	}, {
	  element: 'Bi',
	  orbital: '5p3/2',
	  be: 93
	}, {
	  element: 'Po',
	  orbital: '5p3/2',
	  be: 104
	}, {
	  element: 'At',
	  orbital: '5p3/2',
	  be: 115
	}, {
	  element: 'Rn',
	  orbital: '5p3/2',
	  be: 127
	}, {
	  element: 'Fr',
	  orbital: '5p3/2',
	  be: 140
	}, {
	  element: 'Ra',
	  orbital: '5p3/2',
	  be: 153
	}, {
	  element: 'Ac',
	  orbital: '5p3/2',
	  be: 167
	}, {
	  element: 'Th',
	  orbital: '5p3/2',
	  be: 182
	}, {
	  element: 'In',
	  orbital: '5p',
	  be: 1
	}, {
	  element: 'Sn',
	  orbital: '5p',
	  be: 1
	}, {
	  element: 'Sb',
	  orbital: '5p',
	  be: 2
	}, {
	  element: 'Te',
	  orbital: '5p',
	  be: 2
	}, {
	  element: 'I',
	  orbital: '5p',
	  be: 3
	}, {
	  element: 'Xe',
	  orbital: '5p',
	  be: 7
	}, {
	  element: 'Cs',
	  orbital: '5p',
	  be: 12
	}, {
	  element: 'Ba',
	  orbital: '5p',
	  be: 15
	}, {
	  element: 'La',
	  orbital: '5p',
	  be: 15
	}, {
	  element: 'Ce',
	  orbital: '5p',
	  be: 20
	}, {
	  element: 'Pr',
	  orbital: '5p',
	  be: 23
	}, {
	  element: 'Nd',
	  orbital: '5p',
	  be: 22
	}, {
	  element: 'Pm',
	  orbital: '5p',
	  be: 22
	}, {
	  element: 'Sm',
	  orbital: '5p',
	  be: 22
	}, {
	  element: 'Eu',
	  orbital: '5p',
	  be: 22
	}, {
	  element: 'Gd',
	  orbital: '5p',
	  be: 21
	}, {
	  element: 'Tb',
	  orbital: '5p',
	  be: 26
	}, {
	  element: 'Dy',
	  orbital: '5p',
	  be: 26
	}, {
	  element: 'Ho',
	  orbital: '5p',
	  be: 20
	}, {
	  element: 'Er',
	  orbital: '5p',
	  be: 29
	}, {
	  element: 'Tm',
	  orbital: '5p',
	  be: 32
	}, {
	  element: 'Yb',
	  orbital: '5p',
	  be: 23
	}, {
	  element: 'Lu',
	  orbital: '5p',
	  be: 28
	}, {
	  element: 'Hf',
	  orbital: '5p',
	  be: 31
	}, {
	  element: 'Ta',
	  orbital: '5p',
	  be: 37
	}, {
	  element: 'W',
	  orbital: '5p',
	  be: 37
	}, {
	  element: 'Re',
	  orbital: '5p',
	  be: 35
	}, {
	  element: 'Os',
	  orbital: '5p',
	  be: 46
	}, {
	  element: 'Ir',
	  orbital: '5p',
	  be: 51
	}, {
	  element: 'Pt',
	  orbital: '5p',
	  be: 51
	}, {
	  element: 'Au',
	  orbital: '5p',
	  be: 54
	}, {
	  element: 'Hg',
	  orbital: '5p',
	  be: 58
	}, {
	  element: 'Tl',
	  orbital: '5p',
	  be: 76
	}, {
	  element: 'Pb',
	  orbital: '5p',
	  be: 86
	}, {
	  element: 'Bi',
	  orbital: '5p',
	  be: 93
	}, {
	  element: 'Po',
	  orbital: '5p',
	  be: 104
	}, {
	  element: 'At',
	  orbital: '5p',
	  be: 115
	}, {
	  element: 'Rn',
	  orbital: '5p',
	  be: 127
	}, {
	  element: 'Fr',
	  orbital: '5p',
	  be: 140
	}, {
	  element: 'Ra',
	  orbital: '5p',
	  be: 153
	}, {
	  element: 'Ac',
	  orbital: '5p',
	  be: 167
	}, {
	  element: 'Th',
	  orbital: '5p',
	  be: 182
	}, {
	  element: 'Lu',
	  orbital: '5d3/2',
	  be: 5
	}, {
	  element: 'Hf',
	  orbital: '5d3/2',
	  be: 7
	}, {
	  element: 'Ta',
	  orbital: '5d3/2',
	  be: 6
	}, {
	  element: 'W',
	  orbital: '5d3/2',
	  be: 6
	}, {
	  element: 'Re',
	  orbital: '5d3/2',
	  be: 4
	}, {
	  element: 'Ir',
	  orbital: '5d3/2',
	  be: 4
	}, {
	  element: 'Pt',
	  orbital: '5d3/2',
	  be: 2
	}, {
	  element: 'Au',
	  orbital: '5d3/2',
	  be: 3
	}, {
	  element: 'Hg',
	  orbital: '5d3/2',
	  be: 7
	}, {
	  element: 'Tl',
	  orbital: '5d3/2',
	  be: 16
	}, {
	  element: 'Pb',
	  orbital: '5d3/2',
	  be: 22
	}, {
	  element: 'Bi',
	  orbital: '5d3/2',
	  be: 27
	}, {
	  element: 'Po',
	  orbital: '5d3/2',
	  be: 31
	}, {
	  element: 'At',
	  orbital: '5d3/2',
	  be: 40
	}, {
	  element: 'Rn',
	  orbital: '5d3/2',
	  be: 48
	}, {
	  element: 'Fr',
	  orbital: '5d3/2',
	  be: 58
	}, {
	  element: 'Ra',
	  orbital: '5d3/2',
	  be: 68
	}, {
	  element: 'Ac',
	  orbital: '5d3/2',
	  be: 80
	}, {
	  element: 'Th',
	  orbital: '5d3/2',
	  be: 95
	}, {
	  element: 'U',
	  orbital: '5d3/2',
	  be: 104
	}, {
	  element: 'Lu',
	  orbital: '5d5/2',
	  be: 5
	}, {
	  element: 'Hf',
	  orbital: '5d5/2',
	  be: 7
	}, {
	  element: 'Ta',
	  orbital: '5d5/2',
	  be: 6
	}, {
	  element: 'W',
	  orbital: '5d5/2',
	  be: 6
	}, {
	  element: 'Re',
	  orbital: '5d5/2',
	  be: 4
	}, {
	  element: 'Ir',
	  orbital: '5d5/2',
	  be: 4
	}, {
	  element: 'Pt',
	  orbital: '5d5/2',
	  be: 2
	}, {
	  element: 'Au',
	  orbital: '5d5/2',
	  be: 3
	}, {
	  element: 'Hg',
	  orbital: '5d5/2',
	  be: 7
	}, {
	  element: 'Tl',
	  orbital: '5d5/2',
	  be: 13
	}, {
	  element: 'Pb',
	  orbital: '5d5/2',
	  be: 20
	}, {
	  element: 'Bi',
	  orbital: '5d5/2',
	  be: 25
	}, {
	  element: 'Po',
	  orbital: '5d5/2',
	  be: 31
	}, {
	  element: 'At',
	  orbital: '5d5/2',
	  be: 40
	}, {
	  element: 'Rn',
	  orbital: '5d5/2',
	  be: 48
	}, {
	  element: 'Fr',
	  orbital: '5d5/2',
	  be: 58
	}, {
	  element: 'Ra',
	  orbital: '5d5/2',
	  be: 68
	}, {
	  element: 'Ac',
	  orbital: '5d5/2',
	  be: 80
	}, {
	  element: 'Th',
	  orbital: '5d5/2',
	  be: 88
	}, {
	  element: 'U',
	  orbital: '5d5/2',
	  be: 96
	}, {
	  element: 'Lu',
	  orbital: '5d',
	  be: 5
	}, {
	  element: 'Hf',
	  orbital: '5d',
	  be: 7
	}, {
	  element: 'Ta',
	  orbital: '5d',
	  be: 6
	}, {
	  element: 'W',
	  orbital: '5d',
	  be: 6
	}, {
	  element: 'Re',
	  orbital: '5d',
	  be: 4
	}, {
	  element: 'Ir',
	  orbital: '5d',
	  be: 4
	}, {
	  element: 'Pt',
	  orbital: '5d',
	  be: 2
	}, {
	  element: 'Au',
	  orbital: '5d',
	  be: 3
	}, {
	  element: 'Hg',
	  orbital: '5d',
	  be: 7
	}, {
	  element: 'Tl',
	  orbital: '5d',
	  be: 13
	}, {
	  element: 'Pb',
	  orbital: '5d',
	  be: 20
	}, {
	  element: 'Bi',
	  orbital: '5d',
	  be: 25
	}, {
	  element: 'Po',
	  orbital: '5d',
	  be: 31
	}, {
	  element: 'At',
	  orbital: '5d',
	  be: 40
	}, {
	  element: 'Rn',
	  orbital: '5d',
	  be: 48
	}, {
	  element: 'Fr',
	  orbital: '5d',
	  be: 58
	}, {
	  element: 'Ra',
	  orbital: '5d',
	  be: 68
	}, {
	  element: 'Ac',
	  orbital: '5d',
	  be: 80
	}, {
	  element: 'Th',
	  orbital: '5d',
	  be: 88
	}, {
	  element: 'U',
	  orbital: '5d',
	  be: 96
	}, {
	  element: 'Pb',
	  orbital: '6s',
	  be: 3
	}, {
	  element: 'Bi',
	  orbital: '6s',
	  be: 8
	}, {
	  element: 'Po',
	  orbital: '6s',
	  be: 12
	}, {
	  element: 'At',
	  orbital: '6s',
	  be: 18
	}, {
	  element: 'Rn',
	  orbital: '6s',
	  be: 26
	}, {
	  element: 'Fr',
	  orbital: '6s',
	  be: 34
	}, {
	  element: 'Ra',
	  orbital: '6s',
	  be: 44
	}, {
	  element: 'Th',
	  orbital: '6s',
	  be: 60
	}, {
	  element: 'Pb',
	  orbital: '6p1/2',
	  be: 1
	}, {
	  element: 'Bi',
	  orbital: '6p1/2',
	  be: 3
	}, {
	  element: 'Po',
	  orbital: '6p1/2',
	  be: 5
	}, {
	  element: 'At',
	  orbital: '6p1/2',
	  be: 8
	}, {
	  element: 'Rn',
	  orbital: '6p1/2',
	  be: 11
	}, {
	  element: 'Fr',
	  orbital: '6p1/2',
	  be: 15
	}, {
	  element: 'Ra',
	  orbital: '6p1/2',
	  be: 19
	}, {
	  element: 'Th',
	  orbital: '6p1/2',
	  be: 49
	}, {
	  element: 'U',
	  orbital: '6p1/2',
	  be: 28.9
	}, {
	  element: 'Pb',
	  orbital: '6p3/2',
	  be: 1
	}, {
	  element: 'Bi',
	  orbital: '6p3/2',
	  be: 3
	}, {
	  element: 'Po',
	  orbital: '6p3/2',
	  be: 5
	}, {
	  element: 'At',
	  orbital: '6p3/2',
	  be: 8
	}, {
	  element: 'Rn',
	  orbital: '6p3/2',
	  be: 11
	}, {
	  element: 'Fr',
	  orbital: '6p3/2',
	  be: 15
	}, {
	  element: 'Ra',
	  orbital: '6p3/2',
	  be: 19
	}, {
	  element: 'Th',
	  orbital: '6p3/2',
	  be: 43
	}, {
	  element: 'U',
	  orbital: '6p3/2',
	  be: 19
	}, {
	  element: 'Pb',
	  orbital: '6p',
	  be: 1
	}, {
	  element: 'Bi',
	  orbital: '6p',
	  be: 3
	}, {
	  element: 'Po',
	  orbital: '6p',
	  be: 5
	}, {
	  element: 'At',
	  orbital: '6p',
	  be: 8
	}, {
	  element: 'Rn',
	  orbital: '6p',
	  be: 11
	}, {
	  element: 'Fr',
	  orbital: '6p',
	  be: 15
	}, {
	  element: 'Ra',
	  orbital: '6p',
	  be: 19
	}, {
	  element: 'Th',
	  orbital: '6p',
	  be: 43
	}, {
	  element: 'U',
	  orbital: '6p',
	  be: 19
	}];

	const JSGraph = {
	  ...JSGraph$1,
	  getAnnotations
	};

	exports.AnalysesManager = AnalysesManager;
	exports.Analysis = Analysis;
	exports.JSGraph = JSGraph;
	exports.fromJcamp = fromJcamp;
	exports.fromVamas = fromVamas;
	exports.peakPicking = peakPicking;
	exports.references = references;
	exports.toJcamp = toJcamp;
	exports.toJcamps = toJcamps;

	Object.defineProperty(exports, '__esModule', { value: true });

}));
//# sourceMappingURL=xps-analysis.js.map
