/**
 * legoino-util - Create and parse compact logs.
 * @version v2.0.1
 * @link https://github.com/Hackuarium/legoino-util#readme
 * @license MIT
 */
(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory() :
  typeof define === 'function' && define.amd ? define(factory) :
  (global = typeof globalThis !== 'undefined' ? globalThis : global || self, global.legoinoUtil = factory());
})(this, (function () { 'use strict';

  var browser = {exports: {}};

  /**
   * Helpers.
   */
  var s = 1000;
  var m = s * 60;
  var h = m * 60;
  var d = h * 24;
  var w = d * 7;
  var y = d * 365.25;
  /**
   * Parse or format the given `val`.
   *
   * Options:
   *
   *  - `long` verbose formatting [false]
   *
   * @param {String|Number} val
   * @param {Object} [options]
   * @throws {Error} throw an error if val is not a non-empty string or a number
   * @return {String|Number}
   * @api public
   */

  var ms = function (val, options) {
    options = options || {};
    var type = typeof val;

    if (type === 'string' && val.length > 0) {
      return parse(val);
    } else if (type === 'number' && isFinite(val)) {
      return options.long ? fmtLong(val) : fmtShort(val);
    }

    throw new Error('val is not a non-empty string or a valid number. val=' + JSON.stringify(val));
  };
  /**
   * Parse the given `str` and return milliseconds.
   *
   * @param {String} str
   * @return {Number}
   * @api private
   */


  function parse(str) {
    str = String(str);

    if (str.length > 100) {
      return;
    }

    var match = /^(-?(?:\d+)?\.?\d+) *(milliseconds?|msecs?|ms|seconds?|secs?|s|minutes?|mins?|m|hours?|hrs?|h|days?|d|weeks?|w|years?|yrs?|y)?$/i.exec(str);

    if (!match) {
      return;
    }

    var n = parseFloat(match[1]);
    var type = (match[2] || 'ms').toLowerCase();

    switch (type) {
      case 'years':
      case 'year':
      case 'yrs':
      case 'yr':
      case 'y':
        return n * y;

      case 'weeks':
      case 'week':
      case 'w':
        return n * w;

      case 'days':
      case 'day':
      case 'd':
        return n * d;

      case 'hours':
      case 'hour':
      case 'hrs':
      case 'hr':
      case 'h':
        return n * h;

      case 'minutes':
      case 'minute':
      case 'mins':
      case 'min':
      case 'm':
        return n * m;

      case 'seconds':
      case 'second':
      case 'secs':
      case 'sec':
      case 's':
        return n * s;

      case 'milliseconds':
      case 'millisecond':
      case 'msecs':
      case 'msec':
      case 'ms':
        return n;

      default:
        return undefined;
    }
  }
  /**
   * Short format for `ms`.
   *
   * @param {Number} ms
   * @return {String}
   * @api private
   */


  function fmtShort(ms) {
    var msAbs = Math.abs(ms);

    if (msAbs >= d) {
      return Math.round(ms / d) + 'd';
    }

    if (msAbs >= h) {
      return Math.round(ms / h) + 'h';
    }

    if (msAbs >= m) {
      return Math.round(ms / m) + 'm';
    }

    if (msAbs >= s) {
      return Math.round(ms / s) + 's';
    }

    return ms + 'ms';
  }
  /**
   * Long format for `ms`.
   *
   * @param {Number} ms
   * @return {String}
   * @api private
   */


  function fmtLong(ms) {
    var msAbs = Math.abs(ms);

    if (msAbs >= d) {
      return plural(ms, msAbs, d, 'day');
    }

    if (msAbs >= h) {
      return plural(ms, msAbs, h, 'hour');
    }

    if (msAbs >= m) {
      return plural(ms, msAbs, m, 'minute');
    }

    if (msAbs >= s) {
      return plural(ms, msAbs, s, 'second');
    }

    return ms + ' ms';
  }
  /**
   * Pluralization helper.
   */


  function plural(ms, msAbs, n, name) {
    var isPlural = msAbs >= n * 1.5;
    return Math.round(ms / n) + ' ' + name + (isPlural ? 's' : '');
  }

  /**
   * This is the common logic for both the Node.js and web browser
   * implementations of `debug()`.
   */

  function setup(env) {
    createDebug.debug = createDebug;
    createDebug.default = createDebug;
    createDebug.coerce = coerce;
    createDebug.disable = disable;
    createDebug.enable = enable;
    createDebug.enabled = enabled;
    createDebug.humanize = ms;
    createDebug.destroy = destroy;
    Object.keys(env).forEach(key => {
      createDebug[key] = env[key];
    });
    /**
    * The currently active debug mode names, and names to skip.
    */

    createDebug.names = [];
    createDebug.skips = [];
    /**
    * Map of special "%n" handling functions, for the debug "format" argument.
    *
    * Valid key names are a single, lower or upper-case letter, i.e. "n" and "N".
    */

    createDebug.formatters = {};
    /**
    * Selects a color for a debug namespace
    * @param {String} namespace The namespace string for the debug instance to be colored
    * @return {Number|String} An ANSI color code for the given namespace
    * @api private
    */

    function selectColor(namespace) {
      let hash = 0;

      for (let i = 0; i < namespace.length; i++) {
        hash = (hash << 5) - hash + namespace.charCodeAt(i);
        hash |= 0; // Convert to 32bit integer
      }

      return createDebug.colors[Math.abs(hash) % createDebug.colors.length];
    }

    createDebug.selectColor = selectColor;
    /**
    * Create a debugger with the given `namespace`.
    *
    * @param {String} namespace
    * @return {Function}
    * @api public
    */

    function createDebug(namespace) {
      let prevTime;
      let enableOverride = null;
      let namespacesCache;
      let enabledCache;

      function debug() {
        for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
          args[_key] = arguments[_key];
        }

        // Disabled?
        if (!debug.enabled) {
          return;
        }

        const self = debug; // Set `diff` timestamp

        const curr = Number(new Date());
        const ms = curr - (prevTime || curr);
        self.diff = ms;
        self.prev = prevTime;
        self.curr = curr;
        prevTime = curr;
        args[0] = createDebug.coerce(args[0]);

        if (typeof args[0] !== 'string') {
          // Anything else let's inspect with %O
          args.unshift('%O');
        } // Apply any `formatters` transformations


        let index = 0;
        args[0] = args[0].replace(/%([a-zA-Z%])/g, (match, format) => {
          // If we encounter an escaped % then don't increase the array index
          if (match === '%%') {
            return '%';
          }

          index++;
          const formatter = createDebug.formatters[format];

          if (typeof formatter === 'function') {
            const val = args[index];
            match = formatter.call(self, val); // Now we need to remove `args[index]` since it's inlined in the `format`

            args.splice(index, 1);
            index--;
          }

          return match;
        }); // Apply env-specific formatting (colors, etc.)

        createDebug.formatArgs.call(self, args);
        const logFn = self.log || createDebug.log;
        logFn.apply(self, args);
      }

      debug.namespace = namespace;
      debug.useColors = createDebug.useColors();
      debug.color = createDebug.selectColor(namespace);
      debug.extend = extend;
      debug.destroy = createDebug.destroy; // XXX Temporary. Will be removed in the next major release.

      Object.defineProperty(debug, 'enabled', {
        enumerable: true,
        configurable: false,
        get: () => {
          if (enableOverride !== null) {
            return enableOverride;
          }

          if (namespacesCache !== createDebug.namespaces) {
            namespacesCache = createDebug.namespaces;
            enabledCache = createDebug.enabled(namespace);
          }

          return enabledCache;
        },
        set: v => {
          enableOverride = v;
        }
      }); // Env-specific initialization logic for debug instances

      if (typeof createDebug.init === 'function') {
        createDebug.init(debug);
      }

      return debug;
    }

    function extend(namespace, delimiter) {
      const newDebug = createDebug(this.namespace + (typeof delimiter === 'undefined' ? ':' : delimiter) + namespace);
      newDebug.log = this.log;
      return newDebug;
    }
    /**
    * Enables a debug mode by namespaces. This can include modes
    * separated by a colon and wildcards.
    *
    * @param {String} namespaces
    * @api public
    */


    function enable(namespaces) {
      createDebug.save(namespaces);
      createDebug.namespaces = namespaces;
      createDebug.names = [];
      createDebug.skips = [];
      let i;
      const split = (typeof namespaces === 'string' ? namespaces : '').split(/[\s,]+/);
      const len = split.length;

      for (i = 0; i < len; i++) {
        if (!split[i]) {
          // ignore empty strings
          continue;
        }

        namespaces = split[i].replace(/\*/g, '.*?');

        if (namespaces[0] === '-') {
          createDebug.skips.push(new RegExp('^' + namespaces.slice(1) + '$'));
        } else {
          createDebug.names.push(new RegExp('^' + namespaces + '$'));
        }
      }
    }
    /**
    * Disable debug output.
    *
    * @return {String} namespaces
    * @api public
    */


    function disable() {
      const namespaces = [...createDebug.names.map(toNamespace), ...createDebug.skips.map(toNamespace).map(namespace => '-' + namespace)].join(',');
      createDebug.enable('');
      return namespaces;
    }
    /**
    * Returns true if the given mode name is enabled, false otherwise.
    *
    * @param {String} name
    * @return {Boolean}
    * @api public
    */


    function enabled(name) {
      if (name[name.length - 1] === '*') {
        return true;
      }

      let i;
      let len;

      for (i = 0, len = createDebug.skips.length; i < len; i++) {
        if (createDebug.skips[i].test(name)) {
          return false;
        }
      }

      for (i = 0, len = createDebug.names.length; i < len; i++) {
        if (createDebug.names[i].test(name)) {
          return true;
        }
      }

      return false;
    }
    /**
    * Convert regexp to namespace
    *
    * @param {RegExp} regxep
    * @return {String} namespace
    * @api private
    */


    function toNamespace(regexp) {
      return regexp.toString().substring(2, regexp.toString().length - 2).replace(/\.\*\?$/, '*');
    }
    /**
    * Coerce `val`.
    *
    * @param {Mixed} val
    * @return {Mixed}
    * @api private
    */


    function coerce(val) {
      if (val instanceof Error) {
        return val.stack || val.message;
      }

      return val;
    }
    /**
    * XXX DO NOT USE. This is a temporary stub function.
    * XXX It WILL be removed in the next major release.
    */


    function destroy() {
      console.warn('Instance method `debug.destroy()` is deprecated and no longer does anything. It will be removed in the next major version of `debug`.');
    }

    createDebug.enable(createDebug.load());
    return createDebug;
  }

  var common = setup;

  /* eslint-env browser */

  (function (module, exports) {
    /**
     * This is the web browser implementation of `debug()`.
     */
    exports.formatArgs = formatArgs;
    exports.save = save;
    exports.load = load;
    exports.useColors = useColors;
    exports.storage = localstorage();

    exports.destroy = (() => {
      let warned = false;
      return () => {
        if (!warned) {
          warned = true;
          console.warn('Instance method `debug.destroy()` is deprecated and no longer does anything. It will be removed in the next major version of `debug`.');
        }
      };
    })();
    /**
     * Colors.
     */


    exports.colors = ['#0000CC', '#0000FF', '#0033CC', '#0033FF', '#0066CC', '#0066FF', '#0099CC', '#0099FF', '#00CC00', '#00CC33', '#00CC66', '#00CC99', '#00CCCC', '#00CCFF', '#3300CC', '#3300FF', '#3333CC', '#3333FF', '#3366CC', '#3366FF', '#3399CC', '#3399FF', '#33CC00', '#33CC33', '#33CC66', '#33CC99', '#33CCCC', '#33CCFF', '#6600CC', '#6600FF', '#6633CC', '#6633FF', '#66CC00', '#66CC33', '#9900CC', '#9900FF', '#9933CC', '#9933FF', '#99CC00', '#99CC33', '#CC0000', '#CC0033', '#CC0066', '#CC0099', '#CC00CC', '#CC00FF', '#CC3300', '#CC3333', '#CC3366', '#CC3399', '#CC33CC', '#CC33FF', '#CC6600', '#CC6633', '#CC9900', '#CC9933', '#CCCC00', '#CCCC33', '#FF0000', '#FF0033', '#FF0066', '#FF0099', '#FF00CC', '#FF00FF', '#FF3300', '#FF3333', '#FF3366', '#FF3399', '#FF33CC', '#FF33FF', '#FF6600', '#FF6633', '#FF9900', '#FF9933', '#FFCC00', '#FFCC33'];
    /**
     * Currently only WebKit-based Web Inspectors, Firefox >= v31,
     * and the Firebug extension (any Firefox version) are known
     * to support "%c" CSS customizations.
     *
     * TODO: add a `localStorage` variable to explicitly enable/disable colors
     */
    // eslint-disable-next-line complexity

    function useColors() {
      // NB: In an Electron preload script, document will be defined but not fully
      // initialized. Since we know we're in Chrome, we'll just detect this case
      // explicitly
      if (typeof window !== 'undefined' && window.process && (window.process.type === 'renderer' || window.process.__nwjs)) {
        return true;
      } // Internet Explorer and Edge do not support colors.


      if (typeof navigator !== 'undefined' && navigator.userAgent && navigator.userAgent.toLowerCase().match(/(edge|trident)\/(\d+)/)) {
        return false;
      } // Is webkit? http://stackoverflow.com/a/16459606/376773
      // document is undefined in react-native: https://github.com/facebook/react-native/pull/1632


      return typeof document !== 'undefined' && document.documentElement && document.documentElement.style && document.documentElement.style.WebkitAppearance || // Is firebug? http://stackoverflow.com/a/398120/376773
      typeof window !== 'undefined' && window.console && (window.console.firebug || window.console.exception && window.console.table) || // Is firefox >= v31?
      // https://developer.mozilla.org/en-US/docs/Tools/Web_Console#Styling_messages
      typeof navigator !== 'undefined' && navigator.userAgent && navigator.userAgent.toLowerCase().match(/firefox\/(\d+)/) && parseInt(RegExp.$1, 10) >= 31 || // Double check webkit in userAgent just in case we are in a worker
      typeof navigator !== 'undefined' && navigator.userAgent && navigator.userAgent.toLowerCase().match(/applewebkit\/(\d+)/);
    }
    /**
     * Colorize log arguments if enabled.
     *
     * @api public
     */


    function formatArgs(args) {
      args[0] = (this.useColors ? '%c' : '') + this.namespace + (this.useColors ? ' %c' : ' ') + args[0] + (this.useColors ? '%c ' : ' ') + '+' + module.exports.humanize(this.diff);

      if (!this.useColors) {
        return;
      }

      const c = 'color: ' + this.color;
      args.splice(1, 0, c, 'color: inherit'); // The final "%c" is somewhat tricky, because there could be other
      // arguments passed either before or after the %c, so we need to
      // figure out the correct index to insert the CSS into

      let index = 0;
      let lastC = 0;
      args[0].replace(/%[a-zA-Z%]/g, match => {
        if (match === '%%') {
          return;
        }

        index++;

        if (match === '%c') {
          // We only are interested in the *last* %c
          // (the user may have provided their own)
          lastC = index;
        }
      });
      args.splice(lastC, 0, c);
    }
    /**
     * Invokes `console.debug()` when available.
     * No-op when `console.debug` is not a "function".
     * If `console.debug` is not available, falls back
     * to `console.log`.
     *
     * @api public
     */


    exports.log = console.debug || console.log || (() => {});
    /**
     * Save `namespaces`.
     *
     * @param {String} namespaces
     * @api private
     */


    function save(namespaces) {
      try {
        if (namespaces) {
          exports.storage.setItem('debug', namespaces);
        } else {
          exports.storage.removeItem('debug');
        }
      } catch (error) {// Swallow
        // XXX (@Qix-) should we be logging these?
      }
    }
    /**
     * Load `namespaces`.
     *
     * @return {String} returns the previously persisted debug modes
     * @api private
     */


    function load() {
      let r;

      try {
        r = exports.storage.getItem('debug');
      } catch (error) {// Swallow
        // XXX (@Qix-) should we be logging these?
      } // If debug isn't set in LS, and we're in Electron, try to load $DEBUG


      if (!r && typeof process !== 'undefined' && 'env' in process) {
        r = process.env.DEBUG;
      }

      return r;
    }
    /**
     * Localstorage attempts to return the localstorage.
     *
     * This is necessary because safari throws
     * when a user disables cookies/localstorage
     * and you attempt to access it.
     *
     * @return {LocalStorage}
     * @api private
     */


    function localstorage() {
      try {
        // TVMLKit (Apple TV JS Runtime) does not have a window object, just localStorage in the global context
        // The Browser also has localStorage in the global context.
        return localStorage;
      } catch (error) {// Swallow
        // XXX (@Qix-) should we be logging these?
      }
    }

    module.exports = common(exports);
    const {
      formatters
    } = module.exports;
    /**
     * Map %j to `JSON.stringify()`, since no Web Inspectors do that by default.
     */

    formatters.j = function (v) {
      try {
        return JSON.stringify(v);
      } catch (error) {
        return '[UnexpectedJSONParseError]: ' + error.message;
      }
    };
  })(browser, browser.exports);

  var calculateCheckDigit$4 = function calculateCheckDigit(hexString) {
    let checkDigit = 0;

    for (let i = 0; i < hexString.length; i = i + 2) {
      checkDigit ^= parseInt(`${hexString[i]}${hexString[i + 1]}`, 16);
    } // console.log(checkDigit.toString(16));


    return checkDigit;
  };

  const calculateCheckDigit$3 = calculateCheckDigit$4;

  var checkCheckDigit$2 = function checkCheckDigit(line) {
    if (calculateCheckDigit$3(line) === 0) return true;
    return false;
  };

  var hexToInt16$3 = function hexToInt16(hexa) {
    let value = parseInt(`${hexa}`, 16);

    if (value > 32767) {
      return (65536 - value) * -1;
    }

    return value;
  };

  /**
   * add possibly missing parameters in the array
   * @param {*} device
   * @returns
   */


  function ensureAllParameters$1(device) {
    const existingParameters = device.parameters;
    let currentPosition = 0;
    const parameters = [];

    for (let parameter of existingParameters) {
      if (!parameter.label) {
        throw new Error(`missing label for ${JSON.stringify(parameter)}`);
      }

      const labelPosition = labelToNumber$2(parameter.label);

      if (labelPosition < currentPosition) {
        throw new Error(`expectedPosition > currentPosition for ${JSON.stringify(parameter)}`);
      }

      while (labelPosition > currentPosition) {
        parameters.push(undefined);
        currentPosition++;
      }

      parameters.push(parameter);
      currentPosition++;
    }

    device.parameters = parameters;
    return device;
  }

  function labelToNumber$2(code) {
    let value = 0;

    for (let char of code) {
      value *= 26;
      value += char.charCodeAt(0) - 64;
    }

    return value - 1;
  }

  var ensureAllParameters_1 = ensureAllParameters$1;

  var OpenBio = {
    name: 'Open bioreactor',
    kind: 'OpenBio',
    description: '',
    url: '',
    id: '$',
    numberParameters: 52,
    numberLogParameters: 26,
    parameters: [{
      label: 'A',
      variable: 'liquidTemp',
      name: 'T° LIQ',
      description: 'Temperature of the bioreactor solution',
      factor: 100,
      unit: '°C',
      writable: false
    }, {
      label: 'B',
      variable: 'pcbTemp',
      name: 'T° PCB',
      description: 'Temperature of the bioreactor circuit',
      factor: 100,
      unit: '°C',
      writable: false
    }, {
      label: 'C',
      variable: 'pidTemp',
      name: 'Pid',
      description: 'PID absolute value',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'D',
      variable: 'targetTemp',
      name: 'T° target',
      description: 'Target temperature',
      factor: 100,
      unit: '°C',
      writable: true
    }, {
      label: 'E',
      variable: 'weight',
      name: 'Weight',
      description: 'Weight of the bioreactor tank, in internal value',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'F',
      variable: 'grWeight',
      name: 'Weight (g)',
      description: 'Weight of the bioreactor tank, in gr if calibrated',
      factor: 1,
      unit: 'g',
      writable: false
    }, {
      label: 'G',
      variable: 'waitSinceLast',
      name: 'Time since last event',
      description: 'Time in min since last weight event',
      factor: 1,
      unit: 'min',
      writable: false
    }, {
      label: 'H',
      variable: 'minWeight',
      name: 'Weight min',
      description: 'Weight min in internal unit',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'I',
      variable: 'maxWeight',
      name: 'Weight max',
      description: 'Weight max in internal unit',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'J'
    }, {
      label: 'K'
    }, {
      label: 'L'
    }, {
      label: 'M'
    }, {
      label: 'N'
    }, {
      label: 'O'
    }, {
      label: 'P'
    }, {
      label: 'Q'
    }, {
      label: 'R'
    }, {
      label: 'S'
    }, {
      label: 'T'
    }, {
      label: 'U'
    }, {
      label: 'V'
    }, {
      label: 'W'
    }, {
      label: 'X'
    }, {
      label: 'Y',
      variable: 'error',
      name: 'Error',
      unit: '',
      factor: 1,
      description: `
            bit 0: pcb probe, 1: liquid probe, 2: pcb temperature,
            3: liquid temperature, 4: target temp. range, 5: weight range`,
      writable: true,
      flags: {
        pcbProbe: {
          bit: 0,
          description: 'PCB temperature probe not responding'
        },
        liquidProbe: {
          bit: 1,
          description: 'Liquid temperature probe not responding'
        },
        pcbTemperature: {
          bit: 2,
          description: 'PCB temperature out of range'
        },
        liquidTemperature: {
          bit: 3,
          description: 'Liquid temperature out of range'
        },
        targetTemperature: {
          bit: 4,
          description: 'Target temperature out of range'
        },
        weight: {
          bit: 5,
          description: 'Weight out of range'
        }
      }
    }, {
      label: 'Z',
      variable: 'status',
      name: 'Status',
      description: `Status of the Bioreactor, the bits of this integer code
        for the state of specific elements of the reactor (eg. motor ON/OFF, PUMP ON/OFF etc.).
        bits: 0: stepper, 1: weight, 2: pid, 7: sedimentation, 8: filling, 9: emptying
        `,
      factor: 1,
      unit: '',
      writable: true,
      flags: {
        stepper: {
          bit: 0,
          description: 'Stepper running'
        },
        food: {
          bit: 1,
          description: 'Food running'
        },
        pid: {
          bit: 2,
          description: 'PID running'
        },
        sedimentation: {
          bit: 7,
          description: 'Sedimentation running'
        },
        filling: {
          bit: 8,
          description: 'Filling tank'
        },
        emptying: {
          bit: 9,
          description: 'Emptying tank'
        }
      }
    }, {
      label: 'AA',
      variable: 'stepperSpeed',
      name: 'Stepper speed',
      description: '',
      factor: 1,
      unit: 'RPM',
      writable: true
    }, {
      label: 'AB',
      variable: 'stepperOnDelay',
      name: 'Stepper on delay',
      description: 'Time in (s) for which the stepper stays on',
      factor: 1,
      unit: 's',
      writable: true
    }, {
      label: 'AC',
      variable: 'stepperOffDelay',
      name: 'Stepper off delay',
      description: 'Time to wait in (s) before between stirring periods',
      factor: 1,
      unit: 's',
      writable: true
    }, {
      label: 'AD'
    }, {
      label: 'AE'
    }, {
      label: 'AF',
      variable: 'sedTime',
      name: 'Sedimentation Time',
      description: 'Sedimentation time in min after Semi-batch operation,' + 'corresponds to the waiting time without stirring before emptying the reactor to the minimum level',
      min: 0,
      max: 32767,
      factor: 1,
      unit: 'min',
      writable: true
    }, {
      label: 'AG',
      variable: 'filledTime',
      name: 'Filled Time',
      description: 'Filled time in min after Semi-batch operation,' + 'corresponds to the total time with and without stirring before emptying the reactor to the minimum level' + 'must be set longer than the sedimentation time if stirring is desired',
      min: 0,
      max: 32767,
      factor: 1,
      unit: 'min',
      writable: true
    }, {
      label: 'AH',
      variable: 'weightFactor',
      name: 'Weight factor',
      description: 'Factor allowing to convert the internal weight value to g',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AI',
      variable: 'weightOffset',
      name: 'Weight offset',
      description: '',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AJ'
    }, {
      label: 'AK'
    }, {
      label: 'AL'
    }, {
      label: 'AM'
    }, {
      label: 'AN'
    }, {
      label: 'AO'
    }, {
      label: 'AP'
    }, {
      label: 'AQ'
    }, {
      label: 'AR'
    }, {
      label: 'AS'
    }, {
      label: 'AT'
    }, {
      label: 'AU'
    }, {
      label: 'AV'
    }, {
      label: 'AW'
    }, {
      label: 'AX'
    }, {
      label: 'AY'
    }, {
      label: 'AZ',
      variable: 'enable',
      name: 'Enable',
      description: 'pid - food - stepper : ex. 1: only stepper',
      factor: 1,
      unit: '',
      writable: true,
      flags: {
        stepper: {
          bit: 0,
          description: 'Stepper control'
        },
        food: {
          bit: 1,
          description: 'Food control'
        },
        pid: {
          bit: 2,
          description: 'PID control'
        }
      }
    }],
    events: [{
      id: 1,
      name: 'Arduino boot',
      description: ''
    }, {
      id: 2,
      name: 'Set safe mode',
      description: ''
    }, {
      id: 3,
      name: 'Status enable',
      description: '0:stepper, 1:food, 2:pid, 7:sedimentation, 8:filling, 9:emptying',
      flags: {
        0: 'stepper',
        1: 'food',
        2: 'pid',
        7: 'sedimentation',
        8: 'filling',
        9: 'emptying'
      }
    }, {
      id: 4,
      name: 'Status disable',
      description: '0:stepper, 1:food, 2:pid, 7:sedimentation, 8:filling, 9:emptying',
      flags: {
        0: 'stepper',
        1: 'food',
        2: 'pid',
        7: 'sedimentation',
        8: 'filling',
        9: 'emptying'
      }
    }, {
      id: 6,
      name: 'Error: failed',
      description: '0:stepper, 1:food, 2:pid, 7:sedimentation, 8:filling, 9:emptying',
      flags: {
        0: 'pcb temperature probe',
        1: 'liquid temperature probe',
        2: 'pcb temperature range',
        3: 'liquid temperature range',
        4: 'target temperature range',
        5: 'weight range'
      }
    }, {
      id: 7,
      name: 'Error: recover',
      description: '0:stepper, 1:food, 2:pid, 7:sedimentation, 8:filling, 9:emptying',
      flags: {
        0: 'pcb temperature probe',
        1: 'liquid temperature probe',
        2: 'pcb temperature range',
        3: 'liquid temperature range',
        4: 'target temperature range',
        5: 'weight range'
      }
    }, {
      id: 20,
      name: 'Rotation start',
      description: ''
    }, {
      id: 21,
      name: 'Rotation stop',
      description: ''
    }, {
      id: 150,
      name: 'Not found log entry N',
      description: ''
    }, {
      id: 255,
      name: 'Save all parameters',
      description: ''
    }, {
      id: 256,
      name: 'Change value of A',
      description: ''
    }, {
      id: 257,
      name: 'Change value of B',
      description: ''
    }]
  };

  var OpenBio6 = {
    name: 'Open bioreactor v6',
    kind: 'OpenBio6',
    description: '',
    url: '',
    id: '6',
    numberParameters: 68,
    numberLogParameters: 26,
    parameters: [{
      label: 'A',
      variable: 'externalTemperature1',
      name: 'T° EXT 1',
      description: 'External temperature 1',
      factor: 100,
      unit: '°C',
      writable: false
    }, {
      label: 'B',
      variable: 'externalTemperature2',
      name: 'T° EXT 2',
      description: 'External temperature 2',
      factor: 100,
      unit: '°C',
      writable: false
    }, {
      label: 'C',
      variable: 'pcbTemp',
      name: 'T° PCB',
      description: 'Temperature of the bioreactor circuit',
      factor: 100,
      unit: '°C',
      writable: false
    }, {
      label: 'D',
      variable: 'pidTemp',
      name: 'Pid',
      description: 'PID absolute value',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'E',
      variable: 'targetTemp',
      name: 'T° target',
      description: 'Target temperature',
      factor: 100,
      unit: '°C',
      writable: true
    }, {
      label: 'F',
      variable: 'weight',
      name: 'Weight',
      description: 'Weight of the bioreactor tank, in internal value',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'G',
      variable: 'grWeight',
      name: 'Weight (g)',
      description: 'Weight of the bioreactor tank, in gr if calibrated',
      factor: 1,
      unit: 'g',
      writable: false
    }, {
      label: 'H',
      variable: 'minWeight',
      name: 'Weight min',
      description: 'Weight min in internal unit',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'I',
      variable: 'maxWeight',
      name: 'Weight max',
      description: 'Weight max in internal unit',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'J'
    }, {
      label: 'K'
    }, {
      label: 'L'
    }, {
      label: 'M'
    }, {
      label: 'N'
    }, {
      label: 'O'
    }, {
      label: 'P'
    }, {
      label: 'Q'
    }, {
      label: 'R'
    }, {
      label: 'S'
    }, {
      label: 'T'
    }, {
      label: 'U'
    }, {
      label: 'V'
    }, {
      label: 'W',
      variable: 'currentStep',
      name: 'Current step',
      description: 'Current step in the pipeline',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'X',
      variable: 'currentWaitTime',
      name: 'Current wait time',
      description: 'Current step wait time',
      factor: 1,
      unit: 'min',
      writable: false
    }, {
      label: 'Y',
      variable: 'error',
      name: 'Error',
      unit: '',
      factor: 1,
      description: `
            bit 0: pcb probe, 1: liquid probe, 2: pcb temperature,
            3: liquid temperature, 4: target temp. range, 5: weight range`,
      writable: true,
      flags: {
        0: 'PCB temperature probe error',
        1: 'External temperature 1 probe error',
        2: 'External temperature 2 probe error',
        3: 'PCB temperature out of range',
        4: 'External temperature 1 out of range',
        5: 'External temperature 2 out of range',
        6: 'Target temperature out of range',
        7: 'Weight out of range'
      }
    }, {
      label: 'Z',
      variable: 'status',
      name: 'Status',
      description: `Status of the Bioreactor, the bits of this integer code
        for the state of specific elements of the reactor (eg. motor ON/OFF, PUMP ON/OFF etc.).
        bits: 0: stepper, 1: weight, 2: pid, 7: sedimentation, 8: filling, 9: emptying
        `,
      factor: 1,
      unit: '',
      writable: true,
      flags: {
        0: 'pid',
        1: 'stepper',
        2: 'output 1',
        3: 'output 2',
        4: 'output 3',
        5: 'output 4'
      }
    }, {
      label: 'AA',
      variable: 'stepperSpeed',
      name: 'Stepper speed',
      description: '',
      factor: 1,
      unit: 'RPM',
      writable: true
    }, {
      label: 'AB',
      variable: 'stepperSteps',
      name: 'Stepper steps',
      description: 'Number of step before changing direction. 1 tour = 200 steps',
      factor: 1,
      unit: '',
      writable: true
    }, {
      label: 'AC',
      variable: 'stepperOffDelay',
      name: 'Stepper off delay',
      description: 'Time to wait in (s) before between stirring periods',
      factor: 1,
      unit: 's',
      writable: true
    }, {
      label: 'AD',
      variable: 'weightFactor',
      name: 'Weight factor',
      description: 'Factor allowing to convert the internal weight value to g',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AE',
      variable: 'weightOffset',
      name: 'Weight offset',
      description: '',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AF'
    }, {
      label: 'AG'
    }, {
      label: 'AH'
    }, {
      label: 'AI'
    }, {
      label: 'AJ'
    }, {
      label: 'AK'
    }, {
      label: 'AL'
    }, {
      label: 'AM'
    }, {
      label: 'AN'
    }, {
      label: 'AO'
    }, {
      label: 'AP'
    }, {
      label: 'AQ'
    }, {
      label: 'AR'
    }, {
      label: 'AS'
    }, {
      label: 'AT'
    }, {
      label: 'AU'
    }, {
      label: 'AV'
    }, {
      label: 'AW'
    }, {
      label: 'AX'
    }, {
      label: 'AY'
    }, {
      label: 'AZ',
      variable: 'enable',
      name: 'Enable',
      description: 'pid - food - stepper : ex. 1: only stepper',
      factor: 1,
      unit: '',
      writable: true,
      flags: {
        0: 'pid',
        1: 'stepper',
        2: 'output 1',
        3: 'output 2',
        4: 'output 3',
        5: 'output 4'
      }
    }, {
      label: 'BA',
      variable: 'step01',
      name: 'Step 1',
      description: 'Step 1'
    }, {
      label: 'BB',
      variable: 'step02',
      name: 'Step 2',
      description: 'Step 2'
    }, {
      label: 'BC',
      variable: 'step03',
      name: 'Step 3',
      description: 'Step 3'
    }, {
      label: 'BD',
      variable: 'step04',
      name: 'Step 4',
      description: 'Step 4'
    }, {
      label: 'BE',
      variable: 'step05',
      name: 'Step 5',
      description: 'Step 5'
    }, {
      label: 'BF',
      variable: 'step06',
      name: 'Step 6',
      description: 'Step 6'
    }, {
      label: 'BG',
      variable: 'step07',
      name: 'Step 7',
      description: 'Step 7'
    }, {
      label: 'BH',
      variable: 'step08',
      name: 'Step 8',
      description: 'Step 8'
    }, {
      label: 'BI',
      variable: 'step09',
      name: 'Step 9',
      description: 'Step 9'
    }, {
      label: 'BJ',
      variable: 'step10',
      name: 'Step 10',
      description: 'Step 10'
    }, {
      label: 'BK',
      variable: 'step11',
      name: 'Step 11',
      description: 'Step 11'
    }, {
      label: 'BL',
      variable: 'step12',
      name: 'Step 12',
      description: 'Step 12'
    }, {
      label: 'BM',
      variable: 'step13',
      name: 'Step 13',
      description: 'Step 13'
    }, {
      label: 'BN',
      variable: 'step14',
      name: 'Step 14',
      description: 'Step 14'
    }, {
      label: 'BO',
      variable: 'step15',
      name: 'Step 15',
      description: 'Step 15'
    }, {
      label: 'BP',
      variable: 'step16',
      name: 'Step 16',
      description: 'Step 16'
    }],
    events: [{
      id: 1,
      name: 'Arduino boot',
      description: ''
    }, {
      id: 2,
      name: 'Set safe mode',
      description: ''
    }, {
      id: 3,
      name: 'Status enable',
      description: '0:stepper, 1:food, 2:pid, 7:sedimentation, 8:filling, 9:emptying',
      flags: {
        0: 'stepper',
        1: 'food',
        2: 'pid',
        7: 'sedimentation',
        8: 'filling',
        9: 'emptying'
      }
    }, {
      id: 4,
      name: 'Status disable',
      description: '0:stepper, 1:food, 2:pid, 7:sedimentation, 8:filling, 9:emptying',
      flags: {
        0: 'stepper',
        1: 'food',
        2: 'pid',
        7: 'sedimentation',
        8: 'filling',
        9: 'emptying'
      }
    }, {
      id: 6,
      name: 'Error: failed',
      description: '0:stepper, 1:food, 2:pid, 7:sedimentation, 8:filling, 9:emptying',
      flags: {
        0: 'pcb temperature probe',
        1: 'liquid temperature probe',
        2: 'pcb temperature range',
        3: 'liquid temperature range',
        4: 'target temperature range',
        5: 'weight range'
      }
    }, {
      id: 7,
      name: 'Error: recover',
      description: '0:stepper, 1:food, 2:pid, 7:sedimentation, 8:filling, 9:emptying',
      flags: {
        0: 'pcb temperature probe',
        1: 'liquid temperature probe',
        2: 'pcb temperature range',
        3: 'liquid temperature range',
        4: 'target temperature range',
        5: 'weight range'
      }
    }, {
      id: 20,
      name: 'Rotation start',
      description: ''
    }, {
      id: 21,
      name: 'Rotation stop',
      description: ''
    }, {
      id: 150,
      name: 'Not found log entry N',
      description: ''
    }, {
      id: 255,
      name: 'Save all parameters',
      description: ''
    }, {
      id: 256,
      name: 'Change value of A',
      description: ''
    }, {
      id: 257,
      name: 'Change value of B',
      description: ''
    }]
  };

  var OpenSpectro = {
    name: 'Open spectrophotometer',
    kind: 'OpenSpectro',
    numberParameters: 26,
    id: 'S',
    description: '',
    url: '',
    parameters: [{
      label: 'A',
      name: 'Red point',
      description: 'which point of the linear detector is the maximum for red',
      factor: 1,
      unit: 'pixel#',
      writable: false
    }, {
      label: 'B',
      name: 'Green point',
      description: 'which point of the linear detector is the maximum for green',
      factor: 1,
      unit: 'pixel#',
      writable: false
    }, {
      label: 'C',
      name: 'Blue point',
      description: 'which point of the linear detector is the maximum for blue',
      factor: 1,
      unit: 'pixel#',
      writable: false
    }, {
      label: 'D',
      name: 'Compression',
      description: '0 means no compression, can be set',
      factor: 1,
      unit: '',
      writable: true
    }, {
      label: 'E',
      name: 'R-Intensity',
      description: 'Red led intensity (0 to 255)',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'F',
      name: 'G-Intensity',
      description: 'Green led intensity (0 to 255)',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'G',
      name: 'B-Intensity',
      description: 'Blue led intensity (0 to 255)',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'H',
      name: 'Scan#',
      description: 'Number of scans (maximum 64)',
      factor: 1,
      unit: '',
      min: 1,
      max: 64,
      writable: true
    }, {
      label: 'I',
      name: 'DelayExp',
      description: 'Delay between experiments in seconds',
      factor: 1,
      unit: 's',
      writable: false
    }, {
      label: 'J',
      name: 'Acq Time',
      description: 'Accumulation time in ms(in ms, good value=30)',
      factor: 1,
      unit: 'ms',
      writable: true
    }, {
      label: 'K',
      name: 'lambda-R',
      description: 'Red maximum wavelength (nm)',
      factor: 1,
      unit: 'nm',
      writable: false
    }, {
      label: 'L',
      name: 'lambda-G',
      description: 'Green maximum wavelength (nm)',
      factor: 1,
      unit: 'nm',
      writable: false
    }, {
      label: 'M',
      name: 'lambda-M',
      description: 'Blue maximum wavelength (nm)',
      factor: 1,
      unit: 'nm',
      writable: false
    }, {
      label: 'U',
      name: 'red test',
      description: 'Set intensity of red led for test (0 -> 255)',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'V',
      name: 'green test',
      description: 'Set intensity of green led for test (0 -> 255)',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'W',
      name: 'blue test',
      description: 'Set intensity of blue led for test (0 -> 255)',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'X',
      name: 'white test',
      description: 'Set intensity of white led for test (0 -> 255)',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'Y'
    }, {
      label: 'Z'
    }]
  };

  var SimpleSpectro = {
    name: 'Simple spectrophotometer',
    kind: 'SimpleSpectro',
    numberParameters: 26,
    description: '',
    id: 'T',
    url: '',
    parameters: [{
      label: 'A',
      name: 'Transmission of sample (Red)',
      description: 'Frequency related to the energy of red led through sample',
      factor: 1,
      unit: 'Hz',
      writable: false
    }, {
      label: 'B',
      name: 'Transmission of sample (Green)',
      description: 'Frequency related to the energy of green led through sample',
      factor: 1,
      unit: 'Hz',
      writable: false
    }, {
      label: 'C',
      name: 'Transmission of sample (Blue)',
      description: 'Frequency related to the energy of blue led through sample',
      factor: 1,
      unit: 'Hz',
      writable: false
    }, {
      label: 'D',
      name: 'Emission of sample (Blue)',
      description: 'Frequency related to the energy of blue perpendicular led re-emitted by sample (fluorescence)',
      factor: 1,
      unit: 'Hz',
      writable: false
    }, {
      label: 'F',
      name: 'Transmission of blank (Red)',
      description: 'Frequency related to the energy of red led through blank',
      factor: 1,
      unit: 'Hz',
      writable: false
    }, {
      label: 'G',
      name: 'Transmission of blank (Green)',
      description: 'Frequency related to the energy of green led through blank',
      factor: 1,
      unit: 'Hz',
      writable: false
    }, {
      label: 'H',
      name: 'Transmission of blank (Blue)',
      description: 'Frequency related to the energy of blue led through blank',
      factor: 1,
      unit: 'Hz',
      writable: false
    }, {
      label: 'I',
      name: 'Emission of blank (Blue)',
      description: 'Frequency related to the energy of blue perpendicular led re-emitted by blank (fluorescence)',
      factor: 1,
      unit: 'Hz',
      writable: false
    }, {
      label: 'K',
      name: 'Delay before blank',
      description: 'Delay before the acquisition of the blank in seconds',
      factor: 1,
      unit: 's',
      writable: true
    }, {
      label: 'L',
      name: 'Delay before sample',
      description: 'Delay before the acquisition of the sample in seconds',
      factor: 1,
      unit: 's',
      writable: true
    }, {
      label: 'M',
      name: 'Delay between experiments',
      description: 'Delay between the acquisition of the experiments (kinetic) in seconds',
      factor: 1,
      unit: 's',
      writable: true
    }, {
      label: 'N',
      name: 'Nb experiments',
      description: 'Number of experiments for kinetic (max 240 / (number colors + 1))',
      factor: 1,
      unit: '',
      writable: true
    }, {
      label: 'O',
      name: 'Next exp. number',
      description: 'Number of the next experiment',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'P',
      name: 'Waiting time',
      description: 'Current waiting time before next experiment',
      factor: 1,
      unit: 's',
      writable: false
    }, {
      label: 'Q',
      name: 'Nb sampling',
      description: 'Number of acquisitions of 100ms that will be taken (default 10). This value could be reduced for fast kinetic',
      factor: 1,
      unit: '',
      writable: true
    }, {
      label: 'R',
      name: 'Invert rotary',
      description: ' Invert the rotary button direction',
      factor: 1,
      unit: '',
      writable: true
    }, {
      label: 'S',
      name: 'Battery voltage',
      description: 'Current battery voltage',
      factor: 100,
      unit: 'V',
      writable: true
    }, {
      label: 'T',
      name: 'Temperature',
      description: 'Current temperature',
      factor: 100,
      unit: '°C',
      writable: true
    }, {
      label: 'V',
      name: 'Active channels',
      description: 'Active leds and other. A number between 0 and 63. Each bit represents a function (Red, Green, Blue, UV, Voltage, Temperature). 5 would correspond to Red and Blue (binary combination).',
      factor: 1,
      unit: '',
      writable: true
    }, {
      label: 'W',
      name: 'Error',
      description: 'Error',
      factor: 1,
      unit: '',
      writable: true
    }, {
      label: 'X',
      name: 'Result channel',
      description: 'Value of the channel that will be displayed in the result',
      factor: 1,
      unit: '',
      writable: true
    }, {
      label: 'Y',
      name: 'Status',
      description: 'Status',
      factor: 1,
      unit: '',
      writable: true
    }, {
      label: 'Z',
      name: 'Current menu',
      description: 'Current menu',
      factor: 1,
      unit: '',
      writable: true
    }]
  };

  var Solar2015 = {
    name: 'Solar decathlon 2015',
    kind: 'Solar',
    numberParameters: 4,
    numberLogParameters: 4,
    id: '#',
    description: '',
    url: '',
    parameters: [{
      label: 'A',
      name: 'Temperature',
      description: '',
      factor: 100,
      unit: '°C',
      writable: false
    }, {
      label: 'B',
      name: 'Light',
      description: '',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'C',
      name: 'Pressure',
      description: '',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'D',
      name: 'Humidity',
      description: '',
      factor: 10,
      unit: '%',
      writable: false
    }]
  };

  var Beemos = {
    name: 'Bee Monistoring System',
    kind: 'Beemos',
    numberParameters: 52,
    numberLogParameters: 26,
    id: 'B',
    description: 'Bee Monitoring System data result',
    url: '',
    parameters: [{
      label: 'A',
      variable: 'externalTemperature',
      name: 'Ext temperature',
      description: 'External temperature',
      factor: 100,
      unit: '°C',
      writable: false
    }, {
      label: 'B',
      variable: 'externalHumidity',
      name: 'Ext humidity',
      description: 'External humidity',
      factor: 100,
      unit: '%',
      writable: false
    }, {
      label: 'C',
      variable: 'pressure',
      name: 'Pressure',
      description: 'Press',
      factor: 10,
      unit: 'mbar',
      writable: false
    }, {
      label: 'D',
      variable: 'luminosity',
      name: 'Luminosity',
      description: 'Luminosity (Arbitrary unit)',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'E',
      variable: 'red',
      name: 'Red',
      description: 'Red luminosity (Arbitrary unit)',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'F',
      variable: 'green',
      name: 'Green',
      description: 'Green luminosity (Arbitrary unit)',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'G',
      variable: 'blue',
      name: 'Blue',
      description: 'Blue luminosity (Arbitrary unit)',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'H',
      variable: 'infrared',
      name: 'Infrared',
      description: 'Infrared luminosity (Arbitrary unit)',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'I',
      variable: 'latitude',
      name: 'Latitude',
      description: 'Latitude',
      factor: 100,
      unit: '°',
      writable: false
    }, {
      label: 'J',
      variable: 'longitude',
      name: 'Longitude',
      description: 'Longitude',
      factor: 100,
      unit: '°',
      writable: false
    }, {
      label: 'K',
      variable: 'internalTemperature',
      name: 'Int temperature',
      description: 'Internal temperature',
      factor: 100,
      unit: '°C',
      writable: false
    }, {
      label: 'L',
      variable: 'internalHumidity',
      name: 'Int humidity',
      description: 'Internal humidity',
      factor: 100,
      unit: '%',
      writable: false
    }, {
      label: 'M',
      variable: 'internalTemperatureA',
      name: 'Int temperature A',
      description: 'Internal temperature A',
      factor: 100,
      unit: '°C',
      writable: false
    }, {
      label: 'N',
      variable: 'internalTemperatureB',
      name: 'Int temperature B',
      description: 'Internal temperature B',
      factor: 100,
      unit: '°C',
      writable: false
    }, {
      label: 'O',
      variable: 'weightInternalUnit',
      name: 'Weight internal unit',
      description: 'Weight in internal unit',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'P',
      variable: 'weight',
      name: 'Weight',
      description: 'Weight',
      factor: 100,
      unit: 'kg',
      writable: false
    }, {
      label: 'W',
      variable: 'battery',
      name: 'Battery',
      description: 'Battery voltage',
      factor: 1000,
      unit: 'V',
      writable: false
    }, {
      label: 'X',
      variable: 'charging',
      name: 'Charging',
      description: 'Indication showing if the battery is charging',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'Y',
      variable: 'rssi',
      name: 'Wifi RSSI',
      description: 'Power of Wifi signal',
      factor: 1,
      unit: 'dB',
      writable: false
    }, {
      label: 'Z',
      variable: 'error',
      name: 'Error',
      description: 'Current error code',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AA',
      variable: 'loggingInterval',
      name: 'Logging interval',
      description: 'Interval in seconds between logs',
      factor: 1,
      unit: 's',
      writable: true
    }, {
      label: 'AB',
      variable: 'weightOffset',
      name: 'Weight offset',
      description: 'Offset to convert weight from internal unit',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AC',
      variable: 'weightFactor',
      name: 'Weight factor',
      description: 'Factor to convert the weight from internal unit',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AK',
      variable: 'gate1In',
      name: 'Gate 1 IN',
      description: 'Number of input on gate 1',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AL',
      variable: 'gate1Out',
      name: 'Gate 1 OUT',
      description: 'Number of output on gate 1',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AM',
      variable: 'gate2In',
      name: 'Gate 2 IN',
      description: 'Number of input on gate 2',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AN',
      variable: 'gate2Out',
      name: 'Gate 2 OUT',
      description: 'Number of output on gate 2',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AO',
      variable: 'gate3In',
      name: 'Gate 3 IN',
      description: 'Number of input on gate 3',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AP',
      variable: 'gate3Out',
      name: 'Gate 3 OUT',
      description: 'Number of output on gate 3',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AQ',
      variable: 'gate4In',
      name: 'Gate 4 IN',
      description: 'Number of input on gate 4',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AR',
      variable: 'gate4Out',
      name: 'Gate 4 OUT',
      description: 'Number of output on gate 4',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AS',
      variable: 'gate5In',
      name: 'Gate 5 IN',
      description: 'Number of input on gate 5',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AT',
      variable: 'gate5Out',
      name: 'Gate 5 OUT',
      description: 'Number of output on gate 5',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AU',
      variable: 'gate6In',
      name: 'Gate 6 IN',
      description: 'Number of input on gate 6',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AV',
      variable: 'gate6Out',
      name: 'Gate 6 OUT',
      description: 'Number of output on gate 6',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AW',
      variable: 'gate7In',
      name: 'Gate 7 IN',
      description: 'Number of input on gate 7',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AX',
      variable: 'gate7Out',
      name: 'Gate 7 OUT',
      description: 'Number of output on gate 7',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AY',
      variable: 'gate8In',
      name: 'Gate 8 IN',
      description: 'Number of input on gate 8',
      factor: 1,
      unit: '',
      writable: false
    }, {
      label: 'AZ',
      variable: 'gate8Out',
      name: 'Gate 8 OUT',
      description: 'Number of output on gate 8',
      factor: 1,
      unit: '',
      writable: false
    }]
  };

  var Computer = {
    name: 'Computer monitoring',
    kind: 'Computer',
    numberParameters: 16,
    numberLogParameters: 16,
    description: '',
    id: 'C',
    url: '',
    parameters: [{
      label: 'A',
      variable: 'cpuTemperature',
      name: 'CPU Temperature',
      description: '',
      factor: 1,
      unit: '°C',
      writable: false
    }, {
      label: 'B',
      variable: 'memFree',
      name: 'Free memory',
      description: 'Free memory in percent',
      factor: 1,
      unit: '%',
      writable: false
    }, {
      label: 'C',
      variable: 'swapFree',
      name: 'Free swap',
      description: 'Free swap in percent',
      factor: 1,
      unit: '%',
      writable: false
    }, {
      label: 'D',
      variable: 'fsRead',
      name: 'FS Read',
      description: 'File system read in kb',
      factor: 1,
      unit: 'kb',
      writable: false
    }, {
      label: 'E',
      variable: 'fsWrite',
      name: 'FS Write',
      description: 'File system read in kb',
      factor: 1,
      unit: 'kb',
      writable: false
    }, {
      label: 'F',
      variable: 'networkRead',
      name: 'Network Read',
      description: 'File system read in kb',
      factor: 1,
      unit: 'kb',
      writable: false
    }, {
      label: 'G',
      variable: 'networkWrite',
      name: 'Network Write',
      description: 'File system read in kb',
      factor: 1,
      unit: 'kb',
      writable: false
    }, {
      label: 'H',
      variable: 'load',
      name: 'Load',
      description: 'Total load',
      factor: 1,
      unit: '%',
      writable: false
    }, {
      label: 'I',
      variable: 'userLoad',
      name: 'User load',
      description: 'Load from user',
      factor: 1,
      unit: '%',
      writable: false
    }, {
      label: 'J',
      variable: 'systemLoad',
      name: 'System load',
      description: 'Load from system',
      factor: 1,
      unit: '%',
      writable: false
    }, {
      label: 'K',
      variable: 'niceLoad',
      name: 'Nice load',
      description: 'Load for Nice',
      factor: 1,
      unit: '%',
      writable: false
    }, {
      label: 'L',
      variable: 'idleLoad',
      name: 'Idle load',
      description: 'Idle percent of time',
      factor: 1,
      unit: '%',
      writable: false
    }, {
      label: 'M',
      variable: 'irqLoad',
      name: 'IRQ load',
      description: 'Load due to IRQ',
      factor: 1,
      unit: '%',
      writable: false
    }, {
      label: 'N',
      variable: 'fsMinimalUse',
      name: 'FS minimal use',
      description: 'Minimal percent spaced used in a filesystem',
      factor: 1,
      unit: '%',
      writable: false
    }, {
      label: 'O',
      variable: 'fsMaximalUse',
      name: 'FS maximal use',
      description: 'Maximal percent spaced used in a filesystem',
      factor: 1,
      unit: '%',
      writable: false
    }, {
      label: 'P'
    }]
  };

  const ensureAllParameters = ensureAllParameters_1;
  let devices = {
    OpenBio: ensureAllParameters(OpenBio),
    OpenBio6: ensureAllParameters(OpenBio6),
    OpenSpectro: ensureAllParameters(OpenSpectro),
    SimpleSpectro: ensureAllParameters(SimpleSpectro),
    Solar2015: ensureAllParameters(Solar2015),
    Beemos: ensureAllParameters(Beemos),
    Computer: ensureAllParameters(Computer),
    fromDeviceID
  };
  /**
   * Return a device information from the deviceID
   * @param {string|number} id
   * @returns
   */

  function fromDeviceID(id) {
    if (typeof id === 'number') {
      id = String.fromCharCode(id / 256 >> 0) + String.fromCharCode(id % 256);
    }

    if (typeof id !== 'string') {
      throw Error('Device ID not a string or number');
    }

    if (id.length !== 2) {
      throw Error(`Device ID should be 2 character long and found: ${id.ldength}`);
    }

    const firstCharacter = id.substring(0, 1);

    for (let key in devices) {
      if (devices[key].id === firstCharacter) {
        return ensureAllParameters(devices[key]);
      }
    }

    return undefined;
  }

  var src$1 = devices;

  var numberToLabel$3 = function numberToLabel(value) {
    if (value < 26) {
      return String.fromCharCode(65 + value);
    } else {
      return String.fromCharCode(Math.floor(value / 26) + 64, 65 + value % 26);
    }
  };

  const DeviceInformation$1 = src$1;
  const hexToInt16$2 = hexToInt16$3;
  const numberToLabel$2 = numberToLabel$3;
  /**
   * Parse a buffer (String) containing 4 hexadecimal symbols per parameter
   * @param {string} buffer
   * @param {object} [options={}]
   * @param {boolean} [options.parameterLabel=false] Use the variable name of device info as property name
   * @param {boolean} [options.parameterInfo=false] Show all the information about the parameter in the value
   * @param {boolean} [options.flagInfo=false] Show all the information about the flags, can only be true if `options.parameterInfo=true`!
   * @param {string} [options.kind=undefined] Specify a device type from those that exist in `legoino-device-information`
   * @param {object} [options.deviceInformation=undefined] Pass information for a device that does not exist in `legoino-device-information`. To use if `options.kind` is undefined.
   * @return {object} The parsed parameters
   */

  var parseParameters$2 = function parseParameters(buffer) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    let {
      parameterLabel = false,
      parameterInfo = false,
      flagInfo = false,
      kind = undefined,
      deviceInformation = DeviceInformation$1[kind]
    } = options;

    if (parameterInfo === false && flagInfo === true) {
      throw new Error('parameterInfo must be true when flagInfo is true');
    }

    let parameters = {};
    let parametersArray = [];
    let numberParameters = buffer.length / 4;

    if (deviceInformation && numberParameters !== deviceInformation.numberParameters && numberParameters !== deviceInformation.numberLogParameters) {
      throw new Error(`The number of parameters is not equal to the one described in the deviceInformation. ${buffer} ${kind} ${deviceInformation.numberParameters} != ${deviceInformation.numberLogParameters}`);
    }

    if (!deviceInformation) deviceInformation = {
      parameters: []
    };

    for (let i = 0; i < numberParameters; i++) {
      if (!deviceInformation.parameters[i]) {
        deviceInformation.parameters[i] = {
          variable: numberToLabel$2(i),
          label: numberToLabel$2(i),
          factor: 1
        };
      }

      let valueNumber = hexToInt16$2(buffer.substring(i * 4, i * 4 + 4));
      if (valueNumber === -32768) valueNumber = undefined;
      let label = parameterLabel ? deviceInformation.parameters[i].variable || deviceInformation.parameters[i].name : numberToLabel$2(i);
      if (label === undefined) continue;
      let value;

      if (parameterInfo) {
        value = Object.assign({}, deviceInformation.parameters[i], {
          index: i,
          originalValue: valueNumber,
          value: valueNumber === undefined ? valueNumber : valueNumber / (deviceInformation.parameters[i].factor || 1)
        });

        if (flagInfo) {
          let flags = deviceInformation.parameters[i].flags;

          if (flags !== undefined) {
            for (let key in flags) {
              flags[key].value = (value.value & 1 << flags[key].bit) >> flags[key].bit;
            }

            value.flags = flags;
          }
        }
      } else {
        value = valueNumber === undefined ? valueNumber : valueNumber / (deviceInformation.parameters[i].factor || 1);
      }

      if (value !== undefined) parameters[label] = value;
      parametersArray.push(value);
    }

    return {
      parameters,
      parametersArray
    };
  };

  const debug$2 = browser.exports('legoino:parser:processMultilogLine');
  const checkCheckDigit$1 = checkCheckDigit$2;
  const hexToInt16$1 = hexToInt16$3;
  const parseParameters$1 = parseParameters$2;
  /**
   * Parse a multilog line.
   * @param {string} line
   * @param {object} [options={}]
   * @param {boolean} [options.hasEvent=true] Specify wether the log contains an event
   * @param {boolean} [options.flatten=false] The parsed log will have all properties at the same level (no sub-object for the parameters)
   * @param {boolean} [options.parametersArray=false] Add an array with all the parameters to the result
   * @param {boolean} [options.parameterLabel=false] Use the variable property of device info as property name
   * @param {boolean} [options.parameterInfo=false] Show all the information about the parameter in the value
   * @param {string} [options.kind=undefined] Specify a device type from those that exist in `legoino-device-information`
   * @param {object} [options.deviceInformation=undefined] Pass information for a device that does not exist in `legoino-device-information`. To use if `options.kind` is undefined.
   * @return {object} The parsed line.
   *
   * Warning: parameters that are undefined are not returned!
   */

  var parseMultilogLine$1 = function parseMultilogLine(line, options) {
    let {
      hasEvent = true,
      flatten = false,
      parametersArray = false
    } = options; // keep only valid characters

    line = line.replace(/[^0-9A-F]/gi, '');
    const entry = {};

    if (checkCheckDigit$1(line)) {
      entry.id = parseInt(`${line.substr(0, 8)}`, 16);
      entry.epoch = parseInt(`${line.substr(8, 8)}`, 16) * 1000;
      let parseResult = parseParameters$1(line.substring(16, line.length - 6 - (hasEvent ? 8 : 0)), options);

      if (flatten) {
        Object.assign(entry, parseResult.parameters);
      } else {
        entry.parameters = parseResult.parameters;
      }

      if (parametersArray) {
        entry.parametersArray = parseResult.parametersArray;
      }

      if (hasEvent) {
        entry.eventId = hexToInt16$1(line.substr(line.length - 14, 4));
        entry.eventValue = hexToInt16$1(line.substr(line.length - 10, 4));
      }

      entry.deviceId = hexToInt16$1(line.substr(line.length - 6, 4));

      if (!entry.deviceId) {
        throw new Error('Could not parse device id in processMultilogLine');
      }
    } else {
      debug$2('Check digit error', line);
      throw new Error('Check digit error');
    }

    return entry;
  };

  const debug$1 = browser.exports('legoino:parser:parseMutilog');
  const parseMultilogLine = parseMultilogLine$1;
  /**
   * Parse a multilog string
   * @param {*} buffer
   * @param {object} [options={}]
   */

  var parseMultilog = function parseMultiLog(buffer) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    let lines = buffer.split(/[\r\n]+/).filter(line => line);
    let entries = [];

    for (let line of lines) {
      let entry = parseMultilogLine(line, options);
      if (entry) entries.push(entry);
    } // Check that all entries come from the same device!!


    if (entries.length > 0) {
      let deviceId = entries[0].deviceId;

      for (let i = 1; i < entries.length; i++) {
        if (entries[i].deviceId !== deviceId) {
          debug$1(`checkdigit is ok but all lines did not come from the same device. There are at least 2 device ids: ${entries[0].deviceId}, ${entries[i].deviceId}`);
          throw new Error('all lines do not have the same id');
        }
      }
    }

    return entries;
  };

  var deviceIdNumberToString$1 = function deviceIdNumberToString(idNumber) {
    return String.fromCharCode(idNumber / 256 | 0) + String.fromCharCode(idNumber % 256);
  };

  const debug = browser.exports('legoino:parser:parseCurrentSettings');
  const calculateCheckDigit$2 = calculateCheckDigit$4;
  const checkCheckDigit = checkCheckDigit$2;
  const deviceIdNumberToString = deviceIdNumberToString$1;
  const hexToInt16 = hexToInt16$3;
  const parseParameters = parseParameters$2;
  /**
   * Parse a current settings log.
   * @param {string} line
   * @param {object} [options={}]
   * @param {boolean} [options.flatten=false] The parsed log will have all properties at the same level (no sub-object for the parameters)
   * @param {boolean} [options.parametersArray=false] Add an array with all the parameters to the result
   * @param {boolean} [options.parameterLabel=false] Use the variable property of device info as property name
   * @param {boolean} [options.parameterInfo=false] Show all the information about the parameter in the value
   * @param {string} [options.kind=undefined] Specify a device type from those that exist in `legoino-device-information`
   * @param {object} [options.deviceInformation=undefined] Pass information for a device that does not exist in `legoino-device-information`. To use if `options.kind` is undefined.
   * @return {object} The parsed settings.
   *
   * Warning: parameters that are undefined are not returned!
   */

  var parseCurrentSettings = function parseCurrentSettings(line, options) {
    let {
      flatten = false,
      parametersArray = false
    } = options; // keep only valid characters

    line = line.replace(/[^0-9A-F]/gi, '');
    let entry = {};
    let parseResult;

    if (checkCheckDigit(line)) {
      entry.epoch = parseInt(line.substring(0, 8), 16) * 1000;
      parseResult = parseParameters(line.substring(8, line.length - 6), options);
      entry.deviceId = hexToInt16(line.substring(line.length - 6, line.length - 2));
      entry.deviceCode = deviceIdNumberToString(entry.deviceId);
    } else {
      debug('Check digit error', line);
      throw new Error(`Check digit error. Should be: ${calculateCheckDigit$2(line).toString(16)}`);
    }

    if (flatten) {
      Object.assign(entry, parseResult.parameters);
    } else {
      entry.parameters = parseResult.parameters;
    }

    if (parametersArray) {
      entry.parametersArray = parseResult.parametersArray;
    }

    return entry;
  };

  var deviceIdStringToNumber = function deviceIdStringToNumber(idString) {
    if (idString === undefined) return undefined;

    if (idString.length === 2) {
      return idString.charCodeAt(0) * 256 + idString.charCodeAt(1);
    } else {
      throw new Error('Id does not have the expected 2 char format');
    } // return idString;

  };

  var labelToNumber$1 = function labelToNumber(code) {
    let value = 0;

    for (let char of code) {
      value *= 26;
      value += char.charCodeAt(0) - 64;
    }

    return value - 1;
  };

  const DeviceInformation = src$1;
  const labelToNumber = labelToNumber$1;

  var valueToRawNumber = function valueToRawNumber(label, value, kind) {
    const deviceInformation = DeviceInformation[kind];
    return value * (deviceInformation.parameters[labelToNumber(label)].factor || 1);
  };

  var int16ToHex$2 = function int16ToHex() {
    let value = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : null;
    if (value > 32767 || value < -32767 || value === null) value = -32768;

    if (value < 0) {
      value += 65536;
    }

    return Number(value).toString(16).padStart(4, '0').toUpperCase();
  };

  /**
   * A log entry is a hexadecimal line composed of :
   * - epoch (8)
   * - a list of parameters values (n * 4)
   * - a device ID (4)
   * - a checkdigit (2)
   *
   * This means that for 26 parameters, the length of a log is 134 hexadecimal characters.
   */


  const calculateCheckDigit$1 = calculateCheckDigit$4;
  const int16ToHex$1 = int16ToHex$2;
  const numberToLabel$1 = numberToLabel$3;

  var createCompactLog = function createCompactLog() {
    let data = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    let numberParameters = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 26;
    if (!data.parameters) data.parameters = [];
    let result = '';
    result += Number(data.epoch | 0).toString(16).padStart(8, '0');

    for (let i = 0; i < numberParameters; i++) {
      let label = numberToLabel$1(i);
      result += int16ToHex$1(data.parameters[label]);
    }

    result += int16ToHex$1(data.deviceId);
    result += calculateCheckDigit$1(result).toString(16).padStart(2, '0');
    return result.toUpperCase();
  };

  /**
   * A log entry is a hexadecimal line composed of :
   * - a sequential ID (8)
   * - epoch (8)
   * - a list of parameters values (n * 4)
   * - a log event value (4)
   * - a device ID (4)
   * - a check digit (2)
   *
   * This means that for 26 parameters, the length of a log is 134 hexadecimal characters.
   */


  const calculateCheckDigit = calculateCheckDigit$4;
  const int16ToHex = int16ToHex$2;
  const numberToLabel = numberToLabel$3;

  var createMultiLog = function createCompactLog() {
    let data = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    let numberParameters = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 26;
    if (!data.parameters) data.parameters = [];
    let result = '';
    result += Number(data.id | 0).toString(16).padStart(8, '0');
    result += Number(data.epoch | 0).toString(16).padStart(8, '0');

    for (let i = 0; i < numberParameters; i++) {
      let label = numberToLabel(i);
      result += int16ToHex(data.parameters[label]);
    }

    result += int16ToHex(data.eventId);
    result += int16ToHex(data.eventValue);
    result += int16ToHex(data.deviceId);
    result += calculateCheckDigit(result).toString(16).padStart(2, '0');
    return result.toUpperCase();
  };

  var src = {
    parseMultilog: parseMultilog,
    parseMultilogLine: parseMultilogLine$1,
    parseCurrentSettings: parseCurrentSettings,
    deviceIdNumberToString: deviceIdNumberToString$1,
    deviceIdStringToNumber: deviceIdStringToNumber,
    calculateCheckDigit: calculateCheckDigit$4,
    valueToRawNumber: valueToRawNumber,
    createCompactLog: createCompactLog,
    createMultiLog: createMultiLog,
    DevicesInfo: src$1,
    labelToNumber: labelToNumber$1,
    numberToLabel: numberToLabel$3
  };

  return src;

}));
//# sourceMappingURL=legoino-util.js.map
